use crate::params::{NTRU_N, NTRU_PACK_DEG};
use crate::poly::Poly;
use crate::poly::MODQ;

#[cfg(any(feature = "ntruhps2048509", feature = "ntruhps2048677"))]
pub fn poly_sq_tobytes(r: &mut [u8], a: &Poly) {
    let mut t = [0u16; 8];
    for i in 0..NTRU_PACK_DEG / 8 {
        for (j, val) in t.iter_mut().enumerate() {
            *val = MODQ(a.coeffs[8 * i + j]);
        }
        r[11 * i] = t[0] as u8;
        r[11 * i + 2] = (((t[1]).wrapping_shr(5)) | ((t[2] & 0x03).wrapping_shl(6))) as u8;
        r[11 * i + 1] = (((t[0]).wrapping_shr(8)) | ((t[1] & 0x1f).wrapping_shl(3))) as u8;
        r[11 * i + 3] = (((t[2]).wrapping_shr(2)) & 0xff) as u8;
        r[11 * i + 4] = (((t[2]).wrapping_shr(10)) | ((t[3] & 0x7f).wrapping_shl(1))) as u8;
        r[11 * i + 5] = (((t[3]).wrapping_shr(7)) | ((t[4] & 0x0f).wrapping_shl(4))) as u8;
        r[11 * i + 6] = (((t[4]).wrapping_shr(4)) | ((t[5] & 0x01).wrapping_shl(7))) as u8;
        r[11 * i + 7] = (((t[5]).wrapping_shr(1)) & 0xff) as u8;
        r[11 * i + 8] = (((t[5]).wrapping_shr(9)) | ((t[6] & 0x3f).wrapping_shl(2))) as u8;
        r[11 * i + 9] = (((t[6]).wrapping_shr(6)) | ((t[7] & 0x07).wrapping_shl(5))) as u8;
        r[11 * i + 10] = (t[7]).wrapping_shr(3) as u8;
    }

    let i = NTRU_PACK_DEG as i16 / 8;
    for j in 0..NTRU_PACK_DEG as i16 - 8 * i {
        t[j as usize] = MODQ(a.coeffs[(8 * i + j) as usize]);
    }
    let j = NTRU_PACK_DEG as i16 - 8 * i;
    for x in j..8 {
        t[x as usize] = 0;
    }

    match NTRU_PACK_DEG & 0x07 {
        // cases 0 and 6 are impossible since 2 generates (Z/n)* and
        // p mod 8 in {1, 7} implies that 2 is a quadratic residue.
        4 => {
            r[(11 * i) as usize] = (t[0] & 0xff) as u8;
            r[(11 * i + 1) as usize] = ((t[0] >> 8) | ((t[1] & 0x1f) << 3)) as u8;
            r[(11 * i + 2) as usize] = ((t[1] >> 5) | ((t[2] & 0x03) << 6)) as u8;
            r[(11 * i + 3) as usize] = ((t[2] >> 2) & 0xff) as u8;
            r[(11 * i + 4) as usize] = ((t[2] >> 10) | ((t[3] & 0x7f) << 1)) as u8;
            r[(11 * i + 5) as usize] = ((t[3] >> 7) | ((t[4] & 0x0f) << 4)) as u8;
        }
        2 => {
            r[(11 * i) as usize] = (t[0] & 0xff) as u8;
            r[(11 * i + 1) as usize] = ((t[0] >> 8) | ((t[1] & 0x1f) << 3)) as u8;
            r[(11 * i + 2) as usize] = ((t[1] >> 5) | ((t[2] & 0x03) << 6)) as u8;
        }
        _ => {}
    }
}

#[cfg(feature = "ntruhps4096821")]
pub fn poly_sq_tobytes(r: &mut [u8], a: &Poly) {
    for i in 0..NTRU_PACK_DEG / 2 {
        r[3 * i] = (MODQ(a.coeffs[2 * i] as u16) & 0xff) as u8;
        r[3 * i + 1] =
            ((MODQ(a.coeffs[2 * i]) >> 8) | ((MODQ(a.coeffs[2 * i + 1]) & 0x0f) << 4)) as u8;
        r[3 * i + 2] = (MODQ(a.coeffs[2 * i + 1]) >> 4) as u8;
    }
}

#[cfg(feature = "ntruhrss701")]
pub fn poly_sq_tobytes(r: &mut [u8], a: &Poly) {
    let mut t = [0u16; 8];

    for i in 0..NTRU_PACK_DEG / 8 {
        for (j, item) in t.iter_mut().enumerate() {
            *item = MODQ(a.coeffs[8 * i + j]);
        }
        r[13 * i] = (t[0] & 0xff) as u8;
        r[13 * i + 1] = ((t[0] >> 8) | ((t[1] & 0x07) << 5)) as u8;
        r[13 * i + 2] = ((t[1] >> 3) & 0xff) as u8;
        r[13 * i + 3] = ((t[1] >> 11) | ((t[2] & 0x3f) << 2)) as u8;
        r[13 * i + 4] = ((t[2] >> 6) | ((t[3] & 0x01) << 7)) as u8;
        r[13 * i + 5] = ((t[3] >> 1) & 0xff) as u8;
        r[13 * i + 6] = ((t[3] >> 9) | ((t[4] & 0x0f) << 4)) as u8;
        r[13 * i + 7] = ((t[4] >> 4) & 0xff) as u8;
        r[13 * i + 8] = ((t[4] >> 12) | ((t[5] & 0x7f) << 1)) as u8;
        r[13 * i + 9] = ((t[5] >> 7) | ((t[6] & 0x03) << 6)) as u8;
        r[13 * i + 10] = ((t[6] >> 2) & 0xff) as u8;
        r[13 * i + 11] = ((t[6] >> 10) | ((t[7] & 0x1f) << 3)) as u8;
        r[13 * i + 12] = (t[7] >> 5) as u8;
    }

    let i = (NTRU_PACK_DEG / 8) as i16;
    for j in 0..(NTRU_PACK_DEG as i16 - 8 * i) {
        t[j as usize] = MODQ(a.coeffs[(8 * i + j) as usize]);
    }
    let j = (NTRU_PACK_DEG as i16 - 8 * i) - 1;
    for x in j + 1..8 {
        t[x as usize] = 0;
    }

    match NTRU_PACK_DEG - 8 * (NTRU_PACK_DEG / 8) {
        // cases 0 and 6 are impossible since 2 generates (Z/n)* and
        // p mod 8 in {1, 7} implies that 2 is a quadratic residue.
        4 => {
            r[(13 * i) as usize] = (t[0] & 0xff) as u8;
            r[(13 * i + 1) as usize] = ((t[0] >> 8) | ((t[1] & 0x07) << 5)) as u8;
            r[(13 * i + 2) as usize] = ((t[1] >> 3) & 0xff) as u8;
            r[(13 * i + 3) as usize] = ((t[1] >> 11) | ((t[2] & 0x3f) << 2)) as u8;
            r[(13 * i + 4) as usize] = ((t[2] >> 6) | ((t[3] & 0x01) << 7)) as u8;
            r[(13 * i + 5) as usize] = ((t[3] >> 1) & 0xff) as u8;
            r[(13 * i + 6) as usize] = ((t[3] >> 9) | ((t[4] & 0x0f) << 4)) as u8;
        }
        2 => {
            r[(13 * i) as usize] = (t[0] & 0xff) as u8;
            r[(13 * i + 1) as usize] = ((t[0] >> 8) | ((t[1] & 0x07) << 5)) as u8;
            r[(13 * i + 2) as usize] = ((t[1] >> 3) & 0xff) as u8;
            r[(13 * i + 3) as usize] = ((t[1] >> 11) | ((t[2] & 0x3f) << 2)) as u8;
        }
        _ => {}
    }
}

#[cfg(any(feature = "ntruhps2048509", feature = "ntruhps2048677"))]
pub fn poly_sq_frombytes(r: &mut Poly, a: &[u8]) {
    for i in 0..(NTRU_PACK_DEG / 8) {
        r.coeffs[8 * i] = a[11 * i] as u16 | ((a[11 * i + 1] as u16 & 0x07) << 8);
        r.coeffs[8 * i + 1] = (a[11 * i + 1] >> 3) as u16 | ((a[11 * i + 2] as u16 & 0x3f) << 5);
        r.coeffs[8 * i + 2] = (a[11 * i + 2] as u16 >> 6)
            | ((a[11 * i + 3] as u16) << 2)
            | ((a[11 * i + 4] as u16 & 0x01) << 10);
        r.coeffs[8 * i + 3] = (a[11 * i + 4] as u16 >> 1) | ((a[11 * i + 5] as u16 & 0x0f) << 7);
        r.coeffs[8 * i + 4] = (a[11 * i + 5] as u16 >> 4) | ((a[11 * i + 6] as u16 & 0x7f) << 4);
        r.coeffs[8 * i + 5] = (a[11 * i + 6] as u16 >> 7)
            | ((a[11 * i + 7] as u16) << 1)
            | ((a[11 * i + 8] as u16 & 0x03) << 9);
        r.coeffs[8 * i + 6] = (a[11 * i + 8] as u16 >> 2) | ((a[11 * i + 9] as u16 & 0x1f) << 6);
        r.coeffs[8 * i + 7] = (a[11 * i + 9] as u16 >> 5) | ((a[11 * i + 10] as u16) << 3);
    }
    let i = NTRU_PACK_DEG / 8;
    match NTRU_PACK_DEG & 0x07 {
        // cases 0 and 6 are impossible since 2 generates (Z/n)* and
        // p mod 8 in {1, 7} implies that 2 is a quadratic residue.
        4 => {
            r.coeffs[8 * i] = (a[11 * i] as u16) | ((a[11 * i + 1] as u16 & 0x07) << 8);
            r.coeffs[8 * i + 1] =
                (a[11 * i + 1] as u16 >> 3) | ((a[11 * i + 2] as u16 & 0x3f) << 5);
            r.coeffs[8 * i + 2] = (a[11 * i + 2] as u16 >> 6)
                | ((a[11 * i + 3] as u16) << 2)
                | ((a[11 * i + 4] as u16 & 0x01) << 10);
            r.coeffs[8 * i + 3] =
                (a[11 * i + 4] as u16 >> 1) | ((a[11 * i + 5] as u16 & 0x0f) << 7);
        }
        2 => {
            r.coeffs[8 * i] = (a[11 * i] as u16) | ((a[11 * i + 1] as u16 & 0x07) << 8);
            r.coeffs[8 * i + 1] =
                (a[11 * i + 1] as u16 >> 3) | ((a[11 * i + 2] as u16 & 0x3f) << 5);
        }
        _ => {}
    }
    r.coeffs[NTRU_N - 1] = 0;
}

#[cfg(feature = "ntruhps4096821")]
pub fn poly_sq_frombytes(r: &mut Poly, a: &[u8]) {
    for i in 0..NTRU_PACK_DEG / 2 {
        r.coeffs[2 * i] = a[3 * i] as u16 | ((a[3 * i + 1] as u16 & 0x0f) << 8);
        r.coeffs[2 * i + 1] = ((a[3 * i + 1] as u16) >> 4) | ((a[3 * i + 2] as u16 & 0xff) << 4);
    }
    r.coeffs[NTRU_N - 1] = 0;
}

#[cfg(feature = "ntruhrss701")]
pub fn poly_sq_frombytes(r: &mut Poly, a: &[u8]) {
    for i in 0..(NTRU_PACK_DEG / 8) {
        r.coeffs[8 * i] = (a[13 * i] as u16 | ((a[13 * i + 1] as u16 & 0x1f) << 8)) as u16;
        r.coeffs[8 * i + 1] = ((a[13 * i + 1] as u16 >> 5)
            | ((a[13 * i + 2] as u16) << 3)
            | ((a[13 * i + 3] as u16 & 0x03) << 11)) as u16;
        r.coeffs[8 * i + 2] =
            ((a[13 * i + 3] as u16 >> 2) | ((a[13 * i + 4] as u16 & 0x7f) << 6)) as u16;
        r.coeffs[8 * i + 3] = ((a[13 * i + 4] as u16 >> 7)
            | ((a[13 * i + 5] as u16) << 1)
            | ((a[13 * i + 6] as u16 & 0x0f) << 9)) as u16;
        r.coeffs[8 * i + 4] = ((a[13 * i + 6] as u16 >> 4)
            | ((a[13 * i + 7] as u16) << 4)
            | ((a[13 * i + 8] as u16 & 0x01) << 12)) as u16;
        r.coeffs[8 * i + 5] =
            ((a[13 * i + 8] as u16 >> 1) | ((a[13 * i + 9] as u16 & 0x3f) << 7)) as u16;
        r.coeffs[8 * i + 6] = ((a[13 * i + 9] as u16 >> 6)
            | ((a[13 * i + 10] as u16) << 2)
            | ((a[13 * i + 11] as u16 & 0x07) << 10)) as u16;
        r.coeffs[8 * i + 7] =
            ((a[13 * i + 11] as u16 >> 3) | ((a[13 * i + 12] as u16) << 5)) as u16;
    }
    let i = NTRU_PACK_DEG / 8;
    match NTRU_PACK_DEG & 0x07 {
        // cases 0 and 6 are impossible since 2 generates (Z/n)* and
        // p mod 8 in {1, 7} implies that 2 is a quadratic residue.
        4 => {
            r.coeffs[8 * i] = (a[13 * i] as u16 | ((a[13 * i + 1] as u16 & 0x1f) << 8)) as u16;
            r.coeffs[8 * i + 1] = ((a[13 * i + 1] as u16 >> 5)
                | ((a[13 * i + 2] as u16) << 3)
                | ((a[13 * i + 3] as u16 & 0x03) << 11)) as u16;
            r.coeffs[8 * i + 2] =
                ((a[13 * i + 3] as u16 >> 2) | ((a[13 * i + 4] as u16 & 0x7f) << 6)) as u16;
            r.coeffs[8 * i + 3] = ((a[13 * i + 4] as u16 >> 7)
                | ((a[13 * i + 5] as u16) << 1)
                | ((a[13 * i + 6] as u16 & 0x0f) << 9)) as u16;
        }
        2 => {
            r.coeffs[8 * i] = (a[13 * i] as u16 | ((a[13 * i + 1] as u16 & 0x1f) << 8)) as u16;
            r.coeffs[8 * i + 1] = ((a[13 * i + 1] as u16 >> 5)
                | ((a[13 * i + 2] as u16) << 3)
                | ((a[13 * i + 3] as u16 & 0x03) << 11)) as u16;
        }
        _ => {}
    }
    r.coeffs[NTRU_N - 1] = 0;
}

pub fn poly_rq_sum_zero_tobytes(r: &mut [u8], a: &mut Poly) {
    poly_sq_tobytes(r, a);
}

pub fn poly_rq_sum_zero_frombytes(r: &mut Poly, a: &[u8]) {
    poly_sq_frombytes(r, a);

    /* Set r[n-1] so that the sum of coefficients is zero mod q */
    r.coeffs[NTRU_N - 1] = 0;
    for i in 0..NTRU_PACK_DEG {
        r.coeffs[NTRU_N - 1] = r.coeffs[NTRU_N - 1].wrapping_sub(r.coeffs[i]);
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_poly_sq_tobytes() {
        #[cfg(feature = "ntruhps2048509")]
        {
            let mut a = Poly::new();
            a.coeffs = [7; NTRU_N];
            let mut r = [0u8; 1 + 11 * NTRU_PACK_DEG / 8];
            poly_sq_tobytes(&mut r, &a);
            let expected_r = [
                0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0,
                0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70,
                0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C,
                0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07,
                0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01,
                0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80,
                0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0,
                0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38,
                0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E,
                0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03,
                0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00,
                0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0,
                0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70,
                0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C,
                0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07,
                0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01,
                0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80,
                0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0,
                0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38,
                0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E,
                0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03,
                0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00,
                0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0,
                0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70,
                0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C,
                0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07,
                0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01,
                0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80,
                0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0,
                0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38,
                0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E,
                0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03,
                0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00,
                0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0,
                0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70,
                0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C,
                0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07,
                0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01,
                0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80,
                0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0,
                0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38,
                0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E,
                0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03,
                0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00,
                0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0,
                0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70,
                0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C,
                0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07,
                0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01,
                0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x00,
            ];
            assert_eq!(r, expected_r);
        }

        #[cfg(feature = "ntruhps2048677")]
        {
            let mut a = Poly::new();
            a.coeffs = [7; NTRU_N];
            let mut r = [0u8; 1 + 11 * NTRU_PACK_DEG / 8];
            poly_sq_tobytes(&mut r, &a);
            let expected_r = [
                0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0,
                0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70,
                0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C,
                0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07,
                0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01,
                0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80,
                0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0,
                0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38,
                0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E,
                0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03,
                0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00,
                0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0,
                0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70,
                0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C,
                0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07,
                0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01,
                0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80,
                0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0,
                0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38,
                0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E,
                0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03,
                0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00,
                0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0,
                0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70,
                0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C,
                0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07,
                0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01,
                0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80,
                0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0,
                0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38,
                0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E,
                0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03,
                0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00,
                0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0,
                0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70,
                0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C,
                0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07,
                0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01,
                0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80,
                0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0,
                0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38,
                0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E,
                0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03,
                0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00,
                0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0,
                0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70,
                0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C,
                0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07,
                0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01,
                0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80,
                0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0,
                0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38,
                0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E,
                0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03,
                0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00,
                0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0,
                0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70,
                0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C,
                0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07,
                0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01,
                0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80,
                0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0,
                0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38,
                0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E,
                0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03,
                0x1C, 0xE0, 0x00, 0x07, 0x38, 0xC0, 0x01, 0x0E, 0x70, 0x80, 0x03, 0x1C, 0xE0, 0x00,
                0x07, 0x38, 0xC0, 0x01, 0x0E, 0x00,
            ];
            assert_eq!(r, expected_r);
        }

        #[cfg(feature = "ntruhps4096821")]
        {
            let mut a = Poly::new();
            a.coeffs = [7; NTRU_N];
            let mut r = [0u8; 3 * NTRU_PACK_DEG / 2];
            poly_sq_tobytes(&mut r, &a);
            let expected_r = [
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70,
                0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07,
                0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
                0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00, 0x07, 0x70, 0x00,
            ];
            assert_eq!(r, expected_r);
        }

        #[cfg(feature = "ntruhrss701")]
        {
            let mut a = Poly::new();
            // unlike the previous testcases, we use some non-constant value here
            a.coeffs = [
                0x724F, 0xACE5, 0x13B9, 0x28EB, 0xC443, 0x5CD0, 0xA106, 0x0EFB, 0x5F30, 0x049A,
                0xDB4E, 0x7E7F, 0xD304, 0xD47A, 0x7283, 0x83F7, 0x4D41, 0xFCC5, 0x0C45, 0xB8A5,
                0x29E8, 0xC643, 0xFBFF, 0x6DE9, 0xF36B, 0x8B26, 0xDAF0, 0x1F45, 0x2F7A, 0xA1EA,
                0x7CD6, 0x23FC, 0xDBAC, 0x5FF4, 0x0156, 0xFFD3, 0x319F, 0x3359, 0x483B, 0xA6B5,
                0x6817, 0x5C4F, 0xBD04, 0x2039, 0x110D, 0xD5A4, 0xDA70, 0x61A0, 0x9A89, 0xCEFD,
                0xA969, 0x3CF6, 0xDA71, 0x8F3F, 0x98C8, 0x2595, 0x4E2F, 0xED55, 0x81ED, 0xC15B,
                0x55F3, 0x6CDD, 0xF25A, 0x9D03, 0x398D, 0xE6D9, 0xC369, 0xEB17, 0x398C, 0xBEF7,
                0xDF20, 0x1D97, 0x7EE9, 0xE33B, 0x06CC, 0xD19C, 0x01E0, 0x777C, 0x594B, 0xFE43,
                0xA73F, 0x0057, 0x80E6, 0x0697, 0x7E06, 0xDA10, 0xD45A, 0x6343, 0x109C, 0x381E,
                0x0030, 0x33E4, 0x6BEE, 0x5EF8, 0xE763, 0xB228, 0xD953, 0x9A18, 0x24D8, 0x30FE,
                0xC812, 0x9102, 0x4B55, 0xA2C2, 0xC21C, 0xEFF4, 0xB57E, 0xFD83, 0x9881, 0x0592,
                0x9C00, 0xCA37, 0x4E20, 0x8011, 0xA40D, 0xC8F0, 0x70FB, 0x3DC4, 0x8B96, 0x717D,
                0xB3F0, 0x5160, 0x9EDB, 0x95AF, 0x72C5, 0x3413, 0x05B4, 0x68DE, 0x6234, 0xE2AA,
                0x51C8, 0x0D7F, 0x2036, 0xBC53, 0xE416, 0x767A, 0x5074, 0x0F14, 0xF47C, 0x08C8,
                0xAD2B, 0x00BB, 0xBAD3, 0xA53B, 0x997F, 0x63D9, 0xE092, 0xB0EA, 0x1E3B, 0x444C,
                0x769C, 0xFCB0, 0xDBB4, 0x014C, 0x1CD9, 0xA5B4, 0x8276, 0x5686, 0x2428, 0x8DA7,
                0x89B1, 0x6C6C, 0x8286, 0xBA08, 0x7043, 0x4796, 0x3F5D, 0xE9DF, 0x70DB, 0x6234,
                0xB85A, 0xCB64, 0x190C, 0x0A04, 0x0164, 0x845A, 0x82F9, 0xB31C, 0x4F05, 0x7BD1,
                0x15AE, 0x1B2B, 0x2E75, 0x5DF6, 0x7BB5, 0x8D90, 0x373F, 0xF079, 0x1827, 0x7E69,
                0x20E3, 0xB86C, 0xE138, 0x1B37, 0x3F12, 0x879A, 0xCAC9, 0x8F65, 0x72E5, 0x25A6,
                0x91BD, 0x285F, 0xE6C7, 0xE6A7, 0xA9ED, 0xC8FC, 0x51F6, 0xFB73, 0x820F, 0x1979,
                0xAC9A, 0x2F79, 0x0EDB, 0x4C28, 0x58A2, 0x9E68, 0xA49F, 0x5C35, 0xE788, 0x9CE7,
                0xB488, 0xE872, 0x1606, 0x1897, 0x6564, 0x467C, 0xAE19, 0xC1A7, 0x9CDA, 0xA18E,
                0x1835, 0x6D64, 0x491E, 0x66D4, 0xD030, 0x3A17, 0x1E18, 0x454E, 0x8626, 0x7780,
                0x0565, 0x7FE0, 0x48F5, 0xF109, 0xAC0A, 0x7222, 0xB4A5, 0xD3D1, 0xC227, 0x0862,
                0xC5DB, 0x8E63, 0x0B67, 0xE5E0, 0x3A3C, 0x8509, 0xBBDF, 0xA33B, 0x4BD4, 0x5E17,
                0xD667, 0x33CB, 0x3820, 0x3B40, 0x0680, 0x4A3B, 0x60A4, 0x39BB, 0x386B, 0x8BB2,
                0xC6A8, 0x08A9, 0x8389, 0xBC2D, 0xF8DE, 0x6148, 0x2460, 0xFF39, 0xA8B6, 0x420A,
                0x0252, 0x763C, 0xD814, 0x5FAB, 0xA1C5, 0x14AD, 0x3288, 0x6B66, 0x9BE5, 0x71C8,
                0xBB46, 0xBFB2, 0xD4B6, 0x5A7D, 0x8BBC, 0x2DC1, 0x4143, 0xBA60, 0x9C5B, 0x365C,
                0x9533, 0x3B7A, 0x002B, 0x64AA, 0x8802, 0x9882, 0x17B2, 0x5803, 0x32F2, 0xCDFD,
                0xFBD2, 0xA970, 0xBAEC, 0x5359, 0xC207, 0xD5AC, 0x7CFD, 0x9F5E, 0x0553, 0x7B19,
                0x0E75, 0xFC46, 0x523C, 0xD91E, 0x6FB2, 0xBD5D, 0xACE3, 0xC806, 0xD562, 0xD0B0,
                0xC19E, 0xF5BB, 0xC6B4, 0xF7E0, 0x930B, 0xD97C, 0x286F, 0xC379, 0x59FD, 0xC61E,
                0x9CA2, 0x2546, 0x1EDE, 0xEAB1, 0x8F00, 0x20A0, 0x0A8E, 0xC259, 0xF19A, 0x1C93,
                0x55FE, 0x93BA, 0xC477, 0xEA5E, 0x09AF, 0xFD6F, 0x7D9A, 0xFC37, 0x6BDA, 0xA4AD,
                0x5D7A, 0xD881, 0xF307, 0xA9C0, 0x0D92, 0x2EC7, 0x7FC6, 0x390A, 0x0697, 0x8124,
                0x6E32, 0x90EC, 0x9545, 0x8E9E, 0xB65A, 0x76A8, 0x127B, 0x91EC, 0x9C38, 0x3E5A,
                0x0B14, 0xBA00, 0x0829, 0xF770, 0x3D21, 0x4C5D, 0xFC5E, 0xB29F, 0xC557, 0x4734,
                0x3125, 0x9D98, 0x8B7E, 0xB393, 0x7C68, 0x1EE7, 0xBAA6, 0x760F, 0x2CEF, 0xF676,
                0x7A1D, 0x0FF6, 0xF6F6, 0x803C, 0xBA75, 0x5609, 0xC003, 0xBAAA, 0x9BA9, 0xCCA2,
                0x0AA2, 0x2F9F, 0x9847, 0xA3BD, 0xAF46, 0xBFF9, 0x69AE, 0x107D, 0x9C06, 0xFDDD,
                0x22A5, 0xFAD6, 0x5B62, 0x3361, 0xC2F1, 0x8E52, 0x69EE, 0x8044, 0x8EDB, 0x1F39,
                0xACC6, 0xEB90, 0x909F, 0x767E, 0xC305, 0x4028, 0x1510, 0x8E35, 0x9E07, 0x8F95,
                0xF0C7, 0x80A0, 0x701A, 0x81F4, 0x156E, 0x84D3, 0x5A3E, 0xFFFF, 0x99D8, 0xC5B0,
                0x2F93, 0xE85C, 0x46D9, 0xFA15, 0x38E2, 0x6620, 0x17D9, 0x1C74, 0x1876, 0x2016,
                0x691E, 0x4DC7, 0xA5F3, 0x6946, 0xD76F, 0xB8A4, 0x2538, 0x5BBF, 0xF2DD, 0xA3F3,
                0x8223, 0x9CD6, 0xDAB0, 0x3AB8, 0x91BE, 0x9FA0, 0xD27C, 0x5B7D, 0xB7F7, 0xB3E8,
                0x67BF, 0x27F6, 0xB03E, 0xF1A0, 0x763B, 0xB776, 0x03C5, 0x743C, 0x5EA9, 0xCE81,
                0xA933, 0xEA64, 0x85EC, 0x8D46, 0xBAB7, 0xD97E, 0x7131, 0xFF8D, 0x2AE1, 0x33BC,
                0xF033, 0x4EBF, 0x92F1, 0xB992, 0xFADE, 0xFA1B, 0x0FEA, 0x4513, 0xD290, 0xE50F,
                0x4FC3, 0x3ACC, 0xE1C4, 0xB386, 0xBD94, 0x6C75, 0x4AD0, 0x3744, 0x0F6A, 0x17CB,
                0x9BCE, 0x0C8F, 0x4913, 0x0D95, 0x10FB, 0x9499, 0xF29F, 0xD135, 0x39EA, 0x7D06,
                0x0C37, 0xDC75, 0x9EAF, 0xD522, 0xF7DD, 0x10CA, 0x1349, 0x19F0, 0x019D, 0x905F,
                0xF762, 0x7151, 0xB47F, 0xEF56, 0xE9B2, 0x11F9, 0xBFB2, 0xFFED, 0xF71B, 0x09F2,
                0xA821, 0x1DBF, 0xE272, 0x98B0, 0xD9AE, 0x31AA, 0x091A, 0x71B9, 0xDE17, 0x9630,
                0x3DB3, 0xA03A, 0xB6FE, 0x2955, 0x7705, 0xD37B, 0x270C, 0x786C, 0x0700, 0x1DCE,
                0x0A99, 0xD3D2, 0x7F2B, 0x076E, 0x9DEA, 0xA3C2, 0x2738, 0x06D8, 0x575D, 0x07BC,
                0x54A4, 0x3061, 0xB580, 0xA4F9, 0x46F4, 0xB7C2, 0x16EA, 0x4565, 0xF34C, 0x2ECB,
                0x8109, 0x3186, 0x3CE3, 0xB31B, 0x29DC, 0x8FF8, 0xB97F, 0x663D, 0xE804, 0x8EE8,
                0x0F2A, 0x7E05, 0x99D1, 0x910A, 0x5080, 0xCFB5, 0xB7A9, 0xF2A0, 0xDD06, 0x4B36,
                0x9840, 0x74A5, 0x0CDD, 0x7916, 0x3CD7, 0x65F6, 0x47FF, 0x45A8, 0x9D02, 0x67E4,
                0x9125, 0x4DAA, 0xD8A0, 0xFB7E, 0x321E, 0xF284, 0xC787, 0xB208, 0x8A85, 0x4A95,
                0x4537, 0x399E, 0x2790, 0x1FFB, 0xC5A8, 0xD9A6, 0xBCE1, 0xC3B3, 0x7991, 0x1EA7,
                0xCB14, 0x8365, 0x5841, 0x3112, 0x21ED, 0xB702, 0xFDF8, 0x1931, 0x71F2, 0x2C86,
                0xA2C2, 0xA83E, 0x57C4, 0x8347, 0x79CA, 0xAB46, 0xD20F, 0xD41A, 0x05E2, 0x4F0A,
                0x3B03, 0xC5C8, 0x40AA, 0x79C5, 0x9A1F, 0xC901, 0xE55C, 0xBEC6, 0x05A1, 0xC9CE,
                0x5339, 0xA8BD, 0x6633, 0x3EE0, 0x2AC0, 0xBE0F, 0x4B68, 0x3FB9, 0x3E6A, 0x8CB0,
                0x2E36, 0xC382, 0x13E4, 0x843C, 0x7E5D, 0x37A1, 0xAD78, 0x9073, 0x47C5, 0xBDAE,
                0x57E9, 0xA68F, 0xEB5D, 0x18F1, 0x5702, 0x9482, 0x92E6, 0x3973, 0xC733, 0xED15,
                0x5252,
            ];
            let mut r = [0u8; 1138];
            poly_sq_tobytes(&mut r, &a);
            let expected_r = [
                0x4F, 0xB2, 0x9C, 0xE5, 0xCE, 0x75, 0x34, 0x44, 0xA0, 0xB9, 0x41, 0xD8, 0x77, 0x30,
                0x5F, 0x93, 0x38, 0xED, 0x3F, 0x4F, 0x30, 0xF5, 0xE8, 0xA0, 0xBC, 0x1F, 0x41, 0xAD,
                0x98, 0x17, 0xB1, 0x52, 0x8C, 0x9E, 0x86, 0xCC, 0xFF, 0x4E, 0x6F, 0x6B, 0xD3, 0x64,
                0xC1, 0xEB, 0xA2, 0xAF, 0xF7, 0xD4, 0x83, 0x35, 0xE7, 0x1F, 0xAC, 0x9B, 0xFE, 0x5B,
                0x85, 0xE9, 0xFF, 0x19, 0xB3, 0xE6, 0x0E, 0xAA, 0x35, 0x17, 0xE8, 0x89, 0x13, 0xF4,
                0x1C, 0xD0, 0x10, 0x49, 0x2B, 0x9C, 0x06, 0x0D, 0x89, 0xBA, 0xDF, 0xA5, 0x25, 0x7B,
                0x1E, 0xA7, 0x7F, 0x1E, 0x32, 0xAE, 0x2C, 0x2F, 0xAE, 0xAA, 0xB5, 0x87, 0xAD, 0x30,
                0x5F, 0xBB, 0x99, 0x96, 0x1C, 0xE8, 0x8D, 0x39, 0xDB, 0xA4, 0x8D, 0x8B, 0xC5, 0x98,
                0xEF, 0x3D, 0xC8, 0xBF, 0xEC, 0xE9, 0x7E, 0x67, 0x30, 0x1B, 0xCE, 0x08, 0x1E, 0xF8,
                0xEE, 0x52, 0x1E, 0xF2, 0x3F, 0xE7, 0x0A, 0x98, 0x83, 0x4B, 0x63, 0xE0, 0x21, 0xB4,
                0x16, 0x1D, 0x1A, 0x9C, 0xD0, 0x03, 0xC3, 0x00, 0xF2, 0xE9, 0xBE, 0xF0, 0xFD, 0xD8,
                0x41, 0x91, 0x53, 0x19, 0x43, 0x63, 0x13, 0x7F, 0x28, 0x81, 0x04, 0x62, 0xD5, 0x12,
                0x16, 0x1C, 0x82, 0xFE, 0xF9, 0xD5, 0xC1, 0x1E, 0x88, 0x25, 0x0B, 0x00, 0xBF, 0x51,
                0x20, 0x2E, 0x02, 0x34, 0x10, 0x78, 0xB4, 0x0F, 0x89, 0xBB, 0xE5, 0xEA, 0x8B, 0xF0,
                0x13, 0x2C, 0x6E, 0xFB, 0xD7, 0x5A, 0x2C, 0x27, 0x28, 0x6D, 0xF1, 0x46, 0x34, 0x42,
                0x55, 0x20, 0xC7, 0xBF, 0x66, 0x03, 0xA6, 0xB8, 0x05, 0xD1, 0xB3, 0x74, 0x90, 0xE2,
                0xF1, 0x51, 0x64, 0xB4, 0xD2, 0x76, 0xC1, 0xB4, 0xDE, 0x29, 0x7F, 0x39, 0x7B, 0x48,
                0x02, 0x75, 0xB8, 0xE3, 0x99, 0x08, 0xA7, 0x85, 0xE5, 0xB4, 0x9B, 0x29, 0x64, 0x73,
                0xDA, 0x62, 0x27, 0x0C, 0x2D, 0x0A, 0x39, 0x6D, 0xB1, 0x89, 0x8D, 0x19, 0x0A, 0x04,
                0x3D, 0x04, 0x2D, 0x4F, 0xD7, 0xFF, 0x4E, 0xDB, 0x90, 0x46, 0x68, 0x61, 0xB2, 0xC5,
                0x90, 0x09, 0x14, 0x59, 0xD0, 0x22, 0xF9, 0x82, 0x63, 0x16, 0xBC, 0xE8, 0xED, 0x5A,
                0x57, 0x76, 0x9D, 0xB3, 0xEF, 0xB5, 0x1B, 0xB2, 0xFD, 0xDC, 0x3C, 0x78, 0x82, 0xD3,
                0xFC, 0x38, 0x60, 0xC3, 0x38, 0xE1, 0x66, 0x4B, 0x7C, 0xCD, 0x93, 0xAC, 0xCA, 0x5E,
                0xB9, 0x34, 0x2D, 0xBD, 0xF1, 0x0B, 0x1D, 0x9B, 0x53, 0xD3, 0x9E, 0xF8, 0x91, 0x7D,
                0x9C, 0xDB, 0x0F, 0x22, 0x2F, 0x6B, 0xB2, 0xBC, 0xB7, 0xED, 0x50, 0x98, 0x28, 0x46,
                0xF3, 0x9F, 0xA4, 0x86, 0x23, 0x9E, 0x73, 0x8E, 0x48, 0xE5, 0x90, 0x81, 0xBD, 0xC4,
                0x64, 0x85, 0xCF, 0x64, 0xB8, 0xD3, 0xA0, 0xCD, 0x1D, 0x43, 0x0D, 0x26, 0x6B, 0x1E,
                0x89, 0xDA, 0xC0, 0xC0, 0x0B, 0x8D, 0xE1, 0x9D, 0x8A, 0x89, 0x01, 0xBC, 0x65, 0x05,
                0xFC, 0xD7, 0xA3, 0x84, 0xA8, 0xC0, 0x44, 0x64, 0x29, 0x8D, 0x9E, 0x27, 0x42, 0x0C,
                0x6D, 0x97, 0x31, 0x77, 0xB6, 0xC0, 0x0B, 0x8F, 0x4E, 0x28, 0xDF, 0x7B, 0x67, 0x50,
                0xAF, 0x0B, 0x7F, 0x66, 0x97, 0x27, 0x08, 0x06, 0xDA, 0x80, 0x66, 0x47, 0x91, 0x82,
                0xDD, 0xBC, 0x86, 0x65, 0x17, 0xAA, 0x49, 0x45, 0x89, 0xA3, 0x85, 0x7B, 0x63, 0xA4,
                0x00, 0x46, 0x72, 0xBE, 0x2D, 0x52, 0x10, 0x52, 0x82, 0xC7, 0x52, 0xE0, 0xD5, 0x5F,
                0x1C, 0x5A, 0x29, 0xA2, 0x34, 0x5B, 0xE5, 0x1B, 0x39, 0x1A, 0x6D, 0xD9, 0x6F, 0x4B,
                0xFB, 0x34, 0xEF, 0x0A, 0x6E, 0x43, 0x01, 0x4C, 0x6F, 0x71, 0x2E, 0x3B, 0x53, 0xF5,
                0xF6, 0x0A, 0x50, 0x25, 0x02, 0x48, 0x10, 0xCB, 0xDE, 0x01, 0x2C, 0x2F, 0xFB, 0x9B,
                0xF4, 0x86, 0x4B, 0xEC, 0x3A, 0x6B, 0x1E, 0x08, 0xD6, 0xDA, 0xCF, 0xBD, 0xFE, 0x54,
                0xC9, 0xD8, 0x75, 0xCE, 0x88, 0xF3, 0x48, 0x8F, 0x2C, 0xFB, 0xBA, 0xFA, 0x38, 0x33,
                0x40, 0x62, 0x15, 0x16, 0x7A, 0x86, 0xDD, 0x4A, 0x6B, 0xC0, 0xEF, 0xC2, 0xE4, 0xCB,
                0x6F, 0x28, 0x6F, 0xF4, 0x67, 0x0F, 0x23, 0xCA, 0x8D, 0x8A, 0xB7, 0x8F, 0x55, 0x00,
                0x0F, 0x14, 0x38, 0xAA, 0x2C, 0xA1, 0x19, 0x27, 0xB9, 0x7F, 0xD5, 0x9D, 0x77, 0xC4,
                0x4B, 0xBD, 0xA6, 0xB7, 0xAE, 0xD9, 0x6F, 0xB8, 0xF6, 0x6A, 0x25, 0x7A, 0x3D, 0x10,
                0x1F, 0x4C, 0xE0, 0x24, 0xD9, 0x8E, 0x9D, 0xF1, 0x57, 0xC8, 0x97, 0x86, 0x24, 0xC8,
                0x38, 0x76, 0x58, 0x54, 0x3D, 0x9D, 0x96, 0x45, 0xB5, 0x7B, 0x92, 0x3D, 0xE2, 0x70,
                0x2D, 0x4F, 0xB1, 0x00, 0x74, 0x0A, 0x82, 0xBB, 0x21, 0xBD, 0x8B, 0x79, 0xF1, 0x4F,
                0x79, 0x55, 0x68, 0x4E, 0x49, 0xC4, 0xEC, 0x7E, 0x6B, 0x72, 0xA2, 0xF1, 0x73, 0x6F,
                0xAA, 0x1F, 0xEC, 0x3B, 0xB3, 0xB3, 0x1D, 0xDA, 0xFE, 0xD9, 0x5B, 0x1E, 0x50, 0xA7,
                0x13, 0xEC, 0x00, 0x50, 0xD5, 0xA9, 0x5B, 0x94, 0x89, 0xAA, 0xCF, 0x77, 0x84, 0x7B,
                0x87, 0xD1, 0xCB, 0xFF, 0xAE, 0xA9, 0x0F, 0x1A, 0xF0, 0xEE, 0x5E, 0x2A, 0xAC, 0xB5,
                0xD8, 0x0E, 0x9B, 0xF1, 0x42, 0xCA, 0xB9, 0x27, 0x22, 0xB0, 0xED, 0x72, 0xBE, 0x31,
                0x83, 0x5C, 0x9F, 0xD0, 0xCF, 0x16, 0x0C, 0x14, 0x00, 0x51, 0x6B, 0xDC, 0x81, 0xAF,
                0x7C, 0xC7, 0x10, 0x14, 0x68, 0x40, 0xFA, 0xE0, 0x56, 0xA7, 0x89, 0x8F, 0xFE, 0xFF,
                0xD8, 0x19, 0xB6, 0x4C, 0x3E, 0x2E, 0x94, 0x6D, 0x2A, 0xB4, 0x38, 0x06, 0x31, 0xD9,
                0x97, 0x8E, 0xDB, 0x61, 0x0B, 0xE0, 0x91, 0x8E, 0xDB, 0x7C, 0x31, 0x4A, 0x6F, 0x97,
                0x14, 0xE3, 0x94, 0xDF, 0xDD, 0x2D, 0xE7, 0xC7, 0x88, 0xB0, 0xE6, 0xB0, 0x1A, 0x57,
                0xFB, 0x46, 0xD0, 0xCF, 0x27, 0xFB, 0xF6, 0xFD, 0x45, 0x9F, 0xBF, 0xC7, 0xFE, 0xF8,
                0x40, 0xD0, 0xB8, 0x63, 0xED, 0x6E, 0xF1, 0xE0, 0xA1, 0xA9, 0x3E, 0xD0, 0xCD, 0x24,
                0x32, 0xC5, 0x5E, 0x8C, 0xDA, 0xAD, 0xF6, 0xCB, 0x31, 0xB1, 0xF1, 0x87, 0x2B, 0xDE,
                0x39, 0x03, 0x7F, 0x5D, 0xBC, 0x94, 0xCC, 0xDE, 0x7A, 0x43, 0xAB, 0xBF, 0x89, 0x02,
                0x29, 0x1F, 0xCA, 0xF0, 0x63, 0xD6, 0xC4, 0xC1, 0x70, 0x52, 0xF6, 0x3A, 0x06, 0xAD,
                0x88, 0xAE, 0xDA, 0x5B, 0xBE, 0xCE, 0xFB, 0x91, 0x4D, 0xA4, 0xCA, 0xB6, 0x0F, 0x33,
                0xE9, 0xA7, 0xAC, 0x89, 0xEA, 0xD9, 0xA0, 0xDF, 0xB0, 0x3A, 0xFE, 0xEA, 0x45, 0x6A,
                0xF7, 0x55, 0x86, 0x49, 0x13, 0x3E, 0x77, 0x86, 0x2F, 0x28, 0x76, 0xA3, 0xE2, 0x1F,
                0xB5, 0x7A, 0xB2, 0x29, 0x3F, 0xCA, 0xFE, 0xF6, 0xBF, 0x71, 0xE5, 0x53, 0x08, 0xFA,
                0xED, 0x72, 0x02, 0x16, 0xBB, 0x66, 0xD5, 0xA8, 0x91, 0x72, 0xE3, 0x85, 0x87, 0xB1,
                0xB3, 0x5D, 0x07, 0xF8, 0xDB, 0xAA, 0x54, 0x70, 0xF7, 0x26, 0xC3, 0x61, 0xC3, 0x00,
                0xC7, 0xB9, 0x67, 0x2A, 0xE9, 0xB9, 0xF2, 0xDD, 0x8E, 0x7A, 0x17, 0x1E, 0x38, 0x07,
                0xDB, 0x74, 0x5D, 0xDE, 0x43, 0x4A, 0xC3, 0x20, 0x60, 0xCD, 0x27, 0xF4, 0x46, 0xF8,
                0xAA, 0xDB, 0xB2, 0xC2, 0x34, 0x97, 0x5D, 0x42, 0x30, 0x8C, 0xE3, 0x7C, 0x63, 0x72,
                0x27, 0xFC, 0xF7, 0x97, 0x7B, 0x0C, 0x01, 0x42, 0x77, 0x2A, 0xAF, 0xC0, 0x47, 0x67,
                0x85, 0x08, 0x08, 0x6B, 0x5F, 0xEA, 0x05, 0x95, 0x06, 0xDD, 0x66, 0x01, 0xE1, 0x52,
                0xDA, 0xCD, 0x2C, 0xF2, 0x35, 0xB7, 0x2F, 0xFF, 0x07, 0xB5, 0x08, 0x74, 0xF2, 0x53,
                0x12, 0x55, 0x1B, 0x28, 0xF6, 0xDB, 0x1E, 0x92, 0x50, 0x1E, 0x1E, 0x04, 0x59, 0xA8,
                0x2A, 0xD5, 0x4D, 0xF1, 0xCC, 0x90, 0x67, 0xFF, 0xA3, 0x16, 0xD3, 0x1C, 0xCE, 0x67,
                0x47, 0x64, 0x3E, 0xF5, 0x14, 0xAB, 0x6C, 0x04, 0x61, 0x89, 0xD8, 0x1E, 0x04, 0x2E,
                0x7E, 0x8F, 0xC9, 0xF2, 0xD1, 0x90, 0x09, 0x0B, 0x1F, 0x44, 0x7C, 0x8F, 0x86, 0x72,
                0x36, 0x5A, 0x0F, 0x52, 0x83, 0x8A, 0x17, 0x85, 0x37, 0xB0, 0x91, 0x8B, 0x2A, 0x28,
                0xCE, 0x1F, 0x3A, 0x20, 0x71, 0x15, 0x63, 0x1F, 0x5A, 0x9C, 0x53, 0xCE, 0xEC, 0x45,
                0x33, 0x06, 0xDC, 0x03, 0xAB, 0x07, 0x8F, 0xB6, 0x72, 0xBF, 0x9A, 0x87, 0x65, 0x36,
                0x4E, 0x70, 0x90, 0x4F, 0x1E, 0xD2, 0xE5, 0x43, 0x2F, 0x5E, 0x9B, 0x83, 0xC5, 0xC7,
                0xB5, 0xA7, 0xDF, 0x47, 0xD3, 0xB5, 0xE2, 0xB1, 0xC0, 0x15, 0xA4, 0xE6, 0x72, 0x2E,
                0xCF, 0x9C, 0x8A, 0x06,
            ];
            assert_eq!(r, expected_r);
        }
    }
}
