//! State transition types
use crate::constants::account_size::{
    DISTRIBUTE_TOKEN_PREFIX_SIZE, LIST_DISTRIBUTE_USER_SIZE, NUMBER_USERS_DISTRIBUTE,
    USER_DISTRIBUTE_SIZE,
};
use crate::constants::constant::PUBKEY_SIZE;
use crate::{
    constants::{account_size::DISTRIBUTE_TOKEN_SIZE, account_type::TYPE_ACCOUNT_AIRDROP_ACCOUNT},
    error::PortfolioError,
};
use arrayref::{array_mut_ref, array_ref, array_refs, mut_array_refs};
use serde::{Deserialize, Serialize};
use solana_program::{
    entrypoint_deprecated::ProgramResult,
    program_error::ProgramError,
    program_pack::{IsInitialized, Sealed},
    pubkey::Pubkey,
};
use std::vec::Vec;

const TYPE_SIZE: usize = 1;
const TOTAL_AMOUNT_SIZE: usize = 8;
const NONCE_SIZE: usize = 1;
const STATE_USER_DISTRIBUTE_SIZE: usize = 1;
const NFT_FIND_SIZE: usize = 1;

#[repr(C)]
#[derive(Clone, Debug, Default, PartialEq, Serialize, Deserialize)]

pub struct UserDistributeStruct {
    /// The public address owned by user .
    pub user_account: Pubkey,
    /// State of the user portfolio.
    pub state_user_distribute: u8,
    ///  The total amount deposit.
    pub amount: u64,
    /// _0. if he wasn't staked an nft token.
    /// _1. if he was staked an nft token.
    pub nft_find: u8,
}
impl UserDistributeStruct {
    /// add user distribute
    pub fn add_user_distribute(
        &mut self,
        user_portfolio_account: Pubkey,
        amount: u64,
        nft_find: u8,
        state_user_distribute: u8,
    ) -> ProgramResult {
        self.user_account = user_portfolio_account;
        self.amount = self
            .amount
            .checked_add(amount)
            .ok_or(PortfolioError::InvalidAmount)?;

        self.nft_find = nft_find;
        self.state_user_distribute = state_user_distribute;
        Ok(())
    }
    /// update the state of  user distribute
    pub fn update_state_user_distribute(&mut self, state_user_distribute: u8) -> ProgramResult {
        self.state_user_distribute = state_user_distribute;
        Ok(())
    }
}

///  distribute account data.
#[repr(C)]
#[derive(Clone, Debug, Default, PartialEq, Serialize, Deserialize)]
pub struct DistributeTokens {
    /// The type of account .
    pub type_account: u8,
    /// The portfolio depends of distribute token.
    pub portfolio_address: Pubkey,
    /// The amount fixed for this portfolio
    pub total_amount: u64,
    /// Authority of distribute account.
    pub authority: Pubkey,
    /// Nonce used in cross program.
    pub nonce: u8,
    /// Create account program used in cross program.
    pub create_account_program: Pubkey,
    /// List of [`UserDistributeStruct`]
    pub list_user_distribute: Vec<UserDistributeStruct>,
}

impl Sealed for DistributeTokens {}

impl IsInitialized for DistributeTokens {
    fn is_initialized(&self) -> bool {
        return self.type_account == TYPE_ACCOUNT_AIRDROP_ACCOUNT;
    }
}
/// pack and unpack for distribute token
pub trait PackDistributeTokens {
    /// unpack distribute token
    fn unpack_distribute_token(src: &[u8]) -> Result<DistributeTokens, ProgramError>;
    /// pack diestribute token
    fn pack_distribute_token(&self, dst: &mut [u8]);
}

impl PackDistributeTokens for DistributeTokens {
    /// unpack distribute token
    fn unpack_distribute_token(src: &[u8]) -> Result<Self, ProgramError> {
        let src_fix = array_ref![&src, 0, DISTRIBUTE_TOKEN_PREFIX_SIZE];
        let (
            type_account,
            portfolio_address,
            total_amount,
            authority,
            nonce,
            create_account_program,
        ) = array_refs![
            src_fix,
            TYPE_SIZE,
            PUBKEY_SIZE,
            TOTAL_AMOUNT_SIZE,
            PUBKEY_SIZE,
            NONCE_SIZE,
            PUBKEY_SIZE
        ];
        let mut user_distribute_vec: Vec<UserDistributeStruct> =
            Vec::with_capacity(NUMBER_USERS_DISTRIBUTE);

        // unpack the list of user distribure
        let list_user_distribute_data = &src[DISTRIBUTE_TOKEN_PREFIX_SIZE
            ..DISTRIBUTE_TOKEN_PREFIX_SIZE + LIST_DISTRIBUTE_USER_SIZE as usize];
        let mut offset = 0;
        for _ in 0..NUMBER_USERS_DISTRIBUTE {
            let user_data = array_ref![list_user_distribute_data, offset, USER_DISTRIBUTE_SIZE];
            #[allow(clippy::ptr_offset_with_cast)]
            let (user_account,state_user_distribute, amount, nft_find) = array_refs![
                user_data,
                PUBKEY_SIZE,
                STATE_USER_DISTRIBUTE_SIZE,
                TOTAL_AMOUNT_SIZE,
                NFT_FIND_SIZE
            ];
            user_distribute_vec.push(UserDistributeStruct {
                user_account: Pubkey::new_from_array(*user_account),
                state_user_distribute: u8::from_le_bytes(*state_user_distribute),
                amount: u64::from_le_bytes(*amount),
                nft_find: u8::from_le_bytes(*nft_find),
               
            });
            offset += USER_DISTRIBUTE_SIZE;
        }

        Ok(DistributeTokens {
            type_account: u8::from_le_bytes(*type_account),
            portfolio_address: Pubkey::new_from_array(*portfolio_address),
            total_amount: u64::from_le_bytes(*total_amount),
            authority: Pubkey::new_from_array(*authority),
            nonce: u8::from_le_bytes(*nonce),
            create_account_program: Pubkey::new_from_array(*create_account_program),
            list_user_distribute: user_distribute_vec.to_vec(),
        })
    }

    /// pack distribute token
    fn pack_distribute_token(&self, dst: &mut [u8]) {
        let dst_tmp = array_mut_ref![dst, 0, DISTRIBUTE_TOKEN_PREFIX_SIZE];
        let (
            type_account_dst,
            portfolio_address_dst,
            total_amount_dst,
            authority_dst,
            nonce_dst,
            create_account_program_dst,
        ) = mut_array_refs![
            dst_tmp,
            TYPE_SIZE,
            PUBKEY_SIZE,
            TOTAL_AMOUNT_SIZE,
            PUBKEY_SIZE,
            NONCE_SIZE,
            PUBKEY_SIZE
        ];

        let DistributeTokens {
            type_account,
            portfolio_address,
            total_amount,
            authority,
            nonce,
            create_account_program,
            list_user_distribute,
        } = self;

        let user_distribute_vec_tmp = bincode::serialize(&list_user_distribute).unwrap();
        let mut user_distribute_data_tmp = [0; LIST_DISTRIBUTE_USER_SIZE]; // maximum 100
        let len_user_distribute = user_distribute_vec_tmp.len();
        let nbre_user_distribute = len_user_distribute / USER_DISTRIBUTE_SIZE; //100
        let len_splu_reel = nbre_user_distribute * USER_DISTRIBUTE_SIZE;
        user_distribute_data_tmp[0..len_splu_reel as usize]
            .clone_from_slice(&user_distribute_vec_tmp[8..len_user_distribute]); // 0..8 : 8 bytes contain length of struct
        *type_account_dst = type_account.to_le_bytes();
        portfolio_address_dst.copy_from_slice(portfolio_address.as_ref());
        authority_dst.copy_from_slice(authority.as_ref());
        *total_amount_dst = total_amount.to_le_bytes();
        *nonce_dst = nonce.to_le_bytes();
        create_account_program_dst.copy_from_slice(create_account_program.as_ref());
        dst[DISTRIBUTE_TOKEN_PREFIX_SIZE..DISTRIBUTE_TOKEN_PREFIX_SIZE + len_splu_reel as usize]
            .copy_from_slice(&user_distribute_data_tmp[0..len_splu_reel]);
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::constants::account_type::TYPE_ACCOUNT_AIRDROP_ACCOUNT;
    #[test]
    fn test_pack_distribute_token() {
        let user_account = Pubkey::new_unique();
        let state_user_distribute = 0;
        let amount = 20;
        let nft_find = 1;

        let mut list_user_distribute = Vec::new();

        for _i in 0..NUMBER_USERS_DISTRIBUTE {
            let user_distribute = UserDistributeStruct {
                user_account,
                state_user_distribute,
                amount,
                nft_find
            };
            list_user_distribute.push(user_distribute);
        }

        let distribute_token = DistributeTokens {
            type_account: TYPE_ACCOUNT_AIRDROP_ACCOUNT,
            portfolio_address: Pubkey::new_unique(),
            total_amount: 200,
            authority: Pubkey::new_unique(),
            nonce: 254,
            create_account_program: Pubkey::new_unique(),
            list_user_distribute,
        };

        const LEN: usize = DISTRIBUTE_TOKEN_SIZE;
        let mut packed = [0u8; LEN];
        PackDistributeTokens::pack_distribute_token(&distribute_token, &mut packed[..]);
        let unpacked = DistributeTokens::unpack_distribute_token(&packed).unwrap();
        assert_eq!(distribute_token, unpacked);
        assert_eq!(unpacked.type_account, TYPE_ACCOUNT_AIRDROP_ACCOUNT);
    }
}
