//! State transition types
use crate::constants::account_type::TYPE_ACCOUNT_TRIGGUER_TURING_MASHINE_ACCOUNT;
use crate::constants::account_size::SPLU_TERTAIRY_STRUCT_LEN;
use crate::constants::account_size::TURING_MASHINE_ACCOUNT_PREFIX;
use solana_program::entrypoint::ProgramResult;
use arrayref::{array_mut_ref, mut_array_refs,array_ref, array_refs};
use serde::{Deserialize, Serialize};
use solana_program::{
    program_error::ProgramError,
    program_pack::{IsInitialized, Sealed},
    pubkey::Pubkey,
};

use crate::
    constants::constant::PUBKEY_SIZE;

const TYPE_SIZE: usize = 1;
const STATE_SIZE: usize = 1;
const AMOUNT_SIZE: usize = 8;
const VERSION_SIZE: usize = 1;

const STATE_SPLU_TERTIARY1: usize = 1;
const STATE_SPLU_TERTIARY2: usize = 1;

/// struct of SPLU of turing account
#[repr(C)]
#[derive(Clone, Debug, Default, PartialEq, Serialize, Deserialize)]
pub struct SpluStateTrigguerStruct {
    /// State of first SPLU tertiary.
    pub state_splu_tertiary1: u8,
    /// State of second SPLU tertiary.
    pub state_splu_tertiary2: u8,
}

impl SpluStateTrigguerStruct {
   
    /// reset the state of splu teriary
    pub fn reset_splu_tertiary(&mut self) -> ProgramResult {
        self.state_splu_tertiary1 = 0;
        self.state_splu_tertiary2 = 0;

        Ok(())
    }

    /// update the state splu teriary
    pub fn update_state_tertiary(&mut self, state1: u8, state2: u8) -> ProgramResult {
        self.state_splu_tertiary1 = state1;
        self.state_splu_tertiary2 = state2;

        Ok(())
    }
}



/// Account data.
#[repr(C)]
#[derive(Clone, Debug, Default, PartialEq, Serialize, Deserialize)]
pub struct TrigguerTuringMashineAccount {
    /// The type of account.
    pub type_account: u8,
    /// The owner of this account.
    pub owner: Pubkey,
    /// The user portfolio linked to this account.
    pub splm_n_asset: Pubkey,
    /// State of Turing account to know which step was completed.
    pub state: u8,
    /// Last amount deposited or withrawal.
    pub amount: u64,
    /// Version of Turing struct.
    pub version: u8,
    /// List of splu state contain the state of each splu after swap.
    pub splu_state_list: Vec<SpluStateTrigguerStruct>,
}

impl Sealed for TrigguerTuringMashineAccount {}

impl IsInitialized for TrigguerTuringMashineAccount {
    fn is_initialized(&self) -> bool {
        return self.type_account == TYPE_ACCOUNT_TRIGGUER_TURING_MASHINE_ACCOUNT;
    }
}
/// pack and unpack for turing trigguer account
pub trait PackTrigguerTuringMashineAccount {
    /// unpack turing Trigguer account
    fn unpack_trigguer_turing_machine_account(src: &[u8]) -> Result<TrigguerTuringMashineAccount, ProgramError>;
    /// pack turing Trigguer account
    fn pack_trigguer_turing_machine_account(&self, dst: &mut [u8]);
}
/// pack and unpack for turing trigger account.
impl PackTrigguerTuringMashineAccount for TrigguerTuringMashineAccount {
 
    // unpack trigguer turing machine account

        fn unpack_trigguer_turing_machine_account(src: &[u8]) -> Result<Self, ProgramError> {
            let numbre_splu_state = (&src.len() - TURING_MASHINE_ACCOUNT_PREFIX) / SPLU_TERTAIRY_STRUCT_LEN;
            let len_splu_data = &src.len() - TURING_MASHINE_ACCOUNT_PREFIX;
    
            let src_fix = array_ref![&src, 0, TURING_MASHINE_ACCOUNT_PREFIX];

        let (type_account, owner, splm_n_asset, state, amount, version) = array_refs![
            src_fix,
            TYPE_SIZE,
            PUBKEY_SIZE,
            PUBKEY_SIZE,
            STATE_SIZE,
            AMOUNT_SIZE,
            VERSION_SIZE
        ];


        let mut splu_vec: Vec<SpluStateTrigguerStruct> = Vec::with_capacity(numbre_splu_state);
        let list_splu_data =
            &src[TURING_MASHINE_ACCOUNT_PREFIX..TURING_MASHINE_ACCOUNT_PREFIX + (len_splu_data) as usize];

        // unpack list of splu struct
        let mut offset = 0;
        for _ in 0..numbre_splu_state {
            let splu_data = array_ref![list_splu_data, offset, SPLU_TERTAIRY_STRUCT_LEN];
            #[allow(clippy::ptr_offset_with_cast)]
            let (
                state_splu_tertiary1,
                state_splu_tertiary2,
            ) = array_refs![
                splu_data,
                STATE_SPLU_TERTIARY1,
                STATE_SPLU_TERTIARY2
            ];
            splu_vec.push(SpluStateTrigguerStruct {
                state_splu_tertiary1: u8::from_le_bytes(*state_splu_tertiary1),
                state_splu_tertiary2: u8::from_le_bytes(*state_splu_tertiary2),
            });
            offset += SPLU_TERTAIRY_STRUCT_LEN;
        }

        Ok(TrigguerTuringMashineAccount {
            type_account: u8::from_le_bytes(*type_account),
            owner: Pubkey::new_from_array(*owner),
            splm_n_asset: Pubkey::new_from_array(*splm_n_asset),
            state: u8::from_le_bytes(*state),
            amount: u64::from_le_bytes(*amount),
            version: u8::from_le_bytes(*version),
            splu_state_list: splu_vec.to_vec()
            })
    }

    // pack trigguer turing machine account
    fn pack_trigguer_turing_machine_account(&self, dst: &mut [u8]) {
        let dst_tmp = array_mut_ref![dst, 0, TURING_MASHINE_ACCOUNT_PREFIX];
        let (
            type_account_dst,
            owner_dst,
            splm_n_asset_dst,
            state_dst,
            amount_dst,
            version_dst,
        ) = mut_array_refs![
            dst_tmp,
            TYPE_SIZE,
            PUBKEY_SIZE,
            PUBKEY_SIZE,
            STATE_SIZE,
            AMOUNT_SIZE,
            VERSION_SIZE
        ];
        let TrigguerTuringMashineAccount {
            type_account,
            owner,
            splm_n_asset,
            state,
            amount,
            version,
            splu_state_list
        } = self;
      
        let splu_vec_tmp = bincode::serialize(&splu_state_list).unwrap();
    
        let len_splu = splu_vec_tmp.len();
        let nbre_splu = (len_splu-8) / SPLU_TERTAIRY_STRUCT_LEN;
        let len_slu_reel = nbre_splu * SPLU_TERTAIRY_STRUCT_LEN;
        let mut splu_data_tmp = [0; SPLU_TERTAIRY_STRUCT_LEN * 10]; // maximum 10 splu
        splu_data_tmp[0..len_slu_reel as usize].clone_from_slice(&splu_vec_tmp[8..len_splu]); // 0..8 : 8 bytes contain length of struct
      

        *type_account_dst = type_account.to_le_bytes();
        owner_dst.copy_from_slice(owner.as_ref());
        splm_n_asset_dst.copy_from_slice(splm_n_asset.as_ref());
        *state_dst =  state.to_le_bytes();
        *amount_dst = amount.to_le_bytes();
        *version_dst = version.to_le_bytes();
        dst[TURING_MASHINE_ACCOUNT_PREFIX..TURING_MASHINE_ACCOUNT_PREFIX+len_slu_reel as usize]
        .copy_from_slice(&splu_data_tmp[0..len_slu_reel]);
     
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::constants::account_type::TYPE_ACCOUNT_TRIGGUER_TURING_MASHINE_ACCOUNT;

    #[test]
    fn test_pack_turing_trigguer_account() {
        let type_account = TYPE_ACCOUNT_TRIGGUER_TURING_MASHINE_ACCOUNT;
        let owner = Pubkey::new_unique();
        let state = 0;
        let splm_n_asset = Pubkey::new_unique();
        let amount = 2;
        let version = 1;
        
        let splu_struct= SpluStateTrigguerStruct{
            state_splu_tertiary1:0,
            state_splu_tertiary2:0
        };

        let mut splu_state_list = Vec::new();
        splu_state_list.push(splu_struct);

        let turing = TrigguerTuringMashineAccount {
            type_account,
            owner,
            splm_n_asset,
            state,
            amount,
            version,
            splu_state_list
        };
        const LEN: usize = TURING_MASHINE_ACCOUNT_PREFIX + (SPLU_TERTAIRY_STRUCT_LEN * 1); // 204: @préfix + 164:splu_len * 1
        let mut packed = [0u8; LEN];
        TrigguerTuringMashineAccount::pack_trigguer_turing_machine_account(&turing, &mut packed[..]);
        let unpacked = TrigguerTuringMashineAccount::unpack_trigguer_turing_machine_account(&packed).unwrap();
        assert_eq!(turing, unpacked);
        assert_eq!(unpacked.type_account, TYPE_ACCOUNT_TRIGGUER_TURING_MASHINE_ACCOUNT);
    }
}
