#ifndef NSTD_GL_H_INCLUDED
#define NSTD_GL_H_INCLUDED
#include "core/def.h"
#include "core/slice.h"
#include "gui.h"
#include "nstd.h"
#include "string.h"
#ifdef NSTDCPP
extern "C"
{
#endif

/// Represents a color.
// Must match https://docs.rs/wgpu/0.12.0/wgpu/struct.Color.html.
typedef struct
{
    // Red color value.
    NSTDFloat64 r;
    // Green color value.
    NSTDFloat64 g;
    // Blue color value.
    NSTDFloat64 b;
    // Alpha color value.
    NSTDFloat64 a;
} NSTDGLColor;

/// Represents a graphical surface.
typedef NSTDAny NSTDGLSurface;

/// Represents a surface config.
typedef NSTDAny NSTDGLSurfaceConfiguration;

/// Represents a handle to a physical graphics device.
typedef NSTDAny NSTDGLDeviceHandle;

/// Represents a graphics device.
typedef NSTDAny NSTDGLDevice;

/// Represents a graphics device command queue.
typedef NSTDAny NSTDGLQueue;

/// Represents a shader module.
typedef NSTDAny NSTDGLShaderModule;

/// Represents a render pipeline.
typedef NSTDAny NSTDGLRenderPipeline;

/// Represents a render pass object.
typedef NSTDAny NSTDGLRenderPass;

/// Represents a GPU buffer.
typedef NSTDAny NSTDGLBuffer;

/// Represents a GL state.
typedef struct
{
    /// The surface to draw on.
    NSTDGLSurface surface;
    /// The surface configuration.
    NSTDGLSurfaceConfiguration config;
    /// A handle to the drawing device.
    NSTDGLDeviceHandle device_handle;
    /// The drawing device.
    NSTDGLDevice device;
    /// The device's command queue.
    NSTDGLQueue queue;
    /// The size of the window.
    NSTDWindowSize size;
    /// The window's clear color.
    NSTDGLColor clear_color;
} NSTDGLState;

/// Represents a graphics backend.
typedef enum
{
    /// An unknown graphics backend.
    NSTD_GL_BACKEND_UNKNOWN,
    /// Vulkan.
    NSTD_GL_BACKEND_VULKAN,
    /// Metal.
    NSTD_GL_BACKEND_METAL,
    /// Direct3D 12.
    NSTD_GL_BACKEND_DX12,
    /// Direct3D 11.
    NSTD_GL_BACKEND_DX11,
    /// OpenGL.
    NSTD_GL_BACKEND_GL,
    /// Web based GPU.
    NSTD_GL_BACKEND_WEBGPU
} NSTDGLBackend;

/// Represents a device type.
typedef enum
{
    /// An unknown device type.
    NSTD_GL_DEVICE_TYPE_UNKNOWN,
    /// `wgpu`'s integrated GPU.
    NSTD_GL_DEVICE_TYPE_INTEGRATED_GPU,
    /// A physical GPU.
    NSTD_GL_DEVICE_TYPE_DISCRETE_GPU,
    /// A virtual/hosted GPU.
    NSTD_GL_DEVICE_TYPE_VIRTUAL_GPU,
    /// CPU/Software rendering.
    NSTD_GL_DEVICE_TYPE_CPU
} NSTDGLDeviceType;

/// Contains information on a device.
typedef struct
{
    /// The name of the drawing device.
    NSTDString name;
    /// The device's vendor.
    NSTDUSize vendor;
    /// The ID of the device adapter.
    NSTDUSize device;
    /// The type of drawing device.
    NSTDGLDeviceType device_type;
    /// The drawing backend in use.
    NSTDGLBackend backend;
} NSTDGLDeviceInfo;

/// Represents a state's presentation mode.
typedef enum
{
    /// `wgpu`'s presentation engine will request drawing immediately.
    NSTD_GL_PRESENTATION_MODE_IMMEDIATE,
    /// Waits for the vertical blanking period, but frames are submitted immediately.
    NSTD_GL_PRESENTATION_MODE_MAILBOX,
    /// Waits for the vertical blanking period, and frames are
    /// submitted with the monitor's referesh rate.
    NSTD_GL_PRESENTATION_MODE_FIFO
} NSTDGLPresentationMode;

/// Represents a power preference.
typedef enum
{
    /// Use the default power preference.
    NSTD_GL_POWER_PREFERENCE_DEFAULT,
    /// Use low GPU power.
    NSTD_GL_POWER_PREFERENCE_LOW,
    /// Use high GPU power.
    NSTD_GL_POWER_PREFERENCE_HIGH
} NSTDGLPowerPreference;

/// Configures a GL state upon creation.
/// For `backend`, `NSTD_GL_BACKEND_UNKNOWN` will pick a default backend to use.
typedef struct
{
    /// The graphics backend to use.
    NSTDGLBackend backend;
    /// The amount of GPU power to be used.
    NSTDGLPowerPreference power_preference;
    /// The way frames will be presented to the display.
    NSTDGLPresentationMode presentation_mode;
} NSTDGLStateDescriptor;

/// Represents a vertex format when sending data to the vertex shader.
typedef enum
{
    /// Two `UINT8`s.
    NSTD_GL_VERTEX_FORMAT_UINT8X2,
    /// Four `UINT8`s.
    NSTD_GL_VERTEX_FORMAT_UINT8X4,
    /// Two `INT8`s.
    NSTD_GL_VERTEX_FORMAT_INT8X2,
    /// Four `INT8`s.
    NSTD_GL_VERTEX_FORMAT_INT8X4,
    /// Two `UNORM8`s.
    NSTD_GL_VERTEX_FORMAT_UNORM8X2,
    /// Four `UNORM8`s.
    NSTD_GL_VERTEX_FORMAT_UNORM8X4,
    /// Two `NORM8`s.
    NSTD_GL_VERTEX_FORMAT_NORM8X2,
    /// Four `NORM8`s.
    NSTD_GL_VERTEX_FORMAT_NORM8X4,
    /// Two `UINT16`s.
    NSTD_GL_VERTEX_FORMAT_UINT16X2,
    /// Four `UINT16`s.
    NSTD_GL_VERTEX_FORMAT_UINT16X4,
    /// Two `INT16`s.
    NSTD_GL_VERTEX_FORMAT_INT16X2,
    /// Four `INT16`s.
    NSTD_GL_VERTEX_FORMAT_INT16X4,
    /// Two `UNORM16`s.
    NSTD_GL_VERTEX_FORMAT_UNORM16X2,
    /// Four `UNORM16`s.
    NSTD_GL_VERTEX_FORMAT_UNORM16X4,
    /// Two `NORM16`s.
    NSTD_GL_VERTEX_FORMAT_NORM16X2,
    /// Four `NORM16`s.
    NSTD_GL_VERTEX_FORMAT_NORM16X4,
    /// Two `FLOAT16`s.
    NSTD_GL_VERTEX_FORMAT_FLOAT16X2,
    /// Four `FLOAT16`s.
    NSTD_GL_VERTEX_FORMAT_FLOAT16X4,
    /// One `FLOAT32`.
    NSTD_GL_VERTEX_FORMAT_FLOAT32,
    /// Two `FLOAT32`s.
    NSTD_GL_VERTEX_FORMAT_FLOAT32X2,
    /// Three `FLOAT32`s.
    NSTD_GL_VERTEX_FORMAT_FLOAT32X3,
    /// Four `FLOAT32`s.
    NSTD_GL_VERTEX_FORMAT_FLOAT32X4,
    /// One `UINT32`.
    NSTD_GL_VERTEX_FORMAT_UINT32,
    /// Two `UINT32`s.
    NSTD_GL_VERTEX_FORMAT_UINT32X2,
    /// Three `UINT32`s.
    NSTD_GL_VERTEX_FORMAT_UINT32X3,
    /// Four `UINT32`s.
    NSTD_GL_VERTEX_FORMAT_UINT32X4,
    /// One `INT32`.
    NSTD_GL_VERTEX_FORMAT_INT32,
    /// Two `INT32`s.
    NSTD_GL_VERTEX_FORMAT_INT32X2,
    /// Three `INT32`s.
    NSTD_GL_VERTEX_FORMAT_INT32X3,
    /// Four `INT32`s.
    NSTD_GL_VERTEX_FORMAT_INT32X4,
    /// One `FLOAT64`.
    NSTD_GL_VERTEX_FORMAT_FLOAT64,
    /// Two `FLOAT64`s.
    NSTD_GL_VERTEX_FORMAT_FLOAT64X2,
    /// Three `FLOAT64`s.
    NSTD_GL_VERTEX_FORMAT_FLOAT64X3,
    /// Four `FLOAT64`s.
    NSTD_GL_VERTEX_FORMAT_FLOAT64X4
} NSTDGLVertexFormat;

/// Represents an index format when drawing indexed verticies.
typedef enum
{
    /// `NSTDUInt16`.
    NSTD_GL_INDEX_FORMAT_UINT16,
    /// `NSTDUInt32`.
    NSTD_GL_INDEX_FORMAT_UINT32,
} NSTDGLIndexFormat;

/// Represents a vertex attribute.
/// NOTE: This struct must directly mirror `VertexAttribute` defined by wgpu in terms of size,
/// alignment, and order of fields.
typedef struct
{
    /// The vertex format to be used.
    NSTDGLVertexFormat format;
    /// The offset in bytes from the start of the input.
    NSTDUInt64 offset;
    /// The location for this input.
    NSTDUInt32 location;
} NSTDGLVertexAttribute;

/// Represents a vertex stepping mode.
typedef enum
{
    /// Vertex data is advanced each vertex.
    NSTD_GL_VERTEX_STEP_MODE_VERTEX,
    /// Vertex data is advanced each instance.
    NSTD_GL_VERTEX_STEP_MODE_INSTANCE,
} NSTDGLVertexStepMode;

/// Represents a vertex buffer layout.
/// `attributes` - `&mut [NSTDGLVertexAttribute]`.
typedef struct
{
    /// The number of bytes between each element.
    NSTDUInt64 stride;
    /// Determines how often the vertex data is advanced.
    NSTDGLVertexStepMode step_mode;
    /// A slice of `NSTDGLVertexAttribute`s.
    NSTDSlice attributes;
} NSTDGLVertexBufferLayout;

/// Creates a new GL state.
/// Parameters:
///     `const NSTDWindow window` - The window in which the GL state will live in.
///     `const NSTDGLStateDescriptor descriptor` - Configures the state.
/// Returns: `NSTDGLState state` - The new GL state.
NSTDAPI NSTDGLState nstd_gl_state_new(
    const NSTDWindow window,
    const NSTDGLStateDescriptor descriptor);

/// Pushes the current frame to the display.
/// Parameters:
///     `const NSTDGLState *const state` - The GL state.
///     `void(*callback)(NSTDGLRenderPass)` - Manipulates the render pass.
/// Returns: `NSTDErrorCode errc` - Nonzero on error.
NSTDAPI NSTDErrorCode nstd_gl_state_render(
    const NSTDGLState *const state,
    void(*callback)(NSTDGLRenderPass));

/// Resizes a GL state's context.
/// Parameters:
///     `NSTDGLState *const state` - The GL state.
///     `const NSTDWindowSize *const new_size` - The new context size.
NSTDAPI void nstd_gl_state_resize(NSTDGLState *const state, const NSTDWindowSize *const new_size);

/// Frees and destroys a GL state.
/// Parameters:
///     `NSTDGLState *const state` - The GL state.
NSTDAPI void nstd_gl_state_free(NSTDGLState *const state);

/// Retrieves info on a device.
/// Parameters:
///     `const NSTDGLDeviceHandle device_handle` - Handle to a device.
/// Returns: `NSTDGLDeviceInfo device_info` - Contains information about a device.
NSTDAPI NSTDGLDeviceInfo nstd_gl_device_handle_get_info(const NSTDGLDeviceHandle device_handle);

/// Creates a new shader module.
/// Parameters:
///     `const NSTDSlice *const data` - Raw spirv data.
///     `const NSTDGLDevice device` - The device to create the shader module on.
/// Returns: `NSTDGLShaderModule shader` - The new shader module.
NSTDAPI NSTDGLShaderModule nstd_gl_shader_module_new(
    const NSTDSlice *const data,
    const NSTDGLDevice device);

/// Frees a shader module.
/// Parameters:
///     `NSTDGLShaderModule *const shader` - Pointer to a shader module.
NSTDAPI void nstd_gl_shader_module_free(NSTDGLShaderModule *const shader);

/// Creates a new render pipeline from a vertex and fragment shader.
/// Parameters:
///     `const NSTDGLShaderModule vert` - The vertex shader module.
///     `const NSTDGLShaderModule frag` - The fragment shader module.
///     `const NSTDSlice *const buffers` - A slice of `NSTDGLVertexBufferLayout`s.
///     `const NSTDGLDevice device` - The device to create the render pipeline on.
///     `const NSTDGLSurfaceConfiguration config` - The surface configuration.
/// Returns: `NSTDGLRenderPipeline pipeline` - The new render pipeline.
NSTDAPI NSTDGLRenderPipeline nstd_gl_render_pipeline_new(
    const NSTDGLShaderModule vert,
    const NSTDGLShaderModule frag,
    const NSTDSlice *const buffers,
    const NSTDGLDevice device,
    const NSTDGLSurfaceConfiguration config);

/// Frees a render pipeline.
/// Parameters:
///     `NSTDGLRenderPipeline *const pipeline` - Pointer to a render pipeline.
NSTDAPI void nstd_gl_render_pipeline_free(NSTDGLRenderPipeline *const pipeline);

/// Sets a render pipeline for a render pass.
/// Parameters:
///     `const NSTDGLRenderPass render_pass` - The render pass.
///     `const NSTDGLRenderPipeline pipeline` - The render pipeline.
NSTDAPI void nstd_gl_render_pass_set_pipeline(
    const NSTDGLRenderPass render_pass,
    const NSTDGLRenderPipeline pipeline);

/// Sets a render pass' vertex buffer.
/// Parameters:
///     `const NSTDGLRenderPass render_pass` - The render pass.
///     `const NSTDGLBuffer buffer` - The GPU vertex buffer.
///     `const NSTDUInt32 slot` - The buffer slot (the index of the buffer layout).
NSTDAPI void nstd_gl_render_pass_set_vertex_buffer(
    const NSTDGLRenderPass render_pass,
    const NSTDGLBuffer buffer,
    const NSTDUInt32 slot);

/// Sets a render pass' index buffer.
/// Parameters:
///     `const NSTDGLRenderPass render_pass` - The render pass.
///     `const NSTDGLBuffer buffer` - The GPU index buffer.
///     `const NSTDGLIndexFormat format` - The index format of the buffer.
NSTDAPI void nstd_gl_render_pass_set_index_buffer(
    const NSTDGLRenderPass render_pass,
    const NSTDGLBuffer buffer,
    const NSTDGLIndexFormat format);

/// Draws primitives from active vertex buffers.
/// Parameters:
///     `const NSTDGLRenderPass render_pass` - The render pass.
///     `const NSTDUInt32 verticies` - Number of verticies to draw.
///     `const NSTDUInt32 instances` - Number of instnaces.
NSTDAPI void nstd_gl_render_pass_draw(
    const NSTDGLRenderPass render_pass,
    const NSTDUInt32 verticies,
    const NSTDUInt32 instances);

/// Draws primitives from active vertex and index buffers.
/// Parameters:
///     `const NSTDGLRenderPass render_pass` - The render pass.
///     `const NSTDUInt32 indicies` - The indicies to draw.
///     `const NSTDUInt32 instances` - The instances to draw.
///     `const NSTDInt32 base` - The base vertex.
NSTDAPI void nstd_gl_render_pass_draw_indexed(
    const NSTDGLRenderPass render_pass,
    const NSTDUInt32 indicies,
    const NSTDUInt32 instances,
    const NSTDInt32 base);

/// Frees an `NSTDGLDeviceInfo` object.
/// Parameters:
///     `NSTDGLDeviceInfo *const device_info` - Pointer to an `NSTDGLDeviceInfo` object.
NSTDAPI void nstd_gl_device_info_free(NSTDGLDeviceInfo *const device_info);

/// Creates a new GPU buffer.
/// Parameters:
///     `const NSTDSlice *const bytes` - The bytes to send to the GPU.
///     `const NSTDGLDevice device` - The device to create the buffer on.
/// Returns: `NSTDGLBuffer buffer` - The new GPU buffer.
NSTDAPI NSTDGLBuffer nstd_gl_buffer_new(const NSTDSlice *const bytes, const NSTDGLDevice device);

/// Frees a GPU buffer.
/// Parameters:
///     `NSTDGLBuffer *const buffer` - The GPU buffer.
NSTDAPI void nstd_gl_buffer_free(NSTDGLBuffer *const buffer);


#ifdef NSTDCPP
}
#endif
#endif
