use crate::{
    core::def::{NSTDAny, NSTDAnyConst, NSTDBool, NSTDErrorCode},
    fs::file::NSTDFile,
    string::NSTDString,
};
use cpal::{
    traits::*, BufferSize, BuildStreamError, Device, Host, Sample, SampleFormat, SampleRate,
    Stream, StreamConfig,
};
use rodio::{Decoder, OutputStream, OutputStreamHandle, Sink};
use std::io::BufReader;

/// Represents an audio host.
pub type NSTDAudioHost = *mut Host;

/// Represents an audio device.
pub type NSTDAudioDevice = *mut Device;

/// Represents an audio stream.
pub type NSTDAudioStream = *mut Stream;

/// Represents an audio sample format.
#[repr(C)]
#[allow(non_camel_case_types)]
pub enum NSTDAudioSampleFormat {
    /// Signed 16-bit integer.
    INT16,
    /// Unsigned 16-bit integer.
    UINT16,
    /// 32-bit float.
    FLOAT32,
}

/// Represents a stream config.
/// NOTE: `buffer_size` will be set to 0 for default.
#[repr(C)]
pub struct NSTDAudioStreamConfig {
    /// The number of channels this audio stream owns.
    pub channels: u16,
    /// The sample rate of the audio stream.
    pub sample_rate: u32,
    /// The size of the sample buffer.
    pub buffer_size: u32,
    /// The data format of each sample.
    pub format: NSTDAudioSampleFormat,
}

/// Represents an audio play stream.
#[repr(C)]
pub struct NSTDAudioPlayStream {
    /// The output stream.
    pub stream: *mut OutputStream,
    /// A handle to the output stream.
    pub handle: *mut OutputStreamHandle,
}
impl Default for NSTDAudioPlayStream {
    fn default() -> Self {
        Self {
            stream: std::ptr::null_mut(),
            handle: std::ptr::null_mut(),
        }
    }
}

/// Represents an audio sink.
pub type NSTDAudioSink = *mut Sink;

/// Gets the default audio host.
/// Returns: `NSTDAudioHost host` - The default audio host.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_host_default() -> NSTDAudioHost {
    Box::into_raw(Box::new(cpal::default_host()))
}

/// Generates `nstd_audio_host_default_*_device` functions.
macro_rules! generate_default_device {
    ($name: ident, $method: ident) => {
        #[inline]
        #[cfg_attr(feature = "clib", no_mangle)]
        pub unsafe extern "C" fn $name(host: NSTDAudioHost) -> NSTDAudioDevice {
            match (*host).$method() {
                Some(device) => Box::into_raw(Box::new(device)),
                _ => std::ptr::null_mut(),
            }
        }
    };
}
generate_default_device!(nstd_audio_host_default_input_device, default_input_device);
generate_default_device!(nstd_audio_host_default_output_device, default_output_device);

/// Frees a host's memory.
/// Parameters:
///     `NSTDAudioHost *const host` - Pointer to an audio host.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_host_free(host: *mut NSTDAudioHost) {
    Box::from_raw(*host);
    *host = std::ptr::null_mut();
}

/// Gets the name of a device.
/// Parameters:
///     `const NSTDAudioDevice device` - The device.
/// Returns: `NSTDString name` - The device name.
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_device_name(device: NSTDAudioDevice) -> NSTDString {
    match (*device).name() {
        Ok(name) => NSTDString::from(name.as_bytes()),
        _ => {
            let null = crate::core::slice::nstd_core_slice_new(0, 0, std::ptr::null_mut());
            let null = crate::collections::vec::nstd_collections_vec_from_existing(0, &null);
            crate::string::nstd_string_from_existing(&null)
        }
    }
}

/// Generates `nstd_audio_device_default_*_config` functions.
macro_rules! generate_device_default_config {
    ($name: ident, $method: ident) => {
        #[cfg_attr(feature = "clib", no_mangle)]
        pub unsafe extern "C" fn $name(
            device: NSTDAudioDevice,
            config: *mut NSTDAudioStreamConfig,
        ) -> NSTDErrorCode {
            match (*device).$method() {
                Ok(cpal_supported_config) => {
                    let format = cpal_supported_config.sample_format();
                    let cpal_config = cpal_supported_config.config();
                    *config = NSTDAudioStreamConfig {
                        channels: cpal_config.channels,
                        sample_rate: cpal_config.sample_rate.0,
                        buffer_size: match cpal_config.buffer_size {
                            BufferSize::Default => 0,
                            BufferSize::Fixed(size) => size,
                        },
                        format: match format {
                            SampleFormat::I16 => NSTDAudioSampleFormat::INT16,
                            SampleFormat::U16 => NSTDAudioSampleFormat::UINT16,
                            SampleFormat::F32 => NSTDAudioSampleFormat::FLOAT32,
                        },
                    };
                    0
                }
                _ => 1,
            }
        }
    };
}
generate_device_default_config!(nstd_audio_device_default_input_config, default_input_config);
generate_device_default_config!(
    nstd_audio_device_default_output_config,
    default_output_config
);

/// Generates `nstd_audio_device_build_*_stream` functions.
macro_rules! generate_device_build_stream {
    ($name: ident, $func: ident, $ptr_ty: ty) => {
        #[cfg_attr(feature = "clib", no_mangle)]
        pub unsafe extern "C" fn $name(
            device: NSTDAudioDevice,
            config: *const NSTDAudioStreamConfig,
            format: NSTDAudioSampleFormat,
            callback: extern "C" fn($ptr_ty, usize),
            err_callback: extern "C" fn(),
        ) -> NSTDAudioStream {
            let config = StreamConfig {
                channels: (*config).channels,
                sample_rate: SampleRate((*config).sample_rate),
                buffer_size: match (*config).buffer_size {
                    0 => BufferSize::Default,
                    size => BufferSize::Fixed(size),
                },
            };
            match match format {
                NSTDAudioSampleFormat::INT16 => {
                    $func::<i16>(&*device, &config, callback, err_callback)
                }
                NSTDAudioSampleFormat::UINT16 => {
                    $func::<u16>(&*device, &config, callback, err_callback)
                }
                NSTDAudioSampleFormat::FLOAT32 => {
                    $func::<f32>(&*device, &config, callback, err_callback)
                }
            } {
                Ok(stream) => match stream.play() {
                    Ok(_) => Box::into_raw(Box::new(stream)),
                    _ => std::ptr::null_mut(),
                },
                _ => std::ptr::null_mut(),
            }
        }
    };
}
generate_device_build_stream!(
    nstd_audio_device_build_input_stream,
    build_input_stream,
    NSTDAnyConst
);
generate_device_build_stream!(
    nstd_audio_device_build_output_stream,
    build_output_stream,
    NSTDAny
);

/// Frees a device.
/// Parameters:
///     `NSTDAudioDevice *const device` - Pointer to a device.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_device_free(device: *mut NSTDAudioDevice) {
    Box::from_raw(*device);
    *device = std::ptr::null_mut();
}

/// Generates `nstd_audio_stream_*` functions.
macro_rules! generate_stream_play_pause {
    ($name: ident, $method: ident) => {
        #[inline]
        #[cfg_attr(feature = "clib", no_mangle)]
        pub unsafe extern "C" fn $name(stream: NSTDAudioStream) -> NSTDErrorCode {
            match (*stream).$method() {
                Ok(_) => 0,
                _ => 1,
            }
        }
    };
}
generate_stream_play_pause!(nstd_audio_stream_play, play);
generate_stream_play_pause!(nstd_audio_stream_pause, pause);

/// Frees an audio stream
/// Parameters:
///     `NSTDAudioStream *const stream` - Pointer to an audio stream.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_stream_free(stream: *mut NSTDAudioStream) {
    Box::from_raw(*stream);
    *stream = std::ptr::null_mut();
}

/// Generates `build_*_stream` functions.
macro_rules! generate_build_stream {
    ($name: ident, $ptr_method: ident, $cbptr_ty: ty, $slice_ty: ty) => {
        #[inline]
        fn $name<T: Sample>(
            device: &Device,
            config: &StreamConfig,
            callback: extern "C" fn($cbptr_ty, usize),
            err_callback: extern "C" fn(),
        ) -> Result<Stream, BuildStreamError> {
            device.$name(
                &config,
                move |data: $slice_ty, _| callback(data.$ptr_method() as $cbptr_ty, data.len()),
                move |_| err_callback(),
            )
        }
    };
}
generate_build_stream!(build_input_stream, as_ptr, NSTDAnyConst, &[T]);
generate_build_stream!(build_output_stream, as_mut_ptr, NSTDAny, &mut [T]);

/// Creates a play stream.
/// Returns: `NSTDAudioPlayStream stream` - The new play stream.
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_play_stream_new() -> NSTDAudioPlayStream {
    match OutputStream::try_default() {
        Ok((stream, handle)) => NSTDAudioPlayStream {
            stream: Box::into_raw(Box::new(stream)),
            handle: Box::into_raw(Box::new(handle)),
        },
        _ => NSTDAudioPlayStream::default(),
    }
}

/// Frees a play stream.
/// Parameters:
///     `NSTDAudioPlayStream *const stream` - The play stream.
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_play_stream_free(stream: &mut NSTDAudioPlayStream) {
    Box::from_raw(stream.stream);
    Box::from_raw(stream.handle);
    stream.stream = std::ptr::null_mut();
    stream.handle = std::ptr::null_mut();
}

/// Creates a new audio sink.
/// Parameters:
///     `const NSTDAudioPlayStream *const stream` - The stream to create the sink on.
/// Returns: `NSTDAudioSink sink` - The new audio sink.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_sink_new(stream: &NSTDAudioPlayStream) -> NSTDAudioSink {
    match Sink::try_new(&*stream.handle) {
        Ok(sink) => Box::into_raw(Box::new(sink)),
        _ => std::ptr::null_mut(),
    }
}

/// Appends audio to a sink from a file.
/// Parameters:
///     `const NSTDAudioSink sink` - The audio sink.
///     `const NSTDFile *const file` - The audio file.
///     `const NSTDBool should_loop` - Nonzero if the audio should be looped.
/// Returns: `NSTDErrorCode errc` - Nonzero on error.
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_sink_append_from_file(
    sink: NSTDAudioSink,
    file: &NSTDFile,
    should_loop: NSTDBool,
) -> NSTDErrorCode {
    let buf = BufReader::new((&*file.handle).get_ref());
    match should_loop {
        NSTDBool::NSTD_BOOL_FALSE => match Decoder::new(buf) {
            Ok(decoder) => {
                (*sink).append(decoder);
                0
            }
            _ => 1,
        },
        _ => match Decoder::new_looped(buf) {
            Ok(decoder) => {
                (*sink).append(decoder);
                0
            }
            _ => 1,
        },
    }
}

/// Plays an audio sink.
/// Parameters:
///     `const NSTDAudioSink sink` - The audio sink.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_sink_play(sink: NSTDAudioSink) {
    (*sink).play();
}

/// Pauses an audio sink.
/// Parameters:
///     `const NSTDAudioSink sink` - The audio sink.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_sink_pause(sink: NSTDAudioSink) {
    (*sink).pause();
}

/// Checks if an audio sink is paused.
/// Parameters:
///     `const NSTDAudioSink sink` - The audio sink.
/// Returns: `NSTDBool is_paused` - Whether or not the audio sink is paused.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_sink_is_paused(sink: NSTDAudioSink) -> NSTDBool {
    NSTDBool::from((*sink).is_paused())
}

/// Stops audio playback for a sink by clearing it's queue.
/// Parameters:
///     `const NSTDAudioSink sink` - The audio sink.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_sink_stop(sink: NSTDAudioSink) {
    (*sink).stop();
}

/// Sleeps the current thread until all sounds in the sink are done playing.
/// Parameters:
///     `const NSTDAudioSink sink` - The audio sink.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_sink_sleep_until_end(sink: NSTDAudioSink) {
    (*sink).sleep_until_end();
}

/// Returns the volume of the audio sink.
/// Parameters:
///     `const NSTDAudioSink sink` - The audio sink.
/// Returns: `NSTDFloat32 volume` - The volume of the sink.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_sink_get_volume(sink: NSTDAudioSink) -> f32 {
    (*sink).volume()
}

/// Sets the volume of the audio sink.
/// Parameters:
///     `const NSTDAudioSink sink` - The audio sink.
///     `const NSTDFloat32 volume` - The volume of the sink.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_sink_set_volume(sink: NSTDAudioSink, volume: f32) {
    (*sink).set_volume(volume);
}

/// Gets the number of audio sources currently in a sink.
/// Parameters:
///     `const NSTDAudioSink sink` - The audio sink.
/// Returns: `NSTDUSize size` - The number of audio sources in an audio sink.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_sink_length(sink: NSTDAudioSink) -> usize {
    (*sink).len()
}

/// Detaches a sink from it's thread while freeing its memory.
/// Parameters:
///     `NSTDAudioSink *const sink` - The audio sink.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_sink_detach(sink: &mut NSTDAudioSink) {
    let boxed_sink = Box::from_raw(*sink);
    boxed_sink.detach();
    *sink = std::ptr::null_mut();
}

/// Frees an audio sink.
/// Parameters:
///     `NSTDAudioSink *const sink` - The audio sink.
#[inline]
#[cfg_attr(feature = "clib", no_mangle)]
pub unsafe extern "C" fn nstd_audio_sink_free(sink: &mut NSTDAudioSink) {
    Box::from_raw(*sink);
    *sink = std::ptr::null_mut();
}
