use num_derive::*;
use num_traits::*;
use winit::event::VirtualKeyCode;
pub const NSTD_INPUT_KEY_SHIFT_BIT: u8 = 0b00000001;
pub const NSTD_INPUT_KEY_CTRL_BIT: u8 = 0b00000010;
pub const NSTD_INPUT_KEY_ALT_BIT: u8 = 0b00000100;
pub const NSTD_INPUT_KEY_LOGO_BIT: u8 = 0b00001000;

/// Represents a virtual key code.
#[repr(C)]
#[allow(non_camel_case_types)]
#[derive(FromPrimitive, ToPrimitive)]
pub enum NSTDKey {
    NSTD_KEY_NONE,
    NSTD_KEY_1,
    NSTD_KEY_2,
    NSTD_KEY_3,
    NSTD_KEY_4,
    NSTD_KEY_5,
    NSTD_KEY_6,
    NSTD_KEY_7,
    NSTD_KEY_8,
    NSTD_KEY_9,
    NSTD_KEY_0,
    NSTD_KEY_A,
    NSTD_KEY_B,
    NSTD_KEY_C,
    NSTD_KEY_D,
    NSTD_KEY_E,
    NSTD_KEY_F,
    NSTD_KEY_G,
    NSTD_KEY_H,
    NSTD_KEY_I,
    NSTD_KEY_J,
    NSTD_KEY_K,
    NSTD_KEY_L,
    NSTD_KEY_M,
    NSTD_KEY_N,
    NSTD_KEY_O,
    NSTD_KEY_P,
    NSTD_KEY_Q,
    NSTD_KEY_R,
    NSTD_KEY_S,
    NSTD_KEY_T,
    NSTD_KEY_U,
    NSTD_KEY_V,
    NSTD_KEY_W,
    NSTD_KEY_X,
    NSTD_KEY_Y,
    NSTD_KEY_Z,
    NSTD_KEY_ESCAPE,
    NSTD_KEY_F1,
    NSTD_KEY_F2,
    NSTD_KEY_F3,
    NSTD_KEY_F4,
    NSTD_KEY_F5,
    NSTD_KEY_F6,
    NSTD_KEY_F7,
    NSTD_KEY_F8,
    NSTD_KEY_F9,
    NSTD_KEY_F10,
    NSTD_KEY_F11,
    NSTD_KEY_F12,
    NSTD_KEY_F13,
    NSTD_KEY_F14,
    NSTD_KEY_F15,
    NSTD_KEY_F16,
    NSTD_KEY_F17,
    NSTD_KEY_F18,
    NSTD_KEY_F19,
    NSTD_KEY_F20,
    NSTD_KEY_F21,
    NSTD_KEY_F22,
    NSTD_KEY_F23,
    NSTD_KEY_F24,
    NSTD_KEY_SNAPSHOT,
    NSTD_KEY_SCROLL,
    NSTD_KEY_PAUSE,
    NSTD_KEY_INSERT,
    NSTD_KEY_HOME,
    NSTD_KEY_DELETE,
    NSTD_KEY_END,
    NSTD_KEY_PAGE_DOWN,
    NSTD_KEY_PAGE_UP,
    NSTD_KEY_LEFT,
    NSTD_KEY_UP,
    NSTD_KEY_RIGHT,
    NSTD_KEY_DOWN,
    NSTD_KEY_BACK,
    NSTD_KEY_RETURN,
    NSTD_KEY_SPACE,
    NSTD_KEY_COMPOSE,
    NSTD_KEY_CARET,
    NSTD_KEY_NUMLOCK,
    NSTD_KEY_NUMPAD_0,
    NSTD_KEY_NUMPAD_1,
    NSTD_KEY_NUMPAD_2,
    NSTD_KEY_NUMPAD_3,
    NSTD_KEY_NUMPAD_4,
    NSTD_KEY_NUMPAD_5,
    NSTD_KEY_NUMPAD_6,
    NSTD_KEY_NUMPAD_7,
    NSTD_KEY_NUMPAD_8,
    NSTD_KEY_NUMPAD_9,
    NSTD_KEY_NUMPAD_ADD,
    NSTD_KEY_NUMPAD_DIVIDE,
    NSTD_KEY_NUMPAD_DECIMAL,
    NSTD_KEY_NUMPAD_COMMA,
    NSTD_KEY_NUMPAD_ENTER,
    NSTD_KEY_NUMPAD_EQUALS,
    NSTD_KEY_NUMPAD_MULTIPLY,
    NSTD_KEY_NUMPAD_SUBTRACT,
    NSTD_KEY_ABNTC1,
    NSTD_KEY_ABNTC2,
    NSTD_KEY_APOSTROPHE,
    NSTD_KEY_APPS,
    NSTD_KEY_ASTERISK,
    NSTD_KEY_AT,
    NSTD_KEY_AX,
    NSTD_KEY_BACKSLASH,
    NSTD_KEY_CALCULATOR,
    NSTD_KEY_CAPITAL,
    NSTD_KEY_COLON,
    NSTD_KEY_COMMA,
    NSTD_KEY_CONVERT,
    NSTD_KEY_EQUALS,
    NSTD_KEY_GRAVE,
    NSTD_KEY_KANA,
    NSTD_KEY_KANJI,
    NSTD_KEY_LEFT_ALT,
    NSTD_KEY_LEFT_BRACKET,
    NSTD_KEY_LEFT_CONTROL,
    NSTD_KEY_LEFT_SHIFT,
    NSTD_KEY_LEFT_WIN,
    NSTD_KEY_MAIL,
    NSTD_KEY_MEDIA_SELECT,
    NSTD_KEY_MEDIA_STOP,
    NSTD_KEY_MINUS,
    NSTD_KEY_MUTE,
    NSTD_KEY_MY_COMPUTER,
    NSTD_KEY_NAVIGATE_FORWARD,
    NSTD_KEY_NAVIGATE_BACKWARD,
    NSTD_KEY_NEXT_TRACK,
    NSTD_KEY_NO_CONVERT,
    NSTD_KEY_OEM102,
    NSTD_KEY_PERIOD,
    NSTD_KEY_PLAY_PAUSE,
    NSTD_KEY_PLUS,
    NSTD_KEY_POWER,
    NSTD_KEY_PREV_TRACK,
    NSTD_KEY_RIGHT_ALT,
    NSTD_KEY_RIGHT_BRACKET,
    NSTD_KEY_RIGHT_CONTROL,
    NSTD_KEY_RIGHT_SHIFT,
    NSTD_KEY_RIGHT_WIN,
    NSTD_KEY_SEMICOLON,
    NSTD_KEY_SLASH,
    NSTD_KEY_SLEEP,
    NSTD_KEY_STOP,
    NSTD_KEY_SYSRQ,
    NSTD_KEY_TAB,
    NSTD_KEY_UNDERLINE,
    NSTD_KEY_UNLABELED,
    NSTD_KEY_VOLUME_DOWN,
    NSTD_KEY_VOLUME_UP,
    NSTD_KEY_WAKE,
    NSTD_KEY_WEB_BACK,
    NSTD_KEY_WEB_FAVORITES,
    NSTD_KEY_WEB_FORWARD,
    NSTD_KEY_WEB_HOME,
    NSTD_KEY_WEB_REFRESH,
    NSTD_KEY_WEB_SEARCH,
    NSTD_KEY_WEB_STOP,
    NSTD_KEY_YEN,
    NSTD_KEY_COPY,
    NSTD_KEY_PASTE,
    NSTD_KEY_CUT,
}
impl Default for NSTDKey {
    #[inline]
    fn default() -> Self {
        Self::NSTD_KEY_NONE
    }
}
impl From<VirtualKeyCode> for NSTDKey {
    #[inline]
    fn from(key: VirtualKeyCode) -> Self {
        match key {
            // English alphabet.
            VirtualKeyCode::A => Self::NSTD_KEY_A,
            VirtualKeyCode::B => Self::NSTD_KEY_B,
            VirtualKeyCode::C => Self::NSTD_KEY_C,
            VirtualKeyCode::D => Self::NSTD_KEY_D,
            VirtualKeyCode::E => Self::NSTD_KEY_E,
            VirtualKeyCode::F => Self::NSTD_KEY_F,
            VirtualKeyCode::G => Self::NSTD_KEY_G,
            VirtualKeyCode::H => Self::NSTD_KEY_H,
            VirtualKeyCode::I => Self::NSTD_KEY_I,
            VirtualKeyCode::J => Self::NSTD_KEY_J,
            VirtualKeyCode::K => Self::NSTD_KEY_K,
            VirtualKeyCode::L => Self::NSTD_KEY_L,
            VirtualKeyCode::M => Self::NSTD_KEY_M,
            VirtualKeyCode::N => Self::NSTD_KEY_N,
            VirtualKeyCode::O => Self::NSTD_KEY_O,
            VirtualKeyCode::P => Self::NSTD_KEY_P,
            VirtualKeyCode::Q => Self::NSTD_KEY_Q,
            VirtualKeyCode::R => Self::NSTD_KEY_R,
            VirtualKeyCode::S => Self::NSTD_KEY_S,
            VirtualKeyCode::T => Self::NSTD_KEY_T,
            VirtualKeyCode::U => Self::NSTD_KEY_U,
            VirtualKeyCode::V => Self::NSTD_KEY_V,
            VirtualKeyCode::W => Self::NSTD_KEY_W,
            VirtualKeyCode::X => Self::NSTD_KEY_X,
            VirtualKeyCode::Y => Self::NSTD_KEY_Y,
            VirtualKeyCode::Z => Self::NSTD_KEY_Z,
            // Row 1.
            VirtualKeyCode::Escape => Self::NSTD_KEY_ESCAPE,
            VirtualKeyCode::F1 => Self::NSTD_KEY_F1,
            VirtualKeyCode::F2 => Self::NSTD_KEY_F2,
            VirtualKeyCode::F3 => Self::NSTD_KEY_F3,
            VirtualKeyCode::F4 => Self::NSTD_KEY_F4,
            VirtualKeyCode::F5 => Self::NSTD_KEY_F5,
            VirtualKeyCode::F6 => Self::NSTD_KEY_F6,
            VirtualKeyCode::F7 => Self::NSTD_KEY_F7,
            VirtualKeyCode::F8 => Self::NSTD_KEY_F8,
            VirtualKeyCode::F9 => Self::NSTD_KEY_F9,
            VirtualKeyCode::F10 => Self::NSTD_KEY_F10,
            VirtualKeyCode::F11 => Self::NSTD_KEY_F11,
            VirtualKeyCode::F12 => Self::NSTD_KEY_F12,
            VirtualKeyCode::F13 => Self::NSTD_KEY_F13,
            VirtualKeyCode::F14 => Self::NSTD_KEY_F14,
            VirtualKeyCode::F15 => Self::NSTD_KEY_F15,
            VirtualKeyCode::F16 => Self::NSTD_KEY_F16,
            VirtualKeyCode::F17 => Self::NSTD_KEY_F17,
            VirtualKeyCode::F18 => Self::NSTD_KEY_F18,
            VirtualKeyCode::F19 => Self::NSTD_KEY_F19,
            VirtualKeyCode::F20 => Self::NSTD_KEY_F20,
            VirtualKeyCode::F21 => Self::NSTD_KEY_F21,
            VirtualKeyCode::F22 => Self::NSTD_KEY_F22,
            VirtualKeyCode::F23 => Self::NSTD_KEY_F23,
            VirtualKeyCode::F24 => Self::NSTD_KEY_F24,
            // Row 2.
            VirtualKeyCode::Grave => Self::NSTD_KEY_GRAVE,
            VirtualKeyCode::Key1 => Self::NSTD_KEY_1,
            VirtualKeyCode::Key2 => Self::NSTD_KEY_2,
            VirtualKeyCode::Key3 => Self::NSTD_KEY_3,
            VirtualKeyCode::Key4 => Self::NSTD_KEY_4,
            VirtualKeyCode::Key5 => Self::NSTD_KEY_5,
            VirtualKeyCode::Key6 => Self::NSTD_KEY_6,
            VirtualKeyCode::Key7 => Self::NSTD_KEY_7,
            VirtualKeyCode::Key8 => Self::NSTD_KEY_8,
            VirtualKeyCode::Key9 => Self::NSTD_KEY_9,
            VirtualKeyCode::Key0 => Self::NSTD_KEY_0,
            VirtualKeyCode::Minus => Self::NSTD_KEY_MINUS,
            VirtualKeyCode::Equals => Self::NSTD_KEY_EQUALS,
            VirtualKeyCode::Back => Self::NSTD_KEY_BACK,
            // Row 3.
            VirtualKeyCode::Tab => Self::NSTD_KEY_TAB,
            VirtualKeyCode::LBracket => Self::NSTD_KEY_LEFT_BRACKET,
            VirtualKeyCode::RBracket => Self::NSTD_KEY_RIGHT_BRACKET,
            VirtualKeyCode::Backslash => Self::NSTD_KEY_BACKSLASH,
            // Row 4.
            VirtualKeyCode::Capital => Self::NSTD_KEY_CAPITAL,
            VirtualKeyCode::Semicolon => Self::NSTD_KEY_SEMICOLON,
            VirtualKeyCode::Apostrophe => Self::NSTD_KEY_APOSTROPHE,
            VirtualKeyCode::Return => Self::NSTD_KEY_RETURN,
            // Row 5.
            VirtualKeyCode::LShift => Self::NSTD_KEY_LEFT_SHIFT,
            VirtualKeyCode::RShift => Self::NSTD_KEY_RIGHT_SHIFT,
            VirtualKeyCode::Comma => Self::NSTD_KEY_COMMA,
            VirtualKeyCode::Period => Self::NSTD_KEY_PERIOD,
            VirtualKeyCode::Slash => Self::NSTD_KEY_SLASH,
            // Row 6.
            VirtualKeyCode::LControl => Self::NSTD_KEY_LEFT_CONTROL,
            VirtualKeyCode::RControl => Self::NSTD_KEY_RIGHT_CONTROL,
            VirtualKeyCode::LAlt => Self::NSTD_KEY_LEFT_ALT,
            VirtualKeyCode::RAlt => Self::NSTD_KEY_RIGHT_ALT,
            VirtualKeyCode::Space => Self::NSTD_KEY_SPACE,
            // 60%.
            VirtualKeyCode::Insert => Self::NSTD_KEY_INSERT,
            VirtualKeyCode::Home => Self::NSTD_KEY_HOME,
            VirtualKeyCode::PageUp => Self::NSTD_KEY_PAGE_UP,
            VirtualKeyCode::PageDown => Self::NSTD_KEY_PAGE_DOWN,
            VirtualKeyCode::Delete => Self::NSTD_KEY_DELETE,
            VirtualKeyCode::End => Self::NSTD_KEY_END,
            VirtualKeyCode::Up => Self::NSTD_KEY_UP,
            VirtualKeyCode::Down => Self::NSTD_KEY_DOWN,
            VirtualKeyCode::Left => Self::NSTD_KEY_LEFT,
            VirtualKeyCode::Right => Self::NSTD_KEY_RIGHT,
            // Numpad.
            VirtualKeyCode::Numlock => Self::NSTD_KEY_NUMLOCK,
            VirtualKeyCode::NumpadDivide => Self::NSTD_KEY_NUMPAD_DIVIDE,
            VirtualKeyCode::NumpadMultiply => Self::NSTD_KEY_NUMPAD_MULTIPLY,
            VirtualKeyCode::NumpadSubtract => Self::NSTD_KEY_NUMPAD_SUBTRACT,
            VirtualKeyCode::NumpadAdd => Self::NSTD_KEY_NUMPAD_ADD,
            VirtualKeyCode::NumpadEnter => Self::NSTD_KEY_NUMPAD_ENTER,
            VirtualKeyCode::NumpadDecimal => Self::NSTD_KEY_NUMPAD_DECIMAL,
            VirtualKeyCode::Numpad1 => Self::NSTD_KEY_NUMPAD_1,
            VirtualKeyCode::Numpad2 => Self::NSTD_KEY_NUMPAD_2,
            VirtualKeyCode::Numpad3 => Self::NSTD_KEY_NUMPAD_3,
            VirtualKeyCode::Numpad4 => Self::NSTD_KEY_NUMPAD_4,
            VirtualKeyCode::Numpad5 => Self::NSTD_KEY_NUMPAD_5,
            VirtualKeyCode::Numpad6 => Self::NSTD_KEY_NUMPAD_6,
            VirtualKeyCode::Numpad7 => Self::NSTD_KEY_NUMPAD_7,
            VirtualKeyCode::Numpad8 => Self::NSTD_KEY_NUMPAD_8,
            VirtualKeyCode::Numpad9 => Self::NSTD_KEY_NUMPAD_9,
            VirtualKeyCode::Numpad0 => Self::NSTD_KEY_NUMPAD_0,
            // Fallback to implicit conversion.
            _ => match FromPrimitive::from_i32((key as i32) + 1) {
                Some(key) => key,
                _ => Self::NSTD_KEY_NONE,
            },
        }
    }
}
impl TryInto<VirtualKeyCode> for NSTDKey {
    type Error = u32;

    #[inline]
    fn try_into(self) -> Result<VirtualKeyCode, Self::Error> {
        match self {
            // English alphabet.
            Self::NSTD_KEY_A => Ok(VirtualKeyCode::A),
            Self::NSTD_KEY_B => Ok(VirtualKeyCode::B),
            Self::NSTD_KEY_C => Ok(VirtualKeyCode::C),
            Self::NSTD_KEY_D => Ok(VirtualKeyCode::D),
            Self::NSTD_KEY_E => Ok(VirtualKeyCode::E),
            Self::NSTD_KEY_F => Ok(VirtualKeyCode::F),
            Self::NSTD_KEY_G => Ok(VirtualKeyCode::G),
            Self::NSTD_KEY_H => Ok(VirtualKeyCode::H),
            Self::NSTD_KEY_I => Ok(VirtualKeyCode::I),
            Self::NSTD_KEY_J => Ok(VirtualKeyCode::J),
            Self::NSTD_KEY_K => Ok(VirtualKeyCode::K),
            Self::NSTD_KEY_L => Ok(VirtualKeyCode::L),
            Self::NSTD_KEY_M => Ok(VirtualKeyCode::M),
            Self::NSTD_KEY_N => Ok(VirtualKeyCode::N),
            Self::NSTD_KEY_O => Ok(VirtualKeyCode::O),
            Self::NSTD_KEY_P => Ok(VirtualKeyCode::P),
            Self::NSTD_KEY_Q => Ok(VirtualKeyCode::Q),
            Self::NSTD_KEY_R => Ok(VirtualKeyCode::R),
            Self::NSTD_KEY_S => Ok(VirtualKeyCode::S),
            Self::NSTD_KEY_T => Ok(VirtualKeyCode::T),
            Self::NSTD_KEY_U => Ok(VirtualKeyCode::U),
            Self::NSTD_KEY_V => Ok(VirtualKeyCode::V),
            Self::NSTD_KEY_W => Ok(VirtualKeyCode::W),
            Self::NSTD_KEY_X => Ok(VirtualKeyCode::X),
            Self::NSTD_KEY_Y => Ok(VirtualKeyCode::Y),
            Self::NSTD_KEY_Z => Ok(VirtualKeyCode::Z),
            // Row 1.
            Self::NSTD_KEY_ESCAPE => Ok(VirtualKeyCode::Escape),
            Self::NSTD_KEY_F1 => Ok(VirtualKeyCode::F1),
            Self::NSTD_KEY_F2 => Ok(VirtualKeyCode::F2),
            Self::NSTD_KEY_F3 => Ok(VirtualKeyCode::F3),
            Self::NSTD_KEY_F4 => Ok(VirtualKeyCode::F4),
            Self::NSTD_KEY_F5 => Ok(VirtualKeyCode::F5),
            Self::NSTD_KEY_F6 => Ok(VirtualKeyCode::F6),
            Self::NSTD_KEY_F7 => Ok(VirtualKeyCode::F7),
            Self::NSTD_KEY_F8 => Ok(VirtualKeyCode::F8),
            Self::NSTD_KEY_F9 => Ok(VirtualKeyCode::F9),
            Self::NSTD_KEY_F10 => Ok(VirtualKeyCode::F10),
            Self::NSTD_KEY_F11 => Ok(VirtualKeyCode::F11),
            Self::NSTD_KEY_F12 => Ok(VirtualKeyCode::F12),
            Self::NSTD_KEY_F13 => Ok(VirtualKeyCode::F13),
            Self::NSTD_KEY_F14 => Ok(VirtualKeyCode::F14),
            Self::NSTD_KEY_F15 => Ok(VirtualKeyCode::F15),
            Self::NSTD_KEY_F16 => Ok(VirtualKeyCode::F16),
            Self::NSTD_KEY_F17 => Ok(VirtualKeyCode::F17),
            Self::NSTD_KEY_F18 => Ok(VirtualKeyCode::F18),
            Self::NSTD_KEY_F19 => Ok(VirtualKeyCode::F19),
            Self::NSTD_KEY_F20 => Ok(VirtualKeyCode::F20),
            Self::NSTD_KEY_F21 => Ok(VirtualKeyCode::F21),
            Self::NSTD_KEY_F22 => Ok(VirtualKeyCode::F22),
            Self::NSTD_KEY_F23 => Ok(VirtualKeyCode::F23),
            Self::NSTD_KEY_F24 => Ok(VirtualKeyCode::F24),
            // Row 2.
            Self::NSTD_KEY_GRAVE => Ok(VirtualKeyCode::Grave),
            Self::NSTD_KEY_1 => Ok(VirtualKeyCode::Key1),
            Self::NSTD_KEY_2 => Ok(VirtualKeyCode::Key2),
            Self::NSTD_KEY_3 => Ok(VirtualKeyCode::Key3),
            Self::NSTD_KEY_4 => Ok(VirtualKeyCode::Key4),
            Self::NSTD_KEY_5 => Ok(VirtualKeyCode::Key5),
            Self::NSTD_KEY_6 => Ok(VirtualKeyCode::Key6),
            Self::NSTD_KEY_7 => Ok(VirtualKeyCode::Key7),
            Self::NSTD_KEY_8 => Ok(VirtualKeyCode::Key8),
            Self::NSTD_KEY_9 => Ok(VirtualKeyCode::Key9),
            Self::NSTD_KEY_0 => Ok(VirtualKeyCode::Key0),
            Self::NSTD_KEY_MINUS => Ok(VirtualKeyCode::Minus),
            Self::NSTD_KEY_EQUALS => Ok(VirtualKeyCode::Equals),
            Self::NSTD_KEY_BACK => Ok(VirtualKeyCode::Back),
            // Row 3.
            Self::NSTD_KEY_TAB => Ok(VirtualKeyCode::Tab),
            Self::NSTD_KEY_LEFT_BRACKET => Ok(VirtualKeyCode::LBracket),
            Self::NSTD_KEY_RIGHT_BRACKET => Ok(VirtualKeyCode::RBracket),
            Self::NSTD_KEY_BACKSLASH => Ok(VirtualKeyCode::Backslash),
            // Row 4.
            Self::NSTD_KEY_CAPITAL => Ok(VirtualKeyCode::Capital),
            Self::NSTD_KEY_SEMICOLON => Ok(VirtualKeyCode::Semicolon),
            Self::NSTD_KEY_APOSTROPHE => Ok(VirtualKeyCode::Apostrophe),
            Self::NSTD_KEY_RETURN => Ok(VirtualKeyCode::Return),
            // Row 5.
            Self::NSTD_KEY_LEFT_SHIFT => Ok(VirtualKeyCode::LShift),
            Self::NSTD_KEY_RIGHT_SHIFT => Ok(VirtualKeyCode::RShift),
            Self::NSTD_KEY_COMMA => Ok(VirtualKeyCode::Comma),
            Self::NSTD_KEY_PERIOD => Ok(VirtualKeyCode::Period),
            Self::NSTD_KEY_SLASH => Ok(VirtualKeyCode::Slash),
            // Row 6.
            Self::NSTD_KEY_LEFT_CONTROL => Ok(VirtualKeyCode::LControl),
            Self::NSTD_KEY_RIGHT_CONTROL => Ok(VirtualKeyCode::RControl),
            Self::NSTD_KEY_LEFT_ALT => Ok(VirtualKeyCode::LAlt),
            Self::NSTD_KEY_RIGHT_ALT => Ok(VirtualKeyCode::RAlt),
            Self::NSTD_KEY_SPACE => Ok(VirtualKeyCode::Space),
            // 60%.
            Self::NSTD_KEY_INSERT => Ok(VirtualKeyCode::Insert),
            Self::NSTD_KEY_HOME => Ok(VirtualKeyCode::Home),
            Self::NSTD_KEY_PAGE_UP => Ok(VirtualKeyCode::PageUp),
            Self::NSTD_KEY_PAGE_DOWN => Ok(VirtualKeyCode::PageDown),
            Self::NSTD_KEY_DELETE => Ok(VirtualKeyCode::Delete),
            Self::NSTD_KEY_END => Ok(VirtualKeyCode::End),
            Self::NSTD_KEY_UP => Ok(VirtualKeyCode::Up),
            Self::NSTD_KEY_DOWN => Ok(VirtualKeyCode::Down),
            Self::NSTD_KEY_LEFT => Ok(VirtualKeyCode::Left),
            Self::NSTD_KEY_RIGHT => Ok(VirtualKeyCode::Right),
            // Numpad.
            Self::NSTD_KEY_NUMLOCK => Ok(VirtualKeyCode::Numlock),
            Self::NSTD_KEY_NUMPAD_DIVIDE => Ok(VirtualKeyCode::NumpadDivide),
            Self::NSTD_KEY_NUMPAD_MULTIPLY => Ok(VirtualKeyCode::NumpadMultiply),
            Self::NSTD_KEY_NUMPAD_SUBTRACT => Ok(VirtualKeyCode::NumpadSubtract),
            Self::NSTD_KEY_NUMPAD_ADD => Ok(VirtualKeyCode::NumpadAdd),
            Self::NSTD_KEY_NUMPAD_ENTER => Ok(VirtualKeyCode::NumpadEnter),
            Self::NSTD_KEY_NUMPAD_DECIMAL => Ok(VirtualKeyCode::NumpadDecimal),
            Self::NSTD_KEY_NUMPAD_1 => Ok(VirtualKeyCode::Numpad1),
            Self::NSTD_KEY_NUMPAD_2 => Ok(VirtualKeyCode::Numpad2),
            Self::NSTD_KEY_NUMPAD_3 => Ok(VirtualKeyCode::Numpad3),
            Self::NSTD_KEY_NUMPAD_4 => Ok(VirtualKeyCode::Numpad4),
            Self::NSTD_KEY_NUMPAD_5 => Ok(VirtualKeyCode::Numpad5),
            Self::NSTD_KEY_NUMPAD_6 => Ok(VirtualKeyCode::Numpad6),
            Self::NSTD_KEY_NUMPAD_7 => Ok(VirtualKeyCode::Numpad7),
            Self::NSTD_KEY_NUMPAD_8 => Ok(VirtualKeyCode::Numpad8),
            Self::NSTD_KEY_NUMPAD_9 => Ok(VirtualKeyCode::Numpad9),
            Self::NSTD_KEY_NUMPAD_0 => Ok(VirtualKeyCode::Numpad0),
            // Fallback to implicit conversion.
            _ => Err(self as u32),
        }
    }
}

/// Represents a key state.
#[repr(C)]
#[allow(non_camel_case_types)]
pub enum NSTDKeyState {
    NSTD_KEY_STATE_PRESSED,
    NSTD_KEY_STATE_RELEASED,
}
impl Default for NSTDKeyState {
    #[inline]
    fn default() -> Self {
        Self::NSTD_KEY_STATE_RELEASED
    }
}

/// Represents a key event.
#[repr(C)]
#[derive(Default)]
pub struct NSTDKeyEvent {
    pub key: NSTDKey,
    pub state: NSTDKeyState,
    pub scan_code: u32,
}
