/*
 * Copyright (c) 2021 Nordic Semiconductor ASA
 *
 * SPDX-License-Identifier: LicenseRef-Nordic-5-Clause
 */

/**@file
 * @defgroup nrf_oberon_ecdsa_p224 ECDSA secp224r1 low-level APIs
 * @ingroup nrf_oberon
 * @{
 * @brief Type declarations and APIs to do Elliptic Curve Digital Signature Algorith using the
 *        NIST secp224r1 curve.
 *
 * ECDSA P-224 is a specific implementation of a digital signature scheme.
 */

#ifndef OCRYPTO_ECDSA_P224_H
#define OCRYPTO_ECDSA_P224_H

#include <stddef.h>
#include <stdint.h>


#ifdef __cplusplus
extern "C" {
#endif


/**
 * ECDSA P-224 public key generation.
 *
 * Given a secret key @p sk the corresponding public key is computed and put
 * into @p pk.
 *
 * @param[out] pk Generated public key.
 * @param      sk Secret key. Must be pre-filled with random data.
 *
 * @retval 0  If @p sk is a valid secret key.
 * @retval -1 Otherwise.
 */
int ocrypto_ecdsa_p224_public_key(
    uint8_t pk[56],
    const uint8_t sk[28]);

/**
 * ECDSA P-224 signature generation.
 *
 * The message @p m is signed using the secret key @p sk and the ephemeral
 * session key @p ek. The signature is put into @p sig.
 *
 * @param[out] sig  Generated signature.
 * @param      m    Input message.
 * @param      mlen Length of @p m.
 * @param      sk   Secret key.
 * @param      ek   Session key.
 *
 * @retval 0  If @p ek is a valid session key.
 * @retval -1 Otherwise.
 */
int ocrypto_ecdsa_p224_sign(
    uint8_t sig[56],
    const uint8_t *m, size_t mlen,
    const uint8_t sk[28],
    const uint8_t ek[28]);

/**
 * ECDSA P-224 signature generation from SHA224 hash.
 *
 * The message hash @p hash is signed using the secret key @p sk and the ephemeral
 * session key @p ek. The signature is put into @p sig.
 *
 * @param[out] sig  Generated signature.
 * @param      hash Input hash.
 * @param      sk   Secret key.
 * @param      ek   Session key.
 *
 * @retval 0  If @p ek is a valid session key.
 * @retval -1 Otherwise.
 */
int ocrypto_ecdsa_p224_sign_hash(
    uint8_t sig[56],
    const uint8_t hash[28],
    const uint8_t sk[28],
    const uint8_t ek[28]);

/**
 * ECDSA P-224 signature verification.
 *
 * The signature @p sig of the input message @p m is verified using the signer's
 * public key @p pk.
 *
 * @param sig  Input signature.
 * @param m    Input message.
 * @param mlen Length of @p m.
 * @param pk   Signer's public key.
 *
 * @retval 0  If the signature is valid.
 * @retval -1 Otherwise.
 */
int ocrypto_ecdsa_p224_verify(
    const uint8_t sig[56],
    const uint8_t *m, size_t mlen,
    const uint8_t pk[56]);

/**
 * ECDSA P-224 signature verification from SHA224 hash.
 *
 * The signature @p sig of the message hash @p hash is verified using the signer's
 * public key @p pk.
 *
 * @param sig  Input signature.
 * @param hash Input hash.
 * @param pk   Signer's public key.
 *
 * @retval 0  If the signature is valid.
 * @retval -1 Otherwise.
 */
int ocrypto_ecdsa_p224_verify_hash(
    const uint8_t sig[56],
    const uint8_t hash[28],
    const uint8_t pk[56]);


#ifdef __cplusplus
}
#endif

#endif

/** @} */
