// libnozzle interface for Rust
// Copyright (c) 2021 Red Hat, Inc.
//
// All rights reserved.
//
// Author: Christine Caulfield (ccaulfi@redhat.com)
//


// For the code generated by bindgen
use crate::sys::libnozzle as ffi;

use std::io::{Result, Error, ErrorKind};
use std::os::raw::{c_char, c_void};
use std::ptr::{null_mut};
use libc::free;
use std::fmt;

/// A handle into the nozzle library. Returned from [open] and needed for all other calls
#[derive(Copy, Clone, PartialEq)]
pub struct Handle {
    nozzle_handle: ffi::nozzle_t,
}

const IFNAMSZ: usize = 16;

/// Create a new tap device on the system
pub fn open(devname: &mut String, updownpath: &str) -> Result<Handle>
{
    let mut c_devname: [c_char; IFNAMSZ] = [0; IFNAMSZ];
    let mut c_updownpath: [c_char; libc::PATH_MAX as usize] = [0; libc::PATH_MAX as usize];
    let c_devname_size = IFNAMSZ;

    crate::string_to_bytes(&devname, &mut c_devname)?;
    crate::string_to_bytes(&updownpath, &mut c_updownpath)?;

    let res = unsafe {
	ffi::nozzle_open(c_devname.as_mut_ptr(), c_devname_size, c_updownpath.as_ptr())
    };

    if res.is_null() {
	Err(Error::last_os_error())
    } else {
	let temp = crate::string_from_bytes(c_devname.as_ptr(), IFNAMSZ)?;
	*devname = temp;
	Ok(Handle{nozzle_handle: res})
    }
}

/// Deconfigure and destroy a nozzle device
pub fn close(handle: Handle) -> Result<()>
{
    let res = unsafe {
	ffi::nozzle_close(handle.nozzle_handle)
    };
    if res == 0 {
	Ok(())
    } else {
	Err(Error::last_os_error())
    }
}

/// Which script to run when [run_updown] is called
pub enum Action {
    PreUp,
    Up,
    Down,
    PostDown
}
impl Action {
    pub fn to_u8(self: &Action) -> u8
    {
	match self {
	    Action::PreUp => 0,
	    Action::Up => 1,
	    Action::Down => 2,
	    Action::PostDown => 3,
	}
    }
}

/// Run an up/down script before/after configuring a device. See [Action]
pub fn run_updown(handle: Handle, action: Action) -> Result<String>
{
    let c_exec_string : *mut *mut ::std::os::raw::c_char = &mut [0;0].as_mut_ptr();
    let c_action = action.to_u8();

    let res = unsafe {
	ffi::nozzle_run_updown(handle.nozzle_handle, c_action, c_exec_string)
    };

    match res {
	0 => {
	    unsafe {
		// This is unsafe because we deference a raw pointer
		let resstring = crate::string_from_bytes(*c_exec_string as *mut ::std::os::raw::c_char, libc::PATH_MAX as usize)?;
		free(*c_exec_string as *mut c_void);
		Ok(resstring)
	    }
	},
	-1 => Err(Error::last_os_error()),
	-2 => Err(Error::new(ErrorKind::Other, "error executing shell scripts")),
	_ => Err(Error::new(ErrorKind::Other, "unknown error returned from nozzle_tun_updown()")),
    }
}

/// Mark nozzle device as "up"
pub fn set_up(handle: Handle) -> Result<()>
{
    let res = unsafe {
	ffi::nozzle_set_up(handle.nozzle_handle)
    };
    if res == 0 {
	Ok(())
    } else {
	Err(Error::last_os_error())
    }
}

/// mark nozzle device as "down"
pub fn set_down(handle: Handle) -> Result<()>
{
    let res = unsafe {
	ffi::nozzle_set_down(handle.nozzle_handle)
    };
    if res == 0 {
	Ok(())
    } else {
	Err(Error::last_os_error())
    }
}
const IPADDR_CHAR_MAX: usize = 128;
const PREFIX_CHAR_MAX: usize = 4;

/// Add an ip address to a nozzle device. multiple addresses can be added to one device.
/// The prefix is a the number that comes after the ip address when configuring:
/// eg: 192.168.0.1/24 - the prefix is "24"
pub fn add_ip(handle: Handle, ipaddr: &str, prefix: &str) -> Result<()>
{
    let mut c_ipaddr: [c_char; IPADDR_CHAR_MAX] = [0; IPADDR_CHAR_MAX];
    let mut c_prefix: [c_char; PREFIX_CHAR_MAX] = [0; PREFIX_CHAR_MAX];

    crate::string_to_bytes(&ipaddr, &mut c_ipaddr)?;
    crate::string_to_bytes(&prefix, &mut c_prefix)?;
    let res = unsafe {
	ffi::nozzle_add_ip(handle.nozzle_handle, c_ipaddr.as_ptr(), c_prefix.as_ptr())
    };
    if res == 0 {
	Ok(())
    } else {
	Err(Error::last_os_error())
    }

}

/// remove an ip address from a nozzle device
pub fn del_ip(handle: Handle, ipaddr: &str, prefix: &str) -> Result<()>
{
    let mut c_ipaddr: [c_char; IPADDR_CHAR_MAX] = [0; IPADDR_CHAR_MAX];
    let mut c_prefix: [c_char; PREFIX_CHAR_MAX] = [0; PREFIX_CHAR_MAX];

    crate::string_to_bytes(&ipaddr, &mut c_ipaddr)?;
    crate::string_to_bytes(&prefix, &mut c_prefix)?;
    let res = unsafe {
	ffi::nozzle_del_ip(handle.nozzle_handle, c_ipaddr.as_ptr(), c_prefix.as_ptr())
    };
    if res == 0 {
	Ok(())
    } else {
	Err(Error::last_os_error())
    }
}

/// IpV4/IpV6 in the struct returned from [get_ips]
pub enum Domain {
    IpV4,
    IpV6
}
impl Domain {
    fn new(c_dom: i32) -> Domain
    {
	match c_dom {
	    libc::AF_INET => Domain::IpV4,
	    libc::AF_INET6 => Domain::IpV6,
	    _ => Domain::IpV4,
	}
    }
}
impl fmt::Display for Domain {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
	match self {
	    Domain::IpV4 => write!(f, "IPv4"),
	    Domain::IpV6 => write!(f, "IPv6"),
	}
    }
}

/// IP address info as returned from [get_ips]
pub struct Ip
{
    pub ipaddr: String,
    pub prefix: String,
    pub domain: Domain,
}
impl Ip {
    pub fn new(c_ip: &ffi::nozzle_ip) -> Ip
    {
	Ip {
	    ipaddr: crate::string_from_bytes_safe(c_ip.ipaddr.as_ptr(), IPADDR_CHAR_MAX),
	    prefix: crate::string_from_bytes_safe(c_ip.prefix.as_ptr(), PREFIX_CHAR_MAX),
	    domain: Domain::new(c_ip.domain)
	}
    }
}
impl fmt::Display for Ip {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
	write!(f,"{} {}/{}", self.domain, self.ipaddr, self.prefix)?;
	Ok(())
    }
}

/// Return a Vec of Ip adressess attached to this device
pub fn get_ips(handle: Handle) -> Result<Vec<Ip>>
{
    let mut c_ips : &mut ffi::nozzle_ip = &mut ffi::nozzle_ip{ipaddr: [0;129], prefix: [0;5], domain:0, next: null_mut()};
    let res = unsafe {
	ffi::nozzle_get_ips(handle.nozzle_handle, &mut c_ips as *mut _ as *mut *mut ffi::nozzle_ip)
    };
    let mut ipvec = Vec::<Ip>::new();
    if res == 0 {
	let mut ips : *mut ffi::nozzle_ip = c_ips;
	unsafe {
	    while !ips.is_null() {
		ipvec.push(Ip::new(&*ips));
		ips = (*ips).next;
	    }
	}
	Ok(ipvec)
    } else {
	Err(Error::last_os_error())
    }
}

/// Get the MTU of the device
pub fn get_mtu(handle: Handle) -> Result<i32>
{
    let res = unsafe {
	ffi::nozzle_get_mtu(handle.nozzle_handle)
    };
    if res != -1 {
	Ok(res)
    } else {
	Err(Error::last_os_error())
    }
}

/// Set the MTU of the device
pub fn set_mtu(handle: Handle, new_mtu: i32) -> Result<()>
{
    let res = unsafe {
	ffi::nozzle_set_mtu(handle.nozzle_handle, new_mtu)
    };
    if res != -1 {
	Ok(())
    } else {
	Err(Error::last_os_error())
    }
}


/// Reset the device's MTU back to the default
pub fn reset_mtu(handle: Handle) -> Result<()>
{
    let res = unsafe {
	ffi::nozzle_reset_mtu(handle.nozzle_handle)
    };
    if res != -1 {
	Ok(())
    } else {
	Err(Error::last_os_error())
    }
}

/// Returns the MAC address of the device
pub fn get_mac(handle: Handle) -> Result<String>
{
    let mut c_mac: *mut c_char = null_mut();
    let res = unsafe {
	ffi::nozzle_get_mac(handle.nozzle_handle, &mut c_mac)
    };
    if res == 0 {
	let mac = crate::string_from_bytes(c_mac, 24_usize)?; // Needs to be 8byte aligned
	unsafe { free(c_mac as *mut c_void); }// Was created with strdup(
	Ok(mac)
    } else {
	Err(Error::last_os_error())
    }
}

/// Setsthe MAC address of the device
pub fn set_mac(handle: Handle, ether_addr: &str) -> Result<()>
{
    let mut c_mac: [c_char; 24_usize] = [0; 24_usize]; // Needs to be 8byte aligned
    crate::string_to_bytes(&ether_addr, &mut c_mac)?;
    let res = unsafe {
	ffi::nozzle_set_mac(handle.nozzle_handle, c_mac.as_ptr())
    };
    if res == 0 {
	Ok(())
    } else {
	Err(Error::last_os_error())
    }
}

/// Reset the device's MAC address to the defaut
pub fn reset_mac(handle: Handle) -> Result<()>
{
    let res = unsafe {
	ffi::nozzle_reset_mac(handle.nozzle_handle)
    };
    if res == 0 {
	Ok(())
    } else {
	Err(Error::last_os_error())
    }
}

/// Find the nozzle handle of a device by giving its name
pub fn get_handle_by_name(devname: &str) -> Result<Handle>
{
    let mut c_devname: [c_char; IFNAMSZ] = [0; IFNAMSZ];
    crate::string_to_bytes(&devname, &mut c_devname)?;
    let res = unsafe {
	ffi::nozzle_get_handle_by_name(c_devname.as_ptr())
    };
    if !res.is_null() {
	Ok(Handle{nozzle_handle:res})
    } else {
	Err(Error::last_os_error())
    }
}

/// Return the name of the device
pub fn get_name_by_handle(handle: Handle) -> Result<String>
{
    let res = unsafe {
	ffi::nozzle_get_name_by_handle(handle.nozzle_handle)
    };
    if !res.is_null() {
	crate::string_from_bytes(res, IFNAMSZ)
    } else {
	Err(Error::last_os_error())
    }
}

/// Return a unix FD for the device
pub fn get_fd(handle: Handle) -> Result<i32>
{
    let res = unsafe {
	ffi::nozzle_get_fd(handle.nozzle_handle)
    };
    if res != -1 {
	Ok(res)
    } else {
	Err(Error::last_os_error())
    }
}
