//! Module for letters with the font weight light and size 18.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 18;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 10;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight light and font size 16px
#[inline]
pub const fn get_char(c: char) -> Option<&'static [&'static [u8]]> {
    match c {
        // letter: ' ' / 0x20
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '!' / 0x21
        '!' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 186, 88, 0, 0, 0, 0],
            &[0, 0, 0, 0, 181, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 177, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 173, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 168, 70, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 66, 0, 0, 0, 0],
            &[0, 0, 0, 0, 160, 61, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 75, 33, 0, 0, 0, 0],
            &[0, 0, 0, 0, 213, 115, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '"' / 0x22
        '"' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 103, 0, 38, 66, 0, 0, 0],
            &[0, 0, 0, 227, 0, 83, 148, 0, 0, 0],
            &[0, 0, 0, 203, 0, 71, 136, 0, 0, 0],
            &[0, 0, 0, 180, 0, 59, 124, 0, 0, 0],
            &[0, 0, 0, 112, 0, 34, 80, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '#' / 0x23
        '#' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 36, 0, 37, 42, 0, 0],
            &[0, 0, 0, 143, 54, 0, 120, 68, 0, 0],
            &[0, 0, 0, 185, 12, 0, 164, 26, 0, 0],
            &[0, 0, 0, 200, 0, 0, 191, 0, 0, 0],
            &[19, 184, 193, 229, 184, 189, 234, 184, 135, 0],
            &[0, 0, 63, 132, 0, 45, 150, 0, 0, 0],
            &[0, 0, 107, 88, 0, 89, 107, 0, 0, 0],
            &[42, 65, 173, 103, 65, 160, 117, 65, 13, 0],
            &[79, 122, 224, 122, 122, 220, 127, 122, 25, 0],
            &[0, 0, 195, 0, 0, 196, 0, 0, 0, 0],
            &[0, 25, 168, 0, 13, 183, 0, 0, 0, 0],
            &[0, 68, 124, 0, 57, 139, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '$' / 0x24
        '$' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 155, 29, 0, 0, 0, 0],
            &[0, 0, 110, 215, 237, 213, 211, 98, 0, 0],
            &[0, 80, 194, 13, 153, 26, 12, 32, 0, 0],
            &[0, 139, 99, 0, 153, 26, 0, 0, 0, 0],
            &[0, 101, 168, 1, 153, 26, 0, 0, 0, 0],
            &[0, 4, 159, 193, 199, 31, 0, 0, 0, 0],
            &[0, 0, 0, 40, 200, 225, 141, 16, 0, 0],
            &[0, 0, 0, 0, 153, 35, 120, 201, 1, 0],
            &[0, 0, 0, 0, 153, 26, 0, 206, 34, 0],
            &[0, 0, 0, 0, 153, 26, 6, 222, 12, 0],
            &[0, 94, 139, 91, 181, 114, 191, 119, 0, 0],
            &[0, 16, 85, 122, 209, 127, 43, 0, 0, 0],
            &[0, 0, 0, 0, 142, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '%' / 0x25
        '%' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[3, 97, 126, 41, 0, 0, 0, 90, 1, 0],
            &[144, 103, 44, 191, 26, 0, 60, 156, 0, 0],
            &[202, 0, 0, 97, 104, 0, 185, 31, 0, 0],
            &[201, 0, 0, 87, 114, 65, 151, 0, 0, 0],
            &[178, 37, 0, 162, 56, 187, 28, 0, 0, 0],
            &[30, 164, 170, 113, 69, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 190, 26, 24, 58, 11, 0],
            &[0, 0, 0, 73, 143, 72, 176, 110, 189, 20],
            &[0, 0, 0, 192, 23, 190, 15, 0, 87, 117],
            &[0, 0, 78, 138, 0, 202, 0, 0, 58, 144],
            &[0, 1, 194, 20, 0, 179, 26, 0, 100, 107],
            &[0, 82, 134, 0, 0, 49, 188, 146, 184, 12],
            &[0, 0, 0, 0, 0, 0, 3, 23, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '&' / 0x26
        '&' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 116, 138, 50, 0, 0, 0, 0],
            &[0, 0, 176, 107, 58, 209, 30, 0, 0, 0],
            &[0, 0, 219, 0, 0, 128, 91, 0, 0, 0],
            &[0, 0, 215, 7, 0, 162, 68, 0, 0, 0],
            &[0, 0, 130, 103, 82, 184, 2, 0, 0, 0],
            &[0, 0, 14, 255, 213, 15, 0, 0, 0, 0],
            &[0, 9, 180, 184, 200, 7, 0, 13, 58, 0],
            &[0, 144, 125, 0, 106, 157, 0, 82, 136, 0],
            &[0, 226, 6, 0, 0, 164, 102, 171, 60, 0],
            &[0, 234, 0, 0, 0, 11, 255, 249, 0, 0],
            &[0, 189, 84, 0, 0, 50, 250, 251, 21, 0],
            &[0, 36, 208, 189, 199, 191, 42, 86, 190, 3],
            &[0, 0, 0, 29, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ''' / 0x27
        '\'' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 74, 31, 0, 0, 0, 0],
            &[0, 0, 0, 0, 167, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 155, 52, 0, 0, 0, 0],
            &[0, 0, 0, 0, 143, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 21, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '(' / 0x28
        '(' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 52, 0, 0, 0],
            &[0, 0, 0, 0, 12, 201, 22, 0, 0, 0],
            &[0, 0, 0, 0, 125, 111, 0, 0, 0, 0],
            &[0, 0, 0, 4, 218, 15, 0, 0, 0, 0],
            &[0, 0, 0, 53, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 111, 126, 0, 0, 0, 0, 0],
            &[0, 0, 0, 138, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 154, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 142, 95, 0, 0, 0, 0, 0],
            &[0, 0, 0, 119, 119, 0, 0, 0, 0, 0],
            &[0, 0, 0, 63, 175, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 222, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 202, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 58, 0, 0, 0],
        ]),
        // letter: ')' / 0x29
        ')' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 133, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 104, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 225, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 199, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 184, 52, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 223, 14, 0, 0, 0],
            &[0, 0, 0, 0, 24, 212, 0, 0, 0, 0],
            &[0, 0, 0, 0, 95, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 36, 0, 0, 0, 0],
            &[0, 0, 0, 86, 150, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 11, 0, 0, 0, 0, 0],
        ]),
        // letter: '*' / 0x2a
        '*' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 87, 0, 0, 0, 0],
            &[0, 62, 170, 93, 102, 98, 98, 176, 63, 0],
            &[0, 0, 36, 93, 214, 216, 96, 41, 1, 0],
            &[0, 0, 0, 61, 153, 159, 65, 0, 0, 0],
            &[0, 0, 23, 211, 20, 23, 216, 25, 0, 0],
            &[0, 0, 6, 49, 0, 0, 53, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '+' / 0x2b
        '+' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 51, 0, 0, 0, 0],
            &[0, 6, 8, 8, 148, 57, 8, 8, 3, 0],
            &[0, 150, 188, 188, 226, 201, 188, 188, 75, 0],
            &[0, 0, 0, 0, 145, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ',' / 0x2c
        ',' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 129, 158, 0, 0, 0, 0],
            &[0, 0, 0, 0, 178, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 228, 21, 0, 0, 0, 0],
            &[0, 0, 0, 40, 180, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 17, 0, 0, 0, 0, 0],
        ]),
        // letter: '-' / 0x2d
        '-' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 204, 204, 204, 204, 40, 0, 0],
            &[0, 0, 4, 8, 8, 8, 8, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '.' / 0x2e
        '.' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 33, 0, 0, 0, 0],
            &[0, 0, 0, 0, 217, 115, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '/' / 0x2f
        '/' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 37, 0, 0],
            &[0, 0, 0, 0, 0, 0, 209, 23, 0, 0],
            &[0, 0, 0, 0, 0, 55, 177, 0, 0, 0],
            &[0, 0, 0, 0, 0, 152, 80, 0, 0, 0],
            &[0, 0, 0, 0, 9, 218, 5, 0, 0, 0],
            &[0, 0, 0, 0, 91, 142, 0, 0, 0, 0],
            &[0, 0, 0, 0, 188, 44, 0, 0, 0, 0],
            &[0, 0, 0, 31, 201, 0, 0, 0, 0, 0],
            &[0, 0, 0, 126, 106, 0, 0, 0, 0, 0],
            &[0, 0, 1, 214, 17, 0, 0, 0, 0, 0],
            &[0, 0, 65, 168, 0, 0, 0, 0, 0, 0],
            &[0, 0, 162, 71, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '0' / 0x30
        '0' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 140, 127, 39, 0, 0, 0],
            &[0, 0, 155, 168, 74, 92, 216, 64, 0, 0],
            &[0, 52, 195, 1, 0, 0, 174, 254, 0, 0],
            &[0, 134, 95, 0, 0, 39, 203, 193, 34, 0],
            &[0, 179, 46, 0, 0, 183, 61, 143, 78, 0],
            &[0, 198, 25, 0, 78, 165, 0, 122, 96, 0],
            &[0, 204, 19, 7, 208, 26, 0, 117, 103, 0],
            &[0, 191, 30, 123, 118, 0, 0, 128, 91, 0],
            &[0, 159, 92, 205, 5, 0, 0, 160, 61, 0],
            &[0, 101, 255, 71, 0, 0, 1, 216, 10, 0],
            &[0, 13, 255, 35, 0, 0, 110, 153, 0, 0],
            &[0, 0, 57, 210, 183, 201, 176, 13, 0, 0],
            &[0, 0, 0, 1, 28, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '1' / 0x31
        '1' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 89, 0, 0, 0, 0],
            &[0, 0, 5, 119, 204, 210, 0, 0, 0, 0],
            &[0, 2, 197, 124, 18, 210, 0, 0, 0, 0],
            &[0, 0, 14, 0, 17, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 210, 0, 0, 0, 0],
            &[0, 0, 54, 174, 205, 255, 183, 170, 7, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '2' / 0x32
        '2' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 98, 139, 127, 44, 0, 0, 0],
            &[0, 54, 220, 125, 71, 99, 217, 78, 0, 0],
            &[0, 7, 19, 0, 0, 0, 51, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 236, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 213, 0, 0],
            &[0, 0, 0, 0, 0, 0, 133, 134, 0, 0],
            &[0, 0, 0, 0, 0, 44, 219, 17, 0, 0],
            &[0, 0, 0, 0, 25, 214, 50, 0, 0, 0],
            &[0, 0, 0, 22, 206, 71, 0, 0, 0, 0],
            &[0, 0, 19, 203, 74, 0, 0, 0, 0, 0],
            &[0, 17, 201, 77, 0, 0, 0, 0, 0, 0],
            &[0, 138, 245, 217, 217, 217, 217, 217, 114, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '3' / 0x33
        '3' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 102, 141, 132, 59, 0, 0, 0],
            &[0, 63, 218, 119, 71, 90, 205, 114, 0, 0],
            &[0, 8, 10, 0, 0, 0, 23, 235, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 240, 2, 0],
            &[0, 0, 0, 0, 0, 0, 74, 184, 0, 0],
            &[0, 0, 0, 89, 96, 139, 159, 24, 0, 0],
            &[0, 0, 0, 125, 131, 155, 179, 64, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 219, 26, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 88, 0],
            &[0, 0, 0, 0, 0, 0, 0, 176, 76, 0],
            &[0, 44, 10, 0, 0, 0, 74, 225, 9, 0],
            &[0, 97, 213, 201, 180, 208, 191, 41, 0, 0],
            &[0, 0, 0, 16, 34, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '4' / 0x34
        '4' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 121, 0, 0, 0],
            &[0, 0, 0, 0, 0, 177, 234, 0, 0, 0],
            &[0, 0, 0, 0, 92, 138, 221, 0, 0, 0],
            &[0, 0, 0, 19, 204, 11, 226, 0, 0, 0],
            &[0, 0, 0, 161, 75, 0, 229, 0, 0, 0],
            &[0, 0, 68, 168, 0, 0, 229, 0, 0, 0],
            &[0, 9, 204, 23, 0, 0, 229, 0, 0, 0],
            &[0, 137, 101, 0, 0, 0, 229, 0, 0, 0],
            &[20, 247, 167, 165, 165, 165, 246, 165, 119, 0],
            &[4, 38, 38, 38, 38, 38, 233, 38, 28, 0],
            &[0, 0, 0, 0, 0, 0, 229, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 229, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '5' / 0x35
        '5' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 84, 108, 108, 108, 108, 65, 0, 0],
            &[0, 0, 214, 106, 104, 104, 104, 62, 0, 0],
            &[0, 0, 221, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 216, 0, 0, 0, 0, 0, 0, 0],
            &[0, 24, 195, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 233, 194, 201, 202, 121, 6, 0, 0],
            &[0, 0, 23, 0, 0, 16, 160, 158, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 232, 17, 0],
            &[0, 0, 0, 0, 0, 0, 0, 200, 42, 0],
            &[0, 0, 0, 0, 0, 0, 3, 231, 16, 0],
            &[0, 40, 10, 0, 0, 4, 139, 166, 0, 0],
            &[0, 81, 210, 197, 183, 219, 157, 13, 0, 0],
            &[0, 0, 0, 17, 33, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '6' / 0x36
        '6' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 90, 139, 146, 58, 0, 0],
            &[0, 0, 33, 206, 131, 65, 60, 45, 0, 0],
            &[0, 1, 201, 59, 0, 0, 0, 0, 0, 0],
            &[0, 69, 164, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 86, 0, 10, 17, 0, 0, 0, 0],
            &[0, 182, 75, 173, 174, 182, 201, 56, 0, 0],
            &[0, 199, 173, 19, 0, 0, 34, 216, 14, 0],
            &[0, 196, 73, 0, 0, 0, 0, 151, 77, 0],
            &[0, 169, 58, 0, 0, 0, 0, 128, 102, 0],
            &[0, 109, 124, 0, 0, 0, 0, 164, 71, 0],
            &[0, 15, 218, 45, 0, 0, 51, 216, 6, 0],
            &[0, 0, 52, 207, 188, 188, 201, 44, 0, 0],
            &[0, 0, 0, 0, 28, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '7' / 0x37
        '7' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 53, 108, 108, 108, 108, 108, 108, 43, 0],
            &[0, 51, 104, 104, 104, 104, 104, 207, 75, 0],
            &[0, 0, 0, 0, 0, 0, 14, 216, 3, 0],
            &[0, 0, 0, 0, 0, 0, 106, 130, 0, 0],
            &[0, 0, 0, 0, 0, 0, 208, 30, 0, 0],
            &[0, 0, 0, 0, 0, 60, 180, 0, 0, 0],
            &[0, 0, 0, 0, 0, 164, 77, 0, 0, 0],
            &[0, 0, 0, 0, 20, 220, 3, 0, 0, 0],
            &[0, 0, 0, 0, 118, 128, 0, 0, 0, 0],
            &[0, 0, 0, 1, 218, 28, 0, 0, 0, 0],
            &[0, 0, 0, 72, 178, 0, 0, 0, 0, 0],
            &[0, 0, 0, 176, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '8' / 0x38
        '8' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 102, 142, 125, 40, 0, 0, 0],
            &[0, 15, 210, 123, 64, 90, 212, 71, 0, 0],
            &[0, 102, 143, 0, 0, 0, 57, 188, 0, 0],
            &[0, 113, 122, 0, 0, 0, 49, 192, 0, 0],
            &[0, 30, 208, 39, 0, 6, 177, 89, 0, 0],
            &[0, 0, 46, 197, 158, 197, 84, 0, 0, 0],
            &[0, 0, 88, 190, 118, 197, 135, 7, 0, 0],
            &[0, 90, 189, 9, 0, 0, 123, 172, 0, 0],
            &[0, 197, 44, 0, 0, 0, 0, 220, 27, 0],
            &[0, 209, 33, 0, 0, 0, 0, 205, 35, 0],
            &[0, 133, 145, 1, 0, 0, 74, 206, 1, 0],
            &[0, 6, 152, 206, 169, 194, 185, 31, 0, 0],
            &[0, 0, 0, 9, 34, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '9' / 0x39
        '9' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 104, 145, 122, 29, 0, 0, 0],
            &[0, 17, 207, 123, 62, 92, 218, 50, 0, 0],
            &[0, 136, 113, 0, 0, 0, 49, 199, 0, 0],
            &[0, 200, 35, 0, 0, 0, 0, 193, 33, 0],
            &[0, 203, 24, 0, 0, 0, 0, 156, 78, 0],
            &[0, 157, 80, 0, 0, 0, 6, 216, 94, 0],
            &[0, 36, 212, 78, 27, 58, 168, 177, 89, 0],
            &[0, 0, 29, 131, 163, 128, 25, 170, 62, 0],
            &[0, 0, 0, 0, 0, 0, 1, 221, 13, 0],
            &[0, 0, 0, 0, 0, 0, 75, 175, 0, 0],
            &[0, 0, 0, 0, 0, 49, 217, 40, 0, 0],
            &[0, 0, 196, 172, 199, 194, 49, 0, 0, 0],
            &[0, 0, 17, 36, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ':' / 0x3a
        ':' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 114, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 189, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 217, 112, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ';' / 0x3b
        ';' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 153, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 199, 77, 0, 0, 0, 0],
            &[0, 0, 0, 6, 235, 9, 0, 0, 0, 0],
            &[0, 0, 0, 60, 159, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 14, 0, 0, 0, 0, 0],
        ]),
        // letter: '<' / 0x3c
        '<' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 39, 36, 0],
            &[0, 0, 0, 0, 0, 34, 159, 171, 27, 0],
            &[0, 0, 0, 29, 153, 171, 48, 0, 0, 0],
            &[0, 25, 146, 170, 50, 0, 0, 0, 0, 0],
            &[0, 140, 197, 40, 0, 0, 0, 0, 0, 0],
            &[0, 0, 46, 166, 169, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 158, 179, 63, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 148, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '=' / 0x3d
        '=' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 142, 161, 161, 161, 161, 161, 161, 77, 0],
            &[0, 45, 51, 51, 51, 51, 51, 51, 24, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 111, 126, 126, 126, 126, 126, 126, 60, 0],
            &[0, 79, 90, 90, 90, 90, 90, 90, 43, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '>' / 0x3e
        '>' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 82, 196, 108, 6, 0, 0, 0, 0, 0],
            &[0, 0, 4, 99, 192, 101, 4, 0, 0, 0],
            &[0, 0, 0, 0, 5, 100, 189, 95, 3, 0],
            &[0, 0, 0, 0, 0, 2, 92, 228, 54, 0],
            &[0, 0, 0, 6, 99, 195, 119, 13, 0, 0],
            &[0, 10, 111, 198, 109, 9, 0, 0, 0, 0],
            &[0, 138, 99, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '?' / 0x3f
        '?' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 50, 115, 142, 123, 39, 0, 0, 0],
            &[0, 17, 177, 100, 68, 97, 217, 69, 0, 0],
            &[0, 0, 0, 0, 0, 0, 60, 185, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 195, 0, 0],
            &[0, 0, 0, 0, 0, 0, 106, 138, 0, 0],
            &[0, 0, 0, 0, 0, 85, 199, 14, 0, 0],
            &[0, 0, 0, 0, 110, 180, 12, 0, 0, 0],
            &[0, 0, 0, 20, 209, 4, 0, 0, 0, 0],
            &[0, 0, 0, 57, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 78, 0, 0, 0, 0, 0],
            &[0, 0, 0, 108, 220, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '@' / 0x40
        '@' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 91, 93, 38, 0, 0, 0],
            &[0, 1, 138, 183, 104, 91, 163, 145, 1, 0],
            &[0, 124, 125, 0, 0, 0, 0, 117, 110, 0],
            &[20, 196, 0, 0, 35, 59, 17, 2, 197, 1],
            &[102, 102, 0, 128, 164, 125, 220, 0, 147, 45],
            &[156, 41, 36, 197, 0, 0, 206, 0, 111, 76],
            &[182, 16, 100, 125, 0, 4, 204, 0, 105, 86],
            &[188, 7, 113, 102, 0, 29, 194, 0, 121, 69],
            &[168, 27, 80, 136, 0, 89, 212, 0, 171, 18],
            &[123, 80, 7, 199, 141, 175, 124, 165, 150, 0],
            &[43, 174, 0, 7, 44, 4, 1, 38, 0, 0],
            &[0, 156, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 154, 174, 103, 90, 124, 174, 0, 0],
            &[0, 0, 0, 32, 85, 98, 65, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'A' / 0x41
        'A' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 43, 0, 0, 0, 0],
            &[0, 0, 0, 13, 233, 158, 0, 0, 0, 0],
            &[0, 0, 0, 94, 136, 215, 7, 0, 0, 0],
            &[0, 0, 0, 183, 48, 147, 80, 0, 0, 0],
            &[0, 0, 21, 212, 0, 59, 170, 0, 0, 0],
            &[0, 0, 105, 129, 0, 1, 217, 12, 0, 0],
            &[0, 0, 195, 41, 0, 0, 141, 92, 0, 0],
            &[0, 29, 255, 196, 196, 196, 249, 196, 0, 0],
            &[0, 117, 135, 20, 20, 20, 21, 221, 20, 0],
            &[0, 206, 35, 0, 0, 0, 0, 135, 104, 0],
            &[40, 202, 0, 0, 0, 0, 0, 48, 193, 0],
            &[129, 115, 0, 0, 0, 0, 0, 0, 215, 28],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'B' / 0x42
        'B' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 34, 108, 108, 105, 84, 36, 0, 0, 0],
            &[0, 81, 189, 100, 103, 124, 199, 155, 0, 0],
            &[0, 81, 147, 0, 0, 0, 3, 204, 56, 0],
            &[0, 81, 147, 0, 0, 0, 0, 155, 87, 0],
            &[0, 81, 147, 0, 0, 0, 6, 209, 35, 0],
            &[0, 81, 190, 102, 104, 126, 173, 89, 0, 0],
            &[0, 81, 192, 106, 107, 124, 171, 138, 4, 0],
            &[0, 81, 147, 0, 0, 0, 0, 155, 111, 0],
            &[0, 81, 147, 0, 0, 0, 0, 78, 168, 0],
            &[0, 81, 147, 0, 0, 0, 0, 109, 148, 0],
            &[0, 81, 147, 0, 0, 0, 58, 226, 51, 0],
            &[0, 81, 237, 212, 214, 223, 188, 62, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'C' / 0x43
        'C' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 89, 137, 136, 100, 21, 0],
            &[0, 0, 44, 216, 142, 76, 77, 125, 123, 0],
            &[0, 12, 220, 59, 0, 0, 0, 0, 0, 0],
            &[0, 109, 150, 0, 0, 0, 0, 0, 0, 0],
            &[0, 182, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 219, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 228, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 209, 35, 0, 0, 0, 0, 0, 0, 0],
            &[0, 162, 87, 0, 0, 0, 0, 0, 0, 0],
            &[0, 77, 190, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 183, 142, 9, 0, 0, 3, 17, 0],
            &[0, 0, 8, 145, 227, 193, 197, 222, 85, 0],
            &[0, 0, 0, 0, 3, 29, 25, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'D' / 0x44
        'D' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 108, 108, 90, 45, 0, 0, 0, 0],
            &[0, 171, 140, 101, 122, 188, 192, 26, 0, 0],
            &[0, 171, 65, 0, 0, 0, 91, 208, 5, 0],
            &[0, 171, 65, 0, 0, 0, 0, 169, 89, 0],
            &[0, 171, 65, 0, 0, 0, 0, 85, 161, 0],
            &[0, 171, 65, 0, 0, 0, 0, 51, 189, 0],
            &[0, 171, 65, 0, 0, 0, 0, 41, 196, 0],
            &[0, 171, 65, 0, 0, 0, 0, 63, 175, 0],
            &[0, 171, 65, 0, 0, 0, 0, 122, 120, 0],
            &[0, 171, 65, 0, 0, 0, 15, 219, 32, 0],
            &[0, 171, 65, 0, 0, 54, 201, 104, 0, 0],
            &[0, 171, 223, 213, 213, 179, 66, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'E' / 0x45
        'E' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 34, 108, 108, 108, 108, 108, 108, 22, 0],
            &[0, 81, 196, 104, 104, 104, 104, 104, 21, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 240, 217, 217, 217, 217, 217, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'F' / 0x46
        'F' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 34, 108, 108, 108, 108, 108, 108, 22, 0],
            &[0, 81, 196, 104, 104, 104, 104, 104, 21, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 239, 212, 212, 212, 212, 192, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'G' / 0x47
        'G' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 119, 145, 123, 51, 0, 0],
            &[0, 0, 114, 213, 106, 71, 96, 170, 8, 0],
            &[0, 64, 209, 14, 0, 0, 0, 0, 0, 0],
            &[0, 185, 72, 0, 0, 0, 0, 0, 0, 0],
            &[7, 236, 5, 0, 0, 0, 0, 0, 0, 0],
            &[31, 211, 0, 0, 0, 0, 0, 0, 0, 0],
            &[46, 198, 0, 0, 0, 202, 210, 210, 60, 0],
            &[31, 213, 0, 0, 0, 5, 6, 161, 73, 0],
            &[5, 237, 9, 0, 0, 0, 0, 159, 73, 0],
            &[0, 172, 94, 0, 0, 0, 0, 159, 73, 0],
            &[0, 42, 225, 55, 0, 0, 0, 163, 73, 0],
            &[0, 0, 57, 201, 206, 183, 213, 195, 39, 0],
            &[0, 0, 0, 0, 15, 32, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'H' / 0x48
        'H' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 69, 31, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 144, 106, 106, 106, 106, 202, 73, 0],
            &[0, 171, 147, 110, 110, 110, 110, 203, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'I' / 0x49
        'I' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 108, 108, 108, 108, 47, 0, 0],
            &[0, 0, 55, 79, 200, 131, 73, 27, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 146, 188, 236, 211, 182, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'J' / 0x4a
        'J' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 91, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 216, 21, 0, 0],
            &[0, 0, 0, 0, 0, 1, 236, 6, 0, 0],
            &[0, 14, 9, 0, 0, 82, 194, 0, 0, 0],
            &[0, 41, 214, 188, 186, 217, 44, 0, 0, 0],
            &[0, 0, 2, 31, 36, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'K' / 0x4b
        'K' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 75, 46, 0],
            &[0, 171, 65, 0, 0, 0, 82, 198, 9, 0],
            &[0, 171, 65, 0, 0, 56, 215, 23, 0, 0],
            &[0, 171, 65, 0, 34, 220, 44, 0, 0, 0],
            &[0, 171, 65, 18, 213, 71, 0, 0, 0, 0],
            &[0, 171, 70, 193, 148, 0, 0, 0, 0, 0],
            &[0, 171, 199, 130, 218, 25, 0, 0, 0, 0],
            &[0, 171, 134, 0, 88, 179, 0, 0, 0, 0],
            &[0, 171, 65, 0, 0, 172, 95, 0, 0, 0],
            &[0, 171, 65, 0, 0, 22, 219, 27, 0, 0],
            &[0, 171, 65, 0, 0, 0, 86, 182, 0, 0],
            &[0, 171, 65, 0, 0, 0, 0, 171, 98, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'L' / 0x4c
        'L' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 223, 221, 221, 221, 221, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'M' / 0x4d
        'M' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[13, 108, 30, 0, 0, 0, 0, 75, 78, 0],
            &[30, 245, 137, 0, 0, 0, 6, 227, 184, 0],
            &[30, 191, 212, 1, 0, 0, 79, 176, 184, 0],
            &[30, 194, 157, 63, 0, 0, 170, 92, 184, 0],
            &[30, 198, 65, 155, 0, 14, 208, 41, 184, 0],
            &[30, 198, 1, 210, 7, 96, 125, 40, 184, 0],
            &[30, 198, 0, 136, 81, 187, 33, 40, 184, 0],
            &[30, 198, 0, 43, 196, 193, 0, 40, 184, 0],
            &[30, 198, 0, 0, 197, 100, 0, 40, 184, 0],
            &[30, 198, 0, 0, 0, 0, 0, 40, 184, 0],
            &[30, 198, 0, 0, 0, 0, 0, 40, 184, 0],
            &[30, 198, 0, 0, 0, 0, 0, 40, 184, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'N' / 0x4e
        'N' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 65, 0, 0, 0, 0, 67, 31, 0],
            &[0, 171, 224, 9, 0, 0, 0, 159, 73, 0],
            &[0, 171, 154, 112, 0, 0, 0, 159, 73, 0],
            &[0, 171, 63, 208, 8, 0, 0, 159, 73, 0],
            &[0, 171, 56, 119, 107, 0, 0, 159, 73, 0],
            &[0, 171, 57, 13, 208, 6, 0, 159, 73, 0],
            &[0, 171, 57, 0, 125, 102, 0, 159, 73, 0],
            &[0, 171, 57, 0, 15, 207, 4, 159, 73, 0],
            &[0, 171, 57, 0, 0, 131, 97, 157, 73, 0],
            &[0, 171, 57, 0, 0, 18, 206, 154, 73, 0],
            &[0, 171, 57, 0, 0, 0, 136, 228, 73, 0],
            &[0, 171, 57, 0, 0, 0, 21, 245, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'O' / 0x4f
        'O' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 85, 140, 133, 51, 0, 0, 0],
            &[0, 5, 184, 156, 73, 85, 200, 103, 0, 0],
            &[0, 112, 152, 0, 0, 0, 13, 220, 27, 0],
            &[0, 212, 32, 0, 0, 0, 0, 129, 115, 0],
            &[15, 227, 0, 0, 0, 0, 0, 71, 169, 0],
            &[38, 201, 0, 0, 0, 0, 0, 45, 195, 0],
            &[47, 194, 0, 0, 0, 0, 0, 40, 200, 0],
            &[32, 209, 0, 0, 0, 0, 0, 55, 185, 0],
            &[3, 237, 4, 0, 0, 0, 0, 95, 146, 0],
            &[0, 176, 77, 0, 0, 0, 0, 174, 75, 0],
            &[0, 55, 215, 27, 0, 0, 85, 203, 2, 0],
            &[0, 0, 84, 219, 190, 205, 188, 27, 0, 0],
            &[0, 0, 0, 3, 31, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'P' / 0x50
        'P' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 34, 108, 108, 108, 96, 51, 0, 0, 0],
            &[0, 81, 194, 100, 101, 122, 191, 189, 11, 0],
            &[0, 81, 155, 0, 0, 0, 0, 165, 116, 0],
            &[0, 81, 155, 0, 0, 0, 0, 85, 168, 0],
            &[0, 81, 155, 0, 0, 0, 0, 89, 156, 0],
            &[0, 81, 155, 0, 0, 0, 5, 191, 78, 0],
            &[0, 81, 202, 120, 123, 146, 214, 127, 0, 0],
            &[0, 81, 191, 92, 89, 70, 16, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Q' / 0x51
        'Q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 85, 140, 133, 51, 0, 0, 0],
            &[0, 5, 184, 156, 73, 85, 200, 103, 0, 0],
            &[0, 112, 152, 0, 0, 0, 13, 220, 27, 0],
            &[0, 212, 32, 0, 0, 0, 0, 129, 115, 0],
            &[15, 227, 0, 0, 0, 0, 0, 71, 169, 0],
            &[38, 201, 0, 0, 0, 0, 0, 45, 195, 0],
            &[47, 194, 0, 0, 0, 0, 0, 40, 200, 0],
            &[32, 209, 0, 0, 0, 0, 0, 55, 185, 0],
            &[3, 237, 4, 0, 0, 0, 0, 95, 146, 0],
            &[0, 176, 77, 0, 0, 0, 0, 174, 75, 0],
            &[0, 55, 215, 27, 0, 0, 85, 203, 2, 0],
            &[0, 0, 84, 219, 190, 250, 255, 27, 0, 0],
            &[0, 0, 0, 3, 31, 33, 221, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 150, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 183, 21, 0],
        ]),
        // letter: 'R' / 0x52
        'R' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[34, 108, 108, 104, 82, 24, 0, 0, 0, 0],
            &[81, 194, 100, 107, 137, 219, 118, 0, 0, 0],
            &[81, 155, 0, 0, 0, 18, 228, 34, 0, 0],
            &[81, 155, 0, 0, 0, 0, 169, 81, 0, 0],
            &[81, 155, 0, 0, 0, 0, 183, 66, 0, 0],
            &[81, 155, 0, 0, 0, 72, 212, 6, 0, 0],
            &[81, 228, 186, 188, 215, 151, 25, 0, 0, 0],
            &[81, 166, 26, 26, 163, 98, 0, 0, 0, 0],
            &[81, 155, 0, 0, 24, 219, 16, 0, 0, 0],
            &[81, 155, 0, 0, 0, 114, 147, 0, 0, 0],
            &[81, 155, 0, 0, 0, 4, 210, 48, 0, 0],
            &[81, 155, 0, 0, 0, 0, 67, 196, 1, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'S' / 0x53
        'S' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 71, 132, 141, 116, 49, 0, 0],
            &[0, 0, 173, 176, 94, 72, 103, 169, 8, 0],
            &[0, 62, 193, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 147, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 218, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 224, 122, 21, 0, 0, 0, 0],
            &[0, 0, 0, 21, 126, 222, 158, 28, 0, 0],
            &[0, 0, 0, 0, 0, 5, 112, 223, 18, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 93, 0],
            &[0, 0, 0, 0, 0, 0, 0, 149, 97, 0],
            &[0, 26, 3, 0, 0, 0, 45, 228, 27, 0],
            &[0, 105, 216, 196, 173, 197, 196, 61, 0, 0],
            &[0, 0, 0, 16, 33, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'T' / 0x54
        'T' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[25, 108, 108, 108, 108, 108, 108, 108, 92, 0],
            &[24, 104, 104, 104, 204, 144, 104, 104, 88, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'U' / 0x55
        'U' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 69, 31, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 161, 75, 0, 0, 0, 0, 174, 63, 0],
            &[0, 127, 118, 0, 0, 0, 0, 218, 26, 0],
            &[0, 42, 220, 21, 0, 0, 94, 189, 0, 0],
            &[0, 0, 96, 221, 186, 204, 192, 26, 0, 0],
            &[0, 0, 0, 4, 31, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'V' / 0x56
        'V' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[43, 62, 0, 0, 0, 0, 0, 0, 103, 2],
            &[41, 205, 0, 0, 0, 0, 0, 51, 200, 0],
            &[0, 212, 33, 0, 0, 0, 0, 132, 115, 0],
            &[0, 127, 116, 0, 0, 0, 0, 213, 31, 0],
            &[0, 41, 199, 0, 0, 0, 40, 202, 0, 0],
            &[0, 0, 211, 27, 0, 0, 122, 117, 0, 0],
            &[0, 0, 127, 109, 0, 0, 203, 33, 0, 0],
            &[0, 0, 41, 192, 0, 30, 203, 0, 0, 0],
            &[0, 0, 0, 209, 22, 111, 119, 0, 0, 0],
            &[0, 0, 0, 127, 98, 189, 35, 0, 0, 0],
            &[0, 0, 0, 41, 182, 200, 0, 0, 0, 0],
            &[0, 0, 0, 0, 212, 122, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'W' / 0x57
        'W' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[48, 47, 0, 0, 0, 0, 0, 0, 88, 6],
            &[88, 135, 0, 0, 0, 0, 0, 0, 224, 0],
            &[53, 169, 0, 0, 0, 0, 0, 12, 212, 0],
            &[19, 203, 0, 0, 92, 42, 0, 46, 178, 0],
            &[0, 221, 0, 0, 205, 127, 0, 81, 144, 0],
            &[0, 205, 15, 28, 154, 174, 0, 115, 110, 0],
            &[0, 170, 49, 87, 97, 177, 3, 149, 76, 0],
            &[0, 135, 82, 146, 39, 132, 48, 181, 42, 0],
            &[0, 101, 113, 185, 0, 74, 108, 205, 9, 0],
            &[0, 66, 147, 174, 0, 17, 167, 203, 0, 0],
            &[0, 31, 217, 121, 0, 0, 177, 189, 0, 0],
            &[0, 2, 249, 63, 0, 0, 154, 162, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'X' / 0x58
        'X' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 93, 16, 0, 0, 0, 0, 50, 57, 0],
            &[0, 109, 146, 0, 0, 0, 7, 209, 32, 0],
            &[0, 3, 201, 44, 0, 0, 125, 120, 0, 0],
            &[0, 0, 53, 188, 0, 32, 204, 5, 0, 0],
            &[0, 0, 0, 150, 84, 175, 62, 0, 0, 0],
            &[0, 0, 0, 16, 217, 160, 0, 0, 0, 0],
            &[0, 0, 0, 9, 217, 140, 0, 0, 0, 0],
            &[0, 0, 0, 133, 107, 189, 45, 0, 0, 0],
            &[0, 0, 42, 199, 2, 45, 193, 1, 0, 0],
            &[0, 1, 192, 55, 0, 0, 143, 100, 0, 0],
            &[0, 97, 154, 0, 0, 0, 14, 212, 20, 0],
            &[20, 216, 19, 0, 0, 0, 0, 92, 158, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Y' / 0x59
        'Y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[21, 88, 0, 0, 0, 0, 0, 20, 88, 0],
            &[1, 208, 46, 0, 0, 0, 0, 137, 114, 0],
            &[0, 77, 172, 0, 0, 0, 20, 216, 8, 0],
            &[0, 0, 197, 46, 0, 0, 134, 103, 0, 0],
            &[0, 0, 65, 172, 0, 18, 207, 4, 0, 0],
            &[0, 0, 0, 185, 46, 132, 91, 0, 0, 0],
            &[0, 0, 0, 54, 184, 197, 2, 0, 0, 0],
            &[0, 0, 0, 0, 185, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Z' / 0x5a
        'Z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 101, 108, 108, 108, 108, 98, 0, 0],
            &[0, 0, 93, 100, 100, 100, 141, 196, 0, 0],
            &[0, 0, 0, 0, 0, 0, 176, 67, 0, 0],
            &[0, 0, 0, 0, 0, 55, 188, 0, 0, 0],
            &[0, 0, 0, 0, 0, 189, 55, 0, 0, 0],
            &[0, 0, 0, 0, 69, 176, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 44, 0, 0, 0, 0],
            &[0, 0, 0, 83, 163, 0, 0, 0, 0, 0],
            &[0, 0, 3, 209, 34, 0, 0, 0, 0, 0],
            &[0, 0, 97, 150, 0, 0, 0, 0, 0, 0],
            &[0, 7, 215, 25, 0, 0, 0, 0, 0, 0],
            &[0, 76, 246, 217, 217, 217, 217, 217, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '[' / 0x5b
        '[' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 108, 108, 108, 25, 0, 0, 0],
            &[0, 0, 45, 208, 88, 88, 20, 0, 0, 0],
            &[0, 0, 45, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 202, 65, 65, 15, 0, 0, 0],
            &[0, 0, 23, 135, 135, 135, 31, 0, 0, 0],
        ]),
        // letter: '\' / 0x5c
        '\\' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 80, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 112, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 206, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 182, 50, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 219, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 205, 27, 0, 0, 0],
            &[0, 0, 0, 0, 0, 111, 121, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 212, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 173, 59, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ']' / 0x5d
        ']' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 79, 128, 128, 102, 0, 0, 0],
            &[0, 0, 0, 44, 71, 90, 202, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 202, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 202, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 202, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 202, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 202, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 202, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 202, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 202, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 202, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 202, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 202, 0, 0, 0],
            &[0, 0, 0, 50, 81, 99, 202, 0, 0, 0],
            &[0, 0, 0, 70, 114, 114, 90, 0, 0, 0],
        ]),
        // letter: '^' / 0x5e
        '^' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 25, 0, 0, 0, 0],
            &[0, 0, 0, 17, 209, 149, 0, 0, 0, 0],
            &[0, 0, 0, 131, 74, 177, 31, 0, 0, 0],
            &[0, 0, 18, 188, 0, 55, 155, 0, 0, 0],
            &[0, 0, 134, 76, 0, 0, 178, 35, 0, 0],
            &[0, 19, 191, 0, 0, 0, 54, 160, 0, 0],
            &[0, 136, 78, 0, 0, 0, 0, 178, 39, 0],
            &[2, 133, 0, 0, 0, 0, 0, 48, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '_' / 0x5f
        '_' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[188, 188, 188, 188, 188, 188, 188, 188, 188, 113],
            &[28, 28, 28, 28, 28, 28, 28, 28, 28, 17],
        ]),
        // letter: '`' / 0x60
        '`' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 197, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 9, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'a' / 0x61
        'a' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 125, 160, 149, 83, 0, 0, 0],
            &[0, 0, 144, 81, 39, 53, 175, 125, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 0, 0],
            &[0, 0, 71, 174, 187, 182, 180, 245, 0, 0],
            &[0, 85, 190, 29, 0, 0, 0, 224, 0, 0],
            &[0, 171, 59, 0, 0, 0, 14, 249, 0, 0],
            &[0, 148, 100, 0, 0, 4, 146, 240, 0, 0],
            &[0, 27, 202, 184, 169, 187, 65, 202, 0, 0],
            &[0, 0, 0, 23, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'b' / 0x62
        'b' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 97, 48, 145, 163, 105, 5, 0, 0],
            &[0, 131, 155, 171, 58, 51, 154, 178, 0, 0],
            &[0, 131, 219, 3, 0, 0, 0, 201, 64, 0],
            &[0, 131, 136, 0, 0, 0, 0, 122, 129, 0],
            &[0, 131, 107, 0, 0, 0, 0, 97, 151, 0],
            &[0, 131, 117, 0, 0, 0, 0, 112, 141, 0],
            &[0, 131, 161, 0, 0, 0, 0, 161, 94, 0],
            &[0, 131, 226, 46, 0, 0, 52, 226, 14, 0],
            &[0, 131, 89, 176, 183, 186, 212, 57, 0, 0],
            &[0, 0, 0, 0, 22, 30, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'c' / 0x63
        'c' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 41, 130, 163, 151, 93, 1, 0],
            &[0, 0, 99, 212, 94, 49, 61, 111, 0, 0],
            &[0, 30, 224, 18, 0, 0, 0, 0, 0, 0],
            &[0, 111, 136, 0, 0, 0, 0, 0, 0, 0],
            &[0, 142, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 105, 0, 0, 0, 0, 0, 0, 0],
            &[0, 88, 167, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 217, 81, 0, 0, 0, 7, 0, 0],
            &[0, 0, 33, 186, 208, 174, 192, 199, 0, 0],
            &[0, 0, 0, 0, 13, 34, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'd' / 0x64
        'd' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 32, 5, 0],
            &[0, 0, 0, 0, 0, 0, 0, 204, 32, 0],
            &[0, 0, 0, 0, 0, 0, 0, 204, 32, 0],
            &[0, 0, 0, 0, 0, 0, 0, 204, 32, 0],
            &[0, 0, 26, 128, 162, 127, 19, 203, 32, 0],
            &[0, 31, 224, 110, 42, 70, 179, 204, 32, 0],
            &[0, 155, 116, 0, 0, 0, 52, 255, 32, 0],
            &[0, 223, 27, 0, 0, 0, 0, 233, 32, 0],
            &[0, 243, 2, 0, 0, 0, 0, 209, 32, 0],
            &[0, 241, 6, 0, 0, 0, 0, 220, 32, 0],
            &[0, 202, 48, 0, 0, 0, 15, 249, 32, 0],
            &[0, 107, 177, 6, 0, 0, 136, 229, 32, 0],
            &[0, 2, 143, 216, 172, 196, 101, 172, 32, 0],
            &[0, 0, 0, 12, 32, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'e' / 0x65
        'e' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 92, 154, 158, 94, 2, 0, 0],
            &[0, 3, 183, 149, 53, 43, 147, 161, 0, 0],
            &[0, 105, 156, 0, 0, 0, 0, 193, 52, 0],
            &[0, 191, 51, 0, 0, 0, 0, 122, 115, 0],
            &[0, 224, 199, 196, 196, 196, 196, 205, 114, 0],
            &[0, 217, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 83, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 214, 32, 0, 0, 0, 15, 12, 0],
            &[0, 0, 66, 201, 191, 170, 195, 197, 32, 0],
            &[0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'f' / 0x66
        'f' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 49, 53, 23, 0, 0],
            &[0, 0, 0, 60, 221, 159, 146, 166, 0, 0],
            &[0, 0, 0, 192, 70, 0, 0, 0, 0, 0],
            &[0, 0, 0, 225, 15, 0, 0, 0, 0, 0],
            &[73, 112, 123, 241, 136, 131, 131, 64, 0, 0],
            &[53, 73, 73, 235, 80, 73, 73, 36, 0, 0],
            &[0, 0, 0, 227, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 227, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 227, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 227, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 227, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 227, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 227, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'g' / 0x67
        'g' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 131, 165, 134, 26, 85, 16, 0],
            &[0, 27, 222, 107, 42, 64, 174, 206, 32, 0],
            &[0, 151, 112, 0, 0, 0, 29, 252, 32, 0],
            &[0, 222, 25, 0, 0, 0, 0, 211, 32, 0],
            &[0, 243, 1, 0, 0, 0, 0, 190, 32, 0],
            &[0, 242, 8, 0, 0, 0, 0, 200, 32, 0],
            &[0, 201, 54, 0, 0, 0, 6, 238, 32, 0],
            &[0, 103, 183, 8, 0, 0, 123, 233, 32, 0],
            &[0, 1, 138, 218, 173, 196, 118, 191, 32, 0],
            &[0, 0, 0, 11, 34, 9, 0, 199, 28, 0],
            &[0, 0, 0, 0, 0, 0, 7, 228, 6, 0],
            &[0, 57, 169, 203, 203, 205, 126, 6, 0, 0],
        ]),
        // letter: 'h' / 0x68
        'h' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 106, 41, 137, 159, 107, 6, 0, 0],
            &[0, 131, 151, 159, 51, 45, 155, 172, 0, 0],
            &[0, 131, 204, 2, 0, 0, 3, 228, 19, 0],
            &[0, 131, 128, 0, 0, 0, 0, 189, 50, 0],
            &[0, 131, 107, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'i' / 0x69
        'i' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 139, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 131, 131, 131, 72, 0, 0, 0, 0],
            &[0, 11, 46, 59, 139, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 193, 180, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'j' / 0x6a
        'j' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 218, 23, 0, 0],
            &[0, 0, 0, 0, 0, 19, 140, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 105, 131, 131, 131, 4, 0, 0],
            &[0, 0, 0, 33, 53, 66, 236, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 229, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 229, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 229, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 229, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 229, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 229, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 229, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 229, 8, 0, 0],
            &[0, 0, 0, 0, 0, 5, 240, 1, 0, 0],
            &[0, 0, 166, 203, 212, 164, 24, 0, 0, 0],
        ]),
        // letter: 'k' / 0x6b
        'k' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 229, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 229, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 229, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 229, 0, 0, 0, 0, 94, 54, 0],
            &[0, 8, 229, 0, 0, 0, 113, 174, 3, 0],
            &[0, 8, 229, 0, 0, 102, 184, 6, 0, 0],
            &[0, 8, 227, 0, 91, 193, 10, 0, 0, 0],
            &[0, 8, 221, 79, 215, 162, 0, 0, 0, 0],
            &[0, 8, 240, 171, 8, 173, 93, 0, 0, 0],
            &[0, 8, 236, 5, 0, 18, 214, 37, 0, 0],
            &[0, 8, 229, 0, 0, 0, 63, 203, 6, 0],
            &[0, 8, 229, 0, 0, 0, 0, 133, 143, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'l' / 0x6c
        'l' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 40, 40, 40, 22, 0, 0, 0, 0],
            &[0, 43, 137, 149, 195, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 193, 180, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'm' / 0x6d
        'm' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 84, 70, 159, 71, 22, 149, 127, 5, 0],
            &[10, 210, 142, 69, 220, 158, 75, 167, 106, 0],
            &[10, 250, 18, 0, 174, 128, 0, 60, 156, 0],
            &[10, 224, 0, 0, 152, 85, 0, 42, 175, 0],
            &[10, 207, 0, 0, 149, 67, 0, 40, 176, 0],
            &[10, 206, 0, 0, 149, 67, 0, 40, 176, 0],
            &[10, 206, 0, 0, 149, 67, 0, 40, 176, 0],
            &[10, 206, 0, 0, 149, 67, 0, 40, 176, 0],
            &[10, 206, 0, 0, 149, 67, 0, 40, 176, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'n' / 0x6e
        'n' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 67, 34, 45, 137, 160, 112, 8, 0, 0],
            &[0, 131, 144, 159, 55, 44, 146, 180, 0, 0],
            &[0, 131, 207, 1, 0, 0, 1, 224, 22, 0],
            &[0, 131, 133, 0, 0, 0, 0, 188, 51, 0],
            &[0, 131, 108, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'o' / 0x6f
        'o' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 110, 161, 151, 75, 0, 0, 0],
            &[0, 17, 209, 121, 46, 53, 172, 137, 0, 0],
            &[0, 142, 121, 0, 0, 0, 2, 206, 45, 0],
            &[0, 220, 24, 0, 0, 0, 0, 121, 121, 0],
            &[0, 239, 0, 0, 0, 0, 0, 91, 148, 0],
            &[0, 236, 3, 0, 0, 0, 0, 102, 140, 0],
            &[0, 188, 57, 0, 0, 0, 0, 160, 88, 0],
            &[0, 75, 195, 10, 0, 0, 58, 216, 8, 0],
            &[0, 0, 99, 211, 170, 186, 197, 39, 0, 0],
            &[0, 0, 0, 3, 32, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'p' / 0x70
        'p' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 67, 35, 42, 136, 163, 110, 7, 0, 0],
            &[0, 131, 134, 165, 58, 48, 143, 185, 1, 0],
            &[0, 131, 214, 2, 0, 0, 0, 192, 68, 0],
            &[0, 131, 134, 0, 0, 0, 0, 119, 130, 0],
            &[0, 131, 108, 0, 0, 0, 0, 96, 151, 0],
            &[0, 131, 122, 0, 0, 0, 0, 111, 139, 0],
            &[0, 131, 168, 0, 0, 0, 0, 157, 94, 0],
            &[0, 131, 220, 53, 0, 0, 46, 224, 15, 0],
            &[0, 131, 110, 160, 184, 184, 211, 58, 0, 0],
            &[0, 131, 102, 0, 11, 28, 2, 0, 0, 0],
            &[0, 131, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 110, 85, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'q' / 0x71
        'q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 131, 164, 128, 21, 85, 16, 0],
            &[0, 27, 219, 102, 42, 77, 185, 197, 32, 0],
            &[0, 151, 109, 0, 0, 0, 57, 255, 32, 0],
            &[0, 222, 25, 0, 0, 0, 0, 235, 32, 0],
            &[0, 244, 1, 0, 0, 0, 0, 210, 32, 0],
            &[0, 242, 7, 0, 0, 0, 0, 221, 32, 0],
            &[0, 201, 51, 0, 0, 0, 17, 250, 32, 0],
            &[0, 104, 179, 5, 0, 1, 143, 229, 32, 0],
            &[0, 2, 143, 215, 172, 196, 93, 201, 32, 0],
            &[0, 0, 0, 12, 33, 6, 0, 204, 32, 0],
            &[0, 0, 0, 0, 0, 0, 0, 204, 32, 0],
            &[0, 0, 0, 0, 0, 0, 0, 171, 27, 0],
        ]),
        // letter: 'r' / 0x72
        'r' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[56, 131, 131, 104, 6, 112, 160, 136, 6, 0],
            &[17, 50, 66, 214, 155, 110, 53, 81, 0, 0],
            &[0, 0, 0, 249, 109, 0, 0, 0, 0, 0],
            &[0, 0, 0, 255, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 241, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 0, 0, 0, 0, 0, 0],
            &[121, 184, 197, 252, 197, 184, 134, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 's' / 0x73
        's' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 107, 160, 161, 127, 40, 0, 0],
            &[0, 0, 175, 135, 46, 43, 85, 63, 0, 0],
            &[0, 0, 238, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 179, 132, 9, 0, 0, 0, 0, 0],
            &[0, 0, 5, 122, 212, 146, 39, 0, 0, 0],
            &[0, 0, 0, 0, 3, 79, 211, 85, 0, 0],
            &[0, 0, 0, 0, 0, 0, 60, 196, 0, 0],
            &[0, 12, 17, 0, 0, 0, 92, 170, 0, 0],
            &[0, 31, 204, 198, 171, 191, 196, 33, 0, 0],
            &[0, 0, 0, 12, 34, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 't' / 0x74
        't' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 151, 0, 0, 0, 0, 0],
            &[0, 92, 120, 174, 204, 131, 131, 131, 0, 0],
            &[0, 64, 73, 134, 181, 73, 73, 73, 0, 0],
            &[0, 0, 0, 85, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 79, 166, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 224, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 219, 170, 185, 9, 0],
            &[0, 0, 0, 0, 0, 11, 35, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'u' / 0x75
        'u' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 77, 44, 0, 0, 0, 0, 104, 16, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 86, 0, 0, 0, 0, 212, 32, 0],
            &[0, 134, 105, 0, 0, 0, 8, 245, 32, 0],
            &[0, 79, 190, 4, 0, 0, 126, 229, 32, 0],
            &[0, 0, 150, 214, 168, 188, 92, 170, 32, 0],
            &[0, 0, 0, 9, 29, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'v' / 0x76
        'v' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 113, 12, 0, 0, 0, 0, 66, 60, 0],
            &[0, 147, 95, 0, 0, 0, 0, 200, 44, 0],
            &[0, 51, 189, 0, 0, 0, 38, 202, 0, 0],
            &[0, 0, 208, 29, 0, 0, 131, 105, 0, 0],
            &[0, 0, 113, 121, 0, 1, 214, 16, 0, 0],
            &[0, 0, 22, 209, 0, 62, 165, 0, 0, 0],
            &[0, 0, 0, 176, 53, 155, 68, 0, 0, 0],
            &[0, 0, 0, 79, 153, 208, 1, 0, 0, 0],
            &[0, 0, 0, 5, 231, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'w' / 0x77
        'w' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[80, 33, 0, 0, 0, 0, 0, 0, 81, 30],
            &[116, 105, 0, 0, 0, 0, 0, 0, 196, 21],
            &[64, 158, 0, 0, 191, 117, 0, 3, 216, 0],
            &[13, 210, 0, 23, 184, 191, 0, 43, 176, 0],
            &[0, 212, 12, 99, 109, 188, 21, 94, 126, 0],
            &[0, 163, 62, 176, 35, 117, 95, 145, 76, 0],
            &[0, 110, 112, 203, 0, 41, 168, 187, 26, 0],
            &[0, 58, 194, 140, 0, 0, 198, 194, 0, 0],
            &[0, 9, 250, 64, 0, 0, 146, 180, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'x' / 0x78
        'x' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 78, 59, 0, 0, 0, 0, 110, 25, 0],
            &[0, 29, 214, 22, 0, 0, 100, 163, 0, 0],
            &[0, 0, 80, 179, 0, 41, 207, 12, 0, 0],
            &[0, 0, 0, 149, 115, 201, 49, 0, 0, 0],
            &[0, 0, 0, 8, 233, 134, 0, 0, 0, 0],
            &[0, 0, 0, 95, 176, 209, 20, 0, 0, 0],
            &[0, 0, 40, 208, 13, 79, 180, 1, 0, 0],
            &[0, 8, 203, 52, 0, 0, 149, 115, 0, 0],
            &[0, 151, 117, 0, 0, 0, 8, 205, 54, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'y' / 0x79
        'y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 106, 15, 0, 0, 0, 0, 64, 60, 0],
            &[0, 130, 104, 0, 0, 0, 0, 195, 45, 0],
            &[0, 31, 201, 0, 0, 0, 32, 203, 0, 0],
            &[0, 0, 182, 47, 0, 0, 123, 108, 0, 0],
            &[0, 0, 81, 146, 0, 0, 208, 18, 0, 0],
            &[0, 0, 4, 213, 8, 51, 171, 0, 0, 0],
            &[0, 0, 0, 133, 88, 141, 75, 0, 0, 0],
            &[0, 0, 0, 33, 178, 202, 3, 0, 0, 0],
            &[0, 0, 0, 0, 185, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 186, 41, 0, 0, 0, 0],
            &[0, 0, 0, 33, 196, 0, 0, 0, 0, 0],
            &[0, 156, 209, 124, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'z' / 0x7a
        'z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 131, 131, 131, 131, 131, 118, 0, 0],
            &[0, 0, 61, 61, 61, 61, 150, 167, 0, 0],
            &[0, 0, 0, 0, 0, 37, 211, 15, 0, 0],
            &[0, 0, 0, 0, 5, 199, 57, 0, 0, 0],
            &[0, 0, 0, 0, 139, 121, 0, 0, 0, 0],
            &[0, 0, 0, 70, 186, 2, 0, 0, 0, 0],
            &[0, 0, 22, 210, 25, 0, 0, 0, 0, 0],
            &[0, 1, 181, 73, 0, 0, 0, 0, 0, 0],
            &[0, 76, 248, 208, 208, 208, 208, 195, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '{' / 0x7b
        '{' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 93, 21, 0, 0],
            &[0, 0, 0, 0, 67, 222, 117, 18, 0, 0],
            &[0, 0, 0, 0, 165, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 178, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 178, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 183, 51, 0, 0, 0, 0],
            &[0, 0, 0, 27, 222, 14, 0, 0, 0, 0],
            &[0, 0, 120, 238, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 220, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 184, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 178, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 178, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 168, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 206, 87, 13, 0, 0],
            &[0, 0, 0, 0, 0, 49, 117, 26, 0, 0],
        ]),
        // letter: '|' / 0x7c
        '|' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 42, 0, 0, 0, 0],
        ]),
        // letter: '}' / 0x7d
        '}' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 61, 73, 6, 0, 0, 0, 0, 0],
            &[0, 0, 60, 143, 205, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 180, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 157, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 157, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 80, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 153, 1, 0, 0, 0],
            &[0, 0, 0, 0, 2, 167, 234, 31, 0, 0],
            &[0, 0, 0, 0, 90, 173, 14, 0, 0, 0],
            &[0, 0, 0, 0, 152, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 157, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 157, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 179, 66, 0, 0, 0, 0],
            &[0, 0, 44, 125, 218, 10, 0, 0, 0, 0],
            &[0, 0, 77, 102, 17, 0, 0, 0, 0, 0],
        ]),
        // letter: '~' / 0x7e
        '~' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 85, 5, 0, 0, 48, 29, 0],
            &[0, 68, 153, 102, 182, 19, 0, 157, 35, 0],
            &[0, 156, 22, 0, 47, 192, 134, 168, 0, 0],
            &[0, 44, 0, 0, 0, 10, 46, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ' ' / 0xa0
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¡' / 0xa1
        '¡' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 96, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 72, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 68, 0, 0, 0, 0],
            &[0, 0, 0, 0, 179, 72, 0, 0, 0, 0],
            &[0, 0, 0, 0, 183, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 188, 81, 0, 0, 0, 0],
            &[0, 0, 0, 0, 184, 82, 0, 0, 0, 0],
        ]),
        // letter: '¢' / 0xa2
        '¢' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 196, 44, 5, 0, 0],
            &[0, 0, 25, 186, 190, 146, 163, 165, 0, 0],
            &[0, 2, 204, 87, 0, 0, 0, 0, 0, 0],
            &[0, 65, 184, 0, 0, 0, 0, 0, 0, 0],
            &[0, 115, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 124, 113, 0, 0, 0, 0, 0, 0, 0],
            &[0, 95, 149, 0, 0, 0, 0, 0, 0, 0],
            &[0, 24, 227, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 100, 214, 100, 68, 88, 116, 0, 0],
            &[0, 0, 0, 38, 125, 219, 122, 54, 0, 0],
            &[0, 0, 0, 0, 14, 174, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '£' / 0xa3
        '£' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 121, 130, 78, 2, 0],
            &[0, 0, 0, 54, 211, 87, 73, 141, 47, 0],
            &[0, 0, 0, 167, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 202, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 208, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 208, 24, 0, 0, 0, 0, 0],
            &[0, 70, 184, 242, 191, 184, 173, 0, 0, 0],
            &[0, 0, 0, 208, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 209, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 232, 4, 0, 0, 0, 0, 0],
            &[0, 0, 103, 145, 0, 0, 0, 0, 0, 0],
            &[0, 125, 255, 218, 217, 217, 217, 217, 111, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¤' / 0xa4
        '¤' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 36, 0, 0, 0, 0, 0, 21, 12, 0],
            &[0, 132, 146, 150, 185, 185, 108, 198, 46, 0],
            &[0, 2, 212, 55, 0, 2, 128, 132, 0, 0],
            &[0, 63, 128, 0, 0, 0, 0, 191, 0, 0],
            &[0, 85, 99, 0, 0, 0, 0, 188, 0, 0],
            &[0, 29, 172, 0, 0, 0, 30, 178, 0, 0],
            &[0, 24, 217, 167, 79, 100, 209, 155, 0, 0],
            &[0, 133, 52, 42, 109, 94, 13, 136, 54, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¥' / 0xa5
        '¥' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[29, 77, 0, 0, 0, 0, 0, 11, 96, 0],
            &[5, 211, 34, 0, 0, 0, 0, 126, 123, 0],
            &[0, 79, 168, 0, 0, 0, 24, 212, 8, 0],
            &[0, 0, 190, 54, 0, 0, 154, 87, 0, 0],
            &[0, 0, 50, 192, 0, 43, 194, 0, 0, 0],
            &[0, 0, 0, 160, 80, 180, 53, 0, 0, 0],
            &[0, 15, 75, 103, 234, 205, 75, 62, 0, 0],
            &[0, 21, 104, 104, 202, 141, 104, 86, 0, 0],
            &[0, 2, 10, 10, 169, 71, 10, 8, 0, 0],
            &[0, 35, 169, 169, 225, 191, 169, 140, 0, 0],
            &[0, 0, 0, 0, 165, 63, 0, 0, 0, 0],
            &[0, 0, 0, 0, 165, 63, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¦' / 0xa6
        '¦' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 124, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 143, 44, 0, 0, 0, 0],
        ]),
        // letter: '§' / 0xa7
        '§' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 32, 38, 12, 0, 0, 0],
            &[0, 0, 63, 194, 145, 142, 178, 103, 0, 0],
            &[0, 0, 198, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 174, 102, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 203, 207, 106, 13, 0, 0, 0],
            &[0, 0, 126, 123, 23, 97, 204, 60, 0, 0],
            &[0, 0, 201, 0, 0, 0, 33, 197, 0, 0],
            &[0, 0, 185, 79, 0, 0, 22, 185, 0, 0],
            &[0, 0, 15, 158, 186, 103, 180, 47, 0, 0],
            &[0, 0, 0, 0, 21, 117, 207, 48, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 181, 0, 0],
            &[0, 7, 14, 0, 0, 0, 79, 158, 0, 0],
            &[0, 18, 185, 185, 165, 181, 171, 21, 0, 0],
            &[0, 0, 0, 0, 13, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¨' / 0xa8
        '¨' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 87, 0, 4, 131, 5, 0, 0],
            &[0, 0, 87, 135, 0, 12, 196, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '©' / 0xa9
        '©' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 147, 146, 150, 134, 18, 0, 0],
            &[0, 83, 130, 6, 44, 57, 39, 172, 19, 0],
            &[7, 162, 0, 153, 111, 94, 46, 31, 139, 0],
            &[60, 95, 49, 122, 0, 0, 0, 0, 154, 0],
            &[75, 80, 73, 94, 0, 0, 0, 0, 150, 0],
            &[36, 128, 23, 163, 3, 0, 0, 1, 154, 0],
            &[0, 161, 29, 65, 148, 143, 36, 105, 73, 0],
            &[0, 20, 161, 111, 58, 71, 144, 101, 0, 0],
            &[0, 0, 0, 43, 84, 72, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ª' / 0xaa
        'ª' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 82, 130, 114, 17, 0, 0, 0],
            &[0, 0, 0, 72, 22, 68, 162, 0, 0, 0],
            &[0, 0, 0, 19, 73, 99, 189, 0, 0, 0],
            &[0, 0, 34, 167, 59, 36, 182, 0, 0, 0],
            &[0, 0, 85, 116, 0, 42, 205, 0, 0, 0],
            &[0, 0, 9, 153, 149, 103, 111, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '«' / 0xab
        '«' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 74, 0, 12, 68, 0, 0],
            &[0, 0, 0, 157, 96, 1, 173, 73, 0, 0],
            &[0, 0, 109, 151, 0, 130, 127, 0, 0, 0],
            &[0, 44, 214, 6, 63, 199, 1, 0, 0, 0],
            &[0, 4, 185, 72, 9, 198, 55, 0, 0, 0],
            &[0, 0, 17, 200, 31, 27, 202, 21, 0, 0],
            &[0, 0, 0, 40, 129, 0, 55, 118, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¬' / 0xac
        '¬' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 8, 8, 8, 8, 8, 8, 0, 0],
            &[0, 180, 188, 188, 188, 188, 188, 232, 28, 0],
            &[0, 0, 0, 0, 0, 0, 0, 167, 28, 0],
            &[0, 0, 0, 0, 0, 0, 0, 167, 28, 0],
            &[0, 0, 0, 0, 0, 0, 0, 154, 26, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '®' / 0xae
        '®' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 147, 146, 150, 134, 18, 0, 0],
            &[0, 83, 130, 71, 101, 72, 36, 172, 19, 0],
            &[7, 162, 0, 149, 38, 107, 117, 31, 139, 0],
            &[60, 95, 0, 147, 0, 38, 138, 0, 154, 0],
            &[75, 80, 0, 155, 132, 194, 15, 0, 150, 0],
            &[36, 128, 0, 147, 0, 128, 43, 1, 154, 0],
            &[0, 161, 29, 120, 0, 11, 133, 105, 73, 0],
            &[0, 20, 161, 111, 58, 71, 144, 101, 0, 0],
            &[0, 0, 0, 43, 84, 72, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¯' / 0xaf
        '¯' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 80, 196, 196, 196, 196, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '°' / 0xb0
        '°' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 45, 137, 115, 9, 0, 0, 0],
            &[0, 0, 44, 167, 35, 68, 165, 0, 0, 0],
            &[0, 0, 126, 53, 0, 0, 151, 24, 0, 0],
            &[0, 0, 101, 93, 0, 0, 178, 10, 0, 0],
            &[0, 0, 5, 156, 159, 178, 85, 0, 0, 0],
            &[0, 0, 0, 0, 15, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '±' / 0xb1
        '±' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 57, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 42, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 42, 0, 0, 0, 0],
            &[0, 10, 12, 12, 158, 53, 12, 12, 4, 0],
            &[0, 149, 180, 180, 225, 192, 180, 180, 72, 0],
            &[0, 0, 0, 0, 153, 42, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 42, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 42, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 0, 0, 0, 0, 0],
            &[0, 157, 196, 196, 196, 196, 196, 196, 78, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '²' / 0xb2
        '²' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 86, 154, 139, 27, 0, 0, 0],
            &[0, 0, 26, 102, 14, 49, 184, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 194, 0, 0, 0],
            &[0, 0, 0, 0, 0, 97, 117, 0, 0, 0],
            &[0, 0, 0, 0, 76, 154, 1, 0, 0, 0],
            &[0, 0, 0, 75, 154, 2, 0, 0, 0, 0],
            &[0, 0, 66, 215, 94, 92, 92, 12, 0, 0],
            &[0, 0, 44, 88, 88, 88, 88, 11, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '³' / 0xb3
        '³' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 93, 148, 130, 21, 0, 0, 0],
            &[0, 0, 40, 91, 21, 73, 176, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 192, 0, 0, 0],
            &[0, 0, 0, 75, 144, 195, 52, 0, 0, 0],
            &[0, 0, 0, 11, 24, 74, 183, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 199, 2, 0, 0],
            &[0, 0, 80, 96, 63, 118, 164, 0, 0, 0],
            &[0, 0, 12, 79, 104, 70, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '´' / 0xb4
        '´' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 65, 6, 0, 0, 0],
            &[0, 0, 0, 0, 125, 153, 0, 0, 0, 0],
            &[0, 0, 0, 72, 163, 3, 0, 0, 0, 0],
            &[0, 0, 0, 57, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'µ' / 0xb5
        'µ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 77, 44, 0, 0, 0, 0, 104, 16, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 247, 140, 0, 0, 0, 0, 212, 32, 0],
            &[0, 255, 191, 0, 0, 0, 8, 245, 32, 0],
            &[0, 230, 255, 4, 0, 0, 126, 229, 32, 0],
            &[0, 152, 236, 214, 168, 188, 92, 170, 32, 0],
            &[0, 151, 85, 9, 29, 2, 0, 0, 0, 0],
            &[0, 151, 85, 0, 0, 0, 0, 0, 0, 0],
            &[0, 127, 72, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¶' / 0xb6
        '¶' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 40, 40, 40, 40, 7, 0],
            &[0, 28, 205, 255, 255, 187, 126, 188, 45, 0],
            &[0, 163, 255, 255, 255, 120, 0, 122, 45, 0],
            &[0, 228, 255, 255, 255, 120, 0, 122, 45, 0],
            &[0, 245, 255, 255, 255, 120, 0, 122, 45, 0],
            &[0, 226, 255, 255, 255, 120, 0, 122, 45, 0],
            &[0, 159, 255, 255, 255, 120, 0, 122, 45, 0],
            &[0, 27, 208, 255, 255, 120, 0, 122, 45, 0],
            &[0, 0, 0, 22, 57, 120, 0, 122, 45, 0],
            &[0, 0, 0, 0, 47, 120, 0, 122, 45, 0],
            &[0, 0, 0, 0, 47, 120, 0, 122, 45, 0],
            &[0, 0, 0, 0, 47, 120, 0, 122, 45, 0],
            &[0, 0, 0, 0, 47, 120, 0, 122, 45, 0],
            &[0, 0, 0, 0, 47, 120, 0, 122, 45, 0],
            &[0, 0, 0, 0, 47, 120, 0, 122, 45, 0],
            &[0, 0, 0, 0, 1, 3, 0, 3, 1, 0],
        ]),
        // letter: '·' / 0xb7
        '·' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 193, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¸' / 0xb8
        '¸' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 123, 52, 0, 0, 0, 0],
            &[0, 0, 0, 0, 98, 180, 39, 0, 0, 0],
            &[0, 0, 0, 85, 170, 147, 7, 0, 0, 0],
        ]),
        // letter: '¹' / 0xb9
        '¹' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 97, 48, 0, 0, 0, 0],
            &[0, 0, 47, 192, 168, 96, 0, 0, 0, 0],
            &[0, 0, 14, 13, 91, 96, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 96, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 96, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 96, 0, 0, 0, 0],
            &[0, 0, 2, 63, 168, 172, 62, 2, 0, 0],
            &[0, 0, 3, 88, 119, 121, 88, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'º' / 0xba
        'º' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 128, 112, 9, 0, 0, 0],
            &[0, 0, 17, 185, 32, 80, 159, 0, 0, 0],
            &[0, 0, 83, 99, 0, 0, 193, 0, 0, 0],
            &[0, 0, 90, 90, 0, 0, 184, 1, 0, 0],
            &[0, 0, 31, 167, 1, 30, 174, 0, 0, 0],
            &[0, 0, 0, 78, 157, 149, 25, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '»' / 0xbb
        '»' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 0, 1, 39, 0, 0, 0, 0],
            &[0, 2, 180, 61, 6, 189, 45, 0, 0, 0],
            &[0, 0, 20, 204, 32, 30, 204, 20, 0, 0],
            &[0, 0, 0, 54, 207, 3, 74, 191, 0, 0],
            &[0, 0, 0, 110, 152, 0, 134, 130, 0, 0],
            &[0, 0, 59, 188, 5, 79, 175, 1, 0, 0],
            &[0, 3, 186, 21, 10, 190, 13, 0, 0, 0],
            &[0, 0, 2, 0, 0, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¼' / 0xbc
        '¼' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 48, 69, 0, 0, 0, 0, 81, 10, 0],
            &[124, 169, 163, 0, 0, 0, 35, 181, 0, 0],
            &[7, 23, 163, 0, 0, 0, 162, 54, 0, 0],
            &[0, 24, 163, 0, 0, 42, 174, 0, 0, 0],
            &[0, 24, 163, 0, 0, 170, 46, 0, 0, 0],
            &[48, 151, 255, 115, 50, 166, 0, 0, 0, 0],
            &[13, 35, 53, 31, 178, 38, 0, 109, 35, 0],
            &[0, 0, 0, 58, 158, 0, 76, 216, 61, 0],
            &[0, 0, 0, 184, 32, 23, 165, 123, 61, 0],
            &[0, 0, 67, 149, 0, 167, 35, 127, 67, 1],
            &[0, 0, 190, 25, 15, 155, 155, 203, 179, 21],
            &[0, 76, 140, 0, 0, 0, 0, 122, 61, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '½' / 0xbd
        '½' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 48, 69, 0, 0, 0, 0, 81, 10, 0],
            &[124, 169, 163, 0, 0, 0, 35, 181, 0, 0],
            &[7, 23, 163, 0, 0, 0, 162, 54, 0, 0],
            &[0, 24, 163, 0, 0, 42, 174, 0, 0, 0],
            &[0, 24, 163, 0, 0, 170, 46, 0, 0, 0],
            &[48, 151, 255, 115, 50, 166, 0, 0, 0, 0],
            &[13, 35, 53, 31, 178, 54, 128, 162, 70, 0],
            &[0, 0, 0, 58, 158, 31, 54, 19, 210, 3],
            &[0, 0, 0, 184, 32, 0, 0, 7, 194, 0],
            &[0, 0, 67, 149, 0, 0, 2, 159, 65, 0],
            &[0, 0, 190, 25, 0, 3, 156, 72, 0, 0],
            &[0, 76, 140, 0, 0, 123, 224, 180, 180, 33],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¾' / 0xbe
        '¾' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[21, 113, 133, 42, 0, 0, 0, 81, 10, 0],
            &[53, 61, 51, 209, 0, 0, 35, 181, 0, 0],
            &[0, 20, 71, 180, 0, 0, 162, 54, 0, 0],
            &[0, 119, 172, 139, 0, 42, 174, 0, 0, 0],
            &[0, 0, 0, 204, 10, 170, 46, 0, 0, 0],
            &[104, 128, 137, 175, 50, 166, 0, 0, 0, 0],
            &[3, 45, 39, 0, 178, 38, 0, 109, 35, 0],
            &[0, 0, 0, 58, 158, 0, 76, 216, 61, 0],
            &[0, 0, 0, 184, 32, 23, 165, 123, 61, 0],
            &[0, 0, 67, 149, 0, 167, 35, 127, 67, 1],
            &[0, 0, 190, 25, 15, 155, 155, 203, 179, 21],
            &[0, 76, 140, 0, 0, 0, 0, 122, 61, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¿' / 0xbf
        '¿' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 80, 0, 0, 0, 0],
            &[0, 0, 0, 0, 148, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 100, 0, 0, 0, 0],
            &[0, 0, 0, 27, 212, 26, 0, 0, 0, 0],
            &[0, 0, 49, 207, 57, 0, 0, 0, 0, 0],
            &[0, 25, 213, 31, 0, 0, 0, 0, 0, 0],
            &[0, 118, 113, 0, 0, 0, 0, 0, 0, 0],
            &[0, 146, 86, 0, 0, 0, 0, 0, 0, 0],
            &[0, 93, 178, 5, 0, 0, 27, 13, 0, 0],
            &[0, 0, 0, 4, 24, 6, 0, 0, 0, 0],
        ]),
        // letter: 'À' / 0xc0
        'À' => Some(&[
            &[0, 0, 3, 183, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 182, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 43, 0, 0, 0, 0],
            &[0, 0, 0, 13, 233, 158, 0, 0, 0, 0],
            &[0, 0, 0, 94, 136, 215, 7, 0, 0, 0],
            &[0, 0, 0, 183, 48, 147, 80, 0, 0, 0],
            &[0, 0, 21, 212, 0, 59, 170, 0, 0, 0],
            &[0, 0, 105, 129, 0, 1, 217, 12, 0, 0],
            &[0, 0, 195, 41, 0, 0, 141, 92, 0, 0],
            &[0, 29, 255, 196, 196, 196, 249, 196, 0, 0],
            &[0, 117, 135, 20, 20, 20, 21, 221, 20, 0],
            &[0, 206, 35, 0, 0, 0, 0, 135, 104, 0],
            &[40, 202, 0, 0, 0, 0, 0, 48, 193, 0],
            &[129, 115, 0, 0, 0, 0, 0, 0, 215, 28],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 41, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Á' / 0xc1
        'Á' => Some(&[
            &[0, 0, 0, 0, 0, 121, 160, 0, 0, 0],
            &[0, 0, 0, 0, 67, 171, 4, 0, 0, 0],
            &[0, 0, 0, 0, 75, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 43, 0, 0, 0, 0],
            &[0, 0, 0, 13, 233, 158, 0, 0, 0, 0],
            &[0, 0, 0, 94, 136, 215, 7, 0, 0, 0],
            &[0, 0, 0, 183, 48, 147, 80, 0, 0, 0],
            &[0, 0, 21, 212, 0, 59, 170, 0, 0, 0],
            &[0, 0, 105, 129, 0, 1, 217, 12, 0, 0],
            &[0, 0, 195, 41, 0, 0, 141, 92, 0, 0],
            &[0, 29, 255, 196, 196, 196, 249, 196, 0, 0],
            &[0, 117, 135, 20, 20, 20, 21, 221, 20, 0],
            &[0, 206, 35, 0, 0, 0, 0, 135, 104, 0],
            &[40, 202, 0, 0, 0, 0, 0, 48, 193, 0],
            &[129, 115, 0, 0, 0, 0, 0, 0, 215, 28],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 42, 2, 0, 0],
        ]),
        // letter: 'Â' / 0xc2
        'Â' => Some(&[
            &[0, 0, 0, 37, 212, 179, 1, 0, 0, 0],
            &[0, 0, 15, 182, 29, 101, 130, 0, 0, 0],
            &[0, 0, 48, 30, 0, 0, 68, 10, 0, 0],
            &[0, 0, 0, 0, 92, 43, 0, 0, 0, 0],
            &[0, 0, 0, 13, 233, 158, 0, 0, 0, 0],
            &[0, 0, 0, 94, 136, 215, 7, 0, 0, 0],
            &[0, 0, 0, 183, 48, 147, 80, 0, 0, 0],
            &[0, 0, 21, 212, 0, 59, 170, 0, 0, 0],
            &[0, 0, 105, 129, 0, 1, 217, 12, 0, 0],
            &[0, 0, 195, 41, 0, 0, 141, 92, 0, 0],
            &[0, 29, 255, 196, 196, 196, 249, 196, 0, 0],
            &[0, 117, 135, 20, 20, 20, 21, 221, 20, 0],
            &[0, 206, 35, 0, 0, 0, 0, 135, 104, 0],
            &[40, 202, 0, 0, 0, 0, 0, 48, 193, 0],
            &[129, 115, 0, 0, 0, 0, 0, 0, 215, 28],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 12, 0, 0, 0, 0],
        ]),
        // letter: 'Ã' / 0xc3
        'Ã' => Some(&[
            &[0, 0, 23, 116, 58, 0, 21, 66, 0, 0],
            &[0, 0, 170, 85, 158, 161, 180, 57, 0, 0],
            &[0, 0, 48, 0, 0, 44, 38, 0, 0, 0],
            &[0, 0, 0, 0, 92, 43, 0, 0, 0, 0],
            &[0, 0, 0, 13, 233, 158, 0, 0, 0, 0],
            &[0, 0, 0, 94, 136, 215, 7, 0, 0, 0],
            &[0, 0, 0, 183, 48, 147, 80, 0, 0, 0],
            &[0, 0, 21, 212, 0, 59, 170, 0, 0, 0],
            &[0, 0, 105, 129, 0, 1, 217, 12, 0, 0],
            &[0, 0, 195, 41, 0, 0, 141, 92, 0, 0],
            &[0, 29, 255, 196, 196, 196, 249, 196, 0, 0],
            &[0, 117, 135, 20, 20, 20, 21, 221, 20, 0],
            &[0, 206, 35, 0, 0, 0, 0, 135, 104, 0],
            &[40, 202, 0, 0, 0, 0, 0, 48, 193, 0],
            &[129, 115, 0, 0, 0, 0, 0, 0, 215, 28],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ä' / 0xc4
        'Ä' => Some(&[
            &[0, 0, 43, 71, 0, 2, 108, 3, 0, 0],
            &[0, 0, 98, 151, 0, 14, 218, 16, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 43, 0, 0, 0, 0],
            &[0, 0, 0, 13, 233, 158, 0, 0, 0, 0],
            &[0, 0, 0, 94, 136, 215, 7, 0, 0, 0],
            &[0, 0, 0, 183, 48, 147, 80, 0, 0, 0],
            &[0, 0, 21, 212, 0, 59, 170, 0, 0, 0],
            &[0, 0, 105, 129, 0, 1, 217, 12, 0, 0],
            &[0, 0, 195, 41, 0, 0, 141, 92, 0, 0],
            &[0, 29, 255, 196, 196, 196, 249, 196, 0, 0],
            &[0, 117, 135, 20, 20, 20, 21, 221, 20, 0],
            &[0, 206, 35, 0, 0, 0, 0, 135, 104, 0],
            &[40, 202, 0, 0, 0, 0, 0, 48, 193, 0],
            &[129, 115, 0, 0, 0, 0, 0, 0, 215, 28],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Å' / 0xc5
        'Å' => Some(&[
            &[0, 0, 0, 178, 0, 68, 109, 0, 0, 0],
            &[0, 0, 0, 160, 97, 163, 61, 0, 0, 0],
            &[0, 0, 0, 5, 73, 42, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 43, 0, 0, 0, 0],
            &[0, 0, 0, 13, 233, 158, 0, 0, 0, 0],
            &[0, 0, 0, 94, 136, 215, 7, 0, 0, 0],
            &[0, 0, 0, 183, 48, 147, 80, 0, 0, 0],
            &[0, 0, 21, 212, 0, 59, 170, 0, 0, 0],
            &[0, 0, 105, 129, 0, 1, 217, 12, 0, 0],
            &[0, 0, 195, 41, 0, 0, 141, 92, 0, 0],
            &[0, 29, 255, 196, 196, 196, 249, 196, 0, 0],
            &[0, 117, 135, 20, 20, 20, 21, 221, 20, 0],
            &[0, 206, 35, 0, 0, 0, 0, 135, 104, 0],
            &[40, 202, 0, 0, 0, 0, 0, 48, 193, 0],
            &[129, 115, 0, 0, 0, 0, 0, 0, 215, 28],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 86, 167, 166, 20, 0, 0, 0],
        ]),
        // letter: 'Æ' / 0xc6
        'Æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 108, 108, 108, 108, 108, 2],
            &[0, 0, 0, 162, 168, 202, 104, 104, 104, 2],
            &[0, 0, 2, 217, 66, 165, 0, 0, 0, 0],
            &[0, 0, 55, 169, 63, 165, 0, 0, 0, 0],
            &[0, 0, 129, 96, 63, 165, 0, 0, 0, 0],
            &[0, 0, 202, 25, 63, 203, 106, 106, 80, 0],
            &[0, 23, 205, 0, 63, 203, 106, 106, 80, 0],
            &[0, 96, 255, 196, 255, 177, 0, 0, 0, 0],
            &[0, 170, 72, 12, 75, 166, 0, 0, 0, 0],
            &[4, 224, 4, 0, 63, 165, 0, 0, 0, 0],
            &[62, 172, 0, 0, 63, 165, 0, 0, 0, 0],
            &[136, 99, 0, 0, 63, 242, 217, 217, 217, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ç' / 0xc7
        'Ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 89, 137, 136, 100, 21, 0],
            &[0, 0, 44, 216, 142, 76, 77, 125, 123, 0],
            &[0, 12, 220, 59, 0, 0, 0, 0, 0, 0],
            &[0, 109, 150, 0, 0, 0, 0, 0, 0, 0],
            &[0, 182, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 219, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 228, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 209, 35, 0, 0, 0, 0, 0, 0, 0],
            &[0, 162, 87, 0, 0, 0, 0, 0, 0, 0],
            &[0, 77, 190, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 183, 142, 9, 0, 0, 3, 17, 0],
            &[0, 0, 8, 145, 227, 201, 197, 222, 85, 0],
            &[0, 0, 0, 0, 15, 193, 25, 2, 0, 0],
            &[0, 0, 0, 0, 32, 147, 139, 0, 0, 0],
            &[0, 0, 0, 2, 162, 175, 68, 0, 0, 0],
        ]),
        // letter: 'È' / 0xc8
        'È' => Some(&[
            &[0, 0, 0, 153, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 165, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 76, 0, 0, 0, 0],
            &[0, 34, 108, 108, 108, 108, 108, 108, 22, 0],
            &[0, 81, 196, 104, 104, 104, 104, 104, 21, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 240, 217, 217, 217, 217, 217, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 42, 4, 0, 0, 0, 0, 0],
        ]),
        // letter: 'É' / 0xc9
        'É' => Some(&[
            &[0, 0, 0, 0, 0, 88, 189, 4, 0, 0],
            &[0, 0, 0, 0, 43, 186, 13, 0, 0, 0],
            &[0, 0, 0, 0, 67, 11, 0, 0, 0, 0],
            &[0, 34, 108, 108, 108, 108, 108, 108, 22, 0],
            &[0, 81, 196, 104, 104, 104, 104, 104, 21, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 240, 217, 217, 217, 217, 217, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 40, 8, 0, 0],
        ]),
        // letter: 'Ê' / 0xca
        'Ê' => Some(&[
            &[0, 0, 0, 19, 204, 197, 8, 0, 0, 0],
            &[0, 0, 5, 173, 48, 74, 157, 0, 0, 0],
            &[0, 0, 35, 43, 0, 0, 56, 22, 0, 0],
            &[0, 34, 108, 108, 108, 108, 108, 108, 22, 0],
            &[0, 81, 196, 104, 104, 104, 104, 104, 21, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 240, 217, 217, 217, 217, 217, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 17, 0, 0, 0, 0],
        ]),
        // letter: 'Ë' / 0xcb
        'Ë' => Some(&[
            &[0, 0, 28, 87, 0, 0, 99, 14, 0, 0],
            &[0, 0, 69, 180, 0, 0, 207, 41, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 34, 108, 108, 108, 108, 108, 108, 22, 0],
            &[0, 81, 196, 104, 104, 104, 104, 104, 21, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 240, 217, 217, 217, 217, 217, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ì' / 0xcc
        'Ì' => Some(&[
            &[0, 0, 3, 183, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 182, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 69, 0, 0, 0, 0],
            &[0, 0, 91, 108, 108, 108, 108, 47, 0, 0],
            &[0, 0, 55, 79, 200, 131, 73, 27, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 146, 188, 236, 211, 182, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 41, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Í' / 0xcd
        'Í' => Some(&[
            &[0, 0, 0, 0, 0, 121, 160, 0, 0, 0],
            &[0, 0, 0, 0, 67, 171, 4, 0, 0, 0],
            &[0, 0, 0, 0, 75, 3, 0, 0, 0, 0],
            &[0, 0, 91, 108, 108, 108, 108, 47, 0, 0],
            &[0, 0, 55, 79, 200, 131, 73, 27, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 146, 188, 236, 211, 182, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 42, 2, 0, 0],
        ]),
        // letter: 'Î' / 0xce
        'Î' => Some(&[
            &[0, 0, 0, 37, 212, 179, 1, 0, 0, 0],
            &[0, 0, 15, 182, 29, 101, 130, 0, 0, 0],
            &[0, 0, 48, 30, 0, 0, 68, 10, 0, 0],
            &[0, 0, 91, 108, 108, 108, 108, 47, 0, 0],
            &[0, 0, 55, 79, 200, 131, 73, 27, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 146, 188, 236, 211, 182, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 12, 0, 0, 0, 0],
        ]),
        // letter: 'Ï' / 0xcf
        'Ï' => Some(&[
            &[0, 0, 43, 71, 0, 2, 108, 3, 0, 0],
            &[0, 0, 98, 151, 0, 14, 218, 16, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 108, 108, 108, 108, 47, 0, 0],
            &[0, 0, 55, 79, 200, 131, 73, 27, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 146, 188, 236, 211, 182, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ð' / 0xd0
        'Ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 108, 108, 94, 56, 1, 0, 0, 0],
            &[0, 131, 165, 100, 118, 172, 210, 46, 0, 0],
            &[0, 131, 106, 0, 0, 0, 59, 224, 20, 0],
            &[0, 131, 106, 0, 0, 0, 0, 128, 129, 0],
            &[0, 131, 106, 0, 0, 0, 0, 44, 202, 0],
            &[40, 171, 147, 40, 22, 0, 0, 10, 230, 0],
            &[167, 255, 255, 167, 91, 0, 0, 2, 236, 0],
            &[0, 131, 106, 0, 0, 0, 0, 22, 216, 0],
            &[0, 131, 106, 0, 0, 0, 0, 81, 161, 0],
            &[0, 131, 106, 0, 0, 0, 3, 198, 65, 0],
            &[0, 131, 106, 0, 0, 38, 179, 143, 0, 0],
            &[0, 131, 230, 213, 212, 192, 88, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ñ' / 0xd1
        'Ñ' => Some(&[
            &[0, 0, 16, 114, 67, 0, 11, 77, 0, 0],
            &[0, 0, 158, 91, 147, 170, 172, 75, 0, 0],
            &[0, 0, 48, 0, 0, 38, 44, 0, 0, 0],
            &[0, 72, 65, 0, 0, 0, 0, 67, 31, 0],
            &[0, 171, 224, 9, 0, 0, 0, 159, 73, 0],
            &[0, 171, 154, 112, 0, 0, 0, 159, 73, 0],
            &[0, 171, 63, 208, 8, 0, 0, 159, 73, 0],
            &[0, 171, 56, 119, 107, 0, 0, 159, 73, 0],
            &[0, 171, 57, 13, 208, 6, 0, 159, 73, 0],
            &[0, 171, 57, 0, 125, 102, 0, 159, 73, 0],
            &[0, 171, 57, 0, 15, 207, 4, 159, 73, 0],
            &[0, 171, 57, 0, 0, 131, 97, 157, 73, 0],
            &[0, 171, 57, 0, 0, 18, 206, 154, 73, 0],
            &[0, 171, 57, 0, 0, 0, 136, 228, 73, 0],
            &[0, 171, 57, 0, 0, 0, 21, 245, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ò' / 0xd2
        'Ò' => Some(&[
            &[0, 0, 3, 183, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 182, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 69, 0, 0, 0, 0],
            &[0, 0, 2, 85, 140, 133, 51, 0, 0, 0],
            &[0, 5, 184, 156, 73, 85, 200, 103, 0, 0],
            &[0, 112, 152, 0, 0, 0, 13, 220, 27, 0],
            &[0, 212, 32, 0, 0, 0, 0, 129, 115, 0],
            &[15, 227, 0, 0, 0, 0, 0, 71, 169, 0],
            &[38, 201, 0, 0, 0, 0, 0, 45, 195, 0],
            &[47, 194, 0, 0, 0, 0, 0, 40, 200, 0],
            &[32, 209, 0, 0, 0, 0, 0, 55, 185, 0],
            &[3, 237, 4, 0, 0, 0, 0, 95, 146, 0],
            &[0, 176, 77, 0, 0, 0, 0, 174, 75, 0],
            &[0, 55, 215, 27, 0, 0, 85, 203, 2, 0],
            &[0, 0, 84, 219, 190, 205, 188, 27, 0, 0],
            &[0, 0, 0, 3, 31, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 41, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ó' / 0xd3
        'Ó' => Some(&[
            &[0, 0, 0, 0, 0, 121, 160, 0, 0, 0],
            &[0, 0, 0, 0, 67, 171, 4, 0, 0, 0],
            &[0, 0, 0, 0, 75, 3, 0, 0, 0, 0],
            &[0, 0, 2, 85, 140, 133, 51, 0, 0, 0],
            &[0, 5, 184, 156, 73, 85, 200, 103, 0, 0],
            &[0, 112, 152, 0, 0, 0, 13, 220, 27, 0],
            &[0, 212, 32, 0, 0, 0, 0, 129, 115, 0],
            &[15, 227, 0, 0, 0, 0, 0, 71, 169, 0],
            &[38, 201, 0, 0, 0, 0, 0, 45, 195, 0],
            &[47, 194, 0, 0, 0, 0, 0, 40, 200, 0],
            &[32, 209, 0, 0, 0, 0, 0, 55, 185, 0],
            &[3, 237, 4, 0, 0, 0, 0, 95, 146, 0],
            &[0, 176, 77, 0, 0, 0, 0, 174, 75, 0],
            &[0, 55, 215, 27, 0, 0, 85, 203, 2, 0],
            &[0, 0, 84, 219, 190, 205, 188, 27, 0, 0],
            &[0, 0, 0, 3, 31, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 42, 2, 0, 0],
        ]),
        // letter: 'Ô' / 0xd4
        'Ô' => Some(&[
            &[0, 0, 0, 37, 212, 179, 1, 0, 0, 0],
            &[0, 0, 15, 182, 29, 101, 130, 0, 0, 0],
            &[0, 0, 48, 30, 0, 0, 68, 10, 0, 0],
            &[0, 0, 2, 85, 140, 133, 51, 0, 0, 0],
            &[0, 5, 184, 156, 73, 85, 200, 103, 0, 0],
            &[0, 112, 152, 0, 0, 0, 13, 220, 27, 0],
            &[0, 212, 32, 0, 0, 0, 0, 129, 115, 0],
            &[15, 227, 0, 0, 0, 0, 0, 71, 169, 0],
            &[38, 201, 0, 0, 0, 0, 0, 45, 195, 0],
            &[47, 194, 0, 0, 0, 0, 0, 40, 200, 0],
            &[32, 209, 0, 0, 0, 0, 0, 55, 185, 0],
            &[3, 237, 4, 0, 0, 0, 0, 95, 146, 0],
            &[0, 176, 77, 0, 0, 0, 0, 174, 75, 0],
            &[0, 55, 215, 27, 0, 0, 85, 203, 2, 0],
            &[0, 0, 84, 219, 190, 205, 188, 27, 0, 0],
            &[0, 0, 0, 3, 31, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 12, 0, 0, 0, 0],
        ]),
        // letter: 'Õ' / 0xd5
        'Õ' => Some(&[
            &[0, 0, 23, 116, 58, 0, 21, 66, 0, 0],
            &[0, 0, 170, 85, 158, 161, 180, 57, 0, 0],
            &[0, 0, 48, 0, 0, 44, 38, 0, 0, 0],
            &[0, 0, 2, 85, 140, 133, 51, 0, 0, 0],
            &[0, 5, 184, 156, 73, 85, 200, 103, 0, 0],
            &[0, 112, 152, 0, 0, 0, 13, 220, 27, 0],
            &[0, 212, 32, 0, 0, 0, 0, 129, 115, 0],
            &[15, 227, 0, 0, 0, 0, 0, 71, 169, 0],
            &[38, 201, 0, 0, 0, 0, 0, 45, 195, 0],
            &[47, 194, 0, 0, 0, 0, 0, 40, 200, 0],
            &[32, 209, 0, 0, 0, 0, 0, 55, 185, 0],
            &[3, 237, 4, 0, 0, 0, 0, 95, 146, 0],
            &[0, 176, 77, 0, 0, 0, 0, 174, 75, 0],
            &[0, 55, 215, 27, 0, 0, 85, 203, 2, 0],
            &[0, 0, 84, 219, 190, 205, 188, 27, 0, 0],
            &[0, 0, 0, 3, 31, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ö' / 0xd6
        'Ö' => Some(&[
            &[0, 0, 43, 71, 0, 2, 108, 3, 0, 0],
            &[0, 0, 98, 151, 0, 14, 218, 16, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 85, 140, 133, 51, 0, 0, 0],
            &[0, 5, 184, 156, 73, 85, 200, 103, 0, 0],
            &[0, 112, 152, 0, 0, 0, 13, 220, 27, 0],
            &[0, 212, 32, 0, 0, 0, 0, 129, 115, 0],
            &[15, 227, 0, 0, 0, 0, 0, 71, 169, 0],
            &[38, 201, 0, 0, 0, 0, 0, 45, 195, 0],
            &[47, 194, 0, 0, 0, 0, 0, 40, 200, 0],
            &[32, 209, 0, 0, 0, 0, 0, 55, 185, 0],
            &[3, 237, 4, 0, 0, 0, 0, 95, 146, 0],
            &[0, 176, 77, 0, 0, 0, 0, 174, 75, 0],
            &[0, 55, 215, 27, 0, 0, 85, 203, 2, 0],
            &[0, 0, 84, 219, 190, 205, 188, 27, 0, 0],
            &[0, 0, 0, 3, 31, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '×' / 0xd7
        '×' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 0, 0, 0, 0, 0, 1, 0, 0],
            &[0, 95, 125, 0, 0, 0, 18, 188, 14, 0],
            &[0, 0, 147, 123, 0, 18, 193, 60, 0, 0],
            &[0, 0, 0, 147, 132, 194, 61, 0, 0, 0],
            &[0, 0, 0, 18, 234, 157, 0, 0, 0, 0],
            &[0, 0, 18, 194, 63, 149, 125, 0, 0, 0],
            &[0, 18, 194, 61, 0, 0, 147, 125, 0, 0],
            &[0, 78, 61, 0, 0, 0, 0, 123, 14, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ø' / 0xd8
        'Ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 85, 140, 133, 51, 82, 76, 0],
            &[0, 5, 184, 156, 73, 85, 209, 255, 16, 0],
            &[0, 112, 152, 0, 0, 0, 136, 255, 27, 0],
            &[0, 212, 32, 0, 0, 25, 200, 133, 115, 0],
            &[15, 227, 0, 0, 0, 158, 70, 71, 169, 0],
            &[38, 201, 0, 0, 51, 178, 0, 45, 195, 0],
            &[47, 194, 0, 0, 190, 39, 0, 40, 200, 0],
            &[32, 209, 0, 86, 144, 0, 0, 55, 185, 0],
            &[3, 237, 12, 204, 17, 0, 0, 95, 146, 0],
            &[0, 176, 200, 108, 0, 0, 0, 174, 75, 0],
            &[0, 81, 255, 31, 0, 0, 85, 203, 2, 0],
            &[0, 159, 156, 219, 190, 205, 188, 27, 0, 0],
            &[0, 77, 0, 3, 31, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ù' / 0xd9
        'Ù' => Some(&[
            &[0, 0, 3, 183, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 182, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 69, 0, 0, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 69, 31, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 161, 75, 0, 0, 0, 0, 174, 63, 0],
            &[0, 127, 118, 0, 0, 0, 0, 218, 26, 0],
            &[0, 42, 220, 21, 0, 0, 94, 189, 0, 0],
            &[0, 0, 96, 221, 186, 204, 192, 26, 0, 0],
            &[0, 0, 0, 4, 31, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 41, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ú' / 0xda
        'Ú' => Some(&[
            &[0, 0, 0, 0, 0, 121, 160, 0, 0, 0],
            &[0, 0, 0, 0, 67, 171, 4, 0, 0, 0],
            &[0, 0, 0, 0, 75, 3, 0, 0, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 69, 31, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 161, 75, 0, 0, 0, 0, 174, 63, 0],
            &[0, 127, 118, 0, 0, 0, 0, 218, 26, 0],
            &[0, 42, 220, 21, 0, 0, 94, 189, 0, 0],
            &[0, 0, 96, 221, 186, 204, 192, 26, 0, 0],
            &[0, 0, 0, 4, 31, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 42, 2, 0, 0],
        ]),
        // letter: 'Û' / 0xdb
        'Û' => Some(&[
            &[0, 0, 0, 37, 212, 179, 1, 0, 0, 0],
            &[0, 0, 15, 182, 29, 101, 130, 0, 0, 0],
            &[0, 0, 48, 30, 0, 0, 68, 10, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 69, 31, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 161, 75, 0, 0, 0, 0, 174, 63, 0],
            &[0, 127, 118, 0, 0, 0, 0, 218, 26, 0],
            &[0, 42, 220, 21, 0, 0, 94, 189, 0, 0],
            &[0, 0, 96, 221, 186, 204, 192, 26, 0, 0],
            &[0, 0, 0, 4, 31, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 12, 0, 0, 0, 0],
        ]),
        // letter: 'Ü' / 0xdc
        'Ü' => Some(&[
            &[0, 0, 43, 71, 0, 2, 108, 3, 0, 0],
            &[0, 0, 98, 151, 0, 14, 218, 16, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 69, 31, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 161, 75, 0, 0, 0, 0, 174, 63, 0],
            &[0, 127, 118, 0, 0, 0, 0, 218, 26, 0],
            &[0, 42, 220, 21, 0, 0, 94, 189, 0, 0],
            &[0, 0, 96, 221, 186, 204, 192, 26, 0, 0],
            &[0, 0, 0, 4, 31, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ý' / 0xdd
        'Ý' => Some(&[
            &[0, 0, 0, 0, 0, 121, 160, 0, 0, 0],
            &[0, 0, 0, 0, 67, 171, 4, 0, 0, 0],
            &[0, 0, 0, 0, 75, 3, 0, 0, 0, 0],
            &[21, 88, 0, 0, 0, 0, 0, 20, 88, 0],
            &[1, 208, 46, 0, 0, 0, 0, 137, 114, 0],
            &[0, 77, 172, 0, 0, 0, 20, 216, 8, 0],
            &[0, 0, 197, 46, 0, 0, 134, 103, 0, 0],
            &[0, 0, 65, 172, 0, 18, 207, 4, 0, 0],
            &[0, 0, 0, 185, 46, 132, 91, 0, 0, 0],
            &[0, 0, 0, 54, 184, 197, 2, 0, 0, 0],
            &[0, 0, 0, 0, 185, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 42, 2, 0, 0],
        ]),
        // letter: 'Þ' / 0xde
        'Þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 34, 65, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 180, 63, 63, 52, 13, 0, 0, 0],
            &[0, 81, 212, 145, 146, 167, 223, 145, 2, 0],
            &[0, 81, 155, 0, 0, 0, 7, 193, 97, 0],
            &[0, 81, 155, 0, 0, 0, 0, 93, 164, 0],
            &[0, 81, 155, 0, 0, 0, 0, 82, 161, 0],
            &[0, 81, 155, 0, 0, 0, 0, 168, 96, 0],
            &[0, 81, 185, 75, 78, 102, 183, 166, 2, 0],
            &[0, 81, 209, 137, 134, 115, 53, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ß' / 0xdf
        'ß' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 54, 44, 3, 0, 0, 0],
            &[0, 0, 138, 214, 159, 168, 217, 48, 0, 0],
            &[0, 71, 195, 4, 0, 0, 89, 174, 0, 0],
            &[0, 126, 109, 0, 0, 0, 82, 161, 0, 0],
            &[0, 135, 98, 0, 0, 54, 205, 35, 0, 0],
            &[0, 135, 98, 0, 28, 213, 27, 0, 0, 0],
            &[0, 135, 98, 0, 75, 160, 0, 0, 0, 0],
            &[0, 135, 98, 0, 15, 210, 95, 0, 0, 0],
            &[0, 135, 98, 0, 0, 18, 178, 160, 3, 0],
            &[0, 135, 98, 0, 0, 0, 0, 151, 120, 0],
            &[0, 135, 98, 0, 0, 0, 0, 47, 186, 0],
            &[0, 135, 98, 9, 3, 0, 0, 91, 163, 0],
            &[0, 135, 98, 46, 208, 176, 179, 202, 32, 0],
            &[0, 0, 0, 0, 2, 27, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'à' / 0xe0
        'à' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 65, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 146, 133, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 156, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 58, 0, 0, 0, 0],
            &[0, 0, 42, 125, 160, 149, 83, 0, 0, 0],
            &[0, 0, 144, 81, 39, 53, 175, 125, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 0, 0],
            &[0, 0, 71, 174, 187, 182, 180, 245, 0, 0],
            &[0, 85, 190, 29, 0, 0, 0, 224, 0, 0],
            &[0, 171, 59, 0, 0, 0, 14, 249, 0, 0],
            &[0, 148, 100, 0, 0, 4, 146, 240, 0, 0],
            &[0, 27, 202, 184, 169, 187, 65, 202, 0, 0],
            &[0, 0, 0, 23, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'á' / 0xe1
        'á' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 64, 8, 0, 0],
            &[0, 0, 0, 0, 0, 117, 161, 0, 0, 0],
            &[0, 0, 0, 0, 65, 169, 4, 0, 0, 0],
            &[0, 0, 0, 0, 56, 3, 0, 0, 0, 0],
            &[0, 0, 42, 125, 160, 149, 83, 0, 0, 0],
            &[0, 0, 144, 81, 39, 53, 175, 125, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 0, 0],
            &[0, 0, 71, 174, 187, 182, 180, 245, 0, 0],
            &[0, 85, 190, 29, 0, 0, 0, 224, 0, 0],
            &[0, 171, 59, 0, 0, 0, 14, 249, 0, 0],
            &[0, 148, 100, 0, 0, 4, 146, 240, 0, 0],
            &[0, 27, 202, 184, 169, 187, 65, 202, 0, 0],
            &[0, 0, 0, 23, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'â' / 0xe2
        'â' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 26, 0, 0, 0, 0],
            &[0, 0, 0, 35, 203, 197, 9, 0, 0, 0],
            &[0, 0, 15, 179, 29, 67, 160, 1, 0, 0],
            &[0, 0, 33, 26, 0, 0, 44, 16, 0, 0],
            &[0, 0, 42, 125, 160, 149, 83, 0, 0, 0],
            &[0, 0, 144, 81, 39, 53, 175, 125, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 0, 0],
            &[0, 0, 71, 174, 187, 182, 180, 245, 0, 0],
            &[0, 85, 190, 29, 0, 0, 0, 224, 0, 0],
            &[0, 171, 59, 0, 0, 0, 14, 249, 0, 0],
            &[0, 148, 100, 0, 0, 4, 146, 240, 0, 0],
            &[0, 27, 202, 184, 169, 187, 65, 202, 0, 0],
            &[0, 0, 0, 23, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ã' / 0xe3
        'ã' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 136, 89, 0, 15, 88, 0, 0],
            &[0, 0, 163, 68, 125, 187, 186, 64, 0, 0],
            &[0, 0, 33, 0, 0, 20, 26, 0, 0, 0],
            &[0, 0, 42, 125, 160, 149, 83, 0, 0, 0],
            &[0, 0, 144, 81, 39, 53, 175, 125, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 0, 0],
            &[0, 0, 71, 174, 187, 182, 180, 245, 0, 0],
            &[0, 85, 190, 29, 0, 0, 0, 224, 0, 0],
            &[0, 171, 59, 0, 0, 0, 14, 249, 0, 0],
            &[0, 148, 100, 0, 0, 4, 146, 240, 0, 0],
            &[0, 27, 202, 184, 169, 187, 65, 202, 0, 0],
            &[0, 0, 0, 23, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ä' / 0xe4
        'ä' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 99, 0, 0, 126, 14, 0, 0],
            &[0, 0, 71, 152, 0, 3, 191, 27, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 125, 160, 149, 83, 0, 0, 0],
            &[0, 0, 144, 81, 39, 53, 175, 125, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 0, 0],
            &[0, 0, 71, 174, 187, 182, 180, 245, 0, 0],
            &[0, 85, 190, 29, 0, 0, 0, 224, 0, 0],
            &[0, 171, 59, 0, 0, 0, 14, 249, 0, 0],
            &[0, 148, 100, 0, 0, 4, 146, 240, 0, 0],
            &[0, 27, 202, 184, 169, 187, 65, 202, 0, 0],
            &[0, 0, 0, 23, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'å' / 0xe5
        'å' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 86, 170, 175, 40, 0, 0, 0],
            &[0, 0, 0, 175, 1, 45, 131, 0, 0, 0],
            &[0, 0, 0, 130, 128, 164, 69, 0, 0, 0],
            &[0, 0, 0, 0, 46, 30, 0, 0, 0, 0],
            &[0, 0, 42, 125, 160, 149, 83, 0, 0, 0],
            &[0, 0, 144, 81, 39, 53, 175, 125, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 0, 0],
            &[0, 0, 71, 174, 187, 182, 180, 245, 0, 0],
            &[0, 85, 190, 29, 0, 0, 0, 224, 0, 0],
            &[0, 171, 59, 0, 0, 0, 14, 249, 0, 0],
            &[0, 148, 100, 0, 0, 4, 146, 240, 0, 0],
            &[0, 27, 202, 184, 169, 187, 65, 202, 0, 0],
            &[0, 0, 0, 23, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'æ' / 0xe6
        'æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[1, 80, 154, 142, 23, 30, 149, 140, 17, 0],
            &[2, 121, 45, 84, 177, 182, 62, 114, 169, 0],
            &[0, 0, 0, 0, 194, 116, 0, 0, 217, 6],
            &[0, 0, 0, 0, 174, 61, 0, 0, 191, 33],
            &[0, 94, 191, 186, 232, 202, 192, 192, 205, 35],
            &[69, 193, 18, 0, 179, 48, 0, 0, 0, 0],
            &[136, 89, 0, 0, 209, 85, 0, 0, 0, 0],
            &[118, 111, 0, 47, 170, 181, 2, 0, 13, 0],
            &[24, 211, 176, 200, 25, 102, 201, 172, 180, 0],
            &[0, 4, 29, 2, 0, 0, 12, 28, 1, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ç' / 0xe7
        'ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 41, 130, 163, 151, 93, 1, 0],
            &[0, 0, 99, 212, 94, 49, 61, 111, 0, 0],
            &[0, 30, 224, 18, 0, 0, 0, 0, 0, 0],
            &[0, 111, 136, 0, 0, 0, 0, 0, 0, 0],
            &[0, 142, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 105, 0, 0, 0, 0, 0, 0, 0],
            &[0, 88, 167, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 217, 81, 0, 0, 0, 7, 0, 0],
            &[0, 0, 33, 186, 208, 183, 192, 199, 0, 0],
            &[0, 0, 0, 0, 67, 156, 16, 0, 0, 0],
            &[0, 0, 0, 0, 61, 168, 89, 0, 0, 0],
            &[0, 0, 0, 41, 165, 167, 35, 0, 0, 0],
        ]),
        // letter: 'è' / 0xe8
        'è' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 65, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 150, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 159, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 58, 0, 0, 0, 0],
            &[0, 0, 4, 92, 154, 158, 94, 2, 0, 0],
            &[0, 3, 183, 149, 53, 43, 147, 161, 0, 0],
            &[0, 105, 156, 0, 0, 0, 0, 193, 52, 0],
            &[0, 191, 51, 0, 0, 0, 0, 122, 115, 0],
            &[0, 224, 199, 196, 196, 196, 196, 205, 114, 0],
            &[0, 217, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 83, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 214, 32, 0, 0, 0, 15, 12, 0],
            &[0, 0, 66, 201, 191, 170, 195, 197, 32, 0],
            &[0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'é' / 0xe9
        'é' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 65, 7, 0, 0],
            &[0, 0, 0, 0, 0, 121, 157, 0, 0, 0],
            &[0, 0, 0, 0, 69, 166, 3, 0, 0, 0],
            &[0, 0, 0, 0, 57, 2, 0, 0, 0, 0],
            &[0, 0, 4, 92, 154, 158, 94, 2, 0, 0],
            &[0, 3, 183, 149, 53, 43, 147, 161, 0, 0],
            &[0, 105, 156, 0, 0, 0, 0, 193, 52, 0],
            &[0, 191, 51, 0, 0, 0, 0, 122, 115, 0],
            &[0, 224, 199, 196, 196, 196, 196, 205, 114, 0],
            &[0, 217, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 83, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 214, 32, 0, 0, 0, 15, 12, 0],
            &[0, 0, 66, 201, 191, 170, 195, 197, 32, 0],
            &[0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ê' / 0xea
        'ê' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 25, 0, 0, 0, 0],
            &[0, 0, 0, 37, 203, 196, 8, 0, 0, 0],
            &[0, 0, 17, 180, 27, 70, 157, 1, 0, 0],
            &[0, 0, 35, 25, 0, 0, 45, 15, 0, 0],
            &[0, 0, 4, 92, 154, 158, 94, 2, 0, 0],
            &[0, 3, 183, 149, 53, 43, 147, 161, 0, 0],
            &[0, 105, 156, 0, 0, 0, 0, 193, 52, 0],
            &[0, 191, 51, 0, 0, 0, 0, 122, 115, 0],
            &[0, 224, 199, 196, 196, 196, 196, 205, 114, 0],
            &[0, 217, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 83, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 214, 32, 0, 0, 0, 15, 12, 0],
            &[0, 0, 66, 201, 191, 170, 195, 197, 32, 0],
            &[0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ë' / 0xeb
        'ë' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 97, 0, 0, 128, 12, 0, 0],
            &[0, 0, 74, 148, 0, 4, 193, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 92, 154, 158, 94, 2, 0, 0],
            &[0, 3, 183, 149, 53, 43, 147, 161, 0, 0],
            &[0, 105, 156, 0, 0, 0, 0, 193, 52, 0],
            &[0, 191, 51, 0, 0, 0, 0, 122, 115, 0],
            &[0, 224, 199, 196, 196, 196, 196, 205, 114, 0],
            &[0, 217, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 83, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 214, 32, 0, 0, 0, 15, 12, 0],
            &[0, 0, 66, 201, 191, 170, 195, 197, 32, 0],
            &[0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ì' / 0xec
        'ì' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 64, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 166, 112, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 169, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 56, 0, 0, 0, 0],
            &[0, 37, 131, 131, 131, 72, 0, 0, 0, 0],
            &[0, 11, 46, 59, 139, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 193, 180, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'í' / 0xed
        'í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 65, 3, 0, 0],
            &[0, 0, 0, 0, 0, 137, 141, 0, 0, 0],
            &[0, 0, 0, 0, 83, 155, 1, 0, 0, 0],
            &[0, 0, 0, 0, 59, 0, 0, 0, 0, 0],
            &[0, 37, 131, 131, 131, 72, 0, 0, 0, 0],
            &[0, 11, 46, 59, 139, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 193, 180, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'î' / 0xee
        'î' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 21, 0, 0, 0, 0],
            &[0, 0, 0, 48, 204, 188, 4, 0, 0, 0],
            &[0, 0, 23, 180, 20, 83, 146, 0, 0, 0],
            &[0, 0, 40, 20, 0, 0, 50, 10, 0, 0],
            &[0, 37, 131, 131, 131, 72, 0, 0, 0, 0],
            &[0, 11, 46, 59, 139, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 193, 180, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ï' / 0xef
        'ï' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 87, 0, 4, 131, 5, 0, 0],
            &[0, 0, 87, 135, 0, 12, 196, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 131, 131, 131, 72, 0, 0, 0, 0],
            &[0, 11, 46, 59, 139, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 193, 180, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ð' / 0xf0
        'ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 0, 0, 0, 2, 0, 0],
            &[0, 0, 0, 137, 165, 42, 161, 101, 0, 0],
            &[0, 0, 0, 5, 192, 255, 81, 0, 0, 0],
            &[0, 0, 5, 186, 96, 53, 199, 8, 0, 0],
            &[0, 0, 0, 3, 0, 0, 106, 127, 0, 0],
            &[0, 0, 3, 74, 119, 109, 36, 208, 8, 0],
            &[0, 15, 196, 145, 78, 85, 173, 216, 77, 0],
            &[0, 147, 111, 0, 0, 0, 0, 185, 129, 0],
            &[0, 227, 12, 0, 0, 0, 0, 97, 150, 0],
            &[0, 236, 0, 0, 0, 0, 0, 93, 141, 0],
            &[0, 210, 31, 0, 0, 0, 0, 144, 93, 0],
            &[0, 100, 172, 5, 0, 0, 48, 217, 12, 0],
            &[0, 0, 115, 207, 169, 185, 201, 46, 0, 0],
            &[0, 0, 0, 4, 31, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ñ' / 0xf1
        'ñ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 133, 99, 2, 6, 98, 0, 0],
            &[0, 0, 149, 78, 113, 191, 180, 81, 0, 0],
            &[0, 0, 33, 0, 0, 16, 30, 0, 0, 0],
            &[0, 67, 34, 45, 137, 160, 112, 8, 0, 0],
            &[0, 131, 144, 159, 55, 44, 146, 180, 0, 0],
            &[0, 131, 207, 1, 0, 0, 1, 224, 22, 0],
            &[0, 131, 133, 0, 0, 0, 0, 188, 51, 0],
            &[0, 131, 108, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ò' / 0xf2
        'ò' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 64, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 166, 112, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 169, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 56, 0, 0, 0, 0],
            &[0, 0, 13, 110, 161, 151, 75, 0, 0, 0],
            &[0, 17, 209, 121, 46, 53, 172, 137, 0, 0],
            &[0, 142, 121, 0, 0, 0, 2, 206, 45, 0],
            &[0, 220, 24, 0, 0, 0, 0, 121, 121, 0],
            &[0, 239, 0, 0, 0, 0, 0, 91, 148, 0],
            &[0, 236, 3, 0, 0, 0, 0, 102, 140, 0],
            &[0, 188, 57, 0, 0, 0, 0, 160, 88, 0],
            &[0, 75, 195, 10, 0, 0, 58, 216, 8, 0],
            &[0, 0, 99, 211, 170, 186, 197, 39, 0, 0],
            &[0, 0, 0, 3, 32, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ó' / 0xf3
        'ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 65, 3, 0, 0],
            &[0, 0, 0, 0, 0, 137, 141, 0, 0, 0],
            &[0, 0, 0, 0, 83, 155, 1, 0, 0, 0],
            &[0, 0, 0, 0, 59, 0, 0, 0, 0, 0],
            &[0, 0, 13, 110, 161, 151, 75, 0, 0, 0],
            &[0, 17, 209, 121, 46, 53, 172, 137, 0, 0],
            &[0, 142, 121, 0, 0, 0, 2, 206, 45, 0],
            &[0, 220, 24, 0, 0, 0, 0, 121, 121, 0],
            &[0, 239, 0, 0, 0, 0, 0, 91, 148, 0],
            &[0, 236, 3, 0, 0, 0, 0, 102, 140, 0],
            &[0, 188, 57, 0, 0, 0, 0, 160, 88, 0],
            &[0, 75, 195, 10, 0, 0, 58, 216, 8, 0],
            &[0, 0, 99, 211, 170, 186, 197, 39, 0, 0],
            &[0, 0, 0, 3, 32, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ô' / 0xf4
        'ô' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 21, 0, 0, 0, 0],
            &[0, 0, 0, 48, 204, 188, 4, 0, 0, 0],
            &[0, 0, 23, 180, 20, 83, 146, 0, 0, 0],
            &[0, 0, 40, 20, 0, 0, 50, 10, 0, 0],
            &[0, 0, 13, 110, 161, 151, 75, 0, 0, 0],
            &[0, 17, 209, 121, 46, 53, 172, 137, 0, 0],
            &[0, 142, 121, 0, 0, 0, 2, 206, 45, 0],
            &[0, 220, 24, 0, 0, 0, 0, 121, 121, 0],
            &[0, 239, 0, 0, 0, 0, 0, 91, 148, 0],
            &[0, 236, 3, 0, 0, 0, 0, 102, 140, 0],
            &[0, 188, 57, 0, 0, 0, 0, 160, 88, 0],
            &[0, 75, 195, 10, 0, 0, 58, 216, 8, 0],
            &[0, 0, 99, 211, 170, 186, 197, 39, 0, 0],
            &[0, 0, 0, 3, 32, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'õ' / 0xf5
        'õ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 139, 79, 0, 28, 76, 0, 0],
            &[0, 0, 174, 63, 137, 181, 191, 47, 0, 0],
            &[0, 0, 33, 0, 0, 24, 21, 0, 0, 0],
            &[0, 0, 13, 110, 161, 151, 75, 0, 0, 0],
            &[0, 17, 209, 121, 46, 53, 172, 137, 0, 0],
            &[0, 142, 121, 0, 0, 0, 2, 206, 45, 0],
            &[0, 220, 24, 0, 0, 0, 0, 121, 121, 0],
            &[0, 239, 0, 0, 0, 0, 0, 91, 148, 0],
            &[0, 236, 3, 0, 0, 0, 0, 102, 140, 0],
            &[0, 188, 57, 0, 0, 0, 0, 160, 88, 0],
            &[0, 75, 195, 10, 0, 0, 58, 216, 8, 0],
            &[0, 0, 99, 211, 170, 186, 197, 39, 0, 0],
            &[0, 0, 0, 3, 32, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ö' / 0xf6
        'ö' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 87, 0, 4, 131, 5, 0, 0],
            &[0, 0, 87, 135, 0, 12, 196, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 110, 161, 151, 75, 0, 0, 0],
            &[0, 17, 209, 121, 46, 53, 172, 137, 0, 0],
            &[0, 142, 121, 0, 0, 0, 2, 206, 45, 0],
            &[0, 220, 24, 0, 0, 0, 0, 121, 121, 0],
            &[0, 239, 0, 0, 0, 0, 0, 91, 148, 0],
            &[0, 236, 3, 0, 0, 0, 0, 102, 140, 0],
            &[0, 188, 57, 0, 0, 0, 0, 160, 88, 0],
            &[0, 75, 195, 10, 0, 0, 58, 216, 8, 0],
            &[0, 0, 99, 211, 170, 186, 197, 39, 0, 0],
            &[0, 0, 0, 3, 32, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '÷' / 0xf7
        '÷' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 199, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 12, 0, 0, 0, 0],
            &[0, 6, 8, 8, 8, 8, 8, 8, 3, 0],
            &[0, 150, 188, 188, 188, 188, 188, 188, 75, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 157, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ø' / 0xf8
        'ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 110, 161, 151, 75, 96, 88, 0],
            &[0, 17, 209, 121, 46, 53, 217, 255, 6, 0],
            &[0, 142, 121, 0, 0, 13, 201, 238, 45, 0],
            &[0, 220, 24, 0, 0, 167, 77, 121, 121, 0],
            &[0, 239, 0, 0, 107, 137, 0, 91, 148, 0],
            &[0, 236, 3, 52, 188, 4, 0, 102, 140, 0],
            &[0, 188, 74, 201, 27, 0, 0, 160, 88, 0],
            &[0, 76, 255, 80, 0, 0, 58, 216, 8, 0],
            &[0, 117, 228, 211, 170, 186, 197, 39, 0, 0],
            &[0, 60, 2, 3, 32, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ù' / 0xf9
        'ù' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 64, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 166, 112, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 169, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 56, 0, 0, 0, 0],
            &[0, 77, 44, 0, 0, 0, 0, 104, 16, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 86, 0, 0, 0, 0, 212, 32, 0],
            &[0, 134, 105, 0, 0, 0, 8, 245, 32, 0],
            &[0, 79, 190, 4, 0, 0, 126, 229, 32, 0],
            &[0, 0, 150, 214, 168, 188, 92, 170, 32, 0],
            &[0, 0, 0, 9, 29, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ú' / 0xfa
        'ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 65, 3, 0, 0],
            &[0, 0, 0, 0, 0, 137, 141, 0, 0, 0],
            &[0, 0, 0, 0, 83, 155, 1, 0, 0, 0],
            &[0, 0, 0, 0, 59, 0, 0, 0, 0, 0],
            &[0, 77, 44, 0, 0, 0, 0, 104, 16, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 86, 0, 0, 0, 0, 212, 32, 0],
            &[0, 134, 105, 0, 0, 0, 8, 245, 32, 0],
            &[0, 79, 190, 4, 0, 0, 126, 229, 32, 0],
            &[0, 0, 150, 214, 168, 188, 92, 170, 32, 0],
            &[0, 0, 0, 9, 29, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'û' / 0xfb
        'û' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 21, 0, 0, 0, 0],
            &[0, 0, 0, 48, 204, 188, 4, 0, 0, 0],
            &[0, 0, 23, 180, 20, 83, 146, 0, 0, 0],
            &[0, 0, 40, 20, 0, 0, 50, 10, 0, 0],
            &[0, 77, 44, 0, 0, 0, 0, 104, 16, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 86, 0, 0, 0, 0, 212, 32, 0],
            &[0, 134, 105, 0, 0, 0, 8, 245, 32, 0],
            &[0, 79, 190, 4, 0, 0, 126, 229, 32, 0],
            &[0, 0, 150, 214, 168, 188, 92, 170, 32, 0],
            &[0, 0, 0, 9, 29, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ü' / 0xfc
        'ü' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 87, 0, 4, 131, 5, 0, 0],
            &[0, 0, 87, 135, 0, 12, 196, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 77, 44, 0, 0, 0, 0, 104, 16, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 86, 0, 0, 0, 0, 212, 32, 0],
            &[0, 134, 105, 0, 0, 0, 8, 245, 32, 0],
            &[0, 79, 190, 4, 0, 0, 126, 229, 32, 0],
            &[0, 0, 150, 214, 168, 188, 92, 170, 32, 0],
            &[0, 0, 0, 9, 29, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ý' / 0xfd
        'ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 65, 3, 0, 0],
            &[0, 0, 0, 0, 0, 137, 141, 0, 0, 0],
            &[0, 0, 0, 0, 83, 155, 1, 0, 0, 0],
            &[0, 0, 0, 0, 59, 0, 0, 0, 0, 0],
            &[0, 106, 15, 0, 0, 0, 0, 64, 60, 0],
            &[0, 130, 104, 0, 0, 0, 0, 195, 45, 0],
            &[0, 31, 201, 0, 0, 0, 32, 203, 0, 0],
            &[0, 0, 182, 47, 0, 0, 123, 108, 0, 0],
            &[0, 0, 81, 146, 0, 0, 208, 18, 0, 0],
            &[0, 0, 4, 213, 8, 51, 171, 0, 0, 0],
            &[0, 0, 0, 133, 88, 141, 75, 0, 0, 0],
            &[0, 0, 0, 33, 178, 202, 3, 0, 0, 0],
            &[0, 0, 0, 0, 185, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 186, 41, 0, 0, 0, 0],
            &[0, 0, 0, 33, 196, 0, 0, 0, 0, 0],
            &[0, 156, 209, 124, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'þ' / 0xfe
        'þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 103, 81, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 102, 42, 136, 163, 110, 7, 0, 0],
            &[0, 131, 152, 165, 58, 48, 143, 185, 1, 0],
            &[0, 131, 214, 2, 0, 0, 0, 192, 68, 0],
            &[0, 131, 134, 0, 0, 0, 0, 119, 130, 0],
            &[0, 131, 108, 0, 0, 0, 0, 96, 151, 0],
            &[0, 131, 122, 0, 0, 0, 0, 111, 139, 0],
            &[0, 131, 168, 0, 0, 0, 0, 157, 94, 0],
            &[0, 131, 220, 53, 0, 0, 46, 224, 15, 0],
            &[0, 131, 110, 160, 184, 184, 211, 58, 0, 0],
            &[0, 131, 102, 0, 11, 28, 2, 0, 0, 0],
            &[0, 131, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 110, 85, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ÿ' / 0xff
        'ÿ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 87, 0, 4, 131, 5, 0, 0],
            &[0, 0, 87, 135, 0, 12, 196, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 106, 15, 0, 0, 0, 0, 64, 60, 0],
            &[0, 130, 104, 0, 0, 0, 0, 195, 45, 0],
            &[0, 31, 201, 0, 0, 0, 32, 203, 0, 0],
            &[0, 0, 182, 47, 0, 0, 123, 108, 0, 0],
            &[0, 0, 81, 146, 0, 0, 208, 18, 0, 0],
            &[0, 0, 4, 213, 8, 51, 171, 0, 0, 0],
            &[0, 0, 0, 133, 88, 141, 75, 0, 0, 0],
            &[0, 0, 0, 33, 178, 202, 3, 0, 0, 0],
            &[0, 0, 0, 0, 185, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 186, 41, 0, 0, 0, 0],
            &[0, 0, 0, 33, 196, 0, 0, 0, 0, 0],
            &[0, 156, 209, 124, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ā' / 0x100
        'Ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 72, 178, 178, 178, 178, 1, 0, 0],
            &[0, 0, 7, 18, 18, 18, 18, 0, 0, 0],
            &[0, 0, 0, 0, 92, 43, 0, 0, 0, 0],
            &[0, 0, 0, 13, 233, 158, 0, 0, 0, 0],
            &[0, 0, 0, 94, 136, 215, 7, 0, 0, 0],
            &[0, 0, 0, 183, 48, 147, 80, 0, 0, 0],
            &[0, 0, 21, 212, 0, 59, 170, 0, 0, 0],
            &[0, 0, 105, 129, 0, 1, 217, 12, 0, 0],
            &[0, 0, 195, 41, 0, 0, 141, 92, 0, 0],
            &[0, 29, 255, 196, 196, 196, 249, 196, 0, 0],
            &[0, 117, 135, 20, 20, 20, 21, 221, 20, 0],
            &[0, 206, 35, 0, 0, 0, 0, 135, 104, 0],
            &[40, 202, 0, 0, 0, 0, 0, 48, 193, 0],
            &[129, 115, 0, 0, 0, 0, 0, 0, 215, 28],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ā' / 0x101
        'ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 64, 196, 196, 196, 196, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 125, 160, 149, 83, 0, 0, 0],
            &[0, 0, 144, 81, 39, 53, 175, 125, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 0, 0],
            &[0, 0, 71, 174, 187, 182, 180, 245, 0, 0],
            &[0, 85, 190, 29, 0, 0, 0, 224, 0, 0],
            &[0, 171, 59, 0, 0, 0, 14, 249, 0, 0],
            &[0, 148, 100, 0, 0, 4, 146, 240, 0, 0],
            &[0, 27, 202, 184, 169, 187, 65, 202, 0, 0],
            &[0, 0, 0, 23, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ă' / 0x102
        'Ă' => Some(&[
            &[0, 0, 57, 64, 0, 0, 116, 0, 0, 0],
            &[0, 0, 18, 194, 113, 152, 137, 0, 0, 0],
            &[0, 0, 0, 11, 67, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 43, 0, 0, 0, 0],
            &[0, 0, 0, 13, 233, 158, 0, 0, 0, 0],
            &[0, 0, 0, 94, 136, 215, 7, 0, 0, 0],
            &[0, 0, 0, 183, 48, 147, 80, 0, 0, 0],
            &[0, 0, 21, 212, 0, 59, 170, 0, 0, 0],
            &[0, 0, 105, 129, 0, 1, 217, 12, 0, 0],
            &[0, 0, 195, 41, 0, 0, 141, 92, 0, 0],
            &[0, 29, 255, 196, 196, 196, 249, 196, 0, 0],
            &[0, 117, 135, 20, 20, 20, 21, 221, 20, 0],
            &[0, 206, 35, 0, 0, 0, 0, 135, 104, 0],
            &[40, 202, 0, 0, 0, 0, 0, 48, 193, 0],
            &[129, 115, 0, 0, 0, 0, 0, 0, 215, 28],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ă' / 0x103
        'ă' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 91, 0, 0, 128, 5, 0, 0],
            &[0, 0, 6, 181, 140, 165, 137, 0, 0, 0],
            &[0, 0, 0, 2, 42, 30, 0, 0, 0, 0],
            &[0, 0, 42, 125, 160, 149, 83, 0, 0, 0],
            &[0, 0, 144, 81, 39, 53, 175, 125, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 0, 0],
            &[0, 0, 71, 174, 187, 182, 180, 245, 0, 0],
            &[0, 85, 190, 29, 0, 0, 0, 224, 0, 0],
            &[0, 171, 59, 0, 0, 0, 14, 249, 0, 0],
            &[0, 148, 100, 0, 0, 4, 146, 240, 0, 0],
            &[0, 27, 202, 184, 169, 187, 65, 202, 0, 0],
            &[0, 0, 0, 23, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ą' / 0x104
        'Ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 43, 0, 0, 0, 0],
            &[0, 0, 0, 13, 233, 158, 0, 0, 0, 0],
            &[0, 0, 0, 94, 136, 215, 7, 0, 0, 0],
            &[0, 0, 0, 183, 48, 147, 80, 0, 0, 0],
            &[0, 0, 21, 212, 0, 59, 170, 0, 0, 0],
            &[0, 0, 105, 129, 0, 1, 217, 12, 0, 0],
            &[0, 0, 195, 41, 0, 0, 141, 92, 0, 0],
            &[0, 29, 255, 196, 196, 196, 249, 196, 0, 0],
            &[0, 117, 135, 20, 20, 20, 21, 221, 20, 0],
            &[0, 206, 35, 0, 0, 0, 0, 135, 104, 0],
            &[40, 202, 0, 0, 0, 0, 0, 48, 193, 0],
            &[129, 115, 0, 0, 0, 0, 0, 0, 231, 33],
            &[0, 0, 0, 0, 0, 0, 0, 27, 165, 11],
            &[0, 0, 0, 0, 0, 0, 0, 155, 34, 0],
            &[0, 0, 0, 0, 0, 0, 0, 24, 114, 64],
        ]),
        // letter: 'ą' / 0x105
        'ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 125, 160, 149, 83, 0, 0, 0],
            &[0, 0, 144, 81, 39, 53, 175, 125, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 0, 0],
            &[0, 0, 71, 174, 187, 182, 180, 245, 0, 0],
            &[0, 85, 190, 29, 0, 0, 0, 224, 0, 0],
            &[0, 171, 59, 0, 0, 0, 14, 249, 0, 0],
            &[0, 148, 100, 0, 0, 4, 146, 240, 0, 0],
            &[0, 27, 202, 184, 169, 187, 65, 222, 0, 0],
            &[0, 0, 0, 23, 21, 0, 84, 119, 0, 0],
            &[0, 0, 0, 0, 0, 2, 187, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 57, 117, 27, 0],
        ]),
        // letter: 'Ć' / 0x106
        'Ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 131, 150, 0, 0],
            &[0, 0, 0, 0, 0, 75, 165, 3, 0, 0],
            &[0, 0, 0, 0, 0, 76, 2, 0, 0, 0],
            &[0, 0, 0, 9, 89, 137, 136, 100, 21, 0],
            &[0, 0, 44, 216, 142, 76, 77, 125, 123, 0],
            &[0, 12, 220, 59, 0, 0, 0, 0, 0, 0],
            &[0, 109, 150, 0, 0, 0, 0, 0, 0, 0],
            &[0, 182, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 219, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 228, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 209, 35, 0, 0, 0, 0, 0, 0, 0],
            &[0, 162, 87, 0, 0, 0, 0, 0, 0, 0],
            &[0, 77, 190, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 183, 142, 9, 0, 0, 3, 17, 0],
            &[0, 0, 8, 145, 227, 193, 197, 222, 85, 0],
            &[0, 0, 0, 0, 3, 29, 25, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 42, 1, 0],
        ]),
        // letter: 'ć' / 0x107
        'ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 41, 33, 0, 0],
            &[0, 0, 0, 0, 0, 33, 216, 29, 0, 0],
            &[0, 0, 0, 0, 11, 183, 44, 0, 0, 0],
            &[0, 0, 0, 0, 30, 29, 0, 0, 0, 0],
            &[0, 0, 0, 41, 130, 163, 151, 93, 1, 0],
            &[0, 0, 99, 212, 94, 49, 61, 111, 0, 0],
            &[0, 30, 224, 18, 0, 0, 0, 0, 0, 0],
            &[0, 111, 136, 0, 0, 0, 0, 0, 0, 0],
            &[0, 142, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 105, 0, 0, 0, 0, 0, 0, 0],
            &[0, 88, 167, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 217, 81, 0, 0, 0, 7, 0, 0],
            &[0, 0, 33, 186, 208, 174, 192, 199, 0, 0],
            &[0, 0, 0, 0, 13, 34, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĉ' / 0x108
        'Ĉ' => Some(&[
            &[0, 0, 0, 0, 43, 214, 171, 0, 0, 0],
            &[0, 0, 0, 19, 183, 24, 111, 121, 0, 0],
            &[0, 0, 0, 52, 26, 0, 0, 72, 7, 0],
            &[0, 0, 0, 9, 89, 137, 136, 100, 21, 0],
            &[0, 0, 44, 216, 142, 76, 77, 125, 123, 0],
            &[0, 12, 220, 59, 0, 0, 0, 0, 0, 0],
            &[0, 109, 150, 0, 0, 0, 0, 0, 0, 0],
            &[0, 182, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 219, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 228, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 209, 35, 0, 0, 0, 0, 0, 0, 0],
            &[0, 162, 87, 0, 0, 0, 0, 0, 0, 0],
            &[0, 77, 190, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 183, 142, 9, 0, 0, 3, 17, 0],
            &[0, 0, 8, 145, 227, 193, 197, 222, 85, 0],
            &[0, 0, 0, 0, 3, 29, 25, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 10, 0, 0, 0],
        ]),
        // letter: 'ĉ' / 0x109
        'ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 49, 0, 0, 0, 0],
            &[0, 0, 0, 1, 173, 206, 65, 0, 0, 0],
            &[0, 0, 0, 127, 97, 13, 181, 34, 0, 0],
            &[0, 0, 4, 55, 0, 0, 14, 46, 0, 0],
            &[0, 0, 0, 41, 130, 163, 151, 93, 1, 0],
            &[0, 0, 99, 212, 94, 49, 61, 111, 0, 0],
            &[0, 30, 224, 18, 0, 0, 0, 0, 0, 0],
            &[0, 111, 136, 0, 0, 0, 0, 0, 0, 0],
            &[0, 142, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 105, 0, 0, 0, 0, 0, 0, 0],
            &[0, 88, 167, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 217, 81, 0, 0, 0, 7, 0, 0],
            &[0, 0, 33, 186, 208, 174, 192, 199, 0, 0],
            &[0, 0, 0, 0, 13, 34, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ċ' / 0x10a
        'Ċ' => Some(&[
            &[0, 0, 0, 0, 0, 150, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 70, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 89, 137, 136, 100, 21, 0],
            &[0, 0, 44, 216, 142, 76, 77, 125, 123, 0],
            &[0, 12, 220, 59, 0, 0, 0, 0, 0, 0],
            &[0, 109, 150, 0, 0, 0, 0, 0, 0, 0],
            &[0, 182, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 219, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 228, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 209, 35, 0, 0, 0, 0, 0, 0, 0],
            &[0, 162, 87, 0, 0, 0, 0, 0, 0, 0],
            &[0, 77, 190, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 183, 142, 9, 0, 0, 3, 17, 0],
            &[0, 0, 8, 145, 227, 193, 197, 222, 85, 0],
            &[0, 0, 0, 0, 3, 29, 25, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ċ' / 0x10b
        'ċ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 201, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 124, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 41, 130, 163, 151, 93, 1, 0],
            &[0, 0, 99, 212, 94, 49, 61, 111, 0, 0],
            &[0, 30, 224, 18, 0, 0, 0, 0, 0, 0],
            &[0, 111, 136, 0, 0, 0, 0, 0, 0, 0],
            &[0, 142, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 105, 0, 0, 0, 0, 0, 0, 0],
            &[0, 88, 167, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 217, 81, 0, 0, 0, 7, 0, 0],
            &[0, 0, 33, 186, 208, 174, 192, 199, 0, 0],
            &[0, 0, 0, 0, 13, 34, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Č' / 0x10c
        'Č' => Some(&[
            &[0, 0, 0, 51, 168, 6, 68, 160, 2, 0],
            &[0, 0, 0, 0, 84, 185, 192, 8, 0, 0],
            &[0, 0, 0, 0, 0, 83, 32, 0, 0, 0],
            &[0, 0, 0, 9, 89, 137, 136, 100, 21, 0],
            &[0, 0, 44, 216, 142, 76, 77, 125, 123, 0],
            &[0, 12, 220, 59, 0, 0, 0, 0, 0, 0],
            &[0, 109, 150, 0, 0, 0, 0, 0, 0, 0],
            &[0, 182, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 219, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 228, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 209, 35, 0, 0, 0, 0, 0, 0, 0],
            &[0, 162, 87, 0, 0, 0, 0, 0, 0, 0],
            &[0, 77, 190, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 183, 142, 9, 0, 0, 3, 17, 0],
            &[0, 0, 8, 145, 227, 193, 197, 222, 85, 0],
            &[0, 0, 0, 0, 3, 29, 25, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 4, 0, 0, 24, 4, 0],
        ]),
        // letter: 'č' / 0x10d
        'č' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 41, 0, 0, 10, 38, 0, 0],
            &[0, 0, 0, 142, 86, 11, 178, 43, 0, 0],
            &[0, 0, 0, 3, 185, 201, 69, 0, 0, 0],
            &[0, 0, 0, 0, 23, 62, 0, 0, 0, 0],
            &[0, 0, 0, 41, 130, 163, 151, 93, 1, 0],
            &[0, 0, 99, 212, 94, 49, 61, 111, 0, 0],
            &[0, 30, 224, 18, 0, 0, 0, 0, 0, 0],
            &[0, 111, 136, 0, 0, 0, 0, 0, 0, 0],
            &[0, 142, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 105, 0, 0, 0, 0, 0, 0, 0],
            &[0, 88, 167, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 217, 81, 0, 0, 0, 7, 0, 0],
            &[0, 0, 33, 186, 208, 174, 192, 199, 0, 0],
            &[0, 0, 0, 0, 13, 34, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ď' / 0x10e
        'Ď' => Some(&[
            &[0, 0, 74, 150, 1, 92, 138, 0, 0, 0],
            &[0, 0, 0, 115, 178, 175, 2, 0, 0, 0],
            &[0, 0, 0, 1, 94, 20, 0, 0, 0, 0],
            &[0, 72, 108, 108, 90, 45, 0, 0, 0, 0],
            &[0, 171, 140, 101, 122, 188, 192, 26, 0, 0],
            &[0, 171, 65, 0, 0, 0, 91, 208, 5, 0],
            &[0, 171, 65, 0, 0, 0, 0, 169, 89, 0],
            &[0, 171, 65, 0, 0, 0, 0, 85, 161, 0],
            &[0, 171, 65, 0, 0, 0, 0, 51, 189, 0],
            &[0, 171, 65, 0, 0, 0, 0, 41, 196, 0],
            &[0, 171, 65, 0, 0, 0, 0, 63, 175, 0],
            &[0, 171, 65, 0, 0, 0, 0, 122, 120, 0],
            &[0, 171, 65, 0, 0, 0, 15, 219, 32, 0],
            &[0, 171, 65, 0, 0, 54, 201, 104, 0, 0],
            &[0, 171, 223, 213, 213, 179, 66, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 0, 0, 0, 28, 0, 0, 0],
        ]),
        // letter: 'ď' / 0x10f
        'ď' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 19, 0, 38],
            &[0, 0, 0, 0, 0, 0, 112, 124, 15, 210],
            &[0, 0, 0, 0, 0, 0, 112, 124, 79, 108],
            &[0, 0, 0, 0, 0, 0, 112, 124, 54, 7],
            &[0, 0, 59, 151, 151, 65, 111, 124, 0, 0],
            &[0, 73, 213, 66, 41, 147, 171, 124, 0, 0],
            &[0, 203, 53, 0, 0, 0, 209, 124, 0, 0],
            &[12, 232, 0, 0, 0, 0, 139, 124, 0, 0],
            &[32, 208, 0, 0, 0, 0, 116, 124, 0, 0],
            &[29, 218, 0, 0, 0, 0, 127, 124, 0, 0],
            &[3, 235, 9, 0, 0, 0, 169, 124, 0, 0],
            &[0, 160, 118, 0, 0, 44, 220, 124, 0, 0],
            &[0, 19, 194, 195, 180, 171, 93, 124, 0, 0],
            &[0, 0, 0, 26, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Đ' / 0x110
        'Đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 108, 108, 94, 56, 1, 0, 0, 0],
            &[0, 131, 165, 100, 118, 172, 210, 46, 0, 0],
            &[0, 131, 106, 0, 0, 0, 59, 224, 20, 0],
            &[0, 131, 106, 0, 0, 0, 0, 128, 129, 0],
            &[0, 131, 106, 0, 0, 0, 0, 44, 202, 0],
            &[40, 171, 147, 40, 22, 0, 0, 10, 230, 0],
            &[167, 255, 255, 167, 91, 0, 0, 2, 236, 0],
            &[0, 131, 106, 0, 0, 0, 0, 22, 216, 0],
            &[0, 131, 106, 0, 0, 0, 0, 81, 161, 0],
            &[0, 131, 106, 0, 0, 0, 3, 198, 65, 0],
            &[0, 131, 106, 0, 0, 38, 179, 143, 0, 0],
            &[0, 131, 230, 213, 212, 192, 88, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'đ' / 0x111
        'đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 32, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 32, 0, 0],
            &[0, 0, 0, 140, 155, 155, 255, 188, 93, 0],
            &[0, 0, 0, 29, 32, 32, 237, 65, 19, 0],
            &[0, 20, 115, 150, 115, 14, 203, 32, 0, 0],
            &[27, 221, 121, 54, 82, 179, 202, 32, 0, 0],
            &[151, 121, 0, 0, 0, 56, 255, 32, 0, 0],
            &[222, 28, 0, 0, 0, 0, 234, 32, 0, 0],
            &[243, 2, 0, 0, 0, 0, 209, 32, 0, 0],
            &[241, 6, 0, 0, 0, 0, 220, 32, 0, 0],
            &[203, 48, 0, 0, 0, 14, 248, 32, 0, 0],
            &[108, 177, 6, 0, 0, 134, 229, 32, 0, 0],
            &[2, 143, 216, 172, 195, 101, 172, 32, 0, 0],
            &[0, 0, 12, 32, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ē' / 0x112
        'Ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 178, 178, 178, 178, 24, 0, 0],
            &[0, 0, 5, 18, 18, 18, 18, 2, 0, 0],
            &[0, 34, 108, 108, 108, 108, 108, 108, 22, 0],
            &[0, 81, 196, 104, 104, 104, 104, 104, 21, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 240, 217, 217, 217, 217, 217, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ē' / 0x113
        'ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 67, 196, 196, 196, 196, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 92, 154, 158, 94, 2, 0, 0],
            &[0, 3, 183, 149, 53, 43, 147, 161, 0, 0],
            &[0, 105, 156, 0, 0, 0, 0, 193, 52, 0],
            &[0, 191, 51, 0, 0, 0, 0, 122, 115, 0],
            &[0, 224, 199, 196, 196, 196, 196, 205, 114, 0],
            &[0, 217, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 83, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 214, 32, 0, 0, 0, 15, 12, 0],
            &[0, 0, 66, 201, 191, 170, 195, 197, 32, 0],
            &[0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĕ' / 0x114
        'Ĕ' => Some(&[
            &[0, 0, 36, 85, 0, 0, 104, 12, 0, 0],
            &[0, 0, 6, 186, 120, 139, 164, 0, 0, 0],
            &[0, 0, 0, 5, 63, 55, 1, 0, 0, 0],
            &[0, 34, 108, 108, 108, 108, 108, 108, 22, 0],
            &[0, 81, 196, 104, 104, 104, 104, 104, 21, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 240, 217, 217, 217, 217, 217, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĕ' / 0x115
        'ĕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 51, 88, 0, 0, 130, 4, 0, 0],
            &[0, 0, 7, 183, 139, 167, 134, 0, 0, 0],
            &[0, 0, 0, 2, 43, 29, 0, 0, 0, 0],
            &[0, 0, 4, 92, 154, 158, 94, 2, 0, 0],
            &[0, 3, 183, 149, 53, 43, 147, 161, 0, 0],
            &[0, 105, 156, 0, 0, 0, 0, 193, 52, 0],
            &[0, 191, 51, 0, 0, 0, 0, 122, 115, 0],
            &[0, 224, 199, 196, 196, 196, 196, 205, 114, 0],
            &[0, 217, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 83, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 214, 32, 0, 0, 0, 15, 12, 0],
            &[0, 0, 66, 201, 191, 170, 195, 197, 32, 0],
            &[0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ė' / 0x116
        'Ė' => Some(&[
            &[0, 0, 0, 0, 120, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 34, 108, 108, 108, 108, 108, 108, 22, 0],
            &[0, 81, 196, 104, 104, 104, 104, 104, 21, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 240, 217, 217, 217, 217, 217, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ė' / 0x117
        'ė' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 173, 108, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 92, 154, 158, 94, 2, 0, 0],
            &[0, 3, 183, 149, 53, 43, 147, 161, 0, 0],
            &[0, 105, 156, 0, 0, 0, 0, 193, 52, 0],
            &[0, 191, 51, 0, 0, 0, 0, 122, 115, 0],
            &[0, 224, 199, 196, 196, 196, 196, 205, 114, 0],
            &[0, 217, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 83, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 214, 32, 0, 0, 0, 15, 12, 0],
            &[0, 0, 66, 201, 191, 170, 195, 197, 32, 0],
            &[0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ę' / 0x118
        'Ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 34, 108, 108, 108, 108, 108, 108, 22, 0],
            &[0, 81, 196, 104, 104, 104, 104, 104, 21, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 240, 217, 217, 217, 232, 221, 45, 0],
            &[0, 0, 0, 0, 0, 25, 166, 12, 0, 0],
            &[0, 0, 0, 0, 0, 151, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 113, 66, 0, 0],
        ]),
        // letter: 'ę' / 0x119
        'ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 92, 154, 158, 94, 2, 0, 0],
            &[0, 3, 183, 149, 53, 43, 147, 161, 0, 0],
            &[0, 105, 156, 0, 0, 0, 0, 193, 52, 0],
            &[0, 191, 51, 0, 0, 0, 0, 122, 115, 0],
            &[0, 224, 199, 196, 196, 196, 196, 205, 114, 0],
            &[0, 217, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 83, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 214, 32, 0, 0, 0, 15, 12, 0],
            &[0, 0, 66, 201, 191, 170, 195, 255, 38, 0],
            &[0, 0, 0, 0, 17, 34, 149, 57, 0, 0],
            &[0, 0, 0, 0, 0, 46, 147, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 82, 114, 2, 0],
        ]),
        // letter: 'Ě' / 0x11a
        'Ě' => Some(&[
            &[0, 0, 26, 179, 20, 40, 177, 12, 0, 0],
            &[0, 0, 0, 49, 195, 201, 24, 0, 0, 0],
            &[0, 0, 0, 0, 67, 49, 0, 0, 0, 0],
            &[0, 34, 108, 108, 108, 108, 108, 108, 22, 0],
            &[0, 81, 196, 104, 104, 104, 104, 104, 21, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 197, 106, 106, 106, 106, 94, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 240, 217, 217, 217, 217, 217, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 11, 0, 0, 17, 11, 0, 0],
        ]),
        // letter: 'ě' / 0x11b
        'ě' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 29, 17, 0, 0, 34, 13, 0, 0],
            &[0, 0, 24, 182, 22, 65, 164, 3, 0, 0],
            &[0, 0, 0, 48, 203, 197, 10, 0, 0, 0],
            &[0, 0, 0, 0, 54, 31, 0, 0, 0, 0],
            &[0, 0, 4, 92, 154, 158, 94, 2, 0, 0],
            &[0, 3, 183, 149, 53, 43, 147, 161, 0, 0],
            &[0, 105, 156, 0, 0, 0, 0, 193, 52, 0],
            &[0, 191, 51, 0, 0, 0, 0, 122, 115, 0],
            &[0, 224, 199, 196, 196, 196, 196, 205, 114, 0],
            &[0, 217, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 83, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 214, 32, 0, 0, 0, 15, 12, 0],
            &[0, 0, 66, 201, 191, 170, 195, 197, 32, 0],
            &[0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => Some(&[
            &[0, 0, 0, 1, 177, 214, 37, 0, 0, 0],
            &[0, 0, 0, 128, 98, 31, 184, 16, 0, 0],
            &[0, 0, 10, 68, 0, 0, 30, 48, 0, 0],
            &[0, 0, 0, 36, 119, 145, 123, 51, 0, 0],
            &[0, 0, 114, 213, 106, 71, 96, 170, 8, 0],
            &[0, 64, 209, 14, 0, 0, 0, 0, 0, 0],
            &[0, 185, 72, 0, 0, 0, 0, 0, 0, 0],
            &[7, 236, 5, 0, 0, 0, 0, 0, 0, 0],
            &[31, 211, 0, 0, 0, 0, 0, 0, 0, 0],
            &[46, 198, 0, 0, 0, 202, 210, 210, 60, 0],
            &[31, 213, 0, 0, 0, 5, 6, 161, 73, 0],
            &[5, 237, 9, 0, 0, 0, 0, 159, 73, 0],
            &[0, 172, 94, 0, 0, 0, 0, 159, 73, 0],
            &[0, 42, 225, 55, 0, 0, 0, 163, 73, 0],
            &[0, 0, 57, 201, 206, 183, 213, 195, 39, 0],
            &[0, 0, 0, 0, 15, 32, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 27, 0, 0, 0, 0],
        ]),
        // letter: 'ĝ' / 0x11d
        'ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 21, 0, 0, 0, 0],
            &[0, 0, 0, 48, 204, 188, 4, 0, 0, 0],
            &[0, 0, 23, 180, 20, 83, 146, 0, 0, 0],
            &[0, 0, 40, 20, 0, 0, 50, 10, 0, 0],
            &[0, 0, 25, 131, 165, 134, 26, 85, 16, 0],
            &[0, 27, 222, 107, 42, 64, 174, 206, 32, 0],
            &[0, 151, 112, 0, 0, 0, 29, 252, 32, 0],
            &[0, 222, 25, 0, 0, 0, 0, 211, 32, 0],
            &[0, 243, 1, 0, 0, 0, 0, 190, 32, 0],
            &[0, 242, 8, 0, 0, 0, 0, 200, 32, 0],
            &[0, 201, 54, 0, 0, 0, 6, 238, 32, 0],
            &[0, 103, 183, 8, 0, 0, 123, 233, 32, 0],
            &[0, 1, 138, 218, 173, 196, 118, 191, 32, 0],
            &[0, 0, 0, 11, 34, 9, 0, 199, 28, 0],
            &[0, 0, 0, 0, 0, 0, 7, 228, 6, 0],
            &[0, 57, 169, 203, 203, 205, 126, 6, 0, 0],
        ]),
        // letter: 'Ğ' / 0x11e
        'Ğ' => Some(&[
            &[0, 0, 0, 121, 0, 0, 61, 55, 0, 0],
            &[0, 0, 0, 141, 146, 122, 190, 16, 0, 0],
            &[0, 0, 0, 0, 49, 65, 10, 0, 0, 0],
            &[0, 0, 0, 36, 119, 145, 123, 51, 0, 0],
            &[0, 0, 114, 213, 106, 71, 96, 170, 8, 0],
            &[0, 64, 209, 14, 0, 0, 0, 0, 0, 0],
            &[0, 185, 72, 0, 0, 0, 0, 0, 0, 0],
            &[7, 236, 5, 0, 0, 0, 0, 0, 0, 0],
            &[31, 211, 0, 0, 0, 0, 0, 0, 0, 0],
            &[46, 198, 0, 0, 0, 202, 210, 210, 60, 0],
            &[31, 213, 0, 0, 0, 5, 6, 161, 73, 0],
            &[5, 237, 9, 0, 0, 0, 0, 159, 73, 0],
            &[0, 172, 94, 0, 0, 0, 0, 159, 73, 0],
            &[0, 42, 225, 55, 0, 0, 0, 163, 73, 0],
            &[0, 0, 57, 201, 206, 183, 213, 195, 39, 0],
            &[0, 0, 0, 0, 15, 32, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ğ' / 0x11f
        'ğ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 76, 0, 0, 134, 0, 0, 0],
            &[0, 0, 13, 189, 136, 172, 120, 0, 0, 0],
            &[0, 0, 0, 4, 44, 26, 0, 0, 0, 0],
            &[0, 0, 25, 131, 165, 134, 26, 85, 16, 0],
            &[0, 27, 222, 107, 42, 64, 174, 206, 32, 0],
            &[0, 151, 112, 0, 0, 0, 29, 252, 32, 0],
            &[0, 222, 25, 0, 0, 0, 0, 211, 32, 0],
            &[0, 243, 1, 0, 0, 0, 0, 190, 32, 0],
            &[0, 242, 8, 0, 0, 0, 0, 200, 32, 0],
            &[0, 201, 54, 0, 0, 0, 6, 238, 32, 0],
            &[0, 103, 183, 8, 0, 0, 123, 233, 32, 0],
            &[0, 1, 138, 218, 173, 196, 118, 191, 32, 0],
            &[0, 0, 0, 11, 34, 9, 0, 199, 28, 0],
            &[0, 0, 0, 0, 0, 0, 7, 228, 6, 0],
            &[0, 57, 169, 203, 203, 205, 126, 6, 0, 0],
        ]),
        // letter: 'Ġ' / 0x120
        'Ġ' => Some(&[
            &[0, 0, 0, 0, 72, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 156, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 119, 145, 123, 51, 0, 0],
            &[0, 0, 114, 213, 106, 71, 96, 170, 8, 0],
            &[0, 64, 209, 14, 0, 0, 0, 0, 0, 0],
            &[0, 185, 72, 0, 0, 0, 0, 0, 0, 0],
            &[7, 236, 5, 0, 0, 0, 0, 0, 0, 0],
            &[31, 211, 0, 0, 0, 0, 0, 0, 0, 0],
            &[46, 198, 0, 0, 0, 202, 210, 210, 60, 0],
            &[31, 213, 0, 0, 0, 5, 6, 161, 73, 0],
            &[5, 237, 9, 0, 0, 0, 0, 159, 73, 0],
            &[0, 172, 94, 0, 0, 0, 0, 159, 73, 0],
            &[0, 42, 225, 55, 0, 0, 0, 163, 73, 0],
            &[0, 0, 57, 201, 206, 183, 213, 195, 39, 0],
            &[0, 0, 0, 0, 15, 32, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ġ' / 0x121
        'ġ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 131, 165, 134, 26, 85, 16, 0],
            &[0, 27, 222, 107, 42, 64, 174, 206, 32, 0],
            &[0, 151, 112, 0, 0, 0, 29, 252, 32, 0],
            &[0, 222, 25, 0, 0, 0, 0, 211, 32, 0],
            &[0, 243, 1, 0, 0, 0, 0, 190, 32, 0],
            &[0, 242, 8, 0, 0, 0, 0, 200, 32, 0],
            &[0, 201, 54, 0, 0, 0, 6, 238, 32, 0],
            &[0, 103, 183, 8, 0, 0, 123, 233, 32, 0],
            &[0, 1, 138, 218, 173, 196, 118, 191, 32, 0],
            &[0, 0, 0, 11, 34, 9, 0, 199, 28, 0],
            &[0, 0, 0, 0, 0, 0, 7, 228, 6, 0],
            &[0, 57, 169, 203, 203, 205, 126, 6, 0, 0],
        ]),
        // letter: 'Ģ' / 0x122
        'Ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 119, 145, 123, 51, 0, 0],
            &[0, 0, 114, 213, 106, 71, 96, 170, 8, 0],
            &[0, 64, 209, 14, 0, 0, 0, 0, 0, 0],
            &[0, 185, 72, 0, 0, 0, 0, 0, 0, 0],
            &[7, 236, 5, 0, 0, 0, 0, 0, 0, 0],
            &[31, 211, 0, 0, 0, 0, 0, 0, 0, 0],
            &[46, 198, 0, 0, 0, 202, 210, 210, 60, 0],
            &[31, 213, 0, 0, 0, 5, 6, 161, 73, 0],
            &[5, 237, 9, 0, 0, 0, 0, 159, 73, 0],
            &[0, 172, 94, 0, 0, 0, 0, 159, 73, 0],
            &[0, 42, 225, 55, 0, 0, 0, 163, 73, 0],
            &[0, 0, 57, 201, 206, 183, 213, 195, 39, 0],
            &[0, 0, 0, 0, 15, 37, 6, 0, 0, 0],
            &[0, 0, 0, 0, 46, 184, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ģ' / 0x123
        'ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 36, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 8, 0, 0, 0, 0],
            &[0, 0, 25, 131, 165, 134, 26, 85, 16, 0],
            &[0, 27, 222, 107, 42, 64, 174, 206, 32, 0],
            &[0, 151, 112, 0, 0, 0, 29, 252, 32, 0],
            &[0, 222, 25, 0, 0, 0, 0, 211, 32, 0],
            &[0, 243, 1, 0, 0, 0, 0, 190, 32, 0],
            &[0, 242, 8, 0, 0, 0, 0, 200, 32, 0],
            &[0, 201, 54, 0, 0, 0, 6, 238, 32, 0],
            &[0, 103, 183, 8, 0, 0, 123, 233, 32, 0],
            &[0, 1, 138, 218, 173, 196, 118, 191, 32, 0],
            &[0, 0, 0, 11, 34, 9, 0, 199, 28, 0],
            &[0, 0, 0, 0, 0, 0, 7, 228, 6, 0],
            &[0, 57, 169, 203, 203, 205, 126, 6, 0, 0],
        ]),
        // letter: 'Ĥ' / 0x124
        'Ĥ' => Some(&[
            &[0, 0, 0, 37, 212, 179, 1, 0, 0, 0],
            &[0, 0, 15, 182, 29, 101, 130, 0, 0, 0],
            &[0, 0, 48, 30, 0, 0, 68, 10, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 69, 31, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 144, 106, 106, 106, 106, 202, 73, 0],
            &[0, 171, 147, 110, 110, 110, 110, 203, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 12, 0, 0, 0, 0],
        ]),
        // letter: 'ĥ' / 0x125
        'ĥ' => Some(&[
            &[0, 136, 97, 129, 109, 0, 0, 0, 0, 0],
            &[40, 96, 0, 0, 112, 24, 0, 0, 0, 0],
            &[0, 0, 20, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 106, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 106, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 106, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 106, 41, 137, 159, 107, 6, 0],
            &[0, 0, 131, 151, 159, 51, 45, 155, 172, 0],
            &[0, 0, 131, 204, 2, 0, 0, 3, 228, 19],
            &[0, 0, 131, 128, 0, 0, 0, 0, 189, 50],
            &[0, 0, 131, 107, 0, 0, 0, 0, 184, 53],
            &[0, 0, 131, 106, 0, 0, 0, 0, 184, 53],
            &[0, 0, 131, 106, 0, 0, 0, 0, 184, 53],
            &[0, 0, 131, 106, 0, 0, 0, 0, 184, 53],
            &[0, 0, 131, 106, 0, 0, 0, 0, 184, 53],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 177, 159, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ħ' / 0x126
        'Ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 69, 31, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[47, 219, 112, 47, 47, 47, 47, 210, 120, 28],
            &[153, 255, 219, 153, 153, 153, 153, 255, 227, 92],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 144, 106, 106, 106, 106, 202, 73, 0],
            &[0, 171, 147, 110, 110, 110, 110, 203, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ħ' / 0x127
        'ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 106, 0, 0, 0, 0, 0, 0, 0],
            &[155, 255, 255, 155, 155, 78, 0, 0, 0, 0],
            &[32, 163, 139, 32, 32, 16, 0, 0, 0, 0],
            &[0, 131, 106, 41, 137, 159, 107, 6, 0, 0],
            &[0, 131, 151, 159, 51, 45, 155, 172, 0, 0],
            &[0, 131, 204, 2, 0, 0, 3, 228, 19, 0],
            &[0, 131, 128, 0, 0, 0, 0, 189, 50, 0],
            &[0, 131, 107, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĩ' / 0x128
        'Ĩ' => Some(&[
            &[0, 0, 23, 116, 58, 0, 21, 66, 0, 0],
            &[0, 0, 170, 85, 158, 161, 180, 57, 0, 0],
            &[0, 0, 48, 0, 0, 44, 38, 0, 0, 0],
            &[0, 0, 91, 108, 108, 108, 108, 47, 0, 0],
            &[0, 0, 55, 79, 200, 131, 73, 27, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 146, 188, 236, 211, 182, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĩ' / 0x129
        'ĩ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 139, 79, 0, 28, 76, 0, 0],
            &[0, 0, 174, 63, 137, 181, 191, 47, 0, 0],
            &[0, 0, 33, 0, 0, 24, 21, 0, 0, 0],
            &[0, 37, 131, 131, 131, 72, 0, 0, 0, 0],
            &[0, 11, 46, 59, 139, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 193, 180, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ī' / 0x12a
        'Ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 72, 178, 178, 178, 178, 1, 0, 0],
            &[0, 0, 7, 18, 18, 18, 18, 0, 0, 0],
            &[0, 0, 91, 108, 108, 108, 108, 47, 0, 0],
            &[0, 0, 55, 79, 200, 131, 73, 27, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 146, 188, 236, 211, 182, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ī' / 0x12b
        'ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 80, 196, 196, 196, 196, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 131, 131, 131, 72, 0, 0, 0, 0],
            &[0, 11, 46, 59, 139, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 193, 180, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => Some(&[
            &[0, 0, 57, 64, 0, 0, 116, 0, 0, 0],
            &[0, 0, 18, 194, 113, 152, 137, 0, 0, 0],
            &[0, 0, 0, 11, 67, 46, 0, 0, 0, 0],
            &[0, 0, 91, 108, 108, 108, 108, 47, 0, 0],
            &[0, 0, 55, 79, 200, 131, 73, 27, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 146, 188, 236, 211, 182, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĭ' / 0x12d
        'ĭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 76, 0, 0, 134, 0, 0, 0],
            &[0, 0, 13, 189, 136, 172, 120, 0, 0, 0],
            &[0, 0, 0, 4, 44, 26, 0, 0, 0, 0],
            &[0, 37, 131, 131, 131, 72, 0, 0, 0, 0],
            &[0, 11, 46, 59, 139, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 193, 180, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Į' / 0x12e
        'Į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 108, 108, 108, 108, 47, 0, 0],
            &[0, 0, 55, 79, 200, 131, 73, 27, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 146, 188, 236, 211, 202, 75, 0, 0],
            &[0, 0, 0, 0, 0, 126, 78, 0, 0, 0],
            &[0, 0, 0, 0, 36, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 115, 6, 0, 0],
        ]),
        // letter: 'į' / 0x12f
        'į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 139, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 131, 131, 131, 72, 0, 0, 0, 0],
            &[0, 11, 46, 59, 139, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 208, 185, 95, 0],
            &[0, 0, 0, 0, 0, 25, 166, 12, 0, 0],
            &[0, 0, 0, 0, 0, 151, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 113, 66, 0, 0],
        ]),
        // letter: 'İ' / 0x130
        'İ' => Some(&[
            &[0, 0, 0, 0, 143, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 159, 78, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 108, 108, 108, 108, 47, 0, 0],
            &[0, 0, 55, 79, 200, 131, 73, 27, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 146, 188, 236, 211, 182, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ı' / 0x131
        'ı' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 131, 131, 131, 72, 0, 0, 0, 0],
            &[0, 11, 46, 59, 139, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 193, 180, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĳ' / 0x132
        'Ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 29, 71, 0, 0, 0, 4, 96, 0, 0],
            &[0, 69, 167, 0, 0, 0, 10, 227, 0, 0],
            &[0, 69, 167, 0, 0, 0, 10, 227, 0, 0],
            &[0, 69, 167, 0, 0, 0, 10, 227, 0, 0],
            &[0, 69, 167, 0, 0, 0, 10, 227, 0, 0],
            &[0, 69, 167, 0, 0, 0, 10, 227, 0, 0],
            &[0, 69, 167, 0, 0, 0, 10, 227, 0, 0],
            &[0, 69, 167, 0, 0, 0, 10, 227, 0, 0],
            &[0, 69, 167, 0, 0, 0, 12, 226, 0, 0],
            &[0, 15, 36, 0, 0, 0, 39, 206, 0, 0],
            &[0, 12, 10, 0, 0, 1, 154, 133, 0, 0],
            &[0, 35, 217, 192, 174, 214, 172, 11, 0, 0],
            &[0, 0, 0, 27, 43, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĳ' / 0x133
        'ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 218, 43, 0, 0, 128, 155, 0, 0],
            &[0, 7, 140, 21, 0, 0, 75, 94, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 120, 1, 0, 0, 47, 74, 0, 0],
            &[0, 0, 235, 2, 0, 0, 92, 145, 0, 0],
            &[0, 0, 235, 2, 0, 0, 92, 145, 0, 0],
            &[0, 0, 235, 2, 0, 0, 92, 145, 0, 0],
            &[0, 0, 235, 2, 0, 0, 92, 145, 0, 0],
            &[0, 0, 235, 2, 0, 0, 92, 145, 0, 0],
            &[0, 0, 235, 2, 0, 0, 92, 145, 0, 0],
            &[0, 0, 235, 2, 0, 0, 92, 145, 0, 0],
            &[0, 0, 235, 2, 0, 0, 92, 145, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 145, 0, 0],
            &[0, 0, 0, 0, 0, 0, 120, 128, 0, 0],
            &[0, 0, 157, 203, 202, 194, 83, 0, 0, 0],
        ]),
        // letter: 'Ĵ' / 0x134
        'Ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 71, 220, 138, 0, 0],
            &[0, 0, 0, 0, 36, 179, 11, 144, 87, 0],
            &[0, 0, 0, 0, 65, 13, 0, 0, 78, 0],
            &[0, 0, 0, 0, 0, 0, 91, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 216, 21, 0, 0],
            &[0, 0, 0, 0, 0, 1, 236, 6, 0, 0],
            &[0, 14, 9, 0, 0, 82, 194, 0, 0, 0],
            &[0, 41, 214, 188, 186, 217, 44, 0, 0, 0],
            &[0, 0, 2, 31, 36, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 5, 0, 0],
        ]),
        // letter: 'ĵ' / 0x135
        'ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 58, 0, 0, 0, 0],
            &[0, 0, 0, 0, 141, 204, 99, 0, 0, 0],
            &[0, 0, 0, 93, 132, 4, 167, 58, 0, 0],
            &[0, 0, 0, 60, 0, 0, 4, 55, 0, 0],
            &[0, 0, 105, 131, 131, 131, 4, 0, 0, 0],
            &[0, 0, 33, 53, 66, 236, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 229, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 229, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 229, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 229, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 229, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 229, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 229, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 229, 8, 0, 0, 0],
            &[0, 0, 0, 0, 5, 240, 1, 0, 0, 0],
            &[0, 166, 203, 212, 164, 24, 0, 0, 0, 0],
        ]),
        // letter: 'Ķ' / 0x136
        'Ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 75, 46, 0],
            &[0, 171, 65, 0, 0, 0, 82, 198, 9, 0],
            &[0, 171, 65, 0, 0, 56, 215, 23, 0, 0],
            &[0, 171, 65, 0, 34, 220, 44, 0, 0, 0],
            &[0, 171, 65, 18, 213, 71, 0, 0, 0, 0],
            &[0, 171, 70, 193, 148, 0, 0, 0, 0, 0],
            &[0, 171, 199, 130, 218, 25, 0, 0, 0, 0],
            &[0, 171, 134, 0, 88, 179, 0, 0, 0, 0],
            &[0, 171, 65, 0, 0, 172, 95, 0, 0, 0],
            &[0, 171, 65, 0, 0, 22, 219, 27, 0, 0],
            &[0, 171, 65, 0, 0, 0, 86, 182, 0, 0],
            &[0, 171, 65, 0, 0, 0, 0, 171, 98, 0],
            &[0, 0, 0, 0, 2, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 124, 106, 0, 0, 0, 0],
            &[0, 0, 0, 5, 79, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ķ' / 0x137
        'ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 229, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 229, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 229, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 229, 0, 0, 0, 0, 94, 54, 0],
            &[0, 8, 229, 0, 0, 0, 113, 174, 3, 0],
            &[0, 8, 229, 0, 0, 102, 184, 6, 0, 0],
            &[0, 8, 227, 0, 91, 193, 10, 0, 0, 0],
            &[0, 8, 221, 79, 215, 162, 0, 0, 0, 0],
            &[0, 8, 240, 171, 8, 173, 93, 0, 0, 0],
            &[0, 8, 236, 5, 0, 18, 214, 37, 0, 0],
            &[0, 8, 229, 0, 0, 0, 63, 203, 6, 0],
            &[0, 8, 229, 0, 0, 0, 0, 133, 143, 0],
            &[0, 0, 0, 0, 0, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 196, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 2, 0, 0, 0, 0],
        ]),
        // letter: 'ĸ' / 0x138
        'ĸ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 117, 0, 0, 0, 0, 94, 54, 0],
            &[0, 8, 229, 0, 0, 0, 113, 174, 3, 0],
            &[0, 8, 229, 0, 0, 102, 184, 6, 0, 0],
            &[0, 8, 227, 0, 91, 193, 10, 0, 0, 0],
            &[0, 8, 221, 79, 215, 162, 0, 0, 0, 0],
            &[0, 8, 240, 171, 8, 173, 93, 0, 0, 0],
            &[0, 8, 236, 5, 0, 18, 214, 37, 0, 0],
            &[0, 8, 229, 0, 0, 0, 63, 203, 6, 0],
            &[0, 8, 229, 0, 0, 0, 0, 133, 143, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĺ' / 0x139
        'Ĺ' => Some(&[
            &[0, 0, 0, 174, 107, 0, 0, 0, 0, 0],
            &[0, 0, 116, 127, 0, 0, 0, 0, 0, 0],
            &[0, 4, 74, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 94, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 223, 221, 221, 221, 221, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 36, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĺ' / 0x13a
        'ĺ' => Some(&[
            &[0, 0, 0, 0, 29, 199, 26, 0, 0, 0],
            &[0, 0, 0, 0, 112, 26, 0, 0, 0, 0],
            &[0, 13, 40, 40, 40, 22, 0, 0, 0, 0],
            &[0, 43, 137, 149, 195, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 193, 180, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 74, 182, 3, 0, 0],
        ]),
        // letter: 'Ļ' / 0x13b
        'Ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 223, 221, 221, 221, 221, 45, 0, 0],
            &[0, 0, 0, 0, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 188, 0, 0, 0, 0, 0],
            &[0, 0, 0, 83, 1, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ļ' / 0x13c
        'ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 40, 40, 40, 22, 0, 0, 0, 0],
            &[0, 43, 137, 149, 195, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 193, 180, 95, 0],
            &[0, 0, 0, 0, 2, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 118, 0, 0, 0, 0],
            &[0, 0, 0, 1, 83, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ľ' / 0x13d
        'Ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 6, 0, 0, 8, 92, 0, 0, 0],
            &[0, 223, 14, 0, 0, 57, 159, 0, 0, 0],
            &[0, 223, 14, 0, 0, 128, 48, 0, 0, 0],
            &[0, 223, 14, 0, 0, 19, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 223, 221, 221, 221, 221, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ľ' / 0x13e
        'ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 40, 40, 40, 22, 0, 17, 20, 0],
            &[0, 43, 137, 149, 195, 141, 0, 139, 87, 0],
            &[0, 0, 0, 0, 96, 141, 0, 180, 7, 0],
            &[0, 0, 0, 0, 96, 141, 3, 58, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 193, 180, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 42, 129, 0, 0, 0],
            &[0, 223, 14, 0, 0, 77, 212, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 223, 223, 221, 221, 221, 221, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŀ' / 0x140
        'ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[13, 40, 40, 40, 22, 0, 0, 0, 0, 0],
            &[43, 137, 149, 195, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 141, 0, 0, 0, 23, 0],
            &[0, 0, 0, 96, 141, 0, 0, 11, 253, 69],
            &[0, 0, 0, 96, 141, 0, 0, 0, 90, 14],
            &[0, 0, 0, 96, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 141, 0, 0, 0, 0, 0],
            &[95, 180, 191, 221, 231, 193, 180, 95, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ł' / 0x141
        'Ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 75, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 178, 0, 16, 3, 0, 0, 0],
            &[0, 0, 59, 178, 92, 204, 36, 0, 0, 0],
            &[0, 0, 99, 255, 140, 12, 0, 0, 0, 0],
            &[0, 115, 249, 223, 0, 0, 0, 0, 0, 0],
            &[0, 34, 60, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 245, 221, 221, 221, 221, 187, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ł' / 0x142
        'ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 40, 40, 40, 22, 0, 0, 0, 0],
            &[0, 43, 137, 149, 195, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 52, 73, 0, 0],
            &[0, 0, 0, 0, 108, 255, 178, 31, 0, 0],
            &[0, 0, 0, 75, 255, 229, 0, 0, 0, 0],
            &[0, 0, 75, 155, 113, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 141, 0, 0, 0, 0],
            &[0, 95, 180, 191, 221, 231, 193, 180, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ń' / 0x143
        'Ń' => Some(&[
            &[0, 0, 0, 0, 0, 100, 179, 2, 0, 0],
            &[0, 0, 0, 0, 51, 182, 9, 0, 0, 0],
            &[0, 0, 0, 0, 70, 8, 0, 0, 0, 0],
            &[0, 72, 65, 0, 0, 0, 0, 67, 31, 0],
            &[0, 171, 224, 9, 0, 0, 0, 159, 73, 0],
            &[0, 171, 154, 112, 0, 0, 0, 159, 73, 0],
            &[0, 171, 63, 208, 8, 0, 0, 159, 73, 0],
            &[0, 171, 56, 119, 107, 0, 0, 159, 73, 0],
            &[0, 171, 57, 13, 208, 6, 0, 159, 73, 0],
            &[0, 171, 57, 0, 125, 102, 0, 159, 73, 0],
            &[0, 171, 57, 0, 15, 207, 4, 159, 73, 0],
            &[0, 171, 57, 0, 0, 131, 97, 157, 73, 0],
            &[0, 171, 57, 0, 0, 18, 206, 154, 73, 0],
            &[0, 171, 57, 0, 0, 0, 136, 228, 73, 0],
            &[0, 171, 57, 0, 0, 0, 21, 245, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 6, 0, 0],
        ]),
        // letter: 'ń' / 0x144
        'ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 13, 0, 0],
            &[0, 0, 0, 0, 0, 96, 179, 2, 0, 0],
            &[0, 0, 0, 0, 50, 179, 9, 0, 0, 0],
            &[0, 0, 0, 0, 52, 7, 0, 0, 0, 0],
            &[0, 67, 34, 45, 137, 160, 112, 8, 0, 0],
            &[0, 131, 144, 159, 55, 44, 146, 180, 0, 0],
            &[0, 131, 207, 1, 0, 0, 1, 224, 22, 0],
            &[0, 131, 133, 0, 0, 0, 0, 188, 51, 0],
            &[0, 131, 108, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ņ' / 0x145
        'Ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 65, 0, 0, 0, 0, 67, 31, 0],
            &[0, 171, 224, 9, 0, 0, 0, 159, 73, 0],
            &[0, 171, 154, 112, 0, 0, 0, 159, 73, 0],
            &[0, 171, 63, 208, 8, 0, 0, 159, 73, 0],
            &[0, 171, 56, 119, 107, 0, 0, 159, 73, 0],
            &[0, 171, 57, 13, 208, 6, 0, 159, 73, 0],
            &[0, 171, 57, 0, 125, 102, 0, 159, 73, 0],
            &[0, 171, 57, 0, 15, 207, 4, 159, 73, 0],
            &[0, 171, 57, 0, 0, 131, 97, 157, 73, 0],
            &[0, 171, 57, 0, 0, 18, 206, 154, 73, 0],
            &[0, 171, 57, 0, 0, 0, 136, 228, 73, 0],
            &[0, 171, 57, 0, 0, 0, 21, 245, 73, 0],
            &[0, 0, 0, 0, 3, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 186, 45, 0, 0, 0, 0],
            &[0, 0, 0, 36, 48, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ņ' / 0x146
        'ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 67, 34, 45, 137, 160, 112, 8, 0, 0],
            &[0, 131, 144, 159, 55, 44, 146, 180, 0, 0],
            &[0, 131, 207, 1, 0, 0, 1, 224, 22, 0],
            &[0, 131, 133, 0, 0, 0, 0, 188, 51, 0],
            &[0, 131, 108, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 0, 0, 0, 3, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 77, 0, 0, 0, 0],
            &[0, 0, 0, 18, 65, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ň' / 0x147
        'Ň' => Some(&[
            &[0, 0, 32, 177, 15, 48, 173, 9, 0, 0],
            &[0, 0, 0, 58, 193, 200, 19, 0, 0, 0],
            &[0, 0, 0, 0, 71, 44, 0, 0, 0, 0],
            &[0, 72, 65, 0, 0, 0, 0, 67, 31, 0],
            &[0, 171, 224, 9, 0, 0, 0, 159, 73, 0],
            &[0, 171, 154, 112, 0, 0, 0, 159, 73, 0],
            &[0, 171, 63, 208, 8, 0, 0, 159, 73, 0],
            &[0, 171, 56, 119, 107, 0, 0, 159, 73, 0],
            &[0, 171, 57, 13, 208, 6, 0, 159, 73, 0],
            &[0, 171, 57, 0, 125, 102, 0, 159, 73, 0],
            &[0, 171, 57, 0, 15, 207, 4, 159, 73, 0],
            &[0, 171, 57, 0, 0, 131, 97, 157, 73, 0],
            &[0, 171, 57, 0, 0, 18, 206, 154, 73, 0],
            &[0, 171, 57, 0, 0, 0, 136, 228, 73, 0],
            &[0, 171, 57, 0, 0, 0, 21, 245, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 9, 0, 0, 19, 9, 0, 0],
        ]),
        // letter: 'ň' / 0x148
        'ň' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 23, 0, 0, 28, 19, 0, 0],
            &[0, 0, 14, 180, 34, 48, 175, 8, 0, 0],
            &[0, 0, 0, 32, 203, 203, 20, 0, 0, 0],
            &[0, 0, 0, 0, 47, 38, 0, 0, 0, 0],
            &[0, 67, 34, 45, 137, 160, 112, 8, 0, 0],
            &[0, 131, 144, 159, 55, 44, 146, 180, 0, 0],
            &[0, 131, 207, 1, 0, 0, 1, 224, 22, 0],
            &[0, 131, 133, 0, 0, 0, 0, 188, 51, 0],
            &[0, 131, 108, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 184, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŉ' / 0x149
        'ŉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 55, 0, 0, 0, 0, 0, 0, 0],
            &[0, 174, 92, 0, 0, 0, 0, 0, 0, 0],
            &[0, 222, 21, 0, 0, 0, 0, 0, 0, 0],
            &[34, 180, 29, 72, 20, 127, 161, 119, 11, 0],
            &[45, 49, 57, 172, 174, 66, 42, 141, 186, 0],
            &[0, 0, 57, 249, 28, 0, 0, 1, 223, 24],
            &[0, 0, 57, 206, 0, 0, 0, 0, 188, 51],
            &[0, 0, 57, 182, 0, 0, 0, 0, 184, 53],
            &[0, 0, 57, 180, 0, 0, 0, 0, 184, 53],
            &[0, 0, 57, 180, 0, 0, 0, 0, 184, 53],
            &[0, 0, 57, 180, 0, 0, 0, 0, 184, 53],
            &[0, 0, 57, 180, 0, 0, 0, 0, 184, 53],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 65, 0, 0, 0, 0, 67, 31, 0],
            &[0, 171, 224, 9, 0, 0, 0, 159, 73, 0],
            &[0, 171, 154, 112, 0, 0, 0, 159, 73, 0],
            &[0, 171, 63, 208, 8, 0, 0, 159, 73, 0],
            &[0, 171, 56, 119, 107, 0, 0, 159, 73, 0],
            &[0, 171, 57, 13, 208, 6, 0, 159, 73, 0],
            &[0, 171, 57, 0, 125, 102, 0, 159, 73, 0],
            &[0, 171, 57, 0, 15, 207, 4, 159, 73, 0],
            &[0, 171, 57, 0, 0, 131, 97, 157, 73, 0],
            &[0, 171, 57, 0, 0, 18, 206, 154, 73, 0],
            &[0, 171, 57, 0, 0, 0, 136, 228, 73, 0],
            &[0, 171, 57, 0, 0, 0, 21, 255, 136, 0],
            &[0, 0, 0, 0, 0, 0, 0, 184, 53, 0],
            &[0, 0, 0, 0, 7, 0, 63, 219, 9, 0],
            &[0, 0, 0, 0, 133, 217, 189, 52, 0, 0],
        ]),
        // letter: 'ŋ' / 0x14b
        'ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 67, 34, 45, 137, 160, 112, 8, 0, 0],
            &[0, 131, 144, 159, 55, 44, 146, 180, 0, 0],
            &[0, 131, 207, 1, 0, 0, 1, 224, 22, 0],
            &[0, 131, 133, 0, 0, 0, 0, 188, 51, 0],
            &[0, 131, 108, 0, 0, 0, 0, 184, 53, 0],
            &[0, 131, 106, 0, 0, 0, 0, 254, 74, 0],
            &[0, 131, 106, 0, 0, 0, 0, 255, 106, 0],
            &[0, 131, 106, 0, 0, 0, 0, 255, 106, 0],
            &[0, 131, 106, 0, 0, 0, 0, 255, 106, 0],
            &[0, 0, 0, 0, 0, 0, 0, 184, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 188, 45, 0],
            &[0, 0, 0, 0, 1, 191, 188, 74, 0, 0],
        ]),
        // letter: 'Ō' / 0x14c
        'Ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 72, 178, 178, 178, 178, 1, 0, 0],
            &[0, 0, 7, 18, 18, 18, 18, 0, 0, 0],
            &[0, 0, 2, 85, 140, 133, 51, 0, 0, 0],
            &[0, 5, 184, 156, 73, 85, 200, 103, 0, 0],
            &[0, 112, 152, 0, 0, 0, 13, 220, 27, 0],
            &[0, 212, 32, 0, 0, 0, 0, 129, 115, 0],
            &[15, 227, 0, 0, 0, 0, 0, 71, 169, 0],
            &[38, 201, 0, 0, 0, 0, 0, 45, 195, 0],
            &[47, 194, 0, 0, 0, 0, 0, 40, 200, 0],
            &[32, 209, 0, 0, 0, 0, 0, 55, 185, 0],
            &[3, 237, 4, 0, 0, 0, 0, 95, 146, 0],
            &[0, 176, 77, 0, 0, 0, 0, 174, 75, 0],
            &[0, 55, 215, 27, 0, 0, 85, 203, 2, 0],
            &[0, 0, 84, 219, 190, 205, 188, 27, 0, 0],
            &[0, 0, 0, 3, 31, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ō' / 0x14d
        'ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 80, 196, 196, 196, 196, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 110, 161, 151, 75, 0, 0, 0],
            &[0, 17, 209, 121, 46, 53, 172, 137, 0, 0],
            &[0, 142, 121, 0, 0, 0, 2, 206, 45, 0],
            &[0, 220, 24, 0, 0, 0, 0, 121, 121, 0],
            &[0, 239, 0, 0, 0, 0, 0, 91, 148, 0],
            &[0, 236, 3, 0, 0, 0, 0, 102, 140, 0],
            &[0, 188, 57, 0, 0, 0, 0, 160, 88, 0],
            &[0, 75, 195, 10, 0, 0, 58, 216, 8, 0],
            &[0, 0, 99, 211, 170, 186, 197, 39, 0, 0],
            &[0, 0, 0, 3, 32, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => Some(&[
            &[0, 0, 57, 64, 0, 0, 116, 0, 0, 0],
            &[0, 0, 18, 194, 113, 152, 137, 0, 0, 0],
            &[0, 0, 0, 11, 67, 46, 0, 0, 0, 0],
            &[0, 0, 2, 85, 140, 133, 51, 0, 0, 0],
            &[0, 5, 184, 156, 73, 85, 200, 103, 0, 0],
            &[0, 112, 152, 0, 0, 0, 13, 220, 27, 0],
            &[0, 212, 32, 0, 0, 0, 0, 129, 115, 0],
            &[15, 227, 0, 0, 0, 0, 0, 71, 169, 0],
            &[38, 201, 0, 0, 0, 0, 0, 45, 195, 0],
            &[47, 194, 0, 0, 0, 0, 0, 40, 200, 0],
            &[32, 209, 0, 0, 0, 0, 0, 55, 185, 0],
            &[3, 237, 4, 0, 0, 0, 0, 95, 146, 0],
            &[0, 176, 77, 0, 0, 0, 0, 174, 75, 0],
            &[0, 55, 215, 27, 0, 0, 85, 203, 2, 0],
            &[0, 0, 84, 219, 190, 205, 188, 27, 0, 0],
            &[0, 0, 0, 3, 31, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŏ' / 0x14f
        'ŏ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 76, 0, 0, 134, 0, 0, 0],
            &[0, 0, 13, 189, 136, 172, 120, 0, 0, 0],
            &[0, 0, 0, 4, 44, 26, 0, 0, 0, 0],
            &[0, 0, 13, 110, 161, 151, 75, 0, 0, 0],
            &[0, 17, 209, 121, 46, 53, 172, 137, 0, 0],
            &[0, 142, 121, 0, 0, 0, 2, 206, 45, 0],
            &[0, 220, 24, 0, 0, 0, 0, 121, 121, 0],
            &[0, 239, 0, 0, 0, 0, 0, 91, 148, 0],
            &[0, 236, 3, 0, 0, 0, 0, 102, 140, 0],
            &[0, 188, 57, 0, 0, 0, 0, 160, 88, 0],
            &[0, 75, 195, 10, 0, 0, 58, 216, 8, 0],
            &[0, 0, 99, 211, 170, 186, 197, 39, 0, 0],
            &[0, 0, 0, 3, 32, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ő' / 0x150
        'Ő' => Some(&[
            &[0, 0, 0, 64, 191, 2, 182, 74, 0, 0],
            &[0, 0, 11, 186, 23, 94, 127, 0, 0, 0],
            &[0, 0, 38, 32, 0, 70, 0, 0, 0, 0],
            &[0, 0, 2, 85, 140, 133, 51, 0, 0, 0],
            &[0, 5, 184, 156, 73, 85, 200, 103, 0, 0],
            &[0, 112, 152, 0, 0, 0, 13, 220, 27, 0],
            &[0, 212, 32, 0, 0, 0, 0, 129, 115, 0],
            &[15, 227, 0, 0, 0, 0, 0, 71, 169, 0],
            &[38, 201, 0, 0, 0, 0, 0, 45, 195, 0],
            &[47, 194, 0, 0, 0, 0, 0, 40, 200, 0],
            &[32, 209, 0, 0, 0, 0, 0, 55, 185, 0],
            &[3, 237, 4, 0, 0, 0, 0, 95, 146, 0],
            &[0, 176, 77, 0, 0, 0, 0, 174, 75, 0],
            &[0, 55, 215, 27, 0, 0, 85, 203, 2, 0],
            &[0, 0, 84, 219, 190, 205, 188, 27, 0, 0],
            &[0, 0, 0, 3, 31, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 5, 16, 28, 0, 0],
        ]),
        // letter: 'ő' / 0x151
        'ő' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 61, 7, 27, 42, 0, 0],
            &[0, 0, 0, 78, 177, 1, 192, 60, 0, 0],
            &[0, 0, 17, 184, 15, 109, 107, 0, 0, 0],
            &[0, 0, 31, 22, 0, 53, 0, 0, 0, 0],
            &[0, 0, 13, 110, 161, 151, 75, 0, 0, 0],
            &[0, 17, 209, 121, 46, 53, 172, 137, 0, 0],
            &[0, 142, 121, 0, 0, 0, 2, 206, 45, 0],
            &[0, 220, 24, 0, 0, 0, 0, 121, 121, 0],
            &[0, 239, 0, 0, 0, 0, 0, 91, 148, 0],
            &[0, 236, 3, 0, 0, 0, 0, 102, 140, 0],
            &[0, 188, 57, 0, 0, 0, 0, 160, 88, 0],
            &[0, 75, 195, 10, 0, 0, 58, 216, 8, 0],
            &[0, 0, 99, 211, 170, 186, 197, 39, 0, 0],
            &[0, 0, 0, 3, 32, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Œ' / 0x152
        'Œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 95, 125, 112, 108, 108, 108, 11],
            &[0, 28, 213, 123, 106, 226, 104, 104, 104, 10],
            &[0, 163, 95, 0, 26, 206, 0, 0, 0, 0],
            &[3, 237, 12, 0, 26, 206, 0, 0, 0, 0],
            &[32, 211, 0, 0, 26, 206, 0, 0, 0, 0],
            &[52, 193, 0, 0, 26, 227, 106, 106, 90, 0],
            &[55, 186, 0, 0, 26, 227, 106, 106, 90, 0],
            &[42, 198, 0, 0, 26, 206, 0, 0, 0, 0],
            &[11, 231, 0, 0, 26, 206, 0, 0, 0, 0],
            &[0, 206, 48, 0, 26, 206, 0, 0, 0, 0],
            &[0, 95, 188, 10, 26, 206, 0, 0, 0, 0],
            &[0, 0, 125, 223, 207, 248, 217, 217, 217, 22],
            &[0, 0, 0, 1, 18, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'œ' / 0x153
        'œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 27, 139, 142, 18, 8, 131, 140, 14, 0],
            &[2, 201, 69, 67, 161, 146, 74, 107, 148, 0],
            &[63, 153, 0, 0, 166, 170, 0, 2, 208, 0],
            &[109, 101, 0, 0, 137, 134, 0, 0, 203, 9],
            &[130, 85, 0, 0, 118, 217, 184, 184, 204, 15],
            &[126, 93, 0, 0, 128, 125, 0, 0, 0, 0],
            &[94, 123, 0, 0, 152, 157, 0, 0, 0, 0],
            &[27, 202, 3, 4, 183, 187, 13, 0, 5, 0],
            &[0, 119, 198, 193, 76, 65, 199, 161, 166, 0],
            &[0, 0, 20, 14, 0, 0, 10, 28, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŕ' / 0x154
        'Ŕ' => Some(&[
            &[0, 0, 0, 0, 121, 160, 0, 0, 0, 0],
            &[0, 0, 0, 67, 171, 4, 0, 0, 0, 0],
            &[0, 0, 0, 75, 3, 0, 0, 0, 0, 0],
            &[34, 108, 108, 104, 82, 24, 0, 0, 0, 0],
            &[81, 194, 100, 107, 137, 219, 118, 0, 0, 0],
            &[81, 155, 0, 0, 0, 18, 228, 34, 0, 0],
            &[81, 155, 0, 0, 0, 0, 169, 81, 0, 0],
            &[81, 155, 0, 0, 0, 0, 183, 66, 0, 0],
            &[81, 155, 0, 0, 0, 72, 212, 6, 0, 0],
            &[81, 228, 186, 188, 215, 151, 25, 0, 0, 0],
            &[81, 166, 26, 26, 163, 98, 0, 0, 0, 0],
            &[81, 155, 0, 0, 24, 219, 16, 0, 0, 0],
            &[81, 155, 0, 0, 0, 114, 147, 0, 0, 0],
            &[81, 155, 0, 0, 0, 4, 210, 48, 0, 0],
            &[81, 155, 0, 0, 0, 0, 67, 196, 1, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 42, 2, 0, 0, 0],
        ]),
        // letter: 'ŕ' / 0x155
        'ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 27, 48, 0, 0, 0],
            &[0, 0, 0, 0, 8, 203, 67, 0, 0, 0],
            &[0, 0, 0, 0, 155, 83, 0, 0, 0, 0],
            &[0, 0, 0, 12, 47, 0, 0, 0, 0, 0],
            &[56, 131, 131, 104, 6, 112, 160, 136, 6, 0],
            &[17, 50, 66, 214, 155, 110, 53, 81, 0, 0],
            &[0, 0, 0, 249, 109, 0, 0, 0, 0, 0],
            &[0, 0, 0, 255, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 241, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 0, 0, 0, 0, 0, 0],
            &[121, 184, 197, 252, 197, 184, 134, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŗ' / 0x156
        'Ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[34, 108, 108, 104, 82, 24, 0, 0, 0, 0],
            &[81, 194, 100, 107, 137, 219, 118, 0, 0, 0],
            &[81, 155, 0, 0, 0, 18, 228, 34, 0, 0],
            &[81, 155, 0, 0, 0, 0, 169, 81, 0, 0],
            &[81, 155, 0, 0, 0, 0, 183, 66, 0, 0],
            &[81, 155, 0, 0, 0, 72, 212, 6, 0, 0],
            &[81, 228, 186, 188, 215, 151, 25, 0, 0, 0],
            &[81, 166, 26, 26, 163, 98, 0, 0, 0, 0],
            &[81, 155, 0, 0, 24, 219, 16, 0, 0, 0],
            &[81, 155, 0, 0, 0, 114, 147, 0, 0, 0],
            &[81, 155, 0, 0, 0, 4, 210, 48, 0, 0],
            &[81, 155, 0, 0, 0, 0, 67, 196, 1, 0],
            &[0, 0, 0, 0, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 188, 0, 0, 0, 0, 0],
            &[0, 0, 0, 83, 1, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŗ' / 0x157
        'ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[56, 131, 131, 104, 6, 112, 160, 136, 6, 0],
            &[17, 50, 66, 214, 155, 110, 53, 81, 0, 0],
            &[0, 0, 0, 249, 109, 0, 0, 0, 0, 0],
            &[0, 0, 0, 255, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 241, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 0, 0, 0, 0, 0, 0],
            &[121, 184, 197, 252, 197, 184, 134, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 77, 6, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ř' / 0x158
        'Ř' => Some(&[
            &[0, 44, 172, 9, 61, 165, 4, 0, 0, 0],
            &[0, 0, 75, 188, 196, 11, 0, 0, 0, 0],
            &[0, 0, 0, 79, 36, 0, 0, 0, 0, 0],
            &[34, 108, 108, 104, 82, 24, 0, 0, 0, 0],
            &[81, 194, 100, 107, 137, 219, 118, 0, 0, 0],
            &[81, 155, 0, 0, 0, 18, 228, 34, 0, 0],
            &[81, 155, 0, 0, 0, 0, 169, 81, 0, 0],
            &[81, 155, 0, 0, 0, 0, 183, 66, 0, 0],
            &[81, 155, 0, 0, 0, 72, 212, 6, 0, 0],
            &[81, 228, 186, 188, 215, 151, 25, 0, 0, 0],
            &[81, 166, 26, 26, 163, 98, 0, 0, 0, 0],
            &[81, 155, 0, 0, 24, 219, 16, 0, 0, 0],
            &[81, 155, 0, 0, 0, 114, 147, 0, 0, 0],
            &[81, 155, 0, 0, 0, 4, 210, 48, 0, 0],
            &[81, 155, 0, 0, 0, 0, 67, 196, 1, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 6, 0, 0, 22, 6, 0, 0, 0],
        ]),
        // letter: 'ř' / 0x159
        'ř' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 46, 0, 0, 0, 47, 0, 0, 0],
            &[0, 0, 90, 138, 0, 147, 85, 0, 0, 0],
            &[0, 0, 0, 137, 197, 125, 0, 0, 0, 0],
            &[0, 0, 0, 6, 76, 3, 0, 0, 0, 0],
            &[56, 131, 131, 104, 6, 112, 160, 136, 6, 0],
            &[17, 50, 66, 214, 155, 110, 53, 81, 0, 0],
            &[0, 0, 0, 249, 109, 0, 0, 0, 0, 0],
            &[0, 0, 0, 255, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 241, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 0, 0, 0, 0, 0, 0],
            &[121, 184, 197, 252, 197, 184, 134, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ś' / 0x15a
        'Ś' => Some(&[
            &[0, 0, 0, 0, 0, 46, 215, 20, 0, 0],
            &[0, 0, 0, 0, 18, 190, 35, 0, 0, 0],
            &[0, 0, 0, 0, 50, 28, 0, 0, 0, 0],
            &[0, 0, 0, 71, 132, 141, 116, 49, 0, 0],
            &[0, 0, 173, 176, 94, 72, 103, 169, 8, 0],
            &[0, 62, 193, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 147, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 218, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 224, 122, 21, 0, 0, 0, 0],
            &[0, 0, 0, 21, 126, 222, 158, 28, 0, 0],
            &[0, 0, 0, 0, 0, 5, 112, 223, 18, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 93, 0],
            &[0, 0, 0, 0, 0, 0, 0, 149, 97, 0],
            &[0, 26, 3, 0, 0, 0, 45, 228, 27, 0],
            &[0, 105, 216, 196, 173, 197, 196, 61, 0, 0],
            &[0, 0, 0, 16, 33, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 32, 16, 0, 0],
        ]),
        // letter: 'ś' / 0x15b
        'ś' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 65, 6, 0, 0],
            &[0, 0, 0, 0, 0, 125, 153, 0, 0, 0],
            &[0, 0, 0, 0, 72, 163, 3, 0, 0, 0],
            &[0, 0, 0, 0, 57, 2, 0, 0, 0, 0],
            &[0, 0, 9, 107, 160, 161, 127, 40, 0, 0],
            &[0, 0, 175, 135, 46, 43, 85, 63, 0, 0],
            &[0, 0, 238, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 179, 132, 9, 0, 0, 0, 0, 0],
            &[0, 0, 5, 122, 212, 146, 39, 0, 0, 0],
            &[0, 0, 0, 0, 3, 79, 211, 85, 0, 0],
            &[0, 0, 0, 0, 0, 0, 60, 196, 0, 0],
            &[0, 12, 17, 0, 0, 0, 92, 170, 0, 0],
            &[0, 31, 204, 198, 171, 191, 196, 33, 0, 0],
            &[0, 0, 0, 12, 34, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => Some(&[
            &[0, 0, 0, 4, 187, 210, 27, 0, 0, 0],
            &[0, 0, 0, 142, 84, 40, 181, 10, 0, 0],
            &[0, 0, 16, 62, 0, 0, 37, 42, 0, 0],
            &[0, 0, 0, 71, 132, 141, 116, 49, 0, 0],
            &[0, 0, 173, 176, 94, 72, 103, 169, 8, 0],
            &[0, 62, 193, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 147, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 218, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 224, 122, 21, 0, 0, 0, 0],
            &[0, 0, 0, 21, 126, 222, 158, 28, 0, 0],
            &[0, 0, 0, 0, 0, 5, 112, 223, 18, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 93, 0],
            &[0, 0, 0, 0, 0, 0, 0, 149, 97, 0],
            &[0, 26, 3, 0, 0, 0, 45, 228, 27, 0],
            &[0, 105, 216, 196, 173, 197, 196, 61, 0, 0],
            &[0, 0, 0, 16, 33, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 24, 0, 0, 0, 0],
        ]),
        // letter: 'ŝ' / 0x15d
        'ŝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 24, 0, 0, 0, 0],
            &[0, 0, 0, 40, 204, 194, 7, 0, 0, 0],
            &[0, 0, 18, 180, 25, 73, 155, 0, 0, 0],
            &[0, 0, 36, 23, 0, 0, 46, 13, 0, 0],
            &[0, 0, 9, 107, 160, 161, 127, 40, 0, 0],
            &[0, 0, 175, 135, 46, 43, 85, 63, 0, 0],
            &[0, 0, 238, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 179, 132, 9, 0, 0, 0, 0, 0],
            &[0, 0, 5, 122, 212, 146, 39, 0, 0, 0],
            &[0, 0, 0, 0, 3, 79, 211, 85, 0, 0],
            &[0, 0, 0, 0, 0, 0, 60, 196, 0, 0],
            &[0, 12, 17, 0, 0, 0, 92, 170, 0, 0],
            &[0, 31, 204, 198, 171, 191, 196, 33, 0, 0],
            &[0, 0, 0, 12, 34, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ş' / 0x15e
        'Ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 71, 132, 141, 116, 49, 0, 0],
            &[0, 0, 173, 176, 94, 72, 103, 169, 8, 0],
            &[0, 62, 193, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 147, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 218, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 224, 122, 21, 0, 0, 0, 0],
            &[0, 0, 0, 21, 126, 222, 158, 28, 0, 0],
            &[0, 0, 0, 0, 0, 5, 112, 223, 18, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 93, 0],
            &[0, 0, 0, 0, 0, 0, 0, 149, 97, 0],
            &[0, 26, 3, 0, 0, 0, 45, 228, 27, 0],
            &[0, 105, 216, 196, 180, 199, 196, 61, 0, 0],
            &[0, 0, 0, 16, 202, 17, 0, 0, 0, 0],
            &[0, 0, 0, 9, 130, 175, 4, 0, 0, 0],
            &[0, 0, 0, 129, 175, 104, 0, 0, 0, 0],
        ]),
        // letter: 'ş' / 0x15f
        'ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 107, 160, 161, 127, 40, 0, 0],
            &[0, 0, 175, 135, 46, 43, 85, 63, 0, 0],
            &[0, 0, 238, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 179, 132, 9, 0, 0, 0, 0, 0],
            &[0, 0, 5, 122, 212, 146, 39, 0, 0, 0],
            &[0, 0, 0, 0, 3, 79, 211, 85, 0, 0],
            &[0, 0, 0, 0, 0, 0, 60, 196, 0, 0],
            &[0, 12, 17, 0, 0, 0, 92, 170, 0, 0],
            &[0, 31, 204, 198, 176, 194, 196, 33, 0, 0],
            &[0, 0, 0, 12, 199, 31, 0, 0, 0, 0],
            &[0, 0, 0, 4, 127, 177, 9, 0, 0, 0],
            &[0, 0, 0, 121, 174, 113, 0, 0, 0, 0],
        ]),
        // letter: 'Š' / 0x160
        'Š' => Some(&[
            &[0, 0, 8, 172, 45, 17, 179, 33, 0, 0],
            &[0, 0, 0, 21, 201, 194, 54, 0, 0, 0],
            &[0, 0, 0, 0, 47, 68, 0, 0, 0, 0],
            &[0, 0, 0, 71, 132, 141, 116, 49, 0, 0],
            &[0, 0, 173, 176, 94, 72, 103, 169, 8, 0],
            &[0, 62, 193, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 147, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 218, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 224, 122, 21, 0, 0, 0, 0],
            &[0, 0, 0, 21, 126, 222, 158, 28, 0, 0],
            &[0, 0, 0, 0, 0, 5, 112, 223, 18, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 93, 0],
            &[0, 0, 0, 0, 0, 0, 0, 149, 97, 0],
            &[0, 26, 3, 0, 0, 0, 45, 228, 27, 0],
            &[0, 105, 216, 196, 173, 197, 196, 61, 0, 0],
            &[0, 0, 0, 16, 33, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 18, 0, 0, 10, 18, 0, 0],
        ]),
        // letter: 'š' / 0x161
        'š' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 30, 16, 0, 0, 35, 12, 0, 0],
            &[0, 0, 26, 181, 20, 68, 162, 2, 0, 0],
            &[0, 0, 0, 51, 203, 196, 9, 0, 0, 0],
            &[0, 0, 0, 0, 55, 29, 0, 0, 0, 0],
            &[0, 0, 9, 107, 160, 161, 127, 40, 0, 0],
            &[0, 0, 175, 135, 46, 43, 85, 63, 0, 0],
            &[0, 0, 238, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 179, 132, 9, 0, 0, 0, 0, 0],
            &[0, 0, 5, 122, 212, 146, 39, 0, 0, 0],
            &[0, 0, 0, 0, 3, 79, 211, 85, 0, 0],
            &[0, 0, 0, 0, 0, 0, 60, 196, 0, 0],
            &[0, 12, 17, 0, 0, 0, 92, 170, 0, 0],
            &[0, 31, 204, 198, 171, 191, 196, 33, 0, 0],
            &[0, 0, 0, 12, 34, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ţ' / 0x162
        'Ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[25, 108, 108, 108, 108, 108, 108, 108, 92, 0],
            &[24, 104, 104, 104, 204, 144, 104, 104, 88, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 176, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 168, 7, 0, 0, 0, 0],
            &[0, 0, 0, 9, 130, 175, 4, 0, 0, 0],
            &[0, 0, 0, 129, 175, 104, 0, 0, 0, 0],
        ]),
        // letter: 'ţ' / 0x163
        'ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 151, 0, 0, 0, 0, 0],
            &[0, 92, 120, 174, 204, 131, 131, 131, 0, 0],
            &[0, 64, 73, 134, 181, 73, 73, 73, 0, 0],
            &[0, 0, 0, 85, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 79, 166, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 224, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 221, 176, 185, 9, 0],
            &[0, 0, 0, 0, 0, 129, 93, 13, 0, 0],
            &[0, 0, 0, 0, 0, 95, 180, 43, 0, 0],
            &[0, 0, 0, 0, 81, 169, 149, 9, 0, 0],
        ]),
        // letter: 'Ť' / 0x164
        'Ť' => Some(&[
            &[0, 0, 44, 172, 9, 61, 165, 4, 0, 0],
            &[0, 0, 0, 75, 188, 196, 11, 0, 0, 0],
            &[0, 0, 0, 0, 79, 36, 0, 0, 0, 0],
            &[25, 108, 108, 108, 108, 108, 108, 108, 92, 0],
            &[24, 104, 104, 104, 204, 144, 104, 104, 88, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 6, 0, 0, 22, 6, 0, 0],
        ]),
        // letter: 'ť' / 0x165
        'ť' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 195, 0, 0],
            &[0, 0, 0, 34, 106, 0, 97, 88, 0, 0],
            &[0, 0, 0, 68, 151, 0, 43, 1, 0, 0],
            &[0, 92, 120, 174, 204, 131, 131, 131, 0, 0],
            &[0, 64, 73, 134, 181, 73, 73, 73, 0, 0],
            &[0, 0, 0, 85, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 79, 166, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 224, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 219, 170, 185, 9, 0],
            &[0, 0, 0, 0, 0, 11, 35, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŧ' / 0x166
        'Ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[25, 108, 108, 108, 108, 108, 108, 108, 92, 0],
            &[24, 104, 104, 104, 204, 144, 104, 104, 88, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 15, 61, 61, 231, 128, 61, 53, 0, 0],
            &[0, 36, 143, 143, 255, 210, 143, 124, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŧ' / 0x167
        'ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 151, 0, 0, 0, 0, 0],
            &[0, 92, 120, 174, 204, 131, 131, 131, 0, 0],
            &[0, 64, 73, 134, 181, 73, 73, 73, 0, 0],
            &[0, 0, 0, 85, 151, 0, 0, 0, 0, 0],
            &[0, 55, 63, 149, 214, 63, 63, 63, 7, 0],
            &[0, 106, 120, 206, 255, 120, 120, 120, 13, 0],
            &[0, 0, 0, 85, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 79, 166, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 224, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 219, 170, 185, 9, 0],
            &[0, 0, 0, 0, 0, 11, 35, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ũ' / 0x168
        'Ũ' => Some(&[
            &[0, 0, 23, 116, 58, 0, 21, 66, 0, 0],
            &[0, 0, 170, 85, 158, 161, 180, 57, 0, 0],
            &[0, 0, 48, 0, 0, 44, 38, 0, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 69, 31, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 161, 75, 0, 0, 0, 0, 174, 63, 0],
            &[0, 127, 118, 0, 0, 0, 0, 218, 26, 0],
            &[0, 42, 220, 21, 0, 0, 94, 189, 0, 0],
            &[0, 0, 96, 221, 186, 204, 192, 26, 0, 0],
            &[0, 0, 0, 4, 31, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ũ' / 0x169
        'ũ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 139, 79, 0, 28, 76, 0, 0],
            &[0, 0, 174, 63, 137, 181, 191, 47, 0, 0],
            &[0, 0, 33, 0, 0, 24, 21, 0, 0, 0],
            &[0, 77, 44, 0, 0, 0, 0, 104, 16, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 86, 0, 0, 0, 0, 212, 32, 0],
            &[0, 134, 105, 0, 0, 0, 8, 245, 32, 0],
            &[0, 79, 190, 4, 0, 0, 126, 229, 32, 0],
            &[0, 0, 150, 214, 168, 188, 92, 170, 32, 0],
            &[0, 0, 0, 9, 29, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ū' / 0x16a
        'Ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 72, 178, 178, 178, 178, 1, 0, 0],
            &[0, 0, 7, 18, 18, 18, 18, 0, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 69, 31, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 161, 75, 0, 0, 0, 0, 174, 63, 0],
            &[0, 127, 118, 0, 0, 0, 0, 218, 26, 0],
            &[0, 42, 220, 21, 0, 0, 94, 189, 0, 0],
            &[0, 0, 96, 221, 186, 204, 192, 26, 0, 0],
            &[0, 0, 0, 4, 31, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ū' / 0x16b
        'ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 80, 196, 196, 196, 196, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 77, 44, 0, 0, 0, 0, 104, 16, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 86, 0, 0, 0, 0, 212, 32, 0],
            &[0, 134, 105, 0, 0, 0, 8, 245, 32, 0],
            &[0, 79, 190, 4, 0, 0, 126, 229, 32, 0],
            &[0, 0, 150, 214, 168, 188, 92, 170, 32, 0],
            &[0, 0, 0, 9, 29, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => Some(&[
            &[0, 0, 57, 64, 0, 0, 116, 0, 0, 0],
            &[0, 0, 18, 194, 113, 152, 137, 0, 0, 0],
            &[0, 0, 0, 11, 67, 46, 0, 0, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 69, 31, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 161, 75, 0, 0, 0, 0, 174, 63, 0],
            &[0, 127, 118, 0, 0, 0, 0, 218, 26, 0],
            &[0, 42, 220, 21, 0, 0, 94, 189, 0, 0],
            &[0, 0, 96, 221, 186, 204, 192, 26, 0, 0],
            &[0, 0, 0, 4, 31, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŭ' / 0x16d
        'ŭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 76, 0, 0, 134, 0, 0, 0],
            &[0, 0, 13, 189, 136, 172, 120, 0, 0, 0],
            &[0, 0, 0, 4, 44, 26, 0, 0, 0, 0],
            &[0, 77, 44, 0, 0, 0, 0, 104, 16, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 86, 0, 0, 0, 0, 212, 32, 0],
            &[0, 134, 105, 0, 0, 0, 8, 245, 32, 0],
            &[0, 79, 190, 4, 0, 0, 126, 229, 32, 0],
            &[0, 0, 150, 214, 168, 188, 92, 170, 32, 0],
            &[0, 0, 0, 9, 29, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ů' / 0x16e
        'Ů' => Some(&[
            &[0, 0, 0, 178, 0, 68, 109, 0, 0, 0],
            &[0, 0, 0, 160, 97, 163, 61, 0, 0, 0],
            &[0, 0, 0, 5, 73, 42, 0, 0, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 69, 31, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 161, 75, 0, 0, 0, 0, 174, 63, 0],
            &[0, 127, 118, 0, 0, 0, 0, 218, 26, 0],
            &[0, 42, 220, 21, 0, 0, 94, 189, 0, 0],
            &[0, 0, 96, 221, 186, 204, 192, 26, 0, 0],
            &[0, 0, 0, 4, 31, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 86, 167, 166, 20, 0, 0, 0],
        ]),
        // letter: 'ů' / 0x16f
        'ů' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 103, 166, 175, 28, 0, 0, 0],
            &[0, 0, 0, 177, 0, 66, 110, 0, 0, 0],
            &[0, 0, 0, 148, 119, 173, 52, 0, 0, 0],
            &[0, 0, 0, 1, 50, 25, 0, 0, 0, 0],
            &[0, 77, 44, 0, 0, 0, 0, 104, 16, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 86, 0, 0, 0, 0, 212, 32, 0],
            &[0, 134, 105, 0, 0, 0, 8, 245, 32, 0],
            &[0, 79, 190, 4, 0, 0, 126, 229, 32, 0],
            &[0, 0, 150, 214, 168, 188, 92, 170, 32, 0],
            &[0, 0, 0, 9, 29, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ű' / 0x170
        'Ű' => Some(&[
            &[0, 0, 0, 64, 191, 2, 182, 74, 0, 0],
            &[0, 0, 11, 186, 23, 94, 127, 0, 0, 0],
            &[0, 0, 38, 32, 0, 70, 0, 0, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 69, 31, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 161, 75, 0, 0, 0, 0, 174, 63, 0],
            &[0, 127, 118, 0, 0, 0, 0, 218, 26, 0],
            &[0, 42, 220, 21, 0, 0, 94, 189, 0, 0],
            &[0, 0, 96, 221, 186, 204, 192, 26, 0, 0],
            &[0, 0, 0, 4, 31, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 5, 16, 28, 0, 0],
        ]),
        // letter: 'ű' / 0x171
        'ű' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 61, 7, 27, 42, 0, 0],
            &[0, 0, 0, 78, 177, 1, 192, 60, 0, 0],
            &[0, 0, 17, 184, 15, 109, 107, 0, 0, 0],
            &[0, 0, 31, 22, 0, 53, 0, 0, 0, 0],
            &[0, 77, 44, 0, 0, 0, 0, 104, 16, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 86, 0, 0, 0, 0, 212, 32, 0],
            &[0, 134, 105, 0, 0, 0, 8, 245, 32, 0],
            &[0, 79, 190, 4, 0, 0, 126, 229, 32, 0],
            &[0, 0, 150, 214, 168, 188, 92, 170, 32, 0],
            &[0, 0, 0, 9, 29, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ų' / 0x172
        'Ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 27, 0, 0, 0, 0, 69, 31, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 171, 65, 0, 0, 0, 0, 163, 73, 0],
            &[0, 161, 75, 0, 0, 0, 0, 174, 63, 0],
            &[0, 127, 118, 0, 0, 0, 0, 218, 26, 0],
            &[0, 42, 220, 21, 0, 0, 94, 189, 0, 0],
            &[0, 0, 96, 221, 186, 242, 234, 26, 0, 0],
            &[0, 0, 0, 4, 114, 150, 4, 0, 0, 0],
            &[0, 0, 0, 16, 193, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 64, 117, 20, 0, 0, 0],
        ]),
        // letter: 'ų' / 0x173
        'ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 77, 44, 0, 0, 0, 0, 104, 16, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 85, 0, 0, 0, 0, 204, 32, 0],
            &[0, 151, 86, 0, 0, 0, 0, 212, 32, 0],
            &[0, 134, 105, 0, 0, 0, 8, 245, 32, 0],
            &[0, 79, 190, 4, 0, 0, 126, 229, 32, 0],
            &[0, 0, 150, 214, 168, 188, 92, 190, 33, 0],
            &[0, 0, 0, 9, 29, 2, 59, 143, 1, 0],
            &[0, 0, 0, 0, 0, 0, 184, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 44, 117, 41, 0],
        ]),
        // letter: 'Ŵ' / 0x174
        'Ŵ' => Some(&[
            &[0, 0, 0, 37, 212, 179, 1, 0, 0, 0],
            &[0, 0, 15, 182, 29, 101, 130, 0, 0, 0],
            &[0, 0, 48, 30, 0, 0, 68, 10, 0, 0],
            &[48, 47, 0, 0, 0, 0, 0, 0, 88, 6],
            &[88, 135, 0, 0, 0, 0, 0, 0, 224, 0],
            &[53, 169, 0, 0, 0, 0, 0, 12, 212, 0],
            &[19, 203, 0, 0, 92, 42, 0, 46, 178, 0],
            &[0, 221, 0, 0, 205, 127, 0, 81, 144, 0],
            &[0, 205, 15, 28, 154, 174, 0, 115, 110, 0],
            &[0, 170, 49, 87, 97, 177, 3, 149, 76, 0],
            &[0, 135, 82, 146, 39, 132, 48, 181, 42, 0],
            &[0, 101, 113, 185, 0, 74, 108, 205, 9, 0],
            &[0, 66, 147, 174, 0, 17, 167, 203, 0, 0],
            &[0, 31, 217, 121, 0, 0, 177, 189, 0, 0],
            &[0, 2, 249, 63, 0, 0, 154, 162, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 12, 0, 0, 0, 0],
        ]),
        // letter: 'ŵ' / 0x175
        'ŵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 25, 0, 0, 0, 0],
            &[0, 0, 0, 37, 203, 196, 8, 0, 0, 0],
            &[0, 0, 17, 180, 27, 70, 157, 1, 0, 0],
            &[0, 0, 35, 25, 0, 0, 45, 15, 0, 0],
            &[80, 33, 0, 0, 0, 0, 0, 0, 81, 30],
            &[116, 105, 0, 0, 0, 0, 0, 0, 196, 21],
            &[64, 158, 0, 0, 191, 117, 0, 3, 216, 0],
            &[13, 210, 0, 23, 184, 191, 0, 43, 176, 0],
            &[0, 212, 12, 99, 109, 188, 21, 94, 126, 0],
            &[0, 163, 62, 176, 35, 117, 95, 145, 76, 0],
            &[0, 110, 112, 203, 0, 41, 168, 187, 26, 0],
            &[0, 58, 194, 140, 0, 0, 198, 194, 0, 0],
            &[0, 9, 250, 64, 0, 0, 146, 180, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŷ' / 0x176
        'Ŷ' => Some(&[
            &[0, 0, 0, 37, 212, 179, 1, 0, 0, 0],
            &[0, 0, 15, 182, 29, 101, 130, 0, 0, 0],
            &[0, 0, 48, 30, 0, 0, 68, 10, 0, 0],
            &[21, 88, 0, 0, 0, 0, 0, 20, 88, 0],
            &[1, 208, 46, 0, 0, 0, 0, 137, 114, 0],
            &[0, 77, 172, 0, 0, 0, 20, 216, 8, 0],
            &[0, 0, 197, 46, 0, 0, 134, 103, 0, 0],
            &[0, 0, 65, 172, 0, 18, 207, 4, 0, 0],
            &[0, 0, 0, 185, 46, 132, 91, 0, 0, 0],
            &[0, 0, 0, 54, 184, 197, 2, 0, 0, 0],
            &[0, 0, 0, 0, 185, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 12, 0, 0, 0, 0],
        ]),
        // letter: 'ŷ' / 0x177
        'ŷ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 21, 0, 0, 0, 0],
            &[0, 0, 0, 48, 204, 188, 4, 0, 0, 0],
            &[0, 0, 23, 180, 20, 83, 146, 0, 0, 0],
            &[0, 0, 40, 20, 0, 0, 50, 10, 0, 0],
            &[0, 106, 15, 0, 0, 0, 0, 64, 60, 0],
            &[0, 130, 104, 0, 0, 0, 0, 195, 45, 0],
            &[0, 31, 201, 0, 0, 0, 32, 203, 0, 0],
            &[0, 0, 182, 47, 0, 0, 123, 108, 0, 0],
            &[0, 0, 81, 146, 0, 0, 208, 18, 0, 0],
            &[0, 0, 4, 213, 8, 51, 171, 0, 0, 0],
            &[0, 0, 0, 133, 88, 141, 75, 0, 0, 0],
            &[0, 0, 0, 33, 178, 202, 3, 0, 0, 0],
            &[0, 0, 0, 0, 185, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 186, 41, 0, 0, 0, 0],
            &[0, 0, 0, 33, 196, 0, 0, 0, 0, 0],
            &[0, 156, 209, 124, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ÿ' / 0x178
        'Ÿ' => Some(&[
            &[0, 0, 43, 71, 0, 2, 108, 3, 0, 0],
            &[0, 0, 98, 151, 0, 14, 218, 16, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[21, 88, 0, 0, 0, 0, 0, 20, 88, 0],
            &[1, 208, 46, 0, 0, 0, 0, 137, 114, 0],
            &[0, 77, 172, 0, 0, 0, 20, 216, 8, 0],
            &[0, 0, 197, 46, 0, 0, 134, 103, 0, 0],
            &[0, 0, 65, 172, 0, 18, 207, 4, 0, 0],
            &[0, 0, 0, 185, 46, 132, 91, 0, 0, 0],
            &[0, 0, 0, 54, 184, 197, 2, 0, 0, 0],
            &[0, 0, 0, 0, 185, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ź' / 0x179
        'Ź' => Some(&[
            &[0, 0, 0, 0, 0, 62, 207, 12, 0, 0],
            &[0, 0, 0, 0, 27, 191, 25, 0, 0, 0],
            &[0, 0, 0, 0, 58, 20, 0, 0, 0, 0],
            &[0, 0, 101, 108, 108, 108, 108, 98, 0, 0],
            &[0, 0, 93, 100, 100, 100, 141, 196, 0, 0],
            &[0, 0, 0, 0, 0, 0, 176, 67, 0, 0],
            &[0, 0, 0, 0, 0, 55, 188, 0, 0, 0],
            &[0, 0, 0, 0, 0, 189, 55, 0, 0, 0],
            &[0, 0, 0, 0, 69, 176, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 44, 0, 0, 0, 0],
            &[0, 0, 0, 83, 163, 0, 0, 0, 0, 0],
            &[0, 0, 3, 209, 34, 0, 0, 0, 0, 0],
            &[0, 0, 97, 150, 0, 0, 0, 0, 0, 0],
            &[0, 7, 215, 25, 0, 0, 0, 0, 0, 0],
            &[0, 76, 246, 217, 217, 217, 217, 217, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 35, 13, 0, 0],
        ]),
        // letter: 'ź' / 0x17a
        'ź' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 13, 0, 0],
            &[0, 0, 0, 0, 0, 96, 179, 2, 0, 0],
            &[0, 0, 0, 0, 50, 179, 9, 0, 0, 0],
            &[0, 0, 0, 0, 52, 7, 0, 0, 0, 0],
            &[0, 2, 131, 131, 131, 131, 131, 118, 0, 0],
            &[0, 0, 61, 61, 61, 61, 150, 167, 0, 0],
            &[0, 0, 0, 0, 0, 37, 211, 15, 0, 0],
            &[0, 0, 0, 0, 5, 199, 57, 0, 0, 0],
            &[0, 0, 0, 0, 139, 121, 0, 0, 0, 0],
            &[0, 0, 0, 70, 186, 2, 0, 0, 0, 0],
            &[0, 0, 22, 210, 25, 0, 0, 0, 0, 0],
            &[0, 1, 181, 73, 0, 0, 0, 0, 0, 0],
            &[0, 76, 248, 208, 208, 208, 208, 195, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ż' / 0x17b
        'Ż' => Some(&[
            &[0, 0, 0, 0, 98, 114, 0, 0, 0, 0],
            &[0, 0, 0, 0, 110, 127, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 101, 108, 108, 108, 108, 98, 0, 0],
            &[0, 0, 93, 100, 100, 100, 141, 196, 0, 0],
            &[0, 0, 0, 0, 0, 0, 176, 67, 0, 0],
            &[0, 0, 0, 0, 0, 55, 188, 0, 0, 0],
            &[0, 0, 0, 0, 0, 189, 55, 0, 0, 0],
            &[0, 0, 0, 0, 69, 176, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 44, 0, 0, 0, 0],
            &[0, 0, 0, 83, 163, 0, 0, 0, 0, 0],
            &[0, 0, 3, 209, 34, 0, 0, 0, 0, 0],
            &[0, 0, 97, 150, 0, 0, 0, 0, 0, 0],
            &[0, 7, 215, 25, 0, 0, 0, 0, 0, 0],
            &[0, 76, 246, 217, 217, 217, 217, 217, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ż' / 0x17c
        'ż' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 150, 131, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 131, 131, 131, 131, 131, 118, 0, 0],
            &[0, 0, 61, 61, 61, 61, 150, 167, 0, 0],
            &[0, 0, 0, 0, 0, 37, 211, 15, 0, 0],
            &[0, 0, 0, 0, 5, 199, 57, 0, 0, 0],
            &[0, 0, 0, 0, 139, 121, 0, 0, 0, 0],
            &[0, 0, 0, 70, 186, 2, 0, 0, 0, 0],
            &[0, 0, 22, 210, 25, 0, 0, 0, 0, 0],
            &[0, 1, 181, 73, 0, 0, 0, 0, 0, 0],
            &[0, 76, 248, 208, 208, 208, 208, 195, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ž' / 0x17d
        'Ž' => Some(&[
            &[0, 0, 14, 177, 33, 26, 180, 23, 0, 0],
            &[0, 0, 0, 31, 200, 198, 40, 0, 0, 0],
            &[0, 0, 0, 0, 55, 60, 0, 0, 0, 0],
            &[0, 0, 101, 108, 108, 108, 108, 98, 0, 0],
            &[0, 0, 93, 100, 100, 100, 141, 196, 0, 0],
            &[0, 0, 0, 0, 0, 0, 176, 67, 0, 0],
            &[0, 0, 0, 0, 0, 55, 188, 0, 0, 0],
            &[0, 0, 0, 0, 0, 189, 55, 0, 0, 0],
            &[0, 0, 0, 0, 69, 176, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 44, 0, 0, 0, 0],
            &[0, 0, 0, 83, 163, 0, 0, 0, 0, 0],
            &[0, 0, 3, 209, 34, 0, 0, 0, 0, 0],
            &[0, 0, 97, 150, 0, 0, 0, 0, 0, 0],
            &[0, 7, 215, 25, 0, 0, 0, 0, 0, 0],
            &[0, 76, 246, 217, 217, 217, 217, 217, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 15, 0, 0, 13, 15, 0, 0],
        ]),
        // letter: 'ž' / 0x17e
        'ž' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 23, 0, 0, 28, 19, 0, 0],
            &[0, 0, 14, 180, 34, 48, 175, 8, 0, 0],
            &[0, 0, 0, 32, 203, 203, 20, 0, 0, 0],
            &[0, 0, 0, 0, 47, 38, 0, 0, 0, 0],
            &[0, 2, 131, 131, 131, 131, 131, 118, 0, 0],
            &[0, 0, 61, 61, 61, 61, 150, 167, 0, 0],
            &[0, 0, 0, 0, 0, 37, 211, 15, 0, 0],
            &[0, 0, 0, 0, 5, 199, 57, 0, 0, 0],
            &[0, 0, 0, 0, 139, 121, 0, 0, 0, 0],
            &[0, 0, 0, 70, 186, 2, 0, 0, 0, 0],
            &[0, 0, 22, 210, 25, 0, 0, 0, 0, 0],
            &[0, 1, 181, 73, 0, 0, 0, 0, 0, 0],
            &[0, 76, 248, 208, 208, 208, 208, 195, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ſ' / 0x17f
        'ſ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 50, 50, 8, 0, 0],
            &[0, 0, 0, 66, 217, 156, 152, 65, 0, 0],
            &[0, 0, 1, 221, 40, 0, 0, 0, 0, 0],
            &[0, 0, 26, 214, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 196, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 196, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 196, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 196, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 196, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 196, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 196, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 196, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 196, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        _ => None,
    }
}
