//! Module for letters with the font weight bold and size 32.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 32;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 19;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight bold and font size 27px
#[inline]
pub const fn get_char(c: char) -> Option<&'static [&'static [u8]]> {
    match c {
        // letter: ' ' / 0x20
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '!' / 0x21
        '!' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 11, 71, 71, 71, 71, 27, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 34, 255, 255, 255, 255, 92, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 22, 255, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 11, 255, 255, 255, 255, 69, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 1, 254, 255, 255, 255, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 245, 255, 255, 255, 47, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 233, 255, 255, 255, 35, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 24, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 211, 255, 255, 255, 13, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 199, 255, 255, 255, 2, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 188, 255, 255, 246, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 235, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 166, 255, 255, 224, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 94, 153, 153, 129, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 20, 101, 107, 33, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 15, 229, 255, 255, 245, 38, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 75, 255, 255, 255, 255, 119, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 65, 255, 255, 255, 255, 108, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 3, 186, 255, 255, 212, 15, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 1, 45, 52, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '"' / 0x22
        '"' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 34, 71, 71, 71, 17, 3, 71, 71, 71, 48, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 110, 255, 255, 255, 52, 3, 254, 255, 255, 161, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 91, 255, 255, 255, 33, 0, 238, 255, 255, 143, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 73, 255, 255, 255, 14, 0, 219, 255, 255, 124, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 54, 255, 255, 251, 0, 0, 200, 255, 255, 105, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 35, 255, 255, 233, 0, 0, 181, 255, 255, 86, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 16, 255, 255, 214, 0, 0, 163, 255, 255, 67, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 175, 176, 136, 0, 0, 101, 176, 176, 35, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '#' / 0x23
        '#' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 17, 71, 71, 34, 0, 13, 71, 71, 35, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 90, 255, 255, 95, 0, 76, 255, 255, 102, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 255, 53, 0, 118, 255, 255, 60, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 175, 255, 254, 11, 0, 159, 255, 255, 19, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 218, 255, 223, 0, 0, 201, 255, 233, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 8, 253, 255, 180, 0, 0, 242, 255, 192, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 199, 245, 245, 246, 255, 255, 251, 245, 245, 255, 255, 252, 245, 245, 195, 0,
                0,
            ],
            &[
                0, 0, 207, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 203, 0,
                0,
            ],
            &[
                0, 0, 148, 182, 182, 222, 255, 255, 196, 182, 218, 255, 255, 197, 182, 182, 145, 0,
                0,
            ],
            &[
                0, 0, 0, 0, 0, 166, 255, 255, 19, 0, 153, 255, 255, 25, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 208, 255, 233, 0, 0, 196, 255, 238, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 7, 11, 11, 14, 248, 255, 194, 11, 11, 238, 255, 199, 11, 11, 11, 0, 0, 0,
            ],
            &[
                0, 152, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 3, 0,
                0,
            ],
            &[
                0, 152, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 3, 0,
                0,
            ],
            &[
                0, 91, 153, 153, 206, 255, 255, 171, 153, 204, 255, 255, 179, 153, 153, 153, 1, 0,
                0,
            ],
            &[
                0, 0, 0, 0, 161, 255, 255, 18, 0, 157, 255, 255, 35, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 200, 255, 234, 0, 0, 199, 255, 246, 2, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 240, 255, 194, 0, 0, 241, 255, 206, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 24, 255, 255, 154, 0, 28, 255, 255, 164, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 63, 255, 255, 115, 0, 70, 255, 255, 122, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '$' / 0x24
        '$' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 89, 126, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 11, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 51, 121, 226, 255, 156, 117, 78, 14, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 30, 189, 255, 255, 255, 255, 255, 255, 255, 248, 130, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 15, 218, 255, 255, 255, 255, 255, 255, 255, 255, 255, 120, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 119, 255, 255, 255, 179, 216, 255, 127, 165, 232, 254, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 183, 255, 255, 197, 0, 181, 255, 11, 0, 3, 57, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 199, 255, 255, 173, 0, 181, 255, 11, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 163, 255, 255, 252, 107, 184, 255, 11, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 59, 253, 255, 255, 255, 255, 255, 28, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 103, 251, 255, 255, 255, 255, 247, 157, 34, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 50, 191, 255, 255, 255, 255, 255, 247, 103, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 52, 217, 255, 255, 255, 255, 255, 84, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 137, 251, 255, 255, 207, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 11, 130, 255, 255, 250, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 36, 0, 0, 0, 0, 181, 255, 11, 101, 255, 255, 239, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 207, 194, 100, 29, 0, 181, 255, 70, 215, 255, 255, 192, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 208, 255, 255, 255, 237, 244, 255, 255, 255, 255, 255, 66, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 207, 255, 255, 255, 255, 255, 255, 255, 255, 255, 130, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 28, 117, 183, 227, 252, 255, 255, 232, 166, 58, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 1, 184, 255, 11, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 11, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 102, 144, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '%' / 0x25
        '%' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 56, 108, 95, 35, 0, 0, 0, 0, 0, 44, 71, 71, 18, 0, 0, 0,
            ],
            &[
                0, 3, 161, 255, 255, 255, 249, 106, 0, 0, 0, 10, 232, 255, 231, 9, 0, 0, 0,
            ],
            &[
                0, 98, 255, 255, 231, 252, 255, 251, 35, 0, 0, 117, 255, 255, 110, 0, 0, 0, 0,
            ],
            &[
                0, 199, 255, 228, 9, 70, 255, 255, 129, 0, 12, 235, 255, 227, 7, 0, 0, 0, 0,
            ],
            &[
                0, 234, 255, 176, 0, 0, 248, 255, 163, 0, 123, 255, 255, 105, 0, 0, 0, 0, 0,
            ],
            &[
                0, 242, 255, 181, 0, 0, 244, 255, 172, 15, 238, 255, 224, 6, 0, 0, 0, 0, 0,
            ],
            &[
                0, 208, 255, 226, 1, 39, 255, 255, 138, 129, 255, 255, 100, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 125, 255, 255, 196, 238, 255, 255, 76, 241, 255, 220, 4, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 11, 201, 255, 255, 255, 255, 147, 135, 255, 255, 95, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 5, 99, 151, 137, 73, 21, 244, 255, 217, 3, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 140, 255, 255, 90, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 25, 246, 255, 213, 3, 106, 193, 219, 177, 58, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 146, 255, 255, 85, 143, 255, 255, 255, 255, 251, 70, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 28, 248, 255, 209, 35, 251, 255, 203, 135, 247, 255, 205, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 152, 255, 255, 80, 92, 255, 255, 66, 0, 151, 255, 255, 11, 0,
            ],
            &[
                0, 0, 0, 0, 32, 250, 255, 204, 0, 123, 255, 255, 37, 0, 119, 255, 255, 41, 0,
            ],
            &[
                0, 0, 0, 0, 158, 255, 255, 75, 0, 106, 255, 255, 65, 0, 132, 255, 255, 25, 0,
            ],
            &[
                0, 0, 0, 37, 251, 255, 200, 0, 0, 64, 255, 255, 148, 49, 219, 255, 238, 0, 0,
            ],
            &[
                0, 0, 0, 163, 255, 255, 70, 0, 0, 0, 200, 255, 255, 255, 255, 255, 123, 0, 0,
            ],
            &[
                0, 0, 41, 253, 255, 195, 0, 0, 0, 0, 28, 193, 255, 255, 253, 142, 6, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 23, 48, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '&' / 0x26
        '&' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 13, 80, 120, 102, 55, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 61, 226, 255, 255, 255, 255, 177, 14, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 14, 230, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 103, 255, 255, 240, 71, 118, 255, 255, 250, 4, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 144, 255, 255, 175, 0, 6, 250, 255, 255, 39, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 156, 255, 255, 182, 0, 36, 255, 255, 255, 28, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 121, 255, 255, 233, 6, 160, 255, 255, 210, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 41, 253, 255, 255, 252, 255, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 160, 255, 255, 255, 255, 255, 153, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 72, 255, 255, 255, 255, 182, 4, 0, 20, 45, 45, 45, 1, 0, 0,
            ],
            &[
                0, 0, 0, 50, 238, 255, 255, 255, 255, 156, 0, 0, 146, 255, 255, 235, 0, 0, 0,
            ],
            &[
                0, 0, 19, 227, 255, 255, 255, 255, 255, 255, 89, 0, 207, 255, 255, 184, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 115, 215, 255, 255, 243, 77, 255, 255, 255, 117, 0, 0, 0,
            ],
            &[
                0, 3, 237, 255, 255, 169, 0, 48, 249, 255, 255, 255, 255, 255, 254, 32, 0, 0, 0,
            ],
            &[
                0, 32, 255, 255, 255, 93, 0, 0, 113, 255, 255, 255, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 39, 255, 255, 255, 95, 0, 0, 1, 186, 255, 255, 255, 255, 55, 0, 0, 0, 0,
            ],
            &[
                0, 12, 252, 255, 255, 213, 25, 0, 36, 206, 255, 255, 255, 255, 45, 0, 0, 0, 0,
            ],
            &[
                0, 0, 187, 255, 255, 255, 253, 238, 255, 255, 255, 255, 255, 255, 217, 13, 0, 0, 0,
            ],
            &[
                0, 0, 47, 247, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 168, 0, 0, 0,
            ],
            &[
                0, 0, 0, 54, 202, 255, 255, 255, 255, 203, 87, 7, 206, 255, 255, 255, 106, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 28, 58, 53, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ''' / 0x27
        '\'' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 55, 71, 71, 67, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 187, 255, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 168, 255, 255, 212, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 149, 255, 255, 194, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 130, 255, 255, 175, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 112, 255, 255, 156, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 93, 255, 255, 137, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 53, 176, 176, 83, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '(' / 0x28
        '(' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 11, 71, 71, 70, 4, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 156, 255, 255, 165, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 62, 255, 255, 244, 24, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 202, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 65, 255, 255, 251, 28, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 165, 255, 255, 183, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 6, 244, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 65, 255, 255, 255, 34, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 119, 255, 255, 236, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 164, 255, 255, 193, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 193, 255, 255, 164, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 211, 255, 255, 146, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 220, 255, 255, 138, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 215, 255, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 202, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 174, 255, 255, 187, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 136, 255, 255, 228, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 84, 255, 255, 255, 25, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 253, 255, 255, 93, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 192, 255, 255, 175, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 95, 255, 255, 248, 20, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 6, 227, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 96, 255, 255, 236, 14, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 1, 189, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 26, 116, 116, 111, 3, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ')' / 0x29
        ')' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 71, 71, 25, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 113, 255, 255, 204, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 4, 218, 255, 255, 120, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 93, 255, 255, 243, 19, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 5, 231, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 140, 255, 255, 225, 1, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 55, 255, 255, 255, 55, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 2, 238, 255, 255, 125, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 186, 255, 255, 178, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 255, 220, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 112, 255, 255, 248, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 94, 255, 255, 255, 8, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 87, 255, 255, 255, 15, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 93, 255, 255, 255, 8, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 109, 255, 255, 248, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 136, 255, 255, 221, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 179, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 228, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 37, 255, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 117, 255, 255, 229, 3, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 64, 255, 255, 249, 26, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 193, 255, 255, 139, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 85, 255, 255, 224, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 93, 116, 116, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '*' / 0x2a
        '*' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 44, 98, 98, 66, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 101, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 81, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 0, 0, 0, 60, 255, 255, 114, 0, 0, 0, 6, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 116, 216, 149, 82, 57, 255, 255, 99, 66, 137, 209, 179, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 175, 255, 255, 255, 255, 255, 255, 253, 255, 255, 255, 232, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 191, 246, 255, 255, 255, 255, 255, 255, 255, 255, 254, 231, 19, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 25, 89, 253, 255, 255, 255, 131, 36, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 6, 200, 255, 244, 225, 255, 234, 28, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 146, 255, 255, 132, 92, 255, 255, 199, 6, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 70, 255, 255, 238, 16, 2, 209, 255, 255, 128, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 99, 234, 120, 0, 0, 75, 248, 131, 9, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 16, 9, 0, 0, 0, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '+' / 0x2b
        '+' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 76, 251, 251, 147, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 27, 246, 246, 246, 246, 249, 255, 255, 251, 246, 246, 246, 246, 76, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 25, 230, 230, 230, 230, 238, 255, 255, 245, 230, 230, 230, 230, 71, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 65, 215, 215, 125, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ',' / 0x2c
        ',' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 61, 171, 171, 171, 88, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 126, 255, 255, 255, 90, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 167, 255, 255, 254, 22, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 212, 255, 255, 203, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 6, 250, 255, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 47, 255, 255, 255, 40, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 100, 255, 255, 207, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 155, 255, 255, 105, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 40, 54, 54, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '-' / 0x2d
        '-' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 221, 232, 232, 232, 232, 232, 232, 232, 34, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 243, 255, 255, 255, 255, 255, 255, 255, 38, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 243, 255, 255, 255, 255, 255, 255, 255, 38, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 113, 118, 118, 118, 118, 118, 118, 118, 17, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '.' / 0x2e
        '.' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 20, 101, 107, 33, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 15, 229, 255, 255, 245, 38, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 75, 255, 255, 255, 255, 119, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 65, 255, 255, 255, 255, 108, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 3, 186, 255, 255, 212, 15, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 1, 45, 52, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '/' / 0x2f
        '/' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 93, 98, 98, 42, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 54, 255, 255, 255, 42, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 151, 255, 255, 200, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 240, 255, 255, 103, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 90, 255, 255, 246, 15, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 188, 255, 255, 165, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 31, 254, 255, 255, 68, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 127, 255, 255, 224, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 1, 223, 255, 255, 129, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 66, 255, 255, 254, 33, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 163, 255, 255, 191, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 15, 246, 255, 255, 94, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 102, 255, 255, 242, 10, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 200, 255, 255, 156, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 42, 255, 255, 255, 58, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 139, 255, 255, 217, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 4, 232, 255, 255, 120, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 78, 255, 255, 252, 26, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 175, 255, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 22, 250, 255, 255, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 7, 27, 27, 27, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '0' / 0x30
        '0' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 22, 88, 122, 132, 101, 32, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 137, 245, 255, 255, 255, 255, 254, 159, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 156, 255, 255, 255, 255, 255, 255, 255, 255, 175, 3, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 75, 255, 255, 255, 255, 213, 206, 252, 255, 255, 255, 105, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 195, 255, 255, 255, 104, 0, 0, 101, 255, 255, 255, 222, 3, 0, 0, 0,
            ],
            &[
                0, 0, 20, 255, 255, 255, 201, 0, 0, 0, 173, 255, 255, 255, 255, 59, 0, 0, 0,
            ],
            &[
                0, 0, 75, 255, 255, 255, 123, 0, 0, 72, 255, 255, 255, 255, 255, 115, 0, 0, 0,
            ],
            &[
                0, 0, 121, 255, 255, 255, 80, 0, 7, 219, 255, 255, 255, 255, 255, 166, 0, 0, 0,
            ],
            &[
                0, 0, 139, 255, 255, 255, 52, 0, 125, 255, 255, 206, 255, 255, 255, 186, 0, 0, 0,
            ],
            &[
                0, 0, 154, 255, 255, 255, 41, 32, 246, 255, 247, 46, 255, 255, 255, 202, 0, 0, 0,
            ],
            &[
                0, 0, 161, 255, 255, 255, 36, 178, 255, 255, 125, 5, 255, 255, 255, 214, 0, 0, 0,
            ],
            &[
                0, 0, 146, 255, 255, 255, 124, 255, 255, 217, 7, 16, 255, 255, 255, 206, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 255, 255, 68, 0, 29, 255, 255, 255, 193, 0, 0, 0,
            ],
            &[
                0, 0, 103, 255, 255, 255, 255, 255, 167, 0, 0, 51, 255, 255, 255, 161, 0, 0, 0,
            ],
            &[
                0, 0, 48, 255, 255, 255, 255, 241, 25, 0, 0, 105, 255, 255, 255, 120, 0, 0, 0,
            ],
            &[
                0, 0, 3, 241, 255, 255, 255, 111, 0, 0, 0, 174, 255, 255, 255, 54, 0, 0, 0,
            ],
            &[
                0, 0, 0, 141, 255, 255, 255, 134, 5, 1, 97, 253, 255, 255, 218, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 27, 243, 255, 255, 255, 246, 242, 255, 255, 255, 255, 93, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 68, 249, 255, 255, 255, 255, 255, 255, 255, 154, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 57, 195, 255, 255, 255, 255, 218, 101, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 20, 58, 59, 32, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '1' / 0x31
        '1' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 53, 71, 71, 49, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 54, 189, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 39, 172, 255, 255, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 7, 155, 252, 255, 255, 255, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 190, 255, 255, 255, 255, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 50, 254, 255, 215, 102, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 141, 105, 4, 45, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 52, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 56, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 57, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 57, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 57, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 57, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 57, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 57, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 57, 255, 255, 255, 177, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 2, 81, 255, 255, 255, 209, 10, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 188, 226, 251, 255, 255, 255, 255, 255, 255, 237, 209, 59, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '2' / 0x32
        '2' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 50, 105, 130, 132, 109, 56, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 212, 255, 255, 255, 255, 255, 255, 217, 62, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 5, 144, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 70, 0, 0, 0, 0,
            ],
            &[
                0, 0, 60, 252, 255, 255, 255, 254, 231, 240, 255, 255, 255, 255, 229, 5, 0, 0, 0,
            ],
            &[
                0, 0, 0, 107, 255, 250, 141, 30, 0, 0, 99, 255, 255, 255, 255, 73, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 128, 57, 0, 0, 0, 0, 0, 191, 255, 255, 255, 109, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 165, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 212, 255, 255, 255, 60, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 73, 255, 255, 255, 224, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 23, 227, 255, 255, 255, 93, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 14, 203, 255, 255, 255, 173, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 12, 197, 255, 255, 255, 201, 10, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 11, 193, 255, 255, 255, 195, 14, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 11, 192, 255, 255, 255, 187, 10, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 11, 192, 255, 255, 255, 170, 7, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 11, 192, 255, 255, 255, 142, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 11, 192, 255, 255, 255, 203, 89, 89, 89, 89, 89, 89, 89, 87, 0, 0, 0,
            ],
            &[
                0, 0, 114, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '3' / 0x33
        '3' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 5, 56, 107, 130, 130, 104, 44, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 21, 140, 235, 255, 255, 255, 255, 255, 255, 197, 43, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 5, 225, 255, 255, 255, 255, 255, 255, 255, 255, 255, 237, 32, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 255, 253, 219, 229, 255, 255, 255, 255, 152, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 199, 194, 81, 9, 0, 0, 103, 255, 255, 255, 204, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0, 234, 255, 255, 213, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 238, 255, 255, 167, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 118, 255, 255, 255, 62, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 33, 121, 126, 147, 210, 255, 255, 255, 126, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 70, 255, 255, 255, 255, 255, 188, 67, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 70, 255, 255, 255, 255, 255, 247, 188, 74, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 46, 168, 171, 185, 226, 255, 255, 255, 255, 117, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 77, 248, 255, 255, 251, 25, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 169, 255, 255, 255, 90, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 156, 255, 255, 255, 105, 0, 0, 0,
            ],
            &[
                0, 0, 4, 13, 0, 0, 0, 0, 0, 0, 7, 218, 255, 255, 255, 77, 0, 0, 0,
            ],
            &[
                0, 0, 21, 243, 163, 92, 43, 15, 19, 59, 184, 255, 255, 255, 249, 25, 0, 0, 0,
            ],
            &[
                0, 0, 21, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 150, 0, 0, 0, 0,
            ],
            &[
                0, 0, 21, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 181, 7, 0, 0, 0, 0,
            ],
            &[
                0, 0, 4, 106, 186, 243, 255, 255, 255, 255, 252, 199, 90, 1, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 2, 32, 57, 64, 45, 14, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '4' / 0x34
        '4' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 71, 71, 71, 45, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 128, 255, 255, 255, 162, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 42, 249, 255, 255, 255, 162, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 1, 196, 255, 255, 255, 255, 162, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 104, 255, 255, 255, 255, 255, 162, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 26, 241, 255, 234, 245, 255, 255, 162, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 174, 255, 255, 112, 248, 255, 255, 162, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 81, 255, 255, 214, 9, 255, 255, 255, 162, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 14, 229, 255, 255, 67, 14, 255, 255, 255, 162, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 150, 255, 255, 170, 0, 21, 255, 255, 255, 162, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 59, 254, 255, 244, 29, 0, 23, 255, 255, 255, 162, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 6, 213, 255, 255, 121, 0, 0, 23, 255, 255, 255, 162, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 232, 46, 39, 39, 59, 255, 255, 255, 177, 39, 39, 4, 0, 0,
            ],
            &[
                0, 0, 233, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 30, 0,
                0,
            ],
            &[
                0, 0, 235, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 30, 0,
                0,
            ],
            &[
                0, 0, 235, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 30, 0,
                0,
            ],
            &[
                0, 0, 0, 1, 1, 1, 1, 1, 1, 23, 255, 255, 255, 163, 1, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 23, 255, 255, 255, 162, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 23, 255, 255, 255, 162, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 23, 255, 255, 255, 162, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '5' / 0x35
        '5' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 62, 71, 71, 71, 71, 71, 71, 71, 71, 71, 8, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 237, 255, 255, 255, 255, 255, 255, 255, 255, 255, 31, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 31, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 20, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 31, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 40, 255, 255, 255, 162, 17, 17, 17, 17, 17, 17, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 60, 255, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 79, 255, 255, 255, 116, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 99, 255, 255, 255, 112, 52, 59, 34, 1, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 119, 255, 255, 255, 255, 255, 255, 255, 224, 112, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 169, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 109, 251, 255, 254, 238, 252, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 33, 59, 5, 0, 9, 71, 226, 255, 255, 255, 208, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 255, 255, 255, 255, 11, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 250, 255, 255, 255, 36, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 251, 255, 255, 255, 20, 0, 0, 0,
            ],
            &[
                0, 0, 0, 25, 0, 0, 0, 0, 0, 0, 76, 255, 255, 255, 241, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 238, 182, 112, 76, 45, 53, 112, 236, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 235, 31, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 242, 255, 255, 255, 255, 255, 255, 255, 255, 232, 55, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 101, 197, 252, 255, 255, 255, 255, 222, 136, 16, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 9, 45, 63, 53, 28, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '6' / 0x36
        '6' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 5, 59, 95, 114, 122, 107, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 33, 158, 243, 255, 255, 255, 255, 255, 93, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 88, 248, 255, 255, 255, 255, 255, 255, 255, 93, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 83, 254, 255, 255, 255, 254, 213, 185, 184, 208, 86, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 22, 236, 255, 255, 255, 168, 31, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 126, 255, 255, 255, 159, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 224, 255, 255, 243, 16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 29, 255, 255, 255, 166, 0, 34, 97, 127, 111, 47, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 80, 255, 255, 255, 105, 132, 250, 255, 255, 255, 255, 170, 9, 0, 0, 0, 0,
            ],
            &[
                0, 0, 105, 255, 255, 255, 182, 255, 255, 255, 255, 255, 255, 255, 162, 0, 0, 0, 0,
            ],
            &[
                0, 0, 121, 255, 255, 255, 255, 255, 213, 168, 224, 255, 255, 255, 255, 42, 0, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 255, 117, 0, 0, 10, 203, 255, 255, 255, 121, 0, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 255, 196, 0, 0, 0, 0, 97, 255, 255, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 89, 255, 255, 255, 145, 0, 0, 0, 0, 66, 255, 255, 255, 178, 0, 0, 0,
            ],
            &[
                0, 0, 39, 255, 255, 255, 160, 0, 0, 0, 0, 83, 255, 255, 255, 159, 0, 0, 0,
            ],
            &[
                0, 0, 1, 226, 255, 255, 234, 13, 0, 0, 0, 155, 255, 255, 255, 126, 0, 0, 0,
            ],
            &[
                0, 0, 0, 129, 255, 255, 255, 198, 70, 48, 126, 252, 255, 255, 255, 45, 0, 0, 0,
            ],
            &[
                0, 0, 0, 13, 227, 255, 255, 255, 255, 255, 255, 255, 255, 255, 180, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 51, 239, 255, 255, 255, 255, 255, 255, 255, 216, 24, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 32, 171, 249, 255, 255, 255, 240, 147, 14, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 14, 54, 68, 43, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '7' / 0x37
        '7' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 41, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 54, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0,
            ],
            &[
                0, 0, 11, 17, 17, 17, 17, 17, 17, 17, 17, 180, 255, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 33, 252, 255, 255, 225, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 146, 255, 255, 255, 117, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 18, 244, 255, 255, 244, 16, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 123, 255, 255, 255, 149, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 7, 232, 255, 255, 254, 39, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 99, 255, 255, 255, 181, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 1, 214, 255, 255, 255, 70, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 76, 255, 255, 255, 213, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 192, 255, 255, 255, 102, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 53, 255, 255, 255, 236, 9, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 169, 255, 255, 255, 134, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 33, 252, 255, 255, 251, 27, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 145, 255, 255, 255, 166, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 17, 244, 255, 255, 255, 55, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 122, 255, 255, 255, 199, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '8' / 0x38
        '8' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 24, 90, 120, 124, 100, 41, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 164, 253, 255, 255, 255, 255, 255, 199, 49, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 13, 214, 255, 255, 255, 255, 255, 255, 255, 255, 241, 46, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 124, 255, 255, 255, 254, 194, 185, 248, 255, 255, 255, 178, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 185, 255, 255, 255, 91, 0, 0, 45, 250, 255, 255, 237, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 191, 255, 255, 255, 19, 0, 0, 0, 223, 255, 255, 245, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 150, 255, 255, 255, 86, 0, 0, 34, 253, 255, 255, 201, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 46, 252, 255, 255, 240, 72, 43, 211, 255, 255, 255, 89, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 132, 255, 255, 255, 255, 250, 255, 255, 255, 142, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 119, 255, 255, 255, 255, 255, 249, 89, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 29, 189, 255, 255, 255, 255, 255, 255, 187, 27, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 32, 229, 255, 255, 255, 217, 234, 255, 255, 255, 229, 37, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 194, 255, 255, 255, 154, 8, 20, 176, 255, 255, 255, 213, 4, 0, 0, 0,
            ],
            &[
                0, 0, 43, 255, 255, 255, 198, 3, 0, 0, 4, 186, 255, 255, 255, 86, 0, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 117, 0, 0, 0, 0, 73, 255, 255, 255, 142, 0, 0, 0,
            ],
            &[
                0, 0, 90, 255, 255, 255, 129, 0, 0, 0, 0, 77, 255, 255, 255, 137, 0, 0, 0,
            ],
            &[
                0, 0, 54, 255, 255, 255, 241, 71, 0, 0, 46, 214, 255, 255, 255, 94, 0, 0, 0,
            ],
            &[
                0, 0, 1, 208, 255, 255, 255, 255, 245, 242, 255, 255, 255, 255, 228, 8, 0, 0, 0,
            ],
            &[
                0, 0, 0, 43, 232, 255, 255, 255, 255, 255, 255, 255, 255, 236, 49, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 23, 145, 232, 255, 255, 255, 255, 232, 145, 21, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 34, 57, 59, 36, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '9' / 0x39
        '9' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 44, 108, 136, 131, 93, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 27, 190, 255, 255, 255, 255, 255, 247, 123, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 22, 230, 255, 255, 255, 255, 255, 255, 255, 255, 154, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 170, 255, 255, 255, 255, 239, 244, 255, 255, 255, 255, 84, 0, 0, 0, 0,
            ],
            &[
                0, 0, 21, 251, 255, 255, 254, 90, 0, 3, 104, 254, 255, 255, 213, 0, 0, 0, 0,
            ],
            &[
                0, 0, 87, 255, 255, 255, 179, 0, 0, 0, 0, 170, 255, 255, 255, 50, 0, 0, 0,
            ],
            &[
                0, 0, 114, 255, 255, 255, 127, 0, 0, 0, 0, 106, 255, 255, 255, 109, 0, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 120, 0, 0, 0, 0, 98, 255, 255, 255, 152, 0, 0, 0,
            ],
            &[
                0, 0, 106, 255, 255, 255, 166, 0, 0, 0, 0, 172, 255, 255, 255, 169, 0, 0, 0,
            ],
            &[
                0, 0, 54, 255, 255, 255, 250, 78, 0, 16, 135, 255, 255, 255, 255, 179, 0, 0, 0,
            ],
            &[
                0, 0, 1, 215, 255, 255, 255, 255, 242, 252, 255, 255, 255, 255, 255, 169, 0, 0, 0,
            ],
            &[
                0, 0, 0, 64, 249, 255, 255, 255, 255, 255, 255, 138, 255, 255, 255, 153, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 65, 211, 255, 255, 255, 233, 104, 68, 255, 255, 255, 116, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 31, 57, 35, 4, 0, 133, 255, 255, 255, 67, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 229, 255, 255, 243, 6, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 158, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 9, 16, 0, 0, 8, 70, 191, 255, 255, 255, 240, 28, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 56, 255, 244, 240, 254, 255, 255, 255, 255, 252, 83, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 56, 255, 255, 255, 255, 255, 255, 255, 234, 78, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 56, 255, 255, 255, 255, 253, 210, 121, 17, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 5, 46, 64, 62, 46, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ':' / 0x3a
        ':' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 130, 237, 242, 159, 4, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 51, 255, 255, 255, 255, 92, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 80, 255, 255, 255, 255, 124, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 28, 246, 255, 255, 255, 61, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 61, 166, 172, 83, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 20, 101, 107, 33, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 15, 229, 255, 255, 245, 38, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 75, 255, 255, 255, 255, 119, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 65, 255, 255, 255, 255, 108, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 3, 186, 255, 255, 212, 15, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 1, 45, 52, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ';' / 0x3b
        ';' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 29, 192, 248, 223, 80, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 241, 3, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 184, 255, 255, 255, 255, 20, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 123, 255, 255, 255, 212, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 5, 118, 178, 151, 30, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 61, 171, 171, 171, 88, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 126, 255, 255, 255, 90, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 167, 255, 255, 254, 22, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 212, 255, 255, 203, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 6, 250, 255, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 47, 255, 255, 255, 40, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 100, 255, 255, 207, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 155, 255, 255, 105, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 40, 54, 54, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '<' / 0x3c
        '<' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 60, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 65, 197, 231, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 57, 189, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 49, 180, 255, 255, 255, 255, 206, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 42, 172, 254, 255, 255, 255, 219, 97, 4, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 36, 164, 253, 255, 255, 255, 217, 93, 3, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 123, 251, 255, 255, 255, 214, 90, 2, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 211, 32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 100, 236, 255, 255, 255, 251, 163, 45, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 117, 232, 255, 255, 255, 254, 181, 62, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 9, 110, 227, 255, 255, 255, 255, 199, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 6, 103, 222, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 95, 217, 255, 231, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 88, 187, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '=' / 0x3d
        '=' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 20, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 56, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 14, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 41, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 1, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 3, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 5, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 15, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '>' / 0x3e
        '>' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 180, 219, 91, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 180, 255, 255, 213, 83, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 155, 255, 255, 255, 255, 206, 75, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 72, 198, 255, 255, 255, 255, 198, 66, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 68, 194, 255, 255, 255, 255, 190, 58, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 65, 191, 255, 255, 255, 255, 170, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 15, 177, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 26, 139, 242, 255, 255, 255, 248, 140, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 40, 157, 249, 255, 255, 255, 245, 142, 25, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 52, 176, 254, 255, 255, 255, 242, 135, 21, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 238, 127, 17, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 234, 120, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 154, 113, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '?' / 0x3f
        '?' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 2, 53, 103, 127, 133, 111, 54, 2, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 24, 137, 230, 255, 255, 255, 255, 255, 255, 210, 55, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 160, 255, 255, 255, 255, 255, 255, 255, 255, 255, 243, 47, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 40, 253, 255, 255, 238, 196, 193, 249, 255, 255, 255, 178, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 156, 158, 54, 0, 0, 0, 40, 245, 255, 255, 241, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 210, 255, 255, 255, 4, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 249, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 191, 255, 255, 255, 145, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 33, 212, 255, 255, 255, 208, 14, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 48, 233, 255, 255, 255, 179, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 5, 211, 255, 255, 254, 134, 2, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 58, 255, 255, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 96, 255, 255, 255, 35, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 63, 153, 153, 153, 11, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 86, 113, 59, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 176, 255, 255, 255, 98, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 8, 253, 255, 255, 255, 188, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 3, 248, 255, 255, 255, 178, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 122, 254, 255, 239, 58, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 30, 58, 15, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '@' / 0x40
        '@' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 33, 61, 58, 18, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 20, 141, 233, 255, 255, 255, 254, 188, 54, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 66, 231, 255, 255, 255, 250, 253, 255, 255, 250, 85, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 54, 245, 255, 249, 136, 39, 3, 7, 83, 226, 255, 252, 67, 0, 0, 0,
            ],
            &[
                0, 0, 10, 226, 255, 246, 55, 0, 0, 0, 0, 0, 28, 228, 255, 215, 5, 0, 0,
            ],
            &[
                0, 0, 128, 255, 255, 90, 0, 0, 0, 3, 0, 0, 0, 86, 255, 255, 98, 0, 0,
            ],
            &[
                0, 4, 234, 255, 208, 0, 0, 110, 223, 254, 235, 194, 121, 6, 234, 255, 185, 0, 0,
            ],
            &[
                0, 76, 255, 255, 97, 0, 132, 255, 255, 255, 255, 255, 254, 1, 172, 255, 243, 1, 0,
            ],
            &[
                0, 140, 255, 255, 29, 38, 251, 255, 214, 105, 209, 255, 243, 0, 132, 255, 255, 26,
                0,
            ],
            &[
                0, 192, 255, 224, 0, 127, 255, 255, 57, 0, 181, 255, 230, 0, 116, 255, 255, 44, 0,
            ],
            &[
                0, 225, 255, 196, 0, 184, 255, 225, 0, 0, 193, 255, 217, 0, 112, 255, 255, 46, 0,
            ],
            &[
                0, 241, 255, 178, 0, 206, 255, 197, 0, 0, 211, 255, 204, 0, 129, 255, 255, 31, 0,
            ],
            &[
                0, 252, 255, 171, 0, 200, 255, 192, 0, 0, 239, 255, 192, 0, 152, 255, 248, 3, 0,
            ],
            &[
                0, 243, 255, 189, 0, 173, 255, 221, 0, 40, 255, 255, 209, 0, 211, 255, 192, 0, 0,
            ],
            &[
                0, 217, 255, 212, 0, 110, 255, 255, 126, 189, 255, 255, 246, 140, 254, 255, 102, 0,
                0,
            ],
            &[
                0, 174, 255, 253, 18, 17, 238, 255, 255, 255, 229, 230, 255, 255, 255, 214, 7, 0, 0,
            ],
            &[
                0, 108, 255, 255, 90, 0, 54, 214, 255, 216, 43, 61, 235, 255, 196, 31, 0, 0, 0,
            ],
            &[
                0, 21, 248, 255, 214, 3, 0, 0, 9, 0, 0, 0, 2, 10, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 147, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 10, 211, 255, 255, 179, 53, 1, 0, 0, 13, 76, 163, 88, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 24, 204, 255, 255, 255, 247, 220, 230, 254, 255, 255, 100, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 6, 115, 225, 255, 255, 255, 255, 255, 250, 181, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 1, 42, 84, 99, 91, 62, 14, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'A' / 0x41
        'A' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 42, 91, 91, 91, 91, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 165, 255, 255, 255, 255, 210, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 1, 234, 255, 255, 255, 255, 255, 25, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 50, 255, 255, 255, 247, 255, 255, 95, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 120, 255, 255, 230, 179, 255, 255, 166, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 190, 255, 255, 172, 119, 255, 255, 235, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 11, 249, 255, 255, 112, 59, 255, 255, 255, 51, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 255, 255, 255, 49, 7, 247, 255, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 145, 255, 255, 239, 2, 0, 193, 255, 255, 193, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 216, 255, 255, 178, 0, 0, 131, 255, 255, 250, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 30, 255, 255, 255, 114, 0, 0, 70, 255, 255, 255, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 100, 255, 255, 255, 55, 4, 4, 17, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 238, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 34, 0, 0, 0,
            ],
            &[
                0, 0, 56, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 105, 0, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 107, 56, 56, 56, 56, 75, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 196, 255, 255, 240, 2, 0, 0, 0, 0, 0, 212, 255, 255, 241, 4, 0, 0,
            ],
            &[
                0, 15, 251, 255, 255, 179, 0, 0, 0, 0, 0, 0, 150, 255, 255, 255, 61, 0, 0,
            ],
            &[
                0, 81, 255, 255, 255, 115, 0, 0, 0, 0, 0, 0, 88, 255, 255, 255, 131, 0, 0,
            ],
            &[
                0, 151, 255, 255, 255, 51, 0, 0, 0, 0, 0, 0, 27, 255, 255, 255, 202, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'B' / 0x42
        'B' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 59, 71, 71, 71, 71, 71, 67, 53, 19, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 214, 112, 5, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 182, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 252, 239, 239, 250, 255, 255, 255, 255, 255, 88, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 205, 0, 0, 1, 44, 210, 255, 255, 255, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 205, 0, 0, 0, 0, 90, 255, 255, 255, 175, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 205, 0, 0, 0, 0, 79, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 205, 0, 0, 0, 4, 171, 255, 255, 255, 79, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 176, 176, 186, 228, 255, 255, 255, 187, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 246, 123, 9, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 80, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 237, 161, 161, 171, 213, 255, 255, 255, 254, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 205, 0, 0, 0, 0, 121, 255, 255, 255, 209, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 205, 0, 0, 0, 0, 17, 255, 255, 255, 255, 11, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 205, 0, 0, 0, 0, 4, 252, 255, 255, 255, 23, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 205, 0, 0, 0, 0, 61, 255, 255, 255, 250, 2, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 216, 54, 54, 59, 99, 225, 255, 255, 255, 186, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 253, 66, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 249, 105, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 240, 213, 144, 41, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'C' / 0x43
        'C' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 30, 93, 119, 135, 125, 88, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 69, 202, 255, 255, 255, 255, 255, 255, 249, 171, 5, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 189, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 129, 255, 255, 255, 255, 255, 245, 233, 253, 255, 255, 86, 0, 0, 0,
            ],
            &[
                0, 0, 0, 50, 251, 255, 255, 255, 169, 45, 0, 0, 5, 68, 154, 5, 0, 0, 0,
            ],
            &[
                0, 0, 0, 168, 255, 255, 255, 156, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 16, 250, 255, 255, 243, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 72, 255, 255, 255, 173, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 123, 255, 255, 255, 122, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 94, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 160, 255, 255, 255, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 150, 255, 255, 255, 96, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 130, 255, 255, 255, 117, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 101, 255, 255, 255, 172, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 34, 255, 255, 255, 242, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 255, 255, 170, 4, 0, 0, 0, 0, 0, 4, 16, 0, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 255, 255, 204, 99, 58, 53, 84, 144, 232, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 194, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 17, 196, 255, 255, 255, 255, 255, 255, 255, 255, 255, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 4, 106, 211, 255, 255, 255, 255, 255, 227, 151, 31, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 20, 54, 64, 43, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'D' / 0x44
        'D' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 71, 71, 66, 46, 7, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 247, 175, 54, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 253, 126, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 116, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 29, 11, 26, 74, 196, 255, 255, 255, 249, 36, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 6, 191, 255, 255, 255, 151, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 50, 255, 255, 255, 235, 1, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 0, 227, 255, 255, 255, 39, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 0, 177, 255, 255, 255, 75, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 0, 159, 255, 255, 255, 92, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 0, 152, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 0, 164, 255, 255, 255, 84, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 0, 197, 255, 255, 255, 55, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 10, 246, 255, 255, 251, 12, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 103, 255, 255, 255, 192, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 51, 239, 255, 255, 255, 89, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 95, 83, 101, 162, 250, 255, 255, 255, 202, 2, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 230, 33, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 199, 33, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 247, 221, 166, 77, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'E' / 0x45
        'E' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 59, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 22, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 242, 24, 24, 24, 24, 24, 24, 24, 24, 7, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 176, 176, 176, 176, 176, 176, 176, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 175, 175, 175, 175, 175, 175, 175, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 246, 95, 95, 95, 95, 95, 95, 95, 95, 29, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'F' / 0x46
        'F' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 59, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 22, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 242, 24, 24, 24, 24, 24, 24, 24, 24, 7, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 254, 225, 225, 225, 225, 225, 225, 225, 204, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 248, 126, 126, 126, 126, 126, 126, 126, 114, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'G' / 0x47
        'G' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 26, 88, 124, 136, 126, 97, 46, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 41, 179, 254, 255, 255, 255, 255, 255, 255, 227, 106, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 79, 245, 255, 255, 255, 255, 255, 255, 255, 255, 255, 125, 0, 0,
            ],
            &[
                0, 0, 0, 0, 54, 250, 255, 255, 255, 255, 252, 225, 229, 253, 255, 253, 28, 0, 0,
            ],
            &[
                0, 0, 0, 2, 211, 255, 255, 255, 237, 101, 8, 0, 0, 7, 83, 130, 0, 0, 0,
            ],
            &[
                0, 0, 0, 79, 255, 255, 255, 243, 45, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 168, 255, 255, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 230, 255, 255, 255, 34, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 37, 255, 255, 255, 204, 0, 0, 0, 49, 218, 218, 218, 218, 218, 215, 0, 0,
            ],
            &[
                0, 0, 45, 255, 255, 255, 190, 0, 0, 0, 57, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 39, 255, 255, 255, 203, 0, 0, 0, 57, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 17, 255, 255, 255, 230, 0, 0, 0, 22, 98, 98, 185, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 235, 255, 255, 255, 33, 0, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 174, 255, 255, 255, 132, 0, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 87, 255, 255, 255, 243, 50, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 4, 217, 255, 255, 255, 239, 118, 43, 19, 39, 173, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 57, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 72, 239, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 25, 144, 232, 255, 255, 255, 255, 255, 242, 179, 88, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 31, 58, 65, 56, 30, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'H' / 0x48
        'H' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 9, 0, 0, 0, 0, 0, 66, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 199, 190, 190, 190, 190, 190, 251, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 173, 161, 161, 161, 161, 161, 249, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'I' / 0x49
        'I' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 40, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 66, 139, 168, 198, 248, 255, 255, 253, 202, 173, 144, 91, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 13, 214, 255, 255, 241, 17, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 210, 240, 255, 255, 255, 255, 255, 255, 244, 215, 146, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'J' / 0x4a
        'J' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 42, 71, 71, 71, 16, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 158, 255, 255, 255, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 183, 255, 255, 255, 33, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 23, 244, 255, 255, 249, 9, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 105, 112, 70, 47, 41, 83, 211, 255, 255, 255, 184, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 255, 255, 255, 255, 255, 255, 72, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 255, 255, 255, 255, 255, 136, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 140, 242, 255, 255, 255, 255, 253, 192, 80, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 34, 54, 63, 42, 15, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'K' / 0x4b
        'K' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 3, 0, 0, 0, 0, 7, 71, 71, 71, 71, 13, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 0, 0, 135, 255, 255, 255, 186, 1, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 0, 53, 252, 255, 255, 236, 26, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 7, 213, 255, 255, 255, 82, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 135, 255, 255, 255, 157, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 53, 252, 255, 255, 220, 12, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 7, 213, 255, 255, 252, 56, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 132, 255, 255, 255, 128, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 54, 250, 255, 255, 199, 4, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 191, 255, 255, 255, 94, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 214, 3, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 104, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 248, 116, 252, 255, 255, 232, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 71, 0, 156, 255, 255, 255, 132, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 29, 247, 255, 255, 246, 27, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 141, 255, 255, 255, 159, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 20, 242, 255, 255, 253, 48, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 0, 126, 255, 255, 255, 187, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 0, 12, 234, 255, 255, 255, 73, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 0, 0, 111, 255, 255, 255, 212, 3, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'L' / 0x4c
        'L' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 71, 71, 71, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 164, 116, 116, 116, 116, 116, 116, 116, 36, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'M' / 0x4d
        'M' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 65, 71, 71, 71, 71, 27, 0, 0, 0, 67, 71, 71, 71, 71, 6, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 255, 255, 125, 0, 0, 17, 255, 255, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 255, 255, 163, 0, 0, 62, 255, 255, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 255, 255, 202, 0, 0, 108, 255, 255, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 245, 227, 255, 240, 0, 0, 153, 255, 242, 232, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 254, 179, 255, 255, 23, 0, 199, 255, 198, 235, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 149, 255, 255, 61, 1, 243, 255, 154, 252, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 121, 255, 255, 100, 34, 255, 255, 122, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 91, 255, 255, 139, 79, 255, 255, 94, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 81, 236, 255, 177, 125, 255, 255, 64, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 97, 188, 255, 216, 170, 255, 228, 60, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 113, 139, 255, 250, 225, 255, 183, 73, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 125, 90, 255, 255, 255, 255, 138, 84, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 137, 41, 255, 255, 255, 255, 92, 94, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 144, 3, 245, 255, 255, 255, 47, 100, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 149, 0, 99, 118, 118, 118, 6, 105, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 150, 0, 0, 0, 0, 0, 0, 106, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 150, 0, 0, 0, 0, 0, 0, 106, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 150, 0, 0, 0, 0, 0, 0, 106, 255, 255, 255, 23, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 150, 0, 0, 0, 0, 0, 0, 106, 255, 255, 255, 23, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'N' / 0x4e
        'N' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 71, 41, 0, 0, 0, 0, 35, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 201, 0, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 254, 30, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 113, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 222, 255, 255, 196, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 151, 253, 255, 254, 26, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 141, 195, 255, 255, 108, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 155, 112, 255, 255, 191, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 166, 30, 255, 255, 252, 22, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 175, 0, 202, 255, 255, 103, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 119, 255, 255, 187, 0, 126, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 36, 255, 255, 251, 19, 123, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 209, 255, 255, 98, 118, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 126, 255, 255, 182, 111, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 43, 255, 255, 249, 117, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 216, 255, 255, 182, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 133, 255, 255, 248, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 50, 255, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 0, 223, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 0, 140, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'O' / 0x4f
        'O' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 47, 114, 144, 144, 119, 55, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 187, 255, 255, 255, 255, 255, 255, 208, 49, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 25, 226, 255, 255, 255, 255, 255, 255, 255, 255, 244, 52, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 181, 255, 255, 255, 255, 224, 211, 253, 255, 255, 255, 221, 6, 0, 0, 0,
            ],
            &[
                0, 0, 52, 255, 255, 255, 252, 84, 0, 0, 46, 231, 255, 255, 255, 102, 0, 0, 0,
            ],
            &[
                0, 0, 136, 255, 255, 255, 145, 0, 0, 0, 0, 95, 255, 255, 255, 190, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 50, 0, 0, 0, 0, 17, 252, 255, 255, 251, 7, 0, 0,
            ],
            &[
                0, 2, 249, 255, 255, 250, 4, 0, 0, 0, 0, 0, 204, 255, 255, 255, 45, 0, 0,
            ],
            &[
                0, 17, 255, 255, 255, 221, 0, 0, 0, 0, 0, 0, 180, 255, 255, 255, 76, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 208, 0, 0, 0, 0, 0, 0, 162, 255, 255, 255, 89, 0, 0,
            ],
            &[
                0, 44, 255, 255, 255, 202, 0, 0, 0, 0, 0, 0, 151, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 212, 0, 0, 0, 0, 0, 0, 161, 255, 255, 255, 85, 0, 0,
            ],
            &[
                0, 18, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 179, 255, 255, 255, 69, 0, 0,
            ],
            &[
                0, 0, 236, 255, 255, 255, 12, 0, 0, 0, 0, 0, 219, 255, 255, 255, 32, 0, 0,
            ],
            &[
                0, 0, 190, 255, 255, 255, 77, 0, 0, 0, 0, 31, 254, 255, 255, 239, 1, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 190, 1, 0, 0, 0, 145, 255, 255, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 24, 244, 255, 255, 255, 168, 53, 45, 134, 254, 255, 255, 255, 64, 0, 0, 0,
            ],
            &[
                0, 0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 175, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 163, 255, 255, 255, 255, 255, 255, 255, 255, 202, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 94, 209, 255, 255, 255, 255, 225, 124, 6, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 54, 59, 34, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'P' / 0x50
        'P' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 59, 71, 71, 71, 71, 71, 69, 56, 24, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 217, 105, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 168, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 98, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 241, 11, 11, 16, 55, 190, 255, 255, 255, 204, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 19, 246, 255, 255, 253, 7, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 210, 255, 255, 255, 24, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 215, 255, 255, 255, 11, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 50, 255, 255, 255, 235, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 243, 45, 45, 58, 113, 234, 255, 255, 255, 159, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 247, 43, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 246, 89, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 232, 151, 42, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 242, 36, 36, 30, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Q' / 0x51
        'Q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 47, 114, 144, 144, 119, 55, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 187, 255, 255, 255, 255, 255, 255, 208, 49, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 25, 226, 255, 255, 255, 255, 255, 255, 255, 255, 244, 52, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 181, 255, 255, 255, 255, 224, 211, 253, 255, 255, 255, 221, 6, 0, 0, 0,
            ],
            &[
                0, 0, 52, 255, 255, 255, 252, 84, 0, 0, 46, 231, 255, 255, 255, 102, 0, 0, 0,
            ],
            &[
                0, 0, 136, 255, 255, 255, 145, 0, 0, 0, 0, 95, 255, 255, 255, 190, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 50, 0, 0, 0, 0, 17, 252, 255, 255, 251, 7, 0, 0,
            ],
            &[
                0, 2, 249, 255, 255, 250, 4, 0, 0, 0, 0, 0, 204, 255, 255, 255, 45, 0, 0,
            ],
            &[
                0, 17, 255, 255, 255, 221, 0, 0, 0, 0, 0, 0, 180, 255, 255, 255, 76, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 208, 0, 0, 0, 0, 0, 0, 162, 255, 255, 255, 89, 0, 0,
            ],
            &[
                0, 44, 255, 255, 255, 202, 0, 0, 0, 0, 0, 0, 151, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 212, 0, 0, 0, 0, 0, 0, 161, 255, 255, 255, 85, 0, 0,
            ],
            &[
                0, 18, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 179, 255, 255, 255, 69, 0, 0,
            ],
            &[
                0, 0, 236, 255, 255, 255, 12, 0, 0, 0, 0, 0, 219, 255, 255, 255, 32, 0, 0,
            ],
            &[
                0, 0, 190, 255, 255, 255, 77, 0, 0, 0, 0, 31, 254, 255, 255, 239, 1, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 190, 1, 0, 0, 0, 145, 255, 255, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 24, 244, 255, 255, 255, 168, 53, 45, 134, 254, 255, 255, 255, 64, 0, 0, 0,
            ],
            &[
                0, 0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 175, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 163, 255, 255, 255, 255, 255, 255, 255, 255, 202, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 94, 209, 255, 255, 255, 255, 255, 255, 20, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 26, 54, 179, 255, 255, 255, 131, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 235, 255, 255, 244, 25, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 152, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 11, 234, 255, 255, 251, 39, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 109, 220, 220, 220, 141, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'R' / 0x52
        'R' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 59, 71, 71, 71, 71, 66, 46, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 254, 201, 96, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 180, 4, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 137, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 29, 20, 53, 166, 255, 255, 255, 241, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 2, 212, 255, 255, 255, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 164, 255, 255, 255, 79, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 189, 255, 255, 255, 64, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 93, 252, 255, 255, 249, 12, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 157, 162, 207, 255, 255, 255, 255, 152, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 207, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 155, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 193, 209, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 13, 233, 255, 255, 240, 22, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 104, 255, 255, 255, 155, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 4, 214, 255, 255, 253, 50, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 77, 255, 255, 255, 195, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 191, 255, 255, 255, 88, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 51, 254, 255, 255, 226, 10, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 0, 163, 255, 255, 255, 129, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'S' / 0x53
        'S' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 15, 80, 118, 134, 118, 93, 59, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 139, 248, 255, 255, 255, 255, 255, 255, 247, 166, 30, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 247, 18, 0, 0, 0,
            ],
            &[
                0, 0, 0, 76, 255, 255, 255, 255, 236, 191, 199, 233, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 185, 9, 0, 0, 0, 50, 136, 51, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 179, 255, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 162, 255, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 99, 255, 255, 255, 239, 77, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 218, 255, 255, 255, 255, 181, 50, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 42, 229, 255, 255, 255, 255, 254, 164, 29, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 23, 170, 255, 255, 255, 255, 255, 246, 121, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 215, 255, 255, 255, 255, 255, 153, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 2, 93, 231, 255, 255, 255, 255, 66, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 19, 198, 255, 255, 255, 149, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 86, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 12, 47, 0, 0, 0, 0, 0, 0, 0, 103, 255, 255, 255, 160, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 195, 111, 50, 10, 0, 22, 93, 234, 255, 255, 255, 99, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 253, 255, 255, 255, 255, 255, 222, 11, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 229, 44, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 90, 181, 243, 255, 255, 255, 255, 255, 223, 132, 15, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 2, 32, 57, 65, 53, 22, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'T' / 0x54
        'T' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 13, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 27, 0, 0,
            ],
            &[
                0, 48, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0,
                0,
            ],
            &[
                0, 48, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0,
                0,
            ],
            &[
                0, 48, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0,
                0,
            ],
            &[
                0, 4, 24, 24, 24, 24, 24, 226, 255, 255, 255, 34, 24, 24, 24, 24, 9, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'U' / 0x55
        'U' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 9, 0, 0, 0, 0, 0, 66, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 238, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 44, 0, 0, 0, 0, 0, 248, 255, 255, 247, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 255, 255, 76, 0, 0, 0, 0, 23, 255, 255, 255, 224, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 160, 0, 0, 0, 0, 104, 255, 255, 255, 187, 0, 0, 0,
            ],
            &[
                0, 0, 59, 255, 255, 255, 254, 136, 45, 38, 103, 243, 255, 255, 255, 104, 0, 0, 0,
            ],
            &[
                0, 0, 1, 191, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 11, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 217, 255, 255, 255, 255, 255, 255, 255, 255, 230, 49, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 137, 228, 255, 255, 255, 255, 235, 148, 22, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 2, 30, 55, 57, 32, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'V' / 0x56
        'V' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 49, 71, 71, 71, 17, 0, 0, 0, 0, 0, 0, 7, 71, 71, 71, 63, 0, 0,
            ],
            &[
                0, 128, 255, 255, 255, 105, 0, 0, 0, 0, 0, 0, 66, 255, 255, 255, 179, 0, 0,
            ],
            &[
                0, 53, 255, 255, 255, 171, 0, 0, 0, 0, 0, 0, 129, 255, 255, 255, 104, 0, 0,
            ],
            &[
                0, 1, 232, 255, 255, 235, 1, 0, 0, 0, 0, 0, 192, 255, 255, 255, 30, 0, 0,
            ],
            &[
                0, 0, 159, 255, 255, 255, 46, 0, 0, 0, 0, 8, 248, 255, 255, 210, 0, 0, 0,
            ],
            &[
                0, 0, 84, 255, 255, 255, 112, 0, 0, 0, 0, 63, 255, 255, 255, 135, 0, 0, 0,
            ],
            &[
                0, 0, 14, 250, 255, 255, 178, 0, 0, 0, 0, 127, 255, 255, 255, 61, 0, 0, 0,
            ],
            &[
                0, 0, 0, 190, 255, 255, 240, 3, 0, 0, 0, 190, 255, 255, 238, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 115, 255, 255, 255, 54, 0, 0, 6, 246, 255, 255, 167, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 41, 255, 255, 255, 119, 0, 0, 61, 255, 255, 255, 92, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 221, 255, 255, 185, 0, 0, 124, 255, 255, 253, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 147, 255, 255, 245, 6, 0, 188, 255, 255, 198, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 72, 255, 255, 255, 60, 3, 244, 255, 255, 123, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 8, 245, 255, 255, 122, 45, 255, 255, 255, 48, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 178, 255, 255, 176, 95, 255, 255, 228, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 103, 255, 255, 224, 140, 255, 255, 154, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 29, 255, 255, 254, 216, 255, 255, 79, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 209, 255, 255, 255, 255, 248, 11, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 134, 255, 255, 255, 255, 185, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 59, 255, 255, 255, 255, 110, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'W' / 0x57
        'W' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 51, 71, 71, 48, 0, 0, 0, 0, 0, 0, 0, 0, 34, 71, 71, 65, 0, 0,
            ],
            &[
                0, 172, 255, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 134, 255, 255, 224, 0, 0,
            ],
            &[
                0, 153, 255, 255, 214, 0, 0, 0, 0, 0, 0, 0, 0, 153, 255, 255, 206, 0, 0,
            ],
            &[
                0, 134, 255, 255, 238, 0, 0, 0, 0, 0, 0, 0, 0, 171, 255, 255, 188, 0, 0,
            ],
            &[
                0, 115, 255, 255, 255, 7, 0, 0, 0, 0, 0, 0, 0, 189, 255, 255, 170, 0, 0,
            ],
            &[
                0, 96, 255, 255, 255, 30, 0, 150, 189, 189, 189, 8, 0, 207, 255, 255, 152, 0, 0,
            ],
            &[
                0, 77, 255, 255, 255, 49, 0, 237, 255, 255, 255, 45, 0, 225, 255, 255, 134, 0, 0,
            ],
            &[
                0, 58, 255, 255, 255, 67, 21, 255, 255, 255, 255, 83, 0, 244, 255, 255, 116, 0, 0,
            ],
            &[
                0, 39, 255, 255, 255, 85, 60, 255, 255, 255, 255, 122, 6, 255, 255, 255, 98, 0, 0,
            ],
            &[
                0, 20, 255, 255, 255, 98, 100, 255, 254, 212, 255, 160, 23, 255, 255, 255, 80, 0, 0,
            ],
            &[
                0, 2, 254, 255, 255, 107, 140, 255, 230, 165, 255, 199, 36, 255, 255, 255, 62, 0, 0,
            ],
            &[
                0, 0, 237, 255, 255, 113, 179, 255, 196, 132, 255, 238, 43, 255, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 218, 255, 255, 115, 216, 255, 159, 98, 255, 255, 67, 255, 255, 255, 26, 0, 0,
            ],
            &[
                0, 0, 199, 255, 255, 119, 249, 255, 119, 61, 255, 255, 106, 255, 255, 255, 9, 0, 0,
            ],
            &[
                0, 0, 180, 255, 255, 146, 255, 255, 81, 23, 255, 255, 139, 255, 255, 246, 0, 0, 0,
            ],
            &[
                0, 0, 161, 255, 255, 177, 255, 255, 43, 0, 241, 255, 171, 255, 255, 229, 0, 0, 0,
            ],
            &[
                0, 0, 142, 255, 255, 214, 255, 253, 8, 0, 203, 255, 210, 255, 255, 211, 0, 0, 0,
            ],
            &[
                0, 0, 124, 255, 255, 255, 255, 224, 0, 0, 166, 255, 255, 255, 255, 193, 0, 0, 0,
            ],
            &[
                0, 0, 105, 255, 255, 255, 255, 186, 0, 0, 128, 255, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 86, 255, 255, 255, 255, 148, 0, 0, 90, 255, 255, 255, 255, 157, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'X' / 0x58
        'X' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 1, 69, 71, 71, 69, 0, 0, 0, 0, 0, 0, 59, 71, 71, 71, 14, 0, 0,
            ],
            &[
                0, 0, 161, 255, 255, 255, 82, 0, 0, 0, 0, 45, 253, 255, 255, 209, 3, 0, 0,
            ],
            &[
                0, 0, 26, 243, 255, 255, 213, 2, 0, 0, 0, 175, 255, 255, 255, 64, 0, 0, 0,
            ],
            &[
                0, 0, 0, 121, 255, 255, 255, 94, 0, 0, 52, 255, 255, 255, 171, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 7, 222, 255, 255, 223, 6, 0, 183, 255, 255, 247, 32, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 81, 255, 255, 255, 107, 59, 255, 255, 255, 131, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 189, 255, 255, 231, 196, 255, 255, 227, 10, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 45, 252, 255, 255, 255, 255, 255, 90, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 149, 255, 255, 255, 255, 196, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 19, 238, 255, 255, 253, 51, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 11, 228, 255, 255, 251, 42, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 132, 255, 255, 255, 255, 186, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 33, 247, 255, 255, 255, 255, 255, 79, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 174, 255, 255, 234, 212, 255, 255, 220, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 67, 255, 255, 255, 112, 79, 255, 255, 255, 120, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 211, 255, 255, 226, 7, 0, 202, 255, 255, 243, 25, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 108, 255, 255, 255, 98, 0, 0, 70, 255, 255, 255, 161, 0, 0, 0, 0,
            ],
            &[
                0, 0, 19, 238, 255, 255, 215, 3, 0, 0, 0, 194, 255, 255, 254, 55, 0, 0, 0,
            ],
            &[
                0, 0, 150, 255, 255, 255, 84, 0, 0, 0, 0, 62, 255, 255, 255, 200, 1, 0, 0,
            ],
            &[
                0, 46, 252, 255, 255, 203, 0, 0, 0, 0, 0, 0, 186, 255, 255, 255, 94, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Y' / 0x59
        'Y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 28, 71, 71, 71, 51, 0, 0, 0, 0, 0, 0, 35, 71, 71, 71, 42, 0, 0,
            ],
            &[
                0, 28, 249, 255, 255, 242, 13, 0, 0, 0, 0, 0, 197, 255, 255, 255, 74, 0, 0,
            ],
            &[
                0, 0, 157, 255, 255, 255, 108, 0, 0, 0, 0, 48, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 37, 253, 255, 255, 215, 0, 0, 0, 0, 155, 255, 255, 255, 88, 0, 0, 0,
            ],
            &[
                0, 0, 0, 169, 255, 255, 255, 68, 0, 0, 17, 245, 255, 255, 219, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 48, 255, 255, 255, 176, 0, 0, 114, 255, 255, 255, 101, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 181, 255, 255, 252, 31, 1, 219, 255, 255, 229, 6, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 59, 255, 255, 255, 136, 72, 255, 255, 255, 114, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 193, 255, 255, 236, 187, 255, 255, 237, 12, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 72, 255, 255, 255, 255, 255, 255, 128, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 205, 255, 255, 255, 255, 244, 18, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 255, 141, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 2, 230, 255, 255, 255, 34, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Z' / 0x5a
        'Z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 11, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 18, 0, 0, 0,
            ],
            &[
                0, 0, 0, 42, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 65, 0, 0, 0,
            ],
            &[
                0, 0, 0, 42, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 65, 0, 0, 0,
            ],
            &[
                0, 0, 0, 42, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 11, 11, 11, 11, 11, 11, 107, 255, 255, 255, 150, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 14, 232, 255, 255, 237, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 139, 255, 255, 255, 105, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 37, 249, 255, 255, 207, 3, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 180, 255, 255, 255, 60, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 72, 255, 255, 255, 165, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 5, 215, 255, 255, 243, 27, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 113, 255, 255, 255, 120, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 21, 240, 255, 255, 218, 6, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 154, 255, 255, 255, 74, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 49, 253, 255, 255, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 194, 255, 255, 248, 36, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 88, 255, 255, 255, 197, 95, 95, 95, 95, 95, 95, 95, 29, 0, 0, 0,
            ],
            &[
                0, 0, 0, 205, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '[' / 0x5b
        '[' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 94, 119, 119, 119, 119, 119, 82, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 255, 255, 255, 176, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 255, 255, 255, 176, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 175, 73, 73, 50, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 188, 104, 104, 71, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 255, 255, 255, 176, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 202, 255, 255, 255, 255, 255, 176, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 75, 95, 95, 95, 95, 95, 66, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '\' / 0x5c
        '\\' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 98, 98, 98, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 7, 238, 255, 255, 105, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 149, 255, 255, 203, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 52, 255, 255, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 211, 255, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 114, 255, 255, 234, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 22, 250, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 175, 255, 255, 178, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 78, 255, 255, 251, 24, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 4, 232, 255, 255, 117, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 140, 255, 255, 214, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 43, 255, 255, 255, 56, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 201, 255, 255, 154, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 104, 255, 255, 241, 10, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 16, 247, 255, 255, 93, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 166, 255, 255, 190, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 69, 255, 255, 254, 33, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 225, 255, 255, 129, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 131, 255, 255, 225, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 35, 254, 255, 255, 68, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 25, 27, 27, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ']' / 0x5d
        ']' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 82, 167, 167, 167, 167, 167, 166, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 125, 255, 255, 255, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 125, 255, 255, 255, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 15, 31, 31, 112, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 71, 145, 145, 185, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 125, 255, 255, 255, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 125, 255, 255, 255, 255, 255, 253, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 23, 47, 47, 47, 47, 47, 46, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '^' / 0x5e
        '^' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 84, 77, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 111, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 7, 228, 255, 255, 198, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 103, 255, 255, 255, 255, 75, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 4, 223, 255, 255, 255, 255, 206, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 95, 255, 255, 186, 209, 255, 255, 84, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 217, 255, 255, 63, 80, 255, 255, 214, 2, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 87, 255, 255, 198, 0, 0, 203, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 210, 255, 255, 78, 0, 0, 72, 255, 255, 221, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 79, 255, 255, 211, 1, 0, 0, 0, 197, 255, 255, 102, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 203, 255, 255, 92, 0, 0, 0, 0, 65, 255, 255, 227, 7, 0, 0, 0,
            ],
            &[
                0, 0, 72, 255, 255, 223, 4, 0, 0, 0, 0, 0, 190, 255, 255, 111, 0, 0, 0,
            ],
            &[
                0, 0, 151, 209, 209, 96, 0, 0, 0, 0, 0, 0, 57, 209, 209, 186, 3, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '_' / 0x5f
        '_' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 9, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
                51, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
                51, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
                51, 0,
            ],
            &[
                0, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 2, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '`' / 0x60
        '`' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 35, 173, 174, 174, 173, 26, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 111, 255, 255, 255, 170, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 253, 255, 255, 84, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 88, 249, 255, 236, 27, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 59, 163, 163, 92, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'a' / 0x61
        'a' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 63, 143, 200, 237, 252, 250, 230, 183, 101, 6, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 214, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 200, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 178, 182, 107, 65, 39, 49, 109, 242, 255, 255, 255, 52, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 255, 255, 98, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 4, 23, 31, 38, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 50, 160, 225, 253, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 112, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 40, 254, 255, 255, 254, 158, 94, 60, 50, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 127, 255, 255, 255, 118, 0, 0, 0, 0, 78, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 255, 55, 0, 0, 0, 0, 141, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 152, 5, 0, 10, 107, 252, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 85, 255, 255, 255, 255, 239, 218, 252, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 2, 203, 255, 255, 255, 255, 255, 255, 249, 105, 248, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 28, 185, 255, 255, 255, 255, 189, 59, 0, 176, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 23, 59, 54, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'b' / 0x62
        'b' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 68, 133, 133, 133, 42, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 69, 25, 148, 224, 251, 234, 170, 44, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 95, 237, 255, 255, 255, 255, 255, 246, 66, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 245, 255, 255, 255, 255, 255, 255, 255, 232, 11, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 214, 95, 76, 149, 255, 255, 255, 255, 110, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 237, 14, 0, 0, 0, 136, 255, 255, 255, 191, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 144, 0, 0, 0, 0, 27, 255, 255, 255, 241, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 107, 0, 0, 0, 0, 0, 243, 255, 255, 255, 16, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 87, 0, 0, 0, 0, 0, 223, 255, 255, 255, 27, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 105, 0, 0, 0, 0, 0, 240, 255, 255, 255, 17, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 138, 0, 0, 0, 0, 19, 255, 255, 255, 246, 2, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 228, 4, 0, 0, 0, 112, 255, 255, 255, 201, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 175, 34, 11, 88, 245, 255, 255, 255, 125, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 245, 24, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 244, 154, 255, 255, 255, 255, 255, 255, 255, 99, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 163, 0, 96, 223, 255, 255, 255, 224, 86, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 38, 62, 41, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'c' / 0x63
        'c' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 58, 152, 213, 239, 251, 232, 207, 156, 74, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 152, 255, 255, 255, 255, 255, 255, 255, 255, 210, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 255, 255, 255, 255, 255, 255, 255, 255, 117, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 61, 255, 255, 255, 255, 208, 112, 85, 103, 155, 225, 26, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 153, 255, 255, 255, 187, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 226, 255, 255, 255, 52, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 2, 252, 255, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 20, 255, 255, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 13, 255, 255, 255, 236, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 246, 255, 255, 255, 28, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 199, 255, 255, 255, 139, 0, 0, 0, 0, 0, 0, 17, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 118, 255, 255, 255, 255, 147, 47, 20, 46, 91, 183, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 15, 226, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 57, 233, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 25, 149, 233, 255, 255, 255, 255, 248, 180, 55, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 3, 30, 53, 59, 35, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'd' / 0x64
        'd' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 15, 133, 133, 133, 95, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 30, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 30, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 30, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 30, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 24, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 13, 124, 211, 245, 235, 190, 73, 7, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 19, 212, 255, 255, 255, 255, 255, 255, 115, 233, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 179, 255, 255, 255, 255, 255, 255, 255, 254, 249, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 51, 255, 255, 255, 255, 199, 97, 70, 131, 249, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 130, 255, 255, 255, 206, 8, 0, 0, 0, 118, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 97, 0, 0, 0, 0, 18, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 45, 0, 0, 0, 0, 0, 229, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 230, 255, 255, 255, 21, 0, 0, 0, 0, 0, 213, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 223, 255, 255, 255, 30, 0, 0, 0, 0, 0, 221, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 59, 0, 0, 0, 0, 2, 248, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 255, 148, 0, 0, 0, 0, 85, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 83, 255, 255, 255, 251, 114, 21, 12, 80, 234, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 6, 224, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 69, 252, 255, 255, 255, 255, 255, 255, 106, 224, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 67, 215, 255, 255, 255, 222, 90, 0, 119, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 37, 61, 34, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'e' / 0x65
        'e' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 21, 120, 203, 235, 250, 226, 178, 81, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 64, 235, 255, 255, 255, 255, 255, 255, 255, 174, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 43, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 197, 255, 255, 255, 168, 39, 5, 52, 219, 255, 255, 255, 49, 0, 0, 0,
            ],
            &[
                0, 0, 38, 255, 255, 255, 192, 2, 0, 0, 0, 29, 246, 255, 255, 137, 0, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 255, 97, 0, 0, 0, 0, 0, 186, 255, 255, 184, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 238, 232, 232, 232, 232, 232, 248, 255, 255, 209, 0, 0, 0,
            ],
            &[
                0, 0, 159, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 206, 195, 195, 195, 195, 195, 195, 195, 195, 165, 0, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 255, 199, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 198, 255, 255, 255, 206, 81, 15, 0, 9, 38, 95, 171, 94, 0, 0, 0,
            ],
            &[
                0, 0, 0, 42, 244, 255, 255, 255, 255, 255, 247, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 68, 231, 255, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 132, 218, 255, 255, 255, 255, 255, 230, 152, 32, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 45, 63, 45, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'f' / 0x66
        'f' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 48, 119, 150, 162, 146, 123, 73, 17, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 7, 170, 255, 255, 255, 255, 255, 255, 255, 123, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 147, 255, 255, 255, 255, 255, 255, 255, 255, 38, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 10, 244, 255, 255, 255, 248, 191, 177, 205, 203, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 43, 255, 255, 255, 242, 38, 0, 0, 0, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 62, 255, 255, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 18, 70, 123, 191, 255, 255, 255, 233, 189, 189, 189, 189, 79, 0, 0, 0,
            ],
            &[
                0, 0, 39, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 41, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 20, 127, 127, 127, 159, 255, 255, 255, 212, 127, 127, 127, 127, 53, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 170, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 170, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 170, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 170, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 170, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 170, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 170, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 170, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 170, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 170, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 170, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'g' / 0x67
        'g' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 20, 149, 230, 255, 241, 174, 46, 0, 78, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 226, 255, 255, 255, 255, 255, 248, 71, 201, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 190, 255, 255, 255, 255, 255, 255, 255, 248, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 59, 255, 255, 255, 255, 182, 87, 72, 149, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 192, 3, 0, 0, 0, 147, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 86, 0, 0, 0, 0, 48, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 219, 255, 255, 255, 40, 0, 0, 0, 0, 6, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 231, 255, 255, 255, 19, 0, 0, 0, 0, 0, 248, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 223, 255, 255, 255, 34, 0, 0, 0, 0, 2, 251, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 201, 255, 255, 255, 68, 0, 0, 0, 0, 25, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 155, 255, 255, 255, 162, 0, 0, 0, 0, 110, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 79, 255, 255, 255, 254, 126, 25, 13, 89, 243, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 5, 217, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 58, 246, 255, 255, 255, 255, 255, 255, 166, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 55, 204, 255, 255, 255, 235, 109, 53, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 31, 61, 47, 4, 0, 70, 255, 255, 255, 169, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 122, 255, 255, 255, 142, 0, 0, 0,
            ],
            &[
                0, 0, 0, 172, 106, 34, 0, 0, 0, 0, 77, 242, 255, 255, 255, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 228, 255, 255, 239, 210, 209, 240, 255, 255, 255, 255, 222, 4, 0, 0, 0,
            ],
            &[
                0, 0, 0, 228, 255, 255, 255, 255, 255, 255, 255, 255, 255, 244, 48, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 254, 185, 36, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 46, 100, 126, 146, 147, 126, 96, 25, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'h' / 0x68
        'h' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 68, 133, 133, 133, 53, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 97, 17, 139, 217, 248, 243, 196, 94, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 104, 214, 255, 255, 255, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 245, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 223, 100, 68, 120, 243, 255, 255, 255, 157, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 249, 34, 0, 0, 0, 109, 255, 255, 255, 201, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 177, 0, 0, 0, 0, 41, 255, 255, 255, 216, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 128, 0, 0, 0, 0, 18, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 109, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'i' / 0x69
        'i' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 39, 192, 234, 201, 54, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 255, 202, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 205, 255, 255, 255, 227, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 141, 255, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 5, 100, 140, 103, 7, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 123, 189, 189, 189, 189, 189, 189, 189, 85, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 12, 42, 70, 98, 183, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 17, 146, 255, 255, 255, 141, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 147, 189, 231, 255, 255, 255, 255, 255, 255, 254, 220, 173, 110, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'j' / 0x6a
        'j' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 67, 207, 234, 184, 28, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 225, 255, 255, 255, 153, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 251, 255, 255, 255, 178, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 189, 255, 255, 255, 109, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 115, 139, 84, 1, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 98, 189, 189, 189, 189, 189, 189, 189, 110, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 132, 255, 255, 255, 255, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 132, 255, 255, 255, 255, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 8, 38, 66, 94, 163, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 85, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 97, 255, 255, 255, 147, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 144, 255, 255, 255, 132, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 47, 6, 0, 0, 0, 57, 243, 255, 255, 255, 92, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 235, 251, 222, 205, 222, 255, 255, 255, 255, 248, 23, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 235, 255, 255, 255, 255, 255, 255, 255, 255, 131, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 235, 255, 255, 255, 255, 255, 255, 248, 129, 2, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 27, 81, 123, 142, 152, 136, 93, 21, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'k' / 0x6b
        'k' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 50, 133, 133, 133, 71, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 77, 189, 189, 189, 183, 28, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 48, 244, 255, 255, 254, 90, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 26, 227, 255, 255, 255, 119, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 10, 204, 255, 255, 255, 150, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 133, 1, 173, 255, 255, 255, 177, 3, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 121, 125, 255, 255, 255, 200, 10, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 161, 253, 255, 255, 255, 60, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 198, 2, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 251, 255, 255, 255, 116, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 183, 37, 232, 255, 255, 247, 39, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 170, 1, 0, 84, 255, 255, 255, 199, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 174, 255, 255, 255, 117, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 25, 239, 255, 255, 247, 40, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 99, 255, 255, 255, 199, 3, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 0, 188, 255, 255, 255, 118, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'l' / 0x6c
        'l' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 57, 133, 133, 133, 133, 133, 133, 133, 52, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 255, 255, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 26, 79, 132, 215, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 19, 157, 255, 255, 255, 129, 11, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 147, 190, 232, 255, 255, 255, 255, 255, 255, 254, 219, 172, 110, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'm' / 0x6d
        'm' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 36, 189, 189, 103, 13, 165, 241, 232, 109, 0, 42, 198, 248, 216, 61, 0, 0, 0,
            ],
            &[
                0, 48, 255, 255, 191, 148, 255, 255, 255, 254, 53, 222, 255, 255, 255, 224, 1, 0, 0,
            ],
            &[
                0, 48, 255, 255, 252, 252, 255, 255, 255, 255, 223, 255, 255, 255, 255, 255, 43, 0,
                0,
            ],
            &[
                0, 48, 255, 255, 255, 183, 76, 223, 255, 255, 255, 119, 112, 255, 255, 255, 77, 0,
                0,
            ],
            &[
                0, 48, 255, 255, 255, 68, 0, 159, 255, 255, 233, 0, 3, 254, 255, 255, 92, 0, 0,
            ],
            &[
                0, 48, 255, 255, 255, 49, 0, 153, 255, 255, 214, 0, 0, 252, 255, 255, 99, 0, 0,
            ],
            &[
                0, 48, 255, 255, 255, 47, 0, 153, 255, 255, 211, 0, 0, 252, 255, 255, 99, 0, 0,
            ],
            &[
                0, 48, 255, 255, 255, 47, 0, 153, 255, 255, 211, 0, 0, 252, 255, 255, 99, 0, 0,
            ],
            &[
                0, 48, 255, 255, 255, 47, 0, 153, 255, 255, 211, 0, 0, 252, 255, 255, 99, 0, 0,
            ],
            &[
                0, 48, 255, 255, 255, 47, 0, 153, 255, 255, 211, 0, 0, 252, 255, 255, 99, 0, 0,
            ],
            &[
                0, 48, 255, 255, 255, 47, 0, 153, 255, 255, 211, 0, 0, 252, 255, 255, 99, 0, 0,
            ],
            &[
                0, 48, 255, 255, 255, 47, 0, 153, 255, 255, 211, 0, 0, 252, 255, 255, 99, 0, 0,
            ],
            &[
                0, 48, 255, 255, 255, 47, 0, 153, 255, 255, 211, 0, 0, 252, 255, 255, 99, 0, 0,
            ],
            &[
                0, 48, 255, 255, 255, 47, 0, 153, 255, 255, 211, 0, 0, 252, 255, 255, 99, 0, 0,
            ],
            &[
                0, 48, 255, 255, 255, 47, 0, 153, 255, 255, 211, 0, 0, 252, 255, 255, 99, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'n' / 0x6e
        'n' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 97, 189, 189, 110, 0, 28, 144, 216, 248, 245, 200, 99, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 224, 42, 236, 255, 255, 255, 255, 255, 255, 156, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 244, 255, 255, 255, 255, 255, 255, 255, 255, 69, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 236, 114, 71, 125, 248, 255, 255, 255, 158, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 254, 54, 0, 0, 0, 118, 255, 255, 255, 202, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 197, 0, 0, 0, 0, 42, 255, 255, 255, 216, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 138, 0, 0, 0, 0, 18, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 115, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'o' / 0x6f
        'o' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 31, 134, 209, 239, 247, 219, 164, 62, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 247, 255, 255, 255, 255, 255, 255, 255, 151, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 82, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 233, 255, 255, 255, 182, 74, 51, 129, 253, 255, 255, 254, 57, 0, 0, 0,
            ],
            &[
                0, 0, 95, 255, 255, 255, 199, 4, 0, 0, 0, 135, 255, 255, 255, 154, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 91, 0, 0, 0, 0, 30, 255, 255, 255, 229, 0, 0, 0,
            ],
            &[
                0, 0, 206, 255, 255, 255, 42, 0, 0, 0, 0, 0, 237, 255, 255, 254, 4, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 18, 0, 0, 0, 0, 0, 221, 255, 255, 255, 24, 0, 0,
            ],
            &[
                0, 0, 220, 255, 255, 255, 27, 0, 0, 0, 0, 0, 231, 255, 255, 255, 14, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 68, 0, 0, 0, 0, 16, 254, 255, 255, 240, 0, 0, 0,
            ],
            &[
                0, 0, 132, 255, 255, 255, 160, 0, 0, 0, 0, 100, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 36, 253, 255, 255, 254, 116, 5, 0, 65, 238, 255, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 255, 255, 249, 241, 255, 255, 255, 255, 194, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 7, 174, 255, 255, 255, 255, 255, 255, 255, 255, 209, 25, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 97, 212, 255, 255, 255, 255, 228, 127, 11, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 55, 60, 35, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'p' / 0x70
        'p' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 97, 189, 189, 111, 0, 22, 138, 216, 249, 235, 171, 46, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 226, 37, 229, 255, 255, 255, 255, 255, 247, 66, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 242, 255, 255, 255, 255, 255, 255, 255, 232, 11, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 219, 99, 74, 145, 255, 255, 255, 255, 109, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 238, 19, 0, 0, 0, 145, 255, 255, 255, 190, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 142, 0, 0, 0, 0, 45, 255, 255, 255, 240, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 101, 0, 0, 0, 0, 6, 255, 255, 255, 255, 15, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 84, 0, 0, 0, 0, 0, 243, 255, 255, 255, 26, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 104, 0, 0, 0, 0, 3, 254, 255, 255, 255, 13, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 137, 0, 0, 0, 0, 29, 255, 255, 255, 248, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 228, 4, 0, 0, 0, 109, 255, 255, 255, 196, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 178, 36, 9, 75, 239, 255, 255, 255, 124, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 240, 19, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 175, 255, 255, 255, 255, 255, 255, 253, 91, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 64, 94, 218, 255, 255, 255, 217, 79, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 80, 0, 1, 33, 60, 34, 2, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 63, 122, 122, 122, 39, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'q' / 0x71
        'q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 19, 143, 223, 251, 236, 177, 63, 0, 69, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 226, 255, 255, 255, 255, 255, 254, 115, 160, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 193, 255, 255, 255, 255, 255, 255, 255, 254, 248, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 60, 255, 255, 255, 255, 192, 90, 81, 172, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 209, 7, 0, 0, 0, 170, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 190, 255, 255, 255, 107, 0, 0, 0, 0, 56, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 220, 255, 255, 255, 61, 0, 0, 0, 0, 12, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 231, 255, 255, 255, 40, 0, 0, 0, 0, 0, 243, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 224, 255, 255, 255, 52, 0, 0, 0, 0, 0, 247, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 201, 255, 255, 255, 84, 0, 0, 0, 0, 23, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 255, 171, 0, 0, 0, 0, 112, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 83, 255, 255, 255, 255, 125, 22, 15, 90, 244, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 6, 224, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 69, 252, 255, 255, 255, 255, 255, 255, 143, 253, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 68, 217, 255, 255, 255, 228, 107, 24, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 38, 62, 41, 2, 0, 30, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 30, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 30, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 30, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 30, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 30, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 14, 122, 122, 122, 87, 0, 0, 0,
            ],
        ]),
        // letter: 'r' / 0x72
        'r' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 31, 189, 189, 189, 189, 189, 185, 3, 0, 63, 185, 237, 252, 224, 112, 0, 0,
            ],
            &[
                0, 0, 41, 255, 255, 255, 255, 255, 255, 49, 97, 252, 255, 255, 255, 255, 111, 0, 0,
            ],
            &[
                0, 0, 41, 255, 255, 255, 255, 255, 255, 160, 249, 255, 255, 255, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 2, 35, 72, 118, 255, 255, 255, 255, 255, 255, 228, 196, 213, 216, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 255, 246, 88, 1, 0, 0, 4, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 255, 113, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 252, 16, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 228, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 48, 255, 255, 255, 227, 26, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 119, 185, 238, 255, 255, 255, 255, 255, 255, 229, 175, 92, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 183, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 183, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 's' / 0x73
        's' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 64, 164, 220, 245, 254, 242, 211, 165, 99, 17, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 148, 255, 255, 255, 255, 255, 255, 255, 255, 255, 213, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 79, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 119, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 158, 255, 255, 255, 190, 77, 61, 88, 123, 194, 245, 21, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 178, 255, 255, 255, 111, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 138, 255, 255, 255, 250, 152, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 234, 255, 255, 255, 255, 255, 210, 101, 9, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 39, 194, 255, 255, 255, 255, 255, 255, 234, 85, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 61, 178, 252, 255, 255, 255, 255, 254, 79, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 26, 127, 240, 255, 255, 255, 207, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 18, 0, 0, 0, 0, 0, 0, 47, 255, 255, 255, 250, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 212, 189, 113, 58, 20, 4, 27, 99, 255, 255, 255, 245, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 186, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 237, 48, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 116, 208, 252, 255, 255, 255, 255, 255, 235, 155, 34, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 10, 34, 53, 64, 52, 30, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 't' / 0x74
        't' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 30, 105, 105, 76, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 104, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 150, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 195, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 1, 42, 99, 155, 247, 255, 255, 237, 189, 189, 189, 189, 189, 48, 0, 0, 0,
            ],
            &[
                0, 0, 142, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 65, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 65, 0, 0, 0,
            ],
            &[
                0, 0, 72, 127, 127, 152, 255, 255, 255, 220, 127, 127, 127, 127, 127, 32, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 187, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 35, 255, 255, 255, 220, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 10, 252, 255, 255, 255, 148, 29, 6, 25, 64, 36, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 184, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 46, 245, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 45, 185, 250, 255, 255, 255, 255, 226, 72, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 14, 47, 64, 51, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'u' / 0x75
        'u' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 123, 189, 189, 189, 50, 0, 0, 0, 0, 37, 189, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 55, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 68, 0, 0, 0, 0, 69, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 83, 0, 0, 0, 0, 108, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 155, 255, 255, 255, 132, 0, 0, 0, 0, 192, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 122, 255, 255, 255, 237, 67, 5, 23, 142, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 46, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 157, 255, 255, 255, 255, 255, 255, 255, 130, 199, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 4, 134, 245, 255, 255, 255, 219, 96, 0, 109, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 7, 51, 59, 27, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'v' / 0x76
        'v' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 61, 189, 189, 189, 165, 0, 0, 0, 0, 0, 0, 111, 189, 189, 189, 98, 0, 0,
            ],
            &[
                0, 11, 243, 255, 255, 255, 37, 0, 0, 0, 0, 0, 220, 255, 255, 255, 50, 0, 0,
            ],
            &[
                0, 0, 160, 255, 255, 255, 119, 0, 0, 0, 0, 44, 255, 255, 255, 210, 0, 0, 0,
            ],
            &[
                0, 0, 65, 255, 255, 255, 200, 0, 0, 0, 0, 125, 255, 255, 255, 114, 0, 0, 0,
            ],
            &[
                0, 0, 1, 224, 255, 255, 254, 27, 0, 0, 0, 205, 255, 255, 251, 23, 0, 0, 0,
            ],
            &[
                0, 0, 0, 130, 255, 255, 255, 107, 0, 0, 30, 255, 255, 255, 179, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 36, 255, 255, 255, 189, 0, 0, 110, 255, 255, 255, 83, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 196, 255, 255, 251, 18, 0, 190, 255, 255, 237, 6, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 101, 255, 255, 255, 95, 18, 252, 255, 255, 148, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 15, 246, 255, 255, 168, 86, 255, 255, 255, 52, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 166, 255, 255, 232, 148, 255, 255, 213, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 71, 255, 255, 255, 220, 255, 255, 117, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 3, 229, 255, 255, 255, 255, 252, 25, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 137, 255, 255, 255, 255, 182, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 42, 255, 255, 255, 255, 86, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'w' / 0x77
        'w' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 164, 189, 189, 128, 0, 0, 0, 0, 0, 0, 0, 0, 75, 189, 189, 189, 17, 0,
            ],
            &[
                0, 190, 255, 255, 193, 0, 0, 0, 0, 0, 0, 0, 0, 121, 255, 255, 246, 1, 0,
            ],
            &[
                0, 154, 255, 255, 215, 0, 0, 90, 148, 148, 127, 0, 0, 144, 255, 255, 212, 0, 0,
            ],
            &[
                0, 118, 255, 255, 238, 0, 0, 195, 255, 255, 251, 8, 0, 166, 255, 255, 176, 0, 0,
            ],
            &[
                0, 82, 255, 255, 255, 6, 2, 243, 255, 255, 255, 55, 0, 189, 255, 255, 140, 0, 0,
            ],
            &[
                0, 45, 255, 255, 255, 28, 41, 255, 255, 255, 255, 106, 0, 212, 255, 255, 104, 0, 0,
            ],
            &[
                0, 10, 254, 255, 255, 50, 92, 255, 255, 236, 255, 156, 0, 234, 255, 255, 67, 0, 0,
            ],
            &[
                0, 0, 229, 255, 255, 73, 144, 255, 231, 179, 255, 207, 3, 253, 255, 255, 31, 0, 0,
            ],
            &[
                0, 0, 193, 255, 255, 97, 193, 255, 190, 138, 255, 250, 29, 255, 255, 249, 2, 0, 0,
            ],
            &[
                0, 0, 157, 255, 255, 121, 237, 255, 146, 95, 255, 255, 91, 255, 255, 215, 0, 0, 0,
            ],
            &[
                0, 0, 121, 255, 255, 163, 255, 255, 99, 49, 255, 255, 156, 255, 255, 179, 0, 0, 0,
            ],
            &[
                0, 0, 85, 255, 255, 225, 255, 255, 52, 7, 251, 255, 224, 255, 255, 143, 0, 0, 0,
            ],
            &[
                0, 0, 49, 255, 255, 255, 255, 252, 8, 0, 213, 255, 255, 255, 255, 107, 0, 0, 0,
            ],
            &[
                0, 0, 13, 255, 255, 255, 255, 214, 0, 0, 168, 255, 255, 255, 255, 71, 0, 0, 0,
            ],
            &[
                0, 0, 0, 233, 255, 255, 255, 167, 0, 0, 122, 255, 255, 255, 255, 35, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'x' / 0x78
        'x' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 8, 173, 189, 189, 189, 131, 0, 0, 0, 0, 49, 189, 189, 189, 189, 75, 0, 0,
            ],
            &[
                0, 0, 93, 255, 255, 255, 254, 52, 0, 0, 1, 200, 255, 255, 255, 198, 3, 0, 0,
            ],
            &[
                0, 0, 0, 173, 255, 255, 255, 196, 0, 0, 100, 255, 255, 255, 243, 34, 0, 0, 0,
            ],
            &[
                0, 0, 0, 20, 232, 255, 255, 255, 88, 19, 236, 255, 255, 255, 98, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 78, 255, 255, 255, 226, 162, 255, 255, 255, 174, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 158, 255, 255, 255, 255, 255, 255, 231, 20, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 14, 224, 255, 255, 255, 255, 255, 72, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 107, 255, 255, 255, 255, 193, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 226, 255, 255, 255, 255, 254, 68, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 164, 255, 255, 255, 255, 255, 255, 225, 14, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 85, 255, 255, 255, 231, 168, 255, 255, 255, 157, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 26, 237, 255, 255, 255, 91, 23, 242, 255, 255, 255, 75, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 184, 255, 255, 255, 191, 0, 0, 119, 255, 255, 255, 230, 17, 0, 0, 0,
            ],
            &[
                0, 0, 107, 255, 255, 255, 250, 42, 0, 0, 7, 221, 255, 255, 255, 165, 0, 0, 0,
            ],
            &[
                0, 38, 245, 255, 255, 255, 138, 0, 0, 0, 0, 82, 255, 255, 255, 255, 82, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'y' / 0x79
        'y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 59, 189, 189, 189, 175, 1, 0, 0, 0, 0, 0, 125, 189, 189, 189, 99, 0, 0,
            ],
            &[
                0, 8, 238, 255, 255, 255, 56, 0, 0, 0, 0, 1, 233, 255, 255, 255, 53, 0, 0,
            ],
            &[
                0, 0, 146, 255, 255, 255, 142, 0, 0, 0, 0, 55, 255, 255, 255, 216, 0, 0, 0,
            ],
            &[
                0, 0, 44, 255, 255, 255, 226, 1, 0, 0, 0, 131, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[
                0, 0, 0, 198, 255, 255, 255, 57, 0, 0, 0, 206, 255, 255, 254, 31, 0, 0, 0,
            ],
            &[
                0, 0, 0, 97, 255, 255, 255, 143, 0, 0, 27, 255, 255, 255, 193, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 241, 255, 255, 227, 1, 0, 102, 255, 255, 255, 100, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 150, 255, 255, 255, 58, 0, 178, 255, 255, 247, 15, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 48, 255, 255, 255, 143, 8, 245, 255, 255, 169, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 203, 255, 255, 220, 70, 255, 255, 255, 76, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 101, 255, 255, 255, 157, 255, 255, 234, 4, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 12, 243, 255, 255, 245, 255, 255, 146, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 255, 255, 53, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 53, 255, 255, 255, 255, 216, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 208, 255, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 1, 220, 255, 255, 254, 32, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 83, 255, 255, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 55, 232, 255, 255, 255, 70, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 95, 210, 228, 255, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 255, 230, 29, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 210, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 137, 152, 132, 74, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'z' / 0x7a
        'z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 159, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 51, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 50, 0, 0, 0,
            ],
            &[
                0, 0, 0, 72, 85, 85, 85, 85, 85, 98, 245, 255, 255, 255, 174, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 1, 172, 255, 255, 255, 209, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 124, 255, 255, 255, 234, 31, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 254, 255, 255, 250, 61, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 40, 241, 255, 255, 255, 100, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 217, 255, 255, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 181, 255, 255, 255, 186, 3, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 135, 255, 255, 255, 218, 17, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 86, 255, 255, 255, 246, 60, 19, 19, 19, 19, 19, 19, 8, 0, 0, 0,
            ],
            &[
                0, 0, 33, 245, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 62, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 62, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '{' / 0x7b
        '{' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 19, 67, 85, 47, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 28, 183, 255, 255, 255, 128, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 199, 255, 255, 255, 255, 128, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 30, 255, 255, 255, 251, 168, 68, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 58, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 43, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 67, 255, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 110, 255, 255, 255, 29, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 88, 238, 255, 255, 236, 2, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 76, 255, 255, 255, 255, 247, 86, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 77, 255, 255, 255, 204, 51, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 76, 255, 255, 255, 255, 249, 91, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 16, 85, 237, 255, 255, 238, 3, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 109, 255, 255, 255, 31, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 66, 255, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 63, 255, 255, 255, 42, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 58, 255, 255, 255, 96, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 30, 255, 255, 255, 248, 172, 70, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 198, 255, 255, 255, 255, 128, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 27, 182, 255, 255, 255, 128, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 18, 64, 82, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '|' / 0x7c
        '|' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 54, 133, 133, 77, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 50, 122, 122, 71, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: '}' / 0x7d
        '}' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 88, 73, 30, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 77, 255, 255, 255, 209, 56, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 77, 255, 255, 255, 255, 238, 12, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 40, 157, 236, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 43, 255, 255, 255, 109, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 246, 255, 255, 115, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 243, 255, 255, 115, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 243, 255, 255, 115, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 243, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 235, 255, 255, 162, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 189, 255, 255, 253, 116, 26, 2, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 52, 233, 255, 255, 255, 255, 128, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 29, 175, 255, 255, 255, 128, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 51, 233, 255, 255, 255, 255, 127, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 188, 255, 255, 252, 112, 23, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 234, 255, 255, 160, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 243, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 243, 255, 255, 115, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 243, 255, 255, 115, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 1, 247, 255, 255, 115, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 51, 255, 255, 255, 109, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 41, 160, 240, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 77, 255, 255, 255, 255, 236, 12, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 77, 255, 255, 255, 205, 52, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 28, 85, 70, 26, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '~' / 0x7e
        '~' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 7, 77, 97, 50, 0, 0, 0, 0, 0, 70, 107, 36, 0, 0, 0,
            ],
            &[
                0, 0, 0, 22, 215, 255, 255, 255, 173, 13, 0, 0, 0, 186, 255, 66, 0, 0, 0,
            ],
            &[
                0, 0, 0, 149, 255, 223, 161, 240, 255, 209, 28, 0, 4, 237, 255, 30, 0, 0, 0,
            ],
            &[
                0, 0, 3, 235, 254, 34, 0, 41, 223, 255, 228, 103, 160, 255, 201, 0, 0, 0, 0,
            ],
            &[
                0, 0, 27, 255, 217, 0, 0, 0, 22, 194, 255, 255, 255, 247, 65, 0, 0, 0, 0,
            ],
            &[
                0, 0, 32, 174, 136, 0, 0, 0, 0, 2, 90, 158, 154, 57, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ' ' / 0xa0
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¡' / 0xa1
        '¡' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 22, 116, 125, 47, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 12, 226, 255, 255, 251, 49, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 69, 255, 255, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 55, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 166, 255, 255, 212, 19, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 27, 36, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 146, 226, 226, 185, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 176, 255, 255, 220, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 187, 255, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 198, 255, 255, 243, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 210, 255, 255, 253, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 221, 255, 255, 255, 9, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 232, 255, 255, 255, 20, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 243, 255, 255, 255, 32, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 1, 254, 255, 255, 255, 43, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 10, 255, 255, 255, 255, 54, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 21, 255, 255, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 33, 255, 255, 255, 255, 77, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 44, 254, 254, 254, 254, 88, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¢' / 0xa2
        '¢' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 129, 140, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 237, 255, 59, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 9, 60, 243, 255, 141, 76, 35, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 24, 150, 248, 255, 255, 255, 255, 255, 255, 207, 33, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 41, 226, 255, 255, 255, 255, 255, 255, 255, 255, 236, 5, 0, 0, 0,
            ],
            &[
                0, 0, 0, 5, 211, 255, 255, 255, 253, 200, 170, 186, 224, 255, 151, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 99, 255, 255, 255, 214, 39, 0, 0, 0, 0, 48, 35, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 182, 255, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 238, 255, 255, 209, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 9, 255, 255, 255, 172, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 19, 255, 255, 255, 164, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 1, 249, 255, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 210, 255, 255, 247, 16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 134, 255, 255, 255, 167, 5, 0, 0, 0, 0, 0, 52, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 31, 245, 255, 255, 255, 213, 126, 96, 114, 146, 215, 238, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 105, 254, 255, 255, 255, 255, 255, 255, 255, 255, 238, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 90, 231, 255, 255, 255, 255, 255, 255, 255, 223, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 8, 86, 142, 249, 255, 195, 139, 81, 6, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 237, 255, 59, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 237, 255, 59, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 69, 16, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '£' / 0xa3
        '£' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 40, 106, 133, 130, 102, 41, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 20, 175, 255, 255, 255, 255, 255, 255, 202, 39, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 3, 203, 255, 255, 255, 255, 255, 255, 255, 244, 12, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 87, 255, 255, 255, 255, 215, 189, 223, 255, 161, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 158, 255, 255, 255, 134, 0, 0, 0, 51, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 186, 255, 255, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 194, 255, 255, 255, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 194, 255, 255, 255, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 55, 66, 210, 255, 255, 255, 69, 66, 66, 66, 27, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 107, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 107, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 118, 140, 228, 255, 255, 255, 142, 140, 140, 140, 59, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 194, 255, 255, 255, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 200, 255, 255, 251, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 236, 255, 255, 213, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 80, 255, 255, 255, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 91, 241, 255, 255, 186, 82, 82, 82, 82, 82, 82, 82, 47, 0, 0, 0,
            ],
            &[
                0, 0, 0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0,
            ],
            &[
                0, 0, 0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0,
            ],
            &[
                0, 0, 0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¤' / 0xa4
        '¤' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 0, 0, 0, 0, 0, 0, 0, 0, 55, 3, 0, 0, 0, 0],
            &[
                0, 0, 0, 128, 255, 116, 0, 65, 114, 118, 74, 2, 83, 251, 171, 5, 0, 0, 0,
            ],
            &[
                0, 0, 36, 247, 255, 255, 222, 255, 255, 255, 255, 221, 251, 255, 253, 70, 0, 0, 0,
            ],
            &[
                0, 0, 0, 68, 246, 255, 255, 255, 255, 255, 255, 255, 255, 253, 95, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 191, 255, 255, 192, 95, 84, 164, 255, 255, 231, 6, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 39, 255, 255, 205, 8, 0, 0, 0, 157, 255, 255, 84, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 87, 255, 255, 123, 0, 0, 0, 0, 71, 255, 255, 130, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 84, 255, 255, 130, 0, 0, 0, 0, 78, 255, 255, 128, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 35, 254, 255, 215, 16, 0, 0, 3, 177, 255, 255, 79, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 188, 255, 255, 213, 122, 114, 193, 255, 255, 229, 3, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 82, 250, 255, 255, 255, 255, 255, 255, 255, 255, 255, 109, 0, 0, 0, 0,
            ],
            &[
                0, 0, 31, 249, 255, 252, 200, 255, 255, 255, 255, 204, 247, 255, 255, 57, 0, 0, 0,
            ],
            &[
                0, 0, 0, 97, 250, 89, 0, 40, 85, 88, 43, 0, 69, 246, 136, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 32, 0, 0, 0, 0, 0, 0, 0, 0, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¥' / 0xa5
        '¥' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 18, 71, 71, 71, 50, 0, 0, 0, 0, 0, 0, 35, 71, 71, 71, 32, 0, 0,
            ],
            &[
                0, 8, 230, 255, 255, 240, 13, 0, 0, 0, 0, 0, 202, 255, 255, 251, 37, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 111, 0, 0, 0, 0, 60, 255, 255, 255, 158, 0, 0, 0,
            ],
            &[
                0, 0, 8, 230, 255, 255, 222, 3, 0, 0, 0, 174, 255, 255, 250, 33, 0, 0, 0,
            ],
            &[
                0, 0, 0, 111, 255, 255, 255, 84, 0, 0, 35, 253, 255, 255, 154, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 8, 230, 255, 255, 198, 0, 0, 147, 255, 255, 249, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 111, 255, 255, 255, 56, 17, 244, 255, 255, 150, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 8, 230, 255, 255, 170, 119, 255, 255, 248, 28, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 111, 255, 255, 252, 234, 255, 255, 146, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 8, 230, 255, 255, 255, 255, 246, 25, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 80, 128, 128, 200, 255, 255, 255, 255, 215, 128, 128, 106, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 159, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 210, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 134, 216, 216, 216, 245, 255, 255, 253, 216, 216, 216, 178, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 188, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 50, 81, 81, 81, 209, 255, 255, 244, 81, 81, 81, 67, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 159, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 210, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 159, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 210, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 188, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 188, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 188, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¦' / 0xa6
        '¦' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 49, 119, 119, 69, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 23, 57, 57, 33, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 53, 130, 130, 76, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 22, 53, 53, 31, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '§' / 0xa7
        '§' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 26, 108, 139, 162, 151, 114, 47, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 8, 154, 250, 255, 255, 255, 255, 255, 255, 218, 21, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 148, 255, 255, 255, 241, 220, 250, 255, 255, 194, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 242, 255, 255, 192, 5, 0, 4, 58, 164, 75, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 247, 255, 255, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 174, 255, 255, 255, 154, 30, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 24, 216, 255, 255, 255, 251, 168, 52, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 144, 255, 255, 255, 255, 255, 254, 156, 9, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 134, 255, 255, 215, 135, 236, 255, 255, 255, 180, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 11, 247, 255, 255, 51, 0, 10, 162, 255, 255, 255, 60, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 40, 255, 255, 255, 40, 0, 0, 2, 237, 255, 255, 105, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 16, 250, 255, 255, 205, 28, 0, 1, 234, 255, 255, 85, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 145, 255, 255, 255, 246, 138, 129, 255, 255, 233, 12, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 3, 134, 249, 255, 255, 255, 255, 255, 235, 57, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 32, 157, 251, 255, 255, 255, 239, 59, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 33, 154, 254, 255, 255, 229, 6, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 115, 255, 255, 255, 61, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 27, 55, 0, 0, 0, 0, 0, 64, 255, 255, 255, 67, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 56, 255, 207, 130, 81, 61, 92, 214, 255, 255, 245, 14, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 56, 255, 255, 255, 255, 255, 255, 255, 255, 254, 98, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 14, 138, 226, 255, 255, 255, 255, 254, 195, 67, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 26, 56, 65, 53, 15, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¨' / 0xa8
        '¨' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 7, 89, 88, 7, 0, 0, 0, 64, 100, 29, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 168, 255, 255, 174, 0, 0, 92, 255, 255, 237, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 226, 255, 255, 236, 0, 0, 151, 255, 255, 255, 62, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 175, 255, 255, 181, 0, 0, 97, 255, 255, 241, 21, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 14, 111, 110, 14, 0, 0, 0, 83, 123, 42, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '©' / 0xa9
        '©' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 7, 89, 155, 185, 191, 161, 110, 17, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 76, 233, 255, 255, 255, 253, 255, 255, 246, 113, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 103, 254, 253, 150, 49, 9, 12, 36, 122, 241, 255, 147, 0, 0, 0, 0,
            ],
            &[
                0, 0, 47, 248, 250, 75, 18, 161, 232, 255, 237, 148, 35, 227, 255, 83, 0, 0, 0,
            ],
            &[
                0, 0, 168, 255, 129, 6, 217, 255, 240, 194, 230, 160, 0, 64, 255, 213, 0, 0, 0,
            ],
            &[
                0, 12, 248, 250, 19, 109, 255, 228, 25, 0, 3, 18, 0, 0, 206, 255, 50, 0, 0,
            ],
            &[
                0, 45, 255, 202, 0, 169, 255, 129, 0, 0, 0, 0, 0, 0, 149, 255, 93, 0, 0,
            ],
            &[
                0, 68, 255, 179, 0, 199, 255, 96, 0, 0, 0, 0, 0, 0, 128, 255, 119, 0, 0,
            ],
            &[
                0, 49, 255, 197, 0, 174, 255, 114, 0, 0, 0, 0, 0, 0, 149, 255, 102, 0, 0,
            ],
            &[
                0, 15, 251, 243, 10, 122, 255, 206, 10, 0, 0, 6, 0, 0, 206, 255, 68, 0, 0,
            ],
            &[
                0, 0, 180, 255, 104, 13, 232, 255, 228, 180, 204, 167, 0, 59, 255, 233, 5, 0, 0,
            ],
            &[
                0, 0, 62, 254, 241, 45, 32, 186, 248, 255, 249, 130, 25, 221, 255, 122, 0, 0, 0,
            ],
            &[
                0, 0, 0, 138, 255, 242, 113, 15, 3, 27, 16, 97, 231, 255, 186, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 120, 252, 255, 254, 230, 226, 252, 255, 255, 150, 8, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 29, 132, 189, 219, 222, 194, 139, 43, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ª' / 0xaa
        'ª' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 1, 57, 95, 118, 90, 28, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 37, 223, 255, 255, 255, 255, 245, 90, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 2, 214, 204, 148, 158, 238, 255, 240, 14, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 12, 0, 0, 0, 104, 255, 255, 58, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 72, 153, 195, 216, 234, 255, 255, 79, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 140, 255, 255, 227, 185, 190, 255, 255, 79, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 11, 252, 255, 184, 2, 0, 83, 255, 255, 79, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 20, 255, 255, 163, 1, 23, 203, 255, 255, 79, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 211, 255, 255, 241, 253, 242, 225, 255, 79, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 39, 187, 247, 249, 187, 47, 121, 223, 69, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '«' / 0xab
        '«' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 71, 0, 0, 0, 0, 38, 45, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 167, 255, 194, 57, 0, 11, 213, 253, 164, 32, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 105, 255, 255, 255, 104, 0, 162, 255, 255, 251, 52, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 51, 248, 255, 255, 181, 0, 100, 255, 255, 255, 128, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 16, 221, 255, 255, 236, 24, 47, 247, 255, 255, 203, 4, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 175, 255, 255, 255, 83, 14, 218, 255, 255, 246, 40, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 255, 190, 0, 133, 255, 255, 255, 139, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 8, 207, 255, 255, 251, 52, 33, 239, 255, 255, 232, 20, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 36, 241, 255, 255, 216, 10, 79, 255, 255, 255, 173, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 140, 255, 255, 255, 95, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 145, 255, 255, 255, 72, 4, 197, 255, 255, 241, 31, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 6, 201, 255, 231, 89, 0, 28, 235, 255, 209, 54, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 31, 122, 11, 0, 0, 0, 68, 94, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¬' / 0xac
        '¬' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 9, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 10, 0, 0, 0,
            ],
            &[
                0, 0, 25, 230, 230, 230, 230, 230, 230, 230, 230, 230, 252, 255, 255, 10, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 217, 255, 255, 10, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 217, 255, 255, 10, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 217, 255, 255, 10, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 217, 255, 255, 10, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 165, 194, 194, 7, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '®' / 0xae
        '®' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 6, 88, 155, 185, 190, 161, 108, 15, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 73, 232, 255, 255, 255, 253, 255, 255, 245, 108, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 98, 254, 253, 151, 49, 9, 4, 36, 122, 241, 255, 141, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 247, 252, 79, 166, 203, 203, 185, 137, 25, 35, 227, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 163, 255, 136, 0, 208, 255, 193, 228, 255, 211, 1, 65, 255, 208, 0, 0, 0,
            ],
            &[
                0, 8, 246, 252, 23, 0, 208, 255, 32, 13, 249, 255, 30, 0, 210, 255, 45, 0, 0,
            ],
            &[
                0, 39, 255, 209, 0, 0, 208, 255, 32, 62, 255, 248, 8, 0, 154, 255, 87, 0, 0,
            ],
            &[
                0, 62, 255, 186, 0, 0, 208, 255, 243, 255, 255, 131, 0, 0, 134, 255, 112, 0, 0,
            ],
            &[
                0, 41, 255, 204, 0, 0, 208, 255, 131, 222, 250, 23, 0, 0, 156, 255, 95, 0, 0,
            ],
            &[
                0, 10, 247, 246, 13, 0, 208, 255, 32, 120, 255, 128, 0, 0, 213, 255, 61, 0, 0,
            ],
            &[
                0, 0, 171, 255, 110, 0, 208, 255, 32, 26, 252, 233, 7, 66, 255, 228, 3, 0, 0,
            ],
            &[
                0, 0, 53, 252, 243, 50, 147, 181, 23, 0, 138, 181, 87, 224, 255, 116, 0, 0, 0,
            ],
            &[
                0, 0, 0, 125, 255, 245, 119, 20, 0, 0, 14, 104, 234, 255, 183, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 108, 249, 255, 255, 237, 233, 255, 255, 255, 147, 7, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 22, 123, 182, 212, 216, 188, 134, 39, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¯' / 0xaf
        '¯' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 116, 119, 119, 119, 119, 119, 119, 119, 24, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 250, 255, 255, 255, 255, 255, 255, 255, 52, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 250, 255, 255, 255, 255, 255, 255, 255, 52, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 51, 52, 52, 52, 52, 52, 52, 52, 10, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '°' / 0xb0
        '°' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 54, 121, 127, 75, 3, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 5, 164, 255, 255, 255, 255, 202, 25, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 138, 255, 255, 225, 212, 255, 255, 195, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 9, 244, 255, 171, 3, 0, 117, 255, 255, 56, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 40, 255, 255, 73, 0, 0, 15, 255, 255, 98, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 23, 255, 255, 112, 0, 0, 52, 255, 255, 82, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 198, 255, 244, 128, 112, 223, 255, 245, 16, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 46, 241, 255, 255, 255, 255, 254, 103, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 37, 163, 226, 233, 188, 74, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '±' / 0xb1
        '±' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 72, 237, 237, 139, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 13, 245, 245, 245, 245, 248, 255, 255, 251, 245, 245, 245, 245, 76, 0, 0, 0,
            ],
            &[
                0, 0, 14, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 12, 230, 230, 230, 230, 238, 255, 255, 245, 230, 230, 230, 230, 71, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 63, 208, 208, 122, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 25, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 70, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '²' / 0xb2
        '²' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 3, 100, 202, 251, 253, 221, 109, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 170, 255, 255, 255, 255, 255, 255, 96, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 79, 255, 178, 100, 153, 255, 255, 183, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 60, 0, 0, 5, 247, 255, 210, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 38, 255, 255, 175, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 4, 187, 255, 252, 64, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 5, 171, 255, 255, 117, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 8, 181, 255, 255, 128, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 10, 188, 255, 255, 117, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 178, 255, 255, 226, 148, 148, 148, 148, 62, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 215, 255, 255, 255, 255, 255, 255, 255, 107, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 165, 196, 196, 196, 196, 196, 196, 196, 82, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '³' / 0xb3
        '³' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 11, 120, 209, 250, 247, 206, 88, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 181, 255, 255, 255, 255, 255, 255, 87, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 55, 241, 129, 70, 123, 255, 255, 180, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 12, 0, 0, 2, 241, 255, 182, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 36, 47, 131, 255, 252, 73, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 250, 255, 255, 237, 66, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 234, 245, 255, 255, 227, 57, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 35, 237, 255, 224, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 39, 0, 0, 0, 0, 178, 255, 255, 26, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 228, 181, 117, 85, 128, 249, 255, 245, 4, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 121, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 67, 169, 218, 243, 226, 187, 80, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '´' / 0xb4
        '´' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 89, 174, 174, 174, 143, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 24, 239, 255, 255, 232, 38, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 180, 255, 255, 225, 37, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 105, 255, 255, 210, 28, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 152, 163, 146, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'µ' / 0xb5
        'µ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 123, 189, 189, 189, 50, 0, 0, 0, 0, 37, 189, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 55, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 187, 255, 255, 255, 76, 0, 0, 0, 0, 69, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 255, 255, 255, 255, 151, 0, 0, 0, 0, 108, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 255, 255, 255, 255, 199, 0, 0, 0, 0, 192, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 255, 255, 255, 255, 255, 67, 5, 23, 142, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 212, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 255, 130, 199, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 255, 255, 255, 219, 96, 0, 109, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 119, 59, 27, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 79, 122, 122, 122, 32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: '¶' / 0xb6
        '¶' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 11, 79, 111, 131, 133, 133, 133, 133, 133, 133, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 126, 246, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 148, 255, 255, 255, 255, 255, 255, 255, 246, 191, 206, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 53, 255, 255, 255, 255, 255, 255, 255, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 255, 255, 255, 255, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 255, 255, 255, 255, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 219, 255, 255, 255, 255, 255, 255, 255, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 255, 255, 255, 255, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 209, 255, 255, 255, 255, 255, 255, 255, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 177, 255, 255, 255, 255, 255, 255, 255, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 103, 255, 255, 255, 255, 255, 255, 255, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 14, 225, 255, 255, 255, 255, 255, 255, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 55, 230, 255, 255, 255, 255, 255, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 114, 184, 210, 214, 197, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 50, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 50, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 50, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 50, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 50, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 50, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 50, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 50, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 50, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 50, 255, 218, 0, 58, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 109, 93, 0, 25, 109, 93, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '·' / 0xb7
        '·' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 74, 175, 181, 97, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 33, 250, 255, 255, 255, 69, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 81, 255, 255, 255, 255, 124, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 45, 255, 255, 255, 255, 86, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 114, 227, 234, 144, 1, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¸' / 0xb8
        '¸' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 29, 34, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 27, 251, 240, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 132, 255, 255, 196, 35, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 46, 130, 255, 255, 199, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 1, 229, 255, 252, 2, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 109, 141, 182, 255, 255, 223, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 160, 255, 255, 255, 244, 78, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 47, 107, 115, 88, 22, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: '¹' / 0xb9
        '¹' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 10, 140, 201, 183, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 66, 221, 255, 255, 232, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 153, 254, 255, 255, 255, 232, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 161, 255, 192, 185, 255, 232, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 20, 101, 1, 167, 255, 232, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 172, 255, 232, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 174, 255, 232, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 174, 255, 232, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 174, 255, 232, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 11, 35, 230, 255, 255, 41, 16, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 95, 255, 255, 255, 255, 255, 255, 255, 139, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 75, 196, 196, 255, 255, 255, 196, 196, 109, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'º' / 0xba
        'º' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 45, 99, 113, 60, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 3, 149, 255, 255, 255, 255, 182, 11, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 103, 255, 255, 194, 185, 255, 255, 150, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 217, 255, 201, 1, 0, 170, 255, 247, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 5, 252, 255, 138, 0, 0, 106, 255, 255, 53, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 22, 255, 255, 129, 0, 0, 92, 255, 255, 67, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 5, 248, 255, 159, 0, 0, 122, 255, 255, 42, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 182, 255, 240, 58, 41, 221, 255, 227, 2, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 48, 244, 255, 255, 255, 255, 253, 88, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 37, 180, 238, 243, 196, 62, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '»' / 0xbb
        '»' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 58, 0, 0, 0, 0, 47, 36, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 13, 132, 245, 240, 35, 0, 31, 163, 253, 209, 9, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 22, 234, 255, 255, 206, 7, 60, 253, 255, 255, 157, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 80, 255, 255, 255, 151, 0, 134, 255, 255, 255, 95, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 158, 255, 255, 255, 90, 5, 206, 255, 255, 245, 43, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 14, 223, 255, 255, 243, 40, 42, 247, 255, 255, 215, 12, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 88, 255, 255, 255, 184, 0, 139, 255, 255, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 3, 197, 255, 255, 253, 69, 20, 231, 255, 255, 236, 30, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 120, 255, 255, 255, 129, 0, 173, 255, 255, 254, 74, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 47, 249, 255, 255, 188, 2, 95, 255, 255, 255, 134, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 6, 210, 255, 255, 230, 23, 31, 241, 255, 255, 193, 3, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 29, 189, 255, 252, 63, 0, 60, 215, 255, 232, 25, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 68, 98, 0, 0, 0, 3, 99, 65, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¼' / 0xbc
        '¼' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 65, 71, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 63, 209, 255, 255, 26, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 127, 255, 255, 255, 255, 26, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 53, 247, 127, 254, 255, 26, 0, 0, 0, 0, 0, 5, 152, 168, 160, 13, 0, 0,
            ],
            &[
                0, 0, 19, 8, 255, 255, 26, 0, 0, 0, 0, 0, 144, 255, 255, 104, 0, 0, 0,
            ],
            &[
                0, 0, 0, 13, 255, 255, 26, 0, 0, 0, 0, 82, 255, 255, 167, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 14, 255, 255, 26, 0, 0, 0, 34, 240, 255, 217, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 25, 255, 255, 38, 0, 0, 7, 205, 255, 246, 46, 0, 0, 0, 0, 0,
            ],
            &[
                0, 31, 245, 255, 255, 255, 255, 246, 29, 149, 255, 255, 100, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 26, 196, 207, 255, 255, 216, 196, 112, 255, 255, 163, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 38, 242, 255, 215, 11, 3, 18, 18, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 8, 209, 255, 245, 44, 0, 134, 255, 255, 51, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 155, 255, 255, 97, 0, 52, 252, 255, 255, 51, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 92, 255, 255, 160, 0, 6, 211, 232, 246, 255, 51, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 41, 244, 255, 212, 10, 0, 132, 255, 96, 249, 255, 51, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 212, 255, 244, 41, 0, 50, 251, 183, 0, 252, 255, 51, 0, 0, 0,
            ],
            &[
                0, 0, 0, 161, 255, 255, 93, 0, 0, 204, 255, 152, 123, 253, 255, 150, 74, 0, 0,
            ],
            &[
                0, 0, 98, 255, 255, 156, 0, 0, 0, 237, 255, 255, 255, 255, 255, 255, 155, 0, 0,
            ],
            &[
                0, 0, 5, 7, 7, 1, 0, 0, 0, 77, 83, 83, 83, 253, 255, 118, 50, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 252, 255, 51, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '½' / 0xbd
        '½' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 65, 71, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 63, 209, 255, 255, 26, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 127, 255, 255, 255, 255, 26, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 53, 247, 127, 254, 255, 26, 0, 0, 0, 0, 0, 5, 152, 168, 160, 13, 0, 0,
            ],
            &[
                0, 0, 19, 8, 255, 255, 26, 0, 0, 0, 0, 0, 144, 255, 255, 104, 0, 0, 0,
            ],
            &[
                0, 0, 0, 13, 255, 255, 26, 0, 0, 0, 0, 82, 255, 255, 167, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 14, 255, 255, 26, 0, 0, 0, 34, 240, 255, 217, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 25, 255, 255, 38, 0, 0, 7, 205, 255, 246, 46, 0, 0, 0, 0, 0,
            ],
            &[
                0, 31, 245, 255, 255, 255, 255, 246, 29, 149, 255, 255, 100, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 26, 196, 207, 255, 255, 216, 196, 112, 255, 255, 163, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 38, 242, 255, 215, 12, 44, 65, 32, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 8, 209, 255, 245, 88, 208, 255, 255, 255, 157, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 155, 255, 255, 97, 114, 255, 202, 198, 255, 255, 64, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 92, 255, 255, 160, 0, 0, 71, 0, 3, 242, 255, 91, 0, 0,
            ],
            &[
                0, 0, 0, 0, 41, 244, 255, 212, 10, 0, 0, 0, 0, 62, 255, 253, 35, 0, 0,
            ],
            &[
                0, 0, 0, 10, 212, 255, 244, 41, 0, 0, 0, 0, 46, 236, 255, 136, 0, 0, 0,
            ],
            &[
                0, 0, 0, 161, 255, 255, 93, 0, 0, 0, 0, 43, 234, 255, 163, 2, 0, 0, 0,
            ],
            &[
                0, 0, 98, 255, 255, 156, 0, 0, 0, 0, 39, 232, 255, 154, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 5, 7, 7, 1, 0, 0, 0, 0, 178, 255, 255, 248, 248, 248, 190, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 182, 255, 255, 255, 255, 255, 196, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¾' / 0xbe
        '¾' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 62, 110, 118, 63, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 6, 194, 255, 255, 255, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 136, 188, 99, 164, 255, 250, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 2, 0, 0, 71, 255, 225, 0, 0, 0, 0, 5, 152, 168, 160, 13, 0, 0,
            ],
            &[
                0, 0, 0, 113, 202, 249, 241, 76, 0, 0, 0, 0, 144, 255, 255, 104, 0, 0, 0,
            ],
            &[
                0, 0, 0, 150, 254, 255, 246, 134, 0, 0, 0, 82, 255, 255, 167, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 4, 66, 253, 255, 50, 0, 34, 240, 255, 217, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 149, 48, 15, 78, 255, 255, 62, 7, 205, 255, 246, 46, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 255, 255, 255, 255, 255, 218, 9, 149, 255, 255, 100, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 116, 196, 222, 206, 149, 27, 87, 255, 255, 163, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 38, 242, 255, 215, 11, 3, 18, 18, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 8, 209, 255, 245, 44, 0, 134, 255, 255, 51, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 155, 255, 255, 97, 0, 52, 252, 255, 255, 51, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 92, 255, 255, 160, 0, 6, 211, 232, 246, 255, 51, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 41, 244, 255, 212, 10, 0, 132, 255, 96, 249, 255, 51, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 212, 255, 244, 41, 0, 50, 251, 183, 0, 252, 255, 51, 0, 0, 0,
            ],
            &[
                0, 0, 0, 161, 255, 255, 93, 0, 0, 204, 255, 152, 123, 253, 255, 150, 74, 0, 0,
            ],
            &[
                0, 0, 98, 255, 255, 156, 0, 0, 0, 237, 255, 255, 255, 255, 255, 255, 155, 0, 0,
            ],
            &[
                0, 0, 5, 7, 7, 1, 0, 0, 0, 77, 83, 83, 83, 253, 255, 118, 50, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 252, 255, 51, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¿' / 0xbf
        '¿' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 16, 111, 127, 56, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 5, 212, 255, 255, 254, 67, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 48, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 34, 255, 255, 255, 255, 135, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 147, 255, 255, 221, 30, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 23, 38, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 117, 226, 226, 217, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 164, 255, 255, 229, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 57, 247, 255, 255, 183, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 93, 247, 255, 255, 254, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 136, 255, 255, 255, 250, 96, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 143, 255, 255, 255, 235, 63, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 37, 253, 255, 255, 240, 37, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 102, 255, 255, 255, 128, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 118, 255, 255, 255, 97, 0, 0, 0, 0, 0, 25, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 90, 255, 255, 255, 209, 44, 3, 31, 79, 173, 251, 87, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 17, 242, 255, 255, 255, 255, 254, 255, 255, 255, 255, 212, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 91, 253, 255, 255, 255, 255, 255, 255, 255, 255, 245, 47, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 60, 192, 252, 255, 255, 255, 253, 206, 121, 21, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 17, 50, 62, 47, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'À' / 0xc0
        'À' => Some(&[
            &[
                0, 0, 0, 0, 32, 227, 255, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 36, 225, 255, 255, 216, 9, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 30, 216, 255, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 19, 194, 255, 254, 69, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 26, 26, 12, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 42, 91, 91, 91, 91, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 165, 255, 255, 255, 255, 210, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 1, 234, 255, 255, 255, 255, 255, 25, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 50, 255, 255, 255, 247, 255, 255, 95, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 120, 255, 255, 230, 179, 255, 255, 166, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 190, 255, 255, 172, 119, 255, 255, 235, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 11, 249, 255, 255, 112, 59, 255, 255, 255, 51, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 255, 255, 255, 49, 7, 247, 255, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 145, 255, 255, 239, 2, 0, 193, 255, 255, 193, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 216, 255, 255, 178, 0, 0, 131, 255, 255, 250, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 30, 255, 255, 255, 114, 0, 0, 70, 255, 255, 255, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 100, 255, 255, 255, 55, 4, 4, 17, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 238, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 34, 0, 0, 0,
            ],
            &[
                0, 0, 56, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 105, 0, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 107, 56, 56, 56, 56, 75, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 196, 255, 255, 240, 2, 0, 0, 0, 0, 0, 212, 255, 255, 241, 4, 0, 0,
            ],
            &[
                0, 15, 251, 255, 255, 179, 0, 0, 0, 0, 0, 0, 150, 255, 255, 255, 61, 0, 0,
            ],
            &[
                0, 81, 255, 255, 255, 115, 0, 0, 0, 0, 0, 0, 88, 255, 255, 255, 131, 0, 0,
            ],
            &[
                0, 151, 255, 255, 255, 51, 0, 0, 0, 0, 0, 0, 27, 255, 255, 255, 202, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 55, 55, 55, 48, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Á' / 0xc1
        'Á' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 255, 255, 179, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 49, 250, 255, 255, 180, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 8, 212, 255, 255, 165, 4, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 143, 255, 255, 134, 1, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 19, 26, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 42, 91, 91, 91, 91, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 165, 255, 255, 255, 255, 210, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 1, 234, 255, 255, 255, 255, 255, 25, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 50, 255, 255, 255, 247, 255, 255, 95, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 120, 255, 255, 230, 179, 255, 255, 166, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 190, 255, 255, 172, 119, 255, 255, 235, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 11, 249, 255, 255, 112, 59, 255, 255, 255, 51, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 255, 255, 255, 49, 7, 247, 255, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 145, 255, 255, 239, 2, 0, 193, 255, 255, 193, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 216, 255, 255, 178, 0, 0, 131, 255, 255, 250, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 30, 255, 255, 255, 114, 0, 0, 70, 255, 255, 255, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 100, 255, 255, 255, 55, 4, 4, 17, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 238, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 34, 0, 0, 0,
            ],
            &[
                0, 0, 56, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 105, 0, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 107, 56, 56, 56, 56, 75, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 196, 255, 255, 240, 2, 0, 0, 0, 0, 0, 212, 255, 255, 241, 4, 0, 0,
            ],
            &[
                0, 15, 251, 255, 255, 179, 0, 0, 0, 0, 0, 0, 150, 255, 255, 255, 61, 0, 0,
            ],
            &[
                0, 81, 255, 255, 255, 115, 0, 0, 0, 0, 0, 0, 88, 255, 255, 255, 131, 0, 0,
            ],
            &[
                0, 151, 255, 255, 255, 51, 0, 0, 0, 0, 0, 0, 27, 255, 255, 255, 202, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 55, 55, 55, 55, 9, 0, 0, 0, 0],
        ]),
        // letter: 'Â' / 0xc2
        'Â' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 255, 145, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 27, 234, 255, 255, 249, 255, 253, 70, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 6, 196, 255, 254, 108, 66, 245, 255, 234, 29, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 245, 89, 0, 0, 54, 227, 255, 205, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 26, 16, 0, 0, 0, 0, 10, 26, 26, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 42, 91, 91, 91, 91, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 165, 255, 255, 255, 255, 210, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 1, 234, 255, 255, 255, 255, 255, 25, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 50, 255, 255, 255, 247, 255, 255, 95, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 120, 255, 255, 230, 179, 255, 255, 166, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 190, 255, 255, 172, 119, 255, 255, 235, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 11, 249, 255, 255, 112, 59, 255, 255, 255, 51, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 255, 255, 255, 49, 7, 247, 255, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 145, 255, 255, 239, 2, 0, 193, 255, 255, 193, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 216, 255, 255, 178, 0, 0, 131, 255, 255, 250, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 30, 255, 255, 255, 114, 0, 0, 70, 255, 255, 255, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 100, 255, 255, 255, 55, 4, 4, 17, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 238, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 34, 0, 0, 0,
            ],
            &[
                0, 0, 56, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 105, 0, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 107, 56, 56, 56, 56, 75, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 196, 255, 255, 240, 2, 0, 0, 0, 0, 0, 212, 255, 255, 241, 4, 0, 0,
            ],
            &[
                0, 15, 251, 255, 255, 179, 0, 0, 0, 0, 0, 0, 150, 255, 255, 255, 61, 0, 0,
            ],
            &[
                0, 81, 255, 255, 255, 115, 0, 0, 0, 0, 0, 0, 88, 255, 255, 255, 131, 0, 0,
            ],
            &[
                0, 151, 255, 255, 255, 51, 0, 0, 0, 0, 0, 0, 27, 255, 255, 255, 202, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 52, 55, 55, 55, 9, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ã' / 0xc3
        'Ã' => Some(&[
            &[
                0, 0, 0, 0, 0, 100, 207, 210, 136, 26, 0, 1, 213, 242, 36, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 255, 255, 255, 255, 249, 177, 183, 255, 248, 4, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 185, 255, 234, 212, 255, 255, 255, 255, 255, 150, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 237, 255, 34, 0, 46, 160, 244, 253, 178, 15, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 33, 30, 0, 0, 0, 0, 2, 11, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 42, 91, 91, 91, 91, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 165, 255, 255, 255, 255, 210, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 1, 234, 255, 255, 255, 255, 255, 25, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 50, 255, 255, 255, 247, 255, 255, 95, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 120, 255, 255, 230, 179, 255, 255, 166, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 190, 255, 255, 172, 119, 255, 255, 235, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 11, 249, 255, 255, 112, 59, 255, 255, 255, 51, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 255, 255, 255, 49, 7, 247, 255, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 145, 255, 255, 239, 2, 0, 193, 255, 255, 193, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 216, 255, 255, 178, 0, 0, 131, 255, 255, 250, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 30, 255, 255, 255, 114, 0, 0, 70, 255, 255, 255, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 100, 255, 255, 255, 55, 4, 4, 17, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 238, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 34, 0, 0, 0,
            ],
            &[
                0, 0, 56, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 105, 0, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 107, 56, 56, 56, 56, 75, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 196, 255, 255, 240, 2, 0, 0, 0, 0, 0, 212, 255, 255, 241, 4, 0, 0,
            ],
            &[
                0, 15, 251, 255, 255, 179, 0, 0, 0, 0, 0, 0, 150, 255, 255, 255, 61, 0, 0,
            ],
            &[
                0, 81, 255, 255, 255, 115, 0, 0, 0, 0, 0, 0, 88, 255, 255, 255, 131, 0, 0,
            ],
            &[
                0, 151, 255, 255, 255, 51, 0, 0, 0, 0, 0, 0, 27, 255, 255, 255, 202, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ä' / 0xc4
        'Ä' => Some(&[
            &[
                0, 0, 0, 0, 96, 231, 220, 68, 0, 0, 44, 209, 236, 130, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 236, 255, 255, 204, 0, 0, 161, 255, 255, 255, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 235, 255, 255, 203, 0, 0, 160, 255, 255, 255, 29, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 95, 235, 223, 67, 0, 0, 41, 210, 240, 129, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 42, 91, 91, 91, 91, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 165, 255, 255, 255, 255, 210, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 1, 234, 255, 255, 255, 255, 255, 25, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 50, 255, 255, 255, 247, 255, 255, 95, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 120, 255, 255, 230, 179, 255, 255, 166, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 190, 255, 255, 172, 119, 255, 255, 235, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 11, 249, 255, 255, 112, 59, 255, 255, 255, 51, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 255, 255, 255, 49, 7, 247, 255, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 145, 255, 255, 239, 2, 0, 193, 255, 255, 193, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 216, 255, 255, 178, 0, 0, 131, 255, 255, 250, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 30, 255, 255, 255, 114, 0, 0, 70, 255, 255, 255, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 100, 255, 255, 255, 55, 4, 4, 17, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 238, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 34, 0, 0, 0,
            ],
            &[
                0, 0, 56, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 105, 0, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 107, 56, 56, 56, 56, 75, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 196, 255, 255, 240, 2, 0, 0, 0, 0, 0, 212, 255, 255, 241, 4, 0, 0,
            ],
            &[
                0, 15, 251, 255, 255, 179, 0, 0, 0, 0, 0, 0, 150, 255, 255, 255, 61, 0, 0,
            ],
            &[
                0, 81, 255, 255, 255, 115, 0, 0, 0, 0, 0, 0, 88, 255, 255, 255, 131, 0, 0,
            ],
            &[
                0, 151, 255, 255, 255, 51, 0, 0, 0, 0, 0, 0, 27, 255, 255, 255, 202, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Å' / 0xc5
        'Å' => Some(&[
            &[
                0, 0, 0, 0, 0, 15, 255, 243, 15, 4, 222, 255, 57, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 13, 254, 242, 12, 4, 223, 255, 55, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 190, 255, 239, 232, 255, 219, 7, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 27, 183, 252, 253, 195, 41, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 6, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 42, 91, 91, 91, 91, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 165, 255, 255, 255, 255, 210, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 1, 234, 255, 255, 255, 255, 255, 25, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 50, 255, 255, 255, 247, 255, 255, 95, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 120, 255, 255, 230, 179, 255, 255, 166, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 190, 255, 255, 172, 119, 255, 255, 235, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 11, 249, 255, 255, 112, 59, 255, 255, 255, 51, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 255, 255, 255, 49, 7, 247, 255, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 145, 255, 255, 239, 2, 0, 193, 255, 255, 193, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 216, 255, 255, 178, 0, 0, 131, 255, 255, 250, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 30, 255, 255, 255, 114, 0, 0, 70, 255, 255, 255, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 100, 255, 255, 255, 55, 4, 4, 17, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 238, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 34, 0, 0, 0,
            ],
            &[
                0, 0, 56, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 105, 0, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 107, 56, 56, 56, 56, 75, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 196, 255, 255, 240, 2, 0, 0, 0, 0, 0, 212, 255, 255, 241, 4, 0, 0,
            ],
            &[
                0, 15, 251, 255, 255, 179, 0, 0, 0, 0, 0, 0, 150, 255, 255, 255, 61, 0, 0,
            ],
            &[
                0, 81, 255, 255, 255, 115, 0, 0, 0, 0, 0, 0, 88, 255, 255, 255, 131, 0, 0,
            ],
            &[
                0, 151, 255, 255, 255, 51, 0, 0, 0, 0, 0, 0, 27, 255, 255, 255, 202, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 194, 255, 239, 231, 255, 221, 8, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'Æ' / 0xc6
        'Æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 12, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 27, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 83, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 144, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 205, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0, 0,
            ],
            &[
                0, 0, 0, 0, 13, 252, 255, 255, 88, 174, 255, 255, 163, 17, 17, 17, 6, 0, 0,
            ],
            &[
                0, 0, 0, 0, 71, 255, 255, 255, 21, 168, 255, 255, 156, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 220, 0, 168, 255, 255, 156, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 192, 255, 255, 164, 0, 168, 255, 255, 156, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 6, 247, 255, 255, 107, 0, 168, 255, 255, 225, 176, 176, 159, 0, 0, 0,
            ],
            &[
                0, 0, 0, 58, 255, 255, 255, 51, 0, 168, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 0, 119, 255, 255, 246, 4, 0, 168, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 220, 25, 25, 193, 255, 255, 221, 168, 168, 151, 0, 0, 0,
            ],
            &[
                0, 0, 2, 239, 255, 255, 255, 255, 255, 255, 255, 255, 156, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 46, 255, 255, 255, 255, 255, 255, 255, 255, 255, 156, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 107, 255, 255, 255, 255, 255, 255, 255, 255, 255, 156, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 168, 255, 255, 255, 63, 63, 63, 231, 255, 255, 156, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 229, 255, 255, 168, 0, 0, 0, 168, 255, 255, 190, 89, 89, 89, 34, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 112, 0, 0, 0, 168, 255, 255, 255, 255, 255, 255, 99, 0, 0,
            ],
            &[
                0, 95, 255, 255, 255, 55, 0, 0, 0, 168, 255, 255, 255, 255, 255, 255, 99, 0, 0,
            ],
            &[
                0, 156, 255, 255, 248, 6, 0, 0, 0, 168, 255, 255, 255, 255, 255, 255, 99, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ç' / 0xc7
        'Ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 30, 93, 119, 135, 125, 88, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 69, 202, 255, 255, 255, 255, 255, 255, 249, 171, 5, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 189, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 129, 255, 255, 255, 255, 255, 245, 233, 253, 255, 255, 86, 0, 0, 0,
            ],
            &[
                0, 0, 0, 50, 251, 255, 255, 255, 169, 45, 0, 0, 5, 68, 154, 5, 0, 0, 0,
            ],
            &[
                0, 0, 0, 168, 255, 255, 255, 156, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 16, 250, 255, 255, 243, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 72, 255, 255, 255, 173, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 123, 255, 255, 255, 122, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 94, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 160, 255, 255, 255, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 150, 255, 255, 255, 96, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 130, 255, 255, 255, 117, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 101, 255, 255, 255, 172, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 34, 255, 255, 255, 242, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 255, 255, 170, 4, 0, 0, 0, 0, 0, 4, 16, 0, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 255, 255, 204, 99, 58, 53, 84, 144, 232, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 194, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 17, 196, 255, 255, 255, 255, 255, 255, 255, 255, 255, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 4, 106, 211, 255, 255, 255, 255, 255, 227, 151, 31, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 58, 255, 255, 46, 18, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 146, 255, 255, 188, 29, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 51, 139, 255, 255, 185, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 240, 255, 241, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 118, 141, 188, 255, 255, 209, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 174, 255, 255, 255, 241, 67, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 52, 109, 114, 86, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'È' / 0xc8
        'È' => Some(&[
            &[
                0, 0, 0, 0, 14, 204, 255, 255, 255, 103, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 17, 202, 255, 255, 239, 27, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 14, 190, 255, 255, 184, 1, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 7, 165, 255, 255, 110, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 25, 26, 16, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 59, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 22, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 242, 24, 24, 24, 24, 24, 24, 24, 24, 7, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 176, 176, 176, 176, 176, 176, 176, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 175, 175, 175, 175, 175, 175, 175, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 246, 95, 95, 95, 95, 95, 95, 95, 95, 29, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 55, 55, 55, 55, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'É' / 0xc9
        'É' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 93, 255, 255, 255, 208, 17, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 23, 235, 255, 255, 207, 21, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 178, 255, 255, 196, 16, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 101, 255, 255, 169, 8, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 15, 26, 26, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 59, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 22, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 242, 24, 24, 24, 24, 24, 24, 24, 24, 7, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 176, 176, 176, 176, 176, 176, 176, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 175, 175, 175, 175, 175, 175, 175, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 246, 95, 95, 95, 95, 95, 95, 95, 95, 29, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 54, 55, 55, 55, 18, 0, 0, 0, 0],
        ]),
        // letter: 'Ê' / 0xca
        'Ê' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 48, 251, 255, 255, 255, 186, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 9, 211, 255, 255, 248, 255, 255, 110, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 161, 255, 255, 147, 41, 230, 255, 249, 56, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 112, 255, 253, 122, 0, 0, 32, 206, 255, 229, 19, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 17, 26, 20, 0, 0, 0, 0, 6, 26, 26, 5, 0, 0, 0, 0],
            &[
                0, 0, 0, 59, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 22, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 242, 24, 24, 24, 24, 24, 24, 24, 24, 7, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 176, 176, 176, 176, 176, 176, 176, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 175, 175, 175, 175, 175, 175, 175, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 246, 95, 95, 95, 95, 95, 95, 95, 95, 29, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 43, 55, 55, 55, 18, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ë' / 0xcb
        'Ë' => Some(&[
            &[
                0, 0, 0, 0, 66, 221, 230, 98, 0, 0, 21, 193, 239, 162, 3, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 195, 255, 255, 244, 1, 0, 120, 255, 255, 255, 71, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 194, 255, 255, 243, 1, 0, 118, 255, 255, 255, 70, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 65, 224, 234, 97, 0, 0, 19, 193, 243, 163, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 59, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 22, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 242, 24, 24, 24, 24, 24, 24, 24, 24, 7, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 176, 176, 176, 176, 176, 176, 176, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 175, 175, 175, 175, 175, 175, 175, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 246, 95, 95, 95, 95, 95, 95, 95, 95, 29, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ì' / 0xcc
        'Ì' => Some(&[
            &[
                0, 0, 0, 0, 32, 227, 255, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 36, 225, 255, 255, 216, 9, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 30, 216, 255, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 19, 194, 255, 254, 69, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 26, 26, 12, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 40, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 66, 139, 168, 198, 248, 255, 255, 253, 202, 173, 144, 91, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 13, 214, 255, 255, 241, 17, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 210, 240, 255, 255, 255, 255, 255, 255, 244, 215, 146, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 55, 55, 55, 48, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Í' / 0xcd
        'Í' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 255, 255, 179, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 49, 250, 255, 255, 180, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 8, 212, 255, 255, 165, 4, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 143, 255, 255, 134, 1, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 19, 26, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 40, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 66, 139, 168, 198, 248, 255, 255, 253, 202, 173, 144, 91, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 13, 214, 255, 255, 241, 17, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 210, 240, 255, 255, 255, 255, 255, 255, 244, 215, 146, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 55, 55, 55, 55, 9, 0, 0, 0, 0],
        ]),
        // letter: 'Î' / 0xce
        'Î' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 255, 145, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 27, 234, 255, 255, 249, 255, 253, 70, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 6, 196, 255, 254, 108, 66, 245, 255, 234, 29, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 245, 89, 0, 0, 54, 227, 255, 205, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 26, 16, 0, 0, 0, 0, 10, 26, 26, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 40, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 66, 139, 168, 198, 248, 255, 255, 253, 202, 173, 144, 91, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 13, 214, 255, 255, 241, 17, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 210, 240, 255, 255, 255, 255, 255, 255, 244, 215, 146, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 52, 55, 55, 55, 9, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ï' / 0xcf
        'Ï' => Some(&[
            &[
                0, 0, 0, 0, 96, 231, 220, 68, 0, 0, 44, 209, 236, 130, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 236, 255, 255, 204, 0, 0, 161, 255, 255, 255, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 235, 255, 255, 203, 0, 0, 160, 255, 255, 255, 29, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 95, 235, 223, 67, 0, 0, 41, 210, 240, 129, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 40, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 66, 139, 168, 198, 248, 255, 255, 253, 202, 173, 144, 91, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 13, 214, 255, 255, 241, 17, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 210, 240, 255, 255, 255, 255, 255, 255, 244, 215, 146, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ð' / 0xd0
        'Ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 71, 71, 66, 46, 7, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 247, 175, 54, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 253, 126, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 116, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 29, 11, 26, 74, 196, 255, 255, 255, 249, 36, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 6, 191, 255, 255, 255, 151, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 50, 255, 255, 255, 235, 1, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 0, 227, 255, 255, 255, 39, 0, 0,
            ],
            &[
                0, 38, 239, 255, 255, 255, 58, 38, 36, 0, 0, 0, 177, 255, 255, 255, 75, 0, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 240, 0, 0, 0, 159, 255, 255, 255, 92, 0, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 240, 0, 0, 0, 152, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 240, 0, 0, 0, 164, 255, 255, 255, 84, 0, 0,
            ],
            &[
                0, 22, 223, 255, 255, 255, 42, 22, 21, 0, 0, 0, 197, 255, 255, 255, 55, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 10, 246, 255, 255, 251, 12, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 103, 255, 255, 255, 192, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 51, 239, 255, 255, 255, 89, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 95, 83, 101, 162, 250, 255, 255, 255, 202, 2, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 230, 33, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 199, 33, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 247, 221, 166, 77, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ñ' / 0xd1
        'Ñ' => Some(&[
            &[
                0, 0, 0, 0, 0, 76, 198, 216, 150, 38, 0, 0, 182, 242, 69, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 45, 251, 255, 255, 255, 253, 186, 171, 254, 255, 31, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 150, 255, 243, 207, 252, 255, 255, 255, 255, 185, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 202, 255, 68, 0, 33, 145, 238, 255, 197, 28, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 29, 33, 2, 0, 0, 0, 0, 12, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 71, 41, 0, 0, 0, 0, 35, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 201, 0, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 254, 30, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 113, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 222, 255, 255, 196, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 151, 253, 255, 254, 26, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 141, 195, 255, 255, 108, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 155, 112, 255, 255, 191, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 166, 30, 255, 255, 252, 22, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 175, 0, 202, 255, 255, 103, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 119, 255, 255, 187, 0, 126, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 36, 255, 255, 251, 19, 123, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 209, 255, 255, 98, 118, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 126, 255, 255, 182, 111, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 43, 255, 255, 249, 117, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 216, 255, 255, 182, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 133, 255, 255, 248, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 50, 255, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 0, 223, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 0, 140, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ò' / 0xd2
        'Ò' => Some(&[
            &[
                0, 0, 0, 0, 32, 227, 255, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 36, 225, 255, 255, 216, 9, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 30, 216, 255, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 19, 194, 255, 254, 69, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 26, 26, 12, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 47, 114, 144, 144, 119, 55, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 187, 255, 255, 255, 255, 255, 255, 208, 49, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 25, 226, 255, 255, 255, 255, 255, 255, 255, 255, 244, 52, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 181, 255, 255, 255, 255, 224, 211, 253, 255, 255, 255, 221, 6, 0, 0, 0,
            ],
            &[
                0, 0, 52, 255, 255, 255, 252, 84, 0, 0, 46, 231, 255, 255, 255, 102, 0, 0, 0,
            ],
            &[
                0, 0, 136, 255, 255, 255, 145, 0, 0, 0, 0, 95, 255, 255, 255, 190, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 50, 0, 0, 0, 0, 17, 252, 255, 255, 251, 7, 0, 0,
            ],
            &[
                0, 2, 249, 255, 255, 250, 4, 0, 0, 0, 0, 0, 204, 255, 255, 255, 45, 0, 0,
            ],
            &[
                0, 17, 255, 255, 255, 221, 0, 0, 0, 0, 0, 0, 180, 255, 255, 255, 76, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 208, 0, 0, 0, 0, 0, 0, 162, 255, 255, 255, 89, 0, 0,
            ],
            &[
                0, 44, 255, 255, 255, 202, 0, 0, 0, 0, 0, 0, 151, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 212, 0, 0, 0, 0, 0, 0, 161, 255, 255, 255, 85, 0, 0,
            ],
            &[
                0, 18, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 179, 255, 255, 255, 69, 0, 0,
            ],
            &[
                0, 0, 236, 255, 255, 255, 12, 0, 0, 0, 0, 0, 219, 255, 255, 255, 32, 0, 0,
            ],
            &[
                0, 0, 190, 255, 255, 255, 77, 0, 0, 0, 0, 31, 254, 255, 255, 239, 1, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 190, 1, 0, 0, 0, 145, 255, 255, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 24, 244, 255, 255, 255, 168, 53, 45, 134, 254, 255, 255, 255, 64, 0, 0, 0,
            ],
            &[
                0, 0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 175, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 163, 255, 255, 255, 255, 255, 255, 255, 255, 202, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 94, 209, 255, 255, 255, 255, 225, 124, 6, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 54, 59, 34, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 55, 55, 55, 48, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ó' / 0xd3
        'Ó' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 255, 255, 179, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 49, 250, 255, 255, 180, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 8, 212, 255, 255, 165, 4, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 143, 255, 255, 134, 1, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 19, 26, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 47, 114, 144, 144, 119, 55, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 187, 255, 255, 255, 255, 255, 255, 208, 49, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 25, 226, 255, 255, 255, 255, 255, 255, 255, 255, 244, 52, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 181, 255, 255, 255, 255, 224, 211, 253, 255, 255, 255, 221, 6, 0, 0, 0,
            ],
            &[
                0, 0, 52, 255, 255, 255, 252, 84, 0, 0, 46, 231, 255, 255, 255, 102, 0, 0, 0,
            ],
            &[
                0, 0, 136, 255, 255, 255, 145, 0, 0, 0, 0, 95, 255, 255, 255, 190, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 50, 0, 0, 0, 0, 17, 252, 255, 255, 251, 7, 0, 0,
            ],
            &[
                0, 2, 249, 255, 255, 250, 4, 0, 0, 0, 0, 0, 204, 255, 255, 255, 45, 0, 0,
            ],
            &[
                0, 17, 255, 255, 255, 221, 0, 0, 0, 0, 0, 0, 180, 255, 255, 255, 76, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 208, 0, 0, 0, 0, 0, 0, 162, 255, 255, 255, 89, 0, 0,
            ],
            &[
                0, 44, 255, 255, 255, 202, 0, 0, 0, 0, 0, 0, 151, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 212, 0, 0, 0, 0, 0, 0, 161, 255, 255, 255, 85, 0, 0,
            ],
            &[
                0, 18, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 179, 255, 255, 255, 69, 0, 0,
            ],
            &[
                0, 0, 236, 255, 255, 255, 12, 0, 0, 0, 0, 0, 219, 255, 255, 255, 32, 0, 0,
            ],
            &[
                0, 0, 190, 255, 255, 255, 77, 0, 0, 0, 0, 31, 254, 255, 255, 239, 1, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 190, 1, 0, 0, 0, 145, 255, 255, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 24, 244, 255, 255, 255, 168, 53, 45, 134, 254, 255, 255, 255, 64, 0, 0, 0,
            ],
            &[
                0, 0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 175, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 163, 255, 255, 255, 255, 255, 255, 255, 255, 202, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 94, 209, 255, 255, 255, 255, 225, 124, 6, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 54, 59, 34, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 55, 55, 55, 55, 9, 0, 0, 0, 0],
        ]),
        // letter: 'Ô' / 0xd4
        'Ô' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 255, 145, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 27, 234, 255, 255, 249, 255, 253, 70, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 6, 196, 255, 254, 108, 66, 245, 255, 234, 29, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 245, 89, 0, 0, 54, 227, 255, 205, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 26, 16, 0, 0, 0, 0, 10, 26, 26, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 47, 114, 144, 144, 119, 55, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 187, 255, 255, 255, 255, 255, 255, 208, 49, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 25, 226, 255, 255, 255, 255, 255, 255, 255, 255, 244, 52, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 181, 255, 255, 255, 255, 224, 211, 253, 255, 255, 255, 221, 6, 0, 0, 0,
            ],
            &[
                0, 0, 52, 255, 255, 255, 252, 84, 0, 0, 46, 231, 255, 255, 255, 102, 0, 0, 0,
            ],
            &[
                0, 0, 136, 255, 255, 255, 145, 0, 0, 0, 0, 95, 255, 255, 255, 190, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 50, 0, 0, 0, 0, 17, 252, 255, 255, 251, 7, 0, 0,
            ],
            &[
                0, 2, 249, 255, 255, 250, 4, 0, 0, 0, 0, 0, 204, 255, 255, 255, 45, 0, 0,
            ],
            &[
                0, 17, 255, 255, 255, 221, 0, 0, 0, 0, 0, 0, 180, 255, 255, 255, 76, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 208, 0, 0, 0, 0, 0, 0, 162, 255, 255, 255, 89, 0, 0,
            ],
            &[
                0, 44, 255, 255, 255, 202, 0, 0, 0, 0, 0, 0, 151, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 212, 0, 0, 0, 0, 0, 0, 161, 255, 255, 255, 85, 0, 0,
            ],
            &[
                0, 18, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 179, 255, 255, 255, 69, 0, 0,
            ],
            &[
                0, 0, 236, 255, 255, 255, 12, 0, 0, 0, 0, 0, 219, 255, 255, 255, 32, 0, 0,
            ],
            &[
                0, 0, 190, 255, 255, 255, 77, 0, 0, 0, 0, 31, 254, 255, 255, 239, 1, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 190, 1, 0, 0, 0, 145, 255, 255, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 24, 244, 255, 255, 255, 168, 53, 45, 134, 254, 255, 255, 255, 64, 0, 0, 0,
            ],
            &[
                0, 0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 175, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 163, 255, 255, 255, 255, 255, 255, 255, 255, 202, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 94, 209, 255, 255, 255, 255, 225, 124, 6, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 54, 59, 34, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 52, 55, 55, 55, 9, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Õ' / 0xd5
        'Õ' => Some(&[
            &[
                0, 0, 0, 0, 0, 100, 207, 210, 136, 26, 0, 1, 213, 242, 36, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 255, 255, 255, 255, 249, 177, 183, 255, 248, 4, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 185, 255, 234, 212, 255, 255, 255, 255, 255, 150, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 237, 255, 34, 0, 46, 160, 244, 253, 178, 15, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 33, 30, 0, 0, 0, 0, 2, 11, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 47, 114, 144, 144, 119, 55, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 187, 255, 255, 255, 255, 255, 255, 208, 49, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 25, 226, 255, 255, 255, 255, 255, 255, 255, 255, 244, 52, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 181, 255, 255, 255, 255, 224, 211, 253, 255, 255, 255, 221, 6, 0, 0, 0,
            ],
            &[
                0, 0, 52, 255, 255, 255, 252, 84, 0, 0, 46, 231, 255, 255, 255, 102, 0, 0, 0,
            ],
            &[
                0, 0, 136, 255, 255, 255, 145, 0, 0, 0, 0, 95, 255, 255, 255, 190, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 50, 0, 0, 0, 0, 17, 252, 255, 255, 251, 7, 0, 0,
            ],
            &[
                0, 2, 249, 255, 255, 250, 4, 0, 0, 0, 0, 0, 204, 255, 255, 255, 45, 0, 0,
            ],
            &[
                0, 17, 255, 255, 255, 221, 0, 0, 0, 0, 0, 0, 180, 255, 255, 255, 76, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 208, 0, 0, 0, 0, 0, 0, 162, 255, 255, 255, 89, 0, 0,
            ],
            &[
                0, 44, 255, 255, 255, 202, 0, 0, 0, 0, 0, 0, 151, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 212, 0, 0, 0, 0, 0, 0, 161, 255, 255, 255, 85, 0, 0,
            ],
            &[
                0, 18, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 179, 255, 255, 255, 69, 0, 0,
            ],
            &[
                0, 0, 236, 255, 255, 255, 12, 0, 0, 0, 0, 0, 219, 255, 255, 255, 32, 0, 0,
            ],
            &[
                0, 0, 190, 255, 255, 255, 77, 0, 0, 0, 0, 31, 254, 255, 255, 239, 1, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 190, 1, 0, 0, 0, 145, 255, 255, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 24, 244, 255, 255, 255, 168, 53, 45, 134, 254, 255, 255, 255, 64, 0, 0, 0,
            ],
            &[
                0, 0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 175, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 163, 255, 255, 255, 255, 255, 255, 255, 255, 202, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 94, 209, 255, 255, 255, 255, 225, 124, 6, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 54, 59, 34, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ö' / 0xd6
        'Ö' => Some(&[
            &[
                0, 0, 0, 0, 96, 231, 220, 68, 0, 0, 44, 209, 236, 130, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 236, 255, 255, 204, 0, 0, 161, 255, 255, 255, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 235, 255, 255, 203, 0, 0, 160, 255, 255, 255, 29, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 95, 235, 223, 67, 0, 0, 41, 210, 240, 129, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 47, 114, 144, 144, 119, 55, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 187, 255, 255, 255, 255, 255, 255, 208, 49, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 25, 226, 255, 255, 255, 255, 255, 255, 255, 255, 244, 52, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 181, 255, 255, 255, 255, 224, 211, 253, 255, 255, 255, 221, 6, 0, 0, 0,
            ],
            &[
                0, 0, 52, 255, 255, 255, 252, 84, 0, 0, 46, 231, 255, 255, 255, 102, 0, 0, 0,
            ],
            &[
                0, 0, 136, 255, 255, 255, 145, 0, 0, 0, 0, 95, 255, 255, 255, 190, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 50, 0, 0, 0, 0, 17, 252, 255, 255, 251, 7, 0, 0,
            ],
            &[
                0, 2, 249, 255, 255, 250, 4, 0, 0, 0, 0, 0, 204, 255, 255, 255, 45, 0, 0,
            ],
            &[
                0, 17, 255, 255, 255, 221, 0, 0, 0, 0, 0, 0, 180, 255, 255, 255, 76, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 208, 0, 0, 0, 0, 0, 0, 162, 255, 255, 255, 89, 0, 0,
            ],
            &[
                0, 44, 255, 255, 255, 202, 0, 0, 0, 0, 0, 0, 151, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 212, 0, 0, 0, 0, 0, 0, 161, 255, 255, 255, 85, 0, 0,
            ],
            &[
                0, 18, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 179, 255, 255, 255, 69, 0, 0,
            ],
            &[
                0, 0, 236, 255, 255, 255, 12, 0, 0, 0, 0, 0, 219, 255, 255, 255, 32, 0, 0,
            ],
            &[
                0, 0, 190, 255, 255, 255, 77, 0, 0, 0, 0, 31, 254, 255, 255, 239, 1, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 190, 1, 0, 0, 0, 145, 255, 255, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 24, 244, 255, 255, 255, 168, 53, 45, 134, 254, 255, 255, 255, 64, 0, 0, 0,
            ],
            &[
                0, 0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 175, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 163, 255, 255, 255, 255, 255, 255, 255, 255, 202, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 94, 209, 255, 255, 255, 255, 225, 124, 6, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 54, 59, 34, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '×' / 0xd7
        '×' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 48, 0, 0, 0, 0, 0, 0, 30, 27, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 6, 181, 244, 63, 0, 0, 0, 0, 33, 223, 219, 28, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 147, 255, 255, 245, 67, 0, 0, 35, 226, 255, 255, 205, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 28, 220, 255, 255, 247, 71, 38, 228, 255, 255, 240, 53, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 220, 255, 255, 248, 233, 255, 255, 240, 53, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 28, 220, 255, 255, 255, 255, 240, 53, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 65, 255, 255, 255, 255, 109, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 34, 227, 255, 255, 255, 255, 244, 61, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 32, 225, 255, 255, 239, 219, 255, 255, 243, 59, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 30, 223, 255, 255, 239, 52, 25, 216, 255, 255, 242, 56, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 138, 255, 255, 238, 51, 0, 0, 25, 216, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 168, 237, 50, 0, 0, 0, 0, 25, 216, 205, 17, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 4, 37, 0, 0, 0, 0, 0, 0, 25, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ø' / 0xd8
        'Ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 5, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 47, 114, 144, 144, 119, 55, 0, 0, 178, 207, 49, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 187, 255, 255, 255, 255, 255, 255, 208, 144, 255, 255, 149, 0, 0,
            ],
            &[
                0, 0, 0, 25, 226, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 223, 12, 0, 0,
            ],
            &[
                0, 0, 0, 181, 255, 255, 255, 255, 224, 211, 253, 255, 255, 255, 255, 71, 0, 0, 0,
            ],
            &[
                0, 0, 52, 255, 255, 255, 252, 84, 0, 0, 46, 255, 255, 255, 255, 102, 0, 0, 0,
            ],
            &[
                0, 0, 136, 255, 255, 255, 145, 0, 0, 0, 30, 255, 255, 255, 255, 190, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 50, 0, 0, 0, 186, 255, 255, 255, 255, 251, 7, 0, 0,
            ],
            &[
                0, 2, 249, 255, 255, 250, 4, 0, 0, 103, 255, 255, 255, 255, 255, 255, 45, 0, 0,
            ],
            &[
                0, 17, 255, 255, 255, 221, 0, 0, 32, 243, 255, 220, 191, 255, 255, 255, 76, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 208, 0, 1, 190, 255, 254, 62, 162, 255, 255, 255, 89, 0, 0,
            ],
            &[
                0, 44, 255, 255, 255, 202, 0, 108, 255, 255, 144, 0, 151, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 212, 35, 244, 255, 219, 10, 0, 161, 255, 255, 255, 85, 0, 0,
            ],
            &[
                0, 18, 255, 255, 255, 231, 194, 255, 253, 60, 0, 0, 179, 255, 255, 255, 69, 0, 0,
            ],
            &[
                0, 0, 236, 255, 255, 255, 255, 255, 142, 0, 0, 0, 219, 255, 255, 255, 32, 0, 0,
            ],
            &[
                0, 0, 190, 255, 255, 255, 255, 217, 9, 0, 0, 31, 254, 255, 255, 239, 1, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 255, 60, 0, 0, 0, 145, 255, 255, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 24, 255, 255, 255, 255, 168, 53, 45, 134, 254, 255, 255, 255, 64, 0, 0, 0,
            ],
            &[
                0, 0, 41, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 175, 0, 0, 0, 0,
            ],
            &[
                0, 3, 201, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 202, 13, 0, 0, 0, 0,
            ],
            &[
                0, 105, 255, 255, 139, 94, 209, 255, 255, 255, 255, 225, 124, 6, 0, 0, 0, 0, 0,
            ],
            &[
                0, 7, 136, 211, 8, 0, 0, 26, 54, 59, 34, 1, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ù' / 0xd9
        'Ù' => Some(&[
            &[
                0, 0, 0, 0, 32, 227, 255, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 36, 225, 255, 255, 216, 9, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 30, 216, 255, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 19, 194, 255, 254, 69, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 26, 26, 12, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 9, 0, 0, 0, 0, 0, 66, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 238, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 44, 0, 0, 0, 0, 0, 248, 255, 255, 247, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 255, 255, 76, 0, 0, 0, 0, 23, 255, 255, 255, 224, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 160, 0, 0, 0, 0, 104, 255, 255, 255, 187, 0, 0, 0,
            ],
            &[
                0, 0, 59, 255, 255, 255, 254, 136, 45, 38, 103, 243, 255, 255, 255, 104, 0, 0, 0,
            ],
            &[
                0, 0, 1, 191, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 11, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 217, 255, 255, 255, 255, 255, 255, 255, 255, 230, 49, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 137, 228, 255, 255, 255, 255, 235, 148, 22, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 2, 30, 55, 57, 32, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 55, 55, 55, 48, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ú' / 0xda
        'Ú' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 255, 255, 179, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 49, 250, 255, 255, 180, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 8, 212, 255, 255, 165, 4, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 143, 255, 255, 134, 1, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 19, 26, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 9, 0, 0, 0, 0, 0, 66, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 238, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 44, 0, 0, 0, 0, 0, 248, 255, 255, 247, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 255, 255, 76, 0, 0, 0, 0, 23, 255, 255, 255, 224, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 160, 0, 0, 0, 0, 104, 255, 255, 255, 187, 0, 0, 0,
            ],
            &[
                0, 0, 59, 255, 255, 255, 254, 136, 45, 38, 103, 243, 255, 255, 255, 104, 0, 0, 0,
            ],
            &[
                0, 0, 1, 191, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 11, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 217, 255, 255, 255, 255, 255, 255, 255, 255, 230, 49, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 137, 228, 255, 255, 255, 255, 235, 148, 22, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 2, 30, 55, 57, 32, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 55, 55, 55, 55, 9, 0, 0, 0, 0],
        ]),
        // letter: 'Û' / 0xdb
        'Û' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 255, 145, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 27, 234, 255, 255, 249, 255, 253, 70, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 6, 196, 255, 254, 108, 66, 245, 255, 234, 29, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 245, 89, 0, 0, 54, 227, 255, 205, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 26, 16, 0, 0, 0, 0, 10, 26, 26, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 55, 71, 71, 71, 9, 0, 0, 0, 0, 0, 66, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 238, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 44, 0, 0, 0, 0, 0, 248, 255, 255, 247, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 255, 255, 76, 0, 0, 0, 0, 23, 255, 255, 255, 224, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 160, 0, 0, 0, 0, 104, 255, 255, 255, 187, 0, 0, 0,
            ],
            &[
                0, 0, 59, 255, 255, 255, 254, 136, 45, 38, 103, 243, 255, 255, 255, 104, 0, 0, 0,
            ],
            &[
                0, 0, 1, 191, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 11, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 217, 255, 255, 255, 255, 255, 255, 255, 255, 230, 49, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 137, 228, 255, 255, 255, 255, 235, 148, 22, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 2, 30, 55, 57, 32, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 52, 55, 55, 55, 9, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ü' / 0xdc
        'Ü' => Some(&[
            &[
                0, 0, 0, 0, 96, 231, 220, 68, 0, 0, 44, 209, 236, 130, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 236, 255, 255, 204, 0, 0, 161, 255, 255, 255, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 235, 255, 255, 203, 0, 0, 160, 255, 255, 255, 29, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 95, 235, 223, 67, 0, 0, 41, 210, 240, 129, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 9, 0, 0, 0, 0, 0, 66, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 238, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 44, 0, 0, 0, 0, 0, 248, 255, 255, 247, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 255, 255, 76, 0, 0, 0, 0, 23, 255, 255, 255, 224, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 160, 0, 0, 0, 0, 104, 255, 255, 255, 187, 0, 0, 0,
            ],
            &[
                0, 0, 59, 255, 255, 255, 254, 136, 45, 38, 103, 243, 255, 255, 255, 104, 0, 0, 0,
            ],
            &[
                0, 0, 1, 191, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 11, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 217, 255, 255, 255, 255, 255, 255, 255, 255, 230, 49, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 137, 228, 255, 255, 255, 255, 235, 148, 22, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 2, 30, 55, 57, 32, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ý' / 0xdd
        'Ý' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 255, 255, 179, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 49, 250, 255, 255, 180, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 8, 212, 255, 255, 165, 4, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 143, 255, 255, 134, 1, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 19, 26, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 28, 71, 71, 71, 51, 0, 0, 0, 0, 0, 0, 35, 71, 71, 71, 42, 0, 0,
            ],
            &[
                0, 28, 249, 255, 255, 242, 13, 0, 0, 0, 0, 0, 197, 255, 255, 255, 74, 0, 0,
            ],
            &[
                0, 0, 157, 255, 255, 255, 108, 0, 0, 0, 0, 48, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 37, 253, 255, 255, 215, 0, 0, 0, 0, 155, 255, 255, 255, 88, 0, 0, 0,
            ],
            &[
                0, 0, 0, 169, 255, 255, 255, 68, 0, 0, 17, 245, 255, 255, 219, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 48, 255, 255, 255, 176, 0, 0, 114, 255, 255, 255, 101, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 181, 255, 255, 252, 31, 1, 219, 255, 255, 229, 6, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 59, 255, 255, 255, 136, 72, 255, 255, 255, 114, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 193, 255, 255, 236, 187, 255, 255, 237, 12, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 72, 255, 255, 255, 255, 255, 255, 128, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 205, 255, 255, 255, 255, 244, 18, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 255, 141, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 2, 230, 255, 255, 255, 34, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 55, 55, 55, 55, 9, 0, 0, 0, 0],
        ]),
        // letter: 'Þ' / 0xde
        'Þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 71, 71, 66, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 241, 9, 9, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 250, 219, 155, 48, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 110, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 254, 64, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 244, 72, 72, 78, 116, 229, 255, 255, 255, 184, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 42, 252, 255, 255, 247, 3, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 217, 255, 255, 255, 21, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 207, 255, 255, 255, 17, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 22, 249, 255, 255, 246, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 1, 51, 200, 255, 255, 255, 179, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 254, 240, 240, 251, 255, 255, 255, 255, 255, 73, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 143, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 213, 94, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 246, 97, 97, 92, 69, 38, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ß' / 0xdf
        'ß' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 12, 85, 137, 159, 153, 126, 63, 2, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 100, 239, 255, 255, 255, 255, 255, 255, 211, 54, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 102, 255, 255, 255, 255, 255, 255, 255, 255, 255, 244, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 11, 240, 255, 255, 255, 218, 151, 150, 234, 255, 255, 255, 163, 0, 0, 0, 0,
            ],
            &[
                0, 0, 83, 255, 255, 255, 204, 11, 0, 0, 37, 255, 255, 255, 211, 0, 0, 0, 0,
            ],
            &[
                0, 0, 116, 255, 255, 255, 112, 0, 0, 0, 9, 250, 255, 255, 189, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 76, 0, 0, 0, 104, 255, 255, 255, 119, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 74, 0, 0, 87, 251, 255, 255, 204, 6, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 74, 0, 70, 253, 255, 255, 203, 18, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 74, 0, 209, 255, 255, 233, 15, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 74, 4, 252, 255, 255, 220, 4, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 74, 0, 225, 255, 255, 255, 187, 19, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 74, 0, 100, 255, 255, 255, 255, 225, 46, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 74, 0, 0, 124, 254, 255, 255, 255, 242, 54, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 74, 0, 0, 0, 78, 237, 255, 255, 255, 217, 7, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 74, 0, 0, 0, 0, 31, 226, 255, 255, 255, 72, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 74, 0, 0, 0, 0, 0, 129, 255, 255, 255, 100, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 74, 84, 99, 9, 0, 3, 180, 255, 255, 255, 83, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 74, 112, 255, 252, 215, 230, 255, 255, 255, 251, 25, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 74, 112, 255, 255, 255, 255, 255, 255, 255, 139, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 74, 60, 208, 255, 255, 255, 255, 228, 112, 2, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 26, 54, 60, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'à' / 0xe0
        'à' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 23, 170, 174, 174, 174, 39, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 92, 252, 255, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 87, 250, 255, 255, 105, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 72, 244, 255, 244, 40, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 47, 162, 163, 105, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 63, 143, 200, 237, 252, 250, 230, 183, 101, 6, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 214, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 200, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 178, 182, 107, 65, 39, 49, 109, 242, 255, 255, 255, 52, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 255, 255, 98, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 4, 23, 31, 38, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 50, 160, 225, 253, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 112, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 40, 254, 255, 255, 254, 158, 94, 60, 50, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 127, 255, 255, 255, 118, 0, 0, 0, 0, 78, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 255, 55, 0, 0, 0, 0, 141, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 152, 5, 0, 10, 107, 252, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 85, 255, 255, 255, 255, 239, 218, 252, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 2, 203, 255, 255, 255, 255, 255, 255, 249, 105, 248, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 28, 185, 255, 255, 255, 255, 189, 59, 0, 176, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 23, 59, 54, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'á' / 0xe1
        'á' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 171, 174, 174, 174, 44, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 153, 255, 255, 255, 125, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 71, 255, 255, 255, 118, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 20, 230, 255, 251, 98, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 81, 163, 163, 68, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 63, 143, 200, 237, 252, 250, 230, 183, 101, 6, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 214, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 200, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 178, 182, 107, 65, 39, 49, 109, 242, 255, 255, 255, 52, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 255, 255, 98, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 4, 23, 31, 38, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 50, 160, 225, 253, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 112, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 40, 254, 255, 255, 254, 158, 94, 60, 50, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 127, 255, 255, 255, 118, 0, 0, 0, 0, 78, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 255, 55, 0, 0, 0, 0, 141, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 152, 5, 0, 10, 107, 252, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 85, 255, 255, 255, 255, 239, 218, 252, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 2, 203, 255, 255, 255, 255, 255, 255, 249, 105, 248, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 28, 185, 255, 255, 255, 255, 189, 59, 0, 176, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 23, 59, 54, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'â' / 0xe2
        'â' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 1, 150, 174, 174, 174, 93, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 108, 255, 255, 255, 255, 244, 35, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 247, 255, 242, 165, 255, 255, 206, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 22, 223, 255, 237, 56, 0, 131, 255, 255, 158, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 93, 163, 158, 33, 0, 0, 0, 85, 163, 163, 33, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 63, 143, 200, 237, 252, 250, 230, 183, 101, 6, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 214, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 200, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 178, 182, 107, 65, 39, 49, 109, 242, 255, 255, 255, 52, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 255, 255, 98, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 4, 23, 31, 38, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 50, 160, 225, 253, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 112, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 40, 254, 255, 255, 254, 158, 94, 60, 50, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 127, 255, 255, 255, 118, 0, 0, 0, 0, 78, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 255, 55, 0, 0, 0, 0, 141, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 152, 5, 0, 10, 107, 252, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 85, 255, 255, 255, 255, 239, 218, 252, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 2, 203, 255, 255, 255, 255, 255, 255, 249, 105, 248, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 28, 185, 255, 255, 255, 255, 189, 59, 0, 176, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 23, 59, 54, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ã' / 0xe3
        'ã' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 2, 56, 81, 25, 0, 0, 0, 60, 105, 43, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 170, 255, 255, 251, 163, 55, 28, 216, 255, 75, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 77, 255, 255, 255, 255, 255, 255, 255, 255, 248, 23, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 159, 255, 181, 86, 158, 249, 255, 255, 255, 132, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 129, 170, 41, 0, 0, 24, 114, 150, 89, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 63, 143, 200, 237, 252, 250, 230, 183, 101, 6, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 214, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 200, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 178, 182, 107, 65, 39, 49, 109, 242, 255, 255, 255, 52, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 255, 255, 98, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 4, 23, 31, 38, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 50, 160, 225, 253, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 112, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 40, 254, 255, 255, 254, 158, 94, 60, 50, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 127, 255, 255, 255, 118, 0, 0, 0, 0, 78, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 255, 55, 0, 0, 0, 0, 141, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 152, 5, 0, 10, 107, 252, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 85, 255, 255, 255, 255, 239, 218, 252, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 2, 203, 255, 255, 255, 255, 255, 255, 249, 105, 248, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 28, 185, 255, 255, 255, 255, 189, 59, 0, 176, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 23, 59, 54, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ä' / 0xe4
        'ä' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 2, 80, 96, 15, 0, 0, 0, 51, 102, 41, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 133, 255, 255, 208, 0, 0, 59, 254, 255, 249, 40, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 191, 255, 255, 255, 15, 0, 116, 255, 255, 255, 96, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 140, 255, 255, 215, 0, 0, 63, 255, 255, 252, 45, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 6, 101, 118, 24, 0, 0, 0, 68, 124, 57, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 63, 143, 200, 237, 252, 250, 230, 183, 101, 6, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 214, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 200, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 178, 182, 107, 65, 39, 49, 109, 242, 255, 255, 255, 52, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 255, 255, 98, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 4, 23, 31, 38, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 50, 160, 225, 253, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 112, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 40, 254, 255, 255, 254, 158, 94, 60, 50, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 127, 255, 255, 255, 118, 0, 0, 0, 0, 78, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 255, 55, 0, 0, 0, 0, 141, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 152, 5, 0, 10, 107, 252, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 85, 255, 255, 255, 255, 239, 218, 252, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 2, 203, 255, 255, 255, 255, 255, 255, 249, 105, 248, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 28, 185, 255, 255, 255, 255, 189, 59, 0, 176, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 23, 59, 54, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'å' / 0xe5
        'å' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 30, 114, 133, 82, 1, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 55, 244, 255, 255, 255, 181, 2, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 187, 255, 172, 102, 230, 255, 82, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 225, 255, 29, 0, 144, 255, 123, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 190, 255, 154, 85, 224, 255, 86, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 61, 249, 255, 255, 255, 194, 3, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 40, 131, 150, 99, 4, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 63, 143, 200, 237, 252, 250, 230, 183, 101, 6, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 214, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 200, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 178, 182, 107, 65, 39, 49, 109, 242, 255, 255, 255, 52, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 255, 255, 98, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 4, 23, 31, 38, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 50, 160, 225, 253, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 112, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 40, 254, 255, 255, 254, 158, 94, 60, 50, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 127, 255, 255, 255, 118, 0, 0, 0, 0, 78, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 255, 55, 0, 0, 0, 0, 141, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 152, 5, 0, 10, 107, 252, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 85, 255, 255, 255, 255, 239, 218, 252, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 2, 203, 255, 255, 255, 255, 255, 255, 249, 105, 248, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 28, 185, 255, 255, 255, 255, 189, 59, 0, 176, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 23, 59, 54, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'æ' / 0xe6
        'æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 38, 144, 209, 243, 248, 203, 92, 7, 128, 208, 247, 220, 116, 3, 0, 0, 0,
            ],
            &[
                0, 0, 203, 255, 255, 255, 255, 255, 255, 230, 255, 255, 255, 255, 255, 141, 0, 0, 0,
            ],
            &[
                0, 0, 86, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 254, 37, 0, 0,
            ],
            &[
                0, 0, 3, 135, 69, 38, 80, 237, 255, 255, 255, 143, 61, 231, 255, 255, 131, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 147, 255, 255, 249, 11, 0, 136, 255, 255, 186, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 134, 255, 255, 222, 0, 0, 101, 255, 255, 218, 0, 0,
            ],
            &[
                0, 0, 5, 114, 199, 236, 252, 255, 255, 255, 253, 239, 239, 245, 255, 255, 233, 0, 0,
            ],
            &[
                0, 2, 187, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 237, 0,
                0,
            ],
            &[
                0, 84, 255, 255, 255, 156, 65, 160, 255, 255, 229, 139, 139, 139, 139, 139, 130, 0,
                0,
            ],
            &[
                0, 155, 255, 255, 223, 2, 0, 137, 255, 255, 218, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 179, 255, 255, 191, 0, 0, 164, 255, 255, 255, 35, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 163, 255, 255, 241, 9, 15, 237, 255, 255, 255, 193, 16, 0, 0, 49, 45, 0, 0,
            ],
            &[
                0, 126, 255, 255, 255, 232, 236, 255, 253, 255, 255, 255, 242, 190, 211, 253, 79,
                0, 0,
            ],
            &[
                0, 28, 248, 255, 255, 255, 255, 255, 105, 168, 255, 255, 255, 255, 255, 255, 79, 0,
                0,
            ],
            &[
                0, 0, 78, 237, 255, 255, 242, 112, 0, 4, 127, 238, 255, 255, 255, 180, 26, 0, 0,
            ],
            &[
                0, 0, 0, 5, 45, 54, 12, 0, 0, 0, 0, 7, 47, 57, 21, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ç' / 0xe7
        'ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 58, 152, 213, 239, 251, 232, 207, 156, 74, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 152, 255, 255, 255, 255, 255, 255, 255, 255, 210, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 255, 255, 255, 255, 255, 255, 255, 255, 117, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 61, 255, 255, 255, 255, 208, 112, 85, 103, 155, 225, 26, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 153, 255, 255, 255, 187, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 226, 255, 255, 255, 52, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 2, 252, 255, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 20, 255, 255, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 13, 255, 255, 255, 236, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 246, 255, 255, 255, 28, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 199, 255, 255, 255, 139, 0, 0, 0, 0, 0, 0, 17, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 118, 255, 255, 255, 255, 147, 47, 20, 46, 91, 183, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 15, 226, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 57, 233, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 25, 149, 233, 255, 255, 255, 255, 248, 180, 55, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 3, 108, 255, 253, 35, 8, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 188, 255, 252, 165, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 66, 165, 255, 255, 143, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 29, 255, 255, 200, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 142, 142, 204, 255, 255, 168, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 229, 38, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 68, 112, 110, 80, 8, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'è' / 0xe8
        'è' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 17, 167, 174, 174, 174, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 81, 250, 255, 255, 201, 3, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 77, 247, 255, 255, 119, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 62, 239, 255, 248, 49, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 40, 160, 163, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 21, 120, 203, 235, 250, 226, 178, 81, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 64, 235, 255, 255, 255, 255, 255, 255, 255, 174, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 43, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 197, 255, 255, 255, 168, 39, 5, 52, 219, 255, 255, 255, 49, 0, 0, 0,
            ],
            &[
                0, 0, 38, 255, 255, 255, 192, 2, 0, 0, 0, 29, 246, 255, 255, 137, 0, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 255, 97, 0, 0, 0, 0, 0, 186, 255, 255, 184, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 238, 232, 232, 232, 232, 232, 248, 255, 255, 209, 0, 0, 0,
            ],
            &[
                0, 0, 159, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 206, 195, 195, 195, 195, 195, 195, 195, 195, 165, 0, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 255, 199, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 198, 255, 255, 255, 206, 81, 15, 0, 9, 38, 95, 171, 94, 0, 0, 0,
            ],
            &[
                0, 0, 0, 42, 244, 255, 255, 255, 255, 255, 247, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 68, 231, 255, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 132, 218, 255, 255, 255, 255, 255, 230, 152, 32, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 45, 63, 45, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'é' / 0xe9
        'é' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 11, 167, 174, 174, 174, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 140, 255, 255, 255, 139, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 59, 253, 255, 255, 131, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 14, 222, 255, 253, 110, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 72, 163, 163, 76, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 21, 120, 203, 235, 250, 226, 178, 81, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 64, 235, 255, 255, 255, 255, 255, 255, 255, 174, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 43, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 197, 255, 255, 255, 168, 39, 5, 52, 219, 255, 255, 255, 49, 0, 0, 0,
            ],
            &[
                0, 0, 38, 255, 255, 255, 192, 2, 0, 0, 0, 29, 246, 255, 255, 137, 0, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 255, 97, 0, 0, 0, 0, 0, 186, 255, 255, 184, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 238, 232, 232, 232, 232, 232, 248, 255, 255, 209, 0, 0, 0,
            ],
            &[
                0, 0, 159, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 206, 195, 195, 195, 195, 195, 195, 195, 195, 165, 0, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 255, 199, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 198, 255, 255, 255, 206, 81, 15, 0, 9, 38, 95, 171, 94, 0, 0, 0,
            ],
            &[
                0, 0, 0, 42, 244, 255, 255, 255, 255, 255, 247, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 68, 231, 255, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 132, 218, 255, 255, 255, 255, 255, 230, 152, 32, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 45, 63, 45, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ê' / 0xea
        'ê' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 142, 174, 174, 174, 103, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 94, 255, 255, 255, 255, 249, 45, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 39, 243, 255, 246, 161, 255, 255, 215, 12, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 16, 215, 255, 241, 66, 0, 119, 254, 255, 171, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 84, 163, 160, 40, 0, 0, 0, 76, 163, 163, 42, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 21, 120, 203, 235, 250, 226, 178, 81, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 64, 235, 255, 255, 255, 255, 255, 255, 255, 174, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 43, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 197, 255, 255, 255, 168, 39, 5, 52, 219, 255, 255, 255, 49, 0, 0, 0,
            ],
            &[
                0, 0, 38, 255, 255, 255, 192, 2, 0, 0, 0, 29, 246, 255, 255, 137, 0, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 255, 97, 0, 0, 0, 0, 0, 186, 255, 255, 184, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 238, 232, 232, 232, 232, 232, 248, 255, 255, 209, 0, 0, 0,
            ],
            &[
                0, 0, 159, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 206, 195, 195, 195, 195, 195, 195, 195, 195, 165, 0, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 255, 199, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 198, 255, 255, 255, 206, 81, 15, 0, 9, 38, 95, 171, 94, 0, 0, 0,
            ],
            &[
                0, 0, 0, 42, 244, 255, 255, 255, 255, 255, 247, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 68, 231, 255, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 132, 218, 255, 255, 255, 255, 255, 230, 152, 32, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 45, 63, 45, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ë' / 0xeb
        'ë' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 1, 75, 98, 18, 0, 0, 0, 46, 102, 46, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 120, 255, 255, 219, 3, 0, 47, 252, 255, 252, 51, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 177, 255, 255, 255, 29, 0, 102, 255, 255, 255, 110, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 127, 255, 255, 225, 4, 0, 51, 253, 255, 254, 56, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 4, 96, 120, 29, 0, 0, 0, 62, 124, 63, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 21, 120, 203, 235, 250, 226, 178, 81, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 64, 235, 255, 255, 255, 255, 255, 255, 255, 174, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 43, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 197, 255, 255, 255, 168, 39, 5, 52, 219, 255, 255, 255, 49, 0, 0, 0,
            ],
            &[
                0, 0, 38, 255, 255, 255, 192, 2, 0, 0, 0, 29, 246, 255, 255, 137, 0, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 255, 97, 0, 0, 0, 0, 0, 186, 255, 255, 184, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 238, 232, 232, 232, 232, 232, 248, 255, 255, 209, 0, 0, 0,
            ],
            &[
                0, 0, 159, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 206, 195, 195, 195, 195, 195, 195, 195, 195, 165, 0, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 255, 199, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 198, 255, 255, 255, 206, 81, 15, 0, 9, 38, 95, 171, 94, 0, 0, 0,
            ],
            &[
                0, 0, 0, 42, 244, 255, 255, 255, 255, 255, 247, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 68, 231, 255, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 132, 218, 255, 255, 255, 255, 255, 230, 152, 32, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 45, 63, 45, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ì' / 0xec
        'ì' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 57, 174, 174, 174, 166, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 144, 255, 255, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 137, 255, 255, 252, 54, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 117, 254, 255, 217, 12, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 81, 163, 163, 69, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 123, 189, 189, 189, 189, 189, 189, 189, 85, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 12, 42, 70, 98, 183, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 17, 146, 255, 255, 255, 141, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 147, 189, 231, 255, 255, 255, 255, 255, 255, 254, 220, 173, 110, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'í' / 0xed
        'í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 52, 174, 174, 174, 166, 15, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 4, 205, 255, 255, 250, 76, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 125, 255, 255, 245, 72, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 54, 251, 255, 235, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 116, 163, 160, 36, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 123, 189, 189, 189, 189, 189, 189, 189, 85, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 12, 42, 70, 98, 183, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 17, 146, 255, 255, 255, 141, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 147, 189, 231, 255, 255, 255, 255, 255, 255, 254, 220, 173, 110, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'î' / 0xee
        'î' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 19, 171, 174, 174, 174, 56, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 163, 255, 255, 255, 255, 215, 9, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 96, 255, 255, 218, 189, 255, 255, 158, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 54, 246, 255, 212, 26, 8, 178, 255, 255, 103, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 128, 163, 143, 13, 0, 0, 2, 118, 163, 157, 3, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 123, 189, 189, 189, 189, 189, 189, 189, 85, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 12, 42, 70, 98, 183, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 17, 146, 255, 255, 255, 141, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 147, 189, 231, 255, 255, 255, 255, 255, 255, 254, 220, 173, 110, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ï' / 0xef
        'ï' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 12, 94, 83, 4, 0, 0, 0, 71, 99, 22, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 189, 255, 255, 153, 0, 0, 113, 255, 255, 226, 8, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 246, 255, 255, 215, 0, 0, 172, 255, 255, 255, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 196, 255, 255, 161, 0, 0, 118, 255, 255, 232, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 20, 116, 104, 9, 0, 0, 2, 91, 121, 34, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 123, 189, 189, 189, 189, 189, 189, 189, 85, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 12, 42, 70, 98, 183, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 17, 146, 255, 255, 255, 141, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 147, 189, 231, 255, 255, 255, 255, 255, 255, 254, 220, 173, 110, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ð' / 0xf0
        'ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 67, 0, 0, 0, 0, 68, 47, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 14, 233, 255, 197, 49, 25, 167, 255, 208, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 93, 255, 255, 255, 255, 255, 255, 255, 219, 36, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 71, 255, 255, 255, 255, 255, 136, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 155, 255, 255, 255, 255, 255, 181, 3, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 177, 255, 255, 230, 232, 255, 255, 255, 139, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 41, 244, 141, 12, 1, 166, 255, 255, 253, 53, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 19, 4, 18, 4, 6, 212, 255, 255, 196, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 124, 217, 253, 255, 250, 191, 127, 255, 255, 255, 47, 0, 0, 0,
            ],
            &[
                0, 0, 0, 33, 215, 255, 255, 255, 255, 255, 255, 246, 255, 255, 255, 139, 0, 0, 0,
            ],
            &[
                0, 0, 4, 202, 255, 255, 255, 252, 211, 210, 251, 255, 255, 255, 255, 199, 0, 0, 0,
            ],
            &[
                0, 0, 84, 255, 255, 255, 221, 35, 0, 0, 30, 206, 255, 255, 255, 248, 0, 0, 0,
            ],
            &[
                0, 0, 170, 255, 255, 255, 89, 0, 0, 0, 0, 45, 255, 255, 255, 255, 13, 0, 0,
            ],
            &[
                0, 0, 208, 255, 255, 255, 27, 0, 0, 0, 0, 0, 237, 255, 255, 255, 26, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 10, 0, 0, 0, 0, 0, 217, 255, 255, 255, 12, 0, 0,
            ],
            &[
                0, 0, 209, 255, 255, 255, 25, 0, 0, 0, 0, 1, 241, 255, 255, 237, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 86, 0, 0, 0, 0, 51, 255, 255, 255, 177, 0, 0, 0,
            ],
            &[
                0, 0, 82, 255, 255, 255, 215, 19, 0, 0, 9, 192, 255, 255, 255, 78, 0, 0, 0,
            ],
            &[
                0, 0, 4, 200, 255, 255, 255, 238, 168, 166, 230, 255, 255, 255, 186, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 27, 213, 255, 255, 255, 255, 255, 255, 255, 255, 200, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 8, 126, 222, 255, 255, 255, 255, 223, 116, 7, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 30, 56, 57, 31, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ñ' / 0xf1
        'ñ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 5, 63, 77, 19, 0, 0, 0, 71, 105, 32, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 8, 191, 255, 255, 246, 151, 47, 35, 234, 255, 48, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 104, 255, 255, 255, 255, 255, 255, 255, 255, 236, 7, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 187, 255, 163, 89, 170, 252, 255, 255, 255, 105, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 147, 170, 23, 0, 0, 32, 121, 148, 75, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 97, 189, 189, 110, 0, 28, 144, 216, 248, 245, 200, 99, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 224, 42, 236, 255, 255, 255, 255, 255, 255, 156, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 244, 255, 255, 255, 255, 255, 255, 255, 255, 69, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 236, 114, 71, 125, 248, 255, 255, 255, 158, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 254, 54, 0, 0, 0, 118, 255, 255, 255, 202, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 197, 0, 0, 0, 0, 42, 255, 255, 255, 216, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 138, 0, 0, 0, 0, 18, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 115, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ò' / 0xf2
        'ò' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 57, 174, 174, 174, 166, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 144, 255, 255, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 137, 255, 255, 252, 54, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 117, 254, 255, 217, 12, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 81, 163, 163, 69, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 31, 134, 209, 239, 247, 219, 164, 62, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 247, 255, 255, 255, 255, 255, 255, 255, 151, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 82, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 233, 255, 255, 255, 182, 74, 51, 129, 253, 255, 255, 254, 57, 0, 0, 0,
            ],
            &[
                0, 0, 95, 255, 255, 255, 199, 4, 0, 0, 0, 135, 255, 255, 255, 154, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 91, 0, 0, 0, 0, 30, 255, 255, 255, 229, 0, 0, 0,
            ],
            &[
                0, 0, 206, 255, 255, 255, 42, 0, 0, 0, 0, 0, 237, 255, 255, 254, 4, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 18, 0, 0, 0, 0, 0, 221, 255, 255, 255, 24, 0, 0,
            ],
            &[
                0, 0, 220, 255, 255, 255, 27, 0, 0, 0, 0, 0, 231, 255, 255, 255, 14, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 68, 0, 0, 0, 0, 16, 254, 255, 255, 240, 0, 0, 0,
            ],
            &[
                0, 0, 132, 255, 255, 255, 160, 0, 0, 0, 0, 100, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 36, 253, 255, 255, 254, 116, 5, 0, 65, 238, 255, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 255, 255, 249, 241, 255, 255, 255, 255, 194, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 7, 174, 255, 255, 255, 255, 255, 255, 255, 255, 209, 25, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 97, 212, 255, 255, 255, 255, 228, 127, 11, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 55, 60, 35, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ó' / 0xf3
        'ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 52, 174, 174, 174, 166, 15, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 4, 205, 255, 255, 250, 76, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 125, 255, 255, 245, 72, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 54, 251, 255, 235, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 116, 163, 160, 36, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 31, 134, 209, 239, 247, 219, 164, 62, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 247, 255, 255, 255, 255, 255, 255, 255, 151, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 82, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 233, 255, 255, 255, 182, 74, 51, 129, 253, 255, 255, 254, 57, 0, 0, 0,
            ],
            &[
                0, 0, 95, 255, 255, 255, 199, 4, 0, 0, 0, 135, 255, 255, 255, 154, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 91, 0, 0, 0, 0, 30, 255, 255, 255, 229, 0, 0, 0,
            ],
            &[
                0, 0, 206, 255, 255, 255, 42, 0, 0, 0, 0, 0, 237, 255, 255, 254, 4, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 18, 0, 0, 0, 0, 0, 221, 255, 255, 255, 24, 0, 0,
            ],
            &[
                0, 0, 220, 255, 255, 255, 27, 0, 0, 0, 0, 0, 231, 255, 255, 255, 14, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 68, 0, 0, 0, 0, 16, 254, 255, 255, 240, 0, 0, 0,
            ],
            &[
                0, 0, 132, 255, 255, 255, 160, 0, 0, 0, 0, 100, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 36, 253, 255, 255, 254, 116, 5, 0, 65, 238, 255, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 255, 255, 249, 241, 255, 255, 255, 255, 194, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 7, 174, 255, 255, 255, 255, 255, 255, 255, 255, 209, 25, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 97, 212, 255, 255, 255, 255, 228, 127, 11, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 55, 60, 35, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ô' / 0xf4
        'ô' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 19, 171, 174, 174, 174, 56, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 163, 255, 255, 255, 255, 215, 9, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 96, 255, 255, 218, 189, 255, 255, 158, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 54, 246, 255, 212, 26, 8, 178, 255, 255, 103, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 128, 163, 143, 13, 0, 0, 2, 118, 163, 157, 3, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 31, 134, 209, 239, 247, 219, 164, 62, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 247, 255, 255, 255, 255, 255, 255, 255, 151, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 82, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 233, 255, 255, 255, 182, 74, 51, 129, 253, 255, 255, 254, 57, 0, 0, 0,
            ],
            &[
                0, 0, 95, 255, 255, 255, 199, 4, 0, 0, 0, 135, 255, 255, 255, 154, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 91, 0, 0, 0, 0, 30, 255, 255, 255, 229, 0, 0, 0,
            ],
            &[
                0, 0, 206, 255, 255, 255, 42, 0, 0, 0, 0, 0, 237, 255, 255, 254, 4, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 18, 0, 0, 0, 0, 0, 221, 255, 255, 255, 24, 0, 0,
            ],
            &[
                0, 0, 220, 255, 255, 255, 27, 0, 0, 0, 0, 0, 231, 255, 255, 255, 14, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 68, 0, 0, 0, 0, 16, 254, 255, 255, 240, 0, 0, 0,
            ],
            &[
                0, 0, 132, 255, 255, 255, 160, 0, 0, 0, 0, 100, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 36, 253, 255, 255, 254, 116, 5, 0, 65, 238, 255, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 255, 255, 249, 241, 255, 255, 255, 255, 194, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 7, 174, 255, 255, 255, 255, 255, 255, 255, 255, 209, 25, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 97, 212, 255, 255, 255, 255, 228, 127, 11, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 55, 60, 35, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'õ' / 0xf5
        'õ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 9, 70, 73, 13, 0, 0, 0, 83, 105, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 18, 208, 255, 255, 241, 138, 40, 47, 249, 255, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 132, 255, 255, 255, 255, 255, 255, 255, 255, 216, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 214, 255, 143, 94, 183, 255, 255, 255, 252, 80, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 165, 167, 7, 0, 0, 42, 128, 145, 62, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 31, 134, 209, 239, 247, 219, 164, 62, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 247, 255, 255, 255, 255, 255, 255, 255, 151, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 82, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 233, 255, 255, 255, 182, 74, 51, 129, 253, 255, 255, 254, 57, 0, 0, 0,
            ],
            &[
                0, 0, 95, 255, 255, 255, 199, 4, 0, 0, 0, 135, 255, 255, 255, 154, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 91, 0, 0, 0, 0, 30, 255, 255, 255, 229, 0, 0, 0,
            ],
            &[
                0, 0, 206, 255, 255, 255, 42, 0, 0, 0, 0, 0, 237, 255, 255, 254, 4, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 18, 0, 0, 0, 0, 0, 221, 255, 255, 255, 24, 0, 0,
            ],
            &[
                0, 0, 220, 255, 255, 255, 27, 0, 0, 0, 0, 0, 231, 255, 255, 255, 14, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 68, 0, 0, 0, 0, 16, 254, 255, 255, 240, 0, 0, 0,
            ],
            &[
                0, 0, 132, 255, 255, 255, 160, 0, 0, 0, 0, 100, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 36, 253, 255, 255, 254, 116, 5, 0, 65, 238, 255, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 255, 255, 249, 241, 255, 255, 255, 255, 194, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 7, 174, 255, 255, 255, 255, 255, 255, 255, 255, 209, 25, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 97, 212, 255, 255, 255, 255, 228, 127, 11, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 55, 60, 35, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ö' / 0xf6
        'ö' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 12, 94, 83, 4, 0, 0, 0, 71, 99, 22, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 189, 255, 255, 153, 0, 0, 113, 255, 255, 226, 8, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 246, 255, 255, 215, 0, 0, 172, 255, 255, 255, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 196, 255, 255, 161, 0, 0, 118, 255, 255, 232, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 20, 116, 104, 9, 0, 0, 2, 91, 121, 34, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 31, 134, 209, 239, 247, 219, 164, 62, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 247, 255, 255, 255, 255, 255, 255, 255, 151, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 82, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 233, 255, 255, 255, 182, 74, 51, 129, 253, 255, 255, 254, 57, 0, 0, 0,
            ],
            &[
                0, 0, 95, 255, 255, 255, 199, 4, 0, 0, 0, 135, 255, 255, 255, 154, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 91, 0, 0, 0, 0, 30, 255, 255, 255, 229, 0, 0, 0,
            ],
            &[
                0, 0, 206, 255, 255, 255, 42, 0, 0, 0, 0, 0, 237, 255, 255, 254, 4, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 18, 0, 0, 0, 0, 0, 221, 255, 255, 255, 24, 0, 0,
            ],
            &[
                0, 0, 220, 255, 255, 255, 27, 0, 0, 0, 0, 0, 231, 255, 255, 255, 14, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 68, 0, 0, 0, 0, 16, 254, 255, 255, 240, 0, 0, 0,
            ],
            &[
                0, 0, 132, 255, 255, 255, 160, 0, 0, 0, 0, 100, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 36, 253, 255, 255, 254, 116, 5, 0, 65, 238, 255, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 255, 255, 249, 241, 255, 255, 255, 255, 194, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 7, 174, 255, 255, 255, 255, 255, 255, 255, 255, 209, 25, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 97, 212, 255, 255, 255, 255, 228, 127, 11, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 55, 60, 35, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '÷' / 0xf7
        '÷' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 72, 88, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 113, 255, 255, 171, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 172, 255, 255, 230, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 134, 255, 255, 192, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 11, 125, 143, 30, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 76, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 25, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 71, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 18, 143, 160, 42, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 142, 255, 255, 200, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 171, 255, 255, 229, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 101, 255, 255, 159, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 60, 78, 4, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ø' / 0xf8
        'ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 79, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 31, 134, 209, 239, 247, 219, 164, 171, 255, 175, 13, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 247, 255, 255, 255, 255, 255, 255, 255, 255, 210, 4, 0, 0, 0,
            ],
            &[
                0, 0, 0, 82, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 233, 255, 255, 255, 182, 74, 51, 246, 255, 255, 255, 254, 57, 0, 0, 0,
            ],
            &[
                0, 0, 95, 255, 255, 255, 199, 4, 0, 42, 247, 255, 255, 255, 255, 154, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 91, 0, 4, 203, 255, 255, 255, 255, 255, 229, 0, 0, 0,
            ],
            &[
                0, 0, 206, 255, 255, 255, 42, 0, 124, 255, 255, 107, 237, 255, 255, 254, 4, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 18, 47, 250, 255, 188, 1, 221, 255, 255, 255, 24, 0, 0,
            ],
            &[
                0, 0, 220, 255, 255, 255, 33, 209, 255, 241, 30, 0, 231, 255, 255, 255, 14, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 201, 255, 255, 97, 0, 16, 254, 255, 255, 240, 0, 0, 0,
            ],
            &[
                0, 0, 132, 255, 255, 255, 255, 255, 179, 0, 0, 100, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 36, 253, 255, 255, 255, 255, 30, 0, 65, 238, 255, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 255, 255, 249, 241, 255, 255, 255, 255, 194, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 67, 255, 255, 255, 255, 255, 255, 255, 255, 255, 209, 25, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 196, 255, 255, 232, 255, 255, 255, 255, 228, 127, 11, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 21, 173, 78, 0, 26, 55, 60, 35, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ù' / 0xf9
        'ù' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 57, 174, 174, 174, 166, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 144, 255, 255, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 137, 255, 255, 252, 54, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 117, 254, 255, 217, 12, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 81, 163, 163, 69, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 123, 189, 189, 189, 50, 0, 0, 0, 0, 37, 189, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 55, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 68, 0, 0, 0, 0, 69, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 83, 0, 0, 0, 0, 108, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 155, 255, 255, 255, 132, 0, 0, 0, 0, 192, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 122, 255, 255, 255, 237, 67, 5, 23, 142, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 46, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 157, 255, 255, 255, 255, 255, 255, 255, 130, 199, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 4, 134, 245, 255, 255, 255, 219, 96, 0, 109, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 7, 51, 59, 27, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ú' / 0xfa
        'ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 52, 174, 174, 174, 166, 15, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 4, 205, 255, 255, 250, 76, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 125, 255, 255, 245, 72, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 54, 251, 255, 235, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 116, 163, 160, 36, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 123, 189, 189, 189, 50, 0, 0, 0, 0, 37, 189, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 55, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 68, 0, 0, 0, 0, 69, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 83, 0, 0, 0, 0, 108, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 155, 255, 255, 255, 132, 0, 0, 0, 0, 192, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 122, 255, 255, 255, 237, 67, 5, 23, 142, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 46, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 157, 255, 255, 255, 255, 255, 255, 255, 130, 199, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 4, 134, 245, 255, 255, 255, 219, 96, 0, 109, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 7, 51, 59, 27, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'û' / 0xfb
        'û' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 19, 171, 174, 174, 174, 56, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 163, 255, 255, 255, 255, 215, 9, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 96, 255, 255, 218, 189, 255, 255, 158, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 54, 246, 255, 212, 26, 8, 178, 255, 255, 103, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 128, 163, 143, 13, 0, 0, 2, 118, 163, 157, 3, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 123, 189, 189, 189, 50, 0, 0, 0, 0, 37, 189, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 55, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 68, 0, 0, 0, 0, 69, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 83, 0, 0, 0, 0, 108, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 155, 255, 255, 255, 132, 0, 0, 0, 0, 192, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 122, 255, 255, 255, 237, 67, 5, 23, 142, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 46, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 157, 255, 255, 255, 255, 255, 255, 255, 130, 199, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 4, 134, 245, 255, 255, 255, 219, 96, 0, 109, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 7, 51, 59, 27, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ü' / 0xfc
        'ü' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 12, 94, 83, 4, 0, 0, 0, 71, 99, 22, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 189, 255, 255, 153, 0, 0, 113, 255, 255, 226, 8, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 246, 255, 255, 215, 0, 0, 172, 255, 255, 255, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 196, 255, 255, 161, 0, 0, 118, 255, 255, 232, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 20, 116, 104, 9, 0, 0, 2, 91, 121, 34, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 123, 189, 189, 189, 50, 0, 0, 0, 0, 37, 189, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 55, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 68, 0, 0, 0, 0, 69, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 83, 0, 0, 0, 0, 108, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 155, 255, 255, 255, 132, 0, 0, 0, 0, 192, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 122, 255, 255, 255, 237, 67, 5, 23, 142, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 46, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 157, 255, 255, 255, 255, 255, 255, 255, 130, 199, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 4, 134, 245, 255, 255, 255, 219, 96, 0, 109, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 7, 51, 59, 27, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ý' / 0xfd
        'ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 52, 174, 174, 174, 166, 15, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 4, 205, 255, 255, 250, 76, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 125, 255, 255, 245, 72, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 54, 251, 255, 235, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 116, 163, 160, 36, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 59, 189, 189, 189, 175, 1, 0, 0, 0, 0, 0, 125, 189, 189, 189, 99, 0, 0,
            ],
            &[
                0, 8, 238, 255, 255, 255, 56, 0, 0, 0, 0, 1, 233, 255, 255, 255, 53, 0, 0,
            ],
            &[
                0, 0, 146, 255, 255, 255, 142, 0, 0, 0, 0, 55, 255, 255, 255, 216, 0, 0, 0,
            ],
            &[
                0, 0, 44, 255, 255, 255, 226, 1, 0, 0, 0, 131, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[
                0, 0, 0, 198, 255, 255, 255, 57, 0, 0, 0, 206, 255, 255, 254, 31, 0, 0, 0,
            ],
            &[
                0, 0, 0, 97, 255, 255, 255, 143, 0, 0, 27, 255, 255, 255, 193, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 241, 255, 255, 227, 1, 0, 102, 255, 255, 255, 100, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 150, 255, 255, 255, 58, 0, 178, 255, 255, 247, 15, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 48, 255, 255, 255, 143, 8, 245, 255, 255, 169, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 203, 255, 255, 220, 70, 255, 255, 255, 76, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 101, 255, 255, 255, 157, 255, 255, 234, 4, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 12, 243, 255, 255, 245, 255, 255, 146, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 255, 255, 53, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 53, 255, 255, 255, 255, 216, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 208, 255, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 1, 220, 255, 255, 254, 32, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 83, 255, 255, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 55, 232, 255, 255, 255, 70, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 95, 210, 228, 255, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 255, 230, 29, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 210, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 137, 152, 132, 74, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'þ' / 0xfe
        'þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 68, 133, 133, 133, 42, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 75, 22, 138, 216, 249, 235, 171, 46, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 96, 229, 255, 255, 255, 255, 255, 247, 66, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 246, 255, 255, 255, 255, 255, 255, 255, 232, 11, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 219, 99, 74, 145, 255, 255, 255, 255, 109, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 238, 19, 0, 0, 0, 145, 255, 255, 255, 190, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 142, 0, 0, 0, 0, 45, 255, 255, 255, 240, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 101, 0, 0, 0, 0, 6, 255, 255, 255, 255, 15, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 84, 0, 0, 0, 0, 0, 243, 255, 255, 255, 26, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 104, 0, 0, 0, 0, 3, 254, 255, 255, 255, 13, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 137, 0, 0, 0, 0, 29, 255, 255, 255, 248, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 228, 4, 0, 0, 0, 109, 255, 255, 255, 196, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 178, 36, 9, 75, 239, 255, 255, 255, 124, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 240, 19, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 175, 255, 255, 255, 255, 255, 255, 253, 91, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 64, 94, 218, 255, 255, 255, 217, 79, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 80, 0, 1, 33, 60, 34, 2, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 63, 122, 122, 122, 39, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ÿ' / 0xff
        'ÿ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 12, 94, 83, 4, 0, 0, 0, 71, 99, 22, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 189, 255, 255, 153, 0, 0, 113, 255, 255, 226, 8, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 246, 255, 255, 215, 0, 0, 172, 255, 255, 255, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 196, 255, 255, 161, 0, 0, 118, 255, 255, 232, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 20, 116, 104, 9, 0, 0, 2, 91, 121, 34, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 59, 189, 189, 189, 175, 1, 0, 0, 0, 0, 0, 125, 189, 189, 189, 99, 0, 0,
            ],
            &[
                0, 8, 238, 255, 255, 255, 56, 0, 0, 0, 0, 1, 233, 255, 255, 255, 53, 0, 0,
            ],
            &[
                0, 0, 146, 255, 255, 255, 142, 0, 0, 0, 0, 55, 255, 255, 255, 216, 0, 0, 0,
            ],
            &[
                0, 0, 44, 255, 255, 255, 226, 1, 0, 0, 0, 131, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[
                0, 0, 0, 198, 255, 255, 255, 57, 0, 0, 0, 206, 255, 255, 254, 31, 0, 0, 0,
            ],
            &[
                0, 0, 0, 97, 255, 255, 255, 143, 0, 0, 27, 255, 255, 255, 193, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 241, 255, 255, 227, 1, 0, 102, 255, 255, 255, 100, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 150, 255, 255, 255, 58, 0, 178, 255, 255, 247, 15, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 48, 255, 255, 255, 143, 8, 245, 255, 255, 169, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 203, 255, 255, 220, 70, 255, 255, 255, 76, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 101, 255, 255, 255, 157, 255, 255, 234, 4, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 12, 243, 255, 255, 245, 255, 255, 146, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 255, 255, 53, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 53, 255, 255, 255, 255, 216, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 208, 255, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 1, 220, 255, 255, 254, 32, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 83, 255, 255, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 55, 232, 255, 255, 255, 70, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 95, 210, 228, 255, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 255, 230, 29, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 210, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 137, 152, 132, 74, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'Ā' / 0x100
        'Ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 171, 171, 171, 171, 171, 171, 171, 171, 30, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 42, 91, 91, 91, 91, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 165, 255, 255, 255, 255, 210, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 1, 234, 255, 255, 255, 255, 255, 25, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 50, 255, 255, 255, 247, 255, 255, 95, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 120, 255, 255, 230, 179, 255, 255, 166, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 190, 255, 255, 172, 119, 255, 255, 235, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 11, 249, 255, 255, 112, 59, 255, 255, 255, 51, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 255, 255, 255, 49, 7, 247, 255, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 145, 255, 255, 239, 2, 0, 193, 255, 255, 193, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 216, 255, 255, 178, 0, 0, 131, 255, 255, 250, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 30, 255, 255, 255, 114, 0, 0, 70, 255, 255, 255, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 100, 255, 255, 255, 55, 4, 4, 17, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 238, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 34, 0, 0, 0,
            ],
            &[
                0, 0, 56, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 105, 0, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 107, 56, 56, 56, 56, 75, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 196, 255, 255, 240, 2, 0, 0, 0, 0, 0, 212, 255, 255, 241, 4, 0, 0,
            ],
            &[
                0, 15, 251, 255, 255, 179, 0, 0, 0, 0, 0, 0, 150, 255, 255, 255, 61, 0, 0,
            ],
            &[
                0, 81, 255, 255, 255, 115, 0, 0, 0, 0, 0, 0, 88, 255, 255, 255, 131, 0, 0,
            ],
            &[
                0, 151, 255, 255, 255, 51, 0, 0, 0, 0, 0, 0, 27, 255, 255, 255, 202, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ā' / 0x101
        'ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 94, 119, 119, 119, 119, 119, 119, 119, 46, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 201, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 201, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 41, 52, 52, 52, 52, 52, 52, 52, 20, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 63, 143, 200, 237, 252, 250, 230, 183, 101, 6, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 214, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 200, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 178, 182, 107, 65, 39, 49, 109, 242, 255, 255, 255, 52, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 255, 255, 98, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 4, 23, 31, 38, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 50, 160, 225, 253, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 112, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 40, 254, 255, 255, 254, 158, 94, 60, 50, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 127, 255, 255, 255, 118, 0, 0, 0, 0, 78, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 255, 55, 0, 0, 0, 0, 141, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 152, 5, 0, 10, 107, 252, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 85, 255, 255, 255, 255, 239, 218, 252, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 2, 203, 255, 255, 255, 255, 255, 255, 249, 105, 248, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 28, 185, 255, 255, 255, 255, 189, 59, 0, 176, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 23, 59, 54, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ă' / 0x102
        'Ă' => Some(&[
            &[
                0, 0, 0, 0, 73, 255, 252, 29, 0, 0, 6, 217, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 247, 255, 231, 142, 130, 207, 255, 254, 55, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 114, 255, 255, 255, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 87, 213, 252, 253, 219, 104, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 42, 91, 91, 91, 91, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 165, 255, 255, 255, 255, 210, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 1, 234, 255, 255, 255, 255, 255, 25, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 50, 255, 255, 255, 247, 255, 255, 95, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 120, 255, 255, 230, 179, 255, 255, 166, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 190, 255, 255, 172, 119, 255, 255, 235, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 11, 249, 255, 255, 112, 59, 255, 255, 255, 51, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 255, 255, 255, 49, 7, 247, 255, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 145, 255, 255, 239, 2, 0, 193, 255, 255, 193, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 216, 255, 255, 178, 0, 0, 131, 255, 255, 250, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 30, 255, 255, 255, 114, 0, 0, 70, 255, 255, 255, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 100, 255, 255, 255, 55, 4, 4, 17, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 238, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 34, 0, 0, 0,
            ],
            &[
                0, 0, 56, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 105, 0, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 107, 56, 56, 56, 56, 75, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 196, 255, 255, 240, 2, 0, 0, 0, 0, 0, 212, 255, 255, 241, 4, 0, 0,
            ],
            &[
                0, 15, 251, 255, 255, 179, 0, 0, 0, 0, 0, 0, 150, 255, 255, 255, 61, 0, 0,
            ],
            &[
                0, 81, 255, 255, 255, 115, 0, 0, 0, 0, 0, 0, 88, 255, 255, 255, 131, 0, 0,
            ],
            &[
                0, 151, 255, 255, 255, 51, 0, 0, 0, 0, 0, 0, 27, 255, 255, 255, 202, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 31, 76, 65, 0, 0, 0, 0, 49, 76, 46, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ă' / 0x103
        'ă' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 28, 195, 195, 29, 0, 0, 0, 96, 195, 151, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 3, 247, 255, 167, 17, 0, 45, 224, 255, 153, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 151, 255, 255, 255, 246, 255, 255, 250, 49, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 12, 189, 255, 255, 255, 255, 244, 97, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 76, 131, 150, 110, 30, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 63, 143, 200, 237, 252, 250, 230, 183, 101, 6, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 214, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 200, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 178, 182, 107, 65, 39, 49, 109, 242, 255, 255, 255, 52, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 255, 255, 98, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 4, 23, 31, 38, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 50, 160, 225, 253, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 112, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 40, 254, 255, 255, 254, 158, 94, 60, 50, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 127, 255, 255, 255, 118, 0, 0, 0, 0, 78, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 255, 55, 0, 0, 0, 0, 141, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 152, 5, 0, 10, 107, 252, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 85, 255, 255, 255, 255, 239, 218, 252, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 2, 203, 255, 255, 255, 255, 255, 255, 249, 105, 248, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 28, 185, 255, 255, 255, 255, 189, 59, 0, 176, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 23, 59, 54, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ą' / 0x104
        'Ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 42, 91, 91, 91, 91, 58, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 165, 255, 255, 255, 255, 210, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 1, 234, 255, 255, 255, 255, 255, 25, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 50, 255, 255, 255, 247, 255, 255, 95, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 120, 255, 255, 230, 179, 255, 255, 166, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 190, 255, 255, 172, 119, 255, 255, 235, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 11, 249, 255, 255, 112, 59, 255, 255, 255, 51, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 255, 255, 255, 49, 7, 247, 255, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 145, 255, 255, 239, 2, 0, 193, 255, 255, 193, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 216, 255, 255, 178, 0, 0, 131, 255, 255, 250, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 30, 255, 255, 255, 114, 0, 0, 70, 255, 255, 255, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 100, 255, 255, 255, 55, 4, 4, 17, 255, 255, 255, 149, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 238, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 34, 0, 0, 0,
            ],
            &[
                0, 0, 56, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 105, 0, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 107, 56, 56, 56, 56, 75, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 196, 255, 255, 240, 2, 0, 0, 0, 0, 0, 212, 255, 255, 241, 4, 0, 0,
            ],
            &[
                0, 15, 251, 255, 255, 179, 0, 0, 0, 0, 0, 0, 150, 255, 255, 255, 61, 0, 0,
            ],
            &[
                0, 81, 255, 255, 255, 115, 0, 0, 0, 0, 0, 0, 88, 255, 255, 255, 131, 0, 0,
            ],
            &[
                0, 151, 255, 255, 255, 51, 0, 0, 0, 0, 0, 0, 27, 255, 255, 255, 227, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 36, 220, 255, 124, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 206, 255, 179, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 82, 255, 255, 82, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 128, 255, 255, 126, 6, 18, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 89, 255, 255, 255, 255, 134, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 180, 255, 255, 255, 134, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 55, 93, 80, 20, 0],
        ]),
        // letter: 'ą' / 0x105
        'ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 63, 143, 200, 237, 252, 250, 230, 183, 101, 6, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 214, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 200, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 178, 182, 107, 65, 39, 49, 109, 242, 255, 255, 255, 52, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 255, 255, 98, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 4, 23, 31, 38, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 50, 160, 225, 253, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 112, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 40, 254, 255, 255, 254, 158, 94, 60, 50, 104, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 127, 255, 255, 255, 118, 0, 0, 0, 0, 78, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 255, 55, 0, 0, 0, 0, 141, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 152, 5, 0, 10, 107, 252, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 85, 255, 255, 255, 255, 239, 218, 252, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 2, 203, 255, 255, 255, 255, 255, 255, 249, 105, 248, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 28, 185, 255, 255, 255, 255, 189, 59, 0, 195, 255, 255, 113, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 23, 59, 54, 22, 0, 0, 36, 219, 255, 124, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 206, 255, 179, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 82, 255, 255, 83, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 127, 255, 255, 126, 6, 18, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 89, 255, 255, 255, 255, 134, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 179, 255, 255, 255, 134, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 55, 93, 80, 20, 0, 0, 0],
        ]),
        // letter: 'Ć' / 0x106
        'Ć' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 29, 244, 255, 255, 243, 57, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 182, 255, 255, 242, 63, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 103, 255, 255, 235, 52, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 35, 245, 255, 217, 35, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 7, 26, 26, 9, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 30, 93, 119, 135, 125, 88, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 69, 202, 255, 255, 255, 255, 255, 255, 249, 171, 5, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 189, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 129, 255, 255, 255, 255, 255, 245, 233, 253, 255, 255, 86, 0, 0, 0,
            ],
            &[
                0, 0, 0, 50, 251, 255, 255, 255, 169, 45, 0, 0, 5, 68, 154, 5, 0, 0, 0,
            ],
            &[
                0, 0, 0, 168, 255, 255, 255, 156, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 16, 250, 255, 255, 243, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 72, 255, 255, 255, 173, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 123, 255, 255, 255, 122, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 94, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 160, 255, 255, 255, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 150, 255, 255, 255, 96, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 130, 255, 255, 255, 117, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 101, 255, 255, 255, 172, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 34, 255, 255, 255, 242, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 255, 255, 170, 4, 0, 0, 0, 0, 0, 4, 16, 0, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 255, 255, 204, 99, 58, 53, 84, 144, 232, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 194, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 17, 196, 255, 255, 255, 255, 255, 255, 255, 255, 255, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 4, 106, 211, 255, 255, 255, 255, 255, 227, 151, 31, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 20, 54, 64, 43, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 38, 55, 55, 55, 35, 0, 0, 0],
        ]),
        // letter: 'ć' / 0x107
        'ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 108, 174, 174, 174, 124, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 42, 249, 255, 255, 218, 24, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 4, 203, 255, 255, 210, 24, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 193, 17, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 9, 160, 163, 136, 6, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 58, 152, 213, 239, 251, 232, 207, 156, 74, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 152, 255, 255, 255, 255, 255, 255, 255, 255, 210, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 255, 255, 255, 255, 255, 255, 255, 255, 117, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 61, 255, 255, 255, 255, 208, 112, 85, 103, 155, 225, 26, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 153, 255, 255, 255, 187, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 226, 255, 255, 255, 52, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 2, 252, 255, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 20, 255, 255, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 13, 255, 255, 255, 236, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 246, 255, 255, 255, 28, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 199, 255, 255, 255, 139, 0, 0, 0, 0, 0, 0, 17, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 118, 255, 255, 255, 255, 147, 47, 20, 46, 91, 183, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 15, 226, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 57, 233, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 25, 149, 233, 255, 255, 255, 255, 248, 180, 55, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 3, 30, 53, 59, 35, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĉ' / 0x108
        'Ĉ' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 7, 216, 255, 255, 255, 238, 24, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 145, 255, 255, 248, 255, 255, 184, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 86, 255, 255, 205, 28, 185, 255, 255, 125, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 46, 246, 255, 181, 15, 0, 7, 156, 255, 254, 71, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 9, 26, 26, 2, 0, 0, 0, 0, 24, 26, 13, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 30, 93, 119, 135, 125, 88, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 69, 202, 255, 255, 255, 255, 255, 255, 249, 171, 5, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 189, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 129, 255, 255, 255, 255, 255, 245, 233, 253, 255, 255, 86, 0, 0, 0,
            ],
            &[
                0, 0, 0, 50, 251, 255, 255, 255, 169, 45, 0, 0, 5, 68, 154, 5, 0, 0, 0,
            ],
            &[
                0, 0, 0, 168, 255, 255, 255, 156, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 16, 250, 255, 255, 243, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 72, 255, 255, 255, 173, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 123, 255, 255, 255, 122, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 94, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 160, 255, 255, 255, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 150, 255, 255, 255, 96, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 130, 255, 255, 255, 117, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 101, 255, 255, 255, 172, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 34, 255, 255, 255, 242, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 255, 255, 170, 4, 0, 0, 0, 0, 0, 4, 16, 0, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 255, 255, 204, 99, 58, 53, 84, 144, 232, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 194, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 17, 196, 255, 255, 255, 255, 255, 255, 255, 255, 255, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 4, 106, 211, 255, 255, 255, 255, 255, 227, 151, 31, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 20, 54, 64, 43, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 26, 55, 55, 55, 35, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĉ' / 0x109
        'ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 72, 174, 174, 174, 164, 9, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 17, 229, 255, 255, 255, 255, 142, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 1, 178, 255, 255, 174, 233, 255, 254, 76, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 128, 255, 255, 153, 2, 42, 227, 255, 239, 36, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 162, 163, 103, 0, 0, 0, 22, 151, 163, 108, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 58, 152, 213, 239, 251, 232, 207, 156, 74, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 152, 255, 255, 255, 255, 255, 255, 255, 255, 210, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 255, 255, 255, 255, 255, 255, 255, 255, 117, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 61, 255, 255, 255, 255, 208, 112, 85, 103, 155, 225, 26, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 153, 255, 255, 255, 187, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 226, 255, 255, 255, 52, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 2, 252, 255, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 20, 255, 255, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 13, 255, 255, 255, 236, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 246, 255, 255, 255, 28, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 199, 255, 255, 255, 139, 0, 0, 0, 0, 0, 0, 17, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 118, 255, 255, 255, 255, 147, 47, 20, 46, 91, 183, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 15, 226, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 57, 233, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 25, 149, 233, 255, 255, 255, 255, 248, 180, 55, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 3, 30, 53, 59, 35, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ċ' / 0x10a
        'Ċ' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 125, 255, 255, 255, 151, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 203, 255, 255, 255, 226, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 185, 255, 255, 255, 204, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 54, 219, 255, 221, 62, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 30, 93, 119, 135, 125, 88, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 69, 202, 255, 255, 255, 255, 255, 255, 249, 171, 5, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 189, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 129, 255, 255, 255, 255, 255, 245, 233, 253, 255, 255, 86, 0, 0, 0,
            ],
            &[
                0, 0, 0, 50, 251, 255, 255, 255, 169, 45, 0, 0, 5, 68, 154, 5, 0, 0, 0,
            ],
            &[
                0, 0, 0, 168, 255, 255, 255, 156, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 16, 250, 255, 255, 243, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 72, 255, 255, 255, 173, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 123, 255, 255, 255, 122, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 94, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 160, 255, 255, 255, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 150, 255, 255, 255, 96, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 130, 255, 255, 255, 117, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 101, 255, 255, 255, 172, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 34, 255, 255, 255, 242, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 255, 255, 170, 4, 0, 0, 0, 0, 0, 4, 16, 0, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 255, 255, 204, 99, 58, 53, 84, 144, 232, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 194, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 17, 196, 255, 255, 255, 255, 255, 255, 255, 255, 255, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 4, 106, 211, 255, 255, 255, 255, 255, 227, 151, 31, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 20, 54, 64, 43, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 73, 116, 82, 2, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ċ' / 0x10b
        'ċ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 14, 169, 232, 216, 89, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 121, 255, 255, 255, 247, 9, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 149, 255, 255, 255, 255, 26, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 86, 255, 255, 255, 212, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 78, 138, 118, 20, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 58, 152, 213, 239, 251, 232, 207, 156, 74, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 152, 255, 255, 255, 255, 255, 255, 255, 255, 210, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 255, 255, 255, 255, 255, 255, 255, 255, 117, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 61, 255, 255, 255, 255, 208, 112, 85, 103, 155, 225, 26, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 153, 255, 255, 255, 187, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 226, 255, 255, 255, 52, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 2, 252, 255, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 20, 255, 255, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 13, 255, 255, 255, 236, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 246, 255, 255, 255, 28, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 199, 255, 255, 255, 139, 0, 0, 0, 0, 0, 0, 17, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 118, 255, 255, 255, 255, 147, 47, 20, 46, 91, 183, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 15, 226, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 57, 233, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 25, 149, 233, 255, 255, 255, 255, 248, 180, 55, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 3, 30, 53, 59, 35, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Č' / 0x10c
        'Č' => Some(&[
            &[
                0, 0, 0, 0, 0, 35, 236, 255, 204, 28, 0, 20, 190, 255, 248, 55, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 64, 252, 255, 222, 56, 213, 255, 255, 96, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 124, 255, 255, 255, 255, 255, 158, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 3, 202, 255, 255, 255, 224, 13, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 11, 26, 26, 26, 14, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 30, 93, 119, 135, 125, 88, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 69, 202, 255, 255, 255, 255, 255, 255, 249, 171, 5, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 139, 255, 255, 255, 255, 255, 255, 255, 255, 255, 189, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 129, 255, 255, 255, 255, 255, 245, 233, 253, 255, 255, 86, 0, 0, 0,
            ],
            &[
                0, 0, 0, 50, 251, 255, 255, 255, 169, 45, 0, 0, 5, 68, 154, 5, 0, 0, 0,
            ],
            &[
                0, 0, 0, 168, 255, 255, 255, 156, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 16, 250, 255, 255, 243, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 72, 255, 255, 255, 173, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 123, 255, 255, 255, 122, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 94, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 160, 255, 255, 255, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 150, 255, 255, 255, 96, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 130, 255, 255, 255, 117, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 101, 255, 255, 255, 172, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 34, 255, 255, 255, 242, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 255, 255, 170, 4, 0, 0, 0, 0, 0, 4, 16, 0, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 255, 255, 204, 99, 58, 53, 84, 144, 232, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 194, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 17, 196, 255, 255, 255, 255, 255, 255, 255, 255, 255, 93, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 4, 106, 211, 255, 255, 255, 255, 255, 227, 151, 31, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 20, 54, 64, 43, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 21, 55, 55, 8, 0, 0, 0, 3, 54, 55, 27, 0, 0, 0,
            ],
        ]),
        // letter: 'č' / 0x10d
        'č' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 19, 172, 174, 114, 0, 0, 0, 31, 164, 174, 113, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 117, 255, 255, 163, 3, 56, 237, 255, 232, 29, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 170, 255, 255, 186, 242, 255, 252, 64, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 14, 225, 255, 255, 255, 255, 129, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 65, 163, 163, 163, 150, 5, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 58, 152, 213, 239, 251, 232, 207, 156, 74, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 152, 255, 255, 255, 255, 255, 255, 255, 255, 210, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 255, 255, 255, 255, 255, 255, 255, 255, 117, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 61, 255, 255, 255, 255, 208, 112, 85, 103, 155, 225, 26, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 153, 255, 255, 255, 187, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 226, 255, 255, 255, 52, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 2, 252, 255, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 20, 255, 255, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 13, 255, 255, 255, 236, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 246, 255, 255, 255, 28, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 199, 255, 255, 255, 139, 0, 0, 0, 0, 0, 0, 17, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 118, 255, 255, 255, 255, 147, 47, 20, 46, 91, 183, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 15, 226, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 57, 233, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 25, 149, 233, 255, 255, 255, 255, 248, 180, 55, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 3, 30, 53, 59, 35, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ď' / 0x10e
        'Ď' => Some(&[
            &[
                0, 0, 0, 20, 223, 255, 219, 41, 0, 11, 171, 255, 253, 77, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 43, 245, 255, 235, 61, 195, 255, 255, 125, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 95, 255, 255, 255, 255, 255, 184, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 177, 255, 255, 255, 239, 26, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 8, 26, 26, 26, 17, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 71, 71, 66, 46, 7, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 247, 175, 54, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 253, 126, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 116, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 29, 11, 26, 74, 196, 255, 255, 255, 249, 36, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 6, 191, 255, 255, 255, 151, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 50, 255, 255, 255, 235, 1, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 0, 227, 255, 255, 255, 39, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 0, 177, 255, 255, 255, 75, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 0, 159, 255, 255, 255, 92, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 0, 152, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 0, 164, 255, 255, 255, 84, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 0, 197, 255, 255, 255, 55, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 10, 246, 255, 255, 251, 12, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 103, 255, 255, 255, 192, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 51, 239, 255, 255, 255, 89, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 95, 83, 101, 162, 250, 255, 255, 255, 202, 2, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 230, 33, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 199, 33, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 247, 221, 166, 77, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 15, 55, 55, 14, 0, 0, 0, 0, 51, 55, 33, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ď' / 0x10f
        'ď' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 62, 133, 133, 133, 48, 36, 133, 133, 133, 44,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 255, 255, 93, 100, 255, 255, 242, 16,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 255, 255, 93, 149, 255, 255, 131, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 255, 255, 93, 206, 255, 234, 15, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 97, 202, 204, 90, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 114, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 69, 193, 238, 239, 185, 46, 96, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 98, 255, 255, 255, 255, 255, 244, 115, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                0, 29, 249, 255, 255, 255, 255, 255, 255, 248, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                0, 138, 255, 255, 255, 227, 94, 70, 156, 255, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                0, 209, 255, 255, 255, 67, 0, 0, 0, 188, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                9, 253, 255, 255, 245, 1, 0, 0, 0, 99, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                29, 255, 255, 255, 221, 0, 0, 0, 0, 60, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                44, 255, 255, 255, 204, 0, 0, 0, 0, 46, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                38, 255, 255, 255, 218, 0, 0, 0, 0, 54, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                17, 255, 255, 255, 241, 2, 0, 0, 0, 81, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                0, 233, 255, 255, 255, 60, 0, 0, 0, 161, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                0, 168, 255, 255, 255, 210, 40, 12, 106, 251, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                0, 68, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 255, 255, 242, 102, 255, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 9, 158, 252, 255, 255, 216, 53, 0, 208, 255, 255, 93, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 22, 59, 43, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Đ' / 0x110
        'Đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 71, 71, 66, 46, 7, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 247, 175, 54, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 253, 126, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 116, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 29, 11, 26, 74, 196, 255, 255, 255, 249, 36, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 6, 191, 255, 255, 255, 151, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 50, 255, 255, 255, 235, 1, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 0, 227, 255, 255, 255, 39, 0, 0,
            ],
            &[
                0, 38, 239, 255, 255, 255, 58, 38, 36, 0, 0, 0, 177, 255, 255, 255, 75, 0, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 240, 0, 0, 0, 159, 255, 255, 255, 92, 0, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 240, 0, 0, 0, 152, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 240, 0, 0, 0, 164, 255, 255, 255, 84, 0, 0,
            ],
            &[
                0, 22, 223, 255, 255, 255, 42, 22, 21, 0, 0, 0, 197, 255, 255, 255, 55, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 10, 246, 255, 255, 251, 12, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 0, 103, 255, 255, 255, 192, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 19, 0, 0, 0, 51, 239, 255, 255, 255, 89, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 95, 83, 101, 162, 250, 255, 255, 255, 202, 2, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 230, 33, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 199, 33, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 247, 221, 166, 77, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'đ' / 0x111
        'đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 15, 133, 133, 133, 95, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 30, 255, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 87, 92, 92, 122, 255, 255, 255, 255, 92, 18, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 243, 255, 255, 255, 255, 255, 255, 255, 255, 51, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 243, 255, 255, 255, 255, 255, 255, 255, 255, 51, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 43, 45, 45, 71, 255, 255, 255, 228, 45, 9, 0, 0,
            ],
            &[
                0, 0, 0, 0, 58, 143, 176, 166, 121, 18, 12, 255, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 164, 255, 255, 255, 255, 255, 241, 61, 241, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 0, 143, 255, 255, 255, 255, 255, 255, 255, 239, 240, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 32, 250, 255, 255, 255, 237, 164, 138, 198, 255, 255, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 118, 255, 255, 255, 228, 34, 0, 0, 0, 158, 255, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 180, 255, 255, 255, 117, 0, 0, 0, 0, 33, 255, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 211, 255, 255, 255, 49, 0, 0, 0, 0, 0, 234, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 229, 255, 255, 255, 23, 0, 0, 0, 0, 0, 214, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 225, 255, 255, 255, 27, 0, 0, 0, 0, 0, 218, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 204, 255, 255, 255, 56, 0, 0, 0, 0, 1, 246, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 159, 255, 255, 255, 143, 0, 0, 0, 0, 78, 255, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 88, 255, 255, 255, 249, 110, 20, 11, 77, 230, 255, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 8, 227, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 0, 72, 252, 255, 255, 255, 255, 255, 255, 106, 224, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 68, 215, 255, 255, 255, 222, 90, 0, 119, 255, 255, 182, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 37, 61, 34, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ē' / 0x112
        'Ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 215, 255, 255, 255, 255, 255, 255, 255, 86, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 215, 255, 255, 255, 255, 255, 255, 255, 86, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 144, 171, 171, 171, 171, 171, 171, 171, 58, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 59, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 22, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 242, 24, 24, 24, 24, 24, 24, 24, 24, 7, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 176, 176, 176, 176, 176, 176, 176, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 175, 175, 175, 175, 175, 175, 175, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 246, 95, 95, 95, 95, 95, 95, 95, 95, 29, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ē' / 0x113
        'ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 87, 119, 119, 119, 119, 119, 119, 119, 53, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 187, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 187, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 38, 52, 52, 52, 52, 52, 52, 52, 23, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 21, 120, 203, 235, 250, 226, 178, 81, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 64, 235, 255, 255, 255, 255, 255, 255, 255, 174, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 43, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 197, 255, 255, 255, 168, 39, 5, 52, 219, 255, 255, 255, 49, 0, 0, 0,
            ],
            &[
                0, 0, 38, 255, 255, 255, 192, 2, 0, 0, 0, 29, 246, 255, 255, 137, 0, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 255, 97, 0, 0, 0, 0, 0, 186, 255, 255, 184, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 238, 232, 232, 232, 232, 232, 248, 255, 255, 209, 0, 0, 0,
            ],
            &[
                0, 0, 159, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 206, 195, 195, 195, 195, 195, 195, 195, 195, 165, 0, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 255, 199, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 198, 255, 255, 255, 206, 81, 15, 0, 9, 38, 95, 171, 94, 0, 0, 0,
            ],
            &[
                0, 0, 0, 42, 244, 255, 255, 255, 255, 255, 247, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 68, 231, 255, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 132, 218, 255, 255, 255, 255, 255, 230, 152, 32, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 45, 63, 45, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĕ' / 0x114
        'Ĕ' => Some(&[
            &[
                0, 0, 0, 0, 31, 255, 255, 67, 0, 0, 0, 181, 255, 163, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 226, 255, 244, 151, 127, 189, 255, 255, 96, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 74, 255, 255, 255, 255, 255, 255, 189, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 63, 200, 248, 255, 227, 130, 7, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 3, 12, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 59, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 22, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 242, 24, 24, 24, 24, 24, 24, 24, 24, 7, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 176, 176, 176, 176, 176, 176, 176, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 175, 175, 175, 175, 175, 175, 175, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 246, 95, 95, 95, 95, 95, 95, 95, 95, 29, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 18, 76, 76, 1, 0, 0, 0, 37, 76, 58, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ĕ' / 0x115
        'ĕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 18, 195, 195, 39, 0, 0, 0, 85, 195, 162, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 237, 255, 177, 21, 0, 39, 216, 255, 167, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 137, 255, 255, 255, 245, 255, 255, 253, 59, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 7, 180, 255, 255, 255, 255, 247, 108, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 71, 129, 151, 113, 35, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 21, 120, 203, 235, 250, 226, 178, 81, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 64, 235, 255, 255, 255, 255, 255, 255, 255, 174, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 43, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 197, 255, 255, 255, 168, 39, 5, 52, 219, 255, 255, 255, 49, 0, 0, 0,
            ],
            &[
                0, 0, 38, 255, 255, 255, 192, 2, 0, 0, 0, 29, 246, 255, 255, 137, 0, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 255, 97, 0, 0, 0, 0, 0, 186, 255, 255, 184, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 238, 232, 232, 232, 232, 232, 248, 255, 255, 209, 0, 0, 0,
            ],
            &[
                0, 0, 159, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 206, 195, 195, 195, 195, 195, 195, 195, 195, 165, 0, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 255, 199, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 198, 255, 255, 255, 206, 81, 15, 0, 9, 38, 95, 171, 94, 0, 0, 0,
            ],
            &[
                0, 0, 0, 42, 244, 255, 255, 255, 255, 255, 247, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 68, 231, 255, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 132, 218, 255, 255, 255, 255, 255, 230, 152, 32, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 45, 63, 45, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ė' / 0x116
        'Ė' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 201, 255, 255, 255, 76, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 23, 255, 255, 255, 255, 150, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 9, 251, 255, 255, 255, 129, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 108, 240, 252, 191, 20, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 59, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 22, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 242, 24, 24, 24, 24, 24, 24, 24, 24, 7, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 176, 176, 176, 176, 176, 176, 176, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 175, 175, 175, 175, 175, 175, 175, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 246, 95, 95, 95, 95, 95, 95, 95, 95, 29, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 11, 95, 114, 55, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ė' / 0x117
        'ė' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 67, 207, 234, 184, 28, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 225, 255, 255, 255, 153, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 1, 251, 255, 255, 255, 179, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 189, 255, 255, 255, 110, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 16, 115, 139, 85, 1, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 21, 120, 203, 235, 250, 226, 178, 81, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 64, 235, 255, 255, 255, 255, 255, 255, 255, 174, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 43, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 197, 255, 255, 255, 168, 39, 5, 52, 219, 255, 255, 255, 49, 0, 0, 0,
            ],
            &[
                0, 0, 38, 255, 255, 255, 192, 2, 0, 0, 0, 29, 246, 255, 255, 137, 0, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 255, 97, 0, 0, 0, 0, 0, 186, 255, 255, 184, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 238, 232, 232, 232, 232, 232, 248, 255, 255, 209, 0, 0, 0,
            ],
            &[
                0, 0, 159, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 206, 195, 195, 195, 195, 195, 195, 195, 195, 165, 0, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 255, 199, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 198, 255, 255, 255, 206, 81, 15, 0, 9, 38, 95, 171, 94, 0, 0, 0,
            ],
            &[
                0, 0, 0, 42, 244, 255, 255, 255, 255, 255, 247, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 68, 231, 255, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 132, 218, 255, 255, 255, 255, 255, 230, 152, 32, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 45, 63, 45, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ę' / 0x118
        'Ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 59, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 22, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 242, 24, 24, 24, 24, 24, 24, 24, 24, 7, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 176, 176, 176, 176, 176, 176, 176, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 175, 175, 175, 175, 175, 175, 175, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 246, 95, 95, 95, 95, 95, 95, 95, 95, 29, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 39, 223, 255, 118, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 211, 255, 173, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 88, 255, 255, 76, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 134, 255, 255, 120, 6, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 95, 255, 255, 255, 255, 127, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 185, 255, 255, 255, 127, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 56, 93, 79, 18, 0, 0, 0, 0],
        ]),
        // letter: 'ę' / 0x119
        'ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 21, 120, 203, 235, 250, 226, 178, 81, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 64, 235, 255, 255, 255, 255, 255, 255, 255, 174, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 43, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 197, 255, 255, 255, 168, 39, 5, 52, 219, 255, 255, 255, 49, 0, 0, 0,
            ],
            &[
                0, 0, 38, 255, 255, 255, 192, 2, 0, 0, 0, 29, 246, 255, 255, 137, 0, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 255, 97, 0, 0, 0, 0, 0, 186, 255, 255, 184, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 238, 232, 232, 232, 232, 232, 248, 255, 255, 209, 0, 0, 0,
            ],
            &[
                0, 0, 159, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 206, 195, 195, 195, 195, 195, 195, 195, 195, 165, 0, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 255, 199, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 198, 255, 255, 255, 206, 81, 15, 0, 9, 38, 95, 171, 94, 0, 0, 0,
            ],
            &[
                0, 0, 0, 42, 244, 255, 255, 255, 255, 255, 247, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 68, 231, 255, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 132, 218, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 21, 45, 70, 239, 255, 200, 14, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 250, 28, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 230, 255, 203, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 252, 255, 235, 29, 20, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 208, 255, 255, 255, 255, 10, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 60, 236, 255, 255, 255, 10, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 74, 93, 57, 0, 0, 0, 0],
        ]),
        // letter: 'Ě' / 0x11a
        'Ě' => Some(&[
            &[
                0, 0, 0, 0, 92, 255, 255, 152, 4, 0, 57, 229, 255, 214, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 136, 255, 255, 173, 73, 245, 255, 240, 36, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 4, 195, 255, 255, 255, 255, 255, 83, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 35, 245, 255, 255, 255, 161, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 19, 26, 26, 26, 6, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 59, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 22, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 242, 24, 24, 24, 24, 24, 24, 24, 24, 7, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 176, 176, 176, 176, 176, 176, 176, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 251, 175, 175, 175, 175, 175, 175, 175, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 246, 95, 95, 95, 95, 95, 95, 95, 95, 29, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 37, 55, 47, 0, 0, 0, 0, 19, 55, 55, 11, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ě' / 0x11b
        'ě' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 87, 174, 171, 47, 0, 0, 0, 92, 174, 174, 42, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 208, 255, 245, 74, 0, 140, 255, 255, 158, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 34, 240, 255, 249, 180, 255, 255, 206, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 88, 255, 255, 255, 255, 245, 36, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 131, 163, 163, 163, 90, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 21, 120, 203, 235, 250, 226, 178, 81, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 64, 235, 255, 255, 255, 255, 255, 255, 255, 174, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 43, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 158, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 197, 255, 255, 255, 168, 39, 5, 52, 219, 255, 255, 255, 49, 0, 0, 0,
            ],
            &[
                0, 0, 38, 255, 255, 255, 192, 2, 0, 0, 0, 29, 246, 255, 255, 137, 0, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 255, 97, 0, 0, 0, 0, 0, 186, 255, 255, 184, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 238, 232, 232, 232, 232, 232, 248, 255, 255, 209, 0, 0, 0,
            ],
            &[
                0, 0, 159, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 206, 195, 195, 195, 195, 195, 195, 195, 195, 165, 0, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 255, 199, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 198, 255, 255, 255, 206, 81, 15, 0, 9, 38, 95, 171, 94, 0, 0, 0,
            ],
            &[
                0, 0, 0, 42, 244, 255, 255, 255, 255, 255, 247, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 68, 231, 255, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 132, 218, 255, 255, 255, 255, 255, 230, 152, 32, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 45, 63, 45, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 255, 76, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 76, 254, 255, 248, 255, 255, 231, 23, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 34, 238, 255, 239, 54, 125, 255, 255, 190, 4, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 9, 214, 255, 221, 44, 0, 0, 101, 248, 255, 138, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 2, 26, 26, 9, 0, 0, 0, 0, 17, 26, 20, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 26, 88, 124, 136, 126, 97, 46, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 41, 179, 254, 255, 255, 255, 255, 255, 255, 227, 106, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 79, 245, 255, 255, 255, 255, 255, 255, 255, 255, 255, 125, 0, 0,
            ],
            &[
                0, 0, 0, 0, 54, 250, 255, 255, 255, 255, 252, 225, 229, 253, 255, 253, 28, 0, 0,
            ],
            &[
                0, 0, 0, 2, 211, 255, 255, 255, 237, 101, 8, 0, 0, 7, 83, 130, 0, 0, 0,
            ],
            &[
                0, 0, 0, 79, 255, 255, 255, 243, 45, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 168, 255, 255, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 230, 255, 255, 255, 34, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 37, 255, 255, 255, 204, 0, 0, 0, 49, 218, 218, 218, 218, 218, 215, 0, 0,
            ],
            &[
                0, 0, 45, 255, 255, 255, 190, 0, 0, 0, 57, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 39, 255, 255, 255, 203, 0, 0, 0, 57, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 17, 255, 255, 255, 230, 0, 0, 0, 22, 98, 98, 185, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 235, 255, 255, 255, 33, 0, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 174, 255, 255, 255, 132, 0, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 87, 255, 255, 255, 243, 50, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 4, 217, 255, 255, 255, 239, 118, 43, 19, 39, 173, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 57, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 72, 239, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 25, 144, 232, 255, 255, 255, 255, 255, 242, 179, 88, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 31, 58, 65, 56, 30, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 11, 55, 55, 55, 50, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĝ' / 0x11d
        'ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 19, 171, 174, 174, 174, 56, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 163, 255, 255, 255, 255, 215, 9, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 96, 255, 255, 218, 189, 255, 255, 158, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 54, 246, 255, 212, 26, 8, 178, 255, 255, 103, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 128, 163, 143, 13, 0, 0, 2, 118, 163, 157, 3, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 20, 149, 230, 255, 241, 174, 46, 0, 78, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 226, 255, 255, 255, 255, 255, 248, 71, 201, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 190, 255, 255, 255, 255, 255, 255, 255, 248, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 59, 255, 255, 255, 255, 182, 87, 72, 149, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 192, 3, 0, 0, 0, 147, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 86, 0, 0, 0, 0, 48, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 219, 255, 255, 255, 40, 0, 0, 0, 0, 6, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 231, 255, 255, 255, 19, 0, 0, 0, 0, 0, 248, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 223, 255, 255, 255, 34, 0, 0, 0, 0, 2, 251, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 201, 255, 255, 255, 68, 0, 0, 0, 0, 25, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 155, 255, 255, 255, 162, 0, 0, 0, 0, 110, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 79, 255, 255, 255, 254, 126, 25, 13, 89, 243, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 5, 217, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 58, 246, 255, 255, 255, 255, 255, 255, 166, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 55, 204, 255, 255, 255, 235, 109, 53, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 31, 61, 47, 4, 0, 70, 255, 255, 255, 169, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 122, 255, 255, 255, 142, 0, 0, 0,
            ],
            &[
                0, 0, 0, 172, 106, 34, 0, 0, 0, 0, 77, 242, 255, 255, 255, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 228, 255, 255, 239, 210, 209, 240, 255, 255, 255, 255, 222, 4, 0, 0, 0,
            ],
            &[
                0, 0, 0, 228, 255, 255, 255, 255, 255, 255, 255, 255, 255, 244, 48, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 254, 185, 36, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 46, 100, 126, 146, 147, 126, 96, 25, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'Ğ' / 0x11e
        'Ğ' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 142, 255, 208, 3, 0, 0, 38, 255, 255, 52, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 81, 255, 255, 204, 132, 143, 232, 255, 234, 7, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 183, 255, 255, 255, 255, 255, 254, 81, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 8, 130, 228, 255, 248, 199, 61, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 3, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 26, 88, 124, 136, 126, 97, 46, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 41, 179, 254, 255, 255, 255, 255, 255, 255, 227, 106, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 79, 245, 255, 255, 255, 255, 255, 255, 255, 255, 255, 125, 0, 0,
            ],
            &[
                0, 0, 0, 0, 54, 250, 255, 255, 255, 255, 252, 225, 229, 253, 255, 253, 28, 0, 0,
            ],
            &[
                0, 0, 0, 2, 211, 255, 255, 255, 237, 101, 8, 0, 0, 7, 83, 130, 0, 0, 0,
            ],
            &[
                0, 0, 0, 79, 255, 255, 255, 243, 45, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 168, 255, 255, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 230, 255, 255, 255, 34, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 37, 255, 255, 255, 204, 0, 0, 0, 49, 218, 218, 218, 218, 218, 215, 0, 0,
            ],
            &[
                0, 0, 45, 255, 255, 255, 190, 0, 0, 0, 57, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 39, 255, 255, 255, 203, 0, 0, 0, 57, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 17, 255, 255, 255, 230, 0, 0, 0, 22, 98, 98, 185, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 235, 255, 255, 255, 33, 0, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 174, 255, 255, 255, 132, 0, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 87, 255, 255, 255, 243, 50, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 4, 217, 255, 255, 255, 239, 118, 43, 19, 39, 173, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 57, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 72, 239, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 25, 144, 232, 255, 255, 255, 255, 255, 242, 179, 88, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 31, 58, 65, 56, 30, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 51, 76, 44, 0, 0, 0, 0, 70, 76, 25, 0, 0, 0,
            ],
        ]),
        // letter: 'ğ' / 0x11f
        'ğ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 71, 195, 180, 1, 0, 0, 0, 138, 195, 109, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 50, 255, 255, 121, 8, 1, 77, 246, 255, 98, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 3, 203, 255, 255, 253, 248, 255, 255, 227, 16, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 40, 217, 255, 255, 255, 255, 227, 59, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 7, 95, 140, 143, 100, 13, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 20, 149, 230, 255, 241, 174, 46, 0, 78, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 226, 255, 255, 255, 255, 255, 248, 71, 201, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 190, 255, 255, 255, 255, 255, 255, 255, 248, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 59, 255, 255, 255, 255, 182, 87, 72, 149, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 192, 3, 0, 0, 0, 147, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 86, 0, 0, 0, 0, 48, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 219, 255, 255, 255, 40, 0, 0, 0, 0, 6, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 231, 255, 255, 255, 19, 0, 0, 0, 0, 0, 248, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 223, 255, 255, 255, 34, 0, 0, 0, 0, 2, 251, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 201, 255, 255, 255, 68, 0, 0, 0, 0, 25, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 155, 255, 255, 255, 162, 0, 0, 0, 0, 110, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 79, 255, 255, 255, 254, 126, 25, 13, 89, 243, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 5, 217, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 58, 246, 255, 255, 255, 255, 255, 255, 166, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 55, 204, 255, 255, 255, 235, 109, 53, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 31, 61, 47, 4, 0, 70, 255, 255, 255, 169, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 122, 255, 255, 255, 142, 0, 0, 0,
            ],
            &[
                0, 0, 0, 172, 106, 34, 0, 0, 0, 0, 77, 242, 255, 255, 255, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 228, 255, 255, 239, 210, 209, 240, 255, 255, 255, 255, 222, 4, 0, 0, 0,
            ],
            &[
                0, 0, 0, 228, 255, 255, 255, 255, 255, 255, 255, 255, 255, 244, 48, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 254, 185, 36, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 46, 100, 126, 146, 147, 126, 96, 25, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'Ġ' / 0x120
        'Ġ' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 59, 253, 255, 255, 216, 4, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 134, 255, 255, 255, 255, 40, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 116, 255, 255, 255, 253, 21, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 16, 190, 252, 240, 112, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 26, 88, 124, 136, 126, 97, 46, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 41, 179, 254, 255, 255, 255, 255, 255, 255, 227, 106, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 79, 245, 255, 255, 255, 255, 255, 255, 255, 255, 255, 125, 0, 0,
            ],
            &[
                0, 0, 0, 0, 54, 250, 255, 255, 255, 255, 252, 225, 229, 253, 255, 253, 28, 0, 0,
            ],
            &[
                0, 0, 0, 2, 211, 255, 255, 255, 237, 101, 8, 0, 0, 7, 83, 130, 0, 0, 0,
            ],
            &[
                0, 0, 0, 79, 255, 255, 255, 243, 45, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 168, 255, 255, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 230, 255, 255, 255, 34, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 37, 255, 255, 255, 204, 0, 0, 0, 49, 218, 218, 218, 218, 218, 215, 0, 0,
            ],
            &[
                0, 0, 45, 255, 255, 255, 190, 0, 0, 0, 57, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 39, 255, 255, 255, 203, 0, 0, 0, 57, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 17, 255, 255, 255, 230, 0, 0, 0, 22, 98, 98, 185, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 235, 255, 255, 255, 33, 0, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 174, 255, 255, 255, 132, 0, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 87, 255, 255, 255, 243, 50, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 4, 217, 255, 255, 255, 239, 118, 43, 19, 39, 173, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 57, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 72, 239, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 25, 144, 232, 255, 255, 255, 255, 255, 242, 179, 88, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 31, 58, 65, 56, 30, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 47, 112, 100, 16, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ġ' / 0x121
        'ġ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 115, 222, 229, 151, 3, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 39, 255, 255, 255, 255, 84, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 66, 255, 255, 255, 255, 110, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 18, 239, 255, 255, 252, 44, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 40, 129, 133, 53, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 20, 149, 230, 255, 241, 174, 46, 0, 78, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 226, 255, 255, 255, 255, 255, 248, 71, 201, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 190, 255, 255, 255, 255, 255, 255, 255, 248, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 59, 255, 255, 255, 255, 182, 87, 72, 149, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 192, 3, 0, 0, 0, 147, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 86, 0, 0, 0, 0, 48, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 219, 255, 255, 255, 40, 0, 0, 0, 0, 6, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 231, 255, 255, 255, 19, 0, 0, 0, 0, 0, 248, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 223, 255, 255, 255, 34, 0, 0, 0, 0, 2, 251, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 201, 255, 255, 255, 68, 0, 0, 0, 0, 25, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 155, 255, 255, 255, 162, 0, 0, 0, 0, 110, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 79, 255, 255, 255, 254, 126, 25, 13, 89, 243, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 5, 217, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 58, 246, 255, 255, 255, 255, 255, 255, 166, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 55, 204, 255, 255, 255, 235, 109, 53, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 31, 61, 47, 4, 0, 70, 255, 255, 255, 169, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 122, 255, 255, 255, 142, 0, 0, 0,
            ],
            &[
                0, 0, 0, 172, 106, 34, 0, 0, 0, 0, 77, 242, 255, 255, 255, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 228, 255, 255, 239, 210, 209, 240, 255, 255, 255, 255, 222, 4, 0, 0, 0,
            ],
            &[
                0, 0, 0, 228, 255, 255, 255, 255, 255, 255, 255, 255, 255, 244, 48, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 254, 185, 36, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 46, 100, 126, 146, 147, 126, 96, 25, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'Ģ' / 0x122
        'Ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 26, 88, 124, 136, 126, 97, 46, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 41, 179, 254, 255, 255, 255, 255, 255, 255, 227, 106, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 79, 245, 255, 255, 255, 255, 255, 255, 255, 255, 255, 125, 0, 0,
            ],
            &[
                0, 0, 0, 0, 54, 250, 255, 255, 255, 255, 252, 225, 229, 253, 255, 253, 28, 0, 0,
            ],
            &[
                0, 0, 0, 2, 211, 255, 255, 255, 237, 101, 8, 0, 0, 7, 83, 130, 0, 0, 0,
            ],
            &[
                0, 0, 0, 79, 255, 255, 255, 243, 45, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 168, 255, 255, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 230, 255, 255, 255, 34, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 37, 255, 255, 255, 204, 0, 0, 0, 49, 218, 218, 218, 218, 218, 215, 0, 0,
            ],
            &[
                0, 0, 45, 255, 255, 255, 190, 0, 0, 0, 57, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 39, 255, 255, 255, 203, 0, 0, 0, 57, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 17, 255, 255, 255, 230, 0, 0, 0, 22, 98, 98, 185, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 235, 255, 255, 255, 33, 0, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 174, 255, 255, 255, 132, 0, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 87, 255, 255, 255, 243, 50, 0, 0, 0, 0, 141, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 4, 217, 255, 255, 255, 239, 118, 43, 19, 39, 173, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 57, 248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 72, 239, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 25, 144, 232, 255, 255, 255, 255, 255, 242, 179, 88, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 31, 58, 65, 56, 30, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 81, 90, 90, 64, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 5, 251, 255, 255, 114, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 48, 255, 255, 232, 11, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 106, 255, 255, 106, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 162, 247, 201, 3, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ģ' / 0x123
        'ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 24, 173, 174, 51, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 164, 255, 255, 26, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 51, 254, 255, 222, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 179, 255, 255, 170, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 11, 162, 163, 163, 86, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 20, 149, 230, 255, 241, 174, 46, 0, 78, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 226, 255, 255, 255, 255, 255, 248, 71, 201, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 190, 255, 255, 255, 255, 255, 255, 255, 248, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 59, 255, 255, 255, 255, 182, 87, 72, 149, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 192, 3, 0, 0, 0, 147, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 86, 0, 0, 0, 0, 48, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 219, 255, 255, 255, 40, 0, 0, 0, 0, 6, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 231, 255, 255, 255, 19, 0, 0, 0, 0, 0, 248, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 223, 255, 255, 255, 34, 0, 0, 0, 0, 2, 251, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 201, 255, 255, 255, 68, 0, 0, 0, 0, 25, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 155, 255, 255, 255, 162, 0, 0, 0, 0, 110, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 79, 255, 255, 255, 254, 126, 25, 13, 89, 243, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 5, 217, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 58, 246, 255, 255, 255, 255, 255, 255, 166, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 55, 204, 255, 255, 255, 235, 109, 53, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 31, 61, 47, 4, 0, 70, 255, 255, 255, 169, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 122, 255, 255, 255, 142, 0, 0, 0,
            ],
            &[
                0, 0, 0, 172, 106, 34, 0, 0, 0, 0, 77, 242, 255, 255, 255, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 228, 255, 255, 239, 210, 209, 240, 255, 255, 255, 255, 222, 4, 0, 0, 0,
            ],
            &[
                0, 0, 0, 228, 255, 255, 255, 255, 255, 255, 255, 255, 255, 244, 48, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 254, 185, 36, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 46, 100, 126, 146, 147, 126, 96, 25, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'Ĥ' / 0x124
        'Ĥ' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 255, 145, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 27, 234, 255, 255, 249, 255, 253, 70, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 6, 196, 255, 254, 108, 66, 245, 255, 234, 29, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 245, 89, 0, 0, 54, 227, 255, 205, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 26, 16, 0, 0, 0, 0, 10, 26, 26, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 55, 71, 71, 71, 9, 0, 0, 0, 0, 0, 66, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 199, 190, 190, 190, 190, 190, 251, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 173, 161, 161, 161, 161, 161, 249, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 52, 55, 55, 55, 9, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĥ' / 0x125
        'ĥ' => Some(&[
            &[
                0, 0, 7, 204, 255, 255, 212, 255, 255, 205, 7, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 155, 255, 255, 141, 1, 149, 255, 255, 155, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 72, 220, 219, 106, 0, 0, 0, 109, 219, 220, 66, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 68, 133, 133, 133, 53, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 97, 17, 139, 217, 248, 243, 196, 94, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 104, 214, 255, 255, 255, 255, 255, 255, 149, 0, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 245, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 255, 223, 100, 68, 120, 243, 255, 255, 255, 157, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 249, 34, 0, 0, 0, 109, 255, 255, 255, 201, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 177, 0, 0, 0, 0, 41, 255, 255, 255, 216, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 128, 0, 0, 0, 0, 18, 255, 255, 255, 217, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 109, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0,
            ],
            &[
                0, 0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 36, 246, 255, 255, 255, 246, 37, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'Ħ' / 0x126
        'Ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 32, 71, 71, 71, 32, 0, 0, 0, 0, 16, 71, 71, 71, 48, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 116, 0, 0, 0, 0, 58, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 116, 0, 0, 0, 0, 58, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 168, 255, 255, 255, 255, 255, 168, 168, 168, 168, 226, 255, 255, 255, 255, 168,
                33, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
                51, 0,
            ],
            &[
                0, 231, 255, 255, 255, 255, 255, 231, 231, 231, 231, 255, 255, 255, 255, 255, 231,
                46, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 116, 0, 0, 0, 0, 58, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 116, 0, 0, 0, 0, 58, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 220, 190, 190, 190, 190, 205, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 204, 161, 161, 161, 161, 182, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 116, 0, 0, 0, 0, 58, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 116, 0, 0, 0, 0, 58, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 116, 0, 0, 0, 0, 58, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 116, 0, 0, 0, 0, 58, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 116, 0, 0, 0, 0, 58, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 116, 0, 0, 0, 0, 58, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 116, 0, 0, 0, 0, 58, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 116, 0, 0, 0, 0, 58, 255, 255, 255, 176, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ħ' / 0x127
        'ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 68, 133, 133, 133, 53, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 92, 223, 255, 255, 255, 194, 92, 92, 92, 36, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 255, 255, 255, 255, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 255, 255, 255, 255, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 45, 177, 255, 255, 255, 146, 45, 45, 45, 18, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 98, 0, 70, 147, 180, 174, 127, 32, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 92, 163, 255, 255, 255, 255, 255, 248, 88, 0, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 214, 255, 255, 255, 255, 255, 255, 255, 248, 36, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 255, 250, 169, 137, 189, 255, 255, 255, 255, 138, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 255, 75, 0, 0, 0, 149, 255, 255, 255, 193, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 198, 0, 0, 0, 0, 50, 255, 255, 255, 213, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 138, 0, 0, 0, 0, 22, 255, 255, 255, 217, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 113, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 103, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĩ' / 0x128
        'Ĩ' => Some(&[
            &[
                0, 0, 0, 0, 0, 100, 207, 210, 136, 26, 0, 1, 213, 242, 36, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 255, 255, 255, 255, 249, 177, 183, 255, 248, 4, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 185, 255, 234, 212, 255, 255, 255, 255, 255, 150, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 237, 255, 34, 0, 46, 160, 244, 253, 178, 15, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 33, 30, 0, 0, 0, 0, 2, 11, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 40, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 66, 139, 168, 198, 248, 255, 255, 253, 202, 173, 144, 91, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 13, 214, 255, 255, 241, 17, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 210, 240, 255, 255, 255, 255, 255, 255, 244, 215, 146, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĩ' / 0x129
        'ĩ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 9, 70, 73, 13, 0, 0, 0, 83, 105, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 18, 208, 255, 255, 241, 138, 40, 47, 249, 255, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 132, 255, 255, 255, 255, 255, 255, 255, 255, 216, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 214, 255, 143, 94, 183, 255, 255, 255, 252, 80, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 165, 167, 7, 0, 0, 42, 128, 145, 62, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 123, 189, 189, 189, 189, 189, 189, 189, 85, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 12, 42, 70, 98, 183, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 17, 146, 255, 255, 255, 141, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 147, 189, 231, 255, 255, 255, 255, 255, 255, 254, 220, 173, 110, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ī' / 0x12a
        'Ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 171, 171, 171, 171, 171, 171, 171, 171, 30, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 40, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 66, 139, 168, 198, 248, 255, 255, 253, 202, 173, 144, 91, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 13, 214, 255, 255, 241, 17, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 210, 240, 255, 255, 255, 255, 255, 255, 244, 215, 146, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ī' / 0x12b
        'ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 119, 119, 119, 119, 119, 119, 119, 119, 21, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 52, 52, 52, 52, 52, 52, 52, 52, 9, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 123, 189, 189, 189, 189, 189, 189, 189, 85, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 12, 42, 70, 98, 183, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 17, 146, 255, 255, 255, 141, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 147, 189, 231, 255, 255, 255, 255, 255, 255, 254, 220, 173, 110, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => Some(&[
            &[
                0, 0, 0, 0, 73, 255, 252, 29, 0, 0, 6, 217, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 247, 255, 231, 142, 130, 207, 255, 254, 55, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 114, 255, 255, 255, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 87, 213, 252, 253, 219, 104, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 40, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 66, 139, 168, 198, 248, 255, 255, 253, 202, 173, 144, 91, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 13, 214, 255, 255, 241, 17, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 210, 240, 255, 255, 255, 255, 255, 255, 244, 215, 146, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 31, 76, 65, 0, 0, 0, 0, 49, 76, 46, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ĭ' / 0x12d
        'ĭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 71, 195, 180, 1, 0, 0, 0, 138, 195, 109, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 50, 255, 255, 121, 8, 1, 77, 246, 255, 98, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 3, 203, 255, 255, 253, 248, 255, 255, 227, 16, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 40, 217, 255, 255, 255, 255, 227, 59, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 7, 95, 140, 143, 100, 13, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 123, 189, 189, 189, 189, 189, 189, 189, 85, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 12, 42, 70, 98, 183, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 17, 146, 255, 255, 255, 141, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 147, 189, 231, 255, 255, 255, 255, 255, 255, 254, 220, 173, 110, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Į' / 0x12e
        'Į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 40, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 66, 139, 168, 198, 248, 255, 255, 253, 202, 173, 144, 91, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 13, 214, 255, 255, 241, 17, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 210, 240, 255, 255, 255, 255, 255, 255, 244, 215, 146, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 100, 251, 241, 42, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 55, 252, 255, 84, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 178, 255, 239, 4, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 224, 255, 249, 39, 15, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 185, 255, 255, 255, 255, 38, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 47, 231, 255, 255, 255, 38, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 13, 73, 94, 63, 3, 0, 0, 0, 0, 0],
        ]),
        // letter: 'į' / 0x12f
        'į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 39, 192, 234, 201, 54, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 255, 202, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 205, 255, 255, 255, 227, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 141, 255, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 5, 100, 140, 103, 7, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 123, 189, 189, 189, 189, 189, 189, 189, 85, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 12, 42, 70, 98, 183, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 17, 146, 255, 255, 255, 141, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 147, 189, 231, 255, 255, 255, 255, 255, 255, 254, 220, 173, 110, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 39, 223, 255, 118, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 211, 255, 173, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 88, 255, 255, 76, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 134, 255, 255, 120, 6, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 95, 255, 255, 255, 255, 127, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 185, 255, 255, 255, 127, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 56, 93, 79, 18, 0, 0, 0, 0],
        ]),
        // letter: 'İ' / 0x130
        'İ' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 13, 229, 255, 255, 249, 40, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 64, 255, 255, 255, 255, 109, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 46, 255, 255, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 143, 247, 249, 168, 4, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 40, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 66, 139, 168, 198, 248, 255, 255, 253, 202, 173, 144, 91, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 13, 214, 255, 255, 241, 17, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 210, 240, 255, 255, 255, 255, 255, 255, 244, 215, 146, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 22, 103, 110, 39, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ı' / 0x131
        'ı' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 123, 189, 189, 189, 189, 189, 189, 189, 85, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 166, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 12, 42, 70, 98, 183, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 17, 146, 255, 255, 255, 141, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 147, 189, 231, 255, 255, 255, 255, 255, 255, 254, 220, 173, 110, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĳ' / 0x132
        'Ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 50, 71, 71, 71, 3, 0, 0, 0, 66, 71, 71, 64, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 12, 0, 0, 0, 237, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 12, 0, 0, 0, 237, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 12, 0, 0, 0, 237, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 12, 0, 0, 0, 237, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 12, 0, 0, 0, 237, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 12, 0, 0, 0, 237, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 12, 0, 0, 0, 237, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 12, 0, 0, 0, 237, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 12, 0, 0, 0, 237, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 12, 0, 0, 0, 237, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 12, 0, 0, 0, 237, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 12, 0, 0, 0, 237, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 180, 255, 255, 255, 12, 0, 0, 0, 241, 255, 255, 226, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 30, 42, 42, 42, 2, 0, 0, 13, 255, 255, 255, 206, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 255, 174, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 101, 113, 75, 48, 37, 57, 132, 248, 255, 255, 255, 99, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 173, 255, 255, 255, 255, 255, 255, 255, 255, 255, 226, 14, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 173, 255, 255, 255, 255, 255, 255, 255, 255, 238, 60, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 134, 241, 255, 255, 255, 255, 255, 237, 157, 29, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 27, 54, 65, 54, 32, 4, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĳ' / 0x133
        'ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 58, 203, 234, 190, 35, 0, 0, 12, 167, 231, 217, 94, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 210, 255, 255, 255, 168, 0, 0, 115, 255, 255, 255, 249, 13, 0, 0, 0,
            ],
            &[
                0, 0, 0, 238, 255, 255, 255, 194, 0, 0, 143, 255, 255, 255, 255, 33, 0, 0, 0,
            ],
            &[
                0, 0, 0, 174, 255, 255, 255, 125, 0, 0, 79, 255, 255, 255, 216, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 12, 111, 139, 91, 2, 0, 0, 0, 75, 138, 120, 22, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 113, 189, 189, 189, 60, 0, 0, 32, 189, 189, 189, 141, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 81, 0, 0, 44, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 81, 0, 0, 44, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 81, 0, 0, 44, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 81, 0, 0, 44, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 81, 0, 0, 44, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 81, 0, 0, 44, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 81, 0, 0, 44, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 81, 0, 0, 44, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 81, 0, 0, 44, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 81, 0, 0, 44, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 81, 0, 0, 44, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 81, 0, 0, 44, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 81, 0, 0, 44, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 81, 0, 0, 44, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 56, 255, 255, 255, 188, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 103, 255, 255, 255, 173, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 42, 10, 0, 0, 0, 36, 223, 255, 255, 255, 133, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 194, 253, 226, 206, 216, 253, 255, 255, 255, 255, 56, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 194, 255, 255, 255, 255, 255, 255, 255, 255, 172, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 194, 255, 255, 255, 255, 255, 255, 253, 157, 10, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 72, 119, 139, 153, 140, 102, 32, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'Ĵ' / 0x134
        'Ĵ' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 54, 252, 255, 255, 255, 179, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 12, 216, 255, 255, 248, 255, 255, 102, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 168, 255, 255, 140, 45, 233, 255, 247, 50, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 120, 255, 252, 116, 0, 0, 36, 211, 255, 225, 15, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 18, 26, 19, 0, 0, 0, 0, 6, 26, 26, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 42, 71, 71, 71, 16, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 255, 58, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 158, 255, 255, 255, 54, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 183, 255, 255, 255, 33, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 23, 244, 255, 255, 249, 9, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 105, 112, 70, 47, 41, 83, 211, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 180, 255, 255, 255, 255, 255, 255, 255, 255, 255, 72, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 180, 255, 255, 255, 255, 255, 255, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 140, 242, 255, 255, 255, 255, 253, 192, 80, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 2, 34, 54, 63, 42, 15, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 45, 55, 55, 55, 16, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĵ' / 0x135
        'ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 2, 154, 174, 174, 174, 88, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 115, 255, 255, 255, 255, 242, 31, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 54, 249, 255, 240, 167, 255, 255, 200, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 25, 227, 255, 234, 52, 0, 138, 255, 255, 151, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 97, 163, 156, 30, 0, 0, 0, 90, 163, 163, 28, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 98, 189, 189, 189, 189, 189, 189, 189, 110, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 132, 255, 255, 255, 255, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 132, 255, 255, 255, 255, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 8, 38, 66, 94, 163, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 85, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 97, 255, 255, 255, 147, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 144, 255, 255, 255, 132, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 47, 6, 0, 0, 0, 57, 243, 255, 255, 255, 92, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 235, 251, 222, 205, 222, 255, 255, 255, 255, 248, 23, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 255, 255, 255, 255, 255, 255, 131, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 235, 255, 255, 255, 255, 255, 255, 248, 129, 2, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 27, 81, 123, 142, 152, 136, 93, 21, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'Ķ' / 0x136
        'Ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 3, 0, 0, 0, 0, 7, 71, 71, 71, 71, 13, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 0, 0, 135, 255, 255, 255, 186, 1, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 0, 53, 252, 255, 255, 236, 26, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 7, 213, 255, 255, 255, 82, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 135, 255, 255, 255, 157, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 53, 252, 255, 255, 220, 12, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 7, 213, 255, 255, 252, 56, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 132, 255, 255, 255, 128, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 54, 250, 255, 255, 199, 4, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 191, 255, 255, 255, 94, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 214, 3, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 104, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 248, 116, 252, 255, 255, 232, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 71, 0, 156, 255, 255, 255, 132, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 29, 247, 255, 255, 246, 27, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 141, 255, 255, 255, 159, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 20, 242, 255, 255, 253, 48, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 0, 126, 255, 255, 255, 187, 0, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 0, 12, 234, 255, 255, 255, 73, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 12, 0, 0, 0, 0, 111, 255, 255, 255, 212, 3, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 49, 90, 90, 90, 6, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 166, 255, 255, 204, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 214, 255, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 17, 254, 255, 195, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 75, 247, 244, 47, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ķ' / 0x137
        'ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 50, 133, 133, 133, 71, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 77, 189, 189, 189, 183, 28, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 48, 244, 255, 255, 254, 90, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 26, 227, 255, 255, 255, 119, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 10, 204, 255, 255, 255, 150, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 133, 1, 173, 255, 255, 255, 177, 3, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 121, 125, 255, 255, 255, 200, 10, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 161, 253, 255, 255, 255, 60, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 198, 2, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 251, 255, 255, 255, 116, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 183, 37, 232, 255, 255, 247, 39, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 170, 1, 0, 84, 255, 255, 255, 199, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 174, 255, 255, 255, 117, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 25, 239, 255, 255, 247, 40, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 99, 255, 255, 255, 199, 3, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 0, 188, 255, 255, 255, 118, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 17, 90, 90, 90, 37, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 253, 40, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 124, 255, 255, 168, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 182, 255, 249, 36, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 235, 247, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĸ' / 0x138
        'ĸ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 72, 189, 189, 189, 101, 0, 0, 0, 0, 77, 189, 189, 189, 183, 28, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 48, 244, 255, 255, 254, 90, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 26, 227, 255, 255, 255, 119, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 10, 204, 255, 255, 255, 150, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 133, 1, 173, 255, 255, 255, 177, 3, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 121, 125, 255, 255, 255, 200, 10, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 161, 253, 255, 255, 255, 60, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 198, 2, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 251, 255, 255, 255, 116, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 183, 37, 232, 255, 255, 247, 39, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 170, 1, 0, 84, 255, 255, 255, 199, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 174, 255, 255, 255, 117, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 25, 239, 255, 255, 247, 40, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 99, 255, 255, 255, 199, 3, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 136, 0, 0, 0, 0, 0, 188, 255, 255, 255, 118, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĺ' / 0x139
        'Ĺ' => Some(&[
            &[
                0, 0, 0, 0, 0, 72, 255, 255, 255, 221, 26, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 13, 223, 255, 255, 219, 31, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 157, 255, 255, 209, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 80, 255, 255, 184, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 13, 26, 26, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 71, 71, 71, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 164, 116, 116, 116, 116, 116, 116, 116, 36, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 50, 55, 55, 55, 23, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĺ' / 0x13a
        'ĺ' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 86, 255, 255, 255, 123, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 27, 237, 255, 252, 105, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 143, 220, 216, 75, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 57, 133, 133, 133, 133, 133, 133, 133, 52, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 255, 255, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 26, 79, 132, 215, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 19, 157, 255, 255, 255, 129, 11, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 147, 190, 232, 255, 255, 255, 255, 255, 255, 254, 219, 172, 110, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 173, 255, 255, 255, 125, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'Ļ' / 0x13b
        'Ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 71, 71, 71, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 164, 116, 116, 116, 116, 116, 116, 116, 36, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 86, 90, 90, 59, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 15, 255, 255, 255, 100, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 62, 255, 255, 224, 6, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 120, 255, 255, 92, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 175, 247, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ļ' / 0x13c
        'ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 57, 133, 133, 133, 133, 133, 133, 133, 52, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 255, 255, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 26, 79, 132, 215, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 19, 157, 255, 255, 255, 129, 11, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 147, 190, 232, 255, 255, 255, 255, 255, 255, 254, 219, 172, 110, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 32, 90, 90, 90, 23, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 118, 255, 255, 239, 13, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 166, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 223, 255, 231, 12, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 28, 247, 247, 90, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ľ' / 0x13d
        'Ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 40, 71, 71, 71, 24, 0, 0, 0, 31, 71, 71, 71, 12, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 137, 255, 255, 231, 8, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 184, 255, 255, 116, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 1, 237, 255, 227, 9, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 42, 255, 255, 87, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 2, 10, 10, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 164, 116, 116, 116, 116, 116, 116, 116, 36, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ľ' / 0x13e
        'ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 57, 133, 133, 133, 133, 133, 133, 133, 52, 0, 75, 133, 133, 132, 5, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 101, 0, 176, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 106, 255, 255, 255, 255, 255, 255, 255, 101, 0, 225, 255, 255, 56, 0, 0, 0,
            ],
            &[
                0, 0, 0, 26, 79, 132, 215, 255, 255, 255, 101, 25, 255, 255, 173, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 62, 204, 202, 31, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 157, 255, 255, 255, 129, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 44, 147, 190, 232, 255, 255, 255, 255, 255, 255, 254, 219, 172, 110, 0, 0, 0, 0,
            ],
            &[
                0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0, 0,
            ],
            &[
                0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 71, 71, 71, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 67, 193, 213, 150, 11, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 7, 240, 255, 255, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 33, 255, 255, 255, 255, 172, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 6, 236, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 60, 189, 211, 143, 8, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 164, 116, 116, 116, 116, 116, 116, 116, 36, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[
                0, 0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŀ' / 0x140
        'ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 57, 133, 133, 133, 133, 133, 133, 133, 52, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 255, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 106, 255, 255, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 26, 79, 132, 215, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 22, 119, 137, 70, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 4, 217, 255, 255, 255, 84, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 38, 255, 255, 255, 255, 161, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 21, 254, 255, 255, 255, 141, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 126, 252, 255, 213, 27, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 14, 30, 2, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 157, 255, 255, 255, 129, 11, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                44, 147, 190, 232, 255, 255, 255, 255, 255, 255, 254, 219, 172, 110, 0, 0, 0, 0, 0,
            ],
            &[
                97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0, 0, 0,
            ],
            &[
                97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ł' / 0x141
        'Ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 71, 71, 71, 30, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 125, 255, 255, 255, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 125, 255, 255, 255, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 125, 255, 255, 255, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 125, 255, 255, 255, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 125, 255, 255, 255, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 125, 255, 255, 255, 108, 13, 65, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 125, 255, 255, 255, 195, 229, 243, 26, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 125, 255, 255, 255, 255, 255, 255, 165, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 134, 255, 255, 255, 255, 255, 212, 63, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 74, 255, 255, 255, 255, 255, 115, 4, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 152, 255, 255, 255, 255, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 99, 255, 255, 255, 255, 255, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 2, 193, 244, 255, 255, 255, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 125, 255, 255, 255, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 125, 255, 255, 255, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 125, 255, 255, 255, 175, 116, 116, 116, 116, 116, 116, 116, 45, 0, 0,
            ],
            &[
                0, 0, 0, 0, 125, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0, 0,
            ],
            &[
                0, 0, 0, 0, 125, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0, 0,
            ],
            &[
                0, 0, 0, 0, 125, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ł' / 0x142
        'ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 57, 133, 133, 133, 133, 133, 133, 133, 52, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 255, 255, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 26, 79, 132, 215, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 69, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 255, 255, 197, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 255, 255, 255, 88, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 185, 255, 255, 255, 255, 248, 130, 8, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 148, 255, 255, 255, 255, 255, 37, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 86, 231, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 129, 255, 255, 255, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 10, 227, 199, 181, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 34, 1, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 133, 255, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 19, 157, 255, 255, 255, 129, 11, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 44, 147, 190, 232, 255, 255, 255, 255, 255, 255, 254, 219, 172, 110, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ń' / 0x143
        'Ń' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 100, 255, 255, 255, 204, 14, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 26, 238, 255, 255, 203, 18, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 1, 185, 255, 255, 191, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 108, 255, 255, 163, 6, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 16, 26, 25, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 71, 41, 0, 0, 0, 0, 35, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 201, 0, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 254, 30, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 113, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 222, 255, 255, 196, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 151, 253, 255, 254, 26, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 141, 195, 255, 255, 108, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 155, 112, 255, 255, 191, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 166, 30, 255, 255, 252, 22, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 175, 0, 202, 255, 255, 103, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 119, 255, 255, 187, 0, 126, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 36, 255, 255, 251, 19, 123, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 209, 255, 255, 98, 118, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 126, 255, 255, 182, 111, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 43, 255, 255, 249, 117, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 216, 255, 255, 182, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 133, 255, 255, 248, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 50, 255, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 0, 223, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 0, 140, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 55, 55, 55, 55, 17, 0, 0, 0, 0],
        ]),
        // letter: 'ń' / 0x144
        'ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 33, 174, 174, 174, 172, 28, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 255, 254, 99, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 97, 255, 255, 251, 93, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 35, 242, 255, 244, 77, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 99, 163, 162, 50, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 97, 189, 189, 110, 0, 28, 144, 216, 248, 245, 200, 99, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 224, 42, 236, 255, 255, 255, 255, 255, 255, 156, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 244, 255, 255, 255, 255, 255, 255, 255, 255, 69, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 236, 114, 71, 125, 248, 255, 255, 255, 158, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 254, 54, 0, 0, 0, 118, 255, 255, 255, 202, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 197, 0, 0, 0, 0, 42, 255, 255, 255, 216, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 138, 0, 0, 0, 0, 18, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 115, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ņ' / 0x145
        'Ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 71, 41, 0, 0, 0, 0, 35, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 201, 0, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 254, 30, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 113, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 222, 255, 255, 196, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 151, 253, 255, 254, 26, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 141, 195, 255, 255, 108, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 155, 112, 255, 255, 191, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 166, 30, 255, 255, 252, 22, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 175, 0, 202, 255, 255, 103, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 119, 255, 255, 187, 0, 126, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 36, 255, 255, 251, 19, 123, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 209, 255, 255, 98, 118, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 126, 255, 255, 182, 111, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 43, 255, 255, 249, 117, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 216, 255, 255, 182, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 133, 255, 255, 248, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 50, 255, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 0, 223, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 0, 140, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 68, 90, 90, 76, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 149, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 15, 254, 255, 248, 30, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 71, 255, 255, 140, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 128, 247, 224, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ņ' / 0x146
        'ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 97, 189, 189, 110, 0, 28, 144, 216, 248, 245, 200, 99, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 224, 42, 236, 255, 255, 255, 255, 255, 255, 156, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 244, 255, 255, 255, 255, 255, 255, 255, 255, 69, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 236, 114, 71, 125, 248, 255, 255, 255, 158, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 254, 54, 0, 0, 0, 118, 255, 255, 255, 202, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 197, 0, 0, 0, 0, 42, 255, 255, 255, 216, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 138, 0, 0, 0, 0, 18, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 115, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 59, 90, 90, 86, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 194, 255, 255, 176, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 1, 240, 255, 254, 52, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 43, 255, 255, 168, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 102, 247, 236, 28, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ň' / 0x147
        'Ň' => Some(&[
            &[
                0, 0, 0, 0, 98, 255, 255, 146, 3, 0, 61, 231, 255, 210, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 143, 255, 255, 168, 77, 247, 255, 237, 32, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 5, 201, 255, 255, 255, 255, 255, 76, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 40, 248, 255, 255, 255, 154, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 20, 26, 26, 26, 6, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 71, 41, 0, 0, 0, 0, 35, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 201, 0, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 254, 30, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 113, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 222, 255, 255, 196, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 151, 253, 255, 254, 26, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 141, 195, 255, 255, 108, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 155, 112, 255, 255, 191, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 166, 30, 255, 255, 252, 22, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 175, 0, 202, 255, 255, 103, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 119, 255, 255, 187, 0, 126, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 36, 255, 255, 251, 19, 123, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 209, 255, 255, 98, 118, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 126, 255, 255, 182, 111, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 43, 255, 255, 249, 117, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 216, 255, 255, 182, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 133, 255, 255, 248, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 50, 255, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 0, 223, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 0, 140, 255, 255, 255, 255, 252, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 39, 55, 46, 0, 0, 0, 0, 20, 55, 55, 9, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ň' / 0x148
        'ň' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 115, 174, 163, 27, 0, 0, 1, 119, 174, 172, 16, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 30, 231, 255, 232, 46, 7, 175, 255, 255, 117, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 63, 252, 255, 236, 192, 255, 255, 171, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 129, 255, 255, 255, 255, 225, 14, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 6, 152, 163, 163, 163, 63, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 97, 189, 189, 110, 0, 28, 144, 216, 248, 245, 200, 99, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 224, 42, 236, 255, 255, 255, 255, 255, 255, 156, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 244, 255, 255, 255, 255, 255, 255, 255, 255, 69, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 236, 114, 71, 125, 248, 255, 255, 255, 158, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 254, 54, 0, 0, 0, 118, 255, 255, 255, 202, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 197, 0, 0, 0, 0, 42, 255, 255, 255, 216, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 138, 0, 0, 0, 0, 18, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 115, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŉ' / 0x149
        'ŉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 55, 71, 71, 70, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 227, 255, 255, 244, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 16, 255, 255, 255, 176, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 64, 255, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 115, 255, 255, 241, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 168, 255, 255, 150, 46, 189, 189, 161, 0, 12, 119, 209, 244, 247, 205, 107, 2, 0,
            ],
            &[
                0, 226, 255, 255, 42, 62, 255, 255, 255, 50, 202, 255, 255, 255, 255, 255, 255,
                162, 0,
            ],
            &[
                3, 86, 86, 76, 0, 62, 255, 255, 255, 244, 255, 255, 255, 255, 255, 255, 255, 255,
                72,
            ],
            &[
                0, 0, 0, 0, 0, 62, 255, 255, 255, 255, 252, 133, 73, 117, 245, 255, 255, 255, 160,
            ],
            &[
                0, 0, 0, 0, 0, 62, 255, 255, 255, 255, 114, 0, 0, 0, 114, 255, 255, 255, 202,
            ],
            &[
                0, 0, 0, 0, 0, 62, 255, 255, 255, 248, 14, 0, 0, 0, 41, 255, 255, 255, 216,
            ],
            &[
                0, 0, 0, 0, 0, 62, 255, 255, 255, 205, 0, 0, 0, 0, 18, 255, 255, 255, 217,
            ],
            &[
                0, 0, 0, 0, 0, 62, 255, 255, 255, 183, 0, 0, 0, 0, 16, 255, 255, 255, 217,
            ],
            &[
                0, 0, 0, 0, 0, 62, 255, 255, 255, 171, 0, 0, 0, 0, 16, 255, 255, 255, 217,
            ],
            &[
                0, 0, 0, 0, 0, 62, 255, 255, 255, 171, 0, 0, 0, 0, 16, 255, 255, 255, 217,
            ],
            &[
                0, 0, 0, 0, 0, 62, 255, 255, 255, 171, 0, 0, 0, 0, 16, 255, 255, 255, 217,
            ],
            &[
                0, 0, 0, 0, 0, 62, 255, 255, 255, 171, 0, 0, 0, 0, 16, 255, 255, 255, 217,
            ],
            &[
                0, 0, 0, 0, 0, 62, 255, 255, 255, 171, 0, 0, 0, 0, 16, 255, 255, 255, 217,
            ],
            &[
                0, 0, 0, 0, 0, 62, 255, 255, 255, 171, 0, 0, 0, 0, 16, 255, 255, 255, 217,
            ],
            &[
                0, 0, 0, 0, 0, 62, 255, 255, 255, 171, 0, 0, 0, 0, 16, 255, 255, 255, 217,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 71, 41, 0, 0, 0, 0, 35, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 201, 0, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 254, 30, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 255, 255, 113, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 222, 255, 255, 196, 0, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 151, 253, 255, 254, 26, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 141, 195, 255, 255, 108, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 155, 112, 255, 255, 191, 0, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 166, 30, 255, 255, 252, 22, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 175, 0, 202, 255, 255, 103, 0, 127, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 119, 255, 255, 187, 0, 126, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 36, 255, 255, 251, 19, 123, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 209, 255, 255, 98, 118, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 126, 255, 255, 182, 111, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 43, 255, 255, 249, 117, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 216, 255, 255, 182, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 133, 255, 255, 248, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 50, 255, 255, 255, 255, 255, 255, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 0, 223, 255, 255, 255, 255, 255, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 178, 0, 0, 0, 0, 140, 255, 255, 255, 255, 255, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 73, 255, 255, 241, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 211, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 64, 78, 62, 93, 159, 255, 255, 255, 137, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 146, 255, 255, 255, 255, 255, 255, 229, 21, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 146, 255, 255, 255, 255, 255, 198, 37, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 54, 129, 150, 147, 119, 55, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŋ' / 0x14b
        'ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 97, 189, 189, 110, 0, 28, 144, 216, 248, 245, 200, 99, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 224, 42, 236, 255, 255, 255, 255, 255, 255, 156, 0, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 244, 255, 255, 255, 255, 255, 255, 255, 255, 69, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 255, 236, 114, 71, 125, 248, 255, 255, 255, 158, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 254, 54, 0, 0, 0, 118, 255, 255, 255, 202, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 197, 0, 0, 0, 0, 42, 255, 255, 255, 216, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 138, 0, 0, 0, 0, 18, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 115, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 33, 255, 255, 255, 255, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 33, 255, 255, 255, 255, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 255, 255, 102, 0, 0, 0, 0, 33, 255, 255, 255, 255, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 255, 255, 255, 217, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 27, 255, 255, 255, 216, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 87, 255, 255, 255, 196, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 59, 181, 175, 248, 255, 255, 255, 140, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 71, 255, 255, 255, 255, 255, 246, 35, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 71, 255, 255, 255, 255, 237, 73, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 13, 81, 111, 111, 79, 9, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'Ō' / 0x14c
        'Ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 171, 171, 171, 171, 171, 171, 171, 171, 30, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 47, 114, 144, 144, 119, 55, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 187, 255, 255, 255, 255, 255, 255, 208, 49, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 25, 226, 255, 255, 255, 255, 255, 255, 255, 255, 244, 52, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 181, 255, 255, 255, 255, 224, 211, 253, 255, 255, 255, 221, 6, 0, 0, 0,
            ],
            &[
                0, 0, 52, 255, 255, 255, 252, 84, 0, 0, 46, 231, 255, 255, 255, 102, 0, 0, 0,
            ],
            &[
                0, 0, 136, 255, 255, 255, 145, 0, 0, 0, 0, 95, 255, 255, 255, 190, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 50, 0, 0, 0, 0, 17, 252, 255, 255, 251, 7, 0, 0,
            ],
            &[
                0, 2, 249, 255, 255, 250, 4, 0, 0, 0, 0, 0, 204, 255, 255, 255, 45, 0, 0,
            ],
            &[
                0, 17, 255, 255, 255, 221, 0, 0, 0, 0, 0, 0, 180, 255, 255, 255, 76, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 208, 0, 0, 0, 0, 0, 0, 162, 255, 255, 255, 89, 0, 0,
            ],
            &[
                0, 44, 255, 255, 255, 202, 0, 0, 0, 0, 0, 0, 151, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 212, 0, 0, 0, 0, 0, 0, 161, 255, 255, 255, 85, 0, 0,
            ],
            &[
                0, 18, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 179, 255, 255, 255, 69, 0, 0,
            ],
            &[
                0, 0, 236, 255, 255, 255, 12, 0, 0, 0, 0, 0, 219, 255, 255, 255, 32, 0, 0,
            ],
            &[
                0, 0, 190, 255, 255, 255, 77, 0, 0, 0, 0, 31, 254, 255, 255, 239, 1, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 190, 1, 0, 0, 0, 145, 255, 255, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 24, 244, 255, 255, 255, 168, 53, 45, 134, 254, 255, 255, 255, 64, 0, 0, 0,
            ],
            &[
                0, 0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 175, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 163, 255, 255, 255, 255, 255, 255, 255, 255, 202, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 94, 209, 255, 255, 255, 255, 225, 124, 6, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 54, 59, 34, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ō' / 0x14d
        'ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 119, 119, 119, 119, 119, 119, 119, 119, 21, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 52, 52, 52, 52, 52, 52, 52, 52, 9, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 31, 134, 209, 239, 247, 219, 164, 62, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 247, 255, 255, 255, 255, 255, 255, 255, 151, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 82, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 233, 255, 255, 255, 182, 74, 51, 129, 253, 255, 255, 254, 57, 0, 0, 0,
            ],
            &[
                0, 0, 95, 255, 255, 255, 199, 4, 0, 0, 0, 135, 255, 255, 255, 154, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 91, 0, 0, 0, 0, 30, 255, 255, 255, 229, 0, 0, 0,
            ],
            &[
                0, 0, 206, 255, 255, 255, 42, 0, 0, 0, 0, 0, 237, 255, 255, 254, 4, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 18, 0, 0, 0, 0, 0, 221, 255, 255, 255, 24, 0, 0,
            ],
            &[
                0, 0, 220, 255, 255, 255, 27, 0, 0, 0, 0, 0, 231, 255, 255, 255, 14, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 68, 0, 0, 0, 0, 16, 254, 255, 255, 240, 0, 0, 0,
            ],
            &[
                0, 0, 132, 255, 255, 255, 160, 0, 0, 0, 0, 100, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 36, 253, 255, 255, 254, 116, 5, 0, 65, 238, 255, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 255, 255, 249, 241, 255, 255, 255, 255, 194, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 7, 174, 255, 255, 255, 255, 255, 255, 255, 255, 209, 25, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 97, 212, 255, 255, 255, 255, 228, 127, 11, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 55, 60, 35, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => Some(&[
            &[
                0, 0, 0, 0, 73, 255, 252, 29, 0, 0, 6, 217, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 247, 255, 231, 142, 130, 207, 255, 254, 55, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 114, 255, 255, 255, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 87, 213, 252, 253, 219, 104, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 47, 114, 144, 144, 119, 55, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 187, 255, 255, 255, 255, 255, 255, 208, 49, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 25, 226, 255, 255, 255, 255, 255, 255, 255, 255, 244, 52, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 181, 255, 255, 255, 255, 224, 211, 253, 255, 255, 255, 221, 6, 0, 0, 0,
            ],
            &[
                0, 0, 52, 255, 255, 255, 252, 84, 0, 0, 46, 231, 255, 255, 255, 102, 0, 0, 0,
            ],
            &[
                0, 0, 136, 255, 255, 255, 145, 0, 0, 0, 0, 95, 255, 255, 255, 190, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 50, 0, 0, 0, 0, 17, 252, 255, 255, 251, 7, 0, 0,
            ],
            &[
                0, 2, 249, 255, 255, 250, 4, 0, 0, 0, 0, 0, 204, 255, 255, 255, 45, 0, 0,
            ],
            &[
                0, 17, 255, 255, 255, 221, 0, 0, 0, 0, 0, 0, 180, 255, 255, 255, 76, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 208, 0, 0, 0, 0, 0, 0, 162, 255, 255, 255, 89, 0, 0,
            ],
            &[
                0, 44, 255, 255, 255, 202, 0, 0, 0, 0, 0, 0, 151, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 212, 0, 0, 0, 0, 0, 0, 161, 255, 255, 255, 85, 0, 0,
            ],
            &[
                0, 18, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 179, 255, 255, 255, 69, 0, 0,
            ],
            &[
                0, 0, 236, 255, 255, 255, 12, 0, 0, 0, 0, 0, 219, 255, 255, 255, 32, 0, 0,
            ],
            &[
                0, 0, 190, 255, 255, 255, 77, 0, 0, 0, 0, 31, 254, 255, 255, 239, 1, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 190, 1, 0, 0, 0, 145, 255, 255, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 24, 244, 255, 255, 255, 168, 53, 45, 134, 254, 255, 255, 255, 64, 0, 0, 0,
            ],
            &[
                0, 0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 175, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 163, 255, 255, 255, 255, 255, 255, 255, 255, 202, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 94, 209, 255, 255, 255, 255, 225, 124, 6, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 54, 59, 34, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 31, 76, 65, 0, 0, 0, 0, 49, 76, 46, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ŏ' / 0x14f
        'ŏ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 71, 195, 180, 1, 0, 0, 0, 138, 195, 109, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 50, 255, 255, 121, 8, 1, 77, 246, 255, 98, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 3, 203, 255, 255, 253, 248, 255, 255, 227, 16, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 40, 217, 255, 255, 255, 255, 227, 59, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 7, 95, 140, 143, 100, 13, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 31, 134, 209, 239, 247, 219, 164, 62, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 247, 255, 255, 255, 255, 255, 255, 255, 151, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 82, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 233, 255, 255, 255, 182, 74, 51, 129, 253, 255, 255, 254, 57, 0, 0, 0,
            ],
            &[
                0, 0, 95, 255, 255, 255, 199, 4, 0, 0, 0, 135, 255, 255, 255, 154, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 91, 0, 0, 0, 0, 30, 255, 255, 255, 229, 0, 0, 0,
            ],
            &[
                0, 0, 206, 255, 255, 255, 42, 0, 0, 0, 0, 0, 237, 255, 255, 254, 4, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 18, 0, 0, 0, 0, 0, 221, 255, 255, 255, 24, 0, 0,
            ],
            &[
                0, 0, 220, 255, 255, 255, 27, 0, 0, 0, 0, 0, 231, 255, 255, 255, 14, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 68, 0, 0, 0, 0, 16, 254, 255, 255, 240, 0, 0, 0,
            ],
            &[
                0, 0, 132, 255, 255, 255, 160, 0, 0, 0, 0, 100, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 36, 253, 255, 255, 254, 116, 5, 0, 65, 238, 255, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 255, 255, 249, 241, 255, 255, 255, 255, 194, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 7, 174, 255, 255, 255, 255, 255, 255, 255, 255, 209, 25, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 97, 212, 255, 255, 255, 255, 228, 127, 11, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 55, 60, 35, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ő' / 0x150
        'Ő' => Some(&[
            &[
                0, 0, 0, 0, 0, 2, 209, 255, 255, 203, 21, 236, 255, 255, 166, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 100, 255, 255, 225, 31, 137, 255, 255, 201, 12, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 16, 233, 255, 225, 34, 39, 249, 255, 201, 15, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 144, 255, 203, 25, 0, 185, 255, 181, 11, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 19, 26, 6, 0, 0, 23, 26, 3, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 47, 114, 144, 144, 119, 55, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 187, 255, 255, 255, 255, 255, 255, 208, 49, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 25, 226, 255, 255, 255, 255, 255, 255, 255, 255, 244, 52, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 181, 255, 255, 255, 255, 224, 211, 253, 255, 255, 255, 221, 6, 0, 0, 0,
            ],
            &[
                0, 0, 52, 255, 255, 255, 252, 84, 0, 0, 46, 231, 255, 255, 255, 102, 0, 0, 0,
            ],
            &[
                0, 0, 136, 255, 255, 255, 145, 0, 0, 0, 0, 95, 255, 255, 255, 190, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 50, 0, 0, 0, 0, 17, 252, 255, 255, 251, 7, 0, 0,
            ],
            &[
                0, 2, 249, 255, 255, 250, 4, 0, 0, 0, 0, 0, 204, 255, 255, 255, 45, 0, 0,
            ],
            &[
                0, 17, 255, 255, 255, 221, 0, 0, 0, 0, 0, 0, 180, 255, 255, 255, 76, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 208, 0, 0, 0, 0, 0, 0, 162, 255, 255, 255, 89, 0, 0,
            ],
            &[
                0, 44, 255, 255, 255, 202, 0, 0, 0, 0, 0, 0, 151, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 212, 0, 0, 0, 0, 0, 0, 161, 255, 255, 255, 85, 0, 0,
            ],
            &[
                0, 18, 255, 255, 255, 229, 0, 0, 0, 0, 0, 0, 179, 255, 255, 255, 69, 0, 0,
            ],
            &[
                0, 0, 236, 255, 255, 255, 12, 0, 0, 0, 0, 0, 219, 255, 255, 255, 32, 0, 0,
            ],
            &[
                0, 0, 190, 255, 255, 255, 77, 0, 0, 0, 0, 31, 254, 255, 255, 239, 1, 0, 0,
            ],
            &[
                0, 0, 113, 255, 255, 255, 190, 1, 0, 0, 0, 145, 255, 255, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 24, 244, 255, 255, 255, 168, 53, 45, 134, 254, 255, 255, 255, 64, 0, 0, 0,
            ],
            &[
                0, 0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 175, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 163, 255, 255, 255, 255, 255, 255, 255, 255, 202, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 94, 209, 255, 255, 255, 255, 225, 124, 6, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 54, 59, 34, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 27, 55, 55, 55, 10, 36, 55, 55, 55, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ő' / 0x151
        'ő' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 108, 174, 174, 170, 16, 136, 174, 174, 157, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 30, 247, 255, 255, 119, 59, 255, 255, 254, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 169, 255, 255, 140, 2, 205, 255, 254, 100, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 66, 255, 255, 131, 0, 106, 255, 251, 95, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 113, 163, 91, 0, 0, 140, 163, 70, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 31, 134, 209, 239, 247, 219, 164, 62, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 247, 255, 255, 255, 255, 255, 255, 255, 151, 5, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 82, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 233, 255, 255, 255, 182, 74, 51, 129, 253, 255, 255, 254, 57, 0, 0, 0,
            ],
            &[
                0, 0, 95, 255, 255, 255, 199, 4, 0, 0, 0, 135, 255, 255, 255, 154, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 91, 0, 0, 0, 0, 30, 255, 255, 255, 229, 0, 0, 0,
            ],
            &[
                0, 0, 206, 255, 255, 255, 42, 0, 0, 0, 0, 0, 237, 255, 255, 254, 4, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 18, 0, 0, 0, 0, 0, 221, 255, 255, 255, 24, 0, 0,
            ],
            &[
                0, 0, 220, 255, 255, 255, 27, 0, 0, 0, 0, 0, 231, 255, 255, 255, 14, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 255, 68, 0, 0, 0, 0, 16, 254, 255, 255, 240, 0, 0, 0,
            ],
            &[
                0, 0, 132, 255, 255, 255, 160, 0, 0, 0, 0, 100, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 36, 253, 255, 255, 254, 116, 5, 0, 65, 238, 255, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 255, 255, 249, 241, 255, 255, 255, 255, 194, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 7, 174, 255, 255, 255, 255, 255, 255, 255, 255, 209, 25, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 97, 212, 255, 255, 255, 255, 228, 127, 11, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 26, 55, 60, 35, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Œ' / 0x152
        'Œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 15, 93, 128, 133, 92, 71, 71, 71, 71, 71, 71, 46, 0, 0,
            ],
            &[
                0, 0, 0, 0, 96, 244, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 168, 0, 0,
            ],
            &[
                0, 0, 0, 82, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 168, 0, 0,
            ],
            &[
                0, 0, 7, 232, 255, 255, 255, 254, 228, 255, 255, 255, 255, 255, 255, 255, 168, 0, 0,
            ],
            &[
                0, 0, 92, 255, 255, 255, 226, 34, 0, 184, 255, 255, 246, 17, 17, 17, 11, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 85, 0, 0, 181, 255, 255, 245, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 223, 255, 255, 253, 14, 0, 0, 181, 255, 255, 245, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 6, 253, 255, 255, 212, 0, 0, 0, 181, 255, 255, 245, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 29, 255, 255, 255, 191, 0, 0, 0, 181, 255, 255, 252, 176, 176, 176, 30, 0, 0,
            ],
            &[
                0, 40, 255, 255, 255, 177, 0, 0, 0, 181, 255, 255, 255, 255, 255, 255, 44, 0, 0,
            ],
            &[
                0, 44, 255, 255, 255, 167, 0, 0, 0, 181, 255, 255, 255, 255, 255, 255, 44, 0, 0,
            ],
            &[
                0, 32, 255, 255, 255, 173, 0, 0, 0, 181, 255, 255, 252, 168, 168, 168, 29, 0, 0,
            ],
            &[
                0, 19, 255, 255, 255, 184, 0, 0, 0, 181, 255, 255, 245, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 3, 247, 255, 255, 209, 0, 0, 0, 181, 255, 255, 245, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 204, 255, 255, 247, 7, 0, 0, 181, 255, 255, 245, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 153, 255, 255, 255, 82, 0, 0, 181, 255, 255, 245, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 63, 255, 255, 255, 229, 79, 47, 207, 255, 255, 249, 89, 89, 89, 58, 0, 0,
            ],
            &[
                0, 0, 0, 209, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 168, 0, 0,
            ],
            &[
                0, 0, 0, 40, 243, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 168, 0, 0,
            ],
            &[
                0, 0, 0, 0, 53, 204, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 168, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 31, 62, 57, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'œ' / 0x153
        'œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 22, 158, 231, 241, 189, 54, 0, 34, 172, 230, 215, 124, 5, 0, 0, 0,
            ],
            &[
                0, 0, 17, 221, 255, 255, 255, 255, 247, 68, 228, 255, 255, 255, 255, 156, 0, 0, 0,
            ],
            &[
                0, 0, 144, 255, 255, 251, 239, 255, 255, 254, 255, 255, 195, 247, 255, 255, 48, 0,
                0,
            ],
            &[
                0, 5, 241, 255, 255, 72, 16, 214, 255, 255, 255, 157, 0, 95, 255, 255, 139, 0, 0,
            ],
            &[
                0, 57, 255, 255, 238, 0, 0, 119, 255, 255, 255, 63, 0, 21, 255, 255, 193, 0, 0,
            ],
            &[
                0, 97, 255, 255, 198, 0, 0, 68, 255, 255, 255, 28, 0, 2, 254, 255, 220, 0, 0,
            ],
            &[
                0, 119, 255, 255, 183, 0, 0, 49, 255, 255, 255, 220, 219, 219, 254, 255, 234, 0, 0,
            ],
            &[
                0, 128, 255, 255, 174, 0, 0, 34, 255, 255, 255, 255, 255, 255, 255, 255, 237, 0, 0,
            ],
            &[
                0, 116, 255, 255, 181, 0, 0, 45, 255, 255, 255, 146, 139, 139, 139, 139, 130, 0, 0,
            ],
            &[
                0, 99, 255, 255, 193, 0, 0, 63, 255, 255, 255, 26, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 56, 255, 255, 228, 0, 0, 99, 255, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 8, 246, 255, 255, 40, 0, 185, 255, 255, 255, 168, 0, 0, 0, 10, 48, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 215, 182, 255, 255, 255, 255, 255, 185, 118, 150, 233, 141, 0,
                0,
            ],
            &[
                0, 0, 32, 237, 255, 255, 255, 255, 255, 130, 247, 255, 255, 255, 255, 255, 141, 0,
                0,
            ],
            &[
                0, 0, 0, 57, 217, 255, 255, 244, 113, 0, 76, 235, 255, 255, 255, 188, 55, 0, 0,
            ],
            &[
                0, 0, 0, 0, 3, 40, 57, 13, 0, 0, 0, 7, 52, 53, 17, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŕ' / 0x154
        'Ŕ' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 255, 255, 179, 5, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 49, 250, 255, 255, 180, 7, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 8, 212, 255, 255, 165, 4, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 143, 255, 255, 134, 1, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 19, 26, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 59, 71, 71, 71, 71, 66, 46, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 254, 201, 96, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 180, 4, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 137, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 29, 20, 53, 166, 255, 255, 255, 241, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 2, 212, 255, 255, 255, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 164, 255, 255, 255, 79, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 189, 255, 255, 255, 64, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 93, 252, 255, 255, 249, 12, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 157, 162, 207, 255, 255, 255, 255, 152, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 207, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 155, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 193, 209, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 13, 233, 255, 255, 240, 22, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 104, 255, 255, 255, 155, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 4, 214, 255, 255, 253, 50, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 77, 255, 255, 255, 195, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 191, 255, 255, 255, 88, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 51, 254, 255, 255, 226, 10, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 0, 163, 255, 255, 255, 129, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 8, 55, 55, 55, 55, 9, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŕ' / 0x155
        'ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 85, 174, 174, 174, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 236, 255, 255, 234, 42, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 173, 255, 255, 228, 41, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 98, 255, 255, 214, 31, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 147, 163, 148, 16, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 31, 189, 189, 189, 189, 189, 185, 3, 0, 63, 185, 237, 252, 224, 112, 0, 0,
            ],
            &[
                0, 0, 41, 255, 255, 255, 255, 255, 255, 49, 97, 252, 255, 255, 255, 255, 111, 0, 0,
            ],
            &[
                0, 0, 41, 255, 255, 255, 255, 255, 255, 160, 249, 255, 255, 255, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 2, 35, 72, 118, 255, 255, 255, 255, 255, 255, 228, 196, 213, 216, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 255, 246, 88, 1, 0, 0, 4, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 255, 113, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 252, 16, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 228, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 48, 255, 255, 255, 227, 26, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 119, 185, 238, 255, 255, 255, 255, 255, 255, 229, 175, 92, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 183, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 183, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŗ' / 0x156
        'Ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 59, 71, 71, 71, 71, 66, 46, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 254, 201, 96, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 180, 4, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 137, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 29, 20, 53, 166, 255, 255, 255, 241, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 2, 212, 255, 255, 255, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 164, 255, 255, 255, 79, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 189, 255, 255, 255, 64, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 93, 252, 255, 255, 249, 12, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 157, 162, 207, 255, 255, 255, 255, 152, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 207, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 155, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 193, 209, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 13, 233, 255, 255, 240, 22, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 104, 255, 255, 255, 155, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 4, 214, 255, 255, 253, 50, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 77, 255, 255, 255, 195, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 191, 255, 255, 255, 88, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 51, 254, 255, 255, 226, 10, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 0, 163, 255, 255, 255, 129, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 90, 90, 90, 47, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 49, 255, 255, 255, 65, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 97, 255, 255, 196, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 154, 255, 254, 58, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 209, 247, 157, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŗ' / 0x157
        'ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 31, 189, 189, 189, 189, 189, 185, 3, 0, 63, 185, 237, 252, 224, 112, 0, 0,
            ],
            &[
                0, 0, 41, 255, 255, 255, 255, 255, 255, 49, 97, 252, 255, 255, 255, 255, 111, 0, 0,
            ],
            &[
                0, 0, 41, 255, 255, 255, 255, 255, 255, 160, 249, 255, 255, 255, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 2, 35, 72, 118, 255, 255, 255, 255, 255, 255, 228, 196, 213, 216, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 255, 246, 88, 1, 0, 0, 4, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 255, 113, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 252, 16, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 228, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 48, 255, 255, 255, 227, 26, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 119, 185, 238, 255, 255, 255, 255, 255, 255, 229, 175, 92, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 183, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 183, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 90, 90, 69, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 242, 255, 255, 128, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 34, 255, 255, 240, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 92, 255, 255, 120, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 148, 247, 211, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ř' / 0x158
        'Ř' => Some(&[
            &[
                0, 0, 0, 133, 255, 253, 118, 0, 0, 84, 243, 255, 184, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 176, 255, 255, 137, 101, 253, 255, 219, 15, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 17, 224, 255, 255, 255, 255, 248, 48, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 67, 255, 255, 255, 255, 120, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 23, 26, 26, 26, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 59, 71, 71, 71, 71, 66, 46, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 254, 201, 96, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 180, 4, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 137, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 29, 20, 53, 166, 255, 255, 255, 241, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 2, 212, 255, 255, 255, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 164, 255, 255, 255, 79, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 189, 255, 255, 255, 64, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 93, 252, 255, 255, 249, 12, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 157, 162, 207, 255, 255, 255, 255, 152, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 207, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 155, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 193, 209, 255, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 13, 233, 255, 255, 240, 22, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 104, 255, 255, 255, 155, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 4, 214, 255, 255, 253, 50, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 77, 255, 255, 255, 195, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 191, 255, 255, 255, 88, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 51, 254, 255, 255, 226, 10, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 19, 0, 0, 0, 0, 163, 255, 255, 255, 129, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 46, 55, 38, 0, 0, 0, 0, 28, 55, 55, 2, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ř' / 0x159
        'ř' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 2, 165, 174, 134, 5, 0, 0, 18, 154, 174, 136, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 83, 254, 255, 190, 12, 36, 223, 255, 246, 51, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 136, 255, 255, 200, 229, 255, 255, 96, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 4, 201, 255, 255, 255, 255, 163, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 43, 163, 163, 163, 160, 17, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 31, 189, 189, 189, 189, 189, 185, 3, 0, 63, 185, 237, 252, 224, 112, 0, 0,
            ],
            &[
                0, 0, 41, 255, 255, 255, 255, 255, 255, 49, 97, 252, 255, 255, 255, 255, 111, 0, 0,
            ],
            &[
                0, 0, 41, 255, 255, 255, 255, 255, 255, 160, 249, 255, 255, 255, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 2, 35, 72, 118, 255, 255, 255, 255, 255, 255, 228, 196, 213, 216, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 255, 246, 88, 1, 0, 0, 4, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 255, 113, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 252, 16, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 228, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 15, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 48, 255, 255, 255, 227, 26, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 119, 185, 238, 255, 255, 255, 255, 255, 255, 229, 175, 92, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 183, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 183, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ś' / 0x15a
        'Ś' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 43, 250, 255, 255, 236, 44, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 3, 199, 255, 255, 235, 49, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 124, 255, 255, 226, 40, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 50, 251, 255, 206, 26, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 10, 26, 26, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 15, 80, 118, 134, 118, 93, 59, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 139, 248, 255, 255, 255, 255, 255, 255, 247, 166, 30, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 247, 18, 0, 0, 0,
            ],
            &[
                0, 0, 0, 76, 255, 255, 255, 255, 236, 191, 199, 233, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 185, 9, 0, 0, 0, 50, 136, 51, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 179, 255, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 162, 255, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 99, 255, 255, 255, 239, 77, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 218, 255, 255, 255, 255, 181, 50, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 42, 229, 255, 255, 255, 255, 254, 164, 29, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 23, 170, 255, 255, 255, 255, 255, 246, 121, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 215, 255, 255, 255, 255, 255, 153, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 2, 93, 231, 255, 255, 255, 255, 66, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 19, 198, 255, 255, 255, 149, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 86, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 12, 47, 0, 0, 0, 0, 0, 0, 0, 103, 255, 255, 255, 160, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 195, 111, 50, 10, 0, 22, 93, 234, 255, 255, 255, 99, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 253, 255, 255, 255, 255, 255, 222, 11, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 229, 44, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 90, 181, 243, 255, 255, 255, 255, 255, 223, 132, 15, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 2, 32, 57, 65, 53, 22, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 43, 55, 55, 55, 30, 0, 0, 0, 0],
        ]),
        // letter: 'ś' / 0x15b
        'ś' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 28, 174, 174, 174, 172, 32, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 174, 255, 255, 255, 105, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 90, 255, 255, 252, 99, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 31, 239, 255, 246, 82, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 94, 163, 163, 55, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 64, 164, 220, 245, 254, 242, 211, 165, 99, 17, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 148, 255, 255, 255, 255, 255, 255, 255, 255, 255, 213, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 79, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 119, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 158, 255, 255, 255, 190, 77, 61, 88, 123, 194, 245, 21, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 178, 255, 255, 255, 111, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 138, 255, 255, 255, 250, 152, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 234, 255, 255, 255, 255, 255, 210, 101, 9, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 39, 194, 255, 255, 255, 255, 255, 255, 234, 85, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 61, 178, 252, 255, 255, 255, 255, 254, 79, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 26, 127, 240, 255, 255, 255, 207, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 18, 0, 0, 0, 0, 0, 0, 47, 255, 255, 255, 250, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 212, 189, 113, 58, 20, 4, 27, 99, 255, 255, 255, 245, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 186, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 237, 48, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 116, 208, 252, 255, 255, 255, 255, 255, 235, 155, 34, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 10, 34, 53, 64, 52, 30, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => Some(&[
            &[
                0, 0, 0, 0, 0, 0, 15, 229, 255, 255, 255, 227, 14, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 165, 255, 255, 248, 255, 255, 165, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 106, 255, 255, 192, 27, 199, 255, 255, 105, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 61, 251, 255, 166, 9, 0, 12, 171, 255, 250, 54, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 26, 25, 0, 0, 0, 0, 0, 25, 26, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 15, 80, 118, 134, 118, 93, 59, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 139, 248, 255, 255, 255, 255, 255, 255, 247, 166, 30, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 247, 18, 0, 0, 0,
            ],
            &[
                0, 0, 0, 76, 255, 255, 255, 255, 236, 191, 199, 233, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 185, 9, 0, 0, 0, 50, 136, 51, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 179, 255, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 162, 255, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 99, 255, 255, 255, 239, 77, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 218, 255, 255, 255, 255, 181, 50, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 42, 229, 255, 255, 255, 255, 254, 164, 29, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 23, 170, 255, 255, 255, 255, 255, 246, 121, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 215, 255, 255, 255, 255, 255, 153, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 2, 93, 231, 255, 255, 255, 255, 66, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 19, 198, 255, 255, 255, 149, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 86, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 12, 47, 0, 0, 0, 0, 0, 0, 0, 103, 255, 255, 255, 160, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 195, 111, 50, 10, 0, 22, 93, 234, 255, 255, 255, 99, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 253, 255, 255, 255, 255, 255, 222, 11, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 229, 44, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 90, 181, 243, 255, 255, 255, 255, 255, 223, 132, 15, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 2, 32, 57, 65, 53, 22, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 31, 55, 55, 55, 30, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŝ' / 0x15d
        'ŝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 6, 160, 174, 174, 174, 79, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 129, 255, 255, 255, 255, 236, 24, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 65, 252, 255, 235, 173, 255, 255, 190, 2, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 32, 233, 255, 229, 43, 1, 150, 255, 255, 137, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 106, 163, 153, 24, 0, 0, 0, 98, 163, 162, 20, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 64, 164, 220, 245, 254, 242, 211, 165, 99, 17, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 148, 255, 255, 255, 255, 255, 255, 255, 255, 255, 213, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 79, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 119, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 158, 255, 255, 255, 190, 77, 61, 88, 123, 194, 245, 21, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 178, 255, 255, 255, 111, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 138, 255, 255, 255, 250, 152, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 234, 255, 255, 255, 255, 255, 210, 101, 9, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 39, 194, 255, 255, 255, 255, 255, 255, 234, 85, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 61, 178, 252, 255, 255, 255, 255, 254, 79, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 26, 127, 240, 255, 255, 255, 207, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 18, 0, 0, 0, 0, 0, 0, 47, 255, 255, 255, 250, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 212, 189, 113, 58, 20, 4, 27, 99, 255, 255, 255, 245, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 186, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 237, 48, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 116, 208, 252, 255, 255, 255, 255, 255, 235, 155, 34, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 10, 34, 53, 64, 52, 30, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ş' / 0x15e
        'Ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 15, 80, 118, 134, 118, 93, 59, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 139, 248, 255, 255, 255, 255, 255, 255, 247, 166, 30, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 247, 18, 0, 0, 0,
            ],
            &[
                0, 0, 0, 76, 255, 255, 255, 255, 236, 191, 199, 233, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 185, 9, 0, 0, 0, 50, 136, 51, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 179, 255, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 162, 255, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 99, 255, 255, 255, 239, 77, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 218, 255, 255, 255, 255, 181, 50, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 42, 229, 255, 255, 255, 255, 254, 164, 29, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 23, 170, 255, 255, 255, 255, 255, 246, 121, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 215, 255, 255, 255, 255, 255, 153, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 2, 93, 231, 255, 255, 255, 255, 66, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 19, 198, 255, 255, 255, 149, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 86, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 12, 47, 0, 0, 0, 0, 0, 0, 0, 103, 255, 255, 255, 160, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 195, 111, 50, 10, 0, 22, 93, 234, 255, 255, 255, 99, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 253, 255, 255, 255, 255, 255, 222, 11, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 229, 44, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 90, 181, 243, 255, 255, 255, 255, 255, 223, 132, 15, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 2, 32, 90, 255, 255, 28, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 139, 255, 255, 192, 32, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 48, 135, 255, 255, 192, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 2, 234, 255, 248, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 113, 141, 185, 255, 255, 216, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 167, 255, 255, 255, 243, 73, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 50, 108, 114, 87, 20, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ş' / 0x15f
        'ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 64, 164, 220, 245, 254, 242, 211, 165, 99, 17, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 148, 255, 255, 255, 255, 255, 255, 255, 255, 255, 213, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 79, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 119, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 158, 255, 255, 255, 190, 77, 61, 88, 123, 194, 245, 21, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 178, 255, 255, 255, 111, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 138, 255, 255, 255, 250, 152, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 234, 255, 255, 255, 255, 255, 210, 101, 9, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 39, 194, 255, 255, 255, 255, 255, 255, 234, 85, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 61, 178, 252, 255, 255, 255, 255, 254, 79, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 26, 127, 240, 255, 255, 255, 207, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 18, 0, 0, 0, 0, 0, 0, 47, 255, 255, 255, 250, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 212, 189, 113, 58, 20, 4, 27, 99, 255, 255, 255, 245, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 186, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 237, 48, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 116, 208, 252, 255, 255, 255, 255, 255, 235, 155, 34, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 10, 34, 64, 255, 255, 54, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 105, 255, 255, 209, 49, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 37, 115, 252, 255, 224, 2, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 202, 255, 255, 27, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 92, 143, 170, 255, 255, 246, 5, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 132, 255, 255, 255, 249, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 38, 105, 116, 91, 30, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'Š' / 0x160
        'Š' => Some(&[
            &[
                0, 0, 0, 0, 47, 244, 255, 192, 20, 0, 28, 202, 255, 241, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 82, 255, 255, 211, 57, 224, 255, 254, 77, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 144, 255, 255, 255, 255, 255, 138, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 8, 217, 255, 255, 255, 210, 6, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 13, 26, 26, 26, 12, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 15, 80, 118, 134, 118, 93, 59, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 139, 248, 255, 255, 255, 255, 255, 255, 247, 166, 30, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 247, 18, 0, 0, 0,
            ],
            &[
                0, 0, 0, 76, 255, 255, 255, 255, 236, 191, 199, 233, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 152, 255, 255, 255, 185, 9, 0, 0, 0, 50, 136, 51, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 179, 255, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 162, 255, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 99, 255, 255, 255, 239, 77, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 218, 255, 255, 255, 255, 181, 50, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 42, 229, 255, 255, 255, 255, 254, 164, 29, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 23, 170, 255, 255, 255, 255, 255, 246, 121, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 77, 215, 255, 255, 255, 255, 255, 153, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 2, 93, 231, 255, 255, 255, 255, 66, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 19, 198, 255, 255, 255, 149, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 86, 255, 255, 255, 175, 0, 0, 0,
            ],
            &[
                0, 0, 12, 47, 0, 0, 0, 0, 0, 0, 0, 103, 255, 255, 255, 160, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 195, 111, 50, 10, 0, 22, 93, 234, 255, 255, 255, 99, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 253, 255, 255, 255, 255, 255, 222, 11, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 229, 44, 0, 0, 0, 0,
            ],
            &[
                0, 0, 3, 90, 181, 243, 255, 255, 255, 255, 255, 223, 132, 15, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 2, 32, 57, 65, 53, 22, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 25, 55, 55, 4, 0, 0, 0, 6, 55, 55, 23, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'š' / 0x161
        'š' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 111, 174, 165, 30, 0, 0, 1, 115, 174, 173, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 26, 228, 255, 234, 50, 5, 170, 255, 255, 124, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 57, 251, 255, 239, 190, 255, 255, 178, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 122, 255, 255, 255, 255, 229, 17, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 4, 149, 163, 163, 163, 68, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 64, 164, 220, 245, 254, 242, 211, 165, 99, 17, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 148, 255, 255, 255, 255, 255, 255, 255, 255, 255, 213, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 79, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 119, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 158, 255, 255, 255, 190, 77, 61, 88, 123, 194, 245, 21, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 178, 255, 255, 255, 111, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 138, 255, 255, 255, 250, 152, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 234, 255, 255, 255, 255, 255, 210, 101, 9, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 39, 194, 255, 255, 255, 255, 255, 255, 234, 85, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 61, 178, 252, 255, 255, 255, 255, 254, 79, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 26, 127, 240, 255, 255, 255, 207, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 18, 0, 0, 0, 0, 0, 0, 47, 255, 255, 255, 250, 2, 0, 0, 0,
            ],
            &[
                0, 0, 0, 212, 189, 113, 58, 20, 4, 27, 99, 255, 255, 255, 245, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 186, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 237, 48, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 116, 208, 252, 255, 255, 255, 255, 255, 235, 155, 34, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 10, 34, 53, 64, 52, 30, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ţ' / 0x162
        'Ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 13, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 27, 0, 0,
            ],
            &[
                0, 48, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0,
                0,
            ],
            &[
                0, 48, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0,
                0,
            ],
            &[
                0, 48, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0,
                0,
            ],
            &[
                0, 4, 24, 24, 24, 24, 24, 226, 255, 255, 255, 34, 24, 24, 24, 24, 9, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 32, 253, 236, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 139, 255, 255, 192, 32, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 48, 135, 255, 255, 192, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 2, 234, 255, 248, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 113, 141, 185, 255, 255, 216, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 167, 255, 255, 255, 243, 73, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 50, 108, 114, 87, 20, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ţ' / 0x163
        'ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 30, 105, 105, 76, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 104, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 150, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 195, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 1, 42, 99, 155, 247, 255, 255, 237, 189, 189, 189, 189, 189, 48, 0, 0, 0,
            ],
            &[
                0, 0, 142, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 65, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 65, 0, 0, 0,
            ],
            &[
                0, 0, 72, 127, 127, 152, 255, 255, 255, 220, 127, 127, 127, 127, 127, 32, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 187, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 35, 255, 255, 255, 220, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 10, 252, 255, 255, 255, 148, 29, 6, 25, 64, 36, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 184, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 46, 245, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 45, 185, 254, 255, 255, 255, 255, 226, 72, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 99, 255, 250, 51, 26, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 195, 255, 251, 161, 11, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 68, 170, 255, 255, 136, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 36, 255, 255, 192, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 146, 142, 207, 255, 255, 160, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 255, 255, 227, 33, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 71, 113, 110, 78, 7, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'Ť' / 0x164
        'Ť' => Some(&[
            &[
                0, 0, 0, 0, 133, 255, 253, 118, 0, 0, 84, 243, 255, 184, 1, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 176, 255, 255, 137, 101, 253, 255, 219, 15, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 17, 224, 255, 255, 255, 255, 248, 48, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 67, 255, 255, 255, 255, 120, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 23, 26, 26, 26, 2, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 13, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 27, 0, 0,
            ],
            &[
                0, 48, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0,
                0,
            ],
            &[
                0, 48, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0,
                0,
            ],
            &[
                0, 48, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0,
                0,
            ],
            &[
                0, 4, 24, 24, 24, 24, 24, 226, 255, 255, 255, 34, 24, 24, 24, 24, 9, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 46, 55, 38, 0, 0, 0, 0, 28, 55, 55, 2, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ť' / 0x165
        'ť' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 38, 209, 209, 209, 80, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 86, 255, 255, 237, 12, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 30, 105, 105, 76, 0, 0, 137, 255, 255, 119, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 104, 255, 255, 184, 0, 0, 195, 255, 223, 9, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 150, 255, 255, 184, 0, 0, 118, 128, 59, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 195, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 1, 42, 99, 155, 247, 255, 255, 237, 189, 189, 189, 189, 189, 48, 0, 0, 0,
            ],
            &[
                0, 0, 142, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 65, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 65, 0, 0, 0,
            ],
            &[
                0, 0, 72, 127, 127, 152, 255, 255, 255, 220, 127, 127, 127, 127, 127, 32, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 187, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 35, 255, 255, 255, 220, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 10, 252, 255, 255, 255, 148, 29, 6, 25, 64, 36, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 184, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 46, 245, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 45, 185, 250, 255, 255, 255, 255, 226, 72, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 14, 47, 64, 51, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŧ' / 0x166
        'Ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 13, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 27, 0, 0,
            ],
            &[
                0, 48, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0,
                0,
            ],
            &[
                0, 48, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0,
                0,
            ],
            &[
                0, 48, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0,
                0,
            ],
            &[
                0, 4, 24, 24, 24, 24, 24, 226, 255, 255, 255, 34, 24, 24, 24, 24, 9, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 7, 38, 38, 38, 255, 255, 255, 255, 49, 38, 38, 15, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 15, 15, 15, 238, 255, 255, 255, 27, 15, 15, 6, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 222, 255, 255, 255, 11, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŧ' / 0x167
        'ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 30, 105, 105, 76, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 104, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 150, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 195, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 1, 42, 99, 155, 247, 255, 255, 237, 189, 189, 189, 189, 189, 48, 0, 0, 0,
            ],
            &[
                0, 0, 142, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 65, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 65, 0, 0, 0,
            ],
            &[
                0, 0, 72, 127, 127, 152, 255, 255, 255, 220, 127, 127, 127, 127, 127, 32, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 30, 52, 52, 102, 255, 255, 255, 237, 52, 52, 52, 52, 52, 20, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 48, 84, 84, 134, 255, 255, 255, 255, 84, 84, 84, 84, 84, 33, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 49, 255, 255, 255, 187, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 35, 255, 255, 255, 220, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 10, 252, 255, 255, 255, 148, 29, 6, 25, 64, 36, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 184, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 46, 245, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 45, 185, 250, 255, 255, 255, 255, 226, 72, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 14, 47, 64, 51, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ũ' / 0x168
        'Ũ' => Some(&[
            &[
                0, 0, 0, 0, 0, 100, 207, 210, 136, 26, 0, 1, 213, 242, 36, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 75, 255, 255, 255, 255, 249, 177, 183, 255, 248, 4, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 185, 255, 234, 212, 255, 255, 255, 255, 255, 150, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 237, 255, 34, 0, 46, 160, 244, 253, 178, 15, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 33, 30, 0, 0, 0, 0, 2, 11, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 9, 0, 0, 0, 0, 0, 66, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 238, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 44, 0, 0, 0, 0, 0, 248, 255, 255, 247, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 255, 255, 76, 0, 0, 0, 0, 23, 255, 255, 255, 224, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 160, 0, 0, 0, 0, 104, 255, 255, 255, 187, 0, 0, 0,
            ],
            &[
                0, 0, 59, 255, 255, 255, 254, 136, 45, 38, 103, 243, 255, 255, 255, 104, 0, 0, 0,
            ],
            &[
                0, 0, 1, 191, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 11, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 217, 255, 255, 255, 255, 255, 255, 255, 255, 230, 49, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 137, 228, 255, 255, 255, 255, 235, 148, 22, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 2, 30, 55, 57, 32, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ũ' / 0x169
        'ũ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 9, 70, 73, 13, 0, 0, 0, 83, 105, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 18, 208, 255, 255, 241, 138, 40, 47, 249, 255, 20, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 132, 255, 255, 255, 255, 255, 255, 255, 255, 216, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 214, 255, 143, 94, 183, 255, 255, 255, 252, 80, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 165, 167, 7, 0, 0, 42, 128, 145, 62, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 123, 189, 189, 189, 50, 0, 0, 0, 0, 37, 189, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 55, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 68, 0, 0, 0, 0, 69, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 83, 0, 0, 0, 0, 108, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 155, 255, 255, 255, 132, 0, 0, 0, 0, 192, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 122, 255, 255, 255, 237, 67, 5, 23, 142, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 46, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 157, 255, 255, 255, 255, 255, 255, 255, 130, 199, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 4, 134, 245, 255, 255, 255, 219, 96, 0, 109, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 7, 51, 59, 27, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ū' / 0x16a
        'Ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 171, 171, 171, 171, 171, 171, 171, 171, 30, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 9, 0, 0, 0, 0, 0, 66, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 238, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 44, 0, 0, 0, 0, 0, 248, 255, 255, 247, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 255, 255, 76, 0, 0, 0, 0, 23, 255, 255, 255, 224, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 160, 0, 0, 0, 0, 104, 255, 255, 255, 187, 0, 0, 0,
            ],
            &[
                0, 0, 59, 255, 255, 255, 254, 136, 45, 38, 103, 243, 255, 255, 255, 104, 0, 0, 0,
            ],
            &[
                0, 0, 1, 191, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 11, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 217, 255, 255, 255, 255, 255, 255, 255, 255, 230, 49, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 137, 228, 255, 255, 255, 255, 235, 148, 22, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 2, 30, 55, 57, 32, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ū' / 0x16b
        'ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 119, 119, 119, 119, 119, 119, 119, 119, 21, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 52, 52, 52, 52, 52, 52, 52, 52, 9, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 123, 189, 189, 189, 50, 0, 0, 0, 0, 37, 189, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 55, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 68, 0, 0, 0, 0, 69, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 83, 0, 0, 0, 0, 108, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 155, 255, 255, 255, 132, 0, 0, 0, 0, 192, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 122, 255, 255, 255, 237, 67, 5, 23, 142, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 46, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 157, 255, 255, 255, 255, 255, 255, 255, 130, 199, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 4, 134, 245, 255, 255, 255, 219, 96, 0, 109, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 7, 51, 59, 27, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => Some(&[
            &[
                0, 0, 0, 0, 73, 255, 252, 29, 0, 0, 6, 217, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 247, 255, 231, 142, 130, 207, 255, 254, 55, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 114, 255, 255, 255, 255, 255, 255, 149, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 87, 213, 252, 253, 219, 104, 1, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 7, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 9, 0, 0, 0, 0, 0, 66, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 238, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 44, 0, 0, 0, 0, 0, 248, 255, 255, 247, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 255, 255, 76, 0, 0, 0, 0, 23, 255, 255, 255, 224, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 160, 0, 0, 0, 0, 104, 255, 255, 255, 187, 0, 0, 0,
            ],
            &[
                0, 0, 59, 255, 255, 255, 254, 136, 45, 38, 103, 243, 255, 255, 255, 104, 0, 0, 0,
            ],
            &[
                0, 0, 1, 191, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 11, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 217, 255, 255, 255, 255, 255, 255, 255, 255, 230, 49, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 137, 228, 255, 255, 255, 255, 235, 148, 22, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 2, 30, 55, 57, 32, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 31, 76, 65, 0, 0, 0, 0, 49, 76, 46, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ŭ' / 0x16d
        'ŭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 71, 195, 180, 1, 0, 0, 0, 138, 195, 109, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 50, 255, 255, 121, 8, 1, 77, 246, 255, 98, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 3, 203, 255, 255, 253, 248, 255, 255, 227, 16, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 40, 217, 255, 255, 255, 255, 227, 59, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 7, 95, 140, 143, 100, 13, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 123, 189, 189, 189, 50, 0, 0, 0, 0, 37, 189, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 55, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 68, 0, 0, 0, 0, 69, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 83, 0, 0, 0, 0, 108, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 155, 255, 255, 255, 132, 0, 0, 0, 0, 192, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 122, 255, 255, 255, 237, 67, 5, 23, 142, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 46, 255, 255, 255, 255, 255, 254, 255, 255, 255, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 157, 255, 255, 255, 255, 255, 255, 255, 130, 199, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 0, 4, 134, 245, 255, 255, 255, 219, 96, 0, 109, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 7, 51, 59, 27, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ů' / 0x16e
        'Ů' => Some(&[
            &[
                0, 0, 0, 0, 0, 15, 255, 243, 15, 4, 222, 255, 57, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 13, 254, 242, 12, 4, 223, 255, 55, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 190, 255, 239, 232, 255, 219, 7, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 27, 183, 252, 253, 195, 41, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 6, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 55, 71, 71, 71, 9, 0, 0, 0, 0, 0, 66, 71, 71, 70, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 200, 255, 255, 255, 33, 0, 0, 0, 0, 0, 238, 255, 255, 252, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 255, 255, 44, 0, 0, 0, 0, 0, 248, 255, 255, 247, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 255, 255, 76, 0, 0, 0, 0, 23, 255, 255, 255, 224, 0, 0, 0,
            ],
            &[
                0, 0, 140, 255, 255, 255, 160, 0, 0, 0, 0, 104, 255, 255, 255, 187, 0, 0, 0,
            ],
            &[
                0, 0, 59, 255, 255, 255, 254, 136, 45, 38, 103, 243, 255, 255, 255, 104, 0, 0, 0,
            ],
            &[
                0, 0, 1, 191, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 219, 11, 0, 0, 0,
            ],
            &[
                0, 0, 0, 29, 217, 255, 255, 255, 255, 255, 255, 255, 255, 230, 49, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 137, 228, 255, 255, 255, 255, 235, 148, 22, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 2, 30, 55, 57, 32, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 194, 255, 239, 231, 255, 221, 8, 0, 0, 0, 0, 0, 0,
            ],
        ]),
        // letter: 'ů' / 0x16f
        'ů' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 51, 123, 127, 60, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 101, 253, 255, 255, 255, 129, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 1, 241, 255, 136, 116, 252, 253, 29, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 24, 255, 230, 0, 0, 200, 255, 68, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 2, 243, 255, 115, 99, 249, 254, 31, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 110, 255, 255, 255, 255, 143, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 64, 140, 143, 76, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 123, 189, 189, 189, 50, 0, 0, 0, 0, 37, 189, 189, 189, 135, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 51, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 67, 0, 0, 0, 0, 55, 255, 255, 255, 182, 0, 0, 0,
            ],
            &[
                0, 0, 166, 255, 255, 255, 68, 0, 0, 0, 0, 69, 255, 255, 255, 182, 0, 