//! Module for letters with the font weight regular and size 20.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 20;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 11;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight regular and font size 17px
#[inline]
pub const fn get_char(c: char) -> Option<&'static [&'static [u8]]> {
    match c {
        // letter: ' ' / 0x20
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '!' / 0x21
        '!' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 35, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 204, 255, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 251, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 188, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 180, 235, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 172, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 211, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 148, 203, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 111, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 195, 238, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 197, 239, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '"' / 0x22
        '"' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 35, 11, 4, 35, 16, 0, 0, 0],
            &[0, 0, 58, 255, 73, 21, 255, 110, 0, 0, 0],
            &[0, 0, 39, 255, 53, 3, 254, 90, 0, 0, 0],
            &[0, 0, 19, 255, 33, 0, 238, 70, 0, 0, 0],
            &[0, 0, 2, 253, 13, 0, 218, 50, 0, 0, 0],
            &[0, 0, 0, 60, 0, 0, 51, 9, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '#' / 0x23
        '#' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 7, 0, 23, 17, 0, 0],
            &[0, 0, 0, 27, 255, 29, 0, 198, 101, 0, 0],
            &[0, 0, 0, 76, 236, 0, 2, 244, 53, 0, 0],
            &[0, 0, 0, 124, 188, 0, 39, 252, 8, 0, 0],
            &[4, 136, 136, 211, 208, 136, 171, 241, 136, 117, 0],
            &[4, 150, 150, 246, 179, 150, 216, 211, 150, 129, 0],
            &[0, 0, 15, 254, 36, 0, 196, 112, 0, 0, 0],
            &[0, 0, 65, 241, 1, 2, 243, 62, 0, 0, 0],
            &[116, 177, 207, 241, 177, 184, 255, 186, 177, 49, 0],
            &[75, 114, 211, 187, 114, 170, 229, 114, 114, 32, 0],
            &[0, 0, 212, 95, 0, 139, 169, 0, 0, 0, 0],
            &[0, 7, 251, 47, 0, 189, 119, 0, 0, 0, 0],
            &[0, 49, 249, 6, 0, 239, 68, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '$' / 0x24
        '$' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 122, 129, 0, 0, 0, 0, 0],
            &[0, 0, 21, 119, 210, 216, 142, 88, 10, 0, 0],
            &[0, 28, 231, 236, 221, 216, 199, 251, 46, 0, 0],
            &[0, 133, 247, 27, 135, 143, 0, 18, 0, 0, 0],
            &[0, 154, 227, 0, 135, 143, 0, 0, 0, 0, 0],
            &[0, 89, 255, 140, 157, 143, 0, 0, 0, 0, 0],
            &[0, 0, 123, 242, 255, 214, 91, 9, 0, 0, 0],
            &[0, 0, 0, 14, 174, 239, 254, 227, 44, 0, 0],
            &[0, 0, 0, 0, 135, 143, 38, 228, 177, 0, 0],
            &[0, 0, 0, 0, 135, 143, 0, 172, 203, 0, 0],
            &[0, 60, 63, 7, 135, 146, 70, 243, 141, 0, 0],
            &[0, 112, 255, 253, 246, 255, 255, 172, 11, 0, 0],
            &[0, 0, 39, 83, 183, 176, 31, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '%' / 0x25
        '%' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 57, 14, 0, 0, 0, 20, 26, 0, 0],
            &[76, 239, 200, 232, 46, 0, 2, 216, 116, 0, 0],
            &[210, 113, 0, 152, 172, 0, 90, 233, 10, 0, 0],
            &[246, 71, 0, 111, 211, 2, 215, 116, 0, 0, 0],
            &[221, 97, 0, 135, 187, 89, 234, 11, 0, 0, 0],
            &[106, 227, 142, 239, 80, 214, 117, 0, 0, 0, 0],
            &[0, 64, 112, 52, 89, 234, 11, 0, 0, 0, 0],
            &[0, 0, 0, 2, 214, 118, 58, 140, 115, 9, 0],
            &[0, 0, 0, 88, 234, 64, 245, 124, 186, 181, 0],
            &[0, 0, 2, 213, 119, 143, 175, 0, 41, 255, 23],
            &[0, 0, 87, 235, 11, 158, 159, 0, 25, 255, 40],
            &[0, 2, 213, 119, 0, 110, 211, 5, 81, 241, 8],
            &[0, 86, 235, 12, 0, 8, 187, 232, 241, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 23, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '&' / 0x26
        '&' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 51, 69, 18, 0, 0, 0, 0, 0],
            &[0, 0, 169, 255, 242, 238, 51, 0, 0, 0, 0],
            &[0, 50, 255, 87, 7, 209, 170, 0, 0, 0, 0],
            &[0, 64, 255, 34, 0, 168, 188, 0, 0, 0, 0],
            &[0, 17, 249, 97, 41, 244, 112, 0, 0, 0, 0],
            &[0, 0, 150, 255, 238, 180, 2, 0, 0, 0, 0],
            &[0, 0, 106, 255, 255, 8, 0, 0, 0, 0, 0],
            &[0, 96, 255, 188, 251, 118, 0, 55, 238, 41, 0],
            &[14, 243, 157, 0, 109, 252, 71, 134, 232, 1, 0],
            &[72, 255, 47, 0, 0, 166, 253, 255, 133, 0, 0],
            &[72, 255, 61, 0, 0, 13, 255, 255, 31, 0, 0],
            &[13, 240, 206, 60, 69, 181, 255, 255, 171, 0, 0],
            &[0, 70, 230, 255, 255, 205, 60, 84, 255, 121, 0],
            &[0, 0, 3, 33, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ''' / 0x27
        '\'' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 150, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 177, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '(' / 0x28
        '(' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 35, 9, 0, 0, 0],
            &[0, 0, 0, 0, 0, 137, 223, 10, 0, 0, 0],
            &[0, 0, 0, 0, 50, 253, 80, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 222, 1, 0, 0, 0, 0],
            &[0, 0, 0, 12, 249, 128, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 255, 65, 0, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 137, 249, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 148, 239, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 131, 253, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 103, 255, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 227, 165, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 121, 248, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 224, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 172, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ')' / 0x29
        ')' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 35, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 179, 189, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 250, 105, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 220, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 255, 63, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 253, 124, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 218, 171, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 196, 191, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 188, 198, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 207, 180, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 237, 149, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 255, 92, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 250, 23, 0, 0, 0],
            &[0, 0, 0, 0, 1, 218, 166, 0, 0, 0, 0],
            &[0, 0, 0, 0, 95, 245, 38, 0, 0, 0, 0],
            &[0, 0, 0, 2, 154, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '*' / 0x2a
        '*' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 124, 174, 0, 0, 0, 0, 0],
            &[0, 26, 56, 4, 104, 157, 2, 52, 53, 0, 0],
            &[0, 90, 251, 242, 215, 226, 236, 255, 158, 0, 0],
            &[0, 0, 3, 49, 240, 243, 103, 18, 0, 0, 0],
            &[0, 0, 2, 182, 173, 121, 226, 21, 0, 0, 0],
            &[0, 0, 62, 247, 43, 11, 232, 125, 0, 0, 0],
            &[0, 0, 0, 24, 0, 0, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '+' / 0x2b
        '+' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 119, 176, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 126, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 126, 186, 0, 0, 0, 0, 0],
            &[0, 97, 152, 152, 203, 227, 152, 152, 128, 0, 0],
            &[0, 100, 156, 156, 205, 228, 156, 156, 131, 0, 0],
            &[0, 0, 0, 0, 126, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 126, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 119, 175, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ',' / 0x2c
        ',' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 49, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 81, 0, 0, 0, 0],
            &[0, 0, 0, 0, 201, 244, 10, 0, 0, 0, 0],
            &[0, 0, 0, 5, 248, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 255, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 35, 90, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '-' / 0x2d
        '-' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 31, 69, 69, 69, 69, 44, 0, 0, 0],
            &[0, 0, 115, 255, 255, 255, 255, 166, 0, 0, 0],
            &[0, 0, 10, 23, 23, 23, 23, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '.' / 0x2e
        '.' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 195, 238, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 197, 239, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '/' / 0x2f
        '/' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 35, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 230, 3, 0, 0],
            &[0, 0, 0, 0, 0, 5, 233, 135, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 255, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 178, 195, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 251, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 119, 243, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 216, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 59, 255, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 157, 215, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 243, 118, 0, 0, 0, 0, 0, 0],
            &[0, 0, 98, 251, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 196, 177, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '0' / 0x30
        '0' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 57, 63, 18, 0, 0, 0, 0],
            &[0, 0, 79, 236, 255, 255, 246, 111, 0, 0, 0],
            &[0, 32, 247, 178, 36, 26, 189, 255, 69, 0, 0],
            &[0, 134, 241, 12, 0, 1, 197, 255, 181, 0, 0],
            &[0, 194, 172, 0, 0, 92, 252, 161, 241, 1, 0],
            &[0, 230, 133, 0, 13, 230, 150, 71, 255, 24, 0],
            &[0, 244, 119, 0, 139, 236, 18, 55, 255, 39, 0],
            &[0, 242, 125, 40, 250, 102, 0, 55, 255, 39, 0],
            &[0, 225, 144, 185, 204, 2, 0, 74, 255, 23, 0],
            &[0, 182, 255, 254, 56, 0, 0, 118, 240, 0, 0],
            &[0, 114, 255, 181, 0, 0, 2, 208, 172, 0, 0],
            &[0, 16, 235, 215, 69, 59, 168, 252, 55, 0, 0],
            &[0, 0, 47, 207, 255, 255, 231, 87, 0, 0, 0],
            &[0, 0, 0, 0, 21, 32, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '1' / 0x31
        '1' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 35, 9, 0, 0, 0, 0],
            &[0, 0, 2, 92, 218, 255, 69, 0, 0, 0, 0],
            &[0, 32, 215, 251, 178, 255, 69, 0, 0, 0, 0],
            &[0, 2, 134, 35, 40, 255, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 46, 255, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 69, 0, 0, 0, 0],
            &[0, 0, 8, 26, 91, 255, 114, 27, 10, 0, 0],
            &[0, 0, 189, 255, 255, 255, 255, 255, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '2' / 0x32
        '2' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 20, 61, 67, 19, 0, 0, 0, 0],
            &[0, 25, 170, 251, 255, 255, 251, 127, 0, 0, 0],
            &[0, 94, 230, 97, 23, 29, 157, 255, 72, 0, 0],
            &[0, 0, 9, 0, 0, 0, 7, 251, 143, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 244, 147, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 255, 93, 0, 0],
            &[0, 0, 0, 0, 0, 4, 208, 226, 8, 0, 0],
            &[0, 0, 0, 0, 1, 158, 249, 64, 0, 0, 0],
            &[0, 0, 0, 0, 143, 253, 88, 0, 0, 0, 0],
            &[0, 0, 0, 144, 253, 98, 0, 0, 0, 0, 0],
            &[0, 1, 148, 252, 92, 0, 0, 0, 0, 0, 0],
            &[0, 142, 255, 175, 92, 92, 92, 92, 92, 24, 0],
            &[0, 207, 255, 255, 255, 255, 255, 255, 255, 67, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '3' / 0x33
        '3' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 68, 67, 21, 0, 0, 0, 0],
            &[0, 51, 194, 255, 255, 255, 252, 134, 0, 0, 0],
            &[0, 71, 190, 73, 18, 31, 160, 255, 74, 0, 0],
            &[0, 0, 0, 0, 0, 0, 16, 255, 132, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 255, 110, 0, 0],
            &[0, 0, 0, 0, 0, 39, 194, 223, 18, 0, 0],
            &[0, 0, 0, 233, 249, 255, 181, 22, 0, 0, 0],
            &[0, 0, 0, 99, 107, 134, 220, 219, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 223, 182, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 168, 229, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 217, 203, 0, 0],
            &[0, 153, 128, 75, 51, 76, 183, 255, 93, 0, 0],
            &[0, 125, 238, 255, 255, 255, 219, 97, 0, 0, 0],
            &[0, 0, 1, 21, 37, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '4' / 0x34
        '4' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 52, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 148, 255, 148, 0, 0, 0],
            &[0, 0, 0, 0, 50, 247, 228, 148, 0, 0, 0],
            &[0, 0, 0, 2, 201, 153, 211, 148, 0, 0, 0],
            &[0, 0, 0, 103, 238, 21, 219, 148, 0, 0, 0],
            &[0, 0, 20, 237, 106, 0, 224, 148, 0, 0, 0],
            &[0, 0, 158, 205, 2, 0, 225, 148, 0, 0, 0],
            &[0, 58, 252, 56, 0, 0, 225, 148, 0, 0, 0],
            &[3, 209, 161, 7, 7, 7, 226, 152, 7, 2, 0],
            &[38, 255, 255, 255, 255, 255, 255, 255, 255, 89, 0],
            &[10, 71, 71, 71, 71, 71, 233, 178, 71, 25, 0],
            &[0, 0, 0, 0, 0, 0, 225, 148, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 225, 148, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '5' / 0x35
        '5' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 35, 35, 35, 35, 35, 4, 0, 0],
            &[0, 0, 249, 255, 255, 255, 255, 255, 32, 0, 0],
            &[0, 14, 255, 125, 56, 56, 56, 56, 7, 0, 0],
            &[0, 35, 255, 67, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 255, 44, 0, 0, 0, 0, 0, 0, 0],
            &[0, 76, 255, 104, 109, 97, 38, 0, 0, 0, 0],
            &[0, 65, 246, 221, 209, 242, 255, 152, 2, 0, 0],
            &[0, 0, 2, 0, 0, 6, 137, 255, 105, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 226, 187, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 201, 201, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 243, 158, 0, 0],
            &[0, 131, 118, 67, 54, 93, 211, 246, 45, 0, 0],
            &[0, 101, 230, 255, 255, 254, 198, 56, 0, 0, 0],
            &[0, 0, 0, 20, 36, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '6' / 0x36
        '6' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 50, 72, 64, 7, 0, 0],
            &[0, 0, 2, 121, 237, 255, 255, 255, 45, 0, 0],
            &[0, 0, 146, 251, 128, 33, 3, 14, 7, 0, 0],
            &[0, 48, 255, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 143, 232, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 201, 179, 32, 134, 167, 143, 48, 0, 0, 0],
            &[0, 233, 190, 230, 160, 141, 204, 251, 84, 0, 0],
            &[0, 247, 240, 38, 0, 0, 2, 183, 229, 1, 0],
            &[0, 239, 162, 0, 0, 0, 0, 93, 255, 33, 0],
            &[0, 207, 162, 0, 0, 0, 0, 86, 255, 35, 0],
            &[0, 139, 237, 17, 0, 0, 0, 144, 240, 6, 0],
            &[0, 27, 240, 200, 64, 40, 115, 250, 131, 0, 0],
            &[0, 0, 53, 207, 255, 255, 242, 134, 3, 0, 0],
            &[0, 0, 0, 0, 23, 34, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '7' / 0x37
        '7' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 34, 35, 35, 35, 35, 35, 35, 35, 6, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 55, 56, 56, 56, 56, 56, 163, 242, 13, 0],
            &[0, 0, 0, 0, 0, 0, 5, 230, 147, 0, 0],
            &[0, 0, 0, 0, 0, 0, 90, 254, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 201, 187, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 167, 224, 3, 0, 0, 0],
            &[0, 0, 0, 0, 27, 251, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 134, 247, 20, 0, 0, 0, 0],
            &[0, 0, 0, 8, 236, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 100, 255, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 210, 200, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '8' / 0x38
        '8' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 63, 69, 24, 0, 0, 0, 0],
            &[0, 0, 111, 245, 255, 254, 254, 151, 3, 0, 0],
            &[0, 53, 255, 144, 13, 7, 106, 255, 101, 0, 0],
            &[0, 107, 255, 21, 0, 0, 0, 230, 153, 0, 0],
            &[0, 73, 255, 76, 0, 0, 29, 253, 115, 0, 0],
            &[0, 0, 174, 240, 93, 64, 217, 201, 11, 0, 0],
            &[0, 0, 5, 186, 255, 255, 189, 8, 0, 0, 0],
            &[0, 1, 158, 240, 114, 141, 248, 177, 13, 0, 0],
            &[0, 118, 249, 55, 0, 0, 49, 235, 161, 0, 0],
            &[0, 202, 193, 0, 0, 0, 0, 143, 247, 1, 0],
            &[0, 197, 197, 0, 0, 0, 0, 154, 241, 0, 0],
            &[0, 106, 255, 129, 36, 33, 110, 251, 140, 0, 0],
            &[0, 0, 121, 237, 255, 255, 239, 135, 4, 0, 0],
            &[0, 0, 0, 3, 31, 32, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '9' / 0x39
        '9' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 65, 66, 16, 0, 0, 0, 0],
            &[0, 0, 134, 249, 254, 255, 248, 117, 0, 0, 0],
            &[0, 104, 255, 113, 12, 23, 139, 255, 82, 0, 0],
            &[0, 208, 181, 0, 0, 0, 0, 184, 197, 0, 0],
            &[0, 243, 133, 0, 0, 0, 0, 102, 251, 8, 0],
            &[0, 232, 150, 0, 0, 0, 0, 114, 255, 34, 0],
            &[0, 163, 235, 26, 0, 0, 35, 228, 255, 41, 0],
            &[0, 27, 220, 241, 180, 185, 237, 159, 255, 24, 0],
            &[0, 0, 10, 94, 133, 109, 27, 134, 248, 4, 0],
            &[0, 0, 0, 0, 0, 0, 0, 201, 190, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 255, 88, 0, 0],
            &[0, 0, 48, 36, 54, 134, 248, 174, 1, 0, 0],
            &[0, 0, 206, 255, 255, 231, 125, 4, 0, 0, 0],
            &[0, 0, 16, 39, 26, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ':' / 0x3a
        ':' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 205, 248, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 218, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 195, 236, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 197, 238, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ';' / 0x3b
        ';' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 50, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 110, 255, 105, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 239, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 49, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 81, 0, 0, 0, 0],
            &[0, 0, 0, 0, 201, 244, 10, 0, 0, 0, 0],
            &[0, 0, 0, 5, 248, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 255, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 35, 90, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '<' / 0x3c
        '<' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 25, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 174, 171, 0, 0],
            &[0, 0, 0, 0, 18, 143, 248, 161, 29, 0, 0],
            &[0, 0, 6, 113, 238, 186, 48, 0, 0, 0, 0],
            &[0, 56, 220, 209, 71, 0, 0, 0, 0, 0, 0],
            &[0, 75, 238, 201, 78, 1, 0, 0, 0, 0, 0],
            &[0, 0, 14, 122, 236, 209, 87, 2, 0, 0, 0],
            &[0, 0, 0, 0, 12, 119, 234, 217, 78, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 115, 147, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '=' / 0x3d
        '=' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 71, 71, 71, 71, 71, 71, 71, 0, 0],
            &[0, 207, 255, 255, 255, 255, 255, 255, 255, 2, 0],
            &[0, 16, 20, 20, 20, 20, 20, 20, 20, 0, 0],
            &[0, 10, 12, 12, 12, 12, 12, 12, 12, 0, 0],
            &[0, 207, 255, 255, 255, 255, 255, 255, 255, 2, 0],
            &[0, 64, 79, 79, 79, 79, 79, 79, 79, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '>' / 0x3e
        '>' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 24, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 201, 60, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 133, 245, 172, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 158, 247, 142, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 183, 237, 90, 0, 0],
            &[0, 0, 0, 0, 0, 53, 176, 248, 115, 0, 0],
            &[0, 0, 0, 63, 186, 247, 148, 28, 0, 0, 0],
            &[0, 49, 195, 246, 144, 26, 0, 0, 0, 0, 0],
            &[0, 109, 140, 23, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '?' / 0x3f
        '?' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 71, 67, 22, 0, 0, 0, 0],
            &[0, 57, 220, 255, 255, 255, 253, 149, 2, 0, 0],
            &[0, 19, 152, 65, 23, 19, 115, 255, 103, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 230, 139, 0, 0],
            &[0, 0, 0, 0, 0, 0, 115, 253, 52, 0, 0],
            &[0, 0, 0, 0, 2, 140, 252, 101, 0, 0, 0],
            &[0, 0, 0, 0, 139, 242, 73, 0, 0, 0, 0],
            &[0, 0, 0, 2, 248, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 144, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 58, 253, 128, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 254, 131, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '@' / 0x40
        '@' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 26, 2, 0, 0, 0, 0],
            &[0, 0, 40, 184, 253, 255, 245, 141, 6, 0, 0],
            &[0, 42, 240, 150, 34, 4, 54, 207, 171, 0, 0],
            &[0, 197, 147, 0, 0, 0, 0, 21, 237, 68, 0],
            &[51, 245, 18, 25, 161, 207, 184, 77, 139, 157, 0],
            &[122, 183, 3, 207, 177, 68, 219, 135, 79, 211, 0],
            &[165, 135, 65, 255, 29, 0, 210, 121, 55, 232, 0],
            &[183, 117, 109, 234, 0, 0, 226, 108, 56, 231, 0],
            &[180, 119, 104, 229, 0, 6, 250, 100, 74, 206, 0],
            &[158, 144, 55, 254, 42, 82, 237, 138, 140, 145, 0],
            &[104, 205, 0, 166, 250, 228, 45, 237, 238, 29, 0],
            &[27, 248, 61, 0, 25, 8, 0, 16, 14, 0, 0],
            &[0, 133, 228, 48, 0, 0, 0, 5, 21, 0, 0],
            &[0, 1, 142, 250, 199, 168, 188, 237, 101, 0, 0],
            &[0, 0, 0, 32, 98, 123, 101, 46, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'A' / 0x41
        'A' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 236, 255, 32, 0, 0, 0, 0],
            &[0, 0, 0, 67, 254, 236, 114, 0, 0, 0, 0],
            &[0, 0, 0, 149, 210, 162, 197, 0, 0, 0, 0],
            &[0, 0, 2, 230, 134, 87, 254, 26, 0, 0, 0],
            &[0, 0, 59, 255, 55, 14, 249, 108, 0, 0, 0],
            &[0, 0, 142, 230, 1, 0, 184, 191, 0, 0, 0],
            &[0, 0, 224, 171, 21, 21, 126, 252, 21, 0, 0],
            &[0, 52, 255, 255, 255, 255, 255, 255, 101, 0, 0],
            &[0, 134, 255, 77, 70, 70, 70, 229, 184, 0, 0],
            &[0, 217, 169, 0, 0, 0, 0, 122, 250, 17, 0],
            &[44, 255, 90, 0, 0, 0, 0, 43, 255, 95, 0],
            &[127, 250, 16, 0, 0, 0, 0, 0, 219, 178, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'B' / 0x42
        'B' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 35, 35, 35, 19, 0, 0, 0, 0, 0],
            &[0, 98, 255, 255, 255, 255, 241, 151, 16, 0, 0],
            &[0, 98, 255, 60, 48, 64, 132, 254, 163, 0, 0],
            &[0, 98, 255, 15, 0, 0, 0, 185, 227, 0, 0],
            &[0, 98, 255, 15, 0, 0, 0, 178, 213, 0, 0],
            &[0, 98, 255, 16, 1, 14, 86, 249, 109, 0, 0],
            &[0, 98, 255, 255, 255, 255, 245, 101, 0, 0, 0],
            &[0, 98, 255, 92, 82, 91, 147, 249, 132, 0, 0],
            &[0, 98, 255, 15, 0, 0, 0, 138, 254, 30, 0],
            &[0, 98, 255, 15, 0, 0, 0, 88, 255, 60, 0],
            &[0, 98, 255, 15, 0, 0, 0, 141, 255, 29, 0],
            &[0, 98, 255, 93, 83, 95, 151, 251, 170, 0, 0],
            &[0, 98, 255, 255, 255, 251, 222, 133, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'C' / 0x43
        'C' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 60, 70, 43, 3, 0, 0],
            &[0, 0, 2, 132, 246, 255, 255, 255, 229, 37, 0],
            &[0, 0, 150, 254, 138, 39, 22, 56, 123, 1, 0],
            &[0, 51, 255, 134, 0, 0, 0, 0, 0, 0, 0],
            &[0, 143, 250, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 196, 206, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 217, 181, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 213, 185, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 191, 211, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 253, 22, 0, 0, 0, 0, 0, 0, 0],
            &[0, 45, 253, 152, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 141, 255, 175, 86, 70, 101, 135, 0, 0],
            &[0, 0, 0, 113, 230, 255, 255, 252, 167, 0, 0],
            &[0, 0, 0, 0, 0, 21, 35, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'D' / 0x44
        'D' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 35, 35, 24, 1, 0, 0, 0, 0, 0],
            &[0, 207, 255, 255, 255, 245, 160, 26, 0, 0, 0],
            &[0, 207, 197, 48, 63, 120, 235, 231, 32, 0, 0],
            &[0, 207, 184, 0, 0, 0, 35, 241, 183, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 254, 27, 0],
            &[0, 207, 184, 0, 0, 0, 0, 65, 255, 80, 0],
            &[0, 207, 184, 0, 0, 0, 0, 40, 255, 103, 0],
            &[0, 207, 184, 0, 0, 0, 0, 42, 255, 95, 0],
            &[0, 207, 184, 0, 0, 0, 0, 76, 255, 69, 0],
            &[0, 207, 184, 0, 0, 0, 0, 156, 245, 10, 0],
            &[0, 207, 184, 0, 0, 0, 69, 252, 146, 0, 0],
            &[0, 207, 207, 84, 102, 167, 252, 195, 10, 0, 0],
            &[0, 207, 255, 255, 236, 199, 108, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'E' / 0x45
        'E' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 35, 35, 35, 35, 35, 35, 26, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 98, 255, 85, 56, 56, 56, 56, 42, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 41, 5, 5, 5, 5, 2, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 110, 0, 0],
            &[0, 98, 255, 111, 86, 86, 86, 86, 37, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 115, 92, 92, 92, 92, 69, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'F' / 0x46
        'F' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 35, 35, 35, 35, 35, 35, 26, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 98, 255, 85, 56, 56, 56, 56, 42, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 145, 126, 126, 126, 126, 54, 0, 0],
            &[0, 98, 255, 226, 221, 221, 221, 221, 95, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'G' / 0x47
        'G' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 63, 63, 26, 0, 0, 0],
            &[0, 0, 27, 182, 254, 255, 255, 254, 152, 0, 0],
            &[0, 15, 224, 237, 103, 34, 33, 84, 76, 0, 0],
            &[0, 137, 254, 56, 0, 0, 0, 0, 0, 0, 0],
            &[0, 225, 183, 0, 0, 0, 0, 0, 0, 0, 0],
            &[26, 255, 119, 0, 0, 0, 0, 0, 0, 0, 0],
            &[47, 255, 93, 0, 0, 71, 104, 104, 104, 1, 0],
            &[52, 255, 91, 0, 0, 165, 243, 249, 255, 2, 0],
            &[32, 255, 116, 0, 0, 0, 0, 120, 255, 2, 0],
            &[2, 237, 177, 0, 0, 0, 0, 120, 255, 2, 0],
            &[0, 155, 252, 51, 0, 0, 0, 120, 255, 2, 0],
            &[0, 25, 232, 236, 116, 61, 64, 172, 255, 2, 0],
            &[0, 0, 32, 174, 252, 255, 255, 245, 178, 1, 0],
            &[0, 0, 0, 0, 7, 34, 24, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'H' / 0x48
        'H' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 18, 35, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 185, 5, 5, 5, 5, 135, 255, 2, 0],
            &[0, 207, 255, 255, 255, 255, 255, 255, 255, 2, 0],
            &[0, 207, 208, 86, 86, 86, 86, 174, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'I' / 0x49
        'I' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 35, 35, 35, 35, 35, 35, 11, 0, 0],
            &[0, 30, 248, 255, 255, 255, 255, 251, 78, 0, 0],
            &[0, 0, 0, 16, 184, 222, 20, 1, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 1, 28, 51, 195, 227, 55, 32, 5, 0, 0],
            &[0, 33, 255, 255, 255, 255, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'J' / 0x4a
        'J' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 22, 31, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 231, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 231, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 231, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 231, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 231, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 231, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 231, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 231, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 167, 226, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 220, 195, 0, 0],
            &[0, 0, 55, 132, 83, 82, 171, 255, 103, 0, 0],
            &[0, 0, 55, 232, 255, 255, 245, 136, 1, 0, 0],
            &[0, 0, 0, 1, 23, 35, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'K' / 0x4b
        'K' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 21, 35, 5, 0],
            &[0, 207, 184, 0, 0, 0, 27, 235, 183, 1, 0],
            &[0, 207, 184, 0, 0, 3, 195, 230, 21, 0, 0],
            &[0, 207, 184, 0, 0, 134, 253, 65, 0, 0, 0],
            &[0, 207, 184, 0, 71, 254, 129, 0, 0, 0, 0],
            &[0, 207, 184, 24, 234, 192, 3, 0, 0, 0, 0],
            &[0, 207, 184, 180, 255, 80, 0, 0, 0, 0, 0],
            &[0, 207, 243, 228, 212, 214, 6, 0, 0, 0, 0],
            &[0, 207, 242, 47, 55, 253, 126, 0, 0, 0, 0],
            &[0, 207, 184, 0, 0, 149, 248, 40, 0, 0, 0],
            &[0, 207, 184, 0, 0, 15, 230, 194, 1, 0, 0],
            &[0, 207, 184, 0, 0, 0, 86, 255, 102, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 182, 239, 24, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'L' / 0x4c
        'L' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 35, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 178, 105, 105, 105, 105, 79, 0, 0],
            &[0, 11, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'M' / 0x4d
        'M' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 35, 35, 2, 0, 0, 0, 31, 35, 12, 0],
            &[38, 255, 255, 57, 0, 0, 15, 251, 255, 89, 0],
            &[38, 255, 236, 130, 0, 0, 83, 247, 244, 89, 0],
            &[38, 255, 183, 202, 0, 0, 155, 184, 252, 89, 0],
            &[38, 255, 124, 253, 21, 0, 228, 119, 255, 89, 0],
            &[38, 255, 75, 238, 92, 45, 255, 51, 255, 89, 0],
            &[38, 255, 75, 165, 164, 118, 211, 24, 255, 89, 0],
            &[38, 255, 75, 89, 235, 193, 135, 24, 255, 89, 0],
            &[38, 255, 75, 17, 251, 255, 59, 24, 255, 89, 0],
            &[38, 255, 75, 0, 189, 231, 3, 24, 255, 89, 0],
            &[38, 255, 75, 0, 0, 0, 0, 24, 255, 89, 0],
            &[38, 255, 75, 0, 0, 0, 0, 24, 255, 89, 0],
            &[38, 255, 75, 0, 0, 0, 0, 24, 255, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'N' / 0x4e
        'N' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 35, 8, 0, 0, 0, 16, 35, 0, 0],
            &[0, 207, 255, 125, 0, 0, 0, 120, 255, 2, 0],
            &[0, 207, 230, 231, 6, 0, 0, 120, 255, 2, 0],
            &[0, 207, 152, 243, 93, 0, 0, 120, 255, 2, 0],
            &[0, 207, 155, 146, 205, 0, 0, 120, 255, 2, 0],
            &[0, 207, 167, 36, 253, 61, 0, 120, 255, 2, 0],
            &[0, 207, 171, 0, 178, 173, 0, 120, 255, 2, 0],
            &[0, 207, 171, 0, 65, 252, 32, 119, 255, 2, 0],
            &[0, 207, 171, 0, 0, 208, 140, 115, 255, 2, 0],
            &[0, 207, 171, 0, 0, 97, 240, 119, 255, 2, 0],
            &[0, 207, 171, 0, 0, 7, 233, 199, 255, 2, 0],
            &[0, 207, 171, 0, 0, 0, 128, 255, 255, 2, 0],
            &[0, 207, 171, 0, 0, 0, 23, 248, 255, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'O' / 0x4f
        'O' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 65, 76, 28, 0, 0, 0, 0],
            &[0, 0, 96, 241, 255, 255, 253, 144, 1, 0, 0],
            &[0, 59, 253, 169, 33, 17, 115, 255, 112, 0, 0],
            &[0, 179, 226, 6, 0, 0, 0, 179, 231, 2, 0],
            &[4, 244, 146, 0, 0, 0, 0, 99, 255, 50, 0],
            &[36, 255, 105, 0, 0, 0, 0, 59, 255, 89, 0],
            &[52, 255, 89, 0, 0, 0, 0, 41, 255, 106, 0],
            &[53, 255, 91, 0, 0, 0, 0, 39, 255, 104, 0],
            &[34, 255, 110, 0, 0, 0, 0, 58, 255, 83, 0],
            &[3, 243, 157, 0, 0, 0, 0, 104, 255, 39, 0],
            &[0, 169, 237, 17, 0, 0, 1, 198, 216, 0, 0],
            &[0, 44, 246, 200, 76, 64, 165, 255, 82, 0, 0],
            &[0, 0, 66, 218, 255, 255, 232, 97, 0, 0, 0],
            &[0, 0, 0, 0, 27, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'P' / 0x50
        'P' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 35, 35, 35, 29, 6, 0, 0, 0, 0],
            &[0, 98, 255, 255, 255, 255, 249, 173, 25, 0, 0],
            &[0, 98, 255, 78, 48, 63, 121, 246, 200, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 129, 255, 40, 0],
            &[0, 98, 255, 37, 0, 0, 0, 84, 255, 61, 0],
            &[0, 98, 255, 37, 0, 0, 0, 121, 255, 28, 0],
            &[0, 98, 255, 37, 1, 18, 90, 241, 181, 0, 0],
            &[0, 98, 255, 255, 255, 255, 254, 180, 20, 0, 0],
            &[0, 98, 255, 108, 82, 67, 29, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Q' / 0x51
        'Q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 65, 76, 28, 0, 0, 0, 0],
            &[0, 0, 96, 241, 255, 255, 253, 144, 1, 0, 0],
            &[0, 59, 253, 169, 33, 17, 115, 255, 112, 0, 0],
            &[0, 179, 226, 6, 0, 0, 0, 179, 231, 2, 0],
            &[4, 244, 146, 0, 0, 0, 0, 99, 255, 50, 0],
            &[36, 255, 105, 0, 0, 0, 0, 59, 255, 89, 0],
            &[52, 255, 89, 0, 0, 0, 0, 41, 255, 106, 0],
            &[53, 255, 91, 0, 0, 0, 0, 39, 255, 104, 0],
            &[34, 255, 110, 0, 0, 0, 0, 58, 255, 83, 0],
            &[3, 243, 157, 0, 0, 0, 0, 104, 255, 39, 0],
            &[0, 169, 237, 17, 0, 0, 1, 198, 216, 0, 0],
            &[0, 44, 246, 200, 76, 64, 226, 255, 82, 0, 0],
            &[0, 0, 66, 218, 255, 255, 255, 179, 0, 0, 0],
            &[0, 0, 0, 0, 27, 32, 205, 213, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 73, 255, 95, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 200, 224, 6, 0],
            &[0, 0, 0, 0, 0, 0, 0, 7, 15, 2, 0],
        ]),
        // letter: 'R' / 0x52
        'R' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 35, 35, 32, 14, 0, 0, 0, 0, 0],
            &[0, 98, 255, 255, 255, 255, 214, 87, 0, 0, 0],
            &[0, 98, 255, 78, 53, 82, 195, 254, 68, 0, 0],
            &[0, 98, 255, 37, 0, 0, 15, 251, 160, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 233, 177, 0, 0],
            &[0, 98, 255, 37, 0, 0, 32, 255, 134, 0, 0],
            &[0, 98, 255, 100, 77, 107, 217, 226, 23, 0, 0],
            &[0, 98, 255, 255, 255, 255, 185, 16, 0, 0, 0],
            &[0, 98, 255, 44, 9, 173, 232, 15, 0, 0, 0],
            &[0, 98, 255, 37, 0, 31, 246, 146, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 127, 252, 47, 0, 0],
            &[0, 98, 255, 37, 0, 0, 9, 223, 196, 1, 0],
            &[0, 98, 255, 37, 0, 0, 0, 82, 255, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'S' / 0x53
        'S' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 49, 72, 52, 15, 0, 0, 0],
            &[0, 0, 60, 224, 255, 255, 255, 250, 126, 0, 0],
            &[0, 10, 235, 220, 66, 22, 40, 100, 61, 0, 0],
            &[0, 59, 255, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 54, 255, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 219, 240, 90, 2, 0, 0, 0, 0, 0],
            &[0, 0, 34, 197, 255, 218, 98, 5, 0, 0, 0],
            &[0, 0, 0, 0, 74, 196, 255, 215, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 242, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 166, 245, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 194, 231, 0, 0],
            &[0, 110, 134, 69, 41, 52, 137, 255, 138, 0, 0],
            &[0, 89, 221, 255, 255, 255, 240, 136, 4, 0, 0],
            &[0, 0, 0, 12, 35, 31, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'T' / 0x54
        'T' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[8, 35, 35, 35, 35, 35, 35, 35, 35, 15, 0],
            &[60, 255, 255, 255, 255, 255, 255, 255, 255, 111, 0],
            &[13, 56, 56, 56, 189, 229, 56, 56, 56, 24, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'U' / 0x55
        'U' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 18, 35, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 206, 186, 0, 0, 0, 0, 135, 255, 1, 0],
            &[0, 191, 213, 0, 0, 0, 0, 159, 242, 0, 0],
            &[0, 146, 248, 18, 0, 0, 0, 211, 195, 0, 0],
            &[0, 47, 253, 193, 71, 61, 159, 255, 89, 0, 0],
            &[0, 0, 85, 227, 255, 255, 238, 114, 0, 0, 0],
            &[0, 0, 0, 2, 27, 30, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'V' / 0x56
        'V' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[19, 35, 1, 0, 0, 0, 0, 0, 30, 26, 0],
            &[94, 255, 54, 0, 0, 0, 0, 12, 249, 146, 0],
            &[17, 250, 131, 0, 0, 0, 0, 80, 255, 65, 0],
            &[0, 185, 208, 0, 0, 0, 0, 155, 236, 3, 0],
            &[0, 103, 255, 29, 0, 0, 0, 228, 159, 0, 0],
            &[0, 23, 253, 105, 0, 0, 48, 255, 78, 0, 0],
            &[0, 0, 194, 182, 0, 0, 122, 244, 8, 0, 0],
            &[0, 0, 111, 247, 11, 0, 196, 172, 0, 0, 0],
            &[0, 0, 30, 255, 79, 18, 252, 91, 0, 0, 0],
            &[0, 0, 0, 202, 153, 86, 250, 16, 0, 0, 0],
            &[0, 0, 0, 120, 222, 152, 185, 0, 0, 0, 0],
            &[0, 0, 0, 37, 255, 232, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 211, 254, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'W' / 0x57
        'W' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[20, 29, 0, 0, 0, 0, 0, 0, 22, 27, 0],
            &[128, 229, 0, 0, 0, 0, 0, 0, 177, 181, 0],
            &[100, 251, 2, 0, 0, 0, 0, 0, 198, 156, 0],
            &[71, 255, 22, 0, 17, 24, 0, 0, 220, 130, 0],
            &[42, 255, 46, 0, 193, 247, 6, 0, 242, 104, 0],
            &[14, 255, 71, 6, 247, 231, 59, 9, 255, 78, 0],
            &[0, 241, 95, 56, 231, 163, 121, 30, 255, 52, 0],
            &[0, 212, 117, 116, 176, 107, 182, 50, 255, 27, 0],
            &[0, 184, 137, 174, 117, 46, 240, 71, 253, 3, 0],
            &[0, 155, 154, 230, 58, 2, 238, 130, 231, 0, 0],
            &[0, 126, 196, 248, 6, 0, 179, 196, 205, 0, 0],
            &[0, 98, 249, 195, 0, 0, 117, 249, 179, 0, 0],
            &[0, 69, 255, 136, 0, 0, 55, 255, 153, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'X' / 0x58
        'X' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[3, 35, 18, 0, 0, 0, 0, 10, 35, 10, 0],
            &[0, 194, 214, 4, 0, 0, 0, 160, 232, 15, 0],
            &[0, 45, 251, 106, 0, 0, 51, 254, 94, 0, 0],
            &[0, 0, 143, 234, 15, 0, 194, 196, 1, 0, 0],
            &[0, 0, 14, 230, 137, 84, 252, 47, 0, 0, 0],
            &[0, 0, 0, 90, 248, 226, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 203, 243, 16, 0, 0, 0, 0],
            &[0, 0, 0, 31, 245, 255, 73, 0, 0, 0, 0],
            &[0, 0, 0, 175, 205, 149, 220, 7, 0, 0, 0],
            &[0, 0, 74, 255, 61, 18, 237, 126, 0, 0, 0],
            &[0, 8, 220, 169, 0, 0, 107, 247, 33, 0, 0],
            &[0, 127, 247, 31, 0, 0, 3, 211, 179, 0, 0],
            &[34, 247, 131, 0, 0, 0, 0, 67, 255, 77, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Y' / 0x59
        'Y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[13, 35, 10, 0, 0, 0, 0, 2, 35, 20, 0],
            &[28, 248, 146, 0, 0, 0, 0, 82, 255, 71, 0],
            &[0, 146, 245, 19, 0, 0, 0, 201, 197, 0, 0],
            &[0, 25, 246, 127, 0, 0, 65, 255, 66, 0, 0],
            &[0, 0, 141, 235, 10, 0, 184, 192, 0, 0, 0],
            &[0, 0, 22, 244, 108, 48, 255, 62, 0, 0, 0],
            &[0, 0, 0, 136, 222, 170, 187, 0, 0, 0, 0],
            &[0, 0, 0, 19, 242, 255, 57, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Z' / 0x5a
        'Z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 35, 35, 35, 35, 35, 35, 20, 0, 0],
            &[0, 11, 255, 255, 255, 255, 255, 255, 149, 0, 0],
            &[0, 2, 48, 48, 48, 48, 94, 255, 88, 0, 0],
            &[0, 0, 0, 0, 0, 0, 187, 200, 1, 0, 0],
            &[0, 0, 0, 0, 0, 75, 255, 60, 0, 0, 0],
            &[0, 0, 0, 0, 4, 214, 173, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 250, 36, 0, 0, 0, 0],
            &[0, 0, 0, 15, 234, 144, 0, 0, 0, 0, 0],
            &[0, 0, 0, 137, 239, 19, 0, 0, 0, 0, 0],
            &[0, 0, 32, 248, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 168, 222, 7, 0, 0, 0, 0, 0, 0],
            &[0, 54, 255, 164, 92, 92, 92, 92, 53, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 149, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '[' / 0x5b
        '[' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 35, 35, 31, 0, 0, 0, 0],
            &[0, 0, 0, 235, 255, 255, 231, 0, 0, 0, 0],
            &[0, 0, 0, 235, 148, 22, 19, 0, 0, 0, 0],
            &[0, 0, 0, 235, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 201, 137, 124, 0, 0, 0, 0],
            &[0, 0, 0, 161, 175, 175, 158, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '\' / 0x5c
        '\\' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 183, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 84, 254, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 236, 131, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 144, 227, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 203, 170, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 105, 249, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 247, 110, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 165, 209, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 66, 255, 51, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 222, 149, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 239, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ']' / 0x5d
        ']' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 55, 78, 78, 78, 9, 0, 0, 0],
            &[0, 0, 0, 165, 234, 241, 255, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 30, 0, 0, 0],
            &[0, 0, 0, 127, 181, 206, 255, 30, 0, 0, 0],
            &[0, 0, 0, 93, 132, 132, 132, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '^' / 0x5e
        '^' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 216, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 83, 221, 239, 60, 0, 0, 0, 0],
            &[0, 0, 0, 204, 106, 125, 194, 0, 0, 0, 0],
            &[0, 0, 70, 236, 10, 14, 237, 72, 0, 0, 0],
            &[0, 0, 191, 131, 0, 0, 122, 204, 1, 0, 0],
            &[0, 56, 248, 23, 0, 0, 12, 235, 83, 0, 0],
            &[0, 177, 156, 0, 0, 0, 0, 118, 214, 2, 0],
            &[6, 116, 33, 0, 0, 0, 0, 11, 117, 27, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '_' / 0x5f
        '_' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 37],
            &[158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 31],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '`' / 0x60
        '`' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 130, 255, 74, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 163, 221, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 134, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'a' / 0x61
        'a' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 62, 58, 22, 0, 0, 0, 0],
            &[0, 8, 189, 254, 254, 253, 253, 154, 1, 0, 0],
            &[0, 0, 114, 55, 8, 6, 98, 254, 97, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 207, 154, 0, 0],
            &[0, 0, 2, 59, 106, 126, 133, 223, 171, 0, 0],
            &[0, 27, 208, 247, 185, 156, 145, 224, 171, 0, 0],
            &[0, 157, 236, 38, 0, 0, 0, 191, 171, 0, 0],
            &[0, 197, 177, 0, 0, 0, 14, 239, 171, 0, 0],
            &[0, 154, 238, 69, 24, 63, 198, 241, 171, 0, 0],
            &[0, 22, 194, 255, 255, 234, 99, 159, 171, 0, 0],
            &[0, 0, 0, 19, 31, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'b' / 0x62
        'b' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 150, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 214, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 214, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 213, 0, 28, 65, 39, 0, 0, 0, 0],
            &[0, 163, 207, 147, 254, 253, 255, 185, 14, 0, 0],
            &[0, 163, 253, 204, 37, 7, 88, 248, 156, 0, 0],
            &[0, 163, 255, 48, 0, 0, 0, 159, 248, 10, 0],
            &[0, 163, 239, 0, 0, 0, 0, 99, 255, 51, 0],
            &[0, 163, 221, 0, 0, 0, 0, 82, 255, 63, 0],
            &[0, 163, 243, 0, 0, 0, 0, 103, 255, 47, 0],
            &[0, 163, 255, 59, 0, 0, 0, 171, 242, 6, 0],
            &[0, 163, 248, 217, 64, 33, 116, 253, 137, 0, 0],
            &[0, 163, 164, 120, 243, 255, 251, 154, 5, 0, 0],
            &[0, 0, 0, 0, 9, 36, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'c' / 0x63
        'c' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 62, 55, 17, 0, 0, 0],
            &[0, 0, 27, 184, 254, 255, 255, 250, 94, 0, 0],
            &[0, 2, 205, 238, 91, 23, 26, 87, 34, 0, 0],
            &[0, 66, 255, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 118, 255, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 136, 255, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 119, 255, 26, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 255, 104, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 201, 243, 110, 46, 52, 94, 74, 0, 0],
            &[0, 0, 22, 169, 249, 255, 255, 239, 90, 0, 0],
            &[0, 0, 0, 0, 10, 35, 25, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'd' / 0x64
        'd' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 158, 197, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 172, 214, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 172, 214, 0],
            &[0, 0, 0, 0, 22, 61, 45, 1, 171, 214, 0],
            &[0, 0, 3, 146, 251, 254, 254, 198, 173, 214, 0],
            &[0, 0, 109, 255, 143, 17, 12, 154, 254, 214, 0],
            &[0, 0, 209, 219, 2, 0, 0, 8, 240, 214, 0],
            &[0, 1, 252, 155, 0, 0, 0, 0, 191, 214, 0],
            &[0, 12, 255, 134, 0, 0, 0, 0, 174, 214, 0],
            &[0, 1, 249, 152, 0, 0, 0, 0, 192, 214, 0],
            &[0, 0, 199, 217, 1, 0, 0, 14, 241, 214, 0],
            &[0, 0, 90, 255, 152, 40, 46, 179, 245, 214, 0],
            &[0, 0, 0, 120, 244, 255, 254, 154, 128, 214, 0],
            &[0, 0, 0, 0, 7, 36, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'e' / 0x65
        'e' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 46, 63, 27, 0, 0, 0, 0],
            &[0, 0, 67, 223, 255, 252, 253, 161, 8, 0, 0],
            &[0, 44, 248, 178, 30, 1, 77, 248, 139, 0, 0],
            &[0, 163, 231, 10, 0, 0, 0, 148, 240, 5, 0],
            &[0, 225, 213, 125, 125, 125, 125, 178, 255, 33, 0],
            &[0, 244, 227, 187, 187, 187, 187, 187, 187, 33, 0],
            &[0, 223, 170, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 155, 239, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 239, 204, 74, 30, 46, 95, 136, 0, 0],
            &[0, 0, 39, 183, 252, 255, 255, 233, 127, 0, 0],
            &[0, 0, 0, 0, 11, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'f' / 0x66
        'f' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 161, 235, 252, 233, 115, 0],
            &[0, 0, 0, 0, 161, 252, 119, 70, 88, 51, 0],
            &[0, 0, 0, 0, 221, 182, 0, 0, 0, 0, 0],
            &[0, 0, 1, 17, 233, 171, 28, 28, 28, 1, 0],
            &[0, 207, 252, 255, 255, 255, 255, 255, 255, 15, 0],
            &[0, 36, 41, 41, 235, 176, 41, 41, 41, 2, 0],
            &[0, 0, 0, 0, 231, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 231, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 231, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 231, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 231, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 231, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 231, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'g' / 0x67
        'g' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 65, 45, 0, 11, 24, 0],
            &[0, 0, 0, 137, 253, 254, 254, 196, 141, 214, 0],
            &[0, 0, 96, 255, 133, 15, 11, 135, 255, 214, 0],
            &[0, 0, 202, 213, 0, 0, 0, 1, 216, 214, 0],
            &[0, 2, 251, 152, 0, 0, 0, 0, 156, 214, 0],
            &[0, 12, 255, 134, 0, 0, 0, 0, 139, 214, 0],
            &[0, 2, 250, 155, 0, 0, 0, 0, 158, 214, 0],
            &[0, 0, 199, 224, 3, 0, 0, 2, 216, 214, 0],
            &[0, 0, 85, 255, 164, 45, 39, 153, 254, 214, 0],
            &[0, 0, 0, 111, 241, 255, 255, 192, 163, 214, 0],
            &[0, 0, 0, 0, 6, 36, 23, 0, 160, 208, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 213, 174, 0],
            &[0, 0, 106, 101, 49, 23, 45, 155, 255, 76, 0],
            &[0, 0, 0, 0, 11, 34, 24, 1, 0, 0, 0],
        ]),
        // letter: 'h' / 0x68
        'h' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 150, 209, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 227, 0, 15, 59, 42, 2, 0, 0, 0],
            &[0, 163, 221, 95, 243, 253, 255, 205, 26, 0, 0],
            &[0, 163, 242, 206, 40, 3, 79, 252, 160, 0, 0],
            &[0, 163, 255, 57, 0, 0, 0, 185, 216, 0, 0],
            &[0, 163, 244, 1, 0, 0, 0, 156, 236, 0, 0],
            &[0, 163, 228, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'i' / 0x69
        'i' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 124, 205, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 181, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 31, 0, 0, 0, 0, 0],
            &[0, 10, 28, 28, 28, 28, 3, 0, 0, 0, 0],
            &[0, 89, 254, 255, 255, 255, 35, 0, 0, 0, 0],
            &[0, 0, 3, 20, 122, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 7, 28, 48, 139, 255, 87, 41, 19, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'j' / 0x6a
        'j' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 203, 131, 0, 0, 0],
            &[0, 0, 0, 0, 0, 27, 253, 189, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 10, 0, 0, 0],
            &[0, 0, 0, 21, 28, 28, 28, 21, 0, 0, 0],
            &[0, 0, 0, 187, 255, 255, 255, 191, 0, 0, 0],
            &[0, 0, 0, 0, 9, 28, 208, 191, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 199, 191, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 199, 191, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 199, 191, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 199, 191, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 199, 191, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 199, 191, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 199, 191, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 202, 191, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 237, 166, 0, 0, 0],
            &[0, 0, 84, 41, 49, 151, 255, 91, 0, 0, 0],
            &[0, 0, 2, 27, 31, 3, 0, 0, 0, 0, 0],
        ]),
        // letter: 'k' / 0x6b
        'k' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 235, 74, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 255, 80, 0, 0, 0, 12, 28, 11, 0],
            &[0, 55, 255, 80, 0, 0, 21, 218, 210, 16, 0],
            &[0, 55, 255, 80, 0, 11, 202, 221, 25, 0, 0],
            &[0, 55, 255, 79, 5, 184, 231, 35, 0, 0, 0],
            &[0, 55, 255, 72, 161, 247, 47, 0, 0, 0, 0],
            &[0, 55, 255, 177, 237, 255, 87, 0, 0, 0, 0],
            &[0, 55, 255, 222, 33, 158, 241, 35, 0, 0, 0],
            &[0, 55, 255, 84, 0, 11, 217, 205, 6, 0, 0],
            &[0, 55, 255, 80, 0, 0, 49, 249, 146, 0, 0],
            &[0, 55, 255, 80, 0, 0, 0, 112, 255, 81, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'l' / 0x6c
        'l' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 126, 235, 235, 235, 235, 52, 0, 0, 0, 0],
            &[0, 20, 52, 69, 137, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 7, 27, 47, 122, 255, 104, 42, 20, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'm' / 0x6d
        'm' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[4, 26, 0, 52, 39, 0, 18, 63, 14, 0, 0],
            &[38, 252, 145, 255, 254, 86, 231, 255, 219, 5, 0],
            &[38, 255, 196, 42, 222, 255, 124, 87, 255, 61, 0],
            &[38, 255, 97, 0, 158, 247, 9, 8, 255, 87, 0],
            &[38, 255, 68, 0, 144, 223, 0, 0, 254, 93, 0],
            &[38, 255, 55, 0, 144, 205, 0, 0, 254, 93, 0],
            &[38, 255, 53, 0, 144, 203, 0, 0, 254, 93, 0],
            &[38, 255, 53, 0, 144, 203, 0, 0, 254, 93, 0],
            &[38, 255, 53, 0, 144, 203, 0, 0, 254, 93, 0],
            &[38, 255, 53, 0, 144, 203, 0, 0, 254, 93, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'n' / 0x6e
        'n' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 17, 0, 18, 59, 46, 4, 0, 0, 0],
            &[0, 163, 175, 116, 247, 253, 255, 213, 32, 0, 0],
            &[0, 163, 245, 199, 38, 2, 61, 247, 168, 0, 0],
            &[0, 163, 255, 50, 0, 0, 0, 177, 220, 0, 0],
            &[0, 163, 248, 0, 0, 0, 0, 155, 236, 0, 0],
            &[0, 163, 229, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'o' / 0x6f
        'o' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 55, 59, 15, 0, 0, 0, 0],
            &[0, 0, 106, 240, 255, 253, 247, 136, 3, 0, 0],
            &[0, 81, 255, 142, 19, 6, 102, 252, 125, 0, 0],
            &[0, 197, 211, 0, 0, 0, 0, 161, 238, 6, 0],
            &[1, 249, 141, 0, 0, 0, 0, 93, 255, 44, 0],
            &[11, 255, 120, 0, 0, 0, 0, 76, 255, 62, 0],
            &[1, 247, 142, 0, 0, 0, 0, 99, 255, 44, 0],
            &[0, 185, 218, 3, 0, 0, 0, 176, 235, 3, 0],
            &[0, 61, 252, 167, 42, 32, 132, 255, 113, 0, 0],
            &[0, 0, 77, 220, 255, 255, 239, 118, 0, 0, 0],
            &[0, 0, 0, 0, 28, 35, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'p' / 0x70
        'p' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 17, 0, 26, 63, 40, 0, 0, 0, 0],
            &[0, 163, 181, 136, 253, 253, 255, 188, 16, 0, 0],
            &[0, 163, 251, 186, 28, 6, 81, 245, 160, 0, 0],
            &[0, 163, 255, 34, 0, 0, 0, 151, 249, 11, 0],
            &[0, 163, 239, 0, 0, 0, 0, 97, 255, 51, 0],
            &[0, 163, 227, 0, 0, 0, 0, 82, 255, 63, 0],
            &[0, 163, 247, 0, 0, 0, 0, 102, 255, 45, 0],
            &[0, 163, 255, 56, 0, 0, 0, 168, 242, 5, 0],
            &[0, 163, 248, 211, 60, 33, 114, 252, 138, 0, 0],
            &[0, 163, 214, 106, 241, 255, 251, 154, 6, 0, 0],
            &[0, 163, 223, 0, 4, 33, 13, 0, 0, 0, 0],
            &[0, 163, 223, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 223, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 17, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'q' / 0x71
        'q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 64, 41, 0, 11, 24, 0],
            &[0, 0, 0, 133, 252, 254, 255, 181, 135, 214, 0],
            &[0, 0, 96, 255, 125, 13, 20, 157, 252, 214, 0],
            &[0, 0, 204, 211, 0, 0, 0, 9, 242, 214, 0],
            &[0, 2, 252, 151, 0, 0, 0, 0, 194, 214, 0],
            &[0, 12, 255, 134, 0, 0, 0, 0, 175, 214, 0],
            &[0, 2, 249, 155, 0, 0, 0, 0, 191, 214, 0],
            &[0, 0, 196, 225, 2, 0, 0, 12, 239, 214, 0],
            &[0, 0, 84, 255, 161, 42, 45, 175, 247, 214, 0],
            &[0, 0, 0, 113, 242, 255, 252, 152, 169, 214, 0],
            &[0, 0, 0, 0, 7, 36, 16, 0, 172, 214, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 172, 214, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 172, 214, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 13, 17, 0],
        ]),
        // letter: 'r' / 0x72
        'r' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 19, 28, 28, 28, 1, 5, 58, 55, 8, 0],
            &[0, 167, 255, 255, 255, 51, 212, 255, 255, 149, 0],
            &[0, 0, 11, 71, 255, 216, 163, 33, 41, 37, 0],
            &[0, 0, 0, 44, 255, 211, 2, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 97, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 95, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 95, 0, 0, 0, 0, 0],
            &[0, 15, 39, 94, 255, 134, 42, 18, 0, 0, 0],
            &[0, 242, 255, 255, 255, 255, 255, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 's' / 0x73
        's' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 52, 66, 45, 4, 0, 0, 0],
            &[0, 0, 84, 237, 255, 255, 255, 238, 28, 0, 0],
            &[0, 1, 239, 180, 25, 3, 27, 86, 0, 0, 0],
            &[0, 5, 252, 149, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 127, 255, 178, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 194, 255, 199, 56, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 199, 248, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 255, 98, 0, 0],
            &[0, 42, 139, 59, 28, 40, 154, 255, 52, 0, 0],
            &[0, 34, 209, 255, 255, 255, 234, 106, 0, 0, 0],
            &[0, 0, 0, 13, 36, 29, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 't' / 0x74
        't' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 133, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 148, 247, 28, 28, 28, 14, 0],
            &[0, 10, 243, 255, 255, 255, 255, 255, 255, 132, 0],
            &[0, 2, 41, 41, 162, 248, 41, 41, 41, 21, 0],
            &[0, 0, 0, 0, 144, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 144, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 144, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 144, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 136, 255, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 255, 154, 39, 31, 49, 0],
            &[0, 0, 0, 0, 2, 143, 249, 255, 255, 191, 0],
            &[0, 0, 0, 0, 0, 0, 8, 37, 29, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'u' / 0x75
        'u' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 23, 0, 0, 0, 0, 19, 24, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 177, 214, 0, 0],
            &[0, 185, 207, 0, 0, 0, 0, 194, 214, 0, 0],
            &[0, 166, 233, 1, 0, 0, 11, 242, 214, 0, 0],
            &[0, 107, 255, 129, 31, 43, 174, 243, 214, 0, 0],
            &[0, 5, 163, 254, 255, 249, 143, 122, 214, 0, 0],
            &[0, 0, 0, 10, 35, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'v' / 0x76
        'v' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[6, 28, 9, 0, 0, 0, 0, 5, 28, 11, 0],
            &[12, 245, 139, 0, 0, 0, 0, 99, 255, 48, 0],
            &[0, 163, 227, 1, 0, 0, 0, 187, 208, 0, 0],
            &[0, 69, 255, 63, 0, 0, 22, 252, 113, 0, 0],
            &[0, 2, 227, 153, 0, 0, 106, 251, 22, 0, 0],
            &[0, 0, 135, 237, 6, 0, 194, 178, 0, 0, 0],
            &[0, 0, 41, 255, 78, 28, 254, 82, 0, 0, 0],
            &[0, 0, 0, 202, 167, 117, 236, 6, 0, 0, 0],
            &[0, 0, 0, 108, 244, 213, 147, 0, 0, 0, 0],
            &[0, 0, 0, 19, 249, 255, 52, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'w' / 0x77
        'w' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[20, 20, 0, 0, 0, 0, 0, 0, 13, 26, 0],
            &[161, 202, 0, 0, 0, 0, 0, 0, 139, 216, 0],
            &[120, 239, 0, 0, 81, 99, 0, 0, 180, 174, 0],
            &[79, 255, 21, 10, 247, 255, 46, 0, 220, 132, 0],
            &[37, 255, 59, 75, 241, 201, 116, 8, 253, 90, 0],
            &[3, 249, 96, 148, 180, 136, 187, 44, 255, 48, 0],
            &[0, 211, 128, 214, 111, 71, 247, 85, 253, 8, 0],
            &[0, 170, 165, 254, 43, 11, 250, 160, 219, 0, 0],
            &[0, 129, 230, 232, 0, 0, 197, 230, 177, 0, 0],
            &[0, 88, 255, 166, 0, 0, 131, 255, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'x' / 0x78
        'x' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 19, 0, 0, 0, 0, 14, 28, 6, 0],
            &[0, 157, 243, 35, 0, 0, 11, 216, 202, 5, 0],
            &[0, 10, 214, 201, 4, 0, 155, 240, 33, 0, 0],
            &[0, 0, 45, 246, 133, 82, 255, 84, 0, 0, 0],
            &[0, 0, 0, 103, 253, 241, 151, 0, 0, 0, 0],
            &[0, 0, 0, 12, 239, 255, 44, 0, 0, 0, 0],
            &[0, 0, 0, 164, 234, 206, 204, 6, 0, 0, 0],
            &[0, 0, 97, 254, 75, 36, 243, 144, 0, 0, 0],
            &[0, 42, 245, 144, 0, 0, 96, 255, 79, 0, 0],
            &[9, 211, 208, 7, 0, 0, 0, 168, 237, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'y' / 0x79
        'y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[6, 28, 8, 0, 0, 0, 0, 4, 28, 11, 0],
            &[10, 239, 130, 0, 0, 0, 0, 89, 255, 48, 0],
            &[0, 144, 229, 4, 0, 0, 0, 175, 209, 0, 0],
            &[0, 39, 254, 82, 0, 0, 13, 247, 114, 0, 0],
            &[0, 0, 187, 186, 0, 0, 90, 251, 23, 0, 0],
            &[0, 0, 81, 254, 36, 0, 176, 180, 0, 0, 0],
            &[0, 0, 3, 226, 135, 12, 247, 85, 0, 0, 0],
            &[0, 0, 0, 124, 226, 88, 238, 7, 0, 0, 0],
            &[0, 0, 0, 23, 249, 213, 151, 0, 0, 0, 0],
            &[0, 0, 0, 0, 167, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 240, 115, 0, 0, 0, 0, 0],
            &[0, 20, 29, 169, 238, 14, 0, 0, 0, 0, 0],
            &[0, 22, 35, 5, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'z' / 0x7a
        'z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 28, 28, 28, 28, 28, 28, 15, 0, 0],
            &[0, 50, 255, 255, 255, 255, 255, 255, 136, 0, 0],
            &[0, 3, 15, 15, 15, 15, 130, 250, 59, 0, 0],
            &[0, 0, 0, 0, 0, 63, 251, 110, 0, 0, 0],
            &[0, 0, 0, 0, 25, 232, 165, 0, 0, 0, 0],
            &[0, 0, 0, 4, 195, 210, 10, 0, 0, 0, 0],
            &[0, 0, 0, 143, 240, 36, 0, 0, 0, 0, 0],
            &[0, 0, 86, 253, 77, 0, 0, 0, 0, 0, 0],
            &[0, 40, 243, 188, 74, 74, 74, 74, 50, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '{' / 0x7b
        '{' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 126, 240, 209, 0, 0, 0],
            &[0, 0, 0, 0, 81, 255, 140, 34, 0, 0, 0],
            &[0, 0, 0, 0, 134, 248, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 238, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 238, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 224, 0, 0, 0, 0, 0],
            &[0, 0, 61, 145, 249, 103, 0, 0, 0, 0, 0],
            &[0, 0, 145, 253, 166, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 217, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 236, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 238, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 239, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 121, 254, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 239, 223, 124, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 129, 136, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '|' / 0x7c
        '|' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 14, 0, 0, 0, 0, 0],
        ]),
        // letter: '}' / 0x7d
        '}' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 158, 248, 164, 10, 0, 0, 0, 0, 0],
            &[0, 0, 20, 93, 250, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 197, 185, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 190, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 190, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 213, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 69, 246, 164, 79, 0, 0, 0],
            &[0, 0, 0, 0, 5, 130, 251, 199, 0, 0, 0],
            &[0, 0, 0, 0, 131, 244, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 184, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 190, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 188, 190, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 229, 173, 0, 0, 0, 0, 0],
            &[0, 0, 94, 213, 252, 69, 0, 0, 0, 0, 0],
            &[0, 0, 104, 140, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '~' / 0x7e
        '~' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 146, 204, 119, 2, 0, 46, 182, 0, 0],
            &[0, 97, 187, 62, 194, 176, 15, 119, 155, 0, 0],
            &[0, 165, 91, 0, 7, 157, 246, 226, 34, 0, 0],
            &[0, 22, 8, 0, 0, 0, 14, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ' ' / 0xa0
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¡' / 0xa1
        '¡' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 238, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 238, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 107, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 152, 199, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 160, 207, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 168, 215, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 176, 223, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 184, 231, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 192, 239, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 254, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 30, 37, 0, 0, 0, 0, 0],
        ]),
        // letter: '¢' / 0xa2
        '¢' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 77, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 252, 43, 0, 0, 0, 0],
            &[0, 0, 0, 85, 190, 255, 234, 198, 85, 0, 0],
            &[0, 0, 118, 255, 174, 100, 103, 147, 72, 0, 0],
            &[0, 20, 248, 156, 0, 0, 0, 0, 0, 0, 0],
            &[0, 86, 255, 54, 0, 0, 0, 0, 0, 0, 0],
            &[0, 112, 255, 22, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 255, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 70, 255, 82, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 226, 216, 31, 0, 0, 7, 28, 0, 0],
            &[0, 0, 57, 233, 251, 212, 217, 246, 132, 0, 0],
            &[0, 0, 0, 11, 87, 253, 140, 83, 12, 0, 0],
            &[0, 0, 0, 0, 0, 252, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 42, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '£' / 0xa3
        '£' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 42, 63, 27, 0, 0, 0],
            &[0, 0, 0, 19, 197, 255, 255, 254, 130, 0, 0],
            &[0, 0, 0, 149, 243, 54, 6, 57, 50, 0, 0],
            &[0, 0, 0, 210, 172, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 227, 155, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 227, 155, 0, 0, 0, 0, 0, 0],
            &[0, 60, 143, 243, 211, 143, 143, 88, 0, 0, 0],
            &[0, 60, 143, 243, 211, 143, 143, 88, 0, 0, 0],
            &[0, 0, 0, 227, 155, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 236, 145, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 255, 84, 0, 0, 0, 0, 0, 0],
            &[0, 44, 207, 228, 94, 92, 92, 92, 85, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 236, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¤' / 0xa4
        '¤' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 116, 144, 27, 108, 116, 44, 95, 166, 0, 0],
            &[0, 50, 239, 250, 192, 178, 241, 252, 90, 0, 0],
            &[0, 7, 230, 97, 0, 0, 46, 242, 43, 0, 0],
            &[0, 55, 236, 0, 0, 0, 0, 176, 114, 0, 0],
            &[0, 45, 244, 6, 0, 0, 0, 190, 107, 0, 0],
            &[0, 1, 215, 158, 13, 5, 105, 250, 26, 0, 0],
            &[0, 93, 249, 212, 252, 246, 228, 239, 141, 0, 0],
            &[0, 67, 82, 0, 41, 51, 6, 48, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¥' / 0xa5
        '¥' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[13, 35, 7, 0, 0, 0, 0, 0, 34, 20, 0],
            &[27, 248, 125, 0, 0, 0, 0, 70, 255, 69, 0],
            &[0, 144, 237, 13, 0, 0, 0, 198, 191, 0, 0],
            &[0, 24, 245, 119, 0, 0, 70, 255, 58, 0, 0],
            &[0, 0, 138, 234, 10, 0, 197, 180, 0, 0, 0],
            &[0, 0, 20, 243, 114, 69, 254, 47, 0, 0, 0],
            &[0, 0, 0, 132, 231, 200, 168, 0, 0, 0, 0],
            &[0, 18, 195, 207, 255, 255, 215, 195, 57, 0, 0],
            &[0, 7, 83, 83, 195, 223, 83, 83, 24, 0, 0],
            &[0, 6, 71, 71, 191, 221, 71, 71, 21, 0, 0],
            &[0, 19, 206, 206, 238, 246, 206, 206, 61, 0, 0],
            &[0, 0, 0, 0, 165, 208, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 165, 208, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¦' / 0xa6
        '¦' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 124, 167, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 75, 101, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 51, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 17, 0, 0, 0, 0, 0],
        ]),
        // letter: '§' / 0xa7
        '§' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 35, 175, 235, 249, 226, 168, 10, 0, 0],
            &[0, 0, 200, 203, 48, 33, 61, 114, 0, 0, 0],
            &[0, 0, 230, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 99, 252, 178, 60, 0, 0, 0, 0, 0],
            &[0, 0, 45, 230, 201, 251, 198, 46, 0, 0, 0],
            &[0, 0, 216, 108, 0, 24, 167, 240, 26, 0, 0],
            &[0, 0, 248, 84, 0, 0, 6, 249, 76, 0, 0],
            &[0, 0, 153, 240, 101, 8, 70, 251, 30, 0, 0],
            &[0, 0, 1, 112, 228, 235, 248, 90, 0, 0, 0],
            &[0, 0, 0, 0, 6, 97, 227, 206, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 50, 255, 68, 0, 0],
            &[0, 23, 99, 22, 0, 3, 118, 253, 32, 0, 0],
            &[0, 23, 212, 255, 243, 251, 225, 89, 0, 0, 0],
            &[0, 0, 0, 16, 36, 21, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¨' / 0xa8
        '¨' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 55, 99, 0, 0, 69, 85, 0, 0, 0],
            &[0, 0, 169, 249, 6, 0, 197, 228, 0, 0, 0],
            &[0, 0, 35, 68, 0, 0, 45, 58, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '©' / 0xa9
        '©' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 102, 109, 68, 2, 0, 0, 0],
            &[0, 10, 171, 183, 109, 104, 162, 207, 33, 0, 0],
            &[0, 152, 118, 30, 167, 200, 109, 74, 205, 1, 0],
            &[15, 216, 1, 200, 98, 21, 33, 0, 176, 62, 0],
            &[52, 172, 16, 233, 0, 0, 0, 0, 122, 104, 0],
            &[47, 179, 13, 237, 1, 0, 0, 0, 127, 97, 0],
            &[8, 225, 12, 182, 136, 68, 58, 0, 196, 40, 0],
            &[0, 121, 171, 19, 114, 145, 63, 122, 165, 0, 0],
            &[0, 1, 119, 214, 163, 156, 205, 147, 8, 0, 0],
            &[0, 0, 0, 7, 42, 47, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ª' / 0xaa
        'ª' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 52, 56, 7, 0, 0, 0, 0],
            &[0, 0, 1, 205, 175, 191, 218, 11, 0, 0, 0],
            &[0, 0, 0, 1, 0, 0, 220, 61, 0, 0, 0],
            &[0, 0, 6, 140, 197, 197, 242, 70, 0, 0, 0],
            &[0, 0, 78, 213, 3, 0, 217, 70, 0, 0, 0],
            &[0, 0, 65, 238, 112, 152, 236, 70, 0, 0, 0],
            &[0, 0, 0, 73, 111, 47, 57, 27, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '«' / 0xab
        '«' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 0, 0, 2, 0, 0, 0],
            &[0, 0, 0, 42, 217, 33, 6, 192, 95, 0, 0],
            &[0, 0, 17, 219, 174, 1, 156, 228, 23, 0, 0],
            &[0, 3, 186, 212, 12, 112, 248, 54, 0, 0, 0],
            &[0, 92, 255, 66, 21, 250, 147, 0, 0, 0, 0],
            &[0, 5, 192, 206, 8, 119, 246, 50, 0, 0, 0],
            &[0, 0, 20, 222, 160, 0, 163, 224, 20, 0, 0],
            &[0, 0, 0, 44, 222, 31, 7, 197, 98, 0, 0],
            &[0, 0, 0, 0, 2, 0, 0, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¬' / 0xac
        '¬' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 110, 152, 152, 152, 152, 152, 152, 115, 0, 0],
            &[0, 113, 156, 156, 156, 156, 156, 201, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 115, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 115, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 95, 159, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '®' / 0xae
        '®' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 102, 109, 68, 2, 0, 0, 0],
            &[0, 10, 171, 183, 109, 104, 162, 207, 33, 0, 0],
            &[0, 152, 118, 84, 197, 182, 70, 74, 205, 1, 0],
            &[15, 216, 1, 110, 113, 46, 220, 0, 176, 62, 0],
            &[52, 172, 0, 110, 151, 119, 176, 0, 122, 104, 0],
            &[47, 179, 0, 110, 180, 212, 54, 0, 127, 97, 0],
            &[8, 225, 12, 110, 111, 60, 195, 1, 196, 40, 0],
            &[0, 121, 171, 61, 53, 0, 99, 150, 165, 0, 0],
            &[0, 1, 119, 214, 163, 156, 205, 147, 8, 0, 0],
            &[0, 0, 0, 7, 42, 47, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¯' / 0xaf
        '¯' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 75, 182, 182, 182, 182, 115, 0, 0, 0],
            &[0, 0, 52, 126, 126, 126, 126, 80, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '°' / 0xb0
        '°' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 64, 5, 0, 0, 0, 0],
            &[0, 0, 4, 181, 220, 203, 215, 21, 0, 0, 0],
            &[0, 0, 82, 201, 3, 0, 149, 133, 0, 0, 0],
            &[0, 0, 110, 160, 0, 0, 103, 161, 0, 0, 0],
            &[0, 0, 42, 239, 87, 62, 213, 88, 0, 0, 0],
            &[0, 0, 0, 74, 191, 202, 108, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '±' / 0xb1
        '±' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 172, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 173, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 173, 0, 0, 0, 0, 0],
            &[0, 117, 165, 165, 214, 226, 165, 165, 139, 0, 0],
            &[0, 101, 143, 143, 204, 219, 143, 143, 120, 0, 0],
            &[0, 0, 0, 0, 139, 173, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 173, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 124, 154, 0, 0, 0, 0, 0],
            &[0, 33, 52, 52, 52, 52, 52, 52, 44, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '²' / 0xb2
        '²' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 97, 114, 41, 0, 0, 0, 0],
            &[0, 0, 19, 229, 158, 156, 248, 47, 0, 0, 0],
            &[0, 0, 0, 14, 0, 0, 195, 109, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 233, 54, 0, 0, 0],
            &[0, 0, 0, 0, 11, 194, 134, 0, 0, 0, 0],
            &[0, 0, 0, 13, 195, 140, 0, 0, 0, 0, 0],
            &[0, 0, 15, 199, 160, 32, 32, 23, 0, 0, 0],
            &[0, 0, 86, 237, 237, 237, 237, 174, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '³' / 0xb3
        '³' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 37, 104, 112, 36, 0, 0, 0, 0],
            &[0, 0, 25, 227, 147, 167, 244, 29, 0, 0, 0],
            &[0, 0, 0, 4, 0, 4, 246, 77, 0, 0, 0],
            &[0, 0, 0, 29, 80, 144, 215, 17, 0, 0, 0],
            &[0, 0, 0, 61, 163, 200, 196, 25, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 194, 128, 0, 0, 0],
            &[0, 0, 38, 41, 0, 17, 220, 108, 0, 0, 0],
            &[0, 0, 61, 225, 243, 242, 162, 7, 0, 0, 0],
            &[0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '´' / 0xb4
        '´' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 5, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 254, 80, 0, 0, 0, 0],
            &[0, 0, 0, 39, 247, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 123, 100, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'µ' / 0xb5
        'µ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 23, 0, 0, 0, 0, 19, 24, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 177, 214, 0, 0],
            &[0, 255, 255, 0, 0, 0, 0, 194, 214, 0, 0],
            &[0, 255, 255, 1, 0, 0, 11, 242, 214, 0, 0],
            &[0, 255, 255, 129, 31, 43, 174, 243, 214, 0, 0],
            &[0, 191, 255, 254, 255, 249, 143, 122, 214, 0, 0],
            &[0, 185, 205, 10, 35, 11, 0, 0, 0, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 16, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¶' / 0xb6
        '¶' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 120, 209, 233, 235, 235, 235, 235, 38, 0],
            &[0, 130, 255, 255, 255, 255, 117, 8, 212, 41, 0],
            &[1, 233, 255, 255, 255, 255, 113, 0, 210, 41, 0],
            &[21, 255, 255, 255, 255, 255, 113, 0, 210, 41, 0],
            &[33, 255, 255, 255, 255, 255, 113, 0, 210, 41, 0],
            &[15, 255, 255, 255, 255, 255, 113, 0, 210, 41, 0],
            &[0, 212, 255, 255, 255, 255, 113, 0, 210, 41, 0],
            &[0, 76, 250, 255, 255, 255, 113, 0, 210, 41, 0],
            &[0, 0, 39, 114, 124, 185, 113, 0, 210, 41, 0],
            &[0, 0, 0, 0, 0, 138, 113, 0, 210, 41, 0],
            &[0, 0, 0, 0, 0, 138, 113, 0, 210, 41, 0],
            &[0, 0, 0, 0, 0, 138, 113, 0, 210, 41, 0],
            &[0, 0, 0, 0, 0, 138, 113, 0, 210, 41, 0],
            &[0, 0, 0, 0, 0, 138, 113, 0, 210, 41, 0],
            &[0, 0, 0, 0, 0, 138, 113, 0, 210, 41, 0],
            &[0, 0, 0, 0, 0, 22, 18, 0, 33, 6, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '·' / 0xb7
        '·' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 151, 196, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 215, 250, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¸' / 0xb8
        '¸' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 140, 118, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 140, 227, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 156, 177, 0, 0, 0, 0],
            &[0, 0, 0, 6, 12, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¹' / 0xb9
        '¹' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 46, 81, 0, 0, 0, 0, 0],
            &[0, 0, 23, 158, 247, 234, 0, 0, 0, 0, 0],
            &[0, 0, 42, 168, 76, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 248, 4, 0, 0, 0, 0],
            &[0, 0, 0, 216, 255, 255, 236, 167, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'º' / 0xba
        'º' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 56, 3, 0, 0, 0, 0],
            &[0, 0, 0, 166, 205, 190, 208, 12, 0, 0, 0],
            &[0, 0, 56, 222, 3, 0, 183, 107, 0, 0, 0],
            &[0, 0, 93, 178, 0, 0, 137, 142, 0, 0, 0],
            &[0, 0, 69, 205, 0, 0, 162, 116, 0, 0, 0],
            &[0, 0, 4, 209, 150, 132, 229, 24, 0, 0, 0],
            &[0, 0, 0, 12, 98, 105, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '»' / 0xbb
        '»' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 0, 0, 2, 0, 0, 0, 0, 0],
            &[0, 52, 220, 29, 4, 204, 86, 0, 0, 0, 0],
            &[0, 3, 187, 206, 10, 106, 245, 50, 0, 0, 0],
            &[0, 0, 19, 223, 172, 1, 160, 227, 24, 0, 0],
            &[0, 0, 0, 91, 255, 71, 18, 248, 148, 0, 0],
            &[0, 0, 20, 225, 168, 1, 167, 225, 22, 0, 0],
            &[0, 4, 190, 204, 9, 119, 245, 49, 0, 0, 0],
            &[0, 51, 217, 27, 6, 203, 86, 0, 0, 0, 0],
            &[0, 0, 2, 0, 0, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¼' / 0xbc
        '¼' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 33, 0, 0, 0, 0, 11, 33, 0, 0],
            &[94, 231, 245, 0, 0, 0, 0, 161, 169, 0, 0],
            &[117, 109, 245, 0, 0, 0, 40, 249, 40, 0, 0],
            &[0, 37, 245, 0, 0, 0, 170, 161, 0, 0, 0],
            &[0, 37, 245, 0, 0, 48, 249, 34, 0, 0, 0],
            &[14, 86, 255, 44, 10, 179, 152, 0, 0, 0, 0],
            &[88, 223, 255, 194, 122, 247, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 189, 143, 0, 142, 208, 0, 0],
            &[0, 0, 0, 65, 244, 22, 72, 229, 223, 0, 0],
            &[0, 0, 0, 198, 135, 21, 222, 94, 223, 0, 0],
            &[0, 0, 75, 240, 18, 173, 170, 110, 232, 53, 0],
            &[0, 1, 206, 126, 0, 155, 172, 190, 245, 130, 0],
            &[0, 84, 236, 13, 0, 0, 0, 54, 223, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '½' / 0xbd
        '½' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 33, 0, 0, 0, 0, 11, 33, 0, 0],
            &[94, 231, 245, 0, 0, 0, 0, 161, 169, 0, 0],
            &[117, 109, 245, 0, 0, 0, 40, 249, 40, 0, 0],
            &[0, 37, 245, 0, 0, 0, 170, 161, 0, 0, 0],
            &[0, 37, 245, 0, 0, 48, 249, 34, 0, 0, 0],
            &[14, 86, 255, 44, 10, 179, 152, 0, 0, 0, 0],
            &[88, 223, 255, 194, 122, 247, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 189, 165, 188, 248, 217, 42, 0],
            &[0, 0, 0, 65, 244, 25, 78, 7, 174, 144, 0],
            &[0, 0, 0, 198, 135, 0, 0, 4, 202, 101, 0],
            &[0, 0, 75, 240, 18, 0, 6, 174, 167, 2, 0],
            &[0, 1, 206, 126, 0, 9, 183, 164, 16, 10, 0],
            &[0, 84, 236, 13, 0, 94, 255, 255, 255, 193, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¾' / 0xbe
        '¾' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 63, 14, 0, 0, 0, 11, 33, 0, 0],
            &[101, 223, 201, 232, 21, 0, 0, 161, 169, 0, 0],
            &[5, 4, 12, 252, 57, 0, 40, 249, 40, 0, 0],
            &[0, 166, 227, 197, 4, 0, 170, 161, 0, 0, 0],
            &[0, 43, 79, 243, 75, 48, 249, 34, 0, 0, 0],
            &[59, 50, 58, 240, 86, 179, 152, 0, 0, 0, 0],
            &[87, 207, 206, 135, 59, 247, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 189, 143, 0, 142, 208, 0, 0],
            &[0, 0, 0, 65, 244, 22, 72, 229, 223, 0, 0],
            &[0, 0, 0, 198, 135, 21, 222, 94, 223, 0, 0],
            &[0, 0, 75, 240, 18, 173, 170, 110, 232, 53, 0],
            &[0, 1, 206, 126, 0, 155, 172, 190, 245, 130, 0],
            &[0, 84, 236, 13, 0, 0, 0, 54, 223, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¿' / 0xbf
        '¿' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 183, 245, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 183, 245, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 163, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 113, 253, 88, 0, 0, 0, 0, 0],
            &[0, 1, 147, 250, 97, 0, 0, 0, 0, 0, 0],
            &[0, 102, 252, 67, 0, 0, 0, 0, 0, 0, 0],
            &[0, 192, 178, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 215, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 157, 244, 78, 12, 29, 78, 145, 0, 0, 0],
            &[0, 17, 187, 255, 255, 255, 255, 203, 23, 0, 0],
            &[0, 0, 0, 36, 73, 69, 30, 0, 0, 0, 0],
        ]),
        // letter: 'À' / 0xc0
        'À' => Some(&[
            &[0, 0, 6, 206, 232, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 219, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 162, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 236, 255, 32, 0, 0, 0, 0],
            &[0, 0, 0, 67, 254, 236, 114, 0, 0, 0, 0],
            &[0, 0, 0, 149, 210, 162, 197, 0, 0, 0, 0],
            &[0, 0, 2, 230, 134, 87, 254, 26, 0, 0, 0],
            &[0, 0, 59, 255, 55, 14, 249, 108, 0, 0, 0],
            &[0, 0, 142, 230, 1, 0, 184, 191, 0, 0, 0],
            &[0, 0, 224, 171, 21, 21, 126, 252, 21, 0, 0],
            &[0, 52, 255, 255, 255, 255, 255, 255, 101, 0, 0],
            &[0, 134, 255, 77, 70, 70, 70, 229, 184, 0, 0],
            &[0, 217, 169, 0, 0, 0, 0, 122, 250, 17, 0],
            &[44, 255, 90, 0, 0, 0, 0, 43, 255, 95, 0],
            &[127, 250, 16, 0, 0, 0, 0, 0, 219, 178, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 12, 8, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Á' / 0xc1
        'Á' => Some(&[
            &[0, 0, 0, 0, 0, 75, 255, 131, 0, 0, 0],
            &[0, 0, 0, 0, 11, 222, 161, 1, 0, 0, 0],
            &[0, 0, 0, 0, 82, 129, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 236, 255, 32, 0, 0, 0, 0],
            &[0, 0, 0, 67, 254, 236, 114, 0, 0, 0, 0],
            &[0, 0, 0, 149, 210, 162, 197, 0, 0, 0, 0],
            &[0, 0, 2, 230, 134, 87, 254, 26, 0, 0, 0],
            &[0, 0, 59, 255, 55, 14, 249, 108, 0, 0, 0],
            &[0, 0, 142, 230, 1, 0, 184, 191, 0, 0, 0],
            &[0, 0, 224, 171, 21, 21, 126, 252, 21, 0, 0],
            &[0, 52, 255, 255, 255, 255, 255, 255, 101, 0, 0],
            &[0, 134, 255, 77, 70, 70, 70, 229, 184, 0, 0],
            &[0, 217, 169, 0, 0, 0, 0, 122, 250, 17, 0],
            &[44, 255, 90, 0, 0, 0, 0, 43, 255, 95, 0],
            &[127, 250, 16, 0, 0, 0, 0, 0, 219, 178, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 10, 0, 0, 0],
        ]),
        // letter: 'Â' / 0xc2
        'Â' => Some(&[
            &[0, 0, 0, 33, 241, 253, 62, 0, 0, 0, 0],
            &[0, 0, 14, 210, 148, 116, 233, 35, 0, 0, 0],
            &[0, 0, 105, 112, 0, 0, 86, 138, 0, 0, 0],
            &[0, 0, 0, 0, 36, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 236, 255, 32, 0, 0, 0, 0],
            &[0, 0, 0, 67, 254, 236, 114, 0, 0, 0, 0],
            &[0, 0, 0, 149, 210, 162, 197, 0, 0, 0, 0],
            &[0, 0, 2, 230, 134, 87, 254, 26, 0, 0, 0],
            &[0, 0, 59, 255, 55, 14, 249, 108, 0, 0, 0],
            &[0, 0, 142, 230, 1, 0, 184, 191, 0, 0, 0],
            &[0, 0, 224, 171, 21, 21, 126, 252, 21, 0, 0],
            &[0, 52, 255, 255, 255, 255, 255, 255, 101, 0, 0],
            &[0, 134, 255, 77, 70, 70, 70, 229, 184, 0, 0],
            &[0, 217, 169, 0, 0, 0, 0, 122, 250, 17, 0],
            &[44, 255, 90, 0, 0, 0, 0, 43, 255, 95, 0],
            &[127, 250, 16, 0, 0, 0, 0, 0, 219, 178, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 10, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ã' / 0xc3
        'Ã' => Some(&[
            &[0, 0, 7, 104, 90, 5, 0, 107, 13, 0, 0],
            &[0, 0, 154, 213, 225, 217, 162, 219, 1, 0, 0],
            &[0, 0, 150, 11, 6, 106, 152, 43, 0, 0, 0],
            &[0, 0, 0, 0, 36, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 236, 255, 32, 0, 0, 0, 0],
            &[0, 0, 0, 67, 254, 236, 114, 0, 0, 0, 0],
            &[0, 0, 0, 149, 210, 162, 197, 0, 0, 0, 0],
            &[0, 0, 2, 230, 134, 87, 254, 26, 0, 0, 0],
            &[0, 0, 59, 255, 55, 14, 249, 108, 0, 0, 0],
            &[0, 0, 142, 230, 1, 0, 184, 191, 0, 0, 0],
            &[0, 0, 224, 171, 21, 21, 126, 252, 21, 0, 0],
            &[0, 52, 255, 255, 255, 255, 255, 255, 101, 0, 0],
            &[0, 134, 255, 77, 70, 70, 70, 229, 184, 0, 0],
            &[0, 217, 169, 0, 0, 0, 0, 122, 250, 17, 0],
            &[44, 255, 90, 0, 0, 0, 0, 43, 255, 95, 0],
            &[127, 250, 16, 0, 0, 0, 0, 0, 219, 178, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ä' / 0xc4
        'Ä' => Some(&[
            &[0, 0, 61, 103, 0, 0, 76, 88, 0, 0, 0],
            &[0, 0, 173, 247, 4, 0, 201, 224, 0, 0, 0],
            &[0, 0, 33, 61, 0, 0, 43, 51, 0, 0, 0],
            &[0, 0, 0, 0, 36, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 236, 255, 32, 0, 0, 0, 0],
            &[0, 0, 0, 67, 254, 236, 114, 0, 0, 0, 0],
            &[0, 0, 0, 149, 210, 162, 197, 0, 0, 0, 0],
            &[0, 0, 2, 230, 134, 87, 254, 26, 0, 0, 0],
            &[0, 0, 59, 255, 55, 14, 249, 108, 0, 0, 0],
            &[0, 0, 142, 230, 1, 0, 184, 191, 0, 0, 0],
            &[0, 0, 224, 171, 21, 21, 126, 252, 21, 0, 0],
            &[0, 52, 255, 255, 255, 255, 255, 255, 101, 0, 0],
            &[0, 134, 255, 77, 70, 70, 70, 229, 184, 0, 0],
            &[0, 217, 169, 0, 0, 0, 0, 122, 250, 17, 0],
            &[44, 255, 90, 0, 0, 0, 0, 43, 255, 95, 0],
            &[127, 250, 16, 0, 0, 0, 0, 0, 219, 178, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Å' / 0xc5
        'Å' => Some(&[
            &[0, 0, 0, 179, 95, 49, 229, 0, 0, 0, 0],
            &[0, 0, 0, 160, 148, 112, 209, 0, 0, 0, 0],
            &[0, 0, 0, 21, 142, 153, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 236, 255, 32, 0, 0, 0, 0],
            &[0, 0, 0, 67, 254, 236, 114, 0, 0, 0, 0],
            &[0, 0, 0, 149, 210, 162, 197, 0, 0, 0, 0],
            &[0, 0, 2, 230, 134, 87, 254, 26, 0, 0, 0],
            &[0, 0, 59, 255, 55, 14, 249, 108, 0, 0, 0],
            &[0, 0, 142, 230, 1, 0, 184, 191, 0, 0, 0],
            &[0, 0, 224, 171, 21, 21, 126, 252, 21, 0, 0],
            &[0, 52, 255, 255, 255, 255, 255, 255, 101, 0, 0],
            &[0, 134, 255, 77, 70, 70, 70, 229, 184, 0, 0],
            &[0, 217, 169, 0, 0, 0, 0, 122, 250, 17, 0],
            &[44, 255, 90, 0, 0, 0, 0, 43, 255, 95, 0],
            &[127, 250, 16, 0, 0, 0, 0, 0, 219, 178, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 52, 204, 213, 80, 0, 0, 0, 0],
        ]),
        // letter: 'Æ' / 0xc6
        'Æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 33, 35, 35, 35, 35, 35, 8, 0],
            &[0, 0, 21, 255, 255, 255, 255, 255, 255, 59, 0],
            &[0, 0, 78, 254, 122, 255, 90, 56, 56, 13, 0],
            &[0, 0, 135, 215, 74, 255, 43, 0, 0, 0, 0],
            &[0, 0, 193, 160, 74, 255, 43, 0, 0, 0, 0],
            &[0, 4, 245, 106, 74, 255, 48, 5, 4, 0, 0],
            &[0, 52, 255, 51, 74, 255, 255, 255, 232, 0, 0],
            &[0, 109, 252, 13, 82, 255, 112, 82, 74, 0, 0],
            &[0, 166, 255, 255, 255, 255, 43, 0, 0, 0, 0],
            &[0, 224, 206, 70, 145, 255, 43, 0, 0, 0, 0],
            &[25, 255, 89, 0, 74, 255, 43, 0, 0, 0, 0],
            &[82, 255, 35, 0, 74, 255, 120, 92, 92, 21, 0],
            &[140, 236, 0, 0, 74, 255, 255, 255, 255, 59, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ç' / 0xc7
        'Ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 60, 70, 43, 3, 0, 0],
            &[0, 0, 2, 132, 246, 255, 255, 255, 229, 37, 0],
            &[0, 0, 150, 254, 138, 39, 22, 56, 123, 1, 0],
            &[0, 51, 255, 134, 0, 0, 0, 0, 0, 0, 0],
            &[0, 143, 250, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 196, 206, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 217, 181, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 213, 185, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 191, 211, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 253, 22, 0, 0, 0, 0, 0, 0, 0],
            &[0, 45, 253, 152, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 141, 255, 175, 86, 70, 101, 135, 0, 0],
            &[0, 0, 0, 113, 230, 255, 255, 252, 167, 0, 0],
            &[0, 0, 0, 0, 0, 209, 106, 11, 0, 0, 0],
            &[0, 0, 0, 0, 10, 165, 226, 58, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 203, 130, 0, 0, 0],
            &[0, 0, 0, 0, 9, 9, 0, 0, 0, 0, 0],
        ]),
        // letter: 'È' / 0xc8
        'È' => Some(&[
            &[0, 0, 0, 185, 245, 27, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 204, 177, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 154, 46, 0, 0, 0, 0],
            &[0, 13, 35, 35, 35, 35, 35, 35, 26, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 98, 255, 85, 56, 56, 56, 56, 42, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 41, 5, 5, 5, 5, 2, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 110, 0, 0],
            &[0, 98, 255, 111, 86, 86, 86, 86, 37, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 115, 92, 92, 92, 92, 69, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 9, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'É' / 0xc9
        'É' => Some(&[
            &[0, 0, 0, 0, 0, 50, 253, 157, 0, 0, 0],
            &[0, 0, 0, 0, 4, 204, 183, 6, 0, 0, 0],
            &[0, 0, 0, 0, 64, 142, 7, 0, 0, 0, 0],
            &[0, 13, 35, 35, 35, 35, 35, 35, 26, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 98, 255, 85, 56, 56, 56, 56, 42, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 41, 5, 5, 5, 5, 2, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 110, 0, 0],
            &[0, 98, 255, 111, 86, 86, 86, 86, 37, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 115, 92, 92, 92, 92, 69, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 11, 0, 0, 0],
        ]),
        // letter: 'Ê' / 0xca
        'Ê' => Some(&[
            &[0, 0, 0, 19, 228, 255, 85, 0, 0, 0, 0],
            &[0, 0, 6, 192, 170, 96, 240, 51, 0, 0, 0],
            &[0, 0, 87, 127, 2, 0, 68, 151, 4, 0, 0],
            &[0, 13, 35, 35, 35, 35, 35, 35, 26, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 98, 255, 85, 56, 56, 56, 56, 42, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 41, 5, 5, 5, 5, 2, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 110, 0, 0],
            &[0, 98, 255, 111, 86, 86, 86, 86, 37, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 115, 92, 92, 92, 92, 69, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 11, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ë' / 0xcb
        'Ë' => Some(&[
            &[0, 0, 48, 116, 0, 0, 62, 102, 0, 0, 0],
            &[0, 0, 147, 255, 22, 0, 175, 246, 3, 0, 0],
            &[0, 0, 24, 69, 0, 0, 33, 60, 0, 0, 0],
            &[0, 13, 35, 35, 35, 35, 35, 35, 26, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 98, 255, 85, 56, 56, 56, 56, 42, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 41, 5, 5, 5, 5, 2, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 110, 0, 0],
            &[0, 98, 255, 111, 86, 86, 86, 86, 37, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 115, 92, 92, 92, 92, 69, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ì' / 0xcc
        'Ì' => Some(&[
            &[0, 0, 6, 206, 232, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 219, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 162, 29, 0, 0, 0, 0],
            &[0, 4, 35, 35, 35, 35, 35, 35, 11, 0, 0],
            &[0, 30, 248, 255, 255, 255, 255, 251, 78, 0, 0],
            &[0, 0, 0, 16, 184, 222, 20, 1, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 1, 28, 51, 195, 227, 55, 32, 5, 0, 0],
            &[0, 33, 255, 255, 255, 255, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 12, 8, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Í' / 0xcd
        'Í' => Some(&[
            &[0, 0, 0, 0, 0, 75, 255, 131, 0, 0, 0],
            &[0, 0, 0, 0, 11, 222, 161, 1, 0, 0, 0],
            &[0, 0, 0, 0, 82, 129, 2, 0, 0, 0, 0],
            &[0, 4, 35, 35, 35, 35, 35, 35, 11, 0, 0],
            &[0, 30, 248, 255, 255, 255, 255, 251, 78, 0, 0],
            &[0, 0, 0, 16, 184, 222, 20, 1, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 1, 28, 51, 195, 227, 55, 32, 5, 0, 0],
            &[0, 33, 255, 255, 255, 255, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 10, 0, 0, 0],
        ]),
        // letter: 'Î' / 0xce
        'Î' => Some(&[
            &[0, 0, 0, 33, 241, 253, 62, 0, 0, 0, 0],
            &[0, 0, 14, 210, 148, 116, 233, 35, 0, 0, 0],
            &[0, 0, 105, 112, 0, 0, 86, 138, 0, 0, 0],
            &[0, 4, 35, 35, 35, 35, 35, 35, 11, 0, 0],
            &[0, 30, 248, 255, 255, 255, 255, 251, 78, 0, 0],
            &[0, 0, 0, 16, 184, 222, 20, 1, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 1, 28, 51, 195, 227, 55, 32, 5, 0, 0],
            &[0, 33, 255, 255, 255, 255, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 10, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ï' / 0xcf
        'Ï' => Some(&[
            &[0, 0, 61, 103, 0, 0, 76, 88, 0, 0, 0],
            &[0, 0, 173, 247, 4, 0, 201, 224, 0, 0, 0],
            &[0, 0, 33, 61, 0, 0, 43, 51, 0, 0, 0],
            &[0, 4, 35, 35, 35, 35, 35, 35, 11, 0, 0],
            &[0, 30, 248, 255, 255, 255, 255, 251, 78, 0, 0],
            &[0, 0, 0, 16, 184, 222, 20, 1, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 1, 28, 51, 195, 227, 55, 32, 5, 0, 0],
            &[0, 33, 255, 255, 255, 255, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ð' / 0xd0
        'Ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 28, 3, 0, 0, 0, 0, 0],
            &[0, 163, 255, 255, 255, 251, 178, 46, 0, 0, 0],
            &[0, 163, 233, 48, 58, 105, 219, 245, 61, 0, 0],
            &[0, 163, 227, 0, 0, 0, 15, 217, 222, 4, 0],
            &[0, 163, 227, 0, 0, 0, 0, 89, 255, 69, 0],
            &[0, 163, 227, 0, 0, 0, 0, 21, 255, 124, 0],
            &[169, 255, 255, 169, 163, 0, 0, 1, 251, 146, 0],
            &[169, 255, 255, 169, 163, 0, 0, 1, 253, 139, 0],
            &[0, 163, 227, 0, 0, 0, 0, 32, 255, 113, 0],
            &[0, 163, 227, 0, 0, 0, 0, 112, 255, 43, 0],
            &[0, 163, 227, 0, 0, 0, 40, 237, 189, 0, 0],
            &[0, 163, 236, 83, 97, 151, 245, 219, 30, 0, 0],
            &[0, 163, 255, 255, 241, 209, 126, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ñ' / 0xd1
        'Ñ' => Some(&[
            &[0, 0, 3, 97, 97, 8, 0, 96, 25, 0, 0],
            &[0, 0, 132, 220, 219, 225, 159, 228, 9, 0, 0],
            &[0, 0, 138, 23, 3, 97, 153, 55, 0, 0, 0],
            &[0, 28, 35, 8, 0, 0, 0, 16, 35, 0, 0],
            &[0, 207, 255, 125, 0, 0, 0, 120, 255, 2, 0],
            &[0, 207, 230, 231, 6, 0, 0, 120, 255, 2, 0],
            &[0, 207, 152, 243, 93, 0, 0, 120, 255, 2, 0],
            &[0, 207, 155, 146, 205, 0, 0, 120, 255, 2, 0],
            &[0, 207, 167, 36, 253, 61, 0, 120, 255, 2, 0],
            &[0, 207, 171, 0, 178, 173, 0, 120, 255, 2, 0],
            &[0, 207, 171, 0, 65, 252, 32, 119, 255, 2, 0],
            &[0, 207, 171, 0, 0, 208, 140, 115, 255, 2, 0],
            &[0, 207, 171, 0, 0, 97, 240, 119, 255, 2, 0],
            &[0, 207, 171, 0, 0, 7, 233, 199, 255, 2, 0],
            &[0, 207, 171, 0, 0, 0, 128, 255, 255, 2, 0],
            &[0, 207, 171, 0, 0, 0, 23, 248, 255, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ò' / 0xd2
        'Ò' => Some(&[
            &[0, 0, 6, 206, 232, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 219, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 162, 29, 0, 0, 0, 0],
            &[0, 0, 0, 16, 65, 76, 28, 0, 0, 0, 0],
            &[0, 0, 96, 241, 255, 255, 253, 144, 1, 0, 0],
            &[0, 59, 253, 169, 33, 17, 115, 255, 112, 0, 0],
            &[0, 179, 226, 6, 0, 0, 0, 179, 231, 2, 0],
            &[4, 244, 146, 0, 0, 0, 0, 99, 255, 50, 0],
            &[36, 255, 105, 0, 0, 0, 0, 59, 255, 89, 0],
            &[52, 255, 89, 0, 0, 0, 0, 41, 255, 106, 0],
            &[53, 255, 91, 0, 0, 0, 0, 39, 255, 104, 0],
            &[34, 255, 110, 0, 0, 0, 0, 58, 255, 83, 0],
            &[3, 243, 157, 0, 0, 0, 0, 104, 255, 39, 0],
            &[0, 169, 237, 17, 0, 0, 1, 198, 216, 0, 0],
            &[0, 44, 246, 200, 76, 64, 165, 255, 82, 0, 0],
            &[0, 0, 66, 218, 255, 255, 232, 97, 0, 0, 0],
            &[0, 0, 0, 0, 27, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 12, 8, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ó' / 0xd3
        'Ó' => Some(&[
            &[0, 0, 0, 0, 0, 75, 255, 131, 0, 0, 0],
            &[0, 0, 0, 0, 11, 222, 161, 1, 0, 0, 0],
            &[0, 0, 0, 0, 82, 129, 2, 0, 0, 0, 0],
            &[0, 0, 0, 16, 65, 76, 28, 0, 0, 0, 0],
            &[0, 0, 96, 241, 255, 255, 253, 144, 1, 0, 0],
            &[0, 59, 253, 169, 33, 17, 115, 255, 112, 0, 0],
            &[0, 179, 226, 6, 0, 0, 0, 179, 231, 2, 0],
            &[4, 244, 146, 0, 0, 0, 0, 99, 255, 50, 0],
            &[36, 255, 105, 0, 0, 0, 0, 59, 255, 89, 0],
            &[52, 255, 89, 0, 0, 0, 0, 41, 255, 106, 0],
            &[53, 255, 91, 0, 0, 0, 0, 39, 255, 104, 0],
            &[34, 255, 110, 0, 0, 0, 0, 58, 255, 83, 0],
            &[3, 243, 157, 0, 0, 0, 0, 104, 255, 39, 0],
            &[0, 169, 237, 17, 0, 0, 1, 198, 216, 0, 0],
            &[0, 44, 246, 200, 76, 64, 165, 255, 82, 0, 0],
            &[0, 0, 66, 218, 255, 255, 232, 97, 0, 0, 0],
            &[0, 0, 0, 0, 27, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 10, 0, 0, 0],
        ]),
        // letter: 'Ô' / 0xd4
        'Ô' => Some(&[
            &[0, 0, 0, 33, 241, 253, 62, 0, 0, 0, 0],
            &[0, 0, 14, 210, 148, 116, 233, 35, 0, 0, 0],
            &[0, 0, 105, 112, 0, 0, 86, 138, 0, 0, 0],
            &[0, 0, 0, 16, 65, 76, 28, 0, 0, 0, 0],
            &[0, 0, 96, 241, 255, 255, 253, 144, 1, 0, 0],
            &[0, 59, 253, 169, 33, 17, 115, 255, 112, 0, 0],
            &[0, 179, 226, 6, 0, 0, 0, 179, 231, 2, 0],
            &[4, 244, 146, 0, 0, 0, 0, 99, 255, 50, 0],
            &[36, 255, 105, 0, 0, 0, 0, 59, 255, 89, 0],
            &[52, 255, 89, 0, 0, 0, 0, 41, 255, 106, 0],
            &[53, 255, 91, 0, 0, 0, 0, 39, 255, 104, 0],
            &[34, 255, 110, 0, 0, 0, 0, 58, 255, 83, 0],
            &[3, 243, 157, 0, 0, 0, 0, 104, 255, 39, 0],
            &[0, 169, 237, 17, 0, 0, 1, 198, 216, 0, 0],
            &[0, 44, 246, 200, 76, 64, 165, 255, 82, 0, 0],
            &[0, 0, 66, 218, 255, 255, 232, 97, 0, 0, 0],
            &[0, 0, 0, 0, 27, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 10, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Õ' / 0xd5
        'Õ' => Some(&[
            &[0, 0, 7, 104, 90, 5, 0, 107, 13, 0, 0],
            &[0, 0, 154, 213, 225, 217, 162, 219, 1, 0, 0],
            &[0, 0, 150, 11, 6, 106, 152, 43, 0, 0, 0],
            &[0, 0, 0, 16, 65, 76, 28, 0, 0, 0, 0],
            &[0, 0, 96, 241, 255, 255, 253, 144, 1, 0, 0],
            &[0, 59, 253, 169, 33, 17, 115, 255, 112, 0, 0],
            &[0, 179, 226, 6, 0, 0, 0, 179, 231, 2, 0],
            &[4, 244, 146, 0, 0, 0, 0, 99, 255, 50, 0],
            &[36, 255, 105, 0, 0, 0, 0, 59, 255, 89, 0],
            &[52, 255, 89, 0, 0, 0, 0, 41, 255, 106, 0],
            &[53, 255, 91, 0, 0, 0, 0, 39, 255, 104, 0],
            &[34, 255, 110, 0, 0, 0, 0, 58, 255, 83, 0],
            &[3, 243, 157, 0, 0, 0, 0, 104, 255, 39, 0],
            &[0, 169, 237, 17, 0, 0, 1, 198, 216, 0, 0],
            &[0, 44, 246, 200, 76, 64, 165, 255, 82, 0, 0],
            &[0, 0, 66, 218, 255, 255, 232, 97, 0, 0, 0],
            &[0, 0, 0, 0, 27, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ö' / 0xd6
        'Ö' => Some(&[
            &[0, 0, 61, 103, 0, 0, 76, 88, 0, 0, 0],
            &[0, 0, 173, 247, 4, 0, 201, 224, 0, 0, 0],
            &[0, 0, 33, 61, 0, 0, 43, 51, 0, 0, 0],
            &[0, 0, 0, 16, 65, 76, 28, 0, 0, 0, 0],
            &[0, 0, 96, 241, 255, 255, 253, 144, 1, 0, 0],
            &[0, 59, 253, 169, 33, 17, 115, 255, 112, 0, 0],
            &[0, 179, 226, 6, 0, 0, 0, 179, 231, 2, 0],
            &[4, 244, 146, 0, 0, 0, 0, 99, 255, 50, 0],
            &[36, 255, 105, 0, 0, 0, 0, 59, 255, 89, 0],
            &[52, 255, 89, 0, 0, 0, 0, 41, 255, 106, 0],
            &[53, 255, 91, 0, 0, 0, 0, 39, 255, 104, 0],
            &[34, 255, 110, 0, 0, 0, 0, 58, 255, 83, 0],
            &[3, 243, 157, 0, 0, 0, 0, 104, 255, 39, 0],
            &[0, 169, 237, 17, 0, 0, 1, 198, 216, 0, 0],
            &[0, 44, 246, 200, 76, 64, 165, 255, 82, 0, 0],
            &[0, 0, 66, 218, 255, 255, 232, 97, 0, 0, 0],
            &[0, 0, 0, 0, 27, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '×' / 0xd7
        '×' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 93, 0, 0, 0, 0, 65, 45, 0, 0],
            &[0, 62, 247, 113, 0, 0, 67, 245, 107, 0, 0],
            &[0, 0, 71, 247, 112, 68, 246, 115, 0, 0, 0],
            &[0, 0, 0, 71, 247, 249, 115, 0, 0, 0, 0],
            &[0, 0, 0, 70, 247, 251, 114, 0, 0, 0, 0],
            &[0, 0, 68, 246, 116, 76, 248, 113, 0, 0, 0],
            &[0, 57, 246, 117, 0, 0, 73, 247, 103, 0, 0],
            &[0, 15, 94, 0, 0, 0, 0, 68, 42, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ø' / 0xd8
        'Ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 65, 76, 28, 5, 134, 19, 0],
            &[0, 0, 96, 241, 255, 255, 253, 255, 237, 16, 0],
            &[0, 59, 253, 169, 33, 17, 135, 255, 218, 0, 0],
            &[0, 179, 226, 6, 0, 0, 150, 255, 235, 2, 0],
            &[4, 244, 146, 0, 0, 44, 252, 170, 255, 50, 0],
            &[36, 255, 105, 0, 0, 186, 180, 59, 255, 89, 0],
            &[52, 255, 89, 0, 77, 250, 39, 41, 255, 106, 0],
            &[53, 255, 91, 5, 217, 144, 0, 39, 255, 104, 0],
            &[34, 255, 110, 114, 236, 17, 0, 58, 255, 83, 0],
            &[3, 243, 178, 239, 108, 0, 0, 104, 255, 39, 0],
            &[0, 169, 255, 231, 4, 0, 1, 198, 216, 0, 0],
            &[0, 89, 255, 255, 76, 64, 165, 255, 82, 0, 0],
            &[0, 187, 248, 218, 255, 255, 232, 97, 0, 0, 0],
            &[0, 88, 39, 0, 27, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ù' / 0xd9
        'Ù' => Some(&[
            &[0, 0, 6, 206, 232, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 219, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 162, 29, 0, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 18, 35, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 206, 186, 0, 0, 0, 0, 135, 255, 1, 0],
            &[0, 191, 213, 0, 0, 0, 0, 159, 242, 0, 0],
            &[0, 146, 248, 18, 0, 0, 0, 211, 195, 0, 0],
            &[0, 47, 253, 193, 71, 61, 159, 255, 89, 0, 0],
            &[0, 0, 85, 227, 255, 255, 238, 114, 0, 0, 0],
            &[0, 0, 0, 2, 27, 30, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 12, 8, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ú' / 0xda
        'Ú' => Some(&[
            &[0, 0, 0, 0, 0, 75, 255, 131, 0, 0, 0],
            &[0, 0, 0, 0, 11, 222, 161, 1, 0, 0, 0],
            &[0, 0, 0, 0, 82, 129, 2, 0, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 18, 35, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 206, 186, 0, 0, 0, 0, 135, 255, 1, 0],
            &[0, 191, 213, 0, 0, 0, 0, 159, 242, 0, 0],
            &[0, 146, 248, 18, 0, 0, 0, 211, 195, 0, 0],
            &[0, 47, 253, 193, 71, 61, 159, 255, 89, 0, 0],
            &[0, 0, 85, 227, 255, 255, 238, 114, 0, 0, 0],
            &[0, 0, 0, 2, 27, 30, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 10, 0, 0, 0],
        ]),
        // letter: 'Û' / 0xdb
        'Û' => Some(&[
            &[0, 0, 0, 33, 241, 253, 62, 0, 0, 0, 0],
            &[0, 0, 14, 210, 148, 116, 233, 35, 0, 0, 0],
            &[0, 0, 105, 112, 0, 0, 86, 138, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 18, 35, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 206, 186, 0, 0, 0, 0, 135, 255, 1, 0],
            &[0, 191, 213, 0, 0, 0, 0, 159, 242, 0, 0],
            &[0, 146, 248, 18, 0, 0, 0, 211, 195, 0, 0],
            &[0, 47, 253, 193, 71, 61, 159, 255, 89, 0, 0],
            &[0, 0, 85, 227, 255, 255, 238, 114, 0, 0, 0],
            &[0, 0, 0, 2, 27, 30, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 10, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ü' / 0xdc
        'Ü' => Some(&[
            &[0, 0, 61, 103, 0, 0, 76, 88, 0, 0, 0],
            &[0, 0, 173, 247, 4, 0, 201, 224, 0, 0, 0],
            &[0, 0, 33, 61, 0, 0, 43, 51, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 18, 35, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 206, 186, 0, 0, 0, 0, 135, 255, 1, 0],
            &[0, 191, 213, 0, 0, 0, 0, 159, 242, 0, 0],
            &[0, 146, 248, 18, 0, 0, 0, 211, 195, 0, 0],
            &[0, 47, 253, 193, 71, 61, 159, 255, 89, 0, 0],
            &[0, 0, 85, 227, 255, 255, 238, 114, 0, 0, 0],
            &[0, 0, 0, 2, 27, 30, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ý' / 0xdd
        'Ý' => Some(&[
            &[0, 0, 0, 0, 0, 75, 255, 131, 0, 0, 0],
            &[0, 0, 0, 0, 11, 222, 161, 1, 0, 0, 0],
            &[0, 0, 0, 0, 82, 129, 2, 0, 0, 0, 0],
            &[13, 35, 10, 0, 0, 0, 0, 2, 35, 20, 0],
            &[28, 248, 146, 0, 0, 0, 0, 82, 255, 71, 0],
            &[0, 146, 245, 19, 0, 0, 0, 201, 197, 0, 0],
            &[0, 25, 246, 127, 0, 0, 65, 255, 66, 0, 0],
            &[0, 0, 141, 235, 10, 0, 184, 192, 0, 0, 0],
            &[0, 0, 22, 244, 108, 48, 255, 62, 0, 0, 0],
            &[0, 0, 0, 136, 222, 170, 187, 0, 0, 0, 0],
            &[0, 0, 0, 19, 242, 255, 57, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 10, 0, 0, 0],
        ]),
        // letter: 'Þ' / 0xde
        'Þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 35, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 240, 237, 231, 201, 119, 6, 0, 0],
            &[0, 98, 255, 124, 101, 118, 182, 255, 166, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 161, 255, 29, 0],
            &[0, 98, 255, 37, 0, 0, 0, 88, 255, 61, 0],
            &[0, 98, 255, 37, 0, 0, 0, 106, 255, 39, 0],
            &[0, 98, 255, 37, 0, 0, 37, 220, 209, 0, 0],
            &[0, 98, 255, 210, 203, 220, 254, 225, 44, 0, 0],
            &[0, 98, 255, 154, 136, 121, 82, 8, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ß' / 0xdf
        'ß' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 205, 247, 248, 212, 112, 1, 0, 0],
            &[0, 83, 255, 184, 92, 85, 166, 255, 110, 0, 0],
            &[0, 167, 225, 1, 0, 0, 0, 237, 163, 0, 0],
            &[0, 185, 197, 0, 0, 0, 64, 255, 99, 0, 0],
            &[0, 185, 197, 0, 0, 82, 247, 149, 0, 0, 0],
            &[0, 185, 197, 0, 7, 245, 151, 0, 0, 0, 0],
            &[0, 185, 197, 0, 18, 254, 144, 0, 0, 0, 0],
            &[0, 185, 197, 0, 0, 125, 255, 179, 19, 0, 0],
            &[0, 185, 197, 0, 0, 0, 83, 238, 223, 23, 0],
            &[0, 185, 197, 0, 0, 0, 0, 36, 240, 147, 0],
            &[0, 185, 197, 0, 0, 0, 0, 0, 195, 188, 0],
            &[0, 185, 197, 0, 129, 65, 28, 72, 246, 145, 0],
            &[0, 185, 197, 0, 166, 254, 255, 254, 183, 17, 0],
            &[0, 0, 0, 0, 0, 12, 37, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'à' / 0xe0
        'à' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 193, 242, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 211, 168, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 163, 43, 0, 0, 0, 0],
            &[0, 0, 0, 23, 62, 58, 22, 0, 0, 0, 0],
            &[0, 8, 189, 254, 254, 253, 253, 154, 1, 0, 0],
            &[0, 0, 114, 55, 8, 6, 98, 254, 97, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 207, 154, 0, 0],
            &[0, 0, 2, 59, 106, 126, 133, 223, 171, 0, 0],
            &[0, 27, 208, 247, 185, 156, 145, 224, 171, 0, 0],
            &[0, 157, 236, 38, 0, 0, 0, 191, 171, 0, 0],
            &[0, 197, 177, 0, 0, 0, 14, 239, 171, 0, 0],
            &[0, 154, 238, 69, 24, 63, 198, 241, 171, 0, 0],
            &[0, 22, 194, 255, 255, 234, 99, 159, 171, 0, 0],
            &[0, 0, 0, 19, 31, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'á' / 0xe1
        'á' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 254, 157, 0, 0, 0],
            &[0, 0, 0, 0, 4, 204, 184, 6, 0, 0, 0],
            &[0, 0, 0, 0, 68, 147, 7, 0, 0, 0, 0],
            &[0, 0, 0, 23, 62, 58, 22, 0, 0, 0, 0],
            &[0, 8, 189, 254, 254, 253, 253, 154, 1, 0, 0],
            &[0, 0, 114, 55, 8, 6, 98, 254, 97, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 207, 154, 0, 0],
            &[0, 0, 2, 59, 106, 126, 133, 223, 171, 0, 0],
            &[0, 27, 208, 247, 185, 156, 145, 224, 171, 0, 0],
            &[0, 157, 236, 38, 0, 0, 0, 191, 171, 0, 0],
            &[0, 197, 177, 0, 0, 0, 14, 239, 171, 0, 0],
            &[0, 154, 238, 69, 24, 63, 198, 241, 171, 0, 0],
            &[0, 22, 194, 255, 255, 234, 99, 159, 171, 0, 0],
            &[0, 0, 0, 19, 31, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'â' / 0xe2
        'â' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 228, 255, 77, 0, 0, 0, 0],
            &[0, 0, 6, 191, 172, 105, 238, 44, 0, 0, 0],
            &[0, 0, 92, 131, 3, 0, 77, 153, 3, 0, 0],
            &[0, 0, 0, 23, 62, 58, 22, 0, 0, 0, 0],
            &[0, 8, 189, 254, 254, 253, 253, 154, 1, 0, 0],
            &[0, 0, 114, 55, 8, 6, 98, 254, 97, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 207, 154, 0, 0],
            &[0, 0, 2, 59, 106, 126, 133, 223, 171, 0, 0],
            &[0, 27, 208, 247, 185, 156, 145, 224, 171, 0, 0],
            &[0, 157, 236, 38, 0, 0, 0, 191, 171, 0, 0],
            &[0, 197, 177, 0, 0, 0, 14, 239, 171, 0, 0],
            &[0, 154, 238, 69, 24, 63, 198, 241, 171, 0, 0],
            &[0, 22, 194, 255, 255, 234, 99, 159, 171, 0, 0],
            &[0, 0, 0, 19, 31, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ã' / 0xe3
        'ã' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 90, 91, 6, 0, 90, 24, 0, 0],
            &[0, 0, 129, 224, 224, 220, 152, 229, 9, 0, 0],
            &[0, 0, 142, 25, 4, 103, 160, 59, 0, 0, 0],
            &[0, 0, 0, 23, 62, 58, 22, 0, 0, 0, 0],
            &[0, 8, 189, 254, 254, 253, 253, 154, 1, 0, 0],
            &[0, 0, 114, 55, 8, 6, 98, 254, 97, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 207, 154, 0, 0],
            &[0, 0, 2, 59, 106, 126, 133, 223, 171, 0, 0],
            &[0, 27, 208, 247, 185, 156, 145, 224, 171, 0, 0],
            &[0, 157, 236, 38, 0, 0, 0, 191, 171, 0, 0],
            &[0, 197, 177, 0, 0, 0, 14, 239, 171, 0, 0],
            &[0, 154, 238, 69, 24, 63, 198, 241, 171, 0, 0],
            &[0, 22, 194, 255, 255, 234, 99, 159, 171, 0, 0],
            &[0, 0, 0, 19, 31, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ä' / 0xe4
        'ä' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 47, 107, 0, 0, 60, 94, 0, 0, 0],
            &[0, 0, 152, 255, 18, 0, 179, 244, 2, 0, 0],
            &[0, 0, 29, 74, 0, 0, 38, 64, 0, 0, 0],
            &[0, 0, 0, 23, 62, 58, 22, 0, 0, 0, 0],
            &[0, 8, 189, 254, 254, 253, 253, 154, 1, 0, 0],
            &[0, 0, 114, 55, 8, 6, 98, 254, 97, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 207, 154, 0, 0],
            &[0, 0, 2, 59, 106, 126, 133, 223, 171, 0, 0],
            &[0, 27, 208, 247, 185, 156, 145, 224, 171, 0, 0],
            &[0, 157, 236, 38, 0, 0, 0, 191, 171, 0, 0],
            &[0, 197, 177, 0, 0, 0, 14, 239, 171, 0, 0],
            &[0, 154, 238, 69, 24, 63, 198, 241, 171, 0, 0],
            &[0, 22, 194, 255, 255, 234, 99, 159, 171, 0, 0],
            &[0, 0, 0, 19, 31, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'å' / 0xe5
        'å' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 191, 212, 90, 0, 0, 0, 0],
            &[0, 0, 0, 156, 121, 37, 242, 3, 0, 0, 0],
            &[0, 0, 0, 141, 159, 90, 232, 0, 0, 0, 0],
            &[0, 0, 0, 16, 141, 163, 54, 0, 0, 0, 0],
            &[0, 0, 0, 23, 62, 58, 22, 0, 0, 0, 0],
            &[0, 8, 189, 254, 254, 253, 253, 154, 1, 0, 0],
            &[0, 0, 114, 55, 8, 6, 98, 254, 97, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 207, 154, 0, 0],
            &[0, 0, 2, 59, 106, 126, 133, 223, 171, 0, 0],
            &[0, 27, 208, 247, 185, 156, 145, 224, 171, 0, 0],
            &[0, 157, 236, 38, 0, 0, 0, 191, 171, 0, 0],
            &[0, 197, 177, 0, 0, 0, 14, 239, 171, 0, 0],
            &[0, 154, 238, 69, 24, 63, 198, 241, 171, 0, 0],
            &[0, 22, 194, 255, 255, 234, 99, 159, 171, 0, 0],
            &[0, 0, 0, 19, 31, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'æ' / 0xe6
        'æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 40, 60, 7, 0, 28, 61, 9, 0, 0],
            &[1, 193, 255, 254, 218, 97, 251, 254, 219, 20, 0],
            &[0, 83, 23, 23, 229, 255, 93, 25, 233, 129, 0],
            &[0, 0, 0, 0, 167, 226, 0, 0, 161, 192, 0],
            &[0, 16, 91, 115, 203, 223, 116, 116, 194, 215, 0],
            &[28, 230, 233, 172, 220, 235, 187, 187, 187, 161, 0],
            &[133, 233, 16, 0, 169, 193, 0, 0, 0, 0, 0],
            &[161, 187, 0, 0, 201, 239, 6, 0, 0, 0, 0],
            &[127, 235, 53, 107, 231, 218, 153, 38, 57, 98, 0],
            &[22, 216, 255, 237, 70, 58, 230, 255, 252, 117, 0],
            &[0, 3, 33, 7, 0, 0, 4, 34, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ç' / 0xe7
        'ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 62, 55, 17, 0, 0, 0],
            &[0, 0, 27, 184, 254, 255, 255, 250, 94, 0, 0],
            &[0, 2, 205, 238, 91, 23, 26, 87, 34, 0, 0],
            &[0, 66, 255, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 118, 255, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 136, 255, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 119, 255, 26, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 255, 104, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 201, 243, 110, 46, 52, 94, 74, 0, 0],
            &[0, 0, 22, 169, 249, 255, 255, 239, 90, 0, 0],
            &[0, 0, 0, 0, 33, 255, 33, 3, 0, 0, 0],
            &[0, 0, 0, 0, 56, 200, 192, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 30, 255, 47, 0, 0, 0],
            &[0, 0, 0, 0, 15, 3, 0, 0, 0, 0, 0],
        ]),
        // letter: 'è' / 0xe8
        'è' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 197, 240, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 20, 213, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 164, 40, 0, 0, 0, 0],
            &[0, 0, 0, 2, 46, 63, 27, 0, 0, 0, 0],
            &[0, 0, 67, 223, 255, 252, 253, 161, 8, 0, 0],
            &[0, 44, 248, 178, 30, 1, 77, 248, 139, 0, 0],
            &[0, 163, 231, 10, 0, 0, 0, 148, 240, 5, 0],
            &[0, 225, 213, 125, 125, 125, 125, 178, 255, 33, 0],
            &[0, 244, 227, 187, 187, 187, 187, 187, 187, 33, 0],
            &[0, 223, 170, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 155, 239, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 239, 204, 74, 30, 46, 95, 136, 0, 0],
            &[0, 0, 39, 183, 252, 255, 255, 233, 127, 0, 0],
            &[0, 0, 0, 0, 11, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'é' / 0xe9
        'é' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 55, 254, 153, 0, 0, 0],
            &[0, 0, 0, 0, 5, 207, 181, 5, 0, 0, 0],
            &[0, 0, 0, 0, 71, 145, 6, 0, 0, 0, 0],
            &[0, 0, 0, 2, 46, 63, 27, 0, 0, 0, 0],
            &[0, 0, 67, 223, 255, 252, 253, 161, 8, 0, 0],
            &[0, 44, 248, 178, 30, 1, 77, 248, 139, 0, 0],
            &[0, 163, 231, 10, 0, 0, 0, 148, 240, 5, 0],
            &[0, 225, 213, 125, 125, 125, 125, 178, 255, 33, 0],
            &[0, 244, 227, 187, 187, 187, 187, 187, 187, 33, 0],
            &[0, 223, 170, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 155, 239, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 239, 204, 74, 30, 46, 95, 136, 0, 0],
            &[0, 0, 39, 183, 252, 255, 255, 233, 127, 0, 0],
            &[0, 0, 0, 0, 11, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ê' / 0xea
        'ê' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 231, 255, 73, 0, 0, 0, 0],
            &[0, 0, 7, 194, 169, 109, 237, 42, 0, 0, 0],
            &[0, 0, 95, 129, 2, 0, 80, 151, 2, 0, 0],
            &[0, 0, 0, 2, 46, 63, 27, 0, 0, 0, 0],
            &[0, 0, 67, 223, 255, 252, 253, 161, 8, 0, 0],
            &[0, 44, 248, 178, 30, 1, 77, 248, 139, 0, 0],
            &[0, 163, 231, 10, 0, 0, 0, 148, 240, 5, 0],
            &[0, 225, 213, 125, 125, 125, 125, 178, 255, 33, 0],
            &[0, 244, 227, 187, 187, 187, 187, 187, 187, 33, 0],
            &[0, 223, 170, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 155, 239, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 239, 204, 74, 30, 46, 95, 136, 0, 0],
            &[0, 0, 39, 183, 252, 255, 255, 233, 127, 0, 0],
            &[0, 0, 0, 0, 11, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ë' / 0xeb
        'ë' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 105, 0, 0, 62, 92, 0, 0, 0],
            &[0, 0, 156, 254, 15, 0, 184, 240, 1, 0, 0],
            &[0, 0, 30, 73, 0, 0, 40, 63, 0, 0, 0],
            &[0, 0, 0, 2, 46, 63, 27, 0, 0, 0, 0],
            &[0, 0, 67, 223, 255, 252, 253, 161, 8, 0, 0],
            &[0, 44, 248, 178, 30, 1, 77, 248, 139, 0, 0],
            &[0, 163, 231, 10, 0, 0, 0, 148, 240, 5, 0],
            &[0, 225, 213, 125, 125, 125, 125, 178, 255, 33, 0],
            &[0, 244, 227, 187, 187, 187, 187, 187, 187, 33, 0],
            &[0, 223, 170, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 155, 239, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 239, 204, 74, 30, 46, 95, 136, 0, 0],
            &[0, 0, 39, 183, 252, 255, 255, 233, 127, 0, 0],
            &[0, 0, 0, 0, 11, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ì' / 0xec
        'ì' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 209, 230, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 222, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 168, 29, 0, 0, 0, 0],
            &[0, 10, 28, 28, 28, 28, 3, 0, 0, 0, 0],
            &[0, 89, 254, 255, 255, 255, 35, 0, 0, 0, 0],
            &[0, 0, 3, 20, 122, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 7, 28, 48, 139, 255, 87, 41, 19, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'í' / 0xed
        'í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 135, 0, 0, 0],
            &[0, 0, 0, 0, 10, 219, 167, 2, 0, 0, 0],
            &[0, 0, 0, 0, 84, 136, 3, 0, 0, 0, 0],
            &[0, 10, 28, 28, 28, 28, 3, 0, 0, 0, 0],
            &[0, 89, 254, 255, 255, 255, 35, 0, 0, 0, 0],
            &[0, 0, 3, 20, 122, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 7, 28, 48, 139, 255, 87, 41, 19, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'î' / 0xee
        'î' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 239, 252, 58, 0, 0, 0, 0],
            &[0, 0, 12, 207, 154, 122, 231, 32, 0, 0, 0],
            &[0, 0, 107, 119, 0, 0, 92, 141, 0, 0, 0],
            &[0, 10, 28, 28, 28, 28, 3, 0, 0, 0, 0],
            &[0, 89, 254, 255, 255, 255, 35, 0, 0, 0, 0],
            &[0, 0, 3, 20, 122, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 7, 28, 48, 139, 255, 87, 41, 19, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ï' / 0xef
        'ï' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 97, 0, 0, 71, 83, 0, 0, 0],
            &[0, 0, 174, 247, 4, 0, 201, 224, 0, 0, 0],
            &[0, 0, 36, 67, 0, 0, 47, 56, 0, 0, 0],
            &[0, 10, 28, 28, 28, 28, 3, 0, 0, 0, 0],
            &[0, 89, 254, 255, 255, 255, 35, 0, 0, 0, 0],
            &[0, 0, 3, 20, 122, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 7, 28, 48, 139, 255, 87, 41, 19, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ð' / 0xf0
        'ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 154, 138, 10, 50, 182, 31, 0, 0],
            &[0, 0, 0, 73, 246, 255, 255, 134, 12, 0, 0],
            &[0, 0, 6, 118, 255, 255, 255, 27, 0, 0, 0],
            &[0, 0, 18, 186, 67, 5, 195, 195, 1, 0, 0],
            &[0, 0, 0, 0, 2, 2, 23, 242, 90, 0, 0],
            &[0, 0, 78, 205, 251, 252, 197, 188, 197, 0, 0],
            &[0, 79, 255, 160, 61, 52, 127, 251, 253, 15, 0],
            &[0, 206, 196, 0, 0, 0, 0, 143, 255, 52, 0],
            &[4, 253, 127, 0, 0, 0, 0, 79, 255, 63, 0],
            &[5, 253, 124, 0, 0, 0, 0, 79, 255, 43, 0],
            &[0, 208, 188, 0, 0, 0, 0, 143, 237, 3, 0],
            &[0, 81, 255, 140, 35, 24, 102, 248, 118, 0, 0],
            &[0, 0, 87, 222, 255, 255, 240, 121, 1, 0, 0],
            &[0, 0, 0, 1, 26, 34, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ñ' / 0xf1
        'ñ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 99, 12, 0, 75, 39, 0, 0],
            &[0, 0, 98, 233, 216, 231, 150, 232, 27, 0, 0],
            &[0, 0, 121, 47, 1, 89, 160, 77, 0, 0, 0],
            &[0, 18, 17, 0, 18, 59, 46, 4, 0, 0, 0],
            &[0, 163, 175, 116, 247, 253, 255, 213, 32, 0, 0],
            &[0, 163, 245, 199, 38, 2, 61, 247, 168, 0, 0],
            &[0, 163, 255, 50, 0, 0, 0, 177, 220, 0, 0],
            &[0, 163, 248, 0, 0, 0, 0, 155, 236, 0, 0],
            &[0, 163, 229, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ò' / 0xf2
        'ò' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 209, 230, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 222, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 168, 29, 0, 0, 0, 0],
            &[0, 0, 0, 10, 55, 59, 15, 0, 0, 0, 0],
            &[0, 0, 106, 240, 255, 253, 247, 136, 3, 0, 0],
            &[0, 81, 255, 142, 19, 6, 102, 252, 125, 0, 0],
            &[0, 197, 211, 0, 0, 0, 0, 161, 238, 6, 0],
            &[1, 249, 141, 0, 0, 0, 0, 93, 255, 44, 0],
            &[11, 255, 120, 0, 0, 0, 0, 76, 255, 62, 0],
            &[1, 247, 142, 0, 0, 0, 0, 99, 255, 44, 0],
            &[0, 185, 218, 3, 0, 0, 0, 176, 235, 3, 0],
            &[0, 61, 252, 167, 42, 32, 132, 255, 113, 0, 0],
            &[0, 0, 77, 220, 255, 255, 239, 118, 0, 0, 0],
            &[0, 0, 0, 0, 28, 35, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ó' / 0xf3
        'ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 135, 0, 0, 0],
            &[0, 0, 0, 0, 10, 219, 167, 2, 0, 0, 0],
            &[0, 0, 0, 0, 84, 136, 3, 0, 0, 0, 0],
            &[0, 0, 0, 10, 55, 59, 15, 0, 0, 0, 0],
            &[0, 0, 106, 240, 255, 253, 247, 136, 3, 0, 0],
            &[0, 81, 255, 142, 19, 6, 102, 252, 125, 0, 0],
            &[0, 197, 211, 0, 0, 0, 0, 161, 238, 6, 0],
            &[1, 249, 141, 0, 0, 0, 0, 93, 255, 44, 0],
            &[11, 255, 120, 0, 0, 0, 0, 76, 255, 62, 0],
            &[1, 247, 142, 0, 0, 0, 0, 99, 255, 44, 0],
            &[0, 185, 218, 3, 0, 0, 0, 176, 235, 3, 0],
            &[0, 61, 252, 167, 42, 32, 132, 255, 113, 0, 0],
            &[0, 0, 77, 220, 255, 255, 239, 118, 0, 0, 0],
            &[0, 0, 0, 0, 28, 35, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ô' / 0xf4
        'ô' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 239, 252, 58, 0, 0, 0, 0],
            &[0, 0, 12, 207, 154, 122, 231, 32, 0, 0, 0],
            &[0, 0, 107, 119, 0, 0, 92, 141, 0, 0, 0],
            &[0, 0, 0, 10, 55, 59, 15, 0, 0, 0, 0],
            &[0, 0, 106, 240, 255, 253, 247, 136, 3, 0, 0],
            &[0, 81, 255, 142, 19, 6, 102, 252, 125, 0, 0],
            &[0, 197, 211, 0, 0, 0, 0, 161, 238, 6, 0],
            &[1, 249, 141, 0, 0, 0, 0, 93, 255, 44, 0],
            &[11, 255, 120, 0, 0, 0, 0, 76, 255, 62, 0],
            &[1, 247, 142, 0, 0, 0, 0, 99, 255, 44, 0],
            &[0, 185, 218, 3, 0, 0, 0, 176, 235, 3, 0],
            &[0, 61, 252, 167, 42, 32, 132, 255, 113, 0, 0],
            &[0, 0, 77, 220, 255, 255, 239, 118, 0, 0, 0],
            &[0, 0, 0, 0, 28, 35, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'õ' / 0xf5
        'õ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 97, 83, 3, 0, 101, 13, 0, 0],
            &[0, 0, 151, 219, 230, 212, 155, 221, 1, 0, 0],
            &[0, 0, 155, 13, 8, 113, 158, 48, 0, 0, 0],
            &[0, 0, 0, 10, 55, 59, 15, 0, 0, 0, 0],
            &[0, 0, 106, 240, 255, 253, 247, 136, 3, 0, 0],
            &[0, 81, 255, 142, 19, 6, 102, 252, 125, 0, 0],
            &[0, 197, 211, 0, 0, 0, 0, 161, 238, 6, 0],
            &[1, 249, 141, 0, 0, 0, 0, 93, 255, 44, 0],
            &[11, 255, 120, 0, 0, 0, 0, 76, 255, 62, 0],
            &[1, 247, 142, 0, 0, 0, 0, 99, 255, 44, 0],
            &[0, 185, 218, 3, 0, 0, 0, 176, 235, 3, 0],
            &[0, 61, 252, 167, 42, 32, 132, 255, 113, 0, 0],
            &[0, 0, 77, 220, 255, 255, 239, 118, 0, 0, 0],
            &[0, 0, 0, 0, 28, 35, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ö' / 0xf6
        'ö' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 97, 0, 0, 71, 83, 0, 0, 0],
            &[0, 0, 174, 247, 4, 0, 201, 224, 0, 0, 0],
            &[0, 0, 36, 67, 0, 0, 47, 56, 0, 0, 0],
            &[0, 0, 0, 10, 55, 59, 15, 0, 0, 0, 0],
            &[0, 0, 106, 240, 255, 253, 247, 136, 3, 0, 0],
            &[0, 81, 255, 142, 19, 6, 102, 252, 125, 0, 0],
            &[0, 197, 211, 0, 0, 0, 0, 161, 238, 6, 0],
            &[1, 249, 141, 0, 0, 0, 0, 93, 255, 44, 0],
            &[11, 255, 120, 0, 0, 0, 0, 76, 255, 62, 0],
            &[1, 247, 142, 0, 0, 0, 0, 99, 255, 44, 0],
            &[0, 185, 218, 3, 0, 0, 0, 176, 235, 3, 0],
            &[0, 61, 252, 167, 42, 32, 132, 255, 113, 0, 0],
            &[0, 0, 77, 220, 255, 255, 239, 118, 0, 0, 0],
            &[0, 0, 0, 0, 28, 35, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '÷' / 0xf7
        '÷' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 175, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 182, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 10, 0, 0, 0, 0, 0],
            &[0, 97, 152, 152, 152, 152, 152, 152, 128, 0, 0],
            &[0, 100, 156, 156, 156, 156, 156, 156, 131, 0, 0],
            &[0, 0, 0, 0, 2, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 181, 220, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 143, 180, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ø' / 0xf8
        'ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 55, 59, 15, 26, 85, 0, 0],
            &[0, 0, 106, 240, 255, 253, 249, 255, 186, 0, 0],
            &[0, 81, 255, 142, 19, 6, 219, 255, 149, 0, 0],
            &[0, 197, 211, 0, 0, 50, 248, 236, 238, 6, 0],
            &[1, 249, 141, 0, 10, 216, 147, 93, 255, 44, 0],
            &[11, 255, 120, 0, 153, 211, 8, 76, 255, 62, 0],
            &[1, 247, 142, 80, 247, 45, 0, 99, 255, 44, 0],
            &[0, 185, 245, 240, 110, 0, 0, 176, 235, 3, 0],
            &[0, 62, 255, 255, 43, 32, 132, 255, 113, 0, 0],
            &[0, 114, 255, 243, 255, 255, 239, 118, 0, 0, 0],
            &[0, 29, 58, 0, 28, 35, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ù' / 0xf9
        'ù' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 209, 230, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 222, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 168, 29, 0, 0, 0, 0],
            &[0, 20, 23, 0, 0, 0, 0, 19, 24, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 177, 214, 0, 0],
            &[0, 185, 207, 0, 0, 0, 0, 194, 214, 0, 0],
            &[0, 166, 233, 1, 0, 0, 11, 242, 214, 0, 0],
            &[0, 107, 255, 129, 31, 43, 174, 243, 214, 0, 0],
            &[0, 5, 163, 254, 255, 249, 143, 122, 214, 0, 0],
            &[0, 0, 0, 10, 35, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ú' / 0xfa
        'ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 135, 0, 0, 0],
            &[0, 0, 0, 0, 10, 219, 167, 2, 0, 0, 0],
            &[0, 0, 0, 0, 84, 136, 3, 0, 0, 0, 0],
            &[0, 20, 23, 0, 0, 0, 0, 19, 24, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 177, 214, 0, 0],
            &[0, 185, 207, 0, 0, 0, 0, 194, 214, 0, 0],
            &[0, 166, 233, 1, 0, 0, 11, 242, 214, 0, 0],
            &[0, 107, 255, 129, 31, 43, 174, 243, 214, 0, 0],
            &[0, 5, 163, 254, 255, 249, 143, 122, 214, 0, 0],
            &[0, 0, 0, 10, 35, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'û' / 0xfb
        'û' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 239, 252, 58, 0, 0, 0, 0],
            &[0, 0, 12, 207, 154, 122, 231, 32, 0, 0, 0],
            &[0, 0, 107, 119, 0, 0, 92, 141, 0, 0, 0],
            &[0, 20, 23, 0, 0, 0, 0, 19, 24, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 177, 214, 0, 0],
            &[0, 185, 207, 0, 0, 0, 0, 194, 214, 0, 0],
            &[0, 166, 233, 1, 0, 0, 11, 242, 214, 0, 0],
            &[0, 107, 255, 129, 31, 43, 174, 243, 214, 0, 0],
            &[0, 5, 163, 254, 255, 249, 143, 122, 214, 0, 0],
            &[0, 0, 0, 10, 35, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ü' / 0xfc
        'ü' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 97, 0, 0, 71, 83, 0, 0, 0],
            &[0, 0, 174, 247, 4, 0, 201, 224, 0, 0, 0],
            &[0, 0, 36, 67, 0, 0, 47, 56, 0, 0, 0],
            &[0, 20, 23, 0, 0, 0, 0, 19, 24, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 177, 214, 0, 0],
            &[0, 185, 207, 0, 0, 0, 0, 194, 214, 0, 0],
            &[0, 166, 233, 1, 0, 0, 11, 242, 214, 0, 0],
            &[0, 107, 255, 129, 31, 43, 174, 243, 214, 0, 0],
            &[0, 5, 163, 254, 255, 249, 143, 122, 214, 0, 0],
            &[0, 0, 0, 10, 35, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ý' / 0xfd
        'ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 135, 0, 0, 0],
            &[0, 0, 0, 0, 10, 219, 167, 2, 0, 0, 0],
            &[0, 0, 0, 0, 84, 136, 3, 0, 0, 0, 0],
            &[6, 28, 8, 0, 0, 0, 0, 4, 28, 11, 0],
            &[10, 239, 130, 0, 0, 0, 0, 89, 255, 48, 0],
            &[0, 144, 229, 4, 0, 0, 0, 175, 209, 0, 0],
            &[0, 39, 254, 82, 0, 0, 13, 247, 114, 0, 0],
            &[0, 0, 187, 186, 0, 0, 90, 251, 23, 0, 0],
            &[0, 0, 81, 254, 36, 0, 176, 180, 0, 0, 0],
            &[0, 0, 3, 226, 135, 12, 247, 85, 0, 0, 0],
            &[0, 0, 0, 124, 226, 88, 238, 7, 0, 0, 0],
            &[0, 0, 0, 23, 249, 213, 151, 0, 0, 0, 0],
            &[0, 0, 0, 0, 167, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 240, 115, 0, 0, 0, 0, 0],
            &[0, 20, 29, 169, 238, 14, 0, 0, 0, 0, 0],
            &[0, 22, 35, 5, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'þ' / 0xfe
        'þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 150, 205, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 223, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 223, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 223, 0, 26, 63, 40, 0, 0, 0, 0],
            &[0, 163, 216, 136, 253, 253, 255, 188, 16, 0, 0],
            &[0, 163, 251, 186, 28, 6, 81, 245, 160, 0, 0],
            &[0, 163, 255, 34, 0, 0, 0, 151, 249, 11, 0],
            &[0, 163, 239, 0, 0, 0, 0, 97, 255, 51, 0],
            &[0, 163, 227, 0, 0, 0, 0, 82, 255, 63, 0],
            &[0, 163, 247, 0, 0, 0, 0, 102, 255, 45, 0],
            &[0, 163, 255, 56, 0, 0, 0, 168, 242, 5, 0],
            &[0, 163, 248, 211, 60, 33, 114, 252, 138, 0, 0],
            &[0, 163, 214, 106, 241, 255, 251, 154, 6, 0, 0],
            &[0, 163, 223, 0, 4, 33, 13, 0, 0, 0, 0],
            &[0, 163, 223, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 223, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 17, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ÿ' / 0xff
        'ÿ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 97, 0, 0, 71, 83, 0, 0, 0],
            &[0, 0, 174, 247, 4, 0, 201, 224, 0, 0, 0],
            &[0, 0, 36, 67, 0, 0, 47, 56, 0, 0, 0],
            &[6, 28, 8, 0, 0, 0, 0, 4, 28, 11, 0],
            &[10, 239, 130, 0, 0, 0, 0, 89, 255, 48, 0],
            &[0, 144, 229, 4, 0, 0, 0, 175, 209, 0, 0],
            &[0, 39, 254, 82, 0, 0, 13, 247, 114, 0, 0],
            &[0, 0, 187, 186, 0, 0, 90, 251, 23, 0, 0],
            &[0, 0, 81, 254, 36, 0, 176, 180, 0, 0, 0],
            &[0, 0, 3, 226, 135, 12, 247, 85, 0, 0, 0],
            &[0, 0, 0, 124, 226, 88, 238, 7, 0, 0, 0],
            &[0, 0, 0, 23, 249, 213, 151, 0, 0, 0, 0],
            &[0, 0, 0, 0, 167, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 240, 115, 0, 0, 0, 0, 0],
            &[0, 20, 29, 169, 238, 14, 0, 0, 0, 0, 0],
            &[0, 22, 35, 5, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ā' / 0x100
        'Ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 189, 189, 189, 189, 116, 0, 0, 0],
            &[0, 0, 51, 119, 119, 119, 119, 73, 0, 0, 0],
            &[0, 0, 0, 0, 36, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 236, 255, 32, 0, 0, 0, 0],
            &[0, 0, 0, 67, 254, 236, 114, 0, 0, 0, 0],
            &[0, 0, 0, 149, 210, 162, 197, 0, 0, 0, 0],
            &[0, 0, 2, 230, 134, 87, 254, 26, 0, 0, 0],
            &[0, 0, 59, 255, 55, 14, 249, 108, 0, 0, 0],
            &[0, 0, 142, 230, 1, 0, 184, 191, 0, 0, 0],
            &[0, 0, 224, 171, 21, 21, 126, 252, 21, 0, 0],
            &[0, 52, 255, 255, 255, 255, 255, 255, 101, 0, 0],
            &[0, 134, 255, 77, 70, 70, 70, 229, 184, 0, 0],
            &[0, 217, 169, 0, 0, 0, 0, 122, 250, 17, 0],
            &[44, 255, 90, 0, 0, 0, 0, 43, 255, 95, 0],
            &[127, 250, 16, 0, 0, 0, 0, 0, 219, 178, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ā' / 0x101
        'ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 182, 182, 182, 182, 127, 0, 0, 0],
            &[0, 0, 43, 126, 126, 126, 126, 88, 0, 0, 0],
            &[0, 0, 0, 23, 62, 58, 22, 0, 0, 0, 0],
            &[0, 8, 189, 254, 254, 253, 253, 154, 1, 0, 0],
            &[0, 0, 114, 55, 8, 6, 98, 254, 97, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 207, 154, 0, 0],
            &[0, 0, 2, 59, 106, 126, 133, 223, 171, 0, 0],
            &[0, 27, 208, 247, 185, 156, 145, 224, 171, 0, 0],
            &[0, 157, 236, 38, 0, 0, 0, 191, 171, 0, 0],
            &[0, 197, 177, 0, 0, 0, 14, 239, 171, 0, 0],
            &[0, 154, 238, 69, 24, 63, 198, 241, 171, 0, 0],
            &[0, 22, 194, 255, 255, 234, 99, 159, 171, 0, 0],
            &[0, 0, 0, 19, 31, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ă' / 0x102
        'Ă' => Some(&[
            &[0, 0, 75, 90, 0, 0, 59, 104, 0, 0, 0],
            &[0, 0, 52, 240, 132, 126, 226, 95, 0, 0, 0],
            &[0, 0, 0, 60, 151, 154, 80, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 236, 255, 32, 0, 0, 0, 0],
            &[0, 0, 0, 67, 254, 236, 114, 0, 0, 0, 0],
            &[0, 0, 0, 149, 210, 162, 197, 0, 0, 0, 0],
            &[0, 0, 2, 230, 134, 87, 254, 26, 0, 0, 0],
            &[0, 0, 59, 255, 55, 14, 249, 108, 0, 0, 0],
            &[0, 0, 142, 230, 1, 0, 184, 191, 0, 0, 0],
            &[0, 0, 224, 171, 21, 21, 126, 252, 21, 0, 0],
            &[0, 52, 255, 255, 255, 255, 255, 255, 101, 0, 0],
            &[0, 134, 255, 77, 70, 70, 70, 229, 184, 0, 0],
            &[0, 217, 169, 0, 0, 0, 0, 122, 250, 17, 0],
            &[44, 255, 90, 0, 0, 0, 0, 43, 255, 95, 0],
            &[127, 250, 16, 0, 0, 0, 0, 0, 219, 178, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ă' / 0x103
        'ă' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 98, 0, 0, 43, 112, 0, 0, 0],
            &[0, 0, 38, 240, 133, 116, 214, 121, 0, 0, 0],
            &[0, 0, 0, 55, 154, 164, 97, 2, 0, 0, 0],
            &[0, 0, 0, 23, 62, 58, 22, 0, 0, 0, 0],
            &[0, 8, 189, 254, 254, 253, 253, 154, 1, 0, 0],
            &[0, 0, 114, 55, 8, 6, 98, 254, 97, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 207, 154, 0, 0],
            &[0, 0, 2, 59, 106, 126, 133, 223, 171, 0, 0],
            &[0, 27, 208, 247, 185, 156, 145, 224, 171, 0, 0],
            &[0, 157, 236, 38, 0, 0, 0, 191, 171, 0, 0],
            &[0, 197, 177, 0, 0, 0, 14, 239, 171, 0, 0],
            &[0, 154, 238, 69, 24, 63, 198, 241, 171, 0, 0],
            &[0, 22, 194, 255, 255, 234, 99, 159, 171, 0, 0],
            &[0, 0, 0, 19, 31, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ą' / 0x104
        'Ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 236, 255, 32, 0, 0, 0, 0],
            &[0, 0, 0, 67, 254, 236, 114, 0, 0, 0, 0],
            &[0, 0, 0, 149, 210, 162, 197, 0, 0, 0, 0],
            &[0, 0, 2, 230, 134, 87, 254, 26, 0, 0, 0],
            &[0, 0, 59, 255, 55, 14, 249, 108, 0, 0, 0],
            &[0, 0, 142, 230, 1, 0, 184, 191, 0, 0, 0],
            &[0, 0, 224, 171, 21, 21, 126, 252, 21, 0, 0],
            &[0, 52, 255, 255, 255, 255, 255, 255, 101, 0, 0],
            &[0, 134, 255, 77, 70, 70, 70, 229, 184, 0, 0],
            &[0, 217, 169, 0, 0, 0, 0, 122, 250, 17, 0],
            &[44, 255, 90, 0, 0, 0, 0, 43, 255, 95, 0],
            &[127, 250, 16, 0, 0, 0, 0, 0, 226, 207, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 184, 108, 0],
            &[0, 0, 0, 0, 0, 0, 0, 85, 216, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 230, 80, 24],
            &[0, 0, 0, 0, 0, 0, 0, 16, 152, 181, 45],
        ]),
        // letter: 'ą' / 0x105
        'ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 62, 58, 22, 0, 0, 0, 0],
            &[0, 8, 189, 254, 254, 253, 253, 154, 1, 0, 0],
            &[0, 0, 114, 55, 8, 6, 98, 254, 97, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 207, 154, 0, 0],
            &[0, 0, 2, 59, 106, 126, 133, 223, 171, 0, 0],
            &[0, 27, 208, 247, 185, 156, 145, 224, 171, 0, 0],
            &[0, 157, 236, 38, 0, 0, 0, 191, 171, 0, 0],
            &[0, 197, 177, 0, 0, 0, 14, 239, 171, 0, 0],
            &[0, 154, 238, 69, 24, 63, 198, 241, 171, 0, 0],
            &[0, 22, 194, 255, 255, 234, 99, 182, 184, 0, 0],
            &[0, 0, 0, 19, 31, 3, 30, 218, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 160, 141, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 190, 184, 76, 0, 0],
            &[0, 0, 0, 0, 0, 0, 50, 172, 172, 0, 0],
        ]),
        // letter: 'Ć' / 0x106
        'Ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 74, 255, 131, 0, 0],
            &[0, 0, 0, 0, 0, 11, 222, 162, 1, 0, 0],
            &[0, 0, 0, 0, 0, 81, 130, 2, 0, 0, 0],
            &[0, 0, 0, 0, 11, 60, 70, 43, 3, 0, 0],
            &[0, 0, 2, 132, 246, 255, 255, 255, 229, 37, 0],
            &[0, 0, 150, 254, 138, 39, 22, 56, 123, 1, 0],
            &[0, 51, 255, 134, 0, 0, 0, 0, 0, 0, 0],
            &[0, 143, 250, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 196, 206, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 217, 181, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 213, 185, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 191, 211, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 253, 22, 0, 0, 0, 0, 0, 0, 0],
            &[0, 45, 253, 152, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 141, 255, 175, 86, 70, 101, 135, 0, 0],
            &[0, 0, 0, 113, 230, 255, 255, 252, 167, 0, 0],
            &[0, 0, 0, 0, 0, 21, 35, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 10, 0, 0],
        ]),
        // letter: 'ć' / 0x107
        'ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 5, 1, 0, 0],
            &[0, 0, 0, 0, 0, 1, 203, 232, 25, 0, 0],
            &[0, 0, 0, 0, 0, 108, 240, 50, 0, 0, 0],
            &[0, 0, 0, 0, 8, 166, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 62, 55, 17, 0, 0, 0],
            &[0, 0, 27, 184, 254, 255, 255, 250, 94, 0, 0],
            &[0, 2, 205, 238, 91, 23, 26, 87, 34, 0, 0],
            &[0, 66, 255, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 118, 255, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 136, 255, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 119, 255, 26, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 255, 104, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 201, 243, 110, 46, 52, 94, 74, 0, 0],
            &[0, 0, 22, 169, 249, 255, 255, 239, 90, 0, 0],
            &[0, 0, 0, 0, 10, 35, 25, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĉ' / 0x108
        'Ĉ' => Some(&[
            &[0, 0, 0, 0, 32, 240, 253, 63, 0, 0, 0],
            &[0, 0, 0, 14, 210, 149, 115, 233, 35, 0, 0],
            &[0, 0, 0, 104, 112, 0, 0, 85, 139, 0, 0],
            &[0, 0, 0, 0, 11, 60, 70, 43, 3, 0, 0],
            &[0, 0, 2, 132, 246, 255, 255, 255, 229, 37, 0],
            &[0, 0, 150, 254, 138, 39, 22, 56, 123, 1, 0],
            &[0, 51, 255, 134, 0, 0, 0, 0, 0, 0, 0],
            &[0, 143, 250, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 196, 206, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 217, 181, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 213, 185, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 191, 211, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 253, 22, 0, 0, 0, 0, 0, 0, 0],
            &[0, 45, 253, 152, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 141, 255, 175, 86, 70, 101, 135, 0, 0],
            &[0, 0, 0, 113, 230, 255, 255, 252, 167, 0, 0],
            &[0, 0, 0, 0, 0, 21, 35, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 10, 0, 0, 0, 0],
        ]),
        // letter: 'ĉ' / 0x109
        'ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 5, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 255, 176, 0, 0, 0, 0],
            &[0, 0, 0, 98, 233, 76, 223, 129, 0, 0, 0],
            &[0, 0, 26, 166, 34, 0, 23, 162, 48, 0, 0],
            &[0, 0, 0, 0, 25, 62, 55, 17, 0, 0, 0],
            &[0, 0, 27, 184, 254, 255, 255, 250, 94, 0, 0],
            &[0, 2, 205, 238, 91, 23, 26, 87, 34, 0, 0],
            &[0, 66, 255, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 118, 255, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 136, 255, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 119, 255, 26, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 255, 104, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 201, 243, 110, 46, 52, 94, 74, 0, 0],
            &[0, 0, 22, 169, 249, 255, 255, 239, 90, 0, 0],
            &[0, 0, 0, 0, 10, 35, 25, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ċ' / 0x10a
        'Ċ' => Some(&[
            &[0, 0, 0, 0, 0, 159, 197, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 212, 244, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 60, 70, 43, 3, 0, 0],
            &[0, 0, 2, 132, 246, 255, 255, 255, 229, 37, 0],
            &[0, 0, 150, 254, 138, 39, 22, 56, 123, 1, 0],
            &[0, 51, 255, 134, 0, 0, 0, 0, 0, 0, 0],
            &[0, 143, 250, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 196, 206, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 217, 181, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 213, 185, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 191, 211, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 253, 22, 0, 0, 0, 0, 0, 0, 0],
            &[0, 45, 253, 152, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 141, 255, 175, 86, 70, 101, 135, 0, 0],
            &[0, 0, 0, 113, 230, 255, 255, 252, 167, 0, 0],
            &[0, 0, 0, 0, 0, 21, 35, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ċ' / 0x10b
        'ċ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 212, 78, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 124, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 62, 55, 17, 0, 0, 0],
            &[0, 0, 27, 184, 254, 255, 255, 250, 94, 0, 0],
            &[0, 2, 205, 238, 91, 23, 26, 87, 34, 0, 0],
            &[0, 66, 255, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 118, 255, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 136, 255, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 119, 255, 26, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 255, 104, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 201, 243, 110, 46, 52, 94, 74, 0, 0],
            &[0, 0, 22, 169, 249, 255, 255, 239, 90, 0, 0],
            &[0, 0, 0, 0, 10, 35, 25, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Č' / 0x10c
        'Č' => Some(&[
            &[0, 0, 0, 115, 210, 34, 16, 184, 165, 0, 0],
            &[0, 0, 0, 0, 144, 229, 210, 183, 4, 0, 0],
            &[0, 0, 0, 0, 4, 153, 167, 18, 0, 0, 0],
            &[0, 0, 0, 0, 11, 60, 70, 43, 3, 0, 0],
            &[0, 0, 2, 132, 246, 255, 255, 255, 229, 37, 0],
            &[0, 0, 150, 254, 138, 39, 22, 56, 123, 1, 0],
            &[0, 51, 255, 134, 0, 0, 0, 0, 0, 0, 0],
            &[0, 143, 250, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 196, 206, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 217, 181, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 213, 185, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 191, 211, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 253, 22, 0, 0, 0, 0, 0, 0, 0],
            &[0, 45, 253, 152, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 141, 255, 175, 86, 70, 101, 135, 0, 0],
            &[0, 0, 0, 113, 230, 255, 255, 252, 167, 0, 0],
            &[0, 0, 0, 0, 0, 21, 35, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 3, 0, 0, 1, 11, 0, 0],
        ]),
        // letter: 'č' / 0x10d
        'č' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 4, 0, 0, 0, 3, 2, 0, 0],
            &[0, 0, 27, 222, 107, 0, 84, 228, 49, 0, 0],
            &[0, 0, 0, 44, 240, 178, 247, 65, 0, 0, 0],
            &[0, 0, 0, 0, 80, 178, 101, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 62, 55, 17, 0, 0, 0],
            &[0, 0, 27, 184, 254, 255, 255, 250, 94, 0, 0],
            &[0, 2, 205, 238, 91, 23, 26, 87, 34, 0, 0],
            &[0, 66, 255, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 118, 255, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 136, 255, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 119, 255, 26, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 255, 104, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 201, 243, 110, 46, 52, 94, 74, 0, 0],
            &[0, 0, 22, 169, 249, 255, 255, 239, 90, 0, 0],
            &[0, 0, 0, 0, 10, 35, 25, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ď' / 0x10e
        'Ď' => Some(&[
            &[0, 10, 204, 141, 2, 66, 228, 70, 0, 0, 0],
            &[0, 0, 22, 222, 190, 248, 88, 0, 0, 0, 0],
            &[0, 0, 0, 53, 171, 118, 0, 0, 0, 0, 0],
            &[0, 28, 35, 35, 24, 1, 0, 0, 0, 0, 0],
            &[0, 207, 255, 255, 255, 245, 160, 26, 0, 0, 0],
            &[0, 207, 197, 48, 63, 120, 235, 231, 32, 0, 0],
            &[0, 207, 184, 0, 0, 0, 35, 241, 183, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 254, 27, 0],
            &[0, 207, 184, 0, 0, 0, 0, 65, 255, 80, 0],
            &[0, 207, 184, 0, 0, 0, 0, 40, 255, 103, 0],
            &[0, 207, 184, 0, 0, 0, 0, 42, 255, 95, 0],
            &[0, 207, 184, 0, 0, 0, 0, 76, 255, 69, 0],
            &[0, 207, 184, 0, 0, 0, 0, 156, 245, 10, 0],
            &[0, 207, 184, 0, 0, 0, 69, 252, 146, 0, 0],
            &[0, 207, 207, 84, 102, 167, 252, 195, 10, 0, 0],
            &[0, 207, 255, 255, 236, 199, 108, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 10, 0, 0, 0, 6, 7, 0, 0, 0],
        ]),
        // letter: 'ď' / 0x10f
        'ď' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 235, 37, 150, 188],
            &[0, 0, 0, 0, 0, 0, 90, 255, 40, 212, 104],
            &[0, 0, 0, 0, 0, 0, 90, 255, 54, 182, 5],
            &[0, 0, 0, 46, 59, 15, 89, 255, 40, 0, 0],
            &[0, 22, 202, 255, 253, 239, 139, 255, 40, 0, 0],
            &[0, 166, 248, 73, 2, 72, 248, 255, 40, 0, 0],
            &[9, 248, 159, 0, 0, 0, 163, 255, 40, 0, 0],
            &[43, 255, 103, 0, 0, 0, 118, 255, 40, 0, 0],
            &[56, 255, 88, 0, 0, 0, 93, 255, 40, 0, 0],
            &[41, 255, 105, 0, 0, 0, 109, 255, 40, 0, 0],
            &[5, 244, 161, 0, 0, 0, 168, 255, 40, 0, 0],
            &[0, 151, 249, 96, 32, 107, 242, 255, 40, 0, 0],
            &[0, 12, 178, 255, 255, 214, 80, 255, 40, 0, 0],
            &[0, 0, 0, 21, 32, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Đ' / 0x110
        'Đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 28, 3, 0, 0, 0, 0, 0],
            &[0, 163, 255, 255, 255, 251, 178, 46, 0, 0, 0],
            &[0, 163, 233, 48, 58, 105, 219, 245, 61, 0, 0],
            &[0, 163, 227, 0, 0, 0, 15, 217, 222, 4, 0],
            &[0, 163, 227, 0, 0, 0, 0, 89, 255, 69, 0],
            &[0, 163, 227, 0, 0, 0, 0, 21, 255, 124, 0],
            &[169, 255, 255, 169, 163, 0, 0, 1, 251, 146, 0],
            &[169, 255, 255, 169, 163, 0, 0, 1, 253, 139, 0],
            &[0, 163, 227, 0, 0, 0, 0, 32, 255, 113, 0],
            &[0, 163, 227, 0, 0, 0, 0, 112, 255, 43, 0],
            &[0, 163, 227, 0, 0, 0, 40, 237, 189, 0, 0],
            &[0, 163, 236, 83, 97, 151, 245, 219, 30, 0, 0],
            &[0, 163, 255, 255, 241, 209, 126, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'đ' / 0x111
        'đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 158, 197, 0, 0],
            &[0, 0, 0, 0, 68, 99, 99, 255, 255, 99, 19],
            &[0, 0, 0, 0, 127, 187, 187, 255, 255, 187, 37],
            &[0, 0, 0, 8, 39, 23, 0, 171, 214, 0, 0],
            &[0, 0, 125, 243, 255, 255, 178, 169, 214, 0, 0],
            &[0, 99, 255, 160, 37, 32, 172, 251, 214, 0, 0],
            &[0, 205, 224, 3, 0, 0, 12, 243, 214, 0, 0],
            &[1, 252, 156, 0, 0, 0, 0, 193, 214, 0, 0],
            &[12, 255, 134, 0, 0, 0, 0, 174, 214, 0, 0],
            &[2, 250, 152, 0, 0, 0, 0, 191, 214, 0, 0],
            &[0, 201, 216, 1, 0, 0, 13, 241, 214, 0, 0],
            &[0, 92, 255, 151, 40, 46, 178, 245, 214, 0, 0],
            &[0, 0, 121, 244, 255, 254, 154, 128, 214, 0, 0],
            &[0, 0, 0, 7, 36, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ē' / 0x112
        'Ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 189, 189, 189, 189, 135, 0, 0, 0],
            &[0, 0, 39, 119, 119, 119, 119, 85, 0, 0, 0],
            &[0, 13, 35, 35, 35, 35, 35, 35, 26, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 98, 255, 85, 56, 56, 56, 56, 42, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 41, 5, 5, 5, 5, 2, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 110, 0, 0],
            &[0, 98, 255, 111, 86, 86, 86, 86, 37, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 115, 92, 92, 92, 92, 69, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ē' / 0x113
        'ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 66, 182, 182, 182, 182, 124, 0, 0, 0],
            &[0, 0, 46, 126, 126, 126, 126, 86, 0, 0, 0],
            &[0, 0, 0, 2, 46, 63, 27, 0, 0, 0, 0],
            &[0, 0, 67, 223, 255, 252, 253, 161, 8, 0, 0],
            &[0, 44, 248, 178, 30, 1, 77, 248, 139, 0, 0],
            &[0, 163, 231, 10, 0, 0, 0, 148, 240, 5, 0],
            &[0, 225, 213, 125, 125, 125, 125, 178, 255, 33, 0],
            &[0, 244, 227, 187, 187, 187, 187, 187, 187, 33, 0],
            &[0, 223, 170, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 155, 239, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 239, 204, 74, 30, 46, 95, 136, 0, 0],
            &[0, 0, 39, 183, 252, 255, 255, 233, 127, 0, 0],
            &[0, 0, 0, 0, 11, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĕ' / 0x114
        'Ĕ' => Some(&[
            &[0, 0, 59, 106, 0, 0, 43, 119, 0, 0, 0],
            &[0, 0, 33, 240, 142, 122, 216, 121, 0, 0, 0],
            &[0, 0, 0, 47, 147, 157, 92, 1, 0, 0, 0],
            &[0, 13, 35, 35, 35, 35, 35, 35, 26, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 98, 255, 85, 56, 56, 56, 56, 42, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 41, 5, 5, 5, 5, 2, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 110, 0, 0],
            &[0, 98, 255, 111, 86, 86, 86, 86, 37, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 115, 92, 92, 92, 92, 69, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĕ' / 0x115
        'ĕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 96, 0, 0, 46, 110, 0, 0, 0],
            &[0, 0, 42, 241, 132, 116, 216, 116, 0, 0, 0],
            &[0, 0, 0, 57, 155, 163, 95, 1, 0, 0, 0],
            &[0, 0, 0, 2, 46, 63, 27, 0, 0, 0, 0],
            &[0, 0, 67, 223, 255, 252, 253, 161, 8, 0, 0],
            &[0, 44, 248, 178, 30, 1, 77, 248, 139, 0, 0],
            &[0, 163, 231, 10, 0, 0, 0, 148, 240, 5, 0],
            &[0, 225, 213, 125, 125, 125, 125, 178, 255, 33, 0],
            &[0, 244, 227, 187, 187, 187, 187, 187, 187, 33, 0],
            &[0, 223, 170, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 155, 239, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 239, 204, 74, 30, 46, 95, 136, 0, 0],
            &[0, 0, 39, 183, 252, 255, 255, 233, 127, 0, 0],
            &[0, 0, 0, 0, 11, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ė' / 0x116
        'Ė' => Some(&[
            &[0, 0, 0, 0, 137, 209, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 252, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 24, 0, 0, 0, 0, 0],
            &[0, 13, 35, 35, 35, 35, 35, 35, 26, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 98, 255, 85, 56, 56, 56, 56, 42, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 41, 5, 5, 5, 5, 2, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 110, 0, 0],
            &[0, 98, 255, 111, 86, 86, 86, 86, 37, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 115, 92, 92, 92, 92, 69, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ė' / 0x117
        'ė' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 199, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 251, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 46, 63, 27, 0, 0, 0, 0],
            &[0, 0, 67, 223, 255, 252, 253, 161, 8, 0, 0],
            &[0, 44, 248, 178, 30, 1, 77, 248, 139, 0, 0],
            &[0, 163, 231, 10, 0, 0, 0, 148, 240, 5, 0],
            &[0, 225, 213, 125, 125, 125, 125, 178, 255, 33, 0],
            &[0, 244, 227, 187, 187, 187, 187, 187, 187, 33, 0],
            &[0, 223, 170, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 155, 239, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 239, 204, 74, 30, 46, 95, 136, 0, 0],
            &[0, 0, 39, 183, 252, 255, 255, 233, 127, 0, 0],
            &[0, 0, 0, 0, 11, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ę' / 0x118
        'Ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 35, 35, 35, 35, 35, 35, 26, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 98, 255, 85, 56, 56, 56, 56, 42, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 41, 5, 5, 5, 5, 2, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 110, 0, 0],
            &[0, 98, 255, 111, 86, 86, 86, 86, 37, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 115, 92, 92, 92, 92, 69, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 3, 189, 102, 0, 0, 0],
            &[0, 0, 0, 0, 0, 91, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 226, 79, 22, 0, 0],
            &[0, 0, 0, 0, 0, 18, 153, 181, 41, 0, 0],
        ]),
        // letter: 'ę' / 0x119
        'ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 46, 63, 27, 0, 0, 0, 0],
            &[0, 0, 67, 223, 255, 252, 253, 161, 8, 0, 0],
            &[0, 44, 248, 178, 30, 1, 77, 248, 139, 0, 0],
            &[0, 163, 231, 10, 0, 0, 0, 148, 240, 5, 0],
            &[0, 225, 213, 125, 125, 125, 125, 178, 255, 33, 0],
            &[0, 244, 227, 187, 187, 187, 187, 187, 187, 33, 0],
            &[0, 223, 170, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 155, 239, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 239, 204, 74, 30, 46, 95, 136, 0, 0],
            &[0, 0, 39, 183, 252, 255, 255, 255, 186, 0, 0],
            &[0, 0, 0, 0, 11, 35, 137, 175, 3, 0, 0],
            &[0, 0, 0, 0, 0, 12, 250, 36, 0, 0, 0],
            &[0, 0, 0, 0, 0, 27, 254, 122, 48, 0, 0],
            &[0, 0, 0, 0, 0, 0, 101, 180, 103, 0, 0],
        ]),
        // letter: 'Ě' / 0x11a
        'Ě' => Some(&[
            &[0, 0, 91, 220, 47, 9, 166, 186, 4, 0, 0],
            &[0, 0, 0, 119, 239, 200, 202, 11, 0, 0, 0],
            &[0, 0, 0, 0, 140, 171, 32, 0, 0, 0, 0],
            &[0, 13, 35, 35, 35, 35, 35, 35, 26, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 98, 255, 85, 56, 56, 56, 56, 42, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 41, 5, 5, 5, 5, 2, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 110, 0, 0],
            &[0, 98, 255, 111, 86, 86, 86, 86, 37, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 255, 115, 92, 92, 92, 92, 69, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 4, 0, 0, 0, 12, 0, 0, 0],
        ]),
        // letter: 'ě' / 0x11b
        'ě' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 1, 0, 0, 0, 5, 0, 0, 0],
            &[0, 0, 104, 213, 38, 9, 166, 184, 2, 0, 0],
            &[0, 0, 0, 133, 233, 198, 200, 10, 0, 0, 0],
            &[0, 0, 0, 2, 151, 177, 29, 0, 0, 0, 0],
            &[0, 0, 0, 2, 46, 63, 27, 0, 0, 0, 0],
            &[0, 0, 67, 223, 255, 252, 253, 161, 8, 0, 0],
            &[0, 44, 248, 178, 30, 1, 77, 248, 139, 0, 0],
            &[0, 163, 231, 10, 0, 0, 0, 148, 240, 5, 0],
            &[0, 225, 213, 125, 125, 125, 125, 178, 255, 33, 0],
            &[0, 244, 227, 187, 187, 187, 187, 187, 187, 33, 0],
            &[0, 223, 170, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 155, 239, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 239, 204, 74, 30, 46, 95, 136, 0, 0],
            &[0, 0, 39, 183, 252, 255, 255, 233, 127, 0, 0],
            &[0, 0, 0, 0, 11, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => Some(&[
            &[0, 0, 0, 0, 177, 255, 155, 0, 0, 0, 0],
            &[0, 0, 0, 129, 217, 69, 231, 109, 0, 0, 0],
            &[0, 0, 41, 156, 20, 0, 30, 161, 32, 0, 0],
            &[0, 0, 0, 0, 27, 63, 63, 26, 0, 0, 0],
            &[0, 0, 27, 182, 254, 255, 255, 254, 152, 0, 0],
            &[0, 15, 224, 237, 103, 34, 33, 84, 76, 0, 0],
            &[0, 137, 254, 56, 0, 0, 0, 0, 0, 0, 0],
            &[0, 225, 183, 0, 0, 0, 0, 0, 0, 0, 0],
            &[26, 255, 119, 0, 0, 0, 0, 0, 0, 0, 0],
            &[47, 255, 93, 0, 0, 71, 104, 104, 104, 1, 0],
            &[52, 255, 91, 0, 0, 165, 243, 249, 255, 2, 0],
            &[32, 255, 116, 0, 0, 0, 0, 120, 255, 2, 0],
            &[2, 237, 177, 0, 0, 0, 0, 120, 255, 2, 0],
            &[0, 155, 252, 51, 0, 0, 0, 120, 255, 2, 0],
            &[0, 25, 232, 236, 116, 61, 64, 172, 255, 2, 0],
            &[0, 0, 32, 174, 252, 255, 255, 245, 178, 1, 0],
            &[0, 0, 0, 0, 7, 34, 24, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 12, 2, 0, 0, 0, 0],
        ]),
        // letter: 'ĝ' / 0x11d
        'ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 30, 239, 252, 58, 0, 0, 0],
            &[0, 0, 0, 12, 207, 154, 122, 231, 32, 0, 0],
            &[0, 0, 0, 107, 119, 0, 0, 92, 141, 0, 0],
            &[0, 0, 0, 0, 24, 65, 45, 0, 11, 24, 0],
            &[0, 0, 0, 137, 253, 254, 254, 196, 141, 214, 0],
            &[0, 0, 96, 255, 133, 15, 11, 135, 255, 214, 0],
            &[0, 0, 202, 213, 0, 0, 0, 1, 216, 214, 0],
            &[0, 2, 251, 152, 0, 0, 0, 0, 156, 214, 0],
            &[0, 12, 255, 134, 0, 0, 0, 0, 139, 214, 0],
            &[0, 2, 250, 155, 0, 0, 0, 0, 158, 214, 0],
            &[0, 0, 199, 224, 3, 0, 0, 2, 216, 214, 0],
            &[0, 0, 85, 255, 164, 45, 39, 153, 254, 214, 0],
            &[0, 0, 0, 111, 241, 255, 255, 192, 163, 214, 0],
            &[0, 0, 0, 0, 6, 36, 23, 0, 160, 208, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 213, 174, 0],
            &[0, 0, 106, 101, 49, 23, 45, 155, 255, 76, 0],
            &[0, 0, 0, 0, 11, 34, 24, 1, 0, 0, 0],
        ]),
        // letter: 'Ğ' / 0x11e
        'Ğ' => Some(&[
            &[0, 0, 17, 145, 3, 0, 5, 151, 7, 0, 0],
            &[0, 0, 1, 208, 175, 114, 185, 191, 0, 0, 0],
            &[0, 0, 0, 20, 130, 162, 120, 13, 0, 0, 0],
            &[0, 0, 0, 0, 27, 63, 63, 26, 0, 0, 0],
            &[0, 0, 27, 182, 254, 255, 255, 254, 152, 0, 0],
            &[0, 15, 224, 237, 103, 34, 33, 84, 76, 0, 0],
            &[0, 137, 254, 56, 0, 0, 0, 0, 0, 0, 0],
            &[0, 225, 183, 0, 0, 0, 0, 0, 0, 0, 0],
            &[26, 255, 119, 0, 0, 0, 0, 0, 0, 0, 0],
            &[47, 255, 93, 0, 0, 71, 104, 104, 104, 1, 0],
            &[52, 255, 91, 0, 0, 165, 243, 249, 255, 2, 0],
            &[32, 255, 116, 0, 0, 0, 0, 120, 255, 2, 0],
            &[2, 237, 177, 0, 0, 0, 0, 120, 255, 2, 0],
            &[0, 155, 252, 51, 0, 0, 0, 120, 255, 2, 0],
            &[0, 25, 232, 236, 116, 61, 64, 172, 255, 2, 0],
            &[0, 0, 32, 174, 252, 255, 255, 245, 178, 1, 0],
            &[0, 0, 0, 0, 7, 34, 24, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ğ' / 0x11f
        'ğ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 85, 0, 0, 56, 100, 0, 0],
            &[0, 0, 0, 55, 240, 126, 119, 223, 99, 0, 0],
            &[0, 0, 0, 0, 66, 158, 161, 86, 0, 0, 0],
            &[0, 0, 0, 0, 24, 65, 45, 0, 11, 24, 0],
            &[0, 0, 0, 137, 253, 254, 254, 196, 141, 214, 0],
            &[0, 0, 96, 255, 133, 15, 11, 135, 255, 214, 0],
            &[0, 0, 202, 213, 0, 0, 0, 1, 216, 214, 0],
            &[0, 2, 251, 152, 0, 0, 0, 0, 156, 214, 0],
            &[0, 12, 255, 134, 0, 0, 0, 0, 139, 214, 0],
            &[0, 2, 250, 155, 0, 0, 0, 0, 158, 214, 0],
            &[0, 0, 199, 224, 3, 0, 0, 2, 216, 214, 0],
            &[0, 0, 85, 255, 164, 45, 39, 153, 254, 214, 0],
            &[0, 0, 0, 111, 241, 255, 255, 192, 163, 214, 0],
            &[0, 0, 0, 0, 6, 36, 23, 0, 160, 208, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 213, 174, 0],
            &[0, 0, 106, 101, 49, 23, 45, 155, 255, 76, 0],
            &[0, 0, 0, 0, 11, 34, 24, 1, 0, 0, 0],
        ]),
        // letter: 'Ġ' / 0x120
        'Ġ' => Some(&[
            &[0, 0, 0, 0, 79, 219, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 255, 95, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 63, 63, 26, 0, 0, 0],
            &[0, 0, 27, 182, 254, 255, 255, 254, 152, 0, 0],
            &[0, 15, 224, 237, 103, 34, 33, 84, 76, 0, 0],
            &[0, 137, 254, 56, 0, 0, 0, 0, 0, 0, 0],
            &[0, 225, 183, 0, 0, 0, 0, 0, 0, 0, 0],
            &[26, 255, 119, 0, 0, 0, 0, 0, 0, 0, 0],
            &[47, 255, 93, 0, 0, 71, 104, 104, 104, 1, 0],
            &[52, 255, 91, 0, 0, 165, 243, 249, 255, 2, 0],
            &[32, 255, 116, 0, 0, 0, 0, 120, 255, 2, 0],
            &[2, 237, 177, 0, 0, 0, 0, 120, 255, 2, 0],
            &[0, 155, 252, 51, 0, 0, 0, 120, 255, 2, 0],
            &[0, 25, 232, 236, 116, 61, 64, 172, 255, 2, 0],
            &[0, 0, 32, 174, 252, 255, 255, 245, 178, 1, 0],
            &[0, 0, 0, 0, 7, 34, 24, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ġ' / 0x121
        'ġ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 190, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 216, 245, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 65, 45, 0, 11, 24, 0],
            &[0, 0, 0, 137, 253, 254, 254, 196, 141, 214, 0],
            &[0, 0, 96, 255, 133, 15, 11, 135, 255, 214, 0],
            &[0, 0, 202, 213, 0, 0, 0, 1, 216, 214, 0],
            &[0, 2, 251, 152, 0, 0, 0, 0, 156, 214, 0],
            &[0, 12, 255, 134, 0, 0, 0, 0, 139, 214, 0],
            &[0, 2, 250, 155, 0, 0, 0, 0, 158, 214, 0],
            &[0, 0, 199, 224, 3, 0, 0, 2, 216, 214, 0],
            &[0, 0, 85, 255, 164, 45, 39, 153, 254, 214, 0],
            &[0, 0, 0, 111, 241, 255, 255, 192, 163, 214, 0],
            &[0, 0, 0, 0, 6, 36, 23, 0, 160, 208, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 213, 174, 0],
            &[0, 0, 106, 101, 49, 23, 45, 155, 255, 76, 0],
            &[0, 0, 0, 0, 11, 34, 24, 1, 0, 0, 0],
        ]),
        // letter: 'Ģ' / 0x122
        'Ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 63, 63, 26, 0, 0, 0],
            &[0, 0, 27, 182, 254, 255, 255, 254, 152, 0, 0],
            &[0, 15, 224, 237, 103, 34, 33, 84, 76, 0, 0],
            &[0, 137, 254, 56, 0, 0, 0, 0, 0, 0, 0],
            &[0, 225, 183, 0, 0, 0, 0, 0, 0, 0, 0],
            &[26, 255, 119, 0, 0, 0, 0, 0, 0, 0, 0],
            &[47, 255, 93, 0, 0, 71, 104, 104, 104, 1, 0],
            &[52, 255, 91, 0, 0, 165, 243, 249, 255, 2, 0],
            &[32, 255, 116, 0, 0, 0, 0, 120, 255, 2, 0],
            &[2, 237, 177, 0, 0, 0, 0, 120, 255, 2, 0],
            &[0, 155, 252, 51, 0, 0, 0, 120, 255, 2, 0],
            &[0, 25, 232, 236, 116, 61, 64, 172, 255, 2, 0],
            &[0, 0, 32, 174, 252, 255, 255, 245, 178, 1, 0],
            &[0, 0, 0, 0, 7, 34, 24, 2, 0, 0, 0],
            &[0, 0, 0, 0, 18, 246, 88, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 228, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 76, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ģ' / 0x123
        'ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 229, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 144, 190, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 164, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 65, 45, 0, 11, 24, 0],
            &[0, 0, 0, 137, 253, 254, 254, 196, 141, 214, 0],
            &[0, 0, 96, 255, 133, 15, 11, 135, 255, 214, 0],
            &[0, 0, 202, 213, 0, 0, 0, 1, 216, 214, 0],
            &[0, 2, 251, 152, 0, 0, 0, 0, 156, 214, 0],
            &[0, 12, 255, 134, 0, 0, 0, 0, 139, 214, 0],
            &[0, 2, 250, 155, 0, 0, 0, 0, 158, 214, 0],
            &[0, 0, 199, 224, 3, 0, 0, 2, 216, 214, 0],
            &[0, 0, 85, 255, 164, 45, 39, 153, 254, 214, 0],
            &[0, 0, 0, 111, 241, 255, 255, 192, 163, 214, 0],
            &[0, 0, 0, 0, 6, 36, 23, 0, 160, 208, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 213, 174, 0],
            &[0, 0, 106, 101, 49, 23, 45, 155, 255, 76, 0],
            &[0, 0, 0, 0, 11, 34, 24, 1, 0, 0, 0],
        ]),
        // letter: 'Ĥ' / 0x124
        'Ĥ' => Some(&[
            &[0, 0, 0, 33, 241, 253, 62, 0, 0, 0, 0],
            &[0, 0, 14, 210, 148, 116, 233, 35, 0, 0, 0],
            &[0, 0, 105, 112, 0, 0, 86, 138, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 18, 35, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 185, 5, 5, 5, 5, 135, 255, 2, 0],
            &[0, 207, 255, 255, 255, 255, 255, 255, 255, 2, 0],
            &[0, 207, 208, 86, 86, 86, 86, 174, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 10, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĥ' / 0x125
        'ĥ' => Some(&[
            &[0, 165, 200, 161, 214, 16, 0, 0, 0, 0, 0],
            &[108, 174, 14, 1, 133, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 150, 209, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 163, 227, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 163, 227, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 163, 227, 0, 15, 59, 42, 2, 0, 0],
            &[0, 0, 163, 221, 95, 243, 253, 255, 205, 26, 0],
            &[0, 0, 163, 242, 206, 40, 3, 79, 252, 160, 0],
            &[0, 0, 163, 255, 57, 0, 0, 0, 185, 216, 0],
            &[0, 0, 163, 244, 1, 0, 0, 0, 156, 236, 0],
            &[0, 0, 163, 228, 0, 0, 0, 0, 154, 236, 0],
            &[0, 0, 163, 227, 0, 0, 0, 0, 154, 236, 0],
            &[0, 0, 163, 227, 0, 0, 0, 0, 154, 236, 0],
            &[0, 0, 163, 227, 0, 0, 0, 0, 154, 236, 0],
            &[0, 0, 163, 227, 0, 0, 0, 0, 154, 236, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 188, 210, 40, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ħ' / 0x126
        'Ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 18, 35, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[29, 236, 213, 29, 29, 29, 29, 162, 255, 31, 5],
            &[255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 51],
            &[27, 235, 212, 27, 27, 27, 27, 160, 255, 30, 5],
            &[0, 207, 185, 5, 5, 5, 5, 135, 255, 2, 0],
            &[0, 207, 255, 255, 255, 255, 255, 255, 255, 2, 0],
            &[0, 207, 208, 86, 86, 86, 86, 174, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ħ' / 0x127
        'ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 150, 209, 0, 0, 0, 0, 0, 0, 0],
            &[0, 99, 255, 255, 99, 99, 88, 0, 0, 0, 0],
            &[0, 187, 255, 255, 187, 187, 165, 0, 0, 0, 0],
            &[0, 0, 163, 227, 0, 15, 59, 42, 2, 0, 0],
            &[0, 0, 163, 221, 95, 243, 253, 255, 205, 26, 0],
            &[0, 0, 163, 242, 206, 40, 3, 79, 252, 160, 0],
            &[0, 0, 163, 255, 57, 0, 0, 0, 185, 216, 0],
            &[0, 0, 163, 244, 1, 0, 0, 0, 156, 236, 0],
            &[0, 0, 163, 228, 0, 0, 0, 0, 154, 236, 0],
            &[0, 0, 163, 227, 0, 0, 0, 0, 154, 236, 0],
            &[0, 0, 163, 227, 0, 0, 0, 0, 154, 236, 0],
            &[0, 0, 163, 227, 0, 0, 0, 0, 154, 236, 0],
            &[0, 0, 163, 227, 0, 0, 0, 0, 154, 236, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĩ' / 0x128
        'Ĩ' => Some(&[
            &[0, 0, 7, 104, 90, 5, 0, 107, 13, 0, 0],
            &[0, 0, 154, 213, 225, 217, 162, 219, 1, 0, 0],
            &[0, 0, 150, 11, 6, 106, 152, 43, 0, 0, 0],
            &[0, 4, 35, 35, 35, 35, 35, 35, 11, 0, 0],
            &[0, 30, 248, 255, 255, 255, 255, 251, 78, 0, 0],
            &[0, 0, 0, 16, 184, 222, 20, 1, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 1, 28, 51, 195, 227, 55, 32, 5, 0, 0],
            &[0, 33, 255, 255, 255, 255, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĩ' / 0x129
        'ĩ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 97, 83, 3, 0, 101, 13, 0, 0],
            &[0, 0, 151, 219, 230, 212, 155, 221, 1, 0, 0],
            &[0, 0, 155, 13, 8, 113, 158, 48, 0, 0, 0],
            &[0, 10, 28, 28, 28, 28, 3, 0, 0, 0, 0],
            &[0, 89, 254, 255, 255, 255, 35, 0, 0, 0, 0],
            &[0, 0, 3, 20, 122, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 7, 28, 48, 139, 255, 87, 41, 19, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ī' / 0x12a
        'Ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 189, 189, 189, 189, 116, 0, 0, 0],
            &[0, 0, 51, 119, 119, 119, 119, 73, 0, 0, 0],
            &[0, 4, 35, 35, 35, 35, 35, 35, 11, 0, 0],
            &[0, 30, 248, 255, 255, 255, 255, 251, 78, 0, 0],
            &[0, 0, 0, 16, 184, 222, 20, 1, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 1, 28, 51, 195, 227, 55, 32, 5, 0, 0],
            &[0, 33, 255, 255, 255, 255, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ī' / 0x12b
        'ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 182, 182, 182, 182, 112, 0, 0, 0],
            &[0, 0, 54, 126, 126, 126, 126, 77, 0, 0, 0],
            &[0, 10, 28, 28, 28, 28, 3, 0, 0, 0, 0],
            &[0, 89, 254, 255, 255, 255, 35, 0, 0, 0, 0],
            &[0, 0, 3, 20, 122, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 7, 28, 48, 139, 255, 87, 41, 19, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => Some(&[
            &[0, 0, 75, 90, 0, 0, 59, 104, 0, 0, 0],
            &[0, 0, 52, 240, 132, 126, 226, 95, 0, 0, 0],
            &[0, 0, 0, 60, 151, 154, 80, 0, 0, 0, 0],
            &[0, 4, 35, 35, 35, 35, 35, 35, 11, 0, 0],
            &[0, 30, 248, 255, 255, 255, 255, 251, 78, 0, 0],
            &[0, 0, 0, 16, 184, 222, 20, 1, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 1, 28, 51, 195, 227, 55, 32, 5, 0, 0],
            &[0, 33, 255, 255, 255, 255, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĭ' / 0x12d
        'ĭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 72, 85, 0, 0, 56, 100, 0, 0, 0],
            &[0, 0, 55, 240, 126, 119, 223, 99, 0, 0, 0],
            &[0, 0, 0, 66, 158, 161, 86, 0, 0, 0, 0],
            &[0, 10, 28, 28, 28, 28, 3, 0, 0, 0, 0],
            &[0, 89, 254, 255, 255, 255, 35, 0, 0, 0, 0],
            &[0, 0, 3, 20, 122, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 7, 28, 48, 139, 255, 87, 41, 19, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Į' / 0x12e
        'Į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 35, 35, 35, 35, 35, 35, 11, 0, 0],
            &[0, 30, 248, 255, 255, 255, 255, 251, 78, 0, 0],
            &[0, 0, 0, 16, 184, 222, 20, 1, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 1, 28, 51, 195, 227, 55, 32, 5, 0, 0],
            &[0, 33, 255, 255, 255, 255, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 36, 218, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 131, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 200, 177, 73, 0, 0, 0],
            &[0, 0, 0, 0, 0, 55, 173, 165, 0, 0, 0],
        ]),
        // letter: 'į' / 0x12f
        'į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 124, 205, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 181, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 31, 0, 0, 0, 0, 0],
            &[0, 10, 28, 28, 28, 28, 3, 0, 0, 0, 0],
            &[0, 89, 254, 255, 255, 255, 35, 0, 0, 0, 0],
            &[0, 0, 3, 20, 122, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 7, 28, 48, 139, 255, 87, 41, 19, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 3, 189, 102, 0, 0, 0],
            &[0, 0, 0, 0, 0, 91, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 226, 79, 22, 0, 0],
            &[0, 0, 0, 0, 0, 18, 153, 181, 41, 0, 0],
        ]),
        // letter: 'İ' / 0x130
        'İ' => Some(&[
            &[0, 0, 0, 0, 160, 196, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 213, 243, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 20, 0, 0, 0, 0, 0],
            &[0, 4, 35, 35, 35, 35, 35, 35, 11, 0, 0],
            &[0, 30, 248, 255, 255, 255, 255, 251, 78, 0, 0],
            &[0, 0, 0, 16, 184, 222, 20, 1, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 217, 0, 0, 0, 0, 0],
            &[0, 1, 28, 51, 195, 227, 55, 32, 5, 0, 0],
            &[0, 33, 255, 255, 255, 255, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ı' / 0x131
        'ı' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 28, 28, 28, 28, 3, 0, 0, 0, 0],
            &[0, 89, 254, 255, 255, 255, 35, 0, 0, 0, 0],
            &[0, 0, 3, 20, 122, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 35, 0, 0, 0, 0],
            &[0, 7, 28, 48, 139, 255, 87, 41, 19, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĳ' / 0x132
        'Ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 35, 8, 0, 0, 1, 35, 17, 0, 0],
            &[0, 76, 255, 58, 0, 0, 7, 255, 127, 0, 0],
            &[0, 76, 255, 58, 0, 0, 7, 255, 127, 0, 0],
            &[0, 76, 255, 58, 0, 0, 7, 255, 127, 0, 0],
            &[0, 76, 255, 58, 0, 0, 7, 255, 127, 0, 0],
            &[0, 76, 255, 58, 0, 0, 7, 255, 127, 0, 0],
            &[0, 76, 255, 58, 0, 0, 7, 255, 127, 0, 0],
            &[0, 76, 255, 58, 0, 0, 7, 255, 127, 0, 0],
            &[0, 76, 255, 58, 0, 0, 7, 255, 127, 0, 0],
            &[0, 46, 153, 35, 0, 0, 16, 255, 121, 0, 0],
            &[0, 0, 0, 0, 0, 0, 80, 255, 87, 0, 0],
            &[0, 49, 135, 84, 71, 117, 235, 229, 12, 0, 0],
            &[0, 49, 233, 255, 255, 255, 201, 46, 0, 0, 0],
            &[0, 0, 0, 23, 38, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĳ' / 0x133
        'ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 19, 206, 122, 0, 0, 80, 212, 54, 0, 0],
            &[0, 40, 255, 178, 0, 0, 123, 255, 90, 0, 0],
            &[0, 0, 31, 9, 0, 0, 2, 37, 0, 0, 0],
            &[0, 0, 28, 14, 0, 0, 7, 28, 7, 0, 0],
            &[0, 2, 255, 132, 0, 0, 64, 255, 71, 0, 0],
            &[0, 2, 255, 132, 0, 0, 64, 255, 71, 0, 0],
            &[0, 2, 255, 132, 0, 0, 64, 255, 71, 0, 0],
            &[0, 2, 255, 132, 0, 0, 64, 255, 71, 0, 0],
            &[0, 2, 255, 132, 0, 0, 64, 255, 71, 0, 0],
            &[0, 2, 255, 132, 0, 0, 64, 255, 71, 0, 0],
            &[0, 2, 255, 132, 0, 0, 64, 255, 71, 0, 0],
            &[0, 2, 255, 132, 0, 0, 64, 255, 71, 0, 0],
            &[0, 2, 255, 132, 0, 0, 64, 255, 71, 0, 0],
            &[0, 0, 0, 0, 0, 0, 67, 255, 71, 0, 0],
            &[0, 0, 0, 0, 0, 0, 105, 255, 44, 0, 0],
            &[0, 0, 74, 50, 37, 83, 233, 217, 5, 0, 0],
            &[0, 0, 0, 20, 36, 11, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĵ' / 0x134
        'Ĵ' => Some(&[
            &[0, 0, 0, 0, 27, 236, 254, 71, 0, 0, 0],
            &[0, 0, 0, 11, 203, 157, 107, 236, 41, 0, 0],
            &[0, 0, 0, 98, 118, 1, 0, 79, 144, 0, 0],
            &[0, 0, 0, 0, 0, 22, 31, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 159, 231, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 159, 231, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 159, 231, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 159, 231, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 159, 231, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 159, 231, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 159, 231, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 159, 231, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 167, 226, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 220, 195, 0, 0, 0, 0],
            &[55, 132, 83, 82, 171, 255, 103, 0, 0, 0, 0],
            &[55, 232, 255, 255, 245, 136, 1, 0, 0, 0, 0],
            &[0, 1, 23, 35, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 11, 0, 0, 0, 0],
        ]),
        // letter: 'ĵ' / 0x135
        'ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 95, 255, 217, 11, 0, 0, 0],
            &[0, 0, 0, 55, 240, 91, 191, 178, 3, 0, 0],
            &[0, 0, 5, 158, 63, 0, 7, 142, 84, 0, 0],
            &[0, 0, 21, 28, 28, 28, 21, 0, 0, 0, 0],
            &[0, 0, 187, 255, 255, 255, 191, 0, 0, 0, 0],
            &[0, 0, 0, 9, 28, 208, 191, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 199, 191, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 199, 191, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 199, 191, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 199, 191, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 199, 191, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 199, 191, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 199, 191, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 202, 191, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 237, 166, 0, 0, 0, 0],
            &[0, 84, 41, 49, 151, 255, 91, 0, 0, 0, 0],
            &[0, 2, 27, 31, 3, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ķ' / 0x136
        'Ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 21, 35, 5, 0],
            &[0, 207, 184, 0, 0, 0, 27, 235, 183, 1, 0],
            &[0, 207, 184, 0, 0, 3, 195, 230, 21, 0, 0],
            &[0, 207, 184, 0, 0, 134, 253, 65, 0, 0, 0],
            &[0, 207, 184, 0, 71, 254, 129, 0, 0, 0, 0],
            &[0, 207, 184, 24, 234, 192, 3, 0, 0, 0, 0],
            &[0, 207, 184, 180, 255, 80, 0, 0, 0, 0, 0],
            &[0, 207, 243, 228, 212, 214, 6, 0, 0, 0, 0],
            &[0, 207, 242, 47, 55, 253, 126, 0, 0, 0, 0],
            &[0, 207, 184, 0, 0, 149, 248, 40, 0, 0, 0],
            &[0, 207, 184, 0, 0, 15, 230, 194, 1, 0, 0],
            &[0, 207, 184, 0, 0, 0, 86, 255, 102, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 182, 239, 24, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 220, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 119, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 172, 6, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ķ' / 0x137
        'ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 235, 74, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 255, 80, 0, 0, 0, 12, 28, 11, 0],
            &[0, 55, 255, 80, 0, 0, 21, 218, 210, 16, 0],
            &[0, 55, 255, 80, 0, 11, 202, 221, 25, 0, 0],
            &[0, 55, 255, 79, 5, 184, 231, 35, 0, 0, 0],
            &[0, 55, 255, 72, 161, 247, 47, 0, 0, 0, 0],
            &[0, 55, 255, 177, 237, 255, 87, 0, 0, 0, 0],
            &[0, 55, 255, 222, 33, 158, 241, 35, 0, 0, 0],
            &[0, 55, 255, 84, 0, 11, 217, 205, 6, 0, 0],
            &[0, 55, 255, 80, 0, 0, 49, 249, 146, 0, 0],
            &[0, 55, 255, 80, 0, 0, 0, 112, 255, 81, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 244, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 233, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 110, 82, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĸ' / 0x138
        'ĸ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 28, 9, 0, 0, 0, 12, 28, 11, 0],
            &[0, 55, 255, 80, 0, 0, 21, 218, 210, 16, 0],
            &[0, 55, 255, 80, 0, 11, 202, 221, 25, 0, 0],
            &[0, 55, 255, 79, 5, 184, 231, 35, 0, 0, 0],
            &[0, 55, 255, 72, 162, 247, 47, 0, 0, 0, 0],
            &[0, 55, 255, 178, 237, 255, 87, 0, 0, 0, 0],
            &[0, 55, 255, 222, 33, 158, 241, 35, 0, 0, 0],
            &[0, 55, 255, 84, 0, 11, 217, 205, 6, 0, 0],
            &[0, 55, 255, 80, 0, 0, 49, 249, 146, 0, 0],
            &[0, 55, 255, 80, 0, 0, 0, 112, 255, 81, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĺ' / 0x139
        'Ĺ' => Some(&[
            &[0, 0, 0, 168, 246, 47, 0, 0, 0, 0, 0],
            &[0, 0, 72, 250, 75, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 69, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 35, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 178, 105, 105, 105, 105, 79, 0, 0],
            &[0, 11, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 12, 5, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĺ' / 0x13a
        'ĺ' => Some(&[
            &[0, 0, 0, 0, 1, 196, 202, 13, 0, 0, 0],
            &[0, 0, 0, 0, 92, 185, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 126, 235, 235, 235, 235, 52, 0, 0, 0, 0],
            &[0, 20, 52, 69, 137, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 7, 27, 47, 122, 255, 104, 42, 20, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 212, 129, 0, 0, 0],
        ]),
        // letter: 'Ļ' / 0x13b
        'Ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 35, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 178, 105, 105, 105, 105, 79, 0, 0],
            &[0, 11, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 246, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 205, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 140, 52, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ļ' / 0x13c
        'ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 126, 235, 235, 235, 235, 52, 0, 0, 0, 0],
            &[0, 20, 52, 69, 137, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 7, 27, 47, 122, 255, 104, 42, 20, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 244, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 163, 29, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ľ' / 0x13d
        'Ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 35, 17, 0, 0, 0, 32, 20, 0, 0],
            &[0, 11, 255, 124, 0, 0, 7, 252, 100, 0, 0],
            &[0, 11, 255, 124, 0, 0, 58, 233, 10, 0, 0],
            &[0, 11, 255, 124, 0, 0, 64, 76, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 178, 105, 105, 105, 105, 79, 0, 0],
            &[0, 11, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ľ' / 0x13e
        'ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 126, 235, 235, 235, 235, 52, 2, 228, 107, 0],
            &[0, 20, 52, 69, 137, 255, 56, 44, 247, 24, 0],
            &[0, 0, 0, 0, 78, 255, 56, 82, 119, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 7, 27, 47, 122, 255, 104, 42, 20, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 35, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 5, 97, 36, 0, 0],
            &[0, 11, 255, 124, 0, 0, 65, 255, 168, 0, 0],
            &[0, 11, 255, 124, 0, 0, 20, 185, 87, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 255, 178, 105, 105, 105, 105, 79, 0, 0],
            &[0, 11, 255, 255, 255, 255, 255, 255, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŀ' / 0x140
        'ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[126, 235, 235, 235, 235, 52, 0, 0, 0, 0, 0],
            &[20, 52, 69, 137, 255, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 255, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 255, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 255, 56, 0, 0, 2, 13, 0],
            &[0, 0, 0, 78, 255, 56, 0, 0, 169, 248, 23],
            &[0, 0, 0, 78, 255, 56, 0, 0, 172, 249, 24],
            &[0, 0, 0, 78, 255, 56, 0, 0, 3, 16, 0],
            &[0, 0, 0, 78, 255, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 255, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 255, 56, 0, 0, 0, 0, 0],
            &[7, 27, 47, 122, 255, 104, 42, 20, 1, 0, 0],
            &[163, 255, 255, 255, 255, 255, 255, 255, 46, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ł' / 0x141
        'Ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 35, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 255, 42, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 255, 42, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 255, 42, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 255, 42, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 255, 42, 90, 80, 0, 0, 0, 0],
            &[0, 0, 93, 255, 236, 244, 106, 0, 0, 0, 0],
            &[0, 17, 241, 255, 207, 27, 0, 0, 0, 0, 0],
            &[1, 216, 255, 255, 42, 0, 0, 0, 0, 0, 0],
            &[0, 40, 95, 255, 42, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 255, 42, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 255, 130, 105, 105, 105, 105, 45, 0],
            &[0, 0, 93, 255, 255, 255, 255, 255, 255, 111, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ł' / 0x142
        'ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 126, 235, 235, 235, 235, 52, 0, 0, 0, 0],
            &[0, 20, 52, 69, 137, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 22, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 196, 246, 27, 0, 0],
            &[0, 0, 0, 0, 156, 255, 255, 59, 0, 0, 0],
            &[0, 0, 28, 175, 255, 255, 61, 0, 0, 0, 0],
            &[0, 0, 59, 177, 110, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 56, 0, 0, 0, 0],
            &[0, 7, 27, 47, 122, 255, 104, 42, 20, 1, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ń' / 0x143
        'Ń' => Some(&[
            &[0, 0, 0, 0, 0, 54, 254, 152, 0, 0, 0],
            &[0, 0, 0, 0, 5, 207, 179, 5, 0, 0, 0],
            &[0, 0, 0, 0, 67, 140, 6, 0, 0, 0, 0],
            &[0, 28, 35, 8, 0, 0, 0, 16, 35, 0, 0],
            &[0, 207, 255, 125, 0, 0, 0, 120, 255, 2, 0],
            &[0, 207, 230, 231, 6, 0, 0, 120, 255, 2, 0],
            &[0, 207, 152, 243, 93, 0, 0, 120, 255, 2, 0],
            &[0, 207, 155, 146, 205, 0, 0, 120, 255, 2, 0],
            &[0, 207, 167, 36, 253, 61, 0, 120, 255, 2, 0],
            &[0, 207, 171, 0, 178, 173, 0, 120, 255, 2, 0],
            &[0, 207, 171, 0, 65, 252, 32, 119, 255, 2, 0],
            &[0, 207, 171, 0, 0, 208, 140, 115, 255, 2, 0],
            &[0, 207, 171, 0, 0, 97, 240, 119, 255, 2, 0],
            &[0, 207, 171, 0, 0, 7, 233, 199, 255, 2, 0],
            &[0, 207, 171, 0, 0, 0, 128, 255, 255, 2, 0],
            &[0, 207, 171, 0, 0, 0, 23, 248, 255, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 11, 0, 0, 0],
        ]),
        // letter: 'ń' / 0x144
        'ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 245, 187, 1, 0, 0],
            &[0, 0, 0, 0, 0, 177, 206, 15, 0, 0, 0],
            &[0, 0, 0, 0, 47, 159, 16, 0, 0, 0, 0],
            &[0, 18, 17, 0, 18, 59, 46, 4, 0, 0, 0],
            &[0, 163, 175, 116, 247, 253, 255, 213, 32, 0, 0],
            &[0, 163, 245, 199, 38, 2, 61, 247, 168, 0, 0],
            &[0, 163, 255, 50, 0, 0, 0, 177, 220, 0, 0],
            &[0, 163, 248, 0, 0, 0, 0, 155, 236, 0, 0],
            &[0, 163, 229, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ņ' / 0x145
        'Ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 35, 8, 0, 0, 0, 16, 35, 0, 0],
            &[0, 207, 255, 125, 0, 0, 0, 120, 255, 2, 0],
            &[0, 207, 230, 231, 6, 0, 0, 120, 255, 2, 0],
            &[0, 207, 152, 243, 93, 0, 0, 120, 255, 2, 0],
            &[0, 207, 155, 146, 205, 0, 0, 120, 255, 2, 0],
            &[0, 207, 167, 36, 253, 61, 0, 120, 255, 2, 0],
            &[0, 207, 171, 0, 178, 173, 0, 120, 255, 2, 0],
            &[0, 207, 171, 0, 65, 252, 32, 119, 255, 2, 0],
            &[0, 207, 171, 0, 0, 208, 140, 115, 255, 2, 0],
            &[0, 207, 171, 0, 0, 97, 240, 119, 255, 2, 0],
            &[0, 207, 171, 0, 0, 7, 233, 199, 255, 2, 0],
            &[0, 207, 171, 0, 0, 0, 128, 255, 255, 2, 0],
            &[0, 207, 171, 0, 0, 0, 23, 248, 255, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 173, 180, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 237, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 45, 147, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ņ' / 0x146
        'ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 17, 0, 18, 59, 46, 4, 0, 0, 0],
            &[0, 163, 175, 116, 247, 253, 255, 213, 32, 0, 0],
            &[0, 163, 245, 199, 38, 2, 61, 247, 168, 0, 0],
            &[0, 163, 255, 50, 0, 0, 0, 177, 220, 0, 0],
            &[0, 163, 248, 0, 0, 0, 0, 155, 236, 0, 0],
            &[0, 163, 229, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 205, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 215, 101, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 164, 2, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ň' / 0x147
        'Ň' => Some(&[
            &[0, 0, 95, 219, 44, 10, 169, 183, 3, 0, 0],
            &[0, 0, 0, 123, 237, 202, 199, 9, 0, 0, 0],
            &[0, 0, 0, 0, 142, 171, 29, 0, 0, 0, 0],
            &[0, 28, 35, 8, 0, 0, 0, 16, 35, 0, 0],
            &[0, 207, 255, 125, 0, 0, 0, 120, 255, 2, 0],
            &[0, 207, 230, 231, 6, 0, 0, 120, 255, 2, 0],
            &[0, 207, 152, 243, 93, 0, 0, 120, 255, 2, 0],
            &[0, 207, 155, 146, 205, 0, 0, 120, 255, 2, 0],
            &[0, 207, 167, 36, 253, 61, 0, 120, 255, 2, 0],
            &[0, 207, 171, 0, 178, 173, 0, 120, 255, 2, 0],
            &[0, 207, 171, 0, 65, 252, 32, 119, 255, 2, 0],
            &[0, 207, 171, 0, 0, 208, 140, 115, 255, 2, 0],
            &[0, 207, 171, 0, 0, 97, 240, 119, 255, 2, 0],
            &[0, 207, 171, 0, 0, 7, 233, 199, 255, 2, 0],
            &[0, 207, 171, 0, 0, 0, 128, 255, 255, 2, 0],
            &[0, 207, 171, 0, 0, 0, 23, 248, 255, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 4, 0, 0, 0, 12, 0, 0, 0],
        ]),
        // letter: 'ň' / 0x148
        'ň' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 2, 0, 0, 0, 5, 0, 0, 0],
            &[0, 0, 73, 225, 57, 2, 139, 207, 13, 0, 0],
            &[0, 0, 0, 99, 245, 186, 222, 23, 0, 0, 0],
            &[0, 0, 0, 0, 129, 178, 52, 0, 0, 0, 0],
            &[0, 18, 17, 0, 18, 59, 46, 4, 0, 0, 0],
            &[0, 163, 175, 116, 247, 253, 255, 213, 32, 0, 0],
            &[0, 163, 245, 199, 38, 2, 61, 247, 168, 0, 0],
            &[0, 163, 255, 50, 0, 0, 0, 177, 220, 0, 0],
            &[0, 163, 248, 0, 0, 0, 0, 155, 236, 0, 0],
            &[0, 163, 229, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŉ' / 0x149
        'ŉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[1, 35, 27, 0, 0, 0, 0, 0, 0, 0, 0],
            &[32, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0],
            &[79, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[131, 246, 26, 24, 0, 11, 56, 49, 5, 0, 0],
            &[187, 157, 98, 240, 70, 236, 253, 255, 218, 35, 0],
            &[5, 2, 98, 255, 220, 58, 3, 64, 248, 170, 0],
            &[0, 0, 98, 255, 113, 0, 0, 0, 178, 221, 0],
            &[0, 0, 98, 255, 57, 0, 0, 0, 155, 236, 0],
            &[0, 0, 98, 255, 38, 0, 0, 0, 154, 236, 0],
            &[0, 0, 98, 255, 37, 0, 0, 0, 154, 236, 0],
            &[0, 0, 98, 255, 37, 0, 0, 0, 154, 236, 0],
            &[0, 0, 98, 255, 37, 0, 0, 0, 154, 236, 0],
            &[0, 0, 98, 255, 37, 0, 0, 0, 154, 236, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 35, 8, 0, 0, 0, 16, 35, 0, 0],
            &[0, 207, 255, 125, 0, 0, 0, 120, 255, 2, 0],
            &[0, 207, 230, 231, 6, 0, 0, 120, 255, 2, 0],
            &[0, 207, 152, 243, 93, 0, 0, 120, 255, 2, 0],
            &[0, 207, 155, 146, 205, 0, 0, 120, 255, 2, 0],
            &[0, 207, 167, 36, 253, 61, 0, 120, 255, 2, 0],
            &[0, 207, 171, 0, 178, 173, 0, 120, 255, 2, 0],
            &[0, 207, 171, 0, 65, 252, 32, 119, 255, 2, 0],
            &[0, 207, 171, 0, 0, 208, 140, 115, 255, 2, 0],
            &[0, 207, 171, 0, 0, 97, 240, 119, 255, 2, 0],
            &[0, 207, 171, 0, 0, 7, 233, 199, 255, 2, 0],
            &[0, 207, 171, 0, 0, 0, 128, 255, 255, 3, 0],
            &[0, 207, 171, 0, 0, 0, 23, 255, 255, 4, 0],
            &[0, 0, 0, 0, 0, 0, 0, 128, 235, 0, 0],
            &[0, 0, 0, 0, 28, 36, 77, 235, 174, 0, 0],
            &[0, 0, 0, 0, 122, 255, 255, 202, 30, 0, 0],
            &[0, 0, 0, 0, 9, 51, 31, 1, 0, 0, 0],
        ]),
        // letter: 'ŋ' / 0x14b
        'ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 17, 0, 18, 59, 46, 4, 0, 0, 0],
            &[0, 163, 175, 116, 247, 253, 255, 213, 32, 0, 0],
            &[0, 163, 245, 199, 38, 2, 61, 247, 168, 0, 0],
            &[0, 163, 255, 50, 0, 0, 0, 177, 220, 0, 0],
            &[0, 163, 248, 0, 0, 0, 0, 155, 236, 0, 0],
            &[0, 163, 229, 0, 0, 0, 0, 180, 255, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 255, 255, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 255, 255, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 255, 255, 0, 0],
            &[0, 163, 227, 0, 0, 0, 0, 255, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 154, 236, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 232, 0, 0],
            &[0, 0, 0, 0, 0, 46, 67, 224, 191, 0, 0],
            &[0, 0, 0, 0, 0, 2, 14, 0, 0, 0, 0],
        ]),
        // letter: 'Ō' / 0x14c
        'Ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 189, 189, 189, 189, 116, 0, 0, 0],
            &[0, 0, 51, 119, 119, 119, 119, 73, 0, 0, 0],
            &[0, 0, 0, 16, 65, 76, 28, 0, 0, 0, 0],
            &[0, 0, 96, 241, 255, 255, 253, 144, 1, 0, 0],
            &[0, 59, 253, 169, 33, 17, 115, 255, 112, 0, 0],
            &[0, 179, 226, 6, 0, 0, 0, 179, 231, 2, 0],
            &[4, 244, 146, 0, 0, 0, 0, 99, 255, 50, 0],
            &[36, 255, 105, 0, 0, 0, 0, 59, 255, 89, 0],
            &[52, 255, 89, 0, 0, 0, 0, 41, 255, 106, 0],
            &[53, 255, 91, 0, 0, 0, 0, 39, 255, 104, 0],
            &[34, 255, 110, 0, 0, 0, 0, 58, 255, 83, 0],
            &[3, 243, 157, 0, 0, 0, 0, 104, 255, 39, 0],
            &[0, 169, 237, 17, 0, 0, 1, 198, 216, 0, 0],
            &[0, 44, 246, 200, 76, 64, 165, 255, 82, 0, 0],
            &[0, 0, 66, 218, 255, 255, 232, 97, 0, 0, 0],
            &[0, 0, 0, 0, 27, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ō' / 0x14d
        'ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 182, 182, 182, 182, 112, 0, 0, 0],
            &[0, 0, 54, 126, 126, 126, 126, 77, 0, 0, 0],
            &[0, 0, 0, 10, 55, 59, 15, 0, 0, 0, 0],
            &[0, 0, 106, 240, 255, 253, 247, 136, 3, 0, 0],
            &[0, 81, 255, 142, 19, 6, 102, 252, 125, 0, 0],
            &[0, 197, 211, 0, 0, 0, 0, 161, 238, 6, 0],
            &[1, 249, 141, 0, 0, 0, 0, 93, 255, 44, 0],
            &[11, 255, 120, 0, 0, 0, 0, 76, 255, 62, 0],
            &[1, 247, 142, 0, 0, 0, 0, 99, 255, 44, 0],
            &[0, 185, 218, 3, 0, 0, 0, 176, 235, 3, 0],
            &[0, 61, 252, 167, 42, 32, 132, 255, 113, 0, 0],
            &[0, 0, 77, 220, 255, 255, 239, 118, 0, 0, 0],
            &[0, 0, 0, 0, 28, 35, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => Some(&[
            &[0, 0, 75, 90, 0, 0, 59, 104, 0, 0, 0],
            &[0, 0, 52, 240, 132, 126, 226, 95, 0, 0, 0],
            &[0, 0, 0, 60, 151, 154, 80, 0, 0, 0, 0],
            &[0, 0, 0, 16, 65, 76, 28, 0, 0, 0, 0],
            &[0, 0, 96, 241, 255, 255, 253, 144, 1, 0, 0],
            &[0, 59, 253, 169, 33, 17, 115, 255, 112, 0, 0],
            &[0, 179, 226, 6, 0, 0, 0, 179, 231, 2, 0],
            &[4, 244, 146, 0, 0, 0, 0, 99, 255, 50, 0],
            &[36, 255, 105, 0, 0, 0, 0, 59, 255, 89, 0],
            &[52, 255, 89, 0, 0, 0, 0, 41, 255, 106, 0],
            &[53, 255, 91, 0, 0, 0, 0, 39, 255, 104, 0],
            &[34, 255, 110, 0, 0, 0, 0, 58, 255, 83, 0],
            &[3, 243, 157, 0, 0, 0, 0, 104, 255, 39, 0],
            &[0, 169, 237, 17, 0, 0, 1, 198, 216, 0, 0],
            &[0, 44, 246, 200, 76, 64, 165, 255, 82, 0, 0],
            &[0, 0, 66, 218, 255, 255, 232, 97, 0, 0, 0],
            &[0, 0, 0, 0, 27, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŏ' / 0x14f
        'ŏ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 72, 85, 0, 0, 56, 100, 0, 0, 0],
            &[0, 0, 55, 240, 126, 119, 223, 99, 0, 0, 0],
            &[0, 0, 0, 66, 158, 161, 86, 0, 0, 0, 0],
            &[0, 0, 0, 10, 55, 59, 15, 0, 0, 0, 0],
            &[0, 0, 106, 240, 255, 253, 247, 136, 3, 0, 0],
            &[0, 81, 255, 142, 19, 6, 102, 252, 125, 0, 0],
            &[0, 197, 211, 0, 0, 0, 0, 161, 238, 6, 0],
            &[1, 249, 141, 0, 0, 0, 0, 93, 255, 44, 0],
            &[11, 255, 120, 0, 0, 0, 0, 76, 255, 62, 0],
            &[1, 247, 142, 0, 0, 0, 0, 99, 255, 44, 0],
            &[0, 185, 218, 3, 0, 0, 0, 176, 235, 3, 0],
            &[0, 61, 252, 167, 42, 32, 132, 255, 113, 0, 0],
            &[0, 0, 77, 220, 255, 255, 239, 118, 0, 0, 0],
            &[0, 0, 0, 0, 28, 35, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ő' / 0x150
        'Ő' => Some(&[
            &[0, 0, 0, 110, 252, 58, 110, 251, 56, 0, 0],
            &[0, 0, 18, 236, 105, 19, 236, 103, 0, 0, 0],
            &[0, 0, 87, 101, 0, 89, 100, 0, 0, 0, 0],
            &[0, 0, 0, 16, 65, 76, 28, 0, 0, 0, 0],
            &[0, 0, 96, 241, 255, 255, 253, 144, 1, 0, 0],
            &[0, 59, 253, 169, 33, 17, 115, 255, 112, 0, 0],
            &[0, 179, 226, 6, 0, 0, 0, 179, 231, 2, 0],
            &[4, 244, 146, 0, 0, 0, 0, 99, 255, 50, 0],
            &[36, 255, 105, 0, 0, 0, 0, 59, 255, 89, 0],
            &[52, 255, 89, 0, 0, 0, 0, 41, 255, 106, 0],
            &[53, 255, 91, 0, 0, 0, 0, 39, 255, 104, 0],
            &[34, 255, 110, 0, 0, 0, 0, 58, 255, 83, 0],
            &[3, 243, 157, 0, 0, 0, 0, 104, 255, 39, 0],
            &[0, 169, 237, 17, 0, 0, 1, 198, 216, 0, 0],
            &[0, 44, 246, 200, 76, 64, 165, 255, 82, 0, 0],
            &[0, 0, 66, 218, 255, 255, 232, 97, 0, 0, 0],
            &[0, 0, 0, 0, 27, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 12, 6, 1, 12, 5, 0, 0],
        ]),
        // letter: 'ő' / 0x151
        'ő' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 2, 0, 5, 2, 0, 0],
            &[0, 0, 0, 106, 253, 61, 107, 252, 59, 0, 0],
            &[0, 0, 16, 234, 110, 17, 234, 108, 0, 0, 0],
            &[0, 0, 89, 108, 0, 91, 106, 0, 0, 0, 0],
            &[0, 0, 0, 10, 55, 59, 15, 0, 0, 0, 0],
            &[0, 0, 106, 240, 255, 253, 247, 136, 3, 0, 0],
            &[0, 81, 255, 142, 19, 6, 102, 252, 125, 0, 0],
            &[0, 197, 211, 0, 0, 0, 0, 161, 238, 6, 0],
            &[1, 249, 141, 0, 0, 0, 0, 93, 255, 44, 0],
            &[11, 255, 120, 0, 0, 0, 0, 76, 255, 62, 0],
            &[1, 247, 142, 0, 0, 0, 0, 99, 255, 44, 0],
            &[0, 185, 218, 3, 0, 0, 0, 176, 235, 3, 0],
            &[0, 61, 252, 167, 42, 32, 132, 255, 113, 0, 0],
            &[0, 0, 77, 220, 255, 255, 239, 118, 0, 0, 0],
            &[0, 0, 0, 0, 28, 35, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Œ' / 0x152
        'Œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 73, 43, 35, 35, 35, 24, 0],
            &[0, 8, 183, 255, 255, 255, 255, 255, 255, 176, 0],
            &[0, 128, 253, 88, 22, 204, 205, 56, 56, 39, 0],
            &[0, 229, 174, 0, 0, 186, 191, 0, 0, 0, 0],
            &[32, 255, 112, 0, 0, 186, 191, 0, 0, 0, 0],
            &[64, 255, 82, 0, 0, 186, 193, 5, 5, 1, 0],
            &[77, 255, 68, 0, 0, 186, 255, 255, 255, 93, 0],
            &[74, 255, 67, 0, 0, 186, 212, 82, 82, 30, 0],
            &[59, 255, 81, 0, 0, 186, 191, 0, 0, 0, 0],
            &[23, 255, 115, 0, 0, 186, 191, 0, 0, 0, 0],
            &[0, 219, 184, 0, 0, 186, 191, 0, 0, 0, 0],
            &[0, 116, 255, 115, 62, 215, 214, 92, 92, 63, 0],
            &[0, 3, 162, 255, 255, 255, 255, 255, 255, 176, 0],
            &[0, 0, 0, 17, 33, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'œ' / 0x153
        'œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 64, 6, 0, 16, 60, 13, 0, 0],
            &[0, 127, 255, 247, 207, 48, 234, 243, 227, 24, 0],
            &[44, 255, 87, 7, 212, 239, 141, 7, 208, 134, 0],
            &[114, 235, 1, 0, 120, 255, 51, 0, 132, 193, 0],
            &[140, 201, 0, 0, 92, 255, 115, 99, 168, 215, 0],
            &[155, 191, 0, 0, 79, 255, 184, 178, 178, 153, 0],
            &[141, 205, 0, 0, 93, 255, 34, 0, 0, 0, 0],
            &[99, 242, 4, 0, 124, 255, 68, 0, 0, 0, 0],
            &[22, 248, 120, 37, 220, 224, 182, 14, 35, 62, 0],
            &[0, 95, 247, 255, 182, 27, 212, 255, 248, 94, 0],
            &[0, 0, 12, 31, 0, 0, 2, 33, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŕ' / 0x154
        'Ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 75, 255, 131, 0, 0, 0],
            &[0, 0, 0, 0, 11, 222, 161, 1, 0, 0, 0],
            &[0, 0, 0, 0, 82, 129, 2, 0, 0, 0, 0],
            &[0, 13, 35, 35, 32, 14, 0, 0, 0, 0, 0],
            &[0, 98, 255, 255, 255, 255, 214, 87, 0, 0, 0],
            &[0, 98, 255, 78, 53, 82, 195, 254, 68, 0, 0],
            &[0, 98, 255, 37, 0, 0, 15, 251, 160, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 233, 177, 0, 0],
            &[0, 98, 255, 37, 0, 0, 32, 255, 134, 0, 0],
            &[0, 98, 255, 100, 77, 107, 217, 226, 23, 0, 0],
            &[0, 98, 255, 255, 255, 255, 185, 16, 0, 0, 0],
            &[0, 98, 255, 44, 9, 173, 232, 15, 0, 0, 0],
            &[0, 98, 255, 37, 0, 31, 246, 146, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 127, 252, 47, 0, 0],
            &[0, 98, 255, 37, 0, 0, 9, 223, 196, 1, 0],
            &[0, 98, 255, 37, 0, 0, 0, 82, 255, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 10, 0, 0, 0],
        ]),
        // letter: 'ŕ' / 0x155
        'ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 5, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 153, 250, 59, 0, 0],
            &[0, 0, 0, 0, 0, 58, 251, 90, 0, 0, 0],
            &[0, 0, 0, 0, 0, 141, 82, 0, 0, 0, 0],
            &[0, 19, 28, 28, 28, 1, 5, 58, 55, 8, 0],
            &[0, 167, 255, 255, 255, 51, 212, 255, 255, 149, 0],
            &[0, 0, 11, 71, 255, 216, 163, 33, 41, 37, 0],
            &[0, 0, 0, 44, 255, 211, 2, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 97, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 95, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 95, 0, 0, 0, 0, 0],
            &[0, 15, 39, 94, 255, 134, 42, 18, 0, 0, 0],
            &[0, 242, 255, 255, 255, 255, 255, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŗ' / 0x156
        'Ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 35, 35, 32, 14, 0, 0, 0, 0, 0],
            &[0, 98, 255, 255, 255, 255, 214, 87, 0, 0, 0],
            &[0, 98, 255, 78, 53, 82, 195, 254, 68, 0, 0],
            &[0, 98, 255, 37, 0, 0, 15, 251, 160, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 233, 177, 0, 0],
            &[0, 98, 255, 37, 0, 0, 32, 255, 134, 0, 0],
            &[0, 98, 255, 100, 77, 107, 217, 226, 23, 0, 0],
            &[0, 98, 255, 255, 255, 255, 185, 16, 0, 0, 0],
            &[0, 98, 255, 44, 9, 173, 232, 15, 0, 0, 0],
            &[0, 98, 255, 37, 0, 31, 246, 146, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 127, 252, 47, 0, 0],
            &[0, 98, 255, 37, 0, 0, 9, 223, 196, 1, 0],
            &[0, 98, 255, 37, 0, 0, 0, 82, 255, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 246, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 205, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 140, 52, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŗ' / 0x157
        'ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 19, 28, 28, 28, 1, 5, 58, 55, 8, 0],
            &[0, 167, 255, 255, 255, 51, 212, 255, 255, 149, 0],
            &[0, 0, 11, 71, 255, 216, 163, 33, 41, 37, 0],
            &[0, 0, 0, 44, 255, 211, 2, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 97, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 95, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 95, 0, 0, 0, 0, 0],
            &[0, 15, 39, 94, 255, 134, 42, 18, 0, 0, 0],
            &[0, 242, 255, 255, 255, 255, 255, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 48, 246, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 111, 204, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 140, 52, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ř' / 0x158
        'Ř' => Some(&[
            &[0, 0, 116, 209, 33, 16, 184, 164, 0, 0, 0],
            &[0, 0, 0, 145, 228, 210, 183, 4, 0, 0, 0],
            &[0, 0, 0, 4, 153, 167, 18, 0, 0, 0, 0],
            &[0, 13, 35, 35, 32, 14, 0, 0, 0, 0, 0],
            &[0, 98, 255, 255, 255, 255, 214, 87, 0, 0, 0],
            &[0, 98, 255, 78, 53, 82, 195, 254, 68, 0, 0],
            &[0, 98, 255, 37, 0, 0, 15, 251, 160, 0, 0],
            &[0, 98, 255, 37, 0, 0, 0, 233, 177, 0, 0],
            &[0, 98, 255, 37, 0, 0, 32, 255, 134, 0, 0],
            &[0, 98, 255, 100, 77, 107, 217, 226, 23, 0, 0],
            &[0, 98, 255, 255, 255, 255, 185, 16, 0, 0, 0],
            &[0, 98, 255, 44, 9, 173, 232, 15, 0, 0, 0],
            &[0, 98, 255, 37, 0, 31, 246, 146, 0, 0, 0],
            &[0, 98, 255, 37, 0, 0, 127, 252, 47, 0, 0],
            &[0, 98, 255, 37, 0, 0, 9, 223, 196, 1, 0],
            &[0, 98, 255, 37, 0, 0, 0, 82, 255, 95, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 3, 0, 0, 1, 11, 0, 0, 0],
        ]),
        // letter: 'ř' / 0x159
        'ř' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 2, 3, 0, 0],
            &[0, 0, 5, 196, 149, 4, 50, 222, 89, 0, 0],
            &[0, 0, 0, 17, 215, 189, 242, 111, 0, 0, 0],
            &[0, 0, 0, 0, 44, 178, 137, 0, 0, 0, 0],
            &[0, 19, 28, 28, 28, 1, 5, 58, 55, 8, 0],
            &[0, 167, 255, 255, 255, 51, 212, 255, 255, 149, 0],
            &[0, 0, 11, 71, 255, 216, 163, 33, 41, 37, 0],
            &[0, 0, 0, 44, 255, 211, 2, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 97, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 95, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 95, 0, 0, 0, 0, 0],
            &[0, 15, 39, 94, 255, 134, 42, 18, 0, 0, 0],
            &[0, 242, 255, 255, 255, 255, 255, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ś' / 0x15a
        'Ś' => Some(&[
            &[0, 0, 0, 0, 0, 14, 233, 206, 7, 0, 0],
            &[0, 0, 0, 0, 0, 152, 220, 25, 0, 0, 0],
            &[0, 0, 0, 0, 27, 161, 25, 0, 0, 0, 0],
            &[0, 0, 0, 2, 49, 72, 52, 15, 0, 0, 0],
            &[0, 0, 60, 224, 255, 255, 255, 250, 126, 0, 0],
            &[0, 10, 235, 220, 66, 22, 40, 100, 61, 0, 0],
            &[0, 59, 255, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 54, 255, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 219, 240, 90, 2, 0, 0, 0, 0, 0],
            &[0, 0, 34, 197, 255, 218, 98, 5, 0, 0, 0],
            &[0, 0, 0, 0, 74, 196, 255, 215, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 242, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 166, 245, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 194, 231, 0, 0],
            &[0, 110, 134, 69, 41, 52, 137, 255, 138, 0, 0],
            &[0, 89, 221, 255, 255, 255, 240, 136, 4, 0, 0],
            &[0, 0, 0, 12, 35, 31, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 12, 1, 0, 0],
        ]),
        // letter: 'ś' / 0x15b
        'ś' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 254, 157, 0, 0, 0],
            &[0, 0, 0, 0, 4, 204, 184, 6, 0, 0, 0],
            &[0, 0, 0, 0, 68, 147, 7, 0, 0, 0, 0],
            &[0, 0, 0, 6, 52, 66, 45, 4, 0, 0, 0],
            &[0, 0, 84, 237, 255, 255, 255, 238, 28, 0, 0],
            &[0, 1, 239, 180, 25, 3, 27, 86, 0, 0, 0],
            &[0, 5, 252, 149, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 127, 255, 178, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 194, 255, 199, 56, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 199, 248, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 255, 98, 0, 0],
            &[0, 42, 139, 59, 28, 40, 154, 255, 52, 0, 0],
            &[0, 34, 209, 255, 255, 255, 234, 106, 0, 0, 0],
            &[0, 0, 0, 13, 36, 29, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => Some(&[
            &[0, 0, 0, 2, 189, 255, 142, 0, 0, 0, 0],
            &[0, 0, 0, 142, 210, 71, 236, 96, 0, 0, 0],
            &[0, 0, 49, 152, 15, 0, 36, 162, 25, 0, 0],
            &[0, 0, 0, 2, 49, 72, 52, 15, 0, 0, 0],
            &[0, 0, 60, 224, 255, 255, 255, 250, 126, 0, 0],
            &[0, 10, 235, 220, 66, 22, 40, 100, 61, 0, 0],
            &[0, 59, 255, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 54, 255, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 219, 240, 90, 2, 0, 0, 0, 0, 0],
            &[0, 0, 34, 197, 255, 218, 98, 5, 0, 0, 0],
            &[0, 0, 0, 0, 74, 196, 255, 215, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 242, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 166, 245, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 194, 231, 0, 0],
            &[0, 110, 134, 69, 41, 52, 137, 255, 138, 0, 0],
            &[0, 89, 221, 255, 255, 255, 240, 136, 4, 0, 0],
            &[0, 0, 0, 12, 35, 31, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 12, 2, 0, 0, 0, 0],
        ]),
        // letter: 'ŝ' / 0x15d
        'ŝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 228, 255, 77, 0, 0, 0, 0],
            &[0, 0, 6, 191, 172, 105, 238, 44, 0, 0, 0],
            &[0, 0, 92, 131, 3, 0, 77, 153, 3, 0, 0],
            &[0, 0, 0, 6, 52, 66, 45, 4, 0, 0, 0],
            &[0, 0, 84, 237, 255, 255, 255, 238, 28, 0, 0],
            &[0, 1, 239, 180, 25, 3, 27, 86, 0, 0, 0],
            &[0, 5, 252, 149, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 127, 255, 178, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 194, 255, 199, 56, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 199, 248, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 255, 98, 0, 0],
            &[0, 42, 139, 59, 28, 40, 154, 255, 52, 0, 0],
            &[0, 34, 209, 255, 255, 255, 234, 106, 0, 0, 0],
            &[0, 0, 0, 13, 36, 29, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ş' / 0x15e
        'Ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 49, 72, 52, 15, 0, 0, 0],
            &[0, 0, 60, 224, 255, 255, 255, 250, 126, 0, 0],
            &[0, 10, 235, 220, 66, 22, 40, 100, 61, 0, 0],
            &[0, 59, 255, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 54, 255, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 219, 240, 90, 2, 0, 0, 0, 0, 0],
            &[0, 0, 34, 197, 255, 218, 98, 5, 0, 0, 0],
            &[0, 0, 0, 0, 74, 196, 255, 215, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 242, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 166, 245, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 194, 231, 0, 0],
            &[0, 110, 134, 69, 41, 52, 137, 255, 138, 0, 0],
            &[0, 89, 221, 255, 255, 255, 240, 136, 4, 0, 0],
            &[0, 0, 0, 12, 189, 137, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 228, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 169, 164, 0, 0, 0, 0],
            &[0, 0, 0, 7, 11, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ş' / 0x15f
        'ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 52, 66, 45, 4, 0, 0, 0],
            &[0, 0, 84, 237, 255, 255, 255, 238, 28, 0, 0],
            &[0, 1, 239, 180, 25, 3, 27, 86, 0, 0, 0],
            &[0, 5, 252, 149, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 127, 255, 178, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 194, 255, 199, 56, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 199, 248, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 255, 98, 0, 0],
            &[0, 42, 139, 59, 28, 40, 154, 255, 52, 0, 0],
            &[0, 34, 209, 255, 255, 255, 234, 106, 0, 0, 0],
            &[0, 0, 0, 13, 168, 156, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 134, 227, 99, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 147, 186, 0, 0, 0, 0],
            &[0, 0, 0, 5, 13, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Š' / 0x160
        'Š' => Some(&[
            &[0, 0, 47, 228, 83, 0, 121, 219, 24, 0, 0],
            &[0, 0, 0, 67, 248, 186, 234, 35, 0, 0, 0],
            &[0, 0, 0, 0, 102, 171, 69, 0, 0, 0, 0],
            &[0, 0, 0, 2, 49, 72, 52, 15, 0, 0, 0],
            &[0, 0, 60, 224, 255, 255, 255, 250, 126, 0, 0],
            &[0, 10, 235, 220, 66, 22, 40, 100, 61, 0, 0],
            &[0, 59, 255, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 54, 255, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 219, 240, 90, 2, 0, 0, 0, 0, 0],
            &[0, 0, 34, 197, 255, 218, 98, 5, 0, 0, 0],
            &[0, 0, 0, 0, 74, 196, 255, 215, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 242, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 166, 245, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 194, 231, 0, 0],
            &[0, 110, 134, 69, 41, 52, 137, 255, 138, 0, 0],
            &[0, 89, 221, 255, 255, 255, 240, 136, 4, 0, 0],
            &[0, 0, 0, 12, 35, 31, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 7, 0, 0, 0, 9, 3, 0, 0],
        ]),
        // letter: 'š' / 0x161
        'š' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 1, 0, 0, 0, 5, 0, 0, 0],
            &[0, 0, 99, 215, 40, 8, 163, 187, 3, 0, 0],
            &[0, 0, 0, 129, 234, 197, 203, 11, 0, 0, 0],
            &[0, 0, 0, 1, 149, 177, 32, 0, 0, 0, 0],
            &[0, 0, 0, 6, 52, 66, 45, 4, 0, 0, 0],
            &[0, 0, 84, 237, 255, 255, 255, 238, 28, 0, 0],
            &[0, 1, 239, 180, 25, 3, 27, 86, 0, 0, 0],
            &[0, 5, 252, 149, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 127, 255, 178, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 194, 255, 199, 56, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 199, 248, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 255, 98, 0, 0],
            &[0, 42, 139, 59, 28, 40, 154, 255, 52, 0, 0],
            &[0, 34, 209, 255, 255, 255, 234, 106, 0, 0, 0],
            &[0, 0, 0, 13, 36, 29, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ţ' / 0x162
        'Ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[8, 35, 35, 35, 35, 35, 35, 35, 35, 15, 0],
            &[60, 255, 255, 255, 255, 255, 255, 255, 255, 111, 0],
            &[13, 56, 56, 56, 189, 229, 56, 56, 56, 24, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 177, 235, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 105, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 228, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 169, 164, 0, 0, 0, 0],
            &[0, 0, 0, 7, 11, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ţ' / 0x163
        'ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 133, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 148, 247, 28, 28, 28, 14, 0],
            &[0, 10, 243, 255, 255, 255, 255, 255, 255, 132, 0],
            &[0, 2, 41, 41, 162, 248, 41, 41, 41, 21, 0],
            &[0, 0, 0, 0, 144, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 144, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 144, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 144, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 136, 255, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 255, 154, 39, 31, 49, 0],
            &[0, 0, 0, 0, 2, 143, 251, 255, 255, 191, 0],
            &[0, 0, 0, 0, 0, 0, 96, 208, 29, 2, 0],
            &[0, 0, 0, 0, 0, 0, 103, 221, 136, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 103, 230, 0, 0],
            &[0, 0, 0, 0, 0, 2, 15, 0, 0, 0, 0],
        ]),
        // letter: 'Ť' / 0x164
        'Ť' => Some(&[
            &[0, 0, 116, 209, 33, 16, 184, 164, 0, 0, 0],
            &[0, 0, 0, 145, 228, 210, 183, 4, 0, 0, 0],
            &[0, 0, 0, 4, 153, 167, 18, 0, 0, 0, 0],
            &[8, 35, 35, 35, 35, 35, 35, 35, 35, 15, 0],
            &[60, 255, 255, 255, 255, 255, 255, 255, 255, 111, 0],
            &[13, 56, 56, 56, 189, 229, 56, 56, 56, 24, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 3, 0, 0, 1, 11, 0, 0, 0],
        ]),
        // letter: 'ť' / 0x165
        'ť' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 19, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 223, 138, 0],
            &[0, 0, 0, 0, 40, 133, 0, 22, 250, 32, 0],
            &[0, 0, 0, 0, 100, 246, 0, 46, 103, 0, 0],
            &[0, 0, 0, 15, 148, 247, 28, 28, 28, 14, 0],
            &[0, 10, 243, 255, 255, 255, 255, 255, 255, 132, 0],
            &[0, 2, 41, 41, 162, 248, 41, 41, 41, 21, 0],
            &[0, 0, 0, 0, 144, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 144, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 144, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 144, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 136, 255, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 255, 154, 39, 31, 49, 0],
            &[0, 0, 0, 0, 2, 143, 249, 255, 255, 191, 0],
            &[0, 0, 0, 0, 0, 0, 8, 37, 29, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŧ' / 0x166
        'Ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[8, 35, 35, 35, 35, 35, 35, 35, 35, 15, 0],
            &[60, 255, 255, 255, 255, 255, 255, 255, 255, 111, 0],
            &[13, 56, 56, 56, 189, 229, 56, 56, 56, 24, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 36, 200, 200, 255, 255, 200, 200, 79, 0, 0],
            &[0, 22, 121, 121, 255, 255, 121, 121, 48, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŧ' / 0x167
        'ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 133, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 148, 247, 28, 28, 28, 14, 0],
            &[0, 10, 243, 255, 255, 255, 255, 255, 255, 132, 0],
            &[0, 2, 41, 41, 162, 248, 41, 41, 41, 21, 0],
            &[0, 0, 0, 0, 144, 246, 0, 0, 0, 0, 0],
            &[0, 10, 225, 225, 255, 255, 225, 225, 225, 174, 0],
            &[0, 2, 61, 61, 206, 255, 61, 61, 61, 47, 0],
            &[0, 0, 0, 0, 144, 246, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 136, 255, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 255, 154, 39, 31, 49, 0],
            &[0, 0, 0, 0, 2, 143, 249, 255, 255, 191, 0],
            &[0, 0, 0, 0, 0, 0, 8, 37, 29, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ũ' / 0x168
        'Ũ' => Some(&[
            &[0, 0, 7, 104, 90, 5, 0, 107, 13, 0, 0],
            &[0, 0, 154, 213, 225, 217, 162, 219, 1, 0, 0],
            &[0, 0, 150, 11, 6, 106, 152, 43, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 18, 35, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 206, 186, 0, 0, 0, 0, 135, 255, 1, 0],
            &[0, 191, 213, 0, 0, 0, 0, 159, 242, 0, 0],
            &[0, 146, 248, 18, 0, 0, 0, 211, 195, 0, 0],
            &[0, 47, 253, 193, 71, 61, 159, 255, 89, 0, 0],
            &[0, 0, 85, 227, 255, 255, 238, 114, 0, 0, 0],
            &[0, 0, 0, 2, 27, 30, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ũ' / 0x169
        'ũ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 97, 83, 3, 0, 101, 13, 0, 0],
            &[0, 0, 151, 219, 230, 212, 155, 221, 1, 0, 0],
            &[0, 0, 155, 13, 8, 113, 158, 48, 0, 0, 0],
            &[0, 20, 23, 0, 0, 0, 0, 19, 24, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 177, 214, 0, 0],
            &[0, 185, 207, 0, 0, 0, 0, 194, 214, 0, 0],
            &[0, 166, 233, 1, 0, 0, 11, 242, 214, 0, 0],
            &[0, 107, 255, 129, 31, 43, 174, 243, 214, 0, 0],
            &[0, 5, 163, 254, 255, 249, 143, 122, 214, 0, 0],
            &[0, 0, 0, 10, 35, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ū' / 0x16a
        'Ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 189, 189, 189, 189, 116, 0, 0, 0],
            &[0, 0, 51, 119, 119, 119, 119, 73, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 18, 35, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 206, 186, 0, 0, 0, 0, 135, 255, 1, 0],
            &[0, 191, 213, 0, 0, 0, 0, 159, 242, 0, 0],
            &[0, 146, 248, 18, 0, 0, 0, 211, 195, 0, 0],
            &[0, 47, 253, 193, 71, 61, 159, 255, 89, 0, 0],
            &[0, 0, 85, 227, 255, 255, 238, 114, 0, 0, 0],
            &[0, 0, 0, 2, 27, 30, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ū' / 0x16b
        'ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 182, 182, 182, 182, 112, 0, 0, 0],
            &[0, 0, 54, 126, 126, 126, 126, 77, 0, 0, 0],
            &[0, 20, 23, 0, 0, 0, 0, 19, 24, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 177, 214, 0, 0],
            &[0, 185, 207, 0, 0, 0, 0, 194, 214, 0, 0],
            &[0, 166, 233, 1, 0, 0, 11, 242, 214, 0, 0],
            &[0, 107, 255, 129, 31, 43, 174, 243, 214, 0, 0],
            &[0, 5, 163, 254, 255, 249, 143, 122, 214, 0, 0],
            &[0, 0, 0, 10, 35, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => Some(&[
            &[0, 0, 75, 90, 0, 0, 59, 104, 0, 0, 0],
            &[0, 0, 52, 240, 132, 126, 226, 95, 0, 0, 0],
            &[0, 0, 0, 60, 151, 154, 80, 0, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 18, 35, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 206, 186, 0, 0, 0, 0, 135, 255, 1, 0],
            &[0, 191, 213, 0, 0, 0, 0, 159, 242, 0, 0],
            &[0, 146, 248, 18, 0, 0, 0, 211, 195, 0, 0],
            &[0, 47, 253, 193, 71, 61, 159, 255, 89, 0, 0],
            &[0, 0, 85, 227, 255, 255, 238, 114, 0, 0, 0],
            &[0, 0, 0, 2, 27, 30, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŭ' / 0x16d
        'ŭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 72, 85, 0, 0, 56, 100, 0, 0, 0],
            &[0, 0, 55, 240, 126, 119, 223, 99, 0, 0, 0],
            &[0, 0, 0, 66, 158, 161, 86, 0, 0, 0, 0],
            &[0, 20, 23, 0, 0, 0, 0, 19, 24, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 177, 214, 0, 0],
            &[0, 185, 207, 0, 0, 0, 0, 194, 214, 0, 0],
            &[0, 166, 233, 1, 0, 0, 11, 242, 214, 0, 0],
            &[0, 107, 255, 129, 31, 43, 174, 243, 214, 0, 0],
            &[0, 5, 163, 254, 255, 249, 143, 122, 214, 0, 0],
            &[0, 0, 0, 10, 35, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ů' / 0x16e
        'Ů' => Some(&[
            &[0, 0, 0, 179, 95, 49, 229, 0, 0, 0, 0],
            &[0, 0, 0, 160, 148, 112, 209, 0, 0, 0, 0],
            &[0, 0, 0, 21, 142, 153, 39, 0, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 18, 35, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 206, 186, 0, 0, 0, 0, 135, 255, 1, 0],
            &[0, 191, 213, 0, 0, 0, 0, 159, 242, 0, 0],
            &[0, 146, 248, 18, 0, 0, 0, 211, 195, 0, 0],
            &[0, 47, 253, 193, 71, 61, 159, 255, 89, 0, 0],
            &[0, 0, 85, 227, 255, 255, 238, 114, 0, 0, 0],
            &[0, 0, 0, 2, 27, 30, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 52, 204, 213, 80, 0, 0, 0, 0],
        ]),
        // letter: 'ů' / 0x16f
        'ů' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 48, 198, 208, 75, 0, 0, 0, 0],
            &[0, 0, 0, 178, 100, 54, 228, 0, 0, 0, 0],
            &[0, 0, 0, 163, 143, 106, 212, 0, 0, 0, 0],
            &[0, 0, 0, 24, 148, 159, 43, 0, 0, 0, 0],
            &[0, 20, 23, 0, 0, 0, 0, 19, 24, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 177, 214, 0, 0],
            &[0, 185, 207, 0, 0, 0, 0, 194, 214, 0, 0],
            &[0, 166, 233, 1, 0, 0, 11, 242, 214, 0, 0],
            &[0, 107, 255, 129, 31, 43, 174, 243, 214, 0, 0],
            &[0, 5, 163, 254, 255, 249, 143, 122, 214, 0, 0],
            &[0, 0, 0, 10, 35, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ű' / 0x170
        'Ű' => Some(&[
            &[0, 0, 0, 110, 252, 58, 110, 251, 56, 0, 0],
            &[0, 0, 18, 236, 105, 19, 236, 103, 0, 0, 0],
            &[0, 0, 87, 101, 0, 89, 100, 0, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 18, 35, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 206, 186, 0, 0, 0, 0, 135, 255, 1, 0],
            &[0, 191, 213, 0, 0, 0, 0, 159, 242, 0, 0],
            &[0, 146, 248, 18, 0, 0, 0, 211, 195, 0, 0],
            &[0, 47, 253, 193, 71, 61, 159, 255, 89, 0, 0],
            &[0, 0, 85, 227, 255, 255, 238, 114, 0, 0, 0],
            &[0, 0, 0, 2, 27, 30, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 12, 6, 1, 12, 5, 0, 0],
        ]),
        // letter: 'ű' / 0x171
        'ű' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 2, 0, 5, 2, 0, 0],
            &[0, 0, 0, 106, 253, 61, 107, 252, 59, 0, 0],
            &[0, 0, 16, 234, 110, 17, 234, 108, 0, 0, 0],
            &[0, 0, 89, 108, 0, 91, 106, 0, 0, 0, 0],
            &[0, 20, 23, 0, 0, 0, 0, 19, 24, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 177, 214, 0, 0],
            &[0, 185, 207, 0, 0, 0, 0, 194, 214, 0, 0],
            &[0, 166, 233, 1, 0, 0, 11, 242, 214, 0, 0],
            &[0, 107, 255, 129, 31, 43, 174, 243, 214, 0, 0],
            &[0, 5, 163, 254, 255, 249, 143, 122, 214, 0, 0],
            &[0, 0, 0, 10, 35, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ų' / 0x172
        'Ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 25, 0, 0, 0, 0, 18, 35, 0, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 207, 184, 0, 0, 0, 0, 133, 255, 2, 0],
            &[0, 206, 186, 0, 0, 0, 0, 135, 255, 1, 0],
            &[0, 191, 213, 0, 0, 0, 0, 159, 242, 0, 0],
            &[0, 146, 248, 18, 0, 0, 0, 211, 195, 0, 0],
            &[0, 47, 253, 193, 71, 61, 159, 255, 89, 0, 0],
            &[0, 0, 85, 227, 255, 255, 255, 114, 0, 0, 0],
            &[0, 0, 0, 2, 92, 244, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 233, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 241, 146, 58, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 178, 130, 0, 0, 0, 0],
        ]),
        // letter: 'ų' / 0x173
        'ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 23, 0, 0, 0, 0, 19, 24, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 176, 214, 0, 0],
            &[0, 185, 205, 0, 0, 0, 0, 177, 214, 0, 0],
            &[0, 185, 207, 0, 0, 0, 0, 194, 214, 0, 0],
            &[0, 166, 233, 1, 0, 0, 11, 242, 214, 0, 0],
            &[0, 107, 255, 129, 31, 43, 174, 243, 214, 0, 0],
            &[0, 5, 163, 254, 255, 249, 143, 136, 237, 0, 0],
            &[0, 0, 0, 10, 35, 11, 9, 203, 83, 0, 0],
            &[0, 0, 0, 0, 0, 0, 112, 189, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 143, 214, 77, 15, 0],
            &[0, 0, 0, 0, 0, 0, 27, 160, 179, 27, 0],
        ]),
        // letter: 'Ŵ' / 0x174
        'Ŵ' => Some(&[
            &[0, 0, 0, 33, 241, 253, 62, 0, 0, 0, 0],
            &[0, 0, 14, 210, 148, 116, 233, 35, 0, 0, 0],
            &[0, 0, 105, 112, 0, 0, 86, 138, 0, 0, 0],
            &[20, 29, 0, 0, 0, 0, 0, 0, 22, 27, 0],
            &[128, 229, 0, 0, 0, 0, 0, 0, 177, 181, 0],
            &[100, 251, 2, 0, 0, 0, 0, 0, 198, 156, 0],
            &[71, 255, 22, 0, 17, 24, 0, 0, 220, 130, 0],
            &[42, 255, 46, 0, 193, 247, 6, 0, 242, 104, 0],
            &[14, 255, 71, 6, 247, 231, 59, 9, 255, 78, 0],
            &[0, 241, 95, 56, 231, 163, 121, 30, 255, 52, 0],
            &[0, 212, 117, 116, 176, 107, 182, 50, 255, 27, 0],
            &[0, 184, 137, 174, 117, 46, 240, 71, 253, 3, 0],
            &[0, 155, 154, 230, 58, 2, 238, 130, 231, 0, 0],
            &[0, 126, 196, 248, 6, 0, 179, 196, 205, 0, 0],
            &[0, 98, 249, 195, 0, 0, 117, 249, 179, 0, 0],
            &[0, 69, 255, 136, 0, 0, 55, 255, 153, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 10, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŵ' / 0x175
        'ŵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 237, 253, 61, 0, 0, 0, 0],
            &[0, 0, 11, 204, 158, 119, 233, 34, 0, 0, 0],
            &[0, 0, 104, 121, 1, 0, 89, 144, 0, 0, 0],
            &[20, 20, 0, 0, 0, 0, 0, 0, 13, 26, 0],
            &[161, 202, 0, 0, 0, 0, 0, 0, 139, 216, 0],
            &[120, 239, 0, 0, 81, 99, 0, 0, 180, 174, 0],
            &[79, 255, 21, 10, 247, 255, 46, 0, 220, 132, 0],
            &[37, 255, 59, 75, 241, 201, 116, 8, 253, 90, 0],
            &[3, 249, 96, 148, 180, 136, 187, 44, 255, 48, 0],
            &[0, 211, 128, 214, 111, 71, 247, 85, 253, 8, 0],
            &[0, 170, 165, 254, 43, 11, 250, 160, 219, 0, 0],
            &[0, 129, 230, 232, 0, 0, 197, 230, 177, 0, 0],
            &[0, 88, 255, 166, 0, 0, 131, 255, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŷ' / 0x176
        'Ŷ' => Some(&[
            &[0, 0, 0, 33, 241, 253, 62, 0, 0, 0, 0],
            &[0, 0, 14, 210, 148, 116, 233, 35, 0, 0, 0],
            &[0, 0, 105, 112, 0, 0, 86, 138, 0, 0, 0],
            &[13, 35, 10, 0, 0, 0, 0, 2, 35, 20, 0],
            &[28, 248, 146, 0, 0, 0, 0, 82, 255, 71, 0],
            &[0, 146, 245, 19, 0, 0, 0, 201, 197, 0, 0],
            &[0, 25, 246, 127, 0, 0, 65, 255, 66, 0, 0],
            &[0, 0, 141, 235, 10, 0, 184, 192, 0, 0, 0],
            &[0, 0, 22, 244, 108, 48, 255, 62, 0, 0, 0],
            &[0, 0, 0, 136, 222, 170, 187, 0, 0, 0, 0],
            &[0, 0, 0, 19, 242, 255, 57, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 10, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŷ' / 0x177
        'ŷ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 239, 252, 58, 0, 0, 0, 0],
            &[0, 0, 12, 207, 154, 122, 231, 32, 0, 0, 0],
            &[0, 0, 107, 119, 0, 0, 92, 141, 0, 0, 0],
            &[6, 28, 8, 0, 0, 0, 0, 4, 28, 11, 0],
            &[10, 239, 130, 0, 0, 0, 0, 89, 255, 48, 0],
            &[0, 144, 229, 4, 0, 0, 0, 175, 209, 0, 0],
            &[0, 39, 254, 82, 0, 0, 13, 247, 114, 0, 0],
            &[0, 0, 187, 186, 0, 0, 90, 251, 23, 0, 0],
            &[0, 0, 81, 254, 36, 0, 176, 180, 0, 0, 0],
            &[0, 0, 3, 226, 135, 12, 247, 85, 0, 0, 0],
            &[0, 0, 0, 124, 226, 88, 238, 7, 0, 0, 0],
            &[0, 0, 0, 23, 249, 213, 151, 0, 0, 0, 0],
            &[0, 0, 0, 0, 167, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 240, 115, 0, 0, 0, 0, 0],
            &[0, 20, 29, 169, 238, 14, 0, 0, 0, 0, 0],
            &[0, 22, 35, 5, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ÿ' / 0x178
        'Ÿ' => Some(&[
            &[0, 0, 61, 103, 0, 0, 76, 88, 0, 0, 0],
            &[0, 0, 173, 247, 4, 0, 201, 224, 0, 0, 0],
            &[0, 0, 33, 61, 0, 0, 43, 51, 0, 0, 0],
            &[13, 35, 10, 0, 0, 0, 0, 2, 35, 20, 0],
            &[28, 248, 146, 0, 0, 0, 0, 82, 255, 71, 0],
            &[0, 146, 245, 19, 0, 0, 0, 201, 197, 0, 0],
            &[0, 25, 246, 127, 0, 0, 65, 255, 66, 0, 0],
            &[0, 0, 141, 235, 10, 0, 184, 192, 0, 0, 0],
            &[0, 0, 22, 244, 108, 48, 255, 62, 0, 0, 0],
            &[0, 0, 0, 136, 222, 170, 187, 0, 0, 0, 0],
            &[0, 0, 0, 19, 242, 255, 57, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ź' / 0x179
        'Ź' => Some(&[
            &[0, 0, 0, 0, 0, 31, 247, 182, 0, 0, 0],
            &[0, 0, 0, 0, 0, 181, 202, 13, 0, 0, 0],
            &[0, 0, 0, 0, 47, 153, 14, 0, 0, 0, 0],
            &[0, 1, 35, 35, 35, 35, 35, 35, 20, 0, 0],
            &[0, 11, 255, 255, 255, 255, 255, 255, 149, 0, 0],
            &[0, 2, 48, 48, 48, 48, 94, 255, 88, 0, 0],
            &[0, 0, 0, 0, 0, 0, 187, 200, 1, 0, 0],
            &[0, 0, 0, 0, 0, 75, 255, 60, 0, 0, 0],
            &[0, 0, 0, 0, 4, 214, 173, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 250, 36, 0, 0, 0, 0],
            &[0, 0, 0, 15, 234, 144, 0, 0, 0, 0, 0],
            &[0, 0, 0, 137, 239, 19, 0, 0, 0, 0, 0],
            &[0, 0, 32, 248, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 168, 222, 7, 0, 0, 0, 0, 0, 0],
            &[0, 54, 255, 164, 92, 92, 92, 92, 53, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 149, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 12, 0, 0, 0],
        ]),
        // letter: 'ź' / 0x17a
        'ź' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 248, 179, 0, 0, 0],
            &[0, 0, 0, 0, 0, 185, 200, 12, 0, 0, 0],
            &[0, 0, 0, 0, 53, 156, 13, 0, 0, 0, 0],
            &[0, 5, 28, 28, 28, 28, 28, 28, 15, 0, 0],
            &[0, 50, 255, 255, 255, 255, 255, 255, 136, 0, 0],
            &[0, 3, 15, 15, 15, 15, 130, 250, 59, 0, 0],
            &[0, 0, 0, 0, 0, 63, 251, 110, 0, 0, 0],
            &[0, 0, 0, 0, 25, 232, 165, 0, 0, 0, 0],
            &[0, 0, 0, 4, 195, 210, 10, 0, 0, 0, 0],
            &[0, 0, 0, 143, 240, 36, 0, 0, 0, 0, 0],
            &[0, 0, 86, 253, 77, 0, 0, 0, 0, 0, 0],
            &[0, 40, 243, 188, 74, 74, 74, 74, 50, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ż' / 0x17b
        'Ż' => Some(&[
            &[0, 0, 0, 0, 114, 215, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 27, 0, 0, 0, 0, 0],
            &[0, 1, 35, 35, 35, 35, 35, 35, 20, 0, 0],
            &[0, 11, 255, 255, 255, 255, 255, 255, 149, 0, 0],
            &[0, 2, 48, 48, 48, 48, 94, 255, 88, 0, 0],
            &[0, 0, 0, 0, 0, 0, 187, 200, 1, 0, 0],
            &[0, 0, 0, 0, 0, 75, 255, 60, 0, 0, 0],
            &[0, 0, 0, 0, 4, 214, 173, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 250, 36, 0, 0, 0, 0],
            &[0, 0, 0, 15, 234, 144, 0, 0, 0, 0, 0],
            &[0, 0, 0, 137, 239, 19, 0, 0, 0, 0, 0],
            &[0, 0, 32, 248, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 168, 222, 7, 0, 0, 0, 0, 0, 0],
            &[0, 54, 255, 164, 92, 92, 92, 92, 53, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 149, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ż' / 0x17c
        'ż' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 116, 207, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 172, 255, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 32, 0, 0, 0, 0, 0],
            &[0, 5, 28, 28, 28, 28, 28, 28, 15, 0, 0],
            &[0, 50, 255, 255, 255, 255, 255, 255, 136, 0, 0],
            &[0, 3, 15, 15, 15, 15, 130, 250, 59, 0, 0],
            &[0, 0, 0, 0, 0, 63, 251, 110, 0, 0, 0],
            &[0, 0, 0, 0, 25, 232, 165, 0, 0, 0, 0],
            &[0, 0, 0, 4, 195, 210, 10, 0, 0, 0, 0],
            &[0, 0, 0, 143, 240, 36, 0, 0, 0, 0, 0],
            &[0, 0, 86, 253, 77, 0, 0, 0, 0, 0, 0],
            &[0, 40, 243, 188, 74, 74, 74, 74, 50, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ž' / 0x17d
        'Ž' => Some(&[
            &[0, 0, 69, 227, 62, 3, 146, 204, 11, 0, 0],
            &[0, 0, 0, 93, 246, 192, 218, 20, 0, 0, 0],
            &[0, 0, 0, 0, 123, 171, 49, 0, 0, 0, 0],
            &[0, 1, 35, 35, 35, 35, 35, 35, 20, 0, 0],
            &[0, 11, 255, 255, 255, 255, 255, 255, 149, 0, 0],
            &[0, 2, 48, 48, 48, 48, 94, 255, 88, 0, 0],
            &[0, 0, 0, 0, 0, 0, 187, 200, 1, 0, 0],
            &[0, 0, 0, 0, 0, 75, 255, 60, 0, 0, 0],
            &[0, 0, 0, 0, 4, 214, 173, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 250, 36, 0, 0, 0, 0],
            &[0, 0, 0, 15, 234, 144, 0, 0, 0, 0, 0],
            &[0, 0, 0, 137, 239, 19, 0, 0, 0, 0, 0],
            &[0, 0, 32, 248, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 168, 222, 7, 0, 0, 0, 0, 0, 0],
            &[0, 54, 255, 164, 92, 92, 92, 92, 53, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 149, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 5, 0, 0, 0, 11, 2, 0, 0],
        ]),
        // letter: 'ž' / 0x17e
        'ž' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 2, 0, 0, 0, 5, 0, 0, 0],
            &[0, 0, 80, 223, 52, 3, 146, 202, 9, 0, 0],
            &[0, 0, 0, 107, 242, 189, 217, 19, 0, 0, 0],
            &[0, 0, 0, 0, 135, 178, 46, 0, 0, 0, 0],
            &[0, 5, 28, 28, 28, 28, 28, 28, 15, 0, 0],
            &[0, 50, 255, 255, 255, 255, 255, 255, 136, 0, 0],
            &[0, 3, 15, 15, 15, 15, 130, 250, 59, 0, 0],
            &[0, 0, 0, 0, 0, 63, 251, 110, 0, 0, 0],
            &[0, 0, 0, 0, 25, 232, 165, 0, 0, 0, 0],
            &[0, 0, 0, 4, 195, 210, 10, 0, 0, 0, 0],
            &[0, 0, 0, 143, 240, 36, 0, 0, 0, 0, 0],
            &[0, 0, 86, 253, 77, 0, 0, 0, 0, 0, 0],
            &[0, 40, 243, 188, 74, 74, 74, 74, 50, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ſ' / 0x17f
        'ſ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 35, 176, 240, 250, 224, 38, 0, 0],
            &[0, 0, 2, 217, 228, 94, 68, 99, 3, 0, 0],
            &[0, 0, 43, 255, 102, 0, 0, 0, 0, 0, 0],
            &[0, 0, 61, 255, 67, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        _ => None,
    }
}
