//! Module for letters with the font weight regular and size 18.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 18;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 11;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight regular and font size 16px
#[inline]
pub const fn get_char(c: char) -> Option<&'static [&'static [u8]]> {
    match c {
        // letter: ' ' / 0x20
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '!' / 0x21
        '!' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 108, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 255, 166, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 255, 158, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 248, 150, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 240, 142, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 232, 134, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 224, 126, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 216, 118, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 161, 85, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 227, 137, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 246, 161, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '"' / 0x22
        '"' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 108, 6, 50, 108, 2, 0, 0, 0],
            &[0, 0, 94, 254, 3, 104, 247, 0, 0, 0, 0],
            &[0, 0, 74, 238, 0, 84, 227, 0, 0, 0, 0],
            &[0, 0, 54, 218, 0, 64, 208, 0, 0, 0, 0],
            &[0, 0, 26, 141, 0, 33, 134, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '#' / 0x23
        '#' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 96, 0, 13, 105, 0, 0, 0],
            &[0, 0, 0, 101, 192, 0, 67, 215, 0, 0, 0],
            &[0, 0, 0, 150, 144, 0, 115, 167, 0, 0, 0],
            &[0, 8, 8, 199, 102, 8, 165, 123, 8, 2, 0],
            &[22, 255, 255, 255, 255, 255, 255, 255, 255, 71, 0],
            &[0, 6, 44, 245, 8, 22, 254, 24, 6, 1, 0],
            &[0, 0, 89, 199, 0, 64, 225, 0, 0, 0, 0],
            &[72, 106, 182, 199, 106, 167, 215, 106, 78, 0, 0],
            &[114, 167, 238, 196, 167, 229, 205, 167, 123, 0, 0],
            &[0, 0, 237, 51, 0, 214, 76, 0, 0, 0, 0],
            &[0, 28, 251, 8, 11, 253, 26, 0, 0, 0, 0],
            &[0, 75, 212, 0, 58, 231, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '$' / 0x24
        '$' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 202, 59, 0, 0, 0, 0, 0],
            &[0, 0, 112, 216, 253, 244, 209, 117, 0, 0, 0],
            &[0, 99, 252, 127, 217, 116, 121, 111, 0, 0, 0],
            &[0, 174, 186, 0, 202, 59, 0, 0, 0, 0, 0],
            &[0, 154, 221, 13, 202, 59, 0, 0, 0, 0, 0],
            &[0, 38, 231, 236, 240, 93, 0, 0, 0, 0, 0],
            &[0, 0, 15, 114, 240, 255, 214, 72, 0, 0, 0],
            &[0, 0, 0, 0, 202, 103, 172, 249, 27, 0, 0],
            &[0, 0, 0, 0, 202, 59, 28, 255, 72, 0, 0],
            &[0, 47, 20, 0, 202, 60, 115, 252, 31, 0, 0],
            &[0, 141, 253, 224, 244, 244, 246, 103, 0, 0, 0],
            &[0, 4, 61, 98, 226, 118, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 188, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '%' / 0x25
        '%' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[2, 95, 126, 45, 0, 0, 9, 108, 16, 0, 0],
            &[144, 197, 121, 239, 38, 0, 111, 203, 0, 0, 0],
            &[233, 66, 0, 178, 123, 8, 230, 75, 0, 0, 0],
            &[242, 55, 0, 167, 134, 111, 203, 0, 0, 0, 0],
            &[180, 141, 36, 228, 83, 230, 76, 0, 0, 0, 0],
            &[27, 174, 208, 121, 110, 204, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 229, 77, 28, 57, 9, 0, 0],
            &[0, 0, 0, 109, 204, 88, 232, 189, 218, 17, 0],
            &[0, 0, 7, 229, 78, 214, 88, 0, 187, 115, 0],
            &[0, 0, 108, 205, 0, 240, 58, 0, 159, 144, 0],
            &[0, 7, 228, 78, 0, 201, 103, 1, 201, 106, 0],
            &[0, 108, 206, 0, 0, 59, 229, 220, 195, 10, 0],
            &[0, 0, 0, 0, 0, 0, 4, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '&' / 0x26
        '&' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 40, 133, 132, 36, 0, 0, 0, 0, 0],
            &[0, 27, 242, 186, 187, 239, 22, 0, 0, 0, 0],
            &[0, 92, 246, 8, 9, 248, 85, 0, 0, 0, 0],
            &[0, 72, 251, 10, 28, 253, 64, 0, 0, 0, 0],
            &[0, 10, 233, 121, 200, 186, 2, 0, 0, 0, 0],
            &[0, 0, 139, 255, 234, 17, 0, 0, 0, 0, 0],
            &[0, 58, 241, 255, 220, 12, 0, 97, 100, 0, 0],
            &[9, 229, 168, 14, 217, 174, 2, 222, 118, 0, 0],
            &[74, 255, 27, 0, 43, 244, 199, 251, 34, 0, 0],
            &[86, 255, 18, 0, 0, 104, 255, 234, 0, 0, 0],
            &[29, 250, 162, 33, 68, 210, 255, 254, 38, 0, 0],
            &[0, 95, 240, 255, 251, 168, 27, 199, 215, 13, 0],
            &[0, 0, 7, 34, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ''' / 0x27
        '\'' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 103, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 229, 122, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 209, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 189, 83, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 121, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '(' / 0x28
        '(' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 64, 81, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 242, 78, 0, 0, 0, 0],
            &[0, 0, 0, 0, 168, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 251, 89, 0, 0, 0, 0, 0],
            &[0, 0, 0, 97, 253, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 154, 210, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 183, 180, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 199, 164, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 187, 177, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 163, 203, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 109, 250, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 39, 255, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 186, 181, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 248, 60, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 85, 94, 0, 0, 0, 0],
        ]),
        // letter: ')' / 0x29
        ')' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 108, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 177, 177, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 254, 72, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 190, 181, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 113, 248, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 57, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 255, 83, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 255, 97, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 255, 83, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 57, 0, 0, 0],
            &[0, 0, 0, 0, 0, 108, 249, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 182, 184, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 253, 78, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 190, 1, 0, 0, 0, 0],
            &[0, 0, 0, 16, 134, 30, 0, 0, 0, 0, 0],
        ]),
        // letter: '*' / 0x2a
        '*' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 192, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 83, 0, 0, 0, 0, 0],
            &[0, 79, 148, 83, 180, 107, 115, 185, 10, 0, 0],
            &[0, 59, 139, 177, 255, 237, 170, 139, 20, 0, 0],
            &[0, 0, 0, 144, 178, 237, 59, 0, 0, 0, 0],
            &[0, 0, 87, 250, 34, 130, 231, 23, 0, 0, 0],
            &[0, 0, 19, 90, 0, 14, 101, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '+' / 0x2b
        '+' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 194, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 194, 100, 0, 0, 0, 0, 0],
            &[0, 38, 53, 53, 207, 132, 53, 53, 17, 0, 0],
            &[0, 170, 237, 237, 251, 244, 237, 237, 75, 0, 0],
            &[0, 0, 0, 0, 194, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 194, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 194, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ',' / 0x2c
        ',' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 242, 4, 0, 0, 0, 0],
            &[0, 0, 0, 10, 252, 167, 0, 0, 0, 0, 0],
            &[0, 0, 0, 59, 255, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 116, 222, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 37, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '-' / 0x2d
        '-' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 147, 245, 245, 245, 245, 49, 0, 0, 0],
            &[0, 0, 49, 81, 81, 81, 81, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '.' / 0x2e
        '.' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 227, 137, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 246, 161, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '/' / 0x2f
        '/' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 84, 64, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 249, 83, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 235, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 209, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 255, 44, 0, 0, 0],
            &[0, 0, 0, 0, 0, 149, 202, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 239, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 246, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 188, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 254, 65, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 221, 1, 0, 0, 0, 0, 0],
            &[0, 0, 2, 225, 124, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '0' / 0x30
        '0' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 88, 137, 123, 41, 0, 0, 0],
            &[0, 0, 5, 188, 247, 187, 212, 252, 86, 0, 0],
            &[0, 0, 100, 247, 47, 0, 3, 255, 252, 10, 0],
            &[0, 0, 187, 166, 0, 0, 102, 255, 255, 82, 0],
            &[0, 0, 235, 110, 0, 17, 235, 118, 202, 131, 0],
            &[0, 1, 253, 86, 0, 148, 216, 5, 177, 152, 0],
            &[0, 5, 255, 83, 47, 252, 70, 0, 173, 160, 0],
            &[0, 0, 247, 101, 194, 174, 0, 0, 191, 148, 0],
            &[0, 0, 213, 225, 246, 31, 0, 0, 225, 116, 0],
            &[0, 0, 150, 255, 127, 0, 0, 53, 255, 54, 0],
            &[0, 0, 42, 255, 152, 38, 72, 221, 197, 0, 0],
            &[0, 0, 0, 79, 230, 255, 255, 191, 27, 0, 0],
            &[0, 0, 0, 0, 1, 28, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '1' / 0x31
        '1' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 98, 0, 0, 0, 0, 0],
            &[0, 0, 73, 203, 255, 231, 0, 0, 0, 0, 0],
            &[0, 57, 250, 152, 139, 231, 0, 0, 0, 0, 0],
            &[0, 0, 25, 0, 116, 231, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 119, 231, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 231, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 231, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 231, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 231, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 231, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 149, 255, 20, 4, 0, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 79, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '2' / 0x32
        '2' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 104, 139, 127, 49, 0, 0, 0, 0],
            &[0, 95, 247, 230, 182, 216, 253, 101, 0, 0, 0],
            &[0, 52, 118, 4, 0, 1, 168, 237, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 93, 255, 21, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 247, 6, 0, 0],
            &[0, 0, 0, 0, 0, 14, 228, 166, 0, 0, 0],
            &[0, 0, 0, 0, 1, 166, 238, 30, 0, 0, 0],
            &[0, 0, 0, 0, 145, 247, 60, 0, 0, 0, 0],
            &[0, 0, 0, 142, 248, 75, 0, 0, 0, 0, 0],
            &[0, 0, 146, 246, 71, 0, 0, 0, 0, 0, 0],
            &[0, 145, 255, 138, 71, 71, 71, 71, 51, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 255, 184, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '3' / 0x33
        '3' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 35, 113, 143, 128, 52, 0, 0, 0, 0],
            &[0, 115, 253, 217, 180, 217, 254, 106, 0, 0, 0],
            &[0, 36, 70, 0, 0, 1, 174, 236, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 254, 4, 0, 0],
            &[0, 0, 0, 0, 0, 12, 204, 192, 0, 0, 0],
            &[0, 0, 18, 139, 150, 214, 169, 27, 0, 0, 0],
            &[0, 0, 25, 188, 203, 232, 195, 64, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 139, 246, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 255, 92, 0, 0],
            &[0, 0, 0, 0, 0, 0, 59, 255, 78, 0, 0],
            &[0, 142, 94, 49, 36, 79, 217, 227, 9, 0, 0],
            &[0, 139, 240, 255, 255, 251, 184, 38, 0, 0, 0],
            &[0, 0, 3, 25, 33, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '4' / 0x34
        '4' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 80, 124, 16, 0, 0],
            &[0, 0, 0, 0, 0, 33, 246, 255, 34, 0, 0],
            &[0, 0, 0, 0, 0, 180, 184, 255, 34, 0, 0],
            &[0, 0, 0, 0, 79, 239, 74, 255, 34, 0, 0],
            &[0, 0, 0, 10, 224, 107, 59, 255, 34, 0, 0],
            &[0, 0, 0, 134, 206, 3, 61, 255, 34, 0, 0],
            &[0, 0, 40, 248, 57, 0, 61, 255, 34, 0, 0],
            &[0, 0, 189, 158, 0, 0, 61, 255, 34, 0, 0],
            &[0, 46, 255, 216, 202, 202, 215, 255, 209, 162, 0],
            &[0, 22, 112, 112, 112, 112, 146, 255, 132, 90, 0],
            &[0, 0, 0, 0, 0, 0, 61, 255, 34, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 255, 34, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '5' / 0x35
        '5' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 108, 108, 108, 108, 108, 70, 0, 0, 0],
            &[0, 31, 255, 225, 219, 219, 219, 141, 0, 0, 0],
            &[0, 51, 255, 31, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 255, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 92, 240, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 113, 251, 239, 253, 228, 143, 13, 0, 0, 0],
            &[0, 13, 84, 52, 50, 115, 244, 184, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 105, 255, 41, 0, 0],
            &[0, 0, 0, 0, 0, 0, 52, 255, 70, 0, 0],
            &[0, 0, 0, 0, 0, 0, 98, 255, 39, 0, 0],
            &[0, 123, 85, 42, 42, 103, 238, 183, 0, 0, 0],
            &[0, 122, 240, 255, 255, 245, 155, 15, 0, 0, 0],
            &[0, 0, 2, 24, 31, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '6' / 0x36
        '6' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 86, 130, 145, 88, 0, 0],
            &[0, 0, 0, 53, 226, 240, 181, 166, 132, 0, 0],
            &[0, 0, 18, 233, 172, 13, 0, 0, 0, 0, 0],
            &[0, 0, 123, 233, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 197, 157, 1, 33, 44, 7, 0, 0, 0],
            &[0, 0, 239, 143, 201, 241, 246, 242, 102, 0, 0],
            &[0, 3, 255, 240, 88, 1, 2, 115, 255, 61, 0],
            &[0, 4, 255, 142, 0, 0, 0, 0, 221, 142, 0],
            &[0, 0, 235, 113, 0, 0, 0, 0, 198, 156, 0],
            &[0, 0, 173, 190, 0, 0, 0, 7, 235, 121, 0],
            &[0, 0, 58, 253, 143, 30, 35, 164, 242, 29, 0],
            &[0, 0, 0, 85, 229, 255, 255, 215, 62, 0, 0],
            &[0, 0, 0, 0, 1, 30, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '7' / 0x37
        '7' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 108, 108, 108, 108, 108, 108, 108, 69, 0],
            &[0, 8, 219, 219, 219, 219, 219, 219, 250, 159, 0],
            &[0, 0, 0, 0, 0, 0, 0, 28, 251, 78, 0],
            &[0, 0, 0, 0, 0, 0, 0, 135, 223, 2, 0],
            &[0, 0, 0, 0, 0, 0, 9, 236, 118, 0, 0],
            &[0, 0, 0, 0, 0, 0, 101, 247, 19, 0, 0],
            &[0, 0, 0, 0, 0, 0, 211, 159, 0, 0, 0],
            &[0, 0, 0, 0, 0, 67, 255, 51, 0, 0, 0],
            &[0, 0, 0, 0, 0, 178, 199, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 253, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 144, 233, 6, 0, 0, 0, 0],
            &[0, 0, 0, 13, 241, 131, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '8' / 0x38
        '8' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 96, 141, 131, 59, 0, 0, 0, 0],
            &[0, 17, 215, 234, 168, 188, 254, 128, 0, 0, 0],
            &[0, 111, 247, 24, 0, 0, 123, 250, 10, 0, 0],
            &[0, 117, 238, 2, 0, 0, 90, 255, 17, 0, 0],
            &[0, 41, 247, 121, 0, 19, 205, 184, 0, 0, 0],
            &[0, 0, 79, 240, 206, 237, 164, 13, 0, 0, 0],
            &[0, 0, 102, 238, 180, 244, 183, 28, 0, 0, 0],
            &[0, 96, 249, 68, 0, 23, 190, 217, 16, 0, 0],
            &[0, 211, 164, 0, 0, 0, 17, 252, 103, 0, 0],
            &[0, 220, 146, 0, 0, 0, 6, 247, 112, 0, 0],
            &[0, 143, 241, 74, 12, 30, 163, 246, 34, 0, 0],
            &[0, 7, 153, 247, 255, 255, 212, 65, 0, 0, 0],
            &[0, 0, 0, 8, 34, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '9' / 0x39
        '9' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 102, 141, 125, 43, 0, 0, 0],
            &[0, 0, 34, 226, 228, 170, 209, 252, 95, 0, 0],
            &[0, 0, 177, 209, 11, 0, 0, 137, 244, 19, 0],
            &[0, 0, 246, 111, 0, 0, 0, 8, 240, 97, 0],
            &[0, 4, 255, 97, 0, 0, 0, 0, 210, 143, 0],
            &[0, 0, 218, 156, 0, 0, 0, 39, 251, 159, 0],
            &[0, 0, 94, 252, 148, 89, 119, 211, 231, 149, 0],
            &[0, 0, 0, 64, 167, 196, 158, 38, 238, 125, 0],
            &[0, 0, 0, 0, 0, 0, 0, 35, 255, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 158, 229, 6, 0],
            &[0, 0, 0, 35, 19, 50, 157, 253, 80, 0, 0],
            &[0, 0, 0, 239, 255, 255, 206, 71, 0, 0, 0],
            &[0, 0, 0, 20, 36, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ':' / 0x3a
        ':' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 153, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 255, 179, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 76, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 227, 134, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 246, 158, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ';' / 0x3b
        ';' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 186, 254, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 242, 4, 0, 0, 0, 0],
            &[0, 0, 0, 10, 252, 167, 0, 0, 0, 0, 0],
            &[0, 0, 0, 59, 255, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 116, 222, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 37, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '<' / 0x3c
        '<' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 91, 27, 0, 0],
            &[0, 0, 0, 0, 0, 61, 203, 220, 23, 0, 0],
            &[0, 0, 0, 36, 174, 235, 110, 5, 0, 0, 0],
            &[0, 18, 143, 241, 135, 15, 0, 0, 0, 0, 0],
            &[0, 135, 255, 118, 3, 0, 0, 0, 0, 0, 0],
            &[0, 3, 90, 214, 218, 98, 5, 0, 0, 0, 0],
            &[0, 0, 0, 2, 87, 211, 225, 107, 5, 0, 0],
            &[0, 0, 0, 0, 0, 1, 83, 208, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '=' / 0x3d
        '=' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 178, 202, 202, 202, 202, 202, 202, 97, 0, 0],
            &[0, 109, 124, 124, 124, 124, 124, 124, 59, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 169, 192, 192, 192, 192, 192, 192, 92, 0, 0],
            &[0, 118, 135, 135, 135, 135, 135, 135, 64, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '>' / 0x3e
        '>' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 39, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 96, 244, 147, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 166, 238, 116, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 191, 222, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 199, 253, 35, 0, 0],
            &[0, 0, 0, 30, 147, 244, 166, 42, 0, 0, 0],
            &[0, 34, 157, 246, 162, 39, 0, 0, 0, 0, 0],
            &[0, 138, 159, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '?' / 0x3f
        '?' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 56, 122, 144, 128, 56, 0, 0, 0, 0],
            &[0, 95, 255, 215, 180, 201, 255, 127, 0, 0, 0],
            &[0, 11, 40, 0, 0, 0, 122, 250, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 63, 255, 28, 0, 0],
            &[0, 0, 0, 0, 0, 0, 132, 232, 1, 0, 0],
            &[0, 0, 0, 0, 0, 112, 248, 84, 0, 0, 0],
            &[0, 0, 0, 0, 138, 243, 75, 0, 0, 0, 0],
            &[0, 0, 0, 38, 254, 65, 0, 0, 0, 0, 0],
            &[0, 0, 0, 58, 188, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 97, 236, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 118, 254, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '@' / 0x40
        '@' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 91, 93, 38, 0, 0, 0, 0],
            &[0, 2, 143, 243, 185, 175, 239, 143, 1, 0, 0],
            &[0, 129, 211, 28, 0, 0, 24, 214, 107, 0, 0],
            &[22, 245, 40, 0, 52, 71, 32, 62, 229, 1, 0],
            &[103, 187, 0, 168, 221, 199, 255, 28, 231, 45, 0],
            &[156, 127, 73, 244, 19, 42, 255, 14, 193, 76, 0],
            &[183, 100, 139, 185, 0, 56, 254, 2, 185, 86, 0],
            &[188, 93, 149, 167, 0, 81, 244, 0, 198, 68, 0],
            &[169, 113, 112, 208, 1, 142, 251, 16, 238, 18, 0],
            &[124, 167, 21, 226, 212, 190, 130, 233, 159, 0, 0],
            &[44, 244, 25, 14, 57, 7, 4, 49, 1, 0, 0],
            &[0, 160, 198, 24, 0, 0, 0, 23, 0, 0, 0],
            &[0, 6, 160, 245, 190, 175, 207, 221, 0, 0, 0],
            &[0, 0, 0, 34, 86, 98, 66, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'A' / 0x41
        'A' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 120, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 255, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 155, 199, 255, 49, 0, 0, 0, 0],
            &[0, 0, 3, 234, 105, 210, 132, 0, 0, 0, 0],
            &[0, 0, 64, 254, 27, 133, 215, 0, 0, 0, 0],
            &[0, 0, 147, 203, 0, 53, 255, 42, 0, 0, 0],
            &[0, 1, 228, 123, 0, 1, 229, 126, 0, 0, 0],
            &[0, 57, 255, 245, 200, 200, 255, 209, 0, 0, 0],
            &[0, 140, 255, 127, 126, 126, 187, 255, 36, 0, 0],
            &[0, 222, 141, 0, 0, 0, 6, 241, 119, 0, 0],
            &[49, 255, 61, 0, 0, 0, 0, 168, 202, 0, 0],
            &[132, 235, 3, 0, 0, 0, 0, 88, 255, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'B' / 0x42
        'B' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 108, 108, 106, 85, 34, 0, 0, 0, 0],
            &[0, 122, 250, 210, 213, 237, 254, 165, 1, 0, 0],
            &[0, 122, 225, 0, 0, 1, 115, 255, 68, 0, 0],
            &[0, 122, 225, 0, 0, 0, 25, 255, 93, 0, 0],
            &[0, 122, 225, 0, 0, 0, 77, 254, 44, 0, 0],
            &[0, 122, 243, 151, 152, 176, 220, 103, 0, 0, 0],
            &[0, 122, 245, 167, 168, 187, 225, 142, 6, 0, 0],
            &[0, 122, 225, 0, 0, 0, 38, 250, 123, 0, 0],
            &[0, 122, 225, 0, 0, 0, 0, 207, 176, 0, 0],
            &[0, 122, 225, 0, 0, 0, 6, 232, 156, 0, 0],
            &[0, 122, 232, 63, 63, 85, 182, 252, 57, 0, 0],
            &[0, 122, 255, 255, 255, 245, 199, 73, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'C' / 0x43
        'C' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 97, 139, 132, 93, 15, 0, 0],
            &[0, 0, 57, 232, 243, 189, 196, 241, 152, 0, 0],
            &[0, 20, 235, 188, 17, 0, 0, 9, 17, 0, 0],
            &[0, 123, 249, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 194, 186, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 229, 146, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 235, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 217, 158, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 212, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 87, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 193, 242, 116, 55, 59, 108, 48, 0, 0],
            &[0, 0, 12, 150, 246, 255, 255, 235, 57, 0, 0],
            &[0, 0, 0, 0, 3, 29, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'D' / 0x44
        'D' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 95, 108, 108, 90, 44, 0, 0, 0, 0, 0],
            &[0, 225, 236, 211, 233, 255, 192, 26, 0, 0, 0],
            &[0, 225, 143, 0, 0, 60, 223, 212, 7, 0, 0],
            &[0, 225, 143, 0, 0, 0, 50, 254, 101, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 206, 178, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 166, 209, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 156, 217, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 180, 194, 0, 0],
            &[0, 225, 143, 0, 0, 0, 7, 238, 137, 0, 0],
            &[0, 225, 143, 0, 0, 0, 135, 250, 42, 0, 0],
            &[0, 225, 171, 66, 93, 179, 254, 111, 0, 0, 0],
            &[0, 225, 255, 253, 230, 177, 66, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'E' / 0x45
        'E' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 108, 108, 108, 108, 108, 108, 26, 0, 0],
            &[0, 122, 254, 219, 219, 219, 219, 219, 52, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 251, 155, 155, 155, 155, 145, 0, 0, 0],
            &[0, 122, 252, 171, 171, 171, 171, 160, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 248, 71, 71, 71, 71, 71, 17, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'F' / 0x46
        'F' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 108, 108, 108, 108, 108, 108, 26, 0, 0],
            &[0, 122, 254, 219, 219, 219, 219, 219, 52, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 246, 28, 28, 28, 28, 26, 0, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 239, 0, 0, 0],
            &[0, 122, 247, 42, 42, 42, 42, 40, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'G' / 0x47
        'G' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 113, 141, 122, 63, 0, 0],
            &[0, 0, 0, 121, 252, 232, 191, 210, 253, 42, 0],
            &[0, 0, 81, 255, 132, 5, 0, 0, 28, 0, 0],
            &[0, 0, 204, 195, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 254, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 255, 65, 0, 0, 8, 8, 8, 3, 0],
            &[0, 66, 255, 53, 0, 0, 253, 255, 255, 122, 0],
            &[0, 51, 255, 70, 0, 0, 62, 63, 238, 122, 0],
            &[0, 16, 252, 120, 0, 0, 0, 0, 233, 122, 0],
            &[0, 0, 192, 221, 10, 0, 0, 0, 233, 122, 0],
            &[0, 0, 55, 249, 197, 72, 37, 60, 241, 122, 0],
            &[0, 0, 0, 62, 201, 255, 255, 255, 221, 78, 0],
            &[0, 0, 0, 0, 0, 13, 33, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'H' / 0x48
        'H' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 0, 104, 52, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 211, 155, 155, 155, 155, 251, 122, 0, 0],
            &[0, 225, 218, 171, 171, 171, 171, 252, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'I' / 0x49
        'I' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 108, 108, 108, 108, 108, 91, 0, 0],
            &[0, 0, 35, 162, 185, 252, 224, 175, 130, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 15, 38, 242, 151, 28, 6, 0, 0],
            &[0, 0, 61, 255, 255, 255, 255, 255, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'J' / 0x4a
        'J' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 108, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 255, 112, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 255, 112, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 255, 112, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 255, 112, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 255, 112, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 255, 112, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 255, 112, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 255, 110, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 255, 85, 0, 0],
            &[0, 0, 61, 99, 57, 76, 212, 240, 17, 0, 0],
            &[0, 0, 77, 240, 255, 255, 225, 69, 0, 0, 0],
            &[0, 0, 0, 3, 27, 28, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'K' / 0x4b
        'K' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 16, 108, 54, 0, 0],
            &[0, 225, 143, 0, 0, 0, 174, 228, 20, 0, 0],
            &[0, 225, 143, 0, 0, 109, 252, 62, 0, 0, 0],
            &[0, 225, 143, 0, 51, 249, 126, 0, 0, 0, 0],
            &[0, 225, 143, 15, 221, 189, 2, 0, 0, 0, 0],
            &[0, 225, 143, 164, 245, 25, 0, 0, 0, 0, 0],
            &[0, 225, 213, 241, 255, 79, 0, 0, 0, 0, 0],
            &[0, 225, 246, 55, 168, 228, 13, 0, 0, 0, 0],
            &[0, 225, 148, 0, 24, 239, 148, 0, 0, 0, 0],
            &[0, 225, 143, 0, 0, 104, 253, 56, 0, 0, 0],
            &[0, 225, 143, 0, 0, 1, 198, 211, 5, 0, 0],
            &[0, 225, 143, 0, 0, 0, 45, 251, 123, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'L' / 0x4c
        'L' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 108, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 132, 83, 83, 83, 83, 20, 0, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'M' / 0x4d
        'M' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 108, 100, 0, 0, 0, 36, 108, 86, 0],
            &[0, 51, 255, 255, 32, 0, 0, 138, 255, 204, 0],
            &[0, 51, 251, 220, 105, 0, 0, 211, 215, 204, 0],
            &[0, 51, 255, 162, 177, 0, 28, 255, 159, 204, 0],
            &[0, 51, 255, 99, 243, 6, 100, 213, 136, 204, 0],
            &[0, 51, 255, 44, 240, 67, 173, 137, 142, 204, 0],
            &[0, 51, 255, 40, 169, 144, 241, 61, 143, 204, 0],
            &[0, 51, 255, 40, 93, 246, 238, 3, 143, 204, 0],
            &[0, 51, 255, 40, 19, 252, 166, 0, 143, 204, 0],
            &[0, 51, 255, 40, 0, 34, 18, 0, 143, 204, 0],
            &[0, 51, 255, 40, 0, 0, 0, 0, 143, 204, 0],
            &[0, 51, 255, 40, 0, 0, 0, 0, 143, 204, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'N' / 0x4e
        'N' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 95, 108, 12, 0, 0, 0, 98, 52, 0, 0],
            &[0, 225, 255, 109, 0, 0, 0, 233, 122, 0, 0],
            &[0, 225, 198, 219, 2, 0, 0, 233, 122, 0, 0],
            &[0, 225, 120, 241, 77, 0, 0, 233, 122, 0, 0],
            &[0, 225, 123, 142, 189, 0, 0, 233, 122, 0, 0],
            &[0, 225, 130, 32, 252, 45, 0, 233, 122, 0, 0],
            &[0, 225, 131, 0, 173, 156, 0, 233, 122, 0, 0],
            &[0, 225, 131, 0, 61, 247, 21, 230, 122, 0, 0],
            &[0, 225, 131, 0, 0, 204, 124, 223, 122, 0, 0],
            &[0, 225, 131, 0, 0, 92, 230, 214, 122, 0, 0],
            &[0, 225, 131, 0, 0, 5, 230, 254, 122, 0, 0],
            &[0, 225, 131, 0, 0, 0, 123, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'O' / 0x4f
        'O' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 96, 145, 137, 59, 0, 0, 0],
            &[0, 0, 12, 204, 245, 184, 199, 254, 120, 0, 0],
            &[0, 0, 136, 241, 39, 0, 0, 114, 253, 41, 0],
            &[0, 0, 234, 144, 0, 0, 0, 7, 244, 135, 0],
            &[0, 34, 255, 84, 0, 0, 0, 0, 188, 189, 0],
            &[0, 58, 255, 58, 0, 0, 0, 0, 166, 216, 0],
            &[0, 67, 255, 52, 0, 0, 0, 0, 155, 221, 0],
            &[0, 54, 255, 66, 0, 0, 0, 0, 168, 206, 0],
            &[0, 18, 254, 105, 0, 0, 0, 0, 206, 168, 0],
            &[0, 0, 203, 191, 0, 0, 0, 35, 254, 97, 0],
            &[0, 0, 78, 255, 144, 45, 64, 207, 219, 7, 0],
            &[0, 0, 0, 101, 235, 255, 255, 194, 35, 0, 0],
            &[0, 0, 0, 0, 4, 31, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'P' / 0x50
        'P' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 108, 108, 108, 96, 50, 0, 0, 0, 0],
            &[0, 122, 254, 210, 212, 235, 255, 183, 4, 0, 0],
            &[0, 122, 245, 0, 0, 0, 86, 254, 107, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 212, 166, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 207, 167, 0, 0],
            &[0, 122, 245, 0, 0, 0, 63, 254, 98, 0, 0],
            &[0, 122, 252, 165, 168, 197, 253, 178, 3, 0, 0],
            &[0, 122, 251, 153, 151, 131, 70, 1, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Q' / 0x51
        'Q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 96, 145, 137, 59, 0, 0, 0],
            &[0, 0, 12, 204, 245, 184, 199, 254, 120, 0, 0],
            &[0, 0, 136, 241, 39, 0, 0, 114, 253, 41, 0],
            &[0, 0, 234, 144, 0, 0, 0, 7, 244, 135, 0],
            &[0, 34, 255, 84, 0, 0, 0, 0, 188, 189, 0],
            &[0, 58, 255, 58, 0, 0, 0, 0, 166, 216, 0],
            &[0, 67, 255, 52, 0, 0, 0, 0, 155, 221, 0],
            &[0, 54, 255, 66, 0, 0, 0, 0, 168, 206, 0],
            &[0, 18, 254, 105, 0, 0, 0, 0, 206, 168, 0],
            &[0, 0, 203, 191, 0, 0, 0, 35, 254, 97, 0],
            &[0, 0, 78, 255, 144, 45, 66, 250, 219, 7, 0],
            &[0, 0, 0, 101, 235, 255, 255, 255, 39, 0, 0],
            &[0, 0, 0, 0, 4, 31, 59, 252, 102, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 163, 228, 8, 0],
            &[0, 0, 0, 0, 0, 0, 0, 37, 224, 95, 0],
        ]),
        // letter: 'R' / 0x52
        'R' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 108, 108, 102, 75, 14, 0, 0, 0, 0],
            &[0, 122, 254, 210, 220, 249, 240, 79, 0, 0, 0],
            &[0, 122, 245, 0, 0, 18, 190, 238, 8, 0, 0],
            &[0, 122, 245, 0, 0, 0, 88, 255, 43, 0, 0],
            &[0, 122, 245, 0, 0, 0, 104, 255, 32, 0, 0],
            &[0, 122, 245, 0, 0, 37, 210, 188, 0, 0, 0],
            &[0, 122, 255, 235, 242, 255, 169, 21, 0, 0, 0],
            &[0, 122, 249, 83, 97, 251, 102, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 142, 236, 19, 0, 0, 0],
            &[0, 122, 245, 0, 0, 15, 232, 154, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 97, 253, 53, 0, 0],
            &[0, 122, 245, 0, 0, 0, 1, 200, 203, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'S' / 0x53
        'S' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 131, 140, 113, 49, 0, 0, 0],
            &[0, 0, 167, 255, 208, 187, 220, 250, 21, 0, 0],
            &[0, 56, 255, 114, 0, 0, 0, 28, 0, 0, 0],
            &[0, 90, 255, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 255, 132, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 142, 255, 199, 78, 1, 0, 0, 0, 0],
            &[0, 0, 0, 64, 193, 255, 210, 59, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 209, 246, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 105, 0, 0],
            &[0, 0, 0, 0, 0, 0, 38, 255, 92, 0, 0],
            &[0, 107, 99, 51, 27, 60, 195, 247, 29, 0, 0],
            &[0, 109, 233, 255, 255, 255, 212, 66, 0, 0, 0],
            &[0, 0, 0, 16, 36, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'T' / 0x54
        'T' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 108, 108, 108, 108, 108, 108, 108, 95, 0],
            &[0, 61, 219, 219, 219, 252, 238, 219, 219, 192, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'U' / 0x55
        'U' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 0, 104, 52, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 246, 122, 0, 0],
            &[0, 214, 164, 0, 0, 0, 9, 255, 112, 0, 0],
            &[0, 177, 209, 0, 0, 0, 52, 255, 72, 0, 0],
            &[0, 84, 255, 136, 41, 60, 203, 224, 7, 0, 0],
            &[0, 0, 121, 241, 255, 255, 206, 44, 0, 0, 0],
            &[0, 0, 0, 5, 31, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'V' / 0x56
        'V' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[57, 104, 0, 0, 0, 0, 0, 41, 108, 14, 0],
            &[77, 255, 46, 0, 0, 0, 0, 150, 230, 1, 0],
            &[7, 242, 122, 0, 0, 0, 0, 224, 151, 0, 0],
            &[0, 168, 199, 0, 0, 0, 43, 255, 70, 0, 0],
            &[0, 85, 253, 22, 0, 0, 117, 239, 5, 0, 0],
            &[0, 12, 247, 97, 0, 0, 192, 164, 0, 0, 0],
            &[0, 0, 177, 173, 0, 15, 251, 83, 0, 0, 0],
            &[0, 0, 94, 243, 6, 85, 246, 11, 0, 0, 0],
            &[0, 0, 17, 250, 69, 157, 177, 0, 0, 0, 0],
            &[0, 0, 0, 185, 139, 224, 96, 0, 0, 0, 0],
            &[0, 0, 0, 103, 226, 251, 19, 0, 0, 0, 0],
            &[0, 0, 0, 23, 253, 189, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'W' / 0x57
        'W' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[62, 81, 0, 0, 0, 0, 0, 15, 108, 19, 0],
            &[127, 208, 0, 0, 0, 0, 0, 53, 255, 27, 0],
            &[98, 233, 0, 0, 0, 0, 0, 74, 253, 4, 0],
            &[69, 253, 3, 0, 138, 84, 0, 96, 231, 0, 0],
            &[41, 255, 25, 31, 252, 194, 0, 119, 205, 0, 0],
            &[12, 255, 50, 90, 187, 245, 7, 141, 180, 0, 0],
            &[0, 239, 73, 150, 124, 210, 61, 161, 154, 0, 0],
            &[0, 210, 94, 209, 66, 150, 122, 180, 128, 0, 0],
            &[0, 182, 124, 248, 11, 88, 182, 195, 102, 0, 0],
            &[0, 153, 190, 203, 0, 27, 237, 209, 76, 0, 0],
            &[0, 124, 246, 144, 0, 0, 221, 246, 50, 0, 0],
            &[0, 96, 255, 84, 0, 0, 159, 255, 25, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'X' / 0x58
        'X' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[8, 107, 51, 0, 0, 0, 0, 91, 73, 0, 0],
            &[0, 164, 217, 5, 0, 0, 62, 254, 63, 0, 0],
            &[0, 24, 241, 110, 0, 2, 205, 166, 0, 0, 0],
            &[0, 0, 111, 236, 16, 97, 242, 26, 0, 0, 0],
            &[0, 0, 3, 209, 152, 230, 115, 0, 0, 0, 0],
            &[0, 0, 0, 60, 254, 213, 5, 0, 0, 0, 0],
            &[0, 0, 0, 37, 249, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 184, 188, 243, 83, 0, 0, 0, 0],
            &[0, 0, 83, 248, 34, 120, 226, 11, 0, 0, 0],
            &[0, 11, 226, 136, 0, 7, 221, 136, 0, 0, 0],
            &[0, 136, 231, 13, 0, 0, 80, 250, 40, 0, 0],
            &[40, 250, 98, 0, 0, 0, 0, 188, 188, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Y' / 0x59
        'Y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[36, 108, 26, 0, 0, 0, 0, 64, 100, 0, 0],
            &[12, 235, 146, 0, 0, 0, 6, 230, 145, 0, 0],
            &[0, 117, 245, 19, 0, 0, 100, 246, 24, 0, 0],
            &[0, 10, 232, 127, 0, 2, 217, 141, 0, 0, 0],
            &[0, 0, 113, 235, 10, 82, 244, 22, 0, 0, 0],
            &[0, 0, 8, 229, 108, 201, 136, 0, 0, 0, 0],
            &[0, 0, 0, 108, 241, 242, 19, 0, 0, 0, 0],
            &[0, 0, 0, 7, 239, 144, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Z' / 0x5a
        'Z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 108, 108, 108, 108, 108, 108, 8, 0, 0],
            &[0, 33, 210, 210, 210, 210, 231, 254, 16, 0, 0],
            &[0, 0, 0, 0, 0, 1, 200, 164, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 247, 30, 0, 0, 0],
            &[0, 0, 0, 0, 8, 224, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 234, 14, 0, 0, 0, 0],
            &[0, 0, 0, 22, 241, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 216, 4, 0, 0, 0, 0, 0],
            &[0, 0, 42, 252, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 182, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 69, 255, 121, 71, 71, 71, 71, 5, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '[' / 0x5b
        '[' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 108, 108, 108, 47, 0, 0, 0],
            &[0, 0, 0, 40, 255, 201, 186, 81, 0, 0, 0],
            &[0, 0, 0, 40, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 180, 159, 70, 0, 0, 0],
            &[0, 0, 0, 21, 135, 135, 135, 59, 0, 0, 0],
        ]),
        // letter: '\' / 0x5c
        '\\' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 106, 40, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 185, 166, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 87, 247, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 237, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 146, 205, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 145, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 108, 236, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 248, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 167, 184, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 69, 253, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 225, 124, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ']' / 0x5d
        ']' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 125, 149, 149, 113, 0, 0, 0, 0],
            &[0, 0, 0, 122, 145, 213, 194, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 157, 194, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 157, 194, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 157, 194, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 157, 194, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 157, 194, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 157, 194, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 157, 194, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 157, 194, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 157, 194, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 157, 194, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 157, 194, 0, 0, 0, 0],
            &[0, 0, 0, 168, 200, 234, 194, 0, 0, 0, 0],
            &[0, 0, 0, 79, 94, 94, 71, 0, 0, 0, 0],
        ]),
        // letter: '^' / 0x5e
        '^' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 115, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 64, 255, 140, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 135, 240, 26, 0, 0, 0],
            &[0, 0, 0, 51, 234, 9, 151, 151, 0, 0, 0],
            &[0, 0, 0, 171, 129, 0, 28, 243, 34, 0, 0],
            &[0, 0, 39, 245, 22, 0, 0, 147, 162, 0, 0],
            &[0, 0, 158, 154, 0, 0, 0, 26, 244, 43, 0],
            &[0, 10, 180, 39, 0, 0, 0, 0, 117, 113, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '_' / 0x5f
        '_' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[208, 208, 208, 208, 208, 208, 208, 208, 208, 125, 0],
            &[118, 118, 118, 118, 118, 118, 118, 118, 118, 71, 0],
        ]),
        // letter: '`' / 0x60
        '`' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 63, 50, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 250, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 198, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 67, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'a' / 0x61
        'a' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 73, 153, 182, 165, 103, 6, 0, 0, 0],
            &[0, 21, 232, 158, 120, 141, 239, 164, 0, 0, 0],
            &[0, 0, 3, 0, 0, 0, 89, 254, 10, 0, 0],
            &[0, 0, 0, 4, 31, 46, 86, 255, 39, 0, 0],
            &[0, 13, 159, 244, 239, 218, 217, 255, 40, 0, 0],
            &[0, 157, 228, 60, 0, 0, 41, 255, 40, 0, 0],
            &[0, 215, 140, 0, 0, 0, 87, 255, 40, 0, 0],
            &[0, 182, 204, 32, 19, 82, 223, 255, 40, 0, 0],
            &[0, 42, 214, 255, 255, 205, 58, 255, 40, 0, 0],
            &[0, 0, 0, 25, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'b' / 0x62
        'b' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 29, 27, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 171, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 171, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 171, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 169, 59, 162, 180, 126, 14, 0, 0, 0],
            &[0, 184, 219, 233, 136, 138, 237, 203, 5, 0, 0],
            &[0, 184, 255, 51, 0, 0, 60, 255, 91, 0, 0],
            &[0, 184, 208, 0, 0, 0, 0, 226, 157, 0, 0],
            &[0, 184, 179, 0, 0, 0, 0, 199, 179, 0, 0],
            &[0, 184, 193, 0, 0, 0, 0, 212, 170, 0, 0],
            &[0, 184, 240, 10, 0, 0, 18, 248, 124, 0, 0],
            &[0, 184, 252, 162, 28, 29, 171, 246, 31, 0, 0],
            &[0, 184, 132, 156, 253, 255, 231, 75, 0, 0, 0],
            &[0, 0, 0, 0, 16, 33, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'c' / 0x63
        'c' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 157, 181, 161, 93, 2, 0, 0],
            &[0, 0, 153, 254, 179, 138, 161, 197, 0, 0, 0],
            &[0, 58, 255, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 244, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 156, 218, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 147, 228, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 103, 255, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 232, 206, 62, 25, 46, 106, 0, 0, 0],
            &[0, 0, 47, 199, 255, 255, 255, 208, 0, 0, 0],
            &[0, 0, 0, 0, 16, 34, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'd' / 0x64
        'd' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 40, 13, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 255, 81, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 255, 81, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 255, 81, 0],
            &[0, 0, 0, 49, 151, 180, 142, 48, 255, 81, 0],
            &[0, 0, 63, 249, 207, 125, 157, 218, 255, 81, 0],
            &[0, 0, 197, 219, 8, 0, 0, 144, 255, 81, 0],
            &[0, 7, 252, 128, 0, 0, 0, 57, 255, 81, 0],
            &[0, 25, 255, 99, 0, 0, 0, 30, 255, 81, 0],
            &[0, 16, 255, 108, 0, 0, 0, 39, 255, 81, 0],
            &[0, 0, 227, 161, 0, 0, 0, 91, 255, 81, 0],
            &[0, 0, 127, 249, 93, 15, 55, 217, 255, 81, 0],
            &[0, 0, 5, 155, 251, 255, 237, 82, 238, 81, 0],
            &[0, 0, 0, 0, 13, 32, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'e' / 0x65
        'e' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 113, 172, 173, 109, 6, 0, 0],
            &[0, 0, 16, 212, 228, 133, 128, 232, 184, 1, 0],
            &[0, 0, 144, 233, 22, 0, 0, 50, 255, 78, 0],
            &[0, 0, 228, 165, 42, 42, 42, 42, 233, 143, 0],
            &[0, 3, 254, 253, 251, 251, 251, 251, 251, 159, 0],
            &[0, 0, 246, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 190, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 252, 149, 37, 13, 42, 114, 44, 0],
            &[0, 0, 0, 70, 208, 255, 255, 254, 202, 38, 0],
            &[0, 0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'f' / 0x66
        'f' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 52, 50, 18, 0, 0],
            &[0, 0, 0, 0, 103, 247, 253, 249, 245, 6, 0],
            &[0, 0, 0, 9, 246, 168, 9, 0, 21, 0, 0],
            &[0, 0, 0, 34, 255, 83, 0, 0, 0, 0, 0],
            &[0, 97, 121, 156, 255, 179, 147, 147, 77, 0, 0],
            &[0, 150, 159, 173, 255, 188, 159, 159, 84, 0, 0],
            &[0, 0, 0, 36, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'g' / 0x67
        'g' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 155, 182, 142, 22, 130, 47, 0],
            &[0, 0, 52, 246, 201, 124, 153, 213, 253, 81, 0],
            &[0, 0, 188, 212, 6, 0, 0, 115, 255, 81, 0],
            &[0, 6, 250, 125, 0, 0, 0, 25, 255, 81, 0],
            &[0, 25, 255, 97, 0, 0, 0, 1, 253, 81, 0],
            &[0, 18, 255, 110, 0, 0, 0, 8, 254, 81, 0],
            &[0, 0, 228, 169, 0, 0, 0, 56, 255, 81, 0],
            &[0, 0, 123, 251, 103, 17, 43, 203, 255, 81, 0],
            &[0, 0, 3, 147, 250, 255, 249, 124, 254, 81, 0],
            &[0, 0, 0, 0, 12, 35, 11, 16, 255, 74, 0],
            &[0, 0, 0, 0, 0, 0, 0, 76, 255, 37, 0],
            &[0, 0, 71, 170, 211, 229, 208, 132, 11, 0, 0],
        ]),
        // letter: 'h' / 0x68
        'h' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 29, 29, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 184, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 184, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 184, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 182, 33, 146, 178, 137, 23, 0, 0, 0],
            &[0, 184, 201, 217, 137, 131, 234, 218, 8, 0, 0],
            &[0, 184, 255, 59, 0, 0, 71, 255, 68, 0, 0],
            &[0, 184, 215, 0, 0, 0, 17, 255, 99, 0, 0],
            &[0, 184, 187, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'i' / 0x69
        'i' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 235, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 173, 135, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 147, 147, 147, 114, 0, 0, 0, 0, 0],
            &[0, 51, 124, 143, 222, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 14, 34, 185, 209, 35, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'j' / 0x6a
        'j' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 95, 255, 83, 0, 0],
            &[0, 0, 0, 0, 0, 0, 57, 200, 49, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 147, 147, 147, 51, 0, 0],
            &[0, 0, 0, 0, 102, 132, 160, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 65, 255, 62, 0, 0],
            &[0, 0, 6, 191, 224, 219, 166, 34, 0, 0, 0],
        ]),
        // letter: 'k' / 0x6b
        'k' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 40, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 30, 0, 0, 17, 143, 92, 0, 0],
            &[0, 81, 255, 30, 0, 9, 196, 212, 18, 0, 0],
            &[0, 81, 255, 30, 3, 177, 224, 27, 0, 0, 0],
            &[0, 81, 255, 25, 154, 233, 37, 0, 0, 0, 0],
            &[0, 81, 255, 131, 255, 196, 3, 0, 0, 0, 0],
            &[0, 81, 255, 217, 65, 242, 131, 0, 0, 0, 0],
            &[0, 81, 255, 44, 0, 91, 253, 68, 0, 0, 0],
            &[0, 81, 255, 30, 0, 0, 162, 232, 24, 0, 0],
            &[0, 81, 255, 30, 0, 0, 12, 219, 188, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'l' / 0x6c
        'l' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 40, 40, 40, 35, 0, 0, 0, 0, 0],
            &[0, 136, 232, 249, 255, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 152, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 13, 33, 169, 226, 36, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'm' / 0x6d
        'm' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 29, 126, 63, 177, 116, 20, 161, 164, 25, 0],
            &[0, 51, 252, 209, 173, 254, 207, 168, 235, 148, 0],
            &[0, 51, 255, 89, 2, 237, 197, 0, 141, 187, 0],
            &[0, 51, 255, 39, 0, 214, 141, 0, 121, 206, 0],
            &[0, 51, 255, 24, 0, 210, 121, 0, 118, 208, 0],
            &[0, 51, 255, 20, 0, 210, 116, 0, 118, 208, 0],
            &[0, 51, 255, 20, 0, 210, 116, 0, 118, 208, 0],
            &[0, 51, 255, 20, 0, 210, 116, 0, 118, 208, 0],
            &[0, 51, 255, 20, 0, 210, 116, 0, 118, 208, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'n' / 0x6e
        'n' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 106, 70, 44, 149, 179, 143, 29, 0, 0, 0],
            &[0, 184, 194, 222, 139, 126, 224, 225, 12, 0, 0],
            &[0, 184, 255, 48, 0, 0, 58, 255, 73, 0, 0],
            &[0, 184, 215, 0, 0, 0, 14, 255, 100, 0, 0],
            &[0, 184, 188, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'o' / 0x6f
        'o' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 133, 178, 165, 89, 1, 0, 0],
            &[0, 0, 39, 235, 207, 127, 143, 242, 164, 0, 0],
            &[0, 0, 180, 213, 8, 0, 0, 68, 255, 73, 0],
            &[0, 4, 250, 117, 0, 0, 0, 0, 223, 150, 0],
            &[0, 24, 255, 85, 0, 0, 0, 0, 193, 177, 0],
            &[0, 14, 255, 96, 0, 0, 0, 0, 206, 169, 0],
            &[0, 0, 217, 163, 0, 0, 0, 21, 250, 116, 0],
            &[0, 0, 98, 252, 105, 13, 34, 184, 235, 20, 0],
            &[0, 0, 0, 112, 237, 255, 255, 206, 50, 0, 0],
            &[0, 0, 0, 0, 3, 33, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'p' / 0x70
        'p' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 106, 74, 55, 159, 180, 128, 15, 0, 0, 0],
            &[0, 184, 207, 220, 132, 137, 234, 206, 6, 0, 0],
            &[0, 184, 252, 34, 0, 0, 50, 255, 94, 0, 0],
            &[0, 184, 204, 0, 0, 0, 0, 222, 158, 0, 0],
            &[0, 184, 183, 0, 0, 0, 0, 199, 179, 0, 0],
            &[0, 184, 198, 0, 0, 0, 0, 212, 167, 0, 0],
            &[0, 184, 242, 8, 0, 0, 17, 247, 123, 0, 0],
            &[0, 184, 252, 154, 26, 29, 168, 247, 31, 0, 0],
            &[0, 184, 175, 153, 253, 255, 229, 76, 0, 0, 0],
            &[0, 184, 180, 0, 11, 30, 3, 0, 0, 0, 0],
            &[0, 184, 180, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 154, 151, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'q' / 0x71
        'q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 43, 153, 181, 134, 11, 130, 47, 0],
            &[0, 0, 52, 245, 196, 124, 165, 197, 252, 81, 0],
            &[0, 0, 189, 209, 4, 0, 0, 145, 255, 81, 0],
            &[0, 7, 250, 124, 0, 0, 0, 59, 255, 81, 0],
            &[0, 25, 255, 97, 0, 0, 0, 32, 255, 81, 0],
            &[0, 17, 255, 110, 0, 0, 0, 39, 255, 81, 0],
            &[0, 0, 226, 169, 0, 0, 0, 87, 255, 81, 0],
            &[0, 0, 122, 251, 99, 16, 54, 214, 255, 81, 0],
            &[0, 0, 4, 149, 250, 255, 236, 89, 255, 81, 0],
            &[0, 0, 0, 0, 13, 33, 6, 26, 255, 81, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 255, 81, 0],
            &[0, 0, 0, 0, 0, 0, 0, 22, 214, 68, 0],
        ]),
        // letter: 'r' / 0x72
        'r' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 110, 147, 147, 116, 11, 137, 180, 149, 10, 0],
            &[0, 86, 134, 194, 226, 172, 202, 144, 172, 0, 0],
            &[0, 0, 0, 102, 255, 186, 3, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 14, 0, 0, 0, 0, 0],
            &[0, 3, 25, 128, 255, 51, 15, 0, 0, 0, 0],
            &[1, 253, 255, 255, 255, 255, 255, 139, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 's' / 0x73
        's' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 127, 177, 178, 146, 62, 0, 0],
            &[0, 0, 4, 217, 217, 132, 127, 169, 122, 0, 0],
            &[0, 0, 36, 255, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 222, 207, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 167, 253, 199, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 154, 254, 131, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 227, 0, 0],
            &[0, 0, 53, 102, 31, 10, 41, 209, 192, 0, 0],
            &[0, 0, 54, 223, 255, 255, 255, 200, 39, 0, 0],
            &[0, 0, 0, 0, 17, 35, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 't' / 0x74
        't' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 118, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 167, 171, 0, 0, 0, 0, 0],
            &[0, 9, 111, 139, 230, 220, 147, 147, 147, 2, 0],
            &[0, 16, 159, 159, 233, 224, 159, 159, 159, 2, 0],
            &[0, 0, 0, 0, 196, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 189, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 246, 73, 11, 28, 14, 0],
            &[0, 0, 0, 0, 22, 192, 255, 255, 254, 60, 0],
            &[0, 0, 0, 0, 0, 0, 18, 35, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'u' / 0x75
        'u' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 117, 94, 0, 0, 0, 17, 147, 47, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 41, 255, 81, 0, 0],
            &[0, 191, 183, 0, 0, 0, 91, 255, 81, 0, 0],
            &[0, 139, 248, 70, 10, 51, 214, 255, 81, 0, 0],
            &[0, 19, 191, 255, 255, 230, 74, 235, 81, 0, 0],
            &[0, 0, 0, 16, 30, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'v' / 0x76
        'v' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 147, 43, 0, 0, 0, 0, 108, 111, 0],
            &[0, 1, 224, 146, 0, 0, 0, 12, 246, 118, 0],
            &[0, 0, 131, 232, 3, 0, 0, 89, 252, 26, 0],
            &[0, 0, 37, 255, 70, 0, 0, 177, 183, 0, 0],
            &[0, 0, 0, 198, 160, 0, 16, 249, 87, 0, 0],
            &[0, 0, 0, 103, 241, 8, 97, 239, 8, 0, 0],
            &[0, 0, 0, 16, 248, 84, 187, 152, 0, 0, 0],
            &[0, 0, 0, 0, 170, 194, 252, 57, 0, 0, 0],
            &[0, 0, 0, 0, 75, 255, 217, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'w' / 0x77
        'w' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[102, 94, 0, 0, 0, 0, 0, 1, 145, 46, 0],
            &[146, 194, 0, 0, 0, 0, 0, 31, 255, 46, 0],
            &[105, 231, 0, 33, 245, 176, 0, 71, 252, 7, 0],
            &[64, 255, 13, 107, 225, 245, 7, 112, 218, 0, 0],
            &[23, 255, 50, 180, 128, 234, 67, 152, 176, 0, 0],
            &[0, 237, 89, 244, 59, 171, 136, 187, 134, 0, 0],
            &[0, 196, 159, 242, 4, 106, 198, 212, 92, 0, 0],
            &[0, 155, 226, 179, 0, 40, 246, 235, 50, 0, 0],
            &[0, 114, 255, 113, 0, 0, 230, 254, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'x' / 0x78
        'x' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 131, 103, 0, 0, 0, 17, 145, 74, 0],
            &[0, 0, 84, 254, 69, 0, 0, 170, 221, 14, 0],
            &[0, 0, 0, 151, 230, 20, 98, 249, 52, 0, 0],
            &[0, 0, 0, 8, 209, 192, 243, 113, 0, 0, 0],
            &[0, 0, 0, 0, 57, 255, 206, 0, 0, 0, 0],
            &[0, 0, 0, 0, 175, 226, 252, 71, 0, 0, 0],
            &[0, 0, 0, 109, 247, 45, 140, 233, 25, 0, 0],
            &[0, 0, 50, 248, 107, 0, 6, 205, 189, 2, 0],
            &[0, 13, 219, 177, 0, 0, 0, 39, 244, 126, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'y' / 0x79
        'y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 23, 147, 36, 0, 0, 0, 0, 103, 111, 0],
            &[0, 0, 212, 144, 0, 0, 0, 6, 240, 118, 0],
            &[0, 0, 106, 239, 9, 0, 0, 77, 252, 26, 0],
            &[0, 0, 13, 242, 96, 0, 0, 162, 184, 0, 0],
            &[0, 0, 0, 149, 200, 0, 7, 241, 89, 0, 0],
            &[0, 0, 0, 43, 255, 47, 77, 241, 9, 0, 0],
            &[0, 0, 0, 0, 192, 141, 160, 155, 0, 0, 0],
            &[0, 0, 0, 0, 86, 224, 236, 60, 0, 0, 0],
            &[0, 0, 0, 0, 5, 230, 220, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 216, 126, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 250, 28, 0, 0, 0, 0],
            &[0, 0, 215, 226, 157, 12, 0, 0, 0, 0, 0],
        ]),
        // letter: 'z' / 0x7a
        'z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 44, 147, 147, 147, 147, 147, 147, 4, 0, 0],
            &[0, 41, 135, 135, 135, 135, 220, 239, 4, 0, 0],
            &[0, 0, 0, 0, 0, 72, 252, 77, 0, 0, 0],
            &[0, 0, 0, 0, 31, 237, 130, 0, 0, 0, 0],
            &[0, 0, 0, 7, 203, 183, 2, 0, 0, 0, 0],
            &[0, 0, 0, 153, 222, 18, 0, 0, 0, 0, 0],
            &[0, 0, 96, 247, 49, 0, 0, 0, 0, 0, 0],
            &[0, 48, 246, 147, 55, 55, 55, 55, 8, 0, 0],
            &[0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '{' / 0x7b
        '{' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 85, 37, 0, 0, 0],
            &[0, 0, 0, 0, 59, 241, 231, 70, 0, 0, 0],
            &[0, 0, 0, 0, 182, 208, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 206, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 206, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 213, 145, 0, 0, 0, 0, 0],
            &[0, 0, 16, 90, 251, 82, 0, 0, 0, 0, 0],
            &[0, 0, 194, 249, 111, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 110, 251, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 215, 144, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 206, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 206, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 188, 190, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 251, 204, 59, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 111, 47, 0, 0, 0],
        ]),
        // letter: '|' / 0x7c
        '|' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 84, 0, 0, 0, 0, 0],
        ]),
        // letter: '}' / 0x7d
        '}' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 75, 63, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 151, 247, 195, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 79, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 252, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 248, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 189, 201, 40, 3, 0, 0, 0],
            &[0, 0, 0, 0, 18, 190, 255, 92, 0, 0, 0],
            &[0, 0, 0, 0, 167, 217, 72, 15, 0, 0, 0],
            &[0, 0, 0, 0, 245, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 252, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 255, 85, 0, 0, 0, 0, 0],
            &[0, 0, 135, 239, 212, 10, 0, 0, 0, 0, 0],
            &[0, 0, 96, 89, 8, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '~' / 0x7e
        '~' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 81, 6, 0, 0, 62, 35, 0, 0],
            &[0, 72, 216, 177, 209, 30, 0, 200, 54, 0, 0],
            &[0, 168, 84, 0, 99, 230, 162, 205, 1, 0, 0],
            &[0, 81, 18, 0, 0, 37, 87, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ' ' / 0xa0
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¡' / 0xa1
        '¡' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 125, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 180, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 108, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 49, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 216, 110, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 224, 118, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 232, 126, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 240, 134, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 248, 142, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 255, 150, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 255, 158, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 245, 159, 0, 0, 0, 0],
        ]),
        // letter: '¢' / 0xa2
        '¢' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 120, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 98, 221, 57, 14, 0, 0, 0],
            &[0, 0, 44, 212, 255, 240, 253, 243, 10, 0, 0],
            &[0, 10, 226, 195, 29, 0, 6, 46, 0, 0, 0],
            &[0, 85, 255, 40, 0, 0, 0, 0, 0, 0, 0],
            &[0, 128, 238, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 136, 231, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 110, 252, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 252, 137, 0, 0, 0, 4, 0, 0, 0],
            &[0, 0, 118, 254, 207, 166, 184, 237, 4, 0, 0],
            &[0, 0, 0, 47, 158, 234, 133, 76, 0, 0, 0],
            &[0, 0, 0, 0, 71, 202, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '£' / 0xa3
        '£' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 60, 128, 124, 63, 0, 0, 0],
            &[0, 0, 0, 105, 254, 186, 180, 239, 21, 0, 0],
            &[0, 0, 0, 233, 143, 0, 0, 10, 0, 0, 0],
            &[0, 0, 14, 255, 91, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 255, 85, 0, 0, 0, 0, 0, 0],
            &[0, 23, 60, 255, 115, 45, 45, 7, 0, 0, 0],
            &[0, 115, 227, 255, 235, 225, 225, 37, 0, 0, 0],
            &[0, 0, 18, 255, 85, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 255, 81, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 254, 34, 0, 0, 0, 0, 0, 0],
            &[0, 46, 211, 193, 71, 71, 71, 71, 28, 0, 0],
            &[0, 143, 255, 255, 255, 255, 255, 255, 102, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¤' / 0xa4
        '¤' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 31, 15, 0, 0, 0, 0, 45, 2, 0, 0],
            &[0, 155, 204, 162, 231, 215, 140, 245, 57, 0, 0],
            &[0, 9, 241, 158, 46, 67, 217, 155, 0, 0, 0],
            &[0, 63, 216, 0, 0, 0, 65, 223, 0, 0, 0],
            &[0, 84, 199, 0, 0, 0, 38, 246, 0, 0, 0],
            &[0, 25, 243, 53, 0, 0, 139, 182, 0, 0, 0],
            &[0, 72, 245, 242, 190, 211, 248, 204, 14, 0, 0],
            &[0, 116, 86, 29, 93, 78, 14, 160, 41, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¥' / 0xa5
        '¥' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[35, 108, 16, 0, 0, 0, 0, 58, 100, 0, 0],
            &[12, 235, 128, 0, 0, 0, 5, 223, 142, 0, 0],
            &[0, 116, 239, 14, 0, 0, 100, 243, 21, 0, 0],
            &[0, 9, 231, 123, 0, 4, 222, 130, 0, 0, 0],
            &[0, 0, 110, 236, 12, 99, 237, 15, 0, 0, 0],
            &[0, 0, 7, 227, 122, 222, 119, 0, 0, 0, 0],
            &[0, 22, 108, 184, 252, 253, 118, 87, 0, 0, 0],
            &[0, 31, 153, 153, 246, 201, 153, 124, 0, 0, 0],
            &[0, 4, 22, 22, 233, 132, 22, 18, 0, 0, 0],
            &[0, 49, 239, 239, 254, 247, 239, 193, 0, 0, 0],
            &[0, 0, 0, 0, 231, 120, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 231, 120, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¦' / 0xa6
        '¦' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 202, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 202, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 202, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 202, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 202, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 189, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 202, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 202, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 202, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 202, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 173, 82, 0, 0, 0, 0, 0],
        ]),
        // letter: '§' / 0xa7
        '§' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 48, 48, 18, 0, 0, 0],
            &[0, 0, 0, 112, 243, 215, 218, 246, 119, 0, 0],
            &[0, 0, 6, 249, 104, 0, 0, 13, 22, 0, 0],
            &[0, 0, 2, 231, 171, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 49, 249, 241, 139, 26, 0, 0, 0],
            &[0, 0, 0, 183, 162, 76, 176, 242, 65, 0, 0],
            &[0, 0, 22, 255, 25, 0, 0, 143, 198, 0, 0],
            &[0, 0, 6, 234, 149, 9, 0, 126, 187, 0, 0],
            &[0, 0, 0, 44, 208, 235, 156, 228, 50, 0, 0],
            &[0, 0, 0, 0, 0, 71, 191, 240, 63, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 171, 194, 0, 0],
            &[0, 0, 32, 65, 4, 0, 11, 188, 171, 0, 0],
            &[0, 0, 44, 226, 247, 227, 247, 188, 27, 0, 0],
            &[0, 0, 0, 1, 20, 34, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¨' / 0xa8
        '¨' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 123, 0, 12, 182, 59, 0, 0, 0],
            &[0, 0, 180, 170, 0, 23, 240, 87, 0, 0, 0],
            &[0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '©' / 0xa9
        '©' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 165, 199, 197, 136, 18, 0, 0],
            &[0, 0, 82, 190, 46, 55, 66, 100, 208, 18, 0],
            &[0, 5, 211, 14, 172, 169, 154, 44, 99, 139, 0],
            &[0, 53, 159, 43, 198, 0, 0, 0, 9, 207, 0],
            &[0, 67, 146, 65, 170, 0, 0, 0, 0, 211, 0],
            &[0, 29, 194, 19, 220, 30, 14, 8, 38, 179, 0],
            &[0, 0, 176, 88, 71, 182, 187, 32, 178, 70, 0],
            &[0, 0, 18, 179, 172, 116, 128, 201, 102, 0, 0],
            &[0, 0, 0, 0, 36, 76, 66, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ª' / 0xaa
        'ª' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 88, 131, 113, 17, 0, 0, 0],
            &[0, 0, 0, 12, 136, 83, 162, 172, 0, 0, 0],
            &[0, 0, 0, 0, 29, 81, 129, 215, 0, 0, 0],
            &[0, 0, 0, 66, 215, 102, 107, 217, 0, 0, 0],
            &[0, 0, 0, 131, 153, 1, 117, 217, 0, 0, 0],
            &[0, 0, 0, 36, 192, 196, 104, 164, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '«' / 0xab
        '«' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 80, 0, 10, 95, 0, 0, 0],
            &[0, 0, 6, 197, 181, 1, 171, 202, 6, 0, 0],
            &[0, 0, 156, 217, 15, 127, 232, 28, 0, 0, 0],
            &[0, 97, 251, 41, 71, 254, 66, 0, 0, 0, 0],
            &[0, 36, 237, 115, 22, 224, 147, 0, 0, 0, 0],
            &[0, 0, 67, 247, 65, 46, 243, 95, 0, 0, 0],
            &[0, 0, 0, 108, 195, 0, 80, 221, 7, 0, 0],
            &[0, 0, 0, 0, 6, 0, 0, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¬' / 0xac
        '¬' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 53, 53, 53, 53, 53, 53, 12, 0, 0],
            &[0, 189, 237, 237, 237, 237, 237, 253, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 229, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 229, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 218, 58, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '®' / 0xae
        '®' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 165, 199, 197, 136, 18, 0, 0],
            &[0, 0, 82, 190, 86, 64, 39, 96, 208, 18, 0],
            &[0, 5, 211, 14, 163, 148, 203, 78, 99, 139, 0],
            &[0, 53, 159, 0, 163, 45, 115, 109, 9, 207, 0],
            &[0, 67, 146, 0, 163, 200, 219, 14, 0, 211, 0],
            &[0, 29, 194, 0, 163, 45, 183, 55, 38, 179, 0],
            &[0, 0, 176, 88, 112, 30, 41, 134, 178, 70, 0],
            &[0, 0, 18, 179, 172, 116, 128, 201, 102, 0, 0],
            &[0, 0, 0, 0, 36, 76, 66, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¯' / 0xaf
        '¯' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 6, 6, 6, 6, 1, 0, 0, 0],
            &[0, 0, 145, 255, 255, 255, 255, 47, 0, 0, 0],
            &[0, 0, 16, 28, 28, 28, 28, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '°' / 0xb0
        '°' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 49, 137, 116, 13, 0, 0, 0, 0],
            &[0, 0, 52, 227, 114, 151, 199, 0, 0, 0, 0],
            &[0, 0, 144, 111, 0, 0, 209, 42, 0, 0, 0],
            &[0, 0, 129, 143, 0, 8, 227, 28, 0, 0, 0],
            &[0, 0, 23, 217, 195, 220, 139, 0, 0, 0, 0],
            &[0, 0, 0, 5, 56, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '±' / 0xb1
        '±' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 206, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 206, 88, 0, 0, 0, 0, 0],
            &[0, 51, 65, 65, 219, 131, 65, 65, 20, 0, 0],
            &[0, 176, 225, 225, 250, 235, 225, 225, 72, 0, 0],
            &[0, 0, 0, 0, 206, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 206, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 206, 88, 0, 0, 0, 0, 0],
            &[0, 25, 34, 34, 38, 36, 34, 34, 11, 0, 0],
            &[0, 184, 255, 255, 255, 255, 255, 255, 81, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '²' / 0xb2
        '²' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 82, 148, 138, 32, 0, 0, 0, 0],
            &[0, 0, 47, 183, 90, 153, 215, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 246, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 144, 155, 0, 0, 0, 0],
            &[0, 0, 0, 0, 126, 191, 10, 0, 0, 0, 0],
            &[0, 0, 0, 131, 186, 10, 0, 0, 0, 0, 0],
            &[0, 0, 102, 253, 172, 165, 165, 46, 0, 0, 0],
            &[0, 0, 45, 88, 88, 88, 88, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '³' / 0xb3
        '³' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 97, 152, 135, 26, 0, 0, 0, 0],
            &[0, 0, 51, 164, 84, 174, 191, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 95, 215, 0, 0, 0, 0],
            &[0, 0, 0, 96, 170, 234, 72, 0, 0, 0, 0],
            &[0, 0, 0, 36, 67, 146, 211, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 255, 18, 0, 0, 0],
            &[0, 0, 117, 158, 121, 191, 188, 0, 0, 0, 0],
            &[0, 0, 15, 89, 113, 81, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '´' / 0xb4
        '´' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 65, 9, 0, 0, 0],
            &[0, 0, 0, 0, 16, 233, 170, 0, 0, 0, 0],
            &[0, 0, 0, 0, 160, 182, 7, 0, 0, 0, 0],
            &[0, 0, 0, 3, 75, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'µ' / 0xb5
        'µ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 117, 94, 0, 0, 0, 17, 147, 47, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 255, 255, 0, 0, 0, 41, 255, 81, 0, 0],
            &[0, 255, 255, 0, 0, 0, 91, 255, 81, 0, 0],
            &[0, 255, 255, 70, 10, 51, 214, 255, 81, 0, 0],
            &[0, 224, 255, 255, 255, 230, 74, 235, 81, 0, 0],
            &[0, 204, 163, 16, 30, 3, 0, 0, 0, 0, 0],
            &[0, 204, 163, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 171, 137, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¶' / 0xb6
        '¶' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 40, 40, 40, 40, 25, 0],
            &[0, 0, 37, 207, 255, 255, 255, 192, 208, 159, 0],
            &[0, 0, 195, 255, 255, 255, 255, 16, 77, 159, 0],
            &[0, 19, 254, 255, 255, 255, 255, 16, 77, 159, 0],
            &[0, 43, 255, 255, 255, 255, 255, 16, 77, 159, 0],
            &[0, 39, 255, 255, 255, 255, 255, 16, 77, 159, 0],
            &[0, 9, 248, 255, 255, 255, 255, 16, 77, 159, 0],
            &[0, 0, 158, 255, 255, 255, 255, 16, 77, 159, 0],
            &[0, 0, 7, 125, 188, 185, 243, 16, 77, 159, 0],
            &[0, 0, 0, 0, 0, 0, 221, 16, 77, 159, 0],
            &[0, 0, 0, 0, 0, 0, 221, 16, 77, 159, 0],
            &[0, 0, 0, 0, 0, 0, 221, 16, 77, 159, 0],
            &[0, 0, 0, 0, 0, 0, 221, 16, 77, 159, 0],
            &[0, 0, 0, 0, 0, 0, 221, 16, 77, 159, 0],
            &[0, 0, 0, 0, 0, 0, 221, 16, 77, 159, 0],
            &[0, 0, 0, 0, 0, 0, 7, 0, 2, 5, 0],
        ]),
        // letter: '·' / 0xb7
        '·' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 255, 180, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 131, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¸' / 0xb8
        '¸' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 209, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 151, 227, 32, 0, 0, 0, 0],
            &[0, 0, 0, 169, 201, 130, 2, 0, 0, 0, 0],
        ]),
        // letter: '¹' / 0xb9
        '¹' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 110, 73, 0, 0, 0, 0, 0],
            &[0, 0, 69, 228, 226, 145, 0, 0, 0, 0, 0],
            &[0, 0, 41, 71, 119, 145, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 145, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 145, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 145, 0, 0, 0, 0, 0],
            &[0, 0, 14, 131, 255, 255, 133, 34, 0, 0, 0],
            &[0, 0, 10, 88, 129, 138, 88, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'º' / 0xba
        'º' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 43, 126, 109, 11, 0, 0, 0, 0],
            &[0, 0, 36, 227, 96, 152, 181, 0, 0, 0, 0],
            &[0, 0, 118, 139, 0, 2, 246, 17, 0, 0, 0],
            &[0, 0, 128, 125, 0, 0, 236, 22, 0, 0, 0],
            &[0, 0, 69, 201, 13, 69, 218, 0, 0, 0, 0],
            &[0, 0, 0, 117, 207, 185, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '»' / 0xbb
        '»' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 89, 0, 13, 96, 0, 0, 0, 0, 0],
            &[0, 53, 246, 80, 31, 236, 108, 0, 0, 0, 0],
            &[0, 0, 98, 243, 46, 69, 248, 67, 0, 0, 0],
            &[0, 0, 0, 162, 224, 7, 131, 237, 20, 0, 0],
            &[0, 0, 22, 227, 142, 11, 209, 169, 0, 0, 0],
            &[0, 4, 193, 183, 4, 171, 207, 10, 0, 0, 0],
            &[0, 74, 213, 15, 48, 225, 30, 0, 0, 0, 0],
            &[0, 0, 6, 0, 0, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¼' / 0xbc
        '¼' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 71, 85, 0, 0, 0, 0, 93, 38, 0, 0],
            &[161, 236, 200, 0, 0, 0, 59, 237, 14, 0, 0],
            &[44, 72, 200, 0, 0, 0, 191, 120, 0, 0, 0],
            &[0, 65, 200, 0, 0, 68, 232, 11, 0, 0, 0],
            &[0, 65, 200, 0, 0, 200, 111, 0, 0, 0, 0],
            &[88, 255, 255, 187, 91, 227, 7, 0, 0, 0, 0],
            &[15, 41, 58, 34, 211, 102, 3, 135, 52, 0, 0],
            &[0, 0, 0, 87, 221, 5, 124, 248, 90, 0, 0],
            &[0, 0, 3, 216, 94, 54, 205, 176, 90, 0, 0],
            &[0, 0, 96, 215, 14, 217, 84, 184, 113, 6, 0],
            &[0, 5, 223, 85, 38, 192, 192, 235, 214, 35, 0],
            &[0, 105, 207, 1, 0, 0, 0, 171, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '½' / 0xbd
        '½' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 71, 85, 0, 0, 0, 0, 93, 38, 0, 0],
            &[161, 236, 200, 0, 0, 0, 59, 237, 14, 0, 0],
            &[44, 72, 200, 0, 0, 0, 191, 120, 0, 0, 0],
            &[0, 65, 200, 0, 0, 68, 232, 11, 0, 0, 0],
            &[0, 65, 200, 0, 0, 200, 111, 0, 0, 0, 0],
            &[88, 255, 255, 187, 91, 227, 7, 0, 0, 0, 0],
            &[15, 41, 58, 34, 211, 128, 137, 160, 60, 0, 0],
            &[0, 0, 0, 87, 221, 80, 116, 114, 240, 2, 0],
            &[0, 0, 3, 216, 94, 0, 0, 48, 234, 1, 0],
            &[0, 0, 96, 215, 3, 0, 30, 215, 78, 0, 0],
            &[0, 5, 223, 85, 0, 37, 215, 75, 0, 0, 0],
            &[0, 105, 207, 1, 0, 177, 255, 253, 253, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¾' / 0xbe
        '¾' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[22, 115, 132, 43, 0, 0, 0, 93, 38, 0, 0],
            &[87, 131, 142, 234, 1, 0, 59, 237, 14, 0, 0],
            &[0, 41, 119, 220, 0, 0, 191, 120, 0, 0, 0],
            &[0, 172, 220, 172, 1, 68, 232, 11, 0, 0, 0],
            &[2, 0, 21, 255, 34, 200, 111, 0, 0, 0, 0],
            &[137, 194, 217, 192, 81, 227, 7, 0, 0, 0, 0],
            &[4, 46, 38, 2, 208, 102, 3, 135, 52, 0, 0],
            &[0, 0, 0, 87, 221, 5, 124, 248, 90, 0, 0],
            &[0, 0, 3, 216, 94, 54, 205, 176, 90, 0, 0],
            &[0, 0, 96, 215, 14, 217, 84, 184, 113, 6, 0],
            &[0, 5, 223, 85, 38, 192, 192, 235, 214, 35, 0],
            &[0, 105, 207, 1, 0, 0, 0, 171, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¿' / 0xbf
        '¿' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 118, 72, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 255, 192, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 61, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 42, 31, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 121, 0, 0, 0, 0],
            &[0, 0, 0, 0, 110, 252, 44, 0, 0, 0, 0],
            &[0, 0, 1, 141, 245, 80, 0, 0, 0, 0, 0],
            &[0, 0, 112, 242, 50, 0, 0, 0, 0, 0, 0],
            &[0, 0, 210, 140, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 231, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 166, 232, 81, 41, 68, 138, 90, 0, 0],
            &[0, 0, 0, 0, 7, 25, 12, 0, 0, 0, 0],
        ]),
        // letter: 'À' / 0xc0
        'À' => Some(&[
            &[0, 0, 19, 224, 185, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 39, 230, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 120, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 255, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 155, 199, 255, 49, 0, 0, 0, 0],
            &[0, 0, 3, 234, 105, 210, 132, 0, 0, 0, 0],
            &[0, 0, 64, 254, 27, 133, 215, 0, 0, 0, 0],
            &[0, 0, 147, 203, 0, 53, 255, 42, 0, 0, 0],
            &[0, 1, 228, 123, 0, 1, 229, 126, 0, 0, 0],
            &[0, 57, 255, 245, 200, 200, 255, 209, 0, 0, 0],
            &[0, 140, 255, 127, 126, 126, 187, 255, 36, 0, 0],
            &[0, 222, 141, 0, 0, 0, 6, 241, 119, 0, 0],
            &[49, 255, 61, 0, 0, 0, 0, 168, 202, 0, 0],
            &[132, 235, 3, 0, 0, 0, 0, 88, 255, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 26, 10, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Á' / 0xc1
        'Á' => Some(&[
            &[0, 0, 0, 0, 0, 174, 230, 25, 0, 0, 0],
            &[0, 0, 0, 0, 80, 236, 45, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 120, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 255, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 155, 199, 255, 49, 0, 0, 0, 0],
            &[0, 0, 3, 234, 105, 210, 132, 0, 0, 0, 0],
            &[0, 0, 64, 254, 27, 133, 215, 0, 0, 0, 0],
            &[0, 0, 147, 203, 0, 53, 255, 42, 0, 0, 0],
            &[0, 1, 228, 123, 0, 1, 229, 126, 0, 0, 0],
            &[0, 57, 255, 245, 200, 200, 255, 209, 0, 0, 0],
            &[0, 140, 255, 127, 126, 126, 187, 255, 36, 0, 0],
            &[0, 222, 141, 0, 0, 0, 6, 241, 119, 0, 0],
            &[49, 255, 61, 0, 0, 0, 0, 168, 202, 0, 0],
            &[132, 235, 3, 0, 0, 0, 0, 88, 255, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 26, 9, 0, 0, 0],
        ]),
        // letter: 'Â' / 0xc2
        'Â' => Some(&[
            &[0, 0, 0, 94, 253, 222, 15, 0, 0, 0, 0],
            &[0, 0, 58, 231, 59, 156, 189, 7, 0, 0, 0],
            &[0, 0, 96, 38, 0, 0, 89, 49, 0, 0, 0],
            &[0, 0, 0, 5, 120, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 255, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 155, 199, 255, 49, 0, 0, 0, 0],
            &[0, 0, 3, 234, 105, 210, 132, 0, 0, 0, 0],
            &[0, 0, 64, 254, 27, 133, 215, 0, 0, 0, 0],
            &[0, 0, 147, 203, 0, 53, 255, 42, 0, 0, 0],
            &[0, 1, 228, 123, 0, 1, 229, 126, 0, 0, 0],
            &[0, 57, 255, 245, 200, 200, 255, 209, 0, 0, 0],
            &[0, 140, 255, 127, 126, 126, 187, 255, 36, 0, 0],
            &[0, 222, 141, 0, 0, 0, 6, 241, 119, 0, 0],
            &[49, 255, 61, 0, 0, 0, 0, 168, 202, 0, 0],
            &[132, 235, 3, 0, 0, 0, 0, 88, 255, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 14, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ã' / 0xc3
        'Ã' => Some(&[
            &[0, 0, 34, 136, 88, 1, 39, 95, 0, 0, 0],
            &[0, 0, 206, 159, 216, 214, 225, 85, 0, 0, 0],
            &[0, 3, 97, 0, 4, 84, 71, 0, 0, 0, 0],
            &[0, 0, 0, 5, 120, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 255, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 155, 199, 255, 49, 0, 0, 0, 0],
            &[0, 0, 3, 234, 105, 210, 132, 0, 0, 0, 0],
            &[0, 0, 64, 254, 27, 133, 215, 0, 0, 0, 0],
            &[0, 0, 147, 203, 0, 53, 255, 42, 0, 0, 0],
            &[0, 1, 228, 123, 0, 1, 229, 126, 0, 0, 0],
            &[0, 57, 255, 245, 200, 200, 255, 209, 0, 0, 0],
            &[0, 140, 255, 127, 126, 126, 187, 255, 36, 0, 0],
            &[0, 222, 141, 0, 0, 0, 6, 241, 119, 0, 0],
            &[49, 255, 61, 0, 0, 0, 0, 168, 202, 0, 0],
            &[132, 235, 3, 0, 0, 0, 0, 88, 255, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ä' / 0xc4
        'Ä' => Some(&[
            &[0, 0, 101, 89, 0, 7, 144, 39, 0, 0, 0],
            &[0, 0, 202, 185, 0, 33, 255, 99, 0, 0, 0],
            &[0, 0, 15, 11, 0, 0, 25, 1, 0, 0, 0],
            &[0, 0, 0, 5, 120, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 255, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 155, 199, 255, 49, 0, 0, 0, 0],
            &[0, 0, 3, 234, 105, 210, 132, 0, 0, 0, 0],
            &[0, 0, 64, 254, 27, 133, 215, 0, 0, 0, 0],
            &[0, 0, 147, 203, 0, 53, 255, 42, 0, 0, 0],
            &[0, 1, 228, 123, 0, 1, 229, 126, 0, 0, 0],
            &[0, 57, 255, 245, 200, 200, 255, 209, 0, 0, 0],
            &[0, 140, 255, 127, 126, 126, 187, 255, 36, 0, 0],
            &[0, 222, 141, 0, 0, 0, 6, 241, 119, 0, 0],
            &[49, 255, 61, 0, 0, 0, 0, 168, 202, 0, 0],
            &[132, 235, 3, 0, 0, 0, 0, 88, 255, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Å' / 0xc5
        'Å' => Some(&[
            &[0, 0, 0, 229, 22, 125, 128, 0, 0, 0, 0],
            &[0, 0, 0, 195, 137, 199, 90, 0, 0, 0, 0],
            &[0, 0, 0, 18, 102, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 120, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 255, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 155, 199, 255, 49, 0, 0, 0, 0],
            &[0, 0, 3, 234, 105, 210, 132, 0, 0, 0, 0],
            &[0, 0, 64, 254, 27, 133, 215, 0, 0, 0, 0],
            &[0, 0, 147, 203, 0, 53, 255, 42, 0, 0, 0],
            &[0, 1, 228, 123, 0, 1, 229, 126, 0, 0, 0],
            &[0, 57, 255, 245, 200, 200, 255, 209, 0, 0, 0],
            &[0, 140, 255, 127, 126, 126, 187, 255, 36, 0, 0],
            &[0, 222, 141, 0, 0, 0, 6, 241, 119, 0, 0],
            &[49, 255, 61, 0, 0, 0, 0, 168, 202, 0, 0],
            &[132, 235, 3, 0, 0, 0, 0, 88, 255, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 214, 191, 27, 0, 0, 0, 0],
        ]),
        // letter: 'Æ' / 0xc6
        'Æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 108, 108, 108, 108, 108, 74, 0],
            &[0, 0, 0, 83, 250, 239, 248, 219, 219, 150, 0],
            &[0, 0, 0, 141, 188, 145, 206, 0, 0, 0, 0],
            &[0, 0, 0, 198, 133, 145, 206, 0, 0, 0, 0],
            &[0, 0, 7, 248, 79, 145, 206, 0, 0, 0, 0],
            &[0, 0, 57, 255, 24, 145, 236, 155, 155, 59, 0],
            &[0, 0, 114, 225, 0, 145, 238, 167, 167, 64, 0],
            &[0, 0, 171, 255, 188, 255, 206, 0, 0, 0, 0],
            &[0, 0, 229, 238, 126, 246, 206, 0, 0, 0, 0],
            &[0, 30, 255, 62, 0, 145, 206, 0, 0, 0, 0],
            &[0, 87, 252, 11, 0, 145, 220, 71, 71, 49, 0],
            &[0, 145, 209, 0, 0, 145, 255, 255, 255, 176, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ç' / 0xc7
        'Ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 97, 139, 132, 93, 15, 0, 0],
            &[0, 0, 57, 232, 243, 189, 196, 241, 152, 0, 0],
            &[0, 20, 235, 188, 17, 0, 0, 9, 17, 0, 0],
            &[0, 123, 249, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 194, 186, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 229, 146, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 235, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 217, 158, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 212, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 87, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 193, 242, 116, 55, 59, 108, 48, 0, 0],
            &[0, 0, 12, 150, 246, 255, 255, 235, 57, 0, 0],
            &[0, 0, 0, 0, 26, 251, 29, 2, 0, 0, 0],
            &[0, 0, 0, 0, 44, 182, 197, 3, 0, 0, 0],
            &[0, 0, 0, 9, 209, 193, 91, 0, 0, 0, 0],
        ]),
        // letter: 'È' / 0xc8
        'È' => Some(&[
            &[0, 0, 9, 208, 206, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 218, 117, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 110, 0, 0, 0, 0, 0],
            &[0, 52, 108, 108, 108, 108, 108, 108, 26, 0, 0],
            &[0, 122, 254, 219, 219, 219, 219, 219, 52, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 251, 155, 155, 155, 155, 145, 0, 0, 0],
            &[0, 122, 252, 171, 171, 171, 171, 160, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 248, 71, 71, 71, 71, 71, 17, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 26, 13, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'É' / 0xc9
        'É' => Some(&[
            &[0, 0, 0, 0, 0, 149, 241, 39, 0, 0, 0],
            &[0, 0, 0, 0, 58, 242, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 49, 0, 0, 0, 0, 0],
            &[0, 52, 108, 108, 108, 108, 108, 108, 26, 0, 0],
            &[0, 122, 254, 219, 219, 219, 219, 219, 52, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 251, 155, 155, 155, 155, 145, 0, 0, 0],
            &[0, 122, 252, 171, 171, 171, 171, 160, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 248, 71, 71, 71, 71, 71, 17, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 26, 12, 0, 0, 0],
        ]),
        // letter: 'Ê' / 0xca
        'Ê' => Some(&[
            &[0, 0, 0, 71, 252, 236, 27, 0, 0, 0, 0],
            &[0, 0, 41, 230, 74, 133, 207, 14, 0, 0, 0],
            &[0, 0, 85, 50, 0, 0, 79, 60, 0, 0, 0],
            &[0, 52, 108, 108, 108, 108, 108, 108, 26, 0, 0],
            &[0, 122, 254, 219, 219, 219, 219, 219, 52, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 251, 155, 155, 155, 155, 145, 0, 0, 0],
            &[0, 122, 252, 171, 171, 171, 171, 160, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 248, 71, 71, 71, 71, 71, 17, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 16, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ë' / 0xcb
        'Ë' => Some(&[
            &[0, 0, 85, 105, 0, 0, 137, 53, 0, 0, 0],
            &[0, 0, 177, 209, 0, 13, 250, 123, 0, 0, 0],
            &[0, 0, 10, 16, 0, 0, 23, 3, 0, 0, 0],
            &[0, 52, 108, 108, 108, 108, 108, 108, 26, 0, 0],
            &[0, 122, 254, 219, 219, 219, 219, 219, 52, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 251, 155, 155, 155, 155, 145, 0, 0, 0],
            &[0, 122, 252, 171, 171, 171, 171, 160, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 248, 71, 71, 71, 71, 71, 17, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ì' / 0xcc
        'Ì' => Some(&[
            &[0, 0, 0, 19, 224, 185, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 230, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 100, 0, 0, 0, 0],
            &[0, 0, 26, 108, 108, 108, 108, 108, 91, 0, 0],
            &[0, 0, 35, 162, 185, 252, 224, 175, 130, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 15, 38, 242, 151, 28, 6, 0, 0],
            &[0, 0, 61, 255, 255, 255, 255, 255, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 26, 10, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Í' / 0xcd
        'Í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 174, 230, 25, 0, 0],
            &[0, 0, 0, 0, 0, 80, 236, 45, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 38, 0, 0, 0, 0],
            &[0, 0, 26, 108, 108, 108, 108, 108, 91, 0, 0],
            &[0, 0, 35, 162, 185, 252, 224, 175, 130, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 15, 38, 242, 151, 28, 6, 0, 0],
            &[0, 0, 61, 255, 255, 255, 255, 255, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 26, 9, 0, 0],
        ]),
        // letter: 'Î' / 0xce
        'Î' => Some(&[
            &[0, 0, 0, 0, 94, 253, 222, 15, 0, 0, 0],
            &[0, 0, 0, 58, 231, 59, 156, 189, 7, 0, 0],
            &[0, 0, 0, 96, 38, 0, 0, 89, 49, 0, 0],
            &[0, 0, 26, 108, 108, 108, 108, 108, 91, 0, 0],
            &[0, 0, 35, 162, 185, 252, 224, 175, 130, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 15, 38, 242, 151, 28, 6, 0, 0],
            &[0, 0, 61, 255, 255, 255, 255, 255, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 14, 0, 0, 0, 0],
        ]),
        // letter: 'Ï' / 0xcf
        'Ï' => Some(&[
            &[0, 0, 0, 101, 89, 0, 7, 144, 39, 0, 0],
            &[0, 0, 0, 202, 185, 0, 33, 255, 99, 0, 0],
            &[0, 0, 0, 15, 11, 0, 0, 25, 1, 0, 0],
            &[0, 0, 26, 108, 108, 108, 108, 108, 91, 0, 0],
            &[0, 0, 35, 162, 185, 252, 224, 175, 130, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 15, 38, 242, 151, 28, 6, 0, 0],
            &[0, 0, 61, 255, 255, 255, 255, 255, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ð' / 0xd0
        'Ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 78, 108, 108, 95, 56, 0, 0, 0, 0, 0],
            &[0, 184, 243, 211, 228, 255, 213, 46, 0, 0, 0],
            &[0, 184, 184, 0, 0, 40, 201, 236, 24, 0, 0],
            &[0, 184, 184, 0, 0, 0, 23, 241, 142, 0, 0],
            &[0, 184, 184, 0, 0, 0, 0, 165, 219, 0, 0],
            &[69, 253, 253, 69, 46, 0, 0, 125, 249, 0, 0],
            &[249, 255, 255, 249, 167, 0, 0, 115, 254, 3, 0],
            &[0, 184, 184, 0, 0, 0, 0, 139, 235, 0, 0],
            &[0, 184, 184, 0, 0, 0, 0, 205, 178, 0, 0],
            &[0, 184, 184, 0, 0, 0, 94, 255, 78, 0, 0],
            &[0, 184, 202, 65, 86, 160, 252, 150, 0, 0, 0],
            &[0, 184, 255, 254, 235, 190, 88, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ñ' / 0xd1
        'Ñ' => Some(&[
            &[0, 0, 25, 134, 97, 4, 27, 107, 0, 0, 0],
            &[0, 0, 187, 166, 208, 221, 219, 105, 0, 0, 0],
            &[0, 0, 101, 0, 2, 78, 79, 0, 0, 0, 0],
            &[0, 95, 108, 12, 0, 0, 0, 98, 52, 0, 0],
            &[0, 225, 255, 109, 0, 0, 0, 233, 122, 0, 0],
            &[0, 225, 198, 219, 2, 0, 0, 233, 122, 0, 0],
            &[0, 225, 120, 241, 77, 0, 0, 233, 122, 0, 0],
            &[0, 225, 123, 142, 189, 0, 0, 233, 122, 0, 0],
            &[0, 225, 130, 32, 252, 45, 0, 233, 122, 0, 0],
            &[0, 225, 131, 0, 173, 156, 0, 233, 122, 0, 0],
            &[0, 225, 131, 0, 61, 247, 21, 230, 122, 0, 0],
            &[0, 225, 131, 0, 0, 204, 124, 223, 122, 0, 0],
            &[0, 225, 131, 0, 0, 92, 230, 214, 122, 0, 0],
            &[0, 225, 131, 0, 0, 5, 230, 254, 122, 0, 0],
            &[0, 225, 131, 0, 0, 0, 123, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ò' / 0xd2
        'Ò' => Some(&[
            &[0, 0, 0, 19, 224, 185, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 230, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 100, 0, 0, 0, 0],
            &[0, 0, 0, 5, 96, 145, 137, 59, 0, 0, 0],
            &[0, 0, 12, 204, 245, 184, 199, 254, 120, 0, 0],
            &[0, 0, 136, 241, 39, 0, 0, 114, 253, 41, 0],
            &[0, 0, 234, 144, 0, 0, 0, 7, 244, 135, 0],
            &[0, 34, 255, 84, 0, 0, 0, 0, 188, 189, 0],
            &[0, 58, 255, 58, 0, 0, 0, 0, 166, 216, 0],
            &[0, 67, 255, 52, 0, 0, 0, 0, 155, 221, 0],
            &[0, 54, 255, 66, 0, 0, 0, 0, 168, 206, 0],
            &[0, 18, 254, 105, 0, 0, 0, 0, 206, 168, 0],
            &[0, 0, 203, 191, 0, 0, 0, 35, 254, 97, 0],
            &[0, 0, 78, 255, 144, 45, 64, 207, 219, 7, 0],
            &[0, 0, 0, 101, 235, 255, 255, 194, 35, 0, 0],
            &[0, 0, 0, 0, 4, 31, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 26, 10, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ó' / 0xd3
        'Ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 174, 230, 25, 0, 0],
            &[0, 0, 0, 0, 0, 80, 236, 45, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 38, 0, 0, 0, 0],
            &[0, 0, 0, 5, 96, 145, 137, 59, 0, 0, 0],
            &[0, 0, 12, 204, 245, 184, 199, 254, 120, 0, 0],
            &[0, 0, 136, 241, 39, 0, 0, 114, 253, 41, 0],
            &[0, 0, 234, 144, 0, 0, 0, 7, 244, 135, 0],
            &[0, 34, 255, 84, 0, 0, 0, 0, 188, 189, 0],
            &[0, 58, 255, 58, 0, 0, 0, 0, 166, 216, 0],
            &[0, 67, 255, 52, 0, 0, 0, 0, 155, 221, 0],
            &[0, 54, 255, 66, 0, 0, 0, 0, 168, 206, 0],
            &[0, 18, 254, 105, 0, 0, 0, 0, 206, 168, 0],
            &[0, 0, 203, 191, 0, 0, 0, 35, 254, 97, 0],
            &[0, 0, 78, 255, 144, 45, 64, 207, 219, 7, 0],
            &[0, 0, 0, 101, 235, 255, 255, 194, 35, 0, 0],
            &[0, 0, 0, 0, 4, 31, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 26, 9, 0, 0],
        ]),
        // letter: 'Ô' / 0xd4
        'Ô' => Some(&[
            &[0, 0, 0, 0, 94, 253, 222, 15, 0, 0, 0],
            &[0, 0, 0, 58, 231, 59, 156, 189, 7, 0, 0],
            &[0, 0, 0, 96, 38, 0, 0, 89, 49, 0, 0],
            &[0, 0, 0, 5, 96, 145, 137, 59, 0, 0, 0],
            &[0, 0, 12, 204, 245, 184, 199, 254, 120, 0, 0],
            &[0, 0, 136, 241, 39, 0, 0, 114, 253, 41, 0],
            &[0, 0, 234, 144, 0, 0, 0, 7, 244, 135, 0],
            &[0, 34, 255, 84, 0, 0, 0, 0, 188, 189, 0],
            &[0, 58, 255, 58, 0, 0, 0, 0, 166, 216, 0],
            &[0, 67, 255, 52, 0, 0, 0, 0, 155, 221, 0],
            &[0, 54, 255, 66, 0, 0, 0, 0, 168, 206, 0],
            &[0, 18, 254, 105, 0, 0, 0, 0, 206, 168, 0],
            &[0, 0, 203, 191, 0, 0, 0, 35, 254, 97, 0],
            &[0, 0, 78, 255, 144, 45, 64, 207, 219, 7, 0],
            &[0, 0, 0, 101, 235, 255, 255, 194, 35, 0, 0],
            &[0, 0, 0, 0, 4, 31, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 14, 0, 0, 0, 0],
        ]),
        // letter: 'Õ' / 0xd5
        'Õ' => Some(&[
            &[0, 0, 0, 34, 136, 88, 1, 39, 95, 0, 0],
            &[0, 0, 0, 206, 159, 216, 214, 225, 85, 0, 0],
            &[0, 0, 3, 97, 0, 4, 84, 71, 0, 0, 0],
            &[0, 0, 0, 5, 96, 145, 137, 59, 0, 0, 0],
            &[0, 0, 12, 204, 245, 184, 199, 254, 120, 0, 0],
            &[0, 0, 136, 241, 39, 0, 0, 114, 253, 41, 0],
            &[0, 0, 234, 144, 0, 0, 0, 7, 244, 135, 0],
            &[0, 34, 255, 84, 0, 0, 0, 0, 188, 189, 0],
            &[0, 58, 255, 58, 0, 0, 0, 0, 166, 216, 0],
            &[0, 67, 255, 52, 0, 0, 0, 0, 155, 221, 0],
            &[0, 54, 255, 66, 0, 0, 0, 0, 168, 206, 0],
            &[0, 18, 254, 105, 0, 0, 0, 0, 206, 168, 0],
            &[0, 0, 203, 191, 0, 0, 0, 35, 254, 97, 0],
            &[0, 0, 78, 255, 144, 45, 64, 207, 219, 7, 0],
            &[0, 0, 0, 101, 235, 255, 255, 194, 35, 0, 0],
            &[0, 0, 0, 0, 4, 31, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ö' / 0xd6
        'Ö' => Some(&[
            &[0, 0, 0, 101, 89, 0, 7, 144, 39, 0, 0],
            &[0, 0, 0, 202, 185, 0, 33, 255, 99, 0, 0],
            &[0, 0, 0, 15, 11, 0, 0, 25, 1, 0, 0],
            &[0, 0, 0, 5, 96, 145, 137, 59, 0, 0, 0],
            &[0, 0, 12, 204, 245, 184, 199, 254, 120, 0, 0],
            &[0, 0, 136, 241, 39, 0, 0, 114, 253, 41, 0],
            &[0, 0, 234, 144, 0, 0, 0, 7, 244, 135, 0],
            &[0, 34, 255, 84, 0, 0, 0, 0, 188, 189, 0],
            &[0, 58, 255, 58, 0, 0, 0, 0, 166, 216, 0],
            &[0, 67, 255, 52, 0, 0, 0, 0, 155, 221, 0],
            &[0, 54, 255, 66, 0, 0, 0, 0, 168, 206, 0],
            &[0, 18, 254, 105, 0, 0, 0, 0, 206, 168, 0],
            &[0, 0, 203, 191, 0, 0, 0, 35, 254, 97, 0],
            &[0, 0, 78, 255, 144, 45, 64, 207, 219, 7, 0],
            &[0, 0, 0, 101, 235, 255, 255, 194, 35, 0, 0],
            &[0, 0, 0, 0, 4, 31, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '×' / 0xd7
        '×' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0],
            &[0, 88, 182, 7, 0, 0, 49, 212, 16, 0, 0],
            &[0, 15, 202, 181, 7, 50, 237, 115, 0, 0, 0],
            &[0, 0, 15, 202, 190, 238, 115, 0, 0, 0, 0],
            &[0, 0, 0, 68, 255, 215, 7, 0, 0, 0, 0],
            &[0, 0, 51, 238, 127, 205, 182, 7, 0, 0, 0],
            &[0, 49, 237, 117, 0, 17, 202, 181, 6, 0, 0],
            &[0, 50, 111, 0, 0, 0, 15, 138, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ø' / 0xd8
        'Ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 96, 145, 137, 59, 123, 118, 0],
            &[0, 0, 12, 204, 245, 184, 199, 255, 255, 69, 0],
            &[0, 0, 136, 241, 39, 0, 0, 255, 255, 41, 0],
            &[0, 0, 234, 144, 0, 0, 56, 255, 255, 135, 0],
            &[0, 34, 255, 84, 0, 1, 200, 143, 188, 189, 0],
            &[0, 58, 255, 58, 0, 92, 236, 17, 166, 216, 0],
            &[0, 67, 255, 52, 10, 227, 107, 0, 155, 221, 0],
            &[0, 54, 255, 66, 128, 213, 4, 0, 168, 206, 0],
            &[0, 18, 254, 134, 245, 71, 0, 0, 206, 168, 0],
            &[0, 0, 203, 255, 181, 0, 0, 35, 254, 97, 0],
            &[0, 0, 135, 255, 184, 45, 64, 207, 219, 7, 0],
            &[0, 1, 201, 247, 235, 255, 255, 194, 35, 0, 0],
            &[0, 1, 94, 17, 4, 31, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ù' / 0xd9
        'Ù' => Some(&[
            &[0, 0, 19, 224, 185, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 39, 230, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 100, 0, 0, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 0, 104, 52, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 246, 122, 0, 0],
            &[0, 214, 164, 0, 0, 0, 9, 255, 112, 0, 0],
            &[0, 177, 209, 0, 0, 0, 52, 255, 72, 0, 0],
            &[0, 84, 255, 136, 41, 60, 203, 224, 7, 0, 0],
            &[0, 0, 121, 241, 255, 255, 206, 44, 0, 0, 0],
            &[0, 0, 0, 5, 31, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 26, 10, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ú' / 0xda
        'Ú' => Some(&[
            &[0, 0, 0, 0, 0, 174, 230, 25, 0, 0, 0],
            &[0, 0, 0, 0, 80, 236, 45, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 38, 0, 0, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 0, 104, 52, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 246, 122, 0, 0],
            &[0, 214, 164, 0, 0, 0, 9, 255, 112, 0, 0],
            &[0, 177, 209, 0, 0, 0, 52, 255, 72, 0, 0],
            &[0, 84, 255, 136, 41, 60, 203, 224, 7, 0, 0],
            &[0, 0, 121, 241, 255, 255, 206, 44, 0, 0, 0],
            &[0, 0, 0, 5, 31, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 26, 9, 0, 0, 0],
        ]),
        // letter: 'Û' / 0xdb
        'Û' => Some(&[
            &[0, 0, 0, 94, 253, 222, 15, 0, 0, 0, 0],
            &[0, 0, 58, 231, 59, 156, 189, 7, 0, 0, 0],
            &[0, 0, 96, 38, 0, 0, 89, 49, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 0, 104, 52, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 246, 122, 0, 0],
            &[0, 214, 164, 0, 0, 0, 9, 255, 112, 0, 0],
            &[0, 177, 209, 0, 0, 0, 52, 255, 72, 0, 0],
            &[0, 84, 255, 136, 41, 60, 203, 224, 7, 0, 0],
            &[0, 0, 121, 241, 255, 255, 206, 44, 0, 0, 0],
            &[0, 0, 0, 5, 31, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 14, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ü' / 0xdc
        'Ü' => Some(&[
            &[0, 0, 101, 89, 0, 7, 144, 39, 0, 0, 0],
            &[0, 0, 202, 185, 0, 33, 255, 99, 0, 0, 0],
            &[0, 0, 15, 11, 0, 0, 25, 1, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 0, 104, 52, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 246, 122, 0, 0],
            &[0, 214, 164, 0, 0, 0, 9, 255, 112, 0, 0],
            &[0, 177, 209, 0, 0, 0, 52, 255, 72, 0, 0],
            &[0, 84, 255, 136, 41, 60, 203, 224, 7, 0, 0],
            &[0, 0, 121, 241, 255, 255, 206, 44, 0, 0, 0],
            &[0, 0, 0, 5, 31, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ý' / 0xdd
        'Ý' => Some(&[
            &[0, 0, 0, 0, 0, 174, 230, 25, 0, 0, 0],
            &[0, 0, 0, 0, 80, 236, 45, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 38, 0, 0, 0, 0, 0],
            &[36, 108, 26, 0, 0, 0, 0, 64, 100, 0, 0],
            &[12, 235, 146, 0, 0, 0, 6, 230, 145, 0, 0],
            &[0, 117, 245, 19, 0, 0, 100, 246, 24, 0, 0],
            &[0, 10, 232, 127, 0, 2, 217, 141, 0, 0, 0],
            &[0, 0, 113, 235, 10, 82, 244, 22, 0, 0, 0],
            &[0, 0, 8, 229, 108, 201, 136, 0, 0, 0, 0],
            &[0, 0, 0, 108, 241, 242, 19, 0, 0, 0, 0],
            &[0, 0, 0, 7, 239, 144, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 26, 9, 0, 0, 0],
        ]),
        // letter: 'Þ' / 0xde
        'Þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 104, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 249, 88, 87, 76, 30, 0, 0, 0, 0],
            &[0, 122, 254, 231, 233, 252, 255, 163, 1, 0, 0],
            &[0, 122, 245, 0, 0, 6, 117, 255, 98, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 218, 162, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 204, 170, 0, 0],
            &[0, 122, 245, 0, 0, 0, 48, 252, 107, 0, 0],
            &[0, 122, 251, 145, 148, 176, 247, 195, 6, 0, 0],
            &[0, 122, 252, 174, 171, 151, 90, 5, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ß' / 0xdf
        'ß' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 55, 45, 5, 0, 0, 0, 0],
            &[0, 18, 187, 255, 254, 255, 239, 99, 0, 0, 0],
            &[0, 147, 238, 70, 5, 20, 164, 251, 19, 0, 0],
            &[0, 198, 163, 0, 0, 0, 96, 255, 21, 0, 0],
            &[0, 204, 155, 0, 0, 45, 228, 156, 0, 0, 0],
            &[0, 204, 155, 0, 32, 240, 148, 2, 0, 0, 0],
            &[0, 204, 155, 0, 97, 252, 11, 0, 0, 0, 0],
            &[0, 204, 155, 0, 35, 240, 194, 30, 0, 0, 0],
            &[0, 204, 155, 0, 0, 44, 214, 236, 61, 0, 0],
            &[0, 204, 155, 0, 0, 0, 10, 178, 234, 4, 0],
            &[0, 204, 155, 0, 0, 0, 0, 65, 255, 39, 0],
            &[0, 204, 155, 16, 105, 31, 13, 147, 248, 16, 0],
            &[0, 204, 155, 15, 218, 255, 255, 238, 93, 0, 0],
            &[0, 0, 0, 0, 0, 21, 33, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'à' / 0xe0
        'à' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 65, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 187, 220, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 196, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 76, 0, 0, 0, 0, 0],
            &[0, 2, 73, 153, 182, 165, 103, 6, 0, 0, 0],
            &[0, 21, 232, 158, 120, 141, 239, 164, 0, 0, 0],
            &[0, 0, 3, 0, 0, 0, 89, 254, 10, 0, 0],
            &[0, 0, 0, 4, 31, 46, 86, 255, 39, 0, 0],
            &[0, 13, 159, 244, 239, 218, 217, 255, 40, 0, 0],
            &[0, 157, 228, 60, 0, 0, 41, 255, 40, 0, 0],
            &[0, 215, 140, 0, 0, 0, 87, 255, 40, 0, 0],
            &[0, 182, 204, 32, 19, 82, 223, 255, 40, 0, 0],
            &[0, 42, 214, 255, 255, 205, 58, 255, 40, 0, 0],
            &[0, 0, 0, 25, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'á' / 0xe1
        'á' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 65, 27, 0, 0, 0],
            &[0, 0, 0, 0, 0, 177, 223, 21, 0, 0, 0],
            &[0, 0, 0, 0, 86, 227, 36, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 25, 0, 0, 0, 0, 0],
            &[0, 2, 73, 153, 182, 165, 103, 6, 0, 0, 0],
            &[0, 21, 232, 158, 120, 141, 239, 164, 0, 0, 0],
            &[0, 0, 3, 0, 0, 0, 89, 254, 10, 0, 0],
            &[0, 0, 0, 4, 31, 46, 86, 255, 39, 0, 0],
            &[0, 13, 159, 244, 239, 218, 217, 255, 40, 0, 0],
            &[0, 157, 228, 60, 0, 0, 41, 255, 40, 0, 0],
            &[0, 215, 140, 0, 0, 0, 87, 255, 40, 0, 0],
            &[0, 182, 204, 32, 19, 82, 223, 255, 40, 0, 0],
            &[0, 42, 214, 255, 255, 205, 58, 255, 40, 0, 0],
            &[0, 0, 0, 25, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'â' / 0xe2
        'â' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 61, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 103, 241, 244, 43, 0, 0, 0, 0],
            &[0, 0, 69, 224, 45, 99, 220, 28, 0, 0, 0],
            &[0, 0, 62, 24, 0, 0, 46, 43, 0, 0, 0],
            &[0, 2, 73, 153, 182, 165, 103, 6, 0, 0, 0],
            &[0, 21, 232, 158, 120, 141, 239, 164, 0, 0, 0],
            &[0, 0, 3, 0, 0, 0, 89, 254, 10, 0, 0],
            &[0, 0, 0, 4, 31, 46, 86, 255, 39, 0, 0],
            &[0, 13, 159, 244, 239, 218, 217, 255, 40, 0, 0],
            &[0, 157, 228, 60, 0, 0, 41, 255, 40, 0, 0],
            &[0, 215, 140, 0, 0, 0, 87, 255, 40, 0, 0],
            &[0, 182, 204, 32, 19, 82, 223, 255, 40, 0, 0],
            &[0, 42, 214, 255, 255, 205, 58, 255, 40, 0, 0],
            &[0, 0, 0, 25, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ã' / 0xe3
        'ã' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 173, 136, 18, 41, 131, 0, 0, 0],
            &[0, 0, 201, 127, 171, 244, 242, 82, 0, 0, 0],
            &[0, 0, 66, 0, 0, 40, 42, 0, 0, 0, 0],
            &[0, 2, 73, 153, 182, 165, 103, 6, 0, 0, 0],
            &[0, 21, 232, 158, 120, 141, 239, 164, 0, 0, 0],
            &[0, 0, 3, 0, 0, 0, 89, 254, 10, 0, 0],
            &[0, 0, 0, 4, 31, 46, 86, 255, 39, 0, 0],
            &[0, 13, 159, 244, 239, 218, 217, 255, 40, 0, 0],
            &[0, 157, 228, 60, 0, 0, 41, 255, 40, 0, 0],
            &[0, 215, 140, 0, 0, 0, 87, 255, 40, 0, 0],
            &[0, 182, 204, 32, 19, 82, 223, 255, 40, 0, 0],
            &[0, 42, 214, 255, 255, 205, 58, 255, 40, 0, 0],
            &[0, 0, 0, 25, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ä' / 0xe4
        'ä' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 136, 0, 5, 178, 70, 0, 0, 0],
            &[0, 0, 163, 186, 0, 12, 235, 102, 0, 0, 0],
            &[0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0],
            &[0, 2, 73, 153, 182, 165, 103, 6, 0, 0, 0],
            &[0, 21, 232, 158, 120, 141, 239, 164, 0, 0, 0],
            &[0, 0, 3, 0, 0, 0, 89, 254, 10, 0, 0],
            &[0, 0, 0, 4, 31, 46, 86, 255, 39, 0, 0],
            &[0, 13, 159, 244, 239, 218, 217, 255, 40, 0, 0],
            &[0, 157, 228, 60, 0, 0, 41, 255, 40, 0, 0],
            &[0, 215, 140, 0, 0, 0, 87, 255, 40, 0, 0],
            &[0, 182, 204, 32, 19, 82, 223, 255, 40, 0, 0],
            &[0, 42, 214, 255, 255, 205, 58, 255, 40, 0, 0],
            &[0, 0, 0, 25, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'å' / 0xe5
        'å' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 110, 221, 221, 58, 0, 0, 0, 0],
            &[0, 0, 0, 213, 29, 95, 151, 0, 0, 0, 0],
            &[0, 0, 0, 154, 182, 210, 90, 0, 0, 0, 0],
            &[0, 0, 0, 2, 60, 42, 0, 0, 0, 0, 0],
            &[0, 2, 73, 153, 182, 165, 103, 6, 0, 0, 0],
            &[0, 21, 232, 158, 120, 141, 239, 164, 0, 0, 0],
            &[0, 0, 3, 0, 0, 0, 89, 254, 10, 0, 0],
            &[0, 0, 0, 4, 31, 46, 86, 255, 39, 0, 0],
            &[0, 13, 159, 244, 239, 218, 217, 255, 40, 0, 0],
            &[0, 157, 228, 60, 0, 0, 41, 255, 40, 0, 0],
            &[0, 215, 140, 0, 0, 0, 87, 255, 40, 0, 0],
            &[0, 182, 204, 32, 19, 82, 223, 255, 40, 0, 0],
            &[0, 42, 214, 255, 255, 205, 58, 255, 40, 0, 0],
            &[0, 0, 0, 25, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'æ' / 0xe6
        'æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 165, 168, 53, 54, 170, 156, 25, 0, 0],
            &[4, 198, 132, 170, 232, 235, 143, 203, 192, 0, 0],
            &[0, 0, 0, 5, 244, 168, 0, 52, 255, 27, 0],
            &[0, 0, 17, 34, 229, 133, 34, 50, 255, 62, 0],
            &[13, 185, 255, 231, 252, 253, 251, 251, 251, 70, 0],
            &[124, 230, 36, 0, 232, 101, 0, 0, 0, 0, 0],
            &[166, 163, 0, 6, 249, 144, 0, 0, 0, 0, 0],
            &[140, 210, 29, 133, 211, 239, 66, 16, 91, 12, 0],
            &[33, 227, 255, 218, 33, 132, 251, 255, 218, 11, 0],
            &[0, 5, 30, 2, 0, 0, 15, 28, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ç' / 0xe7
        'ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 157, 181, 161, 93, 2, 0, 0],
            &[0, 0, 153, 254, 179, 138, 161, 197, 0, 0, 0],
            &[0, 58, 255, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 244, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 156, 218, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 147, 228, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 103, 255, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 232, 206, 62, 25, 46, 106, 0, 0, 0],
            &[0, 0, 47, 199, 255, 255, 255, 208, 0, 0, 0],
            &[0, 0, 0, 0, 109, 191, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 86, 214, 126, 0, 0, 0, 0],
            &[0, 0, 0, 71, 209, 176, 46, 0, 0, 0, 0],
        ]),
        // letter: 'è' / 0xe8
        'è' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 65, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 190, 217, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 198, 136, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 75, 0, 0, 0, 0],
            &[0, 0, 0, 12, 113, 172, 173, 109, 6, 0, 0],
            &[0, 0, 16, 212, 228, 133, 128, 232, 184, 1, 0],
            &[0, 0, 144, 233, 22, 0, 0, 50, 255, 78, 0],
            &[0, 0, 228, 165, 42, 42, 42, 42, 233, 143, 0],
            &[0, 3, 254, 253, 251, 251, 251, 251, 251, 159, 0],
            &[0, 0, 246, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 190, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 252, 149, 37, 13, 42, 114, 44, 0],
            &[0, 0, 0, 70, 208, 255, 255, 254, 202, 38, 0],
            &[0, 0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'é' / 0xe9
        'é' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 65, 26, 0, 0],
            &[0, 0, 0, 0, 0, 0, 181, 221, 19, 0, 0],
            &[0, 0, 0, 0, 0, 91, 225, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 60, 23, 0, 0, 0, 0],
            &[0, 0, 0, 12, 113, 172, 173, 109, 6, 0, 0],
            &[0, 0, 16, 212, 228, 133, 128, 232, 184, 1, 0],
            &[0, 0, 144, 233, 22, 0, 0, 50, 255, 78, 0],
            &[0, 0, 228, 165, 42, 42, 42, 42, 233, 143, 0],
            &[0, 3, 254, 253, 251, 251, 251, 251, 251, 159, 0],
            &[0, 0, 246, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 190, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 252, 149, 37, 13, 42, 114, 44, 0],
            &[0, 0, 0, 70, 208, 255, 255, 254, 202, 38, 0],
            &[0, 0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ê' / 0xea
        'ê' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 43, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 240, 243, 40, 0, 0, 0],
            &[0, 0, 0, 72, 224, 42, 103, 218, 26, 0, 0],
            &[0, 0, 0, 63, 23, 0, 0, 47, 42, 0, 0],
            &[0, 0, 0, 12, 113, 172, 173, 109, 6, 0, 0],
            &[0, 0, 16, 212, 228, 133, 128, 232, 184, 1, 0],
            &[0, 0, 144, 233, 22, 0, 0, 50, 255, 78, 0],
            &[0, 0, 228, 165, 42, 42, 42, 42, 233, 143, 0],
            &[0, 3, 254, 253, 251, 251, 251, 251, 251, 159, 0],
            &[0, 0, 246, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 190, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 252, 149, 37, 13, 42, 114, 44, 0],
            &[0, 0, 0, 70, 208, 255, 255, 254, 202, 38, 0],
            &[0, 0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ë' / 0xeb
        'ë' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 121, 133, 0, 6, 179, 67, 0, 0],
            &[0, 0, 0, 167, 182, 0, 15, 236, 98, 0, 0],
            &[0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0],
            &[0, 0, 0, 12, 113, 172, 173, 109, 6, 0, 0],
            &[0, 0, 16, 212, 228, 133, 128, 232, 184, 1, 0],
            &[0, 0, 144, 233, 22, 0, 0, 50, 255, 78, 0],
            &[0, 0, 228, 165, 42, 42, 42, 42, 233, 143, 0],
            &[0, 3, 254, 253, 251, 251, 251, 251, 251, 159, 0],
            &[0, 0, 246, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 190, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 252, 149, 37, 13, 42, 114, 44, 0],
            &[0, 0, 0, 70, 208, 255, 255, 254, 202, 38, 0],
            &[0, 0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ì' / 0xec
        'ì' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 65, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 203, 205, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 20, 209, 120, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 70, 0, 0, 0, 0, 0],
            &[0, 68, 147, 147, 147, 114, 0, 0, 0, 0, 0],
            &[0, 51, 124, 143, 222, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 14, 34, 185, 209, 35, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'í' / 0xed
        'í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 65, 22, 0, 0, 0],
            &[0, 0, 0, 0, 1, 196, 211, 12, 0, 0, 0],
            &[0, 0, 0, 0, 107, 216, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 18, 0, 0, 0, 0, 0],
            &[0, 68, 147, 147, 147, 114, 0, 0, 0, 0, 0],
            &[0, 51, 124, 143, 222, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 14, 34, 185, 209, 35, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'î' / 0xee
        'î' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 64, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 124, 239, 237, 31, 0, 0, 0, 0],
            &[0, 0, 86, 219, 33, 117, 211, 19, 0, 0, 0],
            &[0, 0, 68, 18, 0, 0, 52, 37, 0, 0, 0],
            &[0, 68, 147, 147, 147, 114, 0, 0, 0, 0, 0],
            &[0, 51, 124, 143, 222, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 14, 34, 185, 209, 35, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ï' / 0xef
        'ï' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 134, 120, 0, 14, 183, 56, 0, 0, 0],
            &[0, 0, 184, 166, 0, 26, 240, 83, 0, 0, 0],
            &[0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0],
            &[0, 68, 147, 147, 147, 114, 0, 0, 0, 0, 0],
            &[0, 51, 124, 143, 222, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 14, 34, 185, 209, 35, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ð' / 0xf0
        'ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 0, 0, 0, 7, 0, 0],
            &[0, 0, 0, 4, 225, 164, 52, 177, 171, 0, 0],
            &[0, 0, 0, 0, 49, 255, 255, 165, 12, 0, 0],
            &[0, 0, 0, 50, 232, 180, 200, 217, 14, 0, 0],
            &[0, 0, 0, 6, 62, 0, 6, 197, 150, 0, 0],
            &[0, 0, 0, 17, 102, 137, 112, 89, 251, 26, 0],
            &[0, 0, 42, 230, 218, 154, 169, 238, 246, 109, 0],
            &[0, 0, 192, 197, 6, 0, 0, 50, 252, 160, 0],
            &[0, 10, 254, 101, 0, 0, 0, 0, 207, 179, 0],
            &[0, 22, 255, 84, 0, 0, 0, 0, 189, 167, 0],
            &[0, 1, 236, 132, 0, 0, 0, 5, 235, 117, 0],
            &[0, 0, 120, 243, 84, 8, 22, 153, 237, 22, 0],
            &[0, 0, 1, 122, 237, 255, 255, 208, 52, 0, 0],
            &[0, 0, 0, 0, 4, 31, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ñ' / 0xf1
        'ñ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 31, 167, 147, 28, 21, 151, 0, 0, 0],
            &[0, 0, 174, 142, 158, 246, 240, 109, 0, 0, 0],
            &[0, 0, 65, 1, 0, 32, 50, 0, 0, 0, 0],
            &[0, 106, 70, 44, 149, 179, 143, 29, 0, 0, 0],
            &[0, 184, 194, 222, 139, 126, 224, 225, 12, 0, 0],
            &[0, 184, 255, 48, 0, 0, 58, 255, 73, 0, 0],
            &[0, 184, 215, 0, 0, 0, 14, 255, 100, 0, 0],
            &[0, 184, 188, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ò' / 0xf2
        'ò' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 65, 29, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 203, 205, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 209, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 70, 0, 0, 0, 0],
            &[0, 0, 0, 27, 133, 178, 165, 89, 1, 0, 0],
            &[0, 0, 39, 235, 207, 127, 143, 242, 164, 0, 0],
            &[0, 0, 180, 213, 8, 0, 0, 68, 255, 73, 0],
            &[0, 4, 250, 117, 0, 0, 0, 0, 223, 150, 0],
            &[0, 24, 255, 85, 0, 0, 0, 0, 193, 177, 0],
            &[0, 14, 255, 96, 0, 0, 0, 0, 206, 169, 0],
            &[0, 0, 217, 163, 0, 0, 0, 21, 250, 116, 0],
            &[0, 0, 98, 252, 105, 13, 34, 184, 235, 20, 0],
            &[0, 0, 0, 112, 237, 255, 255, 206, 50, 0, 0],
            &[0, 0, 0, 0, 3, 33, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ó' / 0xf3
        'ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 65, 22, 0, 0],
            &[0, 0, 0, 0, 0, 1, 196, 211, 12, 0, 0],
            &[0, 0, 0, 0, 0, 107, 216, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 65, 18, 0, 0, 0, 0],
            &[0, 0, 0, 27, 133, 178, 165, 89, 1, 0, 0],
            &[0, 0, 39, 235, 207, 127, 143, 242, 164, 0, 0],
            &[0, 0, 180, 213, 8, 0, 0, 68, 255, 73, 0],
            &[0, 4, 250, 117, 0, 0, 0, 0, 223, 150, 0],
            &[0, 24, 255, 85, 0, 0, 0, 0, 193, 177, 0],
            &[0, 14, 255, 96, 0, 0, 0, 0, 206, 169, 0],
            &[0, 0, 217, 163, 0, 0, 0, 21, 250, 116, 0],
            &[0, 0, 98, 252, 105, 13, 34, 184, 235, 20, 0],
            &[0, 0, 0, 112, 237, 255, 255, 206, 50, 0, 0],
            &[0, 0, 0, 0, 3, 33, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ô' / 0xf4
        'ô' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 64, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 124, 239, 237, 31, 0, 0, 0],
            &[0, 0, 0, 86, 219, 33, 117, 211, 19, 0, 0],
            &[0, 0, 0, 68, 18, 0, 0, 52, 37, 0, 0],
            &[0, 0, 0, 27, 133, 178, 165, 89, 1, 0, 0],
            &[0, 0, 39, 235, 207, 127, 143, 242, 164, 0, 0],
            &[0, 0, 180, 213, 8, 0, 0, 68, 255, 73, 0],
            &[0, 4, 250, 117, 0, 0, 0, 0, 223, 150, 0],
            &[0, 24, 255, 85, 0, 0, 0, 0, 193, 177, 0],
            &[0, 14, 255, 96, 0, 0, 0, 0, 206, 169, 0],
            &[0, 0, 217, 163, 0, 0, 0, 21, 250, 116, 0],
            &[0, 0, 98, 252, 105, 13, 34, 184, 235, 20, 0],
            &[0, 0, 0, 112, 237, 255, 255, 206, 50, 0, 0],
            &[0, 0, 0, 0, 3, 33, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'õ' / 0xf5
        'õ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 57, 175, 127, 13, 57, 115, 0, 0],
            &[0, 0, 0, 218, 120, 182, 242, 242, 64, 0, 0],
            &[0, 0, 3, 62, 0, 0, 46, 36, 0, 0, 0],
            &[0, 0, 0, 27, 133, 178, 165, 89, 1, 0, 0],
            &[0, 0, 39, 235, 207, 127, 143, 242, 164, 0, 0],
            &[0, 0, 180, 213, 8, 0, 0, 68, 255, 73, 0],
            &[0, 4, 250, 117, 0, 0, 0, 0, 223, 150, 0],
            &[0, 24, 255, 85, 0, 0, 0, 0, 193, 177, 0],
            &[0, 14, 255, 96, 0, 0, 0, 0, 206, 169, 0],
            &[0, 0, 217, 163, 0, 0, 0, 21, 250, 116, 0],
            &[0, 0, 98, 252, 105, 13, 34, 184, 235, 20, 0],
            &[0, 0, 0, 112, 237, 255, 255, 206, 50, 0, 0],
            &[0, 0, 0, 0, 3, 33, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ö' / 0xf6
        'ö' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 134, 120, 0, 14, 183, 56, 0, 0],
            &[0, 0, 0, 184, 166, 0, 26, 240, 83, 0, 0],
            &[0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0],
            &[0, 0, 0, 27, 133, 178, 165, 89, 1, 0, 0],
            &[0, 0, 39, 235, 207, 127, 143, 242, 164, 0, 0],
            &[0, 0, 180, 213, 8, 0, 0, 68, 255, 73, 0],
            &[0, 4, 250, 117, 0, 0, 0, 0, 223, 150, 0],
            &[0, 24, 255, 85, 0, 0, 0, 0, 193, 177, 0],
            &[0, 14, 255, 96, 0, 0, 0, 0, 206, 169, 0],
            &[0, 0, 217, 163, 0, 0, 0, 21, 250, 116, 0],
            &[0, 0, 98, 252, 105, 13, 34, 184, 235, 20, 0],
            &[0, 0, 0, 112, 237, 255, 255, 206, 50, 0, 0],
            &[0, 0, 0, 0, 3, 33, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '÷' / 0xf7
        '÷' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 53, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 251, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 53, 0, 0, 0, 0, 0],
            &[0, 38, 53, 53, 53, 53, 53, 53, 17, 0, 0],
            &[0, 170, 237, 237, 237, 237, 237, 237, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 187, 95, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 231, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ø' / 0xf8
        'ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 133, 178, 165, 94, 195, 65, 0],
            &[0, 0, 39, 235, 207, 127, 143, 255, 255, 8, 0],
            &[0, 0, 180, 213, 8, 0, 64, 255, 255, 73, 0],
            &[0, 4, 250, 117, 0, 17, 227, 107, 223, 150, 0],
            &[0, 24, 255, 85, 0, 171, 179, 0, 193, 177, 0],
            &[0, 14, 255, 96, 98, 231, 21, 0, 206, 169, 0],
            &[0, 0, 217, 200, 242, 71, 0, 21, 250, 116, 0],
            &[0, 0, 103, 255, 249, 13, 34, 184, 235, 20, 0],
            &[0, 0, 132, 255, 245, 255, 255, 206, 50, 0, 0],
            &[0, 0, 38, 39, 3, 33, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ù' / 0xf9
        'ù' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 65, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 203, 205, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 20, 209, 120, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 70, 0, 0, 0, 0, 0],
            &[0, 117, 94, 0, 0, 0, 17, 147, 47, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 41, 255, 81, 0, 0],
            &[0, 191, 183, 0, 0, 0, 91, 255, 81, 0, 0],
            &[0, 139, 248, 70, 10, 51, 214, 255, 81, 0, 0],
            &[0, 19, 191, 255, 255, 230, 74, 235, 81, 0, 0],
            &[0, 0, 0, 16, 30, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ú' / 0xfa
        'ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 65, 22, 0, 0, 0],
            &[0, 0, 0, 0, 1, 196, 211, 12, 0, 0, 0],
            &[0, 0, 0, 0, 107, 216, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 18, 0, 0, 0, 0, 0],
            &[0, 117, 94, 0, 0, 0, 17, 147, 47, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 41, 255, 81, 0, 0],
            &[0, 191, 183, 0, 0, 0, 91, 255, 81, 0, 0],
            &[0, 139, 248, 70, 10, 51, 214, 255, 81, 0, 0],
            &[0, 19, 191, 255, 255, 230, 74, 235, 81, 0, 0],
            &[0, 0, 0, 16, 30, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'û' / 0xfb
        'û' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 64, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 124, 239, 237, 31, 0, 0, 0, 0],
            &[0, 0, 86, 219, 33, 117, 211, 19, 0, 0, 0],
            &[0, 0, 68, 18, 0, 0, 52, 37, 0, 0, 0],
            &[0, 117, 94, 0, 0, 0, 17, 147, 47, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 41, 255, 81, 0, 0],
            &[0, 191, 183, 0, 0, 0, 91, 255, 81, 0, 0],
            &[0, 139, 248, 70, 10, 51, 214, 255, 81, 0, 0],
            &[0, 19, 191, 255, 255, 230, 74, 235, 81, 0, 0],
            &[0, 0, 0, 16, 30, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ü' / 0xfc
        'ü' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 134, 120, 0, 14, 183, 56, 0, 0, 0],
            &[0, 0, 184, 166, 0, 26, 240, 83, 0, 0, 0],
            &[0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0],
            &[0, 117, 94, 0, 0, 0, 17, 147, 47, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 41, 255, 81, 0, 0],
            &[0, 191, 183, 0, 0, 0, 91, 255, 81, 0, 0],
            &[0, 139, 248, 70, 10, 51, 214, 255, 81, 0, 0],
            &[0, 19, 191, 255, 255, 230, 74, 235, 81, 0, 0],
            &[0, 0, 0, 16, 30, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ý' / 0xfd
        'ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 65, 22, 0, 0],
            &[0, 0, 0, 0, 0, 1, 196, 211, 12, 0, 0],
            &[0, 0, 0, 0, 0, 107, 216, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 65, 18, 0, 0, 0, 0],
            &[0, 23, 147, 36, 0, 0, 0, 0, 103, 111, 0],
            &[0, 0, 212, 144, 0, 0, 0, 6, 240, 118, 0],
            &[0, 0, 106, 239, 9, 0, 0, 77, 252, 26, 0],
            &[0, 0, 13, 242, 96, 0, 0, 162, 184, 0, 0],
            &[0, 0, 0, 149, 200, 0, 7, 241, 89, 0, 0],
            &[0, 0, 0, 43, 255, 47, 77, 241, 9, 0, 0],
            &[0, 0, 0, 0, 192, 141, 160, 155, 0, 0, 0],
            &[0, 0, 0, 0, 86, 224, 236, 60, 0, 0, 0],
            &[0, 0, 0, 0, 5, 230, 220, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 216, 126, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 250, 28, 0, 0, 0, 0],
            &[0, 0, 215, 226, 157, 12, 0, 0, 0, 0, 0],
        ]),
        // letter: 'þ' / 0xfe
        'þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 29, 28, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 180, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 180, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 180, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 178, 55, 159, 180, 128, 15, 0, 0, 0],
            &[0, 184, 217, 220, 132, 137, 234, 206, 6, 0, 0],
            &[0, 184, 252, 34, 0, 0, 50, 255, 94, 0, 0],
            &[0, 184, 204, 0, 0, 0, 0, 222, 158, 0, 0],
            &[0, 184, 183, 0, 0, 0, 0, 199, 179, 0, 0],
            &[0, 184, 198, 0, 0, 0, 0, 212, 167, 0, 0],
            &[0, 184, 242, 8, 0, 0, 17, 247, 123, 0, 0],
            &[0, 184, 252, 154, 26, 29, 168, 247, 31, 0, 0],
            &[0, 184, 175, 153, 253, 255, 229, 76, 0, 0, 0],
            &[0, 184, 180, 0, 11, 30, 3, 0, 0, 0, 0],
            &[0, 184, 180, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 154, 151, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ÿ' / 0xff
        'ÿ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 134, 120, 0, 14, 183, 56, 0, 0],
            &[0, 0, 0, 184, 166, 0, 26, 240, 83, 0, 0],
            &[0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0],
            &[0, 23, 147, 36, 0, 0, 0, 0, 103, 111, 0],
            &[0, 0, 212, 144, 0, 0, 0, 6, 240, 118, 0],
            &[0, 0, 106, 239, 9, 0, 0, 77, 252, 26, 0],
            &[0, 0, 13, 242, 96, 0, 0, 162, 184, 0, 0],
            &[0, 0, 0, 149, 200, 0, 7, 241, 89, 0, 0],
            &[0, 0, 0, 43, 255, 47, 77, 241, 9, 0, 0],
            &[0, 0, 0, 0, 192, 141, 160, 155, 0, 0, 0],
            &[0, 0, 0, 0, 86, 224, 236, 60, 0, 0, 0],
            &[0, 0, 0, 0, 5, 230, 220, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 216, 126, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 250, 28, 0, 0, 0, 0],
            &[0, 0, 215, 226, 157, 12, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ā' / 0x100
        'Ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 223, 223, 223, 223, 37, 0, 0, 0],
            &[0, 0, 39, 67, 67, 67, 67, 11, 0, 0, 0],
            &[0, 0, 0, 5, 120, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 255, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 155, 199, 255, 49, 0, 0, 0, 0],
            &[0, 0, 3, 234, 105, 210, 132, 0, 0, 0, 0],
            &[0, 0, 64, 254, 27, 133, 215, 0, 0, 0, 0],
            &[0, 0, 147, 203, 0, 53, 255, 42, 0, 0, 0],
            &[0, 1, 228, 123, 0, 1, 229, 126, 0, 0, 0],
            &[0, 57, 255, 245, 200, 200, 255, 209, 0, 0, 0],
            &[0, 140, 255, 127, 126, 126, 187, 255, 36, 0, 0],
            &[0, 222, 141, 0, 0, 0, 6, 241, 119, 0, 0],
            &[49, 255, 61, 0, 0, 0, 0, 168, 202, 0, 0],
            &[132, 235, 3, 0, 0, 0, 0, 88, 255, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ā' / 0x101
        'ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 6, 6, 6, 6, 1, 0, 0, 0],
            &[0, 0, 128, 255, 255, 255, 255, 63, 0, 0, 0],
            &[0, 0, 14, 28, 28, 28, 28, 7, 0, 0, 0],
            &[0, 2, 73, 153, 182, 165, 103, 6, 0, 0, 0],
            &[0, 21, 232, 158, 120, 141, 239, 164, 0, 0, 0],
            &[0, 0, 3, 0, 0, 0, 89, 254, 10, 0, 0],
            &[0, 0, 0, 4, 31, 46, 86, 255, 39, 0, 0],
            &[0, 13, 159, 244, 239, 218, 217, 255, 40, 0, 0],
            &[0, 157, 228, 60, 0, 0, 41, 255, 40, 0, 0],
            &[0, 215, 140, 0, 0, 0, 87, 255, 40, 0, 0],
            &[0, 182, 204, 32, 19, 82, 223, 255, 40, 0, 0],
            &[0, 42, 214, 255, 255, 205, 58, 255, 40, 0, 0],
            &[0, 0, 0, 25, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ă' / 0x102
        'Ă' => Some(&[
            &[0, 0, 109, 71, 0, 0, 141, 36, 0, 0, 0],
            &[0, 0, 69, 236, 155, 188, 207, 6, 0, 0, 0],
            &[0, 0, 0, 42, 104, 86, 8, 0, 0, 0, 0],
            &[0, 0, 0, 5, 120, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 255, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 155, 199, 255, 49, 0, 0, 0, 0],
            &[0, 0, 3, 234, 105, 210, 132, 0, 0, 0, 0],
            &[0, 0, 64, 254, 27, 133, 215, 0, 0, 0, 0],
            &[0, 0, 147, 203, 0, 53, 255, 42, 0, 0, 0],
            &[0, 1, 228, 123, 0, 1, 229, 126, 0, 0, 0],
            &[0, 57, 255, 245, 200, 200, 255, 209, 0, 0, 0],
            &[0, 140, 255, 127, 126, 126, 187, 255, 36, 0, 0],
            &[0, 222, 141, 0, 0, 0, 6, 241, 119, 0, 0],
            &[49, 255, 61, 0, 0, 0, 0, 168, 202, 0, 0],
            &[132, 235, 3, 0, 0, 0, 0, 88, 255, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ă' / 0x103
        'ă' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 114, 0, 1, 165, 58, 0, 0, 0],
            &[0, 0, 35, 231, 197, 218, 192, 6, 0, 0, 0],
            &[0, 0, 0, 13, 64, 52, 1, 0, 0, 0, 0],
            &[0, 2, 73, 153, 182, 165, 103, 6, 0, 0, 0],
            &[0, 21, 232, 158, 120, 141, 239, 164, 0, 0, 0],
            &[0, 0, 3, 0, 0, 0, 89, 254, 10, 0, 0],
            &[0, 0, 0, 4, 31, 46, 86, 255, 39, 0, 0],
            &[0, 13, 159, 244, 239, 218, 217, 255, 40, 0, 0],
            &[0, 157, 228, 60, 0, 0, 41, 255, 40, 0, 0],
            &[0, 215, 140, 0, 0, 0, 87, 255, 40, 0, 0],
            &[0, 182, 204, 32, 19, 82, 223, 255, 40, 0, 0],
            &[0, 42, 214, 255, 255, 205, 58, 255, 40, 0, 0],
            &[0, 0, 0, 25, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ą' / 0x104
        'Ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 120, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 255, 221, 0, 0, 0, 0, 0],
            &[0, 0, 0, 155, 199, 255, 49, 0, 0, 0, 0],
            &[0, 0, 3, 234, 105, 210, 132, 0, 0, 0, 0],
            &[0, 0, 64, 254, 27, 133, 215, 0, 0, 0, 0],
            &[0, 0, 147, 203, 0, 53, 255, 42, 0, 0, 0],
            &[0, 1, 228, 123, 0, 1, 229, 126, 0, 0, 0],
            &[0, 57, 255, 245, 200, 200, 255, 209, 0, 0, 0],
            &[0, 140, 255, 127, 126, 126, 187, 255, 36, 0, 0],
            &[0, 222, 141, 0, 0, 0, 6, 241, 119, 0, 0],
            &[49, 255, 61, 0, 0, 0, 0, 168, 202, 0, 0],
            &[132, 235, 3, 0, 0, 0, 0, 88, 255, 34, 0],
            &[0, 0, 0, 0, 0, 0, 0, 68, 198, 11, 0],
            &[0, 0, 0, 0, 0, 0, 0, 208, 78, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 41, 123, 77, 0],
        ]),
        // letter: 'ą' / 0x105
        'ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 73, 153, 182, 165, 103, 6, 0, 0, 0],
            &[0, 21, 232, 158, 120, 141, 239, 164, 0, 0, 0],
            &[0, 0, 3, 0, 0, 0, 89, 254, 10, 0, 0],
            &[0, 0, 0, 4, 31, 46, 86, 255, 39, 0, 0],
            &[0, 13, 159, 244, 239, 218, 217, 255, 40, 0, 0],
            &[0, 157, 228, 60, 0, 0, 41, 255, 40, 0, 0],
            &[0, 215, 140, 0, 0, 0, 87, 255, 40, 0, 0],
            &[0, 182, 204, 32, 19, 82, 223, 255, 40, 0, 0],
            &[0, 42, 214, 255, 255, 205, 58, 255, 41, 0, 0],
            &[0, 0, 0, 25, 21, 0, 117, 160, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 250, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 65, 126, 50, 0, 0],
        ]),
        // letter: 'Ć' / 0x106
        'Ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 188, 223, 19, 0, 0],
            &[0, 0, 0, 0, 0, 95, 230, 37, 0, 0, 0],
            &[0, 0, 0, 0, 0, 99, 31, 0, 0, 0, 0],
            &[0, 0, 0, 14, 97, 139, 132, 93, 15, 0, 0],
            &[0, 0, 57, 232, 243, 189, 196, 241, 152, 0, 0],
            &[0, 20, 235, 188, 17, 0, 0, 9, 17, 0, 0],
            &[0, 123, 249, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 194, 186, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 229, 146, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 235, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 217, 158, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 212, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 87, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 193, 242, 116, 55, 59, 108, 48, 0, 0],
            &[0, 0, 12, 150, 246, 255, 255, 235, 57, 0, 0],
            &[0, 0, 0, 0, 3, 29, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 26, 8, 0, 0],
        ]),
        // letter: 'ć' / 0x107
        'ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 62, 51, 0, 0, 0],
            &[0, 0, 0, 0, 0, 82, 254, 84, 0, 0, 0],
            &[0, 0, 0, 0, 18, 227, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 30, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 157, 181, 161, 93, 2, 0, 0],
            &[0, 0, 153, 254, 179, 138, 161, 197, 0, 0, 0],
            &[0, 58, 255, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 244, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 156, 218, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 147, 228, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 103, 255, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 232, 206, 62, 25, 46, 106, 0, 0, 0],
            &[0, 0, 47, 199, 255, 255, 255, 208, 0, 0, 0],
            &[0, 0, 0, 0, 16, 34, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĉ' / 0x108
        'Ĉ' => Some(&[
            &[0, 0, 0, 0, 109, 253, 213, 10, 0, 0, 0],
            &[0, 0, 0, 69, 229, 52, 168, 178, 4, 0, 0],
            &[0, 0, 0, 102, 32, 0, 1, 95, 42, 0, 0],
            &[0, 0, 0, 14, 97, 139, 132, 93, 15, 0, 0],
            &[0, 0, 57, 232, 243, 189, 196, 241, 152, 0, 0],
            &[0, 20, 235, 188, 17, 0, 0, 9, 17, 0, 0],
            &[0, 123, 249, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 194, 186, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 229, 146, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 235, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 217, 158, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 212, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 87, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 193, 242, 116, 55, 59, 108, 48, 0, 0],
            &[0, 0, 12, 150, 246, 255, 255, 235, 57, 0, 0],
            &[0, 0, 0, 0, 3, 29, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 12, 0, 0, 0, 0],
        ]),
        // letter: 'ĉ' / 0x109
        'ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 65, 3, 0, 0, 0, 0],
            &[0, 0, 0, 29, 235, 240, 127, 0, 0, 0, 0],
            &[0, 0, 15, 208, 115, 35, 220, 92, 0, 0, 0],
            &[0, 0, 33, 53, 0, 0, 17, 72, 0, 0, 0],
            &[0, 0, 0, 78, 157, 181, 161, 93, 2, 0, 0],
            &[0, 0, 153, 254, 179, 138, 161, 197, 0, 0, 0],
            &[0, 58, 255, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 244, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 156, 218, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 147, 228, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 103, 255, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 232, 206, 62, 25, 46, 106, 0, 0, 0],
            &[0, 0, 47, 199, 255, 255, 255, 208, 0, 0, 0],
            &[0, 0, 0, 0, 16, 34, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ċ' / 0x10a
        'Ċ' => Some(&[
            &[0, 0, 0, 0, 25, 230, 122, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 233, 125, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 97, 139, 132, 93, 15, 0, 0],
            &[0, 0, 57, 232, 243, 189, 196, 241, 152, 0, 0],
            &[0, 20, 235, 188, 17, 0, 0, 9, 17, 0, 0],
            &[0, 123, 249, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 194, 186, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 229, 146, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 235, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 217, 158, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 212, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 87, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 193, 242, 116, 55, 59, 108, 48, 0, 0],
            &[0, 0, 12, 150, 246, 255, 255, 235, 57, 0, 0],
            &[0, 0, 0, 0, 3, 29, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ċ' / 0x10b
        'ċ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 250, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 105, 192, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 157, 181, 161, 93, 2, 0, 0],
            &[0, 0, 153, 254, 179, 138, 161, 197, 0, 0, 0],
            &[0, 58, 255, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 244, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 156, 218, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 147, 228, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 103, 255, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 232, 206, 62, 25, 46, 106, 0, 0, 0],
            &[0, 0, 47, 199, 255, 255, 255, 208, 0, 0, 0],
            &[0, 0, 0, 0, 16, 34, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Č' / 0x10c
        'Č' => Some(&[
            &[0, 0, 0, 150, 183, 13, 90, 223, 38, 0, 0],
            &[0, 0, 0, 2, 177, 220, 244, 50, 0, 0, 0],
            &[0, 0, 0, 0, 17, 116, 70, 0, 0, 0, 0],
            &[0, 0, 0, 14, 97, 139, 132, 93, 15, 0, 0],
            &[0, 0, 57, 232, 243, 189, 196, 241, 152, 0, 0],
            &[0, 20, 235, 188, 17, 0, 0, 9, 17, 0, 0],
            &[0, 123, 249, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 194, 186, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 229, 146, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 235, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 217, 158, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 212, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 87, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 193, 242, 116, 55, 59, 108, 48, 0, 0],
            &[0, 0, 12, 150, 246, 255, 255, 235, 57, 0, 0],
            &[0, 0, 0, 0, 3, 29, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 1, 0, 0, 16, 11, 0, 0],
        ]),
        // letter: 'č' / 0x10d
        'č' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 29, 44, 0, 0, 13, 62, 0, 0, 0],
            &[0, 0, 23, 216, 117, 36, 217, 109, 0, 0, 0],
            &[0, 0, 0, 36, 239, 243, 137, 0, 0, 0, 0],
            &[0, 0, 0, 0, 46, 77, 4, 0, 0, 0, 0],
            &[0, 0, 0, 78, 157, 181, 161, 93, 2, 0, 0],
            &[0, 0, 153, 254, 179, 138, 161, 197, 0, 0, 0],
            &[0, 58, 255, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 244, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 156, 218, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 147, 228, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 103, 255, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 232, 206, 62, 25, 46, 106, 0, 0, 0],
            &[0, 0, 47, 199, 255, 255, 255, 208, 0, 0, 0],
            &[0, 0, 0, 0, 16, 34, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ď' / 0x10e
        'Ď' => Some(&[
            &[0, 17, 209, 119, 4, 155, 185, 6, 0, 0, 0],
            &[0, 0, 29, 231, 219, 205, 11, 0, 0, 0, 0],
            &[0, 0, 0, 54, 116, 34, 0, 0, 0, 0, 0],
            &[0, 95, 108, 108, 90, 44, 0, 0, 0, 0, 0],
            &[0, 225, 236, 211, 233, 255, 192, 26, 0, 0, 0],
            &[0, 225, 143, 0, 0, 60, 223, 212, 7, 0, 0],
            &[0, 225, 143, 0, 0, 0, 50, 254, 101, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 206, 178, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 166, 209, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 156, 217, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 180, 194, 0, 0],
            &[0, 225, 143, 0, 0, 0, 7, 238, 137, 0, 0],
            &[0, 225, 143, 0, 0, 0, 135, 250, 42, 0, 0],
            &[0, 225, 171, 66, 93, 179, 254, 111, 0, 0, 0],
            &[0, 225, 255, 253, 230, 177, 66, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 20, 0, 0, 0, 24, 3, 0, 0, 0],
        ]),
        // letter: 'ď' / 0x10f
        'ď' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 30, 27, 5, 40, 11],
            &[0, 0, 0, 0, 0, 0, 190, 174, 58, 252, 25],
            &[0, 0, 0, 0, 0, 0, 190, 174, 114, 162, 0],
            &[0, 0, 0, 0, 0, 0, 190, 174, 64, 24, 0],
            &[0, 0, 92, 172, 168, 73, 186, 174, 0, 0, 0],
            &[0, 117, 255, 160, 124, 221, 224, 174, 0, 0, 0],
            &[4, 240, 162, 0, 0, 45, 255, 174, 0, 0, 0],
            &[47, 255, 79, 0, 0, 0, 226, 174, 0, 0, 0],
            &[66, 255, 54, 0, 0, 0, 198, 174, 0, 0, 0],
            &[59, 255, 64, 0, 0, 0, 202, 174, 0, 0, 0],
            &[20, 254, 109, 0, 0, 9, 241, 174, 0, 0, 0],
            &[0, 183, 225, 49, 21, 151, 250, 174, 0, 0, 0],
            &[0, 28, 203, 255, 255, 166, 150, 174, 0, 0, 0],
            &[0, 0, 0, 28, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Đ' / 0x110
        'Đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 78, 108, 108, 95, 56, 0, 0, 0, 0, 0],
            &[0, 184, 243, 211, 228, 255, 213, 46, 0, 0, 0],
            &[0, 184, 184, 0, 0, 40, 201, 236, 24, 0, 0],
            &[0, 184, 184, 0, 0, 0, 23, 241, 142, 0, 0],
            &[0, 184, 184, 0, 0, 0, 0, 165, 219, 0, 0],
            &[69, 253, 253, 69, 46, 0, 0, 125, 249, 0, 0],
            &[249, 255, 255, 249, 167, 0, 0, 115, 254, 3, 0],
            &[0, 184, 184, 0, 0, 0, 0, 139, 235, 0, 0],
            &[0, 184, 184, 0, 0, 0, 0, 205, 178, 0, 0],
            &[0, 184, 184, 0, 0, 0, 94, 255, 78, 0, 0],
            &[0, 184, 202, 65, 86, 160, 252, 150, 0, 0, 0],
            &[0, 184, 255, 254, 235, 190, 88, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'đ' / 0x111
        'đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 40, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 255, 81, 0, 0],
            &[0, 0, 0, 0, 172, 184, 210, 255, 255, 110, 0],
            &[0, 0, 0, 0, 80, 85, 112, 255, 167, 51, 0],
            &[0, 0, 34, 130, 159, 121, 39, 255, 81, 0, 0],
            &[0, 53, 243, 220, 145, 178, 209, 255, 81, 0, 0],
            &[0, 191, 224, 14, 0, 0, 155, 255, 81, 0, 0],
            &[6, 251, 131, 0, 0, 0, 60, 255, 81, 0, 0],
            &[25, 255, 99, 0, 0, 0, 30, 255, 81, 0, 0],
            &[17, 255, 107, 0, 0, 0, 39, 255, 81, 0, 0],
            &[0, 229, 159, 0, 0, 0, 90, 255, 81, 0, 0],
            &[0, 129, 248, 92, 15, 55, 216, 255, 81, 0, 0],
            &[0, 6, 155, 251, 255, 237, 82, 238, 81, 0, 0],
            &[0, 0, 0, 13, 32, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ē' / 0x112
        'Ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 223, 223, 223, 223, 58, 0, 0, 0],
            &[0, 0, 32, 67, 67, 67, 67, 17, 0, 0, 0],
            &[0, 52, 108, 108, 108, 108, 108, 108, 26, 0, 0],
            &[0, 122, 254, 219, 219, 219, 219, 219, 52, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 251, 155, 155, 155, 155, 145, 0, 0, 0],
            &[0, 122, 252, 171, 171, 171, 171, 160, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 248, 71, 71, 71, 71, 71, 17, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ē' / 0x113
        'ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 6, 6, 6, 6, 1, 0, 0],
            &[0, 0, 0, 133, 255, 255, 255, 255, 59, 0, 0],
            &[0, 0, 0, 14, 28, 28, 28, 28, 6, 0, 0],
            &[0, 0, 0, 12, 113, 172, 173, 109, 6, 0, 0],
            &[0, 0, 16, 212, 228, 133, 128, 232, 184, 1, 0],
            &[0, 0, 144, 233, 22, 0, 0, 50, 255, 78, 0],
            &[0, 0, 228, 165, 42, 42, 42, 42, 233, 143, 0],
            &[0, 3, 254, 253, 251, 251, 251, 251, 251, 159, 0],
            &[0, 0, 246, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 190, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 252, 149, 37, 13, 42, 114, 44, 0],
            &[0, 0, 0, 70, 208, 255, 255, 254, 202, 38, 0],
            &[0, 0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĕ' / 0x114
        'Ĕ' => Some(&[
            &[0, 0, 92, 88, 0, 0, 124, 53, 0, 0, 0],
            &[0, 0, 49, 237, 159, 179, 221, 15, 0, 0, 0],
            &[0, 0, 0, 34, 102, 91, 13, 0, 0, 0, 0],
            &[0, 52, 108, 108, 108, 108, 108, 108, 26, 0, 0],
            &[0, 122, 254, 219, 219, 219, 219, 219, 52, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 251, 155, 155, 155, 155, 145, 0, 0, 0],
            &[0, 122, 252, 171, 171, 171, 171, 160, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 248, 71, 71, 71, 71, 71, 17, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĕ' / 0x115
        'ĕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 116, 111, 0, 1, 168, 54, 0, 0],
            &[0, 0, 0, 37, 232, 196, 220, 189, 5, 0, 0],
            &[0, 0, 0, 0, 13, 64, 51, 1, 0, 0, 0],
            &[0, 0, 0, 12, 113, 172, 173, 109, 6, 0, 0],
            &[0, 0, 16, 212, 228, 133, 128, 232, 184, 1, 0],
            &[0, 0, 144, 233, 22, 0, 0, 50, 255, 78, 0],
            &[0, 0, 228, 165, 42, 42, 42, 42, 233, 143, 0],
            &[0, 3, 254, 253, 251, 251, 251, 251, 251, 159, 0],
            &[0, 0, 246, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 190, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 252, 149, 37, 13, 42, 114, 44, 0],
            &[0, 0, 0, 70, 208, 255, 255, 254, 202, 38, 0],
            &[0, 0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ė' / 0x116
        'Ė' => Some(&[
            &[0, 0, 0, 4, 214, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 217, 162, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 108, 108, 108, 108, 108, 108, 26, 0, 0],
            &[0, 122, 254, 219, 219, 219, 219, 219, 52, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 251, 155, 155, 155, 155, 145, 0, 0, 0],
            &[0, 122, 252, 171, 171, 171, 171, 160, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 248, 71, 71, 71, 71, 71, 17, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ė' / 0x117
        'ė' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 243, 179, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 183, 121, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 113, 172, 173, 109, 6, 0, 0],
            &[0, 0, 16, 212, 228, 133, 128, 232, 184, 1, 0],
            &[0, 0, 144, 233, 22, 0, 0, 50, 255, 78, 0],
            &[0, 0, 228, 165, 42, 42, 42, 42, 233, 143, 0],
            &[0, 3, 254, 253, 251, 251, 251, 251, 251, 159, 0],
            &[0, 0, 246, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 190, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 252, 149, 37, 13, 42, 114, 44, 0],
            &[0, 0, 0, 70, 208, 255, 255, 254, 202, 38, 0],
            &[0, 0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ę' / 0x118
        'Ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 108, 108, 108, 108, 108, 108, 26, 0, 0],
            &[0, 122, 254, 219, 219, 219, 219, 219, 52, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 251, 155, 155, 155, 155, 145, 0, 0, 0],
            &[0, 122, 252, 171, 171, 171, 171, 160, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 248, 71, 71, 71, 71, 71, 17, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 0, 50, 207, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 183, 102, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 121, 90, 0, 0, 0],
        ]),
        // letter: 'ę' / 0x119
        'ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 113, 172, 173, 109, 6, 0, 0],
            &[0, 0, 16, 212, 228, 133, 128, 232, 184, 1, 0],
            &[0, 0, 144, 233, 22, 0, 0, 50, 255, 78, 0],
            &[0, 0, 228, 165, 42, 42, 42, 42, 233, 143, 0],
            &[0, 3, 254, 253, 251, 251, 251, 251, 251, 159, 0],
            &[0, 0, 246, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 190, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 252, 149, 37, 13, 42, 114, 44, 0],
            &[0, 0, 0, 70, 208, 255, 255, 255, 255, 45, 0],
            &[0, 0, 0, 0, 0, 17, 46, 211, 62, 0, 0],
            &[0, 0, 0, 0, 0, 0, 102, 182, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 98, 123, 11, 0],
        ]),
        // letter: 'Ě' / 0x11a
        'Ě' => Some(&[
            &[0, 0, 111, 206, 28, 63, 225, 63, 0, 0, 0],
            &[0, 0, 0, 141, 227, 246, 80, 0, 0, 0, 0],
            &[0, 0, 0, 4, 112, 88, 0, 0, 0, 0, 0],
            &[0, 52, 108, 108, 108, 108, 108, 108, 26, 0, 0],
            &[0, 122, 254, 219, 219, 219, 219, 219, 52, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 251, 155, 155, 155, 155, 145, 0, 0, 0],
            &[0, 122, 252, 171, 171, 171, 171, 160, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 248, 71, 71, 71, 71, 71, 17, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 5, 0, 0, 12, 15, 0, 0, 0],
        ]),
        // letter: 'ě' / 0x11b
        'ě' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 18, 0, 0, 38, 36, 0, 0],
            &[0, 0, 0, 86, 225, 45, 101, 225, 36, 0, 0],
            &[0, 0, 0, 0, 118, 243, 246, 48, 0, 0, 0],
            &[0, 0, 0, 0, 1, 74, 52, 0, 0, 0, 0],
            &[0, 0, 0, 12, 113, 172, 173, 109, 6, 0, 0],
            &[0, 0, 16, 212, 228, 133, 128, 232, 184, 1, 0],
            &[0, 0, 144, 233, 22, 0, 0, 50, 255, 78, 0],
            &[0, 0, 228, 165, 42, 42, 42, 42, 233, 143, 0],
            &[0, 3, 254, 253, 251, 251, 251, 251, 251, 159, 0],
            &[0, 0, 246, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 190, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 252, 149, 37, 13, 42, 114, 44, 0],
            &[0, 0, 0, 70, 208, 255, 255, 254, 202, 38, 0],
            &[0, 0, 0, 0, 0, 17, 34, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => Some(&[
            &[0, 0, 0, 0, 25, 234, 252, 73, 0, 0, 0],
            &[0, 0, 0, 11, 203, 131, 76, 232, 46, 0, 0],
            &[0, 0, 0, 55, 80, 0, 0, 49, 90, 0, 0],
            &[0, 0, 0, 0, 33, 113, 141, 122, 63, 0, 0],
            &[0, 0, 0, 121, 252, 232, 191, 210, 253, 42, 0],
            &[0, 0, 81, 255, 132, 5, 0, 0, 28, 0, 0],
            &[0, 0, 204, 195, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 254, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 255, 65, 0, 0, 8, 8, 8, 3, 0],
            &[0, 66, 255, 53, 0, 0, 253, 255, 255, 122, 0],
            &[0, 51, 255, 70, 0, 0, 62, 63, 238, 122, 0],
            &[0, 16, 252, 120, 0, 0, 0, 0, 233, 122, 0],
            &[0, 0, 192, 221, 10, 0, 0, 0, 233, 122, 0],
            &[0, 0, 55, 249, 197, 72, 37, 60, 241, 122, 0],
            &[0, 0, 0, 62, 201, 255, 255, 255, 221, 78, 0],
            &[0, 0, 0, 0, 0, 13, 33, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 23, 0, 0, 0, 0],
        ]),
        // letter: 'ĝ' / 0x11d
        'ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 64, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 124, 239, 237, 31, 0, 0, 0],
            &[0, 0, 0, 86, 219, 33, 117, 211, 19, 0, 0],
            &[0, 0, 0, 68, 18, 0, 0, 52, 37, 0, 0],
            &[0, 0, 0, 47, 155, 182, 142, 22, 130, 47, 0],
            &[0, 0, 52, 246, 201, 124, 153, 213, 253, 81, 0],
            &[0, 0, 188, 212, 6, 0, 0, 115, 255, 81, 0],
            &[0, 6, 250, 125, 0, 0, 0, 25, 255, 81, 0],
            &[0, 25, 255, 97, 0, 0, 0, 1, 253, 81, 0],
            &[0, 18, 255, 110, 0, 0, 0, 8, 254, 81, 0],
            &[0, 0, 228, 169, 0, 0, 0, 56, 255, 81, 0],
            &[0, 0, 123, 251, 103, 17, 43, 203, 255, 81, 0],
            &[0, 0, 3, 147, 250, 255, 249, 124, 254, 81, 0],
            &[0, 0, 0, 0, 12, 35, 11, 16, 255, 74, 0],
            &[0, 0, 0, 0, 0, 0, 0, 76, 255, 37, 0],
            &[0, 0, 71, 170, 211, 229, 208, 132, 11, 0, 0],
        ]),
        // letter: 'Ğ' / 0x11e
        'Ğ' => Some(&[
            &[0, 0, 0, 47, 133, 0, 0, 79, 98, 0, 0],
            &[0, 0, 0, 12, 219, 178, 163, 235, 55, 0, 0],
            &[0, 0, 0, 0, 14, 93, 101, 33, 0, 0, 0],
            &[0, 0, 0, 0, 33, 113, 141, 122, 63, 0, 0],
            &[0, 0, 0, 121, 252, 232, 191, 210, 253, 42, 0],
            &[0, 0, 81, 255, 132, 5, 0, 0, 28, 0, 0],
            &[0, 0, 204, 195, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 254, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 255, 65, 0, 0, 8, 8, 8, 3, 0],
            &[0, 66, 255, 53, 0, 0, 253, 255, 255, 122, 0],
            &[0, 51, 255, 70, 0, 0, 62, 63, 238, 122, 0],
            &[0, 16, 252, 120, 0, 0, 0, 0, 233, 122, 0],
            &[0, 0, 192, 221, 10, 0, 0, 0, 233, 122, 0],
            &[0, 0, 55, 249, 197, 72, 37, 60, 241, 122, 0],
            &[0, 0, 0, 62, 201, 255, 255, 255, 221, 78, 0],
            &[0, 0, 0, 0, 0, 13, 33, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ğ' / 0x11f
        'ğ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 130, 97, 0, 3, 179, 41, 0, 0],
            &[0, 0, 0, 49, 235, 194, 224, 176, 2, 0, 0],
            &[0, 0, 0, 0, 17, 66, 47, 0, 0, 0, 0],
            &[0, 0, 0, 47, 155, 182, 142, 22, 130, 47, 0],
            &[0, 0, 52, 246, 201, 124, 153, 213, 253, 81, 0],
            &[0, 0, 188, 212, 6, 0, 0, 115, 255, 81, 0],
            &[0, 6, 250, 125, 0, 0, 0, 25, 255, 81, 0],
            &[0, 25, 255, 97, 0, 0, 0, 1, 253, 81, 0],
            &[0, 18, 255, 110, 0, 0, 0, 8, 254, 81, 0],
            &[0, 0, 228, 169, 0, 0, 0, 56, 255, 81, 0],
            &[0, 0, 123, 251, 103, 17, 43, 203, 255, 81, 0],
            &[0, 0, 3, 147, 250, 255, 249, 124, 254, 81, 0],
            &[0, 0, 0, 0, 12, 35, 11, 16, 255, 74, 0],
            &[0, 0, 0, 0, 0, 0, 0, 76, 255, 37, 0],
            &[0, 0, 71, 170, 211, 229, 208, 132, 11, 0, 0],
        ]),
        // letter: 'Ġ' / 0x120
        'Ġ' => Some(&[
            &[0, 0, 0, 0, 0, 154, 217, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 158, 219, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 113, 141, 122, 63, 0, 0],
            &[0, 0, 0, 121, 252, 232, 191, 210, 253, 42, 0],
            &[0, 0, 81, 255, 132, 5, 0, 0, 28, 0, 0],
            &[0, 0, 204, 195, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 254, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 255, 65, 0, 0, 8, 8, 8, 3, 0],
            &[0, 66, 255, 53, 0, 0, 253, 255, 255, 122, 0],
            &[0, 51, 255, 70, 0, 0, 62, 63, 238, 122, 0],
            &[0, 16, 252, 120, 0, 0, 0, 0, 233, 122, 0],
            &[0, 0, 192, 221, 10, 0, 0, 0, 233, 122, 0],
            &[0, 0, 55, 249, 197, 72, 37, 60, 241, 122, 0],
            &[0, 0, 0, 62, 201, 255, 255, 255, 221, 78, 0],
            &[0, 0, 0, 0, 0, 13, 33, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ġ' / 0x121
        'ġ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 249, 162, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 191, 108, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 155, 182, 142, 22, 130, 47, 0],
            &[0, 0, 52, 246, 201, 124, 153, 213, 253, 81, 0],
            &[0, 0, 188, 212, 6, 0, 0, 115, 255, 81, 0],
            &[0, 6, 250, 125, 0, 0, 0, 25, 255, 81, 0],
            &[0, 25, 255, 97, 0, 0, 0, 1, 253, 81, 0],
            &[0, 18, 255, 110, 0, 0, 0, 8, 254, 81, 0],
            &[0, 0, 228, 169, 0, 0, 0, 56, 255, 81, 0],
            &[0, 0, 123, 251, 103, 17, 43, 203, 255, 81, 0],
            &[0, 0, 3, 147, 250, 255, 249, 124, 254, 81, 0],
            &[0, 0, 0, 0, 12, 35, 11, 16, 255, 74, 0],
            &[0, 0, 0, 0, 0, 0, 0, 76, 255, 37, 0],
            &[0, 0, 71, 170, 211, 229, 208, 132, 11, 0, 0],
        ]),
        // letter: 'Ģ' / 0x122
        'Ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 113, 141, 122, 63, 0, 0],
            &[0, 0, 0, 121, 252, 232, 191, 210, 253, 42, 0],
            &[0, 0, 81, 255, 132, 5, 0, 0, 28, 0, 0],
            &[0, 0, 204, 195, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 254, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 255, 65, 0, 0, 8, 8, 8, 3, 0],
            &[0, 66, 255, 53, 0, 0, 253, 255, 255, 122, 0],
            &[0, 51, 255, 70, 0, 0, 62, 63, 238, 122, 0],
            &[0, 16, 252, 120, 0, 0, 0, 0, 233, 122, 0],
            &[0, 0, 192, 221, 10, 0, 0, 0, 233, 122, 0],
            &[0, 0, 55, 249, 197, 72, 37, 60, 241, 122, 0],
            &[0, 0, 0, 62, 201, 255, 255, 255, 221, 78, 0],
            &[0, 0, 0, 0, 0, 15, 40, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 96, 238, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 117, 7, 0, 0, 0, 0],
        ]),
        // letter: 'ģ' / 0x123
        'ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 117, 157, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 238, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 77, 14, 0, 0, 0, 0],
            &[0, 0, 0, 47, 155, 182, 142, 22, 130, 47, 0],
            &[0, 0, 52, 246, 201, 124, 153, 213, 253, 81, 0],
            &[0, 0, 188, 212, 6, 0, 0, 115, 255, 81, 0],
            &[0, 6, 250, 125, 0, 0, 0, 25, 255, 81, 0],
            &[0, 25, 255, 97, 0, 0, 0, 1, 253, 81, 0],
            &[0, 18, 255, 110, 0, 0, 0, 8, 254, 81, 0],
            &[0, 0, 228, 169, 0, 0, 0, 56, 255, 81, 0],
            &[0, 0, 123, 251, 103, 17, 43, 203, 255, 81, 0],
            &[0, 0, 3, 147, 250, 255, 249, 124, 254, 81, 0],
            &[0, 0, 0, 0, 12, 35, 11, 16, 255, 74, 0],
            &[0, 0, 0, 0, 0, 0, 0, 76, 255, 37, 0],
            &[0, 0, 71, 170, 211, 229, 208, 132, 11, 0, 0],
        ]),
        // letter: 'Ĥ' / 0x124
        'Ĥ' => Some(&[
            &[0, 0, 0, 94, 253, 222, 15, 0, 0, 0, 0],
            &[0, 0, 58, 231, 59, 156, 189, 7, 0, 0, 0],
            &[0, 0, 96, 38, 0, 0, 89, 49, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 0, 104, 52, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 211, 155, 155, 155, 155, 251, 122, 0, 0],
            &[0, 225, 218, 171, 171, 171, 171, 252, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 14, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĥ' / 0x125
        'ĥ' => Some(&[
            &[4, 186, 163, 169, 186, 5, 0, 0, 0, 0, 0],
            &[91, 128, 2, 3, 130, 96, 0, 0, 0, 0, 0],
            &[0, 0, 29, 29, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 184, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 184, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 184, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 182, 33, 146, 178, 137, 23, 0, 0],
            &[0, 0, 184, 201, 217, 137, 131, 234, 218, 8, 0],
            &[0, 0, 184, 255, 59, 0, 0, 71, 255, 68, 0],
            &[0, 0, 184, 215, 0, 0, 0, 17, 255, 99, 0],
            &[0, 0, 184, 187, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 184, 184, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 184, 184, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 184, 184, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 184, 184, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 200, 200, 17, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ħ' / 0x126
        'Ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 0, 104, 52, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[133, 255, 255, 133, 133, 133, 133, 255, 255, 79, 0],
            &[161, 255, 255, 161, 161, 161, 161, 255, 255, 97, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 211, 155, 155, 155, 155, 251, 122, 0, 0],
            &[0, 225, 218, 171, 171, 171, 171, 252, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ħ' / 0x127
        'ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 29, 29, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 184, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 255, 255, 184, 184, 98, 0, 0, 0, 0],
            &[0, 85, 255, 255, 85, 85, 46, 0, 0, 0, 0],
            &[0, 0, 184, 182, 33, 146, 178, 137, 23, 0, 0],
            &[0, 0, 184, 201, 217, 137, 131, 234, 218, 8, 0],
            &[0, 0, 184, 255, 59, 0, 0, 71, 255, 68, 0],
            &[0, 0, 184, 215, 0, 0, 0, 17, 255, 99, 0],
            &[0, 0, 184, 187, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 184, 184, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 184, 184, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 184, 184, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 184, 184, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĩ' / 0x128
        'Ĩ' => Some(&[
            &[0, 0, 0, 34, 136, 88, 1, 39, 95, 0, 0],
            &[0, 0, 0, 206, 159, 216, 214, 225, 85, 0, 0],
            &[0, 0, 3, 97, 0, 4, 84, 71, 0, 0, 0],
            &[0, 0, 26, 108, 108, 108, 108, 108, 91, 0, 0],
            &[0, 0, 35, 162, 185, 252, 224, 175, 130, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 15, 38, 242, 151, 28, 6, 0, 0],
            &[0, 0, 61, 255, 255, 255, 255, 255, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĩ' / 0x129
        'ĩ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 175, 127, 13, 57, 115, 0, 0, 0],
            &[0, 0, 218, 120, 182, 242, 242, 64, 0, 0, 0],
            &[0, 3, 62, 0, 0, 46, 36, 0, 0, 0, 0],
            &[0, 68, 147, 147, 147, 114, 0, 0, 0, 0, 0],
            &[0, 51, 124, 143, 222, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 14, 34, 185, 209, 35, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ī' / 0x12a
        'Ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 130, 223, 223, 223, 223, 37, 0, 0],
            &[0, 0, 0, 39, 67, 67, 67, 67, 11, 0, 0],
            &[0, 0, 26, 108, 108, 108, 108, 108, 91, 0, 0],
            &[0, 0, 35, 162, 185, 252, 224, 175, 130, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 15, 38, 242, 151, 28, 6, 0, 0],
            &[0, 0, 61, 255, 255, 255, 255, 255, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ī' / 0x12b
        'ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 6, 6, 6, 6, 1, 0, 0, 0],
            &[0, 0, 149, 255, 255, 255, 255, 42, 0, 0, 0],
            &[0, 0, 16, 28, 28, 28, 28, 4, 0, 0, 0],
            &[0, 68, 147, 147, 147, 114, 0, 0, 0, 0, 0],
            &[0, 51, 124, 143, 222, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 14, 34, 185, 209, 35, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => Some(&[
            &[0, 0, 0, 109, 71, 0, 0, 141, 36, 0, 0],
            &[0, 0, 0, 69, 236, 155, 188, 207, 6, 0, 0],
            &[0, 0, 0, 0, 42, 104, 86, 8, 0, 0, 0],
            &[0, 0, 26, 108, 108, 108, 108, 108, 91, 0, 0],
            &[0, 0, 35, 162, 185, 252, 224, 175, 130, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 15, 38, 242, 151, 28, 6, 0, 0],
            &[0, 0, 61, 255, 255, 255, 255, 255, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĭ' / 0x12d
        'ĭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 97, 0, 3, 179, 41, 0, 0, 0],
            &[0, 0, 49, 235, 194, 224, 176, 2, 0, 0, 0],
            &[0, 0, 0, 17, 66, 47, 0, 0, 0, 0, 0],
            &[0, 68, 147, 147, 147, 114, 0, 0, 0, 0, 0],
            &[0, 51, 124, 143, 222, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 14, 34, 185, 209, 35, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Į' / 0x12e
        'Į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 108, 108, 108, 108, 108, 91, 0, 0],
            &[0, 0, 35, 162, 185, 252, 224, 175, 130, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 15, 38, 242, 151, 28, 6, 0, 0],
            &[0, 0, 61, 255, 255, 255, 255, 255, 215, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 166, 1, 0, 0],
            &[0, 0, 0, 0, 0, 8, 248, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 63, 126, 53, 0, 0],
        ]),
        // letter: 'į' / 0x12f
        'į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 235, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 173, 135, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 147, 147, 147, 114, 0, 0, 0, 0, 0],
            &[0, 51, 124, 143, 222, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 14, 34, 185, 209, 35, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 0, 50, 207, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 183, 102, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 121, 90, 0, 0, 0],
        ]),
        // letter: 'İ' / 0x130
        'İ' => Some(&[
            &[0, 0, 0, 0, 15, 226, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 229, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 108, 108, 108, 108, 108, 91, 0, 0],
            &[0, 0, 35, 162, 185, 252, 224, 175, 130, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 128, 0, 0, 0, 0],
            &[0, 0, 0, 15, 38, 242, 151, 28, 6, 0, 0],
            &[0, 0, 61, 255, 255, 255, 255, 255, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ı' / 0x131
        'ı' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 147, 147, 147, 114, 0, 0, 0, 0, 0],
            &[0, 51, 124, 143, 222, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 198, 0, 0, 0, 0, 0],
            &[0, 0, 14, 34, 185, 209, 35, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĳ' / 0x132
        'Ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 108, 4, 0, 0, 47, 108, 0, 0],
            &[0, 0, 102, 255, 10, 0, 0, 112, 255, 0, 0],
            &[0, 0, 102, 255, 10, 0, 0, 112, 255, 0, 0],
            &[0, 0, 102, 255, 10, 0, 0, 112, 255, 0, 0],
            &[0, 0, 102, 255, 10, 0, 0, 112, 255, 0, 0],
            &[0, 0, 102, 255, 10, 0, 0, 112, 255, 0, 0],
            &[0, 0, 102, 255, 10, 0, 0, 112, 255, 0, 0],
            &[0, 0, 102, 255, 10, 0, 0, 112, 255, 0, 0],
            &[0, 0, 81, 204, 8, 0, 0, 117, 253, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 165, 225, 0, 0],
            &[0, 0, 56, 103, 58, 59, 129, 253, 134, 0, 0],
            &[0, 0, 71, 242, 255, 255, 247, 152, 7, 0, 0],
            &[0, 0, 0, 1, 27, 34, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĳ' / 0x133
        'ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 0, 0, 0, 9, 9, 0, 0],
            &[0, 0, 64, 254, 118, 0, 0, 218, 216, 0, 0],
            &[0, 0, 35, 199, 73, 0, 0, 154, 152, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 147, 45, 0, 0, 95, 116, 0, 0],
            &[0, 0, 32, 255, 79, 0, 0, 165, 202, 0, 0],
            &[0, 0, 32, 255, 79, 0, 0, 165, 202, 0, 0],
            &[0, 0, 32, 255, 79, 0, 0, 165, 202, 0, 0],
            &[0, 0, 32, 255, 79, 0, 0, 165, 202, 0, 0],
            &[0, 0, 32, 255, 79, 0, 0, 165, 202, 0, 0],
            &[0, 0, 32, 255, 79, 0, 0, 165, 202, 0, 0],
            &[0, 0, 32, 255, 79, 0, 0, 165, 202, 0, 0],
            &[0, 0, 32, 255, 79, 0, 0, 165, 202, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 169, 202, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 211, 173, 0, 0],
            &[0, 0, 0, 171, 220, 226, 192, 85, 0, 0, 0],
        ]),
        // letter: 'Ĵ' / 0x134
        'Ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 115, 253, 209, 9, 0, 0],
            &[0, 0, 0, 0, 73, 228, 50, 173, 173, 3, 0],
            &[0, 0, 0, 0, 105, 29, 0, 2, 97, 39, 0],
            &[0, 0, 0, 0, 0, 0, 108, 47, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 112, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 112, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 112, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 112, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 112, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 112, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 112, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 255, 110, 0, 0, 0],
            &[0, 0, 0, 0, 0, 47, 255, 85, 0, 0, 0],
            &[0, 61, 99, 57, 76, 212, 240, 17, 0, 0, 0],
            &[0, 77, 240, 255, 255, 225, 69, 0, 0, 0, 0],
            &[0, 0, 3, 27, 28, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 26, 12, 0, 0, 0],
        ]),
        // letter: 'ĵ' / 0x135
        'ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 65, 15, 0, 0, 0],
            &[0, 0, 0, 0, 9, 207, 239, 174, 1, 0, 0],
            &[0, 0, 0, 2, 173, 159, 17, 193, 140, 0, 0],
            &[0, 0, 0, 18, 68, 0, 0, 5, 75, 9, 0],
            &[0, 0, 0, 130, 147, 147, 147, 51, 0, 0, 0],
            &[0, 0, 0, 102, 132, 160, 255, 90, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 255, 90, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 255, 90, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 255, 90, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 255, 90, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 255, 90, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 255, 90, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 255, 90, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 255, 89, 0, 0, 0],
            &[0, 0, 0, 0, 0, 65, 255, 62, 0, 0, 0],
            &[0, 6, 191, 224, 219, 166, 34, 0, 0, 0, 0],
        ]),
        // letter: 'Ķ' / 0x136
        'Ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 16, 108, 54, 0, 0],
            &[0, 225, 143, 0, 0, 0, 174, 228, 20, 0, 0],
            &[0, 225, 143, 0, 0, 109, 252, 62, 0, 0, 0],
            &[0, 225, 143, 0, 51, 249, 126, 0, 0, 0, 0],
            &[0, 225, 143, 15, 221, 189, 2, 0, 0, 0, 0],
            &[0, 225, 143, 164, 245, 25, 0, 0, 0, 0, 0],
            &[0, 225, 213, 241, 255, 79, 0, 0, 0, 0, 0],
            &[0, 225, 246, 55, 168, 228, 13, 0, 0, 0, 0],
            &[0, 225, 148, 0, 24, 239, 148, 0, 0, 0, 0],
            &[0, 225, 143, 0, 0, 104, 253, 56, 0, 0, 0],
            &[0, 225, 143, 0, 0, 1, 198, 211, 5, 0, 0],
            &[0, 225, 143, 0, 0, 0, 45, 251, 123, 0, 0],
            &[0, 0, 0, 0, 4, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 135, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 79, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ķ' / 0x137
        'ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 40, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 30, 0, 0, 17, 143, 92, 0, 0],
            &[0, 81, 255, 30, 0, 9, 196, 212, 18, 0, 0],
            &[0, 81, 255, 30, 3, 177, 224, 27, 0, 0, 0],
            &[0, 81, 255, 25, 154, 233, 37, 0, 0, 0, 0],
            &[0, 81, 255, 131, 255, 196, 3, 0, 0, 0, 0],
            &[0, 81, 255, 217, 65, 242, 131, 0, 0, 0, 0],
            &[0, 81, 255, 44, 0, 91, 253, 68, 0, 0, 0],
            &[0, 81, 255, 30, 0, 0, 162, 232, 24, 0, 0],
            &[0, 81, 255, 30, 0, 0, 12, 219, 188, 2, 0],
            &[0, 0, 0, 0, 1, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 88, 243, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 114, 9, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĸ' / 0x138
        'ĸ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 47, 147, 17, 0, 0, 17, 143, 92, 0, 0],
            &[0, 81, 255, 30, 0, 9, 196, 212, 18, 0, 0],
            &[0, 81, 255, 30, 3, 177, 224, 27, 0, 0, 0],
            &[0, 81, 255, 25, 154, 233, 37, 0, 0, 0, 0],
            &[0, 81, 255, 132, 255, 196, 3, 0, 0, 0, 0],
            &[0, 81, 255, 217, 65, 242, 131, 0, 0, 0, 0],
            &[0, 81, 255, 44, 0, 91, 253, 68, 0, 0, 0],
            &[0, 81, 255, 30, 0, 0, 162, 232, 24, 0, 0],
            &[0, 81, 255, 30, 0, 0, 12, 219, 188, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĺ' / 0x139
        'Ĺ' => Some(&[
            &[0, 0, 8, 223, 194, 4, 0, 0, 0, 0, 0],
            &[0, 0, 138, 207, 17, 0, 0, 0, 0, 0, 0],
            &[0, 5, 111, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 108, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 132, 83, 83, 83, 83, 20, 0, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 26, 3, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĺ' / 0x13a
        'ĺ' => Some(&[
            &[0, 0, 0, 0, 42, 247, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 88, 0, 0, 0, 0, 0],
            &[0, 25, 40, 40, 40, 35, 0, 0, 0, 0, 0],
            &[0, 136, 232, 249, 255, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 152, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 13, 33, 169, 226, 36, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 123, 209, 35, 0, 0, 0],
        ]),
        // letter: 'Ļ' / 0x13b
        'Ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 108, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 132, 83, 83, 83, 83, 20, 0, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 2, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 120, 1, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ļ' / 0x13c
        'ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 40, 40, 40, 35, 0, 0, 0, 0, 0],
            &[0, 136, 232, 249, 255, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 152, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 13, 33, 169, 226, 36, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 3, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 106, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ľ' / 0x13d
        'Ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 108, 30, 0, 0, 39, 107, 3, 0, 0],
            &[0, 40, 255, 71, 0, 0, 125, 197, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 185, 72, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 28, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 132, 83, 83, 83, 83, 20, 0, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ľ' / 0x13e
        'ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 40, 40, 40, 35, 0, 16, 40, 0, 0],
            &[0, 136, 232, 249, 255, 219, 0, 126, 210, 0, 0],
            &[0, 0, 0, 0, 152, 219, 0, 182, 94, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 87, 1, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 13, 33, 169, 226, 36, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 108, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 1, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 130, 236, 18, 0, 0],
            &[0, 40, 255, 71, 0, 0, 145, 248, 24, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 12, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 132, 83, 83, 83, 83, 20, 0, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŀ' / 0x140
        'ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[25, 40, 40, 40, 35, 0, 0, 0, 0, 0, 0],
            &[136, 232, 249, 255, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 152, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 149, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 149, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 149, 219, 0, 0, 19, 147, 44, 0],
            &[0, 0, 0, 149, 219, 0, 0, 77, 255, 128, 0],
            &[0, 0, 0, 149, 219, 0, 0, 10, 106, 27, 0],
            &[0, 0, 0, 149, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 149, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 149, 219, 0, 0, 0, 0, 0, 0],
            &[0, 13, 33, 169, 226, 36, 14, 0, 0, 0, 0],
            &[181, 255, 255, 255, 255, 255, 255, 161, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ł' / 0x141
        'Ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 99, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 133, 235, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 133, 235, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 133, 235, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 133, 235, 0, 28, 0, 0, 0, 0],
            &[0, 0, 0, 133, 246, 132, 247, 51, 0, 0, 0],
            &[0, 0, 0, 219, 255, 204, 59, 0, 0, 0, 0],
            &[0, 5, 188, 255, 255, 2, 0, 0, 0, 0, 0],
            &[0, 0, 88, 147, 235, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 133, 235, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 133, 242, 83, 83, 83, 83, 73, 0],
            &[0, 0, 0, 133, 255, 255, 255, 255, 255, 225, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ł' / 0x142
        'ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 40, 40, 40, 35, 0, 0, 0, 0, 0],
            &[0, 136, 232, 249, 255, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 152, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 86, 93, 0, 0, 0],
            &[0, 0, 0, 0, 183, 255, 232, 83, 0, 0, 0],
            &[0, 0, 5, 122, 255, 255, 16, 0, 0, 0, 0],
            &[0, 0, 121, 207, 204, 219, 0, 0, 0, 0, 0],
            &[0, 0, 8, 3, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 219, 0, 0, 0, 0, 0],
            &[0, 0, 13, 33, 169, 226, 36, 14, 0, 0, 0],
            &[0, 181, 255, 255, 255, 255, 255, 255, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ń' / 0x143
        'Ń' => Some(&[
            &[0, 0, 0, 0, 0, 153, 239, 37, 0, 0, 0],
            &[0, 0, 0, 0, 62, 241, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 47, 0, 0, 0, 0, 0],
            &[0, 95, 108, 12, 0, 0, 0, 98, 52, 0, 0],
            &[0, 225, 255, 109, 0, 0, 0, 233, 122, 0, 0],
            &[0, 225, 198, 219, 2, 0, 0, 233, 122, 0, 0],
            &[0, 225, 120, 241, 77, 0, 0, 233, 122, 0, 0],
            &[0, 225, 123, 142, 189, 0, 0, 233, 122, 0, 0],
            &[0, 225, 130, 32, 252, 45, 0, 233, 122, 0, 0],
            &[0, 225, 131, 0, 173, 156, 0, 233, 122, 0, 0],
            &[0, 225, 131, 0, 61, 247, 21, 230, 122, 0, 0],
            &[0, 225, 131, 0, 0, 204, 124, 223, 122, 0, 0],
            &[0, 225, 131, 0, 0, 92, 230, 214, 122, 0, 0],
            &[0, 225, 131, 0, 0, 5, 230, 254, 122, 0, 0],
            &[0, 225, 131, 0, 0, 0, 123, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 26, 11, 0, 0, 0],
        ]),
        // letter: 'ń' / 0x144
        'ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 65, 35, 0, 0, 0],
            &[0, 0, 0, 0, 0, 148, 237, 36, 0, 0, 0],
            &[0, 0, 0, 0, 60, 236, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 33, 0, 0, 0, 0, 0],
            &[0, 106, 70, 44, 149, 179, 143, 29, 0, 0, 0],
            &[0, 184, 194, 222, 139, 126, 224, 225, 12, 0, 0],
            &[0, 184, 255, 48, 0, 0, 58, 255, 73, 0, 0],
            &[0, 184, 215, 0, 0, 0, 14, 255, 100, 0, 0],
            &[0, 184, 188, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ņ' / 0x145
        'Ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 95, 108, 12, 0, 0, 0, 98, 52, 0, 0],
            &[0, 225, 255, 109, 0, 0, 0, 233, 122, 0, 0],
            &[0, 225, 198, 219, 2, 0, 0, 233, 122, 0, 0],
            &[0, 225, 120, 241, 77, 0, 0, 233, 122, 0, 0],
            &[0, 225, 123, 142, 189, 0, 0, 233, 122, 0, 0],
            &[0, 225, 130, 32, 252, 45, 0, 233, 122, 0, 0],
            &[0, 225, 131, 0, 173, 156, 0, 233, 122, 0, 0],
            &[0, 225, 131, 0, 61, 247, 21, 230, 122, 0, 0],
            &[0, 225, 131, 0, 0, 204, 124, 223, 122, 0, 0],
            &[0, 225, 131, 0, 0, 92, 230, 214, 122, 0, 0],
            &[0, 225, 131, 0, 0, 5, 230, 254, 122, 0, 0],
            &[0, 225, 131, 0, 0, 0, 123, 255, 122, 0, 0],
            &[0, 0, 0, 0, 5, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 245, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 66, 58, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ņ' / 0x146
        'ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 106, 70, 44, 149, 179, 143, 29, 0, 0, 0],
            &[0, 184, 194, 222, 139, 126, 224, 225, 12, 0, 0],
            &[0, 184, 255, 48, 0, 0, 58, 255, 73, 0, 0],
            &[0, 184, 215, 0, 0, 0, 14, 255, 100, 0, 0],
            &[0, 184, 188, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 0, 0, 0, 4, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 223, 118, 0, 0, 0, 0, 0],
            &[0, 0, 0, 53, 71, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ň' / 0x147
        'Ň' => Some(&[
            &[0, 0, 115, 204, 26, 66, 225, 60, 0, 0, 0],
            &[0, 0, 0, 145, 226, 247, 77, 0, 0, 0, 0],
            &[0, 0, 0, 5, 112, 86, 0, 0, 0, 0, 0],
            &[0, 95, 108, 12, 0, 0, 0, 98, 52, 0, 0],
            &[0, 225, 255, 109, 0, 0, 0, 233, 122, 0, 0],
            &[0, 225, 198, 219, 2, 0, 0, 233, 122, 0, 0],
            &[0, 225, 120, 241, 77, 0, 0, 233, 122, 0, 0],
            &[0, 225, 123, 142, 189, 0, 0, 233, 122, 0, 0],
            &[0, 225, 130, 32, 252, 45, 0, 233, 122, 0, 0],
            &[0, 225, 131, 0, 173, 156, 0, 233, 122, 0, 0],
            &[0, 225, 131, 0, 61, 247, 21, 230, 122, 0, 0],
            &[0, 225, 131, 0, 0, 204, 124, 223, 122, 0, 0],
            &[0, 225, 131, 0, 0, 92, 230, 214, 122, 0, 0],
            &[0, 225, 131, 0, 0, 5, 230, 254, 122, 0, 0],
            &[0, 225, 131, 0, 0, 0, 123, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 5, 0, 0, 12, 15, 0, 0, 0],
        ]),
        // letter: 'ň' / 0x148
        'ň' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 46, 27, 0, 0, 30, 45, 0, 0, 0],
            &[0, 0, 61, 230, 66, 76, 230, 55, 0, 0, 0],
            &[0, 0, 0, 85, 247, 249, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 62, 0, 0, 0, 0, 0],
            &[0, 106, 70, 44, 149, 179, 143, 29, 0, 0, 0],
            &[0, 184, 194, 222, 139, 126, 224, 225, 12, 0, 0],
            &[0, 184, 255, 48, 0, 0, 58, 255, 73, 0, 0],
            &[0, 184, 215, 0, 0, 0, 14, 255, 100, 0, 0],
            &[0, 184, 188, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŉ' / 0x149
        'ŉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 108, 75, 0, 0, 0, 0, 0, 0, 0, 0],
            &[45, 255, 132, 0, 0, 0, 0, 0, 0, 0, 0],
            &[94, 255, 45, 0, 0, 0, 0, 0, 0, 0, 0],
            &[147, 202, 70, 106, 24, 138, 178, 147, 32, 0, 0],
            &[94, 60, 122, 225, 208, 148, 125, 226, 228, 13, 0],
            &[0, 0, 122, 255, 105, 0, 0, 60, 255, 74, 0],
            &[0, 0, 122, 255, 19, 0, 0, 14, 255, 101, 0],
            &[0, 0, 122, 249, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 122, 245, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 122, 245, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 122, 245, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 122, 245, 0, 0, 0, 10, 255, 102, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 95, 108, 12, 0, 0, 0, 98, 52, 0, 0],
            &[0, 225, 255, 109, 0, 0, 0, 233, 122, 0, 0],
            &[0, 225, 198, 219, 2, 0, 0, 233, 122, 0, 0],
            &[0, 225, 120, 241, 77, 0, 0, 233, 122, 0, 0],
            &[0, 225, 123, 142, 189, 0, 0, 233, 122, 0, 0],
            &[0, 225, 130, 32, 252, 45, 0, 233, 122, 0, 0],
            &[0, 225, 131, 0, 173, 156, 0, 233, 122, 0, 0],
            &[0, 225, 131, 0, 61, 247, 21, 230, 122, 0, 0],
            &[0, 225, 131, 0, 0, 204, 124, 223, 122, 0, 0],
            &[0, 225, 131, 0, 0, 92, 230, 214, 122, 0, 0],
            &[0, 225, 131, 0, 0, 5, 230, 255, 161, 0, 0],
            &[0, 225, 131, 0, 0, 0, 123, 255, 245, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 248, 100, 0, 0],
            &[0, 0, 0, 0, 59, 67, 161, 250, 38, 0, 0],
            &[0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŋ' / 0x14b
        'ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 106, 70, 44, 149, 179, 143, 29, 0, 0, 0],
            &[0, 184, 194, 222, 139, 126, 224, 225, 12, 0, 0],
            &[0, 184, 255, 48, 0, 0, 58, 255, 73, 0, 0],
            &[0, 184, 215, 0, 0, 0, 14, 255, 100, 0, 0],
            &[0, 184, 188, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 184, 184, 0, 0, 0, 19, 255, 203, 0, 0],
            &[0, 184, 184, 0, 0, 0, 20, 255, 204, 0, 0],
            &[0, 184, 184, 0, 0, 0, 20, 255, 204, 0, 0],
            &[0, 184, 184, 0, 0, 0, 20, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 255, 102, 0, 0],
            &[0, 0, 0, 0, 0, 0, 16, 255, 96, 0, 0],
            &[0, 0, 0, 0, 9, 197, 202, 103, 0, 0, 0],
        ]),
        // letter: 'Ō' / 0x14c
        'Ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 130, 223, 223, 223, 223, 37, 0, 0],
            &[0, 0, 0, 39, 67, 67, 67, 67, 11, 0, 0],
            &[0, 0, 0, 5, 96, 145, 137, 59, 0, 0, 0],
            &[0, 0, 12, 204, 245, 184, 199, 254, 120, 0, 0],
            &[0, 0, 136, 241, 39, 0, 0, 114, 253, 41, 0],
            &[0, 0, 234, 144, 0, 0, 0, 7, 244, 135, 0],
            &[0, 34, 255, 84, 0, 0, 0, 0, 188, 189, 0],
            &[0, 58, 255, 58, 0, 0, 0, 0, 166, 216, 0],
            &[0, 67, 255, 52, 0, 0, 0, 0, 155, 221, 0],
            &[0, 54, 255, 66, 0, 0, 0, 0, 168, 206, 0],
            &[0, 18, 254, 105, 0, 0, 0, 0, 206, 168, 0],
            &[0, 0, 203, 191, 0, 0, 0, 35, 254, 97, 0],
            &[0, 0, 78, 255, 144, 45, 64, 207, 219, 7, 0],
            &[0, 0, 0, 101, 235, 255, 255, 194, 35, 0, 0],
            &[0, 0, 0, 0, 4, 31, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ō' / 0x14d
        'ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 6, 6, 6, 6, 1, 0, 0],
            &[0, 0, 0, 149, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 0, 16, 28, 28, 28, 28, 4, 0, 0],
            &[0, 0, 0, 27, 133, 178, 165, 89, 1, 0, 0],
            &[0, 0, 39, 235, 207, 127, 143, 242, 164, 0, 0],
            &[0, 0, 180, 213, 8, 0, 0, 68, 255, 73, 0],
            &[0, 4, 250, 117, 0, 0, 0, 0, 223, 150, 0],
            &[0, 24, 255, 85, 0, 0, 0, 0, 193, 177, 0],
            &[0, 14, 255, 96, 0, 0, 0, 0, 206, 169, 0],
            &[0, 0, 217, 163, 0, 0, 0, 21, 250, 116, 0],
            &[0, 0, 98, 252, 105, 13, 34, 184, 235, 20, 0],
            &[0, 0, 0, 112, 237, 255, 255, 206, 50, 0, 0],
            &[0, 0, 0, 0, 3, 33, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => Some(&[
            &[0, 0, 0, 109, 71, 0, 0, 141, 36, 0, 0],
            &[0, 0, 0, 69, 236, 155, 188, 207, 6, 0, 0],
            &[0, 0, 0, 0, 42, 104, 86, 8, 0, 0, 0],
            &[0, 0, 0, 5, 96, 145, 137, 59, 0, 0, 0],
            &[0, 0, 12, 204, 245, 184, 199, 254, 120, 0, 0],
            &[0, 0, 136, 241, 39, 0, 0, 114, 253, 41, 0],
            &[0, 0, 234, 144, 0, 0, 0, 7, 244, 135, 0],
            &[0, 34, 255, 84, 0, 0, 0, 0, 188, 189, 0],
            &[0, 58, 255, 58, 0, 0, 0, 0, 166, 216, 0],
            &[0, 67, 255, 52, 0, 0, 0, 0, 155, 221, 0],
            &[0, 54, 255, 66, 0, 0, 0, 0, 168, 206, 0],
            &[0, 18, 254, 105, 0, 0, 0, 0, 206, 168, 0],
            &[0, 0, 203, 191, 0, 0, 0, 35, 254, 97, 0],
            &[0, 0, 78, 255, 144, 45, 64, 207, 219, 7, 0],
            &[0, 0, 0, 101, 235, 255, 255, 194, 35, 0, 0],
            &[0, 0, 0, 0, 4, 31, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŏ' / 0x14f
        'ŏ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 130, 97, 0, 3, 179, 41, 0, 0],
            &[0, 0, 0, 49, 235, 194, 224, 176, 2, 0, 0],
            &[0, 0, 0, 0, 17, 66, 47, 0, 0, 0, 0],
            &[0, 0, 0, 27, 133, 178, 165, 89, 1, 0, 0],
            &[0, 0, 39, 235, 207, 127, 143, 242, 164, 0, 0],
            &[0, 0, 180, 213, 8, 0, 0, 68, 255, 73, 0],
            &[0, 4, 250, 117, 0, 0, 0, 0, 223, 150, 0],
            &[0, 24, 255, 85, 0, 0, 0, 0, 193, 177, 0],
            &[0, 14, 255, 96, 0, 0, 0, 0, 206, 169, 0],
            &[0, 0, 217, 163, 0, 0, 0, 21, 250, 116, 0],
            &[0, 0, 98, 252, 105, 13, 34, 184, 235, 20, 0],
            &[0, 0, 0, 112, 237, 255, 255, 206, 50, 0, 0],
            &[0, 0, 0, 0, 3, 33, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ő' / 0x150
        'Ő' => Some(&[
            &[0, 0, 0, 0, 176, 208, 11, 217, 169, 0, 0],
            &[0, 0, 0, 67, 228, 31, 114, 201, 11, 0, 0],
            &[0, 0, 0, 82, 33, 0, 101, 14, 0, 0, 0],
            &[0, 0, 0, 5, 96, 145, 137, 59, 0, 0, 0],
            &[0, 0, 12, 204, 245, 184, 199, 254, 120, 0, 0],
            &[0, 0, 136, 241, 39, 0, 0, 114, 253, 41, 0],
            &[0, 0, 234, 144, 0, 0, 0, 7, 244, 135, 0],
            &[0, 34, 255, 84, 0, 0, 0, 0, 188, 189, 0],
            &[0, 58, 255, 58, 0, 0, 0, 0, 166, 216, 0],
            &[0, 67, 255, 52, 0, 0, 0, 0, 155, 221, 0],
            &[0, 54, 255, 66, 0, 0, 0, 0, 168, 206, 0],
            &[0, 18, 254, 105, 0, 0, 0, 0, 206, 168, 0],
            &[0, 0, 203, 191, 0, 0, 0, 35, 254, 97, 0],
            &[0, 0, 78, 255, 144, 45, 64, 207, 219, 7, 0],
            &[0, 0, 0, 101, 235, 255, 255, 194, 35, 0, 0],
            &[0, 0, 0, 0, 4, 31, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 26, 4, 15, 26, 0, 0],
        ]),
        // letter: 'ő' / 0x151
        'ő' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 65, 10, 40, 63, 0, 0],
            &[0, 0, 0, 0, 196, 185, 13, 231, 139, 0, 0],
            &[0, 0, 0, 91, 207, 15, 138, 173, 3, 0, 0],
            &[0, 0, 0, 58, 15, 0, 69, 3, 0, 0, 0],
            &[0, 0, 0, 27, 133, 178, 165, 89, 1, 0, 0],
            &[0, 0, 39, 235, 207, 127, 143, 242, 164, 0, 0],
            &[0, 0, 180, 213, 8, 0, 0, 68, 255, 73, 0],
            &[0, 4, 250, 117, 0, 0, 0, 0, 223, 150, 0],
            &[0, 24, 255, 85, 0, 0, 0, 0, 193, 177, 0],
            &[0, 14, 255, 96, 0, 0, 0, 0, 206, 169, 0],
            &[0, 0, 217, 163, 0, 0, 0, 21, 250, 116, 0],
            &[0, 0, 98, 252, 105, 13, 34, 184, 235, 20, 0],
            &[0, 0, 0, 112, 237, 255, 255, 206, 50, 0, 0],
            &[0, 0, 0, 0, 3, 33, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Œ' / 0x152
        'Œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 40, 130, 139, 109, 108, 108, 108, 13, 0],
            &[0, 56, 246, 215, 201, 255, 232, 219, 219, 26, 0],
            &[0, 196, 200, 2, 10, 255, 90, 0, 0, 0, 0],
            &[21, 254, 103, 0, 10, 255, 90, 0, 0, 0, 0],
            &[63, 255, 55, 0, 10, 255, 90, 0, 0, 0, 0],
            &[84, 255, 39, 0, 10, 255, 190, 155, 126, 0, 0],
            &[88, 255, 31, 0, 10, 255, 198, 167, 136, 0, 0],
            &[76, 255, 41, 0, 10, 255, 90, 0, 0, 0, 0],
            &[45, 255, 69, 0, 10, 255, 90, 0, 0, 0, 0],
            &[6, 242, 130, 0, 10, 255, 90, 0, 0, 0, 0],
            &[0, 150, 237, 66, 64, 255, 136, 71, 71, 8, 0],
            &[0, 15, 193, 255, 255, 255, 255, 255, 255, 30, 0],
            &[0, 0, 0, 24, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'œ' / 0x153
        'œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 158, 171, 44, 27, 157, 162, 31, 0, 0],
            &[13, 233, 167, 155, 223, 199, 151, 175, 199, 0, 0],
            &[95, 231, 3, 0, 217, 234, 2, 25, 255, 29, 0],
            &[139, 181, 0, 0, 168, 198, 18, 18, 246, 63, 0],
            &[160, 166, 0, 0, 151, 252, 243, 243, 243, 68, 0],
            &[152, 174, 0, 0, 159, 192, 0, 0, 0, 0, 0],
            &[116, 208, 0, 0, 184, 224, 0, 0, 0, 0, 0],
            &[40, 253, 67, 43, 245, 238, 83, 1, 64, 0, 0],
            &[0, 121, 252, 252, 123, 86, 246, 248, 205, 0, 0],
            &[0, 0, 17, 21, 0, 0, 13, 27, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŕ' / 0x154
        'Ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 174, 230, 25, 0, 0, 0],
            &[0, 0, 0, 0, 80, 236, 45, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 38, 0, 0, 0, 0, 0],
            &[0, 52, 108, 108, 102, 75, 14, 0, 0, 0, 0],
            &[0, 122, 254, 210, 220, 249, 240, 79, 0, 0, 0],
            &[0, 122, 245, 0, 0, 18, 190, 238, 8, 0, 0],
            &[0, 122, 245, 0, 0, 0, 88, 255, 43, 0, 0],
            &[0, 122, 245, 0, 0, 0, 104, 255, 32, 0, 0],
            &[0, 122, 245, 0, 0, 37, 210, 188, 0, 0, 0],
            &[0, 122, 255, 235, 242, 255, 169, 21, 0, 0, 0],
            &[0, 122, 249, 83, 97, 251, 102, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 142, 236, 19, 0, 0, 0],
            &[0, 122, 245, 0, 0, 15, 232, 154, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 97, 253, 53, 0, 0],
            &[0, 122, 245, 0, 0, 0, 1, 200, 203, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 26, 9, 0, 0, 0],
        ]),
        // letter: 'ŕ' / 0x155
        'ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 50, 64, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 250, 132, 0, 0, 0],
            &[0, 0, 0, 0, 3, 196, 150, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 68, 0, 0, 0, 0, 0],
            &[0, 110, 147, 147, 116, 11, 137, 180, 149, 10, 0],
            &[0, 86, 134, 194, 226, 172, 202, 144, 172, 0, 0],
            &[0, 0, 0, 102, 255, 186, 3, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 14, 0, 0, 0, 0, 0],
            &[0, 3, 25, 128, 255, 51, 15, 0, 0, 0, 0],
            &[1, 253, 255, 255, 255, 255, 255, 139, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŗ' / 0x156
        'Ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 108, 108, 102, 75, 14, 0, 0, 0, 0],
            &[0, 122, 254, 210, 220, 249, 240, 79, 0, 0, 0],
            &[0, 122, 245, 0, 0, 18, 190, 238, 8, 0, 0],
            &[0, 122, 245, 0, 0, 0, 88, 255, 43, 0, 0],
            &[0, 122, 245, 0, 0, 0, 104, 255, 32, 0, 0],
            &[0, 122, 245, 0, 0, 37, 210, 188, 0, 0, 0],
            &[0, 122, 255, 235, 242, 255, 169, 21, 0, 0, 0],
            &[0, 122, 249, 83, 97, 251, 102, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 142, 236, 19, 0, 0, 0],
            &[0, 122, 245, 0, 0, 15, 232, 154, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 97, 253, 53, 0, 0],
            &[0, 122, 245, 0, 0, 0, 1, 200, 203, 2, 0],
            &[0, 0, 0, 0, 2, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 120, 1, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŗ' / 0x157
        'ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 110, 147, 147, 116, 11, 137, 180, 149, 10, 0],
            &[0, 86, 134, 194, 226, 172, 202, 144, 172, 0, 0],
            &[0, 0, 0, 102, 255, 186, 3, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 14, 0, 0, 0, 0, 0],
            &[0, 3, 25, 128, 255, 51, 15, 0, 0, 0, 0],
            &[1, 253, 255, 255, 255, 255, 255, 139, 0, 0, 0],
            &[0, 0, 0, 2, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 111, 229, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 120, 3, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ř' / 0x158
        'Ř' => Some(&[
            &[0, 0, 135, 192, 18, 80, 225, 47, 0, 0, 0],
            &[0, 0, 0, 165, 222, 246, 60, 0, 0, 0, 0],
            &[0, 0, 0, 12, 115, 77, 0, 0, 0, 0, 0],
            &[0, 52, 108, 108, 102, 75, 14, 0, 0, 0, 0],
            &[0, 122, 254, 210, 220, 249, 240, 79, 0, 0, 0],
            &[0, 122, 245, 0, 0, 18, 190, 238, 8, 0, 0],
            &[0, 122, 245, 0, 0, 0, 88, 255, 43, 0, 0],
            &[0, 122, 245, 0, 0, 0, 104, 255, 32, 0, 0],
            &[0, 122, 245, 0, 0, 37, 210, 188, 0, 0, 0],
            &[0, 122, 255, 235, 242, 255, 169, 21, 0, 0, 0],
            &[0, 122, 249, 83, 97, 251, 102, 0, 0, 0, 0],
            &[0, 122, 245, 0, 0, 142, 236, 19, 0, 0, 0],
            &[0, 122, 245, 0, 0, 15, 232, 154, 0, 0, 0],
            &[0, 122, 245, 0, 0, 0, 97, 253, 53, 0, 0],
            &[0, 122, 245, 0, 0, 0, 1, 200, 203, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 24, 3, 0, 0, 14, 13, 0, 0, 0],
        ]),
        // letter: 'ř' / 0x159
        'ř' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 56, 0, 0, 3, 63, 9, 0, 0],
            &[0, 0, 6, 186, 160, 19, 189, 156, 1, 0, 0],
            &[0, 0, 0, 13, 213, 243, 183, 3, 0, 0, 0],
            &[0, 0, 0, 0, 31, 77, 19, 0, 0, 0, 0],
            &[0, 110, 147, 147, 116, 11, 137, 180, 149, 10, 0],
            &[0, 86, 134, 194, 226, 172, 202, 144, 172, 0, 0],
            &[0, 0, 0, 102, 255, 186, 3, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 255, 14, 0, 0, 0, 0, 0],
            &[0, 3, 25, 128, 255, 51, 15, 0, 0, 0, 0],
            &[1, 253, 255, 255, 255, 255, 255, 139, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ś' / 0x15a
        'Ś' => Some(&[
            &[0, 0, 0, 0, 0, 96, 254, 80, 0, 0, 0],
            &[0, 0, 0, 0, 23, 234, 105, 0, 0, 0, 0],
            &[0, 0, 0, 0, 57, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 131, 140, 113, 49, 0, 0, 0],
            &[0, 0, 167, 255, 208, 187, 220, 250, 21, 0, 0],
            &[0, 56, 255, 114, 0, 0, 0, 28, 0, 0, 0],
            &[0, 90, 255, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 255, 132, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 142, 255, 199, 78, 1, 0, 0, 0, 0],
            &[0, 0, 0, 64, 193, 255, 210, 59, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 209, 246, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 105, 0, 0],
            &[0, 0, 0, 0, 0, 0, 38, 255, 92, 0, 0],
            &[0, 107, 99, 51, 27, 60, 195, 247, 29, 0, 0],
            &[0, 109, 233, 255, 255, 255, 212, 66, 0, 0, 0],
            &[0, 0, 0, 16, 36, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 26, 17, 0, 0, 0],
        ]),
        // letter: 'ś' / 0x15b
        'ś' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 65, 27, 0, 0],
            &[0, 0, 0, 0, 0, 0, 177, 223, 21, 0, 0],
            &[0, 0, 0, 0, 0, 86, 227, 36, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 25, 0, 0, 0, 0],
            &[0, 0, 0, 21, 127, 177, 178, 146, 62, 0, 0],
            &[0, 0, 4, 217, 217, 132, 127, 169, 122, 0, 0],
            &[0, 0, 36, 255, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 222, 207, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 167, 253, 199, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 154, 254, 131, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 227, 0, 0],
            &[0, 0, 53, 102, 31, 10, 41, 209, 192, 0, 0],
            &[0, 0, 54, 223, 255, 255, 255, 200, 39, 0, 0],
            &[0, 0, 0, 0, 17, 35, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => Some(&[
            &[0, 0, 0, 32, 240, 250, 63, 0, 0, 0, 0],
            &[0, 0, 16, 210, 119, 86, 229, 39, 0, 0, 0],
            &[0, 0, 60, 74, 0, 0, 54, 84, 0, 0, 0],
            &[0, 0, 0, 69, 131, 140, 113, 49, 0, 0, 0],
            &[0, 0, 167, 255, 208, 187, 220, 250, 21, 0, 0],
            &[0, 56, 255, 114, 0, 0, 0, 28, 0, 0, 0],
            &[0, 90, 255, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 255, 132, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 142, 255, 199, 78, 1, 0, 0, 0, 0],
            &[0, 0, 0, 64, 193, 255, 210, 59, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 209, 246, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 105, 0, 0],
            &[0, 0, 0, 0, 0, 0, 38, 255, 92, 0, 0],
            &[0, 107, 99, 51, 27, 60, 195, 247, 29, 0, 0],
            &[0, 109, 233, 255, 255, 255, 212, 66, 0, 0, 0],
            &[0, 0, 0, 16, 36, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 22, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŝ' / 0x15d
        'ŝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 103, 241, 244, 43, 0, 0, 0],
            &[0, 0, 0, 69, 224, 45, 99, 220, 28, 0, 0],
            &[0, 0, 0, 62, 24, 0, 0, 46, 43, 0, 0],
            &[0, 0, 0, 21, 127, 177, 178, 146, 62, 0, 0],
            &[0, 0, 4, 217, 217, 132, 127, 169, 122, 0, 0],
            &[0, 0, 36, 255, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 222, 207, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 167, 253, 199, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 154, 254, 131, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 227, 0, 0],
            &[0, 0, 53, 102, 31, 10, 41, 209, 192, 0, 0],
            &[0, 0, 54, 223, 255, 255, 255, 200, 39, 0, 0],
            &[0, 0, 0, 0, 17, 35, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ş' / 0x15e
        'Ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 131, 140, 113, 49, 0, 0, 0],
            &[0, 0, 167, 255, 208, 187, 220, 250, 21, 0, 0],
            &[0, 56, 255, 114, 0, 0, 0, 28, 0, 0, 0],
            &[0, 90, 255, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 255, 132, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 142, 255, 199, 78, 1, 0, 0, 0, 0],
            &[0, 0, 0, 64, 193, 255, 210, 59, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 209, 246, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 105, 0, 0],
            &[0, 0, 0, 0, 0, 0, 38, 255, 92, 0, 0],
            &[0, 107, 99, 51, 27, 60, 195, 247, 29, 0, 0],
            &[0, 109, 233, 255, 255, 255, 212, 66, 0, 0, 0],
            &[0, 0, 0, 20, 253, 47, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 157, 222, 25, 0, 0, 0, 0],
            &[0, 0, 0, 179, 200, 122, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ş' / 0x15f
        'ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 127, 177, 178, 146, 62, 0, 0],
            &[0, 0, 4, 217, 217, 132, 127, 169, 122, 0, 0],
            &[0, 0, 36, 255, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 222, 207, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 167, 253, 199, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 154, 254, 131, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 227, 0, 0],
            &[0, 0, 53, 102, 31, 10, 41, 209, 192, 0, 0],
            &[0, 0, 54, 223, 255, 255, 255, 200, 39, 0, 0],
            &[0, 0, 0, 0, 17, 238, 63, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 147, 228, 38, 0, 0, 0],
            &[0, 0, 0, 0, 162, 202, 134, 3, 0, 0, 0],
        ]),
        // letter: 'Š' / 0x160
        'Š' => Some(&[
            &[0, 0, 65, 223, 57, 33, 210, 108, 0, 0, 0],
            &[0, 0, 0, 88, 243, 231, 132, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 109, 2, 0, 0, 0, 0],
            &[0, 0, 0, 69, 131, 140, 113, 49, 0, 0, 0],
            &[0, 0, 167, 255, 208, 187, 220, 250, 21, 0, 0],
            &[0, 56, 255, 114, 0, 0, 0, 28, 0, 0, 0],
            &[0, 90, 255, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 255, 132, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 142, 255, 199, 78, 1, 0, 0, 0, 0],
            &[0, 0, 0, 64, 193, 255, 210, 59, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 209, 246, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 105, 0, 0],
            &[0, 0, 0, 0, 0, 0, 38, 255, 92, 0, 0],
            &[0, 107, 99, 51, 27, 60, 195, 247, 29, 0, 0],
            &[0, 109, 233, 255, 255, 255, 212, 66, 0, 0, 0],
            &[0, 0, 0, 16, 36, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 11, 0, 0, 6, 21, 0, 0, 0],
        ]),
        // letter: 'š' / 0x161
        'š' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 53, 19, 0, 0, 37, 37, 0, 0],
            &[0, 0, 0, 83, 226, 47, 98, 226, 38, 0, 0],
            &[0, 0, 0, 0, 114, 244, 247, 51, 0, 0, 0],
            &[0, 0, 0, 0, 0, 74, 53, 0, 0, 0, 0],
            &[0, 0, 0, 21, 127, 177, 178, 146, 62, 0, 0],
            &[0, 0, 4, 217, 217, 132, 127, 169, 122, 0, 0],
            &[0, 0, 36, 255, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 222, 207, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 167, 253, 199, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 154, 254, 131, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 227, 0, 0],
            &[0, 0, 53, 102, 31, 10, 41, 209, 192, 0, 0],
            &[0, 0, 54, 223, 255, 255, 255, 200, 39, 0, 0],
            &[0, 0, 0, 0, 17, 35, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ţ' / 0x162
        'Ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 108, 108, 108, 108, 108, 108, 108, 95, 0],
            &[0, 61, 219, 219, 219, 252, 238, 219, 219, 192, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 248, 139, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 217, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 157, 222, 25, 0, 0, 0],
            &[0, 0, 0, 0, 179, 200, 122, 0, 0, 0, 0],
        ]),
        // letter: 'ţ' / 0x163
        'ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 118, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 167, 171, 0, 0, 0, 0, 0],
            &[0, 9, 111, 139, 230, 220, 147, 147, 147, 2, 0],
            &[0, 16, 159, 159, 233, 224, 159, 159, 159, 2, 0],
            &[0, 0, 0, 0, 196, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 189, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 246, 73, 11, 28, 14, 0],
            &[0, 0, 0, 0, 22, 192, 255, 255, 254, 60, 0],
            &[0, 0, 0, 0, 0, 0, 194, 108, 14, 0, 0],
            &[0, 0, 0, 0, 0, 3, 136, 230, 57, 0, 0],
            &[0, 0, 0, 0, 0, 140, 204, 148, 10, 0, 0],
        ]),
        // letter: 'Ť' / 0x164
        'Ť' => Some(&[
            &[0, 0, 0, 135, 192, 18, 80, 225, 47, 0, 0],
            &[0, 0, 0, 0, 165, 222, 246, 60, 0, 0, 0],
            &[0, 0, 0, 0, 12, 115, 77, 0, 0, 0, 0],
            &[0, 30, 108, 108, 108, 108, 108, 108, 108, 95, 0],
            &[0, 61, 219, 219, 219, 252, 238, 219, 219, 192, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 3, 0, 0, 14, 13, 0, 0],
        ]),
        // letter: 'ť' / 0x165
        'ť' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 19, 85, 14, 0],
            &[0, 0, 0, 0, 0, 0, 0, 87, 233, 5, 0],
            &[0, 0, 0, 0, 118, 148, 0, 147, 117, 0, 0],
            &[0, 0, 0, 0, 167, 171, 0, 41, 6, 0, 0],
            &[0, 9, 111, 139, 230, 220, 147, 147, 147, 2, 0],
            &[0, 16, 159, 159, 233, 224, 159, 159, 159, 2, 0],
            &[0, 0, 0, 0, 196, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 189, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 246, 73, 11, 28, 14, 0],
            &[0, 0, 0, 0, 22, 192, 255, 255, 254, 60, 0],
            &[0, 0, 0, 0, 0, 0, 18, 35, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŧ' / 0x166
        'Ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 108, 108, 108, 108, 108, 108, 108, 95, 0],
            &[0, 61, 219, 219, 219, 252, 238, 219, 219, 192, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 28, 98, 98, 255, 231, 98, 88, 0, 0],
            &[0, 0, 58, 204, 204, 255, 255, 204, 185, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŧ' / 0x167
        'ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 118, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 167, 171, 0, 0, 0, 0, 0],
            &[0, 9, 111, 139, 230, 220, 147, 147, 147, 2, 0],
            &[0, 16, 159, 159, 233, 224, 159, 159, 159, 2, 0],
            &[0, 0, 0, 0, 196, 171, 0, 0, 0, 0, 0],
            &[0, 14, 137, 137, 255, 255, 137, 137, 137, 35, 0],
            &[0, 13, 133, 133, 255, 255, 133, 133, 133, 34, 0],
            &[0, 0, 0, 0, 196, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 189, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 246, 73, 11, 28, 14, 0],
            &[0, 0, 0, 0, 22, 192, 255, 255, 254, 60, 0],
            &[0, 0, 0, 0, 0, 0, 18, 35, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ũ' / 0x168
        'Ũ' => Some(&[
            &[0, 0, 34, 136, 88, 1, 39, 95, 0, 0, 0],
            &[0, 0, 206, 159, 216, 214, 225, 85, 0, 0, 0],
            &[0, 3, 97, 0, 4, 84, 71, 0, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 0, 104, 52, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 246, 122, 0, 0],
            &[0, 214, 164, 0, 0, 0, 9, 255, 112, 0, 0],
            &[0, 177, 209, 0, 0, 0, 52, 255, 72, 0, 0],
            &[0, 84, 255, 136, 41, 60, 203, 224, 7, 0, 0],
            &[0, 0, 121, 241, 255, 255, 206, 44, 0, 0, 0],
            &[0, 0, 0, 5, 31, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ũ' / 0x169
        'ũ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 175, 127, 13, 57, 115, 0, 0, 0],
            &[0, 0, 218, 120, 182, 242, 242, 64, 0, 0, 0],
            &[0, 3, 62, 0, 0, 46, 36, 0, 0, 0, 0],
            &[0, 117, 94, 0, 0, 0, 17, 147, 47, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 41, 255, 81, 0, 0],
            &[0, 191, 183, 0, 0, 0, 91, 255, 81, 0, 0],
            &[0, 139, 248, 70, 10, 51, 214, 255, 81, 0, 0],
            &[0, 19, 191, 255, 255, 230, 74, 235, 81, 0, 0],
            &[0, 0, 0, 16, 30, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ū' / 0x16a
        'Ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 223, 223, 223, 223, 37, 0, 0, 0],
            &[0, 0, 39, 67, 67, 67, 67, 11, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 0, 104, 52, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 246, 122, 0, 0],
            &[0, 214, 164, 0, 0, 0, 9, 255, 112, 0, 0],
            &[0, 177, 209, 0, 0, 0, 52, 255, 72, 0, 0],
            &[0, 84, 255, 136, 41, 60, 203, 224, 7, 0, 0],
            &[0, 0, 121, 241, 255, 255, 206, 44, 0, 0, 0],
            &[0, 0, 0, 5, 31, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ū' / 0x16b
        'ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 6, 6, 6, 6, 1, 0, 0, 0],
            &[0, 0, 149, 255, 255, 255, 255, 42, 0, 0, 0],
            &[0, 0, 16, 28, 28, 28, 28, 4, 0, 0, 0],
            &[0, 117, 94, 0, 0, 0, 17, 147, 47, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 41, 255, 81, 0, 0],
            &[0, 191, 183, 0, 0, 0, 91, 255, 81, 0, 0],
            &[0, 139, 248, 70, 10, 51, 214, 255, 81, 0, 0],
            &[0, 19, 191, 255, 255, 230, 74, 235, 81, 0, 0],
            &[0, 0, 0, 16, 30, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => Some(&[
            &[0, 0, 109, 71, 0, 0, 141, 36, 0, 0, 0],
            &[0, 0, 69, 236, 155, 188, 207, 6, 0, 0, 0],
            &[0, 0, 0, 42, 104, 86, 8, 0, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 0, 104, 52, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 246, 122, 0, 0],
            &[0, 214, 164, 0, 0, 0, 9, 255, 112, 0, 0],
            &[0, 177, 209, 0, 0, 0, 52, 255, 72, 0, 0],
            &[0, 84, 255, 136, 41, 60, 203, 224, 7, 0, 0],
            &[0, 0, 121, 241, 255, 255, 206, 44, 0, 0, 0],
            &[0, 0, 0, 5, 31, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŭ' / 0x16d
        'ŭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 97, 0, 3, 179, 41, 0, 0, 0],
            &[0, 0, 49, 235, 194, 224, 176, 2, 0, 0, 0],
            &[0, 0, 0, 17, 66, 47, 0, 0, 0, 0, 0],
            &[0, 117, 94, 0, 0, 0, 17, 147, 47, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 41, 255, 81, 0, 0],
            &[0, 191, 183, 0, 0, 0, 91, 255, 81, 0, 0],
            &[0, 139, 248, 70, 10, 51, 214, 255, 81, 0, 0],
            &[0, 19, 191, 255, 255, 230, 74, 235, 81, 0, 0],
            &[0, 0, 0, 16, 30, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ů' / 0x16e
        'Ů' => Some(&[
            &[0, 0, 0, 229, 22, 125, 128, 0, 0, 0, 0],
            &[0, 0, 0, 195, 137, 199, 90, 0, 0, 0, 0],
            &[0, 0, 0, 18, 102, 72, 0, 0, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 0, 104, 52, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 246, 122, 0, 0],
            &[0, 214, 164, 0, 0, 0, 9, 255, 112, 0, 0],
            &[0, 177, 209, 0, 0, 0, 52, 255, 72, 0, 0],
            &[0, 84, 255, 136, 41, 60, 203, 224, 7, 0, 0],
            &[0, 0, 121, 241, 255, 255, 206, 44, 0, 0, 0],
            &[0, 0, 0, 5, 31, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 214, 191, 27, 0, 0, 0, 0],
        ]),
        // letter: 'ů' / 0x16f
        'ů' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 219, 218, 44, 0, 0, 0, 0],
            &[0, 0, 0, 231, 12, 115, 131, 0, 0, 0, 0],
            &[0, 0, 0, 174, 173, 217, 71, 0, 0, 0, 0],
            &[0, 0, 0, 4, 64, 36, 0, 0, 0, 0, 0],
            &[0, 117, 94, 0, 0, 0, 17, 147, 47, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 41, 255, 81, 0, 0],
            &[0, 191, 183, 0, 0, 0, 91, 255, 81, 0, 0],
            &[0, 139, 248, 70, 10, 51, 214, 255, 81, 0, 0],
            &[0, 19, 191, 255, 255, 230, 74, 235, 81, 0, 0],
            &[0, 0, 0, 16, 30, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ű' / 0x170
        'Ű' => Some(&[
            &[0, 0, 0, 176, 208, 11, 217, 169, 0, 0, 0],
            &[0, 0, 67, 228, 31, 114, 201, 11, 0, 0, 0],
            &[0, 0, 82, 33, 0, 101, 14, 0, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 0, 104, 52, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 246, 122, 0, 0],
            &[0, 214, 164, 0, 0, 0, 9, 255, 112, 0, 0],
            &[0, 177, 209, 0, 0, 0, 52, 255, 72, 0, 0],
            &[0, 84, 255, 136, 41, 60, 203, 224, 7, 0, 0],
            &[0, 0, 121, 241, 255, 255, 206, 44, 0, 0, 0],
            &[0, 0, 0, 5, 31, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 26, 4, 15, 26, 0, 0, 0],
        ]),
        // letter: 'ű' / 0x171
        'ű' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 65, 10, 40, 63, 0, 0, 0],
            &[0, 0, 0, 196, 185, 13, 231, 139, 0, 0, 0],
            &[0, 0, 91, 207, 15, 138, 173, 3, 0, 0, 0],
            &[0, 0, 58, 15, 0, 69, 3, 0, 0, 0, 0],
            &[0, 117, 94, 0, 0, 0, 17, 147, 47, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 41, 255, 81, 0, 0],
            &[0, 191, 183, 0, 0, 0, 91, 255, 81, 0, 0],
            &[0, 139, 248, 70, 10, 51, 214, 255, 81, 0, 0],
            &[0, 19, 191, 255, 255, 230, 74, 235, 81, 0, 0],
            &[0, 0, 0, 16, 30, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ų' / 0x172
        'Ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 95, 60, 0, 0, 0, 0, 104, 52, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 245, 122, 0, 0],
            &[0, 225, 143, 0, 0, 0, 0, 246, 122, 0, 0],
            &[0, 214, 164, 0, 0, 0, 9, 255, 112, 0, 0],
            &[0, 177, 209, 0, 0, 0, 52, 255, 72, 0, 0],
            &[0, 84, 255, 136, 41, 60, 203, 224, 7, 0, 0],
            &[0, 0, 121, 241, 255, 255, 249, 44, 0, 0, 0],
            &[0, 0, 0, 5, 162, 183, 8, 0, 0, 0, 0],
            &[0, 0, 0, 45, 245, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 126, 39, 0, 0, 0, 0],
        ]),
        // letter: 'ų' / 0x173
        'ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 117, 94, 0, 0, 0, 17, 147, 47, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 30, 255, 81, 0, 0],
            &[0, 204, 163, 0, 0, 0, 41, 255, 81, 0, 0],
            &[0, 191, 183, 0, 0, 0, 91, 255, 81, 0, 0],
            &[0, 139, 248, 70, 10, 51, 214, 255, 81, 0, 0],
            &[0, 19, 191, 255, 255, 230, 74, 255, 84, 0, 0],
            &[0, 0, 0, 16, 30, 3, 76, 193, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 218, 67, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 46, 124, 72, 0, 0],
        ]),
        // letter: 'Ŵ' / 0x174
        'Ŵ' => Some(&[
            &[0, 0, 0, 94, 253, 222, 15, 0, 0, 0, 0],
            &[0, 0, 58, 231, 59, 156, 189, 7, 0, 0, 0],
            &[0, 0, 96, 38, 0, 0, 89, 49, 0, 0, 0],
            &[62, 81, 0, 0, 0, 0, 0, 15, 108, 19, 0],
            &[127, 208, 0, 0, 0, 0, 0, 53, 255, 27, 0],
            &[98, 233, 0, 0, 0, 0, 0, 74, 253, 4, 0],
            &[69, 253, 3, 0, 138, 84, 0, 96, 231, 0, 0],
            &[41, 255, 25, 31, 252, 194, 0, 119, 205, 0, 0],
            &[12, 255, 50, 90, 187, 245, 7, 141, 180, 0, 0],
            &[0, 239, 73, 150, 124, 210, 61, 161, 154, 0, 0],
            &[0, 210, 94, 209, 66, 150, 122, 180, 128, 0, 0],
            &[0, 182, 124, 248, 11, 88, 182, 195, 102, 0, 0],
            &[0, 153, 190, 203, 0, 27, 237, 209, 76, 0, 0],
            &[0, 124, 246, 144, 0, 0, 221, 246, 50, 0, 0],
            &[0, 96, 255, 84, 0, 0, 159, 255, 25, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 14, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŵ' / 0x175
        'ŵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 64, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 120, 240, 239, 33, 0, 0, 0, 0],
            &[0, 0, 82, 220, 36, 113, 213, 21, 0, 0, 0],
            &[0, 0, 67, 20, 0, 0, 51, 38, 0, 0, 0],
            &[102, 94, 0, 0, 0, 0, 0, 1, 145, 46, 0],
            &[146, 194, 0, 0, 0, 0, 0, 31, 255, 46, 0],
            &[105, 231, 0, 33, 245, 176, 0, 71, 252, 7, 0],
            &[64, 255, 13, 107, 225, 245, 7, 112, 218, 0, 0],
            &[23, 255, 50, 180, 128, 234, 67, 152, 176, 0, 0],
            &[0, 237, 89, 244, 59, 171, 136, 187, 134, 0, 0],
            &[0, 196, 159, 242, 4, 106, 198, 212, 92, 0, 0],
            &[0, 155, 226, 179, 0, 40, 246, 235, 50, 0, 0],
            &[0, 114, 255, 113, 0, 0, 230, 254, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŷ' / 0x176
        'Ŷ' => Some(&[
            &[0, 0, 0, 94, 253, 222, 15, 0, 0, 0, 0],
            &[0, 0, 58, 231, 59, 156, 189, 7, 0, 0, 0],
            &[0, 0, 96, 38, 0, 0, 89, 49, 0, 0, 0],
            &[36, 108, 26, 0, 0, 0, 0, 64, 100, 0, 0],
            &[12, 235, 146, 0, 0, 0, 6, 230, 145, 0, 0],
            &[0, 117, 245, 19, 0, 0, 100, 246, 24, 0, 0],
            &[0, 10, 232, 127, 0, 2, 217, 141, 0, 0, 0],
            &[0, 0, 113, 235, 10, 82, 244, 22, 0, 0, 0],
            &[0, 0, 8, 229, 108, 201, 136, 0, 0, 0, 0],
            &[0, 0, 0, 108, 241, 242, 19, 0, 0, 0, 0],
            &[0, 0, 0, 7, 239, 144, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 14, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŷ' / 0x177
        'ŷ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 64, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 124, 239, 237, 31, 0, 0, 0],
            &[0, 0, 0, 86, 219, 33, 117, 211, 19, 0, 0],
            &[0, 0, 0, 68, 18, 0, 0, 52, 37, 0, 0],
            &[0, 23, 147, 36, 0, 0, 0, 0, 103, 111, 0],
            &[0, 0, 212, 144, 0, 0, 0, 6, 240, 118, 0],
            &[0, 0, 106, 239, 9, 0, 0, 77, 252, 26, 0],
            &[0, 0, 13, 242, 96, 0, 0, 162, 184, 0, 0],
            &[0, 0, 0, 149, 200, 0, 7, 241, 89, 0, 0],
            &[0, 0, 0, 43, 255, 47, 77, 241, 9, 0, 0],
            &[0, 0, 0, 0, 192, 141, 160, 155, 0, 0, 0],
            &[0, 0, 0, 0, 86, 224, 236, 60, 0, 0, 0],
            &[0, 0, 0, 0, 5, 230, 220, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 216, 126, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 250, 28, 0, 0, 0, 0],
            &[0, 0, 215, 226, 157, 12, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ÿ' / 0x178
        'Ÿ' => Some(&[
            &[0, 0, 101, 89, 0, 7, 144, 39, 0, 0, 0],
            &[0, 0, 202, 185, 0, 33, 255, 99, 0, 0, 0],
            &[0, 0, 15, 11, 0, 0, 25, 1, 0, 0, 0],
            &[36, 108, 26, 0, 0, 0, 0, 64, 100, 0, 0],
            &[12, 235, 146, 0, 0, 0, 6, 230, 145, 0, 0],
            &[0, 117, 245, 19, 0, 0, 100, 246, 24, 0, 0],
            &[0, 10, 232, 127, 0, 2, 217, 141, 0, 0, 0],
            &[0, 0, 113, 235, 10, 82, 244, 22, 0, 0, 0],
            &[0, 0, 8, 229, 108, 201, 136, 0, 0, 0, 0],
            &[0, 0, 0, 108, 241, 242, 19, 0, 0, 0, 0],
            &[0, 0, 0, 7, 239, 144, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ź' / 0x179
        'Ź' => Some(&[
            &[0, 0, 0, 0, 0, 125, 248, 56, 0, 0, 0],
            &[0, 0, 0, 0, 40, 241, 81, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 60, 0, 0, 0, 0, 0],
            &[0, 17, 108, 108, 108, 108, 108, 108, 8, 0, 0],
            &[0, 33, 210, 210, 210, 210, 231, 254, 16, 0, 0],
            &[0, 0, 0, 0, 0, 1, 200, 164, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 247, 30, 0, 0, 0],
            &[0, 0, 0, 0, 8, 224, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 234, 14, 0, 0, 0, 0],
            &[0, 0, 0, 22, 241, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 216, 4, 0, 0, 0, 0, 0],
            &[0, 0, 42, 252, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 182, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 69, 255, 121, 71, 71, 71, 71, 5, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 26, 14, 0, 0, 0],
        ]),
        // letter: 'ź' / 0x17a
        'ź' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 65, 33, 0, 0, 0],
            &[0, 0, 0, 0, 0, 156, 233, 31, 0, 0, 0],
            &[0, 0, 0, 0, 67, 234, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 31, 0, 0, 0, 0, 0],
            &[0, 44, 147, 147, 147, 147, 147, 147, 4, 0, 0],
            &[0, 41, 135, 135, 135, 135, 220, 239, 4, 0, 0],
            &[0, 0, 0, 0, 0, 72, 252, 77, 0, 0, 0],
            &[0, 0, 0, 0, 31, 237, 130, 0, 0, 0, 0],
            &[0, 0, 0, 7, 203, 183, 2, 0, 0, 0, 0],
            &[0, 0, 0, 153, 222, 18, 0, 0, 0, 0, 0],
            &[0, 0, 96, 247, 49, 0, 0, 0, 0, 0, 0],
            &[0, 48, 246, 147, 55, 55, 55, 55, 8, 0, 0],
            &[0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ż' / 0x17b
        'Ż' => Some(&[
            &[0, 0, 0, 0, 195, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 108, 108, 108, 108, 108, 108, 8, 0, 0],
            &[0, 33, 210, 210, 210, 210, 231, 254, 16, 0, 0],
            &[0, 0, 0, 0, 0, 1, 200, 164, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 247, 30, 0, 0, 0],
            &[0, 0, 0, 0, 8, 224, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 234, 14, 0, 0, 0, 0],
            &[0, 0, 0, 22, 241, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 216, 4, 0, 0, 0, 0, 0],
            &[0, 0, 42, 252, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 182, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 69, 255, 121, 71, 71, 71, 71, 5, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ż' / 0x17c
        'ż' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 229, 203, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 44, 147, 147, 147, 147, 147, 147, 4, 0, 0],
            &[0, 41, 135, 135, 135, 135, 220, 239, 4, 0, 0],
            &[0, 0, 0, 0, 0, 72, 252, 77, 0, 0, 0],
            &[0, 0, 0, 0, 31, 237, 130, 0, 0, 0, 0],
            &[0, 0, 0, 7, 203, 183, 2, 0, 0, 0, 0],
            &[0, 0, 0, 153, 222, 18, 0, 0, 0, 0, 0],
            &[0, 0, 96, 247, 49, 0, 0, 0, 0, 0, 0],
            &[0, 48, 246, 147, 55, 55, 55, 55, 8, 0, 0],
            &[0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ž' / 0x17d
        'Ž' => Some(&[
            &[0, 0, 88, 217, 40, 48, 221, 82, 0, 0, 0],
            &[0, 0, 0, 117, 234, 240, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 104, 99, 0, 0, 0, 0, 0],
            &[0, 17, 108, 108, 108, 108, 108, 108, 8, 0, 0],
            &[0, 33, 210, 210, 210, 210, 231, 254, 16, 0, 0],
            &[0, 0, 0, 0, 0, 1, 200, 164, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 247, 30, 0, 0, 0],
            &[0, 0, 0, 0, 8, 224, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 234, 14, 0, 0, 0, 0],
            &[0, 0, 0, 22, 241, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 216, 4, 0, 0, 0, 0, 0],
            &[0, 0, 42, 252, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 182, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 69, 255, 121, 71, 71, 71, 71, 5, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 8, 0, 0, 9, 18, 0, 0, 0],
        ]),
        // letter: 'ž' / 0x17e
        'ž' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 25, 0, 0, 32, 43, 0, 0, 0],
            &[0, 0, 67, 229, 60, 82, 229, 50, 0, 0, 0],
            &[0, 0, 0, 93, 246, 249, 66, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 59, 0, 0, 0, 0, 0],
            &[0, 44, 147, 147, 147, 147, 147, 147, 4, 0, 0],
            &[0, 41, 135, 135, 135, 135, 220, 239, 4, 0, 0],
            &[0, 0, 0, 0, 0, 72, 252, 77, 0, 0, 0],
            &[0, 0, 0, 0, 31, 237, 130, 0, 0, 0, 0],
            &[0, 0, 0, 7, 203, 183, 2, 0, 0, 0, 0],
            &[0, 0, 0, 153, 222, 18, 0, 0, 0, 0, 0],
            &[0, 0, 96, 247, 49, 0, 0, 0, 0, 0, 0],
            &[0, 48, 246, 147, 55, 55, 55, 55, 8, 0, 0],
            &[0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ſ' / 0x17f
        'ſ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 54, 47, 9, 0, 0, 0],
            &[0, 0, 0, 139, 252, 247, 251, 157, 0, 0, 0],
            &[0, 0, 62, 255, 110, 1, 3, 13, 0, 0, 0],
            &[0, 0, 110, 252, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 241, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 241, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 241, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 241, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 241, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 241, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 241, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 241, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 241, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        _ => None,
    }
}
