//! Module for letters with the font weight light and size 22.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 22;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 12;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight light and font size 19px
#[inline]
pub const fn get_char(c: char) -> Option<&'static [&'static [u8]]> {
    match c {
        // letter: ' ' / 0x20
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '!' / 0x21
        '!' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 136, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 237, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 233, 84, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 228, 79, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 224, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 219, 71, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 215, 66, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 202, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 85, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 161, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 252, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '"' / 0x22
        '"' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 62, 101, 0, 43, 123, 0, 0, 0, 0],
            &[0, 0, 0, 101, 169, 0, 67, 208, 0, 0, 0, 0],
            &[0, 0, 0, 89, 157, 0, 55, 197, 0, 0, 0, 0],
            &[0, 0, 0, 77, 145, 0, 43, 185, 0, 0, 0, 0],
            &[0, 0, 0, 65, 133, 0, 31, 173, 0, 0, 0, 0],
            &[0, 0, 0, 19, 42, 0, 7, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '#' / 0x23
        '#' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 20, 0, 15, 111, 0, 0, 0],
            &[0, 0, 0, 0, 228, 6, 0, 61, 163, 0, 0, 0],
            &[0, 0, 0, 20, 215, 0, 0, 105, 120, 0, 0, 0],
            &[0, 0, 0, 64, 172, 0, 0, 149, 77, 0, 0, 0],
            &[0, 115, 123, 174, 195, 123, 123, 218, 146, 123, 46, 0],
            &[0, 89, 95, 197, 140, 95, 96, 240, 95, 95, 35, 0],
            &[0, 0, 0, 197, 35, 0, 31, 201, 0, 0, 0, 0],
            &[0, 0, 0, 230, 2, 0, 75, 158, 0, 0, 0, 0],
            &[8, 13, 39, 208, 13, 13, 125, 123, 13, 10, 0, 0],
            &[122, 209, 226, 235, 209, 209, 242, 219, 209, 156, 0, 0],
            &[0, 0, 116, 116, 0, 0, 207, 25, 0, 0, 0, 0],
            &[0, 0, 159, 72, 0, 4, 228, 0, 0, 0, 0, 0],
            &[0, 0, 202, 27, 0, 42, 190, 0, 0, 0, 0, 0],
            &[0, 1, 227, 0, 0, 88, 145, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '$' / 0x24
        '$' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 83, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 203, 32, 0, 0, 0, 0, 0],
            &[0, 0, 32, 177, 246, 249, 234, 240, 167, 11, 0, 0],
            &[0, 5, 220, 135, 11, 198, 15, 22, 89, 0, 0, 0],
            &[0, 62, 228, 0, 0, 198, 15, 0, 0, 0, 0, 0],
            &[0, 63, 227, 0, 0, 198, 15, 0, 0, 0, 0, 0],
            &[0, 7, 225, 124, 1, 198, 15, 0, 0, 0, 0, 0],
            &[0, 0, 37, 193, 219, 233, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 225, 242, 176, 47, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 29, 119, 242, 51, 0, 0],
            &[0, 0, 0, 0, 0, 198, 15, 0, 140, 156, 0, 0],
            &[0, 0, 0, 0, 0, 198, 15, 0, 108, 165, 0, 0],
            &[0, 7, 6, 0, 0, 198, 15, 12, 204, 98, 0, 0],
            &[0, 46, 241, 180, 146, 226, 167, 229, 143, 0, 0, 0],
            &[0, 0, 23, 79, 106, 225, 99, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '%' / 0x25
        '%' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[1, 103, 169, 140, 19, 0, 0, 0, 80, 65, 0, 0],
            &[121, 158, 30, 87, 201, 1, 0, 10, 223, 23, 0, 0],
            &[221, 21, 0, 0, 190, 51, 0, 118, 139, 0, 0, 0],
            &[241, 0, 0, 0, 162, 78, 12, 223, 21, 0, 0, 0],
            &[218, 23, 0, 0, 193, 51, 122, 135, 0, 0, 0, 0],
            &[115, 160, 29, 90, 204, 16, 224, 18, 0, 0, 0, 0],
            &[0, 101, 170, 141, 20, 126, 130, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 224, 16, 9, 53, 30, 0, 0],
            &[0, 0, 0, 0, 131, 126, 41, 209, 146, 190, 129, 0],
            &[0, 0, 0, 19, 224, 14, 173, 75, 0, 1, 208, 36],
            &[0, 0, 0, 135, 122, 0, 225, 14, 0, 0, 152, 87],
            &[0, 0, 21, 223, 12, 0, 220, 18, 0, 0, 156, 83],
            &[0, 0, 139, 117, 0, 0, 159, 93, 0, 8, 217, 29],
            &[0, 24, 222, 10, 0, 0, 27, 199, 172, 208, 109, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 28, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '&' / 0x26
        '&' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 79, 176, 176, 76, 0, 0, 0, 0, 0],
            &[0, 0, 63, 223, 61, 62, 225, 56, 0, 0, 0, 0],
            &[0, 0, 145, 116, 0, 0, 126, 135, 0, 0, 0, 0],
            &[0, 0, 141, 120, 0, 0, 130, 136, 0, 0, 0, 0],
            &[0, 0, 79, 192, 0, 17, 223, 59, 0, 0, 0, 0],
            &[0, 0, 2, 212, 89, 195, 137, 0, 0, 0, 0, 0],
            &[0, 0, 0, 110, 255, 133, 0, 0, 0, 0, 0, 0],
            &[0, 0, 98, 223, 174, 190, 3, 0, 0, 57, 27, 0],
            &[0, 57, 236, 38, 0, 172, 139, 0, 0, 216, 43, 0],
            &[0, 166, 114, 0, 0, 14, 217, 85, 38, 230, 1, 0],
            &[0, 203, 70, 0, 0, 0, 45, 233, 195, 124, 0, 0],
            &[0, 184, 103, 0, 0, 0, 0, 164, 255, 31, 0, 0],
            &[0, 95, 223, 34, 0, 1, 95, 235, 218, 175, 1, 0],
            &[0, 0, 132, 240, 211, 229, 196, 50, 7, 204, 121, 0],
            &[0, 0, 0, 11, 41, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ''' / 0x27
        '\'' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 126, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 214, 61, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 202, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 179, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '(' / 0x28
        '(' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 59, 95, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 218, 36, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 139, 136, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 235, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 89, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 151, 130, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 205, 76, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 228, 52, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 247, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 239, 42, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 220, 61, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 184, 101, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 122, 161, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 232, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 192, 91, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 211, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 143, 132, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ')' / 0x29
        ')' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 37, 117, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 189, 89, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 233, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 183, 101, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 249, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 227, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 187, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 197, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 216, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 249, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 144, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 227, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 113, 162, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 37, 215, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '*' / 0x2a
        '*' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 42, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 127, 88, 0, 0, 0, 0, 0],
            &[0, 0, 145, 81, 12, 115, 77, 21, 98, 135, 0, 0],
            &[0, 3, 96, 153, 203, 221, 216, 204, 151, 90, 0, 0],
            &[0, 0, 0, 0, 32, 196, 204, 19, 0, 0, 0, 0],
            &[0, 0, 0, 7, 203, 60, 97, 183, 2, 0, 0, 0],
            &[0, 0, 0, 143, 159, 0, 1, 196, 116, 0, 0, 0],
            &[0, 0, 0, 26, 19, 0, 0, 33, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '+' / 0x2b
        '+' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 188, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 188, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 188, 44, 0, 0, 0, 0, 0],
            &[0, 23, 41, 41, 41, 199, 79, 41, 41, 40, 0, 0],
            &[0, 110, 191, 191, 191, 239, 202, 191, 191, 186, 0, 0],
            &[0, 0, 0, 0, 0, 188, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 188, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 188, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 128, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ',' / 0x2c
        ',' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 65, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 185, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 232, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 250, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 177, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 54, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '-' / 0x2d
        '-' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 179, 179, 179, 179, 179, 98, 0, 0, 0],
            &[0, 0, 11, 73, 73, 73, 73, 73, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '.' / 0x2e
        '.' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 163, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 253, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '/' / 0x2f
        '/' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 54, 102, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 171, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 241, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 166, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 207, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 226, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 145, 131, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 235, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 193, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 181, 95, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 239, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 119, 157, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 216, 60, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 218, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '0' / 0x30
        '0' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 33, 146, 186, 168, 88, 0, 0, 0, 0],
            &[0, 0, 46, 237, 131, 67, 89, 210, 144, 0, 0, 0],
            &[0, 0, 199, 104, 0, 0, 0, 71, 255, 50, 0, 0],
            &[0, 38, 235, 5, 0, 0, 1, 201, 217, 144, 0, 0],
            &[0, 92, 175, 0, 0, 0, 97, 193, 65, 197, 0, 0],
            &[0, 128, 138, 0, 0, 14, 229, 46, 29, 232, 0, 0],
            &[0, 142, 123, 0, 0, 142, 146, 0, 15, 246, 0, 0],
            &[0, 147, 118, 0, 41, 229, 16, 0, 11, 250, 0, 0],
            &[0, 133, 129, 0, 187, 99, 0, 0, 21, 240, 0, 0],
            &[0, 110, 156, 81, 202, 1, 0, 0, 48, 213, 0, 0],
            &[0, 60, 217, 221, 54, 0, 0, 0, 98, 168, 0, 0],
            &[0, 6, 255, 202, 0, 0, 0, 0, 187, 92, 0, 0],
            &[0, 0, 108, 214, 27, 0, 2, 117, 214, 6, 0, 0],
            &[0, 0, 0, 122, 239, 213, 234, 196, 33, 0, 0, 0],
            &[0, 0, 0, 0, 6, 37, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '1' / 0x31
        '1' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 66, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 164, 220, 234, 0, 0, 0, 0, 0],
            &[0, 0, 87, 232, 122, 36, 234, 0, 0, 0, 0, 0],
            &[0, 0, 55, 40, 0, 35, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 165, 211, 255, 255, 220, 208, 110, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '2' / 0x32
        '2' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 76, 155, 185, 170, 93, 1, 0, 0, 0],
            &[0, 11, 198, 199, 103, 66, 97, 213, 164, 0, 0, 0],
            &[0, 0, 62, 0, 0, 0, 0, 21, 239, 52, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 185, 102, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 182, 100, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 234, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 96, 213, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 233, 67, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 214, 112, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 198, 143, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 193, 149, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 189, 153, 1, 0, 0, 0, 0, 0, 0],
            &[0, 7, 185, 159, 3, 1, 1, 1, 1, 1, 0, 0],
            &[0, 73, 255, 255, 255, 255, 255, 255, 255, 255, 32, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '3' / 0x33
        '3' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 85, 158, 186, 175, 112, 9, 0, 0, 0],
            &[0, 16, 213, 196, 100, 66, 88, 192, 203, 8, 0, 0],
            &[0, 0, 55, 0, 0, 0, 0, 6, 217, 92, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 160, 130, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 190, 101, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 224, 15, 0, 0],
            &[0, 0, 0, 59, 138, 148, 193, 158, 28, 0, 0, 0],
            &[0, 0, 0, 50, 118, 125, 156, 210, 127, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 10, 192, 127, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 70, 221, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 57, 235, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 122, 190, 0, 0],
            &[0, 55, 87, 10, 0, 0, 1, 93, 243, 66, 0, 0],
            &[0, 44, 180, 245, 218, 213, 243, 203, 68, 0, 0, 0],
            &[0, 0, 0, 4, 29, 40, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '4' / 0x34
        '4' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 164, 46, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 182, 235, 72, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 180, 188, 72, 0, 0, 0],
            &[0, 0, 0, 0, 18, 224, 35, 194, 72, 0, 0, 0],
            &[0, 0, 0, 0, 159, 121, 0, 198, 72, 0, 0, 0],
            &[0, 0, 0, 66, 209, 5, 0, 199, 72, 0, 0, 0],
            &[0, 0, 8, 217, 55, 0, 0, 199, 72, 0, 0, 0],
            &[0, 0, 135, 147, 0, 0, 0, 199, 72, 0, 0, 0],
            &[0, 46, 222, 13, 0, 0, 0, 199, 72, 0, 0, 0],
            &[0, 199, 155, 84, 84, 84, 84, 218, 133, 84, 30, 0],
            &[0, 151, 158, 158, 158, 158, 158, 234, 185, 158, 56, 0],
            &[0, 0, 0, 0, 0, 0, 0, 199, 72, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 199, 72, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 199, 72, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '5' / 0x35
        '5' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 144, 144, 144, 144, 144, 144, 3, 0, 0],
            &[0, 0, 110, 194, 107, 107, 107, 107, 107, 2, 0, 0],
            &[0, 0, 132, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 153, 108, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 175, 85, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 197, 66, 34, 47, 24, 0, 0, 0, 0, 0],
            &[0, 0, 198, 230, 198, 193, 232, 204, 55, 0, 0, 0],
            &[0, 0, 5, 0, 0, 0, 2, 108, 241, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 172, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 111, 176, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 126, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 199, 110, 0, 0],
            &[0, 39, 91, 7, 0, 0, 18, 155, 219, 13, 0, 0],
            &[0, 29, 176, 246, 222, 221, 246, 171, 26, 0, 0, 0],
            &[0, 0, 0, 3, 30, 37, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '6' / 0x36
        '6' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 136, 181, 193, 149, 0, 0, 0],
            &[0, 0, 0, 106, 230, 122, 62, 51, 74, 0, 0, 0],
            &[0, 0, 68, 224, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 81, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 233, 3, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 178, 0, 47, 108, 112, 61, 0, 0, 0, 0],
            &[0, 126, 146, 146, 184, 124, 135, 208, 184, 9, 0, 0],
            &[0, 141, 225, 94, 0, 0, 0, 4, 181, 136, 0, 0],
            &[0, 140, 192, 0, 0, 0, 0, 0, 59, 225, 0, 0],
            &[0, 119, 157, 0, 0, 0, 0, 0, 22, 253, 0, 0],
            &[0, 72, 198, 0, 0, 0, 0, 0, 41, 238, 0, 0],
            &[0, 8, 236, 44, 0, 0, 0, 0, 108, 179, 0, 0],
            &[0, 0, 107, 213, 37, 0, 0, 60, 233, 57, 0, 0],
            &[0, 0, 0, 113, 236, 215, 222, 221, 78, 0, 0, 0],
            &[0, 0, 0, 0, 6, 37, 30, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '7' / 0x37
        '7' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 144, 144, 144, 144, 144, 144, 144, 141, 0, 0],
            &[0, 23, 107, 107, 107, 107, 107, 107, 149, 222, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 155, 123, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 15, 240, 25, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 108, 173, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 70, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 62, 222, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 167, 121, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 244, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 120, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 223, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 74, 220, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 179, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 247, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '8' / 0x38
        '8' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 71, 159, 187, 167, 87, 1, 0, 0, 0],
            &[0, 0, 128, 218, 96, 58, 87, 205, 156, 0, 0, 0],
            &[0, 18, 248, 32, 0, 0, 0, 18, 244, 36, 0, 0],
            &[0, 47, 232, 0, 0, 0, 0, 0, 211, 63, 0, 0],
            &[0, 13, 243, 37, 0, 0, 0, 26, 239, 21, 0, 0],
            &[0, 0, 90, 225, 57, 0, 46, 213, 103, 0, 0, 0],
            &[0, 0, 0, 65, 229, 210, 224, 61, 0, 0, 0, 0],
            &[0, 0, 27, 171, 188, 103, 208, 176, 32, 0, 0, 0],
            &[0, 17, 222, 100, 0, 0, 1, 100, 228, 25, 0, 0],
            &[0, 118, 174, 0, 0, 0, 0, 0, 147, 139, 0, 0],
            &[0, 158, 124, 0, 0, 0, 0, 0, 100, 178, 0, 0],
            &[0, 132, 163, 0, 0, 0, 0, 0, 146, 147, 0, 0],
            &[0, 34, 239, 88, 0, 0, 0, 85, 239, 41, 0, 0],
            &[0, 0, 55, 204, 226, 198, 227, 200, 56, 0, 0, 0],
            &[0, 0, 0, 0, 22, 42, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '9' / 0x39
        '9' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 73, 164, 191, 165, 69, 0, 0, 0, 0],
            &[0, 0, 130, 220, 93, 54, 89, 217, 119, 0, 0, 0],
            &[0, 45, 239, 21, 0, 0, 0, 22, 232, 40, 0, 0],
            &[0, 126, 154, 0, 0, 0, 0, 0, 135, 139, 0, 0],
            &[0, 154, 120, 0, 0, 0, 0, 0, 69, 199, 0, 0],
            &[0, 143, 138, 0, 0, 0, 0, 0, 64, 231, 0, 0],
            &[0, 82, 212, 3, 0, 0, 0, 0, 167, 241, 0, 0],
            &[0, 3, 192, 171, 45, 14, 49, 166, 137, 233, 0, 0],
            &[0, 0, 9, 126, 201, 216, 182, 63, 67, 207, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 116, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 200, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 82, 224, 7, 0, 0],
            &[0, 0, 1, 0, 0, 5, 96, 236, 62, 0, 0, 0],
            &[0, 0, 153, 212, 207, 238, 190, 51, 0, 0, 0, 0],
            &[0, 0, 6, 37, 42, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ':' / 0x3a
        ':' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 42, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 254, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 141, 50, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 163, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 253, 117, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ';' / 0x3b
        ';' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 185, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 57, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 210, 140, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 249, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 238, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 153, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 41, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '<' / 0x3c
        '<' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 37, 165, 185, 0, 0],
            &[0, 0, 0, 0, 0, 33, 158, 207, 84, 1, 0, 0],
            &[0, 0, 0, 28, 152, 206, 85, 1, 0, 0, 0, 0],
            &[0, 23, 145, 205, 86, 1, 0, 0, 0, 0, 0, 0],
            &[0, 85, 238, 111, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 135, 219, 121, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 125, 221, 134, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 115, 221, 133, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 7, 83, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '=' / 0x3d
        '=' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 16, 16, 16, 16, 16, 16, 16, 16, 1, 0],
            &[0, 158, 236, 236, 236, 236, 236, 236, 236, 236, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 86, 86, 86, 86, 86, 86, 86, 86, 6, 0],
            &[0, 114, 170, 170, 170, 170, 170, 170, 170, 170, 11, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '>' / 0x3e
        '>' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 89, 210, 86, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 161, 206, 80, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 162, 200, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 38, 164, 195, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 60, 206, 177, 0, 0],
            &[0, 0, 0, 0, 0, 73, 194, 184, 61, 0, 0, 0],
            &[0, 0, 2, 85, 205, 174, 52, 0, 0, 0, 0, 0],
            &[0, 60, 213, 164, 43, 0, 0, 0, 0, 0, 0, 0],
            &[0, 56, 35, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '?' / 0x3f
        '?' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 120, 164, 187, 165, 85, 0, 0, 0, 0],
            &[0, 0, 157, 149, 87, 62, 95, 213, 153, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 23, 244, 32, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 209, 68, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 239, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 113, 192, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 125, 213, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 155, 189, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 67, 213, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 123, 130, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 60, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 114, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 192, 197, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '@' / 0x40
        '@' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 79, 128, 128, 81, 3, 0, 0, 0],
            &[0, 0, 49, 215, 162, 101, 90, 152, 209, 32, 0, 0],
            &[0, 32, 223, 57, 0, 0, 0, 0, 64, 211, 6, 0],
            &[0, 173, 91, 0, 0, 0, 0, 0, 0, 137, 109, 0],
            &[27, 215, 2, 0, 64, 181, 200, 158, 24, 36, 192, 0],
            &[99, 142, 0, 54, 215, 45, 17, 212, 46, 0, 228, 0],
            &[147, 88, 0, 174, 87, 0, 0, 213, 32, 0, 208, 15],
            &[169, 66, 0, 233, 23, 0, 0, 228, 19, 0, 204, 23],
            &[178, 55, 0, 249, 6, 0, 3, 250, 8, 0, 221, 6],
            &[159, 72, 0, 226, 29, 0, 45, 255, 10, 11, 211, 0],
            &[131, 107, 0, 152, 132, 7, 163, 184, 78, 97, 137, 0],
            &[63, 177, 0, 16, 171, 212, 110, 22, 196, 169, 12, 0],
            &[4, 217, 43, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 76, 209, 22, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 223, 124, 69, 57, 88, 158, 94, 0, 0],
            &[0, 0, 0, 20, 113, 154, 165, 136, 69, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'A' / 0x41
        'A' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 153, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 244, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 127, 239, 12, 0, 0, 0, 0],
            &[0, 0, 0, 6, 237, 30, 178, 91, 0, 0, 0, 0],
            &[0, 0, 0, 78, 197, 0, 90, 181, 0, 0, 0, 0],
            &[0, 0, 0, 167, 110, 0, 12, 241, 19, 0, 0, 0],
            &[0, 0, 11, 242, 25, 0, 0, 172, 103, 0, 0, 0],
            &[0, 0, 90, 190, 0, 0, 0, 85, 192, 0, 0, 0],
            &[0, 0, 182, 255, 169, 169, 169, 179, 255, 28, 0, 0],
            &[0, 18, 255, 108, 88, 88, 88, 88, 212, 115, 0, 0],
            &[0, 102, 183, 0, 0, 0, 0, 0, 79, 204, 0, 0],
            &[0, 191, 96, 0, 0, 0, 0, 0, 7, 240, 38, 0],
            &[27, 246, 15, 0, 0, 0, 0, 0, 0, 161, 127, 0],
            &[114, 176, 0, 0, 0, 0, 0, 0, 0, 74, 216, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'B' / 0x42
        'B' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 140, 117, 77, 4, 0, 0, 0],
            &[0, 1, 255, 112, 103, 109, 130, 196, 227, 47, 0, 0],
            &[0, 1, 255, 15, 0, 0, 0, 0, 146, 190, 0, 0],
            &[0, 1, 255, 15, 0, 0, 0, 0, 59, 237, 0, 0],
            &[0, 1, 255, 15, 0, 0, 0, 0, 72, 214, 0, 0],
            &[0, 1, 255, 15, 0, 0, 0, 14, 194, 110, 0, 0],
            &[0, 1, 255, 159, 153, 156, 180, 203, 98, 0, 0, 0],
            &[0, 1, 255, 103, 94, 96, 113, 170, 199, 50, 0, 0],
            &[0, 1, 255, 15, 0, 0, 0, 0, 80, 233, 9, 0],
            &[0, 1, 255, 15, 0, 0, 0, 0, 0, 230, 65, 0],
            &[0, 1, 255, 15, 0, 0, 0, 0, 0, 229, 68, 0],
            &[0, 1, 255, 15, 0, 0, 0, 0, 49, 250, 23, 0],
            &[0, 1, 255, 15, 0, 1, 25, 90, 225, 132, 0, 0],
            &[0, 1, 255, 253, 252, 253, 239, 193, 90, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'C' / 0x43
        'C' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 134, 178, 183, 156, 89, 10, 0],
            &[0, 0, 0, 124, 237, 135, 75, 70, 104, 185, 55, 0],
            &[0, 0, 102, 225, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 238, 68, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 85, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 175, 115, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 156, 133, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 119, 177, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 241, 14, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 191, 141, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 232, 146, 22, 0, 0, 9, 73, 3, 0],
            &[0, 0, 0, 35, 167, 250, 230, 228, 249, 198, 5, 0],
            &[0, 0, 0, 0, 0, 5, 34, 35, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'D' / 0x44
        'D' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 144, 144, 142, 118, 73, 3, 0, 0, 0, 0],
            &[0, 108, 206, 103, 108, 133, 204, 226, 60, 0, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 89, 242, 47, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 126, 193, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 18, 251, 26, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 0, 207, 80, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 0, 184, 102, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 0, 177, 105, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 0, 196, 84, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 4, 242, 42, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 83, 218, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 26, 225, 86, 0, 0],
            &[0, 108, 173, 0, 5, 33, 108, 229, 133, 0, 0, 0],
            &[0, 108, 254, 252, 248, 222, 168, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'E' / 0x45
        'E' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 144, 144, 144, 144, 108, 0, 0],
            &[0, 1, 255, 122, 107, 107, 107, 107, 107, 80, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 167, 158, 158, 158, 158, 158, 58, 0, 0],
            &[0, 1, 255, 110, 94, 94, 94, 94, 94, 34, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 26, 1, 1, 1, 1, 1, 1, 0, 0],
            &[0, 1, 255, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'F' / 0x46
        'F' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 144, 144, 144, 144, 108, 0, 0],
            &[0, 1, 255, 122, 107, 107, 107, 107, 107, 80, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 253, 252, 252, 252, 252, 252, 97, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'G' / 0x47
        'G' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 96, 163, 190, 170, 116, 23, 0],
            &[0, 0, 0, 25, 212, 197, 98, 67, 87, 158, 111, 0],
            &[0, 0, 1, 202, 150, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 85, 224, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 167, 126, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 227, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 249, 39, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 255, 27, 0, 0, 3, 234, 234, 234, 197, 0],
            &[0, 0, 247, 42, 0, 0, 0, 23, 23, 79, 215, 0],
            &[0, 0, 220, 77, 0, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 155, 147, 0, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 55, 244, 30, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 0, 144, 219, 61, 0, 0, 1, 97, 215, 0],
            &[0, 0, 0, 0, 110, 226, 237, 216, 241, 219, 116, 0],
            &[0, 0, 0, 0, 0, 0, 23, 39, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'H' / 0x48
        'H' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 37, 121, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 224, 158, 158, 158, 158, 158, 183, 215, 0, 0],
            &[0, 108, 205, 99, 99, 99, 99, 99, 140, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'I' / 0x49
        'I' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 144, 144, 144, 144, 144, 120, 0, 0, 0],
            &[0, 0, 25, 68, 83, 231, 129, 74, 50, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 88, 213, 228, 253, 238, 219, 171, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'J' / 0x4a
        'J' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 60, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 186, 98, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 230, 64, 0, 0, 0],
            &[0, 0, 75, 5, 0, 1, 122, 226, 6, 0, 0, 0],
            &[0, 0, 172, 245, 211, 228, 222, 54, 0, 0, 0, 0],
            &[0, 0, 0, 15, 47, 39, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'K' / 0x4b
        'K' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 51, 134, 7, 0],
            &[0, 108, 173, 0, 0, 0, 0, 32, 230, 82, 0, 0],
            &[0, 108, 173, 0, 0, 0, 16, 214, 118, 0, 0, 0],
            &[0, 108, 173, 0, 0, 6, 191, 156, 0, 0, 0, 0],
            &[0, 108, 173, 0, 0, 163, 189, 5, 0, 0, 0, 0],
            &[0, 108, 173, 0, 130, 215, 16, 0, 0, 0, 0, 0],
            &[0, 108, 173, 90, 255, 103, 0, 0, 0, 0, 0, 0],
            &[0, 108, 208, 221, 100, 232, 20, 0, 0, 0, 0, 0],
            &[0, 108, 238, 46, 0, 147, 170, 0, 0, 0, 0, 0],
            &[0, 108, 173, 0, 0, 11, 220, 86, 0, 0, 0, 0],
            &[0, 108, 173, 0, 0, 0, 62, 233, 22, 0, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 145, 173, 0, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 10, 220, 89, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 61, 234, 24, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'L' / 0x4c
        'L' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 68, 91, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 163, 6, 6, 6, 6, 6, 4, 0, 0],
            &[0, 0, 121, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'M' / 0x4d
        'M' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 138, 104, 0, 0, 0, 0, 0, 49, 144, 51, 0],
            &[0, 241, 237, 11, 0, 0, 0, 0, 158, 247, 90, 0],
            &[0, 243, 179, 91, 0, 0, 0, 8, 236, 183, 90, 0],
            &[0, 244, 101, 182, 0, 0, 0, 85, 181, 171, 90, 0],
            &[0, 244, 33, 233, 22, 0, 0, 176, 88, 176, 90, 0],
            &[0, 244, 27, 151, 109, 0, 18, 236, 9, 176, 90, 0],
            &[0, 244, 27, 59, 200, 0, 103, 160, 0, 176, 90, 0],
            &[0, 244, 27, 0, 221, 36, 194, 68, 0, 176, 90, 0],
            &[0, 244, 27, 0, 130, 158, 227, 2, 0, 176, 90, 0],
            &[0, 244, 27, 0, 38, 255, 139, 0, 0, 176, 90, 0],
            &[0, 244, 27, 0, 0, 85, 28, 0, 0, 176, 90, 0],
            &[0, 244, 27, 0, 0, 0, 0, 0, 0, 176, 90, 0],
            &[0, 244, 27, 0, 0, 0, 0, 0, 0, 176, 90, 0],
            &[0, 244, 27, 0, 0, 0, 0, 0, 0, 176, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'N' / 0x4e
        'N' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 144, 15, 0, 0, 0, 0, 34, 121, 0, 0],
            &[0, 108, 249, 124, 0, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 167, 229, 12, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 154, 149, 119, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 161, 30, 228, 10, 0, 0, 61, 215, 0, 0],
            &[0, 108, 163, 0, 155, 114, 0, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 34, 227, 8, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 161, 109, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 38, 225, 7, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 166, 105, 60, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 43, 223, 60, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 172, 146, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 47, 243, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 0, 177, 215, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'O' / 0x4f
        'O' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 148, 187, 176, 103, 4, 0, 0, 0],
            &[0, 0, 80, 238, 124, 66, 82, 190, 189, 5, 0, 0],
            &[0, 20, 240, 61, 0, 0, 0, 2, 191, 123, 0, 0],
            &[0, 125, 172, 0, 0, 0, 0, 0, 60, 231, 1, 0],
            &[0, 193, 93, 0, 0, 0, 0, 0, 2, 238, 47, 0],
            &[0, 239, 47, 0, 0, 0, 0, 0, 0, 198, 88, 0],
            &[2, 254, 28, 0, 0, 0, 0, 0, 0, 178, 107, 0],
            &[8, 255, 23, 0, 0, 0, 0, 0, 0, 176, 110, 0],
            &[0, 250, 37, 0, 0, 0, 0, 0, 0, 190, 97, 0],
            &[0, 216, 73, 0, 0, 0, 0, 0, 0, 224, 63, 0],
            &[0, 158, 135, 0, 0, 0, 0, 0, 28, 248, 12, 0],
            &[0, 66, 232, 14, 0, 0, 0, 0, 135, 170, 0, 0],
            &[0, 0, 173, 186, 25, 0, 1, 99, 238, 37, 0, 0],
            &[0, 0, 9, 149, 244, 221, 239, 205, 51, 0, 0, 0],
            &[0, 0, 0, 0, 11, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'P' / 0x50
        'P' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 143, 132, 92, 16, 0, 0, 0],
            &[0, 1, 255, 117, 103, 106, 129, 187, 239, 73, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 105, 236, 9, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 1, 244, 61, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 219, 73, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 11, 246, 38, 0],
            &[0, 1, 255, 24, 0, 0, 0, 3, 146, 193, 0, 0],
            &[0, 1, 255, 139, 127, 131, 155, 221, 194, 26, 0, 0],
            &[0, 1, 255, 138, 125, 121, 101, 47, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Q' / 0x51
        'Q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 148, 187, 176, 103, 4, 0, 0, 0],
            &[0, 0, 80, 238, 124, 66, 82, 190, 189, 5, 0, 0],
            &[0, 20, 240, 61, 0, 0, 0, 2, 191, 123, 0, 0],
            &[0, 125, 172, 0, 0, 0, 0, 0, 60, 231, 1, 0],
            &[0, 193, 93, 0, 0, 0, 0, 0, 2, 238, 47, 0],
            &[0, 239, 47, 0, 0, 0, 0, 0, 0, 198, 88, 0],
            &[2, 254, 28, 0, 0, 0, 0, 0, 0, 178, 107, 0],
            &[8, 255, 23, 0, 0, 0, 0, 0, 0, 176, 110, 0],
            &[0, 250, 37, 0, 0, 0, 0, 0, 0, 190, 97, 0],
            &[0, 216, 73, 0, 0, 0, 0, 0, 0, 224, 63, 0],
            &[0, 158, 135, 0, 0, 0, 0, 0, 28, 248, 12, 0],
            &[0, 66, 232, 14, 0, 0, 0, 0, 135, 170, 0, 0],
            &[0, 0, 173, 186, 25, 0, 1, 99, 238, 37, 0, 0],
            &[0, 0, 9, 149, 244, 221, 255, 255, 52, 0, 0, 0],
            &[0, 0, 0, 0, 11, 40, 31, 222, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 214, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 207, 103, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 38, 64, 0, 0],
        ]),
        // letter: 'R' / 0x52
        'R' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 138, 116, 59, 0, 0, 0, 0],
            &[0, 1, 255, 117, 103, 115, 144, 226, 193, 15, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 6, 195, 146, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 90, 214, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 66, 226, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 118, 183, 0, 0],
            &[0, 1, 255, 24, 0, 0, 1, 75, 235, 59, 0, 0],
            &[0, 1, 255, 210, 205, 209, 236, 187, 53, 0, 0, 0],
            &[0, 1, 255, 67, 47, 47, 185, 124, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 38, 239, 31, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 136, 174, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 12, 228, 71, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 89, 217, 6, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 193, 121, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'S' / 0x53
        'S' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 130, 178, 185, 161, 109, 26, 0, 0],
            &[0, 0, 68, 241, 147, 89, 68, 95, 162, 109, 0, 0],
            &[0, 0, 218, 97, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 252, 37, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 162, 201, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 151, 242, 161, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 135, 235, 197, 67, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 104, 240, 94, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 222, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 35, 252, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 82, 221, 0, 0],
            &[0, 38, 66, 5, 0, 0, 0, 62, 228, 107, 0, 0],
            &[0, 49, 198, 247, 223, 205, 231, 221, 100, 0, 0, 0],
            &[0, 0, 0, 5, 28, 43, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'T' / 0x54
        'T' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[12, 144, 144, 144, 144, 144, 144, 144, 144, 144, 73, 0],
            &[9, 107, 107, 107, 107, 233, 145, 107, 107, 107, 54, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'U' / 0x55
        'U' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 37, 121, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 102, 180, 0, 0, 0, 0, 0, 73, 209, 0, 0],
            &[0, 80, 206, 0, 0, 0, 0, 0, 100, 186, 0, 0],
            &[0, 25, 250, 24, 0, 0, 0, 0, 173, 126, 0, 0],
            &[0, 0, 168, 181, 18, 0, 2, 103, 238, 23, 0, 0],
            &[0, 0, 12, 165, 244, 217, 237, 208, 53, 0, 0, 0],
            &[0, 0, 0, 0, 13, 40, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'V' / 0x56
        'V' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[40, 127, 0, 0, 0, 0, 0, 0, 0, 69, 101, 0],
            &[13, 247, 34, 0, 0, 0, 0, 0, 0, 186, 112, 0],
            &[0, 175, 117, 0, 0, 0, 0, 0, 17, 249, 29, 0],
            &[0, 89, 200, 0, 0, 0, 0, 0, 93, 199, 0, 0],
            &[0, 13, 246, 28, 0, 0, 0, 0, 175, 114, 0, 0],
            &[0, 0, 175, 110, 0, 0, 0, 11, 245, 30, 0, 0],
            &[0, 0, 89, 193, 0, 0, 0, 83, 201, 0, 0, 0],
            &[0, 0, 13, 244, 23, 0, 0, 165, 116, 0, 0, 0],
            &[0, 0, 0, 175, 104, 0, 6, 240, 32, 0, 0, 0],
            &[0, 0, 0, 89, 187, 0, 72, 203, 0, 0, 0, 0],
            &[0, 0, 0, 13, 242, 17, 153, 118, 0, 0, 0, 0],
            &[0, 0, 0, 0, 175, 88, 227, 34, 0, 0, 0, 0],
            &[0, 0, 0, 0, 89, 199, 205, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 247, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'W' / 0x57
        'W' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[48, 102, 0, 0, 0, 0, 0, 0, 0, 42, 109, 0],
            &[58, 208, 0, 0, 0, 0, 0, 0, 0, 101, 166, 0],
            &[23, 242, 0, 0, 0, 0, 0, 0, 0, 134, 132, 0],
            &[0, 244, 20, 0, 0, 23, 7, 0, 0, 168, 98, 0],
            &[0, 209, 53, 0, 4, 244, 100, 0, 0, 203, 64, 0],
            &[0, 175, 87, 0, 51, 198, 159, 0, 0, 238, 30, 0],
            &[0, 140, 121, 0, 110, 108, 207, 0, 16, 249, 2, 0],
            &[0, 105, 155, 0, 169, 49, 192, 22, 50, 217, 0, 0],
            &[0, 70, 189, 0, 216, 3, 134, 80, 83, 183, 0, 0],
            &[0, 36, 220, 31, 189, 0, 76, 140, 108, 149, 0, 0],
            &[0, 5, 243, 87, 132, 0, 18, 200, 132, 115, 0, 0],
            &[0, 0, 222, 158, 74, 0, 0, 209, 160, 81, 0, 0],
            &[0, 0, 187, 227, 17, 0, 0, 156, 225, 47, 0, 0],
            &[0, 0, 153, 214, 0, 0, 0, 97, 255, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'X' / 0x58
        'X' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 91, 83, 0, 0, 0, 0, 0, 12, 141, 16, 0],
            &[0, 43, 236, 23, 0, 0, 0, 0, 137, 159, 0, 0],
            &[0, 0, 137, 159, 0, 0, 0, 40, 231, 20, 0, 0],
            &[0, 0, 11, 224, 55, 0, 0, 188, 100, 0, 0, 0],
            &[0, 0, 0, 80, 201, 1, 85, 197, 1, 0, 0, 0],
            &[0, 0, 0, 0, 179, 109, 223, 45, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 245, 140, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 247, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 196, 93, 214, 58, 0, 0, 0, 0],
            &[0, 0, 0, 100, 188, 0, 69, 209, 4, 0, 0, 0],
            &[0, 0, 21, 231, 39, 0, 0, 171, 115, 0, 0, 0],
            &[0, 0, 162, 134, 0, 0, 0, 29, 232, 28, 0, 0],
            &[0, 65, 224, 10, 0, 0, 0, 0, 121, 174, 0, 0],
            &[6, 216, 80, 0, 0, 0, 0, 0, 6, 217, 75, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Y' / 0x59
        'Y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[8, 141, 25, 0, 0, 0, 0, 0, 0, 106, 65, 0],
            &[0, 159, 144, 0, 0, 0, 0, 0, 35, 240, 23, 0],
            &[0, 32, 240, 25, 0, 0, 0, 0, 157, 134, 0, 0],
            &[0, 0, 147, 144, 0, 0, 0, 33, 234, 17, 0, 0],
            &[0, 0, 24, 236, 25, 0, 0, 155, 123, 0, 0, 0],
            &[0, 0, 0, 135, 144, 0, 31, 228, 11, 0, 0, 0],
            &[0, 0, 0, 17, 231, 25, 152, 112, 0, 0, 0, 0],
            &[0, 0, 0, 0, 123, 171, 220, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 236, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Z' / 0x5a
        'Z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 144, 144, 144, 144, 144, 144, 55, 0, 0],
            &[0, 0, 56, 103, 103, 103, 103, 106, 247, 62, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 189, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 226, 55, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 113, 176, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 232, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 127, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 236, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 141, 150, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 238, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 156, 137, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 236, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 170, 126, 1, 1, 1, 1, 1, 0, 0, 0],
            &[0, 1, 254, 255, 255, 255, 255, 255, 255, 127, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '[' / 0x5b
        '[' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 66, 144, 144, 144, 85, 0, 0, 0, 0],
            &[0, 0, 0, 117, 189, 88, 88, 52, 0, 0, 0, 0],
            &[0, 0, 0, 117, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 117, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 117, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 117, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 117, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 117, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 117, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 117, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 117, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 117, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 117, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 117, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 117, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 117, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 117, 248, 237, 237, 139, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '\' / 0x5c
        '\\' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 112, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 240, 23, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 161, 115, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 64, 212, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 222, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 126, 150, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 237, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 187, 89, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 90, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 238, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 152, 124, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 55, 220, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 63, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 160, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ']' / 0x5d
        ']' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 169, 169, 169, 148, 0, 0, 0, 0],
            &[0, 0, 0, 11, 69, 69, 103, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 1, 48, 224, 0, 0, 0, 0],
            &[0, 0, 0, 39, 232, 232, 232, 203, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '^' / 0x5e
        '^' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 228, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 54, 207, 36, 0, 0, 0, 0],
            &[0, 0, 0, 65, 180, 0, 85, 163, 0, 0, 0, 0],
            &[0, 0, 0, 194, 54, 0, 1, 208, 41, 0, 0, 0],
            &[0, 0, 67, 182, 0, 0, 0, 84, 169, 0, 0, 0],
            &[0, 0, 196, 56, 0, 0, 0, 1, 209, 45, 0, 0],
            &[0, 70, 184, 0, 0, 0, 0, 0, 84, 174, 0, 0],
            &[0, 152, 56, 0, 0, 0, 0, 0, 1, 179, 31, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '_' / 0x5f
        '_' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 51],
            &[129, 129, 129, 129, 129, 129, 129, 129, 129, 129, 129, 51],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '`' / 0x60
        '`' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 139, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 116, 203, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 127, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 88, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'a' / 0x61
        'a' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 36, 66, 51, 12, 0, 0, 0, 0],
            &[0, 0, 84, 225, 202, 171, 189, 234, 99, 0, 0, 0],
            &[0, 0, 33, 30, 0, 0, 0, 42, 239, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 167, 99, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 137, 126, 0, 0],
            &[0, 0, 3, 89, 166, 205, 216, 214, 236, 127, 0, 0],
            &[0, 4, 190, 181, 67, 15, 0, 0, 134, 127, 0, 0],
            &[0, 83, 204, 1, 0, 0, 0, 0, 153, 127, 0, 0],
            &[0, 111, 161, 0, 0, 0, 0, 7, 225, 127, 0, 0],
            &[0, 59, 231, 30, 0, 0, 19, 169, 192, 127, 0, 0],
            &[0, 0, 119, 238, 202, 204, 209, 72, 110, 127, 0, 0],
            &[0, 0, 0, 7, 37, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'b' / 0x62
        'b' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 93, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 212, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 212, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 212, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 212, 0, 5, 53, 67, 23, 0, 0, 0, 0],
            &[0, 59, 209, 58, 213, 189, 186, 239, 141, 1, 0, 0],
            &[0, 59, 219, 197, 32, 0, 0, 28, 217, 111, 0, 0],
            &[0, 59, 255, 81, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 59, 251, 3, 0, 0, 0, 0, 15, 255, 27, 0],
            &[0, 59, 226, 0, 0, 0, 0, 0, 0, 244, 51, 0],
            &[0, 59, 226, 0, 0, 0, 0, 0, 1, 250, 47, 0],
            &[0, 59, 250, 2, 0, 0, 0, 0, 25, 255, 16, 0],
            &[0, 59, 255, 77, 0, 0, 0, 0, 112, 203, 0, 0],
            &[0, 59, 219, 202, 33, 0, 0, 57, 235, 77, 0, 0],
            &[0, 59, 177, 61, 216, 210, 220, 234, 97, 0, 0, 0],
            &[0, 0, 0, 0, 2, 31, 37, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'c' / 0x63
        'c' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 66, 58, 19, 0, 0, 0],
            &[0, 0, 0, 68, 214, 225, 186, 193, 236, 93, 0, 0],
            &[0, 0, 68, 240, 84, 1, 0, 0, 6, 13, 0, 0],
            &[0, 0, 214, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 39, 245, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 70, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 73, 213, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 44, 244, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 92, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 238, 82, 2, 0, 0, 16, 32, 0, 0],
            &[0, 0, 0, 77, 210, 235, 206, 222, 232, 85, 0, 0],
            &[0, 0, 0, 0, 0, 20, 42, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'd' / 0x64
        'd' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 50, 73, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 0, 0, 1, 46, 66, 27, 0, 114, 166, 0, 0],
            &[0, 0, 48, 216, 218, 173, 206, 158, 114, 166, 0, 0],
            &[0, 14, 233, 121, 0, 0, 0, 110, 211, 166, 0, 0],
            &[0, 108, 194, 0, 0, 0, 0, 0, 213, 166, 0, 0],
            &[0, 171, 119, 0, 0, 0, 0, 0, 147, 166, 0, 0],
            &[0, 195, 89, 0, 0, 0, 0, 0, 121, 166, 0, 0],
            &[0, 198, 92, 0, 0, 0, 0, 0, 124, 166, 0, 0],
            &[0, 173, 120, 0, 0, 0, 0, 0, 152, 166, 0, 0],
            &[0, 111, 200, 0, 0, 0, 0, 3, 226, 166, 0, 0],
            &[0, 16, 231, 122, 0, 0, 2, 146, 187, 166, 0, 0],
            &[0, 0, 44, 204, 236, 202, 231, 120, 74, 166, 0, 0],
            &[0, 0, 0, 0, 26, 42, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'e' / 0x65
        'e' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 60, 60, 16, 0, 0, 0, 0],
            &[0, 0, 7, 153, 232, 184, 177, 232, 121, 0, 0, 0],
            &[0, 0, 168, 178, 17, 0, 0, 21, 212, 94, 0, 0],
            &[0, 55, 231, 15, 0, 0, 0, 0, 76, 209, 0, 0],
            &[0, 134, 155, 0, 0, 0, 0, 0, 15, 255, 10, 0],
            &[0, 167, 232, 213, 213, 213, 213, 213, 213, 255, 30, 0],
            &[0, 170, 128, 20, 20, 20, 20, 20, 20, 20, 2, 0],
            &[0, 138, 153, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 233, 15, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 160, 191, 34, 0, 0, 0, 24, 74, 0, 0],
            &[0, 0, 2, 121, 228, 218, 201, 225, 223, 118, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'f' / 0x66
        'f' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 116, 131, 114, 55, 0],
            &[0, 0, 0, 0, 0, 111, 235, 137, 107, 123, 93, 0],
            &[0, 0, 0, 0, 3, 240, 67, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 254, 8, 0, 0, 0, 0, 0],
            &[0, 0, 1, 11, 49, 252, 27, 27, 27, 27, 2, 0],
            &[0, 99, 214, 215, 220, 255, 215, 215, 215, 215, 19, 0],
            &[0, 0, 0, 0, 29, 252, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 252, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 252, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 252, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 252, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 252, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 252, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 252, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 252, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'g' / 0x67
        'g' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 49, 69, 35, 0, 6, 18, 0, 0],
            &[0, 0, 46, 217, 205, 168, 200, 172, 86, 166, 0, 0],
            &[0, 12, 228, 101, 0, 0, 0, 82, 228, 166, 0, 0],
            &[0, 105, 204, 0, 0, 0, 0, 0, 183, 166, 0, 0],
            &[0, 169, 123, 0, 0, 0, 0, 0, 120, 166, 0, 0],
            &[0, 197, 95, 0, 0, 0, 0, 0, 99, 166, 0, 0],
            &[0, 198, 95, 0, 0, 0, 0, 0, 102, 166, 0, 0],
            &[0, 172, 123, 0, 0, 0, 0, 0, 129, 166, 0, 0],
            &[0, 109, 207, 0, 0, 0, 0, 0, 207, 166, 0, 0],
            &[0, 14, 227, 132, 1, 0, 1, 131, 201, 166, 0, 0],
            &[0, 0, 40, 200, 238, 198, 219, 139, 102, 166, 0, 0],
            &[0, 0, 0, 0, 25, 40, 11, 0, 106, 163, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 140, 140, 0, 0],
            &[0, 2, 0, 0, 0, 0, 0, 15, 233, 67, 0, 0],
            &[0, 0, 51, 121, 152, 165, 140, 66, 0, 0, 0, 0],
        ]),
        // letter: 'h' / 0x68
        'h' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 97, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 222, 0, 4, 55, 64, 24, 0, 0, 0, 0],
            &[0, 59, 220, 50, 205, 179, 180, 238, 146, 0, 0, 0],
            &[0, 59, 224, 177, 22, 0, 0, 26, 230, 90, 0, 0],
            &[0, 59, 255, 60, 0, 0, 0, 0, 129, 159, 0, 0],
            &[0, 59, 244, 1, 0, 0, 0, 0, 96, 188, 0, 0],
            &[0, 59, 224, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'i' / 0x69
        'i' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 218, 181, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 118, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 27, 27, 27, 16, 0, 0, 0, 0, 0],
            &[0, 0, 171, 190, 202, 234, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'j' / 0x6a
        'j' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 78, 255, 66, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 30, 156, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 27, 27, 27, 27, 4, 0, 0, 0],
            &[0, 0, 0, 69, 183, 195, 208, 253, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 251, 32, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 60, 244, 2, 0, 0, 0],
            &[0, 0, 68, 141, 164, 143, 67, 0, 0, 0, 0, 0],
        ]),
        // letter: 'k' / 0x6b
        'k' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 74, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 0, 3, 27, 6, 0, 0],
            &[0, 169, 112, 0, 0, 0, 2, 164, 175, 4, 0, 0],
            &[0, 169, 112, 0, 0, 0, 153, 186, 7, 0, 0, 0],
            &[0, 169, 112, 0, 0, 141, 196, 10, 0, 0, 0, 0],
            &[0, 169, 109, 0, 129, 206, 15, 0, 0, 0, 0, 0],
            &[0, 169, 103, 116, 244, 149, 0, 0, 0, 0, 0, 0],
            &[0, 169, 184, 190, 27, 222, 79, 0, 0, 0, 0, 0],
            &[0, 169, 189, 9, 0, 57, 233, 28, 0, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 125, 194, 3, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 3, 194, 128, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 0, 29, 235, 63, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'l' / 0x6c
        'l' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 112, 112, 112, 112, 66, 0, 0, 0, 0, 0],
            &[0, 2, 93, 105, 118, 192, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'm' / 0x6d
        'm' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 0, 35, 59, 2, 0, 36, 64, 5, 0, 0],
            &[0, 213, 87, 203, 231, 157, 88, 225, 222, 190, 0, 0],
            &[0, 220, 175, 0, 29, 245, 196, 6, 12, 238, 41, 0],
            &[0, 220, 93, 0, 0, 216, 114, 0, 0, 194, 73, 0],
            &[0, 220, 57, 0, 0, 195, 74, 0, 0, 177, 81, 0],
            &[0, 220, 38, 0, 0, 193, 64, 0, 0, 176, 81, 0],
            &[0, 220, 37, 0, 0, 193, 64, 0, 0, 176, 81, 0],
            &[0, 220, 37, 0, 0, 193, 64, 0, 0, 176, 81, 0],
            &[0, 220, 37, 0, 0, 193, 64, 0, 0, 176, 81, 0],
            &[0, 220, 37, 0, 0, 193, 64, 0, 0, 176, 81, 0],
            &[0, 220, 37, 0, 0, 193, 64, 0, 0, 176, 81, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'n' / 0x6e
        'n' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 18, 0, 5, 55, 66, 27, 0, 0, 0, 0],
            &[0, 59, 181, 58, 206, 187, 179, 236, 154, 1, 0, 0],
            &[0, 59, 220, 181, 24, 0, 0, 20, 223, 97, 0, 0],
            &[0, 59, 255, 61, 0, 0, 0, 0, 124, 163, 0, 0],
            &[0, 59, 250, 2, 0, 0, 0, 0, 94, 190, 0, 0],
            &[0, 59, 226, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'o' / 0x6f
        'o' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 66, 52, 7, 0, 0, 0, 0],
            &[0, 0, 25, 186, 216, 169, 188, 229, 92, 0, 0, 0],
            &[0, 7, 212, 123, 1, 0, 0, 40, 229, 73, 0, 0],
            &[0, 97, 210, 1, 0, 0, 0, 0, 93, 204, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 15, 253, 20, 0],
            &[0, 195, 90, 0, 0, 0, 0, 0, 0, 238, 47, 0],
            &[0, 195, 87, 0, 0, 0, 0, 0, 0, 241, 46, 0],
            &[0, 163, 124, 0, 0, 0, 0, 0, 23, 253, 14, 0],
            &[0, 87, 211, 2, 0, 0, 0, 0, 111, 191, 0, 0],
            &[0, 3, 198, 149, 9, 0, 0, 69, 237, 57, 0, 0],
            &[0, 0, 16, 163, 232, 199, 218, 216, 69, 0, 0, 0],
            &[0, 0, 0, 0, 12, 41, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'p' / 0x70
        'p' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 18, 0, 5, 55, 68, 26, 0, 0, 0, 0],
            &[0, 59, 182, 50, 205, 189, 183, 236, 149, 3, 0, 0],
            &[0, 59, 212, 183, 29, 0, 0, 22, 207, 118, 0, 0],
            &[0, 59, 255, 73, 0, 0, 0, 0, 77, 228, 0, 0],
            &[0, 59, 251, 2, 0, 0, 0, 0, 12, 255, 29, 0],
            &[0, 59, 225, 0, 0, 0, 0, 0, 0, 244, 51, 0],
            &[0, 59, 232, 0, 0, 0, 0, 0, 1, 250, 44, 0],
            &[0, 59, 253, 6, 0, 0, 0, 0, 23, 254, 17, 0],
            &[0, 59, 255, 86, 0, 0, 0, 0, 106, 204, 0, 0],
            &[0, 59, 219, 193, 40, 0, 0, 50, 231, 79, 0, 0],
            &[0, 59, 215, 43, 202, 213, 216, 233, 100, 0, 0, 0],
            &[0, 59, 217, 0, 0, 31, 36, 5, 0, 0, 0, 0],
            &[0, 59, 217, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 217, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 121, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'q' / 0x71
        'q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 50, 68, 30, 0, 6, 18, 0, 0],
            &[0, 0, 45, 216, 212, 174, 213, 157, 81, 166, 0, 0],
            &[0, 13, 227, 109, 0, 0, 0, 121, 214, 166, 0, 0],
            &[0, 105, 202, 0, 0, 0, 0, 1, 219, 166, 0, 0],
            &[0, 170, 123, 0, 0, 0, 0, 0, 150, 166, 0, 0],
            &[0, 197, 95, 0, 0, 0, 0, 0, 124, 166, 0, 0],
            &[0, 198, 93, 0, 0, 0, 0, 0, 125, 166, 0, 0],
            &[0, 172, 122, 0, 0, 0, 0, 0, 155, 166, 0, 0],
            &[0, 109, 203, 0, 0, 0, 0, 6, 230, 166, 0, 0],
            &[0, 15, 229, 122, 0, 0, 5, 157, 187, 166, 0, 0],
            &[0, 0, 44, 205, 235, 202, 228, 110, 112, 166, 0, 0],
            &[0, 0, 0, 0, 27, 38, 8, 0, 115, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 64, 93, 0, 0],
        ]),
        // letter: 'r' / 0x72
        'r' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 27, 27, 27, 18, 0, 9, 56, 57, 14, 0],
            &[0, 23, 181, 196, 221, 182, 52, 224, 196, 199, 172, 0],
            &[0, 0, 0, 0, 63, 204, 195, 38, 0, 0, 1, 0],
            &[0, 0, 0, 0, 63, 255, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 249, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 225, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 93, 212, 226, 242, 253, 234, 221, 208, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 's' / 0x73
        's' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 66, 64, 36, 0, 0, 0, 0],
            &[0, 0, 10, 176, 229, 179, 178, 214, 206, 1, 0, 0],
            &[0, 0, 114, 191, 4, 0, 0, 0, 18, 0, 0, 0],
            &[0, 0, 145, 144, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 73, 235, 77, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 74, 202, 215, 113, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 154, 238, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 55, 253, 31, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 232, 60, 0, 0],
            &[0, 0, 83, 17, 0, 0, 0, 82, 241, 15, 0, 0],
            &[0, 0, 145, 237, 224, 201, 224, 214, 64, 0, 0, 0],
            &[0, 0, 0, 2, 24, 43, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 't' / 0x74
        't' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 82, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 140, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 23, 174, 131, 27, 27, 27, 13, 0, 0],
            &[0, 140, 215, 215, 241, 233, 215, 215, 215, 107, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 165, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 151, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 227, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 142, 238, 197, 207, 130, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 41, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'u' / 0x75
        'u' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 21, 0, 0, 0, 0, 0, 12, 18, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 118, 166, 0, 0],
            &[0, 78, 206, 0, 0, 0, 0, 0, 142, 166, 0, 0],
            &[0, 47, 238, 4, 0, 0, 0, 2, 214, 166, 0, 0],
            &[0, 4, 218, 125, 0, 0, 5, 138, 182, 166, 0, 0],
            &[0, 0, 50, 214, 226, 198, 220, 108, 72, 166, 0, 0],
            &[0, 0, 0, 0, 22, 33, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'v' / 0x76
        'v' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 10, 0, 0, 0, 0, 0, 0, 27, 3, 0],
            &[0, 136, 153, 0, 0, 0, 0, 0, 55, 233, 5, 0],
            &[0, 40, 238, 7, 0, 0, 0, 0, 148, 141, 0, 0],
            &[0, 0, 199, 85, 0, 0, 0, 5, 235, 43, 0, 0],
            &[0, 0, 102, 179, 0, 0, 0, 79, 201, 0, 0, 0],
            &[0, 0, 15, 241, 21, 0, 0, 172, 104, 0, 0, 0],
            &[0, 0, 0, 165, 111, 0, 16, 239, 15, 0, 0, 0],
            &[0, 0, 0, 68, 205, 0, 102, 164, 0, 0, 0, 0],
            &[0, 0, 0, 2, 225, 44, 195, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 166, 222, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 254, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'w' / 0x77
        'w' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[16, 12, 0, 0, 0, 0, 0, 0, 0, 0, 27, 0],
            &[121, 141, 0, 0, 0, 0, 0, 0, 0, 27, 230, 0],
            &[69, 194, 0, 0, 0, 54, 24, 0, 0, 78, 180, 0],
            &[17, 244, 3, 0, 18, 249, 145, 0, 0, 130, 130, 0],
            &[0, 220, 46, 0, 92, 163, 220, 0, 0, 181, 80, 0],
            &[0, 168, 99, 0, 169, 77, 206, 42, 0, 232, 30, 0],
            &[0, 115, 153, 5, 234, 11, 133, 118, 27, 236, 0, 0],
            &[0, 63, 200, 67, 184, 0, 57, 194, 73, 186, 0, 0],
            &[0, 13, 237, 134, 108, 0, 2, 231, 123, 136, 0, 0],
            &[0, 0, 214, 207, 33, 0, 0, 162, 207, 86, 0, 0],
            &[0, 0, 161, 213, 0, 0, 0, 86, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'x' / 0x78
        'x' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 20, 0, 0, 0, 0, 0, 8, 25, 0, 0],
            &[0, 39, 235, 42, 0, 0, 0, 1, 183, 129, 0, 0],
            &[0, 0, 97, 209, 7, 0, 0, 114, 193, 3, 0, 0],
            &[0, 0, 0, 166, 143, 0, 51, 230, 27, 0, 0, 0],
            &[0, 0, 0, 14, 219, 83, 217, 76, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 253, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 222, 206, 7, 0, 0, 0, 0],
            &[0, 0, 0, 52, 229, 26, 160, 147, 0, 0, 0, 0],
            &[0, 0, 14, 219, 78, 0, 12, 217, 81, 0, 0, 0],
            &[0, 0, 167, 147, 0, 0, 0, 51, 232, 31, 0, 0],
            &[0, 101, 209, 8, 0, 0, 0, 0, 115, 198, 4, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'y' / 0x79
        'y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 11, 0, 0, 0, 0, 0, 0, 27, 3, 0],
            &[0, 119, 161, 0, 0, 0, 0, 0, 49, 234, 5, 0],
            &[0, 23, 240, 14, 0, 0, 0, 0, 141, 142, 0, 0],
            &[0, 0, 171, 104, 0, 0, 0, 2, 230, 46, 0, 0],
            &[0, 0, 70, 203, 0, 0, 0, 69, 205, 0, 0, 0],
            &[0, 0, 1, 222, 46, 0, 0, 161, 109, 0, 0, 0],
            &[0, 0, 0, 122, 146, 0, 10, 236, 19, 0, 0, 0],
            &[0, 0, 0, 25, 233, 7, 88, 172, 0, 0, 0, 0],
            &[0, 0, 0, 0, 175, 85, 177, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 186, 224, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 225, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 226, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 201, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 186, 90, 0, 0, 0, 0, 0, 0],
            &[0, 83, 166, 133, 15, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'z' / 0x7a
        'z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 27, 27, 27, 27, 27, 27, 10, 0, 0],
            &[0, 0, 129, 200, 200, 200, 200, 201, 254, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 108, 201, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 233, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 213, 88, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 152, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 215, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 230, 47, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 193, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 126, 176, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 252, 250, 247, 247, 247, 247, 247, 105, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '{' / 0x7b
        '{' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 44, 119, 76, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 242, 135, 55, 0, 0, 0],
            &[0, 0, 0, 0, 0, 203, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 226, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 227, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 227, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 237, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 233, 7, 0, 0, 0, 0, 0],
            &[0, 0, 18, 213, 212, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 62, 168, 176, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 248, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 227, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 227, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 227, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 219, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 157, 176, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 151, 227, 131, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '|' / 0x7c
        '|' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 85, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 108, 25, 0, 0, 0, 0, 0],
        ]),
        // letter: '}' / 0x7d
        '}' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 130, 79, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 110, 195, 187, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 237, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 204, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 203, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 203, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 197, 83, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 143, 164, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 153, 227, 96, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 221, 93, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 182, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 202, 74, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 203, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 203, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 223, 65, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 100, 243, 16, 0, 0, 0, 0, 0],
            &[0, 0, 35, 240, 194, 60, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '~' / 0x7e
        '~' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 189, 154, 22, 0, 0, 8, 166, 0, 0],
            &[0, 22, 204, 19, 82, 209, 36, 0, 63, 155, 0, 0],
            &[0, 92, 119, 0, 0, 54, 213, 143, 206, 45, 0, 0],
            &[0, 43, 33, 0, 0, 0, 15, 75, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ' ' / 0xa0
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¡' / 0xa1
        '¡' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 255, 117, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 150, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 97, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 207, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 216, 57, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 220, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 224, 66, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 229, 70, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 233, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 238, 79, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 242, 84, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 126, 45, 0, 0, 0, 0, 0],
        ]),
        // letter: '¢' / 0xa2
        '¢' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 205, 27, 2, 0, 0, 0],
            &[0, 0, 0, 45, 186, 235, 205, 217, 235, 54, 0, 0],
            &[0, 0, 43, 236, 108, 5, 0, 0, 19, 8, 0, 0],
            &[0, 0, 183, 129, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 250, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 45, 236, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 51, 230, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 251, 11, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 94, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 87, 234, 54, 0, 0, 0, 4, 10, 0, 0],
            &[0, 0, 0, 99, 234, 203, 175, 193, 232, 57, 0, 0],
            &[0, 0, 0, 0, 6, 77, 209, 57, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 190, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '£' / 0xa3
        '£' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 159, 176, 140, 46, 0, 0],
            &[0, 0, 0, 0, 117, 214, 88, 65, 111, 157, 0, 0],
            &[0, 0, 0, 5, 243, 42, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 37, 239, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 221, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 221, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 22, 73, 224, 22, 22, 22, 5, 0, 0, 0],
            &[0, 15, 196, 209, 247, 196, 196, 196, 48, 0, 0, 0],
            &[0, 0, 0, 56, 221, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 221, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 70, 210, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 121, 158, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 216, 35, 1, 1, 1, 1, 1, 0, 0],
            &[0, 67, 255, 255, 255, 255, 255, 255, 255, 255, 27, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¤' / 0xa4
        '¤' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 1, 0, 0, 0, 0, 0, 0, 3, 0, 0],
            &[0, 107, 156, 5, 90, 141, 120, 30, 61, 195, 1, 0],
            &[0, 4, 174, 231, 142, 79, 101, 204, 227, 44, 0, 0],
            &[0, 0, 160, 90, 0, 0, 0, 13, 210, 25, 0, 0],
            &[0, 1, 219, 1, 0, 0, 0, 0, 122, 101, 0, 0],
            &[0, 8, 211, 0, 0, 0, 0, 0, 106, 118, 0, 0],
            &[0, 0, 202, 32, 0, 0, 0, 0, 178, 63, 0, 0],
            &[0, 0, 114, 207, 41, 0, 10, 131, 223, 2, 0, 0],
            &[0, 62, 221, 96, 195, 224, 218, 130, 169, 156, 0, 0],
            &[0, 37, 41, 0, 0, 0, 0, 0, 3, 80, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¥' / 0xa5
        '¥' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[17, 142, 10, 0, 0, 0, 0, 0, 0, 92, 77, 0],
            &[0, 173, 124, 0, 0, 0, 0, 0, 28, 237, 31, 0],
            &[0, 36, 236, 22, 0, 0, 0, 0, 159, 133, 0, 0],
            &[0, 0, 142, 150, 0, 0, 0, 47, 228, 12, 0, 0],
            &[0, 0, 17, 233, 40, 0, 0, 187, 97, 0, 0, 0],
            &[0, 0, 0, 111, 176, 0, 73, 206, 2, 0, 0, 0],
            &[0, 0, 0, 5, 217, 65, 211, 62, 0, 0, 0, 0],
            &[0, 0, 64, 73, 145, 240, 211, 73, 73, 21, 0, 0],
            &[0, 0, 122, 139, 139, 236, 166, 139, 139, 40, 0, 0],
            &[0, 0, 0, 0, 0, 212, 59, 0, 0, 0, 0, 0],
            &[0, 0, 136, 156, 156, 239, 179, 156, 156, 45, 0, 0],
            &[0, 0, 50, 57, 57, 222, 103, 57, 57, 16, 0, 0],
            &[0, 0, 0, 0, 0, 212, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 212, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¦' / 0xa6
        '¦' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 94, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 128, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 26, 0, 0, 0, 0, 0],
        ]),
        // letter: '§' / 0xa7
        '§' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 66, 107, 111, 85, 22, 0, 0, 0],
            &[0, 0, 12, 198, 164, 103, 104, 137, 192, 3, 0, 0],
            &[0, 0, 91, 176, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 86, 206, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 149, 214, 103, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 113, 190, 153, 223, 150, 15, 0, 0, 0],
            &[0, 0, 73, 176, 1, 0, 5, 120, 217, 11, 0, 0],
            &[0, 0, 126, 116, 0, 0, 0, 0, 194, 68, 0, 0],
            &[0, 0, 77, 211, 27, 0, 0, 0, 202, 45, 0, 0],
            &[0, 0, 0, 112, 224, 149, 56, 148, 162, 0, 0, 0],
            &[0, 0, 0, 0, 9, 97, 200, 238, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 87, 237, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 221, 47, 0, 0],
            &[0, 0, 69, 15, 0, 0, 0, 71, 228, 7, 0, 0],
            &[0, 0, 118, 218, 210, 195, 213, 185, 46, 0, 0, 0],
            &[0, 0, 0, 0, 3, 17, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¨' / 0xa8
        '¨' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 168, 54, 0, 0, 162, 59, 0, 0, 0],
            &[0, 0, 3, 213, 74, 0, 1, 208, 80, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '©' / 0xa9
        '©' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 143, 172, 162, 104, 13, 0, 0, 0],
            &[0, 1, 147, 141, 35, 1, 14, 88, 192, 42, 0, 0],
            &[0, 115, 104, 0, 99, 174, 178, 85, 24, 192, 6, 0],
            &[3, 189, 0, 93, 146, 2, 0, 12, 0, 111, 82, 0],
            &[31, 152, 0, 183, 15, 0, 0, 0, 0, 45, 134, 0],
            &[41, 143, 0, 195, 2, 0, 0, 0, 0, 35, 143, 0],
            &[12, 181, 0, 147, 66, 0, 0, 0, 0, 78, 106, 0],
            &[0, 167, 46, 22, 184, 151, 143, 73, 1, 179, 20, 0],
            &[0, 25, 195, 50, 0, 21, 26, 11, 153, 91, 0, 0],
            &[0, 0, 21, 150, 178, 162, 169, 170, 61, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ª' / 0xaa
        'ª' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 37, 139, 174, 155, 46, 0, 0, 0, 0],
            &[0, 0, 0, 37, 52, 5, 45, 212, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 37, 199, 22, 0, 0, 0],
            &[0, 0, 0, 65, 181, 142, 122, 217, 24, 0, 0, 0],
            &[0, 0, 0, 201, 15, 0, 0, 203, 24, 0, 0, 0],
            &[0, 0, 0, 191, 101, 31, 107, 235, 24, 0, 0, 0],
            &[0, 0, 0, 27, 135, 137, 61, 81, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '«' / 0xab
        '«' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 88, 119, 0, 12, 181, 11, 0, 0],
            &[0, 0, 0, 50, 224, 28, 1, 174, 120, 0, 0, 0],
            &[0, 0, 22, 222, 65, 0, 129, 176, 1, 0, 0, 0],
            &[0, 0, 194, 119, 0, 78, 220, 15, 0, 0, 0, 0],
            &[0, 0, 156, 160, 0, 49, 230, 37, 0, 0, 0, 0],
            &[0, 0, 6, 195, 102, 0, 84, 209, 10, 0, 0, 0],
            &[0, 0, 0, 21, 216, 51, 0, 128, 164, 0, 0, 0],
            &[0, 0, 0, 0, 46, 85, 0, 1, 124, 11, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¬' / 0xac
        '¬' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 31, 41, 41, 41, 41, 41, 41, 41, 26, 0, 0],
            &[0, 146, 191, 191, 191, 191, 191, 191, 209, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 71, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 71, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 71, 161, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 37, 85, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '®' / 0xae
        '®' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 143, 172, 162, 104, 13, 0, 0, 0],
            &[0, 1, 147, 141, 35, 1, 14, 88, 192, 42, 0, 0],
            &[0, 115, 104, 2, 222, 167, 188, 74, 24, 192, 6, 0],
            &[3, 189, 0, 2, 172, 0, 6, 187, 0, 111, 82, 0],
            &[31, 152, 0, 2, 172, 0, 41, 172, 0, 45, 134, 0],
            &[41, 143, 0, 2, 223, 159, 214, 17, 0, 35, 143, 0],
            &[12, 181, 0, 2, 172, 0, 148, 55, 0, 78, 106, 0],
            &[0, 167, 46, 2, 172, 0, 18, 187, 5, 179, 20, 0],
            &[0, 25, 195, 51, 37, 0, 0, 46, 164, 91, 0, 0],
            &[0, 0, 21, 150, 178, 162, 169, 170, 61, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¯' / 0xaf
        '¯' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 190, 206, 206, 206, 206, 66, 0, 0, 0],
            &[0, 0, 0, 25, 27, 27, 27, 27, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '°' / 0xb0
        '°' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 119, 184, 157, 37, 0, 0, 0, 0],
            &[0, 0, 0, 156, 124, 19, 59, 206, 21, 0, 0, 0],
            &[0, 0, 5, 210, 0, 0, 0, 109, 102, 0, 0, 0],
            &[0, 0, 6, 210, 0, 0, 0, 104, 105, 0, 0, 0],
            &[0, 0, 0, 165, 116, 7, 44, 207, 27, 0, 0, 0],
            &[0, 0, 0, 9, 133, 197, 174, 52, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '±' / 0xb1
        '±' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 35, 0, 0, 0, 0, 0],
            &[0, 28, 46, 46, 46, 208, 75, 46, 46, 45, 0, 0],
            &[0, 111, 181, 181, 181, 239, 192, 181, 181, 177, 0, 0],
            &[0, 0, 0, 0, 0, 198, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 127, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 233, 233, 233, 233, 233, 233, 233, 227, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '²' / 0xb2
        '²' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 44, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 104, 206, 155, 191, 148, 0, 0, 0, 0],
            &[0, 0, 0, 34, 3, 0, 0, 208, 35, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 203, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 177, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 198, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 198, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 196, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 241, 207, 189, 189, 189, 89, 0, 0, 0],
            &[0, 0, 0, 24, 24, 24, 24, 24, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '³' / 0xb3
        '³' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 37, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 130, 207, 162, 201, 133, 0, 0, 0, 0],
            &[0, 0, 0, 32, 0, 0, 4, 227, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 225, 10, 0, 0, 0],
            &[0, 0, 0, 0, 119, 141, 201, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 62, 113, 202, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 162, 79, 0, 0, 0],
            &[0, 0, 1, 7, 0, 0, 2, 201, 52, 0, 0, 0],
            &[0, 0, 6, 207, 175, 154, 204, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 44, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '´' / 0xb4
        '´' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 136, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 145, 178, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 186, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 107, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'µ' / 0xb5
        'µ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 21, 0, 0, 0, 0, 0, 12, 18, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 110, 255, 0, 0, 0, 0, 0, 118, 166, 0, 0],
            &[0, 162, 255, 0, 0, 0, 0, 0, 142, 166, 0, 0],
            &[0, 131, 255, 4, 0, 0, 0, 2, 214, 166, 0, 0],
            &[0, 88, 255, 125, 0, 0, 5, 138, 182, 166, 0, 0],
            &[0, 83, 248, 214, 226, 198, 220, 108, 72, 166, 0, 0],
            &[0, 83, 198, 0, 22, 33, 2, 0, 0, 0, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 47, 110, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¶' / 0xb6
        '¶' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 63, 105, 112, 112, 112, 112, 79, 0, 0],
            &[0, 3, 182, 255, 255, 255, 170, 86, 98, 181, 0, 0],
            &[0, 96, 255, 255, 255, 255, 127, 0, 17, 181, 0, 0],
            &[0, 168, 255, 255, 255, 255, 127, 0, 17, 181, 0, 0],
            &[0, 193, 255, 255, 255, 255, 127, 0, 17, 181, 0, 0],
            &[0, 189, 255, 255, 255, 255, 127, 0, 17, 181, 0, 0],
            &[0, 155, 255, 255, 255, 255, 127, 0, 17, 181, 0, 0],
            &[0, 71, 255, 255, 255, 255, 127, 0, 17, 181, 0, 0],
            &[0, 0, 136, 251, 255, 255, 127, 0, 17, 181, 0, 0],
            &[0, 0, 0, 19, 46, 90, 127, 0, 17, 181, 0, 0],
            &[0, 0, 0, 0, 0, 71, 127, 0, 17, 181, 0, 0],
            &[0, 0, 0, 0, 0, 71, 127, 0, 17, 181, 0, 0],
            &[0, 0, 0, 0, 0, 71, 127, 0, 17, 181, 0, 0],
            &[0, 0, 0, 0, 0, 71, 127, 0, 17, 181, 0, 0],
            &[0, 0, 0, 0, 0, 71, 127, 0, 17, 181, 0, 0],
            &[0, 0, 0, 0, 0, 71, 127, 0, 17, 181, 0, 0],
            &[0, 0, 0, 0, 0, 71, 127, 0, 17, 181, 0, 0],
            &[0, 0, 0, 0, 0, 29, 52, 0, 7, 74, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '·' / 0xb7
        '·' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 246, 112, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 187, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¸' / 0xb8
        '¸' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 151, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 185, 175, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 100, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 131, 70, 0, 0, 0, 0, 0],
        ]),
        // letter: '¹' / 0xb9
        '¹' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 117, 236, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 185, 132, 131, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 0, 124, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 125, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 125, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 125, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 125, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 159, 255, 255, 157, 53, 0, 0, 0],
            &[0, 0, 0, 11, 24, 36, 34, 24, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'º' / 0xba
        'º' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 174, 153, 31, 0, 0, 0, 0],
            &[0, 0, 0, 107, 150, 13, 68, 211, 4, 0, 0, 0],
            &[0, 0, 0, 203, 17, 0, 0, 171, 59, 0, 0, 0],
            &[0, 0, 0, 217, 0, 0, 0, 144, 79, 0, 0, 0],
            &[0, 0, 0, 196, 23, 0, 0, 177, 48, 0, 0, 0],
            &[0, 0, 0, 87, 169, 44, 95, 187, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 141, 113, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '»' / 0xbb
        '»' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 97, 53, 0, 19, 127, 0, 0, 0, 0, 0],
            &[0, 0, 47, 219, 26, 0, 172, 117, 0, 0, 0, 0],
            &[0, 0, 0, 94, 202, 8, 13, 214, 75, 0, 0, 0],
            &[0, 0, 0, 0, 151, 166, 0, 44, 230, 41, 0, 0],
            &[0, 0, 0, 0, 128, 185, 0, 30, 229, 54, 0, 0],
            &[0, 0, 0, 72, 216, 16, 7, 203, 98, 0, 0, 0],
            &[0, 0, 32, 222, 40, 0, 155, 146, 0, 0, 0, 0],
            &[0, 0, 114, 75, 0, 23, 165, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¼' / 0xbc
        '¼' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 138, 19, 0, 0, 0, 0, 58, 86, 0, 0],
            &[112, 207, 215, 34, 0, 0, 0, 1, 205, 50, 0, 0],
            &[26, 3, 187, 34, 0, 0, 0, 84, 173, 0, 0, 0],
            &[0, 0, 189, 34, 0, 0, 2, 211, 42, 0, 0, 0],
            &[0, 0, 189, 34, 0, 0, 93, 163, 0, 0, 0, 0],
            &[0, 0, 189, 34, 0, 5, 216, 35, 0, 0, 0, 0],
            &[52, 178, 255, 219, 120, 103, 154, 0, 0, 0, 0, 0],
            &[2, 6, 12, 7, 12, 220, 29, 0, 56, 126, 0, 0],
            &[0, 0, 0, 0, 112, 145, 0, 14, 195, 200, 0, 0],
            &[0, 0, 0, 11, 222, 23, 0, 161, 81, 200, 0, 0],
            &[0, 0, 0, 121, 135, 0, 88, 139, 17, 200, 0, 0],
            &[0, 0, 15, 223, 18, 21, 231, 112, 116, 223, 89, 0],
            &[0, 0, 131, 126, 0, 16, 88, 88, 100, 219, 75, 0],
            &[0, 20, 223, 13, 0, 0, 0, 0, 17, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '½' / 0xbd
        '½' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 138, 19, 0, 0, 0, 0, 58, 86, 0, 0],
            &[112, 207, 215, 34, 0, 0, 0, 1, 205, 50, 0, 0],
            &[26, 3, 187, 34, 0, 0, 0, 84, 173, 0, 0, 0],
            &[0, 0, 189, 34, 0, 0, 2, 211, 42, 0, 0, 0],
            &[0, 0, 189, 34, 0, 0, 93, 163, 0, 0, 0, 0],
            &[0, 0, 189, 34, 0, 5, 216, 35, 0, 0, 0, 0],
            &[52, 178, 255, 219, 120, 103, 154, 0, 0, 0, 0, 0],
            &[2, 6, 12, 7, 12, 220, 36, 117, 178, 148, 20, 0],
            &[0, 0, 0, 0, 112, 145, 31, 105, 25, 132, 157, 0],
            &[0, 0, 0, 11, 222, 23, 0, 0, 0, 64, 168, 0],
            &[0, 0, 0, 121, 135, 0, 0, 0, 8, 194, 56, 0],
            &[0, 0, 15, 223, 18, 0, 0, 7, 180, 88, 0, 0],
            &[0, 0, 131, 126, 0, 0, 8, 179, 87, 0, 0, 0],
            &[0, 20, 223, 13, 0, 0, 128, 244, 213, 213, 193, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¾' / 0xbe
        '¾' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 132, 178, 145, 20, 0, 0, 0, 58, 86, 0, 0],
            &[40, 81, 25, 139, 152, 0, 0, 1, 205, 50, 0, 0],
            &[0, 0, 0, 112, 141, 0, 0, 84, 173, 0, 0, 0],
            &[0, 123, 208, 228, 25, 0, 2, 211, 42, 0, 0, 0],
            &[0, 0, 8, 94, 173, 0, 93, 163, 0, 0, 0, 0],
            &[4, 0, 0, 49, 200, 5, 216, 35, 0, 0, 0, 0],
            &[107, 195, 172, 213, 68, 103, 154, 0, 0, 0, 0, 0],
            &[0, 13, 27, 3, 8, 220, 29, 0, 56, 126, 0, 0],
            &[0, 0, 0, 0, 112, 145, 0, 14, 195, 200, 0, 0],
            &[0, 0, 0, 11, 222, 23, 0, 161, 81, 200, 0, 0],
            &[0, 0, 0, 121, 135, 0, 88, 139, 17, 200, 0, 0],
            &[0, 0, 15, 223, 18, 21, 231, 112, 116, 223, 89, 0],
            &[0, 0, 131, 126, 0, 16, 88, 88, 100, 219, 75, 0],
            &[0, 20, 223, 13, 0, 0, 0, 0, 17, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¿' / 0xbf
        '¿' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 181, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 114, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 60, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 146, 108, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 227, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 207, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 223, 100, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 211, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 61, 223, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 82, 195, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 240, 21, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 156, 215, 104, 76, 102, 167, 156, 0, 0],
            &[0, 0, 0, 0, 79, 154, 173, 149, 103, 20, 0, 0],
        ]),
        // letter: 'À' / 0xc0
        'À' => Some(&[
            &[0, 0, 0, 88, 233, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 185, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 117, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 153, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 244, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 127, 239, 12, 0, 0, 0, 0],
            &[0, 0, 0, 6, 237, 30, 178, 91, 0, 0, 0, 0],
            &[0, 0, 0, 78, 197, 0, 90, 181, 0, 0, 0, 0],
            &[0, 0, 0, 167, 110, 0, 12, 241, 19, 0, 0, 0],
            &[0, 0, 11, 242, 25, 0, 0, 172, 103, 0, 0, 0],
            &[0, 0, 90, 190, 0, 0, 0, 85, 192, 0, 0, 0],
            &[0, 0, 182, 255, 169, 169, 169, 179, 255, 28, 0, 0],
            &[0, 18, 255, 108, 88, 88, 88, 88, 212, 115, 0, 0],
            &[0, 102, 183, 0, 0, 0, 0, 0, 79, 204, 0, 0],
            &[0, 191, 96, 0, 0, 0, 0, 0, 7, 240, 38, 0],
            &[27, 246, 15, 0, 0, 0, 0, 0, 0, 161, 127, 0],
            &[114, 176, 0, 0, 0, 0, 0, 0, 0, 74, 216, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Á' / 0xc1
        'Á' => Some(&[
            &[0, 0, 0, 0, 0, 0, 63, 239, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 219, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 185, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 153, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 244, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 127, 239, 12, 0, 0, 0, 0],
            &[0, 0, 0, 6, 237, 30, 178, 91, 0, 0, 0, 0],
            &[0, 0, 0, 78, 197, 0, 90, 181, 0, 0, 0, 0],
            &[0, 0, 0, 167, 110, 0, 12, 241, 19, 0, 0, 0],
            &[0, 0, 11, 242, 25, 0, 0, 172, 103, 0, 0, 0],
            &[0, 0, 90, 190, 0, 0, 0, 85, 192, 0, 0, 0],
            &[0, 0, 182, 255, 169, 169, 169, 179, 255, 28, 0, 0],
            &[0, 18, 255, 108, 88, 88, 88, 88, 212, 115, 0, 0],
            &[0, 102, 183, 0, 0, 0, 0, 0, 79, 204, 0, 0],
            &[0, 191, 96, 0, 0, 0, 0, 0, 7, 240, 38, 0],
            &[27, 246, 15, 0, 0, 0, 0, 0, 0, 161, 127, 0],
            &[114, 176, 0, 0, 0, 0, 0, 0, 0, 74, 216, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0],
        ]),
        // letter: 'Â' / 0xc2
        'Â' => Some(&[
            &[0, 0, 0, 0, 47, 246, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 208, 63, 191, 88, 0, 0, 0, 0],
            &[0, 0, 4, 180, 67, 0, 13, 186, 56, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 2, 3, 0, 0, 0],
            &[0, 0, 0, 0, 2, 153, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 244, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 127, 239, 12, 0, 0, 0, 0],
            &[0, 0, 0, 6, 237, 30, 178, 91, 0, 0, 0, 0],
            &[0, 0, 0, 78, 197, 0, 90, 181, 0, 0, 0, 0],
            &[0, 0, 0, 167, 110, 0, 12, 241, 19, 0, 0, 0],
            &[0, 0, 11, 242, 25, 0, 0, 172, 103, 0, 0, 0],
            &[0, 0, 90, 190, 0, 0, 0, 85, 192, 0, 0, 0],
            &[0, 0, 182, 255, 169, 169, 169, 179, 255, 28, 0, 0],
            &[0, 18, 255, 108, 88, 88, 88, 88, 212, 115, 0, 0],
            &[0, 102, 183, 0, 0, 0, 0, 0, 79, 204, 0, 0],
            &[0, 191, 96, 0, 0, 0, 0, 0, 7, 240, 38, 0],
            &[27, 246, 15, 0, 0, 0, 0, 0, 0, 161, 127, 0],
            &[114, 176, 0, 0, 0, 0, 0, 0, 0, 74, 216, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ã' / 0xc3
        'Ã' => Some(&[
            &[0, 0, 0, 14, 43, 2, 0, 0, 48, 0, 0, 0],
            &[0, 0, 28, 218, 191, 201, 50, 39, 193, 0, 0, 0],
            &[0, 0, 105, 85, 0, 58, 195, 208, 54, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 153, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 244, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 127, 239, 12, 0, 0, 0, 0],
            &[0, 0, 0, 6, 237, 30, 178, 91, 0, 0, 0, 0],
            &[0, 0, 0, 78, 197, 0, 90, 181, 0, 0, 0, 0],
            &[0, 0, 0, 167, 110, 0, 12, 241, 19, 0, 0, 0],
            &[0, 0, 11, 242, 25, 0, 0, 172, 103, 0, 0, 0],
            &[0, 0, 90, 190, 0, 0, 0, 85, 192, 0, 0, 0],
            &[0, 0, 182, 255, 169, 169, 169, 179, 255, 28, 0, 0],
            &[0, 18, 255, 108, 88, 88, 88, 88, 212, 115, 0, 0],
            &[0, 102, 183, 0, 0, 0, 0, 0, 79, 204, 0, 0],
            &[0, 191, 96, 0, 0, 0, 0, 0, 7, 240, 38, 0],
            &[27, 246, 15, 0, 0, 0, 0, 0, 0, 161, 127, 0],
            &[114, 176, 0, 0, 0, 0, 0, 0, 0, 74, 216, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ä' / 0xc4
        'Ä' => Some(&[
            &[0, 0, 0, 35, 5, 0, 0, 33, 6, 0, 0, 0],
            &[0, 0, 4, 249, 98, 0, 1, 244, 105, 0, 0, 0],
            &[0, 0, 0, 96, 25, 0, 0, 93, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 153, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 244, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 127, 239, 12, 0, 0, 0, 0],
            &[0, 0, 0, 6, 237, 30, 178, 91, 0, 0, 0, 0],
            &[0, 0, 0, 78, 197, 0, 90, 181, 0, 0, 0, 0],
            &[0, 0, 0, 167, 110, 0, 12, 241, 19, 0, 0, 0],
            &[0, 0, 11, 242, 25, 0, 0, 172, 103, 0, 0, 0],
            &[0, 0, 90, 190, 0, 0, 0, 85, 192, 0, 0, 0],
            &[0, 0, 182, 255, 169, 169, 169, 179, 255, 28, 0, 0],
            &[0, 18, 255, 108, 88, 88, 88, 88, 212, 115, 0, 0],
            &[0, 102, 183, 0, 0, 0, 0, 0, 79, 204, 0, 0],
            &[0, 191, 96, 0, 0, 0, 0, 0, 7, 240, 38, 0],
            &[27, 246, 15, 0, 0, 0, 0, 0, 0, 161, 127, 0],
            &[114, 176, 0, 0, 0, 0, 0, 0, 0, 74, 216, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Å' / 0xc5
        'Å' => Some(&[
            &[0, 0, 0, 48, 172, 0, 69, 150, 0, 0, 0, 0],
            &[0, 0, 0, 53, 166, 0, 64, 153, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 196, 190, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 153, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 244, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 127, 239, 12, 0, 0, 0, 0],
            &[0, 0, 0, 6, 237, 30, 178, 91, 0, 0, 0, 0],
            &[0, 0, 0, 78, 197, 0, 90, 181, 0, 0, 0, 0],
            &[0, 0, 0, 167, 110, 0, 12, 241, 19, 0, 0, 0],
            &[0, 0, 11, 242, 25, 0, 0, 172, 103, 0, 0, 0],
            &[0, 0, 90, 190, 0, 0, 0, 85, 192, 0, 0, 0],
            &[0, 0, 182, 255, 169, 169, 169, 179, 255, 28, 0, 0],
            &[0, 18, 255, 108, 88, 88, 88, 88, 212, 115, 0, 0],
            &[0, 102, 183, 0, 0, 0, 0, 0, 79, 204, 0, 0],
            &[0, 191, 96, 0, 0, 0, 0, 0, 7, 240, 38, 0],
            &[27, 246, 15, 0, 0, 0, 0, 0, 0, 161, 127, 0],
            &[114, 176, 0, 0, 0, 0, 0, 0, 0, 74, 216, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 191, 179, 26, 0, 0, 0, 0],
        ]),
        // letter: 'Æ' / 0xc6
        'Æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 144, 144, 144, 144, 144, 103, 0],
            &[0, 0, 0, 8, 240, 162, 212, 107, 107, 107, 77, 0],
            &[0, 0, 0, 73, 191, 91, 180, 0, 0, 0, 0, 0],
            &[0, 0, 0, 147, 118, 91, 180, 0, 0, 0, 0, 0],
            &[0, 0, 0, 221, 46, 91, 180, 0, 0, 0, 0, 0],
            &[0, 0, 39, 228, 0, 91, 180, 0, 0, 0, 0, 0],
            &[0, 0, 114, 156, 0, 91, 227, 158, 158, 158, 62, 0],
            &[0, 0, 188, 83, 0, 91, 208, 94, 94, 94, 37, 0],
            &[0, 12, 255, 184, 169, 255, 183, 0, 0, 0, 0, 0],
            &[0, 80, 231, 78, 78, 168, 180, 0, 0, 0, 0, 0],
            &[0, 154, 121, 0, 0, 91, 180, 0, 0, 0, 0, 0],
            &[0, 228, 49, 0, 0, 91, 180, 0, 0, 0, 0, 0],
            &[47, 231, 1, 0, 0, 91, 181, 1, 1, 1, 1, 0],
            &[121, 159, 0, 0, 0, 91, 255, 255, 255, 255, 183, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ç' / 0xc7
        'Ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 134, 178, 183, 156, 89, 10, 0],
            &[0, 0, 0, 124, 237, 135, 75, 70, 104, 185, 55, 0],
            &[0, 0, 102, 225, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 238, 68, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 85, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 175, 115, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 156, 133, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 119, 177, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 241, 14, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 191, 141, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 232, 146, 22, 0, 0, 9, 73, 3, 0],
            &[0, 0, 0, 35, 167, 250, 240, 229, 249, 198, 5, 0],
            &[0, 0, 0, 0, 0, 18, 227, 36, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 67, 197, 132, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 200, 43, 0, 0, 0],
            &[0, 0, 0, 0, 32, 138, 117, 29, 0, 0, 0, 0],
        ]),
        // letter: 'È' / 0xc8
        'È' => Some(&[
            &[0, 0, 0, 55, 242, 45, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 210, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 173, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 144, 144, 144, 144, 108, 0, 0],
            &[0, 1, 255, 122, 107, 107, 107, 107, 107, 80, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 167, 158, 158, 158, 158, 158, 58, 0, 0],
            &[0, 1, 255, 110, 94, 94, 94, 94, 94, 34, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 26, 1, 1, 1, 1, 1, 1, 0, 0],
            &[0, 1, 255, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'É' / 0xc9
        'É' => Some(&[
            &[0, 0, 0, 0, 0, 0, 34, 240, 67, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 202, 99, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 161, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 144, 144, 144, 144, 108, 0, 0],
            &[0, 1, 255, 122, 107, 107, 107, 107, 107, 80, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 167, 158, 158, 158, 158, 158, 58, 0, 0],
            &[0, 1, 255, 110, 94, 94, 94, 94, 94, 34, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 26, 1, 1, 1, 1, 1, 1, 0, 0],
            &[0, 1, 255, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0],
        ]),
        // letter: 'Ê' / 0xca
        'Ê' => Some(&[
            &[0, 0, 0, 0, 23, 231, 187, 1, 0, 0, 0, 0],
            &[0, 0, 0, 4, 190, 87, 159, 127, 0, 0, 0, 0],
            &[0, 0, 0, 154, 98, 0, 4, 164, 88, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 0, 4, 0, 0, 0],
            &[0, 0, 144, 144, 144, 144, 144, 144, 144, 108, 0, 0],
            &[0, 1, 255, 122, 107, 107, 107, 107, 107, 80, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 167, 158, 158, 158, 158, 158, 58, 0, 0],
            &[0, 1, 255, 110, 94, 94, 94, 94, 94, 34, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 26, 1, 1, 1, 1, 1, 1, 0, 0],
            &[0, 1, 255, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ë' / 0xcb
        'Ë' => Some(&[
            &[0, 0, 0, 29, 11, 0, 0, 26, 13, 0, 0, 0],
            &[0, 0, 0, 215, 137, 0, 0, 207, 144, 0, 0, 0],
            &[0, 0, 0, 80, 41, 0, 0, 76, 45, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 144, 144, 144, 144, 108, 0, 0],
            &[0, 1, 255, 122, 107, 107, 107, 107, 107, 80, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 167, 158, 158, 158, 158, 158, 58, 0, 0],
            &[0, 1, 255, 110, 94, 94, 94, 94, 94, 34, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 26, 1, 1, 1, 1, 1, 1, 0, 0],
            &[0, 1, 255, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ì' / 0xcc
        'Ì' => Some(&[
            &[0, 0, 0, 88, 233, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 185, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 117, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0],
            &[0, 0, 63, 144, 144, 144, 144, 144, 120, 0, 0, 0],
            &[0, 0, 25, 68, 83, 231, 129, 74, 50, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 88, 213, 228, 253, 238, 219, 171, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Í' / 0xcd
        'Í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 63, 239, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 219, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 185, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 144, 144, 144, 144, 144, 120, 0, 0, 0],
            &[0, 0, 25, 68, 83, 231, 129, 74, 50, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 88, 213, 228, 253, 238, 219, 171, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0],
        ]),
        // letter: 'Î' / 0xce
        'Î' => Some(&[
            &[0, 0, 0, 0, 47, 246, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 208, 63, 191, 88, 0, 0, 0, 0],
            &[0, 0, 4, 180, 67, 0, 13, 186, 56, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 2, 3, 0, 0, 0],
            &[0, 0, 63, 144, 144, 144, 144, 144, 120, 0, 0, 0],
            &[0, 0, 25, 68, 83, 231, 129, 74, 50, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 88, 213, 228, 253, 238, 219, 171, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ï' / 0xcf
        'Ï' => Some(&[
            &[0, 0, 0, 35, 5, 0, 0, 33, 6, 0, 0, 0],
            &[0, 0, 4, 249, 98, 0, 1, 244, 105, 0, 0, 0],
            &[0, 0, 0, 96, 25, 0, 0, 93, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 144, 144, 144, 144, 144, 120, 0, 0, 0],
            &[0, 0, 25, 68, 83, 231, 129, 74, 50, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 88, 213, 228, 253, 238, 219, 171, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ð' / 0xd0
        'Ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 144, 144, 143, 124, 87, 10, 0, 0, 0, 0],
            &[0, 59, 235, 103, 105, 128, 185, 239, 94, 0, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 56, 233, 89, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 78, 235, 7, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 0, 222, 74, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 0, 159, 129, 0],
            &[80, 140, 255, 80, 80, 31, 0, 0, 0, 135, 151, 0],
            &[167, 226, 255, 167, 167, 66, 0, 0, 0, 128, 154, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 0, 148, 132, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 0, 197, 90, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 36, 248, 16, 0],
            &[0, 59, 222, 0, 0, 0, 0, 9, 193, 135, 0, 0],
            &[0, 59, 222, 0, 2, 24, 89, 211, 173, 7, 0, 0],
            &[0, 59, 255, 252, 251, 228, 184, 87, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ñ' / 0xd1
        'Ñ' => Some(&[
            &[0, 0, 0, 10, 45, 4, 0, 0, 45, 2, 0, 0],
            &[0, 0, 15, 212, 189, 209, 62, 27, 206, 0, 0, 0],
            &[0, 0, 83, 107, 0, 45, 186, 214, 70, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 144, 15, 0, 0, 0, 0, 34, 121, 0, 0],
            &[0, 108, 249, 124, 0, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 167, 229, 12, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 154, 149, 119, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 161, 30, 228, 10, 0, 0, 61, 215, 0, 0],
            &[0, 108, 163, 0, 155, 114, 0, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 34, 227, 8, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 161, 109, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 38, 225, 7, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 166, 105, 60, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 43, 223, 60, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 172, 146, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 47, 243, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 0, 177, 215, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ò' / 0xd2
        'Ò' => Some(&[
            &[0, 0, 0, 88, 233, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 185, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 117, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 148, 187, 176, 103, 4, 0, 0, 0],
            &[0, 0, 80, 238, 124, 66, 82, 190, 189, 5, 0, 0],
            &[0, 20, 240, 61, 0, 0, 0, 2, 191, 123, 0, 0],
            &[0, 125, 172, 0, 0, 0, 0, 0, 60, 231, 1, 0],
            &[0, 193, 93, 0, 0, 0, 0, 0, 2, 238, 47, 0],
            &[0, 239, 47, 0, 0, 0, 0, 0, 0, 198, 88, 0],
            &[2, 254, 28, 0, 0, 0, 0, 0, 0, 178, 107, 0],
            &[8, 255, 23, 0, 0, 0, 0, 0, 0, 176, 110, 0],
            &[0, 250, 37, 0, 0, 0, 0, 0, 0, 190, 97, 0],
            &[0, 216, 73, 0, 0, 0, 0, 0, 0, 224, 63, 0],
            &[0, 158, 135, 0, 0, 0, 0, 0, 28, 248, 12, 0],
            &[0, 66, 232, 14, 0, 0, 0, 0, 135, 170, 0, 0],
            &[0, 0, 173, 186, 25, 0, 1, 99, 238, 37, 0, 0],
            &[0, 0, 9, 149, 244, 221, 239, 205, 51, 0, 0, 0],
            &[0, 0, 0, 0, 11, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ó' / 0xd3
        'Ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 63, 239, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 219, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 185, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 148, 187, 176, 103, 4, 0, 0, 0],
            &[0, 0, 80, 238, 124, 66, 82, 190, 189, 5, 0, 0],
            &[0, 20, 240, 61, 0, 0, 0, 2, 191, 123, 0, 0],
            &[0, 125, 172, 0, 0, 0, 0, 0, 60, 231, 1, 0],
            &[0, 193, 93, 0, 0, 0, 0, 0, 2, 238, 47, 0],
            &[0, 239, 47, 0, 0, 0, 0, 0, 0, 198, 88, 0],
            &[2, 254, 28, 0, 0, 0, 0, 0, 0, 178, 107, 0],
            &[8, 255, 23, 0, 0, 0, 0, 0, 0, 176, 110, 0],
            &[0, 250, 37, 0, 0, 0, 0, 0, 0, 190, 97, 0],
            &[0, 216, 73, 0, 0, 0, 0, 0, 0, 224, 63, 0],
            &[0, 158, 135, 0, 0, 0, 0, 0, 28, 248, 12, 0],
            &[0, 66, 232, 14, 0, 0, 0, 0, 135, 170, 0, 0],
            &[0, 0, 173, 186, 25, 0, 1, 99, 238, 37, 0, 0],
            &[0, 0, 9, 149, 244, 221, 239, 205, 51, 0, 0, 0],
            &[0, 0, 0, 0, 11, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0],
        ]),
        // letter: 'Ô' / 0xd4
        'Ô' => Some(&[
            &[0, 0, 0, 0, 47, 246, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 208, 63, 191, 88, 0, 0, 0, 0],
            &[0, 0, 4, 180, 67, 0, 13, 186, 56, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 2, 3, 0, 0, 0],
            &[0, 0, 0, 40, 148, 187, 176, 103, 4, 0, 0, 0],
            &[0, 0, 80, 238, 124, 66, 82, 190, 189, 5, 0, 0],
            &[0, 20, 240, 61, 0, 0, 0, 2, 191, 123, 0, 0],
            &[0, 125, 172, 0, 0, 0, 0, 0, 60, 231, 1, 0],
            &[0, 193, 93, 0, 0, 0, 0, 0, 2, 238, 47, 0],
            &[0, 239, 47, 0, 0, 0, 0, 0, 0, 198, 88, 0],
            &[2, 254, 28, 0, 0, 0, 0, 0, 0, 178, 107, 0],
            &[8, 255, 23, 0, 0, 0, 0, 0, 0, 176, 110, 0],
            &[0, 250, 37, 0, 0, 0, 0, 0, 0, 190, 97, 0],
            &[0, 216, 73, 0, 0, 0, 0, 0, 0, 224, 63, 0],
            &[0, 158, 135, 0, 0, 0, 0, 0, 28, 248, 12, 0],
            &[0, 66, 232, 14, 0, 0, 0, 0, 135, 170, 0, 0],
            &[0, 0, 173, 186, 25, 0, 1, 99, 238, 37, 0, 0],
            &[0, 0, 9, 149, 244, 221, 239, 205, 51, 0, 0, 0],
            &[0, 0, 0, 0, 11, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Õ' / 0xd5
        'Õ' => Some(&[
            &[0, 0, 0, 14, 43, 2, 0, 0, 48, 0, 0, 0],
            &[0, 0, 28, 218, 191, 201, 50, 39, 193, 0, 0, 0],
            &[0, 0, 105, 85, 0, 58, 195, 208, 54, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 148, 187, 176, 103, 4, 0, 0, 0],
            &[0, 0, 80, 238, 124, 66, 82, 190, 189, 5, 0, 0],
            &[0, 20, 240, 61, 0, 0, 0, 2, 191, 123, 0, 0],
            &[0, 125, 172, 0, 0, 0, 0, 0, 60, 231, 1, 0],
            &[0, 193, 93, 0, 0, 0, 0, 0, 2, 238, 47, 0],
            &[0, 239, 47, 0, 0, 0, 0, 0, 0, 198, 88, 0],
            &[2, 254, 28, 0, 0, 0, 0, 0, 0, 178, 107, 0],
            &[8, 255, 23, 0, 0, 0, 0, 0, 0, 176, 110, 0],
            &[0, 250, 37, 0, 0, 0, 0, 0, 0, 190, 97, 0],
            &[0, 216, 73, 0, 0, 0, 0, 0, 0, 224, 63, 0],
            &[0, 158, 135, 0, 0, 0, 0, 0, 28, 248, 12, 0],
            &[0, 66, 232, 14, 0, 0, 0, 0, 135, 170, 0, 0],
            &[0, 0, 173, 186, 25, 0, 1, 99, 238, 37, 0, 0],
            &[0, 0, 9, 149, 244, 221, 239, 205, 51, 0, 0, 0],
            &[0, 0, 0, 0, 11, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ö' / 0xd6
        'Ö' => Some(&[
            &[0, 0, 0, 35, 5, 0, 0, 33, 6, 0, 0, 0],
            &[0, 0, 4, 249, 98, 0, 1, 244, 105, 0, 0, 0],
            &[0, 0, 0, 96, 25, 0, 0, 93, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 148, 187, 176, 103, 4, 0, 0, 0],
            &[0, 0, 80, 238, 124, 66, 82, 190, 189, 5, 0, 0],
            &[0, 20, 240, 61, 0, 0, 0, 2, 191, 123, 0, 0],
            &[0, 125, 172, 0, 0, 0, 0, 0, 60, 231, 1, 0],
            &[0, 193, 93, 0, 0, 0, 0, 0, 2, 238, 47, 0],
            &[0, 239, 47, 0, 0, 0, 0, 0, 0, 198, 88, 0],
            &[2, 254, 28, 0, 0, 0, 0, 0, 0, 178, 107, 0],
            &[8, 255, 23, 0, 0, 0, 0, 0, 0, 176, 110, 0],
            &[0, 250, 37, 0, 0, 0, 0, 0, 0, 190, 97, 0],
            &[0, 216, 73, 0, 0, 0, 0, 0, 0, 224, 63, 0],
            &[0, 158, 135, 0, 0, 0, 0, 0, 28, 248, 12, 0],
            &[0, 66, 232, 14, 0, 0, 0, 0, 135, 170, 0, 0],
            &[0, 0, 173, 186, 25, 0, 1, 99, 238, 37, 0, 0],
            &[0, 0, 9, 149, 244, 221, 239, 205, 51, 0, 0, 0],
            &[0, 0, 0, 0, 11, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '×' / 0xd7
        '×' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 123, 0, 0, 0, 0, 0, 51, 102, 0, 0],
            &[0, 6, 178, 139, 0, 0, 0, 51, 220, 49, 0, 0],
            &[0, 0, 6, 178, 138, 0, 51, 221, 50, 0, 0, 0],
            &[0, 0, 0, 6, 178, 159, 221, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 57, 253, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 51, 222, 59, 180, 141, 0, 0, 0, 0],
            &[0, 0, 51, 222, 52, 0, 6, 178, 141, 0, 0, 0],
            &[0, 32, 222, 52, 0, 0, 0, 5, 175, 123, 0, 0],
            &[0, 1, 32, 0, 0, 0, 0, 0, 5, 27, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ø' / 0xd8
        'Ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0],
            &[0, 0, 0, 40, 148, 187, 176, 103, 20, 212, 8, 0],
            &[0, 0, 80, 238, 124, 66, 82, 190, 255, 134, 0, 0],
            &[0, 20, 240, 61, 0, 0, 0, 41, 255, 133, 0, 0],
            &[0, 125, 172, 0, 0, 0, 0, 179, 153, 231, 1, 0],
            &[0, 193, 93, 0, 0, 0, 69, 201, 3, 238, 47, 0],
            &[0, 239, 47, 0, 0, 3, 211, 57, 0, 198, 88, 0],
            &[2, 254, 28, 0, 0, 106, 166, 0, 0, 178, 107, 0],
            &[8, 255, 23, 0, 16, 226, 29, 0, 0, 176, 110, 0],
            &[0, 250, 37, 0, 143, 130, 0, 0, 0, 190, 97, 0],
            &[0, 216, 73, 39, 223, 11, 0, 0, 0, 224, 63, 0],
            &[0, 158, 135, 180, 94, 0, 0, 0, 28, 248, 12, 0],
            &[0, 66, 255, 217, 1, 0, 0, 0, 135, 170, 0, 0],
            &[0, 4, 255, 245, 25, 0, 1, 99, 238, 37, 0, 0],
            &[0, 107, 177, 149, 244, 221, 239, 205, 51, 0, 0, 0],
            &[0, 79, 30, 0, 11, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ù' / 0xd9
        'Ù' => Some(&[
            &[0, 0, 0, 88, 233, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 185, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 117, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 37, 121, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 102, 180, 0, 0, 0, 0, 0, 73, 209, 0, 0],
            &[0, 80, 206, 0, 0, 0, 0, 0, 100, 186, 0, 0],
            &[0, 25, 250, 24, 0, 0, 0, 0, 173, 126, 0, 0],
            &[0, 0, 168, 181, 18, 0, 2, 103, 238, 23, 0, 0],
            &[0, 0, 12, 165, 244, 217, 237, 208, 53, 0, 0, 0],
            &[0, 0, 0, 0, 13, 40, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ú' / 0xda
        'Ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 63, 239, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 219, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 185, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 37, 121, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 102, 180, 0, 0, 0, 0, 0, 73, 209, 0, 0],
            &[0, 80, 206, 0, 0, 0, 0, 0, 100, 186, 0, 0],
            &[0, 25, 250, 24, 0, 0, 0, 0, 173, 126, 0, 0],
            &[0, 0, 168, 181, 18, 0, 2, 103, 238, 23, 0, 0],
            &[0, 0, 12, 165, 244, 217, 237, 208, 53, 0, 0, 0],
            &[0, 0, 0, 0, 13, 40, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0],
        ]),
        // letter: 'Û' / 0xdb
        'Û' => Some(&[
            &[0, 0, 0, 0, 47, 246, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 208, 63, 191, 88, 0, 0, 0, 0],
            &[0, 0, 4, 180, 67, 0, 13, 186, 56, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 2, 3, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 37, 121, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 102, 180, 0, 0, 0, 0, 0, 73, 209, 0, 0],
            &[0, 80, 206, 0, 0, 0, 0, 0, 100, 186, 0, 0],
            &[0, 25, 250, 24, 0, 0, 0, 0, 173, 126, 0, 0],
            &[0, 0, 168, 181, 18, 0, 2, 103, 238, 23, 0, 0],
            &[0, 0, 12, 165, 244, 217, 237, 208, 53, 0, 0, 0],
            &[0, 0, 0, 0, 13, 40, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ü' / 0xdc
        'Ü' => Some(&[
            &[0, 0, 0, 35, 5, 0, 0, 33, 6, 0, 0, 0],
            &[0, 0, 4, 249, 98, 0, 1, 244, 105, 0, 0, 0],
            &[0, 0, 0, 96, 25, 0, 0, 93, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 37, 121, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 102, 180, 0, 0, 0, 0, 0, 73, 209, 0, 0],
            &[0, 80, 206, 0, 0, 0, 0, 0, 100, 186, 0, 0],
            &[0, 25, 250, 24, 0, 0, 0, 0, 173, 126, 0, 0],
            &[0, 0, 168, 181, 18, 0, 2, 103, 238, 23, 0, 0],
            &[0, 0, 12, 165, 244, 217, 237, 208, 53, 0, 0, 0],
            &[0, 0, 0, 0, 13, 40, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ý' / 0xdd
        'Ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 63, 239, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 219, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 185, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0],
            &[8, 141, 25, 0, 0, 0, 0, 0, 0, 106, 65, 0],
            &[0, 159, 144, 0, 0, 0, 0, 0, 35, 240, 23, 0],
            &[0, 32, 240, 25, 0, 0, 0, 0, 157, 134, 0, 0],
            &[0, 0, 147, 144, 0, 0, 0, 33, 234, 17, 0, 0],
            &[0, 0, 24, 236, 25, 0, 0, 155, 123, 0, 0, 0],
            &[0, 0, 0, 135, 144, 0, 31, 228, 11, 0, 0, 0],
            &[0, 0, 0, 17, 231, 25, 152, 112, 0, 0, 0, 0],
            &[0, 0, 0, 0, 123, 171, 220, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 236, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0],
        ]),
        // letter: 'Þ' / 0xde
        'Þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 14, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 248, 247, 248, 239, 200, 108, 3, 0, 0],
            &[0, 1, 255, 24, 0, 1, 22, 81, 217, 156, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 41, 254, 29, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 229, 73, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 226, 63, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 46, 243, 14, 0],
            &[0, 1, 255, 24, 0, 0, 9, 78, 223, 114, 0, 0],
            &[0, 1, 255, 236, 233, 238, 247, 194, 82, 0, 0, 0],
            &[0, 1, 255, 41, 18, 14, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ß' / 0xdf
        'ß' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 101, 130, 119, 57, 0, 0, 0, 0],
            &[0, 0, 79, 238, 156, 112, 133, 225, 151, 0, 0, 0],
            &[0, 7, 235, 78, 0, 0, 0, 28, 253, 31, 0, 0],
            &[0, 48, 233, 0, 0, 0, 0, 8, 248, 29, 0, 0],
            &[0, 64, 213, 0, 0, 0, 1, 142, 182, 0, 0, 0],
            &[0, 64, 212, 0, 0, 3, 170, 174, 11, 0, 0, 0],
            &[0, 64, 212, 0, 0, 82, 208, 3, 0, 0, 0, 0],
            &[0, 64, 212, 0, 0, 102, 182, 0, 0, 0, 0, 0],
            &[0, 64, 212, 0, 0, 20, 220, 139, 5, 0, 0, 0],
            &[0, 64, 212, 0, 0, 0, 22, 185, 200, 28, 0, 0],
            &[0, 64, 212, 0, 0, 0, 0, 1, 134, 212, 10, 0],
            &[0, 64, 212, 0, 0, 0, 0, 0, 0, 210, 84, 0],
            &[0, 64, 212, 0, 0, 0, 0, 0, 0, 180, 101, 0],
            &[0, 64, 212, 0, 44, 13, 0, 0, 40, 240, 43, 0],
            &[0, 64, 212, 0, 99, 234, 210, 204, 232, 100, 0, 0],
            &[0, 0, 0, 0, 0, 4, 32, 38, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'à' / 0xe0
        'à' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 129, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 195, 123, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 196, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 36, 66, 51, 12, 0, 0, 0, 0],
            &[0, 0, 84, 225, 202, 171, 189, 234, 99, 0, 0, 0],
            &[0, 0, 33, 30, 0, 0, 0, 42, 239, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 167, 99, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 137, 126, 0, 0],
            &[0, 0, 3, 89, 166, 205, 216, 214, 236, 127, 0, 0],
            &[0, 4, 190, 181, 67, 15, 0, 0, 134, 127, 0, 0],
            &[0, 83, 204, 1, 0, 0, 0, 0, 153, 127, 0, 0],
            &[0, 111, 161, 0, 0, 0, 0, 7, 225, 127, 0, 0],
            &[0, 59, 231, 30, 0, 0, 19, 169, 192, 127, 0, 0],
            &[0, 0, 119, 238, 202, 204, 209, 72, 110, 127, 0, 0],
            &[0, 0, 0, 7, 37, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'á' / 0xe1
        'á' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 134, 53, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 135, 186, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 82, 192, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 107, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 36, 66, 51, 12, 0, 0, 0, 0],
            &[0, 0, 84, 225, 202, 171, 189, 234, 99, 0, 0, 0],
            &[0, 0, 33, 30, 0, 0, 0, 42, 239, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 167, 99, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 137, 126, 0, 0],
            &[0, 0, 3, 89, 166, 205, 216, 214, 236, 127, 0, 0],
            &[0, 4, 190, 181, 67, 15, 0, 0, 134, 127, 0, 0],
            &[0, 83, 204, 1, 0, 0, 0, 0, 153, 127, 0, 0],
            &[0, 111, 161, 0, 0, 0, 0, 7, 225, 127, 0, 0],
            &[0, 59, 231, 30, 0, 0, 19, 169, 192, 127, 0, 0],
            &[0, 0, 119, 238, 202, 204, 209, 72, 110, 127, 0, 0],
            &[0, 0, 0, 7, 37, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'â' / 0xe2
        'â' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 122, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 118, 184, 222, 35, 0, 0, 0, 0],
            &[0, 0, 0, 74, 186, 8, 58, 200, 15, 0, 0, 0],
            &[0, 0, 0, 106, 9, 0, 0, 52, 64, 0, 0, 0],
            &[0, 0, 0, 2, 36, 66, 51, 12, 0, 0, 0, 0],
            &[0, 0, 84, 225, 202, 171, 189, 234, 99, 0, 0, 0],
            &[0, 0, 33, 30, 0, 0, 0, 42, 239, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 167, 99, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 137, 126, 0, 0],
            &[0, 0, 3, 89, 166, 205, 216, 214, 236, 127, 0, 0],
            &[0, 4, 190, 181, 67, 15, 0, 0, 134, 127, 0, 0],
            &[0, 83, 204, 1, 0, 0, 0, 0, 153, 127, 0, 0],
            &[0, 111, 161, 0, 0, 0, 0, 7, 225, 127, 0, 0],
            &[0, 59, 231, 30, 0, 0, 19, 169, 192, 127, 0, 0],
            &[0, 0, 119, 238, 202, 204, 209, 72, 110, 127, 0, 0],
            &[0, 0, 0, 7, 37, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ã' / 0xe3
        'ã' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 116, 184, 104, 1, 0, 159, 2, 0, 0],
            &[0, 0, 59, 180, 51, 156, 198, 166, 159, 0, 0, 0],
            &[0, 0, 39, 32, 0, 0, 48, 75, 3, 0, 0, 0],
            &[0, 0, 0, 2, 36, 66, 51, 12, 0, 0, 0, 0],
            &[0, 0, 84, 225, 202, 171, 189, 234, 99, 0, 0, 0],
            &[0, 0, 33, 30, 0, 0, 0, 42, 239, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 167, 99, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 137, 126, 0, 0],
            &[0, 0, 3, 89, 166, 205, 216, 214, 236, 127, 0, 0],
            &[0, 4, 190, 181, 67, 15, 0, 0, 134, 127, 0, 0],
            &[0, 83, 204, 1, 0, 0, 0, 0, 153, 127, 0, 0],
            &[0, 111, 161, 0, 0, 0, 0, 7, 225, 127, 0, 0],
            &[0, 59, 231, 30, 0, 0, 19, 169, 192, 127, 0, 0],
            &[0, 0, 119, 238, 202, 204, 209, 72, 110, 127, 0, 0],
            &[0, 0, 0, 7, 37, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ä' / 0xe4
        'ä' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 153, 71, 0, 0, 145, 76, 0, 0, 0],
            &[0, 0, 0, 196, 95, 0, 0, 188, 101, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 36, 66, 51, 12, 0, 0, 0, 0],
            &[0, 0, 84, 225, 202, 171, 189, 234, 99, 0, 0, 0],
            &[0, 0, 33, 30, 0, 0, 0, 42, 239, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 167, 99, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 137, 126, 0, 0],
            &[0, 0, 3, 89, 166, 205, 216, 214, 236, 127, 0, 0],
            &[0, 4, 190, 181, 67, 15, 0, 0, 134, 127, 0, 0],
            &[0, 83, 204, 1, 0, 0, 0, 0, 153, 127, 0, 0],
            &[0, 111, 161, 0, 0, 0, 0, 7, 225, 127, 0, 0],
            &[0, 59, 231, 30, 0, 0, 19, 169, 192, 127, 0, 0],
            &[0, 0, 119, 238, 202, 204, 209, 72, 110, 127, 0, 0],
            &[0, 0, 0, 7, 37, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'å' / 0xe5
        'å' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 105, 76, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 205, 89, 147, 124, 0, 0, 0, 0],
            &[0, 0, 0, 40, 167, 0, 17, 189, 0, 0, 0, 0],
            &[0, 0, 0, 7, 205, 102, 158, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 96, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 36, 66, 51, 12, 0, 0, 0, 0],
            &[0, 0, 84, 225, 202, 171, 189, 234, 99, 0, 0, 0],
            &[0, 0, 33, 30, 0, 0, 0, 42, 239, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 167, 99, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 137, 126, 0, 0],
            &[0, 0, 3, 89, 166, 205, 216, 214, 236, 127, 0, 0],
            &[0, 4, 190, 181, 67, 15, 0, 0, 134, 127, 0, 0],
            &[0, 83, 204, 1, 0, 0, 0, 0, 153, 127, 0, 0],
            &[0, 111, 161, 0, 0, 0, 0, 7, 225, 127, 0, 0],
            &[0, 59, 231, 30, 0, 0, 19, 169, 192, 127, 0, 0],
            &[0, 0, 119, 238, 202, 204, 209, 72, 110, 127, 0, 0],
            &[0, 0, 0, 7, 37, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'æ' / 0xe6
        'æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 66, 22, 0, 0, 36, 62, 7, 0, 0],
            &[0, 165, 209, 175, 231, 66, 111, 214, 185, 212, 15, 0],
            &[0, 30, 0, 0, 57, 204, 199, 8, 0, 165, 120, 0],
            &[0, 0, 0, 0, 0, 239, 103, 0, 0, 76, 187, 0],
            &[0, 0, 0, 0, 0, 223, 56, 0, 0, 50, 216, 0],
            &[0, 24, 139, 195, 213, 250, 223, 218, 218, 223, 226, 0],
            &[15, 225, 135, 36, 2, 227, 46, 10, 10, 10, 9, 0],
            &[94, 185, 0, 0, 0, 235, 58, 0, 0, 0, 0, 0],
            &[115, 147, 0, 0, 23, 246, 119, 0, 0, 0, 0, 0],
            &[75, 205, 5, 0, 149, 103, 217, 31, 0, 5, 47, 0],
            &[1, 177, 222, 216, 152, 0, 76, 228, 200, 226, 103, 0],
            &[0, 0, 30, 21, 0, 0, 0, 7, 38, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ç' / 0xe7
        'ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 66, 58, 19, 0, 0, 0],
            &[0, 0, 0, 68, 214, 225, 186, 193, 236, 93, 0, 0],
            &[0, 0, 68, 240, 84, 1, 0, 0, 6, 13, 0, 0],
            &[0, 0, 214, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 39, 245, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 70, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 73, 213, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 44, 244, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 92, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 238, 82, 2, 0, 0, 16, 32, 0, 0],
            &[0, 0, 0, 77, 210, 235, 218, 222, 232, 85, 0, 0],
            &[0, 0, 0, 0, 0, 89, 180, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 123, 194, 80, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 225, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 137, 99, 10, 0, 0, 0, 0],
        ]),
        // letter: 'è' / 0xe8
        'è' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 63, 128, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 199, 118, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 199, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 60, 60, 16, 0, 0, 0, 0],
            &[0, 0, 7, 153, 232, 184, 177, 232, 121, 0, 0, 0],
            &[0, 0, 168, 178, 17, 0, 0, 21, 212, 94, 0, 0],
            &[0, 55, 231, 15, 0, 0, 0, 0, 76, 209, 0, 0],
            &[0, 134, 155, 0, 0, 0, 0, 0, 15, 255, 10, 0],
            &[0, 167, 232, 213, 213, 213, 213, 213, 213, 255, 30, 0],
            &[0, 170, 128, 20, 20, 20, 20, 20, 20, 20, 2, 0],
            &[0, 138, 153, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 233, 15, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 160, 191, 34, 0, 0, 0, 24, 74, 0, 0],
            &[0, 0, 2, 121, 228, 218, 201, 225, 223, 118, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'é' / 0xe9
        'é' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 135, 50, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 182, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 189, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 107, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 60, 60, 16, 0, 0, 0, 0],
            &[0, 0, 7, 153, 232, 184, 177, 232, 121, 0, 0, 0],
            &[0, 0, 168, 178, 17, 0, 0, 21, 212, 94, 0, 0],
            &[0, 55, 231, 15, 0, 0, 0, 0, 76, 209, 0, 0],
            &[0, 134, 155, 0, 0, 0, 0, 0, 15, 255, 10, 0],
            &[0, 167, 232, 213, 213, 213, 213, 213, 213, 255, 30, 0],
            &[0, 170, 128, 20, 20, 20, 20, 20, 20, 20, 2, 0],
            &[0, 138, 153, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 233, 15, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 160, 191, 34, 0, 0, 0, 24, 74, 0, 0],
            &[0, 0, 2, 121, 228, 218, 201, 225, 223, 118, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ê' / 0xea
        'ê' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 124, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 123, 182, 222, 32, 0, 0, 0, 0],
            &[0, 0, 0, 78, 183, 6, 62, 198, 13, 0, 0, 0],
            &[0, 0, 0, 107, 8, 0, 0, 55, 61, 0, 0, 0],
            &[0, 0, 0, 0, 20, 60, 60, 16, 0, 0, 0, 0],
            &[0, 0, 7, 153, 232, 184, 177, 232, 121, 0, 0, 0],
            &[0, 0, 168, 178, 17, 0, 0, 21, 212, 94, 0, 0],
            &[0, 55, 231, 15, 0, 0, 0, 0, 76, 209, 0, 0],
            &[0, 134, 155, 0, 0, 0, 0, 0, 15, 255, 10, 0],
            &[0, 167, 232, 213, 213, 213, 213, 213, 213, 255, 30, 0],
            &[0, 170, 128, 20, 20, 20, 20, 20, 20, 20, 2, 0],
            &[0, 138, 153, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 233, 15, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 160, 191, 34, 0, 0, 0, 24, 74, 0, 0],
            &[0, 0, 2, 121, 228, 218, 201, 225, 223, 118, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ë' / 0xeb
        'ë' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 156, 67, 0, 0, 149, 73, 0, 0, 0],
            &[0, 0, 0, 200, 90, 0, 0, 192, 97, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 60, 60, 16, 0, 0, 0, 0],
            &[0, 0, 7, 153, 232, 184, 177, 232, 121, 0, 0, 0],
            &[0, 0, 168, 178, 17, 0, 0, 21, 212, 94, 0, 0],
            &[0, 55, 231, 15, 0, 0, 0, 0, 76, 209, 0, 0],
            &[0, 134, 155, 0, 0, 0, 0, 0, 15, 255, 10, 0],
            &[0, 167, 232, 213, 213, 213, 213, 213, 213, 255, 30, 0],
            &[0, 170, 128, 20, 20, 20, 20, 20, 20, 20, 2, 0],
            &[0, 138, 153, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 233, 15, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 160, 191, 34, 0, 0, 0, 24, 74, 0, 0],
            &[0, 0, 2, 121, 228, 218, 201, 225, 223, 118, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ì' / 0xec
        'ì' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 73, 119, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 212, 98, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 206, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 96, 0, 0, 0, 0, 0],
            &[0, 0, 26, 27, 27, 27, 16, 0, 0, 0, 0, 0],
            &[0, 0, 171, 190, 202, 234, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'í' / 0xed
        'í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 139, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 105, 175, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 105, 3, 0, 0, 0, 0, 0],
            &[0, 0, 26, 27, 27, 27, 16, 0, 0, 0, 0, 0],
            &[0, 0, 171, 190, 202, 234, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'î' / 0xee
        'î' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 130, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 176, 218, 22, 0, 0, 0, 0],
            &[0, 0, 0, 95, 169, 2, 76, 189, 7, 0, 0, 0],
            &[0, 0, 5, 106, 4, 0, 0, 64, 52, 0, 0, 0],
            &[0, 0, 26, 27, 27, 27, 16, 0, 0, 0, 0, 0],
            &[0, 0, 171, 190, 202, 234, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ï' / 0xef
        'ï' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 168, 54, 0, 0, 162, 59, 0, 0, 0],
            &[0, 0, 3, 213, 74, 0, 1, 208, 80, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 27, 27, 27, 16, 0, 0, 0, 0, 0],
            &[0, 0, 171, 190, 202, 234, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ð' / 0xf0
        'ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 64, 0, 0, 0, 40, 0, 0, 0],
            &[0, 0, 0, 17, 192, 189, 60, 176, 181, 4, 0, 0],
            &[0, 0, 0, 0, 11, 228, 255, 118, 0, 0, 0, 0],
            &[0, 0, 0, 64, 215, 122, 73, 220, 21, 0, 0, 0],
            &[0, 0, 0, 29, 28, 0, 0, 104, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 201, 54, 0, 0],
            &[0, 0, 1, 101, 193, 225, 212, 141, 107, 158, 0, 0],
            &[0, 0, 170, 189, 57, 10, 19, 98, 216, 234, 0, 0],
            &[0, 84, 213, 7, 0, 0, 0, 0, 85, 255, 27, 0],
            &[0, 169, 116, 0, 0, 0, 0, 0, 3, 245, 49, 0],
            &[0, 197, 83, 0, 0, 0, 0, 0, 0, 232, 47, 0],
            &[0, 182, 99, 0, 0, 0, 0, 0, 10, 250, 16, 0],
            &[0, 115, 179, 0, 0, 0, 0, 0, 92, 200, 0, 0],
            &[0, 13, 218, 122, 4, 0, 0, 59, 231, 67, 0, 0],
            &[0, 0, 24, 171, 231, 197, 216, 221, 78, 0, 0, 0],
            &[0, 0, 0, 0, 12, 40, 29, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ñ' / 0xf1
        'ñ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 99, 184, 116, 4, 0, 146, 14, 0, 0],
            &[0, 0, 36, 196, 51, 140, 206, 160, 179, 0, 0, 0],
            &[0, 0, 30, 41, 0, 0, 40, 79, 8, 0, 0, 0],
            &[0, 6, 18, 0, 5, 55, 66, 27, 0, 0, 0, 0],
            &[0, 59, 181, 58, 206, 187, 179, 236, 154, 1, 0, 0],
            &[0, 59, 220, 181, 24, 0, 0, 20, 223, 97, 0, 0],
            &[0, 59, 255, 61, 0, 0, 0, 0, 124, 163, 0, 0],
            &[0, 59, 250, 2, 0, 0, 0, 0, 94, 190, 0, 0],
            &[0, 59, 226, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ò' / 0xf2
        'ò' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 73, 119, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 212, 98, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 206, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 66, 52, 7, 0, 0, 0, 0],
            &[0, 0, 25, 186, 216, 169, 188, 229, 92, 0, 0, 0],
            &[0, 7, 212, 123, 1, 0, 0, 40, 229, 73, 0, 0],
            &[0, 97, 210, 1, 0, 0, 0, 0, 93, 204, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 15, 253, 20, 0],
            &[0, 195, 90, 0, 0, 0, 0, 0, 0, 238, 47, 0],
            &[0, 195, 87, 0, 0, 0, 0, 0, 0, 241, 46, 0],
            &[0, 163, 124, 0, 0, 0, 0, 0, 23, 253, 14, 0],
            &[0, 87, 211, 2, 0, 0, 0, 0, 111, 191, 0, 0],
            &[0, 3, 198, 149, 9, 0, 0, 69, 237, 57, 0, 0],
            &[0, 0, 16, 163, 232, 199, 218, 216, 69, 0, 0, 0],
            &[0, 0, 0, 0, 12, 41, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ó' / 0xf3
        'ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 139, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 105, 175, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 105, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 66, 52, 7, 0, 0, 0, 0],
            &[0, 0, 25, 186, 216, 169, 188, 229, 92, 0, 0, 0],
            &[0, 7, 212, 123, 1, 0, 0, 40, 229, 73, 0, 0],
            &[0, 97, 210, 1, 0, 0, 0, 0, 93, 204, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 15, 253, 20, 0],
            &[0, 195, 90, 0, 0, 0, 0, 0, 0, 238, 47, 0],
            &[0, 195, 87, 0, 0, 0, 0, 0, 0, 241, 46, 0],
            &[0, 163, 124, 0, 0, 0, 0, 0, 23, 253, 14, 0],
            &[0, 87, 211, 2, 0, 0, 0, 0, 111, 191, 0, 0],
            &[0, 3, 198, 149, 9, 0, 0, 69, 237, 57, 0, 0],
            &[0, 0, 16, 163, 232, 199, 218, 216, 69, 0, 0, 0],
            &[0, 0, 0, 0, 12, 41, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ô' / 0xf4
        'ô' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 130, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 176, 218, 22, 0, 0, 0, 0],
            &[0, 0, 0, 95, 169, 2, 76, 189, 7, 0, 0, 0],
            &[0, 0, 5, 106, 4, 0, 0, 64, 52, 0, 0, 0],
            &[0, 0, 0, 0, 31, 66, 52, 7, 0, 0, 0, 0],
            &[0, 0, 25, 186, 216, 169, 188, 229, 92, 0, 0, 0],
            &[0, 7, 212, 123, 1, 0, 0, 40, 229, 73, 0, 0],
            &[0, 97, 210, 1, 0, 0, 0, 0, 93, 204, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 15, 253, 20, 0],
            &[0, 195, 90, 0, 0, 0, 0, 0, 0, 238, 47, 0],
            &[0, 195, 87, 0, 0, 0, 0, 0, 0, 241, 46, 0],
            &[0, 163, 124, 0, 0, 0, 0, 0, 23, 253, 14, 0],
            &[0, 87, 211, 2, 0, 0, 0, 0, 111, 191, 0, 0],
            &[0, 3, 198, 149, 9, 0, 0, 69, 237, 57, 0, 0],
            &[0, 0, 16, 163, 232, 199, 218, 216, 69, 0, 0, 0],
            &[0, 0, 0, 0, 12, 41, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'õ' / 0xf5
        'õ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 131, 182, 90, 0, 1, 159, 0, 0, 0],
            &[0, 0, 83, 162, 52, 172, 188, 176, 135, 0, 0, 0],
            &[0, 0, 48, 23, 0, 0, 56, 70, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 66, 52, 7, 0, 0, 0, 0],
            &[0, 0, 25, 186, 216, 169, 188, 229, 92, 0, 0, 0],
            &[0, 7, 212, 123, 1, 0, 0, 40, 229, 73, 0, 0],
            &[0, 97, 210, 1, 0, 0, 0, 0, 93, 204, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 15, 253, 20, 0],
            &[0, 195, 90, 0, 0, 0, 0, 0, 0, 238, 47, 0],
            &[0, 195, 87, 0, 0, 0, 0, 0, 0, 241, 46, 0],
            &[0, 163, 124, 0, 0, 0, 0, 0, 23, 253, 14, 0],
            &[0, 87, 211, 2, 0, 0, 0, 0, 111, 191, 0, 0],
            &[0, 3, 198, 149, 9, 0, 0, 69, 237, 57, 0, 0],
            &[0, 0, 16, 163, 232, 199, 218, 216, 69, 0, 0, 0],
            &[0, 0, 0, 0, 12, 41, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ö' / 0xf6
        'ö' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 168, 54, 0, 0, 162, 59, 0, 0, 0],
            &[0, 0, 3, 213, 74, 0, 1, 208, 80, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 66, 52, 7, 0, 0, 0, 0],
            &[0, 0, 25, 186, 216, 169, 188, 229, 92, 0, 0, 0],
            &[0, 7, 212, 123, 1, 0, 0, 40, 229, 73, 0, 0],
            &[0, 97, 210, 1, 0, 0, 0, 0, 93, 204, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 15, 253, 20, 0],
            &[0, 195, 90, 0, 0, 0, 0, 0, 0, 238, 47, 0],
            &[0, 195, 87, 0, 0, 0, 0, 0, 0, 241, 46, 0],
            &[0, 163, 124, 0, 0, 0, 0, 0, 23, 253, 14, 0],
            &[0, 87, 211, 2, 0, 0, 0, 0, 111, 191, 0, 0],
            &[0, 3, 198, 149, 9, 0, 0, 69, 237, 57, 0, 0],
            &[0, 0, 16, 163, 232, 199, 218, 216, 69, 0, 0, 0],
            &[0, 0, 0, 0, 12, 41, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '÷' / 0xf7
        '÷' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 193, 66, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 185, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 23, 41, 41, 41, 41, 41, 41, 41, 40, 0, 0],
            &[0, 110, 191, 191, 191, 191, 191, 191, 191, 186, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 250, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 88, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ø' / 0xf8
        'ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 66, 52, 7, 0, 105, 10, 0],
            &[0, 0, 25, 186, 216, 169, 188, 229, 195, 185, 2, 0],
            &[0, 7, 212, 123, 1, 0, 0, 89, 255, 94, 0, 0],
            &[0, 97, 210, 1, 0, 0, 15, 215, 152, 204, 0, 0],
            &[0, 168, 122, 0, 0, 0, 173, 116, 15, 253, 20, 0],
            &[0, 195, 90, 0, 0, 112, 177, 0, 0, 238, 47, 0],
            &[0, 195, 87, 0, 56, 217, 16, 0, 0, 241, 46, 0],
            &[0, 163, 124, 19, 219, 52, 0, 0, 23, 253, 14, 0],
            &[0, 87, 212, 185, 107, 0, 0, 0, 111, 191, 0, 0],
            &[0, 3, 255, 255, 9, 0, 0, 69, 237, 57, 0, 0],
            &[0, 65, 230, 176, 232, 199, 218, 216, 69, 0, 0, 0],
            &[0, 45, 43, 0, 12, 41, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ù' / 0xf9
        'ù' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 73, 119, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 212, 98, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 206, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 96, 0, 0, 0, 0, 0],
            &[0, 9, 21, 0, 0, 0, 0, 0, 12, 18, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 118, 166, 0, 0],
            &[0, 78, 206, 0, 0, 0, 0, 0, 142, 166, 0, 0],
            &[0, 47, 238, 4, 0, 0, 0, 2, 214, 166, 0, 0],
            &[0, 4, 218, 125, 0, 0, 5, 138, 182, 166, 0, 0],
            &[0, 0, 50, 214, 226, 198, 220, 108, 72, 166, 0, 0],
            &[0, 0, 0, 0, 22, 33, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ú' / 0xfa
        'ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 139, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 105, 175, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 105, 3, 0, 0, 0, 0, 0],
            &[0, 9, 21, 0, 0, 0, 0, 0, 12, 18, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 118, 166, 0, 0],
            &[0, 78, 206, 0, 0, 0, 0, 0, 142, 166, 0, 0],
            &[0, 47, 238, 4, 0, 0, 0, 2, 214, 166, 0, 0],
            &[0, 4, 218, 125, 0, 0, 5, 138, 182, 166, 0, 0],
            &[0, 0, 50, 214, 226, 198, 220, 108, 72, 166, 0, 0],
            &[0, 0, 0, 0, 22, 33, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'û' / 0xfb
        'û' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 130, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 176, 218, 22, 0, 0, 0, 0],
            &[0, 0, 0, 95, 169, 2, 76, 189, 7, 0, 0, 0],
            &[0, 0, 5, 106, 4, 0, 0, 64, 52, 0, 0, 0],
            &[0, 9, 21, 0, 0, 0, 0, 0, 12, 18, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 118, 166, 0, 0],
            &[0, 78, 206, 0, 0, 0, 0, 0, 142, 166, 0, 0],
            &[0, 47, 238, 4, 0, 0, 0, 2, 214, 166, 0, 0],
            &[0, 4, 218, 125, 0, 0, 5, 138, 182, 166, 0, 0],
            &[0, 0, 50, 214, 226, 198, 220, 108, 72, 166, 0, 0],
            &[0, 0, 0, 0, 22, 33, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ü' / 0xfc
        'ü' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 168, 54, 0, 0, 162, 59, 0, 0, 0],
            &[0, 0, 3, 213, 74, 0, 1, 208, 80, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 21, 0, 0, 0, 0, 0, 12, 18, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 118, 166, 0, 0],
            &[0, 78, 206, 0, 0, 0, 0, 0, 142, 166, 0, 0],
            &[0, 47, 238, 4, 0, 0, 0, 2, 214, 166, 0, 0],
            &[0, 4, 218, 125, 0, 0, 5, 138, 182, 166, 0, 0],
            &[0, 0, 50, 214, 226, 198, 220, 108, 72, 166, 0, 0],
            &[0, 0, 0, 0, 22, 33, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ý' / 0xfd
        'ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 139, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 105, 175, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 105, 3, 0, 0, 0, 0, 0],
            &[0, 18, 11, 0, 0, 0, 0, 0, 0, 27, 3, 0],
            &[0, 119, 161, 0, 0, 0, 0, 0, 49, 234, 5, 0],
            &[0, 23, 240, 14, 0, 0, 0, 0, 141, 142, 0, 0],
            &[0, 0, 171, 104, 0, 0, 0, 2, 230, 46, 0, 0],
            &[0, 0, 70, 203, 0, 0, 0, 69, 205, 0, 0, 0],
            &[0, 0, 1, 222, 46, 0, 0, 161, 109, 0, 0, 0],
            &[0, 0, 0, 122, 146, 0, 10, 236, 19, 0, 0, 0],
            &[0, 0, 0, 25, 233, 7, 88, 172, 0, 0, 0, 0],
            &[0, 0, 0, 0, 175, 85, 177, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 186, 224, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 225, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 226, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 201, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 186, 90, 0, 0, 0, 0, 0, 0],
            &[0, 83, 166, 133, 15, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'þ' / 0xfe
        'þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 217, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 217, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 217, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 217, 0, 5, 55, 68, 26, 0, 0, 0, 0],
            &[0, 59, 216, 50, 205, 189, 183, 236, 149, 3, 0, 0],
            &[0, 59, 222, 183, 29, 0, 0, 22, 207, 118, 0, 0],
            &[0, 59, 255, 73, 0, 0, 0, 0, 77, 228, 0, 0],
            &[0, 59, 251, 2, 0, 0, 0, 0, 12, 255, 29, 0],
            &[0, 59, 225, 0, 0, 0, 0, 0, 0, 244, 51, 0],
            &[0, 59, 232, 0, 0, 0, 0, 0, 1, 250, 44, 0],
            &[0, 59, 253, 6, 0, 0, 0, 0, 23, 254, 17, 0],
            &[0, 59, 255, 86, 0, 0, 0, 0, 106, 204, 0, 0],
            &[0, 59, 219, 193, 40, 0, 0, 50, 231, 79, 0, 0],
            &[0, 59, 215, 43, 202, 213, 216, 233, 100, 0, 0, 0],
            &[0, 59, 217, 0, 0, 31, 36, 5, 0, 0, 0, 0],
            &[0, 59, 217, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 217, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 121, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ÿ' / 0xff
        'ÿ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 168, 54, 0, 0, 162, 59, 0, 0, 0],
            &[0, 0, 3, 213, 74, 0, 1, 208, 80, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 11, 0, 0, 0, 0, 0, 0, 27, 3, 0],
            &[0, 119, 161, 0, 0, 0, 0, 0, 49, 234, 5, 0],
            &[0, 23, 240, 14, 0, 0, 0, 0, 141, 142, 0, 0],
            &[0, 0, 171, 104, 0, 0, 0, 2, 230, 46, 0, 0],
            &[0, 0, 70, 203, 0, 0, 0, 69, 205, 0, 0, 0],
            &[0, 0, 1, 222, 46, 0, 0, 161, 109, 0, 0, 0],
            &[0, 0, 0, 122, 146, 0, 10, 236, 19, 0, 0, 0],
            &[0, 0, 0, 25, 233, 7, 88, 172, 0, 0, 0, 0],
            &[0, 0, 0, 0, 175, 85, 177, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 186, 224, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 225, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 226, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 201, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 186, 90, 0, 0, 0, 0, 0, 0],
            &[0, 83, 166, 133, 15, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ā' / 0x100
        'Ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 62, 67, 67, 67, 67, 21, 0, 0, 0],
            &[0, 0, 0, 152, 165, 165, 165, 165, 53, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 153, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 244, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 127, 239, 12, 0, 0, 0, 0],
            &[0, 0, 0, 6, 237, 30, 178, 91, 0, 0, 0, 0],
            &[0, 0, 0, 78, 197, 0, 90, 181, 0, 0, 0, 0],
            &[0, 0, 0, 167, 110, 0, 12, 241, 19, 0, 0, 0],
            &[0, 0, 11, 242, 25, 0, 0, 172, 103, 0, 0, 0],
            &[0, 0, 90, 190, 0, 0, 0, 85, 192, 0, 0, 0],
            &[0, 0, 182, 255, 169, 169, 169, 179, 255, 28, 0, 0],
            &[0, 18, 255, 108, 88, 88, 88, 88, 212, 115, 0, 0],
            &[0, 102, 183, 0, 0, 0, 0, 0, 79, 204, 0, 0],
            &[0, 191, 96, 0, 0, 0, 0, 0, 7, 240, 38, 0],
            &[27, 246, 15, 0, 0, 0, 0, 0, 0, 161, 127, 0],
            &[114, 176, 0, 0, 0, 0, 0, 0, 0, 74, 216, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ā' / 0x101
        'ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 170, 206, 206, 206, 206, 86, 0, 0, 0],
            &[0, 0, 0, 22, 27, 27, 27, 27, 11, 0, 0, 0],
            &[0, 0, 0, 2, 36, 66, 51, 12, 0, 0, 0, 0],
            &[0, 0, 84, 225, 202, 171, 189, 234, 99, 0, 0, 0],
            &[0, 0, 33, 30, 0, 0, 0, 42, 239, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 167, 99, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 137, 126, 0, 0],
            &[0, 0, 3, 89, 166, 205, 216, 214, 236, 127, 0, 0],
            &[0, 4, 190, 181, 67, 15, 0, 0, 134, 127, 0, 0],
            &[0, 83, 204, 1, 0, 0, 0, 0, 153, 127, 0, 0],
            &[0, 111, 161, 0, 0, 0, 0, 7, 225, 127, 0, 0],
            &[0, 59, 231, 30, 0, 0, 19, 169, 192, 127, 0, 0],
            &[0, 0, 119, 238, 202, 204, 209, 72, 110, 127, 0, 0],
            &[0, 0, 0, 7, 37, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ă' / 0x102
        'Ă' => Some(&[
            &[0, 0, 0, 85, 0, 0, 0, 53, 30, 0, 0, 0],
            &[0, 0, 0, 172, 91, 0, 21, 204, 27, 0, 0, 0],
            &[0, 0, 0, 23, 170, 211, 198, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 153, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 244, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 127, 239, 12, 0, 0, 0, 0],
            &[0, 0, 0, 6, 237, 30, 178, 91, 0, 0, 0, 0],
            &[0, 0, 0, 78, 197, 0, 90, 181, 0, 0, 0, 0],
            &[0, 0, 0, 167, 110, 0, 12, 241, 19, 0, 0, 0],
            &[0, 0, 11, 242, 25, 0, 0, 172, 103, 0, 0, 0],
            &[0, 0, 90, 190, 0, 0, 0, 85, 192, 0, 0, 0],
            &[0, 0, 182, 255, 169, 169, 169, 179, 255, 28, 0, 0],
            &[0, 18, 255, 108, 88, 88, 88, 88, 212, 115, 0, 0],
            &[0, 102, 183, 0, 0, 0, 0, 0, 79, 204, 0, 0],
            &[0, 191, 96, 0, 0, 0, 0, 0, 7, 240, 38, 0],
            &[27, 246, 15, 0, 0, 0, 0, 0, 0, 161, 127, 0],
            &[114, 176, 0, 0, 0, 0, 0, 0, 0, 74, 216, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ă' / 0x103
        'ă' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 176, 36, 0, 0, 127, 77, 0, 0, 0],
            &[0, 0, 0, 68, 213, 125, 146, 195, 8, 0, 0, 0],
            &[0, 0, 0, 0, 29, 87, 70, 4, 0, 0, 0, 0],
            &[0, 0, 0, 2, 36, 66, 51, 12, 0, 0, 0, 0],
            &[0, 0, 84, 225, 202, 171, 189, 234, 99, 0, 0, 0],
            &[0, 0, 33, 30, 0, 0, 0, 42, 239, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 167, 99, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 137, 126, 0, 0],
            &[0, 0, 3, 89, 166, 205, 216, 214, 236, 127, 0, 0],
            &[0, 4, 190, 181, 67, 15, 0, 0, 134, 127, 0, 0],
            &[0, 83, 204, 1, 0, 0, 0, 0, 153, 127, 0, 0],
            &[0, 111, 161, 0, 0, 0, 0, 7, 225, 127, 0, 0],
            &[0, 59, 231, 30, 0, 0, 19, 169, 192, 127, 0, 0],
            &[0, 0, 119, 238, 202, 204, 209, 72, 110, 127, 0, 0],
            &[0, 0, 0, 7, 37, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ą' / 0x104
        'Ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 153, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 244, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 127, 239, 12, 0, 0, 0, 0],
            &[0, 0, 0, 6, 237, 30, 178, 91, 0, 0, 0, 0],
            &[0, 0, 0, 78, 197, 0, 90, 181, 0, 0, 0, 0],
            &[0, 0, 0, 167, 110, 0, 12, 241, 19, 0, 0, 0],
            &[0, 0, 11, 242, 25, 0, 0, 172, 103, 0, 0, 0],
            &[0, 0, 90, 190, 0, 0, 0, 85, 192, 0, 0, 0],
            &[0, 0, 182, 255, 169, 169, 169, 179, 255, 28, 0, 0],
            &[0, 18, 255, 108, 88, 88, 88, 88, 212, 115, 0, 0],
            &[0, 102, 183, 0, 0, 0, 0, 0, 79, 204, 0, 0],
            &[0, 191, 96, 0, 0, 0, 0, 0, 7, 240, 38, 0],
            &[27, 246, 15, 0, 0, 0, 0, 0, 0, 161, 127, 0],
            &[114, 176, 0, 0, 0, 0, 0, 0, 0, 74, 244, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 105, 140, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 35, 189, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 92, 154, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 29, 3],
        ]),
        // letter: 'ą' / 0x105
        'ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 36, 66, 51, 12, 0, 0, 0, 0],
            &[0, 0, 84, 225, 202, 171, 189, 234, 99, 0, 0, 0],
            &[0, 0, 33, 30, 0, 0, 0, 42, 239, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 167, 99, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 137, 126, 0, 0],
            &[0, 0, 3, 89, 166, 205, 216, 214, 236, 127, 0, 0],
            &[0, 4, 190, 181, 67, 15, 0, 0, 134, 127, 0, 0],
            &[0, 83, 204, 1, 0, 0, 0, 0, 153, 127, 0, 0],
            &[0, 111, 161, 0, 0, 0, 0, 7, 225, 127, 0, 0],
            &[0, 59, 231, 30, 0, 0, 19, 169, 192, 127, 0, 0],
            &[0, 0, 119, 238, 202, 204, 209, 72, 128, 138, 0, 0],
            &[0, 0, 0, 7, 37, 19, 0, 29, 188, 28, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 178, 47, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 24, 15, 0, 0],
        ]),
        // letter: 'Ć' / 0x106
        'Ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 36, 241, 64, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 204, 96, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 164, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 134, 178, 183, 156, 89, 10, 0],
            &[0, 0, 0, 124, 237, 135, 75, 70, 104, 185, 55, 0],
            &[0, 0, 102, 225, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 238, 68, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 85, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 175, 115, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 156, 133, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 119, 177, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 241, 14, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 191, 141, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 232, 146, 22, 0, 0, 9, 73, 3, 0],
            &[0, 0, 0, 35, 167, 250, 230, 228, 249, 198, 5, 0],
            &[0, 0, 0, 0, 0, 5, 34, 35, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0],
        ]),
        // letter: 'ć' / 0x107
        'ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 75, 117, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 34, 231, 60, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 200, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 57, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 66, 58, 19, 0, 0, 0],
            &[0, 0, 0, 68, 214, 225, 186, 193, 236, 93, 0, 0],
            &[0, 0, 68, 240, 84, 1, 0, 0, 6, 13, 0, 0],
            &[0, 0, 214, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 39, 245, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 69, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 73, 213, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 44, 244, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 92, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 238, 82, 2, 0, 0, 16, 32, 0, 0],
            &[0, 0, 0, 77, 210, 235, 206, 222, 232, 85, 0, 0],
            &[0, 0, 0, 0, 0, 20, 42, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĉ' / 0x108
        'Ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 24, 233, 184, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 191, 85, 162, 124, 0, 0, 0],
            &[0, 0, 0, 0, 157, 95, 0, 4, 166, 85, 0, 0],
            &[0, 0, 0, 0, 5, 0, 0, 0, 1, 4, 0, 0],
            &[0, 0, 0, 0, 42, 134, 178, 183, 156, 89, 10, 0],
            &[0, 0, 0, 124, 237, 135, 75, 70, 104, 185, 55, 0],
            &[0, 0, 102, 225, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 238, 68, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 85, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 175, 115, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 156, 133, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 119, 177, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 241, 14, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 191, 141, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 232, 146, 22, 0, 0, 9, 73, 3, 0],
            &[0, 0, 0, 35, 167, 250, 230, 228, 249, 198, 5, 0],
            &[0, 0, 0, 0, 0, 5, 34, 35, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĉ' / 0x109
        'ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 129, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 218, 180, 136, 0, 0, 0, 0],
            &[0, 0, 0, 9, 191, 67, 5, 179, 90, 0, 0, 0],
            &[0, 0, 0, 56, 60, 0, 0, 5, 107, 3, 0, 0],
            &[0, 0, 0, 0, 0, 37, 66, 58, 19, 0, 0, 0],
            &[0, 0, 0, 68, 214, 225, 186, 193, 236, 93, 0, 0],
            &[0, 0, 68, 240, 84, 1, 0, 0, 6, 13, 0, 0],
            &[0, 0, 214, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 39, 245, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 70, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 73, 213, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 44, 244, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 92, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 238, 82, 2, 0, 0, 16, 32, 0, 0],
            &[0, 0, 0, 77, 210, 235, 206, 222, 232, 85, 0, 0],
            &[0, 0, 0, 0, 0, 20, 42, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ċ' / 0x10a
        'Ċ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 92, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 244, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 53, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 134, 178, 183, 156, 89, 10, 0],
            &[0, 0, 0, 124, 237, 135, 75, 70, 104, 185, 55, 0],
            &[0, 0, 102, 225, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 238, 68, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 85, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 175, 115, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 156, 133, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 119, 177, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 241, 14, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 191, 141, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 232, 146, 22, 0, 0, 9, 73, 3, 0],
            &[0, 0, 0, 35, 167, 250, 230, 228, 249, 198, 5, 0],
            &[0, 0, 0, 0, 0, 5, 34, 35, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ċ' / 0x10b
        'ċ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 135, 249, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 146, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 66, 58, 19, 0, 0, 0],
            &[0, 0, 0, 68, 214, 225, 186, 193, 236, 93, 0, 0],
            &[0, 0, 68, 240, 84, 1, 0, 0, 6, 13, 0, 0],
            &[0, 0, 214, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 39, 245, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 70, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 73, 213, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 44, 244, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 92, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 238, 82, 2, 0, 0, 16, 32, 0, 0],
            &[0, 0, 0, 77, 210, 235, 206, 222, 232, 85, 0, 0],
            &[0, 0, 0, 0, 0, 20, 42, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Č' / 0x10c
        'Č' => Some(&[
            &[0, 0, 0, 0, 164, 88, 0, 4, 164, 90, 0, 0],
            &[0, 0, 0, 0, 8, 199, 82, 165, 124, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 240, 182, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 134, 178, 183, 156, 89, 10, 0],
            &[0, 0, 0, 124, 237, 135, 75, 70, 104, 185, 55, 0],
            &[0, 0, 102, 225, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 238, 68, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 85, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 175, 115, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 156, 133, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 119, 177, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 241, 14, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 191, 141, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 232, 146, 22, 0, 0, 9, 73, 3, 0],
            &[0, 0, 0, 35, 167, 250, 230, 228, 249, 198, 5, 0],
            &[0, 0, 0, 0, 0, 5, 34, 35, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'č' / 0x10d
        'č' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 68, 0, 0, 10, 119, 3, 0, 0],
            &[0, 0, 0, 7, 190, 79, 10, 192, 77, 0, 0, 0],
            &[0, 0, 0, 0, 22, 220, 198, 118, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 113, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 66, 58, 19, 0, 0, 0],
            &[0, 0, 0, 68, 214, 225, 186, 193, 236, 93, 0, 0],
            &[0, 0, 68, 240, 84, 1, 0, 0, 6, 13, 0, 0],
            &[0, 0, 214, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 39, 245, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 70, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 73, 213, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 44, 244, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 223, 92, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 238, 82, 2, 0, 0, 16, 32, 0, 0],
            &[0, 0, 0, 77, 210, 235, 206, 222, 232, 85, 0, 0],
            &[0, 0, 0, 0, 0, 20, 42, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ď' / 0x10e
        'Ď' => Some(&[
            &[0, 0, 25, 194, 33, 0, 33, 196, 30, 0, 0, 0],
            &[0, 0, 0, 49, 213, 52, 213, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 101, 254, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0],
            &[0, 61, 144, 144, 142, 118, 73, 3, 0, 0, 0, 0],
            &[0, 108, 206, 103, 108, 133, 204, 226, 60, 0, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 89, 242, 47, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 126, 193, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 18, 251, 26, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 0, 207, 80, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 0, 184, 102, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 0, 177, 105, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 0, 196, 84, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 4, 242, 42, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 83, 218, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 26, 225, 86, 0, 0],
            &[0, 108, 173, 0, 5, 33, 108, 229, 133, 0, 0, 0],
            &[0, 108, 254, 252, 248, 222, 168, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ď' / 0x10f
        'ď' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 100, 0, 29, 94, 0],
            &[0, 0, 0, 0, 0, 0, 53, 228, 0, 105, 158, 0],
            &[0, 0, 0, 0, 0, 0, 53, 228, 0, 170, 54, 0],
            &[0, 0, 0, 0, 0, 0, 53, 228, 0, 112, 0, 0],
            &[0, 0, 18, 63, 44, 3, 53, 228, 0, 0, 0, 0],
            &[0, 107, 239, 181, 181, 200, 76, 228, 0, 0, 0, 0],
            &[58, 240, 44, 0, 0, 47, 200, 228, 0, 0, 0, 0],
            &[165, 135, 0, 0, 0, 0, 147, 228, 0, 0, 0, 0],
            &[222, 67, 0, 0, 0, 0, 84, 228, 0, 0, 0, 0],
            &[244, 41, 0, 0, 0, 0, 60, 228, 0, 0, 0, 0],
            &[247, 43, 0, 0, 0, 0, 62, 228, 0, 0, 0, 0],
            &[224, 67, 0, 0, 0, 0, 89, 228, 0, 0, 0, 0],
            &[167, 136, 0, 0, 0, 0, 162, 228, 0, 0, 0, 0],
            &[60, 238, 54, 0, 0, 76, 186, 228, 0, 0, 0, 0],
            &[0, 99, 238, 212, 220, 178, 27, 228, 0, 0, 0, 0],
            &[0, 0, 7, 40, 22, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Đ' / 0x110
        'Đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 144, 144, 143, 124, 87, 10, 0, 0, 0, 0],
            &[0, 59, 235, 103, 105, 128, 185, 239, 94, 0, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 56, 233, 89, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 78, 235, 7, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 0, 222, 74, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 0, 159, 129, 0],
            &[80, 140, 255, 80, 80, 31, 0, 0, 0, 135, 151, 0],
            &[167, 226, 255, 167, 167, 66, 0, 0, 0, 128, 154, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 0, 148, 132, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 0, 197, 90, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 36, 248, 16, 0],
            &[0, 59, 222, 0, 0, 0, 0, 9, 193, 135, 0, 0],
            &[0, 59, 222, 0, 2, 24, 89, 211, 173, 7, 0, 0],
            &[0, 59, 255, 252, 251, 228, 184, 87, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'đ' / 0x111
        'đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 50, 73, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 115, 166, 0, 0, 0],
            &[0, 0, 0, 20, 152, 152, 152, 255, 255, 152, 61, 0],
            &[0, 0, 0, 9, 70, 70, 70, 186, 237, 70, 28, 0],
            &[0, 0, 0, 31, 51, 14, 0, 114, 166, 0, 0, 0],
            &[0, 39, 203, 230, 187, 218, 145, 112, 166, 0, 0, 0],
            &[12, 228, 133, 1, 0, 1, 124, 204, 166, 0, 0, 0],
            &[106, 199, 0, 0, 0, 0, 1, 216, 166, 0, 0, 0],
            &[170, 120, 0, 0, 0, 0, 0, 148, 166, 0, 0, 0],
            &[195, 89, 0, 0, 0, 0, 0, 122, 166, 0, 0, 0],
            &[198, 92, 0, 0, 0, 0, 0, 124, 166, 0, 0, 0],
            &[173, 120, 0, 0, 0, 0, 0, 151, 166, 0, 0, 0],
            &[113, 199, 0, 0, 0, 0, 3, 225, 166, 0, 0, 0],
            &[17, 232, 122, 0, 0, 2, 144, 187, 166, 0, 0, 0],
            &[0, 45, 204, 236, 202, 231, 120, 74, 166, 0, 0, 0],
            &[0, 0, 0, 26, 42, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ē' / 0x112
        'Ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 52, 67, 67, 67, 67, 32, 0, 0, 0],
            &[0, 0, 0, 127, 165, 165, 165, 165, 78, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 144, 144, 144, 144, 108, 0, 0],
            &[0, 1, 255, 122, 107, 107, 107, 107, 107, 80, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 167, 158, 158, 158, 158, 158, 58, 0, 0],
            &[0, 1, 255, 110, 94, 94, 94, 94, 94, 34, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 26, 1, 1, 1, 1, 1, 1, 0, 0],
            &[0, 1, 255, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ē' / 0x113
        'ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 174, 206, 206, 206, 206, 82, 0, 0, 0],
            &[0, 0, 0, 22, 27, 27, 27, 27, 10, 0, 0, 0],
            &[0, 0, 0, 0, 20, 60, 60, 16, 0, 0, 0, 0],
            &[0, 0, 7, 153, 232, 184, 177, 232, 121, 0, 0, 0],
            &[0, 0, 168, 178, 17, 0, 0, 21, 212, 94, 0, 0],
            &[0, 55, 231, 15, 0, 0, 0, 0, 76, 209, 0, 0],
            &[0, 134, 155, 0, 0, 0, 0, 0, 15, 255, 10, 0],
            &[0, 167, 232, 213, 213, 213, 213, 213, 213, 255, 30, 0],
            &[0, 170, 128, 20, 20, 20, 20, 20, 20, 20, 2, 0],
            &[0, 138, 153, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 233, 15, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 160, 191, 34, 0, 0, 0, 24, 74, 0, 0],
            &[0, 0, 2, 121, 228, 218, 201, 225, 223, 118, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĕ' / 0x114
        'Ĕ' => Some(&[
            &[0, 0, 0, 76, 9, 0, 0, 37, 45, 0, 0, 0],
            &[0, 0, 0, 133, 129, 1, 11, 182, 58, 0, 0, 0],
            &[0, 0, 0, 9, 152, 211, 204, 101, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 144, 144, 144, 144, 108, 0, 0],
            &[0, 1, 255, 122, 107, 107, 107, 107, 107, 80, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 167, 158, 158, 158, 158, 158, 58, 0, 0],
            &[0, 1, 255, 110, 94, 94, 94, 94, 94, 34, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 26, 1, 1, 1, 1, 1, 1, 0, 0],
            &[0, 1, 255, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĕ' / 0x115
        'ĕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 180, 32, 0, 0, 132, 73, 0, 0, 0],
            &[0, 0, 0, 71, 212, 124, 147, 193, 6, 0, 0, 0],
            &[0, 0, 0, 0, 30, 87, 69, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 60, 60, 16, 0, 0, 0, 0],
            &[0, 0, 7, 153, 232, 184, 177, 232, 121, 0, 0, 0],
            &[0, 0, 168, 178, 17, 0, 0, 21, 212, 94, 0, 0],
            &[0, 55, 231, 15, 0, 0, 0, 0, 76, 209, 0, 0],
            &[0, 134, 155, 0, 0, 0, 0, 0, 15, 255, 10, 0],
            &[0, 167, 232, 213, 213, 213, 213, 213, 213, 255, 30, 0],
            &[0, 170, 128, 20, 20, 20, 20, 20, 20, 20, 2, 0],
            &[0, 138, 153, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 233, 15, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 160, 191, 34, 0, 0, 0, 24, 74, 0, 0],
            &[0, 0, 2, 121, 228, 218, 201, 225, 223, 118, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ė' / 0x116
        'Ė' => Some(&[
            &[0, 0, 0, 0, 0, 91, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 242, 168, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 144, 144, 144, 144, 108, 0, 0],
            &[0, 1, 255, 122, 107, 107, 107, 107, 107, 80, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 167, 158, 158, 158, 158, 158, 58, 0, 0],
            &[0, 1, 255, 110, 94, 94, 94, 94, 94, 34, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 26, 1, 1, 1, 1, 1, 1, 0, 0],
            &[0, 1, 255, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ė' / 0x117
        'ė' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 247, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 143, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 60, 60, 16, 0, 0, 0, 0],
            &[0, 0, 7, 153, 232, 184, 177, 232, 121, 0, 0, 0],
            &[0, 0, 168, 178, 17, 0, 0, 21, 212, 94, 0, 0],
            &[0, 55, 231, 15, 0, 0, 0, 0, 76, 209, 0, 0],
            &[0, 134, 155, 0, 0, 0, 0, 0, 15, 255, 10, 0],
            &[0, 167, 232, 213, 213, 213, 213, 213, 213, 255, 30, 0],
            &[0, 170, 128, 20, 20, 20, 20, 20, 20, 20, 2, 0],
            &[0, 138, 153, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 233, 15, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 160, 191, 34, 0, 0, 0, 24, 74, 0, 0],
            &[0, 0, 2, 121, 228, 218, 201, 225, 223, 118, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ę' / 0x118
        'Ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 144, 144, 144, 144, 108, 0, 0],
            &[0, 1, 255, 122, 107, 107, 107, 107, 107, 80, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 167, 158, 158, 158, 158, 158, 58, 0, 0],
            &[0, 1, 255, 110, 94, 94, 94, 94, 94, 34, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 26, 1, 1, 1, 1, 1, 1, 0, 0],
            &[0, 1, 255, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 178, 61, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 183, 63, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 17, 22, 0, 0, 0],
        ]),
        // letter: 'ę' / 0x119
        'ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 60, 60, 16, 0, 0, 0, 0],
            &[0, 0, 7, 153, 232, 184, 177, 232, 121, 0, 0, 0],
            &[0, 0, 168, 178, 17, 0, 0, 21, 212, 94, 0, 0],
            &[0, 55, 231, 15, 0, 0, 0, 0, 76, 209, 0, 0],
            &[0, 134, 155, 0, 0, 0, 0, 0, 15, 255, 10, 0],
            &[0, 167, 232, 213, 213, 213, 213, 213, 213, 255, 30, 0],
            &[0, 170, 128, 20, 20, 20, 20, 20, 20, 20, 2, 0],
            &[0, 138, 153, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 233, 15, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 160, 191, 34, 0, 0, 0, 24, 74, 0, 0],
            &[0, 0, 2, 121, 228, 218, 201, 225, 255, 184, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 101, 150, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 216, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 56, 192, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 28, 6, 0, 0],
        ]),
        // letter: 'Ě' / 0x11a
        'Ě' => Some(&[
            &[0, 0, 0, 161, 91, 0, 4, 162, 93, 0, 0, 0],
            &[0, 0, 0, 7, 197, 84, 162, 127, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 239, 184, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 2, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 144, 144, 144, 144, 108, 0, 0],
            &[0, 1, 255, 122, 107, 107, 107, 107, 107, 80, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 167, 158, 158, 158, 158, 158, 58, 0, 0],
            &[0, 1, 255, 110, 94, 94, 94, 94, 94, 34, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 26, 1, 1, 1, 1, 1, 1, 0, 0],
            &[0, 1, 255, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ě' / 0x11b
        'ě' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 116, 11, 0, 0, 67, 66, 0, 0, 0],
            &[0, 0, 0, 73, 193, 11, 78, 193, 10, 0, 0, 0],
            &[0, 0, 0, 0, 117, 197, 220, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 114, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 60, 60, 16, 0, 0, 0, 0],
            &[0, 0, 7, 153, 232, 184, 177, 232, 121, 0, 0, 0],
            &[0, 0, 168, 178, 17, 0, 0, 21, 212, 94, 0, 0],
            &[0, 55, 231, 15, 0, 0, 0, 0, 76, 209, 0, 0],
            &[0, 134, 155, 0, 0, 0, 0, 0, 15, 255, 10, 0],
            &[0, 167, 232, 213, 213, 213, 213, 213, 213, 255, 30, 0],
            &[0, 170, 128, 20, 20, 20, 20, 20, 20, 20, 2, 0],
            &[0, 138, 153, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 233, 15, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 160, 191, 34, 0, 0, 0, 24, 74, 0, 0],
            &[0, 0, 2, 121, 228, 218, 201, 225, 223, 118, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 178, 236, 27, 0, 0, 0],
            &[0, 0, 0, 0, 0, 118, 161, 85, 195, 6, 0, 0],
            &[0, 0, 0, 0, 81, 167, 4, 0, 94, 161, 0, 0],
            &[0, 0, 0, 0, 4, 1, 0, 0, 0, 5, 0, 0],
            &[0, 0, 0, 0, 5, 96, 163, 190, 170, 116, 23, 0],
            &[0, 0, 0, 25, 212, 197, 98, 67, 87, 158, 111, 0],
            &[0, 0, 1, 202, 150, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 85, 224, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 167, 126, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 227, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 249, 39, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 255, 27, 0, 0, 3, 234, 234, 234, 197, 0],
            &[0, 0, 247, 42, 0, 0, 0, 23, 23, 79, 215, 0],
            &[0, 0, 220, 77, 0, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 155, 147, 0, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 55, 244, 30, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 0, 144, 219, 61, 0, 0, 1, 97, 215, 0],
            &[0, 0, 0, 0, 110, 226, 237, 216, 241, 219, 116, 0],
            &[0, 0, 0, 0, 0, 0, 23, 39, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĝ' / 0x11d
        'ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 130, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 176, 218, 22, 0, 0, 0, 0],
            &[0, 0, 0, 95, 169, 2, 76, 189, 7, 0, 0, 0],
            &[0, 0, 5, 106, 4, 0, 0, 64, 52, 0, 0, 0],
            &[0, 0, 0, 2, 49, 69, 35, 0, 6, 18, 0, 0],
            &[0, 0, 46, 217, 205, 168, 200, 172, 86, 166, 0, 0],
            &[0, 12, 228, 101, 0, 0, 0, 82, 228, 166, 0, 0],
            &[0, 105, 204, 0, 0, 0, 0, 0, 183, 166, 0, 0],
            &[0, 169, 123, 0, 0, 0, 0, 0, 120, 166, 0, 0],
            &[0, 197, 95, 0, 0, 0, 0, 0, 99, 166, 0, 0],
            &[0, 198, 95, 0, 0, 0, 0, 0, 102, 166, 0, 0],
            &[0, 172, 123, 0, 0, 0, 0, 0, 129, 166, 0, 0],
            &[0, 109, 207, 0, 0, 0, 0, 0, 207, 166, 0, 0],
            &[0, 14, 227, 132, 1, 0, 1, 131, 201, 166, 0, 0],
            &[0, 0, 40, 200, 238, 198, 219, 139, 102, 166, 0, 0],
            &[0, 0, 0, 0, 25, 40, 11, 0, 106, 163, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 140, 140, 0, 0],
            &[0, 2, 0, 0, 0, 0, 0, 15, 233, 67, 0, 0],
            &[0, 0, 51, 121, 152, 165, 140, 66, 0, 0, 0, 0],
        ]),
        // letter: 'Ğ' / 0x11e
        'Ğ' => Some(&[
            &[0, 0, 0, 0, 45, 40, 0, 0, 6, 77, 0, 0],
            &[0, 0, 0, 0, 56, 193, 14, 0, 115, 135, 0, 0],
            &[0, 0, 0, 0, 0, 101, 208, 207, 151, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 96, 163, 190, 170, 116, 23, 0],
            &[0, 0, 0, 25, 212, 197, 98, 67, 87, 158, 111, 0],
            &[0, 0, 1, 202, 150, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 85, 224, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 167, 126, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 227, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 249, 39, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 255, 27, 0, 0, 3, 234, 234, 234, 197, 0],
            &[0, 0, 247, 42, 0, 0, 0, 23, 23, 79, 215, 0],
            &[0, 0, 220, 77, 0, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 155, 147, 0, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 55, 244, 30, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 0, 144, 219, 61, 0, 0, 1, 97, 215, 0],
            &[0, 0, 0, 0, 110, 226, 237, 216, 241, 219, 116, 0],
            &[0, 0, 0, 0, 0, 0, 23, 39, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ğ' / 0x11f
        'ğ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 193, 19, 0, 0, 150, 54, 0, 0, 0],
            &[0, 0, 0, 88, 207, 122, 155, 181, 2, 0, 0, 0],
            &[0, 0, 0, 0, 35, 89, 65, 1, 0, 0, 0, 0],
            &[0, 0, 0, 2, 49, 69, 35, 0, 6, 18, 0, 0],
            &[0, 0, 46, 217, 205, 168, 200, 172, 86, 166, 0, 0],
            &[0, 12, 228, 101, 0, 0, 0, 82, 228, 166, 0, 0],
            &[0, 105, 204, 0, 0, 0, 0, 0, 183, 166, 0, 0],
            &[0, 169, 123, 0, 0, 0, 0, 0, 120, 166, 0, 0],
            &[0, 197, 95, 0, 0, 0, 0, 0, 99, 166, 0, 0],
            &[0, 198, 95, 0, 0, 0, 0, 0, 102, 166, 0, 0],
            &[0, 172, 123, 0, 0, 0, 0, 0, 129, 166, 0, 0],
            &[0, 109, 207, 0, 0, 0, 0, 0, 207, 166, 0, 0],
            &[0, 14, 227, 132, 1, 0, 1, 131, 201, 166, 0, 0],
            &[0, 0, 40, 200, 238, 198, 219, 139, 102, 166, 0, 0],
            &[0, 0, 0, 0, 25, 40, 11, 0, 106, 163, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 140, 140, 0, 0],
            &[0, 2, 0, 0, 0, 0, 0, 15, 233, 67, 0, 0],
            &[0, 0, 51, 121, 152, 165, 140, 66, 0, 0, 0, 0],
        ]),
        // letter: 'Ġ' / 0x120
        'Ġ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 51, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 166, 244, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 52, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 96, 163, 190, 170, 116, 23, 0],
            &[0, 0, 0, 25, 212, 197, 98, 67, 87, 158, 111, 0],
            &[0, 0, 1, 202, 150, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 85, 224, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 167, 126, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 227, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 249, 39, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 255, 27, 0, 0, 3, 234, 234, 234, 197, 0],
            &[0, 0, 247, 42, 0, 0, 0, 23, 23, 79, 215, 0],
            &[0, 0, 220, 77, 0, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 155, 147, 0, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 55, 244, 30, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 0, 144, 219, 61, 0, 0, 1, 97, 215, 0],
            &[0, 0, 0, 0, 110, 226, 237, 216, 241, 219, 116, 0],
            &[0, 0, 0, 0, 0, 0, 23, 39, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ġ' / 0x121
        'ġ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 253, 122, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 151, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 49, 69, 35, 0, 6, 18, 0, 0],
            &[0, 0, 46, 217, 205, 168, 200, 172, 86, 166, 0, 0],
            &[0, 12, 228, 101, 0, 0, 0, 82, 228, 166, 0, 0],
            &[0, 105, 204, 0, 0, 0, 0, 0, 183, 166, 0, 0],
            &[0, 169, 123, 0, 0, 0, 0, 0, 120, 166, 0, 0],
            &[0, 197, 95, 0, 0, 0, 0, 0, 99, 166, 0, 0],
            &[0, 198, 95, 0, 0, 0, 0, 0, 102, 166, 0, 0],
            &[0, 172, 123, 0, 0, 0, 0, 0, 129, 166, 0, 0],
            &[0, 109, 207, 0, 0, 0, 0, 0, 207, 166, 0, 0],
            &[0, 14, 227, 132, 1, 0, 1, 131, 201, 166, 0, 0],
            &[0, 0, 40, 200, 238, 198, 219, 139, 102, 166, 0, 0],
            &[0, 0, 0, 0, 25, 40, 11, 0, 106, 163, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 140, 140, 0, 0],
            &[0, 2, 0, 0, 0, 0, 0, 15, 233, 67, 0, 0],
            &[0, 0, 51, 121, 152, 165, 140, 66, 0, 0, 0, 0],
        ]),
        // letter: 'Ģ' / 0x122
        'Ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 96, 163, 190, 170, 116, 23, 0],
            &[0, 0, 0, 25, 212, 197, 98, 67, 87, 158, 111, 0],
            &[0, 0, 1, 202, 150, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 85, 224, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 167, 126, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 227, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 249, 39, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 255, 27, 0, 0, 3, 234, 234, 234, 197, 0],
            &[0, 0, 247, 42, 0, 0, 0, 23, 23, 79, 215, 0],
            &[0, 0, 220, 77, 0, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 155, 147, 0, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 55, 244, 30, 0, 0, 0, 0, 61, 215, 0],
            &[0, 0, 0, 144, 219, 61, 0, 0, 1, 97, 215, 0],
            &[0, 0, 0, 0, 110, 226, 237, 216, 241, 219, 116, 0],
            &[0, 0, 0, 0, 0, 0, 23, 39, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 50, 184, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 117, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ģ' / 0x123
        'ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 90, 137, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 202, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 128, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 49, 69, 35, 0, 6, 18, 0, 0],
            &[0, 0, 46, 217, 205, 168, 200, 172, 86, 166, 0, 0],
            &[0, 12, 228, 101, 0, 0, 0, 82, 228, 166, 0, 0],
            &[0, 105, 204, 0, 0, 0, 0, 0, 183, 166, 0, 0],
            &[0, 169, 123, 0, 0, 0, 0, 0, 120, 166, 0, 0],
            &[0, 197, 95, 0, 0, 0, 0, 0, 99, 166, 0, 0],
            &[0, 198, 95, 0, 0, 0, 0, 0, 102, 166, 0, 0],
            &[0, 172, 123, 0, 0, 0, 0, 0, 129, 166, 0, 0],
            &[0, 109, 207, 0, 0, 0, 0, 0, 207, 166, 0, 0],
            &[0, 14, 227, 132, 1, 0, 1, 131, 201, 166, 0, 0],
            &[0, 0, 40, 200, 238, 198, 219, 139, 102, 166, 0, 0],
            &[0, 0, 0, 0, 25, 40, 11, 0, 106, 163, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 140, 140, 0, 0],
            &[0, 2, 0, 0, 0, 0, 0, 15, 233, 67, 0, 0],
            &[0, 0, 51, 121, 152, 165, 140, 66, 0, 0, 0, 0],
        ]),
        // letter: 'Ĥ' / 0x124
        'Ĥ' => Some(&[
            &[0, 0, 0, 0, 47, 246, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 208, 63, 191, 88, 0, 0, 0, 0],
            &[0, 0, 4, 180, 67, 0, 13, 186, 56, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 2, 3, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 37, 121, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 224, 158, 158, 158, 158, 158, 183, 215, 0, 0],
            &[0, 108, 205, 99, 99, 99, 99, 99, 140, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĥ' / 0x125
        'ĥ' => Some(&[
            &[0, 61, 206, 94, 202, 9, 0, 0, 0, 0, 0, 0],
            &[37, 195, 26, 0, 91, 169, 2, 0, 0, 0, 0, 0],
            &[19, 14, 0, 0, 0, 29, 4, 0, 0, 0, 0, 0],
            &[0, 0, 26, 97, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 222, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 222, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 222, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 222, 0, 4, 55, 64, 24, 0, 0, 0],
            &[0, 0, 59, 220, 50, 205, 179, 180, 238, 146, 0, 0],
            &[0, 0, 59, 224, 177, 22, 0, 0, 26, 230, 90, 0],
            &[0, 0, 59, 255, 60, 0, 0, 0, 0, 129, 159, 0],
            &[0, 0, 59, 244, 1, 0, 0, 0, 0, 96, 188, 0],
            &[0, 0, 59, 224, 0, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 114, 217, 36, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ħ' / 0x126
        'Ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 37, 121, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[231, 255, 255, 231, 231, 231, 231, 231, 255, 255, 231, 92],
            &[6, 114, 180, 6, 6, 6, 6, 6, 72, 221, 6, 2],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 224, 158, 158, 158, 158, 158, 183, 215, 0, 0],
            &[0, 108, 205, 99, 99, 99, 99, 99, 140, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ħ' / 0x127
        'ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 97, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 222, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 152, 212, 255, 152, 152, 152, 81, 0, 0, 0, 0],
            &[0, 70, 130, 255, 70, 70, 70, 37, 0, 0, 0, 0],
            &[0, 0, 59, 222, 0, 4, 55, 64, 24, 0, 0, 0],
            &[0, 0, 59, 220, 50, 205, 179, 180, 238, 146, 0, 0],
            &[0, 0, 59, 224, 177, 22, 0, 0, 26, 230, 90, 0],
            &[0, 0, 59, 255, 60, 0, 0, 0, 0, 129, 159, 0],
            &[0, 0, 59, 244, 1, 0, 0, 0, 0, 96, 188, 0],
            &[0, 0, 59, 224, 0, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĩ' / 0x128
        'Ĩ' => Some(&[
            &[0, 0, 0, 14, 43, 2, 0, 0, 48, 0, 0, 0],
            &[0, 0, 28, 218, 191, 201, 50, 39, 193, 0, 0, 0],
            &[0, 0, 105, 85, 0, 58, 195, 208, 54, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 144, 144, 144, 144, 144, 120, 0, 0, 0],
            &[0, 0, 25, 68, 83, 231, 129, 74, 50, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 88, 213, 228, 253, 238, 219, 171, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĩ' / 0x129
        'ĩ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 131, 182, 90, 0, 1, 159, 0, 0, 0],
            &[0, 0, 83, 162, 52, 172, 188, 176, 135, 0, 0, 0],
            &[0, 0, 48, 23, 0, 0, 56, 70, 0, 0, 0, 0],
            &[0, 0, 26, 27, 27, 27, 16, 0, 0, 0, 0, 0],
            &[0, 0, 171, 190, 202, 234, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ī' / 0x12a
        'Ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 62, 67, 67, 67, 67, 21, 0, 0, 0],
            &[0, 0, 0, 152, 165, 165, 165, 165, 53, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 144, 144, 144, 144, 144, 120, 0, 0, 0],
            &[0, 0, 25, 68, 83, 231, 129, 74, 50, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 88, 213, 228, 253, 238, 219, 171, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ī' / 0x12b
        'ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 190, 206, 206, 206, 206, 66, 0, 0, 0],
            &[0, 0, 0, 25, 27, 27, 27, 27, 8, 0, 0, 0],
            &[0, 0, 26, 27, 27, 27, 16, 0, 0, 0, 0, 0],
            &[0, 0, 171, 190, 202, 234, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => Some(&[
            &[0, 0, 0, 85, 0, 0, 0, 53, 30, 0, 0, 0],
            &[0, 0, 0, 172, 91, 0, 21, 204, 27, 0, 0, 0],
            &[0, 0, 0, 23, 170, 211, 198, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 144, 144, 144, 144, 144, 120, 0, 0, 0],
            &[0, 0, 25, 68, 83, 231, 129, 74, 50, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 88, 213, 228, 253, 238, 219, 171, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĭ' / 0x12d
        'ĭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 193, 19, 0, 0, 150, 54, 0, 0, 0],
            &[0, 0, 0, 88, 207, 122, 155, 181, 2, 0, 0, 0],
            &[0, 0, 0, 0, 35, 89, 65, 1, 0, 0, 0, 0],
            &[0, 0, 26, 27, 27, 27, 16, 0, 0, 0, 0, 0],
            &[0, 0, 171, 190, 202, 234, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Į' / 0x12e
        'Į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 144, 144, 144, 144, 144, 120, 0, 0, 0],
            &[0, 0, 25, 68, 83, 231, 129, 74, 50, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 88, 213, 228, 253, 238, 248, 171, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 106, 139, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 36, 189, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 92, 153, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 29, 3, 0, 0, 0],
        ]),
        // letter: 'į' / 0x12f
        'į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 218, 181, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 118, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 27, 27, 27, 16, 0, 0, 0, 0, 0],
            &[0, 0, 171, 190, 202, 234, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 237, 242, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 6, 178, 61, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 183, 63, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 17, 22, 0, 0, 0],
        ]),
        // letter: 'İ' / 0x130
        'İ' => Some(&[
            &[0, 0, 0, 0, 0, 107, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 255, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 144, 144, 144, 144, 144, 120, 0, 0, 0],
            &[0, 0, 25, 68, 83, 231, 129, 74, 50, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 88, 213, 228, 253, 238, 219, 171, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ı' / 0x131
        'ı' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 27, 27, 27, 16, 0, 0, 0, 0, 0],
            &[0, 0, 171, 190, 202, 234, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĳ' / 0x132
        'Ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 137, 22, 0, 0, 0, 0, 106, 53, 0, 0],
            &[0, 0, 242, 39, 0, 0, 0, 0, 188, 93, 0, 0],
            &[0, 0, 242, 39, 0, 0, 0, 0, 188, 93, 0, 0],
            &[0, 0, 242, 39, 0, 0, 0, 0, 188, 93, 0, 0],
            &[0, 0, 242, 39, 0, 0, 0, 0, 188, 93, 0, 0],
            &[0, 0, 242, 39, 0, 0, 0, 0, 188, 93, 0, 0],
            &[0, 0, 242, 39, 0, 0, 0, 0, 188, 93, 0, 0],
            &[0, 0, 242, 39, 0, 0, 0, 0, 188, 93, 0, 0],
            &[0, 0, 242, 39, 0, 0, 0, 0, 188, 93, 0, 0],
            &[0, 0, 242, 39, 0, 0, 0, 0, 188, 93, 0, 0],
            &[0, 0, 168, 27, 0, 0, 0, 0, 200, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 10, 244, 46, 0, 0],
            &[0, 0, 73, 12, 0, 0, 9, 161, 203, 0, 0, 0],
            &[0, 0, 164, 245, 217, 206, 242, 191, 30, 0, 0, 0],
            &[0, 0, 0, 8, 43, 51, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĳ' / 0x133
        'ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 14, 0, 0, 0, 0, 24, 0, 0, 0],
            &[0, 0, 190, 210, 0, 0, 0, 103, 255, 42, 0, 0],
            &[0, 0, 98, 114, 0, 0, 0, 44, 154, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 15, 0, 0, 0, 3, 27, 0, 0, 0],
            &[0, 0, 135, 146, 0, 0, 0, 29, 252, 0, 0, 0],
            &[0, 0, 135, 146, 0, 0, 0, 29, 252, 0, 0, 0],
            &[0, 0, 135, 146, 0, 0, 0, 29, 252, 0, 0, 0],
            &[0, 0, 135, 146, 0, 0, 0, 29, 252, 0, 0, 0],
            &[0, 0, 135, 146, 0, 0, 0, 29, 252, 0, 0, 0],
            &[0, 0, 135, 146, 0, 0, 0, 29, 252, 0, 0, 0],
            &[0, 0, 135, 146, 0, 0, 0, 29, 252, 0, 0, 0],
            &[0, 0, 135, 146, 0, 0, 0, 29, 252, 0, 0, 0],
            &[0, 0, 135, 146, 0, 0, 0, 29, 252, 0, 0, 0],
            &[0, 0, 135, 146, 0, 0, 0, 29, 252, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 29, 252, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 38, 242, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 109, 198, 0, 0, 0],
            &[0, 0, 39, 122, 156, 161, 126, 31, 0, 0, 0, 0],
        ]),
        // letter: 'Ĵ' / 0x134
        'Ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 21, 229, 190, 1, 0, 0, 0],
            &[0, 0, 0, 0, 3, 187, 90, 156, 130, 0, 0, 0],
            &[0, 0, 0, 0, 151, 101, 0, 3, 161, 91, 0, 0],
            &[0, 0, 0, 0, 5, 0, 0, 0, 0, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 60, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 175, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 230, 64, 0, 0, 0, 0],
            &[0, 75, 5, 0, 1, 122, 226, 6, 0, 0, 0, 0],
            &[0, 172, 245, 211, 228, 222, 54, 0, 0, 0, 0, 0],
            &[0, 0, 15, 47, 39, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĵ' / 0x135
        'ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 136, 21, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 202, 169, 178, 1, 0, 0, 0],
            &[0, 0, 0, 1, 164, 103, 0, 141, 133, 0, 0, 0],
            &[0, 0, 0, 34, 81, 0, 0, 0, 98, 18, 0, 0],
            &[0, 0, 10, 27, 27, 27, 27, 4, 0, 0, 0, 0],
            &[0, 0, 69, 183, 195, 208, 253, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 240, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 251, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 60, 244, 2, 0, 0, 0, 0],
            &[0, 68, 141, 164, 143, 67, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ķ' / 0x136
        'Ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 51, 134, 7, 0],
            &[0, 108, 173, 0, 0, 0, 0, 32, 230, 82, 0, 0],
            &[0, 108, 173, 0, 0, 0, 16, 214, 118, 0, 0, 0],
            &[0, 108, 173, 0, 0, 6, 191, 156, 0, 0, 0, 0],
            &[0, 108, 173, 0, 0, 163, 189, 5, 0, 0, 0, 0],
            &[0, 108, 173, 0, 130, 215, 16, 0, 0, 0, 0, 0],
            &[0, 108, 173, 90, 255, 103, 0, 0, 0, 0, 0, 0],
            &[0, 108, 208, 221, 100, 232, 20, 0, 0, 0, 0, 0],
            &[0, 108, 238, 46, 0, 147, 170, 0, 0, 0, 0, 0],
            &[0, 108, 173, 0, 0, 11, 220, 86, 0, 0, 0, 0],
            &[0, 108, 173, 0, 0, 0, 62, 233, 22, 0, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 145, 173, 0, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 10, 220, 89, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 61, 234, 24, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 128, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 43, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 16, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ķ' / 0x137
        'ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 74, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 0, 3, 27, 6, 0, 0],
            &[0, 169, 112, 0, 0, 0, 2, 164, 175, 4, 0, 0],
            &[0, 169, 112, 0, 0, 0, 153, 186, 7, 0, 0, 0],
            &[0, 169, 112, 0, 0, 141, 196, 10, 0, 0, 0, 0],
            &[0, 169, 109, 0, 129, 206, 15, 0, 0, 0, 0, 0],
            &[0, 169, 103, 116, 244, 149, 0, 0, 0, 0, 0, 0],
            &[0, 169, 184, 190, 27, 222, 79, 0, 0, 0, 0, 0],
            &[0, 169, 189, 9, 0, 57, 233, 28, 0, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 125, 194, 3, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 3, 194, 128, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 0, 29, 235, 63, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 196, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 102, 149, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĸ' / 0x138
        'ĸ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 12, 0, 0, 0, 0, 3, 27, 6, 0, 0],
            &[0, 169, 112, 0, 0, 0, 2, 164, 175, 4, 0, 0],
            &[0, 169, 112, 0, 0, 0, 153, 186, 7, 0, 0, 0],
            &[0, 169, 112, 0, 0, 141, 196, 10, 0, 0, 0, 0],
            &[0, 169, 109, 0, 129, 206, 15, 0, 0, 0, 0, 0],
            &[0, 169, 103, 116, 244, 149, 0, 0, 0, 0, 0, 0],
            &[0, 169, 184, 190, 27, 222, 79, 0, 0, 0, 0, 0],
            &[0, 169, 189, 9, 0, 57, 233, 28, 0, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 125, 194, 3, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 3, 194, 128, 0, 0, 0],
            &[0, 169, 112, 0, 0, 0, 0, 29, 235, 63, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĺ' / 0x139
        'Ĺ' => Some(&[
            &[0, 0, 0, 7, 212, 121, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 152, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 151, 2, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 68, 91, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 163, 6, 6, 6, 6, 6, 4, 0, 0],
            &[0, 0, 121, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĺ' / 0x13a
        'ĺ' => Some(&[
            &[0, 0, 0, 0, 0, 11, 210, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 171, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 28, 0, 0, 0, 0, 0, 0],
            &[0, 2, 112, 112, 112, 112, 66, 0, 0, 0, 0, 0],
            &[0, 2, 93, 105, 118, 192, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 218, 40, 0, 0, 0],
        ]),
        // letter: 'Ļ' / 0x13b
        'Ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 68, 91, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 163, 6, 6, 6, 6, 6, 4, 0, 0],
            &[0, 0, 121, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 188, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ļ' / 0x13c
        'ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 112, 112, 112, 112, 66, 0, 0, 0, 0, 0],
            &[0, 2, 93, 105, 118, 192, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 118, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 195, 57, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 19, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ľ' / 0x13d
        'Ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 68, 91, 0, 0, 0, 0, 113, 45, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 2, 234, 22, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 55, 164, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 55, 31, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 163, 6, 6, 6, 6, 6, 4, 0, 0],
            &[0, 0, 121, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ľ' / 0x13e
        'ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 112, 112, 112, 112, 66, 0, 3, 112, 8, 0],
            &[0, 2, 93, 105, 118, 192, 151, 0, 45, 218, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 111, 113, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 103, 9, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 68, 91, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 17, 205, 72, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 25, 240, 89, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 1, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 160, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 163, 6, 6, 6, 6, 6, 4, 0, 0],
            &[0, 0, 121, 255, 255, 255, 255, 255, 255, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŀ' / 0x140
        'ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[2, 112, 112, 112, 112, 66, 0, 0, 0, 0, 0, 0],
            &[2, 93, 105, 118, 192, 151, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 151, 0, 0, 0, 22, 111, 9],
            &[0, 0, 0, 0, 130, 151, 0, 0, 0, 96, 255, 55],
            &[0, 0, 0, 0, 130, 151, 0, 0, 0, 13, 83, 4],
            &[0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0, 0],
            &[58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ł' / 0x141
        'Ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 103, 56, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 182, 99, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 182, 99, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 182, 99, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 182, 99, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 182, 99, 0, 68, 43, 0, 0, 0, 0],
            &[0, 0, 0, 182, 124, 166, 206, 47, 0, 0, 0, 0],
            &[0, 0, 2, 255, 255, 110, 3, 0, 0, 0, 0, 0],
            &[0, 48, 203, 255, 125, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 68, 182, 99, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 182, 99, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 182, 99, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 182, 103, 6, 6, 6, 6, 6, 3, 0],
            &[0, 0, 0, 182, 255, 255, 255, 255, 255, 255, 129, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ł' / 0x142
        'ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 112, 112, 112, 112, 66, 0, 0, 0, 0, 0],
            &[0, 2, 93, 105, 118, 192, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 153, 102, 203, 2, 0, 0],
            &[0, 0, 0, 0, 0, 171, 255, 171, 25, 0, 0, 0],
            &[0, 0, 0, 7, 129, 255, 229, 0, 0, 0, 0, 0],
            &[0, 0, 1, 200, 145, 143, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 151, 0, 0, 0, 0, 0],
            &[0, 58, 208, 220, 231, 248, 249, 231, 219, 207, 33, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ń' / 0x143
        'Ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 43, 242, 55, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 210, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 172, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0, 0],
            &[0, 61, 144, 15, 0, 0, 0, 0, 34, 121, 0, 0],
            &[0, 108, 249, 124, 0, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 167, 229, 12, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 154, 149, 119, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 161, 30, 228, 10, 0, 0, 61, 215, 0, 0],
            &[0, 108, 163, 0, 155, 114, 0, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 34, 227, 8, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 161, 109, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 38, 225, 7, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 166, 105, 60, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 43, 223, 60, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 172, 146, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 47, 243, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 0, 177, 215, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0],
        ]),
        // letter: 'ń' / 0x144
        'ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 125, 66, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 203, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 203, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 100, 15, 0, 0, 0, 0, 0],
            &[0, 6, 18, 0, 5, 55, 66, 27, 0, 0, 0, 0],
            &[0, 59, 181, 58, 206, 187, 179, 236, 154, 1, 0, 0],
            &[0, 59, 220, 181, 24, 0, 0, 20, 223, 97, 0, 0],
            &[0, 59, 255, 61, 0, 0, 0, 0, 124, 163, 0, 0],
            &[0, 59, 250, 2, 0, 0, 0, 0, 94, 190, 0, 0],
            &[0, 59, 226, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ņ' / 0x145
        'Ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 144, 15, 0, 0, 0, 0, 34, 121, 0, 0],
            &[0, 108, 249, 124, 0, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 167, 229, 12, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 154, 149, 119, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 161, 30, 228, 10, 0, 0, 61, 215, 0, 0],
            &[0, 108, 163, 0, 155, 114, 0, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 34, 227, 8, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 161, 109, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 38, 225, 7, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 166, 105, 60, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 43, 223, 60, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 172, 146, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 47, 243, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 0, 177, 215, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 189, 45, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 222, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 2, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ņ' / 0x146
        'ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 18, 0, 5, 55, 66, 27, 0, 0, 0, 0],
            &[0, 59, 181, 58, 206, 187, 179, 236, 154, 1, 0, 0],
            &[0, 59, 220, 181, 24, 0, 0, 20, 223, 97, 0, 0],
            &[0, 59, 255, 61, 0, 0, 0, 0, 124, 163, 0, 0],
            &[0, 59, 250, 2, 0, 0, 0, 0, 94, 190, 0, 0],
            &[0, 59, 226, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 156, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 229, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 9, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ň' / 0x147
        'Ň' => Some(&[
            &[0, 0, 0, 173, 79, 0, 7, 172, 80, 0, 0, 0],
            &[0, 0, 0, 12, 205, 74, 175, 113, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 244, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 2, 0, 0, 0, 0, 0],
            &[0, 61, 144, 15, 0, 0, 0, 0, 34, 121, 0, 0],
            &[0, 108, 249, 124, 0, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 167, 229, 12, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 154, 149, 119, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 161, 30, 228, 10, 0, 0, 61, 215, 0, 0],
            &[0, 108, 163, 0, 155, 114, 0, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 34, 227, 8, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 161, 109, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 38, 225, 7, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 166, 105, 60, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 43, 223, 60, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 172, 146, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 47, 243, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 0, 177, 215, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ň' / 0x148
        'ň' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 107, 21, 0, 0, 51, 82, 0, 0, 0],
            &[0, 0, 0, 51, 204, 21, 57, 204, 20, 0, 0, 0],
            &[0, 0, 0, 0, 88, 207, 224, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 70, 0, 0, 0, 0, 0],
            &[0, 6, 18, 0, 5, 55, 66, 27, 0, 0, 0, 0],
            &[0, 59, 181, 58, 206, 187, 179, 236, 154, 1, 0, 0],
            &[0, 59, 220, 181, 24, 0, 0, 20, 223, 97, 0, 0],
            &[0, 59, 255, 61, 0, 0, 0, 0, 124, 163, 0, 0],
            &[0, 59, 250, 2, 0, 0, 0, 0, 94, 190, 0, 0],
            &[0, 59, 226, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŉ' / 0x149
        'ŉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 71, 114, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 157, 160, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 208, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[15, 240, 10, 24, 0, 0, 33, 65, 32, 0, 0, 0],
            &[84, 150, 0, 228, 31, 179, 196, 176, 234, 166, 3, 0],
            &[11, 8, 0, 228, 187, 60, 0, 0, 17, 220, 102, 0],
            &[0, 0, 0, 228, 145, 0, 0, 0, 0, 123, 164, 0],
            &[0, 0, 0, 228, 83, 0, 0, 0, 0, 94, 190, 0],
            &[0, 0, 0, 228, 57, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 0, 228, 53, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 0, 228, 53, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 0, 228, 53, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 0, 228, 53, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 0, 228, 53, 0, 0, 0, 0, 90, 191, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 144, 15, 0, 0, 0, 0, 34, 121, 0, 0],
            &[0, 108, 249, 124, 0, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 167, 229, 12, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 154, 149, 119, 0, 0, 0, 61, 215, 0, 0],
            &[0, 108, 161, 30, 228, 10, 0, 0, 61, 215, 0, 0],
            &[0, 108, 163, 0, 155, 114, 0, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 34, 227, 8, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 161, 109, 0, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 38, 225, 7, 61, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 166, 105, 60, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 43, 223, 60, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 172, 146, 215, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 47, 244, 216, 0, 0],
            &[0, 108, 164, 0, 0, 0, 0, 0, 234, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 72, 194, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 167, 142, 0, 0],
            &[0, 0, 0, 0, 0, 145, 148, 198, 217, 21, 0, 0],
            &[0, 0, 0, 0, 0, 69, 109, 80, 9, 0, 0, 0],
        ]),
        // letter: 'ŋ' / 0x14b
        'ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 18, 0, 5, 55, 66, 27, 0, 0, 0, 0],
            &[0, 59, 181, 58, 206, 187, 179, 236, 154, 1, 0, 0],
            &[0, 59, 220, 181, 24, 0, 0, 20, 223, 97, 0, 0],
            &[0, 59, 255, 61, 0, 0, 0, 0, 124, 163, 0, 0],
            &[0, 59, 250, 2, 0, 0, 0, 0, 94, 190, 0, 0],
            &[0, 59, 226, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 91, 196, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 181, 255, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 181, 255, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 181, 255, 0, 0],
            &[0, 59, 222, 0, 0, 0, 0, 0, 181, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 90, 191, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 92, 190, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 122, 167, 0, 0],
            &[0, 0, 0, 0, 0, 6, 120, 135, 65, 0, 0, 0],
        ]),
        // letter: 'Ō' / 0x14c
        'Ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 62, 67, 67, 67, 67, 21, 0, 0, 0],
            &[0, 0, 0, 152, 165, 165, 165, 165, 53, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 148, 187, 176, 103, 4, 0, 0, 0],
            &[0, 0, 80, 238, 124, 66, 82, 190, 189, 5, 0, 0],
            &[0, 20, 240, 61, 0, 0, 0, 2, 191, 123, 0, 0],
            &[0, 125, 172, 0, 0, 0, 0, 0, 60, 231, 1, 0],
            &[0, 193, 93, 0, 0, 0, 0, 0, 2, 238, 47, 0],
            &[0, 239, 47, 0, 0, 0, 0, 0, 0, 198, 88, 0],
            &[2, 254, 28, 0, 0, 0, 0, 0, 0, 178, 107, 0],
            &[8, 255, 23, 0, 0, 0, 0, 0, 0, 176, 110, 0],
            &[0, 250, 37, 0, 0, 0, 0, 0, 0, 190, 97, 0],
            &[0, 216, 73, 0, 0, 0, 0, 0, 0, 224, 63, 0],
            &[0, 158, 135, 0, 0, 0, 0, 0, 28, 248, 12, 0],
            &[0, 66, 232, 14, 0, 0, 0, 0, 135, 170, 0, 0],
            &[0, 0, 173, 186, 25, 0, 1, 99, 238, 37, 0, 0],
            &[0, 0, 9, 149, 244, 221, 239, 205, 51, 0, 0, 0],
            &[0, 0, 0, 0, 11, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ō' / 0x14d
        'ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 190, 206, 206, 206, 206, 66, 0, 0, 0],
            &[0, 0, 0, 25, 27, 27, 27, 27, 8, 0, 0, 0],
            &[0, 0, 0, 0, 31, 66, 52, 7, 0, 0, 0, 0],
            &[0, 0, 25, 186, 216, 169, 188, 229, 92, 0, 0, 0],
            &[0, 7, 212, 123, 1, 0, 0, 40, 229, 73, 0, 0],
            &[0, 97, 210, 1, 0, 0, 0, 0, 93, 204, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 15, 253, 20, 0],
            &[0, 195, 90, 0, 0, 0, 0, 0, 0, 238, 47, 0],
            &[0, 195, 87, 0, 0, 0, 0, 0, 0, 241, 46, 0],
            &[0, 163, 124, 0, 0, 0, 0, 0, 23, 253, 14, 0],
            &[0, 87, 211, 2, 0, 0, 0, 0, 111, 191, 0, 0],
            &[0, 3, 198, 149, 9, 0, 0, 69, 237, 57, 0, 0],
            &[0, 0, 16, 163, 232, 199, 218, 216, 69, 0, 0, 0],
            &[0, 0, 0, 0, 12, 41, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => Some(&[
            &[0, 0, 0, 85, 0, 0, 0, 53, 30, 0, 0, 0],
            &[0, 0, 0, 172, 91, 0, 21, 204, 27, 0, 0, 0],
            &[0, 0, 0, 23, 170, 211, 198, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 148, 187, 176, 103, 4, 0, 0, 0],
            &[0, 0, 80, 238, 124, 66, 82, 190, 189, 5, 0, 0],
            &[0, 20, 240, 61, 0, 0, 0, 2, 191, 123, 0, 0],
            &[0, 125, 172, 0, 0, 0, 0, 0, 60, 231, 1, 0],
            &[0, 193, 93, 0, 0, 0, 0, 0, 2, 238, 47, 0],
            &[0, 239, 47, 0, 0, 0, 0, 0, 0, 198, 88, 0],
            &[2, 254, 28, 0, 0, 0, 0, 0, 0, 178, 107, 0],
            &[8, 255, 23, 0, 0, 0, 0, 0, 0, 176, 110, 0],
            &[0, 250, 37, 0, 0, 0, 0, 0, 0, 190, 97, 0],
            &[0, 216, 73, 0, 0, 0, 0, 0, 0, 224, 63, 0],
            &[0, 158, 135, 0, 0, 0, 0, 0, 28, 248, 12, 0],
            &[0, 66, 232, 14, 0, 0, 0, 0, 135, 170, 0, 0],
            &[0, 0, 173, 186, 25, 0, 1, 99, 238, 37, 0, 0],
            &[0, 0, 9, 149, 244, 221, 239, 205, 51, 0, 0, 0],
            &[0, 0, 0, 0, 11, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŏ' / 0x14f
        'ŏ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 193, 19, 0, 0, 150, 54, 0, 0, 0],
            &[0, 0, 0, 88, 207, 122, 155, 181, 2, 0, 0, 0],
            &[0, 0, 0, 0, 35, 89, 65, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 66, 52, 7, 0, 0, 0, 0],
            &[0, 0, 25, 186, 216, 169, 188, 229, 92, 0, 0, 0],
            &[0, 7, 212, 123, 1, 0, 0, 40, 229, 73, 0, 0],
            &[0, 97, 210, 1, 0, 0, 0, 0, 93, 204, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 15, 253, 20, 0],
            &[0, 195, 90, 0, 0, 0, 0, 0, 0, 238, 47, 0],
            &[0, 195, 87, 0, 0, 0, 0, 0, 0, 241, 46, 0],
            &[0, 163, 124, 0, 0, 0, 0, 0, 23, 253, 14, 0],
            &[0, 87, 211, 2, 0, 0, 0, 0, 111, 191, 0, 0],
            &[0, 3, 198, 149, 9, 0, 0, 69, 237, 57, 0, 0],
            &[0, 0, 16, 163, 232, 199, 218, 216, 69, 0, 0, 0],
            &[0, 0, 0, 0, 12, 41, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ő' / 0x150
        'Ő' => Some(&[
            &[0, 0, 0, 0, 98, 210, 5, 96, 210, 5, 0, 0],
            &[0, 0, 0, 20, 220, 41, 19, 219, 41, 0, 0, 0],
            &[0, 0, 0, 167, 68, 0, 167, 68, 0, 0, 0, 0],
            &[0, 0, 0, 4, 0, 0, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 148, 187, 176, 103, 4, 0, 0, 0],
            &[0, 0, 80, 238, 124, 66, 82, 190, 189, 5, 0, 0],
            &[0, 20, 240, 61, 0, 0, 0, 2, 191, 123, 0, 0],
            &[0, 125, 172, 0, 0, 0, 0, 0, 60, 231, 1, 0],
            &[0, 193, 93, 0, 0, 0, 0, 0, 2, 238, 47, 0],
            &[0, 239, 47, 0, 0, 0, 0, 0, 0, 198, 88, 0],
            &[2, 254, 28, 0, 0, 0, 0, 0, 0, 178, 107, 0],
            &[8, 255, 23, 0, 0, 0, 0, 0, 0, 176, 110, 0],
            &[0, 250, 37, 0, 0, 0, 0, 0, 0, 190, 97, 0],
            &[0, 216, 73, 0, 0, 0, 0, 0, 0, 224, 63, 0],
            &[0, 158, 135, 0, 0, 0, 0, 0, 28, 248, 12, 0],
            &[0, 66, 232, 14, 0, 0, 0, 0, 135, 170, 0, 0],
            &[0, 0, 173, 186, 25, 0, 1, 99, 238, 37, 0, 0],
            &[0, 0, 9, 149, 244, 221, 239, 205, 51, 0, 0, 0],
            &[0, 0, 0, 0, 11, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 0, 0, 3, 0, 0, 0],
        ]),
        // letter: 'ő' / 0x151
        'ő' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 135, 5, 35, 135, 5, 0, 0],
            &[0, 0, 0, 0, 180, 118, 0, 178, 118, 0, 0, 0],
            &[0, 0, 0, 93, 166, 0, 92, 166, 0, 0, 0, 0],
            &[0, 0, 0, 99, 5, 0, 99, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 66, 52, 7, 0, 0, 0, 0],
            &[0, 0, 25, 186, 216, 169, 188, 229, 92, 0, 0, 0],
            &[0, 7, 212, 123, 1, 0, 0, 40, 229, 73, 0, 0],
            &[0, 97, 210, 1, 0, 0, 0, 0, 93, 204, 0, 0],
            &[0, 168, 122, 0, 0, 0, 0, 0, 15, 253, 20, 0],
            &[0, 195, 90, 0, 0, 0, 0, 0, 0, 238, 47, 0],
            &[0, 195, 87, 0, 0, 0, 0, 0, 0, 241, 46, 0],
            &[0, 163, 124, 0, 0, 0, 0, 0, 23, 253, 14, 0],
            &[0, 87, 211, 2, 0, 0, 0, 0, 111, 191, 0, 0],
            &[0, 3, 198, 149, 9, 0, 0, 69, 237, 57, 0, 0],
            &[0, 0, 16, 163, 232, 199, 218, 216, 69, 0, 0, 0],
            &[0, 0, 0, 0, 12, 41, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Œ' / 0x152
        'Œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 74, 147, 166, 149, 144, 144, 144, 117, 0],
            &[0, 0, 152, 211, 103, 121, 240, 107, 107, 107, 87, 0],
            &[0, 73, 226, 14, 0, 47, 229, 0, 0, 0, 0, 0],
            &[0, 170, 121, 0, 0, 47, 229, 0, 0, 0, 0, 0],
            &[0, 226, 60, 0, 0, 47, 229, 0, 0, 0, 0, 0],
            &[4, 253, 34, 0, 0, 47, 229, 0, 0, 0, 0, 0],
            &[16, 255, 18, 0, 0, 47, 245, 158, 158, 158, 80, 0],
            &[17, 255, 12, 0, 0, 47, 239, 94, 94, 94, 47, 0],
            &[4, 255, 24, 0, 0, 47, 229, 0, 0, 0, 0, 0],
            &[0, 235, 55, 0, 0, 47, 229, 0, 0, 0, 0, 0],
            &[0, 186, 107, 0, 0, 47, 229, 0, 0, 0, 0, 0],
            &[0, 104, 202, 1, 0, 47, 229, 0, 0, 0, 0, 0],
            &[0, 11, 217, 142, 8, 48, 229, 1, 1, 1, 1, 0],
            &[0, 0, 34, 185, 247, 242, 255, 255, 255, 255, 207, 0],
            &[0, 0, 0, 0, 7, 22, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'œ' / 0x153
        'œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 66, 22, 0, 0, 17, 61, 8, 0, 0],
            &[0, 68, 229, 170, 217, 51, 54, 213, 170, 203, 6, 0],
            &[2, 221, 59, 0, 41, 181, 191, 29, 0, 170, 93, 0],
            &[50, 214, 0, 0, 0, 203, 181, 0, 0, 90, 157, 0],
            &[93, 167, 0, 0, 0, 168, 142, 0, 0, 64, 187, 0],
            &[110, 150, 0, 0, 0, 153, 231, 208, 208, 217, 197, 0],
            &[106, 152, 0, 0, 0, 155, 133, 10, 10, 10, 8, 0],
            &[86, 170, 0, 0, 0, 172, 147, 0, 0, 0, 0, 0],
            &[39, 224, 0, 0, 0, 210, 198, 0, 0, 0, 0, 0],
            &[0, 205, 80, 0, 66, 167, 187, 63, 0, 1, 31, 0],
            &[0, 48, 225, 202, 213, 31, 39, 217, 187, 213, 96, 0],
            &[0, 0, 6, 36, 5, 0, 0, 5, 37, 12, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŕ' / 0x154
        'Ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 63, 239, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 219, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 185, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 138, 116, 59, 0, 0, 0, 0],
            &[0, 1, 255, 117, 103, 115, 144, 226, 193, 15, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 6, 195, 146, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 90, 214, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 66, 226, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 118, 183, 0, 0],
            &[0, 1, 255, 24, 0, 0, 1, 75, 235, 59, 0, 0],
            &[0, 1, 255, 210, 205, 209, 236, 187, 53, 0, 0, 0],
            &[0, 1, 255, 67, 47, 47, 185, 124, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 38, 239, 31, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 136, 174, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 12, 228, 71, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 89, 217, 6, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 193, 121, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0],
        ]),
        // letter: 'ŕ' / 0x155
        'ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 37, 139, 15, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 200, 119, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 152, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 90, 0, 0, 0, 0, 0],
            &[0, 3, 27, 27, 27, 18, 0, 9, 56, 57, 14, 0],
            &[0, 23, 181, 196, 221, 182, 52, 224, 196, 199, 172, 0],
            &[0, 0, 0, 0, 63, 204, 195, 38, 0, 0, 1, 0],
            &[0, 0, 0, 0, 63, 255, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 249, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 225, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 93, 212, 226, 242, 253, 234, 221, 208, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŗ' / 0x156
        'Ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 138, 116, 59, 0, 0, 0, 0],
            &[0, 1, 255, 117, 103, 115, 144, 226, 193, 15, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 6, 195, 146, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 90, 214, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 66, 226, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 118, 183, 0, 0],
            &[0, 1, 255, 24, 0, 0, 1, 75, 235, 59, 0, 0],
            &[0, 1, 255, 210, 205, 209, 236, 187, 53, 0, 0, 0],
            &[0, 1, 255, 67, 47, 47, 185, 124, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 38, 239, 31, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 136, 174, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 12, 228, 71, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 89, 217, 6, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 193, 121, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 188, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŗ' / 0x157
        'ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 27, 27, 27, 18, 0, 9, 56, 57, 14, 0],
            &[0, 23, 181, 196, 221, 182, 52, 224, 196, 199, 172, 0],
            &[0, 0, 0, 0, 63, 204, 195, 38, 0, 0, 1, 0],
            &[0, 0, 0, 0, 63, 255, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 249, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 225, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 93, 212, 226, 242, 253, 234, 221, 208, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 60, 174, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 128, 123, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 32, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ř' / 0x158
        'Ř' => Some(&[
            &[0, 0, 5, 185, 62, 0, 13, 185, 61, 0, 0, 0],
            &[0, 0, 0, 21, 213, 62, 193, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 251, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 1, 0, 0, 0, 0, 0],
            &[0, 0, 144, 144, 144, 138, 116, 59, 0, 0, 0, 0],
            &[0, 1, 255, 117, 103, 115, 144, 226, 193, 15, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 6, 195, 146, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 90, 214, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 66, 226, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 118, 183, 0, 0],
            &[0, 1, 255, 24, 0, 0, 1, 75, 235, 59, 0, 0],
            &[0, 1, 255, 210, 205, 209, 236, 187, 53, 0, 0, 0],
            &[0, 1, 255, 67, 47, 47, 185, 124, 0, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 38, 239, 31, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 136, 174, 0, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 12, 228, 71, 0, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 89, 217, 6, 0],
            &[0, 1, 255, 24, 0, 0, 0, 0, 0, 193, 121, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ř' / 0x159
        'ř' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 104, 0, 0, 0, 102, 30, 0, 0],
            &[0, 0, 0, 0, 135, 141, 0, 138, 142, 0, 0, 0],
            &[0, 0, 0, 0, 2, 182, 189, 183, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 124, 24, 0, 0, 0, 0],
            &[0, 3, 27, 27, 27, 18, 0, 9, 56, 57, 14, 0],
            &[0, 23, 181, 196, 221, 182, 52, 224, 196, 199, 172, 0],
            &[0, 0, 0, 0, 63, 204, 195, 38, 0, 0, 1, 0],
            &[0, 0, 0, 0, 63, 255, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 249, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 225, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 218, 0, 0, 0, 0, 0, 0],
            &[0, 93, 212, 226, 242, 253, 234, 221, 208, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ś' / 0x15a
        'Ś' => Some(&[
            &[0, 0, 0, 0, 0, 0, 7, 212, 122, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 152, 155, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 106, 152, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 130, 178, 185, 161, 109, 26, 0, 0],
            &[0, 0, 68, 241, 147, 89, 68, 95, 162, 109, 0, 0],
            &[0, 0, 218, 97, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 252, 37, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 162, 201, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 151, 242, 161, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 135, 235, 197, 67, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 104, 240, 94, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 222, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 35, 252, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 82, 221, 0, 0],
            &[0, 38, 66, 5, 0, 0, 0, 62, 228, 107, 0, 0],
            &[0, 49, 198, 247, 223, 205, 231, 221, 100, 0, 0, 0],
            &[0, 0, 0, 5, 28, 43, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0],
        ]),
        // letter: 'ś' / 0x15b
        'ś' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 136, 47, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 145, 178, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 91, 186, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 107, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 66, 64, 36, 0, 0, 0, 0],
            &[0, 0, 10, 176, 229, 179, 178, 214, 206, 1, 0, 0],
            &[0, 0, 114, 191, 4, 0, 0, 0, 18, 0, 0, 0],
            &[0, 0, 145, 144, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 73, 235, 77, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 74, 202, 215, 113, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 154, 238, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 55, 253, 31, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 232, 60, 0, 0],
            &[0, 0, 83, 17, 0, 0, 0, 82, 241, 15, 0, 0],
            &[0, 0, 145, 237, 224, 201, 224, 214, 64, 0, 0, 0],
            &[0, 0, 0, 2, 24, 43, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => Some(&[
            &[0, 0, 0, 0, 3, 195, 226, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 138, 143, 102, 182, 2, 0, 0, 0],
            &[0, 0, 0, 98, 152, 1, 0, 112, 144, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 0, 5, 0, 0, 0],
            &[0, 0, 0, 32, 130, 178, 185, 161, 109, 26, 0, 0],
            &[0, 0, 68, 241, 147, 89, 68, 95, 162, 109, 0, 0],
            &[0, 0, 218, 97, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 252, 37, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 162, 201, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 151, 242, 161, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 135, 235, 197, 67, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 104, 240, 94, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 222, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 35, 252, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 82, 221, 0, 0],
            &[0, 38, 66, 5, 0, 0, 0, 62, 228, 107, 0, 0],
            &[0, 49, 198, 247, 223, 205, 231, 221, 100, 0, 0, 0],
            &[0, 0, 0, 5, 28, 43, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŝ' / 0x15d
        'ŝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 126, 67, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 128, 181, 221, 29, 0, 0, 0, 0],
            &[0, 0, 0, 82, 180, 5, 65, 196, 12, 0, 0, 0],
            &[0, 0, 1, 107, 7, 0, 0, 57, 59, 0, 0, 0],
            &[0, 0, 0, 0, 28, 66, 64, 36, 0, 0, 0, 0],
            &[0, 0, 10, 176, 229, 179, 178, 214, 206, 1, 0, 0],
            &[0, 0, 114, 191, 4, 0, 0, 0, 18, 0, 0, 0],
            &[0, 0, 145, 144, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 73, 235, 77, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 74, 202, 215, 113, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 154, 238, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 55, 253, 31, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 232, 60, 0, 0],
            &[0, 0, 83, 17, 0, 0, 0, 82, 241, 15, 0, 0],
            &[0, 0, 145, 237, 224, 201, 224, 214, 64, 0, 0, 0],
            &[0, 0, 0, 2, 24, 43, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ş' / 0x15e
        'Ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 130, 178, 185, 161, 109, 26, 0, 0],
            &[0, 0, 68, 241, 147, 89, 68, 95, 162, 109, 0, 0],
            &[0, 0, 218, 97, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 252, 37, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 162, 201, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 151, 242, 161, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 135, 235, 197, 67, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 104, 240, 94, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 222, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 35, 252, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 82, 221, 0, 0],
            &[0, 38, 66, 5, 0, 0, 0, 62, 228, 107, 0, 0],
            &[0, 49, 198, 247, 223, 214, 234, 221, 100, 0, 0, 0],
            &[0, 0, 0, 5, 34, 240, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 197, 143, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 182, 61, 0, 0, 0, 0],
            &[0, 0, 0, 23, 137, 120, 35, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ş' / 0x15f
        'ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 66, 64, 36, 0, 0, 0, 0],
            &[0, 0, 10, 176, 229, 179, 178, 214, 206, 1, 0, 0],
            &[0, 0, 114, 191, 4, 0, 0, 0, 18, 0, 0, 0],
            &[0, 0, 145, 144, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 73, 235, 77, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 74, 202, 215, 113, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 154, 238, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 55, 253, 31, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 232, 60, 0, 0],
            &[0, 0, 83, 17, 0, 0, 0, 82, 241, 15, 0, 0],
            &[0, 0, 145, 237, 224, 210, 227, 214, 64, 0, 0, 0],
            &[0, 0, 0, 2, 26, 238, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 196, 151, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 168, 75, 0, 0, 0, 0],
            &[0, 0, 0, 16, 136, 123, 41, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Š' / 0x160
        'Š' => Some(&[
            &[0, 0, 0, 107, 144, 1, 0, 110, 149, 0, 0, 0],
            &[0, 0, 0, 0, 149, 139, 105, 182, 3, 0, 0, 0],
            &[0, 0, 0, 0, 5, 206, 226, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 130, 178, 185, 161, 109, 26, 0, 0],
            &[0, 0, 68, 241, 147, 89, 68, 95, 162, 109, 0, 0],
            &[0, 0, 218, 97, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 252, 37, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 162, 201, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 151, 242, 161, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 135, 235, 197, 67, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 104, 240, 94, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 222, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 35, 252, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 82, 221, 0, 0],
            &[0, 38, 66, 5, 0, 0, 0, 62, 228, 107, 0, 0],
            &[0, 49, 198, 247, 223, 205, 231, 221, 100, 0, 0, 0],
            &[0, 0, 0, 5, 28, 43, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'š' / 0x161
        'š' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 117, 10, 0, 0, 70, 63, 0, 0, 0],
            &[0, 0, 0, 77, 191, 9, 82, 190, 8, 0, 0, 0],
            &[0, 0, 0, 0, 122, 196, 219, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 116, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 66, 64, 36, 0, 0, 0, 0],
            &[0, 0, 10, 176, 229, 179, 178, 214, 206, 1, 0, 0],
            &[0, 0, 114, 191, 4, 0, 0, 0, 18, 0, 0, 0],
            &[0, 0, 145, 144, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 73, 235, 77, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 74, 202, 215, 113, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 154, 238, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 55, 253, 31, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 232, 60, 0, 0],
            &[0, 0, 83, 17, 0, 0, 0, 82, 241, 15, 0, 0],
            &[0, 0, 145, 237, 224, 201, 224, 214, 64, 0, 0, 0],
            &[0, 0, 0, 2, 24, 43, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ţ' / 0x162
        'Ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[12, 144, 144, 144, 144, 144, 144, 144, 144, 144, 73, 0],
            &[9, 107, 107, 107, 107, 233, 145, 107, 107, 107, 54, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 227, 66, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 196, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 197, 143, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 182, 61, 0, 0, 0, 0],
            &[0, 0, 0, 23, 137, 120, 35, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ţ' / 0x163
        'ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 82, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 140, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 23, 174, 131, 27, 27, 27, 13, 0, 0],
            &[0, 140, 215, 215, 241, 233, 215, 215, 215, 107, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 165, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 151, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 227, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 142, 240, 207, 207, 130, 0, 0],
            &[0, 0, 0, 0, 0, 0, 108, 152, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 145, 190, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 45, 199, 0, 0, 0],
            &[0, 0, 0, 0, 0, 85, 135, 91, 5, 0, 0, 0],
        ]),
        // letter: 'Ť' / 0x164
        'Ť' => Some(&[
            &[0, 0, 5, 185, 62, 0, 13, 185, 61, 0, 0, 0],
            &[0, 0, 0, 21, 213, 62, 193, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 251, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 1, 0, 0, 0, 0, 0],
            &[12, 144, 144, 144, 144, 144, 144, 144, 144, 144, 73, 0],
            &[9, 107, 107, 107, 107, 233, 145, 107, 107, 107, 54, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ť' / 0x165
        'ť' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 97, 58, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 218, 42, 0, 0],
            &[0, 0, 0, 0, 85, 82, 0, 33, 189, 0, 0, 0],
            &[0, 0, 0, 0, 140, 115, 0, 45, 42, 0, 0, 0],
            &[0, 0, 7, 23, 174, 131, 27, 27, 27, 13, 0, 0],
            &[0, 140, 215, 215, 241, 233, 215, 215, 215, 107, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 165, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 151, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 227, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 142, 238, 197, 207, 130, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 41, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŧ' / 0x166
        'Ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[12, 144, 144, 144, 144, 144, 144, 144, 144, 144, 73, 0],
            &[9, 107, 107, 107, 107, 233, 145, 107, 107, 107, 54, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 97, 104, 104, 255, 169, 104, 104, 36, 0, 0],
            &[0, 0, 128, 138, 138, 255, 202, 138, 138, 48, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŧ' / 0x167
        'ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 82, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 140, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 23, 174, 131, 27, 27, 27, 13, 0, 0],
            &[0, 140, 215, 215, 241, 233, 215, 215, 215, 107, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 39, 59, 59, 225, 175, 59, 59, 59, 37, 0, 0],
            &[0, 106, 159, 159, 255, 255, 159, 159, 159, 100, 0, 0],
            &[0, 0, 0, 0, 166, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 165, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 151, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 227, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 142, 238, 197, 207, 130, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 41, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ũ' / 0x168
        'Ũ' => Some(&[
            &[0, 0, 0, 14, 43, 2, 0, 0, 48, 0, 0, 0],
            &[0, 0, 28, 218, 191, 201, 50, 39, 193, 0, 0, 0],
            &[0, 0, 105, 85, 0, 58, 195, 208, 54, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 37, 121, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 102, 180, 0, 0, 0, 0, 0, 73, 209, 0, 0],
            &[0, 80, 206, 0, 0, 0, 0, 0, 100, 186, 0, 0],
            &[0, 25, 250, 24, 0, 0, 0, 0, 173, 126, 0, 0],
            &[0, 0, 168, 181, 18, 0, 2, 103, 238, 23, 0, 0],
            &[0, 0, 12, 165, 244, 217, 237, 208, 53, 0, 0, 0],
            &[0, 0, 0, 0, 13, 40, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ũ' / 0x169
        'ũ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 131, 182, 90, 0, 1, 159, 0, 0, 0],
            &[0, 0, 83, 162, 52, 172, 188, 176, 135, 0, 0, 0],
            &[0, 0, 48, 23, 0, 0, 56, 70, 0, 0, 0, 0],
            &[0, 9, 21, 0, 0, 0, 0, 0, 12, 18, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 118, 166, 0, 0],
            &[0, 78, 206, 0, 0, 0, 0, 0, 142, 166, 0, 0],
            &[0, 47, 238, 4, 0, 0, 0, 2, 214, 166, 0, 0],
            &[0, 4, 218, 125, 0, 0, 5, 138, 182, 166, 0, 0],
            &[0, 0, 50, 214, 226, 198, 220, 108, 72, 166, 0, 0],
            &[0, 0, 0, 0, 22, 33, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ū' / 0x16a
        'Ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 62, 67, 67, 67, 67, 21, 0, 0, 0],
            &[0, 0, 0, 152, 165, 165, 165, 165, 53, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 37, 121, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 102, 180, 0, 0, 0, 0, 0, 73, 209, 0, 0],
            &[0, 80, 206, 0, 0, 0, 0, 0, 100, 186, 0, 0],
            &[0, 25, 250, 24, 0, 0, 0, 0, 173, 126, 0, 0],
            &[0, 0, 168, 181, 18, 0, 2, 103, 238, 23, 0, 0],
            &[0, 0, 12, 165, 244, 217, 237, 208, 53, 0, 0, 0],
            &[0, 0, 0, 0, 13, 40, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ū' / 0x16b
        'ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 190, 206, 206, 206, 206, 66, 0, 0, 0],
            &[0, 0, 0, 25, 27, 27, 27, 27, 8, 0, 0, 0],
            &[0, 9, 21, 0, 0, 0, 0, 0, 12, 18, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 118, 166, 0, 0],
            &[0, 78, 206, 0, 0, 0, 0, 0, 142, 166, 0, 0],
            &[0, 47, 238, 4, 0, 0, 0, 2, 214, 166, 0, 0],
            &[0, 4, 218, 125, 0, 0, 5, 138, 182, 166, 0, 0],
            &[0, 0, 50, 214, 226, 198, 220, 108, 72, 166, 0, 0],
            &[0, 0, 0, 0, 22, 33, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => Some(&[
            &[0, 0, 0, 85, 0, 0, 0, 53, 30, 0, 0, 0],
            &[0, 0, 0, 172, 91, 0, 21, 204, 27, 0, 0, 0],
            &[0, 0, 0, 23, 170, 211, 198, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 37, 121, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 102, 180, 0, 0, 0, 0, 0, 73, 209, 0, 0],
            &[0, 80, 206, 0, 0, 0, 0, 0, 100, 186, 0, 0],
            &[0, 25, 250, 24, 0, 0, 0, 0, 173, 126, 0, 0],
            &[0, 0, 168, 181, 18, 0, 2, 103, 238, 23, 0, 0],
            &[0, 0, 12, 165, 244, 217, 237, 208, 53, 0, 0, 0],
            &[0, 0, 0, 0, 13, 40, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŭ' / 0x16d
        'ŭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 193, 19, 0, 0, 150, 54, 0, 0, 0],
            &[0, 0, 0, 88, 207, 122, 155, 181, 2, 0, 0, 0],
            &[0, 0, 0, 0, 35, 89, 65, 1, 0, 0, 0, 0],
            &[0, 9, 21, 0, 0, 0, 0, 0, 12, 18, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 118, 166, 0, 0],
            &[0, 78, 206, 0, 0, 0, 0, 0, 142, 166, 0, 0],
            &[0, 47, 238, 4, 0, 0, 0, 2, 214, 166, 0, 0],
            &[0, 4, 218, 125, 0, 0, 5, 138, 182, 166, 0, 0],
            &[0, 0, 50, 214, 226, 198, 220, 108, 72, 166, 0, 0],
            &[0, 0, 0, 0, 22, 33, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ů' / 0x16e
        'Ů' => Some(&[
            &[0, 0, 0, 48, 172, 0, 69, 150, 0, 0, 0, 0],
            &[0, 0, 0, 53, 166, 0, 64, 153, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 196, 190, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 37, 121, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 102, 180, 0, 0, 0, 0, 0, 73, 209, 0, 0],
            &[0, 80, 206, 0, 0, 0, 0, 0, 100, 186, 0, 0],
            &[0, 25, 250, 24, 0, 0, 0, 0, 173, 126, 0, 0],
            &[0, 0, 168, 181, 18, 0, 2, 103, 238, 23, 0, 0],
            &[0, 0, 12, 165, 244, 217, 237, 208, 53, 0, 0, 0],
            &[0, 0, 0, 0, 13, 40, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 191, 179, 26, 0, 0, 0, 0],
        ]),
        // letter: 'ů' / 0x16f
        'ů' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 107, 67, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 206, 84, 163, 100, 0, 0, 0, 0],
            &[0, 0, 0, 64, 143, 0, 40, 166, 0, 0, 0, 0],
            &[0, 0, 0, 18, 207, 98, 173, 96, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 97, 59, 0, 0, 0, 0, 0],
            &[0, 9, 21, 0, 0, 0, 0, 0, 12, 18, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 118, 166, 0, 0],
            &[0, 78, 206, 0, 0, 0, 0, 0, 142, 166, 0, 0],
            &[0, 47, 238, 4, 0, 0, 0, 2, 214, 166, 0, 0],
            &[0, 4, 218, 125, 0, 0, 5, 138, 182, 166, 0, 0],
            &[0, 0, 50, 214, 226, 198, 220, 108, 72, 166, 0, 0],
            &[0, 0, 0, 0, 22, 33, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ű' / 0x170
        'Ű' => Some(&[
            &[0, 0, 0, 0, 98, 210, 5, 96, 210, 5, 0, 0],
            &[0, 0, 0, 20, 220, 41, 19, 219, 41, 0, 0, 0],
            &[0, 0, 0, 167, 68, 0, 167, 68, 0, 0, 0, 0],
            &[0, 0, 0, 4, 0, 0, 4, 0, 0, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 37, 121, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 102, 180, 0, 0, 0, 0, 0, 73, 209, 0, 0],
            &[0, 80, 206, 0, 0, 0, 0, 0, 100, 186, 0, 0],
            &[0, 25, 250, 24, 0, 0, 0, 0, 173, 126, 0, 0],
            &[0, 0, 168, 181, 18, 0, 2, 103, 238, 23, 0, 0],
            &[0, 0, 12, 165, 244, 217, 237, 208, 53, 0, 0, 0],
            &[0, 0, 0, 0, 13, 40, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 0, 0, 3, 0, 0, 0],
        ]),
        // letter: 'ű' / 0x171
        'ű' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 135, 5, 35, 135, 5, 0, 0],
            &[0, 0, 0, 0, 180, 118, 0, 178, 118, 0, 0, 0],
            &[0, 0, 0, 93, 166, 0, 92, 166, 0, 0, 0, 0],
            &[0, 0, 0, 99, 5, 0, 99, 5, 0, 0, 0, 0],
            &[0, 9, 21, 0, 0, 0, 0, 0, 12, 18, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 118, 166, 0, 0],
            &[0, 78, 206, 0, 0, 0, 0, 0, 142, 166, 0, 0],
            &[0, 47, 238, 4, 0, 0, 0, 2, 214, 166, 0, 0],
            &[0, 4, 218, 125, 0, 0, 5, 138, 182, 166, 0, 0],
            &[0, 0, 50, 214, 226, 198, 220, 108, 72, 166, 0, 0],
            &[0, 0, 0, 0, 22, 33, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ų' / 0x172
        'Ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 98, 0, 0, 0, 0, 0, 37, 121, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 108, 173, 0, 0, 0, 0, 0, 66, 215, 0, 0],
            &[0, 102, 180, 0, 0, 0, 0, 0, 73, 209, 0, 0],
            &[0, 80, 206, 0, 0, 0, 0, 0, 100, 186, 0, 0],
            &[0, 25, 250, 24, 0, 0, 0, 0, 173, 126, 0, 0],
            &[0, 0, 168, 181, 18, 0, 2, 103, 238, 23, 0, 0],
            &[0, 0, 12, 165, 244, 217, 255, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 13, 126, 185, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 199, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 102, 150, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 29, 3, 0, 0, 0, 0],
        ]),
        // letter: 'ų' / 0x173
        'ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 21, 0, 0, 0, 0, 0, 12, 18, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 115, 166, 0, 0],
            &[0, 83, 198, 0, 0, 0, 0, 0, 118, 166, 0, 0],
            &[0, 78, 206, 0, 0, 0, 0, 0, 142, 166, 0, 0],
            &[0, 47, 238, 4, 0, 0, 0, 2, 214, 166, 0, 0],
            &[0, 4, 218, 125, 0, 0, 5, 138, 182, 166, 0, 0],
            &[0, 0, 50, 214, 226, 198, 220, 108, 82, 185, 0, 0],
            &[0, 0, 0, 0, 22, 33, 2, 11, 184, 49, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 139, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 201, 45, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 19, 20, 0, 0],
        ]),
        // letter: 'Ŵ' / 0x174
        'Ŵ' => Some(&[
            &[0, 0, 0, 0, 47, 246, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 208, 63, 191, 88, 0, 0, 0, 0],
            &[0, 0, 4, 180, 67, 0, 13, 186, 56, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 2, 3, 0, 0, 0],
            &[48, 102, 0, 0, 0, 0, 0, 0, 0, 42, 109, 0],
            &[58, 208, 0, 0, 0, 0, 0, 0, 0, 101, 166, 0],
            &[23, 242, 0, 0, 0, 0, 0, 0, 0, 134, 132, 0],
            &[0, 244, 20, 0, 0, 23, 7, 0, 0, 168, 98, 0],
            &[0, 209, 53, 0, 4, 244, 100, 0, 0, 203, 64, 0],
            &[0, 175, 87, 0, 51, 198, 159, 0, 0, 238, 30, 0],
            &[0, 140, 121, 0, 110, 108, 207, 0, 16, 249, 2, 0],
            &[0, 105, 155, 0, 169, 49, 192, 22, 50, 217, 0, 0],
            &[0, 70, 189, 0, 216, 3, 134, 80, 83, 183, 0, 0],
            &[0, 36, 220, 31, 189, 0, 76, 140, 108, 149, 0, 0],
            &[0, 5, 243, 87, 132, 0, 18, 200, 132, 115, 0, 0],
            &[0, 0, 222, 158, 74, 0, 0, 209, 160, 81, 0, 0],
            &[0, 0, 187, 227, 17, 0, 0, 156, 225, 47, 0, 0],
            &[0, 0, 153, 214, 0, 0, 0, 97, 255, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŵ' / 0x175
        'ŵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 124, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 123, 182, 222, 32, 0, 0, 0, 0],
            &[0, 0, 0, 78, 183, 6, 62, 198, 13, 0, 0, 0],
            &[0, 0, 0, 107, 8, 0, 0, 55, 61, 0, 0, 0],
            &[16, 12, 0, 0, 0, 0, 0, 0, 0, 0, 27, 0],
            &[121, 141, 0, 0, 0, 0, 0, 0, 0, 27, 230, 0],
            &[69, 194, 0, 0, 0, 54, 24, 0, 0, 78, 180, 0],
            &[17, 244, 3, 0, 18, 249, 145, 0, 0, 130, 130, 0],
            &[0, 220, 46, 0, 92, 163, 220, 0, 0, 181, 80, 0],
            &[0, 168, 99, 0, 169, 77, 206, 42, 0, 232, 30, 0],
            &[0, 115, 153, 5, 234, 11, 133, 118, 27, 236, 0, 0],
            &[0, 63, 200, 67, 184, 0, 57, 194, 73, 186, 0, 0],
            &[0, 13, 237, 134, 108, 0, 2, 231, 123, 136, 0, 0],
            &[0, 0, 214, 207, 33, 0, 0, 162, 207, 86, 0, 0],
            &[0, 0, 161, 213, 0, 0, 0, 86, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŷ' / 0x176
        'Ŷ' => Some(&[
            &[0, 0, 0, 0, 47, 246, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 208, 63, 191, 88, 0, 0, 0, 0],
            &[0, 0, 4, 180, 67, 0, 13, 186, 56, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 2, 3, 0, 0, 0],
            &[8, 141, 25, 0, 0, 0, 0, 0, 0, 106, 65, 0],
            &[0, 159, 144, 0, 0, 0, 0, 0, 35, 240, 23, 0],
            &[0, 32, 240, 25, 0, 0, 0, 0, 157, 134, 0, 0],
            &[0, 0, 147, 144, 0, 0, 0, 33, 234, 17, 0, 0],
            &[0, 0, 24, 236, 25, 0, 0, 155, 123, 0, 0, 0],
            &[0, 0, 0, 135, 144, 0, 31, 228, 11, 0, 0, 0],
            &[0, 0, 0, 17, 231, 25, 152, 112, 0, 0, 0, 0],
            &[0, 0, 0, 0, 123, 171, 220, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 236, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŷ' / 0x177
        'ŷ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 130, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 176, 218, 22, 0, 0, 0, 0],
            &[0, 0, 0, 95, 169, 2, 76, 189, 7, 0, 0, 0],
            &[0, 0, 5, 106, 4, 0, 0, 64, 52, 0, 0, 0],
            &[0, 18, 11, 0, 0, 0, 0, 0, 0, 27, 3, 0],
            &[0, 119, 161, 0, 0, 0, 0, 0, 49, 234, 5, 0],
            &[0, 23, 240, 14, 0, 0, 0, 0, 141, 142, 0, 0],
            &[0, 0, 171, 104, 0, 0, 0, 2, 230, 46, 0, 0],
            &[0, 0, 70, 203, 0, 0, 0, 69, 205, 0, 0, 0],
            &[0, 0, 1, 222, 46, 0, 0, 161, 109, 0, 0, 0],
            &[0, 0, 0, 122, 146, 0, 10, 236, 19, 0, 0, 0],
            &[0, 0, 0, 25, 233, 7, 88, 172, 0, 0, 0, 0],
            &[0, 0, 0, 0, 175, 85, 177, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 186, 224, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 225, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 226, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 201, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 186, 90, 0, 0, 0, 0, 0, 0],
            &[0, 83, 166, 133, 15, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ÿ' / 0x178
        'Ÿ' => Some(&[
            &[0, 0, 0, 35, 5, 0, 0, 33, 6, 0, 0, 0],
            &[0, 0, 4, 249, 98, 0, 1, 244, 105, 0, 0, 0],
            &[0, 0, 0, 96, 25, 0, 0, 93, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[8, 141, 25, 0, 0, 0, 0, 0, 0, 106, 65, 0],
            &[0, 159, 144, 0, 0, 0, 0, 0, 35, 240, 23, 0],
            &[0, 32, 240, 25, 0, 0, 0, 0, 157, 134, 0, 0],
            &[0, 0, 147, 144, 0, 0, 0, 33, 234, 17, 0, 0],
            &[0, 0, 24, 236, 25, 0, 0, 155, 123, 0, 0, 0],
            &[0, 0, 0, 135, 144, 0, 31, 228, 11, 0, 0, 0],
            &[0, 0, 0, 17, 231, 25, 152, 112, 0, 0, 0, 0],
            &[0, 0, 0, 0, 123, 171, 220, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 236, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ź' / 0x179
        'Ź' => Some(&[
            &[0, 0, 0, 0, 0, 0, 16, 228, 97, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 175, 132, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 130, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 144, 144, 144, 144, 144, 144, 55, 0, 0],
            &[0, 0, 56, 103, 103, 103, 103, 106, 247, 62, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 189, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 226, 55, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 113, 176, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 232, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 127, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 236, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 141, 150, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 238, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 156, 137, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 236, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 170, 126, 1, 1, 1, 1, 1, 0, 0, 0],
            &[0, 1, 254, 255, 255, 255, 255, 255, 255, 127, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0],
        ]),
        // letter: 'ź' / 0x17a
        'ź' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 125, 66, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 203, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 203, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 100, 15, 0, 0, 0, 0, 0],
            &[0, 0, 17, 27, 27, 27, 27, 27, 27, 10, 0, 0],
            &[0, 0, 129, 200, 200, 200, 200, 201, 254, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 108, 201, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 233, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 213, 88, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 152, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 215, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 230, 47, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 193, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 126, 176, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 252, 250, 247, 247, 247, 247, 247, 105, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ż' / 0x17b
        'Ż' => Some(&[
            &[0, 0, 0, 0, 0, 74, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 209, 202, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 144, 144, 144, 144, 144, 144, 55, 0, 0],
            &[0, 0, 56, 103, 103, 103, 103, 106, 247, 62, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 189, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 226, 55, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 113, 176, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 232, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 127, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 236, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 141, 150, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 238, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 156, 137, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 236, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 170, 126, 1, 1, 1, 1, 1, 0, 0, 0],
            &[0, 1, 254, 255, 255, 255, 255, 255, 255, 127, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ż' / 0x17c
        'ż' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 227, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 125, 86, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 27, 27, 27, 27, 27, 27, 10, 0, 0],
            &[0, 0, 129, 200, 200, 200, 200, 201, 254, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 108, 201, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 233, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 213, 88, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 152, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 215, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 230, 47, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 193, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 126, 176, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 252, 250, 247, 247, 247, 247, 247, 105, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ž' / 0x17d
        'Ž' => Some(&[
            &[0, 0, 0, 131, 121, 0, 0, 133, 126, 0, 0, 0],
            &[0, 0, 0, 1, 172, 114, 129, 161, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 221, 212, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 3, 0, 0, 0, 0, 0],
            &[0, 0, 79, 144, 144, 144, 144, 144, 144, 55, 0, 0],
            &[0, 0, 56, 103, 103, 103, 103, 106, 247, 62, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 189, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 226, 55, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 113, 176, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 232, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 127, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 236, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 141, 150, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 238, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 156, 137, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 236, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 170, 126, 1, 1, 1, 1, 1, 0, 0, 0],
            &[0, 1, 254, 255, 255, 255, 255, 255, 255, 127, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ž' / 0x17e
        'ž' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 107, 21, 0, 0, 51, 82, 0, 0, 0],
            &[0, 0, 0, 51, 204, 21, 57, 204, 20, 0, 0, 0],
            &[0, 0, 0, 0, 88, 207, 224, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 70, 0, 0, 0, 0, 0],
            &[0, 0, 17, 27, 27, 27, 27, 27, 27, 10, 0, 0],
            &[0, 0, 129, 200, 200, 200, 200, 201, 254, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 108, 201, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 233, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 213, 88, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 152, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 215, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 230, 47, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 193, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 126, 176, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 252, 250, 247, 247, 247, 247, 247, 105, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ſ' / 0x17f
        'ſ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 117, 131, 102, 1, 0, 0, 0],
            &[0, 0, 0, 120, 231, 131, 108, 123, 0, 0, 0, 0],
            &[0, 0, 29, 248, 35, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 89, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 172, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 169, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 169, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 169, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 169, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 169, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 169, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 169, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 169, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 169, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 169, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        _ => None,
    }
}
