//! Module for letters with the font weight light and size 20.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 20;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 11;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight light and font size 17px
#[inline]
pub const fn get_char(c: char) -> Option<&'static [&'static [u8]]> {
    match c {
        // letter: ' ' / 0x20
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '!' / 0x21
        '!' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 119, 175, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 170, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 110, 166, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 102, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 97, 153, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 88, 144, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 112, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 87, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 146, 203, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '"' / 0x22
        '"' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 7, 0, 35, 1, 0, 0, 0],
            &[0, 0, 0, 202, 47, 0, 251, 2, 0, 0, 0],
            &[0, 0, 0, 190, 35, 0, 230, 0, 0, 0, 0],
            &[0, 0, 0, 178, 23, 0, 206, 0, 0, 0, 0],
            &[0, 0, 0, 166, 11, 0, 183, 0, 0, 0, 0],
            &[0, 0, 0, 39, 1, 0, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '#' / 0x23
        '#' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 22, 0, 0, 27, 0, 0, 0],
            &[0, 0, 0, 72, 136, 0, 6, 194, 0, 0, 0],
            &[0, 0, 0, 116, 93, 0, 45, 157, 0, 0, 0],
            &[0, 0, 0, 160, 50, 0, 89, 114, 0, 0, 0],
            &[5, 123, 123, 224, 133, 123, 186, 165, 123, 34, 0],
            &[3, 72, 75, 219, 72, 72, 207, 86, 72, 20, 0],
            &[0, 0, 37, 171, 0, 0, 208, 0, 0, 0, 0],
            &[0, 0, 81, 127, 0, 14, 194, 0, 0, 0, 0],
            &[83, 133, 187, 179, 133, 155, 211, 133, 96, 0, 0],
            &[41, 66, 195, 91, 66, 146, 141, 66, 47, 0, 0],
            &[0, 0, 203, 3, 0, 147, 61, 0, 0, 0, 0],
            &[0, 4, 201, 0, 0, 192, 16, 0, 0, 0, 0],
            &[0, 42, 162, 0, 0, 208, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '$' / 0x24
        '$' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 75, 97, 0, 0, 0, 0, 0],
            &[0, 0, 20, 127, 201, 213, 156, 91, 7, 0, 0],
            &[0, 18, 222, 126, 120, 132, 74, 145, 9, 0, 0],
            &[0, 103, 158, 0, 83, 108, 0, 0, 0, 0, 0],
            &[0, 110, 150, 0, 83, 108, 0, 0, 0, 0, 0],
            &[0, 30, 228, 81, 83, 108, 0, 0, 0, 0, 0],
            &[0, 0, 42, 177, 236, 166, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 95, 195, 224, 146, 3, 0, 0],
            &[0, 0, 0, 0, 83, 108, 8, 183, 114, 0, 0],
            &[0, 0, 0, 0, 83, 108, 0, 83, 166, 0, 0],
            &[0, 0, 0, 0, 83, 108, 0, 136, 127, 0, 0],
            &[0, 88, 175, 120, 146, 172, 168, 196, 12, 0, 0],
            &[0, 6, 64, 109, 170, 173, 67, 1, 0, 0, 0],
            &[0, 0, 0, 0, 82, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '%' / 0x25
        '%' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 57, 13, 0, 0, 0, 12, 19, 0, 0],
            &[77, 186, 122, 195, 36, 0, 0, 163, 66, 0, 0],
            &[203, 15, 0, 65, 152, 0, 40, 189, 0, 0, 0],
            &[215, 0, 0, 25, 190, 0, 168, 62, 0, 0, 0],
            &[210, 6, 0, 52, 166, 44, 186, 0, 0, 0, 0],
            &[110, 151, 67, 185, 63, 172, 58, 0, 0, 0, 0],
            &[0, 68, 112, 49, 47, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 176, 54, 50, 138, 120, 12, 0],
            &[0, 0, 0, 51, 178, 36, 197, 42, 80, 182, 0],
            &[0, 0, 0, 180, 50, 121, 94, 0, 0, 190, 25],
            &[0, 0, 55, 174, 0, 136, 78, 0, 0, 174, 40],
            &[0, 0, 183, 46, 0, 89, 131, 0, 3, 210, 8],
            &[0, 59, 170, 0, 0, 3, 172, 160, 191, 97, 0],
            &[0, 0, 0, 0, 0, 0, 0, 21, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '&' / 0x26
        '&' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 72, 26, 0, 0, 0, 0, 0],
            &[0, 0, 81, 206, 133, 209, 65, 0, 0, 0, 0],
            &[0, 0, 201, 39, 0, 60, 179, 0, 0, 0, 0],
            &[0, 0, 210, 23, 0, 44, 192, 0, 0, 0, 0],
            &[0, 0, 151, 91, 0, 150, 116, 0, 0, 0, 0],
            &[0, 0, 35, 212, 138, 178, 2, 0, 0, 0, 0],
            &[0, 0, 20, 255, 255, 4, 0, 0, 0, 0, 0],
            &[0, 23, 216, 87, 173, 102, 0, 0, 120, 33, 0],
            &[0, 158, 113, 0, 14, 211, 54, 2, 224, 7, 0],
            &[0, 228, 16, 0, 0, 46, 217, 101, 168, 0, 0],
            &[0, 228, 22, 0, 0, 0, 106, 255, 52, 0, 0],
            &[0, 158, 139, 0, 0, 22, 188, 255, 139, 0, 0],
            &[0, 19, 187, 215, 188, 219, 102, 8, 204, 85, 0],
            &[0, 0, 0, 23, 34, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ''' / 0x27
        '\'' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 101, 152, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 89, 140, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 117, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 27, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '(' / 0x28
        '(' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 156, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 218, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 118, 133, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 214, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 230, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 182, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 89, 162, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 100, 152, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 83, 169, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 197, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 239, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 181, 74, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 181, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 187, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 30, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ')' / 0x29
        ')' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 145, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 194, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 173, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 233, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 177, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 129, 123, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 109, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 101, 150, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 119, 132, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 149, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 223, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 119, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 209, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 106, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '*' / 0x2a
        '*' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 176, 0, 0, 0, 0, 0],
            &[0, 16, 80, 11, 11, 166, 0, 39, 81, 0, 0],
            &[0, 25, 141, 191, 178, 220, 196, 181, 111, 0, 0],
            &[0, 0, 0, 0, 149, 194, 61, 0, 0, 0, 0],
            &[0, 0, 0, 89, 158, 21, 212, 23, 0, 0, 0],
            &[0, 0, 7, 206, 20, 0, 103, 141, 0, 0, 0],
            &[0, 0, 0, 2, 0, 0, 1, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '+' / 0x2b
        '+' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 69, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 74, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 74, 134, 0, 0, 0, 0, 0],
            &[0, 75, 104, 104, 148, 184, 104, 104, 96, 0, 0],
            &[0, 75, 104, 104, 148, 183, 104, 104, 96, 0, 0],
            &[0, 0, 0, 0, 74, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 74, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ',' / 0x2c
        ',' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 245, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 101, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 219, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '-' / 0x2d
        '-' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 25, 25, 25, 25, 16, 0, 0, 0],
            &[0, 0, 90, 200, 200, 200, 200, 130, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '.' / 0x2e
        '.' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 58, 87, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 150, 203, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '/' / 0x2f
        '/' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 82, 165, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 179, 68, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 221, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 118, 129, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 213, 33, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 191, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 227, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 92, 155, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 189, 58, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 215, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 127, 120, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '0' / 0x30
        '0' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 59, 67, 19, 0, 0, 0, 0],
            &[0, 0, 53, 221, 170, 159, 222, 94, 0, 0, 0],
            &[0, 5, 221, 58, 0, 0, 29, 255, 37, 0, 0],
            &[0, 79, 171, 0, 0, 0, 125, 247, 133, 0, 0],
            &[0, 136, 103, 0, 0, 29, 218, 56, 189, 0, 0],
            &[0, 170, 67, 0, 0, 170, 89, 10, 222, 0, 0],
            &[0, 184, 54, 0, 65, 192, 0, 0, 233, 0, 0],
            &[0, 181, 54, 3, 207, 46, 0, 0, 233, 0, 0],
            &[0, 166, 71, 109, 146, 0, 0, 15, 219, 0, 0],
            &[0, 126, 131, 217, 16, 0, 0, 53, 183, 0, 0],
            &[0, 62, 255, 99, 0, 0, 0, 125, 119, 0, 0],
            &[0, 0, 227, 90, 0, 0, 42, 223, 21, 0, 0],
            &[0, 0, 28, 194, 208, 195, 219, 63, 0, 0, 0],
            &[0, 0, 0, 0, 20, 32, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '1' / 0x31
        '1' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 6, 0, 0, 0, 0],
            &[0, 0, 0, 18, 154, 251, 48, 0, 0, 0, 0],
            &[0, 0, 94, 217, 99, 188, 48, 0, 0, 0, 0],
            &[0, 0, 104, 25, 0, 193, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 195, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 195, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 195, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 195, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 195, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 195, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 195, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 195, 48, 0, 0, 0, 0],
            &[0, 0, 28, 182, 195, 255, 248, 187, 97, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '2' / 0x32
        '2' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 61, 67, 18, 0, 0, 0, 0],
            &[0, 15, 153, 220, 162, 165, 231, 108, 0, 0, 0],
            &[0, 24, 116, 7, 0, 0, 23, 231, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 156, 102, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 150, 103, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 211, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 86, 204, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 222, 48, 0, 0, 0],
            &[0, 0, 0, 0, 20, 212, 78, 0, 0, 0, 0],
            &[0, 0, 0, 18, 205, 94, 0, 0, 0, 0, 0],
            &[0, 0, 15, 202, 98, 0, 0, 0, 0, 0, 0],
            &[0, 13, 198, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 117, 252, 230, 230, 230, 230, 230, 230, 14, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '3' / 0x33
        '3' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 64, 70, 26, 0, 0, 0, 0],
            &[0, 23, 164, 220, 162, 161, 225, 145, 0, 0, 0],
            &[0, 26, 100, 3, 0, 0, 9, 211, 77, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 132, 128, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 157, 103, 0, 0],
            &[0, 0, 0, 0, 0, 0, 80, 212, 14, 0, 0],
            &[0, 0, 0, 154, 194, 224, 172, 19, 0, 0, 0],
            &[0, 0, 0, 31, 40, 59, 125, 208, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 93, 178, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 33, 224, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 71, 201, 0, 0],
            &[0, 52, 29, 0, 0, 0, 33, 211, 95, 0, 0],
            &[0, 77, 209, 221, 192, 208, 220, 104, 0, 0, 0],
            &[0, 0, 0, 11, 36, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '4' / 0x34
        '4' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 41, 21, 0, 0, 0],
            &[0, 0, 0, 0, 0, 49, 248, 105, 0, 0, 0],
            &[0, 0, 0, 0, 3, 198, 163, 105, 0, 0, 0],
            &[0, 0, 0, 0, 114, 135, 133, 105, 0, 0, 0],
            &[0, 0, 0, 32, 209, 9, 137, 105, 0, 0, 0],
            &[0, 0, 0, 183, 68, 0, 138, 105, 0, 0, 0],
            &[0, 0, 90, 161, 0, 0, 138, 105, 0, 0, 0],
            &[0, 19, 213, 20, 0, 0, 138, 105, 0, 0, 0],
            &[0, 159, 93, 0, 0, 0, 138, 105, 0, 0, 0],
            &[13, 248, 218, 217, 217, 217, 238, 233, 217, 57, 0],
            &[0, 0, 0, 0, 0, 0, 138, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 138, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 138, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '5' / 0x35
        '5' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 35, 35, 35, 35, 35, 2, 0, 0],
            &[0, 0, 172, 205, 190, 190, 190, 190, 14, 0, 0],
            &[0, 0, 194, 41, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 232, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 231, 53, 80, 72, 19, 0, 0, 0, 0],
            &[0, 9, 182, 152, 130, 151, 227, 126, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 16, 215, 76, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 107, 157, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 87, 171, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 138, 132, 0, 0],
            &[0, 45, 30, 0, 0, 0, 74, 234, 32, 0, 0],
            &[0, 61, 206, 216, 191, 220, 203, 55, 0, 0, 0],
            &[0, 0, 0, 11, 36, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '6' / 0x36
        '6' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 58, 78, 55, 0, 0, 0],
            &[0, 0, 0, 105, 221, 163, 139, 158, 0, 0, 0],
            &[0, 0, 101, 188, 16, 0, 0, 0, 0, 0, 0],
            &[0, 12, 223, 20, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 158, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 139, 100, 24, 119, 139, 101, 13, 0, 0, 0],
            &[0, 171, 127, 178, 89, 71, 126, 215, 36, 0, 0],
            &[0, 182, 185, 2, 0, 0, 0, 99, 170, 0, 0],
            &[0, 172, 93, 0, 0, 0, 0, 20, 231, 0, 0],
            &[0, 138, 102, 0, 0, 0, 0, 11, 235, 0, 0],
            &[0, 70, 180, 0, 0, 0, 0, 58, 194, 0, 0],
            &[0, 1, 201, 101, 0, 0, 16, 195, 87, 0, 0],
            &[0, 0, 25, 186, 214, 188, 221, 114, 0, 0, 0],
            &[0, 0, 0, 0, 20, 34, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '7' / 0x37
        '7' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 35, 35, 35, 35, 35, 35, 32, 0, 0],
            &[0, 76, 190, 190, 190, 190, 190, 200, 228, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 103, 145, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 207, 42, 0, 0],
            &[0, 0, 0, 0, 0, 0, 56, 195, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 161, 92, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 229, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 115, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 218, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 193, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 173, 91, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 232, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 127, 141, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '8' / 0x38
        '8' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 67, 66, 15, 0, 0, 0, 0],
            &[0, 0, 114, 221, 154, 156, 223, 103, 0, 0, 0],
            &[0, 46, 220, 13, 0, 0, 19, 229, 30, 0, 0],
            &[0, 94, 155, 0, 0, 0, 0, 181, 70, 0, 0],
            &[0, 56, 204, 4, 0, 0, 10, 226, 29, 0, 0],
            &[0, 0, 134, 181, 34, 41, 192, 105, 0, 0, 0],
            &[0, 0, 0, 121, 251, 249, 80, 0, 0, 0, 0],
            &[0, 2, 145, 179, 42, 80, 203, 122, 0, 0, 0],
            &[0, 107, 177, 3, 0, 0, 10, 196, 84, 0, 0],
            &[0, 188, 66, 0, 0, 0, 0, 93, 167, 0, 0],
            &[0, 185, 74, 0, 0, 0, 0, 98, 159, 0, 0],
            &[0, 96, 195, 14, 0, 0, 26, 215, 66, 0, 0],
            &[0, 0, 119, 219, 184, 191, 215, 89, 0, 0, 0],
            &[0, 0, 0, 4, 33, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '9' / 0x39
        '9' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 69, 67, 10, 0, 0, 0, 0],
            &[0, 0, 115, 218, 154, 154, 220, 74, 0, 0, 0],
            &[0, 65, 208, 13, 0, 0, 38, 223, 27, 0, 0],
            &[0, 160, 92, 0, 0, 0, 0, 118, 129, 0, 0],
            &[0, 191, 54, 0, 0, 0, 0, 48, 191, 0, 0],
            &[0, 178, 73, 0, 0, 0, 0, 50, 222, 0, 0],
            &[0, 105, 170, 0, 0, 0, 3, 161, 229, 0, 0],
            &[0, 3, 174, 187, 114, 120, 186, 93, 216, 0, 0],
            &[0, 0, 0, 54, 97, 73, 15, 65, 182, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 131, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 14, 225, 30, 0, 0],
            &[0, 0, 0, 0, 0, 24, 189, 122, 0, 0, 0],
            &[0, 0, 185, 183, 198, 221, 102, 0, 0, 0, 0],
            &[0, 0, 13, 38, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ':' / 0x3a
        ':' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 200, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 53, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 58, 86, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 150, 199, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ';' / 0x3b
        ';' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 69, 253, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 86, 229, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 79, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 224, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '<' / 0x3c
        '<' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 120, 174, 0, 0],
            &[0, 0, 0, 0, 8, 114, 202, 102, 5, 0, 0],
            &[0, 0, 6, 107, 199, 103, 6, 0, 0, 0, 0],
            &[0, 79, 199, 105, 6, 0, 0, 0, 0, 0, 0],
            &[0, 66, 195, 147, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 190, 159, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 183, 171, 50, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 52, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '=' / 0x3d
        '=' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 27, 27, 27, 27, 27, 27, 27, 0, 0],
            &[0, 160, 198, 198, 198, 198, 198, 198, 198, 1, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 161, 198, 198, 198, 198, 198, 198, 198, 1, 0],
            &[0, 25, 31, 31, 31, 31, 31, 31, 31, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '>' / 0x3e
        '>' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 146, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 76, 195, 140, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 193, 133, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 79, 194, 115, 0, 0],
            &[0, 0, 0, 0, 0, 15, 123, 203, 97, 0, 0],
            &[0, 0, 0, 22, 135, 202, 96, 5, 0, 0, 0],
            &[0, 30, 147, 199, 86, 2, 0, 0, 0, 0, 0],
            &[0, 110, 77, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '?' / 0x3f
        '?' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 66, 64, 14, 0, 0, 0, 0],
            &[0, 4, 207, 199, 157, 162, 228, 100, 0, 0, 0],
            &[0, 0, 19, 0, 0, 0, 25, 231, 27, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 176, 71, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 202, 42, 0, 0],
            &[0, 0, 0, 0, 0, 0, 85, 194, 1, 0, 0],
            &[0, 0, 0, 0, 0, 110, 199, 24, 0, 0, 0],
            &[0, 0, 0, 0, 110, 178, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 222, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 180, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 120, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '@' / 0x40
        '@' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 26, 2, 0, 0, 0, 0],
            &[0, 0, 37, 180, 200, 172, 201, 143, 7, 0, 0],
            &[0, 39, 210, 53, 0, 0, 0, 97, 172, 0, 0],
            &[0, 191, 48, 0, 0, 0, 0, 0, 150, 71, 0],
            &[49, 170, 0, 10, 138, 190, 167, 42, 46, 158, 0],
            &[121, 94, 0, 167, 107, 1, 150, 80, 1, 201, 0],
            &[165, 46, 25, 214, 1, 0, 153, 67, 0, 201, 0],
            &[183, 27, 69, 169, 0, 0, 170, 53, 0, 203, 0],
            &[180, 28, 67, 159, 0, 0, 202, 47, 2, 198, 0],
            &[158, 52, 25, 207, 1, 27, 234, 78, 58, 145, 0],
            &[103, 112, 0, 135, 195, 192, 51, 207, 205, 28, 0],
            &[26, 204, 3, 0, 13, 3, 0, 8, 3, 0, 0],
            &[0, 129, 142, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 137, 194, 105, 77, 99, 158, 88, 0, 0],
            &[0, 0, 0, 30, 97, 123, 101, 45, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'A' / 0x41
        'A' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 224, 188, 50, 0, 0, 0, 0],
            &[0, 0, 0, 88, 156, 101, 140, 0, 0, 0, 0],
            &[0, 0, 0, 178, 69, 19, 222, 1, 0, 0, 0],
            &[0, 0, 17, 227, 3, 0, 182, 62, 0, 0, 0],
            &[0, 0, 100, 149, 0, 0, 95, 151, 0, 0, 0],
            &[0, 0, 193, 79, 16, 16, 32, 235, 4, 0, 0],
            &[0, 25, 255, 215, 213, 213, 213, 255, 74, 0, 0],
            &[0, 112, 142, 0, 0, 0, 0, 90, 163, 0, 0],
            &[0, 201, 55, 0, 0, 0, 0, 12, 233, 9, 0],
            &[35, 222, 0, 0, 0, 0, 0, 0, 171, 86, 0],
            &[124, 135, 0, 0, 0, 0, 0, 0, 84, 175, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'B' / 0x42
        'B' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 35, 35, 34, 18, 0, 0, 0, 0, 0],
            &[0, 55, 237, 186, 187, 203, 228, 152, 10, 0, 0],
            &[0, 55, 188, 0, 0, 0, 9, 160, 150, 0, 0],
            &[0, 55, 188, 0, 0, 0, 0, 43, 215, 0, 0],
            &[0, 55, 188, 0, 0, 0, 0, 52, 204, 0, 0],
            &[0, 55, 188, 0, 0, 0, 22, 185, 92, 0, 0],
            &[0, 55, 242, 204, 204, 220, 234, 95, 0, 0, 0],
            &[0, 55, 193, 17, 17, 26, 68, 190, 119, 0, 0],
            &[0, 55, 188, 0, 0, 0, 0, 12, 239, 21, 0],
            &[0, 55, 188, 0, 0, 0, 0, 0, 211, 52, 0],
            &[0, 55, 188, 0, 0, 0, 0, 12, 243, 23, 0],
            &[0, 55, 188, 0, 0, 0, 33, 179, 160, 0, 0],
            &[0, 55, 248, 226, 226, 234, 213, 121, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'C' / 0x43
        'C' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 56, 72, 48, 5, 0, 0],
            &[0, 0, 0, 115, 228, 175, 154, 182, 227, 39, 0],
            &[0, 0, 130, 190, 21, 0, 0, 0, 13, 1, 0],
            &[0, 37, 231, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 78, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 207, 52, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 205, 54, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 183, 79, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 125, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 233, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 202, 38, 0, 0, 0, 37, 0, 0],
            &[0, 0, 0, 105, 224, 216, 200, 228, 200, 0, 0],
            &[0, 0, 0, 0, 0, 20, 35, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'D' / 0x44
        'D' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 35, 35, 23, 1, 0, 0, 0, 0, 0],
            &[0, 150, 214, 186, 202, 230, 162, 27, 0, 0, 0],
            &[0, 150, 101, 0, 0, 7, 110, 226, 29, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 116, 173, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 9, 240, 15, 0],
            &[0, 150, 101, 0, 0, 0, 0, 0, 196, 60, 0],
            &[0, 150, 101, 0, 0, 0, 0, 0, 174, 81, 0],
            &[0, 150, 101, 0, 0, 0, 0, 0, 176, 74, 0],
            &[0, 150, 101, 0, 0, 0, 0, 0, 207, 49, 0],
            &[0, 150, 101, 0, 0, 0, 0, 25, 231, 3, 0],
            &[0, 150, 101, 0, 0, 0, 0, 157, 132, 0, 0],
            &[0, 150, 101, 0, 0, 41, 164, 191, 8, 0, 0],
            &[0, 150, 238, 226, 224, 200, 108, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'E' / 0x45
        'E' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 25, 0, 0],
            &[0, 55, 241, 190, 190, 190, 190, 190, 137, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 245, 209, 209, 209, 209, 209, 79, 0, 0],
            &[0, 55, 201, 17, 17, 17, 17, 17, 6, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 250, 230, 230, 230, 230, 230, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'F' / 0x46
        'F' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 25, 0, 0],
            &[0, 55, 241, 190, 190, 190, 190, 190, 137, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 212, 65, 65, 65, 65, 65, 26, 0, 0],
            &[0, 55, 234, 160, 160, 160, 160, 160, 63, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'G' / 0x47
        'G' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 70, 64, 22, 0, 0],
            &[0, 0, 0, 24, 184, 214, 160, 166, 219, 107, 0],
            &[0, 0, 8, 215, 112, 1, 0, 0, 2, 13, 0],
            &[0, 0, 117, 166, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 59, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 249, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 233, 0, 0, 0, 31, 48, 48, 38, 0],
            &[0, 30, 229, 0, 0, 0, 117, 182, 194, 206, 0],
            &[0, 11, 248, 2, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 218, 48, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 135, 152, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 16, 223, 112, 4, 0, 0, 53, 206, 0],
            &[0, 0, 0, 28, 174, 230, 196, 209, 226, 123, 0],
            &[0, 0, 0, 0, 0, 8, 35, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'H' / 0x48
        'H' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 6, 28, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 227, 209, 209, 209, 209, 217, 206, 0, 0],
            &[0, 150, 114, 21, 21, 21, 21, 63, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'I' / 0x49
        'I' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 24, 35, 35, 35, 35, 31, 0, 0, 0],
            &[0, 0, 104, 161, 205, 222, 164, 135, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 129, 196, 227, 237, 199, 167, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'J' / 0x4a
        'J' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 16, 18, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 120, 133, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 150, 110, 0, 0],
            &[0, 0, 11, 27, 0, 0, 28, 233, 40, 0, 0],
            &[0, 0, 21, 213, 209, 187, 233, 110, 0, 0, 0],
            &[0, 0, 0, 0, 26, 44, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'K' / 0x4b
        'K' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 3, 35, 3, 0],
            &[0, 150, 101, 0, 0, 0, 0, 152, 154, 0, 0],
            &[0, 150, 101, 0, 0, 0, 119, 187, 4, 0, 0],
            &[0, 150, 101, 0, 0, 87, 212, 15, 0, 0, 0],
            &[0, 150, 101, 0, 60, 226, 33, 0, 0, 0, 0],
            &[0, 150, 101, 38, 228, 58, 0, 0, 0, 0, 0],
            &[0, 150, 115, 211, 220, 11, 0, 0, 0, 0, 0],
            &[0, 150, 234, 87, 136, 147, 0, 0, 0, 0, 0],
            &[0, 150, 131, 0, 7, 212, 64, 0, 0, 0, 0],
            &[0, 150, 101, 0, 0, 53, 218, 12, 0, 0, 0],
            &[0, 150, 101, 0, 0, 0, 135, 150, 0, 0, 0],
            &[0, 150, 101, 0, 0, 0, 7, 211, 67, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 52, 220, 13, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'L' / 0x4c
        'L' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 240, 234, 234, 234, 234, 169, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'M' / 0x4d
        'M' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[2, 35, 13, 0, 0, 0, 0, 6, 35, 9, 0],
            &[16, 255, 146, 0, 0, 0, 0, 99, 255, 67, 0],
            &[16, 214, 222, 4, 0, 0, 0, 191, 203, 67, 0],
            &[16, 219, 163, 73, 0, 0, 28, 210, 166, 67, 0],
            &[16, 225, 70, 164, 0, 0, 117, 119, 171, 67, 0],
            &[16, 226, 3, 219, 11, 0, 206, 29, 171, 67, 0],
            &[16, 226, 0, 141, 90, 44, 191, 0, 171, 67, 0],
            &[16, 226, 0, 48, 182, 135, 99, 0, 171, 67, 0],
            &[16, 226, 0, 0, 207, 223, 15, 0, 171, 67, 0],
            &[16, 226, 0, 0, 98, 137, 0, 0, 171, 67, 0],
            &[16, 226, 0, 0, 0, 0, 0, 0, 171, 67, 0],
            &[16, 226, 0, 0, 0, 0, 0, 0, 171, 67, 0],
            &[16, 226, 0, 0, 0, 0, 0, 0, 171, 67, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'N' / 0x4e
        'N' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 24, 0, 0, 0, 0, 5, 28, 0, 0],
            &[0, 150, 235, 11, 0, 0, 0, 41, 206, 0, 0],
            &[0, 150, 193, 116, 0, 0, 0, 41, 206, 0, 0],
            &[0, 150, 98, 216, 9, 0, 0, 41, 206, 0, 0],
            &[0, 150, 90, 129, 111, 0, 0, 41, 206, 0, 0],
            &[0, 150, 92, 18, 216, 7, 0, 41, 206, 0, 0],
            &[0, 150, 92, 0, 135, 106, 0, 41, 206, 0, 0],
            &[0, 150, 92, 0, 21, 215, 6, 41, 206, 0, 0],
            &[0, 150, 92, 0, 0, 141, 101, 41, 206, 0, 0],
            &[0, 150, 92, 0, 0, 24, 213, 42, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 146, 127, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 28, 227, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 0, 152, 206, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'O' / 0x4f
        'O' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 59, 72, 23, 0, 0, 0, 0],
            &[0, 0, 75, 223, 169, 154, 220, 128, 0, 0, 0],
            &[0, 38, 227, 37, 0, 0, 10, 195, 91, 0, 0],
            &[0, 155, 112, 0, 0, 0, 0, 51, 212, 0, 0],
            &[0, 226, 29, 0, 0, 0, 0, 0, 229, 28, 0],
            &[14, 241, 0, 0, 0, 0, 0, 0, 188, 68, 0],
            &[31, 224, 0, 0, 0, 0, 0, 0, 171, 84, 0],
            &[31, 226, 0, 0, 0, 0, 0, 0, 175, 82, 0],
            &[11, 244, 1, 0, 0, 0, 0, 0, 194, 61, 0],
            &[0, 220, 40, 0, 0, 0, 0, 2, 237, 18, 0],
            &[0, 140, 132, 0, 0, 0, 0, 74, 193, 0, 0],
            &[0, 26, 229, 70, 0, 0, 34, 220, 64, 0, 0],
            &[0, 0, 51, 209, 213, 202, 222, 86, 0, 0, 0],
            &[0, 0, 0, 0, 25, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'P' / 0x50
        'P' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 35, 35, 35, 29, 5, 0, 0, 0, 0],
            &[0, 55, 240, 186, 186, 201, 231, 175, 28, 0, 0],
            &[0, 55, 197, 0, 0, 0, 6, 134, 205, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 1, 237, 36, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 210, 51, 0],
            &[0, 55, 197, 0, 0, 0, 0, 10, 237, 17, 0],
            &[0, 55, 197, 0, 0, 0, 22, 169, 151, 0, 0],
            &[0, 55, 245, 208, 208, 221, 217, 121, 5, 0, 0],
            &[0, 55, 201, 17, 17, 4, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Q' / 0x51
        'Q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 59, 72, 23, 0, 0, 0, 0],
            &[0, 0, 75, 223, 169, 154, 220, 128, 0, 0, 0],
            &[0, 38, 227, 37, 0, 0, 10, 195, 91, 0, 0],
            &[0, 155, 112, 0, 0, 0, 0, 51, 212, 0, 0],
            &[0, 226, 29, 0, 0, 0, 0, 0, 229, 28, 0],
            &[14, 241, 0, 0, 0, 0, 0, 0, 188, 68, 0],
            &[31, 224, 0, 0, 0, 0, 0, 0, 171, 84, 0],
            &[31, 226, 0, 0, 0, 0, 0, 0, 175, 82, 0],
            &[11, 244, 1, 0, 0, 0, 0, 0, 194, 61, 0],
            &[0, 220, 40, 0, 0, 0, 0, 2, 237, 18, 0],
            &[0, 140, 132, 0, 0, 0, 0, 74, 193, 0, 0],
            &[0, 26, 229, 70, 0, 0, 34, 220, 64, 0, 0],
            &[0, 0, 51, 209, 213, 204, 255, 94, 0, 0, 0],
            &[0, 0, 0, 0, 25, 31, 160, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 230, 18, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 135, 138, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'R' / 0x52
        'R' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 35, 35, 33, 19, 0, 0, 0, 0, 0],
            &[0, 55, 240, 186, 189, 211, 225, 122, 1, 0, 0],
            &[0, 55, 197, 0, 0, 0, 26, 202, 115, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 72, 202, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 46, 215, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 106, 167, 0, 0],
            &[0, 55, 202, 22, 22, 37, 108, 217, 36, 0, 0],
            &[0, 55, 244, 204, 204, 229, 168, 11, 0, 0, 0],
            &[0, 55, 197, 0, 0, 38, 223, 13, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 137, 140, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 12, 224, 42, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 90, 189, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 194, 87, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'S' / 0x53
        'S' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 49, 73, 55, 16, 0, 0, 0],
            &[0, 0, 65, 222, 192, 156, 172, 224, 109, 0, 0],
            &[0, 13, 234, 54, 0, 0, 0, 1, 11, 0, 0],
            &[0, 64, 193, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 210, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 204, 153, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 146, 231, 148, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 117, 222, 169, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 160, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 24, 234, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 40, 225, 0, 0],
            &[0, 34, 16, 0, 0, 0, 16, 181, 134, 0, 0],
            &[0, 85, 219, 215, 187, 194, 227, 133, 3, 0, 0],
            &[0, 0, 0, 12, 35, 31, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'T' / 0x54
        'T' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[6, 35, 35, 35, 35, 35, 35, 35, 35, 14, 0],
            &[35, 190, 190, 190, 216, 229, 190, 190, 190, 76, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'U' / 0x55
        'U' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 6, 28, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 149, 102, 0, 0, 0, 0, 47, 205, 0, 0],
            &[0, 135, 117, 0, 0, 0, 0, 62, 190, 0, 0],
            &[0, 93, 169, 0, 0, 0, 0, 117, 145, 0, 0],
            &[0, 16, 232, 61, 0, 0, 35, 227, 47, 0, 0],
            &[0, 0, 60, 218, 206, 200, 224, 88, 0, 0, 0],
            &[0, 0, 0, 1, 27, 30, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'V' / 0x56
        'V' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[14, 21, 0, 0, 0, 0, 0, 0, 14, 22, 0],
            &[57, 206, 0, 0, 0, 0, 0, 0, 154, 113, 0],
            &[1, 226, 33, 0, 0, 0, 0, 2, 232, 30, 0],
            &[0, 143, 116, 0, 0, 0, 0, 61, 200, 0, 0],
            &[0, 57, 199, 0, 0, 0, 0, 143, 116, 0, 0],
            &[0, 1, 225, 28, 0, 0, 0, 224, 32, 0, 0],
            &[0, 0, 143, 110, 0, 0, 51, 202, 0, 0, 0],
            &[0, 0, 57, 193, 0, 0, 133, 118, 0, 0, 0],
            &[0, 0, 1, 224, 23, 0, 214, 34, 0, 0, 0],
            &[0, 0, 0, 143, 103, 40, 204, 0, 0, 0, 0],
            &[0, 0, 0, 57, 180, 117, 120, 0, 0, 0, 0],
            &[0, 0, 0, 1, 217, 194, 36, 0, 0, 0, 0],
            &[0, 0, 0, 0, 143, 207, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'W' / 0x57
        'W' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[15, 17, 0, 0, 0, 0, 0, 0, 10, 22, 0],
            &[90, 148, 0, 0, 0, 0, 0, 0, 93, 146, 0],
            &[55, 182, 0, 0, 0, 0, 0, 0, 127, 112, 0],
            &[20, 216, 0, 0, 4, 6, 0, 0, 160, 78, 0],
            &[0, 234, 1, 0, 128, 183, 0, 0, 195, 44, 0],
            &[0, 206, 28, 0, 180, 185, 2, 0, 229, 10, 0],
            &[0, 172, 61, 3, 190, 144, 46, 9, 230, 0, 0],
            &[0, 137, 95, 49, 147, 87, 105, 42, 197, 0, 0],
            &[0, 102, 129, 108, 89, 28, 164, 72, 163, 0, 0],
            &[0, 67, 158, 165, 31, 0, 195, 96, 129, 0, 0],
            &[0, 33, 183, 194, 0, 0, 167, 145, 95, 0, 0],
            &[0, 3, 217, 171, 0, 0, 108, 217, 61, 0, 0],
            &[0, 0, 219, 113, 0, 0, 50, 255, 27, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'X' / 0x58
        'X' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 31, 7, 0, 0, 0, 0, 0, 31, 5, 0],
            &[0, 137, 136, 0, 0, 0, 0, 62, 202, 2, 0],
            &[0, 11, 219, 36, 0, 0, 4, 207, 51, 0, 0],
            &[0, 0, 80, 180, 0, 0, 112, 146, 0, 0, 0],
            &[0, 0, 0, 179, 74, 24, 215, 14, 0, 0, 0],
            &[0, 0, 0, 32, 208, 167, 88, 0, 0, 0, 0],
            &[0, 0, 0, 0, 136, 201, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 207, 214, 35, 0, 0, 0, 0],
            &[0, 0, 0, 122, 134, 67, 183, 0, 0, 0, 0],
            &[0, 0, 34, 215, 10, 0, 169, 85, 0, 0, 0],
            &[0, 0, 183, 80, 0, 0, 28, 217, 13, 0, 0],
            &[0, 86, 181, 0, 0, 0, 0, 119, 144, 0, 0],
            &[14, 222, 35, 0, 0, 0, 0, 5, 212, 49, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Y' / 0x59
        'Y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[8, 30, 0, 0, 0, 0, 0, 0, 21, 15, 0],
            &[9, 226, 38, 0, 0, 0, 0, 5, 221, 42, 0],
            &[0, 105, 163, 0, 0, 0, 0, 101, 161, 0, 0],
            &[0, 5, 218, 38, 0, 0, 4, 217, 33, 0, 0],
            &[0, 0, 92, 163, 0, 0, 98, 150, 0, 0, 0],
            &[0, 0, 2, 209, 38, 4, 211, 26, 0, 0, 0],
            &[0, 0, 0, 80, 163, 95, 139, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 214, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 98, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Z' / 0x5a
        'Z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 35, 35, 35, 35, 35, 14, 0, 0],
            &[0, 0, 150, 186, 186, 186, 186, 238, 91, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 220, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 155, 103, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 216, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 169, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 209, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 183, 78, 0, 0, 0, 0, 0],
            &[0, 0, 0, 62, 199, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 197, 65, 0, 0, 0, 0, 0, 0],
            &[0, 0, 76, 186, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 208, 53, 0, 0, 0, 0, 0, 0, 0],
            &[0, 51, 253, 230, 230, 230, 230, 230, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '[' / 0x5b
        '[' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 33, 35, 35, 24, 0, 0, 0, 0],
            &[0, 0, 0, 239, 174, 173, 118, 0, 0, 0, 0],
            &[0, 0, 0, 239, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 239, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 239, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 239, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 239, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 239, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 239, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 239, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 239, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 239, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 239, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 239, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 239, 40, 37, 25, 0, 0, 0, 0],
            &[0, 0, 0, 164, 175, 175, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '\' / 0x5c
        '\\' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 114, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 222, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 175, 72, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 169, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 225, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 140, 107, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 204, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 201, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 104, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 226, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 166, 81, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 69, 178, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ']' / 0x5d
        ']' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 57, 57, 57, 8, 0, 0, 0],
            &[0, 0, 0, 72, 156, 156, 235, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 39, 0, 0, 0],
            &[0, 0, 0, 25, 55, 55, 215, 39, 0, 0, 0],
            &[0, 0, 0, 71, 153, 153, 153, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '^' / 0x5e
        '^' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 150, 191, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 187, 151, 69, 0, 0, 0, 0],
            &[0, 0, 0, 152, 67, 29, 192, 0, 0, 0, 0],
            &[0, 0, 31, 190, 0, 0, 150, 74, 0, 0, 0],
            &[0, 0, 154, 69, 0, 0, 28, 197, 0, 0, 0],
            &[0, 33, 193, 0, 0, 0, 0, 149, 80, 0, 0],
            &[0, 157, 71, 0, 0, 0, 0, 28, 202, 1, 0],
            &[0, 81, 0, 0, 0, 0, 0, 0, 68, 15, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '_' / 0x5f
        '_' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 33],
            &[62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 12],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '`' / 0x60
        '`' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 75, 218, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 101, 166, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 92, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'a' / 0x61
        'a' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 33, 42, 9, 0, 0, 0, 0],
            &[0, 0, 133, 216, 177, 172, 222, 120, 0, 0, 0],
            &[0, 0, 34, 2, 0, 0, 10, 211, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 128, 110, 0, 0],
            &[0, 0, 0, 1, 36, 61, 72, 153, 127, 0, 0],
            &[0, 0, 114, 212, 166, 133, 119, 172, 127, 0, 0],
            &[0, 91, 187, 9, 0, 0, 0, 113, 127, 0, 0],
            &[0, 153, 90, 0, 0, 0, 0, 165, 127, 0, 0],
            &[0, 119, 151, 0, 0, 0, 88, 214, 127, 0, 0],
            &[0, 10, 181, 210, 176, 201, 125, 87, 127, 0, 0],
            &[0, 0, 0, 17, 31, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'b' / 0x62
        'b' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 125, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 136, 0, 16, 47, 21, 0, 0, 0, 0],
            &[0, 107, 134, 134, 201, 172, 220, 159, 5, 0, 0],
            &[0, 107, 216, 119, 0, 0, 6, 178, 127, 0, 0],
            &[0, 107, 222, 2, 0, 0, 0, 44, 229, 0, 0],
            &[0, 107, 164, 0, 0, 0, 0, 0, 247, 21, 0],
            &[0, 107, 143, 0, 0, 0, 0, 0, 230, 33, 0],
            &[0, 107, 162, 0, 0, 0, 0, 2, 250, 15, 0],
            &[0, 107, 216, 0, 0, 0, 0, 58, 217, 0, 0],
            &[0, 107, 227, 106, 0, 0, 14, 195, 107, 0, 0],
            &[0, 107, 109, 147, 205, 184, 227, 133, 1, 0, 0],
            &[0, 0, 0, 0, 13, 36, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'c' / 0x63
        'c' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 38, 43, 12, 0, 0, 0],
            &[0, 0, 6, 136, 227, 188, 181, 219, 90, 0, 0],
            &[0, 0, 159, 169, 12, 0, 0, 0, 7, 0, 0],
            &[0, 36, 229, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 101, 156, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 133, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 152, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 221, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 181, 150, 7, 0, 0, 1, 15, 0, 0],
            &[0, 0, 12, 154, 228, 190, 193, 218, 90, 0, 0],
            &[0, 0, 0, 0, 7, 33, 28, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'd' / 0x64
        'd' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 82, 149, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 0, 0, 8, 42, 25, 0, 89, 162, 0, 0],
            &[0, 0, 102, 225, 182, 178, 183, 103, 162, 0, 0],
            &[0, 62, 226, 34, 0, 0, 46, 216, 162, 0, 0],
            &[0, 166, 109, 0, 0, 0, 0, 160, 162, 0, 0],
            &[0, 217, 49, 0, 0, 0, 0, 107, 162, 0, 0],
            &[0, 233, 26, 0, 0, 0, 0, 92, 162, 0, 0],
            &[0, 221, 43, 0, 0, 0, 0, 112, 162, 0, 0],
            &[0, 173, 97, 0, 0, 0, 0, 165, 162, 0, 0],
            &[0, 69, 218, 28, 0, 0, 62, 208, 162, 0, 0],
            &[0, 0, 106, 227, 188, 198, 166, 66, 162, 0, 0],
            &[0, 0, 0, 6, 35, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'e' / 0x65
        'e' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 46, 15, 0, 0, 0, 0],
            &[0, 0, 41, 200, 198, 167, 212, 141, 2, 0, 0],
            &[0, 17, 227, 71, 0, 0, 2, 169, 113, 0, 0],
            &[0, 122, 141, 0, 0, 0, 0, 36, 216, 0, 0],
            &[0, 188, 111, 56, 56, 56, 56, 56, 250, 4, 0],
            &[0, 209, 171, 152, 152, 152, 152, 152, 152, 9, 0],
            &[0, 192, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 137, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 225, 76, 0, 0, 0, 4, 43, 0, 0],
            &[0, 0, 35, 182, 215, 181, 195, 216, 123, 0, 0],
            &[0, 0, 0, 0, 10, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'f' / 0x66
        'f' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 132, 219, 214, 212, 105, 0],
            &[0, 0, 0, 0, 95, 198, 15, 0, 0, 2, 0],
            &[0, 0, 0, 0, 153, 107, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 164, 98, 11, 11, 11, 0, 0],
            &[0, 119, 195, 205, 237, 223, 206, 206, 206, 5, 0],
            &[0, 0, 0, 0, 161, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'g' / 0x67
        'g' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 46, 34, 0, 1, 7, 0, 0],
            &[0, 0, 98, 227, 182, 175, 189, 81, 162, 0, 0],
            &[0, 57, 222, 31, 0, 0, 31, 218, 162, 0, 0],
            &[0, 164, 106, 0, 0, 0, 0, 133, 162, 0, 0],
            &[0, 217, 47, 0, 0, 0, 0, 85, 162, 0, 0],
            &[0, 233, 25, 0, 0, 0, 0, 73, 162, 0, 0],
            &[0, 221, 45, 0, 0, 0, 0, 91, 162, 0, 0],
            &[0, 172, 103, 0, 0, 0, 0, 144, 162, 0, 0],
            &[0, 65, 222, 32, 0, 0, 53, 209, 162, 0, 0],
            &[0, 0, 101, 227, 191, 199, 178, 93, 162, 0, 0],
            &[0, 0, 0, 5, 36, 20, 0, 83, 159, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 123, 130, 0, 0],
            &[0, 36, 18, 0, 0, 0, 37, 227, 44, 0, 0],
            &[0, 0, 0, 13, 39, 28, 1, 0, 0, 0, 0],
        ]),
        // letter: 'h' / 0x68
        'h' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 133, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 145, 0, 8, 43, 19, 0, 0, 0, 0],
            &[0, 107, 143, 121, 194, 168, 219, 163, 3, 0, 0],
            &[0, 107, 211, 102, 0, 0, 5, 190, 105, 0, 0],
            &[0, 107, 209, 0, 0, 0, 0, 92, 164, 0, 0],
            &[0, 107, 157, 0, 0, 0, 0, 68, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'i' / 0x69
        'i' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 223, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 11, 11, 11, 10, 0, 0, 0, 0, 0],
            &[0, 30, 175, 188, 205, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 197, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'j' / 0x6a
        'j' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 132, 63, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 207, 108, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 11, 11, 11, 4, 0, 0, 0],
            &[0, 0, 0, 114, 181, 194, 233, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 147, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 147, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 147, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 147, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 147, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 147, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 147, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 147, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 147, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 170, 91, 0, 0, 0],
            &[0, 0, 0, 0, 0, 30, 232, 38, 0, 0, 0],
            &[0, 0, 1, 26, 37, 6, 0, 0, 0, 0, 0],
        ]),
        // letter: 'k' / 0x6b
        'k' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 214, 18, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 232, 19, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 232, 19, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 232, 19, 0, 0, 0, 0, 11, 1, 0, 0],
            &[0, 232, 19, 0, 0, 0, 135, 171, 3, 0, 0],
            &[0, 232, 19, 0, 0, 123, 182, 5, 0, 0, 0],
            &[0, 232, 19, 0, 111, 192, 9, 0, 0, 0, 0],
            &[0, 232, 15, 99, 217, 13, 0, 0, 0, 0, 0],
            &[0, 232, 93, 186, 214, 54, 0, 0, 0, 0, 0],
            &[0, 232, 174, 8, 53, 216, 14, 0, 0, 0, 0],
            &[0, 232, 22, 0, 0, 120, 167, 0, 0, 0, 0],
            &[0, 232, 19, 0, 0, 2, 189, 99, 0, 0, 0],
            &[0, 232, 19, 0, 0, 0, 27, 225, 41, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'l' / 0x6c
        'l' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 187, 199, 213, 211, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 197, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'm' / 0x6d
        'm' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 1, 37, 15, 0, 12, 36, 0, 0, 0],
            &[0, 192, 145, 190, 229, 36, 212, 207, 179, 0, 0],
            &[0, 241, 105, 0, 145, 228, 32, 3, 225, 20, 0],
            &[0, 250, 18, 0, 94, 191, 0, 0, 182, 50, 0],
            &[0, 240, 0, 0, 79, 161, 0, 0, 171, 59, 0],
            &[0, 230, 0, 0, 78, 151, 0, 0, 171, 59, 0],
            &[0, 230, 0, 0, 78, 151, 0, 0, 171, 59, 0],
            &[0, 230, 0, 0, 78, 151, 0, 0, 171, 59, 0],
            &[0, 230, 0, 0, 78, 151, 0, 0, 171, 59, 0],
            &[0, 230, 0, 0, 78, 151, 0, 0, 171, 59, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'n' / 0x6e
        'n' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 4, 0, 9, 43, 21, 0, 0, 0, 0],
            &[0, 107, 109, 131, 197, 168, 216, 170, 5, 0, 0],
            &[0, 107, 215, 100, 0, 0, 3, 181, 111, 0, 0],
            &[0, 107, 209, 0, 0, 0, 0, 89, 167, 0, 0],
            &[0, 107, 163, 0, 0, 0, 0, 68, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'o' / 0x6f
        'o' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 35, 42, 8, 0, 0, 0, 0],
            &[0, 0, 71, 213, 189, 171, 217, 115, 0, 0, 0],
            &[0, 45, 225, 43, 0, 0, 10, 192, 95, 0, 0],
            &[0, 158, 110, 0, 0, 0, 0, 48, 213, 0, 0],
            &[0, 215, 45, 0, 0, 0, 0, 0, 240, 15, 0],
            &[0, 232, 20, 0, 0, 0, 0, 0, 224, 32, 0],
            &[0, 215, 39, 0, 0, 0, 0, 1, 241, 15, 0],
            &[0, 156, 108, 0, 0, 0, 0, 57, 209, 0, 0],
            &[0, 41, 224, 40, 0, 0, 17, 203, 87, 0, 0],
            &[0, 0, 65, 211, 190, 183, 221, 104, 0, 0, 0],
            &[0, 0, 0, 0, 27, 34, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'p' / 0x70
        'p' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 4, 0, 16, 48, 23, 0, 0, 0, 0],
            &[0, 107, 109, 122, 200, 170, 217, 166, 7, 0, 0],
            &[0, 107, 204, 112, 0, 0, 3, 166, 133, 0, 0],
            &[0, 107, 216, 1, 0, 0, 0, 36, 232, 0, 0],
            &[0, 107, 164, 0, 0, 0, 0, 0, 245, 22, 0],
            &[0, 107, 145, 0, 0, 0, 0, 0, 230, 32, 0],
            &[0, 107, 167, 0, 0, 0, 0, 1, 250, 14, 0],
            &[0, 107, 222, 2, 0, 0, 0, 53, 216, 0, 0],
            &[0, 107, 216, 117, 0, 0, 11, 189, 108, 0, 0],
            &[0, 107, 138, 124, 205, 183, 225, 135, 1, 0, 0],
            &[0, 107, 140, 0, 4, 33, 10, 0, 0, 0, 0],
            &[0, 107, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 11, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'q' / 0x71
        'q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 46, 30, 0, 1, 7, 0, 0],
            &[0, 0, 96, 224, 179, 189, 181, 73, 162, 0, 0],
            &[0, 57, 220, 28, 0, 0, 53, 220, 162, 0, 0],
            &[0, 165, 105, 0, 0, 0, 0, 163, 162, 0, 0],
            &[0, 217, 47, 0, 0, 0, 0, 112, 162, 0, 0],
            &[0, 234, 25, 0, 0, 0, 0, 93, 162, 0, 0],
            &[0, 221, 44, 0, 0, 0, 0, 114, 162, 0, 0],
            &[0, 172, 100, 0, 0, 0, 0, 170, 162, 0, 0],
            &[0, 67, 219, 27, 0, 0, 70, 209, 162, 0, 0],
            &[0, 0, 105, 227, 188, 201, 160, 94, 162, 0, 0],
            &[0, 0, 0, 6, 36, 16, 0, 89, 162, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 7, 13, 0, 0],
        ]),
        // letter: 'r' / 0x72
        'r' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 11, 11, 11, 0, 0, 33, 37, 3, 0],
            &[0, 56, 179, 194, 243, 44, 191, 194, 192, 133, 0],
            &[0, 0, 0, 0, 191, 187, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 61, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 60, 0, 0, 0, 0, 0],
            &[0, 113, 193, 207, 245, 222, 201, 188, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 's' / 0x73
        's' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 48, 26, 0, 0, 0, 0],
            &[0, 0, 50, 214, 188, 169, 194, 206, 4, 0, 0],
            &[0, 0, 190, 80, 0, 0, 0, 7, 0, 0, 0],
            &[0, 0, 205, 61, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 220, 102, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 137, 221, 150, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 109, 225, 16, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 200, 67, 0, 0],
            &[0, 7, 41, 0, 0, 0, 22, 232, 32, 0, 0],
            &[0, 11, 195, 217, 186, 189, 221, 98, 0, 0, 0],
            &[0, 0, 0, 8, 33, 31, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 't' / 0x74
        't' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 221, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 33, 226, 11, 11, 11, 5, 0, 0],
            &[0, 153, 202, 211, 249, 206, 206, 206, 103, 0, 0],
            &[0, 0, 0, 27, 224, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 224, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 224, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 224, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 225, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 245, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 222, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 223, 183, 182, 114, 0, 0],
            &[0, 0, 0, 0, 0, 3, 33, 30, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'u' / 0x75
        'u' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 5, 0, 0, 0, 0, 3, 7, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 125, 0, 0, 0, 0, 102, 162, 0, 0],
            &[0, 105, 149, 0, 0, 0, 0, 154, 162, 0, 0],
            &[0, 45, 230, 22, 0, 0, 56, 204, 162, 0, 0],
            &[0, 0, 111, 229, 183, 192, 157, 61, 162, 0, 0],
            &[0, 0, 0, 4, 32, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'v' / 0x76
        'v' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 1, 0, 0, 0, 0, 0, 10, 0, 0],
            &[0, 176, 83, 0, 0, 0, 0, 36, 224, 2, 0],
            &[0, 79, 177, 0, 0, 0, 0, 129, 129, 0, 0],
            &[0, 5, 228, 20, 0, 0, 0, 219, 33, 0, 0],
            &[0, 0, 142, 109, 0, 0, 59, 190, 0, 0, 0],
            &[0, 0, 45, 203, 0, 0, 152, 92, 0, 0, 0],
            &[0, 0, 0, 204, 41, 7, 223, 9, 0, 0, 0],
            &[0, 0, 0, 108, 135, 83, 153, 0, 0, 0, 0],
            &[0, 0, 0, 18, 218, 176, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'w' / 0x77
        'w' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[7, 3, 0, 0, 0, 0, 0, 0, 0, 9, 0],
            &[136, 99, 0, 0, 0, 0, 0, 0, 38, 192, 0],
            &[83, 153, 0, 0, 41, 71, 0, 0, 89, 142, 0],
            &[31, 206, 0, 0, 164, 223, 3, 0, 140, 92, 0],
            &[0, 229, 9, 3, 211, 157, 61, 0, 191, 42, 0],
            &[0, 182, 57, 63, 159, 86, 137, 1, 231, 3, 0],
            &[0, 129, 109, 140, 85, 16, 209, 36, 198, 0, 0],
            &[0, 77, 151, 206, 14, 0, 192, 107, 148, 0, 0],
            &[0, 24, 200, 188, 0, 0, 116, 199, 98, 0, 0],
            &[0, 0, 227, 114, 0, 0, 41, 255, 48, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'x' / 0x78
        'x' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 4, 0, 0, 0, 0, 2, 10, 0, 0],
            &[0, 79, 200, 4, 0, 0, 0, 151, 129, 0, 0],
            &[0, 0, 148, 132, 0, 0, 82, 193, 3, 0, 0],
            &[0, 0, 8, 206, 61, 29, 218, 27, 0, 0, 0],
            &[0, 0, 0, 40, 214, 196, 76, 0, 0, 0, 0],
            &[0, 0, 0, 0, 172, 217, 3, 0, 0, 0, 0],
            &[0, 0, 0, 79, 192, 147, 129, 0, 0, 0, 0],
            &[0, 0, 30, 219, 28, 7, 205, 65, 0, 0, 0],
            &[0, 4, 196, 81, 0, 0, 40, 219, 21, 0, 0],
            &[0, 134, 150, 0, 0, 0, 0, 99, 183, 1, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'y' / 0x79
        'y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 1, 0, 0, 0, 0, 0, 10, 0, 0],
            &[0, 160, 90, 0, 0, 0, 0, 32, 224, 2, 0],
            &[0, 58, 189, 0, 0, 0, 0, 122, 131, 0, 0],
            &[0, 0, 211, 34, 0, 0, 0, 213, 35, 0, 0],
            &[0, 0, 111, 132, 0, 0, 50, 194, 0, 0, 0],
            &[0, 0, 18, 220, 3, 0, 142, 97, 0, 0, 0],
            &[0, 0, 0, 164, 75, 3, 219, 12, 0, 0, 0],
            &[0, 0, 0, 62, 172, 68, 160, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 168, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 114, 222, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 210, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 89, 171, 0, 0, 0, 0, 0, 0],
            &[0, 19, 39, 4, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'z' / 0x7a
        'z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 11, 11, 11, 11, 11, 4, 0, 0],
            &[0, 0, 172, 193, 193, 193, 193, 246, 92, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 199, 5, 0, 0],
            &[0, 0, 0, 0, 0, 26, 220, 33, 0, 0, 0],
            &[0, 0, 0, 0, 2, 188, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 209, 10, 0, 0, 0, 0, 0],
            &[0, 0, 13, 213, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 107, 0, 0, 0, 0, 0, 0, 0],
            &[0, 53, 255, 222, 221, 221, 221, 221, 95, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '{' / 0x7b
        '{' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 19, 0, 0, 0],
            &[0, 0, 0, 0, 2, 150, 222, 121, 0, 0, 0],
            &[0, 0, 0, 0, 73, 205, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 143, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 109, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 109, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 126, 126, 0, 0, 0, 0, 0],
            &[0, 0, 18, 80, 214, 42, 0, 0, 0, 0, 0],
            &[0, 0, 71, 212, 141, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 171, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 113, 136, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 109, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 109, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 230, 89, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 43, 139, 109, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '|' / 0x7c
        '|' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 72, 123, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 10, 0, 0, 0, 0, 0],
        ]),
        // letter: '}' / 0x7d
        '}' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 11, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 82, 210, 181, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 152, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 176, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 218, 93, 25, 0, 0, 0],
            &[0, 0, 0, 0, 0, 98, 221, 112, 0, 0, 0],
            &[0, 0, 0, 0, 42, 218, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 151, 0, 0, 0, 0, 0],
            &[0, 0, 19, 78, 221, 75, 0, 0, 0, 0, 0],
            &[0, 0, 77, 151, 68, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '~' / 0x7e
        '~' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 150, 190, 113, 0, 0, 3, 167, 0, 0],
            &[0, 88, 113, 2, 124, 152, 4, 67, 134, 0, 0],
            &[0, 142, 38, 0, 0, 97, 199, 177, 18, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ' ' / 0xa0
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¡' / 0xa1
        '¡' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 151, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 60, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 108, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 140, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 97, 144, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 101, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 153, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 110, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 119, 166, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 123, 170, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 25, 0, 0, 0, 0, 0],
        ]),
        // letter: '¢' / 0xa2
        '¢' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 58, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 13, 0, 0, 0, 0],
            &[0, 0, 0, 57, 169, 241, 216, 181, 44, 0, 0],
            &[0, 0, 85, 219, 69, 5, 3, 46, 18, 0, 0],
            &[0, 8, 231, 40, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 99, 153, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 93, 159, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 53, 211, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 212, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 220, 161, 107, 113, 157, 61, 0, 0],
            &[0, 0, 0, 7, 79, 219, 115, 68, 4, 0, 0],
            &[0, 0, 0, 0, 0, 191, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '£' / 0xa3
        '£' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 65, 36, 0, 0, 0],
            &[0, 0, 0, 0, 160, 202, 152, 187, 166, 0, 0],
            &[0, 0, 0, 81, 185, 1, 0, 0, 9, 0, 0],
            &[0, 0, 0, 137, 109, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 157, 90, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 157, 90, 0, 0, 0, 0, 0, 0],
            &[0, 24, 87, 191, 146, 87, 87, 32, 0, 0, 0],
            &[0, 30, 108, 199, 160, 108, 108, 41, 0, 0, 0],
            &[0, 0, 0, 157, 90, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 161, 87, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 197, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 87, 183, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 255, 235, 230, 230, 230, 230, 230, 10, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¤' / 0xa4
        '¤' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 69, 20, 98, 107, 38, 33, 160, 0, 0],
            &[0, 23, 211, 195, 101, 88, 175, 224, 51, 0, 0],
            &[0, 8, 198, 13, 0, 0, 0, 176, 41, 0, 0],
            &[0, 54, 142, 0, 0, 0, 0, 86, 113, 0, 0],
            &[0, 47, 150, 0, 0, 0, 0, 95, 107, 0, 0],
            &[0, 3, 193, 32, 0, 0, 10, 196, 29, 0, 0],
            &[0, 35, 209, 197, 145, 134, 200, 212, 69, 0, 0],
            &[0, 106, 48, 4, 58, 67, 14, 22, 137, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¥' / 0xa5
        '¥' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[11, 26, 0, 0, 0, 0, 0, 0, 18, 18, 0],
            &[17, 225, 24, 0, 0, 0, 0, 2, 209, 54, 0],
            &[0, 111, 154, 0, 0, 0, 0, 99, 163, 0, 0],
            &[0, 5, 214, 42, 0, 0, 11, 219, 28, 0, 0],
            &[0, 0, 79, 180, 0, 0, 127, 128, 0, 0, 0],
            &[0, 0, 0, 191, 65, 24, 216, 10, 0, 0, 0],
            &[0, 0, 0, 50, 202, 156, 92, 0, 0, 0, 0],
            &[0, 15, 160, 160, 237, 251, 161, 160, 50, 0, 0],
            &[0, 2, 30, 30, 115, 160, 30, 30, 9, 0, 0],
            &[0, 5, 58, 58, 132, 172, 58, 58, 18, 0, 0],
            &[0, 12, 132, 132, 178, 203, 132, 132, 41, 0, 0],
            &[0, 0, 0, 0, 96, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¦' / 0xa6
        '¦' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 23, 0, 0, 0, 0, 0],
        ]),
        // letter: '§' / 0xa7
        '§' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 141, 193, 191, 195, 160, 7, 0, 0],
            &[0, 0, 136, 124, 0, 0, 0, 39, 0, 0, 0],
            &[0, 0, 169, 84, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 211, 123, 25, 0, 0, 0, 0, 0],
            &[0, 0, 18, 176, 157, 195, 171, 33, 0, 0, 0],
            &[0, 0, 155, 64, 0, 0, 60, 218, 25, 0, 0],
            &[0, 0, 192, 33, 0, 0, 0, 151, 75, 0, 0],
            &[0, 0, 94, 194, 44, 0, 7, 197, 29, 0, 0],
            &[0, 0, 0, 60, 177, 194, 205, 85, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 148, 185, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 194, 54, 0, 0],
            &[0, 0, 37, 0, 0, 0, 17, 218, 22, 0, 0],
            &[0, 1, 176, 202, 178, 183, 197, 76, 0, 0, 0],
            &[0, 0, 0, 0, 12, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¨' / 0xa8
        '¨' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 54, 0, 0, 38, 24, 0, 0, 0],
            &[0, 0, 78, 234, 0, 0, 178, 133, 0, 0, 0],
            &[0, 0, 4, 31, 0, 0, 22, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '©' / 0xa9
        '©' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 106, 113, 70, 2, 0, 0, 0],
            &[0, 10, 157, 119, 49, 43, 98, 177, 34, 0, 0],
            &[0, 147, 49, 27, 148, 157, 117, 19, 177, 2, 0],
            &[20, 151, 0, 180, 27, 0, 4, 0, 103, 68, 0],
            &[60, 103, 24, 152, 0, 0, 0, 0, 48, 112, 0],
            &[56, 110, 20, 158, 0, 0, 0, 0, 53, 105, 0],
            &[12, 167, 0, 171, 67, 15, 31, 0, 122, 47, 0],
            &[0, 125, 90, 8, 105, 136, 67, 47, 155, 0, 0],
            &[0, 2, 123, 158, 101, 96, 144, 139, 9, 0, 0],
            &[0, 0, 0, 10, 51, 55, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ª' / 0xaa
        'ª' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 51, 57, 7, 0, 0, 0, 0],
            &[0, 0, 0, 128, 111, 121, 193, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 132, 55, 0, 0, 0],
            &[0, 0, 0, 112, 146, 143, 198, 61, 0, 0, 0],
            &[0, 0, 32, 163, 0, 0, 143, 61, 0, 0, 0],
            &[0, 0, 21, 209, 78, 98, 201, 61, 0, 0, 0],
            &[0, 0, 0, 35, 92, 47, 29, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '«' / 0xab
        '«' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 185, 5, 7, 182, 15, 0, 0],
            &[0, 0, 3, 183, 85, 0, 160, 106, 0, 0, 0],
            &[0, 0, 139, 140, 0, 114, 163, 0, 0, 0, 0],
            &[0, 26, 241, 15, 11, 240, 31, 0, 0, 0, 0],
            &[0, 0, 109, 166, 0, 86, 188, 3, 0, 0, 0],
            &[0, 0, 0, 154, 108, 0, 129, 135, 0, 0, 0],
            &[0, 0, 0, 5, 149, 4, 1, 147, 15, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¬' / 0xac
        '¬' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 93, 104, 104, 104, 104, 104, 104, 64, 0, 0],
            &[0, 93, 104, 104, 104, 104, 104, 134, 158, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 50, 158, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 50, 158, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 39, 125, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '®' / 0xae
        '®' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 106, 113, 70, 2, 0, 0, 0],
            &[0, 10, 157, 119, 49, 43, 98, 177, 34, 0, 0],
            &[0, 147, 49, 101, 164, 162, 115, 19, 177, 2, 0],
            &[20, 151, 0, 110, 46, 0, 179, 2, 103, 68, 0],
            &[60, 103, 0, 110, 82, 73, 164, 0, 48, 112, 0],
            &[56, 110, 0, 110, 124, 173, 40, 0, 53, 105, 0],
            &[12, 167, 0, 110, 46, 20, 164, 0, 122, 47, 0],
            &[0, 125, 90, 67, 28, 0, 84, 80, 155, 0, 0],
            &[0, 2, 123, 158, 101, 96, 144, 139, 9, 0, 0],
            &[0, 0, 0, 10, 51, 55, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¯' / 0xaf
        '¯' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 117, 117, 117, 117, 52, 0, 0, 0],
            &[0, 0, 22, 91, 91, 91, 91, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '°' / 0xb0
        '°' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 64, 5, 0, 0, 0, 0],
            &[0, 0, 2, 170, 137, 121, 189, 17, 0, 0, 0],
            &[0, 0, 67, 131, 0, 0, 77, 118, 0, 0, 0],
            &[0, 0, 86, 106, 0, 0, 49, 138, 0, 0, 0],
            &[0, 0, 18, 198, 47, 26, 176, 55, 0, 0, 0],
            &[0, 0, 0, 37, 146, 158, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '±' / 0xb1
        '±' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 120, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 125, 0, 0, 0, 0, 0],
            &[0, 82, 109, 109, 156, 181, 109, 109, 100, 0, 0],
            &[0, 72, 95, 95, 147, 174, 95, 95, 88, 0, 0],
            &[0, 0, 0, 0, 83, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 74, 112, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 151, 208, 208, 208, 208, 208, 208, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '²' / 0xb2
        '²' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 104, 117, 38, 0, 0, 0, 0],
            &[0, 0, 6, 163, 76, 65, 203, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 129, 78, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 191, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 147, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 143, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 142, 102, 0, 0, 0, 0, 0, 0],
            &[0, 0, 72, 228, 191, 191, 191, 111, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '³' / 0xb3
        '³' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 100, 108, 31, 0, 0, 0, 0],
            &[0, 0, 17, 163, 79, 88, 208, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 60, 0, 0, 0],
            &[0, 0, 0, 17, 53, 90, 175, 7, 0, 0, 0],
            &[0, 0, 0, 38, 118, 145, 167, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 112, 106, 0, 0, 0],
            &[0, 0, 11, 1, 0, 0, 144, 86, 0, 0, 0],
            &[0, 0, 56, 188, 178, 185, 145, 2, 0, 0, 0],
            &[0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '´' / 0xb4
        '´' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 222, 66, 0, 0, 0, 0],
            &[0, 0, 0, 1, 173, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 70, 86, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'µ' / 0xb5
        'µ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 5, 0, 0, 0, 0, 3, 7, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 239, 230, 0, 0, 0, 0, 102, 162, 0, 0],
            &[0, 234, 255, 0, 0, 0, 0, 154, 162, 0, 0],
            &[0, 174, 255, 22, 0, 0, 56, 204, 162, 0, 0],
            &[0, 128, 235, 229, 183, 192, 157, 61, 162, 0, 0],
            &[0, 128, 123, 4, 32, 9, 0, 0, 0, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 9, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¶' / 0xb6
        '¶' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 116, 212, 234, 224, 178, 178, 161, 0, 0],
            &[0, 101, 255, 255, 255, 208, 0, 2, 175, 0, 0],
            &[0, 192, 255, 255, 255, 208, 0, 2, 175, 0, 0],
            &[0, 224, 255, 255, 255, 208, 0, 2, 175, 0, 0],
            &[0, 223, 255, 255, 255, 208, 0, 2, 175, 0, 0],
            &[0, 186, 255, 255, 255, 208, 0, 2, 175, 0, 0],
            &[0, 89, 255, 255, 255, 208, 0, 2, 175, 0, 0],
            &[0, 0, 102, 194, 194, 198, 0, 2, 175, 0, 0],
            &[0, 0, 0, 0, 0, 178, 0, 2, 175, 0, 0],
            &[0, 0, 0, 0, 0, 178, 0, 2, 175, 0, 0],
            &[0, 0, 0, 0, 0, 178, 0, 2, 175, 0, 0],
            &[0, 0, 0, 0, 0, 178, 0, 2, 175, 0, 0],
            &[0, 0, 0, 0, 0, 178, 0, 2, 175, 0, 0],
            &[0, 0, 0, 0, 0, 178, 0, 2, 175, 0, 0],
            &[0, 0, 0, 0, 0, 178, 0, 2, 175, 0, 0],
            &[0, 0, 0, 0, 0, 28, 0, 0, 27, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '·' / 0xb7
        '·' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 47, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 206, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¸' / 0xb8
        '¸' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 137, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 173, 95, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 195, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¹' / 0xb9
        '¹' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 63, 0, 0, 0, 0, 0],
            &[0, 0, 13, 138, 192, 182, 0, 0, 0, 0, 0],
            &[0, 0, 15, 83, 16, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 135, 188, 255, 166, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'º' / 0xba
        'º' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 60, 4, 0, 0, 0, 0],
            &[0, 0, 0, 138, 140, 121, 188, 5, 0, 0, 0],
            &[0, 0, 23, 177, 0, 0, 129, 81, 0, 0, 0],
            &[0, 0, 54, 139, 0, 0, 89, 111, 0, 0, 0],
            &[0, 0, 27, 169, 0, 0, 119, 82, 0, 0, 0],
            &[0, 0, 0, 156, 112, 91, 187, 6, 0, 0, 0],
            &[0, 0, 0, 2, 68, 78, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '»' / 0xbb
        '»' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 17, 0, 128, 29, 0, 0, 0, 0],
            &[0, 0, 74, 186, 4, 52, 199, 10, 0, 0, 0],
            &[0, 0, 0, 129, 148, 0, 103, 171, 1, 0, 0],
            &[0, 0, 0, 2, 219, 62, 0, 196, 86, 0, 0],
            &[0, 0, 0, 97, 178, 1, 76, 196, 6, 0, 0],
            &[0, 0, 49, 204, 13, 35, 212, 24, 0, 0, 0],
            &[0, 0, 171, 35, 0, 156, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¼' / 0xbc
        '¼' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 28, 0, 0, 0, 0, 7, 24, 0, 0],
            &[67, 194, 201, 0, 0, 0, 0, 130, 99, 0, 0],
            &[67, 23, 197, 0, 0, 0, 20, 205, 4, 0, 0],
            &[0, 0, 199, 0, 0, 0, 139, 90, 0, 0, 0],
            &[0, 0, 200, 0, 0, 25, 202, 2, 0, 0, 0],
            &[0, 0, 200, 0, 0, 149, 81, 0, 0, 0, 0],
            &[60, 168, 255, 164, 65, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 71, 0, 93, 180, 0, 0],
            &[0, 0, 0, 38, 191, 0, 33, 164, 191, 0, 0],
            &[0, 0, 0, 167, 62, 3, 176, 25, 193, 0, 0],
            &[0, 0, 46, 184, 0, 126, 118, 42, 203, 28, 0],
            &[0, 0, 176, 53, 0, 106, 133, 134, 225, 94, 0],
            &[0, 54, 176, 0, 0, 0, 0, 2, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '½' / 0xbd
        '½' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 28, 0, 0, 0, 0, 7, 24, 0, 0],
            &[67, 194, 201, 0, 0, 0, 0, 130, 99, 0, 0],
            &[67, 23, 197, 0, 0, 0, 20, 205, 4, 0, 0],
            &[0, 0, 199, 0, 0, 0, 139, 90, 0, 0, 0],
            &[0, 0, 200, 0, 0, 25, 202, 2, 0, 0, 0],
            &[0, 0, 200, 0, 0, 149, 81, 0, 0, 0, 0],
            &[60, 168, 255, 164, 65, 197, 0, 1, 0, 0, 0],
            &[0, 0, 0, 0, 158, 73, 161, 179, 204, 51, 0],
            &[0, 0, 0, 38, 191, 0, 17, 0, 74, 145, 0],
            &[0, 0, 0, 167, 62, 0, 0, 0, 131, 88, 0],
            &[0, 0, 46, 184, 0, 0, 0, 93, 151, 0, 0],
            &[0, 0, 176, 53, 0, 0, 94, 151, 1, 0, 0],
            &[0, 54, 176, 0, 0, 42, 250, 192, 191, 145, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¾' / 0xbe
        '¾' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 64, 14, 0, 0, 0, 7, 24, 0, 0],
            &[72, 152, 124, 213, 14, 0, 0, 130, 99, 0, 0],
            &[0, 0, 0, 191, 31, 0, 20, 205, 4, 0, 0],
            &[0, 133, 203, 158, 0, 0, 139, 90, 0, 0, 0],
            &[0, 1, 29, 179, 47, 25, 202, 2, 0, 0, 0],
            &[21, 0, 0, 179, 63, 149, 81, 0, 0, 0, 0],
            &[87, 184, 180, 125, 32, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 71, 0, 93, 180, 0, 0],
            &[0, 0, 0, 38, 191, 0, 33, 164, 191, 0, 0],
            &[0, 0, 0, 167, 62, 3, 176, 25, 193, 0, 0],
            &[0, 0, 46, 184, 0, 126, 118, 42, 203, 28, 0],
            &[0, 0, 176, 53, 0, 106, 133, 134, 225, 94, 0],
            &[0, 54, 176, 0, 0, 0, 0, 2, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¿' / 0xbf
        '¿' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 94, 247, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 113, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 19, 161, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 64, 175, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 207, 63, 0, 0, 0, 0],
            &[0, 0, 0, 49, 213, 71, 0, 0, 0, 0, 0],
            &[0, 0, 17, 219, 44, 0, 0, 0, 0, 0, 0],
            &[0, 0, 95, 149, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 124, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 77, 200, 5, 0, 0, 0, 18, 0, 0],
            &[0, 0, 1, 146, 219, 154, 162, 212, 162, 0, 0],
            &[0, 0, 0, 0, 26, 71, 62, 25, 0, 0, 0],
        ]),
        // letter: 'À' / 0xc0
        'À' => Some(&[
            &[0, 0, 0, 154, 149, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 172, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 119, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 224, 188, 50, 0, 0, 0, 0],
            &[0, 0, 0, 88, 156, 101, 140, 0, 0, 0, 0],
            &[0, 0, 0, 178, 69, 19, 222, 1, 0, 0, 0],
            &[0, 0, 17, 227, 3, 0, 182, 62, 0, 0, 0],
            &[0, 0, 100, 149, 0, 0, 95, 151, 0, 0, 0],
            &[0, 0, 193, 79, 16, 16, 32, 235, 4, 0, 0],
            &[0, 25, 255, 215, 213, 213, 213, 255, 74, 0, 0],
            &[0, 112, 142, 0, 0, 0, 0, 90, 163, 0, 0],
            &[0, 201, 55, 0, 0, 0, 0, 12, 233, 9, 0],
            &[35, 222, 0, 0, 0, 0, 0, 0, 171, 86, 0],
            &[124, 135, 0, 0, 0, 0, 0, 0, 84, 175, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 6, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Á' / 0xc1
        'Á' => Some(&[
            &[0, 0, 0, 0, 0, 30, 228, 44, 0, 0, 0],
            &[0, 0, 0, 0, 8, 192, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 74, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 224, 188, 50, 0, 0, 0, 0],
            &[0, 0, 0, 88, 156, 101, 140, 0, 0, 0, 0],
            &[0, 0, 0, 178, 69, 19, 222, 1, 0, 0, 0],
            &[0, 0, 17, 227, 3, 0, 182, 62, 0, 0, 0],
            &[0, 0, 100, 149, 0, 0, 95, 151, 0, 0, 0],
            &[0, 0, 193, 79, 16, 16, 32, 235, 4, 0, 0],
            &[0, 25, 255, 215, 213, 213, 213, 255, 74, 0, 0],
            &[0, 112, 142, 0, 0, 0, 0, 90, 163, 0, 0],
            &[0, 201, 55, 0, 0, 0, 0, 12, 233, 9, 0],
            &[35, 222, 0, 0, 0, 0, 0, 0, 171, 86, 0],
            &[124, 135, 0, 0, 0, 0, 0, 0, 84, 175, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 15, 0, 0, 0],
        ]),
        // letter: 'Â' / 0xc2
        'Â' => Some(&[
            &[0, 0, 0, 3, 192, 219, 23, 0, 0, 0, 0],
            &[0, 0, 0, 144, 101, 59, 184, 6, 0, 0, 0],
            &[0, 0, 38, 93, 0, 0, 60, 72, 0, 0, 0],
            &[0, 0, 0, 0, 21, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 224, 188, 50, 0, 0, 0, 0],
            &[0, 0, 0, 88, 156, 101, 140, 0, 0, 0, 0],
            &[0, 0, 0, 178, 69, 19, 222, 1, 0, 0, 0],
            &[0, 0, 17, 227, 3, 0, 182, 62, 0, 0, 0],
            &[0, 0, 100, 149, 0, 0, 95, 151, 0, 0, 0],
            &[0, 0, 193, 79, 16, 16, 32, 235, 4, 0, 0],
            &[0, 25, 255, 215, 213, 213, 213, 255, 74, 0, 0],
            &[0, 112, 142, 0, 0, 0, 0, 90, 163, 0, 0],
            &[0, 201, 55, 0, 0, 0, 0, 12, 233, 9, 0],
            &[35, 222, 0, 0, 0, 0, 0, 0, 171, 86, 0],
            &[124, 135, 0, 0, 0, 0, 0, 0, 84, 175, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 15, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ã' / 0xc3
        'Ã' => Some(&[
            &[0, 0, 2, 83, 62, 0, 0, 76, 0, 0, 0],
            &[0, 0, 129, 143, 161, 169, 100, 178, 0, 0, 0],
            &[0, 0, 88, 0, 0, 59, 113, 21, 0, 0, 0],
            &[0, 0, 0, 0, 21, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 224, 188, 50, 0, 0, 0, 0],
            &[0, 0, 0, 88, 156, 101, 140, 0, 0, 0, 0],
            &[0, 0, 0, 178, 69, 19, 222, 1, 0, 0, 0],
            &[0, 0, 17, 227, 3, 0, 182, 62, 0, 0, 0],
            &[0, 0, 100, 149, 0, 0, 95, 151, 0, 0, 0],
            &[0, 0, 193, 79, 16, 16, 32, 235, 4, 0, 0],
            &[0, 25, 255, 215, 213, 213, 213, 255, 74, 0, 0],
            &[0, 112, 142, 0, 0, 0, 0, 90, 163, 0, 0],
            &[0, 201, 55, 0, 0, 0, 0, 12, 233, 9, 0],
            &[35, 222, 0, 0, 0, 0, 0, 0, 171, 86, 0],
            &[124, 135, 0, 0, 0, 0, 0, 0, 84, 175, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ä' / 0xc4
        'Ä' => Some(&[
            &[0, 0, 16, 75, 0, 0, 54, 36, 0, 0, 0],
            &[0, 0, 75, 231, 0, 0, 175, 130, 0, 0, 0],
            &[0, 0, 0, 12, 0, 0, 9, 3, 0, 0, 0],
            &[0, 0, 0, 0, 21, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 224, 188, 50, 0, 0, 0, 0],
            &[0, 0, 0, 88, 156, 101, 140, 0, 0, 0, 0],
            &[0, 0, 0, 178, 69, 19, 222, 1, 0, 0, 0],
            &[0, 0, 17, 227, 3, 0, 182, 62, 0, 0, 0],
            &[0, 0, 100, 149, 0, 0, 95, 151, 0, 0, 0],
            &[0, 0, 193, 79, 16, 16, 32, 235, 4, 0, 0],
            &[0, 25, 255, 215, 213, 213, 213, 255, 74, 0, 0],
            &[0, 112, 142, 0, 0, 0, 0, 90, 163, 0, 0],
            &[0, 201, 55, 0, 0, 0, 0, 12, 233, 9, 0],
            &[35, 222, 0, 0, 0, 0, 0, 0, 171, 86, 0],
            &[124, 135, 0, 0, 0, 0, 0, 0, 84, 175, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Å' / 0xc5
        'Å' => Some(&[
            &[0, 0, 0, 159, 32, 3, 187, 0, 0, 0, 0],
            &[0, 0, 0, 131, 102, 63, 174, 0, 0, 0, 0],
            &[0, 0, 0, 7, 108, 120, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 224, 188, 50, 0, 0, 0, 0],
            &[0, 0, 0, 88, 156, 101, 140, 0, 0, 0, 0],
            &[0, 0, 0, 178, 69, 19, 222, 1, 0, 0, 0],
            &[0, 0, 17, 227, 3, 0, 182, 62, 0, 0, 0],
            &[0, 0, 100, 149, 0, 0, 95, 151, 0, 0, 0],
            &[0, 0, 193, 79, 16, 16, 32, 235, 4, 0, 0],
            &[0, 25, 255, 215, 213, 213, 213, 255, 74, 0, 0],
            &[0, 112, 142, 0, 0, 0, 0, 90, 163, 0, 0],
            &[0, 201, 55, 0, 0, 0, 0, 12, 233, 9, 0],
            &[35, 222, 0, 0, 0, 0, 0, 0, 171, 86, 0],
            &[124, 135, 0, 0, 0, 0, 0, 0, 84, 175, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 43, 179, 176, 72, 0, 0, 0, 0],
        ]),
        // letter: 'Æ' / 0xc6
        'Æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 35, 35, 35, 35, 35, 20, 0],
            &[0, 0, 0, 83, 222, 252, 190, 190, 190, 112, 0],
            &[0, 0, 0, 157, 78, 243, 0, 0, 0, 0, 0],
            &[0, 0, 1, 224, 12, 243, 0, 0, 0, 0, 0],
            &[0, 0, 50, 189, 0, 243, 0, 0, 0, 0, 0],
            &[0, 0, 124, 116, 0, 243, 0, 0, 0, 0, 0],
            &[0, 0, 198, 43, 0, 243, 209, 209, 209, 62, 0],
            &[0, 19, 233, 17, 16, 250, 17, 17, 17, 5, 0],
            &[0, 90, 255, 204, 204, 255, 0, 0, 0, 0, 0],
            &[0, 165, 81, 0, 0, 243, 0, 0, 0, 0, 0],
            &[2, 231, 14, 0, 0, 243, 0, 0, 0, 0, 0],
            &[57, 192, 0, 0, 0, 243, 0, 0, 0, 0, 0],
            &[131, 119, 0, 0, 0, 243, 230, 230, 230, 135, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ç' / 0xc7
        'Ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 56, 72, 48, 5, 0, 0],
            &[0, 0, 0, 115, 228, 175, 154, 182, 227, 39, 0],
            &[0, 0, 130, 190, 21, 0, 0, 0, 13, 1, 0],
            &[0, 37, 231, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 78, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 207, 52, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 205, 54, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 183, 79, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 125, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 233, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 202, 38, 0, 0, 0, 37, 0, 0],
            &[0, 0, 0, 105, 224, 221, 204, 228, 200, 0, 0],
            &[0, 0, 0, 0, 0, 179, 62, 11, 0, 0, 0],
            &[0, 0, 0, 0, 3, 144, 173, 21, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 151, 79, 0, 0, 0],
            &[0, 0, 0, 0, 8, 9, 0, 0, 0, 0, 0],
        ]),
        // letter: 'È' / 0xc8
        'È' => Some(&[
            &[0, 0, 0, 119, 183, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 141, 122, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 109, 23, 0, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 25, 0, 0],
            &[0, 55, 241, 190, 190, 190, 190, 190, 137, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 245, 209, 209, 209, 209, 209, 79, 0, 0],
            &[0, 55, 201, 17, 17, 17, 17, 17, 6, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 250, 230, 230, 230, 230, 230, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 10, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'É' / 0xc9
        'É' => Some(&[
            &[0, 0, 0, 0, 0, 13, 219, 69, 0, 0, 0],
            &[0, 0, 0, 0, 1, 171, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 53, 80, 0, 0, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 25, 0, 0],
            &[0, 55, 241, 190, 190, 190, 190, 190, 137, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 245, 209, 209, 209, 209, 209, 79, 0, 0],
            &[0, 55, 201, 17, 17, 17, 17, 17, 6, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 250, 230, 230, 230, 230, 230, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 16, 19, 0, 0, 0],
        ]),
        // letter: 'Ê' / 0xca
        'Ê' => Some(&[
            &[0, 0, 0, 0, 165, 229, 44, 0, 0, 0, 0],
            &[0, 0, 0, 110, 135, 37, 196, 18, 0, 0, 0],
            &[0, 0, 20, 111, 0, 0, 40, 92, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 25, 0, 0],
            &[0, 55, 241, 190, 190, 190, 190, 190, 137, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 245, 209, 209, 209, 209, 209, 79, 0, 0],
            &[0, 55, 201, 17, 17, 17, 17, 17, 6, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 250, 230, 230, 230, 230, 230, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 19, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ë' / 0xcb
        'Ë' => Some(&[
            &[0, 0, 6, 85, 0, 0, 39, 50, 0, 0, 0],
            &[0, 0, 41, 252, 14, 0, 141, 165, 0, 0, 0],
            &[0, 0, 0, 13, 0, 0, 4, 8, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 25, 0, 0],
            &[0, 55, 241, 190, 190, 190, 190, 190, 137, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 245, 209, 209, 209, 209, 209, 79, 0, 0],
            &[0, 55, 201, 17, 17, 17, 17, 17, 6, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 250, 230, 230, 230, 230, 230, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ì' / 0xcc
        'Ì' => Some(&[
            &[0, 0, 0, 154, 149, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 172, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 119, 9, 0, 0, 0, 0],
            &[0, 0, 24, 35, 35, 35, 35, 31, 0, 0, 0],
            &[0, 0, 104, 161, 205, 222, 164, 135, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 129, 196, 227, 237, 199, 167, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 6, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Í' / 0xcd
        'Í' => Some(&[
            &[0, 0, 0, 0, 0, 30, 228, 44, 0, 0, 0],
            &[0, 0, 0, 0, 8, 192, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 74, 59, 0, 0, 0, 0, 0],
            &[0, 0, 24, 35, 35, 35, 35, 31, 0, 0, 0],
            &[0, 0, 104, 161, 205, 222, 164, 135, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 129, 196, 227, 237, 199, 167, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 15, 0, 0, 0],
        ]),
        // letter: 'Î' / 0xce
        'Î' => Some(&[
            &[0, 0, 0, 3, 192, 219, 23, 0, 0, 0, 0],
            &[0, 0, 0, 144, 101, 59, 184, 6, 0, 0, 0],
            &[0, 0, 38, 93, 0, 0, 60, 72, 0, 0, 0],
            &[0, 0, 24, 35, 35, 35, 35, 31, 0, 0, 0],
            &[0, 0, 104, 161, 205, 222, 164, 135, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 129, 196, 227, 237, 199, 167, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 15, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ï' / 0xcf
        'Ï' => Some(&[
            &[0, 0, 16, 75, 0, 0, 54, 36, 0, 0, 0],
            &[0, 0, 75, 231, 0, 0, 175, 130, 0, 0, 0],
            &[0, 0, 0, 12, 0, 0, 9, 3, 0, 0, 0],
            &[0, 0, 24, 35, 35, 35, 35, 31, 0, 0, 0],
            &[0, 0, 104, 161, 205, 222, 164, 135, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 129, 196, 227, 237, 199, 167, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ð' / 0xd0
        'Ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 35, 35, 27, 3, 0, 0, 0, 0, 0],
            &[0, 107, 225, 186, 197, 229, 179, 48, 0, 0, 0],
            &[0, 107, 145, 0, 0, 2, 83, 229, 58, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 73, 215, 1, 0],
            &[0, 107, 145, 0, 0, 0, 0, 0, 212, 52, 0],
            &[0, 107, 145, 0, 0, 0, 0, 0, 152, 104, 0],
            &[139, 246, 255, 139, 115, 0, 0, 0, 130, 125, 0],
            &[82, 189, 227, 82, 68, 0, 0, 0, 133, 117, 0],
            &[0, 107, 145, 0, 0, 0, 0, 0, 164, 93, 0],
            &[0, 107, 145, 0, 0, 0, 0, 3, 228, 27, 0],
            &[0, 107, 145, 0, 0, 0, 0, 113, 176, 0, 0],
            &[0, 107, 145, 0, 0, 28, 136, 214, 26, 0, 0],
            &[0, 107, 243, 226, 225, 210, 127, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ñ' / 0xd1
        'Ñ' => Some(&[
            &[0, 0, 0, 77, 70, 0, 0, 69, 6, 0, 0],
            &[0, 0, 107, 156, 150, 180, 98, 190, 0, 0, 0],
            &[0, 0, 84, 5, 0, 50, 114, 29, 0, 0, 0],
            &[0, 20, 24, 0, 0, 0, 0, 5, 28, 0, 0],
            &[0, 150, 235, 11, 0, 0, 0, 41, 206, 0, 0],
            &[0, 150, 193, 116, 0, 0, 0, 41, 206, 0, 0],
            &[0, 150, 98, 216, 9, 0, 0, 41, 206, 0, 0],
            &[0, 150, 90, 129, 111, 0, 0, 41, 206, 0, 0],
            &[0, 150, 92, 18, 216, 7, 0, 41, 206, 0, 0],
            &[0, 150, 92, 0, 135, 106, 0, 41, 206, 0, 0],
            &[0, 150, 92, 0, 21, 215, 6, 41, 206, 0, 0],
            &[0, 150, 92, 0, 0, 141, 101, 41, 206, 0, 0],
            &[0, 150, 92, 0, 0, 24, 213, 42, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 146, 127, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 28, 227, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 0, 152, 206, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ò' / 0xd2
        'Ò' => Some(&[
            &[0, 0, 0, 154, 149, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 172, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 119, 9, 0, 0, 0, 0],
            &[0, 0, 0, 10, 59, 72, 23, 0, 0, 0, 0],
            &[0, 0, 75, 223, 169, 154, 220, 128, 0, 0, 0],
            &[0, 38, 227, 37, 0, 0, 10, 195, 91, 0, 0],
            &[0, 155, 112, 0, 0, 0, 0, 51, 212, 0, 0],
            &[0, 226, 29, 0, 0, 0, 0, 0, 229, 28, 0],
            &[14, 241, 0, 0, 0, 0, 0, 0, 188, 68, 0],
            &[31, 224, 0, 0, 0, 0, 0, 0, 171, 84, 0],
            &[31, 226, 0, 0, 0, 0, 0, 0, 175, 82, 0],
            &[11, 244, 1, 0, 0, 0, 0, 0, 194, 61, 0],
            &[0, 220, 40, 0, 0, 0, 0, 2, 237, 18, 0],
            &[0, 140, 132, 0, 0, 0, 0, 74, 193, 0, 0],
            &[0, 26, 229, 70, 0, 0, 34, 220, 64, 0, 0],
            &[0, 0, 51, 209, 213, 202, 222, 86, 0, 0, 0],
            &[0, 0, 0, 0, 25, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 6, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ó' / 0xd3
        'Ó' => Some(&[
            &[0, 0, 0, 0, 0, 30, 228, 44, 0, 0, 0],
            &[0, 0, 0, 0, 8, 192, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 74, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 59, 72, 23, 0, 0, 0, 0],
            &[0, 0, 75, 223, 169, 154, 220, 128, 0, 0, 0],
            &[0, 38, 227, 37, 0, 0, 10, 195, 91, 0, 0],
            &[0, 155, 112, 0, 0, 0, 0, 51, 212, 0, 0],
            &[0, 226, 29, 0, 0, 0, 0, 0, 229, 28, 0],
            &[14, 241, 0, 0, 0, 0, 0, 0, 188, 68, 0],
            &[31, 224, 0, 0, 0, 0, 0, 0, 171, 84, 0],
            &[31, 226, 0, 0, 0, 0, 0, 0, 175, 82, 0],
            &[11, 244, 1, 0, 0, 0, 0, 0, 194, 61, 0],
            &[0, 220, 40, 0, 0, 0, 0, 2, 237, 18, 0],
            &[0, 140, 132, 0, 0, 0, 0, 74, 193, 0, 0],
            &[0, 26, 229, 70, 0, 0, 34, 220, 64, 0, 0],
            &[0, 0, 51, 209, 213, 202, 222, 86, 0, 0, 0],
            &[0, 0, 0, 0, 25, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 15, 0, 0, 0],
        ]),
        // letter: 'Ô' / 0xd4
        'Ô' => Some(&[
            &[0, 0, 0, 3, 192, 219, 23, 0, 0, 0, 0],
            &[0, 0, 0, 144, 101, 59, 184, 6, 0, 0, 0],
            &[0, 0, 38, 93, 0, 0, 60, 72, 0, 0, 0],
            &[0, 0, 0, 10, 59, 72, 23, 0, 0, 0, 0],
            &[0, 0, 75, 223, 169, 154, 220, 128, 0, 0, 0],
            &[0, 38, 227, 37, 0, 0, 10, 195, 91, 0, 0],
            &[0, 155, 112, 0, 0, 0, 0, 51, 212, 0, 0],
            &[0, 226, 29, 0, 0, 0, 0, 0, 229, 28, 0],
            &[14, 241, 0, 0, 0, 0, 0, 0, 188, 68, 0],
            &[31, 224, 0, 0, 0, 0, 0, 0, 171, 84, 0],
            &[31, 226, 0, 0, 0, 0, 0, 0, 175, 82, 0],
            &[11, 244, 1, 0, 0, 0, 0, 0, 194, 61, 0],
            &[0, 220, 40, 0, 0, 0, 0, 2, 237, 18, 0],
            &[0, 140, 132, 0, 0, 0, 0, 74, 193, 0, 0],
            &[0, 26, 229, 70, 0, 0, 34, 220, 64, 0, 0],
            &[0, 0, 51, 209, 213, 202, 222, 86, 0, 0, 0],
            &[0, 0, 0, 0, 25, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 15, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Õ' / 0xd5
        'Õ' => Some(&[
            &[0, 0, 2, 83, 62, 0, 0, 76, 0, 0, 0],
            &[0, 0, 129, 143, 161, 169, 100, 178, 0, 0, 0],
            &[0, 0, 88, 0, 0, 59, 113, 21, 0, 0, 0],
            &[0, 0, 0, 10, 59, 72, 23, 0, 0, 0, 0],
            &[0, 0, 75, 223, 169, 154, 220, 128, 0, 0, 0],
            &[0, 38, 227, 37, 0, 0, 10, 195, 91, 0, 0],
            &[0, 155, 112, 0, 0, 0, 0, 51, 212, 0, 0],
            &[0, 226, 29, 0, 0, 0, 0, 0, 229, 28, 0],
            &[14, 241, 0, 0, 0, 0, 0, 0, 188, 68, 0],
            &[31, 224, 0, 0, 0, 0, 0, 0, 171, 84, 0],
            &[31, 226, 0, 0, 0, 0, 0, 0, 175, 82, 0],
            &[11, 244, 1, 0, 0, 0, 0, 0, 194, 61, 0],
            &[0, 220, 40, 0, 0, 0, 0, 2, 237, 18, 0],
            &[0, 140, 132, 0, 0, 0, 0, 74, 193, 0, 0],
            &[0, 26, 229, 70, 0, 0, 34, 220, 64, 0, 0],
            &[0, 0, 51, 209, 213, 202, 222, 86, 0, 0, 0],
            &[0, 0, 0, 0, 25, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ö' / 0xd6
        'Ö' => Some(&[
            &[0, 0, 16, 75, 0, 0, 54, 36, 0, 0, 0],
            &[0, 0, 75, 231, 0, 0, 175, 130, 0, 0, 0],
            &[0, 0, 0, 12, 0, 0, 9, 3, 0, 0, 0],
            &[0, 0, 0, 10, 59, 72, 23, 0, 0, 0, 0],
            &[0, 0, 75, 223, 169, 154, 220, 128, 0, 0, 0],
            &[0, 38, 227, 37, 0, 0, 10, 195, 91, 0, 0],
            &[0, 155, 112, 0, 0, 0, 0, 51, 212, 0, 0],
            &[0, 226, 29, 0, 0, 0, 0, 0, 229, 28, 0],
            &[14, 241, 0, 0, 0, 0, 0, 0, 188, 68, 0],
            &[31, 224, 0, 0, 0, 0, 0, 0, 171, 84, 0],
            &[31, 226, 0, 0, 0, 0, 0, 0, 175, 82, 0],
            &[11, 244, 1, 0, 0, 0, 0, 0, 194, 61, 0],
            &[0, 220, 40, 0, 0, 0, 0, 2, 237, 18, 0],
            &[0, 140, 132, 0, 0, 0, 0, 74, 193, 0, 0],
            &[0, 26, 229, 70, 0, 0, 34, 220, 64, 0, 0],
            &[0, 0, 51, 209, 213, 202, 222, 86, 0, 0, 0],
            &[0, 0, 0, 0, 25, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '×' / 0xd7
        '×' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 48, 58, 0, 0, 0, 0, 27, 80, 0, 0],
            &[0, 27, 205, 57, 0, 0, 27, 204, 56, 0, 0],
            &[0, 0, 27, 204, 56, 27, 204, 57, 0, 0, 0],
            &[0, 0, 0, 27, 204, 208, 58, 0, 0, 0, 0],
            &[0, 0, 0, 27, 205, 210, 58, 0, 0, 0, 0],
            &[0, 0, 27, 205, 58, 28, 205, 58, 0, 0, 0],
            &[0, 27, 205, 58, 0, 0, 26, 204, 58, 0, 0],
            &[0, 45, 58, 0, 0, 0, 0, 25, 77, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ø' / 0xd8
        'Ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 59, 72, 23, 0, 100, 2, 0],
            &[0, 0, 75, 223, 169, 154, 220, 196, 174, 0, 0],
            &[0, 38, 227, 37, 0, 0, 13, 255, 126, 0, 0],
            &[0, 155, 112, 0, 0, 0, 104, 190, 212, 0, 0],
            &[0, 226, 29, 0, 0, 15, 212, 14, 229, 28, 0],
            &[14, 241, 0, 0, 0, 141, 102, 0, 188, 68, 0],
            &[31, 224, 0, 0, 37, 203, 2, 0, 171, 84, 0],
            &[31, 226, 0, 0, 178, 66, 0, 0, 175, 82, 0],
            &[11, 244, 1, 68, 176, 0, 0, 0, 194, 61, 0],
            &[0, 220, 43, 205, 36, 0, 0, 2, 237, 18, 0],
            &[0, 140, 237, 140, 0, 0, 0, 74, 193, 0, 0],
            &[0, 42, 255, 86, 0, 0, 34, 220, 64, 0, 0],
            &[0, 142, 155, 209, 213, 202, 222, 86, 0, 0, 0],
            &[0, 84, 3, 0, 25, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ù' / 0xd9
        'Ù' => Some(&[
            &[0, 0, 0, 154, 149, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 172, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 119, 9, 0, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 6, 28, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 149, 102, 0, 0, 0, 0, 47, 205, 0, 0],
            &[0, 135, 117, 0, 0, 0, 0, 62, 190, 0, 0],
            &[0, 93, 169, 0, 0, 0, 0, 117, 145, 0, 0],
            &[0, 16, 232, 61, 0, 0, 35, 227, 47, 0, 0],
            &[0, 0, 60, 218, 206, 200, 224, 88, 0, 0, 0],
            &[0, 0, 0, 1, 27, 30, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 6, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ú' / 0xda
        'Ú' => Some(&[
            &[0, 0, 0, 0, 0, 30, 228, 44, 0, 0, 0],
            &[0, 0, 0, 0, 8, 192, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 74, 59, 0, 0, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 6, 28, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 149, 102, 0, 0, 0, 0, 47, 205, 0, 0],
            &[0, 135, 117, 0, 0, 0, 0, 62, 190, 0, 0],
            &[0, 93, 169, 0, 0, 0, 0, 117, 145, 0, 0],
            &[0, 16, 232, 61, 0, 0, 35, 227, 47, 0, 0],
            &[0, 0, 60, 218, 206, 200, 224, 88, 0, 0, 0],
            &[0, 0, 0, 1, 27, 30, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 15, 0, 0, 0],
        ]),
        // letter: 'Û' / 0xdb
        'Û' => Some(&[
            &[0, 0, 0, 3, 192, 219, 23, 0, 0, 0, 0],
            &[0, 0, 0, 144, 101, 59, 184, 6, 0, 0, 0],
            &[0, 0, 38, 93, 0, 0, 60, 72, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 6, 28, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 149, 102, 0, 0, 0, 0, 47, 205, 0, 0],
            &[0, 135, 117, 0, 0, 0, 0, 62, 190, 0, 0],
            &[0, 93, 169, 0, 0, 0, 0, 117, 145, 0, 0],
            &[0, 16, 232, 61, 0, 0, 35, 227, 47, 0, 0],
            &[0, 0, 60, 218, 206, 200, 224, 88, 0, 0, 0],
            &[0, 0, 0, 1, 27, 30, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 15, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ü' / 0xdc
        'Ü' => Some(&[
            &[0, 0, 16, 75, 0, 0, 54, 36, 0, 0, 0],
            &[0, 0, 75, 231, 0, 0, 175, 130, 0, 0, 0],
            &[0, 0, 0, 12, 0, 0, 9, 3, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 6, 28, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 149, 102, 0, 0, 0, 0, 47, 205, 0, 0],
            &[0, 135, 117, 0, 0, 0, 0, 62, 190, 0, 0],
            &[0, 93, 169, 0, 0, 0, 0, 117, 145, 0, 0],
            &[0, 16, 232, 61, 0, 0, 35, 227, 47, 0, 0],
            &[0, 0, 60, 218, 206, 200, 224, 88, 0, 0, 0],
            &[0, 0, 0, 1, 27, 30, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ý' / 0xdd
        'Ý' => Some(&[
            &[0, 0, 0, 0, 0, 30, 228, 44, 0, 0, 0],
            &[0, 0, 0, 0, 8, 192, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 74, 59, 0, 0, 0, 0, 0],
            &[8, 30, 0, 0, 0, 0, 0, 0, 21, 15, 0],
            &[9, 226, 38, 0, 0, 0, 0, 5, 221, 42, 0],
            &[0, 105, 163, 0, 0, 0, 0, 101, 161, 0, 0],
            &[0, 5, 218, 38, 0, 0, 4, 217, 33, 0, 0],
            &[0, 0, 92, 163, 0, 0, 98, 150, 0, 0, 0],
            &[0, 0, 2, 209, 38, 4, 211, 26, 0, 0, 0],
            &[0, 0, 0, 80, 163, 95, 139, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 214, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 98, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 15, 0, 0, 0],
        ]),
        // letter: 'Þ' / 0xde
        'Þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 27, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 245, 211, 211, 205, 177, 96, 2, 0, 0],
            &[0, 55, 199, 10, 10, 24, 67, 198, 151, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 20, 246, 21, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 213, 53, 0],
            &[0, 55, 197, 0, 0, 0, 0, 1, 230, 30, 0],
            &[0, 55, 197, 0, 0, 0, 0, 102, 194, 0, 0],
            &[0, 55, 226, 128, 129, 144, 198, 196, 28, 0, 0],
            &[0, 55, 219, 97, 97, 82, 41, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ß' / 0xdf
        'ß' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 50, 187, 221, 221, 203, 72, 0, 0, 0],
            &[0, 18, 231, 73, 0, 0, 53, 238, 22, 0, 0],
            &[0, 89, 168, 0, 0, 0, 0, 210, 47, 0, 0],
            &[0, 110, 137, 0, 0, 0, 68, 211, 4, 0, 0],
            &[0, 111, 136, 0, 0, 92, 201, 27, 0, 0, 0],
            &[0, 111, 136, 0, 2, 235, 21, 0, 0, 0, 0],
            &[0, 111, 136, 0, 2, 234, 34, 0, 0, 0, 0],
            &[0, 111, 136, 0, 0, 78, 222, 71, 0, 0, 0],
            &[0, 111, 136, 0, 0, 0, 43, 211, 110, 0, 0],
            &[0, 111, 136, 0, 0, 0, 0, 18, 227, 28, 0],
            &[0, 111, 136, 0, 0, 0, 0, 0, 174, 74, 0],
            &[0, 111, 136, 1, 22, 0, 0, 12, 224, 39, 0],
            &[0, 111, 136, 3, 208, 199, 179, 220, 113, 0, 0],
            &[0, 0, 0, 0, 0, 19, 36, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'à' / 0xe0
        'à' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 154, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 174, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 134, 17, 0, 0, 0, 0],
            &[0, 0, 0, 4, 33, 42, 9, 0, 0, 0, 0],
            &[0, 0, 133, 216, 177, 172, 222, 120, 0, 0, 0],
            &[0, 0, 34, 2, 0, 0, 10, 211, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 128, 110, 0, 0],
            &[0, 0, 0, 1, 36, 61, 72, 153, 127, 0, 0],
            &[0, 0, 114, 212, 166, 133, 119, 172, 127, 0, 0],
            &[0, 91, 187, 9, 0, 0, 0, 113, 127, 0, 0],
            &[0, 153, 90, 0, 0, 0, 0, 165, 127, 0, 0],
            &[0, 119, 151, 0, 0, 0, 88, 214, 127, 0, 0],
            &[0, 10, 181, 210, 176, 201, 125, 87, 127, 0, 0],
            &[0, 0, 0, 17, 31, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'á' / 0xe1
        'á' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 218, 73, 0, 0, 0],
            &[0, 0, 0, 0, 0, 167, 102, 0, 0, 0, 0],
            &[0, 0, 0, 0, 64, 92, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 33, 42, 9, 0, 0, 0, 0],
            &[0, 0, 133, 216, 177, 172, 222, 120, 0, 0, 0],
            &[0, 0, 34, 2, 0, 0, 10, 211, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 128, 110, 0, 0],
            &[0, 0, 0, 1, 36, 61, 72, 153, 127, 0, 0],
            &[0, 0, 114, 212, 166, 133, 119, 172, 127, 0, 0],
            &[0, 91, 187, 9, 0, 0, 0, 113, 127, 0, 0],
            &[0, 153, 90, 0, 0, 0, 0, 165, 127, 0, 0],
            &[0, 119, 151, 0, 0, 0, 88, 214, 127, 0, 0],
            &[0, 10, 181, 210, 176, 201, 125, 87, 127, 0, 0],
            &[0, 0, 0, 17, 31, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'â' / 0xe2
        'â' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 228, 25, 0, 0, 0, 0],
            &[0, 0, 0, 103, 144, 59, 187, 6, 0, 0, 0],
            &[0, 0, 26, 126, 1, 0, 63, 91, 0, 0, 0],
            &[0, 0, 0, 4, 33, 42, 9, 0, 0, 0, 0],
            &[0, 0, 133, 216, 177, 172, 222, 120, 0, 0, 0],
            &[0, 0, 34, 2, 0, 0, 10, 211, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 128, 110, 0, 0],
            &[0, 0, 0, 1, 36, 61, 72, 153, 127, 0, 0],
            &[0, 0, 114, 212, 166, 133, 119, 172, 127, 0, 0],
            &[0, 91, 187, 9, 0, 0, 0, 113, 127, 0, 0],
            &[0, 153, 90, 0, 0, 0, 0, 165, 127, 0, 0],
            &[0, 119, 151, 0, 0, 0, 88, 214, 127, 0, 0],
            &[0, 10, 181, 210, 176, 201, 125, 87, 127, 0, 0],
            &[0, 0, 0, 17, 31, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ã' / 0xe3
        'ã' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 53, 46, 0, 0, 52, 6, 0, 0],
            &[0, 0, 94, 176, 173, 159, 73, 192, 0, 0, 0],
            &[0, 0, 98, 9, 0, 71, 138, 44, 0, 0, 0],
            &[0, 0, 0, 4, 33, 42, 9, 0, 0, 0, 0],
            &[0, 0, 133, 216, 177, 172, 222, 120, 0, 0, 0],
            &[0, 0, 34, 2, 0, 0, 10, 211, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 128, 110, 0, 0],
            &[0, 0, 0, 1, 36, 61, 72, 153, 127, 0, 0],
            &[0, 0, 114, 212, 166, 133, 119, 172, 127, 0, 0],
            &[0, 91, 187, 9, 0, 0, 0, 113, 127, 0, 0],
            &[0, 153, 90, 0, 0, 0, 0, 165, 127, 0, 0],
            &[0, 119, 151, 0, 0, 0, 88, 214, 127, 0, 0],
            &[0, 10, 181, 210, 176, 201, 125, 87, 127, 0, 0],
            &[0, 0, 0, 17, 31, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ä' / 0xe4
        'ä' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 58, 0, 0, 31, 31, 0, 0, 0],
            &[0, 0, 56, 251, 5, 0, 156, 155, 0, 0, 0],
            &[0, 0, 1, 33, 0, 0, 17, 17, 0, 0, 0],
            &[0, 0, 0, 4, 33, 42, 9, 0, 0, 0, 0],
            &[0, 0, 133, 216, 177, 172, 222, 120, 0, 0, 0],
            &[0, 0, 34, 2, 0, 0, 10, 211, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 128, 110, 0, 0],
            &[0, 0, 0, 1, 36, 61, 72, 153, 127, 0, 0],
            &[0, 0, 114, 212, 166, 133, 119, 172, 127, 0, 0],
            &[0, 91, 187, 9, 0, 0, 0, 113, 127, 0, 0],
            &[0, 153, 90, 0, 0, 0, 0, 165, 127, 0, 0],
            &[0, 119, 151, 0, 0, 0, 88, 214, 127, 0, 0],
            &[0, 10, 181, 210, 176, 201, 125, 87, 127, 0, 0],
            &[0, 0, 0, 17, 31, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'å' / 0xe5
        'å' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 165, 172, 69, 0, 0, 0, 0],
            &[0, 0, 0, 132, 62, 2, 191, 0, 0, 0, 0],
            &[0, 0, 0, 119, 100, 30, 189, 0, 0, 0, 0],
            &[0, 0, 0, 8, 124, 149, 39, 0, 0, 0, 0],
            &[0, 0, 0, 4, 33, 42, 9, 0, 0, 0, 0],
            &[0, 0, 133, 216, 177, 172, 222, 120, 0, 0, 0],
            &[0, 0, 34, 2, 0, 0, 10, 211, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 128, 110, 0, 0],
            &[0, 0, 0, 1, 36, 61, 72, 153, 127, 0, 0],
            &[0, 0, 114, 212, 166, 133, 119, 172, 127, 0, 0],
            &[0, 91, 187, 9, 0, 0, 0, 113, 127, 0, 0],
            &[0, 153, 90, 0, 0, 0, 0, 165, 127, 0, 0],
            &[0, 119, 151, 0, 0, 0, 88, 214, 127, 0, 0],
            &[0, 10, 181, 210, 176, 201, 125, 87, 127, 0, 0],
            &[0, 0, 0, 17, 31, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'æ' / 0xe6
        'æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 29, 37, 0, 0, 12, 43, 4, 0, 0],
            &[0, 187, 183, 190, 168, 47, 209, 187, 201, 10, 0],
            &[0, 16, 0, 0, 173, 203, 27, 0, 148, 103, 0],
            &[0, 0, 0, 0, 114, 173, 0, 0, 75, 162, 0],
            &[0, 0, 14, 49, 141, 166, 60, 60, 101, 184, 0],
            &[1, 147, 211, 151, 184, 198, 144, 144, 144, 106, 0],
            &[84, 184, 3, 0, 114, 135, 0, 0, 0, 0, 0],
            &[130, 107, 0, 0, 148, 180, 0, 0, 0, 0, 0],
            &[104, 143, 0, 19, 194, 177, 46, 0, 0, 23, 0],
            &[14, 208, 187, 217, 61, 34, 208, 181, 206, 85, 0],
            &[0, 2, 31, 7, 0, 0, 3, 32, 12, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ç' / 0xe7
        'ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 38, 43, 12, 0, 0, 0],
            &[0, 0, 6, 136, 227, 188, 181, 219, 90, 0, 0],
            &[0, 0, 159, 169, 12, 0, 0, 0, 7, 0, 0],
            &[0, 36, 229, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 101, 156, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 133, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 152, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 221, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 181, 150, 7, 0, 0, 1, 15, 0, 0],
            &[0, 0, 12, 154, 228, 198, 194, 218, 90, 0, 0],
            &[0, 0, 0, 0, 12, 212, 29, 4, 0, 0, 0],
            &[0, 0, 0, 0, 32, 157, 152, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 216, 14, 0, 0, 0],
            &[0, 0, 0, 0, 12, 4, 0, 0, 0, 0, 0],
        ]),
        // letter: 'è' / 0xe8
        'è' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 155, 150, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 177, 86, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 134, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 46, 15, 0, 0, 0, 0],
            &[0, 0, 41, 200, 198, 167, 212, 141, 2, 0, 0],
            &[0, 17, 227, 71, 0, 0, 2, 169, 113, 0, 0],
            &[0, 122, 141, 0, 0, 0, 0, 36, 216, 0, 0],
            &[0, 188, 111, 56, 56, 56, 56, 56, 250, 4, 0],
            &[0, 209, 171, 152, 152, 152, 152, 152, 152, 9, 0],
            &[0, 192, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 137, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 225, 76, 0, 0, 0, 4, 43, 0, 0],
            &[0, 0, 35, 182, 215, 181, 195, 216, 123, 0, 0],
            &[0, 0, 0, 0, 10, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'é' / 0xe9
        'é' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 220, 70, 0, 0, 0],
            &[0, 0, 0, 0, 1, 170, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 67, 89, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 46, 15, 0, 0, 0, 0],
            &[0, 0, 41, 200, 198, 167, 212, 141, 2, 0, 0],
            &[0, 17, 227, 71, 0, 0, 2, 169, 113, 0, 0],
            &[0, 122, 141, 0, 0, 0, 0, 36, 216, 0, 0],
            &[0, 188, 111, 56, 56, 56, 56, 56, 250, 4, 0],
            &[0, 209, 171, 152, 152, 152, 152, 152, 152, 9, 0],
            &[0, 192, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 137, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 225, 76, 0, 0, 0, 4, 43, 0, 0],
            &[0, 0, 35, 182, 215, 181, 195, 216, 123, 0, 0],
            &[0, 0, 0, 0, 10, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ê' / 0xea
        'ê' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 165, 226, 22, 0, 0, 0, 0],
            &[0, 0, 0, 107, 140, 62, 185, 5, 0, 0, 0],
            &[0, 0, 28, 124, 0, 0, 67, 88, 0, 0, 0],
            &[0, 0, 0, 0, 26, 46, 15, 0, 0, 0, 0],
            &[0, 0, 41, 200, 198, 167, 212, 141, 2, 0, 0],
            &[0, 17, 227, 71, 0, 0, 2, 169, 113, 0, 0],
            &[0, 122, 141, 0, 0, 0, 0, 36, 216, 0, 0],
            &[0, 188, 111, 56, 56, 56, 56, 56, 250, 4, 0],
            &[0, 209, 171, 152, 152, 152, 152, 152, 152, 9, 0],
            &[0, 192, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 137, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 225, 76, 0, 0, 0, 4, 43, 0, 0],
            &[0, 0, 35, 182, 215, 181, 195, 216, 123, 0, 0],
            &[0, 0, 0, 0, 10, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ë' / 0xeb
        'ë' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 57, 0, 0, 33, 30, 0, 0, 0],
            &[0, 0, 60, 248, 3, 0, 161, 150, 0, 0, 0],
            &[0, 0, 2, 33, 0, 0, 18, 16, 0, 0, 0],
            &[0, 0, 0, 0, 26, 46, 15, 0, 0, 0, 0],
            &[0, 0, 41, 200, 198, 167, 212, 141, 2, 0, 0],
            &[0, 17, 227, 71, 0, 0, 2, 169, 113, 0, 0],
            &[0, 122, 141, 0, 0, 0, 0, 36, 216, 0, 0],
            &[0, 188, 111, 56, 56, 56, 56, 56, 250, 4, 0],
            &[0, 209, 171, 152, 152, 152, 152, 152, 152, 9, 0],
            &[0, 192, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 137, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 225, 76, 0, 0, 0, 4, 43, 0, 0],
            &[0, 0, 35, 182, 215, 181, 195, 216, 123, 0, 0],
            &[0, 0, 0, 0, 10, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ì' / 0xec
        'ì' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 173, 132, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 188, 71, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 137, 9, 0, 0, 0, 0],
            &[0, 2, 11, 11, 11, 10, 0, 0, 0, 0, 0],
            &[0, 30, 175, 188, 205, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 197, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'í' / 0xed
        'í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 226, 56, 0, 0, 0],
            &[0, 0, 0, 0, 3, 183, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 77, 0, 0, 0, 0, 0],
            &[0, 2, 11, 11, 11, 10, 0, 0, 0, 0, 0],
            &[0, 30, 175, 188, 205, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 197, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'î' / 0xee
        'î' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 181, 217, 14, 0, 0, 0, 0],
            &[0, 0, 0, 125, 122, 76, 174, 2, 0, 0, 0],
            &[0, 0, 38, 115, 0, 0, 79, 76, 0, 0, 0],
            &[0, 2, 11, 11, 11, 10, 0, 0, 0, 0, 0],
            &[0, 30, 175, 188, 205, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 197, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ï' / 0xef
        'ï' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 54, 0, 0, 38, 24, 0, 0, 0],
            &[0, 0, 78, 234, 0, 0, 178, 133, 0, 0, 0],
            &[0, 0, 4, 31, 0, 0, 22, 12, 0, 0, 0],
            &[0, 2, 11, 11, 11, 10, 0, 0, 0, 0, 0],
            &[0, 30, 175, 188, 205, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 197, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ð' / 0xf0
        'ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 101, 138, 12, 38, 161, 8, 0, 0],
            &[0, 0, 0, 3, 123, 255, 222, 47, 0, 0, 0],
            &[0, 0, 0, 91, 202, 153, 209, 22, 0, 0, 0],
            &[0, 0, 0, 80, 12, 0, 82, 177, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 173, 61, 0, 0],
            &[0, 0, 40, 172, 213, 208, 183, 120, 164, 0, 0],
            &[0, 37, 223, 71, 0, 0, 26, 184, 234, 0, 0],
            &[0, 162, 102, 0, 0, 0, 0, 27, 253, 20, 0],
            &[0, 220, 31, 0, 0, 0, 0, 0, 222, 33, 0],
            &[0, 226, 24, 0, 0, 0, 0, 0, 232, 16, 0],
            &[0, 180, 78, 0, 0, 0, 0, 40, 215, 0, 0],
            &[0, 62, 213, 27, 0, 0, 13, 192, 98, 0, 0],
            &[0, 0, 79, 211, 187, 181, 222, 114, 0, 0, 0],
            &[0, 0, 0, 1, 26, 34, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ñ' / 0xf1
        'ñ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 53, 0, 0, 45, 13, 0, 0],
            &[0, 0, 73, 186, 164, 172, 72, 193, 8, 0, 0],
            &[0, 0, 87, 20, 0, 61, 138, 54, 0, 0, 0],
            &[0, 4, 4, 0, 9, 43, 21, 0, 0, 0, 0],
            &[0, 107, 109, 131, 197, 168, 216, 170, 5, 0, 0],
            &[0, 107, 215, 100, 0, 0, 3, 181, 111, 0, 0],
            &[0, 107, 209, 0, 0, 0, 0, 89, 167, 0, 0],
            &[0, 107, 163, 0, 0, 0, 0, 68, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ò' / 0xf2
        'ò' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 173, 132, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 188, 71, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 137, 9, 0, 0, 0, 0],
            &[0, 0, 0, 2, 35, 42, 8, 0, 0, 0, 0],
            &[0, 0, 71, 213, 189, 171, 217, 115, 0, 0, 0],
            &[0, 45, 225, 43, 0, 0, 10, 192, 95, 0, 0],
            &[0, 158, 110, 0, 0, 0, 0, 48, 213, 0, 0],
            &[0, 215, 45, 0, 0, 0, 0, 0, 240, 15, 0],
            &[0, 232, 20, 0, 0, 0, 0, 0, 224, 32, 0],
            &[0, 215, 39, 0, 0, 0, 0, 1, 241, 15, 0],
            &[0, 156, 108, 0, 0, 0, 0, 57, 209, 0, 0],
            &[0, 41, 224, 40, 0, 0, 17, 203, 87, 0, 0],
            &[0, 0, 65, 211, 190, 183, 221, 104, 0, 0, 0],
            &[0, 0, 0, 0, 27, 34, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ó' / 0xf3
        'ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 226, 56, 0, 0, 0],
            &[0, 0, 0, 0, 3, 183, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 35, 42, 8, 0, 0, 0, 0],
            &[0, 0, 71, 213, 189, 171, 217, 115, 0, 0, 0],
            &[0, 45, 225, 43, 0, 0, 10, 192, 95, 0, 0],
            &[0, 158, 110, 0, 0, 0, 0, 48, 213, 0, 0],
            &[0, 215, 45, 0, 0, 0, 0, 0, 240, 15, 0],
            &[0, 232, 20, 0, 0, 0, 0, 0, 224, 32, 0],
            &[0, 215, 39, 0, 0, 0, 0, 1, 241, 15, 0],
            &[0, 156, 108, 0, 0, 0, 0, 57, 209, 0, 0],
            &[0, 41, 224, 40, 0, 0, 17, 203, 87, 0, 0],
            &[0, 0, 65, 211, 190, 183, 221, 104, 0, 0, 0],
            &[0, 0, 0, 0, 27, 34, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ô' / 0xf4
        'ô' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 181, 217, 14, 0, 0, 0, 0],
            &[0, 0, 0, 125, 122, 76, 174, 2, 0, 0, 0],
            &[0, 0, 38, 115, 0, 0, 79, 76, 0, 0, 0],
            &[0, 0, 0, 2, 35, 42, 8, 0, 0, 0, 0],
            &[0, 0, 71, 213, 189, 171, 217, 115, 0, 0, 0],
            &[0, 45, 225, 43, 0, 0, 10, 192, 95, 0, 0],
            &[0, 158, 110, 0, 0, 0, 0, 48, 213, 0, 0],
            &[0, 215, 45, 0, 0, 0, 0, 0, 240, 15, 0],
            &[0, 232, 20, 0, 0, 0, 0, 0, 224, 32, 0],
            &[0, 215, 39, 0, 0, 0, 0, 1, 241, 15, 0],
            &[0, 156, 108, 0, 0, 0, 0, 57, 209, 0, 0],
            &[0, 41, 224, 40, 0, 0, 17, 203, 87, 0, 0],
            &[0, 0, 65, 211, 190, 183, 221, 104, 0, 0, 0],
            &[0, 0, 0, 0, 27, 34, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'õ' / 0xf5
        'õ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 59, 40, 0, 0, 58, 0, 0, 0],
            &[0, 0, 115, 165, 182, 146, 76, 183, 0, 0, 0],
            &[0, 0, 105, 2, 0, 82, 137, 33, 0, 0, 0],
            &[0, 0, 0, 2, 35, 42, 8, 0, 0, 0, 0],
            &[0, 0, 71, 213, 189, 171, 217, 115, 0, 0, 0],
            &[0, 45, 225, 43, 0, 0, 10, 192, 95, 0, 0],
            &[0, 158, 110, 0, 0, 0, 0, 48, 213, 0, 0],
            &[0, 215, 45, 0, 0, 0, 0, 0, 240, 15, 0],
            &[0, 232, 20, 0, 0, 0, 0, 0, 224, 32, 0],
            &[0, 215, 39, 0, 0, 0, 0, 1, 241, 15, 0],
            &[0, 156, 108, 0, 0, 0, 0, 57, 209, 0, 0],
            &[0, 41, 224, 40, 0, 0, 17, 203, 87, 0, 0],
            &[0, 0, 65, 211, 190, 183, 221, 104, 0, 0, 0],
            &[0, 0, 0, 0, 27, 34, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ö' / 0xf6
        'ö' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 54, 0, 0, 38, 24, 0, 0, 0],
            &[0, 0, 78, 234, 0, 0, 178, 133, 0, 0, 0],
            &[0, 0, 4, 31, 0, 0, 22, 12, 0, 0, 0],
            &[0, 0, 0, 2, 35, 42, 8, 0, 0, 0, 0],
            &[0, 0, 71, 213, 189, 171, 217, 115, 0, 0, 0],
            &[0, 45, 225, 43, 0, 0, 10, 192, 95, 0, 0],
            &[0, 158, 110, 0, 0, 0, 0, 48, 213, 0, 0],
            &[0, 215, 45, 0, 0, 0, 0, 0, 240, 15, 0],
            &[0, 232, 20, 0, 0, 0, 0, 0, 224, 32, 0],
            &[0, 215, 39, 0, 0, 0, 0, 1, 241, 15, 0],
            &[0, 156, 108, 0, 0, 0, 0, 57, 209, 0, 0],
            &[0, 41, 224, 40, 0, 0, 17, 203, 87, 0, 0],
            &[0, 0, 65, 211, 190, 183, 221, 104, 0, 0, 0],
            &[0, 0, 0, 0, 27, 34, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '÷' / 0xf7
        '÷' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 128, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 75, 104, 104, 104, 104, 104, 104, 96, 0, 0],
            &[0, 75, 104, 104, 104, 104, 104, 104, 96, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 89, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 112, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ø' / 0xf8
        'ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 35, 42, 8, 0, 75, 4, 0],
            &[0, 0, 71, 213, 189, 171, 217, 194, 178, 1, 0],
            &[0, 45, 225, 43, 0, 0, 43, 255, 114, 0, 0],
            &[0, 158, 110, 0, 0, 7, 198, 103, 213, 0, 0],
            &[0, 215, 45, 0, 0, 149, 110, 0, 240, 15, 0],
            &[0, 232, 20, 0, 88, 171, 0, 0, 224, 32, 0],
            &[0, 215, 39, 39, 206, 14, 0, 1, 241, 15, 0],
            &[0, 156, 118, 202, 48, 0, 0, 57, 209, 0, 0],
            &[0, 41, 255, 142, 0, 0, 17, 203, 87, 0, 0],
            &[0, 99, 228, 211, 190, 183, 221, 104, 0, 0, 0],
            &[0, 60, 11, 0, 27, 34, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ù' / 0xf9
        'ù' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 173, 132, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 188, 71, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 137, 9, 0, 0, 0, 0],
            &[0, 5, 5, 0, 0, 0, 0, 3, 7, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 125, 0, 0, 0, 0, 102, 162, 0, 0],
            &[0, 105, 149, 0, 0, 0, 0, 154, 162, 0, 0],
            &[0, 45, 230, 22, 0, 0, 56, 204, 162, 0, 0],
            &[0, 0, 111, 229, 183, 192, 157, 61, 162, 0, 0],
            &[0, 0, 0, 4, 32, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ú' / 0xfa
        'ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 226, 56, 0, 0, 0],
            &[0, 0, 0, 0, 3, 183, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 77, 0, 0, 0, 0, 0],
            &[0, 5, 5, 0, 0, 0, 0, 3, 7, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 125, 0, 0, 0, 0, 102, 162, 0, 0],
            &[0, 105, 149, 0, 0, 0, 0, 154, 162, 0, 0],
            &[0, 45, 230, 22, 0, 0, 56, 204, 162, 0, 0],
            &[0, 0, 111, 229, 183, 192, 157, 61, 162, 0, 0],
            &[0, 0, 0, 4, 32, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'û' / 0xfb
        'û' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 181, 217, 14, 0, 0, 0, 0],
            &[0, 0, 0, 125, 122, 76, 174, 2, 0, 0, 0],
            &[0, 0, 38, 115, 0, 0, 79, 76, 0, 0, 0],
            &[0, 5, 5, 0, 0, 0, 0, 3, 7, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 125, 0, 0, 0, 0, 102, 162, 0, 0],
            &[0, 105, 149, 0, 0, 0, 0, 154, 162, 0, 0],
            &[0, 45, 230, 22, 0, 0, 56, 204, 162, 0, 0],
            &[0, 0, 111, 229, 183, 192, 157, 61, 162, 0, 0],
            &[0, 0, 0, 4, 32, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ü' / 0xfc
        'ü' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 54, 0, 0, 38, 24, 0, 0, 0],
            &[0, 0, 78, 234, 0, 0, 178, 133, 0, 0, 0],
            &[0, 0, 4, 31, 0, 0, 22, 12, 0, 0, 0],
            &[0, 5, 5, 0, 0, 0, 0, 3, 7, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 125, 0, 0, 0, 0, 102, 162, 0, 0],
            &[0, 105, 149, 0, 0, 0, 0, 154, 162, 0, 0],
            &[0, 45, 230, 22, 0, 0, 56, 204, 162, 0, 0],
            &[0, 0, 111, 229, 183, 192, 157, 61, 162, 0, 0],
            &[0, 0, 0, 4, 32, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ý' / 0xfd
        'ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 226, 56, 0, 0, 0],
            &[0, 0, 0, 0, 3, 183, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 77, 0, 0, 0, 0, 0],
            &[0, 9, 1, 0, 0, 0, 0, 0, 10, 0, 0],
            &[0, 160, 90, 0, 0, 0, 0, 32, 224, 2, 0],
            &[0, 58, 189, 0, 0, 0, 0, 122, 131, 0, 0],
            &[0, 0, 211, 34, 0, 0, 0, 213, 35, 0, 0],
            &[0, 0, 111, 132, 0, 0, 50, 194, 0, 0, 0],
            &[0, 0, 18, 220, 3, 0, 142, 97, 0, 0, 0],
            &[0, 0, 0, 164, 75, 3, 219, 12, 0, 0, 0],
            &[0, 0, 0, 62, 172, 68, 160, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 168, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 114, 222, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 210, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 89, 171, 0, 0, 0, 0, 0, 0],
            &[0, 19, 39, 4, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'þ' / 0xfe
        'þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 56, 74, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 140, 0, 16, 48, 23, 0, 0, 0, 0],
            &[0, 107, 140, 122, 200, 170, 217, 166, 7, 0, 0],
            &[0, 107, 211, 112, 0, 0, 3, 166, 133, 0, 0],
            &[0, 107, 216, 1, 0, 0, 0, 36, 232, 0, 0],
            &[0, 107, 164, 0, 0, 0, 0, 0, 245, 22, 0],
            &[0, 107, 145, 0, 0, 0, 0, 0, 230, 32, 0],
            &[0, 107, 167, 0, 0, 0, 0, 1, 250, 14, 0],
            &[0, 107, 222, 2, 0, 0, 0, 53, 216, 0, 0],
            &[0, 107, 216, 117, 0, 0, 11, 189, 108, 0, 0],
            &[0, 107, 138, 124, 205, 183, 225, 135, 1, 0, 0],
            &[0, 107, 140, 0, 4, 33, 10, 0, 0, 0, 0],
            &[0, 107, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 11, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ÿ' / 0xff
        'ÿ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 54, 0, 0, 38, 24, 0, 0, 0],
            &[0, 0, 78, 234, 0, 0, 178, 133, 0, 0, 0],
            &[0, 0, 4, 31, 0, 0, 22, 12, 0, 0, 0],
            &[0, 9, 1, 0, 0, 0, 0, 0, 10, 0, 0],
            &[0, 160, 90, 0, 0, 0, 0, 32, 224, 2, 0],
            &[0, 58, 189, 0, 0, 0, 0, 122, 131, 0, 0],
            &[0, 0, 211, 34, 0, 0, 0, 213, 35, 0, 0],
            &[0, 0, 111, 132, 0, 0, 50, 194, 0, 0, 0],
            &[0, 0, 18, 220, 3, 0, 142, 97, 0, 0, 0],
            &[0, 0, 0, 164, 75, 3, 219, 12, 0, 0, 0],
            &[0, 0, 0, 62, 172, 68, 160, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 168, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 114, 222, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 210, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 89, 171, 0, 0, 0, 0, 0, 0],
            &[0, 19, 39, 4, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ā' / 0x100
        'Ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 141, 141, 141, 141, 63, 0, 0, 0],
            &[0, 0, 16, 67, 67, 67, 67, 30, 0, 0, 0],
            &[0, 0, 0, 0, 21, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 224, 188, 50, 0, 0, 0, 0],
            &[0, 0, 0, 88, 156, 101, 140, 0, 0, 0, 0],
            &[0, 0, 0, 178, 69, 19, 222, 1, 0, 0, 0],
            &[0, 0, 17, 227, 3, 0, 182, 62, 0, 0, 0],
            &[0, 0, 100, 149, 0, 0, 95, 151, 0, 0, 0],
            &[0, 0, 193, 79, 16, 16, 32, 235, 4, 0, 0],
            &[0, 25, 255, 215, 213, 213, 213, 255, 74, 0, 0],
            &[0, 112, 142, 0, 0, 0, 0, 90, 163, 0, 0],
            &[0, 201, 55, 0, 0, 0, 0, 12, 233, 9, 0],
            &[35, 222, 0, 0, 0, 0, 0, 0, 171, 86, 0],
            &[124, 135, 0, 0, 0, 0, 0, 0, 84, 175, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ā' / 0x101
        'ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 117, 117, 117, 117, 62, 0, 0, 0],
            &[0, 0, 14, 91, 91, 91, 91, 48, 0, 0, 0],
            &[0, 0, 0, 4, 33, 42, 9, 0, 0, 0, 0],
            &[0, 0, 133, 216, 177, 172, 222, 120, 0, 0, 0],
            &[0, 0, 34, 2, 0, 0, 10, 211, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 128, 110, 0, 0],
            &[0, 0, 0, 1, 36, 61, 72, 153, 127, 0, 0],
            &[0, 0, 114, 212, 166, 133, 119, 172, 127, 0, 0],
            &[0, 91, 187, 9, 0, 0, 0, 113, 127, 0, 0],
            &[0, 153, 90, 0, 0, 0, 0, 165, 127, 0, 0],
            &[0, 119, 151, 0, 0, 0, 88, 214, 127, 0, 0],
            &[0, 10, 181, 210, 176, 201, 125, 87, 127, 0, 0],
            &[0, 0, 0, 17, 31, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ă' / 0x102
        'Ă' => Some(&[
            &[0, 0, 28, 83, 0, 0, 49, 57, 0, 0, 0],
            &[0, 0, 7, 198, 92, 72, 193, 32, 0, 0, 0],
            &[0, 0, 0, 18, 109, 116, 34, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 224, 188, 50, 0, 0, 0, 0],
            &[0, 0, 0, 88, 156, 101, 140, 0, 0, 0, 0],
            &[0, 0, 0, 178, 69, 19, 222, 1, 0, 0, 0],
            &[0, 0, 17, 227, 3, 0, 182, 62, 0, 0, 0],
            &[0, 0, 100, 149, 0, 0, 95, 151, 0, 0, 0],
            &[0, 0, 193, 79, 16, 16, 32, 235, 4, 0, 0],
            &[0, 25, 255, 215, 213, 213, 213, 255, 74, 0, 0],
            &[0, 112, 142, 0, 0, 0, 0, 90, 163, 0, 0],
            &[0, 201, 55, 0, 0, 0, 0, 12, 233, 9, 0],
            &[35, 222, 0, 0, 0, 0, 0, 0, 171, 86, 0],
            &[124, 135, 0, 0, 0, 0, 0, 0, 84, 175, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ă' / 0x103
        'ă' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 77, 0, 0, 29, 59, 0, 0, 0],
            &[0, 0, 2, 197, 77, 44, 175, 58, 0, 0, 0],
            &[0, 0, 0, 23, 128, 142, 63, 0, 0, 0, 0],
            &[0, 0, 0, 4, 33, 42, 9, 0, 0, 0, 0],
            &[0, 0, 133, 216, 177, 172, 222, 120, 0, 0, 0],
            &[0, 0, 34, 2, 0, 0, 10, 211, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 128, 110, 0, 0],
            &[0, 0, 0, 1, 36, 61, 72, 153, 127, 0, 0],
            &[0, 0, 114, 212, 166, 133, 119, 172, 127, 0, 0],
            &[0, 91, 187, 9, 0, 0, 0, 113, 127, 0, 0],
            &[0, 153, 90, 0, 0, 0, 0, 165, 127, 0, 0],
            &[0, 119, 151, 0, 0, 0, 88, 214, 127, 0, 0],
            &[0, 10, 181, 210, 176, 201, 125, 87, 127, 0, 0],
            &[0, 0, 0, 17, 31, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ą' / 0x104
        'Ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 224, 188, 50, 0, 0, 0, 0],
            &[0, 0, 0, 88, 156, 101, 140, 0, 0, 0, 0],
            &[0, 0, 0, 178, 69, 19, 222, 1, 0, 0, 0],
            &[0, 0, 17, 227, 3, 0, 182, 62, 0, 0, 0],
            &[0, 0, 100, 149, 0, 0, 95, 151, 0, 0, 0],
            &[0, 0, 193, 79, 16, 16, 32, 235, 4, 0, 0],
            &[0, 25, 255, 215, 213, 213, 213, 255, 74, 0, 0],
            &[0, 112, 142, 0, 0, 0, 0, 90, 163, 0, 0],
            &[0, 201, 55, 0, 0, 0, 0, 12, 233, 9, 0],
            &[35, 222, 0, 0, 0, 0, 0, 0, 171, 86, 0],
            &[124, 135, 0, 0, 0, 0, 0, 0, 84, 198, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 117, 101, 0],
            &[0, 0, 0, 0, 0, 0, 0, 32, 169, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 65, 178, 12, 4],
            &[0, 0, 0, 0, 0, 0, 0, 2, 131, 172, 30],
        ]),
        // letter: 'ą' / 0x105
        'ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 33, 42, 9, 0, 0, 0, 0],
            &[0, 0, 133, 216, 177, 172, 222, 120, 0, 0, 0],
            &[0, 0, 34, 2, 0, 0, 10, 211, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 128, 110, 0, 0],
            &[0, 0, 0, 1, 36, 61, 72, 153, 127, 0, 0],
            &[0, 0, 114, 212, 166, 133, 119, 172, 127, 0, 0],
            &[0, 91, 187, 9, 0, 0, 0, 113, 127, 0, 0],
            &[0, 153, 90, 0, 0, 0, 0, 165, 127, 0, 0],
            &[0, 119, 151, 0, 0, 0, 88, 214, 127, 0, 0],
            &[0, 10, 181, 210, 176, 201, 125, 99, 138, 0, 0],
            &[0, 0, 0, 17, 31, 2, 14, 175, 28, 0, 0],
            &[0, 0, 0, 0, 0, 0, 134, 67, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 170, 81, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 41, 165, 130, 0, 0],
        ]),
        // letter: 'Ć' / 0x106
        'Ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 27, 227, 47, 0, 0],
            &[0, 0, 0, 0, 0, 6, 189, 69, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 56, 72, 48, 5, 0, 0],
            &[0, 0, 0, 115, 228, 175, 154, 182, 227, 39, 0],
            &[0, 0, 130, 190, 21, 0, 0, 0, 13, 1, 0],
            &[0, 37, 231, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 78, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 207, 52, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 205, 54, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 183, 79, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 125, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 233, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 202, 38, 0, 0, 0, 37, 0, 0],
            &[0, 0, 0, 105, 224, 216, 200, 228, 200, 0, 0],
            &[0, 0, 0, 0, 0, 20, 35, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 19, 15, 0, 0],
        ]),
        // letter: 'ć' / 0x107
        'ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 128, 175, 0, 0, 0],
            &[0, 0, 0, 0, 0, 67, 191, 10, 0, 0, 0],
            &[0, 0, 0, 0, 8, 137, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 38, 43, 12, 0, 0, 0],
            &[0, 0, 6, 136, 227, 188, 181, 219, 90, 0, 0],
            &[0, 0, 159, 169, 12, 0, 0, 0, 7, 0, 0],
            &[0, 36, 229, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 101, 156, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 133, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 152, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 221, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 181, 150, 7, 0, 0, 1, 15, 0, 0],
            &[0, 0, 12, 154, 228, 190, 193, 218, 90, 0, 0],
            &[0, 0, 0, 0, 7, 33, 28, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĉ' / 0x108
        'Ĉ' => Some(&[
            &[0, 0, 0, 0, 3, 189, 220, 25, 0, 0, 0],
            &[0, 0, 0, 0, 139, 106, 55, 187, 8, 0, 0],
            &[0, 0, 0, 35, 96, 0, 0, 57, 75, 0, 0],
            &[0, 0, 0, 0, 6, 56, 72, 48, 5, 0, 0],
            &[0, 0, 0, 115, 228, 175, 154, 182, 227, 39, 0],
            &[0, 0, 130, 190, 21, 0, 0, 0, 13, 1, 0],
            &[0, 37, 231, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 78, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 207, 52, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 205, 54, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 183, 79, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 125, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 233, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 202, 38, 0, 0, 0, 37, 0, 0],
            &[0, 0, 0, 105, 224, 216, 200, 228, 200, 0, 0],
            &[0, 0, 0, 0, 0, 20, 35, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 16, 0, 0, 0, 0],
        ]),
        // letter: 'ĉ' / 0x109
        'ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 244, 110, 0, 0, 0, 0],
            &[0, 0, 0, 25, 196, 33, 186, 59, 0, 0, 0],
            &[0, 0, 0, 122, 31, 0, 12, 137, 5, 0, 0],
            &[0, 0, 0, 0, 6, 38, 43, 12, 0, 0, 0],
            &[0, 0, 6, 136, 227, 188, 181, 219, 90, 0, 0],
            &[0, 0, 159, 169, 12, 0, 0, 0, 7, 0, 0],
            &[0, 36, 229, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 101, 156, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 133, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 152, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 221, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 181, 150, 7, 0, 0, 1, 15, 0, 0],
            &[0, 0, 12, 154, 228, 190, 193, 218, 90, 0, 0],
            &[0, 0, 0, 0, 7, 33, 28, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ċ' / 0x10a
        'Ċ' => Some(&[
            &[0, 0, 0, 0, 0, 75, 116, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 128, 188, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 56, 72, 48, 5, 0, 0],
            &[0, 0, 0, 115, 228, 175, 154, 182, 227, 39, 0],
            &[0, 0, 130, 190, 21, 0, 0, 0, 13, 1, 0],
            &[0, 37, 231, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 78, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 207, 52, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 205, 54, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 183, 79, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 125, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 233, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 202, 38, 0, 0, 0, 37, 0, 0],
            &[0, 0, 0, 105, 224, 216, 200, 228, 200, 0, 0],
            &[0, 0, 0, 0, 0, 20, 35, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ċ' / 0x10b
        'ċ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 152, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 61, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 38, 43, 12, 0, 0, 0],
            &[0, 0, 6, 136, 227, 188, 181, 219, 90, 0, 0],
            &[0, 0, 159, 169, 12, 0, 0, 0, 7, 0, 0],
            &[0, 36, 229, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 101, 156, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 133, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 152, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 221, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 181, 150, 7, 0, 0, 1, 15, 0, 0],
            &[0, 0, 12, 154, 228, 190, 193, 218, 90, 0, 0],
            &[0, 0, 0, 0, 7, 33, 28, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Č' / 0x10c
        'Č' => Some(&[
            &[0, 0, 0, 27, 184, 25, 6, 168, 67, 0, 0],
            &[0, 0, 0, 0, 49, 200, 171, 96, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 121, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 56, 72, 48, 5, 0, 0],
            &[0, 0, 0, 115, 228, 175, 154, 182, 227, 39, 0],
            &[0, 0, 130, 190, 21, 0, 0, 0, 13, 1, 0],
            &[0, 37, 231, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 78, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 207, 52, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 205, 54, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 183, 79, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 125, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 233, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 202, 38, 0, 0, 0, 37, 0, 0],
            &[0, 0, 0, 105, 224, 216, 200, 228, 200, 0, 0],
            &[0, 0, 0, 0, 0, 20, 35, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 8, 0, 0, 3, 16, 0, 0],
        ]),
        // letter: 'č' / 0x10d
        'č' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 146, 84, 0, 53, 176, 6, 0, 0],
            &[0, 0, 0, 4, 185, 111, 202, 17, 0, 0, 0],
            &[0, 0, 0, 0, 26, 176, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 38, 43, 12, 0, 0, 0],
            &[0, 0, 6, 136, 227, 188, 181, 219, 90, 0, 0],
            &[0, 0, 159, 169, 12, 0, 0, 0, 7, 0, 0],
            &[0, 36, 229, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 101, 156, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 133, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 152, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 221, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 181, 150, 7, 0, 0, 1, 15, 0, 0],
            &[0, 0, 12, 154, 228, 190, 193, 218, 90, 0, 0],
            &[0, 0, 0, 0, 7, 33, 28, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ď' / 0x10e
        'Ď' => Some(&[
            &[0, 0, 56, 172, 8, 21, 185, 35, 0, 0, 0],
            &[0, 0, 0, 89, 174, 197, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 118, 92, 0, 0, 0, 0, 0],
            &[0, 20, 35, 35, 23, 1, 0, 0, 0, 0, 0],
            &[0, 150, 214, 186, 202, 230, 162, 27, 0, 0, 0],
            &[0, 150, 101, 0, 0, 7, 110, 226, 29, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 116, 173, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 9, 240, 15, 0],
            &[0, 150, 101, 0, 0, 0, 0, 0, 196, 60, 0],
            &[0, 150, 101, 0, 0, 0, 0, 0, 174, 81, 0],
            &[0, 150, 101, 0, 0, 0, 0, 0, 176, 74, 0],
            &[0, 150, 101, 0, 0, 0, 0, 0, 207, 49, 0],
            &[0, 150, 101, 0, 0, 0, 0, 25, 231, 3, 0],
            &[0, 150, 101, 0, 0, 0, 0, 157, 132, 0, 0],
            &[0, 150, 101, 0, 0, 41, 164, 191, 8, 0, 0],
            &[0, 150, 238, 226, 224, 200, 108, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 4, 0, 0, 8, 11, 0, 0, 0],
        ]),
        // letter: 'ď' / 0x10f
        'ď' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 225, 0, 104, 122],
            &[0, 0, 0, 0, 0, 0, 7, 244, 0, 170, 44],
            &[0, 0, 0, 0, 0, 0, 7, 244, 2, 136, 0],
            &[0, 0, 0, 23, 42, 5, 7, 244, 0, 0, 0],
            &[0, 5, 161, 215, 167, 201, 68, 244, 0, 0, 0],
            &[0, 120, 178, 2, 0, 5, 167, 241, 0, 0, 0],
            &[0, 215, 49, 0, 0, 0, 74, 244, 0, 0, 0],
            &[6, 248, 2, 0, 0, 0, 24, 244, 0, 0, 0],
            &[21, 236, 0, 0, 0, 0, 10, 244, 0, 0, 0],
            &[10, 250, 1, 0, 0, 0, 29, 244, 0, 0, 0],
            &[0, 222, 43, 0, 0, 0, 79, 244, 0, 0, 0],
            &[0, 127, 171, 3, 0, 13, 173, 236, 0, 0, 0],
            &[0, 6, 166, 220, 183, 203, 55, 217, 0, 0, 0],
            &[0, 0, 0, 19, 33, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Đ' / 0x110
        'Đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 35, 35, 27, 3, 0, 0, 0, 0, 0],
            &[0, 107, 225, 186, 197, 229, 179, 48, 0, 0, 0],
            &[0, 107, 145, 0, 0, 2, 83, 229, 58, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 73, 215, 1, 0],
            &[0, 107, 145, 0, 0, 0, 0, 0, 212, 52, 0],
            &[0, 107, 145, 0, 0, 0, 0, 0, 152, 104, 0],
            &[139, 246, 255, 139, 115, 0, 0, 0, 130, 125, 0],
            &[82, 189, 227, 82, 68, 0, 0, 0, 133, 117, 0],
            &[0, 107, 145, 0, 0, 0, 0, 0, 164, 93, 0],
            &[0, 107, 145, 0, 0, 0, 0, 3, 228, 27, 0],
            &[0, 107, 145, 0, 0, 0, 0, 113, 176, 0, 0],
            &[0, 107, 145, 0, 0, 28, 136, 214, 26, 0, 0],
            &[0, 107, 243, 226, 225, 210, 127, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'đ' / 0x111
        'đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 82, 149, 0, 0, 0],
            &[0, 0, 0, 44, 69, 69, 158, 232, 69, 13, 0],
            &[0, 0, 0, 84, 130, 130, 220, 255, 130, 26, 0],
            &[0, 0, 3, 29, 13, 0, 89, 162, 0, 0, 0],
            &[0, 90, 222, 195, 190, 175, 100, 162, 0, 0, 0],
            &[57, 229, 41, 0, 0, 55, 213, 162, 0, 0, 0],
            &[164, 112, 0, 0, 0, 0, 163, 162, 0, 0, 0],
            &[216, 50, 0, 0, 0, 0, 108, 162, 0, 0, 0],
            &[233, 26, 0, 0, 0, 0, 92, 162, 0, 0, 0],
            &[221, 43, 0, 0, 0, 0, 111, 162, 0, 0, 0],
            &[174, 96, 0, 0, 0, 0, 164, 162, 0, 0, 0],
            &[71, 218, 28, 0, 0, 60, 208, 162, 0, 0, 0],
            &[0, 107, 227, 188, 198, 165, 66, 162, 0, 0, 0],
            &[0, 0, 6, 35, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ē' / 0x112
        'Ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 141, 141, 141, 141, 82, 0, 0, 0],
            &[0, 0, 7, 67, 67, 67, 67, 39, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 25, 0, 0],
            &[0, 55, 241, 190, 190, 190, 190, 190, 137, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 245, 209, 209, 209, 209, 209, 79, 0, 0],
            &[0, 55, 201, 17, 17, 17, 17, 17, 6, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 250, 230, 230, 230, 230, 230, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ē' / 0x113
        'ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 117, 117, 117, 117, 60, 0, 0, 0],
            &[0, 0, 16, 91, 91, 91, 91, 47, 0, 0, 0],
            &[0, 0, 0, 0, 26, 46, 15, 0, 0, 0, 0],
            &[0, 0, 41, 200, 198, 167, 212, 141, 2, 0, 0],
            &[0, 17, 227, 71, 0, 0, 2, 169, 113, 0, 0],
            &[0, 122, 141, 0, 0, 0, 0, 36, 216, 0, 0],
            &[0, 188, 111, 56, 56, 56, 56, 56, 250, 4, 0],
            &[0, 209, 171, 152, 152, 152, 152, 152, 152, 9, 0],
            &[0, 192, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 137, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 225, 76, 0, 0, 0, 4, 43, 0, 0],
            &[0, 0, 35, 182, 215, 181, 195, 216, 123, 0, 0],
            &[0, 0, 0, 0, 10, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĕ' / 0x114
        'Ĕ' => Some(&[
            &[0, 0, 8, 102, 0, 0, 29, 77, 0, 0, 0],
            &[0, 0, 0, 182, 107, 66, 182, 58, 0, 0, 0],
            &[0, 0, 0, 9, 100, 120, 48, 0, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 25, 0, 0],
            &[0, 55, 241, 190, 190, 190, 190, 190, 137, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 245, 209, 209, 209, 209, 209, 79, 0, 0],
            &[0, 55, 201, 17, 17, 17, 17, 17, 6, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 250, 230, 230, 230, 230, 230, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĕ' / 0x115
        'ĕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 75, 0, 0, 31, 57, 0, 0, 0],
            &[0, 0, 3, 198, 75, 45, 177, 54, 0, 0, 0],
            &[0, 0, 0, 25, 129, 142, 61, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 46, 15, 0, 0, 0, 0],
            &[0, 0, 41, 200, 198, 167, 212, 141, 2, 0, 0],
            &[0, 17, 227, 71, 0, 0, 2, 169, 113, 0, 0],
            &[0, 122, 141, 0, 0, 0, 0, 36, 216, 0, 0],
            &[0, 188, 111, 56, 56, 56, 56, 56, 250, 4, 0],
            &[0, 209, 171, 152, 152, 152, 152, 152, 152, 9, 0],
            &[0, 192, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 137, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 225, 76, 0, 0, 0, 4, 43, 0, 0],
            &[0, 0, 35, 182, 215, 181, 195, 216, 123, 0, 0],
            &[0, 0, 0, 0, 10, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ė' / 0x116
        'Ė' => Some(&[
            &[0, 0, 0, 0, 56, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 215, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 25, 0, 0],
            &[0, 55, 241, 190, 190, 190, 190, 190, 137, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 245, 209, 209, 209, 209, 209, 79, 0, 0],
            &[0, 55, 201, 17, 17, 17, 17, 17, 6, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 250, 230, 230, 230, 230, 230, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ė' / 0x117
        'ė' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 46, 15, 0, 0, 0, 0],
            &[0, 0, 41, 200, 198, 167, 212, 141, 2, 0, 0],
            &[0, 17, 227, 71, 0, 0, 2, 169, 113, 0, 0],
            &[0, 122, 141, 0, 0, 0, 0, 36, 216, 0, 0],
            &[0, 188, 111, 56, 56, 56, 56, 56, 250, 4, 0],
            &[0, 209, 171, 152, 152, 152, 152, 152, 152, 9, 0],
            &[0, 192, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 137, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 225, 76, 0, 0, 0, 4, 43, 0, 0],
            &[0, 0, 35, 182, 215, 181, 195, 216, 123, 0, 0],
            &[0, 0, 0, 0, 10, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ę' / 0x118
        'Ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 25, 0, 0],
            &[0, 55, 241, 190, 190, 190, 190, 190, 137, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 245, 209, 209, 209, 209, 209, 79, 0, 0],
            &[0, 55, 201, 17, 17, 17, 17, 17, 6, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 250, 230, 230, 230, 230, 253, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 78, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 144, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 154, 11, 2, 0, 0],
            &[0, 0, 0, 0, 0, 9, 143, 170, 13, 0, 0],
        ]),
        // letter: 'ę' / 0x119
        'ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 46, 15, 0, 0, 0, 0],
            &[0, 0, 41, 200, 198, 167, 212, 141, 2, 0, 0],
            &[0, 17, 227, 71, 0, 0, 2, 169, 113, 0, 0],
            &[0, 122, 141, 0, 0, 0, 0, 36, 216, 0, 0],
            &[0, 188, 111, 56, 56, 56, 56, 56, 250, 4, 0],
            &[0, 209, 171, 152, 152, 152, 152, 152, 152, 9, 0],
            &[0, 192, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 137, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 225, 76, 0, 0, 0, 4, 43, 0, 0],
            &[0, 0, 35, 182, 215, 181, 195, 254, 182, 0, 0],
            &[0, 0, 0, 0, 10, 35, 76, 153, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 194, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 223, 32, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 75, 171, 85, 0, 0],
        ]),
        // letter: 'Ě' / 0x11a
        'Ě' => Some(&[
            &[0, 0, 14, 182, 39, 1, 148, 91, 0, 0, 0],
            &[0, 0, 0, 31, 205, 154, 126, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 137, 2, 0, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 25, 0, 0],
            &[0, 55, 241, 190, 190, 190, 190, 190, 137, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 245, 209, 209, 209, 209, 209, 79, 0, 0],
            &[0, 55, 201, 17, 17, 17, 17, 17, 6, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 250, 230, 230, 230, 230, 230, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 11, 0, 0, 0, 18, 0, 0, 0],
        ]),
        // letter: 'ě' / 0x11b
        'ě' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 29, 180, 20, 0, 142, 93, 0, 0, 0],
            &[0, 0, 0, 54, 195, 142, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 152, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 46, 15, 0, 0, 0, 0],
            &[0, 0, 41, 200, 198, 167, 212, 141, 2, 0, 0],
            &[0, 17, 227, 71, 0, 0, 2, 169, 113, 0, 0],
            &[0, 122, 141, 0, 0, 0, 0, 36, 216, 0, 0],
            &[0, 188, 111, 56, 56, 56, 56, 56, 250, 4, 0],
            &[0, 209, 171, 152, 152, 152, 152, 152, 152, 9, 0],
            &[0, 192, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 137, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 225, 76, 0, 0, 0, 4, 43, 0, 0],
            &[0, 0, 35, 182, 215, 181, 195, 216, 123, 0, 0],
            &[0, 0, 0, 0, 10, 35, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 95, 237, 105, 0, 0, 0],
            &[0, 0, 0, 0, 50, 185, 17, 185, 57, 0, 0],
            &[0, 0, 0, 0, 119, 12, 0, 11, 121, 1, 0],
            &[0, 0, 0, 0, 0, 32, 70, 64, 22, 0, 0],
            &[0, 0, 0, 24, 184, 214, 160, 166, 219, 107, 0],
            &[0, 0, 8, 215, 112, 1, 0, 0, 2, 13, 0],
            &[0, 0, 117, 166, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 59, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 249, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 233, 0, 0, 0, 31, 48, 48, 38, 0],
            &[0, 30, 229, 0, 0, 0, 117, 182, 194, 206, 0],
            &[0, 11, 248, 2, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 218, 48, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 135, 152, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 16, 223, 112, 4, 0, 0, 53, 206, 0],
            &[0, 0, 0, 28, 174, 230, 196, 209, 226, 123, 0],
            &[0, 0, 0, 0, 0, 8, 35, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 24, 0, 0, 0, 0],
        ]),
        // letter: 'ĝ' / 0x11d
        'ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 181, 217, 14, 0, 0, 0, 0],
            &[0, 0, 0, 125, 122, 76, 174, 2, 0, 0, 0],
            &[0, 0, 38, 115, 0, 0, 79, 76, 0, 0, 0],
            &[0, 0, 0, 8, 46, 34, 0, 1, 7, 0, 0],
            &[0, 0, 98, 227, 182, 175, 189, 81, 162, 0, 0],
            &[0, 57, 222, 31, 0, 0, 31, 218, 162, 0, 0],
            &[0, 164, 106, 0, 0, 0, 0, 133, 162, 0, 0],
            &[0, 217, 47, 0, 0, 0, 0, 85, 162, 0, 0],
            &[0, 233, 25, 0, 0, 0, 0, 73, 162, 0, 0],
            &[0, 221, 45, 0, 0, 0, 0, 91, 162, 0, 0],
            &[0, 172, 103, 0, 0, 0, 0, 144, 162, 0, 0],
            &[0, 65, 222, 32, 0, 0, 53, 209, 162, 0, 0],
            &[0, 0, 101, 227, 191, 199, 178, 93, 162, 0, 0],
            &[0, 0, 0, 5, 36, 20, 0, 83, 159, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 123, 130, 0, 0],
            &[0, 36, 18, 0, 0, 0, 37, 227, 44, 0, 0],
            &[0, 0, 0, 13, 39, 28, 1, 0, 0, 0, 0],
        ]),
        // letter: 'Ğ' / 0x11e
        'Ğ' => Some(&[
            &[0, 0, 0, 0, 108, 2, 0, 0, 107, 0, 0],
            &[0, 0, 0, 0, 120, 148, 64, 138, 124, 0, 0],
            &[0, 0, 0, 0, 0, 77, 123, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 70, 64, 22, 0, 0],
            &[0, 0, 0, 24, 184, 214, 160, 166, 219, 107, 0],
            &[0, 0, 8, 215, 112, 1, 0, 0, 2, 13, 0],
            &[0, 0, 117, 166, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 59, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 249, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 233, 0, 0, 0, 31, 48, 48, 38, 0],
            &[0, 30, 229, 0, 0, 0, 117, 182, 194, 206, 0],
            &[0, 11, 248, 2, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 218, 48, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 135, 152, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 16, 223, 112, 4, 0, 0, 53, 206, 0],
            &[0, 0, 0, 28, 174, 230, 196, 209, 226, 123, 0],
            &[0, 0, 0, 0, 0, 8, 35, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ğ' / 0x11f
        'ğ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 67, 0, 0, 39, 49, 0, 0, 0],
            &[0, 0, 10, 200, 68, 48, 185, 40, 0, 0, 0],
            &[0, 0, 0, 31, 133, 140, 53, 0, 0, 0, 0],
            &[0, 0, 0, 8, 46, 34, 0, 1, 7, 0, 0],
            &[0, 0, 98, 227, 182, 175, 189, 81, 162, 0, 0],
            &[0, 57, 222, 31, 0, 0, 31, 218, 162, 0, 0],
            &[0, 164, 106, 0, 0, 0, 0, 133, 162, 0, 0],
            &[0, 217, 47, 0, 0, 0, 0, 85, 162, 0, 0],
            &[0, 233, 25, 0, 0, 0, 0, 73, 162, 0, 0],
            &[0, 221, 45, 0, 0, 0, 0, 91, 162, 0, 0],
            &[0, 172, 103, 0, 0, 0, 0, 144, 162, 0, 0],
            &[0, 65, 222, 32, 0, 0, 53, 209, 162, 0, 0],
            &[0, 0, 101, 227, 191, 199, 178, 93, 162, 0, 0],
            &[0, 0, 0, 5, 36, 20, 0, 83, 159, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 123, 130, 0, 0],
            &[0, 36, 18, 0, 0, 0, 37, 227, 44, 0, 0],
            &[0, 0, 0, 13, 39, 28, 1, 0, 0, 0, 0],
        ]),
        // letter: 'Ġ' / 0x120
        'Ġ' => Some(&[
            &[0, 0, 0, 0, 0, 18, 151, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 232, 44, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 70, 64, 22, 0, 0],
            &[0, 0, 0, 24, 184, 214, 160, 166, 219, 107, 0],
            &[0, 0, 8, 215, 112, 1, 0, 0, 2, 13, 0],
            &[0, 0, 117, 166, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 59, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 249, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 233, 0, 0, 0, 31, 48, 48, 38, 0],
            &[0, 30, 229, 0, 0, 0, 117, 182, 194, 206, 0],
            &[0, 11, 248, 2, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 218, 48, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 135, 152, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 16, 223, 112, 4, 0, 0, 53, 206, 0],
            &[0, 0, 0, 28, 174, 230, 196, 209, 226, 123, 0],
            &[0, 0, 0, 0, 0, 8, 35, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ġ' / 0x121
        'ġ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 114, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 46, 34, 0, 1, 7, 0, 0],
            &[0, 0, 98, 227, 182, 175, 189, 81, 162, 0, 0],
            &[0, 57, 222, 31, 0, 0, 31, 218, 162, 0, 0],
            &[0, 164, 106, 0, 0, 0, 0, 133, 162, 0, 0],
            &[0, 217, 47, 0, 0, 0, 0, 85, 162, 0, 0],
            &[0, 233, 25, 0, 0, 0, 0, 73, 162, 0, 0],
            &[0, 221, 45, 0, 0, 0, 0, 91, 162, 0, 0],
            &[0, 172, 103, 0, 0, 0, 0, 144, 162, 0, 0],
            &[0, 65, 222, 32, 0, 0, 53, 209, 162, 0, 0],
            &[0, 0, 101, 227, 191, 199, 178, 93, 162, 0, 0],
            &[0, 0, 0, 5, 36, 20, 0, 83, 159, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 123, 130, 0, 0],
            &[0, 36, 18, 0, 0, 0, 37, 227, 44, 0, 0],
            &[0, 0, 0, 13, 39, 28, 1, 0, 0, 0, 0],
        ]),
        // letter: 'Ģ' / 0x122
        'Ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 70, 64, 22, 0, 0],
            &[0, 0, 0, 24, 184, 214, 160, 166, 219, 107, 0],
            &[0, 0, 8, 215, 112, 1, 0, 0, 2, 13, 0],
            &[0, 0, 117, 166, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 59, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 249, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 233, 0, 0, 0, 31, 48, 48, 38, 0],
            &[0, 30, 229, 0, 0, 0, 117, 182, 194, 206, 0],
            &[0, 11, 248, 2, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 218, 48, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 135, 152, 0, 0, 0, 0, 41, 206, 0],
            &[0, 0, 16, 223, 112, 4, 0, 0, 53, 206, 0],
            &[0, 0, 0, 28, 174, 230, 196, 209, 226, 123, 0],
            &[0, 0, 0, 0, 0, 8, 35, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 212, 25, 0, 0, 0],
            &[0, 0, 0, 0, 0, 30, 185, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 50, 0, 0, 0, 0],
        ]),
        // letter: 'ģ' / 0x123
        'ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 171, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 84, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 46, 34, 0, 1, 7, 0, 0],
            &[0, 0, 98, 227, 182, 175, 189, 81, 162, 0, 0],
            &[0, 57, 222, 31, 0, 0, 31, 218, 162, 0, 0],
            &[0, 164, 106, 0, 0, 0, 0, 133, 162, 0, 0],
            &[0, 217, 47, 0, 0, 0, 0, 85, 162, 0, 0],
            &[0, 233, 25, 0, 0, 0, 0, 73, 162, 0, 0],
            &[0, 221, 45, 0, 0, 0, 0, 91, 162, 0, 0],
            &[0, 172, 103, 0, 0, 0, 0, 144, 162, 0, 0],
            &[0, 65, 222, 32, 0, 0, 53, 209, 162, 0, 0],
            &[0, 0, 101, 227, 191, 199, 178, 93, 162, 0, 0],
            &[0, 0, 0, 5, 36, 20, 0, 83, 159, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 123, 130, 0, 0],
            &[0, 36, 18, 0, 0, 0, 37, 227, 44, 0, 0],
            &[0, 0, 0, 13, 39, 28, 1, 0, 0, 0, 0],
        ]),
        // letter: 'Ĥ' / 0x124
        'Ĥ' => Some(&[
            &[0, 0, 0, 3, 192, 219, 23, 0, 0, 0, 0],
            &[0, 0, 0, 144, 101, 59, 184, 6, 0, 0, 0],
            &[0, 0, 38, 93, 0, 0, 60, 72, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 6, 28, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 227, 209, 209, 209, 209, 217, 206, 0, 0],
            &[0, 150, 114, 21, 21, 21, 21, 63, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 15, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĥ' / 0x125
        'ĥ' => Some(&[
            &[0, 110, 140, 111, 144, 0, 0, 0, 0, 0, 0],
            &[45, 137, 1, 0, 115, 70, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 98, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 145, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 145, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 145, 0, 8, 43, 19, 0, 0, 0],
            &[0, 0, 107, 143, 121, 194, 168, 219, 163, 3, 0],
            &[0, 0, 107, 211, 102, 0, 0, 5, 190, 105, 0],
            &[0, 0, 107, 209, 0, 0, 0, 0, 92, 164, 0],
            &[0, 0, 107, 157, 0, 0, 0, 0, 68, 184, 0],
            &[0, 0, 107, 145, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 107, 145, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 107, 145, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 107, 145, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 107, 145, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 160, 186, 4, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ħ' / 0x126
        'Ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 6, 28, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[193, 255, 255, 193, 193, 193, 193, 240, 255, 193, 38],
            &[19, 169, 120, 19, 19, 19, 19, 65, 225, 19, 3],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 227, 209, 209, 209, 209, 217, 206, 0, 0],
            &[0, 150, 114, 21, 21, 21, 21, 63, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ħ' / 0x127
        'ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 98, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 69, 176, 214, 69, 69, 58, 0, 0, 0, 0],
            &[0, 130, 237, 255, 130, 130, 110, 0, 0, 0, 0],
            &[0, 0, 107, 145, 0, 8, 43, 19, 0, 0, 0],
            &[0, 0, 107, 143, 121, 194, 168, 219, 163, 3, 0],
            &[0, 0, 107, 211, 102, 0, 0, 5, 190, 105, 0],
            &[0, 0, 107, 209, 0, 0, 0, 0, 92, 164, 0],
            &[0, 0, 107, 157, 0, 0, 0, 0, 68, 184, 0],
            &[0, 0, 107, 145, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 107, 145, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 107, 145, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 107, 145, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 107, 145, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĩ' / 0x128
        'Ĩ' => Some(&[
            &[0, 0, 2, 83, 62, 0, 0, 76, 0, 0, 0],
            &[0, 0, 129, 143, 161, 169, 100, 178, 0, 0, 0],
            &[0, 0, 88, 0, 0, 59, 113, 21, 0, 0, 0],
            &[0, 0, 24, 35, 35, 35, 35, 31, 0, 0, 0],
            &[0, 0, 104, 161, 205, 222, 164, 135, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 129, 196, 227, 237, 199, 167, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĩ' / 0x129
        'ĩ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 59, 40, 0, 0, 58, 0, 0, 0],
            &[0, 0, 115, 165, 182, 146, 76, 183, 0, 0, 0],
            &[0, 0, 105, 2, 0, 82, 137, 33, 0, 0, 0],
            &[0, 2, 11, 11, 11, 10, 0, 0, 0, 0, 0],
            &[0, 30, 175, 188, 205, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 197, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ī' / 0x12a
        'Ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 141, 141, 141, 141, 63, 0, 0, 0],
            &[0, 0, 16, 67, 67, 67, 67, 30, 0, 0, 0],
            &[0, 0, 24, 35, 35, 35, 35, 31, 0, 0, 0],
            &[0, 0, 104, 161, 205, 222, 164, 135, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 129, 196, 227, 237, 199, 167, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ī' / 0x12b
        'ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 117, 117, 117, 117, 52, 0, 0, 0],
            &[0, 0, 22, 91, 91, 91, 91, 40, 0, 0, 0],
            &[0, 2, 11, 11, 11, 10, 0, 0, 0, 0, 0],
            &[0, 30, 175, 188, 205, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 197, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => Some(&[
            &[0, 0, 28, 83, 0, 0, 49, 57, 0, 0, 0],
            &[0, 0, 7, 198, 92, 72, 193, 32, 0, 0, 0],
            &[0, 0, 0, 18, 109, 116, 34, 0, 0, 0, 0],
            &[0, 0, 24, 35, 35, 35, 35, 31, 0, 0, 0],
            &[0, 0, 104, 161, 205, 222, 164, 135, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 129, 196, 227, 237, 199, 167, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĭ' / 0x12d
        'ĭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 67, 0, 0, 39, 49, 0, 0, 0],
            &[0, 0, 10, 200, 68, 48, 185, 40, 0, 0, 0],
            &[0, 0, 0, 31, 133, 140, 53, 0, 0, 0, 0],
            &[0, 2, 11, 11, 11, 10, 0, 0, 0, 0, 0],
            &[0, 30, 175, 188, 205, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 197, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Į' / 0x12e
        'Į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 24, 35, 35, 35, 35, 31, 0, 0, 0],
            &[0, 0, 104, 161, 205, 222, 164, 135, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 129, 196, 227, 237, 220, 169, 0, 0, 0],
            &[0, 0, 0, 0, 0, 53, 160, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 190, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 226, 26, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 82, 172, 82, 0, 0, 0],
        ]),
        // letter: 'į' / 0x12f
        'į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 223, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 11, 11, 11, 10, 0, 0, 0, 0, 0],
            &[0, 30, 175, 188, 205, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 220, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 140, 78, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 144, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 154, 11, 2, 0, 0],
            &[0, 0, 0, 0, 0, 9, 143, 170, 13, 0, 0],
        ]),
        // letter: 'İ' / 0x130
        'İ' => Some(&[
            &[0, 0, 0, 0, 78, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 24, 35, 35, 35, 35, 31, 0, 0, 0],
            &[0, 0, 104, 161, 205, 222, 164, 135, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 129, 196, 227, 237, 199, 167, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ı' / 0x131
        'ı' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 11, 11, 11, 10, 0, 0, 0, 0, 0],
            &[0, 30, 175, 188, 205, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 197, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĳ' / 0x132
        'Ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 29, 0, 0, 0, 0, 21, 13, 0, 0],
            &[0, 41, 210, 0, 0, 0, 0, 154, 97, 0, 0],
            &[0, 41, 210, 0, 0, 0, 0, 154, 97, 0, 0],
            &[0, 41, 210, 0, 0, 0, 0, 154, 97, 0, 0],
            &[0, 41, 210, 0, 0, 0, 0, 154, 97, 0, 0],
            &[0, 41, 210, 0, 0, 0, 0, 154, 97, 0, 0],
            &[0, 41, 210, 0, 0, 0, 0, 154, 97, 0, 0],
            &[0, 41, 210, 0, 0, 0, 0, 154, 97, 0, 0],
            &[0, 41, 210, 0, 0, 0, 0, 154, 97, 0, 0],
            &[0, 41, 210, 0, 0, 0, 0, 159, 94, 0, 0],
            &[0, 1, 8, 0, 0, 0, 0, 194, 69, 0, 0],
            &[0, 8, 30, 0, 0, 0, 74, 231, 9, 0, 0],
            &[0, 15, 213, 214, 185, 205, 220, 59, 0, 0, 0],
            &[0, 0, 0, 21, 46, 33, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĳ' / 0x133
        'ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 51, 0, 0, 21, 154, 20, 0, 0],
            &[0, 4, 220, 90, 0, 0, 42, 231, 41, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 2, 0, 0, 0, 10, 0, 0, 0],
            &[0, 0, 202, 50, 0, 0, 0, 241, 10, 0, 0],
            &[0, 0, 202, 50, 0, 0, 0, 241, 10, 0, 0],
            &[0, 0, 202, 50, 0, 0, 0, 241, 10, 0, 0],
            &[0, 0, 202, 50, 0, 0, 0, 241, 10, 0, 0],
            &[0, 0, 202, 50, 0, 0, 0, 241, 10, 0, 0],
            &[0, 0, 202, 50, 0, 0, 0, 241, 10, 0, 0],
            &[0, 0, 202, 50, 0, 0, 0, 241, 10, 0, 0],
            &[0, 0, 202, 50, 0, 0, 0, 241, 10, 0, 0],
            &[0, 0, 202, 50, 0, 0, 0, 241, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 241, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 246, 3, 0, 0],
            &[0, 0, 5, 0, 0, 0, 104, 191, 0, 0, 0],
            &[0, 0, 0, 17, 39, 18, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĵ' / 0x134
        'Ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 7, 201, 213, 15, 0, 0],
            &[0, 0, 0, 0, 0, 157, 87, 71, 176, 3, 0],
            &[0, 0, 0, 0, 47, 84, 0, 0, 70, 62, 0],
            &[0, 0, 0, 0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 135, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 135, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 135, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 135, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 135, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 135, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 135, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 135, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 120, 133, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 150, 110, 0, 0, 0],
            &[0, 11, 27, 0, 0, 28, 233, 40, 0, 0, 0],
            &[0, 21, 213, 209, 187, 233, 110, 0, 0, 0, 0],
            &[0, 0, 0, 26, 44, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 14, 0, 0, 0],
        ]),
        // letter: 'ĵ' / 0x135
        'ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 233, 149, 0, 0, 0, 0],
            &[0, 0, 0, 10, 190, 48, 160, 91, 0, 0, 0],
            &[0, 0, 0, 100, 53, 0, 3, 132, 20, 0, 0],
            &[0, 0, 7, 11, 11, 11, 4, 0, 0, 0, 0],
            &[0, 0, 114, 181, 194, 233, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 147, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 147, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 147, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 147, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 147, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 147, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 147, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 147, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 147, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 30, 232, 38, 0, 0, 0, 0],
            &[0, 1, 26, 37, 6, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ķ' / 0x136
        'Ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 3, 35, 3, 0],
            &[0, 150, 101, 0, 0, 0, 0, 152, 154, 0, 0],
            &[0, 150, 101, 0, 0, 0, 119, 187, 4, 0, 0],
            &[0, 150, 101, 0, 0, 87, 212, 15, 0, 0, 0],
            &[0, 150, 101, 0, 60, 226, 33, 0, 0, 0, 0],
            &[0, 150, 101, 38, 228, 58, 0, 0, 0, 0, 0],
            &[0, 150, 115, 211, 220, 11, 0, 0, 0, 0, 0],
            &[0, 150, 234, 87, 136, 147, 0, 0, 0, 0, 0],
            &[0, 150, 131, 0, 7, 212, 64, 0, 0, 0, 0],
            &[0, 150, 101, 0, 0, 53, 218, 12, 0, 0, 0],
            &[0, 150, 101, 0, 0, 0, 135, 150, 0, 0, 0],
            &[0, 150, 101, 0, 0, 0, 7, 211, 67, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 52, 220, 13, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 46, 191, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 102, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 5, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ķ' / 0x137
        'ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 214, 18, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 232, 19, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 232, 19, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 232, 19, 0, 0, 0, 0, 11, 1, 0, 0],
            &[0, 232, 19, 0, 0, 0, 135, 171, 3, 0, 0],
            &[0, 232, 19, 0, 0, 123, 182, 5, 0, 0, 0],
            &[0, 232, 19, 0, 111, 192, 9, 0, 0, 0, 0],
            &[0, 232, 15, 99, 217, 13, 0, 0, 0, 0, 0],
            &[0, 232, 93, 186, 214, 54, 0, 0, 0, 0, 0],
            &[0, 232, 174, 8, 53, 216, 14, 0, 0, 0, 0],
            &[0, 232, 22, 0, 0, 120, 167, 0, 0, 0, 0],
            &[0, 232, 19, 0, 0, 2, 189, 99, 0, 0, 0],
            &[0, 232, 19, 0, 0, 0, 27, 225, 41, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 195, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 71, 60, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĸ' / 0x138
        'ĸ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 0, 0, 0, 0, 0, 11, 1, 0, 0],
            &[0, 232, 19, 0, 0, 0, 135, 171, 3, 0, 0],
            &[0, 232, 19, 0, 0, 123, 182, 5, 0, 0, 0],
            &[0, 232, 19, 0, 111, 192, 9, 0, 0, 0, 0],
            &[0, 232, 15, 99, 217, 13, 0, 0, 0, 0, 0],
            &[0, 232, 93, 186, 214, 54, 0, 0, 0, 0, 0],
            &[0, 232, 174, 8, 53, 216, 14, 0, 0, 0, 0],
            &[0, 232, 22, 0, 0, 120, 167, 0, 0, 0, 0],
            &[0, 232, 19, 0, 0, 2, 189, 99, 0, 0, 0],
            &[0, 232, 19, 0, 0, 0, 27, 225, 41, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĺ' / 0x139
        'Ĺ' => Some(&[
            &[0, 0, 0, 104, 193, 5, 0, 0, 0, 0, 0],
            &[0, 0, 51, 198, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 240, 234, 234, 234, 234, 169, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 4, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĺ' / 0x13a
        'ĺ' => Some(&[
            &[0, 0, 0, 0, 0, 164, 109, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 105, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 187, 199, 213, 211, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 197, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 203, 59, 0, 0, 0],
        ]),
        // letter: 'Ļ' / 0x13b
        'Ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 240, 234, 234, 234, 234, 169, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 188, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 53, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ļ' / 0x13c
        'ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 187, 199, 213, 211, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 197, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 204, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 115, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 121, 9, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ľ' / 0x13d
        'Ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 8, 0, 0, 0, 21, 13, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 185, 56, 0, 0],
            &[0, 0, 189, 63, 0, 0, 5, 198, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 33, 64, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 240, 234, 234, 234, 234, 169, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ľ' / 0x13e
        'ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 187, 199, 213, 211, 0, 8, 216, 2, 0],
            &[0, 0, 0, 0, 22, 230, 0, 64, 150, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 106, 32, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 197, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 3, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 1, 232, 100, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 134, 49, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 189, 240, 234, 234, 234, 234, 169, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŀ' / 0x140
        'ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[41, 187, 199, 213, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 230, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 230, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 230, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 230, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 230, 0, 0, 0, 52, 118, 0],
            &[0, 0, 0, 22, 230, 0, 0, 0, 115, 227, 0],
            &[0, 0, 0, 22, 230, 0, 0, 0, 0, 8, 0],
            &[0, 0, 0, 22, 230, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 230, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 230, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 230, 0, 0, 0, 0, 0, 0],
            &[85, 189, 201, 216, 251, 210, 197, 185, 16, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ł' / 0x141
        'Ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 237, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 237, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 237, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 237, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 237, 0, 53, 92, 0, 0, 0, 0],
            &[0, 0, 15, 253, 150, 193, 45, 0, 0, 0, 0],
            &[0, 0, 106, 255, 97, 1, 0, 0, 0, 0, 0],
            &[0, 133, 169, 255, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 15, 237, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 237, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 237, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 254, 234, 234, 234, 234, 234, 94, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ł' / 0x142
        'ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 187, 199, 213, 211, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 5, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 98, 201, 5, 0, 0],
            &[0, 0, 0, 0, 60, 255, 147, 13, 0, 0, 0],
            &[0, 0, 5, 124, 224, 255, 0, 0, 0, 0, 0],
            &[0, 0, 34, 119, 27, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 0, 0, 0, 0, 0],
            &[0, 85, 189, 201, 216, 251, 210, 197, 185, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ń' / 0x143
        'Ń' => Some(&[
            &[0, 0, 0, 0, 0, 19, 224, 59, 0, 0, 0],
            &[0, 0, 0, 0, 3, 180, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 61, 72, 0, 0, 0, 0, 0],
            &[0, 20, 24, 0, 0, 0, 0, 5, 28, 0, 0],
            &[0, 150, 235, 11, 0, 0, 0, 41, 206, 0, 0],
            &[0, 150, 193, 116, 0, 0, 0, 41, 206, 0, 0],
            &[0, 150, 98, 216, 9, 0, 0, 41, 206, 0, 0],
            &[0, 150, 90, 129, 111, 0, 0, 41, 206, 0, 0],
            &[0, 150, 92, 18, 216, 7, 0, 41, 206, 0, 0],
            &[0, 150, 92, 0, 135, 106, 0, 41, 206, 0, 0],
            &[0, 150, 92, 0, 21, 215, 6, 41, 206, 0, 0],
            &[0, 150, 92, 0, 0, 141, 101, 41, 206, 0, 0],
            &[0, 150, 92, 0, 0, 24, 213, 42, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 146, 127, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 28, 227, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 0, 152, 206, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 17, 17, 0, 0, 0],
        ]),
        // letter: 'ń' / 0x144
        'ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 205, 93, 0, 0, 0],
            &[0, 0, 0, 0, 0, 147, 122, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 107, 0, 0, 0, 0, 0],
            &[0, 4, 4, 0, 9, 43, 21, 0, 0, 0, 0],
            &[0, 107, 109, 131, 197, 168, 216, 170, 5, 0, 0],
            &[0, 107, 215, 100, 0, 0, 3, 181, 111, 0, 0],
            &[0, 107, 209, 0, 0, 0, 0, 89, 167, 0, 0],
            &[0, 107, 163, 0, 0, 0, 0, 68, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ņ' / 0x145
        'Ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 24, 0, 0, 0, 0, 5, 28, 0, 0],
            &[0, 150, 235, 11, 0, 0, 0, 41, 206, 0, 0],
            &[0, 150, 193, 116, 0, 0, 0, 41, 206, 0, 0],
            &[0, 150, 98, 216, 9, 0, 0, 41, 206, 0, 0],
            &[0, 150, 90, 129, 111, 0, 0, 41, 206, 0, 0],
            &[0, 150, 92, 18, 216, 7, 0, 41, 206, 0, 0],
            &[0, 150, 92, 0, 135, 106, 0, 41, 206, 0, 0],
            &[0, 150, 92, 0, 21, 215, 6, 41, 206, 0, 0],
            &[0, 150, 92, 0, 0, 141, 101, 41, 206, 0, 0],
            &[0, 150, 92, 0, 0, 24, 213, 42, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 146, 127, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 28, 227, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 0, 152, 206, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 109, 128, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 176, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 124, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ņ' / 0x146
        'ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 4, 0, 9, 43, 21, 0, 0, 0, 0],
            &[0, 107, 109, 131, 197, 168, 216, 170, 5, 0, 0],
            &[0, 107, 215, 100, 0, 0, 3, 181, 111, 0, 0],
            &[0, 107, 209, 0, 0, 0, 0, 89, 167, 0, 0],
            &[0, 107, 163, 0, 0, 0, 0, 68, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 75, 162, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 143, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ň' / 0x147
        'Ň' => Some(&[
            &[0, 0, 19, 184, 32, 2, 158, 80, 0, 0, 0],
            &[0, 0, 0, 38, 204, 161, 113, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 130, 0, 0, 0, 0, 0],
            &[0, 20, 24, 0, 0, 0, 0, 5, 28, 0, 0],
            &[0, 150, 235, 11, 0, 0, 0, 41, 206, 0, 0],
            &[0, 150, 193, 116, 0, 0, 0, 41, 206, 0, 0],
            &[0, 150, 98, 216, 9, 0, 0, 41, 206, 0, 0],
            &[0, 150, 90, 129, 111, 0, 0, 41, 206, 0, 0],
            &[0, 150, 92, 18, 216, 7, 0, 41, 206, 0, 0],
            &[0, 150, 92, 0, 135, 106, 0, 41, 206, 0, 0],
            &[0, 150, 92, 0, 21, 215, 6, 41, 206, 0, 0],
            &[0, 150, 92, 0, 0, 141, 101, 41, 206, 0, 0],
            &[0, 150, 92, 0, 0, 24, 213, 42, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 146, 127, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 28, 227, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 0, 152, 206, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 10, 0, 0, 1, 17, 0, 0, 0],
        ]),
        // letter: 'ň' / 0x148
        'ň' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 182, 31, 0, 118, 118, 0, 0, 0],
            &[0, 0, 0, 37, 202, 126, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 164, 7, 0, 0, 0, 0],
            &[0, 4, 4, 0, 9, 43, 21, 0, 0, 0, 0],
            &[0, 107, 109, 131, 197, 168, 216, 170, 5, 0, 0],
            &[0, 107, 215, 100, 0, 0, 3, 181, 111, 0, 0],
            &[0, 107, 209, 0, 0, 0, 0, 89, 167, 0, 0],
            &[0, 107, 163, 0, 0, 0, 0, 68, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŉ' / 0x149
        'ŉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 20, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 154, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 201, 67, 0, 0, 0, 0, 0, 0, 0, 0],
            &[10, 227, 5, 7, 0, 4, 39, 25, 0, 0, 0],
            &[75, 135, 28, 187, 77, 199, 170, 212, 180, 8, 0],
            &[3, 2, 28, 229, 153, 4, 0, 2, 178, 114, 0],
            &[0, 0, 28, 255, 30, 0, 0, 0, 88, 168, 0],
            &[0, 0, 28, 240, 0, 0, 0, 0, 68, 184, 0],
            &[0, 0, 28, 223, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 28, 223, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 28, 223, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 28, 223, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 28, 223, 0, 0, 0, 0, 67, 184, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 24, 0, 0, 0, 0, 5, 28, 0, 0],
            &[0, 150, 235, 11, 0, 0, 0, 41, 206, 0, 0],
            &[0, 150, 193, 116, 0, 0, 0, 41, 206, 0, 0],
            &[0, 150, 98, 216, 9, 0, 0, 41, 206, 0, 0],
            &[0, 150, 90, 129, 111, 0, 0, 41, 206, 0, 0],
            &[0, 150, 92, 18, 216, 7, 0, 41, 206, 0, 0],
            &[0, 150, 92, 0, 135, 106, 0, 41, 206, 0, 0],
            &[0, 150, 92, 0, 21, 215, 6, 41, 206, 0, 0],
            &[0, 150, 92, 0, 0, 141, 101, 41, 206, 0, 0],
            &[0, 150, 92, 0, 0, 24, 213, 42, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 146, 127, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 28, 227, 206, 0, 0],
            &[0, 150, 92, 0, 0, 0, 0, 185, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 54, 185, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 167, 122, 0, 0],
            &[0, 0, 0, 0, 88, 215, 233, 165, 9, 0, 0],
            &[0, 0, 0, 0, 0, 13, 5, 0, 0, 0, 0],
        ]),
        // letter: 'ŋ' / 0x14b
        'ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 4, 0, 9, 43, 21, 0, 0, 0, 0],
            &[0, 107, 109, 131, 197, 168, 216, 170, 5, 0, 0],
            &[0, 107, 215, 100, 0, 0, 3, 181, 111, 0, 0],
            &[0, 107, 209, 0, 0, 0, 0, 89, 167, 0, 0],
            &[0, 107, 163, 0, 0, 0, 0, 68, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 106, 255, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 135, 255, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 135, 255, 0, 0],
            &[0, 107, 145, 0, 0, 0, 0, 135, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 67, 184, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 71, 182, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 121, 147, 0, 0],
            &[0, 0, 0, 0, 0, 1, 11, 0, 0, 0, 0],
        ]),
        // letter: 'Ō' / 0x14c
        'Ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 141, 141, 141, 141, 63, 0, 0, 0],
            &[0, 0, 16, 67, 67, 67, 67, 30, 0, 0, 0],
            &[0, 0, 0, 10, 59, 72, 23, 0, 0, 0, 0],
            &[0, 0, 75, 223, 169, 154, 220, 128, 0, 0, 0],
            &[0, 38, 227, 37, 0, 0, 10, 195, 91, 0, 0],
            &[0, 155, 112, 0, 0, 0, 0, 51, 212, 0, 0],
            &[0, 226, 29, 0, 0, 0, 0, 0, 229, 28, 0],
            &[14, 241, 0, 0, 0, 0, 0, 0, 188, 68, 0],
            &[31, 224, 0, 0, 0, 0, 0, 0, 171, 84, 0],
            &[31, 226, 0, 0, 0, 0, 0, 0, 175, 82, 0],
            &[11, 244, 1, 0, 0, 0, 0, 0, 194, 61, 0],
            &[0, 220, 40, 0, 0, 0, 0, 2, 237, 18, 0],
            &[0, 140, 132, 0, 0, 0, 0, 74, 193, 0, 0],
            &[0, 26, 229, 70, 0, 0, 34, 220, 64, 0, 0],
            &[0, 0, 51, 209, 213, 202, 222, 86, 0, 0, 0],
            &[0, 0, 0, 0, 25, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ō' / 0x14d
        'ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 117, 117, 117, 117, 52, 0, 0, 0],
            &[0, 0, 22, 91, 91, 91, 91, 40, 0, 0, 0],
            &[0, 0, 0, 2, 35, 42, 8, 0, 0, 0, 0],
            &[0, 0, 71, 213, 189, 171, 217, 115, 0, 0, 0],
            &[0, 45, 225, 43, 0, 0, 10, 192, 95, 0, 0],
            &[0, 158, 110, 0, 0, 0, 0, 48, 213, 0, 0],
            &[0, 215, 45, 0, 0, 0, 0, 0, 240, 15, 0],
            &[0, 232, 20, 0, 0, 0, 0, 0, 224, 32, 0],
            &[0, 215, 39, 0, 0, 0, 0, 1, 241, 15, 0],
            &[0, 156, 108, 0, 0, 0, 0, 57, 209, 0, 0],
            &[0, 41, 224, 40, 0, 0, 17, 203, 87, 0, 0],
            &[0, 0, 65, 211, 190, 183, 221, 104, 0, 0, 0],
            &[0, 0, 0, 0, 27, 34, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => Some(&[
            &[0, 0, 28, 83, 0, 0, 49, 57, 0, 0, 0],
            &[0, 0, 7, 198, 92, 72, 193, 32, 0, 0, 0],
            &[0, 0, 0, 18, 109, 116, 34, 0, 0, 0, 0],
            &[0, 0, 0, 10, 59, 72, 23, 0, 0, 0, 0],
            &[0, 0, 75, 223, 169, 154, 220, 128, 0, 0, 0],
            &[0, 38, 227, 37, 0, 0, 10, 195, 91, 0, 0],
            &[0, 155, 112, 0, 0, 0, 0, 51, 212, 0, 0],
            &[0, 226, 29, 0, 0, 0, 0, 0, 229, 28, 0],
            &[14, 241, 0, 0, 0, 0, 0, 0, 188, 68, 0],
            &[31, 224, 0, 0, 0, 0, 0, 0, 171, 84, 0],
            &[31, 226, 0, 0, 0, 0, 0, 0, 175, 82, 0],
            &[11, 244, 1, 0, 0, 0, 0, 0, 194, 61, 0],
            &[0, 220, 40, 0, 0, 0, 0, 2, 237, 18, 0],
            &[0, 140, 132, 0, 0, 0, 0, 74, 193, 0, 0],
            &[0, 26, 229, 70, 0, 0, 34, 220, 64, 0, 0],
            &[0, 0, 51, 209, 213, 202, 222, 86, 0, 0, 0],
            &[0, 0, 0, 0, 25, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŏ' / 0x14f
        'ŏ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 67, 0, 0, 39, 49, 0, 0, 0],
            &[0, 0, 10, 200, 68, 48, 185, 40, 0, 0, 0],
            &[0, 0, 0, 31, 133, 140, 53, 0, 0, 0, 0],
            &[0, 0, 0, 2, 35, 42, 8, 0, 0, 0, 0],
            &[0, 0, 71, 213, 189, 171, 217, 115, 0, 0, 0],
            &[0, 45, 225, 43, 0, 0, 10, 192, 95, 0, 0],
            &[0, 158, 110, 0, 0, 0, 0, 48, 213, 0, 0],
            &[0, 215, 45, 0, 0, 0, 0, 0, 240, 15, 0],
            &[0, 232, 20, 0, 0, 0, 0, 0, 224, 32, 0],
            &[0, 215, 39, 0, 0, 0, 0, 1, 241, 15, 0],
            &[0, 156, 108, 0, 0, 0, 0, 57, 209, 0, 0],
            &[0, 41, 224, 40, 0, 0, 17, 203, 87, 0, 0],
            &[0, 0, 65, 211, 190, 183, 221, 104, 0, 0, 0],
            &[0, 0, 0, 0, 27, 34, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ő' / 0x150
        'Ő' => Some(&[
            &[0, 0, 0, 13, 223, 41, 69, 202, 4, 0, 0],
            &[0, 0, 0, 151, 90, 11, 198, 31, 0, 0, 0],
            &[0, 0, 24, 96, 0, 72, 47, 0, 0, 0, 0],
            &[0, 0, 0, 10, 59, 72, 23, 0, 0, 0, 0],
            &[0, 0, 75, 223, 169, 154, 220, 128, 0, 0, 0],
            &[0, 38, 227, 37, 0, 0, 10, 195, 91, 0, 0],
            &[0, 155, 112, 0, 0, 0, 0, 51, 212, 0, 0],
            &[0, 226, 29, 0, 0, 0, 0, 0, 229, 28, 0],
            &[14, 241, 0, 0, 0, 0, 0, 0, 188, 68, 0],
            &[31, 224, 0, 0, 0, 0, 0, 0, 171, 84, 0],
            &[31, 226, 0, 0, 0, 0, 0, 0, 175, 82, 0],
            &[11, 244, 1, 0, 0, 0, 0, 0, 194, 61, 0],
            &[0, 220, 40, 0, 0, 0, 0, 2, 237, 18, 0],
            &[0, 140, 132, 0, 0, 0, 0, 74, 193, 0, 0],
            &[0, 26, 229, 70, 0, 0, 34, 220, 64, 0, 0],
            &[0, 0, 51, 209, 213, 202, 222, 86, 0, 0, 0],
            &[0, 0, 0, 0, 25, 31, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 13, 0, 28, 4, 0, 0],
        ]),
        // letter: 'ő' / 0x151
        'ő' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 2, 0, 5, 0, 0, 0],
            &[0, 0, 0, 8, 219, 52, 56, 214, 7, 0, 0],
            &[0, 0, 0, 135, 110, 6, 195, 43, 0, 0, 0],
            &[0, 0, 24, 116, 0, 77, 63, 0, 0, 0, 0],
            &[0, 0, 0, 2, 35, 42, 8, 0, 0, 0, 0],
            &[0, 0, 71, 213, 189, 171, 217, 115, 0, 0, 0],
            &[0, 45, 225, 43, 0, 0, 10, 192, 95, 0, 0],
            &[0, 158, 110, 0, 0, 0, 0, 48, 213, 0, 0],
            &[0, 215, 45, 0, 0, 0, 0, 0, 240, 15, 0],
            &[0, 232, 20, 0, 0, 0, 0, 0, 224, 32, 0],
            &[0, 215, 39, 0, 0, 0, 0, 1, 241, 15, 0],
            &[0, 156, 108, 0, 0, 0, 0, 57, 209, 0, 0],
            &[0, 41, 224, 40, 0, 0, 17, 203, 87, 0, 0],
            &[0, 0, 65, 211, 190, 183, 221, 104, 0, 0, 0],
            &[0, 0, 0, 0, 27, 34, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Œ' / 0x152
        'Œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 49, 45, 35, 35, 35, 23, 0],
            &[0, 0, 129, 218, 172, 242, 202, 190, 190, 128, 0],
            &[0, 89, 195, 8, 0, 204, 43, 0, 0, 0, 0],
            &[0, 197, 72, 0, 0, 204, 43, 0, 0, 0, 0],
            &[5, 246, 11, 0, 0, 204, 43, 0, 0, 0, 0],
            &[31, 231, 0, 0, 0, 204, 43, 0, 0, 0, 0],
            &[43, 215, 0, 0, 0, 204, 217, 209, 209, 83, 0],
            &[38, 216, 0, 0, 0, 204, 57, 17, 17, 6, 0],
            &[23, 234, 0, 0, 0, 204, 43, 0, 0, 0, 0],
            &[0, 238, 19, 0, 0, 204, 43, 0, 0, 0, 0],
            &[0, 173, 98, 0, 0, 204, 43, 0, 0, 0, 0],
            &[0, 58, 226, 36, 0, 204, 43, 0, 0, 0, 0],
            &[0, 0, 88, 224, 214, 250, 234, 230, 230, 155, 0],
            &[0, 0, 0, 0, 17, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'œ' / 0x153
        'œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 37, 0, 0, 2, 39, 4, 0, 0],
            &[0, 94, 209, 174, 151, 15, 191, 164, 191, 4, 0],
            &[10, 216, 12, 0, 157, 161, 74, 0, 155, 78, 0],
            &[78, 151, 0, 0, 82, 243, 2, 0, 86, 136, 0],
            &[108, 117, 0, 0, 58, 223, 51, 51, 104, 158, 0],
            &[125, 105, 0, 0, 44, 234, 144, 144, 144, 92, 0],
            &[114, 118, 0, 0, 61, 216, 0, 0, 0, 0, 0],
            &[76, 155, 0, 0, 87, 248, 6, 0, 0, 0, 0],
            &[12, 219, 17, 0, 169, 151, 97, 0, 0, 12, 0],
            &[0, 93, 216, 192, 130, 9, 192, 172, 192, 79, 0],
            &[0, 0, 15, 24, 0, 0, 1, 31, 11, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŕ' / 0x154
        'Ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 30, 228, 44, 0, 0, 0],
            &[0, 0, 0, 0, 8, 192, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 74, 59, 0, 0, 0, 0, 0],
            &[0, 7, 35, 35, 33, 19, 0, 0, 0, 0, 0],
            &[0, 55, 240, 186, 189, 211, 225, 122, 1, 0, 0],
            &[0, 55, 197, 0, 0, 0, 26, 202, 115, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 72, 202, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 46, 215, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 106, 167, 0, 0],
            &[0, 55, 202, 22, 22, 37, 108, 217, 36, 0, 0],
            &[0, 55, 244, 204, 204, 229, 168, 11, 0, 0, 0],
            &[0, 55, 197, 0, 0, 38, 223, 13, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 137, 140, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 12, 224, 42, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 90, 189, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 194, 87, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 15, 0, 0, 0],
        ]),
        // letter: 'ŕ' / 0x155
        'ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 68, 219, 16, 0, 0],
            &[0, 0, 0, 0, 0, 27, 206, 35, 0, 0, 0],
            &[0, 0, 0, 0, 0, 121, 35, 0, 0, 0, 0],
            &[0, 3, 11, 11, 11, 0, 0, 33, 37, 3, 0],
            &[0, 56, 179, 194, 243, 44, 191, 194, 192, 133, 0],
            &[0, 0, 0, 0, 191, 187, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 61, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 60, 0, 0, 0, 0, 0],
            &[0, 113, 193, 207, 245, 222, 201, 188, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŗ' / 0x156
        'Ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 35, 35, 33, 19, 0, 0, 0, 0, 0],
            &[0, 55, 240, 186, 189, 211, 225, 122, 1, 0, 0],
            &[0, 55, 197, 0, 0, 0, 26, 202, 115, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 72, 202, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 46, 215, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 106, 167, 0, 0],
            &[0, 55, 202, 22, 22, 37, 108, 217, 36, 0, 0],
            &[0, 55, 244, 204, 204, 229, 168, 11, 0, 0, 0],
            &[0, 55, 197, 0, 0, 38, 223, 13, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 137, 140, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 12, 224, 42, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 90, 189, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 194, 87, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 188, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 53, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŗ' / 0x157
        'ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 11, 11, 11, 0, 0, 33, 37, 3, 0],
            &[0, 56, 179, 194, 243, 44, 191, 194, 192, 133, 0],
            &[0, 0, 0, 0, 191, 187, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 61, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 60, 0, 0, 0, 0, 0],
            &[0, 113, 193, 207, 245, 222, 201, 188, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 197, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 198, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 62, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ř' / 0x158
        'Ř' => Some(&[
            &[0, 0, 30, 184, 22, 7, 171, 63, 0, 0, 0],
            &[0, 0, 0, 53, 198, 174, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 118, 0, 0, 0, 0, 0],
            &[0, 7, 35, 35, 33, 19, 0, 0, 0, 0, 0],
            &[0, 55, 240, 186, 189, 211, 225, 122, 1, 0, 0],
            &[0, 55, 197, 0, 0, 0, 26, 202, 115, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 72, 202, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 46, 215, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 106, 167, 0, 0],
            &[0, 55, 202, 22, 22, 37, 108, 217, 36, 0, 0],
            &[0, 55, 244, 204, 204, 229, 168, 11, 0, 0, 0],
            &[0, 55, 197, 0, 0, 38, 223, 13, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 137, 140, 0, 0, 0],
            &[0, 55, 197, 0, 0, 0, 12, 224, 42, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 90, 189, 0, 0],
            &[0, 55, 197, 0, 0, 0, 0, 0, 194, 87, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 8, 0, 0, 3, 15, 0, 0, 0],
        ]),
        // letter: 'ř' / 0x159
        'ř' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 92, 138, 0, 22, 183, 31, 0, 0],
            &[0, 0, 0, 0, 133, 138, 197, 51, 0, 0, 0],
            &[0, 0, 0, 0, 3, 156, 94, 0, 0, 0, 0],
            &[0, 3, 11, 11, 11, 0, 0, 33, 37, 3, 0],
            &[0, 56, 179, 194, 243, 44, 191, 194, 192, 133, 0],
            &[0, 0, 0, 0, 191, 187, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 61, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 60, 0, 0, 0, 0, 0],
            &[0, 113, 193, 207, 245, 222, 201, 188, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ś' / 0x15a
        'Ś' => Some(&[
            &[0, 0, 0, 0, 0, 1, 183, 118, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 142, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 109, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 49, 73, 55, 16, 0, 0, 0],
            &[0, 0, 65, 222, 192, 156, 172, 224, 109, 0, 0],
            &[0, 13, 234, 54, 0, 0, 0, 1, 11, 0, 0],
            &[0, 64, 193, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 210, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 204, 153, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 146, 231, 148, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 117, 222, 169, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 160, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 24, 234, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 40, 225, 0, 0],
            &[0, 34, 16, 0, 0, 0, 16, 181, 134, 0, 0],
            &[0, 85, 219, 215, 187, 194, 227, 133, 3, 0, 0],
            &[0, 0, 0, 12, 35, 31, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 25, 0, 0, 0],
        ]),
        // letter: 'ś' / 0x15b
        'ś' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 222, 66, 0, 0, 0],
            &[0, 0, 0, 0, 1, 173, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 86, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 48, 26, 0, 0, 0, 0],
            &[0, 0, 50, 214, 188, 169, 194, 206, 4, 0, 0],
            &[0, 0, 190, 80, 0, 0, 0, 7, 0, 0, 0],
            &[0, 0, 205, 61, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 220, 102, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 137, 221, 150, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 109, 225, 16, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 200, 67, 0, 0],
            &[0, 7, 41, 0, 0, 0, 22, 232, 32, 0, 0],
            &[0, 11, 195, 217, 186, 189, 221, 98, 0, 0, 0],
            &[0, 0, 0, 8, 33, 31, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => Some(&[
            &[0, 0, 0, 0, 113, 237, 88, 0, 0, 0, 0],
            &[0, 0, 0, 63, 177, 18, 192, 45, 0, 0, 0],
            &[0, 0, 2, 121, 8, 0, 16, 116, 0, 0, 0],
            &[0, 0, 0, 3, 49, 73, 55, 16, 0, 0, 0],
            &[0, 0, 65, 222, 192, 156, 172, 224, 109, 0, 0],
            &[0, 13, 234, 54, 0, 0, 0, 1, 11, 0, 0],
            &[0, 64, 193, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 210, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 204, 153, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 146, 231, 148, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 117, 222, 169, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 160, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 24, 234, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 40, 225, 0, 0],
            &[0, 34, 16, 0, 0, 0, 16, 181, 134, 0, 0],
            &[0, 85, 219, 215, 187, 194, 227, 133, 3, 0, 0],
            &[0, 0, 0, 12, 35, 31, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 24, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŝ' / 0x15d
        'ŝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 224, 20, 0, 0, 0, 0],
            &[0, 0, 0, 112, 135, 65, 182, 5, 0, 0, 0],
            &[0, 0, 30, 122, 0, 0, 70, 85, 0, 0, 0],
            &[0, 0, 0, 0, 34, 48, 26, 0, 0, 0, 0],
            &[0, 0, 50, 214, 188, 169, 194, 206, 4, 0, 0],
            &[0, 0, 190, 80, 0, 0, 0, 7, 0, 0, 0],
            &[0, 0, 205, 61, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 220, 102, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 137, 221, 150, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 109, 225, 16, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 200, 67, 0, 0],
            &[0, 7, 41, 0, 0, 0, 22, 232, 32, 0, 0],
            &[0, 11, 195, 217, 186, 189, 221, 98, 0, 0, 0],
            &[0, 0, 0, 8, 33, 31, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ş' / 0x15e
        'Ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 49, 73, 55, 16, 0, 0, 0],
            &[0, 0, 65, 222, 192, 156, 172, 224, 109, 0, 0],
            &[0, 13, 234, 54, 0, 0, 0, 1, 11, 0, 0],
            &[0, 64, 193, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 210, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 204, 153, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 146, 231, 148, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 117, 222, 169, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 160, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 24, 234, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 40, 225, 0, 0],
            &[0, 34, 16, 0, 0, 0, 16, 181, 134, 0, 0],
            &[0, 85, 219, 215, 190, 201, 227, 133, 3, 0, 0],
            &[0, 0, 0, 12, 157, 95, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 119, 177, 45, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 108, 121, 0, 0, 0, 0],
            &[0, 0, 0, 5, 12, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ş' / 0x15f
        'ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 48, 26, 0, 0, 0, 0],
            &[0, 0, 50, 214, 188, 169, 194, 206, 4, 0, 0],
            &[0, 0, 190, 80, 0, 0, 0, 7, 0, 0, 0],
            &[0, 0, 205, 61, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 220, 102, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 137, 221, 150, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 109, 225, 16, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 200, 67, 0, 0],
            &[0, 7, 41, 0, 0, 0, 22, 232, 32, 0, 0],
            &[0, 11, 195, 217, 188, 196, 221, 98, 0, 0, 0],
            &[0, 0, 0, 8, 142, 108, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 177, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 95, 134, 0, 0, 0, 0],
            &[0, 0, 0, 4, 12, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Š' / 0x160
        'Š' => Some(&[
            &[0, 0, 1, 162, 73, 0, 101, 140, 0, 0, 0],
            &[0, 0, 0, 9, 196, 135, 175, 1, 0, 0, 0],
            &[0, 0, 0, 0, 39, 152, 18, 0, 0, 0, 0],
            &[0, 0, 0, 3, 49, 73, 55, 16, 0, 0, 0],
            &[0, 0, 65, 222, 192, 156, 172, 224, 109, 0, 0],
            &[0, 13, 234, 54, 0, 0, 0, 1, 11, 0, 0],
            &[0, 64, 193, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 210, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 204, 153, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 146, 231, 148, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 117, 222, 169, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 160, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 24, 234, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 40, 225, 0, 0],
            &[0, 34, 16, 0, 0, 0, 16, 181, 134, 0, 0],
            &[0, 85, 219, 215, 187, 194, 227, 133, 3, 0, 0],
            &[0, 0, 0, 12, 35, 31, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 17, 0, 0, 0, 19, 0, 0, 0],
        ]),
        // letter: 'š' / 0x161
        'š' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 180, 18, 1, 145, 89, 0, 0, 0],
            &[0, 0, 0, 58, 193, 145, 124, 0, 0, 0, 0],
            &[0, 0, 0, 0, 103, 150, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 48, 26, 0, 0, 0, 0],
            &[0, 0, 50, 214, 188, 169, 194, 206, 4, 0, 0],
            &[0, 0, 190, 80, 0, 0, 0, 7, 0, 0, 0],
            &[0, 0, 205, 61, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 220, 102, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 137, 221, 150, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 109, 225, 16, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 200, 67, 0, 0],
            &[0, 7, 41, 0, 0, 0, 22, 232, 32, 0, 0],
            &[0, 11, 195, 217, 186, 189, 221, 98, 0, 0, 0],
            &[0, 0, 0, 8, 33, 31, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ţ' / 0x162
        'Ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[6, 35, 35, 35, 35, 35, 35, 35, 35, 14, 0],
            &[35, 190, 190, 190, 216, 229, 190, 190, 190, 76, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 103, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 121, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 119, 177, 45, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 108, 121, 0, 0, 0, 0],
            &[0, 0, 0, 5, 12, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ţ' / 0x163
        'ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 221, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 33, 226, 11, 11, 11, 5, 0, 0],
            &[0, 153, 202, 211, 249, 206, 206, 206, 103, 0, 0],
            &[0, 0, 0, 27, 224, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 224, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 224, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 224, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 225, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 245, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 222, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 223, 193, 182, 114, 0, 0],
            &[0, 0, 0, 0, 0, 33, 188, 30, 1, 0, 0],
            &[0, 0, 0, 0, 0, 60, 170, 111, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 214, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 1, 0, 0, 0, 0],
        ]),
        // letter: 'Ť' / 0x164
        'Ť' => Some(&[
            &[0, 0, 30, 184, 22, 7, 171, 63, 0, 0, 0],
            &[0, 0, 0, 53, 198, 174, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 118, 0, 0, 0, 0, 0],
            &[6, 35, 35, 35, 35, 35, 35, 35, 35, 14, 0],
            &[35, 190, 190, 190, 216, 229, 190, 190, 190, 76, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 8, 0, 0, 3, 15, 0, 0, 0],
        ]),
        // letter: 'ť' / 0x165
        'ť' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 158, 87, 0, 0],
            &[0, 0, 0, 0, 77, 0, 0, 203, 9, 0, 0],
            &[0, 0, 0, 2, 221, 0, 22, 96, 0, 0, 0],
            &[0, 0, 1, 33, 226, 11, 11, 11, 5, 0, 0],
            &[0, 153, 202, 211, 249, 206, 206, 206, 103, 0, 0],
            &[0, 0, 0, 27, 224, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 224, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 224, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 224, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 225, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 245, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 222, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 223, 183, 182, 114, 0, 0],
            &[0, 0, 0, 0, 0, 3, 33, 30, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŧ' / 0x166
        'Ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[6, 35, 35, 35, 35, 35, 35, 35, 35, 14, 0],
            &[35, 190, 190, 190, 216, 229, 190, 190, 190, 76, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 23, 161, 161, 255, 255, 161, 161, 58, 0, 0],
            &[0, 8, 56, 56, 156, 208, 56, 56, 20, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŧ' / 0x167
        'ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 221, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 33, 226, 11, 11, 11, 5, 0, 0],
            &[0, 153, 202, 211, 249, 206, 206, 206, 103, 0, 0],
            &[0, 0, 0, 27, 224, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 224, 0, 0, 0, 0, 0, 0],
            &[0, 119, 147, 174, 255, 147, 147, 147, 91, 0, 0],
            &[0, 39, 48, 75, 255, 48, 48, 48, 29, 0, 0],
            &[0, 0, 0, 27, 225, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 245, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 222, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 223, 183, 182, 114, 0, 0],
            &[0, 0, 0, 0, 0, 3, 33, 30, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ũ' / 0x168
        'Ũ' => Some(&[
            &[0, 0, 2, 83, 62, 0, 0, 76, 0, 0, 0],
            &[0, 0, 129, 143, 161, 169, 100, 178, 0, 0, 0],
            &[0, 0, 88, 0, 0, 59, 113, 21, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 6, 28, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 149, 102, 0, 0, 0, 0, 47, 205, 0, 0],
            &[0, 135, 117, 0, 0, 0, 0, 62, 190, 0, 0],
            &[0, 93, 169, 0, 0, 0, 0, 117, 145, 0, 0],
            &[0, 16, 232, 61, 0, 0, 35, 227, 47, 0, 0],
            &[0, 0, 60, 218, 206, 200, 224, 88, 0, 0, 0],
            &[0, 0, 0, 1, 27, 30, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ũ' / 0x169
        'ũ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 59, 40, 0, 0, 58, 0, 0, 0],
            &[0, 0, 115, 165, 182, 146, 76, 183, 0, 0, 0],
            &[0, 0, 105, 2, 0, 82, 137, 33, 0, 0, 0],
            &[0, 5, 5, 0, 0, 0, 0, 3, 7, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 125, 0, 0, 0, 0, 102, 162, 0, 0],
            &[0, 105, 149, 0, 0, 0, 0, 154, 162, 0, 0],
            &[0, 45, 230, 22, 0, 0, 56, 204, 162, 0, 0],
            &[0, 0, 111, 229, 183, 192, 157, 61, 162, 0, 0],
            &[0, 0, 0, 4, 32, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ū' / 0x16a
        'Ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 141, 141, 141, 141, 63, 0, 0, 0],
            &[0, 0, 16, 67, 67, 67, 67, 30, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 6, 28, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 149, 102, 0, 0, 0, 0, 47, 205, 0, 0],
            &[0, 135, 117, 0, 0, 0, 0, 62, 190, 0, 0],
            &[0, 93, 169, 0, 0, 0, 0, 117, 145, 0, 0],
            &[0, 16, 232, 61, 0, 0, 35, 227, 47, 0, 0],
            &[0, 0, 60, 218, 206, 200, 224, 88, 0, 0, 0],
            &[0, 0, 0, 1, 27, 30, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ū' / 0x16b
        'ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 117, 117, 117, 117, 52, 0, 0, 0],
            &[0, 0, 22, 91, 91, 91, 91, 40, 0, 0, 0],
            &[0, 5, 5, 0, 0, 0, 0, 3, 7, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 125, 0, 0, 0, 0, 102, 162, 0, 0],
            &[0, 105, 149, 0, 0, 0, 0, 154, 162, 0, 0],
            &[0, 45, 230, 22, 0, 0, 56, 204, 162, 0, 0],
            &[0, 0, 111, 229, 183, 192, 157, 61, 162, 0, 0],
            &[0, 0, 0, 4, 32, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => Some(&[
            &[0, 0, 28, 83, 0, 0, 49, 57, 0, 0, 0],
            &[0, 0, 7, 198, 92, 72, 193, 32, 0, 0, 0],
            &[0, 0, 0, 18, 109, 116, 34, 0, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 6, 28, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 149, 102, 0, 0, 0, 0, 47, 205, 0, 0],
            &[0, 135, 117, 0, 0, 0, 0, 62, 190, 0, 0],
            &[0, 93, 169, 0, 0, 0, 0, 117, 145, 0, 0],
            &[0, 16, 232, 61, 0, 0, 35, 227, 47, 0, 0],
            &[0, 0, 60, 218, 206, 200, 224, 88, 0, 0, 0],
            &[0, 0, 0, 1, 27, 30, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŭ' / 0x16d
        'ŭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 67, 0, 0, 39, 49, 0, 0, 0],
            &[0, 0, 10, 200, 68, 48, 185, 40, 0, 0, 0],
            &[0, 0, 0, 31, 133, 140, 53, 0, 0, 0, 0],
            &[0, 5, 5, 0, 0, 0, 0, 3, 7, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 125, 0, 0, 0, 0, 102, 162, 0, 0],
            &[0, 105, 149, 0, 0, 0, 0, 154, 162, 0, 0],
            &[0, 45, 230, 22, 0, 0, 56, 204, 162, 0, 0],
            &[0, 0, 111, 229, 183, 192, 157, 61, 162, 0, 0],
            &[0, 0, 0, 4, 32, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ů' / 0x16e
        'Ů' => Some(&[
            &[0, 0, 0, 159, 32, 3, 187, 0, 0, 0, 0],
            &[0, 0, 0, 131, 102, 63, 174, 0, 0, 0, 0],
            &[0, 0, 0, 7, 108, 120, 19, 0, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 6, 28, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 149, 102, 0, 0, 0, 0, 47, 205, 0, 0],
            &[0, 135, 117, 0, 0, 0, 0, 62, 190, 0, 0],
            &[0, 93, 169, 0, 0, 0, 0, 117, 145, 0, 0],
            &[0, 16, 232, 61, 0, 0, 35, 227, 47, 0, 0],
            &[0, 0, 60, 218, 206, 200, 224, 88, 0, 0, 0],
            &[0, 0, 0, 1, 27, 30, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 43, 179, 176, 72, 0, 0, 0, 0],
        ]),
        // letter: 'ů' / 0x16f
        'ů' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 169, 171, 56, 0, 0, 0, 0],
            &[0, 0, 0, 154, 40, 8, 185, 0, 0, 0, 0],
            &[0, 0, 0, 141, 79, 39, 180, 0, 0, 0, 0],
            &[0, 0, 0, 14, 132, 144, 30, 0, 0, 0, 0],
            &[0, 5, 5, 0, 0, 0, 0, 3, 7, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 125, 0, 0, 0, 0, 102, 162, 0, 0],
            &[0, 105, 149, 0, 0, 0, 0, 154, 162, 0, 0],
            &[0, 45, 230, 22, 0, 0, 56, 204, 162, 0, 0],
            &[0, 0, 111, 229, 183, 192, 157, 61, 162, 0, 0],
            &[0, 0, 0, 4, 32, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ű' / 0x170
        'Ű' => Some(&[
            &[0, 0, 0, 13, 223, 41, 69, 202, 4, 0, 0],
            &[0, 0, 0, 151, 90, 11, 198, 31, 0, 0, 0],
            &[0, 0, 24, 96, 0, 72, 47, 0, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 6, 28, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 149, 102, 0, 0, 0, 0, 47, 205, 0, 0],
            &[0, 135, 117, 0, 0, 0, 0, 62, 190, 0, 0],
            &[0, 93, 169, 0, 0, 0, 0, 117, 145, 0, 0],
            &[0, 16, 232, 61, 0, 0, 35, 227, 47, 0, 0],
            &[0, 0, 60, 218, 206, 200, 224, 88, 0, 0, 0],
            &[0, 0, 0, 1, 27, 30, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 13, 0, 28, 4, 0, 0],
        ]),
        // letter: 'ű' / 0x171
        'ű' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 2, 0, 5, 0, 0, 0],
            &[0, 0, 0, 8, 219, 52, 56, 214, 7, 0, 0],
            &[0, 0, 0, 135, 110, 6, 195, 43, 0, 0, 0],
            &[0, 0, 24, 116, 0, 77, 63, 0, 0, 0, 0],
            &[0, 5, 5, 0, 0, 0, 0, 3, 7, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 125, 0, 0, 0, 0, 102, 162, 0, 0],
            &[0, 105, 149, 0, 0, 0, 0, 154, 162, 0, 0],
            &[0, 45, 230, 22, 0, 0, 56, 204, 162, 0, 0],
            &[0, 0, 111, 229, 183, 192, 157, 61, 162, 0, 0],
            &[0, 0, 0, 4, 32, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ų' / 0x172
        'Ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 14, 0, 0, 0, 0, 6, 28, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 150, 101, 0, 0, 0, 0, 46, 206, 0, 0],
            &[0, 149, 102, 0, 0, 0, 0, 47, 205, 0, 0],
            &[0, 135, 117, 0, 0, 0, 0, 62, 190, 0, 0],
            &[0, 93, 169, 0, 0, 0, 0, 117, 145, 0, 0],
            &[0, 16, 232, 61, 0, 0, 35, 227, 47, 0, 0],
            &[0, 0, 60, 218, 206, 209, 255, 88, 0, 0, 0],
            &[0, 0, 0, 1, 57, 200, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 193, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 215, 40, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 67, 171, 96, 0, 0, 0, 0],
        ]),
        // letter: 'ų' / 0x173
        'ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 5, 0, 0, 0, 0, 3, 7, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 123, 0, 0, 0, 0, 89, 162, 0, 0],
            &[0, 128, 125, 0, 0, 0, 0, 102, 162, 0, 0],
            &[0, 105, 149, 0, 0, 0, 0, 154, 162, 0, 0],
            &[0, 45, 230, 22, 0, 0, 56, 204, 162, 0, 0],
            &[0, 0, 111, 229, 183, 192, 157, 66, 180, 0, 0],
            &[0, 0, 0, 4, 32, 9, 3, 166, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 102, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 136, 115, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 156, 154, 0, 0],
        ]),
        // letter: 'Ŵ' / 0x174
        'Ŵ' => Some(&[
            &[0, 0, 0, 3, 192, 219, 23, 0, 0, 0, 0],
            &[0, 0, 0, 144, 101, 59, 184, 6, 0, 0, 0],
            &[0, 0, 38, 93, 0, 0, 60, 72, 0, 0, 0],
            &[15, 17, 0, 0, 0, 0, 0, 0, 10, 22, 0],
            &[90, 148, 0, 0, 0, 0, 0, 0, 93, 146, 0],
            &[55, 182, 0, 0, 0, 0, 0, 0, 127, 112, 0],
            &[20, 216, 0, 0, 4, 6, 0, 0, 160, 78, 0],
            &[0, 234, 1, 0, 128, 183, 0, 0, 195, 44, 0],
            &[0, 206, 28, 0, 180, 185, 2, 0, 229, 10, 0],
            &[0, 172, 61, 3, 190, 144, 46, 9, 230, 0, 0],
            &[0, 137, 95, 49, 147, 87, 105, 42, 197, 0, 0],
            &[0, 102, 129, 108, 89, 28, 164, 72, 163, 0, 0],
            &[0, 67, 158, 165, 31, 0, 195, 96, 129, 0, 0],
            &[0, 33, 183, 194, 0, 0, 167, 145, 95, 0, 0],
            &[0, 3, 217, 171, 0, 0, 108, 217, 61, 0, 0],
            &[0, 0, 219, 113, 0, 0, 50, 255, 27, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 15, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŵ' / 0x175
        'ŵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 165, 226, 22, 0, 0, 0, 0],
            &[0, 0, 0, 107, 140, 62, 185, 5, 0, 0, 0],
            &[0, 0, 28, 124, 0, 0, 67, 88, 0, 0, 0],
            &[7, 3, 0, 0, 0, 0, 0, 0, 0, 9, 0],
            &[136, 99, 0, 0, 0, 0, 0, 0, 38, 192, 0],
            &[83, 153, 0, 0, 41, 71, 0, 0, 89, 142, 0],
            &[31, 206, 0, 0, 164, 223, 3, 0, 140, 92, 0],
            &[0, 229, 9, 3, 211, 157, 61, 0, 191, 42, 0],
            &[0, 182, 57, 63, 159, 86, 137, 1, 231, 3, 0],
            &[0, 129, 109, 140, 85, 16, 209, 36, 198, 0, 0],
            &[0, 77, 151, 206, 14, 0, 192, 107, 148, 0, 0],
            &[0, 24, 200, 188, 0, 0, 116, 199, 98, 0, 0],
            &[0, 0, 227, 114, 0, 0, 41, 255, 48, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŷ' / 0x176
        'Ŷ' => Some(&[
            &[0, 0, 0, 3, 192, 219, 23, 0, 0, 0, 0],
            &[0, 0, 0, 144, 101, 59, 184, 6, 0, 0, 0],
            &[0, 0, 38, 93, 0, 0, 60, 72, 0, 0, 0],
            &[8, 30, 0, 0, 0, 0, 0, 0, 21, 15, 0],
            &[9, 226, 38, 0, 0, 0, 0, 5, 221, 42, 0],
            &[0, 105, 163, 0, 0, 0, 0, 101, 161, 0, 0],
            &[0, 5, 218, 38, 0, 0, 4, 217, 33, 0, 0],
            &[0, 0, 92, 163, 0, 0, 98, 150, 0, 0, 0],
            &[0, 0, 2, 209, 38, 4, 211, 26, 0, 0, 0],
            &[0, 0, 0, 80, 163, 95, 139, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 214, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 98, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 15, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŷ' / 0x177
        'ŷ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 181, 217, 14, 0, 0, 0, 0],
            &[0, 0, 0, 125, 122, 76, 174, 2, 0, 0, 0],
            &[0, 0, 38, 115, 0, 0, 79, 76, 0, 0, 0],
            &[0, 9, 1, 0, 0, 0, 0, 0, 10, 0, 0],
            &[0, 160, 90, 0, 0, 0, 0, 32, 224, 2, 0],
            &[0, 58, 189, 0, 0, 0, 0, 122, 131, 0, 0],
            &[0, 0, 211, 34, 0, 0, 0, 213, 35, 0, 0],
            &[0, 0, 111, 132, 0, 0, 50, 194, 0, 0, 0],
            &[0, 0, 18, 220, 3, 0, 142, 97, 0, 0, 0],
            &[0, 0, 0, 164, 75, 3, 219, 12, 0, 0, 0],
            &[0, 0, 0, 62, 172, 68, 160, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 168, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 114, 222, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 210, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 89, 171, 0, 0, 0, 0, 0, 0],
            &[0, 19, 39, 4, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ÿ' / 0x178
        'Ÿ' => Some(&[
            &[0, 0, 16, 75, 0, 0, 54, 36, 0, 0, 0],
            &[0, 0, 75, 231, 0, 0, 175, 130, 0, 0, 0],
            &[0, 0, 0, 12, 0, 0, 9, 3, 0, 0, 0],
            &[8, 30, 0, 0, 0, 0, 0, 0, 21, 15, 0],
            &[9, 226, 38, 0, 0, 0, 0, 5, 221, 42, 0],
            &[0, 105, 163, 0, 0, 0, 0, 101, 161, 0, 0],
            &[0, 5, 218, 38, 0, 0, 4, 217, 33, 0, 0],
            &[0, 0, 92, 163, 0, 0, 98, 150, 0, 0, 0],
            &[0, 0, 2, 209, 38, 4, 211, 26, 0, 0, 0],
            &[0, 0, 0, 80, 163, 95, 139, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 214, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 98, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ź' / 0x179
        'Ź' => Some(&[
            &[0, 0, 0, 0, 0, 4, 201, 96, 0, 0, 0],
            &[0, 0, 0, 0, 0, 144, 121, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 98, 0, 0, 0, 0, 0],
            &[0, 0, 28, 35, 35, 35, 35, 35, 14, 0, 0],
            &[0, 0, 150, 186, 186, 186, 186, 238, 91, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 220, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 155, 103, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 216, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 169, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 209, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 183, 78, 0, 0, 0, 0, 0],
            &[0, 0, 0, 62, 199, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 197, 65, 0, 0, 0, 0, 0, 0],
            &[0, 0, 76, 186, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 208, 53, 0, 0, 0, 0, 0, 0, 0],
            &[0, 51, 253, 230, 230, 230, 230, 230, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 22, 0, 0, 0],
        ]),
        // letter: 'ź' / 0x17a
        'ź' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 205, 93, 0, 0, 0],
            &[0, 0, 0, 0, 0, 147, 122, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 107, 0, 0, 0, 0, 0],
            &[0, 0, 10, 11, 11, 11, 11, 11, 4, 0, 0],
            &[0, 0, 172, 193, 193, 193, 193, 246, 92, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 199, 5, 0, 0],
            &[0, 0, 0, 0, 0, 26, 220, 33, 0, 0, 0],
            &[0, 0, 0, 0, 2, 188, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 209, 10, 0, 0, 0, 0, 0],
            &[0, 0, 13, 213, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 107, 0, 0, 0, 0, 0, 0, 0],
            &[0, 53, 255, 222, 221, 221, 221, 221, 95, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ż' / 0x17b
        'Ż' => Some(&[
            &[0, 0, 0, 0, 39, 148, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 229, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 35, 35, 35, 35, 35, 14, 0, 0],
            &[0, 0, 150, 186, 186, 186, 186, 238, 91, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 220, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 155, 103, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 216, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 169, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 209, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 183, 78, 0, 0, 0, 0, 0],
            &[0, 0, 0, 62, 199, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 197, 65, 0, 0, 0, 0, 0, 0],
            &[0, 0, 76, 186, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 208, 53, 0, 0, 0, 0, 0, 0, 0],
            &[0, 51, 253, 230, 230, 230, 230, 230, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ż' / 0x17c
        'ż' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 219, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 11, 11, 11, 11, 11, 4, 0, 0],
            &[0, 0, 172, 193, 193, 193, 193, 246, 92, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 199, 5, 0, 0],
            &[0, 0, 0, 0, 0, 26, 220, 33, 0, 0, 0],
            &[0, 0, 0, 0, 2, 188, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 209, 10, 0, 0, 0, 0, 0],
            &[0, 0, 13, 213, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 107, 0, 0, 0, 0, 0, 0, 0],
            &[0, 53, 255, 222, 221, 221, 221, 221, 95, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ž' / 0x17d
        'Ž' => Some(&[
            &[0, 0, 5, 173, 58, 0, 121, 120, 0, 0, 0],
            &[0, 0, 0, 16, 203, 140, 156, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 148, 9, 0, 0, 0, 0],
            &[0, 0, 28, 35, 35, 35, 35, 35, 14, 0, 0],
            &[0, 0, 150, 186, 186, 186, 186, 238, 91, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 220, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 155, 103, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 216, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 169, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 209, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 183, 78, 0, 0, 0, 0, 0],
            &[0, 0, 0, 62, 199, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 197, 65, 0, 0, 0, 0, 0, 0],
            &[0, 0, 76, 186, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 208, 53, 0, 0, 0, 0, 0, 0, 0],
            &[0, 51, 253, 230, 230, 230, 230, 230, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 14, 0, 0, 0, 19, 0, 0, 0],
        ]),
        // letter: 'ž' / 0x17e
        'ž' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 182, 31, 0, 118, 118, 0, 0, 0],
            &[0, 0, 0, 37, 202, 126, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 164, 7, 0, 0, 0, 0],
            &[0, 0, 10, 11, 11, 11, 11, 11, 4, 0, 0],
            &[0, 0, 172, 193, 193, 193, 193, 246, 92, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 199, 5, 0, 0],
            &[0, 0, 0, 0, 0, 26, 220, 33, 0, 0, 0],
            &[0, 0, 0, 0, 2, 188, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 209, 10, 0, 0, 0, 0, 0],
            &[0, 0, 13, 213, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 107, 0, 0, 0, 0, 0, 0, 0],
            &[0, 53, 255, 222, 221, 221, 221, 221, 95, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ſ' / 0x17f
        'ſ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 133, 219, 214, 192, 0, 0, 0],
            &[0, 0, 0, 120, 178, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 206, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 231, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 231, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 231, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 231, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 231, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 231, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 231, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 231, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 231, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        _ => None,
    }
}
