//! Module for letters with the font weight light and size 16.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 16;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 9;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight light and font size 14px
#[inline]
pub const fn get_char(c: char) -> Option<&'static [&'static [u8]]> {
    match c {
        // letter: ' ' / 0x20
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '!' / 0x21
        '!' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 173, 0, 0, 0, 0],
            &[0, 0, 0, 64, 170, 0, 0, 0, 0],
            &[0, 0, 0, 59, 165, 0, 0, 0, 0],
            &[0, 0, 0, 55, 161, 0, 0, 0, 0],
            &[0, 0, 0, 50, 156, 0, 0, 0, 0],
            &[0, 0, 0, 46, 152, 0, 0, 0, 0],
            &[0, 0, 0, 42, 148, 0, 0, 0, 0],
            &[0, 0, 0, 14, 53, 0, 0, 0, 0],
            &[0, 0, 0, 11, 38, 0, 0, 0, 0],
            &[0, 0, 0, 89, 196, 0, 0, 0, 0],
            &[0, 0, 0, 1, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '"' / 0x22
        '"' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 114, 12, 196, 0, 0, 0],
            &[0, 0, 79, 102, 1, 184, 0, 0, 0],
            &[0, 0, 68, 91, 0, 162, 0, 0, 0],
            &[0, 0, 35, 50, 0, 88, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '#' / 0x23
        '#' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 171, 0, 59, 105, 0, 0],
            &[0, 0, 27, 146, 0, 103, 62, 0, 0],
            &[0, 0, 71, 103, 0, 147, 20, 0, 0],
            &[34, 161, 209, 177, 161, 224, 161, 103, 0],
            &[0, 0, 158, 12, 0, 171, 0, 0, 0],
            &[0, 0, 171, 0, 27, 144, 0, 0, 0],
            &[114, 165, 225, 164, 187, 203, 164, 29, 0],
            &[0, 35, 136, 0, 116, 55, 0, 0, 0],
            &[0, 78, 91, 0, 160, 11, 0, 0, 0],
            &[0, 121, 47, 0, 171, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '$' / 0x24
        '$' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 74, 0, 0, 0, 0],
            &[0, 0, 78, 168, 206, 136, 54, 0, 0],
            &[0, 104, 161, 71, 130, 56, 70, 0, 0],
            &[0, 184, 26, 38, 119, 0, 0, 0, 0],
            &[0, 145, 99, 38, 119, 0, 0, 0, 0],
            &[0, 12, 155, 195, 153, 6, 0, 0, 0],
            &[0, 0, 0, 53, 195, 201, 85, 0, 0],
            &[0, 0, 0, 38, 119, 11, 209, 13, 0],
            &[0, 0, 0, 38, 119, 0, 179, 23, 0],
            &[0, 87, 71, 69, 144, 113, 173, 0, 0],
            &[0, 47, 121, 164, 192, 86, 2, 0, 0],
            &[0, 0, 0, 31, 96, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '%' / 0x25
        '%' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 7, 0, 0, 0, 0, 0, 0],
            &[95, 153, 158, 81, 0, 8, 179, 1, 0],
            &[179, 0, 0, 178, 0, 110, 79, 0, 0],
            &[176, 0, 0, 177, 9, 179, 0, 0, 0],
            &[167, 27, 36, 161, 114, 74, 0, 0, 0],
            &[21, 134, 132, 28, 178, 0, 0, 0, 0],
            &[0, 0, 0, 119, 70, 85, 143, 67, 0],
            &[0, 0, 13, 176, 65, 131, 4, 158, 34],
            &[0, 0, 123, 66, 126, 50, 0, 84, 91],
            &[0, 15, 174, 0, 106, 72, 0, 107, 73],
            &[0, 127, 62, 0, 16, 171, 128, 169, 4],
            &[0, 0, 0, 0, 0, 0, 19, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '&' / 0x26
        '&' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 6, 0, 0, 0, 0],
            &[0, 2, 173, 145, 189, 41, 0, 0, 0],
            &[0, 51, 144, 0, 57, 138, 0, 0, 0],
            &[0, 39, 155, 0, 74, 126, 0, 0, 0],
            &[0, 0, 186, 45, 191, 23, 0, 0, 0],
            &[0, 0, 116, 255, 36, 0, 0, 0, 0],
            &[0, 78, 170, 117, 120, 0, 32, 95, 0],
            &[4, 202, 7, 0, 163, 68, 106, 90, 0],
            &[27, 175, 0, 0, 11, 199, 224, 9, 0],
            &[6, 211, 9, 0, 4, 196, 255, 8, 0],
            &[0, 86, 202, 160, 195, 92, 87, 157, 0],
            &[0, 0, 7, 29, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ''' / 0x27
        '\'' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 53, 155, 0, 0, 0, 0],
            &[0, 0, 0, 42, 144, 0, 0, 0, 0],
            &[0, 0, 0, 30, 132, 0, 0, 0, 0],
            &[0, 0, 0, 12, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '(' / 0x28
        '(' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 118, 0, 0, 0],
            &[0, 0, 0, 10, 193, 3, 0, 0, 0],
            &[0, 0, 0, 102, 105, 0, 0, 0, 0],
            &[0, 0, 0, 178, 29, 0, 0, 0, 0],
            &[0, 0, 0, 207, 0, 0, 0, 0, 0],
            &[0, 0, 1, 205, 0, 0, 0, 0, 0],
            &[0, 0, 5, 202, 0, 0, 0, 0, 0],
            &[0, 0, 0, 208, 0, 0, 0, 0, 0],
            &[0, 0, 0, 193, 16, 0, 0, 0, 0],
            &[0, 0, 0, 126, 84, 0, 0, 0, 0],
            &[0, 0, 0, 25, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 80, 0, 0, 0],
            &[0, 0, 0, 0, 4, 38, 0, 0, 0],
        ]),
        // letter: ')' / 0x29
        ')' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 173, 1, 0, 0, 0, 0],
            &[0, 0, 0, 113, 94, 0, 0, 0, 0],
            &[0, 0, 0, 14, 195, 0, 0, 0, 0],
            &[0, 0, 0, 0, 180, 28, 0, 0, 0],
            &[0, 0, 0, 0, 127, 81, 0, 0, 0],
            &[0, 0, 0, 0, 104, 103, 0, 0, 0],
            &[0, 0, 0, 0, 101, 105, 0, 0, 0],
            &[0, 0, 0, 0, 121, 85, 0, 0, 0],
            &[0, 0, 0, 0, 169, 37, 0, 0, 0],
            &[0, 0, 0, 5, 202, 0, 0, 0, 0],
            &[0, 0, 0, 88, 118, 0, 0, 0, 0],
            &[0, 0, 12, 184, 9, 0, 0, 0, 0],
            &[0, 0, 17, 25, 0, 0, 0, 0, 0],
        ]),
        // letter: '*' / 0x2a
        '*' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 0, 0, 0, 0],
            &[0, 73, 93, 19, 143, 39, 117, 38, 0],
            &[0, 21, 81, 156, 241, 129, 71, 11, 0],
            &[0, 0, 2, 168, 88, 130, 0, 0, 0],
            &[0, 0, 106, 99, 0, 166, 48, 0, 0],
            &[0, 0, 4, 1, 0, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '+' / 0x2b
        '+' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 24, 0, 0, 0, 0],
            &[0, 0, 0, 31, 140, 0, 0, 0, 0],
            &[0, 0, 0, 31, 140, 0, 0, 0, 0],
            &[0, 67, 71, 93, 172, 71, 71, 24, 0],
            &[0, 95, 100, 119, 186, 100, 100, 35, 0],
            &[0, 0, 0, 31, 140, 0, 0, 0, 0],
            &[0, 0, 0, 31, 140, 0, 0, 0, 0],
            &[0, 0, 0, 9, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ',' / 0x2c
        ',' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 174, 0, 0, 0, 0],
            &[0, 0, 0, 57, 186, 0, 0, 0, 0],
            &[0, 0, 0, 112, 105, 0, 0, 0, 0],
            &[0, 0, 0, 146, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '-' / 0x2d
        '-' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 46, 51, 51, 51, 15, 0, 0],
            &[0, 0, 121, 135, 135, 135, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '.' / 0x2e
        '.' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 38, 0, 0, 0, 0],
            &[0, 0, 0, 92, 196, 0, 0, 0, 0],
            &[0, 0, 0, 1, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '/' / 0x2f
        '/' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 125, 78, 0, 0],
            &[0, 0, 0, 0, 1, 198, 4, 0, 0],
            &[0, 0, 0, 0, 63, 140, 0, 0, 0],
            &[0, 0, 0, 0, 160, 43, 0, 0, 0],
            &[0, 0, 0, 13, 190, 0, 0, 0, 0],
            &[0, 0, 0, 99, 104, 0, 0, 0, 0],
            &[0, 0, 0, 187, 16, 0, 0, 0, 0],
            &[0, 0, 38, 165, 0, 0, 0, 0, 0],
            &[0, 0, 134, 69, 0, 0, 0, 0, 0],
            &[0, 3, 198, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '0' / 0x30
        '0' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 25, 0, 0, 0, 0],
            &[0, 0, 144, 186, 161, 195, 36, 0, 0],
            &[0, 87, 138, 0, 0, 98, 240, 0, 0],
            &[0, 177, 25, 0, 3, 191, 191, 26, 0],
            &[0, 197, 0, 0, 106, 107, 123, 69, 0],
            &[0, 196, 0, 19, 190, 3, 105, 87, 0],
            &[0, 199, 0, 150, 61, 0, 110, 87, 0],
            &[0, 200, 48, 162, 0, 0, 129, 68, 0],
            &[0, 174, 217, 26, 0, 0, 181, 24, 0],
            &[0, 85, 217, 0, 0, 37, 183, 0, 0],
            &[0, 0, 136, 184, 162, 193, 34, 0, 0],
            &[0, 0, 0, 10, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '1' / 0x31
        '1' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 109, 236, 24, 0, 0, 0],
            &[0, 27, 189, 100, 171, 24, 0, 0, 0],
            &[0, 4, 25, 0, 175, 24, 0, 0, 0],
            &[0, 0, 0, 0, 176, 24, 0, 0, 0],
            &[0, 0, 0, 0, 176, 24, 0, 0, 0],
            &[0, 0, 0, 0, 176, 24, 0, 0, 0],
            &[0, 0, 0, 0, 176, 24, 0, 0, 0],
            &[0, 0, 0, 0, 176, 24, 0, 0, 0],
            &[0, 0, 0, 0, 176, 24, 0, 0, 0],
            &[0, 0, 97, 158, 255, 187, 150, 6, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '2' / 0x32
        '2' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 25, 0, 0, 0, 0],
            &[0, 51, 185, 183, 166, 202, 45, 0, 0],
            &[0, 51, 29, 0, 0, 41, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 212, 0, 0],
            &[0, 0, 0, 0, 0, 8, 211, 0, 0],
            &[0, 0, 0, 0, 0, 110, 117, 0, 0],
            &[0, 0, 0, 0, 45, 194, 8, 0, 0],
            &[0, 0, 0, 32, 198, 28, 0, 0, 0],
            &[0, 0, 29, 195, 36, 0, 0, 0, 0],
            &[0, 26, 194, 39, 0, 0, 0, 0, 0],
            &[0, 178, 218, 189, 189, 189, 189, 87, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '3' / 0x33
        '3' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 27, 2, 0, 0, 0],
            &[0, 55, 184, 182, 163, 205, 67, 0, 0],
            &[0, 48, 21, 0, 0, 23, 213, 0, 0],
            &[0, 0, 0, 0, 0, 0, 211, 1, 0],
            &[0, 0, 0, 0, 0, 50, 173, 0, 0],
            &[0, 0, 52, 143, 169, 153, 14, 0, 0],
            &[0, 0, 17, 48, 73, 137, 151, 0, 0],
            &[0, 0, 0, 0, 0, 0, 162, 55, 0],
            &[0, 0, 0, 0, 0, 0, 140, 76, 0],
            &[0, 17, 0, 0, 0, 18, 212, 20, 0],
            &[0, 141, 191, 163, 170, 199, 68, 0, 0],
            &[0, 0, 3, 26, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '4' / 0x34
        '4' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 1, 0, 0],
            &[0, 0, 0, 0, 50, 248, 26, 0, 0],
            &[0, 0, 0, 3, 181, 178, 26, 0, 0],
            &[0, 0, 0, 116, 89, 171, 26, 0, 0],
            &[0, 0, 33, 173, 0, 173, 26, 0, 0],
            &[0, 0, 174, 32, 0, 174, 26, 0, 0],
            &[0, 92, 115, 0, 0, 174, 26, 0, 0],
            &[18, 206, 52, 49, 49, 189, 70, 30, 0],
            &[29, 129, 129, 129, 129, 215, 143, 81, 0],
            &[0, 0, 0, 0, 0, 174, 26, 0, 0],
            &[0, 0, 0, 0, 0, 174, 26, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '5' / 0x35
        '5' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 230, 186, 186, 186, 121, 0, 0],
            &[0, 42, 151, 0, 0, 0, 0, 0, 0],
            &[0, 64, 128, 0, 0, 0, 0, 0, 0],
            &[0, 86, 105, 3, 3, 0, 0, 0, 0],
            &[0, 78, 187, 168, 181, 184, 39, 0, 0],
            &[0, 0, 0, 0, 0, 47, 206, 1, 0],
            &[0, 0, 0, 0, 0, 0, 184, 30, 0],
            &[0, 0, 0, 0, 0, 0, 190, 26, 0],
            &[0, 16, 0, 0, 0, 54, 197, 0, 0],
            &[0, 125, 187, 160, 181, 189, 33, 0, 0],
            &[0, 0, 2, 26, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '6' / 0x36
        '6' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 33, 7, 0, 0],
            &[0, 0, 40, 181, 166, 146, 95, 0, 0],
            &[0, 15, 196, 29, 0, 0, 0, 0, 0],
            &[0, 118, 87, 0, 0, 0, 0, 0, 0],
            &[0, 187, 11, 19, 51, 17, 0, 0, 0],
            &[0, 200, 115, 145, 119, 178, 109, 0, 0],
            &[0, 233, 67, 0, 0, 0, 180, 32, 0],
            &[0, 211, 0, 0, 0, 0, 116, 83, 0],
            &[0, 179, 20, 0, 0, 0, 127, 77, 0],
            &[0, 85, 145, 0, 0, 10, 201, 15, 0],
            &[0, 0, 131, 189, 156, 196, 72, 0, 0],
            &[0, 0, 0, 10, 27, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '7' / 0x37
        '7' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 127, 186, 186, 186, 186, 223, 81, 0],
            &[0, 0, 0, 0, 0, 0, 192, 12, 0],
            &[0, 0, 0, 0, 0, 54, 152, 0, 0],
            &[0, 0, 0, 0, 0, 159, 49, 0, 0],
            &[0, 0, 0, 0, 17, 193, 0, 0, 0],
            &[0, 0, 0, 0, 113, 99, 0, 0, 0],
            &[0, 0, 0, 0, 202, 11, 0, 0, 0],
            &[0, 0, 0, 66, 150, 0, 0, 0, 0],
            &[0, 0, 0, 171, 47, 0, 0, 0, 0],
            &[0, 0, 25, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '8' / 0x38
        '8' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 24, 0, 0, 0, 0],
            &[0, 18, 179, 168, 158, 194, 42, 0, 0],
            &[0, 138, 84, 0, 0, 40, 183, 0, 0],
            &[0, 164, 51, 0, 0, 12, 198, 0, 0],
            &[0, 64, 174, 11, 4, 144, 96, 0, 0],
            &[0, 0, 66, 225, 216, 84, 0, 0, 0],
            &[0, 33, 184, 74, 77, 187, 56, 0, 0],
            &[0, 193, 32, 0, 0, 15, 206, 7, 0],
            &[0, 209, 0, 0, 0, 0, 174, 35, 0],
            &[0, 195, 37, 0, 0, 19, 207, 6, 0],
            &[0, 40, 190, 159, 156, 191, 57, 0, 0],
            &[0, 0, 0, 22, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '9' / 0x39
        '9' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 20, 25, 0, 0, 0, 0],
            &[0, 20, 179, 166, 157, 185, 26, 0, 0],
            &[0, 160, 67, 0, 0, 51, 178, 0, 0],
            &[0, 205, 0, 0, 0, 0, 173, 25, 0],
            &[0, 199, 0, 0, 0, 0, 144, 70, 0],
            &[0, 178, 34, 0, 0, 11, 213, 81, 0],
            &[0, 39, 190, 132, 129, 160, 149, 63, 0],
            &[0, 0, 3, 38, 31, 0, 170, 30, 0],
            &[0, 0, 0, 0, 0, 12, 195, 0, 0],
            &[0, 0, 0, 0, 4, 159, 88, 0, 0],
            &[0, 47, 159, 157, 196, 101, 0, 0, 0],
            &[0, 0, 24, 27, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ':' / 0x3a
        ':' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 92, 0, 0, 0, 0],
            &[0, 0, 0, 65, 152, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 37, 0, 0, 0, 0],
            &[0, 0, 0, 92, 193, 0, 0, 0, 0],
            &[0, 0, 0, 1, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ';' / 0x3b
        ';' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 112, 0, 0, 0, 0],
            &[0, 0, 0, 18, 186, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 162, 0, 0, 0, 0],
            &[0, 0, 0, 75, 168, 0, 0, 0, 0],
            &[0, 0, 0, 129, 87, 0, 0, 0, 0],
            &[0, 0, 0, 155, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '<' / 0x3c
        '<' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 19, 0],
            &[0, 0, 0, 0, 9, 115, 172, 29, 0],
            &[0, 0, 7, 109, 169, 65, 0, 0, 0],
            &[0, 97, 166, 67, 0, 0, 0, 0, 0],
            &[0, 107, 172, 76, 0, 0, 0, 0, 0],
            &[0, 0, 7, 105, 177, 88, 3, 0, 0],
            &[0, 0, 0, 0, 4, 95, 179, 38, 0],
            &[0, 0, 0, 0, 0, 0, 2, 10, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '=' / 0x3d
        '=' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[3, 173, 173, 173, 173, 173, 173, 72, 0],
            &[0, 12, 12, 12, 12, 12, 12, 5, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[3, 189, 189, 189, 189, 189, 189, 79, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '>' / 0x3e
        '>' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 31, 0, 0, 0, 0, 0, 0, 0],
            &[0, 128, 158, 40, 0, 0, 0, 0, 0],
            &[0, 0, 24, 140, 152, 35, 0, 0, 0],
            &[0, 0, 0, 0, 24, 139, 146, 20, 0],
            &[0, 0, 0, 0, 32, 145, 155, 24, 0],
            &[0, 0, 41, 155, 150, 35, 0, 0, 0],
            &[0, 147, 142, 28, 0, 0, 0, 0, 0],
            &[0, 13, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '?' / 0x3f
        '?' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 23, 0, 0, 0, 0],
            &[0, 56, 197, 166, 163, 196, 41, 0, 0],
            &[0, 4, 1, 0, 0, 45, 180, 0, 0],
            &[0, 0, 0, 0, 0, 2, 198, 0, 0],
            &[0, 0, 0, 0, 0, 72, 144, 0, 0],
            &[0, 0, 0, 0, 73, 181, 14, 0, 0],
            &[0, 0, 0, 67, 168, 7, 0, 0, 0],
            &[0, 0, 0, 166, 26, 0, 0, 0, 0],
            &[0, 0, 0, 67, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 6, 0, 0, 0, 0],
            &[0, 0, 0, 221, 63, 0, 0, 0, 0],
            &[0, 0, 0, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '@' / 0x40
        '@' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 111, 175, 164, 169, 77, 0, 0],
            &[0, 134, 101, 0, 0, 2, 142, 67, 0],
            &[37, 153, 0, 9, 52, 24, 6, 170, 0],
            &[125, 53, 27, 182, 106, 213, 9, 155, 13],
            &[170, 3, 142, 58, 0, 180, 0, 126, 38],
            &[173, 0, 190, 5, 0, 185, 0, 128, 39],
            &[171, 0, 183, 4, 2, 220, 0, 151, 13],
            &[157, 18, 112, 117, 115, 171, 91, 158, 0],
            &[86, 100, 1, 86, 68, 7, 86, 19, 0],
            &[3, 181, 34, 0, 0, 0, 0, 0, 0],
            &[0, 23, 169, 146, 110, 132, 158, 0, 0],
            &[0, 0, 0, 25, 54, 33, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'A' / 0x41
        'A' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 6, 0, 0, 0, 0],
            &[0, 0, 0, 104, 205, 0, 0, 0, 0],
            &[0, 0, 0, 185, 167, 40, 0, 0, 0],
            &[0, 0, 28, 174, 70, 129, 0, 0, 0],
            &[0, 0, 116, 88, 4, 197, 0, 0, 0],
            &[0, 0, 194, 11, 0, 151, 51, 0, 0],
            &[0, 38, 200, 44, 44, 108, 145, 0, 0],
            &[0, 127, 205, 145, 145, 148, 235, 2, 0],
            &[0, 202, 8, 0, 0, 0, 146, 63, 0],
            &[50, 162, 0, 0, 0, 0, 59, 152, 0],
            &[139, 74, 0, 0, 0, 0, 1, 207, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'B' / 0x42
        'B' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 201, 182, 183, 189, 85, 0, 0],
            &[0, 135, 64, 0, 0, 10, 201, 35, 0],
            &[0, 135, 64, 0, 0, 0, 137, 75, 0],
            &[0, 135, 64, 0, 0, 4, 191, 26, 0],
            &[0, 135, 179, 153, 165, 189, 62, 0, 0],
            &[0, 135, 86, 29, 36, 85, 181, 23, 0],
            &[0, 135, 64, 0, 0, 0, 88, 131, 0],
            &[0, 135, 64, 0, 0, 0, 78, 142, 0],
            &[0, 135, 64, 0, 0, 13, 187, 67, 0],
            &[0, 135, 203, 186, 192, 195, 92, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'C' / 0x43
        'C' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 26, 4, 0, 0],
            &[0, 0, 47, 186, 176, 160, 192, 107, 0],
            &[0, 30, 206, 32, 0, 0, 0, 3, 0],
            &[0, 153, 76, 0, 0, 0, 0, 0, 0],
            &[0, 214, 2, 0, 0, 0, 0, 0, 0],
            &[4, 210, 0, 0, 0, 0, 0, 0, 0],
            &[4, 209, 0, 0, 0, 0, 0, 0, 0],
            &[0, 215, 2, 0, 0, 0, 0, 0, 0],
            &[0, 163, 64, 0, 0, 0, 0, 0, 0],
            &[0, 42, 204, 24, 0, 0, 0, 1, 0],
            &[0, 0, 64, 200, 179, 168, 195, 76, 0],
            &[0, 0, 0, 0, 16, 26, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'D' / 0x44
        'D' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 211, 182, 185, 194, 130, 7, 0, 0],
            &[0, 207, 0, 0, 2, 104, 180, 0, 0],
            &[0, 207, 0, 0, 0, 0, 157, 71, 0],
            &[0, 207, 0, 0, 0, 0, 72, 143, 0],
            &[0, 207, 0, 0, 0, 0, 41, 170, 0],
            &[0, 207, 0, 0, 0, 0, 40, 166, 0],
            &[0, 207, 0, 0, 0, 0, 76, 135, 0],
            &[0, 207, 0, 0, 0, 0, 168, 57, 0],
            &[0, 207, 0, 0, 4, 119, 161, 0, 0],
            &[0, 212, 186, 185, 193, 114, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'E' / 0x45
        'E' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 205, 186, 186, 186, 186, 33, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 184, 157, 157, 157, 141, 0, 0],
            &[0, 135, 93, 29, 29, 29, 26, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 208, 189, 189, 189, 189, 34, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'F' / 0x46
        'F' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 205, 186, 186, 186, 186, 33, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 100, 38, 38, 38, 35, 0, 0],
            &[0, 135, 177, 147, 147, 147, 135, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'G' / 0x47
        'G' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 26, 18, 0, 0, 0],
            &[0, 0, 101, 202, 164, 172, 190, 6, 0],
            &[0, 89, 170, 5, 0, 0, 4, 0, 0],
            &[0, 208, 18, 0, 0, 0, 0, 0, 0],
            &[33, 183, 0, 0, 0, 0, 0, 0, 0],
            &[63, 149, 0, 0, 14, 24, 24, 6, 0],
            &[69, 144, 0, 0, 97, 165, 214, 64, 0],
            &[47, 170, 0, 0, 0, 0, 139, 64, 0],
            &[3, 216, 6, 0, 0, 0, 139, 64, 0],
            &[0, 124, 140, 1, 0, 0, 139, 64, 0],
            &[0, 1, 131, 201, 163, 179, 195, 38, 0],
            &[0, 0, 0, 3, 27, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'H' / 0x48
        'H' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 0, 142, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 211, 157, 157, 157, 157, 212, 64, 0],
            &[0, 208, 32, 32, 32, 32, 157, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'I' / 0x49
        'I' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 156, 187, 222, 162, 76, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 16, 156, 187, 223, 162, 76, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'J' / 0x4a
        'J' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 155, 51, 0, 0],
            &[0, 0, 0, 0, 0, 156, 51, 0, 0],
            &[0, 0, 0, 0, 0, 156, 51, 0, 0],
            &[0, 0, 0, 0, 0, 156, 51, 0, 0],
            &[0, 0, 0, 0, 0, 156, 51, 0, 0],
            &[0, 0, 0, 0, 0, 156, 51, 0, 0],
            &[0, 0, 0, 0, 0, 156, 51, 0, 0],
            &[0, 0, 0, 0, 0, 167, 43, 0, 0],
            &[0, 4, 0, 0, 16, 217, 7, 0, 0],
            &[0, 88, 186, 155, 204, 97, 0, 0, 0],
            &[0, 0, 9, 35, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'K' / 0x4b
        'K' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 14, 194, 42, 0],
            &[0, 207, 0, 0, 4, 182, 70, 0, 0],
            &[0, 207, 0, 0, 157, 104, 0, 0, 0],
            &[0, 207, 0, 124, 141, 0, 0, 0, 0],
            &[0, 207, 85, 243, 17, 0, 0, 0, 0],
            &[0, 211, 164, 82, 156, 0, 0, 0, 0],
            &[0, 209, 8, 0, 161, 72, 0, 0, 0],
            &[0, 207, 0, 0, 17, 201, 15, 0, 0],
            &[0, 207, 0, 0, 0, 76, 159, 0, 0],
            &[0, 207, 0, 0, 0, 0, 160, 75, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'L' / 0x4c
        'L' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 171, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 235, 193, 193, 193, 193, 35, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'M' / 0x4d
        'M' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[58, 251, 29, 0, 0, 0, 183, 160, 0],
            &[58, 201, 119, 0, 0, 23, 197, 161, 0],
            &[58, 143, 189, 0, 0, 111, 118, 161, 0],
            &[58, 141, 147, 45, 0, 186, 43, 161, 0],
            &[58, 141, 54, 136, 38, 155, 35, 161, 0],
            &[58, 141, 0, 188, 130, 63, 35, 161, 0],
            &[58, 141, 0, 125, 215, 1, 35, 161, 0],
            &[58, 141, 0, 20, 54, 0, 35, 161, 0],
            &[58, 141, 0, 0, 0, 0, 35, 161, 0],
            &[58, 141, 0, 0, 0, 0, 35, 161, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'N' / 0x4e
        'N' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 213, 109, 0, 0, 0, 139, 64, 0],
            &[0, 185, 188, 6, 0, 0, 139, 64, 0],
            &[0, 195, 94, 104, 0, 0, 139, 64, 0],
            &[0, 200, 4, 189, 5, 0, 139, 64, 0],
            &[0, 200, 0, 99, 99, 0, 139, 64, 0],
            &[0, 200, 0, 5, 190, 4, 139, 64, 0],
            &[0, 200, 0, 0, 105, 94, 139, 64, 0],
            &[0, 200, 0, 0, 7, 190, 137, 64, 0],
            &[0, 200, 0, 0, 0, 110, 213, 64, 0],
            &[0, 200, 0, 0, 0, 9, 232, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'O' / 0x4f
        'O' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 29, 2, 0, 0, 0],
            &[0, 5, 158, 183, 158, 198, 57, 0, 0],
            &[0, 135, 105, 0, 0, 19, 204, 12, 0],
            &[3, 210, 2, 0, 0, 0, 119, 96, 0],
            &[45, 165, 0, 0, 0, 0, 60, 150, 0],
            &[68, 141, 0, 0, 0, 0, 37, 173, 0],
            &[70, 142, 0, 0, 0, 0, 39, 172, 0],
            &[45, 167, 0, 0, 0, 0, 63, 148, 0],
            &[4, 211, 2, 0, 0, 0, 123, 92, 0],
            &[0, 137, 109, 0, 0, 24, 205, 10, 0],
            &[0, 7, 161, 188, 167, 199, 53, 0, 0],
            &[0, 0, 0, 13, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'P' / 0x50
        'P' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 203, 182, 183, 191, 100, 0, 0],
            &[0, 135, 72, 0, 0, 8, 177, 71, 0],
            &[0, 135, 72, 0, 0, 0, 77, 138, 0],
            &[0, 135, 72, 0, 0, 0, 82, 133, 0],
            &[0, 135, 72, 0, 0, 21, 194, 46, 0],
            &[0, 135, 205, 186, 186, 180, 64, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Q' / 0x51
        'Q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 29, 2, 0, 0, 0],
            &[0, 5, 158, 183, 158, 198, 57, 0, 0],
            &[0, 135, 105, 0, 0, 19, 204, 12, 0],
            &[3, 210, 2, 0, 0, 0, 119, 96, 0],
            &[45, 165, 0, 0, 0, 0, 60, 150, 0],
            &[68, 141, 0, 0, 0, 0, 37, 173, 0],
            &[70, 142, 0, 0, 0, 0, 39, 172, 0],
            &[45, 167, 0, 0, 0, 0, 63, 148, 0],
            &[4, 211, 2, 0, 0, 0, 123, 92, 0],
            &[0, 137, 109, 0, 0, 24, 205, 10, 0],
            &[0, 7, 161, 188, 171, 255, 53, 0, 0],
            &[0, 0, 0, 13, 26, 177, 50, 0, 0],
            &[0, 0, 0, 0, 0, 42, 186, 0, 0],
            &[0, 0, 0, 0, 0, 0, 88, 16, 0],
        ]),
        // letter: 'R' / 0x52
        'R' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 203, 183, 189, 186, 61, 0, 0],
            &[0, 135, 72, 0, 0, 29, 217, 19, 0],
            &[0, 135, 72, 0, 0, 0, 151, 69, 0],
            &[0, 135, 72, 0, 0, 0, 165, 53, 0],
            &[0, 135, 74, 3, 16, 96, 172, 1, 0],
            &[0, 135, 203, 183, 213, 139, 3, 0, 0],
            &[0, 135, 72, 0, 24, 198, 4, 0, 0],
            &[0, 135, 72, 0, 0, 115, 113, 0, 0],
            &[0, 135, 72, 0, 0, 5, 200, 24, 0],
            &[0, 135, 72, 0, 0, 0, 68, 163, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'S' / 0x53
        'S' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 29, 11, 0, 0, 0],
            &[0, 4, 149, 198, 160, 178, 190, 6, 0],
            &[0, 102, 141, 0, 0, 0, 3, 0, 0],
            &[0, 145, 71, 0, 0, 0, 0, 0, 0],
            &[0, 95, 166, 4, 0, 0, 0, 0, 0],
            &[0, 0, 123, 202, 110, 13, 0, 0, 0],
            &[0, 0, 0, 12, 108, 205, 109, 0, 0],
            &[0, 0, 0, 0, 0, 9, 196, 52, 0],
            &[0, 0, 0, 0, 0, 0, 126, 85, 0],
            &[0, 4, 0, 0, 0, 7, 195, 33, 0],
            &[0, 145, 187, 159, 160, 195, 84, 0, 0],
            &[0, 0, 3, 25, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'T' / 0x54
        'T' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[61, 186, 186, 200, 227, 186, 186, 138, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'U' / 0x55
        'U' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 0, 142, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 146, 61, 0],
            &[0, 189, 21, 0, 0, 0, 172, 38, 0],
            &[0, 119, 117, 0, 0, 28, 205, 0, 0],
            &[0, 9, 170, 183, 165, 201, 52, 0, 0],
            &[0, 0, 0, 15, 25, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'V' / 0x56
        'V' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[94, 121, 0, 0, 0, 0, 22, 197, 0],
            &[16, 199, 0, 0, 0, 0, 100, 117, 0],
            &[0, 180, 33, 0, 0, 0, 182, 33, 0],
            &[0, 95, 115, 0, 0, 15, 197, 0, 0],
            &[0, 16, 192, 0, 0, 90, 119, 0, 0],
            &[0, 0, 179, 27, 0, 171, 34, 0, 0],
            &[0, 0, 95, 109, 9, 194, 0, 0, 0],
            &[0, 0, 16, 184, 78, 121, 0, 0, 0],
            &[0, 0, 0, 175, 165, 36, 0, 0, 0],
            &[0, 0, 0, 95, 205, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'W' / 0x57
        'W' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[119, 76, 0, 0, 0, 0, 0, 196, 0],
            &[85, 110, 0, 0, 0, 0, 6, 190, 0],
            &[50, 144, 0, 27, 75, 0, 38, 158, 0],
            &[16, 177, 0, 104, 187, 0, 73, 124, 0],
            &[0, 192, 0, 154, 143, 16, 107, 90, 0],
            &[0, 191, 0, 161, 85, 73, 141, 56, 0],
            &[0, 167, 48, 137, 26, 132, 170, 22, 0],
            &[0, 132, 132, 79, 0, 161, 181, 0, 0],
            &[0, 98, 208, 21, 0, 154, 170, 0, 0],
            &[0, 63, 218, 0, 0, 106, 175, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'X' / 0x58
        'X' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[2, 193, 28, 0, 0, 0, 150, 68, 0],
            &[0, 53, 165, 0, 0, 51, 164, 0, 0],
            &[0, 0, 150, 62, 1, 185, 24, 0, 0],
            &[0, 0, 16, 185, 102, 108, 0, 0, 0],
            &[0, 0, 0, 99, 205, 2, 0, 0, 0],
            &[0, 0, 0, 153, 199, 26, 0, 0, 0],
            &[0, 0, 59, 152, 43, 164, 0, 0, 0],
            &[0, 5, 192, 18, 0, 141, 70, 0, 0],
            &[0, 119, 100, 0, 0, 13, 194, 8, 0],
            &[32, 189, 1, 0, 0, 0, 90, 129, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Y' / 0x59
        'Y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[38, 186, 0, 0, 0, 0, 80, 140, 0],
            &[0, 157, 62, 0, 0, 0, 193, 20, 0],
            &[0, 30, 183, 0, 0, 78, 130, 0, 0],
            &[0, 0, 144, 62, 0, 185, 15, 0, 0],
            &[0, 0, 22, 179, 75, 119, 0, 0, 0],
            &[0, 0, 0, 132, 204, 10, 0, 0, 0],
            &[0, 0, 0, 47, 156, 0, 0, 0, 0],
            &[0, 0, 0, 45, 155, 0, 0, 0, 0],
            &[0, 0, 0, 45, 155, 0, 0, 0, 0],
            &[0, 0, 0, 45, 155, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Z' / 0x5a
        'Z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 182, 182, 182, 185, 223, 0, 0],
            &[0, 0, 0, 0, 0, 96, 116, 0, 0],
            &[0, 0, 0, 0, 6, 197, 8, 0, 0],
            &[0, 0, 0, 0, 110, 103, 0, 0, 0],
            &[0, 0, 0, 12, 197, 4, 0, 0, 0],
            &[0, 0, 0, 125, 90, 0, 0, 0, 0],
            &[0, 0, 18, 195, 2, 0, 0, 0, 0],
            &[0, 0, 139, 77, 0, 0, 0, 0, 0],
            &[0, 27, 189, 0, 0, 0, 0, 0, 0],
            &[0, 126, 219, 189, 189, 189, 189, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '[' / 0x5b
        '[' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 166, 182, 171, 56, 0, 0, 0],
            &[0, 0, 167, 33, 0, 0, 0, 0, 0],
            &[0, 0, 167, 33, 0, 0, 0, 0, 0],
            &[0, 0, 167, 33, 0, 0, 0, 0, 0],
            &[0, 0, 167, 33, 0, 0, 0, 0, 0],
            &[0, 0, 167, 33, 0, 0, 0, 0, 0],
            &[0, 0, 167, 33, 0, 0, 0, 0, 0],
            &[0, 0, 167, 33, 0, 0, 0, 0, 0],
            &[0, 0, 167, 33, 0, 0, 0, 0, 0],
            &[0, 0, 167, 33, 0, 0, 0, 0, 0],
            &[0, 0, 167, 33, 0, 0, 0, 0, 0],
            &[0, 0, 167, 138, 121, 39, 0, 0, 0],
            &[0, 0, 35, 54, 54, 17, 0, 0, 0],
        ]),
        // letter: '\' / 0x5c
        '\\' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 198, 2, 0, 0, 0, 0, 0],
            &[0, 0, 135, 69, 0, 0, 0, 0, 0],
            &[0, 0, 38, 165, 0, 0, 0, 0, 0],
            &[0, 0, 0, 188, 16, 0, 0, 0, 0],
            &[0, 0, 0, 99, 104, 0, 0, 0, 0],
            &[0, 0, 0, 13, 190, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 43, 0, 0, 0],
            &[0, 0, 0, 0, 64, 140, 0, 0, 0],
            &[0, 0, 0, 0, 1, 198, 4, 0, 0],
            &[0, 0, 0, 0, 0, 125, 78, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ']' / 0x5d
        ']' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 16, 16, 1, 0, 0, 0],
            &[0, 0, 145, 158, 228, 17, 0, 0, 0],
            &[0, 0, 0, 0, 183, 17, 0, 0, 0],
            &[0, 0, 0, 0, 183, 17, 0, 0, 0],
            &[0, 0, 0, 0, 183, 17, 0, 0, 0],
            &[0, 0, 0, 0, 183, 17, 0, 0, 0],
            &[0, 0, 0, 0, 183, 17, 0, 0, 0],
            &[0, 0, 0, 0, 183, 17, 0, 0, 0],
            &[0, 0, 0, 0, 183, 17, 0, 0, 0],
            &[0, 0, 0, 0, 183, 17, 0, 0, 0],
            &[0, 0, 0, 0, 183, 17, 0, 0, 0],
            &[0, 0, 0, 0, 183, 17, 0, 0, 0],
            &[0, 0, 123, 135, 221, 17, 0, 0, 0],
            &[0, 0, 33, 36, 36, 2, 0, 0, 0],
        ]),
        // letter: '^' / 0x5e
        '^' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 7, 0, 0, 0, 0],
            &[0, 0, 0, 89, 181, 0, 0, 0, 0],
            &[0, 0, 2, 174, 122, 61, 0, 0, 0],
            &[0, 0, 92, 89, 12, 171, 0, 0, 0],
            &[0, 3, 177, 2, 0, 119, 66, 0, 0],
            &[0, 94, 91, 0, 0, 12, 176, 0, 0],
            &[4, 181, 3, 0, 0, 0, 119, 72, 0],
            &[7, 25, 0, 0, 0, 0, 7, 26, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '_' / 0x5f
        '_' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[189, 189, 189, 189, 189, 189, 189, 189, 75],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '`' / 0x60
        '`' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 160, 10, 0, 0, 0, 0],
            &[0, 0, 0, 68, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'a' / 0x61
        'a' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 67, 126, 139, 91, 3, 0, 0],
            &[0, 22, 122, 48, 36, 122, 150, 0, 0],
            &[0, 0, 0, 0, 0, 0, 197, 0, 0],
            &[0, 0, 3, 50, 80, 91, 217, 0, 0],
            &[0, 57, 190, 124, 84, 66, 208, 0, 0],
            &[0, 192, 19, 0, 0, 0, 212, 0, 0],
            &[0, 197, 13, 0, 0, 70, 245, 0, 0],
            &[0, 76, 196, 147, 163, 113, 178, 0, 0],
            &[0, 0, 4, 26, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'b' / 0x62
        'b' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 114, 14, 0, 0, 0, 0, 0, 0],
            &[0, 178, 22, 0, 0, 0, 0, 0, 0],
            &[0, 178, 22, 0, 0, 0, 0, 0, 0],
            &[0, 178, 24, 94, 142, 109, 10, 0, 0],
            &[0, 178, 156, 89, 38, 106, 191, 2, 0],
            &[0, 178, 111, 0, 0, 0, 156, 71, 0],
            &[0, 178, 44, 0, 0, 0, 97, 123, 0],
            &[0, 178, 29, 0, 0, 0, 87, 131, 0],
            &[0, 178, 57, 0, 0, 0, 119, 101, 0],
            &[0, 178, 152, 0, 0, 12, 207, 25, 0],
            &[0, 175, 86, 176, 153, 201, 86, 0, 0],
            &[0, 0, 0, 6, 29, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'c' / 0x63
        'c' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 80, 136, 137, 86, 0, 0],
            &[0, 5, 183, 132, 51, 47, 93, 0, 0],
            &[0, 105, 127, 0, 0, 0, 0, 0, 0],
            &[0, 174, 38, 0, 0, 0, 0, 0, 0],
            &[0, 190, 20, 0, 0, 0, 0, 0, 0],
            &[0, 161, 56, 0, 0, 0, 0, 0, 0],
            &[0, 68, 180, 5, 0, 0, 0, 0, 0],
            &[0, 0, 105, 199, 157, 163, 180, 0, 0],
            &[0, 0, 0, 3, 26, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'd' / 0x64
        'd' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 114, 18, 0],
            &[0, 0, 0, 0, 0, 0, 179, 28, 0],
            &[0, 0, 0, 0, 0, 0, 179, 28, 0],
            &[0, 0, 55, 134, 128, 37, 178, 28, 0],
            &[0, 84, 191, 55, 41, 155, 201, 28, 0],
            &[0, 208, 23, 0, 0, 10, 241, 28, 0],
            &[13, 205, 0, 0, 0, 0, 193, 28, 0],
            &[25, 188, 0, 0, 0, 0, 184, 28, 0],
            &[7, 210, 0, 0, 0, 0, 212, 28, 0],
            &[0, 180, 62, 0, 0, 57, 243, 28, 0],
            &[0, 30, 195, 165, 161, 144, 153, 28, 0],
            &[0, 0, 0, 25, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'e' / 0x65
        'e' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 115, 142, 99, 7, 0, 0],
            &[0, 39, 199, 77, 34, 99, 176, 0, 0],
            &[0, 175, 47, 0, 0, 0, 150, 61, 0],
            &[0, 224, 76, 76, 76, 76, 147, 108, 0],
            &[5, 225, 95, 95, 95, 95, 95, 44, 0],
            &[0, 211, 3, 0, 0, 0, 0, 0, 0],
            &[0, 131, 112, 0, 0, 0, 1, 4, 0],
            &[0, 5, 141, 184, 150, 165, 184, 30, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'f' / 0x66
        'f' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 163, 173, 126, 0],
            &[0, 0, 0, 45, 197, 25, 1, 19, 0],
            &[0, 0, 0, 98, 114, 0, 0, 0, 0],
            &[0, 80, 101, 169, 172, 114, 114, 49, 0],
            &[0, 56, 64, 141, 142, 64, 64, 27, 0],
            &[0, 0, 0, 102, 104, 0, 0, 0, 0],
            &[0, 0, 0, 102, 104, 0, 0, 0, 0],
            &[0, 0, 0, 102, 104, 0, 0, 0, 0],
            &[0, 0, 0, 102, 104, 0, 0, 0, 0],
            &[0, 0, 0, 102, 104, 0, 0, 0, 0],
            &[0, 0, 0, 102, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'g' / 0x67
        'g' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 56, 137, 134, 43, 65, 12, 0],
            &[0, 80, 188, 55, 39, 143, 205, 28, 0],
            &[0, 205, 21, 0, 0, 3, 226, 28, 0],
            &[14, 203, 0, 0, 0, 0, 175, 28, 0],
            &[26, 188, 0, 0, 0, 0, 167, 28, 0],
            &[6, 213, 0, 0, 0, 0, 194, 28, 0],
            &[0, 177, 69, 0, 0, 45, 242, 28, 0],
            &[0, 27, 194, 167, 162, 154, 171, 28, 0],
            &[0, 0, 0, 24, 17, 0, 176, 23, 0],
            &[0, 0, 0, 0, 0, 6, 209, 2, 0],
            &[0, 8, 70, 103, 98, 38, 0, 0, 0],
        ]),
        // letter: 'h' / 0x68
        'h' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 114, 18, 0, 0, 0, 0, 0, 0],
            &[0, 178, 29, 0, 0, 0, 0, 0, 0],
            &[0, 178, 29, 0, 0, 0, 0, 0, 0],
            &[0, 178, 30, 86, 139, 109, 10, 0, 0],
            &[0, 178, 147, 81, 34, 104, 190, 0, 0],
            &[0, 178, 98, 0, 0, 0, 190, 24, 0],
            &[0, 178, 38, 0, 0, 0, 162, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'i' / 0x69
        'i' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 84, 0, 0, 0, 0],
            &[0, 0, 0, 42, 205, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 114, 114, 98, 0, 0, 0, 0],
            &[0, 17, 44, 57, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 171, 159, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'j' / 0x6a
        'j' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 24, 0, 0],
            &[0, 0, 0, 0, 0, 176, 73, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 114, 114, 114, 31, 0, 0],
            &[0, 0, 2, 39, 52, 165, 71, 0, 0],
            &[0, 0, 0, 0, 0, 136, 71, 0, 0],
            &[0, 0, 0, 0, 0, 136, 71, 0, 0],
            &[0, 0, 0, 0, 0, 136, 71, 0, 0],
            &[0, 0, 0, 0, 0, 136, 71, 0, 0],
            &[0, 0, 0, 0, 0, 136, 71, 0, 0],
            &[0, 0, 0, 0, 0, 136, 71, 0, 0],
            &[0, 0, 0, 0, 0, 137, 71, 0, 0],
            &[0, 0, 0, 0, 0, 169, 50, 0, 0],
            &[0, 0, 75, 105, 83, 6, 0, 0, 0],
        ]),
        // letter: 'k' / 0x6b
        'k' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 45, 87, 0, 0, 0, 0, 0, 0],
            &[0, 71, 136, 0, 0, 0, 0, 0, 0],
            &[0, 71, 136, 0, 0, 0, 0, 0, 0],
            &[0, 71, 136, 0, 0, 0, 72, 41, 0],
            &[0, 71, 136, 0, 0, 85, 167, 3, 0],
            &[0, 71, 136, 0, 75, 176, 5, 0, 0],
            &[0, 71, 133, 66, 208, 9, 0, 0, 0],
            &[0, 71, 176, 165, 168, 66, 0, 0, 0],
            &[0, 71, 187, 5, 15, 198, 21, 0, 0],
            &[0, 71, 136, 0, 0, 59, 178, 1, 0],
            &[0, 71, 136, 0, 0, 0, 127, 114, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'l' / 0x6c
        'l' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 159, 163, 140, 0, 0, 0, 0],
            &[0, 0, 0, 8, 208, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 171, 159, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'm' / 0x6d
        'm' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[18, 50, 104, 121, 4, 111, 115, 3, 0],
            &[40, 212, 75, 149, 178, 91, 142, 97, 0],
            &[40, 190, 0, 51, 196, 0, 48, 142, 0],
            &[40, 157, 0, 35, 162, 0, 35, 154, 0],
            &[40, 148, 0, 34, 155, 0, 35, 154, 0],
            &[40, 148, 0, 34, 155, 0, 35, 154, 0],
            &[40, 148, 0, 34, 155, 0, 35, 154, 0],
            &[40, 148, 0, 34, 155, 0, 35, 154, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'n' / 0x6e
        'n' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 77, 2, 88, 139, 112, 13, 0, 0],
            &[0, 178, 143, 82, 35, 97, 194, 0, 0],
            &[0, 178, 98, 0, 0, 0, 186, 26, 0],
            &[0, 178, 43, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'o' / 0x6f
        'o' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 37, 125, 140, 85, 2, 0, 0],
            &[0, 65, 192, 62, 36, 119, 162, 0, 0],
            &[0, 199, 26, 0, 0, 0, 162, 57, 0],
            &[12, 201, 0, 0, 0, 0, 92, 118, 0],
            &[24, 183, 0, 0, 0, 0, 81, 130, 0],
            &[4, 208, 0, 0, 0, 0, 116, 98, 0],
            &[0, 156, 80, 0, 0, 14, 203, 19, 0],
            &[0, 13, 167, 168, 150, 194, 67, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'p' / 0x70
        'p' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 78, 1, 86, 140, 112, 12, 0, 0],
            &[0, 178, 136, 87, 37, 99, 192, 3, 0],
            &[0, 178, 105, 0, 0, 0, 149, 74, 0],
            &[0, 178, 44, 0, 0, 0, 96, 123, 0],
            &[0, 178, 32, 0, 0, 0, 87, 129, 0],
            &[0, 178, 63, 0, 0, 0, 117, 100, 0],
            &[0, 178, 153, 0, 0, 9, 203, 26, 0],
            &[0, 178, 90, 173, 153, 198, 88, 0, 0],
            &[0, 178, 25, 1, 25, 5, 0, 0, 0],
            &[0, 178, 25, 0, 0, 0, 0, 0, 0],
            &[0, 64, 9, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'q' / 0x71
        'q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 55, 137, 131, 37, 65, 12, 0],
            &[0, 79, 184, 52, 49, 158, 196, 28, 0],
            &[0, 205, 19, 0, 0, 12, 242, 28, 0],
            &[14, 203, 0, 0, 0, 0, 197, 28, 0],
            &[26, 188, 0, 0, 0, 0, 184, 28, 0],
            &[6, 212, 0, 0, 0, 0, 214, 28, 0],
            &[0, 178, 64, 0, 0, 64, 241, 28, 0],
            &[0, 29, 194, 164, 163, 140, 176, 28, 0],
            &[0, 0, 0, 25, 14, 0, 179, 28, 0],
            &[0, 0, 0, 0, 0, 0, 179, 28, 0],
            &[0, 0, 0, 0, 0, 0, 64, 10, 0],
        ]),
        // letter: 'r' / 0x72
        'r' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 71, 114, 114, 22, 75, 138, 109, 0],
            &[0, 23, 49, 158, 149, 127, 48, 62, 0],
            &[0, 0, 0, 127, 168, 0, 0, 0, 0],
            &[0, 0, 0, 127, 97, 0, 0, 0, 0],
            &[0, 0, 0, 127, 80, 0, 0, 0, 0],
            &[0, 0, 0, 127, 79, 0, 0, 0, 0],
            &[0, 0, 0, 127, 79, 0, 0, 0, 0],
            &[0, 132, 165, 215, 200, 163, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 's' / 0x73
        's' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 123, 144, 120, 41, 0, 0],
            &[0, 17, 205, 64, 34, 63, 61, 0, 0],
            &[0, 50, 165, 0, 0, 0, 0, 0, 0],
            &[0, 3, 171, 141, 34, 0, 0, 0, 0],
            &[0, 0, 0, 61, 163, 174, 28, 0, 0],
            &[0, 0, 0, 0, 0, 61, 186, 0, 0],
            &[0, 6, 0, 0, 0, 29, 192, 0, 0],
            &[0, 76, 188, 159, 155, 195, 59, 0, 0],
            &[0, 0, 1, 22, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 't' / 0x74
        't' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 61, 1, 0, 0, 0, 0],
            &[0, 0, 0, 183, 4, 0, 0, 0, 0],
            &[1, 94, 110, 226, 117, 114, 114, 0, 0],
            &[1, 64, 64, 216, 67, 64, 64, 0, 0],
            &[0, 0, 0, 203, 4, 0, 0, 0, 0],
            &[0, 0, 0, 203, 4, 0, 0, 0, 0],
            &[0, 0, 0, 203, 4, 0, 0, 0, 0],
            &[0, 0, 0, 201, 10, 0, 0, 0, 0],
            &[0, 0, 0, 171, 60, 0, 0, 0, 0],
            &[0, 0, 0, 50, 205, 155, 160, 7, 0],
            &[0, 0, 0, 0, 3, 29, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'u' / 0x75
        'u' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 88, 5, 0, 0, 0, 80, 12, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 180, 28, 0],
            &[0, 190, 19, 0, 0, 0, 204, 28, 0],
            &[0, 148, 79, 0, 0, 51, 240, 28, 0],
            &[0, 29, 199, 158, 157, 134, 151, 28, 0],
            &[0, 0, 0, 22, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'v' / 0x76
        'v' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[2, 93, 0, 0, 0, 0, 51, 46, 0],
            &[0, 184, 27, 0, 0, 0, 180, 34, 0],
            &[0, 90, 119, 0, 0, 23, 186, 0, 0],
            &[0, 9, 197, 0, 0, 112, 93, 0, 0],
            &[0, 0, 153, 51, 0, 191, 10, 0, 0],
            &[0, 0, 56, 145, 42, 153, 0, 0, 0],
            &[0, 0, 0, 193, 139, 56, 0, 0, 0],
            &[0, 0, 0, 118, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'w' / 0x77
        'w' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[75, 11, 0, 0, 0, 0, 0, 76, 8],
            &[130, 63, 0, 3, 13, 0, 0, 190, 0],
            &[78, 117, 0, 93, 210, 0, 7, 184, 0],
            &[26, 170, 0, 165, 153, 37, 54, 139, 0],
            &[0, 197, 6, 178, 71, 113, 105, 89, 0],
            &[0, 176, 85, 116, 7, 177, 151, 39, 0],
            &[0, 124, 185, 40, 0, 169, 180, 1, 0],
            &[0, 72, 216, 0, 0, 100, 190, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'x' / 0x78
        'x' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 88, 16, 0, 0, 0, 84, 19, 0],
            &[0, 64, 167, 0, 0, 69, 161, 0, 0],
            &[0, 0, 131, 96, 22, 193, 12, 0, 0],
            &[0, 0, 3, 183, 187, 48, 0, 0, 0],
            &[0, 0, 0, 135, 210, 12, 0, 0, 0],
            &[0, 0, 64, 162, 65, 161, 0, 0, 0],
            &[0, 21, 195, 14, 0, 133, 97, 0, 0],
            &[1, 178, 55, 0, 0, 4, 188, 41, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'y' / 0x79
        'y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 92, 0, 0, 0, 0, 49, 46, 0],
            &[0, 170, 35, 0, 0, 0, 175, 34, 0],
            &[0, 70, 133, 0, 0, 18, 187, 0, 0],
            &[0, 1, 195, 3, 0, 104, 96, 0, 0],
            &[0, 0, 122, 75, 0, 184, 12, 0, 0],
            &[0, 0, 25, 170, 32, 158, 0, 0, 0],
            &[0, 0, 0, 169, 138, 63, 0, 0, 0],
            &[0, 0, 0, 73, 217, 1, 0, 0, 0],
            &[0, 0, 0, 73, 126, 0, 0, 0, 0],
            &[0, 0, 0, 172, 29, 0, 0, 0, 0],
            &[0, 97, 77, 3, 0, 0, 0, 0, 0],
        ]),
        // letter: 'z' / 0x7a
        'z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 114, 114, 114, 114, 105, 0, 0],
            &[0, 14, 53, 53, 53, 118, 167, 0, 0],
            &[0, 0, 0, 0, 19, 195, 15, 0, 0],
            &[0, 0, 0, 0, 172, 56, 0, 0, 0],
            &[0, 0, 0, 107, 120, 0, 0, 0, 0],
            &[0, 0, 45, 179, 1, 0, 0, 0, 0],
            &[0, 9, 190, 24, 0, 0, 0, 0, 0],
            &[0, 123, 222, 182, 182, 182, 172, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '{' / 0x7b
        '{' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 129, 190, 52, 0, 0],
            &[0, 0, 0, 38, 188, 1, 0, 0, 0],
            &[0, 0, 0, 59, 144, 0, 0, 0, 0],
            &[0, 0, 0, 59, 144, 0, 0, 0, 0],
            &[0, 0, 0, 69, 137, 0, 0, 0, 0],
            &[0, 0, 65, 176, 54, 0, 0, 0, 0],
            &[0, 0, 109, 174, 33, 0, 0, 0, 0],
            &[0, 0, 0, 78, 132, 0, 0, 0, 0],
            &[0, 0, 0, 59, 144, 0, 0, 0, 0],
            &[0, 0, 0, 59, 144, 0, 0, 0, 0],
            &[0, 0, 0, 48, 164, 0, 0, 0, 0],
            &[0, 0, 0, 2, 178, 154, 37, 0, 0],
            &[0, 0, 0, 0, 0, 30, 15, 0, 0],
        ]),
        // letter: '|' / 0x7c
        '|' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 90, 0, 0, 0, 0],
            &[0, 0, 0, 34, 140, 0, 0, 0, 0],
            &[0, 0, 0, 34, 140, 0, 0, 0, 0],
            &[0, 0, 0, 34, 140, 0, 0, 0, 0],
            &[0, 0, 0, 34, 140, 0, 0, 0, 0],
            &[0, 0, 0, 34, 140, 0, 0, 0, 0],
            &[0, 0, 0, 34, 140, 0, 0, 0, 0],
            &[0, 0, 0, 34, 140, 0, 0, 0, 0],
            &[0, 0, 0, 34, 140, 0, 0, 0, 0],
            &[0, 0, 0, 34, 140, 0, 0, 0, 0],
            &[0, 0, 0, 34, 140, 0, 0, 0, 0],
            &[0, 0, 0, 34, 140, 0, 0, 0, 0],
            &[0, 0, 0, 34, 140, 0, 0, 0, 0],
            &[0, 0, 0, 12, 50, 0, 0, 0, 0],
        ]),
        // letter: '}' / 0x7d
        '}' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 160, 178, 27, 0, 0, 0, 0],
            &[0, 0, 0, 80, 140, 0, 0, 0, 0],
            &[0, 0, 0, 42, 162, 0, 0, 0, 0],
            &[0, 0, 0, 41, 162, 0, 0, 0, 0],
            &[0, 0, 0, 36, 170, 0, 0, 0, 0],
            &[0, 0, 0, 2, 185, 94, 16, 0, 0],
            &[0, 0, 0, 0, 133, 151, 31, 0, 0],
            &[0, 0, 0, 28, 182, 0, 0, 0, 0],
            &[0, 0, 0, 41, 162, 0, 0, 0, 0],
            &[0, 0, 0, 41, 162, 0, 0, 0, 0],
            &[0, 0, 0, 65, 151, 0, 0, 0, 0],
            &[0, 0, 122, 205, 52, 0, 0, 0, 0],
            &[0, 0, 40, 7, 0, 0, 0, 0, 0],
        ]),
        // letter: '~' / 0x7e
        '~' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 106, 62, 0, 0, 51, 29, 0],
            &[0, 141, 56, 122, 117, 0, 149, 20, 0],
            &[0, 139, 0, 0, 88, 163, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ' ' / 0xa0
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¡' / 0xa1
        '¡' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 87, 0, 0, 0, 0],
            &[0, 0, 0, 71, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 33, 110, 0, 0, 0, 0],
            &[0, 0, 0, 47, 146, 0, 0, 0, 0],
            &[0, 0, 0, 51, 150, 0, 0, 0, 0],
            &[0, 0, 0, 56, 155, 0, 0, 0, 0],
            &[0, 0, 0, 60, 159, 0, 0, 0, 0],
            &[0, 0, 0, 65, 163, 0, 0, 0, 0],
            &[0, 0, 0, 69, 168, 0, 0, 0, 0],
            &[0, 0, 0, 43, 101, 0, 0, 0, 0],
        ]),
        // letter: '¢' / 0xa2
        '¢' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 168, 0, 0, 0, 0],
            &[0, 0, 45, 174, 190, 180, 151, 0, 0],
            &[0, 26, 205, 32, 0, 0, 2, 0, 0],
            &[0, 125, 91, 0, 0, 0, 0, 0, 0],
            &[0, 169, 39, 0, 0, 0, 0, 0, 0],
            &[0, 166, 42, 0, 0, 0, 0, 0, 0],
            &[0, 119, 104, 0, 0, 0, 0, 0, 0],
            &[0, 19, 208, 68, 2, 7, 47, 0, 0],
            &[0, 0, 26, 138, 226, 176, 106, 0, 0],
            &[0, 0, 0, 0, 164, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '£' / 0xa3
        '£' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 20, 0, 0, 0],
            &[0, 0, 0, 97, 192, 157, 188, 37, 0],
            &[0, 0, 12, 200, 1, 0, 0, 1, 0],
            &[0, 0, 47, 157, 0, 0, 0, 0, 0],
            &[0, 0, 54, 149, 0, 0, 0, 0, 0],
            &[0, 33, 99, 173, 56, 56, 4, 0, 0],
            &[0, 61, 136, 192, 104, 104, 7, 0, 0],
            &[0, 0, 54, 149, 0, 0, 0, 0, 0],
            &[0, 0, 68, 138, 0, 0, 0, 0, 0],
            &[0, 0, 149, 63, 0, 0, 0, 0, 0],
            &[0, 156, 238, 189, 189, 189, 189, 85, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¤' / 0xa4
        '¤' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 0, 0, 0, 0, 6, 2, 0],
            &[0, 169, 98, 159, 161, 113, 172, 42, 0],
            &[0, 48, 180, 9, 0, 83, 150, 0, 0],
            &[0, 129, 36, 0, 0, 0, 166, 0, 0],
            &[0, 128, 36, 0, 0, 0, 169, 0, 0],
            &[0, 42, 184, 15, 0, 93, 149, 0, 0],
            &[0, 162, 100, 165, 167, 113, 176, 39, 0],
            &[0, 9, 0, 0, 0, 0, 8, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¥' / 0xa5
        '¥' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[49, 169, 0, 0, 0, 0, 66, 152, 0],
            &[0, 160, 57, 0, 0, 1, 191, 22, 0],
            &[0, 27, 187, 0, 0, 94, 118, 0, 0],
            &[0, 0, 129, 83, 9, 192, 7, 0, 0],
            &[0, 0, 11, 193, 127, 82, 0, 0, 0],
            &[0, 68, 157, 211, 246, 157, 133, 0, 0],
            &[0, 0, 0, 49, 151, 0, 0, 0, 0],
            &[0, 68, 157, 176, 215, 157, 133, 0, 0],
            &[0, 0, 0, 49, 151, 0, 0, 0, 0],
            &[0, 0, 0, 49, 151, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¦' / 0xa6
        '¦' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 91, 0, 0, 0, 0],
            &[0, 0, 0, 38, 137, 0, 0, 0, 0],
            &[0, 0, 0, 38, 137, 0, 0, 0, 0],
            &[0, 0, 0, 38, 137, 0, 0, 0, 0],
            &[0, 0, 0, 38, 137, 0, 0, 0, 0],
            &[0, 0, 0, 22, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 110, 0, 0, 0, 0],
            &[0, 0, 0, 38, 137, 0, 0, 0, 0],
            &[0, 0, 0, 38, 137, 0, 0, 0, 0],
            &[0, 0, 0, 38, 137, 0, 0, 0, 0],
            &[0, 0, 0, 17, 60, 0, 0, 0, 0],
        ]),
        // letter: '§' / 0xa7
        '§' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 38, 138, 157, 143, 59, 0, 0],
            &[0, 5, 195, 25, 0, 20, 41, 0, 0],
            &[0, 14, 199, 8, 0, 0, 0, 0, 0],
            &[0, 0, 76, 222, 125, 25, 0, 0, 0],
            &[0, 5, 165, 44, 56, 171, 88, 0, 0],
            &[0, 37, 144, 0, 0, 2, 190, 0, 0],
            &[0, 4, 179, 94, 5, 40, 155, 0, 0],
            &[0, 0, 5, 94, 178, 219, 17, 0, 0],
            &[0, 0, 0, 0, 0, 81, 162, 0, 0],
            &[0, 3, 0, 0, 0, 20, 181, 0, 0],
            &[0, 61, 172, 150, 150, 173, 46, 0, 0],
            &[0, 0, 0, 6, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¨' / 0xa8
        '¨' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 0, 0, 16, 0, 0, 0],
            &[0, 0, 196, 54, 0, 203, 46, 0, 0],
            &[0, 0, 11, 0, 0, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '©' / 0xa9
        '©' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 56, 127, 129, 104, 10, 0, 0],
            &[0, 108, 91, 18, 52, 52, 155, 14, 0],
            &[30, 117, 47, 145, 76, 60, 24, 124, 0],
            &[91, 44, 142, 6, 0, 0, 0, 133, 0],
            &[90, 47, 143, 5, 0, 0, 0, 132, 0],
            &[28, 125, 48, 148, 89, 54, 28, 116, 0],
            &[0, 105, 103, 15, 36, 55, 146, 8, 0],
            &[0, 0, 48, 121, 124, 83, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ª' / 0xaa
        'ª' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 17, 0, 0, 0, 0],
            &[0, 0, 48, 130, 129, 143, 0, 0, 0],
            &[0, 0, 0, 15, 49, 175, 0, 0, 0],
            &[0, 0, 97, 124, 68, 177, 0, 0, 0],
            &[0, 0, 162, 27, 22, 208, 0, 0, 0],
            &[0, 0, 31, 130, 97, 69, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '«' / 0xab
        '«' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 75, 14, 7, 81, 0, 0],
            &[0, 0, 55, 166, 1, 157, 61, 0, 0],
            &[0, 26, 190, 13, 115, 113, 0, 0, 0],
            &[0, 86, 146, 0, 196, 36, 0, 0, 0],
            &[0, 0, 143, 78, 28, 185, 10, 0, 0],
            &[0, 0, 3, 165, 13, 56, 129, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¬' / 0xac
        '¬' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[6, 71, 71, 71, 71, 71, 71, 6, 0],
            &[9, 100, 100, 100, 100, 100, 189, 25, 0],
            &[0, 0, 0, 0, 0, 0, 146, 25, 0],
            &[0, 0, 0, 0, 0, 0, 146, 25, 0],
            &[0, 0, 0, 0, 0, 0, 26, 4, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '®' / 0xae
        '®' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 56, 127, 129, 104, 10, 0, 0],
            &[0, 108, 96, 92, 79, 49, 155, 14, 0],
            &[30, 117, 15, 131, 53, 149, 24, 124, 0],
            &[91, 44, 15, 125, 45, 133, 0, 133, 0],
            &[90, 47, 15, 165, 159, 30, 0, 132, 0],
            &[28, 125, 15, 113, 12, 140, 28, 116, 0],
            &[0, 105, 107, 31, 0, 82, 148, 8, 0],
            &[0, 0, 48, 121, 124, 83, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¯' / 0xaf
        '¯' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 111, 111, 111, 14, 0, 0],
            &[0, 0, 44, 60, 60, 60, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '°' / 0xb0
        '°' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 15, 0, 0, 0, 0],
            &[0, 0, 64, 154, 128, 134, 0, 0, 0],
            &[0, 0, 160, 0, 0, 113, 34, 0, 0],
            &[0, 0, 151, 7, 0, 127, 25, 0, 0],
            &[0, 0, 38, 154, 144, 108, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '±' / 0xb1
        '±' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 27, 0, 0, 0, 0],
            &[0, 0, 0, 38, 133, 0, 0, 0, 0],
            &[0, 0, 0, 38, 133, 0, 0, 0, 0],
            &[0, 73, 74, 101, 169, 74, 74, 26, 0],
            &[0, 91, 93, 117, 178, 93, 93, 32, 0],
            &[0, 0, 0, 38, 133, 0, 0, 0, 0],
            &[0, 0, 0, 38, 133, 0, 0, 0, 0],
            &[0, 0, 0, 10, 35, 0, 0, 0, 0],
            &[0, 163, 171, 171, 171, 171, 171, 60, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '²' / 0xb2
        '²' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 71, 155, 162, 120, 0, 0, 0],
            &[0, 0, 13, 0, 0, 180, 4, 0, 0],
            &[0, 0, 0, 0, 8, 171, 0, 0, 0],
            &[0, 0, 0, 3, 152, 44, 0, 0, 0],
            &[0, 0, 3, 149, 49, 0, 0, 0, 0],
            &[0, 0, 155, 160, 112, 112, 27, 0, 0],
            &[0, 0, 36, 45, 45, 45, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '³' / 0xb3
        '³' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 86, 153, 161, 112, 0, 0, 0],
            &[0, 0, 10, 0, 0, 185, 0, 0, 0],
            &[0, 0, 0, 56, 89, 131, 0, 0, 0],
            &[0, 0, 0, 80, 104, 156, 3, 0, 0],
            &[0, 0, 0, 0, 0, 147, 29, 0, 0],
            &[0, 0, 120, 95, 106, 170, 1, 0, 0],
            &[0, 0, 9, 54, 46, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '´' / 0xb4
        '´' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 159, 14, 0, 0, 0],
            &[0, 0, 0, 149, 74, 0, 0, 0, 0],
            &[0, 0, 30, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'µ' / 0xb5
        'µ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 88, 5, 0, 0, 0, 80, 12, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 231, 13, 0, 0, 0, 180, 28, 0],
            &[0, 255, 30, 0, 0, 0, 204, 28, 0],
            &[0, 255, 90, 0, 0, 51, 240, 28, 0],
            &[0, 226, 210, 158, 157, 134, 151, 28, 0],
            &[0, 196, 11, 22, 9, 0, 0, 0, 0],
            &[0, 196, 11, 0, 0, 0, 0, 0, 0],
            &[0, 70, 4, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¶' / 0xb6
        '¶' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 118, 160, 160, 146, 153, 25, 0],
            &[0, 167, 255, 255, 201, 0, 107, 39, 0],
            &[6, 249, 255, 255, 201, 0, 107, 39, 0],
            &[21, 255, 255, 255, 201, 0, 107, 39, 0],
            &[7, 251, 255, 255, 201, 0, 107, 39, 0],
            &[0, 181, 255, 255, 201, 0, 107, 39, 0],
            &[0, 22, 150, 179, 181, 0, 107, 39, 0],
            &[0, 0, 0, 0, 146, 0, 107, 39, 0],
            &[0, 0, 0, 0, 146, 0, 107, 39, 0],
            &[0, 0, 0, 0, 146, 0, 107, 39, 0],
            &[0, 0, 0, 0, 146, 0, 107, 39, 0],
            &[0, 0, 0, 0, 146, 0, 107, 39, 0],
            &[0, 0, 0, 0, 114, 0, 83, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '·' / 0xb7
        '·' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 28, 0, 0, 0, 0],
            &[0, 0, 0, 93, 196, 0, 0, 0, 0],
            &[0, 0, 0, 5, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¸' / 0xb8
        '¸' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 0, 0, 0, 0],
            &[0, 0, 0, 27, 137, 0, 0, 0, 0],
            &[0, 0, 0, 22, 129, 108, 0, 0, 0],
            &[0, 0, 0, 82, 60, 0, 0, 0, 0],
        ]),
        // letter: '¹' / 0xb9
        '¹' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 124, 143, 0, 0, 0, 0],
            &[0, 0, 105, 57, 162, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 0, 0, 0, 0],
            &[0, 0, 34, 93, 255, 89, 13, 0, 0],
            &[0, 0, 18, 45, 74, 45, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'º' / 0xba
        'º' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 19, 0, 0, 0, 0],
            &[0, 0, 48, 158, 129, 126, 0, 0, 0],
            &[0, 0, 153, 10, 0, 153, 8, 0, 0],
            &[0, 0, 159, 0, 0, 136, 25, 0, 0],
            &[0, 0, 110, 74, 17, 170, 1, 0, 0],
            &[0, 0, 1, 96, 124, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '»' / 0xbb
        '»' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 19, 40, 0, 55, 1, 0, 0, 0],
            &[0, 11, 182, 22, 82, 132, 0, 0, 0],
            &[0, 0, 39, 182, 6, 139, 88, 0, 0],
            &[0, 0, 0, 163, 69, 28, 204, 0, 0],
            &[0, 0, 82, 142, 4, 180, 42, 0, 0],
            &[0, 30, 172, 3, 131, 79, 0, 0, 0],
            &[0, 1, 10, 0, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¼' / 0xbc
        '¼' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[80, 201, 79, 0, 0, 1, 179, 8, 0],
            &[43, 86, 79, 0, 0, 85, 104, 0, 0],
            &[0, 85, 79, 0, 3, 181, 5, 0, 0],
            &[0, 85, 79, 0, 95, 94, 0, 0, 0],
            &[64, 216, 210, 66, 181, 3, 0, 0, 0],
            &[0, 0, 0, 104, 85, 1, 173, 47, 0],
            &[0, 0, 8, 180, 1, 109, 160, 53, 0],
            &[0, 0, 113, 76, 46, 121, 107, 53, 0],
            &[0, 12, 177, 0, 135, 147, 190, 166, 0],
            &[0, 123, 66, 0, 0, 0, 107, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '½' / 0xbd
        '½' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[80, 201, 79, 0, 0, 1, 179, 8, 0],
            &[43, 86, 79, 0, 0, 85, 104, 0, 0],
            &[0, 85, 79, 0, 3, 181, 5, 0, 0],
            &[0, 85, 79, 0, 95, 94, 0, 0, 0],
            &[64, 216, 210, 66, 181, 3, 0, 0, 0],
            &[0, 0, 0, 104, 89, 151, 157, 126, 0],
            &[0, 0, 8, 180, 1, 1, 0, 176, 0],
            &[0, 0, 113, 76, 0, 0, 79, 113, 0],
            &[0, 12, 177, 0, 0, 74, 129, 0, 0],
            &[0, 123, 66, 0, 39, 231, 157, 157, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¾' / 0xbe
        '¾' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 11, 0, 0, 0, 0, 0, 0],
            &[78, 136, 179, 61, 0, 1, 179, 8, 0],
            &[0, 1, 108, 89, 0, 85, 104, 0, 0],
            &[0, 146, 194, 41, 3, 181, 5, 0, 0],
            &[0, 0, 56, 134, 95, 94, 0, 0, 0],
            &[102, 151, 173, 48, 181, 3, 0, 0, 0],
            &[0, 0, 0, 104, 85, 1, 173, 47, 0],
            &[0, 0, 8, 180, 1, 109, 160, 53, 0],
            &[0, 0, 113, 76, 46, 121, 107, 53, 0],
            &[0, 12, 177, 0, 135, 147, 190, 166, 0],
            &[0, 123, 66, 0, 0, 0, 107, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¿' / 0xbf
        '¿' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 107, 0, 0, 0, 0],
            &[0, 0, 0, 35, 190, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 144, 0, 0, 0, 0],
            &[0, 0, 0, 47, 157, 0, 0, 0, 0],
            &[0, 0, 49, 190, 29, 0, 0, 0, 0],
            &[0, 45, 185, 17, 0, 0, 0, 0, 0],
            &[0, 163, 42, 0, 0, 0, 0, 0, 0],
            &[0, 190, 14, 0, 0, 0, 0, 0, 0],
            &[0, 118, 144, 15, 12, 68, 42, 0, 0],
            &[0, 1, 100, 173, 172, 130, 32, 0, 0],
        ]),
        // letter: 'À' / 0xc0
        'À' => Some(&[
            &[0, 0, 25, 195, 20, 0, 0, 0, 0],
            &[0, 0, 0, 33, 152, 2, 0, 0, 0],
            &[0, 0, 0, 2, 6, 0, 0, 0, 0],
            &[0, 0, 0, 104, 205, 0, 0, 0, 0],
            &[0, 0, 0, 185, 167, 40, 0, 0, 0],
            &[0, 0, 28, 174, 70, 129, 0, 0, 0],
            &[0, 0, 116, 88, 4, 197, 0, 0, 0],
            &[0, 0, 194, 11, 0, 151, 51, 0, 0],
            &[0, 38, 200, 44, 44, 108, 145, 0, 0],
            &[0, 127, 205, 145, 145, 148, 235, 2, 0],
            &[0, 202, 8, 0, 0, 0, 146, 63, 0],
            &[50, 162, 0, 0, 0, 0, 59, 152, 0],
            &[139, 74, 0, 0, 0, 0, 1, 207, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 36, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Á' / 0xc1
        'Á' => Some(&[
            &[0, 0, 0, 0, 77, 161, 0, 0, 0],
            &[0, 0, 0, 36, 149, 4, 0, 0, 0],
            &[0, 0, 0, 2, 6, 0, 0, 0, 0],
            &[0, 0, 0, 104, 205, 0, 0, 0, 0],
            &[0, 0, 0, 185, 167, 40, 0, 0, 0],
            &[0, 0, 28, 174, 70, 129, 0, 0, 0],
            &[0, 0, 116, 88, 4, 197, 0, 0, 0],
            &[0, 0, 194, 11, 0, 151, 51, 0, 0],
            &[0, 38, 200, 44, 44, 108, 145, 0, 0],
            &[0, 127, 205, 145, 145, 148, 235, 2, 0],
            &[0, 202, 8, 0, 0, 0, 146, 63, 0],
            &[50, 162, 0, 0, 0, 0, 59, 152, 0],
            &[139, 74, 0, 0, 0, 0, 1, 207, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 10, 0, 0],
        ]),
        // letter: 'Â' / 0xc2
        'Â' => Some(&[
            &[0, 0, 2, 168, 183, 45, 0, 0, 0],
            &[0, 0, 121, 61, 10, 155, 19, 0, 0],
            &[0, 0, 0, 2, 6, 0, 0, 0, 0],
            &[0, 0, 0, 104, 205, 0, 0, 0, 0],
            &[0, 0, 0, 185, 167, 40, 0, 0, 0],
            &[0, 0, 28, 174, 70, 129, 0, 0, 0],
            &[0, 0, 116, 88, 4, 197, 0, 0, 0],
            &[0, 0, 194, 11, 0, 151, 51, 0, 0],
            &[0, 38, 200, 44, 44, 108, 145, 0, 0],
            &[0, 127, 205, 145, 145, 148, 235, 2, 0],
            &[0, 202, 8, 0, 0, 0, 146, 63, 0],
            &[50, 162, 0, 0, 0, 0, 59, 152, 0],
            &[139, 74, 0, 0, 0, 0, 1, 207, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 45, 0, 0, 0, 0],
        ]),
        // letter: 'Ã' / 0xc3
        'Ã' => Some(&[
            &[0, 0, 108, 154, 34, 13, 94, 0, 0],
            &[0, 21, 124, 23, 153, 175, 38, 0, 0],
            &[0, 0, 0, 2, 6, 0, 0, 0, 0],
            &[0, 0, 0, 104, 205, 0, 0, 0, 0],
            &[0, 0, 0, 185, 167, 40, 0, 0, 0],
            &[0, 0, 28, 174, 70, 129, 0, 0, 0],
            &[0, 0, 116, 88, 4, 197, 0, 0, 0],
            &[0, 0, 194, 11, 0, 151, 51, 0, 0],
            &[0, 38, 200, 44, 44, 108, 145, 0, 0],
            &[0, 127, 205, 145, 145, 148, 235, 2, 0],
            &[0, 202, 8, 0, 0, 0, 146, 63, 0],
            &[50, 162, 0, 0, 0, 0, 59, 152, 0],
            &[139, 74, 0, 0, 0, 0, 1, 207, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ä' / 0xc4
        'Ä' => Some(&[
            &[0, 0, 125, 32, 0, 129, 27, 0, 0],
            &[0, 0, 98, 22, 0, 102, 19, 0, 0],
            &[0, 0, 0, 2, 6, 0, 0, 0, 0],
            &[0, 0, 0, 104, 205, 0, 0, 0, 0],
            &[0, 0, 0, 185, 167, 40, 0, 0, 0],
            &[0, 0, 28, 174, 70, 129, 0, 0, 0],
            &[0, 0, 116, 88, 4, 197, 0, 0, 0],
            &[0, 0, 194, 11, 0, 151, 51, 0, 0],
            &[0, 38, 200, 44, 44, 108, 145, 0, 0],
            &[0, 127, 205, 145, 145, 148, 235, 2, 0],
            &[0, 202, 8, 0, 0, 0, 146, 63, 0],
            &[50, 162, 0, 0, 0, 0, 59, 152, 0],
            &[139, 74, 0, 0, 0, 0, 1, 207, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Å' / 0xc5
        'Å' => Some(&[
            &[0, 0, 58, 98, 4, 151, 0, 0, 0],
            &[0, 0, 9, 154, 151, 63, 0, 0, 0],
            &[0, 0, 0, 2, 6, 0, 0, 0, 0],
            &[0, 0, 0, 104, 205, 0, 0, 0, 0],
            &[0, 0, 0, 185, 167, 40, 0, 0, 0],
            &[0, 0, 28, 174, 70, 129, 0, 0, 0],
            &[0, 0, 116, 88, 4, 197, 0, 0, 0],
            &[0, 0, 194, 11, 0, 151, 51, 0, 0],
            &[0, 38, 200, 44, 44, 108, 145, 0, 0],
            &[0, 127, 205, 145, 145, 148, 235, 2, 0],
            &[0, 202, 8, 0, 0, 0, 146, 63, 0],
            &[50, 162, 0, 0, 0, 0, 59, 152, 0],
            &[139, 74, 0, 0, 0, 0, 1, 207, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 155, 148, 65, 0, 0, 0],
        ]),
        // letter: 'Æ' / 0xc6
        'Æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 219, 240, 186, 186, 166, 0],
            &[0, 0, 65, 129, 200, 0, 0, 0, 0],
            &[0, 0, 139, 56, 200, 0, 0, 0, 0],
            &[0, 0, 195, 2, 200, 0, 0, 0, 0],
            &[0, 32, 167, 0, 209, 157, 157, 103, 0],
            &[0, 107, 138, 44, 221, 29, 29, 19, 0],
            &[0, 193, 162, 138, 255, 0, 0, 0, 0],
            &[8, 195, 0, 0, 200, 0, 0, 0, 0],
            &[72, 132, 0, 0, 200, 0, 0, 0, 0],
            &[147, 59, 0, 0, 211, 189, 189, 170, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ç' / 0xc7
        'Ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 26, 4, 0, 0],
            &[0, 0, 47, 186, 176, 160, 192, 107, 0],
            &[0, 30, 206, 32, 0, 0, 0, 3, 0],
            &[0, 153, 76, 0, 0, 0, 0, 0, 0],
            &[0, 214, 2, 0, 0, 0, 0, 0, 0],
            &[4, 210, 0, 0, 0, 0, 0, 0, 0],
            &[4, 209, 0, 0, 0, 0, 0, 0, 0],
            &[0, 215, 2, 0, 0, 0, 0, 0, 0],
            &[0, 163, 64, 0, 0, 0, 0, 0, 0],
            &[0, 42, 204, 24, 0, 0, 0, 1, 0],
            &[0, 0, 64, 200, 183, 170, 195, 76, 0],
            &[0, 0, 0, 0, 166, 40, 3, 0, 0],
            &[0, 0, 0, 0, 71, 186, 2, 0, 0],
            &[0, 0, 0, 42, 82, 19, 0, 0, 0],
        ]),
        // letter: 'È' / 0xc8
        'È' => Some(&[
            &[0, 0, 13, 193, 34, 0, 0, 0, 0],
            &[0, 0, 0, 20, 158, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 205, 186, 186, 186, 186, 33, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 184, 157, 157, 157, 141, 0, 0],
            &[0, 135, 93, 29, 29, 29, 26, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 208, 189, 189, 189, 189, 34, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 44, 0, 0, 0, 0, 0],
        ]),
        // letter: 'É' / 0xc9
        'É' => Some(&[
            &[0, 0, 0, 0, 53, 181, 4, 0, 0],
            &[0, 0, 0, 21, 157, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 205, 186, 186, 186, 186, 33, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 184, 157, 157, 157, 141, 0, 0],
            &[0, 135, 93, 29, 29, 29, 26, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 208, 189, 189, 189, 189, 34, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 18, 0, 0],
        ]),
        // letter: 'Ê' / 0xca
        'Ê' => Some(&[
            &[0, 0, 0, 150, 181, 67, 0, 0, 0],
            &[0, 0, 100, 82, 3, 148, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 205, 186, 186, 186, 186, 33, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 184, 157, 157, 157, 141, 0, 0],
            &[0, 135, 93, 29, 29, 29, 26, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 208, 189, 189, 189, 189, 34, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 53, 0, 0, 0, 0],
        ]),
        // letter: 'Ë' / 0xcb
        'Ë' => Some(&[
            &[0, 0, 108, 50, 0, 112, 44, 0, 0],
            &[0, 0, 84, 37, 0, 88, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 205, 186, 186, 186, 186, 33, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 184, 157, 157, 157, 141, 0, 0],
            &[0, 135, 93, 29, 29, 29, 26, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 208, 189, 189, 189, 189, 34, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ì' / 0xcc
        'Ì' => Some(&[
            &[0, 0, 25, 195, 20, 0, 0, 0, 0],
            &[0, 0, 0, 33, 152, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 156, 187, 222, 162, 76, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 16, 156, 187, 223, 162, 76, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 36, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Í' / 0xcd
        'Í' => Some(&[
            &[0, 0, 0, 0, 77, 161, 0, 0, 0],
            &[0, 0, 0, 36, 149, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 156, 187, 222, 162, 76, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 16, 156, 187, 223, 162, 76, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 10, 0, 0],
        ]),
        // letter: 'Î' / 0xce
        'Î' => Some(&[
            &[0, 0, 2, 168, 183, 45, 0, 0, 0],
            &[0, 0, 121, 61, 10, 155, 19, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 156, 187, 222, 162, 76, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 16, 156, 187, 223, 162, 76, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 45, 0, 0, 0, 0],
        ]),
        // letter: 'Ï' / 0xcf
        'Ï' => Some(&[
            &[0, 0, 125, 32, 0, 129, 27, 0, 0],
            &[0, 0, 98, 22, 0, 102, 19, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 156, 187, 222, 162, 76, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 16, 156, 187, 223, 162, 76, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ð' / 0xd0
        'Ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 177, 190, 184, 193, 147, 16, 0, 0],
            &[0, 178, 29, 0, 0, 79, 200, 7, 0],
            &[0, 178, 29, 0, 0, 0, 121, 107, 0],
            &[0, 178, 29, 0, 0, 0, 36, 179, 0],
            &[99, 255, 128, 97, 0, 0, 5, 205, 0],
            &[82, 255, 112, 80, 0, 0, 5, 201, 0],
            &[0, 178, 29, 0, 0, 0, 41, 170, 0],
            &[0, 178, 29, 0, 0, 0, 132, 93, 0],
            &[0, 178, 29, 0, 1, 93, 188, 2, 0],
            &[0, 178, 194, 186, 194, 132, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ñ' / 0xd1
        'Ñ' => Some(&[
            &[0, 0, 96, 158, 42, 6, 102, 0, 0],
            &[0, 9, 136, 18, 147, 176, 49, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 213, 109, 0, 0, 0, 139, 64, 0],
            &[0, 185, 188, 6, 0, 0, 139, 64, 0],
            &[0, 195, 94, 104, 0, 0, 139, 64, 0],
            &[0, 200, 4, 189, 5, 0, 139, 64, 0],
            &[0, 200, 0, 99, 99, 0, 139, 64, 0],
            &[0, 200, 0, 5, 190, 4, 139, 64, 0],
            &[0, 200, 0, 0, 105, 94, 139, 64, 0],
            &[0, 200, 0, 0, 7, 190, 137, 64, 0],
            &[0, 200, 0, 0, 0, 110, 213, 64, 0],
            &[0, 200, 0, 0, 0, 9, 232, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ò' / 0xd2
        'Ò' => Some(&[
            &[0, 0, 25, 195, 20, 0, 0, 0, 0],
            &[0, 0, 0, 33, 152, 2, 0, 0, 0],
            &[0, 0, 0, 11, 29, 2, 0, 0, 0],
            &[0, 5, 158, 183, 158, 198, 57, 0, 0],
            &[0, 135, 105, 0, 0, 19, 204, 12, 0],
            &[3, 210, 2, 0, 0, 0, 119, 96, 0],
            &[45, 165, 0, 0, 0, 0, 60, 150, 0],
            &[68, 141, 0, 0, 0, 0, 37, 173, 0],
            &[70, 142, 0, 0, 0, 0, 39, 172, 0],
            &[45, 167, 0, 0, 0, 0, 63, 148, 0],
            &[4, 211, 2, 0, 0, 0, 123, 92, 0],
            &[0, 137, 109, 0, 0, 24, 205, 10, 0],
            &[0, 7, 161, 188, 167, 199, 53, 0, 0],
            &[0, 0, 0, 13, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 36, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ó' / 0xd3
        'Ó' => Some(&[
            &[0, 0, 0, 0, 77, 161, 0, 0, 0],
            &[0, 0, 0, 36, 149, 4, 0, 0, 0],
            &[0, 0, 0, 11, 29, 2, 0, 0, 0],
            &[0, 5, 158, 183, 158, 198, 57, 0, 0],
            &[0, 135, 105, 0, 0, 19, 204, 12, 0],
            &[3, 210, 2, 0, 0, 0, 119, 96, 0],
            &[45, 165, 0, 0, 0, 0, 60, 150, 0],
            &[68, 141, 0, 0, 0, 0, 37, 173, 0],
            &[70, 142, 0, 0, 0, 0, 39, 172, 0],
            &[45, 167, 0, 0, 0, 0, 63, 148, 0],
            &[4, 211, 2, 0, 0, 0, 123, 92, 0],
            &[0, 137, 109, 0, 0, 24, 205, 10, 0],
            &[0, 7, 161, 188, 167, 199, 53, 0, 0],
            &[0, 0, 0, 13, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 10, 0, 0],
        ]),
        // letter: 'Ô' / 0xd4
        'Ô' => Some(&[
            &[0, 0, 2, 168, 183, 45, 0, 0, 0],
            &[0, 0, 121, 61, 10, 155, 19, 0, 0],
            &[0, 0, 0, 11, 29, 2, 0, 0, 0],
            &[0, 5, 158, 183, 158, 198, 57, 0, 0],
            &[0, 135, 105, 0, 0, 19, 204, 12, 0],
            &[3, 210, 2, 0, 0, 0, 119, 96, 0],
            &[45, 165, 0, 0, 0, 0, 60, 150, 0],
            &[68, 141, 0, 0, 0, 0, 37, 173, 0],
            &[70, 142, 0, 0, 0, 0, 39, 172, 0],
            &[45, 167, 0, 0, 0, 0, 63, 148, 0],
            &[4, 211, 2, 0, 0, 0, 123, 92, 0],
            &[0, 137, 109, 0, 0, 24, 205, 10, 0],
            &[0, 7, 161, 188, 167, 199, 53, 0, 0],
            &[0, 0, 0, 13, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 45, 0, 0, 0, 0],
        ]),
        // letter: 'Õ' / 0xd5
        'Õ' => Some(&[
            &[0, 0, 108, 154, 34, 13, 94, 0, 0],
            &[0, 21, 124, 23, 153, 175, 38, 0, 0],
            &[0, 0, 0, 11, 29, 2, 0, 0, 0],
            &[0, 5, 158, 183, 158, 198, 57, 0, 0],
            &[0, 135, 105, 0, 0, 19, 204, 12, 0],
            &[3, 210, 2, 0, 0, 0, 119, 96, 0],
            &[45, 165, 0, 0, 0, 0, 60, 150, 0],
            &[68, 141, 0, 0, 0, 0, 37, 173, 0],
            &[70, 142, 0, 0, 0, 0, 39, 172, 0],
            &[45, 167, 0, 0, 0, 0, 63, 148, 0],
            &[4, 211, 2, 0, 0, 0, 123, 92, 0],
            &[0, 137, 109, 0, 0, 24, 205, 10, 0],
            &[0, 7, 161, 188, 167, 199, 53, 0, 0],
            &[0, 0, 0, 13, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ö' / 0xd6
        'Ö' => Some(&[
            &[0, 0, 125, 32, 0, 129, 27, 0, 0],
            &[0, 0, 98, 22, 0, 102, 19, 0, 0],
            &[0, 0, 0, 11, 29, 2, 0, 0, 0],
            &[0, 5, 158, 183, 158, 198, 57, 0, 0],
            &[0, 135, 105, 0, 0, 19, 204, 12, 0],
            &[3, 210, 2, 0, 0, 0, 119, 96, 0],
            &[45, 165, 0, 0, 0, 0, 60, 150, 0],
            &[68, 141, 0, 0, 0, 0, 37, 173, 0],
            &[70, 142, 0, 0, 0, 0, 39, 172, 0],
            &[45, 167, 0, 0, 0, 0, 63, 148, 0],
            &[4, 211, 2, 0, 0, 0, 123, 92, 0],
            &[0, 137, 109, 0, 0, 24, 205, 10, 0],
            &[0, 7, 161, 188, 167, 199, 53, 0, 0],
            &[0, 0, 0, 13, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '×' / 0xd7
        '×' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 48, 0, 0, 5, 154, 11, 0],
            &[0, 13, 177, 47, 5, 165, 67, 0, 0],
            &[0, 0, 13, 176, 175, 68, 0, 0, 0],
            &[0, 0, 5, 166, 191, 48, 0, 0, 0],
            &[0, 5, 165, 68, 13, 178, 48, 0, 0],
            &[0, 127, 68, 0, 0, 12, 171, 11, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ø' / 0xd8
        'Ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 29, 2, 21, 25, 0],
            &[0, 5, 158, 183, 158, 198, 211, 45, 0],
            &[0, 135, 105, 0, 0, 69, 255, 12, 0],
            &[3, 210, 2, 0, 0, 178, 140, 96, 0],
            &[45, 165, 0, 0, 84, 116, 60, 150, 0],
            &[68, 141, 0, 7, 186, 6, 37, 173, 0],
            &[70, 142, 0, 120, 80, 0, 39, 172, 0],
            &[45, 167, 24, 177, 0, 0, 63, 148, 0],
            &[4, 211, 157, 47, 0, 0, 123, 92, 0],
            &[0, 187, 255, 0, 0, 24, 205, 10, 0],
            &[0, 187, 184, 188, 167, 199, 53, 0, 0],
            &[3, 56, 0, 13, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ù' / 0xd9
        'Ù' => Some(&[
            &[0, 0, 25, 195, 20, 0, 0, 0, 0],
            &[0, 0, 0, 33, 152, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 0, 142, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 146, 61, 0],
            &[0, 189, 21, 0, 0, 0, 172, 38, 0],
            &[0, 119, 117, 0, 0, 28, 205, 0, 0],
            &[0, 9, 170, 183, 165, 201, 52, 0, 0],
            &[0, 0, 0, 15, 25, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 36, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ú' / 0xda
        'Ú' => Some(&[
            &[0, 0, 0, 0, 77, 161, 0, 0, 0],
            &[0, 0, 0, 36, 149, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 0, 142, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 146, 61, 0],
            &[0, 189, 21, 0, 0, 0, 172, 38, 0],
            &[0, 119, 117, 0, 0, 28, 205, 0, 0],
            &[0, 9, 170, 183, 165, 201, 52, 0, 0],
            &[0, 0, 0, 15, 25, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 10, 0, 0],
        ]),
        // letter: 'Û' / 0xdb
        'Û' => Some(&[
            &[0, 0, 2, 168, 183, 45, 0, 0, 0],
            &[0, 0, 121, 61, 10, 155, 19, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 0, 142, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 146, 61, 0],
            &[0, 189, 21, 0, 0, 0, 172, 38, 0],
            &[0, 119, 117, 0, 0, 28, 205, 0, 0],
            &[0, 9, 170, 183, 165, 201, 52, 0, 0],
            &[0, 0, 0, 15, 25, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 45, 0, 0, 0, 0],
        ]),
        // letter: 'Ü' / 0xdc
        'Ü' => Some(&[
            &[0, 0, 125, 32, 0, 129, 27, 0, 0],
            &[0, 0, 98, 22, 0, 102, 19, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 0, 142, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 146, 61, 0],
            &[0, 189, 21, 0, 0, 0, 172, 38, 0],
            &[0, 119, 117, 0, 0, 28, 205, 0, 0],
            &[0, 9, 170, 183, 165, 201, 52, 0, 0],
            &[0, 0, 0, 15, 25, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ý' / 0xdd
        'Ý' => Some(&[
            &[0, 0, 0, 0, 77, 161, 0, 0, 0],
            &[0, 0, 0, 36, 149, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[38, 186, 0, 0, 0, 0, 80, 140, 0],
            &[0, 157, 62, 0, 0, 0, 193, 20, 0],
            &[0, 30, 183, 0, 0, 78, 130, 0, 0],
            &[0, 0, 144, 62, 0, 185, 15, 0, 0],
            &[0, 0, 22, 179, 75, 119, 0, 0, 0],
            &[0, 0, 0, 132, 204, 10, 0, 0, 0],
            &[0, 0, 0, 47, 156, 0, 0, 0, 0],
            &[0, 0, 0, 45, 155, 0, 0, 0, 0],
            &[0, 0, 0, 45, 155, 0, 0, 0, 0],
            &[0, 0, 0, 45, 155, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 10, 0, 0],
        ]),
        // letter: 'Þ' / 0xde
        'Þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 71, 0, 0, 0, 0, 0, 0],
            &[0, 135, 89, 23, 14, 0, 0, 0, 0],
            &[0, 135, 186, 159, 170, 199, 125, 0, 0],
            &[0, 135, 72, 0, 0, 2, 164, 82, 0],
            &[0, 135, 72, 0, 0, 0, 74, 140, 0],
            &[0, 135, 72, 0, 0, 0, 88, 127, 0],
            &[0, 135, 72, 0, 0, 44, 200, 34, 0],
            &[0, 135, 205, 186, 187, 158, 42, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ß' / 0xdf
        'ß' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 78, 162, 174, 123, 11, 0, 0],
            &[0, 84, 183, 32, 8, 110, 156, 0, 0],
            &[0, 168, 41, 0, 0, 29, 182, 0, 0],
            &[0, 182, 22, 0, 15, 175, 64, 0, 0],
            &[0, 182, 22, 0, 177, 55, 0, 0, 0],
            &[0, 182, 22, 0, 205, 11, 0, 0, 0],
            &[0, 182, 22, 0, 73, 193, 38, 0, 0],
            &[0, 182, 22, 0, 0, 37, 200, 44, 0],
            &[0, 182, 22, 0, 0, 0, 57, 150, 0],
            &[0, 182, 22, 0, 0, 0, 61, 154, 0],
            &[0, 182, 22, 137, 166, 151, 195, 38, 0],
            &[0, 0, 0, 0, 15, 27, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'à' / 0xe0
        'à' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 46, 137, 0, 0, 0, 0, 0],
            &[0, 0, 0, 123, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 7, 0, 0, 0],
            &[0, 1, 67, 126, 139, 91, 3, 0, 0],
            &[0, 22, 122, 48, 36, 122, 150, 0, 0],
            &[0, 0, 0, 0, 0, 0, 197, 0, 0],
            &[0, 0, 3, 50, 80, 91, 217, 0, 0],
            &[0, 57, 190, 124, 84, 66, 208, 0, 0],
            &[0, 192, 19, 0, 0, 0, 212, 0, 0],
            &[0, 197, 13, 0, 0, 70, 245, 0, 0],
            &[0, 76, 196, 147, 163, 113, 178, 0, 0],
            &[0, 0, 4, 26, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'á' / 0xe1
        'á' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 158, 17, 0, 0],
            &[0, 0, 0, 0, 143, 79, 0, 0, 0],
            &[0, 0, 0, 26, 67, 0, 0, 0, 0],
            &[0, 1, 67, 126, 139, 91, 3, 0, 0],
            &[0, 22, 122, 48, 36, 122, 150, 0, 0],
            &[0, 0, 0, 0, 0, 0, 197, 0, 0],
            &[0, 0, 3, 50, 80, 91, 217, 0, 0],
            &[0, 57, 190, 124, 84, 66, 208, 0, 0],
            &[0, 192, 19, 0, 0, 0, 212, 0, 0],
            &[0, 197, 13, 0, 0, 70, 245, 0, 0],
            &[0, 76, 196, 147, 163, 113, 178, 0, 0],
            &[0, 0, 4, 26, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'â' / 0xe2
        'â' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 64, 161, 4, 0, 0, 0],
            &[0, 0, 28, 168, 77, 140, 0, 0, 0],
            &[0, 0, 80, 12, 0, 65, 27, 0, 0],
            &[0, 1, 67, 126, 139, 91, 3, 0, 0],
            &[0, 22, 122, 48, 36, 122, 150, 0, 0],
            &[0, 0, 0, 0, 0, 0, 197, 0, 0],
            &[0, 0, 3, 50, 80, 91, 217, 0, 0],
            &[0, 57, 190, 124, 84, 66, 208, 0, 0],
            &[0, 192, 19, 0, 0, 0, 212, 0, 0],
            &[0, 197, 13, 0, 0, 70, 245, 0, 0],
            &[0, 76, 196, 147, 163, 113, 178, 0, 0],
            &[0, 0, 4, 26, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ã' / 0xe3
        'ã' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 20, 0, 0, 22, 0, 0],
            &[0, 0, 172, 156, 147, 90, 123, 0, 0],
            &[0, 8, 55, 0, 42, 92, 6, 0, 0],
            &[0, 1, 67, 126, 139, 91, 3, 0, 0],
            &[0, 22, 122, 48, 36, 122, 150, 0, 0],
            &[0, 0, 0, 0, 0, 0, 197, 0, 0],
            &[0, 0, 3, 50, 80, 91, 217, 0, 0],
            &[0, 57, 190, 124, 84, 66, 208, 0, 0],
            &[0, 192, 19, 0, 0, 0, 212, 0, 0],
            &[0, 197, 13, 0, 0, 70, 245, 0, 0],
            &[0, 76, 196, 147, 163, 113, 178, 0, 0],
            &[0, 0, 4, 26, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ä' / 0xe4
        'ä' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 2, 0, 15, 1, 0, 0],
            &[0, 0, 178, 72, 0, 185, 63, 0, 0],
            &[0, 0, 11, 0, 0, 11, 0, 0, 0],
            &[0, 1, 67, 126, 139, 91, 3, 0, 0],
            &[0, 22, 122, 48, 36, 122, 150, 0, 0],
            &[0, 0, 0, 0, 0, 0, 197, 0, 0],
            &[0, 0, 3, 50, 80, 91, 217, 0, 0],
            &[0, 57, 190, 124, 84, 66, 208, 0, 0],
            &[0, 192, 19, 0, 0, 0, 212, 0, 0],
            &[0, 197, 13, 0, 0, 70, 245, 0, 0],
            &[0, 76, 196, 147, 163, 113, 178, 0, 0],
            &[0, 0, 4, 26, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'å' / 0xe5
        'å' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 45, 86, 8, 0, 0, 0],
            &[0, 0, 22, 158, 59, 141, 0, 0, 0],
            &[0, 0, 26, 150, 42, 146, 0, 0, 0],
            &[0, 0, 0, 62, 106, 15, 0, 0, 0],
            &[0, 1, 67, 126, 139, 91, 3, 0, 0],
            &[0, 22, 122, 48, 36, 122, 150, 0, 0],
            &[0, 0, 0, 0, 0, 0, 197, 0, 0],
            &[0, 0, 3, 50, 80, 91, 217, 0, 0],
            &[0, 57, 190, 124, 84, 66, 208, 0, 0],
            &[0, 192, 19, 0, 0, 0, 212, 0, 0],
            &[0, 197, 13, 0, 0, 70, 245, 0, 0],
            &[0, 76, 196, 147, 163, 113, 178, 0, 0],
            &[0, 0, 4, 26, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'æ' / 0xe6
        'æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[3, 89, 140, 74, 5, 113, 126, 13, 0],
            &[12, 96, 37, 167, 158, 82, 91, 155, 0],
            &[0, 0, 0, 68, 184, 0, 0, 195, 0],
            &[0, 7, 59, 117, 179, 79, 79, 213, 2],
            &[36, 198, 113, 121, 176, 88, 88, 88, 1],
            &[139, 66, 0, 75, 155, 0, 0, 0, 0],
            &[143, 53, 0, 138, 181, 8, 0, 1, 0],
            &[49, 197, 167, 126, 53, 186, 151, 146, 0],
            &[0, 9, 18, 0, 0, 6, 25, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ç' / 0xe7
        'ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 80, 136, 137, 86, 0, 0],
            &[0, 5, 183, 132, 51, 47, 93, 0, 0],
            &[0, 105, 127, 0, 0, 0, 0, 0, 0],
            &[0, 174, 38, 0, 0, 0, 0, 0, 0],
            &[0, 190, 20, 0, 0, 0, 0, 0, 0],
            &[0, 161, 56, 0, 0, 0, 0, 0, 0],
            &[0, 68, 180, 5, 0, 0, 0, 0, 0],
            &[0, 0, 105, 199, 164, 163, 180, 0, 0],
            &[0, 0, 0, 4, 189, 19, 0, 0, 0],
            &[0, 0, 0, 7, 92, 160, 0, 0, 0],
            &[0, 0, 0, 61, 75, 7, 0, 0, 0],
        ]),
        // letter: 'è' / 0xe8
        'è' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 135, 0, 0, 0, 0, 0],
            &[0, 0, 0, 127, 95, 0, 0, 0, 0],
            &[0, 0, 0, 0, 88, 6, 0, 0, 0],
            &[0, 0, 23, 115, 142, 99, 7, 0, 0],
            &[0, 39, 199, 77, 34, 99, 176, 0, 0],
            &[0, 175, 47, 0, 0, 0, 150, 61, 0],
            &[0, 224, 76, 76, 76, 76, 147, 108, 0],
            &[5, 225, 95, 95, 95, 95, 95, 44, 0],
            &[0, 211, 3, 0, 0, 0, 0, 0, 0],
            &[0, 131, 112, 0, 0, 0, 1, 4, 0],
            &[0, 5, 141, 184, 150, 165, 184, 30, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'é' / 0xe9
        'é' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 158, 16, 0, 0],
            &[0, 0, 0, 0, 146, 76, 0, 0, 0],
            &[0, 0, 0, 28, 65, 0, 0, 0, 0],
            &[0, 0, 23, 115, 142, 99, 7, 0, 0],
            &[0, 39, 199, 77, 34, 99, 176, 0, 0],
            &[0, 175, 47, 0, 0, 0, 150, 61, 0],
            &[0, 224, 76, 76, 76, 76, 147, 108, 0],
            &[5, 225, 95, 95, 95, 95, 95, 44, 0],
            &[0, 211, 3, 0, 0, 0, 0, 0, 0],
            &[0, 131, 112, 0, 0, 0, 1, 4, 0],
            &[0, 5, 141, 184, 150, 165, 184, 30, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ê' / 0xea
        'ê' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 159, 3, 0, 0, 0],
            &[0, 0, 30, 167, 78, 137, 0, 0, 0],
            &[0, 0, 81, 11, 0, 67, 26, 0, 0],
            &[0, 0, 23, 115, 142, 99, 7, 0, 0],
            &[0, 39, 199, 77, 34, 99, 176, 0, 0],
            &[0, 175, 47, 0, 0, 0, 150, 61, 0],
            &[0, 224, 76, 76, 76, 76, 147, 108, 0],
            &[5, 225, 95, 95, 95, 95, 95, 44, 0],
            &[0, 211, 3, 0, 0, 0, 0, 0, 0],
            &[0, 131, 112, 0, 0, 0, 1, 4, 0],
            &[0, 5, 141, 184, 150, 165, 184, 30, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ë' / 0xeb
        'ë' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 1, 0, 15, 1, 0, 0],
            &[0, 0, 182, 68, 0, 189, 60, 0, 0],
            &[0, 0, 11, 0, 0, 11, 0, 0, 0],
            &[0, 0, 23, 115, 142, 99, 7, 0, 0],
            &[0, 39, 199, 77, 34, 99, 176, 0, 0],
            &[0, 175, 47, 0, 0, 0, 150, 61, 0],
            &[0, 224, 76, 76, 76, 76, 147, 108, 0],
            &[5, 225, 95, 95, 95, 95, 95, 44, 0],
            &[0, 211, 3, 0, 0, 0, 0, 0, 0],
            &[0, 131, 112, 0, 0, 0, 1, 4, 0],
            &[0, 5, 141, 184, 150, 165, 184, 30, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ì' / 0xec
        'ì' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 140, 81, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 2, 0, 0, 0],
            &[0, 57, 114, 114, 98, 0, 0, 0, 0],
            &[0, 17, 44, 57, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 171, 159, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'í' / 0xed
        'í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 159, 10, 0, 0],
            &[0, 0, 0, 0, 156, 65, 0, 0, 0],
            &[0, 0, 0, 36, 58, 0, 0, 0, 0],
            &[0, 57, 114, 114, 98, 0, 0, 0, 0],
            &[0, 17, 44, 57, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 171, 159, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'î' / 0xee
        'î' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 77, 152, 1, 0, 0, 0],
            &[0, 0, 38, 162, 87, 125, 0, 0, 0],
            &[0, 0, 85, 7, 0, 74, 19, 0, 0],
            &[0, 57, 114, 114, 98, 0, 0, 0, 0],
            &[0, 17, 44, 57, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 171, 159, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ï' / 0xef
        'ï' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 0, 0, 16, 0, 0, 0],
            &[0, 0, 196, 54, 0, 203, 46, 0, 0],
            &[0, 0, 11, 0, 0, 11, 0, 0, 0],
            &[0, 57, 114, 114, 98, 0, 0, 0, 0],
            &[0, 17, 44, 57, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 171, 159, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ð' / 0xf0
        'ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 109, 5, 29, 74, 0, 0],
            &[0, 0, 0, 97, 255, 181, 29, 0, 0],
            &[0, 0, 50, 175, 106, 183, 8, 0, 0],
            &[0, 0, 10, 3, 0, 72, 133, 0, 0],
            &[0, 0, 40, 123, 136, 77, 180, 11, 0],
            &[0, 79, 179, 57, 35, 98, 223, 79, 0],
            &[1, 206, 10, 0, 0, 0, 131, 122, 0],
            &[20, 186, 0, 0, 0, 0, 80, 131, 0],
            &[10, 198, 0, 0, 0, 0, 105, 101, 0],
            &[0, 175, 56, 0, 0, 10, 196, 23, 0],
            &[0, 21, 171, 165, 149, 194, 74, 0, 0],
            &[0, 0, 0, 14, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ñ' / 0xf1
        'ñ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 22, 0, 0, 22, 0, 0],
            &[0, 0, 162, 153, 154, 83, 135, 0, 0],
            &[0, 1, 61, 0, 36, 95, 9, 0, 0],
            &[0, 77, 2, 88, 139, 112, 13, 0, 0],
            &[0, 178, 143, 82, 35, 97, 194, 0, 0],
            &[0, 178, 98, 0, 0, 0, 186, 26, 0],
            &[0, 178, 43, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ò' / 0xf2
        'ò' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 140, 81, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 2, 0, 0, 0],
            &[0, 0, 37, 125, 140, 85, 2, 0, 0],
            &[0, 65, 192, 62, 36, 119, 162, 0, 0],
            &[0, 199, 26, 0, 0, 0, 162, 57, 0],
            &[12, 201, 0, 0, 0, 0, 92, 118, 0],
            &[24, 183, 0, 0, 0, 0, 81, 130, 0],
            &[4, 208, 0, 0, 0, 0, 116, 98, 0],
            &[0, 156, 80, 0, 0, 14, 203, 19, 0],
            &[0, 13, 167, 168, 150, 194, 67, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ó' / 0xf3
        'ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 159, 10, 0, 0],
            &[0, 0, 0, 0, 156, 65, 0, 0, 0],
            &[0, 0, 0, 36, 58, 0, 0, 0, 0],
            &[0, 0, 37, 125, 140, 85, 2, 0, 0],
            &[0, 65, 192, 62, 36, 119, 162, 0, 0],
            &[0, 199, 26, 0, 0, 0, 162, 57, 0],
            &[12, 201, 0, 0, 0, 0, 92, 118, 0],
            &[24, 183, 0, 0, 0, 0, 81, 130, 0],
            &[4, 208, 0, 0, 0, 0, 116, 98, 0],
            &[0, 156, 80, 0, 0, 14, 203, 19, 0],
            &[0, 13, 167, 168, 150, 194, 67, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ô' / 0xf4
        'ô' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 77, 152, 1, 0, 0, 0],
            &[0, 0, 38, 162, 87, 125, 0, 0, 0],
            &[0, 0, 85, 7, 0, 74, 19, 0, 0],
            &[0, 0, 37, 125, 140, 85, 2, 0, 0],
            &[0, 65, 192, 62, 36, 119, 162, 0, 0],
            &[0, 199, 26, 0, 0, 0, 162, 57, 0],
            &[12, 201, 0, 0, 0, 0, 92, 118, 0],
            &[24, 183, 0, 0, 0, 0, 81, 130, 0],
            &[4, 208, 0, 0, 0, 0, 116, 98, 0],
            &[0, 156, 80, 0, 0, 14, 203, 19, 0],
            &[0, 13, 167, 168, 150, 194, 67, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'õ' / 0xf5
        'õ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 18, 0, 0, 22, 0, 0],
            &[0, 5, 178, 159, 138, 100, 107, 0, 0],
            &[0, 16, 47, 0, 49, 88, 3, 0, 0],
            &[0, 0, 37, 125, 140, 85, 2, 0, 0],
            &[0, 65, 192, 62, 36, 119, 162, 0, 0],
            &[0, 199, 26, 0, 0, 0, 162, 57, 0],
            &[12, 201, 0, 0, 0, 0, 92, 118, 0],
            &[24, 183, 0, 0, 0, 0, 81, 130, 0],
            &[4, 208, 0, 0, 0, 0, 116, 98, 0],
            &[0, 156, 80, 0, 0, 14, 203, 19, 0],
            &[0, 13, 167, 168, 150, 194, 67, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ö' / 0xf6
        'ö' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 0, 0, 16, 0, 0, 0],
            &[0, 0, 196, 54, 0, 203, 46, 0, 0],
            &[0, 0, 11, 0, 0, 11, 0, 0, 0],
            &[0, 0, 37, 125, 140, 85, 2, 0, 0],
            &[0, 65, 192, 62, 36, 119, 162, 0, 0],
            &[0, 199, 26, 0, 0, 0, 162, 57, 0],
            &[12, 201, 0, 0, 0, 0, 92, 118, 0],
            &[24, 183, 0, 0, 0, 0, 81, 130, 0],
            &[4, 208, 0, 0, 0, 0, 116, 98, 0],
            &[0, 156, 80, 0, 0, 14, 203, 19, 0],
            &[0, 13, 167, 168, 150, 194, 67, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '÷' / 0xf7
        '÷' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 70, 167, 0, 0, 0, 0],
            &[0, 0, 0, 8, 31, 0, 0, 0, 0],
            &[0, 67, 71, 71, 71, 71, 71, 24, 0],
            &[0, 95, 100, 100, 100, 100, 100, 35, 0],
            &[0, 0, 0, 1, 11, 0, 0, 0, 0],
            &[0, 0, 0, 75, 175, 0, 0, 0, 0],
            &[0, 0, 0, 1, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ø' / 0xf8
        'ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 37, 125, 140, 85, 76, 67, 0],
            &[0, 65, 192, 62, 36, 150, 255, 4, 0],
            &[0, 199, 26, 0, 5, 179, 190, 57, 0],
            &[12, 201, 0, 0, 142, 71, 92, 118, 0],
            &[24, 183, 0, 83, 131, 0, 81, 130, 0],
            &[4, 208, 36, 175, 3, 0, 116, 98, 0],
            &[0, 164, 255, 23, 0, 14, 203, 19, 0],
            &[0, 165, 232, 168, 150, 194, 67, 0, 0],
            &[0, 48, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ù' / 0xf9
        'ù' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 140, 81, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 2, 0, 0, 0],
            &[0, 88, 5, 0, 0, 0, 80, 12, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 180, 28, 0],
            &[0, 190, 19, 0, 0, 0, 204, 28, 0],
            &[0, 148, 79, 0, 0, 51, 240, 28, 0],
            &[0, 29, 199, 158, 157, 134, 151, 28, 0],
            &[0, 0, 0, 22, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ú' / 0xfa
        'ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 159, 10, 0, 0],
            &[0, 0, 0, 0, 156, 65, 0, 0, 0],
            &[0, 0, 0, 36, 58, 0, 0, 0, 0],
            &[0, 88, 5, 0, 0, 0, 80, 12, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 180, 28, 0],
            &[0, 190, 19, 0, 0, 0, 204, 28, 0],
            &[0, 148, 79, 0, 0, 51, 240, 28, 0],
            &[0, 29, 199, 158, 157, 134, 151, 28, 0],
            &[0, 0, 0, 22, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'û' / 0xfb
        'û' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 77, 152, 1, 0, 0, 0],
            &[0, 0, 38, 162, 87, 125, 0, 0, 0],
            &[0, 0, 85, 7, 0, 74, 19, 0, 0],
            &[0, 88, 5, 0, 0, 0, 80, 12, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 180, 28, 0],
            &[0, 190, 19, 0, 0, 0, 204, 28, 0],
            &[0, 148, 79, 0, 0, 51, 240, 28, 0],
            &[0, 29, 199, 158, 157, 134, 151, 28, 0],
            &[0, 0, 0, 22, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ü' / 0xfc
        'ü' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 0, 0, 16, 0, 0, 0],
            &[0, 0, 196, 54, 0, 203, 46, 0, 0],
            &[0, 0, 11, 0, 0, 11, 0, 0, 0],
            &[0, 88, 5, 0, 0, 0, 80, 12, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 180, 28, 0],
            &[0, 190, 19, 0, 0, 0, 204, 28, 0],
            &[0, 148, 79, 0, 0, 51, 240, 28, 0],
            &[0, 29, 199, 158, 157, 134, 151, 28, 0],
            &[0, 0, 0, 22, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ý' / 0xfd
        'ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 159, 10, 0, 0],
            &[0, 0, 0, 0, 156, 65, 0, 0, 0],
            &[0, 0, 0, 36, 58, 0, 0, 0, 0],
            &[0, 92, 0, 0, 0, 0, 49, 46, 0],
            &[0, 170, 35, 0, 0, 0, 175, 34, 0],
            &[0, 70, 133, 0, 0, 18, 187, 0, 0],
            &[0, 1, 195, 3, 0, 104, 96, 0, 0],
            &[0, 0, 122, 75, 0, 184, 12, 0, 0],
            &[0, 0, 25, 170, 32, 158, 0, 0, 0],
            &[0, 0, 0, 169, 138, 63, 0, 0, 0],
            &[0, 0, 0, 73, 217, 1, 0, 0, 0],
            &[0, 0, 0, 73, 126, 0, 0, 0, 0],
            &[0, 0, 0, 172, 29, 0, 0, 0, 0],
            &[0, 97, 77, 3, 0, 0, 0, 0, 0],
        ]),
        // letter: 'þ' / 0xfe
        'þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 56, 8, 0, 0, 0, 0, 0, 0],
            &[0, 178, 25, 0, 0, 0, 0, 0, 0],
            &[0, 178, 25, 0, 0, 0, 0, 0, 0],
            &[0, 178, 26, 86, 140, 112, 12, 0, 0],
            &[0, 178, 149, 87, 37, 99, 192, 3, 0],
            &[0, 178, 105, 0, 0, 0, 149, 74, 0],
            &[0, 178, 44, 0, 0, 0, 96, 123, 0],
            &[0, 178, 32, 0, 0, 0, 87, 129, 0],
            &[0, 178, 63, 0, 0, 0, 117, 100, 0],
            &[0, 178, 153, 0, 0, 9, 203, 26, 0],
            &[0, 178, 90, 173, 153, 198, 88, 0, 0],
            &[0, 178, 25, 1, 25, 5, 0, 0, 0],
            &[0, 178, 25, 0, 0, 0, 0, 0, 0],
            &[0, 64, 9, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ÿ' / 0xff
        'ÿ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 0, 0, 16, 0, 0, 0],
            &[0, 0, 196, 54, 0, 203, 46, 0, 0],
            &[0, 0, 11, 0, 0, 11, 0, 0, 0],
            &[0, 92, 0, 0, 0, 0, 49, 46, 0],
            &[0, 170, 35, 0, 0, 0, 175, 34, 0],
            &[0, 70, 133, 0, 0, 18, 187, 0, 0],
            &[0, 1, 195, 3, 0, 104, 96, 0, 0],
            &[0, 0, 122, 75, 0, 184, 12, 0, 0],
            &[0, 0, 25, 170, 32, 158, 0, 0, 0],
            &[0, 0, 0, 169, 138, 63, 0, 0, 0],
            &[0, 0, 0, 73, 217, 1, 0, 0, 0],
            &[0, 0, 0, 73, 126, 0, 0, 0, 0],
            &[0, 0, 0, 172, 29, 0, 0, 0, 0],
            &[0, 97, 77, 3, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ā' / 0x100
        'Ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 125, 171, 171, 171, 22, 0, 0],
            &[0, 0, 0, 2, 6, 0, 0, 0, 0],
            &[0, 0, 0, 104, 205, 0, 0, 0, 0],
            &[0, 0, 0, 185, 167, 40, 0, 0, 0],
            &[0, 0, 28, 174, 70, 129, 0, 0, 0],
            &[0, 0, 116, 88, 4, 197, 0, 0, 0],
            &[0, 0, 194, 11, 0, 151, 51, 0, 0],
            &[0, 38, 200, 44, 44, 108, 145, 0, 0],
            &[0, 127, 205, 145, 145, 148, 235, 2, 0],
            &[0, 202, 8, 0, 0, 0, 146, 63, 0],
            &[50, 162, 0, 0, 0, 0, 59, 152, 0],
            &[139, 74, 0, 0, 0, 0, 1, 207, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ā' / 0x101
        'ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 73, 111, 111, 111, 22, 0, 0],
            &[0, 0, 39, 60, 60, 60, 12, 0, 0],
            &[0, 1, 67, 126, 139, 91, 3, 0, 0],
            &[0, 22, 122, 48, 36, 122, 150, 0, 0],
            &[0, 0, 0, 0, 0, 0, 197, 0, 0],
            &[0, 0, 3, 50, 80, 91, 217, 0, 0],
            &[0, 57, 190, 124, 84, 66, 208, 0, 0],
            &[0, 192, 19, 0, 0, 0, 212, 0, 0],
            &[0, 197, 13, 0, 0, 70, 245, 0, 0],
            &[0, 76, 196, 147, 163, 113, 178, 0, 0],
            &[0, 0, 4, 26, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ă' / 0x102
        'Ă' => Some(&[
            &[0, 0, 127, 13, 0, 124, 11, 0, 0],
            &[0, 0, 39, 164, 165, 104, 0, 0, 0],
            &[0, 0, 0, 2, 6, 0, 0, 0, 0],
            &[0, 0, 0, 104, 205, 0, 0, 0, 0],
            &[0, 0, 0, 185, 167, 40, 0, 0, 0],
            &[0, 0, 28, 174, 70, 129, 0, 0, 0],
            &[0, 0, 116, 88, 4, 197, 0, 0, 0],
            &[0, 0, 194, 11, 0, 151, 51, 0, 0],
            &[0, 38, 200, 44, 44, 108, 145, 0, 0],
            &[0, 127, 205, 145, 145, 148, 235, 2, 0],
            &[0, 202, 8, 0, 0, 0, 146, 63, 0],
            &[50, 162, 0, 0, 0, 0, 59, 152, 0],
            &[139, 74, 0, 0, 0, 0, 1, 207, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ă' / 0x103
        'ă' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 0, 0, 29, 12, 0, 0],
            &[0, 0, 104, 114, 61, 172, 11, 0, 0],
            &[0, 0, 0, 71, 102, 27, 0, 0, 0],
            &[0, 1, 67, 126, 139, 91, 3, 0, 0],
            &[0, 22, 122, 48, 36, 122, 150, 0, 0],
            &[0, 0, 0, 0, 0, 0, 197, 0, 0],
            &[0, 0, 3, 50, 80, 91, 217, 0, 0],
            &[0, 57, 190, 124, 84, 66, 208, 0, 0],
            &[0, 192, 19, 0, 0, 0, 212, 0, 0],
            &[0, 197, 13, 0, 0, 70, 245, 0, 0],
            &[0, 76, 196, 147, 163, 113, 178, 0, 0],
            &[0, 0, 4, 26, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ą' / 0x104
        'Ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 6, 0, 0, 0, 0],
            &[0, 0, 0, 104, 205, 0, 0, 0, 0],
            &[0, 0, 0, 185, 167, 40, 0, 0, 0],
            &[0, 0, 28, 174, 70, 129, 0, 0, 0],
            &[0, 0, 116, 88, 4, 197, 0, 0, 0],
            &[0, 0, 194, 11, 0, 151, 51, 0, 0],
            &[0, 38, 200, 44, 44, 108, 145, 0, 0],
            &[0, 127, 205, 145, 145, 148, 235, 2, 0],
            &[0, 202, 8, 0, 0, 0, 146, 63, 0],
            &[50, 162, 0, 0, 0, 0, 59, 152, 0],
            &[139, 74, 0, 0, 0, 0, 1, 221, 6],
            &[0, 0, 0, 0, 0, 0, 33, 142, 2],
            &[0, 0, 0, 0, 0, 0, 149, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 0],
        ]),
        // letter: 'ą' / 0x105
        'ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 67, 126, 139, 91, 3, 0, 0],
            &[0, 22, 122, 48, 36, 122, 150, 0, 0],
            &[0, 0, 0, 0, 0, 0, 197, 0, 0],
            &[0, 0, 3, 50, 80, 91, 217, 0, 0],
            &[0, 57, 190, 124, 84, 66, 208, 0, 0],
            &[0, 192, 19, 0, 0, 0, 212, 0, 0],
            &[0, 197, 13, 0, 0, 70, 245, 0, 0],
            &[0, 76, 196, 147, 163, 113, 194, 0, 0],
            &[0, 0, 4, 26, 3, 61, 117, 0, 0],
            &[0, 0, 0, 0, 0, 175, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 0, 0],
        ]),
        // letter: 'Ć' / 0x106
        'Ć' => Some(&[
            &[0, 0, 0, 0, 0, 118, 121, 0, 0],
            &[0, 0, 0, 0, 64, 126, 0, 0, 0],
            &[0, 0, 0, 0, 15, 26, 4, 0, 0],
            &[0, 0, 47, 186, 176, 160, 192, 107, 0],
            &[0, 30, 206, 32, 0, 0, 0, 3, 0],
            &[0, 153, 76, 0, 0, 0, 0, 0, 0],
            &[0, 214, 2, 0, 0, 0, 0, 0, 0],
            &[4, 210, 0, 0, 0, 0, 0, 0, 0],
            &[4, 209, 0, 0, 0, 0, 0, 0, 0],
            &[0, 215, 2, 0, 0, 0, 0, 0, 0],
            &[0, 163, 64, 0, 0, 0, 0, 0, 0],
            &[0, 42, 204, 24, 0, 0, 0, 1, 0],
            &[0, 0, 64, 200, 179, 168, 195, 76, 0],
            &[0, 0, 0, 0, 16, 26, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 66, 0, 0],
        ]),
        // letter: 'ć' / 0x107
        'ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 111, 71, 0, 0],
            &[0, 0, 0, 0, 66, 155, 1, 0, 0],
            &[0, 0, 0, 0, 92, 2, 0, 0, 0],
            &[0, 0, 2, 80, 136, 137, 86, 0, 0],
            &[0, 5, 183, 132, 51, 47, 93, 0, 0],
            &[0, 105, 127, 0, 0, 0, 0, 0, 0],
            &[0, 174, 38, 0, 0, 0, 0, 0, 0],
            &[0, 190, 20, 0, 0, 0, 0, 0, 0],
            &[0, 161, 56, 0, 0, 0, 0, 0, 0],
            &[0, 68, 180, 5, 0, 0, 0, 0, 0],
            &[0, 0, 105, 199, 157, 163, 180, 0, 0],
            &[0, 0, 0, 3, 26, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĉ' / 0x108
        'Ĉ' => Some(&[
            &[0, 0, 0, 12, 180, 184, 22, 0, 0],
            &[0, 0, 0, 146, 36, 25, 154, 5, 0],
            &[0, 0, 0, 0, 15, 26, 4, 0, 0],
            &[0, 0, 47, 186, 176, 160, 192, 107, 0],
            &[0, 30, 206, 32, 0, 0, 0, 3, 0],
            &[0, 153, 76, 0, 0, 0, 0, 0, 0],
            &[0, 214, 2, 0, 0, 0, 0, 0, 0],
            &[4, 210, 0, 0, 0, 0, 0, 0, 0],
            &[4, 209, 0, 0, 0, 0, 0, 0, 0],
            &[0, 215, 2, 0, 0, 0, 0, 0, 0],
            &[0, 163, 64, 0, 0, 0, 0, 0, 0],
            &[0, 42, 204, 24, 0, 0, 0, 1, 0],
            &[0, 0, 64, 200, 179, 168, 195, 76, 0],
            &[0, 0, 0, 0, 16, 26, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 35, 0, 0, 0],
        ]),
        // letter: 'ĉ' / 0x109
        'ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 171, 46, 0, 0, 0],
            &[0, 0, 1, 156, 65, 174, 16, 0, 0],
            &[0, 0, 42, 50, 0, 22, 71, 0, 0],
            &[0, 0, 2, 80, 136, 137, 86, 0, 0],
            &[0, 5, 183, 132, 51, 47, 93, 0, 0],
            &[0, 105, 127, 0, 0, 0, 0, 0, 0],
            &[0, 174, 38, 0, 0, 0, 0, 0, 0],
            &[0, 190, 20, 0, 0, 0, 0, 0, 0],
            &[0, 161, 56, 0, 0, 0, 0, 0, 0],
            &[0, 68, 180, 5, 0, 0, 0, 0, 0],
            &[0, 0, 105, 199, 157, 163, 180, 0, 0],
            &[0, 0, 0, 3, 26, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ċ' / 0x10a
        'Ċ' => Some(&[
            &[0, 0, 0, 0, 112, 130, 0, 0, 0],
            &[0, 0, 0, 0, 46, 55, 0, 0, 0],
            &[0, 0, 0, 0, 15, 26, 4, 0, 0],
            &[0, 0, 47, 186, 176, 160, 192, 107, 0],
            &[0, 30, 206, 32, 0, 0, 0, 3, 0],
            &[0, 153, 76, 0, 0, 0, 0, 0, 0],
            &[0, 214, 2, 0, 0, 0, 0, 0, 0],
            &[4, 210, 0, 0, 0, 0, 0, 0, 0],
            &[4, 209, 0, 0, 0, 0, 0, 0, 0],
            &[0, 215, 2, 0, 0, 0, 0, 0, 0],
            &[0, 163, 64, 0, 0, 0, 0, 0, 0],
            &[0, 42, 204, 24, 0, 0, 0, 1, 0],
            &[0, 0, 64, 200, 179, 168, 195, 76, 0],
            &[0, 0, 0, 0, 16, 26, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ċ' / 0x10b
        'ċ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 86, 9, 0, 0, 0],
            &[0, 0, 0, 2, 209, 37, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 80, 136, 137, 86, 0, 0],
            &[0, 5, 183, 132, 51, 47, 93, 0, 0],
            &[0, 105, 127, 0, 0, 0, 0, 0, 0],
            &[0, 174, 38, 0, 0, 0, 0, 0, 0],
            &[0, 190, 20, 0, 0, 0, 0, 0, 0],
            &[0, 161, 56, 0, 0, 0, 0, 0, 0],
            &[0, 68, 180, 5, 0, 0, 0, 0, 0],
            &[0, 0, 105, 199, 157, 163, 180, 0, 0],
            &[0, 0, 0, 3, 26, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Č' / 0x10c
        'Č' => Some(&[
            &[0, 0, 0, 117, 86, 73, 133, 0, 0],
            &[0, 0, 0, 0, 165, 174, 2, 0, 0],
            &[0, 0, 0, 0, 15, 26, 4, 0, 0],
            &[0, 0, 47, 186, 176, 160, 192, 107, 0],
            &[0, 30, 206, 32, 0, 0, 0, 3, 0],
            &[0, 153, 76, 0, 0, 0, 0, 0, 0],
            &[0, 214, 2, 0, 0, 0, 0, 0, 0],
            &[4, 210, 0, 0, 0, 0, 0, 0, 0],
            &[4, 209, 0, 0, 0, 0, 0, 0, 0],
            &[0, 215, 2, 0, 0, 0, 0, 0, 0],
            &[0, 163, 64, 0, 0, 0, 0, 0, 0],
            &[0, 42, 204, 24, 0, 0, 0, 1, 0],
            &[0, 0, 64, 200, 179, 168, 195, 76, 0],
            &[0, 0, 0, 0, 16, 26, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 43, 0, 0, 40, 5, 0],
        ]),
        // letter: 'č' / 0x10d
        'č' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 87, 0, 52, 83, 0, 0],
            &[0, 0, 0, 133, 120, 167, 3, 0, 0],
            &[0, 0, 0, 4, 124, 21, 0, 0, 0],
            &[0, 0, 2, 80, 136, 137, 86, 0, 0],
            &[0, 5, 183, 132, 51, 47, 93, 0, 0],
            &[0, 105, 127, 0, 0, 0, 0, 0, 0],
            &[0, 174, 38, 0, 0, 0, 0, 0, 0],
            &[0, 190, 20, 0, 0, 0, 0, 0, 0],
            &[0, 161, 56, 0, 0, 0, 0, 0, 0],
            &[0, 68, 180, 5, 0, 0, 0, 0, 0],
            &[0, 0, 105, 199, 157, 163, 180, 0, 0],
            &[0, 0, 0, 3, 26, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ď' / 0x10e
        'Ď' => Some(&[
            &[0, 0, 113, 90, 69, 137, 0, 0, 0],
            &[0, 0, 0, 162, 177, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 211, 182, 185, 194, 130, 7, 0, 0],
            &[0, 207, 0, 0, 2, 104, 180, 0, 0],
            &[0, 207, 0, 0, 0, 0, 157, 71, 0],
            &[0, 207, 0, 0, 0, 0, 72, 143, 0],
            &[0, 207, 0, 0, 0, 0, 41, 170, 0],
            &[0, 207, 0, 0, 0, 0, 40, 166, 0],
            &[0, 207, 0, 0, 0, 0, 76, 135, 0],
            &[0, 207, 0, 0, 0, 0, 168, 57, 0],
            &[0, 207, 0, 0, 4, 119, 161, 0, 0],
            &[0, 212, 186, 185, 193, 114, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 0, 0, 39, 7, 0, 0],
        ]),
        // letter: 'ď' / 0x10f
        'ď' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 42, 90, 19, 111],
            &[0, 0, 0, 0, 0, 66, 141, 79, 98],
            &[0, 0, 0, 0, 0, 66, 141, 78, 6],
            &[0, 1, 90, 140, 94, 67, 141, 0, 0],
            &[0, 135, 143, 35, 80, 177, 141, 0, 0],
            &[9, 210, 1, 0, 0, 135, 141, 0, 0],
            &[50, 161, 0, 0, 0, 80, 141, 0, 0],
            &[62, 151, 0, 0, 0, 71, 141, 0, 0],
            &[40, 175, 0, 0, 0, 97, 141, 0, 0],
            &[1, 201, 21, 0, 2, 178, 141, 0, 0],
            &[0, 56, 198, 154, 175, 92, 141, 0, 0],
            &[0, 0, 0, 25, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Đ' / 0x110
        'Đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 177, 190, 184, 193, 147, 16, 0, 0],
            &[0, 178, 29, 0, 0, 79, 200, 7, 0],
            &[0, 178, 29, 0, 0, 0, 121, 107, 0],
            &[0, 178, 29, 0, 0, 0, 36, 179, 0],
            &[99, 255, 128, 97, 0, 0, 5, 205, 0],
            &[82, 255, 112, 80, 0, 0, 5, 201, 0],
            &[0, 178, 29, 0, 0, 0, 41, 170, 0],
            &[0, 178, 29, 0, 0, 0, 132, 93, 0],
            &[0, 178, 29, 0, 1, 93, 188, 2, 0],
            &[0, 178, 194, 186, 194, 132, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'đ' / 0x111
        'đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 114, 18, 0],
            &[0, 0, 0, 30, 72, 72, 251, 100, 28],
            &[0, 0, 0, 38, 92, 92, 255, 121, 37],
            &[0, 0, 46, 123, 117, 31, 178, 28, 0],
            &[0, 78, 196, 66, 52, 160, 198, 28, 0],
            &[0, 207, 26, 0, 0, 11, 242, 28, 0],
            &[13, 205, 0, 0, 0, 0, 194, 28, 0],
            &[25, 188, 0, 0, 0, 0, 183, 28, 0],
            &[7, 210, 0, 0, 0, 0, 211, 28, 0],
            &[0, 181, 62, 0, 0, 56, 243, 28, 0],
            &[0, 31, 195, 165, 161, 144, 153, 28, 0],
            &[0, 0, 0, 25, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ē' / 0x112
        'Ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 106, 171, 171, 171, 41, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 205, 186, 186, 186, 186, 33, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 184, 157, 157, 157, 141, 0, 0],
            &[0, 135, 93, 29, 29, 29, 26, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 208, 189, 189, 189, 189, 34, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ē' / 0x113
        'ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 75, 111, 111, 111, 20, 0, 0],
            &[0, 0, 40, 60, 60, 60, 11, 0, 0],
            &[0, 0, 23, 115, 142, 99, 7, 0, 0],
            &[0, 39, 199, 77, 34, 99, 176, 0, 0],
            &[0, 175, 47, 0, 0, 0, 150, 61, 0],
            &[0, 224, 76, 76, 76, 76, 147, 108, 0],
            &[5, 225, 95, 95, 95, 95, 95, 44, 0],
            &[0, 211, 3, 0, 0, 0, 0, 0, 0],
            &[0, 131, 112, 0, 0, 0, 1, 4, 0],
            &[0, 5, 141, 184, 150, 165, 184, 30, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĕ' / 0x114
        'Ĕ' => Some(&[
            &[0, 0, 111, 30, 0, 103, 32, 0, 0],
            &[0, 0, 24, 161, 163, 124, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 205, 186, 186, 186, 186, 33, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 184, 157, 157, 157, 141, 0, 0],
            &[0, 135, 93, 29, 29, 29, 26, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 208, 189, 189, 189, 189, 34, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĕ' / 0x115
        'ĕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 0, 0, 30, 11, 0, 0],
            &[0, 0, 108, 111, 62, 172, 9, 0, 0],
            &[0, 0, 0, 72, 101, 26, 0, 0, 0],
            &[0, 0, 23, 115, 142, 99, 7, 0, 0],
            &[0, 39, 199, 77, 34, 99, 176, 0, 0],
            &[0, 175, 47, 0, 0, 0, 150, 61, 0],
            &[0, 224, 76, 76, 76, 76, 147, 108, 0],
            &[5, 225, 95, 95, 95, 95, 95, 44, 0],
            &[0, 211, 3, 0, 0, 0, 0, 0, 0],
            &[0, 131, 112, 0, 0, 0, 1, 4, 0],
            &[0, 5, 141, 184, 150, 165, 184, 30, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ė' / 0x116
        'Ė' => Some(&[
            &[0, 0, 0, 52, 190, 0, 0, 0, 0],
            &[0, 0, 0, 17, 84, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 205, 186, 186, 186, 186, 33, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 184, 157, 157, 157, 141, 0, 0],
            &[0, 135, 93, 29, 29, 29, 26, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 208, 189, 189, 189, 189, 34, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ė' / 0x117
        'ė' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 74, 0, 0, 0, 0],
            &[0, 0, 0, 66, 182, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 115, 142, 99, 7, 0, 0],
            &[0, 39, 199, 77, 34, 99, 176, 0, 0],
            &[0, 175, 47, 0, 0, 0, 150, 61, 0],
            &[0, 224, 76, 76, 76, 76, 147, 108, 0],
            &[5, 225, 95, 95, 95, 95, 95, 44, 0],
            &[0, 211, 3, 0, 0, 0, 0, 0, 0],
            &[0, 131, 112, 0, 0, 0, 1, 4, 0],
            &[0, 5, 141, 184, 150, 165, 184, 30, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ę' / 0x118
        'Ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 205, 186, 186, 186, 186, 33, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 184, 157, 157, 157, 141, 0, 0],
            &[0, 135, 93, 29, 29, 29, 26, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 208, 189, 189, 197, 197, 34, 0],
            &[0, 0, 0, 0, 4, 152, 20, 0, 0],
            &[0, 0, 0, 0, 81, 98, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 2, 0, 0],
        ]),
        // letter: 'ę' / 0x119
        'ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 115, 142, 99, 7, 0, 0],
            &[0, 39, 199, 77, 34, 99, 176, 0, 0],
            &[0, 175, 47, 0, 0, 0, 150, 61, 0],
            &[0, 224, 76, 76, 76, 76, 147, 108, 0],
            &[5, 225, 95, 95, 95, 95, 95, 44, 0],
            &[0, 211, 3, 0, 0, 0, 0, 0, 0],
            &[0, 131, 112, 0, 0, 0, 1, 4, 0],
            &[0, 5, 141, 184, 150, 165, 246, 34, 0],
            &[0, 0, 0, 5, 28, 121, 65, 0, 0],
            &[0, 0, 0, 0, 2, 171, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 0, 0],
        ]),
        // letter: 'Ě' / 0x11a
        'Ě' => Some(&[
            &[0, 0, 59, 144, 29, 167, 10, 0, 0],
            &[0, 0, 0, 105, 213, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 205, 186, 186, 186, 186, 33, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 184, 157, 157, 157, 141, 0, 0],
            &[0, 135, 93, 29, 29, 29, 26, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 72, 0, 0, 0, 0, 0, 0],
            &[0, 135, 208, 189, 189, 189, 189, 34, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 0, 0, 22, 23, 0, 0],
        ]),
        // letter: 'ě' / 0x11b
        'ě' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 101, 31, 0, 109, 26, 0, 0],
            &[0, 0, 13, 178, 135, 97, 0, 0, 0],
            &[0, 0, 0, 42, 107, 0, 0, 0, 0],
            &[0, 0, 23, 115, 142, 99, 7, 0, 0],
            &[0, 39, 199, 77, 34, 99, 176, 0, 0],
            &[0, 175, 47, 0, 0, 0, 150, 61, 0],
            &[0, 224, 76, 76, 76, 76, 147, 108, 0],
            &[5, 225, 95, 95, 95, 95, 95, 44, 0],
            &[0, 211, 3, 0, 0, 0, 0, 0, 0],
            &[0, 131, 112, 0, 0, 0, 1, 4, 0],
            &[0, 5, 141, 184, 150, 165, 184, 30, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => Some(&[
            &[0, 0, 0, 100, 177, 122, 0, 0, 0],
            &[0, 0, 55, 127, 0, 112, 73, 0, 0],
            &[0, 0, 0, 1, 26, 18, 0, 0, 0],
            &[0, 0, 101, 202, 164, 172, 190, 6, 0],
            &[0, 89, 170, 5, 0, 0, 4, 0, 0],
            &[0, 208, 18, 0, 0, 0, 0, 0, 0],
            &[33, 183, 0, 0, 0, 0, 0, 0, 0],
            &[63, 149, 0, 0, 14, 24, 24, 6, 0],
            &[69, 144, 0, 0, 97, 165, 214, 64, 0],
            &[47, 170, 0, 0, 0, 0, 139, 64, 0],
            &[3, 216, 6, 0, 0, 0, 139, 64, 0],
            &[0, 124, 140, 1, 0, 0, 139, 64, 0],
            &[0, 1, 131, 201, 163, 179, 195, 38, 0],
            &[0, 0, 0, 3, 27, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 60, 2, 0, 0, 0],
        ]),
        // letter: 'ĝ' / 0x11d
        'ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 77, 152, 1, 0, 0, 0],
            &[0, 0, 38, 162, 87, 125, 0, 0, 0],
            &[0, 0, 85, 7, 0, 74, 19, 0, 0],
            &[0, 0, 56, 137, 134, 43, 65, 12, 0],
            &[0, 80, 188, 55, 39, 143, 205, 28, 0],
            &[0, 205, 21, 0, 0, 3, 226, 28, 0],
            &[14, 203, 0, 0, 0, 0, 175, 28, 0],
            &[26, 188, 0, 0, 0, 0, 167, 28, 0],
            &[6, 213, 0, 0, 0, 0, 194, 28, 0],
            &[0, 177, 69, 0, 0, 45, 242, 28, 0],
            &[0, 27, 194, 167, 162, 154, 171, 28, 0],
            &[0, 0, 0, 24, 17, 0, 176, 23, 0],
            &[0, 0, 0, 0, 0, 6, 209, 2, 0],
            &[0, 8, 70, 103, 98, 38, 0, 0, 0],
        ]),
        // letter: 'Ğ' / 0x11e
        'Ğ' => Some(&[
            &[0, 0, 64, 76, 0, 56, 79, 0, 0],
            &[0, 0, 4, 146, 159, 154, 9, 0, 0],
            &[0, 0, 0, 1, 26, 18, 0, 0, 0],
            &[0, 0, 101, 202, 164, 172, 190, 6, 0],
            &[0, 89, 170, 5, 0, 0, 4, 0, 0],
            &[0, 208, 18, 0, 0, 0, 0, 0, 0],
            &[33, 183, 0, 0, 0, 0, 0, 0, 0],
            &[63, 149, 0, 0, 14, 24, 24, 6, 0],
            &[69, 144, 0, 0, 97, 165, 214, 64, 0],
            &[47, 170, 0, 0, 0, 0, 139, 64, 0],
            &[3, 216, 6, 0, 0, 0, 139, 64, 0],
            &[0, 124, 140, 1, 0, 0, 139, 64, 0],
            &[0, 1, 131, 201, 163, 179, 195, 38, 0],
            &[0, 0, 0, 3, 27, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ğ' / 0x11f
        'ğ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 0, 0, 34, 7, 0, 0],
            &[0, 0, 122, 100, 65, 173, 3, 0, 0],
            &[0, 0, 1, 77, 100, 21, 0, 0, 0],
            &[0, 0, 56, 137, 134, 43, 65, 12, 0],
            &[0, 80, 188, 55, 39, 143, 205, 28, 0],
            &[0, 205, 21, 0, 0, 3, 226, 28, 0],
            &[14, 203, 0, 0, 0, 0, 175, 28, 0],
            &[26, 188, 0, 0, 0, 0, 167, 28, 0],
            &[6, 213, 0, 0, 0, 0, 194, 28, 0],
            &[0, 177, 69, 0, 0, 45, 242, 28, 0],
            &[0, 27, 194, 167, 162, 154, 171, 28, 0],
            &[0, 0, 0, 24, 17, 0, 176, 23, 0],
            &[0, 0, 0, 0, 0, 6, 209, 2, 0],
            &[0, 8, 70, 103, 98, 38, 0, 0, 0],
        ]),
        // letter: 'Ġ' / 0x120
        'Ġ' => Some(&[
            &[0, 0, 0, 10, 209, 22, 0, 0, 0],
            &[0, 0, 0, 0, 96, 5, 0, 0, 0],
            &[0, 0, 0, 1, 26, 18, 0, 0, 0],
            &[0, 0, 101, 202, 164, 172, 190, 6, 0],
            &[0, 89, 170, 5, 0, 0, 4, 0, 0],
            &[0, 208, 18, 0, 0, 0, 0, 0, 0],
            &[33, 183, 0, 0, 0, 0, 0, 0, 0],
            &[63, 149, 0, 0, 14, 24, 24, 6, 0],
            &[69, 144, 0, 0, 97, 165, 214, 64, 0],
            &[47, 170, 0, 0, 0, 0, 139, 64, 0],
            &[3, 216, 6, 0, 0, 0, 139, 64, 0],
            &[0, 124, 140, 1, 0, 0, 139, 64, 0],
            &[0, 1, 131, 201, 163, 179, 195, 38, 0],
            &[0, 0, 0, 3, 27, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ġ' / 0x121
        'ġ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 69, 0, 0, 0, 0],
            &[0, 0, 0, 78, 170, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 56, 137, 134, 43, 65, 12, 0],
            &[0, 80, 188, 55, 39, 143, 205, 28, 0],
            &[0, 205, 21, 0, 0, 3, 226, 28, 0],
            &[14, 203, 0, 0, 0, 0, 175, 28, 0],
            &[26, 188, 0, 0, 0, 0, 167, 28, 0],
            &[6, 213, 0, 0, 0, 0, 194, 28, 0],
            &[0, 177, 69, 0, 0, 45, 242, 28, 0],
            &[0, 27, 194, 167, 162, 154, 171, 28, 0],
            &[0, 0, 0, 24, 17, 0, 176, 23, 0],
            &[0, 0, 0, 0, 0, 6, 209, 2, 0],
            &[0, 8, 70, 103, 98, 38, 0, 0, 0],
        ]),
        // letter: 'Ģ' / 0x122
        'Ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 26, 18, 0, 0, 0],
            &[0, 0, 101, 202, 164, 172, 190, 6, 0],
            &[0, 89, 170, 5, 0, 0, 4, 0, 0],
            &[0, 208, 18, 0, 0, 0, 0, 0, 0],
            &[33, 183, 0, 0, 0, 0, 0, 0, 0],
            &[63, 149, 0, 0, 14, 24, 24, 6, 0],
            &[69, 144, 0, 0, 97, 165, 214, 64, 0],
            &[47, 170, 0, 0, 0, 0, 139, 64, 0],
            &[3, 216, 6, 0, 0, 0, 139, 64, 0],
            &[0, 124, 140, 1, 0, 0, 139, 64, 0],
            &[0, 1, 131, 201, 163, 179, 195, 38, 0],
            &[0, 0, 0, 3, 53, 15, 0, 0, 0],
            &[0, 0, 0, 0, 192, 4, 0, 0, 0],
            &[0, 0, 0, 7, 4, 0, 0, 0, 0],
        ]),
        // letter: 'ģ' / 0x123
        'ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 102, 2, 0, 0, 0],
            &[0, 0, 0, 19, 164, 0, 0, 0, 0],
            &[0, 0, 0, 43, 66, 0, 0, 0, 0],
            &[0, 0, 56, 137, 134, 43, 65, 12, 0],
            &[0, 80, 188, 55, 39, 143, 205, 28, 0],
            &[0, 205, 21, 0, 0, 3, 226, 28, 0],
            &[14, 203, 0, 0, 0, 0, 175, 28, 0],
            &[26, 188, 0, 0, 0, 0, 167, 28, 0],
            &[6, 213, 0, 0, 0, 0, 194, 28, 0],
            &[0, 177, 69, 0, 0, 45, 242, 28, 0],
            &[0, 27, 194, 167, 162, 154, 171, 28, 0],
            &[0, 0, 0, 24, 17, 0, 176, 23, 0],
            &[0, 0, 0, 0, 0, 6, 209, 2, 0],
            &[0, 8, 70, 103, 98, 38, 0, 0, 0],
        ]),
        // letter: 'Ĥ' / 0x124
        'Ĥ' => Some(&[
            &[0, 0, 2, 168, 183, 45, 0, 0, 0],
            &[0, 0, 121, 61, 10, 155, 19, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 0, 142, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 211, 157, 157, 157, 157, 212, 64, 0],
            &[0, 208, 32, 32, 32, 32, 157, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 45, 0, 0, 0, 0],
        ]),
        // letter: 'ĥ' / 0x125
        'ĥ' => Some(&[
            &[5, 161, 35, 151, 51, 0, 0, 0, 0],
            &[25, 31, 0, 2, 54, 0, 0, 0, 0],
            &[0, 0, 114, 18, 0, 0, 0, 0, 0],
            &[0, 0, 178, 29, 0, 0, 0, 0, 0],
            &[0, 0, 178, 29, 0, 0, 0, 0, 0],
            &[0, 0, 178, 30, 86, 139, 109, 10, 0],
            &[0, 0, 178, 147, 81, 34, 104, 190, 0],
            &[0, 0, 178, 98, 0, 0, 0, 190, 24],
            &[0, 0, 178, 38, 0, 0, 0, 162, 46],
            &[0, 0, 178, 29, 0, 0, 0, 161, 46],
            &[0, 0, 178, 29, 0, 0, 0, 161, 46],
            &[0, 0, 178, 29, 0, 0, 0, 161, 46],
            &[0, 0, 178, 29, 0, 0, 0, 161, 46],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 200, 94, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ħ' / 0x126
        'Ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 0, 142, 64, 0],
            &[9, 217, 9, 9, 9, 9, 152, 73, 3],
            &[166, 255, 166, 166, 166, 166, 255, 230, 66],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 211, 157, 157, 157, 157, 212, 64, 0],
            &[0, 208, 32, 32, 32, 32, 157, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ħ' / 0x127
        'ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 114, 18, 0, 0, 0, 0, 0, 0],
            &[72, 250, 101, 72, 58, 0, 0, 0, 0],
            &[92, 255, 121, 92, 75, 0, 0, 0, 0],
            &[0, 178, 30, 86, 139, 109, 10, 0, 0],
            &[0, 178, 147, 81, 34, 104, 190, 0, 0],
            &[0, 178, 98, 0, 0, 0, 190, 24, 0],
            &[0, 178, 38, 0, 0, 0, 162, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĩ' / 0x128
        'Ĩ' => Some(&[
            &[0, 0, 108, 154, 34, 13, 94, 0, 0],
            &[0, 21, 124, 23, 153, 175, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 156, 187, 222, 162, 76, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 16, 156, 187, 223, 162, 76, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĩ' / 0x129
        'ĩ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 18, 0, 0, 22, 0, 0],
            &[0, 5, 178, 159, 138, 100, 107, 0, 0],
            &[0, 16, 47, 0, 49, 88, 3, 0, 0],
            &[0, 57, 114, 114, 98, 0, 0, 0, 0],
            &[0, 17, 44, 57, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 171, 159, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ī' / 0x12a
        'Ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 125, 171, 171, 171, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 156, 187, 222, 162, 76, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 16, 156, 187, 223, 162, 76, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ī' / 0x12b
        'ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 111, 111, 111, 14, 0, 0],
            &[0, 0, 44, 60, 60, 60, 7, 0, 0],
            &[0, 57, 114, 114, 98, 0, 0, 0, 0],
            &[0, 17, 44, 57, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 171, 159, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => Some(&[
            &[0, 0, 127, 13, 0, 124, 11, 0, 0],
            &[0, 0, 39, 164, 165, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 156, 187, 222, 162, 76, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 16, 156, 187, 223, 162, 76, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĭ' / 0x12d
        'ĭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 0, 0, 34, 7, 0, 0],
            &[0, 0, 122, 100, 65, 173, 3, 0, 0],
            &[0, 0, 1, 77, 100, 21, 0, 0, 0],
            &[0, 57, 114, 114, 98, 0, 0, 0, 0],
            &[0, 17, 44, 57, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 171, 159, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Į' / 0x12e
        'Į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 156, 187, 222, 162, 76, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 16, 156, 187, 223, 177, 76, 0, 0],
            &[0, 0, 0, 0, 70, 107, 0, 0, 0],
            &[0, 0, 0, 0, 178, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 0, 0, 0],
        ]),
        // letter: 'į' / 0x12f
        'į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 84, 0, 0, 0, 0],
            &[0, 0, 0, 42, 205, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 114, 114, 98, 0, 0, 0, 0],
            &[0, 17, 44, 57, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 178, 166, 73, 0],
            &[0, 0, 0, 0, 4, 152, 20, 0, 0],
            &[0, 0, 0, 0, 81, 98, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 2, 0, 0],
        ]),
        // letter: 'İ' / 0x130
        'İ' => Some(&[
            &[0, 0, 0, 76, 166, 0, 0, 0, 0],
            &[0, 0, 0, 28, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 156, 187, 222, 162, 76, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 16, 156, 187, 223, 162, 76, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ı' / 0x131
        'ı' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 114, 114, 98, 0, 0, 0, 0],
            &[0, 17, 44, 57, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 171, 159, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĳ' / 0x132
        'Ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 124, 82, 0, 0, 0, 206, 0, 0],
            &[0, 124, 82, 0, 0, 0, 207, 0, 0],
            &[0, 124, 82, 0, 0, 0, 207, 0, 0],
            &[0, 124, 82, 0, 0, 0, 207, 0, 0],
            &[0, 124, 82, 0, 0, 0, 207, 0, 0],
            &[0, 124, 82, 0, 0, 0, 207, 0, 0],
            &[0, 124, 82, 0, 0, 0, 207, 0, 0],
            &[0, 70, 46, 0, 0, 1, 210, 0, 0],
            &[0, 4, 0, 0, 0, 71, 167, 0, 0],
            &[0, 83, 191, 156, 169, 198, 31, 0, 0],
            &[0, 0, 6, 35, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĳ' / 0x133
        'ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 23, 73, 0, 0, 29, 67, 0, 0],
            &[0, 69, 180, 0, 0, 84, 165, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 72, 0, 0, 21, 71, 0, 0],
            &[0, 46, 161, 0, 0, 48, 159, 0, 0],
            &[0, 46, 161, 0, 0, 48, 159, 0, 0],
            &[0, 46, 161, 0, 0, 48, 159, 0, 0],
            &[0, 46, 161, 0, 0, 48, 159, 0, 0],
            &[0, 46, 161, 0, 0, 48, 159, 0, 0],
            &[0, 46, 161, 0, 0, 48, 159, 0, 0],
            &[0, 46, 161, 0, 0, 48, 159, 0, 0],
            &[0, 0, 0, 0, 0, 49, 159, 0, 0],
            &[0, 0, 0, 0, 0, 85, 136, 0, 0],
            &[0, 6, 71, 105, 91, 21, 0, 0, 0],
        ]),
        // letter: 'Ĵ' / 0x134
        'Ĵ' => Some(&[
            &[0, 0, 0, 0, 43, 181, 171, 2, 0],
            &[0, 0, 0, 18, 154, 10, 61, 124, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 155, 51, 0, 0],
            &[0, 0, 0, 0, 0, 156, 51, 0, 0],
            &[0, 0, 0, 0, 0, 156, 51, 0, 0],
            &[0, 0, 0, 0, 0, 156, 51, 0, 0],
            &[0, 0, 0, 0, 0, 156, 51, 0, 0],
            &[0, 0, 0, 0, 0, 156, 51, 0, 0],
            &[0, 0, 0, 0, 0, 156, 51, 0, 0],
            &[0, 0, 0, 0, 0, 167, 43, 0, 0],
            &[0, 4, 0, 0, 16, 217, 7, 0, 0],
            &[0, 88, 186, 155, 204, 97, 0, 0, 0],
            &[0, 0, 9, 35, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 19, 0, 0],
        ]),
        // letter: 'ĵ' / 0x135
        'ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 158, 68, 0, 0],
            &[0, 0, 0, 0, 135, 77, 170, 31, 0],
            &[0, 0, 0, 25, 67, 0, 11, 82, 0],
            &[0, 0, 9, 114, 114, 114, 31, 0, 0],
            &[0, 0, 2, 39, 52, 165, 71, 0, 0],
            &[0, 0, 0, 0, 0, 136, 71, 0, 0],
            &[0, 0, 0, 0, 0, 136, 71, 0, 0],
            &[0, 0, 0, 0, 0, 136, 71, 0, 0],
            &[0, 0, 0, 0, 0, 136, 71, 0, 0],
            &[0, 0, 0, 0, 0, 136, 71, 0, 0],
            &[0, 0, 0, 0, 0, 136, 71, 0, 0],
            &[0, 0, 0, 0, 0, 137, 71, 0, 0],
            &[0, 0, 0, 0, 0, 169, 50, 0, 0],
            &[0, 0, 75, 105, 83, 6, 0, 0, 0],
        ]),
        // letter: 'Ķ' / 0x136
        'Ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 14, 194, 42, 0],
            &[0, 207, 0, 0, 4, 182, 70, 0, 0],
            &[0, 207, 0, 0, 157, 104, 0, 0, 0],
            &[0, 207, 0, 124, 141, 0, 0, 0, 0],
            &[0, 207, 85, 243, 17, 0, 0, 0, 0],
            &[0, 211, 164, 82, 156, 0, 0, 0, 0],
            &[0, 209, 8, 0, 161, 72, 0, 0, 0],
            &[0, 207, 0, 0, 17, 201, 15, 0, 0],
            &[0, 207, 0, 0, 0, 76, 159, 0, 0],
            &[0, 207, 0, 0, 0, 0, 160, 75, 0],
            &[0, 0, 0, 0, 30, 0, 0, 0, 0],
            &[0, 0, 0, 24, 172, 0, 0, 0, 0],
            &[0, 0, 0, 11, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ķ' / 0x137
        'ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 45, 87, 0, 0, 0, 0, 0, 0],
            &[0, 71, 136, 0, 0, 0, 0, 0, 0],
            &[0, 71, 136, 0, 0, 0, 0, 0, 0],
            &[0, 71, 136, 0, 0, 0, 72, 41, 0],
            &[0, 71, 136, 0, 0, 85, 167, 3, 0],
            &[0, 71, 136, 0, 75, 176, 5, 0, 0],
            &[0, 71, 133, 66, 208, 9, 0, 0, 0],
            &[0, 71, 176, 165, 168, 66, 0, 0, 0],
            &[0, 71, 187, 5, 15, 198, 21, 0, 0],
            &[0, 71, 136, 0, 0, 59, 178, 1, 0],
            &[0, 71, 136, 0, 0, 0, 127, 114, 0],
            &[0, 0, 0, 0, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 187, 8, 0, 0, 0],
            &[0, 0, 0, 6, 5, 0, 0, 0, 0],
        ]),
        // letter: 'ĸ' / 0x138
        'ĸ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 31, 61, 0, 0, 0, 72, 41, 0],
            &[0, 71, 136, 0, 0, 85, 167, 3, 0],
            &[0, 71, 136, 0, 75, 176, 5, 0, 0],
            &[0, 71, 133, 66, 208, 9, 0, 0, 0],
            &[0, 71, 176, 165, 168, 66, 0, 0, 0],
            &[0, 71, 187, 5, 15, 198, 21, 0, 0],
            &[0, 71, 136, 0, 0, 59, 178, 1, 0],
            &[0, 71, 136, 0, 0, 0, 127, 114, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĺ' / 0x139
        'Ĺ' => Some(&[
            &[0, 0, 62, 174, 2, 0, 0, 0, 0],
            &[0, 27, 155, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 171, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 235, 193, 193, 193, 193, 35, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 15, 0, 0, 0, 0],
        ]),
        // letter: 'ĺ' / 0x13a
        'ĺ' => Some(&[
            &[0, 0, 0, 7, 171, 36, 0, 0, 0],
            &[0, 0, 0, 28, 27, 0, 0, 0, 0],
            &[0, 81, 159, 163, 140, 0, 0, 0, 0],
            &[0, 0, 0, 8, 208, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 171, 159, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 189, 10, 0, 0],
        ]),
        // letter: 'Ļ' / 0x13b
        'Ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 171, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 235, 193, 193, 193, 193, 35, 0],
            &[0, 0, 0, 0, 25, 5, 0, 0, 0],
            &[0, 0, 0, 0, 191, 5, 0, 0, 0],
            &[0, 0, 0, 6, 4, 0, 0, 0, 0],
        ]),
        // letter: 'ļ' / 0x13c
        'ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 159, 163, 140, 0, 0, 0, 0],
            &[0, 0, 0, 8, 208, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 171, 159, 73, 0],
            &[0, 0, 0, 0, 30, 0, 0, 0, 0],
            &[0, 0, 0, 15, 180, 0, 0, 0, 0],
            &[0, 0, 0, 11, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ľ' / 0x13d
        'Ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 171, 0, 0, 7, 191, 0, 0],
            &[0, 35, 172, 0, 0, 66, 96, 0, 0],
            &[0, 35, 172, 0, 0, 28, 3, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 235, 193, 193, 193, 193, 35, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ľ' / 0x13e
        'ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 159, 163, 140, 0, 69, 61, 0],
            &[0, 0, 0, 8, 208, 0, 151, 26, 0],
            &[0, 0, 0, 0, 207, 0, 85, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 171, 159, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 171, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 33, 177, 0, 0],
            &[0, 35, 172, 0, 0, 19, 123, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 172, 0, 0, 0, 0, 0, 0],
            &[0, 35, 235, 193, 193, 193, 193, 35, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŀ' / 0x140
        'ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[81, 159, 163, 140, 0, 0, 0, 0, 0],
            &[0, 0, 8, 208, 0, 0, 0, 0, 0],
            &[0, 0, 0, 207, 0, 0, 0, 0, 0],
            &[0, 0, 0, 207, 0, 0, 0, 0, 0],
            &[0, 0, 0, 207, 0, 0, 2, 47, 0],
            &[0, 0, 0, 207, 0, 0, 38, 248, 2],
            &[0, 0, 0, 207, 0, 0, 0, 15, 0],
            &[0, 0, 0, 207, 0, 0, 0, 0, 0],
            &[0, 0, 0, 207, 0, 0, 0, 0, 0],
            &[0, 0, 0, 207, 0, 0, 0, 0, 0],
            &[111, 161, 172, 241, 171, 159, 73, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ł' / 0x141
        'Ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 147, 59, 0, 0, 0, 0, 0],
            &[0, 0, 147, 59, 0, 0, 0, 0, 0],
            &[0, 0, 147, 59, 0, 0, 0, 0, 0],
            &[0, 0, 147, 59, 10, 26, 0, 0, 0],
            &[0, 0, 147, 136, 186, 55, 0, 0, 0],
            &[0, 29, 255, 187, 7, 0, 0, 0, 0],
            &[0, 135, 184, 59, 0, 0, 0, 0, 0],
            &[0, 0, 147, 59, 0, 0, 0, 0, 0],
            &[0, 0, 147, 59, 0, 0, 0, 0, 0],
            &[0, 0, 147, 208, 193, 193, 193, 143, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ł' / 0x142
        'ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 159, 163, 140, 0, 0, 0, 0],
            &[0, 0, 0, 8, 208, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 48, 87, 0, 0],
            &[0, 0, 0, 10, 255, 153, 18, 0, 0],
            &[0, 0, 70, 184, 255, 0, 0, 0, 0],
            &[0, 0, 63, 7, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 0, 0, 0, 0],
            &[0, 111, 161, 172, 241, 171, 159, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ń' / 0x143
        'Ń' => Some(&[
            &[0, 0, 0, 0, 62, 175, 2, 0, 0],
            &[0, 0, 0, 26, 155, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 213, 109, 0, 0, 0, 139, 64, 0],
            &[0, 185, 188, 6, 0, 0, 139, 64, 0],
            &[0, 195, 94, 104, 0, 0, 139, 64, 0],
            &[0, 200, 4, 189, 5, 0, 139, 64, 0],
            &[0, 200, 0, 99, 99, 0, 139, 64, 0],
            &[0, 200, 0, 5, 190, 4, 139, 64, 0],
            &[0, 200, 0, 0, 105, 94, 139, 64, 0],
            &[0, 200, 0, 0, 7, 190, 137, 64, 0],
            &[0, 200, 0, 0, 0, 110, 213, 64, 0],
            &[0, 200, 0, 0, 0, 9, 232, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 15, 0, 0],
        ]),
        // letter: 'ń' / 0x144
        'ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 153, 26, 0, 0],
            &[0, 0, 0, 0, 128, 95, 0, 0, 0],
            &[0, 0, 0, 18, 76, 0, 0, 0, 0],
            &[0, 77, 2, 88, 139, 112, 13, 0, 0],
            &[0, 178, 143, 82, 35, 97, 194, 0, 0],
            &[0, 178, 98, 0, 0, 0, 186, 26, 0],
            &[0, 178, 43, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ņ' / 0x145
        'Ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 213, 109, 0, 0, 0, 139, 64, 0],
            &[0, 185, 188, 6, 0, 0, 139, 64, 0],
            &[0, 195, 94, 104, 0, 0, 139, 64, 0],
            &[0, 200, 4, 189, 5, 0, 139, 64, 0],
            &[0, 200, 0, 99, 99, 0, 139, 64, 0],
            &[0, 200, 0, 5, 190, 4, 139, 64, 0],
            &[0, 200, 0, 0, 105, 94, 139, 64, 0],
            &[0, 200, 0, 0, 7, 190, 137, 64, 0],
            &[0, 200, 0, 0, 0, 110, 213, 64, 0],
            &[0, 200, 0, 0, 0, 9, 232, 64, 0],
            &[0, 0, 0, 6, 24, 0, 0, 0, 0],
            &[0, 0, 0, 78, 118, 0, 0, 0, 0],
            &[0, 0, 0, 11, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ņ' / 0x146
        'ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 77, 2, 88, 139, 112, 13, 0, 0],
            &[0, 178, 143, 82, 35, 97, 194, 0, 0],
            &[0, 178, 98, 0, 0, 0, 186, 26, 0],
            &[0, 178, 43, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 0, 0, 2, 28, 0, 0, 0, 0],
            &[0, 0, 0, 49, 147, 0, 0, 0, 0],
            &[0, 0, 0, 11, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ň' / 0x147
        'Ň' => Some(&[
            &[0, 0, 67, 136, 35, 164, 7, 0, 0],
            &[0, 0, 0, 115, 207, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 213, 109, 0, 0, 0, 139, 64, 0],
            &[0, 185, 188, 6, 0, 0, 139, 64, 0],
            &[0, 195, 94, 104, 0, 0, 139, 64, 0],
            &[0, 200, 4, 189, 5, 0, 139, 64, 0],
            &[0, 200, 0, 99, 99, 0, 139, 64, 0],
            &[0, 200, 0, 5, 190, 4, 139, 64, 0],
            &[0, 200, 0, 0, 105, 94, 139, 64, 0],
            &[0, 200, 0, 0, 7, 190, 137, 64, 0],
            &[0, 200, 0, 0, 0, 110, 213, 64, 0],
            &[0, 200, 0, 0, 0, 9, 232, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 0, 0, 25, 20, 0, 0],
        ]),
        // letter: 'ň' / 0x148
        'ň' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 90, 42, 0, 98, 38, 0, 0],
            &[0, 0, 6, 174, 125, 118, 0, 0, 0],
            &[0, 0, 0, 31, 117, 0, 0, 0, 0],
            &[0, 77, 2, 88, 139, 112, 13, 0, 0],
            &[0, 178, 143, 82, 35, 97, 194, 0, 0],
            &[0, 178, 98, 0, 0, 0, 186, 26, 0],
            &[0, 178, 43, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŉ' / 0x149
        'ŉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 66, 0, 0, 0, 0, 0, 0],
            &[0, 208, 9, 0, 0, 0, 0, 0, 0],
            &[26, 163, 51, 26, 66, 136, 117, 16, 0],
            &[34, 38, 114, 164, 107, 36, 93, 199, 1],
            &[0, 0, 114, 160, 0, 0, 0, 186, 27],
            &[0, 0, 114, 107, 0, 0, 0, 161, 46],
            &[0, 0, 114, 93, 0, 0, 0, 161, 46],
            &[0, 0, 114, 93, 0, 0, 0, 161, 46],
            &[0, 0, 114, 93, 0, 0, 0, 161, 46],
            &[0, 0, 114, 93, 0, 0, 0, 161, 46],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 213, 109, 0, 0, 0, 139, 64, 0],
            &[0, 185, 188, 6, 0, 0, 139, 64, 0],
            &[0, 195, 94, 104, 0, 0, 139, 64, 0],
            &[0, 200, 4, 189, 5, 0, 139, 64, 0],
            &[0, 200, 0, 99, 99, 0, 139, 64, 0],
            &[0, 200, 0, 5, 190, 4, 139, 64, 0],
            &[0, 200, 0, 0, 105, 94, 139, 64, 0],
            &[0, 200, 0, 0, 7, 190, 137, 64, 0],
            &[0, 200, 0, 0, 0, 110, 213, 64, 0],
            &[0, 200, 0, 0, 0, 9, 255, 112, 0],
            &[0, 0, 0, 0, 0, 0, 164, 44, 0],
            &[0, 0, 0, 17, 58, 95, 200, 3, 0],
            &[0, 0, 0, 20, 129, 113, 17, 0, 0],
        ]),
        // letter: 'ŋ' / 0x14b
        'ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 77, 2, 88, 139, 112, 13, 0, 0],
            &[0, 178, 143, 82, 35, 97, 194, 0, 0],
            &[0, 178, 98, 0, 0, 0, 186, 26, 0],
            &[0, 178, 43, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 161, 46, 0],
            &[0, 178, 29, 0, 0, 0, 255, 92, 0],
            &[0, 178, 29, 0, 0, 0, 255, 93, 0],
            &[0, 178, 29, 0, 0, 0, 255, 93, 0],
            &[0, 0, 0, 0, 0, 0, 161, 46, 0],
            &[0, 0, 0, 0, 0, 0, 169, 33, 0],
            &[0, 0, 0, 0, 44, 78, 16, 0, 0],
        ]),
        // letter: 'Ō' / 0x14c
        'Ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 125, 171, 171, 171, 22, 0, 0],
            &[0, 0, 0, 11, 29, 2, 0, 0, 0],
            &[0, 5, 158, 183, 158, 198, 57, 0, 0],
            &[0, 135, 105, 0, 0, 19, 204, 12, 0],
            &[3, 210, 2, 0, 0, 0, 119, 96, 0],
            &[45, 165, 0, 0, 0, 0, 60, 150, 0],
            &[68, 141, 0, 0, 0, 0, 37, 173, 0],
            &[70, 142, 0, 0, 0, 0, 39, 172, 0],
            &[45, 167, 0, 0, 0, 0, 63, 148, 0],
            &[4, 211, 2, 0, 0, 0, 123, 92, 0],
            &[0, 137, 109, 0, 0, 24, 205, 10, 0],
            &[0, 7, 161, 188, 167, 199, 53, 0, 0],
            &[0, 0, 0, 13, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ō' / 0x14d
        'ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 111, 111, 111, 14, 0, 0],
            &[0, 0, 44, 60, 60, 60, 7, 0, 0],
            &[0, 0, 37, 125, 140, 85, 2, 0, 0],
            &[0, 65, 192, 62, 36, 119, 162, 0, 0],
            &[0, 199, 26, 0, 0, 0, 162, 57, 0],
            &[12, 201, 0, 0, 0, 0, 92, 118, 0],
            &[24, 183, 0, 0, 0, 0, 81, 130, 0],
            &[4, 208, 0, 0, 0, 0, 116, 98, 0],
            &[0, 156, 80, 0, 0, 14, 203, 19, 0],
            &[0, 13, 167, 168, 150, 194, 67, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => Some(&[
            &[0, 0, 127, 13, 0, 124, 11, 0, 0],
            &[0, 0, 39, 164, 165, 104, 0, 0, 0],
            &[0, 0, 0, 11, 29, 2, 0, 0, 0],
            &[0, 5, 158, 183, 158, 198, 57, 0, 0],
            &[0, 135, 105, 0, 0, 19, 204, 12, 0],
            &[3, 210, 2, 0, 0, 0, 119, 96, 0],
            &[45, 165, 0, 0, 0, 0, 60, 150, 0],
            &[68, 141, 0, 0, 0, 0, 37, 173, 0],
            &[70, 142, 0, 0, 0, 0, 39, 172, 0],
            &[45, 167, 0, 0, 0, 0, 63, 148, 0],
            &[4, 211, 2, 0, 0, 0, 123, 92, 0],
            &[0, 137, 109, 0, 0, 24, 205, 10, 0],
            &[0, 7, 161, 188, 167, 199, 53, 0, 0],
            &[0, 0, 0, 13, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŏ' / 0x14f
        'ŏ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 0, 0, 34, 7, 0, 0],
            &[0, 0, 122, 100, 65, 173, 3, 0, 0],
            &[0, 0, 1, 77, 100, 21, 0, 0, 0],
            &[0, 0, 37, 125, 140, 85, 2, 0, 0],
            &[0, 65, 192, 62, 36, 119, 162, 0, 0],
            &[0, 199, 26, 0, 0, 0, 162, 57, 0],
            &[12, 201, 0, 0, 0, 0, 92, 118, 0],
            &[24, 183, 0, 0, 0, 0, 81, 130, 0],
            &[4, 208, 0, 0, 0, 0, 116, 98, 0],
            &[0, 156, 80, 0, 0, 14, 203, 19, 0],
            &[0, 13, 167, 168, 150, 194, 67, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ő' / 0x150
        'Ő' => Some(&[
            &[0, 0, 4, 188, 27, 154, 64, 0, 0],
            &[0, 0, 112, 59, 69, 102, 0, 0, 0],
            &[0, 0, 0, 11, 29, 2, 0, 0, 0],
            &[0, 5, 158, 183, 158, 198, 57, 0, 0],
            &[0, 135, 105, 0, 0, 19, 204, 12, 0],
            &[3, 210, 2, 0, 0, 0, 119, 96, 0],
            &[45, 165, 0, 0, 0, 0, 60, 150, 0],
            &[68, 141, 0, 0, 0, 0, 37, 173, 0],
            &[70, 142, 0, 0, 0, 0, 39, 172, 0],
            &[45, 167, 0, 0, 0, 0, 63, 148, 0],
            &[4, 211, 2, 0, 0, 0, 123, 92, 0],
            &[0, 137, 109, 0, 0, 24, 205, 10, 0],
            &[0, 7, 161, 188, 167, 199, 53, 0, 0],
            &[0, 0, 0, 13, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 32, 17, 46, 0, 0],
        ]),
        // letter: 'ő' / 0x151
        'ő' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 109, 58, 69, 97, 0, 0],
            &[0, 0, 40, 162, 12, 178, 13, 0, 0],
            &[0, 0, 77, 7, 57, 27, 0, 0, 0],
            &[0, 0, 37, 125, 140, 85, 2, 0, 0],
            &[0, 65, 192, 62, 36, 119, 162, 0, 0],
            &[0, 199, 26, 0, 0, 0, 162, 57, 0],
            &[12, 201, 0, 0, 0, 0, 92, 118, 0],
            &[24, 183, 0, 0, 0, 0, 81, 130, 0],
            &[4, 208, 0, 0, 0, 0, 116, 98, 0],
            &[0, 156, 80, 0, 0, 14, 203, 19, 0],
            &[0, 13, 167, 168, 150, 194, 67, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Œ' / 0x152
        'Œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 8, 0, 0, 0, 0],
            &[0, 26, 180, 176, 234, 191, 186, 179, 0],
            &[0, 180, 53, 0, 183, 20, 0, 0, 0],
            &[20, 201, 0, 0, 183, 20, 0, 0, 0],
            &[62, 151, 0, 0, 183, 20, 0, 0, 0],
            &[79, 134, 0, 0, 183, 165, 157, 116, 0],
            &[76, 133, 0, 0, 183, 47, 29, 21, 0],
            &[57, 154, 0, 0, 183, 20, 0, 0, 0],
            &[14, 204, 0, 0, 183, 20, 0, 0, 0],
            &[0, 172, 71, 0, 183, 20, 0, 0, 0],
            &[0, 22, 186, 182, 235, 195, 189, 183, 0],
            &[0, 0, 0, 10, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'œ' / 0x153
        'œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 134, 70, 0, 89, 125, 10, 0],
            &[19, 185, 37, 157, 108, 110, 86, 136, 0],
            &[100, 89, 0, 50, 233, 3, 0, 177, 0],
            &[135, 50, 0, 17, 221, 72, 72, 196, 0],
            &[147, 43, 0, 8, 220, 88, 88, 83, 0],
            &[127, 62, 0, 28, 220, 0, 0, 0, 0],
            &[69, 129, 0, 81, 209, 33, 0, 0, 0],
            &[1, 169, 155, 169, 30, 183, 141, 131, 0],
            &[0, 0, 23, 2, 0, 4, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŕ' / 0x154
        'Ŕ' => Some(&[
            &[0, 0, 0, 0, 77, 161, 0, 0, 0],
            &[0, 0, 0, 36, 149, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 203, 183, 189, 186, 61, 0, 0],
            &[0, 135, 72, 0, 0, 29, 217, 19, 0],
            &[0, 135, 72, 0, 0, 0, 151, 69, 0],
            &[0, 135, 72, 0, 0, 0, 165, 53, 0],
            &[0, 135, 74, 3, 16, 96, 172, 1, 0],
            &[0, 135, 203, 183, 213, 139, 3, 0, 0],
            &[0, 135, 72, 0, 24, 198, 4, 0, 0],
            &[0, 135, 72, 0, 0, 115, 113, 0, 0],
            &[0, 135, 72, 0, 0, 5, 200, 24, 0],
            &[0, 135, 72, 0, 0, 0, 68, 163, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 10, 0, 0],
        ]),
        // letter: 'ŕ' / 0x155
        'ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 107, 0, 0],
            &[0, 0, 0, 0, 31, 179, 12, 0, 0],
            &[0, 0, 0, 0, 81, 12, 0, 0, 0],
            &[0, 71, 114, 114, 22, 75, 138, 109, 0],
            &[0, 23, 49, 158, 149, 127, 48, 62, 0],
            &[0, 0, 0, 127, 168, 0, 0, 0, 0],
            &[0, 0, 0, 127, 97, 0, 0, 0, 0],
            &[0, 0, 0, 127, 80, 0, 0, 0, 0],
            &[0, 0, 0, 127, 79, 0, 0, 0, 0],
            &[0, 0, 0, 127, 79, 0, 0, 0, 0],
            &[0, 132, 165, 215, 200, 163, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŗ' / 0x156
        'Ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 203, 183, 189, 186, 61, 0, 0],
            &[0, 135, 72, 0, 0, 29, 217, 19, 0],
            &[0, 135, 72, 0, 0, 0, 151, 69, 0],
            &[0, 135, 72, 0, 0, 0, 165, 53, 0],
            &[0, 135, 74, 3, 16, 96, 172, 1, 0],
            &[0, 135, 203, 183, 213, 139, 3, 0, 0],
            &[0, 135, 72, 0, 24, 198, 4, 0, 0],
            &[0, 135, 72, 0, 0, 115, 113, 0, 0],
            &[0, 135, 72, 0, 0, 5, 200, 24, 0],
            &[0, 135, 72, 0, 0, 0, 68, 163, 0],
            &[0, 0, 0, 0, 25, 5, 0, 0, 0],
            &[0, 0, 0, 0, 191, 5, 0, 0, 0],
            &[0, 0, 0, 6, 4, 0, 0, 0, 0],
        ]),
        // letter: 'ŗ' / 0x157
        'ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 71, 114, 114, 22, 75, 138, 109, 0],
            &[0, 23, 49, 158, 149, 127, 48, 62, 0],
            &[0, 0, 0, 127, 168, 0, 0, 0, 0],
            &[0, 0, 0, 127, 97, 0, 0, 0, 0],
            &[0, 0, 0, 127, 80, 0, 0, 0, 0],
            &[0, 0, 0, 127, 79, 0, 0, 0, 0],
            &[0, 0, 0, 127, 79, 0, 0, 0, 0],
            &[0, 132, 165, 215, 200, 163, 122, 0, 0],
            &[0, 0, 0, 17, 13, 0, 0, 0, 0],
            &[0, 0, 0, 153, 43, 0, 0, 0, 0],
            &[0, 0, 1, 9, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ř' / 0x158
        'Ř' => Some(&[
            &[0, 0, 82, 121, 45, 157, 3, 0, 0],
            &[0, 0, 0, 132, 198, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 203, 183, 189, 186, 61, 0, 0],
            &[0, 135, 72, 0, 0, 29, 217, 19, 0],
            &[0, 135, 72, 0, 0, 0, 151, 69, 0],
            &[0, 135, 72, 0, 0, 0, 165, 53, 0],
            &[0, 135, 74, 3, 16, 96, 172, 1, 0],
            &[0, 135, 203, 183, 213, 139, 3, 0, 0],
            &[0, 135, 72, 0, 24, 198, 4, 0, 0],
            &[0, 135, 72, 0, 0, 115, 113, 0, 0],
            &[0, 135, 72, 0, 0, 5, 200, 24, 0],
            &[0, 135, 72, 0, 0, 0, 68, 163, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 0, 0, 29, 16, 0, 0],
        ]),
        // letter: 'ř' / 0x159
        'ř' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 113, 0, 26, 110, 0, 0],
            &[0, 0, 0, 86, 141, 179, 17, 0, 0],
            &[0, 0, 0, 0, 102, 47, 0, 0, 0],
            &[0, 71, 114, 114, 22, 75, 138, 109, 0],
            &[0, 23, 49, 158, 149, 127, 48, 62, 0],
            &[0, 0, 0, 127, 168, 0, 0, 0, 0],
            &[0, 0, 0, 127, 97, 0, 0, 0, 0],
            &[0, 0, 0, 127, 80, 0, 0, 0, 0],
            &[0, 0, 0, 127, 79, 0, 0, 0, 0],
            &[0, 0, 0, 127, 79, 0, 0, 0, 0],
            &[0, 132, 165, 215, 200, 163, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ś' / 0x15a
        'Ś' => Some(&[
            &[0, 0, 0, 0, 26, 195, 17, 0, 0],
            &[0, 0, 0, 6, 156, 27, 0, 0, 0],
            &[0, 0, 0, 7, 29, 11, 0, 0, 0],
            &[0, 4, 149, 198, 160, 178, 190, 6, 0],
            &[0, 102, 141, 0, 0, 0, 3, 0, 0],
            &[0, 145, 71, 0, 0, 0, 0, 0, 0],
            &[0, 95, 166, 4, 0, 0, 0, 0, 0],
            &[0, 0, 123, 202, 110, 13, 0, 0, 0],
            &[0, 0, 0, 12, 108, 205, 109, 0, 0],
            &[0, 0, 0, 0, 0, 9, 196, 52, 0],
            &[0, 0, 0, 0, 0, 0, 126, 85, 0],
            &[0, 4, 0, 0, 0, 7, 195, 33, 0],
            &[0, 145, 187, 159, 160, 195, 84, 0, 0],
            &[0, 0, 3, 25, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 29, 0, 0],
        ]),
        // letter: 'ś' / 0x15b
        'ś' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 159, 14, 0, 0],
            &[0, 0, 0, 0, 149, 74, 0, 0, 0],
            &[0, 0, 0, 30, 64, 0, 0, 0, 0],
            &[0, 0, 33, 123, 144, 120, 41, 0, 0],
            &[0, 17, 205, 64, 34, 63, 61, 0, 0],
            &[0, 50, 165, 0, 0, 0, 0, 0, 0],
            &[0, 3, 171, 141, 34, 0, 0, 0, 0],
            &[0, 0, 0, 61, 163, 174, 28, 0, 0],
            &[0, 0, 0, 0, 0, 61, 186, 0, 0],
            &[0, 6, 0, 0, 0, 29, 192, 0, 0],
            &[0, 76, 188, 159, 155, 195, 59, 0, 0],
            &[0, 0, 1, 22, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => Some(&[
            &[0, 0, 0, 114, 177, 108, 0, 0, 0],
            &[0, 0, 66, 116, 0, 123, 61, 0, 0],
            &[0, 0, 0, 7, 29, 11, 0, 0, 0],
            &[0, 4, 149, 198, 160, 178, 190, 6, 0],
            &[0, 102, 141, 0, 0, 0, 3, 0, 0],
            &[0, 145, 71, 0, 0, 0, 0, 0, 0],
            &[0, 95, 166, 4, 0, 0, 0, 0, 0],
            &[0, 0, 123, 202, 110, 13, 0, 0, 0],
            &[0, 0, 0, 12, 108, 205, 109, 0, 0],
            &[0, 0, 0, 0, 0, 9, 196, 52, 0],
            &[0, 0, 0, 0, 0, 0, 126, 85, 0],
            &[0, 4, 0, 0, 0, 7, 195, 33, 0],
            &[0, 145, 187, 159, 160, 195, 84, 0, 0],
            &[0, 0, 3, 25, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 61, 0, 0, 0, 0],
        ]),
        // letter: 'ŝ' / 0x15d
        'ŝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 158, 3, 0, 0, 0],
            &[0, 0, 32, 166, 80, 134, 0, 0, 0],
            &[0, 0, 83, 10, 0, 69, 24, 0, 0],
            &[0, 0, 33, 123, 144, 120, 41, 0, 0],
            &[0, 17, 205, 64, 34, 63, 61, 0, 0],
            &[0, 50, 165, 0, 0, 0, 0, 0, 0],
            &[0, 3, 171, 141, 34, 0, 0, 0, 0],
            &[0, 0, 0, 61, 163, 174, 28, 0, 0],
            &[0, 0, 0, 0, 0, 61, 186, 0, 0],
            &[0, 6, 0, 0, 0, 29, 192, 0, 0],
            &[0, 76, 188, 159, 155, 195, 59, 0, 0],
            &[0, 0, 1, 22, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ş' / 0x15e
        'Ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 29, 11, 0, 0, 0],
            &[0, 4, 149, 198, 160, 178, 190, 6, 0],
            &[0, 102, 141, 0, 0, 0, 3, 0, 0],
            &[0, 145, 71, 0, 0, 0, 0, 0, 0],
            &[0, 95, 166, 4, 0, 0, 0, 0, 0],
            &[0, 0, 123, 202, 110, 13, 0, 0, 0],
            &[0, 0, 0, 12, 108, 205, 109, 0, 0],
            &[0, 0, 0, 0, 0, 9, 196, 52, 0],
            &[0, 0, 0, 0, 0, 0, 126, 85, 0],
            &[0, 4, 0, 0, 0, 7, 195, 33, 0],
            &[0, 145, 187, 160, 166, 195, 84, 0, 0],
            &[0, 0, 3, 106, 101, 0, 0, 0, 0],
            &[0, 0, 0, 41, 164, 54, 0, 0, 0],
            &[0, 0, 15, 86, 42, 0, 0, 0, 0],
        ]),
        // letter: 'ş' / 0x15f
        'ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 123, 144, 120, 41, 0, 0],
            &[0, 17, 205, 64, 34, 63, 61, 0, 0],
            &[0, 50, 165, 0, 0, 0, 0, 0, 0],
            &[0, 3, 171, 141, 34, 0, 0, 0, 0],
            &[0, 0, 0, 61, 163, 174, 28, 0, 0],
            &[0, 0, 0, 0, 0, 61, 186, 0, 0],
            &[0, 6, 0, 0, 0, 29, 192, 0, 0],
            &[0, 76, 188, 159, 161, 195, 59, 0, 0],
            &[0, 0, 1, 93, 120, 0, 0, 0, 0],
            &[0, 0, 0, 38, 158, 63, 0, 0, 0],
            &[0, 0, 11, 86, 45, 0, 0, 0, 0],
        ]),
        // letter: 'Š' / 0x160
        'Š' => Some(&[
            &[0, 0, 32, 164, 19, 167, 27, 0, 0],
            &[0, 0, 0, 64, 226, 52, 0, 0, 0],
            &[0, 0, 0, 7, 29, 11, 0, 0, 0],
            &[0, 4, 149, 198, 160, 178, 190, 6, 0],
            &[0, 102, 141, 0, 0, 0, 3, 0, 0],
            &[0, 145, 71, 0, 0, 0, 0, 0, 0],
            &[0, 95, 166, 4, 0, 0, 0, 0, 0],
            &[0, 0, 123, 202, 110, 13, 0, 0, 0],
            &[0, 0, 0, 12, 108, 205, 109, 0, 0],
            &[0, 0, 0, 0, 0, 9, 196, 52, 0],
            &[0, 0, 0, 0, 0, 0, 126, 85, 0],
            &[0, 4, 0, 0, 0, 7, 195, 33, 0],
            &[0, 145, 187, 159, 160, 195, 84, 0, 0],
            &[0, 0, 3, 25, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 7, 0, 11, 34, 0, 0],
        ]),
        // letter: 'š' / 0x161
        'š' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 103, 29, 0, 111, 24, 0, 0],
            &[0, 0, 14, 179, 137, 93, 0, 0, 0],
            &[0, 0, 0, 44, 105, 0, 0, 0, 0],
            &[0, 0, 33, 123, 144, 120, 41, 0, 0],
            &[0, 17, 205, 64, 34, 63, 61, 0, 0],
            &[0, 50, 165, 0, 0, 0, 0, 0, 0],
            &[0, 3, 171, 141, 34, 0, 0, 0, 0],
            &[0, 0, 0, 61, 163, 174, 28, 0, 0],
            &[0, 0, 0, 0, 0, 61, 186, 0, 0],
            &[0, 6, 0, 0, 0, 29, 192, 0, 0],
            &[0, 76, 188, 159, 155, 195, 59, 0, 0],
            &[0, 0, 1, 22, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ţ' / 0x162
        'Ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[61, 186, 186, 200, 227, 186, 186, 138, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 53, 160, 0, 0, 0, 0],
            &[0, 0, 0, 81, 84, 0, 0, 0, 0],
            &[0, 0, 0, 41, 164, 54, 0, 0, 0],
            &[0, 0, 15, 86, 42, 0, 0, 0, 0],
        ]),
        // letter: 'ţ' / 0x163
        'ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 61, 1, 0, 0, 0, 0],
            &[0, 0, 0, 183, 4, 0, 0, 0, 0],
            &[1, 94, 110, 226, 117, 114, 114, 0, 0],
            &[1, 64, 64, 216, 67, 64, 64, 0, 0],
            &[0, 0, 0, 203, 4, 0, 0, 0, 0],
            &[0, 0, 0, 203, 4, 0, 0, 0, 0],
            &[0, 0, 0, 203, 4, 0, 0, 0, 0],
            &[0, 0, 0, 201, 10, 0, 0, 0, 0],
            &[0, 0, 0, 171, 60, 0, 0, 0, 0],
            &[0, 0, 0, 50, 205, 162, 160, 7, 0],
            &[0, 0, 0, 0, 51, 146, 13, 0, 0],
            &[0, 0, 0, 0, 30, 145, 84, 0, 0],
            &[0, 0, 0, 4, 86, 53, 0, 0, 0],
        ]),
        // letter: 'Ť' / 0x164
        'Ť' => Some(&[
            &[0, 0, 82, 121, 45, 157, 3, 0, 0],
            &[0, 0, 0, 132, 198, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[61, 186, 186, 200, 227, 186, 186, 138, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 0, 0, 29, 16, 0, 0],
        ]),
        // letter: 'ť' / 0x165
        'ť' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 147, 0, 0],
            &[0, 0, 0, 61, 1, 30, 146, 0, 0],
            &[0, 0, 0, 183, 4, 46, 25, 0, 0],
            &[1, 94, 110, 226, 117, 114, 114, 0, 0],
            &[1, 64, 64, 216, 67, 64, 64, 0, 0],
            &[0, 0, 0, 203, 4, 0, 0, 0, 0],
            &[0, 0, 0, 203, 4, 0, 0, 0, 0],
            &[0, 0, 0, 203, 4, 0, 0, 0, 0],
            &[0, 0, 0, 201, 10, 0, 0, 0, 0],
            &[0, 0, 0, 171, 60, 0, 0, 0, 0],
            &[0, 0, 0, 50, 205, 155, 160, 7, 0],
            &[0, 0, 0, 0, 3, 29, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŧ' / 0x166
        'Ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[61, 186, 186, 200, 227, 186, 186, 138, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 55, 117, 170, 255, 117, 104, 0, 0],
            &[0, 29, 61, 114, 216, 61, 54, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 52, 155, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŧ' / 0x167
        'ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 61, 1, 0, 0, 0, 0],
            &[0, 0, 0, 183, 4, 0, 0, 0, 0],
            &[1, 94, 110, 226, 117, 114, 114, 0, 0],
            &[1, 64, 64, 216, 67, 64, 64, 0, 0],
            &[0, 0, 0, 203, 4, 0, 0, 0, 0],
            &[3, 151, 151, 255, 156, 151, 151, 14, 0],
            &[0, 9, 9, 212, 14, 9, 9, 0, 0],
            &[0, 0, 0, 201, 10, 0, 0, 0, 0],
            &[0, 0, 0, 171, 60, 0, 0, 0, 0],
            &[0, 0, 0, 50, 205, 155, 160, 7, 0],
            &[0, 0, 0, 0, 3, 29, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ũ' / 0x168
        'Ũ' => Some(&[
            &[0, 0, 108, 154, 34, 13, 94, 0, 0],
            &[0, 21, 124, 23, 153, 175, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 0, 142, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 146, 61, 0],
            &[0, 189, 21, 0, 0, 0, 172, 38, 0],
            &[0, 119, 117, 0, 0, 28, 205, 0, 0],
            &[0, 9, 170, 183, 165, 201, 52, 0, 0],
            &[0, 0, 0, 15, 25, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ũ' / 0x169
        'ũ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 18, 0, 0, 22, 0, 0],
            &[0, 5, 178, 159, 138, 100, 107, 0, 0],
            &[0, 16, 47, 0, 49, 88, 3, 0, 0],
            &[0, 88, 5, 0, 0, 0, 80, 12, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 180, 28, 0],
            &[0, 190, 19, 0, 0, 0, 204, 28, 0],
            &[0, 148, 79, 0, 0, 51, 240, 28, 0],
            &[0, 29, 199, 158, 157, 134, 151, 28, 0],
            &[0, 0, 0, 22, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ū' / 0x16a
        'Ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 125, 171, 171, 171, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 0, 142, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 146, 61, 0],
            &[0, 189, 21, 0, 0, 0, 172, 38, 0],
            &[0, 119, 117, 0, 0, 28, 205, 0, 0],
            &[0, 9, 170, 183, 165, 201, 52, 0, 0],
            &[0, 0, 0, 15, 25, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ū' / 0x16b
        'ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 111, 111, 111, 14, 0, 0],
            &[0, 0, 44, 60, 60, 60, 7, 0, 0],
            &[0, 88, 5, 0, 0, 0, 80, 12, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 180, 28, 0],
            &[0, 190, 19, 0, 0, 0, 204, 28, 0],
            &[0, 148, 79, 0, 0, 51, 240, 28, 0],
            &[0, 29, 199, 158, 157, 134, 151, 28, 0],
            &[0, 0, 0, 22, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => Some(&[
            &[0, 0, 127, 13, 0, 124, 11, 0, 0],
            &[0, 0, 39, 164, 165, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 0, 142, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 146, 61, 0],
            &[0, 189, 21, 0, 0, 0, 172, 38, 0],
            &[0, 119, 117, 0, 0, 28, 205, 0, 0],
            &[0, 9, 170, 183, 165, 201, 52, 0, 0],
            &[0, 0, 0, 15, 25, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŭ' / 0x16d
        'ŭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 0, 0, 34, 7, 0, 0],
            &[0, 0, 122, 100, 65, 173, 3, 0, 0],
            &[0, 0, 1, 77, 100, 21, 0, 0, 0],
            &[0, 88, 5, 0, 0, 0, 80, 12, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 180, 28, 0],
            &[0, 190, 19, 0, 0, 0, 204, 28, 0],
            &[0, 148, 79, 0, 0, 51, 240, 28, 0],
            &[0, 29, 199, 158, 157, 134, 151, 28, 0],
            &[0, 0, 0, 22, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ů' / 0x16e
        'Ů' => Some(&[
            &[0, 0, 58, 98, 4, 151, 0, 0, 0],
            &[0, 0, 9, 154, 151, 63, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 0, 142, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 146, 61, 0],
            &[0, 189, 21, 0, 0, 0, 172, 38, 0],
            &[0, 119, 117, 0, 0, 28, 205, 0, 0],
            &[0, 9, 170, 183, 165, 201, 52, 0, 0],
            &[0, 0, 0, 15, 25, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 155, 148, 65, 0, 0, 0],
        ]),
        // letter: 'ů' / 0x16f
        'ů' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 51, 83, 4, 0, 0, 0],
            &[0, 0, 35, 148, 67, 129, 0, 0, 0],
            &[0, 0, 41, 137, 50, 136, 0, 0, 0],
            &[0, 0, 0, 70, 103, 10, 0, 0, 0],
            &[0, 88, 5, 0, 0, 0, 80, 12, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 180, 28, 0],
            &[0, 190, 19, 0, 0, 0, 204, 28, 0],
            &[0, 148, 79, 0, 0, 51, 240, 28, 0],
            &[0, 29, 199, 158, 157, 134, 151, 28, 0],
            &[0, 0, 0, 22, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ű' / 0x170
        'Ű' => Some(&[
            &[0, 0, 4, 188, 27, 154, 64, 0, 0],
            &[0, 0, 112, 59, 69, 102, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 0, 142, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 146, 61, 0],
            &[0, 189, 21, 0, 0, 0, 172, 38, 0],
            &[0, 119, 117, 0, 0, 28, 205, 0, 0],
            &[0, 9, 170, 183, 165, 201, 52, 0, 0],
            &[0, 0, 0, 15, 25, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 32, 17, 46, 0, 0],
        ]),
        // letter: 'ű' / 0x171
        'ű' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 109, 58, 69, 97, 0, 0],
            &[0, 0, 40, 162, 12, 178, 13, 0, 0],
            &[0, 0, 77, 7, 57, 27, 0, 0, 0],
            &[0, 88, 5, 0, 0, 0, 80, 12, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 180, 28, 0],
            &[0, 190, 19, 0, 0, 0, 204, 28, 0],
            &[0, 148, 79, 0, 0, 51, 240, 28, 0],
            &[0, 29, 199, 158, 157, 134, 151, 28, 0],
            &[0, 0, 0, 22, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ų' / 0x172
        'Ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 0, 0, 0, 0, 142, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 143, 64, 0],
            &[0, 207, 0, 0, 0, 0, 146, 61, 0],
            &[0, 189, 21, 0, 0, 0, 172, 38, 0],
            &[0, 119, 117, 0, 0, 28, 205, 0, 0],
            &[0, 9, 170, 183, 174, 252, 52, 0, 0],
            &[0, 0, 0, 42, 171, 18, 0, 0, 0],
            &[0, 0, 0, 137, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 0, 0, 0, 0],
        ]),
        // letter: 'ų' / 0x173
        'ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 88, 5, 0, 0, 0, 80, 12, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 179, 28, 0],
            &[0, 196, 11, 0, 0, 0, 180, 28, 0],
            &[0, 190, 19, 0, 0, 0, 204, 28, 0],
            &[0, 148, 79, 0, 0, 51, 240, 28, 0],
            &[0, 29, 199, 158, 157, 134, 167, 29, 0],
            &[0, 0, 0, 22, 9, 41, 136, 0, 0],
            &[0, 0, 0, 0, 0, 160, 19, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 0, 0],
        ]),
        // letter: 'Ŵ' / 0x174
        'Ŵ' => Some(&[
            &[0, 0, 2, 168, 183, 45, 0, 0, 0],
            &[0, 0, 121, 61, 10, 155, 19, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[119, 76, 0, 0, 0, 0, 0, 196, 0],
            &[85, 110, 0, 0, 0, 0, 6, 190, 0],
            &[50, 144, 0, 27, 75, 0, 38, 158, 0],
            &[16, 177, 0, 104, 187, 0, 73, 124, 0],
            &[0, 192, 0, 154, 143, 16, 107, 90, 0],
            &[0, 191, 0, 161, 85, 73, 141, 56, 0],
            &[0, 167, 48, 137, 26, 132, 170, 22, 0],
            &[0, 132, 132, 79, 0, 161, 181, 0, 0],
            &[0, 98, 208, 21, 0, 154, 170, 0, 0],
            &[0, 63, 218, 0, 0, 106, 175, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 45, 0, 0, 0, 0],
        ]),
        // letter: 'ŵ' / 0x175
        'ŵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 159, 3, 0, 0, 0],
            &[0, 0, 30, 167, 78, 137, 0, 0, 0],
            &[0, 0, 81, 11, 0, 67, 26, 0, 0],
            &[75, 11, 0, 0, 0, 0, 0, 76, 8],
            &[130, 63, 0, 3, 13, 0, 0, 190, 0],
            &[78, 117, 0, 93, 210, 0, 7, 184, 0],
            &[26, 170, 0, 165, 153, 37, 54, 139, 0],
            &[0, 197, 6, 178, 71, 113, 105, 89, 0],
            &[0, 176, 85, 116, 7, 177, 151, 39, 0],
            &[0, 124, 185, 40, 0, 169, 180, 1, 0],
            &[0, 72, 216, 0, 0, 100, 190, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŷ' / 0x176
        'Ŷ' => Some(&[
            &[0, 0, 2, 168, 183, 45, 0, 0, 0],
            &[0, 0, 121, 61, 10, 155, 19, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[38, 186, 0, 0, 0, 0, 80, 140, 0],
            &[0, 157, 62, 0, 0, 0, 193, 20, 0],
            &[0, 30, 183, 0, 0, 78, 130, 0, 0],
            &[0, 0, 144, 62, 0, 185, 15, 0, 0],
            &[0, 0, 22, 179, 75, 119, 0, 0, 0],
            &[0, 0, 0, 132, 204, 10, 0, 0, 0],
            &[0, 0, 0, 47, 156, 0, 0, 0, 0],
            &[0, 0, 0, 45, 155, 0, 0, 0, 0],
            &[0, 0, 0, 45, 155, 0, 0, 0, 0],
            &[0, 0, 0, 45, 155, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 45, 0, 0, 0, 0],
        ]),
        // letter: 'ŷ' / 0x177
        'ŷ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 77, 152, 1, 0, 0, 0],
            &[0, 0, 38, 162, 87, 125, 0, 0, 0],
            &[0, 0, 85, 7, 0, 74, 19, 0, 0],
            &[0, 92, 0, 0, 0, 0, 49, 46, 0],
            &[0, 170, 35, 0, 0, 0, 175, 34, 0],
            &[0, 70, 133, 0, 0, 18, 187, 0, 0],
            &[0, 1, 195, 3, 0, 104, 96, 0, 0],
            &[0, 0, 122, 75, 0, 184, 12, 0, 0],
            &[0, 0, 25, 170, 32, 158, 0, 0, 0],
            &[0, 0, 0, 169, 138, 63, 0, 0, 0],
            &[0, 0, 0, 73, 217, 1, 0, 0, 0],
            &[0, 0, 0, 73, 126, 0, 0, 0, 0],
            &[0, 0, 0, 172, 29, 0, 0, 0, 0],
            &[0, 97, 77, 3, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ÿ' / 0x178
        'Ÿ' => Some(&[
            &[0, 0, 125, 32, 0, 129, 27, 0, 0],
            &[0, 0, 98, 22, 0, 102, 19, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[38, 186, 0, 0, 0, 0, 80, 140, 0],
            &[0, 157, 62, 0, 0, 0, 193, 20, 0],
            &[0, 30, 183, 0, 0, 78, 130, 0, 0],
            &[0, 0, 144, 62, 0, 185, 15, 0, 0],
            &[0, 0, 22, 179, 75, 119, 0, 0, 0],
            &[0, 0, 0, 132, 204, 10, 0, 0, 0],
            &[0, 0, 0, 47, 156, 0, 0, 0, 0],
            &[0, 0, 0, 45, 155, 0, 0, 0, 0],
            &[0, 0, 0, 45, 155, 0, 0, 0, 0],
            &[0, 0, 0, 45, 155, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ź' / 0x179
        'Ź' => Some(&[
            &[0, 0, 0, 0, 36, 192, 11, 0, 0],
            &[0, 0, 0, 11, 158, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 182, 182, 182, 185, 223, 0, 0],
            &[0, 0, 0, 0, 0, 96, 116, 0, 0],
            &[0, 0, 0, 0, 6, 197, 8, 0, 0],
            &[0, 0, 0, 0, 110, 103, 0, 0, 0],
            &[0, 0, 0, 12, 197, 4, 0, 0, 0],
            &[0, 0, 0, 125, 90, 0, 0, 0, 0],
            &[0, 0, 18, 195, 2, 0, 0, 0, 0],
            &[0, 0, 139, 77, 0, 0, 0, 0, 0],
            &[0, 27, 189, 0, 0, 0, 0, 0, 0],
            &[0, 126, 219, 189, 189, 189, 189, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 25, 0, 0],
        ]),
        // letter: 'ź' / 0x17a
        'ź' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 153, 26, 0, 0],
            &[0, 0, 0, 0, 128, 95, 0, 0, 0],
            &[0, 0, 0, 18, 76, 0, 0, 0, 0],
            &[0, 30, 114, 114, 114, 114, 105, 0, 0],
            &[0, 14, 53, 53, 53, 118, 167, 0, 0],
            &[0, 0, 0, 0, 19, 195, 15, 0, 0],
            &[0, 0, 0, 0, 172, 56, 0, 0, 0],
            &[0, 0, 0, 107, 120, 0, 0, 0, 0],
            &[0, 0, 45, 179, 1, 0, 0, 0, 0],
            &[0, 9, 190, 24, 0, 0, 0, 0, 0],
            &[0, 123, 222, 182, 182, 182, 172, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ż' / 0x17b
        'Ż' => Some(&[
            &[0, 0, 0, 33, 206, 3, 0, 0, 0],
            &[0, 0, 0, 9, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 182, 182, 182, 185, 223, 0, 0],
            &[0, 0, 0, 0, 0, 96, 116, 0, 0],
            &[0, 0, 0, 0, 6, 197, 8, 0, 0],
            &[0, 0, 0, 0, 110, 103, 0, 0, 0],
            &[0, 0, 0, 12, 197, 4, 0, 0, 0],
            &[0, 0, 0, 125, 90, 0, 0, 0, 0],
            &[0, 0, 18, 195, 2, 0, 0, 0, 0],
            &[0, 0, 139, 77, 0, 0, 0, 0, 0],
            &[0, 27, 189, 0, 0, 0, 0, 0, 0],
            &[0, 126, 219, 189, 189, 189, 189, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ż' / 0x17c
        'ż' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 82, 0, 0, 0, 0],
            &[0, 0, 0, 48, 200, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 114, 114, 114, 114, 105, 0, 0],
            &[0, 14, 53, 53, 53, 118, 167, 0, 0],
            &[0, 0, 0, 0, 19, 195, 15, 0, 0],
            &[0, 0, 0, 0, 172, 56, 0, 0, 0],
            &[0, 0, 0, 107, 120, 0, 0, 0, 0],
            &[0, 0, 45, 179, 1, 0, 0, 0, 0],
            &[0, 9, 190, 24, 0, 0, 0, 0, 0],
            &[0, 123, 222, 182, 182, 182, 172, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ž' / 0x17d
        'Ž' => Some(&[
            &[0, 0, 42, 157, 21, 169, 19, 0, 0],
            &[0, 0, 0, 81, 222, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 182, 182, 182, 185, 223, 0, 0],
            &[0, 0, 0, 0, 0, 96, 116, 0, 0],
            &[0, 0, 0, 0, 6, 197, 8, 0, 0],
            &[0, 0, 0, 0, 110, 103, 0, 0, 0],
            &[0, 0, 0, 12, 197, 4, 0, 0, 0],
            &[0, 0, 0, 125, 90, 0, 0, 0, 0],
            &[0, 0, 18, 195, 2, 0, 0, 0, 0],
            &[0, 0, 139, 77, 0, 0, 0, 0, 0],
            &[0, 27, 189, 0, 0, 0, 0, 0, 0],
            &[0, 126, 219, 189, 189, 189, 189, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 40, 4, 0, 16, 30, 0, 0],
        ]),
        // letter: 'ž' / 0x17e
        'ž' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 90, 42, 0, 98, 38, 0, 0],
            &[0, 0, 6, 174, 125, 118, 0, 0, 0],
            &[0, 0, 0, 31, 117, 0, 0, 0, 0],
            &[0, 30, 114, 114, 114, 114, 105, 0, 0],
            &[0, 14, 53, 53, 53, 118, 167, 0, 0],
            &[0, 0, 0, 0, 19, 195, 15, 0, 0],
            &[0, 0, 0, 0, 172, 56, 0, 0, 0],
            &[0, 0, 0, 107, 120, 0, 0, 0, 0],
            &[0, 0, 45, 179, 1, 0, 0, 0, 0],
            &[0, 9, 190, 24, 0, 0, 0, 0, 0],
            &[0, 123, 222, 182, 182, 182, 172, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ſ' / 0x17f
        'ſ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 74, 164, 172, 49, 0, 0],
            &[0, 0, 63, 183, 20, 5, 7, 0, 0],
            &[0, 0, 144, 67, 0, 0, 0, 0, 0],
            &[0, 0, 160, 44, 0, 0, 0, 0, 0],
            &[0, 0, 160, 44, 0, 0, 0, 0, 0],
            &[0, 0, 160, 44, 0, 0, 0, 0, 0],
            &[0, 0, 160, 44, 0, 0, 0, 0, 0],
            &[0, 0, 160, 44, 0, 0, 0, 0, 0],
            &[0, 0, 160, 44, 0, 0, 0, 0, 0],
            &[0, 0, 160, 44, 0, 0, 0, 0, 0],
            &[0, 0, 160, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        _ => None,
    }
}
