//! Module for letters with the font weight light and size 14.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 14;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 8;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight light and font size 12px
#[inline]
pub const fn get_char(c: char) -> Option<&'static [&'static [u8]]> {
    match c {
        // letter: ' ' / 0x20
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '!' / 0x21
        '!' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 116, 2, 0, 0, 0],
            &[0, 0, 0, 201, 0, 0, 0, 0],
            &[0, 0, 0, 193, 0, 0, 0, 0],
            &[0, 0, 0, 185, 0, 0, 0, 0],
            &[0, 0, 0, 176, 0, 0, 0, 0],
            &[0, 0, 0, 167, 0, 0, 0, 0],
            &[0, 0, 0, 118, 0, 0, 0, 0],
            &[0, 0, 0, 7, 0, 0, 0, 0],
            &[0, 0, 0, 218, 19, 0, 0, 0],
            &[0, 0, 0, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '"' / 0x22
        '"' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 102, 0, 104, 0, 0, 0],
            &[0, 0, 161, 0, 164, 0, 0, 0],
            &[0, 0, 138, 0, 141, 0, 0, 0],
            &[0, 0, 63, 0, 64, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '#' / 0x23
        '#' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 50, 33, 4, 75, 0, 0],
            &[0, 0, 123, 24, 42, 99, 0, 0],
            &[0, 0, 149, 0, 86, 56, 0, 0],
            &[45, 138, 206, 138, 204, 139, 76, 0],
            &[0, 5, 141, 0, 147, 0, 0, 0],
            &[36, 81, 136, 49, 168, 49, 7, 0],
            &[67, 153, 124, 104, 174, 92, 14, 0],
            &[0, 130, 15, 55, 91, 0, 0, 0],
            &[0, 144, 0, 101, 46, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '$' / 0x24
        '$' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 0, 0, 0, 0],
            &[0, 0, 23, 166, 53, 5, 0, 0],
            &[0, 116, 149, 177, 109, 111, 0, 0],
            &[0, 182, 0, 135, 0, 0, 0, 0],
            &[0, 171, 44, 135, 0, 0, 0, 0],
            &[0, 12, 139, 212, 82, 3, 0, 0],
            &[0, 0, 0, 141, 99, 165, 0, 0],
            &[0, 0, 0, 135, 0, 154, 24, 0],
            &[0, 54, 9, 135, 38, 181, 2, 0],
            &[0, 92, 152, 209, 127, 22, 0, 0],
            &[0, 0, 0, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '%' / 0x25
        '%' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[48, 128, 97, 0, 0, 91, 0, 0],
            &[156, 1, 105, 52, 35, 127, 0, 0],
            &[151, 0, 65, 86, 145, 17, 0, 0],
            &[150, 42, 144, 69, 124, 0, 0, 0],
            &[12, 88, 43, 147, 15, 0, 0, 0],
            &[0, 0, 41, 121, 121, 126, 124, 0],
            &[0, 0, 149, 36, 130, 0, 123, 29],
            &[0, 45, 117, 28, 124, 0, 117, 35],
            &[0, 151, 11, 0, 142, 112, 150, 0],
            &[0, 0, 0, 0, 0, 15, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '&' / 0x26
        '&' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 130, 150, 36, 0, 0, 0],
            &[0, 103, 82, 18, 169, 0, 0, 0],
            &[0, 109, 57, 6, 173, 0, 0, 0],
            &[0, 26, 155, 148, 59, 0, 0, 0],
            &[0, 23, 255, 142, 0, 0, 0, 0],
            &[3, 174, 25, 154, 41, 54, 109, 0],
            &[53, 129, 0, 12, 174, 153, 33, 0],
            &[41, 144, 0, 0, 99, 255, 0, 0],
            &[0, 148, 152, 151, 145, 96, 120, 0],
            &[0, 0, 16, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ''' / 0x27
        '\'' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 104, 0, 0, 0, 0],
            &[0, 0, 0, 164, 0, 0, 0, 0],
            &[0, 0, 0, 141, 0, 0, 0, 0],
            &[0, 0, 0, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '(' / 0x28
        '(' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 97, 0, 0, 0],
            &[0, 0, 0, 107, 70, 0, 0, 0],
            &[0, 0, 2, 175, 0, 0, 0, 0],
            &[0, 0, 47, 130, 0, 0, 0, 0],
            &[0, 0, 95, 82, 0, 0, 0, 0],
            &[0, 0, 114, 63, 0, 0, 0, 0],
            &[0, 0, 104, 73, 0, 0, 0, 0],
            &[0, 0, 70, 109, 0, 0, 0, 0],
            &[0, 0, 12, 167, 0, 0, 0, 0],
            &[0, 0, 0, 148, 30, 0, 0, 0],
            &[0, 0, 0, 20, 136, 0, 0, 0],
        ]),
        // letter: ')' / 0x29
        ')' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 17, 0, 0, 0, 0],
            &[0, 0, 28, 149, 0, 0, 0, 0],
            &[0, 0, 0, 151, 28, 0, 0, 0],
            &[0, 0, 0, 77, 101, 0, 0, 0],
            &[0, 0, 0, 29, 148, 0, 0, 0],
            &[0, 0, 0, 12, 165, 0, 0, 0],
            &[0, 0, 0, 23, 154, 0, 0, 0],
            &[0, 0, 0, 59, 118, 0, 0, 0],
            &[0, 0, 0, 126, 51, 0, 0, 0],
            &[0, 0, 5, 172, 0, 0, 0, 0],
            &[0, 0, 106, 52, 0, 0, 0, 0],
        ]),
        // letter: '*' / 0x2a
        '*' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 0],
            &[0, 0, 0, 142, 8, 0, 0, 0],
            &[0, 55, 19, 127, 2, 58, 19, 0],
            &[0, 58, 123, 207, 155, 102, 17, 0],
            &[0, 0, 70, 105, 160, 4, 0, 0],
            &[0, 0, 128, 1, 72, 62, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '+' / 0x2b
        '+' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 0, 0, 0, 0],
            &[0, 0, 0, 147, 0, 0, 0, 0],
            &[0, 0, 0, 147, 0, 0, 0, 0],
            &[13, 134, 134, 204, 134, 134, 40, 0],
            &[1, 13, 13, 153, 13, 13, 3, 0],
            &[0, 0, 0, 147, 0, 0, 0, 0],
            &[0, 0, 0, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ',' / 0x2c
        ',' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 75, 31, 0, 0, 0],
            &[0, 0, 0, 189, 21, 0, 0, 0],
            &[0, 0, 3, 180, 0, 0, 0, 0],
            &[0, 0, 25, 71, 0, 0, 0, 0],
        ]),
        // letter: '-' / 0x2d
        '-' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 153, 153, 153, 61, 0, 0],
            &[0, 1, 6, 6, 6, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '.' / 0x2e
        '.' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 0, 0, 0, 0],
            &[0, 0, 0, 221, 19, 0, 0, 0],
            &[0, 0, 0, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '/' / 0x2f
        '/' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 66, 0, 0],
            &[0, 0, 0, 0, 133, 41, 0, 0],
            &[0, 0, 0, 2, 172, 0, 0, 0],
            &[0, 0, 0, 71, 103, 0, 0, 0],
            &[0, 0, 0, 159, 15, 0, 0, 0],
            &[0, 0, 17, 157, 0, 0, 0, 0],
            &[0, 0, 107, 67, 0, 0, 0, 0],
            &[0, 0, 173, 1, 0, 0, 0, 0],
            &[0, 45, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '0' / 0x30
        '0' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 108, 160, 131, 13, 0, 0],
            &[0, 116, 96, 0, 62, 207, 0, 0],
            &[0, 174, 0, 0, 132, 202, 19, 0],
            &[10, 158, 0, 36, 147, 104, 61, 0],
            &[24, 143, 0, 161, 20, 89, 76, 0],
            &[17, 155, 73, 107, 0, 101, 70, 0],
            &[1, 178, 170, 3, 0, 131, 39, 0],
            &[0, 210, 91, 0, 7, 181, 1, 0],
            &[0, 29, 181, 146, 183, 65, 0, 0],
            &[0, 0, 0, 21, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '1' / 0x31
        '1' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 96, 53, 0, 0, 0],
            &[0, 41, 168, 144, 94, 0, 0, 0],
            &[0, 35, 14, 76, 94, 0, 0, 0],
            &[0, 0, 0, 77, 94, 0, 0, 0],
            &[0, 0, 0, 77, 94, 0, 0, 0],
            &[0, 0, 0, 77, 94, 0, 0, 0],
            &[0, 0, 0, 77, 94, 0, 0, 0],
            &[0, 0, 0, 77, 94, 0, 0, 0],
            &[0, 0, 128, 219, 235, 129, 4, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '2' / 0x32
        '2' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 23, 126, 162, 134, 16, 0, 0],
            &[0, 108, 48, 1, 55, 169, 0, 0],
            &[0, 0, 0, 0, 0, 180, 0, 0],
            &[0, 0, 0, 0, 1, 181, 0, 0],
            &[0, 0, 0, 0, 90, 108, 0, 0],
            &[0, 0, 0, 46, 167, 3, 0, 0],
            &[0, 0, 40, 172, 11, 0, 0, 0],
            &[0, 36, 172, 13, 0, 0, 0, 0],
            &[0, 215, 177, 162, 162, 162, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '3' / 0x33
        '3' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 130, 163, 142, 27, 0, 0],
            &[0, 105, 43, 1, 40, 187, 0, 0],
            &[0, 0, 0, 0, 0, 177, 0, 0],
            &[0, 0, 0, 0, 46, 156, 0, 0],
            &[0, 0, 121, 189, 195, 29, 0, 0],
            &[0, 0, 0, 0, 24, 172, 14, 0],
            &[0, 0, 0, 0, 0, 117, 66, 0],
            &[0, 0, 0, 0, 0, 172, 31, 0],
            &[0, 163, 155, 138, 174, 102, 0, 0],
            &[0, 0, 9, 23, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '4' / 0x34
        '4' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 121, 36, 0, 0],
            &[0, 0, 0, 72, 197, 59, 0, 0],
            &[0, 0, 11, 164, 111, 59, 0, 0],
            &[0, 0, 135, 42, 112, 59, 0, 0],
            &[0, 51, 126, 0, 112, 59, 0, 0],
            &[4, 167, 7, 0, 112, 59, 0, 0],
            &[67, 185, 153, 153, 198, 177, 82, 0],
            &[0, 0, 0, 0, 112, 59, 0, 0],
            &[0, 0, 0, 0, 112, 59, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '5' / 0x35
        '5' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 145, 145, 145, 101, 0, 0],
            &[0, 105, 71, 14, 14, 10, 0, 0],
            &[0, 127, 38, 0, 0, 0, 0, 0],
            &[0, 148, 38, 41, 11, 0, 0, 0],
            &[0, 89, 122, 110, 171, 99, 0, 0],
            &[0, 0, 0, 0, 0, 187, 9, 0],
            &[0, 0, 0, 0, 0, 154, 29, 0],
            &[0, 0, 0, 0, 7, 192, 3, 0],
            &[0, 150, 151, 142, 183, 64, 0, 0],
            &[0, 0, 9, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '6' / 0x36
        '6' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 137, 153, 95, 0, 0],
            &[0, 38, 170, 26, 0, 2, 0, 0],
            &[0, 157, 20, 0, 0, 0, 0, 0],
            &[0, 169, 16, 74, 59, 1, 0, 0],
            &[16, 191, 139, 67, 100, 159, 0, 0],
            &[20, 204, 0, 0, 0, 124, 51, 0],
            &[2, 168, 0, 0, 0, 99, 74, 0],
            &[0, 159, 26, 0, 0, 158, 27, 0],
            &[0, 28, 176, 138, 163, 105, 0, 0],
            &[0, 0, 0, 22, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '7' / 0x37
        '7' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 126, 145, 145, 145, 145, 43, 0],
            &[0, 12, 14, 14, 14, 158, 25, 0],
            &[0, 0, 0, 0, 14, 162, 0, 0],
            &[0, 0, 0, 0, 107, 71, 0, 0],
            &[0, 0, 0, 0, 178, 2, 0, 0],
            &[0, 0, 0, 61, 122, 0, 0, 0],
            &[0, 0, 0, 160, 24, 0, 0, 0],
            &[0, 0, 21, 165, 0, 0, 0, 0],
            &[0, 0, 119, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '8' / 0x38
        '8' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 127, 156, 130, 15, 0, 0],
            &[0, 160, 52, 0, 49, 162, 0, 0],
            &[0, 177, 0, 0, 0, 180, 0, 0],
            &[0, 107, 114, 6, 114, 102, 0, 0],
            &[0, 2, 161, 215, 148, 1, 0, 0],
            &[0, 149, 70, 0, 84, 147, 0, 0],
            &[24, 164, 0, 0, 0, 159, 24, 0],
            &[14, 177, 0, 0, 0, 178, 11, 0],
            &[0, 99, 164, 128, 165, 90, 0, 0],
            &[0, 0, 5, 24, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '9' / 0x39
        '9' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 129, 155, 122, 8, 0, 0],
            &[0, 167, 49, 0, 63, 151, 0, 0],
            &[17, 159, 0, 0, 0, 153, 18, 0],
            &[19, 150, 0, 0, 0, 128, 61, 0],
            &[0, 175, 23, 0, 32, 198, 67, 0],
            &[0, 26, 141, 141, 116, 130, 46, 0],
            &[0, 0, 0, 0, 0, 166, 5, 0],
            &[0, 0, 0, 0, 60, 134, 0, 0],
            &[0, 76, 130, 154, 151, 7, 0, 0],
            &[0, 4, 26, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ':' / 0x3a
        ':' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 89, 4, 0, 0, 0],
            &[0, 0, 0, 147, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 0, 0, 0, 0],
            &[0, 0, 0, 221, 17, 0, 0, 0],
            &[0, 0, 0, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ';' / 0x3b
        ';' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 65, 22, 0, 0, 0],
            &[0, 0, 0, 110, 42, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 82, 23, 0, 0, 0],
            &[0, 0, 0, 199, 11, 0, 0, 0],
            &[0, 0, 10, 173, 0, 0, 0, 0],
            &[0, 0, 34, 61, 0, 0, 0, 0],
        ]),
        // letter: '<' / 0x3c
        '<' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 0],
            &[0, 0, 0, 0, 71, 155, 30, 0],
            &[0, 0, 64, 150, 84, 1, 0, 0],
            &[4, 180, 92, 1, 0, 0, 0, 0],
            &[0, 52, 148, 112, 11, 0, 0, 0],
            &[0, 0, 0, 43, 145, 123, 11, 0],
            &[0, 0, 0, 0, 0, 35, 24, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '=' / 0x3d
        '=' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[25, 159, 159, 159, 159, 159, 57, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[15, 95, 95, 95, 95, 95, 34, 0],
            &[10, 67, 67, 67, 67, 67, 24, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '>' / 0x3e
        '>' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[1, 5, 0, 0, 0, 0, 0, 0],
            &[3, 152, 97, 3, 0, 0, 0, 0],
            &[0, 0, 58, 149, 90, 2, 0, 0],
            &[0, 0, 0, 0, 60, 186, 31, 0],
            &[0, 0, 3, 89, 154, 76, 0, 0],
            &[1, 102, 154, 66, 0, 0, 0, 0],
            &[3, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '?' / 0x3f
        '?' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 53, 144, 158, 129, 14, 0, 0],
            &[0, 41, 20, 0, 56, 161, 0, 0],
            &[0, 0, 0, 0, 0, 173, 0, 0],
            &[0, 0, 0, 0, 43, 144, 0, 0],
            &[0, 0, 0, 60, 161, 12, 0, 0],
            &[0, 0, 11, 170, 3, 0, 0, 0],
            &[0, 0, 32, 87, 0, 0, 0, 0],
            &[0, 0, 0, 6, 0, 0, 0, 0],
            &[0, 0, 77, 161, 0, 0, 0, 0],
            &[0, 0, 1, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '@' / 0x40
        '@' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 68, 134, 117, 24, 0, 0],
            &[0, 132, 95, 10, 25, 152, 33, 0],
            &[56, 108, 0, 32, 30, 11, 141, 0],
            &[141, 10, 117, 122, 169, 42, 143, 0],
            &[149, 4, 166, 0, 124, 31, 141, 1],
            &[147, 20, 143, 0, 146, 19, 143, 0],
            &[148, 3, 164, 3, 189, 45, 143, 0],
            &[127, 29, 74, 130, 37, 133, 47, 0],
            &[24, 157, 3, 0, 0, 0, 0, 0],
            &[0, 51, 155, 135, 138, 137, 0, 0],
            &[0, 0, 0, 6, 3, 0, 0, 0],
        ]),
        // letter: 'A' / 0x41
        'A' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 133, 10, 0, 0, 0],
            &[0, 0, 37, 204, 88, 0, 0, 0],
            &[0, 0, 126, 55, 162, 0, 0, 0],
            &[0, 0, 175, 0, 155, 17, 0, 0],
            &[0, 48, 128, 0, 74, 100, 0, 0],
            &[0, 155, 187, 147, 153, 215, 0, 0],
            &[1, 182, 15, 15, 15, 159, 25, 0],
            &[60, 121, 0, 0, 0, 69, 111, 0],
            &[149, 34, 0, 0, 0, 3, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'B' / 0x42
        'B' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 145, 140, 113, 28, 0, 0],
            &[0, 172, 11, 16, 52, 189, 18, 0],
            &[0, 171, 0, 0, 0, 119, 62, 0],
            &[0, 171, 0, 0, 8, 170, 19, 0],
            &[0, 182, 156, 165, 215, 85, 0, 0],
            &[0, 171, 0, 0, 2, 142, 57, 0],
            &[0, 171, 0, 0, 0, 61, 121, 0],
            &[0, 171, 0, 0, 0, 130, 81, 0],
            &[0, 182, 159, 161, 173, 126, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'C' / 0x43
        'C' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 138, 160, 144, 47, 0],
            &[0, 52, 176, 32, 0, 23, 34, 0],
            &[0, 181, 16, 0, 0, 0, 0, 0],
            &[16, 170, 0, 0, 0, 0, 0, 0],
            &[41, 142, 0, 0, 0, 0, 0, 0],
            &[31, 151, 0, 0, 0, 0, 0, 0],
            &[2, 186, 0, 0, 0, 0, 0, 0],
            &[0, 136, 83, 0, 0, 0, 0, 0],
            &[0, 5, 147, 171, 142, 165, 57, 0],
            &[0, 0, 0, 5, 23, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'D' / 0x44
        'D' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 145, 145, 128, 62, 0, 0, 0],
            &[1, 180, 12, 34, 128, 140, 0, 0],
            &[1, 177, 0, 0, 0, 144, 52, 0],
            &[1, 177, 0, 0, 0, 58, 125, 0],
            &[1, 177, 0, 0, 0, 32, 147, 0],
            &[1, 177, 0, 0, 0, 44, 134, 0],
            &[1, 177, 0, 0, 0, 104, 80, 0],
            &[1, 177, 0, 0, 36, 183, 5, 0],
            &[1, 226, 160, 163, 153, 21, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'E' / 0x45
        'E' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 145, 145, 145, 145, 22, 0],
            &[0, 178, 14, 14, 14, 14, 2, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 159, 159, 159, 146, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 162, 162, 162, 162, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'F' / 0x46
        'F' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 145, 145, 145, 145, 22, 0],
            &[0, 178, 14, 14, 14, 14, 2, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 182, 106, 106, 106, 98, 0, 0],
            &[0, 180, 53, 53, 53, 49, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'G' / 0x47
        'G' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 71, 155, 161, 114, 4, 0],
            &[0, 107, 138, 12, 1, 53, 0, 0],
            &[10, 184, 0, 0, 0, 0, 0, 0],
            &[68, 117, 0, 0, 0, 0, 0, 0],
            &[94, 88, 0, 20, 94, 94, 20, 0],
            &[89, 93, 0, 15, 68, 156, 55, 0],
            &[50, 139, 0, 0, 0, 119, 55, 0],
            &[1, 188, 26, 0, 0, 119, 55, 0],
            &[0, 35, 178, 150, 145, 185, 36, 0],
            &[0, 0, 0, 15, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'H' / 0x48
        'H' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 0, 0, 0, 69, 31, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 227, 162, 162, 162, 207, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'I' / 0x49
        'I' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 48, 139, 145, 141, 75, 0, 0],
            &[0, 0, 0, 179, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 48, 139, 223, 142, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'J' / 0x4a
        'J' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 45, 0, 0],
            &[0, 0, 0, 0, 97, 80, 0, 0],
            &[0, 0, 0, 0, 97, 80, 0, 0],
            &[0, 0, 0, 0, 97, 80, 0, 0],
            &[0, 0, 0, 0, 97, 80, 0, 0],
            &[0, 0, 0, 0, 97, 80, 0, 0],
            &[0, 0, 0, 0, 104, 78, 0, 0],
            &[0, 0, 0, 0, 153, 47, 0, 0],
            &[0, 116, 146, 159, 161, 0, 0, 0],
            &[0, 0, 19, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'K' / 0x4b
        'K' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 0, 0, 0, 91, 31, 0],
            &[1, 177, 0, 0, 82, 137, 0, 0],
            &[1, 177, 0, 55, 166, 2, 0, 0],
            &[1, 177, 34, 183, 10, 0, 0, 0],
            &[1, 187, 184, 136, 0, 0, 0, 0],
            &[1, 228, 29, 147, 53, 0, 0, 0],
            &[1, 177, 0, 12, 181, 7, 0, 0],
            &[1, 177, 0, 0, 66, 135, 0, 0],
            &[1, 177, 0, 0, 0, 146, 55, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'L' / 0x4c
        'L' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 42, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 192, 165, 165, 165, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'M' / 0x4d
        'M' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[49, 110, 0, 0, 0, 82, 78, 0],
            &[87, 217, 16, 0, 0, 186, 138, 0],
            &[87, 147, 100, 0, 52, 144, 138, 0],
            &[87, 86, 163, 0, 140, 56, 138, 0],
            &[87, 84, 134, 32, 162, 30, 138, 0],
            &[87, 84, 44, 187, 94, 30, 138, 0],
            &[87, 84, 0, 155, 12, 30, 138, 0],
            &[87, 84, 0, 0, 0, 30, 138, 0],
            &[87, 84, 0, 0, 0, 30, 138, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'N' / 0x4e
        'N' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 5, 0, 0, 68, 31, 0],
            &[1, 219, 101, 0, 0, 119, 55, 0],
            &[1, 166, 165, 4, 0, 119, 55, 0],
            &[1, 170, 74, 96, 0, 119, 55, 0],
            &[1, 170, 0, 167, 3, 119, 55, 0],
            &[1, 170, 0, 79, 91, 119, 55, 0],
            &[1, 170, 0, 1, 168, 119, 55, 0],
            &[1, 170, 0, 0, 85, 196, 55, 0],
            &[1, 170, 0, 0, 2, 214, 55, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'O' / 0x4f
        'O' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 112, 159, 138, 21, 0, 0],
            &[0, 148, 79, 0, 38, 181, 3, 0],
            &[23, 161, 0, 0, 0, 116, 76, 0],
            &[77, 103, 0, 0, 0, 54, 131, 0],
            &[97, 83, 0, 0, 0, 33, 150, 0],
            &[90, 91, 0, 0, 0, 39, 141, 0],
            &[50, 133, 0, 0, 0, 79, 102, 0],
            &[1, 187, 13, 0, 0, 173, 24, 0],
            &[0, 49, 182, 142, 175, 86, 0, 0],
            &[0, 0, 1, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'P' / 0x50
        'P' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 145, 142, 117, 32, 0, 0],
            &[0, 178, 11, 14, 48, 185, 38, 0],
            &[0, 178, 0, 0, 0, 69, 115, 0],
            &[0, 178, 0, 0, 0, 76, 109, 0],
            &[0, 179, 26, 30, 69, 184, 22, 0],
            &[0, 183, 133, 129, 98, 16, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Q' / 0x51
        'Q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 112, 159, 138, 21, 0, 0],
            &[0, 148, 79, 0, 38, 181, 3, 0],
            &[23, 161, 0, 0, 0, 116, 76, 0],
            &[77, 103, 0, 0, 0, 54, 131, 0],
            &[97, 83, 0, 0, 0, 33, 150, 0],
            &[90, 91, 0, 0, 0, 39, 141, 0],
            &[50, 133, 0, 0, 0, 79, 102, 0],
            &[1, 187, 13, 0, 0, 173, 24, 0],
            &[0, 49, 182, 142, 255, 87, 0, 0],
            &[0, 0, 1, 23, 111, 88, 0, 0],
            &[0, 0, 0, 0, 0, 14, 5, 0],
        ]),
        // letter: 'R' / 0x52
        'R' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 145, 140, 106, 17, 0, 0],
            &[0, 178, 11, 19, 69, 195, 7, 0],
            &[0, 178, 0, 0, 0, 132, 57, 0],
            &[0, 178, 0, 0, 0, 148, 40, 0],
            &[0, 181, 75, 83, 130, 127, 0, 0],
            &[0, 181, 83, 88, 174, 0, 0, 0],
            &[0, 178, 0, 0, 116, 79, 0, 0],
            &[0, 178, 0, 0, 5, 182, 9, 0],
            &[0, 178, 0, 0, 0, 69, 129, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'S' / 0x53
        'S' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 102, 161, 157, 111, 4, 0],
            &[0, 132, 105, 5, 3, 54, 0, 0],
            &[0, 184, 0, 0, 0, 0, 0, 0],
            &[0, 140, 95, 0, 0, 0, 0, 0],
            &[0, 2, 118, 181, 96, 6, 0, 0],
            &[0, 0, 0, 5, 93, 181, 10, 0],
            &[0, 0, 0, 0, 0, 120, 72, 0],
            &[0, 0, 0, 0, 0, 151, 43, 0],
            &[0, 158, 151, 136, 173, 116, 0, 0],
            &[0, 0, 9, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'T' / 0x54
        'T' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[61, 145, 145, 145, 145, 145, 92, 0],
            &[6, 14, 14, 182, 14, 14, 9, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'U' / 0x55
        'U' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 0, 0, 0, 69, 31, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[0, 179, 0, 0, 0, 135, 44, 0],
            &[0, 179, 14, 0, 1, 185, 6, 0],
            &[0, 53, 184, 139, 175, 85, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'V' / 0x56
        'V' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[74, 32, 0, 0, 0, 4, 102, 0],
            &[63, 121, 0, 0, 0, 68, 118, 0],
            &[2, 180, 0, 0, 0, 150, 34, 0],
            &[0, 148, 32, 0, 1, 180, 0, 0],
            &[0, 63, 115, 0, 58, 120, 0, 0],
            &[0, 2, 173, 0, 140, 36, 0, 0],
            &[0, 0, 147, 26, 173, 0, 0, 0],
            &[0, 0, 63, 143, 122, 0, 0, 0],
            &[0, 0, 2, 224, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'W' / 0x57
        'W' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[82, 13, 0, 0, 0, 0, 95, 0],
            &[117, 50, 0, 0, 0, 2, 166, 0],
            &[82, 84, 0, 75, 0, 30, 138, 0],
            &[47, 118, 1, 212, 38, 65, 104, 0],
            &[13, 151, 43, 130, 98, 99, 70, 0],
            &[0, 164, 102, 37, 135, 132, 36, 0],
            &[0, 159, 138, 0, 137, 154, 5, 0],
            &[0, 149, 136, 0, 116, 166, 0, 0],
            &[0, 129, 116, 0, 58, 186, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'X' / 0x58
        'X' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[12, 98, 0, 0, 0, 61, 46, 0],
            &[0, 146, 42, 0, 7, 173, 4, 0],
            &[0, 16, 165, 0, 124, 57, 0, 0],
            &[0, 0, 93, 114, 145, 0, 0, 0],
            &[0, 0, 0, 223, 30, 0, 0, 0],
            &[0, 0, 80, 139, 134, 0, 0, 0],
            &[0, 12, 169, 1, 134, 45, 0, 0],
            &[0, 138, 49, 0, 12, 170, 1, 0],
            &[48, 143, 0, 0, 0, 88, 99, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Y' / 0x59
        'Y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[48, 61, 0, 0, 0, 29, 79, 0],
            &[8, 180, 1, 0, 0, 146, 39, 0],
            &[0, 102, 81, 0, 28, 150, 0, 0],
            &[0, 4, 171, 1, 141, 31, 0, 0],
            &[0, 0, 90, 107, 139, 0, 0, 0],
            &[0, 0, 2, 209, 24, 0, 0, 0],
            &[0, 0, 0, 171, 0, 0, 0, 0],
            &[0, 0, 0, 171, 0, 0, 0, 0],
            &[0, 0, 0, 171, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Z' / 0x5a
        'Z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 145, 145, 145, 134, 0, 0],
            &[0, 5, 11, 11, 38, 157, 0, 0],
            &[0, 0, 0, 0, 147, 35, 0, 0],
            &[0, 0, 0, 36, 147, 0, 0, 0],
            &[0, 0, 0, 157, 26, 0, 0, 0],
            &[0, 0, 47, 137, 0, 0, 0, 0],
            &[0, 0, 166, 19, 0, 0, 0, 0],
            &[0, 59, 126, 0, 0, 0, 0, 0],
            &[0, 175, 175, 162, 162, 162, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '[' / 0x5b
        '[' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 145, 145, 61, 0, 0],
            &[0, 0, 33, 139, 2, 0, 0, 0],
            &[0, 0, 33, 138, 0, 0, 0, 0],
            &[0, 0, 33, 138, 0, 0, 0, 0],
            &[0, 0, 33, 138, 0, 0, 0, 0],
            &[0, 0, 33, 138, 0, 0, 0, 0],
            &[0, 0, 33, 138, 0, 0, 0, 0],
            &[0, 0, 33, 138, 0, 0, 0, 0],
            &[0, 0, 33, 138, 0, 0, 0, 0],
            &[0, 0, 33, 138, 0, 0, 0, 0],
            &[0, 0, 30, 193, 150, 63, 0, 0],
        ]),
        // letter: '\' / 0x5c
        '\\' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 61, 0, 0, 0, 0, 0],
            &[0, 7, 167, 0, 0, 0, 0, 0],
            &[0, 0, 146, 28, 0, 0, 0, 0],
            &[0, 0, 52, 122, 0, 0, 0, 0],
            &[0, 0, 0, 174, 0, 0, 0, 0],
            &[0, 0, 0, 113, 61, 0, 0, 0],
            &[0, 0, 0, 21, 153, 0, 0, 0],
            &[0, 0, 0, 0, 163, 11, 0, 0],
            &[0, 0, 0, 0, 78, 96, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ']' / 0x5d
        ']' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 31, 150, 153, 55, 0, 0],
            &[0, 0, 0, 0, 83, 88, 0, 0],
            &[0, 0, 0, 0, 83, 88, 0, 0],
            &[0, 0, 0, 0, 83, 88, 0, 0],
            &[0, 0, 0, 0, 83, 88, 0, 0],
            &[0, 0, 0, 0, 83, 88, 0, 0],
            &[0, 0, 0, 0, 83, 88, 0, 0],
            &[0, 0, 0, 0, 83, 88, 0, 0],
            &[0, 0, 0, 0, 83, 88, 0, 0],
            &[0, 0, 0, 0, 83, 88, 0, 0],
            &[0, 0, 31, 147, 169, 73, 0, 0],
        ]),
        // letter: '^' / 0x5e
        '^' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 116, 3, 0, 0, 0],
            &[0, 0, 51, 158, 97, 0, 0, 0],
            &[0, 0, 149, 6, 152, 5, 0, 0],
            &[0, 53, 104, 0, 57, 102, 0, 0],
            &[0, 153, 6, 0, 0, 155, 7, 0],
            &[27, 89, 0, 0, 0, 53, 65, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '_' / 0x5f
        '_' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[13, 13, 13, 13, 13, 13, 13, 2],
            &[149, 149, 149, 149, 149, 149, 149, 29],
        ]),
        // letter: '`' / 0x60
        '`' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 13, 0, 0, 0, 0],
            &[0, 0, 43, 161, 1, 0, 0, 0],
            &[0, 0, 0, 49, 70, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'a' / 0x61
        'a' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 84, 120, 93, 7, 0],
            &[0, 0, 54, 72, 30, 74, 163, 0],
            &[0, 0, 0, 0, 0, 0, 167, 0],
            &[0, 0, 18, 120, 146, 136, 213, 0],
            &[0, 0, 183, 46, 0, 0, 171, 0],
            &[0, 2, 174, 0, 0, 13, 228, 0],
            &[0, 0, 136, 148, 131, 137, 164, 0],
            &[0, 0, 0, 13, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'b' / 0x62
        'b' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 0, 0, 0, 0, 0, 0],
            &[0, 171, 0, 0, 0, 0, 0, 0],
            &[0, 171, 0, 0, 0, 0, 0, 0],
            &[0, 171, 27, 113, 98, 12, 0, 0],
            &[0, 187, 143, 39, 67, 180, 2, 0],
            &[0, 215, 13, 0, 0, 116, 70, 0],
            &[0, 182, 0, 0, 0, 75, 106, 0],
            &[0, 189, 0, 0, 0, 87, 93, 0],
            &[0, 223, 35, 0, 0, 159, 33, 0],
            &[0, 159, 147, 135, 166, 110, 0, 0],
            &[0, 0, 0, 19, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'c' / 0x63
        'c' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 102, 121, 79, 0, 0],
            &[0, 50, 180, 63, 38, 76, 0, 0],
            &[0, 174, 18, 0, 0, 0, 0, 0],
            &[0, 180, 0, 0, 0, 0, 0, 0],
            &[0, 183, 0, 0, 0, 0, 0, 0],
            &[0, 155, 49, 0, 0, 0, 0, 0],
            &[0, 20, 168, 150, 135, 154, 0, 0],
            &[0, 0, 0, 13, 21, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'd' / 0x64
        'd' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 2, 0],
            &[0, 0, 0, 0, 0, 153, 24, 0],
            &[0, 0, 0, 0, 0, 153, 24, 0],
            &[0, 3, 84, 118, 55, 152, 24, 0],
            &[0, 152, 105, 32, 104, 207, 24, 0],
            &[19, 174, 0, 0, 0, 198, 24, 0],
            &[55, 130, 0, 0, 0, 158, 24, 0],
            &[42, 135, 0, 0, 0, 168, 24, 0],
            &[7, 181, 1, 0, 7, 227, 24, 0],
            &[0, 79, 173, 132, 152, 151, 24, 0],
            &[0, 0, 2, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'e' / 0x65
        'e' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 118, 99, 12, 0, 0],
            &[0, 102, 138, 36, 61, 174, 4, 0],
            &[5, 182, 0, 0, 0, 119, 66, 0],
            &[37, 208, 147, 147, 147, 175, 87, 0],
            &[30, 151, 0, 0, 0, 0, 0, 0],
            &[0, 183, 13, 0, 0, 0, 0, 0],
            &[0, 42, 170, 139, 135, 158, 25, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'f' / 0x66
        'f' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 40, 13, 0],
            &[0, 0, 0, 119, 153, 110, 94, 0],
            &[0, 0, 0, 186, 0, 0, 0, 0],
            &[2, 79, 91, 207, 98, 98, 36, 0],
            &[2, 55, 55, 194, 55, 55, 20, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'g' / 0x67
        'g' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 77, 120, 61, 48, 9, 0],
            &[0, 137, 103, 31, 93, 209, 24, 0],
            &[15, 168, 0, 0, 0, 179, 24, 0],
            &[50, 128, 0, 0, 0, 144, 24, 0],
            &[42, 136, 0, 0, 0, 153, 24, 0],
            &[7, 184, 2, 0, 4, 216, 24, 0],
            &[0, 75, 175, 132, 157, 167, 24, 0],
            &[0, 0, 1, 21, 1, 153, 18, 0],
            &[0, 138, 153, 154, 167, 52, 0, 0],
        ]),
        // letter: 'h' / 0x68
        'h' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 177, 23, 108, 106, 16, 0, 0],
            &[0, 189, 130, 35, 64, 189, 1, 0],
            &[0, 211, 6, 0, 0, 153, 28, 0],
            &[0, 180, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'i' / 0x69
        'i' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 169, 50, 0, 0, 0],
            &[0, 0, 0, 28, 4, 0, 0, 0],
            &[0, 70, 98, 98, 16, 0, 0, 0],
            &[0, 23, 43, 160, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 137, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'j' / 0x6a
        'j' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 88, 132, 0, 0],
            &[0, 0, 0, 0, 11, 22, 0, 0],
            &[0, 0, 34, 98, 98, 51, 0, 0],
            &[0, 0, 10, 38, 85, 134, 0, 0],
            &[0, 0, 0, 0, 44, 134, 0, 0],
            &[0, 0, 0, 0, 44, 134, 0, 0],
            &[0, 0, 0, 0, 44, 134, 0, 0],
            &[0, 0, 0, 0, 44, 134, 0, 0],
            &[0, 0, 0, 0, 44, 134, 0, 0],
            &[0, 0, 0, 0, 45, 133, 0, 0],
            &[0, 25, 154, 159, 159, 9, 0, 0],
        ]),
        // letter: 'k' / 0x6b
        'k' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 5, 0, 0, 0, 0, 0],
            &[0, 134, 44, 0, 0, 0, 0, 0],
            &[0, 134, 44, 0, 0, 0, 0, 0],
            &[0, 134, 44, 0, 0, 53, 30, 0],
            &[0, 134, 44, 0, 61, 155, 2, 0],
            &[0, 134, 43, 53, 163, 4, 0, 0],
            &[0, 134, 83, 217, 51, 0, 0, 0],
            &[0, 134, 153, 18, 177, 9, 0, 0],
            &[0, 134, 44, 0, 55, 148, 0, 0],
            &[0, 134, 44, 0, 0, 122, 84, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'l' / 0x6c
        'l' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 23, 30, 30, 5, 0, 0, 0],
            &[0, 75, 110, 192, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 137, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'm' / 0x6d
        'm' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[27, 37, 114, 40, 65, 102, 2, 0],
            &[71, 179, 58, 189, 120, 116, 87, 0],
            &[71, 116, 0, 184, 12, 37, 126, 0],
            &[71, 93, 0, 164, 0, 30, 132, 0],
            &[71, 91, 0, 162, 0, 30, 132, 0],
            &[71, 91, 0, 162, 0, 30, 132, 0],
            &[71, 91, 0, 162, 0, 30, 132, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'n' / 0x6e
        'n' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 25, 108, 108, 18, 0, 0],
            &[0, 183, 133, 37, 60, 190, 1, 0],
            &[0, 212, 7, 0, 0, 150, 29, 0],
            &[0, 182, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'o' / 0x6f
        'o' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 120, 90, 6, 0, 0],
            &[0, 132, 113, 33, 74, 169, 1, 0],
            &[17, 172, 0, 0, 0, 119, 66, 0],
            &[54, 126, 0, 0, 0, 70, 109, 0],
            &[46, 133, 0, 0, 0, 82, 99, 0],
            &[4, 184, 5, 0, 0, 158, 29, 0],
            &[0, 61, 171, 128, 162, 100, 0, 0],
            &[0, 0, 1, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'p' / 0x70
        'p' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 23, 108, 100, 14, 0, 0],
            &[0, 178, 135, 39, 62, 179, 3, 0],
            &[0, 214, 10, 0, 0, 111, 72, 0],
            &[0, 182, 0, 0, 0, 75, 107, 0],
            &[0, 193, 0, 0, 0, 87, 100, 0],
            &[0, 221, 40, 0, 0, 157, 39, 0],
            &[0, 173, 135, 135, 162, 121, 0, 0],
            &[0, 175, 0, 13, 10, 0, 0, 0],
            &[0, 154, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'q' / 0x71
        'q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 84, 121, 55, 47, 9, 0],
            &[0, 136, 99, 34, 109, 204, 24, 0],
            &[16, 167, 0, 0, 0, 200, 24, 0],
            &[50, 128, 0, 0, 0, 160, 24, 0],
            &[42, 135, 0, 0, 0, 170, 24, 0],
            &[7, 182, 1, 0, 9, 228, 24, 0],
            &[0, 78, 173, 132, 152, 166, 24, 0],
            &[0, 0, 2, 20, 0, 153, 24, 0],
            &[0, 0, 0, 0, 0, 135, 21, 0],
        ]),
        // letter: 'r' / 0x72
        'r' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 80, 98, 58, 40, 117, 80, 0],
            &[0, 27, 47, 191, 139, 43, 45, 0],
            &[0, 0, 0, 230, 4, 0, 0, 0],
            &[0, 0, 0, 183, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[3, 133, 146, 224, 143, 109, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 's' / 0x73
        's' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 115, 111, 42, 0, 0],
            &[0, 78, 139, 32, 44, 56, 0, 0],
            &[0, 97, 92, 0, 0, 0, 0, 0],
            &[0, 6, 132, 159, 70, 0, 0, 0],
            &[0, 0, 0, 10, 111, 135, 0, 0],
            &[0, 0, 0, 0, 0, 185, 0, 0],
            &[0, 108, 152, 129, 166, 90, 0, 0],
            &[0, 0, 7, 25, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 't' / 0x74
        't' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 0],
            &[0, 0, 42, 113, 0, 0, 0, 0],
            &[11, 84, 136, 168, 98, 98, 0, 0],
            &[8, 55, 105, 144, 55, 55, 0, 0],
            &[0, 0, 64, 113, 0, 0, 0, 0],
            &[0, 0, 64, 113, 0, 0, 0, 0],
            &[0, 0, 64, 114, 0, 0, 0, 0],
            &[0, 0, 47, 143, 0, 0, 0, 0],
            &[0, 0, 1, 171, 143, 135, 5, 0],
            &[0, 0, 0, 0, 20, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'u' / 0x75
        'u' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 0, 0, 0, 58, 9, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 162, 24, 0],
            &[0, 185, 3, 0, 6, 221, 24, 0],
            &[0, 85, 175, 128, 144, 147, 24, 0],
            &[0, 0, 5, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'v' / 0x76
        'v' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[14, 56, 0, 0, 0, 37, 34, 0],
            &[1, 179, 0, 0, 0, 158, 25, 0],
            &[0, 130, 49, 0, 11, 167, 0, 0],
            &[0, 34, 141, 0, 92, 81, 0, 0],
            &[0, 0, 169, 4, 164, 5, 0, 0],
            &[0, 0, 95, 101, 139, 0, 0, 0],
            &[0, 0, 11, 230, 44, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'w' / 0x77
        'w' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[63, 0, 0, 0, 0, 0, 62, 0],
            &[144, 22, 0, 40, 2, 0, 163, 0],
            &[92, 75, 4, 218, 56, 14, 149, 0],
            &[40, 128, 64, 115, 131, 64, 101, 0],
            &[1, 166, 139, 20, 159, 114, 51, 0],
            &[0, 156, 152, 0, 130, 171, 6, 0],
            &[0, 133, 119, 0, 55, 199, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'x' / 0x78
        'x' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[1, 76, 0, 0, 0, 62, 14, 0],
            &[0, 113, 85, 0, 43, 153, 0, 0],
            &[0, 1, 163, 38, 173, 12, 0, 0],
            &[0, 0, 20, 230, 55, 0, 0, 0],
            &[0, 0, 95, 143, 140, 0, 0, 0],
            &[0, 40, 157, 0, 117, 79, 0, 0],
            &[8, 176, 15, 0, 1, 168, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'y' / 0x79
        'y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[10, 57, 0, 0, 0, 36, 34, 0],
            &[0, 174, 0, 0, 0, 154, 25, 0],
            &[0, 111, 62, 0, 8, 166, 0, 0],
            &[0, 18, 153, 0, 85, 85, 0, 0],
            &[0, 0, 153, 14, 159, 7, 0, 0],
            &[0, 0, 62, 118, 142, 0, 0, 0],
            &[0, 0, 0, 206, 51, 0, 0, 0],
            &[0, 0, 0, 170, 0, 0, 0, 0],
            &[14, 148, 151, 7, 0, 0, 0, 0],
        ]),
        // letter: 'z' / 0x7a
        'z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 98, 98, 98, 91, 0, 0],
            &[0, 23, 46, 46, 87, 164, 0, 0],
            &[0, 0, 0, 7, 175, 15, 0, 0],
            &[0, 0, 0, 140, 55, 0, 0, 0],
            &[0, 0, 75, 118, 0, 0, 0, 0],
            &[0, 25, 165, 1, 0, 0, 0, 0],
            &[0, 169, 179, 156, 156, 149, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '{' / 0x7b
        '{' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 106, 54, 0, 0],
            &[0, 0, 0, 158, 66, 4, 0, 0],
            &[0, 0, 0, 176, 0, 0, 0, 0],
            &[0, 0, 0, 175, 0, 0, 0, 0],
            &[0, 0, 0, 180, 0, 0, 0, 0],
            &[0, 25, 191, 84, 0, 0, 0, 0],
            &[0, 0, 24, 175, 0, 0, 0, 0],
            &[0, 0, 0, 175, 0, 0, 0, 0],
            &[0, 0, 0, 175, 0, 0, 0, 0],
            &[0, 0, 0, 166, 10, 0, 0, 0],
            &[0, 0, 0, 46, 154, 58, 0, 0],
        ]),
        // letter: '|' / 0x7c
        '|' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 132, 0, 0, 0, 0],
        ]),
        // letter: '}' / 0x7d
        '}' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 115, 22, 0, 0, 0, 0],
            &[0, 1, 36, 174, 0, 0, 0, 0],
            &[0, 0, 0, 175, 0, 0, 0, 0],
            &[0, 0, 0, 175, 0, 0, 0, 0],
            &[0, 0, 0, 167, 14, 0, 0, 0],
            &[0, 0, 0, 48, 192, 58, 0, 0],
            &[0, 0, 0, 141, 59, 0, 0, 0],
            &[0, 0, 0, 175, 0, 0, 0, 0],
            &[0, 0, 0, 175, 0, 0, 0, 0],
            &[0, 0, 1, 188, 0, 0, 0, 0],
            &[0, 29, 157, 79, 0, 0, 0, 0],
        ]),
        // letter: '~' / 0x7e
        '~' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 51, 125, 26, 0, 53, 27, 0],
            &[0, 147, 12, 147, 61, 145, 9, 0],
            &[2, 58, 0, 3, 86, 46, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ' ' / 0xa0
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¡' / 0xa1
        '¡' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 86, 3, 0, 0, 0],
            &[0, 0, 0, 154, 12, 0, 0, 0],
            &[0, 0, 0, 14, 0, 0, 0, 0],
            &[0, 0, 0, 161, 0, 0, 0, 0],
            &[0, 0, 0, 170, 0, 0, 0, 0],
            &[0, 0, 0, 179, 0, 0, 0, 0],
            &[0, 0, 0, 188, 0, 0, 0, 0],
            &[0, 0, 0, 196, 0, 0, 0, 0],
            &[0, 0, 0, 46, 0, 0, 0, 0],
        ]),
        // letter: '¢' / 0xa2
        '¢' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 51, 47, 0, 0, 0],
            &[0, 0, 58, 173, 176, 116, 0, 0],
            &[0, 69, 156, 12, 0, 19, 0, 0],
            &[0, 174, 10, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 181, 0, 0, 0, 0, 0, 0],
            &[0, 122, 92, 0, 0, 0, 0, 0],
            &[0, 4, 133, 169, 163, 135, 0, 0],
            &[0, 0, 0, 74, 66, 0, 0, 0],
            &[0, 0, 0, 8, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '£' / 0xa3
        '£' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 102, 153, 130, 17, 0],
            &[0, 0, 89, 111, 0, 31, 11, 0],
            &[0, 0, 147, 26, 0, 0, 0, 0],
            &[0, 0, 156, 18, 0, 0, 0, 0],
            &[0, 95, 203, 131, 121, 25, 0, 0],
            &[0, 12, 162, 33, 16, 3, 0, 0],
            &[0, 0, 160, 15, 0, 0, 0, 0],
            &[0, 2, 177, 0, 0, 0, 0, 0],
            &[0, 184, 202, 162, 162, 162, 62, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¤' / 0xa4
        '¤' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[6, 157, 99, 147, 119, 138, 33, 0],
            &[0, 120, 79, 0, 40, 155, 0, 0],
            &[0, 148, 0, 0, 0, 150, 0, 0],
            &[0, 142, 15, 0, 2, 160, 0, 0],
            &[0, 145, 165, 131, 162, 172, 10, 0],
            &[3, 48, 0, 18, 1, 34, 20, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¥' / 0xa5
        '¥' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[55, 51, 0, 0, 0, 21, 86, 0],
            &[11, 175, 0, 0, 0, 144, 42, 0],
            &[0, 98, 86, 0, 38, 143, 0, 0],
            &[0, 2, 173, 6, 158, 19, 0, 0],
            &[0, 0, 66, 168, 112, 0, 0, 0],
            &[0, 88, 135, 218, 136, 117, 0, 0],
            &[0, 47, 71, 195, 71, 62, 0, 0],
            &[0, 41, 63, 192, 63, 55, 0, 0],
            &[0, 0, 0, 171, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¦' / 0xa6
        '¦' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 82, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 150, 0, 0, 0, 0],
            &[0, 0, 0, 143, 0, 0, 0, 0],
        ]),
        // letter: '§' / 0xa7
        '§' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 27, 19, 0, 0, 0],
            &[0, 27, 162, 106, 118, 94, 0, 0],
            &[0, 86, 90, 0, 0, 0, 0, 0],
            &[0, 9, 170, 143, 41, 0, 0, 0],
            &[0, 53, 120, 25, 121, 111, 0, 0],
            &[0, 91, 66, 0, 0, 162, 0, 0],
            &[0, 10, 146, 128, 101, 104, 0, 0],
            &[0, 0, 0, 27, 136, 101, 0, 0],
            &[0, 0, 0, 0, 0, 167, 0, 0],
            &[0, 90, 141, 125, 149, 78, 0, 0],
            &[0, 0, 0, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¨' / 0xa8
        '¨' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 125, 0, 90, 48, 0, 0],
            &[0, 4, 61, 0, 44, 21, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '©' / 0xa9
        '©' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 95, 68, 3, 0, 0],
            &[0, 118, 69, 52, 72, 137, 10, 0],
            &[66, 59, 120, 77, 64, 17, 108, 0],
            &[114, 1, 125, 0, 0, 0, 113, 0],
            &[96, 25, 136, 1, 0, 0, 115, 0],
            &[15, 133, 40, 110, 74, 102, 41, 0],
            &[0, 28, 115, 107, 111, 46, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ª' / 0xaa
        'ª' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 85, 117, 101, 0, 0, 0],
            &[0, 0, 3, 2, 132, 20, 0, 0],
            &[0, 0, 131, 97, 158, 28, 0, 0],
            &[0, 2, 164, 38, 168, 28, 0, 0],
            &[0, 0, 39, 70, 30, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '«' / 0xab
        '«' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 72, 4, 87, 0, 0],
            &[0, 4, 164, 23, 139, 50, 0, 0],
            &[0, 136, 62, 88, 109, 0, 0, 0],
            &[0, 59, 135, 29, 163, 2, 0, 0],
            &[0, 0, 97, 80, 57, 122, 0, 0],
            &[0, 0, 0, 9, 0, 11, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¬' / 0xac
        '¬' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[29, 134, 134, 134, 134, 134, 11, 0],
            &[2, 13, 13, 13, 13, 132, 21, 0],
            &[0, 0, 0, 0, 0, 125, 21, 0],
            &[0, 0, 0, 0, 0, 55, 9, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '®' / 0xae
        '®' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 95, 68, 3, 0, 0],
            &[0, 118, 105, 89, 74, 137, 10, 0],
            &[66, 59, 111, 32, 144, 17, 108, 0],
            &[114, 1, 114, 97, 121, 0, 113, 0],
            &[96, 25, 110, 20, 113, 0, 115, 0],
            &[15, 133, 71, 0, 71, 112, 41, 0],
            &[0, 28, 115, 107, 111, 46, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¯' / 0xaf
        '¯' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 22, 22, 22, 5, 0, 0],
            &[0, 6, 124, 124, 124, 31, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '°' / 0xb0
        '°' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 64, 121, 87, 0, 0, 0],
            &[0, 8, 130, 0, 86, 49, 0, 0],
            &[0, 14, 121, 0, 68, 62, 0, 0],
            &[0, 0, 89, 122, 113, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '±' / 0xb1
        '±' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 0, 0, 0, 0],
            &[0, 0, 0, 147, 0, 0, 0, 0],
            &[0, 0, 0, 147, 0, 0, 0, 0],
            &[17, 137, 137, 205, 137, 137, 41, 0],
            &[0, 7, 7, 150, 7, 7, 2, 0],
            &[0, 0, 0, 147, 0, 0, 0, 0],
            &[0, 0, 0, 75, 0, 0, 0, 0],
            &[14, 147, 147, 147, 147, 147, 44, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '²' / 0xb2
        '²' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 52, 17, 0, 0, 0],
            &[0, 0, 112, 78, 171, 15, 0, 0],
            &[0, 0, 0, 0, 122, 27, 0, 0],
            &[0, 0, 0, 43, 124, 0, 0, 0],
            &[0, 0, 42, 130, 1, 0, 0, 0],
            &[0, 16, 214, 134, 133, 46, 0, 0],
            &[0, 0, 2, 2, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '³' / 0xb3
        '³' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 47, 13, 0, 0, 0],
            &[0, 1, 118, 83, 175, 9, 0, 0],
            &[0, 0, 0, 0, 152, 9, 0, 0],
            &[0, 0, 48, 133, 146, 4, 0, 0],
            &[0, 0, 0, 0, 92, 54, 0, 0],
            &[0, 17, 131, 114, 156, 11, 0, 0],
            &[0, 0, 2, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '´' / 0xb4
        '´' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 15, 0, 0, 0],
            &[0, 0, 19, 174, 10, 0, 0, 0],
            &[0, 0, 102, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'µ' / 0xb5
        'µ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 0, 0, 0, 58, 9, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 255, 0, 0, 0, 162, 24, 0],
            &[0, 255, 3, 0, 6, 221, 24, 0],
            &[0, 255, 175, 128, 144, 147, 24, 0],
            &[0, 178, 5, 17, 0, 0, 0, 0],
            &[0, 156, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¶' / 0xb6
        '¶' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 30, 30, 30, 4, 0],
            &[0, 141, 254, 255, 111, 153, 33, 0],
            &[24, 255, 255, 255, 26, 92, 33, 0],
            &[54, 255, 255, 255, 26, 92, 33, 0],
            &[35, 255, 255, 255, 26, 92, 33, 0],
            &[0, 194, 255, 255, 26, 92, 33, 0],
            &[0, 9, 75, 144, 26, 92, 33, 0],
            &[0, 0, 0, 99, 26, 92, 33, 0],
            &[0, 0, 0, 99, 26, 92, 33, 0],
            &[0, 0, 0, 99, 26, 92, 33, 0],
            &[0, 0, 0, 99, 26, 92, 33, 0],
            &[0, 0, 0, 52, 13, 48, 17, 0],
        ]),
        // letter: '·' / 0xb7
        '·' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 90, 5, 0, 0, 0],
            &[0, 0, 0, 150, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¸' / 0xb8
        '¸' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 0],
            &[0, 0, 0, 151, 13, 0, 0, 0],
            &[0, 0, 47, 130, 96, 0, 0, 0],
        ]),
        // letter: '¹' / 0xb9
        '¹' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 20, 1, 0, 0, 0],
            &[0, 3, 133, 201, 8, 0, 0, 0],
            &[0, 0, 15, 132, 8, 0, 0, 0],
            &[0, 0, 0, 133, 8, 0, 0, 0],
            &[0, 0, 0, 133, 8, 0, 0, 0],
            &[0, 0, 86, 255, 124, 30, 0, 0],
            &[0, 0, 1, 3, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'º' / 0xba
        'º' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 119, 81, 0, 0, 0],
            &[0, 0, 144, 0, 116, 27, 0, 0],
            &[0, 5, 130, 0, 80, 58, 0, 0],
            &[0, 0, 144, 46, 147, 12, 0, 0],
            &[0, 0, 11, 72, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '»' / 0xbb
        '»' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 14, 30, 37, 0, 0, 0],
            &[0, 26, 158, 10, 163, 15, 0, 0],
            &[0, 0, 64, 134, 31, 166, 0, 0],
            &[0, 0, 107, 88, 65, 132, 0, 0],
            &[0, 57, 129, 28, 159, 2, 0, 0],
            &[0, 25, 2, 13, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¼' / 0xbc
        '¼' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 128, 0, 0, 0, 86, 6, 0],
            &[89, 156, 0, 0, 21, 141, 0, 0],
            &[0, 141, 0, 0, 133, 29, 0, 0],
            &[0, 141, 0, 26, 136, 0, 0, 0],
            &[63, 208, 113, 139, 23, 22, 8, 0],
            &[0, 0, 32, 129, 10, 193, 46, 0],
            &[0, 0, 144, 18, 125, 110, 46, 0],
            &[0, 39, 122, 54, 177, 170, 128, 0],
            &[0, 149, 13, 0, 0, 92, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '½' / 0xbd
        '½' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 128, 0, 0, 0, 86, 6, 0],
            &[89, 156, 0, 0, 21, 141, 0, 0],
            &[0, 141, 0, 0, 133, 29, 0, 0],
            &[0, 141, 0, 26, 136, 0, 0, 0],
            &[63, 208, 113, 139, 35, 56, 8, 0],
            &[0, 0, 32, 129, 85, 77, 151, 0],
            &[0, 0, 144, 18, 0, 20, 134, 0],
            &[0, 39, 122, 0, 17, 145, 12, 0],
            &[0, 149, 13, 0, 193, 147, 119, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¾' / 0xbe
        '¾' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[68, 129, 97, 0, 0, 86, 6, 0],
            &[9, 0, 161, 0, 21, 141, 0, 0],
            &[17, 142, 155, 0, 133, 29, 0, 0],
            &[5, 0, 160, 32, 136, 0, 0, 0],
            &[88, 132, 91, 139, 23, 22, 8, 0],
            &[0, 0, 32, 129, 10, 193, 46, 0],
            &[0, 0, 144, 18, 125, 110, 46, 0],
            &[0, 39, 122, 54, 177, 170, 128, 0],
            &[0, 149, 13, 0, 0, 92, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¿' / 0xbf
        '¿' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 17, 0, 0, 0],
            &[0, 0, 0, 128, 37, 0, 0, 0],
            &[0, 0, 0, 12, 2, 0, 0, 0],
            &[0, 0, 0, 148, 15, 0, 0, 0],
            &[0, 0, 48, 159, 0, 0, 0, 0],
            &[0, 63, 152, 7, 0, 0, 0, 0],
            &[0, 174, 4, 0, 0, 0, 0, 0],
            &[0, 177, 0, 0, 0, 0, 0, 0],
            &[0, 0, 47, 80, 50, 1, 0, 0],
        ]),
        // letter: 'À' / 0xc0
        'À' => Some(&[
            &[0, 0, 75, 7, 0, 0, 0, 0],
            &[0, 0, 63, 134, 0, 0, 0, 0],
            &[0, 0, 0, 60, 26, 0, 0, 0],
            &[0, 0, 0, 133, 10, 0, 0, 0],
            &[0, 0, 37, 204, 88, 0, 0, 0],
            &[0, 0, 126, 55, 162, 0, 0, 0],
            &[0, 0, 175, 0, 155, 17, 0, 0],
            &[0, 48, 128, 0, 74, 100, 0, 0],
            &[0, 155, 187, 147, 153, 215, 0, 0],
            &[1, 182, 15, 15, 15, 159, 25, 0],
            &[60, 121, 0, 0, 0, 69, 111, 0],
            &[149, 34, 0, 0, 0, 3, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Á' / 0xc1
        'Á' => Some(&[
            &[0, 0, 0, 0, 61, 21, 0, 0],
            &[0, 0, 0, 45, 151, 0, 0, 0],
            &[0, 0, 0, 84, 3, 0, 0, 0],
            &[0, 0, 0, 133, 10, 0, 0, 0],
            &[0, 0, 37, 204, 88, 0, 0, 0],
            &[0, 0, 126, 55, 162, 0, 0, 0],
            &[0, 0, 175, 0, 155, 17, 0, 0],
            &[0, 48, 128, 0, 74, 100, 0, 0],
            &[0, 155, 187, 147, 153, 215, 0, 0],
            &[1, 182, 15, 15, 15, 159, 25, 0],
            &[60, 121, 0, 0, 0, 69, 111, 0],
            &[149, 34, 0, 0, 0, 3, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Â' / 0xc2
        'Â' => Some(&[
            &[0, 0, 0, 78, 7, 0, 0, 0],
            &[0, 0, 91, 128, 134, 0, 0, 0],
            &[0, 8, 76, 0, 55, 30, 0, 0],
            &[0, 0, 0, 133, 10, 0, 0, 0],
            &[0, 0, 37, 204, 88, 0, 0, 0],
            &[0, 0, 126, 55, 162, 0, 0, 0],
            &[0, 0, 175, 0, 155, 17, 0, 0],
            &[0, 48, 128, 0, 74, 100, 0, 0],
            &[0, 155, 187, 147, 153, 215, 0, 0],
            &[1, 182, 15, 15, 15, 159, 25, 0],
            &[60, 121, 0, 0, 0, 69, 111, 0],
            &[149, 34, 0, 0, 0, 3, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ã' / 0xc3
        'Ã' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 156, 135, 45, 111, 0, 0],
            &[0, 48, 12, 28, 106, 16, 0, 0],
            &[0, 0, 0, 133, 10, 0, 0, 0],
            &[0, 0, 37, 204, 88, 0, 0, 0],
            &[0, 0, 126, 55, 162, 0, 0, 0],
            &[0, 0, 175, 0, 155, 17, 0, 0],
            &[0, 48, 128, 0, 74, 100, 0, 0],
            &[0, 155, 187, 147, 153, 215, 0, 0],
            &[1, 182, 15, 15, 15, 159, 25, 0],
            &[60, 121, 0, 0, 0, 69, 111, 0],
            &[149, 34, 0, 0, 0, 3, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ä' / 0xc4
        'Ä' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 160, 0, 115, 61, 0, 0],
            &[0, 0, 26, 0, 19, 7, 0, 0],
            &[0, 0, 0, 133, 10, 0, 0, 0],
            &[0, 0, 37, 204, 88, 0, 0, 0],
            &[0, 0, 126, 55, 162, 0, 0, 0],
            &[0, 0, 175, 0, 155, 17, 0, 0],
            &[0, 48, 128, 0, 74, 100, 0, 0],
            &[0, 155, 187, 147, 153, 215, 0, 0],
            &[1, 182, 15, 15, 15, 159, 25, 0],
            &[60, 121, 0, 0, 0, 69, 111, 0],
            &[149, 34, 0, 0, 0, 3, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Å' / 0xc5
        'Å' => Some(&[
            &[0, 0, 84, 125, 109, 0, 0, 0],
            &[0, 0, 133, 12, 138, 0, 0, 0],
            &[0, 0, 24, 115, 43, 0, 0, 0],
            &[0, 0, 0, 133, 10, 0, 0, 0],
            &[0, 0, 37, 204, 88, 0, 0, 0],
            &[0, 0, 126, 55, 162, 0, 0, 0],
            &[0, 0, 175, 0, 155, 17, 0, 0],
            &[0, 48, 128, 0, 74, 100, 0, 0],
            &[0, 155, 187, 147, 153, 215, 0, 0],
            &[1, 182, 15, 15, 15, 159, 25, 0],
            &[60, 121, 0, 0, 0, 69, 111, 0],
            &[149, 34, 0, 0, 0, 3, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Æ' / 0xc6
        'Æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 145, 145, 145, 111, 0],
            &[0, 0, 147, 136, 71, 14, 11, 0],
            &[0, 0, 168, 111, 60, 0, 0, 0],
            &[0, 42, 127, 111, 60, 0, 0, 0],
            &[0, 116, 54, 111, 182, 159, 90, 0],
            &[0, 222, 149, 228, 60, 0, 0, 0],
            &[14, 163, 9, 118, 60, 0, 0, 0],
            &[83, 92, 0, 111, 60, 0, 0, 0],
            &[155, 21, 0, 111, 184, 162, 124, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ç' / 0xc7
        'Ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 138, 160, 144, 47, 0],
            &[0, 52, 176, 32, 0, 23, 34, 0],
            &[0, 181, 16, 0, 0, 0, 0, 0],
            &[16, 170, 0, 0, 0, 0, 0, 0],
            &[41, 142, 0, 0, 0, 0, 0, 0],
            &[31, 151, 0, 0, 0, 0, 0, 0],
            &[2, 186, 0, 0, 0, 0, 0, 0],
            &[0, 136, 83, 0, 0, 0, 0, 0],
            &[0, 5, 147, 171, 147, 165, 57, 0],
            &[0, 0, 0, 79, 114, 4, 0, 0],
            &[0, 0, 1, 124, 142, 7, 0, 0],
        ]),
        // letter: 'È' / 0xc8
        'È' => Some(&[
            &[0, 0, 67, 15, 0, 0, 0, 0],
            &[0, 0, 46, 150, 0, 0, 0, 0],
            &[0, 0, 0, 47, 40, 0, 0, 0],
            &[0, 107, 145, 145, 145, 145, 22, 0],
            &[0, 178, 14, 14, 14, 14, 2, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 159, 159, 159, 146, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 162, 162, 162, 162, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'É' / 0xc9
        'É' => Some(&[
            &[0, 0, 0, 0, 52, 30, 0, 0],
            &[0, 0, 0, 30, 163, 4, 0, 0],
            &[0, 0, 0, 79, 7, 0, 0, 0],
            &[0, 107, 145, 145, 145, 145, 22, 0],
            &[0, 178, 14, 14, 14, 14, 2, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 159, 159, 159, 146, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 162, 162, 162, 162, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ê' / 0xca
        'Ê' => Some(&[
            &[0, 0, 0, 70, 15, 0, 0, 0],
            &[0, 0, 70, 134, 148, 1, 0, 0],
            &[0, 2, 82, 0, 42, 44, 0, 0],
            &[0, 107, 145, 145, 145, 145, 22, 0],
            &[0, 178, 14, 14, 14, 14, 2, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 159, 159, 159, 146, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 162, 162, 162, 162, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ë' / 0xcb
        'Ë' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 175, 0, 94, 83, 0, 0],
            &[0, 0, 27, 0, 14, 12, 0, 0],
            &[0, 107, 145, 145, 145, 145, 22, 0],
            &[0, 178, 14, 14, 14, 14, 2, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 159, 159, 159, 146, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 162, 162, 162, 162, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ì' / 0xcc
        'Ì' => Some(&[
            &[0, 0, 75, 7, 0, 0, 0, 0],
            &[0, 0, 63, 134, 0, 0, 0, 0],
            &[0, 0, 0, 60, 26, 0, 0, 0],
            &[0, 48, 139, 145, 141, 75, 0, 0],
            &[0, 0, 0, 179, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 48, 139, 223, 142, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Í' / 0xcd
        'Í' => Some(&[
            &[0, 0, 0, 0, 61, 21, 0, 0],
            &[0, 0, 0, 45, 151, 0, 0, 0],
            &[0, 0, 0, 84, 3, 0, 0, 0],
            &[0, 48, 139, 145, 141, 75, 0, 0],
            &[0, 0, 0, 179, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 48, 139, 223, 142, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Î' / 0xce
        'Î' => Some(&[
            &[0, 0, 0, 78, 7, 0, 0, 0],
            &[0, 0, 91, 128, 134, 0, 0, 0],
            &[0, 8, 76, 0, 55, 30, 0, 0],
            &[0, 48, 139, 145, 141, 75, 0, 0],
            &[0, 0, 0, 179, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 48, 139, 223, 142, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ï' / 0xcf
        'Ï' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 160, 0, 115, 61, 0, 0],
            &[0, 0, 26, 0, 19, 7, 0, 0],
            &[0, 48, 139, 145, 141, 75, 0, 0],
            &[0, 0, 0, 179, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 48, 139, 223, 142, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ð' / 0xd0
        'Ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 128, 145, 131, 74, 1, 0, 0],
            &[0, 180, 11, 28, 110, 163, 1, 0],
            &[0, 178, 0, 0, 0, 114, 83, 0],
            &[0, 178, 0, 0, 0, 28, 155, 0],
            &[156, 255, 156, 63, 0, 2, 177, 0],
            &[0, 178, 0, 0, 0, 13, 165, 0],
            &[0, 178, 0, 0, 0, 73, 111, 0],
            &[0, 178, 0, 0, 23, 185, 16, 0],
            &[0, 208, 160, 162, 161, 34, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ñ' / 0xd1
        'Ñ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 23, 156, 139, 45, 116, 0, 0],
            &[0, 41, 19, 22, 106, 22, 0, 0],
            &[0, 141, 5, 0, 0, 68, 31, 0],
            &[1, 219, 101, 0, 0, 119, 55, 0],
            &[1, 166, 165, 4, 0, 119, 55, 0],
            &[1, 170, 74, 96, 0, 119, 55, 0],
            &[1, 170, 0, 167, 3, 119, 55, 0],
            &[1, 170, 0, 79, 91, 119, 55, 0],
            &[1, 170, 0, 1, 168, 119, 55, 0],
            &[1, 170, 0, 0, 85, 196, 55, 0],
            &[1, 170, 0, 0, 2, 214, 55, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ò' / 0xd2
        'Ò' => Some(&[
            &[0, 0, 75, 7, 0, 0, 0, 0],
            &[0, 0, 63, 134, 0, 0, 0, 0],
            &[0, 0, 0, 60, 26, 0, 0, 0],
            &[0, 5, 112, 159, 138, 21, 0, 0],
            &[0, 148, 79, 0, 38, 181, 3, 0],
            &[23, 161, 0, 0, 0, 116, 76, 0],
            &[77, 103, 0, 0, 0, 54, 131, 0],
            &[97, 83, 0, 0, 0, 33, 150, 0],
            &[90, 91, 0, 0, 0, 39, 141, 0],
            &[50, 133, 0, 0, 0, 79, 102, 0],
            &[1, 187, 13, 0, 0, 173, 24, 0],
            &[0, 49, 182, 142, 175, 86, 0, 0],
            &[0, 0, 1, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ó' / 0xd3
        'Ó' => Some(&[
            &[0, 0, 0, 0, 61, 21, 0, 0],
            &[0, 0, 0, 45, 151, 0, 0, 0],
            &[0, 0, 0, 84, 3, 0, 0, 0],
            &[0, 5, 112, 159, 138, 21, 0, 0],
            &[0, 148, 79, 0, 38, 181, 3, 0],
            &[23, 161, 0, 0, 0, 116, 76, 0],
            &[77, 103, 0, 0, 0, 54, 131, 0],
            &[97, 83, 0, 0, 0, 33, 150, 0],
            &[90, 91, 0, 0, 0, 39, 141, 0],
            &[50, 133, 0, 0, 0, 79, 102, 0],
            &[1, 187, 13, 0, 0, 173, 24, 0],
            &[0, 49, 182, 142, 175, 86, 0, 0],
            &[0, 0, 1, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ô' / 0xd4
        'Ô' => Some(&[
            &[0, 0, 0, 78, 7, 0, 0, 0],
            &[0, 0, 91, 128, 134, 0, 0, 0],
            &[0, 8, 76, 0, 55, 30, 0, 0],
            &[0, 5, 112, 159, 138, 21, 0, 0],
            &[0, 148, 79, 0, 38, 181, 3, 0],
            &[23, 161, 0, 0, 0, 116, 76, 0],
            &[77, 103, 0, 0, 0, 54, 131, 0],
            &[97, 83, 0, 0, 0, 33, 150, 0],
            &[90, 91, 0, 0, 0, 39, 141, 0],
            &[50, 133, 0, 0, 0, 79, 102, 0],
            &[1, 187, 13, 0, 0, 173, 24, 0],
            &[0, 49, 182, 142, 175, 86, 0, 0],
            &[0, 0, 1, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Õ' / 0xd5
        'Õ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 156, 135, 45, 111, 0, 0],
            &[0, 48, 12, 28, 106, 16, 0, 0],
            &[0, 5, 112, 159, 138, 21, 0, 0],
            &[0, 148, 79, 0, 38, 181, 3, 0],
            &[23, 161, 0, 0, 0, 116, 76, 0],
            &[77, 103, 0, 0, 0, 54, 131, 0],
            &[97, 83, 0, 0, 0, 33, 150, 0],
            &[90, 91, 0, 0, 0, 39, 141, 0],
            &[50, 133, 0, 0, 0, 79, 102, 0],
            &[1, 187, 13, 0, 0, 173, 24, 0],
            &[0, 49, 182, 142, 175, 86, 0, 0],
            &[0, 0, 1, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ö' / 0xd6
        'Ö' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 160, 0, 115, 61, 0, 0],
            &[0, 0, 26, 0, 19, 7, 0, 0],
            &[0, 5, 112, 159, 138, 21, 0, 0],
            &[0, 148, 79, 0, 38, 181, 3, 0],
            &[23, 161, 0, 0, 0, 116, 76, 0],
            &[77, 103, 0, 0, 0, 54, 131, 0],
            &[97, 83, 0, 0, 0, 33, 150, 0],
            &[90, 91, 0, 0, 0, 39, 141, 0],
            &[50, 133, 0, 0, 0, 79, 102, 0],
            &[1, 187, 13, 0, 0, 173, 24, 0],
            &[0, 49, 182, 142, 175, 86, 0, 0],
            &[0, 0, 1, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '×' / 0xd7
        '×' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 118, 7, 0, 0, 116, 8, 0],
            &[0, 40, 156, 7, 129, 74, 0, 0],
            &[0, 0, 40, 210, 83, 0, 0, 0],
            &[0, 0, 129, 110, 158, 7, 0, 0],
            &[0, 128, 76, 0, 40, 157, 6, 0],
            &[0, 28, 0, 0, 0, 26, 1, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ø' / 0xd8
        'Ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 112, 159, 138, 101, 51, 0],
            &[0, 148, 79, 0, 46, 255, 4, 0],
            &[23, 161, 0, 0, 117, 170, 76, 0],
            &[77, 103, 0, 23, 148, 54, 131, 0],
            &[97, 83, 0, 144, 27, 33, 150, 0],
            &[90, 91, 48, 124, 0, 39, 141, 0],
            &[50, 133, 163, 9, 0, 79, 102, 0],
            &[1, 255, 104, 0, 0, 173, 24, 0],
            &[7, 215, 183, 142, 175, 86, 0, 0],
            &[10, 33, 1, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ù' / 0xd9
        'Ù' => Some(&[
            &[0, 0, 75, 7, 0, 0, 0, 0],
            &[0, 0, 63, 134, 0, 0, 0, 0],
            &[0, 0, 0, 60, 26, 0, 0, 0],
            &[0, 100, 0, 0, 0, 69, 31, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[0, 179, 0, 0, 0, 135, 44, 0],
            &[0, 179, 14, 0, 1, 185, 6, 0],
            &[0, 53, 184, 139, 175, 85, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ú' / 0xda
        'Ú' => Some(&[
            &[0, 0, 0, 0, 61, 21, 0, 0],
            &[0, 0, 0, 45, 151, 0, 0, 0],
            &[0, 0, 0, 84, 3, 0, 0, 0],
            &[0, 100, 0, 0, 0, 69, 31, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[0, 179, 0, 0, 0, 135, 44, 0],
            &[0, 179, 14, 0, 1, 185, 6, 0],
            &[0, 53, 184, 139, 175, 85, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Û' / 0xdb
        'Û' => Some(&[
            &[0, 0, 0, 78, 7, 0, 0, 0],
            &[0, 0, 91, 128, 134, 0, 0, 0],
            &[0, 8, 76, 0, 55, 30, 0, 0],
            &[0, 100, 0, 0, 0, 69, 31, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[0, 179, 0, 0, 0, 135, 44, 0],
            &[0, 179, 14, 0, 1, 185, 6, 0],
            &[0, 53, 184, 139, 175, 85, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ü' / 0xdc
        'Ü' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 160, 0, 115, 61, 0, 0],
            &[0, 0, 26, 0, 19, 7, 0, 0],
            &[0, 100, 0, 0, 0, 69, 31, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[0, 179, 0, 0, 0, 135, 44, 0],
            &[0, 179, 14, 0, 1, 185, 6, 0],
            &[0, 53, 184, 139, 175, 85, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ý' / 0xdd
        'Ý' => Some(&[
            &[0, 0, 0, 0, 61, 21, 0, 0],
            &[0, 0, 0, 45, 151, 0, 0, 0],
            &[0, 0, 0, 84, 3, 0, 0, 0],
            &[48, 61, 0, 0, 0, 29, 79, 0],
            &[8, 180, 1, 0, 0, 146, 39, 0],
            &[0, 102, 81, 0, 28, 150, 0, 0],
            &[0, 4, 171, 1, 141, 31, 0, 0],
            &[0, 0, 90, 107, 139, 0, 0, 0],
            &[0, 0, 2, 209, 24, 0, 0, 0],
            &[0, 0, 0, 171, 0, 0, 0, 0],
            &[0, 0, 0, 171, 0, 0, 0, 0],
            &[0, 0, 0, 171, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Þ' / 0xde
        'Þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 101, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 184, 156, 157, 166, 117, 0, 0],
            &[0, 178, 0, 0, 0, 133, 75, 0],
            &[0, 178, 0, 0, 0, 61, 122, 0],
            &[0, 178, 0, 0, 0, 111, 82, 0],
            &[0, 183, 120, 124, 158, 141, 1, 0],
            &[0, 179, 38, 34, 9, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ß' / 0xdf
        'ß' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 38, 10, 0, 0, 0],
            &[0, 79, 181, 118, 160, 95, 0, 0],
            &[0, 189, 4, 0, 0, 179, 0, 0],
            &[0, 175, 0, 0, 116, 97, 0, 0],
            &[0, 175, 0, 87, 108, 0, 0, 0],
            &[0, 175, 0, 97, 105, 0, 0, 0],
            &[0, 175, 0, 1, 125, 141, 2, 0],
            &[0, 175, 0, 0, 0, 97, 100, 0],
            &[0, 175, 0, 0, 0, 40, 131, 0],
            &[0, 175, 31, 154, 128, 172, 37, 0],
            &[0, 0, 0, 4, 24, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'à' / 0xe0
        'à' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 39, 3, 0, 0, 0],
            &[0, 0, 0, 83, 121, 0, 0, 0],
            &[0, 0, 0, 0, 84, 35, 0, 0],
            &[0, 0, 9, 84, 120, 93, 7, 0],
            &[0, 0, 54, 72, 30, 74, 163, 0],
            &[0, 0, 0, 0, 0, 0, 167, 0],
            &[0, 0, 18, 120, 146, 136, 213, 0],
            &[0, 0, 183, 46, 0, 0, 171, 0],
            &[0, 2, 174, 0, 0, 13, 228, 0],
            &[0, 0, 136, 148, 131, 137, 164, 0],
            &[0, 0, 0, 13, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'á' / 0xe1
        'á' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 16, 0],
            &[0, 0, 0, 0, 16, 174, 12, 0],
            &[0, 0, 0, 0, 100, 20, 0, 0],
            &[0, 0, 9, 84, 120, 93, 7, 0],
            &[0, 0, 54, 72, 30, 74, 163, 0],
            &[0, 0, 0, 0, 0, 0, 167, 0],
            &[0, 0, 18, 120, 146, 136, 213, 0],
            &[0, 0, 183, 46, 0, 0, 171, 0],
            &[0, 2, 174, 0, 0, 13, 228, 0],
            &[0, 0, 136, 148, 131, 137, 164, 0],
            &[0, 0, 0, 13, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'â' / 0xe2
        'â' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 3, 0, 0],
            &[0, 0, 0, 48, 170, 120, 0, 0],
            &[0, 0, 4, 110, 1, 78, 39, 0],
            &[0, 0, 9, 84, 120, 93, 7, 0],
            &[0, 0, 54, 72, 30, 74, 163, 0],
            &[0, 0, 0, 0, 0, 0, 167, 0],
            &[0, 0, 18, 120, 146, 136, 213, 0],
            &[0, 0, 183, 46, 0, 0, 171, 0],
            &[0, 2, 174, 0, 0, 13, 228, 0],
            &[0, 0, 136, 148, 131, 137, 164, 0],
            &[0, 0, 0, 13, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ã' / 0xe3
        'ã' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 142, 109, 4, 94, 0],
            &[0, 0, 53, 34, 53, 147, 44, 0],
            &[0, 0, 9, 84, 120, 93, 7, 0],
            &[0, 0, 54, 72, 30, 74, 163, 0],
            &[0, 0, 0, 0, 0, 0, 167, 0],
            &[0, 0, 18, 120, 146, 136, 213, 0],
            &[0, 0, 183, 46, 0, 0, 171, 0],
            &[0, 2, 174, 0, 0, 13, 228, 0],
            &[0, 0, 136, 148, 131, 137, 164, 0],
            &[0, 0, 0, 13, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ä' / 0xe4
        'ä' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 133, 0, 80, 58, 0],
            &[0, 0, 0, 65, 0, 38, 26, 0],
            &[0, 0, 9, 84, 120, 93, 7, 0],
            &[0, 0, 54, 72, 30, 74, 163, 0],
            &[0, 0, 0, 0, 0, 0, 167, 0],
            &[0, 0, 18, 120, 146, 136, 213, 0],
            &[0, 0, 183, 46, 0, 0, 171, 0],
            &[0, 2, 174, 0, 0, 13, 228, 0],
            &[0, 0, 136, 148, 131, 137, 164, 0],
            &[0, 0, 0, 13, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'å' / 0xe5
        'å' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 48, 126, 91, 0, 0],
            &[0, 0, 0, 132, 3, 135, 0, 0],
            &[0, 0, 0, 39, 127, 81, 0, 0],
            &[0, 0, 9, 84, 120, 93, 7, 0],
            &[0, 0, 54, 72, 30, 74, 163, 0],
            &[0, 0, 0, 0, 0, 0, 167, 0],
            &[0, 0, 18, 120, 146, 136, 213, 0],
            &[0, 0, 183, 46, 0, 0, 171, 0],
            &[0, 2, 174, 0, 0, 13, 228, 0],
            &[0, 0, 136, 148, 131, 137, 164, 0],
            &[0, 0, 0, 13, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'æ' / 0xe6
        'æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[7, 93, 111, 13, 69, 110, 9, 0],
            &[26, 62, 53, 177, 119, 71, 138, 0],
            &[0, 0, 0, 193, 6, 0, 167, 0],
            &[6, 110, 139, 214, 144, 144, 176, 0],
            &[127, 74, 0, 178, 0, 0, 0, 0],
            &[162, 11, 8, 218, 32, 0, 0, 0],
            &[81, 171, 166, 51, 170, 134, 109, 0],
            &[0, 14, 6, 0, 2, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ç' / 0xe7
        'ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 102, 121, 79, 0, 0],
            &[0, 50, 180, 63, 38, 76, 0, 0],
            &[0, 174, 18, 0, 0, 0, 0, 0],
            &[0, 180, 0, 0, 0, 0, 0, 0],
            &[0, 183, 0, 0, 0, 0, 0, 0],
            &[0, 155, 49, 0, 0, 0, 0, 0],
            &[0, 20, 168, 152, 137, 154, 0, 0],
            &[0, 0, 0, 132, 66, 1, 0, 0],
            &[0, 0, 23, 126, 124, 0, 0, 0],
        ]),
        // letter: 'è' / 0xe8
        'è' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 2, 0, 0, 0, 0],
            &[0, 0, 86, 118, 0, 0, 0, 0],
            &[0, 0, 0, 86, 33, 0, 0, 0],
            &[0, 0, 54, 118, 99, 12, 0, 0],
            &[0, 102, 138, 36, 61, 174, 4, 0],
            &[5, 182, 0, 0, 0, 119, 66, 0],
            &[37, 208, 147, 147, 147, 175, 87, 0],
            &[30, 151, 0, 0, 0, 0, 0, 0],
            &[0, 183, 13, 0, 0, 0, 0, 0],
            &[0, 42, 170, 139, 135, 158, 25, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'é' / 0xe9
        'é' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 16, 0, 0],
            &[0, 0, 0, 18, 174, 11, 0, 0],
            &[0, 0, 0, 101, 19, 0, 0, 0],
            &[0, 0, 54, 118, 99, 12, 0, 0],
            &[0, 102, 138, 36, 61, 174, 4, 0],
            &[5, 182, 0, 0, 0, 119, 66, 0],
            &[37, 208, 147, 147, 147, 175, 87, 0],
            &[30, 151, 0, 0, 0, 0, 0, 0],
            &[0, 183, 13, 0, 0, 0, 0, 0],
            &[0, 42, 170, 139, 135, 158, 25, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ê' / 0xea
        'ê' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 2, 0, 0, 0],
            &[0, 0, 51, 170, 118, 0, 0, 0],
            &[0, 5, 109, 1, 80, 37, 0, 0],
            &[0, 0, 54, 118, 99, 12, 0, 0],
            &[0, 102, 138, 36, 61, 174, 4, 0],
            &[5, 182, 0, 0, 0, 119, 66, 0],
            &[37, 208, 147, 147, 147, 175, 87, 0],
            &[30, 151, 0, 0, 0, 0, 0, 0],
            &[0, 183, 13, 0, 0, 0, 0, 0],
            &[0, 42, 170, 139, 135, 158, 25, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ë' / 0xeb
        'ë' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 131, 0, 82, 56, 0, 0],
            &[0, 1, 64, 0, 40, 25, 0, 0],
            &[0, 0, 54, 118, 99, 12, 0, 0],
            &[0, 102, 138, 36, 61, 174, 4, 0],
            &[5, 182, 0, 0, 0, 119, 66, 0],
            &[37, 208, 147, 147, 147, 175, 87, 0],
            &[30, 151, 0, 0, 0, 0, 0, 0],
            &[0, 183, 13, 0, 0, 0, 0, 0],
            &[0, 42, 170, 139, 135, 158, 25, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ì' / 0xec
        'ì' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 41, 0, 0, 0, 0, 0],
            &[0, 0, 97, 107, 0, 0, 0, 0],
            &[0, 0, 0, 93, 26, 0, 0, 0],
            &[0, 70, 98, 98, 16, 0, 0, 0],
            &[0, 23, 43, 160, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 137, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'í' / 0xed
        'í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 13, 0, 0],
            &[0, 0, 0, 23, 172, 8, 0, 0],
            &[0, 0, 0, 105, 15, 0, 0, 0],
            &[0, 70, 98, 98, 16, 0, 0, 0],
            &[0, 23, 43, 160, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 137, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'î' / 0xee
        'î' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 35, 0, 0, 0, 0],
            &[0, 0, 60, 171, 108, 0, 0, 0],
            &[0, 8, 107, 0, 87, 30, 0, 0],
            &[0, 70, 98, 98, 16, 0, 0, 0],
            &[0, 23, 43, 160, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 137, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ï' / 0xef
        'ï' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 125, 0, 90, 48, 0, 0],
            &[0, 4, 61, 0, 44, 21, 0, 0],
            &[0, 70, 98, 98, 16, 0, 0, 0],
            &[0, 23, 43, 160, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 137, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ð' / 0xf0
        'ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 0, 0, 1, 0, 0],
            &[0, 0, 77, 161, 137, 97, 0, 0],
            &[0, 0, 80, 184, 204, 9, 0, 0],
            &[0, 0, 33, 0, 44, 133, 0, 0],
            &[0, 6, 107, 147, 121, 158, 15, 0],
            &[0, 165, 65, 0, 26, 193, 80, 0],
            &[35, 145, 0, 0, 0, 87, 111, 0],
            &[53, 123, 0, 0, 0, 74, 100, 0],
            &[11, 178, 1, 0, 0, 151, 37, 0],
            &[0, 72, 166, 127, 160, 107, 0, 0],
            &[0, 0, 2, 23, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ñ' / 0xf1
        'ñ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 138, 116, 6, 95, 0, 0],
            &[0, 43, 44, 45, 146, 52, 0, 0],
            &[0, 57, 25, 108, 108, 18, 0, 0],
            &[0, 183, 133, 37, 60, 190, 1, 0],
            &[0, 212, 7, 0, 0, 150, 29, 0],
            &[0, 182, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ò' / 0xf2
        'ò' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 41, 0, 0, 0, 0, 0],
            &[0, 0, 97, 107, 0, 0, 0, 0],
            &[0, 0, 0, 93, 26, 0, 0, 0],
            &[0, 0, 69, 120, 90, 6, 0, 0],
            &[0, 132, 113, 33, 74, 169, 1, 0],
            &[17, 172, 0, 0, 0, 119, 66, 0],
            &[54, 126, 0, 0, 0, 70, 109, 0],
            &[46, 133, 0, 0, 0, 82, 99, 0],
            &[4, 184, 5, 0, 0, 158, 29, 0],
            &[0, 61, 171, 128, 162, 100, 0, 0],
            &[0, 0, 1, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ó' / 0xf3
        'ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 13, 0, 0],
            &[0, 0, 0, 23, 172, 8, 0, 0],
            &[0, 0, 0, 105, 15, 0, 0, 0],
            &[0, 0, 69, 120, 90, 6, 0, 0],
            &[0, 132, 113, 33, 74, 169, 1, 0],
            &[17, 172, 0, 0, 0, 119, 66, 0],
            &[54, 126, 0, 0, 0, 70, 109, 0],
            &[46, 133, 0, 0, 0, 82, 99, 0],
            &[4, 184, 5, 0, 0, 158, 29, 0],
            &[0, 61, 171, 128, 162, 100, 0, 0],
            &[0, 0, 1, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ô' / 0xf4
        'ô' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 35, 0, 0, 0, 0],
            &[0, 0, 60, 171, 108, 0, 0, 0],
            &[0, 8, 107, 0, 87, 30, 0, 0],
            &[0, 0, 69, 120, 90, 6, 0, 0],
            &[0, 132, 113, 33, 74, 169, 1, 0],
            &[17, 172, 0, 0, 0, 119, 66, 0],
            &[54, 126, 0, 0, 0, 70, 109, 0],
            &[46, 133, 0, 0, 0, 82, 99, 0],
            &[4, 184, 5, 0, 0, 158, 29, 0],
            &[0, 61, 171, 128, 162, 100, 0, 0],
            &[0, 0, 1, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'õ' / 0xf5
        'õ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 145, 102, 5, 92, 0, 0],
            &[0, 63, 24, 61, 147, 35, 0, 0],
            &[0, 0, 69, 120, 90, 6, 0, 0],
            &[0, 132, 113, 33, 74, 169, 1, 0],
            &[17, 172, 0, 0, 0, 119, 66, 0],
            &[54, 126, 0, 0, 0, 70, 109, 0],
            &[46, 133, 0, 0, 0, 82, 99, 0],
            &[4, 184, 5, 0, 0, 158, 29, 0],
            &[0, 61, 171, 128, 162, 100, 0, 0],
            &[0, 0, 1, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ö' / 0xf6
        'ö' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 125, 0, 90, 48, 0, 0],
            &[0, 4, 61, 0, 44, 21, 0, 0],
            &[0, 0, 69, 120, 90, 6, 0, 0],
            &[0, 132, 113, 33, 74, 169, 1, 0],
            &[17, 172, 0, 0, 0, 119, 66, 0],
            &[54, 126, 0, 0, 0, 70, 109, 0],
            &[46, 133, 0, 0, 0, 82, 99, 0],
            &[4, 184, 5, 0, 0, 158, 29, 0],
            &[0, 61, 171, 128, 162, 100, 0, 0],
            &[0, 0, 1, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '÷' / 0xf7
        '÷' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 169, 5, 0, 0, 0],
            &[0, 0, 0, 29, 0, 0, 0, 0],
            &[13, 134, 134, 134, 134, 134, 40, 0],
            &[1, 13, 13, 13, 13, 13, 3, 0],
            &[0, 0, 0, 131, 5, 0, 0, 0],
            &[0, 0, 0, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ø' / 0xf8
        'ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 120, 90, 61, 49, 0],
            &[0, 132, 113, 33, 92, 255, 5, 0],
            &[17, 172, 0, 1, 156, 144, 66, 0],
            &[54, 126, 0, 117, 66, 70, 109, 0],
            &[46, 133, 62, 121, 0, 82, 99, 0],
            &[4, 207, 164, 2, 0, 158, 29, 0],
            &[2, 222, 192, 128, 162, 100, 0, 0],
            &[2, 32, 1, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ù' / 0xf9
        'ù' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 41, 0, 0, 0, 0, 0],
            &[0, 0, 97, 107, 0, 0, 0, 0],
            &[0, 0, 0, 93, 26, 0, 0, 0],
            &[0, 68, 0, 0, 0, 58, 9, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 162, 24, 0],
            &[0, 185, 3, 0, 6, 221, 24, 0],
            &[0, 85, 175, 128, 144, 147, 24, 0],
            &[0, 0, 5, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ú' / 0xfa
        'ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 13, 0, 0],
            &[0, 0, 0, 23, 172, 8, 0, 0],
            &[0, 0, 0, 105, 15, 0, 0, 0],
            &[0, 68, 0, 0, 0, 58, 9, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 162, 24, 0],
            &[0, 185, 3, 0, 6, 221, 24, 0],
            &[0, 85, 175, 128, 144, 147, 24, 0],
            &[0, 0, 5, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'û' / 0xfb
        'û' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 35, 0, 0, 0, 0],
            &[0, 0, 60, 171, 108, 0, 0, 0],
            &[0, 8, 107, 0, 87, 30, 0, 0],
            &[0, 68, 0, 0, 0, 58, 9, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 162, 24, 0],
            &[0, 185, 3, 0, 6, 221, 24, 0],
            &[0, 85, 175, 128, 144, 147, 24, 0],
            &[0, 0, 5, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ü' / 0xfc
        'ü' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 125, 0, 90, 48, 0, 0],
            &[0, 4, 61, 0, 44, 21, 0, 0],
            &[0, 68, 0, 0, 0, 58, 9, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 162, 24, 0],
            &[0, 185, 3, 0, 6, 221, 24, 0],
            &[0, 85, 175, 128, 144, 147, 24, 0],
            &[0, 0, 5, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ý' / 0xfd
        'ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 13, 0, 0],
            &[0, 0, 0, 23, 172, 8, 0, 0],
            &[0, 0, 0, 105, 15, 0, 0, 0],
            &[10, 57, 0, 0, 0, 36, 34, 0],
            &[0, 174, 0, 0, 0, 154, 25, 0],
            &[0, 111, 62, 0, 8, 166, 0, 0],
            &[0, 18, 153, 0, 85, 85, 0, 0],
            &[0, 0, 153, 14, 159, 7, 0, 0],
            &[0, 0, 62, 118, 142, 0, 0, 0],
            &[0, 0, 0, 206, 51, 0, 0, 0],
            &[0, 0, 0, 170, 0, 0, 0, 0],
            &[14, 148, 151, 7, 0, 0, 0, 0],
        ]),
        // letter: 'þ' / 0xfe
        'þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 147, 0, 0, 0, 0, 0, 0],
            &[0, 175, 0, 0, 0, 0, 0, 0],
            &[0, 175, 23, 108, 100, 14, 0, 0],
            &[0, 188, 135, 39, 62, 179, 3, 0],
            &[0, 214, 10, 0, 0, 111, 72, 0],
            &[0, 182, 0, 0, 0, 75, 107, 0],
            &[0, 193, 0, 0, 0, 87, 100, 0],
            &[0, 221, 40, 0, 0, 157, 39, 0],
            &[0, 173, 135, 135, 162, 121, 0, 0],
            &[0, 175, 0, 13, 10, 0, 0, 0],
            &[0, 154, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ÿ' / 0xff
        'ÿ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 125, 0, 90, 48, 0, 0],
            &[0, 4, 61, 0, 44, 21, 0, 0],
            &[10, 57, 0, 0, 0, 36, 34, 0],
            &[0, 174, 0, 0, 0, 154, 25, 0],
            &[0, 111, 62, 0, 8, 166, 0, 0],
            &[0, 18, 153, 0, 85, 85, 0, 0],
            &[0, 0, 153, 14, 159, 7, 0, 0],
            &[0, 0, 62, 118, 142, 0, 0, 0],
            &[0, 0, 0, 206, 51, 0, 0, 0],
            &[0, 0, 0, 170, 0, 0, 0, 0],
            &[14, 148, 151, 7, 0, 0, 0, 0],
        ]),
        // letter: 'Ā' / 0x100
        'Ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 69, 69, 69, 17, 0, 0],
            &[0, 4, 77, 77, 77, 19, 0, 0],
            &[0, 0, 0, 133, 10, 0, 0, 0],
            &[0, 0, 37, 204, 88, 0, 0, 0],
            &[0, 0, 126, 55, 162, 0, 0, 0],
            &[0, 0, 175, 0, 155, 17, 0, 0],
            &[0, 48, 128, 0, 74, 100, 0, 0],
            &[0, 155, 187, 147, 153, 215, 0, 0],
            &[1, 182, 15, 15, 15, 159, 25, 0],
            &[60, 121, 0, 0, 0, 69, 111, 0],
            &[149, 34, 0, 0, 0, 3, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ā' / 0x101
        'ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 22, 22, 7, 0],
            &[0, 0, 0, 124, 124, 124, 39, 0],
            &[0, 0, 9, 84, 120, 93, 7, 0],
            &[0, 0, 54, 72, 30, 74, 163, 0],
            &[0, 0, 0, 0, 0, 0, 167, 0],
            &[0, 0, 18, 120, 146, 136, 213, 0],
            &[0, 0, 183, 46, 0, 0, 171, 0],
            &[0, 2, 174, 0, 0, 13, 228, 0],
            &[0, 0, 136, 148, 131, 137, 164, 0],
            &[0, 0, 0, 13, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ă' / 0x102
        'Ă' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 154, 23, 137, 32, 0, 0],
            &[0, 0, 38, 113, 58, 0, 0, 0],
            &[0, 0, 0, 133, 10, 0, 0, 0],
            &[0, 0, 37, 204, 88, 0, 0, 0],
            &[0, 0, 126, 55, 162, 0, 0, 0],
            &[0, 0, 175, 0, 155, 17, 0, 0],
            &[0, 48, 128, 0, 74, 100, 0, 0],
            &[0, 155, 187, 147, 153, 215, 0, 0],
            &[1, 182, 15, 15, 15, 159, 25, 0],
            &[60, 121, 0, 0, 0, 69, 111, 0],
            &[149, 34, 0, 0, 0, 3, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ă' / 0x103
        'ă' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 123, 0, 83, 45, 0],
            &[0, 0, 0, 62, 137, 109, 0, 0],
            &[0, 0, 9, 84, 120, 93, 7, 0],
            &[0, 0, 54, 72, 30, 74, 163, 0],
            &[0, 0, 0, 0, 0, 0, 167, 0],
            &[0, 0, 18, 120, 146, 136, 213, 0],
            &[0, 0, 183, 46, 0, 0, 171, 0],
            &[0, 2, 174, 0, 0, 13, 228, 0],
            &[0, 0, 136, 148, 131, 137, 164, 0],
            &[0, 0, 0, 13, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ą' / 0x104
        'Ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 133, 10, 0, 0, 0],
            &[0, 0, 37, 204, 88, 0, 0, 0],
            &[0, 0, 126, 55, 162, 0, 0, 0],
            &[0, 0, 175, 0, 155, 17, 0, 0],
            &[0, 48, 128, 0, 74, 100, 0, 0],
            &[0, 155, 187, 147, 153, 215, 0, 0],
            &[1, 182, 15, 15, 15, 159, 25, 0],
            &[60, 121, 0, 0, 0, 69, 111, 0],
            &[149, 34, 0, 0, 0, 3, 190, 0],
            &[0, 0, 0, 0, 0, 39, 114, 0],
            &[0, 0, 0, 0, 0, 34, 129, 24],
        ]),
        // letter: 'ą' / 0x105
        'ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 84, 120, 93, 7, 0, 0],
            &[0, 54, 72, 30, 74, 163, 0, 0],
            &[0, 0, 0, 0, 0, 167, 0, 0],
            &[0, 18, 120, 146, 136, 213, 0, 0],
            &[0, 183, 46, 0, 0, 171, 0, 0],
            &[2, 174, 0, 0, 13, 228, 0, 0],
            &[0, 136, 148, 131, 137, 175, 0, 0],
            &[0, 0, 13, 12, 38, 114, 0, 0],
            &[0, 0, 0, 0, 33, 129, 25, 0],
        ]),
        // letter: 'Ć' / 0x106
        'Ć' => Some(&[
            &[0, 0, 0, 0, 0, 81, 0, 0],
            &[0, 0, 0, 0, 104, 93, 0, 0],
            &[0, 0, 0, 11, 76, 0, 0, 0],
            &[0, 0, 36, 138, 160, 144, 47, 0],
            &[0, 52, 176, 32, 0, 23, 34, 0],
            &[0, 181, 16, 0, 0, 0, 0, 0],
            &[16, 170, 0, 0, 0, 0, 0, 0],
            &[41, 142, 0, 0, 0, 0, 0, 0],
            &[31, 151, 0, 0, 0, 0, 0, 0],
            &[2, 186, 0, 0, 0, 0, 0, 0],
            &[0, 136, 83, 0, 0, 0, 0, 0],
            &[0, 5, 147, 171, 142, 165, 57, 0],
            &[0, 0, 0, 5, 23, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ć' / 0x107
        'ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 31, 0, 0],
            &[0, 0, 0, 0, 153, 51, 0, 0],
            &[0, 0, 0, 62, 58, 0, 0, 0],
            &[0, 0, 23, 102, 121, 79, 0, 0],
            &[0, 50, 180, 63, 38, 76, 0, 0],
            &[0, 174, 18, 0, 0, 0, 0, 0],
            &[0, 180, 0, 0, 0, 0, 0, 0],
            &[0, 183, 0, 0, 0, 0, 0, 0],
            &[0, 155, 49, 0, 0, 0, 0, 0],
            &[0, 20, 168, 150, 135, 154, 0, 0],
            &[0, 0, 0, 13, 21, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĉ' / 0x108
        'Ĉ' => Some(&[
            &[0, 0, 0, 12, 73, 0, 0, 0],
            &[0, 0, 0, 142, 133, 77, 0, 0],
            &[0, 0, 40, 45, 0, 82, 3, 0],
            &[0, 0, 36, 138, 160, 144, 47, 0],
            &[0, 52, 176, 32, 0, 23, 34, 0],
            &[0, 181, 16, 0, 0, 0, 0, 0],
            &[16, 170, 0, 0, 0, 0, 0, 0],
            &[41, 142, 0, 0, 0, 0, 0, 0],
            &[31, 151, 0, 0, 0, 0, 0, 0],
            &[2, 186, 0, 0, 0, 0, 0, 0],
            &[0, 136, 83, 0, 0, 0, 0, 0],
            &[0, 5, 147, 171, 142, 165, 57, 0],
            &[0, 0, 0, 5, 23, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĉ' / 0x109
        'ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 20, 16, 0, 0, 0],
            &[0, 0, 11, 162, 161, 5, 0, 0],
            &[0, 0, 94, 21, 33, 84, 0, 0],
            &[0, 0, 23, 102, 121, 79, 0, 0],
            &[0, 50, 180, 63, 38, 76, 0, 0],
            &[0, 174, 18, 0, 0, 0, 0, 0],
            &[0, 180, 0, 0, 0, 0, 0, 0],
            &[0, 183, 0, 0, 0, 0, 0, 0],
            &[0, 155, 49, 0, 0, 0, 0, 0],
            &[0, 20, 168, 150, 135, 154, 0, 0],
            &[0, 0, 0, 13, 21, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ċ' / 0x10a
        'Ċ' => Some(&[
            &[0, 0, 0, 0, 3, 0, 0, 0],
            &[0, 0, 0, 38, 201, 0, 0, 0],
            &[0, 0, 0, 0, 9, 0, 0, 0],
            &[0, 0, 36, 138, 160, 144, 47, 0],
            &[0, 52, 176, 32, 0, 23, 34, 0],
            &[0, 181, 16, 0, 0, 0, 0, 0],
            &[16, 170, 0, 0, 0, 0, 0, 0],
            &[41, 142, 0, 0, 0, 0, 0, 0],
            &[31, 151, 0, 0, 0, 0, 0, 0],
            &[2, 186, 0, 0, 0, 0, 0, 0],
            &[0, 136, 83, 0, 0, 0, 0, 0],
            &[0, 5, 147, 171, 142, 165, 57, 0],
            &[0, 0, 0, 5, 23, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ċ' / 0x10b
        'ċ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 122, 97, 0, 0, 0],
            &[0, 0, 0, 19, 13, 0, 0, 0],
            &[0, 0, 23, 102, 121, 79, 0, 0],
            &[0, 50, 180, 63, 38, 76, 0, 0],
            &[0, 174, 18, 0, 0, 0, 0, 0],
            &[0, 180, 0, 0, 0, 0, 0, 0],
            &[0, 183, 0, 0, 0, 0, 0, 0],
            &[0, 155, 49, 0, 0, 0, 0, 0],
            &[0, 20, 168, 150, 135, 154, 0, 0],
            &[0, 0, 0, 13, 21, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Č' / 0x10c
        'Č' => Some(&[
            &[0, 0, 34, 21, 0, 53, 4, 0],
            &[0, 0, 7, 154, 94, 104, 0, 0],
            &[0, 0, 0, 29, 115, 0, 0, 0],
            &[0, 0, 36, 138, 160, 144, 47, 0],
            &[0, 52, 176, 32, 0, 23, 34, 0],
            &[0, 181, 16, 0, 0, 0, 0, 0],
            &[16, 170, 0, 0, 0, 0, 0, 0],
            &[41, 142, 0, 0, 0, 0, 0, 0],
            &[31, 151, 0, 0, 0, 0, 0, 0],
            &[2, 186, 0, 0, 0, 0, 0, 0],
            &[0, 136, 83, 0, 0, 0, 0, 0],
            &[0, 5, 147, 171, 142, 165, 57, 0],
            &[0, 0, 0, 5, 23, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'č' / 0x10d
        'č' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 0, 0, 27, 0, 0],
            &[0, 0, 82, 92, 115, 62, 0, 0],
            &[0, 0, 0, 117, 97, 0, 0, 0],
            &[0, 0, 23, 102, 121, 79, 0, 0],
            &[0, 50, 180, 63, 38, 76, 0, 0],
            &[0, 174, 18, 0, 0, 0, 0, 0],
            &[0, 180, 0, 0, 0, 0, 0, 0],
            &[0, 183, 0, 0, 0, 0, 0, 0],
            &[0, 155, 49, 0, 0, 0, 0, 0],
            &[0, 20, 168, 150, 135, 154, 0, 0],
            &[0, 0, 0, 13, 21, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ď' / 0x10e
        'Ď' => Some(&[
            &[0, 19, 36, 0, 41, 16, 0, 0],
            &[0, 0, 141, 85, 134, 0, 0, 0],
            &[0, 0, 10, 128, 5, 0, 0, 0],
            &[0, 145, 145, 128, 62, 0, 0, 0],
            &[1, 180, 12, 34, 128, 140, 0, 0],
            &[1, 177, 0, 0, 0, 144, 52, 0],
            &[1, 177, 0, 0, 0, 58, 125, 0],
            &[1, 177, 0, 0, 0, 32, 147, 0],
            &[1, 177, 0, 0, 0, 44, 134, 0],
            &[1, 177, 0, 0, 0, 104, 80, 0],
            &[1, 177, 0, 0, 36, 183, 5, 0],
            &[1, 226, 160, 163, 153, 21, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ď' / 0x10f
        'ď' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 18, 6, 15],
            &[0, 0, 0, 0, 20, 157, 82, 82],
            &[0, 0, 0, 0, 20, 157, 97, 4],
            &[0, 13, 104, 107, 34, 157, 0, 0],
            &[1, 180, 66, 39, 145, 157, 0, 0],
            &[51, 135, 0, 0, 63, 157, 0, 0],
            &[86, 94, 0, 0, 25, 157, 0, 0],
            &[74, 102, 0, 0, 34, 157, 0, 0],
            &[27, 158, 0, 0, 97, 157, 0, 0],
            &[0, 120, 157, 138, 106, 154, 0, 0],
            &[0, 0, 9, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Đ' / 0x110
        'Đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 128, 145, 131, 74, 1, 0, 0],
            &[0, 180, 11, 28, 110, 163, 1, 0],
            &[0, 178, 0, 0, 0, 114, 83, 0],
            &[0, 178, 0, 0, 0, 28, 155, 0],
            &[156, 255, 156, 63, 0, 2, 177, 0],
            &[0, 178, 0, 0, 0, 13, 165, 0],
            &[0, 178, 0, 0, 0, 73, 111, 0],
            &[0, 178, 0, 0, 23, 185, 16, 0],
            &[0, 208, 160, 162, 161, 34, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'đ' / 0x111
        'đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 2, 0],
            &[0, 0, 0, 0, 0, 153, 24, 0],
            &[0, 0, 0, 131, 141, 255, 165, 28],
            &[0, 1, 74, 109, 48, 153, 24, 0],
            &[0, 147, 113, 41, 112, 204, 24, 0],
            &[19, 176, 0, 0, 0, 200, 24, 0],
            &[54, 130, 0, 0, 0, 158, 24, 0],
            &[42, 135, 0, 0, 0, 168, 24, 0],
            &[7, 182, 1, 0, 7, 226, 24, 0],
            &[0, 79, 173, 132, 152, 151, 24, 0],
            &[0, 0, 2, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ē' / 0x112
        'Ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 66, 69, 69, 24, 0, 0],
            &[0, 0, 74, 77, 77, 27, 0, 0],
            &[0, 107, 145, 145, 145, 145, 22, 0],
            &[0, 178, 14, 14, 14, 14, 2, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 159, 159, 159, 146, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 162, 162, 162, 162, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ē' / 0x113
        'ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 22, 22, 6, 0, 0],
            &[0, 0, 124, 124, 124, 37, 0, 0],
            &[0, 0, 54, 118, 99, 12, 0, 0],
            &[0, 102, 138, 36, 61, 174, 4, 0],
            &[5, 182, 0, 0, 0, 119, 66, 0],
            &[37, 208, 147, 147, 147, 175, 87, 0],
            &[30, 151, 0, 0, 0, 0, 0, 0],
            &[0, 183, 13, 0, 0, 0, 0, 0],
            &[0, 42, 170, 139, 135, 158, 25, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĕ' / 0x114
        'Ĕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 151, 25, 120, 53, 0, 0],
            &[0, 0, 28, 112, 69, 0, 0, 0],
            &[0, 107, 145, 145, 145, 145, 22, 0],
            &[0, 178, 14, 14, 14, 14, 2, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 159, 159, 159, 146, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 162, 162, 162, 162, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĕ' / 0x115
        'ĕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 0, 85, 42, 0, 0],
            &[0, 0, 64, 137, 107, 0, 0, 0],
            &[0, 0, 54, 118, 99, 12, 0, 0],
            &[0, 102, 138, 36, 61, 174, 4, 0],
            &[5, 182, 0, 0, 0, 119, 66, 0],
            &[37, 208, 147, 147, 147, 175, 87, 0],
            &[30, 151, 0, 0, 0, 0, 0, 0],
            &[0, 183, 13, 0, 0, 0, 0, 0],
            &[0, 42, 170, 139, 135, 158, 25, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ė' / 0x116
        'Ė' => Some(&[
            &[0, 0, 0, 3, 0, 0, 0, 0],
            &[0, 0, 0, 197, 42, 0, 0, 0],
            &[0, 0, 0, 9, 0, 0, 0, 0],
            &[0, 107, 145, 145, 145, 145, 22, 0],
            &[0, 178, 14, 14, 14, 14, 2, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 159, 159, 159, 146, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 162, 162, 162, 162, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ė' / 0x117
        'ė' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 190, 29, 0, 0, 0],
            &[0, 0, 0, 31, 1, 0, 0, 0],
            &[0, 0, 54, 118, 99, 12, 0, 0],
            &[0, 102, 138, 36, 61, 174, 4, 0],
            &[5, 182, 0, 0, 0, 119, 66, 0],
            &[37, 208, 147, 147, 147, 175, 87, 0],
            &[30, 151, 0, 0, 0, 0, 0, 0],
            &[0, 183, 13, 0, 0, 0, 0, 0],
            &[0, 42, 170, 139, 135, 158, 25, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ę' / 0x118
        'Ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 145, 145, 145, 145, 22, 0],
            &[0, 178, 14, 14, 14, 14, 2, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 159, 159, 159, 146, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 162, 162, 163, 173, 25, 0],
            &[0, 0, 0, 0, 122, 31, 0, 0],
            &[0, 0, 0, 0, 97, 90, 0, 0],
        ]),
        // letter: 'ę' / 0x119
        'ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 118, 99, 12, 0, 0],
            &[0, 102, 138, 36, 61, 174, 4, 0],
            &[5, 182, 0, 0, 0, 119, 66, 0],
            &[37, 208, 147, 147, 147, 175, 87, 0],
            &[30, 151, 0, 0, 0, 0, 0, 0],
            &[0, 183, 13, 0, 0, 0, 0, 0],
            &[0, 42, 170, 139, 135, 203, 28, 0],
            &[0, 0, 0, 16, 94, 71, 0, 0],
            &[0, 0, 0, 0, 59, 133, 2, 0],
        ]),
        // letter: 'Ě' / 0x11a
        'Ě' => Some(&[
            &[0, 2, 53, 0, 20, 36, 0, 0],
            &[0, 0, 103, 95, 155, 8, 0, 0],
            &[0, 0, 0, 116, 28, 0, 0, 0],
            &[0, 107, 145, 145, 145, 145, 22, 0],
            &[0, 178, 14, 14, 14, 14, 2, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 159, 159, 159, 146, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[0, 185, 162, 162, 162, 162, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ě' / 0x11b
        'ě' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 21, 0, 8, 19, 0, 0],
            &[0, 0, 135, 47, 150, 18, 0, 0],
            &[0, 0, 7, 165, 41, 0, 0, 0],
            &[0, 0, 54, 118, 99, 12, 0, 0],
            &[0, 102, 138, 36, 61, 174, 4, 0],
            &[5, 182, 0, 0, 0, 119, 66, 0],
            &[37, 208, 147, 147, 147, 175, 87, 0],
            &[30, 151, 0, 0, 0, 0, 0, 0],
            &[0, 183, 13, 0, 0, 0, 0, 0],
            &[0, 42, 170, 139, 135, 158, 25, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => Some(&[
            &[0, 0, 0, 52, 33, 0, 0, 0],
            &[0, 0, 35, 146, 159, 11, 0, 0],
            &[0, 0, 81, 3, 18, 67, 0, 0],
            &[0, 0, 71, 155, 161, 114, 4, 0],
            &[0, 107, 138, 12, 1, 53, 0, 0],
            &[10, 184, 0, 0, 0, 0, 0, 0],
            &[68, 117, 0, 0, 0, 0, 0, 0],
            &[94, 88, 0, 20, 94, 94, 20, 0],
            &[89, 93, 0, 15, 68, 156, 55, 0],
            &[50, 139, 0, 0, 0, 119, 55, 0],
            &[1, 188, 26, 0, 0, 119, 55, 0],
            &[0, 35, 178, 150, 145, 185, 36, 0],
            &[0, 0, 0, 15, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĝ' / 0x11d
        'ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 35, 0, 0, 0, 0],
            &[0, 0, 60, 171, 108, 0, 0, 0],
            &[0, 8, 107, 0, 87, 30, 0, 0],
            &[0, 0, 77, 120, 61, 48, 9, 0],
            &[0, 137, 103, 31, 93, 209, 24, 0],
            &[15, 168, 0, 0, 0, 179, 24, 0],
            &[50, 128, 0, 0, 0, 144, 24, 0],
            &[42, 136, 0, 0, 0, 153, 24, 0],
            &[7, 184, 2, 0, 4, 216, 24, 0],
            &[0, 75, 175, 132, 157, 167, 24, 0],
            &[0, 0, 1, 21, 1, 153, 18, 0],
            &[0, 138, 153, 154, 167, 52, 0, 0],
        ]),
        // letter: 'Ğ' / 0x11e
        'Ğ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 135, 37, 77, 100, 0, 0],
            &[0, 0, 13, 106, 88, 2, 0, 0],
            &[0, 0, 71, 155, 161, 114, 4, 0],
            &[0, 107, 138, 12, 1, 53, 0, 0],
            &[10, 184, 0, 0, 0, 0, 0, 0],
            &[68, 117, 0, 0, 0, 0, 0, 0],
            &[94, 88, 0, 20, 94, 94, 20, 0],
            &[89, 93, 0, 15, 68, 156, 55, 0],
            &[50, 139, 0, 0, 0, 119, 55, 0],
            &[1, 188, 26, 0, 0, 119, 55, 0],
            &[0, 35, 178, 150, 145, 185, 36, 0],
            &[0, 0, 0, 15, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ğ' / 0x11f
        'ğ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 120, 0, 95, 32, 0, 0],
            &[0, 0, 71, 137, 100, 0, 0, 0],
            &[0, 0, 77, 120, 61, 48, 9, 0],
            &[0, 137, 103, 31, 93, 209, 24, 0],
            &[15, 168, 0, 0, 0, 179, 24, 0],
            &[50, 128, 0, 0, 0, 144, 24, 0],
            &[42, 136, 0, 0, 0, 153, 24, 0],
            &[7, 184, 2, 0, 4, 216, 24, 0],
            &[0, 75, 175, 132, 157, 167, 24, 0],
            &[0, 0, 1, 21, 1, 153, 18, 0],
            &[0, 138, 153, 154, 167, 52, 0, 0],
        ]),
        // letter: 'Ġ' / 0x120
        'Ġ' => Some(&[
            &[0, 0, 0, 3, 0, 0, 0, 0],
            &[0, 0, 0, 148, 91, 0, 0, 0],
            &[0, 0, 0, 7, 2, 0, 0, 0],
            &[0, 0, 71, 155, 161, 114, 4, 0],
            &[0, 107, 138, 12, 1, 53, 0, 0],
            &[10, 184, 0, 0, 0, 0, 0, 0],
            &[68, 117, 0, 0, 0, 0, 0, 0],
            &[94, 88, 0, 20, 94, 94, 20, 0],
            &[89, 93, 0, 15, 68, 156, 55, 0],
            &[50, 139, 0, 0, 0, 119, 55, 0],
            &[1, 188, 26, 0, 0, 119, 55, 0],
            &[0, 35, 178, 150, 145, 185, 36, 0],
            &[0, 0, 0, 15, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ġ' / 0x121
        'ġ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 200, 19, 0, 0, 0],
            &[0, 0, 0, 32, 0, 0, 0, 0],
            &[0, 0, 77, 120, 61, 48, 9, 0],
            &[0, 137, 103, 31, 93, 209, 24, 0],
            &[15, 168, 0, 0, 0, 179, 24, 0],
            &[50, 128, 0, 0, 0, 144, 24, 0],
            &[42, 136, 0, 0, 0, 153, 24, 0],
            &[7, 184, 2, 0, 4, 216, 24, 0],
            &[0, 75, 175, 132, 157, 167, 24, 0],
            &[0, 0, 1, 21, 1, 153, 18, 0],
            &[0, 138, 153, 154, 167, 52, 0, 0],
        ]),
        // letter: 'Ģ' / 0x122
        'Ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 71, 155, 161, 114, 4, 0],
            &[0, 107, 138, 12, 1, 53, 0, 0],
            &[10, 184, 0, 0, 0, 0, 0, 0],
            &[68, 117, 0, 0, 0, 0, 0, 0],
            &[94, 88, 0, 20, 94, 94, 20, 0],
            &[89, 93, 0, 15, 68, 156, 55, 0],
            &[50, 139, 0, 0, 0, 119, 55, 0],
            &[1, 188, 26, 0, 0, 119, 55, 0],
            &[0, 35, 178, 150, 145, 185, 36, 0],
            &[0, 0, 0, 38, 40, 0, 0, 0],
            &[0, 0, 0, 81, 0, 0, 0, 0],
        ]),
        // letter: 'ģ' / 0x123
        'ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 15, 0, 0, 0],
            &[0, 0, 0, 112, 32, 0, 0, 0],
            &[0, 0, 0, 128, 0, 0, 0, 0],
            &[0, 0, 77, 120, 61, 48, 9, 0],
            &[0, 137, 103, 31, 93, 209, 24, 0],
            &[15, 168, 0, 0, 0, 179, 24, 0],
            &[50, 128, 0, 0, 0, 144, 24, 0],
            &[42, 136, 0, 0, 0, 153, 24, 0],
            &[7, 184, 2, 0, 4, 216, 24, 0],
            &[0, 75, 175, 132, 157, 167, 24, 0],
            &[0, 0, 1, 21, 1, 153, 18, 0],
            &[0, 138, 153, 154, 167, 52, 0, 0],
        ]),
        // letter: 'Ĥ' / 0x124
        'Ĥ' => Some(&[
            &[0, 0, 0, 78, 7, 0, 0, 0],
            &[0, 0, 91, 128, 134, 0, 0, 0],
            &[0, 8, 76, 0, 55, 30, 0, 0],
            &[0, 100, 0, 0, 0, 69, 31, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 227, 162, 162, 162, 207, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĥ' / 0x125
        'ĥ' => Some(&[
            &[0, 49, 197, 35, 0, 0, 0, 0],
            &[22, 137, 11, 143, 13, 0, 0, 0],
            &[0, 0, 21, 0, 0, 0, 0, 0],
            &[0, 0, 178, 0, 0, 0, 0, 0],
            &[0, 0, 178, 0, 0, 0, 0, 0],
            &[0, 0, 177, 23, 108, 106, 16, 0],
            &[0, 0, 189, 130, 35, 64, 189, 1],
            &[0, 0, 211, 6, 0, 0, 153, 28],
            &[0, 0, 180, 0, 0, 0, 138, 39],
            &[0, 0, 178, 0, 0, 0, 138, 39],
            &[0, 0, 178, 0, 0, 0, 138, 39],
            &[0, 0, 178, 0, 0, 0, 138, 39],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ħ' / 0x126
        'Ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 0, 0, 0, 69, 31, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[151, 255, 150, 150, 150, 255, 205, 30],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 227, 162, 162, 162, 207, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ħ' / 0x127
        'ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 0, 0, 0, 0, 0, 0],
            &[0, 178, 0, 0, 0, 0, 0, 0],
            &[141, 255, 141, 141, 18, 0, 0, 0],
            &[0, 177, 23, 108, 106, 16, 0, 0],
            &[0, 189, 130, 35, 64, 189, 1, 0],
            &[0, 211, 6, 0, 0, 153, 28, 0],
            &[0, 180, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĩ' / 0x128
        'Ĩ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 156, 135, 45, 111, 0, 0],
            &[0, 48, 12, 28, 106, 16, 0, 0],
            &[0, 48, 139, 145, 141, 75, 0, 0],
            &[0, 0, 0, 179, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 48, 139, 223, 142, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĩ' / 0x129
        'ĩ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 145, 102, 5, 92, 0, 0],
            &[0, 63, 24, 61, 147, 35, 0, 0],
            &[0, 70, 98, 98, 16, 0, 0, 0],
            &[0, 23, 43, 160, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 137, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ī' / 0x12a
        'Ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 69, 69, 69, 17, 0, 0],
            &[0, 4, 77, 77, 77, 19, 0, 0],
            &[0, 48, 139, 145, 141, 75, 0, 0],
            &[0, 0, 0, 179, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 48, 139, 223, 142, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ī' / 0x12b
        'ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 22, 22, 22, 5, 0, 0],
            &[0, 6, 124, 124, 124, 31, 0, 0],
            &[0, 70, 98, 98, 16, 0, 0, 0],
            &[0, 23, 43, 160, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 137, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 154, 23, 137, 32, 0, 0],
            &[0, 0, 38, 113, 58, 0, 0, 0],
            &[0, 48, 139, 145, 141, 75, 0, 0],
            &[0, 0, 0, 179, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 48, 139, 223, 142, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĭ' / 0x12d
        'ĭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 120, 0, 95, 32, 0, 0],
            &[0, 0, 71, 137, 100, 0, 0, 0],
            &[0, 70, 98, 98, 16, 0, 0, 0],
            &[0, 23, 43, 160, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 137, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Į' / 0x12e
        'Į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 48, 139, 145, 141, 75, 0, 0],
            &[0, 0, 0, 179, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 48, 139, 223, 153, 75, 0, 0],
            &[0, 0, 0, 23, 129, 0, 0, 0],
            &[0, 0, 0, 23, 127, 37, 0, 0],
        ]),
        // letter: 'į' / 0x12f
        'į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 169, 50, 0, 0, 0],
            &[0, 0, 0, 28, 4, 0, 0, 0],
            &[0, 70, 98, 98, 16, 0, 0, 0],
            &[0, 23, 43, 160, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 147, 53, 0],
            &[0, 0, 0, 0, 122, 31, 0, 0],
            &[0, 0, 0, 0, 97, 90, 0, 0],
        ]),
        // letter: 'İ' / 0x130
        'İ' => Some(&[
            &[0, 0, 0, 3, 0, 0, 0, 0],
            &[0, 0, 0, 219, 20, 0, 0, 0],
            &[0, 0, 0, 9, 0, 0, 0, 0],
            &[0, 48, 139, 145, 141, 75, 0, 0],
            &[0, 0, 0, 179, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 48, 139, 223, 142, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ı' / 0x131
        'ı' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 70, 98, 98, 16, 0, 0, 0],
            &[0, 23, 43, 160, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 137, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĳ' / 0x132
        'Ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 101, 0, 0, 0, 101, 0, 0],
            &[0, 178, 0, 0, 0, 178, 0, 0],
            &[0, 178, 0, 0, 0, 178, 0, 0],
            &[0, 178, 0, 0, 0, 178, 0, 0],
            &[0, 178, 0, 0, 0, 178, 0, 0],
            &[0, 178, 0, 0, 0, 178, 0, 0],
            &[0, 162, 0, 0, 0, 179, 0, 0],
            &[0, 0, 0, 0, 12, 182, 0, 0],
            &[0, 113, 150, 133, 187, 61, 0, 0],
            &[0, 0, 15, 30, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĳ' / 0x133
        'ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 91, 0, 42, 178, 0, 0],
            &[0, 21, 12, 0, 3, 30, 0, 0],
            &[0, 43, 25, 0, 1, 66, 0, 0],
            &[0, 112, 65, 0, 5, 172, 0, 0],
            &[0, 112, 65, 0, 5, 172, 0, 0],
            &[0, 112, 65, 0, 5, 172, 0, 0],
            &[0, 112, 65, 0, 5, 172, 0, 0],
            &[0, 112, 65, 0, 5, 172, 0, 0],
            &[0, 112, 65, 0, 5, 172, 0, 0],
            &[0, 0, 0, 0, 7, 172, 0, 0],
            &[0, 68, 153, 158, 163, 21, 0, 0],
        ]),
        // letter: 'Ĵ' / 0x134
        'Ĵ' => Some(&[
            &[0, 0, 0, 0, 45, 40, 0, 0],
            &[0, 0, 0, 24, 152, 157, 19, 0],
            &[0, 0, 0, 77, 7, 12, 74, 0],
            &[0, 0, 0, 0, 55, 45, 0, 0],
            &[0, 0, 0, 0, 97, 80, 0, 0],
            &[0, 0, 0, 0, 97, 80, 0, 0],
            &[0, 0, 0, 0, 97, 80, 0, 0],
            &[0, 0, 0, 0, 97, 80, 0, 0],
            &[0, 0, 0, 0, 97, 80, 0, 0],
            &[0, 0, 0, 0, 104, 78, 0, 0],
            &[0, 0, 0, 0, 153, 47, 0, 0],
            &[0, 116, 146, 159, 161, 0, 0, 0],
            &[0, 0, 19, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĵ' / 0x135
        'ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 21, 0, 0],
            &[0, 0, 0, 3, 157, 164, 13, 0],
            &[0, 0, 0, 82, 34, 21, 97, 0],
            &[0, 0, 34, 98, 98, 51, 0, 0],
            &[0, 0, 10, 38, 85, 134, 0, 0],
            &[0, 0, 0, 0, 44, 134, 0, 0],
            &[0, 0, 0, 0, 44, 134, 0, 0],
            &[0, 0, 0, 0, 44, 134, 0, 0],
            &[0, 0, 0, 0, 44, 134, 0, 0],
            &[0, 0, 0, 0, 44, 134, 0, 0],
            &[0, 0, 0, 0, 45, 133, 0, 0],
            &[0, 25, 154, 159, 159, 9, 0, 0],
        ]),
        // letter: 'Ķ' / 0x136
        'Ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 0, 0, 0, 91, 31, 0],
            &[1, 177, 0, 0, 82, 137, 0, 0],
            &[1, 177, 0, 55, 166, 2, 0, 0],
            &[1, 177, 34, 183, 10, 0, 0, 0],
            &[1, 187, 184, 136, 0, 0, 0, 0],
            &[1, 228, 29, 147, 53, 0, 0, 0],
            &[1, 177, 0, 12, 181, 7, 0, 0],
            &[1, 177, 0, 0, 66, 135, 0, 0],
            &[1, 177, 0, 0, 0, 146, 55, 0],
            &[0, 0, 0, 38, 9, 0, 0, 0],
            &[0, 0, 2, 79, 0, 0, 0, 0],
        ]),
        // letter: 'ķ' / 0x137
        'ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 5, 0, 0, 0, 0, 0],
            &[0, 134, 44, 0, 0, 0, 0, 0],
            &[0, 134, 44, 0, 0, 0, 0, 0],
            &[0, 134, 44, 0, 0, 53, 30, 0],
            &[0, 134, 44, 0, 61, 155, 2, 0],
            &[0, 134, 43, 53, 163, 4, 0, 0],
            &[0, 134, 83, 217, 51, 0, 0, 0],
            &[0, 134, 153, 18, 177, 9, 0, 0],
            &[0, 134, 44, 0, 55, 148, 0, 0],
            &[0, 134, 44, 0, 0, 122, 84, 0],
            &[0, 0, 0, 20, 27, 0, 0, 0],
            &[0, 0, 0, 81, 0, 0, 0, 0],
        ]),
        // letter: 'ĸ' / 0x138
        'ĸ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 51, 16, 0, 0, 53, 30, 0],
            &[0, 134, 44, 0, 61, 155, 2, 0],
            &[0, 134, 43, 53, 163, 4, 0, 0],
            &[0, 134, 83, 217, 51, 0, 0, 0],
            &[0, 134, 153, 18, 177, 9, 0, 0],
            &[0, 134, 44, 0, 55, 148, 0, 0],
            &[0, 134, 44, 0, 0, 122, 84, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĺ' / 0x139
        'Ĺ' => Some(&[
            &[0, 0, 28, 54, 0, 0, 0, 0],
            &[0, 5, 165, 27, 0, 0, 0, 0],
            &[0, 57, 29, 0, 0, 0, 0, 0],
            &[0, 58, 42, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 192, 165, 165, 165, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĺ' / 0x13a
        'ĺ' => Some(&[
            &[0, 0, 0, 4, 169, 22, 0, 0],
            &[0, 0, 0, 124, 46, 0, 0, 0],
            &[0, 23, 30, 30, 5, 0, 0, 0],
            &[0, 75, 110, 192, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 137, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ļ' / 0x13b
        'Ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 42, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 192, 165, 165, 165, 25, 0],
            &[0, 0, 0, 21, 26, 0, 0, 0],
            &[0, 0, 0, 81, 0, 0, 0, 0],
        ]),
        // letter: 'ļ' / 0x13c
        'ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 23, 30, 30, 5, 0, 0, 0],
            &[0, 75, 110, 192, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 137, 53, 0],
            &[0, 0, 0, 35, 12, 0, 0, 0],
            &[0, 0, 0, 81, 0, 0, 0, 0],
        ]),
        // letter: 'Ľ' / 0x13d
        'Ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 42, 0, 0, 99, 0, 0],
            &[0, 103, 74, 0, 11, 137, 0, 0],
            &[0, 103, 74, 0, 20, 20, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 192, 165, 165, 165, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ľ' / 0x13e
        'ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 23, 30, 30, 5, 9, 11, 0],
            &[0, 75, 110, 192, 41, 111, 53, 0],
            &[0, 0, 0, 136, 41, 102, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 137, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 42, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 6, 218, 0, 0],
            &[0, 103, 74, 0, 0, 34, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 74, 0, 0, 0, 0, 0],
            &[0, 103, 192, 165, 165, 165, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŀ' / 0x140
        'ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 30, 30, 5, 0, 0, 0, 0],
            &[75, 110, 192, 41, 0, 0, 0, 0],
            &[0, 0, 136, 41, 0, 0, 0, 0],
            &[0, 0, 136, 41, 0, 0, 0, 0],
            &[0, 0, 136, 41, 0, 16, 54, 0],
            &[0, 0, 136, 41, 0, 51, 139, 0],
            &[0, 0, 136, 41, 0, 0, 0, 0],
            &[0, 0, 136, 41, 0, 0, 0, 0],
            &[0, 0, 136, 41, 0, 0, 0, 0],
            &[120, 142, 206, 166, 137, 53, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ł' / 0x141
        'Ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 101, 0, 0, 0, 0, 0],
            &[0, 0, 178, 0, 0, 0, 0, 0],
            &[0, 0, 178, 0, 0, 0, 0, 0],
            &[0, 0, 178, 5, 52, 0, 0, 0],
            &[0, 0, 238, 165, 57, 0, 0, 0],
            &[5, 145, 255, 4, 0, 0, 0, 0],
            &[0, 20, 178, 0, 0, 0, 0, 0],
            &[0, 0, 178, 0, 0, 0, 0, 0],
            &[0, 0, 215, 165, 165, 165, 105, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ł' / 0x142
        'ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 23, 30, 30, 5, 0, 0, 0],
            &[0, 75, 110, 192, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 87, 94, 0, 0],
            &[0, 0, 8, 255, 170, 8, 0, 0],
            &[0, 6, 157, 180, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 0, 0, 136, 41, 0, 0, 0],
            &[0, 120, 142, 206, 166, 137, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ń' / 0x143
        'Ń' => Some(&[
            &[0, 0, 0, 0, 55, 26, 0, 0],
            &[0, 0, 0, 35, 159, 2, 0, 0],
            &[0, 0, 0, 81, 5, 0, 0, 0],
            &[0, 141, 5, 0, 0, 68, 31, 0],
            &[1, 219, 101, 0, 0, 119, 55, 0],
            &[1, 166, 165, 4, 0, 119, 55, 0],
            &[1, 170, 74, 96, 0, 119, 55, 0],
            &[1, 170, 0, 167, 3, 119, 55, 0],
            &[1, 170, 0, 79, 91, 119, 55, 0],
            &[1, 170, 0, 1, 168, 119, 55, 0],
            &[1, 170, 0, 0, 85, 196, 55, 0],
            &[1, 170, 0, 0, 2, 214, 55, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ń' / 0x144
        'ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 19, 0, 0],
            &[0, 0, 0, 11, 174, 18, 0, 0],
            &[0, 0, 0, 93, 27, 0, 0, 0],
            &[0, 57, 25, 108, 108, 18, 0, 0],
            &[0, 183, 133, 37, 60, 190, 1, 0],
            &[0, 212, 7, 0, 0, 150, 29, 0],
            &[0, 182, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ņ' / 0x145
        'Ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 5, 0, 0, 68, 31, 0],
            &[1, 219, 101, 0, 0, 119, 55, 0],
            &[1, 166, 165, 4, 0, 119, 55, 0],
            &[1, 170, 74, 96, 0, 119, 55, 0],
            &[1, 170, 0, 167, 3, 119, 55, 0],
            &[1, 170, 0, 79, 91, 119, 55, 0],
            &[1, 170, 0, 1, 168, 119, 55, 0],
            &[1, 170, 0, 0, 85, 196, 55, 0],
            &[1, 170, 0, 0, 2, 214, 55, 0],
            &[0, 0, 0, 48, 0, 0, 0, 0],
            &[0, 0, 25, 56, 0, 0, 0, 0],
        ]),
        // letter: 'ņ' / 0x146
        'ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 25, 108, 108, 18, 0, 0],
            &[0, 183, 133, 37, 60, 190, 1, 0],
            &[0, 212, 7, 0, 0, 150, 29, 0],
            &[0, 182, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 0, 0, 44, 3, 0, 0, 0],
            &[0, 0, 10, 71, 0, 0, 0, 0],
        ]),
        // letter: 'Ň' / 0x147
        'Ň' => Some(&[
            &[0, 4, 51, 0, 24, 33, 0, 0],
            &[0, 0, 110, 91, 153, 6, 0, 0],
            &[0, 0, 0, 120, 23, 0, 0, 0],
            &[0, 141, 5, 0, 0, 68, 31, 0],
            &[1, 219, 101, 0, 0, 119, 55, 0],
            &[1, 166, 165, 4, 0, 119, 55, 0],
            &[1, 170, 74, 96, 0, 119, 55, 0],
            &[1, 170, 0, 167, 3, 119, 55, 0],
            &[1, 170, 0, 79, 91, 119, 55, 0],
            &[1, 170, 0, 1, 168, 119, 55, 0],
            &[1, 170, 0, 0, 85, 196, 55, 0],
            &[1, 170, 0, 0, 2, 214, 55, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ň' / 0x148
        'ň' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 24, 0, 4, 22, 0, 0],
            &[0, 0, 125, 54, 146, 26, 0, 0],
            &[0, 0, 3, 156, 54, 0, 0, 0],
            &[0, 57, 25, 108, 108, 18, 0, 0],
            &[0, 183, 133, 37, 60, 190, 1, 0],
            &[0, 212, 7, 0, 0, 150, 29, 0],
            &[0, 182, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 138, 39, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŉ' / 0x149
        'ŉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 16, 0, 0, 0, 0, 0],
            &[0, 190, 1, 0, 0, 0, 0, 0],
            &[19, 145, 57, 11, 101, 111, 22, 0],
            &[25, 28, 163, 134, 43, 57, 192, 2],
            &[0, 0, 170, 47, 0, 0, 150, 30],
            &[0, 0, 170, 11, 0, 0, 138, 39],
            &[0, 0, 170, 7, 0, 0, 138, 39],
            &[0, 0, 170, 7, 0, 0, 138, 39],
            &[0, 0, 170, 7, 0, 0, 138, 39],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 5, 0, 0, 68, 31, 0],
            &[1, 219, 101, 0, 0, 119, 55, 0],
            &[1, 166, 165, 4, 0, 119, 55, 0],
            &[1, 170, 74, 96, 0, 119, 55, 0],
            &[1, 170, 0, 167, 3, 119, 55, 0],
            &[1, 170, 0, 79, 91, 119, 55, 0],
            &[1, 170, 0, 1, 168, 119, 55, 0],
            &[1, 170, 0, 0, 85, 196, 55, 0],
            &[1, 170, 0, 0, 2, 255, 90, 0],
            &[0, 0, 0, 0, 0, 146, 35, 0],
            &[0, 0, 0, 21, 32, 0, 0, 0],
        ]),
        // letter: 'ŋ' / 0x14b
        'ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 25, 108, 108, 18, 0, 0],
            &[0, 183, 133, 37, 60, 190, 1, 0],
            &[0, 212, 7, 0, 0, 150, 29, 0],
            &[0, 182, 0, 0, 0, 138, 39, 0],
            &[0, 178, 0, 0, 0, 211, 61, 0],
            &[0, 178, 0, 0, 0, 255, 79, 0],
            &[0, 178, 0, 0, 0, 255, 79, 0],
            &[0, 0, 0, 0, 0, 138, 39, 0],
            &[0, 0, 0, 38, 156, 130, 0, 0],
        ]),
        // letter: 'Ō' / 0x14c
        'Ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 69, 69, 69, 17, 0, 0],
            &[0, 4, 77, 77, 77, 19, 0, 0],
            &[0, 5, 112, 159, 138, 21, 0, 0],
            &[0, 148, 79, 0, 38, 181, 3, 0],
            &[23, 161, 0, 0, 0, 116, 76, 0],
            &[77, 103, 0, 0, 0, 54, 131, 0],
            &[97, 83, 0, 0, 0, 33, 150, 0],
            &[90, 91, 0, 0, 0, 39, 141, 0],
            &[50, 133, 0, 0, 0, 79, 102, 0],
            &[1, 187, 13, 0, 0, 173, 24, 0],
            &[0, 49, 182, 142, 175, 86, 0, 0],
            &[0, 0, 1, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ō' / 0x14d
        'ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 22, 22, 22, 5, 0, 0],
            &[0, 6, 124, 124, 124, 31, 0, 0],
            &[0, 0, 69, 120, 90, 6, 0, 0],
            &[0, 132, 113, 33, 74, 169, 1, 0],
            &[17, 172, 0, 0, 0, 119, 66, 0],
            &[54, 126, 0, 0, 0, 70, 109, 0],
            &[46, 133, 0, 0, 0, 82, 99, 0],
            &[4, 184, 5, 0, 0, 158, 29, 0],
            &[0, 61, 171, 128, 162, 100, 0, 0],
            &[0, 0, 1, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 154, 23, 137, 32, 0, 0],
            &[0, 0, 38, 113, 58, 0, 0, 0],
            &[0, 5, 112, 159, 138, 21, 0, 0],
            &[0, 148, 79, 0, 38, 181, 3, 0],
            &[23, 161, 0, 0, 0, 116, 76, 0],
            &[77, 103, 0, 0, 0, 54, 131, 0],
            &[97, 83, 0, 0, 0, 33, 150, 0],
            &[90, 91, 0, 0, 0, 39, 141, 0],
            &[50, 133, 0, 0, 0, 79, 102, 0],
            &[1, 187, 13, 0, 0, 173, 24, 0],
            &[0, 49, 182, 142, 175, 86, 0, 0],
            &[0, 0, 1, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŏ' / 0x14f
        'ŏ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 120, 0, 95, 32, 0, 0],
            &[0, 0, 71, 137, 100, 0, 0, 0],
            &[0, 0, 69, 120, 90, 6, 0, 0],
            &[0, 132, 113, 33, 74, 169, 1, 0],
            &[17, 172, 0, 0, 0, 119, 66, 0],
            &[54, 126, 0, 0, 0, 70, 109, 0],
            &[46, 133, 0, 0, 0, 82, 99, 0],
            &[4, 184, 5, 0, 0, 158, 29, 0],
            &[0, 61, 171, 128, 162, 100, 0, 0],
            &[0, 0, 1, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ő' / 0x150
        'Ő' => Some(&[
            &[0, 0, 2, 72, 10, 65, 0, 0],
            &[0, 0, 106, 74, 126, 53, 0, 0],
            &[0, 3, 75, 10, 67, 0, 0, 0],
            &[0, 5, 112, 159, 138, 21, 0, 0],
            &[0, 148, 79, 0, 38, 181, 3, 0],
            &[23, 161, 0, 0, 0, 116, 76, 0],
            &[77, 103, 0, 0, 0, 54, 131, 0],
            &[97, 83, 0, 0, 0, 33, 150, 0],
            &[90, 91, 0, 0, 0, 39, 141, 0],
            &[50, 133, 0, 0, 0, 79, 102, 0],
            &[1, 187, 13, 0, 0, 173, 24, 0],
            &[0, 49, 182, 142, 175, 86, 0, 0],
            &[0, 0, 1, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ő' / 0x151
        'ő' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 38, 2, 35, 0, 0],
            &[0, 0, 78, 108, 103, 82, 0, 0],
            &[0, 3, 105, 11, 98, 0, 0, 0],
            &[0, 0, 69, 120, 90, 6, 0, 0],
            &[0, 132, 113, 33, 74, 169, 1, 0],
            &[17, 172, 0, 0, 0, 119, 66, 0],
            &[54, 126, 0, 0, 0, 70, 109, 0],
            &[46, 133, 0, 0, 0, 82, 99, 0],
            &[4, 184, 5, 0, 0, 158, 29, 0],
            &[0, 61, 171, 128, 162, 100, 0, 0],
            &[0, 0, 1, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Œ' / 0x152
        'Œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 125, 156, 145, 145, 120, 0],
            &[0, 181, 50, 86, 100, 14, 11, 0],
            &[47, 142, 0, 83, 91, 0, 0, 0],
            &[85, 92, 0, 83, 91, 0, 0, 0],
            &[103, 76, 0, 83, 193, 159, 101, 0],
            &[97, 83, 0, 83, 91, 0, 0, 0],
            &[64, 119, 0, 83, 91, 0, 0, 0],
            &[8, 188, 3, 83, 91, 0, 0, 0],
            &[0, 76, 180, 185, 195, 162, 134, 0],
            &[0, 0, 0, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'œ' / 0x153
        'œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 63, 110, 10, 49, 108, 7, 0],
            &[50, 136, 53, 144, 137, 67, 123, 0],
            &[133, 28, 0, 178, 50, 0, 151, 0],
            &[158, 2, 0, 155, 150, 138, 161, 0],
            &[155, 7, 0, 162, 40, 0, 0, 0],
            &[111, 54, 0, 187, 88, 0, 0, 0],
            &[18, 175, 152, 79, 160, 128, 96, 0],
            &[0, 4, 14, 0, 1, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŕ' / 0x154
        'Ŕ' => Some(&[
            &[0, 0, 0, 0, 61, 21, 0, 0],
            &[0, 0, 0, 45, 151, 0, 0, 0],
            &[0, 0, 0, 84, 3, 0, 0, 0],
            &[0, 107, 145, 140, 106, 17, 0, 0],
            &[0, 178, 11, 19, 69, 195, 7, 0],
            &[0, 178, 0, 0, 0, 132, 57, 0],
            &[0, 178, 0, 0, 0, 148, 40, 0],
            &[0, 181, 75, 83, 130, 127, 0, 0],
            &[0, 181, 83, 88, 174, 0, 0, 0],
            &[0, 178, 0, 0, 116, 79, 0, 0],
            &[0, 178, 0, 0, 5, 182, 9, 0],
            &[0, 178, 0, 0, 0, 69, 129, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŕ' / 0x155
        'ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 39, 0, 0],
            &[0, 0, 0, 0, 120, 83, 0, 0],
            &[0, 0, 0, 34, 86, 0, 0, 0],
            &[0, 80, 98, 58, 40, 117, 80, 0],
            &[0, 27, 47, 191, 139, 43, 45, 0],
            &[0, 0, 0, 230, 4, 0, 0, 0],
            &[0, 0, 0, 183, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[3, 133, 146, 224, 143, 109, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŗ' / 0x156
        'Ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 145, 140, 106, 17, 0, 0],
            &[0, 178, 11, 19, 69, 195, 7, 0],
            &[0, 178, 0, 0, 0, 132, 57, 0],
            &[0, 178, 0, 0, 0, 148, 40, 0],
            &[0, 181, 75, 83, 130, 127, 0, 0],
            &[0, 181, 83, 88, 174, 0, 0, 0],
            &[0, 178, 0, 0, 116, 79, 0, 0],
            &[0, 178, 0, 0, 5, 182, 9, 0],
            &[0, 178, 0, 0, 0, 69, 129, 0],
            &[0, 0, 0, 21, 26, 0, 0, 0],
            &[0, 0, 0, 81, 0, 0, 0, 0],
        ]),
        // letter: 'ŗ' / 0x157
        'ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 80, 98, 58, 40, 117, 80, 0],
            &[0, 27, 47, 191, 139, 43, 45, 0],
            &[0, 0, 0, 230, 4, 0, 0, 0],
            &[0, 0, 0, 183, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[3, 133, 146, 224, 143, 109, 0, 0],
            &[0, 0, 0, 47, 0, 0, 0, 0],
            &[0, 0, 64, 17, 0, 0, 0, 0],
        ]),
        // letter: 'Ř' / 0x158
        'Ř' => Some(&[
            &[0, 8, 46, 0, 29, 27, 0, 0],
            &[0, 0, 122, 87, 149, 3, 0, 0],
            &[0, 0, 2, 126, 16, 0, 0, 0],
            &[0, 107, 145, 140, 106, 17, 0, 0],
            &[0, 178, 11, 19, 69, 195, 7, 0],
            &[0, 178, 0, 0, 0, 132, 57, 0],
            &[0, 178, 0, 0, 0, 148, 40, 0],
            &[0, 181, 75, 83, 130, 127, 0, 0],
            &[0, 181, 83, 88, 174, 0, 0, 0],
            &[0, 178, 0, 0, 116, 79, 0, 0],
            &[0, 178, 0, 0, 5, 182, 9, 0],
            &[0, 178, 0, 0, 0, 69, 129, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ř' / 0x159
        'ř' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 0, 0, 27, 0, 0],
            &[0, 0, 51, 124, 83, 94, 0, 0],
            &[0, 0, 0, 89, 124, 0, 0, 0],
            &[0, 80, 98, 58, 40, 117, 80, 0],
            &[0, 27, 47, 191, 139, 43, 45, 0],
            &[0, 0, 0, 230, 4, 0, 0, 0],
            &[0, 0, 0, 183, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[3, 133, 146, 224, 143, 109, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ś' / 0x15a
        'Ś' => Some(&[
            &[0, 0, 0, 0, 38, 44, 0, 0],
            &[0, 0, 0, 13, 168, 15, 0, 0],
            &[0, 0, 0, 68, 18, 0, 0, 0],
            &[0, 3, 102, 161, 157, 111, 4, 0],
            &[0, 132, 105, 5, 3, 54, 0, 0],
            &[0, 184, 0, 0, 0, 0, 0, 0],
            &[0, 140, 95, 0, 0, 0, 0, 0],
            &[0, 2, 118, 181, 96, 6, 0, 0],
            &[0, 0, 0, 5, 93, 181, 10, 0],
            &[0, 0, 0, 0, 0, 120, 72, 0],
            &[0, 0, 0, 0, 0, 151, 43, 0],
            &[0, 158, 151, 136, 173, 116, 0, 0],
            &[0, 0, 9, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ś' / 0x15b
        'ś' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 15, 0, 0],
            &[0, 0, 0, 19, 174, 10, 0, 0],
            &[0, 0, 0, 102, 18, 0, 0, 0],
            &[0, 0, 57, 115, 111, 42, 0, 0],
            &[0, 78, 139, 32, 44, 56, 0, 0],
            &[0, 97, 92, 0, 0, 0, 0, 0],
            &[0, 6, 132, 159, 70, 0, 0, 0],
            &[0, 0, 0, 10, 111, 135, 0, 0],
            &[0, 0, 0, 0, 0, 185, 0, 0],
            &[0, 108, 152, 129, 166, 90, 0, 0],
            &[0, 0, 7, 25, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => Some(&[
            &[0, 0, 0, 56, 29, 0, 0, 0],
            &[0, 0, 43, 143, 158, 7, 0, 0],
            &[0, 0, 83, 1, 23, 62, 0, 0],
            &[0, 3, 102, 161, 157, 111, 4, 0],
            &[0, 132, 105, 5, 3, 54, 0, 0],
            &[0, 184, 0, 0, 0, 0, 0, 0],
            &[0, 140, 95, 0, 0, 0, 0, 0],
            &[0, 2, 118, 181, 96, 6, 0, 0],
            &[0, 0, 0, 5, 93, 181, 10, 0],
            &[0, 0, 0, 0, 0, 120, 72, 0],
            &[0, 0, 0, 0, 0, 151, 43, 0],
            &[0, 158, 151, 136, 173, 116, 0, 0],
            &[0, 0, 9, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŝ' / 0x15d
        'ŝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 2, 0, 0, 0],
            &[0, 0, 53, 170, 116, 0, 0, 0],
            &[0, 5, 109, 1, 82, 35, 0, 0],
            &[0, 0, 57, 115, 111, 42, 0, 0],
            &[0, 78, 139, 32, 44, 56, 0, 0],
            &[0, 97, 92, 0, 0, 0, 0, 0],
            &[0, 6, 132, 159, 70, 0, 0, 0],
            &[0, 0, 0, 10, 111, 135, 0, 0],
            &[0, 0, 0, 0, 0, 185, 0, 0],
            &[0, 108, 152, 129, 166, 90, 0, 0],
            &[0, 0, 7, 25, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ş' / 0x15e
        'Ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 102, 161, 157, 111, 4, 0],
            &[0, 132, 105, 5, 3, 54, 0, 0],
            &[0, 184, 0, 0, 0, 0, 0, 0],
            &[0, 140, 95, 0, 0, 0, 0, 0],
            &[0, 2, 118, 181, 96, 6, 0, 0],
            &[0, 0, 0, 5, 93, 181, 10, 0],
            &[0, 0, 0, 0, 0, 120, 72, 0],
            &[0, 0, 0, 0, 0, 151, 43, 0],
            &[0, 158, 151, 140, 173, 116, 0, 0],
            &[0, 0, 13, 179, 6, 0, 0, 0],
            &[0, 0, 72, 139, 61, 0, 0, 0],
        ]),
        // letter: 'ş' / 0x15f
        'ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 115, 111, 42, 0, 0],
            &[0, 78, 139, 32, 44, 56, 0, 0],
            &[0, 97, 92, 0, 0, 0, 0, 0],
            &[0, 6, 132, 159, 70, 0, 0, 0],
            &[0, 0, 0, 10, 111, 135, 0, 0],
            &[0, 0, 0, 0, 0, 185, 0, 0],
            &[0, 108, 152, 134, 166, 90, 0, 0],
            &[0, 0, 9, 182, 10, 0, 0, 0],
            &[0, 0, 68, 138, 68, 0, 0, 0],
        ]),
        // letter: 'Š' / 0x160
        'Š' => Some(&[
            &[0, 0, 55, 0, 9, 48, 0, 0],
            &[0, 0, 73, 112, 154, 21, 0, 0],
            &[0, 0, 0, 96, 48, 0, 0, 0],
            &[0, 3, 102, 161, 157, 111, 4, 0],
            &[0, 132, 105, 5, 3, 54, 0, 0],
            &[0, 184, 0, 0, 0, 0, 0, 0],
            &[0, 140, 95, 0, 0, 0, 0, 0],
            &[0, 2, 118, 181, 96, 6, 0, 0],
            &[0, 0, 0, 5, 93, 181, 10, 0],
            &[0, 0, 0, 0, 0, 120, 72, 0],
            &[0, 0, 0, 0, 0, 151, 43, 0],
            &[0, 158, 151, 136, 173, 116, 0, 0],
            &[0, 0, 9, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'š' / 0x161
        'š' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 20, 0, 8, 18, 0, 0],
            &[0, 0, 137, 46, 150, 17, 0, 0],
            &[0, 0, 8, 166, 39, 0, 0, 0],
            &[0, 0, 57, 115, 111, 42, 0, 0],
            &[0, 78, 139, 32, 44, 56, 0, 0],
            &[0, 97, 92, 0, 0, 0, 0, 0],
            &[0, 6, 132, 159, 70, 0, 0, 0],
            &[0, 0, 0, 10, 111, 135, 0, 0],
            &[0, 0, 0, 0, 0, 185, 0, 0],
            &[0, 108, 152, 129, 166, 90, 0, 0],
            &[0, 0, 7, 25, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ţ' / 0x162
        'Ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[61, 145, 145, 145, 145, 145, 92, 0],
            &[6, 14, 14, 182, 14, 14, 9, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 182, 0, 0, 0, 0],
            &[0, 0, 3, 156, 4, 0, 0, 0],
            &[0, 0, 72, 139, 61, 0, 0, 0],
        ]),
        // letter: 'ţ' / 0x163
        'ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 0],
            &[0, 0, 42, 113, 0, 0, 0, 0],
            &[11, 84, 136, 168, 98, 98, 0, 0],
            &[8, 55, 105, 144, 55, 55, 0, 0],
            &[0, 0, 64, 113, 0, 0, 0, 0],
            &[0, 0, 64, 113, 0, 0, 0, 0],
            &[0, 0, 64, 114, 0, 0, 0, 0],
            &[0, 0, 47, 143, 0, 0, 0, 0],
            &[0, 0, 1, 171, 148, 135, 5, 0],
            &[0, 0, 0, 5, 175, 17, 0, 0],
            &[0, 0, 0, 76, 141, 57, 0, 0],
        ]),
        // letter: 'Ť' / 0x164
        'Ť' => Some(&[
            &[0, 8, 46, 0, 29, 27, 0, 0],
            &[0, 0, 122, 87, 149, 3, 0, 0],
            &[0, 0, 2, 126, 16, 0, 0, 0],
            &[61, 145, 145, 145, 145, 145, 92, 0],
            &[6, 14, 14, 182, 14, 14, 9, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ť' / 0x165
        'ť' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 0, 0],
            &[0, 0, 0, 3, 0, 163, 0, 0],
            &[0, 0, 42, 113, 20, 71, 0, 0],
            &[11, 84, 136, 168, 98, 98, 0, 0],
            &[8, 55, 105, 144, 55, 55, 0, 0],
            &[0, 0, 64, 113, 0, 0, 0, 0],
            &[0, 0, 64, 113, 0, 0, 0, 0],
            &[0, 0, 64, 114, 0, 0, 0, 0],
            &[0, 0, 47, 143, 0, 0, 0, 0],
            &[0, 0, 1, 171, 143, 135, 5, 0],
            &[0, 0, 0, 0, 20, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŧ' / 0x166
        'Ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[61, 145, 145, 145, 145, 145, 92, 0],
            &[6, 14, 14, 182, 14, 14, 9, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 106, 153, 255, 153, 138, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 178, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŧ' / 0x167
        'ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 0],
            &[0, 0, 42, 113, 0, 0, 0, 0],
            &[11, 84, 136, 168, 98, 98, 0, 0],
            &[8, 55, 105, 144, 55, 55, 0, 0],
            &[0, 0, 64, 113, 0, 0, 0, 0],
            &[22, 138, 202, 251, 138, 138, 11, 0],
            &[0, 0, 64, 114, 0, 0, 0, 0],
            &[0, 0, 47, 143, 0, 0, 0, 0],
            &[0, 0, 1, 171, 143, 135, 5, 0],
            &[0, 0, 0, 0, 20, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ũ' / 0x168
        'Ũ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 156, 135, 45, 111, 0, 0],
            &[0, 48, 12, 28, 106, 16, 0, 0],
            &[0, 100, 0, 0, 0, 69, 31, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[0, 179, 0, 0, 0, 135, 44, 0],
            &[0, 179, 14, 0, 1, 185, 6, 0],
            &[0, 53, 184, 139, 175, 85, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ũ' / 0x169
        'ũ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 145, 102, 5, 92, 0, 0],
            &[0, 63, 24, 61, 147, 35, 0, 0],
            &[0, 68, 0, 0, 0, 58, 9, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 162, 24, 0],
            &[0, 185, 3, 0, 6, 221, 24, 0],
            &[0, 85, 175, 128, 144, 147, 24, 0],
            &[0, 0, 5, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ū' / 0x16a
        'Ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 69, 69, 69, 17, 0, 0],
            &[0, 4, 77, 77, 77, 19, 0, 0],
            &[0, 100, 0, 0, 0, 69, 31, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[0, 179, 0, 0, 0, 135, 44, 0],
            &[0, 179, 14, 0, 1, 185, 6, 0],
            &[0, 53, 184, 139, 175, 85, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ū' / 0x16b
        'ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 22, 22, 22, 5, 0, 0],
            &[0, 6, 124, 124, 124, 31, 0, 0],
            &[0, 68, 0, 0, 0, 58, 9, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 162, 24, 0],
            &[0, 185, 3, 0, 6, 221, 24, 0],
            &[0, 85, 175, 128, 144, 147, 24, 0],
            &[0, 0, 5, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 154, 23, 137, 32, 0, 0],
            &[0, 0, 38, 113, 58, 0, 0, 0],
            &[0, 100, 0, 0, 0, 69, 31, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[0, 179, 0, 0, 0, 135, 44, 0],
            &[0, 179, 14, 0, 1, 185, 6, 0],
            &[0, 53, 184, 139, 175, 85, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŭ' / 0x16d
        'ŭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 120, 0, 95, 32, 0, 0],
            &[0, 0, 71, 137, 100, 0, 0, 0],
            &[0, 68, 0, 0, 0, 58, 9, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 162, 24, 0],
            &[0, 185, 3, 0, 6, 221, 24, 0],
            &[0, 85, 175, 128, 144, 147, 24, 0],
            &[0, 0, 5, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ů' / 0x16e
        'Ů' => Some(&[
            &[0, 0, 84, 125, 109, 0, 0, 0],
            &[0, 0, 133, 12, 138, 0, 0, 0],
            &[0, 0, 24, 115, 43, 0, 0, 0],
            &[0, 100, 0, 0, 0, 69, 31, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[0, 179, 0, 0, 0, 135, 44, 0],
            &[0, 179, 14, 0, 1, 185, 6, 0],
            &[0, 53, 184, 139, 175, 85, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ů' / 0x16f
        'ů' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 125, 83, 0, 0, 0],
            &[0, 0, 135, 0, 135, 0, 0, 0],
            &[0, 0, 48, 127, 73, 0, 0, 0],
            &[0, 68, 0, 0, 0, 58, 9, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 162, 24, 0],
            &[0, 185, 3, 0, 6, 221, 24, 0],
            &[0, 85, 175, 128, 144, 147, 24, 0],
            &[0, 0, 5, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ű' / 0x170
        'Ű' => Some(&[
            &[0, 0, 2, 72, 10, 65, 0, 0],
            &[0, 0, 106, 74, 126, 53, 0, 0],
            &[0, 3, 75, 10, 67, 0, 0, 0],
            &[0, 100, 0, 0, 0, 69, 31, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[0, 179, 0, 0, 0, 135, 44, 0],
            &[0, 179, 14, 0, 1, 185, 6, 0],
            &[0, 53, 184, 139, 175, 85, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ű' / 0x171
        'ű' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 38, 2, 35, 0, 0],
            &[0, 0, 78, 108, 103, 82, 0, 0],
            &[0, 3, 105, 11, 98, 0, 0, 0],
            &[0, 68, 0, 0, 0, 58, 9, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 162, 24, 0],
            &[0, 185, 3, 0, 6, 221, 24, 0],
            &[0, 85, 175, 128, 144, 147, 24, 0],
            &[0, 0, 5, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ų' / 0x172
        'Ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 0, 0, 0, 69, 31, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[1, 177, 0, 0, 0, 122, 55, 0],
            &[0, 179, 0, 0, 0, 135, 44, 0],
            &[0, 179, 14, 0, 1, 185, 6, 0],
            &[0, 53, 184, 139, 220, 85, 0, 0],
            &[0, 0, 2, 143, 45, 0, 0, 0],
            &[0, 0, 0, 117, 82, 0, 0, 0],
        ]),
        // letter: 'ų' / 0x173
        'ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 0, 0, 0, 58, 9, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 153, 24, 0],
            &[0, 178, 0, 0, 0, 162, 24, 0],
            &[0, 185, 3, 0, 6, 221, 24, 0],
            &[0, 85, 175, 128, 144, 158, 24, 0],
            &[0, 0, 5, 17, 24, 128, 0, 0],
            &[0, 0, 0, 0, 23, 127, 36, 0],
        ]),
        // letter: 'Ŵ' / 0x174
        'Ŵ' => Some(&[
            &[0, 0, 0, 78, 7, 0, 0, 0],
            &[0, 0, 91, 128, 134, 0, 0, 0],
            &[0, 8, 76, 0, 55, 30, 0, 0],
            &[82, 13, 0, 0, 0, 0, 95, 0],
            &[117, 50, 0, 0, 0, 2, 166, 0],
            &[82, 84, 0, 75, 0, 30, 138, 0],
            &[47, 118, 1, 212, 38, 65, 104, 0],
            &[13, 151, 43, 130, 98, 99, 70, 0],
            &[0, 164, 102, 37, 135, 132, 36, 0],
            &[0, 159, 138, 0, 137, 154, 5, 0],
            &[0, 149, 136, 0, 116, 166, 0, 0],
            &[0, 129, 116, 0, 58, 186, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŵ' / 0x175
        'ŵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 2, 0, 0, 0],
            &[0, 0, 51, 170, 118, 0, 0, 0],
            &[0, 5, 109, 1, 80, 37, 0, 0],
            &[63, 0, 0, 0, 0, 0, 62, 0],
            &[144, 22, 0, 40, 2, 0, 163, 0],
            &[92, 75, 4, 218, 56, 14, 149, 0],
            &[40, 128, 64, 115, 131, 64, 101, 0],
            &[1, 166, 139, 20, 159, 114, 51, 0],
            &[0, 156, 152, 0, 130, 171, 6, 0],
            &[0, 133, 119, 0, 55, 199, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŷ' / 0x176
        'Ŷ' => Some(&[
            &[0, 0, 0, 78, 7, 0, 0, 0],
            &[0, 0, 91, 128, 134, 0, 0, 0],
            &[0, 8, 76, 0, 55, 30, 0, 0],
            &[48, 61, 0, 0, 0, 29, 79, 0],
            &[8, 180, 1, 0, 0, 146, 39, 0],
            &[0, 102, 81, 0, 28, 150, 0, 0],
            &[0, 4, 171, 1, 141, 31, 0, 0],
            &[0, 0, 90, 107, 139, 0, 0, 0],
            &[0, 0, 2, 209, 24, 0, 0, 0],
            &[0, 0, 0, 171, 0, 0, 0, 0],
            &[0, 0, 0, 171, 0, 0, 0, 0],
            &[0, 0, 0, 171, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŷ' / 0x177
        'ŷ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 35, 0, 0, 0, 0],
            &[0, 0, 60, 171, 108, 0, 0, 0],
            &[0, 8, 107, 0, 87, 30, 0, 0],
            &[10, 57, 0, 0, 0, 36, 34, 0],
            &[0, 174, 0, 0, 0, 154, 25, 0],
            &[0, 111, 62, 0, 8, 166, 0, 0],
            &[0, 18, 153, 0, 85, 85, 0, 0],
            &[0, 0, 153, 14, 159, 7, 0, 0],
            &[0, 0, 62, 118, 142, 0, 0, 0],
            &[0, 0, 0, 206, 51, 0, 0, 0],
            &[0, 0, 0, 170, 0, 0, 0, 0],
            &[14, 148, 151, 7, 0, 0, 0, 0],
        ]),
        // letter: 'Ÿ' / 0x178
        'Ÿ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 160, 0, 115, 61, 0, 0],
            &[0, 0, 26, 0, 19, 7, 0, 0],
            &[48, 61, 0, 0, 0, 29, 79, 0],
            &[8, 180, 1, 0, 0, 146, 39, 0],
            &[0, 102, 81, 0, 28, 150, 0, 0],
            &[0, 4, 171, 1, 141, 31, 0, 0],
            &[0, 0, 90, 107, 139, 0, 0, 0],
            &[0, 0, 2, 209, 24, 0, 0, 0],
            &[0, 0, 0, 171, 0, 0, 0, 0],
            &[0, 0, 0, 171, 0, 0, 0, 0],
            &[0, 0, 0, 171, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ź' / 0x179
        'Ź' => Some(&[
            &[0, 0, 0, 0, 44, 38, 0, 0],
            &[0, 0, 0, 19, 167, 10, 0, 0],
            &[0, 0, 0, 73, 13, 0, 0, 0],
            &[0, 65, 145, 145, 145, 134, 0, 0],
            &[0, 5, 11, 11, 38, 157, 0, 0],
            &[0, 0, 0, 0, 147, 35, 0, 0],
            &[0, 0, 0, 36, 147, 0, 0, 0],
            &[0, 0, 0, 157, 26, 0, 0, 0],
            &[0, 0, 47, 137, 0, 0, 0, 0],
            &[0, 0, 166, 19, 0, 0, 0, 0],
            &[0, 59, 126, 0, 0, 0, 0, 0],
            &[0, 175, 175, 162, 162, 162, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ź' / 0x17a
        'ź' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 19, 0, 0],
            &[0, 0, 0, 11, 174, 18, 0, 0],
            &[0, 0, 0, 93, 27, 0, 0, 0],
            &[0, 50, 98, 98, 98, 91, 0, 0],
            &[0, 23, 46, 46, 87, 164, 0, 0],
            &[0, 0, 0, 7, 175, 15, 0, 0],
            &[0, 0, 0, 140, 55, 0, 0, 0],
            &[0, 0, 75, 118, 0, 0, 0, 0],
            &[0, 25, 165, 1, 0, 0, 0, 0],
            &[0, 169, 179, 156, 156, 149, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ż' / 0x17b
        'Ż' => Some(&[
            &[0, 0, 0, 3, 0, 0, 0, 0],
            &[0, 0, 0, 176, 63, 0, 0, 0],
            &[0, 0, 0, 9, 0, 0, 0, 0],
            &[0, 65, 145, 145, 145, 134, 0, 0],
            &[0, 5, 11, 11, 38, 157, 0, 0],
            &[0, 0, 0, 0, 147, 35, 0, 0],
            &[0, 0, 0, 36, 147, 0, 0, 0],
            &[0, 0, 0, 157, 26, 0, 0, 0],
            &[0, 0, 47, 137, 0, 0, 0, 0],
            &[0, 0, 166, 19, 0, 0, 0, 0],
            &[0, 59, 126, 0, 0, 0, 0, 0],
            &[0, 175, 175, 162, 162, 162, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ż' / 0x17c
        'ż' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 175, 45, 0, 0, 0],
            &[0, 0, 0, 29, 3, 0, 0, 0],
            &[0, 50, 98, 98, 98, 91, 0, 0],
            &[0, 23, 46, 46, 87, 164, 0, 0],
            &[0, 0, 0, 7, 175, 15, 0, 0],
            &[0, 0, 0, 140, 55, 0, 0, 0],
            &[0, 0, 75, 118, 0, 0, 0, 0],
            &[0, 25, 165, 1, 0, 0, 0, 0],
            &[0, 169, 179, 156, 156, 149, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ž' / 0x17d
        'Ž' => Some(&[
            &[0, 0, 55, 0, 13, 44, 0, 0],
            &[0, 0, 85, 104, 155, 15, 0, 0],
            &[0, 0, 0, 104, 40, 0, 0, 0],
            &[0, 65, 145, 145, 145, 134, 0, 0],
            &[0, 5, 11, 11, 38, 157, 0, 0],
            &[0, 0, 0, 0, 147, 35, 0, 0],
            &[0, 0, 0, 36, 147, 0, 0, 0],
            &[0, 0, 0, 157, 26, 0, 0, 0],
            &[0, 0, 47, 137, 0, 0, 0, 0],
            &[0, 0, 166, 19, 0, 0, 0, 0],
            &[0, 59, 126, 0, 0, 0, 0, 0],
            &[0, 175, 175, 162, 162, 162, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ž' / 0x17e
        'ž' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 24, 0, 4, 22, 0, 0],
            &[0, 0, 125, 54, 146, 26, 0, 0],
            &[0, 0, 3, 156, 54, 0, 0, 0],
            &[0, 50, 98, 98, 98, 91, 0, 0],
            &[0, 23, 46, 46, 87, 164, 0, 0],
            &[0, 0, 0, 7, 175, 15, 0, 0],
            &[0, 0, 0, 140, 55, 0, 0, 0],
            &[0, 0, 75, 118, 0, 0, 0, 0],
            &[0, 25, 165, 1, 0, 0, 0, 0],
            &[0, 169, 179, 156, 156, 149, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ſ' / 0x17f
        'ſ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 39, 4, 0, 0],
            &[0, 0, 126, 146, 112, 36, 0, 0],
            &[0, 9, 176, 0, 0, 0, 0, 0],
            &[0, 27, 148, 0, 0, 0, 0, 0],
            &[0, 27, 147, 0, 0, 0, 0, 0],
            &[0, 27, 147, 0, 0, 0, 0, 0],
            &[0, 27, 147, 0, 0, 0, 0, 0],
            &[0, 27, 147, 0, 0, 0, 0, 0],
            &[0, 27, 147, 0, 0, 0, 0, 0],
            &[0, 27, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        _ => None,
    }
}
