//! Module for letters with the font weight bold and size 22.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 22;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 14;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight bold and font size 19px
#[inline]
pub const fn get_char(c: char) -> Option<&'static [&'static [u8]]> {
    match c {
        // letter: ' ' / 0x20
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '!' / 0x21
        '!' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 127, 144, 144, 43, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 255, 255, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 255, 57, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 194, 255, 255, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 183, 255, 255, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 172, 255, 255, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 160, 255, 255, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 149, 255, 254, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 138, 255, 245, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 64, 127, 117, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 28, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 236, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 251, 255, 255, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 177, 255, 238, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 37, 12, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '"' / 0x22
        '"' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 73, 144, 144, 13, 99, 144, 131, 0, 0, 0, 0],
            &[0, 0, 0, 115, 255, 255, 8, 161, 255, 217, 0, 0, 0, 0],
            &[0, 0, 0, 96, 255, 245, 0, 143, 255, 199, 0, 0, 0, 0],
            &[0, 0, 0, 77, 255, 226, 0, 124, 255, 180, 0, 0, 0, 0],
            &[0, 0, 0, 59, 255, 207, 0, 105, 255, 161, 0, 0, 0, 0],
            &[0, 0, 0, 15, 85, 65, 0, 31, 85, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '#' / 0x23
        '#' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 141, 135, 0, 61, 144, 68, 0, 0, 0],
            &[0, 0, 0, 0, 28, 255, 206, 0, 141, 255, 88, 0, 0, 0],
            &[0, 0, 0, 0, 71, 255, 163, 0, 183, 255, 46, 0, 0, 0],
            &[0, 0, 0, 0, 114, 255, 120, 0, 224, 253, 8, 0, 0, 0],
            &[
                0, 34, 210, 210, 235, 255, 227, 210, 254, 252, 210, 210, 24, 0,
            ],
            &[
                0, 41, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 29, 0,
            ],
            &[0, 2, 14, 14, 237, 250, 18, 104, 255, 142, 14, 14, 1, 0],
            &[0, 0, 0, 22, 255, 213, 0, 136, 255, 93, 0, 0, 0, 0],
            &[
                0, 121, 169, 186, 255, 231, 169, 225, 255, 191, 169, 95, 0, 0,
            ],
            &[
                0, 182, 255, 255, 255, 255, 255, 255, 255, 255, 255, 144, 0, 0,
            ],
            &[0, 36, 51, 172, 255, 114, 63, 255, 230, 51, 51, 28, 0, 0],
            &[0, 0, 0, 187, 255, 43, 53, 255, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 226, 252, 6, 96, 255, 143, 0, 0, 0, 0, 0],
            &[0, 0, 11, 254, 219, 0, 138, 255, 101, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '$' / 0x24
        '$' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 93, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 225, 107, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 166, 247, 255, 255, 247, 192, 78, 0, 0, 0],
            &[0, 0, 9, 219, 255, 255, 255, 255, 255, 255, 112, 0, 0, 0],
            &[0, 0, 85, 255, 255, 81, 224, 105, 57, 137, 24, 0, 0, 0],
            &[0, 0, 110, 255, 251, 10, 222, 93, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 255, 255, 199, 237, 93, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 142, 255, 255, 255, 216, 98, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 199, 255, 255, 255, 208, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 223, 210, 254, 255, 161, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 222, 93, 157, 255, 215, 0, 0, 0],
            &[0, 0, 80, 95, 8, 0, 222, 98, 187, 255, 190, 0, 0, 0],
            &[0, 0, 117, 255, 246, 198, 246, 247, 255, 255, 90, 0, 0, 0],
            &[0, 0, 76, 226, 255, 255, 255, 255, 243, 111, 0, 0, 0, 0],
            &[0, 0, 0, 1, 36, 66, 233, 127, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 222, 93, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '%' / 0x25
        '%' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 100, 169, 144, 28, 0, 0, 13, 144, 134, 1, 0, 0],
            &[0, 117, 255, 251, 255, 224, 7, 0, 125, 255, 139, 0, 0, 0],
            &[0, 219, 253, 27, 170, 255, 75, 16, 239, 243, 21, 0, 0, 0],
            &[0, 247, 231, 0, 121, 255, 102, 130, 255, 134, 0, 0, 0, 0],
            &[0, 221, 253, 25, 168, 255, 97, 242, 241, 18, 0, 0, 0, 0],
            &[0, 121, 255, 250, 255, 226, 145, 255, 129, 0, 0, 0, 0, 0],
            &[0, 1, 103, 171, 146, 52, 244, 239, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 142, 255, 124, 12, 55, 28, 0, 0, 0],
            &[0, 0, 0, 0, 26, 247, 236, 67, 234, 255, 255, 131, 0, 0],
            &[0, 0, 0, 0, 148, 255, 119, 195, 255, 133, 225, 253, 37, 0],
            &[0, 0, 0, 29, 249, 233, 12, 246, 232, 0, 140, 255, 86, 0],
            &[0, 0, 0, 153, 255, 114, 1, 245, 233, 0, 142, 255, 85, 0],
            &[0, 0, 33, 250, 230, 9, 0, 188, 255, 157, 233, 251, 32, 0],
            &[0, 0, 159, 255, 109, 0, 0, 42, 220, 255, 250, 110, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 30, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '&' / 0x26
        '&' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 122, 175, 158, 68, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 188, 255, 255, 255, 255, 85, 0, 0, 0, 0, 0],
            &[0, 0, 48, 255, 255, 83, 160, 255, 193, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 250, 2, 103, 255, 216, 0, 0, 0, 0, 0],
            &[0, 0, 55, 255, 255, 56, 203, 255, 155, 0, 0, 0, 0, 0],
            &[0, 0, 2, 213, 255, 255, 255, 234, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 122, 255, 255, 255, 63, 0, 36, 41, 23, 0, 0],
            &[0, 0, 50, 242, 255, 255, 255, 53, 7, 248, 255, 111, 0, 0],
            &[0, 5, 218, 255, 235, 248, 255, 224, 89, 255, 255, 55, 0, 0],
            &[0, 69, 255, 255, 68, 43, 246, 255, 255, 255, 229, 3, 0, 0],
            &[0, 103, 255, 255, 20, 0, 105, 255, 255, 255, 120, 0, 0, 0],
            &[0, 81, 255, 255, 126, 18, 85, 255, 255, 255, 68, 0, 0, 0],
            &[0, 13, 235, 255, 255, 255, 255, 255, 255, 255, 228, 21, 0, 0],
            &[0, 0, 58, 219, 255, 255, 246, 161, 90, 253, 255, 186, 2, 0],
            &[0, 0, 0, 1, 31, 39, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ''' / 0x27
        '\'' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 88, 144, 143, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 141, 255, 238, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 103, 255, 200, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 255, 182, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 85, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '(' / 0x28
        '(' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 128, 144, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 100, 255, 225, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 236, 255, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 98, 255, 240, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 255, 160, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 246, 255, 101, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 51, 255, 255, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 74, 255, 255, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 255, 255, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 255, 255, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 255, 255, 32, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 69, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 232, 255, 127, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 163, 255, 199, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 195, 255, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 40, 248, 253, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 34, 17, 0, 0, 0, 0],
        ]),
        // letter: ')' / 0x29
        ')' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 136, 144, 42, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 134, 255, 203, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 241, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 153, 255, 207, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 255, 255, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 248, 255, 105, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 202, 255, 157, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 177, 255, 179, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 160, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 168, 255, 187, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 186, 255, 168, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 222, 255, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 255, 255, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 255, 245, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 191, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 254, 250, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 197, 255, 132, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 34, 34, 5, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '*' / 0x2a
        '*' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 88, 71, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 85, 255, 191, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 65, 255, 170, 0, 0, 1, 0, 0, 0],
            &[0, 0, 59, 216, 149, 119, 255, 179, 121, 192, 165, 0, 0, 0],
            &[0, 0, 118, 255, 255, 255, 255, 255, 255, 255, 222, 0, 0, 0],
            &[0, 0, 3, 33, 73, 239, 255, 255, 138, 51, 20, 0, 0, 0],
            &[0, 0, 0, 0, 146, 255, 165, 252, 229, 23, 0, 0, 0, 0],
            &[0, 0, 0, 74, 255, 236, 14, 154, 255, 181, 0, 0, 0, 0],
            &[0, 0, 0, 4, 110, 105, 0, 27, 168, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '+' / 0x2b
        '+' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 63, 47, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 182, 182, 203, 255, 237, 182, 182, 182, 12, 0, 0],
            &[0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 51, 76, 76, 128, 255, 210, 76, 76, 76, 5, 0, 0],
            &[0, 0, 0, 0, 0, 73, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 198, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ',' / 0x2c
        ',' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 149, 149, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 118, 255, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 161, 255, 236, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 161, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 248, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 51, 246, 221, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '-' / 0x2d
        '-' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 69, 69, 69, 69, 69, 38, 0, 0, 0, 0],
            &[0, 0, 0, 38, 255, 255, 255, 255, 255, 140, 0, 0, 0, 0],
            &[0, 0, 0, 38, 255, 255, 255, 255, 255, 140, 0, 0, 0, 0],
            &[0, 0, 0, 4, 26, 26, 26, 26, 26, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '.' / 0x2e
        '.' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 28, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 236, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 251, 255, 255, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 177, 255, 238, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 37, 12, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '/' / 0x2f
        '/' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 159, 164, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 79, 255, 250, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 176, 255, 175, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 251, 255, 78, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 115, 255, 232, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 213, 255, 140, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 43, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 152, 255, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 240, 255, 104, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 255, 247, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 188, 255, 166, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 254, 255, 69, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 127, 255, 226, 2, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 223, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 19, 19, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '0' / 0x30
        '0' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 57, 152, 184, 163, 94, 3, 0, 0, 0, 0],
            &[0, 0, 0, 114, 255, 255, 255, 255, 255, 188, 7, 0, 0, 0],
            &[0, 0, 40, 253, 255, 237, 153, 204, 255, 255, 125, 0, 0, 0],
            &[0, 0, 145, 255, 255, 57, 0, 72, 255, 255, 234, 2, 0, 0],
            &[0, 0, 204, 255, 220, 0, 5, 215, 255, 255, 255, 43, 0, 0],
            &[0, 0, 243, 255, 180, 0, 119, 255, 255, 255, 255, 86, 0, 0],
            &[0, 3, 255, 255, 162, 29, 244, 252, 124, 255, 255, 102, 0, 0],
            &[0, 6, 255, 255, 159, 172, 255, 144, 71, 255, 255, 111, 0, 0],
            &[0, 0, 247, 255, 244, 255, 229, 14, 84, 255, 255, 101, 0, 0],
            &[0, 0, 220, 255, 255, 255, 87, 0, 107, 255, 255, 73, 0, 0],
            &[0, 0, 164, 255, 255, 195, 0, 0, 164, 255, 255, 24, 0, 0],
            &[0, 0, 75, 255, 255, 194, 29, 84, 251, 255, 196, 0, 0, 0],
            &[0, 0, 0, 186, 255, 255, 255, 255, 255, 251, 61, 0, 0, 0],
            &[0, 0, 0, 12, 155, 250, 255, 255, 218, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 38, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '1' / 0x31
        '1' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 139, 144, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 157, 252, 255, 255, 134, 0, 0, 0, 0, 0],
            &[0, 0, 99, 247, 255, 255, 255, 255, 134, 0, 0, 0, 0, 0],
            &[0, 0, 49, 253, 255, 197, 255, 255, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 123, 65, 50, 255, 255, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 255, 255, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 102, 255, 255, 183, 24, 1, 0, 0, 0],
            &[0, 0, 0, 210, 255, 255, 255, 255, 255, 255, 249, 16, 0, 0],
            &[0, 0, 0, 218, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '2' / 0x32
        '2' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 89, 164, 188, 176, 124, 24, 0, 0, 0, 0],
            &[0, 0, 49, 219, 255, 255, 255, 255, 255, 238, 54, 0, 0, 0],
            &[0, 0, 166, 255, 255, 216, 171, 222, 255, 255, 214, 1, 0, 0],
            &[0, 0, 10, 193, 104, 0, 0, 9, 226, 255, 255, 30, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 185, 255, 255, 32, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 10, 236, 255, 232, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 157, 255, 255, 119, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 114, 255, 255, 192, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 107, 255, 255, 199, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 254, 255, 191, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 106, 254, 255, 171, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 106, 254, 255, 220, 93, 91, 91, 91, 91, 49, 0, 0],
            &[0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 255, 139, 0, 0],
            &[0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 255, 139, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '3' / 0x33
        '3' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 109, 166, 188, 173, 115, 16, 0, 0, 0, 0],
            &[0, 0, 95, 248, 255, 255, 255, 255, 255, 227, 31, 0, 0, 0],
            &[0, 0, 36, 247, 253, 198, 163, 217, 255, 255, 150, 0, 0, 0],
            &[0, 0, 0, 76, 35, 0, 0, 20, 251, 255, 188, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 6, 243, 255, 154, 0, 0, 0],
            &[0, 0, 0, 0, 12, 25, 57, 170, 255, 244, 42, 0, 0, 0],
            &[0, 0, 0, 0, 172, 255, 255, 255, 191, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 172, 255, 255, 255, 249, 175, 27, 0, 0, 0],
            &[0, 0, 0, 0, 22, 38, 61, 138, 253, 255, 202, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 196, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 196, 255, 255, 27, 0, 0],
            &[0, 0, 145, 150, 81, 48, 48, 133, 255, 255, 230, 1, 0, 0],
            &[0, 0, 166, 255, 255, 255, 255, 255, 255, 254, 93, 0, 0, 0],
            &[0, 0, 90, 199, 253, 255, 255, 254, 205, 78, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 31, 41, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '4' / 0x34
        '4' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 123, 144, 129, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 89, 255, 255, 228, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 233, 255, 255, 228, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 159, 255, 248, 255, 228, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 255, 229, 175, 255, 228, 0, 0, 0, 0],
            &[0, 0, 0, 8, 219, 255, 90, 176, 255, 228, 0, 0, 0, 0],
            &[0, 0, 0, 135, 255, 193, 0, 185, 255, 228, 0, 0, 0, 0],
            &[0, 0, 46, 251, 251, 45, 0, 186, 255, 228, 0, 0, 0, 0],
            &[0, 2, 201, 255, 150, 8, 8, 189, 255, 229, 8, 5, 0, 0],
            &[
                0, 58, 255, 255, 255, 255, 255, 255, 255, 255, 255, 163, 0, 0,
            ],
            &[
                0, 61, 255, 255, 255, 255, 255, 255, 255, 255, 255, 163, 0, 0,
            ],
            &[0, 11, 48, 48, 48, 48, 48, 199, 255, 233, 48, 30, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 186, 255, 228, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 186, 255, 228, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '5' / 0x35
        '5' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 118, 144, 144, 144, 144, 144, 144, 33, 0, 0, 0],
            &[0, 0, 0, 224, 255, 255, 255, 255, 255, 255, 59, 0, 0, 0],
            &[0, 0, 0, 244, 255, 237, 202, 202, 202, 202, 47, 0, 0, 0],
            &[0, 0, 8, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 255, 255, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 47, 255, 255, 192, 154, 143, 82, 3, 0, 0, 0, 0],
            &[0, 0, 67, 255, 255, 255, 255, 255, 255, 191, 8, 0, 0, 0],
            &[0, 0, 35, 192, 192, 160, 185, 253, 255, 255, 130, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 83, 255, 255, 216, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 243, 255, 242, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 15, 253, 255, 221, 0, 0, 0],
            &[0, 0, 130, 164, 94, 63, 74, 188, 255, 255, 147, 0, 0, 0],
            &[0, 0, 142, 255, 255, 255, 255, 255, 255, 222, 23, 0, 0, 0],
            &[0, 0, 77, 207, 255, 255, 255, 241, 156, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 39, 32, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '6' / 0x36
        '6' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 105, 157, 177, 176, 63, 0, 0, 0],
            &[0, 0, 0, 0, 120, 246, 255, 255, 255, 255, 103, 0, 0, 0],
            &[0, 0, 0, 124, 255, 255, 245, 171, 139, 148, 70, 0, 0, 0],
            &[0, 0, 32, 251, 255, 215, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 255, 255, 61, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 188, 255, 238, 33, 153, 198, 185, 98, 0, 0, 0, 0],
            &[0, 0, 224, 255, 215, 228, 255, 255, 255, 255, 128, 0, 0, 0],
            &[0, 0, 239, 255, 255, 231, 126, 139, 250, 255, 247, 16, 0, 0],
            &[0, 0, 234, 255, 255, 54, 0, 0, 154, 255, 255, 69, 0, 0],
            &[0, 0, 210, 255, 241, 0, 0, 0, 114, 255, 255, 85, 0, 0],
            &[0, 0, 156, 255, 252, 19, 0, 0, 147, 255, 255, 64, 0, 0],
            &[0, 0, 64, 255, 255, 188, 67, 99, 243, 255, 243, 9, 0, 0],
            &[0, 0, 0, 167, 255, 255, 255, 255, 255, 255, 119, 0, 0, 0],
            &[0, 0, 0, 6, 143, 245, 255, 255, 234, 114, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 43, 35, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '7' / 0x37
        '7' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 139, 139, 139, 139, 139, 139, 139, 139, 139, 62, 0, 0],
            &[
                0, 12, 255, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0,
            ],
            &[0, 10, 202, 202, 202, 202, 202, 202, 236, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 228, 255, 227, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 93, 255, 255, 120, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 209, 255, 245, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 70, 255, 255, 152, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 255, 254, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 47, 255, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 163, 255, 255, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 250, 255, 215, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 255, 255, 105, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 241, 255, 238, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 255, 137, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '8' / 0x38
        '8' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 152, 182, 170, 114, 16, 0, 0, 0, 0],
            &[0, 0, 0, 161, 255, 255, 255, 255, 255, 231, 39, 0, 0, 0],
            &[0, 0, 66, 255, 255, 234, 152, 199, 255, 255, 170, 0, 0, 0],
            &[0, 0, 108, 255, 255, 93, 0, 2, 231, 255, 211, 0, 0, 0],
            &[0, 0, 74, 255, 255, 123, 0, 25, 248, 255, 176, 0, 0, 0],
            &[0, 0, 3, 210, 255, 251, 141, 218, 255, 247, 54, 0, 0, 0],
            &[0, 0, 0, 27, 212, 255, 255, 255, 230, 59, 0, 0, 0, 0],
            &[0, 0, 0, 84, 239, 255, 255, 255, 254, 144, 4, 0, 0, 0],
            &[0, 0, 59, 253, 255, 207, 66, 183, 255, 255, 136, 0, 0, 0],
            &[0, 0, 177, 255, 246, 23, 0, 3, 188, 255, 253, 19, 0, 0],
            &[0, 0, 214, 255, 213, 0, 0, 0, 112, 255, 255, 60, 0, 0],
            &[0, 0, 189, 255, 253, 97, 17, 47, 210, 255, 254, 29, 0, 0],
            &[0, 0, 77, 254, 255, 255, 255, 255, 255, 255, 156, 0, 0, 0],
            &[0, 0, 0, 79, 210, 255, 255, 255, 231, 123, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 19, 42, 30, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '9' / 0x39
        '9' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 87, 167, 191, 168, 86, 0, 0, 0, 0, 0],
            &[0, 0, 2, 174, 255, 255, 255, 255, 255, 171, 3, 0, 0, 0],
            &[0, 0, 102, 255, 255, 238, 176, 225, 255, 255, 113, 0, 0, 0],
            &[0, 0, 194, 255, 253, 43, 0, 11, 219, 255, 225, 1, 0, 0],
            &[0, 0, 232, 255, 222, 0, 0, 0, 141, 255, 255, 38, 0, 0],
            &[0, 0, 236, 255, 227, 0, 0, 0, 163, 255, 255, 75, 0, 0],
            &[0, 0, 196, 255, 255, 90, 4, 77, 246, 255, 255, 87, 0, 0],
            &[0, 0, 97, 255, 255, 255, 253, 255, 240, 255, 255, 79, 0, 0],
            &[0, 0, 1, 140, 252, 255, 255, 195, 133, 255, 255, 52, 0, 0],
            &[0, 0, 0, 0, 19, 56, 35, 0, 170, 255, 248, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 44, 249, 255, 178, 0, 0, 0],
            &[0, 0, 2, 38, 17, 32, 102, 233, 255, 252, 54, 0, 0, 0],
            &[0, 0, 11, 255, 255, 255, 255, 255, 254, 106, 0, 0, 0, 0],
            &[0, 0, 11, 255, 255, 255, 247, 178, 57, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 45, 33, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ':' / 0x3a
        ':' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 133, 93, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 230, 255, 255, 70, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 243, 255, 255, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 85, 188, 145, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 28, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 236, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 251, 255, 255, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 177, 255, 238, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 37, 12, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ';' / 0x3b
        ';' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 113, 127, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 123, 255, 255, 177, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 137, 255, 255, 192, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 167, 181, 50, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 149, 149, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 118, 255, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 161, 255, 236, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 161, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 248, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 51, 246, 221, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '<' / 0x3c
        '<' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 81, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 7, 109, 231, 200, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 101, 226, 255, 255, 165, 0, 0, 0],
            &[0, 0, 0, 2, 93, 220, 255, 255, 194, 68, 0, 0, 0, 0],
            &[0, 0, 48, 214, 255, 255, 190, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 97, 255, 255, 225, 77, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 94, 216, 255, 255, 212, 95, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 87, 210, 255, 255, 227, 99, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 80, 204, 255, 200, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 73, 144, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '=' / 0x3d
        '=' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 166, 166, 166, 166, 166, 166, 166, 166, 11, 0, 0],
            &[0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 107, 160, 160, 160, 160, 160, 160, 160, 160, 11, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 169, 169, 169, 169, 169, 169, 169, 169, 11, 0, 0],
            &[0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 105, 158, 158, 158, 158, 158, 158, 158, 158, 11, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '>' / 0x3e
        '>' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 50, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 98, 253, 162, 35, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 71, 247, 255, 251, 154, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 143, 246, 255, 248, 146, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 140, 244, 255, 245, 119, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 168, 255, 255, 199, 0, 0, 0],
            &[0, 0, 0, 0, 48, 166, 252, 255, 246, 144, 27, 0, 0, 0],
            &[0, 0, 39, 185, 255, 255, 243, 137, 22, 0, 0, 0, 0, 0],
            &[0, 0, 98, 255, 239, 129, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 80, 122, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '?' / 0x3f
        '?' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 105, 163, 187, 178, 124, 21, 0, 0, 0, 0],
            &[0, 0, 64, 247, 255, 255, 255, 255, 255, 234, 40, 0, 0, 0],
            &[0, 0, 8, 230, 247, 181, 144, 188, 255, 255, 171, 0, 0, 0],
            &[0, 0, 0, 53, 18, 0, 0, 1, 223, 255, 215, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 17, 244, 255, 189, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 197, 255, 255, 82, 0, 0, 0],
            &[0, 0, 0, 0, 0, 36, 223, 255, 251, 107, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 241, 66, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 252, 255, 124, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 127, 127, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 60, 246, 255, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 128, 255, 255, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 61, 247, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 33, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '@' / 0x40
        '@' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 74, 125, 132, 92, 9, 0, 0, 0, 0],
            &[0, 0, 0, 50, 217, 255, 255, 255, 255, 232, 59, 0, 0, 0],
            &[0, 0, 42, 239, 245, 116, 27, 8, 85, 237, 244, 35, 0, 0],
            &[0, 0, 197, 254, 77, 0, 0, 0, 0, 68, 255, 173, 0, 0],
            &[0, 64, 255, 175, 0, 99, 205, 209, 166, 52, 205, 249, 13, 0],
            &[
                0, 150, 255, 75, 87, 255, 249, 233, 255, 110, 146, 255, 66, 0,
            ],
            &[0, 205, 254, 14, 204, 253, 43, 116, 255, 97, 120, 255, 89, 0],
            &[0, 237, 239, 3, 253, 209, 0, 129, 255, 84, 120, 255, 91, 0],
            &[0, 251, 225, 11, 255, 195, 0, 161, 255, 71, 138, 255, 66, 0],
            &[0, 236, 239, 0, 241, 227, 3, 211, 255, 83, 193, 247, 14, 0],
            &[
                0, 210, 254, 19, 163, 255, 226, 254, 251, 235, 255, 158, 0, 0,
            ],
            &[0, 143, 255, 90, 22, 186, 218, 101, 101, 224, 180, 11, 0, 0],
            &[0, 50, 253, 214, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 146, 255, 194, 47, 0, 0, 19, 89, 64, 0, 0, 0],
            &[0, 0, 2, 143, 253, 255, 242, 242, 255, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 35, 124, 158, 164, 136, 69, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'A' / 0x41
        'A' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 159, 159, 159, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 255, 255, 255, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 239, 255, 250, 13, 0, 0, 0, 0],
            &[0, 0, 0, 1, 234, 255, 137, 254, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 255, 60, 212, 255, 150, 0, 0, 0, 0],
            &[0, 0, 0, 121, 255, 246, 6, 151, 255, 220, 0, 0, 0, 0],
            &[0, 0, 0, 191, 255, 189, 0, 90, 255, 255, 35, 0, 0, 0],
            &[0, 0, 12, 249, 255, 125, 0, 28, 255, 255, 106, 0, 0, 0],
            &[0, 0, 76, 255, 255, 226, 164, 164, 255, 255, 176, 0, 0, 0],
            &[0, 0, 146, 255, 255, 255, 255, 255, 255, 255, 242, 5, 0, 0],
            &[0, 0, 217, 255, 255, 163, 163, 163, 246, 255, 255, 62, 0, 0],
            &[0, 31, 255, 255, 126, 0, 0, 0, 37, 255, 255, 132, 0, 0],
            &[0, 101, 255, 255, 62, 0, 0, 0, 0, 231, 255, 203, 0, 0],
            &[0, 172, 255, 247, 7, 0, 0, 0, 0, 170, 255, 253, 20, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'B' / 0x42
        'B' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 144, 125, 97, 21, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 248, 118, 0, 0, 0],
            &[0, 0, 122, 255, 255, 184, 179, 209, 255, 255, 253, 35, 0, 0],
            &[0, 0, 122, 255, 255, 21, 0, 0, 161, 255, 255, 83, 0, 0],
            &[0, 0, 122, 255, 255, 21, 0, 0, 124, 255, 255, 67, 0, 0],
            &[0, 0, 122, 255, 255, 74, 59, 87, 225, 255, 229, 7, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 199, 44, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 244, 124, 0, 0, 0],
            &[0, 0, 122, 255, 255, 47, 28, 56, 194, 255, 255, 79, 0, 0],
            &[0, 0, 122, 255, 255, 21, 0, 0, 76, 255, 255, 151, 0, 0],
            &[0, 0, 122, 255, 255, 21, 0, 0, 84, 255, 255, 150, 0, 0],
            &[0, 0, 122, 255, 255, 82, 66, 89, 216, 255, 255, 96, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 205, 8, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 248, 218, 136, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'C' / 0x43
        'C' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 47, 132, 174, 186, 160, 104, 16, 0, 0],
            &[0, 0, 0, 5, 155, 254, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 0, 157, 255, 255, 248, 190, 176, 215, 248, 19, 0, 0],
            &[0, 0, 57, 255, 255, 214, 29, 0, 0, 0, 31, 0, 0, 0],
            &[0, 0, 157, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 219, 255, 238, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 211, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 255, 255, 195, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 247, 255, 210, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 217, 255, 249, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 251, 255, 253, 147, 74, 69, 109, 195, 26, 0, 0],
            &[0, 0, 0, 120, 255, 255, 255, 255, 255, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 86, 208, 255, 255, 255, 244, 173, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 40, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'D' / 0x44
        'D' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 144, 144, 131, 99, 21, 0, 0, 0, 0, 0],
            &[0, 37, 255, 255, 255, 255, 255, 255, 249, 127, 1, 0, 0, 0],
            &[0, 37, 255, 255, 230, 197, 215, 254, 255, 255, 130, 0, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 50, 233, 255, 251, 34, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 104, 255, 255, 119, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 29, 255, 255, 176, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 2, 254, 255, 198, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 0, 251, 255, 203, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 16, 255, 255, 180, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 73, 255, 255, 136, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 5, 195, 255, 255, 54, 0, 0],
            &[0, 37, 255, 255, 183, 87, 111, 204, 255, 255, 175, 0, 0, 0],
            &[0, 37, 255, 255, 255, 255, 255, 255, 255, 199, 20, 0, 0, 0],
            &[0, 37, 255, 255, 255, 255, 236, 199, 106, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'E' / 0x45
        'E' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 144, 144, 144, 144, 144, 10, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 215, 207, 207, 207, 207, 207, 14, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 93, 58, 58, 58, 58, 45, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 77, 37, 37, 37, 37, 29, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 124, 95, 95, 95, 95, 95, 6, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'F' / 0x46
        'F' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 144, 144, 144, 144, 144, 10, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 215, 207, 207, 207, 207, 207, 14, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 183, 168, 168, 168, 168, 131, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 196, 183, 183, 183, 183, 144, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'G' / 0x47
        'G' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 96, 157, 185, 181, 143, 64, 1, 0, 0],
            &[0, 0, 0, 51, 229, 255, 255, 255, 255, 255, 255, 65, 0, 0],
            &[0, 0, 29, 242, 255, 255, 230, 173, 174, 220, 223, 1, 0, 0],
            &[0, 0, 168, 255, 255, 151, 5, 0, 0, 0, 28, 0, 0, 0],
            &[0, 11, 247, 255, 220, 4, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 255, 255, 137, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 255, 255, 98, 0, 0, 134, 163, 163, 163, 88, 0, 0],
            &[0, 104, 255, 255, 88, 0, 0, 211, 255, 255, 255, 139, 0, 0],
            &[0, 85, 255, 255, 107, 0, 0, 135, 164, 250, 255, 139, 0, 0],
            &[0, 52, 255, 255, 163, 0, 0, 0, 0, 241, 255, 139, 0, 0],
            &[0, 2, 234, 255, 247, 33, 0, 0, 0, 241, 255, 139, 0, 0],
            &[0, 0, 121, 255, 255, 222, 91, 45, 61, 246, 255, 139, 0, 0],
            &[0, 0, 6, 187, 255, 255, 255, 255, 255, 255, 255, 139, 0, 0],
            &[0, 0, 0, 5, 122, 227, 255, 255, 255, 252, 201, 71, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 42, 28, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'H' / 0x48
        'H' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 88, 0, 0, 0, 30, 144, 144, 78, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 182, 67, 67, 67, 107, 255, 255, 139, 0, 0],
            &[
                0, 37, 255, 255, 255, 255, 255, 255, 255, 255, 255, 139, 0, 0,
            ],
            &[
                0, 37, 255, 255, 255, 255, 255, 255, 255, 255, 255, 139, 0, 0,
            ],
            &[0, 37, 255, 255, 167, 28, 28, 28, 75, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'I' / 0x49
        'I' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 144, 144, 144, 144, 144, 144, 144, 99, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 26, 108, 138, 221, 255, 255, 148, 119, 65, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 27, 182, 255, 254, 37, 9, 0, 0, 0, 0],
            &[0, 0, 68, 250, 255, 255, 255, 255, 255, 255, 165, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'J' / 0x4a
        'J' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 56, 144, 144, 44, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 108, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 255, 255, 52, 0, 0, 0],
            &[0, 0, 52, 105, 69, 63, 124, 253, 255, 240, 5, 0, 0, 0],
            &[0, 0, 98, 255, 255, 255, 255, 255, 255, 120, 0, 0, 0, 0],
            &[0, 0, 81, 245, 255, 255, 255, 226, 111, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 31, 43, 25, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'K' / 0x4b
        'K' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 80, 0, 0, 0, 65, 144, 144, 80, 0, 0],
            &[0, 37, 255, 255, 141, 0, 0, 18, 231, 255, 233, 22, 0, 0],
            &[0, 37, 255, 255, 141, 0, 0, 164, 255, 255, 76, 0, 0, 0],
            &[0, 37, 255, 255, 141, 0, 79, 255, 255, 151, 0, 0, 0, 0],
            &[0, 37, 255, 255, 141, 18, 231, 255, 216, 10, 0, 0, 0, 0],
            &[0, 37, 255, 255, 141, 160, 255, 250, 51, 0, 0, 0, 0, 0],
            &[0, 37, 255, 255, 195, 254, 255, 162, 0, 0, 0, 0, 0, 0],
            &[0, 37, 255, 255, 255, 255, 255, 242, 22, 0, 0, 0, 0, 0],
            &[0, 37, 255, 255, 255, 198, 255, 255, 151, 0, 0, 0, 0, 0],
            &[0, 37, 255, 255, 178, 2, 194, 255, 251, 41, 0, 0, 0, 0],
            &[0, 37, 255, 255, 141, 0, 59, 255, 255, 179, 0, 0, 0, 0],
            &[0, 37, 255, 255, 141, 0, 0, 180, 255, 255, 65, 0, 0, 0],
            &[0, 37, 255, 255, 141, 0, 0, 46, 253, 255, 205, 1, 0, 0],
            &[0, 37, 255, 255, 141, 0, 0, 0, 165, 255, 255, 92, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'L' / 0x4c
        'L' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 144, 144, 67, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 178, 110, 110, 110, 110, 110, 7, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'M' / 0x4d
        'M' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 34, 144, 144, 144, 76, 0, 0, 139, 144, 144, 89, 0, 0],
            &[0, 61, 255, 255, 255, 164, 0, 26, 255, 255, 255, 158, 0, 0],
            &[0, 61, 255, 253, 255, 203, 0, 71, 255, 254, 255, 158, 0, 0],
            &[0, 61, 255, 213, 252, 241, 0, 117, 255, 209, 255, 158, 0, 0],
            &[0, 61, 255, 224, 212, 255, 24, 162, 255, 181, 255, 158, 0, 0],
            &[0, 61, 255, 244, 163, 255, 63, 208, 248, 161, 255, 158, 0, 0],
            &[
                0, 61, 255, 255, 122, 255, 106, 249, 206, 173, 255, 158, 0, 0,
            ],
            &[0, 61, 255, 255, 90, 255, 183, 255, 161, 186, 255, 158, 0, 0],
            &[0, 61, 255, 255, 56, 255, 252, 255, 116, 198, 255, 158, 0, 0],
            &[0, 61, 255, 255, 49, 224, 255, 255, 71, 209, 255, 158, 0, 0],
            &[0, 61, 255, 255, 56, 146, 206, 206, 24, 215, 255, 158, 0, 0],
            &[0, 61, 255, 255, 58, 0, 0, 0, 0, 217, 255, 158, 0, 0],
            &[0, 61, 255, 255, 58, 0, 0, 0, 0, 217, 255, 158, 0, 0],
            &[0, 61, 255, 255, 58, 0, 0, 0, 0, 217, 255, 158, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'N' / 0x4e
        'N' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 144, 100, 0, 0, 0, 131, 144, 78, 0, 0],
            &[0, 37, 255, 255, 255, 237, 4, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 255, 255, 69, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 200, 255, 153, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 132, 255, 234, 3, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 74, 244, 255, 65, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 75, 170, 255, 148, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 87, 255, 230, 2, 230, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 12, 247, 255, 60, 226, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 177, 255, 143, 219, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 94, 255, 226, 210, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 17, 250, 255, 246, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 0, 184, 255, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 0, 101, 255, 255, 255, 139, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'O' / 0x4f
        'O' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 77, 163, 194, 183, 124, 19, 0, 0, 0, 0],
            &[0, 0, 2, 164, 255, 255, 255, 255, 255, 232, 42, 0, 0, 0],
            &[0, 0, 105, 255, 255, 236, 162, 208, 255, 255, 210, 2, 0, 0],
            &[0, 0, 223, 255, 245, 35, 0, 2, 185, 255, 255, 72, 0, 0],
            &[0, 34, 255, 255, 169, 0, 0, 0, 76, 255, 255, 143, 0, 0],
            &[0, 80, 255, 255, 120, 0, 0, 0, 21, 255, 255, 184, 0, 0],
            &[0, 98, 255, 255, 100, 0, 0, 0, 3, 254, 255, 203, 0, 0],
            &[0, 105, 255, 255, 96, 0, 0, 0, 0, 250, 255, 208, 0, 0],
            &[0, 92, 255, 255, 109, 0, 0, 0, 7, 255, 255, 195, 0, 0],
            &[0, 59, 255, 255, 145, 0, 0, 0, 45, 255, 255, 161, 0, 0],
            &[0, 11, 246, 255, 221, 4, 0, 0, 126, 255, 255, 104, 0, 0],
            &[0, 0, 163, 255, 255, 166, 60, 102, 246, 255, 244, 21, 0, 0],
            &[0, 0, 30, 233, 255, 255, 255, 255, 255, 255, 109, 0, 0, 0],
            &[0, 0, 0, 38, 184, 253, 255, 255, 224, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 41, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'P' / 0x50
        'P' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 144, 137, 105, 30, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 251, 107, 0, 0, 0],
            &[0, 0, 122, 255, 255, 207, 197, 219, 255, 255, 254, 46, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 122, 255, 255, 130, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 36, 255, 255, 158, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 68, 255, 255, 142, 0, 0],
            &[0, 0, 122, 255, 255, 80, 46, 86, 222, 255, 255, 83, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 204, 5, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 244, 157, 17, 0, 0, 0],
            &[0, 0, 122, 255, 255, 82, 44, 29, 5, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Q' / 0x51
        'Q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 77, 163, 194, 183, 124, 19, 0, 0, 0, 0],
            &[0, 0, 2, 164, 255, 255, 255, 255, 255, 232, 42, 0, 0, 0],
            &[0, 0, 105, 255, 255, 236, 162, 208, 255, 255, 210, 2, 0, 0],
            &[0, 0, 223, 255, 245, 35, 0, 2, 185, 255, 255, 72, 0, 0],
            &[0, 34, 255, 255, 169, 0, 0, 0, 76, 255, 255, 143, 0, 0],
            &[0, 80, 255, 255, 120, 0, 0, 0, 21, 255, 255, 184, 0, 0],
            &[0, 98, 255, 255, 100, 0, 0, 0, 3, 254, 255, 203, 0, 0],
            &[0, 105, 255, 255, 96, 0, 0, 0, 0, 250, 255, 208, 0, 0],
            &[0, 92, 255, 255, 109, 0, 0, 0, 7, 255, 255, 195, 0, 0],
            &[0, 59, 255, 255, 145, 0, 0, 0, 45, 255, 255, 161, 0, 0],
            &[0, 11, 246, 255, 221, 4, 0, 0, 126, 255, 255, 104, 0, 0],
            &[0, 0, 163, 255, 255, 166, 60, 102, 246, 255, 244, 21, 0, 0],
            &[0, 0, 30, 233, 255, 255, 255, 255, 255, 255, 109, 0, 0, 0],
            &[0, 0, 0, 38, 184, 253, 255, 255, 255, 185, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 49, 255, 255, 220, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 107, 255, 255, 109, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 7, 227, 255, 234, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 60, 107, 107, 38, 0, 0],
        ]),
        // letter: 'R' / 0x52
        'R' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 69, 144, 144, 144, 137, 112, 47, 0, 0, 0, 0, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 173, 8, 0, 0, 0, 0],
            &[0, 122, 255, 255, 213, 209, 250, 255, 255, 147, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 31, 237, 255, 242, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 0, 187, 255, 255, 16, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 10, 231, 255, 245, 3, 0, 0, 0],
            &[0, 122, 255, 255, 153, 133, 211, 255, 255, 147, 0, 0, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 191, 13, 0, 0, 0, 0],
            &[0, 122, 255, 255, 235, 244, 255, 242, 12, 0, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 77, 255, 255, 129, 0, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 191, 255, 246, 31, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 51, 254, 255, 170, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 0, 164, 255, 255, 63, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 0, 30, 247, 255, 208, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'S' / 0x53
        'S' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 58, 145, 182, 182, 157, 112, 33, 0, 0, 0],
            &[0, 0, 0, 129, 255, 255, 255, 255, 255, 255, 234, 4, 0, 0],
            &[0, 0, 37, 254, 255, 252, 167, 145, 182, 247, 139, 0, 0, 0],
            &[0, 0, 91, 255, 255, 138, 0, 0, 0, 13, 17, 0, 0, 0],
            &[0, 0, 84, 255, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 244, 255, 253, 132, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 87, 251, 255, 255, 235, 109, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 208, 255, 255, 255, 214, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 97, 226, 255, 255, 224, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 205, 255, 255, 68, 0, 0],
            &[0, 0, 5, 0, 0, 0, 0, 0, 138, 255, 255, 82, 0, 0],
            &[0, 0, 163, 163, 85, 39, 31, 77, 228, 255, 255, 35, 0, 0],
            &[0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 87, 205, 253, 255, 255, 255, 223, 114, 2, 0, 0, 0],
            &[0, 0, 0, 0, 6, 30, 43, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'T' / 0x54
        'T' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 62, 144, 144, 144, 144, 144, 144, 144, 144, 144, 120, 0, 0,
            ],
            &[
                0, 110, 255, 255, 255, 255, 255, 255, 255, 255, 255, 212, 0, 0,
            ],
            &[
                0, 89, 207, 207, 207, 240, 255, 255, 210, 207, 207, 171, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'U' / 0x55
        'U' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 88, 0, 0, 0, 30, 144, 144, 78, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 32, 255, 255, 165, 0, 0, 0, 73, 255, 255, 134, 0, 0],
            &[0, 9, 254, 255, 213, 0, 0, 0, 112, 255, 255, 108, 0, 0],
            &[0, 0, 196, 255, 255, 148, 65, 98, 232, 255, 255, 38, 0, 0],
            &[0, 0, 63, 253, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 67, 204, 255, 255, 255, 231, 123, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 41, 27, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'V' / 0x56
        'V' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 105, 144, 144, 5, 0, 0, 0, 0, 97, 144, 144, 18, 0],
            &[0, 127, 255, 255, 60, 0, 0, 0, 0, 221, 255, 228, 0, 0],
            &[0, 52, 255, 255, 126, 0, 0, 0, 29, 255, 255, 154, 0, 0],
            &[0, 1, 232, 255, 192, 0, 0, 0, 92, 255, 255, 79, 0, 0],
            &[0, 0, 158, 255, 248, 9, 0, 0, 155, 255, 248, 12, 0, 0],
            &[0, 0, 83, 255, 255, 68, 0, 0, 219, 255, 186, 0, 0, 0],
            &[0, 0, 14, 250, 255, 133, 0, 26, 255, 255, 111, 0, 0, 0],
            &[0, 0, 0, 189, 255, 199, 0, 90, 255, 255, 36, 0, 0, 0],
            &[0, 0, 0, 114, 255, 251, 13, 152, 255, 217, 0, 0, 0, 0],
            &[0, 0, 0, 39, 255, 255, 71, 206, 255, 142, 0, 0, 0, 0],
            &[0, 0, 0, 0, 220, 255, 128, 249, 255, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 146, 255, 220, 255, 242, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 255, 255, 255, 173, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 244, 255, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'W' / 0x57
        'W' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 114, 144, 88, 0, 0, 0, 0, 0, 29, 144, 144, 27, 0],
            &[0, 187, 255, 175, 0, 0, 0, 0, 0, 66, 255, 255, 34, 0],
            &[0, 168, 255, 199, 0, 0, 0, 0, 0, 84, 255, 255, 16, 0],
            &[0, 149, 255, 223, 0, 59, 95, 95, 5, 102, 255, 253, 1, 0],
            &[0, 130, 255, 243, 0, 186, 255, 255, 41, 121, 255, 236, 0, 0],
            &[0, 111, 255, 255, 6, 225, 255, 255, 79, 139, 255, 218, 0, 0],
            &[0, 92, 255, 255, 31, 254, 231, 255, 118, 156, 255, 200, 0, 0],
            &[0, 73, 255, 255, 79, 255, 162, 255, 156, 169, 255, 183, 0, 0],
            &[
                0, 54, 255, 255, 120, 255, 102, 246, 196, 173, 255, 165, 0, 0,
            ],
            &[0, 35, 255, 255, 157, 255, 62, 210, 238, 170, 255, 147, 0, 0],
            &[0, 16, 255, 255, 190, 255, 24, 173, 255, 185, 255, 129, 0, 0],
            &[0, 1, 251, 255, 226, 242, 0, 135, 255, 223, 255, 111, 0, 0],
            &[0, 0, 233, 255, 255, 204, 0, 97, 255, 255, 255, 93, 0, 0],
            &[0, 0, 214, 255, 255, 167, 0, 59, 255, 255, 255, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'X' / 0x58
        'X' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 27, 144, 144, 87, 0, 0, 0, 35, 144, 144, 85, 0, 0],
            &[0, 0, 188, 255, 241, 19, 0, 0, 165, 255, 250, 40, 0, 0],
            &[0, 0, 44, 252, 255, 138, 0, 43, 253, 255, 142, 0, 0, 0],
            &[0, 0, 0, 148, 255, 246, 26, 173, 255, 234, 15, 0, 0, 0],
            &[0, 0, 0, 18, 237, 255, 190, 254, 255, 101, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 255, 255, 255, 206, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 212, 255, 255, 61, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 211, 255, 255, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 255, 255, 255, 209, 3, 0, 0, 0, 0],
            &[0, 0, 0, 19, 237, 255, 188, 255, 255, 104, 0, 0, 0, 0],
            &[0, 0, 0, 149, 255, 243, 22, 177, 255, 236, 17, 0, 0, 0],
            &[0, 0, 45, 252, 255, 131, 0, 46, 254, 255, 145, 0, 0, 0],
            &[0, 0, 191, 255, 236, 14, 0, 0, 169, 255, 251, 42, 0, 0],
            &[0, 84, 255, 255, 116, 0, 0, 0, 40, 252, 255, 186, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Y' / 0x59
        'Y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 70, 144, 144, 58, 0, 0, 0, 3, 139, 144, 128, 0, 0],
            &[0, 33, 251, 255, 188, 0, 0, 0, 80, 255, 255, 131, 0, 0],
            &[0, 0, 162, 255, 255, 41, 0, 0, 187, 255, 245, 20, 0, 0],
            &[0, 0, 42, 254, 255, 148, 0, 40, 254, 255, 145, 0, 0, 0],
            &[0, 0, 0, 175, 255, 242, 14, 146, 255, 250, 29, 0, 0, 0],
            &[0, 0, 0, 53, 255, 255, 121, 241, 255, 158, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 255, 251, 255, 253, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 255, 255, 255, 172, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 203, 255, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Z' / 0x5a
        'Z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 144, 144, 144, 144, 144, 144, 144, 144, 4, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 8, 0, 0],
            &[0, 0, 0, 197, 197, 197, 197, 203, 255, 255, 224, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 82, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 243, 255, 187, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 161, 255, 251, 42, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 254, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 200, 255, 232, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 255, 97, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 230, 255, 201, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 135, 255, 254, 53, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 35, 248, 255, 211, 95, 95, 95, 95, 95, 6, 0, 0],
            &[0, 0, 121, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '[' / 0x5b
        '[' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 178, 178, 178, 166, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 255, 255, 237, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 202, 61, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 247, 224, 209, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 255, 255, 237, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 19, 19, 19, 18, 0, 0, 0, 0, 0],
        ]),
        // letter: '\' / 0x5c
        '\\' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 157, 164, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 170, 255, 181, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 73, 255, 252, 26, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 229, 255, 120, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 255, 217, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 255, 255, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 196, 255, 157, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 99, 255, 243, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 245, 255, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 161, 255, 193, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 64, 255, 254, 36, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 221, 255, 132, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 125, 255, 227, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 30, 253, 255, 71, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 17, 19, 9, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ']' / 0x5d
        ']' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 212, 212, 212, 212, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 255, 255, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 31, 104, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 134, 254, 254, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 128, 241, 241, 241, 241, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '^' / 0x5e
        '^' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 154, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 161, 255, 200, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 251, 255, 255, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 255, 220, 255, 208, 1, 0, 0, 0, 0],
            &[0, 0, 0, 28, 249, 250, 36, 221, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 146, 255, 161, 0, 95, 255, 215, 3, 0, 0, 0],
            &[0, 0, 23, 246, 254, 41, 0, 3, 216, 255, 95, 0, 0, 0],
            &[0, 0, 138, 255, 175, 0, 0, 0, 88, 255, 222, 5, 0, 0],
            &[0, 15, 229, 242, 54, 0, 0, 0, 1, 202, 242, 95, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '_' / 0x5f
        '_' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 186, 186, 186, 186, 186, 186, 186, 186, 186, 186, 186, 74, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 102, 0,
            ],
            &[
                0, 141, 141, 141, 141, 141, 141, 141, 141, 141, 141, 141, 56, 0,
            ],
        ]),
        // letter: '`' / 0x60
        '`' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 141, 141, 107, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 142, 255, 254, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 128, 255, 222, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 80, 124, 52, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'a' / 0x61
        'a' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 83, 126, 140, 128, 82, 9, 0, 0, 0, 0],
            &[0, 0, 55, 247, 255, 255, 255, 255, 255, 228, 50, 0, 0, 0],
            &[0, 0, 3, 223, 240, 190, 170, 205, 255, 255, 214, 0, 0, 0],
            &[0, 0, 0, 29, 5, 0, 0, 0, 164, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 40, 85, 104, 111, 180, 255, 255, 42, 0, 0],
            &[0, 0, 28, 200, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 186, 255, 252, 143, 76, 55, 144, 255, 255, 42, 0, 0],
            &[0, 2, 251, 255, 185, 0, 0, 0, 151, 255, 255, 42, 0, 0],
            &[0, 2, 246, 255, 221, 42, 12, 90, 244, 255, 255, 42, 0, 0],
            &[0, 0, 180, 255, 255, 255, 255, 255, 231, 255, 255, 42, 0, 0],
            &[0, 0, 30, 197, 255, 255, 245, 141, 30, 253, 255, 42, 0, 0],
            &[0, 0, 0, 0, 20, 38, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'b' / 0x62
        'b' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 112, 83, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 184, 4, 86, 135, 121, 38, 0, 0, 0, 0],
            &[0, 0, 244, 255, 181, 199, 255, 255, 255, 249, 80, 0, 0, 0],
            &[0, 0, 244, 255, 255, 252, 200, 225, 255, 255, 236, 12, 0, 0],
            &[0, 0, 244, 255, 255, 81, 0, 5, 184, 255, 255, 87, 0, 0],
            &[0, 0, 244, 255, 225, 0, 0, 0, 76, 255, 255, 139, 0, 0],
            &[0, 0, 244, 255, 197, 0, 0, 0, 47, 255, 255, 159, 0, 0],
            &[0, 0, 244, 255, 208, 0, 0, 0, 57, 255, 255, 153, 0, 0],
            &[0, 0, 244, 255, 246, 15, 0, 0, 111, 255, 255, 120, 0, 0],
            &[0, 0, 244, 255, 255, 171, 42, 72, 233, 255, 255, 53, 0, 0],
            &[0, 0, 244, 255, 245, 255, 255, 255, 255, 255, 187, 0, 0, 0],
            &[0, 0, 244, 255, 82, 128, 243, 255, 255, 182, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 40, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'c' / 0x63
        'c' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 62, 118, 138, 123, 83, 13, 0, 0, 0],
            &[0, 0, 0, 30, 202, 255, 255, 255, 255, 255, 182, 0, 0, 0],
            &[0, 0, 3, 203, 255, 255, 245, 205, 219, 255, 92, 0, 0, 0],
            &[0, 0, 70, 255, 255, 206, 19, 0, 0, 26, 8, 0, 0, 0],
            &[0, 0, 136, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 161, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 159, 255, 255, 45, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 255, 255, 113, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 255, 255, 243, 105, 45, 65, 137, 125, 0, 0, 0],
            &[0, 0, 0, 170, 255, 255, 255, 255, 255, 255, 142, 0, 0, 0],
            &[0, 0, 0, 5, 132, 232, 255, 255, 255, 218, 81, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 26, 42, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'd' / 0x64
        'd' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 38, 112, 112, 39, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 87, 255, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 87, 255, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 87, 255, 255, 90, 0, 0],
            &[0, 0, 0, 2, 83, 131, 119, 41, 77, 255, 255, 90, 0, 0],
            &[0, 0, 11, 196, 255, 255, 255, 252, 143, 255, 255, 90, 0, 0],
            &[0, 0, 136, 255, 255, 249, 200, 218, 255, 255, 255, 90, 0, 0],
            &[0, 1, 235, 255, 253, 58, 0, 1, 171, 255, 255, 90, 0, 0],
            &[0, 34, 255, 255, 189, 0, 0, 0, 74, 255, 255, 90, 0, 0],
            &[0, 55, 255, 255, 152, 0, 0, 0, 44, 255, 255, 90, 0, 0],
            &[0, 52, 255, 255, 155, 0, 0, 0, 44, 255, 255, 90, 0, 0],
            &[0, 21, 255, 255, 203, 0, 0, 0, 92, 255, 255, 90, 0, 0],
            &[0, 0, 212, 255, 255, 126, 36, 69, 224, 255, 255, 90, 0, 0],
            &[0, 0, 95, 255, 255, 255, 255, 255, 224, 255, 255, 90, 0, 0],
            &[0, 0, 0, 125, 246, 255, 254, 165, 17, 234, 255, 90, 0, 0],
            &[0, 0, 0, 0, 11, 40, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'e' / 0x65
        'e' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 98, 133, 126, 76, 4, 0, 0, 0, 0],
            &[0, 0, 0, 87, 244, 255, 255, 255, 255, 210, 30, 0, 0, 0],
            &[0, 0, 51, 251, 255, 242, 158, 177, 252, 255, 197, 0, 0, 0],
            &[0, 0, 169, 255, 248, 42, 0, 0, 109, 255, 255, 49, 0, 0],
            &[0, 0, 236, 255, 213, 69, 69, 69, 88, 255, 255, 96, 0, 0],
            &[0, 5, 255, 255, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 248, 255, 222, 156, 156, 156, 156, 156, 156, 70, 0, 0],
            &[0, 0, 208, 255, 222, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 255, 255, 192, 75, 32, 38, 76, 148, 28, 0, 0],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0],
            &[0, 0, 0, 5, 123, 222, 255, 255, 255, 248, 182, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 42, 29, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'f' / 0x66
        'f' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 90, 126, 127, 104, 54, 0, 0, 0],
            &[0, 0, 0, 0, 44, 233, 255, 255, 255, 255, 210, 0, 0, 0],
            &[0, 0, 0, 0, 181, 255, 255, 246, 211, 230, 124, 0, 0, 0],
            &[0, 0, 0, 0, 232, 255, 243, 28, 0, 0, 2, 0, 0, 0],
            &[0, 0, 15, 66, 248, 255, 225, 95, 95, 95, 13, 0, 0, 0],
            &[0, 158, 254, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0, 0],
            &[0, 164, 231, 231, 254, 255, 251, 231, 231, 231, 33, 0, 0, 0],
            &[0, 0, 0, 0, 243, 255, 205, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 243, 255, 205, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 243, 255, 205, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 243, 255, 205, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 243, 255, 205, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 243, 255, 205, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 243, 255, 205, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 243, 255, 205, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'g' / 0x67
        'g' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 98, 141, 119, 30, 0, 78, 95, 33, 0, 0],
            &[0, 0, 15, 207, 255, 255, 255, 243, 95, 252, 255, 90, 0, 0],
            &[0, 0, 145, 255, 255, 246, 196, 225, 255, 255, 255, 90, 0, 0],
            &[0, 2, 239, 255, 250, 46, 0, 4, 189, 255, 255, 90, 0, 0],
            &[0, 37, 255, 255, 183, 0, 0, 0, 91, 255, 255, 90, 0, 0],
            &[0, 56, 255, 255, 150, 0, 0, 0, 63, 255, 255, 90, 0, 0],
            &[0, 51, 255, 255, 158, 0, 0, 0, 67, 255, 255, 90, 0, 0],
            &[0, 21, 255, 255, 212, 0, 0, 0, 111, 255, 255, 90, 0, 0],
            &[0, 0, 210, 255, 255, 136, 38, 73, 232, 255, 255, 90, 0, 0],
            &[0, 0, 87, 255, 255, 255, 255, 255, 242, 255, 255, 90, 0, 0],
            &[0, 0, 0, 112, 243, 255, 255, 180, 118, 255, 255, 90, 0, 0],
            &[0, 0, 0, 0, 8, 40, 22, 0, 121, 255, 255, 77, 0, 0],
            &[0, 0, 60, 53, 0, 0, 0, 17, 209, 255, 255, 37, 0, 0],
            &[0, 0, 132, 255, 236, 206, 201, 243, 255, 255, 187, 0, 0, 0],
            &[0, 0, 15, 82, 137, 158, 156, 131, 65, 2, 0, 0, 0, 0],
        ]),
        // letter: 'h' / 0x68
        'h' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 112, 90, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 202, 1, 81, 131, 117, 54, 0, 0, 0, 0],
            &[0, 0, 244, 255, 199, 177, 255, 255, 255, 254, 125, 0, 0, 0],
            &[0, 0, 244, 255, 255, 254, 201, 213, 255, 255, 251, 34, 0, 0],
            &[0, 0, 244, 255, 255, 100, 0, 0, 163, 255, 255, 88, 0, 0],
            &[0, 0, 244, 255, 243, 3, 0, 0, 90, 255, 255, 114, 0, 0],
            &[0, 0, 244, 255, 213, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'i' / 0x69
        'i' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 35, 167, 170, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 255, 255, 162, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 255, 137, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 93, 92, 7, 0, 0, 0, 0, 0],
            &[0, 0, 33, 95, 95, 95, 95, 95, 33, 0, 0, 0, 0, 0],
            &[0, 0, 88, 255, 255, 255, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 53, 172, 201, 236, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 138, 255, 255, 126, 18, 0, 0, 0, 0],
            &[0, 0, 171, 238, 255, 255, 255, 255, 255, 255, 226, 86, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'j' / 0x6a
        'j' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 100, 181, 130, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 253, 255, 255, 52, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 238, 255, 251, 31, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 41, 105, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 95, 95, 95, 95, 95, 14, 0, 0, 0],
            &[0, 0, 0, 0, 140, 255, 255, 255, 255, 255, 38, 0, 0, 0],
            &[0, 0, 0, 0, 86, 178, 206, 244, 255, 255, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 255, 255, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 179, 255, 255, 35, 0, 0, 0],
            &[0, 0, 4, 1, 0, 0, 31, 237, 255, 253, 13, 0, 0, 0],
            &[0, 0, 61, 240, 203, 201, 245, 255, 255, 195, 0, 0, 0, 0],
            &[0, 0, 16, 100, 144, 162, 154, 107, 18, 0, 0, 0, 0, 0],
        ]),
        // letter: 'k' / 0x6b
        'k' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 96, 112, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 0, 46, 95, 95, 67, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 36, 236, 255, 238, 42, 0, 0, 0],
            &[0, 220, 255, 229, 0, 17, 216, 255, 248, 62, 0, 0, 0, 0],
            &[0, 220, 255, 227, 4, 189, 255, 254, 87, 0, 0, 0, 0, 0],
            &[0, 220, 255, 213, 143, 255, 255, 117, 0, 0, 0, 0, 0, 0],
            &[0, 220, 255, 240, 255, 255, 255, 113, 0, 0, 0, 0, 0, 0],
            &[0, 220, 255, 255, 255, 231, 255, 246, 37, 0, 0, 0, 0, 0],
            &[0, 220, 255, 249, 88, 38, 247, 255, 196, 2, 0, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 119, 255, 255, 114, 0, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 4, 205, 255, 246, 38, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 0, 48, 251, 255, 197, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'l' / 0x6c
        'l' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 112, 112, 112, 112, 112, 35, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 14, 105, 158, 224, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 146, 255, 255, 118, 17, 0, 0, 0, 0],
            &[0, 0, 172, 239, 255, 255, 255, 255, 255, 255, 225, 86, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'm' / 0x6d
        'm' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 95, 45, 31, 127, 105, 4, 28, 128, 110, 6, 0, 0],
            &[
                0, 110, 255, 171, 220, 255, 255, 119, 218, 255, 255, 117, 0, 0,
            ],
            &[
                0, 110, 255, 255, 214, 241, 255, 250, 212, 243, 255, 179, 0, 0,
            ],
            &[0, 110, 255, 254, 11, 135, 255, 250, 8, 147, 255, 198, 0, 0],
            &[0, 110, 255, 242, 0, 127, 255, 235, 0, 139, 255, 211, 0, 0],
            &[0, 110, 255, 241, 0, 127, 255, 234, 0, 139, 255, 212, 0, 0],
            &[0, 110, 255, 241, 0, 127, 255, 234, 0, 139, 255, 212, 0, 0],
            &[0, 110, 255, 241, 0, 127, 255, 234, 0, 139, 255, 212, 0, 0],
            &[0, 110, 255, 241, 0, 127, 255, 234, 0, 139, 255, 212, 0, 0],
            &[0, 110, 255, 241, 0, 127, 255, 234, 0, 139, 255, 212, 0, 0],
            &[0, 110, 255, 241, 0, 127, 255, 234, 0, 139, 255, 212, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'n' / 0x6e
        'n' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 95, 19, 3, 83, 131, 119, 57, 0, 0, 0, 0],
            &[0, 0, 244, 255, 124, 200, 255, 255, 255, 255, 129, 0, 0, 0],
            &[0, 0, 244, 255, 255, 254, 209, 215, 255, 255, 252, 35, 0, 0],
            &[0, 0, 244, 255, 255, 119, 0, 1, 170, 255, 255, 88, 0, 0],
            &[0, 0, 244, 255, 247, 10, 0, 0, 91, 255, 255, 114, 0, 0],
            &[0, 0, 244, 255, 218, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'o' / 0x6f
        'o' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 104, 135, 122, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 109, 250, 255, 255, 255, 255, 197, 24, 0, 0, 0],
            &[0, 0, 80, 255, 255, 244, 185, 213, 255, 255, 192, 1, 0, 0],
            &[0, 0, 209, 255, 252, 50, 0, 1, 181, 255, 255, 61, 0, 0],
            &[0, 25, 255, 255, 186, 0, 0, 0, 84, 255, 255, 129, 0, 0],
            &[0, 52, 255, 255, 150, 0, 0, 0, 50, 255, 255, 155, 0, 0],
            &[0, 49, 255, 255, 154, 0, 0, 0, 62, 255, 255, 151, 0, 0],
            &[0, 13, 252, 255, 211, 0, 0, 0, 108, 255, 255, 110, 0, 0],
            &[0, 0, 179, 255, 255, 131, 20, 55, 228, 255, 251, 29, 0, 0],
            &[0, 0, 38, 240, 255, 255, 255, 255, 255, 255, 125, 0, 0, 0],
            &[0, 0, 0, 42, 186, 253, 255, 255, 227, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 42, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'p' / 0x70
        'p' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 95, 19, 2, 80, 132, 122, 39, 0, 0, 0, 0],
            &[0, 0, 244, 255, 124, 192, 255, 255, 255, 249, 80, 0, 0, 0],
            &[0, 0, 244, 255, 255, 252, 202, 223, 255, 255, 236, 12, 0, 0],
            &[0, 0, 244, 255, 255, 85, 0, 4, 189, 255, 255, 87, 0, 0],
            &[0, 0, 244, 255, 223, 0, 0, 0, 89, 255, 255, 138, 0, 0],
            &[0, 0, 244, 255, 194, 0, 0, 0, 60, 255, 255, 159, 0, 0],
            &[0, 0, 244, 255, 208, 0, 0, 0, 69, 255, 255, 149, 0, 0],
            &[0, 0, 244, 255, 245, 15, 0, 0, 113, 255, 255, 120, 0, 0],
            &[0, 0, 244, 255, 255, 173, 43, 65, 228, 255, 255, 50, 0, 0],
            &[0, 0, 244, 255, 248, 255, 255, 255, 255, 255, 181, 0, 0, 0],
            &[0, 0, 244, 255, 175, 125, 241, 255, 255, 176, 15, 0, 0, 0],
            &[0, 0, 244, 255, 189, 0, 7, 38, 19, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 136, 143, 106, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'q' / 0x71
        'q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 93, 136, 118, 35, 0, 75, 95, 33, 0, 0],
            &[0, 0, 15, 207, 255, 255, 255, 249, 101, 245, 255, 90, 0, 0],
            &[0, 0, 146, 255, 255, 248, 198, 232, 255, 255, 255, 90, 0, 0],
            &[0, 3, 238, 255, 254, 56, 0, 11, 203, 255, 255, 90, 0, 0],
            &[0, 36, 255, 255, 198, 0, 0, 0, 96, 255, 255, 90, 0, 0],
            &[0, 56, 255, 255, 165, 0, 0, 0, 62, 255, 255, 90, 0, 0],
            &[0, 52, 255, 255, 171, 0, 0, 0, 63, 255, 255, 90, 0, 0],
            &[0, 21, 255, 255, 220, 0, 0, 0, 112, 255, 255, 90, 0, 0],
            &[0, 0, 212, 255, 255, 137, 37, 74, 233, 255, 255, 90, 0, 0],
            &[0, 0, 95, 255, 255, 255, 255, 255, 234, 255, 255, 90, 0, 0],
            &[0, 0, 0, 126, 246, 255, 253, 176, 99, 255, 255, 90, 0, 0],
            &[0, 0, 0, 0, 11, 41, 19, 0, 87, 255, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 87, 255, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 87, 255, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 49, 143, 143, 50, 0, 0],
        ]),
        // letter: 'r' / 0x72
        'r' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 67, 95, 95, 95, 58, 0, 30, 117, 138, 101, 2, 0],
            &[0, 0, 181, 255, 255, 255, 196, 64, 244, 255, 255, 220, 0, 0],
            &[0, 0, 115, 194, 240, 255, 248, 233, 255, 255, 255, 146, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 251, 114, 30, 37, 28, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 144, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 72, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 161, 255, 255, 98, 8, 0, 0, 0, 0, 0],
            &[0, 41, 220, 255, 255, 255, 255, 255, 250, 196, 0, 0, 0, 0],
            &[0, 56, 255, 255, 255, 255, 255, 255, 255, 242, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 's' / 0x73
        's' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 112, 138, 127, 102, 52, 1, 0, 0, 0],
            &[0, 0, 0, 146, 255, 255, 255, 255, 255, 255, 169, 0, 0, 0],
            &[0, 0, 55, 255, 255, 243, 190, 199, 235, 255, 90, 0, 0, 0],
            &[0, 0, 95, 255, 255, 140, 0, 0, 1, 54, 7, 0, 0, 0],
            &[0, 0, 50, 255, 255, 252, 168, 71, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 248, 255, 255, 255, 218, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 135, 232, 255, 255, 255, 101, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 111, 255, 255, 204, 0, 0, 0],
            &[0, 0, 110, 167, 89, 45, 34, 90, 254, 255, 211, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 136, 0, 0, 0],
            &[0, 0, 79, 214, 255, 255, 255, 255, 234, 132, 4, 0, 0, 0],
            &[0, 0, 0, 0, 15, 36, 43, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 't' / 0x74
        't' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 243, 245, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 53, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 142, 255, 255, 117, 95, 95, 95, 3, 0, 0],
            &[0, 0, 231, 255, 255, 255, 255, 255, 255, 255, 255, 8, 0, 0],
            &[0, 0, 230, 231, 246, 255, 255, 235, 231, 231, 231, 7, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 255, 255, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 255, 255, 177, 45, 39, 73, 12, 0, 0],
            &[0, 0, 0, 0, 44, 251, 255, 255, 255, 255, 255, 32, 0, 0],
            &[0, 0, 0, 0, 0, 83, 225, 255, 255, 255, 234, 25, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 28, 43, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'u' / 0x75
        'u' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 95, 95, 67, 0, 0, 0, 38, 95, 95, 33, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 105, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 181, 0, 0, 0, 127, 255, 255, 90, 0, 0],
            &[0, 7, 254, 255, 205, 0, 0, 0, 170, 255, 255, 90, 0, 0],
            &[0, 0, 228, 255, 254, 94, 32, 100, 252, 255, 255, 90, 0, 0],
            &[0, 0, 138, 255, 255, 255, 255, 255, 224, 255, 255, 90, 0, 0],
            &[0, 0, 11, 160, 253, 255, 253, 168, 17, 229, 255, 90, 0, 0],
            &[0, 0, 0, 0, 10, 39, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'v' / 0x76
        'v' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 95, 95, 38, 0, 0, 0, 0, 90, 95, 90, 0, 0],
            &[0, 75, 255, 255, 159, 0, 0, 0, 41, 255, 255, 177, 0, 0],
            &[0, 4, 232, 255, 236, 3, 0, 0, 121, 255, 255, 81, 0, 0],
            &[0, 0, 141, 255, 255, 66, 0, 0, 202, 255, 236, 6, 0, 0],
            &[0, 0, 45, 255, 255, 147, 0, 27, 254, 255, 146, 0, 0, 0],
            &[0, 0, 0, 206, 255, 227, 1, 106, 255, 255, 50, 0, 0, 0],
            &[0, 0, 0, 111, 255, 255, 53, 186, 255, 211, 0, 0, 0, 0],
            &[0, 0, 0, 21, 250, 255, 132, 247, 255, 115, 0, 0, 0, 0],
            &[0, 0, 0, 0, 177, 255, 226, 255, 251, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 255, 255, 255, 180, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 236, 255, 255, 84, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'w' / 0x77
        'w' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 87, 95, 55, 0, 0, 0, 0, 0, 11, 95, 95, 32, 0],
            &[0, 209, 255, 164, 0, 0, 0, 0, 0, 47, 255, 255, 61, 0],
            &[0, 173, 255, 187, 0, 118, 218, 211, 2, 70, 255, 255, 25, 0],
            &[0, 137, 255, 209, 0, 186, 255, 255, 42, 92, 255, 244, 0, 0],
            &[0, 101, 255, 232, 0, 237, 255, 255, 93, 115, 255, 208, 0, 0],
            &[0, 65, 255, 252, 35, 255, 203, 255, 144, 138, 255, 172, 0, 0],
            &[
                0, 29, 255, 255, 105, 255, 122, 254, 194, 159, 255, 136, 0, 0,
            ],
            &[0, 1, 247, 255, 171, 255, 66, 220, 239, 180, 255, 100, 0, 0],
            &[0, 0, 213, 255, 231, 255, 19, 175, 255, 230, 255, 64, 0, 0],
            &[0, 0, 176, 255, 255, 228, 0, 129, 255, 255, 255, 28, 0, 0],
            &[0, 0, 140, 255, 255, 181, 0, 83, 255, 255, 246, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'x' / 0x78
        'x' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 95, 95, 91, 1, 0, 0, 38, 95, 95, 84, 0, 0],
            &[0, 6, 210, 255, 255, 92, 0, 2, 205, 255, 255, 104, 0, 0],
            &[0, 0, 47, 249, 255, 228, 11, 107, 255, 255, 179, 0, 0, 0],
            &[0, 0, 0, 121, 255, 255, 151, 239, 255, 233, 22, 0, 0, 0],
            &[0, 0, 0, 3, 198, 255, 255, 255, 255, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 255, 255, 167, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 126, 255, 255, 255, 231, 18, 0, 0, 0, 0],
            &[0, 0, 0, 53, 251, 255, 229, 255, 255, 167, 0, 0, 0, 0],
            &[0, 0, 9, 216, 255, 253, 53, 183, 255, 255, 84, 0, 0, 0],
            &[0, 0, 148, 255, 255, 152, 0, 41, 251, 255, 234, 22, 0, 0],
            &[0, 70, 254, 255, 235, 17, 0, 0, 146, 255, 255, 174, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'y' / 0x79
        'y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 95, 95, 42, 0, 0, 0, 1, 94, 95, 90, 0, 0],
            &[0, 69, 255, 255, 172, 0, 0, 0, 51, 255, 255, 179, 0, 0],
            &[0, 1, 222, 255, 246, 11, 0, 0, 127, 255, 255, 86, 0, 0],
            &[0, 0, 122, 255, 255, 87, 0, 0, 203, 255, 240, 8, 0, 0],
            &[0, 0, 25, 251, 255, 173, 0, 24, 254, 255, 156, 0, 0, 0],
            &[0, 0, 0, 175, 255, 246, 12, 99, 255, 255, 63, 0, 0, 0],
            &[0, 0, 0, 74, 255, 255, 87, 174, 255, 224, 1, 0, 0, 0],
            &[0, 0, 0, 2, 225, 255, 162, 241, 255, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 255, 242, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 252, 255, 255, 203, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 180, 255, 255, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 192, 255, 249, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 255, 255, 164, 0, 0, 0, 0, 0, 0],
            &[0, 0, 177, 210, 253, 255, 248, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 162, 134, 41, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'z' / 0x7a
        'z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 95, 95, 95, 95, 95, 95, 95, 95, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 254, 0, 0, 0],
            &[0, 0, 97, 202, 202, 202, 202, 232, 255, 255, 224, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 236, 255, 244, 48, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 209, 255, 254, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 122, 255, 255, 171, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 75, 253, 255, 206, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 240, 255, 246, 72, 42, 42, 42, 42, 7, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 195, 255, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '{' / 0x7b
        '{' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 94, 144, 126, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 199, 255, 255, 203, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 43, 255, 255, 201, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 255, 255, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 255, 255, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 255, 255, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 91, 255, 255, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 35, 117, 231, 255, 227, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 101, 255, 255, 186, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 79, 227, 255, 255, 155, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 146, 255, 255, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 255, 255, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 255, 255, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 255, 255, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 255, 255, 119, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 246, 255, 255, 202, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 74, 206, 252, 203, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 3, 11, 0, 0, 0, 0],
        ]),
        // letter: '|' / 0x7c
        '|' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 112, 83, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 143, 106, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '}' / 0x7d
        '}' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 64, 152, 120, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 101, 255, 255, 249, 54, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 159, 254, 255, 145, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 201, 255, 165, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 190, 255, 166, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 190, 255, 166, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 185, 255, 194, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 255, 255, 155, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 124, 247, 255, 203, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 243, 255, 245, 164, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 172, 255, 228, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 190, 255, 170, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 190, 255, 166, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 192, 255, 166, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 238, 255, 158, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 99, 255, 255, 255, 108, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 101, 255, 231, 142, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '~' / 0x7e
        '~' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 170, 152, 31, 0, 0, 63, 189, 12, 0, 0],
            &[0, 0, 59, 253, 196, 231, 235, 59, 0, 125, 242, 0, 0, 0],
            &[0, 0, 149, 213, 1, 23, 202, 247, 159, 240, 159, 0, 0, 0],
            &[0, 0, 153, 146, 0, 0, 8, 134, 205, 159, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ' ' / 0xa0
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¡' / 0xa1
        '¡' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 49, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 179, 255, 246, 43, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 247, 255, 255, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 155, 255, 232, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 74, 140, 127, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 143, 255, 241, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 155, 255, 252, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 255, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 177, 255, 255, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 188, 255, 255, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 200, 255, 255, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 255, 255, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 222, 255, 255, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 119, 131, 131, 37, 0, 0, 0, 0, 0],
        ]),
        // letter: '¢' / 0xa2
        '¢' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 193, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 106, 255, 100, 29, 0, 0, 0, 0],
            &[0, 0, 0, 1, 119, 236, 255, 255, 255, 255, 204, 1, 0, 0],
            &[0, 0, 0, 145, 255, 255, 255, 237, 247, 255, 171, 0, 0, 0],
            &[0, 0, 40, 254, 255, 209, 40, 0, 1, 46, 43, 0, 0, 0],
            &[0, 0, 119, 255, 255, 58, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 155, 255, 253, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 161, 255, 251, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 255, 255, 43, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 255, 255, 189, 16, 0, 0, 12, 71, 0, 0, 0],
            &[0, 0, 0, 182, 255, 255, 246, 203, 216, 248, 205, 0, 0, 0],
            &[0, 0, 0, 12, 165, 253, 255, 255, 255, 255, 182, 0, 0, 0],
            &[0, 0, 0, 0, 0, 27, 132, 255, 124, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 81, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 48, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '£' / 0xa3
        '£' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 132, 176, 176, 141, 57, 0, 0, 0],
            &[0, 0, 0, 0, 57, 243, 255, 255, 255, 255, 232, 4, 0, 0],
            &[0, 0, 0, 0, 193, 255, 255, 196, 146, 197, 140, 0, 0, 0],
            &[0, 0, 0, 0, 247, 255, 201, 0, 0, 0, 3, 0, 0, 0],
            &[0, 0, 0, 4, 255, 255, 164, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 255, 255, 164, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 236, 255, 255, 248, 236, 236, 174, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 189, 0, 0, 0, 0],
            &[0, 0, 6, 17, 255, 255, 169, 13, 13, 10, 0, 0, 0, 0],
            &[0, 0, 0, 11, 255, 255, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 64, 255, 255, 95, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 222, 255, 212, 88, 86, 86, 86, 86, 22, 0, 0],
            &[0, 0, 117, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 117, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¤' / 0xa4
        '¤' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 86, 0, 0, 0, 0, 0, 57, 34, 0, 0, 0],
            &[0, 0, 144, 255, 156, 179, 226, 203, 126, 242, 228, 14, 0, 0],
            &[0, 0, 39, 229, 255, 255, 255, 255, 255, 255, 104, 0, 0, 0],
            &[0, 0, 0, 212, 255, 125, 21, 60, 229, 255, 57, 0, 0, 0],
            &[0, 0, 28, 255, 226, 0, 0, 0, 123, 255, 125, 0, 0, 0],
            &[0, 0, 23, 255, 235, 1, 0, 0, 133, 255, 121, 0, 0, 0],
            &[0, 0, 0, 201, 255, 161, 60, 101, 244, 255, 44, 0, 0, 0],
            &[0, 0, 59, 241, 255, 255, 255, 255, 255, 255, 133, 0, 0, 0],
            &[0, 0, 118, 254, 119, 142, 187, 163, 94, 230, 202, 8, 0, 0],
            &[0, 0, 0, 48, 0, 0, 0, 0, 0, 36, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¥' / 0xa5
        '¥' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 144, 144, 57, 0, 0, 0, 4, 139, 144, 113, 0, 0],
            &[0, 14, 238, 255, 190, 0, 0, 0, 87, 255, 255, 98, 0, 0],
            &[0, 0, 125, 255, 255, 48, 0, 0, 201, 255, 219, 4, 0, 0],
            &[0, 0, 14, 238, 255, 162, 0, 60, 255, 255, 93, 0, 0, 0],
            &[0, 0, 0, 125, 255, 250, 26, 174, 255, 216, 3, 0, 0, 0],
            &[0, 0, 0, 14, 238, 255, 168, 253, 255, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 255, 255, 255, 213, 2, 0, 0, 0, 0],
            &[0, 0, 57, 176, 189, 255, 255, 255, 216, 176, 128, 0, 0, 0],
            &[0, 0, 80, 246, 246, 252, 255, 255, 246, 246, 179, 0, 0, 0],
            &[0, 0, 0, 0, 0, 151, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 190, 190, 229, 255, 255, 190, 190, 138, 0, 0, 0],
            &[0, 0, 74, 228, 228, 244, 255, 255, 228, 228, 165, 0, 0, 0],
            &[0, 0, 0, 0, 0, 151, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 151, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¦' / 0xa6
        '¦' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 102, 76, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 31, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 44, 32, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 94, 70, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '§' / 0xa7
        '§' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 88, 123, 120, 88, 21, 0, 0, 0, 0],
            &[0, 0, 0, 53, 236, 255, 255, 255, 255, 252, 35, 0, 0, 0],
            &[0, 0, 0, 195, 255, 217, 79, 76, 138, 171, 0, 0, 0, 0],
            &[0, 0, 0, 216, 255, 185, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 92, 253, 255, 216, 99, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 216, 255, 255, 255, 223, 61, 0, 0, 0, 0],
            &[0, 0, 0, 185, 255, 160, 80, 208, 255, 244, 38, 0, 0, 0],
            &[0, 0, 1, 251, 255, 84, 0, 12, 239, 255, 108, 0, 0, 0],
            &[0, 0, 0, 217, 255, 226, 67, 10, 240, 255, 83, 0, 0, 0],
            &[0, 0, 0, 42, 227, 255, 255, 232, 255, 175, 4, 0, 0, 0],
            &[0, 0, 0, 0, 11, 120, 235, 255, 255, 146, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 149, 255, 255, 47, 0, 0, 0],
            &[0, 0, 5, 63, 0, 0, 0, 52, 255, 255, 74, 0, 0, 0],
            &[0, 0, 11, 255, 223, 164, 151, 221, 255, 232, 17, 0, 0, 0],
            &[0, 0, 5, 166, 247, 255, 255, 255, 188, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 28, 39, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¨' / 0xa8
        '¨' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 75, 16, 0, 0, 57, 49, 0, 0, 0, 0],
            &[0, 0, 0, 184, 255, 199, 0, 74, 255, 255, 57, 0, 0, 0],
            &[0, 0, 0, 184, 255, 199, 0, 73, 255, 255, 58, 0, 0, 0],
            &[0, 0, 0, 16, 81, 18, 0, 0, 61, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '©' / 0xa9
        '©' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 91, 171, 200, 185, 132, 29, 0, 0, 0, 0],
            &[0, 0, 19, 197, 244, 156, 115, 131, 212, 242, 73, 0, 0, 0],
            &[0, 0, 177, 226, 44, 137, 210, 211, 104, 149, 242, 32, 0, 0],
            &[0, 43, 255, 81, 146, 249, 127, 115, 74, 4, 222, 140, 0, 0],
            &[0, 104, 248, 4, 238, 159, 0, 0, 0, 0, 147, 205, 0, 0],
            &[0, 121, 232, 5, 254, 126, 0, 0, 0, 0, 131, 224, 0, 0],
            &[0, 93, 252, 15, 223, 179, 0, 0, 0, 0, 167, 199, 0, 0],
            &[0, 20, 250, 115, 98, 254, 206, 193, 120, 28, 244, 122, 0, 0],
            &[0, 0, 131, 247, 91, 62, 130, 130, 80, 208, 220, 15, 0, 0],
            &[0, 0, 2, 138, 252, 224, 187, 205, 254, 201, 33, 0, 0, 0],
            &[0, 0, 0, 0, 29, 102, 129, 113, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ª' / 0xaa
        'ª' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 128, 168, 165, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 118, 232, 187, 234, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 4, 22, 96, 255, 158, 0, 0, 0, 0],
            &[0, 0, 0, 0, 134, 239, 253, 238, 255, 169, 0, 0, 0, 0],
            &[0, 0, 0, 50, 255, 191, 10, 52, 255, 169, 0, 0, 0, 0],
            &[0, 0, 0, 48, 255, 214, 105, 206, 253, 169, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 238, 218, 111, 151, 148, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '«' / 0xab
        '«' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 79, 3, 0, 0, 78, 6, 0, 0, 0],
            &[0, 0, 0, 0, 118, 255, 212, 9, 101, 255, 221, 17, 0, 0],
            &[0, 0, 0, 60, 251, 255, 129, 48, 247, 255, 147, 0, 0, 0],
            &[0, 0, 21, 229, 255, 203, 19, 219, 255, 217, 10, 0, 0, 0],
            &[0, 0, 177, 255, 253, 40, 160, 255, 255, 58, 0, 0, 0, 0],
            &[0, 0, 108, 255, 255, 104, 92, 255, 255, 124, 0, 0, 0, 0],
            &[0, 0, 0, 170, 255, 244, 37, 153, 255, 250, 51, 0, 0, 0],
            &[0, 0, 0, 14, 218, 255, 201, 12, 207, 255, 214, 8, 0, 0],
            &[0, 0, 0, 0, 47, 226, 111, 5, 36, 225, 120, 8, 0, 0],
            &[0, 0, 0, 0, 0, 2, 0, 0, 0, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¬' / 0xac
        '¬' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 182, 182, 182, 182, 182, 182, 182, 160, 0, 0, 0],
            &[0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 225, 0, 0, 0],
            &[0, 0, 51, 76, 76, 76, 76, 76, 104, 255, 225, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 39, 255, 225, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 39, 255, 225, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 28, 184, 162, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '®' / 0xae
        '®' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 90, 171, 200, 185, 130, 27, 0, 0, 0, 0],
            &[0, 0, 17, 195, 244, 157, 115, 131, 212, 240, 70, 0, 0, 0],
            &[0, 0, 174, 228, 101, 181, 178, 145, 34, 149, 240, 30, 0, 0],
            &[0, 40, 255, 86, 90, 255, 102, 221, 187, 4, 223, 137, 0, 0],
            &[0, 100, 251, 6, 90, 255, 14, 195, 196, 0, 151, 200, 0, 0],
            &[0, 116, 237, 0, 90, 255, 250, 255, 67, 0, 135, 219, 0, 0],
            &[0, 87, 254, 17, 90, 255, 29, 251, 74, 0, 172, 193, 0, 0],
            &[0, 16, 248, 119, 90, 255, 4, 179, 187, 31, 246, 118, 0, 0],
            &[0, 0, 122, 248, 121, 72, 1, 32, 113, 213, 219, 14, 0, 0],
            &[0, 0, 1, 129, 250, 228, 191, 210, 254, 198, 32, 0, 0, 0],
            &[0, 0, 0, 0, 24, 97, 124, 109, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¯' / 0xaf
        '¯' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 178, 178, 178, 178, 178, 105, 0, 0, 0, 0],
            &[0, 0, 0, 43, 255, 255, 255, 255, 255, 150, 0, 0, 0, 0],
            &[0, 0, 0, 7, 46, 46, 46, 46, 46, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '°' / 0xb0
        '°' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 123, 184, 159, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 184, 255, 243, 255, 246, 46, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 180, 7, 67, 255, 158, 0, 0, 0, 0],
            &[0, 0, 0, 71, 255, 129, 0, 21, 254, 179, 0, 0, 0, 0],
            &[0, 0, 0, 17, 243, 234, 118, 181, 255, 116, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 238, 255, 255, 171, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 49, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '±' / 0xb1
        '±' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 67, 233, 174, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 67, 106, 106, 149, 255, 217, 106, 106, 106, 7, 0, 0],
            &[0, 0, 161, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 96, 152, 152, 182, 255, 229, 152, 152, 152, 10, 0, 0],
            &[0, 0, 0, 0, 0, 73, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 8, 8, 12, 22, 18, 8, 8, 8, 0, 0, 0],
            &[0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '²' / 0xb2
        '²' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 49, 31, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 124, 246, 255, 255, 171, 2, 0, 0, 0, 0],
            &[0, 0, 0, 2, 190, 195, 131, 236, 255, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 2, 0, 173, 255, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 32, 239, 238, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 30, 218, 250, 71, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 225, 247, 75, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 229, 255, 200, 142, 142, 105, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 255, 255, 255, 189, 0, 0, 0, 0],
            &[0, 0, 0, 1, 24, 24, 24, 24, 24, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '³' / 0xb3
        '³' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 46, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 145, 250, 255, 254, 158, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 160, 111, 226, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 0, 0, 179, 255, 43, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 149, 214, 249, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 135, 200, 250, 209, 23, 0, 0, 0, 0],
            &[0, 0, 0, 2, 1, 0, 0, 128, 255, 121, 0, 0, 0, 0],
            &[0, 0, 0, 28, 209, 123, 104, 209, 255, 108, 0, 0, 0, 0],
            &[0, 0, 0, 18, 223, 255, 255, 255, 178, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 37, 55, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '´' / 0xb4
        '´' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 138, 141, 133, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 138, 255, 247, 74, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 61, 253, 240, 65, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 89, 124, 42, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'µ' / 0xb5
        'µ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 95, 95, 67, 0, 0, 0, 38, 95, 95, 33, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 105, 255, 255, 90, 0, 0],
            &[0, 16, 255, 255, 238, 0, 0, 0, 127, 255, 255, 90, 0, 0],
            &[0, 20, 255, 255, 255, 0, 0, 0, 170, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 255, 94, 32, 100, 252, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 255, 255, 255, 255, 224, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 255, 255, 253, 168, 17, 229, 255, 90, 0, 0],
            &[0, 12, 255, 255, 190, 39, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 143, 143, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¶' / 0xb6
        '¶' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 95, 112, 112, 112, 112, 112, 50, 0, 0],
            &[0, 0, 17, 188, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 161, 255, 255, 255, 255, 255, 177, 42, 255, 115, 0, 0],
            &[0, 11, 247, 255, 255, 255, 255, 255, 163, 3, 255, 115, 0, 0],
            &[0, 43, 255, 255, 255, 255, 255, 255, 163, 3, 255, 115, 0, 0],
            &[0, 55, 255, 255, 255, 255, 255, 255, 163, 3, 255, 115, 0, 0],
            &[0, 36, 255, 255, 255, 255, 255, 255, 163, 3, 255, 115, 0, 0],
            &[0, 1, 231, 255, 255, 255, 255, 255, 163, 3, 255, 115, 0, 0],
            &[0, 0, 107, 255, 255, 255, 255, 255, 163, 3, 255, 115, 0, 0],
            &[0, 0, 0, 93, 201, 241, 239, 247, 163, 3, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 206, 163, 3, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 206, 163, 3, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 206, 163, 3, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 206, 163, 3, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 206, 163, 3, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 206, 163, 3, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 206, 163, 3, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 85, 67, 1, 105, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '·' / 0xb7
        '·' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 62, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 199, 255, 249, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 251, 255, 255, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 148, 254, 217, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¸' / 0xb8
        '¸' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 255, 106, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 219, 255, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 87, 255, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 129, 132, 82, 2, 0, 0, 0, 0, 0],
        ]),
        // letter: '¹' / 0xb9
        '¹' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 237, 248, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 182, 255, 255, 248, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 193, 129, 212, 248, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 0, 215, 248, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 217, 248, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 217, 248, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 60, 255, 255, 64, 34, 0, 0, 0, 0],
            &[0, 0, 0, 0, 192, 255, 255, 255, 255, 213, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 24, 45, 48, 24, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'º' / 0xba
        'º' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 114, 171, 149, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 211, 247, 236, 25, 0, 0, 0, 0],
            &[0, 0, 0, 25, 255, 191, 0, 102, 255, 123, 0, 0, 0, 0],
            &[0, 0, 0, 57, 255, 148, 0, 57, 255, 157, 0, 0, 0, 0],
            &[0, 0, 0, 35, 255, 171, 0, 78, 255, 133, 0, 0, 0, 0],
            &[0, 0, 0, 1, 202, 249, 141, 215, 253, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 30, 180, 239, 214, 87, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '»' / 0xbb
        '»' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 29, 0, 0, 46, 37, 0, 0, 0, 0, 0],
            &[0, 0, 135, 255, 198, 5, 119, 253, 211, 9, 0, 0, 0, 0],
            &[0, 0, 52, 251, 255, 142, 39, 245, 255, 159, 0, 0, 0, 0],
            &[0, 0, 0, 126, 255, 255, 81, 106, 255, 255, 97, 0, 0, 0],
            &[0, 0, 0, 4, 208, 255, 239, 23, 190, 255, 246, 33, 0, 0],
            &[0, 0, 0, 34, 243, 255, 191, 25, 234, 255, 204, 7, 0, 0],
            &[0, 0, 2, 195, 255, 231, 24, 178, 255, 239, 33, 0, 0, 0],
            &[0, 0, 117, 255, 252, 65, 100, 255, 255, 80, 0, 0, 0, 0],
            &[0, 0, 67, 204, 124, 0, 58, 195, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 1, 0, 0, 0, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¼' / 0xbc
        '¼' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 138, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 114, 247, 255, 151, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 92, 140, 233, 151, 0, 0, 0, 0, 112, 185, 89, 0, 0],
            &[0, 0, 0, 236, 151, 0, 0, 0, 68, 253, 205, 7, 0, 0],
            &[0, 0, 0, 237, 151, 0, 0, 26, 233, 240, 35, 0, 0, 0],
            &[0, 39, 115, 255, 255, 108, 5, 193, 255, 83, 0, 0, 0, 0],
            &[0, 80, 204, 255, 255, 204, 138, 255, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 254, 202, 6, 77, 97, 0, 0, 0],
            &[0, 0, 0, 0, 29, 236, 239, 32, 64, 254, 254, 0, 0, 0],
            &[0, 0, 0, 4, 198, 255, 80, 11, 221, 222, 254, 0, 0, 0],
            &[0, 0, 0, 140, 255, 142, 0, 146, 193, 140, 254, 0, 0, 0],
            &[0, 0, 78, 254, 199, 5, 50, 254, 150, 191, 255, 113, 0, 0],
            &[0, 3, 139, 156, 30, 0, 67, 210, 210, 235, 255, 206, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 139, 254, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '½' / 0xbd
        '½' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 138, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 114, 247, 255, 151, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 92, 140, 233, 151, 0, 0, 0, 0, 112, 185, 89, 0, 0],
            &[0, 0, 0, 236, 151, 0, 0, 0, 68, 253, 205, 7, 0, 0],
            &[0, 0, 0, 237, 151, 0, 0, 26, 233, 240, 35, 0, 0, 0],
            &[0, 39, 115, 255, 255, 108, 5, 193, 255, 83, 0, 0, 0, 0],
            &[0, 80, 204, 255, 255, 204, 138, 255, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 254, 202, 65, 121, 104, 10, 0, 0],
            &[0, 0, 0, 0, 29, 236, 239, 101, 254, 226, 255, 160, 0, 0],
            &[0, 0, 0, 4, 198, 255, 80, 0, 50, 0, 215, 201, 0, 0],
            &[0, 0, 0, 140, 255, 142, 0, 0, 0, 84, 254, 114, 0, 0],
            &[0, 0, 78, 254, 199, 5, 0, 0, 78, 250, 162, 2, 0, 0],
            &[0, 3, 139, 156, 30, 0, 0, 62, 249, 222, 101, 99, 9, 0],
            &[0, 0, 0, 0, 0, 0, 0, 118, 255, 255, 255, 255, 24, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¾' / 0xbe
        '¾' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 111, 173, 163, 49, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 36, 225, 158, 233, 223, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 31, 193, 204, 0, 0, 0, 112, 185, 89, 0, 0],
            &[0, 0, 79, 255, 255, 84, 0, 0, 68, 253, 205, 7, 0, 0],
            &[0, 0, 10, 42, 170, 247, 10, 26, 233, 240, 35, 0, 0, 0],
            &[0, 72, 170, 128, 215, 248, 16, 193, 255, 83, 0, 0, 0, 0],
            &[0, 37, 186, 221, 197, 76, 134, 255, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 254, 202, 6, 77, 97, 0, 0, 0],
            &[0, 0, 0, 0, 29, 236, 239, 32, 64, 254, 254, 0, 0, 0],
            &[0, 0, 0, 4, 198, 255, 80, 11, 221, 222, 254, 0, 0, 0],
            &[0, 0, 0, 140, 255, 142, 0, 146, 193, 140, 254, 0, 0, 0],
            &[0, 0, 78, 254, 199, 5, 50, 254, 150, 191, 255, 113, 0, 0],
            &[0, 3, 139, 156, 30, 0, 67, 210, 210, 235, 255, 206, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 139, 254, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¿' / 0xbf
        '¿' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 48, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 164, 255, 249, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 233, 255, 255, 114, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 141, 255, 236, 43, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 14, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 140, 140, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 143, 255, 242, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 249, 255, 173, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 132, 255, 255, 207, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 105, 255, 255, 178, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 205, 255, 232, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 229, 255, 213, 0, 0, 0, 31, 57, 0, 0, 0, 0],
            &[0, 0, 181, 255, 255, 194, 158, 197, 253, 225, 6, 0, 0, 0],
            &[0, 0, 42, 234, 255, 255, 255, 255, 255, 239, 51, 0, 0, 0],
            &[0, 0, 0, 18, 116, 166, 174, 147, 88, 11, 0, 0, 0, 0],
        ]),
        // letter: 'À' / 0xc0
        'À' => Some(&[
            &[0, 0, 0, 57, 190, 190, 137, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 255, 255, 71, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 101, 251, 228, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 75, 27, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 159, 159, 159, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 255, 255, 255, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 239, 255, 250, 13, 0, 0, 0, 0],
            &[0, 0, 0, 1, 234, 255, 137, 254, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 255, 60, 212, 255, 150, 0, 0, 0, 0],
            &[0, 0, 0, 121, 255, 246, 6, 151, 255, 220, 0, 0, 0, 0],
            &[0, 0, 0, 191, 255, 189, 0, 90, 255, 255, 35, 0, 0, 0],
            &[0, 0, 12, 249, 255, 125, 0, 28, 255, 255, 106, 0, 0, 0],
            &[0, 0, 76, 255, 255, 226, 164, 164, 255, 255, 176, 0, 0, 0],
            &[0, 0, 146, 255, 255, 255, 255, 255, 255, 255, 242, 5, 0, 0],
            &[0, 0, 217, 255, 255, 163, 163, 163, 246, 255, 255, 62, 0, 0],
            &[0, 31, 255, 255, 126, 0, 0, 0, 37, 255, 255, 132, 0, 0],
            &[0, 101, 255, 255, 62, 0, 0, 0, 0, 231, 255, 203, 0, 0],
            &[0, 172, 255, 247, 7, 0, 0, 0, 0, 170, 255, 253, 20, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Á' / 0xc1
        'Á' => Some(&[
            &[0, 0, 0, 0, 0, 0, 7, 175, 190, 180, 21, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 132, 255, 244, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 252, 234, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 75, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 159, 159, 159, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 255, 255, 255, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 239, 255, 250, 13, 0, 0, 0, 0],
            &[0, 0, 0, 1, 234, 255, 137, 254, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 255, 60, 212, 255, 150, 0, 0, 0, 0],
            &[0, 0, 0, 121, 255, 246, 6, 151, 255, 220, 0, 0, 0, 0],
            &[0, 0, 0, 191, 255, 189, 0, 90, 255, 255, 35, 0, 0, 0],
            &[0, 0, 12, 249, 255, 125, 0, 28, 255, 255, 106, 0, 0, 0],
            &[0, 0, 76, 255, 255, 226, 164, 164, 255, 255, 176, 0, 0, 0],
            &[0, 0, 146, 255, 255, 255, 255, 255, 255, 255, 242, 5, 0, 0],
            &[0, 0, 217, 255, 255, 163, 163, 163, 246, 255, 255, 62, 0, 0],
            &[0, 31, 255, 255, 126, 0, 0, 0, 37, 255, 255, 132, 0, 0],
            &[0, 101, 255, 255, 62, 0, 0, 0, 0, 231, 255, 203, 0, 0],
            &[0, 172, 255, 247, 7, 0, 0, 0, 0, 170, 255, 253, 20, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Â' / 0xc2
        'Â' => Some(&[
            &[0, 0, 0, 0, 5, 171, 190, 190, 66, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 255, 232, 255, 226, 19, 0, 0, 0, 0],
            &[0, 0, 0, 82, 254, 201, 19, 119, 254, 184, 3, 0, 0, 0],
            &[0, 0, 0, 58, 74, 8, 0, 0, 51, 75, 13, 0, 0, 0],
            &[0, 0, 0, 0, 24, 159, 159, 159, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 255, 255, 255, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 239, 255, 250, 13, 0, 0, 0, 0],
            &[0, 0, 0, 1, 234, 255, 137, 254, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 255, 60, 212, 255, 150, 0, 0, 0, 0],
            &[0, 0, 0, 121, 255, 246, 6, 151, 255, 220, 0, 0, 0, 0],
            &[0, 0, 0, 191, 255, 189, 0, 90, 255, 255, 35, 0, 0, 0],
            &[0, 0, 12, 249, 255, 125, 0, 28, 255, 255, 106, 0, 0, 0],
            &[0, 0, 76, 255, 255, 226, 164, 164, 255, 255, 176, 0, 0, 0],
            &[0, 0, 146, 255, 255, 255, 255, 255, 255, 255, 242, 5, 0, 0],
            &[0, 0, 217, 255, 255, 163, 163, 163, 246, 255, 255, 62, 0, 0],
            &[0, 31, 255, 255, 126, 0, 0, 0, 37, 255, 255, 132, 0, 0],
            &[0, 101, 255, 255, 62, 0, 0, 0, 0, 231, 255, 203, 0, 0],
            &[0, 172, 255, 247, 7, 0, 0, 0, 0, 170, 255, 253, 20, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ã' / 0xc3
        'Ã' => Some(&[
            &[0, 0, 0, 0, 85, 121, 42, 0, 22, 142, 37, 0, 0, 0],
            &[0, 0, 0, 101, 255, 255, 254, 188, 196, 255, 31, 0, 0, 0],
            &[0, 0, 0, 197, 210, 102, 190, 255, 255, 160, 0, 0, 0, 0],
            &[0, 0, 0, 70, 40, 0, 0, 36, 56, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 159, 159, 159, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 255, 255, 255, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 239, 255, 250, 13, 0, 0, 0, 0],
            &[0, 0, 0, 1, 234, 255, 137, 254, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 255, 60, 212, 255, 150, 0, 0, 0, 0],
            &[0, 0, 0, 121, 255, 246, 6, 151, 255, 220, 0, 0, 0, 0],
            &[0, 0, 0, 191, 255, 189, 0, 90, 255, 255, 35, 0, 0, 0],
            &[0, 0, 12, 249, 255, 125, 0, 28, 255, 255, 106, 0, 0, 0],
            &[0, 0, 76, 255, 255, 226, 164, 164, 255, 255, 176, 0, 0, 0],
            &[0, 0, 146, 255, 255, 255, 255, 255, 255, 255, 242, 5, 0, 0],
            &[0, 0, 217, 255, 255, 163, 163, 163, 246, 255, 255, 62, 0, 0],
            &[0, 31, 255, 255, 126, 0, 0, 0, 37, 255, 255, 132, 0, 0],
            &[0, 101, 255, 255, 62, 0, 0, 0, 0, 231, 255, 203, 0, 0],
            &[0, 172, 255, 247, 7, 0, 0, 0, 0, 170, 255, 253, 20, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ä' / 0xc4
        'Ä' => Some(&[
            &[0, 0, 0, 48, 124, 40, 0, 9, 111, 92, 0, 0, 0, 0],
            &[0, 0, 0, 212, 255, 199, 0, 102, 255, 255, 57, 0, 0, 0],
            &[0, 0, 0, 175, 255, 159, 0, 67, 251, 242, 30, 0, 0, 0],
            &[0, 0, 0, 0, 32, 0, 0, 0, 21, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 159, 159, 159, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 255, 255, 255, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 239, 255, 250, 13, 0, 0, 0, 0],
            &[0, 0, 0, 1, 234, 255, 137, 254, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 255, 60, 212, 255, 150, 0, 0, 0, 0],
            &[0, 0, 0, 121, 255, 246, 6, 151, 255, 220, 0, 0, 0, 0],
            &[0, 0, 0, 191, 255, 189, 0, 90, 255, 255, 35, 0, 0, 0],
            &[0, 0, 12, 249, 255, 125, 0, 28, 255, 255, 106, 0, 0, 0],
            &[0, 0, 76, 255, 255, 226, 164, 164, 255, 255, 176, 0, 0, 0],
            &[0, 0, 146, 255, 255, 255, 255, 255, 255, 255, 242, 5, 0, 0],
            &[0, 0, 217, 255, 255, 163, 163, 163, 246, 255, 255, 62, 0, 0],
            &[0, 31, 255, 255, 126, 0, 0, 0, 37, 255, 255, 132, 0, 0],
            &[0, 101, 255, 255, 62, 0, 0, 0, 0, 231, 255, 203, 0, 0],
            &[0, 172, 255, 247, 7, 0, 0, 0, 0, 170, 255, 253, 20, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Å' / 0xc5
        'Å' => Some(&[
            &[0, 0, 0, 0, 119, 246, 87, 195, 214, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 234, 30, 166, 226, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 233, 255, 252, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 63, 32, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 159, 159, 159, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 255, 255, 255, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 239, 255, 250, 13, 0, 0, 0, 0],
            &[0, 0, 0, 1, 234, 255, 137, 254, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 255, 60, 212, 255, 150, 0, 0, 0, 0],
            &[0, 0, 0, 121, 255, 246, 6, 151, 255, 220, 0, 0, 0, 0],
            &[0, 0, 0, 191, 255, 189, 0, 90, 255, 255, 35, 0, 0, 0],
            &[0, 0, 12, 249, 255, 125, 0, 28, 255, 255, 106, 0, 0, 0],
            &[0, 0, 76, 255, 255, 226, 164, 164, 255, 255, 176, 0, 0, 0],
            &[0, 0, 146, 255, 255, 255, 255, 255, 255, 255, 242, 5, 0, 0],
            &[0, 0, 217, 255, 255, 163, 163, 163, 246, 255, 255, 62, 0, 0],
            &[0, 31, 255, 255, 126, 0, 0, 0, 37, 255, 255, 132, 0, 0],
            &[0, 101, 255, 255, 62, 0, 0, 0, 0, 231, 255, 203, 0, 0],
            &[0, 172, 255, 247, 7, 0, 0, 0, 0, 170, 255, 253, 20, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 194, 255, 228, 65, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Æ' / 0xc6
        'Æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 144, 144, 144, 144, 144, 144, 120, 0, 0],
            &[0, 0, 0, 0, 213, 255, 255, 255, 255, 255, 255, 212, 0, 0],
            &[0, 0, 0, 19, 254, 255, 218, 255, 255, 211, 202, 167, 0, 0],
            &[0, 0, 0, 79, 255, 253, 47, 255, 255, 43, 0, 0, 0, 0],
            &[0, 0, 0, 140, 255, 212, 33, 255, 255, 43, 0, 0, 0, 0],
            &[0, 0, 0, 201, 255, 155, 33, 255, 255, 91, 58, 28, 0, 0],
            &[0, 0, 11, 251, 255, 99, 33, 255, 255, 255, 255, 124, 0, 0],
            &[0, 0, 67, 255, 255, 43, 33, 255, 255, 255, 255, 124, 0, 0],
            &[0, 0, 128, 255, 255, 181, 212, 255, 255, 70, 33, 16, 0, 0],
            &[0, 0, 189, 255, 255, 255, 255, 255, 255, 43, 0, 0, 0, 0],
            &[0, 4, 245, 255, 255, 167, 200, 255, 255, 43, 0, 0, 0, 0],
            &[0, 54, 255, 255, 73, 0, 33, 255, 255, 119, 91, 75, 0, 0],
            &[0, 115, 255, 254, 18, 0, 33, 255, 255, 255, 255, 212, 0, 0],
            &[0, 176, 255, 216, 0, 0, 33, 255, 255, 255, 255, 212, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ç' / 0xc7
        'Ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 47, 132, 174, 186, 160, 104, 16, 0, 0],
            &[0, 0, 0, 5, 155, 254, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 0, 157, 255, 255, 248, 190, 176, 215, 248, 19, 0, 0],
            &[0, 0, 57, 255, 255, 214, 29, 0, 0, 0, 31, 0, 0, 0],
            &[0, 0, 157, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 219, 255, 238, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 211, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 255, 255, 195, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 247, 255, 210, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 217, 255, 249, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 251, 255, 253, 147, 74, 69, 109, 195, 26, 0, 0],
            &[0, 0, 0, 120, 255, 255, 255, 255, 255, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 86, 208, 255, 255, 255, 244, 173, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 151, 255, 56, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 245, 240, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 172, 255, 100, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 137, 123, 46, 0, 0, 0, 0, 0],
        ]),
        // letter: 'È' / 0xc8
        'È' => Some(&[
            &[0, 0, 0, 38, 188, 190, 158, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 252, 255, 99, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 78, 245, 242, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 38, 75, 36, 0, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 144, 144, 144, 144, 144, 10, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 215, 207, 207, 207, 207, 207, 14, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 93, 58, 58, 58, 58, 45, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 77, 37, 37, 37, 37, 29, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 124, 95, 95, 95, 95, 95, 6, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'É' / 0xc9
        'É' => Some(&[
            &[0, 0, 0, 0, 0, 0, 1, 159, 190, 187, 36, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 103, 255, 251, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 244, 243, 74, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 75, 37, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 144, 144, 144, 144, 144, 10, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 215, 207, 207, 207, 207, 207, 14, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 93, 58, 58, 58, 58, 45, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 77, 37, 37, 37, 37, 29, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 124, 95, 95, 95, 95, 95, 6, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ê' / 0xca
        'Ê' => Some(&[
            &[0, 0, 0, 0, 0, 154, 190, 190, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 255, 233, 254, 240, 34, 0, 0, 0, 0],
            &[0, 0, 0, 58, 248, 218, 32, 94, 249, 206, 10, 0, 0, 0],
            &[0, 0, 0, 49, 75, 15, 0, 0, 42, 75, 21, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 144, 144, 144, 144, 144, 10, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 215, 207, 207, 207, 207, 207, 14, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 93, 58, 58, 58, 58, 45, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 77, 37, 37, 37, 37, 29, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 124, 95, 95, 95, 95, 95, 6, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ë' / 0xcb
        'Ë' => Some(&[
            &[0, 0, 0, 36, 124, 52, 0, 2, 101, 104, 5, 0, 0, 0],
            &[0, 0, 0, 183, 255, 228, 0, 73, 255, 255, 86, 0, 0, 0],
            &[0, 0, 0, 146, 255, 188, 0, 42, 247, 248, 53, 0, 0, 0],
            &[0, 0, 0, 0, 32, 1, 0, 0, 15, 17, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 144, 144, 144, 144, 144, 10, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 215, 207, 207, 207, 207, 207, 14, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 93, 58, 58, 58, 58, 45, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 77, 37, 37, 37, 37, 29, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 124, 95, 95, 95, 95, 95, 6, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ì' / 0xcc
        'Ì' => Some(&[
            &[0, 0, 0, 57, 190, 190, 137, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 255, 255, 71, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 101, 251, 228, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 75, 27, 0, 0, 0, 0, 0],
            &[0, 0, 42, 144, 144, 144, 144, 144, 144, 144, 99, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 26, 108, 138, 221, 255, 255, 148, 119, 65, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 27, 182, 255, 254, 37, 9, 0, 0, 0, 0],
            &[0, 0, 68, 250, 255, 255, 255, 255, 255, 255, 165, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Í' / 0xcd
        'Í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 7, 175, 190, 180, 21, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 132, 255, 244, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 252, 234, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 75, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 144, 144, 144, 144, 144, 144, 144, 99, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 26, 108, 138, 221, 255, 255, 148, 119, 65, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 27, 182, 255, 254, 37, 9, 0, 0, 0, 0],
            &[0, 0, 68, 250, 255, 255, 255, 255, 255, 255, 165, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Î' / 0xce
        'Î' => Some(&[
            &[0, 0, 0, 0, 5, 171, 190, 190, 66, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 255, 232, 255, 226, 19, 0, 0, 0, 0],
            &[0, 0, 0, 82, 254, 201, 19, 119, 254, 184, 3, 0, 0, 0],
            &[0, 0, 0, 58, 74, 8, 0, 0, 51, 75, 13, 0, 0, 0],
            &[0, 0, 42, 144, 144, 144, 144, 144, 144, 144, 99, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 26, 108, 138, 221, 255, 255, 148, 119, 65, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 27, 182, 255, 254, 37, 9, 0, 0, 0, 0],
            &[0, 0, 68, 250, 255, 255, 255, 255, 255, 255, 165, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ï' / 0xcf
        'Ï' => Some(&[
            &[0, 0, 0, 48, 124, 40, 0, 9, 111, 92, 0, 0, 0, 0],
            &[0, 0, 0, 212, 255, 199, 0, 102, 255, 255, 57, 0, 0, 0],
            &[0, 0, 0, 175, 255, 159, 0, 67, 251, 242, 30, 0, 0, 0],
            &[0, 0, 0, 0, 32, 0, 0, 0, 21, 11, 0, 0, 0, 0],
            &[0, 0, 42, 144, 144, 144, 144, 144, 144, 144, 99, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 26, 108, 138, 221, 255, 255, 148, 119, 65, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 27, 182, 255, 254, 37, 9, 0, 0, 0, 0],
            &[0, 0, 68, 250, 255, 255, 255, 255, 255, 255, 165, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ð' / 0xd0
        'Ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 144, 144, 131, 99, 21, 0, 0, 0, 0, 0],
            &[0, 37, 255, 255, 255, 255, 255, 255, 249, 127, 1, 0, 0, 0],
            &[0, 37, 255, 255, 230, 197, 215, 254, 255, 255, 130, 0, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 50, 233, 255, 251, 34, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 104, 255, 255, 119, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 29, 255, 255, 176, 0, 0],
            &[0, 253, 255, 255, 255, 216, 127, 0, 2, 254, 255, 198, 0, 0],
            &[0, 255, 255, 255, 255, 255, 149, 0, 0, 251, 255, 203, 0, 0],
            &[0, 147, 255, 255, 255, 110, 64, 0, 16, 255, 255, 180, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 73, 255, 255, 136, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 5, 195, 255, 255, 54, 0, 0],
            &[0, 37, 255, 255, 183, 87, 111, 204, 255, 255, 175, 0, 0, 0],
            &[0, 37, 255, 255, 255, 255, 255, 255, 255, 199, 20, 0, 0, 0],
            &[0, 37, 255, 255, 255, 255, 236, 199, 106, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ñ' / 0xd1
        'Ñ' => Some(&[
            &[0, 0, 0, 0, 73, 123, 52, 0, 9, 141, 51, 0, 0, 0],
            &[0, 0, 0, 77, 255, 255, 255, 196, 187, 255, 55, 0, 0, 0],
            &[0, 0, 0, 173, 224, 101, 179, 254, 255, 184, 0, 0, 0, 0],
            &[0, 0, 0, 63, 48, 0, 0, 30, 59, 4, 0, 0, 0, 0],
            &[0, 21, 144, 144, 144, 100, 0, 0, 0, 131, 144, 78, 0, 0],
            &[0, 37, 255, 255, 255, 237, 4, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 255, 255, 69, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 200, 255, 153, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 132, 255, 234, 3, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 74, 244, 255, 65, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 75, 170, 255, 148, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 87, 255, 230, 2, 230, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 12, 247, 255, 60, 226, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 177, 255, 143, 219, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 94, 255, 226, 210, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 17, 250, 255, 246, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 0, 184, 255, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 0, 101, 255, 255, 255, 139, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ò' / 0xd2
        'Ò' => Some(&[
            &[0, 0, 0, 57, 190, 190, 137, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 255, 255, 71, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 101, 251, 228, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 75, 27, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 77, 163, 194, 183, 124, 19, 0, 0, 0, 0],
            &[0, 0, 2, 164, 255, 255, 255, 255, 255, 232, 42, 0, 0, 0],
            &[0, 0, 105, 255, 255, 236, 162, 208, 255, 255, 210, 2, 0, 0],
            &[0, 0, 223, 255, 245, 35, 0, 2, 185, 255, 255, 72, 0, 0],
            &[0, 34, 255, 255, 169, 0, 0, 0, 76, 255, 255, 143, 0, 0],
            &[0, 80, 255, 255, 120, 0, 0, 0, 21, 255, 255, 184, 0, 0],
            &[0, 98, 255, 255, 100, 0, 0, 0, 3, 254, 255, 203, 0, 0],
            &[0, 105, 255, 255, 96, 0, 0, 0, 0, 250, 255, 208, 0, 0],
            &[0, 92, 255, 255, 109, 0, 0, 0, 7, 255, 255, 195, 0, 0],
            &[0, 59, 255, 255, 145, 0, 0, 0, 45, 255, 255, 161, 0, 0],
            &[0, 11, 246, 255, 221, 4, 0, 0, 126, 255, 255, 104, 0, 0],
            &[0, 0, 163, 255, 255, 166, 60, 102, 246, 255, 244, 21, 0, 0],
            &[0, 0, 30, 233, 255, 255, 255, 255, 255, 255, 109, 0, 0, 0],
            &[0, 0, 0, 38, 184, 253, 255, 255, 224, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 41, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ó' / 0xd3
        'Ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 7, 175, 190, 180, 21, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 132, 255, 244, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 252, 234, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 75, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 77, 163, 194, 183, 124, 19, 0, 0, 0, 0],
            &[0, 0, 2, 164, 255, 255, 255, 255, 255, 232, 42, 0, 0, 0],
            &[0, 0, 105, 255, 255, 236, 162, 208, 255, 255, 210, 2, 0, 0],
            &[0, 0, 223, 255, 245, 35, 0, 2, 185, 255, 255, 72, 0, 0],
            &[0, 34, 255, 255, 169, 0, 0, 0, 76, 255, 255, 143, 0, 0],
            &[0, 80, 255, 255, 120, 0, 0, 0, 21, 255, 255, 184, 0, 0],
            &[0, 98, 255, 255, 100, 0, 0, 0, 3, 254, 255, 203, 0, 0],
            &[0, 105, 255, 255, 96, 0, 0, 0, 0, 250, 255, 208, 0, 0],
            &[0, 92, 255, 255, 109, 0, 0, 0, 7, 255, 255, 195, 0, 0],
            &[0, 59, 255, 255, 145, 0, 0, 0, 45, 255, 255, 161, 0, 0],
            &[0, 11, 246, 255, 221, 4, 0, 0, 126, 255, 255, 104, 0, 0],
            &[0, 0, 163, 255, 255, 166, 60, 102, 246, 255, 244, 21, 0, 0],
            &[0, 0, 30, 233, 255, 255, 255, 255, 255, 255, 109, 0, 0, 0],
            &[0, 0, 0, 38, 184, 253, 255, 255, 224, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 41, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ô' / 0xd4
        'Ô' => Some(&[
            &[0, 0, 0, 0, 5, 171, 190, 190, 66, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 255, 232, 255, 226, 19, 0, 0, 0, 0],
            &[0, 0, 0, 82, 254, 201, 19, 119, 254, 184, 3, 0, 0, 0],
            &[0, 0, 0, 58, 74, 8, 0, 0, 51, 75, 13, 0, 0, 0],
            &[0, 0, 0, 1, 77, 163, 194, 183, 124, 19, 0, 0, 0, 0],
            &[0, 0, 2, 164, 255, 255, 255, 255, 255, 232, 42, 0, 0, 0],
            &[0, 0, 105, 255, 255, 236, 162, 208, 255, 255, 210, 2, 0, 0],
            &[0, 0, 223, 255, 245, 35, 0, 2, 185, 255, 255, 72, 0, 0],
            &[0, 34, 255, 255, 169, 0, 0, 0, 76, 255, 255, 143, 0, 0],
            &[0, 80, 255, 255, 120, 0, 0, 0, 21, 255, 255, 184, 0, 0],
            &[0, 98, 255, 255, 100, 0, 0, 0, 3, 254, 255, 203, 0, 0],
            &[0, 105, 255, 255, 96, 0, 0, 0, 0, 250, 255, 208, 0, 0],
            &[0, 92, 255, 255, 109, 0, 0, 0, 7, 255, 255, 195, 0, 0],
            &[0, 59, 255, 255, 145, 0, 0, 0, 45, 255, 255, 161, 0, 0],
            &[0, 11, 246, 255, 221, 4, 0, 0, 126, 255, 255, 104, 0, 0],
            &[0, 0, 163, 255, 255, 166, 60, 102, 246, 255, 244, 21, 0, 0],
            &[0, 0, 30, 233, 255, 255, 255, 255, 255, 255, 109, 0, 0, 0],
            &[0, 0, 0, 38, 184, 253, 255, 255, 224, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 41, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Õ' / 0xd5
        'Õ' => Some(&[
            &[0, 0, 0, 0, 85, 121, 42, 0, 22, 142, 37, 0, 0, 0],
            &[0, 0, 0, 101, 255, 255, 254, 188, 196, 255, 31, 0, 0, 0],
            &[0, 0, 0, 197, 210, 102, 190, 255, 255, 160, 0, 0, 0, 0],
            &[0, 0, 0, 70, 40, 0, 0, 36, 56, 1, 0, 0, 0, 0],
            &[0, 0, 0, 1, 77, 163, 194, 183, 124, 19, 0, 0, 0, 0],
            &[0, 0, 2, 164, 255, 255, 255, 255, 255, 232, 42, 0, 0, 0],
            &[0, 0, 105, 255, 255, 236, 162, 208, 255, 255, 210, 2, 0, 0],
            &[0, 0, 223, 255, 245, 35, 0, 2, 185, 255, 255, 72, 0, 0],
            &[0, 34, 255, 255, 169, 0, 0, 0, 76, 255, 255, 143, 0, 0],
            &[0, 80, 255, 255, 120, 0, 0, 0, 21, 255, 255, 184, 0, 0],
            &[0, 98, 255, 255, 100, 0, 0, 0, 3, 254, 255, 203, 0, 0],
            &[0, 105, 255, 255, 96, 0, 0, 0, 0, 250, 255, 208, 0, 0],
            &[0, 92, 255, 255, 109, 0, 0, 0, 7, 255, 255, 195, 0, 0],
            &[0, 59, 255, 255, 145, 0, 0, 0, 45, 255, 255, 161, 0, 0],
            &[0, 11, 246, 255, 221, 4, 0, 0, 126, 255, 255, 104, 0, 0],
            &[0, 0, 163, 255, 255, 166, 60, 102, 246, 255, 244, 21, 0, 0],
            &[0, 0, 30, 233, 255, 255, 255, 255, 255, 255, 109, 0, 0, 0],
            &[0, 0, 0, 38, 184, 253, 255, 255, 224, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 41, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ö' / 0xd6
        'Ö' => Some(&[
            &[0, 0, 0, 48, 124, 40, 0, 9, 111, 92, 0, 0, 0, 0],
            &[0, 0, 0, 212, 255, 199, 0, 102, 255, 255, 57, 0, 0, 0],
            &[0, 0, 0, 175, 255, 159, 0, 67, 251, 242, 30, 0, 0, 0],
            &[0, 0, 0, 0, 32, 0, 0, 0, 21, 11, 0, 0, 0, 0],
            &[0, 0, 0, 1, 77, 163, 194, 183, 124, 19, 0, 0, 0, 0],
            &[0, 0, 2, 164, 255, 255, 255, 255, 255, 232, 42, 0, 0, 0],
            &[0, 0, 105, 255, 255, 236, 162, 208, 255, 255, 210, 2, 0, 0],
            &[0, 0, 223, 255, 245, 35, 0, 2, 185, 255, 255, 72, 0, 0],
            &[0, 34, 255, 255, 169, 0, 0, 0, 76, 255, 255, 143, 0, 0],
            &[0, 80, 255, 255, 120, 0, 0, 0, 21, 255, 255, 184, 0, 0],
            &[0, 98, 255, 255, 100, 0, 0, 0, 3, 254, 255, 203, 0, 0],
            &[0, 105, 255, 255, 96, 0, 0, 0, 0, 250, 255, 208, 0, 0],
            &[0, 92, 255, 255, 109, 0, 0, 0, 7, 255, 255, 195, 0, 0],
            &[0, 59, 255, 255, 145, 0, 0, 0, 45, 255, 255, 161, 0, 0],
            &[0, 11, 246, 255, 221, 4, 0, 0, 126, 255, 255, 104, 0, 0],
            &[0, 0, 163, 255, 255, 166, 60, 102, 246, 255, 244, 21, 0, 0],
            &[0, 0, 30, 233, 255, 255, 255, 255, 255, 255, 109, 0, 0, 0],
            &[0, 0, 0, 38, 184, 253, 255, 255, 224, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 41, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '×' / 0xd7
        '×' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 106, 94, 0, 0, 0, 27, 162, 14, 0, 0, 0],
            &[0, 0, 59, 254, 253, 99, 0, 30, 220, 255, 164, 0, 0, 0],
            &[0, 0, 0, 146, 255, 254, 125, 223, 255, 218, 26, 0, 0, 0],
            &[0, 0, 0, 0, 146, 255, 255, 255, 218, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 30, 243, 255, 255, 115, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 220, 255, 240, 255, 253, 92, 0, 0, 0, 0],
            &[0, 0, 26, 218, 255, 217, 26, 141, 255, 252, 89, 0, 0, 0],
            &[0, 0, 29, 222, 216, 25, 0, 0, 141, 254, 101, 0, 0, 0],
            &[0, 0, 0, 32, 24, 0, 0, 0, 0, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ø' / 0xd8
        'Ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 18, 1, 0, 0],
            &[0, 0, 0, 1, 77, 163, 194, 183, 124, 22, 198, 192, 8, 0],
            &[0, 0, 2, 164, 255, 255, 255, 255, 255, 255, 255, 159, 0, 0],
            &[0, 0, 105, 255, 255, 236, 162, 208, 255, 255, 255, 18, 0, 0],
            &[0, 0, 223, 255, 245, 35, 0, 6, 255, 255, 255, 72, 0, 0],
            &[0, 34, 255, 255, 169, 0, 0, 122, 255, 255, 255, 143, 0, 0],
            &[0, 80, 255, 255, 120, 0, 45, 249, 248, 255, 255, 184, 0, 0],
            &[0, 98, 255, 255, 100, 4, 206, 255, 75, 254, 255, 203, 0, 0],
            &[0, 105, 255, 255, 96, 127, 255, 155, 0, 250, 255, 208, 0, 0],
            &[0, 92, 255, 255, 158, 250, 225, 14, 7, 255, 255, 195, 0, 0],
            &[0, 59, 255, 255, 255, 255, 69, 0, 45, 255, 255, 161, 0, 0],
            &[0, 11, 246, 255, 255, 157, 0, 0, 126, 255, 255, 104, 0, 0],
            &[0, 0, 215, 255, 255, 179, 60, 102, 246, 255, 244, 21, 0, 0],
            &[0, 7, 242, 255, 255, 255, 255, 255, 255, 255, 109, 0, 0, 0],
            &[0, 128, 255, 189, 184, 253, 255, 255, 224, 97, 0, 0, 0, 0],
            &[0, 21, 149, 12, 0, 15, 41, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ù' / 0xd9
        'Ù' => Some(&[
            &[0, 0, 0, 57, 190, 190, 137, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 255, 255, 71, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 101, 251, 228, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 75, 27, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 88, 0, 0, 0, 30, 144, 144, 78, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 32, 255, 255, 165, 0, 0, 0, 73, 255, 255, 134, 0, 0],
            &[0, 9, 254, 255, 213, 0, 0, 0, 112, 255, 255, 108, 0, 0],
            &[0, 0, 196, 255, 255, 148, 65, 98, 232, 255, 255, 38, 0, 0],
            &[0, 0, 63, 253, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 67, 204, 255, 255, 255, 231, 123, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 41, 27, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ú' / 0xda
        'Ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 7, 175, 190, 180, 21, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 132, 255, 244, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 252, 234, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 75, 28, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 88, 0, 0, 0, 30, 144, 144, 78, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 32, 255, 255, 165, 0, 0, 0, 73, 255, 255, 134, 0, 0],
            &[0, 9, 254, 255, 213, 0, 0, 0, 112, 255, 255, 108, 0, 0],
            &[0, 0, 196, 255, 255, 148, 65, 98, 232, 255, 255, 38, 0, 0],
            &[0, 0, 63, 253, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 67, 204, 255, 255, 255, 231, 123, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 41, 27, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Û' / 0xdb
        'Û' => Some(&[
            &[0, 0, 0, 0, 5, 171, 190, 190, 66, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 255, 232, 255, 226, 19, 0, 0, 0, 0],
            &[0, 0, 0, 82, 254, 201, 19, 119, 254, 184, 3, 0, 0, 0],
            &[0, 0, 0, 58, 74, 8, 0, 0, 51, 75, 13, 0, 0, 0],
            &[0, 21, 144, 144, 88, 0, 0, 0, 30, 144, 144, 78, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 32, 255, 255, 165, 0, 0, 0, 73, 255, 255, 134, 0, 0],
            &[0, 9, 254, 255, 213, 0, 0, 0, 112, 255, 255, 108, 0, 0],
            &[0, 0, 196, 255, 255, 148, 65, 98, 232, 255, 255, 38, 0, 0],
            &[0, 0, 63, 253, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 67, 204, 255, 255, 255, 231, 123, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 41, 27, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ü' / 0xdc
        'Ü' => Some(&[
            &[0, 0, 0, 48, 124, 40, 0, 9, 111, 92, 0, 0, 0, 0],
            &[0, 0, 0, 212, 255, 199, 0, 102, 255, 255, 57, 0, 0, 0],
            &[0, 0, 0, 175, 255, 159, 0, 67, 251, 242, 30, 0, 0, 0],
            &[0, 0, 0, 0, 32, 0, 0, 0, 21, 11, 0, 0, 0, 0],
            &[0, 21, 144, 144, 88, 0, 0, 0, 30, 144, 144, 78, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 32, 255, 255, 165, 0, 0, 0, 73, 255, 255, 134, 0, 0],
            &[0, 9, 254, 255, 213, 0, 0, 0, 112, 255, 255, 108, 0, 0],
            &[0, 0, 196, 255, 255, 148, 65, 98, 232, 255, 255, 38, 0, 0],
            &[0, 0, 63, 253, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 67, 204, 255, 255, 255, 231, 123, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 41, 27, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ý' / 0xdd
        'Ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 7, 175, 190, 180, 21, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 132, 255, 244, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 252, 234, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 75, 28, 0, 0, 0, 0, 0, 0],
            &[0, 70, 144, 144, 58, 0, 0, 0, 3, 139, 144, 128, 0, 0],
            &[0, 33, 251, 255, 188, 0, 0, 0, 80, 255, 255, 131, 0, 0],
            &[0, 0, 162, 255, 255, 41, 0, 0, 187, 255, 245, 20, 0, 0],
            &[0, 0, 42, 254, 255, 148, 0, 40, 254, 255, 145, 0, 0, 0],
            &[0, 0, 0, 175, 255, 242, 14, 146, 255, 250, 29, 0, 0, 0],
            &[0, 0, 0, 53, 255, 255, 121, 241, 255, 158, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 255, 251, 255, 253, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 255, 255, 255, 172, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 203, 255, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Þ' / 0xde
        'Þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 26, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 106, 73, 65, 34, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 209, 52, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 237, 21, 0, 0],
            &[0, 0, 122, 255, 255, 56, 13, 35, 176, 255, 255, 114, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 45, 255, 255, 155, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 51, 255, 255, 149, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 15, 176, 255, 255, 106, 0, 0],
            &[0, 0, 122, 255, 255, 231, 230, 254, 255, 255, 235, 20, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 225, 57, 0, 0, 0],
            &[0, 0, 122, 255, 255, 141, 115, 100, 65, 3, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ß' / 0xdf
        'ß' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 112, 131, 113, 52, 0, 0, 0, 0, 0],
            &[0, 0, 5, 168, 255, 255, 255, 255, 255, 180, 11, 0, 0, 0],
            &[0, 0, 125, 255, 255, 238, 191, 238, 255, 255, 131, 0, 0, 0],
            &[0, 0, 213, 255, 242, 28, 0, 16, 249, 255, 185, 0, 0, 0],
            &[0, 0, 241, 255, 195, 0, 0, 29, 254, 255, 146, 0, 0, 0],
            &[0, 0, 244, 255, 185, 0, 22, 207, 255, 231, 26, 0, 0, 0],
            &[0, 0, 244, 255, 185, 0, 191, 255, 231, 38, 0, 0, 0, 0],
            &[0, 0, 244, 255, 185, 15, 255, 255, 167, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 185, 1, 232, 255, 255, 136, 3, 0, 0, 0],
            &[0, 0, 244, 255, 185, 0, 69, 244, 255, 255, 180, 7, 0, 0],
            &[0, 0, 244, 255, 185, 0, 0, 46, 216, 255, 255, 126, 0, 0],
            &[0, 0, 244, 255, 185, 0, 0, 0, 19, 245, 255, 197, 0, 0],
            &[0, 0, 244, 255, 185, 60, 73, 10, 46, 252, 255, 201, 0, 0],
            &[0, 0, 244, 255, 185, 98, 255, 253, 255, 255, 255, 126, 0, 0],
            &[0, 0, 244, 255, 185, 67, 237, 255, 255, 248, 153, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 33, 39, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'à' / 0xe0
        'à' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 33, 141, 141, 115, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 128, 255, 255, 77, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 115, 254, 231, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 73, 124, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 83, 126, 140, 128, 82, 9, 0, 0, 0, 0],
            &[0, 0, 55, 247, 255, 255, 255, 255, 255, 228, 50, 0, 0, 0],
            &[0, 0, 3, 223, 240, 190, 170, 205, 255, 255, 214, 0, 0, 0],
            &[0, 0, 0, 29, 5, 0, 0, 0, 164, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 40, 85, 104, 111, 180, 255, 255, 42, 0, 0],
            &[0, 0, 28, 200, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 186, 255, 252, 143, 76, 55, 144, 255, 255, 42, 0, 0],
            &[0, 2, 251, 255, 185, 0, 0, 0, 151, 255, 255, 42, 0, 0],
            &[0, 2, 246, 255, 221, 42, 12, 90, 244, 255, 255, 42, 0, 0],
            &[0, 0, 180, 255, 255, 255, 255, 255, 231, 255, 255, 42, 0, 0],
            &[0, 0, 30, 197, 255, 255, 245, 141, 30, 253, 255, 42, 0, 0],
            &[0, 0, 0, 0, 20, 38, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'á' / 0xe1
        'á' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 105, 141, 141, 42, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 62, 254, 255, 144, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 222, 255, 128, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 124, 80, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 83, 126, 140, 128, 82, 9, 0, 0, 0, 0],
            &[0, 0, 55, 247, 255, 255, 255, 255, 255, 228, 50, 0, 0, 0],
            &[0, 0, 3, 223, 240, 190, 170, 205, 255, 255, 214, 0, 0, 0],
            &[0, 0, 0, 29, 5, 0, 0, 0, 164, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 40, 85, 104, 111, 180, 255, 255, 42, 0, 0],
            &[0, 0, 28, 200, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 186, 255, 252, 143, 76, 55, 144, 255, 255, 42, 0, 0],
            &[0, 2, 251, 255, 185, 0, 0, 0, 151, 255, 255, 42, 0, 0],
            &[0, 2, 246, 255, 221, 42, 12, 90, 244, 255, 255, 42, 0, 0],
            &[0, 0, 180, 255, 255, 255, 255, 255, 231, 255, 255, 42, 0, 0],
            &[0, 0, 30, 197, 255, 255, 245, 141, 30, 253, 255, 42, 0, 0],
            &[0, 0, 0, 0, 20, 38, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'â' / 0xe2
        'â' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 101, 141, 141, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 62, 253, 252, 255, 228, 19, 0, 0, 0, 0],
            &[0, 0, 0, 27, 230, 244, 68, 132, 255, 184, 2, 0, 0, 0],
            &[0, 0, 0, 71, 124, 45, 0, 0, 81, 124, 34, 0, 0, 0],
            &[0, 0, 0, 16, 83, 126, 140, 128, 82, 9, 0, 0, 0, 0],
            &[0, 0, 55, 247, 255, 255, 255, 255, 255, 228, 50, 0, 0, 0],
            &[0, 0, 3, 223, 240, 190, 170, 205, 255, 255, 214, 0, 0, 0],
            &[0, 0, 0, 29, 5, 0, 0, 0, 164, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 40, 85, 104, 111, 180, 255, 255, 42, 0, 0],
            &[0, 0, 28, 200, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 186, 255, 252, 143, 76, 55, 144, 255, 255, 42, 0, 0],
            &[0, 2, 251, 255, 185, 0, 0, 0, 151, 255, 255, 42, 0, 0],
            &[0, 2, 246, 255, 221, 42, 12, 90, 244, 255, 255, 42, 0, 0],
            &[0, 0, 180, 255, 255, 255, 255, 255, 231, 255, 255, 42, 0, 0],
            &[0, 0, 30, 197, 255, 255, 245, 141, 30, 253, 255, 42, 0, 0],
            &[0, 0, 0, 0, 20, 38, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ã' / 0xe3
        'ã' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 74, 19, 0, 0, 90, 40, 0, 0, 0],
            &[0, 0, 0, 42, 242, 255, 246, 152, 136, 255, 81, 0, 0, 0],
            &[0, 0, 0, 146, 243, 151, 219, 255, 255, 225, 9, 0, 0, 0],
            &[0, 0, 0, 92, 89, 0, 2, 73, 109, 26, 0, 0, 0, 0],
            &[0, 0, 0, 16, 83, 126, 140, 128, 82, 9, 0, 0, 0, 0],
            &[0, 0, 55, 247, 255, 255, 255, 255, 255, 228, 50, 0, 0, 0],
            &[0, 0, 3, 223, 240, 190, 170, 205, 255, 255, 214, 0, 0, 0],
            &[0, 0, 0, 29, 5, 0, 0, 0, 164, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 40, 85, 104, 111, 180, 255, 255, 42, 0, 0],
            &[0, 0, 28, 200, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 186, 255, 252, 143, 76, 55, 144, 255, 255, 42, 0, 0],
            &[0, 2, 251, 255, 185, 0, 0, 0, 151, 255, 255, 42, 0, 0],
            &[0, 2, 246, 255, 221, 42, 12, 90, 244, 255, 255, 42, 0, 0],
            &[0, 0, 180, 255, 255, 255, 255, 255, 231, 255, 255, 42, 0, 0],
            &[0, 0, 30, 197, 255, 255, 245, 141, 30, 253, 255, 42, 0, 0],
            &[0, 0, 0, 0, 20, 38, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ä' / 0xe4
        'ä' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 74, 22, 0, 0, 49, 58, 0, 0, 0, 0],
            &[0, 0, 0, 160, 255, 223, 0, 50, 255, 255, 81, 0, 0, 0],
            &[0, 0, 0, 160, 255, 224, 0, 49, 255, 255, 82, 0, 0, 0],
            &[0, 0, 0, 11, 80, 24, 0, 0, 51, 63, 0, 0, 0, 0],
            &[0, 0, 0, 16, 83, 126, 140, 128, 82, 9, 0, 0, 0, 0],
            &[0, 0, 55, 247, 255, 255, 255, 255, 255, 228, 50, 0, 0, 0],
            &[0, 0, 3, 223, 240, 190, 170, 205, 255, 255, 214, 0, 0, 0],
            &[0, 0, 0, 29, 5, 0, 0, 0, 164, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 40, 85, 104, 111, 180, 255, 255, 42, 0, 0],
            &[0, 0, 28, 200, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 186, 255, 252, 143, 76, 55, 144, 255, 255, 42, 0, 0],
            &[0, 2, 251, 255, 185, 0, 0, 0, 151, 255, 255, 42, 0, 0],
            &[0, 2, 246, 255, 221, 42, 12, 90, 244, 255, 255, 42, 0, 0],
            &[0, 0, 180, 255, 255, 255, 255, 255, 231, 255, 255, 42, 0, 0],
            &[0, 0, 30, 197, 255, 255, 245, 141, 30, 253, 255, 42, 0, 0],
            &[0, 0, 0, 0, 20, 38, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'å' / 0xe5
        'å' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 120, 211, 192, 57, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 64, 255, 152, 200, 233, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 253, 15, 104, 255, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 240, 238, 251, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 109, 91, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 83, 126, 140, 128, 82, 9, 0, 0, 0, 0],
            &[0, 0, 55, 247, 255, 255, 255, 255, 255, 228, 50, 0, 0, 0],
            &[0, 0, 3, 223, 240, 190, 170, 205, 255, 255, 214, 0, 0, 0],
            &[0, 0, 0, 29, 5, 0, 0, 0, 164, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 40, 85, 104, 111, 180, 255, 255, 42, 0, 0],
            &[0, 0, 28, 200, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 186, 255, 252, 143, 76, 55, 144, 255, 255, 42, 0, 0],
            &[0, 2, 251, 255, 185, 0, 0, 0, 151, 255, 255, 42, 0, 0],
            &[0, 2, 246, 255, 221, 42, 12, 90, 244, 255, 255, 42, 0, 0],
            &[0, 0, 180, 255, 255, 255, 255, 255, 231, 255, 255, 42, 0, 0],
            &[0, 0, 30, 197, 255, 255, 245, 141, 30, 253, 255, 42, 0, 0],
            &[0, 0, 0, 0, 20, 38, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'æ' / 0xe6
        'æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 37, 110, 134, 100, 10, 27, 114, 128, 46, 0, 0, 0],
            &[0, 29, 252, 255, 255, 255, 223, 240, 255, 255, 251, 63, 0, 0],
            &[0, 0, 172, 191, 189, 250, 255, 255, 212, 226, 255, 198, 0, 0],
            &[0, 0, 0, 0, 0, 145, 255, 254, 25, 69, 255, 253, 13, 0],
            &[0, 0, 0, 45, 78, 160, 255, 245, 73, 97, 255, 255, 42, 0],
            &[
                0, 19, 202, 255, 255, 255, 255, 255, 255, 255, 255, 255, 53, 0,
            ],
            &[
                0, 140, 255, 243, 89, 145, 255, 240, 117, 117, 117, 117, 24, 0,
            ],
            &[0, 195, 255, 167, 0, 124, 255, 249, 12, 0, 0, 0, 0, 0],
            &[0, 191, 255, 204, 11, 193, 255, 255, 163, 13, 5, 69, 0, 0],
            &[
                0, 141, 255, 255, 252, 255, 210, 255, 255, 251, 248, 197, 0, 0,
            ],
            &[0, 23, 207, 255, 255, 165, 6, 132, 245, 255, 254, 136, 0, 0],
            &[0, 0, 1, 32, 24, 0, 0, 0, 10, 40, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ç' / 0xe7
        'ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 62, 118, 138, 123, 83, 13, 0, 0, 0],
            &[0, 0, 0, 30, 202, 255, 255, 255, 255, 255, 182, 0, 0, 0],
            &[0, 0, 3, 203, 255, 255, 245, 205, 219, 255, 92, 0, 0, 0],
            &[0, 0, 70, 255, 255, 206, 19, 0, 0, 26, 8, 0, 0, 0],
            &[0, 0, 136, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 161, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 159, 255, 255, 45, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 255, 255, 113, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 255, 255, 243, 105, 45, 65, 137, 125, 0, 0, 0],
            &[0, 0, 0, 170, 255, 255, 255, 255, 255, 255, 142, 0, 0, 0],
            &[0, 0, 0, 5, 132, 232, 255, 255, 255, 218, 81, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 192, 255, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 160, 252, 228, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 201, 255, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 77, 137, 119, 35, 0, 0, 0, 0, 0],
        ]),
        // letter: 'è' / 0xe8
        'è' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 141, 141, 120, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 119, 255, 255, 86, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 106, 253, 235, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 68, 124, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 98, 133, 126, 76, 4, 0, 0, 0, 0],
            &[0, 0, 0, 87, 244, 255, 255, 255, 255, 210, 30, 0, 0, 0],
            &[0, 0, 51, 251, 255, 242, 158, 177, 252, 255, 197, 0, 0, 0],
            &[0, 0, 169, 255, 248, 42, 0, 0, 109, 255, 255, 49, 0, 0],
            &[0, 0, 236, 255, 213, 69, 69, 69, 88, 255, 255, 96, 0, 0],
            &[0, 5, 255, 255, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 248, 255, 222, 156, 156, 156, 156, 156, 156, 70, 0, 0],
            &[0, 0, 208, 255, 222, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 255, 255, 192, 75, 32, 38, 76, 148, 28, 0, 0],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0],
            &[0, 0, 0, 5, 123, 222, 255, 255, 255, 248, 182, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 42, 29, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'é' / 0xe9
        'é' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 100, 141, 141, 47, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 54, 252, 255, 153, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 216, 255, 136, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 47, 124, 85, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 98, 133, 126, 76, 4, 0, 0, 0, 0],
            &[0, 0, 0, 87, 244, 255, 255, 255, 255, 210, 30, 0, 0, 0],
            &[0, 0, 51, 251, 255, 242, 158, 177, 252, 255, 197, 0, 0, 0],
            &[0, 0, 169, 255, 248, 42, 0, 0, 109, 255, 255, 49, 0, 0],
            &[0, 0, 236, 255, 213, 69, 69, 69, 88, 255, 255, 96, 0, 0],
            &[0, 5, 255, 255, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 248, 255, 222, 156, 156, 156, 156, 156, 156, 70, 0, 0],
            &[0, 0, 208, 255, 222, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 255, 255, 192, 75, 32, 38, 76, 148, 28, 0, 0],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0],
            &[0, 0, 0, 5, 123, 222, 255, 255, 255, 248, 182, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 42, 29, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ê' / 0xea
        'ê' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 95, 141, 141, 67, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 251, 252, 255, 233, 24, 0, 0, 0, 0],
            &[0, 0, 0, 22, 225, 247, 75, 123, 255, 192, 4, 0, 0, 0],
            &[0, 0, 0, 66, 124, 50, 0, 0, 76, 124, 39, 0, 0, 0],
            &[0, 0, 0, 0, 17, 98, 133, 126, 76, 4, 0, 0, 0, 0],
            &[0, 0, 0, 87, 244, 255, 255, 255, 255, 210, 30, 0, 0, 0],
            &[0, 0, 51, 251, 255, 242, 158, 177, 252, 255, 197, 0, 0, 0],
            &[0, 0, 169, 255, 248, 42, 0, 0, 109, 255, 255, 49, 0, 0],
            &[0, 0, 236, 255, 213, 69, 69, 69, 88, 255, 255, 96, 0, 0],
            &[0, 5, 255, 255, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 248, 255, 222, 156, 156, 156, 156, 156, 156, 70, 0, 0],
            &[0, 0, 208, 255, 222, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 255, 255, 192, 75, 32, 38, 76, 148, 28, 0, 0],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0],
            &[0, 0, 0, 5, 123, 222, 255, 255, 255, 248, 182, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 42, 29, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ë' / 0xeb
        'ë' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 74, 24, 0, 0, 45, 61, 1, 0, 0, 0],
            &[0, 0, 0, 150, 255, 232, 0, 41, 254, 255, 91, 0, 0, 0],
            &[0, 0, 0, 150, 255, 232, 0, 41, 254, 255, 91, 0, 0, 0],
            &[0, 0, 0, 9, 79, 27, 0, 0, 48, 66, 1, 0, 0, 0],
            &[0, 0, 0, 0, 17, 98, 133, 126, 76, 4, 0, 0, 0, 0],
            &[0, 0, 0, 87, 244, 255, 255, 255, 255, 210, 30, 0, 0, 0],
            &[0, 0, 51, 251, 255, 242, 158, 177, 252, 255, 197, 0, 0, 0],
            &[0, 0, 169, 255, 248, 42, 0, 0, 109, 255, 255, 49, 0, 0],
            &[0, 0, 236, 255, 213, 69, 69, 69, 88, 255, 255, 96, 0, 0],
            &[0, 5, 255, 255, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 248, 255, 222, 156, 156, 156, 156, 156, 156, 70, 0, 0],
            &[0, 0, 208, 255, 222, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 255, 255, 192, 75, 32, 38, 76, 148, 28, 0, 0],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0],
            &[0, 0, 0, 5, 123, 222, 255, 255, 255, 248, 182, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 42, 29, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ì' / 0xec
        'ì' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 141, 141, 93, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 164, 255, 249, 44, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 150, 255, 206, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 124, 40, 0, 0, 0, 0, 0],
            &[0, 0, 33, 95, 95, 95, 95, 95, 33, 0, 0, 0, 0, 0],
            &[0, 0, 88, 255, 255, 255, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 53, 172, 201, 236, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 138, 255, 255, 126, 18, 0, 0, 0, 0],
            &[0, 0, 171, 238, 255, 255, 255, 255, 255, 255, 226, 86, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'í' / 0xed
        'í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 126, 141, 140, 21, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 254, 107, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 242, 250, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 70, 124, 61, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 95, 95, 95, 95, 95, 33, 0, 0, 0, 0, 0],
            &[0, 0, 88, 255, 255, 255, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 53, 172, 201, 236, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 138, 255, 255, 126, 18, 0, 0, 0, 0],
            &[0, 0, 171, 238, 255, 255, 255, 255, 255, 255, 226, 86, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'î' / 0xee
        'î' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 141, 141, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 255, 252, 255, 203, 6, 0, 0, 0, 0],
            &[0, 0, 0, 49, 246, 231, 47, 167, 255, 148, 0, 0, 0, 0],
            &[0, 0, 0, 90, 123, 27, 0, 1, 98, 124, 16, 0, 0, 0],
            &[0, 0, 33, 95, 95, 95, 95, 95, 33, 0, 0, 0, 0, 0],
            &[0, 0, 88, 255, 255, 255, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 53, 172, 201, 236, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 138, 255, 255, 126, 18, 0, 0, 0, 0],
            &[0, 0, 171, 238, 255, 255, 255, 255, 255, 255, 226, 86, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ï' / 0xef
        'ï' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 75, 13, 0, 1, 62, 44, 0, 0, 0, 0],
            &[0, 0, 0, 198, 255, 184, 0, 88, 255, 254, 44, 0, 0, 0],
            &[0, 0, 0, 199, 255, 185, 0, 88, 255, 254, 44, 0, 0, 0],
            &[0, 0, 0, 20, 81, 15, 0, 1, 65, 48, 0, 0, 0, 0],
            &[0, 0, 33, 95, 95, 95, 95, 95, 33, 0, 0, 0, 0, 0],
            &[0, 0, 88, 255, 255, 255, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 53, 172, 201, 236, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 138, 255, 255, 126, 18, 0, 0, 0, 0],
            &[0, 0, 171, 238, 255, 255, 255, 255, 255, 255, 226, 86, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ð' / 0xf0
        'ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 75, 0, 0, 8, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 150, 255, 200, 119, 221, 251, 45, 0, 0, 0],
            &[0, 0, 0, 0, 75, 254, 255, 255, 255, 111, 3, 0, 0, 0],
            &[0, 0, 0, 3, 118, 255, 255, 255, 255, 88, 0, 0, 0, 0],
            &[0, 0, 0, 1, 194, 217, 71, 174, 255, 246, 36, 0, 0, 0],
            &[0, 0, 0, 0, 28, 44, 40, 15, 220, 255, 174, 0, 0, 0],
            &[0, 0, 0, 100, 230, 255, 255, 249, 204, 255, 252, 27, 0, 0],
            &[0, 0, 96, 255, 255, 249, 212, 239, 255, 255, 255, 96, 0, 0],
            &[0, 1, 226, 255, 244, 44, 0, 9, 191, 255, 255, 141, 0, 0],
            &[0, 36, 255, 255, 172, 0, 0, 0, 65, 255, 255, 158, 0, 0],
            &[0, 54, 255, 255, 146, 0, 0, 0, 41, 255, 255, 149, 0, 0],
            &[0, 31, 255, 255, 178, 0, 0, 0, 77, 255, 255, 107, 0, 0],
            &[0, 0, 213, 255, 251, 59, 0, 12, 196, 255, 250, 27, 0, 0],
            &[0, 0, 67, 252, 255, 253, 219, 243, 255, 255, 117, 0, 0, 0],
            &[0, 0, 0, 60, 199, 255, 255, 255, 222, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 42, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ñ' / 0xf1
        'ñ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 73, 15, 0, 4, 93, 33, 0, 0, 0],
            &[0, 0, 0, 56, 248, 255, 242, 145, 147, 255, 61, 0, 0, 0],
            &[0, 0, 0, 166, 237, 151, 226, 255, 255, 212, 3, 0, 0, 0],
            &[0, 0, 0, 102, 80, 0, 4, 79, 107, 20, 0, 0, 0, 0],
            &[0, 0, 91, 95, 19, 3, 83, 131, 119, 57, 0, 0, 0, 0],
            &[0, 0, 244, 255, 124, 200, 255, 255, 255, 255, 129, 0, 0, 0],
            &[0, 0, 244, 255, 255, 254, 209, 215, 255, 255, 252, 35, 0, 0],
            &[0, 0, 244, 255, 255, 119, 0, 1, 170, 255, 255, 88, 0, 0],
            &[0, 0, 244, 255, 247, 10, 0, 0, 91, 255, 255, 114, 0, 0],
            &[0, 0, 244, 255, 218, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ò' / 0xf2
        'ò' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 141, 141, 93, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 164, 255, 249, 44, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 150, 255, 206, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 124, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 104, 135, 122, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 109, 250, 255, 255, 255, 255, 197, 24, 0, 0, 0],
            &[0, 0, 80, 255, 255, 244, 185, 213, 255, 255, 192, 1, 0, 0],
            &[0, 0, 209, 255, 252, 50, 0, 1, 181, 255, 255, 61, 0, 0],
            &[0, 25, 255, 255, 186, 0, 0, 0, 84, 255, 255, 129, 0, 0],
            &[0, 52, 255, 255, 150, 0, 0, 0, 50, 255, 255, 155, 0, 0],
            &[0, 49, 255, 255, 154, 0, 0, 0, 62, 255, 255, 151, 0, 0],
            &[0, 13, 252, 255, 211, 0, 0, 0, 108, 255, 255, 110, 0, 0],
            &[0, 0, 179, 255, 255, 131, 20, 55, 228, 255, 251, 29, 0, 0],
            &[0, 0, 38, 240, 255, 255, 255, 255, 255, 255, 125, 0, 0, 0],
            &[0, 0, 0, 42, 186, 253, 255, 255, 227, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 42, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ó' / 0xf3
        'ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 126, 141, 140, 21, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 254, 107, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 242, 250, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 70, 124, 61, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 104, 135, 122, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 109, 250, 255, 255, 255, 255, 197, 24, 0, 0, 0],
            &[0, 0, 80, 255, 255, 244, 185, 213, 255, 255, 192, 1, 0, 0],
            &[0, 0, 209, 255, 252, 50, 0, 1, 181, 255, 255, 61, 0, 0],
            &[0, 25, 255, 255, 186, 0, 0, 0, 84, 255, 255, 129, 0, 0],
            &[0, 52, 255, 255, 150, 0, 0, 0, 50, 255, 255, 155, 0, 0],
            &[0, 49, 255, 255, 154, 0, 0, 0, 62, 255, 255, 151, 0, 0],
            &[0, 13, 252, 255, 211, 0, 0, 0, 108, 255, 255, 110, 0, 0],
            &[0, 0, 179, 255, 255, 131, 20, 55, 228, 255, 251, 29, 0, 0],
            &[0, 0, 38, 240, 255, 255, 255, 255, 255, 255, 125, 0, 0, 0],
            &[0, 0, 0, 42, 186, 253, 255, 255, 227, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 42, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ô' / 0xf4
        'ô' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 141, 141, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 255, 252, 255, 203, 6, 0, 0, 0, 0],
            &[0, 0, 0, 49, 246, 231, 47, 167, 255, 148, 0, 0, 0, 0],
            &[0, 0, 0, 90, 123, 27, 0, 1, 98, 124, 16, 0, 0, 0],
            &[0, 0, 0, 0, 23, 104, 135, 122, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 109, 250, 255, 255, 255, 255, 197, 24, 0, 0, 0],
            &[0, 0, 80, 255, 255, 244, 185, 213, 255, 255, 192, 1, 0, 0],
            &[0, 0, 209, 255, 252, 50, 0, 1, 181, 255, 255, 61, 0, 0],
            &[0, 25, 255, 255, 186, 0, 0, 0, 84, 255, 255, 129, 0, 0],
            &[0, 52, 255, 255, 150, 0, 0, 0, 50, 255, 255, 155, 0, 0],
            &[0, 49, 255, 255, 154, 0, 0, 0, 62, 255, 255, 151, 0, 0],
            &[0, 13, 252, 255, 211, 0, 0, 0, 108, 255, 255, 110, 0, 0],
            &[0, 0, 179, 255, 255, 131, 20, 55, 228, 255, 251, 29, 0, 0],
            &[0, 0, 38, 240, 255, 255, 255, 255, 255, 255, 125, 0, 0, 0],
            &[0, 0, 0, 42, 186, 253, 255, 255, 227, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 42, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'õ' / 0xf5
        'õ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 72, 11, 0, 11, 93, 26, 0, 0, 0],
            &[0, 0, 0, 72, 252, 255, 237, 139, 158, 255, 42, 0, 0, 0],
            &[0, 0, 0, 185, 230, 152, 232, 255, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 112, 70, 0, 7, 85, 105, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 104, 135, 122, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 109, 250, 255, 255, 255, 255, 197, 24, 0, 0, 0],
            &[0, 0, 80, 255, 255, 244, 185, 213, 255, 255, 192, 1, 0, 0],
            &[0, 0, 209, 255, 252, 50, 0, 1, 181, 255, 255, 61, 0, 0],
            &[0, 25, 255, 255, 186, 0, 0, 0, 84, 255, 255, 129, 0, 0],
            &[0, 52, 255, 255, 150, 0, 0, 0, 50, 255, 255, 155, 0, 0],
            &[0, 49, 255, 255, 154, 0, 0, 0, 62, 255, 255, 151, 0, 0],
            &[0, 13, 252, 255, 211, 0, 0, 0, 108, 255, 255, 110, 0, 0],
            &[0, 0, 179, 255, 255, 131, 20, 55, 228, 255, 251, 29, 0, 0],
            &[0, 0, 38, 240, 255, 255, 255, 255, 255, 255, 125, 0, 0, 0],
            &[0, 0, 0, 42, 186, 253, 255, 255, 227, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 42, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ö' / 0xf6
        'ö' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 75, 13, 0, 1, 62, 44, 0, 0, 0, 0],
            &[0, 0, 0, 198, 255, 184, 0, 88, 255, 254, 44, 0, 0, 0],
            &[0, 0, 0, 199, 255, 185, 0, 88, 255, 254, 44, 0, 0, 0],
            &[0, 0, 0, 20, 81, 15, 0, 1, 65, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 104, 135, 122, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 109, 250, 255, 255, 255, 255, 197, 24, 0, 0, 0],
            &[0, 0, 80, 255, 255, 244, 185, 213, 255, 255, 192, 1, 0, 0],
            &[0, 0, 209, 255, 252, 50, 0, 1, 181, 255, 255, 61, 0, 0],
            &[0, 25, 255, 255, 186, 0, 0, 0, 84, 255, 255, 129, 0, 0],
            &[0, 52, 255, 255, 150, 0, 0, 0, 50, 255, 255, 155, 0, 0],
            &[0, 49, 255, 255, 154, 0, 0, 0, 62, 255, 255, 151, 0, 0],
            &[0, 13, 252, 255, 211, 0, 0, 0, 108, 255, 255, 110, 0, 0],
            &[0, 0, 179, 255, 255, 131, 20, 55, 228, 255, 251, 29, 0, 0],
            &[0, 0, 38, 240, 255, 255, 255, 255, 255, 255, 125, 0, 0, 0],
            &[0, 0, 0, 42, 186, 253, 255, 255, 227, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 42, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '÷' / 0xf7
        '÷' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 113, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 129, 255, 237, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 108, 255, 215, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 182, 182, 182, 182, 182, 182, 182, 182, 12, 0, 0],
            &[0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 51, 76, 76, 76, 76, 76, 76, 76, 76, 5, 0, 0],
            &[0, 0, 0, 0, 0, 33, 154, 90, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 135, 255, 243, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 251, 189, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ø' / 0xf8
        'ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 104, 135, 122, 66, 152, 85, 0, 0, 0],
            &[0, 0, 0, 109, 250, 255, 255, 255, 255, 255, 208, 0, 0, 0],
            &[0, 0, 80, 255, 255, 244, 185, 236, 255, 255, 219, 1, 0, 0],
            &[0, 0, 209, 255, 252, 50, 0, 177, 255, 255, 255, 61, 0, 0],
            &[0, 25, 255, 255, 186, 0, 92, 255, 255, 255, 255, 129, 0, 0],
            &[0, 52, 255, 255, 150, 26, 238, 234, 73, 255, 255, 155, 0, 0],
            &[0, 49, 255, 255, 155, 182, 255, 83, 62, 255, 255, 151, 0, 0],
            &[0, 13, 252, 255, 255, 255, 165, 0, 108, 255, 255, 110, 0, 0],
            &[0, 0, 179, 255, 255, 255, 38, 55, 228, 255, 251, 29, 0, 0],
            &[0, 0, 39, 255, 255, 255, 255, 255, 255, 255, 125, 0, 0, 0],
            &[0, 0, 91, 255, 255, 253, 255, 255, 227, 103, 0, 0, 0, 0],
            &[0, 0, 4, 116, 13, 15, 42, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ù' / 0xf9
        'ù' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 141, 141, 93, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 164, 255, 249, 44, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 150, 255, 206, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 124, 40, 0, 0, 0, 0, 0],
            &[0, 4, 95, 95, 67, 0, 0, 0, 38, 95, 95, 33, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 105, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 181, 0, 0, 0, 127, 255, 255, 90, 0, 0],
            &[0, 7, 254, 255, 205, 0, 0, 0, 170, 255, 255, 90, 0, 0],
            &[0, 0, 228, 255, 254, 94, 32, 100, 252, 255, 255, 90, 0, 0],
            &[0, 0, 138, 255, 255, 255, 255, 255, 224, 255, 255, 90, 0, 0],
            &[0, 0, 11, 160, 253, 255, 253, 168, 17, 229, 255, 90, 0, 0],
            &[0, 0, 0, 0, 10, 39, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ú' / 0xfa
        'ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 126, 141, 140, 21, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 254, 107, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 242, 250, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 70, 124, 61, 0, 0, 0, 0, 0, 0],
            &[0, 4, 95, 95, 67, 0, 0, 0, 38, 95, 95, 33, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 105, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 181, 0, 0, 0, 127, 255, 255, 90, 0, 0],
            &[0, 7, 254, 255, 205, 0, 0, 0, 170, 255, 255, 90, 0, 0],
            &[0, 0, 228, 255, 254, 94, 32, 100, 252, 255, 255, 90, 0, 0],
            &[0, 0, 138, 255, 255, 255, 255, 255, 224, 255, 255, 90, 0, 0],
            &[0, 0, 11, 160, 253, 255, 253, 168, 17, 229, 255, 90, 0, 0],
            &[0, 0, 0, 0, 10, 39, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'û' / 0xfb
        'û' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 141, 141, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 255, 252, 255, 203, 6, 0, 0, 0, 0],
            &[0, 0, 0, 49, 246, 231, 47, 167, 255, 148, 0, 0, 0, 0],
            &[0, 0, 0, 90, 123, 27, 0, 1, 98, 124, 16, 0, 0, 0],
            &[0, 4, 95, 95, 67, 0, 0, 0, 38, 95, 95, 33, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 105, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 181, 0, 0, 0, 127, 255, 255, 90, 0, 0],
            &[0, 7, 254, 255, 205, 0, 0, 0, 170, 255, 255, 90, 0, 0],
            &[0, 0, 228, 255, 254, 94, 32, 100, 252, 255, 255, 90, 0, 0],
            &[0, 0, 138, 255, 255, 255, 255, 255, 224, 255, 255, 90, 0, 0],
            &[0, 0, 11, 160, 253, 255, 253, 168, 17, 229, 255, 90, 0, 0],
            &[0, 0, 0, 0, 10, 39, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ü' / 0xfc
        'ü' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 75, 13, 0, 1, 62, 44, 0, 0, 0, 0],
            &[0, 0, 0, 198, 255, 184, 0, 88, 255, 254, 44, 0, 0, 0],
            &[0, 0, 0, 199, 255, 185, 0, 88, 255, 254, 44, 0, 0, 0],
            &[0, 0, 0, 20, 81, 15, 0, 1, 65, 48, 0, 0, 0, 0],
            &[0, 4, 95, 95, 67, 0, 0, 0, 38, 95, 95, 33, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 105, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 181, 0, 0, 0, 127, 255, 255, 90, 0, 0],
            &[0, 7, 254, 255, 205, 0, 0, 0, 170, 255, 255, 90, 0, 0],
            &[0, 0, 228, 255, 254, 94, 32, 100, 252, 255, 255, 90, 0, 0],
            &[0, 0, 138, 255, 255, 255, 255, 255, 224, 255, 255, 90, 0, 0],
            &[0, 0, 11, 160, 253, 255, 253, 168, 17, 229, 255, 90, 0, 0],
            &[0, 0, 0, 0, 10, 39, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ý' / 0xfd
        'ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 126, 141, 140, 21, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 254, 107, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 242, 250, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 70, 124, 61, 0, 0, 0, 0, 0, 0],
            &[0, 52, 95, 95, 42, 0, 0, 0, 1, 94, 95, 90, 0, 0],
            &[0, 69, 255, 255, 172, 0, 0, 0, 51, 255, 255, 179, 0, 0],
            &[0, 1, 222, 255, 246, 11, 0, 0, 127, 255, 255, 86, 0, 0],
            &[0, 0, 122, 255, 255, 87, 0, 0, 203, 255, 240, 8, 0, 0],
            &[0, 0, 25, 251, 255, 173, 0, 24, 254, 255, 156, 0, 0, 0],
            &[0, 0, 0, 175, 255, 246, 12, 99, 255, 255, 63, 0, 0, 0],
            &[0, 0, 0, 74, 255, 255, 87, 174, 255, 224, 1, 0, 0, 0],
            &[0, 0, 0, 2, 225, 255, 162, 241, 255, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 255, 242, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 252, 255, 255, 203, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 180, 255, 255, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 192, 255, 249, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 255, 255, 164, 0, 0, 0, 0, 0, 0],
            &[0, 0, 177, 210, 253, 255, 248, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 162, 134, 41, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'þ' / 0xfe
        'þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 112, 83, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 188, 2, 80, 132, 122, 39, 0, 0, 0, 0],
            &[0, 0, 244, 255, 186, 192, 255, 255, 255, 249, 80, 0, 0, 0],
            &[0, 0, 244, 255, 255, 252, 202, 223, 255, 255, 236, 12, 0, 0],
            &[0, 0, 244, 255, 255, 85, 0, 4, 189, 255, 255, 87, 0, 0],
            &[0, 0, 244, 255, 223, 0, 0, 0, 89, 255, 255, 138, 0, 0],
            &[0, 0, 244, 255, 194, 0, 0, 0, 60, 255, 255, 159, 0, 0],
            &[0, 0, 244, 255, 208, 0, 0, 0, 69, 255, 255, 149, 0, 0],
            &[0, 0, 244, 255, 245, 15, 0, 0, 113, 255, 255, 120, 0, 0],
            &[0, 0, 244, 255, 255, 173, 43, 65, 228, 255, 255, 50, 0, 0],
            &[0, 0, 244, 255, 248, 255, 255, 255, 255, 255, 181, 0, 0, 0],
            &[0, 0, 244, 255, 175, 125, 241, 255, 255, 176, 15, 0, 0, 0],
            &[0, 0, 244, 255, 189, 0, 7, 38, 19, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 136, 143, 106, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ÿ' / 0xff
        'ÿ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 75, 13, 0, 1, 62, 44, 0, 0, 0, 0],
            &[0, 0, 0, 198, 255, 184, 0, 88, 255, 254, 44, 0, 0, 0],
            &[0, 0, 0, 199, 255, 185, 0, 88, 255, 254, 44, 0, 0, 0],
            &[0, 0, 0, 20, 81, 15, 0, 1, 65, 48, 0, 0, 0, 0],
            &[0, 52, 95, 95, 42, 0, 0, 0, 1, 94, 95, 90, 0, 0],
            &[0, 69, 255, 255, 172, 0, 0, 0, 51, 255, 255, 179, 0, 0],
            &[0, 1, 222, 255, 246, 11, 0, 0, 127, 255, 255, 86, 0, 0],
            &[0, 0, 122, 255, 255, 87, 0, 0, 203, 255, 240, 8, 0, 0],
            &[0, 0, 25, 251, 255, 173, 0, 24, 254, 255, 156, 0, 0, 0],
            &[0, 0, 0, 175, 255, 246, 12, 99, 255, 255, 63, 0, 0, 0],
            &[0, 0, 0, 74, 255, 255, 87, 174, 255, 224, 1, 0, 0, 0],
            &[0, 0, 0, 2, 225, 255, 162, 241, 255, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 255, 242, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 252, 255, 255, 203, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 180, 255, 255, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 192, 255, 249, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 255, 255, 164, 0, 0, 0, 0, 0, 0],
            &[0, 0, 177, 210, 253, 255, 248, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 162, 134, 41, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ā' / 0x100
        'Ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 228, 228, 228, 228, 228, 129, 0, 0, 0, 0],
            &[0, 0, 0, 47, 253, 253, 253, 253, 253, 144, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 159, 159, 159, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 255, 255, 255, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 239, 255, 250, 13, 0, 0, 0, 0],
            &[0, 0, 0, 1, 234, 255, 137, 254, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 255, 60, 212, 255, 150, 0, 0, 0, 0],
            &[0, 0, 0, 121, 255, 246, 6, 151, 255, 220, 0, 0, 0, 0],
            &[0, 0, 0, 191, 255, 189, 0, 90, 255, 255, 35, 0, 0, 0],
            &[0, 0, 12, 249, 255, 125, 0, 28, 255, 255, 106, 0, 0, 0],
            &[0, 0, 76, 255, 255, 226, 164, 164, 255, 255, 176, 0, 0, 0],
            &[0, 0, 146, 255, 255, 255, 255, 255, 255, 255, 242, 5, 0, 0],
            &[0, 0, 217, 255, 255, 163, 163, 163, 246, 255, 255, 62, 0, 0],
            &[0, 31, 255, 255, 126, 0, 0, 0, 37, 255, 255, 132, 0, 0],
            &[0, 101, 255, 255, 62, 0, 0, 0, 0, 231, 255, 203, 0, 0],
            &[0, 172, 255, 247, 7, 0, 0, 0, 0, 170, 255, 253, 20, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ā' / 0x101
        'ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 178, 178, 178, 178, 178, 128, 0, 0, 0, 0],
            &[0, 0, 0, 9, 255, 255, 255, 255, 255, 184, 0, 0, 0, 0],
            &[0, 0, 0, 1, 46, 46, 46, 46, 46, 33, 0, 0, 0, 0],
            &[0, 0, 0, 16, 83, 126, 140, 128, 82, 9, 0, 0, 0, 0],
            &[0, 0, 55, 247, 255, 255, 255, 255, 255, 228, 50, 0, 0, 0],
            &[0, 0, 3, 223, 240, 190, 170, 205, 255, 255, 214, 0, 0, 0],
            &[0, 0, 0, 29, 5, 0, 0, 0, 164, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 40, 85, 104, 111, 180, 255, 255, 42, 0, 0],
            &[0, 0, 28, 200, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 186, 255, 252, 143, 76, 55, 144, 255, 255, 42, 0, 0],
            &[0, 2, 251, 255, 185, 0, 0, 0, 151, 255, 255, 42, 0, 0],
            &[0, 2, 246, 255, 221, 42, 12, 90, 244, 255, 255, 42, 0, 0],
            &[0, 0, 180, 255, 255, 255, 255, 255, 231, 255, 255, 42, 0, 0],
            &[0, 0, 30, 197, 255, 255, 245, 141, 30, 253, 255, 42, 0, 0],
            &[0, 0, 0, 0, 20, 38, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ă' / 0x102
        'Ă' => Some(&[
            &[0, 0, 0, 85, 205, 45, 0, 0, 165, 166, 0, 0, 0, 0],
            &[0, 0, 0, 49, 254, 224, 143, 186, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 118, 250, 255, 255, 187, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 63, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 159, 159, 159, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 255, 255, 255, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 239, 255, 250, 13, 0, 0, 0, 0],
            &[0, 0, 0, 1, 234, 255, 137, 254, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 255, 60, 212, 255, 150, 0, 0, 0, 0],
            &[0, 0, 0, 121, 255, 246, 6, 151, 255, 220, 0, 0, 0, 0],
            &[0, 0, 0, 191, 255, 189, 0, 90, 255, 255, 35, 0, 0, 0],
            &[0, 0, 12, 249, 255, 125, 0, 28, 255, 255, 106, 0, 0, 0],
            &[0, 0, 76, 255, 255, 226, 164, 164, 255, 255, 176, 0, 0, 0],
            &[0, 0, 146, 255, 255, 255, 255, 255, 255, 255, 242, 5, 0, 0],
            &[0, 0, 217, 255, 255, 163, 163, 163, 246, 255, 255, 62, 0, 0],
            &[0, 31, 255, 255, 126, 0, 0, 0, 37, 255, 255, 132, 0, 0],
            &[0, 101, 255, 255, 62, 0, 0, 0, 0, 231, 255, 203, 0, 0],
            &[0, 172, 255, 247, 7, 0, 0, 0, 0, 170, 255, 253, 20, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ă' / 0x103
        'ă' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 156, 53, 0, 0, 96, 152, 1, 0, 0, 0],
            &[0, 0, 0, 30, 253, 212, 97, 123, 242, 202, 0, 0, 0, 0],
            &[0, 0, 0, 0, 128, 255, 255, 255, 238, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 109, 95, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 83, 126, 140, 128, 82, 9, 0, 0, 0, 0],
            &[0, 0, 55, 247, 255, 255, 255, 255, 255, 228, 50, 0, 0, 0],
            &[0, 0, 3, 223, 240, 190, 170, 205, 255, 255, 214, 0, 0, 0],
            &[0, 0, 0, 29, 5, 0, 0, 0, 164, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 40, 85, 104, 111, 180, 255, 255, 42, 0, 0],
            &[0, 0, 28, 200, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 186, 255, 252, 143, 76, 55, 144, 255, 255, 42, 0, 0],
            &[0, 2, 251, 255, 185, 0, 0, 0, 151, 255, 255, 42, 0, 0],
            &[0, 2, 246, 255, 221, 42, 12, 90, 244, 255, 255, 42, 0, 0],
            &[0, 0, 180, 255, 255, 255, 255, 255, 231, 255, 255, 42, 0, 0],
            &[0, 0, 30, 197, 255, 255, 245, 141, 30, 253, 255, 42, 0, 0],
            &[0, 0, 0, 0, 20, 38, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ą' / 0x104
        'Ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 159, 159, 159, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 255, 255, 255, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 239, 255, 250, 13, 0, 0, 0, 0],
            &[0, 0, 0, 1, 234, 255, 137, 254, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 255, 60, 212, 255, 150, 0, 0, 0, 0],
            &[0, 0, 0, 121, 255, 246, 6, 151, 255, 220, 0, 0, 0, 0],
            &[0, 0, 0, 191, 255, 189, 0, 90, 255, 255, 35, 0, 0, 0],
            &[0, 0, 12, 249, 255, 125, 0, 28, 255, 255, 106, 0, 0, 0],
            &[0, 0, 76, 255, 255, 226, 164, 164, 255, 255, 176, 0, 0, 0],
            &[0, 0, 146, 255, 255, 255, 255, 255, 255, 255, 242, 5, 0, 0],
            &[0, 0, 217, 255, 255, 163, 163, 163, 246, 255, 255, 62, 0, 0],
            &[0, 31, 255, 255, 126, 0, 0, 0, 37, 255, 255, 132, 0, 0],
            &[0, 101, 255, 255, 62, 0, 0, 0, 0, 231, 255, 203, 0, 0],
            &[0, 172, 255, 247, 7, 0, 0, 0, 0, 170, 255, 255, 21, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 29, 222, 192, 5, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 172, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 228, 255, 84, 40, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 92, 121, 58, 0],
        ]),
        // letter: 'ą' / 0x105
        'ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 83, 126, 140, 128, 82, 9, 0, 0, 0, 0],
            &[0, 0, 55, 247, 255, 255, 255, 255, 255, 228, 50, 0, 0, 0],
            &[0, 0, 3, 223, 240, 190, 170, 205, 255, 255, 214, 0, 0, 0],
            &[0, 0, 0, 29, 5, 0, 0, 0, 164, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 40, 85, 104, 111, 180, 255, 255, 42, 0, 0],
            &[0, 0, 28, 200, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 186, 255, 252, 143, 76, 55, 144, 255, 255, 42, 0, 0],
            &[0, 2, 251, 255, 185, 0, 0, 0, 151, 255, 255, 42, 0, 0],
            &[0, 2, 246, 255, 221, 42, 12, 90, 244, 255, 255, 42, 0, 0],
            &[0, 0, 180, 255, 255, 255, 255, 255, 231, 255, 255, 42, 0, 0],
            &[0, 0, 30, 197, 255, 255, 245, 141, 30, 255, 255, 42, 0, 0],
            &[0, 0, 0, 0, 20, 38, 9, 0, 102, 253, 94, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 28, 248, 192, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 76, 255, 203, 56, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 34, 113, 112, 17, 0, 0],
        ]),
        // letter: 'Ć' / 0x106
        'Ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 5, 172, 190, 182, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 125, 255, 246, 71, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 53, 250, 236, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 43, 75, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 47, 132, 174, 186, 160, 104, 16, 0, 0],
            &[0, 0, 0, 5, 155, 254, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 0, 157, 255, 255, 248, 190, 176, 215, 248, 19, 0, 0],
            &[0, 0, 57, 255, 255, 214, 29, 0, 0, 0, 31, 0, 0, 0],
            &[0, 0, 157, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 219, 255, 238, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 211, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 255, 255, 195, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 247, 255, 210, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 217, 255, 249, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 251, 255, 253, 147, 74, 69, 109, 195, 26, 0, 0],
            &[0, 0, 0, 120, 255, 255, 255, 255, 255, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 86, 208, 255, 255, 255, 244, 173, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 40, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ć' / 0x107
        'ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 60, 141, 141, 87, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 222, 255, 207, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 194, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 124, 113, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 62, 118, 138, 123, 83, 13, 0, 0, 0],
            &[0, 0, 0, 30, 202, 255, 255, 255, 255, 255, 182, 0, 0, 0],
            &[0, 0, 3, 203, 255, 255, 245, 205, 219, 255, 92, 0, 0, 0],
            &[0, 0, 70, 255, 255, 206, 19, 0, 0, 26, 8, 0, 0, 0],
            &[0, 0, 136, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 161, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 159, 255, 255, 45, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 255, 255, 113, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 255, 255, 243, 105, 45, 65, 137, 125, 0, 0, 0],
            &[0, 0, 0, 170, 255, 255, 255, 255, 255, 255, 142, 0, 0, 0],
            &[0, 0, 0, 5, 132, 232, 255, 255, 255, 218, 81, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 26, 42, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĉ' / 0x108
        'Ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 3, 168, 190, 190, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 128, 255, 232, 255, 229, 22, 0, 0, 0],
            &[0, 0, 0, 0, 76, 253, 205, 21, 113, 253, 190, 4, 0, 0],
            &[0, 0, 0, 0, 56, 74, 9, 0, 0, 49, 75, 15, 0, 0],
            &[0, 0, 0, 0, 0, 47, 132, 174, 186, 160, 104, 16, 0, 0],
            &[0, 0, 0, 5, 155, 254, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 0, 157, 255, 255, 248, 190, 176, 215, 248, 19, 0, 0],
            &[0, 0, 57, 255, 255, 214, 29, 0, 0, 0, 31, 0, 0, 0],
            &[0, 0, 157, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 219, 255, 238, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 211, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 255, 255, 195, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 247, 255, 210, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 217, 255, 249, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 251, 255, 253, 147, 74, 69, 109, 195, 26, 0, 0],
            &[0, 0, 0, 120, 255, 255, 255, 255, 255, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 86, 208, 255, 255, 255, 244, 173, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 40, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĉ' / 0x109
        'ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 55, 141, 141, 108, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 220, 255, 252, 255, 75, 0, 0, 0, 0],
            &[0, 0, 0, 1, 173, 255, 139, 64, 242, 237, 33, 0, 0, 0],
            &[0, 0, 0, 31, 124, 86, 0, 0, 40, 124, 75, 0, 0, 0],
            &[0, 0, 0, 0, 1, 62, 118, 138, 123, 83, 13, 0, 0, 0],
            &[0, 0, 0, 30, 202, 255, 255, 255, 255, 255, 182, 0, 0, 0],
            &[0, 0, 3, 203, 255, 255, 245, 205, 219, 255, 92, 0, 0, 0],
            &[0, 0, 70, 255, 255, 206, 19, 0, 0, 26, 8, 0, 0, 0],
            &[0, 0, 136, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 161, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 159, 255, 255, 45, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 255, 255, 113, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 255, 255, 243, 105, 45, 65, 137, 125, 0, 0, 0],
            &[0, 0, 0, 170, 255, 255, 255, 255, 255, 255, 142, 0, 0, 0],
            &[0, 0, 0, 5, 132, 232, 255, 255, 255, 218, 81, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 26, 42, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ċ' / 0x10a
        'Ċ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 118, 228, 196, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 255, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 255, 248, 43, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 55, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 47, 132, 174, 186, 160, 104, 16, 0, 0],
            &[0, 0, 0, 5, 155, 254, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 0, 157, 255, 255, 248, 190, 176, 215, 248, 19, 0, 0],
            &[0, 0, 57, 255, 255, 214, 29, 0, 0, 0, 31, 0, 0, 0],
            &[0, 0, 157, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 219, 255, 238, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 211, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 255, 255, 195, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 247, 255, 210, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 217, 255, 249, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 251, 255, 253, 147, 74, 69, 109, 195, 26, 0, 0],
            &[0, 0, 0, 120, 255, 255, 255, 255, 255, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 86, 208, 255, 255, 255, 244, 173, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 40, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ċ' / 0x10b
        'ċ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 156, 177, 61, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 115, 255, 255, 201, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 94, 255, 255, 176, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 82, 100, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 62, 118, 138, 123, 83, 13, 0, 0, 0],
            &[0, 0, 0, 30, 202, 255, 255, 255, 255, 255, 182, 0, 0, 0],
            &[0, 0, 3, 203, 255, 255, 245, 205, 219, 255, 92, 0, 0, 0],
            &[0, 0, 70, 255, 255, 206, 19, 0, 0, 26, 8, 0, 0, 0],
            &[0, 0, 136, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 161, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 159, 255, 255, 45, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 255, 255, 113, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 255, 255, 243, 105, 45, 65, 137, 125, 0, 0, 0],
            &[0, 0, 0, 170, 255, 255, 255, 255, 255, 255, 142, 0, 0, 0],
            &[0, 0, 0, 5, 132, 232, 255, 255, 255, 218, 81, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 26, 42, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Č' / 0x10c
        'Č' => Some(&[
            &[0, 0, 0, 0, 116, 190, 77, 0, 20, 164, 183, 19, 0, 0],
            &[0, 0, 0, 0, 15, 219, 253, 116, 216, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 251, 255, 255, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 56, 75, 75, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 47, 132, 174, 186, 160, 104, 16, 0, 0],
            &[0, 0, 0, 5, 155, 254, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 0, 157, 255, 255, 248, 190, 176, 215, 248, 19, 0, 0],
            &[0, 0, 57, 255, 255, 214, 29, 0, 0, 0, 31, 0, 0, 0],
            &[0, 0, 157, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 219, 255, 238, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 211, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 255, 255, 195, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 247, 255, 210, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 217, 255, 249, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 251, 255, 253, 147, 74, 69, 109, 195, 26, 0, 0],
            &[0, 0, 0, 120, 255, 255, 255, 255, 255, 255, 255, 27, 0, 0],
            &[0, 0, 0, 0, 86, 208, 255, 255, 255, 244, 173, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 40, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'č' / 0x10d
        'č' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 141, 103, 0, 0, 56, 141, 82, 0, 0, 0],
            &[0, 0, 0, 0, 159, 255, 156, 85, 249, 228, 24, 0, 0, 0],
            &[0, 0, 0, 0, 9, 212, 255, 255, 253, 61, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 124, 124, 89, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 62, 118, 138, 123, 83, 13, 0, 0, 0],
            &[0, 0, 0, 30, 202, 255, 255, 255, 255, 255, 182, 0, 0, 0],
            &[0, 0, 3, 203, 255, 255, 245, 205, 219, 255, 92, 0, 0, 0],
            &[0, 0, 70, 255, 255, 206, 19, 0, 0, 26, 8, 0, 0, 0],
            &[0, 0, 136, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 161, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 159, 255, 255, 45, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 255, 255, 113, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 255, 255, 243, 105, 45, 65, 137, 125, 0, 0, 0],
            &[0, 0, 0, 170, 255, 255, 255, 255, 255, 255, 142, 0, 0, 0],
            &[0, 0, 0, 5, 132, 232, 255, 255, 255, 218, 81, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 26, 42, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ď' / 0x10e
        'Ď' => Some(&[
            &[0, 0, 6, 173, 175, 29, 0, 60, 187, 140, 0, 0, 0, 0],
            &[0, 0, 0, 66, 252, 228, 109, 249, 238, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 255, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 75, 75, 66, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 144, 144, 131, 99, 21, 0, 0, 0, 0, 0],
            &[0, 37, 255, 255, 255, 255, 255, 255, 249, 127, 1, 0, 0, 0],
            &[0, 37, 255, 255, 230, 197, 215, 254, 255, 255, 130, 0, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 50, 233, 255, 251, 34, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 104, 255, 255, 119, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 29, 255, 255, 176, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 2, 254, 255, 198, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 0, 251, 255, 203, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 16, 255, 255, 180, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 73, 255, 255, 136, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 5, 195, 255, 255, 54, 0, 0],
            &[0, 37, 255, 255, 183, 87, 111, 204, 255, 255, 175, 0, 0, 0],
            &[0, 37, 255, 255, 255, 255, 255, 255, 255, 199, 20, 0, 0, 0],
            &[0, 37, 255, 255, 255, 255, 236, 199, 106, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ď' / 0x10f
        'ď' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 32, 112, 112, 45, 72, 112, 99, 0],
            &[0, 0, 0, 0, 0, 0, 74, 255, 255, 103, 195, 255, 144, 0],
            &[0, 0, 0, 0, 0, 0, 74, 255, 255, 107, 244, 243, 24, 0],
            &[0, 0, 0, 0, 0, 0, 74, 255, 255, 126, 153, 89, 0, 0],
            &[0, 0, 27, 115, 131, 63, 64, 255, 255, 103, 0, 0, 0, 0],
            &[0, 51, 243, 255, 255, 255, 149, 255, 255, 103, 0, 0, 0, 0],
            &[0, 203, 255, 255, 217, 204, 255, 255, 255, 103, 0, 0, 0, 0],
            &[36, 255, 255, 208, 2, 0, 156, 255, 255, 103, 0, 0, 0, 0],
            &[85, 255, 255, 121, 0, 0, 58, 255, 255, 103, 0, 0, 0, 0],
            &[104, 255, 255, 99, 0, 0, 30, 255, 255, 103, 0, 0, 0, 0],
            &[93, 255, 255, 108, 0, 0, 30, 255, 255, 103, 0, 0, 0, 0],
            &[68, 255, 255, 148, 0, 0, 73, 255, 255, 103, 0, 0, 0, 0],
            &[19, 252, 255, 240, 70, 49, 208, 255, 255, 103, 0, 0, 0, 0],
            &[0, 164, 255, 255, 255, 255, 227, 255, 255, 103, 0, 0, 0, 0],
            &[0, 17, 189, 255, 255, 194, 24, 225, 255, 103, 0, 0, 0, 0],
            &[0, 0, 0, 30, 32, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Đ' / 0x110
        'Đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 144, 144, 131, 99, 21, 0, 0, 0, 0, 0],
            &[0, 37, 255, 255, 255, 255, 255, 255, 249, 127, 1, 0, 0, 0],
            &[0, 37, 255, 255, 230, 197, 215, 254, 255, 255, 130, 0, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 50, 233, 255, 251, 34, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 104, 255, 255, 119, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 29, 255, 255, 176, 0, 0],
            &[0, 253, 255, 255, 255, 216, 127, 0, 2, 254, 255, 198, 0, 0],
            &[0, 255, 255, 255, 255, 255, 149, 0, 0, 251, 255, 203, 0, 0],
            &[0, 147, 255, 255, 255, 110, 64, 0, 16, 255, 255, 180, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 0, 73, 255, 255, 136, 0, 0],
            &[0, 37, 255, 255, 146, 0, 0, 5, 195, 255, 255, 54, 0, 0],
            &[0, 37, 255, 255, 183, 87, 111, 204, 255, 255, 175, 0, 0, 0],
            &[0, 37, 255, 255, 255, 255, 255, 255, 255, 199, 20, 0, 0, 0],
            &[0, 37, 255, 255, 255, 255, 236, 199, 106, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'đ' / 0x111
        'đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 38, 112, 112, 39, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 87, 255, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 9, 235, 235, 255, 255, 255, 255, 94, 0],
            &[0, 0, 0, 0, 0, 9, 221, 221, 255, 255, 255, 255, 88, 0],
            &[0, 0, 0, 0, 37, 83, 71, 11, 79, 255, 255, 90, 0, 0],
            &[0, 0, 2, 155, 254, 255, 255, 233, 113, 255, 255, 90, 0, 0],
            &[0, 0, 116, 255, 255, 255, 245, 251, 250, 255, 255, 90, 0, 0],
            &[0, 0, 226, 255, 255, 97, 2, 16, 196, 255, 255, 90, 0, 0],
            &[0, 29, 255, 255, 199, 0, 0, 0, 81, 255, 255, 90, 0, 0],
            &[0, 53, 255, 255, 155, 0, 0, 0, 47, 255, 255, 90, 0, 0],
            &[0, 53, 255, 255, 154, 0, 0, 0, 42, 255, 255, 90, 0, 0],
            &[0, 23, 255, 255, 200, 0, 0, 0, 87, 255, 255, 90, 0, 0],
            &[0, 0, 216, 255, 255, 123, 36, 67, 221, 255, 255, 90, 0, 0],
            &[0, 0, 98, 255, 255, 255, 255, 255, 224, 255, 255, 90, 0, 0],
            &[0, 0, 0, 126, 246, 255, 254, 165, 17, 234, 255, 90, 0, 0],
            &[0, 0, 0, 0, 11, 40, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ē' / 0x112
        'Ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 228, 228, 228, 228, 228, 155, 0, 0, 0, 0],
            &[0, 0, 0, 18, 253, 253, 253, 253, 253, 173, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 144, 144, 144, 144, 144, 10, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 215, 207, 207, 207, 207, 207, 14, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 93, 58, 58, 58, 58, 45, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 77, 37, 37, 37, 37, 29, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 124, 95, 95, 95, 95, 95, 6, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ē' / 0x113
        'ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 178, 178, 178, 178, 178, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 255, 255, 255, 255, 255, 194, 0, 0, 0, 0],
            &[0, 0, 0, 0, 46, 46, 46, 46, 46, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 98, 133, 126, 76, 4, 0, 0, 0, 0],
            &[0, 0, 0, 87, 244, 255, 255, 255, 255, 210, 30, 0, 0, 0],
            &[0, 0, 51, 251, 255, 242, 158, 177, 252, 255, 197, 0, 0, 0],
            &[0, 0, 169, 255, 248, 42, 0, 0, 109, 255, 255, 49, 0, 0],
            &[0, 0, 236, 255, 213, 69, 69, 69, 88, 255, 255, 96, 0, 0],
            &[0, 5, 255, 255, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 248, 255, 222, 156, 156, 156, 156, 156, 156, 70, 0, 0],
            &[0, 0, 208, 255, 222, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 255, 255, 192, 75, 32, 38, 76, 148, 28, 0, 0],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0],
            &[0, 0, 0, 5, 123, 222, 255, 255, 255, 248, 182, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 42, 29, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĕ' / 0x114
        'Ĕ' => Some(&[
            &[0, 0, 0, 62, 205, 68, 0, 0, 142, 190, 0, 0, 0, 0],
            &[0, 0, 0, 26, 247, 234, 145, 175, 254, 172, 0, 0, 0, 0],
            &[0, 0, 0, 0, 95, 244, 255, 255, 203, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 61, 44, 2, 0, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 144, 144, 144, 144, 144, 10, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 215, 207, 207, 207, 207, 207, 14, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 93, 58, 58, 58, 58, 45, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 77, 37, 37, 37, 37, 29, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 124, 95, 95, 95, 95, 95, 6, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĕ' / 0x115
        'ĕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 38, 156, 59, 0, 0, 90, 155, 4, 0, 0, 0],
            &[0, 0, 0, 22, 251, 217, 98, 120, 238, 212, 0, 0, 0, 0],
            &[0, 0, 0, 0, 119, 254, 255, 255, 241, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 49, 108, 96, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 98, 133, 126, 76, 4, 0, 0, 0, 0],
            &[0, 0, 0, 87, 244, 255, 255, 255, 255, 210, 30, 0, 0, 0],
            &[0, 0, 51, 251, 255, 242, 158, 177, 252, 255, 197, 0, 0, 0],
            &[0, 0, 169, 255, 248, 42, 0, 0, 109, 255, 255, 49, 0, 0],
            &[0, 0, 236, 255, 213, 69, 69, 69, 88, 255, 255, 96, 0, 0],
            &[0, 5, 255, 255, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 248, 255, 222, 156, 156, 156, 156, 156, 156, 70, 0, 0],
            &[0, 0, 208, 255, 222, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 255, 255, 192, 75, 32, 38, 76, 148, 28, 0, 0],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0],
            &[0, 0, 0, 5, 123, 222, 255, 255, 255, 248, 182, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 42, 29, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ė' / 0x116
        'Ė' => Some(&[
            &[0, 0, 0, 0, 0, 99, 227, 204, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 213, 255, 255, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 164, 255, 252, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 53, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 144, 144, 144, 144, 144, 10, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 215, 207, 207, 207, 207, 207, 14, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 93, 58, 58, 58, 58, 45, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 77, 37, 37, 37, 37, 29, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 124, 95, 95, 95, 95, 95, 6, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ė' / 0x117
        'ė' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 175, 161, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 188, 255, 255, 128, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 167, 255, 255, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 100, 82, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 98, 133, 126, 76, 4, 0, 0, 0, 0],
            &[0, 0, 0, 87, 244, 255, 255, 255, 255, 210, 30, 0, 0, 0],
            &[0, 0, 51, 251, 255, 242, 158, 177, 252, 255, 197, 0, 0, 0],
            &[0, 0, 169, 255, 248, 42, 0, 0, 109, 255, 255, 49, 0, 0],
            &[0, 0, 236, 255, 213, 69, 69, 69, 88, 255, 255, 96, 0, 0],
            &[0, 5, 255, 255, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 248, 255, 222, 156, 156, 156, 156, 156, 156, 70, 0, 0],
            &[0, 0, 208, 255, 222, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 255, 255, 192, 75, 32, 38, 76, 148, 28, 0, 0],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0],
            &[0, 0, 0, 5, 123, 222, 255, 255, 255, 248, 182, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 42, 29, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ę' / 0x118
        'Ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 144, 144, 144, 144, 144, 10, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 215, 207, 207, 207, 207, 207, 14, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 93, 58, 58, 58, 58, 45, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 77, 37, 37, 37, 37, 29, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 124, 95, 95, 95, 95, 95, 6, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 48, 236, 164, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 205, 249, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 254, 248, 66, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 11, 101, 120, 44, 0, 0, 0],
        ]),
        // letter: 'ę' / 0x119
        'ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 98, 133, 126, 76, 4, 0, 0, 0, 0],
            &[0, 0, 0, 87, 244, 255, 255, 255, 255, 210, 30, 0, 0, 0],
            &[0, 0, 51, 251, 255, 242, 158, 177, 252, 255, 197, 0, 0, 0],
            &[0, 0, 169, 255, 248, 42, 0, 0, 109, 255, 255, 49, 0, 0],
            &[0, 0, 236, 255, 213, 69, 69, 69, 88, 255, 255, 96, 0, 0],
            &[0, 5, 255, 255, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 248, 255, 222, 156, 156, 156, 156, 156, 156, 70, 0, 0],
            &[0, 0, 208, 255, 222, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 255, 255, 192, 75, 32, 38, 76, 148, 28, 0, 0],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0],
            &[0, 0, 0, 5, 123, 222, 255, 255, 255, 255, 255, 21, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 57, 240, 221, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 131, 255, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 166, 255, 140, 52, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 64, 119, 87, 0, 0, 0],
        ]),
        // letter: 'Ě' / 0x11a
        'Ě' => Some(&[
            &[0, 0, 0, 99, 190, 94, 0, 12, 155, 188, 31, 0, 0, 0],
            &[0, 0, 0, 8, 204, 255, 127, 203, 255, 121, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 244, 255, 255, 186, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 75, 75, 22, 0, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 144, 144, 144, 144, 144, 10, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 215, 207, 207, 207, 207, 207, 14, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 93, 58, 58, 58, 58, 45, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 122, 255, 255, 77, 37, 37, 37, 37, 29, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 124, 95, 95, 95, 95, 95, 6, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ě' / 0x11b
        'ě' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 72, 141, 63, 0, 0, 96, 141, 42, 0, 0, 0],
            &[0, 0, 0, 15, 216, 251, 91, 148, 255, 178, 1, 0, 0, 0],
            &[0, 0, 0, 0, 46, 248, 255, 255, 225, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 124, 124, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 98, 133, 126, 76, 4, 0, 0, 0, 0],
            &[0, 0, 0, 87, 244, 255, 255, 255, 255, 210, 30, 0, 0, 0],
            &[0, 0, 51, 251, 255, 242, 158, 177, 252, 255, 197, 0, 0, 0],
            &[0, 0, 169, 255, 248, 42, 0, 0, 109, 255, 255, 49, 0, 0],
            &[0, 0, 236, 255, 213, 69, 69, 69, 88, 255, 255, 96, 0, 0],
            &[0, 5, 255, 255, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 248, 255, 222, 156, 156, 156, 156, 156, 156, 70, 0, 0],
            &[0, 0, 208, 255, 222, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 255, 255, 192, 75, 32, 38, 76, 148, 28, 0, 0],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0],
            &[0, 0, 0, 5, 123, 222, 255, 255, 255, 248, 182, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 42, 29, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 78, 190, 190, 162, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 233, 254, 233, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 7, 197, 251, 100, 28, 213, 252, 67, 0, 0, 0],
            &[0, 0, 0, 19, 75, 45, 0, 0, 12, 75, 51, 0, 0, 0],
            &[0, 0, 0, 0, 8, 96, 157, 185, 181, 143, 64, 1, 0, 0],
            &[0, 0, 0, 51, 229, 255, 255, 255, 255, 255, 255, 65, 0, 0],
            &[0, 0, 29, 242, 255, 255, 230, 173, 174, 220, 223, 1, 0, 0],
            &[0, 0, 168, 255, 255, 151, 5, 0, 0, 0, 28, 0, 0, 0],
            &[0, 11, 247, 255, 220, 4, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 255, 255, 137, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 255, 255, 98, 0, 0, 134, 163, 163, 163, 88, 0, 0],
            &[0, 104, 255, 255, 88, 0, 0, 211, 255, 255, 255, 139, 0, 0],
            &[0, 85, 255, 255, 107, 0, 0, 135, 164, 250, 255, 139, 0, 0],
            &[0, 52, 255, 255, 163, 0, 0, 0, 0, 241, 255, 139, 0, 0],
            &[0, 2, 234, 255, 247, 33, 0, 0, 0, 241, 255, 139, 0, 0],
            &[0, 0, 121, 255, 255, 222, 91, 45, 61, 246, 255, 139, 0, 0],
            &[0, 0, 6, 187, 255, 255, 255, 255, 255, 255, 255, 139, 0, 0],
            &[0, 0, 0, 5, 122, 227, 255, 255, 255, 252, 201, 71, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 42, 28, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĝ' / 0x11d
        'ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 141, 141, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 255, 252, 255, 203, 6, 0, 0, 0, 0],
            &[0, 0, 0, 49, 246, 231, 47, 167, 255, 148, 0, 0, 0, 0],
            &[0, 0, 0, 90, 123, 27, 0, 1, 98, 124, 16, 0, 0, 0],
            &[0, 0, 0, 8, 98, 141, 119, 30, 0, 78, 95, 33, 0, 0],
            &[0, 0, 15, 207, 255, 255, 255, 243, 95, 252, 255, 90, 0, 0],
            &[0, 0, 145, 255, 255, 246, 196, 225, 255, 255, 255, 90, 0, 0],
            &[0, 2, 239, 255, 250, 46, 0, 4, 189, 255, 255, 90, 0, 0],
            &[0, 37, 255, 255, 183, 0, 0, 0, 91, 255, 255, 90, 0, 0],
            &[0, 56, 255, 255, 150, 0, 0, 0, 63, 255, 255, 90, 0, 0],
            &[0, 51, 255, 255, 158, 0, 0, 0, 67, 255, 255, 90, 0, 0],
            &[0, 21, 255, 255, 212, 0, 0, 0, 111, 255, 255, 90, 0, 0],
            &[0, 0, 210, 255, 255, 136, 38, 73, 232, 255, 255, 90, 0, 0],
            &[0, 0, 87, 255, 255, 255, 255, 255, 242, 255, 255, 90, 0, 0],
            &[0, 0, 0, 112, 243, 255, 255, 180, 118, 255, 255, 90, 0, 0],
            &[0, 0, 0, 0, 8, 40, 22, 0, 121, 255, 255, 77, 0, 0],
            &[0, 0, 60, 53, 0, 0, 0, 17, 209, 255, 255, 37, 0, 0],
            &[0, 0, 132, 255, 236, 206, 201, 243, 255, 255, 187, 0, 0, 0],
            &[0, 0, 15, 82, 137, 158, 156, 131, 65, 2, 0, 0, 0, 0],
        ]),
        // letter: 'Ğ' / 0x11e
        'Ğ' => Some(&[
            &[0, 0, 0, 0, 185, 150, 0, 0, 60, 205, 66, 0, 0, 0],
            &[0, 0, 0, 0, 172, 255, 172, 151, 230, 246, 27, 0, 0, 0],
            &[0, 0, 0, 0, 29, 208, 255, 255, 242, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 47, 60, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 96, 157, 185, 181, 143, 64, 1, 0, 0],
            &[0, 0, 0, 51, 229, 255, 255, 255, 255, 255, 255, 65, 0, 0],
            &[0, 0, 29, 242, 255, 255, 230, 173, 174, 220, 223, 1, 0, 0],
            &[0, 0, 168, 255, 255, 151, 5, 0, 0, 0, 28, 0, 0, 0],
            &[0, 11, 247, 255, 220, 4, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 255, 255, 137, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 255, 255, 98, 0, 0, 134, 163, 163, 163, 88, 0, 0],
            &[0, 104, 255, 255, 88, 0, 0, 211, 255, 255, 255, 139, 0, 0],
            &[0, 85, 255, 255, 107, 0, 0, 135, 164, 250, 255, 139, 0, 0],
            &[0, 52, 255, 255, 163, 0, 0, 0, 0, 241, 255, 139, 0, 0],
            &[0, 2, 234, 255, 247, 33, 0, 0, 0, 241, 255, 139, 0, 0],
            &[0, 0, 121, 255, 255, 222, 91, 45, 61, 246, 255, 139, 0, 0],
            &[0, 0, 6, 187, 255, 255, 255, 255, 255, 255, 255, 139, 0, 0],
            &[0, 0, 0, 5, 122, 227, 255, 255, 255, 252, 201, 71, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 42, 28, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ğ' / 0x11f
        'ğ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 156, 29, 0, 0, 120, 129, 0, 0, 0, 0],
            &[0, 0, 0, 66, 255, 191, 94, 137, 252, 163, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 255, 255, 225, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 66, 112, 88, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 98, 141, 119, 30, 0, 78, 95, 33, 0, 0],
            &[0, 0, 15, 207, 255, 255, 255, 243, 95, 252, 255, 90, 0, 0],
            &[0, 0, 145, 255, 255, 246, 196, 225, 255, 255, 255, 90, 0, 0],
            &[0, 2, 239, 255, 250, 46, 0, 4, 189, 255, 255, 90, 0, 0],
            &[0, 37, 255, 255, 183, 0, 0, 0, 91, 255, 255, 90, 0, 0],
            &[0, 56, 255, 255, 150, 0, 0, 0, 63, 255, 255, 90, 0, 0],
            &[0, 51, 255, 255, 158, 0, 0, 0, 67, 255, 255, 90, 0, 0],
            &[0, 21, 255, 255, 212, 0, 0, 0, 111, 255, 255, 90, 0, 0],
            &[0, 0, 210, 255, 255, 136, 38, 73, 232, 255, 255, 90, 0, 0],
            &[0, 0, 87, 255, 255, 255, 255, 255, 242, 255, 255, 90, 0, 0],
            &[0, 0, 0, 112, 243, 255, 255, 180, 118, 255, 255, 90, 0, 0],
            &[0, 0, 0, 0, 8, 40, 22, 0, 121, 255, 255, 77, 0, 0],
            &[0, 0, 60, 53, 0, 0, 0, 17, 209, 255, 255, 37, 0, 0],
            &[0, 0, 132, 255, 236, 206, 201, 243, 255, 255, 187, 0, 0, 0],
            &[0, 0, 15, 82, 137, 158, 156, 131, 65, 2, 0, 0, 0, 0],
        ]),
        // letter: 'Ġ' / 0x120
        'Ġ' => Some(&[
            &[0, 0, 0, 0, 0, 30, 202, 227, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 111, 255, 255, 215, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 64, 253, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 32, 52, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 96, 157, 185, 181, 143, 64, 1, 0, 0],
            &[0, 0, 0, 51, 229, 255, 255, 255, 255, 255, 255, 65, 0, 0],
            &[0, 0, 29, 242, 255, 255, 230, 173, 174, 220, 223, 1, 0, 0],
            &[0, 0, 168, 255, 255, 151, 5, 0, 0, 0, 28, 0, 0, 0],
            &[0, 11, 247, 255, 220, 4, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 255, 255, 137, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 255, 255, 98, 0, 0, 134, 163, 163, 163, 88, 0, 0],
            &[0, 104, 255, 255, 88, 0, 0, 211, 255, 255, 255, 139, 0, 0],
            &[0, 85, 255, 255, 107, 0, 0, 135, 164, 250, 255, 139, 0, 0],
            &[0, 52, 255, 255, 163, 0, 0, 0, 0, 241, 255, 139, 0, 0],
            &[0, 2, 234, 255, 247, 33, 0, 0, 0, 241, 255, 139, 0, 0],
            &[0, 0, 121, 255, 255, 222, 91, 45, 61, 246, 255, 139, 0, 0],
            &[0, 0, 6, 187, 255, 255, 255, 255, 255, 255, 255, 139, 0, 0],
            &[0, 0, 0, 5, 122, 227, 255, 255, 255, 252, 201, 71, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 42, 28, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ġ' / 0x121
        'ġ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 82, 180, 144, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 237, 255, 255, 79, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 216, 255, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 104, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 98, 141, 119, 30, 0, 78, 95, 33, 0, 0],
            &[0, 0, 15, 207, 255, 255, 255, 243, 95, 252, 255, 90, 0, 0],
            &[0, 0, 145, 255, 255, 246, 196, 225, 255, 255, 255, 90, 0, 0],
            &[0, 2, 239, 255, 250, 46, 0, 4, 189, 255, 255, 90, 0, 0],
            &[0, 37, 255, 255, 183, 0, 0, 0, 91, 255, 255, 90, 0, 0],
            &[0, 56, 255, 255, 150, 0, 0, 0, 63, 255, 255, 90, 0, 0],
            &[0, 51, 255, 255, 158, 0, 0, 0, 67, 255, 255, 90, 0, 0],
            &[0, 21, 255, 255, 212, 0, 0, 0, 111, 255, 255, 90, 0, 0],
            &[0, 0, 210, 255, 255, 136, 38, 73, 232, 255, 255, 90, 0, 0],
            &[0, 0, 87, 255, 255, 255, 255, 255, 242, 255, 255, 90, 0, 0],
            &[0, 0, 0, 112, 243, 255, 255, 180, 118, 255, 255, 90, 0, 0],
            &[0, 0, 0, 0, 8, 40, 22, 0, 121, 255, 255, 77, 0, 0],
            &[0, 0, 60, 53, 0, 0, 0, 17, 209, 255, 255, 37, 0, 0],
            &[0, 0, 132, 255, 236, 206, 201, 243, 255, 255, 187, 0, 0, 0],
            &[0, 0, 15, 82, 137, 158, 156, 131, 65, 2, 0, 0, 0, 0],
        ]),
        // letter: 'Ģ' / 0x122
        'Ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 96, 157, 185, 181, 143, 64, 1, 0, 0],
            &[0, 0, 0, 51, 229, 255, 255, 255, 255, 255, 255, 65, 0, 0],
            &[0, 0, 29, 242, 255, 255, 230, 173, 174, 220, 223, 1, 0, 0],
            &[0, 0, 168, 255, 255, 151, 5, 0, 0, 0, 28, 0, 0, 0],
            &[0, 11, 247, 255, 220, 4, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 255, 255, 137, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 255, 255, 98, 0, 0, 134, 163, 163, 163, 88, 0, 0],
            &[0, 104, 255, 255, 88, 0, 0, 211, 255, 255, 255, 139, 0, 0],
            &[0, 85, 255, 255, 107, 0, 0, 135, 164, 250, 255, 139, 0, 0],
            &[0, 52, 255, 255, 163, 0, 0, 0, 0, 241, 255, 139, 0, 0],
            &[0, 2, 234, 255, 247, 33, 0, 0, 0, 241, 255, 139, 0, 0],
            &[0, 0, 121, 255, 255, 222, 91, 45, 61, 246, 255, 139, 0, 0],
            &[0, 0, 6, 187, 255, 255, 255, 255, 255, 255, 255, 139, 0, 0],
            &[0, 0, 0, 5, 122, 227, 255, 255, 255, 252, 201, 71, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 42, 28, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 214, 215, 91, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 55, 255, 237, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 50, 3, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ģ' / 0x123
        'ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 69, 141, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 230, 253, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 121, 255, 214, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 101, 124, 88, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 98, 141, 119, 30, 0, 78, 95, 33, 0, 0],
            &[0, 0, 15, 207, 255, 255, 255, 243, 95, 252, 255, 90, 0, 0],
            &[0, 0, 145, 255, 255, 246, 196, 225, 255, 255, 255, 90, 0, 0],
            &[0, 2, 239, 255, 250, 46, 0, 4, 189, 255, 255, 90, 0, 0],
            &[0, 37, 255, 255, 183, 0, 0, 0, 91, 255, 255, 90, 0, 0],
            &[0, 56, 255, 255, 150, 0, 0, 0, 63, 255, 255, 90, 0, 0],
            &[0, 51, 255, 255, 158, 0, 0, 0, 67, 255, 255, 90, 0, 0],
            &[0, 21, 255, 255, 212, 0, 0, 0, 111, 255, 255, 90, 0, 0],
            &[0, 0, 210, 255, 255, 136, 38, 73, 232, 255, 255, 90, 0, 0],
            &[0, 0, 87, 255, 255, 255, 255, 255, 242, 255, 255, 90, 0, 0],
            &[0, 0, 0, 112, 243, 255, 255, 180, 118, 255, 255, 90, 0, 0],
            &[0, 0, 0, 0, 8, 40, 22, 0, 121, 255, 255, 77, 0, 0],
            &[0, 0, 60, 53, 0, 0, 0, 17, 209, 255, 255, 37, 0, 0],
            &[0, 0, 132, 255, 236, 206, 201, 243, 255, 255, 187, 0, 0, 0],
            &[0, 0, 15, 82, 137, 158, 156, 131, 65, 2, 0, 0, 0, 0],
        ]),
        // letter: 'Ĥ' / 0x124
        'Ĥ' => Some(&[
            &[0, 0, 0, 0, 5, 171, 190, 190, 66, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 255, 232, 255, 226, 19, 0, 0, 0, 0],
            &[0, 0, 0, 82, 254, 201, 19, 119, 254, 184, 3, 0, 0, 0],
            &[0, 0, 0, 58, 74, 8, 0, 0, 51, 75, 13, 0, 0, 0],
            &[0, 21, 144, 144, 88, 0, 0, 0, 30, 144, 144, 78, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 182, 67, 67, 67, 107, 255, 255, 139, 0, 0],
            &[
                0, 37, 255, 255, 255, 255, 255, 255, 255, 255, 255, 139, 0, 0,
            ],
            &[
                0, 37, 255, 255, 255, 255, 255, 255, 255, 255, 255, 139, 0, 0,
            ],
            &[0, 37, 255, 255, 167, 28, 28, 28, 75, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĥ' / 0x125
        'ĥ' => Some(&[
            &[0, 0, 174, 255, 247, 255, 157, 0, 0, 0, 0, 0, 0, 0],
            &[0, 117, 255, 183, 26, 200, 255, 98, 0, 0, 0, 0, 0, 0],
            &[1, 105, 95, 3, 0, 7, 100, 96, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 107, 112, 90, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 244, 255, 204, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 244, 255, 204, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 244, 255, 204, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 244, 255, 202, 1, 81, 131, 117, 54, 0, 0, 0],
            &[0, 0, 0, 244, 255, 199, 177, 255, 255, 255, 254, 125, 0, 0],
            &[0, 0, 0, 244, 255, 255, 254, 201, 213, 255, 255, 251, 34, 0],
            &[0, 0, 0, 244, 255, 255, 100, 0, 0, 163, 255, 255, 88, 0],
            &[0, 0, 0, 244, 255, 243, 3, 0, 0, 90, 255, 255, 114, 0],
            &[0, 0, 0, 244, 255, 213, 0, 0, 0, 78, 255, 255, 115, 0],
            &[0, 0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0],
            &[0, 0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0],
            &[0, 0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0],
            &[0, 0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0],
            &[0, 0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 157, 158, 154, 13, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ħ' / 0x126
        'Ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 132, 144, 121, 0, 0, 0, 60, 144, 144, 48, 0, 0],
            &[0, 0, 234, 255, 214, 0, 0, 0, 107, 255, 255, 85, 0, 0],
            &[
                0, 185, 255, 255, 255, 185, 185, 185, 255, 255, 255, 255, 74, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 102, 0,
            ],
            &[0, 20, 255, 255, 235, 20, 20, 20, 128, 255, 255, 106, 8, 0],
            &[0, 0, 234, 255, 225, 67, 67, 67, 146, 255, 255, 85, 0, 0],
            &[0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 255, 85, 0, 0],
            &[0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 255, 85, 0, 0],
            &[0, 0, 234, 255, 219, 28, 28, 28, 123, 255, 255, 85, 0, 0],
            &[0, 0, 234, 255, 214, 0, 0, 0, 107, 255, 255, 85, 0, 0],
            &[0, 0, 234, 255, 214, 0, 0, 0, 107, 255, 255, 85, 0, 0],
            &[0, 0, 234, 255, 214, 0, 0, 0, 107, 255, 255, 85, 0, 0],
            &[0, 0, 234, 255, 214, 0, 0, 0, 107, 255, 255, 85, 0, 0],
            &[0, 0, 234, 255, 214, 0, 0, 0, 107, 255, 255, 85, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ħ' / 0x127
        'ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 112, 90, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 235, 255, 255, 255, 235, 235, 144, 0, 0, 0, 0, 0, 0],
            &[0, 221, 255, 255, 255, 221, 221, 135, 0, 0, 0, 0, 0, 0],
            &[0, 0, 244, 255, 203, 0, 35, 83, 68, 19, 0, 0, 0, 0],
            &[0, 0, 244, 255, 196, 130, 253, 255, 255, 241, 86, 0, 0, 0],
            &[0, 0, 244, 255, 248, 255, 245, 250, 255, 255, 242, 22, 0, 0],
            &[0, 0, 244, 255, 255, 140, 4, 12, 191, 255, 255, 81, 0, 0],
            &[0, 0, 244, 255, 249, 10, 0, 0, 96, 255, 255, 111, 0, 0],
            &[0, 0, 244, 255, 216, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 205, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĩ' / 0x128
        'Ĩ' => Some(&[
            &[0, 0, 0, 0, 85, 121, 42, 0, 22, 142, 37, 0, 0, 0],
            &[0, 0, 0, 101, 255, 255, 254, 188, 196, 255, 31, 0, 0, 0],
            &[0, 0, 0, 197, 210, 102, 190, 255, 255, 160, 0, 0, 0, 0],
            &[0, 0, 0, 70, 40, 0, 0, 36, 56, 1, 0, 0, 0, 0],
            &[0, 0, 42, 144, 144, 144, 144, 144, 144, 144, 99, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 26, 108, 138, 221, 255, 255, 148, 119, 65, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 27, 182, 255, 254, 37, 9, 0, 0, 0, 0],
            &[0, 0, 68, 250, 255, 255, 255, 255, 255, 255, 165, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĩ' / 0x129
        'ĩ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 72, 11, 0, 11, 93, 26, 0, 0, 0],
            &[0, 0, 0, 72, 252, 255, 237, 139, 158, 255, 42, 0, 0, 0],
            &[0, 0, 0, 185, 230, 152, 232, 255, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 112, 70, 0, 7, 85, 105, 14, 0, 0, 0, 0],
            &[0, 0, 33, 95, 95, 95, 95, 95, 33, 0, 0, 0, 0, 0],
            &[0, 0, 88, 255, 255, 255, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 53, 172, 201, 236, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 138, 255, 255, 126, 18, 0, 0, 0, 0],
            &[0, 0, 171, 238, 255, 255, 255, 255, 255, 255, 226, 86, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ī' / 0x12a
        'Ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 228, 228, 228, 228, 228, 129, 0, 0, 0, 0],
            &[0, 0, 0, 47, 253, 253, 253, 253, 253, 144, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 144, 144, 144, 144, 144, 144, 144, 99, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 26, 108, 138, 221, 255, 255, 148, 119, 65, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 27, 182, 255, 254, 37, 9, 0, 0, 0, 0],
            &[0, 0, 68, 250, 255, 255, 255, 255, 255, 255, 165, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ī' / 0x12b
        'ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 33, 178, 178, 178, 178, 178, 101, 0, 0, 0, 0],
            &[0, 0, 0, 48, 255, 255, 255, 255, 255, 145, 0, 0, 0, 0],
            &[0, 0, 0, 8, 46, 46, 46, 46, 46, 26, 0, 0, 0, 0],
            &[0, 0, 33, 95, 95, 95, 95, 95, 33, 0, 0, 0, 0, 0],
            &[0, 0, 88, 255, 255, 255, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 53, 172, 201, 236, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 138, 255, 255, 126, 18, 0, 0, 0, 0],
            &[0, 0, 171, 238, 255, 255, 255, 255, 255, 255, 226, 86, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => Some(&[
            &[0, 0, 0, 85, 205, 45, 0, 0, 165, 166, 0, 0, 0, 0],
            &[0, 0, 0, 49, 254, 224, 143, 186, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 118, 250, 255, 255, 187, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 63, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 144, 144, 144, 144, 144, 144, 144, 99, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 26, 108, 138, 221, 255, 255, 148, 119, 65, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 27, 182, 255, 254, 37, 9, 0, 0, 0, 0],
            &[0, 0, 68, 250, 255, 255, 255, 255, 255, 255, 165, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĭ' / 0x12d
        'ĭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 156, 29, 0, 0, 120, 129, 0, 0, 0, 0],
            &[0, 0, 0, 66, 255, 191, 94, 137, 252, 163, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 255, 255, 225, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 66, 112, 88, 11, 0, 0, 0, 0, 0],
            &[0, 0, 33, 95, 95, 95, 95, 95, 33, 0, 0, 0, 0, 0],
            &[0, 0, 88, 255, 255, 255, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 53, 172, 201, 236, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 138, 255, 255, 126, 18, 0, 0, 0, 0],
            &[0, 0, 171, 238, 255, 255, 255, 255, 255, 255, 226, 86, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Į' / 0x12e
        'Į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 144, 144, 144, 144, 144, 144, 144, 99, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 26, 108, 138, 221, 255, 255, 148, 119, 65, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 27, 182, 255, 254, 37, 9, 0, 0, 0, 0],
            &[0, 0, 68, 250, 255, 255, 255, 255, 255, 255, 165, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 40, 231, 176, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 192, 253, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 248, 252, 71, 36, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 97, 120, 49, 0, 0, 0, 0],
        ]),
        // letter: 'į' / 0x12f
        'į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 35, 167, 170, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 255, 255, 162, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 255, 137, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 93, 92, 7, 0, 0, 0, 0, 0],
            &[0, 0, 33, 95, 95, 95, 95, 95, 33, 0, 0, 0, 0, 0],
            &[0, 0, 88, 255, 255, 255, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 53, 172, 201, 236, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 138, 255, 255, 126, 18, 0, 0, 0, 0],
            &[0, 0, 171, 238, 255, 255, 255, 255, 255, 255, 226, 86, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 48, 236, 164, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 205, 249, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 254, 248, 66, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 11, 101, 120, 44, 0, 0, 0],
        ]),
        // letter: 'İ' / 0x130
        'İ' => Some(&[
            &[0, 0, 0, 0, 0, 123, 229, 193, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 242, 255, 255, 84, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 247, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 55, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 144, 144, 144, 144, 144, 144, 144, 99, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 26, 108, 138, 221, 255, 255, 148, 119, 65, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 27, 182, 255, 254, 37, 9, 0, 0, 0, 0],
            &[0, 0, 68, 250, 255, 255, 255, 255, 255, 255, 165, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ı' / 0x131
        'ı' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 95, 95, 95, 95, 95, 33, 0, 0, 0, 0, 0],
            &[0, 0, 88, 255, 255, 255, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 53, 172, 201, 236, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 138, 255, 255, 126, 18, 0, 0, 0, 0],
            &[0, 0, 171, 238, 255, 255, 255, 255, 255, 255, 226, 86, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĳ' / 0x132
        'Ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 55, 144, 144, 37, 0, 0, 132, 144, 113, 0, 0, 0],
            &[0, 0, 98, 255, 255, 65, 0, 0, 233, 255, 200, 0, 0, 0],
            &[0, 0, 98, 255, 255, 65, 0, 0, 233, 255, 200, 0, 0, 0],
            &[0, 0, 98, 255, 255, 65, 0, 0, 233, 255, 200, 0, 0, 0],
            &[0, 0, 98, 255, 255, 65, 0, 0, 233, 255, 200, 0, 0, 0],
            &[0, 0, 98, 255, 255, 65, 0, 0, 233, 255, 200, 0, 0, 0],
            &[0, 0, 98, 255, 255, 65, 0, 0, 233, 255, 200, 0, 0, 0],
            &[0, 0, 98, 255, 255, 65, 0, 0, 233, 255, 200, 0, 0, 0],
            &[0, 0, 98, 255, 255, 65, 0, 0, 233, 255, 200, 0, 0, 0],
            &[0, 0, 88, 229, 229, 58, 0, 0, 242, 255, 195, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 36, 255, 255, 173, 0, 0, 0],
            &[0, 0, 49, 106, 71, 56, 81, 202, 255, 255, 110, 0, 0, 0],
            &[0, 0, 93, 255, 255, 255, 255, 255, 255, 223, 14, 0, 0, 0],
            &[0, 0, 77, 245, 255, 255, 255, 247, 171, 28, 0, 0, 0, 0],
            &[0, 0, 0, 2, 29, 45, 33, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĳ' / 0x133
        'ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 24, 161, 175, 52, 0, 6, 144, 180, 81, 0, 0, 0],
            &[0, 0, 130, 255, 255, 186, 0, 83, 255, 255, 234, 0, 0, 0],
            &[0, 0, 109, 255, 255, 161, 0, 62, 255, 255, 209, 0, 0, 0],
            &[0, 0, 3, 86, 97, 12, 0, 0, 70, 103, 26, 0, 0, 0],
            &[0, 0, 29, 95, 95, 42, 0, 8, 95, 95, 64, 0, 0, 0],
            &[0, 0, 79, 255, 255, 114, 0, 21, 255, 255, 171, 0, 0, 0],
            &[0, 0, 79, 255, 255, 114, 0, 21, 255, 255, 171, 0, 0, 0],
            &[0, 0, 79, 255, 255, 114, 0, 21, 255, 255, 171, 0, 0, 0],
            &[0, 0, 79, 255, 255, 114, 0, 21, 255, 255, 171, 0, 0, 0],
            &[0, 0, 79, 255, 255, 114, 0, 21, 255, 255, 171, 0, 0, 0],
            &[0, 0, 79, 255, 255, 114, 0, 21, 255, 255, 171, 0, 0, 0],
            &[0, 0, 79, 255, 255, 114, 0, 21, 255, 255, 171, 0, 0, 0],
            &[0, 0, 79, 255, 255, 114, 0, 21, 255, 255, 171, 0, 0, 0],
            &[0, 0, 79, 255, 255, 114, 0, 21, 255, 255, 171, 0, 0, 0],
            &[0, 0, 79, 255, 255, 114, 0, 22, 255, 255, 171, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 46, 255, 255, 168, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 135, 255, 255, 144, 0, 0, 0],
            &[0, 0, 0, 182, 220, 194, 222, 255, 255, 255, 72, 0, 0, 0],
            &[0, 0, 0, 57, 125, 154, 162, 137, 65, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĵ' / 0x134
        'Ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 137, 190, 190, 105, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 83, 255, 237, 250, 248, 49, 0, 0, 0],
            &[0, 0, 0, 0, 42, 241, 228, 44, 76, 244, 221, 18, 0, 0],
            &[0, 0, 0, 0, 42, 75, 22, 0, 0, 35, 75, 28, 0, 0],
            &[0, 0, 0, 0, 0, 0, 56, 144, 144, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 108, 255, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 155, 255, 255, 52, 0, 0, 0, 0],
            &[0, 52, 105, 69, 63, 124, 253, 255, 240, 5, 0, 0, 0, 0],
            &[0, 98, 255, 255, 255, 255, 255, 255, 120, 0, 0, 0, 0, 0],
            &[0, 81, 245, 255, 255, 255, 226, 111, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 31, 43, 25, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĵ' / 0x135
        'ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 137, 141, 141, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 141, 255, 252, 255, 167, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 254, 209, 37, 198, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 110, 117, 12, 0, 7, 112, 119, 0, 0, 0],
            &[0, 0, 0, 52, 95, 95, 95, 95, 95, 14, 0, 0, 0, 0],
            &[0, 0, 0, 140, 255, 255, 255, 255, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 86, 178, 206, 244, 255, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 155, 255, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 179, 255, 255, 35, 0, 0, 0, 0],
            &[0, 4, 1, 0, 0, 31, 237, 255, 253, 13, 0, 0, 0, 0],
            &[0, 61, 240, 203, 201, 245, 255, 255, 195, 0, 0, 0, 0, 0],
            &[0, 16, 100, 144, 162, 154, 107, 18, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ķ' / 0x136
        'Ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 80, 0, 0, 0, 65, 144, 144, 80, 0, 0],
            &[0, 37, 255, 255, 141, 0, 0, 18, 231, 255, 233, 22, 0, 0],
            &[0, 37, 255, 255, 141, 0, 0, 164, 255, 255, 76, 0, 0, 0],
            &[0, 37, 255, 255, 141, 0, 79, 255, 255, 151, 0, 0, 0, 0],
            &[0, 37, 255, 255, 141, 18, 231, 255, 216, 10, 0, 0, 0, 0],
            &[0, 37, 255, 255, 141, 160, 255, 250, 51, 0, 0, 0, 0, 0],
            &[0, 37, 255, 255, 195, 254, 255, 162, 0, 0, 0, 0, 0, 0],
            &[0, 37, 255, 255, 255, 255, 255, 242, 22, 0, 0, 0, 0, 0],
            &[0, 37, 255, 255, 255, 198, 255, 255, 151, 0, 0, 0, 0, 0],
            &[0, 37, 255, 255, 178, 2, 194, 255, 251, 41, 0, 0, 0, 0],
            &[0, 37, 255, 255, 141, 0, 59, 255, 255, 179, 0, 0, 0, 0],
            &[0, 37, 255, 255, 141, 0, 0, 180, 255, 255, 65, 0, 0, 0],
            &[0, 37, 255, 255, 141, 0, 0, 46, 253, 255, 205, 1, 0, 0],
            &[0, 37, 255, 255, 141, 0, 0, 0, 165, 255, 255, 92, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 108, 215, 202, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 172, 255, 135, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 50, 30, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ķ' / 0x137
        'ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 96, 112, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 0, 46, 95, 95, 67, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 36, 236, 255, 238, 42, 0, 0, 0],
            &[0, 220, 255, 229, 0, 17, 216, 255, 248, 62, 0, 0, 0, 0],
            &[0, 220, 255, 227, 4, 189, 255, 254, 87, 0, 0, 0, 0, 0],
            &[0, 220, 255, 213, 143, 255, 255, 117, 0, 0, 0, 0, 0, 0],
            &[0, 220, 255, 240, 255, 255, 255, 113, 0, 0, 0, 0, 0, 0],
            &[0, 220, 255, 255, 255, 231, 255, 246, 37, 0, 0, 0, 0, 0],
            &[0, 220, 255, 249, 88, 38, 247, 255, 196, 2, 0, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 119, 255, 255, 114, 0, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 4, 205, 255, 246, 38, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 0, 48, 251, 255, 197, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 215, 215, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 255, 198, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 43, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĸ' / 0x138
        'ĸ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 82, 95, 85, 0, 0, 0, 46, 95, 95, 67, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 36, 236, 255, 238, 42, 0, 0, 0],
            &[0, 220, 255, 229, 0, 17, 216, 255, 248, 62, 0, 0, 0, 0],
            &[0, 220, 255, 227, 4, 189, 255, 254, 87, 0, 0, 0, 0, 0],
            &[0, 220, 255, 213, 143, 255, 255, 117, 0, 0, 0, 0, 0, 0],
            &[0, 220, 255, 240, 255, 255, 255, 113, 0, 0, 0, 0, 0, 0],
            &[0, 220, 255, 255, 255, 231, 255, 246, 37, 0, 0, 0, 0, 0],
            &[0, 220, 255, 249, 88, 38, 247, 255, 196, 2, 0, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 119, 255, 255, 114, 0, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 4, 205, 255, 246, 38, 0, 0, 0],
            &[0, 220, 255, 229, 0, 0, 0, 48, 251, 255, 197, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĺ' / 0x139
        'Ĺ' => Some(&[
            &[0, 0, 0, 0, 114, 190, 190, 79, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 249, 255, 146, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 211, 255, 126, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 75, 55, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 144, 144, 67, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 178, 110, 110, 110, 110, 110, 7, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĺ' / 0x13a
        'ĺ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 110, 255, 252, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 246, 246, 79, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 107, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 112, 112, 112, 112, 112, 35, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 14, 105, 158, 224, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 146, 255, 255, 118, 17, 0, 0, 0, 0],
            &[0, 0, 172, 239, 255, 255, 255, 255, 255, 255, 225, 86, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 143, 158, 155, 21, 0, 0, 0],
        ]),
        // letter: 'Ļ' / 0x13b
        'Ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 144, 144, 67, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 178, 110, 110, 110, 110, 110, 7, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 215, 215, 83, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 65, 255, 231, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 50, 1, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ļ' / 0x13c
        'ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 112, 112, 112, 112, 112, 35, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 14, 105, 158, 224, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 146, 255, 255, 118, 17, 0, 0, 0, 0],
            &[0, 0, 172, 239, 255, 255, 255, 255, 255, 255, 225, 86, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 215, 213, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 138, 255, 169, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 37, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ľ' / 0x13d
        'Ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 144, 144, 67, 0, 0, 86, 144, 132, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 187, 255, 142, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 2, 239, 241, 22, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 13, 121, 72, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 178, 110, 110, 110, 110, 110, 7, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ľ' / 0x13e
        'ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 112, 112, 112, 112, 112, 35, 28, 112, 112, 30, 0, 0],
            &[0, 49, 255, 255, 255, 255, 255, 80, 96, 255, 233, 9, 0, 0],
            &[0, 14, 105, 158, 224, 255, 255, 80, 149, 255, 110, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 255, 80, 117, 143, 4, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 146, 255, 255, 118, 17, 0, 0, 0, 0, 0],
            &[0, 172, 239, 255, 255, 255, 255, 255, 255, 225, 86, 0, 0, 0],
            &[0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 144, 144, 67, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 6, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 167, 255, 214, 18, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 11, 255, 255, 255, 80, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 208, 255, 243, 31, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 10, 61, 23, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 178, 110, 110, 110, 110, 110, 7, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŀ' / 0x140
        'ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[21, 112, 112, 112, 112, 112, 35, 0, 0, 0, 0, 0, 0, 0],
            &[49, 255, 255, 255, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[14, 105, 158, 224, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 255, 255, 80, 0, 3, 154, 208, 125, 0, 0],
            &[0, 0, 0, 112, 255, 255, 80, 0, 60, 255, 255, 255, 23, 0],
            &[0, 0, 0, 112, 255, 255, 80, 0, 35, 252, 255, 239, 9, 0],
            &[0, 0, 0, 112, 255, 255, 80, 0, 0, 60, 115, 43, 0, 0],
            &[0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 146, 255, 255, 118, 17, 0, 0, 0, 0, 0, 0],
            &[172, 239, 255, 255, 255, 255, 255, 255, 225, 86, 0, 0, 0, 0],
            &[220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ł' / 0x141
        'Ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 144, 144, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 135, 255, 255, 57, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 135, 255, 255, 57, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 135, 255, 255, 57, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 135, 255, 255, 60, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 135, 255, 255, 255, 255, 65, 0, 0, 0, 0, 0],
            &[0, 0, 0, 153, 255, 255, 255, 232, 76, 0, 0, 0, 0, 0],
            &[0, 1, 97, 255, 255, 255, 203, 15, 0, 0, 0, 0, 0, 0],
            &[0, 16, 236, 255, 255, 255, 57, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 94, 232, 255, 255, 57, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 135, 255, 255, 57, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 135, 255, 255, 143, 110, 110, 110, 110, 91, 0, 0],
            &[0, 0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 212, 0, 0],
            &[0, 0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 212, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ł' / 0x142
        'ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 112, 112, 112, 112, 112, 35, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 14, 105, 158, 224, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 96, 84, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 255, 252, 46, 0, 0, 0],
            &[0, 0, 0, 0, 0, 152, 255, 255, 255, 202, 44, 0, 0, 0],
            &[0, 0, 0, 9, 131, 255, 255, 255, 185, 2, 0, 0, 0, 0],
            &[0, 0, 0, 149, 255, 255, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 204, 175, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 146, 255, 255, 118, 17, 0, 0, 0, 0],
            &[0, 0, 172, 239, 255, 255, 255, 255, 255, 255, 225, 86, 0, 0],
            &[0, 0, 220, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ń' / 0x143
        'Ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 1, 162, 190, 186, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 108, 255, 250, 85, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 245, 242, 71, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 75, 36, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 144, 100, 0, 0, 0, 131, 144, 78, 0, 0],
            &[0, 37, 255, 255, 255, 237, 4, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 255, 255, 69, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 200, 255, 153, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 132, 255, 234, 3, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 74, 244, 255, 65, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 75, 170, 255, 148, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 87, 255, 230, 2, 230, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 12, 247, 255, 60, 226, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 177, 255, 143, 219, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 94, 255, 226, 210, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 17, 250, 255, 246, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 0, 184, 255, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 0, 101, 255, 255, 255, 139, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ń' / 0x144
        'ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 141, 141, 31, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 80, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 233, 253, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 124, 71, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 95, 19, 3, 83, 131, 119, 57, 0, 0, 0, 0],
            &[0, 0, 244, 255, 124, 200, 255, 255, 255, 255, 129, 0, 0, 0],
            &[0, 0, 244, 255, 255, 254, 209, 215, 255, 255, 252, 35, 0, 0],
            &[0, 0, 244, 255, 255, 119, 0, 1, 170, 255, 255, 88, 0, 0],
            &[0, 0, 244, 255, 247, 10, 0, 0, 91, 255, 255, 114, 0, 0],
            &[0, 0, 244, 255, 218, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ņ' / 0x145
        'Ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 144, 100, 0, 0, 0, 131, 144, 78, 0, 0],
            &[0, 37, 255, 255, 255, 237, 4, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 255, 255, 69, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 200, 255, 153, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 132, 255, 234, 3, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 74, 244, 255, 65, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 75, 170, 255, 148, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 87, 255, 230, 2, 230, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 12, 247, 255, 60, 226, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 177, 255, 143, 219, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 94, 255, 226, 210, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 17, 250, 255, 246, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 0, 184, 255, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 0, 101, 255, 255, 255, 139, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 141, 215, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 255, 96, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 50, 23, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ņ' / 0x146
        'ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 95, 19, 3, 83, 131, 119, 57, 0, 0, 0, 0],
            &[0, 0, 244, 255, 124, 200, 255, 255, 255, 255, 129, 0, 0, 0],
            &[0, 0, 244, 255, 255, 254, 209, 215, 255, 255, 252, 35, 0, 0],
            &[0, 0, 244, 255, 255, 119, 0, 1, 170, 255, 255, 88, 0, 0],
            &[0, 0, 244, 255, 247, 10, 0, 0, 91, 255, 255, 114, 0, 0],
            &[0, 0, 244, 255, 218, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 125, 215, 191, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 255, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 50, 27, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ň' / 0x147
        'Ň' => Some(&[
            &[0, 0, 0, 102, 190, 90, 0, 14, 157, 187, 28, 0, 0, 0],
            &[0, 0, 0, 9, 207, 255, 124, 206, 255, 117, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 246, 255, 255, 181, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 75, 75, 21, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 144, 100, 0, 0, 0, 131, 144, 78, 0, 0],
            &[0, 37, 255, 255, 255, 237, 4, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 255, 255, 69, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 200, 255, 153, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 132, 255, 234, 3, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 74, 244, 255, 65, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 75, 170, 255, 148, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 87, 255, 230, 2, 230, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 12, 247, 255, 60, 226, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 177, 255, 143, 219, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 94, 255, 226, 210, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 17, 250, 255, 246, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 0, 184, 255, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 0, 101, 255, 255, 255, 139, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ň' / 0x148
        'ň' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 89, 141, 47, 0, 2, 110, 141, 25, 0, 0, 0],
            &[0, 0, 0, 28, 232, 244, 73, 173, 255, 150, 0, 0, 0, 0],
            &[0, 0, 0, 0, 69, 254, 255, 255, 205, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 95, 124, 124, 40, 0, 0, 0, 0, 0],
            &[0, 0, 91, 95, 19, 3, 83, 131, 119, 57, 0, 0, 0, 0],
            &[0, 0, 244, 255, 124, 200, 255, 255, 255, 255, 129, 0, 0, 0],
            &[0, 0, 244, 255, 255, 254, 209, 215, 255, 255, 252, 35, 0, 0],
            &[0, 0, 244, 255, 255, 119, 0, 1, 170, 255, 255, 88, 0, 0],
            &[0, 0, 244, 255, 247, 10, 0, 0, 91, 255, 255, 114, 0, 0],
            &[0, 0, 244, 255, 218, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŉ' / 0x149
        'ŉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 144, 144, 64, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 155, 255, 229, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 255, 136, 73, 95, 37, 0, 70, 127, 122, 61, 0, 0],
            &[
                11, 251, 253, 32, 195, 255, 162, 166, 255, 255, 255, 255, 134, 0,
            ],
            &[
                3, 22, 20, 0, 195, 255, 255, 255, 217, 210, 255, 255, 252, 37,
            ],
            &[0, 0, 0, 0, 195, 255, 255, 161, 0, 0, 166, 255, 255, 89],
            &[0, 0, 0, 0, 195, 255, 255, 48, 0, 0, 90, 255, 255, 114],
            &[0, 0, 0, 0, 195, 255, 255, 10, 0, 0, 78, 255, 255, 115],
            &[0, 0, 0, 0, 195, 255, 253, 0, 0, 0, 78, 255, 255, 115],
            &[0, 0, 0, 0, 195, 255, 253, 0, 0, 0, 78, 255, 255, 115],
            &[0, 0, 0, 0, 195, 255, 253, 0, 0, 0, 78, 255, 255, 115],
            &[0, 0, 0, 0, 195, 255, 253, 0, 0, 0, 78, 255, 255, 115],
            &[0, 0, 0, 0, 195, 255, 253, 0, 0, 0, 78, 255, 255, 115],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 144, 100, 0, 0, 0, 131, 144, 78, 0, 0],
            &[0, 37, 255, 255, 255, 237, 4, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 255, 255, 69, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 200, 255, 153, 0, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 132, 255, 234, 3, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 74, 244, 255, 65, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 75, 170, 255, 148, 0, 231, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 87, 255, 230, 2, 230, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 12, 247, 255, 60, 226, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 177, 255, 143, 219, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 94, 255, 226, 210, 255, 139, 0, 0],
            &[0, 37, 255, 255, 78, 0, 17, 250, 255, 255, 255, 152, 0, 0],
            &[0, 37, 255, 255, 78, 0, 0, 184, 255, 255, 255, 255, 0, 0],
            &[0, 37, 255, 255, 78, 0, 0, 101, 255, 255, 255, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 213, 255, 129, 0, 0],
            &[0, 0, 0, 0, 0, 23, 21, 29, 118, 254, 255, 80, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 255, 255, 255, 200, 3, 0, 0],
            &[0, 0, 0, 0, 0, 96, 228, 238, 210, 129, 11, 0, 0, 0],
        ]),
        // letter: 'ŋ' / 0x14b
        'ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 95, 19, 3, 83, 131, 119, 57, 0, 0, 0, 0],
            &[0, 0, 244, 255, 124, 200, 255, 255, 255, 255, 129, 0, 0, 0],
            &[0, 0, 244, 255, 255, 254, 209, 215, 255, 255, 252, 35, 0, 0],
            &[0, 0, 244, 255, 255, 119, 0, 1, 170, 255, 255, 88, 0, 0],
            &[0, 0, 244, 255, 247, 10, 0, 0, 91, 255, 255, 114, 0, 0],
            &[0, 0, 244, 255, 218, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 86, 255, 255, 127, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 156, 255, 255, 230, 0, 0],
            &[0, 0, 244, 255, 204, 0, 0, 0, 156, 255, 255, 230, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 78, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 107, 255, 255, 106, 0, 0],
            &[0, 0, 0, 0, 0, 0, 105, 175, 244, 255, 255, 57, 0, 0],
            &[0, 0, 0, 0, 0, 0, 56, 131, 132, 80, 2, 0, 0, 0],
        ]),
        // letter: 'Ō' / 0x14c
        'Ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 228, 228, 228, 228, 228, 129, 0, 0, 0, 0],
            &[0, 0, 0, 47, 253, 253, 253, 253, 253, 144, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 77, 163, 194, 183, 124, 19, 0, 0, 0, 0],
            &[0, 0, 2, 164, 255, 255, 255, 255, 255, 232, 42, 0, 0, 0],
            &[0, 0, 105, 255, 255, 236, 162, 208, 255, 255, 210, 2, 0, 0],
            &[0, 0, 223, 255, 245, 35, 0, 2, 185, 255, 255, 72, 0, 0],
            &[0, 34, 255, 255, 169, 0, 0, 0, 76, 255, 255, 143, 0, 0],
            &[0, 80, 255, 255, 120, 0, 0, 0, 21, 255, 255, 184, 0, 0],
            &[0, 98, 255, 255, 100, 0, 0, 0, 3, 254, 255, 203, 0, 0],
            &[0, 105, 255, 255, 96, 0, 0, 0, 0, 250, 255, 208, 0, 0],
            &[0, 92, 255, 255, 109, 0, 0, 0, 7, 255, 255, 195, 0, 0],
            &[0, 59, 255, 255, 145, 0, 0, 0, 45, 255, 255, 161, 0, 0],
            &[0, 11, 246, 255, 221, 4, 0, 0, 126, 255, 255, 104, 0, 0],
            &[0, 0, 163, 255, 255, 166, 60, 102, 246, 255, 244, 21, 0, 0],
            &[0, 0, 30, 233, 255, 255, 255, 255, 255, 255, 109, 0, 0, 0],
            &[0, 0, 0, 38, 184, 253, 255, 255, 224, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 41, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ō' / 0x14d
        'ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 33, 178, 178, 178, 178, 178, 101, 0, 0, 0, 0],
            &[0, 0, 0, 48, 255, 255, 255, 255, 255, 145, 0, 0, 0, 0],
            &[0, 0, 0, 8, 46, 46, 46, 46, 46, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 104, 135, 122, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 109, 250, 255, 255, 255, 255, 197, 24, 0, 0, 0],
            &[0, 0, 80, 255, 255, 244, 185, 213, 255, 255, 192, 1, 0, 0],
            &[0, 0, 209, 255, 252, 50, 0, 1, 181, 255, 255, 61, 0, 0],
            &[0, 25, 255, 255, 186, 0, 0, 0, 84, 255, 255, 129, 0, 0],
            &[0, 52, 255, 255, 150, 0, 0, 0, 50, 255, 255, 155, 0, 0],
            &[0, 49, 255, 255, 154, 0, 0, 0, 62, 255, 255, 151, 0, 0],
            &[0, 13, 252, 255, 211, 0, 0, 0, 108, 255, 255, 110, 0, 0],
            &[0, 0, 179, 255, 255, 131, 20, 55, 228, 255, 251, 29, 0, 0],
            &[0, 0, 38, 240, 255, 255, 255, 255, 255, 255, 125, 0, 0, 0],
            &[0, 0, 0, 42, 186, 253, 255, 255, 227, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 42, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => Some(&[
            &[0, 0, 0, 85, 205, 45, 0, 0, 165, 166, 0, 0, 0, 0],
            &[0, 0, 0, 49, 254, 224, 143, 186, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 118, 250, 255, 255, 187, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 63, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 77, 163, 194, 183, 124, 19, 0, 0, 0, 0],
            &[0, 0, 2, 164, 255, 255, 255, 255, 255, 232, 42, 0, 0, 0],
            &[0, 0, 105, 255, 255, 236, 162, 208, 255, 255, 210, 2, 0, 0],
            &[0, 0, 223, 255, 245, 35, 0, 2, 185, 255, 255, 72, 0, 0],
            &[0, 34, 255, 255, 169, 0, 0, 0, 76, 255, 255, 143, 0, 0],
            &[0, 80, 255, 255, 120, 0, 0, 0, 21, 255, 255, 184, 0, 0],
            &[0, 98, 255, 255, 100, 0, 0, 0, 3, 254, 255, 203, 0, 0],
            &[0, 105, 255, 255, 96, 0, 0, 0, 0, 250, 255, 208, 0, 0],
            &[0, 92, 255, 255, 109, 0, 0, 0, 7, 255, 255, 195, 0, 0],
            &[0, 59, 255, 255, 145, 0, 0, 0, 45, 255, 255, 161, 0, 0],
            &[0, 11, 246, 255, 221, 4, 0, 0, 126, 255, 255, 104, 0, 0],
            &[0, 0, 163, 255, 255, 166, 60, 102, 246, 255, 244, 21, 0, 0],
            &[0, 0, 30, 233, 255, 255, 255, 255, 255, 255, 109, 0, 0, 0],
            &[0, 0, 0, 38, 184, 253, 255, 255, 224, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 41, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŏ' / 0x14f
        'ŏ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 156, 29, 0, 0, 120, 129, 0, 0, 0, 0],
            &[0, 0, 0, 66, 255, 191, 94, 137, 252, 163, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 255, 255, 225, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 66, 112, 88, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 104, 135, 122, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 109, 250, 255, 255, 255, 255, 197, 24, 0, 0, 0],
            &[0, 0, 80, 255, 255, 244, 185, 213, 255, 255, 192, 1, 0, 0],
            &[0, 0, 209, 255, 252, 50, 0, 1, 181, 255, 255, 61, 0, 0],
            &[0, 25, 255, 255, 186, 0, 0, 0, 84, 255, 255, 129, 0, 0],
            &[0, 52, 255, 255, 150, 0, 0, 0, 50, 255, 255, 155, 0, 0],
            &[0, 49, 255, 255, 154, 0, 0, 0, 62, 255, 255, 151, 0, 0],
            &[0, 13, 252, 255, 211, 0, 0, 0, 108, 255, 255, 110, 0, 0],
            &[0, 0, 179, 255, 255, 131, 20, 55, 228, 255, 251, 29, 0, 0],
            &[0, 0, 38, 240, 255, 255, 255, 255, 255, 255, 125, 0, 0, 0],
            &[0, 0, 0, 42, 186, 253, 255, 255, 227, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 42, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ő' / 0x150
        'Ő' => Some(&[
            &[0, 0, 0, 0, 54, 190, 190, 57, 166, 190, 132, 0, 0, 0],
            &[0, 0, 0, 1, 198, 255, 151, 93, 255, 224, 29, 0, 0, 0],
            &[0, 0, 0, 94, 255, 144, 16, 231, 218, 32, 0, 0, 0, 0],
            &[0, 0, 0, 52, 61, 0, 22, 75, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 77, 163, 194, 183, 124, 19, 0, 0, 0, 0],
            &[0, 0, 2, 164, 255, 255, 255, 255, 255, 232, 42, 0, 0, 0],
            &[0, 0, 105, 255, 255, 236, 162, 208, 255, 255, 210, 2, 0, 0],
            &[0, 0, 223, 255, 245, 35, 0, 2, 185, 255, 255, 72, 0, 0],
            &[0, 34, 255, 255, 169, 0, 0, 0, 76, 255, 255, 143, 0, 0],
            &[0, 80, 255, 255, 120, 0, 0, 0, 21, 255, 255, 184, 0, 0],
            &[0, 98, 255, 255, 100, 0, 0, 0, 3, 254, 255, 203, 0, 0],
            &[0, 105, 255, 255, 96, 0, 0, 0, 0, 250, 255, 208, 0, 0],
            &[0, 92, 255, 255, 109, 0, 0, 0, 7, 255, 255, 195, 0, 0],
            &[0, 59, 255, 255, 145, 0, 0, 0, 45, 255, 255, 161, 0, 0],
            &[0, 11, 246, 255, 221, 4, 0, 0, 126, 255, 255, 104, 0, 0],
            &[0, 0, 163, 255, 255, 166, 60, 102, 246, 255, 244, 21, 0, 0],
            &[0, 0, 30, 233, 255, 255, 255, 255, 255, 255, 109, 0, 0, 0],
            &[0, 0, 0, 38, 184, 253, 255, 255, 224, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 41, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ő' / 0x151
        'ő' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 141, 141, 51, 117, 141, 107, 0, 0, 0],
            &[0, 0, 0, 0, 170, 255, 190, 70, 255, 244, 53, 0, 0, 0],
            &[0, 0, 0, 65, 255, 188, 16, 211, 241, 61, 0, 0, 0, 0],
            &[0, 0, 0, 82, 110, 5, 32, 124, 45, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 104, 135, 122, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 109, 250, 255, 255, 255, 255, 197, 24, 0, 0, 0],
            &[0, 0, 80, 255, 255, 244, 185, 213, 255, 255, 192, 1, 0, 0],
            &[0, 0, 209, 255, 252, 50, 0, 1, 181, 255, 255, 61, 0, 0],
            &[0, 25, 255, 255, 186, 0, 0, 0, 84, 255, 255, 129, 0, 0],
            &[0, 52, 255, 255, 150, 0, 0, 0, 50, 255, 255, 155, 0, 0],
            &[0, 49, 255, 255, 154, 0, 0, 0, 62, 255, 255, 151, 0, 0],
            &[0, 13, 252, 255, 211, 0, 0, 0, 108, 255, 255, 110, 0, 0],
            &[0, 0, 179, 255, 255, 131, 20, 55, 228, 255, 251, 29, 0, 0],
            &[0, 0, 38, 240, 255, 255, 255, 255, 255, 255, 125, 0, 0, 0],
            &[0, 0, 0, 42, 186, 253, 255, 255, 227, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 42, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Œ' / 0x152
        'Œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 130, 182, 180, 146, 144, 144, 144, 144, 2, 0],
            &[0, 0, 20, 223, 255, 255, 255, 255, 255, 255, 255, 255, 5, 0],
            &[0, 0, 152, 255, 255, 219, 189, 255, 255, 224, 202, 202, 4, 0],
            &[0, 5, 243, 255, 224, 11, 42, 255, 255, 106, 0, 0, 0, 0],
            &[0, 52, 255, 255, 140, 0, 42, 255, 255, 106, 0, 0, 0, 0],
            &[0, 87, 255, 255, 94, 0, 42, 255, 255, 140, 58, 39, 0, 0],
            &[0, 102, 255, 255, 78, 0, 42, 255, 255, 255, 255, 173, 0, 0],
            &[0, 105, 255, 255, 71, 0, 42, 255, 255, 255, 255, 173, 0, 0],
            &[0, 92, 255, 255, 79, 0, 42, 255, 255, 125, 33, 22, 0, 0],
            &[0, 68, 255, 255, 103, 0, 42, 255, 255, 106, 0, 0, 0, 0],
            &[0, 22, 255, 255, 159, 0, 42, 255, 255, 106, 0, 0, 0, 0],
            &[0, 0, 199, 255, 248, 98, 100, 255, 255, 159, 91, 91, 1, 0],
            &[0, 0, 76, 254, 255, 255, 255, 255, 255, 255, 255, 255, 5, 0],
            &[0, 0, 0, 101, 239, 255, 255, 255, 255, 255, 255, 255, 5, 0],
            &[0, 0, 0, 0, 3, 35, 32, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'œ' / 0x153
        'œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 75, 131, 98, 6, 2, 90, 122, 50, 0, 0, 0],
            &[0, 0, 135, 255, 255, 255, 183, 165, 255, 255, 253, 72, 0, 0],
            &[
                0, 33, 252, 255, 142, 230, 255, 255, 194, 134, 255, 205, 0, 0,
            ],
            &[0, 108, 255, 204, 0, 113, 255, 255, 63, 2, 238, 254, 17, 0],
            &[0, 149, 255, 164, 0, 63, 255, 255, 80, 59, 226, 255, 43, 0],
            &[0, 164, 255, 153, 0, 45, 255, 255, 255, 255, 255, 255, 53, 0],
            &[0, 156, 255, 156, 0, 52, 255, 255, 129, 117, 117, 117, 24, 0],
            &[0, 131, 255, 175, 0, 73, 255, 255, 40, 0, 0, 0, 0, 0],
            &[0, 75, 255, 229, 6, 149, 255, 255, 139, 0, 0, 51, 0, 0],
            &[0, 4, 224, 255, 228, 255, 251, 243, 254, 201, 211, 241, 0, 0],
            &[0, 0, 54, 227, 255, 247, 101, 80, 243, 255, 253, 168, 0, 0],
            &[0, 0, 0, 5, 37, 14, 0, 0, 11, 40, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŕ' / 0x154
        'Ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 7, 175, 190, 180, 21, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 132, 255, 244, 67, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 252, 234, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 45, 75, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 69, 144, 144, 144, 137, 112, 47, 0, 0, 0, 0, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 173, 8, 0, 0, 0, 0],
            &[0, 122, 255, 255, 213, 209, 250, 255, 255, 147, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 31, 237, 255, 242, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 0, 187, 255, 255, 16, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 10, 231, 255, 245, 3, 0, 0, 0],
            &[0, 122, 255, 255, 153, 133, 211, 255, 255, 147, 0, 0, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 191, 13, 0, 0, 0, 0],
            &[0, 122, 255, 255, 235, 244, 255, 242, 12, 0, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 77, 255, 255, 129, 0, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 191, 255, 246, 31, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 51, 254, 255, 170, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 0, 164, 255, 255, 63, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 0, 30, 247, 255, 208, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŕ' / 0x155
        'ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 46, 141, 141, 101, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 205, 255, 221, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 130, 255, 209, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 121, 119, 13, 0, 0, 0, 0, 0],
            &[0, 0, 67, 95, 95, 95, 58, 0, 30, 117, 138, 101, 2, 0],
            &[0, 0, 181, 255, 255, 255, 196, 64, 244, 255, 255, 220, 0, 0],
            &[0, 0, 115, 194, 240, 255, 248, 233, 255, 255, 255, 146, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 251, 114, 30, 37, 28, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 144, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 72, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 161, 255, 255, 98, 8, 0, 0, 0, 0, 0],
            &[0, 41, 220, 255, 255, 255, 255, 255, 250, 196, 0, 0, 0, 0],
            &[0, 56, 255, 255, 255, 255, 255, 255, 255, 242, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŗ' / 0x156
        'Ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 69, 144, 144, 144, 137, 112, 47, 0, 0, 0, 0, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 173, 8, 0, 0, 0, 0],
            &[0, 122, 255, 255, 213, 209, 250, 255, 255, 147, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 31, 237, 255, 242, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 0, 187, 255, 255, 16, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 10, 231, 255, 245, 3, 0, 0, 0],
            &[0, 122, 255, 255, 153, 133, 211, 255, 255, 147, 0, 0, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 191, 13, 0, 0, 0, 0],
            &[0, 122, 255, 255, 235, 244, 255, 242, 12, 0, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 77, 255, 255, 129, 0, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 191, 255, 246, 31, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 51, 254, 255, 170, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 0, 164, 255, 255, 63, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 0, 30, 247, 255, 208, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 215, 215, 62, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 89, 255, 214, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 47, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŗ' / 0x157
        'ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 67, 95, 95, 95, 58, 0, 30, 117, 138, 101, 2, 0],
            &[0, 0, 181, 255, 255, 255, 196, 64, 244, 255, 255, 220, 0, 0],
            &[0, 0, 115, 194, 240, 255, 248, 233, 255, 255, 255, 146, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 251, 114, 30, 37, 28, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 144, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 72, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 161, 255, 255, 98, 8, 0, 0, 0, 0, 0],
            &[0, 41, 220, 255, 255, 255, 255, 255, 250, 196, 0, 0, 0, 0],
            &[0, 56, 255, 255, 255, 255, 255, 255, 255, 242, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 89, 215, 210, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 255, 157, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 35, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ř' / 0x158
        'Ř' => Some(&[
            &[0, 0, 121, 190, 73, 0, 22, 167, 182, 16, 0, 0, 0, 0],
            &[0, 0, 18, 223, 252, 113, 219, 255, 92, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 252, 255, 255, 158, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 58, 75, 75, 13, 0, 0, 0, 0, 0, 0],
            &[0, 69, 144, 144, 144, 137, 112, 47, 0, 0, 0, 0, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 173, 8, 0, 0, 0, 0],
            &[0, 122, 255, 255, 213, 209, 250, 255, 255, 147, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 31, 237, 255, 242, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 0, 187, 255, 255, 16, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 10, 231, 255, 245, 3, 0, 0, 0],
            &[0, 122, 255, 255, 153, 133, 211, 255, 255, 147, 0, 0, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 191, 13, 0, 0, 0, 0],
            &[0, 122, 255, 255, 235, 244, 255, 242, 12, 0, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 77, 255, 255, 129, 0, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 191, 255, 246, 31, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 51, 254, 255, 170, 0, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 0, 164, 255, 255, 63, 0, 0, 0],
            &[0, 122, 255, 255, 70, 0, 0, 30, 247, 255, 208, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ř' / 0x159
        'ř' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 141, 114, 3, 0, 43, 141, 95, 0, 0, 0],
            &[0, 0, 0, 0, 135, 255, 176, 71, 243, 240, 37, 0, 0, 0],
            &[0, 0, 0, 0, 3, 194, 255, 255, 255, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 124, 124, 101, 0, 0, 0, 0, 0],
            &[0, 0, 67, 95, 95, 95, 58, 0, 30, 117, 138, 101, 2, 0],
            &[0, 0, 181, 255, 255, 255, 196, 64, 244, 255, 255, 220, 0, 0],
            &[0, 0, 115, 194, 240, 255, 248, 233, 255, 255, 255, 146, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 251, 114, 30, 37, 28, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 144, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 72, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 255, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 161, 255, 255, 98, 8, 0, 0, 0, 0, 0],
            &[0, 41, 220, 255, 255, 255, 255, 255, 250, 196, 0, 0, 0, 0],
            &[0, 56, 255, 255, 255, 255, 255, 255, 255, 242, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ś' / 0x15a
        'Ś' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 132, 190, 190, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 65, 254, 255, 123, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 226, 252, 105, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 75, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 58, 145, 182, 182, 157, 112, 33, 0, 0, 0],
            &[0, 0, 0, 129, 255, 255, 255, 255, 255, 255, 234, 4, 0, 0],
            &[0, 0, 37, 254, 255, 252, 167, 145, 182, 247, 139, 0, 0, 0],
            &[0, 0, 91, 255, 255, 138, 0, 0, 0, 13, 17, 0, 0, 0],
            &[0, 0, 84, 255, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 244, 255, 253, 132, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 87, 251, 255, 255, 235, 109, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 208, 255, 255, 255, 214, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 97, 226, 255, 255, 224, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 205, 255, 255, 68, 0, 0],
            &[0, 0, 5, 0, 0, 0, 0, 0, 138, 255, 255, 82, 0, 0],
            &[0, 0, 163, 163, 85, 39, 31, 77, 228, 255, 255, 35, 0, 0],
            &[0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 87, 205, 253, 255, 255, 255, 223, 114, 2, 0, 0, 0],
            &[0, 0, 0, 0, 6, 30, 43, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ś' / 0x15b
        'ś' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 113, 141, 141, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 75, 255, 255, 129, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 231, 254, 114, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 58, 124, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 112, 138, 127, 102, 52, 1, 0, 0, 0],
            &[0, 0, 0, 146, 255, 255, 255, 255, 255, 255, 169, 0, 0, 0],
            &[0, 0, 55, 255, 255, 243, 190, 199, 235, 255, 90, 0, 0, 0],
            &[0, 0, 95, 255, 255, 140, 0, 0, 1, 54, 7, 0, 0, 0],
            &[0, 0, 50, 255, 255, 252, 168, 71, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 248, 255, 255, 255, 218, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 135, 232, 255, 255, 255, 101, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 111, 255, 255, 204, 0, 0, 0],
            &[0, 0, 110, 167, 89, 45, 34, 90, 254, 255, 211, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 136, 0, 0, 0],
            &[0, 0, 79, 214, 255, 255, 255, 255, 234, 132, 4, 0, 0, 0],
            &[0, 0, 0, 0, 15, 36, 43, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => Some(&[
            &[0, 0, 0, 0, 0, 126, 190, 190, 117, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 69, 253, 241, 246, 252, 61, 0, 0, 0, 0],
            &[0, 0, 0, 33, 235, 234, 54, 65, 239, 230, 25, 0, 0, 0],
            &[0, 0, 0, 38, 75, 27, 0, 0, 31, 75, 33, 0, 0, 0],
            &[0, 0, 0, 0, 58, 145, 182, 182, 157, 112, 33, 0, 0, 0],
            &[0, 0, 0, 129, 255, 255, 255, 255, 255, 255, 234, 4, 0, 0],
            &[0, 0, 37, 254, 255, 252, 167, 145, 182, 247, 139, 0, 0, 0],
            &[0, 0, 91, 255, 255, 138, 0, 0, 0, 13, 17, 0, 0, 0],
            &[0, 0, 84, 255, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 244, 255, 253, 132, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 87, 251, 255, 255, 235, 109, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 208, 255, 255, 255, 214, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 97, 226, 255, 255, 224, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 205, 255, 255, 68, 0, 0],
            &[0, 0, 5, 0, 0, 0, 0, 0, 138, 255, 255, 82, 0, 0],
            &[0, 0, 163, 163, 85, 39, 31, 77, 228, 255, 255, 35, 0, 0],
            &[0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 87, 205, 253, 255, 255, 255, 223, 114, 2, 0, 0, 0],
            &[0, 0, 0, 0, 6, 30, 43, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŝ' / 0x15d
        'ŝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 109, 141, 141, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 75, 255, 252, 255, 219, 13, 0, 0, 0, 0],
            &[0, 0, 0, 34, 237, 240, 59, 146, 255, 172, 0, 0, 0, 0],
            &[0, 0, 0, 78, 124, 38, 0, 0, 88, 124, 27, 0, 0, 0],
            &[0, 0, 0, 0, 44, 112, 138, 127, 102, 52, 1, 0, 0, 0],
            &[0, 0, 0, 146, 255, 255, 255, 255, 255, 255, 169, 0, 0, 0],
            &[0, 0, 55, 255, 255, 243, 190, 199, 235, 255, 90, 0, 0, 0],
            &[0, 0, 95, 255, 255, 140, 0, 0, 1, 54, 7, 0, 0, 0],
            &[0, 0, 50, 255, 255, 252, 168, 71, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 248, 255, 255, 255, 218, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 135, 232, 255, 255, 255, 101, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 111, 255, 255, 204, 0, 0, 0],
            &[0, 0, 110, 167, 89, 45, 34, 90, 254, 255, 211, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 136, 0, 0, 0],
            &[0, 0, 79, 214, 255, 255, 255, 255, 234, 132, 4, 0, 0, 0],
            &[0, 0, 0, 0, 15, 36, 43, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ş' / 0x15e
        'Ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 58, 145, 182, 182, 157, 112, 33, 0, 0, 0],
            &[0, 0, 0, 129, 255, 255, 255, 255, 255, 255, 234, 4, 0, 0],
            &[0, 0, 37, 254, 255, 252, 167, 145, 182, 247, 139, 0, 0, 0],
            &[0, 0, 91, 255, 255, 138, 0, 0, 0, 13, 17, 0, 0, 0],
            &[0, 0, 84, 255, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 244, 255, 253, 132, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 87, 251, 255, 255, 235, 109, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 208, 255, 255, 255, 214, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 97, 226, 255, 255, 224, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 205, 255, 255, 68, 0, 0],
            &[0, 0, 5, 0, 0, 0, 0, 0, 138, 255, 255, 82, 0, 0],
            &[0, 0, 163, 163, 85, 39, 31, 77, 228, 255, 255, 35, 0, 0],
            &[0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 87, 205, 253, 255, 255, 255, 223, 114, 2, 0, 0, 0],
            &[0, 0, 0, 0, 6, 86, 255, 124, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 83, 221, 255, 101, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 91, 255, 181, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 130, 132, 80, 1, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ş' / 0x15f
        'ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 112, 138, 127, 102, 52, 1, 0, 0, 0],
            &[0, 0, 0, 146, 255, 255, 255, 255, 255, 255, 169, 0, 0, 0],
            &[0, 0, 55, 255, 255, 243, 190, 199, 235, 255, 90, 0, 0, 0],
            &[0, 0, 95, 255, 255, 140, 0, 0, 1, 54, 7, 0, 0, 0],
            &[0, 0, 50, 255, 255, 252, 168, 71, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 248, 255, 255, 255, 218, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 135, 232, 255, 255, 255, 101, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 111, 255, 255, 204, 0, 0, 0],
            &[0, 0, 110, 167, 89, 45, 34, 90, 254, 255, 211, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 136, 0, 0, 0],
            &[0, 0, 79, 214, 255, 255, 255, 255, 234, 132, 4, 0, 0, 0],
            &[0, 0, 0, 0, 15, 70, 255, 152, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 67, 212, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 67, 255, 205, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 127, 134, 89, 4, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Š' / 0x160
        'Š' => Some(&[
            &[0, 0, 0, 70, 190, 121, 1, 4, 134, 190, 58, 0, 0, 0],
            &[0, 0, 0, 1, 172, 255, 154, 175, 255, 160, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 226, 255, 255, 216, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 75, 75, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 58, 145, 182, 182, 157, 112, 33, 0, 0, 0],
            &[0, 0, 0, 129, 255, 255, 255, 255, 255, 255, 234, 4, 0, 0],
            &[0, 0, 37, 254, 255, 252, 167, 145, 182, 247, 139, 0, 0, 0],
            &[0, 0, 91, 255, 255, 138, 0, 0, 0, 13, 17, 0, 0, 0],
            &[0, 0, 84, 255, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 244, 255, 253, 132, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 87, 251, 255, 255, 235, 109, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 208, 255, 255, 255, 214, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 97, 226, 255, 255, 224, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 205, 255, 255, 68, 0, 0],
            &[0, 0, 5, 0, 0, 0, 0, 0, 138, 255, 255, 82, 0, 0],
            &[0, 0, 163, 163, 85, 39, 31, 77, 228, 255, 255, 35, 0, 0],
            &[0, 0, 171, 255, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 87, 205, 253, 255, 255, 255, 223, 114, 2, 0, 0, 0],
            &[0, 0, 0, 0, 6, 30, 43, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'š' / 0x161
        'š' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 86, 141, 50, 0, 1, 108, 141, 28, 0, 0, 0],
            &[0, 0, 0, 26, 230, 246, 75, 169, 255, 155, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 254, 255, 255, 209, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 124, 124, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 112, 138, 127, 102, 52, 1, 0, 0, 0],
            &[0, 0, 0, 146, 255, 255, 255, 255, 255, 255, 169, 0, 0, 0],
            &[0, 0, 55, 255, 255, 243, 190, 199, 235, 255, 90, 0, 0, 0],
            &[0, 0, 95, 255, 255, 140, 0, 0, 1, 54, 7, 0, 0, 0],
            &[0, 0, 50, 255, 255, 252, 168, 71, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 248, 255, 255, 255, 218, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 135, 232, 255, 255, 255, 101, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 111, 255, 255, 204, 0, 0, 0],
            &[0, 0, 110, 167, 89, 45, 34, 90, 254, 255, 211, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 136, 0, 0, 0],
            &[0, 0, 79, 214, 255, 255, 255, 255, 234, 132, 4, 0, 0, 0],
            &[0, 0, 0, 0, 15, 36, 43, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ţ' / 0x162
        'Ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 62, 144, 144, 144, 144, 144, 144, 144, 144, 144, 120, 0, 0,
            ],
            &[
                0, 110, 255, 255, 255, 255, 255, 255, 255, 255, 255, 212, 0, 0,
            ],
            &[
                0, 89, 207, 207, 207, 240, 255, 255, 210, 207, 207, 171, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 255, 101, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 83, 221, 255, 101, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 91, 255, 181, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 130, 132, 80, 1, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ţ' / 0x163
        'ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 243, 245, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 53, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 142, 255, 255, 117, 95, 95, 95, 3, 0, 0],
            &[0, 0, 231, 255, 255, 255, 255, 255, 255, 255, 255, 8, 0, 0],
            &[0, 0, 230, 231, 246, 255, 255, 235, 231, 231, 231, 7, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 255, 255, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 255, 255, 177, 45, 39, 73, 12, 0, 0],
            &[0, 0, 0, 0, 44, 251, 255, 255, 255, 255, 255, 32, 0, 0],
            &[0, 0, 0, 0, 0, 83, 225, 255, 255, 255, 234, 25, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 255, 207, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 41, 197, 255, 166, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 27, 255, 246, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 119, 136, 102, 11, 0, 0, 0, 0],
        ]),
        // letter: 'Ť' / 0x164
        'Ť' => Some(&[
            &[0, 0, 0, 121, 190, 73, 0, 22, 167, 182, 16, 0, 0, 0],
            &[0, 0, 0, 18, 223, 252, 113, 219, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 252, 255, 255, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 58, 75, 75, 13, 0, 0, 0, 0, 0],
            &[
                0, 62, 144, 144, 144, 144, 144, 144, 144, 144, 144, 120, 0, 0,
            ],
            &[
                0, 110, 255, 255, 255, 255, 255, 255, 255, 255, 255, 212, 0, 0,
            ],
            &[
                0, 89, 207, 207, 207, 240, 255, 255, 210, 207, 207, 171, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ť' / 0x165
        'ť' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 114, 166, 134, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 213, 255, 109, 0, 0],
            &[0, 0, 0, 0, 10, 243, 245, 33, 15, 253, 222, 7, 0, 0],
            &[0, 0, 0, 0, 53, 255, 255, 35, 20, 100, 48, 0, 0, 0],
            &[0, 0, 0, 34, 142, 255, 255, 117, 95, 95, 95, 3, 0, 0],
            &[0, 0, 231, 255, 255, 255, 255, 255, 255, 255, 255, 8, 0, 0],
            &[0, 0, 230, 231, 246, 255, 255, 235, 231, 231, 231, 7, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 255, 255, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 255, 255, 177, 45, 39, 73, 12, 0, 0],
            &[0, 0, 0, 0, 44, 251, 255, 255, 255, 255, 255, 32, 0, 0],
            &[0, 0, 0, 0, 0, 83, 225, 255, 255, 255, 234, 25, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 28, 43, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŧ' / 0x166
        'Ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 62, 144, 144, 144, 144, 144, 144, 144, 144, 144, 120, 0, 0,
            ],
            &[
                0, 110, 255, 255, 255, 255, 255, 255, 255, 255, 255, 212, 0, 0,
            ],
            &[
                0, 89, 207, 207, 207, 240, 255, 255, 210, 207, 207, 171, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 5, 216, 216, 255, 255, 255, 234, 216, 91, 0, 0, 0],
            &[0, 0, 6, 255, 255, 255, 255, 255, 255, 255, 108, 0, 0, 0],
            &[0, 0, 2, 105, 105, 255, 255, 255, 123, 105, 44, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 255, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŧ' / 0x167
        'ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 243, 245, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 53, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 142, 255, 255, 117, 95, 95, 95, 3, 0, 0],
            &[0, 0, 231, 255, 255, 255, 255, 255, 255, 255, 255, 8, 0, 0],
            &[0, 0, 230, 231, 246, 255, 255, 235, 231, 231, 231, 7, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 122, 255, 255, 255, 157, 122, 122, 122, 15, 0, 0],
            &[0, 0, 254, 255, 255, 255, 255, 255, 255, 255, 255, 32, 0, 0],
            &[0, 0, 78, 78, 236, 255, 255, 113, 78, 78, 78, 9, 0, 0],
            &[0, 0, 0, 0, 154, 255, 255, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 255, 255, 177, 45, 39, 73, 12, 0, 0],
            &[0, 0, 0, 0, 44, 251, 255, 255, 255, 255, 255, 32, 0, 0],
            &[0, 0, 0, 0, 0, 83, 225, 255, 255, 255, 234, 25, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 28, 43, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ũ' / 0x168
        'Ũ' => Some(&[
            &[0, 0, 0, 0, 85, 121, 42, 0, 22, 142, 37, 0, 0, 0],
            &[0, 0, 0, 101, 255, 255, 254, 188, 196, 255, 31, 0, 0, 0],
            &[0, 0, 0, 197, 210, 102, 190, 255, 255, 160, 0, 0, 0, 0],
            &[0, 0, 0, 70, 40, 0, 0, 36, 56, 1, 0, 0, 0, 0],
            &[0, 21, 144, 144, 88, 0, 0, 0, 30, 144, 144, 78, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 32, 255, 255, 165, 0, 0, 0, 73, 255, 255, 134, 0, 0],
            &[0, 9, 254, 255, 213, 0, 0, 0, 112, 255, 255, 108, 0, 0],
            &[0, 0, 196, 255, 255, 148, 65, 98, 232, 255, 255, 38, 0, 0],
            &[0, 0, 63, 253, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 67, 204, 255, 255, 255, 231, 123, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 41, 27, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ũ' / 0x169
        'ũ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 72, 11, 0, 11, 93, 26, 0, 0, 0],
            &[0, 0, 0, 72, 252, 255, 237, 139, 158, 255, 42, 0, 0, 0],
            &[0, 0, 0, 185, 230, 152, 232, 255, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 112, 70, 0, 7, 85, 105, 14, 0, 0, 0, 0],
            &[0, 4, 95, 95, 67, 0, 0, 0, 38, 95, 95, 33, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 105, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 181, 0, 0, 0, 127, 255, 255, 90, 0, 0],
            &[0, 7, 254, 255, 205, 0, 0, 0, 170, 255, 255, 90, 0, 0],
            &[0, 0, 228, 255, 254, 94, 32, 100, 252, 255, 255, 90, 0, 0],
            &[0, 0, 138, 255, 255, 255, 255, 255, 224, 255, 255, 90, 0, 0],
            &[0, 0, 11, 160, 253, 255, 253, 168, 17, 229, 255, 90, 0, 0],
            &[0, 0, 0, 0, 10, 39, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ū' / 0x16a
        'Ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 228, 228, 228, 228, 228, 129, 0, 0, 0, 0],
            &[0, 0, 0, 47, 253, 253, 253, 253, 253, 144, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 88, 0, 0, 0, 30, 144, 144, 78, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 32, 255, 255, 165, 0, 0, 0, 73, 255, 255, 134, 0, 0],
            &[0, 9, 254, 255, 213, 0, 0, 0, 112, 255, 255, 108, 0, 0],
            &[0, 0, 196, 255, 255, 148, 65, 98, 232, 255, 255, 38, 0, 0],
            &[0, 0, 63, 253, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 67, 204, 255, 255, 255, 231, 123, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 41, 27, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ū' / 0x16b
        'ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 33, 178, 178, 178, 178, 178, 101, 0, 0, 0, 0],
            &[0, 0, 0, 48, 255, 255, 255, 255, 255, 145, 0, 0, 0, 0],
            &[0, 0, 0, 8, 46, 46, 46, 46, 46, 26, 0, 0, 0, 0],
            &[0, 4, 95, 95, 67, 0, 0, 0, 38, 95, 95, 33, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 105, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 181, 0, 0, 0, 127, 255, 255, 90, 0, 0],
            &[0, 7, 254, 255, 205, 0, 0, 0, 170, 255, 255, 90, 0, 0],
            &[0, 0, 228, 255, 254, 94, 32, 100, 252, 255, 255, 90, 0, 0],
            &[0, 0, 138, 255, 255, 255, 255, 255, 224, 255, 255, 90, 0, 0],
            &[0, 0, 11, 160, 253, 255, 253, 168, 17, 229, 255, 90, 0, 0],
            &[0, 0, 0, 0, 10, 39, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => Some(&[
            &[0, 0, 0, 85, 205, 45, 0, 0, 165, 166, 0, 0, 0, 0],
            &[0, 0, 0, 49, 254, 224, 143, 186, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 118, 250, 255, 255, 187, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 63, 39, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 88, 0, 0, 0, 30, 144, 144, 78, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 32, 255, 255, 165, 0, 0, 0, 73, 255, 255, 134, 0, 0],
            &[0, 9, 254, 255, 213, 0, 0, 0, 112, 255, 255, 108, 0, 0],
            &[0, 0, 196, 255, 255, 148, 65, 98, 232, 255, 255, 38, 0, 0],
            &[0, 0, 63, 253, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 67, 204, 255, 255, 255, 231, 123, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 41, 27, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŭ' / 0x16d
        'ŭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 156, 29, 0, 0, 120, 129, 0, 0, 0, 0],
            &[0, 0, 0, 66, 255, 191, 94, 137, 252, 163, 0, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 255, 255, 225, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 66, 112, 88, 11, 0, 0, 0, 0, 0],
            &[0, 4, 95, 95, 67, 0, 0, 0, 38, 95, 95, 33, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 105, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 181, 0, 0, 0, 127, 255, 255, 90, 0, 0],
            &[0, 7, 254, 255, 205, 0, 0, 0, 170, 255, 255, 90, 0, 0],
            &[0, 0, 228, 255, 254, 94, 32, 100, 252, 255, 255, 90, 0, 0],
            &[0, 0, 138, 255, 255, 255, 255, 255, 224, 255, 255, 90, 0, 0],
            &[0, 0, 11, 160, 253, 255, 253, 168, 17, 229, 255, 90, 0, 0],
            &[0, 0, 0, 0, 10, 39, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ů' / 0x16e
        'Ů' => Some(&[
            &[0, 0, 0, 0, 119, 246, 87, 195, 214, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 234, 30, 166, 226, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 233, 255, 252, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 63, 32, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 88, 0, 0, 0, 30, 144, 144, 78, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 32, 255, 255, 165, 0, 0, 0, 73, 255, 255, 134, 0, 0],
            &[0, 9, 254, 255, 213, 0, 0, 0, 112, 255, 255, 108, 0, 0],
            &[0, 0, 196, 255, 255, 148, 65, 98, 232, 255, 255, 38, 0, 0],
            &[0, 0, 63, 253, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 67, 204, 255, 255, 255, 231, 123, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 41, 27, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 194, 255, 228, 65, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ů' / 0x16f
        'ů' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 144, 214, 180, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 103, 253, 136, 219, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 228, 1, 143, 232, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 251, 235, 254, 143, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 112, 78, 1, 0, 0, 0, 0, 0],
            &[0, 4, 95, 95, 67, 0, 0, 0, 38, 95, 95, 33, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 105, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 181, 0, 0, 0, 127, 255, 255, 90, 0, 0],
            &[0, 7, 254, 255, 205, 0, 0, 0, 170, 255, 255, 90, 0, 0],
            &[0, 0, 228, 255, 254, 94, 32, 100, 252, 255, 255, 90, 0, 0],
            &[0, 0, 138, 255, 255, 255, 255, 255, 224, 255, 255, 90, 0, 0],
            &[0, 0, 11, 160, 253, 255, 253, 168, 17, 229, 255, 90, 0, 0],
            &[0, 0, 0, 0, 10, 39, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ű' / 0x170
        'Ű' => Some(&[
            &[0, 0, 0, 0, 54, 190, 190, 57, 166, 190, 132, 0, 0, 0],
            &[0, 0, 0, 1, 198, 255, 151, 93, 255, 224, 29, 0, 0, 0],
            &[0, 0, 0, 94, 255, 144, 16, 231, 218, 32, 0, 0, 0, 0],
            &[0, 0, 0, 52, 61, 0, 22, 75, 19, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 88, 0, 0, 0, 30, 144, 144, 78, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 32, 255, 255, 165, 0, 0, 0, 73, 255, 255, 134, 0, 0],
            &[0, 9, 254, 255, 213, 0, 0, 0, 112, 255, 255, 108, 0, 0],
            &[0, 0, 196, 255, 255, 148, 65, 98, 232, 255, 255, 38, 0, 0],
            &[0, 0, 63, 253, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 67, 204, 255, 255, 255, 231, 123, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 41, 27, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ű' / 0x171
        'ű' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 141, 141, 51, 117, 141, 107, 0, 0, 0],
            &[0, 0, 0, 0, 170, 255, 190, 70, 255, 244, 53, 0, 0, 0],
            &[0, 0, 0, 65, 255, 188, 16, 211, 241, 61, 0, 0, 0, 0],
            &[0, 0, 0, 82, 110, 5, 32, 124, 45, 0, 0, 0, 0, 0],
            &[0, 4, 95, 95, 67, 0, 0, 0, 38, 95, 95, 33, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 105, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 181, 0, 0, 0, 127, 255, 255, 90, 0, 0],
            &[0, 7, 254, 255, 205, 0, 0, 0, 170, 255, 255, 90, 0, 0],
            &[0, 0, 228, 255, 254, 94, 32, 100, 252, 255, 255, 90, 0, 0],
            &[0, 0, 138, 255, 255, 255, 255, 255, 224, 255, 255, 90, 0, 0],
            &[0, 0, 11, 160, 253, 255, 253, 168, 17, 229, 255, 90, 0, 0],
            &[0, 0, 0, 0, 10, 39, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ų' / 0x172
        'Ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 88, 0, 0, 0, 30, 144, 144, 78, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 37, 255, 255, 156, 0, 0, 0, 53, 255, 255, 139, 0, 0],
            &[0, 32, 255, 255, 165, 0, 0, 0, 73, 255, 255, 134, 0, 0],
            &[0, 9, 254, 255, 213, 0, 0, 0, 112, 255, 255, 108, 0, 0],
            &[0, 0, 196, 255, 255, 148, 65, 98, 232, 255, 255, 38, 0, 0],
            &[0, 0, 63, 253, 255, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 67, 204, 255, 255, 255, 255, 135, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 36, 252, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 164, 255, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 224, 255, 87, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 89, 121, 59, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ų' / 0x173
        'ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 95, 95, 67, 0, 0, 0, 38, 95, 95, 33, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 102, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 105, 255, 255, 90, 0, 0],
            &[0, 12, 255, 255, 181, 0, 0, 0, 127, 255, 255, 90, 0, 0],
            &[0, 7, 254, 255, 205, 0, 0, 0, 170, 255, 255, 90, 0, 0],
            &[0, 0, 228, 255, 254, 94, 32, 100, 252, 255, 255, 90, 0, 0],
            &[0, 0, 138, 255, 255, 255, 255, 255, 224, 255, 255, 90, 0, 0],
            &[0, 0, 11, 160, 253, 255, 253, 168, 17, 255, 255, 90, 0, 0],
            &[0, 0, 0, 0, 10, 39, 12, 0, 63, 243, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 226, 236, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 28, 255, 235, 60, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 17, 106, 118, 35, 0, 0],
        ]),
        // letter: 'Ŵ' / 0x174
        'Ŵ' => Some(&[
            &[0, 0, 0, 0, 5, 171, 190, 190, 66, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 255, 232, 255, 226, 19, 0, 0, 0, 0],
            &[0, 0, 0, 82, 254, 201, 19, 119, 254, 184, 3, 0, 0, 0],
            &[0, 0, 0, 58, 74, 8, 0, 0, 51, 75, 13, 0, 0, 0],
            &[0, 114, 144, 88, 0, 0, 0, 0, 0, 29, 144, 144, 27, 0],
            &[0, 187, 255, 175, 0, 0, 0, 0, 0, 66, 255, 255, 34, 0],
            &[0, 168, 255, 199, 0, 0, 0, 0, 0, 84, 255, 255, 16, 0],
            &[0, 149, 255, 223, 0, 59, 95, 95, 5, 102, 255, 253, 1, 0],
            &[0, 130, 255, 243, 0, 186, 255, 255, 41, 121, 255, 236, 0, 0],
            &[0, 111, 255, 255, 6, 225, 255, 255, 79, 139, 255, 218, 0, 0],
            &[0, 92, 255, 255, 31, 254, 231, 255, 118, 156, 255, 200, 0, 0],
            &[0, 73, 255, 255, 79, 255, 162, 255, 156, 169, 255, 183, 0, 0],
            &[
                0, 54, 255, 255, 120, 255, 102, 246, 196, 173, 255, 165, 0, 0,
            ],
            &[0, 35, 255, 255, 157, 255, 62, 210, 238, 170, 255, 147, 0, 0],
            &[0, 16, 255, 255, 190, 255, 24, 173, 255, 185, 255, 129, 0, 0],
            &[0, 1, 251, 255, 226, 242, 0, 135, 255, 223, 255, 111, 0, 0],
            &[0, 0, 233, 255, 255, 204, 0, 97, 255, 255, 255, 93, 0, 0],
            &[0, 0, 214, 255, 255, 167, 0, 59, 255, 255, 255, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŵ' / 0x175
        'ŵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 117, 141, 141, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 89, 255, 252, 255, 210, 8, 0, 0, 0, 0],
            &[0, 0, 0, 43, 242, 235, 51, 159, 255, 158, 0, 0, 0, 0],
            &[0, 0, 0, 85, 124, 31, 0, 0, 94, 124, 20, 0, 0, 0],
            &[0, 87, 95, 55, 0, 0, 0, 0, 0, 11, 95, 95, 32, 0],
            &[0, 209, 255, 164, 0, 0, 0, 0, 0, 47, 255, 255, 61, 0],
            &[0, 173, 255, 187, 0, 118, 218, 211, 2, 70, 255, 255, 25, 0],
            &[0, 137, 255, 209, 0, 186, 255, 255, 42, 92, 255, 244, 0, 0],
            &[0, 101, 255, 232, 0, 237, 255, 255, 93, 115, 255, 208, 0, 0],
            &[0, 65, 255, 252, 35, 255, 203, 255, 144, 138, 255, 172, 0, 0],
            &[
                0, 29, 255, 255, 105, 255, 122, 254, 194, 159, 255, 136, 0, 0,
            ],
            &[0, 1, 247, 255, 171, 255, 66, 220, 239, 180, 255, 100, 0, 0],
            &[0, 0, 213, 255, 231, 255, 19, 175, 255, 230, 255, 64, 0, 0],
            &[0, 0, 176, 255, 255, 228, 0, 129, 255, 255, 255, 28, 0, 0],
            &[0, 0, 140, 255, 255, 181, 0, 83, 255, 255, 246, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŷ' / 0x176
        'Ŷ' => Some(&[
            &[0, 0, 0, 0, 5, 171, 190, 190, 66, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 255, 232, 255, 226, 19, 0, 0, 0, 0],
            &[0, 0, 0, 82, 254, 201, 19, 119, 254, 184, 3, 0, 0, 0],
            &[0, 0, 0, 58, 74, 8, 0, 0, 51, 75, 13, 0, 0, 0],
            &[0, 70, 144, 144, 58, 0, 0, 0, 3, 139, 144, 128, 0, 0],
            &[0, 33, 251, 255, 188, 0, 0, 0, 80, 255, 255, 131, 0, 0],
            &[0, 0, 162, 255, 255, 41, 0, 0, 187, 255, 245, 20, 0, 0],
            &[0, 0, 42, 254, 255, 148, 0, 40, 254, 255, 145, 0, 0, 0],
            &[0, 0, 0, 175, 255, 242, 14, 146, 255, 250, 29, 0, 0, 0],
            &[0, 0, 0, 53, 255, 255, 121, 241, 255, 158, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 255, 251, 255, 253, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 255, 255, 255, 172, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 203, 255, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŷ' / 0x177
        'ŷ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 141, 141, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 255, 252, 255, 203, 6, 0, 0, 0, 0],
            &[0, 0, 0, 49, 246, 231, 47, 167, 255, 148, 0, 0, 0, 0],
            &[0, 0, 0, 90, 123, 27, 0, 1, 98, 124, 16, 0, 0, 0],
            &[0, 52, 95, 95, 42, 0, 0, 0, 1, 94, 95, 90, 0, 0],
            &[0, 69, 255, 255, 172, 0, 0, 0, 51, 255, 255, 179, 0, 0],
            &[0, 1, 222, 255, 246, 11, 0, 0, 127, 255, 255, 86, 0, 0],
            &[0, 0, 122, 255, 255, 87, 0, 0, 203, 255, 240, 8, 0, 0],
            &[0, 0, 25, 251, 255, 173, 0, 24, 254, 255, 156, 0, 0, 0],
            &[0, 0, 0, 175, 255, 246, 12, 99, 255, 255, 63, 0, 0, 0],
            &[0, 0, 0, 74, 255, 255, 87, 174, 255, 224, 1, 0, 0, 0],
            &[0, 0, 0, 2, 225, 255, 162, 241, 255, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 255, 242, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 252, 255, 255, 203, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 180, 255, 255, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 192, 255, 249, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 255, 255, 164, 0, 0, 0, 0, 0, 0],
            &[0, 0, 177, 210, 253, 255, 248, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 162, 134, 41, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ÿ' / 0x178
        'Ÿ' => Some(&[
            &[0, 0, 0, 48, 124, 40, 0, 9, 111, 92, 0, 0, 0, 0],
            &[0, 0, 0, 212, 255, 199, 0, 102, 255, 255, 57, 0, 0, 0],
            &[0, 0, 0, 175, 255, 159, 0, 67, 251, 242, 30, 0, 0, 0],
            &[0, 0, 0, 0, 32, 0, 0, 0, 21, 11, 0, 0, 0, 0],
            &[0, 70, 144, 144, 58, 0, 0, 0, 3, 139, 144, 128, 0, 0],
            &[0, 33, 251, 255, 188, 0, 0, 0, 80, 255, 255, 131, 0, 0],
            &[0, 0, 162, 255, 255, 41, 0, 0, 187, 255, 245, 20, 0, 0],
            &[0, 0, 42, 254, 255, 148, 0, 40, 254, 255, 145, 0, 0, 0],
            &[0, 0, 0, 175, 255, 242, 14, 146, 255, 250, 29, 0, 0, 0],
            &[0, 0, 0, 53, 255, 255, 121, 241, 255, 158, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 255, 251, 255, 253, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 255, 255, 255, 172, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 203, 255, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ź' / 0x179
        'Ź' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 139, 190, 190, 55, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 74, 255, 255, 114, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 231, 250, 97, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 75, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 144, 144, 144, 144, 144, 144, 144, 144, 4, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 8, 0, 0],
            &[0, 0, 0, 197, 197, 197, 197, 203, 255, 255, 224, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 82, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 243, 255, 187, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 160, 255, 251, 42, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 254, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 200, 255, 232, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 255, 97, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 230, 255, 201, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 135, 255, 254, 53, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 35, 248, 255, 211, 95, 95, 95, 95, 95, 6, 0, 0],
            &[0, 0, 121, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ź' / 0x17a
        'ź' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 105, 141, 141, 42, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 62, 254, 255, 144, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 222, 255, 128, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 124, 80, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 95, 95, 95, 95, 95, 95, 95, 95, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 254, 0, 0, 0],
            &[0, 0, 97, 202, 202, 202, 202, 232, 255, 255, 224, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 236, 255, 244, 48, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 209, 255, 254, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 122, 255, 255, 171, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 75, 253, 255, 206, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 240, 255, 246, 72, 42, 42, 42, 42, 7, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 195, 255, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ż' / 0x17b
        'Ż' => Some(&[
            &[0, 0, 0, 0, 0, 77, 222, 213, 50, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 184, 255, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 134, 255, 255, 89, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 49, 37, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 144, 144, 144, 144, 144, 144, 144, 144, 4, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 8, 0, 0],
            &[0, 0, 0, 197, 197, 197, 197, 203, 255, 255, 224, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 82, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 243, 255, 187, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 160, 255, 251, 42, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 254, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 200, 255, 232, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 255, 97, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 230, 255, 201, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 135, 255, 254, 53, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 35, 248, 255, 211, 95, 95, 95, 95, 95, 6, 0, 0],
            &[0, 0, 121, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ż' / 0x17c
        'ż' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 58, 176, 158, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 198, 255, 255, 118, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 177, 255, 255, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 101, 79, 1, 0, 0, 0, 0, 0],
            &[0, 0, 45, 95, 95, 95, 95, 95, 95, 95, 95, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 254, 0, 0, 0],
            &[0, 0, 97, 202, 202, 202, 202, 232, 255, 255, 224, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 236, 255, 244, 48, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 209, 255, 254, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 122, 255, 255, 171, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 75, 253, 255, 206, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 240, 255, 246, 72, 42, 42, 42, 42, 7, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 195, 255, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ž' / 0x17d
        'Ž' => Some(&[
            &[0, 0, 0, 77, 190, 115, 0, 6, 140, 190, 50, 0, 0, 0],
            &[0, 0, 0, 2, 181, 255, 147, 183, 255, 151, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 231, 255, 255, 209, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 75, 75, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 144, 144, 144, 144, 144, 144, 144, 144, 4, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 8, 0, 0],
            &[0, 0, 0, 197, 197, 197, 197, 203, 255, 255, 224, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 82, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 243, 255, 187, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 161, 255, 251, 42, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 254, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 200, 255, 232, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 255, 97, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 230, 255, 201, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 135, 255, 254, 53, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 35, 248, 255, 211, 95, 95, 95, 95, 95, 6, 0, 0],
            &[0, 0, 121, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ž' / 0x17e
        'ž' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 141, 58, 0, 0, 101, 141, 36, 0, 0, 0],
            &[0, 0, 0, 19, 222, 249, 84, 157, 255, 169, 0, 0, 0, 0],
            &[0, 0, 0, 0, 53, 251, 255, 255, 219, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 86, 124, 124, 49, 0, 0, 0, 0, 0],
            &[0, 0, 45, 95, 95, 95, 95, 95, 95, 95, 95, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 254, 0, 0, 0],
            &[0, 0, 97, 202, 202, 202, 202, 232, 255, 255, 224, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 236, 255, 244, 48, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 209, 255, 254, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 122, 255, 255, 171, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 75, 253, 255, 206, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 39, 240, 255, 246, 72, 42, 42, 42, 42, 7, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 195, 255, 255, 255, 255, 255, 255, 255, 255, 42, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ſ' / 0x17f
        'ſ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 116, 131, 111, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 123, 255, 255, 255, 255, 188, 0, 0, 0, 0],
            &[0, 0, 0, 13, 247, 255, 255, 205, 213, 107, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 255, 172, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        _ => None,
    }
}
