//! Module for letters with the font weight bold and size 20.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 20;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 12;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight bold and font size 17px
#[inline]
pub const fn get_char(c: char) -> Option<&'static [&'static [u8]]> {
    match c {
        // letter: ' ' / 0x20
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '!' / 0x21
        '!' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 35, 35, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 76, 255, 255, 132, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 255, 255, 121, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 255, 255, 109, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 255, 255, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 255, 255, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 255, 255, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 246, 248, 52, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 177, 190, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 103, 255, 255, 149, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 247, 253, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '"' / 0x22
        '"' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 35, 20, 12, 35, 34, 0, 0, 0],
            &[0, 0, 0, 190, 255, 134, 83, 255, 241, 0, 0, 0],
            &[0, 0, 0, 171, 255, 115, 64, 255, 222, 0, 0, 0],
            &[0, 0, 0, 152, 255, 96, 45, 255, 203, 0, 0, 0],
            &[0, 0, 0, 133, 255, 78, 26, 255, 184, 0, 0, 0],
            &[0, 0, 0, 30, 63, 16, 3, 63, 42, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '#' / 0x23
        '#' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 35, 12, 6, 35, 17, 0, 0],
            &[0, 0, 0, 0, 119, 255, 63, 73, 255, 105, 0, 0],
            &[0, 0, 0, 0, 162, 255, 20, 114, 255, 64, 0, 0],
            &[0, 0, 0, 0, 205, 233, 0, 156, 255, 22, 0, 0],
            &[0, 50, 201, 201, 250, 245, 201, 240, 255, 201, 190, 0],
            &[0, 57, 228, 233, 255, 243, 228, 255, 247, 228, 216, 0],
            &[0, 0, 0, 71, 255, 112, 25, 255, 153, 0, 0, 0],
            &[0, 12, 16, 122, 255, 83, 79, 255, 121, 16, 7, 0],
            &[0, 190, 255, 255, 255, 255, 255, 255, 255, 255, 115, 0],
            &[0, 114, 153, 237, 243, 153, 221, 255, 161, 153, 69, 0],
            &[0, 0, 0, 237, 197, 0, 198, 243, 1, 0, 0, 0],
            &[0, 0, 21, 255, 157, 0, 240, 202, 0, 0, 0, 0],
            &[0, 0, 61, 255, 117, 27, 255, 160, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '$' / 0x24
        '$' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 94, 160, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 119, 206, 234, 163, 112, 25, 0, 0],
            &[0, 0, 20, 222, 255, 255, 255, 255, 255, 133, 0, 0],
            &[0, 0, 122, 255, 213, 146, 193, 80, 154, 41, 0, 0],
            &[0, 0, 151, 255, 155, 104, 177, 0, 0, 0, 0, 0],
            &[0, 0, 95, 255, 253, 209, 179, 0, 0, 0, 0, 0],
            &[0, 0, 1, 145, 254, 255, 254, 179, 52, 0, 0, 0],
            &[0, 0, 0, 0, 49, 188, 255, 255, 251, 71, 0, 0],
            &[0, 0, 0, 0, 0, 104, 193, 178, 255, 189, 0, 0],
            &[0, 0, 39, 4, 0, 104, 177, 95, 255, 199, 0, 0],
            &[0, 0, 159, 227, 159, 175, 224, 230, 255, 130, 0, 0],
            &[0, 0, 131, 253, 255, 255, 255, 255, 172, 10, 0, 0],
            &[0, 0, 0, 24, 71, 160, 203, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 103, 175, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '%' / 0x25
        '%' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 57, 15, 0, 0, 0, 29, 34, 0, 0, 0],
            &[75, 246, 255, 238, 52, 0, 34, 251, 179, 0, 0, 0],
            &[208, 230, 84, 250, 174, 0, 161, 254, 50, 0, 0, 0],
            &[246, 181, 0, 218, 211, 39, 252, 174, 0, 0, 0, 0],
            &[224, 215, 28, 244, 190, 166, 254, 45, 0, 0, 0, 0],
            &[115, 255, 255, 254, 128, 253, 169, 0, 0, 0, 0, 0],
            &[0, 68, 113, 55, 172, 253, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 48, 254, 164, 57, 140, 118, 12, 0, 0],
            &[0, 0, 0, 178, 252, 89, 252, 244, 255, 188, 0, 0],
            &[0, 0, 53, 255, 159, 141, 255, 41, 162, 255, 24, 0],
            &[0, 0, 183, 250, 34, 159, 255, 13, 132, 255, 41, 0],
            &[0, 59, 255, 154, 0, 115, 255, 135, 212, 244, 10, 0],
            &[0, 189, 249, 30, 0, 12, 194, 255, 249, 98, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 24, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '&' / 0x26
        '&' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 61, 17, 0, 0, 0, 0, 0],
            &[0, 0, 1, 174, 255, 255, 243, 80, 0, 0, 0, 0],
            &[0, 0, 81, 255, 222, 130, 255, 226, 0, 0, 0, 0],
            &[0, 0, 124, 255, 146, 3, 249, 255, 5, 0, 0, 0],
            &[0, 0, 102, 255, 192, 96, 255, 205, 0, 0, 0, 0],
            &[0, 0, 19, 240, 255, 255, 251, 54, 0, 0, 0, 0],
            &[0, 0, 3, 255, 255, 255, 115, 0, 78, 104, 44, 0],
            &[0, 0, 153, 255, 255, 255, 194, 5, 233, 255, 68, 0],
            &[0, 48, 255, 251, 83, 231, 255, 199, 255, 246, 13, 0],
            &[0, 113, 255, 196, 0, 70, 254, 255, 255, 155, 0, 0],
            &[0, 110, 255, 232, 10, 5, 255, 255, 255, 68, 0, 0],
            &[0, 42, 253, 255, 235, 242, 255, 255, 255, 200, 5, 0],
            &[0, 0, 103, 241, 255, 255, 206, 105, 244, 255, 144, 0],
            &[0, 0, 0, 7, 35, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ''' / 0x27
        '\'' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 35, 35, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 255, 255, 57, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 247, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 229, 255, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 210, 254, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 49, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '(' / 0x28
        '(' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 35, 21, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 197, 255, 70, 0, 0, 0, 0],
            &[0, 0, 0, 0, 98, 255, 180, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 49, 255, 235, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 111, 255, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 160, 255, 131, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 180, 255, 110, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 192, 255, 100, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 176, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 152, 255, 143, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 255, 199, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 254, 249, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 181, 255, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 58, 253, 221, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 205, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ')' / 0x29
        ')' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 35, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 245, 233, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 253, 248, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 190, 255, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 128, 255, 168, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 58, 255, 233, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 49, 255, 242, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 65, 255, 225, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 92, 255, 199, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 146, 255, 142, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 213, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 255, 224, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 255, 102, 0, 0, 0, 0, 0],
            &[0, 0, 0, 43, 205, 169, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '*' / 0x2a
        '*' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 182, 213, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 199, 251, 1, 0, 0, 0, 0],
            &[0, 0, 73, 135, 68, 187, 233, 52, 124, 114, 0, 0],
            &[0, 0, 153, 255, 255, 255, 255, 255, 255, 207, 0, 0],
            &[0, 0, 16, 52, 148, 255, 255, 186, 62, 30, 0, 0],
            &[0, 0, 0, 26, 233, 218, 182, 251, 63, 0, 0, 0],
            &[0, 0, 0, 138, 255, 88, 44, 253, 192, 0, 0, 0],
            &[0, 0, 0, 0, 76, 1, 0, 75, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '+' / 0x2b
        '+' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 255, 8, 0, 0, 0, 0, 0],
            &[0, 186, 230, 230, 250, 255, 231, 230, 230, 2, 0, 0],
            &[0, 186, 230, 230, 250, 255, 231, 230, 230, 2, 0, 0],
            &[0, 0, 0, 0, 200, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ',' / 0x2c
        ',' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 64, 79, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 234, 255, 127, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 255, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 255, 231, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 116, 255, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 105, 166, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '-' / 0x2d
        '-' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 70, 156, 156, 156, 156, 101, 0, 0, 0],
            &[0, 0, 0, 115, 255, 255, 255, 255, 166, 0, 0, 0],
            &[0, 0, 0, 59, 131, 131, 131, 131, 85, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '.' / 0x2e
        '.' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 177, 190, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 103, 255, 255, 149, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 247, 253, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '/' / 0x2f
        '/' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 35, 52, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 226, 255, 59, 0, 0],
            &[0, 0, 0, 0, 0, 0, 70, 255, 217, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 167, 255, 121, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 247, 252, 27, 0, 0, 0],
            &[0, 0, 0, 0, 0, 106, 255, 182, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 204, 255, 85, 0, 0, 0, 0],
            &[0, 0, 0, 0, 45, 255, 237, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 143, 255, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 234, 255, 50, 0, 0, 0, 0, 0],
            &[0, 0, 0, 82, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 179, 255, 111, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 252, 250, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 17, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '0' / 0x30
        '0' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 60, 63, 19, 0, 0, 0, 0],
            &[0, 0, 0, 100, 241, 255, 255, 247, 125, 0, 0, 0],
            &[0, 0, 60, 254, 255, 228, 217, 255, 255, 94, 0, 0],
            &[0, 0, 174, 255, 210, 8, 10, 255, 255, 214, 0, 0],
            &[0, 0, 236, 255, 117, 0, 130, 255, 255, 255, 25, 0],
            &[0, 17, 255, 255, 77, 36, 248, 255, 255, 255, 65, 0],
            &[0, 32, 255, 255, 63, 183, 255, 100, 255, 255, 81, 0],
            &[0, 28, 255, 255, 147, 255, 174, 27, 255, 255, 84, 0],
            &[0, 11, 255, 255, 255, 244, 29, 41, 255, 255, 67, 0],
            &[0, 0, 222, 255, 255, 116, 0, 83, 255, 255, 29, 0],
            &[0, 0, 150, 255, 255, 18, 3, 186, 255, 214, 0, 0],
            &[0, 0, 34, 242, 255, 245, 237, 255, 255, 90, 0, 0],
            &[0, 0, 0, 59, 211, 255, 255, 237, 110, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 33, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '1' / 0x31
        '1' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 35, 25, 0, 0, 0, 0, 0],
            &[0, 0, 11, 123, 241, 255, 187, 0, 0, 0, 0, 0],
            &[0, 95, 231, 255, 255, 255, 187, 0, 0, 0, 0, 0],
            &[0, 111, 255, 245, 228, 255, 187, 0, 0, 0, 0, 0],
            &[0, 5, 139, 29, 179, 255, 187, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 186, 255, 187, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 255, 187, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 255, 187, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 255, 187, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 255, 187, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 255, 187, 0, 0, 0, 0, 0],
            &[0, 14, 199, 227, 255, 255, 255, 226, 198, 1, 0, 0],
            &[0, 20, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '2' / 0x32
        '2' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 58, 70, 32, 0, 0, 0, 0],
            &[0, 0, 24, 168, 252, 255, 255, 255, 183, 19, 0, 0],
            &[0, 1, 205, 255, 255, 231, 236, 255, 255, 177, 0, 0],
            &[0, 0, 50, 217, 73, 0, 5, 186, 255, 254, 10, 0],
            &[0, 0, 0, 1, 0, 0, 0, 126, 255, 255, 15, 0],
            &[0, 0, 0, 0, 0, 0, 0, 194, 255, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 114, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 88, 252, 255, 147, 0, 0, 0],
            &[0, 0, 0, 0, 85, 251, 255, 147, 1, 0, 0, 0],
            &[0, 0, 0, 85, 251, 255, 131, 0, 0, 0, 0, 0],
            &[0, 0, 85, 251, 255, 133, 27, 27, 27, 27, 12, 0],
            &[0, 6, 251, 255, 255, 255, 255, 255, 255, 255, 111, 0],
            &[0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 111, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '3' / 0x33
        '3' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 68, 69, 26, 0, 0, 0, 0, 0],
            &[0, 74, 204, 255, 255, 255, 254, 168, 10, 0, 0, 0],
            &[0, 112, 255, 254, 223, 230, 255, 255, 128, 0, 0, 0],
            &[0, 3, 121, 31, 0, 4, 212, 255, 178, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 184, 255, 140, 0, 0, 0],
            &[0, 0, 1, 57, 77, 144, 253, 238, 36, 0, 0, 0],
            &[0, 0, 6, 255, 255, 255, 218, 48, 0, 0, 0, 0],
            &[0, 0, 4, 191, 206, 239, 255, 245, 85, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 188, 255, 236, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 255, 255, 16, 0, 0],
            &[0, 108, 58, 7, 0, 27, 209, 255, 239, 0, 0, 0],
            &[0, 202, 255, 254, 239, 254, 255, 255, 124, 0, 0, 0],
            &[0, 126, 224, 255, 255, 255, 227, 112, 0, 0, 0, 0],
            &[0, 0, 0, 14, 36, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '4' / 0x34
        '4' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 35, 33, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 189, 255, 244, 0, 0, 0],
            &[0, 0, 0, 0, 0, 97, 255, 255, 244, 0, 0, 0],
            &[0, 0, 0, 0, 22, 237, 251, 255, 244, 0, 0, 0],
            &[0, 0, 0, 0, 166, 255, 160, 255, 244, 0, 0, 0],
            &[0, 0, 0, 73, 255, 186, 91, 255, 244, 0, 0, 0],
            &[0, 0, 11, 224, 249, 39, 99, 255, 244, 0, 0, 0],
            &[0, 0, 142, 255, 136, 0, 99, 255, 244, 0, 0, 0],
            &[0, 45, 252, 255, 141, 129, 178, 255, 250, 129, 67, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 133, 0],
            &[0, 39, 123, 123, 123, 123, 175, 255, 250, 123, 64, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 244, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 244, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '5' / 0x35
        '5' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 35, 35, 35, 35, 35, 35, 9, 0, 0],
            &[0, 0, 19, 255, 255, 255, 255, 255, 255, 67, 0, 0],
            &[0, 0, 39, 255, 255, 249, 248, 248, 248, 65, 0, 0],
            &[0, 0, 59, 255, 255, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 78, 255, 255, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 98, 255, 255, 168, 176, 131, 26, 0, 0, 0],
            &[0, 0, 118, 255, 255, 255, 255, 255, 235, 34, 0, 0],
            &[0, 0, 22, 118, 88, 81, 165, 255, 255, 164, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 202, 255, 221, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 181, 255, 222, 0, 0],
            &[0, 0, 104, 70, 18, 3, 65, 245, 255, 171, 0, 0],
            &[0, 0, 181, 255, 255, 254, 255, 255, 244, 47, 0, 0],
            &[0, 0, 113, 232, 255, 255, 252, 189, 50, 0, 0, 0],
            &[0, 0, 0, 1, 21, 36, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '6' / 0x36
        '6' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 60, 64, 19, 0, 0],
            &[0, 0, 0, 0, 103, 223, 255, 255, 255, 106, 0, 0],
            &[0, 0, 0, 146, 255, 255, 238, 194, 197, 92, 0, 0],
            &[0, 0, 64, 255, 255, 150, 9, 0, 0, 0, 0, 0],
            &[0, 0, 167, 255, 211, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 227, 255, 133, 130, 208, 214, 140, 9, 0, 0],
            &[0, 3, 254, 255, 220, 255, 255, 255, 255, 155, 0, 0],
            &[0, 12, 255, 255, 246, 88, 37, 185, 255, 251, 16, 0],
            &[0, 2, 253, 255, 158, 0, 0, 71, 255, 255, 52, 0],
            &[0, 0, 216, 255, 139, 0, 0, 71, 255, 255, 54, 0],
            &[0, 0, 139, 255, 230, 41, 11, 175, 255, 248, 14, 0],
            &[0, 0, 23, 236, 255, 255, 254, 255, 255, 147, 0, 0],
            &[0, 0, 0, 46, 203, 255, 255, 246, 145, 6, 0, 0],
            &[0, 0, 0, 0, 0, 26, 39, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '7' / 0x37
        '7' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 30, 30, 30, 30, 30, 30, 30, 30, 10, 0],
            &[0, 38, 255, 255, 255, 255, 255, 255, 255, 255, 89, 0],
            &[0, 37, 248, 248, 248, 248, 248, 251, 255, 255, 75, 0],
            &[0, 0, 0, 0, 0, 0, 0, 156, 255, 228, 4, 0],
            &[0, 0, 0, 0, 0, 0, 24, 248, 255, 121, 0, 0],
            &[0, 0, 0, 0, 0, 0, 132, 255, 246, 19, 0, 0],
            &[0, 0, 0, 0, 0, 11, 237, 255, 153, 0, 0, 0],
            &[0, 0, 0, 0, 0, 109, 255, 254, 42, 0, 0, 0],
            &[0, 0, 0, 0, 3, 222, 255, 185, 0, 0, 0, 0],
            &[0, 0, 0, 0, 86, 255, 255, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 202, 255, 216, 1, 0, 0, 0, 0],
            &[0, 0, 0, 62, 255, 255, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 178, 255, 239, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '8' / 0x38
        '8' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 60, 65, 25, 0, 0, 0, 0, 0],
            &[0, 1, 130, 248, 255, 255, 255, 171, 12, 0, 0, 0],
            &[0, 90, 255, 255, 229, 216, 255, 255, 144, 0, 0, 0],
            &[0, 149, 255, 234, 7, 0, 191, 255, 201, 0, 0, 0],
            &[0, 119, 255, 241, 19, 2, 203, 255, 170, 0, 0, 0],
            &[0, 19, 233, 255, 216, 190, 255, 242, 46, 0, 0, 0],
            &[0, 0, 46, 250, 255, 255, 248, 51, 0, 0, 0, 0],
            &[0, 24, 210, 255, 238, 241, 255, 218, 37, 0, 0, 0],
            &[0, 167, 255, 222, 26, 29, 212, 255, 205, 0, 0, 0],
            &[0, 236, 255, 116, 0, 0, 70, 255, 255, 30, 0, 0],
            &[0, 235, 255, 167, 2, 0, 118, 255, 255, 26, 0, 0],
            &[0, 146, 255, 255, 233, 225, 255, 255, 180, 0, 0, 0],
            &[0, 5, 140, 240, 255, 255, 243, 151, 12, 0, 0, 0],
            &[0, 0, 0, 3, 31, 34, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '9' / 0x39
        '9' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 70, 68, 15, 0, 0, 0, 0],
            &[0, 0, 3, 145, 251, 255, 255, 246, 108, 0, 0, 0],
            &[0, 0, 115, 255, 255, 238, 238, 255, 255, 81, 0, 0],
            &[0, 0, 220, 255, 198, 7, 6, 183, 255, 207, 0, 0],
            &[0, 6, 254, 255, 118, 0, 0, 87, 255, 255, 21, 0],
            &[0, 3, 250, 255, 128, 0, 0, 113, 255, 255, 55, 0],
            &[0, 0, 207, 255, 231, 83, 96, 239, 255, 255, 64, 0],
            &[0, 0, 82, 255, 255, 255, 255, 207, 255, 255, 51, 0],
            &[0, 0, 0, 77, 170, 180, 120, 91, 255, 255, 16, 0],
            &[0, 0, 0, 0, 0, 0, 0, 178, 255, 206, 0, 0],
            &[0, 0, 0, 0, 0, 8, 124, 255, 255, 97, 0, 0],
            &[0, 0, 62, 228, 219, 247, 255, 255, 170, 1, 0, 0],
            &[0, 0, 63, 255, 255, 255, 214, 106, 2, 0, 0, 0],
            &[0, 0, 4, 35, 38, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ':' / 0x3a
        ':' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 83, 96, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 255, 255, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 89, 255, 255, 134, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 103, 117, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 177, 190, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 103, 255, 255, 149, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 247, 253, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ';' / 0x3b
        ';' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 45, 106, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 240, 255, 224, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 244, 255, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 128, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 64, 79, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 234, 255, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 20, 255, 255, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 65, 255, 231, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 146, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 105, 166, 41, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '<' / 0x3c
        '<' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 37, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 57, 189, 193, 0, 0],
            &[0, 0, 0, 0, 0, 49, 180, 255, 255, 155, 0, 0],
            &[0, 0, 0, 42, 172, 254, 255, 186, 60, 0, 0, 0],
            &[0, 0, 107, 253, 255, 184, 57, 0, 0, 0, 0, 0],
            &[0, 0, 101, 249, 255, 212, 94, 5, 0, 0, 0, 0],
            &[0, 0, 0, 29, 147, 247, 255, 226, 112, 12, 0, 0],
            &[0, 0, 0, 0, 0, 24, 140, 244, 255, 187, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 20, 133, 178, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '=' / 0x3d
        '=' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 162, 162, 162, 162, 162, 162, 162, 1, 0, 0],
            &[0, 207, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 83, 103, 103, 103, 103, 103, 103, 103, 1, 0, 0],
            &[0, 13, 16, 16, 16, 16, 16, 16, 16, 0, 0, 0],
            &[0, 207, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 201, 249, 249, 249, 249, 249, 249, 249, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '>' / 0x3e
        '>' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 142, 213, 83, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 252, 255, 206, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 38, 161, 252, 255, 198, 66, 0, 0, 0],
            &[0, 0, 0, 0, 0, 35, 157, 252, 255, 156, 0, 0],
            &[0, 0, 0, 0, 0, 70, 189, 255, 255, 146, 0, 0],
            &[0, 0, 3, 89, 207, 255, 254, 172, 48, 0, 0, 0],
            &[0, 0, 136, 255, 252, 165, 42, 0, 0, 0, 0, 0],
            &[0, 0, 136, 157, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '?' / 0x3f
        '?' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 66, 71, 32, 0, 0, 0, 0],
            &[0, 0, 64, 201, 255, 255, 255, 255, 179, 13, 0, 0],
            &[0, 0, 66, 255, 249, 208, 220, 255, 255, 145, 0, 0],
            &[0, 0, 0, 94, 19, 0, 0, 171, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 195, 255, 182, 0, 0],
            &[0, 0, 0, 0, 0, 5, 154, 255, 253, 70, 0, 0],
            &[0, 0, 0, 0, 2, 185, 255, 243, 80, 0, 0, 0],
            &[0, 0, 0, 0, 81, 255, 243, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 116, 248, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 200, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 220, 255, 255, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 218, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 32, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '@' / 0x40
        '@' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 27, 4, 0, 0, 0, 0, 0],
            &[0, 0, 35, 176, 251, 255, 250, 162, 19, 0, 0, 0],
            &[0, 45, 240, 234, 119, 78, 129, 248, 211, 10, 0, 0],
            &[4, 212, 233, 31, 0, 0, 0, 72, 255, 127, 0, 0],
            &[84, 255, 103, 25, 165, 206, 174, 68, 197, 221, 0, 0],
            &[168, 250, 12, 200, 251, 197, 255, 137, 140, 255, 21, 0],
            &[220, 205, 49, 255, 128, 39, 255, 124, 119, 255, 43, 0],
            &[244, 181, 89, 255, 71, 56, 255, 111, 126, 255, 33, 0],
            &[248, 178, 82, 255, 77, 93, 255, 105, 155, 247, 7, 0],
            &[226, 200, 34, 254, 181, 197, 255, 183, 233, 173, 0, 0],
            &[176, 249, 11, 149, 255, 221, 135, 254, 230, 31, 0, 0],
            &[88, 255, 120, 0, 22, 2, 0, 21, 8, 0, 0, 0],
            &[1, 192, 252, 112, 9, 0, 10, 70, 57, 0, 0, 0],
            &[0, 13, 174, 255, 253, 238, 255, 255, 100, 0, 0, 0],
            &[0, 0, 0, 42, 102, 123, 101, 45, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'A' / 0x41
        'A' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 48, 48, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 171, 255, 255, 218, 0, 0, 0, 0, 0],
            &[0, 0, 3, 238, 255, 250, 255, 33, 0, 0, 0, 0],
            &[0, 0, 56, 255, 233, 183, 255, 104, 0, 0, 0, 0],
            &[0, 0, 126, 255, 175, 122, 255, 174, 0, 0, 0, 0],
            &[0, 0, 196, 255, 112, 62, 255, 241, 4, 0, 0, 0],
            &[0, 15, 251, 255, 48, 7, 248, 255, 60, 0, 0, 0],
            &[0, 81, 255, 246, 14, 12, 205, 255, 130, 0, 0, 0],
            &[0, 151, 255, 255, 255, 255, 255, 255, 201, 0, 0, 0],
            &[0, 222, 255, 255, 253, 253, 255, 255, 253, 18, 0, 0],
            &[36, 255, 255, 48, 0, 0, 13, 252, 255, 86, 0, 0],
            &[106, 255, 238, 2, 0, 0, 0, 203, 255, 157, 0, 0],
            &[177, 255, 177, 0, 0, 0, 0, 142, 255, 227, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'B' / 0x42
        'B' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 35, 26, 3, 0, 0, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 252, 185, 52, 0, 0, 0],
            &[0, 163, 255, 245, 226, 238, 255, 255, 236, 10, 0, 0],
            &[0, 163, 255, 167, 0, 0, 125, 255, 255, 58, 0, 0],
            &[0, 163, 255, 167, 0, 0, 72, 255, 255, 46, 0, 0],
            &[0, 163, 255, 199, 92, 103, 206, 255, 206, 1, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 209, 24, 0, 0, 0],
            &[0, 163, 255, 232, 186, 199, 246, 255, 200, 8, 0, 0],
            &[0, 163, 255, 167, 0, 0, 68, 255, 255, 99, 0, 0],
            &[0, 163, 255, 167, 0, 0, 21, 255, 255, 126, 0, 0],
            &[0, 163, 255, 169, 5, 14, 126, 255, 255, 92, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 220, 11, 0, 0],
            &[0, 163, 255, 255, 255, 253, 231, 160, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'C' / 0x43
        'C' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 53, 72, 52, 11, 0, 0],
            &[0, 0, 0, 5, 133, 242, 255, 255, 255, 243, 75, 0],
            &[0, 0, 0, 176, 255, 255, 246, 224, 247, 250, 23, 0],
            &[0, 0, 90, 255, 255, 138, 9, 0, 7, 58, 0, 0],
            &[0, 0, 194, 255, 215, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 249, 255, 131, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 255, 255, 103, 0, 0, 0, 0, 0, 0, 0],
            &[0, 31, 255, 255, 96, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 213, 255, 201, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 124, 255, 255, 148, 28, 5, 38, 120, 7, 0],
            &[0, 0, 10, 211, 255, 255, 255, 255, 255, 255, 11, 0],
            &[0, 0, 0, 17, 155, 246, 255, 255, 248, 181, 6, 0],
            &[0, 0, 0, 0, 0, 3, 29, 30, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'D' / 0x44
        'D' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 35, 35, 35, 31, 7, 0, 0, 0, 0, 0],
            &[0, 60, 255, 255, 255, 255, 254, 196, 68, 0, 0, 0],
            &[0, 60, 255, 255, 246, 248, 255, 255, 252, 88, 0, 0],
            &[0, 60, 255, 255, 50, 0, 44, 210, 255, 238, 15, 0],
            &[0, 60, 255, 255, 50, 0, 0, 53, 255, 255, 93, 0],
            &[0, 60, 255, 255, 50, 0, 0, 0, 236, 255, 146, 0],
            &[0, 60, 255, 255, 50, 0, 0, 0, 213, 255, 168, 0],
            &[0, 60, 255, 255, 50, 0, 0, 0, 223, 255, 161, 0],
            &[0, 60, 255, 255, 50, 0, 0, 6, 248, 255, 135, 0],
            &[0, 60, 255, 255, 50, 0, 0, 92, 255, 255, 66, 0],
            &[0, 60, 255, 255, 68, 38, 102, 243, 255, 208, 3, 0],
            &[0, 60, 255, 255, 255, 255, 255, 255, 232, 45, 0, 0],
            &[0, 60, 255, 255, 255, 249, 220, 144, 27, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'E' / 0x45
        'E' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 35, 35, 35, 35, 35, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 252, 252, 252, 252, 252, 2, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 213, 92, 92, 92, 92, 69, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 193, 0, 0, 0],
            &[0, 163, 255, 240, 195, 195, 195, 195, 147, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 197, 31, 31, 31, 31, 31, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'F' / 0x46
        'F' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 35, 35, 35, 35, 35, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 252, 252, 252, 252, 252, 2, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 245, 217, 217, 217, 217, 164, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 193, 0, 0, 0],
            &[0, 163, 255, 207, 70, 70, 70, 70, 52, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'G' / 0x47
        'G' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 63, 64, 34, 0, 0, 0],
            &[0, 0, 0, 42, 191, 255, 255, 255, 255, 218, 37, 0],
            &[0, 0, 41, 240, 255, 255, 234, 218, 247, 231, 4, 0],
            &[0, 0, 190, 255, 244, 81, 0, 0, 10, 51, 0, 0],
            &[0, 29, 255, 255, 121, 0, 0, 0, 0, 0, 0, 0],
            &[0, 89, 255, 255, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 112, 255, 254, 4, 0, 101, 213, 213, 213, 92, 0],
            &[0, 116, 255, 254, 3, 0, 121, 255, 255, 255, 111, 0],
            &[0, 93, 255, 255, 34, 0, 25, 52, 213, 255, 111, 0],
            &[0, 37, 255, 255, 117, 0, 0, 0, 202, 255, 111, 0],
            &[0, 0, 199, 255, 244, 79, 1, 1, 206, 255, 111, 0],
            &[0, 0, 48, 240, 255, 255, 247, 248, 255, 255, 111, 0],
            &[0, 0, 0, 43, 177, 252, 255, 255, 253, 196, 60, 0],
            &[0, 0, 0, 0, 0, 6, 33, 29, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'H' / 0x48
        'H' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 35, 35, 8, 0, 0, 1, 35, 35, 15, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 136, 101, 101, 105, 255, 255, 111, 0],
            &[0, 60, 255, 255, 255, 255, 255, 255, 255, 255, 111, 0],
            &[0, 60, 255, 255, 202, 186, 186, 188, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'I' / 0x49
        'I' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 35, 35, 35, 35, 35, 35, 23, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 70, 172, 206, 255, 255, 216, 177, 102, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 87, 207, 237, 255, 255, 242, 212, 126, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'J' / 0x4a
        'J' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 35, 35, 11, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 255, 255, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 50, 255, 255, 69, 0, 0],
            &[0, 0, 35, 29, 2, 23, 171, 255, 254, 26, 0, 0],
            &[0, 0, 142, 255, 254, 255, 255, 255, 173, 0, 0, 0],
            &[0, 0, 123, 252, 255, 255, 245, 157, 14, 0, 0, 0],
            &[0, 0, 0, 10, 34, 32, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'K' / 0x4b
        'K' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 35, 35, 6, 0, 0, 4, 35, 35, 20, 0],
            &[0, 60, 255, 255, 45, 0, 0, 129, 255, 249, 49, 0],
            &[0, 60, 255, 255, 45, 0, 49, 250, 255, 119, 0, 0],
            &[0, 60, 255, 255, 45, 5, 209, 255, 192, 2, 0, 0],
            &[0, 60, 255, 255, 45, 129, 255, 239, 30, 0, 0, 0],
            &[0, 60, 255, 255, 90, 250, 255, 89, 0, 0, 0, 0],
            &[0, 60, 255, 255, 223, 255, 238, 8, 0, 0, 0, 0],
            &[0, 60, 255, 255, 255, 255, 255, 120, 0, 0, 0, 0],
            &[0, 60, 255, 255, 176, 151, 255, 241, 21, 0, 0, 0],
            &[0, 60, 255, 255, 45, 21, 243, 255, 148, 0, 0, 0],
            &[0, 60, 255, 255, 45, 0, 129, 255, 251, 39, 0, 0],
            &[0, 60, 255, 255, 45, 0, 14, 235, 255, 176, 0, 0],
            &[0, 60, 255, 255, 45, 0, 0, 114, 255, 255, 62, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'L' / 0x4c
        'L' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 35, 35, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 45, 45, 45, 45, 45, 0, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'M' / 0x4d
        'M' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 35, 35, 35, 0, 0, 23, 35, 35, 17, 0],
            &[0, 81, 255, 255, 255, 27, 0, 197, 255, 255, 128, 0],
            &[0, 81, 255, 255, 255, 66, 1, 242, 255, 255, 128, 0],
            &[0, 81, 255, 219, 255, 104, 32, 255, 220, 255, 128, 0],
            &[0, 81, 255, 189, 255, 143, 78, 255, 189, 255, 128, 0],
            &[0, 81, 255, 170, 246, 181, 123, 255, 161, 255, 128, 0],
            &[0, 81, 255, 185, 200, 220, 169, 241, 145, 255, 128, 0],
            &[0, 81, 255, 201, 151, 252, 222, 197, 157, 255, 128, 0],
            &[0, 81, 255, 214, 103, 255, 255, 152, 169, 255, 128, 0],
            &[0, 81, 255, 223, 54, 255, 255, 107, 177, 255, 128, 0],
            &[0, 81, 255, 227, 3, 36, 36, 11, 180, 255, 128, 0],
            &[0, 81, 255, 227, 0, 0, 0, 0, 180, 255, 128, 0],
            &[0, 81, 255, 227, 0, 0, 0, 0, 180, 255, 128, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'N' / 0x4e
        'N' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 35, 35, 35, 4, 0, 0, 26, 35, 15, 0],
            &[0, 60, 255, 255, 255, 82, 0, 0, 193, 255, 111, 0],
            &[0, 60, 255, 255, 255, 166, 0, 0, 193, 255, 111, 0],
            &[0, 60, 255, 222, 248, 242, 7, 0, 193, 255, 111, 0],
            &[0, 60, 255, 221, 179, 255, 77, 0, 193, 255, 111, 0],
            &[0, 60, 255, 234, 96, 255, 161, 0, 193, 255, 111, 0],
            &[0, 60, 255, 243, 18, 251, 239, 5, 193, 255, 111, 0],
            &[0, 60, 255, 245, 0, 187, 255, 72, 191, 255, 111, 0],
            &[0, 60, 255, 245, 0, 104, 255, 156, 186, 255, 111, 0],
            &[0, 60, 255, 245, 0, 23, 253, 236, 182, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 194, 255, 233, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 111, 255, 255, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 29, 254, 255, 255, 111, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'O' / 0x4f
        'O' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 67, 77, 34, 0, 0, 0, 0],
            &[0, 0, 0, 135, 249, 255, 255, 255, 175, 13, 0, 0],
            &[0, 0, 115, 255, 255, 232, 228, 255, 255, 170, 0, 0],
            &[0, 6, 236, 255, 187, 6, 0, 148, 255, 255, 41, 0],
            &[0, 58, 255, 255, 71, 0, 0, 28, 255, 255, 114, 0],
            &[0, 100, 255, 255, 24, 0, 0, 0, 232, 255, 154, 0],
            &[0, 118, 255, 255, 6, 0, 0, 0, 214, 255, 171, 0],
            &[0, 118, 255, 255, 8, 0, 0, 0, 213, 255, 169, 0],
            &[0, 98, 255, 255, 29, 0, 0, 0, 235, 255, 149, 0],
            &[0, 53, 255, 255, 84, 0, 0, 36, 255, 255, 104, 0],
            &[0, 2, 227, 255, 210, 27, 12, 176, 255, 251, 28, 0],
            &[0, 0, 90, 255, 255, 255, 253, 255, 255, 140, 0, 0],
            &[0, 0, 0, 95, 228, 255, 255, 240, 131, 3, 0, 0],
            &[0, 0, 0, 0, 1, 28, 33, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'P' / 0x50
        'P' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 35, 32, 11, 0, 0, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 253, 195, 46, 0, 0, 0],
            &[0, 163, 255, 252, 243, 250, 255, 255, 234, 16, 0, 0],
            &[0, 163, 255, 189, 0, 2, 97, 255, 255, 97, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 248, 255, 127, 0, 0],
            &[0, 163, 255, 189, 0, 0, 32, 255, 255, 110, 0, 0],
            &[0, 163, 255, 216, 104, 128, 220, 255, 254, 41, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 130, 0, 0, 0],
            &[0, 163, 255, 234, 174, 167, 139, 58, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Q' / 0x51
        'Q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 67, 77, 34, 0, 0, 0, 0],
            &[0, 0, 0, 135, 249, 255, 255, 255, 175, 13, 0, 0],
            &[0, 0, 115, 255, 255, 232, 228, 255, 255, 170, 0, 0],
            &[0, 6, 236, 255, 187, 6, 0, 148, 255, 255, 41, 0],
            &[0, 58, 255, 255, 71, 0, 0, 28, 255, 255, 114, 0],
            &[0, 100, 255, 255, 24, 0, 0, 0, 232, 255, 154, 0],
            &[0, 118, 255, 255, 6, 0, 0, 0, 214, 255, 171, 0],
            &[0, 118, 255, 255, 8, 0, 0, 0, 213, 255, 169, 0],
            &[0, 98, 255, 255, 29, 0, 0, 0, 235, 255, 149, 0],
            &[0, 53, 255, 255, 84, 0, 0, 36, 255, 255, 104, 0],
            &[0, 2, 227, 255, 210, 27, 12, 176, 255, 251, 28, 0],
            &[0, 0, 90, 255, 255, 255, 254, 255, 255, 140, 0, 0],
            &[0, 0, 0, 95, 228, 255, 255, 255, 243, 3, 0, 0],
            &[0, 0, 0, 0, 1, 28, 170, 255, 235, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 243, 255, 133, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 136, 255, 245, 26, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 15, 15, 5, 0],
        ]),
        // letter: 'R' / 0x52
        'R' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 33, 16, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 220, 105, 0, 0, 0, 0],
            &[0, 163, 255, 253, 247, 255, 255, 255, 110, 0, 0, 0],
            &[0, 163, 255, 211, 0, 29, 206, 255, 224, 0, 0, 0],
            &[0, 163, 255, 211, 0, 0, 128, 255, 254, 1, 0, 0],
            &[0, 163, 255, 211, 0, 4, 190, 255, 229, 0, 0, 0],
            &[0, 163, 255, 240, 174, 220, 255, 255, 112, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 125, 2, 0, 0, 0],
            &[0, 163, 255, 230, 119, 253, 255, 86, 0, 0, 0, 0],
            &[0, 163, 255, 211, 0, 156, 255, 225, 9, 0, 0, 0],
            &[0, 163, 255, 211, 0, 25, 244, 255, 127, 0, 0, 0],
            &[0, 163, 255, 211, 0, 0, 128, 255, 245, 29, 0, 0],
            &[0, 163, 255, 211, 0, 0, 11, 231, 255, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'S' / 0x53
        'S' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 58, 72, 52, 15, 0, 0, 0, 0],
            &[0, 0, 108, 243, 255, 255, 255, 253, 181, 0, 0, 0],
            &[0, 62, 255, 255, 240, 197, 220, 255, 160, 0, 0, 0],
            &[0, 132, 255, 254, 32, 0, 0, 35, 31, 0, 0, 0],
            &[0, 129, 255, 254, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 253, 255, 219, 76, 0, 0, 0, 0, 0, 0],
            &[0, 0, 102, 249, 255, 255, 194, 52, 0, 0, 0, 0],
            &[0, 0, 0, 43, 192, 255, 255, 253, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 226, 255, 250, 21, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 255, 255, 60, 0, 0],
            &[0, 134, 77, 10, 0, 3, 150, 255, 255, 35, 0, 0],
            &[0, 207, 255, 254, 233, 245, 255, 255, 178, 0, 0, 0],
            &[0, 118, 222, 255, 255, 255, 239, 143, 10, 0, 0, 0],
            &[0, 0, 0, 13, 36, 29, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'T' / 0x54
        'T' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 35, 35, 35, 35, 35, 35, 35, 35, 24, 0],
            &[0, 125, 255, 255, 255, 255, 255, 255, 255, 255, 176, 0],
            &[0, 123, 252, 252, 253, 255, 255, 253, 252, 252, 174, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'U' / 0x55
        'U' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 35, 35, 8, 0, 0, 1, 35, 35, 15, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 59, 255, 255, 64, 0, 0, 12, 255, 255, 111, 0],
            &[0, 44, 255, 255, 95, 0, 0, 41, 255, 255, 95, 0],
            &[0, 8, 243, 255, 206, 26, 15, 162, 255, 255, 43, 0],
            &[0, 0, 135, 255, 255, 255, 253, 255, 255, 178, 0, 0],
            &[0, 0, 4, 129, 237, 255, 255, 244, 151, 11, 0, 0],
            &[0, 0, 0, 0, 4, 29, 32, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'V' / 0x56
        'V' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[28, 35, 23, 0, 0, 0, 0, 17, 35, 35, 0, 0],
            &[164, 255, 208, 0, 0, 0, 0, 164, 255, 215, 0, 0],
            &[89, 255, 254, 20, 0, 0, 0, 228, 255, 140, 0, 0],
            &[18, 252, 255, 84, 0, 0, 35, 255, 255, 65, 0, 0],
            &[0, 195, 255, 150, 0, 0, 99, 255, 241, 5, 0, 0],
            &[0, 120, 255, 216, 0, 0, 162, 255, 172, 0, 0, 0],
            &[0, 45, 255, 255, 26, 0, 225, 255, 97, 0, 0, 0],
            &[0, 0, 226, 255, 91, 33, 255, 254, 23, 0, 0, 0],
            &[0, 0, 152, 255, 156, 92, 255, 203, 0, 0, 0, 0],
            &[0, 0, 77, 255, 213, 143, 255, 128, 0, 0, 0, 0],
            &[0, 0, 10, 247, 252, 201, 255, 53, 0, 0, 0, 0],
            &[0, 0, 0, 183, 255, 255, 232, 1, 0, 0, 0, 0],
            &[0, 0, 0, 108, 255, 255, 159, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'W' / 0x57
        'W' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[29, 35, 11, 0, 0, 0, 0, 4, 35, 35, 0, 0],
            &[200, 255, 95, 0, 0, 0, 0, 40, 255, 251, 0, 0],
            &[181, 255, 119, 0, 0, 0, 0, 58, 255, 234, 0, 0],
            &[162, 255, 143, 5, 72, 72, 22, 76, 255, 216, 0, 0],
            &[143, 255, 164, 45, 255, 255, 104, 94, 255, 198, 0, 0],
            &[124, 255, 182, 84, 255, 255, 142, 112, 255, 180, 0, 0],
            &[105, 255, 195, 124, 255, 219, 181, 129, 255, 162, 0, 0],
            &[86, 255, 203, 163, 233, 173, 220, 140, 255, 144, 0, 0],
            &[67, 255, 205, 200, 195, 138, 252, 148, 255, 126, 0, 0],
            &[48, 255, 205, 236, 156, 101, 255, 181, 255, 108, 0, 0],
            &[29, 255, 216, 255, 119, 63, 255, 213, 255, 90, 0, 0],
            &[10, 255, 255, 255, 81, 25, 255, 255, 255, 72, 0, 0],
            &[0, 247, 255, 255, 43, 0, 243, 255, 255, 55, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'X' / 0x58
        'X' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 35, 35, 4, 0, 0, 0, 34, 35, 20, 0],
            &[0, 25, 243, 255, 109, 0, 0, 65, 255, 255, 64, 0],
            &[0, 0, 121, 255, 232, 11, 0, 197, 255, 172, 0, 0],
            &[0, 0, 7, 221, 255, 122, 73, 255, 247, 32, 0, 0],
            &[0, 0, 0, 81, 255, 239, 209, 255, 131, 0, 0, 0],
            &[0, 0, 0, 0, 189, 255, 255, 227, 10, 0, 0, 0],
            &[0, 0, 0, 0, 51, 255, 255, 101, 0, 0, 0, 0],
            &[0, 0, 0, 0, 118, 255, 255, 171, 0, 0, 0, 0],
            &[0, 0, 0, 24, 242, 255, 249, 255, 64, 0, 0, 0],
            &[0, 0, 0, 159, 255, 180, 140, 255, 209, 3, 0, 0],
            &[0, 0, 54, 254, 254, 47, 20, 243, 255, 104, 0, 0],
            &[0, 1, 200, 255, 165, 0, 0, 131, 255, 236, 17, 0],
            &[0, 94, 255, 250, 35, 0, 0, 16, 239, 255, 145, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Y' / 0x59
        'Y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 35, 34, 0, 0, 0, 0, 26, 35, 29, 0],
            &[0, 91, 255, 255, 53, 0, 0, 12, 241, 255, 142, 0],
            &[0, 3, 221, 255, 161, 0, 0, 105, 255, 249, 27, 0],
            &[0, 0, 103, 255, 248, 21, 0, 211, 255, 156, 0, 0],
            &[0, 0, 7, 230, 255, 122, 63, 255, 253, 37, 0, 0],
            &[0, 0, 0, 115, 255, 226, 174, 255, 169, 0, 0, 0],
            &[0, 0, 0, 12, 237, 255, 255, 255, 49, 0, 0, 0],
            &[0, 0, 0, 0, 128, 255, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Z' / 0x5a
        'Z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 34, 0, 0, 0],
            &[0, 55, 255, 255, 255, 255, 255, 255, 249, 0, 0, 0],
            &[0, 52, 243, 243, 243, 243, 253, 255, 232, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 247, 255, 103, 0, 0, 0],
            &[0, 0, 0, 0, 0, 172, 255, 206, 2, 0, 0, 0],
            &[0, 0, 0, 0, 65, 255, 254, 59, 0, 0, 0, 0],
            &[0, 0, 0, 3, 210, 255, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 106, 255, 242, 25, 0, 0, 0, 0, 0],
            &[0, 0, 18, 236, 255, 117, 0, 0, 0, 0, 0, 0],
            &[0, 0, 147, 255, 217, 6, 0, 0, 0, 0, 0, 0],
            &[0, 43, 251, 255, 103, 31, 31, 31, 31, 0, 0, 0],
            &[0, 155, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '[' / 0x5b
        '[' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 65, 65, 65, 65, 0, 0, 0, 0],
            &[0, 0, 0, 23, 255, 255, 255, 253, 0, 0, 0, 0],
            &[0, 0, 0, 23, 255, 255, 124, 120, 0, 0, 0, 0],
            &[0, 0, 0, 23, 255, 255, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 255, 255, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 255, 255, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 255, 255, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 255, 255, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 255, 255, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 255, 255, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 255, 255, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 255, 255, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 255, 255, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 255, 255, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 255, 255, 255, 252, 0, 0, 0, 0],
            &[0, 0, 0, 17, 192, 192, 192, 191, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '\' / 0x5c
        '\\' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 52, 45, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 247, 252, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 166, 255, 121, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 255, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 226, 255, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 255, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 254, 243, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 96, 255, 194, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 243, 255, 36, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 157, 255, 133, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 60, 255, 228, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 218, 255, 72, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 11, 17, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ']' / 0x5d
        ']' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 96, 96, 96, 27, 0, 0, 0, 0],
            &[0, 0, 0, 202, 255, 255, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 75, 95, 226, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 209, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 209, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 209, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 209, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 209, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 209, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 209, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 209, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 209, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 209, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 20, 25, 213, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 202, 255, 255, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 128, 162, 162, 162, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '^' / 0x5e
        '^' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 231, 225, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 255, 255, 108, 0, 0, 0, 0],
            &[0, 0, 0, 5, 226, 245, 244, 231, 10, 0, 0, 0],
            &[0, 0, 0, 100, 255, 144, 135, 255, 117, 0, 0, 0],
            &[0, 0, 3, 220, 250, 29, 18, 241, 236, 13, 0, 0],
            &[0, 0, 92, 255, 158, 0, 0, 128, 255, 126, 0, 0],
            &[0, 2, 214, 253, 39, 0, 0, 15, 237, 241, 18, 0],
            &[0, 24, 122, 97, 0, 0, 0, 0, 74, 122, 48, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '_' / 0x5f
        '_' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 44, 0],
            &[255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 51, 0],
            &[45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 9, 0],
        ]),
        // letter: '`' / 0x60
        '`' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 5, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 87, 252, 255, 81, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 86, 249, 230, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 66, 178, 100, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'a' / 0x61
        'a' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 74, 102, 100, 67, 4, 0, 0, 0],
            &[0, 0, 87, 247, 255, 255, 255, 255, 223, 56, 0, 0],
            &[0, 0, 23, 237, 181, 144, 151, 237, 255, 216, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 86, 255, 255, 16, 0],
            &[0, 0, 0, 75, 151, 180, 191, 211, 255, 255, 24, 0],
            &[0, 0, 143, 255, 255, 221, 188, 199, 255, 255, 24, 0],
            &[0, 12, 251, 255, 130, 0, 0, 75, 255, 255, 24, 0],
            &[0, 27, 255, 255, 97, 0, 7, 178, 255, 255, 24, 0],
            &[0, 1, 229, 255, 254, 214, 239, 247, 255, 255, 24, 0],
            &[0, 0, 67, 231, 255, 255, 196, 40, 236, 255, 24, 0],
            &[0, 0, 0, 2, 30, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'b' / 0x62
        'b' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 15, 235, 235, 82, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 255, 85, 26, 98, 103, 34, 0, 0, 0],
            &[0, 16, 255, 255, 148, 245, 255, 255, 249, 84, 0, 0],
            &[0, 16, 255, 255, 255, 198, 169, 248, 255, 236, 10, 0],
            &[0, 16, 255, 255, 184, 0, 0, 94, 255, 255, 78, 0],
            &[0, 16, 255, 255, 109, 0, 0, 15, 255, 255, 119, 0],
            &[0, 16, 255, 255, 96, 0, 0, 2, 254, 255, 129, 0],
            &[0, 16, 255, 255, 126, 0, 0, 32, 255, 255, 109, 0],
            &[0, 16, 255, 255, 223, 24, 1, 151, 255, 255, 55, 0],
            &[0, 16, 255, 255, 254, 251, 241, 255, 255, 203, 1, 0],
            &[0, 16, 255, 242, 58, 209, 255, 255, 203, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'c' / 0x63
        'c' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 75, 106, 101, 66, 6, 0, 0],
            &[0, 0, 0, 76, 237, 255, 255, 255, 255, 172, 0, 0],
            &[0, 0, 35, 246, 255, 247, 180, 176, 229, 86, 0, 0],
            &[0, 0, 134, 255, 252, 52, 0, 0, 0, 2, 0, 0],
            &[0, 0, 185, 255, 201, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 201, 255, 181, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 215, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 127, 255, 255, 113, 4, 3, 58, 79, 0, 0],
            &[0, 0, 24, 237, 255, 255, 248, 249, 255, 141, 0, 0],
            &[0, 0, 0, 43, 190, 253, 255, 255, 229, 88, 0, 0],
            &[0, 0, 0, 0, 0, 12, 35, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'd' / 0x64
        'd' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 35, 235, 235, 62, 0],
            &[0, 0, 0, 0, 0, 0, 0, 38, 255, 255, 67, 0],
            &[0, 0, 0, 0, 0, 0, 0, 38, 255, 255, 67, 0],
            &[0, 0, 0, 10, 85, 105, 55, 30, 255, 255, 67, 0],
            &[0, 0, 37, 228, 255, 255, 255, 152, 254, 255, 67, 0],
            &[0, 0, 189, 255, 255, 193, 164, 244, 255, 255, 67, 0],
            &[0, 23, 255, 255, 162, 0, 0, 95, 255, 255, 67, 0],
            &[0, 65, 255, 255, 72, 0, 0, 15, 255, 255, 67, 0],
            &[0, 77, 255, 255, 52, 0, 0, 0, 249, 255, 67, 0],
            &[0, 60, 255, 255, 76, 0, 0, 17, 255, 255, 67, 0],
            &[0, 14, 250, 255, 186, 9, 1, 137, 255, 255, 67, 0],
            &[0, 0, 162, 255, 255, 247, 240, 243, 255, 255, 67, 0],
            &[0, 0, 15, 183, 255, 255, 216, 49, 208, 255, 67, 0],
            &[0, 0, 0, 0, 23, 29, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'e' / 0x65
        'e' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 94, 106, 70, 4, 0, 0, 0],
            &[0, 0, 2, 146, 254, 255, 255, 255, 216, 36, 0, 0],
            &[0, 0, 112, 255, 254, 163, 129, 220, 255, 203, 0, 0],
            &[0, 0, 222, 255, 150, 0, 0, 30, 254, 255, 44, 0],
            &[0, 18, 255, 255, 190, 156, 156, 156, 246, 255, 80, 0],
            &[0, 30, 255, 255, 255, 255, 255, 255, 255, 255, 89, 0],
            &[0, 7, 252, 255, 102, 18, 18, 18, 18, 18, 6, 0],
            &[0, 0, 183, 255, 220, 50, 0, 0, 11, 79, 9, 0],
            &[0, 0, 44, 244, 255, 255, 242, 230, 253, 255, 19, 0],
            &[0, 0, 0, 41, 180, 250, 255, 255, 249, 179, 10, 0],
            &[0, 0, 0, 0, 0, 7, 32, 30, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'f' / 0x66
        'f' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 58, 192, 243, 250, 227, 139, 0, 0],
            &[0, 0, 0, 18, 242, 255, 255, 255, 255, 125, 0, 0],
            &[0, 0, 0, 80, 255, 255, 131, 29, 45, 20, 0, 0],
            &[0, 0, 14, 131, 255, 255, 91, 72, 72, 5, 0, 0],
            &[0, 185, 254, 255, 255, 255, 255, 255, 255, 19, 0, 0],
            &[0, 163, 193, 217, 255, 255, 199, 193, 193, 15, 0, 0],
            &[0, 0, 0, 96, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 255, 255, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'g' / 0x67
        'g' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 97, 110, 46, 0, 48, 72, 19, 0],
            &[0, 0, 44, 234, 255, 255, 254, 120, 233, 255, 67, 0],
            &[0, 0, 196, 255, 255, 185, 180, 249, 255, 255, 67, 0],
            &[0, 28, 255, 255, 150, 0, 0, 113, 255, 255, 67, 0],
            &[0, 68, 255, 255, 68, 0, 0, 39, 255, 255, 67, 0],
            &[0, 78, 255, 255, 52, 0, 0, 15, 255, 255, 67, 0],
            &[0, 60, 255, 255, 82, 0, 0, 36, 255, 255, 67, 0],
            &[0, 14, 249, 255, 195, 11, 1, 150, 255, 255, 67, 0],
            &[0, 0, 156, 255, 255, 248, 241, 251, 255, 255, 67, 0],
            &[0, 0, 11, 174, 254, 255, 225, 109, 255, 255, 67, 0],
            &[0, 0, 0, 0, 20, 34, 2, 70, 255, 255, 54, 0],
            &[0, 0, 100, 79, 24, 2, 31, 187, 255, 247, 15, 0],
            &[0, 0, 172, 255, 255, 254, 255, 255, 255, 136, 0, 0],
            &[0, 0, 0, 2, 23, 36, 19, 0, 0, 0, 0, 0],
        ]),
        // letter: 'h' / 0x68
        'h' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 15, 235, 235, 94, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 255, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 255, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 255, 100, 22, 94, 99, 45, 0, 0, 0],
            &[0, 16, 255, 255, 148, 241, 255, 255, 254, 123, 0, 0],
            &[0, 16, 255, 255, 255, 213, 164, 240, 255, 251, 27, 0],
            &[0, 16, 255, 255, 201, 2, 0, 80, 255, 255, 73, 0],
            &[0, 16, 255, 255, 125, 0, 0, 34, 255, 255, 89, 0],
            &[0, 16, 255, 255, 105, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'i' / 0x69
        'i' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 39, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 225, 255, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 211, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 103, 160, 62, 0, 0, 0, 0, 0],
            &[0, 37, 72, 72, 72, 72, 41, 0, 0, 0, 0, 0],
            &[0, 133, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0],
            &[0, 66, 148, 176, 248, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 149, 0, 0, 0, 0, 0],
            &[0, 153, 198, 239, 255, 255, 254, 222, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'j' / 0x6a
        'j' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 31, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 101, 255, 250, 51, 0, 0],
            &[0, 0, 0, 0, 0, 0, 143, 255, 255, 86, 0, 0],
            &[0, 0, 0, 0, 0, 0, 30, 153, 134, 7, 0, 0],
            &[0, 0, 0, 0, 58, 72, 72, 72, 72, 21, 0, 0],
            &[0, 0, 0, 0, 206, 255, 255, 255, 255, 74, 0, 0],
            &[0, 0, 0, 0, 105, 156, 195, 255, 255, 74, 0, 0],
            &[0, 0, 0, 0, 0, 0, 44, 255, 255, 74, 0, 0],
            &[0, 0, 0, 0, 0, 0, 44, 255, 255, 74, 0, 0],
            &[0, 0, 0, 0, 0, 0, 44, 255, 255, 74, 0, 0],
            &[0, 0, 0, 0, 0, 0, 44, 255, 255, 74, 0, 0],
            &[0, 0, 0, 0, 0, 0, 44, 255, 255, 74, 0, 0],
            &[0, 0, 0, 0, 0, 0, 44, 255, 255, 74, 0, 0],
            &[0, 0, 0, 0, 0, 0, 44, 255, 255, 74, 0, 0],
            &[0, 0, 0, 0, 0, 0, 68, 255, 255, 71, 0, 0],
            &[0, 0, 21, 33, 2, 21, 175, 255, 255, 42, 0, 0],
            &[0, 0, 81, 255, 254, 255, 255, 255, 207, 1, 0, 0],
            &[0, 0, 0, 2, 27, 37, 14, 0, 0, 0, 0, 0],
        ]),
        // letter: 'k' / 0x6b
        'k' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 230, 235, 114, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 250, 255, 124, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 250, 255, 124, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 250, 255, 124, 0, 0, 18, 72, 72, 43, 0, 0],
            &[0, 250, 255, 124, 0, 7, 196, 255, 232, 32, 0, 0],
            &[0, 250, 255, 124, 0, 163, 255, 243, 51, 0, 0, 0],
            &[0, 250, 255, 120, 123, 255, 251, 73, 0, 0, 0, 0],
            &[0, 250, 255, 172, 253, 255, 135, 0, 0, 0, 0, 0],
            &[0, 250, 255, 255, 255, 255, 226, 14, 0, 0, 0, 0],
            &[0, 250, 255, 245, 106, 231, 255, 155, 0, 0, 0, 0],
            &[0, 250, 255, 127, 0, 83, 255, 255, 71, 0, 0, 0],
            &[0, 250, 255, 124, 0, 0, 173, 255, 226, 14, 0, 0],
            &[0, 250, 255, 124, 0, 0, 24, 238, 255, 156, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'l' / 0x6c
        'l' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 235, 235, 235, 235, 128, 0, 0, 0, 0, 0],
            &[0, 64, 202, 249, 255, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 237, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 236, 255, 140, 0, 0, 0, 0, 0],
            &[0, 153, 198, 240, 255, 255, 254, 221, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'm' / 0x6d
        'm' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 72, 15, 57, 104, 22, 7, 94, 83, 1, 0],
            &[0, 125, 255, 150, 253, 255, 194, 147, 255, 255, 90, 0],
            &[0, 125, 255, 246, 175, 255, 255, 203, 218, 255, 149, 0],
            &[0, 125, 255, 174, 0, 250, 255, 60, 114, 255, 167, 0],
            &[0, 125, 255, 162, 0, 248, 255, 48, 111, 255, 176, 0],
            &[0, 125, 255, 162, 0, 248, 255, 48, 111, 255, 176, 0],
            &[0, 125, 255, 162, 0, 248, 255, 48, 111, 255, 176, 0],
            &[0, 125, 255, 162, 0, 248, 255, 48, 111, 255, 176, 0],
            &[0, 125, 255, 162, 0, 248, 255, 48, 111, 255, 176, 0],
            &[0, 125, 255, 162, 0, 248, 255, 48, 111, 255, 176, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'n' / 0x6e
        'n' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 72, 61, 0, 24, 93, 100, 48, 0, 0, 0],
            &[0, 16, 255, 252, 99, 247, 255, 255, 254, 127, 0, 0],
            &[0, 16, 255, 255, 255, 211, 164, 242, 255, 251, 28, 0],
            &[0, 16, 255, 255, 214, 7, 0, 84, 255, 255, 73, 0],
            &[0, 16, 255, 255, 132, 0, 0, 34, 255, 255, 89, 0],
            &[0, 16, 255, 255, 106, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'o' / 0x6f
        'o' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 96, 104, 62, 1, 0, 0, 0],
            &[0, 0, 8, 163, 255, 255, 255, 255, 205, 30, 0, 0],
            &[0, 0, 143, 255, 255, 179, 155, 245, 255, 201, 1, 0],
            &[0, 12, 246, 255, 155, 0, 0, 104, 255, 255, 59, 0],
            &[0, 59, 255, 255, 69, 0, 0, 23, 255, 255, 112, 0],
            &[0, 76, 255, 255, 54, 0, 0, 4, 255, 255, 127, 0],
            &[0, 54, 255, 255, 88, 0, 0, 35, 255, 255, 104, 0],
            &[0, 6, 235, 255, 205, 9, 0, 153, 255, 254, 37, 0],
            &[0, 0, 104, 255, 255, 238, 226, 255, 255, 155, 0, 0],
            &[0, 0, 0, 100, 228, 255, 255, 241, 135, 4, 0, 0],
            &[0, 0, 0, 0, 1, 29, 34, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'p' / 0x70
        'p' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 72, 61, 0, 22, 94, 103, 35, 0, 0, 0],
            &[0, 16, 255, 252, 95, 245, 255, 255, 249, 84, 0, 0],
            &[0, 16, 255, 255, 255, 201, 168, 247, 255, 236, 10, 0],
            &[0, 16, 255, 255, 186, 1, 0, 101, 255, 255, 77, 0],
            &[0, 16, 255, 255, 106, 0, 0, 26, 255, 255, 119, 0],
            &[0, 16, 255, 255, 95, 0, 0, 13, 255, 255, 128, 0],
            &[0, 16, 255, 255, 126, 0, 0, 38, 255, 255, 108, 0],
            &[0, 16, 255, 255, 223, 26, 0, 144, 255, 255, 52, 0],
            &[0, 16, 255, 255, 255, 252, 239, 255, 255, 199, 0, 0],
            &[0, 16, 255, 255, 118, 207, 255, 255, 199, 28, 0, 0],
            &[0, 16, 255, 255, 88, 0, 26, 25, 0, 0, 0, 0],
            &[0, 16, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 20, 20, 7, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'q' / 0x71
        'q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 92, 106, 49, 0, 46, 72, 19, 0],
            &[0, 0, 44, 233, 255, 255, 255, 142, 215, 255, 67, 0],
            &[0, 0, 197, 255, 255, 189, 174, 253, 255, 255, 67, 0],
            &[0, 25, 255, 255, 163, 0, 0, 115, 255, 255, 67, 0],
            &[0, 58, 255, 255, 81, 0, 0, 31, 255, 255, 67, 0],
            &[0, 77, 255, 255, 65, 0, 0, 10, 255, 255, 67, 0],
            &[0, 60, 255, 255, 92, 0, 0, 35, 255, 255, 67, 0],
            &[0, 14, 250, 255, 197, 11, 1, 152, 255, 255, 67, 0],
            &[0, 0, 162, 255, 255, 248, 242, 246, 255, 255, 67, 0],
            &[0, 0, 16, 184, 255, 255, 222, 91, 255, 255, 67, 0],
            &[0, 0, 0, 0, 23, 32, 1, 38, 255, 255, 67, 0],
            &[0, 0, 0, 0, 0, 0, 0, 38, 255, 255, 67, 0],
            &[0, 0, 0, 0, 0, 0, 0, 38, 255, 255, 67, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 20, 20, 5, 0],
        ]),
        // letter: 'r' / 0x72
        'r' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 72, 72, 71, 1, 8, 84, 109, 65, 0],
            &[0, 0, 215, 255, 255, 255, 61, 213, 255, 255, 184, 0],
            &[0, 0, 114, 170, 248, 255, 237, 255, 245, 247, 110, 0],
            &[0, 0, 0, 0, 227, 255, 255, 102, 1, 2, 6, 0],
            &[0, 0, 0, 0, 227, 255, 200, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 228, 255, 148, 0, 0, 0, 0, 0],
            &[0, 43, 179, 232, 255, 255, 254, 216, 162, 0, 0, 0],
            &[0, 77, 255, 255, 255, 255, 255, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 's' / 0x73
        's' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 102, 107, 83, 33, 0, 0, 0],
            &[0, 0, 12, 194, 255, 255, 255, 255, 255, 151, 0, 0],
            &[0, 0, 110, 255, 255, 182, 156, 187, 246, 83, 0, 0],
            &[0, 0, 136, 255, 254, 49, 0, 0, 9, 3, 0, 0],
            &[0, 0, 56, 247, 255, 253, 187, 85, 4, 0, 0, 0],
            &[0, 0, 0, 55, 194, 255, 255, 255, 215, 33, 0, 0],
            &[0, 0, 0, 0, 0, 49, 156, 253, 255, 171, 0, 0],
            &[0, 0, 92, 73, 9, 0, 3, 216, 255, 206, 0, 0],
            &[0, 0, 163, 255, 252, 233, 247, 255, 255, 153, 0, 0],
            &[0, 0, 116, 235, 255, 255, 255, 245, 159, 14, 0, 0],
            &[0, 0, 0, 1, 22, 38, 30, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 't' / 0x74
        't' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 152, 74, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 146, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 203, 255, 162, 72, 72, 70, 0, 0],
            &[0, 19, 231, 255, 255, 255, 255, 255, 255, 249, 0, 0],
            &[0, 19, 193, 193, 254, 255, 224, 193, 193, 189, 0, 0],
            &[0, 0, 0, 0, 249, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 249, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 249, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 248, 255, 131, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 231, 255, 199, 9, 0, 13, 2, 0],
            &[0, 0, 0, 0, 162, 255, 255, 247, 235, 254, 15, 0],
            &[0, 0, 0, 0, 20, 180, 253, 255, 255, 236, 12, 0],
            &[0, 0, 0, 0, 0, 0, 14, 38, 27, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'u' / 0x75
        'u' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 72, 72, 22, 0, 0, 14, 72, 72, 19, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 61, 255, 255, 67, 0],
            &[0, 37, 255, 255, 89, 0, 0, 90, 255, 255, 67, 0],
            &[0, 12, 255, 255, 167, 1, 11, 192, 255, 255, 67, 0],
            &[0, 0, 198, 255, 255, 238, 245, 244, 254, 255, 67, 0],
            &[0, 0, 39, 208, 255, 255, 214, 57, 198, 255, 67, 0],
            &[0, 0, 0, 0, 23, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'v' / 0x76
        'v' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 72, 72, 2, 0, 0, 0, 56, 72, 58, 0],
            &[0, 94, 255, 255, 61, 0, 0, 8, 244, 255, 145, 0],
            &[0, 11, 243, 255, 143, 0, 0, 76, 255, 255, 49, 0],
            &[0, 0, 160, 255, 224, 0, 0, 157, 255, 209, 0, 0],
            &[0, 0, 64, 255, 255, 50, 3, 234, 255, 114, 0, 0],
            &[0, 0, 1, 224, 255, 131, 62, 255, 251, 23, 0, 0],
            &[0, 0, 0, 130, 255, 207, 137, 255, 179, 0, 0, 0],
            &[0, 0, 0, 36, 255, 254, 207, 255, 83, 0, 0, 0],
            &[0, 0, 0, 0, 196, 255, 255, 237, 6, 0, 0, 0],
            &[0, 0, 0, 0, 101, 255, 255, 148, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'w' / 0x77
        'w' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[67, 72, 22, 0, 0, 0, 0, 4, 72, 72, 10, 0],
            &[214, 255, 93, 0, 0, 0, 0, 29, 255, 255, 14, 0],
            &[178, 255, 115, 9, 233, 235, 61, 51, 255, 234, 0, 0],
            &[142, 255, 138, 56, 255, 255, 116, 74, 255, 197, 0, 0],
            &[106, 255, 161, 107, 255, 246, 167, 97, 255, 161, 0, 0],
            &[70, 255, 184, 158, 240, 189, 217, 119, 255, 125, 0, 0],
            &[34, 255, 208, 204, 197, 146, 254, 151, 255, 89, 0, 0],
            &[3, 250, 233, 240, 151, 101, 255, 215, 255, 53, 0, 0],
            &[0, 217, 255, 255, 104, 55, 255, 255, 255, 17, 0, 0],
            &[0, 181, 255, 255, 57, 11, 253, 255, 237, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'x' / 0x78
        'x' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 29, 72, 72, 38, 0, 0, 13, 72, 72, 54, 0],
            &[0, 18, 229, 255, 222, 7, 0, 146, 255, 255, 79, 0],
            &[0, 0, 73, 255, 255, 121, 48, 252, 255, 156, 0, 0],
            &[0, 0, 0, 152, 255, 243, 205, 255, 220, 12, 0, 0],
            &[0, 0, 0, 11, 220, 255, 255, 252, 56, 0, 0, 0],
            &[0, 0, 0, 0, 150, 255, 255, 217, 4, 0, 0, 0],
            &[0, 0, 0, 61, 253, 255, 255, 255, 126, 0, 0, 0],
            &[0, 0, 13, 222, 255, 204, 138, 255, 250, 48, 0, 0],
            &[0, 0, 158, 255, 253, 54, 13, 231, 255, 210, 6, 0],
            &[0, 80, 255, 255, 153, 0, 0, 98, 255, 255, 133, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'y' / 0x79
        'y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 72, 72, 5, 0, 0, 0, 60, 72, 58, 0],
            &[0, 89, 255, 255, 73, 0, 0, 12, 249, 255, 147, 0],
            &[0, 7, 236, 255, 159, 0, 0, 81, 255, 255, 54, 0],
            &[0, 0, 142, 255, 238, 5, 0, 157, 255, 216, 0, 0],
            &[0, 0, 41, 255, 255, 74, 1, 231, 255, 124, 0, 0],
            &[0, 0, 0, 195, 255, 159, 53, 255, 254, 32, 0, 0],
            &[0, 0, 0, 93, 255, 237, 129, 255, 194, 0, 0, 0],
            &[0, 0, 0, 9, 239, 255, 225, 255, 101, 0, 0, 0],
            &[0, 0, 0, 0, 146, 255, 255, 247, 15, 0, 0, 0],
            &[0, 0, 0, 0, 45, 255, 255, 170, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 255, 255, 75, 0, 0, 0, 0],
            &[0, 0, 4, 31, 202, 255, 219, 2, 0, 0, 0, 0],
            &[0, 0, 253, 255, 255, 254, 75, 0, 0, 0, 0, 0],
            &[0, 0, 26, 34, 4, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'z' / 0x7a
        'z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 72, 72, 72, 72, 72, 72, 67, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 241, 0, 0, 0],
            &[0, 107, 167, 167, 167, 194, 255, 255, 198, 0, 0, 0],
            &[0, 0, 0, 0, 10, 207, 255, 230, 27, 0, 0, 0],
            &[0, 0, 0, 0, 168, 255, 247, 55, 0, 0, 0, 0],
            &[0, 0, 0, 119, 255, 255, 92, 0, 0, 0, 0, 0],
            &[0, 0, 73, 253, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 37, 239, 255, 180, 2, 0, 0, 0, 0, 0, 0],
            &[0, 202, 255, 255, 244, 240, 240, 240, 240, 22, 0, 0],
            &[0, 229, 255, 255, 255, 255, 255, 255, 255, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '{' / 0x7b
        '{' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 27, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 215, 255, 222, 0, 0, 0],
            &[0, 0, 0, 0, 0, 161, 255, 246, 150, 0, 0, 0],
            &[0, 0, 0, 0, 0, 190, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 255, 100, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 192, 255, 100, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 229, 255, 90, 0, 0, 0, 0],
            &[0, 0, 0, 119, 227, 255, 224, 20, 0, 0, 0, 0],
            &[0, 0, 0, 171, 255, 238, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 121, 254, 255, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 203, 255, 99, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 255, 100, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 255, 101, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 183, 255, 176, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 120, 255, 255, 222, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 101, 167, 163, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '|' / 0x7c
        '|' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 235, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 20, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '}' / 0x7d
        '}' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 34, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 171, 255, 234, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 119, 235, 255, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 255, 241, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 193, 255, 240, 157, 0, 0, 0],
            &[0, 0, 0, 0, 0, 65, 220, 255, 222, 0, 0, 0],
            &[0, 0, 0, 0, 16, 246, 255, 155, 57, 0, 0, 0],
            &[0, 0, 0, 0, 47, 255, 250, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 137, 255, 235, 0, 0, 0, 0, 0],
            &[0, 0, 0, 171, 255, 255, 170, 0, 0, 0, 0, 0],
            &[0, 0, 0, 126, 174, 119, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '~' / 0x7e
        '~' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 147, 197, 123, 3, 0, 53, 207, 2, 0, 0],
            &[0, 130, 225, 129, 241, 180, 11, 121, 217, 0, 0, 0],
            &[0, 202, 117, 0, 44, 222, 240, 252, 106, 0, 0, 0],
            &[0, 86, 37, 0, 0, 9, 83, 57, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ' ' / 0xa0
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¡' / 0xa1
        '¡' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 46, 245, 254, 95, 0, 0, 0, 0],
            &[0, 0, 0, 0, 98, 255, 255, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 177, 194, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 246, 246, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 255, 255, 60, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 255, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 255, 255, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 58, 255, 255, 105, 0, 0, 0, 0],
            &[0, 0, 0, 0, 69, 255, 255, 116, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 255, 255, 127, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 37, 37, 19, 0, 0, 0, 0],
        ]),
        // letter: '¢' / 0xa2
        '¢' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 234, 113, 0, 0, 0, 0, 0],
            &[0, 0, 6, 118, 205, 254, 250, 218, 143, 0, 0, 0],
            &[0, 1, 185, 255, 255, 254, 252, 255, 176, 0, 0, 0],
            &[0, 87, 255, 255, 113, 9, 4, 47, 46, 0, 0, 0],
            &[0, 165, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 197, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 154, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 144, 255, 225, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 48, 252, 255, 207, 112, 106, 146, 173, 0, 0, 0],
            &[0, 0, 104, 249, 255, 255, 255, 255, 197, 0, 0, 0],
            &[0, 0, 0, 28, 106, 245, 190, 103, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 234, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '£' / 0xa3
        '£' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 51, 68, 36, 0, 0, 0, 0],
            &[0, 0, 0, 76, 229, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 15, 243, 255, 250, 200, 233, 159, 0, 0, 0],
            &[0, 0, 69, 255, 255, 60, 0, 3, 18, 0, 0, 0],
            &[0, 0, 84, 255, 255, 13, 0, 0, 0, 0, 0, 0],
            &[0, 14, 99, 255, 255, 34, 22, 18, 0, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 209, 0, 0, 0, 0],
            &[0, 111, 200, 255, 255, 177, 173, 142, 0, 0, 0, 0],
            &[0, 0, 85, 255, 255, 12, 0, 0, 0, 0, 0, 0],
            &[0, 0, 110, 255, 237, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 220, 254, 116, 23, 23, 23, 23, 4, 0, 0],
            &[0, 159, 255, 255, 255, 255, 255, 255, 255, 46, 0, 0],
            &[0, 159, 255, 255, 255, 255, 255, 255, 255, 46, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¤' / 0xa4
        '¤' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 31, 0, 0, 0, 0, 19, 16, 0, 0, 0],
            &[0, 164, 231, 107, 177, 183, 109, 209, 206, 6, 0, 0],
            &[0, 73, 249, 255, 255, 254, 255, 255, 108, 0, 0, 0],
            &[0, 30, 252, 211, 30, 16, 171, 255, 74, 0, 0, 0],
            &[0, 83, 255, 115, 0, 0, 63, 255, 130, 0, 0, 0],
            &[0, 57, 255, 168, 0, 0, 117, 255, 104, 0, 0, 0],
            &[0, 20, 245, 255, 199, 185, 253, 255, 54, 0, 0, 0],
            &[0, 180, 254, 196, 254, 255, 199, 248, 216, 3, 0, 0],
            &[0, 33, 93, 0, 9, 12, 0, 72, 60, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¥' / 0xa5
        '¥' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 19, 35, 33, 0, 0, 0, 0, 26, 35, 26, 0],
            &[0, 65, 255, 255, 53, 0, 0, 15, 242, 255, 115, 0],
            &[0, 0, 193, 255, 167, 0, 0, 116, 255, 231, 9, 0],
            &[0, 0, 65, 255, 251, 30, 4, 226, 255, 111, 0, 0],
            &[0, 0, 0, 193, 255, 139, 88, 255, 228, 8, 0, 0],
            &[0, 0, 0, 65, 255, 240, 208, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 193, 255, 255, 225, 6, 0, 0, 0],
            &[0, 0, 126, 251, 253, 255, 255, 254, 251, 177, 0, 0],
            &[0, 0, 63, 126, 133, 255, 255, 159, 126, 89, 0, 0],
            &[0, 0, 45, 89, 98, 255, 255, 131, 89, 62, 0, 0],
            &[0, 0, 128, 255, 255, 255, 255, 255, 255, 180, 0, 0],
            &[0, 0, 14, 28, 41, 255, 255, 87, 28, 20, 0, 0],
            &[0, 0, 0, 0, 14, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¦' / 0xa6
        '¦' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 193, 226, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 184, 216, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 182, 214, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 232, 8, 0, 0, 0, 0, 0],
        ]),
        // letter: '§' / 0xa7
        '§' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 169, 233, 248, 222, 159, 28, 0, 0],
            &[0, 0, 0, 221, 255, 203, 155, 201, 236, 10, 0, 0],
            &[0, 0, 15, 254, 255, 44, 0, 0, 20, 0, 0, 0],
            &[0, 0, 0, 176, 255, 230, 113, 10, 0, 0, 0, 0],
            &[0, 0, 0, 61, 254, 255, 255, 234, 84, 0, 0, 0],
            &[0, 0, 15, 241, 243, 51, 131, 253, 252, 55, 0, 0],
            &[0, 0, 45, 255, 232, 11, 0, 168, 255, 111, 0, 0],
            &[0, 0, 3, 219, 255, 212, 84, 217, 253, 57, 0, 0],
            &[0, 0, 0, 25, 172, 254, 255, 255, 139, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 179, 255, 243, 27, 0, 0],
            &[0, 0, 10, 3, 0, 0, 0, 230, 255, 82, 0, 0],
            &[0, 0, 63, 226, 147, 108, 144, 255, 249, 34, 0, 0],
            &[0, 0, 35, 201, 255, 255, 255, 218, 76, 0, 0, 0],
            &[0, 0, 0, 0, 11, 35, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¨' / 0xa8
        '¨' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 153, 189, 30, 1, 159, 187, 26, 0, 0],
            &[0, 0, 6, 254, 255, 87, 14, 255, 255, 81, 0, 0],
            &[0, 0, 0, 90, 126, 10, 0, 93, 124, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '©' / 0xa9
        '©' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 77, 129, 134, 93, 11, 0, 0, 0],
            &[0, 0, 32, 209, 227, 154, 147, 208, 232, 58, 0, 0],
            &[0, 3, 208, 175, 60, 181, 210, 138, 124, 235, 22, 0],
            &[0, 74, 246, 33, 236, 169, 79, 82, 0, 204, 122, 0],
            &[0, 126, 191, 72, 255, 21, 0, 0, 0, 138, 176, 0],
            &[0, 129, 186, 77, 255, 11, 0, 0, 0, 137, 182, 0],
            &[0, 82, 239, 30, 244, 143, 59, 68, 0, 199, 139, 0],
            &[0, 8, 222, 148, 74, 204, 231, 137, 105, 247, 38, 0],
            &[0, 0, 51, 232, 200, 123, 118, 183, 245, 86, 0, 0],
            &[0, 0, 0, 17, 112, 160, 164, 122, 29, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ª' / 0xaa
        'ª' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 43, 60, 20, 0, 0, 0, 0],
            &[0, 0, 0, 1, 211, 255, 255, 245, 68, 0, 0, 0],
            &[0, 0, 0, 0, 53, 18, 30, 244, 174, 0, 0, 0],
            &[0, 0, 0, 14, 148, 219, 248, 254, 192, 0, 0, 0],
            &[0, 0, 0, 124, 255, 90, 12, 240, 192, 0, 0, 0],
            &[0, 0, 0, 115, 255, 162, 181, 243, 192, 0, 0, 0],
            &[0, 0, 0, 9, 136, 172, 106, 69, 120, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '«' / 0xab
        '«' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 41, 0, 0, 72, 12, 0, 0, 0],
            &[0, 0, 14, 219, 252, 71, 72, 254, 223, 8, 0, 0],
            &[0, 0, 171, 255, 196, 32, 236, 255, 108, 0, 0, 0],
            &[0, 110, 255, 243, 39, 198, 255, 187, 1, 0, 0, 0],
            &[0, 217, 255, 177, 55, 255, 255, 88, 0, 0, 0, 0],
            &[0, 55, 250, 255, 79, 141, 255, 228, 17, 0, 0, 0],
            &[0, 0, 111, 255, 234, 28, 198, 255, 166, 0, 0, 0],
            &[0, 0, 0, 172, 209, 50, 29, 234, 162, 8, 0, 0],
            &[0, 0, 0, 8, 2, 0, 0, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¬' / 0xac
        '¬' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 186, 230, 230, 230, 230, 230, 230, 193, 0, 0],
            &[0, 0, 186, 230, 230, 230, 230, 230, 255, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 250, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 250, 214, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 240, 206, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '®' / 0xae
        '®' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 76, 129, 134, 92, 10, 0, 0, 0],
            &[0, 0, 31, 208, 228, 154, 147, 208, 230, 56, 0, 0],
            &[0, 2, 205, 178, 137, 175, 159, 62, 124, 234, 21, 0],
            &[0, 71, 248, 19, 188, 159, 151, 240, 1, 205, 118, 0],
            &[0, 122, 195, 0, 188, 158, 160, 228, 0, 141, 172, 0],
            &[0, 125, 190, 0, 188, 204, 236, 99, 0, 141, 177, 0],
            &[0, 77, 241, 9, 188, 124, 139, 181, 0, 203, 134, 0],
            &[0, 5, 217, 152, 127, 83, 38, 169, 124, 247, 36, 0],
            &[0, 0, 46, 228, 204, 127, 123, 187, 244, 84, 0, 0],
            &[0, 0, 0, 14, 107, 155, 160, 119, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¯' / 0xaf
        '¯' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 65, 65, 65, 65, 44, 0, 0, 0],
            &[0, 0, 0, 119, 255, 255, 255, 255, 175, 0, 0, 0],
            &[0, 0, 0, 50, 109, 109, 109, 109, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '°' / 0xb0
        '°' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 54, 64, 6, 0, 0, 0, 0],
            &[0, 0, 0, 8, 190, 255, 255, 221, 32, 0, 0, 0],
            &[0, 0, 0, 109, 255, 117, 78, 248, 165, 0, 0, 0],
            &[0, 0, 0, 145, 252, 10, 0, 206, 201, 0, 0, 0],
            &[0, 0, 0, 84, 255, 178, 135, 254, 143, 0, 0, 0],
            &[0, 0, 0, 0, 130, 243, 252, 172, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '±' / 0xb1
        '±' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 132, 168, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 255, 8, 0, 0, 0, 0, 0],
            &[0, 105, 135, 135, 229, 255, 140, 135, 135, 1, 0, 0],
            &[0, 198, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 53, 69, 69, 215, 255, 75, 69, 69, 0, 0, 0],
            &[0, 0, 0, 0, 200, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 200, 255, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 93, 3, 0, 0, 0, 0, 0],
            &[0, 169, 209, 209, 209, 209, 209, 209, 209, 2, 0, 0],
            &[0, 207, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '²' / 0xb2
        '²' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 111, 120, 48, 0, 0, 0, 0],
            &[0, 0, 0, 50, 245, 255, 255, 251, 46, 0, 0, 0],
            &[0, 0, 0, 2, 118, 20, 109, 255, 108, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 130, 255, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 251, 170, 0, 0, 0, 0],
            &[0, 0, 0, 0, 88, 251, 175, 6, 0, 0, 0, 0],
            &[0, 0, 0, 65, 253, 246, 145, 141, 115, 0, 0, 0],
            &[0, 0, 0, 90, 237, 237, 237, 237, 194, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '³' / 0xb3
        '³' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 101, 113, 38, 0, 0, 0, 0],
            &[0, 0, 0, 52, 251, 248, 254, 248, 42, 0, 0, 0],
            &[0, 0, 0, 0, 82, 4, 94, 255, 90, 0, 0, 0],
            &[0, 0, 0, 0, 22, 117, 203, 217, 18, 0, 0, 0],
            &[0, 0, 0, 0, 47, 231, 255, 180, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 63, 255, 141, 0, 0, 0],
            &[0, 0, 0, 101, 161, 107, 161, 255, 136, 0, 0, 0],
            &[0, 0, 0, 64, 220, 254, 249, 177, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '´' / 0xb4
        '´' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 5, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 211, 255, 195, 8, 0, 0, 0],
            &[0, 0, 0, 0, 130, 255, 188, 12, 0, 0, 0, 0],
            &[0, 0, 0, 19, 175, 142, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'µ' / 0xb5
        'µ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 72, 72, 22, 0, 0, 14, 72, 72, 19, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 61, 255, 255, 67, 0],
            &[0, 70, 255, 255, 159, 0, 0, 90, 255, 255, 67, 0],
            &[0, 50, 255, 255, 248, 1, 11, 192, 255, 255, 67, 0],
            &[0, 38, 255, 255, 255, 238, 245, 244, 254, 255, 67, 0],
            &[0, 38, 255, 255, 255, 255, 214, 57, 198, 255, 67, 0],
            &[0, 38, 255, 255, 103, 25, 0, 0, 0, 0, 0, 0],
            &[0, 38, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 38, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 20, 20, 6, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¶' / 0xb6
        '¶' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 112, 201, 230, 235, 235, 235, 235, 82, 0],
            &[0, 0, 147, 255, 255, 255, 255, 235, 129, 250, 89, 0],
            &[0, 18, 251, 255, 255, 255, 255, 213, 0, 245, 89, 0],
            &[0, 63, 255, 255, 255, 255, 255, 213, 0, 245, 89, 0],
            &[0, 76, 255, 255, 255, 255, 255, 213, 0, 245, 89, 0],
            &[0, 55, 255, 255, 255, 255, 255, 213, 0, 245, 89, 0],
            &[0, 5, 237, 255, 255, 255, 255, 213, 0, 245, 89, 0],
            &[0, 0, 85, 249, 255, 255, 255, 213, 0, 245, 89, 0],
            &[0, 0, 0, 32, 105, 125, 175, 213, 0, 245, 89, 0],
            &[0, 0, 0, 0, 0, 0, 117, 213, 0, 245, 89, 0],
            &[0, 0, 0, 0, 0, 0, 117, 213, 0, 245, 89, 0],
            &[0, 0, 0, 0, 0, 0, 117, 213, 0, 245, 89, 0],
            &[0, 0, 0, 0, 0, 0, 117, 213, 0, 245, 89, 0],
            &[0, 0, 0, 0, 0, 0, 117, 213, 0, 245, 89, 0],
            &[0, 0, 0, 0, 0, 0, 117, 213, 0, 245, 89, 0],
            &[0, 0, 0, 0, 0, 0, 18, 33, 0, 39, 14, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '·' / 0xb7
        '·' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 85, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 255, 255, 137, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 113, 127, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¸' / 0xb8
        '¸' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 186, 188, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 128, 253, 184, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 63, 241, 230, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 15, 1, 0, 0, 0, 0, 0],
        ]),
        // letter: '¹' / 0xb9
        '¹' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 89, 21, 0, 0, 0, 0],
            &[0, 0, 0, 1, 104, 241, 255, 61, 0, 0, 0, 0],
            &[0, 0, 0, 79, 255, 210, 255, 61, 0, 0, 0, 0],
            &[0, 0, 0, 1, 56, 96, 255, 61, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 100, 255, 61, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 100, 255, 61, 0, 0, 0, 0],
            &[0, 0, 0, 3, 55, 178, 255, 136, 47, 0, 0, 0],
            &[0, 0, 0, 22, 237, 255, 255, 255, 214, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'º' / 0xba
        'º' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 42, 55, 3, 0, 0, 0, 0],
            &[0, 0, 0, 3, 181, 255, 255, 211, 18, 0, 0, 0],
            &[0, 0, 0, 86, 255, 106, 64, 255, 134, 0, 0, 0],
            &[0, 0, 0, 130, 255, 34, 0, 243, 180, 0, 0, 0],
            &[0, 0, 0, 112, 255, 58, 16, 255, 159, 0, 0, 0],
            &[0, 0, 0, 29, 245, 210, 193, 255, 68, 0, 0, 0],
            &[0, 0, 0, 0, 48, 158, 166, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '»' / 0xbb
        '»' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 78, 1, 0, 22, 61, 0, 0, 0, 0, 0],
            &[0, 179, 255, 122, 29, 242, 244, 41, 0, 0, 0, 0],
            &[0, 61, 253, 252, 63, 150, 255, 212, 10, 0, 0, 0],
            &[0, 0, 138, 255, 230, 33, 217, 255, 161, 0, 0, 0],
            &[0, 0, 38, 255, 255, 105, 126, 255, 247, 21, 0, 0],
            &[0, 1, 191, 255, 190, 40, 245, 255, 100, 0, 0, 0],
            &[0, 112, 255, 231, 27, 200, 255, 162, 0, 0, 0, 0],
            &[0, 129, 243, 64, 25, 186, 213, 11, 0, 0, 0, 0],
            &[0, 0, 11, 0, 0, 0, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¼' / 0xbc
        '¼' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 35, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 205, 255, 54, 0, 0, 0, 0, 0, 0, 0],
            &[0, 143, 185, 255, 54, 0, 0, 0, 54, 125, 54, 0],
            &[0, 2, 35, 255, 54, 0, 0, 23, 231, 213, 10, 0],
            &[0, 0, 37, 255, 54, 0, 2, 189, 244, 42, 0, 0],
            &[0, 60, 184, 255, 203, 64, 129, 255, 94, 0, 0, 0],
            &[0, 64, 163, 255, 172, 138, 253, 157, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 233, 210, 9, 140, 171, 0, 0],
            &[0, 0, 0, 3, 193, 243, 39, 87, 252, 241, 0, 0],
            &[0, 0, 0, 135, 255, 90, 21, 231, 159, 241, 0, 0],
            &[0, 0, 74, 254, 153, 0, 169, 171, 139, 243, 40, 0],
            &[0, 9, 174, 173, 8, 0, 221, 242, 248, 255, 199, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 111, 241, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '½' / 0xbd
        '½' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 35, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 59, 205, 255, 54, 0, 0, 0, 0, 0, 0, 0],
            &[0, 143, 185, 255, 54, 0, 0, 0, 54, 125, 54, 0],
            &[0, 2, 35, 255, 54, 0, 0, 23, 231, 213, 10, 0],
            &[0, 0, 37, 255, 54, 0, 2, 189, 244, 42, 0, 0],
            &[0, 60, 184, 255, 203, 64, 129, 255, 94, 0, 0, 0],
            &[0, 64, 163, 255, 172, 138, 253, 157, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 233, 214, 128, 198, 186, 41, 0],
            &[0, 0, 0, 3, 193, 243, 46, 169, 110, 232, 164, 0],
            &[0, 0, 0, 135, 255, 90, 0, 0, 16, 231, 133, 0],
            &[0, 0, 74, 254, 153, 0, 0, 14, 200, 208, 13, 0],
            &[0, 9, 174, 173, 8, 0, 9, 196, 234, 81, 57, 0],
            &[0, 0, 0, 0, 0, 0, 38, 255, 255, 255, 237, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¾' / 0xbe
        '¾' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 24, 64, 27, 0, 0, 0, 0, 0, 0, 0],
            &[0, 56, 249, 238, 250, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 38, 17, 253, 91, 0, 0, 54, 125, 54, 0],
            &[0, 0, 113, 249, 217, 11, 0, 23, 231, 213, 10, 0],
            &[0, 1, 22, 61, 237, 122, 2, 189, 244, 42, 0, 0],
            &[0, 92, 181, 162, 253, 114, 129, 255, 94, 0, 0, 0],
            &[0, 29, 135, 156, 105, 73, 253, 157, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 233, 210, 9, 140, 171, 0, 0],
            &[0, 0, 0, 3, 193, 243, 39, 87, 252, 241, 0, 0],
            &[0, 0, 0, 135, 255, 90, 21, 231, 159, 241, 0, 0],
            &[0, 0, 74, 254, 153, 0, 169, 171, 139, 243, 40, 0],
            &[0, 9, 174, 173, 8, 0, 221, 242, 248, 255, 199, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 111, 241, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¿' / 0xbf
        '¿' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 241, 255, 107, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 255, 167, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 172, 196, 58, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 232, 246, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 86, 255, 254, 30, 0, 0, 0, 0],
            &[0, 0, 0, 121, 253, 255, 136, 0, 0, 0, 0, 0],
            &[0, 0, 120, 255, 253, 111, 0, 0, 0, 0, 0, 0],
            &[0, 0, 235, 255, 143, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 253, 255, 117, 0, 0, 33, 77, 0, 0, 0],
            &[0, 0, 199, 255, 253, 207, 215, 253, 245, 22, 0, 0],
            &[0, 0, 37, 210, 255, 255, 255, 252, 181, 37, 0, 0],
            &[0, 0, 0, 1, 46, 75, 62, 16, 0, 0, 0, 0],
        ]),
        // letter: 'À' / 0xc0
        'À' => Some(&[
            &[0, 0, 110, 224, 223, 42, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 141, 255, 199, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 215, 92, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 48, 48, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 171, 255, 255, 218, 0, 0, 0, 0, 0],
            &[0, 0, 3, 238, 255, 250, 255, 33, 0, 0, 0, 0],
            &[0, 0, 56, 255, 233, 183, 255, 104, 0, 0, 0, 0],
            &[0, 0, 126, 255, 175, 122, 255, 174, 0, 0, 0, 0],
            &[0, 0, 196, 255, 112, 62, 255, 241, 4, 0, 0, 0],
            &[0, 15, 251, 255, 48, 7, 248, 255, 60, 0, 0, 0],
            &[0, 81, 255, 246, 14, 12, 205, 255, 130, 0, 0, 0],
            &[0, 151, 255, 255, 255, 255, 255, 255, 201, 0, 0, 0],
            &[0, 222, 255, 255, 253, 253, 255, 255, 253, 18, 0, 0],
            &[36, 255, 255, 48, 0, 0, 13, 252, 255, 86, 0, 0],
            &[106, 255, 238, 2, 0, 0, 0, 203, 255, 157, 0, 0],
            &[177, 255, 177, 0, 0, 0, 0, 142, 255, 227, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Á' / 0xc1
        'Á' => Some(&[
            &[0, 0, 0, 0, 0, 147, 224, 203, 23, 0, 0, 0],
            &[0, 0, 0, 0, 71, 255, 230, 44, 0, 0, 0, 0],
            &[0, 0, 0, 5, 190, 194, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 48, 48, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 171, 255, 255, 218, 0, 0, 0, 0, 0],
            &[0, 0, 3, 238, 255, 250, 255, 33, 0, 0, 0, 0],
            &[0, 0, 56, 255, 233, 183, 255, 104, 0, 0, 0, 0],
            &[0, 0, 126, 255, 175, 122, 255, 174, 0, 0, 0, 0],
            &[0, 0, 196, 255, 112, 62, 255, 241, 4, 0, 0, 0],
            &[0, 15, 251, 255, 48, 7, 248, 255, 60, 0, 0, 0],
            &[0, 81, 255, 246, 14, 12, 205, 255, 130, 0, 0, 0],
            &[0, 151, 255, 255, 255, 255, 255, 255, 201, 0, 0, 0],
            &[0, 222, 255, 255, 253, 253, 255, 255, 253, 18, 0, 0],
            &[36, 255, 255, 48, 0, 0, 13, 252, 255, 86, 0, 0],
            &[106, 255, 238, 2, 0, 0, 0, 203, 255, 157, 0, 0],
            &[177, 255, 177, 0, 0, 0, 0, 142, 255, 227, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Â' / 0xc2
        'Â' => Some(&[
            &[0, 0, 0, 95, 224, 224, 145, 0, 0, 0, 0, 0],
            &[0, 0, 37, 242, 224, 197, 255, 82, 0, 0, 0, 0],
            &[0, 0, 175, 190, 29, 11, 163, 201, 18, 0, 0, 0],
            &[0, 0, 0, 24, 48, 48, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 171, 255, 255, 218, 0, 0, 0, 0, 0],
            &[0, 0, 3, 238, 255, 250, 255, 33, 0, 0, 0, 0],
            &[0, 0, 56, 255, 233, 183, 255, 104, 0, 0, 0, 0],
            &[0, 0, 126, 255, 175, 122, 255, 174, 0, 0, 0, 0],
            &[0, 0, 196, 255, 112, 62, 255, 241, 4, 0, 0, 0],
            &[0, 15, 251, 255, 48, 7, 248, 255, 60, 0, 0, 0],
            &[0, 81, 255, 246, 14, 12, 205, 255, 130, 0, 0, 0],
            &[0, 151, 255, 255, 255, 255, 255, 255, 201, 0, 0, 0],
            &[0, 222, 255, 255, 253, 253, 255, 255, 253, 18, 0, 0],
            &[36, 255, 255, 48, 0, 0, 13, 252, 255, 86, 0, 0],
            &[106, 255, 238, 2, 0, 0, 0, 203, 255, 157, 0, 0],
            &[177, 255, 177, 0, 0, 0, 0, 142, 255, 227, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ã' / 0xc3
        'Ã' => Some(&[
            &[0, 0, 38, 157, 132, 28, 8, 177, 49, 0, 0, 0],
            &[0, 0, 206, 255, 255, 251, 233, 249, 24, 0, 0, 0],
            &[0, 13, 219, 61, 45, 153, 202, 96, 0, 0, 0, 0],
            &[0, 0, 0, 24, 48, 48, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 171, 255, 255, 218, 0, 0, 0, 0, 0],
            &[0, 0, 3, 238, 255, 250, 255, 33, 0, 0, 0, 0],
            &[0, 0, 56, 255, 233, 183, 255, 104, 0, 0, 0, 0],
            &[0, 0, 126, 255, 175, 122, 255, 174, 0, 0, 0, 0],
            &[0, 0, 196, 255, 112, 62, 255, 241, 4, 0, 0, 0],
            &[0, 15, 251, 255, 48, 7, 248, 255, 60, 0, 0, 0],
            &[0, 81, 255, 246, 14, 12, 205, 255, 130, 0, 0, 0],
            &[0, 151, 255, 255, 255, 255, 255, 255, 201, 0, 0, 0],
            &[0, 222, 255, 255, 253, 253, 255, 255, 253, 18, 0, 0],
            &[36, 255, 255, 48, 0, 0, 13, 252, 255, 86, 0, 0],
            &[106, 255, 238, 2, 0, 0, 0, 203, 255, 157, 0, 0],
            &[177, 255, 177, 0, 0, 0, 0, 142, 255, 227, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ä' / 0xc4
        'Ä' => Some(&[
            &[0, 0, 125, 148, 13, 1, 130, 146, 11, 0, 0, 0],
            &[0, 19, 255, 255, 75, 28, 255, 255, 69, 0, 0, 0],
            &[0, 0, 135, 159, 14, 1, 138, 157, 12, 0, 0, 0],
            &[0, 0, 0, 24, 48, 48, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 171, 255, 255, 218, 0, 0, 0, 0, 0],
            &[0, 0, 3, 238, 255, 250, 255, 33, 0, 0, 0, 0],
            &[0, 0, 56, 255, 233, 183, 255, 104, 0, 0, 0, 0],
            &[0, 0, 126, 255, 175, 122, 255, 174, 0, 0, 0, 0],
            &[0, 0, 196, 255, 112, 62, 255, 241, 4, 0, 0, 0],
            &[0, 15, 251, 255, 48, 7, 248, 255, 60, 0, 0, 0],
            &[0, 81, 255, 246, 14, 12, 205, 255, 130, 0, 0, 0],
            &[0, 151, 255, 255, 255, 255, 255, 255, 201, 0, 0, 0],
            &[0, 222, 255, 255, 253, 253, 255, 255, 253, 18, 0, 0],
            &[36, 255, 255, 48, 0, 0, 13, 252, 255, 86, 0, 0],
            &[106, 255, 238, 2, 0, 0, 0, 203, 255, 157, 0, 0],
            &[177, 255, 177, 0, 0, 0, 0, 142, 255, 227, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Å' / 0xc5
        'Å' => Some(&[
            &[0, 0, 0, 222, 136, 102, 254, 14, 0, 0, 0, 0],
            &[0, 0, 0, 206, 174, 150, 244, 7, 0, 0, 0, 0],
            &[0, 0, 0, 50, 187, 196, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 48, 48, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 171, 255, 255, 218, 0, 0, 0, 0, 0],
            &[0, 0, 3, 238, 255, 250, 255, 33, 0, 0, 0, 0],
            &[0, 0, 56, 255, 233, 183, 255, 104, 0, 0, 0, 0],
            &[0, 0, 126, 255, 175, 122, 255, 174, 0, 0, 0, 0],
            &[0, 0, 196, 255, 112, 62, 255, 241, 4, 0, 0, 0],
            &[0, 15, 251, 255, 48, 7, 248, 255, 60, 0, 0, 0],
            &[0, 81, 255, 246, 14, 12, 205, 255, 130, 0, 0, 0],
            &[0, 151, 255, 255, 255, 255, 255, 255, 201, 0, 0, 0],
            &[0, 222, 255, 255, 253, 253, 255, 255, 253, 18, 0, 0],
            &[36, 255, 255, 48, 0, 0, 13, 252, 255, 86, 0, 0],
            &[106, 255, 238, 2, 0, 0, 0, 203, 255, 157, 0, 0],
            &[177, 255, 177, 0, 0, 0, 0, 142, 255, 227, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 90, 243, 249, 121, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Æ' / 0xc6
        'Æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 35, 35, 35, 35, 35, 24, 0],
            &[0, 0, 0, 23, 255, 255, 255, 255, 255, 255, 176, 0],
            &[0, 0, 0, 84, 255, 253, 253, 255, 250, 248, 171, 0],
            &[0, 0, 0, 145, 255, 138, 191, 255, 79, 0, 0, 0],
            &[0, 0, 0, 206, 255, 82, 191, 255, 79, 0, 0, 0],
            &[0, 0, 14, 252, 255, 25, 191, 255, 143, 92, 35, 0],
            &[0, 0, 72, 255, 225, 0, 191, 255, 255, 255, 98, 0],
            &[0, 0, 133, 255, 194, 25, 209, 255, 211, 191, 73, 0],
            &[0, 0, 194, 255, 255, 255, 255, 255, 79, 0, 0, 0],
            &[0, 7, 247, 255, 255, 255, 255, 255, 79, 0, 0, 0],
            &[0, 60, 255, 249, 8, 2, 192, 255, 98, 27, 19, 0],
            &[0, 121, 255, 199, 0, 0, 191, 255, 255, 255, 176, 0],
            &[0, 181, 255, 142, 0, 0, 191, 255, 255, 255, 176, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ç' / 0xc7
        'Ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 53, 72, 52, 11, 0, 0],
            &[0, 0, 0, 5, 133, 242, 255, 255, 255, 243, 75, 0],
            &[0, 0, 0, 176, 255, 255, 246, 224, 247, 250, 23, 0],
            &[0, 0, 90, 255, 255, 138, 9, 0, 7, 58, 0, 0],
            &[0, 0, 194, 255, 215, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 249, 255, 131, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 255, 255, 103, 0, 0, 0, 0, 0, 0, 0],
            &[0, 31, 255, 255, 96, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 213, 255, 201, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 124, 255, 255, 148, 28, 5, 38, 120, 7, 0],
            &[0, 0, 10, 211, 255, 255, 255, 255, 255, 255, 11, 0],
            &[0, 0, 0, 17, 155, 246, 255, 255, 248, 181, 6, 0],
            &[0, 0, 0, 0, 0, 39, 255, 119, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 186, 254, 81, 0, 0, 0],
            &[0, 0, 0, 0, 1, 58, 133, 255, 126, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 10, 0, 0, 0, 0, 0],
        ]),
        // letter: 'È' / 0xc8
        'È' => Some(&[
            &[0, 0, 87, 224, 224, 63, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 218, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 215, 114, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 35, 35, 35, 35, 35, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 252, 252, 252, 252, 252, 2, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 213, 92, 92, 92, 92, 69, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 193, 0, 0, 0],
            &[0, 163, 255, 240, 195, 195, 195, 195, 147, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 197, 31, 31, 31, 31, 31, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'É' / 0xc9
        'É' => Some(&[
            &[0, 0, 0, 0, 0, 124, 224, 212, 36, 0, 0, 0],
            &[0, 0, 0, 0, 49, 250, 240, 60, 0, 0, 0, 0],
            &[0, 0, 0, 0, 173, 202, 45, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 35, 35, 35, 35, 35, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 252, 252, 252, 252, 252, 2, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 213, 92, 92, 92, 92, 69, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 193, 0, 0, 0],
            &[0, 163, 255, 240, 195, 195, 195, 195, 147, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 197, 31, 31, 31, 31, 31, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ê' / 0xca
        'Ê' => Some(&[
            &[0, 0, 0, 72, 224, 224, 167, 0, 0, 0, 0, 0],
            &[0, 0, 23, 230, 236, 185, 255, 107, 0, 0, 0, 0],
            &[0, 0, 154, 199, 42, 4, 148, 209, 33, 0, 0, 0],
            &[0, 22, 35, 35, 35, 35, 35, 35, 35, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 252, 252, 252, 252, 252, 2, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 213, 92, 92, 92, 92, 69, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 193, 0, 0, 0],
            &[0, 163, 255, 240, 195, 195, 195, 195, 147, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 197, 31, 31, 31, 31, 31, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ë' / 0xcb
        'Ë' => Some(&[
            &[0, 0, 108, 155, 24, 0, 113, 153, 21, 0, 0, 0],
            &[0, 0, 248, 255, 101, 4, 254, 255, 95, 0, 0, 0],
            &[0, 0, 117, 166, 26, 0, 121, 164, 24, 0, 0, 0],
            &[0, 22, 35, 35, 35, 35, 35, 35, 35, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 252, 252, 252, 252, 252, 2, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 213, 92, 92, 92, 92, 69, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 193, 0, 0, 0],
            &[0, 163, 255, 240, 195, 195, 195, 195, 147, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 197, 31, 31, 31, 31, 31, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ì' / 0xcc
        'Ì' => Some(&[
            &[0, 0, 0, 110, 224, 223, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 141, 255, 199, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 115, 215, 92, 0, 0, 0, 0],
            &[0, 0, 16, 35, 35, 35, 35, 35, 35, 23, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 70, 172, 206, 255, 255, 216, 177, 102, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 87, 207, 237, 255, 255, 242, 212, 126, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Í' / 0xcd
        'Í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 147, 224, 203, 23, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 230, 44, 0, 0, 0],
            &[0, 0, 0, 0, 5, 190, 194, 32, 0, 0, 0, 0],
            &[0, 0, 16, 35, 35, 35, 35, 35, 35, 23, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 70, 172, 206, 255, 255, 216, 177, 102, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 87, 207, 237, 255, 255, 242, 212, 126, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Î' / 0xce
        'Î' => Some(&[
            &[0, 0, 0, 0, 95, 224, 224, 145, 0, 0, 0, 0],
            &[0, 0, 0, 37, 242, 224, 197, 255, 82, 0, 0, 0],
            &[0, 0, 0, 175, 190, 29, 11, 163, 201, 18, 0, 0],
            &[0, 0, 16, 35, 35, 35, 35, 35, 35, 23, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 70, 172, 206, 255, 255, 216, 177, 102, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 87, 207, 237, 255, 255, 242, 212, 126, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ï' / 0xcf
        'Ï' => Some(&[
            &[0, 0, 0, 125, 148, 13, 1, 130, 146, 11, 0, 0],
            &[0, 0, 19, 255, 255, 75, 28, 255, 255, 69, 0, 0],
            &[0, 0, 0, 135, 159, 14, 1, 138, 157, 12, 0, 0],
            &[0, 0, 16, 35, 35, 35, 35, 35, 35, 23, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 70, 172, 206, 255, 255, 216, 177, 102, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 87, 207, 237, 255, 255, 242, 212, 126, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ð' / 0xd0
        'Ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 35, 35, 35, 31, 7, 0, 0, 0, 0, 0],
            &[0, 60, 255, 255, 255, 255, 254, 196, 68, 0, 0, 0],
            &[0, 60, 255, 255, 246, 248, 255, 255, 252, 88, 0, 0],
            &[0, 60, 255, 255, 50, 0, 44, 210, 255, 238, 15, 0],
            &[0, 60, 255, 255, 50, 0, 0, 53, 255, 255, 93, 0],
            &[0, 65, 255, 255, 55, 5, 0, 0, 236, 255, 146, 0],
            &[0, 255, 255, 255, 255, 255, 0, 0, 213, 255, 168, 0],
            &[0, 255, 255, 255, 255, 255, 0, 0, 223, 255, 161, 0],
            &[0, 64, 255, 255, 55, 4, 0, 6, 248, 255, 135, 0],
            &[0, 60, 255, 255, 50, 0, 0, 92, 255, 255, 66, 0],
            &[0, 60, 255, 255, 68, 38, 102, 243, 255, 208, 3, 0],
            &[0, 60, 255, 255, 255, 255, 255, 255, 232, 45, 0, 0],
            &[0, 60, 255, 255, 255, 249, 220, 144, 27, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ñ' / 0xd1
        'Ñ' => Some(&[
            &[0, 0, 0, 27, 154, 139, 36, 2, 168, 64, 0, 0],
            &[0, 0, 0, 184, 255, 255, 253, 232, 254, 41, 0, 0],
            &[0, 0, 1, 212, 79, 37, 145, 202, 112, 0, 0, 0],
            &[0, 8, 35, 35, 35, 4, 0, 0, 26, 35, 15, 0],
            &[0, 60, 255, 255, 255, 82, 0, 0, 193, 255, 111, 0],
            &[0, 60, 255, 255, 255, 166, 0, 0, 193, 255, 111, 0],
            &[0, 60, 255, 222, 248, 242, 7, 0, 193, 255, 111, 0],
            &[0, 60, 255, 221, 179, 255, 77, 0, 193, 255, 111, 0],
            &[0, 60, 255, 234, 96, 255, 161, 0, 193, 255, 111, 0],
            &[0, 60, 255, 243, 18, 251, 239, 5, 193, 255, 111, 0],
            &[0, 60, 255, 245, 0, 187, 255, 72, 191, 255, 111, 0],
            &[0, 60, 255, 245, 0, 104, 255, 156, 186, 255, 111, 0],
            &[0, 60, 255, 245, 0, 23, 253, 236, 182, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 194, 255, 233, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 111, 255, 255, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 29, 254, 255, 255, 111, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ò' / 0xd2
        'Ò' => Some(&[
            &[0, 0, 0, 110, 224, 223, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 141, 255, 199, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 115, 215, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 67, 77, 34, 0, 0, 0, 0],
            &[0, 0, 0, 135, 249, 255, 255, 255, 175, 13, 0, 0],
            &[0, 0, 115, 255, 255, 232, 228, 255, 255, 170, 0, 0],
            &[0, 6, 236, 255, 187, 6, 0, 148, 255, 255, 41, 0],
            &[0, 58, 255, 255, 71, 0, 0, 28, 255, 255, 114, 0],
            &[0, 100, 255, 255, 24, 0, 0, 0, 232, 255, 154, 0],
            &[0, 118, 255, 255, 6, 0, 0, 0, 214, 255, 171, 0],
            &[0, 118, 255, 255, 8, 0, 0, 0, 213, 255, 169, 0],
            &[0, 98, 255, 255, 29, 0, 0, 0, 235, 255, 149, 0],
            &[0, 53, 255, 255, 84, 0, 0, 36, 255, 255, 104, 0],
            &[0, 2, 227, 255, 210, 27, 12, 176, 255, 251, 28, 0],
            &[0, 0, 90, 255, 255, 255, 253, 255, 255, 140, 0, 0],
            &[0, 0, 0, 95, 228, 255, 255, 240, 131, 3, 0, 0],
            &[0, 0, 0, 0, 1, 28, 33, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ó' / 0xd3
        'Ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 147, 224, 203, 23, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 230, 44, 0, 0, 0],
            &[0, 0, 0, 0, 5, 190, 194, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 67, 77, 34, 0, 0, 0, 0],
            &[0, 0, 0, 135, 249, 255, 255, 255, 175, 13, 0, 0],
            &[0, 0, 115, 255, 255, 232, 228, 255, 255, 170, 0, 0],
            &[0, 6, 236, 255, 187, 6, 0, 148, 255, 255, 41, 0],
            &[0, 58, 255, 255, 71, 0, 0, 28, 255, 255, 114, 0],
            &[0, 100, 255, 255, 24, 0, 0, 0, 232, 255, 154, 0],
            &[0, 118, 255, 255, 6, 0, 0, 0, 214, 255, 171, 0],
            &[0, 118, 255, 255, 8, 0, 0, 0, 213, 255, 169, 0],
            &[0, 98, 255, 255, 29, 0, 0, 0, 235, 255, 149, 0],
            &[0, 53, 255, 255, 84, 0, 0, 36, 255, 255, 104, 0],
            &[0, 2, 227, 255, 210, 27, 12, 176, 255, 251, 28, 0],
            &[0, 0, 90, 255, 255, 255, 253, 255, 255, 140, 0, 0],
            &[0, 0, 0, 95, 228, 255, 255, 240, 131, 3, 0, 0],
            &[0, 0, 0, 0, 1, 28, 33, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ô' / 0xd4
        'Ô' => Some(&[
            &[0, 0, 0, 0, 95, 224, 224, 145, 0, 0, 0, 0],
            &[0, 0, 0, 37, 242, 224, 197, 255, 82, 0, 0, 0],
            &[0, 0, 0, 175, 190, 29, 11, 163, 201, 18, 0, 0],
            &[0, 0, 0, 0, 22, 67, 77, 34, 0, 0, 0, 0],
            &[0, 0, 0, 135, 249, 255, 255, 255, 175, 13, 0, 0],
            &[0, 0, 115, 255, 255, 232, 228, 255, 255, 170, 0, 0],
            &[0, 6, 236, 255, 187, 6, 0, 148, 255, 255, 41, 0],
            &[0, 58, 255, 255, 71, 0, 0, 28, 255, 255, 114, 0],
            &[0, 100, 255, 255, 24, 0, 0, 0, 232, 255, 154, 0],
            &[0, 118, 255, 255, 6, 0, 0, 0, 214, 255, 171, 0],
            &[0, 118, 255, 255, 8, 0, 0, 0, 213, 255, 169, 0],
            &[0, 98, 255, 255, 29, 0, 0, 0, 235, 255, 149, 0],
            &[0, 53, 255, 255, 84, 0, 0, 36, 255, 255, 104, 0],
            &[0, 2, 227, 255, 210, 27, 12, 176, 255, 251, 28, 0],
            &[0, 0, 90, 255, 255, 255, 253, 255, 255, 140, 0, 0],
            &[0, 0, 0, 95, 228, 255, 255, 240, 131, 3, 0, 0],
            &[0, 0, 0, 0, 1, 28, 33, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Õ' / 0xd5
        'Õ' => Some(&[
            &[0, 0, 0, 38, 157, 132, 28, 8, 177, 49, 0, 0],
            &[0, 0, 0, 206, 255, 255, 251, 233, 249, 24, 0, 0],
            &[0, 0, 13, 219, 61, 45, 153, 202, 96, 0, 0, 0],
            &[0, 0, 0, 0, 22, 67, 77, 34, 0, 0, 0, 0],
            &[0, 0, 0, 135, 249, 255, 255, 255, 175, 13, 0, 0],
            &[0, 0, 115, 255, 255, 232, 228, 255, 255, 170, 0, 0],
            &[0, 6, 236, 255, 187, 6, 0, 148, 255, 255, 41, 0],
            &[0, 58, 255, 255, 71, 0, 0, 28, 255, 255, 114, 0],
            &[0, 100, 255, 255, 24, 0, 0, 0, 232, 255, 154, 0],
            &[0, 118, 255, 255, 6, 0, 0, 0, 214, 255, 171, 0],
            &[0, 118, 255, 255, 8, 0, 0, 0, 213, 255, 169, 0],
            &[0, 98, 255, 255, 29, 0, 0, 0, 235, 255, 149, 0],
            &[0, 53, 255, 255, 84, 0, 0, 36, 255, 255, 104, 0],
            &[0, 2, 227, 255, 210, 27, 12, 176, 255, 251, 28, 0],
            &[0, 0, 90, 255, 255, 255, 253, 255, 255, 140, 0, 0],
            &[0, 0, 0, 95, 228, 255, 255, 240, 131, 3, 0, 0],
            &[0, 0, 0, 0, 1, 28, 33, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ö' / 0xd6
        'Ö' => Some(&[
            &[0, 0, 0, 125, 148, 13, 1, 130, 146, 11, 0, 0],
            &[0, 0, 19, 255, 255, 75, 28, 255, 255, 69, 0, 0],
            &[0, 0, 0, 135, 159, 14, 1, 138, 157, 12, 0, 0],
            &[0, 0, 0, 0, 22, 67, 77, 34, 0, 0, 0, 0],
            &[0, 0, 0, 135, 249, 255, 255, 255, 175, 13, 0, 0],
            &[0, 0, 115, 255, 255, 232, 228, 255, 255, 170, 0, 0],
            &[0, 6, 236, 255, 187, 6, 0, 148, 255, 255, 41, 0],
            &[0, 58, 255, 255, 71, 0, 0, 28, 255, 255, 114, 0],
            &[0, 100, 255, 255, 24, 0, 0, 0, 232, 255, 154, 0],
            &[0, 118, 255, 255, 6, 0, 0, 0, 214, 255, 171, 0],
            &[0, 118, 255, 255, 8, 0, 0, 0, 213, 255, 169, 0],
            &[0, 98, 255, 255, 29, 0, 0, 0, 235, 255, 149, 0],
            &[0, 53, 255, 255, 84, 0, 0, 36, 255, 255, 104, 0],
            &[0, 2, 227, 255, 210, 27, 12, 176, 255, 251, 28, 0],
            &[0, 0, 90, 255, 255, 255, 253, 255, 255, 140, 0, 0],
            &[0, 0, 0, 95, 228, 255, 255, 240, 131, 3, 0, 0],
            &[0, 0, 0, 0, 1, 28, 33, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '×' / 0xd7
        '×' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 118, 20, 0, 0, 5, 126, 11, 0, 0],
            &[0, 0, 99, 255, 212, 23, 6, 177, 255, 154, 0, 0],
            &[0, 0, 6, 179, 255, 215, 184, 255, 211, 21, 0, 0],
            &[0, 0, 0, 6, 179, 255, 255, 211, 21, 0, 0, 0],
            &[0, 0, 0, 5, 178, 255, 255, 210, 20, 0, 0, 0],
            &[0, 0, 5, 175, 255, 210, 178, 255, 208, 18, 0, 0],
            &[0, 0, 95, 255, 209, 20, 5, 175, 255, 146, 0, 0],
            &[0, 0, 0, 117, 19, 0, 0, 5, 124, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ø' / 0xd8
        'Ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 67, 77, 34, 0, 101, 63, 0],
            &[0, 0, 0, 135, 249, 255, 255, 255, 220, 255, 181, 0],
            &[0, 0, 115, 255, 255, 232, 228, 255, 255, 255, 27, 0],
            &[0, 6, 236, 255, 187, 6, 0, 255, 255, 255, 41, 0],
            &[0, 58, 255, 255, 71, 0, 47, 255, 255, 255, 114, 0],
            &[0, 100, 255, 255, 24, 5, 208, 239, 255, 255, 154, 0],
            &[0, 118, 255, 255, 6, 130, 255, 94, 214, 255, 171, 0],
            &[0, 118, 255, 255, 59, 251, 179, 0, 213, 255, 169, 0],
            &[0, 98, 255, 255, 241, 238, 25, 0, 235, 255, 149, 0],
            &[0, 53, 255, 255, 255, 92, 0, 36, 255, 255, 104, 0],
            &[0, 2, 255, 255, 255, 27, 12, 176, 255, 251, 28, 0],
            &[0, 8, 255, 255, 255, 255, 253, 255, 255, 140, 0, 0],
            &[0, 133, 255, 186, 228, 255, 255, 240, 131, 3, 0, 0],
            &[0, 26, 120, 0, 1, 28, 33, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ù' / 0xd9
        'Ù' => Some(&[
            &[0, 0, 0, 110, 224, 223, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 141, 255, 199, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 115, 215, 92, 0, 0, 0, 0],
            &[0, 8, 35, 35, 8, 0, 0, 1, 35, 35, 15, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 59, 255, 255, 64, 0, 0, 12, 255, 255, 111, 0],
            &[0, 44, 255, 255, 95, 0, 0, 41, 255, 255, 95, 0],
            &[0, 8, 243, 255, 206, 26, 15, 162, 255, 255, 43, 0],
            &[0, 0, 135, 255, 255, 255, 253, 255, 255, 178, 0, 0],
            &[0, 0, 4, 129, 237, 255, 255, 244, 151, 11, 0, 0],
            &[0, 0, 0, 0, 4, 29, 32, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ú' / 0xda
        'Ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 147, 224, 203, 23, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 230, 44, 0, 0, 0],
            &[0, 0, 0, 0, 5, 190, 194, 32, 0, 0, 0, 0],
            &[0, 8, 35, 35, 8, 0, 0, 1, 35, 35, 15, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 59, 255, 255, 64, 0, 0, 12, 255, 255, 111, 0],
            &[0, 44, 255, 255, 95, 0, 0, 41, 255, 255, 95, 0],
            &[0, 8, 243, 255, 206, 26, 15, 162, 255, 255, 43, 0],
            &[0, 0, 135, 255, 255, 255, 253, 255, 255, 178, 0, 0],
            &[0, 0, 4, 129, 237, 255, 255, 244, 151, 11, 0, 0],
            &[0, 0, 0, 0, 4, 29, 32, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Û' / 0xdb
        'Û' => Some(&[
            &[0, 0, 0, 0, 95, 224, 224, 145, 0, 0, 0, 0],
            &[0, 0, 0, 37, 242, 224, 197, 255, 82, 0, 0, 0],
            &[0, 0, 0, 175, 190, 29, 11, 163, 201, 18, 0, 0],
            &[0, 8, 35, 35, 8, 0, 0, 1, 35, 35, 15, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 59, 255, 255, 64, 0, 0, 12, 255, 255, 111, 0],
            &[0, 44, 255, 255, 95, 0, 0, 41, 255, 255, 95, 0],
            &[0, 8, 243, 255, 206, 26, 15, 162, 255, 255, 43, 0],
            &[0, 0, 135, 255, 255, 255, 253, 255, 255, 178, 0, 0],
            &[0, 0, 4, 129, 237, 255, 255, 244, 151, 11, 0, 0],
            &[0, 0, 0, 0, 4, 29, 32, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ü' / 0xdc
        'Ü' => Some(&[
            &[0, 0, 0, 125, 148, 13, 1, 130, 146, 11, 0, 0],
            &[0, 0, 19, 255, 255, 75, 28, 255, 255, 69, 0, 0],
            &[0, 0, 0, 135, 159, 14, 1, 138, 157, 12, 0, 0],
            &[0, 8, 35, 35, 8, 0, 0, 1, 35, 35, 15, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 59, 255, 255, 64, 0, 0, 12, 255, 255, 111, 0],
            &[0, 44, 255, 255, 95, 0, 0, 41, 255, 255, 95, 0],
            &[0, 8, 243, 255, 206, 26, 15, 162, 255, 255, 43, 0],
            &[0, 0, 135, 255, 255, 255, 253, 255, 255, 178, 0, 0],
            &[0, 0, 4, 129, 237, 255, 255, 244, 151, 11, 0, 0],
            &[0, 0, 0, 0, 4, 29, 32, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ý' / 0xdd
        'Ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 147, 224, 203, 23, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 230, 44, 0, 0, 0],
            &[0, 0, 0, 0, 5, 190, 194, 32, 0, 0, 0, 0],
            &[0, 22, 35, 34, 0, 0, 0, 0, 26, 35, 29, 0],
            &[0, 91, 255, 255, 53, 0, 0, 12, 241, 255, 142, 0],
            &[0, 3, 221, 255, 161, 0, 0, 105, 255, 249, 27, 0],
            &[0, 0, 103, 255, 248, 21, 0, 211, 255, 156, 0, 0],
            &[0, 0, 7, 230, 255, 122, 63, 255, 253, 37, 0, 0],
            &[0, 0, 0, 115, 255, 226, 174, 255, 169, 0, 0, 0],
            &[0, 0, 0, 12, 237, 255, 255, 255, 49, 0, 0, 0],
            &[0, 0, 0, 0, 128, 255, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Þ' / 0xde
        'Þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 35, 26, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 195, 25, 22, 5, 0, 0, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 249, 185, 40, 0, 0, 0],
            &[0, 163, 255, 255, 254, 255, 255, 255, 230, 13, 0, 0],
            &[0, 163, 255, 189, 0, 7, 107, 255, 255, 95, 0, 0],
            &[0, 163, 255, 189, 0, 0, 1, 248, 255, 127, 0, 0],
            &[0, 163, 255, 189, 0, 0, 28, 255, 255, 111, 0, 0],
            &[0, 163, 255, 213, 94, 118, 214, 255, 255, 45, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 139, 0, 0, 0],
            &[0, 163, 255, 237, 184, 177, 150, 68, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ß' / 0xdf
        'ß' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 101, 207, 247, 245, 198, 84, 0, 0, 0],
            &[0, 0, 116, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 231, 255, 194, 23, 32, 216, 255, 175, 0, 0],
            &[0, 11, 255, 255, 97, 0, 0, 202, 255, 151, 0, 0],
            &[0, 16, 255, 255, 84, 0, 138, 255, 236, 34, 0, 0],
            &[0, 16, 255, 255, 84, 89, 255, 242, 45, 0, 0, 0],
            &[0, 16, 255, 255, 84, 148, 255, 227, 17, 0, 0, 0],
            &[0, 16, 255, 255, 84, 75, 255, 255, 220, 44, 0, 0],
            &[0, 16, 255, 255, 84, 0, 100, 249, 255, 240, 36, 0],
            &[0, 16, 255, 255, 84, 0, 0, 51, 242, 255, 146, 0],
            &[0, 16, 255, 255, 84, 47, 0, 0, 214, 255, 171, 0],
            &[0, 16, 255, 255, 84, 222, 224, 219, 255, 255, 118, 0],
            &[0, 16, 255, 255, 84, 183, 255, 255, 251, 162, 6, 0],
            &[0, 0, 0, 0, 0, 0, 20, 37, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'à' / 0xe0
        'à' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 5, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 250, 255, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 76, 246, 237, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 177, 109, 0, 0, 0, 0],
            &[0, 0, 0, 15, 74, 102, 100, 67, 4, 0, 0, 0],
            &[0, 0, 87, 247, 255, 255, 255, 255, 223, 56, 0, 0],
            &[0, 0, 23, 237, 181, 144, 151, 237, 255, 216, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 86, 255, 255, 16, 0],
            &[0, 0, 0, 75, 151, 180, 191, 211, 255, 255, 24, 0],
            &[0, 0, 143, 255, 255, 221, 188, 199, 255, 255, 24, 0],
            &[0, 12, 251, 255, 130, 0, 0, 75, 255, 255, 24, 0],
            &[0, 27, 255, 255, 97, 0, 7, 178, 255, 255, 24, 0],
            &[0, 1, 229, 255, 254, 214, 239, 247, 255, 255, 24, 0],
            &[0, 0, 67, 231, 255, 255, 196, 40, 236, 255, 24, 0],
            &[0, 0, 0, 2, 30, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'á' / 0xe1
        'á' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 5, 5, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 146, 255, 234, 39, 0, 0],
            &[0, 0, 0, 0, 0, 63, 253, 228, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 146, 168, 28, 0, 0, 0, 0],
            &[0, 0, 0, 15, 74, 102, 100, 67, 4, 0, 0, 0],
            &[0, 0, 87, 247, 255, 255, 255, 255, 223, 56, 0, 0],
            &[0, 0, 23, 237, 181, 144, 151, 237, 255, 216, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 86, 255, 255, 16, 0],
            &[0, 0, 0, 75, 151, 180, 191, 211, 255, 255, 24, 0],
            &[0, 0, 143, 255, 255, 221, 188, 199, 255, 255, 24, 0],
            &[0, 12, 251, 255, 130, 0, 0, 75, 255, 255, 24, 0],
            &[0, 27, 255, 255, 97, 0, 7, 178, 255, 255, 24, 0],
            &[0, 1, 229, 255, 254, 214, 239, 247, 255, 255, 24, 0],
            &[0, 0, 67, 231, 255, 255, 196, 40, 236, 255, 24, 0],
            &[0, 0, 0, 2, 30, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'â' / 0xe2
        'â' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 5, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 255, 255, 208, 5, 0, 0, 0],
            &[0, 0, 0, 34, 239, 223, 149, 255, 145, 0, 0, 0],
            &[0, 0, 0, 131, 165, 26, 0, 105, 178, 38, 0, 0],
            &[0, 0, 0, 15, 74, 102, 100, 67, 4, 0, 0, 0],
            &[0, 0, 87, 247, 255, 255, 255, 255, 223, 56, 0, 0],
            &[0, 0, 23, 237, 181, 144, 151, 237, 255, 216, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 86, 255, 255, 16, 0],
            &[0, 0, 0, 75, 151, 180, 191, 211, 255, 255, 24, 0],
            &[0, 0, 143, 255, 255, 221, 188, 199, 255, 255, 24, 0],
            &[0, 12, 251, 255, 130, 0, 0, 75, 255, 255, 24, 0],
            &[0, 27, 255, 255, 97, 0, 7, 178, 255, 255, 24, 0],
            &[0, 1, 229, 255, 254, 214, 239, 247, 255, 255, 24, 0],
            &[0, 0, 67, 231, 255, 255, 196, 40, 236, 255, 24, 0],
            &[0, 0, 0, 2, 30, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ã' / 0xe3
        'ã' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 39, 187, 179, 76, 13, 197, 86, 0, 0],
            &[0, 0, 0, 185, 244, 242, 255, 255, 253, 40, 0, 0],
            &[0, 0, 0, 171, 64, 9, 103, 165, 87, 0, 0, 0],
            &[0, 0, 0, 15, 74, 102, 100, 67, 4, 0, 0, 0],
            &[0, 0, 87, 247, 255, 255, 255, 255, 223, 56, 0, 0],
            &[0, 0, 23, 237, 181, 144, 151, 237, 255, 216, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 86, 255, 255, 16, 0],
            &[0, 0, 0, 75, 151, 180, 191, 211, 255, 255, 24, 0],
            &[0, 0, 143, 255, 255, 221, 188, 199, 255, 255, 24, 0],
            &[0, 12, 251, 255, 130, 0, 0, 75, 255, 255, 24, 0],
            &[0, 27, 255, 255, 97, 0, 7, 178, 255, 255, 24, 0],
            &[0, 1, 229, 255, 254, 214, 239, 247, 255, 255, 24, 0],
            &[0, 0, 67, 231, 255, 255, 196, 40, 236, 255, 24, 0],
            &[0, 0, 0, 2, 30, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ä' / 0xe4
        'ä' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 136, 193, 42, 0, 143, 192, 39, 0, 0],
            &[0, 0, 0, 239, 255, 108, 0, 248, 255, 103, 0, 0],
            &[0, 0, 0, 78, 131, 17, 0, 81, 129, 15, 0, 0],
            &[0, 0, 0, 15, 74, 102, 100, 67, 4, 0, 0, 0],
            &[0, 0, 87, 247, 255, 255, 255, 255, 223, 56, 0, 0],
            &[0, 0, 23, 237, 181, 144, 151, 237, 255, 216, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 86, 255, 255, 16, 0],
            &[0, 0, 0, 75, 151, 180, 191, 211, 255, 255, 24, 0],
            &[0, 0, 143, 255, 255, 221, 188, 199, 255, 255, 24, 0],
            &[0, 12, 251, 255, 130, 0, 0, 75, 255, 255, 24, 0],
            &[0, 27, 255, 255, 97, 0, 7, 178, 255, 255, 24, 0],
            &[0, 1, 229, 255, 254, 214, 239, 247, 255, 255, 24, 0],
            &[0, 0, 67, 231, 255, 255, 196, 40, 236, 255, 24, 0],
            &[0, 0, 0, 2, 30, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'å' / 0xe5
        'å' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 39, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 89, 250, 254, 187, 4, 0, 0, 0],
            &[0, 0, 0, 0, 192, 147, 39, 255, 51, 0, 0, 0],
            &[0, 0, 0, 0, 158, 214, 163, 247, 24, 0, 0, 0],
            &[0, 0, 0, 0, 17, 140, 164, 65, 0, 0, 0, 0],
            &[0, 0, 0, 15, 74, 102, 100, 67, 4, 0, 0, 0],
            &[0, 0, 87, 247, 255, 255, 255, 255, 223, 56, 0, 0],
            &[0, 0, 23, 237, 181, 144, 151, 237, 255, 216, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 86, 255, 255, 16, 0],
            &[0, 0, 0, 75, 151, 180, 191, 211, 255, 255, 24, 0],
            &[0, 0, 143, 255, 255, 221, 188, 199, 255, 255, 24, 0],
            &[0, 12, 251, 255, 130, 0, 0, 75, 255, 255, 24, 0],
            &[0, 27, 255, 255, 97, 0, 7, 178, 255, 255, 24, 0],
            &[0, 1, 229, 255, 254, 214, 239, 247, 255, 255, 24, 0],
            &[0, 0, 67, 231, 255, 255, 196, 40, 236, 255, 24, 0],
            &[0, 0, 0, 2, 30, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'æ' / 0xe6
        'æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 36, 94, 99, 36, 2, 75, 103, 33, 0, 0, 0],
            &[45, 254, 255, 255, 249, 210, 255, 255, 245, 48, 0, 0],
            &[0, 165, 146, 189, 255, 255, 212, 192, 255, 172, 0, 0],
            &[0, 0, 0, 6, 245, 255, 84, 35, 255, 235, 0, 0],
            &[0, 68, 147, 168, 249, 255, 181, 164, 255, 255, 3, 0],
            &[90, 255, 251, 194, 250, 255, 242, 239, 239, 239, 7, 0],
            &[188, 255, 134, 0, 239, 255, 63, 0, 0, 0, 0, 0],
            &[204, 255, 108, 25, 255, 255, 160, 0, 0, 20, 0, 0],
            &[165, 255, 242, 224, 241, 249, 255, 217, 200, 162, 0, 0],
            &[42, 229, 255, 239, 73, 81, 233, 255, 253, 115, 0, 0],
            &[0, 3, 33, 8, 0, 0, 4, 34, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ç' / 0xe7
        'ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 75, 106, 101, 66, 6, 0, 0],
            &[0, 0, 0, 76, 237, 255, 255, 255, 255, 172, 0, 0],
            &[0, 0, 35, 246, 255, 247, 180, 176, 229, 86, 0, 0],
            &[0, 0, 134, 255, 252, 52, 0, 0, 0, 2, 0, 0],
            &[0, 0, 185, 255, 201, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 201, 255, 181, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 215, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 127, 255, 255, 113, 4, 3, 58, 79, 0, 0],
            &[0, 0, 24, 237, 255, 255, 248, 249, 255, 141, 0, 0],
            &[0, 0, 0, 43, 190, 253, 255, 255, 229, 88, 0, 0],
            &[0, 0, 0, 0, 0, 72, 255, 88, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 200, 251, 58, 0, 0, 0],
            &[0, 0, 0, 0, 10, 56, 153, 255, 100, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 9, 0, 0, 0, 0, 0],
        ]),
        // letter: 'è' / 0xe8
        'è' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 5, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 248, 255, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 244, 241, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 176, 115, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 94, 106, 70, 4, 0, 0, 0],
            &[0, 0, 2, 146, 254, 255, 255, 255, 216, 36, 0, 0],
            &[0, 0, 112, 255, 254, 163, 129, 220, 255, 203, 0, 0],
            &[0, 0, 222, 255, 150, 0, 0, 30, 254, 255, 44, 0],
            &[0, 18, 255, 255, 190, 156, 156, 156, 246, 255, 80, 0],
            &[0, 30, 255, 255, 255, 255, 255, 255, 255, 255, 89, 0],
            &[0, 7, 252, 255, 102, 18, 18, 18, 18, 18, 6, 0],
            &[0, 0, 183, 255, 220, 50, 0, 0, 11, 79, 9, 0],
            &[0, 0, 44, 244, 255, 255, 242, 230, 253, 255, 19, 0],
            &[0, 0, 0, 41, 180, 250, 255, 255, 249, 179, 10, 0],
            &[0, 0, 0, 0, 0, 7, 32, 30, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'é' / 0xe9
        'é' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 5, 5, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 138, 255, 237, 44, 0, 0],
            &[0, 0, 0, 0, 0, 56, 252, 232, 46, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 170, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 94, 106, 70, 4, 0, 0, 0],
            &[0, 0, 2, 146, 254, 255, 255, 255, 216, 36, 0, 0],
            &[0, 0, 112, 255, 254, 163, 129, 220, 255, 203, 0, 0],
            &[0, 0, 222, 255, 150, 0, 0, 30, 254, 255, 44, 0],
            &[0, 18, 255, 255, 190, 156, 156, 156, 246, 255, 80, 0],
            &[0, 30, 255, 255, 255, 255, 255, 255, 255, 255, 89, 0],
            &[0, 7, 252, 255, 102, 18, 18, 18, 18, 18, 6, 0],
            &[0, 0, 183, 255, 220, 50, 0, 0, 11, 79, 9, 0],
            &[0, 0, 44, 244, 255, 255, 242, 230, 253, 255, 19, 0],
            &[0, 0, 0, 41, 180, 250, 255, 255, 249, 179, 10, 0],
            &[0, 0, 0, 0, 0, 7, 32, 30, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ê' / 0xea
        'ê' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 5, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 255, 255, 215, 8, 0, 0, 0],
            &[0, 0, 0, 29, 235, 227, 144, 255, 153, 0, 0, 0],
            &[0, 0, 0, 125, 167, 29, 0, 99, 178, 44, 0, 0],
            &[0, 0, 0, 0, 33, 94, 106, 70, 4, 0, 0, 0],
            &[0, 0, 2, 146, 254, 255, 255, 255, 216, 36, 0, 0],
            &[0, 0, 112, 255, 254, 163, 129, 220, 255, 203, 0, 0],
            &[0, 0, 222, 255, 150, 0, 0, 30, 254, 255, 44, 0],
            &[0, 18, 255, 255, 190, 156, 156, 156, 246, 255, 80, 0],
            &[0, 30, 255, 255, 255, 255, 255, 255, 255, 255, 89, 0],
            &[0, 7, 252, 255, 102, 18, 18, 18, 18, 18, 6, 0],
            &[0, 0, 183, 255, 220, 50, 0, 0, 11, 79, 9, 0],
            &[0, 0, 44, 244, 255, 255, 242, 230, 253, 255, 19, 0],
            &[0, 0, 0, 41, 180, 250, 255, 255, 249, 179, 10, 0],
            &[0, 0, 0, 0, 0, 7, 32, 30, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ë' / 0xeb
        'ë' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 195, 48, 0, 136, 194, 44, 0, 0],
            &[0, 0, 0, 230, 255, 117, 0, 239, 255, 111, 0, 0],
            &[0, 0, 0, 73, 132, 20, 0, 76, 131, 18, 0, 0],
            &[0, 0, 0, 0, 33, 94, 106, 70, 4, 0, 0, 0],
            &[0, 0, 2, 146, 254, 255, 255, 255, 216, 36, 0, 0],
            &[0, 0, 112, 255, 254, 163, 129, 220, 255, 203, 0, 0],
            &[0, 0, 222, 255, 150, 0, 0, 30, 254, 255, 44, 0],
            &[0, 18, 255, 255, 190, 156, 156, 156, 246, 255, 80, 0],
            &[0, 30, 255, 255, 255, 255, 255, 255, 255, 255, 89, 0],
            &[0, 7, 252, 255, 102, 18, 18, 18, 18, 18, 6, 0],
            &[0, 0, 183, 255, 220, 50, 0, 0, 11, 79, 9, 0],
            &[0, 0, 44, 244, 255, 255, 242, 230, 253, 255, 19, 0],
            &[0, 0, 0, 41, 180, 250, 255, 255, 249, 179, 10, 0],
            &[0, 0, 0, 0, 0, 7, 32, 30, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ì' / 0xec
        'ì' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 5, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 106, 255, 254, 60, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 104, 253, 217, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 178, 85, 0, 0, 0, 0, 0],
            &[0, 37, 72, 72, 72, 72, 41, 0, 0, 0, 0, 0],
            &[0, 133, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0],
            &[0, 66, 148, 176, 248, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 149, 0, 0, 0, 0, 0],
            &[0, 153, 198, 239, 255, 255, 254, 222, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'í' / 0xed
        'í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 5, 5, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 181, 255, 217, 21, 0, 0, 0],
            &[0, 0, 0, 0, 96, 255, 210, 24, 0, 0, 0, 0],
            &[0, 0, 0, 5, 165, 157, 15, 0, 0, 0, 0, 0],
            &[0, 37, 72, 72, 72, 72, 41, 0, 0, 0, 0, 0],
            &[0, 133, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0],
            &[0, 66, 148, 176, 248, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 149, 0, 0, 0, 0, 0],
            &[0, 153, 198, 239, 255, 255, 254, 222, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'î' / 0xee
        'î' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 5, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 122, 255, 255, 179, 0, 0, 0, 0, 0],
            &[0, 0, 57, 250, 203, 168, 255, 110, 0, 0, 0, 0],
            &[0, 0, 155, 153, 14, 2, 127, 173, 18, 0, 0, 0],
            &[0, 37, 72, 72, 72, 72, 41, 0, 0, 0, 0, 0],
            &[0, 133, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0],
            &[0, 66, 148, 176, 248, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 149, 0, 0, 0, 0, 0],
            &[0, 153, 198, 239, 255, 255, 254, 222, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ï' / 0xef
        'ï' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 162, 185, 22, 4, 167, 183, 19, 0, 0, 0],
            &[0, 18, 255, 255, 73, 27, 255, 255, 68, 0, 0, 0],
            &[0, 0, 98, 122, 6, 0, 101, 120, 5, 0, 0, 0],
            &[0, 37, 72, 72, 72, 72, 41, 0, 0, 0, 0, 0],
            &[0, 133, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0],
            &[0, 66, 148, 176, 248, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 149, 0, 0, 0, 0, 0],
            &[0, 153, 198, 239, 255, 255, 254, 222, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ð' / 0xf0
        'ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 142, 18, 78, 194, 13, 0, 0],
            &[0, 0, 0, 15, 214, 255, 255, 255, 206, 37, 0, 0],
            &[0, 0, 0, 6, 129, 255, 255, 255, 78, 0, 0, 0],
            &[0, 0, 0, 45, 251, 167, 142, 255, 242, 32, 0, 0],
            &[0, 0, 0, 0, 59, 54, 28, 165, 255, 168, 0, 0],
            &[0, 0, 16, 172, 254, 255, 255, 205, 255, 251, 22, 0],
            &[0, 0, 174, 255, 250, 177, 172, 245, 255, 255, 84, 0],
            &[0, 29, 254, 255, 125, 0, 0, 79, 255, 255, 119, 0],
            &[0, 70, 255, 255, 58, 0, 0, 4, 253, 255, 127, 0],
            &[0, 66, 255, 255, 62, 0, 0, 21, 255, 255, 102, 0],
            &[0, 20, 249, 255, 147, 0, 0, 109, 255, 253, 35, 0],
            &[0, 0, 140, 255, 255, 201, 180, 250, 255, 150, 0, 0],
            &[0, 0, 3, 123, 234, 255, 255, 239, 128, 3, 0, 0],
            &[0, 0, 0, 0, 3, 29, 33, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ñ' / 0xf1
        'ñ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 49, 191, 174, 68, 17, 207, 71, 0, 0],
            &[0, 0, 0, 203, 241, 245, 255, 255, 249, 27, 0, 0],
            &[0, 0, 3, 180, 52, 12, 109, 165, 76, 0, 0, 0],
            &[0, 4, 72, 61, 0, 24, 93, 100, 48, 0, 0, 0],
            &[0, 16, 255, 252, 99, 247, 255, 255, 254, 127, 0, 0],
            &[0, 16, 255, 255, 255, 211, 164, 242, 255, 251, 28, 0],
            &[0, 16, 255, 255, 214, 7, 0, 84, 255, 255, 73, 0],
            &[0, 16, 255, 255, 132, 0, 0, 34, 255, 255, 89, 0],
            &[0, 16, 255, 255, 106, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ò' / 0xf2
        'ò' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 5, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 106, 255, 254, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 104, 253, 217, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 178, 85, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 96, 104, 62, 1, 0, 0, 0],
            &[0, 0, 8, 163, 255, 255, 255, 255, 205, 30, 0, 0],
            &[0, 0, 143, 255, 255, 179, 155, 245, 255, 201, 1, 0],
            &[0, 12, 246, 255, 155, 0, 0, 104, 255, 255, 59, 0],
            &[0, 59, 255, 255, 69, 0, 0, 23, 255, 255, 112, 0],
            &[0, 76, 255, 255, 54, 0, 0, 4, 255, 255, 127, 0],
            &[0, 54, 255, 255, 88, 0, 0, 35, 255, 255, 104, 0],
            &[0, 6, 235, 255, 205, 9, 0, 153, 255, 254, 37, 0],
            &[0, 0, 104, 255, 255, 238, 226, 255, 255, 155, 0, 0],
            &[0, 0, 0, 100, 228, 255, 255, 241, 135, 4, 0, 0],
            &[0, 0, 0, 0, 1, 29, 34, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ó' / 0xf3
        'ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 5, 5, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 181, 255, 217, 21, 0, 0],
            &[0, 0, 0, 0, 0, 96, 255, 210, 24, 0, 0, 0],
            &[0, 0, 0, 0, 5, 165, 157, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 96, 104, 62, 1, 0, 0, 0],
            &[0, 0, 8, 163, 255, 255, 255, 255, 205, 30, 0, 0],
            &[0, 0, 143, 255, 255, 179, 155, 245, 255, 201, 1, 0],
            &[0, 12, 246, 255, 155, 0, 0, 104, 255, 255, 59, 0],
            &[0, 59, 255, 255, 69, 0, 0, 23, 255, 255, 112, 0],
            &[0, 76, 255, 255, 54, 0, 0, 4, 255, 255, 127, 0],
            &[0, 54, 255, 255, 88, 0, 0, 35, 255, 255, 104, 0],
            &[0, 6, 235, 255, 205, 9, 0, 153, 255, 254, 37, 0],
            &[0, 0, 104, 255, 255, 238, 226, 255, 255, 155, 0, 0],
            &[0, 0, 0, 100, 228, 255, 255, 241, 135, 4, 0, 0],
            &[0, 0, 0, 0, 1, 29, 34, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ô' / 0xf4
        'ô' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 5, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 122, 255, 255, 179, 0, 0, 0, 0],
            &[0, 0, 0, 57, 250, 203, 168, 255, 110, 0, 0, 0],
            &[0, 0, 0, 155, 153, 14, 2, 127, 173, 18, 0, 0],
            &[0, 0, 0, 0, 40, 96, 104, 62, 1, 0, 0, 0],
            &[0, 0, 8, 163, 255, 255, 255, 255, 205, 30, 0, 0],
            &[0, 0, 143, 255, 255, 179, 155, 245, 255, 201, 1, 0],
            &[0, 12, 246, 255, 155, 0, 0, 104, 255, 255, 59, 0],
            &[0, 59, 255, 255, 69, 0, 0, 23, 255, 255, 112, 0],
            &[0, 76, 255, 255, 54, 0, 0, 4, 255, 255, 127, 0],
            &[0, 54, 255, 255, 88, 0, 0, 35, 255, 255, 104, 0],
            &[0, 6, 235, 255, 205, 9, 0, 153, 255, 254, 37, 0],
            &[0, 0, 104, 255, 255, 238, 226, 255, 255, 155, 0, 0],
            &[0, 0, 0, 100, 228, 255, 255, 241, 135, 4, 0, 0],
            &[0, 0, 0, 0, 1, 29, 34, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'õ' / 0xf5
        'õ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 194, 169, 61, 24, 214, 56, 0, 0],
            &[0, 0, 0, 220, 239, 247, 255, 255, 242, 16, 0, 0],
            &[0, 0, 13, 182, 39, 16, 116, 165, 66, 0, 0, 0],
            &[0, 0, 0, 0, 40, 96, 104, 62, 1, 0, 0, 0],
            &[0, 0, 8, 163, 255, 255, 255, 255, 205, 30, 0, 0],
            &[0, 0, 143, 255, 255, 179, 155, 245, 255, 201, 1, 0],
            &[0, 12, 246, 255, 155, 0, 0, 104, 255, 255, 59, 0],
            &[0, 59, 255, 255, 69, 0, 0, 23, 255, 255, 112, 0],
            &[0, 76, 255, 255, 54, 0, 0, 4, 255, 255, 127, 0],
            &[0, 54, 255, 255, 88, 0, 0, 35, 255, 255, 104, 0],
            &[0, 6, 235, 255, 205, 9, 0, 153, 255, 254, 37, 0],
            &[0, 0, 104, 255, 255, 238, 226, 255, 255, 155, 0, 0],
            &[0, 0, 0, 100, 228, 255, 255, 241, 135, 4, 0, 0],
            &[0, 0, 0, 0, 1, 29, 34, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ö' / 0xf6
        'ö' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 162, 185, 22, 4, 167, 183, 19, 0, 0],
            &[0, 0, 18, 255, 255, 73, 27, 255, 255, 68, 0, 0],
            &[0, 0, 0, 98, 122, 6, 0, 101, 120, 5, 0, 0],
            &[0, 0, 0, 0, 40, 96, 104, 62, 1, 0, 0, 0],
            &[0, 0, 8, 163, 255, 255, 255, 255, 205, 30, 0, 0],
            &[0, 0, 143, 255, 255, 179, 155, 245, 255, 201, 1, 0],
            &[0, 12, 246, 255, 155, 0, 0, 104, 255, 255, 59, 0],
            &[0, 59, 255, 255, 69, 0, 0, 23, 255, 255, 112, 0],
            &[0, 76, 255, 255, 54, 0, 0, 4, 255, 255, 127, 0],
            &[0, 54, 255, 255, 88, 0, 0, 35, 255, 255, 104, 0],
            &[0, 6, 235, 255, 205, 9, 0, 153, 255, 254, 37, 0],
            &[0, 0, 104, 255, 255, 238, 226, 255, 255, 155, 0, 0],
            &[0, 0, 0, 100, 228, 255, 255, 241, 135, 4, 0, 0],
            &[0, 0, 0, 0, 1, 29, 34, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '÷' / 0xf7
        '÷' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 30, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 240, 255, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 249, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 20, 0, 0, 0, 0, 0, 0],
            &[0, 186, 230, 230, 230, 230, 230, 230, 230, 2, 0, 0],
            &[0, 186, 230, 230, 230, 230, 230, 230, 230, 2, 0, 0],
            &[0, 0, 0, 0, 10, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 229, 251, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 239, 255, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 30, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ø' / 0xf8
        'ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 96, 104, 62, 103, 65, 0, 0],
            &[0, 0, 8, 163, 255, 255, 255, 255, 255, 207, 0, 0],
            &[0, 0, 143, 255, 255, 179, 159, 255, 255, 225, 1, 0],
            &[0, 12, 246, 255, 155, 0, 126, 255, 255, 255, 59, 0],
            &[0, 59, 255, 255, 69, 48, 250, 192, 255, 255, 112, 0],
            &[0, 76, 255, 255, 61, 210, 231, 24, 255, 255, 127, 0],
            &[0, 54, 255, 255, 222, 255, 77, 35, 255, 255, 104, 0],
            &[0, 6, 235, 255, 255, 168, 0, 153, 255, 254, 37, 0],
            &[0, 0, 113, 255, 255, 253, 226, 255, 255, 155, 0, 0],
            &[0, 0, 128, 255, 255, 255, 255, 241, 135, 4, 0, 0],
            &[0, 0, 15, 92, 1, 29, 34, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ù' / 0xf9
        'ù' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 5, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 106, 255, 254, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 104, 253, 217, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 178, 85, 0, 0, 0, 0],
            &[0, 10, 72, 72, 22, 0, 0, 14, 72, 72, 19, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 61, 255, 255, 67, 0],
            &[0, 37, 255, 255, 89, 0, 0, 90, 255, 255, 67, 0],
            &[0, 12, 255, 255, 167, 1, 11, 192, 255, 255, 67, 0],
            &[0, 0, 198, 255, 255, 238, 245, 244, 254, 255, 67, 0],
            &[0, 0, 39, 208, 255, 255, 214, 57, 198, 255, 67, 0],
            &[0, 0, 0, 0, 23, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ú' / 0xfa
        'ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 5, 5, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 181, 255, 217, 21, 0, 0],
            &[0, 0, 0, 0, 0, 96, 255, 210, 24, 0, 0, 0],
            &[0, 0, 0, 0, 5, 165, 157, 15, 0, 0, 0, 0],
            &[0, 10, 72, 72, 22, 0, 0, 14, 72, 72, 19, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 61, 255, 255, 67, 0],
            &[0, 37, 255, 255, 89, 0, 0, 90, 255, 255, 67, 0],
            &[0, 12, 255, 255, 167, 1, 11, 192, 255, 255, 67, 0],
            &[0, 0, 198, 255, 255, 238, 245, 244, 254, 255, 67, 0],
            &[0, 0, 39, 208, 255, 255, 214, 57, 198, 255, 67, 0],
            &[0, 0, 0, 0, 23, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'û' / 0xfb
        'û' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 5, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 122, 255, 255, 179, 0, 0, 0, 0],
            &[0, 0, 0, 57, 250, 203, 168, 255, 110, 0, 0, 0],
            &[0, 0, 0, 155, 153, 14, 2, 127, 173, 18, 0, 0],
            &[0, 10, 72, 72, 22, 0, 0, 14, 72, 72, 19, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 61, 255, 255, 67, 0],
            &[0, 37, 255, 255, 89, 0, 0, 90, 255, 255, 67, 0],
            &[0, 12, 255, 255, 167, 1, 11, 192, 255, 255, 67, 0],
            &[0, 0, 198, 255, 255, 238, 245, 244, 254, 255, 67, 0],
            &[0, 0, 39, 208, 255, 255, 214, 57, 198, 255, 67, 0],
            &[0, 0, 0, 0, 23, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ü' / 0xfc
        'ü' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 162, 185, 22, 4, 167, 183, 19, 0, 0],
            &[0, 0, 18, 255, 255, 73, 27, 255, 255, 68, 0, 0],
            &[0, 0, 0, 98, 122, 6, 0, 101, 120, 5, 0, 0],
            &[0, 10, 72, 72, 22, 0, 0, 14, 72, 72, 19, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 61, 255, 255, 67, 0],
            &[0, 37, 255, 255, 89, 0, 0, 90, 255, 255, 67, 0],
            &[0, 12, 255, 255, 167, 1, 11, 192, 255, 255, 67, 0],
            &[0, 0, 198, 255, 255, 238, 245, 244, 254, 255, 67, 0],
            &[0, 0, 39, 208, 255, 255, 214, 57, 198, 255, 67, 0],
            &[0, 0, 0, 0, 23, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ý' / 0xfd
        'ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 5, 5, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 181, 255, 217, 21, 0, 0],
            &[0, 0, 0, 0, 0, 96, 255, 210, 24, 0, 0, 0],
            &[0, 0, 0, 0, 5, 165, 157, 15, 0, 0, 0, 0],
            &[0, 43, 72, 72, 5, 0, 0, 0, 60, 72, 58, 0],
            &[0, 89, 255, 255, 73, 0, 0, 12, 249, 255, 147, 0],
            &[0, 7, 236, 255, 159, 0, 0, 81, 255, 255, 54, 0],
            &[0, 0, 142, 255, 238, 5, 0, 157, 255, 216, 0, 0],
            &[0, 0, 41, 255, 255, 74, 1, 231, 255, 124, 0, 0],
            &[0, 0, 0, 195, 255, 159, 53, 255, 254, 32, 0, 0],
            &[0, 0, 0, 93, 255, 237, 129, 255, 194, 0, 0, 0],
            &[0, 0, 0, 9, 239, 255, 225, 255, 101, 0, 0, 0],
            &[0, 0, 0, 0, 146, 255, 255, 247, 15, 0, 0, 0],
            &[0, 0, 0, 0, 45, 255, 255, 170, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 255, 255, 75, 0, 0, 0, 0],
            &[0, 0, 4, 31, 202, 255, 219, 2, 0, 0, 0, 0],
            &[0, 0, 253, 255, 255, 254, 75, 0, 0, 0, 0, 0],
            &[0, 0, 26, 34, 4, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'þ' / 0xfe
        'þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 15, 235, 235, 82, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 255, 87, 22, 94, 103, 35, 0, 0, 0],
            &[0, 16, 255, 255, 146, 245, 255, 255, 249, 84, 0, 0],
            &[0, 16, 255, 255, 255, 201, 168, 247, 255, 236, 10, 0],
            &[0, 16, 255, 255, 186, 1, 0, 101, 255, 255, 77, 0],
            &[0, 16, 255, 255, 106, 0, 0, 26, 255, 255, 119, 0],
            &[0, 16, 255, 255, 95, 0, 0, 13, 255, 255, 128, 0],
            &[0, 16, 255, 255, 126, 0, 0, 38, 255, 255, 108, 0],
            &[0, 16, 255, 255, 223, 26, 0, 144, 255, 255, 52, 0],
            &[0, 16, 255, 255, 255, 252, 239, 255, 255, 199, 0, 0],
            &[0, 16, 255, 255, 118, 207, 255, 255, 199, 28, 0, 0],
            &[0, 16, 255, 255, 88, 0, 26, 25, 0, 0, 0, 0],
            &[0, 16, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 20, 20, 7, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ÿ' / 0xff
        'ÿ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 162, 185, 22, 4, 167, 183, 19, 0, 0],
            &[0, 0, 18, 255, 255, 73, 27, 255, 255, 68, 0, 0],
            &[0, 0, 0, 98, 122, 6, 0, 101, 120, 5, 0, 0],
            &[0, 43, 72, 72, 5, 0, 0, 0, 60, 72, 58, 0],
            &[0, 89, 255, 255, 73, 0, 0, 12, 249, 255, 147, 0],
            &[0, 7, 236, 255, 159, 0, 0, 81, 255, 255, 54, 0],
            &[0, 0, 142, 255, 238, 5, 0, 157, 255, 216, 0, 0],
            &[0, 0, 41, 255, 255, 74, 1, 231, 255, 124, 0, 0],
            &[0, 0, 0, 195, 255, 159, 53, 255, 254, 32, 0, 0],
            &[0, 0, 0, 93, 255, 237, 129, 255, 194, 0, 0, 0],
            &[0, 0, 0, 9, 239, 255, 225, 255, 101, 0, 0, 0],
            &[0, 0, 0, 0, 146, 255, 255, 247, 15, 0, 0, 0],
            &[0, 0, 0, 0, 45, 255, 255, 170, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 255, 255, 75, 0, 0, 0, 0],
            &[0, 0, 4, 31, 202, 255, 219, 2, 0, 0, 0, 0],
            &[0, 0, 253, 255, 255, 254, 75, 0, 0, 0, 0, 0],
            &[0, 0, 26, 34, 4, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ā' / 0x100
        'Ā' => Some(&[
            &[0, 0, 13, 28, 28, 28, 28, 19, 0, 0, 0, 0],
            &[0, 0, 123, 255, 255, 255, 255, 170, 0, 0, 0, 0],
            &[0, 0, 70, 145, 145, 145, 145, 97, 0, 0, 0, 0],
            &[0, 0, 0, 24, 48, 48, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 171, 255, 255, 218, 0, 0, 0, 0, 0],
            &[0, 0, 3, 238, 255, 250, 255, 33, 0, 0, 0, 0],
            &[0, 0, 56, 255, 233, 183, 255, 104, 0, 0, 0, 0],
            &[0, 0, 126, 255, 175, 122, 255, 174, 0, 0, 0, 0],
            &[0, 0, 196, 255, 112, 62, 255, 241, 4, 0, 0, 0],
            &[0, 15, 251, 255, 48, 7, 248, 255, 60, 0, 0, 0],
            &[0, 81, 255, 246, 14, 12, 205, 255, 130, 0, 0, 0],
            &[0, 151, 255, 255, 255, 255, 255, 255, 201, 0, 0, 0],
            &[0, 222, 255, 255, 253, 253, 255, 255, 253, 18, 0, 0],
            &[36, 255, 255, 48, 0, 0, 13, 252, 255, 86, 0, 0],
            &[106, 255, 238, 2, 0, 0, 0, 203, 255, 157, 0, 0],
            &[177, 255, 177, 0, 0, 0, 0, 142, 255, 227, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ā' / 0x101
        'ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 65, 65, 65, 65, 52, 0, 0, 0],
            &[0, 0, 0, 89, 255, 255, 255, 255, 205, 0, 0, 0],
            &[0, 0, 0, 37, 109, 109, 109, 109, 87, 0, 0, 0],
            &[0, 0, 0, 15, 74, 102, 100, 67, 4, 0, 0, 0],
            &[0, 0, 87, 247, 255, 255, 255, 255, 223, 56, 0, 0],
            &[0, 0, 23, 237, 181, 144, 151, 237, 255, 216, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 86, 255, 255, 16, 0],
            &[0, 0, 0, 75, 151, 180, 191, 211, 255, 255, 24, 0],
            &[0, 0, 143, 255, 255, 221, 188, 199, 255, 255, 24, 0],
            &[0, 12, 251, 255, 130, 0, 0, 75, 255, 255, 24, 0],
            &[0, 27, 255, 255, 97, 0, 7, 178, 255, 255, 24, 0],
            &[0, 1, 229, 255, 254, 214, 239, 247, 255, 255, 24, 0],
            &[0, 0, 67, 231, 255, 255, 196, 40, 236, 255, 24, 0],
            &[0, 0, 0, 2, 30, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ă' / 0x102
        'Ă' => Some(&[
            &[0, 0, 158, 196, 5, 0, 149, 205, 0, 0, 0, 0],
            &[0, 0, 88, 255, 228, 223, 255, 131, 0, 0, 0, 0],
            &[0, 0, 0, 98, 191, 196, 118, 4, 0, 0, 0, 0],
            &[0, 0, 0, 24, 48, 48, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 171, 255, 255, 218, 0, 0, 0, 0, 0],
            &[0, 0, 3, 238, 255, 250, 255, 33, 0, 0, 0, 0],
            &[0, 0, 56, 255, 233, 183, 255, 104, 0, 0, 0, 0],
            &[0, 0, 126, 255, 175, 122, 255, 174, 0, 0, 0, 0],
            &[0, 0, 196, 255, 112, 62, 255, 241, 4, 0, 0, 0],
            &[0, 15, 251, 255, 48, 7, 248, 255, 60, 0, 0, 0],
            &[0, 81, 255, 246, 14, 12, 205, 255, 130, 0, 0, 0],
            &[0, 151, 255, 255, 255, 255, 255, 255, 201, 0, 0, 0],
            &[0, 222, 255, 255, 253, 253, 255, 255, 253, 18, 0, 0],
            &[36, 255, 255, 48, 0, 0, 13, 252, 255, 86, 0, 0],
            &[106, 255, 238, 2, 0, 0, 0, 203, 255, 157, 0, 0],
            &[177, 255, 177, 0, 0, 0, 0, 142, 255, 227, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ă' / 0x103
        'ă' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 15, 0, 0, 6, 18, 1, 0, 0],
            &[0, 0, 0, 131, 243, 30, 5, 147, 246, 3, 0, 0],
            &[0, 0, 0, 40, 247, 254, 248, 255, 143, 0, 0, 0],
            &[0, 0, 0, 0, 45, 149, 164, 98, 2, 0, 0, 0],
            &[0, 0, 0, 15, 74, 102, 100, 67, 4, 0, 0, 0],
            &[0, 0, 87, 247, 255, 255, 255, 255, 223, 56, 0, 0],
            &[0, 0, 23, 237, 181, 144, 151, 237, 255, 216, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 86, 255, 255, 16, 0],
            &[0, 0, 0, 75, 151, 180, 191, 211, 255, 255, 24, 0],
            &[0, 0, 143, 255, 255, 221, 188, 199, 255, 255, 24, 0],
            &[0, 12, 251, 255, 130, 0, 0, 75, 255, 255, 24, 0],
            &[0, 27, 255, 255, 97, 0, 7, 178, 255, 255, 24, 0],
            &[0, 1, 229, 255, 254, 214, 239, 247, 255, 255, 24, 0],
            &[0, 0, 67, 231, 255, 255, 196, 40, 236, 255, 24, 0],
            &[0, 0, 0, 2, 30, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ą' / 0x104
        'Ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 48, 48, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 171, 255, 255, 218, 0, 0, 0, 0, 0],
            &[0, 0, 3, 238, 255, 250, 255, 33, 0, 0, 0, 0],
            &[0, 0, 56, 255, 233, 183, 255, 104, 0, 0, 0, 0],
            &[0, 0, 126, 255, 175, 122, 255, 174, 0, 0, 0, 0],
            &[0, 0, 196, 255, 112, 62, 255, 241, 4, 0, 0, 0],
            &[0, 15, 251, 255, 48, 7, 248, 255, 60, 0, 0, 0],
            &[0, 81, 255, 246, 14, 12, 205, 255, 130, 0, 0, 0],
            &[0, 151, 255, 255, 255, 255, 255, 255, 201, 0, 0, 0],
            &[0, 222, 255, 255, 253, 253, 255, 255, 253, 18, 0, 0],
            &[36, 255, 255, 48, 0, 0, 13, 252, 255, 86, 0, 0],
            &[106, 255, 238, 2, 0, 0, 0, 203, 255, 157, 0, 0],
            &[177, 255, 177, 0, 0, 0, 0, 142, 255, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 223, 152, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 156, 253, 15, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 186, 255, 145, 57, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0],
        ]),
        // letter: 'ą' / 0x105
        'ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 74, 102, 100, 67, 4, 0, 0, 0],
            &[0, 0, 87, 247, 255, 255, 255, 255, 223, 56, 0, 0],
            &[0, 0, 23, 237, 181, 144, 151, 237, 255, 216, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 86, 255, 255, 16, 0],
            &[0, 0, 0, 75, 151, 180, 191, 211, 255, 255, 24, 0],
            &[0, 0, 143, 255, 255, 221, 188, 199, 255, 255, 24, 0],
            &[0, 12, 251, 255, 130, 0, 0, 75, 255, 255, 24, 0],
            &[0, 27, 255, 255, 97, 0, 7, 178, 255, 255, 24, 0],
            &[0, 1, 229, 255, 254, 214, 239, 247, 255, 255, 24, 0],
            &[0, 0, 67, 231, 255, 255, 196, 40, 255, 255, 24, 0],
            &[0, 0, 0, 2, 30, 21, 0, 69, 245, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 222, 202, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 249, 238, 132, 22, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0],
        ]),
        // letter: 'Ć' / 0x106
        'Ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 166, 224, 193, 14, 0],
            &[0, 0, 0, 0, 0, 0, 91, 255, 220, 33, 0, 0],
            &[0, 0, 0, 0, 0, 13, 200, 185, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 53, 72, 52, 11, 0, 0],
            &[0, 0, 0, 5, 133, 242, 255, 255, 255, 243, 75, 0],
            &[0, 0, 0, 176, 255, 255, 246, 224, 247, 250, 23, 0],
            &[0, 0, 90, 255, 255, 138, 9, 0, 7, 58, 0, 0],
            &[0, 0, 194, 255, 215, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 249, 255, 131, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 255, 255, 103, 0, 0, 0, 0, 0, 0, 0],
            &[0, 31, 255, 255, 96, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 213, 255, 201, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 124, 255, 255, 148, 28, 5, 38, 120, 7, 0],
            &[0, 0, 10, 211, 255, 255, 255, 255, 255, 255, 11, 0],
            &[0, 0, 0, 17, 155, 246, 255, 255, 248, 181, 6, 0],
            &[0, 0, 0, 0, 0, 3, 29, 30, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ć' / 0x107
        'ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 5, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 73, 255, 254, 93, 0, 0],
            &[0, 0, 0, 0, 0, 16, 226, 251, 92, 0, 0, 0],
            &[0, 0, 0, 0, 0, 94, 178, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 75, 106, 101, 66, 6, 0, 0],
            &[0, 0, 0, 76, 237, 255, 255, 255, 255, 172, 0, 0],
            &[0, 0, 35, 246, 255, 247, 180, 176, 229, 86, 0, 0],
            &[0, 0, 134, 255, 252, 52, 0, 0, 0, 2, 0, 0],
            &[0, 0, 185, 255, 201, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 201, 255, 181, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 215, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 127, 255, 255, 113, 4, 3, 58, 79, 0, 0],
            &[0, 0, 24, 237, 255, 255, 248, 249, 255, 141, 0, 0],
            &[0, 0, 0, 43, 190, 253, 255, 255, 229, 88, 0, 0],
            &[0, 0, 0, 0, 0, 12, 35, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĉ' / 0x108
        'Ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 113, 224, 224, 126, 0, 0, 0],
            &[0, 0, 0, 0, 52, 248, 213, 208, 252, 64, 0, 0],
            &[0, 0, 0, 6, 187, 180, 21, 17, 175, 193, 9, 0],
            &[0, 0, 0, 0, 0, 6, 53, 72, 52, 11, 0, 0],
            &[0, 0, 0, 5, 133, 242, 255, 255, 255, 243, 75, 0],
            &[0, 0, 0, 176, 255, 255, 246, 224, 247, 250, 23, 0],
            &[0, 0, 90, 255, 255, 138, 9, 0, 7, 58, 0, 0],
            &[0, 0, 194, 255, 215, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 249, 255, 131, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 255, 255, 103, 0, 0, 0, 0, 0, 0, 0],
            &[0, 31, 255, 255, 96, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 213, 255, 201, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 124, 255, 255, 148, 28, 5, 38, 120, 7, 0],
            &[0, 0, 10, 211, 255, 255, 255, 255, 255, 255, 11, 0],
            &[0, 0, 0, 17, 155, 246, 255, 255, 248, 181, 6, 0],
            &[0, 0, 0, 0, 0, 3, 29, 30, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĉ' / 0x109
        'ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 5, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 241, 255, 247, 40, 0, 0, 0],
            &[0, 0, 0, 5, 194, 249, 131, 246, 209, 9, 0, 0],
            &[0, 0, 0, 80, 178, 65, 0, 56, 176, 90, 0, 0],
            &[0, 0, 0, 0, 8, 75, 106, 101, 66, 6, 0, 0],
            &[0, 0, 0, 76, 237, 255, 255, 255, 255, 172, 0, 0],
            &[0, 0, 35, 246, 255, 247, 180, 176, 229, 86, 0, 0],
            &[0, 0, 134, 255, 252, 52, 0, 0, 0, 2, 0, 0],
            &[0, 0, 185, 255, 201, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 201, 255, 181, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 215, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 127, 255, 255, 113, 4, 3, 58, 79, 0, 0],
            &[0, 0, 24, 237, 255, 255, 248, 249, 255, 141, 0, 0],
            &[0, 0, 0, 43, 190, 253, 255, 255, 229, 88, 0, 0],
            &[0, 0, 0, 0, 0, 12, 35, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ċ' / 0x10a
        'Ċ' => Some(&[
            &[0, 0, 0, 0, 0, 56, 241, 242, 61, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 255, 120, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 182, 181, 31, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 53, 72, 52, 11, 0, 0],
            &[0, 0, 0, 5, 133, 242, 255, 255, 255, 243, 75, 0],
            &[0, 0, 0, 176, 255, 255, 246, 224, 247, 250, 23, 0],
            &[0, 0, 90, 255, 255, 138, 9, 0, 7, 58, 0, 0],
            &[0, 0, 194, 255, 215, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 249, 255, 131, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 255, 255, 103, 0, 0, 0, 0, 0, 0, 0],
            &[0, 31, 255, 255, 96, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 213, 255, 201, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 124, 255, 255, 148, 28, 5, 38, 120, 7, 0],
            &[0, 0, 10, 211, 255, 255, 255, 255, 255, 255, 11, 0],
            &[0, 0, 0, 17, 155, 246, 255, 255, 248, 181, 6, 0],
            &[0, 0, 0, 0, 0, 3, 29, 30, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0],
        ]),
        // letter: 'ċ' / 0x10b
        'ċ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 40, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 196, 255, 206, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 239, 255, 245, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 82, 160, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 75, 106, 101, 66, 6, 0, 0],
            &[0, 0, 0, 76, 237, 255, 255, 255, 255, 172, 0, 0],
            &[0, 0, 35, 246, 255, 247, 180, 176, 229, 86, 0, 0],
            &[0, 0, 134, 255, 252, 52, 0, 0, 0, 2, 0, 0],
            &[0, 0, 185, 255, 201, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 201, 255, 181, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 215, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 127, 255, 255, 113, 4, 3, 58, 79, 0, 0],
            &[0, 0, 24, 237, 255, 255, 248, 249, 255, 141, 0, 0],
            &[0, 0, 0, 43, 190, 253, 255, 255, 229, 88, 0, 0],
            &[0, 0, 0, 0, 0, 12, 35, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Č' / 0x10c
        'Č' => Some(&[
            &[0, 0, 0, 6, 191, 190, 25, 24, 187, 197, 9, 0],
            &[0, 0, 0, 0, 46, 247, 218, 218, 250, 56, 0, 0],
            &[0, 0, 0, 0, 0, 106, 215, 215, 115, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 53, 72, 52, 11, 0, 0],
            &[0, 0, 0, 5, 133, 242, 255, 255, 255, 243, 75, 0],
            &[0, 0, 0, 176, 255, 255, 246, 224, 247, 250, 23, 0],
            &[0, 0, 90, 255, 255, 138, 9, 0, 7, 58, 0, 0],
            &[0, 0, 194, 255, 215, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 249, 255, 131, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 255, 255, 103, 0, 0, 0, 0, 0, 0, 0],
            &[0, 31, 255, 255, 96, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 255, 255, 123, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 213, 255, 201, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 124, 255, 255, 148, 28, 5, 38, 120, 7, 0],
            &[0, 0, 10, 211, 255, 255, 255, 255, 255, 255, 11, 0],
            &[0, 0, 0, 17, 155, 246, 255, 255, 248, 181, 6, 0],
            &[0, 0, 0, 0, 0, 3, 29, 30, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'č' / 0x10d
        'č' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 5, 0, 0, 0, 5, 3, 0, 0],
            &[0, 0, 0, 81, 254, 141, 2, 135, 254, 94, 0, 0],
            &[0, 0, 0, 0, 136, 255, 216, 255, 152, 0, 0, 0],
            &[0, 0, 0, 0, 5, 162, 178, 168, 10, 0, 0, 0],
            &[0, 0, 0, 0, 8, 75, 106, 101, 66, 6, 0, 0],
            &[0, 0, 0, 76, 237, 255, 255, 255, 255, 172, 0, 0],
            &[0, 0, 35, 246, 255, 247, 180, 176, 229, 86, 0, 0],
            &[0, 0, 134, 255, 252, 52, 0, 0, 0, 2, 0, 0],
            &[0, 0, 185, 255, 201, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 201, 255, 181, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 215, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 127, 255, 255, 113, 4, 3, 58, 79, 0, 0],
            &[0, 0, 24, 237, 255, 255, 248, 249, 255, 141, 0, 0],
            &[0, 0, 0, 43, 190, 253, 255, 255, 229, 88, 0, 0],
            &[0, 0, 0, 0, 0, 12, 35, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ď' / 0x10e
        'Ď' => Some(&[
            &[0, 0, 28, 213, 163, 8, 47, 206, 165, 0, 0, 0],
            &[0, 0, 0, 86, 255, 197, 239, 232, 25, 0, 0, 0],
            &[0, 0, 0, 0, 147, 215, 215, 75, 0, 0, 0, 0],
            &[0, 8, 35, 35, 35, 31, 7, 0, 0, 0, 0, 0],
            &[0, 60, 255, 255, 255, 255, 254, 196, 68, 0, 0, 0],
            &[0, 60, 255, 255, 246, 248, 255, 255, 252, 88, 0, 0],
            &[0, 60, 255, 255, 50, 0, 44, 210, 255, 238, 15, 0],
            &[0, 60, 255, 255, 50, 0, 0, 53, 255, 255, 93, 0],
            &[0, 60, 255, 255, 50, 0, 0, 0, 236, 255, 146, 0],
            &[0, 60, 255, 255, 50, 0, 0, 0, 213, 255, 168, 0],
            &[0, 60, 255, 255, 50, 0, 0, 0, 223, 255, 161, 0],
            &[0, 60, 255, 255, 50, 0, 0, 6, 248, 255, 135, 0],
            &[0, 60, 255, 255, 50, 0, 0, 92, 255, 255, 66, 0],
            &[0, 60, 255, 255, 68, 38, 102, 243, 255, 208, 3, 0],
            &[0, 60, 255, 255, 255, 255, 255, 255, 232, 45, 0, 0],
            &[0, 60, 255, 255, 255, 249, 220, 144, 27, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ď' / 0x10f
        'ď' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 235, 235, 97, 182, 235, 98],
            &[0, 0, 0, 0, 0, 0, 255, 255, 108, 242, 231, 10],
            &[0, 0, 0, 0, 0, 0, 255, 255, 138, 204, 90, 0],
            &[0, 0, 39, 102, 82, 3, 248, 255, 106, 0, 0, 0],
            &[0, 90, 252, 255, 255, 187, 228, 255, 106, 0, 0, 0],
            &[10, 237, 255, 234, 160, 224, 255, 255, 106, 0, 0, 0],
            &[71, 255, 255, 81, 0, 45, 254, 255, 106, 0, 0, 0],
            &[110, 255, 255, 19, 0, 0, 231, 255, 106, 0, 0, 0],
            &[119, 255, 255, 8, 0, 0, 211, 255, 106, 0, 0, 0],
            &[97, 255, 255, 31, 0, 0, 231, 255, 106, 0, 0, 0],
            &[59, 255, 255, 127, 0, 83, 255, 255, 106, 0, 0, 0],
            &[3, 218, 255, 255, 238, 247, 251, 255, 106, 0, 0, 0],
            &[0, 51, 228, 255, 242, 84, 170, 255, 106, 0, 0, 0],
            &[0, 0, 5, 35, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Đ' / 0x110
        'Đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 35, 35, 35, 31, 7, 0, 0, 0, 0, 0],
            &[0, 60, 255, 255, 255, 255, 254, 196, 68, 0, 0, 0],
            &[0, 60, 255, 255, 246, 248, 255, 255, 252, 88, 0, 0],
            &[0, 60, 255, 255, 50, 0, 44, 210, 255, 238, 15, 0],
            &[0, 60, 255, 255, 50, 0, 0, 53, 255, 255, 93, 0],
            &[0, 65, 255, 255, 55, 5, 0, 0, 236, 255, 146, 0],
            &[0, 255, 255, 255, 255, 255, 0, 0, 213, 255, 168, 0],
            &[0, 255, 255, 255, 255, 255, 0, 0, 223, 255, 161, 0],
            &[0, 64, 255, 255, 55, 4, 0, 6, 248, 255, 135, 0],
            &[0, 60, 255, 255, 50, 0, 0, 92, 255, 255, 66, 0],
            &[0, 60, 255, 255, 68, 38, 102, 243, 255, 208, 3, 0],
            &[0, 60, 255, 255, 255, 255, 255, 255, 232, 45, 0, 0],
            &[0, 60, 255, 255, 255, 249, 220, 144, 27, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'đ' / 0x111
        'đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 35, 235, 235, 62, 0, 0],
            &[0, 0, 0, 0, 80, 143, 181, 255, 255, 211, 28, 0],
            &[0, 0, 0, 0, 144, 255, 255, 255, 255, 255, 51, 0],
            &[0, 0, 0, 42, 67, 29, 41, 255, 255, 77, 1, 0],
            &[0, 20, 198, 255, 255, 247, 114, 255, 255, 67, 0, 0],
            &[0, 172, 255, 255, 230, 207, 253, 255, 255, 67, 0, 0],
            &[17, 253, 255, 183, 5, 0, 118, 255, 255, 67, 0, 0],
            &[63, 255, 255, 77, 0, 0, 19, 255, 255, 67, 0, 0],
            &[75, 255, 255, 52, 0, 0, 0, 250, 255, 67, 0, 0],
            &[52, 255, 255, 74, 0, 0, 14, 255, 255, 67, 0, 0],
            &[15, 251, 255, 184, 8, 0, 133, 255, 255, 67, 0, 0],
            &[0, 165, 255, 255, 247, 240, 243, 255, 255, 67, 0, 0],
            &[0, 16, 183, 255, 255, 216, 49, 208, 255, 67, 0, 0],
            &[0, 0, 0, 23, 29, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ē' / 0x112
        'Ē' => Some(&[
            &[0, 0, 11, 28, 28, 28, 28, 22, 0, 0, 0, 0],
            &[0, 0, 97, 255, 255, 255, 255, 196, 0, 0, 0, 0],
            &[0, 0, 55, 145, 145, 145, 145, 112, 0, 0, 0, 0],
            &[0, 22, 35, 35, 35, 35, 35, 35, 35, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 252, 252, 252, 252, 252, 2, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 213, 92, 92, 92, 92, 69, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 193, 0, 0, 0],
            &[0, 163, 255, 240, 195, 195, 195, 195, 147, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 197, 31, 31, 31, 31, 31, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ē' / 0x113
        'ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 20, 65, 65, 65, 65, 55, 0, 0, 0],
            &[0, 0, 0, 80, 255, 255, 255, 255, 214, 0, 0, 0],
            &[0, 0, 0, 34, 109, 109, 109, 109, 91, 0, 0, 0],
            &[0, 0, 0, 0, 33, 94, 106, 70, 4, 0, 0, 0],
            &[0, 0, 2, 146, 254, 255, 255, 255, 216, 36, 0, 0],
            &[0, 0, 112, 255, 254, 163, 129, 220, 255, 203, 0, 0],
            &[0, 0, 222, 255, 150, 0, 0, 30, 254, 255, 44, 0],
            &[0, 18, 255, 255, 190, 156, 156, 156, 246, 255, 80, 0],
            &[0, 30, 255, 255, 255, 255, 255, 255, 255, 255, 89, 0],
            &[0, 7, 252, 255, 102, 18, 18, 18, 18, 18, 6, 0],
            &[0, 0, 183, 255, 220, 50, 0, 0, 11, 79, 9, 0],
            &[0, 0, 44, 244, 255, 255, 242, 230, 253, 255, 19, 0],
            &[0, 0, 0, 41, 180, 250, 255, 255, 249, 179, 10, 0],
            &[0, 0, 0, 0, 0, 7, 32, 30, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĕ' / 0x114
        'Ĕ' => Some(&[
            &[0, 0, 134, 215, 10, 0, 125, 227, 2, 0, 0, 0],
            &[0, 0, 63, 254, 233, 218, 255, 157, 0, 0, 0, 0],
            &[0, 0, 0, 81, 187, 200, 131, 9, 0, 0, 0, 0],
            &[0, 22, 35, 35, 35, 35, 35, 35, 35, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 252, 252, 252, 252, 252, 2, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 213, 92, 92, 92, 92, 69, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 193, 0, 0, 0],
            &[0, 163, 255, 240, 195, 195, 195, 195, 147, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 197, 31, 31, 31, 31, 31, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĕ' / 0x115
        'ĕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 15, 0, 0, 5, 18, 2, 0, 0],
            &[0, 0, 0, 122, 247, 35, 4, 139, 250, 7, 0, 0],
            &[0, 0, 0, 34, 244, 254, 248, 255, 151, 0, 0, 0],
            &[0, 0, 0, 0, 41, 147, 165, 102, 3, 0, 0, 0],
            &[0, 0, 0, 0, 33, 94, 106, 70, 4, 0, 0, 0],
            &[0, 0, 2, 146, 254, 255, 255, 255, 216, 36, 0, 0],
            &[0, 0, 112, 255, 254, 163, 129, 220, 255, 203, 0, 0],
            &[0, 0, 222, 255, 150, 0, 0, 30, 254, 255, 44, 0],
            &[0, 18, 255, 255, 190, 156, 156, 156, 246, 255, 80, 0],
            &[0, 30, 255, 255, 255, 255, 255, 255, 255, 255, 89, 0],
            &[0, 7, 252, 255, 102, 18, 18, 18, 18, 18, 6, 0],
            &[0, 0, 183, 255, 220, 50, 0, 0, 11, 79, 9, 0],
            &[0, 0, 44, 244, 255, 255, 242, 230, 253, 255, 19, 0],
            &[0, 0, 0, 41, 180, 250, 255, 255, 249, 179, 10, 0],
            &[0, 0, 0, 0, 0, 7, 32, 30, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ė' / 0x116
        'Ė' => Some(&[
            &[0, 0, 0, 23, 226, 251, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 255, 255, 167, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 166, 192, 57, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 35, 35, 35, 35, 35, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 252, 252, 252, 252, 252, 2, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 213, 92, 92, 92, 92, 69, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 193, 0, 0, 0],
            &[0, 163, 255, 240, 195, 195, 195, 195, 147, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 197, 31, 31, 31, 31, 31, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ė' / 0x117
        'ė' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 239, 255, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 255, 181, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 121, 157, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 94, 106, 70, 4, 0, 0, 0],
            &[0, 0, 2, 146, 254, 255, 255, 255, 216, 36, 0, 0],
            &[0, 0, 112, 255, 254, 163, 129, 220, 255, 203, 0, 0],
            &[0, 0, 222, 255, 150, 0, 0, 30, 254, 255, 44, 0],
            &[0, 18, 255, 255, 190, 156, 156, 156, 246, 255, 80, 0],
            &[0, 30, 255, 255, 255, 255, 255, 255, 255, 255, 89, 0],
            &[0, 7, 252, 255, 102, 18, 18, 18, 18, 18, 6, 0],
            &[0, 0, 183, 255, 220, 50, 0, 0, 11, 79, 9, 0],
            &[0, 0, 44, 244, 255, 255, 242, 230, 253, 255, 19, 0],
            &[0, 0, 0, 41, 180, 250, 255, 255, 249, 179, 10, 0],
            &[0, 0, 0, 0, 0, 7, 32, 30, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ę' / 0x118
        'Ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 35, 35, 35, 35, 35, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 252, 252, 252, 252, 252, 2, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 213, 92, 92, 92, 92, 69, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 193, 0, 0, 0],
            &[0, 163, 255, 240, 195, 195, 195, 195, 147, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 197, 31, 31, 31, 31, 31, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 14, 210, 175, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 131, 255, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 162, 255, 157, 70, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0],
        ]),
        // letter: 'ę' / 0x119
        'ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 94, 106, 70, 4, 0, 0, 0],
            &[0, 0, 2, 146, 254, 255, 255, 255, 216, 36, 0, 0],
            &[0, 0, 112, 255, 254, 163, 129, 220, 255, 203, 0, 0],
            &[0, 0, 222, 255, 150, 0, 0, 30, 254, 255, 44, 0],
            &[0, 18, 255, 255, 190, 156, 156, 156, 246, 255, 80, 0],
            &[0, 30, 255, 255, 255, 255, 255, 255, 255, 255, 89, 0],
            &[0, 7, 252, 255, 102, 18, 18, 18, 18, 18, 6, 0],
            &[0, 0, 183, 255, 220, 50, 0, 0, 11, 79, 9, 0],
            &[0, 0, 44, 244, 255, 255, 242, 230, 253, 255, 19, 0],
            &[0, 0, 0, 41, 180, 250, 255, 255, 255, 255, 11, 0],
            &[0, 0, 0, 0, 0, 7, 33, 203, 226, 18, 0, 0],
            &[0, 0, 0, 0, 0, 0, 60, 255, 115, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 74, 255, 204, 112, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0],
        ]),
        // letter: 'Ě' / 0x11a
        'Ě' => Some(&[
            &[0, 0, 157, 208, 48, 8, 161, 216, 32, 0, 0, 0],
            &[0, 0, 19, 226, 239, 197, 255, 97, 0, 0, 0, 0],
            &[0, 0, 0, 67, 215, 215, 155, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 35, 35, 35, 35, 35, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 252, 252, 252, 252, 252, 2, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 213, 92, 92, 92, 92, 69, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 193, 0, 0, 0],
            &[0, 163, 255, 240, 195, 195, 195, 195, 147, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 197, 31, 31, 31, 31, 31, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ě' / 0x11b
        'ě' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 3, 0, 0, 0, 5, 2, 0, 0],
            &[0, 0, 0, 145, 246, 87, 14, 185, 243, 41, 0, 0],
            &[0, 0, 0, 4, 197, 253, 218, 255, 87, 0, 0, 0],
            &[0, 0, 0, 0, 35, 178, 178, 132, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 94, 106, 70, 4, 0, 0, 0],
            &[0, 0, 2, 146, 254, 255, 255, 255, 216, 36, 0, 0],
            &[0, 0, 112, 255, 254, 163, 129, 220, 255, 203, 0, 0],
            &[0, 0, 222, 255, 150, 0, 0, 30, 254, 255, 44, 0],
            &[0, 18, 255, 255, 190, 156, 156, 156, 246, 255, 80, 0],
            &[0, 30, 255, 255, 255, 255, 255, 255, 255, 255, 89, 0],
            &[0, 7, 252, 255, 102, 18, 18, 18, 18, 18, 6, 0],
            &[0, 0, 183, 255, 220, 50, 0, 0, 11, 79, 9, 0],
            &[0, 0, 44, 244, 255, 255, 242, 230, 253, 255, 19, 0],
            &[0, 0, 0, 41, 180, 250, 255, 255, 249, 179, 10, 0],
            &[0, 0, 0, 0, 0, 7, 32, 30, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => Some(&[
            &[0, 0, 0, 0, 12, 204, 224, 218, 30, 0, 0, 0],
            &[0, 0, 0, 0, 162, 255, 169, 252, 194, 4, 0, 0],
            &[0, 0, 0, 77, 215, 103, 0, 79, 212, 104, 0, 0],
            &[0, 0, 0, 0, 0, 28, 63, 64, 34, 0, 0, 0],
            &[0, 0, 0, 42, 191, 255, 255, 255, 255, 218, 37, 0],
            &[0, 0, 41, 240, 255, 255, 234, 218, 247, 231, 4, 0],
            &[0, 0, 190, 255, 244, 81, 0, 0, 10, 51, 0, 0],
            &[0, 29, 255, 255, 121, 0, 0, 0, 0, 0, 0, 0],
            &[0, 89, 255, 255, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 112, 255, 254, 4, 0, 101, 213, 213, 213, 92, 0],
            &[0, 116, 255, 254, 3, 0, 121, 255, 255, 255, 111, 0],
            &[0, 93, 255, 255, 34, 0, 25, 52, 213, 255, 111, 0],
            &[0, 37, 255, 255, 117, 0, 0, 0, 202, 255, 111, 0],
            &[0, 0, 199, 255, 244, 79, 1, 1, 206, 255, 111, 0],
            &[0, 0, 48, 240, 255, 255, 247, 248, 255, 255, 111, 0],
            &[0, 0, 0, 43, 177, 252, 255, 255, 253, 196, 60, 0],
            &[0, 0, 0, 0, 0, 6, 33, 29, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĝ' / 0x11d
        'ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 5, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 122, 255, 255, 179, 0, 0, 0, 0],
            &[0, 0, 0, 57, 250, 203, 168, 255, 110, 0, 0, 0],
            &[0, 0, 0, 155, 153, 14, 2, 127, 173, 18, 0, 0],
            &[0, 0, 0, 18, 97, 110, 46, 0, 48, 72, 19, 0],
            &[0, 0, 44, 234, 255, 255, 254, 120, 233, 255, 67, 0],
            &[0, 0, 196, 255, 255, 185, 180, 249, 255, 255, 67, 0],
            &[0, 28, 255, 255, 150, 0, 0, 113, 255, 255, 67, 0],
            &[0, 68, 255, 255, 68, 0, 0, 39, 255, 255, 67, 0],
            &[0, 78, 255, 255, 52, 0, 0, 15, 255, 255, 67, 0],
            &[0, 60, 255, 255, 82, 0, 0, 36, 255, 255, 67, 0],
            &[0, 14, 249, 255, 195, 11, 1, 150, 255, 255, 67, 0],
            &[0, 0, 156, 255, 255, 248, 241, 251, 255, 255, 67, 0],
            &[0, 0, 11, 174, 254, 255, 225, 109, 255, 255, 67, 0],
            &[0, 0, 0, 0, 20, 34, 2, 70, 255, 255, 54, 0],
            &[0, 0, 100, 79, 24, 2, 31, 187, 255, 247, 15, 0],
            &[0, 0, 172, 255, 255, 254, 255, 255, 255, 136, 0, 0],
            &[0, 0, 0, 2, 23, 36, 19, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ğ' / 0x11e
        'Ğ' => Some(&[
            &[0, 0, 0, 49, 237, 72, 0, 40, 237, 76, 0, 0],
            &[0, 0, 0, 7, 219, 248, 210, 248, 231, 17, 0, 0],
            &[0, 0, 0, 0, 33, 163, 204, 167, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 63, 64, 34, 0, 0, 0],
            &[0, 0, 0, 42, 191, 255, 255, 255, 255, 218, 37, 0],
            &[0, 0, 41, 240, 255, 255, 234, 218, 247, 231, 4, 0],
            &[0, 0, 190, 255, 244, 81, 0, 0, 10, 51, 0, 0],
            &[0, 29, 255, 255, 121, 0, 0, 0, 0, 0, 0, 0],
            &[0, 89, 255, 255, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 112, 255, 254, 4, 0, 101, 213, 213, 213, 92, 0],
            &[0, 116, 255, 254, 3, 0, 121, 255, 255, 255, 111, 0],
            &[0, 93, 255, 255, 34, 0, 25, 52, 213, 255, 111, 0],
            &[0, 37, 255, 255, 117, 0, 0, 0, 202, 255, 111, 0],
            &[0, 0, 199, 255, 244, 79, 1, 1, 206, 255, 111, 0],
            &[0, 0, 48, 240, 255, 255, 247, 248, 255, 255, 111, 0],
            &[0, 0, 0, 43, 177, 252, 255, 255, 253, 196, 60, 0],
            &[0, 0, 0, 0, 0, 6, 33, 29, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ğ' / 0x11f
        'ğ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 12, 0, 0, 8, 17, 0, 0, 0],
            &[0, 0, 0, 165, 220, 18, 10, 177, 215, 0, 0, 0],
            &[0, 0, 0, 67, 254, 252, 250, 255, 108, 0, 0, 0],
            &[0, 0, 0, 0, 62, 155, 160, 81, 0, 0, 0, 0],
            &[0, 0, 0, 18, 97, 110, 46, 0, 48, 72, 19, 0],
            &[0, 0, 44, 234, 255, 255, 254, 120, 233, 255, 67, 0],
            &[0, 0, 196, 255, 255, 185, 180, 249, 255, 255, 67, 0],
            &[0, 28, 255, 255, 150, 0, 0, 113, 255, 255, 67, 0],
            &[0, 68, 255, 255, 68, 0, 0, 39, 255, 255, 67, 0],
            &[0, 78, 255, 255, 52, 0, 0, 15, 255, 255, 67, 0],
            &[0, 60, 255, 255, 82, 0, 0, 36, 255, 255, 67, 0],
            &[0, 14, 249, 255, 195, 11, 1, 150, 255, 255, 67, 0],
            &[0, 0, 156, 255, 255, 248, 241, 251, 255, 255, 67, 0],
            &[0, 0, 11, 174, 254, 255, 225, 109, 255, 255, 67, 0],
            &[0, 0, 0, 0, 20, 34, 2, 70, 255, 255, 54, 0],
            &[0, 0, 100, 79, 24, 2, 31, 187, 255, 247, 15, 0],
            &[0, 0, 172, 255, 255, 254, 255, 255, 255, 136, 0, 0],
            &[0, 0, 0, 2, 23, 36, 19, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ġ' / 0x120
        'Ġ' => Some(&[
            &[0, 0, 0, 0, 0, 159, 255, 186, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 233, 255, 254, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 107, 197, 121, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 63, 64, 34, 0, 0, 0],
            &[0, 0, 0, 42, 191, 255, 255, 255, 255, 218, 37, 0],
            &[0, 0, 41, 240, 255, 255, 234, 218, 247, 231, 4, 0],
            &[0, 0, 190, 255, 244, 81, 0, 0, 10, 51, 0, 0],
            &[0, 29, 255, 255, 121, 0, 0, 0, 0, 0, 0, 0],
            &[0, 89, 255, 255, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 112, 255, 254, 4, 0, 101, 213, 213, 213, 92, 0],
            &[0, 116, 255, 254, 3, 0, 121, 255, 255, 255, 111, 0],
            &[0, 93, 255, 255, 34, 0, 25, 52, 213, 255, 111, 0],
            &[0, 37, 255, 255, 117, 0, 0, 0, 202, 255, 111, 0],
            &[0, 0, 199, 255, 244, 79, 1, 1, 206, 255, 111, 0],
            &[0, 0, 48, 240, 255, 255, 247, 248, 255, 255, 111, 0],
            &[0, 0, 0, 43, 177, 252, 255, 255, 253, 196, 60, 0],
            &[0, 0, 0, 0, 0, 6, 33, 29, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ġ' / 0x121
        'ġ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 250, 255, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 255, 255, 137, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 139, 150, 26, 0, 0, 0, 0],
            &[0, 0, 0, 18, 97, 110, 46, 0, 48, 72, 19, 0],
            &[0, 0, 44, 234, 255, 255, 254, 120, 233, 255, 67, 0],
            &[0, 0, 196, 255, 255, 185, 180, 249, 255, 255, 67, 0],
            &[0, 28, 255, 255, 150, 0, 0, 113, 255, 255, 67, 0],
            &[0, 68, 255, 255, 68, 0, 0, 39, 255, 255, 67, 0],
            &[0, 78, 255, 255, 52, 0, 0, 15, 255, 255, 67, 0],
            &[0, 60, 255, 255, 82, 0, 0, 36, 255, 255, 67, 0],
            &[0, 14, 249, 255, 195, 11, 1, 150, 255, 255, 67, 0],
            &[0, 0, 156, 255, 255, 248, 241, 251, 255, 255, 67, 0],
            &[0, 0, 11, 174, 254, 255, 225, 109, 255, 255, 67, 0],
            &[0, 0, 0, 0, 20, 34, 2, 70, 255, 255, 54, 0],
            &[0, 0, 100, 79, 24, 2, 31, 187, 255, 247, 15, 0],
            &[0, 0, 172, 255, 255, 254, 255, 255, 255, 136, 0, 0],
            &[0, 0, 0, 2, 23, 36, 19, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ģ' / 0x122
        'Ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 63, 64, 34, 0, 0, 0],
            &[0, 0, 0, 42, 191, 255, 255, 255, 255, 218, 37, 0],
            &[0, 0, 41, 240, 255, 255, 234, 218, 247, 231, 4, 0],
            &[0, 0, 190, 255, 244, 81, 0, 0, 10, 51, 0, 0],
            &[0, 29, 255, 255, 121, 0, 0, 0, 0, 0, 0, 0],
            &[0, 89, 255, 255, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 112, 255, 254, 4, 0, 101, 213, 213, 213, 92, 0],
            &[0, 116, 255, 254, 3, 0, 121, 255, 255, 255, 111, 0],
            &[0, 93, 255, 255, 34, 0, 25, 52, 213, 255, 111, 0],
            &[0, 37, 255, 255, 117, 0, 0, 0, 202, 255, 111, 0],
            &[0, 0, 199, 255, 244, 79, 1, 1, 206, 255, 111, 0],
            &[0, 0, 48, 240, 255, 255, 247, 248, 255, 255, 111, 0],
            &[0, 0, 0, 43, 177, 252, 255, 255, 253, 196, 60, 0],
            &[0, 0, 0, 0, 0, 6, 33, 29, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 144, 246, 146, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 199, 249, 34, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 188, 116, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ģ' / 0x123
        'ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 254, 105, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 202, 255, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 178, 177, 5, 0, 0, 0, 0],
            &[0, 0, 0, 18, 97, 110, 46, 0, 48, 72, 19, 0],
            &[0, 0, 44, 234, 255, 255, 254, 120, 233, 255, 67, 0],
            &[0, 0, 196, 255, 255, 185, 180, 249, 255, 255, 67, 0],
            &[0, 28, 255, 255, 150, 0, 0, 113, 255, 255, 67, 0],
            &[0, 68, 255, 255, 68, 0, 0, 39, 255, 255, 67, 0],
            &[0, 78, 255, 255, 52, 0, 0, 15, 255, 255, 67, 0],
            &[0, 60, 255, 255, 82, 0, 0, 36, 255, 255, 67, 0],
            &[0, 14, 249, 255, 195, 11, 1, 150, 255, 255, 67, 0],
            &[0, 0, 156, 255, 255, 248, 241, 251, 255, 255, 67, 0],
            &[0, 0, 11, 174, 254, 255, 225, 109, 255, 255, 67, 0],
            &[0, 0, 0, 0, 20, 34, 2, 70, 255, 255, 54, 0],
            &[0, 0, 100, 79, 24, 2, 31, 187, 255, 247, 15, 0],
            &[0, 0, 172, 255, 255, 254, 255, 255, 255, 136, 0, 0],
            &[0, 0, 0, 2, 23, 36, 19, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĥ' / 0x124
        'Ĥ' => Some(&[
            &[0, 0, 0, 0, 95, 224, 224, 145, 0, 0, 0, 0],
            &[0, 0, 0, 37, 242, 224, 197, 255, 82, 0, 0, 0],
            &[0, 0, 0, 175, 190, 29, 11, 163, 201, 18, 0, 0],
            &[0, 8, 35, 35, 8, 0, 0, 1, 35, 35, 15, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 136, 101, 101, 105, 255, 255, 111, 0],
            &[0, 60, 255, 255, 255, 255, 255, 255, 255, 255, 111, 0],
            &[0, 60, 255, 255, 202, 186, 186, 188, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĥ' / 0x125
        'ĥ' => Some(&[
            &[7, 206, 251, 224, 253, 66, 0, 0, 0, 0, 0, 0],
            &[158, 244, 80, 23, 201, 233, 27, 0, 0, 0, 0, 0],
            &[14, 9, 0, 0, 3, 15, 4, 0, 0, 0, 0, 0],
            &[0, 15, 235, 235, 94, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 255, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 255, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 255, 100, 22, 94, 99, 45, 0, 0, 0],
            &[0, 16, 255, 255, 148, 241, 255, 255, 254, 123, 0, 0],
            &[0, 16, 255, 255, 255, 213, 164, 240, 255, 251, 27, 0],
            &[0, 16, 255, 255, 201, 2, 0, 80, 255, 255, 73, 0],
            &[0, 16, 255, 255, 125, 0, 0, 34, 255, 255, 89, 0],
            &[0, 16, 255, 255, 105, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 168, 168, 112, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ħ' / 0x126
        'Ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[1, 35, 35, 15, 0, 0, 7, 35, 35, 8, 0, 0],
            &[7, 255, 255, 111, 0, 0, 55, 255, 255, 63, 0, 0],
            &[133, 255, 255, 236, 125, 125, 180, 255, 255, 188, 25, 0],
            &[255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 51, 0],
            &[40, 255, 255, 143, 32, 32, 87, 255, 255, 95, 6, 0],
            &[7, 255, 255, 168, 101, 101, 134, 255, 255, 63, 0, 0],
            &[7, 255, 255, 255, 255, 255, 255, 255, 255, 63, 0, 0],
            &[7, 255, 255, 216, 186, 186, 201, 255, 255, 63, 0, 0],
            &[7, 255, 255, 111, 0, 0, 55, 255, 255, 63, 0, 0],
            &[7, 255, 255, 111, 0, 0, 55, 255, 255, 63, 0, 0],
            &[7, 255, 255, 111, 0, 0, 55, 255, 255, 63, 0, 0],
            &[7, 255, 255, 111, 0, 0, 55, 255, 255, 63, 0, 0],
            &[7, 255, 255, 111, 0, 0, 55, 255, 255, 63, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ħ' / 0x127
        'ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 15, 235, 235, 94, 0, 0, 0, 0, 0, 0, 0],
            &[0, 159, 255, 255, 245, 143, 131, 0, 0, 0, 0, 0],
            &[0, 255, 255, 255, 255, 255, 234, 0, 0, 0, 0, 0],
            &[0, 26, 255, 255, 110, 14, 59, 56, 13, 0, 0, 0],
            &[0, 16, 255, 255, 124, 216, 255, 255, 242, 88, 0, 0],
            &[0, 16, 255, 255, 252, 245, 207, 253, 255, 242, 18, 0],
            &[0, 16, 255, 255, 219, 14, 0, 102, 255, 255, 66, 0],
            &[0, 16, 255, 255, 132, 0, 0, 37, 255, 255, 89, 0],
            &[0, 16, 255, 255, 106, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĩ' / 0x128
        'Ĩ' => Some(&[
            &[0, 0, 0, 38, 157, 132, 28, 8, 177, 49, 0, 0],
            &[0, 0, 0, 206, 255, 255, 251, 233, 249, 24, 0, 0],
            &[0, 0, 13, 219, 61, 45, 153, 202, 96, 0, 0, 0],
            &[0, 0, 16, 35, 35, 35, 35, 35, 35, 23, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 70, 172, 206, 255, 255, 216, 177, 102, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 87, 207, 237, 255, 255, 242, 212, 126, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĩ' / 0x129
        'ĩ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 194, 169, 61, 24, 214, 56, 0, 0, 0],
            &[0, 0, 220, 239, 247, 255, 255, 242, 16, 0, 0, 0],
            &[0, 13, 182, 39, 16, 116, 165, 66, 0, 0, 0, 0],
            &[0, 37, 72, 72, 72, 72, 41, 0, 0, 0, 0, 0],
            &[0, 133, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0],
            &[0, 66, 148, 176, 248, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 149, 0, 0, 0, 0, 0],
            &[0, 153, 198, 239, 255, 255, 254, 222, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ī' / 0x12a
        'Ī' => Some(&[
            &[0, 0, 0, 13, 28, 28, 28, 28, 19, 0, 0, 0],
            &[0, 0, 0, 123, 255, 255, 255, 255, 170, 0, 0, 0],
            &[0, 0, 0, 70, 145, 145, 145, 145, 97, 0, 0, 0],
            &[0, 0, 16, 35, 35, 35, 35, 35, 35, 23, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 70, 172, 206, 255, 255, 216, 177, 102, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 87, 207, 237, 255, 255, 242, 212, 126, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ī' / 0x12b
        'ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 31, 65, 65, 65, 65, 43, 0, 0, 0, 0],
            &[0, 0, 123, 255, 255, 255, 255, 170, 0, 0, 0, 0],
            &[0, 0, 52, 109, 109, 109, 109, 72, 0, 0, 0, 0],
            &[0, 37, 72, 72, 72, 72, 41, 0, 0, 0, 0, 0],
            &[0, 133, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0],
            &[0, 66, 148, 176, 248, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 149, 0, 0, 0, 0, 0],
            &[0, 153, 198, 239, 255, 255, 254, 222, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => Some(&[
            &[0, 0, 0, 158, 196, 5, 0, 149, 205, 0, 0, 0],
            &[0, 0, 0, 88, 255, 228, 223, 255, 131, 0, 0, 0],
            &[0, 0, 0, 0, 98, 191, 196, 118, 4, 0, 0, 0],
            &[0, 0, 16, 35, 35, 35, 35, 35, 35, 23, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 70, 172, 206, 255, 255, 216, 177, 102, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 87, 207, 237, 255, 255, 242, 212, 126, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĭ' / 0x12d
        'ĭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 12, 0, 0, 8, 17, 0, 0, 0, 0],
            &[0, 0, 165, 220, 18, 10, 177, 215, 0, 0, 0, 0],
            &[0, 0, 67, 254, 252, 250, 255, 108, 0, 0, 0, 0],
            &[0, 0, 0, 62, 155, 160, 81, 0, 0, 0, 0, 0],
            &[0, 37, 72, 72, 72, 72, 41, 0, 0, 0, 0, 0],
            &[0, 133, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0],
            &[0, 66, 148, 176, 248, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 149, 0, 0, 0, 0, 0],
            &[0, 153, 198, 239, 255, 255, 254, 222, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Į' / 0x12e
        'Į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 35, 35, 35, 35, 35, 35, 23, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 70, 172, 206, 255, 255, 216, 177, 102, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 87, 207, 237, 255, 255, 242, 212, 126, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 3, 183, 205, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 124, 255, 177, 89, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0],
        ]),
        // letter: 'į' / 0x12f
        'į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 39, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 225, 255, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 211, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 103, 160, 62, 0, 0, 0, 0, 0],
            &[0, 37, 72, 72, 72, 72, 41, 0, 0, 0, 0, 0],
            &[0, 133, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0],
            &[0, 66, 148, 176, 248, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 149, 0, 0, 0, 0, 0],
            &[0, 153, 198, 239, 255, 255, 254, 222, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 14, 210, 175, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 131, 255, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 162, 255, 157, 70, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0],
        ]),
        // letter: 'İ' / 0x130
        'İ' => Some(&[
            &[0, 0, 0, 0, 40, 235, 247, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 95, 255, 255, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 176, 187, 42, 0, 0, 0, 0],
            &[0, 0, 16, 35, 35, 35, 35, 35, 35, 23, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 70, 172, 206, 255, 255, 216, 177, 102, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 255, 255, 65, 0, 0, 0, 0],
            &[0, 0, 87, 207, 237, 255, 255, 242, 212, 126, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 3, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ı' / 0x131
        'ı' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 72, 72, 72, 72, 41, 0, 0, 0, 0, 0],
            &[0, 133, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0],
            &[0, 66, 148, 176, 248, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 148, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 149, 0, 0, 0, 0, 0],
            &[0, 153, 198, 239, 255, 255, 254, 222, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĳ' / 0x132
        'Ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 35, 28, 0, 0, 23, 35, 26, 0, 0],
            &[0, 0, 142, 255, 206, 0, 0, 168, 255, 193, 0, 0],
            &[0, 0, 142, 255, 206, 0, 0, 168, 255, 193, 0, 0],
            &[0, 0, 142, 255, 206, 0, 0, 168, 255, 193, 0, 0],
            &[0, 0, 142, 255, 206, 0, 0, 168, 255, 193, 0, 0],
            &[0, 0, 142, 255, 206, 0, 0, 168, 255, 193, 0, 0],
            &[0, 0, 142, 255, 206, 0, 0, 168, 255, 193, 0, 0],
            &[0, 0, 142, 255, 206, 0, 0, 168, 255, 193, 0, 0],
            &[0, 0, 142, 255, 206, 0, 0, 169, 255, 192, 0, 0],
            &[0, 0, 46, 83, 67, 0, 0, 199, 255, 177, 0, 0],
            &[0, 0, 34, 31, 3, 12, 95, 251, 255, 133, 0, 0],
            &[0, 0, 137, 255, 254, 254, 255, 255, 244, 34, 0, 0],
            &[0, 0, 119, 252, 255, 255, 255, 202, 58, 0, 0, 0],
            &[0, 0, 0, 7, 34, 37, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĳ' / 0x133
        'ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 17, 0, 0, 9, 40, 4, 0, 0],
            &[0, 0, 129, 255, 244, 29, 1, 219, 255, 182, 0, 0],
            &[0, 0, 172, 255, 255, 57, 10, 253, 255, 222, 0, 0],
            &[0, 0, 44, 158, 122, 1, 0, 97, 160, 68, 0, 0],
            &[0, 0, 35, 72, 70, 0, 0, 58, 72, 47, 0, 0],
            &[0, 0, 124, 255, 250, 0, 0, 207, 255, 167, 0, 0],
            &[0, 0, 124, 255, 250, 0, 0, 207, 255, 167, 0, 0],
            &[0, 0, 124, 255, 250, 0, 0, 207, 255, 167, 0, 0],
            &[0, 0, 124, 255, 250, 0, 0, 207, 255, 167, 0, 0],
            &[0, 0, 124, 255, 250, 0, 0, 207, 255, 167, 0, 0],
            &[0, 0, 124, 255, 250, 0, 0, 207, 255, 167, 0, 0],
            &[0, 0, 124, 255, 250, 0, 0, 207, 255, 167, 0, 0],
            &[0, 0, 124, 255, 250, 0, 0, 207, 255, 167, 0, 0],
            &[0, 0, 124, 255, 250, 0, 0, 208, 255, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 232, 255, 163, 0, 0],
            &[0, 0, 0, 48, 9, 7, 96, 255, 255, 134, 0, 0],
            &[0, 0, 0, 245, 255, 254, 255, 255, 252, 48, 0, 0],
            &[0, 0, 0, 0, 17, 37, 26, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĵ' / 0x134
        'Ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 101, 224, 224, 138, 0, 0, 0],
            &[0, 0, 0, 0, 42, 244, 221, 200, 254, 76, 0, 0],
            &[0, 0, 0, 2, 179, 187, 26, 13, 167, 199, 15, 0],
            &[0, 0, 0, 0, 0, 2, 35, 35, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 255, 255, 84, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 255, 255, 84, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 255, 255, 84, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 255, 255, 84, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 255, 255, 84, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 255, 255, 84, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 255, 255, 84, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 255, 255, 84, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 255, 255, 69, 0, 0, 0],
            &[0, 35, 29, 2, 23, 171, 255, 254, 26, 0, 0, 0],
            &[0, 142, 255, 254, 255, 255, 255, 173, 0, 0, 0, 0],
            &[0, 123, 252, 255, 255, 245, 157, 14, 0, 0, 0, 0],
            &[0, 0, 10, 34, 32, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĵ' / 0x135
        'ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 5, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 186, 255, 255, 115, 0, 0, 0],
            &[0, 0, 0, 0, 117, 255, 160, 211, 248, 52, 0, 0],
            &[0, 0, 0, 25, 175, 120, 1, 17, 157, 147, 0, 0],
            &[0, 0, 0, 58, 72, 72, 72, 72, 21, 0, 0, 0],
            &[0, 0, 0, 206, 255, 255, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 105, 156, 195, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 255, 255, 71, 0, 0, 0],
            &[0, 21, 33, 2, 21, 175, 255, 255, 42, 0, 0, 0],
            &[0, 81, 255, 254, 255, 255, 255, 207, 1, 0, 0, 0],
            &[0, 0, 2, 27, 37, 14, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ķ' / 0x136
        'Ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 35, 35, 6, 0, 0, 4, 35, 35, 20, 0],
            &[0, 60, 255, 255, 45, 0, 0, 129, 255, 249, 49, 0],
            &[0, 60, 255, 255, 45, 0, 49, 250, 255, 119, 0, 0],
            &[0, 60, 255, 255, 45, 5, 209, 255, 192, 2, 0, 0],
            &[0, 60, 255, 255, 45, 129, 255, 239, 30, 0, 0, 0],
            &[0, 60, 255, 255, 90, 250, 255, 89, 0, 0, 0, 0],
            &[0, 60, 255, 255, 223, 255, 238, 8, 0, 0, 0, 0],
            &[0, 60, 255, 255, 255, 255, 255, 120, 0, 0, 0, 0],
            &[0, 60, 255, 255, 176, 151, 255, 241, 21, 0, 0, 0],
            &[0, 60, 255, 255, 45, 21, 243, 255, 148, 0, 0, 0],
            &[0, 60, 255, 255, 45, 0, 129, 255, 251, 39, 0, 0],
            &[0, 60, 255, 255, 45, 0, 14, 235, 255, 176, 0, 0],
            &[0, 60, 255, 255, 45, 0, 0, 114, 255, 255, 62, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 241, 246, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 255, 179, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 76, 193, 37, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ķ' / 0x137
        'ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 230, 235, 114, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 250, 255, 124, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 250, 255, 124, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 250, 255, 124, 0, 0, 18, 72, 72, 43, 0, 0],
            &[0, 250, 255, 124, 0, 7, 196, 255, 232, 32, 0, 0],
            &[0, 250, 255, 124, 0, 163, 255, 243, 51, 0, 0, 0],
            &[0, 250, 255, 120, 123, 255, 251, 73, 0, 0, 0, 0],
            &[0, 250, 255, 172, 253, 255, 135, 0, 0, 0, 0, 0],
            &[0, 250, 255, 255, 255, 255, 226, 14, 0, 0, 0, 0],
            &[0, 250, 255, 245, 106, 231, 255, 155, 0, 0, 0, 0],
            &[0, 250, 255, 127, 0, 83, 255, 255, 71, 0, 0, 0],
            &[0, 250, 255, 124, 0, 0, 173, 255, 226, 14, 0, 0],
            &[0, 250, 255, 124, 0, 0, 24, 238, 255, 156, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 190, 246, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 243, 227, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 33, 193, 80, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĸ' / 0x138
        'ĸ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 70, 72, 34, 0, 0, 18, 72, 72, 43, 0, 0],
            &[0, 250, 255, 124, 0, 7, 196, 255, 232, 32, 0, 0],
            &[0, 250, 255, 124, 0, 163, 255, 243, 51, 0, 0, 0],
            &[0, 250, 255, 120, 123, 255, 251, 73, 0, 0, 0, 0],
            &[0, 250, 255, 172, 253, 255, 135, 0, 0, 0, 0, 0],
            &[0, 250, 255, 255, 255, 255, 226, 14, 0, 0, 0, 0],
            &[0, 250, 255, 245, 106, 231, 255, 155, 0, 0, 0, 0],
            &[0, 250, 255, 127, 0, 83, 255, 255, 71, 0, 0, 0],
            &[0, 250, 255, 124, 0, 0, 173, 255, 226, 14, 0, 0],
            &[0, 250, 255, 124, 0, 0, 24, 238, 255, 156, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĺ' / 0x139
        'Ĺ' => Some(&[
            &[0, 0, 17, 211, 224, 145, 0, 0, 0, 0, 0, 0],
            &[0, 0, 163, 255, 174, 7, 0, 0, 0, 0, 0, 0],
            &[0, 58, 215, 144, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 35, 35, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 45, 45, 45, 45, 45, 0, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĺ' / 0x13a
        'ĺ' => Some(&[
            &[0, 0, 0, 0, 40, 247, 249, 83, 0, 0, 0, 0],
            &[0, 0, 0, 3, 205, 240, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 15, 9, 0, 0, 0, 0, 0, 0],
            &[0, 90, 235, 235, 235, 235, 128, 0, 0, 0, 0, 0],
            &[0, 64, 202, 249, 255, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 237, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 236, 255, 140, 0, 0, 0, 0, 0],
            &[0, 153, 198, 240, 255, 255, 254, 221, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 99, 168, 164, 23, 0, 0, 0],
        ]),
        // letter: 'Ļ' / 0x13b
        'Ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 35, 35, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 45, 45, 45, 45, 45, 0, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 153, 246, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 247, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 191, 110, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ļ' / 0x13c
        'ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 235, 235, 235, 235, 128, 0, 0, 0, 0, 0],
            &[0, 64, 202, 249, 255, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 237, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 236, 255, 140, 0, 0, 0, 0, 0],
            &[0, 153, 198, 240, 255, 255, 254, 221, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 215, 246, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 254, 208, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 53, 193, 60, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ľ' / 0x13d
        'Ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 35, 35, 0, 0, 12, 35, 32, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 112, 255, 180, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 163, 254, 51, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 121, 110, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 45, 45, 45, 45, 45, 0, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ľ' / 0x13e
        'ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 235, 235, 235, 235, 128, 51, 235, 220, 8, 0],
            &[0, 64, 202, 249, 255, 255, 139, 102, 255, 128, 0, 0],
            &[0, 0, 0, 6, 237, 255, 139, 123, 191, 11, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 236, 255, 140, 0, 0, 0, 0, 0],
            &[0, 153, 198, 240, 255, 255, 254, 221, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 35, 35, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 16, 3, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 129, 255, 225, 17, 0, 0],
            &[0, 120, 255, 254, 0, 0, 197, 255, 255, 56, 0, 0],
            &[0, 120, 255, 254, 0, 0, 83, 213, 168, 5, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 255, 254, 45, 45, 45, 45, 45, 0, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 120, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŀ' / 0x140
        'ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[90, 235, 235, 235, 235, 128, 0, 0, 0, 0, 0, 0],
            &[64, 202, 249, 255, 255, 139, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 237, 255, 139, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 255, 139, 0, 13, 190, 222, 82, 0],
            &[0, 0, 0, 235, 255, 139, 0, 69, 255, 255, 184, 0],
            &[0, 0, 0, 235, 255, 139, 0, 20, 223, 252, 104, 0],
            &[0, 0, 0, 235, 255, 139, 0, 0, 1, 7, 0, 0],
            &[0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 236, 255, 140, 0, 0, 0, 0, 0, 0],
            &[153, 198, 240, 255, 255, 254, 221, 175, 50, 0, 0, 0],
            &[250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ł' / 0x141
        'Ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 35, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 202, 255, 172, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 202, 255, 172, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 202, 255, 172, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 202, 255, 172, 62, 14, 0, 0, 0, 0],
            &[0, 0, 0, 202, 255, 255, 255, 141, 0, 0, 0, 0],
            &[0, 0, 2, 255, 255, 255, 221, 74, 0, 0, 0, 0],
            &[0, 26, 203, 255, 255, 255, 8, 0, 0, 0, 0, 0],
            &[0, 4, 214, 255, 255, 172, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 202, 255, 172, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 202, 255, 187, 45, 45, 45, 45, 31, 0],
            &[0, 0, 0, 202, 255, 255, 255, 255, 255, 255, 176, 0],
            &[0, 0, 0, 202, 255, 255, 255, 255, 255, 255, 176, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ł' / 0x142
        'ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 235, 235, 235, 235, 128, 0, 0, 0, 0, 0],
            &[0, 64, 202, 249, 255, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 237, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 148, 88, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 255, 255, 67, 0, 0, 0],
            &[0, 0, 0, 27, 255, 255, 255, 158, 20, 0, 0, 0],
            &[0, 0, 112, 242, 255, 255, 199, 0, 0, 0, 0, 0],
            &[0, 0, 172, 243, 255, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 27, 28, 235, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 236, 255, 140, 0, 0, 0, 0, 0],
            &[0, 153, 198, 240, 255, 255, 254, 221, 175, 50, 0, 0],
            &[0, 250, 255, 255, 255, 255, 255, 255, 255, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ń' / 0x143
        'Ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 128, 224, 211, 34, 0, 0],
            &[0, 0, 0, 0, 0, 53, 251, 239, 57, 0, 0, 0],
            &[0, 0, 0, 0, 0, 177, 201, 43, 0, 0, 0, 0],
            &[0, 8, 35, 35, 35, 4, 0, 0, 26, 35, 15, 0],
            &[0, 60, 255, 255, 255, 82, 0, 0, 193, 255, 111, 0],
            &[0, 60, 255, 255, 255, 166, 0, 0, 193, 255, 111, 0],
            &[0, 60, 255, 222, 248, 242, 7, 0, 193, 255, 111, 0],
            &[0, 60, 255, 221, 179, 255, 77, 0, 193, 255, 111, 0],
            &[0, 60, 255, 234, 96, 255, 161, 0, 193, 255, 111, 0],
            &[0, 60, 255, 243, 18, 251, 239, 5, 193, 255, 111, 0],
            &[0, 60, 255, 245, 0, 187, 255, 72, 191, 255, 111, 0],
            &[0, 60, 255, 245, 0, 104, 255, 156, 186, 255, 111, 0],
            &[0, 60, 255, 245, 0, 23, 253, 236, 182, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 194, 255, 233, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 111, 255, 255, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 29, 254, 255, 255, 111, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ń' / 0x144
        'ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 5, 5, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 164, 255, 226, 29, 0, 0],
            &[0, 0, 0, 0, 0, 78, 255, 219, 32, 0, 0, 0],
            &[0, 0, 0, 0, 0, 158, 163, 21, 0, 0, 0, 0],
            &[0, 4, 72, 61, 0, 24, 93, 100, 48, 0, 0, 0],
            &[0, 16, 255, 252, 99, 247, 255, 255, 254, 127, 0, 0],
            &[0, 16, 255, 255, 255, 211, 164, 242, 255, 251, 28, 0],
            &[0, 16, 255, 255, 214, 7, 0, 84, 255, 255, 73, 0],
            &[0, 16, 255, 255, 132, 0, 0, 34, 255, 255, 89, 0],
            &[0, 16, 255, 255, 106, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ņ' / 0x145
        'Ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 35, 35, 35, 4, 0, 0, 26, 35, 15, 0],
            &[0, 60, 255, 255, 255, 82, 0, 0, 193, 255, 111, 0],
            &[0, 60, 255, 255, 255, 166, 0, 0, 193, 255, 111, 0],
            &[0, 60, 255, 222, 248, 242, 7, 0, 193, 255, 111, 0],
            &[0, 60, 255, 221, 179, 255, 77, 0, 193, 255, 111, 0],
            &[0, 60, 255, 234, 96, 255, 161, 0, 193, 255, 111, 0],
            &[0, 60, 255, 243, 18, 251, 239, 5, 193, 255, 111, 0],
            &[0, 60, 255, 245, 0, 187, 255, 72, 191, 255, 111, 0],
            &[0, 60, 255, 245, 0, 104, 255, 156, 186, 255, 111, 0],
            &[0, 60, 255, 245, 0, 23, 253, 236, 182, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 194, 255, 233, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 111, 255, 255, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 29, 254, 255, 255, 111, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 246, 238, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 255, 144, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 103, 187, 17, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ņ' / 0x146
        'ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 72, 61, 0, 24, 93, 100, 48, 0, 0, 0],
            &[0, 16, 255, 252, 99, 247, 255, 255, 254, 127, 0, 0],
            &[0, 16, 255, 255, 255, 211, 164, 242, 255, 251, 28, 0],
            &[0, 16, 255, 255, 214, 7, 0, 84, 255, 255, 73, 0],
            &[0, 16, 255, 255, 132, 0, 0, 34, 255, 255, 89, 0],
            &[0, 16, 255, 255, 106, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 246, 243, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 255, 162, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 89, 191, 26, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ň' / 0x147
        'Ň' => Some(&[
            &[0, 0, 0, 160, 207, 45, 9, 163, 215, 30, 0, 0],
            &[0, 0, 0, 21, 229, 238, 199, 255, 93, 0, 0, 0],
            &[0, 0, 0, 0, 70, 215, 215, 151, 0, 0, 0, 0],
            &[0, 8, 35, 35, 35, 4, 0, 0, 26, 35, 15, 0],
            &[0, 60, 255, 255, 255, 82, 0, 0, 193, 255, 111, 0],
            &[0, 60, 255, 255, 255, 166, 0, 0, 193, 255, 111, 0],
            &[0, 60, 255, 222, 248, 242, 7, 0, 193, 255, 111, 0],
            &[0, 60, 255, 221, 179, 255, 77, 0, 193, 255, 111, 0],
            &[0, 60, 255, 234, 96, 255, 161, 0, 193, 255, 111, 0],
            &[0, 60, 255, 243, 18, 251, 239, 5, 193, 255, 111, 0],
            &[0, 60, 255, 245, 0, 187, 255, 72, 191, 255, 111, 0],
            &[0, 60, 255, 245, 0, 104, 255, 156, 186, 255, 111, 0],
            &[0, 60, 255, 245, 0, 23, 253, 236, 182, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 194, 255, 233, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 111, 255, 255, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 29, 254, 255, 255, 111, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ň' / 0x148
        'ň' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 3, 0, 0, 1, 5, 1, 0, 0],
            &[0, 0, 0, 171, 239, 68, 24, 202, 232, 26, 0, 0],
            &[0, 0, 0, 11, 215, 248, 223, 252, 64, 0, 0, 0],
            &[0, 0, 0, 0, 53, 178, 178, 114, 0, 0, 0, 0],
            &[0, 4, 72, 61, 0, 24, 93, 100, 48, 0, 0, 0],
            &[0, 16, 255, 252, 99, 247, 255, 255, 254, 127, 0, 0],
            &[0, 16, 255, 255, 255, 211, 164, 242, 255, 251, 28, 0],
            &[0, 16, 255, 255, 214, 7, 0, 84, 255, 255, 73, 0],
            &[0, 16, 255, 255, 132, 0, 0, 34, 255, 255, 89, 0],
            &[0, 16, 255, 255, 106, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŉ' / 0x149
        'ŉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[4, 35, 35, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[54, 255, 255, 52, 0, 0, 0, 0, 0, 0, 0, 0],
            &[101, 255, 225, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[152, 255, 132, 64, 71, 1, 15, 89, 102, 51, 0, 0],
            &[208, 252, 30, 229, 255, 109, 237, 255, 255, 255, 131, 0],
            &[6, 6, 0, 229, 255, 255, 222, 164, 239, 255, 252, 29],
            &[0, 0, 0, 229, 255, 241, 19, 0, 82, 255, 255, 73],
            &[0, 0, 0, 229, 255, 175, 0, 0, 34, 255, 255, 89],
            &[0, 0, 0, 229, 255, 150, 0, 0, 29, 255, 255, 89],
            &[0, 0, 0, 229, 255, 145, 0, 0, 29, 255, 255, 89],
            &[0, 0, 0, 229, 255, 145, 0, 0, 29, 255, 255, 89],
            &[0, 0, 0, 229, 255, 145, 0, 0, 29, 255, 255, 89],
            &[0, 0, 0, 229, 255, 145, 0, 0, 29, 255, 255, 89],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 35, 35, 35, 4, 0, 0, 26, 35, 15, 0],
            &[0, 60, 255, 255, 255, 82, 0, 0, 193, 255, 111, 0],
            &[0, 60, 255, 255, 255, 166, 0, 0, 193, 255, 111, 0],
            &[0, 60, 255, 222, 248, 242, 7, 0, 193, 255, 111, 0],
            &[0, 60, 255, 221, 179, 255, 77, 0, 193, 255, 111, 0],
            &[0, 60, 255, 234, 96, 255, 161, 0, 193, 255, 111, 0],
            &[0, 60, 255, 243, 18, 251, 239, 5, 193, 255, 111, 0],
            &[0, 60, 255, 245, 0, 187, 255, 72, 191, 255, 111, 0],
            &[0, 60, 255, 245, 0, 104, 255, 156, 186, 255, 111, 0],
            &[0, 60, 255, 245, 0, 23, 253, 236, 182, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 194, 255, 233, 255, 111, 0],
            &[0, 60, 255, 245, 0, 0, 111, 255, 255, 255, 208, 0],
            &[0, 60, 255, 245, 0, 0, 29, 254, 255, 255, 222, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 180, 255, 100, 0],
            &[0, 0, 0, 0, 0, 82, 72, 130, 252, 255, 44, 0],
            &[0, 0, 0, 0, 0, 244, 255, 255, 252, 130, 0, 0],
            &[0, 0, 0, 0, 0, 105, 131, 105, 39, 0, 0, 0],
        ]),
        // letter: 'ŋ' / 0x14b
        'ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 72, 61, 0, 24, 93, 100, 48, 0, 0, 0],
            &[0, 16, 255, 252, 99, 247, 255, 255, 254, 127, 0, 0],
            &[0, 16, 255, 255, 255, 211, 164, 242, 255, 251, 28, 0],
            &[0, 16, 255, 255, 214, 7, 0, 84, 255, 255, 73, 0],
            &[0, 16, 255, 255, 132, 0, 0, 34, 255, 255, 89, 0],
            &[0, 16, 255, 255, 106, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 29, 255, 255, 89, 0],
            &[0, 16, 255, 255, 102, 0, 0, 55, 255, 255, 169, 0],
            &[0, 16, 255, 255, 102, 0, 0, 58, 255, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 30, 255, 255, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 77, 0],
            &[0, 0, 0, 0, 0, 35, 235, 255, 255, 245, 20, 0],
            &[0, 0, 0, 0, 0, 0, 4, 13, 0, 0, 0, 0],
        ]),
        // letter: 'Ō' / 0x14c
        'Ō' => Some(&[
            &[0, 0, 0, 13, 28, 28, 28, 28, 19, 0, 0, 0],
            &[0, 0, 0, 123, 255, 255, 255, 255, 170, 0, 0, 0],
            &[0, 0, 0, 70, 145, 145, 145, 145, 97, 0, 0, 0],
            &[0, 0, 0, 0, 22, 67, 77, 34, 0, 0, 0, 0],
            &[0, 0, 0, 135, 249, 255, 255, 255, 175, 13, 0, 0],
            &[0, 0, 115, 255, 255, 232, 228, 255, 255, 170, 0, 0],
            &[0, 6, 236, 255, 187, 6, 0, 148, 255, 255, 41, 0],
            &[0, 58, 255, 255, 71, 0, 0, 28, 255, 255, 114, 0],
            &[0, 100, 255, 255, 24, 0, 0, 0, 232, 255, 154, 0],
            &[0, 118, 255, 255, 6, 0, 0, 0, 214, 255, 171, 0],
            &[0, 118, 255, 255, 8, 0, 0, 0, 213, 255, 169, 0],
            &[0, 98, 255, 255, 29, 0, 0, 0, 235, 255, 149, 0],
            &[0, 53, 255, 255, 84, 0, 0, 36, 255, 255, 104, 0],
            &[0, 2, 227, 255, 210, 27, 12, 176, 255, 251, 28, 0],
            &[0, 0, 90, 255, 255, 255, 253, 255, 255, 140, 0, 0],
            &[0, 0, 0, 95, 228, 255, 255, 240, 131, 3, 0, 0],
            &[0, 0, 0, 0, 1, 28, 33, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ō' / 0x14d
        'ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 65, 65, 65, 65, 43, 0, 0, 0],
            &[0, 0, 0, 123, 255, 255, 255, 255, 170, 0, 0, 0],
            &[0, 0, 0, 52, 109, 109, 109, 109, 72, 0, 0, 0],
            &[0, 0, 0, 0, 40, 96, 104, 62, 1, 0, 0, 0],
            &[0, 0, 8, 163, 255, 255, 255, 255, 205, 30, 0, 0],
            &[0, 0, 143, 255, 255, 179, 155, 245, 255, 201, 1, 0],
            &[0, 12, 246, 255, 155, 0, 0, 104, 255, 255, 59, 0],
            &[0, 59, 255, 255, 69, 0, 0, 23, 255, 255, 112, 0],
            &[0, 76, 255, 255, 54, 0, 0, 4, 255, 255, 127, 0],
            &[0, 54, 255, 255, 88, 0, 0, 35, 255, 255, 104, 0],
            &[0, 6, 235, 255, 205, 9, 0, 153, 255, 254, 37, 0],
            &[0, 0, 104, 255, 255, 238, 226, 255, 255, 155, 0, 0],
            &[0, 0, 0, 100, 228, 255, 255, 241, 135, 4, 0, 0],
            &[0, 0, 0, 0, 1, 29, 34, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => Some(&[
            &[0, 0, 0, 158, 196, 5, 0, 149, 205, 0, 0, 0],
            &[0, 0, 0, 88, 255, 228, 223, 255, 131, 0, 0, 0],
            &[0, 0, 0, 0, 98, 191, 196, 118, 4, 0, 0, 0],
            &[0, 0, 0, 0, 22, 67, 77, 34, 0, 0, 0, 0],
            &[0, 0, 0, 135, 249, 255, 255, 255, 175, 13, 0, 0],
            &[0, 0, 115, 255, 255, 232, 228, 255, 255, 170, 0, 0],
            &[0, 6, 236, 255, 187, 6, 0, 148, 255, 255, 41, 0],
            &[0, 58, 255, 255, 71, 0, 0, 28, 255, 255, 114, 0],
            &[0, 100, 255, 255, 24, 0, 0, 0, 232, 255, 154, 0],
            &[0, 118, 255, 255, 6, 0, 0, 0, 214, 255, 171, 0],
            &[0, 118, 255, 255, 8, 0, 0, 0, 213, 255, 169, 0],
            &[0, 98, 255, 255, 29, 0, 0, 0, 235, 255, 149, 0],
            &[0, 53, 255, 255, 84, 0, 0, 36, 255, 255, 104, 0],
            &[0, 2, 227, 255, 210, 27, 12, 176, 255, 251, 28, 0],
            &[0, 0, 90, 255, 255, 255, 253, 255, 255, 140, 0, 0],
            &[0, 0, 0, 95, 228, 255, 255, 240, 131, 3, 0, 0],
            &[0, 0, 0, 0, 1, 28, 33, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŏ' / 0x14f
        'ŏ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 12, 0, 0, 8, 17, 0, 0, 0],
            &[0, 0, 0, 165, 220, 18, 10, 177, 215, 0, 0, 0],
            &[0, 0, 0, 67, 254, 252, 250, 255, 108, 0, 0, 0],
            &[0, 0, 0, 0, 62, 155, 160, 81, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 96, 104, 62, 1, 0, 0, 0],
            &[0, 0, 8, 163, 255, 255, 255, 255, 205, 30, 0, 0],
            &[0, 0, 143, 255, 255, 179, 155, 245, 255, 201, 1, 0],
            &[0, 12, 246, 255, 155, 0, 0, 104, 255, 255, 59, 0],
            &[0, 59, 255, 255, 69, 0, 0, 23, 255, 255, 112, 0],
            &[0, 76, 255, 255, 54, 0, 0, 4, 255, 255, 127, 0],
            &[0, 54, 255, 255, 88, 0, 0, 35, 255, 255, 104, 0],
            &[0, 6, 235, 255, 205, 9, 0, 153, 255, 254, 37, 0],
            &[0, 0, 104, 255, 255, 238, 226, 255, 255, 155, 0, 0],
            &[0, 0, 0, 100, 228, 255, 255, 241, 135, 4, 0, 0],
            &[0, 0, 0, 0, 1, 29, 34, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ő' / 0x150
        'Ő' => Some(&[
            &[0, 0, 0, 0, 165, 224, 124, 154, 224, 133, 0, 0],
            &[0, 0, 0, 69, 255, 184, 64, 254, 192, 10, 0, 0],
            &[0, 0, 0, 173, 156, 7, 163, 166, 10, 0, 0, 0],
            &[0, 0, 0, 0, 22, 67, 77, 34, 0, 0, 0, 0],
            &[0, 0, 0, 135, 249, 255, 255, 255, 175, 13, 0, 0],
            &[0, 0, 115, 255, 255, 232, 228, 255, 255, 170, 0, 0],
            &[0, 6, 236, 255, 187, 6, 0, 148, 255, 255, 41, 0],
            &[0, 58, 255, 255, 71, 0, 0, 28, 255, 255, 114, 0],
            &[0, 100, 255, 255, 24, 0, 0, 0, 232, 255, 154, 0],
            &[0, 118, 255, 255, 6, 0, 0, 0, 214, 255, 171, 0],
            &[0, 118, 255, 255, 8, 0, 0, 0, 213, 255, 169, 0],
            &[0, 98, 255, 255, 29, 0, 0, 0, 235, 255, 149, 0],
            &[0, 53, 255, 255, 84, 0, 0, 36, 255, 255, 104, 0],
            &[0, 2, 227, 255, 210, 27, 12, 176, 255, 251, 28, 0],
            &[0, 0, 90, 255, 255, 255, 253, 255, 255, 140, 0, 0],
            &[0, 0, 0, 95, 228, 255, 255, 240, 131, 3, 0, 0],
            &[0, 0, 0, 0, 1, 28, 33, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ő' / 0x151
        'ő' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 5, 4, 2, 5, 4, 0, 0],
            &[0, 0, 0, 0, 199, 255, 126, 187, 255, 137, 0, 0],
            &[0, 0, 0, 91, 255, 153, 79, 255, 163, 2, 0, 0],
            &[0, 0, 0, 150, 119, 0, 142, 129, 2, 0, 0, 0],
            &[0, 0, 0, 0, 40, 96, 104, 62, 1, 0, 0, 0],
            &[0, 0, 8, 163, 255, 255, 255, 255, 205, 30, 0, 0],
            &[0, 0, 143, 255, 255, 179, 155, 245, 255, 201, 1, 0],
            &[0, 12, 246, 255, 155, 0, 0, 104, 255, 255, 59, 0],
            &[0, 59, 255, 255, 69, 0, 0, 23, 255, 255, 112, 0],
            &[0, 76, 255, 255, 54, 0, 0, 4, 255, 255, 127, 0],
            &[0, 54, 255, 255, 88, 0, 0, 35, 255, 255, 104, 0],
            &[0, 6, 235, 255, 205, 9, 0, 153, 255, 254, 37, 0],
            &[0, 0, 104, 255, 255, 238, 226, 255, 255, 155, 0, 0],
            &[0, 0, 0, 100, 228, 255, 255, 241, 135, 4, 0, 0],
            &[0, 0, 0, 0, 1, 29, 34, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Œ' / 0x152
        'Œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 74, 44, 35, 35, 35, 30, 0],
            &[0, 0, 18, 197, 255, 255, 255, 255, 255, 255, 220, 0],
            &[0, 0, 162, 255, 255, 229, 252, 255, 252, 248, 213, 0],
            &[0, 16, 251, 255, 148, 0, 199, 255, 135, 0, 0, 0],
            &[0, 73, 255, 255, 45, 0, 199, 255, 135, 0, 0, 0],
            &[0, 107, 255, 254, 3, 0, 199, 255, 179, 92, 51, 0],
            &[0, 121, 255, 242, 0, 0, 199, 255, 255, 255, 141, 0],
            &[0, 117, 255, 245, 0, 0, 199, 255, 225, 191, 105, 0],
            &[0, 102, 255, 255, 5, 0, 199, 255, 135, 0, 0, 0],
            &[0, 64, 255, 255, 35, 0, 199, 255, 135, 0, 0, 0],
            &[0, 12, 244, 255, 149, 5, 205, 255, 148, 27, 23, 0],
            &[0, 0, 145, 255, 255, 254, 255, 255, 255, 255, 220, 0],
            &[0, 0, 8, 165, 254, 255, 255, 255, 255, 255, 220, 0],
            &[0, 0, 0, 0, 12, 37, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'œ' / 0x153
        'œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 76, 101, 30, 0, 46, 96, 34, 0, 0, 0],
            &[0, 173, 255, 255, 241, 122, 253, 255, 248, 54, 0, 0],
            &[63, 255, 215, 130, 255, 255, 211, 97, 255, 179, 0, 0],
            &[132, 255, 116, 0, 220, 255, 112, 0, 220, 238, 0, 0],
            &[158, 255, 88, 0, 185, 255, 185, 147, 234, 255, 3, 0],
            &[172, 255, 82, 0, 175, 255, 245, 239, 239, 239, 7, 0],
            &[157, 255, 93, 0, 190, 255, 92, 0, 0, 0, 0, 0],
            &[111, 255, 135, 5, 236, 255, 154, 0, 0, 9, 0, 0],
            &[28, 250, 241, 194, 255, 250, 254, 162, 163, 191, 0, 0],
            &[0, 98, 247, 255, 200, 53, 224, 255, 252, 145, 0, 0],
            &[0, 0, 12, 32, 0, 0, 4, 35, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŕ' / 0x154
        'Ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 147, 224, 203, 23, 0, 0, 0],
            &[0, 0, 0, 0, 71, 255, 230, 44, 0, 0, 0, 0],
            &[0, 0, 0, 5, 190, 194, 32, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 33, 16, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 220, 105, 0, 0, 0, 0],
            &[0, 163, 255, 253, 247, 255, 255, 255, 110, 0, 0, 0],
            &[0, 163, 255, 211, 0, 29, 206, 255, 224, 0, 0, 0],
            &[0, 163, 255, 211, 0, 0, 128, 255, 254, 1, 0, 0],
            &[0, 163, 255, 211, 0, 4, 190, 255, 229, 0, 0, 0],
            &[0, 163, 255, 240, 174, 220, 255, 255, 112, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 125, 2, 0, 0, 0],
            &[0, 163, 255, 230, 119, 253, 255, 86, 0, 0, 0, 0],
            &[0, 163, 255, 211, 0, 156, 255, 225, 9, 0, 0, 0],
            &[0, 163, 255, 211, 0, 25, 244, 255, 127, 0, 0, 0],
            &[0, 163, 255, 211, 0, 0, 128, 255, 245, 29, 0, 0],
            &[0, 163, 255, 211, 0, 0, 11, 231, 255, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŕ' / 0x155
        'ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 5, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 53, 253, 255, 113, 0, 0],
            &[0, 0, 0, 0, 0, 8, 212, 254, 111, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 178, 85, 0, 0, 0, 0],
            &[0, 0, 60, 72, 72, 71, 1, 8, 84, 109, 65, 0],
            &[0, 0, 215, 255, 255, 255, 61, 213, 255, 255, 184, 0],
            &[0, 0, 114, 170, 248, 255, 237, 255, 245, 247, 110, 0],
            &[0, 0, 0, 0, 227, 255, 255, 102, 1, 2, 6, 0],
            &[0, 0, 0, 0, 227, 255, 200, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 228, 255, 148, 0, 0, 0, 0, 0],
            &[0, 43, 179, 232, 255, 255, 254, 216, 162, 0, 0, 0],
            &[0, 77, 255, 255, 255, 255, 255, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŗ' / 0x156
        'Ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 33, 16, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 220, 105, 0, 0, 0, 0],
            &[0, 163, 255, 253, 247, 255, 255, 255, 110, 0, 0, 0],
            &[0, 163, 255, 211, 0, 29, 206, 255, 224, 0, 0, 0],
            &[0, 163, 255, 211, 0, 0, 128, 255, 254, 1, 0, 0],
            &[0, 163, 255, 211, 0, 4, 190, 255, 229, 0, 0, 0],
            &[0, 163, 255, 240, 174, 220, 255, 255, 112, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 125, 2, 0, 0, 0],
            &[0, 163, 255, 230, 119, 253, 255, 86, 0, 0, 0, 0],
            &[0, 163, 255, 211, 0, 156, 255, 225, 9, 0, 0, 0],
            &[0, 163, 255, 211, 0, 25, 244, 255, 127, 0, 0, 0],
            &[0, 163, 255, 211, 0, 0, 128, 255, 245, 29, 0, 0],
            &[0, 163, 255, 211, 0, 0, 11, 231, 255, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 173, 246, 117, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 229, 237, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 20, 193, 93, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŗ' / 0x157
        'ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 72, 72, 71, 1, 8, 84, 109, 65, 0],
            &[0, 0, 215, 255, 255, 255, 61, 213, 255, 255, 184, 0],
            &[0, 0, 114, 170, 248, 255, 237, 255, 245, 247, 110, 0],
            &[0, 0, 0, 0, 227, 255, 255, 102, 1, 2, 6, 0],
            &[0, 0, 0, 0, 227, 255, 200, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 228, 255, 148, 0, 0, 0, 0, 0],
            &[0, 43, 179, 232, 255, 255, 254, 216, 162, 0, 0, 0],
            &[0, 77, 255, 255, 255, 255, 255, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 199, 246, 91, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 249, 221, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 41, 193, 73, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ř' / 0x158
        'Ř' => Some(&[
            &[0, 0, 179, 199, 34, 16, 176, 207, 18, 0, 0, 0],
            &[0, 0, 33, 239, 229, 207, 254, 73, 0, 0, 0, 0],
            &[0, 0, 0, 89, 215, 215, 133, 0, 0, 0, 0, 0],
            &[0, 22, 35, 35, 33, 16, 0, 0, 0, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 220, 105, 0, 0, 0, 0],
            &[0, 163, 255, 253, 247, 255, 255, 255, 110, 0, 0, 0],
            &[0, 163, 255, 211, 0, 29, 206, 255, 224, 0, 0, 0],
            &[0, 163, 255, 211, 0, 0, 128, 255, 254, 1, 0, 0],
            &[0, 163, 255, 211, 0, 4, 190, 255, 229, 0, 0, 0],
            &[0, 163, 255, 240, 174, 220, 255, 255, 112, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 125, 2, 0, 0, 0],
            &[0, 163, 255, 230, 119, 253, 255, 86, 0, 0, 0, 0],
            &[0, 163, 255, 211, 0, 156, 255, 225, 9, 0, 0, 0],
            &[0, 163, 255, 211, 0, 25, 244, 255, 127, 0, 0, 0],
            &[0, 163, 255, 211, 0, 0, 128, 255, 245, 29, 0, 0],
            &[0, 163, 255, 211, 0, 0, 11, 231, 255, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ř' / 0x159
        'ř' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 5, 0, 0, 0, 4, 4, 0, 0],
            &[0, 0, 0, 63, 250, 160, 4, 116, 252, 116, 0, 0],
            &[0, 0, 0, 0, 114, 255, 216, 255, 173, 0, 0, 0],
            &[0, 0, 0, 0, 1, 151, 178, 174, 19, 0, 0, 0],
            &[0, 0, 60, 72, 72, 71, 1, 8, 84, 109, 65, 0],
            &[0, 0, 215, 255, 255, 255, 61, 213, 255, 255, 184, 0],
            &[0, 0, 114, 170, 248, 255, 237, 255, 245, 247, 110, 0],
            &[0, 0, 0, 0, 227, 255, 255, 102, 1, 2, 6, 0],
            &[0, 0, 0, 0, 227, 255, 200, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 255, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 228, 255, 148, 0, 0, 0, 0, 0],
            &[0, 43, 179, 232, 255, 255, 254, 216, 162, 0, 0, 0],
            &[0, 77, 255, 255, 255, 255, 255, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ś' / 0x15a
        'Ś' => Some(&[
            &[0, 0, 0, 0, 0, 94, 224, 221, 59, 0, 0, 0],
            &[0, 0, 0, 0, 27, 238, 249, 85, 0, 0, 0, 0],
            &[0, 0, 0, 0, 144, 210, 66, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 58, 72, 52, 15, 0, 0, 0, 0],
            &[0, 0, 108, 243, 255, 255, 255, 253, 181, 0, 0, 0],
            &[0, 62, 255, 255, 240, 197, 220, 255, 160, 0, 0, 0],
            &[0, 132, 255, 254, 32, 0, 0, 35, 31, 0, 0, 0],
            &[0, 129, 255, 254, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 253, 255, 219, 76, 0, 0, 0, 0, 0, 0],
            &[0, 0, 102, 249, 255, 255, 194, 52, 0, 0, 0, 0],
            &[0, 0, 0, 43, 192, 255, 255, 253, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 226, 255, 250, 21, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 255, 255, 60, 0, 0],
            &[0, 134, 77, 10, 0, 3, 150, 255, 255, 35, 0, 0],
            &[0, 207, 255, 254, 233, 245, 255, 255, 178, 0, 0, 0],
            &[0, 118, 222, 255, 255, 255, 239, 143, 10, 0, 0, 0],
            &[0, 0, 0, 13, 36, 29, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ś' / 0x15b
        'ś' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 5, 5, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 160, 255, 228, 32, 0, 0],
            &[0, 0, 0, 0, 0, 74, 255, 222, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 155, 164, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 102, 107, 83, 33, 0, 0, 0],
            &[0, 0, 12, 194, 255, 255, 255, 255, 255, 151, 0, 0],
            &[0, 0, 110, 255, 255, 182, 156, 187, 246, 83, 0, 0],
            &[0, 0, 136, 255, 254, 49, 0, 0, 9, 3, 0, 0],
            &[0, 0, 56, 247, 255, 253, 187, 85, 4, 0, 0, 0],
            &[0, 0, 0, 55, 194, 255, 255, 255, 215, 33, 0, 0],
            &[0, 0, 0, 0, 0, 49, 156, 253, 255, 171, 0, 0],
            &[0, 0, 92, 73, 9, 0, 3, 216, 255, 206, 0, 0],
            &[0, 0, 163, 255, 252, 233, 247, 255, 255, 153, 0, 0],
            &[0, 0, 116, 235, 255, 255, 255, 245, 159, 14, 0, 0],
            &[0, 0, 0, 1, 22, 38, 30, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => Some(&[
            &[0, 0, 0, 43, 222, 224, 192, 5, 0, 0, 0, 0],
            &[0, 0, 9, 209, 247, 174, 255, 142, 0, 0, 0, 0],
            &[0, 0, 124, 208, 62, 0, 123, 215, 57, 0, 0, 0],
            &[0, 0, 0, 8, 58, 72, 52, 15, 0, 0, 0, 0],
            &[0, 0, 108, 243, 255, 255, 255, 253, 181, 0, 0, 0],
            &[0, 62, 255, 255, 240, 197, 220, 255, 160, 0, 0, 0],
            &[0, 132, 255, 254, 32, 0, 0, 35, 31, 0, 0, 0],
            &[0, 129, 255, 254, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 253, 255, 219, 76, 0, 0, 0, 0, 0, 0],
            &[0, 0, 102, 249, 255, 255, 194, 52, 0, 0, 0, 0],
            &[0, 0, 0, 43, 192, 255, 255, 253, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 226, 255, 250, 21, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 255, 255, 60, 0, 0],
            &[0, 134, 77, 10, 0, 3, 150, 255, 255, 35, 0, 0],
            &[0, 207, 255, 254, 233, 245, 255, 255, 178, 0, 0, 0],
            &[0, 118, 222, 255, 255, 255, 239, 143, 10, 0, 0, 0],
            &[0, 0, 0, 13, 36, 29, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŝ' / 0x15d
        'ŝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 5, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 255, 198, 3, 0, 0, 0],
            &[0, 0, 0, 42, 244, 216, 155, 255, 132, 0, 0, 0],
            &[0, 0, 0, 140, 161, 21, 0, 114, 177, 30, 0, 0],
            &[0, 0, 0, 0, 56, 102, 107, 83, 33, 0, 0, 0],
            &[0, 0, 12, 194, 255, 255, 255, 255, 255, 151, 0, 0],
            &[0, 0, 110, 255, 255, 182, 156, 187, 246, 83, 0, 0],
            &[0, 0, 136, 255, 254, 49, 0, 0, 9, 3, 0, 0],
            &[0, 0, 56, 247, 255, 253, 187, 85, 4, 0, 0, 0],
            &[0, 0, 0, 55, 194, 255, 255, 255, 215, 33, 0, 0],
            &[0, 0, 0, 0, 0, 49, 156, 253, 255, 171, 0, 0],
            &[0, 0, 92, 73, 9, 0, 3, 216, 255, 206, 0, 0],
            &[0, 0, 163, 255, 252, 233, 247, 255, 255, 153, 0, 0],
            &[0, 0, 116, 235, 255, 255, 255, 245, 159, 14, 0, 0],
            &[0, 0, 0, 1, 22, 38, 30, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ş' / 0x15e
        'Ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 58, 72, 52, 15, 0, 0, 0, 0],
            &[0, 0, 108, 243, 255, 255, 255, 253, 181, 0, 0, 0],
            &[0, 62, 255, 255, 240, 197, 220, 255, 160, 0, 0, 0],
            &[0, 132, 255, 254, 32, 0, 0, 35, 31, 0, 0, 0],
            &[0, 129, 255, 254, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 253, 255, 219, 76, 0, 0, 0, 0, 0, 0],
            &[0, 0, 102, 249, 255, 255, 194, 52, 0, 0, 0, 0],
            &[0, 0, 0, 43, 192, 255, 255, 253, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 226, 255, 250, 21, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 255, 255, 60, 0, 0],
            &[0, 134, 77, 10, 0, 3, 150, 255, 255, 35, 0, 0],
            &[0, 207, 255, 254, 233, 245, 255, 255, 178, 0, 0, 0],
            &[0, 118, 222, 255, 255, 255, 239, 143, 10, 0, 0, 0],
            &[0, 0, 0, 13, 227, 214, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 130, 254, 179, 0, 0, 0, 0, 0],
            &[0, 0, 0, 41, 65, 244, 226, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 15, 1, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ş' / 0x15f
        'ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 102, 107, 83, 33, 0, 0, 0],
            &[0, 0, 12, 194, 255, 255, 255, 255, 255, 151, 0, 0],
            &[0, 0, 110, 255, 255, 182, 156, 187, 246, 83, 0, 0],
            &[0, 0, 136, 255, 254, 49, 0, 0, 9, 3, 0, 0],
            &[0, 0, 56, 247, 255, 253, 187, 85, 4, 0, 0, 0],
            &[0, 0, 0, 55, 194, 255, 255, 255, 215, 33, 0, 0],
            &[0, 0, 0, 0, 0, 49, 156, 253, 255, 171, 0, 0],
            &[0, 0, 92, 73, 9, 0, 3, 216, 255, 206, 0, 0],
            &[0, 0, 163, 255, 252, 233, 247, 255, 255, 153, 0, 0],
            &[0, 0, 116, 235, 255, 255, 255, 245, 159, 14, 0, 0],
            &[0, 0, 0, 1, 22, 207, 233, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 117, 248, 201, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 60, 231, 247, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 2, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Š' / 0x160
        'Š' => Some(&[
            &[0, 0, 126, 217, 69, 2, 137, 223, 56, 0, 0, 0],
            &[0, 0, 7, 204, 249, 187, 255, 132, 0, 0, 0, 0],
            &[0, 0, 0, 39, 213, 215, 181, 3, 0, 0, 0, 0],
            &[0, 0, 0, 8, 58, 72, 52, 15, 0, 0, 0, 0],
            &[0, 0, 108, 243, 255, 255, 255, 253, 181, 0, 0, 0],
            &[0, 62, 255, 255, 240, 197, 220, 255, 160, 0, 0, 0],
            &[0, 132, 255, 254, 32, 0, 0, 35, 31, 0, 0, 0],
            &[0, 129, 255, 254, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 253, 255, 219, 76, 0, 0, 0, 0, 0, 0],
            &[0, 0, 102, 249, 255, 255, 194, 52, 0, 0, 0, 0],
            &[0, 0, 0, 43, 192, 255, 255, 253, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 226, 255, 250, 21, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 255, 255, 60, 0, 0],
            &[0, 134, 77, 10, 0, 3, 150, 255, 255, 35, 0, 0],
            &[0, 207, 255, 254, 233, 245, 255, 255, 178, 0, 0, 0],
            &[0, 118, 222, 255, 255, 255, 239, 143, 10, 0, 0, 0],
            &[0, 0, 0, 13, 36, 29, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'š' / 0x161
        'š' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 3, 0, 0, 1, 5, 1, 0, 0],
            &[0, 0, 0, 166, 240, 71, 22, 199, 234, 28, 0, 0],
            &[0, 0, 0, 10, 212, 249, 222, 253, 67, 0, 0, 0],
            &[0, 0, 0, 0, 50, 178, 178, 117, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 102, 107, 83, 33, 0, 0, 0],
            &[0, 0, 12, 194, 255, 255, 255, 255, 255, 151, 0, 0],
            &[0, 0, 110, 255, 255, 182, 156, 187, 246, 83, 0, 0],
            &[0, 0, 136, 255, 254, 49, 0, 0, 9, 3, 0, 0],
            &[0, 0, 56, 247, 255, 253, 187, 85, 4, 0, 0, 0],
            &[0, 0, 0, 55, 194, 255, 255, 255, 215, 33, 0, 0],
            &[0, 0, 0, 0, 0, 49, 156, 253, 255, 171, 0, 0],
            &[0, 0, 92, 73, 9, 0, 3, 216, 255, 206, 0, 0],
            &[0, 0, 163, 255, 252, 233, 247, 255, 255, 153, 0, 0],
            &[0, 0, 116, 235, 255, 255, 255, 245, 159, 14, 0, 0],
            &[0, 0, 0, 1, 22, 38, 30, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ţ' / 0x162
        'Ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 35, 35, 35, 35, 35, 35, 35, 35, 24, 0],
            &[0, 125, 255, 255, 255, 255, 255, 255, 255, 255, 176, 0],
            &[0, 123, 252, 252, 253, 255, 255, 253, 252, 252, 174, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 190, 184, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 130, 254, 179, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 65, 244, 226, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 15, 1, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ţ' / 0x163
        'ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 152, 74, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 146, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 203, 255, 162, 72, 72, 70, 0, 0],
            &[0, 19, 231, 255, 255, 255, 255, 255, 255, 249, 0, 0],
            &[0, 19, 193, 193, 254, 255, 224, 193, 193, 189, 0, 0],
            &[0, 0, 0, 0, 249, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 249, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 249, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 248, 255, 131, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 231, 255, 199, 9, 0, 13, 2, 0],
            &[0, 0, 0, 0, 162, 255, 255, 247, 235, 254, 15, 0],
            &[0, 0, 0, 0, 20, 180, 255, 255, 255, 236, 12, 0],
            &[0, 0, 0, 0, 0, 0, 174, 248, 37, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 245, 210, 0, 0, 0],
            &[0, 0, 0, 0, 0, 35, 59, 225, 252, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 14, 2, 0, 0, 0, 0],
        ]),
        // letter: 'Ť' / 0x164
        'Ť' => Some(&[
            &[0, 0, 0, 179, 199, 34, 16, 176, 207, 18, 0, 0],
            &[0, 0, 0, 33, 239, 229, 207, 254, 73, 0, 0, 0],
            &[0, 0, 0, 0, 89, 215, 215, 133, 0, 0, 0, 0],
            &[0, 17, 35, 35, 35, 35, 35, 35, 35, 35, 24, 0],
            &[0, 125, 255, 255, 255, 255, 255, 255, 255, 255, 176, 0],
            &[0, 123, 252, 252, 253, 255, 255, 253, 252, 252, 174, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ť' / 0x165
        'ť' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 14, 27, 20, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 158, 255, 136, 0],
            &[0, 0, 0, 0, 65, 152, 74, 0, 208, 242, 21, 0],
            &[0, 0, 0, 0, 146, 255, 125, 2, 154, 88, 0, 0],
            &[0, 0, 0, 31, 203, 255, 162, 72, 72, 70, 0, 0],
            &[0, 19, 231, 255, 255, 255, 255, 255, 255, 249, 0, 0],
            &[0, 19, 193, 193, 254, 255, 224, 193, 193, 189, 0, 0],
            &[0, 0, 0, 0, 249, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 249, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 249, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 248, 255, 131, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 231, 255, 199, 9, 0, 13, 2, 0],
            &[0, 0, 0, 0, 162, 255, 255, 247, 235, 254, 15, 0],
            &[0, 0, 0, 0, 20, 180, 253, 255, 255, 236, 12, 0],
            &[0, 0, 0, 0, 0, 0, 14, 38, 27, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŧ' / 0x166
        'Ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 35, 35, 35, 35, 35, 35, 35, 35, 24, 0],
            &[0, 125, 255, 255, 255, 255, 255, 255, 255, 255, 176, 0],
            &[0, 123, 252, 252, 253, 255, 255, 253, 252, 252, 174, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 1, 5, 41, 255, 255, 88, 5, 2, 0, 0],
            &[0, 0, 59, 255, 255, 255, 255, 255, 255, 110, 0, 0],
            &[0, 0, 59, 255, 255, 255, 255, 255, 255, 110, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŧ' / 0x167
        'ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 152, 74, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 146, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 203, 255, 162, 72, 72, 70, 0, 0],
            &[0, 19, 231, 255, 255, 255, 255, 255, 255, 249, 0, 0],
            &[0, 19, 193, 193, 254, 255, 224, 193, 193, 189, 0, 0],
            &[0, 0, 0, 0, 249, 255, 125, 0, 0, 0, 0, 0],
            &[0, 20, 203, 203, 255, 255, 255, 203, 203, 203, 12, 0],
            &[0, 20, 204, 204, 255, 255, 255, 204, 204, 204, 12, 0],
            &[0, 0, 0, 0, 248, 255, 131, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 231, 255, 199, 9, 0, 13, 2, 0],
            &[0, 0, 0, 0, 162, 255, 255, 247, 235, 254, 15, 0],
            &[0, 0, 0, 0, 20, 180, 253, 255, 255, 236, 12, 0],
            &[0, 0, 0, 0, 0, 0, 14, 38, 27, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ũ' / 0x168
        'Ũ' => Some(&[
            &[0, 0, 0, 38, 157, 132, 28, 8, 177, 49, 0, 0],
            &[0, 0, 0, 206, 255, 255, 251, 233, 249, 24, 0, 0],
            &[0, 0, 13, 219, 61, 45, 153, 202, 96, 0, 0, 0],
            &[0, 8, 35, 35, 8, 0, 0, 1, 35, 35, 15, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 59, 255, 255, 64, 0, 0, 12, 255, 255, 111, 0],
            &[0, 44, 255, 255, 95, 0, 0, 41, 255, 255, 95, 0],
            &[0, 8, 243, 255, 206, 26, 15, 162, 255, 255, 43, 0],
            &[0, 0, 135, 255, 255, 255, 253, 255, 255, 178, 0, 0],
            &[0, 0, 4, 129, 237, 255, 255, 244, 151, 11, 0, 0],
            &[0, 0, 0, 0, 4, 29, 32, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ũ' / 0x169
        'ũ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 194, 169, 61, 24, 214, 56, 0, 0],
            &[0, 0, 0, 220, 239, 247, 255, 255, 242, 16, 0, 0],
            &[0, 0, 13, 182, 39, 16, 116, 165, 66, 0, 0, 0],
            &[0, 10, 72, 72, 22, 0, 0, 14, 72, 72, 19, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 61, 255, 255, 67, 0],
            &[0, 37, 255, 255, 89, 0, 0, 90, 255, 255, 67, 0],
            &[0, 12, 255, 255, 167, 1, 11, 192, 255, 255, 67, 0],
            &[0, 0, 198, 255, 255, 238, 245, 244, 254, 255, 67, 0],
            &[0, 0, 39, 208, 255, 255, 214, 57, 198, 255, 67, 0],
            &[0, 0, 0, 0, 23, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ū' / 0x16a
        'Ū' => Some(&[
            &[0, 0, 0, 13, 28, 28, 28, 28, 19, 0, 0, 0],
            &[0, 0, 0, 123, 255, 255, 255, 255, 170, 0, 0, 0],
            &[0, 0, 0, 70, 145, 145, 145, 145, 97, 0, 0, 0],
            &[0, 8, 35, 35, 8, 0, 0, 1, 35, 35, 15, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 59, 255, 255, 64, 0, 0, 12, 255, 255, 111, 0],
            &[0, 44, 255, 255, 95, 0, 0, 41, 255, 255, 95, 0],
            &[0, 8, 243, 255, 206, 26, 15, 162, 255, 255, 43, 0],
            &[0, 0, 135, 255, 255, 255, 253, 255, 255, 178, 0, 0],
            &[0, 0, 4, 129, 237, 255, 255, 244, 151, 11, 0, 0],
            &[0, 0, 0, 0, 4, 29, 32, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ū' / 0x16b
        'ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 65, 65, 65, 65, 43, 0, 0, 0],
            &[0, 0, 0, 123, 255, 255, 255, 255, 170, 0, 0, 0],
            &[0, 0, 0, 52, 109, 109, 109, 109, 72, 0, 0, 0],
            &[0, 10, 72, 72, 22, 0, 0, 14, 72, 72, 19, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 61, 255, 255, 67, 0],
            &[0, 37, 255, 255, 89, 0, 0, 90, 255, 255, 67, 0],
            &[0, 12, 255, 255, 167, 1, 11, 192, 255, 255, 67, 0],
            &[0, 0, 198, 255, 255, 238, 245, 244, 254, 255, 67, 0],
            &[0, 0, 39, 208, 255, 255, 214, 57, 198, 255, 67, 0],
            &[0, 0, 0, 0, 23, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => Some(&[
            &[0, 0, 0, 158, 196, 5, 0, 149, 205, 0, 0, 0],
            &[0, 0, 0, 88, 255, 228, 223, 255, 131, 0, 0, 0],
            &[0, 0, 0, 0, 98, 191, 196, 118, 4, 0, 0, 0],
            &[0, 8, 35, 35, 8, 0, 0, 1, 35, 35, 15, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 59, 255, 255, 64, 0, 0, 12, 255, 255, 111, 0],
            &[0, 44, 255, 255, 95, 0, 0, 41, 255, 255, 95, 0],
            &[0, 8, 243, 255, 206, 26, 15, 162, 255, 255, 43, 0],
            &[0, 0, 135, 255, 255, 255, 253, 255, 255, 178, 0, 0],
            &[0, 0, 4, 129, 237, 255, 255, 244, 151, 11, 0, 0],
            &[0, 0, 0, 0, 4, 29, 32, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŭ' / 0x16d
        'ŭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 12, 0, 0, 8, 17, 0, 0, 0],
            &[0, 0, 0, 165, 220, 18, 10, 177, 215, 0, 0, 0],
            &[0, 0, 0, 67, 254, 252, 250, 255, 108, 0, 0, 0],
            &[0, 0, 0, 0, 62, 155, 160, 81, 0, 0, 0, 0],
            &[0, 10, 72, 72, 22, 0, 0, 14, 72, 72, 19, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 61, 255, 255, 67, 0],
            &[0, 37, 255, 255, 89, 0, 0, 90, 255, 255, 67, 0],
            &[0, 12, 255, 255, 167, 1, 11, 192, 255, 255, 67, 0],
            &[0, 0, 198, 255, 255, 238, 245, 244, 254, 255, 67, 0],
            &[0, 0, 39, 208, 255, 255, 214, 57, 198, 255, 67, 0],
            &[0, 0, 0, 0, 23, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ů' / 0x16e
        'Ů' => Some(&[
            &[0, 0, 0, 0, 222, 136, 102, 254, 14, 0, 0, 0],
            &[0, 0, 0, 0, 206, 174, 150, 244, 7, 0, 0, 0],
            &[0, 0, 0, 0, 50, 187, 196, 72, 0, 0, 0, 0],
            &[0, 8, 35, 35, 8, 0, 0, 1, 35, 35, 15, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 59, 255, 255, 64, 0, 0, 12, 255, 255, 111, 0],
            &[0, 44, 255, 255, 95, 0, 0, 41, 255, 255, 95, 0],
            &[0, 8, 243, 255, 206, 26, 15, 162, 255, 255, 43, 0],
            &[0, 0, 135, 255, 255, 255, 253, 255, 255, 178, 0, 0],
            &[0, 0, 4, 129, 237, 255, 255, 244, 151, 11, 0, 0],
            &[0, 0, 0, 0, 4, 29, 32, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 243, 249, 121, 0, 0, 0, 0],
        ]),
        // letter: 'ů' / 0x16f
        'ů' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 119, 254, 254, 156, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 112, 73, 255, 17, 0, 0, 0],
            &[0, 0, 0, 0, 193, 198, 180, 233, 3, 0, 0, 0],
            &[0, 0, 0, 0, 30, 151, 159, 47, 0, 0, 0, 0],
            &[0, 10, 72, 72, 22, 0, 0, 14, 72, 72, 19, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 61, 255, 255, 67, 0],
            &[0, 37, 255, 255, 89, 0, 0, 90, 255, 255, 67, 0],
            &[0, 12, 255, 255, 167, 1, 11, 192, 255, 255, 67, 0],
            &[0, 0, 198, 255, 255, 238, 245, 244, 254, 255, 67, 0],
            &[0, 0, 39, 208, 255, 255, 214, 57, 198, 255, 67, 0],
            &[0, 0, 0, 0, 23, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ű' / 0x170
        'Ű' => Some(&[
            &[0, 0, 0, 0, 165, 224, 124, 154, 224, 133, 0, 0],
            &[0, 0, 0, 69, 255, 184, 64, 254, 192, 10, 0, 0],
            &[0, 0, 0, 173, 156, 7, 163, 166, 10, 0, 0, 0],
            &[0, 8, 35, 35, 8, 0, 0, 1, 35, 35, 15, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 59, 255, 255, 64, 0, 0, 12, 255, 255, 111, 0],
            &[0, 44, 255, 255, 95, 0, 0, 41, 255, 255, 95, 0],
            &[0, 8, 243, 255, 206, 26, 15, 162, 255, 255, 43, 0],
            &[0, 0, 135, 255, 255, 255, 253, 255, 255, 178, 0, 0],
            &[0, 0, 4, 129, 237, 255, 255, 244, 151, 11, 0, 0],
            &[0, 0, 0, 0, 4, 29, 32, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ű' / 0x171
        'ű' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 5, 4, 2, 5, 4, 0, 0],
            &[0, 0, 0, 0, 199, 255, 126, 187, 255, 137, 0, 0],
            &[0, 0, 0, 91, 255, 153, 79, 255, 163, 2, 0, 0],
            &[0, 0, 0, 150, 119, 0, 142, 129, 2, 0, 0, 0],
            &[0, 10, 72, 72, 22, 0, 0, 14, 72, 72, 19, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 61, 255, 255, 67, 0],
            &[0, 37, 255, 255, 89, 0, 0, 90, 255, 255, 67, 0],
            &[0, 12, 255, 255, 167, 1, 11, 192, 255, 255, 67, 0],
            &[0, 0, 198, 255, 255, 238, 245, 244, 254, 255, 67, 0],
            &[0, 0, 39, 208, 255, 255, 214, 57, 198, 255, 67, 0],
            &[0, 0, 0, 0, 23, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ų' / 0x172
        'Ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 35, 35, 8, 0, 0, 1, 35, 35, 15, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 60, 255, 255, 58, 0, 0, 7, 255, 255, 111, 0],
            &[0, 59, 255, 255, 64, 0, 0, 12, 255, 255, 111, 0],
            &[0, 44, 255, 255, 95, 0, 0, 41, 255, 255, 95, 0],
            &[0, 8, 243, 255, 206, 26, 15, 162, 255, 255, 43, 0],
            &[0, 0, 135, 255, 255, 255, 253, 255, 255, 178, 0, 0],
            &[0, 0, 4, 129, 237, 255, 255, 255, 174, 11, 0, 0],
            &[0, 0, 0, 0, 4, 152, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 76, 255, 202, 111, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ų' / 0x173
        'ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 72, 72, 22, 0, 0, 14, 72, 72, 19, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 51, 255, 255, 67, 0],
            &[0, 38, 255, 255, 80, 0, 0, 61, 255, 255, 67, 0],
            &[0, 37, 255, 255, 89, 0, 0, 90, 255, 255, 67, 0],
            &[0, 12, 255, 255, 167, 1, 11, 192, 255, 255, 67, 0],
            &[0, 0, 198, 255, 255, 238, 245, 244, 254, 255, 67, 0],
            &[0, 0, 39, 208, 255, 255, 214, 57, 223, 255, 67, 0],
            &[0, 0, 0, 0, 23, 25, 0, 39, 233, 129, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 178, 241, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 209, 253, 137, 46, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0],
        ]),
        // letter: 'Ŵ' / 0x174
        'Ŵ' => Some(&[
            &[0, 0, 0, 95, 224, 224, 145, 0, 0, 0, 0, 0],
            &[0, 0, 37, 242, 224, 197, 255, 82, 0, 0, 0, 0],
            &[0, 0, 175, 190, 29, 11, 163, 201, 18, 0, 0, 0],
            &[29, 35, 11, 0, 0, 0, 0, 4, 35, 35, 0, 0],
            &[200, 255, 95, 0, 0, 0, 0, 40, 255, 251, 0, 0],
            &[181, 255, 119, 0, 0, 0, 0, 58, 255, 234, 0, 0],
            &[162, 255, 143, 5, 72, 72, 22, 76, 255, 216, 0, 0],
            &[143, 255, 164, 45, 255, 255, 104, 94, 255, 198, 0, 0],
            &[124, 255, 182, 84, 255, 255, 142, 112, 255, 180, 0, 0],
            &[105, 255, 195, 124, 255, 219, 181, 129, 255, 162, 0, 0],
            &[86, 255, 203, 163, 233, 173, 220, 140, 255, 144, 0, 0],
            &[67, 255, 205, 200, 195, 138, 252, 148, 255, 126, 0, 0],
            &[48, 255, 205, 236, 156, 101, 255, 181, 255, 108, 0, 0],
            &[29, 255, 216, 255, 119, 63, 255, 213, 255, 90, 0, 0],
            &[10, 255, 255, 255, 81, 25, 255, 255, 255, 72, 0, 0],
            &[0, 247, 255, 255, 43, 0, 243, 255, 255, 55, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŵ' / 0x175
        'ŵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 5, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 113, 255, 255, 187, 1, 0, 0, 0, 0],
            &[0, 0, 51, 248, 209, 163, 255, 119, 0, 0, 0, 0],
            &[0, 0, 149, 156, 16, 1, 122, 175, 23, 0, 0, 0],
            &[67, 72, 22, 0, 0, 0, 0, 4, 72, 72, 10, 0],
            &[214, 255, 93, 0, 0, 0, 0, 29, 255, 255, 14, 0],
            &[178, 255, 115, 9, 233, 235, 61, 51, 255, 234, 0, 0],
            &[142, 255, 138, 56, 255, 255, 116, 74, 255, 197, 0, 0],
            &[106, 255, 161, 107, 255, 246, 167, 97, 255, 161, 0, 0],
            &[70, 255, 184, 158, 240, 189, 217, 119, 255, 125, 0, 0],
            &[34, 255, 208, 204, 197, 146, 254, 151, 255, 89, 0, 0],
            &[3, 250, 233, 240, 151, 101, 255, 215, 255, 53, 0, 0],
            &[0, 217, 255, 255, 104, 55, 255, 255, 255, 17, 0, 0],
            &[0, 181, 255, 255, 57, 11, 253, 255, 237, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŷ' / 0x176
        'Ŷ' => Some(&[
            &[0, 0, 0, 0, 95, 224, 224, 145, 0, 0, 0, 0],
            &[0, 0, 0, 37, 242, 224, 197, 255, 82, 0, 0, 0],
            &[0, 0, 0, 175, 190, 29, 11, 163, 201, 18, 0, 0],
            &[0, 22, 35, 34, 0, 0, 0, 0, 26, 35, 29, 0],
            &[0, 91, 255, 255, 53, 0, 0, 12, 241, 255, 142, 0],
            &[0, 3, 221, 255, 161, 0, 0, 105, 255, 249, 27, 0],
            &[0, 0, 103, 255, 248, 21, 0, 211, 255, 156, 0, 0],
            &[0, 0, 7, 230, 255, 122, 63, 255, 253, 37, 0, 0],
            &[0, 0, 0, 115, 255, 226, 174, 255, 169, 0, 0, 0],
            &[0, 0, 0, 12, 237, 255, 255, 255, 49, 0, 0, 0],
            &[0, 0, 0, 0, 128, 255, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŷ' / 0x177
        'ŷ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 5, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 122, 255, 255, 179, 0, 0, 0, 0],
            &[0, 0, 0, 57, 250, 203, 168, 255, 110, 0, 0, 0],
            &[0, 0, 0, 155, 153, 14, 2, 127, 173, 18, 0, 0],
            &[0, 43, 72, 72, 5, 0, 0, 0, 60, 72, 58, 0],
            &[0, 89, 255, 255, 73, 0, 0, 12, 249, 255, 147, 0],
            &[0, 7, 236, 255, 159, 0, 0, 81, 255, 255, 54, 0],
            &[0, 0, 142, 255, 238, 5, 0, 157, 255, 216, 0, 0],
            &[0, 0, 41, 255, 255, 74, 1, 231, 255, 124, 0, 0],
            &[0, 0, 0, 195, 255, 159, 53, 255, 254, 32, 0, 0],
            &[0, 0, 0, 93, 255, 237, 129, 255, 194, 0, 0, 0],
            &[0, 0, 0, 9, 239, 255, 225, 255, 101, 0, 0, 0],
            &[0, 0, 0, 0, 146, 255, 255, 247, 15, 0, 0, 0],
            &[0, 0, 0, 0, 45, 255, 255, 170, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 255, 255, 75, 0, 0, 0, 0],
            &[0, 0, 4, 31, 202, 255, 219, 2, 0, 0, 0, 0],
            &[0, 0, 253, 255, 255, 254, 75, 0, 0, 0, 0, 0],
            &[0, 0, 26, 34, 4, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ÿ' / 0x178
        'Ÿ' => Some(&[
            &[0, 0, 0, 125, 148, 13, 1, 130, 146, 11, 0, 0],
            &[0, 0, 19, 255, 255, 75, 28, 255, 255, 69, 0, 0],
            &[0, 0, 0, 135, 159, 14, 1, 138, 157, 12, 0, 0],
            &[0, 22, 35, 34, 0, 0, 0, 0, 26, 35, 29, 0],
            &[0, 91, 255, 255, 53, 0, 0, 12, 241, 255, 142, 0],
            &[0, 3, 221, 255, 161, 0, 0, 105, 255, 249, 27, 0],
            &[0, 0, 103, 255, 248, 21, 0, 211, 255, 156, 0, 0],
            &[0, 0, 7, 230, 255, 122, 63, 255, 253, 37, 0, 0],
            &[0, 0, 0, 115, 255, 226, 174, 255, 169, 0, 0, 0],
            &[0, 0, 0, 12, 237, 255, 255, 255, 49, 0, 0, 0],
            &[0, 0, 0, 0, 128, 255, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ź' / 0x179
        'Ź' => Some(&[
            &[0, 0, 0, 0, 0, 102, 224, 219, 52, 0, 0, 0],
            &[0, 0, 0, 0, 32, 242, 248, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 151, 209, 60, 0, 0, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 34, 0, 0, 0],
            &[0, 55, 255, 255, 255, 255, 255, 255, 249, 0, 0, 0],
            &[0, 52, 243, 243, 243, 243, 253, 255, 232, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 247, 255, 103, 0, 0, 0],
            &[0, 0, 0, 0, 0, 172, 255, 206, 2, 0, 0, 0],
            &[0, 0, 0, 0, 65, 255, 254, 59, 0, 0, 0, 0],
            &[0, 0, 0, 3, 210, 255, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 106, 255, 242, 25, 0, 0, 0, 0, 0],
            &[0, 0, 18, 236, 255, 117, 0, 0, 0, 0, 0, 0],
            &[0, 0, 147, 255, 217, 6, 0, 0, 0, 0, 0, 0],
            &[0, 43, 251, 255, 103, 31, 31, 31, 31, 0, 0, 0],
            &[0, 155, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ź' / 0x17a
        'ź' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 5, 5, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 146, 255, 234, 39, 0, 0, 0],
            &[0, 0, 0, 0, 63, 253, 228, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 146, 168, 28, 0, 0, 0, 0, 0],
            &[0, 46, 72, 72, 72, 72, 72, 72, 67, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 241, 0, 0, 0],
            &[0, 107, 167, 167, 167, 194, 255, 255, 198, 0, 0, 0],
            &[0, 0, 0, 0, 10, 207, 255, 230, 27, 0, 0, 0],
            &[0, 0, 0, 0, 168, 255, 247, 55, 0, 0, 0, 0],
            &[0, 0, 0, 119, 255, 255, 92, 0, 0, 0, 0, 0],
            &[0, 0, 73, 253, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 37, 239, 255, 180, 2, 0, 0, 0, 0, 0, 0],
            &[0, 202, 255, 255, 244, 240, 240, 240, 240, 22, 0, 0],
            &[0, 229, 255, 255, 255, 255, 255, 255, 255, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ż' / 0x17b
        'Ż' => Some(&[
            &[0, 0, 0, 10, 213, 253, 123, 0, 0, 0, 0, 0],
            &[0, 0, 0, 43, 255, 255, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 153, 195, 74, 0, 0, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 34, 0, 0, 0],
            &[0, 55, 255, 255, 255, 255, 255, 255, 249, 0, 0, 0],
            &[0, 52, 243, 243, 243, 243, 253, 255, 232, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 247, 255, 103, 0, 0, 0],
            &[0, 0, 0, 0, 0, 172, 255, 206, 2, 0, 0, 0],
            &[0, 0, 0, 0, 65, 255, 254, 59, 0, 0, 0, 0],
            &[0, 0, 0, 3, 210, 255, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 106, 255, 242, 25, 0, 0, 0, 0, 0],
            &[0, 0, 18, 236, 255, 117, 0, 0, 0, 0, 0, 0],
            &[0, 0, 147, 255, 217, 6, 0, 0, 0, 0, 0, 0],
            &[0, 43, 251, 255, 103, 31, 31, 31, 31, 0, 0, 0],
            &[0, 155, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ż' / 0x17c
        'ż' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 37, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 243, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 57, 255, 255, 172, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 126, 156, 41, 0, 0, 0, 0, 0],
            &[0, 46, 72, 72, 72, 72, 72, 72, 67, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 241, 0, 0, 0],
            &[0, 107, 167, 167, 167, 194, 255, 255, 198, 0, 0, 0],
            &[0, 0, 0, 0, 10, 207, 255, 230, 27, 0, 0, 0],
            &[0, 0, 0, 0, 168, 255, 247, 55, 0, 0, 0, 0],
            &[0, 0, 0, 119, 255, 255, 92, 0, 0, 0, 0, 0],
            &[0, 0, 73, 253, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 37, 239, 255, 180, 2, 0, 0, 0, 0, 0, 0],
            &[0, 202, 255, 255, 244, 240, 240, 240, 240, 22, 0, 0],
            &[0, 229, 255, 255, 255, 255, 255, 255, 255, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ž' / 0x17d
        'Ž' => Some(&[
            &[0, 0, 134, 215, 63, 3, 143, 222, 49, 0, 0, 0],
            &[0, 0, 9, 210, 247, 189, 255, 123, 0, 0, 0, 0],
            &[0, 0, 0, 46, 214, 215, 175, 2, 0, 0, 0, 0],
            &[0, 7, 35, 35, 35, 35, 35, 35, 34, 0, 0, 0],
            &[0, 55, 255, 255, 255, 255, 255, 255, 249, 0, 0, 0],
            &[0, 52, 243, 243, 243, 243, 253, 255, 232, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 247, 255, 103, 0, 0, 0],
            &[0, 0, 0, 0, 0, 172, 255, 206, 2, 0, 0, 0],
            &[0, 0, 0, 0, 65, 255, 254, 59, 0, 0, 0, 0],
            &[0, 0, 0, 3, 210, 255, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 106, 255, 242, 25, 0, 0, 0, 0, 0],
            &[0, 0, 18, 236, 255, 117, 0, 0, 0, 0, 0, 0],
            &[0, 0, 147, 255, 217, 6, 0, 0, 0, 0, 0, 0],
            &[0, 43, 251, 255, 103, 31, 31, 31, 31, 0, 0, 0],
            &[0, 155, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 255, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ž' / 0x17e
        'ž' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 3, 0, 0, 1, 5, 1, 0, 0, 0],
            &[0, 0, 153, 244, 80, 17, 191, 240, 36, 0, 0, 0],
            &[0, 0, 6, 203, 251, 220, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 41, 178, 178, 126, 0, 0, 0, 0, 0],
            &[0, 46, 72, 72, 72, 72, 72, 72, 67, 0, 0, 0],
            &[0, 163, 255, 255, 255, 255, 255, 255, 241, 0, 0, 0],
            &[0, 107, 167, 167, 167, 194, 255, 255, 198, 0, 0, 0],
            &[0, 0, 0, 0, 10, 207, 255, 230, 27, 0, 0, 0],
            &[0, 0, 0, 0, 168, 255, 247, 55, 0, 0, 0, 0],
            &[0, 0, 0, 119, 255, 255, 92, 0, 0, 0, 0, 0],
            &[0, 0, 73, 253, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 37, 239, 255, 180, 2, 0, 0, 0, 0, 0, 0],
            &[0, 202, 255, 255, 244, 240, 240, 240, 240, 22, 0, 0],
            &[0, 229, 255, 255, 255, 255, 255, 255, 255, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ſ' / 0x17f
        'ſ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 121, 224, 251, 237, 151, 0, 0, 0],
            &[0, 0, 0, 81, 255, 255, 255, 255, 136, 0, 0, 0],
            &[0, 0, 0, 144, 255, 249, 40, 25, 20, 0, 0, 0],
            &[0, 0, 0, 157, 255, 210, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 157, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 157, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 157, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 157, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 157, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 157, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 157, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 157, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 157, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        _ => None,
    }
}
