//! Module for letters with the font weight bold and size 18.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 18;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 12;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight bold and font size 16px
#[inline]
pub const fn get_char(c: char) -> Option<&'static [&'static [u8]]> {
    match c {
        // letter: ' ' / 0x20
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '!' / 0x21
        '!' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 58, 108, 108, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 128, 255, 255, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 255, 255, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 255, 255, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 95, 255, 252, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 255, 241, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 72, 255, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 61, 255, 218, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 53, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 153, 118, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 255, 255, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 105, 255, 233, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '"' / 0x22
        '"' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 98, 108, 31, 74, 108, 54, 0, 0, 0],
            &[0, 0, 0, 218, 255, 60, 162, 255, 115, 0, 0, 0],
            &[0, 0, 0, 199, 255, 41, 143, 255, 96, 0, 0, 0],
            &[0, 0, 0, 180, 255, 22, 125, 255, 77, 0, 0, 0],
            &[0, 0, 0, 115, 180, 4, 76, 180, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '#' / 0x23
        '#' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 107, 1, 67, 106, 0, 0, 0],
            &[0, 0, 0, 0, 187, 226, 0, 187, 221, 0, 0, 0],
            &[0, 0, 0, 0, 229, 183, 0, 229, 180, 0, 0, 0],
            &[0, 20, 69, 77, 255, 176, 76, 255, 174, 69, 25, 0],
            &[0, 75, 255, 255, 255, 255, 255, 255, 255, 255, 92, 0],
            &[0, 23, 79, 150, 255, 117, 151, 255, 113, 79, 28, 0],
            &[0, 0, 0, 138, 255, 19, 140, 255, 13, 0, 0, 0],
            &[0, 149, 196, 234, 254, 196, 235, 252, 196, 176, 0, 0],
            &[0, 155, 204, 252, 238, 204, 253, 239, 204, 183, 0, 0],
            &[0, 0, 9, 253, 146, 15, 255, 146, 0, 0, 0, 0],
            &[0, 0, 46, 255, 106, 57, 255, 104, 0, 0, 0, 0],
            &[0, 0, 86, 255, 66, 99, 255, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '$' / 0x24
        '$' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 175, 95, 0, 0, 0, 0, 0],
            &[0, 0, 0, 101, 219, 255, 254, 229, 157, 13, 0, 0],
            &[0, 0, 85, 255, 249, 240, 224, 243, 224, 1, 0, 0],
            &[0, 0, 167, 255, 107, 174, 92, 6, 41, 0, 0, 0],
            &[0, 0, 155, 255, 175, 188, 92, 0, 0, 0, 0, 0],
            &[0, 0, 43, 239, 255, 255, 175, 36, 0, 0, 0, 0],
            &[0, 0, 0, 29, 151, 248, 255, 252, 119, 0, 0, 0],
            &[0, 0, 0, 0, 0, 174, 188, 251, 255, 39, 0, 0],
            &[0, 0, 10, 0, 0, 174, 92, 191, 255, 71, 0, 0],
            &[0, 0, 176, 174, 109, 198, 179, 247, 248, 24, 0, 0],
            &[0, 0, 164, 255, 255, 255, 255, 248, 98, 0, 0, 0],
            &[0, 0, 0, 46, 88, 208, 144, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 161, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '%' / 0x25
        '%' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 95, 127, 49, 0, 0, 35, 108, 42, 0, 0],
            &[0, 142, 255, 240, 250, 41, 0, 176, 242, 19, 0, 0],
            &[0, 232, 177, 32, 255, 124, 51, 255, 131, 0, 0, 0],
            &[0, 244, 160, 14, 255, 135, 181, 240, 16, 0, 0, 0],
            &[0, 186, 237, 175, 255, 136, 255, 126, 0, 0, 0, 0],
            &[0, 32, 177, 209, 127, 188, 237, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 62, 255, 121, 28, 57, 10, 0, 0],
            &[0, 0, 0, 0, 193, 234, 98, 251, 255, 223, 21, 0],
            &[0, 0, 0, 68, 255, 116, 212, 208, 79, 255, 117, 0],
            &[0, 0, 0, 198, 232, 10, 241, 162, 4, 254, 145, 0],
            &[0, 0, 74, 255, 111, 0, 205, 219, 110, 255, 110, 0],
            &[0, 0, 204, 228, 8, 0, 66, 239, 255, 200, 13, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '&' / 0x26
        '&' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 37, 124, 124, 44, 0, 0, 0, 0, 0],
            &[0, 0, 37, 245, 255, 255, 251, 52, 0, 0, 0, 0],
            &[0, 0, 130, 255, 129, 97, 255, 143, 0, 0, 0, 0],
            &[0, 0, 146, 255, 102, 97, 255, 130, 0, 0, 0, 0],
            &[0, 0, 86, 255, 242, 236, 247, 40, 0, 0, 0, 0],
            &[0, 0, 2, 255, 255, 255, 100, 1, 8, 7, 0, 0],
            &[0, 0, 103, 255, 255, 255, 54, 65, 255, 203, 0, 0],
            &[0, 34, 250, 251, 181, 255, 224, 158, 255, 144, 0, 0],
            &[0, 113, 255, 165, 0, 177, 255, 255, 255, 48, 0, 0],
            &[0, 123, 255, 180, 0, 26, 255, 255, 255, 0, 0, 0],
            &[0, 62, 255, 255, 201, 235, 255, 255, 255, 61, 0, 0],
            &[0, 0, 129, 248, 255, 251, 171, 162, 255, 229, 21, 0],
            &[0, 0, 0, 11, 34, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ''' / 0x27
        '\'' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 108, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 64, 255, 213, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 45, 255, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 255, 176, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 180, 112, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '(' / 0x28
        '(' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 91, 107, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 75, 255, 167, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 213, 254, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 64, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 138, 255, 117, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 196, 255, 63, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 223, 255, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 240, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 230, 255, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 208, 255, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 156, 255, 107, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 89, 255, 174, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 235, 250, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 114, 255, 137, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 146, 156, 7, 0, 0, 0, 0],
        ]),
        // letter: ')' / 0x29
        ')' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 108, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 244, 221, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 141, 255, 117, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 222, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 222, 255, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 165, 255, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 136, 255, 123, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 121, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 156, 255, 102, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 207, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 252, 234, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 114, 255, 135, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 226, 239, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 163, 83, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '*' / 0x2a
        '*' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 20, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 255, 177, 0, 0, 0, 0, 0],
            &[0, 0, 6, 0, 2, 253, 156, 0, 7, 2, 0, 0],
            &[0, 0, 142, 222, 155, 247, 198, 182, 245, 45, 0, 0],
            &[0, 0, 117, 189, 233, 255, 255, 212, 180, 54, 0, 0],
            &[0, 0, 0, 9, 208, 229, 255, 117, 0, 0, 0, 0],
            &[0, 0, 0, 156, 255, 74, 182, 252, 63, 0, 0, 0],
            &[0, 0, 0, 54, 148, 0, 47, 145, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '+' / 0x2b
        '+' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 120, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 111, 126, 131, 255, 214, 126, 126, 60, 0, 0],
            &[0, 0, 225, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 45, 51, 57, 255, 190, 51, 51, 24, 0, 0],
            &[0, 0, 0, 0, 8, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 32, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ',' / 0x2c
        ',' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 45, 45, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 220, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 123, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 255, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 116, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '-' / 0x2d
        '-' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 71, 71, 71, 71, 14, 0, 0, 0],
            &[0, 0, 0, 153, 255, 255, 255, 255, 51, 0, 0, 0],
            &[0, 0, 0, 110, 184, 184, 184, 184, 36, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '.' / 0x2e
        '.' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 153, 118, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 255, 255, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 105, 255, 233, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '/' / 0x2f
        '/' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 124, 113, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 96, 255, 159, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 193, 255, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 36, 254, 220, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 132, 255, 124, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 227, 253, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 255, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 169, 255, 89, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 248, 239, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 108, 255, 150, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 205, 255, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 47, 255, 212, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 16, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '0' / 0x30
        '0' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 93, 138, 123, 45, 0, 0, 0, 0],
            &[0, 0, 15, 206, 255, 255, 255, 254, 103, 0, 0, 0],
            &[0, 0, 135, 255, 234, 109, 169, 255, 249, 26, 0, 0],
            &[0, 0, 225, 255, 104, 0, 137, 255, 255, 115, 0, 0],
            &[0, 20, 255, 255, 41, 40, 250, 255, 255, 168, 0, 0],
            &[0, 39, 255, 255, 18, 188, 251, 185, 255, 191, 0, 0],
            &[0, 46, 255, 255, 101, 255, 139, 128, 255, 201, 0, 0],
            &[0, 31, 255, 255, 255, 226, 11, 140, 255, 190, 0, 0],
            &[0, 4, 246, 255, 255, 82, 0, 172, 255, 157, 0, 0],
            &[0, 0, 185, 255, 255, 0, 21, 237, 255, 94, 0, 0],
            &[0, 0, 65, 255, 255, 207, 234, 255, 225, 9, 0, 0],
            &[0, 0, 0, 92, 232, 255, 255, 203, 42, 0, 0, 0],
            &[0, 0, 0, 0, 1, 28, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '1' / 0x31
        '1' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 94, 108, 36, 0, 0, 0, 0],
            &[0, 0, 3, 98, 226, 255, 255, 85, 0, 0, 0, 0],
            &[0, 0, 172, 255, 255, 255, 255, 85, 0, 0, 0, 0],
            &[0, 0, 72, 242, 122, 242, 255, 85, 0, 0, 0, 0],
            &[0, 0, 0, 17, 0, 248, 255, 85, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 255, 85, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 255, 85, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 255, 85, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 255, 85, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 255, 85, 0, 0, 0, 0],
            &[0, 0, 30, 176, 204, 255, 255, 255, 185, 78, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '2' / 0x32
        '2' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 20, 104, 137, 134, 74, 1, 0, 0, 0],
            &[0, 0, 108, 246, 255, 255, 255, 255, 178, 4, 0, 0],
            &[0, 3, 184, 255, 195, 127, 175, 255, 255, 91, 0, 0],
            &[0, 0, 15, 90, 0, 0, 2, 233, 255, 139, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 240, 255, 114, 0, 0],
            &[0, 0, 0, 0, 0, 0, 124, 255, 245, 27, 0, 0],
            &[0, 0, 0, 0, 0, 80, 252, 255, 103, 0, 0, 0],
            &[0, 0, 0, 0, 75, 249, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 75, 249, 253, 108, 0, 0, 0, 0, 0],
            &[0, 0, 75, 249, 250, 83, 0, 0, 0, 0, 0, 0],
            &[0, 16, 249, 255, 255, 251, 251, 251, 251, 221, 0, 0],
            &[0, 22, 255, 255, 255, 255, 255, 255, 255, 225, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '3' / 0x33
        '3' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 41, 112, 143, 131, 65, 0, 0, 0, 0],
            &[0, 0, 148, 255, 255, 255, 255, 255, 154, 0, 0, 0],
            &[0, 0, 75, 246, 161, 113, 174, 255, 255, 31, 0, 0],
            &[0, 0, 0, 14, 0, 0, 19, 255, 255, 37, 0, 0],
            &[0, 0, 0, 0, 0, 0, 95, 255, 214, 1, 0, 0],
            &[0, 0, 0, 40, 206, 232, 255, 200, 39, 0, 0, 0],
            &[0, 0, 0, 51, 255, 255, 255, 218, 109, 0, 0, 0],
            &[0, 0, 0, 2, 15, 35, 109, 255, 255, 78, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 222, 255, 135, 0, 0],
            &[0, 0, 80, 17, 0, 0, 44, 248, 255, 113, 0, 0],
            &[0, 0, 221, 252, 225, 213, 251, 255, 240, 25, 0, 0],
            &[0, 0, 146, 235, 255, 255, 253, 193, 50, 0, 0, 0],
            &[0, 0, 0, 0, 19, 35, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '4' / 0x34
        '4' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 87, 108, 52, 0, 0, 0],
            &[0, 0, 0, 0, 0, 67, 255, 255, 124, 0, 0, 0],
            &[0, 0, 0, 0, 9, 219, 255, 255, 124, 0, 0, 0],
            &[0, 0, 0, 0, 135, 255, 223, 255, 124, 0, 0, 0],
            &[0, 0, 0, 47, 251, 182, 172, 255, 124, 0, 0, 0],
            &[0, 0, 2, 202, 248, 37, 181, 255, 124, 0, 0, 0],
            &[0, 0, 111, 255, 133, 0, 184, 255, 124, 0, 0, 0],
            &[0, 30, 243, 244, 71, 61, 201, 255, 156, 58, 0, 0],
            &[0, 92, 255, 255, 255, 255, 255, 255, 255, 245, 0, 0],
            &[0, 58, 161, 161, 161, 161, 229, 255, 207, 155, 0, 0],
            &[0, 0, 0, 0, 0, 0, 184, 255, 124, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 184, 255, 124, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '5' / 0x35
        '5' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 108, 108, 108, 108, 108, 84, 0, 0, 0],
            &[0, 0, 55, 255, 255, 255, 255, 255, 198, 0, 0, 0],
            &[0, 0, 74, 255, 246, 143, 143, 143, 111, 0, 0, 0],
            &[0, 0, 94, 255, 221, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 114, 255, 211, 59, 52, 5, 0, 0, 0, 0],
            &[0, 0, 133, 255, 255, 255, 255, 231, 66, 0, 0, 0],
            &[0, 0, 81, 203, 167, 176, 249, 255, 239, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 84, 255, 255, 79, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 255, 255, 93, 0, 0],
            &[0, 0, 81, 25, 0, 0, 104, 255, 255, 52, 0, 0],
            &[0, 0, 200, 255, 236, 233, 255, 255, 184, 0, 0, 0],
            &[0, 0, 134, 241, 255, 255, 240, 146, 12, 0, 0, 0],
            &[0, 0, 0, 2, 25, 30, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '6' / 0x36
        '6' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 66, 119, 137, 117, 0, 0, 0],
            &[0, 0, 0, 46, 215, 255, 255, 255, 235, 0, 0, 0],
            &[0, 0, 24, 235, 255, 225, 125, 92, 99, 0, 0, 0],
            &[0, 0, 141, 255, 221, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 221, 255, 103, 48, 97, 67, 4, 0, 0, 0],
            &[0, 9, 255, 255, 166, 255, 255, 255, 194, 6, 0, 0],
            &[0, 25, 255, 255, 250, 143, 150, 254, 255, 103, 0, 0],
            &[0, 20, 255, 255, 127, 0, 0, 189, 255, 160, 0, 0],
            &[0, 1, 242, 255, 81, 0, 0, 167, 255, 175, 0, 0],
            &[0, 0, 175, 255, 170, 3, 18, 229, 255, 137, 0, 0],
            &[0, 0, 53, 251, 255, 237, 245, 255, 247, 38, 0, 0],
            &[0, 0, 0, 77, 227, 255, 255, 222, 72, 0, 0, 0],
            &[0, 0, 0, 0, 1, 33, 29, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '7' / 0x37
        '7' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 104, 104, 104, 104, 104, 104, 104, 83, 0, 0],
            &[0, 51, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 28, 143, 143, 143, 143, 143, 240, 255, 160, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 255, 255, 49, 0, 0],
            &[0, 0, 0, 0, 0, 0, 164, 255, 193, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 251, 255, 81, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 255, 222, 3, 0, 0, 0],
            &[0, 0, 0, 0, 15, 241, 255, 114, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 255, 243, 15, 0, 0, 0, 0],
            &[0, 0, 0, 5, 228, 255, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 93, 255, 253, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 209, 255, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '8' / 0x38
        '8' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 96, 137, 128, 65, 0, 0, 0, 0],
            &[0, 0, 33, 226, 255, 255, 255, 255, 163, 0, 0, 0],
            &[0, 0, 149, 255, 237, 109, 157, 255, 255, 48, 0, 0],
            &[0, 0, 165, 255, 164, 0, 10, 254, 255, 65, 0, 0],
            &[0, 0, 93, 255, 242, 69, 145, 255, 231, 10, 0, 0],
            &[0, 0, 1, 154, 255, 255, 255, 220, 49, 0, 0, 0],
            &[0, 0, 10, 158, 255, 255, 255, 227, 62, 0, 0, 0],
            &[0, 0, 150, 255, 220, 54, 150, 255, 246, 40, 0, 0],
            &[0, 1, 242, 255, 77, 0, 0, 190, 255, 139, 0, 0],
            &[0, 4, 252, 255, 92, 0, 1, 192, 255, 151, 0, 0],
            &[0, 0, 180, 255, 251, 194, 218, 255, 254, 65, 0, 0],
            &[0, 0, 18, 167, 248, 255, 255, 219, 84, 0, 0, 0],
            &[0, 0, 0, 0, 8, 34, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '9' / 0x39
        '9' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 109, 145, 126, 39, 0, 0, 0, 0],
            &[0, 0, 41, 232, 255, 255, 255, 250, 89, 0, 0, 0],
            &[0, 0, 187, 255, 231, 125, 179, 255, 244, 23, 0, 0],
            &[0, 8, 253, 255, 93, 0, 1, 218, 255, 110, 0, 0],
            &[0, 22, 255, 255, 63, 0, 0, 189, 255, 161, 0, 0],
            &[0, 1, 247, 255, 134, 0, 47, 245, 255, 180, 0, 0],
            &[0, 0, 154, 255, 255, 233, 254, 241, 255, 174, 0, 0],
            &[0, 0, 13, 157, 234, 236, 145, 173, 255, 146, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 234, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 13, 167, 255, 235, 11, 0, 0],
            &[0, 0, 77, 196, 194, 240, 255, 251, 79, 0, 0, 0],
            &[0, 0, 90, 255, 255, 251, 188, 58, 0, 0, 0, 0],
            &[0, 0, 6, 35, 33, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ':' / 0x3a
        ':' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 215, 179, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 157, 255, 255, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 220, 182, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 153, 118, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 255, 255, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 105, 255, 233, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ';' / 0x3b
        ';' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 189, 211, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 67, 255, 255, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 192, 216, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 45, 45, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 220, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 123, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 255, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 116, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '<' / 0x3c
        '<' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 100, 43, 0, 0],
            &[0, 0, 0, 0, 0, 2, 92, 220, 255, 61, 0, 0],
            &[0, 0, 0, 1, 84, 213, 255, 235, 121, 10, 0, 0],
            &[0, 0, 63, 206, 255, 233, 118, 12, 0, 0, 0, 0],
            &[0, 0, 160, 255, 250, 104, 5, 0, 0, 0, 0, 0],
            &[0, 0, 11, 115, 231, 255, 226, 112, 11, 0, 0, 0],
            &[0, 0, 0, 0, 8, 108, 226, 255, 237, 42, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 100, 221, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 11, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '=' / 0x3d
        '=' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 32, 32, 32, 32, 32, 32, 15, 0, 0],
            &[0, 0, 225, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 178, 202, 202, 202, 202, 202, 202, 97, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 172, 196, 196, 196, 196, 196, 196, 94, 0, 0],
            &[0, 0, 225, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 34, 38, 38, 38, 38, 38, 38, 18, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '>' / 0x3e
        '>' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 100, 49, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 163, 254, 171, 42, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 172, 254, 253, 163, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 168, 253, 251, 155, 17, 0, 0],
            &[0, 0, 0, 0, 0, 28, 178, 255, 255, 58, 0, 0],
            &[0, 0, 0, 42, 159, 250, 255, 189, 65, 0, 0, 0],
            &[0, 0, 127, 254, 255, 182, 58, 0, 0, 0, 0, 0],
            &[0, 0, 162, 175, 51, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '?' / 0x3f
        '?' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 38, 110, 142, 135, 73, 0, 0, 0, 0],
            &[0, 0, 120, 255, 255, 255, 255, 255, 166, 0, 0, 0],
            &[0, 0, 45, 234, 146, 99, 157, 255, 255, 49, 0, 0],
            &[0, 0, 0, 4, 0, 0, 6, 252, 255, 69, 0, 0],
            &[0, 0, 0, 0, 0, 0, 122, 255, 243, 21, 0, 0],
            &[0, 0, 0, 0, 2, 150, 255, 246, 80, 0, 0, 0],
            &[0, 0, 0, 0, 103, 255, 231, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 167, 255, 111, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 53, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 157, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 254, 255, 192, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 213, 255, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '@' / 0x40
        '@' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 87, 96, 48, 0, 0, 0, 0],
            &[0, 0, 2, 138, 252, 251, 243, 255, 173, 9, 0, 0],
            &[0, 0, 138, 254, 119, 7, 3, 85, 246, 158, 0, 0],
            &[0, 41, 252, 137, 0, 55, 67, 21, 118, 254, 34, 0],
            &[0, 140, 249, 18, 166, 255, 255, 253, 58, 255, 107, 0],
            &[0, 203, 196, 60, 255, 134, 141, 255, 17, 252, 138, 0],
            &[0, 240, 161, 121, 255, 17, 136, 255, 4, 249, 144, 0],
            &[0, 251, 150, 128, 255, 6, 168, 248, 14, 255, 117, 0],
            &[0, 232, 167, 91, 255, 97, 228, 255, 123, 255, 47, 0],
            &[0, 191, 219, 11, 218, 255, 194, 223, 255, 160, 0, 0],
            &[0, 107, 255, 70, 12, 50, 3, 14, 44, 0, 0, 0],
            &[0, 6, 211, 235, 74, 1, 0, 26, 95, 0, 0, 0],
            &[0, 0, 22, 187, 255, 247, 245, 255, 223, 0, 0, 0],
            &[0, 0, 0, 0, 42, 89, 97, 65, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'A' / 0x41
        'A' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 120, 120, 40, 0, 0, 0, 0],
            &[0, 0, 0, 3, 240, 255, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 58, 255, 226, 255, 208, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 135, 235, 254, 24, 0, 0, 0],
            &[0, 0, 0, 199, 255, 73, 175, 255, 94, 0, 0, 0],
            &[0, 0, 17, 252, 252, 13, 114, 255, 165, 0, 0, 0],
            &[0, 0, 84, 255, 201, 0, 52, 255, 234, 1, 0, 0],
            &[0, 0, 154, 255, 255, 192, 196, 255, 255, 50, 0, 0],
            &[0, 0, 224, 255, 255, 255, 255, 255, 255, 121, 0, 0],
            &[0, 39, 255, 255, 56, 42, 42, 152, 255, 191, 0, 0],
            &[0, 109, 255, 202, 0, 0, 0, 61, 255, 250, 12, 0],
            &[0, 179, 255, 138, 0, 0, 0, 7, 248, 255, 77, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'B' / 0x42
        'B' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 78, 108, 108, 108, 91, 56, 1, 0, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 210, 21, 0, 0],
            &[0, 0, 184, 255, 181, 123, 153, 251, 255, 139, 0, 0],
            &[0, 0, 184, 255, 112, 0, 0, 176, 255, 174, 0, 0],
            &[0, 0, 184, 255, 112, 0, 13, 211, 255, 129, 0, 0],
            &[0, 0, 184, 255, 245, 237, 252, 255, 179, 14, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 194, 32, 0, 0],
            &[0, 0, 184, 255, 118, 10, 36, 203, 255, 189, 0, 0],
            &[0, 0, 184, 255, 112, 0, 0, 126, 255, 239, 0, 0],
            &[0, 0, 184, 255, 112, 0, 6, 187, 255, 215, 0, 0],
            &[0, 0, 184, 255, 242, 231, 247, 255, 255, 111, 0, 0],
            &[0, 0, 184, 255, 255, 255, 249, 212, 107, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'C' / 0x43
        'C' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 88, 134, 139, 107, 28, 0, 0],
            &[0, 0, 0, 68, 232, 255, 255, 255, 255, 199, 0, 0],
            &[0, 0, 40, 246, 255, 218, 133, 126, 182, 95, 0, 0],
            &[0, 0, 166, 255, 215, 10, 0, 0, 0, 0, 0, 0],
            &[0, 3, 243, 255, 109, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 255, 255, 61, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 255, 255, 43, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 60, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 241, 255, 123, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 239, 64, 0, 0, 37, 54, 0, 0],
            &[0, 0, 32, 238, 255, 255, 237, 239, 255, 131, 0, 0],
            &[0, 0, 0, 42, 184, 254, 255, 255, 225, 84, 0, 0],
            &[0, 0, 0, 0, 0, 8, 33, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'D' / 0x44
        'D' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 108, 108, 108, 100, 67, 4, 0, 0, 0, 0],
            &[0, 71, 255, 255, 255, 255, 255, 230, 67, 0, 0, 0],
            &[0, 71, 255, 255, 140, 150, 223, 255, 248, 41, 0, 0],
            &[0, 71, 255, 255, 2, 0, 13, 216, 255, 166, 0, 0],
            &[0, 71, 255, 255, 2, 0, 0, 112, 255, 241, 0, 0],
            &[0, 71, 255, 255, 2, 0, 0, 70, 255, 255, 15, 0],
            &[0, 71, 255, 255, 2, 0, 0, 67, 255, 255, 22, 0],
            &[0, 71, 255, 255, 2, 0, 0, 93, 255, 253, 3, 0],
            &[0, 71, 255, 255, 2, 0, 0, 169, 255, 199, 0, 0],
            &[0, 71, 255, 255, 2, 18, 115, 252, 255, 100, 0, 0],
            &[0, 71, 255, 255, 247, 255, 255, 255, 175, 1, 0, 0],
            &[0, 71, 255, 255, 255, 243, 205, 108, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'E' / 0x45
        'E' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 78, 108, 108, 108, 108, 108, 108, 52, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 203, 147, 147, 147, 147, 70, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 247, 237, 237, 237, 237, 56, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 184, 255, 141, 18, 18, 18, 18, 4, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'F' / 0x46
        'F' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 78, 108, 108, 108, 108, 108, 108, 52, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 203, 147, 147, 147, 147, 70, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 188, 114, 114, 114, 114, 27, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 184, 255, 200, 141, 141, 141, 141, 33, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'G' / 0x47
        'G' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 114, 141, 124, 82, 9, 0, 0],
            &[0, 0, 3, 146, 254, 255, 255, 255, 255, 157, 0, 0],
            &[0, 0, 124, 255, 255, 187, 118, 125, 191, 63, 0, 0],
            &[0, 11, 246, 255, 149, 0, 0, 0, 0, 0, 0, 0],
            &[0, 78, 255, 253, 20, 0, 0, 0, 0, 0, 0, 0],
            &[0, 112, 255, 221, 0, 0, 88, 110, 110, 97, 0, 0],
            &[0, 127, 255, 209, 0, 0, 204, 255, 255, 225, 0, 0],
            &[0, 109, 255, 231, 0, 0, 99, 153, 255, 225, 0, 0],
            &[0, 65, 255, 255, 43, 0, 0, 55, 255, 225, 0, 0],
            &[0, 5, 230, 255, 195, 19, 0, 55, 255, 225, 0, 0],
            &[0, 0, 82, 252, 255, 246, 213, 235, 255, 225, 0, 0],
            &[0, 0, 0, 72, 204, 255, 255, 255, 236, 144, 0, 0],
            &[0, 0, 0, 0, 0, 12, 34, 19, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'H' / 0x48
        'H' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 108, 108, 4, 0, 0, 47, 108, 95, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 246, 245, 245, 250, 255, 225, 0, 0],
            &[0, 71, 255, 255, 255, 255, 255, 255, 255, 225, 0, 0],
            &[0, 71, 255, 255, 20, 10, 10, 118, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'I' / 0x49
        'I' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 108, 108, 108, 108, 108, 108, 17, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 29, 76, 152, 255, 239, 92, 64, 7, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 90, 184, 224, 255, 251, 201, 172, 24, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'J' / 0x4a
        'J' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 46, 108, 91, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 255, 214, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 255, 214, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 255, 214, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 255, 214, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 255, 214, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 255, 214, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 255, 214, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 128, 255, 205, 0, 0, 0],
            &[0, 0, 17, 1, 0, 16, 211, 255, 169, 0, 0, 0],
            &[0, 0, 163, 246, 226, 251, 255, 255, 70, 0, 0, 0],
            &[0, 0, 145, 254, 255, 255, 227, 99, 0, 0, 0, 0],
            &[0, 0, 0, 14, 35, 24, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'K' / 0x4b
        'K' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 108, 107, 0, 0, 0, 68, 108, 97, 0, 0],
            &[0, 71, 255, 253, 0, 0, 36, 245, 255, 103, 0, 0],
            &[0, 71, 255, 253, 0, 2, 195, 255, 178, 0, 0, 0],
            &[0, 71, 255, 253, 0, 111, 255, 232, 21, 0, 0, 0],
            &[0, 71, 255, 253, 35, 245, 255, 74, 0, 0, 0, 0],
            &[0, 71, 255, 253, 180, 255, 165, 0, 0, 0, 0, 0],
            &[0, 71, 255, 255, 255, 255, 224, 8, 0, 0, 0, 0],
            &[0, 71, 255, 255, 189, 234, 255, 119, 0, 0, 0, 0],
            &[0, 71, 255, 254, 7, 111, 255, 240, 20, 0, 0, 0],
            &[0, 71, 255, 253, 0, 7, 224, 255, 146, 0, 0, 0],
            &[0, 71, 255, 253, 0, 0, 96, 255, 250, 37, 0, 0],
            &[0, 71, 255, 253, 0, 0, 3, 214, 255, 174, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'L' / 0x4c
        'L' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 108, 82, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 197, 12, 12, 12, 12, 5, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'M' / 0x4d
        'M' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 39, 108, 108, 87, 0, 7, 108, 108, 102, 0, 0],
            &[0, 92, 255, 255, 234, 0, 50, 255, 255, 241, 0, 0],
            &[0, 92, 255, 246, 255, 16, 95, 255, 246, 241, 0, 0],
            &[0, 92, 255, 207, 255, 55, 141, 244, 218, 241, 0, 0],
            &[0, 92, 255, 178, 255, 93, 186, 200, 234, 241, 0, 0],
            &[0, 92, 255, 150, 254, 132, 232, 155, 250, 241, 0, 0],
            &[0, 92, 255, 154, 218, 192, 255, 119, 255, 241, 0, 0],
            &[0, 92, 255, 168, 170, 254, 255, 86, 255, 241, 0, 0],
            &[0, 92, 255, 178, 121, 255, 255, 50, 255, 241, 0, 0],
            &[0, 92, 255, 183, 27, 79, 76, 34, 255, 241, 0, 0],
            &[0, 92, 255, 184, 0, 0, 0, 34, 255, 241, 0, 0],
            &[0, 92, 255, 184, 0, 0, 0, 34, 255, 241, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'N' / 0x4e
        'N' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 108, 108, 102, 0, 0, 19, 108, 95, 0, 0],
            &[0, 71, 255, 255, 255, 44, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 248, 255, 128, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 190, 251, 211, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 185, 188, 255, 39, 47, 255, 225, 0, 0],
            &[0, 71, 255, 196, 105, 255, 123, 47, 255, 225, 0, 0],
            &[0, 71, 255, 200, 24, 253, 206, 46, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 195, 255, 76, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 112, 255, 152, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 30, 255, 225, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 0, 202, 255, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 0, 119, 255, 255, 225, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'O' / 0x4f
        'O' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 105, 146, 139, 72, 0, 0, 0, 0],
            &[0, 0, 37, 229, 255, 255, 255, 255, 166, 2, 0, 0],
            &[0, 0, 193, 255, 229, 115, 158, 255, 255, 96, 0, 0],
            &[0, 38, 255, 255, 69, 0, 0, 177, 255, 195, 0, 0],
            &[0, 95, 255, 247, 2, 0, 0, 99, 255, 247, 2, 0],
            &[0, 119, 255, 221, 0, 0, 0, 72, 255, 255, 21, 0],
            &[0, 129, 255, 216, 0, 0, 0, 63, 255, 255, 26, 0],
            &[0, 115, 255, 230, 0, 0, 0, 77, 255, 255, 13, 0],
            &[0, 76, 255, 255, 20, 0, 0, 124, 255, 230, 0, 0],
            &[0, 13, 246, 255, 141, 0, 21, 225, 255, 157, 0, 0],
            &[0, 0, 129, 255, 255, 231, 245, 255, 244, 38, 0, 0],
            &[0, 0, 2, 131, 241, 255, 255, 211, 60, 0, 0, 0],
            &[0, 0, 0, 0, 5, 32, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'P' / 0x50
        'P' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 78, 108, 108, 108, 101, 64, 3, 0, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 212, 26, 0, 0],
            &[0, 0, 184, 255, 199, 139, 161, 248, 255, 165, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 126, 255, 231, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 106, 255, 236, 0, 0],
            &[0, 0, 184, 255, 136, 10, 49, 212, 255, 192, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 254, 73, 0, 0],
            &[0, 0, 184, 255, 248, 239, 226, 182, 68, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Q' / 0x51
        'Q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 105, 146, 139, 72, 0, 0, 0, 0],
            &[0, 0, 37, 229, 255, 255, 255, 255, 166, 2, 0, 0],
            &[0, 0, 193, 255, 229, 115, 158, 255, 255, 96, 0, 0],
            &[0, 38, 255, 255, 69, 0, 0, 177, 255, 195, 0, 0],
            &[0, 95, 255, 247, 2, 0, 0, 99, 255, 247, 2, 0],
            &[0, 119, 255, 221, 0, 0, 0, 72, 255, 255, 21, 0],
            &[0, 129, 255, 216, 0, 0, 0, 63, 255, 255, 26, 0],
            &[0, 115, 255, 230, 0, 0, 0, 77, 255, 255, 13, 0],
            &[0, 76, 255, 255, 20, 0, 0, 124, 255, 230, 0, 0],
            &[0, 13, 246, 255, 141, 0, 21, 225, 255, 157, 0, 0],
            &[0, 0, 129, 255, 255, 231, 255, 255, 244, 38, 0, 0],
            &[0, 0, 2, 131, 241, 255, 255, 255, 76, 0, 0, 0],
            &[0, 0, 0, 0, 5, 34, 236, 255, 134, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 87, 255, 246, 27, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 193, 225, 130, 0, 0],
        ]),
        // letter: 'R' / 0x52
        'R' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 78, 108, 108, 103, 79, 16, 0, 0, 0, 0, 0],
            &[0, 184, 255, 255, 255, 255, 245, 100, 0, 0, 0, 0],
            &[0, 184, 255, 209, 146, 209, 255, 253, 41, 0, 0, 0],
            &[0, 184, 255, 153, 0, 8, 239, 255, 111, 0, 0, 0],
            &[0, 184, 255, 153, 0, 1, 239, 255, 105, 0, 0, 0],
            &[0, 184, 255, 181, 81, 166, 255, 252, 42, 0, 0, 0],
            &[0, 184, 255, 255, 255, 255, 252, 112, 0, 0, 0, 0],
            &[0, 184, 255, 224, 209, 255, 199, 0, 0, 0, 0, 0],
            &[0, 184, 255, 153, 23, 243, 255, 84, 0, 0, 0, 0],
            &[0, 184, 255, 153, 0, 125, 255, 224, 8, 0, 0, 0],
            &[0, 184, 255, 153, 0, 10, 228, 255, 125, 0, 0, 0],
            &[0, 184, 255, 153, 0, 0, 97, 255, 245, 28, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'S' / 0x53
        'S' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 89, 137, 139, 113, 57, 0, 0, 0],
            &[0, 0, 19, 213, 255, 255, 255, 255, 255, 86, 0, 0],
            &[0, 0, 125, 255, 250, 128, 96, 138, 203, 8, 0, 0],
            &[0, 0, 160, 255, 199, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 255, 246, 67, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 216, 255, 255, 179, 42, 0, 0, 0, 0],
            &[0, 0, 0, 17, 159, 255, 255, 252, 137, 1, 0, 0],
            &[0, 0, 0, 0, 0, 52, 192, 255, 255, 104, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 207, 255, 174, 0, 0],
            &[0, 0, 106, 30, 0, 0, 9, 213, 255, 161, 0, 0],
            &[0, 0, 225, 254, 227, 202, 235, 255, 253, 64, 0, 0],
            &[0, 0, 138, 233, 255, 255, 255, 212, 78, 0, 0, 0],
            &[0, 0, 0, 0, 18, 36, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'T' / 0x54
        'T' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 56, 108, 108, 108, 108, 108, 108, 108, 108, 13, 0],
            &[0, 133, 255, 255, 255, 255, 255, 255, 255, 255, 30, 0],
            &[0, 76, 147, 147, 187, 255, 250, 147, 147, 147, 17, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'U' / 0x55
        'U' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 108, 108, 4, 0, 0, 47, 108, 95, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 11, 0, 0, 114, 255, 225, 0, 0],
            &[0, 61, 255, 255, 37, 0, 0, 138, 255, 214, 0, 0],
            &[0, 24, 253, 255, 131, 0, 14, 217, 255, 172, 0, 0],
            &[0, 0, 170, 255, 255, 235, 249, 255, 254, 63, 0, 0],
            &[0, 0, 14, 158, 247, 255, 255, 220, 84, 0, 0, 0],
            &[0, 0, 0, 0, 8, 32, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'V' / 0x56
        'V' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 108, 59, 0, 0, 0, 0, 105, 108, 41, 0],
            &[0, 145, 255, 187, 0, 0, 0, 40, 255, 255, 43, 0],
            &[0, 70, 255, 246, 7, 0, 0, 103, 255, 224, 0, 0],
            &[0, 7, 244, 255, 63, 0, 0, 166, 255, 149, 0, 0],
            &[0, 0, 176, 255, 129, 0, 0, 229, 255, 74, 0, 0],
            &[0, 0, 102, 255, 194, 0, 37, 255, 246, 9, 0, 0],
            &[0, 0, 27, 255, 249, 10, 101, 255, 180, 0, 0, 0],
            &[0, 0, 0, 208, 255, 70, 162, 255, 105, 0, 0, 0],
            &[0, 0, 0, 133, 255, 129, 215, 255, 31, 0, 0, 0],
            &[0, 0, 0, 58, 255, 188, 252, 211, 0, 0, 0, 0],
            &[0, 0, 0, 2, 236, 255, 255, 136, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 255, 255, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'W' / 0x57
        'W' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 89, 108, 21, 0, 0, 0, 0, 64, 108, 46, 0],
            &[0, 197, 255, 67, 0, 0, 0, 0, 164, 255, 96, 0],
            &[0, 178, 255, 91, 0, 0, 0, 0, 182, 255, 78, 0],
            &[0, 159, 255, 114, 64, 188, 182, 0, 200, 255, 60, 0],
            &[0, 140, 255, 133, 122, 255, 255, 27, 218, 255, 42, 0],
            &[0, 121, 255, 149, 162, 243, 255, 65, 236, 255, 24, 0],
            &[0, 102, 255, 159, 201, 179, 255, 104, 249, 255, 6, 0],
            &[0, 83, 255, 163, 239, 134, 230, 144, 253, 244, 0, 0],
            &[0, 64, 255, 183, 255, 95, 193, 186, 248, 226, 0, 0],
            &[0, 45, 255, 213, 255, 57, 155, 227, 239, 208, 0, 0],
            &[0, 26, 255, 254, 255, 19, 118, 255, 254, 190, 0, 0],
            &[0, 7, 255, 255, 237, 0, 80, 255, 255, 172, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'X' / 0x58
        'X' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 36, 108, 106, 4, 0, 0, 49, 108, 100, 0, 0],
            &[0, 9, 226, 255, 101, 0, 1, 208, 255, 133, 0, 0],
            &[0, 0, 87, 255, 227, 8, 85, 255, 228, 11, 0, 0],
            &[0, 0, 0, 195, 255, 116, 214, 255, 92, 0, 0, 0],
            &[0, 0, 0, 50, 253, 251, 255, 198, 1, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 254, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 255, 248, 31, 0, 0, 0, 0],
            &[0, 0, 0, 27, 244, 255, 255, 171, 0, 0, 0, 0],
            &[0, 0, 0, 164, 255, 150, 235, 255, 64, 0, 0, 0],
            &[0, 0, 58, 254, 240, 18, 117, 255, 209, 3, 0, 0],
            &[0, 2, 204, 255, 122, 0, 9, 231, 255, 104, 0, 0],
            &[0, 99, 255, 232, 11, 0, 0, 109, 255, 236, 17, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Y' / 0x59
        'Y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 62, 108, 93, 0, 0, 0, 26, 108, 108, 19, 0],
            &[0, 61, 255, 255, 41, 0, 0, 138, 255, 213, 1, 0],
            &[0, 0, 195, 255, 148, 0, 8, 236, 255, 94, 0, 0],
            &[0, 0, 73, 255, 242, 13, 96, 255, 224, 4, 0, 0],
            &[0, 0, 0, 207, 255, 108, 203, 255, 107, 0, 0, 0],
            &[0, 0, 0, 86, 255, 236, 255, 233, 9, 0, 0, 0],
            &[0, 0, 0, 2, 217, 255, 255, 121, 0, 0, 0, 0],
            &[0, 0, 0, 0, 109, 255, 252, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Z' / 0x5a
        'Z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 108, 108, 108, 108, 108, 108, 48, 0, 0],
            &[0, 0, 81, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 44, 139, 139, 139, 171, 255, 255, 60, 0, 0],
            &[0, 0, 0, 0, 0, 0, 178, 255, 166, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 244, 27, 0, 0, 0],
            &[0, 0, 0, 0, 4, 214, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 219, 6, 0, 0, 0, 0],
            &[0, 0, 0, 21, 239, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 153, 255, 180, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 253, 249, 37, 0, 0, 0, 0, 0, 0],
            &[0, 0, 170, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '[' / 0x5b
        '[' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 43, 137, 137, 137, 71, 0, 0, 0, 0],
            &[0, 0, 0, 81, 255, 255, 255, 133, 0, 0, 0, 0],
            &[0, 0, 0, 81, 255, 178, 24, 12, 0, 0, 0, 0],
            &[0, 0, 0, 81, 255, 169, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 255, 169, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 255, 169, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 255, 169, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 255, 169, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 255, 169, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 255, 169, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 255, 169, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 255, 169, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 255, 174, 14, 7, 0, 0, 0, 0],
            &[0, 0, 0, 81, 255, 255, 255, 133, 0, 0, 0, 0],
            &[0, 0, 0, 48, 151, 151, 151, 78, 0, 0, 0, 0],
        ]),
        // letter: '\' / 0x5c
        '\\' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 38, 124, 86, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 246, 240, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 165, 255, 91, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 255, 188, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 224, 254, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 129, 255, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 254, 223, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 255, 66, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 94, 255, 164, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 242, 246, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 155, 255, 103, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 58, 255, 200, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 16, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ']' / 0x5d
        ']' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 152, 165, 165, 152, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 255, 235, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 255, 235, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 255, 235, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 255, 235, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 255, 235, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 255, 235, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 255, 235, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 255, 235, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 255, 235, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 255, 235, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 255, 235, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 52, 255, 235, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 255, 255, 235, 0, 0, 0, 0],
            &[0, 0, 0, 0, 113, 122, 122, 113, 0, 0, 0, 0],
        ]),
        // letter: '^' / 0x5e
        '^' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 112, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 208, 255, 255, 62, 0, 0, 0, 0],
            &[0, 0, 0, 77, 255, 167, 250, 195, 0, 0, 0, 0],
            &[0, 0, 0, 201, 247, 23, 152, 255, 71, 0, 0, 0],
            &[0, 0, 69, 255, 150, 0, 28, 248, 203, 0, 0, 0],
            &[0, 0, 193, 251, 33, 0, 0, 145, 255, 80, 0, 0],
            &[0, 34, 190, 133, 0, 0, 0, 23, 189, 146, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '_' / 0x5f
        '_' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 237, 237, 237, 237, 237, 237, 237, 237, 237, 142, 0],
            &[0, 253, 253, 253, 253, 253, 253, 253, 253, 253, 152, 0],
        ]),
        // letter: '`' / 0x60
        '`' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 52, 65, 51, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 246, 251, 48, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 62, 239, 210, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 77, 21, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'a' / 0x61
        'a' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 41, 141, 196, 221, 205, 155, 39, 0, 0, 0],
            &[0, 0, 97, 255, 255, 251, 255, 255, 244, 35, 0, 0],
            &[0, 0, 5, 90, 24, 0, 28, 221, 255, 122, 0, 0],
            &[0, 0, 0, 18, 75, 99, 107, 205, 255, 142, 0, 0],
            &[0, 0, 104, 247, 255, 255, 248, 251, 255, 143, 0, 0],
            &[0, 15, 248, 255, 125, 15, 0, 169, 255, 143, 0, 0],
            &[0, 42, 255, 255, 35, 0, 21, 230, 255, 143, 0, 0],
            &[0, 10, 244, 255, 231, 184, 236, 252, 255, 143, 0, 0],
            &[0, 0, 90, 242, 255, 250, 152, 100, 255, 143, 0, 0],
            &[0, 0, 0, 4, 31, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'b' / 0x62
        'b' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 40, 40, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 60, 166, 221, 198, 77, 0, 0, 0],
            &[0, 30, 255, 255, 230, 255, 255, 255, 252, 55, 0, 0],
            &[0, 30, 255, 255, 192, 28, 51, 231, 255, 167, 0, 0],
            &[0, 30, 255, 255, 72, 0, 0, 135, 255, 214, 0, 0],
            &[0, 30, 255, 255, 45, 0, 0, 106, 255, 238, 0, 0],
            &[0, 30, 255, 255, 63, 0, 0, 125, 255, 219, 0, 0],
            &[0, 30, 255, 255, 160, 0, 10, 211, 255, 181, 0, 0],
            &[0, 30, 255, 255, 255, 218, 232, 255, 255, 84, 0, 0],
            &[0, 30, 255, 210, 92, 235, 255, 249, 132, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 33, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'c' / 0x63
        'c' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 133, 202, 223, 207, 152, 26, 0, 0],
            &[0, 0, 13, 217, 255, 255, 255, 255, 245, 12, 0, 0],
            &[0, 0, 123, 255, 250, 96, 25, 46, 76, 0, 0, 0],
            &[0, 0, 193, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 218, 255, 126, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 255, 145, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 161, 255, 233, 34, 0, 1, 73, 6, 0, 0],
            &[0, 0, 53, 252, 255, 254, 221, 237, 255, 12, 0, 0],
            &[0, 0, 0, 75, 213, 255, 255, 254, 192, 6, 0, 0],
            &[0, 0, 0, 0, 0, 18, 34, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'd' / 0x64
        'd' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 22, 40, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 184, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 184, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 139, 255, 184, 0, 0],
            &[0, 0, 3, 126, 209, 213, 135, 130, 255, 184, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 233, 255, 184, 0, 0],
            &[0, 15, 249, 255, 178, 26, 42, 234, 255, 184, 0, 0],
            &[0, 67, 255, 255, 41, 0, 0, 134, 255, 184, 0, 0],
            &[0, 86, 255, 255, 5, 0, 0, 102, 255, 184, 0, 0],
            &[0, 68, 255, 255, 18, 0, 0, 113, 255, 184, 0, 0],
            &[0, 31, 255, 255, 110, 0, 7, 198, 255, 184, 0, 0],
            &[0, 0, 194, 255, 254, 210, 231, 249, 255, 184, 0, 0],
            &[0, 0, 32, 207, 255, 255, 173, 73, 255, 184, 0, 0],
            &[0, 0, 0, 0, 29, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'e' / 0x65
        'e' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 51, 168, 216, 213, 155, 29, 0, 0, 0],
            &[0, 0, 62, 248, 255, 244, 249, 255, 231, 23, 0, 0],
            &[0, 0, 206, 255, 163, 6, 11, 189, 255, 133, 0, 0],
            &[0, 22, 255, 255, 106, 71, 71, 138, 255, 188, 0, 0],
            &[0, 44, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 26, 255, 255, 100, 77, 77, 77, 77, 62, 0, 0],
            &[0, 0, 216, 255, 164, 11, 0, 0, 13, 35, 0, 0],
            &[0, 0, 79, 254, 255, 242, 202, 212, 249, 139, 0, 0],
            &[0, 0, 0, 72, 203, 255, 255, 255, 225, 87, 0, 0],
            &[0, 0, 0, 0, 0, 13, 34, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'f' / 0x66
        'f' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 55, 47, 13, 0, 0, 0],
            &[0, 0, 0, 6, 172, 255, 255, 255, 253, 32, 0, 0],
            &[0, 0, 0, 100, 255, 255, 221, 193, 189, 0, 0, 0],
            &[0, 0, 0, 145, 255, 216, 3, 0, 0, 0, 0, 0],
            &[0, 81, 140, 223, 255, 237, 188, 188, 102, 0, 0, 0],
            &[0, 233, 255, 255, 255, 255, 255, 255, 139, 0, 0, 0],
            &[0, 42, 47, 170, 255, 199, 47, 47, 25, 0, 0, 0],
            &[0, 0, 0, 151, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'g' / 0x67
        'g' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 139, 220, 214, 117, 38, 188, 135, 0, 0],
            &[0, 0, 152, 255, 255, 255, 255, 225, 255, 184, 0, 0],
            &[0, 19, 251, 255, 166, 23, 63, 242, 255, 184, 0, 0],
            &[0, 70, 255, 255, 36, 0, 0, 154, 255, 184, 0, 0],
            &[0, 88, 255, 255, 4, 0, 0, 123, 255, 184, 0, 0],
            &[0, 77, 255, 255, 22, 0, 0, 131, 255, 184, 0, 0],
            &[0, 32, 255, 255, 120, 0, 10, 210, 255, 184, 0, 0],
            &[0, 0, 188, 255, 255, 212, 233, 254, 255, 184, 0, 0],
            &[0, 0, 27, 200, 255, 255, 186, 166, 255, 184, 0, 0],
            &[0, 0, 0, 0, 27, 23, 0, 173, 255, 170, 0, 0],
            &[0, 0, 124, 93, 47, 33, 90, 247, 255, 118, 0, 0],
            &[0, 0, 97, 188, 222, 227, 202, 132, 16, 0, 0, 0],
        ]),
        // letter: 'h' / 0x68
        'h' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 40, 40, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 50, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 68, 161, 218, 199, 103, 0, 0, 0],
            &[0, 30, 255, 255, 227, 255, 255, 255, 255, 89, 0, 0],
            &[0, 30, 255, 255, 204, 43, 36, 221, 255, 175, 0, 0],
            &[0, 30, 255, 255, 89, 0, 0, 144, 255, 203, 0, 0],
            &[0, 30, 255, 255, 56, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'i' / 0x69
        'i' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 85, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 255, 255, 99, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 254, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 46, 0, 0, 0, 0, 0],
            &[0, 0, 114, 188, 188, 188, 188, 36, 0, 0, 0, 0],
            &[0, 0, 147, 255, 255, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 9, 63, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 4, 135, 177, 222, 255, 255, 223, 173, 105, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'j' / 0x6a
        'j' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 97, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 198, 255, 222, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 255, 205, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 60, 14, 0, 0, 0],
            &[0, 0, 0, 0, 176, 188, 188, 188, 162, 0, 0, 0],
            &[0, 0, 0, 0, 230, 255, 255, 255, 221, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 138, 255, 221, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 255, 221, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 255, 221, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 255, 221, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 255, 221, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 255, 221, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 117, 255, 221, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 216, 0, 0, 0],
            &[0, 0, 33, 58, 30, 64, 230, 255, 185, 0, 0, 0],
            &[0, 0, 55, 191, 223, 229, 192, 85, 0, 0, 0, 0],
        ]),
        // letter: 'k' / 0x6b
        'k' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 40, 40, 11, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 255, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 255, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 255, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 255, 255, 71, 0, 3, 155, 188, 152, 3, 0],
            &[0, 10, 255, 255, 71, 0, 149, 255, 235, 36, 0, 0],
            &[0, 10, 255, 255, 70, 110, 255, 246, 56, 0, 0, 0],
            &[0, 10, 255, 255, 124, 252, 255, 79, 0, 0, 0, 0],
            &[0, 10, 255, 255, 247, 255, 255, 90, 0, 0, 0, 0],
            &[0, 10, 255, 255, 246, 165, 255, 237, 24, 0, 0, 0],
            &[0, 10, 255, 255, 87, 1, 191, 255, 176, 0, 0, 0],
            &[0, 10, 255, 255, 71, 0, 36, 246, 255, 91, 0, 0],
            &[0, 10, 255, 255, 71, 0, 0, 117, 255, 237, 24, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'l' / 0x6c
        'l' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 40, 40, 40, 40, 6, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 40, 123, 185, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 4, 135, 178, 223, 255, 255, 222, 173, 104, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'm' / 0x6d
        'm' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 97, 188, 54, 194, 205, 41, 133, 219, 114, 0, 0],
            &[0, 133, 255, 232, 255, 255, 209, 255, 255, 243, 1, 0],
            &[0, 133, 255, 164, 82, 255, 238, 28, 239, 255, 15, 0],
            &[0, 133, 255, 124, 53, 255, 212, 0, 225, 255, 29, 0],
            &[0, 133, 255, 122, 53, 255, 210, 0, 225, 255, 30, 0],
            &[0, 133, 255, 122, 53, 255, 210, 0, 225, 255, 30, 0],
            &[0, 133, 255, 122, 53, 255, 210, 0, 225, 255, 30, 0],
            &[0, 133, 255, 122, 53, 255, 210, 0, 225, 255, 30, 0],
            &[0, 133, 255, 122, 53, 255, 210, 0, 225, 255, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'n' / 0x6e
        'n' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 188, 144, 31, 164, 217, 200, 107, 1, 0, 0],
            &[0, 30, 255, 251, 223, 255, 255, 255, 255, 91, 0, 0],
            &[0, 30, 255, 255, 215, 39, 39, 225, 255, 175, 0, 0],
            &[0, 30, 255, 255, 99, 0, 0, 145, 255, 203, 0, 0],
            &[0, 30, 255, 255, 60, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'o' / 0x6f
        'o' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 61, 175, 218, 209, 145, 23, 0, 0, 0],
            &[0, 0, 84, 252, 255, 255, 255, 255, 225, 19, 0, 0],
            &[0, 4, 233, 255, 170, 14, 40, 239, 255, 141, 0, 0],
            &[0, 59, 255, 255, 38, 0, 0, 141, 255, 215, 0, 0],
            &[0, 85, 255, 255, 5, 0, 0, 109, 255, 239, 0, 0],
            &[0, 73, 255, 255, 29, 0, 0, 130, 255, 226, 0, 0],
            &[0, 20, 250, 255, 128, 0, 7, 215, 255, 167, 0, 0],
            &[0, 0, 144, 255, 254, 199, 222, 255, 249, 48, 0, 0],
            &[0, 0, 3, 134, 242, 255, 255, 213, 66, 0, 0, 0],
            &[0, 0, 0, 0, 5, 33, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'p' / 0x70
        'p' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 188, 144, 27, 161, 218, 199, 78, 0, 0, 0],
            &[0, 30, 255, 252, 221, 255, 255, 255, 252, 55, 0, 0],
            &[0, 30, 255, 255, 196, 30, 50, 233, 255, 167, 0, 0],
            &[0, 30, 255, 255, 70, 0, 0, 145, 255, 214, 0, 0],
            &[0, 30, 255, 255, 42, 0, 0, 118, 255, 238, 0, 0],
            &[0, 30, 255, 255, 63, 0, 0, 133, 255, 228, 0, 0],
            &[0, 30, 255, 255, 161, 0, 7, 209, 255, 180, 0, 0],
            &[0, 30, 255, 255, 255, 219, 229, 255, 255, 80, 0, 0],
            &[0, 30, 255, 255, 113, 233, 255, 248, 125, 0, 0, 0],
            &[0, 30, 255, 255, 38, 4, 31, 10, 0, 0, 0, 0],
            &[0, 30, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 214, 214, 32, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'q' / 0x71
        'q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 136, 216, 212, 127, 33, 188, 135, 0, 0],
            &[0, 0, 153, 255, 255, 255, 255, 225, 255, 184, 0, 0],
            &[0, 18, 251, 255, 176, 24, 63, 240, 255, 184, 0, 0],
            &[0, 61, 255, 255, 48, 0, 0, 151, 255, 184, 0, 0],
            &[0, 84, 255, 255, 16, 0, 0, 118, 255, 184, 0, 0],
            &[0, 68, 255, 255, 32, 0, 0, 129, 255, 184, 0, 0],
            &[0, 31, 255, 255, 124, 0, 10, 211, 255, 184, 0, 0],
            &[0, 0, 194, 255, 255, 212, 233, 250, 255, 184, 0, 0],
            &[0, 0, 33, 208, 255, 255, 183, 150, 255, 184, 0, 0],
            &[0, 0, 0, 0, 29, 20, 0, 141, 255, 184, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 184, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 118, 214, 154, 0, 0],
        ]),
        // letter: 'r' / 0x72
        'r' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 171, 188, 188, 145, 11, 148, 218, 210, 41, 0],
            &[0, 0, 226, 255, 255, 244, 181, 255, 255, 247, 9, 0],
            &[0, 0, 1, 45, 255, 255, 252, 140, 90, 91, 0, 0],
            &[0, 0, 0, 18, 255, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 79, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 63, 0, 0, 0, 0, 0],
            &[0, 41, 158, 213, 255, 255, 225, 168, 68, 0, 0, 0],
            &[0, 88, 255, 255, 255, 255, 255, 255, 137, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 's' / 0x73
        's' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 87, 187, 222, 214, 185, 116, 17, 0, 0],
            &[0, 0, 87, 255, 255, 255, 255, 255, 245, 15, 0, 0],
            &[0, 0, 158, 255, 219, 16, 20, 63, 88, 0, 0, 0],
            &[0, 0, 122, 255, 254, 170, 70, 1, 0, 0, 0, 0],
            &[0, 0, 4, 146, 251, 255, 255, 217, 69, 0, 0, 0],
            &[0, 0, 0, 0, 31, 132, 235, 255, 248, 26, 0, 0],
            &[0, 0, 72, 27, 0, 0, 54, 255, 255, 73, 0, 0],
            &[0, 0, 184, 254, 223, 204, 237, 255, 253, 34, 0, 0],
            &[0, 0, 136, 242, 255, 255, 255, 223, 91, 0, 0, 0],
            &[0, 0, 0, 3, 25, 36, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 't' / 0x74
        't' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 233, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 215, 255, 42, 0, 0, 0, 0, 0],
            &[0, 10, 102, 158, 252, 255, 199, 188, 188, 84, 0, 0],
            &[0, 38, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 7, 47, 78, 255, 255, 82, 47, 47, 21, 0, 0],
            &[0, 0, 0, 38, 255, 255, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 38, 255, 255, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 38, 255, 255, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 255, 255, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 221, 255, 251, 208, 218, 131, 0, 0],
            &[0, 0, 0, 0, 54, 215, 255, 255, 253, 117, 0, 0],
            &[0, 0, 0, 0, 0, 0, 24, 35, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'u' / 0x75
        'u' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 188, 188, 22, 0, 0, 113, 188, 135, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 158, 255, 184, 0, 0],
            &[0, 51, 255, 255, 35, 0, 0, 182, 255, 184, 0, 0],
            &[0, 31, 255, 255, 91, 0, 26, 241, 255, 184, 0, 0],
            &[0, 2, 223, 255, 249, 204, 241, 249, 255, 184, 0, 0],
            &[0, 0, 60, 226, 255, 254, 176, 68, 255, 184, 0, 0],
            &[0, 0, 0, 1, 28, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'v' / 0x76
        'v' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 102, 188, 174, 1, 0, 0, 53, 188, 188, 26, 0],
            &[0, 56, 255, 255, 53, 0, 0, 142, 255, 209, 0, 0],
            &[0, 0, 216, 255, 135, 0, 0, 222, 255, 113, 0, 0],
            &[0, 0, 122, 255, 216, 0, 47, 255, 251, 22, 0, 0],
            &[0, 0, 29, 253, 255, 42, 127, 255, 178, 0, 0, 0],
            &[0, 0, 0, 187, 255, 120, 204, 255, 82, 0, 0, 0],
            &[0, 0, 0, 92, 255, 195, 253, 236, 6, 0, 0, 0],
            &[0, 0, 0, 10, 242, 255, 255, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'w' / 0x77
        'w' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 169, 188, 36, 0, 0, 0, 0, 102, 188, 97, 0],
            &[0, 199, 255, 68, 25, 116, 99, 0, 159, 255, 100, 0],
            &[0, 162, 255, 91, 93, 255, 249, 6, 181, 255, 64, 0],
            &[0, 126, 255, 114, 145, 255, 255, 50, 204, 255, 28, 0],
            &[0, 90, 255, 137, 196, 200, 255, 101, 227, 246, 1, 0],
            &[0, 54, 255, 162, 243, 135, 237, 150, 247, 212, 0, 0],
            &[0, 18, 255, 210, 255, 89, 192, 207, 255, 176, 0, 0],
            &[0, 0, 238, 255, 255, 41, 146, 255, 255, 140, 0, 0],
            &[0, 0, 202, 255, 247, 3, 100, 255, 255, 104, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'x' / 0x78
        'x' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 53, 188, 188, 86, 0, 0, 136, 188, 178, 14, 0],
            &[0, 0, 175, 255, 231, 13, 64, 255, 255, 105, 0, 0],
            &[0, 0, 22, 234, 255, 142, 213, 255, 181, 0, 0, 0],
            &[0, 0, 0, 80, 255, 255, 255, 234, 23, 0, 0, 0],
            &[0, 0, 0, 0, 196, 255, 255, 114, 0, 0, 0, 0],
            &[0, 0, 0, 65, 254, 255, 255, 221, 11, 0, 0, 0],
            &[0, 0, 15, 226, 255, 166, 228, 255, 150, 0, 0, 0],
            &[0, 0, 163, 255, 238, 20, 92, 255, 254, 68, 0, 0],
            &[0, 85, 255, 255, 104, 0, 1, 200, 255, 226, 14, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'y' / 0x79
        'y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 188, 179, 3, 0, 0, 61, 188, 188, 27, 0],
            &[0, 48, 255, 255, 66, 0, 0, 148, 255, 212, 0, 0],
            &[0, 0, 202, 255, 152, 0, 0, 224, 255, 119, 0, 0],
            &[0, 0, 101, 255, 234, 3, 44, 255, 253, 28, 0, 0],
            &[0, 0, 12, 243, 255, 67, 120, 255, 189, 0, 0, 0],
            &[0, 0, 0, 154, 255, 150, 195, 255, 96, 0, 0, 0],
            &[0, 0, 0, 52, 255, 219, 250, 245, 13, 0, 0, 0],
            &[0, 0, 0, 0, 207, 255, 255, 166, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 255, 255, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 255, 230, 3, 0, 0, 0, 0],
            &[0, 2, 32, 76, 237, 255, 122, 0, 0, 0, 0, 0],
            &[0, 11, 222, 224, 160, 23, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'z' / 0x7a
        'z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 135, 188, 188, 188, 188, 188, 188, 78, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 102, 0, 0],
            &[0, 0, 16, 22, 22, 32, 218, 255, 207, 10, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 233, 30, 0, 0, 0],
            &[0, 0, 0, 0, 118, 255, 249, 59, 0, 0, 0, 0],
            &[0, 0, 0, 71, 253, 255, 97, 0, 0, 0, 0, 0],
            &[0, 0, 36, 239, 255, 142, 0, 0, 0, 0, 0, 0],
            &[0, 0, 209, 255, 255, 214, 210, 210, 210, 118, 0, 0],
            &[0, 0, 245, 255, 255, 255, 255, 255, 255, 143, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '{' / 0x7b
        '{' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 66, 111, 49, 0, 0, 0],
            &[0, 0, 0, 0, 0, 156, 255, 255, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 244, 255, 129, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 255, 255, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 255, 254, 2, 0, 0, 0, 0],
            &[0, 0, 0, 76, 180, 255, 208, 0, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 212, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 98, 201, 255, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 255, 254, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 255, 255, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 255, 255, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 247, 255, 108, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 174, 255, 255, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 93, 138, 60, 0, 0, 0],
        ]),
        // letter: '|' / 0x7c
        '|' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 40, 27, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 214, 146, 0, 0, 0, 0, 0],
        ]),
        // letter: '}' / 0x7d
        '}' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 100, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 251, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 66, 197, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 255, 153, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 255, 153, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 104, 255, 166, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 57, 255, 248, 126, 36, 0, 0, 0],
            &[0, 0, 0, 0, 0, 104, 247, 255, 104, 0, 0, 0],
            &[0, 0, 0, 0, 46, 250, 253, 152, 47, 0, 0, 0],
            &[0, 0, 0, 0, 102, 255, 172, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 255, 153, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 255, 153, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 186, 255, 145, 0, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 118, 126, 55, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '~' / 0x7e
        '~' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 48, 76, 7, 0, 0, 63, 45, 0, 0],
            &[0, 0, 96, 248, 233, 214, 33, 0, 202, 104, 0, 0],
            &[0, 0, 204, 105, 9, 174, 233, 155, 249, 32, 0, 0],
            &[0, 0, 155, 42, 0, 1, 101, 156, 71, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ' ' / 0xa0
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¡' / 0xa1
        '¡' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 143, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 255, 242, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 171, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 255, 231, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 255, 242, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 104, 255, 253, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 255, 255, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 126, 255, 255, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 132, 245, 245, 30, 0, 0, 0, 0],
        ]),
        // letter: '¢' / 0xa2
        '¢' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 149, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 95, 255, 85, 31, 0, 0, 0],
            &[0, 0, 0, 84, 232, 255, 255, 255, 255, 79, 0, 0],
            &[0, 0, 50, 252, 255, 198, 132, 148, 208, 14, 0, 0],
            &[0, 0, 157, 255, 186, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 206, 255, 101, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 91, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 182, 255, 144, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 98, 255, 250, 117, 51, 67, 127, 48, 0, 0],
            &[0, 0, 2, 174, 255, 255, 255, 255, 255, 65, 0, 0],
            &[0, 0, 0, 0, 73, 163, 255, 158, 95, 9, 0, 0],
            &[0, 0, 0, 0, 0, 55, 255, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 40, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '£' / 0xa3
        '£' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 85, 135, 129, 80, 6, 0, 0],
            &[0, 0, 0, 5, 196, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 82, 255, 254, 137, 101, 164, 8, 0, 0],
            &[0, 0, 0, 120, 255, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 124, 255, 192, 0, 0, 0, 0, 0, 0],
            &[0, 0, 123, 212, 255, 235, 171, 171, 61, 0, 0, 0],
            &[0, 0, 182, 254, 255, 255, 253, 253, 91, 0, 0, 0],
            &[0, 0, 0, 124, 255, 192, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 140, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 224, 246, 66, 0, 0, 0, 0, 0, 0],
            &[0, 0, 177, 255, 253, 247, 247, 247, 247, 158, 0, 0],
            &[0, 0, 180, 255, 255, 255, 255, 255, 255, 163, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¤' / 0xa4
        '¤' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 85, 3, 39, 22, 8, 132, 11, 0, 0],
            &[0, 0, 193, 251, 227, 255, 253, 222, 255, 85, 0, 0],
            &[0, 0, 33, 255, 234, 131, 165, 255, 182, 0, 0, 0],
            &[0, 0, 96, 255, 84, 0, 0, 185, 244, 1, 0, 0],
            &[0, 0, 101, 255, 77, 0, 0, 179, 248, 2, 0, 0],
            &[0, 0, 33, 255, 225, 110, 146, 254, 181, 0, 0, 0],
            &[0, 0, 179, 254, 242, 255, 255, 241, 251, 69, 0, 0],
            &[0, 0, 77, 103, 12, 60, 40, 16, 155, 15, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¥' / 0xa5
        '¥' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 53, 108, 91, 0, 0, 0, 26, 108, 108, 10, 0],
            &[0, 38, 252, 254, 42, 0, 0, 144, 255, 186, 0, 0],
            &[0, 0, 163, 255, 156, 0, 15, 242, 255, 57, 0, 0],
            &[0, 0, 38, 252, 247, 22, 116, 255, 182, 0, 0, 0],
            &[0, 0, 0, 163, 255, 132, 226, 255, 53, 0, 0, 0],
            &[0, 0, 0, 38, 252, 252, 255, 178, 0, 0, 0, 0],
            &[0, 0, 95, 161, 236, 255, 255, 194, 161, 31, 0, 0],
            &[0, 0, 115, 194, 212, 255, 249, 194, 194, 37, 0, 0],
            &[0, 0, 23, 38, 101, 255, 231, 38, 38, 7, 0, 0],
            &[0, 0, 151, 255, 255, 255, 255, 255, 255, 49, 0, 0],
            &[0, 0, 33, 57, 114, 255, 233, 57, 57, 11, 0, 0],
            &[0, 0, 0, 0, 73, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¦' / 0xa6
        '¦' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 32, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 53, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 171, 116, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 174, 118, 0, 0, 0, 0, 0],
        ]),
        // letter: '§' / 0xa7
        '§' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 46, 47, 14, 0, 0, 0, 0],
            &[0, 0, 0, 112, 242, 255, 255, 252, 144, 0, 0, 0],
            &[0, 0, 31, 255, 246, 90, 79, 148, 83, 0, 0, 0],
            &[0, 0, 24, 254, 244, 58, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 121, 255, 255, 187, 60, 0, 0, 0, 0],
            &[0, 0, 7, 199, 243, 169, 249, 254, 101, 0, 0, 0],
            &[0, 0, 72, 255, 165, 0, 55, 255, 232, 0, 0, 0],
            &[0, 0, 38, 255, 243, 91, 26, 253, 219, 0, 0, 0],
            &[0, 0, 0, 99, 243, 255, 246, 251, 62, 0, 0, 0],
            &[0, 0, 0, 0, 21, 138, 244, 255, 114, 0, 0, 0],
            &[0, 0, 2, 0, 0, 0, 78, 255, 209, 0, 0, 0],
            &[0, 0, 89, 187, 112, 86, 176, 255, 170, 0, 0, 0],
            &[0, 0, 54, 218, 255, 255, 254, 177, 21, 0, 0, 0],
            &[0, 0, 0, 0, 16, 33, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¨' / 0xa8
        '¨' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 3, 0, 0, 9, 0, 0, 0, 0],
            &[0, 0, 17, 235, 227, 9, 71, 249, 171, 0, 0, 0],
            &[0, 0, 13, 245, 242, 13, 70, 255, 190, 0, 0, 0],
            &[0, 0, 0, 22, 17, 0, 0, 34, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '©' / 0xa9
        '©' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 87, 193, 233, 222, 160, 35, 0, 0, 0],
            &[0, 0, 132, 230, 98, 97, 114, 150, 237, 43, 0, 0],
            &[0, 43, 249, 52, 208, 223, 208, 84, 132, 191, 0, 0],
            &[0, 121, 184, 95, 237, 11, 0, 0, 26, 254, 17, 0],
            &[0, 141, 156, 123, 201, 0, 0, 0, 3, 255, 39, 0],
            &[0, 110, 195, 81, 242, 30, 6, 14, 44, 253, 14, 0],
            &[0, 27, 245, 70, 165, 253, 251, 74, 172, 177, 0, 0],
            &[0, 0, 96, 243, 136, 91, 109, 192, 219, 26, 0, 0],
            &[0, 0, 0, 50, 150, 187, 175, 113, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ª' / 0xaa
        'ª' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 71, 124, 124, 41, 0, 0, 0, 0],
            &[0, 0, 0, 16, 220, 173, 222, 242, 20, 0, 0, 0],
            &[0, 0, 0, 0, 31, 81, 148, 255, 69, 0, 0, 0],
            &[0, 0, 0, 98, 251, 176, 161, 255, 75, 0, 0, 0],
            &[0, 0, 0, 171, 236, 31, 157, 255, 75, 0, 0, 0],
            &[0, 0, 0, 78, 245, 250, 154, 223, 75, 0, 0, 0],
            &[0, 0, 0, 0, 5, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '«' / 0xab
        '«' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 181, 136, 9, 87, 193, 50, 0, 0],
            &[0, 0, 0, 133, 255, 193, 40, 242, 251, 53, 0, 0],
            &[0, 0, 73, 254, 241, 41, 209, 255, 128, 0, 0, 0],
            &[0, 2, 231, 255, 122, 125, 255, 230, 4, 0, 0, 0],
            &[0, 0, 116, 255, 221, 40, 235, 255, 88, 0, 0, 0],
            &[0, 0, 0, 177, 255, 156, 71, 253, 238, 27, 0, 0],
            &[0, 0, 0, 17, 223, 192, 11, 131, 236, 76, 0, 0],
            &[0, 0, 0, 0, 17, 0, 0, 3, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¬' / 0xac
        '¬' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 126, 126, 126, 126, 126, 126, 40, 0, 0],
            &[0, 0, 225, 255, 255, 255, 255, 255, 255, 81, 0, 0],
            &[0, 0, 45, 51, 51, 51, 51, 131, 255, 81, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 100, 255, 81, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 100, 255, 81, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 8, 20, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '®' / 0xae
        '®' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 86, 193, 233, 222, 159, 33, 0, 0, 0],
            &[0, 0, 129, 231, 133, 76, 62, 141, 236, 40, 0, 0],
            &[0, 40, 250, 45, 237, 205, 241, 84, 133, 188, 0, 0],
            &[0, 117, 188, 0, 237, 57, 182, 143, 29, 254, 14, 0],
            &[0, 137, 160, 0, 237, 228, 247, 44, 7, 255, 35, 0],
            &[0, 105, 198, 0, 237, 65, 238, 51, 48, 252, 11, 0],
            &[0, 24, 243, 71, 205, 49, 135, 137, 175, 174, 0, 0],
            &[0, 0, 90, 243, 140, 83, 99, 196, 218, 25, 0, 0],
            &[0, 0, 0, 45, 145, 182, 172, 110, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¯' / 0xaf
        '¯' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 84, 137, 137, 137, 137, 31, 0, 0, 0],
            &[0, 0, 0, 157, 255, 255, 255, 255, 59, 0, 0, 0],
            &[0, 0, 0, 7, 12, 12, 12, 12, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '°' / 0xb0
        '°' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 53, 137, 117, 16, 0, 0, 0, 0],
            &[0, 0, 0, 67, 252, 231, 249, 213, 7, 0, 0, 0],
            &[0, 0, 0, 173, 220, 3, 66, 255, 75, 0, 0, 0],
            &[0, 0, 0, 166, 228, 19, 89, 255, 69, 0, 0, 0],
            &[0, 0, 0, 53, 245, 251, 255, 201, 5, 0, 0, 0],
            &[0, 0, 0, 0, 31, 105, 88, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '±' / 0xb1
        '±' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 19, 22, 29, 255, 181, 22, 22, 10, 0, 0],
            &[0, 0, 217, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 131, 155, 158, 255, 223, 155, 155, 74, 0, 0],
            &[0, 0, 0, 0, 8, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 133, 90, 0, 0, 0, 0, 0],
            &[0, 0, 160, 182, 182, 182, 182, 182, 182, 87, 0, 0],
            &[0, 0, 225, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '²' / 0xb2
        '²' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 91, 159, 144, 36, 0, 0, 0, 0],
            &[0, 0, 0, 91, 251, 207, 250, 213, 0, 0, 0, 0],
            &[0, 0, 0, 2, 46, 0, 176, 246, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 27, 235, 178, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 217, 217, 22, 0, 0, 0, 0],
            &[0, 0, 0, 33, 223, 218, 35, 12, 4, 0, 0, 0],
            &[0, 0, 0, 137, 255, 255, 255, 255, 92, 0, 0, 0],
            &[0, 0, 0, 47, 88, 88, 88, 88, 31, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '³' / 0xb3
        '³' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 96, 151, 136, 29, 0, 0, 0, 0],
            &[0, 0, 0, 88, 242, 189, 246, 209, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 0, 174, 221, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 213, 242, 60, 0, 0, 0, 0],
            &[0, 0, 0, 0, 53, 136, 225, 205, 5, 0, 0, 0],
            &[0, 0, 0, 56, 16, 0, 140, 255, 39, 0, 0, 0],
            &[0, 0, 0, 144, 254, 235, 255, 210, 5, 0, 0, 0],
            &[0, 0, 0, 10, 84, 113, 86, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '´' / 0xb4
        '´' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 60, 65, 43, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 255, 234, 43, 0, 0, 0, 0],
            &[0, 0, 0, 23, 233, 223, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 77, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'µ' / 0xb5
        'µ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 188, 188, 22, 0, 0, 113, 188, 135, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 158, 255, 184, 0, 0],
            &[0, 83, 255, 255, 55, 0, 0, 182, 255, 184, 0, 0],
            &[0, 82, 255, 255, 122, 0, 26, 241, 255, 184, 0, 0],
            &[0, 53, 255, 255, 255, 204, 241, 249, 255, 184, 0, 0],
            &[0, 51, 255, 255, 255, 254, 176, 68, 255, 184, 0, 0],
            &[0, 51, 255, 255, 58, 14, 0, 0, 0, 0, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 214, 214, 25, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¶' / 0xb6
        '¶' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 39, 40, 40, 40, 32, 0, 0],
            &[0, 0, 40, 198, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 1, 216, 255, 255, 255, 255, 137, 137, 204, 0, 0],
            &[0, 53, 255, 255, 255, 255, 255, 110, 110, 204, 0, 0],
            &[0, 83, 255, 255, 255, 255, 255, 110, 110, 204, 0, 0],
            &[0, 78, 255, 255, 255, 255, 255, 110, 110, 204, 0, 0],
            &[0, 35, 255, 255, 255, 255, 255, 110, 110, 204, 0, 0],
            &[0, 0, 174, 255, 255, 255, 255, 110, 110, 204, 0, 0],
            &[0, 0, 6, 116, 182, 190, 236, 110, 110, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 200, 110, 110, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 200, 110, 110, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 200, 110, 110, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 200, 110, 110, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 200, 110, 110, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 200, 110, 110, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 3, 3, 6, 0, 0],
        ]),
        // letter: '·' / 0xb7
        '·' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 235, 200, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 157, 255, 255, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 199, 162, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¸' / 0xb8
        '¸' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 241, 114, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 150, 255, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 160, 207, 159, 16, 0, 0, 0, 0],
        ]),
        // letter: '¹' / 0xb9
        '¹' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 93, 112, 0, 0, 0, 0, 0],
            &[0, 0, 0, 35, 191, 255, 223, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 207, 196, 223, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 167, 223, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 169, 223, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 169, 223, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 192, 255, 255, 197, 78, 0, 0, 0],
            &[0, 0, 0, 23, 88, 146, 164, 88, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'º' / 0xba
        'º' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 126, 108, 11, 0, 0, 0, 0],
            &[0, 0, 0, 53, 250, 204, 237, 195, 0, 0, 0, 0],
            &[0, 0, 0, 150, 247, 4, 99, 255, 46, 0, 0, 0],
            &[0, 0, 0, 166, 229, 0, 83, 255, 60, 0, 0, 0],
            &[0, 0, 0, 116, 255, 70, 162, 247, 19, 0, 0, 0],
            &[0, 0, 0, 9, 178, 254, 242, 96, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '»' / 0xbb
        '»' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 201, 17, 46, 188, 92, 0, 0, 0, 0],
            &[0, 0, 153, 255, 177, 50, 250, 244, 42, 0, 0, 0],
            &[0, 0, 11, 220, 255, 116, 121, 255, 213, 11, 0, 0],
            &[0, 0, 0, 81, 255, 251, 31, 222, 255, 131, 0, 0],
            &[0, 0, 1, 188, 255, 161, 79, 255, 238, 31, 0, 0],
            &[0, 0, 110, 255, 212, 33, 234, 254, 77, 0, 0, 0],
            &[0, 0, 162, 243, 41, 73, 235, 137, 0, 0, 0, 0],
            &[0, 0, 0, 18, 0, 0, 13, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¼' / 0xbc
        '¼' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 46, 108, 2, 0, 0, 0, 0, 0, 0, 0],
            &[0, 128, 252, 255, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 74, 114, 255, 6, 0, 0, 22, 208, 161, 0, 0],
            &[0, 0, 65, 255, 6, 0, 2, 187, 234, 26, 0, 0],
            &[0, 9, 100, 255, 36, 3, 126, 253, 70, 0, 0, 0],
            &[0, 116, 255, 255, 245, 120, 253, 131, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 232, 191, 13, 96, 40, 0, 0],
            &[0, 0, 0, 3, 191, 232, 25, 142, 255, 106, 0, 0],
            &[0, 0, 0, 132, 253, 67, 58, 217, 227, 106, 0, 0],
            &[0, 0, 71, 254, 127, 8, 216, 79, 227, 116, 4, 0],
            &[0, 13, 192, 169, 2, 55, 255, 255, 255, 255, 63, 0],
            &[0, 0, 0, 0, 0, 0, 2, 2, 225, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '½' / 0xbd
        '½' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 46, 108, 2, 0, 0, 0, 0, 0, 0, 0],
            &[0, 128, 252, 255, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 74, 114, 255, 6, 0, 0, 22, 208, 161, 0, 0],
            &[0, 0, 65, 255, 6, 0, 2, 187, 234, 26, 0, 0],
            &[0, 9, 100, 255, 36, 3, 126, 253, 70, 0, 0, 0],
            &[0, 116, 255, 255, 245, 120, 253, 131, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 232, 192, 81, 118, 48, 0, 0],
            &[0, 0, 0, 3, 191, 232, 99, 221, 204, 243, 11, 0],
            &[0, 0, 0, 132, 253, 67, 0, 6, 89, 251, 12, 0],
            &[0, 0, 71, 254, 127, 0, 0, 48, 237, 119, 0, 0],
            &[0, 13, 192, 169, 2, 0, 43, 235, 167, 43, 14, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 255, 255, 88, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¾' / 0xbe
        '¾' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 100, 136, 69, 0, 0, 0, 0, 0, 0, 0],
            &[0, 56, 194, 171, 253, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 138, 227, 9, 0, 22, 208, 161, 0, 0],
            &[0, 0, 102, 205, 203, 17, 2, 187, 234, 26, 0, 0],
            &[0, 52, 59, 67, 255, 70, 126, 253, 70, 0, 0, 0],
            &[0, 74, 237, 249, 178, 76, 253, 131, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 232, 191, 13, 96, 40, 0, 0],
            &[0, 0, 0, 3, 191, 232, 25, 142, 255, 106, 0, 0],
            &[0, 0, 0, 132, 253, 67, 58, 217, 227, 106, 0, 0],
            &[0, 0, 71, 254, 127, 8, 216, 79, 227, 116, 4, 0],
            &[0, 13, 192, 169, 2, 55, 255, 255, 255, 255, 63, 0],
            &[0, 0, 0, 0, 0, 0, 2, 2, 225, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¿' / 0xbf
        '¿' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 140, 117, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 134, 255, 255, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 95, 255, 246, 33, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 171, 155, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 104, 255, 205, 0, 0, 0, 0],
            &[0, 0, 0, 0, 114, 252, 253, 86, 0, 0, 0, 0],
            &[0, 0, 0, 125, 255, 243, 80, 0, 0, 0, 0, 0],
            &[0, 0, 2, 247, 255, 94, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 255, 255, 83, 0, 4, 84, 32, 0, 0],
            &[0, 0, 0, 206, 255, 253, 218, 245, 255, 157, 0, 0],
            &[0, 0, 0, 0, 0, 10, 25, 6, 0, 0, 0, 0],
        ]),
        // letter: 'À' / 0xc0
        'À' => Some(&[
            &[0, 0, 0, 140, 241, 212, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 151, 255, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 109, 156, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 120, 120, 40, 0, 0, 0, 0],
            &[0, 0, 0, 3, 240, 255, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 58, 255, 226, 255, 208, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 135, 235, 254, 24, 0, 0, 0],
            &[0, 0, 0, 199, 255, 73, 175, 255, 94, 0, 0, 0],
            &[0, 0, 17, 252, 252, 13, 114, 255, 165, 0, 0, 0],
            &[0, 0, 84, 255, 201, 0, 52, 255, 234, 1, 0, 0],
            &[0, 0, 154, 255, 255, 192, 196, 255, 255, 50, 0, 0],
            &[0, 0, 224, 255, 255, 255, 255, 255, 255, 121, 0, 0],
            &[0, 39, 255, 255, 56, 42, 42, 152, 255, 191, 0, 0],
            &[0, 109, 255, 202, 0, 0, 0, 61, 255, 250, 12, 0],
            &[0, 179, 255, 138, 0, 0, 0, 7, 248, 255, 77, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Á' / 0xc1
        'Á' => Some(&[
            &[0, 0, 0, 0, 0, 18, 225, 241, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 167, 255, 127, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 157, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 120, 120, 40, 0, 0, 0, 0],
            &[0, 0, 0, 3, 240, 255, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 58, 255, 226, 255, 208, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 135, 235, 254, 24, 0, 0, 0],
            &[0, 0, 0, 199, 255, 73, 175, 255, 94, 0, 0, 0],
            &[0, 0, 17, 252, 252, 13, 114, 255, 165, 0, 0, 0],
            &[0, 0, 84, 255, 201, 0, 52, 255, 234, 1, 0, 0],
            &[0, 0, 154, 255, 255, 192, 196, 255, 255, 50, 0, 0],
            &[0, 0, 224, 255, 255, 255, 255, 255, 255, 121, 0, 0],
            &[0, 39, 255, 255, 56, 42, 42, 152, 255, 191, 0, 0],
            &[0, 109, 255, 202, 0, 0, 0, 61, 255, 250, 12, 0],
            &[0, 179, 255, 138, 0, 0, 0, 7, 248, 255, 77, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Â' / 0xc2
        'Â' => Some(&[
            &[0, 0, 0, 0, 162, 241, 241, 71, 0, 0, 0, 0],
            &[0, 0, 0, 97, 255, 154, 218, 231, 25, 0, 0, 0],
            &[0, 0, 9, 150, 106, 0, 21, 147, 97, 0, 0, 0],
            &[0, 0, 0, 0, 90, 120, 120, 40, 0, 0, 0, 0],
            &[0, 0, 0, 3, 240, 255, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 58, 255, 226, 255, 208, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 135, 235, 254, 24, 0, 0, 0],
            &[0, 0, 0, 199, 255, 73, 175, 255, 94, 0, 0, 0],
            &[0, 0, 17, 252, 252, 13, 114, 255, 165, 0, 0, 0],
            &[0, 0, 84, 255, 201, 0, 52, 255, 234, 1, 0, 0],
            &[0, 0, 154, 255, 255, 192, 196, 255, 255, 50, 0, 0],
            &[0, 0, 224, 255, 255, 255, 255, 255, 255, 121, 0, 0],
            &[0, 39, 255, 255, 56, 42, 42, 152, 255, 191, 0, 0],
            &[0, 109, 255, 202, 0, 0, 0, 61, 255, 250, 12, 0],
            &[0, 179, 255, 138, 0, 0, 0, 7, 248, 255, 77, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ã' / 0xc3
        'Ã' => Some(&[
            &[0, 0, 0, 79, 182, 129, 24, 96, 155, 0, 0, 0],
            &[0, 0, 11, 242, 227, 251, 254, 254, 136, 0, 0, 0],
            &[0, 0, 30, 158, 6, 29, 125, 127, 9, 0, 0, 0],
            &[0, 0, 0, 0, 90, 120, 120, 40, 0, 0, 0, 0],
            &[0, 0, 0, 3, 240, 255, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 58, 255, 226, 255, 208, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 135, 235, 254, 24, 0, 0, 0],
            &[0, 0, 0, 199, 255, 73, 175, 255, 94, 0, 0, 0],
            &[0, 0, 17, 252, 252, 13, 114, 255, 165, 0, 0, 0],
            &[0, 0, 84, 255, 201, 0, 52, 255, 234, 1, 0, 0],
            &[0, 0, 154, 255, 255, 192, 196, 255, 255, 50, 0, 0],
            &[0, 0, 224, 255, 255, 255, 255, 255, 255, 121, 0, 0],
            &[0, 39, 255, 255, 56, 42, 42, 152, 255, 191, 0, 0],
            &[0, 109, 255, 202, 0, 0, 0, 61, 255, 250, 12, 0],
            &[0, 179, 255, 138, 0, 0, 0, 7, 248, 255, 77, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ä' / 0xc4
        'Ä' => Some(&[
            &[0, 0, 10, 166, 143, 0, 45, 180, 96, 0, 0, 0],
            &[0, 0, 36, 255, 254, 9, 105, 255, 198, 0, 0, 0],
            &[0, 0, 0, 99, 78, 0, 14, 115, 47, 0, 0, 0],
            &[0, 0, 0, 0, 90, 120, 120, 40, 0, 0, 0, 0],
            &[0, 0, 0, 3, 240, 255, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 58, 255, 226, 255, 208, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 135, 235, 254, 24, 0, 0, 0],
            &[0, 0, 0, 199, 255, 73, 175, 255, 94, 0, 0, 0],
            &[0, 0, 17, 252, 252, 13, 114, 255, 165, 0, 0, 0],
            &[0, 0, 84, 255, 201, 0, 52, 255, 234, 1, 0, 0],
            &[0, 0, 154, 255, 255, 192, 196, 255, 255, 50, 0, 0],
            &[0, 0, 224, 255, 255, 255, 255, 255, 255, 121, 0, 0],
            &[0, 39, 255, 255, 56, 42, 42, 152, 255, 191, 0, 0],
            &[0, 109, 255, 202, 0, 0, 0, 61, 255, 250, 12, 0],
            &[0, 179, 255, 138, 0, 0, 0, 7, 248, 255, 77, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Å' / 0xc5
        'Å' => Some(&[
            &[0, 0, 0, 16, 255, 62, 166, 164, 0, 0, 0, 0],
            &[0, 0, 0, 4, 235, 166, 221, 130, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 144, 114, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 120, 120, 40, 0, 0, 0, 0],
            &[0, 0, 0, 3, 240, 255, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 58, 255, 226, 255, 208, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 135, 235, 254, 24, 0, 0, 0],
            &[0, 0, 0, 199, 255, 73, 175, 255, 94, 0, 0, 0],
            &[0, 0, 17, 252, 252, 13, 114, 255, 165, 0, 0, 0],
            &[0, 0, 84, 255, 201, 0, 52, 255, 234, 1, 0, 0],
            &[0, 0, 154, 255, 255, 192, 196, 255, 255, 50, 0, 0],
            &[0, 0, 224, 255, 255, 255, 255, 255, 255, 121, 0, 0],
            &[0, 39, 255, 255, 56, 42, 42, 152, 255, 191, 0, 0],
            &[0, 109, 255, 202, 0, 0, 0, 61, 255, 250, 12, 0],
            &[0, 179, 255, 138, 0, 0, 0, 7, 248, 255, 77, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 143, 253, 228, 52, 0, 0, 0, 0],
        ]),
        // letter: 'Æ' / 0xc6
        'Æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 108, 108, 108, 108, 108, 108, 13, 0],
            &[0, 0, 0, 87, 255, 255, 255, 255, 255, 255, 30, 0],
            &[0, 0, 0, 147, 255, 187, 255, 242, 143, 143, 17, 0],
            &[0, 0, 0, 208, 255, 59, 255, 225, 0, 0, 0, 0],
            &[0, 0, 16, 253, 242, 16, 255, 225, 0, 0, 0, 0],
            &[0, 0, 74, 255, 188, 14, 255, 253, 237, 197, 0, 0],
            &[0, 0, 135, 255, 132, 14, 255, 255, 255, 212, 0, 0],
            &[0, 0, 196, 255, 255, 219, 255, 226, 14, 11, 0, 0],
            &[0, 8, 249, 255, 255, 255, 255, 225, 0, 0, 0, 0],
            &[0, 62, 255, 243, 47, 61, 255, 225, 0, 0, 0, 0],
            &[0, 123, 255, 162, 0, 14, 255, 255, 251, 251, 30, 0],
            &[0, 184, 255, 105, 0, 14, 255, 255, 255, 255, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ç' / 0xc7
        'Ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 88, 134, 139, 107, 28, 0, 0],
            &[0, 0, 0, 68, 232, 255, 255, 255, 255, 199, 0, 0],
            &[0, 0, 40, 246, 255, 218, 133, 126, 182, 95, 0, 0],
            &[0, 0, 166, 255, 215, 10, 0, 0, 0, 0, 0, 0],
            &[0, 3, 243, 255, 109, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 255, 255, 61, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 255, 255, 43, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 60, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 241, 255, 123, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 239, 64, 0, 0, 37, 54, 0, 0],
            &[0, 0, 32, 238, 255, 255, 237, 239, 255, 131, 0, 0],
            &[0, 0, 0, 42, 184, 255, 255, 255, 225, 84, 0, 0],
            &[0, 0, 0, 0, 0, 119, 255, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 66, 217, 232, 5, 0, 0, 0],
            &[0, 0, 0, 0, 47, 206, 196, 92, 0, 0, 0, 0],
        ]),
        // letter: 'È' / 0xc8
        'È' => Some(&[
            &[0, 0, 0, 117, 241, 225, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 129, 255, 166, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 94, 157, 32, 0, 0, 0, 0],
            &[0, 0, 78, 108, 108, 108, 108, 108, 108, 52, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 203, 147, 147, 147, 147, 70, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 247, 237, 237, 237, 237, 56, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 184, 255, 141, 18, 18, 18, 18, 4, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'É' / 0xc9
        'É' => Some(&[
            &[0, 0, 0, 0, 0, 8, 211, 241, 139, 0, 0, 0],
            &[0, 0, 0, 0, 0, 143, 255, 150, 2, 0, 0, 0],
            &[0, 0, 0, 0, 18, 156, 108, 0, 0, 0, 0, 0],
            &[0, 0, 78, 108, 108, 108, 108, 108, 108, 52, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 203, 147, 147, 147, 147, 70, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 247, 237, 237, 237, 237, 56, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 184, 255, 141, 18, 18, 18, 18, 4, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ê' / 0xca
        'Ê' => Some(&[
            &[0, 0, 0, 0, 139, 241, 241, 94, 0, 0, 0, 0],
            &[0, 0, 0, 74, 254, 168, 204, 242, 38, 0, 0, 0],
            &[0, 0, 1, 142, 119, 2, 13, 140, 112, 0, 0, 0],
            &[0, 0, 78, 108, 108, 108, 108, 108, 108, 52, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 203, 147, 147, 147, 147, 70, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 247, 237, 237, 237, 237, 56, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 184, 255, 141, 18, 18, 18, 18, 4, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ë' / 0xcb
        'Ë' => Some(&[
            &[0, 0, 2, 154, 157, 5, 31, 177, 113, 0, 0, 0],
            &[0, 0, 15, 252, 255, 32, 80, 255, 223, 0, 0, 0],
            &[0, 0, 0, 86, 91, 0, 7, 111, 58, 0, 0, 0],
            &[0, 0, 78, 108, 108, 108, 108, 108, 108, 52, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 203, 147, 147, 147, 147, 70, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 247, 237, 237, 237, 237, 56, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 184, 255, 141, 18, 18, 18, 18, 4, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ì' / 0xcc
        'Ì' => Some(&[
            &[0, 0, 0, 140, 241, 212, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 151, 255, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 109, 156, 18, 0, 0, 0, 0],
            &[0, 0, 60, 108, 108, 108, 108, 108, 108, 17, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 29, 76, 152, 255, 239, 92, 64, 7, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 90, 184, 224, 255, 251, 201, 172, 24, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Í' / 0xcd
        'Í' => Some(&[
            &[0, 0, 0, 0, 0, 18, 225, 241, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 167, 255, 127, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 157, 93, 0, 0, 0, 0, 0],
            &[0, 0, 60, 108, 108, 108, 108, 108, 108, 17, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 29, 76, 152, 255, 239, 92, 64, 7, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 90, 184, 224, 255, 251, 201, 172, 24, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Î' / 0xce
        'Î' => Some(&[
            &[0, 0, 0, 0, 162, 241, 241, 71, 0, 0, 0, 0],
            &[0, 0, 0, 97, 255, 154, 218, 231, 25, 0, 0, 0],
            &[0, 0, 9, 150, 106, 0, 21, 147, 97, 0, 0, 0],
            &[0, 0, 60, 108, 108, 108, 108, 108, 108, 17, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 29, 76, 152, 255, 239, 92, 64, 7, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 90, 184, 224, 255, 251, 201, 172, 24, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ï' / 0xcf
        'Ï' => Some(&[
            &[0, 0, 10, 166, 143, 0, 45, 180, 96, 0, 0, 0],
            &[0, 0, 36, 255, 254, 9, 105, 255, 198, 0, 0, 0],
            &[0, 0, 0, 99, 78, 0, 14, 115, 47, 0, 0, 0],
            &[0, 0, 60, 108, 108, 108, 108, 108, 108, 17, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 29, 76, 152, 255, 239, 92, 64, 7, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 90, 184, 224, 255, 251, 201, 172, 24, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ð' / 0xd0
        'Ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 108, 108, 108, 100, 67, 4, 0, 0, 0, 0],
            &[0, 71, 255, 255, 255, 255, 255, 230, 67, 0, 0, 0],
            &[0, 71, 255, 255, 140, 150, 223, 255, 248, 41, 0, 0],
            &[0, 71, 255, 255, 2, 0, 13, 216, 255, 166, 0, 0],
            &[0, 71, 255, 255, 2, 0, 0, 112, 255, 241, 0, 0],
            &[0, 227, 255, 255, 157, 109, 0, 70, 255, 255, 15, 0],
            &[0, 255, 255, 255, 255, 180, 0, 67, 255, 255, 22, 0],
            &[0, 151, 255, 255, 81, 56, 0, 93, 255, 253, 3, 0],
            &[0, 71, 255, 255, 2, 0, 0, 169, 255, 199, 0, 0],
            &[0, 71, 255, 255, 2, 18, 115, 252, 255, 100, 0, 0],
            &[0, 71, 255, 255, 247, 255, 255, 255, 175, 1, 0, 0],
            &[0, 71, 255, 255, 255, 243, 205, 108, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ñ' / 0xd1
        'Ñ' => Some(&[
            &[0, 0, 0, 66, 181, 137, 30, 80, 171, 0, 0, 0],
            &[0, 0, 2, 230, 229, 248, 255, 254, 157, 0, 0, 0],
            &[0, 0, 17, 161, 16, 23, 119, 133, 15, 0, 0, 0],
            &[0, 30, 108, 108, 102, 0, 0, 19, 108, 95, 0, 0],
            &[0, 71, 255, 255, 255, 44, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 248, 255, 128, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 190, 251, 211, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 185, 188, 255, 39, 47, 255, 225, 0, 0],
            &[0, 71, 255, 196, 105, 255, 123, 47, 255, 225, 0, 0],
            &[0, 71, 255, 200, 24, 253, 206, 46, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 195, 255, 76, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 112, 255, 152, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 30, 255, 225, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 0, 202, 255, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 0, 119, 255, 255, 225, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ò' / 0xd2
        'Ò' => Some(&[
            &[0, 0, 0, 140, 241, 212, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 151, 255, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 109, 156, 18, 0, 0, 0, 0],
            &[0, 0, 0, 13, 105, 146, 139, 72, 0, 0, 0, 0],
            &[0, 0, 37, 229, 255, 255, 255, 255, 166, 2, 0, 0],
            &[0, 0, 193, 255, 229, 115, 158, 255, 255, 96, 0, 0],
            &[0, 38, 255, 255, 69, 0, 0, 177, 255, 195, 0, 0],
            &[0, 95, 255, 247, 2, 0, 0, 99, 255, 247, 2, 0],
            &[0, 119, 255, 221, 0, 0, 0, 72, 255, 255, 21, 0],
            &[0, 129, 255, 216, 0, 0, 0, 63, 255, 255, 26, 0],
            &[0, 115, 255, 230, 0, 0, 0, 77, 255, 255, 13, 0],
            &[0, 76, 255, 255, 20, 0, 0, 124, 255, 230, 0, 0],
            &[0, 13, 246, 255, 141, 0, 21, 225, 255, 157, 0, 0],
            &[0, 0, 129, 255, 255, 231, 245, 255, 244, 38, 0, 0],
            &[0, 0, 2, 131, 241, 255, 255, 211, 60, 0, 0, 0],
            &[0, 0, 0, 0, 5, 32, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ó' / 0xd3
        'Ó' => Some(&[
            &[0, 0, 0, 0, 0, 18, 225, 241, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 167, 255, 127, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 157, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 105, 146, 139, 72, 0, 0, 0, 0],
            &[0, 0, 37, 229, 255, 255, 255, 255, 166, 2, 0, 0],
            &[0, 0, 193, 255, 229, 115, 158, 255, 255, 96, 0, 0],
            &[0, 38, 255, 255, 69, 0, 0, 177, 255, 195, 0, 0],
            &[0, 95, 255, 247, 2, 0, 0, 99, 255, 247, 2, 0],
            &[0, 119, 255, 221, 0, 0, 0, 72, 255, 255, 21, 0],
            &[0, 129, 255, 216, 0, 0, 0, 63, 255, 255, 26, 0],
            &[0, 115, 255, 230, 0, 0, 0, 77, 255, 255, 13, 0],
            &[0, 76, 255, 255, 20, 0, 0, 124, 255, 230, 0, 0],
            &[0, 13, 246, 255, 141, 0, 21, 225, 255, 157, 0, 0],
            &[0, 0, 129, 255, 255, 231, 245, 255, 244, 38, 0, 0],
            &[0, 0, 2, 131, 241, 255, 255, 211, 60, 0, 0, 0],
            &[0, 0, 0, 0, 5, 32, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ô' / 0xd4
        'Ô' => Some(&[
            &[0, 0, 0, 0, 162, 241, 241, 71, 0, 0, 0, 0],
            &[0, 0, 0, 97, 255, 154, 218, 231, 25, 0, 0, 0],
            &[0, 0, 9, 150, 106, 0, 21, 147, 97, 0, 0, 0],
            &[0, 0, 0, 13, 105, 146, 139, 72, 0, 0, 0, 0],
            &[0, 0, 37, 229, 255, 255, 255, 255, 166, 2, 0, 0],
            &[0, 0, 193, 255, 229, 115, 158, 255, 255, 96, 0, 0],
            &[0, 38, 255, 255, 69, 0, 0, 177, 255, 195, 0, 0],
            &[0, 95, 255, 247, 2, 0, 0, 99, 255, 247, 2, 0],
            &[0, 119, 255, 221, 0, 0, 0, 72, 255, 255, 21, 0],
            &[0, 129, 255, 216, 0, 0, 0, 63, 255, 255, 26, 0],
            &[0, 115, 255, 230, 0, 0, 0, 77, 255, 255, 13, 0],
            &[0, 76, 255, 255, 20, 0, 0, 124, 255, 230, 0, 0],
            &[0, 13, 246, 255, 141, 0, 21, 225, 255, 157, 0, 0],
            &[0, 0, 129, 255, 255, 231, 245, 255, 244, 38, 0, 0],
            &[0, 0, 2, 131, 241, 255, 255, 211, 60, 0, 0, 0],
            &[0, 0, 0, 0, 5, 32, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Õ' / 0xd5
        'Õ' => Some(&[
            &[0, 0, 0, 79, 182, 129, 24, 96, 155, 0, 0, 0],
            &[0, 0, 11, 242, 227, 251, 254, 254, 136, 0, 0, 0],
            &[0, 0, 30, 158, 6, 29, 125, 127, 9, 0, 0, 0],
            &[0, 0, 0, 13, 105, 146, 139, 72, 0, 0, 0, 0],
            &[0, 0, 37, 229, 255, 255, 255, 255, 166, 2, 0, 0],
            &[0, 0, 193, 255, 229, 115, 158, 255, 255, 96, 0, 0],
            &[0, 38, 255, 255, 69, 0, 0, 177, 255, 195, 0, 0],
            &[0, 95, 255, 247, 2, 0, 0, 99, 255, 247, 2, 0],
            &[0, 119, 255, 221, 0, 0, 0, 72, 255, 255, 21, 0],
            &[0, 129, 255, 216, 0, 0, 0, 63, 255, 255, 26, 0],
            &[0, 115, 255, 230, 0, 0, 0, 77, 255, 255, 13, 0],
            &[0, 76, 255, 255, 20, 0, 0, 124, 255, 230, 0, 0],
            &[0, 13, 246, 255, 141, 0, 21, 225, 255, 157, 0, 0],
            &[0, 0, 129, 255, 255, 231, 245, 255, 244, 38, 0, 0],
            &[0, 0, 2, 131, 241, 255, 255, 211, 60, 0, 0, 0],
            &[0, 0, 0, 0, 5, 32, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ö' / 0xd6
        'Ö' => Some(&[
            &[0, 0, 10, 166, 143, 0, 45, 180, 96, 0, 0, 0],
            &[0, 0, 36, 255, 254, 9, 105, 255, 198, 0, 0, 0],
            &[0, 0, 0, 99, 78, 0, 14, 115, 47, 0, 0, 0],
            &[0, 0, 0, 13, 105, 146, 139, 72, 0, 0, 0, 0],
            &[0, 0, 37, 229, 255, 255, 255, 255, 166, 2, 0, 0],
            &[0, 0, 193, 255, 229, 115, 158, 255, 255, 96, 0, 0],
            &[0, 38, 255, 255, 69, 0, 0, 177, 255, 195, 0, 0],
            &[0, 95, 255, 247, 2, 0, 0, 99, 255, 247, 2, 0],
            &[0, 119, 255, 221, 0, 0, 0, 72, 255, 255, 21, 0],
            &[0, 129, 255, 216, 0, 0, 0, 63, 255, 255, 26, 0],
            &[0, 115, 255, 230, 0, 0, 0, 77, 255, 255, 13, 0],
            &[0, 76, 255, 255, 20, 0, 0, 124, 255, 230, 0, 0],
            &[0, 13, 246, 255, 141, 0, 21, 225, 255, 157, 0, 0],
            &[0, 0, 129, 255, 255, 231, 245, 255, 244, 38, 0, 0],
            &[0, 0, 2, 131, 241, 255, 255, 211, 60, 0, 0, 0],
            &[0, 0, 0, 0, 5, 32, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '×' / 0xd7
        '×' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 0, 0, 0, 0, 6, 0, 0, 0],
            &[0, 0, 54, 227, 59, 0, 0, 146, 189, 10, 0, 0],
            &[0, 0, 80, 250, 243, 64, 151, 255, 207, 18, 0, 0],
            &[0, 0, 0, 81, 250, 253, 255, 207, 18, 0, 0, 0],
            &[0, 0, 0, 0, 182, 255, 255, 77, 0, 0, 0, 0],
            &[0, 0, 0, 147, 255, 217, 249, 241, 56, 0, 0, 0],
            &[0, 0, 112, 255, 206, 18, 78, 249, 240, 23, 0, 0],
            &[0, 0, 16, 158, 17, 0, 0, 78, 115, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ø' / 0xd8
        'Ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 105, 146, 139, 72, 46, 197, 32, 0],
            &[0, 0, 37, 229, 255, 255, 255, 255, 255, 226, 13, 0],
            &[0, 0, 193, 255, 229, 115, 158, 255, 255, 164, 0, 0],
            &[0, 38, 255, 255, 69, 0, 49, 255, 255, 195, 0, 0],
            &[0, 95, 255, 247, 2, 6, 209, 255, 255, 247, 2, 0],
            &[0, 119, 255, 221, 0, 132, 254, 138, 255, 255, 21, 0],
            &[0, 129, 255, 216, 52, 251, 148, 63, 255, 255, 26, 0],
            &[0, 115, 255, 238, 213, 221, 11, 77, 255, 255, 13, 0],
            &[0, 76, 255, 255, 255, 63, 0, 124, 255, 230, 0, 0],
            &[0, 13, 255, 255, 255, 0, 21, 225, 255, 157, 0, 0],
            &[0, 8, 255, 255, 255, 231, 245, 255, 244, 38, 0, 0],
            &[0, 135, 255, 193, 241, 255, 255, 211, 60, 0, 0, 0],
            &[0, 28, 101, 0, 5, 32, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ù' / 0xd9
        'Ù' => Some(&[
            &[0, 0, 0, 140, 241, 212, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 151, 255, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 109, 156, 18, 0, 0, 0, 0],
            &[0, 30, 108, 108, 4, 0, 0, 47, 108, 95, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 11, 0, 0, 114, 255, 225, 0, 0],
            &[0, 61, 255, 255, 37, 0, 0, 138, 255, 214, 0, 0],
            &[0, 24, 253, 255, 131, 0, 14, 217, 255, 172, 0, 0],
            &[0, 0, 170, 255, 255, 235, 249, 255, 254, 63, 0, 0],
            &[0, 0, 14, 158, 247, 255, 255, 220, 84, 0, 0, 0],
            &[0, 0, 0, 0, 8, 32, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ú' / 0xda
        'Ú' => Some(&[
            &[0, 0, 0, 0, 0, 18, 225, 241, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 167, 255, 127, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 157, 93, 0, 0, 0, 0, 0],
            &[0, 30, 108, 108, 4, 0, 0, 47, 108, 95, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 11, 0, 0, 114, 255, 225, 0, 0],
            &[0, 61, 255, 255, 37, 0, 0, 138, 255, 214, 0, 0],
            &[0, 24, 253, 255, 131, 0, 14, 217, 255, 172, 0, 0],
            &[0, 0, 170, 255, 255, 235, 249, 255, 254, 63, 0, 0],
            &[0, 0, 14, 158, 247, 255, 255, 220, 84, 0, 0, 0],
            &[0, 0, 0, 0, 8, 32, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Û' / 0xdb
        'Û' => Some(&[
            &[0, 0, 0, 0, 162, 241, 241, 71, 0, 0, 0, 0],
            &[0, 0, 0, 97, 255, 154, 218, 231, 25, 0, 0, 0],
            &[0, 0, 9, 150, 106, 0, 21, 147, 97, 0, 0, 0],
            &[0, 30, 108, 108, 4, 0, 0, 47, 108, 95, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 11, 0, 0, 114, 255, 225, 0, 0],
            &[0, 61, 255, 255, 37, 0, 0, 138, 255, 214, 0, 0],
            &[0, 24, 253, 255, 131, 0, 14, 217, 255, 172, 0, 0],
            &[0, 0, 170, 255, 255, 235, 249, 255, 254, 63, 0, 0],
            &[0, 0, 14, 158, 247, 255, 255, 220, 84, 0, 0, 0],
            &[0, 0, 0, 0, 8, 32, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ü' / 0xdc
        'Ü' => Some(&[
            &[0, 0, 10, 166, 143, 0, 45, 180, 96, 0, 0, 0],
            &[0, 0, 36, 255, 254, 9, 105, 255, 198, 0, 0, 0],
            &[0, 0, 0, 99, 78, 0, 14, 115, 47, 0, 0, 0],
            &[0, 30, 108, 108, 4, 0, 0, 47, 108, 95, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 11, 0, 0, 114, 255, 225, 0, 0],
            &[0, 61, 255, 255, 37, 0, 0, 138, 255, 214, 0, 0],
            &[0, 24, 253, 255, 131, 0, 14, 217, 255, 172, 0, 0],
            &[0, 0, 170, 255, 255, 235, 249, 255, 254, 63, 0, 0],
            &[0, 0, 14, 158, 247, 255, 255, 220, 84, 0, 0, 0],
            &[0, 0, 0, 0, 8, 32, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ý' / 0xdd
        'Ý' => Some(&[
            &[0, 0, 0, 0, 0, 18, 225, 241, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 167, 255, 127, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 157, 93, 0, 0, 0, 0, 0],
            &[0, 62, 108, 93, 0, 0, 0, 26, 108, 108, 19, 0],
            &[0, 61, 255, 255, 41, 0, 0, 138, 255, 213, 1, 0],
            &[0, 0, 195, 255, 148, 0, 8, 236, 255, 94, 0, 0],
            &[0, 0, 73, 255, 242, 13, 96, 255, 224, 4, 0, 0],
            &[0, 0, 0, 207, 255, 108, 203, 255, 107, 0, 0, 0],
            &[0, 0, 0, 86, 255, 236, 255, 233, 9, 0, 0, 0],
            &[0, 0, 0, 2, 217, 255, 255, 121, 0, 0, 0, 0],
            &[0, 0, 0, 0, 109, 255, 252, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Þ' / 0xde
        'Þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 78, 108, 56, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 194, 128, 122, 84, 9, 0, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 226, 35, 0, 0],
            &[0, 0, 184, 255, 190, 118, 140, 241, 255, 173, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 119, 255, 233, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 109, 255, 234, 0, 0],
            &[0, 0, 184, 255, 146, 30, 70, 222, 255, 186, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 250, 61, 0, 0],
            &[0, 0, 184, 255, 238, 218, 205, 162, 51, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ß' / 0xdf
        'ß' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 55, 40, 1, 0, 0, 0, 0],
            &[0, 0, 27, 190, 255, 255, 255, 226, 73, 0, 0, 0],
            &[0, 0, 188, 255, 243, 178, 223, 255, 243, 15, 0, 0],
            &[0, 12, 255, 255, 85, 0, 26, 255, 255, 45, 0, 0],
            &[0, 30, 255, 255, 36, 0, 105, 255, 226, 6, 0, 0],
            &[0, 30, 255, 255, 34, 86, 253, 237, 50, 0, 0, 0],
            &[0, 30, 255, 255, 34, 207, 255, 113, 0, 0, 0, 0],
            &[0, 30, 255, 255, 34, 184, 255, 234, 63, 0, 0, 0],
            &[0, 30, 255, 255, 34, 35, 220, 255, 251, 85, 0, 0],
            &[0, 30, 255, 255, 34, 0, 16, 181, 255, 239, 2, 0],
            &[0, 30, 255, 255, 42, 13, 0, 61, 255, 255, 26, 0],
            &[0, 30, 255, 255, 63, 239, 182, 224, 255, 235, 4, 0],
            &[0, 30, 255, 255, 54, 229, 255, 255, 228, 70, 0, 0],
            &[0, 0, 0, 0, 0, 2, 28, 29, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'à' / 0xe0
        'à' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 49, 65, 54, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 242, 254, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 235, 218, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 77, 25, 0, 0, 0, 0],
            &[0, 0, 41, 141, 196, 221, 205, 155, 39, 0, 0, 0],
            &[0, 0, 97, 255, 255, 251, 255, 255, 244, 35, 0, 0],
            &[0, 0, 5, 90, 24, 0, 28, 221, 255, 122, 0, 0],
            &[0, 0, 0, 18, 75, 99, 107, 205, 255, 142, 0, 0],
            &[0, 0, 104, 247, 255, 255, 248, 251, 255, 143, 0, 0],
            &[0, 15, 248, 255, 125, 15, 0, 169, 255, 143, 0, 0],
            &[0, 42, 255, 255, 35, 0, 21, 230, 255, 143, 0, 0],
            &[0, 10, 244, 255, 231, 184, 236, 252, 255, 143, 0, 0],
            &[0, 0, 90, 242, 255, 250, 152, 100, 255, 143, 0, 0],
            &[0, 0, 0, 4, 31, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'á' / 0xe1
        'á' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 43, 65, 59, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 242, 252, 91, 0, 0, 0],
            &[0, 0, 0, 0, 2, 189, 246, 81, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 77, 41, 0, 0, 0, 0, 0],
            &[0, 0, 41, 141, 196, 221, 205, 155, 39, 0, 0, 0],
            &[0, 0, 97, 255, 255, 251, 255, 255, 244, 35, 0, 0],
            &[0, 0, 5, 90, 24, 0, 28, 221, 255, 122, 0, 0],
            &[0, 0, 0, 18, 75, 99, 107, 205, 255, 142, 0, 0],
            &[0, 0, 104, 247, 255, 255, 248, 251, 255, 143, 0, 0],
            &[0, 15, 248, 255, 125, 15, 0, 169, 255, 143, 0, 0],
            &[0, 42, 255, 255, 35, 0, 21, 230, 255, 143, 0, 0],
            &[0, 10, 244, 255, 231, 184, 236, 252, 255, 143, 0, 0],
            &[0, 0, 90, 242, 255, 250, 152, 100, 255, 143, 0, 0],
            &[0, 0, 0, 4, 31, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'â' / 0xe2
        'â' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 65, 65, 12, 0, 0, 0, 0],
            &[0, 0, 0, 1, 187, 255, 255, 158, 0, 0, 0, 0],
            &[0, 0, 0, 128, 251, 98, 130, 255, 96, 0, 0, 0],
            &[0, 0, 0, 76, 47, 0, 0, 57, 65, 0, 0, 0],
            &[0, 0, 41, 141, 196, 221, 205, 155, 39, 0, 0, 0],
            &[0, 0, 97, 255, 255, 251, 255, 255, 244, 35, 0, 0],
            &[0, 0, 5, 90, 24, 0, 28, 221, 255, 122, 0, 0],
            &[0, 0, 0, 18, 75, 99, 107, 205, 255, 142, 0, 0],
            &[0, 0, 104, 247, 255, 255, 248, 251, 255, 143, 0, 0],
            &[0, 15, 248, 255, 125, 15, 0, 169, 255, 143, 0, 0],
            &[0, 42, 255, 255, 35, 0, 21, 230, 255, 143, 0, 0],
            &[0, 10, 244, 255, 231, 184, 236, 252, 255, 143, 0, 0],
            &[0, 0, 90, 242, 255, 250, 152, 100, 255, 143, 0, 0],
            &[0, 0, 0, 4, 31, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ã' / 0xe3
        'ã' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 0, 0, 4, 23, 0, 0, 0],
            &[0, 0, 0, 116, 252, 221, 114, 144, 222, 0, 0, 0],
            &[0, 0, 2, 242, 167, 187, 255, 255, 123, 0, 0, 0],
            &[0, 0, 7, 81, 7, 0, 36, 56, 0, 0, 0, 0],
            &[0, 0, 41, 141, 196, 221, 205, 155, 39, 0, 0, 0],
            &[0, 0, 97, 255, 255, 251, 255, 255, 244, 35, 0, 0],
            &[0, 0, 5, 90, 24, 0, 28, 221, 255, 122, 0, 0],
            &[0, 0, 0, 18, 75, 99, 107, 205, 255, 142, 0, 0],
            &[0, 0, 104, 247, 255, 255, 248, 251, 255, 143, 0, 0],
            &[0, 15, 248, 255, 125, 15, 0, 169, 255, 143, 0, 0],
            &[0, 42, 255, 255, 35, 0, 21, 230, 255, 143, 0, 0],
            &[0, 10, 244, 255, 231, 184, 236, 252, 255, 143, 0, 0],
            &[0, 0, 90, 242, 255, 250, 152, 100, 255, 143, 0, 0],
            &[0, 0, 0, 4, 31, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ä' / 0xe4
        'ä' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 5, 0, 0, 8, 0, 0, 0, 0],
            &[0, 0, 6, 226, 235, 21, 53, 246, 191, 0, 0, 0],
            &[0, 0, 4, 233, 249, 27, 51, 255, 211, 0, 0, 0],
            &[0, 0, 0, 17, 22, 0, 0, 31, 9, 0, 0, 0],
            &[0, 0, 41, 141, 196, 221, 205, 155, 39, 0, 0, 0],
            &[0, 0, 97, 255, 255, 251, 255, 255, 244, 35, 0, 0],
            &[0, 0, 5, 90, 24, 0, 28, 221, 255, 122, 0, 0],
            &[0, 0, 0, 18, 75, 99, 107, 205, 255, 142, 0, 0],
            &[0, 0, 104, 247, 255, 255, 248, 251, 255, 143, 0, 0],
            &[0, 15, 248, 255, 125, 15, 0, 169, 255, 143, 0, 0],
            &[0, 42, 255, 255, 35, 0, 21, 230, 255, 143, 0, 0],
            &[0, 10, 244, 255, 231, 184, 236, 252, 255, 143, 0, 0],
            &[0, 0, 90, 242, 255, 250, 152, 100, 255, 143, 0, 0],
            &[0, 0, 0, 4, 31, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'å' / 0xe5
        'å' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 78, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 178, 226, 238, 131, 0, 0, 0, 0],
            &[0, 0, 0, 0, 242, 65, 115, 201, 0, 0, 0, 0],
            &[0, 0, 0, 0, 163, 237, 246, 117, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 59, 45, 0, 0, 0, 0, 0],
            &[0, 0, 41, 141, 196, 221, 205, 155, 39, 0, 0, 0],
            &[0, 0, 97, 255, 255, 251, 255, 255, 244, 35, 0, 0],
            &[0, 0, 5, 90, 24, 0, 28, 221, 255, 122, 0, 0],
            &[0, 0, 0, 18, 75, 99, 107, 205, 255, 142, 0, 0],
            &[0, 0, 104, 247, 255, 255, 248, 251, 255, 143, 0, 0],
            &[0, 15, 248, 255, 125, 15, 0, 169, 255, 143, 0, 0],
            &[0, 42, 255, 255, 35, 0, 21, 230, 255, 143, 0, 0],
            &[0, 10, 244, 255, 231, 184, 236, 252, 255, 143, 0, 0],
            &[0, 0, 90, 242, 255, 250, 152, 100, 255, 143, 0, 0],
            &[0, 0, 0, 4, 31, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'æ' / 0xe6
        'æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 31, 160, 214, 205, 110, 121, 211, 196, 53, 0, 0],
            &[0, 23, 247, 253, 255, 255, 255, 254, 255, 229, 6, 0],
            &[0, 0, 36, 3, 102, 255, 253, 27, 186, 255, 67, 0],
            &[0, 0, 14, 68, 111, 255, 234, 75, 171, 255, 102, 0],
            &[0, 58, 240, 255, 253, 255, 255, 255, 255, 255, 112, 0],
            &[0, 180, 255, 127, 43, 255, 225, 45, 45, 45, 19, 0],
            &[0, 208, 255, 68, 72, 255, 253, 40, 0, 3, 2, 0],
            &[0, 176, 255, 213, 224, 243, 255, 235, 169, 215, 18, 0],
            &[0, 54, 237, 255, 220, 39, 154, 252, 255, 218, 10, 0],
            &[0, 0, 5, 31, 3, 0, 0, 15, 29, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ç' / 0xe7
        'ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 133, 202, 223, 207, 152, 26, 0, 0],
            &[0, 0, 13, 217, 255, 255, 255, 255, 245, 12, 0, 0],
            &[0, 0, 123, 255, 250, 96, 25, 46, 76, 0, 0, 0],
            &[0, 0, 193, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 218, 255, 126, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 255, 145, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 161, 255, 233, 34, 0, 1, 73, 6, 0, 0],
            &[0, 0, 53, 252, 255, 254, 221, 237, 255, 12, 0, 0],
            &[0, 0, 0, 75, 213, 255, 255, 254, 192, 6, 0, 0],
            &[0, 0, 0, 0, 0, 154, 252, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 77, 230, 212, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 208, 192, 76, 0, 0, 0, 0],
        ]),
        // letter: 'è' / 0xe8
        'è' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 65, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 240, 255, 65, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 232, 223, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 27, 77, 27, 0, 0, 0, 0],
            &[0, 0, 0, 51, 168, 216, 213, 155, 29, 0, 0, 0],
            &[0, 0, 62, 248, 255, 244, 249, 255, 231, 23, 0, 0],
            &[0, 0, 206, 255, 163, 6, 11, 189, 255, 133, 0, 0],
            &[0, 22, 255, 255, 106, 71, 71, 138, 255, 188, 0, 0],
            &[0, 44, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 26, 255, 255, 100, 77, 77, 77, 77, 62, 0, 0],
            &[0, 0, 216, 255, 164, 11, 0, 0, 13, 35, 0, 0],
            &[0, 0, 79, 254, 255, 242, 202, 212, 249, 139, 0, 0],
            &[0, 0, 0, 72, 203, 255, 255, 255, 225, 87, 0, 0],
            &[0, 0, 0, 0, 0, 13, 34, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'é' / 0xe9
        'é' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 41, 65, 61, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 239, 253, 98, 0, 0, 0],
            &[0, 0, 0, 0, 0, 182, 248, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 77, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 51, 168, 216, 213, 155, 29, 0, 0, 0],
            &[0, 0, 62, 248, 255, 244, 249, 255, 231, 23, 0, 0],
            &[0, 0, 206, 255, 163, 6, 11, 189, 255, 133, 0, 0],
            &[0, 22, 255, 255, 106, 71, 71, 138, 255, 188, 0, 0],
            &[0, 44, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 26, 255, 255, 100, 77, 77, 77, 77, 62, 0, 0],
            &[0, 0, 216, 255, 164, 11, 0, 0, 13, 35, 0, 0],
            &[0, 0, 79, 254, 255, 242, 202, 212, 249, 139, 0, 0],
            &[0, 0, 0, 72, 203, 255, 255, 255, 225, 87, 0, 0],
            &[0, 0, 0, 0, 0, 13, 34, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ê' / 0xea
        'ê' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 65, 65, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 180, 255, 255, 166, 0, 0, 0, 0],
            &[0, 0, 0, 120, 253, 105, 123, 255, 104, 0, 0, 0],
            &[0, 0, 0, 74, 50, 0, 0, 55, 68, 0, 0, 0],
            &[0, 0, 0, 51, 168, 216, 213, 155, 29, 0, 0, 0],
            &[0, 0, 62, 248, 255, 244, 249, 255, 231, 23, 0, 0],
            &[0, 0, 206, 255, 163, 6, 11, 189, 255, 133, 0, 0],
            &[0, 22, 255, 255, 106, 71, 71, 138, 255, 188, 0, 0],
            &[0, 44, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 26, 255, 255, 100, 77, 77, 77, 77, 62, 0, 0],
            &[0, 0, 216, 255, 164, 11, 0, 0, 13, 35, 0, 0],
            &[0, 0, 79, 254, 255, 242, 202, 212, 249, 139, 0, 0],
            &[0, 0, 0, 72, 203, 255, 255, 255, 225, 87, 0, 0],
            &[0, 0, 0, 0, 0, 13, 34, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ë' / 0xeb
        'ë' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 6, 0, 0, 8, 0, 0, 0, 0],
            &[0, 0, 2, 221, 238, 27, 47, 245, 200, 0, 0, 0],
            &[0, 0, 2, 227, 251, 33, 43, 254, 219, 0, 0, 0],
            &[0, 0, 0, 16, 24, 0, 0, 29, 10, 0, 0, 0],
            &[0, 0, 0, 51, 168, 216, 213, 155, 29, 0, 0, 0],
            &[0, 0, 62, 248, 255, 244, 249, 255, 231, 23, 0, 0],
            &[0, 0, 206, 255, 163, 6, 11, 189, 255, 133, 0, 0],
            &[0, 22, 255, 255, 106, 71, 71, 138, 255, 188, 0, 0],
            &[0, 44, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 26, 255, 255, 100, 77, 77, 77, 77, 62, 0, 0],
            &[0, 0, 216, 255, 164, 11, 0, 0, 13, 35, 0, 0],
            &[0, 0, 79, 254, 255, 242, 202, 212, 249, 139, 0, 0],
            &[0, 0, 0, 72, 203, 255, 255, 255, 225, 87, 0, 0],
            &[0, 0, 0, 0, 0, 13, 34, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ì' / 0xec
        'ì' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 57, 65, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 84, 251, 245, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 76, 245, 194, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 77, 15, 0, 0, 0, 0],
            &[0, 0, 114, 188, 188, 188, 188, 36, 0, 0, 0, 0],
            &[0, 0, 147, 255, 255, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 9, 63, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 4, 135, 177, 222, 255, 255, 223, 173, 105, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'í' / 0xed
        'í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 52, 65, 51, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 252, 245, 65, 0, 0, 0],
            &[0, 0, 0, 0, 9, 214, 237, 58, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 77, 31, 0, 0, 0, 0, 0],
            &[0, 0, 114, 188, 188, 188, 188, 36, 0, 0, 0, 0],
            &[0, 0, 147, 255, 255, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 9, 63, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 4, 135, 177, 222, 255, 255, 223, 173, 105, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'î' / 0xee
        'î' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 65, 65, 4, 0, 0, 0, 0],
            &[0, 0, 0, 9, 212, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 160, 245, 75, 160, 252, 66, 0, 0, 0],
            &[0, 0, 8, 77, 37, 0, 0, 67, 55, 0, 0, 0],
            &[0, 0, 114, 188, 188, 188, 188, 36, 0, 0, 0, 0],
            &[0, 0, 147, 255, 255, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 9, 63, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 4, 135, 177, 222, 255, 255, 223, 173, 105, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ï' / 0xef
        'ï' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 2, 0, 0, 9, 0, 0, 0, 0],
            &[0, 0, 25, 239, 220, 3, 82, 251, 159, 0, 0, 0],
            &[0, 0, 20, 250, 237, 6, 83, 255, 178, 0, 0, 0],
            &[0, 0, 0, 25, 15, 0, 0, 35, 4, 0, 0, 0],
            &[0, 0, 114, 188, 188, 188, 188, 36, 0, 0, 0, 0],
            &[0, 0, 147, 255, 255, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 9, 63, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 4, 135, 177, 222, 255, 255, 223, 173, 105, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ð' / 0xf0
        'ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 0, 0, 3, 11, 0, 0, 0],
            &[0, 0, 0, 25, 242, 174, 84, 205, 163, 0, 0, 0],
            &[0, 0, 0, 20, 184, 255, 255, 255, 55, 0, 0, 0],
            &[0, 0, 0, 66, 234, 255, 255, 250, 54, 0, 0, 0],
            &[0, 0, 0, 28, 159, 25, 99, 255, 217, 6, 0, 0],
            &[0, 0, 0, 78, 167, 184, 141, 205, 255, 94, 0, 0],
            &[0, 0, 123, 255, 255, 251, 255, 255, 255, 177, 0, 0],
            &[0, 19, 250, 255, 133, 7, 37, 219, 255, 225, 0, 0],
            &[0, 74, 255, 255, 19, 0, 0, 116, 255, 241, 0, 0],
            &[0, 82, 255, 254, 8, 0, 0, 118, 255, 224, 0, 0],
            &[0, 40, 255, 255, 73, 0, 0, 186, 255, 164, 0, 0],
            &[0, 0, 177, 255, 238, 158, 181, 255, 247, 44, 0, 0],
            &[0, 0, 12, 152, 245, 255, 255, 208, 61, 0, 0, 0],
            &[0, 0, 0, 0, 7, 33, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ñ' / 0xf1
        'ñ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 0, 0, 6, 22, 0, 0, 0],
            &[0, 0, 0, 131, 253, 215, 108, 155, 206, 0, 0, 0],
            &[0, 0, 11, 249, 160, 193, 255, 255, 107, 0, 0, 0],
            &[0, 0, 12, 81, 2, 0, 41, 52, 0, 0, 0, 0],
            &[0, 22, 188, 144, 31, 164, 217, 200, 107, 1, 0, 0],
            &[0, 30, 255, 251, 223, 255, 255, 255, 255, 91, 0, 0],
            &[0, 30, 255, 255, 215, 39, 39, 225, 255, 175, 0, 0],
            &[0, 30, 255, 255, 99, 0, 0, 145, 255, 203, 0, 0],
            &[0, 30, 255, 255, 60, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ò' / 0xf2
        'ò' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 57, 65, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 84, 251, 245, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 76, 245, 194, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 77, 15, 0, 0, 0, 0],
            &[0, 0, 0, 61, 175, 218, 209, 145, 23, 0, 0, 0],
            &[0, 0, 84, 252, 255, 255, 255, 255, 225, 19, 0, 0],
            &[0, 4, 233, 255, 170, 14, 40, 239, 255, 141, 0, 0],
            &[0, 59, 255, 255, 38, 0, 0, 141, 255, 215, 0, 0],
            &[0, 85, 255, 255, 5, 0, 0, 109, 255, 239, 0, 0],
            &[0, 73, 255, 255, 29, 0, 0, 130, 255, 226, 0, 0],
            &[0, 20, 250, 255, 128, 0, 7, 215, 255, 167, 0, 0],
            &[0, 0, 144, 255, 254, 199, 222, 255, 249, 48, 0, 0],
            &[0, 0, 3, 134, 242, 255, 255, 213, 66, 0, 0, 0],
            &[0, 0, 0, 0, 5, 33, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ó' / 0xf3
        'ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 52, 65, 51, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 252, 245, 65, 0, 0, 0],
            &[0, 0, 0, 0, 9, 214, 237, 58, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 77, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 61, 175, 218, 209, 145, 23, 0, 0, 0],
            &[0, 0, 84, 252, 255, 255, 255, 255, 225, 19, 0, 0],
            &[0, 4, 233, 255, 170, 14, 40, 239, 255, 141, 0, 0],
            &[0, 59, 255, 255, 38, 0, 0, 141, 255, 215, 0, 0],
            &[0, 85, 255, 255, 5, 0, 0, 109, 255, 239, 0, 0],
            &[0, 73, 255, 255, 29, 0, 0, 130, 255, 226, 0, 0],
            &[0, 20, 250, 255, 128, 0, 7, 215, 255, 167, 0, 0],
            &[0, 0, 144, 255, 254, 199, 222, 255, 249, 48, 0, 0],
            &[0, 0, 3, 134, 242, 255, 255, 213, 66, 0, 0, 0],
            &[0, 0, 0, 0, 5, 33, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ô' / 0xf4
        'ô' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 65, 65, 4, 0, 0, 0, 0],
            &[0, 0, 0, 9, 212, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 160, 245, 75, 160, 252, 66, 0, 0, 0],
            &[0, 0, 8, 77, 37, 0, 0, 67, 55, 0, 0, 0],
            &[0, 0, 0, 61, 175, 218, 209, 145, 23, 0, 0, 0],
            &[0, 0, 84, 252, 255, 255, 255, 255, 225, 19, 0, 0],
            &[0, 4, 233, 255, 170, 14, 40, 239, 255, 141, 0, 0],
            &[0, 59, 255, 255, 38, 0, 0, 141, 255, 215, 0, 0],
            &[0, 85, 255, 255, 5, 0, 0, 109, 255, 239, 0, 0],
            &[0, 73, 255, 255, 29, 0, 0, 130, 255, 226, 0, 0],
            &[0, 20, 250, 255, 128, 0, 7, 215, 255, 167, 0, 0],
            &[0, 0, 144, 255, 254, 199, 222, 255, 249, 48, 0, 0],
            &[0, 0, 3, 134, 242, 255, 255, 213, 66, 0, 0, 0],
            &[0, 0, 0, 0, 5, 33, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'õ' / 0xf5
        'õ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 0, 0, 8, 20, 0, 0, 0],
            &[0, 0, 0, 147, 254, 209, 103, 167, 189, 0, 0, 0],
            &[0, 0, 24, 254, 153, 200, 255, 255, 91, 0, 0, 0],
            &[0, 0, 17, 78, 0, 0, 45, 47, 0, 0, 0, 0],
            &[0, 0, 0, 61, 175, 218, 209, 145, 23, 0, 0, 0],
            &[0, 0, 84, 252, 255, 255, 255, 255, 225, 19, 0, 0],
            &[0, 4, 233, 255, 170, 14, 40, 239, 255, 141, 0, 0],
            &[0, 59, 255, 255, 38, 0, 0, 141, 255, 215, 0, 0],
            &[0, 85, 255, 255, 5, 0, 0, 109, 255, 239, 0, 0],
            &[0, 73, 255, 255, 29, 0, 0, 130, 255, 226, 0, 0],
            &[0, 20, 250, 255, 128, 0, 7, 215, 255, 167, 0, 0],
            &[0, 0, 144, 255, 254, 199, 222, 255, 249, 48, 0, 0],
            &[0, 0, 3, 134, 242, 255, 255, 213, 66, 0, 0, 0],
            &[0, 0, 0, 0, 5, 33, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ö' / 0xf6
        'ö' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 2, 0, 0, 9, 0, 0, 0, 0],
            &[0, 0, 25, 239, 220, 3, 82, 251, 159, 0, 0, 0],
            &[0, 0, 20, 250, 237, 6, 83, 255, 178, 0, 0, 0],
            &[0, 0, 0, 25, 15, 0, 0, 35, 4, 0, 0, 0],
            &[0, 0, 0, 61, 175, 218, 209, 145, 23, 0, 0, 0],
            &[0, 0, 84, 252, 255, 255, 255, 255, 225, 19, 0, 0],
            &[0, 4, 233, 255, 170, 14, 40, 239, 255, 141, 0, 0],
            &[0, 59, 255, 255, 38, 0, 0, 141, 255, 215, 0, 0],
            &[0, 85, 255, 255, 5, 0, 0, 109, 255, 239, 0, 0],
            &[0, 73, 255, 255, 29, 0, 0, 130, 255, 226, 0, 0],
            &[0, 20, 250, 255, 128, 0, 7, 215, 255, 167, 0, 0],
            &[0, 0, 144, 255, 254, 199, 222, 255, 249, 48, 0, 0],
            &[0, 0, 3, 134, 242, 255, 255, 213, 66, 0, 0, 0],
            &[0, 0, 0, 0, 5, 33, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '÷' / 0xf7
        '÷' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 153, 91, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 205, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 137, 79, 0, 0, 0, 0, 0],
            &[0, 0, 111, 126, 126, 126, 126, 126, 126, 60, 0, 0],
            &[0, 0, 225, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 45, 51, 51, 51, 51, 51, 51, 24, 0, 0],
            &[0, 0, 0, 0, 15, 214, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 255, 199, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ø' / 0xf8
        'ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 12, 0, 0, 0],
            &[0, 0, 0, 61, 175, 218, 209, 201, 255, 49, 0, 0],
            &[0, 0, 84, 252, 255, 255, 255, 255, 255, 26, 0, 0],
            &[0, 4, 233, 255, 170, 14, 183, 255, 255, 141, 0, 0],
            &[0, 59, 255, 255, 38, 62, 253, 255, 255, 215, 0, 0],
            &[0, 85, 255, 255, 17, 221, 203, 113, 255, 239, 0, 0],
            &[0, 73, 255, 255, 180, 247, 41, 130, 255, 226, 0, 0],
            &[0, 20, 250, 255, 255, 115, 7, 215, 255, 167, 0, 0],
            &[0, 0, 158, 255, 255, 201, 222, 255, 249, 48, 0, 0],
            &[0, 0, 150, 255, 255, 255, 255, 213, 66, 0, 0, 0],
            &[0, 0, 23, 72, 5, 33, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ù' / 0xf9
        'ù' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 57, 65, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 84, 251, 245, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 76, 245, 194, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 77, 15, 0, 0, 0, 0],
            &[0, 37, 188, 188, 22, 0, 0, 113, 188, 135, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 158, 255, 184, 0, 0],
            &[0, 51, 255, 255, 35, 0, 0, 182, 255, 184, 0, 0],
            &[0, 31, 255, 255, 91, 0, 26, 241, 255, 184, 0, 0],
            &[0, 2, 223, 255, 249, 204, 241, 249, 255, 184, 0, 0],
            &[0, 0, 60, 226, 255, 254, 176, 68, 255, 184, 0, 0],
            &[0, 0, 0, 1, 28, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ú' / 0xfa
        'ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 52, 65, 51, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 252, 245, 65, 0, 0, 0],
            &[0, 0, 0, 0, 9, 214, 237, 58, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 77, 31, 0, 0, 0, 0, 0],
            &[0, 37, 188, 188, 22, 0, 0, 113, 188, 135, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 158, 255, 184, 0, 0],
            &[0, 51, 255, 255, 35, 0, 0, 182, 255, 184, 0, 0],
            &[0, 31, 255, 255, 91, 0, 26, 241, 255, 184, 0, 0],
            &[0, 2, 223, 255, 249, 204, 241, 249, 255, 184, 0, 0],
            &[0, 0, 60, 226, 255, 254, 176, 68, 255, 184, 0, 0],
            &[0, 0, 0, 1, 28, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'û' / 0xfb
        'û' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 65, 65, 4, 0, 0, 0, 0],
            &[0, 0, 0, 9, 212, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 160, 245, 75, 160, 252, 66, 0, 0, 0],
            &[0, 0, 8, 77, 37, 0, 0, 67, 55, 0, 0, 0],
            &[0, 37, 188, 188, 22, 0, 0, 113, 188, 135, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 158, 255, 184, 0, 0],
            &[0, 51, 255, 255, 35, 0, 0, 182, 255, 184, 0, 0],
            &[0, 31, 255, 255, 91, 0, 26, 241, 255, 184, 0, 0],
            &[0, 2, 223, 255, 249, 204, 241, 249, 255, 184, 0, 0],
            &[0, 0, 60, 226, 255, 254, 176, 68, 255, 184, 0, 0],
            &[0, 0, 0, 1, 28, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ü' / 0xfc
        'ü' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 2, 0, 0, 9, 0, 0, 0, 0],
            &[0, 0, 25, 239, 220, 3, 82, 251, 159, 0, 0, 0],
            &[0, 0, 20, 250, 237, 6, 83, 255, 178, 0, 0, 0],
            &[0, 0, 0, 25, 15, 0, 0, 35, 4, 0, 0, 0],
            &[0, 37, 188, 188, 22, 0, 0, 113, 188, 135, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 158, 255, 184, 0, 0],
            &[0, 51, 255, 255, 35, 0, 0, 182, 255, 184, 0, 0],
            &[0, 31, 255, 255, 91, 0, 26, 241, 255, 184, 0, 0],
            &[0, 2, 223, 255, 249, 204, 241, 249, 255, 184, 0, 0],
            &[0, 0, 60, 226, 255, 254, 176, 68, 255, 184, 0, 0],
            &[0, 0, 0, 1, 28, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ý' / 0xfd
        'ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 52, 65, 51, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 252, 245, 65, 0, 0, 0],
            &[0, 0, 0, 0, 9, 214, 237, 58, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 77, 31, 0, 0, 0, 0, 0],
            &[0, 100, 188, 179, 3, 0, 0, 61, 188, 188, 27, 0],
            &[0, 48, 255, 255, 66, 0, 0, 148, 255, 212, 0, 0],
            &[0, 0, 202, 255, 152, 0, 0, 224, 255, 119, 0, 0],
            &[0, 0, 101, 255, 234, 3, 44, 255, 253, 28, 0, 0],
            &[0, 0, 12, 243, 255, 67, 120, 255, 189, 0, 0, 0],
            &[0, 0, 0, 154, 255, 150, 195, 255, 96, 0, 0, 0],
            &[0, 0, 0, 52, 255, 219, 250, 245, 13, 0, 0, 0],
            &[0, 0, 0, 0, 207, 255, 255, 166, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 255, 255, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 255, 230, 3, 0, 0, 0, 0],
            &[0, 2, 32, 76, 237, 255, 122, 0, 0, 0, 0, 0],
            &[0, 11, 222, 224, 160, 23, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'þ' / 0xfe
        'þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 40, 40, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 61, 161, 218, 199, 78, 0, 0, 0],
            &[0, 30, 255, 255, 229, 255, 255, 255, 252, 55, 0, 0],
            &[0, 30, 255, 255, 196, 30, 50, 233, 255, 167, 0, 0],
            &[0, 30, 255, 255, 70, 0, 0, 145, 255, 214, 0, 0],
            &[0, 30, 255, 255, 42, 0, 0, 118, 255, 238, 0, 0],
            &[0, 30, 255, 255, 63, 0, 0, 133, 255, 228, 0, 0],
            &[0, 30, 255, 255, 161, 0, 7, 209, 255, 180, 0, 0],
            &[0, 30, 255, 255, 255, 219, 229, 255, 255, 80, 0, 0],
            &[0, 30, 255, 255, 113, 233, 255, 248, 125, 0, 0, 0],
            &[0, 30, 255, 255, 38, 4, 31, 10, 0, 0, 0, 0],
            &[0, 30, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 214, 214, 32, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ÿ' / 0xff
        'ÿ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 2, 0, 0, 9, 0, 0, 0, 0],
            &[0, 0, 25, 239, 220, 3, 82, 251, 159, 0, 0, 0],
            &[0, 0, 20, 250, 237, 6, 83, 255, 178, 0, 0, 0],
            &[0, 0, 0, 25, 15, 0, 0, 35, 4, 0, 0, 0],
            &[0, 100, 188, 179, 3, 0, 0, 61, 188, 188, 27, 0],
            &[0, 48, 255, 255, 66, 0, 0, 148, 255, 212, 0, 0],
            &[0, 0, 202, 255, 152, 0, 0, 224, 255, 119, 0, 0],
            &[0, 0, 101, 255, 234, 3, 44, 255, 253, 28, 0, 0],
            &[0, 0, 12, 243, 255, 67, 120, 255, 189, 0, 0, 0],
            &[0, 0, 0, 154, 255, 150, 195, 255, 96, 0, 0, 0],
            &[0, 0, 0, 52, 255, 219, 250, 245, 13, 0, 0, 0],
            &[0, 0, 0, 0, 207, 255, 255, 166, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 255, 255, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 255, 230, 3, 0, 0, 0, 0],
            &[0, 2, 32, 76, 237, 255, 122, 0, 0, 0, 0, 0],
            &[0, 11, 222, 224, 160, 23, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ā' / 0x100
        'Ā' => Some(&[
            &[0, 0, 0, 36, 57, 57, 57, 57, 12, 0, 0, 0],
            &[0, 0, 0, 161, 255, 255, 255, 255, 55, 0, 0, 0],
            &[0, 0, 0, 58, 92, 92, 92, 92, 19, 0, 0, 0],
            &[0, 0, 0, 0, 90, 120, 120, 40, 0, 0, 0, 0],
            &[0, 0, 0, 3, 240, 255, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 58, 255, 226, 255, 208, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 135, 235, 254, 24, 0, 0, 0],
            &[0, 0, 0, 199, 255, 73, 175, 255, 94, 0, 0, 0],
            &[0, 0, 17, 252, 252, 13, 114, 255, 165, 0, 0, 0],
            &[0, 0, 84, 255, 201, 0, 52, 255, 234, 1, 0, 0],
            &[0, 0, 154, 255, 255, 192, 196, 255, 255, 50, 0, 0],
            &[0, 0, 224, 255, 255, 255, 255, 255, 255, 121, 0, 0],
            &[0, 39, 255, 255, 56, 42, 42, 152, 255, 191, 0, 0],
            &[0, 109, 255, 202, 0, 0, 0, 61, 255, 250, 12, 0],
            &[0, 179, 255, 138, 0, 0, 0, 7, 248, 255, 77, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ā' / 0x101
        'ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 137, 137, 137, 137, 47, 0, 0, 0],
            &[0, 0, 0, 128, 255, 255, 255, 255, 88, 0, 0, 0],
            &[0, 0, 0, 6, 12, 12, 12, 12, 4, 0, 0, 0],
            &[0, 0, 41, 141, 196, 221, 205, 155, 39, 0, 0, 0],
            &[0, 0, 97, 255, 255, 251, 255, 255, 244, 35, 0, 0],
            &[0, 0, 5, 90, 24, 0, 28, 221, 255, 122, 0, 0],
            &[0, 0, 0, 18, 75, 99, 107, 205, 255, 142, 0, 0],
            &[0, 0, 104, 247, 255, 255, 248, 251, 255, 143, 0, 0],
            &[0, 15, 248, 255, 125, 15, 0, 169, 255, 143, 0, 0],
            &[0, 42, 255, 255, 35, 0, 21, 230, 255, 143, 0, 0],
            &[0, 10, 244, 255, 231, 184, 236, 252, 255, 143, 0, 0],
            &[0, 0, 90, 242, 255, 250, 152, 100, 255, 143, 0, 0],
            &[0, 0, 0, 4, 31, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ă' / 0x102
        'Ă' => Some(&[
            &[0, 0, 0, 201, 169, 2, 30, 241, 98, 0, 0, 0],
            &[0, 0, 0, 105, 255, 244, 254, 230, 19, 0, 0, 0],
            &[0, 0, 0, 0, 75, 144, 125, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 120, 120, 40, 0, 0, 0, 0],
            &[0, 0, 0, 3, 240, 255, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 58, 255, 226, 255, 208, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 135, 235, 254, 24, 0, 0, 0],
            &[0, 0, 0, 199, 255, 73, 175, 255, 94, 0, 0, 0],
            &[0, 0, 17, 252, 252, 13, 114, 255, 165, 0, 0, 0],
            &[0, 0, 84, 255, 201, 0, 52, 255, 234, 1, 0, 0],
            &[0, 0, 154, 255, 255, 192, 196, 255, 255, 50, 0, 0],
            &[0, 0, 224, 255, 255, 255, 255, 255, 255, 121, 0, 0],
            &[0, 39, 255, 255, 56, 42, 42, 152, 255, 191, 0, 0],
            &[0, 109, 255, 202, 0, 0, 0, 61, 255, 250, 12, 0],
            &[0, 179, 255, 138, 0, 0, 0, 7, 248, 255, 77, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ă' / 0x103
        'ă' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 48, 0, 0, 58, 45, 0, 0, 0],
            &[0, 0, 0, 153, 228, 78, 94, 242, 112, 0, 0, 0],
            &[0, 0, 0, 32, 225, 255, 255, 198, 10, 0, 0, 0],
            &[0, 0, 0, 0, 9, 60, 52, 2, 0, 0, 0, 0],
            &[0, 0, 41, 141, 196, 221, 205, 155, 39, 0, 0, 0],
            &[0, 0, 97, 255, 255, 251, 255, 255, 244, 35, 0, 0],
            &[0, 0, 5, 90, 24, 0, 28, 221, 255, 122, 0, 0],
            &[0, 0, 0, 18, 75, 99, 107, 205, 255, 142, 0, 0],
            &[0, 0, 104, 247, 255, 255, 248, 251, 255, 143, 0, 0],
            &[0, 15, 248, 255, 125, 15, 0, 169, 255, 143, 0, 0],
            &[0, 42, 255, 255, 35, 0, 21, 230, 255, 143, 0, 0],
            &[0, 10, 244, 255, 231, 184, 236, 252, 255, 143, 0, 0],
            &[0, 0, 90, 242, 255, 250, 152, 100, 255, 143, 0, 0],
            &[0, 0, 0, 4, 31, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ą' / 0x104
        'Ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 120, 120, 40, 0, 0, 0, 0],
            &[0, 0, 0, 3, 240, 255, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 58, 255, 226, 255, 208, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 135, 235, 254, 24, 0, 0, 0],
            &[0, 0, 0, 199, 255, 73, 175, 255, 94, 0, 0, 0],
            &[0, 0, 17, 252, 252, 13, 114, 255, 165, 0, 0, 0],
            &[0, 0, 84, 255, 201, 0, 52, 255, 234, 1, 0, 0],
            &[0, 0, 154, 255, 255, 192, 196, 255, 255, 50, 0, 0],
            &[0, 0, 224, 255, 255, 255, 255, 255, 255, 121, 0, 0],
            &[0, 39, 255, 255, 56, 42, 42, 152, 255, 191, 0, 0],
            &[0, 109, 255, 202, 0, 0, 0, 61, 255, 250, 12, 0],
            &[0, 179, 255, 138, 0, 0, 0, 7, 248, 255, 82, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 121, 230, 27, 0],
            &[0, 0, 0, 0, 0, 0, 0, 22, 252, 129, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 107, 193, 141, 0],
        ]),
        // letter: 'ą' / 0x105
        'ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 41, 141, 196, 221, 205, 155, 39, 0, 0, 0],
            &[0, 0, 97, 255, 255, 251, 255, 255, 244, 35, 0, 0],
            &[0, 0, 5, 90, 24, 0, 28, 221, 255, 122, 0, 0],
            &[0, 0, 0, 18, 75, 99, 107, 205, 255, 142, 0, 0],
            &[0, 0, 104, 247, 255, 255, 248, 251, 255, 143, 0, 0],
            &[0, 15, 248, 255, 125, 15, 0, 169, 255, 143, 0, 0],
            &[0, 42, 255, 255, 35, 0, 21, 230, 255, 143, 0, 0],
            &[0, 10, 244, 255, 231, 184, 236, 252, 255, 143, 0, 0],
            &[0, 0, 90, 242, 255, 250, 152, 102, 255, 145, 0, 0],
            &[0, 0, 0, 4, 31, 11, 0, 167, 202, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 65, 255, 82, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 138, 195, 106, 0, 0],
        ]),
        // letter: 'Ć' / 0x106
        'Ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 39, 237, 240, 85, 0, 0],
            &[0, 0, 0, 0, 0, 3, 198, 252, 96, 0, 0, 0],
            &[0, 0, 0, 0, 0, 53, 157, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 88, 134, 139, 107, 28, 0, 0],
            &[0, 0, 0, 68, 232, 255, 255, 255, 255, 199, 0, 0],
            &[0, 0, 40, 246, 255, 218, 133, 126, 182, 95, 0, 0],
            &[0, 0, 166, 255, 215, 10, 0, 0, 0, 0, 0, 0],
            &[0, 3, 243, 255, 109, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 255, 255, 61, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 255, 255, 43, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 60, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 241, 255, 123, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 239, 64, 0, 0, 37, 54, 0, 0],
            &[0, 0, 32, 238, 255, 255, 237, 239, 255, 131, 0, 0],
            &[0, 0, 0, 42, 184, 254, 255, 255, 225, 84, 0, 0],
            &[0, 0, 0, 0, 0, 8, 33, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ć' / 0x107
        'ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 65, 65, 12, 0, 0],
            &[0, 0, 0, 0, 0, 2, 199, 255, 155, 1, 0, 0],
            &[0, 0, 0, 0, 0, 121, 255, 141, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 69, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 133, 202, 223, 207, 152, 26, 0, 0],
            &[0, 0, 13, 217, 255, 255, 255, 255, 245, 12, 0, 0],
            &[0, 0, 123, 255, 250, 96, 25, 46, 76, 0, 0, 0],
            &[0, 0, 193, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 218, 255, 126, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 255, 145, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 161, 255, 233, 34, 0, 1, 73, 6, 0, 0],
            &[0, 0, 53, 252, 255, 254, 221, 237, 255, 12, 0, 0],
            &[0, 0, 0, 75, 213, 255, 255, 254, 192, 6, 0, 0],
            &[0, 0, 0, 0, 0, 18, 34, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĉ' / 0x108
        'Ĉ' => Some(&[
            &[0, 0, 0, 0, 2, 192, 241, 237, 43, 0, 0, 0],
            &[0, 0, 0, 0, 131, 255, 138, 235, 211, 10, 0, 0],
            &[0, 0, 0, 24, 156, 85, 0, 36, 154, 76, 0, 0],
            &[0, 0, 0, 0, 12, 88, 134, 139, 107, 28, 0, 0],
            &[0, 0, 0, 68, 232, 255, 255, 255, 255, 199, 0, 0],
            &[0, 0, 40, 246, 255, 218, 133, 126, 182, 95, 0, 0],
            &[0, 0, 166, 255, 215, 10, 0, 0, 0, 0, 0, 0],
            &[0, 3, 243, 255, 109, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 255, 255, 61, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 255, 255, 43, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 60, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 241, 255, 123, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 239, 64, 0, 0, 37, 54, 0, 0],
            &[0, 0, 32, 238, 255, 255, 237, 239, 255, 131, 0, 0],
            &[0, 0, 0, 42, 184, 254, 255, 255, 225, 84, 0, 0],
            &[0, 0, 0, 0, 0, 8, 33, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĉ' / 0x109
        'ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 65, 65, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 119, 255, 255, 217, 11, 0, 0, 0],
            &[0, 0, 0, 63, 251, 160, 75, 244, 165, 0, 0, 0],
            &[0, 0, 0, 55, 68, 0, 0, 36, 77, 9, 0, 0],
            &[0, 0, 0, 16, 133, 202, 223, 207, 152, 26, 0, 0],
            &[0, 0, 13, 217, 255, 255, 255, 255, 245, 12, 0, 0],
            &[0, 0, 123, 255, 250, 96, 25, 46, 76, 0, 0, 0],
            &[0, 0, 193, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 218, 255, 126, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 255, 145, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 161, 255, 233, 34, 0, 1, 73, 6, 0, 0],
            &[0, 0, 53, 252, 255, 254, 221, 237, 255, 12, 0, 0],
            &[0, 0, 0, 75, 213, 255, 255, 254, 192, 6, 0, 0],
            &[0, 0, 0, 0, 0, 18, 34, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ċ' / 0x10a
        'Ċ' => Some(&[
            &[0, 0, 0, 0, 0, 131, 255, 211, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 180, 255, 251, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 43, 139, 80, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 88, 134, 139, 107, 28, 0, 0],
            &[0, 0, 0, 68, 232, 255, 255, 255, 255, 199, 0, 0],
            &[0, 0, 40, 246, 255, 218, 133, 126, 182, 95, 0, 0],
            &[0, 0, 166, 255, 215, 10, 0, 0, 0, 0, 0, 0],
            &[0, 3, 243, 255, 109, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 255, 255, 61, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 255, 255, 43, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 60, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 241, 255, 123, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 239, 64, 0, 0, 37, 54, 0, 0],
            &[0, 0, 32, 238, 255, 255, 237, 239, 255, 131, 0, 0],
            &[0, 0, 0, 42, 184, 254, 255, 255, 225, 84, 0, 0],
            &[0, 0, 0, 0, 0, 8, 33, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 16, 2, 0, 0, 0, 0],
        ]),
        // letter: 'ċ' / 0x10b
        'ċ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 92, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 254, 255, 132, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 249, 255, 115, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 133, 202, 223, 207, 152, 26, 0, 0],
            &[0, 0, 13, 217, 255, 255, 255, 255, 245, 12, 0, 0],
            &[0, 0, 123, 255, 250, 96, 25, 46, 76, 0, 0, 0],
            &[0, 0, 193, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 218, 255, 126, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 255, 145, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 161, 255, 233, 34, 0, 1, 73, 6, 0, 0],
            &[0, 0, 53, 252, 255, 254, 221, 237, 255, 12, 0, 0],
            &[0, 0, 0, 75, 213, 255, 255, 254, 192, 6, 0, 0],
            &[0, 0, 0, 0, 0, 18, 34, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Č' / 0x10c
        'Č' => Some(&[
            &[0, 0, 0, 24, 221, 168, 8, 104, 238, 85, 0, 0],
            &[0, 0, 0, 0, 69, 254, 217, 255, 154, 0, 0, 0],
            &[0, 0, 0, 0, 0, 110, 157, 152, 11, 0, 0, 0],
            &[0, 0, 0, 0, 12, 88, 134, 139, 107, 28, 0, 0],
            &[0, 0, 0, 68, 232, 255, 255, 255, 255, 199, 0, 0],
            &[0, 0, 40, 246, 255, 218, 133, 126, 182, 95, 0, 0],
            &[0, 0, 166, 255, 215, 10, 0, 0, 0, 0, 0, 0],
            &[0, 3, 243, 255, 109, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 255, 255, 61, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 255, 255, 43, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 60, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 241, 255, 123, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 239, 64, 0, 0, 37, 54, 0, 0],
            &[0, 0, 32, 238, 255, 255, 237, 239, 255, 131, 0, 0],
            &[0, 0, 0, 42, 184, 254, 255, 255, 225, 84, 0, 0],
            &[0, 0, 0, 0, 0, 8, 33, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'č' / 0x10d
        'č' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 56, 0, 0, 30, 65, 8, 0, 0],
            &[0, 0, 0, 71, 252, 148, 69, 242, 171, 1, 0, 0],
            &[0, 0, 0, 0, 129, 255, 254, 220, 13, 0, 0, 0],
            &[0, 0, 0, 0, 5, 77, 77, 36, 0, 0, 0, 0],
            &[0, 0, 0, 16, 133, 202, 223, 207, 152, 26, 0, 0],
            &[0, 0, 13, 217, 255, 255, 255, 255, 245, 12, 0, 0],
            &[0, 0, 123, 255, 250, 96, 25, 46, 76, 0, 0, 0],
            &[0, 0, 193, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 218, 255, 126, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 255, 145, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 161, 255, 233, 34, 0, 1, 73, 6, 0, 0],
            &[0, 0, 53, 252, 255, 254, 221, 237, 255, 12, 0, 0],
            &[0, 0, 0, 75, 213, 255, 255, 254, 192, 6, 0, 0],
            &[0, 0, 0, 0, 0, 18, 34, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ď' / 0x10e
        'Ď' => Some(&[
            &[0, 0, 42, 232, 145, 3, 130, 238, 59, 0, 0, 0],
            &[0, 0, 0, 98, 255, 217, 255, 124, 0, 0, 0, 0],
            &[0, 0, 0, 0, 129, 157, 141, 2, 0, 0, 0, 0],
            &[0, 30, 108, 108, 108, 100, 67, 4, 0, 0, 0, 0],
            &[0, 71, 255, 255, 255, 255, 255, 230, 67, 0, 0, 0],
            &[0, 71, 255, 255, 140, 150, 223, 255, 248, 41, 0, 0],
            &[0, 71, 255, 255, 2, 0, 13, 216, 255, 166, 0, 0],
            &[0, 71, 255, 255, 2, 0, 0, 112, 255, 241, 0, 0],
            &[0, 71, 255, 255, 2, 0, 0, 70, 255, 255, 15, 0],
            &[0, 71, 255, 255, 2, 0, 0, 67, 255, 255, 22, 0],
            &[0, 71, 255, 255, 2, 0, 0, 93, 255, 253, 3, 0],
            &[0, 71, 255, 255, 2, 0, 0, 169, 255, 199, 0, 0],
            &[0, 71, 255, 255, 2, 18, 115, 252, 255, 100, 0, 0],
            &[0, 71, 255, 255, 247, 255, 255, 255, 175, 1, 0, 0],
            &[0, 71, 255, 255, 255, 243, 205, 108, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ď' / 0x10f
        'ď' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 40, 37, 10, 40, 36, 0],
            &[0, 0, 0, 0, 0, 90, 255, 235, 91, 255, 164, 0],
            &[0, 0, 0, 0, 0, 90, 255, 235, 143, 250, 38, 0],
            &[0, 0, 0, 0, 0, 88, 255, 235, 73, 73, 0, 0],
            &[0, 27, 169, 219, 173, 98, 255, 235, 0, 0, 0, 0],
            &[0, 203, 255, 255, 255, 234, 255, 235, 0, 0, 0, 0],
            &[58, 255, 255, 108, 32, 190, 255, 235, 0, 0, 0, 0],
            &[110, 255, 238, 1, 0, 78, 255, 235, 0, 0, 0, 0],
            &[128, 255, 218, 0, 0, 50, 255, 235, 0, 0, 0, 0],
            &[111, 255, 232, 0, 0, 60, 255, 235, 0, 0, 0, 0],
            &[79, 255, 255, 52, 0, 143, 255, 235, 0, 0, 0, 0],
            &[12, 237, 255, 243, 212, 249, 255, 235, 0, 0, 0, 0],
            &[0, 74, 240, 255, 218, 57, 248, 235, 0, 0, 0, 0],
            &[0, 0, 9, 32, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Đ' / 0x110
        'Đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 108, 108, 108, 100, 67, 4, 0, 0, 0, 0],
            &[0, 71, 255, 255, 255, 255, 255, 230, 67, 0, 0, 0],
            &[0, 71, 255, 255, 140, 150, 223, 255, 248, 41, 0, 0],
            &[0, 71, 255, 255, 2, 0, 13, 216, 255, 166, 0, 0],
            &[0, 71, 255, 255, 2, 0, 0, 112, 255, 241, 0, 0],
            &[0, 227, 255, 255, 157, 109, 0, 70, 255, 255, 15, 0],
            &[0, 255, 255, 255, 255, 180, 0, 67, 255, 255, 22, 0],
            &[0, 151, 255, 255, 81, 56, 0, 93, 255, 253, 3, 0],
            &[0, 71, 255, 255, 2, 0, 0, 169, 255, 199, 0, 0],
            &[0, 71, 255, 255, 2, 18, 115, 252, 255, 100, 0, 0],
            &[0, 71, 255, 255, 247, 255, 255, 255, 175, 1, 0, 0],
            &[0, 71, 255, 255, 255, 243, 205, 108, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'đ' / 0x111
        'đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 22, 40, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 184, 0, 0],
            &[0, 0, 0, 0, 0, 185, 225, 255, 255, 255, 135, 0],
            &[0, 0, 0, 0, 0, 131, 159, 255, 255, 255, 95, 0],
            &[0, 0, 0, 88, 169, 172, 94, 128, 255, 184, 0, 0],
            &[0, 0, 119, 255, 255, 255, 255, 215, 255, 184, 0, 0],
            &[0, 10, 244, 255, 201, 67, 82, 244, 255, 184, 0, 0],
            &[0, 63, 255, 255, 50, 0, 0, 143, 255, 184, 0, 0],
            &[0, 86, 255, 255, 7, 0, 0, 104, 255, 184, 0, 0],
            &[0, 69, 255, 255, 16, 0, 0, 111, 255, 184, 0, 0],
            &[0, 33, 255, 255, 107, 0, 6, 195, 255, 184, 0, 0],
            &[0, 0, 197, 255, 254, 210, 230, 249, 255, 184, 0, 0],
            &[0, 0, 33, 207, 255, 255, 173, 73, 255, 184, 0, 0],
            &[0, 0, 0, 0, 29, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ē' / 0x112
        'Ē' => Some(&[
            &[0, 0, 0, 30, 57, 57, 57, 57, 17, 0, 0, 0],
            &[0, 0, 0, 137, 255, 255, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 49, 92, 92, 92, 92, 28, 0, 0, 0],
            &[0, 0, 78, 108, 108, 108, 108, 108, 108, 52, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 203, 147, 147, 147, 147, 70, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 247, 237, 237, 237, 237, 56, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 184, 255, 141, 18, 18, 18, 18, 4, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ē' / 0x113
        'ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 64, 137, 137, 137, 137, 51, 0, 0, 0],
            &[0, 0, 0, 120, 255, 255, 255, 255, 96, 0, 0, 0],
            &[0, 0, 0, 5, 12, 12, 12, 12, 4, 0, 0, 0],
            &[0, 0, 0, 51, 168, 216, 213, 155, 29, 0, 0, 0],
            &[0, 0, 62, 248, 255, 244, 249, 255, 231, 23, 0, 0],
            &[0, 0, 206, 255, 163, 6, 11, 189, 255, 133, 0, 0],
            &[0, 22, 255, 255, 106, 71, 71, 138, 255, 188, 0, 0],
            &[0, 44, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 26, 255, 255, 100, 77, 77, 77, 77, 62, 0, 0],
            &[0, 0, 216, 255, 164, 11, 0, 0, 13, 35, 0, 0],
            &[0, 0, 79, 254, 255, 242, 202, 212, 249, 139, 0, 0],
            &[0, 0, 0, 72, 203, 255, 255, 255, 225, 87, 0, 0],
            &[0, 0, 0, 0, 0, 13, 34, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĕ' / 0x114
        'Ĕ' => Some(&[
            &[0, 0, 0, 177, 190, 5, 20, 227, 123, 0, 0, 0],
            &[0, 0, 0, 81, 255, 246, 253, 241, 32, 0, 0, 0],
            &[0, 0, 0, 0, 63, 141, 130, 33, 0, 0, 0, 0],
            &[0, 0, 78, 108, 108, 108, 108, 108, 108, 52, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 203, 147, 147, 147, 147, 70, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 247, 237, 237, 237, 237, 56, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 184, 255, 141, 18, 18, 18, 18, 4, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĕ' / 0x115
        'ĕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 50, 0, 0, 56, 48, 0, 0, 0],
            &[0, 0, 0, 144, 232, 80, 91, 239, 121, 0, 0, 0],
            &[0, 0, 0, 27, 221, 255, 255, 204, 13, 0, 0, 0],
            &[0, 0, 0, 0, 8, 59, 53, 3, 0, 0, 0, 0],
            &[0, 0, 0, 51, 168, 216, 213, 155, 29, 0, 0, 0],
            &[0, 0, 62, 248, 255, 244, 249, 255, 231, 23, 0, 0],
            &[0, 0, 206, 255, 163, 6, 11, 189, 255, 133, 0, 0],
            &[0, 22, 255, 255, 106, 71, 71, 138, 255, 188, 0, 0],
            &[0, 44, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 26, 255, 255, 100, 77, 77, 77, 77, 62, 0, 0],
            &[0, 0, 216, 255, 164, 11, 0, 0, 13, 35, 0, 0],
            &[0, 0, 79, 254, 255, 242, 202, 212, 249, 139, 0, 0],
            &[0, 0, 0, 72, 203, 255, 255, 255, 225, 87, 0, 0],
            &[0, 0, 0, 0, 0, 13, 34, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ė' / 0x116
        'Ė' => Some(&[
            &[0, 0, 0, 0, 75, 251, 240, 31, 0, 0, 0, 0],
            &[0, 0, 0, 0, 121, 255, 255, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 132, 111, 1, 0, 0, 0, 0],
            &[0, 0, 78, 108, 108, 108, 108, 108, 108, 52, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 203, 147, 147, 147, 147, 70, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 247, 237, 237, 237, 237, 56, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 184, 255, 141, 18, 18, 18, 18, 4, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 7, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ė' / 0x117
        'ė' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 84, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 255, 70, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 255, 254, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 51, 168, 216, 213, 155, 29, 0, 0, 0],
            &[0, 0, 62, 248, 255, 244, 249, 255, 231, 23, 0, 0],
            &[0, 0, 206, 255, 163, 6, 11, 189, 255, 133, 0, 0],
            &[0, 22, 255, 255, 106, 71, 71, 138, 255, 188, 0, 0],
            &[0, 44, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 26, 255, 255, 100, 77, 77, 77, 77, 62, 0, 0],
            &[0, 0, 216, 255, 164, 11, 0, 0, 13, 35, 0, 0],
            &[0, 0, 79, 254, 255, 242, 202, 212, 249, 139, 0, 0],
            &[0, 0, 0, 72, 203, 255, 255, 255, 225, 87, 0, 0],
            &[0, 0, 0, 0, 0, 13, 34, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ę' / 0x118
        'Ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 78, 108, 108, 108, 108, 108, 108, 52, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 203, 147, 147, 147, 147, 70, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 247, 237, 237, 237, 237, 56, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 184, 255, 141, 18, 18, 18, 18, 4, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 74, 243, 61, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 221, 182, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 71, 187, 183, 0, 0, 0],
        ]),
        // letter: 'ę' / 0x119
        'ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 51, 168, 216, 213, 155, 29, 0, 0, 0],
            &[0, 0, 62, 248, 255, 244, 249, 255, 231, 23, 0, 0],
            &[0, 0, 206, 255, 163, 6, 11, 189, 255, 133, 0, 0],
            &[0, 22, 255, 255, 106, 71, 71, 138, 255, 188, 0, 0],
            &[0, 44, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 26, 255, 255, 100, 77, 77, 77, 77, 62, 0, 0],
            &[0, 0, 216, 255, 164, 11, 0, 0, 13, 35, 0, 0],
            &[0, 0, 79, 254, 255, 242, 202, 212, 249, 139, 0, 0],
            &[0, 0, 0, 72, 203, 255, 255, 255, 255, 130, 0, 0],
            &[0, 0, 0, 0, 0, 13, 71, 255, 114, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 152, 252, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 164, 194, 52, 0, 0],
        ]),
        // letter: 'Ě' / 0x11a
        'Ě' => Some(&[
            &[0, 0, 1, 188, 203, 29, 61, 225, 141, 0, 0, 0],
            &[0, 0, 0, 28, 236, 226, 246, 206, 7, 0, 0, 0],
            &[0, 0, 0, 0, 74, 157, 157, 42, 0, 0, 0, 0],
            &[0, 0, 78, 108, 108, 108, 108, 108, 108, 52, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 203, 147, 147, 147, 147, 70, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 247, 237, 237, 237, 237, 56, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 61, 0, 0],
            &[0, 0, 184, 255, 141, 18, 18, 18, 18, 4, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ě' / 0x11b
        'ě' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 63, 40, 0, 0, 45, 57, 0, 0, 0],
            &[0, 0, 0, 129, 250, 94, 117, 254, 111, 0, 0, 0],
            &[0, 0, 0, 1, 188, 254, 255, 172, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 77, 77, 17, 0, 0, 0, 0],
            &[0, 0, 0, 51, 168, 216, 213, 155, 29, 0, 0, 0],
            &[0, 0, 62, 248, 255, 244, 249, 255, 231, 23, 0, 0],
            &[0, 0, 206, 255, 163, 6, 11, 189, 255, 133, 0, 0],
            &[0, 22, 255, 255, 106, 71, 71, 138, 255, 188, 0, 0],
            &[0, 44, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 26, 255, 255, 100, 77, 77, 77, 77, 62, 0, 0],
            &[0, 0, 216, 255, 164, 11, 0, 0, 13, 35, 0, 0],
            &[0, 0, 79, 254, 255, 242, 202, 212, 249, 139, 0, 0],
            &[0, 0, 0, 72, 203, 255, 255, 255, 225, 87, 0, 0],
            &[0, 0, 0, 0, 0, 13, 34, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => Some(&[
            &[0, 0, 0, 0, 59, 240, 241, 175, 0, 0, 0, 0],
            &[0, 0, 0, 18, 223, 223, 150, 255, 111, 0, 0, 0],
            &[0, 0, 0, 91, 150, 25, 0, 100, 153, 12, 0, 0],
            &[0, 0, 0, 0, 38, 114, 141, 124, 82, 9, 0, 0],
            &[0, 0, 3, 146, 254, 255, 255, 255, 255, 157, 0, 0],
            &[0, 0, 124, 255, 255, 187, 118, 125, 191, 63, 0, 0],
            &[0, 11, 246, 255, 149, 0, 0, 0, 0, 0, 0, 0],
            &[0, 78, 255, 253, 20, 0, 0, 0, 0, 0, 0, 0],
            &[0, 112, 255, 221, 0, 0, 88, 110, 110, 97, 0, 0],
            &[0, 127, 255, 209, 0, 0, 204, 255, 255, 225, 0, 0],
            &[0, 109, 255, 231, 0, 0, 99, 153, 255, 225, 0, 0],
            &[0, 65, 255, 255, 43, 0, 0, 55, 255, 225, 0, 0],
            &[0, 5, 230, 255, 195, 19, 0, 55, 255, 225, 0, 0],
            &[0, 0, 82, 252, 255, 246, 213, 235, 255, 225, 0, 0],
            &[0, 0, 0, 72, 204, 255, 255, 255, 236, 144, 0, 0],
            &[0, 0, 0, 0, 0, 12, 34, 19, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĝ' / 0x11d
        'ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 65, 65, 4, 0, 0, 0, 0],
            &[0, 0, 0, 9, 212, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 160, 245, 75, 160, 252, 66, 0, 0, 0],
            &[0, 0, 8, 77, 37, 0, 0, 67, 55, 0, 0, 0],
            &[0, 0, 7, 139, 220, 214, 117, 38, 188, 135, 0, 0],
            &[0, 0, 152, 255, 255, 255, 255, 225, 255, 184, 0, 0],
            &[0, 19, 251, 255, 166, 23, 63, 242, 255, 184, 0, 0],
            &[0, 70, 255, 255, 36, 0, 0, 154, 255, 184, 0, 0],
            &[0, 88, 255, 255, 4, 0, 0, 123, 255, 184, 0, 0],
            &[0, 77, 255, 255, 22, 0, 0, 131, 255, 184, 0, 0],
            &[0, 32, 255, 255, 120, 0, 10, 210, 255, 184, 0, 0],
            &[0, 0, 188, 255, 255, 212, 233, 254, 255, 184, 0, 0],
            &[0, 0, 27, 200, 255, 255, 186, 166, 255, 184, 0, 0],
            &[0, 0, 0, 0, 27, 23, 0, 173, 255, 170, 0, 0],
            &[0, 0, 124, 93, 47, 33, 90, 247, 255, 118, 0, 0],
            &[0, 0, 97, 188, 222, 227, 202, 132, 16, 0, 0, 0],
        ]),
        // letter: 'Ğ' / 0x11e
        'Ğ' => Some(&[
            &[0, 0, 0, 92, 247, 33, 4, 157, 208, 0, 0, 0],
            &[0, 0, 0, 18, 231, 253, 246, 255, 104, 0, 0, 0],
            &[0, 0, 0, 0, 26, 127, 143, 72, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 114, 141, 124, 82, 9, 0, 0],
            &[0, 0, 3, 146, 254, 255, 255, 255, 255, 157, 0, 0],
            &[0, 0, 124, 255, 255, 187, 118, 125, 191, 63, 0, 0],
            &[0, 11, 246, 255, 149, 0, 0, 0, 0, 0, 0, 0],
            &[0, 78, 255, 253, 20, 0, 0, 0, 0, 0, 0, 0],
            &[0, 112, 255, 221, 0, 0, 88, 110, 110, 97, 0, 0],
            &[0, 127, 255, 209, 0, 0, 204, 255, 255, 225, 0, 0],
            &[0, 109, 255, 231, 0, 0, 99, 153, 255, 225, 0, 0],
            &[0, 65, 255, 255, 43, 0, 0, 55, 255, 225, 0, 0],
            &[0, 5, 230, 255, 195, 19, 0, 55, 255, 225, 0, 0],
            &[0, 0, 82, 252, 255, 246, 213, 235, 255, 225, 0, 0],
            &[0, 0, 0, 72, 204, 255, 255, 255, 236, 144, 0, 0],
            &[0, 0, 0, 0, 0, 12, 34, 19, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ğ' / 0x11f
        'ğ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 66, 38, 0, 0, 68, 35, 0, 0, 0],
            &[0, 0, 0, 185, 210, 70, 108, 253, 80, 0, 0, 0],
            &[0, 0, 0, 54, 236, 255, 255, 174, 2, 0, 0, 0],
            &[0, 0, 0, 0, 15, 64, 45, 0, 0, 0, 0, 0],
            &[0, 0, 7, 139, 220, 214, 117, 38, 188, 135, 0, 0],
            &[0, 0, 152, 255, 255, 255, 255, 225, 255, 184, 0, 0],
            &[0, 19, 251, 255, 166, 23, 63, 242, 255, 184, 0, 0],
            &[0, 70, 255, 255, 36, 0, 0, 154, 255, 184, 0, 0],
            &[0, 88, 255, 255, 4, 0, 0, 123, 255, 184, 0, 0],
            &[0, 77, 255, 255, 22, 0, 0, 131, 255, 184, 0, 0],
            &[0, 32, 255, 255, 120, 0, 10, 210, 255, 184, 0, 0],
            &[0, 0, 188, 255, 255, 212, 233, 254, 255, 184, 0, 0],
            &[0, 0, 27, 200, 255, 255, 186, 166, 255, 184, 0, 0],
            &[0, 0, 0, 0, 27, 23, 0, 173, 255, 170, 0, 0],
            &[0, 0, 124, 93, 47, 33, 90, 247, 255, 118, 0, 0],
            &[0, 0, 97, 188, 222, 227, 202, 132, 16, 0, 0, 0],
        ]),
        // letter: 'Ġ' / 0x120
        'Ġ' => Some(&[
            &[0, 0, 0, 0, 13, 228, 254, 102, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 255, 255, 148, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 101, 135, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 114, 141, 124, 82, 9, 0, 0],
            &[0, 0, 3, 146, 254, 255, 255, 255, 255, 157, 0, 0],
            &[0, 0, 124, 255, 255, 187, 118, 125, 191, 63, 0, 0],
            &[0, 11, 246, 255, 149, 0, 0, 0, 0, 0, 0, 0],
            &[0, 78, 255, 253, 20, 0, 0, 0, 0, 0, 0, 0],
            &[0, 112, 255, 221, 0, 0, 88, 110, 110, 97, 0, 0],
            &[0, 127, 255, 209, 0, 0, 204, 255, 255, 225, 0, 0],
            &[0, 109, 255, 231, 0, 0, 99, 153, 255, 225, 0, 0],
            &[0, 65, 255, 255, 43, 0, 0, 55, 255, 225, 0, 0],
            &[0, 5, 230, 255, 195, 19, 0, 55, 255, 225, 0, 0],
            &[0, 0, 82, 252, 255, 246, 213, 235, 255, 225, 0, 0],
            &[0, 0, 0, 72, 204, 255, 255, 255, 236, 144, 0, 0],
            &[0, 0, 0, 0, 0, 12, 34, 19, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 15, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ġ' / 0x121
        'ġ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 92, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 255, 254, 31, 0, 0, 0, 0],
            &[0, 0, 0, 0, 122, 255, 246, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 56, 28, 0, 0, 0, 0, 0],
            &[0, 0, 7, 139, 220, 214, 117, 38, 188, 135, 0, 0],
            &[0, 0, 152, 255, 255, 255, 255, 225, 255, 184, 0, 0],
            &[0, 19, 251, 255, 166, 23, 63, 242, 255, 184, 0, 0],
            &[0, 70, 255, 255, 36, 0, 0, 154, 255, 184, 0, 0],
            &[0, 88, 255, 255, 4, 0, 0, 123, 255, 184, 0, 0],
            &[0, 77, 255, 255, 22, 0, 0, 131, 255, 184, 0, 0],
            &[0, 32, 255, 255, 120, 0, 10, 210, 255, 184, 0, 0],
            &[0, 0, 188, 255, 255, 212, 233, 254, 255, 184, 0, 0],
            &[0, 0, 27, 200, 255, 255, 186, 166, 255, 184, 0, 0],
            &[0, 0, 0, 0, 27, 23, 0, 173, 255, 170, 0, 0],
            &[0, 0, 124, 93, 47, 33, 90, 247, 255, 118, 0, 0],
            &[0, 0, 97, 188, 222, 227, 202, 132, 16, 0, 0, 0],
        ]),
        // letter: 'Ģ' / 0x122
        'Ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 114, 141, 124, 82, 9, 0, 0],
            &[0, 0, 3, 146, 254, 255, 255, 255, 255, 157, 0, 0],
            &[0, 0, 124, 255, 255, 187, 118, 125, 191, 63, 0, 0],
            &[0, 11, 246, 255, 149, 0, 0, 0, 0, 0, 0, 0],
            &[0, 78, 255, 253, 20, 0, 0, 0, 0, 0, 0, 0],
            &[0, 112, 255, 221, 0, 0, 88, 110, 110, 97, 0, 0],
            &[0, 127, 255, 209, 0, 0, 204, 255, 255, 225, 0, 0],
            &[0, 109, 255, 231, 0, 0, 99, 153, 255, 225, 0, 0],
            &[0, 65, 255, 255, 43, 0, 0, 55, 255, 225, 0, 0],
            &[0, 5, 230, 255, 195, 19, 0, 55, 255, 225, 0, 0],
            &[0, 0, 82, 252, 255, 246, 213, 235, 255, 225, 0, 0],
            &[0, 0, 0, 72, 204, 255, 255, 255, 236, 144, 0, 0],
            &[0, 0, 0, 0, 0, 17, 40, 21, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 219, 255, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 137, 30, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ģ' / 0x123
        'ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 19, 65, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 171, 244, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 255, 193, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 77, 49, 0, 0, 0, 0, 0],
            &[0, 0, 7, 139, 220, 214, 117, 38, 188, 135, 0, 0],
            &[0, 0, 152, 255, 255, 255, 255, 225, 255, 184, 0, 0],
            &[0, 19, 251, 255, 166, 23, 63, 242, 255, 184, 0, 0],
            &[0, 70, 255, 255, 36, 0, 0, 154, 255, 184, 0, 0],
            &[0, 88, 255, 255, 4, 0, 0, 123, 255, 184, 0, 0],
            &[0, 77, 255, 255, 22, 0, 0, 131, 255, 184, 0, 0],
            &[0, 32, 255, 255, 120, 0, 10, 210, 255, 184, 0, 0],
            &[0, 0, 188, 255, 255, 212, 233, 254, 255, 184, 0, 0],
            &[0, 0, 27, 200, 255, 255, 186, 166, 255, 184, 0, 0],
            &[0, 0, 0, 0, 27, 23, 0, 173, 255, 170, 0, 0],
            &[0, 0, 124, 93, 47, 33, 90, 247, 255, 118, 0, 0],
            &[0, 0, 97, 188, 222, 227, 202, 132, 16, 0, 0, 0],
        ]),
        // letter: 'Ĥ' / 0x124
        'Ĥ' => Some(&[
            &[0, 0, 0, 0, 162, 241, 241, 71, 0, 0, 0, 0],
            &[0, 0, 0, 97, 255, 154, 218, 231, 25, 0, 0, 0],
            &[0, 0, 9, 150, 106, 0, 21, 147, 97, 0, 0, 0],
            &[0, 30, 108, 108, 4, 0, 0, 47, 108, 95, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 246, 245, 245, 250, 255, 225, 0, 0],
            &[0, 71, 255, 255, 255, 255, 255, 255, 255, 225, 0, 0],
            &[0, 71, 255, 255, 20, 10, 10, 118, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĥ' / 0x125
        'ĥ' => Some(&[
            &[0, 14, 218, 234, 220, 239, 33, 0, 0, 0, 0, 0],
            &[0, 145, 205, 41, 26, 190, 176, 0, 0, 0, 0, 0],
            &[0, 0, 4, 40, 40, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 30, 255, 255, 51, 0, 0, 0, 0, 0, 0],
            &[0, 0, 30, 255, 255, 51, 0, 0, 0, 0, 0, 0],
            &[0, 0, 30, 255, 255, 50, 0, 0, 0, 0, 0, 0],
            &[0, 0, 30, 255, 255, 68, 161, 218, 199, 103, 0, 0],
            &[0, 0, 30, 255, 255, 227, 255, 255, 255, 255, 89, 0],
            &[0, 0, 30, 255, 255, 204, 43, 36, 221, 255, 175, 0],
            &[0, 0, 30, 255, 255, 89, 0, 0, 144, 255, 203, 0],
            &[0, 0, 30, 255, 255, 56, 0, 0, 133, 255, 204, 0],
            &[0, 0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0],
            &[0, 0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0],
            &[0, 0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0],
            &[0, 0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 56, 174, 174, 81, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ħ' / 0x126
        'Ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 108, 108, 25, 0, 0, 66, 108, 76, 0, 0],
            &[0, 22, 255, 255, 59, 0, 0, 157, 255, 180, 0, 0],
            &[0, 245, 255, 255, 255, 223, 223, 255, 255, 255, 133, 0],
            &[0, 188, 255, 255, 225, 165, 165, 255, 255, 255, 99, 0],
            &[0, 22, 255, 255, 59, 0, 0, 157, 255, 180, 0, 0],
            &[0, 22, 255, 255, 248, 245, 245, 251, 255, 180, 0, 0],
            &[0, 22, 255, 255, 255, 255, 255, 255, 255, 180, 0, 0],
            &[0, 22, 255, 255, 67, 10, 10, 161, 255, 180, 0, 0],
            &[0, 22, 255, 255, 59, 0, 0, 157, 255, 180, 0, 0],
            &[0, 22, 255, 255, 59, 0, 0, 157, 255, 180, 0, 0],
            &[0, 22, 255, 255, 59, 0, 0, 157, 255, 180, 0, 0],
            &[0, 22, 255, 255, 59, 0, 0, 157, 255, 180, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ħ' / 0x127
        'ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 40, 40, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 0, 0, 0, 0, 0],
            &[0, 255, 255, 255, 255, 225, 127, 0, 0, 0, 0, 0],
            &[0, 190, 255, 255, 210, 159, 90, 0, 0, 0, 0, 0],
            &[0, 30, 255, 255, 56, 120, 177, 158, 63, 0, 0, 0],
            &[0, 30, 255, 255, 206, 255, 255, 255, 254, 65, 0, 0],
            &[0, 30, 255, 255, 223, 84, 77, 235, 255, 168, 0, 0],
            &[0, 30, 255, 255, 100, 0, 0, 150, 255, 200, 0, 0],
            &[0, 30, 255, 255, 58, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĩ' / 0x128
        'Ĩ' => Some(&[
            &[0, 0, 0, 79, 182, 129, 24, 96, 155, 0, 0, 0],
            &[0, 0, 11, 242, 227, 251, 254, 254, 136, 0, 0, 0],
            &[0, 0, 30, 158, 6, 29, 125, 127, 9, 0, 0, 0],
            &[0, 0, 60, 108, 108, 108, 108, 108, 108, 17, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 29, 76, 152, 255, 239, 92, 64, 7, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 90, 184, 224, 255, 251, 201, 172, 24, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĩ' / 0x129
        'ĩ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 0, 0, 8, 20, 0, 0, 0],
            &[0, 0, 0, 147, 254, 209, 103, 167, 189, 0, 0, 0],
            &[0, 0, 24, 254, 153, 200, 255, 255, 91, 0, 0, 0],
            &[0, 0, 17, 78, 0, 0, 45, 47, 0, 0, 0, 0],
            &[0, 0, 114, 188, 188, 188, 188, 36, 0, 0, 0, 0],
            &[0, 0, 147, 255, 255, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 9, 63, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 4, 135, 177, 222, 255, 255, 223, 173, 105, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ī' / 0x12a
        'Ī' => Some(&[
            &[0, 0, 0, 36, 57, 57, 57, 57, 12, 0, 0, 0],
            &[0, 0, 0, 161, 255, 255, 255, 255, 55, 0, 0, 0],
            &[0, 0, 0, 58, 92, 92, 92, 92, 19, 0, 0, 0],
            &[0, 0, 60, 108, 108, 108, 108, 108, 108, 17, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 29, 76, 152, 255, 239, 92, 64, 7, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 90, 184, 224, 255, 251, 201, 172, 24, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ī' / 0x12b
        'ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 86, 137, 137, 137, 137, 29, 0, 0, 0],
            &[0, 0, 0, 161, 255, 255, 255, 255, 55, 0, 0, 0],
            &[0, 0, 0, 7, 12, 12, 12, 12, 2, 0, 0, 0],
            &[0, 0, 114, 188, 188, 188, 188, 36, 0, 0, 0, 0],
            &[0, 0, 147, 255, 255, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 9, 63, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 4, 135, 177, 222, 255, 255, 223, 173, 105, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => Some(&[
            &[0, 0, 0, 201, 169, 2, 30, 241, 98, 0, 0, 0],
            &[0, 0, 0, 105, 255, 244, 254, 230, 19, 0, 0, 0],
            &[0, 0, 0, 0, 75, 144, 125, 24, 0, 0, 0, 0],
            &[0, 0, 60, 108, 108, 108, 108, 108, 108, 17, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 29, 76, 152, 255, 239, 92, 64, 7, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 90, 184, 224, 255, 251, 201, 172, 24, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĭ' / 0x12d
        'ĭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 66, 38, 0, 0, 68, 35, 0, 0, 0],
            &[0, 0, 0, 185, 210, 70, 108, 253, 80, 0, 0, 0],
            &[0, 0, 0, 54, 236, 255, 255, 174, 2, 0, 0, 0],
            &[0, 0, 0, 0, 15, 64, 45, 0, 0, 0, 0, 0],
            &[0, 0, 114, 188, 188, 188, 188, 36, 0, 0, 0, 0],
            &[0, 0, 147, 255, 255, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 9, 63, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 4, 135, 177, 222, 255, 255, 223, 173, 105, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Į' / 0x12e
        'Į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 108, 108, 108, 108, 108, 108, 17, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 29, 76, 152, 255, 239, 92, 64, 7, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 90, 184, 224, 255, 251, 201, 172, 24, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 38, 233, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 233, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 42, 176, 191, 32, 0, 0, 0],
        ]),
        // letter: 'į' / 0x12f
        'į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 85, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 255, 255, 99, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 254, 255, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 46, 0, 0, 0, 0, 0],
            &[0, 0, 114, 188, 188, 188, 188, 36, 0, 0, 0, 0],
            &[0, 0, 147, 255, 255, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 9, 63, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 4, 135, 177, 222, 255, 255, 223, 173, 105, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 74, 243, 61, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 221, 182, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 71, 187, 183, 0, 0, 0],
        ]),
        // letter: 'İ' / 0x130
        'İ' => Some(&[
            &[0, 0, 0, 0, 97, 254, 231, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 255, 255, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 136, 99, 0, 0, 0, 0, 0],
            &[0, 0, 60, 108, 108, 108, 108, 108, 108, 17, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 29, 76, 152, 255, 239, 92, 64, 7, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 90, 184, 224, 255, 251, 201, 172, 24, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 5, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ı' / 0x131
        'ı' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 114, 188, 188, 188, 188, 36, 0, 0, 0, 0],
            &[0, 0, 147, 255, 255, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 9, 63, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 255, 49, 0, 0, 0, 0],
            &[0, 4, 135, 177, 222, 255, 255, 223, 173, 105, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĳ' / 0x132
        'Ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 108, 63, 0, 3, 108, 108, 26, 0, 0],
            &[0, 0, 163, 255, 149, 0, 8, 255, 255, 61, 0, 0],
            &[0, 0, 163, 255, 149, 0, 8, 255, 255, 61, 0, 0],
            &[0, 0, 163, 255, 149, 0, 8, 255, 255, 61, 0, 0],
            &[0, 0, 163, 255, 149, 0, 8, 255, 255, 61, 0, 0],
            &[0, 0, 163, 255, 149, 0, 8, 255, 255, 61, 0, 0],
            &[0, 0, 163, 255, 149, 0, 8, 255, 255, 61, 0, 0],
            &[0, 0, 163, 255, 149, 0, 8, 255, 255, 61, 0, 0],
            &[0, 0, 89, 139, 81, 0, 28, 255, 255, 51, 0, 0],
            &[0, 0, 16, 2, 0, 2, 134, 255, 252, 18, 0, 0],
            &[0, 0, 159, 247, 224, 245, 255, 255, 169, 0, 0, 0],
            &[0, 0, 142, 254, 255, 255, 246, 159, 15, 0, 0, 0],
            &[0, 0, 0, 11, 35, 30, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĳ' / 0x133
        'ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 97, 45, 0, 2, 81, 80, 1, 0, 0],
            &[0, 0, 180, 255, 239, 1, 86, 255, 255, 78, 0, 0],
            &[0, 0, 167, 255, 223, 0, 73, 255, 255, 63, 0, 0],
            &[0, 0, 8, 60, 18, 0, 0, 46, 41, 0, 0, 0],
            &[0, 0, 108, 188, 140, 0, 33, 188, 188, 27, 0, 0],
            &[0, 0, 147, 255, 190, 0, 45, 255, 255, 36, 0, 0],
            &[0, 0, 147, 255, 190, 0, 45, 255, 255, 36, 0, 0],
            &[0, 0, 147, 255, 190, 0, 45, 255, 255, 36, 0, 0],
            &[0, 0, 147, 255, 190, 0, 45, 255, 255, 36, 0, 0],
            &[0, 0, 147, 255, 190, 0, 45, 255, 255, 36, 0, 0],
            &[0, 0, 147, 255, 190, 0, 45, 255, 255, 36, 0, 0],
            &[0, 0, 147, 255, 190, 0, 45, 255, 255, 36, 0, 0],
            &[0, 0, 147, 255, 190, 0, 45, 255, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 69, 255, 255, 32, 0, 0],
            &[0, 0, 8, 71, 34, 46, 185, 255, 249, 8, 0, 0],
            &[0, 0, 11, 176, 218, 231, 208, 126, 6, 0, 0, 0],
        ]),
        // letter: 'Ĵ' / 0x134
        'Ĵ' => Some(&[
            &[0, 0, 0, 0, 3, 194, 241, 236, 41, 0, 0, 0],
            &[0, 0, 0, 0, 134, 254, 138, 235, 209, 10, 0, 0],
            &[0, 0, 0, 25, 156, 84, 0, 37, 154, 74, 0, 0],
            &[0, 0, 0, 0, 0, 46, 108, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 110, 255, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 110, 255, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 110, 255, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 110, 255, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 110, 255, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 110, 255, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 110, 255, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 128, 255, 205, 0, 0, 0, 0],
            &[0, 17, 1, 0, 16, 211, 255, 169, 0, 0, 0, 0],
            &[0, 163, 246, 226, 251, 255, 255, 70, 0, 0, 0, 0],
            &[0, 145, 254, 255, 255, 227, 99, 0, 0, 0, 0, 0],
            &[0, 0, 14, 35, 24, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĵ' / 0x135
        'ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 65, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 250, 255, 251, 54, 0, 0, 0],
            &[0, 0, 0, 17, 219, 214, 52, 213, 223, 19, 0, 0],
            &[0, 0, 0, 31, 77, 14, 0, 13, 77, 32, 0, 0],
            &[0, 0, 0, 176, 188, 188, 188, 162, 0, 0, 0, 0],
            &[0, 0, 0, 230, 255, 255, 255, 221, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 138, 255, 221, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 221, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 221, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 221, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 221, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 221, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 117, 255, 221, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 255, 216, 0, 0, 0, 0],
            &[0, 33, 58, 30, 64, 230, 255, 185, 0, 0, 0, 0],
            &[0, 55, 191, 223, 229, 192, 85, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ķ' / 0x136
        'Ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 108, 107, 0, 0, 0, 68, 108, 97, 0, 0],
            &[0, 71, 255, 253, 0, 0, 36, 245, 255, 103, 0, 0],
            &[0, 71, 255, 253, 0, 2, 195, 255, 178, 0, 0, 0],
            &[0, 71, 255, 253, 0, 111, 255, 232, 21, 0, 0, 0],
            &[0, 71, 255, 253, 35, 245, 255, 74, 0, 0, 0, 0],
            &[0, 71, 255, 253, 180, 255, 165, 0, 0, 0, 0, 0],
            &[0, 71, 255, 255, 255, 255, 224, 8, 0, 0, 0, 0],
            &[0, 71, 255, 255, 189, 234, 255, 119, 0, 0, 0, 0],
            &[0, 71, 255, 254, 7, 111, 255, 240, 20, 0, 0, 0],
            &[0, 71, 255, 253, 0, 7, 224, 255, 146, 0, 0, 0],
            &[0, 71, 255, 253, 0, 0, 96, 255, 250, 37, 0, 0],
            &[0, 71, 255, 253, 0, 0, 3, 214, 255, 174, 0, 0],
            &[0, 0, 0, 0, 0, 6, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 62, 255, 202, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 86, 114, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ķ' / 0x137
        'ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 40, 40, 11, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 255, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 255, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 255, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 255, 255, 71, 0, 3, 155, 188, 152, 3, 0],
            &[0, 10, 255, 255, 71, 0, 149, 255, 235, 36, 0, 0],
            &[0, 10, 255, 255, 70, 110, 255, 246, 56, 0, 0, 0],
            &[0, 10, 255, 255, 124, 252, 255, 79, 0, 0, 0, 0],
            &[0, 10, 255, 255, 247, 255, 255, 90, 0, 0, 0, 0],
            &[0, 10, 255, 255, 246, 165, 255, 237, 24, 0, 0, 0],
            &[0, 10, 255, 255, 87, 1, 191, 255, 176, 0, 0, 0],
            &[0, 10, 255, 255, 71, 0, 36, 246, 255, 91, 0, 0],
            &[0, 10, 255, 255, 71, 0, 0, 117, 255, 237, 24, 0],
            &[0, 0, 0, 0, 0, 5, 6, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 254, 241, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 57, 134, 8, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĸ' / 0x138
        'ĸ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 188, 188, 52, 0, 3, 155, 188, 152, 3, 0],
            &[0, 10, 255, 255, 71, 0, 149, 255, 235, 36, 0, 0],
            &[0, 10, 255, 255, 70, 110, 255, 246, 56, 0, 0, 0],
            &[0, 10, 255, 255, 124, 252, 255, 79, 0, 0, 0, 0],
            &[0, 10, 255, 255, 247, 255, 255, 90, 0, 0, 0, 0],
            &[0, 10, 255, 255, 246, 165, 255, 237, 24, 0, 0, 0],
            &[0, 10, 255, 255, 87, 1, 191, 255, 176, 0, 0, 0],
            &[0, 10, 255, 255, 71, 0, 36, 246, 255, 91, 0, 0],
            &[0, 10, 255, 255, 71, 0, 0, 117, 255, 237, 24, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĺ' / 0x139
        'Ĺ' => Some(&[
            &[0, 0, 0, 57, 240, 237, 67, 0, 0, 0, 0, 0],
            &[0, 0, 9, 215, 247, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 67, 157, 57, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 108, 82, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 197, 12, 12, 12, 12, 5, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĺ' / 0x13a
        'ĺ' => Some(&[
            &[0, 0, 0, 0, 0, 120, 255, 188, 11, 0, 0, 0],
            &[0, 0, 0, 0, 32, 219, 160, 7, 0, 0, 0, 0],
            &[0, 0, 19, 40, 40, 40, 40, 6, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 40, 123, 185, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 4, 135, 178, 223, 255, 255, 222, 173, 104, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 157, 174, 105, 0, 0, 0],
        ]),
        // letter: 'Ļ' / 0x13b
        'Ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 108, 82, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 197, 12, 12, 12, 12, 5, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 4, 6, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 227, 253, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 137, 25, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ļ' / 0x13c
        'ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 40, 40, 40, 40, 6, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 40, 123, 185, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 4, 135, 178, 223, 255, 255, 222, 173, 104, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 6, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 255, 227, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 127, 2, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ľ' / 0x13d
        'Ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 108, 82, 0, 0, 82, 108, 39, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 226, 244, 20, 0, 0],
            &[0, 0, 143, 255, 194, 0, 26, 255, 127, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 7, 34, 5, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 197, 12, 12, 12, 12, 5, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ľ' / 0x13e
        'ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 40, 40, 40, 40, 6, 25, 40, 21, 0],
            &[0, 0, 122, 255, 255, 255, 255, 40, 183, 255, 72, 0],
            &[0, 0, 40, 123, 185, 255, 255, 41, 234, 196, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 48, 102, 36, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 4, 135, 178, 223, 255, 255, 222, 173, 104, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 108, 82, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 104, 149, 38, 0, 0],
            &[0, 0, 143, 255, 194, 0, 30, 255, 255, 169, 0, 0],
            &[0, 0, 143, 255, 194, 0, 11, 230, 255, 122, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 11, 34, 0, 0, 0],
            &[0, 0, 143, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 197, 12, 12, 12, 12, 5, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 143, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŀ' / 0x140
        'ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 19, 40, 40, 40, 40, 6, 0, 0, 0, 0, 0],
            &[0, 122, 255, 255, 255, 255, 40, 0, 0, 0, 0, 0],
            &[0, 40, 123, 185, 255, 255, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 255, 40, 0, 27, 106, 50, 0],
            &[0, 0, 0, 40, 255, 255, 40, 0, 186, 255, 243, 5],
            &[0, 0, 0, 40, 255, 255, 40, 0, 175, 255, 234, 3],
            &[0, 0, 0, 40, 255, 255, 40, 0, 13, 81, 30, 0],
            &[0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0, 0],
            &[4, 135, 178, 223, 255, 255, 222, 173, 104, 0, 0, 0],
            &[10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ł' / 0x141
        'Ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 99, 108, 43, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 255, 102, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 255, 102, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 255, 102, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 255, 190, 197, 8, 0, 0, 0, 0],
            &[0, 0, 0, 255, 255, 255, 243, 58, 0, 0, 0, 0],
            &[0, 9, 132, 255, 255, 255, 28, 0, 0, 0, 0, 0],
            &[0, 29, 245, 255, 255, 102, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 241, 255, 102, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 235, 255, 109, 12, 12, 12, 12, 1, 0],
            &[0, 0, 0, 235, 255, 255, 255, 255, 255, 255, 30, 0],
            &[0, 0, 0, 235, 255, 255, 255, 255, 255, 255, 30, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ł' / 0x142
        'ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 40, 40, 40, 40, 6, 0, 0, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 40, 123, 185, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 1, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 212, 131, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 255, 255, 178, 0, 0, 0],
            &[0, 0, 0, 56, 248, 255, 255, 127, 0, 0, 0, 0],
            &[0, 0, 6, 230, 255, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 74, 88, 255, 255, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 40, 0, 0, 0, 0],
            &[0, 4, 135, 178, 223, 255, 255, 222, 173, 104, 0, 0],
            &[0, 10, 255, 255, 255, 255, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ń' / 0x143
        'Ń' => Some(&[
            &[0, 0, 0, 0, 0, 10, 214, 241, 136, 0, 0, 0],
            &[0, 0, 0, 0, 0, 147, 255, 146, 1, 0, 0, 0],
            &[0, 0, 0, 0, 20, 156, 105, 0, 0, 0, 0, 0],
            &[0, 30, 108, 108, 102, 0, 0, 19, 108, 95, 0, 0],
            &[0, 71, 255, 255, 255, 44, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 248, 255, 128, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 190, 251, 211, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 185, 188, 255, 39, 47, 255, 225, 0, 0],
            &[0, 71, 255, 196, 105, 255, 123, 47, 255, 225, 0, 0],
            &[0, 71, 255, 200, 24, 253, 206, 46, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 195, 255, 76, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 112, 255, 152, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 30, 255, 225, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 0, 202, 255, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 0, 119, 255, 255, 225, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ń' / 0x144
        'ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 65, 55, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 248, 249, 77, 0, 0, 0],
            &[0, 0, 0, 0, 5, 202, 242, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 77, 36, 0, 0, 0, 0, 0],
            &[0, 22, 188, 144, 31, 164, 217, 200, 107, 1, 0, 0],
            &[0, 30, 255, 251, 223, 255, 255, 255, 255, 91, 0, 0],
            &[0, 30, 255, 255, 215, 39, 39, 225, 255, 175, 0, 0],
            &[0, 30, 255, 255, 99, 0, 0, 145, 255, 203, 0, 0],
            &[0, 30, 255, 255, 60, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ņ' / 0x145
        'Ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 108, 108, 102, 0, 0, 19, 108, 95, 0, 0],
            &[0, 71, 255, 255, 255, 44, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 248, 255, 128, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 190, 251, 211, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 185, 188, 255, 39, 47, 255, 225, 0, 0],
            &[0, 71, 255, 196, 105, 255, 123, 47, 255, 225, 0, 0],
            &[0, 71, 255, 200, 24, 253, 206, 46, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 195, 255, 76, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 112, 255, 152, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 30, 255, 225, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 0, 202, 255, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 0, 119, 255, 255, 225, 0, 0],
            &[0, 0, 0, 0, 1, 6, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 170, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 103, 97, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ņ' / 0x146
        'ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 188, 144, 31, 164, 217, 200, 107, 1, 0, 0],
            &[0, 30, 255, 251, 223, 255, 255, 255, 255, 91, 0, 0],
            &[0, 30, 255, 255, 215, 39, 39, 225, 255, 175, 0, 0],
            &[0, 30, 255, 255, 99, 0, 0, 145, 255, 203, 0, 0],
            &[0, 30, 255, 255, 60, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 0, 0, 0, 1, 6, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 186, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 95, 105, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ň' / 0x147
        'Ň' => Some(&[
            &[0, 0, 2, 191, 201, 27, 64, 226, 137, 0, 0, 0],
            &[0, 0, 0, 30, 237, 225, 247, 203, 6, 0, 0, 0],
            &[0, 0, 0, 0, 76, 157, 157, 39, 0, 0, 0, 0],
            &[0, 30, 108, 108, 102, 0, 0, 19, 108, 95, 0, 0],
            &[0, 71, 255, 255, 255, 44, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 248, 255, 128, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 190, 251, 211, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 185, 188, 255, 39, 47, 255, 225, 0, 0],
            &[0, 71, 255, 196, 105, 255, 123, 47, 255, 225, 0, 0],
            &[0, 71, 255, 200, 24, 253, 206, 46, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 195, 255, 76, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 112, 255, 152, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 30, 255, 225, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 0, 202, 255, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 0, 119, 255, 255, 225, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ň' / 0x148
        'ň' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 65, 34, 0, 0, 52, 51, 0, 0, 0],
            &[0, 0, 0, 153, 245, 75, 139, 255, 87, 0, 0, 0],
            &[0, 0, 0, 7, 207, 254, 255, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 77, 77, 10, 0, 0, 0, 0],
            &[0, 22, 188, 144, 31, 164, 217, 200, 107, 1, 0, 0],
            &[0, 30, 255, 251, 223, 255, 255, 255, 255, 91, 0, 0],
            &[0, 30, 255, 255, 215, 39, 39, 225, 255, 175, 0, 0],
            &[0, 30, 255, 255, 99, 0, 0, 145, 255, 203, 0, 0],
            &[0, 30, 255, 255, 60, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŉ' / 0x149
        'ŉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[8, 108, 108, 23, 0, 0, 0, 0, 0, 0, 0, 0],
            &[51, 255, 250, 15, 0, 0, 0, 0, 0, 0, 0, 0],
            &[99, 255, 178, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[152, 255, 79, 180, 173, 18, 152, 215, 202, 111, 1, 0],
            &[97, 123, 2, 245, 255, 220, 255, 255, 255, 255, 94, 0],
            &[0, 0, 0, 245, 255, 234, 52, 34, 223, 255, 176, 0],
            &[0, 0, 0, 245, 255, 138, 0, 0, 144, 255, 203, 0],
            &[0, 0, 0, 245, 255, 100, 0, 0, 133, 255, 204, 0],
            &[0, 0, 0, 245, 255, 92, 0, 0, 133, 255, 204, 0],
            &[0, 0, 0, 245, 255, 92, 0, 0, 133, 255, 204, 0],
            &[0, 0, 0, 245, 255, 92, 0, 0, 133, 255, 204, 0],
            &[0, 0, 0, 245, 255, 92, 0, 0, 133, 255, 204, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 108, 108, 102, 0, 0, 19, 108, 95, 0, 0],
            &[0, 71, 255, 255, 255, 44, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 248, 255, 128, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 190, 251, 211, 0, 47, 255, 225, 0, 0],
            &[0, 71, 255, 185, 188, 255, 39, 47, 255, 225, 0, 0],
            &[0, 71, 255, 196, 105, 255, 123, 47, 255, 225, 0, 0],
            &[0, 71, 255, 200, 24, 253, 206, 46, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 195, 255, 76, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 112, 255, 152, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 30, 255, 225, 255, 225, 0, 0],
            &[0, 71, 255, 200, 0, 0, 202, 255, 255, 255, 0, 0],
            &[0, 71, 255, 200, 0, 0, 119, 255, 255, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 37, 255, 214, 0, 0],
            &[0, 0, 0, 0, 24, 103, 107, 205, 255, 151, 0, 0],
            &[0, 0, 0, 0, 8, 67, 74, 37, 0, 0, 0, 0],
        ]),
        // letter: 'ŋ' / 0x14b
        'ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 188, 144, 31, 164, 217, 200, 107, 1, 0, 0],
            &[0, 30, 255, 251, 223, 255, 255, 255, 255, 91, 0, 0],
            &[0, 30, 255, 255, 215, 39, 39, 225, 255, 175, 0, 0],
            &[0, 30, 255, 255, 99, 0, 0, 145, 255, 203, 0, 0],
            &[0, 30, 255, 255, 60, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 133, 255, 204, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 236, 255, 255, 0, 0],
            &[0, 30, 255, 255, 51, 0, 0, 255, 255, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 134, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 10, 16, 187, 255, 190, 0, 0],
            &[0, 0, 0, 0, 0, 75, 204, 200, 119, 2, 0, 0],
        ]),
        // letter: 'Ō' / 0x14c
        'Ō' => Some(&[
            &[0, 0, 0, 36, 57, 57, 57, 57, 12, 0, 0, 0],
            &[0, 0, 0, 161, 255, 255, 255, 255, 55, 0, 0, 0],
            &[0, 0, 0, 58, 92, 92, 92, 92, 19, 0, 0, 0],
            &[0, 0, 0, 13, 105, 146, 139, 72, 0, 0, 0, 0],
            &[0, 0, 37, 229, 255, 255, 255, 255, 166, 2, 0, 0],
            &[0, 0, 193, 255, 229, 115, 158, 255, 255, 96, 0, 0],
            &[0, 38, 255, 255, 69, 0, 0, 177, 255, 195, 0, 0],
            &[0, 95, 255, 247, 2, 0, 0, 99, 255, 247, 2, 0],
            &[0, 119, 255, 221, 0, 0, 0, 72, 255, 255, 21, 0],
            &[0, 129, 255, 216, 0, 0, 0, 63, 255, 255, 26, 0],
            &[0, 115, 255, 230, 0, 0, 0, 77, 255, 255, 13, 0],
            &[0, 76, 255, 255, 20, 0, 0, 124, 255, 230, 0, 0],
            &[0, 13, 246, 255, 141, 0, 21, 225, 255, 157, 0, 0],
            &[0, 0, 129, 255, 255, 231, 245, 255, 244, 38, 0, 0],
            &[0, 0, 2, 131, 241, 255, 255, 211, 60, 0, 0, 0],
            &[0, 0, 0, 0, 5, 32, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ō' / 0x14d
        'ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 86, 137, 137, 137, 137, 29, 0, 0, 0],
            &[0, 0, 0, 161, 255, 255, 255, 255, 55, 0, 0, 0],
            &[0, 0, 0, 7, 12, 12, 12, 12, 2, 0, 0, 0],
            &[0, 0, 0, 61, 175, 218, 209, 145, 23, 0, 0, 0],
            &[0, 0, 84, 252, 255, 255, 255, 255, 225, 19, 0, 0],
            &[0, 4, 233, 255, 170, 14, 40, 239, 255, 141, 0, 0],
            &[0, 59, 255, 255, 38, 0, 0, 141, 255, 215, 0, 0],
            &[0, 85, 255, 255, 5, 0, 0, 109, 255, 239, 0, 0],
            &[0, 73, 255, 255, 29, 0, 0, 130, 255, 226, 0, 0],
            &[0, 20, 250, 255, 128, 0, 7, 215, 255, 167, 0, 0],
            &[0, 0, 144, 255, 254, 199, 222, 255, 249, 48, 0, 0],
            &[0, 0, 3, 134, 242, 255, 255, 213, 66, 0, 0, 0],
            &[0, 0, 0, 0, 5, 33, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => Some(&[
            &[0, 0, 0, 201, 169, 2, 30, 241, 98, 0, 0, 0],
            &[0, 0, 0, 105, 255, 244, 254, 230, 19, 0, 0, 0],
            &[0, 0, 0, 0, 75, 144, 125, 24, 0, 0, 0, 0],
            &[0, 0, 0, 13, 105, 146, 139, 72, 0, 0, 0, 0],
            &[0, 0, 37, 229, 255, 255, 255, 255, 166, 2, 0, 0],
            &[0, 0, 193, 255, 229, 115, 158, 255, 255, 96, 0, 0],
            &[0, 38, 255, 255, 69, 0, 0, 177, 255, 195, 0, 0],
            &[0, 95, 255, 247, 2, 0, 0, 99, 255, 247, 2, 0],
            &[0, 119, 255, 221, 0, 0, 0, 72, 255, 255, 21, 0],
            &[0, 129, 255, 216, 0, 0, 0, 63, 255, 255, 26, 0],
            &[0, 115, 255, 230, 0, 0, 0, 77, 255, 255, 13, 0],
            &[0, 76, 255, 255, 20, 0, 0, 124, 255, 230, 0, 0],
            &[0, 13, 246, 255, 141, 0, 21, 225, 255, 157, 0, 0],
            &[0, 0, 129, 255, 255, 231, 245, 255, 244, 38, 0, 0],
            &[0, 0, 2, 131, 241, 255, 255, 211, 60, 0, 0, 0],
            &[0, 0, 0, 0, 5, 32, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŏ' / 0x14f
        'ŏ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 66, 38, 0, 0, 68, 35, 0, 0, 0],
            &[0, 0, 0, 185, 210, 70, 108, 253, 80, 0, 0, 0],
            &[0, 0, 0, 54, 236, 255, 255, 174, 2, 0, 0, 0],
            &[0, 0, 0, 0, 15, 64, 45, 0, 0, 0, 0, 0],
            &[0, 0, 0, 61, 175, 218, 209, 145, 23, 0, 0, 0],
            &[0, 0, 84, 252, 255, 255, 255, 255, 225, 19, 0, 0],
            &[0, 4, 233, 255, 170, 14, 40, 239, 255, 141, 0, 0],
            &[0, 59, 255, 255, 38, 0, 0, 141, 255, 215, 0, 0],
            &[0, 85, 255, 255, 5, 0, 0, 109, 255, 239, 0, 0],
            &[0, 73, 255, 255, 29, 0, 0, 130, 255, 226, 0, 0],
            &[0, 20, 250, 255, 128, 0, 7, 215, 255, 167, 0, 0],
            &[0, 0, 144, 255, 254, 199, 222, 255, 249, 48, 0, 0],
            &[0, 0, 3, 134, 242, 255, 255, 213, 66, 0, 0, 0],
            &[0, 0, 0, 0, 5, 33, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ő' / 0x150
        'Ő' => Some(&[
            &[0, 0, 0, 11, 220, 236, 76, 235, 226, 26, 0, 0],
            &[0, 0, 0, 132, 251, 86, 165, 243, 60, 0, 0, 0],
            &[0, 0, 4, 150, 68, 18, 156, 50, 0, 0, 0, 0],
            &[0, 0, 0, 13, 105, 146, 139, 72, 0, 0, 0, 0],
            &[0, 0, 37, 229, 255, 255, 255, 255, 166, 2, 0, 0],
            &[0, 0, 193, 255, 229, 115, 158, 255, 255, 96, 0, 0],
            &[0, 38, 255, 255, 69, 0, 0, 177, 255, 195, 0, 0],
            &[0, 95, 255, 247, 2, 0, 0, 99, 255, 247, 2, 0],
            &[0, 119, 255, 221, 0, 0, 0, 72, 255, 255, 21, 0],
            &[0, 129, 255, 216, 0, 0, 0, 63, 255, 255, 26, 0],
            &[0, 115, 255, 230, 0, 0, 0, 77, 255, 255, 13, 0],
            &[0, 76, 255, 255, 20, 0, 0, 124, 255, 230, 0, 0],
            &[0, 13, 246, 255, 141, 0, 21, 225, 255, 157, 0, 0],
            &[0, 0, 129, 255, 255, 231, 245, 255, 244, 38, 0, 0],
            &[0, 0, 2, 131, 241, 255, 255, 211, 60, 0, 0, 0],
            &[0, 0, 0, 0, 5, 32, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ő' / 0x151
        'ő' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 65, 27, 59, 65, 18, 0, 0],
            &[0, 0, 0, 36, 250, 224, 84, 255, 203, 8, 0, 0],
            &[0, 0, 0, 180, 220, 37, 209, 202, 18, 0, 0, 0],
            &[0, 0, 4, 77, 19, 14, 77, 11, 0, 0, 0, 0],
            &[0, 0, 0, 61, 175, 218, 209, 145, 23, 0, 0, 0],
            &[0, 0, 84, 252, 255, 255, 255, 255, 225, 19, 0, 0],
            &[0, 4, 233, 255, 170, 14, 40, 239, 255, 141, 0, 0],
            &[0, 59, 255, 255, 38, 0, 0, 141, 255, 215, 0, 0],
            &[0, 85, 255, 255, 5, 0, 0, 109, 255, 239, 0, 0],
            &[0, 73, 255, 255, 29, 0, 0, 130, 255, 226, 0, 0],
            &[0, 20, 250, 255, 128, 0, 7, 215, 255, 167, 0, 0],
            &[0, 0, 144, 255, 254, 199, 222, 255, 249, 48, 0, 0],
            &[0, 0, 3, 134, 242, 255, 255, 213, 66, 0, 0, 0],
            &[0, 0, 0, 0, 5, 33, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Œ' / 0x152
        'Œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 130, 141, 110, 108, 108, 108, 30, 0],
            &[0, 0, 79, 251, 255, 255, 255, 255, 255, 255, 71, 0],
            &[0, 3, 226, 255, 208, 134, 255, 255, 153, 143, 40, 0],
            &[0, 56, 255, 255, 36, 22, 255, 255, 22, 0, 0, 0],
            &[0, 103, 255, 226, 0, 22, 255, 255, 22, 0, 0, 0],
            &[0, 125, 255, 203, 0, 22, 255, 255, 239, 235, 0, 0],
            &[0, 129, 255, 197, 0, 22, 255, 255, 255, 253, 0, 0],
            &[0, 116, 255, 211, 0, 22, 255, 255, 35, 14, 0, 0],
            &[0, 85, 255, 234, 0, 22, 255, 255, 22, 0, 0, 0],
            &[0, 30, 254, 255, 69, 22, 255, 255, 22, 0, 0, 0],
            &[0, 0, 179, 255, 251, 234, 255, 255, 252, 251, 70, 0],
            &[0, 0, 22, 192, 255, 255, 255, 255, 255, 255, 71, 0],
            &[0, 0, 0, 0, 19, 31, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'œ' / 0x153
        'œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 201, 208, 87, 58, 196, 195, 58, 0, 0],
            &[0, 26, 247, 244, 240, 251, 237, 222, 238, 234, 9, 0],
            &[0, 113, 255, 106, 59, 255, 255, 41, 107, 255, 71, 0],
            &[0, 156, 255, 59, 2, 253, 253, 64, 120, 255, 103, 0],
            &[0, 176, 255, 47, 0, 240, 255, 255, 255, 255, 112, 0],
            &[0, 167, 255, 54, 0, 250, 251, 45, 45, 45, 19, 0],
            &[0, 128, 255, 87, 35, 255, 255, 38, 0, 0, 1, 0],
            &[0, 47, 255, 213, 195, 255, 253, 211, 123, 182, 54, 0],
            &[0, 0, 124, 253, 255, 151, 106, 251, 255, 223, 34, 0],
            &[0, 0, 0, 16, 23, 0, 0, 16, 27, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŕ' / 0x154
        'Ŕ' => Some(&[
            &[0, 0, 0, 0, 18, 225, 241, 116, 0, 0, 0, 0],
            &[0, 0, 0, 0, 167, 255, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 157, 93, 0, 0, 0, 0, 0, 0],
            &[0, 78, 108, 108, 103, 79, 16, 0, 0, 0, 0, 0],
            &[0, 184, 255, 255, 255, 255, 245, 100, 0, 0, 0, 0],
            &[0, 184, 255, 209, 146, 209, 255, 253, 41, 0, 0, 0],
            &[0, 184, 255, 153, 0, 8, 239, 255, 111, 0, 0, 0],
            &[0, 184, 255, 153, 0, 1, 239, 255, 105, 0, 0, 0],
            &[0, 184, 255, 181, 81, 166, 255, 252, 42, 0, 0, 0],
            &[0, 184, 255, 255, 255, 255, 252, 112, 0, 0, 0, 0],
            &[0, 184, 255, 224, 209, 255, 199, 0, 0, 0, 0, 0],
            &[0, 184, 255, 153, 23, 243, 255, 84, 0, 0, 0, 0],
            &[0, 184, 255, 153, 0, 125, 255, 224, 8, 0, 0, 0],
            &[0, 184, 255, 153, 0, 10, 228, 255, 125, 0, 0, 0],
            &[0, 184, 255, 153, 0, 0, 97, 255, 245, 28, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŕ' / 0x155
        'ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 65, 65, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 181, 255, 173, 4, 0, 0],
            &[0, 0, 0, 0, 0, 101, 255, 158, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 68, 0, 0, 0, 0, 0],
            &[0, 0, 171, 188, 188, 145, 11, 148, 218, 210, 41, 0],
            &[0, 0, 226, 255, 255, 244, 181, 255, 255, 247, 9, 0],
            &[0, 0, 1, 45, 255, 255, 252, 140, 90, 91, 0, 0],
            &[0, 0, 0, 18, 255, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 79, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 63, 0, 0, 0, 0, 0],
            &[0, 41, 158, 213, 255, 255, 225, 168, 68, 0, 0, 0],
            &[0, 88, 255, 255, 255, 255, 255, 255, 137, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŗ' / 0x156
        'Ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 78, 108, 108, 103, 79, 16, 0, 0, 0, 0, 0],
            &[0, 184, 255, 255, 255, 255, 245, 100, 0, 0, 0, 0],
            &[0, 184, 255, 209, 146, 209, 255, 253, 41, 0, 0, 0],
            &[0, 184, 255, 153, 0, 8, 239, 255, 111, 0, 0, 0],
            &[0, 184, 255, 153, 0, 1, 239, 255, 105, 0, 0, 0],
            &[0, 184, 255, 181, 81, 166, 255, 252, 42, 0, 0, 0],
            &[0, 184, 255, 255, 255, 255, 252, 112, 0, 0, 0, 0],
            &[0, 184, 255, 224, 209, 255, 199, 0, 0, 0, 0, 0],
            &[0, 184, 255, 153, 23, 243, 255, 84, 0, 0, 0, 0],
            &[0, 184, 255, 153, 0, 125, 255, 224, 8, 0, 0, 0],
            &[0, 184, 255, 153, 0, 10, 228, 255, 125, 0, 0, 0],
            &[0, 184, 255, 153, 0, 0, 97, 255, 245, 28, 0, 0],
            &[0, 0, 0, 0, 5, 6, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 246, 248, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 48, 136, 15, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŗ' / 0x157
        'ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 171, 188, 188, 145, 11, 148, 218, 210, 41, 0],
            &[0, 0, 226, 255, 255, 244, 181, 255, 255, 247, 9, 0],
            &[0, 0, 1, 45, 255, 255, 252, 140, 90, 91, 0, 0],
            &[0, 0, 0, 18, 255, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 79, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 63, 0, 0, 0, 0, 0],
            &[0, 41, 158, 213, 255, 255, 225, 168, 68, 0, 0, 0],
            &[0, 88, 255, 255, 255, 255, 255, 255, 137, 0, 0, 0],
            &[0, 0, 0, 0, 5, 6, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 252, 244, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 135, 11, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ř' / 0x158
        'Ř' => Some(&[
            &[0, 9, 204, 190, 19, 78, 232, 118, 0, 0, 0, 0],
            &[0, 0, 42, 245, 221, 251, 187, 2, 0, 0, 0, 0],
            &[0, 0, 0, 89, 157, 157, 27, 0, 0, 0, 0, 0],
            &[0, 78, 108, 108, 103, 79, 16, 0, 0, 0, 0, 0],
            &[0, 184, 255, 255, 255, 255, 245, 100, 0, 0, 0, 0],
            &[0, 184, 255, 209, 146, 209, 255, 253, 41, 0, 0, 0],
            &[0, 184, 255, 153, 0, 8, 239, 255, 111, 0, 0, 0],
            &[0, 184, 255, 153, 0, 1, 239, 255, 105, 0, 0, 0],
            &[0, 184, 255, 181, 81, 166, 255, 252, 42, 0, 0, 0],
            &[0, 184, 255, 255, 255, 255, 252, 112, 0, 0, 0, 0],
            &[0, 184, 255, 224, 209, 255, 199, 0, 0, 0, 0, 0],
            &[0, 184, 255, 153, 23, 243, 255, 84, 0, 0, 0, 0],
            &[0, 184, 255, 153, 0, 125, 255, 224, 8, 0, 0, 0],
            &[0, 184, 255, 153, 0, 10, 228, 255, 125, 0, 0, 0],
            &[0, 184, 255, 153, 0, 0, 97, 255, 245, 28, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ř' / 0x159
        'ř' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 60, 0, 0, 24, 65, 13, 0, 0],
            &[0, 0, 0, 55, 248, 166, 58, 235, 189, 4, 0, 0],
            &[0, 0, 0, 0, 108, 255, 254, 232, 22, 0, 0, 0],
            &[0, 0, 0, 0, 1, 75, 77, 42, 0, 0, 0, 0],
            &[0, 0, 171, 188, 188, 145, 11, 148, 218, 210, 41, 0],
            &[0, 0, 226, 255, 255, 244, 181, 255, 255, 247, 9, 0],
            &[0, 0, 1, 45, 255, 255, 252, 140, 90, 91, 0, 0],
            &[0, 0, 0, 18, 255, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 79, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 255, 63, 0, 0, 0, 0, 0],
            &[0, 41, 158, 213, 255, 255, 225, 168, 68, 0, 0, 0],
            &[0, 88, 255, 255, 255, 255, 255, 255, 137, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ś' / 0x15a
        'Ś' => Some(&[
            &[0, 0, 0, 0, 0, 1, 188, 241, 169, 1, 0, 0],
            &[0, 0, 0, 0, 0, 110, 255, 177, 8, 0, 0, 0],
            &[0, 0, 0, 0, 5, 149, 125, 3, 0, 0, 0, 0],
            &[0, 0, 0, 7, 89, 137, 139, 113, 57, 0, 0, 0],
            &[0, 0, 19, 213, 255, 255, 255, 255, 255, 86, 0, 0],
            &[0, 0, 125, 255, 250, 128, 96, 138, 203, 8, 0, 0],
            &[0, 0, 160, 255, 199, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 255, 246, 67, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 216, 255, 255, 179, 42, 0, 0, 0, 0],
            &[0, 0, 0, 17, 159, 255, 255, 252, 137, 1, 0, 0],
            &[0, 0, 0, 0, 0, 52, 192, 255, 255, 104, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 207, 255, 174, 0, 0],
            &[0, 0, 106, 30, 0, 0, 9, 213, 255, 161, 0, 0],
            &[0, 0, 225, 254, 227, 202, 235, 255, 253, 64, 0, 0],
            &[0, 0, 138, 233, 255, 255, 255, 212, 78, 0, 0, 0],
            &[0, 0, 0, 0, 18, 36, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ś' / 0x15b
        'ś' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 65, 56, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 247, 250, 81, 0, 0, 0],
            &[0, 0, 0, 0, 4, 199, 243, 72, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 77, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 87, 187, 222, 214, 185, 116, 17, 0, 0],
            &[0, 0, 87, 255, 255, 255, 255, 255, 245, 15, 0, 0],
            &[0, 0, 158, 255, 219, 16, 20, 63, 88, 0, 0, 0],
            &[0, 0, 122, 255, 254, 170, 70, 1, 0, 0, 0, 0],
            &[0, 0, 4, 146, 251, 255, 255, 217, 69, 0, 0, 0],
            &[0, 0, 0, 0, 31, 132, 235, 255, 248, 26, 0, 0],
            &[0, 0, 72, 27, 0, 0, 54, 255, 255, 73, 0, 0],
            &[0, 0, 184, 254, 223, 204, 237, 255, 253, 34, 0, 0],
            &[0, 0, 136, 242, 255, 255, 255, 223, 91, 0, 0, 0],
            &[0, 0, 0, 3, 25, 36, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => Some(&[
            &[0, 0, 0, 0, 108, 241, 241, 125, 0, 0, 0, 0],
            &[0, 0, 0, 48, 247, 191, 182, 251, 62, 0, 0, 0],
            &[0, 0, 0, 124, 133, 8, 5, 127, 132, 0, 0, 0],
            &[0, 0, 0, 7, 89, 137, 139, 113, 57, 0, 0, 0],
            &[0, 0, 19, 213, 255, 255, 255, 255, 255, 86, 0, 0],
            &[0, 0, 125, 255, 250, 128, 96, 138, 203, 8, 0, 0],
            &[0, 0, 160, 255, 199, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 255, 246, 67, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 216, 255, 255, 179, 42, 0, 0, 0, 0],
            &[0, 0, 0, 17, 159, 255, 255, 252, 137, 1, 0, 0],
            &[0, 0, 0, 0, 0, 52, 192, 255, 255, 104, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 207, 255, 174, 0, 0],
            &[0, 0, 106, 30, 0, 0, 9, 213, 255, 161, 0, 0],
            &[0, 0, 225, 254, 227, 202, 235, 255, 253, 64, 0, 0],
            &[0, 0, 138, 233, 255, 255, 255, 212, 78, 0, 0, 0],
            &[0, 0, 0, 0, 18, 36, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŝ' / 0x15d
        'ŝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 65, 65, 9, 0, 0, 0, 0],
            &[0, 0, 0, 3, 197, 255, 255, 146, 0, 0, 0, 0],
            &[0, 0, 0, 140, 249, 88, 142, 255, 84, 0, 0, 0],
            &[0, 0, 2, 77, 43, 0, 0, 61, 61, 0, 0, 0],
            &[0, 0, 0, 87, 187, 222, 214, 185, 116, 17, 0, 0],
            &[0, 0, 87, 255, 255, 255, 255, 255, 245, 15, 0, 0],
            &[0, 0, 158, 255, 219, 16, 20, 63, 88, 0, 0, 0],
            &[0, 0, 122, 255, 254, 170, 70, 1, 0, 0, 0, 0],
            &[0, 0, 4, 146, 251, 255, 255, 217, 69, 0, 0, 0],
            &[0, 0, 0, 0, 31, 132, 235, 255, 248, 26, 0, 0],
            &[0, 0, 72, 27, 0, 0, 54, 255, 255, 73, 0, 0],
            &[0, 0, 184, 254, 223, 204, 237, 255, 253, 34, 0, 0],
            &[0, 0, 136, 242, 255, 255, 255, 223, 91, 0, 0, 0],
            &[0, 0, 0, 3, 25, 36, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ş' / 0x15e
        'Ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 89, 137, 139, 113, 57, 0, 0, 0],
            &[0, 0, 19, 213, 255, 255, 255, 255, 255, 86, 0, 0],
            &[0, 0, 125, 255, 250, 128, 96, 138, 203, 8, 0, 0],
            &[0, 0, 160, 255, 199, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 255, 246, 67, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 216, 255, 255, 179, 42, 0, 0, 0, 0],
            &[0, 0, 0, 17, 159, 255, 255, 252, 137, 1, 0, 0],
            &[0, 0, 0, 0, 0, 52, 192, 255, 255, 104, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 207, 255, 174, 0, 0],
            &[0, 0, 106, 30, 0, 0, 9, 213, 255, 161, 0, 0],
            &[0, 0, 225, 254, 227, 202, 235, 255, 253, 64, 0, 0],
            &[0, 0, 138, 233, 255, 255, 255, 212, 78, 0, 0, 0],
            &[0, 0, 0, 0, 33, 255, 128, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 152, 255, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 163, 207, 158, 14, 0, 0, 0, 0],
        ]),
        // letter: 'ş' / 0x15f
        'ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 87, 187, 222, 214, 185, 116, 17, 0, 0],
            &[0, 0, 87, 255, 255, 255, 255, 255, 245, 15, 0, 0],
            &[0, 0, 158, 255, 219, 16, 20, 63, 88, 0, 0, 0],
            &[0, 0, 122, 255, 254, 170, 70, 1, 0, 0, 0, 0],
            &[0, 0, 4, 146, 251, 255, 255, 217, 69, 0, 0, 0],
            &[0, 0, 0, 0, 31, 132, 235, 255, 248, 26, 0, 0],
            &[0, 0, 72, 27, 0, 0, 54, 255, 255, 73, 0, 0],
            &[0, 0, 184, 254, 223, 204, 237, 255, 253, 34, 0, 0],
            &[0, 0, 136, 242, 255, 255, 255, 223, 91, 0, 0, 0],
            &[0, 0, 0, 3, 31, 255, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 139, 255, 110, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 208, 165, 22, 0, 0, 0, 0],
        ]),
        // letter: 'Š' / 0x160
        'Š' => Some(&[
            &[0, 0, 0, 159, 217, 46, 42, 213, 172, 0, 0, 0],
            &[0, 0, 0, 13, 218, 237, 235, 227, 19, 0, 0, 0],
            &[0, 0, 0, 0, 54, 157, 157, 62, 0, 0, 0, 0],
            &[0, 0, 0, 7, 89, 137, 139, 113, 57, 0, 0, 0],
            &[0, 0, 19, 213, 255, 255, 255, 255, 255, 86, 0, 0],
            &[0, 0, 125, 255, 250, 128, 96, 138, 203, 8, 0, 0],
            &[0, 0, 160, 255, 199, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 255, 246, 67, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 216, 255, 255, 179, 42, 0, 0, 0, 0],
            &[0, 0, 0, 17, 159, 255, 255, 252, 137, 1, 0, 0],
            &[0, 0, 0, 0, 0, 52, 192, 255, 255, 104, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 207, 255, 174, 0, 0],
            &[0, 0, 106, 30, 0, 0, 9, 213, 255, 161, 0, 0],
            &[0, 0, 225, 254, 227, 202, 235, 255, 253, 64, 0, 0],
            &[0, 0, 138, 233, 255, 255, 255, 212, 78, 0, 0, 0],
            &[0, 0, 0, 0, 18, 36, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'š' / 0x161
        'š' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 65, 35, 0, 0, 51, 52, 0, 0, 0],
            &[0, 0, 0, 149, 246, 78, 136, 255, 91, 0, 0, 0],
            &[0, 0, 0, 6, 204, 254, 255, 151, 0, 0, 0, 0],
            &[0, 0, 0, 0, 30, 77, 77, 11, 0, 0, 0, 0],
            &[0, 0, 0, 87, 187, 222, 214, 185, 116, 17, 0, 0],
            &[0, 0, 87, 255, 255, 255, 255, 255, 245, 15, 0, 0],
            &[0, 0, 158, 255, 219, 16, 20, 63, 88, 0, 0, 0],
            &[0, 0, 122, 255, 254, 170, 70, 1, 0, 0, 0, 0],
            &[0, 0, 4, 146, 251, 255, 255, 217, 69, 0, 0, 0],
            &[0, 0, 0, 0, 31, 132, 235, 255, 248, 26, 0, 0],
            &[0, 0, 72, 27, 0, 0, 54, 255, 255, 73, 0, 0],
            &[0, 0, 184, 254, 223, 204, 237, 255, 253, 34, 0, 0],
            &[0, 0, 136, 242, 255, 255, 255, 223, 91, 0, 0, 0],
            &[0, 0, 0, 3, 25, 36, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ţ' / 0x162
        'Ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 56, 108, 108, 108, 108, 108, 108, 108, 108, 13, 0],
            &[0, 133, 255, 255, 255, 255, 255, 255, 255, 255, 30, 0],
            &[0, 76, 147, 147, 187, 255, 250, 147, 147, 147, 17, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 252, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 243, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 152, 255, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 163, 207, 158, 14, 0, 0, 0, 0],
        ]),
        // letter: 'ţ' / 0x163
        'ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 233, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 215, 255, 42, 0, 0, 0, 0, 0],
            &[0, 10, 102, 158, 252, 255, 199, 188, 188, 84, 0, 0],
            &[0, 38, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 7, 47, 78, 255, 255, 82, 47, 47, 21, 0, 0],
            &[0, 0, 0, 38, 255, 255, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 38, 255, 255, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 38, 255, 255, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 255, 255, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 221, 255, 251, 208, 218, 131, 0, 0],
            &[0, 0, 0, 0, 54, 215, 255, 255, 253, 117, 0, 0],
            &[0, 0, 0, 0, 0, 8, 255, 160, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 143, 255, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 152, 208, 163, 20, 0, 0, 0],
        ]),
        // letter: 'Ť' / 0x164
        'Ť' => Some(&[
            &[0, 0, 9, 204, 190, 19, 78, 232, 118, 0, 0, 0],
            &[0, 0, 0, 42, 245, 221, 251, 187, 2, 0, 0, 0],
            &[0, 0, 0, 0, 89, 157, 157, 27, 0, 0, 0, 0],
            &[0, 56, 108, 108, 108, 108, 108, 108, 108, 108, 13, 0],
            &[0, 133, 255, 255, 255, 255, 255, 255, 255, 255, 30, 0],
            &[0, 76, 147, 147, 187, 255, 250, 147, 147, 147, 17, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ť' / 0x165
        'ť' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 84, 85, 12, 0],
            &[0, 0, 0, 0, 0, 0, 0, 25, 255, 214, 2, 0],
            &[0, 0, 0, 0, 156, 233, 39, 80, 255, 81, 0, 0],
            &[0, 0, 0, 0, 215, 255, 42, 25, 54, 0, 0, 0],
            &[0, 10, 102, 158, 252, 255, 199, 188, 188, 84, 0, 0],
            &[0, 38, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 7, 47, 78, 255, 255, 82, 47, 47, 21, 0, 0],
            &[0, 0, 0, 38, 255, 255, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 38, 255, 255, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 38, 255, 255, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 255, 255, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 221, 255, 251, 208, 218, 131, 0, 0],
            &[0, 0, 0, 0, 54, 215, 255, 255, 253, 117, 0, 0],
            &[0, 0, 0, 0, 0, 0, 24, 35, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŧ' / 0x166
        'Ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 56, 108, 108, 108, 108, 108, 108, 108, 108, 13, 0],
            &[0, 133, 255, 255, 255, 255, 255, 255, 255, 255, 30, 0],
            &[0, 76, 147, 147, 187, 255, 250, 147, 147, 147, 17, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 52, 155, 249, 255, 255, 155, 145, 0, 0, 0],
            &[0, 0, 85, 255, 255, 255, 255, 255, 239, 0, 0, 0],
            &[0, 0, 25, 75, 169, 255, 255, 75, 70, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŧ' / 0x167
        'ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 233, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 215, 255, 42, 0, 0, 0, 0, 0],
            &[0, 10, 102, 158, 252, 255, 199, 188, 188, 84, 0, 0],
            &[0, 38, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 7, 47, 78, 255, 255, 82, 47, 47, 21, 0, 0],
            &[0, 17, 116, 155, 255, 255, 159, 116, 116, 61, 0, 0],
            &[0, 38, 255, 255, 255, 255, 255, 255, 255, 135, 0, 0],
            &[0, 1, 12, 51, 255, 255, 57, 12, 12, 6, 0, 0],
            &[0, 0, 0, 26, 255, 255, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 221, 255, 251, 208, 218, 131, 0, 0],
            &[0, 0, 0, 0, 54, 215, 255, 255, 253, 117, 0, 0],
            &[0, 0, 0, 0, 0, 0, 24, 35, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ũ' / 0x168
        'Ũ' => Some(&[
            &[0, 0, 0, 79, 182, 129, 24, 96, 155, 0, 0, 0],
            &[0, 0, 11, 242, 227, 251, 254, 254, 136, 0, 0, 0],
            &[0, 0, 30, 158, 6, 29, 125, 127, 9, 0, 0, 0],
            &[0, 30, 108, 108, 4, 0, 0, 47, 108, 95, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 11, 0, 0, 114, 255, 225, 0, 0],
            &[0, 61, 255, 255, 37, 0, 0, 138, 255, 214, 0, 0],
            &[0, 24, 253, 255, 131, 0, 14, 217, 255, 172, 0, 0],
            &[0, 0, 170, 255, 255, 235, 249, 255, 254, 63, 0, 0],
            &[0, 0, 14, 158, 247, 255, 255, 220, 84, 0, 0, 0],
            &[0, 0, 0, 0, 8, 32, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ũ' / 0x169
        'ũ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 0, 0, 8, 20, 0, 0, 0],
            &[0, 0, 0, 147, 254, 209, 103, 167, 189, 0, 0, 0],
            &[0, 0, 24, 254, 153, 200, 255, 255, 91, 0, 0, 0],
            &[0, 0, 17, 78, 0, 0, 45, 47, 0, 0, 0, 0],
            &[0, 37, 188, 188, 22, 0, 0, 113, 188, 135, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 158, 255, 184, 0, 0],
            &[0, 51, 255, 255, 35, 0, 0, 182, 255, 184, 0, 0],
            &[0, 31, 255, 255, 91, 0, 26, 241, 255, 184, 0, 0],
            &[0, 2, 223, 255, 249, 204, 241, 249, 255, 184, 0, 0],
            &[0, 0, 60, 226, 255, 254, 176, 68, 255, 184, 0, 0],
            &[0, 0, 0, 1, 28, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ū' / 0x16a
        'Ū' => Some(&[
            &[0, 0, 0, 36, 57, 57, 57, 57, 12, 0, 0, 0],
            &[0, 0, 0, 161, 255, 255, 255, 255, 55, 0, 0, 0],
            &[0, 0, 0, 58, 92, 92, 92, 92, 19, 0, 0, 0],
            &[0, 30, 108, 108, 4, 0, 0, 47, 108, 95, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 11, 0, 0, 114, 255, 225, 0, 0],
            &[0, 61, 255, 255, 37, 0, 0, 138, 255, 214, 0, 0],
            &[0, 24, 253, 255, 131, 0, 14, 217, 255, 172, 0, 0],
            &[0, 0, 170, 255, 255, 235, 249, 255, 254, 63, 0, 0],
            &[0, 0, 14, 158, 247, 255, 255, 220, 84, 0, 0, 0],
            &[0, 0, 0, 0, 8, 32, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ū' / 0x16b
        'ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 86, 137, 137, 137, 137, 29, 0, 0, 0],
            &[0, 0, 0, 161, 255, 255, 255, 255, 55, 0, 0, 0],
            &[0, 0, 0, 7, 12, 12, 12, 12, 2, 0, 0, 0],
            &[0, 37, 188, 188, 22, 0, 0, 113, 188, 135, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 158, 255, 184, 0, 0],
            &[0, 51, 255, 255, 35, 0, 0, 182, 255, 184, 0, 0],
            &[0, 31, 255, 255, 91, 0, 26, 241, 255, 184, 0, 0],
            &[0, 2, 223, 255, 249, 204, 241, 249, 255, 184, 0, 0],
            &[0, 0, 60, 226, 255, 254, 176, 68, 255, 184, 0, 0],
            &[0, 0, 0, 1, 28, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => Some(&[
            &[0, 0, 0, 201, 169, 2, 30, 241, 98, 0, 0, 0],
            &[0, 0, 0, 105, 255, 244, 254, 230, 19, 0, 0, 0],
            &[0, 0, 0, 0, 75, 144, 125, 24, 0, 0, 0, 0],
            &[0, 30, 108, 108, 4, 0, 0, 47, 108, 95, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 11, 0, 0, 114, 255, 225, 0, 0],
            &[0, 61, 255, 255, 37, 0, 0, 138, 255, 214, 0, 0],
            &[0, 24, 253, 255, 131, 0, 14, 217, 255, 172, 0, 0],
            &[0, 0, 170, 255, 255, 235, 249, 255, 254, 63, 0, 0],
            &[0, 0, 14, 158, 247, 255, 255, 220, 84, 0, 0, 0],
            &[0, 0, 0, 0, 8, 32, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŭ' / 0x16d
        'ŭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 66, 38, 0, 0, 68, 35, 0, 0, 0],
            &[0, 0, 0, 185, 210, 70, 108, 253, 80, 0, 0, 0],
            &[0, 0, 0, 54, 236, 255, 255, 174, 2, 0, 0, 0],
            &[0, 0, 0, 0, 15, 64, 45, 0, 0, 0, 0, 0],
            &[0, 37, 188, 188, 22, 0, 0, 113, 188, 135, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 158, 255, 184, 0, 0],
            &[0, 51, 255, 255, 35, 0, 0, 182, 255, 184, 0, 0],
            &[0, 31, 255, 255, 91, 0, 26, 241, 255, 184, 0, 0],
            &[0, 2, 223, 255, 249, 204, 241, 249, 255, 184, 0, 0],
            &[0, 0, 60, 226, 255, 254, 176, 68, 255, 184, 0, 0],
            &[0, 0, 0, 1, 28, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ů' / 0x16e
        'Ů' => Some(&[
            &[0, 0, 0, 16, 255, 62, 166, 164, 0, 0, 0, 0],
            &[0, 0, 0, 4, 235, 166, 221, 130, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 144, 114, 7, 0, 0, 0, 0],
            &[0, 30, 108, 108, 4, 0, 0, 47, 108, 95, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 11, 0, 0, 114, 255, 225, 0, 0],
            &[0, 61, 255, 255, 37, 0, 0, 138, 255, 214, 0, 0],
            &[0, 24, 253, 255, 131, 0, 14, 217, 255, 172, 0, 0],
            &[0, 0, 170, 255, 255, 235, 249, 255, 254, 63, 0, 0],
            &[0, 0, 14, 158, 247, 255, 255, 220, 84, 0, 0, 0],
            &[0, 0, 0, 0, 8, 32, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 143, 253, 228, 52, 0, 0, 0, 0],
        ]),
        // letter: 'ů' / 0x16f
        'ů' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 82, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 210, 219, 244, 99, 0, 0, 0, 0],
            &[0, 0, 0, 19, 255, 32, 147, 168, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 232, 248, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 64, 36, 0, 0, 0, 0, 0],
            &[0, 37, 188, 188, 22, 0, 0, 113, 188, 135, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 158, 255, 184, 0, 0],
            &[0, 51, 255, 255, 35, 0, 0, 182, 255, 184, 0, 0],
            &[0, 31, 255, 255, 91, 0, 26, 241, 255, 184, 0, 0],
            &[0, 2, 223, 255, 249, 204, 241, 249, 255, 184, 0, 0],
            &[0, 0, 60, 226, 255, 254, 176, 68, 255, 184, 0, 0],
            &[0, 0, 0, 1, 28, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ű' / 0x170
        'Ű' => Some(&[
            &[0, 0, 0, 11, 220, 236, 76, 235, 226, 26, 0, 0],
            &[0, 0, 0, 132, 251, 86, 165, 243, 60, 0, 0, 0],
            &[0, 0, 4, 150, 68, 18, 156, 50, 0, 0, 0, 0],
            &[0, 30, 108, 108, 4, 0, 0, 47, 108, 95, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 11, 0, 0, 114, 255, 225, 0, 0],
            &[0, 61, 255, 255, 37, 0, 0, 138, 255, 214, 0, 0],
            &[0, 24, 253, 255, 131, 0, 14, 217, 255, 172, 0, 0],
            &[0, 0, 170, 255, 255, 235, 249, 255, 254, 63, 0, 0],
            &[0, 0, 14, 158, 247, 255, 255, 220, 84, 0, 0, 0],
            &[0, 0, 0, 0, 8, 32, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ű' / 0x171
        'ű' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 65, 27, 59, 65, 18, 0, 0],
            &[0, 0, 0, 36, 250, 224, 84, 255, 203, 8, 0, 0],
            &[0, 0, 0, 180, 220, 37, 209, 202, 18, 0, 0, 0],
            &[0, 0, 4, 77, 19, 14, 77, 11, 0, 0, 0, 0],
            &[0, 37, 188, 188, 22, 0, 0, 113, 188, 135, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 158, 255, 184, 0, 0],
            &[0, 51, 255, 255, 35, 0, 0, 182, 255, 184, 0, 0],
            &[0, 31, 255, 255, 91, 0, 26, 241, 255, 184, 0, 0],
            &[0, 2, 223, 255, 249, 204, 241, 249, 255, 184, 0, 0],
            &[0, 0, 60, 226, 255, 254, 176, 68, 255, 184, 0, 0],
            &[0, 0, 0, 1, 28, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ų' / 0x172
        'Ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 108, 108, 4, 0, 0, 47, 108, 95, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 10, 0, 0, 112, 255, 225, 0, 0],
            &[0, 71, 255, 255, 11, 0, 0, 114, 255, 225, 0, 0],
            &[0, 61, 255, 255, 37, 0, 0, 138, 255, 214, 0, 0],
            &[0, 24, 253, 255, 131, 0, 14, 217, 255, 172, 0, 0],
            &[0, 0, 170, 255, 255, 235, 249, 255, 254, 63, 0, 0],
            &[0, 0, 14, 158, 247, 255, 255, 255, 84, 0, 0, 0],
            &[0, 0, 0, 0, 15, 223, 223, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 105, 255, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 156, 195, 72, 0, 0, 0, 0],
        ]),
        // letter: 'ų' / 0x173
        'ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 188, 188, 22, 0, 0, 113, 188, 135, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 153, 255, 184, 0, 0],
            &[0, 51, 255, 255, 30, 0, 0, 158, 255, 184, 0, 0],
            &[0, 51, 255, 255, 35, 0, 0, 182, 255, 184, 0, 0],
            &[0, 31, 255, 255, 91, 0, 26, 241, 255, 184, 0, 0],
            &[0, 2, 223, 255, 249, 204, 241, 249, 255, 184, 0, 0],
            &[0, 0, 60, 226, 255, 254, 176, 68, 255, 189, 0, 0],
            &[0, 0, 0, 1, 28, 14, 0, 126, 227, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 27, 253, 123, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 111, 193, 137, 0, 0],
        ]),
        // letter: 'Ŵ' / 0x174
        'Ŵ' => Some(&[
            &[0, 0, 0, 0, 162, 241, 241, 71, 0, 0, 0, 0],
            &[0, 0, 0, 97, 255, 154, 218, 231, 25, 0, 0, 0],
            &[0, 0, 9, 150, 106, 0, 21, 147, 97, 0, 0, 0],
            &[0, 89, 108, 21, 0, 0, 0, 0, 64, 108, 46, 0],
            &[0, 197, 255, 67, 0, 0, 0, 0, 164, 255, 96, 0],
            &[0, 178, 255, 91, 0, 0, 0, 0, 182, 255, 78, 0],
            &[0, 159, 255, 114, 64, 188, 182, 0, 200, 255, 60, 0],
            &[0, 140, 255, 133, 122, 255, 255, 27, 218, 255, 42, 0],
            &[0, 121, 255, 149, 162, 243, 255, 65, 236, 255, 24, 0],
            &[0, 102, 255, 159, 201, 179, 255, 104, 249, 255, 6, 0],
            &[0, 83, 255, 163, 239, 134, 230, 144, 253, 244, 0, 0],
            &[0, 64, 255, 183, 255, 95, 193, 186, 248, 226, 0, 0],
            &[0, 45, 255, 213, 255, 57, 155, 227, 239, 208, 0, 0],
            &[0, 26, 255, 254, 255, 19, 118, 255, 254, 190, 0, 0],
            &[0, 7, 255, 255, 237, 0, 80, 255, 255, 172, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŵ' / 0x175
        'ŵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 65, 65, 6, 0, 0, 0, 0],
            &[0, 0, 0, 6, 207, 255, 255, 134, 0, 0, 0, 0],
            &[0, 0, 0, 152, 247, 80, 153, 253, 73, 0, 0, 0],
            &[0, 0, 6, 77, 40, 0, 0, 65, 58, 0, 0, 0],
            &[0, 169, 188, 36, 0, 0, 0, 0, 102, 188, 97, 0],
            &[0, 199, 255, 68, 25, 116, 99, 0, 159, 255, 100, 0],
            &[0, 162, 255, 91, 93, 255, 249, 6, 181, 255, 64, 0],
            &[0, 126, 255, 114, 145, 255, 255, 50, 204, 255, 28, 0],
            &[0, 90, 255, 137, 196, 200, 255, 101, 227, 246, 1, 0],
            &[0, 54, 255, 162, 243, 135, 237, 150, 247, 212, 0, 0],
            &[0, 18, 255, 210, 255, 89, 192, 207, 255, 176, 0, 0],
            &[0, 0, 238, 255, 255, 41, 146, 255, 255, 140, 0, 0],
            &[0, 0, 202, 255, 247, 3, 100, 255, 255, 104, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŷ' / 0x176
        'Ŷ' => Some(&[
            &[0, 0, 0, 0, 162, 241, 241, 71, 0, 0, 0, 0],
            &[0, 0, 0, 97, 255, 154, 218, 231, 25, 0, 0, 0],
            &[0, 0, 9, 150, 106, 0, 21, 147, 97, 0, 0, 0],
            &[0, 62, 108, 93, 0, 0, 0, 26, 108, 108, 19, 0],
            &[0, 61, 255, 255, 41, 0, 0, 138, 255, 213, 1, 0],
            &[0, 0, 195, 255, 148, 0, 8, 236, 255, 94, 0, 0],
            &[0, 0, 73, 255, 242, 13, 96, 255, 224, 4, 0, 0],
            &[0, 0, 0, 207, 255, 108, 203, 255, 107, 0, 0, 0],
            &[0, 0, 0, 86, 255, 236, 255, 233, 9, 0, 0, 0],
            &[0, 0, 0, 2, 217, 255, 255, 121, 0, 0, 0, 0],
            &[0, 0, 0, 0, 109, 255, 252, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŷ' / 0x177
        'ŷ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 65, 65, 4, 0, 0, 0, 0],
            &[0, 0, 0, 9, 212, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 160, 245, 75, 160, 252, 66, 0, 0, 0],
            &[0, 0, 8, 77, 37, 0, 0, 67, 55, 0, 0, 0],
            &[0, 100, 188, 179, 3, 0, 0, 61, 188, 188, 27, 0],
            &[0, 48, 255, 255, 66, 0, 0, 148, 255, 212, 0, 0],
            &[0, 0, 202, 255, 152, 0, 0, 224, 255, 119, 0, 0],
            &[0, 0, 101, 255, 234, 3, 44, 255, 253, 28, 0, 0],
            &[0, 0, 12, 243, 255, 67, 120, 255, 189, 0, 0, 0],
            &[0, 0, 0, 154, 255, 150, 195, 255, 96, 0, 0, 0],
            &[0, 0, 0, 52, 255, 219, 250, 245, 13, 0, 0, 0],
            &[0, 0, 0, 0, 207, 255, 255, 166, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 255, 255, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 255, 230, 3, 0, 0, 0, 0],
            &[0, 2, 32, 76, 237, 255, 122, 0, 0, 0, 0, 0],
            &[0, 11, 222, 224, 160, 23, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ÿ' / 0x178
        'Ÿ' => Some(&[
            &[0, 0, 10, 166, 143, 0, 45, 180, 96, 0, 0, 0],
            &[0, 0, 36, 255, 254, 9, 105, 255, 198, 0, 0, 0],
            &[0, 0, 0, 99, 78, 0, 14, 115, 47, 0, 0, 0],
            &[0, 62, 108, 93, 0, 0, 0, 26, 108, 108, 19, 0],
            &[0, 61, 255, 255, 41, 0, 0, 138, 255, 213, 1, 0],
            &[0, 0, 195, 255, 148, 0, 8, 236, 255, 94, 0, 0],
            &[0, 0, 73, 255, 242, 13, 96, 255, 224, 4, 0, 0],
            &[0, 0, 0, 207, 255, 108, 203, 255, 107, 0, 0, 0],
            &[0, 0, 0, 86, 255, 236, 255, 233, 9, 0, 0, 0],
            &[0, 0, 0, 2, 217, 255, 255, 121, 0, 0, 0, 0],
            &[0, 0, 0, 0, 109, 255, 252, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ź' / 0x179
        'Ź' => Some(&[
            &[0, 0, 0, 0, 0, 2, 194, 241, 163, 0, 0, 0],
            &[0, 0, 0, 0, 0, 118, 255, 170, 6, 0, 0, 0],
            &[0, 0, 0, 0, 8, 151, 121, 2, 0, 0, 0, 0],
            &[0, 0, 34, 108, 108, 108, 108, 108, 108, 48, 0, 0],
            &[0, 0, 81, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 44, 139, 139, 139, 171, 255, 255, 60, 0, 0],
            &[0, 0, 0, 0, 0, 0, 178, 255, 166, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 244, 27, 0, 0, 0],
            &[0, 0, 0, 0, 4, 214, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 219, 6, 0, 0, 0, 0],
            &[0, 0, 0, 21, 239, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 153, 255, 180, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 253, 249, 37, 0, 0, 0, 0, 0, 0],
            &[0, 0, 170, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ź' / 0x17a
        'ź' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 43, 65, 59, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 242, 252, 91, 0, 0, 0],
            &[0, 0, 0, 0, 2, 189, 246, 81, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 77, 41, 0, 0, 0, 0, 0],
            &[0, 0, 135, 188, 188, 188, 188, 188, 188, 78, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 102, 0, 0],
            &[0, 0, 16, 22, 22, 32, 218, 255, 207, 10, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 233, 30, 0, 0, 0],
            &[0, 0, 0, 0, 118, 255, 249, 59, 0, 0, 0, 0],
            &[0, 0, 0, 71, 253, 255, 97, 0, 0, 0, 0, 0],
            &[0, 0, 36, 239, 255, 142, 0, 0, 0, 0, 0, 0],
            &[0, 0, 209, 255, 255, 214, 210, 210, 210, 118, 0, 0],
            &[0, 0, 245, 255, 255, 255, 255, 255, 255, 143, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ż' / 0x17b
        'Ż' => Some(&[
            &[0, 0, 0, 0, 55, 247, 246, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 255, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 125, 120, 6, 0, 0, 0, 0],
            &[0, 0, 34, 108, 108, 108, 108, 108, 108, 48, 0, 0],
            &[0, 0, 81, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 44, 139, 139, 139, 171, 255, 255, 60, 0, 0],
            &[0, 0, 0, 0, 0, 0, 178, 255, 166, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 244, 27, 0, 0, 0],
            &[0, 0, 0, 0, 4, 214, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 219, 6, 0, 0, 0, 0],
            &[0, 0, 0, 21, 239, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 153, 255, 180, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 253, 249, 37, 0, 0, 0, 0, 0, 0],
            &[0, 0, 170, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 9, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ż' / 0x17c
        'ż' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 86, 74, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 102, 255, 255, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 253, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 37, 0, 0, 0, 0, 0],
            &[0, 0, 135, 188, 188, 188, 188, 188, 188, 78, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 102, 0, 0],
            &[0, 0, 16, 22, 22, 32, 218, 255, 207, 10, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 233, 30, 0, 0, 0],
            &[0, 0, 0, 0, 118, 255, 249, 59, 0, 0, 0, 0],
            &[0, 0, 0, 71, 253, 255, 97, 0, 0, 0, 0, 0],
            &[0, 0, 36, 239, 255, 142, 0, 0, 0, 0, 0, 0],
            &[0, 0, 209, 255, 255, 214, 210, 210, 210, 118, 0, 0],
            &[0, 0, 245, 255, 255, 255, 255, 255, 255, 143, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ž' / 0x17d
        'Ž' => Some(&[
            &[0, 0, 0, 166, 214, 41, 47, 217, 164, 0, 0, 0],
            &[0, 0, 0, 16, 223, 234, 238, 222, 16, 0, 0, 0],
            &[0, 0, 0, 0, 59, 157, 157, 57, 0, 0, 0, 0],
            &[0, 0, 34, 108, 108, 108, 108, 108, 108, 48, 0, 0],
            &[0, 0, 81, 255, 255, 255, 255, 255, 255, 114, 0, 0],
            &[0, 0, 44, 139, 139, 139, 171, 255, 255, 60, 0, 0],
            &[0, 0, 0, 0, 0, 0, 178, 255, 166, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 244, 27, 0, 0, 0],
            &[0, 0, 0, 0, 4, 214, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 219, 6, 0, 0, 0, 0],
            &[0, 0, 0, 21, 239, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 153, 255, 180, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 253, 249, 37, 0, 0, 0, 0, 0, 0],
            &[0, 0, 170, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 122, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ž' / 0x17e
        'ž' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 65, 38, 0, 0, 48, 55, 0, 0, 0],
            &[0, 0, 0, 137, 249, 87, 124, 254, 103, 0, 0, 0],
            &[0, 0, 0, 3, 195, 254, 255, 164, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 77, 77, 15, 0, 0, 0, 0],
            &[0, 0, 135, 188, 188, 188, 188, 188, 188, 78, 0, 0],
            &[0, 0, 184, 255, 255, 255, 255, 255, 255, 102, 0, 0],
            &[0, 0, 16, 22, 22, 32, 218, 255, 207, 10, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 233, 30, 0, 0, 0],
            &[0, 0, 0, 0, 118, 255, 249, 59, 0, 0, 0, 0],
            &[0, 0, 0, 71, 253, 255, 97, 0, 0, 0, 0, 0],
            &[0, 0, 36, 239, 255, 142, 0, 0, 0, 0, 0, 0],
            &[0, 0, 209, 255, 255, 214, 210, 210, 210, 118, 0, 0],
            &[0, 0, 245, 255, 255, 255, 255, 255, 255, 143, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ſ' / 0x17f
        'ſ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 44, 54, 25, 0, 0, 0, 0],
            &[0, 0, 0, 41, 224, 255, 255, 255, 53, 0, 0, 0],
            &[0, 0, 0, 166, 255, 250, 191, 195, 3, 0, 0, 0],
            &[0, 0, 0, 203, 255, 145, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 208, 255, 120, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 208, 255, 120, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 208, 255, 120, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 208, 255, 120, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 208, 255, 120, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 208, 255, 120, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 208, 255, 120, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 208, 255, 120, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 208, 255, 120, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        _ => None,
    }
}
