//! Module for letters with the font weight bold and size 16.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 16;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 10;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight bold and font size 14px
#[inline]
pub const fn get_char(c: char) -> Option<&'static [&'static [u8]]> {
    match c {
        // letter: ' ' / 0x20
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '!' / 0x21
        '!' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 243, 254, 93, 0, 0, 0],
            &[0, 0, 0, 0, 232, 255, 82, 0, 0, 0],
            &[0, 0, 0, 0, 221, 255, 71, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 60, 0, 0, 0],
            &[0, 0, 0, 0, 199, 255, 49, 0, 0, 0],
            &[0, 0, 0, 0, 187, 255, 37, 0, 0, 0],
            &[0, 0, 0, 0, 121, 174, 19, 0, 0, 0],
            &[0, 0, 0, 0, 37, 65, 1, 0, 0, 0],
            &[0, 0, 0, 6, 243, 255, 91, 0, 0, 0],
            &[0, 0, 0, 2, 223, 253, 69, 0, 0, 0],
            &[0, 0, 0, 0, 8, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '"' / 0x22
        '"' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 254, 185, 83, 254, 138, 0, 0],
            &[0, 0, 17, 255, 167, 65, 255, 120, 0, 0],
            &[0, 0, 1, 253, 148, 46, 255, 101, 0, 0],
            &[0, 0, 0, 147, 82, 19, 157, 53, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '#' / 0x23
        '#' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 254, 83, 119, 237, 0, 0, 0],
            &[0, 0, 65, 255, 40, 160, 197, 0, 0, 0],
            &[23, 60, 139, 253, 65, 211, 183, 60, 11, 0],
            &[98, 255, 255, 255, 255, 255, 255, 255, 48, 0],
            &[14, 37, 200, 183, 65, 255, 96, 37, 7, 0],
            &[34, 44, 231, 155, 101, 255, 71, 34, 0, 0],
            &[202, 255, 255, 255, 255, 255, 255, 200, 0, 0],
            &[40, 100, 255, 83, 182, 212, 51, 40, 0, 0],
            &[0, 97, 253, 7, 202, 163, 0, 0, 0, 0],
            &[0, 137, 220, 1, 243, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '$' / 0x24
        '$' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 35, 110, 0, 0, 0, 0, 0],
            &[0, 0, 70, 172, 228, 144, 79, 4, 0, 0],
            &[0, 106, 255, 253, 250, 254, 239, 7, 0, 0],
            &[0, 207, 246, 74, 176, 16, 58, 0, 0, 0],
            &[0, 195, 254, 144, 176, 0, 0, 0, 0, 0],
            &[0, 57, 236, 255, 236, 115, 9, 0, 0, 0],
            &[0, 0, 14, 141, 255, 255, 203, 4, 0, 0],
            &[0, 0, 0, 56, 182, 173, 255, 59, 0, 0],
            &[0, 133, 63, 68, 188, 184, 255, 40, 0, 0],
            &[0, 220, 255, 255, 255, 255, 157, 0, 0, 0],
            &[0, 21, 89, 149, 211, 63, 0, 0, 0, 0],
            &[0, 0, 0, 45, 143, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '%' / 0x25
        '%' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 7, 0, 0, 0, 0, 0, 0, 0],
            &[93, 249, 250, 96, 0, 81, 253, 47, 0, 0],
            &[225, 153, 153, 227, 1, 210, 171, 0, 0, 0],
            &[245, 107, 107, 247, 87, 253, 43, 0, 0, 0],
            &[186, 218, 218, 192, 214, 166, 0, 0, 0, 0],
            &[22, 137, 137, 116, 252, 39, 0, 0, 0, 0],
            &[0, 0, 4, 219, 162, 92, 146, 67, 0, 0],
            &[0, 0, 98, 251, 117, 252, 190, 251, 35, 0],
            &[0, 5, 223, 156, 143, 210, 8, 254, 91, 0],
            &[0, 104, 250, 32, 126, 232, 67, 255, 75, 0],
            &[7, 227, 151, 0, 27, 221, 255, 193, 5, 0],
            &[0, 0, 0, 0, 0, 1, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '&' / 0x26
        '&' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 20, 1, 0, 0, 0, 0, 0],
            &[0, 42, 224, 255, 229, 53, 0, 0, 0, 0],
            &[0, 166, 253, 93, 242, 179, 0, 0, 0, 0],
            &[0, 185, 245, 6, 236, 181, 0, 0, 0, 0],
            &[0, 128, 255, 248, 255, 81, 0, 0, 0, 0],
            &[0, 37, 255, 255, 163, 1, 71, 54, 0, 0],
            &[5, 205, 255, 255, 194, 50, 255, 160, 0, 0],
            &[102, 255, 134, 140, 255, 255, 255, 88, 0, 0],
            &[142, 255, 75, 6, 243, 255, 255, 5, 0, 0],
            &[103, 255, 209, 148, 255, 255, 255, 31, 0, 0],
            &[7, 178, 255, 255, 216, 170, 255, 201, 6, 0],
            &[0, 0, 19, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ''' / 0x27
        '\'' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 189, 254, 32, 0, 0, 0, 0],
            &[0, 0, 0, 171, 255, 14, 0, 0, 0, 0],
            &[0, 0, 0, 152, 250, 0, 0, 0, 0, 0],
            &[0, 0, 0, 84, 145, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '(' / 0x28
        '(' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 236, 19, 0, 0, 0],
            &[0, 0, 0, 68, 255, 116, 0, 0, 0, 0],
            &[0, 0, 0, 177, 249, 19, 0, 0, 0, 0],
            &[0, 0, 6, 246, 195, 0, 0, 0, 0, 0],
            &[0, 0, 50, 255, 143, 0, 0, 0, 0, 0],
            &[0, 0, 73, 255, 120, 0, 0, 0, 0, 0],
            &[0, 0, 80, 255, 114, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 133, 0, 0, 0, 0, 0],
            &[0, 0, 21, 255, 178, 0, 0, 0, 0, 0],
            &[0, 0, 0, 211, 238, 6, 0, 0, 0, 0],
            &[0, 0, 0, 110, 255, 85, 0, 0, 0, 0],
            &[0, 0, 0, 9, 219, 208, 2, 0, 0, 0],
            &[0, 0, 0, 0, 37, 79, 15, 0, 0, 0],
        ]),
        // letter: ')' / 0x29
        ')' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 153, 244, 33, 0, 0, 0, 0],
            &[0, 0, 0, 24, 249, 175, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 253, 31, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 103, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 156, 0, 0, 0],
            &[0, 0, 0, 0, 17, 255, 177, 0, 0, 0],
            &[0, 0, 0, 0, 13, 255, 181, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 162, 0, 0, 0],
            &[0, 0, 0, 0, 75, 255, 118, 0, 0, 0],
            &[0, 0, 0, 0, 139, 255, 52, 0, 0, 0],
            &[0, 0, 0, 4, 230, 207, 0, 0, 0, 0],
            &[0, 0, 0, 108, 255, 72, 0, 0, 0, 0],
            &[0, 0, 0, 64, 68, 0, 0, 0, 0, 0],
        ]),
        // letter: '*' / 0x2a
        '*' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 145, 1, 0, 0, 0, 0],
            &[0, 0, 0, 134, 239, 0, 0, 0, 0, 0],
            &[0, 141, 142, 166, 230, 114, 185, 29, 0, 0],
            &[0, 155, 208, 252, 255, 228, 192, 49, 0, 0],
            &[0, 0, 76, 252, 222, 178, 1, 0, 0, 0],
            &[0, 10, 231, 167, 70, 255, 91, 0, 0, 0],
            &[0, 0, 36, 31, 0, 70, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '+' / 0x2b
        '+' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 248, 0, 0, 0, 0],
            &[0, 3, 175, 175, 217, 253, 175, 175, 73, 0],
            &[0, 4, 204, 204, 231, 254, 204, 204, 85, 0],
            &[0, 0, 0, 0, 135, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ',' / 0x2c
        ',' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 231, 92, 0, 0, 0],
            &[0, 0, 0, 0, 194, 255, 37, 0, 0, 0],
            &[0, 0, 0, 0, 238, 215, 0, 0, 0, 0],
            &[0, 0, 0, 34, 255, 124, 0, 0, 0, 0],
            &[0, 0, 0, 12, 47, 11, 0, 0, 0, 0],
        ]),
        // letter: '-' / 0x2d
        '-' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 142, 158, 158, 158, 47, 0, 0, 0],
            &[0, 0, 230, 255, 255, 255, 76, 0, 0, 0],
            &[0, 0, 29, 33, 33, 33, 9, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '.' / 0x2e
        '.' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 65, 1, 0, 0, 0],
            &[0, 0, 0, 6, 243, 255, 91, 0, 0, 0],
            &[0, 0, 0, 2, 223, 253, 69, 0, 0, 0],
            &[0, 0, 0, 0, 8, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '/' / 0x2f
        '/' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 13, 0, 0, 0],
            &[0, 0, 0, 0, 7, 238, 202, 0, 0, 0],
            &[0, 0, 0, 0, 87, 255, 105, 0, 0, 0],
            &[0, 0, 0, 0, 184, 247, 16, 0, 0, 0],
            &[0, 0, 0, 28, 253, 166, 0, 0, 0, 0],
            &[0, 0, 0, 123, 255, 69, 0, 0, 0, 0],
            &[0, 0, 1, 220, 226, 2, 0, 0, 0, 0],
            &[0, 0, 62, 255, 131, 0, 0, 0, 0, 0],
            &[0, 0, 160, 254, 35, 0, 0, 0, 0, 0],
            &[0, 13, 244, 193, 0, 0, 0, 0, 0, 0],
            &[0, 99, 255, 95, 0, 0, 0, 0, 0, 0],
            &[0, 8, 14, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '0' / 0x30
        '0' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 22, 0, 0, 0, 0],
            &[0, 0, 13, 176, 255, 255, 218, 57, 0, 0],
            &[0, 0, 158, 255, 205, 166, 255, 235, 12, 0],
            &[0, 7, 249, 244, 14, 48, 255, 255, 96, 0],
            &[0, 50, 255, 198, 1, 198, 255, 255, 149, 0],
            &[0, 69, 255, 179, 98, 255, 158, 255, 170, 0],
            &[0, 68, 255, 192, 235, 169, 82, 255, 174, 0],
            &[0, 47, 255, 255, 242, 26, 100, 255, 154, 0],
            &[0, 5, 255, 255, 122, 0, 158, 255, 105, 0],
            &[0, 0, 146, 255, 200, 154, 253, 242, 20, 0],
            &[0, 0, 9, 167, 254, 255, 227, 70, 0, 0],
            &[0, 0, 0, 0, 11, 26, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '1' / 0x31
        '1' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 64, 200, 254, 138, 0, 0, 0],
            &[0, 0, 176, 255, 255, 255, 139, 0, 0, 0],
            &[0, 0, 136, 216, 173, 255, 139, 0, 0, 0],
            &[0, 0, 9, 4, 121, 255, 139, 0, 0, 0],
            &[0, 0, 0, 0, 124, 255, 139, 0, 0, 0],
            &[0, 0, 0, 0, 124, 255, 139, 0, 0, 0],
            &[0, 0, 0, 0, 124, 255, 139, 0, 0, 0],
            &[0, 0, 0, 0, 124, 255, 139, 0, 0, 0],
            &[0, 0, 46, 131, 255, 255, 255, 132, 47, 0],
            &[0, 0, 106, 255, 255, 255, 255, 255, 107, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '2' / 0x32
        '2' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 27, 3, 0, 0, 0],
            &[0, 0, 67, 204, 255, 255, 237, 106, 0, 0],
            &[0, 13, 233, 254, 197, 184, 255, 255, 59, 0],
            &[0, 0, 58, 65, 0, 0, 181, 255, 120, 0],
            &[0, 0, 0, 0, 0, 0, 199, 255, 94, 0],
            &[0, 0, 0, 0, 0, 81, 255, 230, 13, 0],
            &[0, 0, 0, 0, 59, 244, 246, 62, 0, 0],
            &[0, 0, 0, 57, 242, 243, 64, 0, 0, 0],
            &[0, 0, 57, 242, 232, 51, 0, 0, 0, 0],
            &[0, 29, 242, 255, 217, 188, 188, 188, 145, 0],
            &[0, 51, 255, 255, 255, 255, 255, 255, 197, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '3' / 0x33
        '3' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 27, 1, 0, 0, 0],
            &[0, 0, 115, 221, 255, 255, 232, 90, 0, 0],
            &[0, 0, 159, 244, 178, 178, 255, 249, 18, 0],
            &[0, 0, 15, 17, 0, 0, 213, 255, 34, 0],
            &[0, 0, 0, 0, 2, 57, 243, 208, 1, 0],
            &[0, 0, 0, 133, 250, 255, 196, 28, 0, 0],
            &[0, 0, 0, 94, 180, 219, 255, 192, 11, 0],
            &[0, 0, 0, 0, 0, 2, 191, 255, 94, 0],
            &[0, 0, 14, 0, 0, 0, 182, 255, 111, 0],
            &[0, 1, 247, 186, 154, 186, 255, 251, 40, 0],
            &[0, 0, 189, 249, 255, 255, 219, 78, 0, 0],
            &[0, 0, 0, 4, 26, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '4' / 0x34
        '4' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 254, 140, 0, 0],
            &[0, 0, 0, 0, 74, 255, 255, 141, 0, 0],
            &[0, 0, 0, 11, 224, 231, 255, 141, 0, 0],
            &[0, 0, 0, 143, 246, 118, 255, 141, 0, 0],
            &[0, 0, 53, 252, 125, 95, 255, 141, 0, 0],
            &[0, 4, 208, 220, 7, 97, 255, 141, 0, 0],
            &[0, 95, 255, 219, 181, 209, 255, 222, 152, 0],
            &[0, 104, 237, 237, 237, 244, 255, 247, 199, 0],
            &[0, 0, 0, 0, 0, 97, 255, 141, 0, 0],
            &[0, 0, 0, 0, 0, 97, 255, 141, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '5' / 0x35
        '5' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 105, 254, 254, 254, 254, 204, 0, 0, 0],
            &[0, 125, 255, 218, 189, 189, 152, 0, 0, 0],
            &[0, 145, 255, 98, 0, 0, 0, 0, 0, 0],
            &[0, 165, 255, 133, 84, 36, 0, 0, 0, 0],
            &[0, 184, 255, 255, 255, 254, 123, 0, 0, 0],
            &[0, 57, 105, 80, 150, 255, 254, 33, 0, 0],
            &[0, 0, 0, 0, 0, 224, 255, 80, 0, 0],
            &[0, 18, 0, 0, 6, 230, 255, 64, 0, 0],
            &[0, 235, 195, 167, 213, 255, 218, 4, 0, 0],
            &[0, 180, 253, 255, 252, 183, 33, 0, 0, 0],
            &[0, 0, 7, 29, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '6' / 0x36
        '6' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 21, 15, 0, 0],
            &[0, 0, 0, 43, 178, 247, 255, 237, 0, 0],
            &[0, 0, 43, 242, 255, 198, 149, 145, 0, 0],
            &[0, 0, 180, 255, 126, 0, 0, 0, 0, 0],
            &[0, 9, 251, 241, 33, 110, 100, 20, 0, 0],
            &[0, 44, 255, 225, 237, 255, 255, 228, 18, 0],
            &[0, 55, 255, 255, 127, 48, 202, 255, 115, 0],
            &[0, 39, 255, 239, 1, 0, 114, 255, 152, 0],
            &[0, 3, 239, 247, 18, 0, 144, 255, 136, 0],
            &[0, 0, 135, 255, 219, 175, 252, 254, 53, 0],
            &[0, 0, 5, 156, 251, 255, 238, 103, 0, 0],
            &[0, 0, 0, 0, 13, 31, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '7' / 0x37
        '7' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 75, 251, 251, 251, 251, 251, 251, 175, 0],
            &[0, 56, 189, 189, 189, 189, 229, 255, 155, 0],
            &[0, 0, 0, 0, 0, 3, 222, 255, 50, 0],
            &[0, 0, 0, 0, 0, 86, 255, 194, 0, 0],
            &[0, 0, 0, 0, 0, 202, 255, 82, 0, 0],
            &[0, 0, 0, 0, 63, 255, 223, 3, 0, 0],
            &[0, 0, 0, 0, 179, 255, 114, 0, 0, 0],
            &[0, 0, 0, 41, 254, 243, 15, 0, 0, 0],
            &[0, 0, 0, 156, 255, 147, 0, 0, 0, 0],
            &[0, 0, 24, 248, 254, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '8' / 0x38
        '8' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 24, 1, 0, 0, 0],
            &[0, 0, 29, 189, 254, 255, 231, 94, 0, 0],
            &[0, 0, 177, 255, 206, 163, 254, 252, 29, 0],
            &[0, 0, 209, 255, 51, 0, 204, 255, 57, 0],
            &[0, 0, 137, 255, 177, 100, 252, 224, 8, 0],
            &[0, 0, 8, 188, 255, 255, 227, 35, 0, 0],
            &[0, 0, 85, 247, 240, 223, 255, 154, 0, 0],
            &[0, 6, 241, 247, 38, 11, 198, 255, 86, 0],
            &[0, 34, 255, 217, 0, 0, 116, 255, 136, 0],
            &[0, 7, 235, 255, 171, 141, 236, 255, 80, 0],
            &[0, 0, 60, 212, 255, 255, 236, 114, 0, 0],
            &[0, 0, 0, 0, 19, 28, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '9' / 0x39
        '9' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 26, 0, 0, 0, 0],
            &[0, 0, 35, 201, 255, 255, 212, 47, 0, 0],
            &[0, 0, 204, 255, 209, 187, 255, 228, 11, 0],
            &[0, 33, 255, 239, 8, 0, 168, 255, 91, 0],
            &[0, 50, 255, 216, 0, 0, 137, 255, 142, 0],
            &[0, 17, 252, 253, 81, 64, 229, 255, 157, 0],
            &[0, 0, 145, 255, 255, 255, 209, 255, 146, 0],
            &[0, 0, 0, 75, 117, 70, 142, 255, 107, 0],
            &[0, 0, 0, 0, 0, 33, 240, 252, 31, 0],
            &[0, 0, 86, 136, 164, 241, 255, 136, 0, 0],
            &[0, 0, 142, 255, 255, 222, 110, 1, 0, 0],
            &[0, 0, 11, 33, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ':' / 0x3a
        ':' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 143, 183, 33, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 107, 0, 0, 0],
            &[0, 0, 0, 0, 113, 155, 21, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 65, 1, 0, 0, 0],
            &[0, 0, 0, 6, 243, 255, 91, 0, 0, 0],
            &[0, 0, 0, 2, 223, 253, 69, 0, 0, 0],
            &[0, 0, 0, 0, 8, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ';' / 0x3b
        ';' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 191, 83, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 185, 0, 0, 0],
            &[0, 0, 0, 0, 64, 164, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 231, 92, 0, 0, 0],
            &[0, 0, 0, 0, 194, 255, 37, 0, 0, 0],
            &[0, 0, 0, 0, 238, 215, 0, 0, 0, 0],
            &[0, 0, 0, 34, 255, 124, 0, 0, 0, 0],
            &[0, 0, 0, 12, 47, 11, 0, 0, 0, 0],
        ]),
        // letter: '<' / 0x3c
        '<' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 49, 26, 0, 0],
            &[0, 0, 0, 0, 42, 171, 254, 53, 0, 0],
            &[0, 0, 35, 163, 253, 231, 114, 8, 0, 0],
            &[0, 135, 251, 229, 110, 9, 0, 0, 0, 0],
            &[0, 151, 254, 225, 111, 11, 0, 0, 0, 0],
            &[0, 0, 45, 168, 253, 237, 129, 12, 0, 0],
            &[0, 0, 0, 0, 39, 160, 251, 53, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 21, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '=' / 0x3d
        '=' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 28, 28, 28, 28, 28, 12, 0],
            &[0, 5, 255, 255, 255, 255, 255, 255, 107, 0],
            &[0, 2, 145, 145, 145, 145, 145, 145, 61, 0],
            &[0, 0, 44, 44, 44, 44, 44, 44, 18, 0],
            &[0, 5, 255, 255, 255, 255, 255, 255, 107, 0],
            &[0, 2, 129, 129, 129, 129, 129, 129, 54, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '>' / 0x3e
        '>' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 71, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 207, 220, 92, 2, 0, 0, 0, 0, 0],
            &[0, 61, 190, 255, 213, 84, 1, 0, 0, 0],
            &[0, 0, 0, 60, 186, 255, 206, 26, 0, 0],
            &[0, 0, 0, 63, 182, 255, 221, 31, 0, 0],
            &[0, 75, 201, 255, 215, 93, 3, 0, 0, 0],
            &[0, 207, 209, 86, 2, 0, 0, 0, 0, 0],
            &[0, 54, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '?' / 0x3f
        '?' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 28, 3, 0, 0, 0, 0],
            &[0, 99, 215, 255, 255, 237, 98, 0, 0, 0],
            &[0, 126, 236, 165, 165, 253, 252, 30, 0, 0],
            &[0, 9, 8, 0, 0, 198, 255, 60, 0, 0],
            &[0, 0, 0, 0, 67, 249, 237, 16, 0, 0],
            &[0, 0, 0, 91, 251, 235, 60, 0, 0, 0],
            &[0, 0, 4, 243, 235, 32, 0, 0, 0, 0],
            &[0, 0, 17, 174, 124, 0, 0, 0, 0, 0],
            &[0, 0, 1, 65, 37, 0, 0, 0, 0, 0],
            &[0, 0, 90, 255, 243, 7, 0, 0, 0, 0],
            &[0, 0, 69, 253, 222, 3, 0, 0, 0, 0],
            &[0, 0, 0, 20, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '@' / 0x40
        '@' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 108, 214, 246, 215, 97, 0, 0, 0],
            &[0, 146, 239, 105, 51, 112, 249, 104, 0, 0],
            &[57, 255, 68, 19, 62, 29, 122, 238, 6, 0],
            &[159, 197, 44, 240, 254, 255, 84, 255, 64, 0],
            &[219, 131, 164, 200, 56, 255, 47, 252, 93, 0],
            &[247, 103, 208, 133, 63, 255, 32, 253, 87, 0],
            &[245, 105, 197, 147, 104, 255, 65, 255, 45, 0],
            &[217, 140, 121, 248, 236, 234, 238, 195, 0, 0],
            &[144, 222, 8, 97, 65, 36, 102, 18, 0, 0],
            &[25, 239, 167, 21, 0, 14, 79, 0, 0, 0],
            &[0, 45, 204, 255, 242, 255, 207, 0, 0, 0],
            &[0, 0, 0, 31, 55, 33, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'A' / 0x41
        'A' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 9, 9, 4, 0, 0, 0, 0],
            &[0, 0, 63, 255, 255, 162, 0, 0, 0, 0],
            &[0, 0, 134, 255, 241, 232, 1, 0, 0, 0],
            &[0, 0, 204, 245, 150, 255, 48, 0, 0, 0],
            &[0, 20, 254, 188, 85, 255, 118, 0, 0, 0],
            &[0, 89, 255, 124, 24, 255, 189, 0, 0, 0],
            &[0, 159, 255, 100, 40, 245, 249, 11, 0, 0],
            &[0, 229, 255, 255, 255, 255, 255, 75, 0, 0],
            &[44, 255, 255, 133, 133, 205, 255, 145, 0, 0],
            &[114, 255, 124, 0, 0, 33, 255, 216, 0, 0],
            &[184, 255, 61, 0, 0, 0, 227, 255, 31, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'B' / 0x42
        'B' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 224, 254, 254, 247, 219, 128, 2, 0, 0],
            &[0, 225, 255, 172, 182, 247, 255, 103, 0, 0],
            &[0, 225, 255, 2, 0, 128, 255, 151, 0, 0],
            &[0, 225, 255, 18, 25, 174, 255, 106, 0, 0],
            &[0, 225, 255, 255, 255, 255, 157, 4, 0, 0],
            &[0, 225, 255, 169, 179, 242, 248, 76, 0, 0],
            &[0, 225, 255, 2, 0, 96, 255, 187, 0, 0],
            &[0, 225, 255, 2, 0, 98, 255, 202, 0, 0],
            &[0, 225, 255, 171, 177, 243, 255, 124, 0, 0],
            &[0, 225, 255, 255, 253, 227, 134, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'C' / 0x43
        'C' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 28, 5, 0, 0],
            &[0, 0, 0, 61, 199, 254, 255, 250, 138, 0],
            &[0, 0, 60, 249, 255, 200, 171, 218, 105, 0],
            &[0, 0, 202, 255, 126, 0, 0, 0, 3, 0],
            &[0, 27, 255, 242, 8, 0, 0, 0, 0, 0],
            &[0, 64, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 69, 255, 200, 0, 0, 0, 0, 0, 0],
            &[0, 45, 255, 233, 0, 0, 0, 0, 0, 0],
            &[0, 2, 234, 255, 92, 0, 0, 0, 13, 0],
            &[0, 0, 110, 255, 253, 192, 173, 220, 114, 0],
            &[0, 0, 0, 113, 233, 255, 255, 232, 79, 0],
            &[0, 0, 0, 0, 1, 23, 21, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'D' / 0x44
        'D' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 254, 254, 253, 232, 162, 28, 0, 0],
            &[0, 94, 255, 230, 187, 227, 255, 229, 20, 0],
            &[0, 94, 255, 161, 0, 3, 179, 255, 139, 0],
            &[0, 94, 255, 161, 0, 0, 62, 255, 213, 0],
            &[0, 94, 255, 161, 0, 0, 27, 255, 241, 0],
            &[0, 94, 255, 161, 0, 0, 34, 255, 237, 0],
            &[0, 94, 255, 161, 0, 0, 74, 255, 205, 0],
            &[0, 94, 255, 161, 0, 12, 195, 255, 124, 0],
            &[0, 94, 255, 229, 191, 237, 255, 215, 11, 0],
            &[0, 94, 255, 255, 253, 227, 148, 18, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'E' / 0x45
        'E' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 224, 254, 254, 254, 254, 254, 107, 0, 0],
            &[0, 225, 255, 197, 192, 192, 192, 81, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 35, 15, 15, 15, 3, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 53, 0, 0],
            &[0, 225, 255, 182, 176, 176, 176, 36, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 197, 191, 191, 191, 80, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'F' / 0x46
        'F' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 224, 254, 254, 254, 254, 254, 107, 0, 0],
            &[0, 225, 255, 197, 192, 192, 192, 81, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 171, 164, 164, 164, 34, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 53, 0, 0],
            &[0, 225, 255, 45, 27, 27, 27, 5, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'G' / 0x47
        'G' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 18, 0, 0, 0],
            &[0, 0, 1, 118, 232, 255, 255, 235, 99, 0],
            &[0, 0, 138, 255, 250, 181, 168, 223, 74, 0],
            &[0, 30, 252, 250, 54, 0, 0, 2, 1, 0],
            &[0, 103, 255, 172, 0, 0, 0, 0, 0, 0],
            &[0, 134, 255, 125, 0, 72, 160, 160, 123, 0],
            &[0, 139, 255, 122, 0, 115, 255, 255, 197, 0],
            &[0, 113, 255, 161, 0, 5, 28, 255, 197, 0],
            &[0, 45, 255, 244, 33, 0, 16, 255, 197, 0],
            &[0, 0, 168, 255, 238, 162, 170, 255, 197, 0],
            &[0, 0, 7, 143, 243, 255, 255, 240, 135, 0],
            &[0, 0, 0, 0, 2, 26, 21, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'H' / 0x48
        'H' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 254, 168, 0, 0, 66, 254, 196, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 176, 23, 23, 83, 255, 197, 0],
            &[0, 94, 255, 255, 255, 255, 255, 255, 197, 0],
            &[0, 94, 255, 226, 168, 168, 191, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'I' / 0x49
        'I' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 188, 254, 254, 254, 254, 254, 35, 0, 0],
            &[0, 84, 139, 235, 255, 164, 121, 14, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 83, 138, 235, 255, 163, 120, 14, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'J' / 0x4a
        'J' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 254, 219, 0, 0],
            &[0, 0, 0, 0, 0, 32, 255, 220, 0, 0],
            &[0, 0, 0, 0, 0, 32, 255, 220, 0, 0],
            &[0, 0, 0, 0, 0, 32, 255, 220, 0, 0],
            &[0, 0, 0, 0, 0, 32, 255, 220, 0, 0],
            &[0, 0, 0, 0, 0, 32, 255, 220, 0, 0],
            &[0, 0, 0, 0, 0, 36, 255, 217, 0, 0],
            &[0, 0, 0, 0, 0, 88, 255, 195, 0, 0],
            &[0, 0, 170, 175, 175, 243, 255, 117, 0, 0],
            &[0, 0, 192, 255, 255, 246, 150, 5, 0, 0],
            &[0, 0, 0, 20, 29, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'K' / 0x4b
        'K' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 254, 157, 0, 0, 161, 254, 145, 0],
            &[0, 94, 255, 158, 0, 76, 255, 213, 8, 0],
            &[0, 94, 255, 158, 17, 230, 249, 48, 0, 0],
            &[0, 94, 255, 158, 160, 255, 117, 0, 0, 0],
            &[0, 94, 255, 212, 255, 234, 3, 0, 0, 0],
            &[0, 94, 255, 255, 254, 255, 88, 0, 0, 0],
            &[0, 94, 255, 210, 78, 255, 222, 7, 0, 0],
            &[0, 94, 255, 158, 0, 181, 255, 116, 0, 0],
            &[0, 94, 255, 158, 0, 47, 254, 239, 18, 0],
            &[0, 94, 255, 158, 0, 0, 166, 255, 144, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'L' / 0x4c
        'L' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 188, 254, 73, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 218, 202, 202, 202, 85, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'M' / 0x4d
        'M' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 112, 254, 254, 96, 0, 220, 254, 210, 0],
            &[0, 112, 255, 252, 135, 12, 254, 252, 211, 0],
            &[0, 112, 255, 215, 173, 56, 255, 215, 211, 0],
            &[0, 112, 255, 186, 212, 102, 236, 206, 211, 0],
            &[0, 112, 255, 156, 248, 150, 191, 222, 211, 0],
            &[0, 112, 255, 124, 255, 227, 146, 235, 211, 0],
            &[0, 112, 255, 94, 249, 255, 101, 246, 211, 0],
            &[0, 112, 255, 95, 139, 165, 41, 252, 211, 0],
            &[0, 112, 255, 97, 0, 0, 0, 254, 211, 0],
            &[0, 112, 255, 97, 0, 0, 0, 254, 211, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'N' / 0x4e
        'N' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 254, 254, 140, 0, 9, 254, 196, 0],
            &[0, 94, 255, 253, 223, 0, 9, 255, 197, 0],
            &[0, 94, 255, 199, 255, 52, 9, 255, 197, 0],
            &[0, 94, 255, 131, 255, 135, 9, 255, 197, 0],
            &[0, 94, 255, 109, 204, 219, 9, 255, 197, 0],
            &[0, 94, 255, 111, 121, 255, 54, 255, 197, 0],
            &[0, 94, 255, 111, 38, 255, 133, 255, 197, 0],
            &[0, 94, 255, 111, 0, 211, 214, 247, 197, 0],
            &[0, 94, 255, 111, 0, 128, 255, 255, 197, 0],
            &[0, 94, 255, 111, 0, 45, 255, 255, 197, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'O' / 0x4f
        'O' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 33, 4, 0, 0, 0],
            &[0, 0, 29, 195, 255, 255, 236, 95, 0, 0],
            &[0, 1, 205, 255, 205, 173, 252, 254, 59, 0],
            &[0, 60, 255, 223, 6, 0, 133, 255, 166, 0],
            &[0, 117, 255, 152, 0, 0, 53, 255, 221, 0],
            &[0, 140, 255, 127, 0, 0, 28, 255, 244, 0],
            &[0, 142, 255, 127, 0, 0, 25, 255, 244, 0],
            &[0, 118, 255, 153, 0, 0, 52, 255, 220, 0],
            &[0, 62, 255, 224, 8, 0, 132, 255, 164, 0],
            &[0, 2, 205, 255, 210, 175, 252, 253, 56, 0],
            &[0, 0, 30, 191, 255, 255, 232, 90, 0, 0],
            &[0, 0, 0, 0, 16, 27, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'P' / 0x50
        'P' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 224, 254, 254, 252, 226, 132, 3, 0, 0],
            &[0, 225, 255, 191, 192, 247, 255, 111, 0, 0],
            &[0, 225, 255, 20, 0, 87, 255, 190, 0, 0],
            &[0, 225, 255, 20, 0, 64, 255, 205, 0, 0],
            &[0, 225, 255, 85, 89, 204, 255, 152, 0, 0],
            &[0, 225, 255, 255, 255, 255, 219, 26, 0, 0],
            &[0, 225, 255, 125, 108, 75, 7, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Q' / 0x51
        'Q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 33, 4, 0, 0, 0],
            &[0, 0, 29, 195, 255, 255, 236, 95, 0, 0],
            &[0, 1, 205, 255, 205, 173, 252, 254, 59, 0],
            &[0, 60, 255, 223, 6, 0, 133, 255, 166, 0],
            &[0, 117, 255, 152, 0, 0, 53, 255, 221, 0],
            &[0, 140, 255, 127, 0, 0, 28, 255, 244, 0],
            &[0, 142, 255, 127, 0, 0, 25, 255, 244, 0],
            &[0, 118, 255, 153, 0, 0, 52, 255, 220, 0],
            &[0, 62, 255, 224, 8, 0, 132, 255, 164, 0],
            &[0, 2, 205, 255, 210, 175, 255, 253, 56, 0],
            &[0, 0, 30, 191, 255, 255, 255, 118, 0, 0],
            &[0, 0, 0, 0, 16, 144, 255, 158, 0, 0],
            &[0, 0, 0, 0, 0, 10, 233, 253, 44, 0],
            &[0, 0, 0, 0, 0, 0, 76, 133, 76, 0],
        ]),
        // letter: 'R' / 0x52
        'R' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 224, 254, 254, 239, 183, 49, 0, 0, 0],
            &[0, 225, 255, 197, 218, 255, 240, 22, 0, 0],
            &[0, 225, 255, 38, 0, 194, 255, 94, 0, 0],
            &[0, 225, 255, 38, 0, 184, 255, 89, 0, 0],
            &[0, 225, 255, 145, 170, 255, 240, 23, 0, 0],
            &[0, 225, 255, 255, 255, 238, 60, 0, 0, 0],
            &[0, 225, 255, 89, 223, 252, 45, 0, 0, 0],
            &[0, 225, 255, 38, 89, 255, 189, 0, 0, 0],
            &[0, 225, 255, 38, 1, 202, 255, 82, 0, 0],
            &[0, 225, 255, 38, 0, 62, 255, 222, 7, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'S' / 0x53
        'S' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 29, 9, 0, 0, 0],
            &[0, 0, 18, 178, 252, 255, 255, 214, 54, 0],
            &[0, 0, 156, 255, 223, 149, 174, 235, 19, 0],
            &[0, 0, 203, 255, 80, 0, 0, 3, 0, 0],
            &[0, 0, 163, 255, 182, 19, 0, 0, 0, 0],
            &[0, 0, 30, 222, 255, 242, 124, 9, 0, 0],
            &[0, 0, 0, 12, 143, 248, 255, 216, 23, 0],
            &[0, 0, 0, 0, 0, 33, 211, 255, 132, 0],
            &[0, 1, 30, 0, 0, 0, 135, 255, 152, 0],
            &[0, 5, 253, 194, 152, 160, 246, 255, 79, 0],
            &[0, 2, 181, 249, 255, 255, 225, 103, 0, 0],
            &[0, 0, 0, 4, 26, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'T' / 0x54
        'T' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 147, 254, 254, 254, 254, 254, 254, 249, 0],
            &[0, 111, 192, 192, 244, 255, 206, 192, 189, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'U' / 0x55
        'U' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 254, 168, 0, 0, 66, 254, 196, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 92, 255, 178, 0, 0, 75, 255, 194, 0],
            &[0, 68, 255, 224, 7, 0, 127, 255, 169, 0],
            &[0, 8, 230, 255, 208, 177, 248, 255, 79, 0],
            &[0, 0, 51, 208, 255, 255, 237, 115, 0, 0],
            &[0, 0, 0, 0, 18, 26, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'V' / 0x56
        'V' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[182, 254, 79, 0, 0, 1, 236, 254, 29, 0],
            &[108, 255, 145, 0, 0, 46, 255, 210, 0, 0],
            &[33, 255, 211, 0, 0, 110, 255, 135, 0, 0],
            &[0, 214, 254, 22, 0, 173, 255, 60, 0, 0],
            &[0, 139, 255, 86, 1, 235, 237, 3, 0, 0],
            &[0, 64, 255, 152, 44, 255, 166, 0, 0, 0],
            &[0, 4, 240, 216, 101, 255, 91, 0, 0, 0],
            &[0, 0, 170, 254, 165, 253, 19, 0, 0, 0],
            &[0, 0, 95, 255, 248, 197, 0, 0, 0, 0],
            &[0, 0, 22, 253, 255, 122, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'W' / 0x57
        'W' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[209, 242, 0, 0, 0, 0, 137, 254, 57, 0],
            &[191, 255, 11, 0, 0, 0, 156, 255, 39, 0],
            &[172, 255, 34, 132, 164, 37, 174, 255, 21, 0],
            &[153, 255, 53, 237, 255, 90, 192, 255, 4, 0],
            &[134, 255, 90, 255, 234, 128, 209, 241, 0, 0],
            &[115, 255, 138, 255, 168, 167, 220, 223, 0, 0],
            &[96, 255, 176, 228, 121, 208, 220, 206, 0, 0],
            &[77, 255, 210, 190, 84, 247, 214, 188, 0, 0],
            &[58, 255, 247, 152, 46, 255, 246, 170, 0, 0],
            &[39, 255, 255, 114, 10, 254, 255, 152, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'X' / 0x58
        'X' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[60, 254, 205, 1, 0, 110, 254, 161, 0, 0],
            &[0, 168, 255, 85, 10, 231, 243, 26, 0, 0],
            &[0, 30, 246, 215, 121, 255, 122, 0, 0, 0],
            &[0, 0, 128, 255, 252, 221, 7, 0, 0, 0],
            &[0, 0, 10, 230, 255, 85, 0, 0, 0, 0],
            &[0, 0, 34, 248, 255, 130, 0, 0, 0, 0],
            &[0, 0, 175, 254, 227, 247, 31, 0, 0, 0],
            &[0, 68, 255, 172, 79, 255, 171, 0, 0, 0],
            &[4, 212, 252, 40, 0, 203, 255, 64, 0, 0],
            &[109, 255, 157, 0, 0, 71, 255, 209, 3, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Y' / 0x59
        'Y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[123, 254, 161, 0, 0, 55, 254, 221, 3, 0],
            &[16, 241, 248, 21, 0, 162, 255, 105, 0, 0],
            &[0, 136, 255, 121, 21, 248, 231, 8, 0, 0],
            &[0, 23, 246, 226, 124, 255, 118, 0, 0, 0],
            &[0, 0, 148, 255, 249, 239, 13, 0, 0, 0],
            &[0, 0, 31, 251, 255, 132, 0, 0, 0, 0],
            &[0, 0, 0, 207, 255, 57, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Z' / 0x5a
        'Z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 254, 254, 254, 254, 254, 99, 0, 0],
            &[0, 98, 185, 185, 186, 252, 255, 70, 0, 0],
            &[0, 0, 0, 0, 83, 255, 186, 0, 0, 0],
            &[0, 0, 0, 8, 223, 250, 41, 0, 0, 0],
            &[0, 0, 0, 124, 255, 141, 0, 0, 0, 0],
            &[0, 0, 27, 244, 232, 14, 0, 0, 0, 0],
            &[0, 0, 165, 255, 96, 0, 0, 0, 0, 0],
            &[0, 58, 254, 200, 1, 0, 0, 0, 0, 0],
            &[0, 195, 255, 218, 191, 191, 191, 80, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '[' / 0x5b
        '[' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 24, 24, 13, 0, 0, 0],
            &[0, 0, 0, 199, 255, 255, 148, 0, 0, 0],
            &[0, 0, 0, 199, 248, 85, 49, 0, 0, 0],
            &[0, 0, 0, 199, 244, 0, 0, 0, 0, 0],
            &[0, 0, 0, 199, 244, 0, 0, 0, 0, 0],
            &[0, 0, 0, 199, 244, 0, 0, 0, 0, 0],
            &[0, 0, 0, 199, 244, 0, 0, 0, 0, 0],
            &[0, 0, 0, 199, 244, 0, 0, 0, 0, 0],
            &[0, 0, 0, 199, 244, 0, 0, 0, 0, 0],
            &[0, 0, 0, 199, 244, 0, 0, 0, 0, 0],
            &[0, 0, 0, 199, 244, 0, 0, 0, 0, 0],
            &[0, 0, 0, 199, 246, 44, 25, 0, 0, 0],
            &[0, 0, 0, 199, 255, 255, 148, 0, 0, 0],
            &[0, 0, 0, 53, 68, 68, 39, 0, 0, 0],
        ]),
        // letter: '\' / 0x5c
        '\\' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 13, 2, 0, 0, 0, 0, 0, 0],
            &[0, 99, 255, 92, 0, 0, 0, 0, 0, 0],
            &[0, 13, 245, 189, 0, 0, 0, 0, 0, 0],
            &[0, 0, 161, 254, 32, 0, 0, 0, 0, 0],
            &[0, 0, 64, 255, 128, 0, 0, 0, 0, 0],
            &[0, 0, 1, 221, 224, 1, 0, 0, 0, 0],
            &[0, 0, 0, 126, 255, 67, 0, 0, 0, 0],
            &[0, 0, 0, 31, 253, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 246, 15, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 104, 0, 0, 0],
            &[0, 0, 0, 0, 9, 240, 201, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ']' / 0x5d
        ']' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 49, 49, 49, 8, 0, 0, 0],
            &[0, 0, 46, 255, 255, 255, 46, 0, 0, 0],
            &[0, 0, 11, 63, 170, 255, 46, 0, 0, 0],
            &[0, 0, 0, 0, 142, 255, 46, 0, 0, 0],
            &[0, 0, 0, 0, 142, 255, 46, 0, 0, 0],
            &[0, 0, 0, 0, 142, 255, 46, 0, 0, 0],
            &[0, 0, 0, 0, 142, 255, 46, 0, 0, 0],
            &[0, 0, 0, 0, 142, 255, 46, 0, 0, 0],
            &[0, 0, 0, 0, 142, 255, 46, 0, 0, 0],
            &[0, 0, 0, 0, 142, 255, 46, 0, 0, 0],
            &[0, 0, 0, 0, 142, 255, 46, 0, 0, 0],
            &[0, 0, 11, 66, 171, 255, 46, 0, 0, 0],
            &[0, 0, 46, 255, 255, 255, 46, 0, 0, 0],
            &[0, 0, 7, 43, 43, 43, 7, 0, 0, 0],
        ]),
        // letter: '^' / 0x5e
        '^' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 214, 2, 0, 0, 0],
            &[0, 0, 0, 35, 252, 255, 93, 0, 0, 0],
            &[0, 0, 0, 156, 240, 199, 221, 5, 0, 0],
            &[0, 0, 30, 250, 133, 61, 255, 103, 0, 0],
            &[0, 0, 148, 246, 22, 0, 185, 227, 7, 0],
            &[0, 24, 247, 148, 0, 0, 54, 255, 112, 0],
            &[0, 26, 70, 19, 0, 0, 0, 62, 54, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '_' / 0x5f
        '_' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[15, 15, 15, 15, 15, 15, 15, 15, 6, 0],
            &[255, 255, 255, 255, 255, 255, 255, 255, 102, 0],
            &[158, 158, 158, 158, 158, 158, 158, 158, 63, 0],
        ]),
        // letter: '`' / 0x60
        '`' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 155, 185, 62, 0, 0, 0, 0],
            &[0, 0, 0, 33, 221, 219, 11, 0, 0, 0],
            &[0, 0, 0, 0, 22, 129, 55, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'a' / 0x61
        'a' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 55, 146, 183, 188, 149, 42, 0, 0],
            &[0, 0, 141, 254, 229, 230, 255, 241, 33, 0],
            &[0, 0, 18, 26, 0, 0, 152, 255, 105, 0],
            &[0, 0, 27, 131, 174, 187, 220, 255, 125, 0],
            &[0, 12, 230, 254, 170, 126, 179, 255, 125, 0],
            &[0, 66, 255, 191, 0, 0, 146, 255, 125, 0],
            &[0, 44, 255, 244, 140, 157, 253, 255, 125, 0],
            &[0, 0, 147, 254, 255, 209, 97, 255, 125, 0],
            &[0, 0, 0, 12, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'b' / 0x62
        'b' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 163, 124, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 193, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 193, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 186, 87, 183, 183, 77, 0, 0],
            &[0, 58, 255, 236, 253, 244, 255, 253, 51, 0],
            &[0, 58, 255, 252, 50, 2, 165, 255, 154, 0],
            &[0, 58, 255, 210, 0, 0, 72, 255, 196, 0],
            &[0, 58, 255, 203, 0, 0, 65, 255, 204, 0],
            &[0, 58, 255, 243, 13, 0, 118, 255, 173, 0],
            &[0, 58, 255, 255, 207, 156, 247, 255, 87, 0],
            &[0, 58, 255, 147, 187, 253, 254, 158, 2, 0],
            &[0, 0, 0, 0, 0, 13, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'c' / 0x63
        'c' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 153, 191, 185, 138, 20, 0],
            &[0, 0, 60, 248, 255, 252, 251, 242, 9, 0],
            &[0, 0, 188, 255, 160, 13, 4, 40, 0, 0],
            &[0, 0, 243, 255, 39, 0, 0, 0, 0, 0],
            &[0, 1, 253, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 224, 255, 91, 0, 0, 10, 2, 0],
            &[0, 0, 133, 255, 247, 174, 168, 238, 10, 0],
            &[0, 0, 7, 149, 245, 255, 255, 206, 6, 0],
            &[0, 0, 0, 0, 5, 28, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'd' / 0x64
        'd' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 58, 163, 103, 0],
            &[0, 0, 0, 0, 0, 0, 91, 255, 161, 0],
            &[0, 0, 0, 0, 0, 0, 91, 255, 161, 0],
            &[0, 0, 19, 145, 191, 151, 101, 255, 161, 0],
            &[0, 0, 197, 255, 254, 241, 238, 255, 161, 0],
            &[0, 54, 255, 245, 40, 1, 164, 255, 161, 0],
            &[0, 99, 255, 179, 0, 0, 72, 255, 161, 0],
            &[0, 102, 255, 165, 0, 0, 57, 255, 161, 0],
            &[0, 73, 255, 212, 0, 0, 104, 255, 161, 0],
            &[0, 8, 237, 255, 190, 154, 245, 255, 161, 0],
            &[0, 0, 81, 240, 255, 223, 82, 253, 161, 0],
            &[0, 0, 0, 7, 26, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'e' / 0x65
        'e' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 174, 192, 151, 35, 0, 0],
            &[0, 0, 123, 255, 243, 213, 253, 237, 28, 0],
            &[0, 13, 247, 241, 29, 0, 109, 255, 130, 0],
            &[0, 61, 255, 232, 158, 158, 174, 255, 171, 0],
            &[0, 66, 255, 238, 196, 196, 196, 196, 137, 0],
            &[0, 26, 253, 235, 17, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 230, 156, 148, 193, 117, 0],
            &[0, 0, 9, 144, 240, 255, 255, 233, 82, 0],
            &[0, 0, 0, 0, 3, 25, 22, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'f' / 0x66
        'f' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 124, 173, 170, 133, 5, 0],
            &[0, 0, 0, 178, 255, 255, 255, 210, 0, 0],
            &[0, 0, 1, 249, 255, 81, 9, 23, 0, 0],
            &[2, 86, 140, 255, 255, 166, 164, 78, 0, 0],
            &[12, 255, 255, 255, 255, 255, 255, 121, 0, 0],
            &[0, 9, 13, 255, 255, 12, 9, 4, 0, 0],
            &[0, 0, 4, 255, 255, 3, 0, 0, 0, 0],
            &[0, 0, 4, 255, 255, 3, 0, 0, 0, 0],
            &[0, 0, 4, 255, 255, 3, 0, 0, 0, 0],
            &[0, 0, 4, 255, 255, 3, 0, 0, 0, 0],
            &[0, 0, 4, 255, 255, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'g' / 0x67
        'g' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 24, 157, 197, 143, 23, 162, 103, 0],
            &[0, 1, 204, 255, 253, 246, 232, 255, 161, 0],
            &[0, 57, 255, 241, 32, 6, 178, 255, 161, 0],
            &[0, 101, 255, 176, 0, 0, 91, 255, 161, 0],
            &[0, 107, 255, 167, 0, 0, 74, 255, 161, 0],
            &[0, 78, 255, 218, 2, 0, 118, 255, 161, 0],
            &[0, 12, 239, 255, 195, 156, 247, 255, 161, 0],
            &[0, 0, 75, 237, 255, 230, 158, 255, 161, 0],
            &[0, 0, 0, 5, 29, 4, 124, 255, 147, 0],
            &[0, 0, 180, 126, 94, 116, 236, 255, 85, 0],
            &[0, 0, 24, 83, 104, 91, 36, 0, 0, 0],
        ]),
        // letter: 'h' / 0x68
        'h' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 163, 131, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 204, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 204, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 201, 76, 181, 181, 99, 0, 0],
            &[0, 58, 255, 238, 252, 242, 255, 255, 85, 0],
            &[0, 58, 255, 255, 77, 0, 150, 255, 160, 0],
            &[0, 58, 255, 231, 0, 0, 88, 255, 179, 0],
            &[0, 58, 255, 207, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'i' / 0x69
        'i' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 113, 212, 86, 0, 0, 0],
            &[0, 0, 0, 0, 193, 255, 157, 0, 0, 0],
            &[0, 0, 0, 0, 45, 115, 28, 0, 0, 0],
            &[0, 0, 128, 164, 164, 164, 68, 0, 0, 0],
            &[0, 0, 168, 240, 255, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 11, 94, 136, 220, 255, 200, 123, 58, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'j' / 0x6a
        'j' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 202, 172, 1, 0, 0],
            &[0, 0, 0, 0, 86, 255, 253, 11, 0, 0],
            &[0, 0, 0, 0, 7, 106, 75, 0, 0, 0],
            &[0, 0, 31, 164, 164, 164, 164, 0, 0, 0],
            &[0, 0, 40, 223, 250, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 7, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 6, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 6, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 6, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 6, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 6, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 30, 255, 247, 0, 0, 0],
            &[0, 62, 107, 94, 184, 255, 205, 0, 0, 0],
            &[0, 18, 79, 103, 84, 23, 0, 0, 0, 0],
        ]),
        // letter: 'k' / 0x6b
        'k' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[26, 163, 142, 0, 0, 0, 0, 0, 0, 0],
            &[40, 255, 222, 0, 0, 0, 0, 0, 0, 0],
            &[40, 255, 222, 0, 0, 0, 0, 0, 0, 0],
            &[40, 255, 222, 0, 0, 101, 164, 119, 0, 0],
            &[40, 255, 222, 0, 82, 254, 227, 28, 0, 0],
            &[40, 255, 220, 50, 245, 240, 45, 0, 0, 0],
            &[40, 255, 220, 224, 255, 103, 0, 0, 0, 0],
            &[40, 255, 255, 248, 255, 211, 6, 0, 0, 0],
            &[40, 255, 242, 50, 156, 255, 133, 0, 0, 0],
            &[40, 255, 222, 0, 16, 230, 251, 52, 0, 0],
            &[40, 255, 222, 0, 0, 81, 255, 212, 6, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'l' / 0x6c
        'l' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 109, 163, 163, 163, 63, 0, 0, 0],
            &[0, 0, 118, 220, 255, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 164, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 11, 94, 137, 223, 255, 197, 122, 57, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'm' / 0x6d
        'm' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 95, 143, 78, 189, 86, 79, 191, 81, 0],
            &[0, 148, 254, 230, 254, 232, 229, 255, 211, 0],
            &[0, 148, 255, 63, 183, 255, 44, 205, 241, 0],
            &[0, 148, 255, 43, 174, 255, 24, 197, 250, 0],
            &[0, 148, 255, 43, 174, 255, 24, 197, 250, 0],
            &[0, 148, 255, 43, 174, 255, 24, 197, 250, 0],
            &[0, 148, 255, 43, 174, 255, 24, 197, 250, 0],
            &[0, 148, 255, 43, 174, 255, 24, 197, 250, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'n' / 0x6e
        'n' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 164, 69, 86, 181, 182, 101, 0, 0],
            &[0, 58, 255, 228, 255, 242, 255, 255, 87, 0],
            &[0, 58, 255, 255, 76, 0, 155, 255, 160, 0],
            &[0, 58, 255, 229, 0, 0, 88, 255, 179, 0],
            &[0, 58, 255, 206, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'o' / 0x6f
        'o' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 95, 177, 191, 143, 28, 0, 0],
            &[0, 0, 146, 255, 250, 240, 255, 232, 25, 0],
            &[0, 35, 255, 243, 33, 4, 171, 255, 141, 0],
            &[0, 94, 255, 177, 0, 0, 78, 255, 198, 0],
            &[0, 106, 255, 170, 0, 0, 68, 255, 208, 0],
            &[0, 68, 255, 222, 6, 0, 123, 255, 170, 0],
            &[0, 3, 215, 255, 196, 153, 245, 255, 65, 0],
            &[0, 0, 34, 193, 255, 255, 233, 96, 0, 0],
            &[0, 0, 0, 0, 17, 28, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'p' / 0x70
        'p' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 164, 69, 81, 181, 183, 78, 0, 0],
            &[0, 58, 255, 227, 253, 244, 255, 253, 51, 0],
            &[0, 58, 255, 252, 54, 1, 169, 255, 153, 0],
            &[0, 58, 255, 207, 0, 0, 82, 255, 196, 0],
            &[0, 58, 255, 203, 0, 0, 74, 255, 207, 0],
            &[0, 58, 255, 242, 13, 0, 119, 255, 177, 0],
            &[0, 58, 255, 255, 209, 154, 245, 255, 93, 0],
            &[0, 58, 255, 199, 186, 255, 253, 151, 2, 0],
            &[0, 58, 255, 193, 0, 20, 15, 0, 0, 0],
            &[0, 58, 255, 193, 0, 0, 0, 0, 0, 0],
            &[0, 21, 92, 69, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'q' / 0x71
        'q' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 24, 154, 193, 147, 25, 161, 103, 0],
            &[0, 0, 201, 255, 253, 245, 232, 255, 161, 0],
            &[0, 51, 255, 246, 37, 5, 181, 255, 161, 0],
            &[0, 94, 255, 187, 0, 0, 85, 255, 161, 0],
            &[0, 103, 255, 176, 0, 0, 71, 255, 161, 0],
            &[0, 73, 255, 223, 2, 0, 119, 255, 161, 0],
            &[0, 8, 237, 255, 194, 156, 248, 255, 161, 0],
            &[0, 0, 82, 241, 255, 228, 143, 255, 161, 0],
            &[0, 0, 0, 7, 28, 3, 91, 255, 161, 0],
            &[0, 0, 0, 0, 0, 0, 91, 255, 161, 0],
            &[0, 0, 0, 0, 0, 0, 32, 92, 58, 0],
        ]),
        // letter: 'r' / 0x72
        'r' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[7, 164, 164, 164, 28, 100, 189, 180, 12, 0],
            &[9, 224, 253, 255, 189, 255, 255, 220, 0, 0],
            &[0, 0, 113, 255, 255, 127, 48, 46, 0, 0],
            &[0, 0, 112, 255, 196, 0, 0, 0, 0, 0],
            &[0, 0, 112, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 112, 255, 151, 0, 0, 0, 0, 0],
            &[33, 116, 201, 255, 218, 125, 48, 0, 0, 0],
            &[109, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 's' / 0x73
        's' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 116, 183, 193, 169, 107, 13, 0, 0],
            &[0, 147, 255, 250, 234, 254, 241, 11, 0, 0],
            &[0, 202, 255, 106, 0, 14, 52, 0, 0, 0],
            &[0, 120, 255, 254, 187, 84, 2, 0, 0, 0],
            &[0, 0, 81, 199, 255, 255, 181, 3, 0, 0],
            &[0, 14, 0, 0, 51, 241, 255, 57, 0, 0],
            &[0, 221, 195, 158, 162, 253, 255, 42, 0, 0],
            &[0, 181, 252, 255, 255, 238, 120, 0, 0, 0],
            &[0, 0, 9, 29, 25, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 't' / 0x74
        't' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 140, 73, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 133, 0, 0, 0, 0],
            &[0, 16, 100, 182, 255, 212, 164, 164, 64, 0],
            &[0, 66, 255, 255, 255, 255, 255, 255, 100, 0],
            &[0, 2, 9, 134, 255, 137, 9, 9, 3, 0],
            &[0, 0, 0, 129, 255, 133, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 133, 0, 0, 0, 0],
            &[0, 0, 0, 125, 255, 153, 0, 0, 0, 0],
            &[0, 0, 0, 83, 255, 249, 158, 157, 85, 0],
            &[0, 0, 0, 3, 159, 251, 255, 253, 104, 0],
            &[0, 0, 0, 0, 0, 11, 30, 12, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'u' / 0x75
        'u' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 164, 120, 0, 0, 65, 164, 103, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 187, 0, 0, 116, 255, 161, 0],
            &[0, 68, 255, 212, 0, 0, 172, 255, 161, 0],
            &[0, 26, 251, 255, 171, 166, 254, 255, 161, 0],
            &[0, 0, 112, 249, 255, 223, 82, 252, 161, 0],
            &[0, 0, 0, 6, 25, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'v' / 0x76
        'v' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[98, 164, 93, 0, 0, 20, 164, 159, 4, 0],
            &[75, 255, 211, 0, 0, 97, 255, 177, 0, 0],
            &[4, 232, 255, 37, 0, 177, 255, 81, 0, 0],
            &[0, 141, 255, 118, 11, 247, 235, 5, 0, 0],
            &[0, 45, 255, 199, 82, 255, 146, 0, 0, 0],
            &[0, 0, 206, 253, 169, 255, 50, 0, 0, 0],
            &[0, 0, 111, 255, 249, 210, 0, 0, 0, 0],
            &[0, 0, 21, 250, 255, 115, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'w' / 0x77
        'w' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[150, 151, 0, 0, 0, 0, 78, 164, 53, 0],
            &[203, 251, 1, 94, 134, 17, 140, 255, 53, 0],
            &[167, 255, 20, 219, 255, 73, 163, 255, 17, 0],
            &[131, 255, 59, 254, 244, 124, 185, 237, 0, 0],
            &[95, 255, 131, 254, 173, 174, 207, 201, 0, 0],
            &[59, 255, 198, 220, 118, 220, 229, 165, 0, 0],
            &[23, 255, 250, 173, 72, 253, 252, 129, 0, 0],
            &[0, 242, 255, 126, 27, 255, 255, 93, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'x' / 0x78
        'x' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[61, 164, 159, 10, 0, 84, 164, 144, 2, 0],
            &[4, 202, 255, 128, 20, 237, 255, 81, 0, 0],
            &[0, 40, 246, 245, 177, 255, 157, 0, 0, 0],
            &[0, 0, 111, 255, 255, 221, 12, 0, 0, 0],
            &[0, 0, 76, 255, 255, 193, 1, 0, 0, 0],
            &[0, 20, 231, 254, 216, 255, 109, 0, 0, 0],
            &[0, 173, 255, 159, 47, 253, 245, 37, 0, 0],
            &[95, 255, 238, 21, 0, 154, 255, 197, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'y' / 0x79
        'y' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[97, 164, 98, 0, 0, 26, 164, 160, 5, 0],
            &[68, 255, 223, 0, 0, 103, 255, 180, 0, 0],
            &[1, 221, 255, 53, 0, 178, 255, 87, 0, 0],
            &[0, 121, 255, 139, 8, 245, 241, 8, 0, 0],
            &[0, 24, 251, 223, 74, 255, 157, 0, 0, 0],
            &[0, 0, 173, 255, 184, 255, 64, 0, 0, 0],
            &[0, 0, 72, 255, 254, 225, 1, 0, 0, 0],
            &[0, 0, 2, 224, 255, 133, 0, 0, 0, 0],
            &[0, 0, 7, 230, 254, 39, 0, 0, 0, 0],
            &[17, 93, 177, 255, 178, 0, 0, 0, 0, 0],
            &[14, 102, 87, 12, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'z' / 0x7a
        'z' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 145, 164, 164, 164, 164, 164, 59, 0],
            &[0, 0, 214, 243, 243, 246, 255, 255, 84, 0],
            &[0, 0, 0, 0, 0, 163, 255, 180, 2, 0],
            &[0, 0, 0, 0, 114, 255, 214, 14, 0, 0],
            &[0, 0, 0, 69, 252, 237, 35, 0, 0, 0],
            &[0, 0, 34, 237, 251, 66, 0, 0, 0, 0],
            &[0, 6, 211, 255, 221, 152, 152, 152, 74, 0],
            &[0, 23, 255, 255, 255, 255, 255, 255, 125, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '{' / 0x7b
        '{' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 4, 0, 0],
            &[0, 0, 0, 0, 19, 187, 250, 123, 0, 0],
            &[0, 0, 0, 0, 109, 255, 199, 59, 0, 0],
            &[0, 0, 0, 0, 127, 255, 72, 0, 0, 0],
            &[0, 0, 0, 0, 127, 255, 67, 0, 0, 0],
            &[0, 0, 0, 0, 155, 255, 62, 0, 0, 0],
            &[0, 0, 13, 192, 253, 206, 10, 0, 0, 0],
            &[0, 0, 18, 242, 250, 152, 2, 0, 0, 0],
            &[0, 0, 0, 4, 180, 255, 56, 0, 0, 0],
            &[0, 0, 0, 0, 128, 255, 67, 0, 0, 0],
            &[0, 0, 0, 0, 127, 255, 68, 0, 0, 0],
            &[0, 0, 0, 0, 118, 255, 164, 35, 0, 0],
            &[0, 0, 0, 0, 38, 234, 255, 123, 0, 0],
            &[0, 0, 0, 0, 0, 8, 51, 31, 0, 0],
        ]),
        // letter: '|' / 0x7c
        '|' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 95, 158, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 53, 89, 0, 0, 0, 0],
        ]),
        // letter: '}' / 0x7d
        '}' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 255, 224, 80, 0, 0, 0, 0],
            &[0, 0, 9, 154, 254, 211, 0, 0, 0, 0],
            &[0, 0, 0, 0, 224, 230, 0, 0, 0, 0],
            &[0, 0, 0, 0, 221, 230, 0, 0, 0, 0],
            &[0, 0, 0, 0, 216, 246, 11, 0, 0, 0],
            &[0, 0, 0, 0, 120, 251, 221, 84, 0, 0],
            &[0, 0, 0, 0, 77, 224, 252, 111, 0, 0],
            &[0, 0, 0, 0, 209, 251, 36, 0, 0, 0],
            &[0, 0, 0, 0, 221, 230, 0, 0, 0, 0],
            &[0, 0, 0, 0, 222, 230, 0, 0, 0, 0],
            &[0, 0, 4, 92, 251, 220, 0, 0, 0, 0],
            &[0, 0, 20, 255, 254, 120, 0, 0, 0, 0],
            &[0, 0, 5, 60, 24, 0, 0, 0, 0, 0],
        ]),
        // letter: '~' / 0x7e
        '~' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 92, 64, 0, 0, 70, 47, 0],
            &[0, 0, 175, 199, 227, 142, 3, 202, 81, 0],
            &[0, 4, 249, 9, 25, 198, 237, 218, 10, 0],
            &[0, 3, 50, 0, 0, 1, 39, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: ' ' / 0xa0
        ' ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¡' / 0xa1
        '¡' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 122, 13, 0, 0, 0],
            &[0, 0, 0, 5, 251, 255, 107, 0, 0, 0],
            &[0, 0, 0, 0, 172, 218, 51, 0, 0, 0],
            &[0, 0, 0, 0, 15, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 184, 255, 27, 0, 0, 0],
            &[0, 0, 0, 0, 195, 255, 38, 0, 0, 0],
            &[0, 0, 0, 0, 207, 255, 50, 0, 0, 0],
            &[0, 0, 0, 0, 218, 255, 61, 0, 0, 0],
            &[0, 0, 0, 0, 229, 255, 72, 0, 0, 0],
            &[0, 0, 0, 0, 241, 255, 83, 0, 0, 0],
            &[0, 0, 0, 0, 147, 150, 54, 0, 0, 0],
        ]),
        // letter: '¢' / 0xa2
        '¢' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 81, 0, 0, 0],
            &[0, 0, 0, 98, 215, 255, 254, 222, 54, 0],
            &[0, 0, 94, 255, 241, 159, 155, 209, 17, 0],
            &[0, 0, 205, 255, 61, 0, 0, 0, 0, 0],
            &[0, 0, 248, 242, 0, 0, 0, 0, 0, 0],
            &[0, 0, 245, 246, 2, 0, 0, 0, 0, 0],
            &[0, 0, 193, 255, 99, 0, 0, 17, 15, 0],
            &[0, 0, 67, 251, 255, 226, 212, 249, 57, 0],
            &[0, 0, 0, 56, 164, 245, 216, 151, 21, 0],
            &[0, 0, 0, 0, 0, 208, 78, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '£' / 0xa3
        '£' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 22, 1, 0, 0, 0],
            &[0, 0, 16, 184, 255, 255, 233, 59, 0, 0],
            &[0, 0, 143, 255, 219, 151, 202, 18, 0, 0],
            &[0, 0, 194, 255, 53, 0, 0, 0, 0, 0],
            &[0, 0, 205, 255, 40, 0, 0, 0, 0, 0],
            &[0, 188, 247, 255, 220, 214, 94, 0, 0, 0],
            &[0, 139, 236, 255, 173, 158, 69, 0, 0, 0],
            &[0, 0, 208, 255, 36, 0, 0, 0, 0, 0],
            &[0, 18, 243, 226, 5, 0, 0, 0, 0, 0],
            &[0, 197, 255, 211, 184, 184, 184, 103, 0, 0],
            &[0, 221, 255, 255, 255, 255, 255, 143, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¤' / 0xa4
        '¤' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 15, 0, 4, 0, 74, 2, 0],
            &[0, 6, 223, 216, 237, 252, 198, 255, 71, 0],
            &[0, 0, 97, 255, 160, 118, 236, 195, 0, 0],
            &[0, 0, 155, 221, 0, 0, 118, 250, 2, 0],
            &[0, 0, 133, 242, 25, 0, 165, 233, 0, 0],
            &[0, 0, 132, 255, 244, 227, 255, 212, 14, 0],
            &[0, 5, 192, 122, 129, 145, 93, 218, 51, 0],
            &[0, 0, 3, 0, 0, 0, 0, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¥' / 0xa5
        '¥' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[102, 254, 160, 0, 0, 58, 254, 202, 0, 0],
            &[5, 224, 249, 25, 0, 172, 255, 74, 0, 0],
            &[0, 102, 255, 133, 33, 252, 199, 0, 0, 0],
            &[0, 5, 224, 237, 155, 255, 69, 0, 0, 0],
            &[0, 0, 102, 255, 255, 195, 0, 0, 0, 0],
            &[0, 182, 239, 255, 255, 246, 237, 39, 0, 0],
            &[0, 56, 74, 210, 255, 101, 74, 12, 0, 0],
            &[0, 148, 193, 240, 255, 203, 193, 32, 0, 0],
            &[0, 87, 114, 220, 255, 135, 114, 19, 0, 0],
            &[0, 0, 0, 192, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¦' / 0xa6
        '¦' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 151, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 231, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 54, 0, 0, 0, 0],
        ]),
        // letter: '§' / 0xa7
        '§' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 61, 153, 173, 146, 67, 0, 0],
            &[0, 0, 57, 252, 220, 159, 209, 146, 0, 0],
            &[0, 0, 94, 255, 135, 0, 0, 6, 0, 0],
            &[0, 0, 12, 203, 255, 201, 77, 0, 0, 0],
            &[0, 0, 47, 240, 177, 206, 255, 128, 0, 0],
            &[0, 0, 127, 255, 57, 4, 212, 230, 0, 0],
            &[0, 0, 59, 250, 229, 107, 234, 180, 0, 0],
            &[0, 0, 0, 56, 193, 255, 255, 67, 0, 0],
            &[0, 0, 0, 0, 0, 71, 252, 200, 0, 0],
            &[0, 0, 101, 104, 49, 85, 252, 197, 0, 0],
            &[0, 0, 99, 242, 255, 255, 208, 46, 0, 0],
            &[0, 0, 0, 2, 25, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¨' / 0xa8
        '¨' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 130, 38, 0, 113, 74, 0, 0],
            &[0, 0, 79, 255, 131, 13, 254, 202, 0, 0],
            &[0, 0, 8, 93, 22, 0, 77, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '©' / 0xa9
        '©' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 84, 159, 171, 124, 21, 0, 0, 0],
            &[0, 162, 192, 108, 127, 149, 223, 32, 0, 0],
            &[74, 210, 84, 235, 172, 117, 108, 173, 0, 0],
            &[144, 119, 189, 105, 0, 0, 14, 245, 0, 0],
            &[151, 110, 197, 87, 0, 0, 9, 251, 2, 0],
            &[95, 184, 114, 214, 126, 83, 84, 202, 0, 0],
            &[6, 207, 139, 96, 122, 123, 231, 67, 0, 0],
            &[0, 16, 144, 215, 225, 178, 57, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ª' / 0xaa
        'ª' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 18, 0, 0, 0, 0, 0],
            &[0, 0, 87, 245, 250, 213, 17, 0, 0, 0],
            &[0, 0, 14, 38, 73, 253, 87, 0, 0, 0],
            &[0, 0, 150, 228, 154, 251, 98, 0, 0, 0],
            &[0, 0, 241, 141, 108, 255, 98, 0, 0, 0],
            &[0, 0, 107, 202, 148, 128, 73, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '«' / 0xab
        '«' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 59, 203, 52, 60, 203, 52, 0],
            &[0, 0, 20, 228, 238, 48, 228, 238, 27, 0],
            &[0, 2, 185, 255, 90, 185, 255, 89, 0, 0],
            &[0, 14, 228, 250, 55, 229, 251, 41, 0, 0],
            &[0, 0, 60, 251, 203, 66, 251, 204, 4, 0],
            &[0, 0, 0, 118, 252, 75, 119, 252, 74, 0],
            &[0, 0, 0, 1, 34, 0, 1, 34, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¬' / 0xac
        '¬' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 175, 175, 175, 175, 175, 175, 49, 0],
            &[0, 4, 204, 204, 204, 204, 215, 255, 71, 0],
            &[0, 0, 0, 0, 0, 0, 55, 255, 71, 0],
            &[0, 0, 0, 0, 0, 0, 55, 255, 71, 0],
            &[0, 0, 0, 0, 0, 0, 17, 81, 22, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '®' / 0xae
        '®' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 84, 159, 171, 123, 20, 0, 0, 0],
            &[0, 160, 205, 118, 87, 131, 222, 31, 0, 0],
            &[72, 213, 86, 227, 208, 140, 108, 171, 0, 0],
            &[141, 123, 79, 189, 131, 181, 17, 242, 0, 0],
            &[147, 113, 79, 221, 222, 73, 12, 249, 1, 0],
            &[90, 187, 79, 178, 109, 157, 88, 198, 0, 0],
            &[5, 202, 163, 71, 23, 142, 232, 65, 0, 0],
            &[0, 13, 139, 212, 222, 176, 56, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¯' / 0xaf
        '¯' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 24, 24, 24, 7, 0, 0, 0],
            &[0, 0, 233, 255, 255, 255, 83, 0, 0, 0],
            &[0, 0, 68, 74, 74, 74, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '°' / 0xb0
        '°' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 22, 0, 0, 0, 0],
            &[0, 0, 0, 75, 244, 255, 165, 4, 0, 0],
            &[0, 0, 0, 235, 139, 52, 248, 85, 0, 0],
            &[0, 0, 1, 243, 113, 21, 241, 95, 0, 0],
            &[0, 0, 0, 117, 255, 252, 214, 14, 0, 0],
            &[0, 0, 0, 0, 36, 59, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '±' / 0xb1
        '±' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 105, 193, 0, 0, 0, 0],
            &[0, 0, 0, 0, 135, 248, 0, 0, 0, 0],
            &[0, 0, 51, 51, 159, 249, 51, 51, 21, 0],
            &[0, 0, 253, 255, 255, 255, 255, 255, 107, 0],
            &[0, 0, 71, 72, 169, 250, 72, 72, 30, 0],
            &[0, 0, 0, 0, 135, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 206, 0, 0, 0, 0],
            &[0, 2, 127, 127, 127, 127, 127, 127, 53, 0],
            &[0, 5, 255, 255, 255, 255, 255, 255, 107, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '²' / 0xb2
        '²' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 89, 215, 234, 138, 0, 0, 0, 0],
            &[0, 0, 112, 112, 152, 255, 22, 0, 0, 0],
            &[0, 0, 0, 0, 121, 245, 5, 0, 0, 0],
            &[0, 0, 0, 77, 247, 94, 0, 0, 0, 0],
            &[0, 0, 85, 249, 107, 10, 4, 0, 0, 0],
            &[0, 0, 215, 255, 255, 255, 112, 0, 0, 0],
            &[0, 0, 38, 45, 45, 45, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '³' / 0xb3
        '³' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 101, 210, 228, 129, 0, 0, 0, 0],
            &[0, 0, 91, 89, 138, 255, 14, 0, 0, 0],
            &[0, 0, 0, 95, 189, 169, 0, 0, 0, 0],
            &[0, 0, 0, 151, 221, 182, 8, 0, 0, 0],
            &[0, 0, 54, 1, 60, 255, 65, 0, 0, 0],
            &[0, 0, 214, 244, 250, 219, 16, 0, 0, 0],
            &[0, 0, 10, 60, 57, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '´' / 0xb4
        '´' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 132, 185, 89, 0, 0, 0],
            &[0, 0, 0, 73, 255, 155, 2, 0, 0, 0],
            &[0, 0, 0, 105, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'µ' / 0xb5
        'µ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 164, 120, 0, 0, 65, 164, 103, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 90, 255, 220, 0, 0, 116, 255, 161, 0],
            &[0, 145, 255, 255, 0, 0, 172, 255, 161, 0],
            &[0, 103, 255, 255, 171, 166, 254, 255, 161, 0],
            &[0, 76, 255, 255, 255, 223, 82, 252, 161, 0],
            &[0, 76, 255, 193, 25, 1, 0, 0, 0, 0],
            &[0, 76, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 27, 92, 67, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¶' / 0xb6
        '¶' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 109, 155, 163, 163, 163, 114, 0],
            &[0, 3, 200, 255, 255, 255, 211, 182, 179, 0],
            &[0, 66, 255, 255, 255, 255, 160, 96, 179, 0],
            &[0, 104, 255, 255, 255, 255, 160, 96, 179, 0],
            &[0, 99, 255, 255, 255, 255, 160, 96, 179, 0],
            &[0, 47, 255, 255, 255, 255, 160, 96, 179, 0],
            &[0, 0, 144, 255, 255, 255, 160, 96, 179, 0],
            &[0, 0, 0, 33, 71, 145, 160, 96, 179, 0],
            &[0, 0, 0, 0, 0, 111, 160, 96, 179, 0],
            &[0, 0, 0, 0, 0, 111, 160, 96, 179, 0],
            &[0, 0, 0, 0, 0, 111, 160, 96, 179, 0],
            &[0, 0, 0, 0, 0, 111, 160, 96, 179, 0],
            &[0, 0, 0, 0, 0, 86, 125, 75, 139, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '·' / 0xb7
        '·' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 13, 0, 0, 0, 0],
            &[0, 0, 0, 2, 220, 251, 67, 0, 0, 0],
            &[0, 0, 0, 6, 244, 255, 92, 0, 0, 0],
            &[0, 0, 0, 0, 42, 73, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¸' / 0xb8
        '¸' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 199, 16, 0, 0, 0],
            &[0, 0, 0, 0, 40, 226, 162, 0, 0, 0],
            &[0, 0, 0, 13, 85, 66, 2, 0, 0, 0],
        ]),
        // letter: '¹' / 0xb9
        '¹' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 113, 208, 28, 0, 0, 0, 0],
            &[0, 0, 177, 239, 255, 35, 0, 0, 0, 0],
            &[0, 0, 54, 62, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 52, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 52, 255, 35, 0, 0, 0, 0],
            &[0, 0, 116, 255, 255, 244, 97, 0, 0, 0],
            &[0, 0, 27, 54, 90, 51, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'º' / 0xba
        'º' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 15, 0, 0, 0, 0, 0],
            &[0, 0, 76, 241, 254, 160, 0, 0, 0, 0],
            &[0, 0, 215, 152, 47, 255, 60, 0, 0, 0],
            &[0, 0, 241, 105, 8, 255, 85, 0, 0, 0],
            &[0, 0, 189, 192, 125, 252, 37, 0, 0, 0],
            &[0, 0, 33, 174, 197, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '»' / 0xbb
        '»' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 153, 154, 6, 154, 153, 0, 0, 0],
            &[0, 0, 165, 255, 96, 165, 255, 95, 0, 0],
            &[0, 0, 16, 227, 245, 60, 226, 245, 43, 0],
            &[0, 0, 0, 189, 255, 91, 187, 255, 90, 0],
            &[0, 0, 105, 255, 160, 105, 255, 159, 0, 0],
            &[0, 6, 218, 211, 17, 219, 211, 9, 0, 0],
            &[0, 0, 9, 26, 0, 9, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¼' / 0xbc
        '¼' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[74, 218, 164, 0, 0, 0, 0, 0, 0, 0],
            &[119, 179, 165, 0, 0, 1, 138, 111, 0, 0],
            &[0, 120, 165, 0, 0, 121, 239, 33, 0, 0],
            &[27, 187, 234, 36, 62, 251, 80, 0, 0, 0],
            &[98, 255, 255, 146, 230, 143, 0, 0, 0, 0],
            &[0, 0, 2, 187, 200, 25, 176, 65, 0, 0],
            &[0, 0, 127, 237, 31, 167, 241, 93, 0, 0],
            &[0, 67, 252, 77, 81, 170, 196, 93, 0, 0],
            &[23, 227, 138, 0, 202, 219, 244, 224, 18, 0],
            &[0, 0, 0, 0, 27, 33, 204, 114, 3, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '½' / 0xbd
        '½' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[74, 218, 164, 0, 0, 0, 0, 0, 0, 0],
            &[119, 179, 165, 0, 0, 1, 138, 111, 0, 0],
            &[0, 120, 165, 0, 0, 121, 239, 33, 0, 0],
            &[27, 187, 234, 36, 62, 251, 80, 0, 0, 0],
            &[98, 255, 255, 146, 230, 143, 0, 0, 0, 0],
            &[0, 0, 2, 187, 210, 149, 198, 111, 0, 0],
            &[0, 0, 127, 237, 34, 104, 109, 241, 0, 0],
            &[0, 67, 252, 77, 0, 3, 172, 163, 0, 0],
            &[23, 227, 138, 0, 2, 163, 188, 14, 0, 0],
            &[0, 0, 0, 0, 46, 255, 255, 255, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¾' / 0xbe
        '¾' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 16, 0, 0, 0, 0, 0, 0, 0],
            &[75, 234, 243, 107, 0, 0, 0, 0, 0, 0],
            &[6, 44, 172, 134, 0, 1, 138, 111, 0, 0],
            &[0, 142, 232, 89, 0, 121, 239, 33, 0, 0],
            &[65, 74, 160, 186, 62, 251, 80, 0, 0, 0],
            &[64, 183, 174, 73, 230, 143, 0, 0, 0, 0],
            &[0, 0, 2, 187, 200, 25, 176, 65, 0, 0],
            &[0, 0, 127, 237, 31, 167, 241, 93, 0, 0],
            &[0, 67, 252, 77, 81, 170, 196, 93, 0, 0],
            &[23, 227, 138, 0, 202, 219, 244, 224, 18, 0],
            &[0, 0, 0, 0, 27, 33, 204, 114, 3, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '¿' / 0xbf
        '¿' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 124, 16, 0, 0, 0],
            &[0, 0, 0, 0, 245, 255, 118, 0, 0, 0],
            &[0, 0, 0, 0, 163, 220, 59, 0, 0, 0],
            &[0, 0, 0, 0, 14, 22, 3, 0, 0, 0],
            &[0, 0, 0, 0, 177, 255, 35, 0, 0, 0],
            &[0, 0, 0, 101, 252, 218, 3, 0, 0, 0],
            &[0, 0, 130, 255, 212, 32, 0, 0, 0, 0],
            &[0, 23, 254, 235, 17, 0, 0, 0, 0, 0],
            &[0, 44, 255, 231, 20, 12, 86, 54, 0, 0],
            &[0, 3, 211, 255, 253, 254, 255, 188, 0, 0],
            &[0, 0, 19, 131, 179, 172, 116, 21, 0, 0],
        ]),
        // letter: 'À' / 0xc0
        'À' => Some(&[
            &[0, 5, 181, 255, 103, 0, 0, 0, 0, 0],
            &[0, 0, 6, 168, 242, 34, 0, 0, 0, 0],
            &[0, 0, 1, 10, 50, 22, 0, 0, 0, 0],
            &[0, 0, 63, 255, 255, 162, 0, 0, 0, 0],
            &[0, 0, 134, 255, 241, 232, 1, 0, 0, 0],
            &[0, 0, 204, 245, 150, 255, 48, 0, 0, 0],
            &[0, 20, 254, 188, 85, 255, 118, 0, 0, 0],
            &[0, 89, 255, 124, 24, 255, 189, 0, 0, 0],
            &[0, 159, 255, 100, 40, 245, 249, 11, 0, 0],
            &[0, 229, 255, 255, 255, 255, 255, 75, 0, 0],
            &[44, 255, 255, 133, 133, 205, 255, 145, 0, 0],
            &[114, 255, 124, 0, 0, 33, 255, 216, 0, 0],
            &[184, 255, 61, 0, 0, 0, 227, 255, 31, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 19, 19, 1, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Á' / 0xc1
        'Á' => Some(&[
            &[0, 0, 0, 0, 188, 254, 102, 0, 0, 0],
            &[0, 0, 0, 107, 250, 93, 0, 0, 0, 0],
            &[0, 0, 1, 41, 37, 4, 0, 0, 0, 0],
            &[0, 0, 63, 255, 255, 162, 0, 0, 0, 0],
            &[0, 0, 134, 255, 241, 232, 1, 0, 0, 0],
            &[0, 0, 204, 245, 150, 255, 48, 0, 0, 0],
            &[0, 20, 254, 188, 85, 255, 118, 0, 0, 0],
            &[0, 89, 255, 124, 24, 255, 189, 0, 0, 0],
            &[0, 159, 255, 100, 40, 245, 249, 11, 0, 0],
            &[0, 229, 255, 255, 255, 255, 255, 75, 0, 0],
            &[44, 255, 255, 133, 133, 205, 255, 145, 0, 0],
            &[114, 255, 124, 0, 0, 33, 255, 216, 0, 0],
            &[184, 255, 61, 0, 0, 0, 227, 255, 31, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 19, 16, 0, 0, 0],
        ]),
        // letter: 'Â' / 0xc2
        'Â' => Some(&[
            &[0, 0, 54, 252, 255, 159, 0, 0, 0, 0],
            &[0, 21, 223, 165, 81, 245, 95, 0, 0, 0],
            &[0, 15, 41, 10, 9, 28, 31, 0, 0, 0],
            &[0, 0, 63, 255, 255, 162, 0, 0, 0, 0],
            &[0, 0, 134, 255, 241, 232, 1, 0, 0, 0],
            &[0, 0, 204, 245, 150, 255, 48, 0, 0, 0],
            &[0, 20, 254, 188, 85, 255, 118, 0, 0, 0],
            &[0, 89, 255, 124, 24, 255, 189, 0, 0, 0],
            &[0, 159, 255, 100, 40, 245, 249, 11, 0, 0],
            &[0, 229, 255, 255, 255, 255, 255, 75, 0, 0],
            &[44, 255, 255, 133, 133, 205, 255, 145, 0, 0],
            &[114, 255, 124, 0, 0, 33, 255, 216, 0, 0],
            &[184, 255, 61, 0, 0, 0, 227, 255, 31, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 19, 5, 0, 0, 0, 0],
        ]),
        // letter: 'Ã' / 0xc3
        'Ã' => Some(&[
            &[0, 8, 181, 212, 114, 114, 187, 0, 0, 0],
            &[0, 85, 220, 142, 236, 255, 101, 0, 0, 0],
            &[0, 20, 27, 9, 16, 30, 0, 0, 0, 0],
            &[0, 0, 63, 255, 255, 162, 0, 0, 0, 0],
            &[0, 0, 134, 255, 241, 232, 1, 0, 0, 0],
            &[0, 0, 204, 245, 150, 255, 48, 0, 0, 0],
            &[0, 20, 254, 188, 85, 255, 118, 0, 0, 0],
            &[0, 89, 255, 124, 24, 255, 189, 0, 0, 0],
            &[0, 159, 255, 100, 40, 245, 249, 11, 0, 0],
            &[0, 229, 255, 255, 255, 255, 255, 75, 0, 0],
            &[44, 255, 255, 133, 133, 205, 255, 145, 0, 0],
            &[114, 255, 124, 0, 0, 33, 255, 216, 0, 0],
            &[184, 255, 61, 0, 0, 0, 227, 255, 31, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ä' / 0xc4
        'Ä' => Some(&[
            &[0, 54, 220, 76, 8, 208, 135, 0, 0, 0],
            &[0, 70, 251, 97, 14, 239, 166, 0, 0, 0],
            &[0, 0, 9, 9, 9, 12, 0, 0, 0, 0],
            &[0, 0, 63, 255, 255, 162, 0, 0, 0, 0],
            &[0, 0, 134, 255, 241, 232, 1, 0, 0, 0],
            &[0, 0, 204, 245, 150, 255, 48, 0, 0, 0],
            &[0, 20, 254, 188, 85, 255, 118, 0, 0, 0],
            &[0, 89, 255, 124, 24, 255, 189, 0, 0, 0],
            &[0, 159, 255, 100, 40, 245, 249, 11, 0, 0],
            &[0, 229, 255, 255, 255, 255, 255, 75, 0, 0],
            &[44, 255, 255, 133, 133, 205, 255, 145, 0, 0],
            &[114, 255, 124, 0, 0, 33, 255, 216, 0, 0],
            &[184, 255, 61, 0, 0, 0, 227, 255, 31, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Å' / 0xc5
        'Å' => Some(&[
            &[0, 0, 111, 162, 70, 209, 0, 0, 0, 0],
            &[0, 0, 46, 235, 234, 126, 0, 0, 0, 0],
            &[0, 0, 1, 22, 35, 4, 0, 0, 0, 0],
            &[0, 0, 63, 255, 255, 162, 0, 0, 0, 0],
            &[0, 0, 134, 255, 241, 232, 1, 0, 0, 0],
            &[0, 0, 204, 245, 150, 255, 48, 0, 0, 0],
            &[0, 20, 254, 188, 85, 255, 118, 0, 0, 0],
            &[0, 89, 255, 124, 24, 255, 189, 0, 0, 0],
            &[0, 159, 255, 100, 40, 245, 249, 11, 0, 0],
            &[0, 229, 255, 255, 255, 255, 255, 75, 0, 0],
            &[44, 255, 255, 133, 133, 205, 255, 145, 0, 0],
            &[114, 255, 124, 0, 0, 33, 255, 216, 0, 0],
            &[184, 255, 61, 0, 0, 0, 227, 255, 31, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 226, 243, 105, 0, 0, 0, 0],
        ]),
        // letter: 'Æ' / 0xc6
        'Æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 152, 254, 254, 254, 254, 249, 0],
            &[0, 0, 0, 213, 243, 234, 255, 190, 185, 0],
            &[0, 0, 20, 254, 171, 172, 255, 5, 0, 0],
            &[0, 0, 79, 255, 114, 172, 255, 20, 11, 0],
            &[0, 0, 140, 255, 58, 172, 255, 255, 186, 0],
            &[0, 0, 201, 255, 59, 208, 255, 174, 125, 0],
            &[0, 11, 251, 255, 255, 255, 255, 5, 0, 0],
            &[0, 67, 255, 254, 137, 255, 255, 5, 0, 0],
            &[0, 128, 255, 88, 0, 172, 255, 189, 184, 0],
            &[0, 189, 255, 32, 0, 172, 255, 255, 250, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ç' / 0xc7
        'Ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 28, 5, 0, 0],
            &[0, 0, 0, 61, 199, 254, 255, 250, 138, 0],
            &[0, 0, 60, 249, 255, 200, 171, 218, 105, 0],
            &[0, 0, 202, 255, 126, 0, 0, 0, 3, 0],
            &[0, 27, 255, 242, 8, 0, 0, 0, 0, 0],
            &[0, 64, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 69, 255, 200, 0, 0, 0, 0, 0, 0],
            &[0, 45, 255, 233, 0, 0, 0, 0, 0, 0],
            &[0, 2, 234, 255, 92, 0, 0, 0, 13, 0],
            &[0, 0, 110, 255, 253, 192, 173, 220, 114, 0],
            &[0, 0, 0, 113, 233, 255, 255, 232, 79, 0],
            &[0, 0, 0, 0, 19, 255, 106, 1, 0, 0],
            &[0, 0, 0, 0, 9, 131, 254, 33, 0, 0],
            &[0, 0, 0, 0, 55, 83, 28, 0, 0, 0],
        ]),
        // letter: 'È' / 0xc8
        'È' => Some(&[
            &[0, 1, 163, 255, 124, 0, 0, 0, 0, 0],
            &[0, 0, 2, 150, 249, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 21, 0, 0, 0, 0],
            &[0, 224, 254, 254, 254, 254, 254, 107, 0, 0],
            &[0, 225, 255, 197, 192, 192, 192, 81, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 35, 15, 15, 15, 3, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 53, 0, 0],
            &[0, 225, 255, 182, 176, 176, 176, 36, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 197, 191, 191, 191, 80, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 19, 19, 2, 0, 0, 0, 0, 0],
        ]),
        // letter: 'É' / 0xc9
        'É' => Some(&[
            &[0, 0, 0, 0, 167, 255, 122, 0, 0, 0],
            &[0, 0, 0, 85, 253, 111, 0, 0, 0, 0],
            &[0, 0, 0, 28, 31, 0, 0, 0, 0, 0],
            &[0, 224, 254, 254, 254, 254, 254, 107, 0, 0],
            &[0, 225, 255, 197, 192, 192, 192, 81, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 35, 15, 15, 15, 3, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 53, 0, 0],
            &[0, 225, 255, 182, 176, 176, 176, 36, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 197, 191, 191, 191, 80, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 19, 17, 0, 0, 0],
        ]),
        // letter: 'Ê' / 0xca
        'Ê' => Some(&[
            &[0, 0, 39, 246, 255, 179, 0, 0, 0, 0],
            &[0, 12, 210, 182, 71, 239, 117, 0, 0, 0],
            &[0, 11, 41, 3, 0, 20, 35, 0, 0, 0],
            &[0, 224, 254, 254, 254, 254, 254, 107, 0, 0],
            &[0, 225, 255, 197, 192, 192, 192, 81, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 35, 15, 15, 15, 3, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 53, 0, 0],
            &[0, 225, 255, 182, 176, 176, 176, 36, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 197, 191, 191, 191, 80, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 19, 6, 0, 0, 0, 0],
        ]),
        // letter: 'Ë' / 0xcb
        'Ë' => Some(&[
            &[0, 38, 219, 93, 1, 196, 155, 0, 0, 0],
            &[0, 50, 250, 117, 5, 227, 188, 0, 0, 0],
            &[0, 0, 8, 0, 0, 6, 1, 0, 0, 0],
            &[0, 224, 254, 254, 254, 254, 254, 107, 0, 0],
            &[0, 225, 255, 197, 192, 192, 192, 81, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 35, 15, 15, 15, 3, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 53, 0, 0],
            &[0, 225, 255, 182, 176, 176, 176, 36, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 197, 191, 191, 191, 80, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ì' / 0xcc
        'Ì' => Some(&[
            &[0, 5, 181, 255, 103, 0, 0, 0, 0, 0],
            &[0, 0, 6, 168, 242, 34, 0, 0, 0, 0],
            &[0, 0, 0, 1, 41, 18, 0, 0, 0, 0],
            &[0, 188, 254, 254, 254, 254, 254, 35, 0, 0],
            &[0, 84, 139, 235, 255, 164, 121, 14, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 83, 138, 235, 255, 163, 120, 14, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 19, 19, 1, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Í' / 0xcd
        'Í' => Some(&[
            &[0, 0, 0, 0, 188, 254, 102, 0, 0, 0],
            &[0, 0, 0, 107, 250, 93, 0, 0, 0, 0],
            &[0, 0, 0, 32, 28, 0, 0, 0, 0, 0],
            &[0, 188, 254, 254, 254, 254, 254, 35, 0, 0],
            &[0, 84, 139, 235, 255, 164, 121, 14, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 83, 138, 235, 255, 163, 120, 14, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 19, 16, 0, 0, 0],
        ]),
        // letter: 'Î' / 0xce
        'Î' => Some(&[
            &[0, 0, 54, 252, 255, 159, 0, 0, 0, 0],
            &[0, 21, 223, 165, 81, 245, 95, 0, 0, 0],
            &[0, 15, 40, 0, 0, 24, 31, 0, 0, 0],
            &[0, 188, 254, 254, 254, 254, 254, 35, 0, 0],
            &[0, 84, 139, 235, 255, 164, 121, 14, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 83, 138, 235, 255, 163, 120, 14, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 19, 5, 0, 0, 0, 0],
        ]),
        // letter: 'Ï' / 0xcf
        'Ï' => Some(&[
            &[0, 54, 220, 76, 8, 208, 135, 0, 0, 0],
            &[0, 70, 251, 97, 14, 239, 166, 0, 0, 0],
            &[0, 0, 8, 0, 0, 7, 0, 0, 0, 0],
            &[0, 188, 254, 254, 254, 254, 254, 35, 0, 0],
            &[0, 84, 139, 235, 255, 164, 121, 14, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 83, 138, 235, 255, 163, 120, 14, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ð' / 0xd0
        'Ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 254, 254, 253, 232, 162, 28, 0, 0],
            &[0, 94, 255, 230, 187, 227, 255, 229, 20, 0],
            &[0, 94, 255, 161, 0, 3, 179, 255, 139, 0],
            &[0, 94, 255, 161, 0, 0, 62, 255, 213, 0],
            &[0, 255, 255, 255, 200, 23, 27, 255, 241, 0],
            &[0, 255, 255, 255, 229, 26, 34, 255, 237, 0],
            &[0, 94, 255, 161, 0, 0, 74, 255, 205, 0],
            &[0, 94, 255, 161, 0, 12, 195, 255, 124, 0],
            &[0, 94, 255, 229, 191, 237, 255, 215, 11, 0],
            &[0, 94, 255, 255, 253, 227, 148, 18, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ñ' / 0xd1
        'Ñ' => Some(&[
            &[0, 0, 2, 171, 216, 121, 102, 204, 0, 0],
            &[0, 0, 67, 228, 138, 231, 255, 119, 0, 0],
            &[0, 0, 17, 29, 0, 4, 27, 0, 0, 0],
            &[0, 94, 254, 254, 140, 0, 9, 254, 196, 0],
            &[0, 94, 255, 253, 223, 0, 9, 255, 197, 0],
            &[0, 94, 255, 199, 255, 52, 9, 255, 197, 0],
            &[0, 94, 255, 131, 255, 135, 9, 255, 197, 0],
            &[0, 94, 255, 109, 204, 219, 9, 255, 197, 0],
            &[0, 94, 255, 111, 121, 255, 54, 255, 197, 0],
            &[0, 94, 255, 111, 38, 255, 133, 255, 197, 0],
            &[0, 94, 255, 111, 0, 211, 214, 247, 197, 0],
            &[0, 94, 255, 111, 0, 128, 255, 255, 197, 0],
            &[0, 94, 255, 111, 0, 45, 255, 255, 197, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ò' / 0xd2
        'Ò' => Some(&[
            &[0, 0, 5, 181, 255, 103, 0, 0, 0, 0],
            &[0, 0, 0, 6, 168, 242, 34, 0, 0, 0],
            &[0, 0, 0, 0, 20, 74, 22, 0, 0, 0],
            &[0, 0, 29, 195, 255, 255, 236, 95, 0, 0],
            &[0, 1, 205, 255, 205, 173, 252, 254, 59, 0],
            &[0, 60, 255, 223, 6, 0, 133, 255, 166, 0],
            &[0, 117, 255, 152, 0, 0, 53, 255, 221, 0],
            &[0, 140, 255, 127, 0, 0, 28, 255, 244, 0],
            &[0, 142, 255, 127, 0, 0, 25, 255, 244, 0],
            &[0, 118, 255, 153, 0, 0, 52, 255, 220, 0],
            &[0, 62, 255, 224, 8, 0, 132, 255, 164, 0],
            &[0, 2, 205, 255, 210, 175, 252, 253, 56, 0],
            &[0, 0, 30, 191, 255, 255, 232, 90, 0, 0],
            &[0, 0, 0, 0, 16, 27, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 19, 19, 1, 0, 0, 0, 0],
        ]),
        // letter: 'Ó' / 0xd3
        'Ó' => Some(&[
            &[0, 0, 0, 0, 0, 188, 254, 102, 0, 0],
            &[0, 0, 0, 0, 107, 250, 93, 0, 0, 0],
            &[0, 0, 0, 0, 51, 61, 4, 0, 0, 0],
            &[0, 0, 29, 195, 255, 255, 236, 95, 0, 0],
            &[0, 1, 205, 255, 205, 173, 252, 254, 59, 0],
            &[0, 60, 255, 223, 6, 0, 133, 255, 166, 0],
            &[0, 117, 255, 152, 0, 0, 53, 255, 221, 0],
            &[0, 140, 255, 127, 0, 0, 28, 255, 244, 0],
            &[0, 142, 255, 127, 0, 0, 25, 255, 244, 0],
            &[0, 118, 255, 153, 0, 0, 52, 255, 220, 0],
            &[0, 62, 255, 224, 8, 0, 132, 255, 164, 0],
            &[0, 2, 205, 255, 210, 175, 252, 253, 56, 0],
            &[0, 0, 30, 191, 255, 255, 232, 90, 0, 0],
            &[0, 0, 0, 0, 16, 27, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 19, 16, 0, 0],
        ]),
        // letter: 'Ô' / 0xd4
        'Ô' => Some(&[
            &[0, 0, 0, 54, 252, 255, 159, 0, 0, 0],
            &[0, 0, 21, 223, 165, 81, 245, 95, 0, 0],
            &[0, 0, 15, 40, 19, 33, 28, 31, 0, 0],
            &[0, 0, 29, 195, 255, 255, 236, 95, 0, 0],
            &[0, 1, 205, 255, 205, 173, 252, 254, 59, 0],
            &[0, 60, 255, 223, 6, 0, 133, 255, 166, 0],
            &[0, 117, 255, 152, 0, 0, 53, 255, 221, 0],
            &[0, 140, 255, 127, 0, 0, 28, 255, 244, 0],
            &[0, 142, 255, 127, 0, 0, 25, 255, 244, 0],
            &[0, 118, 255, 153, 0, 0, 52, 255, 220, 0],
            &[0, 62, 255, 224, 8, 0, 132, 255, 164, 0],
            &[0, 2, 205, 255, 210, 175, 252, 253, 56, 0],
            &[0, 0, 30, 191, 255, 255, 232, 90, 0, 0],
            &[0, 0, 0, 0, 16, 27, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 19, 5, 0, 0, 0],
        ]),
        // letter: 'Õ' / 0xd5
        'Õ' => Some(&[
            &[0, 0, 8, 181, 212, 114, 114, 187, 0, 0],
            &[0, 0, 85, 220, 142, 236, 255, 101, 0, 0],
            &[0, 0, 20, 26, 18, 40, 30, 0, 0, 0],
            &[0, 0, 29, 195, 255, 255, 236, 95, 0, 0],
            &[0, 1, 205, 255, 205, 173, 252, 254, 59, 0],
            &[0, 60, 255, 223, 6, 0, 133, 255, 166, 0],
            &[0, 117, 255, 152, 0, 0, 53, 255, 221, 0],
            &[0, 140, 255, 127, 0, 0, 28, 255, 244, 0],
            &[0, 142, 255, 127, 0, 0, 25, 255, 244, 0],
            &[0, 118, 255, 153, 0, 0, 52, 255, 220, 0],
            &[0, 62, 255, 224, 8, 0, 132, 255, 164, 0],
            &[0, 2, 205, 255, 210, 175, 252, 253, 56, 0],
            &[0, 0, 30, 191, 255, 255, 232, 90, 0, 0],
            &[0, 0, 0, 0, 16, 27, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ö' / 0xd6
        'Ö' => Some(&[
            &[0, 0, 54, 220, 76, 8, 208, 135, 0, 0],
            &[0, 0, 70, 251, 97, 14, 239, 166, 0, 0],
            &[0, 0, 0, 8, 18, 33, 12, 0, 0, 0],
            &[0, 0, 29, 195, 255, 255, 236, 95, 0, 0],
            &[0, 1, 205, 255, 205, 173, 252, 254, 59, 0],
            &[0, 60, 255, 223, 6, 0, 133, 255, 166, 0],
            &[0, 117, 255, 152, 0, 0, 53, 255, 221, 0],
            &[0, 140, 255, 127, 0, 0, 28, 255, 244, 0],
            &[0, 142, 255, 127, 0, 0, 25, 255, 244, 0],
            &[0, 118, 255, 153, 0, 0, 52, 255, 220, 0],
            &[0, 62, 255, 224, 8, 0, 132, 255, 164, 0],
            &[0, 2, 205, 255, 210, 175, 252, 253, 56, 0],
            &[0, 0, 30, 191, 255, 255, 232, 90, 0, 0],
            &[0, 0, 0, 0, 16, 27, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '×' / 0xd7
        '×' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 86, 173, 6, 0, 85, 177, 7, 0, 0],
            &[0, 114, 255, 181, 97, 252, 199, 13, 0, 0],
            &[0, 0, 115, 255, 255, 199, 14, 0, 0, 0],
            &[0, 0, 87, 253, 255, 177, 5, 0, 0, 0],
            &[0, 84, 251, 198, 121, 255, 174, 4, 0, 0],
            &[0, 111, 197, 13, 0, 112, 199, 13, 0, 0],
            &[0, 0, 4, 0, 0, 0, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ø' / 0xd8
        'Ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 33, 4, 6, 68, 0, 0],
            &[0, 29, 195, 255, 255, 236, 227, 236, 14, 0],
            &[1, 205, 255, 205, 173, 255, 255, 149, 0, 0],
            &[60, 255, 223, 6, 7, 255, 255, 166, 0, 0],
            &[117, 255, 152, 0, 135, 255, 255, 221, 0, 0],
            &[140, 255, 127, 55, 252, 116, 255, 244, 0, 0],
            &[142, 255, 136, 215, 172, 25, 255, 244, 0, 0],
            &[118, 255, 255, 235, 22, 52, 255, 220, 0, 0],
            &[62, 255, 255, 94, 0, 132, 255, 164, 0, 0],
            &[12, 255, 255, 210, 175, 252, 253, 56, 0, 0],
            &[140, 255, 212, 255, 255, 232, 90, 0, 0, 0],
            &[34, 65, 0, 16, 27, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ù' / 0xd9
        'Ù' => Some(&[
            &[0, 0, 5, 181, 255, 103, 0, 0, 0, 0],
            &[0, 0, 0, 6, 168, 242, 34, 0, 0, 0],
            &[0, 0, 0, 0, 1, 41, 18, 0, 0, 0],
            &[0, 94, 254, 168, 0, 0, 66, 254, 196, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 92, 255, 178, 0, 0, 75, 255, 194, 0],
            &[0, 68, 255, 224, 7, 0, 127, 255, 169, 0],
            &[0, 8, 230, 255, 208, 177, 248, 255, 79, 0],
            &[0, 0, 51, 208, 255, 255, 237, 115, 0, 0],
            &[0, 0, 0, 0, 18, 26, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 19, 19, 1, 0, 0, 0, 0],
        ]),
        // letter: 'Ú' / 0xda
        'Ú' => Some(&[
            &[0, 0, 0, 0, 0, 188, 254, 102, 0, 0],
            &[0, 0, 0, 0, 107, 250, 93, 0, 0, 0],
            &[0, 0, 0, 0, 32, 28, 0, 0, 0, 0],
            &[0, 94, 254, 168, 0, 0, 66, 254, 196, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 92, 255, 178, 0, 0, 75, 255, 194, 0],
            &[0, 68, 255, 224, 7, 0, 127, 255, 169, 0],
            &[0, 8, 230, 255, 208, 177, 248, 255, 79, 0],
            &[0, 0, 51, 208, 255, 255, 237, 115, 0, 0],
            &[0, 0, 0, 0, 18, 26, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 19, 16, 0, 0],
        ]),
        // letter: 'Û' / 0xdb
        'Û' => Some(&[
            &[0, 0, 0, 54, 252, 255, 159, 0, 0, 0],
            &[0, 0, 21, 223, 165, 81, 245, 95, 0, 0],
            &[0, 0, 15, 40, 0, 0, 24, 31, 0, 0],
            &[0, 94, 254, 168, 0, 0, 66, 254, 196, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 92, 255, 178, 0, 0, 75, 255, 194, 0],
            &[0, 68, 255, 224, 7, 0, 127, 255, 169, 0],
            &[0, 8, 230, 255, 208, 177, 248, 255, 79, 0],
            &[0, 0, 51, 208, 255, 255, 237, 115, 0, 0],
            &[0, 0, 0, 0, 18, 26, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 19, 5, 0, 0, 0],
        ]),
        // letter: 'Ü' / 0xdc
        'Ü' => Some(&[
            &[0, 0, 54, 220, 76, 8, 208, 135, 0, 0],
            &[0, 0, 70, 251, 97, 14, 239, 166, 0, 0],
            &[0, 0, 0, 8, 0, 0, 7, 0, 0, 0],
            &[0, 94, 254, 168, 0, 0, 66, 254, 196, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 92, 255, 178, 0, 0, 75, 255, 194, 0],
            &[0, 68, 255, 224, 7, 0, 127, 255, 169, 0],
            &[0, 8, 230, 255, 208, 177, 248, 255, 79, 0],
            &[0, 0, 51, 208, 255, 255, 237, 115, 0, 0],
            &[0, 0, 0, 0, 18, 26, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ý' / 0xdd
        'Ý' => Some(&[
            &[0, 0, 0, 0, 188, 254, 102, 0, 0, 0],
            &[0, 0, 0, 107, 250, 93, 0, 0, 0, 0],
            &[0, 0, 0, 32, 28, 0, 0, 0, 0, 0],
            &[123, 254, 161, 0, 0, 55, 254, 221, 3, 0],
            &[16, 241, 248, 21, 0, 162, 255, 105, 0, 0],
            &[0, 136, 255, 121, 21, 248, 231, 8, 0, 0],
            &[0, 23, 246, 226, 124, 255, 118, 0, 0, 0],
            &[0, 0, 148, 255, 249, 239, 13, 0, 0, 0],
            &[0, 0, 31, 251, 255, 132, 0, 0, 0, 0],
            &[0, 0, 0, 207, 255, 57, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 19, 16, 0, 0, 0],
        ]),
        // letter: 'Þ' / 0xde
        'Þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 224, 254, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 94, 78, 52, 2, 0, 0, 0],
            &[0, 225, 255, 255, 255, 255, 212, 21, 0, 0],
            &[0, 225, 255, 116, 111, 206, 255, 148, 0, 0],
            &[0, 225, 255, 20, 0, 61, 255, 201, 0, 0],
            &[0, 225, 255, 20, 0, 90, 255, 195, 0, 0],
            &[0, 225, 255, 161, 171, 244, 255, 116, 0, 0],
            &[0, 225, 255, 255, 255, 244, 145, 4, 0, 0],
            &[0, 225, 255, 49, 26, 4, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ß' / 0xdf
        'ß' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 106, 168, 172, 120, 16, 0, 0],
            &[0, 0, 178, 255, 254, 252, 255, 209, 4, 0],
            &[0, 32, 255, 238, 29, 16, 233, 255, 39, 0],
            &[0, 58, 255, 193, 0, 32, 243, 233, 8, 0],
            &[0, 58, 255, 190, 19, 224, 241, 58, 0, 0],
            &[0, 58, 255, 190, 88, 255, 176, 0, 0, 0],
            &[0, 58, 255, 190, 34, 245, 255, 160, 6, 0],
            &[0, 58, 255, 190, 0, 54, 225, 255, 158, 0],
            &[0, 58, 255, 190, 0, 0, 33, 255, 241, 0],
            &[0, 58, 255, 190, 130, 141, 159, 255, 223, 0],
            &[0, 58, 255, 190, 127, 254, 255, 234, 75, 0],
            &[0, 0, 0, 0, 0, 15, 28, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'à' / 0xe0
        'à' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 149, 185, 70, 0, 0, 0, 0],
            &[0, 0, 0, 28, 215, 226, 15, 0, 0, 0],
            &[0, 0, 0, 0, 18, 127, 61, 0, 0, 0],
            &[0, 0, 55, 146, 183, 188, 149, 42, 0, 0],
            &[0, 0, 141, 254, 229, 230, 255, 241, 33, 0],
            &[0, 0, 18, 26, 0, 0, 152, 255, 105, 0],
            &[0, 0, 27, 131, 174, 187, 220, 255, 125, 0],
            &[0, 12, 230, 254, 170, 126, 179, 255, 125, 0],
            &[0, 66, 255, 191, 0, 0, 146, 255, 125, 0],
            &[0, 44, 255, 244, 140, 157, 253, 255, 125, 0],
            &[0, 0, 147, 254, 255, 209, 97, 255, 125, 0],
            &[0, 0, 0, 12, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'á' / 0xe1
        'á' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 90, 185, 131, 0, 0],
            &[0, 0, 0, 0, 30, 240, 198, 17, 0, 0],
            &[0, 0, 0, 0, 76, 121, 9, 0, 0, 0],
            &[0, 0, 55, 146, 183, 188, 149, 42, 0, 0],
            &[0, 0, 141, 254, 229, 230, 255, 241, 33, 0],
            &[0, 0, 18, 26, 0, 0, 152, 255, 105, 0],
            &[0, 0, 27, 131, 174, 187, 220, 255, 125, 0],
            &[0, 12, 230, 254, 170, 126, 179, 255, 125, 0],
            &[0, 66, 255, 191, 0, 0, 146, 255, 125, 0],
            &[0, 44, 255, 244, 140, 157, 253, 255, 125, 0],
            &[0, 0, 147, 254, 255, 209, 97, 255, 125, 0],
            &[0, 0, 0, 12, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'â' / 0xe2
        'â' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 168, 185, 108, 0, 0, 0],
            &[0, 0, 0, 144, 239, 158, 250, 58, 0, 0],
            &[0, 0, 21, 131, 41, 0, 90, 103, 0, 0],
            &[0, 0, 55, 146, 183, 188, 149, 42, 0, 0],
            &[0, 0, 141, 254, 229, 230, 255, 241, 33, 0],
            &[0, 0, 18, 26, 0, 0, 152, 255, 105, 0],
            &[0, 0, 27, 131, 174, 187, 220, 255, 125, 0],
            &[0, 12, 230, 254, 170, 126, 179, 255, 125, 0],
            &[0, 66, 255, 191, 0, 0, 146, 255, 125, 0],
            &[0, 44, 255, 244, 140, 157, 253, 255, 125, 0],
            &[0, 0, 147, 254, 255, 209, 97, 255, 125, 0],
            &[0, 0, 0, 12, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ã' / 0xe3
        'ã' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 128, 38, 21, 138, 0, 0],
            &[0, 0, 31, 250, 226, 254, 240, 183, 0, 0],
            &[0, 0, 41, 101, 0, 65, 118, 23, 0, 0],
            &[0, 0, 55, 146, 183, 188, 149, 42, 0, 0],
            &[0, 0, 141, 254, 229, 230, 255, 241, 33, 0],
            &[0, 0, 18, 26, 0, 0, 152, 255, 105, 0],
            &[0, 0, 27, 131, 174, 187, 220, 255, 125, 0],
            &[0, 12, 230, 254, 170, 126, 179, 255, 125, 0],
            &[0, 66, 255, 191, 0, 0, 146, 255, 125, 0],
            &[0, 44, 255, 244, 140, 157, 253, 255, 125, 0],
            &[0, 0, 147, 254, 255, 209, 97, 255, 125, 0],
            &[0, 0, 0, 12, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ä' / 0xe4
        'ä' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 128, 46, 0, 103, 84, 0, 0],
            &[0, 0, 61, 255, 149, 3, 246, 220, 0, 0],
            &[0, 0, 4, 92, 28, 0, 69, 55, 0, 0],
            &[0, 0, 55, 146, 183, 188, 149, 42, 0, 0],
            &[0, 0, 141, 254, 229, 230, 255, 241, 33, 0],
            &[0, 0, 18, 26, 0, 0, 152, 255, 105, 0],
            &[0, 0, 27, 131, 174, 187, 220, 255, 125, 0],
            &[0, 12, 230, 254, 170, 126, 179, 255, 125, 0],
            &[0, 66, 255, 191, 0, 0, 146, 255, 125, 0],
            &[0, 44, 255, 244, 140, 157, 253, 255, 125, 0],
            &[0, 0, 147, 254, 255, 209, 97, 255, 125, 0],
            &[0, 0, 0, 12, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'å' / 0xe5
        'å' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 124, 167, 58, 0, 0, 0],
            &[0, 0, 0, 72, 214, 112, 226, 0, 0, 0],
            &[0, 0, 0, 55, 229, 156, 208, 0, 0, 0],
            &[0, 0, 0, 0, 76, 119, 29, 0, 0, 0],
            &[0, 0, 55, 146, 183, 188, 149, 42, 0, 0],
            &[0, 0, 141, 254, 229, 230, 255, 241, 33, 0],
            &[0, 0, 18, 26, 0, 0, 152, 255, 105, 0],
            &[0, 0, 27, 131, 174, 187, 220, 255, 125, 0],
            &[0, 12, 230, 254, 170, 126, 179, 255, 125, 0],
            &[0, 66, 255, 191, 0, 0, 146, 255, 125, 0],
            &[0, 44, 255, 244, 140, 157, 253, 255, 125, 0],
            &[0, 0, 147, 254, 255, 209, 97, 255, 125, 0],
            &[0, 0, 0, 12, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'æ' / 0xe6
        'æ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[37, 154, 191, 149, 74, 176, 175, 40, 0, 0],
            &[39, 245, 232, 255, 255, 246, 253, 212, 0, 0],
            &[0, 7, 0, 189, 255, 71, 146, 255, 36, 0],
            &[7, 114, 163, 225, 255, 174, 204, 255, 62, 0],
            &[147, 255, 174, 208, 255, 175, 167, 167, 43, 0],
            &[212, 249, 5, 175, 255, 62, 0, 0, 0, 0],
            &[196, 255, 140, 244, 254, 221, 119, 153, 0, 0],
            &[81, 249, 254, 134, 99, 243, 255, 205, 0, 0],
            &[0, 11, 20, 0, 0, 8, 25, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ç' / 0xe7
        'ç' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 153, 191, 185, 138, 20, 0],
            &[0, 0, 60, 248, 255, 252, 251, 242, 9, 0],
            &[0, 0, 188, 255, 160, 13, 4, 40, 0, 0],
            &[0, 0, 243, 255, 39, 0, 0, 0, 0, 0],
            &[0, 1, 253, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 224, 255, 91, 0, 0, 10, 2, 0],
            &[0, 0, 133, 255, 247, 174, 168, 238, 10, 0],
            &[0, 0, 7, 149, 245, 255, 255, 206, 6, 0],
            &[0, 0, 0, 0, 38, 255, 80, 0, 0, 0],
            &[0, 0, 0, 0, 14, 148, 249, 16, 0, 0],
            &[0, 0, 0, 0, 62, 82, 22, 0, 0, 0],
        ]),
        // letter: 'è' / 0xe8
        'è' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 145, 185, 75, 0, 0, 0, 0],
            &[0, 0, 0, 25, 211, 230, 18, 0, 0, 0],
            &[0, 0, 0, 0, 16, 126, 65, 0, 0, 0],
            &[0, 0, 0, 85, 174, 192, 151, 35, 0, 0],
            &[0, 0, 123, 255, 243, 213, 253, 237, 28, 0],
            &[0, 13, 247, 241, 29, 0, 109, 255, 130, 0],
            &[0, 61, 255, 232, 158, 158, 174, 255, 171, 0],
            &[0, 66, 255, 238, 196, 196, 196, 196, 137, 0],
            &[0, 26, 253, 235, 17, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 230, 156, 148, 193, 117, 0],
            &[0, 0, 9, 144, 240, 255, 255, 233, 82, 0],
            &[0, 0, 0, 0, 3, 25, 22, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'é' / 0xe9
        'é' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 85, 185, 136, 0, 0],
            &[0, 0, 0, 0, 26, 237, 202, 19, 0, 0],
            &[0, 0, 0, 0, 72, 123, 11, 0, 0, 0],
            &[0, 0, 0, 85, 174, 192, 151, 35, 0, 0],
            &[0, 0, 123, 255, 243, 213, 253, 237, 28, 0],
            &[0, 13, 247, 241, 29, 0, 109, 255, 130, 0],
            &[0, 61, 255, 232, 158, 158, 174, 255, 171, 0],
            &[0, 66, 255, 238, 196, 196, 196, 196, 137, 0],
            &[0, 26, 253, 235, 17, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 230, 156, 148, 193, 117, 0],
            &[0, 0, 9, 144, 240, 255, 255, 233, 82, 0],
            &[0, 0, 0, 0, 3, 25, 22, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ê' / 0xea
        'ê' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 165, 185, 113, 0, 0, 0],
            &[0, 0, 0, 137, 241, 156, 252, 64, 0, 0],
            &[0, 0, 17, 131, 45, 0, 86, 107, 0, 0],
            &[0, 0, 0, 85, 174, 192, 151, 35, 0, 0],
            &[0, 0, 123, 255, 243, 213, 253, 237, 28, 0],
            &[0, 13, 247, 241, 29, 0, 109, 255, 130, 0],
            &[0, 61, 255, 232, 158, 158, 174, 255, 171, 0],
            &[0, 66, 255, 238, 196, 196, 196, 196, 137, 0],
            &[0, 26, 253, 235, 17, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 230, 156, 148, 193, 117, 0],
            &[0, 0, 9, 144, 240, 255, 255, 233, 82, 0],
            &[0, 0, 0, 0, 3, 25, 22, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ë' / 0xeb
        'ë' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 127, 50, 0, 99, 88, 0, 0],
            &[0, 0, 54, 255, 156, 1, 241, 227, 0, 0],
            &[0, 0, 3, 91, 30, 0, 66, 58, 0, 0],
            &[0, 0, 0, 85, 174, 192, 151, 35, 0, 0],
            &[0, 0, 123, 255, 243, 213, 253, 237, 28, 0],
            &[0, 13, 247, 241, 29, 0, 109, 255, 130, 0],
            &[0, 61, 255, 232, 158, 158, 174, 255, 171, 0],
            &[0, 66, 255, 238, 196, 196, 196, 196, 137, 0],
            &[0, 26, 253, 235, 17, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 230, 156, 148, 193, 117, 0],
            &[0, 0, 9, 144, 240, 255, 255, 233, 82, 0],
            &[0, 0, 0, 0, 3, 25, 22, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ì' / 0xec
        'ì' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 163, 185, 49, 0, 0, 0, 0],
            &[0, 0, 0, 43, 229, 207, 6, 0, 0, 0],
            &[0, 0, 0, 0, 30, 131, 46, 0, 0, 0],
            &[0, 0, 128, 164, 164, 164, 68, 0, 0, 0],
            &[0, 0, 168, 240, 255, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 11, 94, 136, 220, 255, 200, 123, 58, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'í' / 0xed
        'í' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 111, 185, 110, 0, 0],
            &[0, 0, 0, 0, 49, 250, 178, 8, 0, 0],
            &[0, 0, 0, 0, 90, 112, 3, 0, 0, 0],
            &[0, 0, 128, 164, 164, 164, 68, 0, 0, 0],
            &[0, 0, 168, 240, 255, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 11, 94, 136, 220, 255, 200, 123, 58, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'î' / 0xee
        'î' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 178, 185, 87, 0, 0, 0],
            &[0, 0, 0, 172, 227, 170, 242, 38, 0, 0],
            &[0, 0, 35, 130, 28, 1, 103, 89, 0, 0],
            &[0, 0, 128, 164, 164, 164, 68, 0, 0, 0],
            &[0, 0, 168, 240, 255, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 11, 94, 136, 220, 255, 200, 123, 58, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ï' / 0xef
        'ï' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 130, 33, 0, 118, 68, 0, 0],
            &[0, 0, 90, 255, 120, 23, 255, 191, 0, 0],
            &[0, 0, 11, 94, 19, 0, 81, 43, 0, 0],
            &[0, 0, 128, 164, 164, 164, 68, 0, 0, 0],
            &[0, 0, 168, 240, 255, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 11, 94, 136, 220, 255, 200, 123, 58, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ð' / 0xf0
        'ð' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 45, 113, 8, 59, 83, 0, 0],
            &[0, 0, 0, 109, 255, 255, 255, 139, 0, 0],
            &[0, 0, 0, 95, 255, 255, 255, 45, 0, 0],
            &[0, 0, 0, 109, 109, 52, 243, 213, 5, 0],
            &[0, 0, 15, 137, 192, 175, 184, 255, 88, 0],
            &[0, 3, 204, 255, 222, 201, 253, 255, 166, 0],
            &[0, 69, 255, 216, 6, 0, 125, 255, 204, 0],
            &[0, 105, 255, 164, 0, 0, 62, 255, 207, 0],
            &[0, 85, 255, 193, 0, 0, 100, 255, 167, 0],
            &[0, 13, 233, 255, 145, 113, 231, 254, 62, 0],
            &[0, 0, 49, 203, 255, 255, 230, 91, 0, 0],
            &[0, 0, 0, 0, 18, 27, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ñ' / 0xf1
        'ñ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 83, 125, 33, 30, 129, 0, 0],
            &[0, 0, 43, 251, 228, 253, 241, 168, 0, 0],
            &[0, 0, 48, 94, 1, 71, 117, 18, 0, 0],
            &[0, 37, 164, 69, 86, 181, 182, 101, 0, 0],
            &[0, 58, 255, 228, 255, 242, 255, 255, 87, 0],
            &[0, 58, 255, 255, 76, 0, 155, 255, 160, 0],
            &[0, 58, 255, 229, 0, 0, 88, 255, 179, 0],
            &[0, 58, 255, 206, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ò' / 0xf2
        'ò' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 163, 185, 49, 0, 0, 0, 0],
            &[0, 0, 0, 43, 229, 207, 6, 0, 0, 0],
            &[0, 0, 0, 0, 30, 131, 46, 0, 0, 0],
            &[0, 0, 0, 95, 177, 191, 143, 28, 0, 0],
            &[0, 0, 146, 255, 250, 240, 255, 232, 25, 0],
            &[0, 35, 255, 243, 33, 4, 171, 255, 141, 0],
            &[0, 94, 255, 177, 0, 0, 78, 255, 198, 0],
            &[0, 106, 255, 170, 0, 0, 68, 255, 208, 0],
            &[0, 68, 255, 222, 6, 0, 123, 255, 170, 0],
            &[0, 3, 215, 255, 196, 153, 245, 255, 65, 0],
            &[0, 0, 34, 193, 255, 255, 233, 96, 0, 0],
            &[0, 0, 0, 0, 17, 28, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ó' / 0xf3
        'ó' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 111, 185, 110, 0, 0],
            &[0, 0, 0, 0, 49, 250, 178, 8, 0, 0],
            &[0, 0, 0, 0, 90, 112, 3, 0, 0, 0],
            &[0, 0, 0, 95, 177, 191, 143, 28, 0, 0],
            &[0, 0, 146, 255, 250, 240, 255, 232, 25, 0],
            &[0, 35, 255, 243, 33, 4, 171, 255, 141, 0],
            &[0, 94, 255, 177, 0, 0, 78, 255, 198, 0],
            &[0, 106, 255, 170, 0, 0, 68, 255, 208, 0],
            &[0, 68, 255, 222, 6, 0, 123, 255, 170, 0],
            &[0, 3, 215, 255, 196, 153, 245, 255, 65, 0],
            &[0, 0, 34, 193, 255, 255, 233, 96, 0, 0],
            &[0, 0, 0, 0, 17, 28, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ô' / 0xf4
        'ô' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 178, 185, 87, 0, 0, 0],
            &[0, 0, 0, 172, 227, 170, 242, 38, 0, 0],
            &[0, 0, 35, 130, 28, 1, 103, 89, 0, 0],
            &[0, 0, 0, 95, 177, 191, 143, 28, 0, 0],
            &[0, 0, 146, 255, 250, 240, 255, 232, 25, 0],
            &[0, 35, 255, 243, 33, 4, 171, 255, 141, 0],
            &[0, 94, 255, 177, 0, 0, 78, 255, 198, 0],
            &[0, 106, 255, 170, 0, 0, 68, 255, 208, 0],
            &[0, 68, 255, 222, 6, 0, 123, 255, 170, 0],
            &[0, 3, 215, 255, 196, 153, 245, 255, 65, 0],
            &[0, 0, 34, 193, 255, 255, 233, 96, 0, 0],
            &[0, 0, 0, 0, 17, 28, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'õ' / 0xf5
        'õ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 122, 28, 38, 121, 0, 0],
            &[0, 0, 57, 250, 230, 251, 242, 154, 0, 0],
            &[0, 0, 56, 86, 2, 76, 115, 13, 0, 0],
            &[0, 0, 0, 95, 177, 191, 143, 28, 0, 0],
            &[0, 0, 146, 255, 250, 240, 255, 232, 25, 0],
            &[0, 35, 255, 243, 33, 4, 171, 255, 141, 0],
            &[0, 94, 255, 177, 0, 0, 78, 255, 198, 0],
            &[0, 106, 255, 170, 0, 0, 68, 255, 208, 0],
            &[0, 68, 255, 222, 6, 0, 123, 255, 170, 0],
            &[0, 3, 215, 255, 196, 153, 245, 255, 65, 0],
            &[0, 0, 34, 193, 255, 255, 233, 96, 0, 0],
            &[0, 0, 0, 0, 17, 28, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ö' / 0xf6
        'ö' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 130, 33, 0, 118, 68, 0, 0],
            &[0, 0, 90, 255, 120, 23, 255, 191, 0, 0],
            &[0, 0, 11, 94, 19, 0, 81, 43, 0, 0],
            &[0, 0, 0, 95, 177, 191, 143, 28, 0, 0],
            &[0, 0, 146, 255, 250, 240, 255, 232, 25, 0],
            &[0, 35, 255, 243, 33, 4, 171, 255, 141, 0],
            &[0, 94, 255, 177, 0, 0, 78, 255, 198, 0],
            &[0, 106, 255, 170, 0, 0, 68, 255, 208, 0],
            &[0, 68, 255, 222, 6, 0, 123, 255, 170, 0],
            &[0, 3, 215, 255, 196, 153, 245, 255, 65, 0],
            &[0, 0, 34, 193, 255, 255, 233, 96, 0, 0],
            &[0, 0, 0, 0, 17, 28, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: '÷' / 0xf7
        '÷' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 45, 95, 0, 0, 0, 0],
            &[0, 0, 0, 0, 167, 255, 17, 0, 0, 0],
            &[0, 0, 0, 0, 55, 113, 0, 0, 0, 0],
            &[0, 3, 175, 175, 175, 175, 175, 175, 73, 0],
            &[0, 4, 204, 204, 204, 204, 204, 204, 85, 0],
            &[0, 0, 0, 0, 39, 85, 0, 0, 0, 0],
            &[0, 0, 0, 0, 167, 255, 17, 0, 0, 0],
            &[0, 0, 0, 0, 62, 126, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ø' / 0xf8
        'ø' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 0, 0],
            &[0, 0, 0, 95, 177, 191, 166, 237, 37, 0],
            &[0, 0, 146, 255, 250, 240, 255, 255, 30, 0],
            &[0, 35, 255, 243, 33, 97, 255, 255, 141, 0],
            &[0, 94, 255, 177, 27, 239, 197, 255, 198, 0],
            &[0, 106, 255, 171, 183, 198, 71, 255, 208, 0],
            &[0, 68, 255, 255, 252, 37, 123, 255, 170, 0],
            &[0, 3, 246, 255, 255, 153, 245, 255, 65, 0],
            &[0, 0, 216, 255, 255, 255, 233, 96, 0, 0],
            &[0, 0, 42, 30, 17, 28, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ù' / 0xf9
        'ù' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 163, 185, 49, 0, 0, 0, 0],
            &[0, 0, 0, 43, 229, 207, 6, 0, 0, 0],
            &[0, 0, 0, 0, 30, 131, 46, 0, 0, 0],
            &[0, 49, 164, 120, 0, 0, 65, 164, 103, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 187, 0, 0, 116, 255, 161, 0],
            &[0, 68, 255, 212, 0, 0, 172, 255, 161, 0],
            &[0, 26, 251, 255, 171, 166, 254, 255, 161, 0],
            &[0, 0, 112, 249, 255, 223, 82, 252, 161, 0],
            &[0, 0, 0, 6, 25, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ú' / 0xfa
        'ú' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 111, 185, 110, 0, 0],
            &[0, 0, 0, 0, 49, 250, 178, 8, 0, 0],
            &[0, 0, 0, 0, 90, 112, 3, 0, 0, 0],
            &[0, 49, 164, 120, 0, 0, 65, 164, 103, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 187, 0, 0, 116, 255, 161, 0],
            &[0, 68, 255, 212, 0, 0, 172, 255, 161, 0],
            &[0, 26, 251, 255, 171, 166, 254, 255, 161, 0],
            &[0, 0, 112, 249, 255, 223, 82, 252, 161, 0],
            &[0, 0, 0, 6, 25, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'û' / 0xfb
        'û' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 178, 185, 87, 0, 0, 0],
            &[0, 0, 0, 172, 227, 170, 242, 38, 0, 0],
            &[0, 0, 35, 130, 28, 1, 103, 89, 0, 0],
            &[0, 49, 164, 120, 0, 0, 65, 164, 103, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 187, 0, 0, 116, 255, 161, 0],
            &[0, 68, 255, 212, 0, 0, 172, 255, 161, 0],
            &[0, 26, 251, 255, 171, 166, 254, 255, 161, 0],
            &[0, 0, 112, 249, 255, 223, 82, 252, 161, 0],
            &[0, 0, 0, 6, 25, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ü' / 0xfc
        'ü' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 130, 33, 0, 118, 68, 0, 0],
            &[0, 0, 90, 255, 120, 23, 255, 191, 0, 0],
            &[0, 0, 11, 94, 19, 0, 81, 43, 0, 0],
            &[0, 49, 164, 120, 0, 0, 65, 164, 103, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 187, 0, 0, 116, 255, 161, 0],
            &[0, 68, 255, 212, 0, 0, 172, 255, 161, 0],
            &[0, 26, 251, 255, 171, 166, 254, 255, 161, 0],
            &[0, 0, 112, 249, 255, 223, 82, 252, 161, 0],
            &[0, 0, 0, 6, 25, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ý' / 0xfd
        'ý' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 185, 110, 0, 0, 0],
            &[0, 0, 0, 49, 250, 178, 8, 0, 0, 0],
            &[0, 0, 0, 90, 112, 3, 0, 0, 0, 0],
            &[97, 164, 98, 0, 0, 26, 164, 160, 5, 0],
            &[68, 255, 223, 0, 0, 103, 255, 180, 0, 0],
            &[1, 221, 255, 53, 0, 178, 255, 87, 0, 0],
            &[0, 121, 255, 139, 8, 245, 241, 8, 0, 0],
            &[0, 24, 251, 223, 74, 255, 157, 0, 0, 0],
            &[0, 0, 173, 255, 184, 255, 64, 0, 0, 0],
            &[0, 0, 72, 255, 254, 225, 1, 0, 0, 0],
            &[0, 0, 2, 224, 255, 133, 0, 0, 0, 0],
            &[0, 0, 7, 230, 254, 39, 0, 0, 0, 0],
            &[17, 93, 177, 255, 178, 0, 0, 0, 0, 0],
            &[14, 102, 87, 12, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'þ' / 0xfe
        'þ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 163, 124, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 193, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 193, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 190, 81, 181, 183, 78, 0, 0],
            &[0, 58, 255, 237, 253, 244, 255, 253, 51, 0],
            &[0, 58, 255, 252, 54, 1, 169, 255, 153, 0],
            &[0, 58, 255, 207, 0, 0, 82, 255, 196, 0],
            &[0, 58, 255, 203, 0, 0, 74, 255, 207, 0],
            &[0, 58, 255, 242, 13, 0, 119, 255, 177, 0],
            &[0, 58, 255, 255, 209, 154, 245, 255, 93, 0],
            &[0, 58, 255, 199, 186, 255, 253, 151, 2, 0],
            &[0, 58, 255, 193, 0, 20, 15, 0, 0, 0],
            &[0, 58, 255, 193, 0, 0, 0, 0, 0, 0],
            &[0, 21, 92, 69, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ÿ' / 0xff
        'ÿ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 130, 33, 0, 118, 68, 0, 0, 0],
            &[0, 90, 255, 120, 23, 255, 191, 0, 0, 0],
            &[0, 11, 94, 19, 0, 81, 43, 0, 0, 0],
            &[97, 164, 98, 0, 0, 26, 164, 160, 5, 0],
            &[68, 255, 223, 0, 0, 103, 255, 180, 0, 0],
            &[1, 221, 255, 53, 0, 178, 255, 87, 0, 0],
            &[0, 121, 255, 139, 8, 245, 241, 8, 0, 0],
            &[0, 24, 251, 223, 74, 255, 157, 0, 0, 0],
            &[0, 0, 173, 255, 184, 255, 64, 0, 0, 0],
            &[0, 0, 72, 255, 254, 225, 1, 0, 0, 0],
            &[0, 0, 2, 224, 255, 133, 0, 0, 0, 0],
            &[0, 0, 7, 230, 254, 39, 0, 0, 0, 0],
            &[17, 93, 177, 255, 178, 0, 0, 0, 0, 0],
            &[14, 102, 87, 12, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ā' / 0x100
        'Ā' => Some(&[
            &[0, 0, 105, 114, 114, 114, 35, 0, 0, 0],
            &[0, 0, 223, 240, 240, 240, 75, 0, 0, 0],
            &[0, 0, 1, 9, 9, 4, 0, 0, 0, 0],
            &[0, 0, 63, 255, 255, 162, 0, 0, 0, 0],
            &[0, 0, 134, 255, 241, 232, 1, 0, 0, 0],
            &[0, 0, 204, 245, 150, 255, 48, 0, 0, 0],
            &[0, 20, 254, 188, 85, 255, 118, 0, 0, 0],
            &[0, 89, 255, 124, 24, 255, 189, 0, 0, 0],
            &[0, 159, 255, 100, 40, 245, 249, 11, 0, 0],
            &[0, 229, 255, 255, 255, 255, 255, 75, 0, 0],
            &[44, 255, 255, 133, 133, 205, 255, 145, 0, 0],
            &[114, 255, 124, 0, 0, 33, 255, 216, 0, 0],
            &[184, 255, 61, 0, 0, 0, 227, 255, 31, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ā' / 0x101
        'ā' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 24, 24, 24, 10, 0, 0],
            &[0, 0, 0, 208, 255, 255, 255, 109, 0, 0],
            &[0, 0, 0, 60, 74, 74, 74, 31, 0, 0],
            &[0, 0, 55, 146, 183, 188, 149, 42, 0, 0],
            &[0, 0, 141, 254, 229, 230, 255, 241, 33, 0],
            &[0, 0, 18, 26, 0, 0, 152, 255, 105, 0],
            &[0, 0, 27, 131, 174, 187, 220, 255, 125, 0],
            &[0, 12, 230, 254, 170, 126, 179, 255, 125, 0],
            &[0, 66, 255, 191, 0, 0, 146, 255, 125, 0],
            &[0, 44, 255, 244, 140, 157, 253, 255, 125, 0],
            &[0, 0, 147, 254, 255, 209, 97, 255, 125, 0],
            &[0, 0, 0, 12, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ă' / 0x102
        'Ă' => Some(&[
            &[0, 11, 252, 108, 63, 218, 107, 0, 0, 0],
            &[0, 0, 115, 251, 255, 193, 10, 0, 0, 0],
            &[0, 0, 1, 24, 36, 5, 0, 0, 0, 0],
            &[0, 0, 63, 255, 255, 162, 0, 0, 0, 0],
            &[0, 0, 134, 255, 241, 232, 1, 0, 0, 0],
            &[0, 0, 204, 245, 150, 255, 48, 0, 0, 0],
            &[0, 20, 254, 188, 85, 255, 118, 0, 0, 0],
            &[0, 89, 255, 124, 24, 255, 189, 0, 0, 0],
            &[0, 159, 255, 100, 40, 245, 249, 11, 0, 0],
            &[0, 229, 255, 255, 255, 255, 255, 75, 0, 0],
            &[44, 255, 255, 133, 133, 205, 255, 145, 0, 0],
            &[114, 255, 124, 0, 0, 33, 255, 216, 0, 0],
            &[184, 255, 61, 0, 0, 0, 227, 255, 31, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 30, 0, 0, 18, 16, 0, 0, 0],
        ]),
        // letter: 'ă' / 0x103
        'ă' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 188, 55, 0, 125, 115, 0, 0],
            &[0, 0, 0, 163, 246, 225, 252, 63, 0, 0],
            &[0, 0, 0, 4, 91, 119, 48, 0, 0, 0],
            &[0, 0, 55, 146, 183, 188, 149, 42, 0, 0],
            &[0, 0, 141, 254, 229, 230, 255, 241, 33, 0],
            &[0, 0, 18, 26, 0, 0, 152, 255, 105, 0],
            &[0, 0, 27, 131, 174, 187, 220, 255, 125, 0],
            &[0, 12, 230, 254, 170, 126, 179, 255, 125, 0],
            &[0, 66, 255, 191, 0, 0, 146, 255, 125, 0],
            &[0, 44, 255, 244, 140, 157, 253, 255, 125, 0],
            &[0, 0, 147, 254, 255, 209, 97, 255, 125, 0],
            &[0, 0, 0, 12, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ą' / 0x104
        'Ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 9, 9, 4, 0, 0, 0, 0],
            &[0, 0, 63, 255, 255, 162, 0, 0, 0, 0],
            &[0, 0, 134, 255, 241, 232, 1, 0, 0, 0],
            &[0, 0, 204, 245, 150, 255, 48, 0, 0, 0],
            &[0, 20, 254, 188, 85, 255, 118, 0, 0, 0],
            &[0, 89, 255, 124, 24, 255, 189, 0, 0, 0],
            &[0, 159, 255, 100, 40, 245, 249, 11, 0, 0],
            &[0, 229, 255, 255, 255, 255, 255, 75, 0, 0],
            &[44, 255, 255, 133, 133, 205, 255, 145, 0, 0],
            &[114, 255, 124, 0, 0, 33, 255, 216, 0, 0],
            &[184, 255, 61, 0, 0, 0, 227, 255, 33, 0],
            &[0, 0, 0, 0, 0, 0, 118, 205, 9, 0],
            &[0, 0, 0, 0, 0, 4, 247, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 16, 72, 32, 0],
        ]),
        // letter: 'ą' / 0x105
        'ą' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 55, 146, 183, 188, 149, 42, 0, 0],
            &[0, 0, 141, 254, 229, 230, 255, 241, 33, 0],
            &[0, 0, 18, 26, 0, 0, 152, 255, 105, 0],
            &[0, 0, 27, 131, 174, 187, 220, 255, 125, 0],
            &[0, 12, 230, 254, 170, 126, 179, 255, 125, 0],
            &[0, 66, 255, 191, 0, 0, 146, 255, 125, 0],
            &[0, 44, 255, 244, 140, 157, 253, 255, 125, 0],
            &[0, 0, 147, 254, 255, 209, 97, 255, 126, 0],
            &[0, 0, 0, 12, 23, 0, 127, 198, 6, 0],
            &[0, 0, 0, 0, 0, 9, 251, 106, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 73, 30, 0],
        ]),
        // letter: 'Ć' / 0x106
        'Ć' => Some(&[
            &[0, 0, 0, 0, 0, 18, 233, 240, 53, 0],
            &[0, 0, 0, 0, 0, 169, 231, 49, 0, 0],
            &[0, 0, 0, 0, 0, 54, 46, 5, 0, 0],
            &[0, 0, 0, 61, 199, 254, 255, 250, 138, 0],
            &[0, 0, 60, 249, 255, 200, 171, 218, 105, 0],
            &[0, 0, 202, 255, 126, 0, 0, 0, 3, 0],
            &[0, 27, 255, 242, 8, 0, 0, 0, 0, 0],
            &[0, 64, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 69, 255, 200, 0, 0, 0, 0, 0, 0],
            &[0, 45, 255, 233, 0, 0, 0, 0, 0, 0],
            &[0, 2, 234, 255, 92, 0, 0, 0, 13, 0],
            &[0, 0, 110, 255, 253, 192, 173, 220, 114, 0],
            &[0, 0, 0, 113, 233, 255, 255, 232, 79, 0],
            &[0, 0, 0, 0, 1, 23, 21, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 19, 11, 0],
        ]),
        // letter: 'ć' / 0x107
        'ć' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 185, 164, 10, 0],
            &[0, 0, 0, 0, 5, 204, 230, 45, 0, 0],
            &[0, 0, 0, 0, 44, 131, 30, 0, 0, 0],
            &[0, 0, 0, 47, 153, 191, 185, 138, 20, 0],
            &[0, 0, 60, 248, 255, 252, 251, 242, 9, 0],
            &[0, 0, 188, 255, 160, 13, 4, 40, 0, 0],
            &[0, 0, 243, 255, 39, 0, 0, 0, 0, 0],
            &[0, 1, 253, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 224, 255, 91, 0, 0, 10, 2, 0],
            &[0, 0, 133, 255, 247, 174, 168, 238, 10, 0],
            &[0, 0, 7, 149, 245, 255, 255, 206, 6, 0],
            &[0, 0, 0, 0, 5, 28, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĉ' / 0x108
        'Ĉ' => Some(&[
            &[0, 0, 0, 0, 114, 255, 255, 96, 0, 0],
            &[0, 0, 0, 58, 247, 109, 129, 243, 44, 0],
            &[0, 0, 0, 25, 31, 12, 28, 39, 21, 0],
            &[0, 0, 0, 61, 199, 254, 255, 250, 138, 0],
            &[0, 0, 60, 249, 255, 200, 171, 218, 105, 0],
            &[0, 0, 202, 255, 126, 0, 0, 0, 3, 0],
            &[0, 27, 255, 242, 8, 0, 0, 0, 0, 0],
            &[0, 64, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 69, 255, 200, 0, 0, 0, 0, 0, 0],
            &[0, 45, 255, 233, 0, 0, 0, 0, 0, 0],
            &[0, 2, 234, 255, 92, 0, 0, 0, 13, 0],
            &[0, 0, 110, 255, 253, 192, 173, 220, 114, 0],
            &[0, 0, 0, 113, 233, 255, 255, 232, 79, 0],
            &[0, 0, 0, 0, 1, 23, 21, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 19, 19, 0, 0, 0],
        ]),
        // letter: 'ĉ' / 0x109
        'ĉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 185, 151, 0, 0, 0],
            &[0, 0, 0, 84, 253, 150, 249, 114, 0, 0],
            &[0, 0, 0, 120, 73, 0, 59, 128, 7, 0],
            &[0, 0, 0, 47, 153, 191, 185, 138, 20, 0],
            &[0, 0, 60, 248, 255, 252, 251, 242, 9, 0],
            &[0, 0, 188, 255, 160, 13, 4, 40, 0, 0],
            &[0, 0, 243, 255, 39, 0, 0, 0, 0, 0],
            &[0, 1, 253, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 224, 255, 91, 0, 0, 10, 2, 0],
            &[0, 0, 133, 255, 247, 174, 168, 238, 10, 0],
            &[0, 0, 7, 149, 245, 255, 255, 206, 6, 0],
            &[0, 0, 0, 0, 5, 28, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ċ' / 0x10a
        'Ċ' => Some(&[
            &[0, 0, 0, 0, 44, 253, 249, 24, 0, 0],
            &[0, 0, 0, 0, 39, 249, 240, 19, 0, 0],
            &[0, 0, 0, 0, 0, 29, 40, 5, 0, 0],
            &[0, 0, 0, 61, 199, 254, 255, 250, 138, 0],
            &[0, 0, 60, 249, 255, 200, 171, 218, 105, 0],
            &[0, 0, 202, 255, 126, 0, 0, 0, 3, 0],
            &[0, 27, 255, 242, 8, 0, 0, 0, 0, 0],
            &[0, 64, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 69, 255, 200, 0, 0, 0, 0, 0, 0],
            &[0, 45, 255, 233, 0, 0, 0, 0, 0, 0],
            &[0, 2, 234, 255, 92, 0, 0, 0, 13, 0],
            &[0, 0, 110, 255, 253, 192, 173, 220, 114, 0],
            &[0, 0, 0, 113, 233, 255, 255, 232, 79, 0],
            &[0, 0, 0, 0, 1, 23, 21, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 26, 0, 0, 0],
        ]),
        // letter: 'ċ' / 0x10b
        'ċ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 212, 108, 0, 0, 0],
            &[0, 0, 0, 0, 165, 255, 186, 0, 0, 0],
            &[0, 0, 0, 0, 33, 115, 40, 0, 0, 0],
            &[0, 0, 0, 47, 153, 191, 185, 138, 20, 0],
            &[0, 0, 60, 248, 255, 252, 251, 242, 9, 0],
            &[0, 0, 188, 255, 160, 13, 4, 40, 0, 0],
            &[0, 0, 243, 255, 39, 0, 0, 0, 0, 0],
            &[0, 1, 253, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 224, 255, 91, 0, 0, 10, 2, 0],
            &[0, 0, 133, 255, 247, 174, 168, 238, 10, 0],
            &[0, 0, 7, 149, 245, 255, 255, 206, 6, 0],
            &[0, 0, 0, 0, 5, 28, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Č' / 0x10c
        'Č' => Some(&[
            &[0, 0, 0, 71, 245, 89, 112, 245, 55, 0],
            &[0, 0, 0, 0, 130, 254, 255, 109, 0, 0],
            &[0, 0, 0, 0, 5, 54, 69, 7, 0, 0],
            &[0, 0, 0, 61, 199, 254, 255, 250, 138, 0],
            &[0, 0, 60, 249, 255, 200, 171, 218, 105, 0],
            &[0, 0, 202, 255, 126, 0, 0, 0, 3, 0],
            &[0, 27, 255, 242, 8, 0, 0, 0, 0, 0],
            &[0, 64, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 69, 255, 200, 0, 0, 0, 0, 0, 0],
            &[0, 45, 255, 233, 0, 0, 0, 0, 0, 0],
            &[0, 2, 234, 255, 92, 0, 0, 0, 13, 0],
            &[0, 0, 110, 255, 253, 192, 173, 220, 114, 0],
            &[0, 0, 0, 113, 233, 255, 255, 232, 79, 0],
            &[0, 0, 0, 0, 1, 23, 21, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 13, 0, 0, 15, 10, 0],
        ]),
        // letter: 'č' / 0x10d
        'č' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 154, 124, 1, 109, 167, 7, 0],
            &[0, 0, 0, 49, 248, 205, 254, 72, 0, 0],
            &[0, 0, 0, 0, 84, 132, 98, 0, 0, 0],
            &[0, 0, 0, 47, 153, 191, 185, 138, 20, 0],
            &[0, 0, 60, 248, 255, 252, 251, 242, 9, 0],
            &[0, 0, 188, 255, 160, 13, 4, 40, 0, 0],
            &[0, 0, 243, 255, 39, 0, 0, 0, 0, 0],
            &[0, 1, 253, 255, 17, 0, 0, 0, 0, 0],
            &[0, 0, 224, 255, 91, 0, 0, 10, 2, 0],
            &[0, 0, 133, 255, 247, 174, 168, 238, 10, 0],
            &[0, 0, 7, 149, 245, 255, 255, 206, 6, 0],
            &[0, 0, 0, 0, 5, 28, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ď' / 0x10e
        'Ď' => Some(&[
            &[0, 0, 67, 245, 93, 108, 246, 59, 0, 0],
            &[0, 0, 0, 125, 254, 255, 114, 0, 0, 0],
            &[0, 0, 0, 4, 41, 41, 2, 0, 0, 0],
            &[0, 94, 254, 254, 253, 232, 162, 28, 0, 0],
            &[0, 94, 255, 230, 187, 227, 255, 229, 20, 0],
            &[0, 94, 255, 161, 0, 3, 179, 255, 139, 0],
            &[0, 94, 255, 161, 0, 0, 62, 255, 213, 0],
            &[0, 94, 255, 161, 0, 0, 27, 255, 241, 0],
            &[0, 94, 255, 161, 0, 0, 34, 255, 237, 0],
            &[0, 94, 255, 161, 0, 0, 74, 255, 205, 0],
            &[0, 94, 255, 161, 0, 12, 195, 255, 124, 0],
            &[0, 94, 255, 229, 191, 237, 255, 215, 11, 0],
            &[0, 94, 255, 255, 253, 227, 148, 18, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 13, 0, 0, 14, 10, 0, 0],
        ]),
        // letter: 'ď' / 0x10f
        'ď' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 163, 152, 64, 163, 70],
            &[0, 0, 0, 0, 14, 255, 237, 139, 242, 21],
            &[0, 0, 0, 0, 14, 255, 237, 111, 89, 0],
            &[0, 49, 172, 184, 69, 253, 237, 0, 0, 0],
            &[15, 236, 255, 248, 245, 249, 237, 0, 0, 0],
            &[95, 255, 205, 9, 79, 255, 237, 0, 0, 0],
            &[136, 255, 136, 0, 0, 248, 237, 0, 0, 0],
            &[139, 255, 130, 0, 0, 236, 237, 0, 0, 0],
            &[113, 255, 171, 0, 24, 254, 237, 0, 0, 0],
            &[38, 255, 254, 161, 210, 255, 237, 0, 0, 0],
            &[0, 122, 253, 251, 104, 209, 237, 0, 0, 0],
            &[0, 0, 18, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Đ' / 0x110
        'Đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 254, 254, 253, 232, 162, 28, 0, 0],
            &[0, 94, 255, 230, 187, 227, 255, 229, 20, 0],
            &[0, 94, 255, 161, 0, 3, 179, 255, 139, 0],
            &[0, 94, 255, 161, 0, 0, 62, 255, 213, 0],
            &[0, 255, 255, 255, 200, 23, 27, 255, 241, 0],
            &[0, 255, 255, 255, 229, 26, 34, 255, 237, 0],
            &[0, 94, 255, 161, 0, 0, 74, 255, 205, 0],
            &[0, 94, 255, 161, 0, 12, 195, 255, 124, 0],
            &[0, 94, 255, 229, 191, 237, 255, 215, 11, 0],
            &[0, 94, 255, 255, 253, 227, 148, 18, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'đ' / 0x111
        'đ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 58, 163, 103, 0, 0],
            &[0, 0, 0, 46, 133, 224, 255, 255, 53, 0],
            &[0, 0, 0, 70, 203, 255, 255, 255, 81, 0],
            &[0, 8, 111, 155, 115, 91, 255, 161, 0, 0],
            &[0, 178, 255, 255, 255, 224, 255, 161, 0, 0],
            &[46, 255, 250, 72, 22, 184, 255, 161, 0, 0],
            &[97, 255, 184, 0, 0, 75, 255, 161, 0, 0],
            &[104, 255, 165, 0, 0, 57, 255, 161, 0, 0],
            &[75, 255, 210, 0, 0, 101, 255, 161, 0, 0],
            &[9, 238, 255, 189, 154, 243, 255, 161, 0, 0],
            &[0, 83, 240, 255, 223, 82, 253, 161, 0, 0],
            &[0, 0, 7, 26, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ē' / 0x112
        'Ē' => Some(&[
            &[0, 0, 96, 114, 114, 114, 45, 0, 0, 0],
            &[0, 0, 203, 240, 240, 240, 95, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 224, 254, 254, 254, 254, 254, 107, 0, 0],
            &[0, 225, 255, 197, 192, 192, 192, 81, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 35, 15, 15, 15, 3, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 53, 0, 0],
            &[0, 225, 255, 182, 176, 176, 176, 36, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 197, 191, 191, 191, 80, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ē' / 0x113
        'ē' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 24, 24, 24, 10, 0, 0],
            &[0, 0, 0, 201, 255, 255, 255, 116, 0, 0],
            &[0, 0, 0, 58, 74, 74, 74, 33, 0, 0],
            &[0, 0, 0, 85, 174, 192, 151, 35, 0, 0],
            &[0, 0, 123, 255, 243, 213, 253, 237, 28, 0],
            &[0, 13, 247, 241, 29, 0, 109, 255, 130, 0],
            &[0, 61, 255, 232, 158, 158, 174, 255, 171, 0],
            &[0, 66, 255, 238, 196, 196, 196, 196, 137, 0],
            &[0, 26, 253, 235, 17, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 230, 156, 148, 193, 117, 0],
            &[0, 0, 9, 144, 240, 255, 255, 233, 82, 0],
            &[0, 0, 0, 0, 3, 25, 22, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĕ' / 0x114
        'Ĕ' => Some(&[
            &[0, 1, 240, 125, 60, 204, 128, 0, 0, 0],
            &[0, 0, 97, 248, 255, 206, 19, 0, 0, 0],
            &[0, 0, 0, 11, 28, 1, 0, 0, 0, 0],
            &[0, 224, 254, 254, 254, 254, 254, 107, 0, 0],
            &[0, 225, 255, 197, 192, 192, 192, 81, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 35, 15, 15, 15, 3, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 53, 0, 0],
            &[0, 225, 255, 182, 176, 176, 176, 36, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 197, 191, 191, 191, 80, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 30, 3, 0, 16, 18, 0, 0, 0],
        ]),
        // letter: 'ĕ' / 0x115
        'ĕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 184, 60, 0, 120, 120, 0, 0],
            &[0, 0, 0, 156, 247, 224, 253, 69, 0, 0],
            &[0, 0, 0, 3, 89, 119, 51, 0, 0, 0],
            &[0, 0, 0, 85, 174, 192, 151, 35, 0, 0],
            &[0, 0, 123, 255, 243, 213, 253, 237, 28, 0],
            &[0, 13, 247, 241, 29, 0, 109, 255, 130, 0],
            &[0, 61, 255, 232, 158, 158, 174, 255, 171, 0],
            &[0, 66, 255, 238, 196, 196, 196, 196, 137, 0],
            &[0, 26, 253, 235, 17, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 230, 156, 148, 193, 117, 0],
            &[0, 0, 9, 144, 240, 255, 255, 233, 82, 0],
            &[0, 0, 0, 0, 3, 25, 22, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ė' / 0x116
        'Ė' => Some(&[
            &[0, 0, 0, 214, 255, 101, 0, 0, 0, 0],
            &[0, 0, 0, 205, 255, 88, 0, 0, 0, 0],
            &[0, 0, 0, 6, 22, 0, 0, 0, 0, 0],
            &[0, 224, 254, 254, 254, 254, 254, 107, 0, 0],
            &[0, 225, 255, 197, 192, 192, 192, 81, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 35, 15, 15, 15, 3, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 53, 0, 0],
            &[0, 225, 255, 182, 176, 176, 176, 36, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 197, 191, 191, 191, 80, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 43, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ė' / 0x117
        'ė' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 210, 67, 0, 0, 0],
            &[0, 0, 0, 0, 219, 255, 132, 0, 0, 0],
            &[0, 0, 0, 0, 56, 113, 19, 0, 0, 0],
            &[0, 0, 0, 85, 174, 192, 151, 35, 0, 0],
            &[0, 0, 123, 255, 243, 213, 253, 237, 28, 0],
            &[0, 13, 247, 241, 29, 0, 109, 255, 130, 0],
            &[0, 61, 255, 232, 158, 158, 174, 255, 171, 0],
            &[0, 66, 255, 238, 196, 196, 196, 196, 137, 0],
            &[0, 26, 253, 235, 17, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 230, 156, 148, 193, 117, 0],
            &[0, 0, 9, 144, 240, 255, 255, 233, 82, 0],
            &[0, 0, 0, 0, 3, 25, 22, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ę' / 0x118
        'Ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 224, 254, 254, 254, 254, 254, 107, 0, 0],
            &[0, 225, 255, 197, 192, 192, 192, 81, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 35, 15, 15, 15, 3, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 53, 0, 0],
            &[0, 225, 255, 182, 176, 176, 176, 36, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 197, 191, 191, 191, 80, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 31, 230, 70, 0, 0, 0],
            &[0, 0, 0, 0, 141, 225, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 66, 0, 0, 0],
        ]),
        // letter: 'ę' / 0x119
        'ę' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 174, 192, 151, 35, 0, 0],
            &[0, 0, 123, 255, 243, 213, 253, 237, 28, 0],
            &[0, 13, 247, 241, 29, 0, 109, 255, 130, 0],
            &[0, 61, 255, 232, 158, 158, 174, 255, 171, 0],
            &[0, 66, 255, 238, 196, 196, 196, 196, 137, 0],
            &[0, 26, 253, 235, 17, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 230, 156, 148, 193, 117, 0],
            &[0, 0, 9, 144, 240, 255, 255, 255, 115, 0],
            &[0, 0, 0, 0, 3, 34, 231, 122, 0, 0],
            &[0, 0, 0, 0, 0, 68, 255, 39, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 73, 13, 0],
        ]),
        // letter: 'Ě' / 0x11a
        'Ě' => Some(&[
            &[0, 19, 221, 162, 55, 230, 132, 0, 0, 0],
            &[0, 0, 51, 251, 253, 191, 2, 0, 0, 0],
            &[0, 0, 0, 33, 41, 15, 0, 0, 0, 0],
            &[0, 224, 254, 254, 254, 254, 254, 107, 0, 0],
            &[0, 225, 255, 197, 192, 192, 192, 81, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 35, 15, 15, 15, 3, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 53, 0, 0],
            &[0, 225, 255, 182, 176, 176, 176, 36, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 20, 0, 0, 0, 0, 0, 0],
            &[0, 225, 255, 197, 191, 191, 191, 80, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 19, 0, 0, 8, 16, 0, 0, 0],
        ]),
        // letter: 'ě' / 0x11b
        'ě' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 176, 86, 6, 142, 135, 0, 0],
            &[0, 0, 0, 96, 254, 207, 239, 33, 0, 0],
            &[0, 0, 0, 0, 112, 132, 70, 0, 0, 0],
            &[0, 0, 0, 85, 174, 192, 151, 35, 0, 0],
            &[0, 0, 123, 255, 243, 213, 253, 237, 28, 0],
            &[0, 13, 247, 241, 29, 0, 109, 255, 130, 0],
            &[0, 61, 255, 232, 158, 158, 174, 255, 171, 0],
            &[0, 66, 255, 238, 196, 196, 196, 196, 137, 0],
            &[0, 26, 253, 235, 17, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 230, 156, 148, 193, 117, 0],
            &[0, 0, 9, 144, 240, 255, 255, 233, 82, 0],
            &[0, 0, 0, 0, 3, 25, 22, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => Some(&[
            &[0, 0, 0, 5, 205, 255, 233, 22, 0, 0],
            &[0, 0, 0, 148, 226, 61, 205, 188, 3, 0],
            &[0, 0, 0, 41, 16, 21, 26, 42, 5, 0],
            &[0, 0, 1, 118, 232, 255, 255, 235, 99, 0],
            &[0, 0, 138, 255, 250, 181, 168, 223, 74, 0],
            &[0, 30, 252, 250, 54, 0, 0, 2, 1, 0],
            &[0, 103, 255, 172, 0, 0, 0, 0, 0, 0],
            &[0, 134, 255, 125, 0, 72, 160, 160, 123, 0],
            &[0, 139, 255, 122, 0, 115, 255, 255, 197, 0],
            &[0, 113, 255, 161, 0, 5, 28, 255, 197, 0],
            &[0, 45, 255, 244, 33, 0, 16, 255, 197, 0],
            &[0, 0, 168, 255, 238, 162, 170, 255, 197, 0],
            &[0, 0, 7, 143, 243, 255, 255, 240, 135, 0],
            &[0, 0, 0, 0, 2, 26, 21, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 19, 12, 0, 0, 0],
        ]),
        // letter: 'ĝ' / 0x11d
        'ĝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 178, 185, 87, 0, 0, 0],
            &[0, 0, 0, 172, 227, 170, 242, 38, 0, 0],
            &[0, 0, 35, 130, 28, 1, 103, 89, 0, 0],
            &[0, 0, 24, 157, 197, 143, 23, 162, 103, 0],
            &[0, 1, 204, 255, 253, 246, 232, 255, 161, 0],
            &[0, 57, 255, 241, 32, 6, 178, 255, 161, 0],
            &[0, 101, 255, 176, 0, 0, 91, 255, 161, 0],
            &[0, 107, 255, 167, 0, 0, 74, 255, 161, 0],
            &[0, 78, 255, 218, 2, 0, 118, 255, 161, 0],
            &[0, 12, 239, 255, 195, 156, 247, 255, 161, 0],
            &[0, 0, 75, 237, 255, 230, 158, 255, 161, 0],
            &[0, 0, 0, 5, 29, 4, 124, 255, 147, 0],
            &[0, 0, 180, 126, 94, 116, 236, 255, 85, 0],
            &[0, 0, 24, 83, 104, 91, 36, 0, 0, 0],
        ]),
        // letter: 'Ğ' / 0x11e
        'Ğ' => Some(&[
            &[0, 0, 0, 166, 183, 54, 152, 204, 0, 0],
            &[0, 0, 0, 43, 227, 255, 236, 64, 0, 0],
            &[0, 0, 0, 0, 4, 52, 24, 0, 0, 0],
            &[0, 0, 1, 118, 232, 255, 255, 235, 99, 0],
            &[0, 0, 138, 255, 250, 181, 168, 223, 74, 0],
            &[0, 30, 252, 250, 54, 0, 0, 2, 1, 0],
            &[0, 103, 255, 172, 0, 0, 0, 0, 0, 0],
            &[0, 134, 255, 125, 0, 72, 160, 160, 123, 0],
            &[0, 139, 255, 122, 0, 115, 255, 255, 197, 0],
            &[0, 113, 255, 161, 0, 5, 28, 255, 197, 0],
            &[0, 45, 255, 244, 33, 0, 16, 255, 197, 0],
            &[0, 0, 168, 255, 238, 162, 170, 255, 197, 0],
            &[0, 0, 7, 143, 243, 255, 255, 240, 135, 0],
            &[0, 0, 0, 0, 2, 26, 21, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 12, 0, 7, 27, 0, 0],
        ]),
        // letter: 'ğ' / 0x11f
        'ğ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 196, 33, 0, 147, 93, 0, 0],
            &[0, 0, 0, 192, 242, 229, 246, 41, 0, 0],
            &[0, 0, 0, 9, 100, 116, 37, 0, 0, 0],
            &[0, 0, 24, 157, 197, 143, 23, 162, 103, 0],
            &[0, 1, 204, 255, 253, 246, 232, 255, 161, 0],
            &[0, 57, 255, 241, 32, 6, 178, 255, 161, 0],
            &[0, 101, 255, 176, 0, 0, 91, 255, 161, 0],
            &[0, 107, 255, 167, 0, 0, 74, 255, 161, 0],
            &[0, 78, 255, 218, 2, 0, 118, 255, 161, 0],
            &[0, 12, 239, 255, 195, 156, 247, 255, 161, 0],
            &[0, 0, 75, 237, 255, 230, 158, 255, 161, 0],
            &[0, 0, 0, 5, 29, 4, 124, 255, 147, 0],
            &[0, 0, 180, 126, 94, 116, 236, 255, 85, 0],
            &[0, 0, 24, 83, 104, 91, 36, 0, 0, 0],
        ]),
        // letter: 'Ġ' / 0x120
        'Ġ' => Some(&[
            &[0, 0, 0, 0, 139, 255, 176, 0, 0, 0],
            &[0, 0, 0, 0, 129, 255, 163, 0, 0, 0],
            &[0, 0, 0, 0, 1, 47, 21, 0, 0, 0],
            &[0, 0, 1, 118, 232, 255, 255, 235, 99, 0],
            &[0, 0, 138, 255, 250, 181, 168, 223, 74, 0],
            &[0, 30, 252, 250, 54, 0, 0, 2, 1, 0],
            &[0, 103, 255, 172, 0, 0, 0, 0, 0, 0],
            &[0, 134, 255, 125, 0, 72, 160, 160, 123, 0],
            &[0, 139, 255, 122, 0, 115, 255, 255, 197, 0],
            &[0, 113, 255, 161, 0, 5, 28, 255, 197, 0],
            &[0, 45, 255, 244, 33, 0, 16, 255, 197, 0],
            &[0, 0, 168, 255, 238, 162, 170, 255, 197, 0],
            &[0, 0, 7, 143, 243, 255, 255, 240, 135, 0],
            &[0, 0, 0, 0, 2, 26, 21, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 46, 9, 0, 0, 0],
        ]),
        // letter: 'ġ' / 0x121
        'ġ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 162, 205, 42, 0, 0, 0],
            &[0, 0, 0, 3, 251, 255, 96, 0, 0, 0],
            &[0, 0, 0, 0, 72, 107, 9, 0, 0, 0],
            &[0, 0, 24, 157, 197, 143, 23, 162, 103, 0],
            &[0, 1, 204, 255, 253, 246, 232, 255, 161, 0],
            &[0, 57, 255, 241, 32, 6, 178, 255, 161, 0],
            &[0, 101, 255, 176, 0, 0, 91, 255, 161, 0],
            &[0, 107, 255, 167, 0, 0, 74, 255, 161, 0],
            &[0, 78, 255, 218, 2, 0, 118, 255, 161, 0],
            &[0, 12, 239, 255, 195, 156, 247, 255, 161, 0],
            &[0, 0, 75, 237, 255, 230, 158, 255, 161, 0],
            &[0, 0, 0, 5, 29, 4, 124, 255, 147, 0],
            &[0, 0, 180, 126, 94, 116, 236, 255, 85, 0],
            &[0, 0, 24, 83, 104, 91, 36, 0, 0, 0],
        ]),
        // letter: 'Ģ' / 0x122
        'Ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 18, 0, 0, 0],
            &[0, 0, 1, 118, 232, 255, 255, 235, 99, 0],
            &[0, 0, 138, 255, 250, 181, 168, 223, 74, 0],
            &[0, 30, 252, 250, 54, 0, 0, 2, 1, 0],
            &[0, 103, 255, 172, 0, 0, 0, 0, 0, 0],
            &[0, 134, 255, 125, 0, 72, 160, 160, 123, 0],
            &[0, 139, 255, 122, 0, 115, 255, 255, 197, 0],
            &[0, 113, 255, 161, 0, 5, 28, 255, 197, 0],
            &[0, 45, 255, 244, 33, 0, 16, 255, 197, 0],
            &[0, 0, 168, 255, 238, 162, 170, 255, 197, 0],
            &[0, 0, 7, 143, 243, 255, 255, 240, 135, 0],
            &[0, 0, 0, 0, 14, 63, 42, 2, 0, 0],
            &[0, 0, 0, 0, 105, 255, 81, 0, 0, 0],
            &[0, 0, 0, 0, 17, 10, 0, 0, 0, 0],
        ]),
        // letter: 'ģ' / 0x123
        'ģ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 178, 54, 0, 0, 0],
            &[0, 0, 0, 0, 132, 255, 25, 0, 0, 0],
            &[0, 0, 0, 0, 116, 126, 0, 0, 0, 0],
            &[0, 0, 24, 157, 197, 143, 23, 162, 103, 0],
            &[0, 1, 204, 255, 253, 246, 232, 255, 161, 0],
            &[0, 57, 255, 241, 32, 6, 178, 255, 161, 0],
            &[0, 101, 255, 176, 0, 0, 91, 255, 161, 0],
            &[0, 107, 255, 167, 0, 0, 74, 255, 161, 0],
            &[0, 78, 255, 218, 2, 0, 118, 255, 161, 0],
            &[0, 12, 239, 255, 195, 156, 247, 255, 161, 0],
            &[0, 0, 75, 237, 255, 230, 158, 255, 161, 0],
            &[0, 0, 0, 5, 29, 4, 124, 255, 147, 0],
            &[0, 0, 180, 126, 94, 116, 236, 255, 85, 0],
            &[0, 0, 24, 83, 104, 91, 36, 0, 0, 0],
        ]),
        // letter: 'Ĥ' / 0x124
        'Ĥ' => Some(&[
            &[0, 0, 0, 54, 252, 255, 159, 0, 0, 0],
            &[0, 0, 21, 223, 165, 81, 245, 95, 0, 0],
            &[0, 0, 15, 40, 0, 0, 24, 31, 0, 0],
            &[0, 94, 254, 168, 0, 0, 66, 254, 196, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 176, 23, 23, 83, 255, 197, 0],
            &[0, 94, 255, 255, 255, 255, 255, 255, 197, 0],
            &[0, 94, 255, 226, 168, 168, 191, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 19, 5, 0, 0, 0],
        ]),
        // letter: 'ĥ' / 0x125
        'ĥ' => Some(&[
            &[34, 239, 171, 227, 177, 1, 0, 0, 0, 0],
            &[88, 106, 1, 27, 131, 37, 0, 0, 0, 0],
            &[0, 37, 163, 131, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 204, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 204, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 201, 76, 181, 181, 99, 0, 0],
            &[0, 58, 255, 238, 252, 242, 255, 255, 85, 0],
            &[0, 58, 255, 255, 77, 0, 150, 255, 160, 0],
            &[0, 58, 255, 231, 0, 0, 88, 255, 179, 0],
            &[0, 58, 255, 207, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 82, 184, 179, 19, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ħ' / 0x126
        'Ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[51, 254, 211, 0, 0, 105, 254, 157, 0, 0],
            &[214, 255, 255, 163, 163, 255, 255, 255, 65, 0],
            &[228, 255, 255, 177, 177, 255, 255, 255, 70, 0],
            &[51, 255, 215, 23, 23, 119, 255, 157, 0, 0],
            &[51, 255, 255, 255, 255, 255, 255, 157, 0, 0],
            &[51, 255, 240, 168, 168, 204, 255, 157, 0, 0],
            &[51, 255, 211, 0, 0, 105, 255, 157, 0, 0],
            &[51, 255, 211, 0, 0, 105, 255, 157, 0, 0],
            &[51, 255, 211, 0, 0, 105, 255, 157, 0, 0],
            &[51, 255, 211, 0, 0, 105, 255, 157, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ħ' / 0x127
        'ħ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 163, 131, 0, 0, 0, 0, 0, 0],
            &[0, 191, 255, 255, 133, 116, 0, 0, 0, 0],
            &[0, 255, 255, 255, 203, 177, 0, 0, 0, 0],
            &[0, 58, 255, 202, 48, 145, 146, 64, 0, 0],
            &[0, 58, 255, 230, 244, 255, 255, 254, 66, 0],
            &[0, 58, 255, 255, 112, 23, 172, 255, 155, 0],
            &[0, 58, 255, 236, 0, 0, 90, 255, 178, 0],
            &[0, 58, 255, 209, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĩ' / 0x128
        'Ĩ' => Some(&[
            &[0, 8, 181, 212, 114, 114, 187, 0, 0, 0],
            &[0, 85, 220, 142, 236, 255, 101, 0, 0, 0],
            &[0, 20, 26, 0, 6, 25, 0, 0, 0, 0],
            &[0, 188, 254, 254, 254, 254, 254, 35, 0, 0],
            &[0, 84, 139, 235, 255, 164, 121, 14, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 83, 138, 235, 255, 163, 120, 14, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĩ' / 0x129
        'ĩ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 122, 28, 38, 121, 0, 0],
            &[0, 0, 57, 250, 230, 251, 242, 154, 0, 0],
            &[0, 0, 56, 86, 2, 76, 115, 13, 0, 0],
            &[0, 0, 128, 164, 164, 164, 68, 0, 0, 0],
            &[0, 0, 168, 240, 255, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 11, 94, 136, 220, 255, 200, 123, 58, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ī' / 0x12a
        'Ī' => Some(&[
            &[0, 0, 105, 114, 114, 114, 35, 0, 0, 0],
            &[0, 0, 223, 240, 240, 240, 75, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 188, 254, 254, 254, 254, 254, 35, 0, 0],
            &[0, 84, 139, 235, 255, 164, 121, 14, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 83, 138, 235, 255, 163, 120, 14, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ī' / 0x12b
        'ī' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 24, 24, 24, 7, 0, 0],
            &[0, 0, 0, 237, 255, 255, 255, 80, 0, 0],
            &[0, 0, 0, 69, 74, 74, 74, 23, 0, 0],
            &[0, 0, 128, 164, 164, 164, 68, 0, 0, 0],
            &[0, 0, 168, 240, 255, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 11, 94, 136, 220, 255, 200, 123, 58, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => Some(&[
            &[0, 11, 252, 108, 63, 218, 107, 0, 0, 0],
            &[0, 0, 115, 251, 255, 193, 10, 0, 0, 0],
            &[0, 0, 0, 14, 26, 0, 0, 0, 0, 0],
            &[0, 188, 254, 254, 254, 254, 254, 35, 0, 0],
            &[0, 84, 139, 235, 255, 164, 121, 14, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 83, 138, 235, 255, 163, 120, 14, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 30, 0, 0, 18, 16, 0, 0, 0],
        ]),
        // letter: 'ĭ' / 0x12d
        'ĭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 196, 33, 0, 147, 93, 0, 0],
            &[0, 0, 0, 192, 242, 229, 246, 41, 0, 0],
            &[0, 0, 0, 9, 100, 116, 37, 0, 0, 0],
            &[0, 0, 128, 164, 164, 164, 68, 0, 0, 0],
            &[0, 0, 168, 240, 255, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 11, 94, 136, 220, 255, 200, 123, 58, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Į' / 0x12e
        'Į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 188, 254, 254, 254, 254, 254, 35, 0, 0],
            &[0, 84, 139, 235, 255, 164, 121, 14, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 83, 138, 235, 255, 163, 120, 14, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 35, 0, 0],
            &[0, 0, 0, 2, 184, 146, 0, 0, 0, 0],
            &[0, 0, 0, 64, 255, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 73, 14, 0, 0, 0],
        ]),
        // letter: 'į' / 0x12f
        'į' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 113, 212, 86, 0, 0, 0],
            &[0, 0, 0, 0, 193, 255, 157, 0, 0, 0],
            &[0, 0, 0, 0, 45, 115, 28, 0, 0, 0],
            &[0, 0, 128, 164, 164, 164, 68, 0, 0, 0],
            &[0, 0, 168, 240, 255, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 11, 94, 136, 220, 255, 200, 123, 58, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 179, 0],
            &[0, 0, 0, 0, 0, 31, 230, 70, 0, 0],
            &[0, 0, 0, 0, 0, 141, 225, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 56, 66, 0, 0],
        ]),
        // letter: 'İ' / 0x130
        'İ' => Some(&[
            &[0, 0, 2, 233, 255, 80, 0, 0, 0, 0],
            &[0, 0, 1, 225, 253, 68, 0, 0, 0, 0],
            &[0, 0, 0, 8, 19, 0, 0, 0, 0, 0],
            &[0, 188, 254, 254, 254, 254, 254, 35, 0, 0],
            &[0, 84, 139, 235, 255, 164, 121, 14, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 0, 0, 203, 255, 38, 0, 0, 0, 0],
            &[0, 83, 138, 235, 255, 163, 120, 14, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 39, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ı' / 0x131
        'ı' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 164, 164, 164, 68, 0, 0, 0],
            &[0, 0, 168, 240, 255, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 106, 0, 0, 0],
            &[0, 11, 94, 136, 220, 255, 200, 123, 58, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĳ' / 0x132
        'Ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 206, 254, 34, 0, 198, 254, 53, 0, 0],
            &[0, 207, 255, 34, 0, 199, 255, 53, 0, 0],
            &[0, 207, 255, 34, 0, 199, 255, 53, 0, 0],
            &[0, 207, 255, 34, 0, 199, 255, 53, 0, 0],
            &[0, 207, 255, 34, 0, 199, 255, 53, 0, 0],
            &[0, 207, 255, 34, 0, 199, 255, 53, 0, 0],
            &[0, 202, 249, 33, 0, 203, 255, 51, 0, 0],
            &[0, 0, 0, 0, 17, 242, 255, 28, 0, 0],
            &[0, 167, 176, 169, 226, 255, 204, 0, 0, 0],
            &[0, 189, 255, 255, 254, 194, 36, 0, 0, 0],
            &[0, 0, 18, 32, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĳ' / 0x133
        'ĳ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 142, 209, 59, 3, 176, 200, 30, 0, 0],
            &[0, 230, 255, 121, 20, 255, 255, 75, 0, 0],
            &[0, 61, 112, 15, 0, 82, 102, 4, 0, 0],
            &[0, 124, 164, 45, 0, 148, 164, 20, 0, 0],
            &[0, 192, 255, 70, 0, 231, 255, 32, 0, 0],
            &[0, 192, 255, 70, 0, 231, 255, 32, 0, 0],
            &[0, 192, 255, 70, 0, 231, 255, 32, 0, 0],
            &[0, 192, 255, 70, 0, 231, 255, 32, 0, 0],
            &[0, 192, 255, 70, 0, 231, 255, 32, 0, 0],
            &[0, 192, 255, 70, 0, 231, 255, 32, 0, 0],
            &[0, 192, 255, 70, 0, 231, 255, 32, 0, 0],
            &[0, 0, 0, 0, 4, 250, 255, 22, 0, 0],
            &[0, 46, 112, 92, 167, 255, 235, 0, 0, 0],
            &[0, 12, 74, 102, 88, 32, 0, 0, 0, 0],
        ]),
        // letter: 'Ĵ' / 0x134
        'Ĵ' => Some(&[
            &[0, 0, 0, 0, 147, 255, 254, 64, 0, 0],
            &[0, 0, 0, 86, 246, 85, 159, 230, 25, 0],
            &[0, 0, 0, 30, 25, 0, 0, 39, 16, 0],
            &[0, 0, 0, 0, 32, 254, 219, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 220, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 220, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 220, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 220, 0, 0, 0],
            &[0, 0, 0, 0, 32, 255, 220, 0, 0, 0],
            &[0, 0, 0, 0, 36, 255, 217, 0, 0, 0],
            &[0, 0, 0, 0, 88, 255, 195, 0, 0, 0],
            &[0, 170, 175, 175, 243, 255, 117, 0, 0, 0],
            &[0, 192, 255, 255, 246, 150, 5, 0, 0, 0],
            &[0, 0, 20, 29, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 19, 17, 0, 0, 0],
        ]),
        // letter: 'ĵ' / 0x135
        'ĵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 185, 181, 20, 0, 0],
            &[0, 0, 0, 33, 238, 172, 225, 180, 1, 0],
            &[0, 0, 0, 86, 106, 2, 26, 129, 38, 0],
            &[0, 0, 31, 164, 164, 164, 164, 0, 0, 0],
            &[0, 0, 40, 223, 250, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 7, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 6, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 6, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 6, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 6, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 6, 255, 255, 1, 0, 0],
            &[0, 0, 0, 0, 30, 255, 247, 0, 0, 0],
            &[0, 62, 107, 94, 184, 255, 205, 0, 0, 0],
            &[0, 18, 79, 103, 84, 23, 0, 0, 0, 0],
        ]),
        // letter: 'Ķ' / 0x136
        'Ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 254, 157, 0, 0, 161, 254, 145, 0],
            &[0, 94, 255, 158, 0, 76, 255, 213, 8, 0],
            &[0, 94, 255, 158, 17, 230, 249, 48, 0, 0],
            &[0, 94, 255, 158, 160, 255, 117, 0, 0, 0],
            &[0, 94, 255, 212, 255, 234, 3, 0, 0, 0],
            &[0, 94, 255, 255, 254, 255, 88, 0, 0, 0],
            &[0, 94, 255, 210, 78, 255, 222, 7, 0, 0],
            &[0, 94, 255, 158, 0, 181, 255, 116, 0, 0],
            &[0, 94, 255, 158, 0, 47, 254, 239, 18, 0],
            &[0, 94, 255, 158, 0, 0, 166, 255, 144, 0],
            &[0, 0, 0, 0, 24, 37, 8, 0, 0, 0],
            &[0, 0, 0, 0, 191, 237, 13, 0, 0, 0],
            &[0, 0, 0, 1, 24, 2, 0, 0, 0, 0],
        ]),
        // letter: 'ķ' / 0x137
        'ķ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[26, 163, 142, 0, 0, 0, 0, 0, 0, 0],
            &[40, 255, 222, 0, 0, 0, 0, 0, 0, 0],
            &[40, 255, 222, 0, 0, 0, 0, 0, 0, 0],
            &[40, 255, 222, 0, 0, 101, 164, 119, 0, 0],
            &[40, 255, 222, 0, 82, 254, 227, 28, 0, 0],
            &[40, 255, 220, 50, 245, 240, 45, 0, 0, 0],
            &[40, 255, 220, 224, 255, 103, 0, 0, 0, 0],
            &[40, 255, 255, 248, 255, 211, 6, 0, 0, 0],
            &[40, 255, 242, 50, 156, 255, 133, 0, 0, 0],
            &[40, 255, 222, 0, 16, 230, 251, 52, 0, 0],
            &[40, 255, 222, 0, 0, 81, 255, 212, 6, 0],
            &[0, 0, 0, 17, 37, 15, 0, 0, 0, 0],
            &[0, 0, 0, 144, 253, 43, 0, 0, 0, 0],
            &[0, 0, 0, 21, 7, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĸ' / 0x138
        'ĸ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[26, 164, 143, 0, 0, 101, 164, 119, 0, 0],
            &[40, 255, 222, 0, 82, 254, 227, 28, 0, 0],
            &[40, 255, 220, 50, 245, 240, 45, 0, 0, 0],
            &[40, 255, 220, 224, 255, 103, 0, 0, 0, 0],
            &[40, 255, 255, 248, 255, 211, 6, 0, 0, 0],
            &[40, 255, 242, 50, 156, 255, 133, 0, 0, 0],
            &[40, 255, 222, 0, 16, 230, 251, 52, 0, 0],
            &[40, 255, 222, 0, 0, 81, 255, 212, 6, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ĺ' / 0x139
        'Ĺ' => Some(&[
            &[0, 0, 175, 255, 115, 0, 0, 0, 0, 0],
            &[0, 93, 252, 104, 0, 0, 0, 0, 0, 0],
            &[0, 29, 30, 0, 0, 0, 0, 0, 0, 0],
            &[0, 188, 254, 73, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 218, 202, 202, 202, 85, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 19, 17, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ĺ' / 0x13a
        'ĺ' => Some(&[
            &[0, 0, 0, 0, 49, 250, 179, 8, 0, 0],
            &[0, 0, 0, 0, 91, 113, 3, 0, 0, 0],
            &[0, 0, 109, 163, 163, 163, 63, 0, 0, 0],
            &[0, 0, 118, 220, 255, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 164, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 11, 94, 137, 223, 255, 197, 122, 57, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 110, 184, 110, 0, 0],
        ]),
        // letter: 'Ļ' / 0x13b
        'Ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 188, 254, 73, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 218, 202, 202, 202, 85, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 12, 37, 19, 0, 0, 0, 0],
            &[0, 0, 0, 112, 255, 73, 0, 0, 0, 0],
            &[0, 0, 0, 18, 10, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ļ' / 0x13c
        'ļ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 109, 163, 163, 163, 63, 0, 0, 0],
            &[0, 0, 118, 220, 255, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 164, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 0, 0, 0],
            &[0, 11, 94, 137, 223, 255, 197, 122, 57, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 179, 0],
            &[0, 0, 0, 0, 20, 37, 11, 0, 0, 0],
            &[0, 0, 0, 0, 165, 248, 27, 0, 0, 0],
            &[0, 0, 0, 0, 23, 5, 0, 0, 0, 0],
        ]),
        // letter: 'Ľ' / 0x13d
        'Ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 188, 254, 73, 0, 151, 253, 46, 0, 0],
            &[0, 189, 255, 74, 0, 203, 170, 0, 0, 0],
            &[0, 189, 255, 74, 0, 58, 18, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 218, 202, 202, 202, 85, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ľ' / 0x13e
        'ľ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 109, 163, 163, 163, 63, 95, 163, 39, 0],
            &[0, 118, 220, 255, 255, 99, 188, 212, 2, 0],
            &[0, 0, 0, 164, 255, 99, 140, 59, 0, 0],
            &[0, 0, 0, 163, 255, 99, 0, 0, 0, 0],
            &[0, 0, 0, 163, 255, 99, 0, 0, 0, 0],
            &[0, 0, 0, 163, 255, 99, 0, 0, 0, 0],
            &[0, 0, 0, 163, 255, 99, 0, 0, 0, 0],
            &[0, 0, 0, 163, 255, 99, 0, 0, 0, 0],
            &[0, 0, 0, 163, 255, 99, 0, 0, 0, 0],
            &[11, 94, 137, 223, 255, 197, 122, 57, 0, 0],
            &[40, 255, 255, 255, 255, 255, 255, 179, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 188, 254, 73, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 88, 162, 44, 0, 0],
            &[0, 189, 255, 74, 0, 221, 255, 150, 0, 0],
            &[0, 189, 255, 74, 0, 108, 189, 57, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 218, 202, 202, 202, 85, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŀ' / 0x140
        'ŀ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 109, 163, 163, 163, 63, 0, 0, 0, 0],
            &[0, 118, 220, 255, 255, 99, 0, 0, 0, 0],
            &[0, 0, 0, 164, 255, 99, 0, 0, 0, 0],
            &[0, 0, 0, 163, 255, 99, 0, 0, 0, 0],
            &[0, 0, 0, 163, 255, 99, 0, 51, 124, 31],
            &[0, 0, 0, 163, 255, 99, 0, 203, 255, 162],
            &[0, 0, 0, 163, 255, 99, 0, 128, 230, 91],
            &[0, 0, 0, 163, 255, 99, 0, 0, 0, 0],
            &[0, 0, 0, 163, 255, 99, 0, 0, 0, 0],
            &[11, 94, 137, 223, 255, 197, 122, 57, 0, 0],
            &[40, 255, 255, 255, 255, 255, 255, 179, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ł' / 0x141
        'Ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 254, 216, 0, 0, 0, 0, 0],
            &[0, 0, 46, 255, 217, 0, 0, 0, 0, 0],
            &[0, 0, 46, 255, 217, 0, 0, 0, 0, 0],
            &[0, 0, 46, 255, 255, 194, 54, 0, 0, 0],
            &[0, 0, 58, 255, 255, 235, 77, 0, 0, 0],
            &[0, 46, 255, 255, 255, 18, 0, 0, 0, 0],
            &[0, 11, 223, 255, 217, 0, 0, 0, 0, 0],
            &[0, 0, 46, 255, 217, 0, 0, 0, 0, 0],
            &[0, 0, 46, 255, 247, 202, 202, 202, 198, 0],
            &[0, 0, 46, 255, 255, 255, 255, 255, 250, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ł' / 0x142
        'ł' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 109, 163, 163, 163, 63, 0, 0, 0],
            &[0, 0, 118, 220, 255, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 164, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 163, 255, 99, 1, 0, 0],
            &[0, 0, 0, 0, 163, 255, 252, 154, 0, 0],
            &[0, 0, 0, 0, 255, 255, 255, 138, 0, 0],
            &[0, 0, 28, 191, 255, 255, 145, 0, 0, 0],
            &[0, 0, 14, 212, 255, 255, 99, 0, 0, 0],
            &[0, 0, 0, 1, 163, 255, 99, 0, 0, 0],
            &[0, 11, 94, 137, 223, 255, 197, 122, 57, 0],
            &[0, 40, 255, 255, 255, 255, 255, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ń' / 0x143
        'Ń' => Some(&[
            &[0, 0, 0, 0, 0, 170, 255, 119, 0, 0],
            &[0, 0, 0, 0, 89, 253, 108, 0, 0, 0],
            &[0, 0, 0, 0, 29, 31, 0, 0, 0, 0],
            &[0, 94, 254, 254, 140, 0, 9, 254, 196, 0],
            &[0, 94, 255, 253, 223, 0, 9, 255, 197, 0],
            &[0, 94, 255, 199, 255, 52, 9, 255, 197, 0],
            &[0, 94, 255, 131, 255, 135, 9, 255, 197, 0],
            &[0, 94, 255, 109, 204, 219, 9, 255, 197, 0],
            &[0, 94, 255, 111, 121, 255, 54, 255, 197, 0],
            &[0, 94, 255, 111, 38, 255, 133, 255, 197, 0],
            &[0, 94, 255, 111, 0, 211, 214, 247, 197, 0],
            &[0, 94, 255, 111, 0, 128, 255, 255, 197, 0],
            &[0, 94, 255, 111, 0, 45, 255, 255, 197, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 19, 17, 0, 0],
        ]),
        // letter: 'ń' / 0x144
        'ń' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 101, 185, 120, 0, 0],
            &[0, 0, 0, 0, 39, 246, 188, 12, 0, 0],
            &[0, 0, 0, 0, 83, 117, 6, 0, 0, 0],
            &[0, 37, 164, 69, 86, 181, 182, 101, 0, 0],
            &[0, 58, 255, 228, 255, 242, 255, 255, 87, 0],
            &[0, 58, 255, 255, 76, 0, 155, 255, 160, 0],
            &[0, 58, 255, 229, 0, 0, 88, 255, 179, 0],
            &[0, 58, 255, 206, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ņ' / 0x145
        'Ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 254, 254, 140, 0, 9, 254, 196, 0],
            &[0, 94, 255, 253, 223, 0, 9, 255, 197, 0],
            &[0, 94, 255, 199, 255, 52, 9, 255, 197, 0],
            &[0, 94, 255, 131, 255, 135, 9, 255, 197, 0],
            &[0, 94, 255, 109, 204, 219, 9, 255, 197, 0],
            &[0, 94, 255, 111, 121, 255, 54, 255, 197, 0],
            &[0, 94, 255, 111, 38, 255, 133, 255, 197, 0],
            &[0, 94, 255, 111, 0, 211, 214, 247, 197, 0],
            &[0, 94, 255, 111, 0, 128, 255, 255, 197, 0],
            &[0, 94, 255, 111, 0, 45, 255, 255, 197, 0],
            &[0, 0, 0, 0, 28, 37, 4, 0, 0, 0],
            &[0, 0, 0, 0, 219, 219, 3, 0, 0, 0],
            &[0, 0, 0, 4, 23, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ņ' / 0x146
        'ņ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 164, 69, 86, 181, 182, 101, 0, 0],
            &[0, 58, 255, 228, 255, 242, 255, 255, 87, 0],
            &[0, 58, 255, 255, 76, 0, 155, 255, 160, 0],
            &[0, 58, 255, 229, 0, 0, 88, 255, 179, 0],
            &[0, 58, 255, 206, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 0, 0, 0, 26, 37, 6, 0, 0, 0],
            &[0, 0, 0, 0, 205, 229, 7, 0, 0, 0],
            &[0, 0, 0, 3, 24, 1, 0, 0, 0, 0],
        ]),
        // letter: 'Ň' / 0x147
        'Ň' => Some(&[
            &[0, 0, 20, 223, 159, 56, 232, 128, 0, 0],
            &[0, 0, 0, 53, 251, 253, 188, 2, 0, 0],
            &[0, 0, 0, 0, 33, 41, 14, 0, 0, 0],
            &[0, 94, 254, 254, 140, 0, 9, 254, 196, 0],
            &[0, 94, 255, 253, 223, 0, 9, 255, 197, 0],
            &[0, 94, 255, 199, 255, 52, 9, 255, 197, 0],
            &[0, 94, 255, 131, 255, 135, 9, 255, 197, 0],
            &[0, 94, 255, 109, 204, 219, 9, 255, 197, 0],
            &[0, 94, 255, 111, 121, 255, 54, 255, 197, 0],
            &[0, 94, 255, 111, 38, 255, 133, 255, 197, 0],
            &[0, 94, 255, 111, 0, 211, 214, 247, 197, 0],
            &[0, 94, 255, 111, 0, 128, 255, 255, 197, 0],
            &[0, 94, 255, 111, 0, 45, 255, 255, 197, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 19, 0, 0, 9, 16, 0, 0],
        ]),
        // letter: 'ň' / 0x148
        'ň' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 181, 71, 11, 152, 120, 0, 0],
            &[0, 0, 0, 118, 250, 210, 229, 22, 0, 0],
            &[0, 0, 0, 2, 121, 132, 59, 0, 0, 0],
            &[0, 37, 164, 69, 86, 181, 182, 101, 0, 0],
            &[0, 58, 255, 228, 255, 242, 255, 255, 87, 0],
            &[0, 58, 255, 255, 76, 0, 155, 255, 160, 0],
            &[0, 58, 255, 229, 0, 0, 88, 255, 179, 0],
            &[0, 58, 255, 206, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŉ' / 0x149
        'ŉ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[4, 248, 245, 7, 0, 0, 0, 0, 0, 0],
            &[45, 255, 173, 0, 0, 0, 0, 0, 0, 0],
            &[98, 255, 90, 164, 92, 68, 176, 184, 105, 1],
            &[60, 108, 25, 255, 234, 250, 243, 255, 255, 89],
            &[0, 0, 23, 255, 255, 107, 0, 152, 255, 161],
            &[0, 0, 23, 255, 254, 9, 0, 88, 255, 179],
            &[0, 0, 23, 255, 242, 0, 0, 84, 255, 179],
            &[0, 0, 23, 255, 240, 0, 0, 84, 255, 179],
            &[0, 0, 23, 255, 240, 0, 0, 84, 255, 179],
            &[0, 0, 23, 255, 240, 0, 0, 84, 255, 179],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 254, 254, 140, 0, 9, 254, 196, 0],
            &[0, 94, 255, 253, 223, 0, 9, 255, 197, 0],
            &[0, 94, 255, 199, 255, 52, 9, 255, 197, 0],
            &[0, 94, 255, 131, 255, 135, 9, 255, 197, 0],
            &[0, 94, 255, 109, 204, 219, 9, 255, 197, 0],
            &[0, 94, 255, 111, 121, 255, 54, 255, 197, 0],
            &[0, 94, 255, 111, 38, 255, 133, 255, 197, 0],
            &[0, 94, 255, 111, 0, 211, 214, 247, 197, 0],
            &[0, 94, 255, 111, 0, 128, 255, 255, 255, 0],
            &[0, 94, 255, 111, 0, 45, 255, 255, 255, 0],
            &[0, 0, 0, 0, 0, 0, 12, 250, 175, 0],
            &[0, 0, 0, 0, 109, 150, 212, 255, 98, 0],
            &[0, 0, 0, 0, 139, 227, 201, 106, 0, 0],
        ]),
        // letter: 'ŋ' / 0x14b
        'ŋ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 164, 69, 86, 181, 182, 101, 0, 0],
            &[0, 58, 255, 228, 255, 242, 255, 255, 87, 0],
            &[0, 58, 255, 255, 76, 0, 155, 255, 160, 0],
            &[0, 58, 255, 229, 0, 0, 88, 255, 179, 0],
            &[0, 58, 255, 206, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 84, 255, 179, 0],
            &[0, 58, 255, 204, 0, 0, 131, 255, 255, 0],
            &[0, 58, 255, 204, 0, 0, 168, 255, 255, 0],
            &[0, 0, 0, 0, 0, 0, 86, 255, 178, 0],
            &[0, 0, 0, 0, 0, 76, 176, 255, 157, 0],
            &[0, 0, 0, 0, 0, 72, 83, 26, 0, 0],
        ]),
        // letter: 'Ō' / 0x14c
        'Ō' => Some(&[
            &[0, 0, 0, 105, 114, 114, 114, 35, 0, 0],
            &[0, 0, 0, 223, 240, 240, 240, 75, 0, 0],
            &[0, 0, 0, 0, 18, 33, 4, 0, 0, 0],
            &[0, 0, 29, 195, 255, 255, 236, 95, 0, 0],
            &[0, 1, 205, 255, 205, 173, 252, 254, 59, 0],
            &[0, 60, 255, 223, 6, 0, 133, 255, 166, 0],
            &[0, 117, 255, 152, 0, 0, 53, 255, 221, 0],
            &[0, 140, 255, 127, 0, 0, 28, 255, 244, 0],
            &[0, 142, 255, 127, 0, 0, 25, 255, 244, 0],
            &[0, 118, 255, 153, 0, 0, 52, 255, 220, 0],
            &[0, 62, 255, 224, 8, 0, 132, 255, 164, 0],
            &[0, 2, 205, 255, 210, 175, 252, 253, 56, 0],
            &[0, 0, 30, 191, 255, 255, 232, 90, 0, 0],
            &[0, 0, 0, 0, 16, 27, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ō' / 0x14d
        'ō' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 24, 24, 24, 7, 0, 0],
            &[0, 0, 0, 237, 255, 255, 255, 80, 0, 0],
            &[0, 0, 0, 69, 74, 74, 74, 23, 0, 0],
            &[0, 0, 0, 95, 177, 191, 143, 28, 0, 0],
            &[0, 0, 146, 255, 250, 240, 255, 232, 25, 0],
            &[0, 35, 255, 243, 33, 4, 171, 255, 141, 0],
            &[0, 94, 255, 177, 0, 0, 78, 255, 198, 0],
            &[0, 106, 255, 170, 0, 0, 68, 255, 208, 0],
            &[0, 68, 255, 222, 6, 0, 123, 255, 170, 0],
            &[0, 3, 215, 255, 196, 153, 245, 255, 65, 0],
            &[0, 0, 34, 193, 255, 255, 233, 96, 0, 0],
            &[0, 0, 0, 0, 17, 28, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => Some(&[
            &[0, 0, 11, 252, 108, 63, 218, 107, 0, 0],
            &[0, 0, 0, 115, 251, 255, 193, 10, 0, 0],
            &[0, 0, 0, 0, 33, 60, 5, 0, 0, 0],
            &[0, 0, 29, 195, 255, 255, 236, 95, 0, 0],
            &[0, 1, 205, 255, 205, 173, 252, 254, 59, 0],
            &[0, 60, 255, 223, 6, 0, 133, 255, 166, 0],
            &[0, 117, 255, 152, 0, 0, 53, 255, 221, 0],
            &[0, 140, 255, 127, 0, 0, 28, 255, 244, 0],
            &[0, 142, 255, 127, 0, 0, 25, 255, 244, 0],
            &[0, 118, 255, 153, 0, 0, 52, 255, 220, 0],
            &[0, 62, 255, 224, 8, 0, 132, 255, 164, 0],
            &[0, 2, 205, 255, 210, 175, 252, 253, 56, 0],
            &[0, 0, 30, 191, 255, 255, 232, 90, 0, 0],
            &[0, 0, 0, 0, 16, 27, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 30, 0, 0, 18, 16, 0, 0],
        ]),
        // letter: 'ŏ' / 0x14f
        'ŏ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 196, 33, 0, 147, 93, 0, 0],
            &[0, 0, 0, 192, 242, 229, 246, 41, 0, 0],
            &[0, 0, 0, 9, 100, 116, 37, 0, 0, 0],
            &[0, 0, 0, 95, 177, 191, 143, 28, 0, 0],
            &[0, 0, 146, 255, 250, 240, 255, 232, 25, 0],
            &[0, 35, 255, 243, 33, 4, 171, 255, 141, 0],
            &[0, 94, 255, 177, 0, 0, 78, 255, 198, 0],
            &[0, 106, 255, 170, 0, 0, 68, 255, 208, 0],
            &[0, 68, 255, 222, 6, 0, 123, 255, 170, 0],
            &[0, 3, 215, 255, 196, 153, 245, 255, 65, 0],
            &[0, 0, 34, 193, 255, 255, 233, 96, 0, 0],
            &[0, 0, 0, 0, 17, 28, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ő' / 0x150
        'Ő' => Some(&[
            &[0, 0, 0, 110, 255, 113, 226, 218, 15, 0],
            &[0, 0, 21, 237, 116, 129, 216, 29, 0, 0],
            &[0, 0, 12, 32, 18, 67, 17, 0, 0, 0],
            &[0, 0, 29, 195, 255, 255, 236, 95, 0, 0],
            &[0, 1, 205, 255, 205, 173, 252, 254, 59, 0],
            &[0, 60, 255, 223, 6, 0, 133, 255, 166, 0],
            &[0, 117, 255, 152, 0, 0, 53, 255, 221, 0],
            &[0, 140, 255, 127, 0, 0, 28, 255, 244, 0],
            &[0, 142, 255, 127, 0, 0, 25, 255, 244, 0],
            &[0, 118, 255, 153, 0, 0, 52, 255, 220, 0],
            &[0, 62, 255, 224, 8, 0, 132, 255, 164, 0],
            &[0, 2, 205, 255, 210, 175, 252, 253, 56, 0],
            &[0, 0, 30, 191, 255, 255, 232, 90, 0, 0],
            &[0, 0, 0, 0, 16, 27, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 19, 14, 12, 19, 5, 0],
        ]),
        // letter: 'ő' / 0x151
        'ő' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 57, 185, 106, 148, 179, 20, 0],
            &[0, 0, 2, 203, 196, 87, 251, 87, 0, 0],
            &[0, 0, 31, 121, 9, 96, 68, 0, 0, 0],
            &[0, 0, 0, 95, 177, 191, 143, 28, 0, 0],
            &[0, 0, 146, 255, 250, 240, 255, 232, 25, 0],
            &[0, 35, 255, 243, 33, 4, 171, 255, 141, 0],
            &[0, 94, 255, 177, 0, 0, 78, 255, 198, 0],
            &[0, 106, 255, 170, 0, 0, 68, 255, 208, 0],
            &[0, 68, 255, 222, 6, 0, 123, 255, 170, 0],
            &[0, 3, 215, 255, 196, 153, 245, 255, 65, 0],
            &[0, 0, 34, 193, 255, 255, 233, 96, 0, 0],
            &[0, 0, 0, 0, 17, 28, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Œ' / 0x152
        'Œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 29, 9, 0, 0, 0, 0, 0],
            &[0, 65, 231, 255, 255, 254, 254, 254, 30, 0],
            &[9, 233, 255, 191, 232, 255, 202, 189, 22, 0],
            &[77, 255, 199, 0, 179, 255, 51, 0, 0, 0],
            &[124, 255, 132, 0, 179, 255, 64, 13, 0, 0],
            &[143, 255, 111, 0, 179, 255, 255, 222, 0, 0],
            &[141, 255, 112, 0, 179, 255, 189, 149, 0, 0],
            &[123, 255, 128, 0, 179, 255, 51, 0, 0, 0],
            &[78, 255, 181, 0, 179, 255, 51, 0, 0, 0],
            &[8, 235, 255, 181, 233, 255, 201, 188, 22, 0],
            &[0, 68, 233, 255, 255, 255, 255, 255, 30, 0],
            &[0, 0, 2, 28, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'œ' / 0x153
        'œ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 88, 186, 133, 20, 154, 173, 44, 0, 0],
            &[53, 253, 211, 251, 235, 224, 217, 218, 1, 0],
            &[143, 254, 17, 163, 255, 87, 84, 255, 38, 0],
            &[176, 238, 0, 122, 255, 175, 177, 255, 63, 0],
            &[185, 235, 0, 116, 255, 187, 167, 167, 43, 0],
            &[160, 250, 4, 140, 255, 83, 0, 0, 0, 0],
            &[92, 255, 131, 227, 255, 203, 77, 135, 12, 0],
            &[4, 180, 255, 227, 96, 237, 255, 220, 10, 0],
            &[0, 0, 22, 1, 0, 3, 23, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŕ' / 0x154
        'Ŕ' => Some(&[
            &[0, 0, 0, 0, 188, 254, 102, 0, 0, 0],
            &[0, 0, 0, 107, 250, 93, 0, 0, 0, 0],
            &[0, 0, 0, 32, 28, 0, 0, 0, 0, 0],
            &[0, 224, 254, 254, 239, 183, 49, 0, 0, 0],
            &[0, 225, 255, 197, 218, 255, 240, 22, 0, 0],
            &[0, 225, 255, 38, 0, 194, 255, 94, 0, 0],
            &[0, 225, 255, 38, 0, 184, 255, 89, 0, 0],
            &[0, 225, 255, 145, 170, 255, 240, 23, 0, 0],
            &[0, 225, 255, 255, 255, 238, 60, 0, 0, 0],
            &[0, 225, 255, 89, 223, 252, 45, 0, 0, 0],
            &[0, 225, 255, 38, 89, 255, 189, 0, 0, 0],
            &[0, 225, 255, 38, 1, 202, 255, 82, 0, 0],
            &[0, 225, 255, 38, 0, 62, 255, 222, 7, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 19, 16, 0, 0, 0],
        ]),
        // letter: 'ŕ' / 0x155
        'ŕ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 184, 171, 16, 0, 0],
            &[0, 0, 0, 1, 190, 237, 56, 0, 0, 0],
            &[0, 0, 0, 35, 132, 39, 0, 0, 0, 0],
            &[7, 164, 164, 164, 28, 100, 189, 180, 12, 0],
            &[9, 224, 253, 255, 189, 255, 255, 220, 0, 0],
            &[0, 0, 113, 255, 255, 127, 48, 46, 0, 0],
            &[0, 0, 112, 255, 196, 0, 0, 0, 0, 0],
            &[0, 0, 112, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 112, 255, 151, 0, 0, 0, 0, 0],
            &[33, 116, 201, 255, 218, 125, 48, 0, 0, 0],
            &[109, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŗ' / 0x156
        'Ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 224, 254, 254, 239, 183, 49, 0, 0, 0],
            &[0, 225, 255, 197, 218, 255, 240, 22, 0, 0],
            &[0, 225, 255, 38, 0, 194, 255, 94, 0, 0],
            &[0, 225, 255, 38, 0, 184, 255, 89, 0, 0],
            &[0, 225, 255, 145, 170, 255, 240, 23, 0, 0],
            &[0, 225, 255, 255, 255, 238, 60, 0, 0, 0],
            &[0, 225, 255, 89, 223, 252, 45, 0, 0, 0],
            &[0, 225, 255, 38, 89, 255, 189, 0, 0, 0],
            &[0, 225, 255, 38, 1, 202, 255, 82, 0, 0],
            &[0, 225, 255, 38, 0, 62, 255, 222, 7, 0],
            &[0, 0, 0, 15, 37, 17, 0, 0, 0, 0],
            &[0, 0, 0, 130, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 20, 8, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŗ' / 0x157
        'ŗ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[7, 164, 164, 164, 28, 100, 189, 180, 12, 0],
            &[9, 224, 253, 255, 189, 255, 255, 220, 0, 0],
            &[0, 0, 113, 255, 255, 127, 48, 46, 0, 0],
            &[0, 0, 112, 255, 196, 0, 0, 0, 0, 0],
            &[0, 0, 112, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 112, 255, 151, 0, 0, 0, 0, 0],
            &[33, 116, 201, 255, 218, 125, 48, 0, 0, 0],
            &[109, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 12, 37, 20, 0, 0, 0, 0, 0],
            &[0, 0, 107, 255, 79, 0, 0, 0, 0, 0],
            &[0, 0, 17, 10, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ř' / 0x158
        'Ř' => Some(&[
            &[0, 29, 232, 143, 65, 238, 110, 0, 0, 0],
            &[0, 0, 68, 254, 253, 171, 0, 0, 0, 0],
            &[0, 0, 0, 36, 41, 11, 0, 0, 0, 0],
            &[0, 224, 254, 254, 239, 183, 49, 0, 0, 0],
            &[0, 225, 255, 197, 218, 255, 240, 22, 0, 0],
            &[0, 225, 255, 38, 0, 194, 255, 94, 0, 0],
            &[0, 225, 255, 38, 0, 184, 255, 89, 0, 0],
            &[0, 225, 255, 145, 170, 255, 240, 23, 0, 0],
            &[0, 225, 255, 255, 255, 238, 60, 0, 0, 0],
            &[0, 225, 255, 89, 223, 252, 45, 0, 0, 0],
            &[0, 225, 255, 38, 89, 255, 189, 0, 0, 0],
            &[0, 225, 255, 38, 1, 202, 255, 82, 0, 0],
            &[0, 225, 255, 38, 0, 62, 255, 222, 7, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 18, 0, 0, 10, 14, 0, 0, 0],
        ]),
        // letter: 'ř' / 0x159
        'ř' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 142, 135, 3, 96, 174, 13, 0, 0],
            &[0, 0, 37, 242, 205, 255, 89, 0, 0, 0],
            &[0, 0, 0, 75, 132, 107, 0, 0, 0, 0],
            &[7, 164, 164, 164, 28, 100, 189, 180, 12, 0],
            &[9, 224, 253, 255, 189, 255, 255, 220, 0, 0],
            &[0, 0, 113, 255, 255, 127, 48, 46, 0, 0],
            &[0, 0, 112, 255, 196, 0, 0, 0, 0, 0],
            &[0, 0, 112, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 112, 255, 151, 0, 0, 0, 0, 0],
            &[33, 116, 201, 255, 218, 125, 48, 0, 0, 0],
            &[109, 255, 255, 255, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ś' / 0x15a
        'Ś' => Some(&[
            &[0, 0, 0, 0, 0, 138, 255, 150, 0, 0],
            &[0, 0, 0, 0, 60, 252, 136, 0, 0, 0],
            &[0, 0, 0, 0, 35, 65, 9, 0, 0, 0],
            &[0, 0, 18, 178, 252, 255, 255, 214, 54, 0],
            &[0, 0, 156, 255, 223, 149, 174, 235, 19, 0],
            &[0, 0, 203, 255, 80, 0, 0, 3, 0, 0],
            &[0, 0, 163, 255, 182, 19, 0, 0, 0, 0],
            &[0, 0, 30, 222, 255, 242, 124, 9, 0, 0],
            &[0, 0, 0, 12, 143, 248, 255, 216, 23, 0],
            &[0, 0, 0, 0, 0, 33, 211, 255, 132, 0],
            &[0, 1, 30, 0, 0, 0, 135, 255, 152, 0],
            &[0, 5, 253, 194, 152, 160, 246, 255, 79, 0],
            &[0, 2, 181, 249, 255, 255, 225, 103, 0, 0],
            &[0, 0, 0, 4, 26, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 19, 19, 0, 0],
        ]),
        // letter: 'ś' / 0x15b
        'ś' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 98, 185, 123, 0, 0, 0],
            &[0, 0, 0, 37, 244, 191, 13, 0, 0, 0],
            &[0, 0, 0, 81, 118, 6, 0, 0, 0, 0],
            &[0, 6, 116, 183, 193, 169, 107, 13, 0, 0],
            &[0, 147, 255, 250, 234, 254, 241, 11, 0, 0],
            &[0, 202, 255, 106, 0, 14, 52, 0, 0, 0],
            &[0, 120, 255, 254, 187, 84, 2, 0, 0, 0],
            &[0, 0, 81, 199, 255, 255, 181, 3, 0, 0],
            &[0, 14, 0, 0, 51, 241, 255, 57, 0, 0],
            &[0, 221, 195, 158, 162, 253, 255, 42, 0, 0],
            &[0, 181, 252, 255, 255, 238, 120, 0, 0, 0],
            &[0, 0, 9, 29, 25, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => Some(&[
            &[0, 0, 0, 22, 234, 255, 203, 5, 0, 0],
            &[0, 0, 4, 190, 201, 62, 228, 145, 0, 0],
            &[0, 0, 6, 41, 19, 29, 25, 40, 0, 0],
            &[0, 0, 18, 178, 252, 255, 255, 214, 54, 0],
            &[0, 0, 156, 255, 223, 149, 174, 235, 19, 0],
            &[0, 0, 203, 255, 80, 0, 0, 3, 0, 0],
            &[0, 0, 163, 255, 182, 19, 0, 0, 0, 0],
            &[0, 0, 30, 222, 255, 242, 124, 9, 0, 0],
            &[0, 0, 0, 12, 143, 248, 255, 216, 23, 0],
            &[0, 0, 0, 0, 0, 33, 211, 255, 132, 0],
            &[0, 1, 30, 0, 0, 0, 135, 255, 152, 0],
            &[0, 5, 253, 194, 152, 160, 246, 255, 79, 0],
            &[0, 2, 181, 249, 255, 255, 225, 103, 0, 0],
            &[0, 0, 0, 4, 26, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 19, 8, 0, 0, 0],
        ]),
        // letter: 'ŝ' / 0x15d
        'ŝ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 173, 185, 100, 0, 0, 0, 0],
            &[0, 0, 155, 235, 162, 248, 50, 0, 0, 0],
            &[0, 26, 131, 36, 0, 95, 98, 0, 0, 0],
            &[0, 6, 116, 183, 193, 169, 107, 13, 0, 0],
            &[0, 147, 255, 250, 234, 254, 241, 11, 0, 0],
            &[0, 202, 255, 106, 0, 14, 52, 0, 0, 0],
            &[0, 120, 255, 254, 187, 84, 2, 0, 0, 0],
            &[0, 0, 81, 199, 255, 255, 181, 3, 0, 0],
            &[0, 14, 0, 0, 51, 241, 255, 57, 0, 0],
            &[0, 221, 195, 158, 162, 253, 255, 42, 0, 0],
            &[0, 181, 252, 255, 255, 238, 120, 0, 0, 0],
            &[0, 0, 9, 29, 25, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ş' / 0x15e
        'Ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 29, 9, 0, 0, 0],
            &[0, 0, 18, 178, 252, 255, 255, 214, 54, 0],
            &[0, 0, 156, 255, 223, 149, 174, 235, 19, 0],
            &[0, 0, 203, 255, 80, 0, 0, 3, 0, 0],
            &[0, 0, 163, 255, 182, 19, 0, 0, 0, 0],
            &[0, 0, 30, 222, 255, 242, 124, 9, 0, 0],
            &[0, 0, 0, 12, 143, 248, 255, 216, 23, 0],
            &[0, 0, 0, 0, 0, 33, 211, 255, 132, 0],
            &[0, 1, 30, 0, 0, 0, 135, 255, 152, 0],
            &[0, 5, 253, 194, 152, 160, 246, 255, 79, 0],
            &[0, 2, 181, 249, 255, 255, 225, 103, 0, 0],
            &[0, 0, 0, 4, 163, 212, 15, 0, 0, 0],
            &[0, 0, 0, 0, 41, 228, 158, 0, 0, 0],
            &[0, 0, 0, 14, 85, 65, 1, 0, 0, 0],
        ]),
        // letter: 'ş' / 0x15f
        'ş' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 116, 183, 193, 169, 107, 13, 0, 0],
            &[0, 147, 255, 250, 234, 254, 241, 11, 0, 0],
            &[0, 202, 255, 106, 0, 14, 52, 0, 0, 0],
            &[0, 120, 255, 254, 187, 84, 2, 0, 0, 0],
            &[0, 0, 81, 199, 255, 255, 181, 3, 0, 0],
            &[0, 14, 0, 0, 51, 241, 255, 57, 0, 0],
            &[0, 221, 195, 158, 162, 253, 255, 42, 0, 0],
            &[0, 181, 252, 255, 255, 238, 120, 0, 0, 0],
            &[0, 0, 9, 148, 234, 21, 0, 0, 0, 0],
            &[0, 0, 0, 36, 216, 176, 0, 0, 0, 0],
            &[0, 0, 10, 84, 69, 3, 0, 0, 0, 0],
        ]),
        // letter: 'Š' / 0x160
        'Š' => Some(&[
            &[0, 0, 8, 203, 184, 45, 218, 160, 0, 0],
            &[0, 0, 0, 31, 241, 253, 213, 9, 0, 0],
            &[0, 0, 0, 0, 40, 71, 29, 0, 0, 0],
            &[0, 0, 18, 178, 252, 255, 255, 214, 54, 0],
            &[0, 0, 156, 255, 223, 149, 174, 235, 19, 0],
            &[0, 0, 203, 255, 80, 0, 0, 3, 0, 0],
            &[0, 0, 163, 255, 182, 19, 0, 0, 0, 0],
            &[0, 0, 30, 222, 255, 242, 124, 9, 0, 0],
            &[0, 0, 0, 12, 143, 248, 255, 216, 23, 0],
            &[0, 0, 0, 0, 0, 33, 211, 255, 132, 0],
            &[0, 1, 30, 0, 0, 0, 135, 255, 152, 0],
            &[0, 5, 253, 194, 152, 160, 246, 255, 79, 0],
            &[0, 2, 181, 249, 255, 255, 225, 103, 0, 0],
            &[0, 0, 0, 4, 26, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 19, 2, 0, 6, 18, 0, 0],
        ]),
        // letter: 'š' / 0x161
        'š' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 180, 73, 10, 151, 122, 0, 0, 0],
            &[0, 0, 114, 251, 210, 231, 23, 0, 0, 0],
            &[0, 0, 1, 119, 132, 61, 0, 0, 0, 0],
            &[0, 6, 116, 183, 193, 169, 107, 13, 0, 0],
            &[0, 147, 255, 250, 234, 254, 241, 11, 0, 0],
            &[0, 202, 255, 106, 0, 14, 52, 0, 0, 0],
            &[0, 120, 255, 254, 187, 84, 2, 0, 0, 0],
            &[0, 0, 81, 199, 255, 255, 181, 3, 0, 0],
            &[0, 14, 0, 0, 51, 241, 255, 57, 0, 0],
            &[0, 221, 195, 158, 162, 253, 255, 42, 0, 0],
            &[0, 181, 252, 255, 255, 238, 120, 0, 0, 0],
            &[0, 0, 9, 29, 25, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ţ' / 0x162
        'Ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 147, 254, 254, 254, 254, 254, 254, 249, 0],
            &[0, 111, 192, 192, 244, 255, 206, 192, 189, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 215, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 136, 196, 15, 0, 0, 0],
            &[0, 0, 0, 0, 41, 228, 158, 0, 0, 0],
            &[0, 0, 0, 14, 85, 65, 1, 0, 0, 0],
        ]),
        // letter: 'ţ' / 0x163
        'ţ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 140, 73, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 133, 0, 0, 0, 0],
            &[0, 16, 100, 182, 255, 212, 164, 164, 64, 0],
            &[0, 66, 255, 255, 255, 255, 255, 255, 100, 0],
            &[0, 2, 9, 134, 255, 137, 9, 9, 3, 0],
            &[0, 0, 0, 129, 255, 133, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 133, 0, 0, 0, 0],
            &[0, 0, 0, 125, 255, 153, 0, 0, 0, 0],
            &[0, 0, 0, 83, 255, 249, 158, 157, 85, 0],
            &[0, 0, 0, 3, 159, 255, 255, 253, 104, 0],
            &[0, 0, 0, 0, 0, 167, 212, 23, 0, 0],
            &[0, 0, 0, 0, 0, 48, 241, 139, 0, 0],
            &[0, 0, 0, 0, 20, 86, 60, 0, 0, 0],
        ]),
        // letter: 'Ť' / 0x164
        'Ť' => Some(&[
            &[0, 0, 29, 232, 143, 65, 238, 110, 0, 0],
            &[0, 0, 0, 68, 254, 253, 171, 0, 0, 0],
            &[0, 0, 0, 0, 36, 41, 11, 0, 0, 0],
            &[0, 147, 254, 254, 254, 254, 254, 254, 249, 0],
            &[0, 111, 192, 192, 244, 255, 206, 192, 189, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 18, 0, 0, 10, 14, 0, 0],
        ]),
        // letter: 'ť' / 0x165
        'ť' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 183, 181, 0, 0],
            &[0, 0, 9, 140, 73, 20, 255, 115, 0, 0],
            &[0, 0, 51, 255, 133, 27, 112, 6, 0, 0],
            &[16, 100, 182, 255, 212, 164, 164, 64, 0, 0],
            &[66, 255, 255, 255, 255, 255, 255, 100, 0, 0],
            &[2, 9, 134, 255, 137, 9, 9, 3, 0, 0],
            &[0, 0, 129, 255, 133, 0, 0, 0, 0, 0],
            &[0, 0, 129, 255, 133, 0, 0, 0, 0, 0],
            &[0, 0, 125, 255, 153, 0, 0, 0, 0, 0],
            &[0, 0, 83, 255, 249, 158, 157, 85, 0, 0],
            &[0, 0, 3, 159, 251, 255, 253, 104, 0, 0],
            &[0, 0, 0, 0, 11, 30, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŧ' / 0x166
        'Ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 147, 254, 254, 254, 254, 254, 254, 249, 0],
            &[0, 111, 192, 192, 244, 255, 206, 192, 189, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 108, 200, 255, 255, 253, 188, 0, 0],
            &[0, 0, 123, 226, 255, 255, 255, 213, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 53, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ŧ' / 0x167
        'ŧ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 140, 73, 0, 0, 0, 0],
            &[0, 0, 0, 51, 255, 133, 0, 0, 0, 0],
            &[0, 16, 100, 182, 255, 212, 164, 164, 64, 0],
            &[0, 66, 255, 255, 255, 255, 255, 255, 100, 0],
            &[0, 2, 9, 134, 255, 137, 9, 9, 3, 0],
            &[0, 51, 198, 255, 255, 255, 198, 198, 91, 0],
            &[0, 35, 138, 255, 255, 255, 138, 138, 64, 0],
            &[0, 0, 0, 125, 255, 153, 0, 0, 0, 0],
            &[0, 0, 0, 83, 255, 249, 158, 157, 85, 0],
            &[0, 0, 0, 3, 159, 251, 255, 253, 104, 0],
            &[0, 0, 0, 0, 0, 11, 30, 12, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ũ' / 0x168
        'Ũ' => Some(&[
            &[0, 0, 8, 181, 212, 114, 114, 187, 0, 0],
            &[0, 0, 85, 220, 142, 236, 255, 101, 0, 0],
            &[0, 0, 20, 26, 0, 6, 25, 0, 0, 0],
            &[0, 94, 254, 168, 0, 0, 66, 254, 196, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 92, 255, 178, 0, 0, 75, 255, 194, 0],
            &[0, 68, 255, 224, 7, 0, 127, 255, 169, 0],
            &[0, 8, 230, 255, 208, 177, 248, 255, 79, 0],
            &[0, 0, 51, 208, 255, 255, 237, 115, 0, 0],
            &[0, 0, 0, 0, 18, 26, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ũ' / 0x169
        'ũ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 122, 28, 38, 121, 0, 0],
            &[0, 0, 57, 250, 230, 251, 242, 154, 0, 0],
            &[0, 0, 56, 86, 2, 76, 115, 13, 0, 0],
            &[0, 49, 164, 120, 0, 0, 65, 164, 103, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 187, 0, 0, 116, 255, 161, 0],
            &[0, 68, 255, 212, 0, 0, 172, 255, 161, 0],
            &[0, 26, 251, 255, 171, 166, 254, 255, 161, 0],
            &[0, 0, 112, 249, 255, 223, 82, 252, 161, 0],
            &[0, 0, 0, 6, 25, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ū' / 0x16a
        'Ū' => Some(&[
            &[0, 0, 0, 105, 114, 114, 114, 35, 0, 0],
            &[0, 0, 0, 223, 240, 240, 240, 75, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 254, 168, 0, 0, 66, 254, 196, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 92, 255, 178, 0, 0, 75, 255, 194, 0],
            &[0, 68, 255, 224, 7, 0, 127, 255, 169, 0],
            &[0, 8, 230, 255, 208, 177, 248, 255, 79, 0],
            &[0, 0, 51, 208, 255, 255, 237, 115, 0, 0],
            &[0, 0, 0, 0, 18, 26, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ū' / 0x16b
        'ū' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 24, 24, 24, 7, 0, 0],
            &[0, 0, 0, 237, 255, 255, 255, 80, 0, 0],
            &[0, 0, 0, 69, 74, 74, 74, 23, 0, 0],
            &[0, 49, 164, 120, 0, 0, 65, 164, 103, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 187, 0, 0, 116, 255, 161, 0],
            &[0, 68, 255, 212, 0, 0, 172, 255, 161, 0],
            &[0, 26, 251, 255, 171, 166, 254, 255, 161, 0],
            &[0, 0, 112, 249, 255, 223, 82, 252, 161, 0],
            &[0, 0, 0, 6, 25, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => Some(&[
            &[0, 0, 11, 252, 108, 63, 218, 107, 0, 0],
            &[0, 0, 0, 115, 251, 255, 193, 10, 0, 0],
            &[0, 0, 0, 0, 14, 26, 0, 0, 0, 0],
            &[0, 94, 254, 168, 0, 0, 66, 254, 196, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 92, 255, 178, 0, 0, 75, 255, 194, 0],
            &[0, 68, 255, 224, 7, 0, 127, 255, 169, 0],
            &[0, 8, 230, 255, 208, 177, 248, 255, 79, 0],
            &[0, 0, 51, 208, 255, 255, 237, 115, 0, 0],
            &[0, 0, 0, 0, 18, 26, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 30, 0, 0, 18, 16, 0, 0],
        ]),
        // letter: 'ŭ' / 0x16d
        'ŭ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 196, 33, 0, 147, 93, 0, 0],
            &[0, 0, 0, 192, 242, 229, 246, 41, 0, 0],
            &[0, 0, 0, 9, 100, 116, 37, 0, 0, 0],
            &[0, 49, 164, 120, 0, 0, 65, 164, 103, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 187, 0, 0, 116, 255, 161, 0],
            &[0, 68, 255, 212, 0, 0, 172, 255, 161, 0],
            &[0, 26, 251, 255, 171, 166, 254, 255, 161, 0],
            &[0, 0, 112, 249, 255, 223, 82, 252, 161, 0],
            &[0, 0, 0, 6, 25, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ů' / 0x16e
        'Ů' => Some(&[
            &[0, 0, 0, 111, 162, 70, 209, 0, 0, 0],
            &[0, 0, 0, 46, 235, 234, 126, 0, 0, 0],
            &[0, 0, 0, 0, 12, 25, 0, 0, 0, 0],
            &[0, 94, 254, 168, 0, 0, 66, 254, 196, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 92, 255, 178, 0, 0, 75, 255, 194, 0],
            &[0, 68, 255, 224, 7, 0, 127, 255, 169, 0],
            &[0, 8, 230, 255, 208, 177, 248, 255, 79, 0],
            &[0, 0, 51, 208, 255, 255, 237, 115, 0, 0],
            &[0, 0, 0, 0, 18, 26, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 226, 243, 105, 0, 0, 0],
        ]),
        // letter: 'ů' / 0x16f
        'ů' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 137, 163, 43, 0, 0, 0],
            &[0, 0, 0, 101, 195, 130, 198, 0, 0, 0],
            &[0, 0, 0, 84, 216, 169, 180, 0, 0, 0],
            &[0, 0, 0, 1, 88, 115, 19, 0, 0, 0],
            &[0, 49, 164, 120, 0, 0, 65, 164, 103, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 187, 0, 0, 116, 255, 161, 0],
            &[0, 68, 255, 212, 0, 0, 172, 255, 161, 0],
            &[0, 26, 251, 255, 171, 166, 254, 255, 161, 0],
            &[0, 0, 112, 249, 255, 223, 82, 252, 161, 0],
            &[0, 0, 0, 6, 25, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ű' / 0x170
        'Ű' => Some(&[
            &[0, 0, 0, 110, 255, 113, 226, 218, 15, 0],
            &[0, 0, 21, 237, 116, 129, 216, 29, 0, 0],
            &[0, 0, 12, 32, 0, 33, 13, 0, 0, 0],
            &[0, 94, 254, 168, 0, 0, 66, 254, 196, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 92, 255, 178, 0, 0, 75, 255, 194, 0],
            &[0, 68, 255, 224, 7, 0, 127, 255, 169, 0],
            &[0, 8, 230, 255, 208, 177, 248, 255, 79, 0],
            &[0, 0, 51, 208, 255, 255, 237, 115, 0, 0],
            &[0, 0, 0, 0, 18, 26, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 19, 14, 12, 19, 5, 0],
        ]),
        // letter: 'ű' / 0x171
        'ű' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 57, 185, 106, 148, 179, 20, 0],
            &[0, 0, 2, 203, 196, 87, 251, 87, 0, 0],
            &[0, 0, 31, 121, 9, 96, 68, 0, 0, 0],
            &[0, 49, 164, 120, 0, 0, 65, 164, 103, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 187, 0, 0, 116, 255, 161, 0],
            &[0, 68, 255, 212, 0, 0, 172, 255, 161, 0],
            &[0, 26, 251, 255, 171, 166, 254, 255, 161, 0],
            &[0, 0, 112, 249, 255, 223, 82, 252, 161, 0],
            &[0, 0, 0, 6, 25, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ų' / 0x172
        'Ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 254, 168, 0, 0, 66, 254, 196, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 94, 255, 168, 0, 0, 66, 255, 197, 0],
            &[0, 92, 255, 178, 0, 0, 75, 255, 194, 0],
            &[0, 68, 255, 224, 7, 0, 127, 255, 169, 0],
            &[0, 8, 230, 255, 208, 177, 248, 255, 79, 0],
            &[0, 0, 51, 208, 255, 255, 255, 117, 0, 0],
            &[0, 0, 0, 0, 112, 254, 49, 0, 0, 0],
            &[0, 0, 0, 0, 232, 132, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 71, 37, 0, 0, 0],
        ]),
        // letter: 'ų' / 0x173
        'ų' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 164, 120, 0, 0, 65, 164, 103, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 186, 0, 0, 102, 255, 161, 0],
            &[0, 76, 255, 187, 0, 0, 116, 255, 161, 0],
            &[0, 68, 255, 212, 0, 0, 172, 255, 161, 0],
            &[0, 26, 251, 255, 171, 166, 254, 255, 161, 0],
            &[0, 0, 112, 249, 255, 223, 82, 255, 164, 0],
            &[0, 0, 0, 6, 25, 1, 93, 220, 18, 0],
            &[0, 0, 0, 0, 0, 0, 225, 142, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 71, 40, 0],
        ]),
        // letter: 'Ŵ' / 0x174
        'Ŵ' => Some(&[
            &[0, 0, 54, 252, 255, 159, 0, 0, 0, 0],
            &[0, 21, 223, 165, 81, 245, 95, 0, 0, 0],
            &[0, 15, 40, 0, 0, 24, 31, 0, 0, 0],
            &[209, 242, 0, 0, 0, 0, 137, 254, 57, 0],
            &[191, 255, 11, 0, 0, 0, 156, 255, 39, 0],
            &[172, 255, 34, 132, 164, 37, 174, 255, 21, 0],
            &[153, 255, 53, 237, 255, 90, 192, 255, 4, 0],
            &[134, 255, 90, 255, 234, 128, 209, 241, 0, 0],
            &[115, 255, 138, 255, 168, 167, 220, 223, 0, 0],
            &[96, 255, 176, 228, 121, 208, 220, 206, 0, 0],
            &[77, 255, 210, 190, 84, 247, 214, 188, 0, 0],
            &[58, 255, 247, 152, 46, 255, 246, 170, 0, 0],
            &[39, 255, 255, 114, 10, 254, 255, 152, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 19, 5, 0, 0, 0, 0],
        ]),
        // letter: 'ŵ' / 0x175
        'ŵ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 176, 185, 92, 0, 0, 0, 0],
            &[0, 0, 165, 230, 166, 244, 43, 0, 0, 0],
            &[0, 32, 131, 31, 0, 100, 92, 0, 0, 0],
            &[150, 151, 0, 0, 0, 0, 78, 164, 53, 0],
            &[203, 251, 1, 94, 134, 17, 140, 255, 53, 0],
            &[167, 255, 20, 219, 255, 73, 163, 255, 17, 0],
            &[131, 255, 59, 254, 244, 124, 185, 237, 0, 0],
            &[95, 255, 131, 254, 173, 174, 207, 201, 0, 0],
            &[59, 255, 198, 220, 118, 220, 229, 165, 0, 0],
            &[23, 255, 250, 173, 72, 253, 252, 129, 0, 0],
            &[0, 242, 255, 126, 27, 255, 255, 93, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ŷ' / 0x176
        'Ŷ' => Some(&[
            &[0, 0, 54, 252, 255, 159, 0, 0, 0, 0],
            &[0, 21, 223, 165, 81, 245, 95, 0, 0, 0],
            &[0, 15, 40, 0, 0, 24, 31, 0, 0, 0],
            &[123, 254, 161, 0, 0, 55, 254, 221, 3, 0],
            &[16, 241, 248, 21, 0, 162, 255, 105, 0, 0],
            &[0, 136, 255, 121, 21, 248, 231, 8, 0, 0],
            &[0, 23, 246, 226, 124, 255, 118, 0, 0, 0],
            &[0, 0, 148, 255, 249, 239, 13, 0, 0, 0],
            &[0, 0, 31, 251, 255, 132, 0, 0, 0, 0],
            &[0, 0, 0, 207, 255, 57, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 19, 5, 0, 0, 0, 0],
        ]),
        // letter: 'ŷ' / 0x177
        'ŷ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 178, 185, 87, 0, 0, 0, 0],
            &[0, 0, 172, 227, 170, 242, 38, 0, 0, 0],
            &[0, 35, 130, 28, 1, 103, 89, 0, 0, 0],
            &[97, 164, 98, 0, 0, 26, 164, 160, 5, 0],
            &[68, 255, 223, 0, 0, 103, 255, 180, 0, 0],
            &[1, 221, 255, 53, 0, 178, 255, 87, 0, 0],
            &[0, 121, 255, 139, 8, 245, 241, 8, 0, 0],
            &[0, 24, 251, 223, 74, 255, 157, 0, 0, 0],
            &[0, 0, 173, 255, 184, 255, 64, 0, 0, 0],
            &[0, 0, 72, 255, 254, 225, 1, 0, 0, 0],
            &[0, 0, 2, 224, 255, 133, 0, 0, 0, 0],
            &[0, 0, 7, 230, 254, 39, 0, 0, 0, 0],
            &[17, 93, 177, 255, 178, 0, 0, 0, 0, 0],
            &[14, 102, 87, 12, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ÿ' / 0x178
        'Ÿ' => Some(&[
            &[0, 54, 220, 76, 8, 208, 135, 0, 0, 0],
            &[0, 70, 251, 97, 14, 239, 166, 0, 0, 0],
            &[0, 0, 8, 0, 0, 7, 0, 0, 0, 0],
            &[123, 254, 161, 0, 0, 55, 254, 221, 3, 0],
            &[16, 241, 248, 21, 0, 162, 255, 105, 0, 0],
            &[0, 136, 255, 121, 21, 248, 231, 8, 0, 0],
            &[0, 23, 246, 226, 124, 255, 118, 0, 0, 0],
            &[0, 0, 148, 255, 249, 239, 13, 0, 0, 0],
            &[0, 0, 31, 251, 255, 132, 0, 0, 0, 0],
            &[0, 0, 0, 207, 255, 57, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ź' / 0x179
        'Ź' => Some(&[
            &[0, 0, 0, 0, 145, 255, 144, 0, 0, 0],
            &[0, 0, 0, 66, 253, 130, 0, 0, 0, 0],
            &[0, 0, 0, 25, 35, 0, 0, 0, 0, 0],
            &[0, 135, 254, 254, 254, 254, 254, 99, 0, 0],
            &[0, 98, 185, 185, 186, 252, 255, 70, 0, 0],
            &[0, 0, 0, 0, 83, 255, 186, 0, 0, 0],
            &[0, 0, 0, 8, 223, 250, 41, 0, 0, 0],
            &[0, 0, 0, 124, 255, 141, 0, 0, 0, 0],
            &[0, 0, 27, 244, 232, 14, 0, 0, 0, 0],
            &[0, 0, 165, 255, 96, 0, 0, 0, 0, 0],
            &[0, 58, 254, 200, 1, 0, 0, 0, 0, 0],
            &[0, 195, 255, 218, 191, 191, 191, 80, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 19, 19, 0, 0, 0],
        ]),
        // letter: 'ź' / 0x17a
        'ź' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 90, 185, 131, 0, 0],
            &[0, 0, 0, 0, 30, 240, 198, 17, 0, 0],
            &[0, 0, 0, 0, 76, 121, 9, 0, 0, 0],
            &[0, 0, 145, 164, 164, 164, 164, 164, 59, 0],
            &[0, 0, 214, 243, 243, 246, 255, 255, 84, 0],
            &[0, 0, 0, 0, 0, 163, 255, 180, 2, 0],
            &[0, 0, 0, 0, 114, 255, 214, 14, 0, 0],
            &[0, 0, 0, 69, 252, 237, 35, 0, 0, 0],
            &[0, 0, 34, 237, 251, 66, 0, 0, 0, 0],
            &[0, 6, 211, 255, 221, 152, 152, 152, 74, 0],
            &[0, 23, 255, 255, 255, 255, 255, 255, 125, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ż' / 0x17b
        'Ż' => Some(&[
            &[0, 0, 0, 193, 255, 123, 0, 0, 0, 0],
            &[0, 0, 0, 183, 255, 109, 0, 0, 0, 0],
            &[0, 0, 0, 4, 24, 0, 0, 0, 0, 0],
            &[0, 135, 254, 254, 254, 254, 254, 99, 0, 0],
            &[0, 98, 185, 185, 186, 252, 255, 70, 0, 0],
            &[0, 0, 0, 0, 83, 255, 186, 0, 0, 0],
            &[0, 0, 0, 8, 223, 250, 41, 0, 0, 0],
            &[0, 0, 0, 124, 255, 141, 0, 0, 0, 0],
            &[0, 0, 27, 244, 232, 14, 0, 0, 0, 0],
            &[0, 0, 165, 255, 96, 0, 0, 0, 0, 0],
            &[0, 58, 254, 200, 1, 0, 0, 0, 0, 0],
            &[0, 195, 255, 218, 191, 191, 191, 80, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 45, 1, 0, 0, 0, 0],
        ]),
        // letter: 'ż' / 0x17c
        'ż' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 209, 62, 0, 0, 0],
            &[0, 0, 0, 0, 226, 255, 125, 0, 0, 0],
            &[0, 0, 0, 0, 59, 112, 17, 0, 0, 0],
            &[0, 0, 145, 164, 164, 164, 164, 164, 59, 0],
            &[0, 0, 214, 243, 243, 246, 255, 255, 84, 0],
            &[0, 0, 0, 0, 0, 163, 255, 180, 2, 0],
            &[0, 0, 0, 0, 114, 255, 214, 14, 0, 0],
            &[0, 0, 0, 69, 252, 237, 35, 0, 0, 0],
            &[0, 0, 34, 237, 251, 66, 0, 0, 0, 0],
            &[0, 6, 211, 255, 221, 152, 152, 152, 74, 0],
            &[0, 23, 255, 255, 255, 255, 255, 255, 125, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'Ž' / 0x17d
        'Ž' => Some(&[
            &[0, 10, 208, 178, 47, 221, 153, 0, 0, 0],
            &[0, 0, 36, 244, 253, 208, 7, 0, 0, 0],
            &[0, 0, 0, 29, 41, 18, 0, 0, 0, 0],
            &[0, 135, 254, 254, 254, 254, 254, 99, 0, 0],
            &[0, 98, 185, 185, 186, 252, 255, 70, 0, 0],
            &[0, 0, 0, 0, 83, 255, 186, 0, 0, 0],
            &[0, 0, 0, 8, 223, 250, 41, 0, 0, 0],
            &[0, 0, 0, 124, 255, 141, 0, 0, 0, 0],
            &[0, 0, 27, 244, 232, 14, 0, 0, 0, 0],
            &[0, 0, 165, 255, 96, 0, 0, 0, 0, 0],
            &[0, 58, 254, 200, 1, 0, 0, 0, 0, 0],
            &[0, 195, 255, 218, 191, 191, 191, 80, 0, 0],
            &[0, 225, 255, 255, 255, 255, 255, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 19, 1, 0, 7, 18, 0, 0, 0],
        ]),
        // letter: 'ž' / 0x17e
        'ž' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 178, 81, 7, 146, 130, 0, 0],
            &[0, 0, 0, 103, 253, 208, 236, 29, 0, 0],
            &[0, 0, 0, 0, 115, 132, 66, 0, 0, 0],
            &[0, 0, 145, 164, 164, 164, 164, 164, 59, 0],
            &[0, 0, 214, 243, 243, 246, 255, 255, 84, 0],
            &[0, 0, 0, 0, 0, 163, 255, 180, 2, 0],
            &[0, 0, 0, 0, 114, 255, 214, 14, 0, 0],
            &[0, 0, 0, 69, 252, 237, 35, 0, 0, 0],
            &[0, 0, 34, 237, 251, 66, 0, 0, 0, 0],
            &[0, 6, 211, 255, 221, 152, 152, 152, 74, 0],
            &[0, 23, 255, 255, 255, 255, 255, 255, 125, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        // letter: 'ſ' / 0x17f
        'ſ' => Some(&[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 43, 152, 176, 152, 28, 0, 0],
            &[0, 0, 8, 234, 255, 254, 250, 14, 0, 0],
            &[0, 0, 46, 255, 233, 19, 15, 0, 0, 0],
            &[0, 0, 54, 255, 201, 0, 0, 0, 0, 0],
            &[0, 0, 54, 255, 201, 0, 0, 0, 0, 0],
            &[0, 0, 54, 255, 201, 0, 0, 0, 0, 0],
            &[0, 0, 54, 255, 201, 0, 0, 0, 0, 0],
            &[0, 0, 54, 255, 201, 0, 0, 0, 0, 0],
            &[0, 0, 54, 255, 201, 0, 0, 0, 0, 0],
            &[0, 0, 54, 255, 201, 0, 0, 0, 0, 0],
            &[0, 0, 54, 255, 201, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ]),
        _ => None,
    }
}
