//! Module for letters with the font weight regular and size 32.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 32;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 17;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight regular and font size 27px
#[inline]
pub const fn get_char(c: char) -> &'static [&'static [u8]] {
    match c {
        // letter: ' ' / 0x20
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '!' / 0x21
        '!' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 71, 71, 41, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 85, 255, 255, 143, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 255, 255, 135, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 69, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 255, 255, 119, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 53, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 255, 255, 103, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 255, 255, 95, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 255, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 255, 255, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 253, 255, 55, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 245, 255, 47, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 130, 139, 22, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 50, 67, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 255, 255, 143, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 137, 255, 255, 188, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 249, 254, 118, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 19, 32, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '"' / 0x22
        '"' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 71, 62, 0, 0, 47, 71, 61, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 158, 255, 211, 0, 0, 159, 255, 209, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 138, 255, 191, 0, 0, 140, 255, 189, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 118, 255, 171, 0, 0, 120, 255, 170, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 99, 255, 151, 0, 0, 100, 255, 150, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 255, 131, 0, 0, 80, 255, 130, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 255, 111, 0, 0, 60, 255, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 176, 65, 0, 0, 30, 176, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '#' / 0x23
        '#' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 46, 71, 20, 0, 0, 25, 71, 35, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 198, 255, 43, 0, 0, 123, 255, 97, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 244, 247, 3, 0, 0, 171, 255, 49, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 255, 202, 0, 0, 0, 220, 250, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 154, 0, 0, 13, 254, 208, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 135, 255, 106, 0, 0, 60, 255, 160, 0, 0, 0, 0],
            &[
                0, 65, 141, 141, 141, 217, 255, 173, 141, 141, 184, 255, 197, 141, 141, 93, 0,
            ],
            &[
                0, 117, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 168, 0,
            ],
            &[
                0, 26, 58, 58, 79, 255, 213, 58, 58, 58, 235, 254, 64, 58, 58, 38, 0,
            ],
            &[0, 0, 0, 0, 73, 255, 156, 0, 0, 17, 255, 217, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 124, 255, 106, 0, 0, 66, 255, 167, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 175, 255, 57, 0, 0, 116, 255, 118, 0, 0, 0, 0, 0],
            &[
                53, 115, 115, 115, 233, 255, 126, 115, 115, 200, 255, 159, 115, 115, 85, 0, 0,
            ],
            &[
                117, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 189, 0, 0,
            ],
            &[
                42, 91, 91, 141, 255, 191, 91, 91, 103, 255, 230, 91, 91, 91, 67, 0, 0,
            ],
            &[0, 0, 0, 116, 255, 116, 0, 0, 59, 255, 174, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 162, 255, 69, 0, 0, 109, 255, 124, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 209, 255, 21, 0, 0, 160, 255, 74, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 250, 229, 0, 0, 0, 210, 255, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 46, 255, 181, 0, 0, 8, 251, 230, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '$' / 0x24
        '$' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 113, 104, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 212, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 108, 243, 235, 128, 97, 39, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 43, 205, 255, 255, 255, 255, 255, 255, 255, 210, 33, 0, 0, 0,
            ],
            &[
                0, 0, 27, 233, 255, 248, 186, 244, 234, 158, 204, 253, 233, 4, 0, 0, 0,
            ],
            &[
                0, 0, 139, 255, 239, 47, 0, 230, 212, 0, 0, 24, 61, 0, 0, 0, 0,
            ],
            &[0, 0, 196, 255, 158, 0, 0, 230, 212, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 145, 0, 0, 230, 212, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 160, 255, 224, 23, 0, 230, 212, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 50, 249, 255, 228, 127, 237, 212, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 79, 239, 255, 255, 255, 239, 112, 27, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 20, 132, 230, 255, 255, 255, 251, 168, 29, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 3, 233, 244, 237, 255, 255, 237, 41, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 230, 212, 5, 88, 242, 255, 180, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 212, 0, 0, 115, 255, 244, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 212, 0, 0, 87, 255, 249, 2, 0, 0],
            &[0, 0, 3, 0, 0, 0, 0, 230, 212, 0, 0, 168, 255, 204, 0, 0, 0],
            &[
                0, 0, 142, 187, 101, 43, 11, 230, 218, 75, 173, 255, 255, 87, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 133, 0, 0, 0, 0,
            ],
            &[
                0, 0, 40, 131, 203, 236, 254, 255, 254, 218, 155, 52, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 6, 232, 212, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 212, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 130, 120, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '%' / 0x25
        '%' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 56, 107, 95, 30, 0, 0, 0, 0, 0, 15, 71, 61, 0, 0, 0],
            &[
                3, 163, 255, 255, 255, 245, 92, 0, 0, 0, 0, 136, 255, 139, 0, 0, 0,
            ],
            &[
                101, 255, 193, 46, 70, 234, 247, 31, 0, 0, 20, 243, 245, 22, 0, 0, 0,
            ],
            &[
                201, 255, 53, 0, 0, 129, 255, 127, 0, 0, 136, 255, 139, 0, 0, 0, 0,
            ],
            &[
                236, 255, 12, 0, 0, 89, 255, 164, 0, 20, 243, 245, 22, 0, 0, 0, 0,
            ],
            &[
                240, 255, 9, 0, 0, 86, 255, 169, 0, 135, 255, 140, 0, 0, 0, 0, 0,
            ],
            &[
                204, 255, 43, 0, 0, 120, 255, 135, 19, 243, 245, 23, 0, 0, 0, 0, 0,
            ],
            &[
                114, 255, 166, 6, 25, 220, 254, 50, 134, 255, 141, 0, 0, 0, 0, 0, 0,
            ],
            &[
                6, 189, 255, 248, 253, 255, 134, 19, 243, 246, 23, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 2, 93, 150, 136, 66, 0, 134, 255, 142, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 18, 242, 246, 24, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 133, 255, 142, 0, 106, 192, 219, 174, 48, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 18, 242, 246, 24, 145, 255, 229, 189, 247, 248, 58, 0,
            ],
            &[
                0, 0, 0, 0, 0, 132, 255, 143, 36, 252, 227, 15, 0, 73, 255, 201, 0,
            ],
            &[
                0, 0, 0, 0, 18, 242, 246, 24, 93, 255, 150, 0, 0, 0, 238, 254, 10,
            ],
            &[
                0, 0, 0, 0, 131, 255, 144, 0, 124, 255, 127, 0, 0, 0, 217, 255, 41,
            ],
            &[
                0, 0, 0, 17, 241, 247, 25, 0, 103, 255, 140, 0, 0, 0, 229, 255, 21,
            ],
            &[
                0, 0, 0, 131, 255, 144, 0, 0, 58, 255, 199, 0, 0, 32, 254, 234, 0,
            ],
            &[
                0, 0, 17, 241, 247, 25, 0, 0, 0, 188, 255, 156, 98, 204, 255, 113, 0,
            ],
            &[
                0, 0, 130, 255, 145, 0, 0, 0, 0, 21, 181, 255, 255, 251, 131, 3, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 48, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '&' / 0x26
        '&' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 96, 134, 114, 53, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 54, 229, 255, 255, 255, 254, 143, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 4, 216, 255, 196, 88, 126, 249, 255, 80, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 45, 255, 253, 22, 0, 0, 150, 255, 160, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 76, 255, 236, 0, 0, 0, 116, 255, 188, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 51, 255, 255, 14, 0, 0, 166, 255, 156, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 3, 233, 255, 94, 0, 61, 251, 255, 65, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 119, 255, 210, 48, 238, 255, 165, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 12, 229, 255, 255, 255, 187, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 203, 255, 255, 229, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 6, 175, 255, 255, 255, 238, 34, 0, 0, 0, 57, 108, 78, 0, 0,
            ],
            &[
                0, 0, 160, 255, 251, 88, 196, 255, 210, 11, 0, 0, 182, 255, 144, 0, 0,
            ],
            &[
                0, 70, 255, 255, 106, 0, 26, 232, 255, 170, 0, 8, 243, 255, 77, 0, 0,
            ],
            &[
                0, 173, 255, 188, 0, 0, 0, 64, 252, 255, 120, 103, 255, 241, 8, 0, 0,
            ],
            &[
                0, 222, 255, 114, 0, 0, 0, 0, 118, 255, 255, 255, 255, 144, 0, 0, 0,
            ],
            &[
                0, 233, 255, 112, 0, 0, 0, 0, 0, 175, 255, 255, 255, 26, 0, 0, 0,
            ],
            &[
                0, 203, 255, 180, 0, 0, 0, 0, 0, 93, 255, 255, 255, 12, 0, 0, 0,
            ],
            &[
                0, 122, 255, 255, 119, 2, 0, 13, 125, 250, 255, 255, 255, 173, 1, 0, 0,
            ],
            &[
                0, 10, 216, 255, 255, 243, 216, 247, 255, 255, 161, 98, 255, 255, 123, 0, 0,
            ],
            &[
                0, 0, 21, 170, 251, 255, 255, 255, 203, 80, 0, 0, 150, 255, 253, 75, 0,
            ],
            &[0, 0, 0, 0, 18, 53, 58, 25, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ''' / 0x27
        '\'' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 71, 71, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 34, 255, 255, 78, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 14, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 250, 255, 39, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 231, 255, 19, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 211, 253, 2, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 191, 235, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 120, 150, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '(' / 0x28
        '(' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 52, 71, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 60, 251, 243, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 8, 221, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 123, 255, 226, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 236, 255, 114, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 94, 255, 250, 19, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 181, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 244, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 255, 255, 55, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 94, 255, 255, 10, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 124, 255, 234, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 143, 255, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 151, 255, 207, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 145, 255, 214, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 129, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 100, 255, 254, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 255, 255, 52, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 249, 255, 108, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 190, 255, 180, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 104, 255, 249, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 16, 242, 255, 112, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 132, 255, 225, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 11, 225, 255, 114, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 62, 251, 242, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 50, 68, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ')' / 0x29
        ')' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 71, 66, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 212, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 255, 249, 40, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 178, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 59, 255, 255, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 215, 255, 157, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 128, 255, 239, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 56, 255, 255, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 6, 249, 255, 109, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 210, 255, 152, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 180, 255, 181, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 163, 255, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 156, 255, 202, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 164, 255, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 183, 255, 175, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 214, 255, 145, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 8, 251, 255, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 255, 45, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 132, 255, 230, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 218, 255, 144, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 64, 255, 254, 43, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 183, 255, 173, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 255, 246, 33, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 217, 255, 104, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 68, 63, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '*' / 0x2a
        '*' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 35, 36, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 237, 255, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 217, 255, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 197, 249, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 31, 89, 25, 0, 0, 177, 231, 0, 0, 25, 95, 74, 0, 0, 0],
            &[
                0, 0, 98, 255, 255, 215, 151, 200, 232, 140, 210, 255, 255, 167, 0, 0, 0,
            ],
            &[
                0, 0, 72, 163, 196, 230, 254, 255, 255, 255, 251, 223, 191, 130, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 142, 255, 250, 212, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 255, 188, 136, 255, 154, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 42, 243, 255, 55, 18, 240, 255, 102, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 202, 255, 178, 0, 0, 124, 255, 248, 31, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 25, 171, 46, 0, 0, 12, 191, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '+' / 0x2b
        '+' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 171, 203, 20, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 216, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 216, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 216, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 216, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 102, 121, 121, 121, 121, 235, 255, 135, 121, 121, 121, 121, 4, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 94, 112, 112, 112, 112, 233, 255, 127, 112, 112, 112, 112, 4, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 216, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 216, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 216, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 216, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 158, 187, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ',' / 0x2c
        ',' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 109, 123, 123, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 251, 255, 253, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 255, 255, 196, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 94, 255, 255, 107, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 146, 255, 249, 20, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 199, 255, 169, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 249, 255, 64, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 42, 199, 172, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '-' / 0x2d
        '-' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 89, 94, 94, 94, 94, 94, 94, 94, 14, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 243, 255, 255, 255, 255, 255, 255, 255, 38, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 192, 202, 202, 202, 202, 202, 202, 202, 30, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '.' / 0x2e
        '.' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 50, 67, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 255, 255, 143, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 137, 255, 255, 188, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 249, 254, 118, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 19, 32, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '/' / 0x2f
        '/' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 29, 71, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 169, 255, 169, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 18, 248, 255, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 109, 255, 226, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 207, 255, 130, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 50, 255, 254, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 148, 255, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 8, 238, 255, 91, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 88, 255, 240, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 186, 255, 151, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 253, 255, 53, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 127, 255, 210, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 223, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 67, 255, 249, 20, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 172, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 247, 255, 73, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 255, 228, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 204, 255, 133, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 255, 254, 36, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 145, 255, 192, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '0' / 0x30
        '0' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 94, 130, 132, 101, 29, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 117, 248, 255, 255, 255, 255, 252, 144, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 106, 255, 255, 232, 172, 167, 222, 255, 255, 144, 0, 0, 0, 0,
            ],
            &[
                0, 0, 26, 250, 255, 171, 12, 0, 0, 4, 205, 255, 255, 62, 0, 0, 0,
            ],
            &[
                0, 0, 131, 255, 227, 11, 0, 0, 0, 5, 216, 255, 255, 176, 0, 0, 0,
            ],
            &[
                0, 0, 212, 255, 131, 0, 0, 0, 0, 116, 255, 255, 255, 250, 8, 0, 0,
            ],
            &[
                0, 11, 252, 255, 62, 0, 0, 0, 25, 242, 255, 107, 248, 255, 55, 0, 0,
            ],
            &[
                0, 53, 255, 255, 18, 0, 0, 0, 163, 255, 205, 2, 206, 255, 102, 0, 0,
            ],
            &[
                0, 71, 255, 248, 0, 0, 0, 59, 254, 254, 57, 0, 178, 255, 121, 0, 0,
            ],
            &[
                0, 85, 255, 237, 0, 0, 2, 206, 255, 159, 0, 0, 165, 255, 136, 0, 0,
            ],
            &[
                0, 93, 255, 229, 0, 0, 105, 255, 240, 23, 0, 0, 158, 255, 145, 0, 0,
            ],
            &[
                0, 79, 255, 237, 0, 19, 237, 255, 111, 0, 0, 0, 165, 255, 136, 0, 0,
            ],
            &[
                0, 64, 255, 251, 2, 151, 255, 211, 4, 0, 0, 0, 182, 255, 123, 0, 0,
            ],
            &[
                0, 38, 255, 255, 79, 253, 255, 64, 0, 0, 0, 0, 213, 255, 91, 0, 0,
            ],
            &[
                0, 1, 240, 255, 255, 255, 168, 0, 0, 0, 0, 13, 252, 255, 51, 0, 0,
            ],
            &[
                0, 0, 185, 255, 255, 244, 27, 0, 0, 0, 0, 89, 255, 238, 3, 0, 0,
            ],
            &[
                0, 0, 86, 255, 255, 153, 0, 0, 0, 0, 7, 212, 255, 156, 0, 0, 0,
            ],
            &[
                0, 0, 4, 218, 255, 223, 55, 0, 0, 36, 186, 255, 249, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 41, 239, 255, 255, 238, 233, 255, 255, 254, 100, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 44, 189, 255, 255, 255, 255, 206, 72, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 21, 58, 57, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '1' / 0x31
        '1' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 20, 71, 68, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 133, 243, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 15, 129, 241, 255, 255, 255, 246, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 40, 239, 255, 255, 212, 143, 255, 246, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 198, 215, 86, 1, 76, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 3, 0, 0, 83, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 88, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 90, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 246, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 66, 174, 191, 208, 255, 255, 255, 218, 201, 184, 167, 6, 0, 0,
            ],
            &[
                0, 0, 0, 104, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '2' / 0x32
        '2' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 43, 105, 132, 127, 98, 26, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 73, 197, 255, 255, 255, 255, 255, 253, 152, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 254, 200, 160, 171, 229, 255, 255, 174, 0, 0, 0, 0,
            ],
            &[
                0, 0, 76, 253, 169, 35, 0, 0, 0, 8, 159, 255, 255, 61, 0, 0, 0,
            ],
            &[0, 0, 0, 44, 0, 0, 0, 0, 0, 0, 9, 243, 255, 139, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 204, 255, 167, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 207, 255, 159, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 247, 255, 118, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 108, 255, 254, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 24, 235, 255, 175, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 6, 190, 255, 241, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 171, 255, 254, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 255, 255, 114, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 149, 255, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 153, 255, 255, 121, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 157, 255, 255, 115, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 160, 255, 254, 109, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 2, 164, 255, 255, 145, 40, 40, 40, 40, 40, 40, 40, 40, 29, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 182, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 182, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '3' / 0x33
        '3' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 55, 109, 130, 127, 100, 29, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 6, 116, 229, 255, 255, 255, 255, 255, 254, 161, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 169, 255, 255, 244, 184, 157, 173, 230, 255, 255, 183, 0, 0, 0, 0,
            ],
            &[
                0, 0, 42, 224, 114, 9, 0, 0, 0, 8, 165, 255, 255, 64, 0, 0, 0,
            ],
            &[0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 16, 254, 255, 126, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 233, 255, 140, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 19, 253, 255, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 150, 255, 241, 22, 0, 0, 0],
            &[0, 0, 0, 0, 1, 11, 11, 25, 70, 178, 255, 247, 79, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 42, 255, 255, 255, 255, 234, 148, 36, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 42, 255, 255, 255, 255, 255, 223, 139, 21, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 4, 29, 30, 39, 65, 126, 231, 255, 228, 32, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 23, 223, 255, 181, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 252, 12, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 82, 255, 255, 37, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 118, 255, 255, 18, 0, 0],
            &[0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 12, 221, 255, 217, 0, 0, 0],
            &[
                0, 0, 219, 123, 51, 7, 0, 0, 7, 67, 203, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 255, 255, 255, 251, 231, 231, 254, 255, 255, 255, 146, 0, 0, 0, 0,
            ],
            &[
                0, 0, 108, 200, 251, 255, 255, 255, 255, 249, 184, 72, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 8, 32, 56, 59, 39, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '4' / 0x34
        '4' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 81, 98, 98, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 64, 255, 255, 255, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 213, 255, 255, 255, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 119, 255, 188, 255, 255, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 244, 250, 81, 255, 255, 10, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 173, 255, 144, 56, 255, 255, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 73, 255, 232, 15, 63, 255, 255, 10, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 8, 220, 255, 94, 0, 68, 255, 255, 10, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 128, 255, 195, 0, 0, 70, 255, 255, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 36, 247, 251, 46, 0, 0, 71, 255, 255, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 183, 255, 144, 0, 0, 0, 71, 255, 255, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 83, 255, 231, 14, 0, 0, 0, 71, 255, 255, 10, 0, 0, 0, 0,
            ],
            &[
                0, 11, 226, 255, 92, 0, 0, 0, 0, 71, 255, 255, 10, 0, 0, 0, 0,
            ],
            &[
                0, 130, 255, 236, 102, 102, 102, 102, 102, 145, 255, 255, 108, 102, 86, 0, 0,
            ],
            &[
                0, 166, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 217, 0, 0,
            ],
            &[
                0, 113, 174, 174, 174, 174, 174, 174, 174, 196, 255, 255, 177, 174, 147, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 71, 255, 255, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 71, 255, 255, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 71, 255, 255, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 71, 255, 255, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '5' / 0x35
        '5' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 51, 71, 71, 71, 71, 71, 71, 71, 71, 64, 0, 0, 0, 0],
            &[
                0, 0, 0, 196, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 217, 255, 236, 225, 225, 225, 225, 225, 225, 204, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 237, 255, 80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 254, 255, 57, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 255, 255, 33, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 255, 255, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 64, 255, 242, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 84, 255, 242, 166, 196, 207, 191, 155, 76, 2, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 96, 255, 255, 255, 255, 255, 255, 255, 255, 199, 32, 0, 0, 0, 0,
            ],
            &[
                0, 0, 1, 93, 105, 65, 45, 50, 89, 182, 255, 255, 216, 15, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 116, 255, 255, 121, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 208, 255, 209, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 242, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 245, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 180, 255, 215, 0, 0, 0],
            &[0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 59, 250, 255, 136, 0, 0, 0],
            &[
                0, 0, 182, 111, 34, 0, 0, 0, 21, 118, 236, 255, 236, 28, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 243, 226, 244, 255, 255, 255, 237, 63, 0, 0, 0, 0,
            ],
            &[
                0, 0, 86, 193, 250, 255, 255, 255, 255, 233, 150, 24, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 7, 43, 63, 55, 32, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '6' / 0x36
        '6' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 38, 95, 126, 137, 129, 97, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 54, 202, 255, 255, 255, 255, 255, 252, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 107, 251, 255, 254, 199, 154, 134, 143, 181, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 75, 253, 255, 185, 34, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 229, 255, 178, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 100, 255, 244, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 183, 255, 162, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 245, 255, 90, 0, 0, 21, 46, 36, 4, 0, 0, 0, 0, 0, 0],
            &[
                0, 34, 255, 255, 45, 60, 199, 255, 255, 255, 243, 159, 27, 0, 0, 0, 0,
            ],
            &[
                0, 67, 255, 255, 93, 253, 250, 201, 185, 220, 254, 255, 233, 39, 0, 0, 0,
            ],
            &[
                0, 83, 255, 255, 246, 165, 22, 0, 0, 0, 62, 226, 255, 206, 0, 0, 0,
            ],
            &[
                0, 93, 255, 255, 173, 0, 0, 0, 0, 0, 0, 74, 255, 255, 60, 0, 0,
            ],
            &[
                0, 85, 255, 255, 47, 0, 0, 0, 0, 0, 0, 0, 236, 255, 119, 0, 0,
            ],
            &[
                0, 68, 255, 255, 15, 0, 0, 0, 0, 0, 0, 0, 201, 255, 141, 0, 0,
            ],
            &[
                0, 26, 255, 255, 45, 0, 0, 0, 0, 0, 0, 0, 201, 255, 129, 0, 0,
            ],
            &[0, 0, 223, 255, 125, 0, 0, 0, 0, 0, 0, 2, 240, 255, 99, 0, 0],
            &[
                0, 0, 130, 255, 236, 29, 0, 0, 0, 0, 0, 88, 255, 252, 24, 0, 0,
            ],
            &[
                0, 0, 18, 233, 255, 215, 56, 0, 0, 1, 100, 241, 255, 157, 0, 0, 0,
            ],
            &[
                0, 0, 0, 64, 244, 255, 255, 230, 205, 242, 255, 255, 203, 12, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 41, 177, 250, 255, 255, 255, 231, 131, 7, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 13, 50, 61, 35, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '7' / 0x37
        '7' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 27, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 41, 0, 0,
            ],
            &[
                0, 97, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[
                0, 85, 225, 225, 225, 225, 225, 225, 225, 225, 225, 225, 252, 255, 141, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 38, 254, 255, 55, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 148, 255, 203, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 15, 243, 255, 95, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 114, 255, 235, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 222, 255, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 80, 255, 252, 31, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 191, 255, 176, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 46, 255, 255, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 157, 255, 215, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 20, 247, 255, 108, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 123, 255, 242, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 229, 255, 148, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 89, 255, 255, 41, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 200, 255, 188, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 55, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 225, 3, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 250, 255, 121, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '8' / 0x38
        '8' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 92, 126, 131, 104, 38, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 7, 148, 251, 255, 255, 255, 255, 255, 179, 25, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 177, 255, 255, 197, 140, 136, 183, 254, 255, 215, 9, 0, 0, 0,
            ],
            &[
                0, 0, 63, 255, 255, 111, 0, 0, 0, 0, 73, 252, 255, 108, 0, 0, 0,
            ],
            &[
                0, 0, 119, 255, 238, 0, 0, 0, 0, 0, 0, 195, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 126, 255, 218, 0, 0, 0, 0, 0, 0, 173, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 76, 255, 252, 36, 0, 0, 0, 0, 11, 234, 255, 116, 0, 0, 0,
            ],
            &[
                0, 0, 4, 209, 255, 221, 35, 0, 0, 11, 192, 255, 228, 13, 0, 0, 0,
            ],
            &[
                0, 0, 0, 35, 226, 255, 243, 129, 99, 226, 255, 224, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 18, 181, 255, 255, 255, 255, 152, 14, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 96, 232, 255, 241, 255, 255, 227, 82, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 1, 151, 255, 250, 140, 14, 57, 194, 255, 255, 157, 3, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 252, 73, 0, 0, 0, 0, 98, 250, 255, 131, 0, 0, 0,
            ],
            &[
                0, 1, 224, 255, 151, 0, 0, 0, 0, 0, 0, 113, 255, 246, 17, 0, 0,
            ],
            &[
                0, 23, 255, 255, 73, 0, 0, 0, 0, 0, 0, 20, 255, 255, 67, 0, 0,
            ],
            &[
                0, 30, 255, 255, 71, 0, 0, 0, 0, 0, 0, 27, 255, 255, 72, 0, 0,
            ],
            &[
                0, 3, 244, 255, 141, 0, 0, 0, 0, 0, 0, 109, 255, 254, 30, 0, 0,
            ],
            &[
                0, 0, 147, 255, 251, 100, 2, 0, 0, 1, 93, 245, 255, 173, 0, 0, 0,
            ],
            &[
                0, 0, 12, 204, 255, 255, 233, 197, 195, 232, 255, 255, 210, 18, 0, 0, 0,
            ],
            &[
                0, 0, 0, 8, 127, 226, 255, 255, 255, 255, 224, 127, 8, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 34, 58, 56, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '9' / 0x39
        '9' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 99, 128, 127, 93, 22, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 18, 172, 254, 255, 255, 255, 255, 249, 137, 3, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 218, 255, 254, 186, 139, 138, 207, 255, 255, 165, 0, 0, 0, 0,
            ],
            &[
                0, 0, 153, 255, 243, 65, 0, 0, 0, 0, 115, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 12, 244, 255, 118, 0, 0, 0, 0, 0, 0, 167, 255, 201, 0, 0, 0,
            ],
            &[
                0, 61, 255, 255, 33, 0, 0, 0, 0, 0, 0, 42, 255, 255, 31, 0, 0,
            ],
            &[0, 87, 255, 250, 3, 0, 0, 0, 0, 0, 0, 0, 237, 255, 82, 0, 0],
            &[0, 84, 255, 254, 8, 0, 0, 0, 0, 0, 0, 0, 224, 255, 120, 0, 0],
            &[
                0, 52, 255, 255, 45, 0, 0, 0, 0, 0, 0, 31, 253, 255, 135, 0, 0,
            ],
            &[
                0, 3, 232, 255, 163, 0, 0, 0, 0, 0, 3, 179, 255, 255, 144, 0, 0,
            ],
            &[
                0, 0, 108, 255, 255, 159, 42, 5, 10, 72, 198, 236, 237, 255, 128, 0, 0,
            ],
            &[
                0, 0, 0, 144, 255, 255, 255, 253, 255, 255, 245, 66, 230, 255, 110, 0, 0,
            ],
            &[
                0, 0, 0, 0, 62, 163, 215, 232, 214, 149, 26, 10, 253, 255, 86, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 60, 255, 255, 37, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 223, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 242, 255, 131, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 189, 255, 239, 19, 0, 0, 0],
            &[0, 0, 0, 6, 0, 0, 0, 5, 75, 211, 255, 254, 84, 0, 0, 0, 0],
            &[
                0, 0, 0, 132, 227, 205, 217, 244, 255, 255, 248, 92, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 132, 255, 255, 255, 255, 234, 155, 32, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 8, 45, 63, 55, 34, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ':' / 0x3a
        ':' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 14, 135, 155, 39, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 115, 255, 255, 174, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 124, 255, 255, 182, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 179, 201, 62, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 50, 66, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 255, 255, 138, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 137, 255, 255, 182, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 249, 254, 114, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 19, 31, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ';' / 0x3b
        ';' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 59, 163, 120, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 219, 255, 255, 76, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 228, 255, 255, 85, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 90, 210, 164, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 109, 123, 123, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 251, 255, 253, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 255, 255, 196, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 94, 255, 255, 107, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 146, 255, 249, 20, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 199, 255, 169, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 249, 255, 64, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 42, 199, 172, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '<' / 0x3c
        '<' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 41, 180, 196, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 150, 251, 255, 174, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 8, 119, 241, 255, 229, 99, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 89, 224, 255, 242, 124, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 200, 255, 251, 149, 22, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 171, 255, 255, 174, 38, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 145, 255, 255, 116, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 61, 205, 255, 255, 189, 65, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 76, 202, 255, 255, 198, 75, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 198, 255, 255, 207, 84, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 69, 195, 255, 255, 215, 83, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 65, 191, 255, 197, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 133, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '=' / 0x3d
        '=' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 4, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 38, 11, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[0, 0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 1, 0, 0],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 3, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 11, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '>' / 0x3e
        '>' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 37, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 145, 206, 66, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 123, 255, 255, 178, 40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 71, 209, 255, 251, 148, 20, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 96, 227, 255, 240, 117, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 121, 241, 255, 223, 87, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 20, 146, 250, 255, 199, 56, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 71, 248, 255, 197, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 42, 164, 252, 255, 225, 92, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 51, 173, 254, 255, 223, 101, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 60, 183, 255, 255, 220, 98, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 192, 255, 255, 217, 94, 3, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 145, 255, 214, 90, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 106, 87, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '?' / 0x3f
        '?' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 76, 118, 133, 127, 100, 33, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 51, 171, 248, 255, 255, 255, 255, 255, 255, 175, 24, 0, 0, 0, 0,
            ],
            &[
                0, 0, 60, 255, 255, 233, 177, 151, 157, 203, 255, 255, 212, 11, 0, 0, 0,
            ],
            &[0, 0, 0, 140, 73, 3, 0, 0, 0, 0, 86, 254, 255, 111, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 177, 255, 174, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 146, 255, 190, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 182, 255, 164, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 41, 248, 255, 94, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 33, 213, 255, 208, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 60, 235, 255, 209, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 85, 248, 255, 188, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 246, 255, 155, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 143, 255, 197, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 182, 255, 106, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 139, 45, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 79, 13, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 30, 249, 255, 213, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 255, 255, 254, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 232, 255, 186, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '@' / 0x40
        '@' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 39, 63, 38, 5, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 26, 153, 241, 255, 255, 255, 247, 145, 24, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 69, 236, 255, 229, 172, 138, 157, 230, 255, 227, 36, 0, 0, 0,
            ],
            &[
                0, 0, 49, 243, 252, 122, 8, 0, 0, 0, 5, 139, 255, 225, 12, 0, 0,
            ],
            &[
                0, 2, 209, 255, 105, 0, 0, 0, 0, 0, 0, 0, 149, 255, 130, 0, 0,
            ],
            &[0, 91, 255, 177, 0, 0, 0, 0, 3, 0, 0, 0, 13, 234, 244, 11, 0],
            &[
                0, 187, 255, 56, 0, 0, 106, 217, 253, 243, 212, 147, 3, 144, 255, 82, 0,
            ],
            &[
                16, 252, 221, 0, 0, 140, 255, 238, 180, 209, 255, 253, 0, 76, 255, 139, 0,
            ],
            &[
                63, 255, 162, 0, 53, 254, 238, 30, 0, 31, 255, 240, 0, 33, 255, 175, 0,
            ],
            &[
                108, 255, 112, 0, 152, 255, 135, 0, 0, 45, 255, 226, 0, 14, 255, 194, 0,
            ],
            &[
                130, 255, 93, 0, 214, 255, 79, 0, 0, 60, 255, 212, 0, 7, 255, 199, 0,
            ],
            &[
                144, 255, 76, 0, 239, 255, 49, 0, 0, 78, 255, 198, 0, 17, 255, 182, 0,
            ],
            &[
                147, 255, 78, 0, 236, 255, 41, 0, 0, 106, 255, 185, 0, 30, 255, 162, 0,
            ],
            &[
                135, 255, 97, 0, 208, 255, 66, 0, 0, 155, 255, 190, 0, 72, 255, 113, 0,
            ],
            &[
                107, 255, 123, 0, 146, 255, 153, 0, 16, 233, 243, 226, 1, 135, 255, 49, 0,
            ],
            &[
                61, 255, 184, 0, 40, 248, 255, 184, 220, 241, 87, 255, 202, 246, 205, 0, 0,
            ],
            &[
                8, 243, 244, 12, 0, 71, 218, 255, 218, 60, 0, 156, 254, 217, 40, 0, 0,
            ],
            &[0, 161, 255, 125, 0, 0, 0, 9, 0, 0, 0, 0, 8, 0, 0, 0, 0],
            &[0, 42, 251, 242, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 127, 255, 238, 77, 0, 0, 0, 0, 0, 0, 31, 36, 0, 0, 0],
            &[
                0, 0, 1, 146, 255, 255, 210, 147, 115, 113, 141, 198, 254, 100, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 78, 206, 255, 255, 255, 255, 255, 247, 168, 39, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 32, 80, 98, 92, 65, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'A' / 0x41
        'A' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 91, 91, 39, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 255, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 255, 255, 241, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 249, 223, 255, 76, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 113, 255, 196, 151, 255, 159, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 196, 255, 127, 79, 255, 237, 4, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 25, 253, 255, 48, 12, 247, 255, 69, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 106, 255, 224, 0, 0, 180, 255, 153, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 189, 255, 145, 0, 0, 101, 255, 233, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 19, 252, 255, 65, 0, 0, 23, 253, 255, 63, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 98, 255, 238, 4, 0, 0, 0, 198, 255, 146, 0, 0, 0, 0],
            &[
                0, 0, 0, 181, 255, 181, 18, 18, 18, 18, 137, 255, 228, 1, 0, 0, 0,
            ],
            &[
                0, 0, 15, 249, 255, 255, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 198, 22, 22, 22, 22, 22, 22, 153, 255, 222, 0, 0, 0,
            ],
            &[
                0, 10, 246, 255, 100, 0, 0, 0, 0, 0, 0, 56, 255, 255, 50, 0, 0,
            ],
            &[
                0, 83, 255, 253, 23, 0, 0, 0, 0, 0, 0, 1, 231, 255, 133, 0, 0,
            ],
            &[
                0, 166, 255, 197, 0, 0, 0, 0, 0, 0, 0, 0, 153, 255, 216, 0, 0,
            ],
            &[
                7, 242, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0, 74, 255, 255, 44, 0,
            ],
            &[
                76, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 7, 243, 255, 127, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'B' / 0x42
        'B' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 30, 71, 71, 71, 71, 70, 62, 39, 4, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 243, 167, 43, 0, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 249, 211, 211, 214, 232, 252, 255, 255, 246, 63, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 220, 0, 0, 0, 0, 11, 104, 251, 255, 209, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 220, 0, 0, 0, 0, 0, 0, 140, 255, 255, 21, 0, 0,
            ],
            &[
                0, 0, 111, 255, 220, 0, 0, 0, 0, 0, 0, 86, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 111, 255, 220, 0, 0, 0, 0, 0, 0, 101, 255, 254, 9, 0, 0,
            ],
            &[
                0, 0, 111, 255, 220, 0, 0, 0, 0, 0, 1, 189, 255, 190, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 224, 31, 31, 31, 46, 79, 184, 255, 233, 46, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 240, 131, 27, 0, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 251, 251, 251, 255, 255, 255, 229, 130, 8, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 220, 0, 0, 0, 5, 33, 111, 239, 255, 196, 5, 0, 0,
            ],
            &[
                0, 0, 111, 255, 220, 0, 0, 0, 0, 0, 0, 67, 255, 255, 95, 0, 0,
            ],
            &[
                0, 0, 111, 255, 220, 0, 0, 0, 0, 0, 0, 0, 225, 255, 159, 0, 0,
            ],
            &[
                0, 0, 111, 255, 220, 0, 0, 0, 0, 0, 0, 0, 206, 255, 175, 0, 0,
            ],
            &[
                0, 0, 111, 255, 220, 0, 0, 0, 0, 0, 0, 2, 240, 255, 151, 0, 0,
            ],
            &[
                0, 0, 111, 255, 220, 0, 0, 0, 0, 0, 0, 104, 255, 255, 88, 0, 0,
            ],
            &[
                0, 0, 111, 255, 223, 27, 27, 27, 39, 70, 150, 252, 255, 216, 6, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 221, 39, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 253, 233, 196, 115, 11, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'C' / 0x43
        'C' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 46, 105, 129, 132, 111, 65, 6, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 69, 212, 255, 255, 255, 255, 255, 255, 241, 120, 0, 0,
            ],
            &[
                0, 0, 0, 0, 114, 255, 255, 255, 206, 168, 172, 202, 247, 255, 108, 0, 0,
            ],
            &[
                0, 0, 0, 83, 255, 255, 202, 44, 0, 0, 0, 0, 15, 100, 10, 0, 0,
            ],
            &[0, 0, 11, 231, 255, 210, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 98, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 185, 255, 214, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 239, 255, 143, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 255, 255, 99, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 255, 255, 72, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 107, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 231, 255, 159, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 225, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 255, 255, 90, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 226, 255, 234, 42, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 72, 253, 255, 238, 125, 38, 7, 8, 34, 90, 167, 2, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 249, 255, 255, 255, 254, 255, 255, 255, 255, 3, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 43, 169, 247, 255, 255, 255, 255, 234, 162, 1, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 5, 44, 63, 54, 31, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'D' / 0x44
        'D' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 71, 71, 69, 57, 22, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 214, 113, 8, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 225, 211, 218, 245, 255, 255, 255, 214, 35, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 47, 154, 255, 255, 227, 23, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 89, 254, 255, 171, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 151, 255, 255, 41, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 28, 255, 255, 130, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 207, 255, 190, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 163, 255, 227, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 143, 255, 244, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 132, 255, 248, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 146, 255, 236, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 178, 255, 207, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 2, 232, 255, 160, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 71, 255, 255, 88, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 5, 208, 255, 233, 8, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 13, 178, 255, 255, 98, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 100, 27, 41, 69, 134, 235, 255, 255, 152, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 246, 118, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 249, 231, 189, 118, 22, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'E' / 0x45
        'E' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 30, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 64, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 225, 225, 225, 225, 225, 225, 225, 225, 204, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 38, 38, 38, 38, 38, 38, 38, 38, 15, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 40, 40, 40, 40, 40, 40, 40, 40, 37, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'F' / 0x46
        'F' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 30, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 64, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 225, 225, 225, 225, 225, 225, 225, 225, 204, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 255, 80, 80, 80, 80, 80, 80, 80, 80, 31, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 216, 216, 216, 216, 216, 216, 216, 216, 84, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'G' / 0x47
        'G' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 86, 124, 136, 125, 87, 19, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 24, 163, 251, 255, 255, 255, 255, 255, 248, 164, 3, 0, 0,
            ],
            &[
                0, 0, 0, 41, 230, 255, 255, 243, 198, 171, 194, 231, 255, 180, 0, 0, 0,
            ],
            &[
                0, 0, 13, 222, 255, 253, 128, 15, 0, 0, 0, 1, 74, 53, 0, 0, 0,
            ],
            &[0, 0, 134, 255, 255, 99, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 239, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 255, 255, 75, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 255, 251, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 166, 255, 216, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 189, 255, 197, 0, 0, 0, 0, 25, 45, 45, 45, 45, 45, 14, 0, 0,
            ],
            &[
                0, 198, 255, 183, 0, 0, 0, 0, 140, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 192, 255, 194, 0, 0, 0, 0, 137, 251, 251, 251, 255, 255, 79, 0, 0,
            ],
            &[0, 175, 255, 214, 0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0],
            &[
                0, 142, 255, 251, 10, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[0, 89, 255, 255, 72, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0],
            &[
                0, 17, 249, 255, 183, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 106, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 22, 232, 255, 253, 155, 40, 3, 0, 3, 55, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 0, 48, 228, 255, 255, 255, 250, 234, 252, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 0, 0, 18, 136, 230, 255, 255, 255, 255, 255, 210, 132, 25, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 32, 59, 64, 50, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'H' / 0x48
        'H' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 8, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 108, 38, 38, 38, 38, 38, 38, 64, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 83, 2, 2, 2, 2, 2, 2, 32, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'I' / 0x49
        'I' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 16, 0, 0, 0],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[
                0, 0, 3, 113, 136, 159, 191, 255, 255, 204, 163, 140, 117, 23, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 255, 255, 74, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 5, 184, 207, 230, 251, 255, 255, 253, 234, 211, 189, 40, 0, 0, 0,
            ],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'J' / 0x4a
        'J' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 62, 71, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 224, 255, 142, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 237, 255, 133, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 254, 255, 110, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 54, 0, 0, 0],
            &[
                0, 0, 0, 58, 141, 67, 29, 2, 20, 116, 249, 255, 214, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 76, 255, 255, 255, 254, 255, 255, 255, 245, 56, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 40, 185, 248, 255, 255, 255, 253, 187, 46, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 8, 36, 60, 50, 19, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'K' / 0x4b
        'K' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 19, 71, 71, 35, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 2, 190, 255, 236, 27, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 128, 255, 254, 75, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 66, 253, 255, 141, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 23, 231, 255, 201, 5, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 2, 188, 255, 240, 33, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 125, 255, 255, 85, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 64, 252, 255, 151, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 18, 228, 255, 209, 8, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 168, 255, 255, 83, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 159, 255, 255, 255, 206, 4, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 243, 110, 254, 255, 117, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 252, 74, 0, 156, 255, 245, 33, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 122, 0, 0, 18, 234, 255, 185, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 93, 255, 255, 92, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 188, 255, 235, 19, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 37, 248, 255, 161, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 125, 255, 255, 68, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 6, 215, 255, 220, 9, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 63, 254, 255, 136, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'L' / 0x4c
        'L' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 63, 71, 38, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 229, 255, 165, 61, 61, 61, 61, 61, 61, 61, 55, 0, 0, 0,
            ],
            &[
                0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'M' / 0x4d
        'M' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 71, 71, 62, 0, 0, 0, 0, 0, 0, 49, 71, 71, 60, 0, 0],
            &[
                0, 166, 255, 255, 252, 17, 0, 0, 0, 0, 0, 225, 255, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 255, 255, 86, 0, 0, 0, 0, 42, 255, 255, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 214, 255, 159, 0, 0, 0, 0, 114, 255, 215, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 162, 255, 230, 1, 0, 0, 0, 187, 255, 160, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 130, 233, 255, 48, 0, 0, 11, 248, 253, 104, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 144, 158, 255, 121, 0, 0, 77, 255, 199, 95, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 155, 82, 255, 193, 0, 0, 149, 255, 123, 104, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 161, 13, 249, 251, 15, 0, 222, 255, 48, 110, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 164, 0, 186, 255, 83, 39, 255, 227, 0, 113, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 165, 0, 110, 255, 156, 111, 255, 152, 0, 113, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 165, 0, 34, 255, 228, 185, 255, 76, 0, 113, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 165, 0, 0, 213, 255, 255, 246, 10, 0, 113, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 165, 0, 0, 137, 255, 255, 181, 0, 0, 113, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 165, 0, 0, 61, 255, 255, 105, 0, 0, 113, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 165, 0, 0, 3, 49, 49, 11, 0, 0, 113, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 165, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 165, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 165, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 217, 0, 0,
            ],
            &[
                0, 166, 255, 165, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 217, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'N' / 0x4e
        'N' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 71, 11, 0, 0, 0, 0, 0, 2, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 255, 113, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 222, 2, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 239, 221, 255, 81, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 252, 113, 255, 192, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 32, 241, 255, 49, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 37, 142, 255, 160, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 50, 33, 252, 248, 23, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 57, 0, 174, 255, 128, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 61, 255, 232, 7, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 205, 255, 96, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 93, 255, 207, 0, 8, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 6, 230, 255, 63, 5, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 124, 255, 175, 0, 254, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 20, 247, 253, 34, 244, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 155, 255, 143, 230, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 44, 255, 241, 225, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 186, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 74, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 1, 216, 255, 255, 79, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'O' / 0x4f
        'O' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 107, 143, 143, 115, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 2, 143, 252, 255, 255, 255, 255, 255, 173, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 228, 169, 153, 202, 255, 255, 203, 8, 0, 0, 0,
            ],
            &[
                0, 0, 78, 255, 255, 156, 9, 0, 0, 0, 86, 252, 255, 134, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 205, 4, 0, 0, 0, 0, 0, 143, 255, 243, 15, 0, 0,
            ],
            &[
                0, 36, 255, 255, 96, 0, 0, 0, 0, 0, 0, 36, 255, 255, 89, 0, 0,
            ],
            &[
                0, 94, 255, 255, 19, 0, 0, 0, 0, 0, 0, 0, 223, 255, 157, 0, 0,
            ],
            &[
                0, 147, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 176, 255, 198, 0, 0,
            ],
            &[
                0, 170, 255, 201, 0, 0, 0, 0, 0, 0, 0, 0, 147, 255, 229, 0, 0,
            ],
            &[
                0, 186, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 242, 0, 0,
            ],
            &[
                0, 197, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 248, 0, 0,
            ],
            &[
                0, 187, 255, 193, 0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 237, 0, 0,
            ],
            &[
                0, 171, 255, 207, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 221, 0, 0,
            ],
            &[
                0, 135, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 192, 255, 183, 0, 0,
            ],
            &[
                0, 90, 255, 255, 39, 0, 0, 0, 0, 0, 0, 3, 237, 255, 136, 0, 0,
            ],
            &[
                0, 19, 252, 255, 127, 0, 0, 0, 0, 0, 0, 70, 255, 255, 62, 0, 0,
            ],
            &[
                0, 0, 174, 255, 236, 28, 0, 0, 0, 0, 4, 200, 255, 216, 2, 0, 0,
            ],
            &[
                0, 0, 44, 248, 255, 214, 64, 3, 0, 38, 181, 255, 255, 81, 0, 0, 0,
            ],
            &[
                0, 0, 0, 91, 251, 255, 255, 249, 242, 255, 255, 255, 131, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 58, 193, 254, 255, 255, 255, 208, 83, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 22, 53, 57, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'P' / 0x50
        'P' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 30, 71, 71, 71, 71, 71, 64, 48, 9, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 252, 187, 63, 0, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 211, 211, 212, 229, 253, 255, 255, 253, 97, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 254, 0, 0, 0, 0, 5, 91, 230, 255, 247, 27, 0, 0,
            ],
            &[
                0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 65, 255, 255, 117, 0, 0,
            ],
            &[
                0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 236, 255, 165, 0, 0,
            ],
            &[
                0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 205, 255, 176, 0, 0,
            ],
            &[
                0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 225, 255, 151, 0, 0,
            ],
            &[
                0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 50, 255, 255, 90, 0, 0,
            ],
            &[
                0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 43, 217, 255, 231, 12, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 135, 135, 138, 154, 199, 252, 255, 250, 74, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 211, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 147, 147, 146, 128, 104, 50, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Q' / 0x51
        'Q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 107, 143, 143, 115, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 2, 143, 252, 255, 255, 255, 255, 255, 173, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 228, 169, 153, 202, 255, 255, 203, 8, 0, 0, 0,
            ],
            &[
                0, 0, 78, 255, 255, 156, 9, 0, 0, 0, 86, 252, 255, 134, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 205, 4, 0, 0, 0, 0, 0, 143, 255, 243, 15, 0, 0,
            ],
            &[
                0, 36, 255, 255, 96, 0, 0, 0, 0, 0, 0, 36, 255, 255, 89, 0, 0,
            ],
            &[
                0, 94, 255, 255, 19, 0, 0, 0, 0, 0, 0, 0, 223, 255, 157, 0, 0,
            ],
            &[
                0, 147, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 176, 255, 198, 0, 0,
            ],
            &[
                0, 170, 255, 201, 0, 0, 0, 0, 0, 0, 0, 0, 147, 255, 229, 0, 0,
            ],
            &[
                0, 186, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 242, 0, 0,
            ],
            &[
                0, 197, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 248, 0, 0,
            ],
            &[
                0, 187, 255, 193, 0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 237, 0, 0,
            ],
            &[
                0, 171, 255, 207, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 221, 0, 0,
            ],
            &[
                0, 135, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 192, 255, 183, 0, 0,
            ],
            &[
                0, 90, 255, 255, 39, 0, 0, 0, 0, 0, 0, 3, 237, 255, 136, 0, 0,
            ],
            &[
                0, 19, 252, 255, 127, 0, 0, 0, 0, 0, 0, 70, 255, 255, 62, 0, 0,
            ],
            &[
                0, 0, 174, 255, 236, 28, 0, 0, 0, 0, 4, 200, 255, 216, 2, 0, 0,
            ],
            &[
                0, 0, 44, 248, 255, 214, 64, 3, 0, 38, 229, 255, 255, 81, 0, 0, 0,
            ],
            &[
                0, 0, 0, 91, 251, 255, 255, 249, 242, 255, 255, 255, 131, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 58, 193, 254, 255, 255, 255, 255, 255, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 22, 53, 57, 100, 255, 255, 78, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 199, 255, 211, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 71, 255, 255, 92, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 198, 255, 221, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 68, 220, 220, 83, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'R' / 0x52
        'R' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 71, 71, 71, 71, 63, 44, 12, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 111, 255, 255, 255, 255, 255, 255, 250, 194, 74, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 255, 211, 211, 226, 250, 255, 255, 255, 135, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 1, 69, 208, 255, 255, 63, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 32, 246, 255, 161, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 202, 255, 199, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 176, 255, 208, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 208, 255, 180, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 60, 255, 255, 103, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 7, 34, 112, 239, 255, 208, 5, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 255, 253, 253, 255, 255, 255, 255, 190, 23, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 255, 255, 255, 255, 255, 255, 126, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 29, 29, 29, 192, 255, 214, 5, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 46, 252, 255, 117, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 147, 255, 243, 27, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 17, 235, 255, 168, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 102, 255, 255, 67, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 2, 204, 255, 214, 5, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 57, 254, 255, 118, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 161, 255, 243, 28, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'S' / 0x53
        'S' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 56, 110, 131, 131, 110, 64, 4, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 72, 221, 255, 255, 255, 255, 255, 255, 236, 133, 0, 0, 0,
            ],
            &[
                0, 0, 0, 73, 251, 255, 252, 210, 170, 176, 205, 246, 255, 135, 0, 0, 0,
            ],
            &[0, 0, 1, 215, 255, 229, 44, 0, 0, 0, 0, 12, 89, 32, 0, 0, 0],
            &[0, 0, 36, 255, 255, 104, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 255, 61, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 230, 255, 214, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 98, 255, 255, 237, 123, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 106, 248, 255, 255, 242, 138, 23, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 30, 154, 250, 255, 255, 246, 145, 14, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 32, 147, 246, 255, 255, 222, 29, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 148, 255, 255, 190, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 140, 255, 255, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 255, 62, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 255, 255, 58, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 156, 255, 251, 16, 0, 0],
            &[
                0, 0, 154, 147, 64, 9, 0, 0, 0, 19, 133, 255, 255, 167, 0, 0, 0,
            ],
            &[
                0, 0, 180, 255, 255, 252, 225, 211, 223, 253, 255, 255, 212, 20, 0, 0, 0,
            ],
            &[
                0, 0, 73, 181, 242, 255, 255, 255, 255, 255, 225, 127, 10, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 29, 56, 65, 55, 27, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'T' / 0x54
        'T' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 55, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 70, 0, 0,
            ],
            &[
                0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 177, 225, 225, 225, 225, 229, 255, 255, 235, 225, 225, 225, 225, 222, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'U' / 0x55
        'U' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 8, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 21, 255, 255, 89, 0, 0, 0, 0, 0, 0, 37, 255, 255, 72, 0, 0,
            ],
            &[
                0, 3, 254, 255, 110, 0, 0, 0, 0, 0, 0, 58, 255, 255, 53, 0, 0,
            ],
            &[
                0, 0, 222, 255, 162, 0, 0, 0, 0, 0, 0, 107, 255, 253, 16, 0, 0,
            ],
            &[
                0, 0, 155, 255, 244, 22, 0, 0, 0, 0, 3, 206, 255, 201, 0, 0, 0,
            ],
            &[
                0, 0, 46, 253, 255, 205, 51, 0, 0, 30, 169, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 125, 255, 255, 255, 245, 240, 255, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 90, 210, 255, 255, 255, 255, 221, 109, 1, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 25, 53, 56, 27, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'V' / 0x56
        'V' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[19, 71, 71, 18, 0, 0, 0, 0, 0, 0, 0, 0, 5, 71, 71, 34, 0],
            &[
                22, 252, 255, 114, 0, 0, 0, 0, 0, 0, 0, 0, 68, 255, 255, 71, 0,
            ],
            &[
                0, 192, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 142, 255, 240, 5, 0,
            ],
            &[
                0, 110, 255, 251, 16, 0, 0, 0, 0, 0, 0, 0, 216, 255, 165, 0, 0,
            ],
            &[
                0, 28, 254, 255, 88, 0, 0, 0, 0, 0, 0, 35, 255, 255, 84, 0, 0,
            ],
            &[
                0, 0, 201, 255, 164, 0, 0, 0, 0, 0, 0, 109, 255, 247, 11, 0, 0,
            ],
            &[
                0, 0, 118, 255, 237, 3, 0, 0, 0, 0, 0, 184, 255, 178, 0, 0, 0,
            ],
            &[
                0, 0, 36, 255, 255, 62, 0, 0, 0, 0, 10, 247, 255, 97, 0, 0, 0,
            ],
            &[
                0, 0, 0, 209, 255, 138, 0, 0, 0, 0, 76, 255, 252, 19, 0, 0, 0,
            ],
            &[
                0, 0, 0, 127, 255, 215, 0, 0, 0, 0, 151, 255, 191, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 44, 255, 255, 36, 0, 0, 0, 225, 255, 110, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 218, 255, 113, 0, 0, 44, 255, 255, 29, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 135, 255, 189, 0, 0, 118, 255, 203, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 53, 255, 250, 14, 0, 191, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 226, 255, 82, 12, 250, 255, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 144, 255, 152, 72, 255, 216, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 255, 218, 135, 255, 135, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 232, 255, 217, 255, 54, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 153, 255, 255, 228, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 70, 255, 255, 148, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'W' / 0x57
        'W' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[22, 71, 66, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 51, 71, 36, 0],
            &[60, 255, 251, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 200, 255, 114, 0],
            &[32, 255, 255, 21, 0, 0, 0, 0, 0, 0, 0, 0, 0, 222, 255, 88, 0],
            &[5, 253, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0, 0, 243, 255, 62, 0],
            &[0, 230, 255, 70, 0, 0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 36, 0],
            &[
                0, 202, 255, 94, 0, 0, 10, 120, 120, 41, 0, 0, 31, 255, 255, 11, 0,
            ],
            &[
                0, 173, 255, 119, 0, 0, 66, 255, 255, 133, 0, 0, 54, 255, 241, 0, 0,
            ],
            &[
                0, 144, 255, 143, 0, 0, 126, 255, 252, 194, 0, 0, 76, 255, 215, 0, 0,
            ],
            &[
                0, 116, 255, 168, 0, 0, 185, 251, 189, 248, 7, 0, 99, 255, 189, 0, 0,
            ],
            &[
                0, 87, 255, 192, 0, 3, 242, 211, 134, 255, 61, 0, 120, 255, 163, 0, 0,
            ],
            &[
                0, 58, 255, 215, 0, 48, 255, 158, 80, 255, 122, 0, 141, 255, 137, 0, 0,
            ],
            &[
                0, 30, 255, 237, 0, 108, 255, 101, 22, 255, 183, 0, 160, 255, 112, 0, 0,
            ],
            &[
                0, 4, 252, 253, 3, 166, 255, 42, 0, 216, 241, 3, 178, 255, 86, 0, 0,
            ],
            &[
                0, 0, 228, 255, 19, 222, 237, 1, 0, 154, 255, 49, 194, 255, 60, 0, 0,
            ],
            &[
                0, 0, 199, 255, 59, 255, 179, 0, 0, 92, 255, 108, 208, 255, 34, 0, 0,
            ],
            &[
                0, 0, 171, 255, 127, 255, 120, 0, 0, 31, 255, 163, 221, 255, 9, 0, 0,
            ],
            &[
                0, 0, 142, 255, 192, 255, 60, 0, 0, 0, 225, 213, 233, 238, 0, 0, 0,
            ],
            &[
                0, 0, 113, 255, 247, 249, 8, 0, 0, 0, 163, 251, 251, 213, 0, 0, 0,
            ],
            &[
                0, 0, 85, 255, 255, 198, 0, 0, 0, 0, 102, 255, 255, 187, 0, 0, 0,
            ],
            &[
                0, 0, 56, 255, 255, 138, 0, 0, 0, 0, 40, 255, 255, 161, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'X' / 0x58
        'X' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 71, 70, 3, 0, 0, 0, 0, 0, 0, 0, 56, 71, 54, 0, 0],
            &[
                0, 49, 252, 255, 103, 0, 0, 0, 0, 0, 0, 43, 252, 255, 98, 0, 0,
            ],
            &[
                0, 0, 147, 255, 233, 13, 0, 0, 0, 0, 0, 185, 255, 199, 1, 0, 0,
            ],
            &[
                0, 0, 16, 233, 255, 134, 0, 0, 0, 0, 74, 255, 253, 50, 0, 0, 0,
            ],
            &[
                0, 0, 0, 95, 255, 247, 30, 0, 0, 4, 215, 255, 150, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 195, 255, 165, 0, 0, 109, 255, 235, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 46, 252, 254, 54, 17, 237, 255, 100, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 144, 255, 195, 144, 255, 201, 1, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 14, 231, 255, 254, 253, 52, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 91, 255, 255, 153, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 80, 255, 255, 134, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 224, 255, 255, 249, 39, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 133, 255, 212, 154, 255, 186, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 38, 249, 255, 69, 19, 238, 255, 85, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 186, 255, 178, 0, 0, 110, 255, 227, 12, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 85, 255, 249, 37, 0, 0, 4, 214, 255, 138, 0, 0, 0, 0,
            ],
            &[
                0, 0, 12, 228, 255, 140, 0, 0, 0, 0, 71, 255, 250, 42, 0, 0, 0,
            ],
            &[
                0, 0, 139, 255, 233, 15, 0, 0, 0, 0, 0, 179, 255, 190, 0, 0, 0,
            ],
            &[
                0, 42, 250, 255, 101, 0, 0, 0, 0, 0, 0, 37, 249, 255, 89, 0, 0,
            ],
            &[
                0, 192, 255, 208, 2, 0, 0, 0, 0, 0, 0, 0, 139, 255, 230, 13, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Y' / 0x59
        'Y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 69, 71, 48, 0, 0, 0, 0, 0, 0, 0, 0, 28, 71, 71, 13, 0],
            &[
                0, 168, 255, 238, 12, 0, 0, 0, 0, 0, 0, 0, 178, 255, 216, 3, 0,
            ],
            &[
                0, 41, 253, 255, 112, 0, 0, 0, 0, 0, 0, 43, 254, 255, 89, 0, 0,
            ],
            &[
                0, 0, 163, 255, 226, 4, 0, 0, 0, 0, 0, 161, 255, 212, 2, 0, 0,
            ],
            &[
                0, 0, 37, 252, 255, 93, 0, 0, 0, 0, 29, 250, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 159, 255, 210, 0, 0, 0, 0, 143, 255, 208, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 33, 250, 255, 74, 0, 0, 18, 244, 255, 80, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 192, 0, 0, 125, 255, 204, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 30, 249, 255, 55, 9, 234, 255, 75, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 149, 255, 173, 108, 255, 200, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 247, 253, 232, 255, 70, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 145, 255, 255, 196, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 255, 82, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Z' / 0x5a
        'Z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 63, 71, 71, 71, 71, 71, 71, 71, 71, 71, 45, 0, 0, 0],
            &[
                0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 162, 0, 0, 0,
            ],
            &[
                0, 0, 0, 189, 211, 211, 211, 211, 211, 211, 211, 252, 255, 153, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 255, 252, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 206, 255, 153, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 96, 255, 243, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 10, 228, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 127, 255, 228, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 244, 255, 96, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 158, 255, 207, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 253, 255, 67, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 189, 255, 181, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 77, 255, 252, 42, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 216, 255, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 255, 242, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 236, 255, 124, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 255, 227, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 34, 249, 255, 130, 40, 40, 40, 40, 40, 40, 40, 25, 0, 0, 0,
            ],
            &[
                0, 0, 110, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 162, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 162, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '[' / 0x5b
        '[' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 36, 71, 71, 71, 71, 71, 39, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 255, 255, 142, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 238, 170, 170, 170, 94, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 133, 255, 239, 173, 173, 173, 96, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 133, 255, 255, 255, 255, 255, 142, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 35, 68, 68, 68, 68, 68, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '\' / 0x5c
        '\\' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 71, 43, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 118, 255, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 251, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 177, 255, 160, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 255, 245, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 232, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 138, 255, 199, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 255, 255, 42, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 198, 255, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 100, 255, 233, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 244, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 255, 178, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 61, 255, 251, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 218, 255, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 216, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 26, 252, 255, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 180, 255, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 82, 255, 243, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 234, 255, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ']' / 0x5d
        ']' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 49, 140, 140, 140, 140, 140, 100, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 90, 255, 255, 255, 255, 255, 184, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 35, 101, 101, 101, 194, 255, 184, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 184, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 86, 242, 242, 242, 250, 255, 184, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 90, 254, 254, 254, 254, 254, 183, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '^' / 0x5e
        '^' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 101, 61, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 255, 232, 11, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 198, 255, 255, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 64, 255, 176, 235, 239, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 185, 255, 50, 117, 255, 133, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 51, 255, 190, 0, 10, 233, 244, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 171, 255, 75, 0, 0, 113, 255, 144, 0, 0, 0, 0, 0],
            &[0, 0, 0, 39, 253, 214, 1, 0, 0, 9, 230, 248, 29, 0, 0, 0, 0],
            &[
                0, 0, 0, 158, 255, 100, 0, 0, 0, 0, 110, 255, 155, 0, 0, 0, 0,
            ],
            &[0, 0, 29, 250, 232, 7, 0, 0, 0, 0, 7, 228, 251, 37, 0, 0, 0],
            &[
                0, 0, 144, 255, 125, 0, 0, 0, 0, 0, 0, 106, 255, 167, 0, 0, 0,
            ],
            &[0, 20, 245, 245, 19, 0, 0, 0, 0, 0, 0, 6, 226, 254, 46, 0, 0],
            &[0, 93, 202, 128, 0, 0, 0, 0, 0, 0, 0, 0, 92, 202, 130, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '_' / 0x5f
        '_' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 51,
            ],
            &[
                255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 51,
            ],
            &[
                40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 8,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '`' / 0x60
        '`' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 174, 173, 23, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 248, 255, 153, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 252, 254, 61, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 87, 250, 221, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 69, 163, 77, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'a' / 0x61
        'a' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 108, 159, 180, 183, 161, 106, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 174, 252, 255, 255, 255, 255, 255, 255, 239, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 163, 248, 163, 95, 66, 75, 119, 226, 255, 248, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 27, 20, 0, 0, 0, 0, 0, 25, 235, 255, 132, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 181, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 196, 0, 0],
            &[
                0, 0, 0, 0, 0, 10, 76, 135, 162, 183, 191, 198, 226, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 232, 255, 255, 255, 255, 253, 245, 246, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 88, 255, 255, 194, 90, 36, 11, 0, 0, 113, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 170, 1, 0, 0, 0, 0, 0, 123, 255, 197, 0, 0,
            ],
            &[
                0, 0, 10, 255, 255, 64, 0, 0, 0, 0, 0, 0, 165, 255, 197, 0, 0,
            ],
            &[
                0, 0, 18, 255, 255, 73, 0, 0, 0, 0, 0, 38, 248, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 227, 255, 192, 16, 0, 0, 1, 73, 226, 255, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 243, 195, 190, 230, 255, 212, 121, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 223, 255, 255, 255, 245, 144, 19, 61, 255, 197, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 1, 29, 59, 39, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'b' / 0x62
        'b' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 133, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 214, 255, 127, 0, 58, 146, 178, 174, 129, 36, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 120, 153, 255, 255, 255, 255, 255, 249, 110, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 226, 255, 192, 90, 65, 99, 209, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 171, 1, 0, 0, 0, 9, 207, 255, 226, 3, 0, 0,
            ],
            &[
                0, 0, 214, 255, 250, 24, 0, 0, 0, 0, 0, 90, 255, 255, 70, 0, 0,
            ],
            &[
                0, 0, 214, 255, 191, 0, 0, 0, 0, 0, 0, 7, 251, 255, 129, 0, 0,
            ],
            &[
                0, 0, 214, 255, 148, 0, 0, 0, 0, 0, 0, 0, 227, 255, 163, 0, 0,
            ],
            &[
                0, 0, 214, 255, 133, 0, 0, 0, 0, 0, 0, 0, 204, 255, 178, 0, 0,
            ],
            &[
                0, 0, 214, 255, 140, 0, 0, 0, 0, 0, 0, 0, 215, 255, 171, 0, 0,
            ],
            &[
                0, 0, 214, 255, 167, 0, 0, 0, 0, 0, 0, 0, 240, 255, 150, 0, 0,
            ],
            &[
                0, 0, 214, 255, 224, 0, 0, 0, 0, 0, 0, 38, 255, 255, 102, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 73, 0, 0, 0, 0, 0, 139, 255, 252, 28, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 231, 52, 0, 0, 0, 82, 249, 255, 164, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 136, 245, 255, 209, 185, 219, 255, 255, 217, 21, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 41, 45, 191, 255, 255, 255, 249, 163, 22, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 25, 58, 54, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'c' / 0x63
        'c' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 3, 65, 138, 168, 185, 171, 129, 44, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 49, 211, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0,
            ],
            &[
                0, 0, 0, 40, 244, 255, 252, 179, 118, 100, 123, 173, 250, 44, 0, 0, 0,
            ],
            &[0, 0, 0, 203, 255, 241, 61, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0],
            &[0, 0, 45, 255, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 255, 254, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 255, 224, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 171, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 166, 255, 212, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 235, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 102, 255, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 253, 255, 167, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 152, 255, 255, 158, 31, 0, 0, 0, 16, 89, 71, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 196, 255, 255, 254, 229, 214, 239, 255, 255, 121, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 4, 118, 217, 255, 255, 255, 255, 251, 191, 64, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 27, 53, 59, 36, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'd' / 0x64
        'd' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 48, 133, 133, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 255, 255, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 255, 255, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 255, 255, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 255, 255, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 255, 255, 10, 0, 0],
            &[
                0, 0, 0, 0, 13, 103, 156, 181, 157, 101, 5, 90, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 68, 239, 255, 255, 255, 255, 255, 213, 92, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 46, 250, 255, 243, 130, 69, 79, 154, 253, 229, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 187, 255, 246, 49, 0, 0, 0, 0, 113, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 20, 253, 255, 141, 0, 0, 0, 0, 0, 3, 226, 255, 255, 10, 0, 0,
            ],
            &[
                0, 82, 255, 255, 58, 0, 0, 0, 0, 0, 0, 154, 255, 255, 10, 0, 0,
            ],
            &[
                0, 108, 255, 255, 14, 0, 0, 0, 0, 0, 0, 113, 255, 255, 10, 0, 0,
            ],
            &[
                0, 126, 255, 252, 0, 0, 0, 0, 0, 0, 0, 97, 255, 255, 10, 0, 0,
            ],
            &[
                0, 119, 255, 255, 7, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 99, 255, 255, 33, 0, 0, 0, 0, 0, 0, 126, 255, 255, 10, 0, 0,
            ],
            &[
                0, 52, 255, 255, 81, 0, 0, 0, 0, 0, 0, 171, 255, 255, 10, 0, 0,
            ],
            &[
                0, 3, 230, 255, 188, 0, 0, 0, 0, 0, 28, 250, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 119, 255, 255, 125, 3, 0, 0, 29, 208, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 4, 190, 255, 255, 233, 188, 212, 252, 252, 119, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 9, 137, 241, 255, 255, 255, 217, 73, 8, 254, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 5, 47, 60, 27, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'e' / 0x65
        'e' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 119, 160, 182, 158, 105, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 134, 251, 255, 255, 255, 255, 255, 238, 70, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 160, 255, 255, 185, 100, 67, 96, 201, 255, 251, 55, 0, 0, 0,
            ],
            &[
                0, 0, 84, 255, 255, 119, 0, 0, 0, 0, 5, 188, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 198, 255, 189, 0, 0, 0, 0, 0, 0, 52, 255, 255, 46, 0, 0,
            ],
            &[
                0, 27, 254, 255, 93, 0, 0, 0, 0, 0, 0, 0, 242, 255, 109, 0, 0,
            ],
            &[
                0, 64, 255, 255, 196, 184, 184, 184, 184, 184, 184, 184, 246, 255, 133, 0, 0,
            ],
            &[
                0, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[
                0, 79, 255, 255, 81, 57, 57, 57, 57, 57, 57, 57, 57, 57, 33, 0, 0,
            ],
            &[0, 53, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 241, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 235, 29, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 25, 233, 255, 220, 73, 3, 0, 0, 0, 14, 98, 177, 0, 0, 0,
            ],
            &[
                0, 0, 0, 57, 229, 255, 255, 238, 202, 191, 213, 252, 255, 238, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 139, 228, 255, 255, 255, 255, 249, 192, 99, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 28, 53, 58, 32, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'f' / 0x66
        'f' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 12, 90, 142, 161, 153, 132, 85, 15, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 61, 233, 255, 255, 255, 255, 255, 255, 54, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 10, 236, 255, 246, 150, 100, 99, 123, 162, 1, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 83, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 115, 255, 254, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 240, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 7, 44, 63, 82, 101, 184, 255, 247, 120, 120, 120, 120, 120, 47, 0, 0,
            ],
            &[
                0, 55, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0,
            ],
            &[
                0, 30, 141, 141, 141, 141, 197, 255, 248, 141, 141, 141, 141, 141, 55, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 126, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'g' / 0x67
        'g' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 15, 107, 167, 183, 163, 96, 5, 0, 114, 120, 4, 0, 0,
            ],
            &[
                0, 0, 0, 56, 230, 255, 255, 255, 255, 255, 206, 39, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 30, 240, 255, 236, 121, 68, 77, 145, 249, 232, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 166, 255, 242, 36, 0, 0, 0, 0, 74, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 17, 249, 255, 147, 0, 0, 0, 0, 0, 0, 177, 255, 255, 10, 0, 0,
            ],
            &[
                0, 72, 255, 255, 57, 0, 0, 0, 0, 0, 0, 100, 255, 255, 10, 0, 0,
            ],
            &[
                0, 110, 255, 255, 24, 0, 0, 0, 0, 0, 0, 59, 255, 255, 10, 0, 0,
            ],
            &[
                0, 126, 255, 253, 2, 0, 0, 0, 0, 0, 0, 42, 255, 255, 10, 0, 0,
            ],
            &[
                0, 121, 255, 254, 2, 0, 0, 0, 0, 0, 0, 40, 255, 255, 10, 0, 0,
            ],
            &[
                0, 101, 255, 255, 27, 0, 0, 0, 0, 0, 0, 57, 255, 255, 10, 0, 0,
            ],
            &[
                0, 54, 255, 255, 85, 0, 0, 0, 0, 0, 0, 109, 255, 255, 10, 0, 0,
            ],
            &[
                0, 3, 228, 255, 191, 0, 0, 0, 0, 0, 3, 219, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 145, 7, 0, 0, 13, 166, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 3, 179, 255, 255, 240, 191, 202, 245, 255, 190, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 5, 125, 236, 255, 255, 255, 244, 134, 59, 255, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 3, 44, 63, 49, 7, 0, 65, 255, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 97, 255, 239, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 168, 255, 198, 0, 0, 0],
            &[0, 0, 76, 13, 0, 0, 0, 0, 0, 0, 67, 254, 255, 112, 0, 0, 0],
            &[
                0, 0, 194, 246, 182, 130, 98, 88, 107, 169, 251, 255, 207, 7, 0, 0, 0,
            ],
            &[
                0, 0, 137, 242, 255, 255, 255, 255, 255, 255, 255, 175, 22, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 6, 70, 118, 145, 152, 134, 105, 33, 0, 0, 0, 0, 0, 0,
            ],
        ],
        // letter: 'h' / 0x68
        'h' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 133, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 214, 255, 149, 0, 18, 119, 164, 182, 151, 71, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 141, 86, 239, 255, 255, 255, 255, 255, 174, 5, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 177, 250, 213, 103, 68, 101, 214, 255, 255, 129, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 197, 10, 0, 0, 0, 9, 224, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 45, 0, 0, 0, 0, 0, 119, 255, 255, 25, 0, 0,
            ],
            &[
                0, 0, 214, 255, 214, 0, 0, 0, 0, 0, 0, 82, 255, 255, 42, 0, 0,
            ],
            &[
                0, 0, 214, 255, 170, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 153, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'i' / 0x69
        'i' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 61, 106, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 63, 255, 255, 211, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 90, 255, 255, 238, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 158, 205, 91, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 49, 120, 120, 120, 120, 120, 120, 90, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 22, 68, 87, 106, 125, 218, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 255, 192, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 140, 185, 205, 225, 245, 255, 255, 255, 246, 224, 203, 181, 95, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'j' / 0x6a
        'j' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 67, 104, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 73, 255, 255, 197, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 97, 255, 255, 224, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 14, 164, 203, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 52, 120, 120, 120, 120, 120, 120, 86, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 111, 255, 255, 255, 255, 255, 255, 183, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 24, 69, 88, 107, 126, 222, 255, 183, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 191, 255, 180, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 230, 255, 158, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 98, 255, 255, 95, 0, 0, 0, 0],
            &[
                0, 0, 0, 89, 189, 139, 112, 118, 173, 254, 255, 216, 7, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 97, 255, 255, 255, 255, 255, 255, 204, 33, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 18, 82, 124, 144, 151, 127, 66, 1, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'k' / 0x6b
        'k' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 133, 133, 35, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 15, 119, 120, 95, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 0, 7, 192, 255, 239, 46, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 2, 172, 255, 245, 59, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 149, 255, 250, 74, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 66, 0, 0, 125, 255, 254, 91, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 61, 0, 100, 255, 255, 109, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 50, 73, 252, 255, 141, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 78, 242, 255, 255, 201, 5, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 231, 251, 123, 246, 255, 141, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 249, 80, 0, 105, 255, 254, 76, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 99, 0, 0, 0, 175, 255, 237, 29, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 18, 227, 255, 196, 4, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 0, 62, 252, 255, 135, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 129, 255, 254, 71, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 3, 196, 255, 234, 26, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'l' / 0x6c
        'l' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 90, 133, 133, 133, 133, 133, 133, 117, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 255, 255, 255, 225, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 30, 60, 77, 95, 112, 196, 255, 225, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 226, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 140, 184, 204, 224, 243, 255, 255, 255, 248, 226, 204, 182, 95, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'm' / 0x6d
        'm' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 78, 120, 9, 46, 161, 177, 107, 0, 2, 107, 176, 166, 61, 0, 0, 0,
            ],
            &[
                0, 166, 255, 93, 243, 255, 255, 255, 108, 143, 255, 255, 255, 248, 42, 0, 0,
            ],
            &[
                0, 166, 255, 250, 200, 116, 218, 255, 236, 253, 176, 126, 239, 255, 141, 0, 0,
            ],
            &[
                0, 166, 255, 250, 25, 0, 72, 255, 255, 220, 4, 0, 139, 255, 190, 0, 0,
            ],
            &[
                0, 166, 255, 198, 0, 0, 13, 255, 255, 135, 0, 0, 90, 255, 213, 0, 0,
            ],
            &[
                0, 166, 255, 165, 0, 0, 0, 251, 255, 89, 0, 0, 78, 255, 222, 0, 0,
            ],
            &[
                0, 166, 255, 150, 0, 0, 0, 243, 255, 65, 0, 0, 72, 255, 224, 0, 0,
            ],
            &[
                0, 166, 255, 135, 0, 0, 0, 243, 255, 55, 0, 0, 72, 255, 224, 0, 0,
            ],
            &[
                0, 166, 255, 130, 0, 0, 0, 243, 255, 52, 0, 0, 72, 255, 224, 0, 0,
            ],
            &[
                0, 166, 255, 130, 0, 0, 0, 243, 255, 52, 0, 0, 72, 255, 224, 0, 0,
            ],
            &[
                0, 166, 255, 130, 0, 0, 0, 243, 255, 52, 0, 0, 72, 255, 224, 0, 0,
            ],
            &[
                0, 166, 255, 130, 0, 0, 0, 243, 255, 52, 0, 0, 72, 255, 224, 0, 0,
            ],
            &[
                0, 166, 255, 130, 0, 0, 0, 243, 255, 52, 0, 0, 72, 255, 224, 0, 0,
            ],
            &[
                0, 166, 255, 130, 0, 0, 0, 243, 255, 52, 0, 0, 72, 255, 224, 0, 0,
            ],
            &[
                0, 166, 255, 130, 0, 0, 0, 243, 255, 52, 0, 0, 72, 255, 224, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'n' / 0x6e
        'n' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 101, 120, 17, 0, 28, 125, 163, 183, 158, 83, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 67, 122, 249, 255, 255, 255, 255, 255, 191, 10, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 187, 255, 187, 92, 65, 92, 190, 255, 255, 144, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 163, 0, 0, 0, 0, 1, 200, 255, 239, 2, 0, 0,
            ],
            &[
                0, 0, 214, 255, 254, 38, 0, 0, 0, 0, 0, 104, 255, 255, 29, 0, 0,
            ],
            &[
                0, 0, 214, 255, 208, 0, 0, 0, 0, 0, 0, 74, 255, 255, 43, 0, 0,
            ],
            &[
                0, 0, 214, 255, 178, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 155, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'o' / 0x6f
        'o' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 142, 173, 176, 146, 72, 2, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 26, 195, 255, 255, 255, 255, 255, 255, 212, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 222, 255, 242, 132, 72, 72, 123, 232, 255, 240, 33, 0, 0, 0,
            ],
            &[
                0, 0, 151, 255, 243, 45, 0, 0, 0, 0, 29, 225, 255, 193, 0, 0, 0,
            ],
            &[
                0, 9, 243, 255, 128, 0, 0, 0, 0, 0, 0, 91, 255, 255, 41, 0, 0,
            ],
            &[
                0, 73, 255, 255, 39, 0, 0, 0, 0, 0, 0, 10, 249, 255, 121, 0, 0,
            ],
            &[
                0, 103, 255, 247, 1, 0, 0, 0, 0, 0, 0, 0, 212, 255, 151, 0, 0,
            ],
            &[
                0, 125, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 195, 255, 175, 0, 0,
            ],
            &[
                0, 117, 255, 241, 0, 0, 0, 0, 0, 0, 0, 0, 203, 255, 170, 0, 0,
            ],
            &[
                0, 94, 255, 255, 16, 0, 0, 0, 0, 0, 0, 0, 232, 255, 147, 0, 0,
            ],
            &[
                0, 37, 255, 255, 82, 0, 0, 0, 0, 0, 0, 42, 255, 255, 91, 0, 0,
            ],
            &[
                0, 0, 205, 255, 201, 2, 0, 0, 0, 0, 0, 160, 255, 244, 17, 0, 0,
            ],
            &[
                0, 0, 72, 254, 255, 165, 18, 0, 0, 7, 128, 255, 255, 129, 0, 0, 0,
            ],
            &[
                0, 0, 0, 119, 254, 255, 244, 199, 193, 236, 255, 255, 176, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 71, 199, 254, 255, 255, 255, 223, 109, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 21, 55, 60, 33, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'p' / 0x70
        'p' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 101, 120, 23, 0, 52, 139, 175, 175, 130, 38, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 77, 139, 254, 255, 255, 255, 255, 250, 116, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 210, 255, 179, 99, 66, 97, 201, 255, 255, 91, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 146, 0, 0, 0, 0, 4, 194, 255, 230, 4, 0, 0,
            ],
            &[
                0, 0, 214, 255, 246, 19, 0, 0, 0, 0, 0, 76, 255, 255, 73, 0, 0,
            ],
            &[
                0, 0, 214, 255, 190, 0, 0, 0, 0, 0, 0, 3, 248, 255, 130, 0, 0,
            ],
            &[
                0, 0, 214, 255, 163, 0, 0, 0, 0, 0, 0, 0, 225, 255, 164, 0, 0,
            ],
            &[
                0, 0, 214, 255, 145, 0, 0, 0, 0, 0, 0, 0, 202, 255, 178, 0, 0,
            ],
            &[
                0, 0, 214, 255, 161, 0, 0, 0, 0, 0, 0, 0, 205, 255, 167, 0, 0,
            ],
            &[
                0, 0, 214, 255, 184, 0, 0, 0, 0, 0, 0, 0, 230, 255, 147, 0, 0,
            ],
            &[
                0, 0, 214, 255, 234, 4, 0, 0, 0, 0, 0, 26, 255, 255, 100, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 74, 0, 0, 0, 0, 0, 123, 255, 254, 28, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 220, 39, 0, 0, 0, 79, 247, 255, 163, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 159, 234, 253, 204, 185, 219, 255, 255, 218, 24, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 136, 36, 177, 255, 255, 255, 249, 161, 22, 0, 0, 0, 0,
            ],
            &[0, 0, 214, 255, 143, 0, 0, 13, 53, 48, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 103, 122, 69, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'q' / 0x71
        'q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 11, 102, 165, 182, 158, 81, 2, 0, 114, 120, 4, 0, 0,
            ],
            &[
                0, 0, 0, 52, 226, 255, 255, 255, 255, 255, 182, 30, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 30, 239, 255, 230, 113, 66, 77, 150, 252, 214, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 168, 255, 238, 29, 0, 0, 0, 0, 100, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 18, 250, 255, 142, 0, 0, 0, 0, 0, 3, 232, 255, 255, 10, 0, 0,
            ],
            &[
                0, 74, 255, 255, 55, 0, 0, 0, 0, 0, 0, 153, 255, 255, 10, 0, 0,
            ],
            &[
                0, 111, 255, 255, 24, 0, 0, 0, 0, 0, 0, 123, 255, 255, 10, 0, 0,
            ],
            &[
                0, 126, 255, 253, 2, 0, 0, 0, 0, 0, 0, 100, 255, 255, 10, 0, 0,
            ],
            &[
                0, 120, 255, 255, 10, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 99, 255, 255, 36, 0, 0, 0, 0, 0, 0, 125, 255, 255, 10, 0, 0,
            ],
            &[
                0, 50, 255, 255, 97, 0, 0, 0, 0, 0, 0, 165, 255, 255, 10, 0, 0,
            ],
            &[
                0, 2, 225, 255, 201, 0, 0, 0, 0, 0, 24, 248, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 110, 255, 255, 139, 4, 0, 0, 27, 199, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 2, 180, 255, 255, 235, 188, 211, 251, 252, 149, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 6, 128, 238, 255, 255, 255, 214, 67, 87, 255, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 4, 46, 62, 37, 0, 0, 92, 255, 255, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 255, 255, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 255, 255, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 255, 255, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 255, 255, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 255, 255, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 44, 122, 122, 4, 0, 0],
        ],
        // letter: 'r' / 0x72
        'r' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 107, 120, 120, 120, 120, 120, 5, 0, 53, 153, 182, 164, 117, 24, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 255, 255, 35, 94, 254, 255, 255, 255, 255, 63, 0,
            ],
            &[
                0, 0, 55, 83, 105, 128, 247, 255, 105, 248, 247, 151, 109, 137, 201, 3, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 236, 255, 248, 236, 44, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 255, 91, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 232, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 168, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 237, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 52, 178, 201, 225, 248, 255, 255, 255, 238, 213, 189, 124, 0, 0, 0, 0,
            ],
            &[
                0, 83, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 's' / 0x73
        's' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 56, 130, 170, 184, 168, 144, 87, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 9, 174, 255, 255, 255, 255, 255, 255, 255, 239, 8, 0, 0, 0,
            ],
            &[
                0, 0, 0, 136, 255, 254, 160, 95, 77, 92, 124, 199, 161, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 213, 255, 151, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0],
            &[0, 0, 0, 222, 255, 147, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 160, 255, 246, 106, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 21, 210, 255, 255, 220, 107, 10, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 119, 233, 255, 255, 236, 130, 21, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 10, 111, 224, 255, 255, 238, 74, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 93, 237, 255, 244, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 255, 255, 76, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 37, 255, 255, 78, 0, 0, 0],
            &[
                0, 0, 36, 162, 65, 3, 0, 0, 0, 12, 164, 255, 251, 29, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 244, 205, 191, 205, 243, 255, 255, 136, 0, 0, 0, 0,
            ],
            &[
                0, 0, 15, 144, 225, 255, 255, 255, 255, 255, 214, 98, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 56, 65, 52, 23, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 't' / 0x74
        't' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 85, 75, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 75, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 108, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 18, 58, 86, 114, 197, 255, 240, 120, 120, 120, 120, 120, 63, 0, 0,
            ],
            &[
                0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 134, 0, 0,
            ],
            &[
                0, 0, 69, 141, 141, 141, 203, 255, 242, 141, 141, 141, 141, 141, 74, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 137, 255, 233, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 114, 255, 253, 20, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 59, 255, 255, 163, 6, 0, 0, 0, 7, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 183, 255, 255, 237, 190, 190, 216, 234, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 10, 142, 241, 255, 255, 255, 255, 222, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 41, 62, 55, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'u' / 0x75
        'u' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 117, 120, 54, 0, 0, 0, 0, 0, 0, 47, 120, 120, 4, 0, 0],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 117, 0, 0, 0, 0, 0, 0, 126, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 243, 255, 137, 0, 0, 0, 0, 0, 0, 175, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 219, 255, 190, 0, 0, 0, 0, 0, 24, 247, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 154, 255, 255, 98, 0, 0, 0, 29, 198, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 37, 244, 255, 255, 224, 189, 205, 250, 246, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 54, 204, 255, 255, 255, 255, 199, 63, 5, 249, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 59, 55, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'v' / 0x76
        'v' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 120, 94, 0, 0, 0, 0, 0, 0, 0, 0, 79, 120, 105, 0, 0],
            &[
                0, 108, 255, 249, 17, 0, 0, 0, 0, 0, 0, 2, 231, 255, 152, 0, 0,
            ],
            &[
                0, 20, 250, 255, 100, 0, 0, 0, 0, 0, 0, 65, 255, 255, 56, 0, 0,
            ],
            &[
                0, 0, 175, 255, 190, 0, 0, 0, 0, 0, 0, 153, 255, 216, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 253, 27, 0, 0, 0, 0, 4, 236, 255, 121, 0, 0, 0,
            ],
            &[
                0, 0, 6, 236, 255, 115, 0, 0, 0, 0, 72, 255, 253, 28, 0, 0, 0,
            ],
            &[
                0, 0, 0, 148, 255, 205, 0, 0, 0, 0, 160, 255, 186, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 53, 255, 255, 39, 0, 0, 7, 240, 255, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 214, 255, 129, 0, 0, 79, 255, 241, 9, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 120, 255, 219, 0, 0, 168, 255, 156, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 253, 255, 53, 12, 246, 255, 60, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 187, 255, 141, 93, 255, 220, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 92, 255, 227, 185, 255, 125, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 243, 255, 255, 254, 31, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 255, 255, 190, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'w' / 0x77
        'w' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[64, 120, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 55, 120, 91, 0],
            &[
                105, 255, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 146, 255, 162, 0,
            ],
            &[64, 255, 249, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 187, 255, 120, 0],
            &[
                23, 255, 255, 35, 0, 0, 68, 190, 190, 100, 0, 0, 0, 227, 255, 78, 0,
            ],
            &[
                0, 238, 255, 72, 0, 0, 155, 255, 255, 196, 0, 0, 13, 255, 255, 36, 0,
            ],
            &[
                0, 197, 255, 110, 0, 0, 228, 253, 234, 251, 15, 0, 52, 255, 247, 2, 0,
            ],
            &[
                0, 156, 255, 148, 0, 46, 255, 207, 164, 255, 81, 0, 93, 255, 208, 0, 0,
            ],
            &[
                0, 114, 255, 185, 0, 120, 255, 145, 104, 255, 152, 0, 133, 255, 166, 0, 0,
            ],
            &[
                0, 73, 255, 219, 0, 190, 255, 77, 40, 255, 221, 0, 169, 255, 124, 0, 0,
            ],
            &[
                0, 32, 255, 249, 8, 247, 250, 12, 0, 230, 255, 33, 201, 255, 82, 0, 0,
            ],
            &[
                0, 1, 245, 255, 73, 255, 194, 0, 0, 165, 255, 94, 225, 255, 40, 0, 0,
            ],
            &[
                0, 0, 206, 255, 142, 255, 128, 0, 0, 99, 255, 150, 245, 249, 4, 0, 0,
            ],
            &[
                0, 0, 165, 255, 206, 255, 61, 0, 0, 34, 255, 206, 255, 212, 0, 0, 0,
            ],
            &[
                0, 0, 124, 255, 255, 244, 5, 0, 0, 0, 225, 255, 255, 170, 0, 0, 0,
            ],
            &[
                0, 0, 83, 255, 255, 184, 0, 0, 0, 0, 159, 255, 255, 128, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'x' / 0x78
        'x' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 41, 120, 120, 46, 0, 0, 0, 0, 0, 0, 21, 120, 120, 65, 0, 0,
            ],
            &[
                0, 4, 199, 255, 219, 12, 0, 0, 0, 0, 0, 180, 255, 231, 23, 0, 0,
            ],
            &[
                0, 0, 32, 239, 255, 159, 0, 0, 0, 0, 108, 255, 253, 67, 0, 0, 0,
            ],
            &[
                0, 0, 0, 83, 255, 255, 87, 0, 0, 44, 247, 255, 131, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 150, 255, 239, 30, 8, 211, 255, 194, 3, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 8, 209, 255, 194, 149, 255, 236, 28, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 40, 244, 255, 255, 254, 76, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 255, 171, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 19, 228, 255, 255, 246, 45, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 1, 180, 255, 221, 186, 255, 214, 11, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 115, 255, 251, 54, 23, 233, 255, 159, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 54, 250, 255, 120, 0, 0, 73, 254, 255, 93, 0, 0, 0, 0,
            ],
            &[
                0, 0, 15, 222, 255, 189, 2, 0, 0, 0, 144, 255, 243, 39, 0, 0, 0,
            ],
            &[
                0, 0, 172, 255, 236, 26, 0, 0, 0, 0, 7, 209, 255, 208, 8, 0, 0,
            ],
            &[
                0, 106, 255, 255, 78, 0, 0, 0, 0, 0, 0, 42, 246, 255, 150, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'y' / 0x79
        'y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 82, 120, 82, 0, 0, 0, 0, 0, 0, 0, 0, 73, 120, 105, 0, 0],
            &[
                0, 97, 255, 241, 11, 0, 0, 0, 0, 0, 0, 0, 217, 255, 152, 0, 0,
            ],
            &[
                0, 9, 237, 255, 100, 0, 0, 0, 0, 0, 0, 47, 255, 255, 57, 0, 0,
            ],
            &[
                0, 0, 140, 255, 204, 0, 0, 0, 0, 0, 0, 133, 255, 218, 0, 0, 0,
            ],
            &[
                0, 0, 35, 254, 255, 52, 0, 0, 0, 0, 0, 218, 255, 123, 0, 0, 0,
            ],
            &[
                0, 0, 0, 183, 255, 156, 0, 0, 0, 0, 48, 255, 254, 30, 0, 0, 0,
            ],
            &[
                0, 0, 0, 77, 255, 245, 15, 0, 0, 0, 133, 255, 189, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 2, 223, 255, 108, 0, 0, 0, 219, 255, 94, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 120, 255, 212, 0, 0, 49, 255, 243, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 248, 255, 56, 0, 133, 255, 160, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 163, 255, 150, 0, 215, 255, 65, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 255, 233, 43, 255, 224, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 206, 255, 179, 255, 131, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 255, 255, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 239, 255, 197, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 219, 255, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 64, 255, 247, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 178, 255, 155, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 69, 255, 254, 40, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 93, 88, 137, 248, 255, 164, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 69, 255, 255, 255, 255, 192, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 29, 133, 152, 135, 76, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'z' / 0x7a
        'z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 16, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 66, 0, 0, 0,
            ],
            &[
                0, 0, 35, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 141, 0, 0, 0,
            ],
            &[
                0, 0, 13, 100, 100, 100, 100, 100, 100, 100, 147, 255, 255, 110, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 220, 255, 183, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 178, 255, 222, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 122, 255, 247, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 67, 252, 255, 95, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 234, 255, 151, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 199, 255, 199, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 148, 255, 233, 28, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 255, 252, 64, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 245, 255, 116, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 14, 217, 255, 180, 13, 13, 13, 13, 13, 13, 13, 10, 0, 0, 0,
            ],
            &[
                0, 0, 133, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '{' / 0x7b
        '{' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 14, 52, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 43, 182, 251, 255, 107, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 42, 240, 255, 253, 214, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 167, 255, 236, 60, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 220, 255, 135, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 252, 255, 92, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 255, 251, 35, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 150, 205, 255, 249, 104, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 255, 212, 54, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 138, 191, 255, 253, 122, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 106, 255, 253, 41, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 252, 255, 95, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 108, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 109, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 220, 255, 138, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 167, 255, 236, 44, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 43, 241, 255, 253, 205, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 46, 182, 250, 255, 107, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 47, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '|' / 0x7c
        '|' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 133, 13, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 255, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 107, 122, 12, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '}' / 0x7d
        '}' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 54, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 255, 206, 75, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 38, 194, 245, 255, 254, 84, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 205, 255, 219, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 87, 255, 255, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 57, 255, 255, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 57, 255, 255, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 57, 255, 255, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 57, 255, 255, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 43, 255, 255, 56, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 235, 255, 159, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 78, 243, 255, 213, 146, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 30, 184, 255, 255, 107, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 68, 237, 255, 220, 153, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 230, 255, 163, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 41, 255, 255, 56, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 57, 255, 255, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 57, 255, 255, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 57, 255, 255, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 57, 255, 255, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 84, 255, 255, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 35, 213, 255, 217, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 39, 207, 250, 255, 252, 78, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 254, 200, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 53, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '~' / 0x7e
        '~' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 89, 108, 51, 0, 0, 0, 0, 0, 62, 121, 13, 0, 0],
            &[
                0, 0, 3, 185, 255, 255, 255, 165, 10, 0, 0, 0, 164, 248, 3, 0, 0,
            ],
            &[
                0, 0, 95, 255, 130, 55, 158, 255, 203, 25, 0, 3, 230, 187, 0, 0, 0,
            ],
            &[
                0, 0, 178, 231, 2, 0, 0, 114, 253, 230, 125, 163, 255, 89, 0, 0, 0,
            ],
            &[
                0, 0, 227, 178, 0, 0, 0, 0, 78, 227, 255, 255, 153, 0, 0, 0, 0,
            ],
            &[0, 0, 63, 39, 0, 0, 0, 0, 0, 5, 46, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ' ' / 0xa0
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¡' / 0xa1
        '¡' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 85, 103, 13, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 105, 255, 255, 156, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 136, 255, 255, 187, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 57, 231, 242, 97, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 7, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 204, 212, 29, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 254, 255, 42, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 255, 255, 50, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 14, 255, 255, 58, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 30, 255, 255, 74, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 38, 255, 255, 82, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 46, 255, 255, 90, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 54, 255, 255, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 62, 255, 255, 106, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 70, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 78, 255, 255, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 86, 255, 255, 130, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 94, 254, 254, 138, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¢' / 0xa2
        '¢' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 140, 112, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 8, 255, 204, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 47, 255, 224, 78, 48, 7, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 1, 100, 223, 255, 255, 255, 255, 255, 244, 125, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 159, 255, 255, 245, 195, 176, 199, 234, 255, 96, 0, 0, 0,
            ],
            &[0, 0, 0, 104, 255, 255, 152, 15, 0, 0, 0, 1, 50, 12, 0, 0, 0],
            &[0, 0, 3, 234, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 255, 255, 65, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 110, 255, 251, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 255, 233, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 120, 255, 245, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 90, 255, 255, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 253, 255, 137, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 172, 255, 250, 81, 0, 0, 0, 0, 0, 2, 21, 0, 0, 0],
            &[
                0, 0, 0, 29, 230, 255, 255, 194, 132, 116, 138, 167, 233, 134, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 31, 189, 255, 255, 255, 255, 255, 255, 255, 125, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 48, 120, 255, 234, 143, 111, 46, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 8, 255, 198, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 8, 255, 198, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 69, 53, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '£' / 0xa3
        '£' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 87, 124, 124, 92, 22, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 83, 238, 255, 255, 255, 255, 247, 138, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 57, 251, 255, 229, 154, 143, 186, 253, 131, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 181, 255, 224, 15, 0, 0, 0, 38, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 245, 255, 124, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 255, 255, 91, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 52, 107, 116, 255, 255, 154, 107, 107, 107, 107, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 125, 255, 255, 255, 255, 255, 255, 255, 255, 255, 24, 0, 0, 0, 0,
            ],
            &[
                0, 0, 45, 92, 101, 255, 255, 144, 92, 92, 92, 92, 8, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 15, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 53, 255, 253, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 255, 171, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 20, 141, 255, 238, 65, 40, 40, 40, 40, 40, 40, 40, 7, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 44, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¤' / 0xa4
        '¤' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 152, 3, 0, 0, 8, 15, 0, 0, 0, 112, 103, 0, 0, 0],
            &[
                0, 3, 214, 255, 170, 96, 212, 253, 255, 227, 121, 127, 255, 245, 24, 0, 0,
            ],
            &[
                0, 0, 35, 229, 255, 255, 247, 189, 180, 233, 255, 255, 249, 71, 0, 0, 0,
            ],
            &[
                0, 0, 0, 120, 255, 190, 22, 0, 0, 4, 136, 255, 185, 0, 0, 0, 0,
            ],
            &[0, 0, 2, 228, 240, 16, 0, 0, 0, 0, 0, 188, 254, 40, 0, 0, 0],
            &[0, 0, 36, 255, 169, 0, 0, 0, 0, 0, 0, 103, 255, 99, 0, 0, 0],
            &[0, 0, 57, 255, 153, 0, 0, 0, 0, 0, 0, 87, 255, 123, 0, 0, 0],
            &[0, 0, 17, 255, 199, 0, 0, 0, 0, 0, 0, 132, 255, 86, 0, 0, 0],
            &[0, 0, 0, 181, 255, 80, 0, 0, 0, 0, 30, 236, 237, 14, 0, 0, 0],
            &[
                0, 0, 0, 140, 255, 251, 144, 66, 57, 113, 233, 255, 196, 3, 0, 0, 0,
            ],
            &[
                0, 0, 116, 255, 246, 225, 255, 255, 255, 255, 243, 234, 255, 163, 2, 0, 0,
            ],
            &[
                0, 1, 176, 243, 64, 5, 90, 141, 151, 108, 19, 36, 225, 217, 18, 0, 0,
            ],
            &[0, 0, 5, 44, 0, 0, 0, 0, 0, 0, 0, 0, 30, 25, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¥' / 0xa5
        '¥' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 69, 71, 37, 0, 0, 0, 0, 0, 0, 0, 0, 21, 71, 71, 12, 0],
            &[
                0, 168, 255, 212, 1, 0, 0, 0, 0, 0, 0, 0, 157, 255, 214, 2, 0,
            ],
            &[
                0, 40, 252, 255, 83, 0, 0, 0, 0, 0, 0, 34, 251, 255, 83, 0, 0,
            ],
            &[
                0, 0, 161, 255, 208, 1, 0, 0, 0, 0, 0, 157, 255, 204, 1, 0, 0,
            ],
            &[
                0, 0, 35, 251, 255, 78, 0, 0, 0, 0, 33, 251, 255, 71, 0, 0, 0,
            ],
            &[
                0, 0, 0, 155, 255, 203, 0, 0, 0, 0, 156, 255, 193, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 31, 249, 255, 73, 0, 0, 33, 250, 255, 60, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 149, 255, 198, 0, 0, 155, 255, 182, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 26, 247, 255, 68, 32, 250, 254, 49, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 143, 255, 193, 154, 255, 170, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 38, 39, 62, 251, 255, 253, 255, 79, 39, 39, 6, 0, 0, 0,
            ],
            &[
                0, 0, 0, 249, 255, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0,
            ],
            &[
                0, 0, 0, 143, 147, 147, 156, 255, 255, 172, 147, 147, 147, 25, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 22, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 52, 53, 53, 71, 255, 255, 101, 53, 53, 53, 9, 0, 0, 0,
            ],
            &[
                0, 0, 0, 249, 255, 255, 255, 255, 255, 255, 255, 255, 255, 45, 0, 0, 0,
            ],
            &[
                0, 0, 0, 129, 132, 132, 143, 255, 255, 161, 132, 132, 132, 23, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 22, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¦' / 0xa6
        '¦' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 119, 133, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 176, 196, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 179, 199, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 255, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 129, 9, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '§' / 0xa7
        '§' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 34, 114, 142, 163, 154, 123, 61, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 1, 152, 253, 255, 255, 255, 255, 255, 255, 199, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 107, 255, 242, 103, 43, 32, 54, 86, 168, 110, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 175, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 166, 255, 193, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 66, 254, 255, 194, 56, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 242, 255, 255, 198, 81, 2, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 113, 249, 243, 232, 255, 255, 216, 80, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 82, 255, 198, 19, 3, 70, 187, 255, 255, 108, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 184, 255, 51, 0, 0, 0, 0, 124, 255, 244, 13, 0, 0, 0,
            ],
            &[0, 0, 0, 206, 255, 54, 0, 0, 0, 0, 0, 227, 255, 49, 0, 0, 0],
            &[
                0, 0, 0, 163, 255, 188, 15, 0, 0, 0, 0, 213, 255, 30, 0, 0, 0,
            ],
            &[
                0, 0, 0, 41, 240, 255, 222, 103, 9, 0, 103, 255, 198, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 41, 192, 255, 255, 235, 182, 255, 227, 35, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 55, 174, 253, 255, 255, 154, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 38, 157, 255, 255, 152, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 111, 255, 252, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 19, 255, 255, 44, 0, 0, 0],
            &[0, 0, 8, 99, 11, 0, 0, 0, 0, 0, 107, 255, 247, 12, 0, 0, 0],
            &[
                0, 0, 14, 255, 248, 187, 144, 123, 140, 194, 255, 255, 121, 0, 0, 0, 0,
            ],
            &[
                0, 0, 5, 146, 231, 255, 255, 255, 255, 255, 208, 88, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 33, 58, 66, 54, 20, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¨' / 0xa8
        '¨' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 191, 195, 38, 0, 0, 7, 170, 207, 67, 0, 0, 0, 0],
            &[
                0, 0, 0, 101, 255, 255, 119, 0, 0, 54, 255, 255, 166, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 49, 244, 246, 63, 0, 0, 17, 229, 252, 104, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 13, 15, 0, 0, 0, 0, 5, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '©' / 0xa9
        '©' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 113, 143, 151, 123, 68, 1, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 26, 183, 255, 230, 193, 188, 221, 255, 220, 61, 0, 0, 0, 0,
            ],
            &[
                0, 0, 33, 233, 216, 67, 1, 0, 0, 0, 49, 192, 251, 79, 0, 0, 0,
            ],
            &[
                0, 1, 202, 212, 12, 2, 124, 215, 249, 221, 105, 3, 178, 238, 24, 0, 0,
            ],
            &[
                0, 67, 255, 54, 0, 146, 254, 156, 103, 143, 80, 0, 19, 246, 124, 0, 0,
            ],
            &[
                0, 151, 214, 0, 29, 250, 143, 0, 0, 0, 0, 0, 0, 168, 205, 0, 0,
            ],
            &[
                0, 184, 172, 0, 79, 255, 59, 0, 0, 0, 0, 0, 0, 123, 237, 0, 0,
            ],
            &[
                0, 199, 160, 0, 102, 255, 40, 0, 0, 0, 0, 0, 0, 108, 249, 0, 0,
            ],
            &[
                0, 174, 186, 0, 72, 255, 71, 0, 0, 0, 0, 0, 0, 132, 223, 0, 0,
            ],
            &[
                0, 126, 244, 12, 13, 236, 187, 7, 0, 0, 7, 0, 0, 199, 173, 0, 0,
            ],
            &[
                0, 34, 254, 123, 0, 88, 247, 231, 193, 223, 107, 0, 64, 255, 76, 0, 0,
            ],
            &[
                0, 0, 138, 251, 84, 0, 41, 113, 145, 118, 28, 45, 229, 178, 0, 0, 0,
            ],
            &[
                0, 0, 5, 162, 255, 172, 77, 37, 31, 63, 146, 246, 189, 17, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 84, 190, 253, 255, 255, 255, 205, 103, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 6, 32, 37, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ª' / 0xaa
        'ª' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 78, 109, 112, 73, 2, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 41, 246, 255, 247, 253, 255, 201, 7, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 112, 39, 1, 16, 136, 255, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 36, 255, 135, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 61, 153, 196, 217, 230, 255, 142, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 96, 255, 209, 125, 97, 104, 255, 142, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 208, 239, 10, 0, 0, 53, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 247, 19, 0, 6, 171, 255, 142, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 122, 255, 238, 195, 231, 178, 246, 142, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 87, 153, 152, 84, 0, 99, 70, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '«' / 0xab
        '«' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 88, 110, 0, 0, 0, 34, 154, 15, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 51, 246, 255, 71, 0, 13, 212, 255, 137, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 23, 227, 255, 150, 0, 2, 179, 255, 202, 7, 0, 0, 0,
            ],
            &[
                0, 0, 0, 6, 196, 255, 194, 5, 0, 136, 255, 233, 28, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 156, 255, 226, 22, 0, 92, 255, 250, 61, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 100, 255, 249, 50, 0, 46, 247, 255, 111, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 64, 251, 255, 91, 0, 25, 228, 255, 159, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 245, 47, 0, 51, 246, 255, 107, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 150, 255, 221, 17, 0, 87, 254, 250, 60, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 4, 190, 255, 183, 2, 0, 129, 255, 231, 26, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 221, 250, 61, 0, 1, 172, 255, 132, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 42, 51, 0, 0, 0, 10, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¬' / 0xac
        '¬' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 118, 121, 121, 121, 121, 121, 121, 121, 121, 121, 121, 110, 0, 0, 0,
            ],
            &[
                0, 0, 249, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 110, 112, 112, 112, 112, 112, 112, 112, 112, 114, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 255, 231, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 255, 231, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 255, 231, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 255, 231, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 139, 125, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '®' / 0xae
        '®' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 113, 143, 151, 123, 68, 1, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 26, 183, 255, 230, 193, 188, 221, 255, 220, 61, 0, 0, 0, 0,
            ],
            &[
                0, 0, 33, 233, 216, 67, 1, 0, 0, 0, 49, 192, 251, 79, 0, 0, 0,
            ],
            &[
                0, 1, 202, 212, 12, 78, 238, 238, 223, 169, 32, 3, 178, 238, 24, 0, 0,
            ],
            &[
                0, 67, 255, 54, 0, 84, 255, 87, 115, 228, 195, 0, 19, 246, 124, 0, 0,
            ],
            &[
                0, 151, 214, 0, 0, 84, 255, 12, 0, 127, 249, 0, 0, 168, 205, 0, 0,
            ],
            &[
                0, 184, 172, 0, 0, 84, 255, 12, 29, 200, 209, 0, 0, 123, 237, 0, 0,
            ],
            &[
                0, 199, 160, 0, 0, 84, 255, 246, 255, 220, 42, 0, 0, 108, 249, 0, 0,
            ],
            &[
                0, 174, 186, 0, 0, 84, 255, 66, 169, 224, 10, 0, 0, 132, 223, 0, 0,
            ],
            &[
                0, 126, 244, 12, 0, 84, 255, 12, 25, 242, 137, 0, 0, 199, 173, 0, 0,
            ],
            &[
                0, 34, 254, 123, 0, 84, 255, 12, 0, 118, 250, 44, 64, 255, 76, 0, 0,
            ],
            &[
                0, 0, 138, 251, 84, 34, 105, 4, 0, 6, 104, 108, 229, 178, 0, 0, 0,
            ],
            &[
                0, 0, 5, 162, 255, 172, 77, 37, 31, 63, 146, 246, 189, 17, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 84, 190, 253, 255, 255, 255, 205, 103, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 6, 32, 37, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¯' / 0xaf
        '¯' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 138, 154, 154, 154, 154, 154, 154, 154, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 31, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 71, 80, 80, 80, 80, 80, 80, 80, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '°' / 0xb0
        '°' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 50, 120, 127, 72, 2, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 1, 150, 255, 255, 255, 255, 190, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 105, 255, 173, 47, 35, 129, 255, 157, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 209, 226, 6, 0, 0, 0, 175, 249, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 242, 182, 0, 0, 0, 0, 124, 255, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 217, 223, 4, 0, 0, 0, 163, 253, 15, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 126, 255, 161, 22, 8, 101, 253, 177, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 6, 181, 255, 255, 253, 255, 214, 26, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 1, 77, 143, 149, 98, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '±' / 0xb1
        '±' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 207, 224, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 4, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 135, 142, 142, 142, 142, 247, 255, 144, 142, 142, 142, 142, 5, 0, 0,
            ],
            &[
                0, 0, 242, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 87, 92, 92, 92, 92, 243, 255, 95, 92, 92, 92, 92, 3, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 255, 4, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 154, 166, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 197, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 9, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '²' / 0xb2
        '²' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 74, 196, 248, 248, 211, 97, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 103, 255, 232, 169, 152, 234, 255, 96, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 11, 123, 11, 0, 0, 42, 255, 193, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 253, 204, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 75, 255, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 24, 227, 235, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 23, 214, 246, 66, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 24, 213, 246, 69, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 27, 217, 243, 66, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 30, 220, 239, 57, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 195, 255, 245, 231, 231, 231, 231, 231, 66, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 160, 196, 196, 196, 196, 196, 196, 196, 56, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '³' / 0xb3
        '³' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 109, 213, 252, 246, 204, 79, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 125, 255, 219, 153, 173, 247, 253, 61, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 6, 86, 2, 0, 0, 111, 255, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 90, 255, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 45, 211, 253, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 60, 244, 250, 255, 251, 92, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 33, 135, 141, 175, 247, 241, 67, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 213, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 249, 248, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 8, 0, 0, 0, 69, 255, 208, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 208, 239, 183, 151, 174, 250, 254, 74, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 78, 177, 225, 244, 220, 177, 56, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '´' / 0xb4
        '´' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 7, 164, 174, 163, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 88, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 34, 245, 255, 116, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 194, 255, 123, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 49, 163, 92, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'µ' / 0xb5
        'µ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 117, 120, 54, 0, 0, 0, 0, 0, 0, 47, 120, 120, 4, 0, 0],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 255, 255, 131, 0, 0, 0, 0, 0, 0, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 255, 255, 233, 0, 0, 0, 0, 0, 0, 126, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 255, 255, 254, 0, 0, 0, 0, 0, 0, 175, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 255, 255, 255, 0, 0, 0, 0, 0, 24, 247, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 255, 255, 255, 98, 0, 0, 0, 29, 198, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 255, 255, 255, 255, 224, 189, 205, 250, 246, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 255, 255, 255, 255, 255, 199, 63, 5, 249, 255, 10, 0, 0,
            ],
            &[0, 0, 249, 255, 116, 27, 59, 55, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 119, 122, 55, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¶' / 0xb6
        '¶' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 22, 93, 123, 133, 133, 133, 133, 133, 133, 133, 73, 0, 0,
            ],
            &[
                0, 0, 3, 134, 252, 255, 255, 255, 255, 255, 254, 253, 253, 255, 141, 0, 0,
            ],
            &[
                0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 59, 0, 3, 255, 141, 0, 0,
            ],
            &[
                0, 15, 248, 255, 255, 255, 255, 255, 255, 255, 59, 0, 3, 255, 141, 0, 0,
            ],
            &[
                0, 87, 255, 255, 255, 255, 255, 255, 255, 255, 59, 0, 3, 255, 141, 0, 0,
            ],
            &[
                0, 132, 255, 255, 255, 255, 255, 255, 255, 255, 59, 0, 3, 255, 141, 0, 0,
            ],
            &[
                0, 151, 255, 255, 255, 255, 255, 255, 255, 255, 59, 0, 3, 255, 141, 0, 0,
            ],
            &[
                0, 160, 255, 255, 255, 255, 255, 255, 255, 255, 59, 0, 3, 255, 141, 0, 0,
            ],
            &[
                0, 143, 255, 255, 255, 255, 255, 255, 255, 255, 59, 0, 3, 255, 141, 0, 0,
            ],
            &[
                0, 117, 255, 255, 255, 255, 255, 255, 255, 255, 59, 0, 3, 255, 141, 0, 0,
            ],
            &[
                0, 51, 255, 255, 255, 255, 255, 255, 255, 255, 59, 0, 3, 255, 141, 0, 0,
            ],
            &[
                0, 1, 205, 255, 255, 255, 255, 255, 255, 255, 59, 0, 3, 255, 141, 0, 0,
            ],
            &[
                0, 0, 47, 233, 255, 255, 255, 255, 255, 255, 59, 0, 3, 255, 141, 0, 0,
            ],
            &[
                0, 0, 0, 26, 132, 196, 217, 205, 202, 255, 59, 0, 3, 255, 141, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 255, 59, 0, 3, 255, 141, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 255, 59, 0, 3, 255, 141, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 255, 59, 0, 3, 255, 141, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 255, 59, 0, 3, 255, 141, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 255, 59, 0, 3, 255, 141, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 255, 59, 0, 3, 255, 141, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 255, 59, 0, 3, 255, 141, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 255, 59, 0, 3, 255, 141, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 255, 59, 0, 3, 255, 141, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 85, 255, 59, 0, 3, 255, 141, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 36, 109, 25, 0, 1, 109, 60, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '·' / 0xb7
        '·' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 207, 224, 79, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 134, 255, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 114, 255, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 112, 129, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¸' / 0xb8
        '¸' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 18, 34, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 202, 201, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 68, 255, 180, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 62, 161, 241, 250, 71, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 80, 255, 168, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 6, 32, 140, 255, 143, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 160, 255, 255, 255, 211, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 118, 102, 65, 1, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¹' / 0xb9
        '¹' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 176, 201, 79, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 152, 252, 255, 255, 101, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 199, 255, 218, 136, 255, 101, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 63, 114, 6, 89, 255, 101, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 101, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 101, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 101, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 101, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 101, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 255, 101, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 21, 166, 183, 255, 255, 255, 183, 168, 45, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 27, 196, 196, 255, 255, 255, 196, 196, 56, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'º' / 0xba
        'º' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 41, 98, 113, 61, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 130, 253, 254, 252, 255, 177, 6, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 66, 255, 171, 16, 15, 137, 255, 122, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 166, 252, 24, 0, 0, 2, 231, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 202, 227, 0, 0, 0, 0, 194, 251, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 223, 0, 0, 0, 0, 186, 249, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 172, 248, 11, 0, 0, 0, 222, 216, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 89, 255, 130, 0, 0, 92, 255, 133, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 1, 172, 255, 226, 219, 254, 203, 13, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 83, 143, 148, 96, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '»' / 0xbb
        '»' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 135, 81, 0, 0, 0, 79, 131, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 76, 255, 244, 47, 0, 19, 237, 255, 102, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 142, 255, 225, 22, 0, 72, 253, 250, 62, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 4, 191, 255, 197, 6, 0, 123, 255, 235, 32, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 226, 255, 160, 0, 1, 176, 255, 210, 12, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 52, 252, 255, 104, 0, 14, 222, 255, 165, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 107, 255, 247, 54, 0, 50, 246, 255, 107, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 61, 250, 255, 91, 0, 22, 226, 255, 155, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 28, 233, 255, 135, 0, 5, 194, 255, 196, 6, 0, 0, 0, 0,
            ],
            &[
                0, 0, 7, 202, 255, 177, 2, 0, 151, 255, 226, 23, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 74, 251, 211, 13, 0, 23, 229, 246, 50, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 57, 33, 0, 0, 0, 21, 67, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¼' / 0xbc
        '¼' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 71, 49, 0, 0, 0, 0, 0, 0, 0, 59, 71, 15, 0, 0],
            &[
                8, 123, 243, 255, 178, 0, 0, 0, 0, 0, 0, 45, 254, 220, 4, 0, 0,
            ],
            &[
                169, 255, 222, 255, 178, 0, 0, 0, 0, 0, 0, 176, 255, 93, 0, 0, 0,
            ],
            &[
                47, 144, 23, 255, 178, 0, 0, 0, 0, 0, 54, 255, 214, 2, 0, 0, 0,
            ],
            &[0, 0, 13, 255, 178, 0, 0, 0, 0, 0, 185, 255, 84, 0, 0, 0, 0],
            &[0, 0, 14, 255, 178, 0, 0, 0, 0, 62, 255, 206, 1, 0, 0, 0, 0],
            &[0, 0, 14, 255, 178, 0, 0, 0, 0, 195, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 14, 255, 178, 0, 0, 0, 72, 255, 199, 0, 0, 0, 0, 0, 0],
            &[
                10, 87, 124, 255, 255, 101, 72, 0, 203, 255, 67, 0, 0, 0, 0, 0, 0,
            ],
            &[
                34, 255, 255, 255, 255, 255, 226, 81, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                3, 23, 24, 46, 39, 23, 22, 211, 255, 58, 0, 0, 93, 108, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 90, 255, 182, 0, 0, 94, 255, 255, 24, 0, 0],
            &[
                0, 0, 0, 0, 0, 4, 218, 254, 51, 0, 33, 241, 249, 255, 24, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 99, 255, 173, 0, 3, 197, 228, 168, 255, 24, 0, 0,
            ],
            &[
                0, 0, 0, 0, 6, 225, 253, 43, 0, 125, 255, 72, 161, 255, 24, 0, 0,
            ],
            &[
                0, 0, 0, 0, 109, 255, 165, 0, 55, 251, 151, 0, 162, 255, 24, 0, 0,
            ],
            &[
                0, 0, 0, 10, 231, 251, 37, 0, 208, 255, 169, 158, 220, 255, 167, 78, 0,
            ],
            &[
                0, 0, 0, 118, 255, 156, 0, 0, 205, 228, 228, 228, 245, 255, 231, 113, 0,
            ],
            &[0, 0, 14, 236, 249, 30, 0, 0, 0, 0, 0, 0, 162, 255, 24, 0, 0],
            &[0, 0, 127, 255, 148, 0, 0, 0, 0, 0, 0, 0, 162, 255, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '½' / 0xbd
        '½' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 71, 49, 0, 0, 0, 0, 0, 0, 0, 59, 71, 15, 0, 0],
            &[
                8, 123, 243, 255, 178, 0, 0, 0, 0, 0, 0, 45, 254, 220, 4, 0, 0,
            ],
            &[
                169, 255, 222, 255, 178, 0, 0, 0, 0, 0, 0, 176, 255, 93, 0, 0, 0,
            ],
            &[
                47, 144, 23, 255, 178, 0, 0, 0, 0, 0, 54, 255, 214, 2, 0, 0, 0,
            ],
            &[0, 0, 13, 255, 178, 0, 0, 0, 0, 0, 185, 255, 84, 0, 0, 0, 0],
            &[0, 0, 14, 255, 178, 0, 0, 0, 0, 62, 255, 206, 1, 0, 0, 0, 0],
            &[0, 0, 14, 255, 178, 0, 0, 0, 0, 195, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 14, 255, 178, 0, 0, 0, 72, 255, 199, 0, 0, 0, 0, 0, 0],
            &[
                10, 87, 124, 255, 255, 101, 72, 0, 203, 255, 67, 0, 0, 0, 0, 0, 0,
            ],
            &[
                34, 255, 255, 255, 255, 255, 226, 81, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                3, 23, 24, 46, 39, 23, 22, 211, 255, 58, 55, 118, 134, 81, 1, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 90, 255, 182, 148, 255, 255, 255, 255, 170, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 4, 218, 254, 51, 80, 142, 28, 34, 222, 255, 39, 0,
            ],
            &[0, 0, 0, 0, 0, 99, 255, 173, 0, 0, 0, 0, 0, 159, 255, 53, 0],
            &[0, 0, 0, 0, 6, 225, 253, 43, 0, 0, 0, 0, 22, 232, 231, 7, 0],
            &[
                0, 0, 0, 0, 109, 255, 165, 0, 0, 0, 0, 18, 204, 251, 70, 0, 0,
            ],
            &[
                0, 0, 0, 10, 231, 251, 37, 0, 0, 0, 22, 208, 249, 80, 0, 0, 0,
            ],
            &[
                0, 0, 0, 118, 255, 156, 0, 0, 0, 28, 216, 242, 68, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 236, 249, 30, 0, 0, 7, 224, 255, 211, 172, 172, 172, 85, 0,
            ],
            &[
                0, 0, 127, 255, 148, 0, 0, 0, 16, 255, 255, 255, 255, 255, 255, 127, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¾' / 0xbe
        '¾' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 83, 122, 103, 27, 0, 0, 0, 0, 0, 0, 59, 71, 15, 0, 0],
            &[
                52, 236, 255, 255, 255, 247, 63, 0, 0, 0, 0, 45, 254, 220, 4, 0, 0,
            ],
            &[
                3, 148, 72, 20, 129, 255, 183, 0, 0, 0, 0, 176, 255, 93, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 67, 255, 196, 0, 0, 0, 54, 255, 214, 2, 0, 0, 0],
            &[
                0, 26, 120, 133, 222, 242, 75, 0, 0, 0, 185, 255, 84, 0, 0, 0, 0,
            ],
            &[
                0, 55, 255, 255, 255, 219, 63, 0, 0, 62, 255, 206, 1, 0, 0, 0, 0,
            ],
            &[0, 0, 3, 24, 89, 253, 233, 1, 0, 195, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 243, 253, 7, 72, 255, 199, 0, 0, 0, 0, 0, 0],
            &[
                63, 172, 102, 82, 163, 255, 196, 0, 203, 255, 67, 0, 0, 0, 0, 0, 0,
            ],
            &[
                54, 244, 255, 255, 255, 196, 35, 81, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 9, 47, 54, 25, 0, 2, 211, 255, 58, 0, 0, 93, 108, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 90, 255, 182, 0, 0, 94, 255, 255, 24, 0, 0],
            &[
                0, 0, 0, 0, 0, 4, 218, 254, 51, 0, 33, 241, 249, 255, 24, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 99, 255, 173, 0, 3, 197, 228, 168, 255, 24, 0, 0,
            ],
            &[
                0, 0, 0, 0, 6, 225, 253, 43, 0, 125, 255, 72, 161, 255, 24, 0, 0,
            ],
            &[
                0, 0, 0, 0, 109, 255, 165, 0, 55, 251, 151, 0, 162, 255, 24, 0, 0,
            ],
            &[
                0, 0, 0, 10, 231, 251, 37, 0, 208, 255, 169, 158, 220, 255, 167, 78, 0,
            ],
            &[
                0, 0, 0, 118, 255, 156, 0, 0, 205, 228, 228, 228, 245, 255, 231, 113, 0,
            ],
            &[0, 0, 14, 236, 249, 30, 0, 0, 0, 0, 0, 0, 162, 255, 24, 0, 0],
            &[0, 0, 127, 255, 148, 0, 0, 0, 0, 0, 0, 0, 162, 255, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¿' / 0xbf
        '¿' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 78, 108, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 84, 255, 255, 177, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 115, 255, 255, 208, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 42, 225, 246, 114, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 167, 212, 70, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 232, 255, 58, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 106, 255, 244, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 107, 252, 255, 120, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 143, 255, 255, 136, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 169, 255, 250, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 118, 255, 250, 64, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 232, 255, 132, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 255, 255, 56, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 255, 31, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 255, 255, 91, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 194, 255, 236, 81, 1, 0, 0, 17, 91, 189, 122, 0, 0, 0,
            ],
            &[
                0, 0, 0, 41, 236, 255, 255, 247, 221, 231, 254, 255, 255, 228, 4, 0, 0,
            ],
            &[
                0, 0, 0, 0, 29, 162, 243, 255, 255, 255, 255, 234, 151, 57, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 7, 42, 62, 54, 31, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'À' / 0xc0
        'À' => &[
            &[0, 0, 0, 0, 12, 215, 255, 230, 11, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 231, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 40, 230, 253, 64, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 33, 195, 186, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 91, 91, 39, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 255, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 255, 255, 241, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 249, 223, 255, 76, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 113, 255, 196, 151, 255, 159, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 196, 255, 127, 79, 255, 237, 4, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 25, 253, 255, 48, 12, 247, 255, 69, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 106, 255, 224, 0, 0, 180, 255, 153, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 189, 255, 145, 0, 0, 101, 255, 233, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 19, 252, 255, 65, 0, 0, 23, 253, 255, 63, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 98, 255, 238, 4, 0, 0, 0, 198, 255, 146, 0, 0, 0, 0],
            &[
                0, 0, 0, 181, 255, 181, 18, 18, 18, 18, 137, 255, 228, 1, 0, 0, 0,
            ],
            &[
                0, 0, 15, 249, 255, 255, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 198, 22, 22, 22, 22, 22, 22, 153, 255, 222, 0, 0, 0,
            ],
            &[
                0, 10, 246, 255, 100, 0, 0, 0, 0, 0, 0, 56, 255, 255, 50, 0, 0,
            ],
            &[
                0, 83, 255, 253, 23, 0, 0, 0, 0, 0, 0, 1, 231, 255, 133, 0, 0,
            ],
            &[
                0, 166, 255, 197, 0, 0, 0, 0, 0, 0, 0, 0, 153, 255, 216, 0, 0,
            ],
            &[
                7, 242, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0, 74, 255, 255, 44, 0,
            ],
            &[
                76, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 7, 243, 255, 127, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 124, 124, 66, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Á' / 0xc1
        'Á' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 18, 237, 255, 207, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 158, 255, 224, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 72, 255, 227, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 189, 190, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 91, 91, 39, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 255, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 255, 255, 241, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 249, 223, 255, 76, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 113, 255, 196, 151, 255, 159, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 196, 255, 127, 79, 255, 237, 4, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 25, 253, 255, 48, 12, 247, 255, 69, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 106, 255, 224, 0, 0, 180, 255, 153, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 189, 255, 145, 0, 0, 101, 255, 233, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 19, 252, 255, 65, 0, 0, 23, 253, 255, 63, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 98, 255, 238, 4, 0, 0, 0, 198, 255, 146, 0, 0, 0, 0],
            &[
                0, 0, 0, 181, 255, 181, 18, 18, 18, 18, 137, 255, 228, 1, 0, 0, 0,
            ],
            &[
                0, 0, 15, 249, 255, 255, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 198, 22, 22, 22, 22, 22, 22, 153, 255, 222, 0, 0, 0,
            ],
            &[
                0, 10, 246, 255, 100, 0, 0, 0, 0, 0, 0, 56, 255, 255, 50, 0, 0,
            ],
            &[
                0, 83, 255, 253, 23, 0, 0, 0, 0, 0, 0, 1, 231, 255, 133, 0, 0,
            ],
            &[
                0, 166, 255, 197, 0, 0, 0, 0, 0, 0, 0, 0, 153, 255, 216, 0, 0,
            ],
            &[
                7, 242, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0, 74, 255, 255, 44, 0,
            ],
            &[
                76, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 7, 243, 255, 127, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 124, 124, 47, 0, 0, 0, 0],
        ],
        // letter: 'Â' / 0xc2
        'Â' => &[
            &[0, 0, 0, 0, 0, 7, 208, 255, 255, 230, 19, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 163, 255, 184, 166, 255, 194, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 130, 255, 179, 8, 2, 158, 255, 170, 3, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 64, 212, 138, 3, 0, 0, 0, 113, 212, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 91, 91, 39, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 255, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 255, 255, 241, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 249, 223, 255, 76, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 113, 255, 196, 151, 255, 159, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 196, 255, 127, 79, 255, 237, 4, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 25, 253, 255, 48, 12, 247, 255, 69, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 106, 255, 224, 0, 0, 180, 255, 153, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 189, 255, 145, 0, 0, 101, 255, 233, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 19, 252, 255, 65, 0, 0, 23, 253, 255, 63, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 98, 255, 238, 4, 0, 0, 0, 198, 255, 146, 0, 0, 0, 0],
            &[
                0, 0, 0, 181, 255, 181, 18, 18, 18, 18, 137, 255, 228, 1, 0, 0, 0,
            ],
            &[
                0, 0, 15, 249, 255, 255, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 198, 22, 22, 22, 22, 22, 22, 153, 255, 222, 0, 0, 0,
            ],
            &[
                0, 10, 246, 255, 100, 0, 0, 0, 0, 0, 0, 56, 255, 255, 50, 0, 0,
            ],
            &[
                0, 83, 255, 253, 23, 0, 0, 0, 0, 0, 0, 1, 231, 255, 133, 0, 0,
            ],
            &[
                0, 166, 255, 197, 0, 0, 0, 0, 0, 0, 0, 0, 153, 255, 216, 0, 0,
            ],
            &[
                7, 242, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0, 74, 255, 255, 44, 0,
            ],
            &[
                76, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 7, 243, 255, 127, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 124, 124, 57, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ã' / 0xc3
        'Ã' => &[
            &[0, 0, 0, 0, 19, 121, 141, 70, 0, 0, 0, 89, 147, 0, 0, 0, 0],
            &[
                0, 0, 0, 12, 222, 255, 255, 255, 190, 66, 62, 228, 178, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 114, 255, 134, 94, 197, 255, 255, 255, 254, 61, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 142, 170, 0, 0, 0, 85, 177, 180, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 91, 91, 39, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 255, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 255, 255, 241, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 249, 223, 255, 76, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 113, 255, 196, 151, 255, 159, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 196, 255, 127, 79, 255, 237, 4, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 25, 253, 255, 48, 12, 247, 255, 69, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 106, 255, 224, 0, 0, 180, 255, 153, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 189, 255, 145, 0, 0, 101, 255, 233, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 19, 252, 255, 65, 0, 0, 23, 253, 255, 63, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 98, 255, 238, 4, 0, 0, 0, 198, 255, 146, 0, 0, 0, 0],
            &[
                0, 0, 0, 181, 255, 181, 18, 18, 18, 18, 137, 255, 228, 1, 0, 0, 0,
            ],
            &[
                0, 0, 15, 249, 255, 255, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 198, 22, 22, 22, 22, 22, 22, 153, 255, 222, 0, 0, 0,
            ],
            &[
                0, 10, 246, 255, 100, 0, 0, 0, 0, 0, 0, 56, 255, 255, 50, 0, 0,
            ],
            &[
                0, 83, 255, 253, 23, 0, 0, 0, 0, 0, 0, 1, 231, 255, 133, 0, 0,
            ],
            &[
                0, 166, 255, 197, 0, 0, 0, 0, 0, 0, 0, 0, 153, 255, 216, 0, 0,
            ],
            &[
                7, 242, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0, 74, 255, 255, 44, 0,
            ],
            &[
                76, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 7, 243, 255, 127, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ä' / 0xc4
        'Ä' => &[
            &[0, 0, 0, 15, 143, 143, 16, 0, 0, 2, 124, 156, 36, 0, 0, 0, 0],
            &[
                0, 0, 0, 104, 255, 255, 108, 0, 0, 57, 255, 255, 155, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 76, 255, 255, 79, 0, 0, 33, 251, 255, 125, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 54, 53, 0, 0, 0, 0, 36, 68, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 91, 91, 39, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 255, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 255, 255, 241, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 249, 223, 255, 76, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 113, 255, 196, 151, 255, 159, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 196, 255, 127, 79, 255, 237, 4, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 25, 253, 255, 48, 12, 247, 255, 69, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 106, 255, 224, 0, 0, 180, 255, 153, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 189, 255, 145, 0, 0, 101, 255, 233, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 19, 252, 255, 65, 0, 0, 23, 253, 255, 63, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 98, 255, 238, 4, 0, 0, 0, 198, 255, 146, 0, 0, 0, 0],
            &[
                0, 0, 0, 181, 255, 181, 18, 18, 18, 18, 137, 255, 228, 1, 0, 0, 0,
            ],
            &[
                0, 0, 15, 249, 255, 255, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 198, 22, 22, 22, 22, 22, 22, 153, 255, 222, 0, 0, 0,
            ],
            &[
                0, 10, 246, 255, 100, 0, 0, 0, 0, 0, 0, 56, 255, 255, 50, 0, 0,
            ],
            &[
                0, 83, 255, 253, 23, 0, 0, 0, 0, 0, 0, 1, 231, 255, 133, 0, 0,
            ],
            &[
                0, 166, 255, 197, 0, 0, 0, 0, 0, 0, 0, 0, 153, 255, 216, 0, 0,
            ],
            &[
                7, 242, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0, 74, 255, 255, 44, 0,
            ],
            &[
                76, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 7, 243, 255, 127, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Å' / 0xc5
        'Å' => &[
            &[0, 0, 0, 0, 0, 202, 210, 0, 0, 166, 248, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 199, 215, 0, 0, 171, 246, 2, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 112, 255, 202, 187, 253, 157, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 1, 105, 189, 195, 124, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 91, 91, 39, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 255, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 255, 255, 241, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 249, 223, 255, 76, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 113, 255, 196, 151, 255, 159, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 196, 255, 127, 79, 255, 237, 4, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 25, 253, 255, 48, 12, 247, 255, 69, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 106, 255, 224, 0, 0, 180, 255, 153, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 189, 255, 145, 0, 0, 101, 255, 233, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 19, 252, 255, 65, 0, 0, 23, 253, 255, 63, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 98, 255, 238, 4, 0, 0, 0, 198, 255, 146, 0, 0, 0, 0],
            &[
                0, 0, 0, 181, 255, 181, 18, 18, 18, 18, 137, 255, 228, 1, 0, 0, 0,
            ],
            &[
                0, 0, 15, 249, 255, 255, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 198, 22, 22, 22, 22, 22, 22, 153, 255, 222, 0, 0, 0,
            ],
            &[
                0, 10, 246, 255, 100, 0, 0, 0, 0, 0, 0, 56, 255, 255, 50, 0, 0,
            ],
            &[
                0, 83, 255, 253, 23, 0, 0, 0, 0, 0, 0, 1, 231, 255, 133, 0, 0,
            ],
            &[
                0, 166, 255, 197, 0, 0, 0, 0, 0, 0, 0, 0, 153, 255, 216, 0, 0,
            ],
            &[
                7, 242, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0, 74, 255, 255, 44, 0,
            ],
            &[
                76, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 7, 243, 255, 127, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 120, 255, 186, 171, 251, 165, 0, 0, 0, 0, 0, 0,
            ],
        ],
        // letter: 'Æ' / 0xc6
        'Æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 71, 71, 71, 71, 71, 71, 71, 71, 71, 47, 0],
            &[
                0, 0, 0, 0, 0, 80, 255, 255, 255, 255, 255, 255, 255, 255, 255, 169, 0,
            ],
            &[
                0, 0, 0, 0, 0, 137, 255, 241, 241, 255, 249, 225, 225, 225, 225, 149, 0,
            ],
            &[
                0, 0, 0, 0, 0, 195, 255, 102, 133, 255, 204, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 5, 246, 255, 48, 133, 255, 204, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 53, 255, 245, 4, 133, 255, 204, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 111, 255, 195, 0, 133, 255, 204, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 168, 255, 140, 0, 133, 255, 204, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 225, 255, 86, 0, 133, 255, 212, 38, 38, 38, 38, 5, 0,
            ],
            &[
                0, 0, 0, 27, 255, 255, 31, 0, 133, 255, 255, 255, 255, 255, 255, 37, 0,
            ],
            &[
                0, 0, 0, 84, 255, 233, 0, 0, 133, 255, 254, 251, 251, 251, 251, 37, 0,
            ],
            &[
                0, 0, 0, 142, 255, 178, 0, 0, 133, 255, 204, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 199, 255, 255, 253, 253, 255, 255, 204, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 7, 249, 255, 255, 255, 255, 255, 255, 204, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 58, 255, 255, 38, 22, 22, 155, 255, 204, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 115, 255, 216, 0, 0, 0, 133, 255, 204, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 172, 255, 162, 0, 0, 0, 133, 255, 204, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 230, 255, 107, 0, 0, 0, 133, 255, 213, 40, 40, 40, 40, 27, 0,
            ],
            &[
                0, 31, 255, 255, 53, 0, 0, 0, 133, 255, 255, 255, 255, 255, 255, 169, 0,
            ],
            &[
                0, 89, 255, 248, 6, 0, 0, 0, 133, 255, 255, 255, 255, 255, 255, 169, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ç' / 0xc7
        'Ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 46, 105, 129, 132, 111, 65, 6, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 69, 212, 255, 255, 255, 255, 255, 255, 241, 120, 0, 0,
            ],
            &[
                0, 0, 0, 0, 114, 255, 255, 255, 206, 168, 172, 202, 247, 255, 108, 0, 0,
            ],
            &[
                0, 0, 0, 83, 255, 255, 202, 44, 0, 0, 0, 0, 15, 100, 10, 0, 0,
            ],
            &[0, 0, 11, 231, 255, 210, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 98, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 185, 255, 214, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 239, 255, 143, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 255, 255, 99, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 255, 255, 72, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 107, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 231, 255, 159, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 225, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 255, 255, 90, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 226, 255, 234, 42, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 72, 253, 255, 238, 125, 38, 7, 8, 34, 90, 167, 2, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 249, 255, 255, 255, 254, 255, 255, 255, 255, 3, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 43, 169, 247, 255, 255, 255, 255, 234, 162, 1, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 5, 171, 255, 81, 31, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 11, 236, 225, 69, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 27, 141, 220, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 247, 243, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 10, 19, 81, 254, 219, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 240, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 114, 110, 82, 12, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'È' / 0xc8
        'È' => &[
            &[0, 0, 0, 0, 2, 184, 255, 249, 34, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 208, 255, 185, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 209, 255, 103, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 16, 177, 204, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 30, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 64, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 225, 225, 225, 225, 225, 225, 225, 225, 204, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 38, 38, 38, 38, 38, 38, 38, 38, 15, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 40, 40, 40, 40, 40, 40, 40, 40, 37, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 124, 124, 86, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'É' / 0xc9
        'É' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 3, 211, 255, 231, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 116, 255, 241, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 40, 246, 242, 62, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 156, 204, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 30, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 64, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 225, 225, 225, 225, 225, 225, 225, 225, 204, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 38, 38, 38, 38, 38, 38, 38, 38, 15, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 40, 40, 40, 40, 40, 40, 40, 40, 37, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 52, 124, 124, 67, 0, 0, 0, 0],
        ],
        // letter: 'Ê' / 0xca
        'Ê' => &[
            &[0, 0, 0, 0, 0, 0, 173, 255, 255, 247, 44, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 122, 255, 212, 138, 255, 221, 21, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 90, 254, 207, 21, 0, 120, 254, 201, 13, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 35, 207, 164, 11, 0, 0, 0, 81, 210, 140, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 30, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 64, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 225, 225, 225, 225, 225, 225, 225, 225, 204, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 38, 38, 38, 38, 38, 38, 38, 38, 15, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 40, 40, 40, 40, 40, 40, 40, 40, 37, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 124, 124, 77, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ë' / 0xcb
        'Ë' => &[
            &[0, 0, 0, 3, 127, 155, 33, 0, 0, 0, 100, 163, 56, 0, 0, 0, 0],
            &[
                0, 0, 0, 63, 255, 255, 149, 0, 0, 16, 255, 255, 196, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 37, 252, 255, 120, 0, 0, 6, 237, 255, 167, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 38, 66, 2, 0, 0, 0, 24, 75, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 30, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 64, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 225, 225, 225, 225, 225, 225, 225, 225, 204, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 38, 38, 38, 38, 38, 38, 38, 38, 15, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 40, 40, 40, 40, 40, 40, 40, 40, 37, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ì' / 0xcc
        'Ì' => &[
            &[0, 0, 0, 0, 12, 215, 255, 230, 11, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 231, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 40, 230, 253, 64, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 33, 195, 186, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 16, 0, 0, 0],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[
                0, 0, 3, 113, 136, 159, 191, 255, 255, 204, 163, 140, 117, 23, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 255, 255, 74, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 5, 184, 207, 230, 251, 255, 255, 253, 234, 211, 189, 40, 0, 0, 0,
            ],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 124, 124, 66, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Í' / 0xcd
        'Í' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 18, 237, 255, 207, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 158, 255, 224, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 72, 255, 227, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 189, 190, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 16, 0, 0, 0],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[
                0, 0, 3, 113, 136, 159, 191, 255, 255, 204, 163, 140, 117, 23, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 255, 255, 74, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 5, 184, 207, 230, 251, 255, 255, 253, 234, 211, 189, 40, 0, 0, 0,
            ],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 124, 124, 47, 0, 0, 0, 0],
        ],
        // letter: 'Î' / 0xce
        'Î' => &[
            &[0, 0, 0, 0, 0, 7, 208, 255, 255, 230, 19, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 163, 255, 184, 166, 255, 194, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 130, 255, 179, 8, 2, 158, 255, 170, 3, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 64, 212, 138, 3, 0, 0, 0, 113, 212, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 16, 0, 0, 0],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[
                0, 0, 3, 113, 136, 159, 191, 255, 255, 204, 163, 140, 117, 23, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 255, 255, 74, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 5, 184, 207, 230, 251, 255, 255, 253, 234, 211, 189, 40, 0, 0, 0,
            ],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 124, 124, 57, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ï' / 0xcf
        'Ï' => &[
            &[0, 0, 0, 15, 143, 143, 16, 0, 0, 2, 124, 156, 36, 0, 0, 0, 0],
            &[
                0, 0, 0, 104, 255, 255, 108, 0, 0, 57, 255, 255, 155, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 76, 255, 255, 79, 0, 0, 33, 251, 255, 125, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 54, 53, 0, 0, 0, 0, 36, 68, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 16, 0, 0, 0],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[
                0, 0, 3, 113, 136, 159, 191, 255, 255, 204, 163, 140, 117, 23, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 255, 255, 74, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 5, 184, 207, 230, 251, 255, 255, 253, 234, 211, 189, 40, 0, 0, 0,
            ],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ð' / 0xd0
        'Ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 71, 71, 71, 70, 61, 34, 1, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 233, 145, 26, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 237, 211, 214, 238, 255, 255, 255, 240, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 25, 120, 242, 255, 251, 68, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 43, 231, 255, 228, 11, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 82, 255, 255, 110, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 214, 255, 199, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 138, 255, 252, 7, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 94, 255, 255, 40, 0,
            ],
            &[
                149, 149, 255, 255, 255, 149, 149, 132, 0, 0, 0, 0, 73, 255, 255, 57, 0,
            ],
            &[
                255, 255, 255, 255, 255, 255, 255, 226, 0, 0, 0, 0, 63, 255, 255, 61, 0,
            ],
            &[
                133, 133, 255, 255, 255, 133, 133, 118, 0, 0, 0, 0, 77, 255, 255, 49, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 109, 255, 255, 20, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 165, 255, 229, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 13, 244, 255, 157, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 145, 255, 255, 55, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 2, 121, 255, 255, 167, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 162, 27, 35, 57, 114, 210, 255, 255, 207, 14, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 164, 15, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 252, 238, 202, 140, 46, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ñ' / 0xd1
        'Ñ' => &[
            &[0, 0, 0, 0, 9, 111, 144, 84, 2, 0, 0, 66, 165, 5, 0, 0, 0],
            &[
                0, 0, 0, 2, 197, 255, 255, 255, 208, 78, 52, 208, 212, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 79, 255, 157, 89, 178, 255, 255, 255, 255, 94, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 114, 194, 3, 0, 0, 67, 170, 186, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 71, 11, 0, 0, 0, 0, 0, 2, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 255, 113, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 222, 2, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 239, 221, 255, 81, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 252, 113, 255, 192, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 32, 241, 255, 49, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 37, 142, 255, 160, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 50, 33, 252, 248, 23, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 57, 0, 174, 255, 128, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 61, 255, 232, 7, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 205, 255, 96, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 93, 255, 207, 0, 8, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 6, 230, 255, 63, 5, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 124, 255, 175, 0, 254, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 20, 247, 253, 34, 244, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 155, 255, 143, 230, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 44, 255, 241, 225, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 186, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 74, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 1, 216, 255, 255, 79, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ò' / 0xd2
        'Ò' => &[
            &[0, 0, 0, 0, 12, 215, 255, 230, 11, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 231, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 40, 230, 253, 64, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 33, 195, 186, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 107, 143, 143, 115, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 2, 143, 252, 255, 255, 255, 255, 255, 173, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 228, 169, 153, 202, 255, 255, 203, 8, 0, 0, 0,
            ],
            &[
                0, 0, 78, 255, 255, 156, 9, 0, 0, 0, 86, 252, 255, 134, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 205, 4, 0, 0, 0, 0, 0, 143, 255, 243, 15, 0, 0,
            ],
            &[
                0, 36, 255, 255, 96, 0, 0, 0, 0, 0, 0, 36, 255, 255, 89, 0, 0,
            ],
            &[
                0, 94, 255, 255, 19, 0, 0, 0, 0, 0, 0, 0, 223, 255, 157, 0, 0,
            ],
            &[
                0, 147, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 176, 255, 198, 0, 0,
            ],
            &[
                0, 170, 255, 201, 0, 0, 0, 0, 0, 0, 0, 0, 147, 255, 229, 0, 0,
            ],
            &[
                0, 186, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 242, 0, 0,
            ],
            &[
                0, 197, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 248, 0, 0,
            ],
            &[
                0, 187, 255, 193, 0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 237, 0, 0,
            ],
            &[
                0, 171, 255, 207, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 221, 0, 0,
            ],
            &[
                0, 135, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 192, 255, 183, 0, 0,
            ],
            &[
                0, 90, 255, 255, 39, 0, 0, 0, 0, 0, 0, 3, 237, 255, 136, 0, 0,
            ],
            &[
                0, 19, 252, 255, 127, 0, 0, 0, 0, 0, 0, 70, 255, 255, 62, 0, 0,
            ],
            &[
                0, 0, 174, 255, 236, 28, 0, 0, 0, 0, 4, 200, 255, 216, 2, 0, 0,
            ],
            &[
                0, 0, 44, 248, 255, 214, 64, 3, 0, 38, 181, 255, 255, 81, 0, 0, 0,
            ],
            &[
                0, 0, 0, 91, 251, 255, 255, 249, 242, 255, 255, 255, 131, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 58, 193, 254, 255, 255, 255, 208, 83, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 22, 53, 57, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 124, 124, 66, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ó' / 0xd3
        'Ó' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 18, 237, 255, 207, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 158, 255, 224, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 72, 255, 227, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 189, 190, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 107, 143, 143, 115, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 2, 143, 252, 255, 255, 255, 255, 255, 173, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 228, 169, 153, 202, 255, 255, 203, 8, 0, 0, 0,
            ],
            &[
                0, 0, 78, 255, 255, 156, 9, 0, 0, 0, 86, 252, 255, 134, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 205, 4, 0, 0, 0, 0, 0, 143, 255, 243, 15, 0, 0,
            ],
            &[
                0, 36, 255, 255, 96, 0, 0, 0, 0, 0, 0, 36, 255, 255, 89, 0, 0,
            ],
            &[
                0, 94, 255, 255, 19, 0, 0, 0, 0, 0, 0, 0, 223, 255, 157, 0, 0,
            ],
            &[
                0, 147, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 176, 255, 198, 0, 0,
            ],
            &[
                0, 170, 255, 201, 0, 0, 0, 0, 0, 0, 0, 0, 147, 255, 229, 0, 0,
            ],
            &[
                0, 186, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 242, 0, 0,
            ],
            &[
                0, 197, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 248, 0, 0,
            ],
            &[
                0, 187, 255, 193, 0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 237, 0, 0,
            ],
            &[
                0, 171, 255, 207, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 221, 0, 0,
            ],
            &[
                0, 135, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 192, 255, 183, 0, 0,
            ],
            &[
                0, 90, 255, 255, 39, 0, 0, 0, 0, 0, 0, 3, 237, 255, 136, 0, 0,
            ],
            &[
                0, 19, 252, 255, 127, 0, 0, 0, 0, 0, 0, 70, 255, 255, 62, 0, 0,
            ],
            &[
                0, 0, 174, 255, 236, 28, 0, 0, 0, 0, 4, 200, 255, 216, 2, 0, 0,
            ],
            &[
                0, 0, 44, 248, 255, 214, 64, 3, 0, 38, 181, 255, 255, 81, 0, 0, 0,
            ],
            &[
                0, 0, 0, 91, 251, 255, 255, 249, 242, 255, 255, 255, 131, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 58, 193, 254, 255, 255, 255, 208, 83, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 22, 53, 57, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 124, 124, 47, 0, 0, 0, 0],
        ],
        // letter: 'Ô' / 0xd4
        'Ô' => &[
            &[0, 0, 0, 0, 0, 7, 208, 255, 255, 230, 19, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 163, 255, 184, 166, 255, 194, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 130, 255, 179, 8, 2, 158, 255, 170, 3, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 64, 212, 138, 3, 0, 0, 0, 113, 212, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 107, 143, 143, 115, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 2, 143, 252, 255, 255, 255, 255, 255, 173, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 228, 169, 153, 202, 255, 255, 203, 8, 0, 0, 0,
            ],
            &[
                0, 0, 78, 255, 255, 156, 9, 0, 0, 0, 86, 252, 255, 134, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 205, 4, 0, 0, 0, 0, 0, 143, 255, 243, 15, 0, 0,
            ],
            &[
                0, 36, 255, 255, 96, 0, 0, 0, 0, 0, 0, 36, 255, 255, 89, 0, 0,
            ],
            &[
                0, 94, 255, 255, 19, 0, 0, 0, 0, 0, 0, 0, 223, 255, 157, 0, 0,
            ],
            &[
                0, 147, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 176, 255, 198, 0, 0,
            ],
            &[
                0, 170, 255, 201, 0, 0, 0, 0, 0, 0, 0, 0, 147, 255, 229, 0, 0,
            ],
            &[
                0, 186, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 242, 0, 0,
            ],
            &[
                0, 197, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 248, 0, 0,
            ],
            &[
                0, 187, 255, 193, 0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 237, 0, 0,
            ],
            &[
                0, 171, 255, 207, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 221, 0, 0,
            ],
            &[
                0, 135, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 192, 255, 183, 0, 0,
            ],
            &[
                0, 90, 255, 255, 39, 0, 0, 0, 0, 0, 0, 3, 237, 255, 136, 0, 0,
            ],
            &[
                0, 19, 252, 255, 127, 0, 0, 0, 0, 0, 0, 70, 255, 255, 62, 0, 0,
            ],
            &[
                0, 0, 174, 255, 236, 28, 0, 0, 0, 0, 4, 200, 255, 216, 2, 0, 0,
            ],
            &[
                0, 0, 44, 248, 255, 214, 64, 3, 0, 38, 181, 255, 255, 81, 0, 0, 0,
            ],
            &[
                0, 0, 0, 91, 251, 255, 255, 249, 242, 255, 255, 255, 131, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 58, 193, 254, 255, 255, 255, 208, 83, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 22, 53, 57, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 124, 124, 57, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Õ' / 0xd5
        'Õ' => &[
            &[0, 0, 0, 0, 19, 121, 141, 70, 0, 0, 0, 89, 147, 0, 0, 0, 0],
            &[
                0, 0, 0, 12, 222, 255, 255, 255, 190, 66, 62, 228, 178, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 114, 255, 134, 94, 197, 255, 255, 255, 254, 61, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 142, 170, 0, 0, 0, 85, 177, 180, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 107, 143, 143, 115, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 2, 143, 252, 255, 255, 255, 255, 255, 173, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 228, 169, 153, 202, 255, 255, 203, 8, 0, 0, 0,
            ],
            &[
                0, 0, 78, 255, 255, 156, 9, 0, 0, 0, 86, 252, 255, 134, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 205, 4, 0, 0, 0, 0, 0, 143, 255, 243, 15, 0, 0,
            ],
            &[
                0, 36, 255, 255, 96, 0, 0, 0, 0, 0, 0, 36, 255, 255, 89, 0, 0,
            ],
            &[
                0, 94, 255, 255, 19, 0, 0, 0, 0, 0, 0, 0, 223, 255, 157, 0, 0,
            ],
            &[
                0, 147, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 176, 255, 198, 0, 0,
            ],
            &[
                0, 170, 255, 201, 0, 0, 0, 0, 0, 0, 0, 0, 147, 255, 229, 0, 0,
            ],
            &[
                0, 186, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 242, 0, 0,
            ],
            &[
                0, 197, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 248, 0, 0,
            ],
            &[
                0, 187, 255, 193, 0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 237, 0, 0,
            ],
            &[
                0, 171, 255, 207, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 221, 0, 0,
            ],
            &[
                0, 135, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 192, 255, 183, 0, 0,
            ],
            &[
                0, 90, 255, 255, 39, 0, 0, 0, 0, 0, 0, 3, 237, 255, 136, 0, 0,
            ],
            &[
                0, 19, 252, 255, 127, 0, 0, 0, 0, 0, 0, 70, 255, 255, 62, 0, 0,
            ],
            &[
                0, 0, 174, 255, 236, 28, 0, 0, 0, 0, 4, 200, 255, 216, 2, 0, 0,
            ],
            &[
                0, 0, 44, 248, 255, 214, 64, 3, 0, 38, 181, 255, 255, 81, 0, 0, 0,
            ],
            &[
                0, 0, 0, 91, 251, 255, 255, 249, 242, 255, 255, 255, 131, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 58, 193, 254, 255, 255, 255, 208, 83, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 22, 53, 57, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ö' / 0xd6
        'Ö' => &[
            &[0, 0, 0, 15, 143, 143, 16, 0, 0, 2, 124, 156, 36, 0, 0, 0, 0],
            &[
                0, 0, 0, 104, 255, 255, 108, 0, 0, 57, 255, 255, 155, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 76, 255, 255, 79, 0, 0, 33, 251, 255, 125, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 54, 53, 0, 0, 0, 0, 36, 68, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 107, 143, 143, 115, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 2, 143, 252, 255, 255, 255, 255, 255, 173, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 228, 169, 153, 202, 255, 255, 203, 8, 0, 0, 0,
            ],
            &[
                0, 0, 78, 255, 255, 156, 9, 0, 0, 0, 86, 252, 255, 134, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 205, 4, 0, 0, 0, 0, 0, 143, 255, 243, 15, 0, 0,
            ],
            &[
                0, 36, 255, 255, 96, 0, 0, 0, 0, 0, 0, 36, 255, 255, 89, 0, 0,
            ],
            &[
                0, 94, 255, 255, 19, 0, 0, 0, 0, 0, 0, 0, 223, 255, 157, 0, 0,
            ],
            &[
                0, 147, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 176, 255, 198, 0, 0,
            ],
            &[
                0, 170, 255, 201, 0, 0, 0, 0, 0, 0, 0, 0, 147, 255, 229, 0, 0,
            ],
            &[
                0, 186, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 242, 0, 0,
            ],
            &[
                0, 197, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 248, 0, 0,
            ],
            &[
                0, 187, 255, 193, 0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 237, 0, 0,
            ],
            &[
                0, 171, 255, 207, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 221, 0, 0,
            ],
            &[
                0, 135, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 192, 255, 183, 0, 0,
            ],
            &[
                0, 90, 255, 255, 39, 0, 0, 0, 0, 0, 0, 3, 237, 255, 136, 0, 0,
            ],
            &[
                0, 19, 252, 255, 127, 0, 0, 0, 0, 0, 0, 70, 255, 255, 62, 0, 0,
            ],
            &[
                0, 0, 174, 255, 236, 28, 0, 0, 0, 0, 4, 200, 255, 216, 2, 0, 0,
            ],
            &[
                0, 0, 44, 248, 255, 214, 64, 3, 0, 38, 181, 255, 255, 81, 0, 0, 0,
            ],
            &[
                0, 0, 0, 91, 251, 255, 255, 249, 242, 255, 255, 255, 131, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 58, 193, 254, 255, 255, 255, 208, 83, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 22, 53, 57, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '×' / 0xd7
        '×' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 0, 0, 0, 0, 0, 0, 0, 0, 28, 0, 0, 0, 0],
            &[0, 0, 61, 243, 121, 0, 0, 0, 0, 0, 0, 68, 246, 114, 0, 0, 0],
            &[
                0, 0, 66, 245, 255, 120, 0, 0, 0, 0, 69, 246, 255, 108, 0, 0, 0,
            ],
            &[
                0, 0, 0, 66, 245, 255, 118, 0, 0, 70, 247, 255, 108, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 66, 245, 255, 117, 72, 247, 255, 108, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 66, 245, 255, 250, 255, 108, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 127, 255, 255, 179, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 249, 255, 250, 255, 121, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 72, 248, 255, 108, 74, 248, 255, 120, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 70, 247, 255, 110, 0, 0, 71, 246, 255, 118, 0, 0, 0, 0,
            ],
            &[
                0, 0, 67, 247, 255, 111, 0, 0, 0, 0, 67, 245, 255, 116, 0, 0, 0,
            ],
            &[0, 0, 49, 236, 113, 0, 0, 0, 0, 0, 0, 64, 241, 93, 0, 0, 0],
            &[0, 0, 0, 19, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ø' / 0xd8
        'Ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 15, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 31, 107, 143, 143, 115, 41, 0, 74, 240, 105, 0, 0,
            ],
            &[
                0, 0, 0, 2, 143, 252, 255, 255, 255, 255, 255, 178, 234, 255, 103, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 228, 169, 153, 202, 255, 255, 255, 219, 3, 0, 0,
            ],
            &[
                0, 0, 78, 255, 255, 156, 9, 0, 0, 0, 105, 255, 255, 202, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 205, 4, 0, 0, 0, 0, 148, 255, 255, 243, 15, 0, 0,
            ],
            &[
                0, 36, 255, 255, 96, 0, 0, 0, 0, 42, 251, 255, 255, 255, 89, 0, 0,
            ],
            &[
                0, 94, 255, 255, 19, 0, 0, 0, 0, 184, 255, 141, 223, 255, 157, 0, 0,
            ],
            &[
                0, 147, 255, 231, 0, 0, 0, 0, 75, 255, 235, 16, 176, 255, 198, 0, 0,
            ],
            &[
                0, 170, 255, 201, 0, 0, 0, 5, 216, 255, 105, 0, 147, 255, 229, 0, 0,
            ],
            &[
                0, 186, 255, 189, 0, 0, 0, 112, 255, 211, 3, 0, 135, 255, 242, 0, 0,
            ],
            &[
                0, 197, 255, 182, 0, 0, 19, 238, 255, 69, 0, 0, 130, 255, 248, 0, 0,
            ],
            &[
                0, 187, 255, 193, 0, 0, 148, 255, 179, 0, 0, 0, 141, 255, 237, 0, 0,
            ],
            &[
                0, 171, 255, 207, 0, 43, 251, 250, 38, 0, 0, 0, 155, 255, 221, 0, 0,
            ],
            &[
                0, 135, 255, 244, 1, 185, 255, 143, 0, 0, 0, 0, 192, 255, 183, 0, 0,
            ],
            &[
                0, 90, 255, 255, 115, 255, 236, 16, 0, 0, 0, 3, 237, 255, 136, 0, 0,
            ],
            &[
                0, 19, 252, 255, 255, 255, 107, 0, 0, 0, 0, 70, 255, 255, 62, 0, 0,
            ],
            &[
                0, 0, 174, 255, 255, 241, 4, 0, 0, 0, 4, 200, 255, 216, 2, 0, 0,
            ],
            &[
                0, 0, 64, 255, 255, 255, 64, 3, 0, 38, 181, 255, 255, 81, 0, 0, 0,
            ],
            &[
                0, 0, 149, 255, 255, 255, 255, 249, 242, 255, 255, 255, 131, 0, 0, 0, 0,
            ],
            &[
                0, 43, 252, 251, 98, 193, 254, 255, 255, 255, 208, 83, 0, 0, 0, 0, 0,
            ],
            &[0, 26, 156, 141, 0, 0, 22, 53, 57, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ù' / 0xd9
        'Ù' => &[
            &[0, 0, 0, 0, 12, 215, 255, 230, 11, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 231, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 40, 230, 253, 64, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 33, 195, 186, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 8, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 21, 255, 255, 89, 0, 0, 0, 0, 0, 0, 37, 255, 255, 72, 0, 0,
            ],
            &[
                0, 3, 254, 255, 110, 0, 0, 0, 0, 0, 0, 58, 255, 255, 53, 0, 0,
            ],
            &[
                0, 0, 222, 255, 162, 0, 0, 0, 0, 0, 0, 107, 255, 253, 16, 0, 0,
            ],
            &[
                0, 0, 155, 255, 244, 22, 0, 0, 0, 0, 3, 206, 255, 201, 0, 0, 0,
            ],
            &[
                0, 0, 46, 253, 255, 205, 51, 0, 0, 30, 169, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 125, 255, 255, 255, 245, 240, 255, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 90, 210, 255, 255, 255, 255, 221, 109, 1, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 25, 53, 56, 27, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 124, 124, 66, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ú' / 0xda
        'Ú' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 18, 237, 255, 207, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 158, 255, 224, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 72, 255, 227, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 189, 190, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 8, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 21, 255, 255, 89, 0, 0, 0, 0, 0, 0, 37, 255, 255, 72, 0, 0,
            ],
            &[
                0, 3, 254, 255, 110, 0, 0, 0, 0, 0, 0, 58, 255, 255, 53, 0, 0,
            ],
            &[
                0, 0, 222, 255, 162, 0, 0, 0, 0, 0, 0, 107, 255, 253, 16, 0, 0,
            ],
            &[
                0, 0, 155, 255, 244, 22, 0, 0, 0, 0, 3, 206, 255, 201, 0, 0, 0,
            ],
            &[
                0, 0, 46, 253, 255, 205, 51, 0, 0, 30, 169, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 125, 255, 255, 255, 245, 240, 255, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 90, 210, 255, 255, 255, 255, 221, 109, 1, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 25, 53, 56, 27, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 124, 124, 47, 0, 0, 0, 0],
        ],
        // letter: 'Û' / 0xdb
        'Û' => &[
            &[0, 0, 0, 0, 0, 7, 208, 255, 255, 230, 19, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 163, 255, 184, 166, 255, 194, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 130, 255, 179, 8, 2, 158, 255, 170, 3, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 64, 212, 138, 3, 0, 0, 0, 113, 212, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 8, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 21, 255, 255, 89, 0, 0, 0, 0, 0, 0, 37, 255, 255, 72, 0, 0,
            ],
            &[
                0, 3, 254, 255, 110, 0, 0, 0, 0, 0, 0, 58, 255, 255, 53, 0, 0,
            ],
            &[
                0, 0, 222, 255, 162, 0, 0, 0, 0, 0, 0, 107, 255, 253, 16, 0, 0,
            ],
            &[
                0, 0, 155, 255, 244, 22, 0, 0, 0, 0, 3, 206, 255, 201, 0, 0, 0,
            ],
            &[
                0, 0, 46, 253, 255, 205, 51, 0, 0, 30, 169, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 125, 255, 255, 255, 245, 240, 255, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 90, 210, 255, 255, 255, 255, 221, 109, 1, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 25, 53, 56, 27, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 124, 124, 57, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ü' / 0xdc
        'Ü' => &[
            &[0, 0, 0, 15, 143, 143, 16, 0, 0, 2, 124, 156, 36, 0, 0, 0, 0],
            &[
                0, 0, 0, 104, 255, 255, 108, 0, 0, 57, 255, 255, 155, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 76, 255, 255, 79, 0, 0, 33, 251, 255, 125, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 54, 53, 0, 0, 0, 0, 36, 68, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 8, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 21, 255, 255, 89, 0, 0, 0, 0, 0, 0, 37, 255, 255, 72, 0, 0,
            ],
            &[
                0, 3, 254, 255, 110, 0, 0, 0, 0, 0, 0, 58, 255, 255, 53, 0, 0,
            ],
            &[
                0, 0, 222, 255, 162, 0, 0, 0, 0, 0, 0, 107, 255, 253, 16, 0, 0,
            ],
            &[
                0, 0, 155, 255, 244, 22, 0, 0, 0, 0, 3, 206, 255, 201, 0, 0, 0,
            ],
            &[
                0, 0, 46, 253, 255, 205, 51, 0, 0, 30, 169, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 125, 255, 255, 255, 245, 240, 255, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 90, 210, 255, 255, 255, 255, 221, 109, 1, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 25, 53, 56, 27, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ý' / 0xdd
        'Ý' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 18, 237, 255, 207, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 158, 255, 224, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 72, 255, 227, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 189, 190, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 69, 71, 48, 0, 0, 0, 0, 0, 0, 0, 0, 28, 71, 71, 13, 0],
            &[
                0, 168, 255, 238, 12, 0, 0, 0, 0, 0, 0, 0, 178, 255, 216, 3, 0,
            ],
            &[
                0, 41, 253, 255, 112, 0, 0, 0, 0, 0, 0, 43, 254, 255, 89, 0, 0,
            ],
            &[
                0, 0, 163, 255, 226, 4, 0, 0, 0, 0, 0, 161, 255, 212, 2, 0, 0,
            ],
            &[
                0, 0, 37, 252, 255, 93, 0, 0, 0, 0, 29, 250, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 159, 255, 210, 0, 0, 0, 0, 143, 255, 208, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 33, 250, 255, 74, 0, 0, 18, 244, 255, 80, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 192, 0, 0, 125, 255, 204, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 30, 249, 255, 55, 9, 234, 255, 75, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 149, 255, 173, 108, 255, 200, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 247, 253, 232, 255, 70, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 145, 255, 255, 196, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 255, 82, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 124, 124, 47, 0, 0, 0, 0],
        ],
        // letter: 'Þ' / 0xde
        'Þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 30, 71, 70, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 255, 196, 196, 196, 190, 173, 131, 57, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 186, 20, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 86, 86, 86, 105, 137, 227, 255, 255, 194, 1, 0, 0,
            ],
            &[
                0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 7, 173, 255, 255, 74, 0, 0,
            ],
            &[
                0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 19, 252, 255, 146, 0, 0,
            ],
            &[
                0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 218, 255, 175, 0, 0,
            ],
            &[
                0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 209, 255, 165, 0, 0,
            ],
            &[
                0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 10, 244, 255, 129, 0, 0,
            ],
            &[
                0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 127, 255, 255, 47, 0, 0,
            ],
            &[
                0, 0, 111, 255, 254, 5, 5, 8, 24, 69, 165, 254, 255, 165, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 181, 14, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 254, 235, 180, 86, 1, 0, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 21, 21, 20, 4, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ß' / 0xdf
        'ß' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 11, 83, 137, 158, 158, 137, 85, 13, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 84, 233, 255, 255, 255, 255, 255, 255, 239, 99, 0, 0, 0, 0, 0,
            ],
            &[
                0, 61, 252, 255, 239, 158, 119, 114, 153, 237, 255, 255, 74, 0, 0, 0, 0,
            ],
            &[
                0, 179, 255, 228, 26, 0, 0, 0, 0, 29, 237, 255, 173, 0, 0, 0, 0,
            ],
            &[
                0, 226, 255, 131, 0, 0, 0, 0, 0, 0, 186, 255, 195, 0, 0, 0, 0,
            ],
            &[
                0, 248, 255, 104, 0, 0, 0, 0, 0, 11, 239, 255, 147, 0, 0, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 0, 0, 0, 14, 190, 255, 229, 22, 0, 0, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 0, 0, 28, 216, 255, 227, 46, 0, 0, 0, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 0, 3, 208, 255, 212, 26, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 0, 58, 255, 255, 53, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 0, 54, 255, 255, 60, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 0, 5, 220, 255, 242, 89, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 0, 0, 38, 225, 255, 255, 162, 10, 0, 0, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 0, 0, 0, 18, 179, 255, 255, 211, 24, 0, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 0, 0, 0, 0, 0, 110, 248, 255, 193, 4, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 0, 0, 0, 0, 0, 0, 85, 255, 255, 78, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 0, 0, 0, 0, 0, 0, 0, 237, 255, 123, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 0, 0, 0, 0, 0, 0, 0, 241, 255, 117, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 78, 107, 11, 0, 0, 0, 111, 255, 255, 66, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 105, 255, 248, 202, 186, 218, 255, 255, 188, 0, 0, 0,
            ],
            &[
                0, 249, 255, 102, 0, 45, 184, 253, 255, 255, 255, 243, 146, 12, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 13, 43, 63, 45, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'à' / 0xe0
        'à' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 36, 174, 174, 126, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 153, 255, 254, 51, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 179, 255, 207, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 7, 178, 255, 132, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 131, 161, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 108, 159, 180, 183, 161, 106, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 174, 252, 255, 255, 255, 255, 255, 255, 239, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 163, 248, 163, 95, 66, 75, 119, 226, 255, 248, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 27, 20, 0, 0, 0, 0, 0, 25, 235, 255, 132, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 181, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 196, 0, 0],
            &[
                0, 0, 0, 0, 0, 10, 76, 135, 162, 183, 191, 198, 226, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 232, 255, 255, 255, 255, 253, 245, 246, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 88, 255, 255, 194, 90, 36, 11, 0, 0, 113, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 170, 1, 0, 0, 0, 0, 0, 123, 255, 197, 0, 0,
            ],
            &[
                0, 0, 10, 255, 255, 64, 0, 0, 0, 0, 0, 0, 165, 255, 197, 0, 0,
            ],
            &[
                0, 0, 18, 255, 255, 73, 0, 0, 0, 0, 0, 38, 248, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 227, 255, 192, 16, 0, 0, 1, 73, 226, 255, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 243, 195, 190, 230, 255, 212, 121, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 223, 255, 255, 255, 245, 144, 19, 61, 255, 197, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 1, 29, 59, 39, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'á' / 0xe1
        'á' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 87, 174, 174, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 234, 255, 203, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 217, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 73, 254, 217, 29, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 133, 152, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 108, 159, 180, 183, 161, 106, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 174, 252, 255, 255, 255, 255, 255, 255, 239, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 163, 248, 163, 95, 66, 75, 119, 226, 255, 248, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 27, 20, 0, 0, 0, 0, 0, 25, 235, 255, 132, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 181, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 196, 0, 0],
            &[
                0, 0, 0, 0, 0, 10, 76, 135, 162, 183, 191, 198, 226, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 232, 255, 255, 255, 255, 253, 245, 246, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 88, 255, 255, 194, 90, 36, 11, 0, 0, 113, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 170, 1, 0, 0, 0, 0, 0, 123, 255, 197, 0, 0,
            ],
            &[
                0, 0, 10, 255, 255, 64, 0, 0, 0, 0, 0, 0, 165, 255, 197, 0, 0,
            ],
            &[
                0, 0, 18, 255, 255, 73, 0, 0, 0, 0, 0, 38, 248, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 227, 255, 192, 16, 0, 0, 1, 73, 226, 255, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 243, 195, 190, 230, 255, 212, 121, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 223, 255, 255, 255, 245, 144, 19, 61, 255, 197, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 1, 29, 59, 39, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'â' / 0xe2
        'â' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 43, 174, 174, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 208, 255, 255, 254, 67, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 1, 170, 255, 172, 93, 251, 238, 40, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 141, 255, 163, 4, 0, 81, 245, 224, 29, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 39, 163, 110, 0, 0, 0, 0, 46, 161, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 108, 159, 180, 183, 161, 106, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 174, 252, 255, 255, 255, 255, 255, 255, 239, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 163, 248, 163, 95, 66, 75, 119, 226, 255, 248, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 27, 20, 0, 0, 0, 0, 0, 25, 235, 255, 132, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 181, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 196, 0, 0],
            &[
                0, 0, 0, 0, 0, 10, 76, 135, 162, 183, 191, 198, 226, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 232, 255, 255, 255, 255, 253, 245, 246, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 88, 255, 255, 194, 90, 36, 11, 0, 0, 113, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 170, 1, 0, 0, 0, 0, 0, 123, 255, 197, 0, 0,
            ],
            &[
                0, 0, 10, 255, 255, 64, 0, 0, 0, 0, 0, 0, 165, 255, 197, 0, 0,
            ],
            &[
                0, 0, 18, 255, 255, 73, 0, 0, 0, 0, 0, 38, 248, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 227, 255, 192, 16, 0, 0, 1, 73, 226, 255, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 243, 195, 190, 230, 255, 212, 121, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 223, 255, 255, 255, 245, 144, 19, 61, 255, 197, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 1, 29, 59, 39, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ã' / 0xe3
        'ã' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 161, 194, 134, 19, 0, 0, 96, 211, 5, 0, 0],
            &[
                0, 0, 0, 0, 9, 225, 255, 255, 255, 239, 128, 102, 228, 196, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 95, 255, 111, 39, 129, 245, 255, 255, 250, 64, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 90, 145, 0, 0, 0, 27, 120, 136, 52, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 108, 159, 180, 183, 161, 106, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 174, 252, 255, 255, 255, 255, 255, 255, 239, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 163, 248, 163, 95, 66, 75, 119, 226, 255, 248, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 27, 20, 0, 0, 0, 0, 0, 25, 235, 255, 132, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 181, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 196, 0, 0],
            &[
                0, 0, 0, 0, 0, 10, 76, 135, 162, 183, 191, 198, 226, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 232, 255, 255, 255, 255, 253, 245, 246, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 88, 255, 255, 194, 90, 36, 11, 0, 0, 113, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 170, 1, 0, 0, 0, 0, 0, 123, 255, 197, 0, 0,
            ],
            &[
                0, 0, 10, 255, 255, 64, 0, 0, 0, 0, 0, 0, 165, 255, 197, 0, 0,
            ],
            &[
                0, 0, 18, 255, 255, 73, 0, 0, 0, 0, 0, 38, 248, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 227, 255, 192, 16, 0, 0, 1, 73, 226, 255, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 243, 195, 190, 230, 255, 212, 121, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 223, 255, 255, 255, 245, 144, 19, 61, 255, 197, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 1, 29, 59, 39, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ä' / 0xe4
        'ä' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 180, 203, 55, 0, 0, 1, 154, 212, 86, 0, 0, 0],
            &[
                0, 0, 0, 0, 73, 255, 255, 146, 0, 0, 26, 255, 255, 193, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 29, 237, 250, 87, 0, 0, 5, 214, 254, 129, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 8, 20, 0, 0, 0, 0, 2, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 108, 159, 180, 183, 161, 106, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 174, 252, 255, 255, 255, 255, 255, 255, 239, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 163, 248, 163, 95, 66, 75, 119, 226, 255, 248, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 27, 20, 0, 0, 0, 0, 0, 25, 235, 255, 132, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 181, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 196, 0, 0],
            &[
                0, 0, 0, 0, 0, 10, 76, 135, 162, 183, 191, 198, 226, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 232, 255, 255, 255, 255, 253, 245, 246, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 88, 255, 255, 194, 90, 36, 11, 0, 0, 113, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 170, 1, 0, 0, 0, 0, 0, 123, 255, 197, 0, 0,
            ],
            &[
                0, 0, 10, 255, 255, 64, 0, 0, 0, 0, 0, 0, 165, 255, 197, 0, 0,
            ],
            &[
                0, 0, 18, 255, 255, 73, 0, 0, 0, 0, 0, 38, 248, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 227, 255, 192, 16, 0, 0, 1, 73, 226, 255, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 243, 195, 190, 230, 255, 212, 121, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 223, 255, 255, 255, 245, 144, 19, 61, 255, 197, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 1, 29, 59, 39, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'å' / 0xe5
        'å' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 144, 242, 252, 197, 45, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 111, 255, 157, 112, 229, 223, 3, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 173, 229, 1, 0, 117, 255, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 156, 249, 26, 1, 161, 255, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 50, 249, 247, 226, 255, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 41, 134, 149, 90, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 108, 159, 180, 183, 161, 106, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 174, 252, 255, 255, 255, 255, 255, 255, 239, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 163, 248, 163, 95, 66, 75, 119, 226, 255, 248, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 27, 20, 0, 0, 0, 0, 0, 25, 235, 255, 132, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 181, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 196, 0, 0],
            &[
                0, 0, 0, 0, 0, 10, 76, 135, 162, 183, 191, 198, 226, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 232, 255, 255, 255, 255, 253, 245, 246, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 88, 255, 255, 194, 90, 36, 11, 0, 0, 113, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 170, 1, 0, 0, 0, 0, 0, 123, 255, 197, 0, 0,
            ],
            &[
                0, 0, 10, 255, 255, 64, 0, 0, 0, 0, 0, 0, 165, 255, 197, 0, 0,
            ],
            &[
                0, 0, 18, 255, 255, 73, 0, 0, 0, 0, 0, 38, 248, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 227, 255, 192, 16, 0, 0, 1, 73, 226, 255, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 243, 195, 190, 230, 255, 212, 121, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 223, 255, 255, 255, 245, 144, 19, 61, 255, 197, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 1, 29, 59, 39, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'æ' / 0xe6
        'æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 20, 113, 165, 175, 120, 14, 0, 25, 135, 180, 154, 47, 0, 0, 0,
            ],
            &[
                0, 66, 241, 255, 255, 255, 255, 209, 34, 228, 255, 255, 255, 250, 65, 0, 0,
            ],
            &[
                0, 21, 240, 148, 79, 94, 229, 255, 233, 255, 162, 73, 169, 255, 218, 2, 0,
            ],
            &[
                0, 0, 17, 0, 0, 0, 86, 255, 255, 196, 1, 0, 16, 243, 255, 62, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 29, 255, 255, 96, 0, 0, 0, 183, 255, 120, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 10, 255, 255, 52, 0, 0, 0, 160, 255, 149, 0,
            ],
            &[
                0, 0, 18, 107, 151, 167, 177, 255, 255, 180, 170, 170, 170, 219, 255, 164, 0,
            ],
            &[
                0, 55, 234, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 168, 0,
            ],
            &[
                5, 225, 255, 215, 87, 33, 31, 255, 255, 71, 57, 57, 57, 57, 57, 37, 0,
            ],
            &[
                69, 255, 245, 27, 0, 0, 27, 255, 255, 35, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                106, 255, 197, 0, 0, 0, 48, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                99, 255, 194, 0, 0, 0, 111, 255, 255, 161, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                65, 255, 246, 31, 0, 30, 228, 229, 225, 252, 84, 0, 0, 1, 94, 58, 0,
            ],
            &[
                4, 215, 255, 237, 199, 251, 255, 91, 94, 255, 255, 211, 193, 235, 255, 79, 0,
            ],
            &[
                0, 48, 220, 255, 255, 239, 107, 0, 0, 106, 236, 255, 255, 255, 179, 29, 0,
            ],
            &[0, 0, 4, 46, 50, 10, 0, 0, 0, 0, 7, 48, 55, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ç' / 0xe7
        'ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 3, 65, 138, 168, 185, 171, 129, 44, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 49, 211, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0,
            ],
            &[
                0, 0, 0, 40, 244, 255, 252, 179, 118, 100, 123, 173, 250, 44, 0, 0, 0,
            ],
            &[0, 0, 0, 203, 255, 241, 61, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0],
            &[0, 0, 45, 255, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 255, 254, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 255, 224, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 171, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 166, 255, 212, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 235, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 102, 255, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 253, 255, 167, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 152, 255, 255, 158, 31, 0, 0, 0, 16, 89, 71, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 196, 255, 255, 254, 229, 214, 239, 255, 255, 121, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 4, 118, 217, 255, 255, 255, 255, 251, 191, 64, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 44, 255, 205, 36, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 123, 255, 143, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 91, 176, 252, 235, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 136, 255, 112, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 17, 7, 42, 183, 255, 87, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 215, 255, 255, 255, 173, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 88, 118, 97, 47, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'è' / 0xe8
        'è' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 174, 174, 121, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 160, 255, 252, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 185, 255, 201, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 183, 255, 126, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 134, 160, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 119, 160, 182, 158, 105, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 134, 251, 255, 255, 255, 255, 255, 238, 70, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 160, 255, 255, 185, 100, 67, 96, 201, 255, 251, 55, 0, 0, 0,
            ],
            &[
                0, 0, 84, 255, 255, 119, 0, 0, 0, 0, 5, 188, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 198, 255, 189, 0, 0, 0, 0, 0, 0, 52, 255, 255, 46, 0, 0,
            ],
            &[
                0, 27, 254, 255, 93, 0, 0, 0, 0, 0, 0, 0, 242, 255, 109, 0, 0,
            ],
            &[
                0, 64, 255, 255, 196, 184, 184, 184, 184, 184, 184, 184, 246, 255, 133, 0, 0,
            ],
            &[
                0, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[
                0, 79, 255, 255, 81, 57, 57, 57, 57, 57, 57, 57, 57, 57, 33, 0, 0,
            ],
            &[0, 53, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 241, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 235, 29, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 25, 233, 255, 220, 73, 3, 0, 0, 0, 14, 98, 177, 0, 0, 0,
            ],
            &[
                0, 0, 0, 57, 229, 255, 255, 238, 202, 191, 213, 252, 255, 238, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 139, 228, 255, 255, 255, 255, 249, 192, 99, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 28, 53, 58, 32, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'é' / 0xe9
        'é' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 174, 174, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 19, 238, 255, 198, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 162, 255, 213, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 80, 255, 213, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 137, 150, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 119, 160, 182, 158, 105, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 134, 251, 255, 255, 255, 255, 255, 238, 70, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 160, 255, 255, 185, 100, 67, 96, 201, 255, 251, 55, 0, 0, 0,
            ],
            &[
                0, 0, 84, 255, 255, 119, 0, 0, 0, 0, 5, 188, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 198, 255, 189, 0, 0, 0, 0, 0, 0, 52, 255, 255, 46, 0, 0,
            ],
            &[
                0, 27, 254, 255, 93, 0, 0, 0, 0, 0, 0, 0, 242, 255, 109, 0, 0,
            ],
            &[
                0, 64, 255, 255, 196, 184, 184, 184, 184, 184, 184, 184, 246, 255, 133, 0, 0,
            ],
            &[
                0, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[
                0, 79, 255, 255, 81, 57, 57, 57, 57, 57, 57, 57, 57, 57, 33, 0, 0,
            ],
            &[0, 53, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 241, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 235, 29, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 25, 233, 255, 220, 73, 3, 0, 0, 0, 14, 98, 177, 0, 0, 0,
            ],
            &[
                0, 0, 0, 57, 229, 255, 255, 238, 202, 191, 213, 252, 255, 238, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 139, 228, 255, 255, 255, 255, 249, 192, 99, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 28, 53, 58, 32, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ê' / 0xea
        'ê' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 174, 174, 110, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 213, 255, 255, 252, 61, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 2, 176, 255, 167, 97, 252, 235, 36, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 147, 255, 157, 3, 0, 86, 246, 221, 26, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 44, 163, 106, 0, 0, 0, 0, 50, 162, 111, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 119, 160, 182, 158, 105, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 134, 251, 255, 255, 255, 255, 255, 238, 70, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 160, 255, 255, 185, 100, 67, 96, 201, 255, 251, 55, 0, 0, 0,
            ],
            &[
                0, 0, 84, 255, 255, 119, 0, 0, 0, 0, 5, 188, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 198, 255, 189, 0, 0, 0, 0, 0, 0, 52, 255, 255, 46, 0, 0,
            ],
            &[
                0, 27, 254, 255, 93, 0, 0, 0, 0, 0, 0, 0, 242, 255, 109, 0, 0,
            ],
            &[
                0, 64, 255, 255, 196, 184, 184, 184, 184, 184, 184, 184, 246, 255, 133, 0, 0,
            ],
            &[
                0, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[
                0, 79, 255, 255, 81, 57, 57, 57, 57, 57, 57, 57, 57, 57, 33, 0, 0,
            ],
            &[0, 53, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 241, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 235, 29, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 25, 233, 255, 220, 73, 3, 0, 0, 0, 14, 98, 177, 0, 0, 0,
            ],
            &[
                0, 0, 0, 57, 229, 255, 255, 238, 202, 191, 213, 252, 255, 238, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 139, 228, 255, 255, 255, 255, 249, 192, 99, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 28, 53, 58, 32, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ë' / 0xeb
        'ë' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 183, 201, 51, 0, 0, 2, 158, 211, 81, 0, 0, 0, 0],
            &[
                0, 0, 0, 80, 255, 255, 139, 0, 0, 33, 255, 255, 186, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 34, 239, 249, 81, 0, 0, 8, 218, 254, 123, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 9, 19, 0, 0, 0, 0, 3, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 119, 160, 182, 158, 105, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 134, 251, 255, 255, 255, 255, 255, 238, 70, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 160, 255, 255, 185, 100, 67, 96, 201, 255, 251, 55, 0, 0, 0,
            ],
            &[
                0, 0, 84, 255, 255, 119, 0, 0, 0, 0, 5, 188, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 198, 255, 189, 0, 0, 0, 0, 0, 0, 52, 255, 255, 46, 0, 0,
            ],
            &[
                0, 27, 254, 255, 93, 0, 0, 0, 0, 0, 0, 0, 242, 255, 109, 0, 0,
            ],
            &[
                0, 64, 255, 255, 196, 184, 184, 184, 184, 184, 184, 184, 246, 255, 133, 0, 0,
            ],
            &[
                0, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[
                0, 79, 255, 255, 81, 57, 57, 57, 57, 57, 57, 57, 57, 57, 33, 0, 0,
            ],
            &[0, 53, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 241, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 235, 29, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 25, 233, 255, 220, 73, 3, 0, 0, 0, 14, 98, 177, 0, 0, 0,
            ],
            &[
                0, 0, 0, 57, 229, 255, 255, 238, 202, 191, 213, 252, 255, 238, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 139, 228, 255, 255, 255, 255, 249, 192, 99, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 28, 53, 58, 32, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ì' / 0xec
        'ì' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 174, 174, 102, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 185, 255, 244, 26, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 204, 255, 177, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 17, 202, 255, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 145, 152, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 49, 120, 120, 120, 120, 120, 120, 90, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 22, 68, 87, 106, 125, 218, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 255, 192, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 140, 185, 205, 225, 245, 255, 255, 255, 246, 224, 203, 181, 95, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'í' / 0xed
        'í' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 110, 174, 174, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 36, 249, 255, 175, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 189, 255, 195, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 107, 255, 197, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 154, 141, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 49, 120, 120, 120, 120, 120, 120, 90, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 22, 68, 87, 106, 125, 218, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 255, 192, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 140, 185, 205, 225, 245, 255, 255, 255, 246, 224, 203, 181, 95, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'î' / 0xee
        'î' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 174, 174, 91, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 230, 255, 255, 245, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 8, 198, 255, 142, 119, 255, 222, 22, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 172, 255, 133, 0, 0, 109, 252, 204, 15, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 61, 163, 89, 0, 0, 0, 0, 67, 163, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 49, 120, 120, 120, 120, 120, 120, 90, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 22, 68, 87, 106, 125, 218, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 255, 192, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 140, 185, 205, 225, 245, 255, 255, 255, 246, 224, 203, 181, 95, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ï' / 0xef
        'ï' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 32, 193, 193, 34, 0, 0, 10, 174, 206, 63, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 108, 255, 255, 112, 0, 0, 61, 255, 255, 159, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 55, 245, 245, 57, 0, 0, 21, 232, 252, 98, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 14, 14, 0, 0, 0, 0, 6, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 49, 120, 120, 120, 120, 120, 120, 90, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 22, 68, 87, 106, 125, 218, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 255, 192, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 140, 185, 205, 225, 245, 255, 255, 255, 246, 224, 203, 181, 95, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ð' / 0xf0
        'ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 72, 0, 0, 0, 0, 23, 60, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 5, 214, 255, 184, 28, 6, 118, 241, 240, 29, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 92, 233, 255, 255, 255, 255, 239, 112, 4, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 62, 255, 255, 255, 240, 20, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 19, 149, 252, 255, 255, 255, 253, 74, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 116, 255, 224, 84, 0, 112, 255, 239, 34, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 3, 102, 8, 0, 0, 0, 151, 255, 182, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 222, 255, 71, 0, 0, 0],
            &[
                0, 0, 0, 0, 12, 96, 160, 187, 185, 151, 74, 92, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 0, 68, 233, 255, 255, 255, 255, 255, 255, 183, 247, 253, 23, 0, 0,
            ],
            &[
                0, 0, 53, 250, 255, 214, 102, 54, 47, 84, 189, 255, 255, 255, 85, 0, 0,
            ],
            &[
                0, 0, 209, 255, 204, 11, 0, 0, 0, 0, 1, 158, 255, 255, 138, 0, 0,
            ],
            &[
                0, 49, 255, 255, 69, 0, 0, 0, 0, 0, 0, 15, 247, 255, 162, 0, 0,
            ],
            &[0, 99, 255, 252, 5, 0, 0, 0, 0, 0, 0, 0, 195, 255, 178, 0, 0],
            &[
                0, 123, 255, 234, 0, 0, 0, 0, 0, 0, 0, 0, 182, 255, 168, 0, 0,
            ],
            &[
                0, 108, 255, 243, 0, 0, 0, 0, 0, 0, 0, 0, 200, 255, 145, 0, 0,
            ],
            &[0, 74, 255, 255, 35, 0, 0, 0, 0, 0, 0, 6, 246, 255, 93, 0, 0],
            &[
                0, 6, 235, 255, 144, 0, 0, 0, 0, 0, 0, 98, 255, 246, 18, 0, 0,
            ],
            &[
                0, 0, 109, 255, 253, 119, 7, 0, 0, 0, 75, 241, 255, 137, 0, 0, 0,
            ],
            &[
                0, 0, 0, 149, 255, 255, 237, 191, 182, 220, 255, 255, 182, 6, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 82, 198, 255, 255, 255, 255, 226, 113, 4, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 21, 51, 60, 35, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ñ' / 0xf1
        'ñ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 13, 143, 195, 151, 36, 0, 0, 55, 216, 41, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 186, 255, 255, 255, 250, 148, 94, 206, 239, 5, 0, 0, 0,
            ],
            &[
                0, 0, 0, 47, 255, 149, 38, 105, 233, 255, 255, 255, 107, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 59, 162, 14, 0, 0, 13, 107, 141, 74, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 101, 120, 17, 0, 28, 125, 163, 183, 158, 83, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 67, 122, 249, 255, 255, 255, 255, 255, 191, 10, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 187, 255, 187, 92, 65, 92, 190, 255, 255, 144, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 163, 0, 0, 0, 0, 1, 200, 255, 239, 2, 0, 0,
            ],
            &[
                0, 0, 214, 255, 254, 38, 0, 0, 0, 0, 0, 104, 255, 255, 29, 0, 0,
            ],
            &[
                0, 0, 214, 255, 208, 0, 0, 0, 0, 0, 0, 74, 255, 255, 43, 0, 0,
            ],
            &[
                0, 0, 214, 255, 178, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 155, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ò' / 0xf2
        'ò' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 174, 174, 102, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 185, 255, 244, 26, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 204, 255, 177, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 17, 202, 255, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 145, 152, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 142, 173, 176, 146, 72, 2, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 26, 195, 255, 255, 255, 255, 255, 255, 212, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 222, 255, 242, 132, 72, 72, 123, 232, 255, 240, 33, 0, 0, 0,
            ],
            &[
                0, 0, 151, 255, 243, 45, 0, 0, 0, 0, 29, 225, 255, 193, 0, 0, 0,
            ],
            &[
                0, 9, 243, 255, 128, 0, 0, 0, 0, 0, 0, 91, 255, 255, 41, 0, 0,
            ],
            &[
                0, 73, 255, 255, 39, 0, 0, 0, 0, 0, 0, 10, 249, 255, 121, 0, 0,
            ],
            &[
                0, 103, 255, 247, 1, 0, 0, 0, 0, 0, 0, 0, 212, 255, 151, 0, 0,
            ],
            &[
                0, 125, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 195, 255, 175, 0, 0,
            ],
            &[
                0, 117, 255, 241, 0, 0, 0, 0, 0, 0, 0, 0, 203, 255, 170, 0, 0,
            ],
            &[
                0, 94, 255, 255, 16, 0, 0, 0, 0, 0, 0, 0, 232, 255, 147, 0, 0,
            ],
            &[
                0, 37, 255, 255, 82, 0, 0, 0, 0, 0, 0, 42, 255, 255, 91, 0, 0,
            ],
            &[
                0, 0, 205, 255, 201, 2, 0, 0, 0, 0, 0, 160, 255, 244, 17, 0, 0,
            ],
            &[
                0, 0, 72, 254, 255, 165, 18, 0, 0, 7, 128, 255, 255, 129, 0, 0, 0,
            ],
            &[
                0, 0, 0, 119, 254, 255, 244, 199, 193, 236, 255, 255, 176, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 71, 199, 254, 255, 255, 255, 223, 109, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 21, 55, 60, 33, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ó' / 0xf3
        'ó' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 110, 174, 174, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 36, 249, 255, 175, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 189, 255, 195, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 107, 255, 197, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 154, 141, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 142, 173, 176, 146, 72, 2, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 26, 195, 255, 255, 255, 255, 255, 255, 212, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 222, 255, 242, 132, 72, 72, 123, 232, 255, 240, 33, 0, 0, 0,
            ],
            &[
                0, 0, 151, 255, 243, 45, 0, 0, 0, 0, 29, 225, 255, 193, 0, 0, 0,
            ],
            &[
                0, 9, 243, 255, 128, 0, 0, 0, 0, 0, 0, 91, 255, 255, 41, 0, 0,
            ],
            &[
                0, 73, 255, 255, 39, 0, 0, 0, 0, 0, 0, 10, 249, 255, 121, 0, 0,
            ],
            &[
                0, 103, 255, 247, 1, 0, 0, 0, 0, 0, 0, 0, 212, 255, 151, 0, 0,
            ],
            &[
                0, 125, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 195, 255, 175, 0, 0,
            ],
            &[
                0, 117, 255, 241, 0, 0, 0, 0, 0, 0, 0, 0, 203, 255, 170, 0, 0,
            ],
            &[
                0, 94, 255, 255, 16, 0, 0, 0, 0, 0, 0, 0, 232, 255, 147, 0, 0,
            ],
            &[
                0, 37, 255, 255, 82, 0, 0, 0, 0, 0, 0, 42, 255, 255, 91, 0, 0,
            ],
            &[
                0, 0, 205, 255, 201, 2, 0, 0, 0, 0, 0, 160, 255, 244, 17, 0, 0,
            ],
            &[
                0, 0, 72, 254, 255, 165, 18, 0, 0, 7, 128, 255, 255, 129, 0, 0, 0,
            ],
            &[
                0, 0, 0, 119, 254, 255, 244, 199, 193, 236, 255, 255, 176, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 71, 199, 254, 255, 255, 255, 223, 109, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 21, 55, 60, 33, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ô' / 0xf4
        'ô' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 174, 174, 91, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 230, 255, 255, 245, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 8, 198, 255, 142, 119, 255, 222, 22, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 172, 255, 133, 0, 0, 109, 252, 204, 15, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 61, 163, 89, 0, 0, 0, 0, 67, 163, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 142, 173, 176, 146, 72, 2, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 26, 195, 255, 255, 255, 255, 255, 255, 212, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 222, 255, 242, 132, 72, 72, 123, 232, 255, 240, 33, 0, 0, 0,
            ],
            &[
                0, 0, 151, 255, 243, 45, 0, 0, 0, 0, 29, 225, 255, 193, 0, 0, 0,
            ],
            &[
                0, 9, 243, 255, 128, 0, 0, 0, 0, 0, 0, 91, 255, 255, 41, 0, 0,
            ],
            &[
                0, 73, 255, 255, 39, 0, 0, 0, 0, 0, 0, 10, 249, 255, 121, 0, 0,
            ],
            &[
                0, 103, 255, 247, 1, 0, 0, 0, 0, 0, 0, 0, 212, 255, 151, 0, 0,
            ],
            &[
                0, 125, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 195, 255, 175, 0, 0,
            ],
            &[
                0, 117, 255, 241, 0, 0, 0, 0, 0, 0, 0, 0, 203, 255, 170, 0, 0,
            ],
            &[
                0, 94, 255, 255, 16, 0, 0, 0, 0, 0, 0, 0, 232, 255, 147, 0, 0,
            ],
            &[
                0, 37, 255, 255, 82, 0, 0, 0, 0, 0, 0, 42, 255, 255, 91, 0, 0,
            ],
            &[
                0, 0, 205, 255, 201, 2, 0, 0, 0, 0, 0, 160, 255, 244, 17, 0, 0,
            ],
            &[
                0, 0, 72, 254, 255, 165, 18, 0, 0, 7, 128, 255, 255, 129, 0, 0, 0,
            ],
            &[
                0, 0, 0, 119, 254, 255, 244, 199, 193, 236, 255, 255, 176, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 71, 199, 254, 255, 255, 255, 223, 109, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 21, 55, 60, 33, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'õ' / 0xf5
        'õ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 47, 171, 191, 119, 11, 0, 0, 125, 187, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 26, 243, 255, 255, 255, 229, 115, 111, 242, 161, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 130, 252, 84, 44, 148, 251, 255, 255, 241, 38, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 112, 123, 0, 0, 0, 40, 128, 130, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 142, 173, 176, 146, 72, 2, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 26, 195, 255, 255, 255, 255, 255, 255, 212, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 222, 255, 242, 132, 72, 72, 123, 232, 255, 240, 33, 0, 0, 0,
            ],
            &[
                0, 0, 151, 255, 243, 45, 0, 0, 0, 0, 29, 225, 255, 193, 0, 0, 0,
            ],
            &[
                0, 9, 243, 255, 128, 0, 0, 0, 0, 0, 0, 91, 255, 255, 41, 0, 0,
            ],
            &[
                0, 73, 255, 255, 39, 0, 0, 0, 0, 0, 0, 10, 249, 255, 121, 0, 0,
            ],
            &[
                0, 103, 255, 247, 1, 0, 0, 0, 0, 0, 0, 0, 212, 255, 151, 0, 0,
            ],
            &[
                0, 125, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 195, 255, 175, 0, 0,
            ],
            &[
                0, 117, 255, 241, 0, 0, 0, 0, 0, 0, 0, 0, 203, 255, 170, 0, 0,
            ],
            &[
                0, 94, 255, 255, 16, 0, 0, 0, 0, 0, 0, 0, 232, 255, 147, 0, 0,
            ],
            &[
                0, 37, 255, 255, 82, 0, 0, 0, 0, 0, 0, 42, 255, 255, 91, 0, 0,
            ],
            &[
                0, 0, 205, 255, 201, 2, 0, 0, 0, 0, 0, 160, 255, 244, 17, 0, 0,
            ],
            &[
                0, 0, 72, 254, 255, 165, 18, 0, 0, 7, 128, 255, 255, 129, 0, 0, 0,
            ],
            &[
                0, 0, 0, 119, 254, 255, 244, 199, 193, 236, 255, 255, 176, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 71, 199, 254, 255, 255, 255, 223, 109, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 21, 55, 60, 33, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ö' / 0xf6
        'ö' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 32, 193, 193, 34, 0, 0, 10, 174, 206, 63, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 108, 255, 255, 112, 0, 0, 61, 255, 255, 159, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 55, 245, 245, 57, 0, 0, 21, 232, 252, 98, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 14, 14, 0, 0, 0, 0, 6, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 142, 173, 176, 146, 72, 2, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 26, 195, 255, 255, 255, 255, 255, 255, 212, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 222, 255, 242, 132, 72, 72, 123, 232, 255, 240, 33, 0, 0, 0,
            ],
            &[
                0, 0, 151, 255, 243, 45, 0, 0, 0, 0, 29, 225, 255, 193, 0, 0, 0,
            ],
            &[
                0, 9, 243, 255, 128, 0, 0, 0, 0, 0, 0, 91, 255, 255, 41, 0, 0,
            ],
            &[
                0, 73, 255, 255, 39, 0, 0, 0, 0, 0, 0, 10, 249, 255, 121, 0, 0,
            ],
            &[
                0, 103, 255, 247, 1, 0, 0, 0, 0, 0, 0, 0, 212, 255, 151, 0, 0,
            ],
            &[
                0, 125, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 195, 255, 175, 0, 0,
            ],
            &[
                0, 117, 255, 241, 0, 0, 0, 0, 0, 0, 0, 0, 203, 255, 170, 0, 0,
            ],
            &[
                0, 94, 255, 255, 16, 0, 0, 0, 0, 0, 0, 0, 232, 255, 147, 0, 0,
            ],
            &[
                0, 37, 255, 255, 82, 0, 0, 0, 0, 0, 0, 42, 255, 255, 91, 0, 0,
            ],
            &[
                0, 0, 205, 255, 201, 2, 0, 0, 0, 0, 0, 160, 255, 244, 17, 0, 0,
            ],
            &[
                0, 0, 72, 254, 255, 165, 18, 0, 0, 7, 128, 255, 255, 129, 0, 0, 0,
            ],
            &[
                0, 0, 0, 119, 254, 255, 244, 199, 193, 236, 255, 255, 176, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 71, 199, 254, 255, 255, 255, 223, 109, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 21, 55, 60, 33, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '÷' / 0xf7
        '÷' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 141, 152, 26, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 97, 255, 255, 128, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 79, 255, 255, 110, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 80, 93, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 102, 121, 121, 121, 121, 121, 121, 121, 121, 121, 121, 121, 4, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 94, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 4, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 88, 99, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 84, 255, 255, 115, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 94, 255, 255, 125, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 133, 146, 21, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ø' / 0xf8
        'ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 18, 4, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 61, 142, 173, 176, 146, 72, 5, 192, 207, 32, 0, 0,
            ],
            &[
                0, 0, 0, 26, 195, 255, 255, 255, 255, 255, 255, 255, 255, 209, 7, 0, 0,
            ],
            &[
                0, 0, 14, 222, 255, 242, 132, 72, 72, 123, 255, 255, 255, 77, 0, 0, 0,
            ],
            &[
                0, 0, 151, 255, 243, 45, 0, 0, 0, 12, 248, 255, 255, 193, 0, 0, 0,
            ],
            &[
                0, 9, 243, 255, 128, 0, 0, 0, 0, 158, 255, 255, 255, 255, 41, 0, 0,
            ],
            &[
                0, 73, 255, 255, 39, 0, 0, 0, 85, 255, 231, 32, 249, 255, 121, 0, 0,
            ],
            &[
                0, 103, 255, 247, 1, 0, 0, 29, 238, 254, 71, 0, 212, 255, 151, 0, 0,
            ],
            &[
                0, 125, 255, 231, 0, 0, 2, 192, 255, 143, 0, 0, 195, 255, 175, 0, 0,
            ],
            &[
                0, 117, 255, 241, 0, 0, 121, 255, 209, 7, 0, 0, 203, 255, 170, 0, 0,
            ],
            &[
                0, 94, 255, 255, 16, 53, 250, 246, 42, 0, 0, 0, 232, 255, 147, 0, 0,
            ],
            &[
                0, 37, 255, 255, 94, 219, 255, 106, 0, 0, 0, 42, 255, 255, 91, 0, 0,
            ],
            &[
                0, 0, 205, 255, 255, 255, 179, 0, 0, 0, 0, 160, 255, 244, 17, 0, 0,
            ],
            &[
                0, 0, 72, 255, 255, 255, 39, 0, 0, 7, 128, 255, 255, 129, 0, 0, 0,
            ],
            &[
                0, 0, 29, 255, 255, 255, 244, 199, 193, 236, 255, 255, 176, 3, 0, 0, 0,
            ],
            &[
                0, 0, 187, 255, 213, 199, 254, 255, 255, 255, 223, 109, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 49, 166, 7, 0, 21, 55, 60, 33, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ù' / 0xf9
        'ù' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 174, 174, 102, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 185, 255, 244, 26, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 204, 255, 177, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 17, 202, 255, 98, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 145, 152, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 117, 120, 54, 0, 0, 0, 0, 0, 0, 47, 120, 120, 4, 0, 0],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 117, 0, 0, 0, 0, 0, 0, 126, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 243, 255, 137, 0, 0, 0, 0, 0, 0, 175, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 219, 255, 190, 0, 0, 0, 0, 0, 24, 247, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 154, 255, 255, 98, 0, 0, 0, 29, 198, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 37, 244, 255, 255, 224, 189, 205, 250, 246, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 54, 204, 255, 255, 255, 255, 199, 63, 5, 249, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 59, 55, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ú' / 0xfa
        'ú' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 110, 174, 174, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 36, 249, 255, 175, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 189, 255, 195, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 107, 255, 197, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 154, 141, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 117, 120, 54, 0, 0, 0, 0, 0, 0, 47, 120, 120, 4, 0, 0],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 117, 0, 0, 0, 0, 0, 0, 126, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 243, 255, 137, 0, 0, 0, 0, 0, 0, 175, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 219, 255, 190, 0, 0, 0, 0, 0, 24, 247, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 154, 255, 255, 98, 0, 0, 0, 29, 198, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 37, 244, 255, 255, 224, 189, 205, 250, 246, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 54, 204, 255, 255, 255, 255, 199, 63, 5, 249, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 59, 55, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'û' / 0xfb
        'û' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 174, 174, 91, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 230, 255, 255, 245, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 8, 198, 255, 142, 119, 255, 222, 22, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 172, 255, 133, 0, 0, 109, 252, 204, 15, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 61, 163, 89, 0, 0, 0, 0, 67, 163, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 117, 120, 54, 0, 0, 0, 0, 0, 0, 47, 120, 120, 4, 0, 0],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 117, 0, 0, 0, 0, 0, 0, 126, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 243, 255, 137, 0, 0, 0, 0, 0, 0, 175, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 219, 255, 190, 0, 0, 0, 0, 0, 24, 247, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 154, 255, 255, 98, 0, 0, 0, 29, 198, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 37, 244, 255, 255, 224, 189, 205, 250, 246, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 54, 204, 255, 255, 255, 255, 199, 63, 5, 249, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 59, 55, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ü' / 0xfc
        'ü' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 32, 193, 193, 34, 0, 0, 10, 174, 206, 63, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 108, 255, 255, 112, 0, 0, 61, 255, 255, 159, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 55, 245, 245, 57, 0, 0, 21, 232, 252, 98, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 14, 14, 0, 0, 0, 0, 6, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 117, 120, 54, 0, 0, 0, 0, 0, 0, 47, 120, 120, 4, 0, 0],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 117, 0, 0, 0, 0, 0, 0, 126, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 243, 255, 137, 0, 0, 0, 0, 0, 0, 175, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 219, 255, 190, 0, 0, 0, 0, 0, 24, 247, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 154, 255, 255, 98, 0, 0, 0, 29, 198, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 37, 244, 255, 255, 224, 189, 205, 250, 246, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 54, 204, 255, 255, 255, 255, 199, 63, 5, 249, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 59, 55, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ý' / 0xfd
        'ý' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 110, 174, 174, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 36, 249, 255, 175, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 189, 255, 195, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 107, 255, 197, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 154, 141, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 82, 120, 82, 0, 0, 0, 0, 0, 0, 0, 0, 73, 120, 105, 0, 0],
            &[
                0, 97, 255, 241, 11, 0, 0, 0, 0, 0, 0, 0, 217, 255, 152, 0, 0,
            ],
            &[
                0, 9, 237, 255, 100, 0, 0, 0, 0, 0, 0, 47, 255, 255, 57, 0, 0,
            ],
            &[
                0, 0, 140, 255, 204, 0, 0, 0, 0, 0, 0, 133, 255, 218, 0, 0, 0,
            ],
            &[
                0, 0, 35, 254, 255, 52, 0, 0, 0, 0, 0, 218, 255, 123, 0, 0, 0,
            ],
            &[
                0, 0, 0, 183, 255, 156, 0, 0, 0, 0, 48, 255, 254, 30, 0, 0, 0,
            ],
            &[
                0, 0, 0, 77, 255, 245, 15, 0, 0, 0, 133, 255, 189, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 2, 223, 255, 108, 0, 0, 0, 219, 255, 94, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 120, 255, 212, 0, 0, 49, 255, 243, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 248, 255, 56, 0, 133, 255, 160, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 163, 255, 150, 0, 215, 255, 65, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 255, 233, 43, 255, 224, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 206, 255, 179, 255, 131, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 255, 255, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 239, 255, 197, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 219, 255, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 64, 255, 247, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 178, 255, 155, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 69, 255, 254, 40, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 93, 88, 137, 248, 255, 164, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 69, 255, 255, 255, 255, 192, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 29, 133, 152, 135, 76, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'þ' / 0xfe
        'þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 133, 74, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 214, 255, 141, 0, 52, 139, 175, 175, 130, 38, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 133, 139, 254, 255, 255, 255, 255, 250, 116, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 217, 255, 179, 99, 66, 97, 201, 255, 255, 91, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 146, 0, 0, 0, 0, 4, 194, 255, 230, 4, 0, 0,
            ],
            &[
                0, 0, 214, 255, 246, 19, 0, 0, 0, 0, 0, 76, 255, 255, 73, 0, 0,
            ],
            &[
                0, 0, 214, 255, 190, 0, 0, 0, 0, 0, 0, 3, 248, 255, 130, 0, 0,
            ],
            &[
                0, 0, 214, 255, 163, 0, 0, 0, 0, 0, 0, 0, 225, 255, 164, 0, 0,
            ],
            &[
                0, 0, 214, 255, 145, 0, 0, 0, 0, 0, 0, 0, 202, 255, 178, 0, 0,
            ],
            &[
                0, 0, 214, 255, 161, 0, 0, 0, 0, 0, 0, 0, 205, 255, 167, 0, 0,
            ],
            &[
                0, 0, 214, 255, 184, 0, 0, 0, 0, 0, 0, 0, 230, 255, 147, 0, 0,
            ],
            &[
                0, 0, 214, 255, 234, 4, 0, 0, 0, 0, 0, 26, 255, 255, 100, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 74, 0, 0, 0, 0, 0, 123, 255, 254, 28, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 220, 39, 0, 0, 0, 79, 247, 255, 163, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 159, 234, 253, 204, 185, 219, 255, 255, 218, 24, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 136, 36, 177, 255, 255, 255, 249, 161, 22, 0, 0, 0, 0,
            ],
            &[0, 0, 214, 255, 143, 0, 0, 13, 53, 48, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 214, 255, 144, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 103, 122, 69, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ÿ' / 0xff
        'ÿ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 32, 193, 193, 34, 0, 0, 10, 174, 206, 63, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 108, 255, 255, 112, 0, 0, 61, 255, 255, 159, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 55, 245, 245, 57, 0, 0, 21, 232, 252, 98, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 14, 14, 0, 0, 0, 0, 6, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 82, 120, 82, 0, 0, 0, 0, 0, 0, 0, 0, 73, 120, 105, 0, 0],
            &[
                0, 97, 255, 241, 11, 0, 0, 0, 0, 0, 0, 0, 217, 255, 152, 0, 0,
            ],
            &[
                0, 9, 237, 255, 100, 0, 0, 0, 0, 0, 0, 47, 255, 255, 57, 0, 0,
            ],
            &[
                0, 0, 140, 255, 204, 0, 0, 0, 0, 0, 0, 133, 255, 218, 0, 0, 0,
            ],
            &[
                0, 0, 35, 254, 255, 52, 0, 0, 0, 0, 0, 218, 255, 123, 0, 0, 0,
            ],
            &[
                0, 0, 0, 183, 255, 156, 0, 0, 0, 0, 48, 255, 254, 30, 0, 0, 0,
            ],
            &[
                0, 0, 0, 77, 255, 245, 15, 0, 0, 0, 133, 255, 189, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 2, 223, 255, 108, 0, 0, 0, 219, 255, 94, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 120, 255, 212, 0, 0, 49, 255, 243, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 248, 255, 56, 0, 133, 255, 160, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 163, 255, 150, 0, 215, 255, 65, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 255, 233, 43, 255, 224, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 206, 255, 179, 255, 131, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 255, 255, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 239, 255, 197, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 219, 255, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 64, 255, 247, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 178, 255, 155, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 69, 255, 254, 40, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 93, 88, 137, 248, 255, 164, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 69, 255, 255, 255, 255, 192, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 29, 133, 152, 135, 76, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ā' / 0x100
        'Ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 96, 104, 104, 104, 104, 104, 104, 104, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 236, 255, 255, 255, 255, 255, 255, 255, 24, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 119, 129, 129, 129, 129, 129, 129, 129, 12, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 91, 91, 39, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 255, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 255, 255, 241, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 249, 223, 255, 76, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 113, 255, 196, 151, 255, 159, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 196, 255, 127, 79, 255, 237, 4, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 25, 253, 255, 48, 12, 247, 255, 69, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 106, 255, 224, 0, 0, 180, 255, 153, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 189, 255, 145, 0, 0, 101, 255, 233, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 19, 252, 255, 65, 0, 0, 23, 253, 255, 63, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 98, 255, 238, 4, 0, 0, 0, 198, 255, 146, 0, 0, 0, 0],
            &[
                0, 0, 0, 181, 255, 181, 18, 18, 18, 18, 137, 255, 228, 1, 0, 0, 0,
            ],
            &[
                0, 0, 15, 249, 255, 255, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 198, 22, 22, 22, 22, 22, 22, 153, 255, 222, 0, 0, 0,
            ],
            &[
                0, 10, 246, 255, 100, 0, 0, 0, 0, 0, 0, 56, 255, 255, 50, 0, 0,
            ],
            &[
                0, 83, 255, 253, 23, 0, 0, 0, 0, 0, 0, 1, 231, 255, 133, 0, 0,
            ],
            &[
                0, 166, 255, 197, 0, 0, 0, 0, 0, 0, 0, 0, 153, 255, 216, 0, 0,
            ],
            &[
                7, 242, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0, 74, 255, 255, 44, 0,
            ],
            &[
                76, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 7, 243, 255, 127, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ā' / 0x101
        'ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 121, 154, 154, 154, 154, 154, 154, 154, 35, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 201, 255, 255, 255, 255, 255, 255, 255, 59, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 63, 80, 80, 80, 80, 80, 80, 80, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 108, 159, 180, 183, 161, 106, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 174, 252, 255, 255, 255, 255, 255, 255, 239, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 163, 248, 163, 95, 66, 75, 119, 226, 255, 248, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 27, 20, 0, 0, 0, 0, 0, 25, 235, 255, 132, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 181, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 196, 0, 0],
            &[
                0, 0, 0, 0, 0, 10, 76, 135, 162, 183, 191, 198, 226, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 232, 255, 255, 255, 255, 253, 245, 246, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 88, 255, 255, 194, 90, 36, 11, 0, 0, 113, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 170, 1, 0, 0, 0, 0, 0, 123, 255, 197, 0, 0,
            ],
            &[
                0, 0, 10, 255, 255, 64, 0, 0, 0, 0, 0, 0, 165, 255, 197, 0, 0,
            ],
            &[
                0, 0, 18, 255, 255, 73, 0, 0, 0, 0, 0, 38, 248, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 227, 255, 192, 16, 0, 0, 1, 73, 226, 255, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 243, 195, 190, 230, 255, 212, 121, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 223, 255, 255, 255, 245, 144, 19, 61, 255, 197, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 1, 29, 59, 39, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ă' / 0x102
        'Ă' => &[
            &[0, 0, 0, 7, 199, 130, 0, 0, 0, 0, 92, 201, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 206, 249, 82, 0, 3, 58, 226, 242, 8, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 56, 247, 255, 242, 243, 255, 252, 90, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 34, 150, 193, 194, 155, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 91, 91, 39, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 255, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 255, 255, 241, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 249, 223, 255, 76, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 113, 255, 196, 151, 255, 159, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 196, 255, 127, 79, 255, 237, 4, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 25, 253, 255, 48, 12, 247, 255, 69, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 106, 255, 224, 0, 0, 180, 255, 153, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 189, 255, 145, 0, 0, 101, 255, 233, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 19, 252, 255, 65, 0, 0, 23, 253, 255, 63, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 98, 255, 238, 4, 0, 0, 0, 198, 255, 146, 0, 0, 0, 0],
            &[
                0, 0, 0, 181, 255, 181, 18, 18, 18, 18, 137, 255, 228, 1, 0, 0, 0,
            ],
            &[
                0, 0, 15, 249, 255, 255, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 198, 22, 22, 22, 22, 22, 22, 153, 255, 222, 0, 0, 0,
            ],
            &[
                0, 10, 246, 255, 100, 0, 0, 0, 0, 0, 0, 56, 255, 255, 50, 0, 0,
            ],
            &[
                0, 83, 255, 253, 23, 0, 0, 0, 0, 0, 0, 1, 231, 255, 133, 0, 0,
            ],
            &[
                0, 166, 255, 197, 0, 0, 0, 0, 0, 0, 0, 0, 153, 255, 216, 0, 0,
            ],
            &[
                7, 242, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0, 74, 255, 255, 44, 0,
            ],
            &[
                76, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 7, 243, 255, 127, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ă' / 0x103
        'ă' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 218, 207, 1, 0, 0, 0, 90, 250, 79, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 149, 255, 156, 43, 35, 94, 232, 242, 21, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 14, 205, 255, 255, 255, 255, 243, 83, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 3, 90, 139, 149, 112, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 108, 159, 180, 183, 161, 106, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 174, 252, 255, 255, 255, 255, 255, 255, 239, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 163, 248, 163, 95, 66, 75, 119, 226, 255, 248, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 27, 20, 0, 0, 0, 0, 0, 25, 235, 255, 132, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 181, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 196, 0, 0],
            &[
                0, 0, 0, 0, 0, 10, 76, 135, 162, 183, 191, 198, 226, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 232, 255, 255, 255, 255, 253, 245, 246, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 88, 255, 255, 194, 90, 36, 11, 0, 0, 113, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 170, 1, 0, 0, 0, 0, 0, 123, 255, 197, 0, 0,
            ],
            &[
                0, 0, 10, 255, 255, 64, 0, 0, 0, 0, 0, 0, 165, 255, 197, 0, 0,
            ],
            &[
                0, 0, 18, 255, 255, 73, 0, 0, 0, 0, 0, 38, 248, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 227, 255, 192, 16, 0, 0, 1, 73, 226, 255, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 105, 255, 255, 243, 195, 190, 230, 255, 212, 121, 255, 197, 0, 0,
            ],
            &[
                0, 0, 0, 0, 92, 223, 255, 255, 255, 245, 144, 19, 61, 255, 197, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 1, 29, 59, 39, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ą' / 0x104
        'Ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 91, 91, 39, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 255, 255, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 255, 255, 241, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 249, 223, 255, 76, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 113, 255, 196, 151, 255, 159, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 196, 255, 127, 79, 255, 237, 4, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 25, 253, 255, 48, 12, 247, 255, 69, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 106, 255, 224, 0, 0, 180, 255, 153, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 189, 255, 145, 0, 0, 101, 255, 233, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 19, 252, 255, 65, 0, 0, 23, 253, 255, 63, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 98, 255, 238, 4, 0, 0, 0, 198, 255, 146, 0, 0, 0, 0],
            &[
                0, 0, 0, 181, 255, 181, 18, 18, 18, 18, 137, 255, 228, 1, 0, 0, 0,
            ],
            &[
                0, 0, 15, 249, 255, 255, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0, 0,
            ],
            &[
                0, 0, 174, 255, 198, 22, 22, 22, 22, 22, 22, 153, 255, 222, 0, 0, 0,
            ],
            &[
                0, 10, 246, 255, 100, 0, 0, 0, 0, 0, 0, 56, 255, 255, 50, 0, 0,
            ],
            &[
                0, 83, 255, 253, 23, 0, 0, 0, 0, 0, 0, 1, 231, 255, 133, 0, 0,
            ],
            &[
                0, 166, 255, 197, 0, 0, 0, 0, 0, 0, 0, 0, 153, 255, 216, 0, 0,
            ],
            &[
                7, 242, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0, 74, 255, 255, 44, 0,
            ],
            &[
                76, 255, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 7, 244, 255, 144, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 160, 249, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 110, 255, 100, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 237, 235, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 255, 240, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 226, 255, 218, 174, 66],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 50, 189, 231, 220, 63],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ą' / 0x105
        'ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 26, 108, 159, 180, 183, 161, 106, 18, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 2, 174, 252, 255, 255, 255, 255, 255, 255, 239, 77, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 163, 248, 163, 95, 66, 75, 119, 226, 255, 248, 33, 0, 0, 0,
            ],
            &[0, 0, 0, 27, 20, 0, 0, 0, 0, 0, 25, 235, 255, 132, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 181, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 196, 0, 0, 0],
            &[
                0, 0, 0, 0, 10, 76, 135, 162, 183, 191, 198, 226, 255, 197, 0, 0, 0,
            ],
            &[
                0, 0, 0, 94, 232, 255, 255, 255, 255, 253, 245, 246, 255, 197, 0, 0, 0,
            ],
            &[
                0, 0, 88, 255, 255, 194, 90, 36, 11, 0, 0, 113, 255, 197, 0, 0, 0,
            ],
            &[
                0, 0, 216, 255, 170, 1, 0, 0, 0, 0, 0, 123, 255, 197, 0, 0, 0,
            ],
            &[
                0, 10, 255, 255, 64, 0, 0, 0, 0, 0, 0, 165, 255, 197, 0, 0, 0,
            ],
            &[
                0, 18, 255, 255, 73, 0, 0, 0, 0, 0, 38, 248, 255, 197, 0, 0, 0,
            ],
            &[
                0, 0, 227, 255, 192, 16, 0, 0, 1, 73, 226, 255, 255, 197, 0, 0, 0,
            ],
            &[
                0, 0, 105, 255, 255, 243, 195, 190, 230, 255, 212, 121, 255, 197, 0, 0, 0,
            ],
            &[
                0, 0, 0, 92, 223, 255, 255, 255, 245, 144, 19, 64, 255, 211, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 29, 59, 39, 7, 0, 2, 172, 245, 54, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 123, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 244, 224, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 34, 255, 231, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 234, 255, 214, 175, 56, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 57, 193, 232, 218, 52, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ć' / 0x106
        'Ć' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 104, 255, 255, 113, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 23, 238, 255, 148, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 176, 255, 158, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 64, 212, 131, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 46, 105, 129, 132, 111, 65, 6, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 69, 212, 255, 255, 255, 255, 255, 255, 241, 120, 0, 0,
            ],
            &[
                0, 0, 0, 0, 114, 255, 255, 255, 206, 168, 172, 202, 247, 255, 108, 0, 0,
            ],
            &[
                0, 0, 0, 83, 255, 255, 202, 44, 0, 0, 0, 0, 15, 100, 10, 0, 0,
            ],
            &[0, 0, 11, 231, 255, 210, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 98, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 185, 255, 214, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 239, 255, 143, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 255, 255, 99, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 255, 255, 72, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 107, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 231, 255, 159, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 225, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 255, 255, 90, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 226, 255, 234, 42, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 72, 253, 255, 238, 125, 38, 7, 8, 34, 90, 167, 2, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 249, 255, 255, 255, 254, 255, 255, 255, 255, 3, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 43, 169, 247, 255, 255, 255, 255, 234, 162, 1, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 5, 44, 63, 54, 31, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 119, 124, 120, 0, 0, 0],
        ],
        // letter: 'ć' / 0x107
        'ć' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 152, 174, 170, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 255, 114, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 19, 233, 255, 143, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 169, 255, 148, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 31, 163, 110, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 3, 65, 138, 168, 185, 171, 129, 44, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 49, 211, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0,
            ],
            &[
                0, 0, 0, 40, 244, 255, 252, 179, 118, 100, 123, 173, 250, 44, 0, 0, 0,
            ],
            &[0, 0, 0, 203, 255, 241, 61, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0],
            &[0, 0, 45, 255, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 255, 254, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 255, 224, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 171, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 166, 255, 212, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 235, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 102, 255, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 253, 255, 167, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 152, 255, 255, 158, 31, 0, 0, 0, 16, 89, 71, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 196, 255, 255, 254, 229, 214, 239, 255, 255, 121, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 4, 118, 217, 255, 255, 255, 255, 251, 191, 64, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 27, 53, 59, 36, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĉ' / 0x108
        'Ĉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 65, 253, 255, 255, 146, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 33, 234, 252, 122, 231, 255, 97, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 19, 215, 250, 89, 0, 39, 226, 250, 74, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 151, 206, 61, 0, 0, 0, 21, 178, 204, 28, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 46, 105, 129, 132, 111, 65, 6, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 69, 212, 255, 255, 255, 255, 255, 255, 241, 120, 0, 0,
            ],
            &[
                0, 0, 0, 0, 114, 255, 255, 255, 206, 168, 172, 202, 247, 255, 108, 0, 0,
            ],
            &[
                0, 0, 0, 83, 255, 255, 202, 44, 0, 0, 0, 0, 15, 100, 10, 0, 0,
            ],
            &[0, 0, 11, 231, 255, 210, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 98, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 185, 255, 214, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 239, 255, 143, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 255, 255, 99, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 255, 255, 72, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 107, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 231, 255, 159, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 225, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 255, 255, 90, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 226, 255, 234, 42, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 72, 253, 255, 238, 125, 38, 7, 8, 34, 90, 167, 2, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 249, 255, 255, 255, 254, 255, 255, 255, 255, 3, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 43, 169, 247, 255, 255, 255, 255, 234, 162, 1, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 5, 44, 63, 54, 31, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 90, 124, 122, 8, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĉ' / 0x109
        'ĉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 109, 174, 174, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 62, 252, 255, 255, 213, 11, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 33, 234, 251, 91, 175, 255, 178, 3, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 216, 245, 80, 0, 4, 163, 255, 156, 1, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 101, 162, 50, 0, 0, 0, 0, 106, 163, 54, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 3, 65, 138, 168, 185, 171, 129, 44, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 49, 211, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0,
            ],
            &[
                0, 0, 0, 40, 244, 255, 252, 179, 118, 100, 123, 173, 250, 44, 0, 0, 0,
            ],
            &[0, 0, 0, 203, 255, 241, 61, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0],
            &[0, 0, 45, 255, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 255, 254, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 255, 224, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 171, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 166, 255, 212, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 235, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 102, 255, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 253, 255, 167, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 152, 255, 255, 158, 31, 0, 0, 0, 16, 89, 71, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 196, 255, 255, 254, 229, 214, 239, 255, 255, 121, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 4, 118, 217, 255, 255, 255, 255, 251, 191, 64, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 27, 53, 59, 36, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ċ' / 0x10a
        'Ċ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 198, 255, 246, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 251, 255, 255, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 144, 249, 203, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 46, 105, 129, 132, 111, 65, 6, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 69, 212, 255, 255, 255, 255, 255, 255, 241, 120, 0, 0,
            ],
            &[
                0, 0, 0, 0, 114, 255, 255, 255, 206, 168, 172, 202, 247, 255, 108, 0, 0,
            ],
            &[
                0, 0, 0, 83, 255, 255, 202, 44, 0, 0, 0, 0, 15, 100, 10, 0, 0,
            ],
            &[0, 0, 11, 231, 255, 210, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 98, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 185, 255, 214, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 239, 255, 143, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 255, 255, 99, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 255, 255, 72, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 107, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 231, 255, 159, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 225, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 255, 255, 90, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 226, 255, 234, 42, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 72, 253, 255, 238, 125, 38, 7, 8, 34, 90, 167, 2, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 249, 255, 255, 255, 254, 255, 255, 255, 255, 3, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 43, 169, 247, 255, 255, 255, 255, 234, 162, 1, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 5, 44, 63, 54, 31, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 3, 58, 16, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ċ' / 0x10b
        'ċ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 16, 101, 73, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 180, 255, 255, 93, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 208, 255, 255, 120, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 72, 199, 170, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 3, 65, 138, 168, 185, 171, 129, 44, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 49, 211, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0,
            ],
            &[
                0, 0, 0, 40, 244, 255, 252, 179, 118, 100, 123, 173, 250, 44, 0, 0, 0,
            ],
            &[0, 0, 0, 203, 255, 241, 61, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0],
            &[0, 0, 45, 255, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 255, 254, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 255, 224, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 171, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 166, 255, 212, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 235, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 102, 255, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 253, 255, 167, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 152, 255, 255, 158, 31, 0, 0, 0, 16, 89, 71, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 196, 255, 255, 254, 229, 214, 239, 255, 255, 121, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 4, 118, 217, 255, 255, 255, 255, 251, 191, 64, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 27, 53, 59, 36, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Č' / 0x10c
        'Č' => &[
            &[
                0, 0, 0, 0, 0, 75, 250, 220, 43, 0, 9, 169, 255, 161, 2, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 94, 255, 236, 66, 195, 255, 176, 3, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 130, 255, 255, 255, 203, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 176, 212, 210, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 46, 105, 129, 132, 111, 65, 6, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 69, 212, 255, 255, 255, 255, 255, 255, 241, 120, 0, 0,
            ],
            &[
                0, 0, 0, 0, 114, 255, 255, 255, 206, 168, 172, 202, 247, 255, 108, 0, 0,
            ],
            &[
                0, 0, 0, 83, 255, 255, 202, 44, 0, 0, 0, 0, 15, 100, 10, 0, 0,
            ],
            &[0, 0, 11, 231, 255, 210, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 98, 255, 255, 62, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 185, 255, 214, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 239, 255, 143, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 255, 255, 99, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 50, 255, 255, 72, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 107, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 231, 255, 159, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 164, 255, 225, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 255, 255, 90, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 226, 255, 234, 42, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 72, 253, 255, 238, 125, 38, 7, 8, 34, 90, 167, 2, 0, 0,
            ],
            &[
                0, 0, 0, 0, 94, 249, 255, 255, 255, 254, 255, 255, 255, 255, 3, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 43, 169, 247, 255, 255, 255, 255, 234, 162, 1, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 5, 44, 63, 54, 31, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 106, 118, 14, 0, 0, 0, 0, 91, 124, 29, 0, 0],
        ],
        // letter: 'č' / 0x10d
        'č' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 110, 173, 63, 0, 0, 0, 2, 120, 174, 59, 0, 0, 0],
            &[
                0, 0, 0, 0, 21, 216, 252, 111, 0, 13, 184, 255, 156, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 32, 231, 255, 134, 203, 255, 174, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 55, 251, 255, 255, 207, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 163, 163, 43, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 3, 65, 138, 168, 185, 171, 129, 44, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 49, 211, 255, 255, 255, 255, 255, 255, 255, 114, 0, 0, 0,
            ],
            &[
                0, 0, 0, 40, 244, 255, 252, 179, 118, 100, 123, 173, 250, 44, 0, 0, 0,
            ],
            &[0, 0, 0, 203, 255, 241, 61, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0],
            &[0, 0, 45, 255, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 255, 254, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 255, 224, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 171, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 166, 255, 212, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 143, 255, 235, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 102, 255, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 253, 255, 167, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 152, 255, 255, 158, 31, 0, 0, 0, 16, 89, 71, 0, 0, 0,
            ],
            &[
                0, 0, 0, 10, 196, 255, 255, 254, 229, 214, 239, 255, 255, 121, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 4, 118, 217, 255, 255, 255, 255, 251, 191, 64, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 27, 53, 59, 36, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ď' / 0x10e
        'Ď' => &[
            &[
                0, 0, 0, 125, 255, 190, 18, 0, 29, 204, 255, 109, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 148, 255, 210, 62, 226, 255, 125, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 182, 255, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 202, 212, 192, 11, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 71, 71, 69, 57, 22, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 214, 113, 8, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 225, 211, 218, 245, 255, 255, 255, 214, 35, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 47, 154, 255, 255, 227, 23, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 89, 254, 255, 171, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 151, 255, 255, 41, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 28, 255, 255, 130, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 207, 255, 190, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 163, 255, 227, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 143, 255, 244, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 132, 255, 248, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 146, 255, 236, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 0, 178, 255, 207, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 2, 232, 255, 160, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 71, 255, 255, 88, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 5, 208, 255, 233, 8, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 13, 178, 255, 255, 98, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 100, 27, 41, 69, 134, 235, 255, 255, 152, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 246, 118, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 249, 231, 189, 118, 22, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 124, 103, 3, 0, 0, 0, 7, 111, 123, 4, 0, 0, 0, 0],
        ],
        // letter: 'ď' / 0x10f
        'ď' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 58, 133, 127, 0, 7, 133, 133, 40],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 245, 0, 46, 255, 251, 19,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 245, 0, 97, 255, 164, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 245, 0, 158, 251, 42, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 245, 0, 176, 128, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 255, 245, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 66, 149, 179, 159, 106, 6, 110, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[
                0, 5, 169, 255, 255, 255, 255, 255, 208, 106, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[
                0, 138, 255, 255, 189, 84, 73, 145, 252, 229, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[
                24, 247, 255, 186, 3, 0, 0, 0, 115, 255, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[
                96, 255, 255, 55, 0, 0, 0, 0, 6, 235, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[
                155, 255, 239, 1, 0, 0, 0, 0, 0, 171, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[
                178, 255, 200, 0, 0, 0, 0, 0, 0, 132, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[
                195, 255, 185, 0, 0, 0, 0, 0, 0, 117, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[
                190, 255, 191, 0, 0, 0, 0, 0, 0, 119, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[
                172, 255, 214, 0, 0, 0, 0, 0, 0, 144, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[
                129, 255, 244, 9, 0, 0, 0, 0, 0, 186, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[
                62, 255, 255, 90, 0, 0, 0, 0, 32, 253, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[
                1, 212, 255, 232, 45, 0, 0, 23, 206, 255, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[
                0, 58, 247, 255, 253, 210, 204, 250, 252, 130, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 59, 209, 255, 255, 255, 223, 74, 27, 255, 245, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 34, 62, 43, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Đ' / 0x110
        'Đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 71, 71, 71, 70, 61, 34, 1, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 233, 145, 26, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 237, 211, 214, 238, 255, 255, 255, 240, 78, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 25, 120, 242, 255, 251, 68, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 43, 231, 255, 228, 11, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 82, 255, 255, 110, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 214, 255, 199, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 138, 255, 252, 7, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 94, 255, 255, 40, 0,
            ],
            &[
                149, 149, 255, 255, 255, 149, 149, 132, 0, 0, 0, 0, 73, 255, 255, 57, 0,
            ],
            &[
                255, 255, 255, 255, 255, 255, 255, 226, 0, 0, 0, 0, 63, 255, 255, 61, 0,
            ],
            &[
                133, 133, 255, 255, 255, 133, 133, 118, 0, 0, 0, 0, 77, 255, 255, 49, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 109, 255, 255, 20, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 165, 255, 229, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 13, 244, 255, 157, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 145, 255, 255, 55, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 2, 121, 255, 255, 167, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 162, 27, 35, 57, 114, 210, 255, 255, 207, 14, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 164, 15, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 252, 238, 202, 140, 46, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'đ' / 0x111
        'đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 48, 133, 133, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 255, 255, 10, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 3, 23, 23, 23, 23, 115, 255, 255, 33, 23, 4, 0,
            ],
            &[
                0, 0, 0, 0, 0, 36, 255, 255, 255, 255, 255, 255, 255, 255, 255, 51, 0,
            ],
            &[
                0, 0, 0, 0, 0, 25, 177, 177, 177, 177, 255, 255, 255, 187, 177, 35, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 255, 255, 10, 0, 0, 0],
            &[
                0, 0, 0, 1, 69, 121, 146, 122, 67, 0, 90, 255, 255, 10, 0, 0, 0,
            ],
            &[
                0, 0, 44, 216, 255, 255, 255, 255, 255, 184, 84, 255, 255, 10, 0, 0, 0,
            ],
            &[
                0, 32, 240, 255, 252, 163, 104, 114, 189, 255, 212, 255, 255, 10, 0, 0, 0,
            ],
            &[
                0, 173, 255, 251, 69, 0, 0, 0, 0, 140, 255, 255, 255, 10, 0, 0, 0,
            ],
            &[
                15, 251, 255, 152, 0, 0, 0, 0, 0, 7, 234, 255, 255, 10, 0, 0, 0,
            ],
            &[
                78, 255, 255, 63, 0, 0, 0, 0, 0, 0, 159, 255, 255, 10, 0, 0, 0,
            ],
            &[
                106, 255, 255, 16, 0, 0, 0, 0, 0, 0, 115, 255, 255, 10, 0, 0, 0,
            ],
            &[
                125, 255, 253, 0, 0, 0, 0, 0, 0, 0, 98, 255, 255, 10, 0, 0, 0,
            ],
            &[
                120, 255, 254, 6, 0, 0, 0, 0, 0, 0, 98, 255, 255, 10, 0, 0, 0,
            ],
            &[
                101, 255, 255, 32, 0, 0, 0, 0, 0, 0, 125, 255, 255, 10, 0, 0, 0,
            ],
            &[
                54, 255, 255, 79, 0, 0, 0, 0, 0, 0, 169, 255, 255, 10, 0, 0, 0,
            ],
            &[
                4, 232, 255, 186, 0, 0, 0, 0, 0, 26, 250, 255, 255, 10, 0, 0, 0,
            ],
            &[
                0, 123, 255, 255, 123, 3, 0, 0, 28, 206, 255, 255, 255, 10, 0, 0, 0,
            ],
            &[
                0, 5, 192, 255, 255, 233, 188, 211, 252, 252, 119, 255, 255, 10, 0, 0, 0,
            ],
            &[
                0, 0, 9, 137, 241, 255, 255, 255, 217, 73, 8, 254, 255, 10, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 5, 47, 60, 27, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ē' / 0x112
        'Ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 79, 104, 104, 104, 104, 104, 104, 104, 27, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 194, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 98, 129, 129, 129, 129, 129, 129, 129, 33, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 30, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 64, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 225, 225, 225, 225, 225, 225, 225, 225, 204, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 38, 38, 38, 38, 38, 38, 38, 38, 15, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 40, 40, 40, 40, 40, 40, 40, 40, 37, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ē' / 0x113
        'ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 125, 154, 154, 154, 154, 154, 154, 154, 31, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 208, 255, 255, 255, 255, 255, 255, 255, 52, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 65, 80, 80, 80, 80, 80, 80, 80, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 119, 160, 182, 158, 105, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 134, 251, 255, 255, 255, 255, 255, 238, 70, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 160, 255, 255, 185, 100, 67, 96, 201, 255, 251, 55, 0, 0, 0,
            ],
            &[
                0, 0, 84, 255, 255, 119, 0, 0, 0, 0, 5, 188, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 198, 255, 189, 0, 0, 0, 0, 0, 0, 52, 255, 255, 46, 0, 0,
            ],
            &[
                0, 27, 254, 255, 93, 0, 0, 0, 0, 0, 0, 0, 242, 255, 109, 0, 0,
            ],
            &[
                0, 64, 255, 255, 196, 184, 184, 184, 184, 184, 184, 184, 246, 255, 133, 0, 0,
            ],
            &[
                0, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[
                0, 79, 255, 255, 81, 57, 57, 57, 57, 57, 57, 57, 57, 57, 33, 0, 0,
            ],
            &[0, 53, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 241, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 235, 29, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 25, 233, 255, 220, 73, 3, 0, 0, 0, 14, 98, 177, 0, 0, 0,
            ],
            &[
                0, 0, 0, 57, 229, 255, 255, 238, 202, 191, 213, 252, 255, 238, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 139, 228, 255, 255, 255, 255, 249, 192, 99, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 28, 53, 58, 32, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĕ' / 0x114
        'Ĕ' => &[
            &[0, 0, 0, 0, 174, 163, 0, 0, 0, 0, 59, 201, 73, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 164, 255, 114, 3, 1, 42, 203, 253, 39, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 234, 255, 247, 239, 255, 255, 128, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 19, 137, 187, 199, 163, 67, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 30, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 64, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 225, 225, 225, 225, 225, 225, 225, 225, 204, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 38, 38, 38, 38, 38, 38, 38, 38, 15, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 40, 40, 40, 40, 40, 40, 40, 40, 37, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĕ' / 0x115
        'ĕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 225, 201, 0, 0, 0, 0, 97, 250, 72, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 156, 255, 151, 41, 36, 96, 234, 239, 17, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 17, 209, 255, 255, 255, 255, 242, 77, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 4, 92, 140, 148, 111, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 119, 160, 182, 158, 105, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 134, 251, 255, 255, 255, 255, 255, 238, 70, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 160, 255, 255, 185, 100, 67, 96, 201, 255, 251, 55, 0, 0, 0,
            ],
            &[
                0, 0, 84, 255, 255, 119, 0, 0, 0, 0, 5, 188, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 198, 255, 189, 0, 0, 0, 0, 0, 0, 52, 255, 255, 46, 0, 0,
            ],
            &[
                0, 27, 254, 255, 93, 0, 0, 0, 0, 0, 0, 0, 242, 255, 109, 0, 0,
            ],
            &[
                0, 64, 255, 255, 196, 184, 184, 184, 184, 184, 184, 184, 246, 255, 133, 0, 0,
            ],
            &[
                0, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[
                0, 79, 255, 255, 81, 57, 57, 57, 57, 57, 57, 57, 57, 57, 33, 0, 0,
            ],
            &[0, 53, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 241, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 235, 29, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 25, 233, 255, 220, 73, 3, 0, 0, 0, 14, 98, 177, 0, 0, 0,
            ],
            &[
                0, 0, 0, 57, 229, 255, 255, 238, 202, 191, 213, 252, 255, 238, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 139, 228, 255, 255, 255, 255, 249, 192, 99, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 28, 53, 58, 32, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ė' / 0x116
        'Ė' => &[
            &[0, 0, 0, 0, 0, 0, 56, 251, 255, 172, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 108, 255, 255, 228, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 214, 245, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 30, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 64, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 225, 225, 225, 225, 225, 225, 225, 225, 204, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 38, 38, 38, 38, 38, 38, 38, 38, 15, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 40, 40, 40, 40, 40, 40, 40, 40, 37, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 21, 54, 1, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ė' / 0x117
        'ė' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 72, 101, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 90, 255, 255, 183, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 118, 255, 255, 210, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 169, 200, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 119, 160, 182, 158, 105, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 134, 251, 255, 255, 255, 255, 255, 238, 70, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 160, 255, 255, 185, 100, 67, 96, 201, 255, 251, 55, 0, 0, 0,
            ],
            &[
                0, 0, 84, 255, 255, 119, 0, 0, 0, 0, 5, 188, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 198, 255, 189, 0, 0, 0, 0, 0, 0, 52, 255, 255, 46, 0, 0,
            ],
            &[
                0, 27, 254, 255, 93, 0, 0, 0, 0, 0, 0, 0, 242, 255, 109, 0, 0,
            ],
            &[
                0, 64, 255, 255, 196, 184, 184, 184, 184, 184, 184, 184, 246, 255, 133, 0, 0,
            ],
            &[
                0, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[
                0, 79, 255, 255, 81, 57, 57, 57, 57, 57, 57, 57, 57, 57, 33, 0, 0,
            ],
            &[0, 53, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 241, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 235, 29, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 25, 233, 255, 220, 73, 3, 0, 0, 0, 14, 98, 177, 0, 0, 0,
            ],
            &[
                0, 0, 0, 57, 229, 255, 255, 238, 202, 191, 213, 252, 255, 238, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 139, 228, 255, 255, 255, 255, 249, 192, 99, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 28, 53, 58, 32, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ę' / 0x118
        'Ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 30, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 64, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 225, 225, 225, 225, 225, 225, 225, 225, 204, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 38, 38, 38, 38, 38, 38, 38, 38, 15, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 40, 40, 40, 40, 40, 40, 40, 40, 37, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 13, 202, 229, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 164, 251, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 39, 255, 183, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 76, 255, 194, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 31, 250, 255, 201, 179, 25, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 82, 204, 233, 212, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ę' / 0x119
        'ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 119, 160, 182, 158, 105, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 134, 251, 255, 255, 255, 255, 255, 238, 70, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 160, 255, 255, 185, 100, 67, 96, 201, 255, 251, 55, 0, 0, 0,
            ],
            &[
                0, 0, 84, 255, 255, 119, 0, 0, 0, 0, 5, 188, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 198, 255, 189, 0, 0, 0, 0, 0, 0, 52, 255, 255, 46, 0, 0,
            ],
            &[
                0, 27, 254, 255, 93, 0, 0, 0, 0, 0, 0, 0, 242, 255, 109, 0, 0,
            ],
            &[
                0, 64, 255, 255, 196, 184, 184, 184, 184, 184, 184, 184, 246, 255, 133, 0, 0,
            ],
            &[
                0, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[
                0, 79, 255, 255, 81, 57, 57, 57, 57, 57, 57, 57, 57, 57, 33, 0, 0,
            ],
            &[0, 53, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 241, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 235, 29, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 25, 233, 255, 220, 73, 3, 0, 0, 0, 14, 98, 177, 0, 0, 0,
            ],
            &[
                0, 0, 0, 57, 229, 255, 255, 238, 202, 191, 213, 252, 255, 238, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 139, 228, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 28, 53, 58, 120, 255, 123, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 53, 248, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 162, 255, 78, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 184, 255, 92, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 122, 255, 244, 182, 130, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 141, 219, 231, 136, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ě' / 0x11a
        'Ě' => &[
            &[
                0, 0, 0, 4, 176, 255, 150, 3, 0, 59, 230, 245, 63, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 8, 195, 255, 171, 80, 246, 251, 74, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 18, 222, 255, 255, 255, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 58, 212, 212, 157, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 30, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 64, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 225, 225, 225, 225, 225, 225, 225, 225, 204, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 38, 38, 38, 38, 38, 38, 38, 38, 15, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0,
            ],
            &[0, 0, 111, 255, 254, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 111, 255, 254, 40, 40, 40, 40, 40, 40, 40, 40, 37, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 35, 124, 79, 0, 0, 0, 0, 23, 122, 100, 0, 0, 0, 0],
        ],
        // letter: 'ě' / 0x11b
        'ě' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 48, 174, 122, 2, 0, 0, 0, 62, 173, 121, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 148, 255, 185, 12, 0, 111, 252, 220, 26, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 2, 172, 255, 199, 138, 255, 232, 34, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 8, 207, 255, 255, 251, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 163, 163, 100, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 26, 119, 160, 182, 158, 105, 13, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 134, 251, 255, 255, 255, 255, 255, 238, 70, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 160, 255, 255, 185, 100, 67, 96, 201, 255, 251, 55, 0, 0, 0,
            ],
            &[
                0, 0, 84, 255, 255, 119, 0, 0, 0, 0, 5, 188, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 198, 255, 189, 0, 0, 0, 0, 0, 0, 52, 255, 255, 46, 0, 0,
            ],
            &[
                0, 27, 254, 255, 93, 0, 0, 0, 0, 0, 0, 0, 242, 255, 109, 0, 0,
            ],
            &[
                0, 64, 255, 255, 196, 184, 184, 184, 184, 184, 184, 184, 246, 255, 133, 0, 0,
            ],
            &[
                0, 88, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[
                0, 79, 255, 255, 81, 57, 57, 57, 57, 57, 57, 57, 57, 57, 33, 0, 0,
            ],
            &[0, 53, 255, 255, 59, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 241, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 235, 29, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 25, 233, 255, 220, 73, 3, 0, 0, 0, 14, 98, 177, 0, 0, 0,
            ],
            &[
                0, 0, 0, 57, 229, 255, 255, 238, 202, 191, 213, 252, 255, 238, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 139, 228, 255, 255, 255, 255, 249, 192, 99, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 28, 53, 58, 32, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => &[
            &[0, 0, 0, 0, 0, 0, 65, 253, 255, 255, 146, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 33, 234, 252, 122, 231, 255, 97, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 19, 214, 250, 89, 0, 39, 226, 250, 74, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 151, 206, 61, 0, 0, 0, 21, 178, 204, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 86, 124, 136, 125, 87, 19, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 24, 163, 251, 255, 255, 255, 255, 255, 248, 164, 3, 0, 0,
            ],
            &[
                0, 0, 0, 41, 230, 255, 255, 243, 198, 171, 194, 231, 255, 180, 0, 0, 0,
            ],
            &[
                0, 0, 13, 222, 255, 253, 128, 15, 0, 0, 0, 1, 74, 53, 0, 0, 0,
            ],
            &[0, 0, 134, 255, 255, 99, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 239, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 255, 255, 75, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 255, 251, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 166, 255, 216, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 189, 255, 197, 0, 0, 0, 0, 25, 45, 45, 45, 45, 45, 14, 0, 0,
            ],
            &[
                0, 198, 255, 183, 0, 0, 0, 0, 140, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 192, 255, 194, 0, 0, 0, 0, 137, 251, 251, 251, 255, 255, 79, 0, 0,
            ],
            &[0, 175, 255, 214, 0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0],
            &[
                0, 142, 255, 251, 10, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[0, 89, 255, 255, 72, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0],
            &[
                0, 17, 249, 255, 183, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 106, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 22, 232, 255, 253, 155, 40, 3, 0, 3, 55, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 0, 48, 228, 255, 255, 255, 250, 234, 252, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 0, 0, 18, 136, 230, 255, 255, 255, 255, 255, 210, 132, 25, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 32, 59, 64, 50, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 89, 124, 122, 9, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĝ' / 0x11d
        'ĝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 174, 174, 91, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 230, 255, 255, 245, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 8, 198, 255, 142, 119, 255, 222, 22, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 172, 255, 133, 0, 0, 109, 252, 204, 15, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 61, 163, 89, 0, 0, 0, 0, 67, 163, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 15, 107, 167, 183, 163, 96, 5, 0, 114, 120, 4, 0, 0,
            ],
            &[
                0, 0, 0, 56, 230, 255, 255, 255, 255, 255, 206, 39, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 30, 240, 255, 236, 121, 68, 77, 145, 249, 232, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 166, 255, 242, 36, 0, 0, 0, 0, 74, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 17, 249, 255, 147, 0, 0, 0, 0, 0, 0, 177, 255, 255, 10, 0, 0,
            ],
            &[
                0, 72, 255, 255, 57, 0, 0, 0, 0, 0, 0, 100, 255, 255, 10, 0, 0,
            ],
            &[
                0, 110, 255, 255, 24, 0, 0, 0, 0, 0, 0, 59, 255, 255, 10, 0, 0,
            ],
            &[
                0, 126, 255, 253, 2, 0, 0, 0, 0, 0, 0, 42, 255, 255, 10, 0, 0,
            ],
            &[
                0, 121, 255, 254, 2, 0, 0, 0, 0, 0, 0, 40, 255, 255, 10, 0, 0,
            ],
            &[
                0, 101, 255, 255, 27, 0, 0, 0, 0, 0, 0, 57, 255, 255, 10, 0, 0,
            ],
            &[
                0, 54, 255, 255, 85, 0, 0, 0, 0, 0, 0, 109, 255, 255, 10, 0, 0,
            ],
            &[
                0, 3, 228, 255, 191, 0, 0, 0, 0, 0, 3, 219, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 145, 7, 0, 0, 13, 166, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 3, 179, 255, 255, 240, 191, 202, 245, 255, 190, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 5, 125, 236, 255, 255, 255, 244, 134, 59, 255, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 3, 44, 63, 49, 7, 0, 65, 255, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 97, 255, 239, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 168, 255, 198, 0, 0, 0],
            &[0, 0, 76, 13, 0, 0, 0, 0, 0, 0, 67, 254, 255, 112, 0, 0, 0],
            &[
                0, 0, 194, 246, 182, 130, 98, 88, 107, 169, 251, 255, 207, 7, 0, 0, 0,
            ],
            &[
                0, 0, 137, 242, 255, 255, 255, 255, 255, 255, 255, 175, 22, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 6, 70, 118, 145, 152, 134, 105, 33, 0, 0, 0, 0, 0, 0,
            ],
        ],
        // letter: 'Ğ' / 0x11e
        'Ğ' => &[
            &[0, 0, 0, 0, 87, 201, 49, 0, 0, 0, 0, 173, 160, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 54, 255, 197, 30, 0, 16, 119, 255, 147, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 151, 255, 254, 233, 254, 255, 221, 17, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 87, 171, 203, 179, 122, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 86, 124, 136, 125, 87, 19, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 24, 163, 251, 255, 255, 255, 255, 255, 248, 164, 3, 0, 0,
            ],
            &[
                0, 0, 0, 41, 230, 255, 255, 243, 198, 171, 194, 231, 255, 180, 0, 0, 0,
            ],
            &[
                0, 0, 13, 222, 255, 253, 128, 15, 0, 0, 0, 1, 74, 53, 0, 0, 0,
            ],
            &[0, 0, 134, 255, 255, 99, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 239, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 255, 255, 75, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 255, 251, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 166, 255, 216, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 189, 255, 197, 0, 0, 0, 0, 25, 45, 45, 45, 45, 45, 14, 0, 0,
            ],
            &[
                0, 198, 255, 183, 0, 0, 0, 0, 140, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 192, 255, 194, 0, 0, 0, 0, 137, 251, 251, 251, 255, 255, 79, 0, 0,
            ],
            &[0, 175, 255, 214, 0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0],
            &[
                0, 142, 255, 251, 10, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[0, 89, 255, 255, 72, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0],
            &[
                0, 17, 249, 255, 183, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 106, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 22, 232, 255, 253, 155, 40, 3, 0, 3, 55, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 0, 48, 228, 255, 255, 255, 250, 234, 252, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 0, 0, 18, 136, 230, 255, 255, 255, 255, 255, 210, 132, 25, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 32, 59, 64, 50, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ğ' / 0x11f
        'ğ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 245, 175, 0, 0, 0, 0, 124, 250, 45, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 184, 255, 131, 36, 41, 107, 244, 225, 3, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 32, 223, 255, 255, 255, 255, 234, 57, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 9, 101, 144, 144, 105, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 15, 107, 167, 183, 163, 96, 5, 0, 114, 120, 4, 0, 0,
            ],
            &[
                0, 0, 0, 56, 230, 255, 255, 255, 255, 255, 206, 39, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 30, 240, 255, 236, 121, 68, 77, 145, 249, 232, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 166, 255, 242, 36, 0, 0, 0, 0, 74, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 17, 249, 255, 147, 0, 0, 0, 0, 0, 0, 177, 255, 255, 10, 0, 0,
            ],
            &[
                0, 72, 255, 255, 57, 0, 0, 0, 0, 0, 0, 100, 255, 255, 10, 0, 0,
            ],
            &[
                0, 110, 255, 255, 24, 0, 0, 0, 0, 0, 0, 59, 255, 255, 10, 0, 0,
            ],
            &[
                0, 126, 255, 253, 2, 0, 0, 0, 0, 0, 0, 42, 255, 255, 10, 0, 0,
            ],
            &[
                0, 121, 255, 254, 2, 0, 0, 0, 0, 0, 0, 40, 255, 255, 10, 0, 0,
            ],
            &[
                0, 101, 255, 255, 27, 0, 0, 0, 0, 0, 0, 57, 255, 255, 10, 0, 0,
            ],
            &[
                0, 54, 255, 255, 85, 0, 0, 0, 0, 0, 0, 109, 255, 255, 10, 0, 0,
            ],
            &[
                0, 3, 228, 255, 191, 0, 0, 0, 0, 0, 3, 219, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 145, 7, 0, 0, 13, 166, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 3, 179, 255, 255, 240, 191, 202, 245, 255, 190, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 5, 125, 236, 255, 255, 255, 244, 134, 59, 255, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 3, 44, 63, 49, 7, 0, 65, 255, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 97, 255, 239, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 168, 255, 198, 0, 0, 0],
            &[0, 0, 76, 13, 0, 0, 0, 0, 0, 0, 67, 254, 255, 112, 0, 0, 0],
            &[
                0, 0, 194, 246, 182, 130, 98, 88, 107, 169, 251, 255, 207, 7, 0, 0, 0,
            ],
            &[
                0, 0, 137, 242, 255, 255, 255, 255, 255, 255, 255, 175, 22, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 6, 70, 118, 145, 152, 134, 105, 33, 0, 0, 0, 0, 0, 0,
            ],
        ],
        // letter: 'Ġ' / 0x120
        'Ġ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 197, 255, 246, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 251, 255, 255, 83, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 144, 249, 203, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 86, 124, 136, 125, 87, 19, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 24, 163, 251, 255, 255, 255, 255, 255, 248, 164, 3, 0, 0,
            ],
            &[
                0, 0, 0, 41, 230, 255, 255, 243, 198, 171, 194, 231, 255, 180, 0, 0, 0,
            ],
            &[
                0, 0, 13, 222, 255, 253, 128, 15, 0, 0, 0, 1, 74, 53, 0, 0, 0,
            ],
            &[0, 0, 134, 255, 255, 99, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 239, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 255, 255, 75, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 255, 251, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 166, 255, 216, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 189, 255, 197, 0, 0, 0, 0, 25, 45, 45, 45, 45, 45, 14, 0, 0,
            ],
            &[
                0, 198, 255, 183, 0, 0, 0, 0, 140, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 192, 255, 194, 0, 0, 0, 0, 137, 251, 251, 251, 255, 255, 79, 0, 0,
            ],
            &[0, 175, 255, 214, 0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0],
            &[
                0, 142, 255, 251, 10, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[0, 89, 255, 255, 72, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0],
            &[
                0, 17, 249, 255, 183, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 106, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 22, 232, 255, 253, 155, 40, 3, 0, 3, 55, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 0, 48, 228, 255, 255, 255, 250, 234, 252, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 0, 0, 18, 136, 230, 255, 255, 255, 255, 255, 210, 132, 25, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 32, 59, 64, 50, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 58, 16, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ġ' / 0x121
        'ġ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 81, 94, 11, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 118, 255, 255, 155, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 145, 255, 255, 183, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 179, 192, 57, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 15, 107, 167, 183, 163, 96, 5, 0, 114, 120, 4, 0, 0,
            ],
            &[
                0, 0, 0, 56, 230, 255, 255, 255, 255, 255, 206, 39, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 30, 240, 255, 236, 121, 68, 77, 145, 249, 232, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 166, 255, 242, 36, 0, 0, 0, 0, 74, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 17, 249, 255, 147, 0, 0, 0, 0, 0, 0, 177, 255, 255, 10, 0, 0,
            ],
            &[
                0, 72, 255, 255, 57, 0, 0, 0, 0, 0, 0, 100, 255, 255, 10, 0, 0,
            ],
            &[
                0, 110, 255, 255, 24, 0, 0, 0, 0, 0, 0, 59, 255, 255, 10, 0, 0,
            ],
            &[
                0, 126, 255, 253, 2, 0, 0, 0, 0, 0, 0, 42, 255, 255, 10, 0, 0,
            ],
            &[
                0, 121, 255, 254, 2, 0, 0, 0, 0, 0, 0, 40, 255, 255, 10, 0, 0,
            ],
            &[
                0, 101, 255, 255, 27, 0, 0, 0, 0, 0, 0, 57, 255, 255, 10, 0, 0,
            ],
            &[
                0, 54, 255, 255, 85, 0, 0, 0, 0, 0, 0, 109, 255, 255, 10, 0, 0,
            ],
            &[
                0, 3, 228, 255, 191, 0, 0, 0, 0, 0, 3, 219, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 145, 7, 0, 0, 13, 166, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 3, 179, 255, 255, 240, 191, 202, 245, 255, 190, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 5, 125, 236, 255, 255, 255, 244, 134, 59, 255, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 3, 44, 63, 49, 7, 0, 65, 255, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 97, 255, 239, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 168, 255, 198, 0, 0, 0],
            &[0, 0, 76, 13, 0, 0, 0, 0, 0, 0, 67, 254, 255, 112, 0, 0, 0],
            &[
                0, 0, 194, 246, 182, 130, 98, 88, 107, 169, 251, 255, 207, 7, 0, 0, 0,
            ],
            &[
                0, 0, 137, 242, 255, 255, 255, 255, 255, 255, 255, 175, 22, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 6, 70, 118, 145, 152, 134, 105, 33, 0, 0, 0, 0, 0, 0,
            ],
        ],
        // letter: 'Ģ' / 0x122
        'Ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 86, 124, 136, 125, 87, 19, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 24, 163, 251, 255, 255, 255, 255, 255, 248, 164, 3, 0, 0,
            ],
            &[
                0, 0, 0, 41, 230, 255, 255, 243, 198, 171, 194, 231, 255, 180, 0, 0, 0,
            ],
            &[
                0, 0, 13, 222, 255, 253, 128, 15, 0, 0, 0, 1, 74, 53, 0, 0, 0,
            ],
            &[0, 0, 134, 255, 255, 99, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 239, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 255, 255, 75, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 255, 251, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 166, 255, 216, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 189, 255, 197, 0, 0, 0, 0, 25, 45, 45, 45, 45, 45, 14, 0, 0,
            ],
            &[
                0, 198, 255, 183, 0, 0, 0, 0, 140, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 192, 255, 194, 0, 0, 0, 0, 137, 251, 251, 251, 255, 255, 79, 0, 0,
            ],
            &[0, 175, 255, 214, 0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0],
            &[
                0, 142, 255, 251, 10, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[0, 89, 255, 255, 72, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0],
            &[
                0, 17, 249, 255, 183, 0, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 156, 255, 255, 106, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 22, 232, 255, 253, 155, 40, 3, 0, 3, 55, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 0, 48, 228, 255, 255, 255, 250, 234, 252, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 0, 0, 0, 18, 136, 230, 255, 255, 255, 255, 255, 210, 132, 25, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 1, 32, 59, 64, 50, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 6, 90, 90, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 48, 255, 252, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 108, 255, 166, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 184, 249, 38, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 243, 119, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ģ' / 0x123
        'ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 127, 138, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 75, 255, 129, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 213, 255, 48, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 78, 255, 235, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 98, 163, 123, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 15, 107, 167, 183, 163, 96, 5, 0, 114, 120, 4, 0, 0,
            ],
            &[
                0, 0, 0, 56, 230, 255, 255, 255, 255, 255, 206, 39, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 30, 240, 255, 236, 121, 68, 77, 145, 249, 232, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 166, 255, 242, 36, 0, 0, 0, 0, 74, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 17, 249, 255, 147, 0, 0, 0, 0, 0, 0, 177, 255, 255, 10, 0, 0,
            ],
            &[
                0, 72, 255, 255, 57, 0, 0, 0, 0, 0, 0, 100, 255, 255, 10, 0, 0,
            ],
            &[
                0, 110, 255, 255, 24, 0, 0, 0, 0, 0, 0, 59, 255, 255, 10, 0, 0,
            ],
            &[
                0, 126, 255, 253, 2, 0, 0, 0, 0, 0, 0, 42, 255, 255, 10, 0, 0,
            ],
            &[
                0, 121, 255, 254, 2, 0, 0, 0, 0, 0, 0, 40, 255, 255, 10, 0, 0,
            ],
            &[
                0, 101, 255, 255, 27, 0, 0, 0, 0, 0, 0, 57, 255, 255, 10, 0, 0,
            ],
            &[
                0, 54, 255, 255, 85, 0, 0, 0, 0, 0, 0, 109, 255, 255, 10, 0, 0,
            ],
            &[
                0, 3, 228, 255, 191, 0, 0, 0, 0, 0, 3, 219, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 145, 7, 0, 0, 13, 166, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 3, 179, 255, 255, 240, 191, 202, 245, 255, 190, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 5, 125, 236, 255, 255, 255, 244, 134, 59, 255, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 3, 44, 63, 49, 7, 0, 65, 255, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 97, 255, 239, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 168, 255, 198, 0, 0, 0],
            &[0, 0, 76, 13, 0, 0, 0, 0, 0, 0, 67, 254, 255, 112, 0, 0, 0],
            &[
                0, 0, 194, 246, 182, 130, 98, 88, 107, 169, 251, 255, 207, 7, 0, 0, 0,
            ],
            &[
                0, 0, 137, 242, 255, 255, 255, 255, 255, 255, 255, 175, 22, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 6, 70, 118, 145, 152, 134, 105, 33, 0, 0, 0, 0, 0, 0,
            ],
        ],
        // letter: 'Ĥ' / 0x124
        'Ĥ' => &[
            &[0, 0, 0, 0, 0, 7, 208, 255, 255, 230, 19, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 163, 255, 184, 166, 255, 194, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 130, 255, 179, 8, 2, 158, 255, 170, 3, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 64, 212, 138, 3, 0, 0, 0, 113, 212, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 8, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 108, 38, 38, 38, 38, 38, 38, 64, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 83, 2, 2, 2, 2, 2, 2, 32, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 124, 124, 57, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĥ' / 0x125
        'ĥ' => &[
            &[
                0, 0, 139, 255, 195, 61, 236, 253, 79, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 109, 255, 184, 12, 0, 45, 225, 246, 62, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[15, 149, 120, 3, 0, 0, 0, 20, 142, 135, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 133, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 214, 255, 149, 0, 18, 119, 164, 182, 151, 71, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 214, 255, 141, 86, 239, 255, 255, 255, 255, 255, 174, 5, 0,
            ],
            &[
                0, 0, 0, 0, 214, 255, 177, 250, 213, 103, 68, 101, 214, 255, 255, 129, 0,
            ],
            &[
                0, 0, 0, 0, 214, 255, 255, 197, 10, 0, 0, 0, 9, 224, 255, 231, 0,
            ],
            &[
                0, 0, 0, 0, 214, 255, 255, 45, 0, 0, 0, 0, 0, 119, 255, 255, 25,
            ],
            &[
                0, 0, 0, 0, 214, 255, 214, 0, 0, 0, 0, 0, 0, 82, 255, 255, 42,
            ],
            &[
                0, 0, 0, 0, 214, 255, 170, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 214, 255, 153, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 182, 255, 255, 255, 117, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ħ' / 0x126
        'Ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 8, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                16, 44, 255, 255, 98, 16, 16, 16, 16, 16, 16, 47, 255, 255, 95, 16, 3,
            ],
            &[
                255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 51,
            ],
            &[
                224, 253, 255, 255, 255, 224, 224, 224, 224, 224, 224, 255, 255, 255, 255, 224, 44,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 108, 38, 38, 38, 38, 38, 38, 64, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 83, 2, 2, 2, 2, 2, 2, 32, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ħ' / 0x127
        'ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 111, 133, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 23, 23, 237, 255, 174, 23, 23, 23, 23, 7, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 177, 177, 255, 255, 255, 177, 177, 177, 177, 60, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 214, 255, 149, 0, 18, 119, 164, 182, 151, 71, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 141, 86, 239, 255, 255, 255, 255, 255, 174, 5, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 177, 250, 213, 103, 68, 101, 214, 255, 255, 129, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 197, 10, 0, 0, 0, 9, 224, 255, 231, 0, 0,
            ],
            &[
                0, 0, 0, 214, 255, 255, 45, 0, 0, 0, 0, 0, 119, 255, 255, 25, 0,
            ],
            &[
                0, 0, 0, 214, 255, 214, 0, 0, 0, 0, 0, 0, 82, 255, 255, 42, 0,
            ],
            &[
                0, 0, 0, 214, 255, 170, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0,
            ],
            &[
                0, 0, 0, 214, 255, 153, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0,
            ],
            &[
                0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0,
            ],
            &[
                0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0,
            ],
            &[
                0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0,
            ],
            &[
                0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0,
            ],
            &[
                0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0,
            ],
            &[
                0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0,
            ],
            &[
                0, 0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĩ' / 0x128
        'Ĩ' => &[
            &[0, 0, 0, 0, 19, 121, 141, 70, 0, 0, 0, 89, 147, 0, 0, 0, 0],
            &[
                0, 0, 0, 12, 222, 255, 255, 255, 190, 66, 62, 228, 178, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 114, 255, 134, 94, 197, 255, 255, 255, 254, 61, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 142, 170, 0, 0, 0, 85, 177, 180, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 16, 0, 0, 0],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[
                0, 0, 3, 113, 136, 159, 191, 255, 255, 204, 163, 140, 117, 23, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 255, 255, 74, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 5, 184, 207, 230, 251, 255, 255, 253, 234, 211, 189, 40, 0, 0, 0,
            ],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĩ' / 0x129
        'ĩ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 47, 171, 191, 119, 11, 0, 0, 125, 187, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 26, 243, 255, 255, 255, 229, 115, 111, 242, 161, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 130, 252, 84, 44, 148, 251, 255, 255, 241, 38, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 112, 123, 0, 0, 0, 40, 128, 130, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 49, 120, 120, 120, 120, 120, 120, 90, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 22, 68, 87, 106, 125, 218, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 255, 192, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 140, 185, 205, 225, 245, 255, 255, 255, 246, 224, 203, 181, 95, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ī' / 0x12a
        'Ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 96, 104, 104, 104, 104, 104, 104, 104, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 236, 255, 255, 255, 255, 255, 255, 255, 24, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 119, 129, 129, 129, 129, 129, 129, 129, 12, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 16, 0, 0, 0],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[
                0, 0, 3, 113, 136, 159, 191, 255, 255, 204, 163, 140, 117, 23, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 255, 255, 74, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 5, 184, 207, 230, 251, 255, 255, 253, 234, 211, 189, 40, 0, 0, 0,
            ],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ī' / 0x12b
        'ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 142, 154, 154, 154, 154, 154, 154, 154, 14, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 236, 255, 255, 255, 255, 255, 255, 255, 24, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 74, 80, 80, 80, 80, 80, 80, 80, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 49, 120, 120, 120, 120, 120, 120, 90, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 22, 68, 87, 106, 125, 218, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 255, 192, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 140, 185, 205, 225, 245, 255, 255, 255, 246, 224, 203, 181, 95, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => &[
            &[0, 0, 0, 7, 199, 130, 0, 0, 0, 0, 92, 201, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 206, 249, 82, 0, 3, 58, 226, 242, 8, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 56, 247, 255, 242, 243, 255, 252, 90, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 34, 150, 193, 194, 155, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 16, 0, 0, 0],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[
                0, 0, 3, 113, 136, 159, 191, 255, 255, 204, 163, 140, 117, 23, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 255, 255, 74, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 5, 184, 207, 230, 251, 255, 255, 253, 234, 211, 189, 40, 0, 0, 0,
            ],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĭ' / 0x12d
        'ĭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 245, 175, 0, 0, 0, 0, 124, 250, 45, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 184, 255, 131, 36, 41, 107, 244, 225, 3, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 32, 223, 255, 255, 255, 255, 234, 57, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 9, 101, 144, 144, 105, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 49, 120, 120, 120, 120, 120, 120, 90, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 22, 68, 87, 106, 125, 218, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 255, 192, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 140, 185, 205, 225, 245, 255, 255, 255, 246, 224, 203, 181, 95, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Į' / 0x12e
        'Į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 16, 0, 0, 0],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[
                0, 0, 3, 113, 136, 159, 191, 255, 255, 204, 163, 140, 117, 23, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 255, 255, 74, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 5, 184, 207, 230, 251, 255, 255, 253, 234, 211, 189, 40, 0, 0, 0,
            ],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 88, 251, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 42, 246, 175, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 163, 255, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 200, 255, 69, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 150, 255, 237, 178, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 17, 159, 223, 230, 125, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'į' / 0x12f
        'į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 61, 106, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 63, 255, 255, 211, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 90, 255, 255, 238, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 158, 205, 91, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 49, 120, 120, 120, 120, 120, 120, 90, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 22, 68, 87, 106, 125, 218, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 255, 192, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 140, 185, 205, 225, 245, 255, 255, 255, 246, 224, 203, 181, 95, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 13, 202, 229, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 164, 251, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 39, 255, 183, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 76, 255, 194, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 31, 250, 255, 201, 179, 25, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 82, 204, 233, 212, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'İ' / 0x130
        'İ' => &[
            &[0, 0, 0, 0, 0, 0, 94, 255, 255, 131, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 149, 255, 255, 186, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 62, 227, 237, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 16, 0, 0, 0],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[
                0, 0, 3, 113, 136, 159, 191, 255, 255, 204, 163, 140, 117, 23, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 255, 255, 74, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 5, 184, 207, 230, 251, 255, 255, 253, 234, 211, 189, 40, 0, 0, 0,
            ],
            &[
                0, 0, 7, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 58, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 33, 44, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ı' / 0x131
        'ı' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 49, 120, 120, 120, 120, 120, 120, 90, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 104, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 22, 68, 87, 106, 125, 218, 255, 191, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 255, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 255, 192, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 140, 185, 205, 225, 245, 255, 255, 255, 246, 224, 203, 181, 95, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĳ' / 0x132
        'Ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 71, 71, 9, 0, 0, 0, 0, 0, 66, 71, 35, 0, 0, 0],
            &[
                0, 0, 76, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 33, 0, 0, 0, 0, 0, 237, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 33, 0, 0, 0, 0, 0, 238, 255, 127, 0, 0, 0,
            ],
            &[
                0, 0, 46, 153, 153, 19, 0, 0, 0, 0, 1, 251, 255, 118, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 42, 255, 255, 93, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 157, 255, 254, 32, 0, 0, 0],
            &[
                0, 0, 48, 142, 77, 28, 5, 4, 44, 157, 255, 255, 183, 0, 0, 0, 0,
            ],
            &[
                0, 0, 62, 255, 255, 255, 255, 254, 255, 255, 255, 227, 27, 0, 0, 0, 0,
            ],
            &[
                0, 0, 33, 188, 248, 255, 255, 255, 255, 240, 154, 19, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 3, 37, 57, 62, 39, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĳ' / 0x133
        'ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 51, 109, 32, 0, 0, 0, 0, 18, 102, 70, 0, 0, 0, 0],
            &[
                0, 0, 38, 253, 255, 236, 3, 0, 0, 0, 184, 255, 255, 86, 0, 0, 0,
            ],
            &[
                0, 0, 61, 255, 255, 254, 12, 0, 0, 0, 209, 255, 255, 112, 0, 0, 0,
            ],
            &[
                0, 0, 3, 144, 209, 111, 0, 0, 0, 0, 74, 201, 169, 20, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 101, 120, 71, 0, 0, 0, 0, 33, 120, 120, 17, 0, 0, 0],
            &[
                0, 0, 0, 215, 255, 150, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 215, 255, 150, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 215, 255, 150, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 215, 255, 150, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 215, 255, 150, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 215, 255, 150, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 215, 255, 150, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 215, 255, 150, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 215, 255, 150, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 215, 255, 150, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 215, 255, 150, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 215, 255, 150, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 215, 255, 150, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0,
            ],
            &[
                0, 0, 0, 215, 255, 150, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 71, 255, 255, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 82, 255, 255, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 125, 255, 254, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 27, 231, 255, 201, 0, 0, 0, 0],
            &[
                0, 0, 0, 114, 187, 141, 114, 114, 152, 238, 255, 253, 70, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 125, 255, 255, 255, 255, 255, 255, 238, 87, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 24, 84, 123, 143, 152, 135, 86, 11, 0, 0, 0, 0, 0, 0,
            ],
        ],
        // letter: 'Ĵ' / 0x134
        'Ĵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 154, 255, 255, 252, 59, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 102, 255, 223, 128, 254, 231, 31, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 73, 251, 218, 30, 0, 103, 252, 214, 20, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 23, 202, 174, 17, 0, 0, 0, 67, 207, 157, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 62, 71, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 223, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 224, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 237, 255, 133, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 254, 255, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 119, 255, 255, 54, 0, 0, 0, 0, 0],
            &[
                0, 58, 141, 67, 29, 2, 20, 116, 249, 255, 214, 1, 0, 0, 0, 0, 0,
            ],
            &[
                0, 76, 255, 255, 255, 254, 255, 255, 255, 245, 56, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 40, 185, 248, 255, 255, 255, 253, 187, 46, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 8, 36, 60, 50, 19, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 11, 123, 124, 87, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĵ' / 0x135
        'ĵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 52, 174, 174, 105, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 13, 218, 255, 255, 251, 56, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 3, 182, 255, 161, 102, 253, 232, 32, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 154, 255, 151, 2, 0, 92, 248, 217, 23, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 48, 163, 102, 0, 0, 0, 0, 54, 162, 107, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 52, 120, 120, 120, 120, 120, 120, 86, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 111, 255, 255, 255, 255, 255, 255, 183, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 24, 69, 88, 107, 126, 222, 255, 183, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 183, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 191, 255, 180, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 230, 255, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 98, 255, 255, 95, 0, 0, 0, 0, 0],
            &[
                0, 0, 89, 189, 139, 112, 118, 173, 254, 255, 216, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 97, 255, 255, 255, 255, 255, 255, 204, 33, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 18, 82, 124, 144, 151, 127, 66, 1, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ķ' / 0x136
        'Ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 19, 71, 71, 35, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 2, 190, 255, 236, 27, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 128, 255, 254, 75, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 66, 253, 255, 141, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 23, 231, 255, 201, 5, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 2, 188, 255, 240, 33, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 125, 255, 255, 85, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 64, 252, 255, 151, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 18, 228, 255, 209, 8, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 168, 255, 255, 83, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 159, 255, 255, 255, 206, 4, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 243, 110, 254, 255, 117, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 252, 74, 0, 156, 255, 245, 33, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 122, 0, 0, 18, 234, 255, 185, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 93, 255, 255, 92, 0, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 188, 255, 235, 19, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 37, 248, 255, 161, 0, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 125, 255, 255, 68, 0, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 6, 215, 255, 220, 9, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 63, 254, 255, 136, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 71, 90, 51, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 235, 255, 88, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 255, 228, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 114, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 195, 185, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ķ' / 0x137
        'ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 133, 133, 35, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 15, 119, 120, 95, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 0, 7, 192, 255, 239, 46, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 2, 172, 255, 245, 59, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 149, 255, 250, 74, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 66, 0, 0, 125, 255, 254, 91, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 61, 0, 100, 255, 255, 109, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 50, 73, 252, 255, 141, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 78, 242, 255, 255, 201, 5, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 231, 251, 123, 246, 255, 141, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 249, 80, 0, 105, 255, 254, 76, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 99, 0, 0, 0, 175, 255, 237, 29, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 18, 227, 255, 196, 4, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 0, 62, 252, 255, 135, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 129, 255, 254, 71, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 3, 196, 255, 234, 26, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 90, 90, 32, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 34, 255, 255, 34, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 94, 255, 180, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 170, 252, 49, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 239, 133, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĸ' / 0x138
        'ĸ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 19, 120, 120, 32, 0, 0, 0, 0, 0, 15, 119, 120, 95, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 0, 7, 192, 255, 239, 46, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 2, 172, 255, 245, 59, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 149, 255, 250, 74, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 66, 0, 0, 125, 255, 254, 91, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 61, 0, 100, 255, 255, 109, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 50, 75, 252, 255, 141, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 80, 243, 255, 255, 201, 5, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 232, 251, 123, 246, 255, 141, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 249, 80, 0, 105, 255, 254, 76, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 99, 0, 0, 0, 175, 255, 237, 29, 0, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 18, 227, 255, 196, 4, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 0, 62, 252, 255, 135, 0, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 129, 255, 254, 71, 0, 0, 0,
            ],
            &[
                0, 42, 255, 255, 67, 0, 0, 0, 0, 0, 3, 196, 255, 234, 26, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĺ' / 0x139
        'Ĺ' => &[
            &[0, 0, 0, 0, 0, 192, 255, 241, 39, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 248, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 237, 248, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 137, 208, 62, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 63, 71, 38, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 229, 255, 165, 61, 61, 61, 61, 61, 61, 61, 55, 0, 0, 0,
            ],
            &[
                0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 124, 124, 78, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĺ' / 0x13a
        'ĺ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 1, 196, 255, 187, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 255, 187, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 144, 128, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 90, 133, 133, 133, 133, 133, 133, 117, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 255, 255, 255, 225, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 30, 60, 77, 95, 112, 196, 255, 225, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 226, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 140, 184, 204, 224, 243, 255, 255, 255, 248, 226, 204, 182, 95, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 41, 251, 255, 166, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ļ' / 0x13b
        'Ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 63, 71, 38, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 229, 255, 165, 61, 61, 61, 61, 61, 61, 61, 55, 0, 0, 0,
            ],
            &[
                0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 23, 90, 90, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 97, 255, 225, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 156, 255, 118, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 229, 228, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 247, 73, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ļ' / 0x13c
        'ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 90, 133, 133, 133, 133, 133, 133, 117, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 255, 255, 255, 225, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 30, 60, 77, 95, 112, 196, 255, 225, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 226, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 140, 184, 204, 224, 243, 255, 255, 255, 248, 226, 204, 182, 95, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 42, 90, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 152, 255, 171, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 212, 255, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 32, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 115, 235, 31, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ľ' / 0x13d
        'Ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 63, 71, 38, 0, 0, 0, 0, 0, 54, 71, 43, 0, 0, 0],
            &[
                0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 221, 255, 106, 0, 0, 0,
            ],
            &[
                0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 15, 254, 244, 18, 0, 0, 0,
            ],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 72, 255, 141, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 141, 235, 17, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 6, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 229, 255, 165, 61, 61, 61, 61, 61, 61, 61, 55, 0, 0, 0,
            ],
            &[
                0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ľ' / 0x13e
        'ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 90, 133, 133, 133, 133, 133, 133, 117, 0, 0, 25, 133, 133, 22, 0, 0,
            ],
            &[
                0, 173, 255, 255, 255, 255, 255, 255, 225, 0, 0, 80, 255, 233, 3, 0, 0,
            ],
            &[
                0, 30, 60, 77, 95, 112, 196, 255, 225, 0, 0, 131, 255, 130, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 192, 240, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 3, 200, 101, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 141, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 140, 184, 204, 224, 243, 255, 255, 255, 248, 226, 204, 182, 95, 0, 0, 0,
            ],
            &[
                0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 63, 71, 38, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 13, 161, 197, 77, 0, 0, 0,
            ],
            &[
                0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 96, 255, 255, 214, 0, 0, 0,
            ],
            &[
                0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 90, 255, 255, 208, 0, 0, 0,
            ],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 7, 134, 172, 57, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 229, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 229, 255, 165, 61, 61, 61, 61, 61, 61, 61, 55, 0, 0, 0,
            ],
            &[
                0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[
                0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 231, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŀ' / 0x140
        'ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                90, 133, 133, 133, 133, 133, 133, 117, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                173, 255, 255, 255, 255, 255, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                30, 60, 77, 95, 112, 196, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 21, 177, 209, 82, 0,
            ],
            &[
                0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 106, 255, 255, 208, 0,
            ],
            &[
                0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 94, 255, 255, 197, 0,
            ],
            &[0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 6, 123, 157, 45, 0],
            &[0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 141, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                140, 184, 204, 224, 243, 255, 255, 255, 248, 226, 204, 182, 95, 0, 0, 0, 0,
            ],
            &[
                214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ł' / 0x141
        'Ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 57, 71, 43, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 255, 157, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 255, 157, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 255, 157, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 255, 157, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 255, 157, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 255, 157, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 255, 157, 0, 0, 32, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 208, 255, 157, 12, 137, 249, 85, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 208, 255, 247, 229, 255, 250, 117, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 255, 255, 255, 255, 182, 39, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 148, 255, 255, 255, 74, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 63, 235, 255, 255, 255, 163, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 23, 239, 156, 231, 255, 157, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 0, 208, 255, 157, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 255, 157, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 255, 157, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 208, 255, 181, 61, 61, 61, 61, 61, 61, 61, 60, 0, 0,
            ],
            &[
                0, 0, 0, 0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 0, 0, 0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ł' / 0x142
        'ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 90, 133, 133, 133, 133, 133, 133, 117, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 173, 255, 255, 255, 255, 255, 255, 225, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 30, 60, 77, 95, 112, 196, 255, 225, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 33, 182, 156, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 140, 255, 255, 246, 255, 216, 11, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 200, 255, 255, 248, 127, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 156, 255, 255, 255, 37, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 95, 234, 255, 255, 255, 226, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 2, 204, 253, 150, 156, 255, 225, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 48, 54, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 255, 225, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 226, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 140, 184, 204, 224, 243, 255, 255, 255, 248, 226, 204, 182, 95, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 148, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ń' / 0x143
        'Ń' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 5, 216, 255, 228, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 123, 255, 239, 47, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 45, 248, 240, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 161, 202, 45, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 71, 11, 0, 0, 0, 0, 0, 2, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 255, 113, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 222, 2, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 239, 221, 255, 81, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 252, 113, 255, 192, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 32, 241, 255, 49, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 37, 142, 255, 160, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 50, 33, 252, 248, 23, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 57, 0, 174, 255, 128, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 61, 255, 232, 7, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 205, 255, 96, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 93, 255, 207, 0, 8, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 6, 230, 255, 63, 5, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 124, 255, 175, 0, 254, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 20, 247, 253, 34, 244, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 155, 255, 143, 230, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 44, 255, 241, 225, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 186, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 74, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 1, 216, 255, 255, 79, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 55, 124, 124, 64, 0, 0, 0, 0],
        ],
        // letter: 'ń' / 0x144
        'ń' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 54, 174, 174, 112, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 200, 255, 232, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 107, 255, 240, 48, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 36, 243, 238, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 102, 161, 41, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 101, 120, 17, 0, 28, 125, 163, 183, 158, 83, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 67, 122, 249, 255, 255, 255, 255, 255, 191, 10, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 187, 255, 187, 92, 65, 92, 190, 255, 255, 144, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 163, 0, 0, 0, 0, 1, 200, 255, 239, 2, 0, 0,
            ],
            &[
                0, 0, 214, 255, 254, 38, 0, 0, 0, 0, 0, 104, 255, 255, 29, 0, 0,
            ],
            &[
                0, 0, 214, 255, 208, 0, 0, 0, 0, 0, 0, 74, 255, 255, 43, 0, 0,
            ],
            &[
                0, 0, 214, 255, 178, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 155, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ņ' / 0x145
        'Ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 71, 11, 0, 0, 0, 0, 0, 2, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 255, 113, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 222, 2, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 239, 221, 255, 81, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 252, 113, 255, 192, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 32, 241, 255, 49, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 37, 142, 255, 160, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 50, 33, 252, 248, 23, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 57, 0, 174, 255, 128, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 61, 255, 232, 7, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 205, 255, 96, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 93, 255, 207, 0, 8, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 6, 230, 255, 63, 5, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 124, 255, 175, 0, 254, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 20, 247, 253, 34, 244, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 155, 255, 143, 230, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 44, 255, 241, 225, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 186, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 74, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 1, 216, 255, 255, 79, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 90, 90, 27, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 255, 252, 23, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 108, 255, 166, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 183, 249, 38, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 243, 120, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ņ' / 0x146
        'ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 101, 120, 17, 0, 28, 125, 163, 183, 158, 83, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 67, 122, 249, 255, 255, 255, 255, 255, 191, 10, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 187, 255, 187, 92, 65, 92, 190, 255, 255, 144, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 163, 0, 0, 0, 0, 1, 200, 255, 239, 2, 0, 0,
            ],
            &[
                0, 0, 214, 255, 254, 38, 0, 0, 0, 0, 0, 104, 255, 255, 29, 0, 0,
            ],
            &[
                0, 0, 214, 255, 208, 0, 0, 0, 0, 0, 0, 74, 255, 255, 43, 0, 0,
            ],
            &[
                0, 0, 214, 255, 178, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 155, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 81, 90, 41, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 253, 255, 61, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 255, 207, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 142, 255, 73, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 222, 160, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ň' / 0x147
        'Ň' => &[
            &[
                0, 0, 0, 5, 182, 255, 144, 2, 0, 63, 233, 243, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 10, 200, 255, 166, 84, 248, 250, 68, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 21, 226, 255, 255, 255, 96, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 64, 212, 212, 151, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 71, 11, 0, 0, 0, 0, 0, 2, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 255, 113, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 222, 2, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 239, 221, 255, 81, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 252, 113, 255, 192, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 32, 241, 255, 49, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 37, 142, 255, 160, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 50, 33, 252, 248, 23, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 57, 0, 174, 255, 128, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 61, 255, 232, 7, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 205, 255, 96, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 93, 255, 207, 0, 8, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 6, 230, 255, 63, 5, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 124, 255, 175, 0, 254, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 20, 247, 253, 34, 244, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 155, 255, 143, 230, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 44, 255, 241, 225, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 186, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 74, 255, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 1, 216, 255, 255, 79, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 38, 124, 76, 0, 0, 0, 0, 26, 123, 97, 0, 0, 0, 0],
        ],
        // letter: 'ň' / 0x148
        'ň' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 170, 148, 13, 0, 0, 0, 33, 165, 158, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 94, 254, 218, 35, 0, 68, 239, 244, 58, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 119, 255, 229, 114, 249, 250, 71, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 161, 255, 255, 255, 105, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 157, 163, 135, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 101, 120, 17, 0, 28, 125, 163, 183, 158, 83, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 67, 122, 249, 255, 255, 255, 255, 255, 191, 10, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 187, 255, 187, 92, 65, 92, 190, 255, 255, 144, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 163, 0, 0, 0, 0, 1, 200, 255, 239, 2, 0, 0,
            ],
            &[
                0, 0, 214, 255, 254, 38, 0, 0, 0, 0, 0, 104, 255, 255, 29, 0, 0,
            ],
            &[
                0, 0, 214, 255, 208, 0, 0, 0, 0, 0, 0, 74, 255, 255, 43, 0, 0,
            ],
            &[
                0, 0, 214, 255, 178, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 155, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŉ' / 0x149
        'ŉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 71, 71, 59, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 40, 255, 255, 203, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 85, 255, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 255, 255, 40, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 186, 255, 201, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                1, 238, 255, 102, 52, 120, 66, 0, 8, 105, 156, 183, 163, 91, 4, 0, 0,
            ],
            &[
                41, 255, 239, 10, 111, 255, 171, 37, 227, 255, 255, 255, 255, 255, 199, 13, 0,
            ],
            &[
                26, 86, 59, 0, 111, 255, 223, 212, 225, 109, 66, 90, 194, 255, 255, 149, 0,
            ],
            &[
                0, 0, 0, 0, 111, 255, 255, 234, 24, 0, 0, 0, 2, 206, 255, 241, 2,
            ],
            &[
                0, 0, 0, 0, 111, 255, 255, 136, 0, 0, 0, 0, 0, 106, 255, 255, 30,
            ],
            &[
                0, 0, 0, 0, 111, 255, 255, 54, 0, 0, 0, 0, 0, 74, 255, 255, 43,
            ],
            &[
                0, 0, 0, 0, 111, 255, 255, 25, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 111, 255, 255, 3, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[
                0, 0, 0, 0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 71, 11, 0, 0, 0, 0, 0, 2, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 255, 113, 0, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 255, 222, 2, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 239, 221, 255, 81, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 252, 113, 255, 192, 0, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 32, 241, 255, 49, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 37, 142, 255, 160, 0, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 50, 33, 252, 248, 23, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 57, 0, 174, 255, 128, 0, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 61, 255, 232, 7, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 205, 255, 96, 0, 9, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 93, 255, 207, 0, 8, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 6, 230, 255, 63, 5, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 124, 255, 175, 0, 254, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 20, 247, 253, 34, 244, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 155, 255, 143, 230, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 44, 255, 241, 225, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 186, 255, 255, 255, 96, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 74, 255, 255, 255, 158, 0, 0,
            ],
            &[
                0, 28, 255, 255, 61, 0, 0, 0, 0, 0, 1, 216, 255, 255, 158, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 255, 255, 67, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 75, 255, 255, 32, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 40, 14, 19, 79, 228, 255, 208, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 209, 255, 255, 255, 255, 247, 58, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 187, 255, 255, 249, 181, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 18, 27, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŋ' / 0x14b
        'ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 101, 120, 17, 0, 28, 125, 163, 183, 158, 83, 2, 0, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 67, 122, 249, 255, 255, 255, 255, 255, 191, 10, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 187, 255, 187, 92, 65, 92, 190, 255, 255, 144, 0, 0, 0,
            ],
            &[
                0, 0, 214, 255, 255, 163, 0, 0, 0, 0, 1, 200, 255, 239, 2, 0, 0,
            ],
            &[
                0, 0, 214, 255, 254, 38, 0, 0, 0, 0, 0, 104, 255, 255, 29, 0, 0,
            ],
            &[
                0, 0, 214, 255, 208, 0, 0, 0, 0, 0, 0, 74, 255, 255, 43, 0, 0,
            ],
            &[
                0, 0, 214, 255, 178, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 155, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 104, 255, 255, 78, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 130, 255, 255, 89, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 130, 255, 255, 89, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 130, 255, 255, 89, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 130, 255, 255, 89, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 130, 255, 255, 89, 0, 0,
            ],
            &[
                0, 0, 214, 255, 150, 0, 0, 0, 0, 0, 0, 130, 255, 255, 89, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 65, 255, 255, 44, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 73, 255, 255, 37, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 124, 255, 251, 11, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 0, 114, 136, 154, 248, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 168, 255, 255, 255, 221, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 41, 105, 114, 80, 7, 0, 0, 0, 0],
        ],
        // letter: 'Ō' / 0x14c
        'Ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 96, 104, 104, 104, 104, 104, 104, 104, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 236, 255, 255, 255, 255, 255, 255, 255, 24, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 119, 129, 129, 129, 129, 129, 129, 129, 12, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 107, 143, 143, 115, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 2, 143, 252, 255, 255, 255, 255, 255, 173, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 228, 169, 153, 202, 255, 255, 203, 8, 0, 0, 0,
            ],
            &[
                0, 0, 78, 255, 255, 156, 9, 0, 0, 0, 86, 252, 255, 134, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 205, 4, 0, 0, 0, 0, 0, 143, 255, 243, 15, 0, 0,
            ],
            &[
                0, 36, 255, 255, 96, 0, 0, 0, 0, 0, 0, 36, 255, 255, 89, 0, 0,
            ],
            &[
                0, 94, 255, 255, 19, 0, 0, 0, 0, 0, 0, 0, 223, 255, 157, 0, 0,
            ],
            &[
                0, 147, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 176, 255, 198, 0, 0,
            ],
            &[
                0, 170, 255, 201, 0, 0, 0, 0, 0, 0, 0, 0, 147, 255, 229, 0, 0,
            ],
            &[
                0, 186, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 242, 0, 0,
            ],
            &[
                0, 197, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 248, 0, 0,
            ],
            &[
                0, 187, 255, 193, 0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 237, 0, 0,
            ],
            &[
                0, 171, 255, 207, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 221, 0, 0,
            ],
            &[
                0, 135, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 192, 255, 183, 0, 0,
            ],
            &[
                0, 90, 255, 255, 39, 0, 0, 0, 0, 0, 0, 3, 237, 255, 136, 0, 0,
            ],
            &[
                0, 19, 252, 255, 127, 0, 0, 0, 0, 0, 0, 70, 255, 255, 62, 0, 0,
            ],
            &[
                0, 0, 174, 255, 236, 28, 0, 0, 0, 0, 4, 200, 255, 216, 2, 0, 0,
            ],
            &[
                0, 0, 44, 248, 255, 214, 64, 3, 0, 38, 181, 255, 255, 81, 0, 0, 0,
            ],
            &[
                0, 0, 0, 91, 251, 255, 255, 249, 242, 255, 255, 255, 131, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 58, 193, 254, 255, 255, 255, 208, 83, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 22, 53, 57, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ō' / 0x14d
        'ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 142, 154, 154, 154, 154, 154, 154, 154, 14, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 236, 255, 255, 255, 255, 255, 255, 255, 24, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 74, 80, 80, 80, 80, 80, 80, 80, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 142, 173, 176, 146, 72, 2, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 26, 195, 255, 255, 255, 255, 255, 255, 212, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 222, 255, 242, 132, 72, 72, 123, 232, 255, 240, 33, 0, 0, 0,
            ],
            &[
                0, 0, 151, 255, 243, 45, 0, 0, 0, 0, 29, 225, 255, 193, 0, 0, 0,
            ],
            &[
                0, 9, 243, 255, 128, 0, 0, 0, 0, 0, 0, 91, 255, 255, 41, 0, 0,
            ],
            &[
                0, 73, 255, 255, 39, 0, 0, 0, 0, 0, 0, 10, 249, 255, 121, 0, 0,
            ],
            &[
                0, 103, 255, 247, 1, 0, 0, 0, 0, 0, 0, 0, 212, 255, 151, 0, 0,
            ],
            &[
                0, 125, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 195, 255, 175, 0, 0,
            ],
            &[
                0, 117, 255, 241, 0, 0, 0, 0, 0, 0, 0, 0, 203, 255, 170, 0, 0,
            ],
            &[
                0, 94, 255, 255, 16, 0, 0, 0, 0, 0, 0, 0, 232, 255, 147, 0, 0,
            ],
            &[
                0, 37, 255, 255, 82, 0, 0, 0, 0, 0, 0, 42, 255, 255, 91, 0, 0,
            ],
            &[
                0, 0, 205, 255, 201, 2, 0, 0, 0, 0, 0, 160, 255, 244, 17, 0, 0,
            ],
            &[
                0, 0, 72, 254, 255, 165, 18, 0, 0, 7, 128, 255, 255, 129, 0, 0, 0,
            ],
            &[
                0, 0, 0, 119, 254, 255, 244, 199, 193, 236, 255, 255, 176, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 71, 199, 254, 255, 255, 255, 223, 109, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 21, 55, 60, 33, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => &[
            &[0, 0, 0, 7, 199, 130, 0, 0, 0, 0, 92, 201, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 206, 249, 82, 0, 3, 58, 226, 242, 8, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 56, 247, 255, 242, 243, 255, 252, 90, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 34, 150, 193, 194, 155, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 107, 143, 143, 115, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 2, 143, 252, 255, 255, 255, 255, 255, 173, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 228, 169, 153, 202, 255, 255, 203, 8, 0, 0, 0,
            ],
            &[
                0, 0, 78, 255, 255, 156, 9, 0, 0, 0, 86, 252, 255, 134, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 205, 4, 0, 0, 0, 0, 0, 143, 255, 243, 15, 0, 0,
            ],
            &[
                0, 36, 255, 255, 96, 0, 0, 0, 0, 0, 0, 36, 255, 255, 89, 0, 0,
            ],
            &[
                0, 94, 255, 255, 19, 0, 0, 0, 0, 0, 0, 0, 223, 255, 157, 0, 0,
            ],
            &[
                0, 147, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 176, 255, 198, 0, 0,
            ],
            &[
                0, 170, 255, 201, 0, 0, 0, 0, 0, 0, 0, 0, 147, 255, 229, 0, 0,
            ],
            &[
                0, 186, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 242, 0, 0,
            ],
            &[
                0, 197, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 248, 0, 0,
            ],
            &[
                0, 187, 255, 193, 0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 237, 0, 0,
            ],
            &[
                0, 171, 255, 207, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 221, 0, 0,
            ],
            &[
                0, 135, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 192, 255, 183, 0, 0,
            ],
            &[
                0, 90, 255, 255, 39, 0, 0, 0, 0, 0, 0, 3, 237, 255, 136, 0, 0,
            ],
            &[
                0, 19, 252, 255, 127, 0, 0, 0, 0, 0, 0, 70, 255, 255, 62, 0, 0,
            ],
            &[
                0, 0, 174, 255, 236, 28, 0, 0, 0, 0, 4, 200, 255, 216, 2, 0, 0,
            ],
            &[
                0, 0, 44, 248, 255, 214, 64, 3, 0, 38, 181, 255, 255, 81, 0, 0, 0,
            ],
            &[
                0, 0, 0, 91, 251, 255, 255, 249, 242, 255, 255, 255, 131, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 58, 193, 254, 255, 255, 255, 208, 83, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 22, 53, 57, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŏ' / 0x14f
        'ŏ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 245, 175, 0, 0, 0, 0, 124, 250, 45, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 184, 255, 131, 36, 41, 107, 244, 225, 3, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 32, 223, 255, 255, 255, 255, 234, 57, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 9, 101, 144, 144, 105, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 142, 173, 176, 146, 72, 2, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 26, 195, 255, 255, 255, 255, 255, 255, 212, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 222, 255, 242, 132, 72, 72, 123, 232, 255, 240, 33, 0, 0, 0,
            ],
            &[
                0, 0, 151, 255, 243, 45, 0, 0, 0, 0, 29, 225, 255, 193, 0, 0, 0,
            ],
            &[
                0, 9, 243, 255, 128, 0, 0, 0, 0, 0, 0, 91, 255, 255, 41, 0, 0,
            ],
            &[
                0, 73, 255, 255, 39, 0, 0, 0, 0, 0, 0, 10, 249, 255, 121, 0, 0,
            ],
            &[
                0, 103, 255, 247, 1, 0, 0, 0, 0, 0, 0, 0, 212, 255, 151, 0, 0,
            ],
            &[
                0, 125, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 195, 255, 175, 0, 0,
            ],
            &[
                0, 117, 255, 241, 0, 0, 0, 0, 0, 0, 0, 0, 203, 255, 170, 0, 0,
            ],
            &[
                0, 94, 255, 255, 16, 0, 0, 0, 0, 0, 0, 0, 232, 255, 147, 0, 0,
            ],
            &[
                0, 37, 255, 255, 82, 0, 0, 0, 0, 0, 0, 42, 255, 255, 91, 0, 0,
            ],
            &[
                0, 0, 205, 255, 201, 2, 0, 0, 0, 0, 0, 160, 255, 244, 17, 0, 0,
            ],
            &[
                0, 0, 72, 254, 255, 165, 18, 0, 0, 7, 128, 255, 255, 129, 0, 0, 0,
            ],
            &[
                0, 0, 0, 119, 254, 255, 244, 199, 193, 236, 255, 255, 176, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 71, 199, 254, 255, 255, 255, 223, 109, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 21, 55, 60, 33, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ő' / 0x150
        'Ő' => &[
            &[
                0, 0, 0, 0, 0, 99, 255, 251, 56, 0, 160, 255, 227, 17, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 230, 255, 106, 0, 51, 253, 245, 53, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 141, 255, 135, 0, 3, 201, 249, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 210, 122, 0, 0, 77, 211, 71, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 107, 143, 143, 115, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 2, 143, 252, 255, 255, 255, 255, 255, 173, 18, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 146, 255, 255, 228, 169, 153, 202, 255, 255, 203, 8, 0, 0, 0,
            ],
            &[
                0, 0, 78, 255, 255, 156, 9, 0, 0, 0, 86, 252, 255, 134, 0, 0, 0,
            ],
            &[
                0, 0, 197, 255, 205, 4, 0, 0, 0, 0, 0, 143, 255, 243, 15, 0, 0,
            ],
            &[
                0, 36, 255, 255, 96, 0, 0, 0, 0, 0, 0, 36, 255, 255, 89, 0, 0,
            ],
            &[
                0, 94, 255, 255, 19, 0, 0, 0, 0, 0, 0, 0, 223, 255, 157, 0, 0,
            ],
            &[
                0, 147, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 176, 255, 198, 0, 0,
            ],
            &[
                0, 170, 255, 201, 0, 0, 0, 0, 0, 0, 0, 0, 147, 255, 229, 0, 0,
            ],
            &[
                0, 186, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 242, 0, 0,
            ],
            &[
                0, 197, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 248, 0, 0,
            ],
            &[
                0, 187, 255, 193, 0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 237, 0, 0,
            ],
            &[
                0, 171, 255, 207, 0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 221, 0, 0,
            ],
            &[
                0, 135, 255, 244, 1, 0, 0, 0, 0, 0, 0, 0, 192, 255, 183, 0, 0,
            ],
            &[
                0, 90, 255, 255, 39, 0, 0, 0, 0, 0, 0, 3, 237, 255, 136, 0, 0,
            ],
            &[
                0, 19, 252, 255, 127, 0, 0, 0, 0, 0, 0, 70, 255, 255, 62, 0, 0,
            ],
            &[
                0, 0, 174, 255, 236, 28, 0, 0, 0, 0, 4, 200, 255, 216, 2, 0, 0,
            ],
            &[
                0, 0, 44, 248, 255, 214, 64, 3, 0, 38, 181, 255, 255, 81, 0, 0, 0,
            ],
            &[
                0, 0, 0, 91, 251, 255, 255, 249, 242, 255, 255, 255, 131, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 58, 193, 254, 255, 255, 255, 208, 83, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 22, 53, 57, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 120, 124, 85, 0, 30, 124, 124, 54, 0, 0, 0],
        ],
        // letter: 'ő' / 0x151
        'ő' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 12, 170, 174, 110, 0, 51, 174, 174, 66, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 126, 255, 240, 32, 0, 187, 255, 203, 5, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 27, 244, 250, 68, 0, 78, 255, 228, 27, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 173, 251, 90, 0, 12, 224, 233, 45, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 163, 77, 0, 0, 67, 161, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 142, 173, 176, 146, 72, 2, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 26, 195, 255, 255, 255, 255, 255, 255, 212, 41, 0, 0, 0, 0,
            ],
            &[
                0, 0, 14, 222, 255, 242, 132, 72, 72, 123, 232, 255, 240, 33, 0, 0, 0,
            ],
            &[
                0, 0, 151, 255, 243, 45, 0, 0, 0, 0, 29, 225, 255, 193, 0, 0, 0,
            ],
            &[
                0, 9, 243, 255, 128, 0, 0, 0, 0, 0, 0, 91, 255, 255, 41, 0, 0,
            ],
            &[
                0, 73, 255, 255, 39, 0, 0, 0, 0, 0, 0, 10, 249, 255, 121, 0, 0,
            ],
            &[
                0, 103, 255, 247, 1, 0, 0, 0, 0, 0, 0, 0, 212, 255, 151, 0, 0,
            ],
            &[
                0, 125, 255, 231, 0, 0, 0, 0, 0, 0, 0, 0, 195, 255, 175, 0, 0,
            ],
            &[
                0, 117, 255, 241, 0, 0, 0, 0, 0, 0, 0, 0, 203, 255, 170, 0, 0,
            ],
            &[
                0, 94, 255, 255, 16, 0, 0, 0, 0, 0, 0, 0, 232, 255, 147, 0, 0,
            ],
            &[
                0, 37, 255, 255, 82, 0, 0, 0, 0, 0, 0, 42, 255, 255, 91, 0, 0,
            ],
            &[
                0, 0, 205, 255, 201, 2, 0, 0, 0, 0, 0, 160, 255, 244, 17, 0, 0,
            ],
            &[
                0, 0, 72, 254, 255, 165, 18, 0, 0, 7, 128, 255, 255, 129, 0, 0, 0,
            ],
            &[
                0, 0, 0, 119, 254, 255, 244, 199, 193, 236, 255, 255, 176, 3, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 71, 199, 254, 255, 255, 255, 223, 109, 2, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 21, 55, 60, 33, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Œ' / 0x152
        'Œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 11, 92, 129, 128, 89, 71, 71, 71, 71, 71, 71, 27, 0,
            ],
            &[
                0, 0, 0, 68, 236, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0,
            ],
            &[
                0, 0, 42, 247, 255, 230, 160, 181, 250, 255, 247, 225, 225, 225, 225, 88, 0,
            ],
            &[
                0, 0, 185, 255, 220, 18, 0, 0, 161, 255, 183, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 35, 254, 255, 92, 0, 0, 0, 161, 255, 183, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 104, 255, 251, 14, 0, 0, 0, 161, 255, 183, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 159, 255, 214, 0, 0, 0, 0, 161, 255, 183, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 192, 255, 183, 0, 0, 0, 0, 161, 255, 183, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 217, 255, 169, 0, 0, 0, 0, 161, 255, 194, 38, 38, 38, 33, 0, 0,
            ],
            &[
                0, 227, 255, 155, 0, 0, 0, 0, 161, 255, 255, 255, 255, 255, 224, 0, 0,
            ],
            &[
                0, 231, 255, 147, 0, 0, 0, 0, 161, 255, 254, 251, 251, 251, 220, 0, 0,
            ],
            &[
                0, 220, 255, 154, 0, 0, 0, 0, 161, 255, 183, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 207, 255, 165, 0, 0, 0, 0, 161, 255, 183, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 182, 255, 192, 0, 0, 0, 0, 161, 255, 183, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 139, 255, 228, 0, 0, 0, 0, 161, 255, 183, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 90, 255, 255, 34, 0, 0, 0, 161, 255, 183, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 14, 247, 255, 130, 0, 0, 0, 161, 255, 183, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 161, 255, 245, 74, 0, 10, 184, 255, 195, 40, 40, 40, 40, 15, 0,
            ],
            &[
                0, 0, 20, 232, 255, 255, 240, 252, 255, 255, 255, 255, 255, 255, 255, 99, 0,
            ],
            &[
                0, 0, 0, 38, 198, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 99, 0,
            ],
            &[0, 0, 0, 0, 0, 24, 61, 43, 4, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'œ' / 0x153
        'œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 4, 106, 178, 182, 119, 9, 0, 1, 102, 170, 159, 61, 0, 0, 0,
            ],
            &[
                0, 3, 183, 255, 255, 255, 255, 184, 1, 163, 255, 255, 255, 254, 80, 0, 0,
            ],
            &[
                0, 108, 255, 243, 95, 57, 195, 255, 158, 255, 198, 43, 102, 255, 229, 3, 0,
            ],
            &[
                0, 216, 255, 108, 0, 0, 30, 253, 255, 255, 43, 0, 0, 212, 255, 68, 0,
            ],
            &[
                28, 255, 255, 26, 0, 0, 0, 205, 255, 221, 0, 0, 0, 138, 255, 121, 0,
            ],
            &[
                69, 255, 229, 0, 0, 0, 0, 165, 255, 187, 0, 0, 0, 114, 255, 151, 0,
            ],
            &[
                91, 255, 211, 0, 0, 0, 0, 144, 255, 217, 143, 143, 143, 186, 255, 165, 0,
            ],
            &[
                100, 255, 197, 0, 0, 0, 0, 136, 255, 255, 255, 255, 255, 255, 255, 168, 0,
            ],
            &[
                90, 255, 205, 0, 0, 0, 0, 139, 255, 178, 43, 43, 43, 43, 43, 28, 0,
            ],
            &[75, 255, 221, 0, 0, 0, 0, 150, 255, 176, 0, 0, 0, 0, 0, 0, 0],
            &[35, 255, 246, 6, 0, 0, 0, 178, 255, 209, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 235, 255, 63, 0, 0, 0, 230, 255, 251, 20, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 142, 255, 191, 2, 0, 85, 255, 239, 255, 143, 0, 0, 0, 57, 15, 0,
            ],
            &[
                0, 27, 234, 255, 219, 176, 249, 248, 48, 231, 255, 195, 156, 201, 255, 30, 0,
            ],
            &[
                0, 0, 54, 216, 255, 255, 236, 79, 0, 48, 220, 255, 255, 251, 164, 11, 0,
            ],
            &[0, 0, 0, 3, 42, 54, 11, 0, 0, 0, 3, 46, 54, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŕ' / 0x154
        'Ŕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 18, 237, 255, 207, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 158, 255, 224, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 255, 227, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 189, 190, 28, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 71, 71, 71, 71, 63, 44, 12, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 111, 255, 255, 255, 255, 255, 255, 250, 194, 74, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 255, 211, 211, 226, 250, 255, 255, 255, 135, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 1, 69, 208, 255, 255, 63, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 32, 246, 255, 161, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 202, 255, 199, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 176, 255, 208, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 208, 255, 180, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 60, 255, 255, 103, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 7, 34, 112, 239, 255, 208, 5, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 255, 253, 253, 255, 255, 255, 255, 190, 23, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 255, 255, 255, 255, 255, 255, 126, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 29, 29, 29, 192, 255, 214, 5, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 46, 252, 255, 117, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 147, 255, 243, 27, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 17, 235, 255, 168, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 102, 255, 255, 67, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 2, 204, 255, 214, 5, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 57, 254, 255, 118, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 161, 255, 243, 28, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 72, 124, 124, 47, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŕ' / 0x155
        'ŕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 96, 174, 174, 69, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 23, 241, 255, 192, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 169, 255, 209, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 86, 255, 209, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 142, 148, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 107, 120, 120, 120, 120, 120, 5, 0, 53, 153, 182, 164, 117, 24, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 255, 255, 35, 94, 254, 255, 255, 255, 255, 63, 0,
            ],
            &[
                0, 0, 55, 83, 105, 128, 247, 255, 105, 248, 247, 151, 109, 137, 201, 3, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 236, 255, 248, 236, 44, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 255, 91, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 232, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 168, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 237, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 52, 178, 201, 225, 248, 255, 255, 255, 238, 213, 189, 124, 0, 0, 0, 0,
            ],
            &[
                0, 83, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŗ' / 0x156
        'Ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 71, 71, 71, 71, 63, 44, 12, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 111, 255, 255, 255, 255, 255, 255, 250, 194, 74, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 255, 211, 211, 226, 250, 255, 255, 255, 135, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 1, 69, 208, 255, 255, 63, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 32, 246, 255, 161, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 202, 255, 199, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 176, 255, 208, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 208, 255, 180, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 60, 255, 255, 103, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 7, 34, 112, 239, 255, 208, 5, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 255, 253, 253, 255, 255, 255, 255, 190, 23, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 255, 255, 255, 255, 255, 255, 126, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 29, 29, 29, 192, 255, 214, 5, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 46, 252, 255, 117, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 147, 255, 243, 27, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 17, 235, 255, 168, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 102, 255, 255, 67, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 2, 204, 255, 214, 5, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 57, 254, 255, 118, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 161, 255, 243, 28, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 90, 90, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 97, 255, 225, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 156, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 229, 228, 10, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 247, 73, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŗ' / 0x157
        'ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 107, 120, 120, 120, 120, 120, 5, 0, 53, 153, 182, 164, 117, 24, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 255, 255, 35, 94, 254, 255, 255, 255, 255, 63, 0,
            ],
            &[
                0, 0, 55, 83, 105, 128, 247, 255, 105, 248, 247, 151, 109, 137, 201, 3, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 236, 255, 248, 236, 44, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 255, 91, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 232, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 168, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 237, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 52, 178, 201, 225, 248, 255, 255, 255, 238, 213, 189, 124, 0, 0, 0, 0,
            ],
            &[
                0, 83, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 90, 46, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 245, 255, 75, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 218, 3, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 128, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 208, 173, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ř' / 0x158
        'Ř' => &[
            &[
                0, 0, 15, 206, 252, 115, 0, 0, 87, 244, 230, 37, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 23, 222, 255, 135, 109, 254, 239, 44, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 39, 242, 255, 255, 252, 64, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 92, 212, 212, 123, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 71, 71, 71, 71, 63, 44, 12, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 111, 255, 255, 255, 255, 255, 255, 250, 194, 74, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 255, 211, 211, 226, 250, 255, 255, 255, 135, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 1, 69, 208, 255, 255, 63, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 32, 246, 255, 161, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 202, 255, 199, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 176, 255, 208, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 208, 255, 180, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 60, 255, 255, 103, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 7, 34, 112, 239, 255, 208, 5, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 255, 253, 253, 255, 255, 255, 255, 190, 23, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 255, 255, 255, 255, 255, 255, 126, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 29, 29, 29, 192, 255, 214, 5, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 46, 252, 255, 117, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 147, 255, 243, 27, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 17, 235, 255, 168, 0, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 102, 255, 255, 67, 0, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 2, 204, 255, 214, 5, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 57, 254, 255, 118, 0, 0, 0,
            ],
            &[
                0, 111, 255, 254, 0, 0, 0, 0, 0, 0, 0, 161, 255, 243, 28, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 55, 124, 59, 0, 0, 0, 0, 41, 124, 80, 0, 0, 0, 0, 0],
        ],
        // letter: 'ř' / 0x159
        'ř' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 53, 174, 117, 1, 0, 0, 0, 66, 174, 116, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 154, 255, 180, 10, 0, 117, 253, 216, 23, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 3, 178, 255, 194, 143, 255, 229, 30, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 10, 213, 255, 255, 249, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 46, 163, 163, 96, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 107, 120, 120, 120, 120, 120, 5, 0, 53, 153, 182, 164, 117, 24, 0,
            ],
            &[
                0, 0, 228, 255, 255, 255, 255, 255, 35, 94, 254, 255, 255, 255, 255, 63, 0,
            ],
            &[
                0, 0, 55, 83, 105, 128, 247, 255, 105, 248, 247, 151, 109, 137, 201, 3, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 236, 255, 248, 236, 44, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 255, 91, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 232, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 168, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 236, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 237, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 52, 178, 201, 225, 248, 255, 255, 255, 238, 213, 189, 124, 0, 0, 0, 0,
            ],
            &[
                0, 83, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 190, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ś' / 0x15a
        'Ś' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 125, 255, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 36, 246, 255, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 194, 255, 139, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 81, 212, 115, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 56, 110, 131, 131, 110, 64, 4, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 72, 221, 255, 255, 255, 255, 255, 255, 236, 133, 0, 0, 0,
            ],
            &[
                0, 0, 0, 73, 251, 255, 252, 210, 170, 176, 205, 246, 255, 135, 0, 0, 0,
            ],
            &[0, 0, 1, 215, 255, 229, 44, 0, 0, 0, 0, 12, 89, 32, 0, 0, 0],
            &[0, 0, 36, 255, 255, 104, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 255, 61, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 230, 255, 214, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 98, 255, 255, 237, 123, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 106, 248, 255, 255, 242, 138, 23, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 30, 154, 250, 255, 255, 246, 145, 14, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 32, 147, 246, 255, 255, 222, 29, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 148, 255, 255, 190, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 140, 255, 255, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 255, 62, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 255, 255, 58, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 156, 255, 251, 16, 0, 0],
            &[
                0, 0, 154, 147, 64, 9, 0, 0, 0, 19, 133, 255, 255, 167, 0, 0, 0,
            ],
            &[
                0, 0, 180, 255, 255, 252, 225, 211, 223, 253, 255, 255, 212, 20, 0, 0, 0,
            ],
            &[
                0, 0, 73, 181, 242, 255, 255, 255, 255, 255, 225, 127, 10, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 29, 56, 65, 55, 27, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 123, 124, 111, 0, 0, 0, 0],
        ],
        // letter: 'ś' / 0x15b
        'ś' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 87, 174, 174, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 16, 234, 255, 203, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 155, 255, 217, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 73, 254, 217, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 133, 152, 19, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 56, 130, 170, 184, 168, 144, 87, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 9, 174, 255, 255, 255, 255, 255, 255, 255, 239, 8, 0, 0, 0,
            ],
            &[
                0, 0, 0, 136, 255, 254, 160, 95, 77, 92, 124, 199, 161, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 213, 255, 151, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0],
            &[0, 0, 0, 222, 255, 147, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 160, 255, 246, 106, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 21, 210, 255, 255, 220, 107, 10, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 119, 233, 255, 255, 236, 130, 21, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 10, 111, 224, 255, 255, 238, 74, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 93, 237, 255, 244, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 255, 255, 76, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 37, 255, 255, 78, 0, 0, 0],
            &[
                0, 0, 36, 162, 65, 3, 0, 0, 0, 12, 164, 255, 251, 29, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 244, 205, 191, 205, 243, 255, 255, 136, 0, 0, 0, 0,
            ],
            &[
                0, 0, 15, 144, 225, 255, 255, 255, 255, 255, 214, 98, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 56, 65, 52, 23, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => &[
            &[0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 125, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 46, 242, 248, 118, 239, 253, 79, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 28, 226, 245, 73, 0, 51, 234, 245, 59, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 168, 201, 49, 0, 0, 0, 30, 187, 197, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 56, 110, 131, 131, 110, 64, 4, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 72, 221, 255, 255, 255, 255, 255, 255, 236, 133, 0, 0, 0,
            ],
            &[
                0, 0, 0, 73, 251, 255, 252, 210, 170, 176, 205, 246, 255, 135, 0, 0, 0,
            ],
            &[0, 0, 1, 215, 255, 229, 44, 0, 0, 0, 0, 12, 89, 32, 0, 0, 0],
            &[0, 0, 36, 255, 255, 104, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 255, 61, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 230, 255, 214, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 98, 255, 255, 237, 123, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 106, 248, 255, 255, 242, 138, 23, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 30, 154, 250, 255, 255, 246, 145, 14, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 32, 147, 246, 255, 255, 222, 29, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 148, 255, 255, 190, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 140, 255, 255, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 255, 62, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 255, 255, 58, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 156, 255, 251, 16, 0, 0],
            &[
                0, 0, 154, 147, 64, 9, 0, 0, 0, 19, 133, 255, 255, 167, 0, 0, 0,
            ],
            &[
                0, 0, 180, 255, 255, 252, 225, 211, 223, 253, 255, 255, 212, 20, 0, 0, 0,
            ],
            &[
                0, 0, 73, 181, 242, 255, 255, 255, 255, 255, 225, 127, 10, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 29, 56, 65, 55, 27, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 100, 124, 117, 3, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŝ' / 0x15d
        'ŝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 43, 174, 174, 115, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 208, 255, 255, 254, 67, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 1, 170, 255, 172, 93, 251, 238, 40, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 141, 255, 163, 4, 0, 81, 245, 224, 29, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 39, 163, 110, 0, 0, 0, 0, 46, 161, 116, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 56, 130, 170, 184, 168, 144, 87, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 9, 174, 255, 255, 255, 255, 255, 255, 255, 239, 8, 0, 0, 0,
            ],
            &[
                0, 0, 0, 136, 255, 254, 160, 95, 77, 92, 124, 199, 161, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 213, 255, 151, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0],
            &[0, 0, 0, 222, 255, 147, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 160, 255, 246, 106, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 21, 210, 255, 255, 220, 107, 10, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 119, 233, 255, 255, 236, 130, 21, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 10, 111, 224, 255, 255, 238, 74, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 93, 237, 255, 244, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 255, 255, 76, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 37, 255, 255, 78, 0, 0, 0],
            &[
                0, 0, 36, 162, 65, 3, 0, 0, 0, 12, 164, 255, 251, 29, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 244, 205, 191, 205, 243, 255, 255, 136, 0, 0, 0, 0,
            ],
            &[
                0, 0, 15, 144, 225, 255, 255, 255, 255, 255, 214, 98, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 56, 65, 52, 23, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ş' / 0x15e
        'Ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 56, 110, 131, 131, 110, 64, 4, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 72, 221, 255, 255, 255, 255, 255, 255, 236, 133, 0, 0, 0,
            ],
            &[
                0, 0, 0, 73, 251, 255, 252, 210, 170, 176, 205, 246, 255, 135, 0, 0, 0,
            ],
            &[0, 0, 1, 215, 255, 229, 44, 0, 0, 0, 0, 12, 89, 32, 0, 0, 0],
            &[0, 0, 36, 255, 255, 104, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 255, 61, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 230, 255, 214, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 98, 255, 255, 237, 123, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 106, 248, 255, 255, 242, 138, 23, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 30, 154, 250, 255, 255, 246, 145, 14, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 32, 147, 246, 255, 255, 222, 29, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 148, 255, 255, 190, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 140, 255, 255, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 255, 62, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 255, 255, 58, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 156, 255, 251, 16, 0, 0],
            &[
                0, 0, 154, 147, 64, 9, 0, 0, 0, 19, 133, 255, 255, 167, 0, 0, 0,
            ],
            &[
                0, 0, 180, 255, 255, 252, 225, 211, 223, 253, 255, 255, 212, 20, 0, 0, 0,
            ],
            &[
                0, 0, 73, 181, 242, 255, 255, 255, 255, 255, 225, 127, 10, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 29, 59, 255, 236, 27, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 88, 255, 167, 31, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 166, 246, 246, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 101, 255, 147, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 6, 35, 157, 255, 122, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 181, 255, 255, 255, 198, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 118, 100, 59, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ş' / 0x15f
        'ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 56, 130, 170, 184, 168, 144, 87, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 9, 174, 255, 255, 255, 255, 255, 255, 255, 239, 8, 0, 0, 0,
            ],
            &[
                0, 0, 0, 136, 255, 254, 160, 95, 77, 92, 124, 199, 161, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 213, 255, 151, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0],
            &[0, 0, 0, 222, 255, 147, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 160, 255, 246, 106, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 21, 210, 255, 255, 220, 107, 10, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 119, 233, 255, 255, 236, 130, 21, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 10, 111, 224, 255, 255, 238, 74, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 93, 237, 255, 244, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 255, 255, 76, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 37, 255, 255, 78, 0, 0, 0],
            &[
                0, 0, 36, 162, 65, 3, 0, 0, 0, 12, 164, 255, 251, 29, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 244, 205, 191, 205, 243, 255, 255, 136, 0, 0, 0, 0,
            ],
            &[
                0, 0, 15, 144, 225, 255, 255, 255, 255, 255, 214, 98, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 56, 254, 255, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 54, 255, 189, 43, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 55, 157, 237, 252, 83, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 66, 255, 181, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 6, 29, 129, 255, 157, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 146, 255, 255, 255, 218, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 118, 104, 68, 2, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Š' / 0x160
        'Š' => &[
            &[
                0, 0, 0, 0, 92, 254, 210, 33, 0, 15, 184, 255, 143, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 114, 255, 227, 62, 208, 255, 158, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 151, 255, 255, 255, 188, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 188, 212, 205, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 56, 110, 131, 131, 110, 64, 4, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 72, 221, 255, 255, 255, 255, 255, 255, 236, 133, 0, 0, 0,
            ],
            &[
                0, 0, 0, 73, 251, 255, 252, 210, 170, 176, 205, 246, 255, 135, 0, 0, 0,
            ],
            &[0, 0, 1, 215, 255, 229, 44, 0, 0, 0, 0, 12, 89, 32, 0, 0, 0],
            &[0, 0, 36, 255, 255, 104, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 255, 61, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 255, 255, 89, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 230, 255, 214, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 98, 255, 255, 237, 123, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 106, 248, 255, 255, 242, 138, 23, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 30, 154, 250, 255, 255, 246, 145, 14, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 32, 147, 246, 255, 255, 222, 29, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 148, 255, 255, 190, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 140, 255, 255, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 255, 62, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72, 255, 255, 58, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 156, 255, 251, 16, 0, 0],
            &[
                0, 0, 154, 147, 64, 9, 0, 0, 0, 19, 133, 255, 255, 167, 0, 0, 0,
            ],
            &[
                0, 0, 180, 255, 255, 252, 225, 211, 223, 253, 255, 255, 212, 20, 0, 0, 0,
            ],
            &[
                0, 0, 73, 181, 242, 255, 255, 255, 255, 255, 225, 127, 10, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 29, 56, 65, 55, 27, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 116, 114, 9, 0, 0, 0, 1, 100, 124, 19, 0, 0, 0],
        ],
        // letter: 'š' / 0x161
        'š' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 174, 125, 3, 0, 0, 0, 58, 173, 125, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 141, 255, 190, 14, 0, 105, 250, 224, 30, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 1, 166, 255, 203, 134, 255, 235, 38, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 6, 202, 255, 255, 252, 60, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 163, 163, 105, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 56, 130, 170, 184, 168, 144, 87, 13, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 9, 174, 255, 255, 255, 255, 255, 255, 255, 239, 8, 0, 0, 0,
            ],
            &[
                0, 0, 0, 136, 255, 254, 160, 95, 77, 92, 124, 199, 161, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 213, 255, 151, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0],
            &[0, 0, 0, 222, 255, 147, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 160, 255, 246, 106, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 21, 210, 255, 255, 220, 107, 10, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 7, 119, 233, 255, 255, 236, 130, 21, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 10, 111, 224, 255, 255, 238, 74, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 93, 237, 255, 244, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 255, 255, 76, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 37, 255, 255, 78, 0, 0, 0],
            &[
                0, 0, 36, 162, 65, 3, 0, 0, 0, 12, 164, 255, 251, 29, 0, 0, 0,
            ],
            &[
                0, 0, 42, 255, 255, 244, 205, 191, 205, 243, 255, 255, 136, 0, 0, 0, 0,
            ],
            &[
                0, 0, 15, 144, 225, 255, 255, 255, 255, 255, 214, 98, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 56, 65, 52, 23, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ţ' / 0x162
        'Ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 55, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 70, 0, 0,
            ],
            &[
                0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 177, 225, 225, 225, 225, 229, 255, 255, 235, 225, 225, 225, 225, 222, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 220, 180, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 88, 255, 167, 31, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 166, 246, 246, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 101, 255, 147, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 6, 35, 157, 255, 122, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 181, 255, 255, 255, 198, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 118, 100, 59, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ţ' / 0x163
        'ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 85, 75, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 75, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 108, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 18, 58, 86, 114, 197, 255, 240, 120, 120, 120, 120, 120, 63, 0, 0,
            ],
            &[
                0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 134, 0, 0,
            ],
            &[
                0, 0, 69, 141, 141, 141, 203, 255, 242, 141, 141, 141, 141, 141, 74, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 137, 255, 233, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 114, 255, 253, 20, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 59, 255, 255, 163, 6, 0, 0, 0, 7, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 183, 255, 255, 237, 190, 190, 216, 234, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 10, 142, 241, 255, 255, 255, 255, 222, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 255, 242, 55, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 89, 255, 166, 31, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 73, 166, 246, 246, 54, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 101, 255, 146, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 16, 6, 36, 157, 255, 122, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 255, 255, 198, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 72, 118, 100, 58, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ť' / 0x164
        'Ť' => &[
            &[
                0, 0, 0, 15, 206, 252, 115, 0, 0, 87, 244, 230, 37, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 23, 222, 255, 135, 109, 254, 239, 44, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 39, 242, 255, 255, 252, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 212, 212, 123, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 55, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 70, 0, 0,
            ],
            &[
                0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 177, 225, 225, 225, 225, 229, 255, 255, 235, 225, 225, 225, 225, 222, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 55, 124, 59, 0, 0, 0, 0, 41, 124, 80, 0, 0, 0, 0],
        ],
        // letter: 'ť' / 0x165
        'ť' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 122, 209, 156, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 190, 255, 112, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 85, 75, 0, 0, 3, 241, 239, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 255, 226, 0, 0, 54, 255, 121, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 75, 255, 226, 0, 0, 52, 125, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 108, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 18, 58, 86, 114, 197, 255, 240, 120, 120, 120, 120, 120, 63, 0, 0,
            ],
            &[
                0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 134, 0, 0,
            ],
            &[
                0, 0, 69, 141, 141, 141, 203, 255, 242, 141, 141, 141, 141, 141, 74, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 137, 255, 233, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 114, 255, 253, 20, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 59, 255, 255, 163, 6, 0, 0, 0, 7, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 183, 255, 255, 237, 190, 190, 216, 234, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 10, 142, 241, 255, 255, 255, 255, 222, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 41, 62, 55, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŧ' / 0x166
        'Ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 55, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 71, 70, 0, 0,
            ],
            &[
                0, 200, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 252, 0, 0,
            ],
            &[
                0, 177, 225, 225, 225, 225, 229, 255, 255, 235, 225, 225, 225, 225, 222, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 21, 197, 197, 197, 227, 255, 255, 255, 197, 197, 197, 66, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 86, 0, 0, 0,
            ],
            &[
                0, 0, 6, 57, 57, 57, 87, 255, 255, 138, 57, 57, 57, 19, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 255, 255, 80, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŧ' / 0x167
        'ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 85, 75, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 75, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 108, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 18, 58, 86, 114, 197, 255, 240, 120, 120, 120, 120, 120, 63, 0, 0,
            ],
            &[
                0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 134, 0, 0,
            ],
            &[
                0, 0, 69, 141, 141, 141, 203, 255, 242, 141, 141, 141, 141, 141, 74, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 42, 87, 87, 87, 227, 255, 255, 87, 87, 87, 87, 87, 81, 0, 0,
            ],
            &[
                0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 238, 0, 0,
            ],
            &[
                0, 0, 54, 112, 112, 112, 252, 255, 255, 112, 112, 112, 112, 112, 104, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 137, 255, 233, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 114, 255, 253, 20, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 59, 255, 255, 163, 6, 0, 0, 0, 7, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 0, 183, 255, 255, 237, 190, 190, 216, 234, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 0, 0, 10, 142, 241, 255, 255, 255, 255, 222, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 41, 62, 55, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ũ' / 0x168
        'Ũ' => &[
            &[0, 0, 0, 0, 19, 121, 141, 70, 0, 0, 0, 89, 147, 0, 0, 0, 0],
            &[
                0, 0, 0, 12, 222, 255, 255, 255, 190, 66, 62, 228, 178, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 114, 255, 134, 94, 197, 255, 255, 255, 254, 61, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 142, 170, 0, 0, 0, 85, 177, 180, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 8, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 21, 255, 255, 89, 0, 0, 0, 0, 0, 0, 37, 255, 255, 72, 0, 0,
            ],
            &[
                0, 3, 254, 255, 110, 0, 0, 0, 0, 0, 0, 58, 255, 255, 53, 0, 0,
            ],
            &[
                0, 0, 222, 255, 162, 0, 0, 0, 0, 0, 0, 107, 255, 253, 16, 0, 0,
            ],
            &[
                0, 0, 155, 255, 244, 22, 0, 0, 0, 0, 3, 206, 255, 201, 0, 0, 0,
            ],
            &[
                0, 0, 46, 253, 255, 205, 51, 0, 0, 30, 169, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 125, 255, 255, 255, 245, 240, 255, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 90, 210, 255, 255, 255, 255, 221, 109, 1, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 25, 53, 56, 27, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ũ' / 0x169
        'ũ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 47, 171, 191, 119, 11, 0, 0, 125, 187, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 26, 243, 255, 255, 255, 229, 115, 111, 242, 161, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 130, 252, 84, 44, 148, 251, 255, 255, 241, 38, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 112, 123, 0, 0, 0, 40, 128, 130, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 117, 120, 54, 0, 0, 0, 0, 0, 0, 47, 120, 120, 4, 0, 0],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 117, 0, 0, 0, 0, 0, 0, 126, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 243, 255, 137, 0, 0, 0, 0, 0, 0, 175, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 219, 255, 190, 0, 0, 0, 0, 0, 24, 247, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 154, 255, 255, 98, 0, 0, 0, 29, 198, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 37, 244, 255, 255, 224, 189, 205, 250, 246, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 54, 204, 255, 255, 255, 255, 199, 63, 5, 249, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 59, 55, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ū' / 0x16a
        'Ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 96, 104, 104, 104, 104, 104, 104, 104, 10, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 236, 255, 255, 255, 255, 255, 255, 255, 24, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 119, 129, 129, 129, 129, 129, 129, 129, 12, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 8, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 21, 255, 255, 89, 0, 0, 0, 0, 0, 0, 37, 255, 255, 72, 0, 0,
            ],
            &[
                0, 3, 254, 255, 110, 0, 0, 0, 0, 0, 0, 58, 255, 255, 53, 0, 0,
            ],
            &[
                0, 0, 222, 255, 162, 0, 0, 0, 0, 0, 0, 107, 255, 253, 16, 0, 0,
            ],
            &[
                0, 0, 155, 255, 244, 22, 0, 0, 0, 0, 3, 206, 255, 201, 0, 0, 0,
            ],
            &[
                0, 0, 46, 253, 255, 205, 51, 0, 0, 30, 169, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 125, 255, 255, 255, 245, 240, 255, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 90, 210, 255, 255, 255, 255, 221, 109, 1, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 25, 53, 56, 27, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ū' / 0x16b
        'ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 142, 154, 154, 154, 154, 154, 154, 154, 14, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 236, 255, 255, 255, 255, 255, 255, 255, 24, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 74, 80, 80, 80, 80, 80, 80, 80, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 117, 120, 54, 0, 0, 0, 0, 0, 0, 47, 120, 120, 4, 0, 0],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 117, 0, 0, 0, 0, 0, 0, 126, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 243, 255, 137, 0, 0, 0, 0, 0, 0, 175, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 219, 255, 190, 0, 0, 0, 0, 0, 24, 247, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 154, 255, 255, 98, 0, 0, 0, 29, 198, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 37, 244, 255, 255, 224, 189, 205, 250, 246, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 54, 204, 255, 255, 255, 255, 199, 63, 5, 249, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 59, 55, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => &[
            &[0, 0, 0, 7, 199, 130, 0, 0, 0, 0, 92, 201, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 206, 249, 82, 0, 3, 58, 226, 242, 8, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 56, 247, 255, 242, 243, 255, 252, 90, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 34, 150, 193, 194, 155, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 8, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 21, 255, 255, 89, 0, 0, 0, 0, 0, 0, 37, 255, 255, 72, 0, 0,
            ],
            &[
                0, 3, 254, 255, 110, 0, 0, 0, 0, 0, 0, 58, 255, 255, 53, 0, 0,
            ],
            &[
                0, 0, 222, 255, 162, 0, 0, 0, 0, 0, 0, 107, 255, 253, 16, 0, 0,
            ],
            &[
                0, 0, 155, 255, 244, 22, 0, 0, 0, 0, 3, 206, 255, 201, 0, 0, 0,
            ],
            &[
                0, 0, 46, 253, 255, 205, 51, 0, 0, 30, 169, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 125, 255, 255, 255, 245, 240, 255, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 90, 210, 255, 255, 255, 255, 221, 109, 1, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 25, 53, 56, 27, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŭ' / 0x16d
        'ŭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 245, 175, 0, 0, 0, 0, 124, 250, 45, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 184, 255, 131, 36, 41, 107, 244, 225, 3, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 32, 223, 255, 255, 255, 255, 234, 57, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 9, 101, 144, 144, 105, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 117, 120, 54, 0, 0, 0, 0, 0, 0, 47, 120, 120, 4, 0, 0],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 117, 0, 0, 0, 0, 0, 0, 126, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 243, 255, 137, 0, 0, 0, 0, 0, 0, 175, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 219, 255, 190, 0, 0, 0, 0, 0, 24, 247, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 154, 255, 255, 98, 0, 0, 0, 29, 198, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 37, 244, 255, 255, 224, 189, 205, 250, 246, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 54, 204, 255, 255, 255, 255, 199, 63, 5, 249, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 59, 55, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ů' / 0x16e
        'Ů' => &[
            &[0, 0, 0, 0, 0, 202, 210, 0, 0, 166, 248, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 199, 215, 0, 0, 171, 246, 2, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 112, 255, 202, 187, 253, 157, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 1, 105, 189, 195, 124, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 8, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 21, 255, 255, 89, 0, 0, 0, 0, 0, 0, 37, 255, 255, 72, 0, 0,
            ],
            &[
                0, 3, 254, 255, 110, 0, 0, 0, 0, 0, 0, 58, 255, 255, 53, 0, 0,
            ],
            &[
                0, 0, 222, 255, 162, 0, 0, 0, 0, 0, 0, 107, 255, 253, 16, 0, 0,
            ],
            &[
                0, 0, 155, 255, 244, 22, 0, 0, 0, 0, 3, 206, 255, 201, 0, 0, 0,
            ],
            &[
                0, 0, 46, 253, 255, 205, 51, 0, 0, 30, 169, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 125, 255, 255, 255, 245, 240, 255, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 90, 210, 255, 255, 255, 255, 221, 109, 1, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 25, 53, 56, 27, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 120, 255, 186, 171, 251, 165, 0, 0, 0, 0, 0, 0,
            ],
        ],
        // letter: 'ů' / 0x16f
        'ů' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 165, 246, 249, 182, 29, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 145, 254, 137, 121, 241, 192, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 208, 196, 0, 0, 152, 252, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 230, 10, 4, 193, 239, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 254, 241, 232, 255, 122, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 56, 140, 145, 74, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 117, 120, 54, 0, 0, 0, 0, 0, 0, 47, 120, 120, 4, 0, 0],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 117, 0, 0, 0, 0, 0, 0, 126, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 243, 255, 137, 0, 0, 0, 0, 0, 0, 175, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 219, 255, 190, 0, 0, 0, 0, 0, 24, 247, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 154, 255, 255, 98, 0, 0, 0, 29, 198, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 37, 244, 255, 255, 224, 189, 205, 250, 246, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 54, 204, 255, 255, 255, 255, 199, 63, 5, 249, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 59, 55, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ű' / 0x170
        'Ű' => &[
            &[
                0, 0, 0, 0, 0, 99, 255, 251, 56, 0, 160, 255, 227, 17, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 12, 230, 255, 106, 0, 51, 253, 245, 53, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 141, 255, 135, 0, 3, 201, 249, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 210, 122, 0, 0, 77, 211, 71, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 8, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 21, 255, 255, 89, 0, 0, 0, 0, 0, 0, 37, 255, 255, 72, 0, 0,
            ],
            &[
                0, 3, 254, 255, 110, 0, 0, 0, 0, 0, 0, 58, 255, 255, 53, 0, 0,
            ],
            &[
                0, 0, 222, 255, 162, 0, 0, 0, 0, 0, 0, 107, 255, 253, 16, 0, 0,
            ],
            &[
                0, 0, 155, 255, 244, 22, 0, 0, 0, 0, 3, 206, 255, 201, 0, 0, 0,
            ],
            &[
                0, 0, 46, 253, 255, 205, 51, 0, 0, 30, 169, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 125, 255, 255, 255, 245, 240, 255, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 90, 210, 255, 255, 255, 255, 221, 109, 1, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 25, 53, 56, 27, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 120, 124, 85, 0, 30, 124, 124, 54, 0, 0, 0],
        ],
        // letter: 'ű' / 0x171
        'ű' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 12, 170, 174, 110, 0, 51, 174, 174, 66, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 126, 255, 240, 32, 0, 187, 255, 203, 5, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 27, 244, 250, 68, 0, 78, 255, 228, 27, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 173, 251, 90, 0, 12, 224, 233, 45, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 163, 77, 0, 0, 67, 161, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 117, 120, 54, 0, 0, 0, 0, 0, 0, 47, 120, 120, 4, 0, 0],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 117, 0, 0, 0, 0, 0, 0, 126, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 243, 255, 137, 0, 0, 0, 0, 0, 0, 175, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 219, 255, 190, 0, 0, 0, 0, 0, 24, 247, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 154, 255, 255, 98, 0, 0, 0, 29, 198, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 37, 244, 255, 255, 224, 189, 205, 250, 246, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 54, 204, 255, 255, 255, 255, 199, 63, 5, 249, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 59, 55, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ų' / 0x172
        'Ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 71, 71, 22, 0, 0, 0, 0, 0, 0, 8, 71, 71, 22, 0, 0],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 28, 255, 255, 81, 0, 0, 0, 0, 0, 0, 30, 255, 255, 79, 0, 0,
            ],
            &[
                0, 21, 255, 255, 89, 0, 0, 0, 0, 0, 0, 37, 255, 255, 72, 0, 0,
            ],
            &[
                0, 3, 254, 255, 110, 0, 0, 0, 0, 0, 0, 58, 255, 255, 53, 0, 0,
            ],
            &[
                0, 0, 222, 255, 162, 0, 0, 0, 0, 0, 0, 107, 255, 253, 16, 0, 0,
            ],
            &[
                0, 0, 155, 255, 244, 22, 0, 0, 0, 0, 3, 206, 255, 201, 0, 0, 0,
            ],
            &[
                0, 0, 46, 253, 255, 205, 51, 0, 0, 30, 169, 255, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 125, 255, 255, 255, 245, 240, 255, 255, 255, 160, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 90, 210, 255, 255, 255, 255, 255, 114, 1, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 25, 96, 255, 232, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 236, 213, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 146, 255, 100, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 169, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 106, 255, 245, 182, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 134, 217, 232, 149, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ų' / 0x173
        'ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 117, 120, 54, 0, 0, 0, 0, 0, 0, 47, 120, 120, 4, 0, 0],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 99, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 116, 0, 0, 0, 0, 0, 0, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 249, 255, 117, 0, 0, 0, 0, 0, 0, 126, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 243, 255, 137, 0, 0, 0, 0, 0, 0, 175, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 219, 255, 190, 0, 0, 0, 0, 0, 24, 247, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 154, 255, 255, 98, 0, 0, 0, 29, 198, 255, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 37, 244, 255, 255, 224, 189, 205, 250, 246, 107, 255, 255, 10, 0, 0,
            ],
            &[
                0, 0, 0, 54, 204, 255, 255, 255, 255, 199, 63, 5, 255, 255, 10, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 59, 55, 25, 0, 0, 100, 253, 120, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 53, 250, 162, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 177, 255, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 214, 255, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 164, 255, 234, 176, 111, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 22, 165, 225, 228, 113, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŵ' / 0x174
        'Ŵ' => &[
            &[0, 0, 0, 0, 0, 7, 208, 255, 255, 230, 19, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 163, 255, 184, 166, 255, 194, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 130, 255, 179, 8, 2, 158, 255, 170, 3, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 64, 212, 138, 3, 0, 0, 0, 113, 212, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[22, 71, 66, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 51, 71, 36, 0],
            &[60, 255, 251, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 200, 255, 114, 0],
            &[32, 255, 255, 21, 0, 0, 0, 0, 0, 0, 0, 0, 0, 222, 255, 88, 0],
            &[5, 253, 255, 46, 0, 0, 0, 0, 0, 0, 0, 0, 0, 243, 255, 62, 0],
            &[0, 230, 255, 70, 0, 0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 36, 0],
            &[
                0, 202, 255, 94, 0, 0, 10, 120, 120, 41, 0, 0, 31, 255, 255, 11, 0,
            ],
            &[
                0, 173, 255, 119, 0, 0, 66, 255, 255, 133, 0, 0, 54, 255, 241, 0, 0,
            ],
            &[
                0, 144, 255, 143, 0, 0, 126, 255, 252, 194, 0, 0, 76, 255, 215, 0, 0,
            ],
            &[
                0, 116, 255, 168, 0, 0, 185, 251, 189, 248, 7, 0, 99, 255, 189, 0, 0,
            ],
            &[
                0, 87, 255, 192, 0, 3, 242, 211, 134, 255, 61, 0, 120, 255, 163, 0, 0,
            ],
            &[
                0, 58, 255, 215, 0, 48, 255, 158, 80, 255, 122, 0, 141, 255, 137, 0, 0,
            ],
            &[
                0, 30, 255, 237, 0, 108, 255, 101, 22, 255, 183, 0, 160, 255, 112, 0, 0,
            ],
            &[
                0, 4, 252, 253, 3, 166, 255, 42, 0, 216, 241, 3, 178, 255, 86, 0, 0,
            ],
            &[
                0, 0, 228, 255, 19, 222, 237, 1, 0, 154, 255, 49, 194, 255, 60, 0, 0,
            ],
            &[
                0, 0, 199, 255, 59, 255, 179, 0, 0, 92, 255, 108, 208, 255, 34, 0, 0,
            ],
            &[
                0, 0, 171, 255, 127, 255, 120, 0, 0, 31, 255, 163, 221, 255, 9, 0, 0,
            ],
            &[
                0, 0, 142, 255, 192, 255, 60, 0, 0, 0, 225, 213, 233, 238, 0, 0, 0,
            ],
            &[
                0, 0, 113, 255, 247, 249, 8, 0, 0, 0, 163, 251, 251, 213, 0, 0, 0,
            ],
            &[
                0, 0, 85, 255, 255, 198, 0, 0, 0, 0, 102, 255, 255, 187, 0, 0, 0,
            ],
            &[
                0, 0, 56, 255, 255, 138, 0, 0, 0, 0, 40, 255, 255, 161, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 124, 124, 57, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŵ' / 0x175
        'ŵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 62, 174, 174, 96, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 226, 255, 255, 247, 46, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 6, 193, 255, 149, 113, 255, 225, 25, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 2, 166, 255, 139, 0, 0, 103, 251, 209, 17, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 57, 163, 93, 0, 0, 0, 0, 62, 163, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[64, 120, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 55, 120, 91, 0],
            &[
                105, 255, 215, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 146, 255, 162, 0,
            ],
            &[64, 255, 249, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 187, 255, 120, 0],
            &[
                23, 255, 255, 35, 0, 0, 68, 190, 190, 100, 0, 0, 0, 227, 255, 78, 0,
            ],
            &[
                0, 238, 255, 72, 0, 0, 155, 255, 255, 196, 0, 0, 13, 255, 255, 36, 0,
            ],
            &[
                0, 197, 255, 110, 0, 0, 228, 253, 234, 251, 15, 0, 52, 255, 247, 2, 0,
            ],
            &[
                0, 156, 255, 148, 0, 46, 255, 207, 164, 255, 81, 0, 93, 255, 208, 0, 0,
            ],
            &[
                0, 114, 255, 185, 0, 120, 255, 145, 104, 255, 152, 0, 133, 255, 166, 0, 0,
            ],
            &[
                0, 73, 255, 219, 0, 190, 255, 77, 40, 255, 221, 0, 169, 255, 124, 0, 0,
            ],
            &[
                0, 32, 255, 249, 8, 247, 250, 12, 0, 230, 255, 33, 201, 255, 82, 0, 0,
            ],
            &[
                0, 1, 245, 255, 73, 255, 194, 0, 0, 165, 255, 94, 225, 255, 40, 0, 0,
            ],
            &[
                0, 0, 206, 255, 142, 255, 128, 0, 0, 99, 255, 150, 245, 249, 4, 0, 0,
            ],
            &[
                0, 0, 165, 255, 206, 255, 61, 0, 0, 34, 255, 206, 255, 212, 0, 0, 0,
            ],
            &[
                0, 0, 124, 255, 255, 244, 5, 0, 0, 0, 225, 255, 255, 170, 0, 0, 0,
            ],
            &[
                0, 0, 83, 255, 255, 184, 0, 0, 0, 0, 159, 255, 255, 128, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŷ' / 0x176
        'Ŷ' => &[
            &[0, 0, 0, 0, 0, 7, 208, 255, 255, 230, 19, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 163, 255, 184, 166, 255, 194, 7, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 130, 255, 179, 8, 2, 158, 255, 170, 3, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 64, 212, 138, 3, 0, 0, 0, 113, 212, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 69, 71, 48, 0, 0, 0, 0, 0, 0, 0, 0, 28, 71, 71, 13, 0],
            &[
                0, 168, 255, 238, 12, 0, 0, 0, 0, 0, 0, 0, 178, 255, 216, 3, 0,
            ],
            &[
                0, 41, 253, 255, 112, 0, 0, 0, 0, 0, 0, 43, 254, 255, 89, 0, 0,
            ],
            &[
                0, 0, 163, 255, 226, 4, 0, 0, 0, 0, 0, 161, 255, 212, 2, 0, 0,
            ],
            &[
                0, 0, 37, 252, 255, 93, 0, 0, 0, 0, 29, 250, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 159, 255, 210, 0, 0, 0, 0, 143, 255, 208, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 33, 250, 255, 74, 0, 0, 18, 244, 255, 80, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 192, 0, 0, 125, 255, 204, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 30, 249, 255, 55, 9, 234, 255, 75, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 149, 255, 173, 108, 255, 200, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 247, 253, 232, 255, 70, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 145, 255, 255, 196, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 255, 82, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 124, 124, 57, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŷ' / 0x177
        'ŷ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 174, 174, 91, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 230, 255, 255, 245, 41, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 8, 198, 255, 142, 119, 255, 222, 22, 0, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 3, 172, 255, 133, 0, 0, 109, 252, 204, 15, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 61, 163, 89, 0, 0, 0, 0, 67, 163, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 82, 120, 82, 0, 0, 0, 0, 0, 0, 0, 0, 73, 120, 105, 0, 0],
            &[
                0, 97, 255, 241, 11, 0, 0, 0, 0, 0, 0, 0, 217, 255, 152, 0, 0,
            ],
            &[
                0, 9, 237, 255, 100, 0, 0, 0, 0, 0, 0, 47, 255, 255, 57, 0, 0,
            ],
            &[
                0, 0, 140, 255, 204, 0, 0, 0, 0, 0, 0, 133, 255, 218, 0, 0, 0,
            ],
            &[
                0, 0, 35, 254, 255, 52, 0, 0, 0, 0, 0, 218, 255, 123, 0, 0, 0,
            ],
            &[
                0, 0, 0, 183, 255, 156, 0, 0, 0, 0, 48, 255, 254, 30, 0, 0, 0,
            ],
            &[
                0, 0, 0, 77, 255, 245, 15, 0, 0, 0, 133, 255, 189, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 2, 223, 255, 108, 0, 0, 0, 219, 255, 94, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 120, 255, 212, 0, 0, 49, 255, 243, 11, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 21, 248, 255, 56, 0, 133, 255, 160, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 163, 255, 150, 0, 215, 255, 65, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 255, 233, 43, 255, 224, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 206, 255, 179, 255, 131, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 255, 255, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 239, 255, 197, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 219, 255, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 64, 255, 247, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 178, 255, 155, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 69, 255, 254, 40, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 93, 88, 137, 248, 255, 164, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 69, 255, 255, 255, 255, 192, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 29, 133, 152, 135, 76, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ÿ' / 0x178
        'Ÿ' => &[
            &[0, 0, 0, 15, 143, 143, 16, 0, 0, 2, 124, 156, 36, 0, 0, 0, 0],
            &[
                0, 0, 0, 104, 255, 255, 108, 0, 0, 57, 255, 255, 155, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 76, 255, 255, 79, 0, 0, 33, 251, 255, 125, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 54, 53, 0, 0, 0, 0, 36, 68, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 69, 71, 48, 0, 0, 0, 0, 0, 0, 0, 0, 28, 71, 71, 13, 0],
            &[
                0, 168, 255, 238, 12, 0, 0, 0, 0, 0, 0, 0, 178, 255, 216, 3, 0,
            ],
            &[
                0, 41, 253, 255, 112, 0, 0, 0, 0, 0, 0, 43, 254, 255, 89, 0, 0,
            ],
            &[
                0, 0, 163, 255, 226, 4, 0, 0, 0, 0, 0, 161, 255, 212, 2, 0, 0,
            ],
            &[
                0, 0, 37, 252, 255, 93, 0, 0, 0, 0, 29, 250, 255, 84, 0, 0, 0,
            ],
            &[
                0, 0, 0, 159, 255, 210, 0, 0, 0, 0, 143, 255, 208, 1, 0, 0, 0,
            ],
            &[
                0, 0, 0, 33, 250, 255, 74, 0, 0, 18, 244, 255, 80, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 154, 255, 192, 0, 0, 125, 255, 204, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 30, 249, 255, 55, 9, 234, 255, 75, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 149, 255, 173, 108, 255, 200, 0, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 27, 247, 253, 232, 255, 70, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 145, 255, 255, 196, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 255, 82, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 66, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ź' / 0x179
        'Ź' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 173, 255, 247, 52, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 75, 255, 252, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 17, 227, 252, 94, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 121, 211, 76, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 63, 71, 71, 71, 71, 71, 71, 71, 71, 71, 45, 0, 0, 0],
            &[
                0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 162, 0, 0, 0,
            ],
            &[
                0, 0, 0, 189, 211, 211, 211, 211, 211, 211, 211, 252, 255, 153, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 255, 252, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 206, 255, 153, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 96, 255, 243, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 10, 228, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 127, 255, 228, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 244, 255, 96, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 158, 255, 207, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 253, 255, 67, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 189, 255, 181, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 77, 255, 252, 42, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 216, 255, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 255, 242, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 236, 255, 124, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 255, 227, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 34, 249, 255, 130, 40, 40, 40, 40, 40, 40, 40, 25, 0, 0, 0,
            ],
            &[
                0, 0, 110, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 162, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 162, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 31, 124, 124, 88, 0, 0, 0, 0],
        ],
        // letter: 'ź' / 0x17a
        'ź' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 63, 174, 174, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 211, 255, 225, 21, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 121, 255, 234, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 46, 248, 232, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 111, 159, 34, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 16, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 66, 0, 0, 0,
            ],
            &[
                0, 0, 35, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 141, 0, 0, 0,
            ],
            &[
                0, 0, 13, 100, 100, 100, 100, 100, 100, 100, 147, 255, 255, 110, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 220, 255, 183, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 178, 255, 222, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 122, 255, 247, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 67, 252, 255, 95, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 234, 255, 151, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 199, 255, 199, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 148, 255, 233, 28, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 255, 252, 64, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 245, 255, 116, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 14, 217, 255, 180, 13, 13, 13, 13, 13, 13, 13, 10, 0, 0, 0,
            ],
            &[
                0, 0, 133, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ż' / 0x17b
        'Ż' => &[
            &[0, 0, 0, 0, 0, 0, 25, 241, 255, 213, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 255, 255, 255, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 194, 250, 158, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 63, 71, 71, 71, 71, 71, 71, 71, 71, 71, 45, 0, 0, 0],
            &[
                0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 162, 0, 0, 0,
            ],
            &[
                0, 0, 0, 189, 211, 211, 211, 211, 211, 211, 211, 252, 255, 153, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 255, 252, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 206, 255, 153, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 96, 255, 243, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 10, 228, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 127, 255, 228, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 244, 255, 96, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 158, 255, 207, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 253, 255, 67, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 189, 255, 181, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 77, 255, 252, 42, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 216, 255, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 255, 242, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 236, 255, 124, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 255, 227, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 34, 249, 255, 130, 40, 40, 40, 40, 40, 40, 40, 25, 0, 0, 0,
            ],
            &[
                0, 0, 110, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 162, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 162, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 60, 4, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ż' / 0x17c
        'ż' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 56, 108, 28, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 50, 254, 255, 224, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 255, 255, 249, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 152, 207, 100, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 16, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 66, 0, 0, 0,
            ],
            &[
                0, 0, 35, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 141, 0, 0, 0,
            ],
            &[
                0, 0, 13, 100, 100, 100, 100, 100, 100, 100, 147, 255, 255, 110, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 220, 255, 183, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 178, 255, 222, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 122, 255, 247, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 67, 252, 255, 95, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 234, 255, 151, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 199, 255, 199, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 148, 255, 233, 28, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 255, 252, 64, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 245, 255, 116, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 14, 217, 255, 180, 13, 13, 13, 13, 13, 13, 13, 10, 0, 0, 0,
            ],
            &[
                0, 0, 133, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ž' / 0x17d
        'Ž' => &[
            &[
                0, 0, 0, 0, 139, 255, 181, 13, 0, 36, 212, 254, 96, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 161, 255, 201, 64, 232, 255, 111, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 5, 194, 255, 255, 255, 144, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 206, 212, 185, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 63, 71, 71, 71, 71, 71, 71, 71, 71, 71, 45, 0, 0, 0],
            &[
                0, 0, 0, 229, 255, 255, 255, 255, 255, 255, 255, 255, 255, 162, 0, 0, 0,
            ],
            &[
                0, 0, 0, 189, 211, 211, 211, 211, 211, 211, 211, 252, 255, 153, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 64, 255, 252, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 206, 255, 153, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 96, 255, 243, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 10, 228, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 127, 255, 228, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 244, 255, 96, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 158, 255, 207, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 253, 255, 67, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 189, 255, 181, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 77, 255, 252, 42, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 216, 255, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 255, 242, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 236, 255, 124, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 255, 227, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 34, 249, 255, 130, 40, 40, 40, 40, 40, 40, 40, 25, 0, 0, 0,
            ],
            &[
                0, 0, 110, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 162, 0, 0, 0,
            ],
            &[
                0, 0, 111, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 162, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 124, 98, 1, 0, 0, 0, 10, 115, 120, 0, 0, 0, 0],
        ],
        // letter: 'ž' / 0x17e
        'ž' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 172, 142, 10, 0, 0, 0, 39, 168, 149, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 107, 255, 211, 28, 0, 78, 243, 239, 49, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 132, 255, 222, 118, 252, 247, 61, 0, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 174, 255, 255, 255, 92, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 160, 163, 127, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 16, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 66, 0, 0, 0,
            ],
            &[
                0, 0, 35, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 141, 0, 0, 0,
            ],
            &[
                0, 0, 13, 100, 100, 100, 100, 100, 100, 100, 147, 255, 255, 110, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 220, 255, 183, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 178, 255, 222, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 122, 255, 247, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 67, 252, 255, 95, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 234, 255, 151, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 199, 255, 199, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 148, 255, 233, 28, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 255, 252, 64, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 245, 255, 116, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 14, 217, 255, 180, 13, 13, 13, 13, 13, 13, 13, 10, 0, 0, 0,
            ],
            &[
                0, 0, 133, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0,
            ],
            &[
                0, 0, 145, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ſ' / 0x17f
        'ſ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 0, 0, 0, 27, 104, 147, 162, 147, 118, 49, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 0, 115, 248, 255, 255, 255, 255, 255, 148, 0, 0, 0, 0,
            ],
            &[
                0, 0, 0, 0, 81, 255, 255, 217, 116, 88, 104, 145, 53, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 192, 255, 224, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 240, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 255, 255, 98, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        _ => panic!("unsupported char"),
    }
}
