//! Module for letters with the font weight regular and size 24.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 24;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 14;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight regular and font size 21px
#[inline]
pub const fn get_char(c: char) -> &'static [&'static [u8]] {
    match c {
        // letter: ' ' / 0x20
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '!' / 0x21
        '!' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 208, 254, 111, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 201, 255, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 193, 255, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 185, 255, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 177, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 169, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 161, 255, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 153, 255, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 145, 255, 48, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 137, 255, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 98, 193, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 78, 140, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 243, 255, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 255, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '"' / 0x22
        '"' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 152, 254, 80, 0, 182, 254, 51, 0, 0, 0, 0],
            &[0, 0, 0, 134, 255, 61, 0, 163, 255, 31, 0, 0, 0, 0],
            &[0, 0, 0, 114, 255, 41, 0, 143, 255, 11, 0, 0, 0, 0],
            &[0, 0, 0, 94, 255, 21, 0, 123, 247, 0, 0, 0, 0, 0],
            &[0, 0, 0, 74, 254, 3, 0, 104, 228, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 103, 0, 0, 38, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '#' / 0x23
        '#' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 254, 95, 0, 9, 250, 108, 0, 0, 0],
            &[0, 0, 0, 0, 83, 255, 47, 0, 53, 255, 61, 0, 0, 0],
            &[0, 0, 0, 0, 131, 249, 5, 0, 101, 254, 14, 0, 0, 0],
            &[0, 0, 0, 0, 179, 206, 0, 0, 150, 220, 0, 0, 0, 0],
            &[
                0, 111, 138, 138, 237, 217, 138, 138, 223, 223, 138, 138, 24, 0,
            ],
            &[
                0, 173, 215, 222, 255, 228, 215, 219, 255, 231, 215, 215, 38, 0,
            ],
            &[0, 0, 0, 68, 255, 52, 0, 53, 255, 71, 0, 0, 0, 0],
            &[0, 0, 0, 119, 251, 7, 0, 103, 255, 22, 0, 0, 0, 0],
            &[0, 0, 0, 170, 209, 0, 0, 152, 228, 0, 0, 0, 0, 0],
            &[
                118, 203, 203, 244, 240, 203, 203, 240, 244, 203, 203, 94, 0, 0,
            ],
            &[
                90, 156, 167, 255, 193, 156, 160, 255, 200, 156, 156, 72, 0, 0,
            ],
            &[0, 0, 62, 255, 62, 0, 46, 255, 79, 0, 0, 0, 0, 0],
            &[0, 0, 108, 255, 15, 0, 96, 255, 29, 0, 0, 0, 0, 0],
            &[0, 0, 155, 222, 0, 0, 146, 234, 0, 0, 0, 0, 0, 0],
            &[0, 0, 201, 175, 0, 0, 197, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '$' / 0x24
        '$' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 114, 208, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 56, 170, 236, 88, 54, 4, 0, 0, 0],
            &[0, 0, 0, 42, 213, 255, 255, 255, 255, 255, 235, 44, 0, 0],
            &[0, 0, 4, 221, 255, 141, 160, 228, 71, 123, 188, 4, 0, 0],
            &[0, 0, 59, 255, 169, 0, 122, 221, 0, 0, 0, 0, 0, 0],
            &[0, 0, 72, 255, 142, 0, 122, 221, 0, 0, 0, 0, 0, 0],
            &[0, 0, 24, 251, 233, 49, 122, 221, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 113, 254, 255, 237, 231, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 183, 253, 255, 249, 173, 52, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 241, 198, 255, 251, 71, 0, 0],
            &[0, 0, 0, 0, 0, 0, 122, 221, 0, 76, 255, 195, 0, 0],
            &[0, 0, 0, 0, 0, 0, 122, 221, 0, 0, 239, 226, 0, 0],
            &[0, 0, 4, 8, 0, 0, 122, 221, 0, 50, 255, 186, 0, 0],
            &[0, 0, 28, 242, 166, 121, 171, 237, 151, 243, 252, 59, 0, 0],
            &[0, 0, 17, 199, 253, 255, 255, 255, 253, 197, 57, 0, 0, 0],
            &[0, 0, 0, 0, 16, 47, 156, 228, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 122, 221, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '%' / 0x25
        '%' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[18, 185, 255, 255, 179, 15, 0, 0, 19, 242, 149, 0, 0, 0],
            &[150, 239, 59, 59, 240, 146, 0, 0, 134, 249, 29, 0, 0, 0],
            &[225, 165, 0, 0, 168, 225, 0, 19, 242, 151, 0, 0, 0, 0],
            &[246, 147, 0, 0, 151, 248, 0, 133, 249, 30, 0, 0, 0, 0],
            &[215, 178, 0, 0, 180, 218, 18, 242, 152, 0, 0, 0, 0, 0],
            &[113, 250, 108, 108, 250, 120, 133, 249, 30, 0, 0, 0, 0, 0],
            &[2, 124, 220, 220, 130, 22, 242, 153, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 132, 250, 31, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 241, 153, 33, 176, 225, 190, 53, 0, 0],
            &[0, 0, 0, 0, 131, 250, 36, 218, 208, 93, 181, 242, 23, 0],
            &[0, 0, 0, 17, 241, 154, 64, 255, 74, 0, 29, 255, 112, 0],
            &[0, 0, 0, 130, 250, 32, 94, 255, 43, 0, 1, 254, 144, 0],
            &[0, 0, 17, 241, 155, 0, 72, 255, 61, 0, 16, 255, 122, 0],
            &[0, 0, 130, 250, 32, 0, 15, 235, 169, 22, 129, 254, 49, 0],
            &[0, 16, 240, 155, 0, 0, 0, 69, 234, 255, 249, 112, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 3, 35, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '&' / 0x26
        '&' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 44, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 190, 255, 255, 240, 87, 0, 0, 0, 0, 0, 0],
            &[0, 0, 154, 255, 126, 74, 214, 246, 26, 0, 0, 0, 0, 0],
            &[0, 0, 219, 221, 0, 0, 97, 255, 81, 0, 0, 0, 0, 0],
            &[0, 0, 206, 228, 0, 0, 113, 255, 74, 0, 0, 0, 0, 0],
            &[0, 0, 151, 255, 45, 20, 224, 237, 10, 0, 0, 0, 0, 0],
            &[0, 0, 39, 253, 196, 202, 253, 84, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 196, 255, 255, 92, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 87, 255, 255, 255, 54, 0, 0, 25, 84, 37, 0, 0],
            &[0, 63, 251, 227, 71, 240, 228, 23, 0, 120, 255, 77, 0, 0],
            &[0, 206, 251, 50, 0, 79, 254, 195, 5, 201, 248, 12, 0, 0],
            &[23, 255, 186, 0, 0, 0, 136, 255, 210, 255, 165, 0, 0, 0],
            &[38, 255, 173, 0, 0, 0, 3, 198, 255, 255, 43, 0, 0, 0],
            &[9, 246, 237, 19, 0, 0, 12, 197, 255, 255, 56, 0, 0, 0],
            &[
                0, 148, 255, 219, 123, 126, 219, 255, 221, 253, 229, 25, 0, 0,
            ],
            &[0, 8, 153, 250, 255, 255, 223, 101, 0, 110, 255, 198, 6, 0],
            &[0, 0, 0, 14, 45, 32, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ''' / 0x27
        '\'' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 254, 63, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 151, 255, 43, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 131, 255, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 111, 255, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 92, 240, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 95, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '(' / 0x28
        '(' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 240, 175, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 195, 245, 27, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 82, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 194, 254, 32, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 254, 201, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 255, 140, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 255, 81, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 171, 255, 50, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 190, 255, 31, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 255, 24, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 180, 255, 42, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 160, 255, 65, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 255, 117, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 51, 255, 178, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 230, 245, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 255, 109, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 235, 221, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 87, 255, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 68, 74, 0, 0, 0, 0, 0],
        ],
        // letter: ')' / 0x29
        ')' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 245, 176, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 124, 255, 101, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 237, 233, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 255, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 46, 255, 185, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 237, 242, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 181, 255, 45, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 151, 255, 72, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 131, 255, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 127, 255, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 147, 255, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 170, 255, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 249, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 255, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 103, 255, 122, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 248, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 255, 143, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 216, 222, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 81, 35, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '*' / 0x2a
        '*' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 87, 170, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 114, 255, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 94, 248, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 173, 132, 68, 83, 231, 31, 101, 172, 122, 0, 0],
            &[0, 0, 8, 204, 238, 255, 254, 254, 255, 255, 241, 150, 0, 0],
            &[0, 0, 0, 0, 0, 37, 240, 246, 190, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 191, 224, 82, 255, 108, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 255, 97, 0, 198, 250, 61, 0, 0, 0],
            &[0, 0, 0, 0, 65, 180, 3, 0, 64, 174, 29, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '+' / 0x2b
        '+' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 72, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 255, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 255, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 255, 19, 0, 0, 0, 0, 0],
            &[0, 0, 12, 37, 37, 37, 132, 255, 54, 37, 37, 34, 0, 0],
            &[0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 235, 0, 0],
            &[0, 0, 28, 87, 87, 87, 160, 255, 100, 87, 87, 80, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 255, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 255, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 255, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 51, 117, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ',' / 0x2c
        ',' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 89, 181, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 162, 255, 178, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 209, 255, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 251, 243, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 255, 154, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 110, 240, 45, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '-' / 0x2d
        '-' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 110, 130, 130, 130, 130, 130, 58, 0, 0, 0, 0],
            &[0, 0, 0, 217, 255, 255, 255, 255, 255, 115, 0, 0, 0, 0],
            &[0, 0, 0, 36, 43, 43, 43, 43, 43, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '.' / 0x2e
        '.' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 78, 140, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 243, 255, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 255, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '/' / 0x2f
        '/' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 229, 224, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 76, 255, 129, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 174, 254, 33, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 250, 189, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 115, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 213, 239, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 55, 255, 150, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 255, 52, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 241, 209, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 94, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 192, 249, 19, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 254, 171, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 133, 255, 73, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 228, 227, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 73, 255, 132, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '0' / 0x30
        '0' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 41, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 83, 222, 255, 255, 253, 167, 24, 0, 0, 0, 0],
            &[0, 0, 78, 253, 244, 151, 128, 192, 255, 208, 11, 0, 0, 0],
            &[0, 2, 221, 245, 45, 0, 0, 5, 255, 255, 114, 0, 0, 0],
            &[0, 65, 255, 151, 0, 0, 0, 114, 255, 255, 214, 0, 0, 0],
            &[0, 119, 255, 77, 0, 0, 23, 241, 216, 174, 254, 16, 0, 0],
            &[0, 160, 255, 35, 0, 0, 160, 255, 70, 124, 255, 56, 0, 0],
            &[0, 175, 255, 17, 0, 57, 254, 174, 0, 104, 255, 72, 0, 0],
            &[0, 185, 255, 8, 2, 204, 246, 32, 0, 95, 255, 83, 0, 0],
            &[0, 174, 255, 15, 102, 255, 127, 0, 0, 101, 255, 76, 0, 0],
            &[0, 157, 255, 54, 235, 222, 8, 0, 0, 123, 255, 57, 0, 0],
            &[0, 114, 255, 227, 255, 79, 0, 0, 0, 165, 255, 21, 0, 0],
            &[0, 57, 255, 255, 183, 0, 0, 0, 6, 234, 216, 0, 0, 0],
            &[0, 0, 214, 255, 80, 0, 0, 0, 130, 255, 124, 0, 0, 0],
            &[0, 0, 67, 251, 240, 147, 123, 184, 255, 217, 13, 0, 0, 0],
            &[0, 0, 0, 74, 218, 255, 255, 253, 178, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 45, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '1' / 0x31
        '1' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 161, 251, 190, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 157, 251, 255, 255, 191, 0, 0, 0, 0, 0, 0],
            &[0, 0, 186, 255, 180, 54, 253, 191, 0, 0, 0, 0, 0, 0],
            &[0, 0, 41, 56, 0, 6, 255, 191, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 255, 191, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 255, 191, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 255, 191, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 255, 191, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 255, 191, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 255, 191, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 255, 191, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 255, 191, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 255, 191, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 78, 95, 127, 255, 255, 107, 90, 68, 0, 0, 0],
            &[0, 0, 52, 255, 255, 255, 255, 255, 255, 255, 235, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '2' / 0x32
        '2' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 28, 43, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 153, 243, 255, 255, 252, 181, 33, 0, 0, 0, 0],
            &[0, 36, 241, 255, 203, 134, 130, 199, 255, 224, 15, 0, 0, 0],
            &[0, 0, 141, 88, 0, 0, 0, 0, 171, 255, 107, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 82, 255, 153, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 75, 255, 152, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 137, 255, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 21, 240, 243, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 183, 255, 117, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 160, 255, 171, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 146, 255, 190, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 255, 196, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 151, 255, 191, 11, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 154, 255, 186, 9, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 114, 255, 255, 182, 174, 174, 174, 174, 174, 174, 77, 0, 0,
            ],
            &[
                0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '3' / 0x33
        '3' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 38, 43, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 59, 179, 248, 255, 255, 253, 186, 37, 0, 0, 0, 0],
            &[0, 37, 252, 251, 179, 128, 131, 200, 255, 228, 16, 0, 0, 0],
            &[0, 0, 90, 36, 0, 0, 0, 0, 179, 255, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 101, 255, 136, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 128, 255, 108, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 50, 238, 235, 20, 0, 0, 0],
            &[0, 0, 0, 36, 150, 155, 190, 250, 194, 44, 0, 0, 0, 0],
            &[0, 0, 0, 61, 255, 255, 255, 228, 149, 40, 0, 0, 0, 0],
            &[0, 0, 0, 5, 23, 26, 47, 110, 233, 247, 64, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 51, 255, 202, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 237, 251, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 10, 252, 241, 0, 0, 0],
            &[0, 34, 12, 0, 0, 0, 0, 1, 142, 255, 174, 0, 0, 0],
            &[0, 114, 245, 184, 148, 122, 142, 207, 255, 239, 39, 0, 0, 0],
            &[0, 55, 194, 250, 255, 255, 255, 244, 169, 33, 0, 0, 0, 0],
            &[0, 0, 0, 6, 30, 48, 32, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '4' / 0x34
        '4' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 15, 19, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 249, 255, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 187, 251, 255, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 166, 255, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 229, 200, 99, 255, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 253, 52, 106, 255, 93, 0, 0, 0, 0],
            &[0, 0, 0, 46, 251, 152, 0, 110, 255, 93, 0, 0, 0, 0],
            &[0, 0, 1, 196, 236, 18, 0, 112, 255, 93, 0, 0, 0, 0],
            &[0, 0, 97, 255, 100, 0, 0, 112, 255, 93, 0, 0, 0, 0],
            &[0, 17, 234, 201, 1, 0, 0, 112, 255, 93, 0, 0, 0, 0],
            &[0, 152, 253, 51, 0, 0, 0, 112, 255, 93, 0, 0, 0, 0],
            &[
                0, 242, 255, 250, 250, 250, 250, 252, 255, 252, 250, 137, 0, 0,
            ],
            &[
                0, 155, 163, 163, 163, 163, 163, 204, 255, 197, 163, 90, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 112, 255, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 112, 255, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 112, 255, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '5' / 0x35
        '5' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 121, 254, 254, 254, 254, 254, 254, 254, 9, 0, 0],
            &[0, 0, 0, 142, 255, 188, 175, 175, 175, 175, 175, 6, 0, 0],
            &[0, 0, 0, 163, 255, 26, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 183, 254, 4, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 204, 235, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 225, 212, 23, 44, 37, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 245, 255, 255, 255, 255, 246, 163, 25, 0, 0, 0],
            &[0, 0, 0, 97, 138, 102, 91, 120, 211, 255, 221, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 178, 255, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 49, 255, 203, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 17, 255, 222, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 56, 255, 194, 0, 0],
            &[0, 0, 25, 12, 0, 0, 0, 0, 11, 195, 255, 110, 0, 0],
            &[0, 0, 81, 240, 173, 138, 124, 155, 236, 255, 195, 6, 0, 0],
            &[0, 0, 37, 187, 250, 255, 255, 255, 229, 131, 8, 0, 0, 0],
            &[0, 0, 0, 0, 4, 29, 47, 27, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '6' / 0x36
        '6' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 34, 48, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 76, 190, 253, 255, 255, 255, 25, 0, 0, 0],
            &[0, 0, 0, 135, 255, 243, 164, 117, 105, 131, 16, 0, 0, 0],
            &[0, 0, 88, 255, 203, 27, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 220, 244, 26, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 255, 154, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 119, 255, 88, 4, 74, 113, 110, 53, 0, 0, 0, 0, 0],
            &[0, 156, 255, 84, 216, 255, 254, 255, 255, 184, 15, 0, 0, 0],
            &[0, 177, 255, 232, 147, 33, 3, 36, 164, 255, 170, 0, 0, 0],
            &[0, 186, 255, 147, 0, 0, 0, 0, 3, 211, 254, 27, 0, 0],
            &[0, 174, 255, 47, 0, 0, 0, 0, 0, 143, 255, 74, 0, 0],
            &[0, 145, 255, 55, 0, 0, 0, 0, 0, 131, 255, 77, 0, 0],
            &[0, 87, 255, 126, 0, 0, 0, 0, 0, 168, 255, 47, 0, 0],
            &[0, 9, 235, 241, 40, 0, 0, 0, 43, 249, 221, 0, 0, 0],
            &[0, 0, 87, 254, 243, 142, 104, 143, 243, 253, 72, 0, 0, 0],
            &[0, 0, 0, 79, 216, 255, 255, 255, 207, 68, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 47, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '7' / 0x37
        '7' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 188, 254, 254, 254, 254, 254, 254, 254, 254, 254, 86, 0, 0,
            ],
            &[
                0, 129, 175, 175, 175, 175, 175, 175, 175, 226, 255, 73, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 5, 230, 229, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 91, 255, 126, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 202, 250, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 57, 255, 167, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 168, 255, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 251, 207, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 134, 255, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 236, 237, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 100, 255, 139, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 253, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 66, 255, 179, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 177, 255, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 35, 253, 218, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '8' / 0x38
        '8' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 32, 45, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 113, 230, 255, 255, 254, 195, 51, 0, 0, 0, 0],
            &[0, 0, 127, 255, 219, 122, 104, 155, 250, 242, 35, 0, 0, 0],
            &[0, 1, 242, 239, 19, 0, 0, 0, 112, 255, 136, 0, 0, 0],
            &[0, 14, 255, 198, 0, 0, 0, 0, 55, 255, 160, 0, 0, 0],
            &[0, 0, 228, 239, 17, 0, 0, 0, 119, 255, 116, 0, 0, 0],
            &[0, 0, 83, 255, 208, 37, 0, 94, 249, 217, 11, 0, 0, 0],
            &[0, 0, 0, 103, 247, 251, 215, 255, 173, 20, 0, 0, 0, 0],
            &[0, 0, 0, 64, 222, 252, 250, 251, 134, 9, 0, 0, 0, 0],
            &[0, 0, 116, 255, 200, 46, 34, 167, 255, 213, 24, 0, 0, 0],
            &[0, 44, 253, 212, 14, 0, 0, 0, 108, 255, 180, 0, 0, 0],
            &[0, 124, 255, 98, 0, 0, 0, 0, 0, 201, 254, 16, 0, 0],
            &[0, 136, 255, 85, 0, 0, 0, 0, 0, 192, 255, 27, 0, 0],
            &[0, 88, 255, 171, 0, 0, 0, 0, 44, 248, 227, 1, 0, 0],
            &[0, 5, 204, 255, 186, 107, 93, 136, 241, 254, 84, 0, 0, 0],
            &[0, 0, 18, 154, 243, 255, 255, 255, 203, 70, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 37, 45, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '9' / 0x39
        '9' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 33, 43, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 131, 236, 255, 255, 250, 171, 26, 0, 0, 0, 0],
            &[0, 0, 173, 255, 208, 122, 105, 179, 255, 220, 14, 0, 0, 0],
            &[0, 71, 255, 188, 4, 0, 0, 0, 134, 255, 137, 0, 0, 0],
            &[0, 152, 255, 66, 0, 0, 0, 0, 5, 223, 232, 1, 0, 0],
            &[0, 181, 255, 28, 0, 0, 0, 0, 0, 156, 255, 35, 0, 0],
            &[0, 174, 255, 41, 0, 0, 0, 0, 0, 154, 255, 69, 0, 0],
            &[0, 125, 255, 109, 0, 0, 0, 0, 19, 234, 255, 82, 0, 0],
            &[0, 27, 242, 240, 91, 14, 6, 67, 210, 233, 255, 73, 0, 0],
            &[0, 0, 67, 234, 255, 255, 254, 254, 148, 154, 255, 55, 0, 0],
            &[0, 0, 0, 11, 84, 119, 101, 44, 0, 192, 255, 23, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 13, 247, 219, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 120, 255, 129, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 80, 249, 232, 18, 0, 0, 0],
            &[0, 0, 44, 120, 103, 124, 191, 255, 240, 59, 0, 0, 0, 0],
            &[0, 0, 74, 255, 255, 255, 242, 163, 34, 0, 0, 0, 0, 0],
            &[0, 0, 3, 34, 49, 34, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ':' / 0x3a
        ':' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 41, 94, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 229, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 221, 255, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 73, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 78, 139, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 243, 255, 136, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 255, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ';' / 0x3b
        ';' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 90, 51, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 110, 255, 248, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 103, 255, 242, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 69, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 181, 164, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 162, 255, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 209, 255, 93, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 251, 243, 12, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 61, 255, 154, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 110, 240, 45, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '<' / 0x3c
        '<' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 31, 103, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 14, 135, 247, 177, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 105, 234, 247, 136, 15, 0, 0],
            &[0, 0, 0, 0, 0, 75, 214, 253, 161, 29, 0, 0, 0, 0],
            &[0, 0, 0, 47, 187, 255, 186, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 24, 253, 253, 96, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 132, 241, 247, 149, 31, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 128, 239, 250, 158, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 125, 237, 253, 168, 44, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 13, 121, 235, 181, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 11, 70, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '=' / 0x3d
        '=' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 58, 58, 58, 58, 58, 58, 58, 58, 58, 7, 0, 0],
            &[
                0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 61, 115, 115, 115, 115, 115, 115, 115, 115, 115, 15, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 60, 60, 60, 60, 60, 60, 60, 60, 60, 7, 0, 0],
            &[
                0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 60, 113, 113, 113, 113, 113, 113, 113, 113, 113, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '>' / 0x3e
        '>' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 109, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 254, 216, 78, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 53, 192, 255, 191, 50, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 77, 214, 253, 160, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 102, 231, 245, 130, 12, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 11, 184, 255, 175, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 75, 198, 255, 207, 71, 0, 0],
            &[0, 0, 0, 0, 2, 84, 207, 255, 203, 78, 0, 0, 0, 0],
            &[0, 0, 2, 94, 215, 255, 200, 74, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 197, 71, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '?' / 0x3f
        '?' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 42, 44, 16, 0, 0, 0, 0, 0],
            &[0, 0, 4, 112, 205, 254, 255, 255, 253, 191, 48, 0, 0, 0],
            &[0, 0, 0, 205, 243, 168, 124, 120, 170, 253, 241, 36, 0, 0],
            &[0, 0, 0, 37, 12, 0, 0, 0, 0, 109, 255, 141, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 31, 255, 172, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 57, 255, 142, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 178, 255, 65, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 11, 173, 255, 151, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 204, 255, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 176, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 255, 172, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 30, 193, 95, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 138, 83, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 250, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 95, 254, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '@' / 0x40
        '@' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 141, 218, 245, 225, 165, 43, 0, 0, 0, 0],
            &[0, 0, 58, 236, 240, 149, 111, 129, 221, 247, 71, 0, 0, 0],
            &[0, 26, 236, 199, 20, 0, 0, 0, 9, 180, 242, 22, 0, 0],
            &[0, 156, 237, 21, 0, 0, 0, 0, 0, 16, 235, 143, 0, 0],
            &[12, 246, 129, 0, 30, 166, 224, 214, 174, 44, 141, 224, 0, 0],
            &[79, 255, 45, 13, 223, 224, 116, 168, 255, 80, 77, 255, 29, 0],
            &[127, 241, 2, 118, 255, 70, 0, 87, 255, 66, 44, 255, 52, 0],
            &[156, 217, 0, 189, 236, 0, 0, 102, 255, 52, 34, 255, 68, 0],
            &[170, 201, 0, 215, 206, 0, 0, 120, 255, 39, 43, 255, 54, 0],
            &[161, 209, 0, 202, 209, 0, 0, 153, 255, 30, 63, 255, 27, 0],
            &[140, 230, 0, 160, 247, 16, 4, 220, 253, 51, 118, 223, 0, 0],
            &[
                98, 255, 29, 54, 251, 210, 195, 203, 147, 212, 244, 117, 0, 0,
            ],
            &[27, 255, 111, 0, 70, 166, 148, 23, 24, 165, 137, 4, 0, 0],
            &[0, 178, 231, 17, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 244, 206, 35, 0, 0, 0, 0, 13, 28, 0, 0, 0],
            &[0, 0, 56, 237, 252, 187, 150, 157, 188, 246, 106, 0, 0, 0],
            &[0, 0, 0, 20, 124, 190, 217, 212, 176, 95, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'A' / 0x41
        'A' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 14, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 226, 255, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 255, 255, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 137, 255, 186, 255, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 219, 228, 78, 255, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 151, 11, 247, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 71, 0, 179, 253, 25, 0, 0, 0, 0],
            &[0, 0, 0, 212, 241, 6, 0, 100, 255, 106, 0, 0, 0, 0],
            &[0, 0, 39, 255, 168, 0, 0, 23, 253, 190, 0, 0, 0, 0],
            &[0, 0, 122, 255, 160, 71, 71, 71, 245, 252, 20, 0, 0, 0],
            &[0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0],
            &[0, 32, 255, 255, 102, 102, 102, 102, 140, 255, 183, 0, 0, 0],
            &[0, 114, 255, 106, 0, 0, 0, 0, 0, 214, 250, 16, 0, 0],
            &[0, 197, 254, 28, 0, 0, 0, 0, 0, 135, 255, 94, 0, 0],
            &[26, 254, 203, 0, 0, 0, 0, 0, 0, 55, 255, 177, 0, 0],
            &[107, 255, 124, 0, 0, 0, 0, 0, 0, 1, 230, 247, 12, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'B' / 0x42
        'B' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 254, 254, 254, 254, 242, 216, 160, 56, 0, 0, 0, 0],
            &[0, 1, 255, 235, 164, 164, 178, 214, 255, 254, 92, 0, 0, 0],
            &[0, 1, 255, 199, 0, 0, 0, 0, 98, 255, 218, 0, 0, 0],
            &[0, 1, 255, 199, 0, 0, 0, 0, 1, 244, 253, 1, 0, 0],
            &[0, 1, 255, 199, 0, 0, 0, 0, 2, 247, 233, 0, 0, 0],
            &[0, 1, 255, 199, 0, 0, 0, 0, 110, 255, 147, 0, 0, 0],
            &[0, 1, 255, 236, 166, 166, 176, 213, 243, 151, 9, 0, 0, 0],
            &[0, 1, 255, 255, 252, 252, 254, 255, 212, 118, 11, 0, 0, 0],
            &[0, 1, 255, 199, 0, 0, 3, 33, 136, 255, 201, 4, 0, 0],
            &[0, 1, 255, 199, 0, 0, 0, 0, 0, 191, 255, 72, 0, 0],
            &[0, 1, 255, 199, 0, 0, 0, 0, 0, 138, 255, 106, 0, 0],
            &[0, 1, 255, 199, 0, 0, 0, 0, 0, 162, 255, 88, 0, 0],
            &[0, 1, 255, 199, 0, 0, 0, 0, 59, 248, 250, 22, 0, 0],
            &[0, 1, 255, 235, 163, 163, 170, 201, 254, 255, 115, 0, 0, 0],
            &[0, 1, 255, 255, 255, 255, 254, 235, 184, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'C' / 0x43
        'C' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 45, 41, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 208, 255, 255, 255, 251, 182, 53, 0, 0],
            &[0, 0, 0, 145, 255, 246, 169, 129, 144, 187, 251, 43, 0, 0],
            &[0, 0, 92, 255, 221, 35, 0, 0, 0, 0, 18, 0, 0, 0],
            &[0, 1, 222, 254, 52, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 195, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 113, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 255, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 152, 255, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 137, 255, 98, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 115, 255, 127, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 255, 188, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 233, 254, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 255, 217, 32, 0, 0, 0, 0, 14, 0, 0, 0],
            &[0, 0, 3, 166, 255, 250, 179, 145, 157, 197, 226, 0, 0, 0],
            &[0, 0, 0, 2, 119, 228, 255, 255, 255, 247, 168, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 49, 34, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'D' / 0x44
        'D' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 254, 254, 242, 214, 146, 42, 0, 0, 0, 0, 0],
            &[0, 135, 255, 197, 164, 181, 226, 255, 250, 104, 0, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 1, 83, 240, 255, 75, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 69, 255, 225, 2, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 193, 255, 61, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 120, 255, 123, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 89, 255, 147, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 75, 255, 163, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 87, 255, 152, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 127, 255, 119, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 203, 255, 57, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 82, 255, 211, 1, 0, 0],
            &[0, 135, 255, 92, 0, 0, 2, 97, 246, 252, 54, 0, 0, 0],
            &[0, 135, 255, 196, 165, 184, 236, 255, 241, 80, 0, 0, 0, 0],
            &[0, 135, 255, 255, 254, 233, 204, 126, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'E' / 0x45
        'E' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 254, 254, 254, 254, 254, 254, 254, 254, 207, 0, 0],
            &[0, 0, 1, 255, 246, 175, 175, 175, 175, 175, 175, 143, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 172, 172, 172, 172, 172, 172, 71, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 1, 255, 226, 1, 1, 1, 1, 1, 1, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 174, 174, 174, 174, 174, 174, 141, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'F' / 0x46
        'F' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 254, 254, 254, 254, 254, 254, 254, 254, 207, 0, 0],
            &[0, 0, 1, 255, 246, 175, 175, 175, 175, 175, 175, 143, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 227, 5, 5, 5, 5, 5, 5, 2, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 1, 255, 245, 168, 168, 168, 168, 168, 168, 70, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'G' / 0x47
        'G' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 24, 44, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 141, 239, 255, 255, 255, 227, 120, 0, 0],
            &[0, 0, 0, 33, 220, 255, 231, 158, 135, 162, 227, 156, 0, 0],
            &[0, 0, 0, 196, 255, 158, 8, 0, 0, 0, 4, 17, 0, 0],
            &[0, 0, 78, 255, 205, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 158, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 220, 255, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 245, 244, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 255, 229, 0, 0, 0, 100, 234, 234, 234, 234, 30, 0],
            &[0, 1, 252, 238, 0, 0, 0, 83, 195, 195, 237, 255, 33, 0],
            &[0, 0, 235, 253, 12, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 188, 255, 67, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 115, 255, 172, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 15, 233, 255, 112, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 0, 72, 245, 255, 206, 144, 127, 156, 236, 255, 33, 0],
            &[0, 0, 0, 0, 49, 174, 251, 255, 255, 255, 234, 159, 15, 0],
            &[0, 0, 0, 0, 0, 0, 7, 35, 48, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'H' / 0x48
        'H' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 0, 193, 254, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[
                0, 135, 255, 202, 172, 172, 172, 172, 172, 235, 255, 33, 0, 0,
            ],
            &[
                0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 135, 255, 93, 1, 1, 1, 1, 1, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'I' / 0x49
        'I' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 175, 254, 254, 254, 254, 254, 254, 254, 73, 0, 0, 0],
            &[0, 0, 59, 106, 129, 218, 255, 165, 119, 96, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 104, 127, 218, 255, 164, 118, 95, 23, 0, 0, 0],
            &[0, 0, 176, 255, 255, 255, 255, 255, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'J' / 0x4a
        'J' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 31, 254, 194, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 38, 255, 192, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 73, 255, 170, 0, 0, 0],
            &[0, 0, 0, 13, 0, 0, 0, 5, 186, 255, 109, 0, 0, 0],
            &[0, 0, 0, 225, 195, 157, 146, 214, 255, 226, 14, 0, 0, 0],
            &[0, 0, 0, 169, 248, 255, 255, 254, 189, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 33, 46, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'K' / 0x4b
        'K' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 50, 247, 230, 22, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 14, 220, 253, 67, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 170, 255, 131, 0, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 105, 255, 194, 3, 0, 0, 0, 0],
            &[0, 135, 255, 92, 0, 48, 248, 236, 28, 0, 0, 0, 0, 0],
            &[0, 135, 255, 92, 13, 218, 254, 76, 0, 0, 0, 0, 0, 0],
            &[0, 135, 255, 92, 157, 255, 150, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 255, 163, 255, 255, 200, 2, 0, 0, 0, 0, 0, 0],
            &[0, 135, 255, 255, 173, 172, 255, 109, 0, 0, 0, 0, 0, 0],
            &[0, 135, 255, 199, 9, 26, 241, 242, 28, 0, 0, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 108, 255, 178, 0, 0, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 2, 201, 255, 84, 0, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 48, 251, 230, 15, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 140, 255, 153, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 11, 225, 254, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'L' / 0x4c
        'L' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 254, 77, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 210, 190, 190, 190, 190, 190, 154, 0, 0, 0],
            &[0, 0, 149, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'M' / 0x4d
        'M' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 241, 254, 228, 0, 0, 0, 0, 81, 254, 254, 139, 0, 0],
            &[0, 243, 252, 255, 46, 0, 0, 0, 154, 253, 255, 140, 0, 0],
            &[0, 243, 205, 255, 119, 0, 0, 0, 226, 204, 255, 140, 0, 0],
            &[0, 243, 181, 227, 192, 0, 0, 44, 255, 149, 255, 140, 0, 0],
            &[0, 243, 197, 151, 250, 14, 0, 116, 255, 86, 255, 140, 0, 0],
            &[0, 243, 207, 75, 255, 81, 0, 189, 221, 54, 255, 140, 0, 0],
            &[0, 243, 212, 9, 246, 154, 12, 249, 145, 58, 255, 140, 0, 0],
            &[0, 243, 213, 0, 179, 226, 79, 255, 70, 60, 255, 140, 0, 0],
            &[0, 243, 213, 0, 103, 255, 195, 243, 6, 60, 255, 140, 0, 0],
            &[0, 243, 213, 0, 27, 255, 255, 174, 0, 60, 255, 140, 0, 0],
            &[0, 243, 213, 0, 0, 206, 255, 98, 0, 60, 255, 140, 0, 0],
            &[0, 243, 213, 0, 0, 41, 67, 13, 0, 60, 255, 140, 0, 0],
            &[0, 243, 213, 0, 0, 0, 0, 0, 0, 60, 255, 140, 0, 0],
            &[0, 243, 213, 0, 0, 0, 0, 0, 0, 60, 255, 140, 0, 0],
            &[0, 243, 213, 0, 0, 0, 0, 0, 0, 60, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'N' / 0x4e
        'N' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 254, 47, 0, 0, 0, 0, 177, 254, 33, 0, 0],
            &[0, 135, 255, 254, 159, 0, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 182, 248, 22, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 91, 255, 126, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 57, 196, 231, 6, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 69, 84, 255, 94, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 75, 3, 224, 206, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 115, 255, 62, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 15, 243, 174, 0, 177, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 146, 252, 33, 173, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 36, 253, 141, 164, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 177, 240, 163, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 65, 255, 236, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 0, 208, 255, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 0, 96, 255, 255, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'O' / 0x4f
        'O' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 33, 54, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 231, 255, 255, 255, 195, 42, 0, 0, 0],
            &[0, 0, 0, 110, 255, 241, 147, 119, 169, 254, 237, 34, 0, 0],
            &[0, 0, 23, 247, 239, 35, 0, 0, 0, 102, 255, 176, 0, 0],
            &[0, 0, 124, 255, 124, 0, 0, 0, 0, 0, 218, 254, 25, 0],
            &[0, 0, 187, 255, 44, 0, 0, 0, 0, 0, 145, 255, 92, 0],
            &[0, 0, 234, 251, 5, 0, 0, 0, 0, 0, 102, 255, 134, 0],
            &[0, 1, 252, 236, 0, 0, 0, 0, 0, 0, 79, 255, 155, 0],
            &[0, 10, 255, 227, 0, 0, 0, 0, 0, 0, 73, 255, 164, 0],
            &[0, 2, 255, 236, 0, 0, 0, 0, 0, 0, 82, 255, 155, 0],
            &[0, 0, 236, 252, 6, 0, 0, 0, 0, 0, 103, 255, 132, 0],
            &[0, 0, 195, 255, 46, 0, 0, 0, 0, 0, 146, 255, 89, 0],
            &[0, 0, 126, 255, 127, 0, 0, 0, 0, 1, 224, 253, 22, 0],
            &[0, 0, 29, 248, 241, 39, 0, 0, 0, 119, 255, 170, 0, 0],
            &[0, 0, 0, 114, 255, 244, 155, 131, 186, 255, 232, 29, 0, 0],
            &[0, 0, 0, 0, 98, 226, 255, 255, 253, 182, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 32, 45, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'P' / 0x50
        'P' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 254, 254, 254, 254, 250, 232, 182, 79, 0, 0, 0, 0],
            &[0, 1, 255, 245, 164, 164, 175, 209, 255, 255, 125, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 66, 249, 251, 28, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 166, 255, 92, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 132, 255, 107, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 160, 255, 79, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 43, 243, 244, 12, 0, 0],
            &[0, 1, 255, 238, 105, 106, 118, 163, 247, 255, 103, 0, 0, 0],
            &[0, 1, 255, 255, 255, 255, 255, 255, 207, 82, 0, 0, 0, 0],
            &[0, 1, 255, 233, 57, 57, 45, 20, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Q' / 0x51
        'Q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 33, 54, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 231, 255, 255, 255, 195, 42, 0, 0, 0],
            &[0, 0, 0, 110, 255, 241, 147, 119, 169, 254, 237, 34, 0, 0],
            &[0, 0, 23, 247, 239, 35, 0, 0, 0, 102, 255, 176, 0, 0],
            &[0, 0, 124, 255, 124, 0, 0, 0, 0, 0, 218, 254, 25, 0],
            &[0, 0, 187, 255, 44, 0, 0, 0, 0, 0, 145, 255, 92, 0],
            &[0, 0, 234, 251, 5, 0, 0, 0, 0, 0, 102, 255, 134, 0],
            &[0, 1, 252, 236, 0, 0, 0, 0, 0, 0, 79, 255, 155, 0],
            &[0, 10, 255, 227, 0, 0, 0, 0, 0, 0, 73, 255, 164, 0],
            &[0, 2, 255, 236, 0, 0, 0, 0, 0, 0, 82, 255, 155, 0],
            &[0, 0, 236, 252, 6, 0, 0, 0, 0, 0, 103, 255, 132, 0],
            &[0, 0, 195, 255, 46, 0, 0, 0, 0, 0, 146, 255, 89, 0],
            &[0, 0, 126, 255, 127, 0, 0, 0, 0, 1, 224, 253, 22, 0],
            &[0, 0, 29, 248, 241, 39, 0, 0, 0, 119, 255, 170, 0, 0],
            &[0, 0, 0, 114, 255, 244, 155, 131, 225, 255, 232, 29, 0, 0],
            &[0, 0, 0, 0, 98, 226, 255, 255, 255, 255, 66, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 32, 45, 116, 255, 161, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 221, 253, 43, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 97, 255, 175, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 182, 198, 31, 0],
        ],
        // letter: 'R' / 0x52
        'R' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 254, 254, 254, 252, 235, 197, 116, 8, 0, 0, 0, 0],
            &[0, 1, 255, 245, 164, 169, 192, 246, 255, 203, 11, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 19, 203, 255, 114, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 82, 255, 177, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 53, 255, 190, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 94, 255, 154, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 46, 224, 253, 53, 0, 0, 0],
            &[0, 1, 255, 249, 197, 199, 222, 255, 238, 92, 0, 0, 0, 0],
            &[0, 1, 255, 252, 222, 222, 246, 254, 44, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 99, 255, 165, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 2, 203, 255, 63, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 55, 254, 212, 4, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 159, 255, 114, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 23, 241, 242, 25, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 114, 255, 165, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'S' / 0x53
        'S' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 43, 43, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 58, 196, 254, 255, 255, 255, 207, 98, 0, 0, 0],
            &[0, 0, 52, 247, 255, 189, 137, 140, 173, 241, 123, 0, 0, 0],
            &[0, 0, 171, 255, 122, 0, 0, 0, 0, 7, 11, 0, 0, 0],
            &[0, 0, 214, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 204, 255, 57, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 255, 208, 33, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 190, 255, 250, 158, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 104, 227, 255, 253, 169, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 101, 218, 255, 244, 61, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 139, 255, 213, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 2, 239, 255, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 228, 255, 16, 0, 0],
            &[0, 15, 22, 0, 0, 0, 0, 0, 75, 255, 219, 0, 0, 0],
            &[0, 55, 252, 193, 137, 111, 114, 165, 251, 254, 83, 0, 0, 0],
            &[0, 25, 167, 240, 255, 255, 255, 254, 203, 73, 0, 0, 0, 0],
            &[0, 0, 0, 2, 30, 48, 44, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'T' / 0x54
        'T' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 13, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 166, 0,
            ],
            &[
                0, 9, 175, 175, 175, 175, 227, 255, 195, 175, 175, 175, 114, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'U' / 0x55
        'U' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 0, 193, 254, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 134, 255, 93, 0, 0, 0, 0, 0, 195, 255, 32, 0, 0],
            &[0, 119, 255, 108, 0, 0, 0, 0, 0, 210, 255, 17, 0, 0],
            &[0, 86, 255, 156, 0, 0, 0, 0, 8, 247, 236, 1, 0, 0],
            &[0, 21, 249, 242, 33, 0, 0, 0, 117, 255, 162, 0, 0, 0],
            &[0, 0, 134, 255, 237, 149, 129, 178, 254, 241, 36, 0, 0, 0],
            &[0, 0, 1, 123, 236, 255, 255, 255, 194, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 32, 44, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'V' / 0x56
        'V' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[80, 254, 164, 0, 0, 0, 0, 0, 0, 19, 251, 231, 2, 0],
            &[9, 244, 238, 3, 0, 0, 0, 0, 0, 90, 255, 154, 0, 0],
            &[0, 171, 255, 62, 0, 0, 0, 0, 0, 165, 255, 73, 0, 0],
            &[0, 89, 255, 139, 0, 0, 0, 0, 2, 236, 241, 6, 0, 0],
            &[0, 14, 248, 216, 0, 0, 0, 0, 57, 255, 166, 0, 0, 0],
            &[0, 0, 180, 255, 37, 0, 0, 0, 132, 255, 85, 0, 0, 0],
            &[0, 0, 98, 255, 113, 0, 0, 0, 206, 248, 12, 0, 0, 0],
            &[0, 0, 19, 251, 190, 0, 0, 26, 254, 179, 0, 0, 0, 0],
            &[0, 0, 0, 189, 251, 15, 0, 99, 255, 98, 0, 0, 0, 0],
            &[0, 0, 0, 106, 255, 87, 0, 173, 252, 21, 0, 0, 0, 0],
            &[0, 0, 0, 25, 254, 162, 4, 241, 192, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 197, 231, 57, 255, 111, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 255, 159, 255, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 255, 251, 205, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 124, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'W' / 0x57
        'W' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[106, 254, 80, 0, 0, 0, 0, 0, 0, 0, 181, 253, 7, 0],
            &[78, 255, 105, 0, 0, 0, 0, 0, 0, 0, 204, 236, 0, 0],
            &[50, 255, 129, 0, 0, 0, 0, 0, 0, 0, 224, 210, 0, 0],
            &[21, 255, 153, 0, 0, 40, 65, 16, 0, 0, 246, 184, 0, 0],
            &[0, 247, 178, 0, 0, 193, 255, 103, 0, 13, 255, 159, 0, 0],
            &[0, 219, 202, 0, 6, 247, 242, 164, 0, 36, 255, 133, 0, 0],
            &[0, 191, 226, 0, 57, 255, 159, 226, 0, 58, 255, 107, 0, 0],
            &[0, 162, 249, 0, 116, 240, 70, 255, 31, 79, 255, 81, 0, 0],
            &[0, 133, 255, 15, 176, 186, 13, 252, 92, 98, 255, 55, 0, 0],
            &[0, 105, 255, 35, 233, 126, 0, 204, 153, 115, 255, 30, 0, 0],
            &[0, 76, 255, 86, 255, 67, 0, 142, 213, 130, 254, 5, 0, 0],
            &[0, 47, 255, 156, 252, 12, 0, 81, 254, 158, 234, 0, 0, 0],
            &[0, 19, 255, 220, 204, 0, 0, 20, 254, 219, 208, 0, 0, 0],
            &[0, 0, 246, 255, 145, 0, 0, 0, 213, 255, 182, 0, 0, 0],
            &[0, 0, 217, 255, 86, 0, 0, 0, 152, 255, 156, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'X' / 0x58
        'X' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 165, 254, 92, 0, 0, 0, 0, 0, 184, 253, 64, 0],
            &[0, 0, 26, 242, 226, 9, 0, 0, 0, 75, 255, 167, 0, 0],
            &[0, 0, 0, 114, 255, 124, 0, 0, 4, 215, 243, 27, 0, 0],
            &[0, 0, 0, 4, 211, 243, 24, 0, 109, 255, 117, 0, 0, 0],
            &[0, 0, 0, 0, 62, 255, 154, 17, 237, 214, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 163, 252, 180, 255, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 240, 255, 170, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 196, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 86, 255, 249, 228, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 228, 224, 76, 255, 139, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 255, 85, 0, 175, 251, 43, 0, 0, 0],
            &[0, 0, 0, 43, 250, 194, 0, 0, 35, 248, 192, 0, 0, 0],
            &[0, 0, 0, 192, 253, 50, 0, 0, 0, 136, 255, 91, 0, 0],
            &[0, 0, 91, 255, 156, 0, 0, 0, 0, 13, 231, 231, 14, 0],
            &[0, 14, 231, 242, 23, 0, 0, 0, 0, 0, 96, 255, 144, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Y' / 0x59
        'Y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 239, 244, 20, 0, 0, 0, 0, 0, 97, 254, 155, 0],
            &[0, 0, 129, 255, 129, 0, 0, 0, 0, 1, 214, 250, 32, 0],
            &[0, 0, 15, 238, 236, 10, 0, 0, 0, 79, 255, 152, 0, 0],
            &[0, 0, 0, 124, 255, 110, 0, 0, 0, 198, 248, 29, 0, 0],
            &[0, 0, 0, 13, 236, 224, 4, 0, 62, 255, 147, 0, 0, 0],
            &[0, 0, 0, 0, 119, 255, 90, 0, 181, 247, 26, 0, 0, 0],
            &[0, 0, 0, 0, 11, 233, 208, 45, 254, 142, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 114, 255, 214, 245, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 231, 255, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 156, 255, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Z' / 0x5a
        'Z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 148, 254, 254, 254, 254, 254, 254, 254, 153, 0, 0],
            &[0, 0, 0, 96, 164, 164, 164, 164, 164, 205, 255, 136, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 2, 209, 243, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 99, 255, 125, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 231, 228, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 130, 255, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 246, 208, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 162, 255, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 254, 182, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 252, 43, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 255, 153, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 218, 243, 23, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 255, 124, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 255, 183, 174, 174, 174, 174, 174, 105, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 154, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '[' / 0x5b
        '[' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 244, 254, 254, 254, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 236, 132, 132, 101, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 224, 49, 49, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 255, 255, 255, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 78, 81, 81, 81, 62, 0, 0, 0, 0, 0],
        ],
        // letter: '\' / 0x5c
        '\\' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 73, 254, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 228, 227, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 133, 255, 72, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 254, 170, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 193, 249, 19, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 95, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 242, 209, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 255, 149, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 213, 239, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 115, 255, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 250, 188, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 175, 254, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 77, 255, 129, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 230, 225, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ']' / 0x5d
        ']' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 52, 52, 52, 52, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 255, 255, 255, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 78, 78, 122, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 62, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 62, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 62, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 62, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 62, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 62, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 62, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 62, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 62, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 62, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 62, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 62, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 62, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 62, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 103, 103, 140, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 255, 255, 255, 255, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 27, 27, 27, 27, 15, 0, 0, 0, 0],
        ],
        // letter: '^' / 0x5e
        '^' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 189, 225, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 255, 253, 109, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 175, 204, 149, 232, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 254, 88, 27, 247, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 162, 225, 4, 0, 145, 238, 16, 0, 0, 0, 0],
            &[0, 0, 31, 251, 113, 0, 0, 25, 246, 132, 0, 0, 0, 0],
            &[0, 0, 148, 240, 13, 0, 0, 0, 141, 243, 22, 0, 0, 0],
            &[0, 22, 246, 138, 0, 0, 0, 0, 22, 245, 143, 0, 0, 0],
            &[0, 134, 250, 28, 0, 0, 0, 0, 0, 138, 248, 29, 0, 0],
            &[0, 86, 78, 0, 0, 0, 0, 0, 0, 18, 100, 44, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '_' / 0x5f
        '_' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 68, 0,
            ],
            &[
                255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 153, 0,
            ],
            &[59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 35, 0],
        ],
        // letter: '`' / 0x60
        '`' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 22, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 188, 255, 127, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 212, 246, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 209, 199, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 70, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'a' / 0x61
        'a' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 68, 106, 114, 91, 27, 0, 0, 0, 0],
            &[0, 0, 0, 103, 237, 255, 255, 255, 255, 252, 131, 0, 0, 0],
            &[0, 0, 0, 71, 180, 83, 41, 27, 76, 217, 255, 77, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 156, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 255, 179, 0, 0],
            &[0, 0, 0, 0, 46, 130, 176, 197, 206, 213, 255, 181, 0, 0],
            &[0, 0, 0, 136, 254, 245, 186, 153, 139, 132, 255, 181, 0, 0],
            &[0, 0, 59, 255, 199, 20, 0, 0, 0, 6, 255, 181, 0, 0],
            &[0, 0, 121, 255, 80, 0, 0, 0, 0, 45, 255, 181, 0, 0],
            &[0, 0, 109, 255, 103, 0, 0, 0, 4, 179, 255, 181, 0, 0],
            &[0, 0, 41, 249, 239, 126, 89, 117, 211, 201, 252, 181, 0, 0],
            &[0, 0, 0, 83, 227, 255, 255, 249, 154, 13, 222, 181, 0, 0],
            &[0, 0, 0, 0, 2, 30, 41, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'b' / 0x62
        'b' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 78, 245, 124, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 128, 0, 60, 107, 104, 44, 0, 0, 0, 0, 0],
            &[0, 81, 255, 134, 189, 255, 255, 255, 255, 158, 4, 0, 0, 0],
            &[0, 81, 255, 246, 227, 78, 30, 69, 214, 255, 130, 0, 0, 0],
            &[0, 81, 255, 254, 42, 0, 0, 0, 26, 250, 241, 10, 0, 0],
            &[0, 81, 255, 191, 0, 0, 0, 0, 0, 181, 255, 66, 0, 0],
            &[0, 81, 255, 152, 0, 0, 0, 0, 0, 145, 255, 100, 0, 0],
            &[0, 81, 255, 137, 0, 0, 0, 0, 0, 131, 255, 109, 0, 0],
            &[0, 81, 255, 160, 0, 0, 0, 0, 0, 152, 255, 94, 0, 0],
            &[0, 81, 255, 211, 0, 0, 0, 0, 0, 201, 255, 51, 0, 0],
            &[0, 81, 255, 255, 77, 0, 0, 0, 62, 255, 222, 1, 0, 0],
            &[0, 81, 255, 217, 249, 142, 96, 134, 244, 254, 82, 0, 0, 0],
            &[0, 81, 255, 65, 124, 241, 255, 255, 230, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 42, 38, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'c' / 0x63
        'c' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 57, 99, 111, 84, 29, 0, 0, 0],
            &[0, 0, 0, 0, 63, 221, 255, 255, 255, 255, 253, 101, 0, 0],
            &[0, 0, 0, 45, 246, 253, 148, 68, 51, 79, 161, 41, 0, 0],
            &[0, 0, 0, 176, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 248, 243, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 255, 200, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 188, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 31, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 243, 249, 13, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 164, 255, 148, 0, 0, 0, 0, 0, 2, 0, 0],
            &[0, 0, 0, 27, 235, 255, 200, 125, 112, 139, 193, 120, 0, 0],
            &[0, 0, 0, 0, 31, 173, 249, 255, 255, 255, 222, 79, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 38, 44, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'd' / 0x64
        'd' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 233, 226, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 242, 235, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 242, 235, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 242, 235, 0, 0],
            &[0, 0, 0, 0, 2, 67, 105, 98, 39, 0, 241, 235, 0, 0],
            &[0, 0, 0, 43, 219, 255, 255, 255, 252, 121, 227, 235, 0, 0],
            &[0, 0, 10, 226, 255, 167, 53, 30, 109, 250, 248, 235, 0, 0],
            &[0, 0, 103, 255, 192, 0, 0, 0, 0, 133, 255, 235, 0, 0],
            &[0, 0, 171, 255, 87, 0, 0, 0, 0, 41, 255, 235, 0, 0],
            &[0, 0, 200, 255, 48, 0, 0, 0, 0, 2, 254, 235, 0, 0],
            &[0, 0, 211, 255, 28, 0, 0, 0, 0, 0, 244, 235, 0, 0],
            &[0, 0, 196, 255, 48, 0, 0, 0, 0, 7, 254, 235, 0, 0],
            &[0, 0, 154, 255, 93, 0, 0, 0, 0, 50, 255, 235, 0, 0],
            &[0, 0, 74, 255, 204, 8, 0, 0, 1, 167, 255, 235, 0, 0],
            &[0, 0, 1, 188, 255, 204, 113, 103, 181, 235, 231, 235, 0, 0],
            &[0, 0, 0, 13, 162, 250, 255, 255, 211, 52, 180, 235, 0, 0],
            &[0, 0, 0, 0, 0, 13, 45, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'e' / 0x65
        'e' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 85, 111, 91, 25, 0, 0, 0, 0, 0],
            &[0, 0, 2, 131, 248, 255, 255, 255, 250, 127, 1, 0, 0, 0],
            &[0, 0, 139, 255, 202, 70, 27, 55, 192, 255, 106, 0, 0, 0],
            &[0, 38, 253, 219, 11, 0, 0, 0, 14, 236, 232, 2, 0, 0],
            &[0, 125, 255, 109, 0, 0, 0, 0, 0, 166, 255, 50, 0, 0],
            &[
                0, 166, 255, 214, 200, 200, 200, 200, 200, 231, 255, 77, 0, 0,
            ],
            &[
                0, 182, 255, 199, 186, 186, 186, 186, 186, 186, 186, 63, 0, 0,
            ],
            &[0, 161, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 109, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 242, 240, 43, 0, 0, 0, 0, 0, 24, 0, 0, 0],
            &[0, 0, 85, 251, 247, 156, 103, 99, 133, 197, 213, 0, 0, 0],
            &[0, 0, 0, 55, 185, 252, 255, 255, 255, 213, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'f' / 0x66
        'f' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 5, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 116, 219, 255, 255, 244, 174, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 238, 152, 131, 153, 129, 0, 0],
            &[0, 0, 0, 0, 0, 211, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 237, 250, 2, 0, 0, 0, 0, 0, 0],
            &[0, 1, 16, 35, 54, 244, 246, 65, 65, 65, 65, 12, 0, 0],
            &[
                0, 156, 255, 255, 255, 255, 255, 255, 255, 255, 255, 49, 0, 0,
            ],
            &[0, 50, 81, 81, 81, 245, 247, 81, 81, 81, 81, 15, 0, 0],
            &[0, 0, 0, 0, 0, 240, 243, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 240, 243, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 240, 243, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 240, 243, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 240, 243, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 240, 243, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 240, 243, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 240, 243, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 240, 243, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'g' / 0x67
        'g' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 71, 111, 101, 36, 0, 40, 60, 0, 0],
            &[0, 0, 0, 34, 213, 255, 255, 255, 252, 117, 186, 235, 0, 0],
            &[0, 0, 6, 213, 255, 155, 51, 29, 100, 243, 250, 235, 0, 0],
            &[0, 0, 91, 255, 181, 0, 0, 0, 0, 94, 255, 235, 0, 0],
            &[0, 0, 164, 255, 82, 0, 0, 0, 0, 7, 247, 235, 0, 0],
            &[0, 0, 201, 255, 44, 0, 0, 0, 0, 0, 214, 235, 0, 0],
            &[0, 0, 212, 255, 29, 0, 0, 0, 0, 0, 201, 235, 0, 0],
            &[0, 0, 198, 255, 52, 0, 0, 0, 0, 0, 220, 235, 0, 0],
            &[0, 0, 155, 255, 103, 0, 0, 0, 0, 11, 249, 235, 0, 0],
            &[0, 0, 72, 255, 214, 13, 0, 0, 0, 123, 255, 235, 0, 0],
            &[0, 0, 0, 180, 255, 213, 119, 97, 160, 252, 242, 235, 0, 0],
            &[0, 0, 0, 10, 153, 248, 255, 255, 238, 94, 212, 235, 0, 0],
            &[0, 0, 0, 0, 0, 11, 44, 40, 4, 0, 224, 229, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 252, 204, 0, 0],
            &[0, 0, 17, 12, 0, 0, 0, 0, 0, 129, 255, 136, 0, 0],
            &[0, 0, 0, 0, 0, 14, 32, 21, 2, 0, 0, 0, 0, 0],
        ],
        // letter: 'h' / 0x68
        'h' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 78, 245, 140, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 145, 0, 35, 94, 103, 59, 0, 0, 0, 0, 0],
            &[0, 81, 255, 137, 130, 251, 255, 255, 255, 191, 19, 0, 0, 0],
            &[0, 81, 255, 212, 231, 83, 27, 60, 205, 255, 156, 0, 0, 0],
            &[0, 81, 255, 254, 54, 0, 0, 0, 39, 255, 233, 0, 0, 0],
            &[0, 81, 255, 202, 0, 0, 0, 0, 0, 237, 255, 3, 0, 0],
            &[0, 81, 255, 159, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 146, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'i' / 0x69
        'i' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 223, 101, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 212, 255, 185, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 69, 157, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 64, 65, 65, 65, 65, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 39, 58, 130, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 22, 83, 103, 123, 175, 255, 208, 127, 105, 84, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'j' / 0x6a
        'j' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 166, 214, 52, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 16, 255, 255, 117, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 107, 149, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 65, 65, 65, 65, 65, 22, 0, 0, 0],
            &[0, 0, 0, 0, 58, 255, 255, 255, 255, 255, 86, 0, 0, 0],
            &[0, 0, 0, 0, 3, 25, 44, 63, 176, 255, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 142, 255, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 172, 255, 78, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 230, 255, 34, 0, 0, 0],
            &[0, 0, 0, 0, 6, 27, 28, 3, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'k' / 0x6b
        'k' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 195, 245, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 0, 46, 65, 36, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 74, 251, 223, 26, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 55, 245, 233, 37, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 39, 236, 241, 49, 0, 0, 0, 0],
            &[0, 0, 203, 255, 19, 26, 224, 247, 62, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 22, 208, 255, 109, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 250, 175, 240, 251, 213, 10, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 236, 50, 123, 255, 158, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 56, 0, 2, 190, 255, 93, 0, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 27, 236, 244, 39, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 78, 255, 209, 8, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 0, 147, 255, 152, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'l' / 0x6c
        'l' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 47, 245, 245, 245, 245, 245, 168, 0, 0, 0, 0, 0, 0],
            &[0, 17, 99, 117, 134, 171, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 22, 82, 102, 121, 163, 255, 220, 129, 107, 85, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'm' / 0x6d
        'm' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 62, 24, 22, 105, 86, 3, 4, 84, 107, 30, 0, 0, 0],
            &[
                0, 243, 148, 226, 255, 255, 138, 166, 255, 255, 234, 17, 0, 0,
            ],
            &[0, 243, 255, 150, 77, 242, 250, 239, 87, 166, 255, 94, 0, 0],
            &[0, 243, 252, 21, 0, 166, 255, 143, 0, 53, 255, 131, 0, 0],
            &[0, 243, 216, 0, 0, 141, 255, 73, 0, 33, 255, 143, 0, 0],
            &[0, 243, 200, 0, 0, 132, 255, 49, 0, 27, 255, 146, 0, 0],
            &[0, 243, 187, 0, 0, 132, 255, 41, 0, 27, 255, 146, 0, 0],
            &[0, 243, 186, 0, 0, 132, 255, 41, 0, 27, 255, 146, 0, 0],
            &[0, 243, 186, 0, 0, 132, 255, 41, 0, 27, 255, 146, 0, 0],
            &[0, 243, 186, 0, 0, 132, 255, 41, 0, 27, 255, 146, 0, 0],
            &[0, 243, 186, 0, 0, 132, 255, 41, 0, 27, 255, 146, 0, 0],
            &[0, 243, 186, 0, 0, 132, 255, 41, 0, 27, 255, 146, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'n' / 0x6e
        'n' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 65, 13, 0, 41, 94, 106, 66, 2, 0, 0, 0, 0],
            &[0, 81, 255, 82, 157, 254, 255, 255, 255, 202, 25, 0, 0, 0],
            &[0, 81, 255, 224, 223, 78, 30, 50, 186, 255, 166, 0, 0, 0],
            &[0, 81, 255, 254, 38, 0, 0, 0, 23, 254, 238, 0, 0, 0],
            &[0, 81, 255, 198, 0, 0, 0, 0, 0, 231, 255, 4, 0, 0],
            &[0, 81, 255, 164, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 146, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'o' / 0x6f
        'o' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 96, 110, 78, 9, 0, 0, 0, 0, 0],
            &[0, 0, 18, 178, 255, 255, 255, 255, 240, 94, 0, 0, 0, 0],
            &[0, 2, 193, 255, 166, 57, 27, 80, 222, 255, 84, 0, 0, 0],
            &[0, 81, 255, 183, 0, 0, 0, 0, 37, 250, 227, 2, 0, 0],
            &[0, 163, 255, 73, 0, 0, 0, 0, 0, 181, 255, 57, 0, 0],
            &[0, 196, 255, 30, 0, 0, 0, 0, 0, 135, 255, 92, 0, 0],
            &[0, 210, 255, 11, 0, 0, 0, 0, 0, 124, 255, 108, 0, 0],
            &[0, 193, 255, 32, 0, 0, 0, 0, 0, 144, 255, 92, 0, 0],
            &[0, 142, 255, 94, 0, 0, 0, 0, 0, 205, 255, 42, 0, 0],
            &[0, 49, 255, 215, 14, 0, 0, 0, 81, 255, 205, 0, 0, 0],
            &[0, 0, 140, 255, 216, 113, 92, 145, 248, 247, 53, 0, 0, 0],
            &[0, 0, 0, 109, 229, 255, 255, 254, 196, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 34, 47, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'p' / 0x70
        'p' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 65, 16, 0, 57, 104, 104, 45, 0, 0, 0, 0, 0],
            &[0, 81, 255, 95, 180, 255, 255, 255, 255, 162, 5, 0, 0, 0],
            &[0, 81, 255, 240, 208, 66, 29, 66, 206, 255, 135, 0, 0, 0],
            &[0, 81, 255, 248, 23, 0, 0, 0, 18, 247, 243, 11, 0, 0],
            &[0, 81, 255, 182, 0, 0, 0, 0, 0, 175, 255, 67, 0, 0],
            &[0, 81, 255, 152, 0, 0, 0, 0, 0, 143, 255, 101, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 131, 255, 108, 0, 0],
            &[0, 81, 255, 166, 0, 0, 0, 0, 0, 152, 255, 90, 0, 0],
            &[0, 81, 255, 214, 0, 0, 0, 0, 0, 199, 255, 49, 0, 0],
            &[0, 81, 255, 255, 68, 0, 0, 0, 57, 255, 224, 1, 0, 0],
            &[0, 81, 255, 219, 244, 136, 95, 134, 243, 255, 84, 0, 0, 0],
            &[0, 81, 255, 132, 109, 238, 255, 255, 228, 91, 0, 0, 0, 0],
            &[0, 81, 255, 140, 0, 3, 36, 34, 3, 0, 0, 0, 0, 0],
            &[0, 81, 255, 140, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 140, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 10, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'q' / 0x71
        'q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 68, 111, 98, 27, 0, 40, 60, 0, 0],
            &[0, 0, 0, 32, 208, 255, 255, 255, 249, 95, 186, 235, 0, 0],
            &[0, 0, 6, 213, 255, 146, 46, 40, 117, 248, 245, 235, 0, 0],
            &[0, 0, 92, 255, 175, 0, 0, 0, 0, 133, 255, 235, 0, 0],
            &[0, 0, 165, 255, 80, 0, 0, 0, 0, 42, 255, 235, 0, 0],
            &[0, 0, 202, 255, 44, 0, 0, 0, 0, 8, 255, 235, 0, 0],
            &[0, 0, 212, 255, 28, 0, 0, 0, 0, 0, 245, 235, 0, 0],
            &[0, 0, 197, 255, 51, 0, 0, 0, 0, 7, 254, 235, 0, 0],
            &[0, 0, 152, 255, 104, 0, 0, 0, 0, 45, 255, 235, 0, 0],
            &[0, 0, 69, 255, 213, 13, 0, 0, 0, 161, 255, 235, 0, 0],
            &[0, 0, 0, 180, 255, 208, 115, 102, 179, 234, 242, 235, 0, 0],
            &[0, 0, 0, 10, 155, 249, 255, 255, 210, 47, 237, 235, 0, 0],
            &[0, 0, 0, 0, 0, 12, 45, 32, 0, 0, 242, 235, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 242, 235, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 242, 235, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 9, 9, 0, 0],
        ],
        // letter: 'r' / 0x72
        'r' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 23, 65, 65, 65, 65, 16, 0, 53, 109, 99, 44, 0, 0],
            &[0, 92, 255, 255, 255, 255, 84, 126, 255, 255, 255, 219, 0, 0],
            &[0, 5, 31, 53, 123, 255, 177, 253, 141, 61, 79, 88, 0, 0],
            &[0, 0, 0, 0, 70, 255, 255, 127, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 241, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 184, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 164, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 162, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 162, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 162, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 94, 118, 170, 255, 214, 125, 101, 70, 0, 0, 0, 0],
            &[0, 178, 255, 255, 255, 255, 255, 255, 255, 233, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 's' / 0x73
        's' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 93, 113, 100, 68, 7, 0, 0, 0, 0],
            &[0, 0, 5, 161, 254, 255, 255, 255, 255, 241, 19, 0, 0, 0],
            &[0, 0, 99, 255, 189, 62, 33, 45, 91, 129, 0, 0, 0, 0],
            &[0, 0, 146, 255, 75, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 105, 255, 192, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 167, 255, 247, 153, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 64, 187, 255, 253, 175, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 51, 175, 255, 234, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 168, 255, 87, 0, 0, 0],
            &[0, 0, 20, 0, 0, 0, 0, 0, 160, 255, 81, 0, 0, 0],
            &[0, 0, 202, 193, 124, 95, 100, 154, 253, 234, 13, 0, 0, 0],
            &[0, 0, 126, 230, 255, 255, 255, 252, 189, 44, 0, 0, 0, 0],
            &[0, 0, 0, 1, 20, 44, 42, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 't' / 0x74
        't' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 66, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 155, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 32, 60, 205, 255, 90, 65, 65, 65, 34, 0, 0],
            &[0, 0, 210, 255, 255, 255, 255, 255, 255, 255, 255, 133, 0, 0],
            &[0, 0, 67, 81, 81, 213, 255, 104, 81, 81, 81, 42, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 255, 47, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 106, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 255, 248, 136, 92, 104, 119, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 235, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 34, 48, 30, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'u' / 0x75
        'u' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 65, 30, 0, 0, 0, 0, 0, 63, 60, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 249, 235, 0, 0],
            &[0, 0, 108, 255, 119, 0, 0, 0, 0, 7, 255, 235, 0, 0],
            &[0, 0, 98, 255, 135, 0, 0, 0, 0, 52, 255, 235, 0, 0],
            &[0, 0, 64, 255, 201, 0, 0, 0, 0, 163, 255, 235, 0, 0],
            &[0, 0, 6, 219, 255, 179, 99, 99, 177, 228, 232, 235, 0, 0],
            &[0, 0, 0, 46, 203, 255, 255, 255, 198, 42, 174, 235, 0, 0],
            &[0, 0, 0, 0, 0, 17, 45, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'v' / 0x76
        'v' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 64, 60, 0, 0, 0, 0, 0, 0, 25, 65, 38, 0],
            &[0, 0, 198, 255, 37, 0, 0, 0, 0, 0, 153, 255, 90, 0],
            &[0, 0, 103, 255, 126, 0, 0, 0, 0, 4, 236, 241, 9, 0],
            &[0, 0, 17, 248, 216, 0, 0, 0, 0, 72, 255, 154, 0, 0],
            &[0, 0, 0, 170, 255, 51, 0, 0, 0, 160, 255, 59, 0, 0],
            &[0, 0, 0, 76, 255, 141, 0, 0, 7, 240, 219, 0, 0, 0],
            &[0, 0, 0, 4, 233, 229, 2, 0, 80, 255, 124, 0, 0, 0],
            &[0, 0, 0, 0, 143, 255, 65, 0, 168, 254, 30, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 155, 12, 245, 189, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 209, 237, 98, 255, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 115, 255, 234, 243, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 252, 255, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'w' / 0x77
        'w' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[42, 65, 7, 0, 0, 0, 0, 0, 0, 0, 29, 65, 17, 0],
            &[138, 255, 54, 0, 0, 0, 0, 0, 0, 0, 142, 255, 41, 0],
            &[97, 255, 91, 0, 0, 26, 34, 12, 0, 0, 183, 250, 4, 0],
            &[56, 255, 129, 0, 2, 236, 255, 129, 0, 0, 223, 213, 0, 0],
            &[15, 255, 167, 0, 56, 255, 241, 199, 0, 10, 254, 171, 0, 0],
            &[0, 230, 204, 0, 129, 253, 136, 252, 17, 48, 255, 129, 0, 0],
            &[0, 189, 241, 0, 202, 204, 60, 255, 85, 87, 255, 87, 0, 0],
            &[0, 148, 255, 34, 253, 134, 5, 245, 153, 121, 255, 44, 0, 0],
            &[0, 107, 255, 113, 255, 66, 0, 185, 215, 146, 252, 6, 0, 0],
            &[0, 66, 255, 181, 247, 8, 0, 120, 254, 182, 216, 0, 0, 0],
            &[0, 25, 255, 243, 188, 0, 0, 54, 255, 243, 174, 0, 0, 0],
            &[0, 0, 239, 255, 122, 0, 0, 3, 241, 255, 132, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'x' / 0x78
        'x' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 65, 24, 0, 0, 0, 0, 0, 50, 65, 22, 0],
            &[0, 0, 74, 254, 203, 5, 0, 0, 0, 59, 252, 215, 11, 0],
            &[0, 0, 0, 141, 255, 136, 0, 0, 15, 223, 247, 45, 0, 0],
            &[0, 0, 0, 5, 202, 253, 66, 0, 166, 255, 103, 0, 0, 0],
            &[0, 0, 0, 0, 34, 241, 228, 112, 255, 169, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 255, 221, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 241, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 183, 255, 214, 255, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 118, 255, 152, 23, 233, 237, 29, 0, 0, 0],
            &[0, 0, 0, 56, 251, 213, 9, 0, 73, 254, 196, 4, 0, 0],
            &[0, 0, 17, 224, 248, 46, 0, 0, 0, 144, 255, 133, 0, 0],
            &[0, 0, 174, 255, 109, 0, 0, 0, 0, 7, 209, 253, 69, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'y' / 0x79
        'y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 64, 55, 0, 0, 0, 0, 0, 0, 22, 65, 38, 0],
            &[0, 0, 189, 252, 28, 0, 0, 0, 0, 0, 140, 255, 90, 0],
            &[0, 0, 83, 255, 129, 0, 0, 0, 0, 1, 225, 241, 9, 0],
            &[0, 0, 4, 228, 229, 4, 0, 0, 0, 56, 255, 156, 0, 0],
            &[0, 0, 0, 126, 255, 81, 0, 0, 0, 141, 255, 61, 0, 0],
            &[0, 0, 0, 24, 250, 185, 0, 0, 1, 226, 221, 0, 0, 0],
            &[0, 0, 0, 0, 169, 254, 35, 0, 57, 255, 127, 0, 0, 0],
            &[0, 0, 0, 0, 62, 255, 135, 0, 141, 254, 33, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 228, 2, 223, 193, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 105, 255, 108, 255, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 242, 241, 245, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 148, 255, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 125, 255, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 224, 224, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 113, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 30, 19, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'z' / 0x7a
        'z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 65, 65, 65, 65, 65, 65, 65, 35, 0, 0],
            &[0, 0, 0, 198, 255, 255, 255, 255, 255, 255, 255, 138, 0, 0],
            &[0, 0, 0, 38, 49, 49, 49, 49, 51, 213, 255, 80, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 140, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 74, 254, 191, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 238, 228, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 205, 249, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 156, 255, 105, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 255, 161, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 247, 207, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 222, 255, 183, 152, 152, 152, 152, 152, 108, 0, 0],
            &[0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '{' / 0x7b
        '{' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 151, 224, 163, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 224, 255, 200, 97, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 96, 255, 166, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 125, 255, 92, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 127, 255, 84, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 127, 255, 84, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 135, 255, 79, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 207, 253, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 173, 233, 235, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 248, 255, 171, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 60, 236, 240, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 145, 255, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 127, 255, 84, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 127, 255, 84, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 127, 255, 84, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 255, 117, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 252, 239, 107, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 86, 233, 255, 168, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 3, 50, 48, 0, 0, 0],
        ],
        // letter: '|' / 0x7c
        '|' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 245, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 10, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '}' / 0x7d
        '}' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 241, 202, 108, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 151, 228, 255, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 237, 246, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 191, 255, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 255, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 255, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 182, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 139, 255, 109, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 185, 253, 195, 106, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 90, 213, 255, 164, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 97, 255, 177, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 174, 255, 43, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 255, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 255, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 187, 255, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 229, 254, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 76, 174, 255, 187, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 255, 255, 187, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 69, 29, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '~' / 0x7e
        '~' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 152, 161, 68, 0, 0, 0, 60, 169, 0, 0, 0],
            &[0, 0, 199, 205, 172, 251, 133, 1, 0, 137, 198, 0, 0, 0],
            &[0, 44, 253, 24, 0, 74, 244, 181, 90, 229, 109, 0, 0, 0],
            &[0, 92, 214, 0, 0, 0, 47, 190, 238, 162, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ' ' / 0xa0
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¡' / 0xa1
        '¡' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 54, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 220, 255, 117, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 239, 255, 136, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 56, 112, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 118, 221, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 143, 255, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 151, 255, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 255, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 167, 255, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 175, 255, 67, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 183, 255, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 191, 255, 83, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 199, 255, 91, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 207, 255, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 190, 226, 95, 0, 0, 0, 0, 0],
        ],
        // letter: '¢' / 0xa2
        '¢' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 42, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 205, 159, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 66, 226, 207, 107, 67, 6, 0, 0],
            &[0, 0, 0, 0, 53, 217, 255, 255, 255, 255, 255, 140, 0, 0],
            &[0, 0, 0, 29, 241, 252, 130, 43, 26, 51, 108, 42, 0, 0],
            &[0, 0, 0, 154, 255, 115, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 231, 250, 12, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 255, 216, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 255, 204, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 255, 223, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 219, 255, 30, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 132, 255, 174, 4, 0, 0, 0, 0, 2, 0, 0],
            &[0, 0, 0, 12, 213, 255, 214, 136, 121, 145, 194, 130, 0, 0],
            &[0, 0, 0, 0, 17, 147, 238, 255, 255, 255, 224, 83, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 210, 169, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 205, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 43, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '£' / 0xa3
        '£' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 34, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 131, 246, 255, 255, 228, 102, 0, 0, 0],
            &[0, 0, 0, 0, 127, 255, 205, 115, 123, 200, 118, 0, 0, 0],
            &[0, 0, 0, 0, 241, 238, 13, 0, 0, 0, 3, 0, 0, 0],
            &[0, 0, 0, 27, 255, 190, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 176, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 176, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 27, 63, 255, 185, 27, 27, 27, 11, 0, 0, 0, 0],
            &[0, 12, 255, 255, 255, 255, 255, 255, 255, 104, 0, 0, 0, 0],
            &[0, 3, 71, 100, 255, 198, 71, 71, 71, 29, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 176, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 45, 255, 170, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 83, 255, 127, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 193, 242, 27, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 215, 255, 217, 174, 174, 174, 174, 174, 174, 4, 0, 0],
            &[0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¤' / 0xa4
        '¤' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 0, 0, 0, 0, 0, 0, 9, 26, 0, 0, 0],
            &[0, 67, 252, 108, 75, 161, 183, 138, 43, 193, 215, 4, 0, 0],
            &[0, 4, 173, 255, 254, 196, 168, 223, 255, 251, 82, 0, 0, 0],
            &[0, 0, 85, 253, 81, 0, 0, 4, 161, 237, 12, 0, 0, 0],
            &[0, 0, 183, 182, 0, 0, 0, 0, 19, 253, 90, 0, 0, 0],
            &[0, 0, 214, 148, 0, 0, 0, 0, 0, 239, 122, 0, 0, 0],
            &[0, 0, 175, 194, 0, 0, 0, 0, 30, 254, 87, 0, 0, 0],
            &[0, 0, 77, 255, 122, 7, 0, 22, 191, 235, 8, 0, 0, 0],
            &[0, 11, 193, 251, 253, 248, 216, 249, 250, 255, 103, 0, 0, 0],
            &[0, 55, 238, 76, 40, 124, 146, 103, 17, 169, 199, 3, 0, 0],
            &[0, 0, 13, 0, 0, 0, 0, 0, 0, 3, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¥' / 0xa5
        '¥' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 239, 228, 8, 0, 0, 0, 0, 0, 80, 254, 153, 0],
            &[0, 0, 127, 255, 107, 0, 0, 0, 0, 0, 206, 248, 29, 0],
            &[0, 0, 14, 237, 226, 6, 0, 0, 0, 79, 255, 143, 0, 0],
            &[0, 0, 0, 121, 255, 102, 0, 0, 0, 206, 243, 22, 0, 0],
            &[0, 0, 0, 12, 234, 222, 4, 0, 78, 255, 132, 0, 0, 0],
            &[0, 0, 0, 0, 115, 255, 97, 0, 205, 238, 15, 0, 0, 0],
            &[0, 0, 0, 0, 9, 230, 219, 81, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 19, 30, 133, 255, 241, 240, 41, 30, 7, 0, 0],
            &[0, 0, 0, 165, 255, 255, 255, 255, 255, 255, 255, 63, 0, 0],
            &[0, 0, 0, 37, 57, 57, 181, 255, 93, 57, 57, 14, 0, 0],
            &[0, 0, 0, 8, 13, 13, 164, 255, 57, 13, 13, 3, 0, 0],
            &[0, 0, 0, 165, 255, 255, 255, 255, 255, 255, 255, 63, 0, 0],
            &[0, 0, 0, 48, 74, 74, 187, 255, 107, 74, 74, 18, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 255, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 255, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¦' / 0xa6
        '¦' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 117, 245, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 114, 240, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 15, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '§' / 0xa7
        '§' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 64, 189, 244, 255, 248, 217, 143, 3, 0, 0, 0],
            &[0, 0, 42, 248, 215, 105, 82, 101, 148, 168, 0, 0, 0, 0],
            &[0, 0, 109, 255, 80, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 255, 188, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 153, 255, 240, 138, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 98, 251, 213, 252, 249, 156, 17, 0, 0, 0, 0],
            &[0, 0, 65, 255, 98, 0, 28, 140, 254, 206, 3, 0, 0, 0],
            &[0, 0, 128, 253, 5, 0, 0, 0, 127, 255, 57, 0, 0, 0],
            &[0, 0, 104, 255, 109, 0, 0, 0, 78, 255, 55, 0, 0, 0],
            &[0, 0, 12, 207, 255, 176, 61, 23, 203, 217, 4, 0, 0, 0],
            &[0, 0, 0, 9, 119, 230, 255, 248, 236, 36, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 100, 217, 255, 156, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 196, 255, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 135, 255, 49, 0, 0, 0],
            &[0, 0, 161, 145, 77, 44, 49, 106, 242, 219, 3, 0, 0, 0],
            &[0, 0, 116, 234, 255, 255, 255, 251, 171, 31, 0, 0, 0, 0],
            &[0, 0, 0, 2, 24, 46, 34, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¨' / 0xa8
        '¨' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 65, 0, 0, 7, 101, 44, 0, 0, 0],
            &[0, 0, 0, 31, 255, 241, 2, 0, 80, 255, 194, 0, 0, 0],
            &[0, 0, 0, 5, 195, 159, 0, 0, 32, 211, 117, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '©' / 0xa9
        '©' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 57, 163, 215, 231, 206, 133, 24, 0, 0, 0],
            &[0, 0, 0, 119, 233, 114, 46, 32, 66, 166, 232, 50, 0, 0],
            &[0, 0, 71, 232, 30, 35, 175, 219, 192, 39, 115, 222, 8, 0],
            &[0, 0, 188, 103, 7, 225, 151, 55, 92, 8, 0, 210, 89, 0],
            &[0, 0, 247, 30, 77, 233, 4, 0, 0, 0, 0, 135, 147, 0],
            &[0, 12, 255, 11, 106, 203, 0, 0, 0, 0, 0, 113, 165, 0],
            &[0, 0, 246, 37, 79, 233, 3, 0, 0, 0, 0, 137, 142, 0],
            &[0, 0, 181, 121, 8, 225, 155, 68, 96, 14, 2, 216, 74, 0],
            &[0, 0, 63, 241, 48, 29, 155, 195, 169, 13, 132, 203, 3, 0],
            &[0, 0, 0, 104, 240, 138, 67, 51, 86, 184, 210, 31, 0, 0],
            &[0, 0, 0, 0, 40, 139, 186, 200, 173, 98, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ª' / 0xaa
        'ª' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 32, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 222, 254, 250, 238, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 78, 9, 9, 147, 222, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 55, 81, 143, 251, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 235, 204, 163, 183, 252, 0, 0, 0, 0],
            &[0, 0, 0, 0, 188, 186, 0, 0, 100, 252, 0, 0, 0, 0],
            &[0, 0, 0, 0, 180, 208, 48, 65, 218, 252, 0, 0, 0, 0],
            &[0, 0, 0, 0, 45, 201, 231, 180, 67, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '«' / 0xab
        '«' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 27, 195, 49, 0, 14, 192, 69, 0, 0],
            &[0, 0, 0, 0, 8, 202, 247, 52, 2, 181, 251, 65, 0, 0],
            &[0, 0, 0, 0, 163, 255, 93, 0, 139, 255, 112, 0, 0, 0],
            &[0, 0, 0, 116, 255, 141, 0, 95, 255, 163, 0, 0, 0, 0],
            &[0, 0, 14, 246, 240, 10, 4, 237, 250, 24, 0, 0, 0, 0],
            &[0, 0, 0, 100, 255, 154, 0, 81, 254, 178, 1, 0, 0, 0],
            &[0, 0, 0, 0, 145, 255, 100, 0, 123, 255, 128, 0, 0, 0],
            &[0, 0, 0, 0, 3, 186, 248, 54, 0, 166, 254, 77, 0, 0],
            &[0, 0, 0, 0, 0, 17, 170, 40, 0, 8, 170, 56, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¬' / 0xac
        '¬' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 37, 37, 37, 37, 37, 37, 37, 37, 30, 0, 0],
            &[0, 0, 108, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0],
            &[0, 0, 37, 87, 87, 87, 87, 87, 87, 87, 201, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 172, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 172, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 172, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 53, 65, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '®' / 0xae
        '®' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 57, 163, 215, 231, 206, 133, 24, 0, 0, 0],
            &[0, 0, 0, 119, 233, 114, 46, 32, 66, 166, 232, 50, 0, 0],
            &[0, 0, 71, 232, 30, 126, 213, 204, 146, 9, 115, 222, 8, 0],
            &[0, 0, 188, 103, 0, 150, 140, 54, 209, 113, 0, 210, 89, 0],
            &[0, 0, 247, 30, 0, 150, 123, 0, 173, 128, 0, 135, 147, 0],
            &[0, 12, 255, 11, 0, 150, 222, 215, 207, 24, 0, 113, 165, 0],
            &[0, 0, 246, 37, 0, 150, 146, 123, 198, 1, 0, 137, 142, 0],
            &[0, 0, 181, 121, 0, 150, 123, 3, 211, 102, 2, 216, 74, 0],
            &[0, 0, 63, 241, 48, 98, 80, 0, 59, 150, 135, 203, 3, 0],
            &[0, 0, 0, 104, 240, 138, 67, 51, 86, 184, 210, 31, 0, 0],
            &[0, 0, 0, 0, 40, 139, 186, 200, 173, 98, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¯' / 0xaf
        '¯' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 120, 120, 120, 120, 120, 51, 0, 0, 0, 0],
            &[0, 0, 0, 206, 255, 255, 255, 255, 255, 109, 0, 0, 0, 0],
            &[0, 0, 0, 4, 5, 5, 5, 5, 5, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '°' / 0xb0
        '°' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 43, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 158, 253, 255, 236, 79, 0, 0, 0, 0, 0],
            &[0, 0, 0, 124, 234, 68, 26, 129, 247, 30, 0, 0, 0, 0],
            &[0, 0, 0, 206, 127, 0, 0, 0, 224, 104, 0, 0, 0, 0],
            &[0, 0, 0, 206, 131, 0, 0, 0, 225, 104, 0, 0, 0, 0],
            &[0, 0, 0, 124, 240, 80, 32, 134, 247, 30, 0, 0, 0, 0],
            &[0, 0, 0, 5, 153, 251, 255, 231, 78, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 32, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '±' / 0xb1
        '±' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 44, 89, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 127, 255, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 127, 255, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 127, 255, 3, 0, 0, 0, 0, 0],
            &[0, 0, 21, 53, 53, 53, 154, 255, 56, 53, 53, 49, 0, 0],
            &[0, 0, 103, 255, 255, 255, 255, 255, 255, 255, 255, 235, 0, 0],
            &[0, 0, 28, 71, 71, 71, 163, 255, 74, 71, 71, 65, 0, 0],
            &[0, 0, 0, 0, 0, 0, 127, 255, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 127, 255, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 127, 255, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 50, 101, 1, 0, 0, 0, 0, 0],
            &[0, 0, 40, 125, 125, 125, 125, 125, 125, 125, 125, 115, 0, 0],
            &[0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 235, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '²' / 0xb2
        '²' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 117, 208, 224, 181, 50, 0, 0, 0, 0, 0],
            &[0, 0, 0, 95, 229, 122, 99, 201, 234, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 0, 0, 61, 255, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 105, 251, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 40, 238, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 230, 158, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 231, 155, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 234, 153, 10, 9, 9, 5, 0, 0, 0, 0],
            &[0, 0, 0, 188, 255, 255, 255, 255, 255, 142, 0, 0, 0, 0],
            &[0, 0, 0, 50, 67, 67, 67, 67, 67, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '³' / 0xb3
        '³' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 140, 215, 222, 175, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 98, 212, 109, 100, 220, 214, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 0, 0, 124, 255, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 203, 201, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 248, 255, 236, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 49, 75, 195, 232, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 57, 255, 77, 0, 0, 0, 0],
            &[0, 0, 0, 64, 12, 0, 0, 118, 255, 46, 0, 0, 0, 0],
            &[0, 0, 0, 181, 248, 211, 217, 254, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 66, 100, 89, 41, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '´' / 0xb4
        '´' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 22, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 253, 240, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 197, 248, 67, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 245, 74, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 58, 37, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'µ' / 0xb5
        'µ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 65, 30, 0, 0, 0, 0, 0, 63, 60, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 249, 235, 0, 0],
            &[0, 0, 192, 255, 210, 0, 0, 0, 0, 7, 255, 235, 0, 0],
            &[0, 0, 207, 255, 254, 0, 0, 0, 0, 52, 255, 235, 0, 0],
            &[0, 0, 172, 255, 255, 0, 0, 0, 0, 163, 255, 235, 0, 0],
            &[0, 0, 115, 255, 255, 179, 99, 99, 177, 228, 232, 235, 0, 0],
            &[0, 0, 108, 255, 255, 255, 255, 255, 198, 42, 174, 235, 0, 0],
            &[0, 0, 108, 255, 118, 17, 45, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 10, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¶' / 0xb6
        '¶' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 156, 222, 244, 245, 245, 245, 245, 245, 78, 0, 0],
            &[0, 17, 228, 255, 255, 255, 255, 247, 55, 55, 235, 81, 0, 0],
            &[0, 129, 255, 255, 255, 255, 255, 245, 0, 0, 230, 81, 0, 0],
            &[0, 200, 255, 255, 255, 255, 255, 245, 0, 0, 230, 81, 0, 0],
            &[0, 227, 255, 255, 255, 255, 255, 245, 0, 0, 230, 81, 0, 0],
            &[0, 237, 255, 255, 255, 255, 255, 245, 0, 0, 230, 81, 0, 0],
            &[0, 221, 255, 255, 255, 255, 255, 245, 0, 0, 230, 81, 0, 0],
            &[0, 177, 255, 255, 255, 255, 255, 245, 0, 0, 230, 81, 0, 0],
            &[0, 88, 255, 255, 255, 255, 255, 245, 0, 0, 230, 81, 0, 0],
            &[0, 0, 155, 255, 255, 255, 255, 245, 0, 0, 230, 81, 0, 0],
            &[0, 0, 0, 46, 101, 109, 128, 245, 0, 0, 230, 81, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 245, 0, 0, 230, 81, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 245, 0, 0, 230, 81, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 245, 0, 0, 230, 81, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 245, 0, 0, 230, 81, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 245, 0, 0, 230, 81, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 245, 0, 0, 230, 81, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 245, 0, 0, 230, 81, 0, 0],
            &[0, 0, 0, 0, 0, 0, 44, 163, 0, 0, 153, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '·' / 0xb7
        '·' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 62, 120, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 240, 255, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 216, 255, 114, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¸' / 0xb8
        '¸' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 26, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 114, 200, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 213, 228, 109, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 108, 255, 61, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 4, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¹' / 0xb9
        '¹' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 125, 185, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 86, 226, 240, 255, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 130, 171, 41, 255, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 255, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 255, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 255, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 255, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 255, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 45, 218, 250, 255, 255, 223, 117, 0, 0, 0, 0],
            &[0, 0, 0, 14, 67, 71, 135, 88, 67, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'º' / 0xba
        'º' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 31, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 145, 252, 253, 230, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 94, 242, 50, 9, 141, 237, 11, 0, 0, 0, 0],
            &[0, 0, 0, 170, 166, 0, 0, 23, 255, 70, 0, 0, 0, 0],
            &[0, 0, 0, 192, 143, 0, 0, 2, 254, 85, 0, 0, 0, 0],
            &[0, 0, 0, 162, 175, 0, 0, 33, 255, 54, 0, 0, 0, 0],
            &[0, 0, 0, 66, 249, 98, 54, 177, 211, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 209, 230, 175, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '»' / 0xbb
        '»' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 150, 136, 0, 0, 124, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 150, 255, 94, 0, 121, 255, 116, 0, 0, 0, 0],
            &[0, 0, 0, 5, 197, 248, 57, 1, 174, 253, 74, 0, 0, 0],
            &[0, 0, 0, 0, 25, 230, 232, 29, 13, 215, 241, 41, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 144, 0, 92, 255, 163, 0, 0],
            &[0, 0, 0, 0, 40, 242, 215, 15, 28, 232, 228, 24, 0, 0],
            &[0, 0, 0, 14, 216, 238, 37, 8, 203, 247, 52, 0, 0, 0],
            &[0, 0, 1, 178, 251, 68, 0, 163, 255, 91, 0, 0, 0, 0],
            &[0, 0, 1, 114, 106, 0, 0, 88, 128, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¼' / 0xbc
        '¼' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[17, 144, 249, 166, 0, 0, 0, 0, 0, 189, 213, 2, 0, 0],
            &[176, 237, 219, 167, 0, 0, 0, 0, 66, 255, 86, 0, 0, 0],
            &[32, 21, 179, 167, 0, 0, 0, 0, 199, 208, 1, 0, 0, 0],
            &[0, 0, 181, 167, 0, 0, 0, 76, 255, 77, 0, 0, 0, 0],
            &[0, 0, 181, 167, 0, 0, 1, 207, 200, 0, 0, 0, 0, 0],
            &[0, 0, 181, 167, 0, 0, 85, 255, 69, 0, 0, 0, 0, 0],
            &[66, 217, 255, 255, 216, 75, 215, 192, 0, 0, 0, 0, 0, 0],
            &[24, 74, 128, 124, 74, 121, 255, 60, 0, 47, 80, 0, 0, 0],
            &[0, 0, 0, 0, 5, 222, 184, 0, 26, 237, 246, 0, 0, 0],
            &[0, 0, 0, 0, 104, 255, 52, 1, 187, 223, 246, 0, 0, 0],
            &[0, 0, 0, 8, 228, 175, 0, 114, 223, 109, 246, 0, 0, 0],
            &[0, 0, 0, 113, 253, 45, 46, 246, 63, 97, 246, 0, 0, 0],
            &[0, 0, 12, 234, 166, 0, 172, 247, 208, 226, 254, 208, 11, 0],
            &[0, 0, 122, 252, 38, 0, 64, 92, 92, 155, 249, 92, 5, 0],
            &[0, 16, 239, 158, 0, 0, 0, 0, 0, 97, 246, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '½' / 0xbd
        '½' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[17, 144, 249, 166, 0, 0, 0, 0, 0, 189, 213, 2, 0, 0],
            &[176, 237, 219, 167, 0, 0, 0, 0, 66, 255, 86, 0, 0, 0],
            &[32, 21, 179, 167, 0, 0, 0, 0, 199, 208, 1, 0, 0, 0],
            &[0, 0, 181, 167, 0, 0, 0, 76, 255, 77, 0, 0, 0, 0],
            &[0, 0, 181, 167, 0, 0, 1, 207, 200, 0, 0, 0, 0, 0],
            &[0, 0, 181, 167, 0, 0, 85, 255, 69, 0, 0, 0, 0, 0],
            &[66, 217, 255, 255, 216, 75, 215, 192, 0, 0, 0, 0, 0, 0],
            &[24, 74, 128, 124, 74, 121, 255, 63, 67, 105, 60, 0, 0, 0],
            &[0, 0, 0, 0, 5, 222, 184, 172, 244, 216, 255, 122, 0, 0],
            &[0, 0, 0, 0, 104, 255, 52, 45, 19, 0, 171, 212, 0, 0],
            &[0, 0, 0, 8, 228, 175, 0, 0, 0, 1, 196, 177, 0, 0],
            &[0, 0, 0, 113, 253, 45, 0, 0, 0, 144, 238, 38, 0, 0],
            &[0, 0, 12, 234, 166, 0, 0, 2, 151, 233, 48, 0, 0, 0],
            &[0, 0, 122, 252, 38, 0, 1, 164, 250, 113, 77, 77, 4, 0],
            &[0, 16, 239, 158, 0, 0, 12, 255, 255, 255, 255, 255, 13, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¾' / 0xbe
        '¾' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 31, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[66, 231, 255, 255, 172, 3, 0, 0, 0, 189, 213, 2, 0, 0],
            &[18, 105, 21, 116, 255, 61, 0, 0, 66, 255, 86, 0, 0, 0],
            &[0, 5, 20, 140, 253, 30, 0, 0, 199, 208, 1, 0, 0, 0],
            &[0, 157, 255, 255, 153, 0, 0, 76, 255, 77, 0, 0, 0, 0],
            &[0, 19, 40, 123, 255, 81, 1, 207, 200, 0, 0, 0, 0, 0],
            &[24, 6, 0, 49, 255, 108, 85, 255, 69, 0, 0, 0, 0, 0],
            &[110, 241, 211, 251, 223, 24, 215, 192, 0, 0, 0, 0, 0, 0],
            &[10, 76, 100, 75, 8, 94, 255, 60, 0, 47, 80, 0, 0, 0],
            &[0, 0, 0, 0, 5, 222, 184, 0, 26, 237, 246, 0, 0, 0],
            &[0, 0, 0, 0, 104, 255, 52, 1, 187, 223, 246, 0, 0, 0],
            &[0, 0, 0, 8, 228, 175, 0, 114, 223, 109, 246, 0, 0, 0],
            &[0, 0, 0, 113, 253, 45, 46, 246, 63, 97, 246, 0, 0, 0],
            &[0, 0, 12, 234, 166, 0, 172, 247, 208, 226, 254, 208, 11, 0],
            &[0, 0, 122, 252, 38, 0, 64, 92, 92, 155, 249, 92, 5, 0],
            &[0, 16, 239, 158, 0, 0, 0, 0, 0, 97, 246, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¿' / 0xbf
        '¿' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 56, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 255, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 222, 255, 152, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 49, 113, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 89, 221, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 103, 254, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 139, 255, 199, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 150, 255, 167, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 55, 254, 183, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 76, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 255, 57, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 255, 153, 0, 0, 0, 0, 23, 61, 0, 0, 0],
            &[0, 0, 14, 218, 255, 207, 150, 149, 190, 251, 234, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 14, 15, 1, 0, 0, 0, 0, 0],
        ],
        // letter: 'À' / 0xc0
        'À' => &[
            &[0, 0, 0, 58, 210, 196, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 152, 251, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 105, 97, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 14, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 226, 255, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 255, 255, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 137, 255, 186, 255, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 219, 228, 78, 255, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 151, 11, 247, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 71, 0, 179, 253, 25, 0, 0, 0, 0],
            &[0, 0, 0, 212, 241, 6, 0, 100, 255, 106, 0, 0, 0, 0],
            &[0, 0, 39, 255, 168, 0, 0, 23, 253, 190, 0, 0, 0, 0],
            &[0, 0, 122, 255, 160, 71, 71, 71, 245, 252, 20, 0, 0, 0],
            &[0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0],
            &[0, 32, 255, 255, 102, 102, 102, 102, 140, 255, 183, 0, 0, 0],
            &[0, 114, 255, 106, 0, 0, 0, 0, 0, 214, 250, 16, 0, 0],
            &[0, 197, 254, 28, 0, 0, 0, 0, 0, 135, 255, 94, 0, 0],
            &[26, 254, 203, 0, 0, 0, 0, 0, 0, 55, 255, 177, 0, 0],
            &[107, 255, 124, 0, 0, 0, 0, 0, 0, 1, 230, 247, 12, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Á' / 0xc1
        'Á' => &[
            &[0, 0, 0, 0, 0, 0, 0, 168, 210, 100, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 86, 255, 185, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 233, 192, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 69, 123, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 14, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 226, 255, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 255, 255, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 137, 255, 186, 255, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 219, 228, 78, 255, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 151, 11, 247, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 71, 0, 179, 253, 25, 0, 0, 0, 0],
            &[0, 0, 0, 212, 241, 6, 0, 100, 255, 106, 0, 0, 0, 0],
            &[0, 0, 39, 255, 168, 0, 0, 23, 253, 190, 0, 0, 0, 0],
            &[0, 0, 122, 255, 160, 71, 71, 71, 245, 252, 20, 0, 0, 0],
            &[0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0],
            &[0, 32, 255, 255, 102, 102, 102, 102, 140, 255, 183, 0, 0, 0],
            &[0, 114, 255, 106, 0, 0, 0, 0, 0, 214, 250, 16, 0, 0],
            &[0, 197, 254, 28, 0, 0, 0, 0, 0, 135, 255, 94, 0, 0],
            &[26, 254, 203, 0, 0, 0, 0, 0, 0, 55, 255, 177, 0, 0],
            &[107, 255, 124, 0, 0, 0, 0, 0, 0, 1, 230, 247, 12, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Â' / 0xc2
        'Â' => &[
            &[0, 0, 0, 0, 11, 190, 210, 107, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 248, 198, 248, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 130, 244, 71, 6, 174, 235, 41, 0, 0, 0, 0],
            &[0, 0, 23, 137, 46, 0, 0, 1, 106, 105, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 14, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 226, 255, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 255, 255, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 137, 255, 186, 255, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 219, 228, 78, 255, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 151, 11, 247, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 71, 0, 179, 253, 25, 0, 0, 0, 0],
            &[0, 0, 0, 212, 241, 6, 0, 100, 255, 106, 0, 0, 0, 0],
            &[0, 0, 39, 255, 168, 0, 0, 23, 253, 190, 0, 0, 0, 0],
            &[0, 0, 122, 255, 160, 71, 71, 71, 245, 252, 20, 0, 0, 0],
            &[0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0],
            &[0, 32, 255, 255, 102, 102, 102, 102, 140, 255, 183, 0, 0, 0],
            &[0, 114, 255, 106, 0, 0, 0, 0, 0, 214, 250, 16, 0, 0],
            &[0, 197, 254, 28, 0, 0, 0, 0, 0, 135, 255, 94, 0, 0],
            &[26, 254, 203, 0, 0, 0, 0, 0, 0, 55, 255, 177, 0, 0],
            &[107, 255, 124, 0, 0, 0, 0, 0, 0, 1, 230, 247, 12, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ã' / 0xc3
        'Ã' => &[
            &[0, 0, 0, 0, 19, 8, 0, 0, 0, 44, 2, 0, 0, 0],
            &[0, 0, 0, 154, 255, 246, 135, 18, 86, 237, 1, 0, 0, 0],
            &[0, 0, 58, 246, 112, 141, 247, 254, 255, 128, 0, 0, 0, 0],
            &[0, 0, 57, 97, 0, 0, 30, 113, 86, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 14, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 226, 255, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 255, 255, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 137, 255, 186, 255, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 219, 228, 78, 255, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 151, 11, 247, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 71, 0, 179, 253, 25, 0, 0, 0, 0],
            &[0, 0, 0, 212, 241, 6, 0, 100, 255, 106, 0, 0, 0, 0],
            &[0, 0, 39, 255, 168, 0, 0, 23, 253, 190, 0, 0, 0, 0],
            &[0, 0, 122, 255, 160, 71, 71, 71, 245, 252, 20, 0, 0, 0],
            &[0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0],
            &[0, 32, 255, 255, 102, 102, 102, 102, 140, 255, 183, 0, 0, 0],
            &[0, 114, 255, 106, 0, 0, 0, 0, 0, 214, 250, 16, 0, 0],
            &[0, 197, 254, 28, 0, 0, 0, 0, 0, 135, 255, 94, 0, 0],
            &[26, 254, 203, 0, 0, 0, 0, 0, 0, 55, 255, 177, 0, 0],
            &[107, 255, 124, 0, 0, 0, 0, 0, 0, 1, 230, 247, 12, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ä' / 0xc4
        'Ä' => &[
            &[0, 0, 0, 27, 14, 0, 0, 0, 35, 6, 0, 0, 0, 0],
            &[0, 0, 24, 251, 220, 0, 0, 69, 255, 172, 0, 0, 0, 0],
            &[0, 0, 19, 247, 211, 0, 0, 61, 254, 162, 0, 0, 0, 0],
            &[0, 0, 0, 18, 8, 0, 0, 0, 24, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 14, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 226, 255, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 255, 255, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 137, 255, 186, 255, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 219, 228, 78, 255, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 151, 11, 247, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 71, 0, 179, 253, 25, 0, 0, 0, 0],
            &[0, 0, 0, 212, 241, 6, 0, 100, 255, 106, 0, 0, 0, 0],
            &[0, 0, 39, 255, 168, 0, 0, 23, 253, 190, 0, 0, 0, 0],
            &[0, 0, 122, 255, 160, 71, 71, 71, 245, 252, 20, 0, 0, 0],
            &[0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0],
            &[0, 32, 255, 255, 102, 102, 102, 102, 140, 255, 183, 0, 0, 0],
            &[0, 114, 255, 106, 0, 0, 0, 0, 0, 214, 250, 16, 0, 0],
            &[0, 197, 254, 28, 0, 0, 0, 0, 0, 135, 255, 94, 0, 0],
            &[26, 254, 203, 0, 0, 0, 0, 0, 0, 55, 255, 177, 0, 0],
            &[107, 255, 124, 0, 0, 0, 0, 0, 0, 1, 230, 247, 12, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Å' / 0xc5
        'Å' => &[
            &[0, 0, 0, 0, 214, 160, 72, 229, 109, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 70, 0, 178, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 223, 170, 249, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 98, 124, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 14, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 226, 255, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 255, 255, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 137, 255, 186, 255, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 219, 228, 78, 255, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 151, 11, 247, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 71, 0, 179, 253, 25, 0, 0, 0, 0],
            &[0, 0, 0, 212, 241, 6, 0, 100, 255, 106, 0, 0, 0, 0],
            &[0, 0, 39, 255, 168, 0, 0, 23, 253, 190, 0, 0, 0, 0],
            &[0, 0, 122, 255, 160, 71, 71, 71, 245, 252, 20, 0, 0, 0],
            &[0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0],
            &[0, 32, 255, 255, 102, 102, 102, 102, 140, 255, 183, 0, 0, 0],
            &[0, 114, 255, 106, 0, 0, 0, 0, 0, 214, 250, 16, 0, 0],
            &[0, 197, 254, 28, 0, 0, 0, 0, 0, 135, 255, 94, 0, 0],
            &[26, 254, 203, 0, 0, 0, 0, 0, 0, 55, 255, 177, 0, 0],
            &[107, 255, 124, 0, 0, 0, 0, 0, 0, 1, 230, 247, 12, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 191, 216, 143, 5, 0, 0, 0, 0, 0],
        ],
        // letter: 'Æ' / 0xc6
        'Æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 254, 254, 254, 254, 254, 254, 254, 102, 0],
            &[0, 0, 0, 0, 141, 255, 194, 255, 225, 175, 175, 175, 70, 0],
            &[0, 0, 0, 0, 199, 232, 46, 255, 159, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 249, 178, 46, 255, 159, 0, 0, 0, 0, 0],
            &[0, 0, 0, 58, 255, 123, 46, 255, 159, 0, 0, 0, 0, 0],
            &[0, 0, 0, 115, 255, 69, 46, 255, 159, 0, 0, 0, 0, 0],
            &[0, 0, 0, 172, 254, 16, 46, 255, 224, 172, 172, 172, 0, 0],
            &[0, 0, 0, 230, 215, 0, 46, 255, 254, 252, 252, 252, 1, 0],
            &[0, 0, 31, 255, 216, 55, 102, 255, 159, 0, 0, 0, 0, 0],
            &[0, 0, 88, 255, 255, 255, 255, 255, 159, 0, 0, 0, 0, 0],
            &[0, 0, 146, 255, 155, 102, 149, 255, 159, 0, 0, 0, 0, 0],
            &[0, 0, 203, 248, 5, 0, 46, 255, 159, 0, 0, 0, 0, 0],
            &[0, 9, 251, 199, 0, 0, 46, 255, 159, 0, 0, 0, 0, 0],
            &[0, 62, 255, 144, 0, 0, 46, 255, 225, 174, 174, 174, 70, 0],
            &[0, 119, 255, 90, 0, 0, 46, 255, 255, 255, 255, 255, 103, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ç' / 0xc7
        'Ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 45, 41, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 208, 255, 255, 255, 251, 182, 53, 0, 0],
            &[0, 0, 0, 145, 255, 246, 169, 129, 144, 187, 251, 43, 0, 0],
            &[0, 0, 92, 255, 221, 35, 0, 0, 0, 0, 18, 0, 0, 0],
            &[0, 1, 222, 254, 52, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 195, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 113, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 255, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 152, 255, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 137, 255, 98, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 115, 255, 127, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 255, 188, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 233, 254, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 255, 217, 32, 0, 0, 0, 0, 14, 0, 0, 0],
            &[0, 0, 3, 166, 255, 250, 179, 145, 157, 197, 226, 0, 0, 0],
            &[0, 0, 0, 2, 119, 228, 255, 255, 255, 247, 168, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 142, 251, 34, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 212, 229, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 106, 255, 63, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 4, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'È' / 0xc8
        'È' => &[
            &[0, 0, 0, 0, 35, 207, 207, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 110, 255, 165, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 122, 255, 81, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 88, 115, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 254, 254, 254, 254, 254, 254, 254, 254, 207, 0, 0],
            &[0, 0, 1, 255, 246, 175, 175, 175, 175, 175, 175, 143, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 172, 172, 172, 172, 172, 172, 71, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 1, 255, 226, 1, 1, 1, 1, 1, 1, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 174, 174, 174, 174, 174, 174, 141, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'É' / 0xc9
        'É' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 141, 210, 127, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 56, 253, 208, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 214, 213, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 52, 131, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 254, 254, 254, 254, 254, 254, 254, 254, 207, 0, 0],
            &[0, 0, 1, 255, 246, 175, 175, 175, 175, 175, 175, 143, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 172, 172, 172, 172, 172, 172, 71, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 1, 255, 226, 1, 1, 1, 1, 1, 1, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 174, 174, 174, 174, 174, 174, 141, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ê' / 0xca
        'Ê' => &[
            &[0, 0, 0, 0, 0, 2, 172, 210, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 254, 192, 254, 84, 0, 0, 0, 0],
            &[0, 0, 0, 0, 98, 250, 96, 1, 147, 247, 62, 0, 0, 0],
            &[0, 0, 0, 9, 133, 63, 0, 0, 0, 91, 122, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 254, 254, 254, 254, 254, 254, 254, 254, 207, 0, 0],
            &[0, 0, 1, 255, 246, 175, 175, 175, 175, 175, 175, 143, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 172, 172, 172, 172, 172, 172, 71, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 1, 255, 226, 1, 1, 1, 1, 1, 1, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 174, 174, 174, 174, 174, 174, 141, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ë' / 0xcb
        'Ë' => &[
            &[0, 0, 0, 0, 20, 21, 0, 0, 0, 30, 11, 0, 0, 0],
            &[0, 0, 0, 6, 237, 242, 10, 0, 37, 255, 204, 0, 0, 0],
            &[0, 0, 0, 4, 230, 236, 7, 0, 32, 251, 195, 0, 0, 0],
            &[0, 0, 0, 0, 12, 14, 0, 0, 0, 21, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 254, 254, 254, 254, 254, 254, 254, 254, 207, 0, 0],
            &[0, 0, 1, 255, 246, 175, 175, 175, 175, 175, 175, 143, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 172, 172, 172, 172, 172, 172, 71, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 1, 255, 226, 1, 1, 1, 1, 1, 1, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 174, 174, 174, 174, 174, 174, 141, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ì' / 0xcc
        'Ì' => &[
            &[0, 0, 0, 58, 210, 196, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 152, 251, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 105, 97, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 175, 254, 254, 254, 254, 254, 254, 254, 73, 0, 0, 0],
            &[0, 0, 59, 106, 129, 218, 255, 165, 119, 96, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 104, 127, 218, 255, 164, 118, 95, 23, 0, 0, 0],
            &[0, 0, 176, 255, 255, 255, 255, 255, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Í' / 0xcd
        'Í' => &[
            &[0, 0, 0, 0, 0, 0, 0, 168, 210, 100, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 86, 255, 185, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 233, 192, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 69, 123, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 175, 254, 254, 254, 254, 254, 254, 254, 73, 0, 0, 0],
            &[0, 0, 59, 106, 129, 218, 255, 165, 119, 96, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 104, 127, 218, 255, 164, 118, 95, 23, 0, 0, 0],
            &[0, 0, 176, 255, 255, 255, 255, 255, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Î' / 0xce
        'Î' => &[
            &[0, 0, 0, 0, 11, 190, 210, 107, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 248, 198, 248, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 130, 244, 71, 6, 174, 235, 41, 0, 0, 0, 0],
            &[0, 0, 23, 137, 46, 0, 0, 1, 106, 105, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 175, 254, 254, 254, 254, 254, 254, 254, 73, 0, 0, 0],
            &[0, 0, 59, 106, 129, 218, 255, 165, 119, 96, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 104, 127, 218, 255, 164, 118, 95, 23, 0, 0, 0],
            &[0, 0, 176, 255, 255, 255, 255, 255, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ï' / 0xcf
        'Ï' => &[
            &[0, 0, 0, 27, 14, 0, 0, 0, 35, 6, 0, 0, 0, 0],
            &[0, 0, 24, 251, 220, 0, 0, 69, 255, 172, 0, 0, 0, 0],
            &[0, 0, 19, 247, 211, 0, 0, 61, 254, 162, 0, 0, 0, 0],
            &[0, 0, 0, 18, 8, 0, 0, 0, 24, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 175, 254, 254, 254, 254, 254, 254, 254, 73, 0, 0, 0],
            &[0, 0, 59, 106, 129, 218, 255, 165, 119, 96, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 104, 127, 218, 255, 164, 118, 95, 23, 0, 0, 0],
            &[0, 0, 176, 255, 255, 255, 255, 255, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ð' / 0xd0
        'Ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 254, 254, 254, 247, 220, 166, 64, 0, 0, 0, 0],
            &[0, 0, 81, 255, 216, 164, 176, 212, 255, 255, 148, 4, 0, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 52, 218, 255, 128, 0, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 28, 242, 251, 30, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 139, 255, 115, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 66, 255, 177, 0],
            &[0, 59, 141, 255, 205, 59, 59, 7, 0, 0, 35, 255, 201, 0],
            &[0, 255, 255, 255, 255, 255, 255, 33, 0, 0, 21, 255, 217, 0],
            &[0, 103, 185, 255, 249, 103, 103, 13, 0, 0, 33, 255, 206, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 73, 255, 172, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 149, 255, 111, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 38, 246, 243, 22, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 66, 226, 255, 104, 0, 0],
            &[0, 0, 81, 255, 216, 164, 178, 223, 255, 253, 122, 0, 0, 0],
            &[0, 0, 81, 255, 255, 255, 239, 212, 146, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ñ' / 0xd1
        'Ñ' => &[
            &[0, 0, 0, 0, 16, 11, 0, 0, 0, 40, 7, 0, 0, 0],
            &[0, 0, 0, 128, 254, 250, 150, 25, 60, 251, 12, 0, 0, 0],
            &[0, 0, 33, 252, 121, 130, 241, 254, 255, 155, 0, 0, 0, 0],
            &[0, 0, 43, 111, 0, 0, 21, 109, 96, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 254, 47, 0, 0, 0, 0, 177, 254, 33, 0, 0],
            &[0, 135, 255, 254, 159, 0, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 182, 248, 22, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 91, 255, 126, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 57, 196, 231, 6, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 69, 84, 255, 94, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 75, 3, 224, 206, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 115, 255, 62, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 15, 243, 174, 0, 177, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 146, 252, 33, 173, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 36, 253, 141, 164, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 177, 240, 163, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 65, 255, 236, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 0, 208, 255, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 0, 96, 255, 255, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ò' / 0xd2
        'Ò' => &[
            &[0, 0, 0, 0, 58, 210, 196, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 142, 255, 133, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 152, 251, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 105, 97, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 33, 54, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 231, 255, 255, 255, 195, 42, 0, 0, 0],
            &[0, 0, 0, 110, 255, 241, 147, 119, 169, 254, 237, 34, 0, 0],
            &[0, 0, 23, 247, 239, 35, 0, 0, 0, 102, 255, 176, 0, 0],
            &[0, 0, 124, 255, 124, 0, 0, 0, 0, 0, 218, 254, 25, 0],
            &[0, 0, 187, 255, 44, 0, 0, 0, 0, 0, 145, 255, 92, 0],
            &[0, 0, 234, 251, 5, 0, 0, 0, 0, 0, 102, 255, 134, 0],
            &[0, 1, 252, 236, 0, 0, 0, 0, 0, 0, 79, 255, 155, 0],
            &[0, 10, 255, 227, 0, 0, 0, 0, 0, 0, 73, 255, 164, 0],
            &[0, 2, 255, 236, 0, 0, 0, 0, 0, 0, 82, 255, 155, 0],
            &[0, 0, 236, 252, 6, 0, 0, 0, 0, 0, 103, 255, 132, 0],
            &[0, 0, 195, 255, 46, 0, 0, 0, 0, 0, 146, 255, 89, 0],
            &[0, 0, 126, 255, 127, 0, 0, 0, 0, 1, 224, 253, 22, 0],
            &[0, 0, 29, 248, 241, 39, 0, 0, 0, 119, 255, 170, 0, 0],
            &[0, 0, 0, 114, 255, 244, 155, 131, 186, 255, 232, 29, 0, 0],
            &[0, 0, 0, 0, 98, 226, 255, 255, 253, 182, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 32, 45, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ó' / 0xd3
        'Ó' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 168, 210, 100, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 86, 255, 185, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 233, 192, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 69, 123, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 33, 54, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 231, 255, 255, 255, 195, 42, 0, 0, 0],
            &[0, 0, 0, 110, 255, 241, 147, 119, 169, 254, 237, 34, 0, 0],
            &[0, 0, 23, 247, 239, 35, 0, 0, 0, 102, 255, 176, 0, 0],
            &[0, 0, 124, 255, 124, 0, 0, 0, 0, 0, 218, 254, 25, 0],
            &[0, 0, 187, 255, 44, 0, 0, 0, 0, 0, 145, 255, 92, 0],
            &[0, 0, 234, 251, 5, 0, 0, 0, 0, 0, 102, 255, 134, 0],
            &[0, 1, 252, 236, 0, 0, 0, 0, 0, 0, 79, 255, 155, 0],
            &[0, 10, 255, 227, 0, 0, 0, 0, 0, 0, 73, 255, 164, 0],
            &[0, 2, 255, 236, 0, 0, 0, 0, 0, 0, 82, 255, 155, 0],
            &[0, 0, 236, 252, 6, 0, 0, 0, 0, 0, 103, 255, 132, 0],
            &[0, 0, 195, 255, 46, 0, 0, 0, 0, 0, 146, 255, 89, 0],
            &[0, 0, 126, 255, 127, 0, 0, 0, 0, 1, 224, 253, 22, 0],
            &[0, 0, 29, 248, 241, 39, 0, 0, 0, 119, 255, 170, 0, 0],
            &[0, 0, 0, 114, 255, 244, 155, 131, 186, 255, 232, 29, 0, 0],
            &[0, 0, 0, 0, 98, 226, 255, 255, 253, 182, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 32, 45, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ô' / 0xd4
        'Ô' => &[
            &[0, 0, 0, 0, 0, 11, 190, 210, 107, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 164, 248, 198, 248, 58, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 244, 71, 6, 174, 235, 41, 0, 0, 0],
            &[0, 0, 0, 23, 137, 46, 0, 0, 1, 106, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 33, 54, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 231, 255, 255, 255, 195, 42, 0, 0, 0],
            &[0, 0, 0, 110, 255, 241, 147, 119, 169, 254, 237, 34, 0, 0],
            &[0, 0, 23, 247, 239, 35, 0, 0, 0, 102, 255, 176, 0, 0],
            &[0, 0, 124, 255, 124, 0, 0, 0, 0, 0, 218, 254, 25, 0],
            &[0, 0, 187, 255, 44, 0, 0, 0, 0, 0, 145, 255, 92, 0],
            &[0, 0, 234, 251, 5, 0, 0, 0, 0, 0, 102, 255, 134, 0],
            &[0, 1, 252, 236, 0, 0, 0, 0, 0, 0, 79, 255, 155, 0],
            &[0, 10, 255, 227, 0, 0, 0, 0, 0, 0, 73, 255, 164, 0],
            &[0, 2, 255, 236, 0, 0, 0, 0, 0, 0, 82, 255, 155, 0],
            &[0, 0, 236, 252, 6, 0, 0, 0, 0, 0, 103, 255, 132, 0],
            &[0, 0, 195, 255, 46, 0, 0, 0, 0, 0, 146, 255, 89, 0],
            &[0, 0, 126, 255, 127, 0, 0, 0, 0, 1, 224, 253, 22, 0],
            &[0, 0, 29, 248, 241, 39, 0, 0, 0, 119, 255, 170, 0, 0],
            &[0, 0, 0, 114, 255, 244, 155, 131, 186, 255, 232, 29, 0, 0],
            &[0, 0, 0, 0, 98, 226, 255, 255, 253, 182, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 32, 45, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Õ' / 0xd5
        'Õ' => &[
            &[0, 0, 0, 0, 0, 19, 8, 0, 0, 0, 44, 2, 0, 0],
            &[0, 0, 0, 0, 154, 255, 246, 135, 18, 86, 237, 1, 0, 0],
            &[0, 0, 0, 58, 246, 112, 141, 247, 254, 255, 128, 0, 0, 0],
            &[0, 0, 0, 57, 97, 0, 0, 30, 113, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 33, 54, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 231, 255, 255, 255, 195, 42, 0, 0, 0],
            &[0, 0, 0, 110, 255, 241, 147, 119, 169, 254, 237, 34, 0, 0],
            &[0, 0, 23, 247, 239, 35, 0, 0, 0, 102, 255, 176, 0, 0],
            &[0, 0, 124, 255, 124, 0, 0, 0, 0, 0, 218, 254, 25, 0],
            &[0, 0, 187, 255, 44, 0, 0, 0, 0, 0, 145, 255, 92, 0],
            &[0, 0, 234, 251, 5, 0, 0, 0, 0, 0, 102, 255, 134, 0],
            &[0, 1, 252, 236, 0, 0, 0, 0, 0, 0, 79, 255, 155, 0],
            &[0, 10, 255, 227, 0, 0, 0, 0, 0, 0, 73, 255, 164, 0],
            &[0, 2, 255, 236, 0, 0, 0, 0, 0, 0, 82, 255, 155, 0],
            &[0, 0, 236, 252, 6, 0, 0, 0, 0, 0, 103, 255, 132, 0],
            &[0, 0, 195, 255, 46, 0, 0, 0, 0, 0, 146, 255, 89, 0],
            &[0, 0, 126, 255, 127, 0, 0, 0, 0, 1, 224, 253, 22, 0],
            &[0, 0, 29, 248, 241, 39, 0, 0, 0, 119, 255, 170, 0, 0],
            &[0, 0, 0, 114, 255, 244, 155, 131, 186, 255, 232, 29, 0, 0],
            &[0, 0, 0, 0, 98, 226, 255, 255, 253, 182, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 32, 45, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ö' / 0xd6
        'Ö' => &[
            &[0, 0, 0, 0, 27, 14, 0, 0, 0, 35, 6, 0, 0, 0],
            &[0, 0, 0, 24, 251, 220, 0, 0, 69, 255, 172, 0, 0, 0],
            &[0, 0, 0, 19, 247, 211, 0, 0, 61, 254, 162, 0, 0, 0],
            &[0, 0, 0, 0, 18, 8, 0, 0, 0, 24, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 33, 54, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 231, 255, 255, 255, 195, 42, 0, 0, 0],
            &[0, 0, 0, 110, 255, 241, 147, 119, 169, 254, 237, 34, 0, 0],
            &[0, 0, 23, 247, 239, 35, 0, 0, 0, 102, 255, 176, 0, 0],
            &[0, 0, 124, 255, 124, 0, 0, 0, 0, 0, 218, 254, 25, 0],
            &[0, 0, 187, 255, 44, 0, 0, 0, 0, 0, 145, 255, 92, 0],
            &[0, 0, 234, 251, 5, 0, 0, 0, 0, 0, 102, 255, 134, 0],
            &[0, 1, 252, 236, 0, 0, 0, 0, 0, 0, 79, 255, 155, 0],
            &[0, 10, 255, 227, 0, 0, 0, 0, 0, 0, 73, 255, 164, 0],
            &[0, 2, 255, 236, 0, 0, 0, 0, 0, 0, 82, 255, 155, 0],
            &[0, 0, 236, 252, 6, 0, 0, 0, 0, 0, 103, 255, 132, 0],
            &[0, 0, 195, 255, 46, 0, 0, 0, 0, 0, 146, 255, 89, 0],
            &[0, 0, 126, 255, 127, 0, 0, 0, 0, 1, 224, 253, 22, 0],
            &[0, 0, 29, 248, 241, 39, 0, 0, 0, 119, 255, 170, 0, 0],
            &[0, 0, 0, 114, 255, 244, 155, 131, 186, 255, 232, 29, 0, 0],
            &[0, 0, 0, 0, 98, 226, 255, 255, 253, 182, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 32, 45, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '×' / 0xd7
        '×' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 144, 73, 0, 0, 0, 0, 2, 149, 69, 0, 0],
            &[0, 0, 10, 199, 248, 72, 0, 0, 2, 158, 255, 109, 0, 0],
            &[0, 0, 0, 14, 199, 247, 71, 2, 160, 255, 112, 0, 0, 0],
            &[0, 0, 0, 0, 14, 199, 247, 184, 255, 112, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 239, 255, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 164, 255, 226, 248, 73, 0, 0, 0, 0],
            &[0, 0, 0, 2, 161, 255, 113, 17, 203, 248, 72, 0, 0, 0],
            &[0, 0, 1, 158, 255, 115, 0, 0, 15, 200, 247, 70, 0, 0],
            &[0, 0, 7, 184, 116, 0, 0, 0, 0, 14, 195, 99, 0, 0],
            &[0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ø' / 0xd8
        'Ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 33, 54, 26, 0, 9, 132, 20, 0],
            &[0, 0, 0, 0, 99, 231, 255, 255, 255, 195, 169, 255, 68, 0],
            &[0, 0, 0, 110, 255, 241, 147, 119, 169, 255, 255, 213, 0, 0],
            &[0, 0, 23, 247, 239, 35, 0, 0, 0, 255, 255, 214, 0, 0],
            &[0, 0, 124, 255, 124, 0, 0, 0, 55, 255, 255, 254, 25, 0],
            &[0, 0, 187, 255, 44, 0, 0, 1, 199, 235, 161, 255, 92, 0],
            &[0, 0, 234, 251, 5, 0, 0, 91, 255, 106, 102, 255, 134, 0],
            &[0, 1, 252, 236, 0, 0, 10, 227, 212, 3, 79, 255, 155, 0],
            &[0, 10, 255, 227, 0, 0, 127, 255, 70, 0, 73, 255, 164, 0],
            &[0, 2, 255, 236, 0, 28, 245, 180, 0, 0, 82, 255, 155, 0],
            &[0, 0, 236, 252, 6, 164, 250, 39, 0, 0, 103, 255, 132, 0],
            &[0, 0, 195, 255, 102, 254, 144, 0, 0, 0, 146, 255, 89, 0],
            &[0, 0, 126, 255, 255, 236, 17, 0, 0, 1, 224, 253, 22, 0],
            &[0, 0, 29, 255, 255, 147, 0, 0, 0, 119, 255, 170, 0, 0],
            &[0, 0, 10, 255, 255, 248, 155, 131, 186, 255, 232, 29, 0, 0],
            &[0, 0, 128, 255, 170, 226, 255, 255, 253, 182, 34, 0, 0, 0],
            &[0, 0, 59, 136, 0, 2, 32, 45, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ù' / 0xd9
        'Ù' => &[
            &[0, 0, 0, 58, 210, 196, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 255, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 152, 251, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 105, 97, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 0, 193, 254, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 134, 255, 93, 0, 0, 0, 0, 0, 195, 255, 32, 0, 0],
            &[0, 119, 255, 108, 0, 0, 0, 0, 0, 210, 255, 17, 0, 0],
            &[0, 86, 255, 156, 0, 0, 0, 0, 8, 247, 236, 1, 0, 0],
            &[0, 21, 249, 242, 33, 0, 0, 0, 117, 255, 162, 0, 0, 0],
            &[0, 0, 134, 255, 237, 149, 129, 178, 254, 241, 36, 0, 0, 0],
            &[0, 0, 1, 123, 236, 255, 255, 255, 194, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 32, 44, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ú' / 0xda
        'Ú' => &[
            &[0, 0, 0, 0, 0, 0, 0, 168, 210, 100, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 86, 255, 185, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 233, 192, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 69, 123, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 0, 193, 254, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 134, 255, 93, 0, 0, 0, 0, 0, 195, 255, 32, 0, 0],
            &[0, 119, 255, 108, 0, 0, 0, 0, 0, 210, 255, 17, 0, 0],
            &[0, 86, 255, 156, 0, 0, 0, 0, 8, 247, 236, 1, 0, 0],
            &[0, 21, 249, 242, 33, 0, 0, 0, 117, 255, 162, 0, 0, 0],
            &[0, 0, 134, 255, 237, 149, 129, 178, 254, 241, 36, 0, 0, 0],
            &[0, 0, 1, 123, 236, 255, 255, 255, 194, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 32, 44, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Û' / 0xdb
        'Û' => &[
            &[0, 0, 0, 0, 11, 190, 210, 107, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 248, 198, 248, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 130, 244, 71, 6, 174, 235, 41, 0, 0, 0, 0],
            &[0, 0, 23, 137, 46, 0, 0, 1, 106, 105, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 0, 193, 254, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 134, 255, 93, 0, 0, 0, 0, 0, 195, 255, 32, 0, 0],
            &[0, 119, 255, 108, 0, 0, 0, 0, 0, 210, 255, 17, 0, 0],
            &[0, 86, 255, 156, 0, 0, 0, 0, 8, 247, 236, 1, 0, 0],
            &[0, 21, 249, 242, 33, 0, 0, 0, 117, 255, 162, 0, 0, 0],
            &[0, 0, 134, 255, 237, 149, 129, 178, 254, 241, 36, 0, 0, 0],
            &[0, 0, 1, 123, 236, 255, 255, 255, 194, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 32, 44, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ü' / 0xdc
        'Ü' => &[
            &[0, 0, 0, 27, 14, 0, 0, 0, 35, 6, 0, 0, 0, 0],
            &[0, 0, 24, 251, 220, 0, 0, 69, 255, 172, 0, 0, 0, 0],
            &[0, 0, 19, 247, 211, 0, 0, 61, 254, 162, 0, 0, 0, 0],
            &[0, 0, 0, 18, 8, 0, 0, 0, 24, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 0, 193, 254, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 134, 255, 93, 0, 0, 0, 0, 0, 195, 255, 32, 0, 0],
            &[0, 119, 255, 108, 0, 0, 0, 0, 0, 210, 255, 17, 0, 0],
            &[0, 86, 255, 156, 0, 0, 0, 0, 8, 247, 236, 1, 0, 0],
            &[0, 21, 249, 242, 33, 0, 0, 0, 117, 255, 162, 0, 0, 0],
            &[0, 0, 134, 255, 237, 149, 129, 178, 254, 241, 36, 0, 0, 0],
            &[0, 0, 1, 123, 236, 255, 255, 255, 194, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 32, 44, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ý' / 0xdd
        'Ý' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 168, 210, 100, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 86, 255, 185, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 233, 192, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 69, 123, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 239, 244, 20, 0, 0, 0, 0, 0, 97, 254, 155, 0],
            &[0, 0, 129, 255, 129, 0, 0, 0, 0, 1, 214, 250, 32, 0],
            &[0, 0, 15, 238, 236, 10, 0, 0, 0, 79, 255, 152, 0, 0],
            &[0, 0, 0, 124, 255, 110, 0, 0, 0, 198, 248, 29, 0, 0],
            &[0, 0, 0, 13, 236, 224, 4, 0, 62, 255, 147, 0, 0, 0],
            &[0, 0, 0, 0, 119, 255, 90, 0, 181, 247, 26, 0, 0, 0],
            &[0, 0, 0, 0, 11, 233, 208, 45, 254, 142, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 114, 255, 214, 245, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 231, 255, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 156, 255, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Þ' / 0xde
        'Þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 254, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 234, 67, 67, 63, 45, 6, 0, 0, 0, 0, 0],
            &[0, 1, 255, 255, 255, 255, 255, 255, 245, 148, 11, 0, 0, 0],
            &[0, 1, 255, 237, 95, 95, 107, 144, 239, 255, 178, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 27, 237, 255, 48, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 155, 255, 100, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 133, 255, 103, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 180, 255, 63, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 1, 103, 254, 221, 3, 0, 0],
            &[0, 1, 255, 246, 174, 175, 187, 231, 255, 238, 56, 0, 0, 0],
            &[0, 1, 255, 254, 244, 244, 232, 207, 138, 30, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ß' / 0xdf
        'ß' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 133, 220, 255, 255, 243, 183, 62, 0, 0, 0, 0],
            &[0, 0, 191, 255, 229, 161, 144, 185, 253, 252, 67, 0, 0, 0],
            &[0, 60, 255, 202, 15, 0, 0, 0, 109, 255, 168, 0, 0, 0],
            &[0, 100, 255, 118, 0, 0, 0, 0, 72, 255, 157, 0, 0, 0],
            &[0, 108, 255, 108, 0, 0, 0, 14, 200, 253, 65, 0, 0, 0],
            &[0, 108, 255, 108, 0, 0, 28, 215, 251, 101, 0, 0, 0, 0],
            &[0, 108, 255, 108, 0, 0, 187, 255, 77, 0, 0, 0, 0, 0],
            &[0, 108, 255, 108, 0, 0, 244, 225, 0, 0, 0, 0, 0, 0],
            &[0, 108, 255, 108, 0, 0, 186, 255, 155, 9, 0, 0, 0, 0],
            &[0, 108, 255, 108, 0, 0, 23, 204, 255, 214, 42, 0, 0, 0],
            &[0, 108, 255, 108, 0, 0, 0, 6, 145, 255, 243, 47, 0, 0],
            &[0, 108, 255, 108, 0, 0, 0, 0, 0, 90, 254, 208, 0, 0],
            &[0, 108, 255, 108, 0, 0, 0, 0, 0, 0, 215, 255, 10, 0],
            &[0, 108, 255, 108, 0, 11, 0, 0, 0, 4, 230, 251, 3, 0],
            &[0, 108, 255, 108, 0, 213, 159, 109, 95, 178, 255, 170, 0, 0],
            &[0, 108, 255, 108, 0, 150, 248, 255, 255, 251, 169, 15, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 36, 45, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'à' / 0xe0
        'à' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 22, 19, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 238, 253, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 247, 200, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 68, 243, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 36, 61, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 68, 106, 114, 91, 27, 0, 0, 0, 0],
            &[0, 0, 0, 103, 237, 255, 255, 255, 255, 252, 131, 0, 0, 0],
            &[0, 0, 0, 71, 180, 83, 41, 27, 76, 217, 255, 77, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 156, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 255, 179, 0, 0],
            &[0, 0, 0, 0, 46, 130, 176, 197, 206, 213, 255, 181, 0, 0],
            &[0, 0, 0, 136, 254, 245, 186, 153, 139, 132, 255, 181, 0, 0],
            &[0, 0, 59, 255, 199, 20, 0, 0, 0, 6, 255, 181, 0, 0],
            &[0, 0, 121, 255, 80, 0, 0, 0, 0, 45, 255, 181, 0, 0],
            &[0, 0, 109, 255, 103, 0, 0, 0, 4, 179, 255, 181, 0, 0],
            &[0, 0, 41, 249, 239, 126, 89, 117, 211, 201, 252, 181, 0, 0],
            &[0, 0, 0, 83, 227, 255, 255, 249, 154, 13, 222, 181, 0, 0],
            &[0, 0, 0, 0, 2, 30, 41, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'á' / 0xe1
        'á' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 10, 22, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 201, 255, 117, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 102, 255, 155, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 34, 241, 157, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 63, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 68, 106, 114, 91, 27, 0, 0, 0, 0],
            &[0, 0, 0, 103, 237, 255, 255, 255, 255, 252, 131, 0, 0, 0],
            &[0, 0, 0, 71, 180, 83, 41, 27, 76, 217, 255, 77, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 156, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 255, 179, 0, 0],
            &[0, 0, 0, 0, 46, 130, 176, 197, 206, 213, 255, 181, 0, 0],
            &[0, 0, 0, 136, 254, 245, 186, 153, 139, 132, 255, 181, 0, 0],
            &[0, 0, 59, 255, 199, 20, 0, 0, 0, 6, 255, 181, 0, 0],
            &[0, 0, 121, 255, 80, 0, 0, 0, 0, 45, 255, 181, 0, 0],
            &[0, 0, 109, 255, 103, 0, 0, 0, 4, 179, 255, 181, 0, 0],
            &[0, 0, 41, 249, 239, 126, 89, 117, 211, 201, 252, 181, 0, 0],
            &[0, 0, 0, 83, 227, 255, 255, 249, 154, 13, 222, 181, 0, 0],
            &[0, 0, 0, 0, 2, 30, 41, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'â' / 0xe2
        'â' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 22, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 19, 230, 255, 187, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 189, 234, 128, 253, 135, 0, 0, 0, 0],
            &[0, 0, 0, 1, 162, 224, 43, 0, 87, 246, 111, 0, 0, 0],
            &[0, 0, 0, 9, 70, 19, 0, 0, 0, 36, 66, 0, 0, 0],
            &[0, 0, 0, 0, 7, 68, 106, 114, 91, 27, 0, 0, 0, 0],
            &[0, 0, 0, 103, 237, 255, 255, 255, 255, 252, 131, 0, 0, 0],
            &[0, 0, 0, 71, 180, 83, 41, 27, 76, 217, 255, 77, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 156, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 255, 179, 0, 0],
            &[0, 0, 0, 0, 46, 130, 176, 197, 206, 213, 255, 181, 0, 0],
            &[0, 0, 0, 136, 254, 245, 186, 153, 139, 132, 255, 181, 0, 0],
            &[0, 0, 59, 255, 199, 20, 0, 0, 0, 6, 255, 181, 0, 0],
            &[0, 0, 121, 255, 80, 0, 0, 0, 0, 45, 255, 181, 0, 0],
            &[0, 0, 109, 255, 103, 0, 0, 0, 4, 179, 255, 181, 0, 0],
            &[0, 0, 41, 249, 239, 126, 89, 117, 211, 201, 252, 181, 0, 0],
            &[0, 0, 0, 83, 227, 255, 255, 249, 154, 13, 222, 181, 0, 0],
            &[0, 0, 0, 0, 2, 30, 41, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ã' / 0xe3
        'ã' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 82, 73, 3, 0, 1, 106, 15, 0, 0],
            &[0, 0, 0, 0, 186, 255, 255, 214, 91, 126, 241, 4, 0, 0],
            &[0, 0, 0, 52, 245, 54, 63, 196, 255, 254, 104, 0, 0, 0],
            &[0, 0, 0, 24, 52, 0, 0, 0, 42, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 68, 106, 114, 91, 27, 0, 0, 0, 0],
            &[0, 0, 0, 103, 237, 255, 255, 255, 255, 252, 131, 0, 0, 0],
            &[0, 0, 0, 71, 180, 83, 41, 27, 76, 217, 255, 77, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 156, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 255, 179, 0, 0],
            &[0, 0, 0, 0, 46, 130, 176, 197, 206, 213, 255, 181, 0, 0],
            &[0, 0, 0, 136, 254, 245, 186, 153, 139, 132, 255, 181, 0, 0],
            &[0, 0, 59, 255, 199, 20, 0, 0, 0, 6, 255, 181, 0, 0],
            &[0, 0, 121, 255, 80, 0, 0, 0, 0, 45, 255, 181, 0, 0],
            &[0, 0, 109, 255, 103, 0, 0, 0, 4, 179, 255, 181, 0, 0],
            &[0, 0, 41, 249, 239, 126, 89, 117, 211, 201, 252, 181, 0, 0],
            &[0, 0, 0, 83, 227, 255, 255, 249, 154, 13, 222, 181, 0, 0],
            &[0, 0, 0, 0, 2, 30, 41, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ä' / 0xe4
        'ä' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 75, 0, 0, 2, 97, 53, 0, 0, 0],
            &[0, 0, 0, 14, 251, 251, 14, 0, 58, 255, 216, 0, 0, 0],
            &[0, 0, 0, 0, 180, 178, 0, 0, 18, 206, 135, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 68, 106, 114, 91, 27, 0, 0, 0, 0],
            &[0, 0, 0, 103, 237, 255, 255, 255, 255, 252, 131, 0, 0, 0],
            &[0, 0, 0, 71, 180, 83, 41, 27, 76, 217, 255, 77, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 156, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 255, 179, 0, 0],
            &[0, 0, 0, 0, 46, 130, 176, 197, 206, 213, 255, 181, 0, 0],
            &[0, 0, 0, 136, 254, 245, 186, 153, 139, 132, 255, 181, 0, 0],
            &[0, 0, 59, 255, 199, 20, 0, 0, 0, 6, 255, 181, 0, 0],
            &[0, 0, 121, 255, 80, 0, 0, 0, 0, 45, 255, 181, 0, 0],
            &[0, 0, 109, 255, 103, 0, 0, 0, 4, 179, 255, 181, 0, 0],
            &[0, 0, 41, 249, 239, 126, 89, 117, 211, 201, 252, 181, 0, 0],
            &[0, 0, 0, 83, 227, 255, 255, 249, 154, 13, 222, 181, 0, 0],
            &[0, 0, 0, 0, 2, 30, 41, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'å' / 0xe5
        'å' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 29, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 241, 250, 221, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 207, 135, 7, 184, 157, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 117, 0, 171, 161, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 94, 250, 233, 238, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 27, 59, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 68, 106, 114, 91, 27, 0, 0, 0, 0],
            &[0, 0, 0, 103, 237, 255, 255, 255, 255, 252, 131, 0, 0, 0],
            &[0, 0, 0, 71, 180, 83, 41, 27, 76, 217, 255, 77, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 156, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 255, 179, 0, 0],
            &[0, 0, 0, 0, 46, 130, 176, 197, 206, 213, 255, 181, 0, 0],
            &[0, 0, 0, 136, 254, 245, 186, 153, 139, 132, 255, 181, 0, 0],
            &[0, 0, 59, 255, 199, 20, 0, 0, 0, 6, 255, 181, 0, 0],
            &[0, 0, 121, 255, 80, 0, 0, 0, 0, 45, 255, 181, 0, 0],
            &[0, 0, 109, 255, 103, 0, 0, 0, 4, 179, 255, 181, 0, 0],
            &[0, 0, 41, 249, 239, 126, 89, 117, 211, 201, 252, 181, 0, 0],
            &[0, 0, 0, 83, 227, 255, 255, 249, 154, 13, 222, 181, 0, 0],
            &[0, 0, 0, 0, 2, 30, 41, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'æ' / 0xe6
        'æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 41, 99, 99, 24, 0, 16, 97, 101, 21, 0, 0, 0],
            &[0, 139, 255, 255, 255, 238, 65, 226, 255, 255, 235, 35, 0, 0],
            &[0, 86, 116, 35, 73, 248, 250, 228, 54, 85, 254, 168, 0, 0],
            &[0, 0, 0, 0, 0, 178, 255, 104, 0, 0, 190, 242, 2, 0],
            &[0, 0, 0, 0, 0, 152, 255, 44, 0, 0, 156, 255, 29, 0],
            &[
                0, 13, 120, 174, 188, 229, 255, 195, 189, 189, 226, 255, 43, 0,
            ],
            &[
                8, 208, 255, 212, 161, 214, 255, 189, 186, 186, 186, 186, 33, 0,
            ],
            &[93, 255, 139, 0, 0, 161, 255, 23, 0, 0, 0, 0, 0, 0],
            &[139, 255, 40, 0, 0, 181, 255, 61, 0, 0, 0, 0, 0, 0],
            &[126, 255, 49, 0, 22, 240, 255, 155, 0, 0, 0, 9, 0, 0],
            &[
                70, 255, 202, 106, 202, 226, 130, 255, 153, 92, 129, 209, 0, 0,
            ],
            &[0, 150, 255, 255, 223, 51, 6, 157, 252, 255, 253, 158, 0, 0],
            &[0, 0, 25, 38, 3, 0, 0, 0, 19, 45, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ç' / 0xe7
        'ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 57, 99, 111, 84, 29, 0, 0, 0],
            &[0, 0, 0, 0, 63, 221, 255, 255, 255, 255, 253, 101, 0, 0],
            &[0, 0, 0, 45, 246, 253, 148, 68, 51, 79, 161, 41, 0, 0],
            &[0, 0, 0, 176, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 248, 243, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 255, 200, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 188, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 31, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 243, 249, 13, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 164, 255, 148, 0, 0, 0, 0, 0, 2, 0, 0],
            &[0, 0, 0, 27, 235, 255, 200, 125, 112, 139, 193, 120, 0, 0],
            &[0, 0, 0, 0, 31, 173, 249, 255, 255, 255, 222, 79, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 251, 144, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 253, 186, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 18, 193, 217, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'è' / 0xe8
        'è' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 22, 18, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 241, 252, 44, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 67, 248, 195, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 245, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 85, 111, 91, 25, 0, 0, 0, 0, 0],
            &[0, 0, 2, 131, 248, 255, 255, 255, 250, 127, 1, 0, 0, 0],
            &[0, 0, 139, 255, 202, 70, 27, 55, 192, 255, 106, 0, 0, 0],
            &[0, 38, 253, 219, 11, 0, 0, 0, 14, 236, 232, 2, 0, 0],
            &[0, 125, 255, 109, 0, 0, 0, 0, 0, 166, 255, 50, 0, 0],
            &[
                0, 166, 255, 214, 200, 200, 200, 200, 200, 231, 255, 77, 0, 0,
            ],
            &[
                0, 182, 255, 199, 186, 186, 186, 186, 186, 186, 186, 63, 0, 0,
            ],
            &[0, 161, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 109, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 242, 240, 43, 0, 0, 0, 0, 0, 24, 0, 0, 0],
            &[0, 0, 85, 251, 247, 156, 103, 99, 133, 197, 213, 0, 0, 0],
            &[0, 0, 0, 55, 185, 252, 255, 255, 255, 213, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'é' / 0xe9
        'é' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 11, 22, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 206, 255, 112, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 108, 255, 150, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 243, 153, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 62, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 85, 111, 91, 25, 0, 0, 0, 0, 0],
            &[0, 0, 2, 131, 248, 255, 255, 255, 250, 127, 1, 0, 0, 0],
            &[0, 0, 139, 255, 202, 70, 27, 55, 192, 255, 106, 0, 0, 0],
            &[0, 38, 253, 219, 11, 0, 0, 0, 14, 236, 232, 2, 0, 0],
            &[0, 125, 255, 109, 0, 0, 0, 0, 0, 166, 255, 50, 0, 0],
            &[
                0, 166, 255, 214, 200, 200, 200, 200, 200, 231, 255, 77, 0, 0,
            ],
            &[
                0, 182, 255, 199, 186, 186, 186, 186, 186, 186, 186, 63, 0, 0,
            ],
            &[0, 161, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 109, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 242, 240, 43, 0, 0, 0, 0, 0, 24, 0, 0, 0],
            &[0, 0, 85, 251, 247, 156, 103, 99, 133, 197, 213, 0, 0, 0],
            &[0, 0, 0, 55, 185, 252, 255, 255, 255, 213, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ê' / 0xea
        'ê' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 22, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 232, 255, 182, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 193, 232, 129, 254, 130, 0, 0, 0, 0, 0],
            &[0, 0, 2, 167, 221, 40, 0, 91, 247, 106, 0, 0, 0, 0],
            &[0, 0, 11, 70, 18, 0, 0, 0, 37, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 85, 111, 91, 25, 0, 0, 0, 0, 0],
            &[0, 0, 2, 131, 248, 255, 255, 255, 250, 127, 1, 0, 0, 0],
            &[0, 0, 139, 255, 202, 70, 27, 55, 192, 255, 106, 0, 0, 0],
            &[0, 38, 253, 219, 11, 0, 0, 0, 14, 236, 232, 2, 0, 0],
            &[0, 125, 255, 109, 0, 0, 0, 0, 0, 166, 255, 50, 0, 0],
            &[
                0, 166, 255, 214, 200, 200, 200, 200, 200, 231, 255, 77, 0, 0,
            ],
            &[
                0, 182, 255, 199, 186, 186, 186, 186, 186, 186, 186, 63, 0, 0,
            ],
            &[0, 161, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 109, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 242, 240, 43, 0, 0, 0, 0, 0, 24, 0, 0, 0],
            &[0, 0, 85, 251, 247, 156, 103, 99, 133, 197, 213, 0, 0, 0],
            &[0, 0, 0, 55, 185, 252, 255, 255, 255, 213, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ë' / 0xeb
        'ë' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 80, 73, 0, 0, 3, 98, 51, 0, 0, 0, 0],
            &[0, 0, 18, 253, 249, 10, 0, 64, 255, 211, 0, 0, 0, 0],
            &[0, 0, 1, 185, 174, 0, 0, 21, 208, 130, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 85, 111, 91, 25, 0, 0, 0, 0, 0],
            &[0, 0, 2, 131, 248, 255, 255, 255, 250, 127, 1, 0, 0, 0],
            &[0, 0, 139, 255, 202, 70, 27, 55, 192, 255, 106, 0, 0, 0],
            &[0, 38, 253, 219, 11, 0, 0, 0, 14, 236, 232, 2, 0, 0],
            &[0, 125, 255, 109, 0, 0, 0, 0, 0, 166, 255, 50, 0, 0],
            &[
                0, 166, 255, 214, 200, 200, 200, 200, 200, 231, 255, 77, 0, 0,
            ],
            &[
                0, 182, 255, 199, 186, 186, 186, 186, 186, 186, 186, 63, 0, 0,
            ],
            &[0, 161, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 109, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 242, 240, 43, 0, 0, 0, 0, 0, 24, 0, 0, 0],
            &[0, 0, 85, 251, 247, 156, 103, 99, 133, 197, 213, 0, 0, 0],
            &[0, 0, 0, 55, 185, 252, 255, 255, 255, 213, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ì' / 0xec
        'ì' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 22, 17, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 49, 248, 246, 29, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 252, 176, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 250, 97, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 43, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 64, 65, 65, 65, 65, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 39, 58, 130, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 22, 83, 103, 123, 175, 255, 208, 127, 105, 84, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'í' / 0xed
        'í' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 22, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 222, 255, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 129, 255, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 250, 133, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 56, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 64, 65, 65, 65, 65, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 39, 58, 130, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 22, 83, 103, 123, 175, 255, 208, 127, 105, 84, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'î' / 0xee
        'î' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 22, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 242, 255, 161, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 208, 222, 138, 255, 108, 0, 0, 0, 0, 0],
            &[0, 0, 6, 185, 210, 29, 0, 108, 248, 87, 0, 0, 0, 0],
            &[0, 0, 17, 70, 12, 0, 0, 0, 43, 59, 0, 0, 0, 0],
            &[0, 0, 64, 65, 65, 65, 65, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 39, 58, 130, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 22, 83, 103, 123, 175, 255, 208, 127, 105, 84, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ï' / 0xef
        'ï' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 90, 63, 0, 0, 8, 102, 42, 0, 0, 0, 0],
            &[0, 0, 36, 255, 237, 1, 0, 85, 255, 189, 0, 0, 0, 0],
            &[0, 0, 7, 198, 154, 0, 0, 35, 212, 112, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 64, 65, 65, 65, 65, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 39, 58, 130, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 22, 83, 103, 123, 175, 255, 208, 127, 105, 84, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ð' / 0xf0
        'ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 194, 135, 11, 0, 84, 189, 13, 0, 0, 0],
            &[0, 0, 0, 11, 164, 255, 255, 236, 255, 177, 26, 0, 0, 0],
            &[0, 0, 0, 0, 22, 225, 255, 255, 123, 0, 0, 0, 0, 0],
            &[0, 0, 0, 84, 240, 238, 149, 228, 235, 30, 0, 0, 0, 0],
            &[0, 0, 0, 36, 146, 19, 0, 45, 244, 193, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 97, 255, 82, 0, 0, 0],
            &[0, 0, 0, 74, 178, 222, 231, 196, 102, 215, 191, 0, 0, 0],
            &[0, 0, 132, 255, 238, 167, 150, 189, 253, 242, 254, 19, 0, 0],
            &[0, 61, 255, 207, 18, 0, 0, 0, 72, 252, 255, 73, 0, 0],
            &[0, 159, 255, 73, 0, 0, 0, 0, 0, 166, 255, 100, 0, 0],
            &[0, 200, 255, 20, 0, 0, 0, 0, 0, 116, 255, 108, 0, 0],
            &[0, 204, 255, 14, 0, 0, 0, 0, 0, 122, 255, 91, 0, 0],
            &[0, 169, 255, 56, 0, 0, 0, 0, 0, 168, 255, 43, 0, 0],
            &[0, 78, 255, 178, 2, 0, 0, 0, 37, 247, 209, 0, 0, 0],
            &[0, 0, 166, 255, 197, 104, 84, 124, 233, 248, 58, 0, 0, 0],
            &[0, 0, 2, 118, 229, 255, 255, 255, 199, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 30, 47, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ñ' / 0xf1
        'ñ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 72, 82, 9, 0, 0, 90, 31, 0, 0, 0],
            &[0, 0, 0, 149, 255, 255, 228, 105, 99, 251, 29, 0, 0, 0],
            &[0, 0, 17, 253, 79, 49, 175, 255, 255, 140, 0, 0, 0, 0],
            &[0, 0, 13, 63, 0, 0, 0, 33, 40, 0, 0, 0, 0, 0],
            &[0, 20, 65, 13, 0, 41, 94, 106, 66, 2, 0, 0, 0, 0],
            &[0, 81, 255, 82, 157, 254, 255, 255, 255, 202, 25, 0, 0, 0],
            &[0, 81, 255, 224, 223, 78, 30, 50, 186, 255, 166, 0, 0, 0],
            &[0, 81, 255, 254, 38, 0, 0, 0, 23, 254, 238, 0, 0, 0],
            &[0, 81, 255, 198, 0, 0, 0, 0, 0, 231, 255, 4, 0, 0],
            &[0, 81, 255, 164, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 146, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ò' / 0xf2
        'ò' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 22, 17, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 49, 248, 246, 29, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 252, 176, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 250, 97, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 43, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 96, 110, 78, 9, 0, 0, 0, 0, 0],
            &[0, 0, 18, 178, 255, 255, 255, 255, 240, 94, 0, 0, 0, 0],
            &[0, 2, 193, 255, 166, 57, 27, 80, 222, 255, 84, 0, 0, 0],
            &[0, 81, 255, 183, 0, 0, 0, 0, 37, 250, 227, 2, 0, 0],
            &[0, 163, 255, 73, 0, 0, 0, 0, 0, 181, 255, 57, 0, 0],
            &[0, 196, 255, 30, 0, 0, 0, 0, 0, 135, 255, 92, 0, 0],
            &[0, 210, 255, 11, 0, 0, 0, 0, 0, 124, 255, 108, 0, 0],
            &[0, 193, 255, 32, 0, 0, 0, 0, 0, 144, 255, 92, 0, 0],
            &[0, 142, 255, 94, 0, 0, 0, 0, 0, 205, 255, 42, 0, 0],
            &[0, 49, 255, 215, 14, 0, 0, 0, 81, 255, 205, 0, 0, 0],
            &[0, 0, 140, 255, 216, 113, 92, 145, 248, 247, 53, 0, 0, 0],
            &[0, 0, 0, 109, 229, 255, 255, 254, 196, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 34, 47, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ó' / 0xf3
        'ó' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 22, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 222, 255, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 129, 255, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 250, 133, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 56, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 96, 110, 78, 9, 0, 0, 0, 0, 0],
            &[0, 0, 18, 178, 255, 255, 255, 255, 240, 94, 0, 0, 0, 0],
            &[0, 2, 193, 255, 166, 57, 27, 80, 222, 255, 84, 0, 0, 0],
            &[0, 81, 255, 183, 0, 0, 0, 0, 37, 250, 227, 2, 0, 0],
            &[0, 163, 255, 73, 0, 0, 0, 0, 0, 181, 255, 57, 0, 0],
            &[0, 196, 255, 30, 0, 0, 0, 0, 0, 135, 255, 92, 0, 0],
            &[0, 210, 255, 11, 0, 0, 0, 0, 0, 124, 255, 108, 0, 0],
            &[0, 193, 255, 32, 0, 0, 0, 0, 0, 144, 255, 92, 0, 0],
            &[0, 142, 255, 94, 0, 0, 0, 0, 0, 205, 255, 42, 0, 0],
            &[0, 49, 255, 215, 14, 0, 0, 0, 81, 255, 205, 0, 0, 0],
            &[0, 0, 140, 255, 216, 113, 92, 145, 248, 247, 53, 0, 0, 0],
            &[0, 0, 0, 109, 229, 255, 255, 254, 196, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 34, 47, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ô' / 0xf4
        'ô' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 22, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 242, 255, 161, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 208, 222, 138, 255, 108, 0, 0, 0, 0, 0],
            &[0, 0, 6, 185, 210, 29, 0, 108, 248, 87, 0, 0, 0, 0],
            &[0, 0, 17, 70, 12, 0, 0, 0, 43, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 96, 110, 78, 9, 0, 0, 0, 0, 0],
            &[0, 0, 18, 178, 255, 255, 255, 255, 240, 94, 0, 0, 0, 0],
            &[0, 2, 193, 255, 166, 57, 27, 80, 222, 255, 84, 0, 0, 0],
            &[0, 81, 255, 183, 0, 0, 0, 0, 37, 250, 227, 2, 0, 0],
            &[0, 163, 255, 73, 0, 0, 0, 0, 0, 181, 255, 57, 0, 0],
            &[0, 196, 255, 30, 0, 0, 0, 0, 0, 135, 255, 92, 0, 0],
            &[0, 210, 255, 11, 0, 0, 0, 0, 0, 124, 255, 108, 0, 0],
            &[0, 193, 255, 32, 0, 0, 0, 0, 0, 144, 255, 92, 0, 0],
            &[0, 142, 255, 94, 0, 0, 0, 0, 0, 205, 255, 42, 0, 0],
            &[0, 49, 255, 215, 14, 0, 0, 0, 81, 255, 205, 0, 0, 0],
            &[0, 0, 140, 255, 216, 113, 92, 145, 248, 247, 53, 0, 0, 0],
            &[0, 0, 0, 109, 229, 255, 255, 254, 196, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 34, 47, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'õ' / 0xf5
        'õ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 86, 65, 1, 0, 7, 110, 3, 0, 0, 0],
            &[0, 0, 5, 208, 255, 255, 201, 83, 146, 218, 0, 0, 0, 0],
            &[0, 0, 79, 231, 45, 74, 209, 255, 250, 81, 0, 0, 0, 0],
            &[0, 0, 31, 45, 0, 0, 2, 46, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 96, 110, 78, 9, 0, 0, 0, 0, 0],
            &[0, 0, 18, 178, 255, 255, 255, 255, 240, 94, 0, 0, 0, 0],
            &[0, 2, 193, 255, 166, 57, 27, 80, 222, 255, 84, 0, 0, 0],
            &[0, 81, 255, 183, 0, 0, 0, 0, 37, 250, 227, 2, 0, 0],
            &[0, 163, 255, 73, 0, 0, 0, 0, 0, 181, 255, 57, 0, 0],
            &[0, 196, 255, 30, 0, 0, 0, 0, 0, 135, 255, 92, 0, 0],
            &[0, 210, 255, 11, 0, 0, 0, 0, 0, 124, 255, 108, 0, 0],
            &[0, 193, 255, 32, 0, 0, 0, 0, 0, 144, 255, 92, 0, 0],
            &[0, 142, 255, 94, 0, 0, 0, 0, 0, 205, 255, 42, 0, 0],
            &[0, 49, 255, 215, 14, 0, 0, 0, 81, 255, 205, 0, 0, 0],
            &[0, 0, 140, 255, 216, 113, 92, 145, 248, 247, 53, 0, 0, 0],
            &[0, 0, 0, 109, 229, 255, 255, 254, 196, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 34, 47, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ö' / 0xf6
        'ö' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 90, 63, 0, 0, 8, 102, 42, 0, 0, 0, 0],
            &[0, 0, 36, 255, 237, 1, 0, 85, 255, 189, 0, 0, 0, 0],
            &[0, 0, 7, 198, 154, 0, 0, 35, 212, 112, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 96, 110, 78, 9, 0, 0, 0, 0, 0],
            &[0, 0, 18, 178, 255, 255, 255, 255, 240, 94, 0, 0, 0, 0],
            &[0, 2, 193, 255, 166, 57, 27, 80, 222, 255, 84, 0, 0, 0],
            &[0, 81, 255, 183, 0, 0, 0, 0, 37, 250, 227, 2, 0, 0],
            &[0, 163, 255, 73, 0, 0, 0, 0, 0, 181, 255, 57, 0, 0],
            &[0, 196, 255, 30, 0, 0, 0, 0, 0, 135, 255, 92, 0, 0],
            &[0, 210, 255, 11, 0, 0, 0, 0, 0, 124, 255, 108, 0, 0],
            &[0, 193, 255, 32, 0, 0, 0, 0, 0, 144, 255, 92, 0, 0],
            &[0, 142, 255, 94, 0, 0, 0, 0, 0, 205, 255, 42, 0, 0],
            &[0, 49, 255, 215, 14, 0, 0, 0, 81, 255, 205, 0, 0, 0],
            &[0, 0, 140, 255, 216, 113, 92, 145, 248, 247, 53, 0, 0, 0],
            &[0, 0, 0, 109, 229, 255, 255, 254, 196, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 34, 47, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '÷' / 0xf7
        '÷' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 182, 255, 65, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 185, 255, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 47, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 37, 37, 37, 37, 37, 37, 37, 37, 34, 0, 0],
            &[0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 235, 0, 0],
            &[0, 0, 28, 87, 87, 87, 87, 87, 87, 87, 87, 80, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 156, 246, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 197, 255, 79, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 89, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ø' / 0xf8
        'ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 96, 110, 78, 9, 81, 136, 6, 0, 0],
            &[0, 0, 18, 178, 255, 255, 255, 255, 255, 255, 194, 3, 0, 0],
            &[0, 2, 193, 255, 166, 57, 27, 82, 255, 255, 115, 0, 0, 0],
            &[0, 81, 255, 183, 0, 0, 0, 118, 255, 255, 227, 2, 0, 0],
            &[0, 163, 255, 73, 0, 0, 51, 250, 160, 181, 255, 57, 0, 0],
            &[0, 196, 255, 30, 0, 11, 217, 220, 13, 135, 255, 92, 0, 0],
            &[0, 210, 255, 11, 0, 155, 251, 55, 0, 124, 255, 108, 0, 0],
            &[0, 193, 255, 32, 82, 255, 123, 0, 0, 144, 255, 92, 0, 0],
            &[0, 142, 255, 122, 237, 193, 2, 0, 0, 205, 255, 42, 0, 0],
            &[0, 49, 255, 255, 253, 30, 0, 0, 81, 255, 205, 0, 0, 0],
            &[0, 0, 255, 255, 255, 113, 92, 145, 248, 247, 53, 0, 0, 0],
            &[0, 46, 250, 255, 229, 255, 255, 254, 196, 53, 0, 0, 0, 0],
            &[0, 5, 116, 12, 1, 34, 47, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ù' / 0xf9
        'ù' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 22, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 248, 246, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 252, 176, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 89, 250, 97, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 43, 53, 0, 0, 0, 0, 0],
            &[0, 0, 27, 65, 30, 0, 0, 0, 0, 0, 63, 60, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 249, 235, 0, 0],
            &[0, 0, 108, 255, 119, 0, 0, 0, 0, 7, 255, 235, 0, 0],
            &[0, 0, 98, 255, 135, 0, 0, 0, 0, 52, 255, 235, 0, 0],
            &[0, 0, 64, 255, 201, 0, 0, 0, 0, 163, 255, 235, 0, 0],
            &[0, 0, 6, 219, 255, 179, 99, 99, 177, 228, 232, 235, 0, 0],
            &[0, 0, 0, 46, 203, 255, 255, 255, 198, 42, 174, 235, 0, 0],
            &[0, 0, 0, 0, 0, 17, 45, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ú' / 0xfa
        'ú' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 12, 22, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 7, 222, 255, 90, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 129, 255, 129, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 52, 250, 133, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 65, 30, 0, 0, 0, 0, 0, 63, 60, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 249, 235, 0, 0],
            &[0, 0, 108, 255, 119, 0, 0, 0, 0, 7, 255, 235, 0, 0],
            &[0, 0, 98, 255, 135, 0, 0, 0, 0, 52, 255, 235, 0, 0],
            &[0, 0, 64, 255, 201, 0, 0, 0, 0, 163, 255, 235, 0, 0],
            &[0, 0, 6, 219, 255, 179, 99, 99, 177, 228, 232, 235, 0, 0],
            &[0, 0, 0, 46, 203, 255, 255, 255, 198, 42, 174, 235, 0, 0],
            &[0, 0, 0, 0, 0, 17, 45, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'û' / 0xfb
        'û' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 22, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 242, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 208, 222, 138, 255, 108, 0, 0, 0, 0],
            &[0, 0, 0, 6, 185, 210, 29, 0, 108, 248, 87, 0, 0, 0],
            &[0, 0, 0, 17, 70, 12, 0, 0, 0, 43, 59, 0, 0, 0],
            &[0, 0, 27, 65, 30, 0, 0, 0, 0, 0, 63, 60, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 249, 235, 0, 0],
            &[0, 0, 108, 255, 119, 0, 0, 0, 0, 7, 255, 235, 0, 0],
            &[0, 0, 98, 255, 135, 0, 0, 0, 0, 52, 255, 235, 0, 0],
            &[0, 0, 64, 255, 201, 0, 0, 0, 0, 163, 255, 235, 0, 0],
            &[0, 0, 6, 219, 255, 179, 99, 99, 177, 228, 232, 235, 0, 0],
            &[0, 0, 0, 46, 203, 255, 255, 255, 198, 42, 174, 235, 0, 0],
            &[0, 0, 0, 0, 0, 17, 45, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ü' / 0xfc
        'ü' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 63, 0, 0, 8, 102, 42, 0, 0, 0],
            &[0, 0, 0, 36, 255, 237, 1, 0, 85, 255, 189, 0, 0, 0],
            &[0, 0, 0, 7, 198, 154, 0, 0, 35, 212, 112, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 65, 30, 0, 0, 0, 0, 0, 63, 60, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 249, 235, 0, 0],
            &[0, 0, 108, 255, 119, 0, 0, 0, 0, 7, 255, 235, 0, 0],
            &[0, 0, 98, 255, 135, 0, 0, 0, 0, 52, 255, 235, 0, 0],
            &[0, 0, 64, 255, 201, 0, 0, 0, 0, 163, 255, 235, 0, 0],
            &[0, 0, 6, 219, 255, 179, 99, 99, 177, 228, 232, 235, 0, 0],
            &[0, 0, 0, 46, 203, 255, 255, 255, 198, 42, 174, 235, 0, 0],
            &[0, 0, 0, 0, 0, 17, 45, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ý' / 0xfd
        'ý' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 12, 22, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 7, 222, 255, 90, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 129, 255, 129, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 52, 250, 133, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 64, 55, 0, 0, 0, 0, 0, 0, 22, 65, 38, 0],
            &[0, 0, 189, 252, 28, 0, 0, 0, 0, 0, 140, 255, 90, 0],
            &[0, 0, 83, 255, 129, 0, 0, 0, 0, 1, 225, 241, 9, 0],
            &[0, 0, 4, 228, 229, 4, 0, 0, 0, 56, 255, 156, 0, 0],
            &[0, 0, 0, 126, 255, 81, 0, 0, 0, 141, 255, 61, 0, 0],
            &[0, 0, 0, 24, 250, 185, 0, 0, 1, 226, 221, 0, 0, 0],
            &[0, 0, 0, 0, 169, 254, 35, 0, 57, 255, 127, 0, 0, 0],
            &[0, 0, 0, 0, 62, 255, 135, 0, 141, 254, 33, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 228, 2, 223, 193, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 105, 255, 108, 255, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 242, 241, 245, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 148, 255, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 125, 255, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 224, 224, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 113, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 30, 19, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'þ' / 0xfe
        'þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 78, 245, 134, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 140, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 140, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 140, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 139, 0, 57, 104, 104, 45, 0, 0, 0, 0, 0],
            &[0, 81, 255, 138, 180, 255, 255, 255, 255, 162, 5, 0, 0, 0],
            &[0, 81, 255, 242, 208, 66, 29, 66, 206, 255, 135, 0, 0, 0],
            &[0, 81, 255, 248, 23, 0, 0, 0, 18, 247, 243, 11, 0, 0],
            &[0, 81, 255, 182, 0, 0, 0, 0, 0, 175, 255, 67, 0, 0],
            &[0, 81, 255, 152, 0, 0, 0, 0, 0, 143, 255, 101, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 131, 255, 108, 0, 0],
            &[0, 81, 255, 166, 0, 0, 0, 0, 0, 152, 255, 90, 0, 0],
            &[0, 81, 255, 214, 0, 0, 0, 0, 0, 199, 255, 49, 0, 0],
            &[0, 81, 255, 255, 68, 0, 0, 0, 57, 255, 224, 1, 0, 0],
            &[0, 81, 255, 219, 244, 136, 95, 134, 243, 255, 84, 0, 0, 0],
            &[0, 81, 255, 132, 109, 238, 255, 255, 228, 91, 0, 0, 0, 0],
            &[0, 81, 255, 140, 0, 3, 36, 34, 3, 0, 0, 0, 0, 0],
            &[0, 81, 255, 140, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 255, 140, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 10, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ÿ' / 0xff
        'ÿ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 63, 0, 0, 8, 102, 42, 0, 0, 0],
            &[0, 0, 0, 36, 255, 237, 1, 0, 85, 255, 189, 0, 0, 0],
            &[0, 0, 0, 7, 198, 154, 0, 0, 35, 212, 112, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 64, 55, 0, 0, 0, 0, 0, 0, 22, 65, 38, 0],
            &[0, 0, 189, 252, 28, 0, 0, 0, 0, 0, 140, 255, 90, 0],
            &[0, 0, 83, 255, 129, 0, 0, 0, 0, 1, 225, 241, 9, 0],
            &[0, 0, 4, 228, 229, 4, 0, 0, 0, 56, 255, 156, 0, 0],
            &[0, 0, 0, 126, 255, 81, 0, 0, 0, 141, 255, 61, 0, 0],
            &[0, 0, 0, 24, 250, 185, 0, 0, 1, 226, 221, 0, 0, 0],
            &[0, 0, 0, 0, 169, 254, 35, 0, 57, 255, 127, 0, 0, 0],
            &[0, 0, 0, 0, 62, 255, 135, 0, 141, 254, 33, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 228, 2, 223, 193, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 105, 255, 108, 255, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 242, 241, 245, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 148, 255, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 125, 255, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 224, 224, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 113, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 30, 19, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ā' / 0x100
        'Ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 43, 52, 52, 52, 52, 52, 21, 0, 0, 0, 0],
            &[0, 0, 0, 212, 255, 255, 255, 255, 255, 104, 0, 0, 0, 0],
            &[0, 0, 0, 60, 72, 72, 72, 72, 72, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 14, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 226, 255, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 255, 255, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 137, 255, 186, 255, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 219, 228, 78, 255, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 151, 11, 247, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 71, 0, 179, 253, 25, 0, 0, 0, 0],
            &[0, 0, 0, 212, 241, 6, 0, 100, 255, 106, 0, 0, 0, 0],
            &[0, 0, 39, 255, 168, 0, 0, 23, 253, 190, 0, 0, 0, 0],
            &[0, 0, 122, 255, 160, 71, 71, 71, 245, 252, 20, 0, 0, 0],
            &[0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0],
            &[0, 32, 255, 255, 102, 102, 102, 102, 140, 255, 183, 0, 0, 0],
            &[0, 114, 255, 106, 0, 0, 0, 0, 0, 214, 250, 16, 0, 0],
            &[0, 197, 254, 28, 0, 0, 0, 0, 0, 135, 255, 94, 0, 0],
            &[26, 254, 203, 0, 0, 0, 0, 0, 0, 55, 255, 177, 0, 0],
            &[107, 255, 124, 0, 0, 0, 0, 0, 0, 1, 230, 247, 12, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ā' / 0x101
        'ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 86, 120, 120, 120, 120, 120, 61, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 255, 255, 255, 255, 131, 0, 0, 0],
            &[0, 0, 0, 0, 4, 5, 5, 5, 5, 5, 2, 0, 0, 0],
            &[0, 0, 0, 0, 7, 68, 106, 114, 91, 27, 0, 0, 0, 0],
            &[0, 0, 0, 103, 237, 255, 255, 255, 255, 252, 131, 0, 0, 0],
            &[0, 0, 0, 71, 180, 83, 41, 27, 76, 217, 255, 77, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 156, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 255, 179, 0, 0],
            &[0, 0, 0, 0, 46, 130, 176, 197, 206, 213, 255, 181, 0, 0],
            &[0, 0, 0, 136, 254, 245, 186, 153, 139, 132, 255, 181, 0, 0],
            &[0, 0, 59, 255, 199, 20, 0, 0, 0, 6, 255, 181, 0, 0],
            &[0, 0, 121, 255, 80, 0, 0, 0, 0, 45, 255, 181, 0, 0],
            &[0, 0, 109, 255, 103, 0, 0, 0, 4, 179, 255, 181, 0, 0],
            &[0, 0, 41, 249, 239, 126, 89, 117, 211, 201, 252, 181, 0, 0],
            &[0, 0, 0, 83, 227, 255, 255, 249, 154, 13, 222, 181, 0, 0],
            &[0, 0, 0, 0, 2, 30, 41, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ă' / 0x102
        'Ă' => &[
            &[0, 0, 0, 67, 22, 0, 0, 0, 52, 37, 0, 0, 0, 0],
            &[0, 0, 0, 205, 172, 6, 0, 36, 241, 98, 0, 0, 0, 0],
            &[0, 0, 0, 69, 247, 235, 217, 254, 197, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 109, 125, 82, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 14, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 226, 255, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 255, 255, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 137, 255, 186, 255, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 219, 228, 78, 255, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 151, 11, 247, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 71, 0, 179, 253, 25, 0, 0, 0, 0],
            &[0, 0, 0, 212, 241, 6, 0, 100, 255, 106, 0, 0, 0, 0],
            &[0, 0, 39, 255, 168, 0, 0, 23, 253, 190, 0, 0, 0, 0],
            &[0, 0, 122, 255, 160, 71, 71, 71, 245, 252, 20, 0, 0, 0],
            &[0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0],
            &[0, 32, 255, 255, 102, 102, 102, 102, 140, 255, 183, 0, 0, 0],
            &[0, 114, 255, 106, 0, 0, 0, 0, 0, 214, 250, 16, 0, 0],
            &[0, 197, 254, 28, 0, 0, 0, 0, 0, 135, 255, 94, 0, 0],
            &[26, 254, 203, 0, 0, 0, 0, 0, 0, 55, 255, 177, 0, 0],
            &[107, 255, 124, 0, 0, 0, 0, 0, 0, 1, 230, 247, 12, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ă' / 0x103
        'ă' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 66, 0, 0, 0, 94, 83, 0, 0, 0],
            &[0, 0, 0, 0, 154, 224, 62, 26, 88, 244, 100, 0, 0, 0],
            &[0, 0, 0, 0, 20, 197, 255, 255, 254, 155, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 38, 59, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 68, 106, 114, 91, 27, 0, 0, 0, 0],
            &[0, 0, 0, 103, 237, 255, 255, 255, 255, 252, 131, 0, 0, 0],
            &[0, 0, 0, 71, 180, 83, 41, 27, 76, 217, 255, 77, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 156, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 12, 255, 179, 0, 0],
            &[0, 0, 0, 0, 46, 130, 176, 197, 206, 213, 255, 181, 0, 0],
            &[0, 0, 0, 136, 254, 245, 186, 153, 139, 132, 255, 181, 0, 0],
            &[0, 0, 59, 255, 199, 20, 0, 0, 0, 6, 255, 181, 0, 0],
            &[0, 0, 121, 255, 80, 0, 0, 0, 0, 45, 255, 181, 0, 0],
            &[0, 0, 109, 255, 103, 0, 0, 0, 4, 179, 255, 181, 0, 0],
            &[0, 0, 41, 249, 239, 126, 89, 117, 211, 201, 252, 181, 0, 0],
            &[0, 0, 0, 83, 227, 255, 255, 249, 154, 13, 222, 181, 0, 0],
            &[0, 0, 0, 0, 2, 30, 41, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ą' / 0x104
        'Ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 14, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 226, 255, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 255, 255, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 137, 255, 186, 255, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 219, 228, 78, 255, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 151, 11, 247, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 71, 0, 179, 253, 25, 0, 0, 0, 0],
            &[0, 0, 0, 212, 241, 6, 0, 100, 255, 106, 0, 0, 0, 0],
            &[0, 0, 39, 255, 168, 0, 0, 23, 253, 190, 0, 0, 0, 0],
            &[0, 0, 122, 255, 160, 71, 71, 71, 245, 252, 20, 0, 0, 0],
            &[0, 0, 204, 255, 255, 255, 255, 255, 255, 255, 100, 0, 0, 0],
            &[0, 32, 255, 255, 102, 102, 102, 102, 140, 255, 183, 0, 0, 0],
            &[0, 114, 255, 106, 0, 0, 0, 0, 0, 214, 250, 16, 0, 0],
            &[0, 197, 254, 28, 0, 0, 0, 0, 0, 135, 255, 94, 0, 0],
            &[26, 254, 203, 0, 0, 0, 0, 0, 0, 55, 255, 177, 0, 0],
            &[107, 255, 124, 0, 0, 0, 0, 0, 0, 1, 234, 255, 14, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 175, 185, 4, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 103, 249, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 175, 215, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 115, 152, 92, 0],
        ],
        // letter: 'ą' / 0x105
        'ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 68, 106, 114, 91, 27, 0, 0, 0, 0, 0],
            &[0, 0, 103, 237, 255, 255, 255, 255, 252, 131, 0, 0, 0, 0],
            &[0, 0, 71, 180, 83, 41, 27, 76, 217, 255, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 61, 255, 156, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 12, 255, 179, 0, 0, 0],
            &[0, 0, 0, 46, 130, 176, 197, 206, 213, 255, 181, 0, 0, 0],
            &[0, 0, 136, 254, 245, 186, 153, 139, 132, 255, 181, 0, 0, 0],
            &[0, 59, 255, 199, 20, 0, 0, 0, 6, 255, 181, 0, 0, 0],
            &[0, 121, 255, 80, 0, 0, 0, 0, 45, 255, 181, 0, 0, 0],
            &[0, 109, 255, 103, 0, 0, 0, 4, 179, 255, 181, 0, 0, 0],
            &[0, 41, 249, 239, 126, 89, 117, 211, 201, 252, 181, 0, 0, 0],
            &[0, 0, 83, 227, 255, 255, 249, 154, 13, 255, 195, 0, 0, 0],
            &[0, 0, 0, 2, 30, 41, 9, 0, 79, 238, 50, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 15, 240, 115, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 72, 255, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 70, 146, 140, 12, 0, 0],
        ],
        // letter: 'Ć' / 0x106
        'Ć' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 118, 210, 151, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 35, 246, 224, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 192, 227, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 135, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 45, 41, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 208, 255, 255, 255, 251, 182, 53, 0, 0],
            &[0, 0, 0, 145, 255, 246, 169, 129, 144, 187, 251, 43, 0, 0],
            &[0, 0, 92, 255, 221, 35, 0, 0, 0, 0, 18, 0, 0, 0],
            &[0, 1, 222, 254, 52, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 195, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 113, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 255, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 152, 255, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 137, 255, 98, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 115, 255, 127, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 255, 188, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 233, 254, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 255, 217, 32, 0, 0, 0, 0, 14, 0, 0, 0],
            &[0, 0, 3, 166, 255, 250, 179, 145, 157, 197, 226, 0, 0, 0],
            &[0, 0, 0, 2, 119, 228, 255, 255, 255, 247, 168, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 49, 34, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ć' / 0x107
        'ć' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 21, 22, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 79, 255, 223, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 11, 223, 236, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 152, 233, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 68, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 57, 99, 111, 84, 29, 0, 0, 0],
            &[0, 0, 0, 0, 63, 221, 255, 255, 255, 255, 253, 101, 0, 0],
            &[0, 0, 0, 45, 246, 253, 148, 68, 51, 79, 161, 41, 0, 0],
            &[0, 0, 0, 176, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 248, 243, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 255, 200, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 188, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 31, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 243, 249, 13, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 164, 255, 148, 0, 0, 0, 0, 0, 2, 0, 0],
            &[0, 0, 0, 27, 235, 255, 200, 125, 112, 139, 193, 120, 0, 0],
            &[0, 0, 0, 0, 31, 173, 249, 255, 255, 255, 222, 79, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 38, 44, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĉ' / 0x108
        'Ĉ' => &[
            &[0, 0, 0, 0, 0, 0, 151, 210, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 104, 255, 190, 255, 112, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 250, 121, 0, 120, 252, 85, 0, 0, 0],
            &[0, 0, 0, 0, 126, 79, 0, 0, 0, 75, 131, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 45, 41, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 208, 255, 255, 255, 251, 182, 53, 0, 0],
            &[0, 0, 0, 145, 255, 246, 169, 129, 144, 187, 251, 43, 0, 0],
            &[0, 0, 92, 255, 221, 35, 0, 0, 0, 0, 18, 0, 0, 0],
            &[0, 1, 222, 254, 52, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 195, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 113, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 255, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 152, 255, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 137, 255, 98, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 115, 255, 127, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 255, 188, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 233, 254, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 255, 217, 32, 0, 0, 0, 0, 14, 0, 0, 0],
            &[0, 0, 3, 166, 255, 250, 179, 145, 157, 197, 226, 0, 0, 0],
            &[0, 0, 0, 2, 119, 228, 255, 255, 255, 247, 168, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 49, 34, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĉ' / 0x109
        'ĉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 22, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 126, 255, 252, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 74, 252, 153, 207, 229, 29, 0, 0, 0],
            &[0, 0, 0, 0, 53, 242, 134, 0, 17, 192, 213, 20, 0, 0],
            &[0, 0, 0, 0, 46, 53, 0, 0, 0, 4, 68, 30, 0, 0],
            &[0, 0, 0, 0, 0, 1, 57, 99, 111, 84, 29, 0, 0, 0],
            &[0, 0, 0, 0, 63, 221, 255, 255, 255, 255, 253, 101, 0, 0],
            &[0, 0, 0, 45, 246, 253, 148, 68, 51, 79, 161, 41, 0, 0],
            &[0, 0, 0, 176, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 248, 243, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 255, 200, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 188, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 31, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 243, 249, 13, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 164, 255, 148, 0, 0, 0, 0, 0, 2, 0, 0],
            &[0, 0, 0, 27, 235, 255, 200, 125, 112, 139, 193, 120, 0, 0],
            &[0, 0, 0, 0, 31, 173, 249, 255, 255, 255, 222, 79, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 38, 44, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ċ' / 0x10a
        'Ċ' => &[
            &[0, 0, 0, 0, 0, 0, 59, 156, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 194, 255, 203, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 108, 224, 115, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 45, 41, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 208, 255, 255, 255, 251, 182, 53, 0, 0],
            &[0, 0, 0, 145, 255, 246, 169, 129, 144, 187, 251, 43, 0, 0],
            &[0, 0, 92, 255, 221, 35, 0, 0, 0, 0, 18, 0, 0, 0],
            &[0, 1, 222, 254, 52, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 195, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 113, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 255, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 152, 255, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 137, 255, 98, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 115, 255, 127, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 255, 188, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 233, 254, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 255, 217, 32, 0, 0, 0, 0, 14, 0, 0, 0],
            &[0, 0, 3, 166, 255, 250, 179, 145, 157, 197, 226, 0, 0, 0],
            &[0, 0, 0, 2, 119, 228, 255, 255, 255, 247, 168, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 49, 34, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ċ' / 0x10b
        'ċ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 43, 211, 183, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 105, 255, 255, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 145, 117, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 57, 99, 111, 84, 29, 0, 0, 0],
            &[0, 0, 0, 0, 63, 221, 255, 255, 255, 255, 253, 101, 0, 0],
            &[0, 0, 0, 45, 246, 253, 148, 68, 51, 79, 161, 41, 0, 0],
            &[0, 0, 0, 176, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 248, 243, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 255, 200, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 188, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 31, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 243, 249, 13, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 164, 255, 148, 0, 0, 0, 0, 0, 2, 0, 0],
            &[0, 0, 0, 27, 235, 255, 200, 125, 112, 139, 193, 120, 0, 0],
            &[0, 0, 0, 0, 31, 173, 249, 255, 255, 255, 222, 79, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 38, 44, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Č' / 0x10c
        'Č' => &[
            &[0, 0, 0, 0, 176, 156, 10, 0, 8, 150, 184, 7, 0, 0],
            &[0, 0, 0, 0, 33, 232, 203, 35, 199, 236, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 249, 252, 250, 58, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 84, 137, 89, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 45, 41, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 208, 255, 255, 255, 251, 182, 53, 0, 0],
            &[0, 0, 0, 145, 255, 246, 169, 129, 144, 187, 251, 43, 0, 0],
            &[0, 0, 92, 255, 221, 35, 0, 0, 0, 0, 18, 0, 0, 0],
            &[0, 1, 222, 254, 52, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 255, 195, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 113, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 255, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 152, 255, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 137, 255, 98, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 115, 255, 127, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 255, 188, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 233, 254, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 255, 217, 32, 0, 0, 0, 0, 14, 0, 0, 0],
            &[0, 0, 3, 166, 255, 250, 179, 145, 157, 197, 226, 0, 0, 0],
            &[0, 0, 0, 2, 119, 228, 255, 255, 255, 247, 168, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 49, 34, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'č' / 0x10d
        'č' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 14, 0, 0, 0, 0, 20, 9, 0, 0],
            &[0, 0, 0, 0, 91, 246, 101, 0, 6, 154, 239, 46, 0, 0],
            &[0, 0, 0, 0, 0, 116, 255, 130, 187, 246, 59, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 161, 255, 255, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 70, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 57, 99, 111, 84, 29, 0, 0, 0],
            &[0, 0, 0, 0, 63, 221, 255, 255, 255, 255, 253, 101, 0, 0],
            &[0, 0, 0, 45, 246, 253, 148, 68, 51, 79, 161, 41, 0, 0],
            &[0, 0, 0, 176, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 248, 243, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 255, 200, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 188, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 31, 255, 205, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 243, 249, 13, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 164, 255, 148, 0, 0, 0, 0, 0, 2, 0, 0],
            &[0, 0, 0, 27, 235, 255, 200, 125, 112, 139, 193, 120, 0, 0],
            &[0, 0, 0, 0, 31, 173, 249, 255, 255, 255, 222, 79, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 38, 44, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ď' / 0x10e
        'Ď' => &[
            &[0, 0, 118, 189, 36, 0, 0, 101, 206, 43, 0, 0, 0, 0],
            &[0, 0, 6, 188, 238, 55, 144, 254, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 216, 252, 255, 124, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 46, 137, 124, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 254, 254, 242, 214, 146, 42, 0, 0, 0, 0, 0],
            &[0, 135, 255, 197, 164, 181, 226, 255, 250, 104, 0, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 1, 83, 240, 255, 75, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 69, 255, 225, 2, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 193, 255, 61, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 120, 255, 123, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 89, 255, 147, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 75, 255, 163, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 87, 255, 152, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 127, 255, 119, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 203, 255, 57, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 82, 255, 211, 1, 0, 0],
            &[0, 135, 255, 92, 0, 0, 2, 97, 246, 252, 54, 0, 0, 0],
            &[0, 135, 255, 196, 165, 184, 236, 255, 241, 80, 0, 0, 0, 0],
            &[0, 135, 255, 255, 254, 233, 204, 126, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ď' / 0x10f
        'ď' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 193, 245, 19, 103, 245, 91],
            &[0, 0, 0, 0, 0, 0, 0, 0, 201, 255, 20, 155, 240, 14],
            &[0, 0, 0, 0, 0, 0, 0, 0, 201, 255, 20, 217, 127, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 201, 255, 23, 102, 9, 0],
            &[0, 0, 0, 28, 95, 106, 61, 0, 200, 255, 20, 0, 0, 0],
            &[0, 0, 108, 250, 255, 255, 255, 165, 187, 255, 20, 0, 0, 0],
            &[0, 55, 255, 243, 89, 27, 74, 230, 247, 255, 20, 0, 0, 0],
            &[0, 165, 255, 109, 0, 0, 0, 83, 255, 255, 20, 0, 0, 0],
            &[0, 228, 255, 22, 0, 0, 0, 5, 248, 255, 20, 0, 0, 0],
            &[1, 253, 239, 0, 0, 0, 0, 0, 215, 255, 20, 0, 0, 0],
            &[10, 255, 229, 0, 0, 0, 0, 0, 203, 255, 20, 0, 0, 0],
            &[0, 251, 246, 0, 0, 0, 0, 0, 220, 255, 20, 0, 0, 0],
            &[0, 214, 255, 29, 0, 0, 0, 12, 248, 255, 20, 0, 0, 0],
            &[0, 142, 255, 130, 0, 0, 0, 114, 255, 255, 20, 0, 0, 0],
            &[0, 28, 241, 254, 149, 96, 147, 249, 218, 255, 20, 0, 0, 0],
            &[0, 0, 57, 217, 255, 255, 236, 88, 139, 255, 20, 0, 0, 0],
            &[0, 0, 0, 0, 35, 42, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Đ' / 0x110
        'Đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 254, 254, 254, 247, 220, 166, 64, 0, 0, 0, 0],
            &[0, 0, 81, 255, 216, 164, 176, 212, 255, 255, 148, 4, 0, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 52, 218, 255, 128, 0, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 28, 242, 251, 30, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 139, 255, 115, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 66, 255, 177, 0],
            &[0, 59, 141, 255, 205, 59, 59, 7, 0, 0, 35, 255, 201, 0],
            &[0, 255, 255, 255, 255, 255, 255, 33, 0, 0, 21, 255, 217, 0],
            &[0, 103, 185, 255, 249, 103, 103, 13, 0, 0, 33, 255, 206, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 73, 255, 172, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 149, 255, 111, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 38, 246, 243, 22, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 66, 226, 255, 104, 0, 0],
            &[0, 0, 81, 255, 216, 164, 178, 223, 255, 253, 122, 0, 0, 0],
            &[0, 0, 81, 255, 255, 255, 239, 212, 146, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'đ' / 0x111
        'đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 233, 226, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 17, 17, 17, 255, 253, 17, 10, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 0, 53, 80, 80, 80, 255, 255, 80, 48, 0],
            &[0, 0, 0, 0, 40, 78, 71, 20, 0, 242, 235, 0, 0, 0],
            &[0, 0, 28, 195, 255, 255, 255, 244, 97, 229, 235, 0, 0, 0],
            &[0, 6, 215, 255, 188, 79, 57, 136, 250, 244, 235, 0, 0, 0],
            &[0, 96, 255, 201, 3, 0, 0, 0, 147, 255, 235, 0, 0, 0],
            &[0, 168, 255, 92, 0, 0, 0, 0, 46, 255, 235, 0, 0, 0],
            &[0, 199, 255, 50, 0, 0, 0, 0, 3, 254, 235, 0, 0, 0],
            &[0, 211, 255, 28, 0, 0, 0, 0, 0, 244, 235, 0, 0, 0],
            &[0, 197, 255, 47, 0, 0, 0, 0, 7, 254, 235, 0, 0, 0],
            &[0, 156, 255, 92, 0, 0, 0, 0, 48, 255, 235, 0, 0, 0],
            &[0, 76, 255, 203, 7, 0, 0, 0, 166, 255, 235, 0, 0, 0],
            &[0, 1, 191, 255, 203, 113, 102, 180, 235, 231, 235, 0, 0, 0],
            &[0, 0, 14, 162, 250, 255, 255, 211, 52, 180, 235, 0, 0, 0],
            &[0, 0, 0, 0, 13, 45, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ē' / 0x112
        'Ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 52, 52, 52, 52, 52, 28, 0, 0, 0],
            &[0, 0, 0, 0, 179, 255, 255, 255, 255, 255, 136, 0, 0, 0],
            &[0, 0, 0, 0, 51, 72, 72, 72, 72, 72, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 254, 254, 254, 254, 254, 254, 254, 254, 207, 0, 0],
            &[0, 0, 1, 255, 246, 175, 175, 175, 175, 175, 175, 143, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 172, 172, 172, 172, 172, 172, 71, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 1, 255, 226, 1, 1, 1, 1, 1, 1, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 174, 174, 174, 174, 174, 174, 141, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ē' / 0x113
        'ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 89, 120, 120, 120, 120, 120, 59, 0, 0, 0, 0],
            &[0, 0, 0, 190, 255, 255, 255, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 4, 5, 5, 5, 5, 5, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 85, 111, 91, 25, 0, 0, 0, 0, 0],
            &[0, 0, 2, 131, 248, 255, 255, 255, 250, 127, 1, 0, 0, 0],
            &[0, 0, 139, 255, 202, 70, 27, 55, 192, 255, 106, 0, 0, 0],
            &[0, 38, 253, 219, 11, 0, 0, 0, 14, 236, 232, 2, 0, 0],
            &[0, 125, 255, 109, 0, 0, 0, 0, 0, 166, 255, 50, 0, 0],
            &[
                0, 166, 255, 214, 200, 200, 200, 200, 200, 231, 255, 77, 0, 0,
            ],
            &[
                0, 182, 255, 199, 186, 186, 186, 186, 186, 186, 186, 63, 0, 0,
            ],
            &[0, 161, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 109, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 242, 240, 43, 0, 0, 0, 0, 0, 24, 0, 0, 0],
            &[0, 0, 85, 251, 247, 156, 103, 99, 133, 197, 213, 0, 0, 0],
            &[0, 0, 0, 55, 185, 252, 255, 255, 255, 213, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĕ' / 0x114
        'Ĕ' => &[
            &[0, 0, 0, 0, 58, 31, 0, 0, 0, 43, 46, 0, 0, 0],
            &[0, 0, 0, 0, 173, 198, 12, 0, 22, 223, 130, 0, 0, 0],
            &[0, 0, 0, 0, 44, 239, 240, 214, 252, 218, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 104, 127, 91, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 254, 254, 254, 254, 254, 254, 254, 254, 207, 0, 0],
            &[0, 0, 1, 255, 246, 175, 175, 175, 175, 175, 175, 143, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 172, 172, 172, 172, 172, 172, 71, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 1, 255, 226, 1, 1, 1, 1, 1, 1, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 174, 174, 174, 174, 174, 174, 141, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĕ' / 0x115
        'ĕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 115, 63, 0, 0, 0, 97, 80, 0, 0, 0, 0],
            &[0, 0, 0, 159, 221, 60, 26, 91, 245, 95, 0, 0, 0, 0],
            &[0, 0, 0, 22, 200, 255, 255, 253, 151, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 39, 59, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 85, 111, 91, 25, 0, 0, 0, 0, 0],
            &[0, 0, 2, 131, 248, 255, 255, 255, 250, 127, 1, 0, 0, 0],
            &[0, 0, 139, 255, 202, 70, 27, 55, 192, 255, 106, 0, 0, 0],
            &[0, 38, 253, 219, 11, 0, 0, 0, 14, 236, 232, 2, 0, 0],
            &[0, 125, 255, 109, 0, 0, 0, 0, 0, 166, 255, 50, 0, 0],
            &[
                0, 166, 255, 214, 200, 200, 200, 200, 200, 231, 255, 77, 0, 0,
            ],
            &[
                0, 182, 255, 199, 186, 186, 186, 186, 186, 186, 186, 63, 0, 0,
            ],
            &[0, 161, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 109, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 242, 240, 43, 0, 0, 0, 0, 0, 24, 0, 0, 0],
            &[0, 0, 85, 251, 247, 156, 103, 99, 133, 197, 213, 0, 0, 0],
            &[0, 0, 0, 55, 185, 252, 255, 255, 255, 213, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ė' / 0x116
        'Ė' => &[
            &[0, 0, 0, 0, 0, 0, 75, 155, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 223, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 131, 224, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 254, 254, 254, 254, 254, 254, 254, 254, 207, 0, 0],
            &[0, 0, 1, 255, 246, 175, 175, 175, 175, 175, 175, 143, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 172, 172, 172, 172, 172, 172, 71, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 1, 255, 226, 1, 1, 1, 1, 1, 1, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 174, 174, 174, 174, 174, 174, 141, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ė' / 0x117
        'ė' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 221, 84, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 234, 255, 164, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 156, 44, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 85, 111, 91, 25, 0, 0, 0, 0, 0],
            &[0, 0, 2, 131, 248, 255, 255, 255, 250, 127, 1, 0, 0, 0],
            &[0, 0, 139, 255, 202, 70, 27, 55, 192, 255, 106, 0, 0, 0],
            &[0, 38, 253, 219, 11, 0, 0, 0, 14, 236, 232, 2, 0, 0],
            &[0, 125, 255, 109, 0, 0, 0, 0, 0, 166, 255, 50, 0, 0],
            &[
                0, 166, 255, 214, 200, 200, 200, 200, 200, 231, 255, 77, 0, 0,
            ],
            &[
                0, 182, 255, 199, 186, 186, 186, 186, 186, 186, 186, 63, 0, 0,
            ],
            &[0, 161, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 109, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 242, 240, 43, 0, 0, 0, 0, 0, 24, 0, 0, 0],
            &[0, 0, 85, 251, 247, 156, 103, 99, 133, 197, 213, 0, 0, 0],
            &[0, 0, 0, 55, 185, 252, 255, 255, 255, 213, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ę' / 0x118
        'Ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 254, 254, 254, 254, 254, 254, 254, 254, 207, 0, 0],
            &[0, 0, 1, 255, 246, 175, 175, 175, 175, 175, 175, 143, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 172, 172, 172, 172, 172, 172, 71, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 1, 255, 226, 1, 1, 1, 1, 1, 1, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 174, 174, 174, 174, 174, 174, 141, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 60, 239, 68, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 227, 140, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 47, 255, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 58, 143, 144, 23, 0, 0],
        ],
        // letter: 'ę' / 0x119
        'ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 85, 111, 91, 25, 0, 0, 0, 0, 0],
            &[0, 0, 2, 131, 248, 255, 255, 255, 250, 127, 1, 0, 0, 0],
            &[0, 0, 139, 255, 202, 70, 27, 55, 192, 255, 106, 0, 0, 0],
            &[0, 38, 253, 219, 11, 0, 0, 0, 14, 236, 232, 2, 0, 0],
            &[0, 125, 255, 109, 0, 0, 0, 0, 0, 166, 255, 50, 0, 0],
            &[
                0, 166, 255, 214, 200, 200, 200, 200, 200, 231, 255, 77, 0, 0,
            ],
            &[
                0, 182, 255, 199, 186, 186, 186, 186, 186, 186, 186, 63, 0, 0,
            ],
            &[0, 161, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 109, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 242, 240, 43, 0, 0, 0, 0, 0, 24, 0, 0, 0],
            &[0, 0, 85, 251, 247, 156, 103, 99, 133, 197, 213, 0, 0, 0],
            &[0, 0, 0, 55, 185, 252, 255, 255, 255, 255, 188, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 51, 211, 154, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 140, 232, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 200, 202, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 115, 152, 83, 0, 0, 0],
        ],
        // letter: 'Ě' / 0x11a
        'Ě' => &[
            &[0, 0, 0, 10, 189, 138, 4, 0, 16, 166, 168, 0, 0, 0],
            &[0, 0, 0, 0, 51, 243, 183, 38, 216, 222, 25, 0, 0, 0],
            &[0, 0, 0, 0, 0, 76, 254, 252, 241, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 100, 137, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 254, 254, 254, 254, 254, 254, 254, 254, 207, 0, 0],
            &[0, 0, 1, 255, 246, 175, 175, 175, 175, 175, 175, 143, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 172, 172, 172, 172, 172, 172, 71, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 1, 255, 226, 1, 1, 1, 1, 1, 1, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 255, 246, 174, 174, 174, 174, 174, 174, 141, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ě' / 0x11b
        'ě' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 22, 3, 0, 0, 0, 9, 20, 0, 0, 0, 0],
            &[0, 0, 11, 207, 195, 25, 0, 60, 229, 157, 0, 0, 0, 0],
            &[0, 0, 0, 23, 222, 220, 105, 248, 173, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 246, 255, 209, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 47, 70, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 85, 111, 91, 25, 0, 0, 0, 0, 0],
            &[0, 0, 2, 131, 248, 255, 255, 255, 250, 127, 1, 0, 0, 0],
            &[0, 0, 139, 255, 202, 70, 27, 55, 192, 255, 106, 0, 0, 0],
            &[0, 38, 253, 219, 11, 0, 0, 0, 14, 236, 232, 2, 0, 0],
            &[0, 125, 255, 109, 0, 0, 0, 0, 0, 166, 255, 50, 0, 0],
            &[
                0, 166, 255, 214, 200, 200, 200, 200, 200, 231, 255, 77, 0, 0,
            ],
            &[
                0, 182, 255, 199, 186, 186, 186, 186, 186, 186, 186, 63, 0, 0,
            ],
            &[0, 161, 255, 67, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 109, 255, 130, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 242, 240, 43, 0, 0, 0, 0, 0, 24, 0, 0, 0],
            &[0, 0, 85, 251, 247, 156, 103, 99, 133, 197, 213, 0, 0, 0],
            &[0, 0, 0, 55, 185, 252, 255, 255, 255, 213, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => &[
            &[0, 0, 0, 0, 0, 0, 104, 210, 192, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 55, 247, 197, 249, 168, 1, 0, 0, 0],
            &[0, 0, 0, 0, 35, 232, 171, 5, 74, 244, 139, 0, 0, 0],
            &[0, 0, 0, 0, 96, 107, 1, 0, 0, 44, 137, 31, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 24, 44, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 141, 239, 255, 255, 255, 227, 120, 0, 0],
            &[0, 0, 0, 33, 220, 255, 231, 158, 135, 162, 227, 156, 0, 0],
            &[0, 0, 0, 196, 255, 158, 8, 0, 0, 0, 4, 17, 0, 0],
            &[0, 0, 78, 255, 205, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 158, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 220, 255, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 245, 244, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 255, 229, 0, 0, 0, 100, 234, 234, 234, 234, 30, 0],
            &[0, 1, 252, 238, 0, 0, 0, 83, 195, 195, 237, 255, 33, 0],
            &[0, 0, 235, 253, 12, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 188, 255, 67, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 115, 255, 172, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 15, 233, 255, 112, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 0, 72, 245, 255, 206, 144, 128, 156, 236, 255, 33, 0],
            &[0, 0, 0, 0, 49, 174, 251, 255, 255, 255, 234, 159, 15, 0],
            &[0, 0, 0, 0, 0, 0, 7, 35, 48, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĝ' / 0x11d
        'ĝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 22, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 242, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 208, 222, 138, 255, 108, 0, 0, 0, 0],
            &[0, 0, 0, 6, 185, 210, 29, 0, 108, 248, 87, 0, 0, 0],
            &[0, 0, 0, 17, 70, 12, 0, 0, 0, 43, 59, 0, 0, 0],
            &[0, 0, 0, 0, 5, 71, 111, 101, 36, 0, 40, 60, 0, 0],
            &[0, 0, 0, 34, 213, 255, 255, 255, 252, 117, 186, 235, 0, 0],
            &[0, 0, 6, 213, 255, 155, 51, 29, 100, 243, 250, 235, 0, 0],
            &[0, 0, 91, 255, 181, 0, 0, 0, 0, 94, 255, 235, 0, 0],
            &[0, 0, 164, 255, 82, 0, 0, 0, 0, 7, 247, 235, 0, 0],
            &[0, 0, 201, 255, 44, 0, 0, 0, 0, 0, 214, 235, 0, 0],
            &[0, 0, 212, 255, 29, 0, 0, 0, 0, 0, 201, 235, 0, 0],
            &[0, 0, 198, 255, 52, 0, 0, 0, 0, 0, 220, 235, 0, 0],
            &[0, 0, 155, 255, 103, 0, 0, 0, 0, 11, 249, 235, 0, 0],
            &[0, 0, 72, 255, 214, 13, 0, 0, 0, 123, 255, 235, 0, 0],
            &[0, 0, 0, 180, 255, 213, 119, 97, 160, 252, 242, 235, 0, 0],
            &[0, 0, 0, 10, 153, 248, 255, 255, 238, 94, 212, 235, 0, 0],
            &[0, 0, 0, 0, 0, 11, 44, 40, 4, 0, 224, 229, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 252, 204, 0, 0],
            &[0, 0, 17, 12, 0, 0, 0, 0, 0, 129, 255, 136, 0, 0],
            &[0, 0, 0, 0, 0, 14, 32, 21, 2, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ğ' / 0x11e
        'Ğ' => &[
            &[0, 0, 0, 0, 34, 55, 0, 0, 0, 19, 70, 0, 0, 0],
            &[0, 0, 0, 0, 87, 249, 47, 0, 2, 156, 216, 0, 0, 0],
            &[0, 0, 0, 0, 4, 193, 251, 213, 242, 247, 75, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 84, 126, 107, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 24, 44, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 141, 239, 255, 255, 255, 227, 120, 0, 0],
            &[0, 0, 0, 33, 220, 255, 231, 158, 135, 162, 227, 156, 0, 0],
            &[0, 0, 0, 196, 255, 158, 8, 0, 0, 0, 4, 17, 0, 0],
            &[0, 0, 78, 255, 205, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 158, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 220, 255, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 245, 244, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 255, 229, 0, 0, 0, 100, 234, 234, 234, 234, 30, 0],
            &[0, 1, 252, 238, 0, 0, 0, 83, 195, 195, 237, 255, 33, 0],
            &[0, 0, 235, 253, 12, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 188, 255, 67, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 115, 255, 172, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 15, 233, 255, 112, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 0, 72, 245, 255, 206, 144, 127, 156, 236, 255, 33, 0],
            &[0, 0, 0, 0, 49, 174, 251, 255, 255, 255, 234, 159, 15, 0],
            &[0, 0, 0, 0, 0, 0, 7, 35, 48, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ğ' / 0x11f
        'ğ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 126, 52, 0, 0, 0, 108, 68, 0, 0, 0],
            &[0, 0, 0, 0, 181, 209, 52, 29, 102, 251, 74, 0, 0, 0],
            &[0, 0, 0, 0, 33, 210, 255, 255, 251, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 42, 58, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 71, 111, 101, 36, 0, 40, 60, 0, 0],
            &[0, 0, 0, 34, 213, 255, 255, 255, 252, 117, 186, 235, 0, 0],
            &[0, 0, 6, 213, 255, 155, 51, 29, 100, 243, 250, 235, 0, 0],
            &[0, 0, 91, 255, 181, 0, 0, 0, 0, 94, 255, 235, 0, 0],
            &[0, 0, 164, 255, 82, 0, 0, 0, 0, 7, 247, 235, 0, 0],
            &[0, 0, 201, 255, 44, 0, 0, 0, 0, 0, 214, 235, 0, 0],
            &[0, 0, 212, 255, 29, 0, 0, 0, 0, 0, 201, 235, 0, 0],
            &[0, 0, 198, 255, 52, 0, 0, 0, 0, 0, 220, 235, 0, 0],
            &[0, 0, 155, 255, 103, 0, 0, 0, 0, 11, 249, 235, 0, 0],
            &[0, 0, 72, 255, 214, 13, 0, 0, 0, 123, 255, 235, 0, 0],
            &[0, 0, 0, 180, 255, 213, 119, 97, 160, 252, 242, 235, 0, 0],
            &[0, 0, 0, 10, 153, 248, 255, 255, 238, 94, 212, 235, 0, 0],
            &[0, 0, 0, 0, 0, 11, 44, 40, 4, 0, 224, 229, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 252, 204, 0, 0],
            &[0, 0, 17, 12, 0, 0, 0, 0, 0, 129, 255, 136, 0, 0],
            &[0, 0, 0, 0, 0, 14, 32, 21, 2, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ġ' / 0x120
        'Ġ' => &[
            &[0, 0, 0, 0, 0, 0, 32, 151, 97, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 137, 255, 254, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 65, 219, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 24, 44, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 141, 239, 255, 255, 255, 227, 120, 0, 0],
            &[0, 0, 0, 33, 220, 255, 231, 158, 135, 162, 227, 156, 0, 0],
            &[0, 0, 0, 196, 255, 158, 8, 0, 0, 0, 4, 17, 0, 0],
            &[0, 0, 78, 255, 205, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 158, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 220, 255, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 245, 244, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 255, 229, 0, 0, 0, 100, 234, 234, 234, 234, 30, 0],
            &[0, 1, 252, 238, 0, 0, 0, 83, 195, 195, 237, 255, 33, 0],
            &[0, 0, 235, 253, 12, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 188, 255, 67, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 115, 255, 172, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 15, 233, 255, 112, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 0, 72, 245, 255, 206, 144, 127, 156, 236, 255, 33, 0],
            &[0, 0, 0, 0, 49, 174, 251, 255, 255, 255, 234, 159, 15, 0],
            &[0, 0, 0, 0, 0, 0, 7, 35, 48, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ġ' / 0x121
        'ġ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 158, 219, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 252, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 94, 153, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 71, 111, 101, 36, 0, 40, 60, 0, 0],
            &[0, 0, 0, 34, 213, 255, 255, 255, 252, 117, 186, 235, 0, 0],
            &[0, 0, 6, 213, 255, 155, 51, 29, 100, 243, 250, 235, 0, 0],
            &[0, 0, 91, 255, 181, 0, 0, 0, 0, 94, 255, 235, 0, 0],
            &[0, 0, 164, 255, 82, 0, 0, 0, 0, 7, 247, 235, 0, 0],
            &[0, 0, 201, 255, 44, 0, 0, 0, 0, 0, 214, 235, 0, 0],
            &[0, 0, 212, 255, 29, 0, 0, 0, 0, 0, 201, 235, 0, 0],
            &[0, 0, 198, 255, 52, 0, 0, 0, 0, 0, 220, 235, 0, 0],
            &[0, 0, 155, 255, 103, 0, 0, 0, 0, 11, 249, 235, 0, 0],
            &[0, 0, 72, 255, 214, 13, 0, 0, 0, 123, 255, 235, 0, 0],
            &[0, 0, 0, 180, 255, 213, 119, 97, 160, 252, 242, 235, 0, 0],
            &[0, 0, 0, 10, 153, 248, 255, 255, 238, 94, 212, 235, 0, 0],
            &[0, 0, 0, 0, 0, 11, 44, 40, 4, 0, 224, 229, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 252, 204, 0, 0],
            &[0, 0, 17, 12, 0, 0, 0, 0, 0, 129, 255, 136, 0, 0],
            &[0, 0, 0, 0, 0, 14, 32, 21, 2, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ģ' / 0x122
        'Ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 24, 44, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 141, 239, 255, 255, 255, 227, 120, 0, 0],
            &[0, 0, 0, 33, 220, 255, 231, 158, 135, 162, 227, 156, 0, 0],
            &[0, 0, 0, 196, 255, 158, 8, 0, 0, 0, 4, 17, 0, 0],
            &[0, 0, 78, 255, 205, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 158, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 220, 255, 22, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 245, 244, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 255, 229, 0, 0, 0, 100, 234, 234, 234, 234, 30, 0],
            &[0, 1, 252, 238, 0, 0, 0, 83, 195, 195, 237, 255, 33, 0],
            &[0, 0, 235, 253, 12, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 188, 255, 67, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 115, 255, 172, 0, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 15, 233, 255, 112, 0, 0, 0, 0, 178, 255, 33, 0],
            &[0, 0, 0, 72, 245, 255, 206, 144, 127, 156, 236, 255, 33, 0],
            &[0, 0, 0, 0, 49, 174, 251, 255, 255, 255, 234, 159, 15, 0],
            &[0, 0, 0, 0, 0, 0, 7, 35, 48, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 160, 172, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 253, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 101, 92, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ģ' / 0x123
        'ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 197, 131, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 96, 255, 45, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 213, 234, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 70, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 71, 111, 101, 36, 0, 40, 60, 0, 0],
            &[0, 0, 0, 34, 213, 255, 255, 255, 252, 117, 186, 235, 0, 0],
            &[0, 0, 6, 213, 255, 155, 51, 29, 100, 243, 250, 235, 0, 0],
            &[0, 0, 91, 255, 181, 0, 0, 0, 0, 94, 255, 235, 0, 0],
            &[0, 0, 164, 255, 82, 0, 0, 0, 0, 7, 247, 235, 0, 0],
            &[0, 0, 201, 255, 44, 0, 0, 0, 0, 0, 214, 235, 0, 0],
            &[0, 0, 212, 255, 29, 0, 0, 0, 0, 0, 201, 235, 0, 0],
            &[0, 0, 198, 255, 52, 0, 0, 0, 0, 0, 220, 235, 0, 0],
            &[0, 0, 155, 255, 103, 0, 0, 0, 0, 11, 249, 235, 0, 0],
            &[0, 0, 72, 255, 214, 13, 0, 0, 0, 123, 255, 235, 0, 0],
            &[0, 0, 0, 180, 255, 213, 119, 97, 160, 252, 242, 235, 0, 0],
            &[0, 0, 0, 10, 153, 248, 255, 255, 238, 94, 212, 235, 0, 0],
            &[0, 0, 0, 0, 0, 11, 44, 40, 4, 0, 224, 229, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 252, 204, 0, 0],
            &[0, 0, 17, 12, 0, 0, 0, 0, 0, 129, 255, 136, 0, 0],
            &[0, 0, 0, 0, 0, 14, 32, 21, 2, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĥ' / 0x124
        'Ĥ' => &[
            &[0, 0, 0, 0, 11, 190, 210, 107, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 248, 198, 248, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 130, 244, 71, 6, 174, 235, 41, 0, 0, 0, 0],
            &[0, 0, 23, 137, 46, 0, 0, 1, 106, 105, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 0, 193, 254, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[
                0, 135, 255, 202, 172, 172, 172, 172, 172, 235, 255, 33, 0, 0,
            ],
            &[
                0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 135, 255, 93, 1, 1, 1, 1, 1, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĥ' / 0x125
        'ĥ' => &[
            &[0, 72, 253, 200, 255, 132, 0, 0, 0, 0, 0, 0, 0, 0],
            &[48, 241, 155, 2, 105, 252, 102, 0, 0, 0, 0, 0, 0, 0],
            &[118, 102, 0, 0, 0, 67, 143, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 78, 245, 140, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 145, 0, 35, 94, 103, 59, 0, 0, 0, 0],
            &[0, 0, 81, 255, 137, 130, 251, 255, 255, 255, 191, 19, 0, 0],
            &[0, 0, 81, 255, 212, 231, 83, 27, 60, 205, 255, 156, 0, 0],
            &[0, 0, 81, 255, 254, 54, 0, 0, 0, 39, 255, 233, 0, 0],
            &[0, 0, 81, 255, 202, 0, 0, 0, 0, 0, 237, 255, 3, 0],
            &[0, 0, 81, 255, 159, 0, 0, 0, 0, 0, 221, 255, 6, 0],
            &[0, 0, 81, 255, 146, 0, 0, 0, 0, 0, 221, 255, 6, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 121, 202, 168, 2, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ħ' / 0x126
        'Ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 0, 193, 254, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[
                126, 255, 255, 218, 126, 126, 126, 126, 126, 255, 255, 159, 76, 0,
            ],
            &[
                255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 153, 0,
            ],
            &[4, 139, 255, 96, 4, 4, 4, 4, 4, 198, 255, 37, 2, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[
                0, 135, 255, 202, 172, 172, 172, 172, 172, 235, 255, 33, 0, 0,
            ],
            &[
                0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 135, 255, 93, 1, 1, 1, 1, 1, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ħ' / 0x127
        'ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 78, 245, 140, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 99, 255, 163, 17, 17, 17, 4, 0, 0, 0, 0, 0],
            &[0, 255, 255, 255, 255, 255, 255, 255, 68, 0, 0, 0, 0, 0],
            &[0, 80, 162, 255, 226, 80, 80, 80, 21, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 145, 0, 35, 94, 103, 59, 0, 0, 0, 0],
            &[0, 0, 81, 255, 137, 130, 251, 255, 255, 255, 191, 19, 0, 0],
            &[0, 0, 81, 255, 212, 231, 83, 27, 60, 205, 255, 156, 0, 0],
            &[0, 0, 81, 255, 254, 54, 0, 0, 0, 39, 255, 233, 0, 0],
            &[0, 0, 81, 255, 202, 0, 0, 0, 0, 0, 237, 255, 3, 0],
            &[0, 0, 81, 255, 159, 0, 0, 0, 0, 0, 221, 255, 6, 0],
            &[0, 0, 81, 255, 146, 0, 0, 0, 0, 0, 221, 255, 6, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0],
            &[0, 0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĩ' / 0x128
        'Ĩ' => &[
            &[0, 0, 0, 0, 19, 8, 0, 0, 0, 44, 2, 0, 0, 0],
            &[0, 0, 0, 154, 255, 246, 135, 18, 86, 237, 1, 0, 0, 0],
            &[0, 0, 58, 246, 112, 141, 247, 254, 255, 128, 0, 0, 0, 0],
            &[0, 0, 57, 97, 0, 0, 30, 113, 86, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 175, 254, 254, 254, 254, 254, 254, 254, 73, 0, 0, 0],
            &[0, 0, 59, 106, 129, 218, 255, 165, 119, 96, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 104, 127, 218, 255, 164, 118, 95, 23, 0, 0, 0],
            &[0, 0, 176, 255, 255, 255, 255, 255, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĩ' / 0x129
        'ĩ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 86, 65, 1, 0, 7, 110, 3, 0, 0, 0],
            &[0, 0, 5, 208, 255, 255, 201, 83, 146, 218, 0, 0, 0, 0],
            &[0, 0, 79, 231, 45, 74, 209, 255, 250, 81, 0, 0, 0, 0],
            &[0, 0, 31, 45, 0, 0, 2, 46, 24, 0, 0, 0, 0, 0],
            &[0, 0, 64, 65, 65, 65, 65, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 39, 58, 130, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 22, 83, 103, 123, 175, 255, 208, 127, 105, 84, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ī' / 0x12a
        'Ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 43, 52, 52, 52, 52, 52, 21, 0, 0, 0, 0],
            &[0, 0, 0, 212, 255, 255, 255, 255, 255, 104, 0, 0, 0, 0],
            &[0, 0, 0, 60, 72, 72, 72, 72, 72, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 175, 254, 254, 254, 254, 254, 254, 254, 73, 0, 0, 0],
            &[0, 0, 59, 106, 129, 218, 255, 165, 119, 96, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 104, 127, 218, 255, 164, 118, 95, 23, 0, 0, 0],
            &[0, 0, 176, 255, 255, 255, 255, 255, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ī' / 0x12b
        'ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 99, 120, 120, 120, 120, 120, 48, 0, 0, 0, 0],
            &[0, 0, 0, 212, 255, 255, 255, 255, 255, 104, 0, 0, 0, 0],
            &[0, 0, 0, 4, 5, 5, 5, 5, 5, 2, 0, 0, 0, 0],
            &[0, 0, 64, 65, 65, 65, 65, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 39, 58, 130, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 22, 83, 103, 123, 175, 255, 208, 127, 105, 84, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => &[
            &[0, 0, 0, 67, 22, 0, 0, 0, 52, 37, 0, 0, 0, 0],
            &[0, 0, 0, 205, 172, 6, 0, 36, 241, 98, 0, 0, 0, 0],
            &[0, 0, 0, 69, 247, 235, 217, 254, 197, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 109, 125, 82, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 175, 254, 254, 254, 254, 254, 254, 254, 73, 0, 0, 0],
            &[0, 0, 59, 106, 129, 218, 255, 165, 119, 96, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 104, 127, 218, 255, 164, 118, 95, 23, 0, 0, 0],
            &[0, 0, 176, 255, 255, 255, 255, 255, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĭ' / 0x12d
        'ĭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 126, 52, 0, 0, 0, 108, 68, 0, 0, 0, 0],
            &[0, 0, 0, 181, 209, 52, 29, 102, 251, 74, 0, 0, 0, 0],
            &[0, 0, 0, 33, 210, 255, 255, 251, 133, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 42, 58, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 64, 65, 65, 65, 65, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 39, 58, 130, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 22, 83, 103, 123, 175, 255, 208, 127, 105, 84, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Į' / 0x12e
        'Į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 175, 254, 254, 254, 254, 254, 254, 254, 73, 0, 0, 0],
            &[0, 0, 59, 106, 129, 218, 255, 165, 119, 96, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 104, 127, 218, 255, 164, 118, 95, 23, 0, 0, 0],
            &[0, 0, 176, 255, 255, 255, 255, 255, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 208, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 71, 255, 43, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 144, 243, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 104, 151, 110, 0, 0, 0, 0],
        ],
        // letter: 'į' / 0x12f
        'į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 223, 101, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 212, 255, 185, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 69, 157, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 64, 65, 65, 65, 65, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 39, 58, 130, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 22, 83, 103, 123, 175, 255, 208, 127, 105, 84, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 60, 239, 68, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 227, 140, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 255, 87, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 58, 143, 144, 23, 0, 0, 0],
        ],
        // letter: 'İ' / 0x130
        'İ' => &[
            &[0, 0, 0, 0, 0, 94, 151, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 252, 255, 142, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 158, 220, 69, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 175, 254, 254, 254, 254, 254, 254, 254, 73, 0, 0, 0],
            &[0, 0, 59, 106, 129, 218, 255, 165, 119, 96, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 104, 127, 218, 255, 164, 118, 95, 23, 0, 0, 0],
            &[0, 0, 176, 255, 255, 255, 255, 255, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ı' / 0x131
        'ı' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 64, 65, 65, 65, 65, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 251, 255, 255, 255, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 39, 58, 130, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 22, 83, 103, 123, 175, 255, 208, 127, 105, 84, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĳ' / 0x132
        'Ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 228, 251, 0, 0, 0, 0, 99, 254, 127, 0, 0, 0],
            &[0, 0, 230, 253, 0, 0, 0, 0, 99, 255, 127, 0, 0, 0],
            &[0, 0, 230, 253, 0, 0, 0, 0, 99, 255, 127, 0, 0, 0],
            &[0, 0, 230, 253, 0, 0, 0, 0, 99, 255, 127, 0, 0, 0],
            &[0, 0, 230, 253, 0, 0, 0, 0, 99, 255, 127, 0, 0, 0],
            &[0, 0, 230, 253, 0, 0, 0, 0, 99, 255, 127, 0, 0, 0],
            &[0, 0, 230, 253, 0, 0, 0, 0, 99, 255, 127, 0, 0, 0],
            &[0, 0, 230, 253, 0, 0, 0, 0, 99, 255, 127, 0, 0, 0],
            &[0, 0, 230, 253, 0, 0, 0, 0, 99, 255, 127, 0, 0, 0],
            &[0, 0, 230, 253, 0, 0, 0, 0, 99, 255, 127, 0, 0, 0],
            &[0, 0, 184, 202, 0, 0, 0, 0, 107, 255, 124, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 147, 255, 100, 0, 0, 0],
            &[0, 0, 12, 0, 0, 0, 0, 42, 241, 255, 33, 0, 0, 0],
            &[0, 0, 215, 202, 158, 141, 170, 246, 255, 153, 0, 0, 0, 0],
            &[0, 0, 163, 248, 255, 255, 255, 235, 128, 4, 0, 0, 0, 0],
            &[0, 0, 0, 4, 34, 49, 33, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĳ' / 0x133
        'ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 222, 96, 0, 0, 0, 167, 214, 51, 0, 0, 0],
            &[0, 0, 218, 255, 179, 0, 0, 17, 255, 255, 116, 0, 0, 0],
            &[0, 0, 72, 157, 52, 0, 0, 0, 108, 148, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 35, 65, 22, 0, 0, 0, 43, 65, 14, 0, 0, 0],
            &[0, 0, 138, 255, 88, 0, 0, 0, 169, 255, 58, 0, 0, 0],
            &[0, 0, 138, 255, 88, 0, 0, 0, 169, 255, 58, 0, 0, 0],
            &[0, 0, 138, 255, 88, 0, 0, 0, 169, 255, 58, 0, 0, 0],
            &[0, 0, 138, 255, 88, 0, 0, 0, 169, 255, 58, 0, 0, 0],
            &[0, 0, 138, 255, 88, 0, 0, 0, 169, 255, 58, 0, 0, 0],
            &[0, 0, 138, 255, 88, 0, 0, 0, 169, 255, 58, 0, 0, 0],
            &[0, 0, 138, 255, 88, 0, 0, 0, 169, 255, 58, 0, 0, 0],
            &[0, 0, 138, 255, 88, 0, 0, 0, 169, 255, 58, 0, 0, 0],
            &[0, 0, 138, 255, 88, 0, 0, 0, 169, 255, 58, 0, 0, 0],
            &[0, 0, 138, 255, 88, 0, 0, 0, 169, 255, 58, 0, 0, 0],
            &[0, 0, 138, 255, 88, 0, 0, 0, 169, 255, 58, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 170, 255, 58, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 188, 255, 49, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 30, 246, 248, 11, 0, 0, 0],
            &[0, 0, 0, 0, 15, 32, 22, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĵ' / 0x134
        'Ĵ' => &[
            &[0, 0, 0, 0, 0, 0, 92, 210, 198, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 243, 201, 245, 180, 3, 0, 0, 0],
            &[0, 0, 0, 0, 27, 225, 183, 9, 63, 240, 153, 0, 0, 0],
            &[0, 0, 0, 0, 88, 114, 3, 0, 0, 37, 136, 39, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 254, 194, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 255, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 38, 255, 192, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 73, 255, 170, 0, 0, 0, 0, 0],
            &[0, 13, 0, 0, 0, 5, 186, 255, 109, 0, 0, 0, 0, 0],
            &[0, 225, 195, 157, 146, 214, 255, 226, 14, 0, 0, 0, 0, 0],
            &[0, 169, 248, 255, 255, 254, 189, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 33, 46, 18, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĵ' / 0x135
        'ĵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 18, 22, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 63, 253, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 30, 231, 198, 162, 251, 71, 0, 0, 0],
            &[0, 0, 0, 0, 18, 213, 186, 13, 1, 143, 243, 56, 0, 0],
            &[0, 0, 0, 0, 28, 67, 3, 0, 0, 0, 55, 48, 0, 0],
            &[0, 0, 0, 14, 65, 65, 65, 65, 65, 22, 0, 0, 0, 0],
            &[0, 0, 0, 58, 255, 255, 255, 255, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 3, 25, 44, 63, 176, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 141, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 142, 255, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 172, 255, 78, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 230, 255, 34, 0, 0, 0, 0],
            &[0, 0, 0, 6, 27, 28, 3, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ķ' / 0x136
        'Ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 50, 247, 230, 22, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 14, 220, 253, 67, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 170, 255, 131, 0, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 105, 255, 194, 3, 0, 0, 0, 0],
            &[0, 135, 255, 92, 0, 48, 248, 236, 28, 0, 0, 0, 0, 0],
            &[0, 135, 255, 92, 13, 218, 254, 76, 0, 0, 0, 0, 0, 0],
            &[0, 135, 255, 92, 157, 255, 150, 0, 0, 0, 0, 0, 0, 0],
            &[0, 135, 255, 163, 255, 255, 200, 2, 0, 0, 0, 0, 0, 0],
            &[0, 135, 255, 255, 173, 172, 255, 109, 0, 0, 0, 0, 0, 0],
            &[0, 135, 255, 199, 9, 26, 241, 242, 28, 0, 0, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 108, 255, 178, 0, 0, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 2, 201, 255, 84, 0, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 48, 251, 230, 15, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 140, 255, 153, 0, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 11, 225, 254, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 183, 68, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 157, 245, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 30, 154, 8, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ķ' / 0x137
        'ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 195, 245, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 0, 46, 65, 36, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 74, 251, 223, 26, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 55, 245, 233, 37, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 39, 236, 241, 49, 0, 0, 0, 0],
            &[0, 0, 203, 255, 19, 26, 224, 247, 62, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 22, 208, 255, 109, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 250, 175, 240, 251, 213, 10, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 236, 50, 123, 255, 158, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 56, 0, 2, 190, 255, 93, 0, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 27, 236, 244, 39, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 78, 255, 209, 8, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 0, 147, 255, 152, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 152, 176, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 248, 166, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 95, 98, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĸ' / 0x138
        'ĸ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 65, 6, 0, 0, 0, 0, 46, 65, 36, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 74, 251, 223, 26, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 55, 245, 233, 37, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 39, 236, 241, 49, 0, 0, 0, 0],
            &[0, 0, 203, 255, 19, 26, 224, 247, 62, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 22, 208, 255, 109, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 250, 176, 240, 251, 213, 10, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 236, 50, 123, 255, 158, 0, 0, 0, 0, 0],
            &[0, 0, 203, 255, 56, 0, 2, 190, 255, 93, 0, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 27, 236, 244, 39, 0, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 78, 255, 209, 8, 0, 0],
            &[0, 0, 203, 255, 24, 0, 0, 0, 0, 147, 255, 152, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĺ' / 0x139
        'Ĺ' => &[
            &[0, 0, 0, 0, 151, 210, 117, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 66, 255, 201, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 221, 206, 18, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 129, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 254, 77, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 210, 190, 190, 190, 190, 190, 154, 0, 0, 0],
            &[0, 0, 149, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĺ' / 0x13a
        'ĺ' => &[
            &[0, 0, 0, 0, 0, 0, 81, 255, 191, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 19, 230, 199, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 132, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 47, 245, 245, 245, 245, 245, 168, 0, 0, 0, 0, 0, 0],
            &[0, 17, 99, 117, 134, 171, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 22, 82, 102, 121, 163, 255, 220, 129, 107, 85, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 202, 99, 0, 0, 0, 0],
        ],
        // letter: 'Ļ' / 0x13b
        'Ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 254, 77, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 210, 190, 190, 190, 190, 190, 154, 0, 0, 0],
            &[0, 0, 149, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 182, 145, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 255, 118, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 126, 67, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ļ' / 0x13c
        'ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 47, 245, 245, 245, 245, 245, 168, 0, 0, 0, 0, 0, 0],
            &[0, 17, 99, 117, 134, 171, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 22, 82, 102, 121, 163, 255, 220, 129, 107, 85, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 183, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 75, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 153, 40, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ľ' / 0x13d
        'Ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 254, 77, 0, 0, 0, 81, 254, 118, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 130, 249, 27, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 193, 148, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 87, 16, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 210, 190, 190, 190, 190, 190, 154, 0, 0, 0],
            &[0, 0, 149, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ľ' / 0x13e
        'ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 47, 245, 245, 245, 245, 245, 168, 0, 19, 245, 174, 0, 0],
            &[0, 17, 99, 117, 134, 171, 255, 175, 0, 67, 255, 86, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 129, 211, 2, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 70, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 22, 82, 102, 121, 163, 255, 220, 129, 107, 85, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 254, 77, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 1, 39, 2, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 158, 255, 165, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 187, 255, 194, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 43, 134, 46, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 149, 255, 210, 190, 190, 190, 190, 190, 154, 0, 0, 0],
            &[0, 0, 149, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŀ' / 0x140
        'ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[47, 245, 245, 245, 245, 245, 168, 0, 0, 0, 0, 0, 0, 0],
            &[17, 99, 117, 134, 171, 255, 175, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 2, 0, 0],
            &[0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 106, 245, 151, 0],
            &[0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 169, 255, 220, 0],
            &[0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 60, 179, 92, 0],
            &[0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0, 0],
            &[22, 82, 102, 121, 163, 255, 220, 129, 107, 85, 23, 0, 0, 0],
            &[81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ł' / 0x141
        'Ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 189, 254, 36, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 190, 255, 37, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 190, 255, 37, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 190, 255, 37, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 190, 255, 37, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 190, 255, 37, 8, 94, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 190, 255, 117, 223, 255, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 230, 255, 255, 211, 66, 0, 0, 0, 0, 0, 0],
            &[0, 13, 138, 255, 255, 144, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 166, 252, 255, 255, 37, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 37, 44, 190, 255, 37, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 190, 255, 37, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 190, 255, 37, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 190, 255, 199, 190, 190, 190, 190, 190, 124, 0, 0],
            &[0, 0, 0, 190, 255, 255, 255, 255, 255, 255, 255, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ł' / 0x142
        'ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 47, 245, 245, 245, 245, 245, 168, 0, 0, 0, 0, 0, 0],
            &[0, 17, 99, 117, 134, 171, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 194, 156, 227, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 255, 247, 124, 6, 0, 0, 0],
            &[0, 0, 0, 0, 40, 243, 255, 255, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 125, 249, 255, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 135, 147, 67, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 175, 0, 0, 0, 0, 0, 0],
            &[0, 22, 82, 102, 121, 163, 255, 220, 129, 107, 85, 23, 0, 0],
            &[0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ń' / 0x143
        'Ń' => &[
            &[0, 0, 0, 0, 0, 0, 0, 146, 210, 122, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 254, 205, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 217, 210, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 55, 130, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 254, 47, 0, 0, 0, 0, 177, 254, 33, 0, 0],
            &[0, 135, 255, 254, 159, 0, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 182, 248, 22, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 91, 255, 126, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 57, 196, 231, 6, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 69, 84, 255, 94, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 75, 3, 224, 206, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 115, 255, 62, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 15, 243, 174, 0, 177, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 146, 252, 33, 173, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 36, 253, 141, 164, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 177, 240, 163, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 65, 255, 236, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 0, 208, 255, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 0, 96, 255, 255, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ń' / 0x144
        'ń' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 7, 22, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 165, 255, 155, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 254, 187, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 223, 186, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 69, 5, 0, 0, 0, 0, 0, 0],
            &[0, 20, 65, 13, 0, 41, 94, 106, 66, 2, 0, 0, 0, 0],
            &[0, 81, 255, 82, 157, 254, 255, 255, 255, 202, 25, 0, 0, 0],
            &[0, 81, 255, 224, 223, 78, 30, 50, 186, 255, 166, 0, 0, 0],
            &[0, 81, 255, 254, 38, 0, 0, 0, 23, 254, 238, 0, 0, 0],
            &[0, 81, 255, 198, 0, 0, 0, 0, 0, 231, 255, 4, 0, 0],
            &[0, 81, 255, 164, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 146, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ņ' / 0x145
        'Ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 254, 47, 0, 0, 0, 0, 177, 254, 33, 0, 0],
            &[0, 135, 255, 254, 159, 0, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 182, 248, 22, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 91, 255, 126, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 57, 196, 231, 6, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 69, 84, 255, 94, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 75, 3, 224, 206, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 115, 255, 62, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 15, 243, 174, 0, 177, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 146, 252, 33, 173, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 36, 253, 141, 164, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 177, 240, 163, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 65, 255, 236, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 0, 208, 255, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 0, 96, 255, 255, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 119, 183, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 212, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 128, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ņ' / 0x146
        'ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 65, 13, 0, 41, 94, 106, 66, 2, 0, 0, 0, 0],
            &[0, 81, 255, 82, 157, 254, 255, 255, 255, 202, 25, 0, 0, 0],
            &[0, 81, 255, 224, 223, 78, 30, 50, 186, 255, 166, 0, 0, 0],
            &[0, 81, 255, 254, 38, 0, 0, 0, 23, 254, 238, 0, 0, 0],
            &[0, 81, 255, 198, 0, 0, 0, 0, 0, 231, 255, 4, 0, 0],
            &[0, 81, 255, 164, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 146, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 95, 183, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 179, 235, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 146, 3, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ň' / 0x147
        'Ň' => &[
            &[0, 0, 12, 192, 134, 3, 0, 18, 169, 163, 0, 0, 0, 0],
            &[0, 0, 0, 54, 245, 179, 39, 218, 219, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 255, 252, 239, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 137, 70, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 254, 47, 0, 0, 0, 0, 177, 254, 33, 0, 0],
            &[0, 135, 255, 254, 159, 0, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 182, 248, 22, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 91, 255, 126, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 57, 196, 231, 6, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 69, 84, 255, 94, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 75, 3, 224, 206, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 115, 255, 62, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 15, 243, 174, 0, 177, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 146, 252, 33, 173, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 36, 253, 141, 164, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 177, 240, 163, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 65, 255, 236, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 0, 208, 255, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 0, 96, 255, 255, 33, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ň' / 0x148
        'ň' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 7, 0, 0, 0, 5, 22, 2, 0, 0, 0],
            &[0, 0, 0, 176, 218, 44, 0, 37, 210, 193, 6, 0, 0, 0],
            &[0, 0, 0, 7, 195, 239, 99, 234, 206, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 228, 255, 235, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 35, 70, 40, 0, 0, 0, 0, 0, 0],
            &[0, 20, 65, 13, 0, 41, 94, 106, 66, 2, 0, 0, 0, 0],
            &[0, 81, 255, 82, 157, 254, 255, 255, 255, 202, 25, 0, 0, 0],
            &[0, 81, 255, 224, 223, 78, 30, 50, 186, 255, 166, 0, 0, 0],
            &[0, 81, 255, 254, 38, 0, 0, 0, 23, 254, 238, 0, 0, 0],
            &[0, 81, 255, 198, 0, 0, 0, 0, 0, 231, 255, 4, 0, 0],
            &[0, 81, 255, 164, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 146, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŉ' / 0x149
        'ŉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 254, 254, 46, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 72, 255, 229, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 121, 255, 142, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 174, 255, 46, 65, 34, 0, 25, 89, 108, 70, 3, 0, 0],
            &[0, 230, 194, 1, 255, 161, 95, 246, 255, 255, 255, 207, 29, 0],
            &[1, 38, 19, 1, 255, 231, 241, 105, 34, 49, 190, 255, 169, 0],
            &[0, 0, 0, 1, 255, 255, 113, 0, 0, 0, 26, 254, 239, 0],
            &[0, 0, 0, 1, 255, 255, 21, 0, 0, 0, 0, 231, 255, 4],
            &[0, 0, 0, 1, 255, 244, 0, 0, 0, 0, 0, 221, 255, 6],
            &[0, 0, 0, 1, 255, 227, 0, 0, 0, 0, 0, 221, 255, 6],
            &[0, 0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 221, 255, 6],
            &[0, 0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 221, 255, 6],
            &[0, 0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 221, 255, 6],
            &[0, 0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 221, 255, 6],
            &[0, 0, 0, 1, 255, 226, 0, 0, 0, 0, 0, 221, 255, 6],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 254, 47, 0, 0, 0, 0, 177, 254, 33, 0, 0],
            &[0, 135, 255, 254, 159, 0, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 182, 248, 22, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 91, 255, 126, 0, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 57, 196, 231, 6, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 69, 84, 255, 94, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 75, 3, 224, 206, 0, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 115, 255, 62, 0, 178, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 15, 243, 174, 0, 177, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 146, 252, 33, 173, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 36, 253, 141, 164, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 177, 240, 163, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 65, 255, 236, 255, 33, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 0, 208, 255, 255, 57, 0, 0],
            &[0, 135, 255, 76, 0, 0, 0, 0, 96, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 181, 255, 21, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 50, 245, 229, 0, 0, 0],
            &[0, 0, 0, 0, 0, 88, 185, 191, 246, 255, 108, 0, 0, 0],
            &[0, 0, 0, 0, 0, 82, 237, 239, 198, 81, 0, 0, 0, 0],
        ],
        // letter: 'ŋ' / 0x14b
        'ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 65, 13, 0, 41, 94, 106, 66, 2, 0, 0, 0, 0],
            &[0, 81, 255, 82, 157, 254, 255, 255, 255, 202, 25, 0, 0, 0],
            &[0, 81, 255, 224, 223, 78, 30, 50, 186, 255, 166, 0, 0, 0],
            &[0, 81, 255, 254, 38, 0, 0, 0, 23, 254, 238, 0, 0, 0],
            &[0, 81, 255, 198, 0, 0, 0, 0, 0, 231, 255, 4, 0, 0],
            &[0, 81, 255, 164, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 81, 255, 146, 0, 0, 0, 0, 0, 255, 255, 7, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 255, 255, 12, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 255, 255, 12, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 255, 255, 12, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 255, 255, 12, 0, 0],
            &[0, 81, 255, 145, 0, 0, 0, 0, 0, 255, 255, 12, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 221, 255, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 222, 255, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 7, 244, 241, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ō' / 0x14c
        'Ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 52, 52, 52, 52, 52, 21, 0, 0, 0],
            &[0, 0, 0, 0, 212, 255, 255, 255, 255, 255, 104, 0, 0, 0],
            &[0, 0, 0, 0, 60, 72, 72, 72, 72, 72, 29, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 33, 54, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 231, 255, 255, 255, 195, 42, 0, 0, 0],
            &[0, 0, 0, 110, 255, 241, 147, 119, 169, 254, 237, 34, 0, 0],
            &[0, 0, 23, 247, 239, 35, 0, 0, 0, 102, 255, 176, 0, 0],
            &[0, 0, 124, 255, 124, 0, 0, 0, 0, 0, 218, 254, 25, 0],
            &[0, 0, 187, 255, 44, 0, 0, 0, 0, 0, 145, 255, 92, 0],
            &[0, 0, 234, 251, 5, 0, 0, 0, 0, 0, 102, 255, 134, 0],
            &[0, 1, 252, 236, 0, 0, 0, 0, 0, 0, 79, 255, 155, 0],
            &[0, 10, 255, 227, 0, 0, 0, 0, 0, 0, 73, 255, 164, 0],
            &[0, 2, 255, 236, 0, 0, 0, 0, 0, 0, 82, 255, 155, 0],
            &[0, 0, 236, 252, 6, 0, 0, 0, 0, 0, 103, 255, 132, 0],
            &[0, 0, 195, 255, 46, 0, 0, 0, 0, 0, 146, 255, 89, 0],
            &[0, 0, 126, 255, 127, 0, 0, 0, 0, 1, 224, 253, 22, 0],
            &[0, 0, 29, 248, 241, 39, 0, 0, 0, 119, 255, 170, 0, 0],
            &[0, 0, 0, 114, 255, 244, 155, 131, 186, 255, 232, 29, 0, 0],
            &[0, 0, 0, 0, 98, 226, 255, 255, 253, 182, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 32, 45, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ō' / 0x14d
        'ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 99, 120, 120, 120, 120, 120, 48, 0, 0, 0, 0],
            &[0, 0, 0, 212, 255, 255, 255, 255, 255, 104, 0, 0, 0, 0],
            &[0, 0, 0, 4, 5, 5, 5, 5, 5, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 96, 110, 78, 9, 0, 0, 0, 0, 0],
            &[0, 0, 18, 178, 255, 255, 255, 255, 240, 94, 0, 0, 0, 0],
            &[0, 2, 193, 255, 166, 57, 27, 80, 222, 255, 84, 0, 0, 0],
            &[0, 81, 255, 183, 0, 0, 0, 0, 37, 250, 227, 2, 0, 0],
            &[0, 163, 255, 73, 0, 0, 0, 0, 0, 181, 255, 57, 0, 0],
            &[0, 196, 255, 30, 0, 0, 0, 0, 0, 135, 255, 92, 0, 0],
            &[0, 210, 255, 11, 0, 0, 0, 0, 0, 124, 255, 108, 0, 0],
            &[0, 193, 255, 32, 0, 0, 0, 0, 0, 144, 255, 92, 0, 0],
            &[0, 142, 255, 94, 0, 0, 0, 0, 0, 205, 255, 42, 0, 0],
            &[0, 49, 255, 215, 14, 0, 0, 0, 81, 255, 205, 0, 0, 0],
            &[0, 0, 140, 255, 216, 113, 92, 145, 248, 247, 53, 0, 0, 0],
            &[0, 0, 0, 109, 229, 255, 255, 254, 196, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 34, 47, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => &[
            &[0, 0, 0, 0, 67, 22, 0, 0, 0, 52, 37, 0, 0, 0],
            &[0, 0, 0, 0, 205, 172, 6, 0, 36, 241, 98, 0, 0, 0],
            &[0, 0, 0, 0, 69, 247, 235, 217, 254, 197, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 109, 125, 82, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 33, 54, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 231, 255, 255, 255, 195, 42, 0, 0, 0],
            &[0, 0, 0, 110, 255, 241, 147, 119, 169, 254, 237, 34, 0, 0],
            &[0, 0, 23, 247, 239, 35, 0, 0, 0, 102, 255, 176, 0, 0],
            &[0, 0, 124, 255, 124, 0, 0, 0, 0, 0, 218, 254, 25, 0],
            &[0, 0, 187, 255, 44, 0, 0, 0, 0, 0, 145, 255, 92, 0],
            &[0, 0, 234, 251, 5, 0, 0, 0, 0, 0, 102, 255, 134, 0],
            &[0, 1, 252, 236, 0, 0, 0, 0, 0, 0, 79, 255, 155, 0],
            &[0, 10, 255, 227, 0, 0, 0, 0, 0, 0, 73, 255, 164, 0],
            &[0, 2, 255, 236, 0, 0, 0, 0, 0, 0, 82, 255, 155, 0],
            &[0, 0, 236, 252, 6, 0, 0, 0, 0, 0, 103, 255, 132, 0],
            &[0, 0, 195, 255, 46, 0, 0, 0, 0, 0, 146, 255, 89, 0],
            &[0, 0, 126, 255, 127, 0, 0, 0, 0, 1, 224, 253, 22, 0],
            &[0, 0, 29, 248, 241, 39, 0, 0, 0, 119, 255, 170, 0, 0],
            &[0, 0, 0, 114, 255, 244, 155, 131, 186, 255, 232, 29, 0, 0],
            &[0, 0, 0, 0, 98, 226, 255, 255, 253, 182, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 32, 45, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŏ' / 0x14f
        'ŏ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 126, 52, 0, 0, 0, 108, 68, 0, 0, 0, 0],
            &[0, 0, 0, 181, 209, 52, 29, 102, 251, 74, 0, 0, 0, 0],
            &[0, 0, 0, 33, 210, 255, 255, 251, 133, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 42, 58, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 96, 110, 78, 9, 0, 0, 0, 0, 0],
            &[0, 0, 18, 178, 255, 255, 255, 255, 240, 94, 0, 0, 0, 0],
            &[0, 2, 193, 255, 166, 57, 27, 80, 222, 255, 84, 0, 0, 0],
            &[0, 81, 255, 183, 0, 0, 0, 0, 37, 250, 227, 2, 0, 0],
            &[0, 163, 255, 73, 0, 0, 0, 0, 0, 181, 255, 57, 0, 0],
            &[0, 196, 255, 30, 0, 0, 0, 0, 0, 135, 255, 92, 0, 0],
            &[0, 210, 255, 11, 0, 0, 0, 0, 0, 124, 255, 108, 0, 0],
            &[0, 193, 255, 32, 0, 0, 0, 0, 0, 144, 255, 92, 0, 0],
            &[0, 142, 255, 94, 0, 0, 0, 0, 0, 205, 255, 42, 0, 0],
            &[0, 49, 255, 215, 14, 0, 0, 0, 81, 255, 205, 0, 0, 0],
            &[0, 0, 140, 255, 216, 113, 92, 145, 248, 247, 53, 0, 0, 0],
            &[0, 0, 0, 109, 229, 255, 255, 254, 196, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 34, 47, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ő' / 0x150
        'Ő' => &[
            &[0, 0, 0, 0, 0, 94, 210, 130, 0, 157, 210, 66, 0, 0],
            &[0, 0, 0, 0, 12, 229, 222, 20, 62, 255, 165, 0, 0, 0],
            &[0, 0, 0, 0, 138, 233, 40, 5, 210, 189, 7, 0, 0, 0],
            &[0, 0, 0, 4, 133, 38, 0, 43, 125, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 33, 54, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 231, 255, 255, 255, 195, 42, 0, 0, 0],
            &[0, 0, 0, 110, 255, 241, 147, 119, 169, 254, 237, 34, 0, 0],
            &[0, 0, 23, 247, 239, 35, 0, 0, 0, 102, 255, 176, 0, 0],
            &[0, 0, 124, 255, 124, 0, 0, 0, 0, 0, 218, 254, 25, 0],
            &[0, 0, 187, 255, 44, 0, 0, 0, 0, 0, 145, 255, 92, 0],
            &[0, 0, 234, 251, 5, 0, 0, 0, 0, 0, 102, 255, 134, 0],
            &[0, 1, 252, 236, 0, 0, 0, 0, 0, 0, 79, 255, 155, 0],
            &[0, 10, 255, 227, 0, 0, 0, 0, 0, 0, 73, 255, 164, 0],
            &[0, 2, 255, 236, 0, 0, 0, 0, 0, 0, 82, 255, 155, 0],
            &[0, 0, 236, 252, 6, 0, 0, 0, 0, 0, 103, 255, 132, 0],
            &[0, 0, 195, 255, 46, 0, 0, 0, 0, 0, 146, 255, 89, 0],
            &[0, 0, 126, 255, 127, 0, 0, 0, 0, 1, 224, 253, 22, 0],
            &[0, 0, 29, 248, 241, 39, 0, 0, 0, 119, 255, 170, 0, 0],
            &[0, 0, 0, 114, 255, 244, 155, 131, 186, 255, 232, 29, 0, 0],
            &[0, 0, 0, 0, 98, 226, 255, 255, 253, 182, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 32, 45, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ő' / 0x151
        'ő' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 22, 17, 0, 12, 22, 10, 0, 0, 0],
            &[0, 0, 0, 0, 138, 255, 129, 2, 212, 251, 56, 0, 0, 0],
            &[0, 0, 0, 32, 247, 183, 4, 100, 255, 109, 0, 0, 0, 0],
            &[0, 0, 0, 181, 195, 13, 23, 236, 130, 0, 0, 0, 0, 0],
            &[0, 0, 4, 70, 9, 0, 25, 58, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 96, 110, 78, 9, 0, 0, 0, 0, 0],
            &[0, 0, 18, 178, 255, 255, 255, 255, 240, 94, 0, 0, 0, 0],
            &[0, 2, 193, 255, 166, 57, 27, 80, 222, 255, 84, 0, 0, 0],
            &[0, 81, 255, 183, 0, 0, 0, 0, 37, 250, 227, 2, 0, 0],
            &[0, 163, 255, 73, 0, 0, 0, 0, 0, 181, 255, 57, 0, 0],
            &[0, 196, 255, 30, 0, 0, 0, 0, 0, 135, 255, 92, 0, 0],
            &[0, 210, 255, 11, 0, 0, 0, 0, 0, 124, 255, 108, 0, 0],
            &[0, 193, 255, 32, 0, 0, 0, 0, 0, 144, 255, 92, 0, 0],
            &[0, 142, 255, 94, 0, 0, 0, 0, 0, 205, 255, 42, 0, 0],
            &[0, 49, 255, 215, 14, 0, 0, 0, 81, 255, 205, 0, 0, 0],
            &[0, 0, 140, 255, 216, 113, 92, 145, 248, 247, 53, 0, 0, 0],
            &[0, 0, 0, 109, 229, 255, 255, 254, 196, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 34, 47, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Œ' / 0x152
        'Œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 46, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 190, 255, 255, 255, 254, 254, 254, 254, 246, 0],
            &[0, 0, 6, 210, 255, 165, 127, 215, 255, 202, 175, 175, 170, 0],
            &[0, 0, 95, 255, 173, 0, 0, 125, 255, 86, 0, 0, 0, 0],
            &[0, 0, 181, 255, 59, 0, 0, 125, 255, 86, 0, 0, 0, 0],
            &[0, 0, 237, 250, 8, 0, 0, 125, 255, 86, 0, 0, 0, 0],
            &[0, 14, 255, 225, 0, 0, 0, 125, 255, 86, 0, 0, 0, 0],
            &[0, 31, 255, 211, 0, 0, 0, 125, 255, 200, 172, 172, 98, 0],
            &[0, 38, 255, 200, 0, 0, 0, 125, 255, 253, 252, 252, 144, 0],
            &[0, 27, 255, 206, 0, 0, 0, 125, 255, 86, 0, 0, 0, 0],
            &[0, 13, 255, 222, 0, 0, 0, 125, 255, 86, 0, 0, 0, 0],
            &[0, 0, 231, 250, 3, 0, 0, 125, 255, 86, 0, 0, 0, 0],
            &[0, 0, 183, 255, 55, 0, 0, 125, 255, 86, 0, 0, 0, 0],
            &[0, 0, 98, 255, 162, 0, 0, 125, 255, 86, 0, 0, 0, 0],
            &[0, 0, 9, 217, 255, 172, 133, 217, 255, 201, 174, 174, 168, 0],
            &[0, 0, 0, 40, 208, 255, 255, 255, 255, 255, 255, 255, 248, 0],
            &[0, 0, 0, 0, 0, 29, 43, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'œ' / 0x153
        'œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 109, 104, 20, 0, 2, 80, 103, 29, 0, 0, 0],
            &[0, 66, 245, 255, 255, 229, 27, 178, 255, 252, 242, 43, 0, 0],
            &[3, 222, 236, 49, 37, 228, 215, 253, 68, 24, 239, 175, 0, 0],
            &[58, 255, 127, 0, 0, 124, 255, 195, 0, 0, 156, 245, 3, 0],
            &[106, 255, 69, 0, 0, 75, 255, 148, 0, 0, 119, 255, 30, 0],
            &[129, 255, 49, 0, 0, 54, 255, 213, 168, 168, 204, 255, 44, 0],
            &[133, 255, 40, 0, 0, 49, 255, 214, 175, 175, 175, 175, 31, 0],
            &[119, 255, 54, 0, 0, 56, 255, 133, 0, 0, 0, 0, 0, 0],
            &[84, 255, 84, 0, 0, 82, 255, 163, 0, 0, 0, 0, 0, 0],
            &[26, 252, 157, 0, 0, 141, 255, 228, 4, 0, 0, 0, 0, 0],
            &[0, 169, 253, 120, 99, 244, 164, 252, 166, 67, 102, 152, 0, 0],
            &[0, 18, 194, 255, 255, 181, 8, 106, 247, 255, 249, 131, 0, 0],
            &[0, 0, 0, 32, 34, 0, 0, 0, 15, 44, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŕ' / 0x154
        'Ŕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 168, 210, 100, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 86, 255, 185, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 233, 192, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 69, 123, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 254, 254, 254, 252, 235, 197, 116, 8, 0, 0, 0, 0],
            &[0, 1, 255, 245, 164, 169, 192, 246, 255, 203, 11, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 19, 203, 255, 114, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 82, 255, 177, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 53, 255, 190, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 94, 255, 154, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 46, 224, 253, 53, 0, 0, 0],
            &[0, 1, 255, 249, 197, 199, 222, 255, 238, 92, 0, 0, 0, 0],
            &[0, 1, 255, 252, 222, 222, 246, 254, 44, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 99, 255, 165, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 2, 203, 255, 63, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 55, 254, 212, 4, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 159, 255, 114, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 23, 241, 242, 25, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 114, 255, 165, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŕ' / 0x155
        'ŕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 16, 22, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 244, 249, 56, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 170, 253, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 88, 251, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 50, 45, 0, 0, 0, 0, 0, 0],
            &[0, 23, 65, 65, 65, 65, 16, 0, 53, 109, 99, 44, 0, 0],
            &[0, 92, 255, 255, 255, 255, 84, 126, 255, 255, 255, 219, 0, 0],
            &[0, 5, 31, 53, 123, 255, 177, 253, 141, 61, 79, 88, 0, 0],
            &[0, 0, 0, 0, 70, 255, 255, 127, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 241, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 184, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 164, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 162, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 162, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 162, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 94, 118, 170, 255, 214, 125, 101, 70, 0, 0, 0, 0],
            &[0, 178, 255, 255, 255, 255, 255, 255, 255, 233, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŗ' / 0x156
        'Ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 254, 254, 254, 252, 235, 197, 116, 8, 0, 0, 0, 0],
            &[0, 1, 255, 245, 164, 169, 192, 246, 255, 203, 11, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 19, 203, 255, 114, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 82, 255, 177, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 53, 255, 190, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 94, 255, 154, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 46, 224, 253, 53, 0, 0, 0],
            &[0, 1, 255, 249, 197, 199, 222, 255, 238, 92, 0, 0, 0, 0],
            &[0, 1, 255, 252, 222, 222, 246, 254, 44, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 99, 255, 165, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 2, 203, 255, 63, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 55, 254, 212, 4, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 159, 255, 114, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 23, 241, 242, 25, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 114, 255, 165, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 182, 145, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 255, 118, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 126, 67, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŗ' / 0x157
        'ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 23, 65, 65, 65, 65, 16, 0, 53, 109, 99, 44, 0, 0],
            &[0, 92, 255, 255, 255, 255, 84, 126, 255, 255, 255, 219, 0, 0],
            &[0, 5, 31, 53, 123, 255, 177, 253, 141, 61, 79, 88, 0, 0],
            &[0, 0, 0, 0, 70, 255, 255, 127, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 241, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 184, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 164, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 162, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 162, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 162, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 94, 118, 170, 255, 214, 125, 101, 70, 0, 0, 0, 0],
            &[0, 178, 255, 255, 255, 255, 255, 255, 255, 233, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 183, 102, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 255, 57, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 160, 29, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ř' / 0x158
        'Ř' => &[
            &[0, 0, 25, 201, 115, 0, 0, 28, 181, 141, 0, 0, 0, 0],
            &[0, 0, 0, 75, 251, 157, 49, 231, 203, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 255, 252, 226, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 137, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 254, 254, 254, 252, 235, 197, 116, 8, 0, 0, 0, 0],
            &[0, 1, 255, 245, 164, 169, 192, 246, 255, 203, 11, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 19, 203, 255, 114, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 82, 255, 177, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 53, 255, 190, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 94, 255, 154, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 46, 224, 253, 53, 0, 0, 0],
            &[0, 1, 255, 249, 197, 199, 222, 255, 238, 92, 0, 0, 0, 0],
            &[0, 1, 255, 252, 222, 222, 246, 254, 44, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 99, 255, 165, 0, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 2, 203, 255, 63, 0, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 55, 254, 212, 4, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 159, 255, 114, 0, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 23, 241, 242, 25, 0, 0],
            &[0, 1, 255, 226, 0, 0, 0, 0, 0, 114, 255, 165, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ř' / 0x159
        'ř' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 19, 0, 0, 0, 0, 14, 15, 0, 0, 0],
            &[0, 0, 0, 42, 239, 151, 5, 0, 104, 247, 95, 0, 0, 0],
            &[0, 0, 0, 0, 60, 247, 180, 137, 255, 114, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 96, 255, 255, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 64, 70, 11, 0, 0, 0, 0, 0],
            &[0, 23, 65, 65, 65, 65, 16, 0, 53, 109, 99, 44, 0, 0],
            &[0, 92, 255, 255, 255, 255, 84, 126, 255, 255, 255, 219, 0, 0],
            &[0, 5, 31, 53, 123, 255, 177, 253, 141, 61, 79, 88, 0, 0],
            &[0, 0, 0, 0, 70, 255, 255, 127, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 241, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 184, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 164, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 162, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 162, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 255, 162, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 94, 118, 170, 255, 214, 125, 101, 70, 0, 0, 0, 0],
            &[0, 178, 255, 255, 255, 255, 255, 255, 255, 233, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ś' / 0x15a
        'Ś' => &[
            &[0, 0, 0, 0, 0, 0, 0, 84, 210, 181, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 227, 242, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 153, 242, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 136, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 43, 43, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 58, 196, 254, 255, 255, 255, 207, 98, 0, 0, 0],
            &[0, 0, 52, 247, 255, 189, 137, 140, 173, 241, 123, 0, 0, 0],
            &[0, 0, 171, 255, 122, 0, 0, 0, 0, 7, 11, 0, 0, 0],
            &[0, 0, 214, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 204, 255, 57, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 255, 208, 33, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 190, 255, 250, 158, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 104, 227, 255, 253, 169, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 101, 218, 255, 244, 61, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 139, 255, 213, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 2, 239, 255, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 228, 255, 16, 0, 0],
            &[0, 15, 22, 0, 0, 0, 0, 0, 75, 255, 219, 0, 0, 0],
            &[0, 55, 252, 193, 137, 111, 114, 165, 251, 254, 83, 0, 0, 0],
            &[0, 25, 167, 240, 255, 255, 255, 254, 203, 73, 0, 0, 0, 0],
            &[0, 0, 0, 2, 30, 48, 44, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ś' / 0x15b
        'ś' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 10, 22, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 201, 255, 117, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 102, 255, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 241, 157, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 63, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 93, 113, 100, 68, 7, 0, 0, 0, 0],
            &[0, 0, 5, 161, 254, 255, 255, 255, 255, 241, 19, 0, 0, 0],
            &[0, 0, 99, 255, 189, 62, 33, 45, 91, 129, 0, 0, 0, 0],
            &[0, 0, 146, 255, 75, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 105, 255, 192, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 167, 255, 247, 153, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 64, 187, 255, 253, 175, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 51, 175, 255, 234, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 168, 255, 87, 0, 0, 0],
            &[0, 0, 20, 0, 0, 0, 0, 0, 160, 255, 81, 0, 0, 0],
            &[0, 0, 202, 193, 124, 95, 100, 154, 253, 234, 13, 0, 0, 0],
            &[0, 0, 126, 230, 255, 255, 255, 252, 189, 44, 0, 0, 0, 0],
            &[0, 0, 0, 1, 20, 44, 42, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => &[
            &[0, 0, 0, 0, 0, 117, 210, 184, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 67, 251, 194, 252, 153, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 239, 158, 2, 86, 248, 123, 0, 0, 0, 0],
            &[0, 0, 0, 105, 100, 0, 0, 0, 53, 136, 22, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 43, 43, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 58, 196, 254, 255, 255, 255, 207, 98, 0, 0, 0],
            &[0, 0, 52, 247, 255, 189, 137, 140, 173, 241, 123, 0, 0, 0],
            &[0, 0, 171, 255, 122, 0, 0, 0, 0, 7, 11, 0, 0, 0],
            &[0, 0, 214, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 204, 255, 57, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 255, 208, 33, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 190, 255, 250, 158, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 104, 227, 255, 253, 169, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 101, 218, 255, 244, 61, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 139, 255, 213, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 2, 239, 255, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 228, 255, 16, 0, 0],
            &[0, 15, 22, 0, 0, 0, 0, 0, 75, 255, 219, 0, 0, 0],
            &[0, 55, 252, 193, 137, 111, 114, 165, 251, 254, 83, 0, 0, 0],
            &[0, 25, 167, 240, 255, 255, 255, 254, 203, 73, 0, 0, 0, 0],
            &[0, 0, 0, 2, 30, 48, 44, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŝ' / 0x15d
        'ŝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 22, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 230, 255, 187, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 189, 234, 128, 253, 135, 0, 0, 0, 0, 0],
            &[0, 0, 1, 162, 224, 43, 0, 87, 246, 111, 0, 0, 0, 0],
            &[0, 0, 9, 70, 19, 0, 0, 0, 36, 66, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 93, 113, 100, 68, 7, 0, 0, 0, 0],
            &[0, 0, 5, 161, 254, 255, 255, 255, 255, 241, 19, 0, 0, 0],
            &[0, 0, 99, 255, 189, 62, 33, 45, 91, 129, 0, 0, 0, 0],
            &[0, 0, 146, 255, 75, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 105, 255, 192, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 167, 255, 247, 153, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 64, 187, 255, 253, 175, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 51, 175, 255, 234, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 168, 255, 87, 0, 0, 0],
            &[0, 0, 20, 0, 0, 0, 0, 0, 160, 255, 81, 0, 0, 0],
            &[0, 0, 202, 193, 124, 95, 100, 154, 253, 234, 13, 0, 0, 0],
            &[0, 0, 126, 230, 255, 255, 255, 252, 189, 44, 0, 0, 0, 0],
            &[0, 0, 0, 1, 20, 44, 42, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ş' / 0x15e
        'Ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 43, 43, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 58, 196, 254, 255, 255, 255, 207, 98, 0, 0, 0],
            &[0, 0, 52, 247, 255, 189, 137, 140, 173, 241, 123, 0, 0, 0],
            &[0, 0, 171, 255, 122, 0, 0, 0, 0, 7, 11, 0, 0, 0],
            &[0, 0, 214, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 204, 255, 57, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 255, 208, 33, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 190, 255, 250, 158, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 104, 227, 255, 253, 169, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 101, 218, 255, 244, 61, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 139, 255, 213, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 2, 239, 255, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 228, 255, 16, 0, 0],
            &[0, 15, 22, 0, 0, 0, 0, 0, 75, 255, 219, 0, 0, 0],
            &[0, 55, 252, 193, 137, 111, 114, 165, 251, 254, 83, 0, 0, 0],
            &[0, 25, 167, 240, 255, 255, 255, 254, 203, 73, 0, 0, 0, 0],
            &[0, 0, 0, 2, 30, 179, 229, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 226, 223, 99, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 122, 255, 45, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ş' / 0x15f
        'ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 93, 113, 100, 68, 7, 0, 0, 0, 0],
            &[0, 0, 5, 161, 254, 255, 255, 255, 255, 241, 19, 0, 0, 0],
            &[0, 0, 99, 255, 189, 62, 33, 45, 91, 129, 0, 0, 0, 0],
            &[0, 0, 146, 255, 75, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 105, 255, 192, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 167, 255, 247, 153, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 64, 187, 255, 253, 175, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 51, 175, 255, 234, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 168, 255, 87, 0, 0, 0],
            &[0, 0, 20, 0, 0, 0, 0, 0, 160, 255, 81, 0, 0, 0],
            &[0, 0, 202, 193, 124, 95, 100, 154, 253, 234, 13, 0, 0, 0],
            &[0, 0, 126, 230, 255, 255, 255, 254, 189, 44, 0, 0, 0, 0],
            &[0, 0, 0, 1, 20, 148, 253, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 204, 232, 115, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 98, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 4, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Š' / 0x160
        'Š' => &[
            &[0, 0, 0, 142, 177, 23, 0, 1, 123, 200, 25, 0, 0, 0],
            &[0, 0, 0, 15, 209, 225, 42, 169, 249, 68, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 233, 252, 255, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 137, 111, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 43, 43, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 58, 196, 254, 255, 255, 255, 207, 98, 0, 0, 0],
            &[0, 0, 52, 247, 255, 189, 137, 140, 173, 241, 123, 0, 0, 0],
            &[0, 0, 171, 255, 122, 0, 0, 0, 0, 7, 11, 0, 0, 0],
            &[0, 0, 214, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 204, 255, 57, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 130, 255, 208, 33, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 190, 255, 250, 158, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 104, 227, 255, 253, 169, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 101, 218, 255, 244, 61, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 139, 255, 213, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 2, 239, 255, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 228, 255, 16, 0, 0],
            &[0, 15, 22, 0, 0, 0, 0, 0, 75, 255, 219, 0, 0, 0],
            &[0, 55, 252, 193, 137, 111, 114, 165, 251, 254, 83, 0, 0, 0],
            &[0, 25, 167, 240, 255, 255, 255, 254, 203, 73, 0, 0, 0, 0],
            &[0, 0, 0, 2, 30, 48, 44, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'š' / 0x161
        'š' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 22, 3, 0, 0, 0, 8, 20, 0, 0, 0, 0],
            &[0, 0, 9, 204, 198, 27, 0, 57, 227, 162, 0, 0, 0, 0],
            &[0, 0, 0, 20, 219, 223, 103, 246, 178, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 245, 255, 213, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 70, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 93, 113, 100, 68, 7, 0, 0, 0, 0],
            &[0, 0, 5, 161, 254, 255, 255, 255, 255, 241, 19, 0, 0, 0],
            &[0, 0, 99, 255, 189, 62, 33, 45, 91, 129, 0, 0, 0, 0],
            &[0, 0, 146, 255, 75, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 105, 255, 192, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 167, 255, 247, 153, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 64, 187, 255, 253, 175, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 51, 175, 255, 234, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 168, 255, 87, 0, 0, 0],
            &[0, 0, 20, 0, 0, 0, 0, 0, 160, 255, 81, 0, 0, 0],
            &[0, 0, 202, 193, 124, 95, 100, 154, 253, 234, 13, 0, 0, 0],
            &[0, 0, 126, 230, 255, 255, 255, 252, 189, 44, 0, 0, 0, 0],
            &[0, 0, 0, 1, 20, 44, 42, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ţ' / 0x162
        'Ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 13, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 166, 0,
            ],
            &[
                0, 9, 175, 175, 175, 175, 227, 255, 195, 175, 175, 175, 114, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 171, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 130, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 226, 223, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 122, 255, 45, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 3, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ţ' / 0x163
        'ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 66, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 155, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 32, 60, 205, 255, 90, 65, 65, 65, 34, 0, 0],
            &[0, 0, 210, 255, 255, 255, 255, 255, 255, 255, 255, 133, 0, 0],
            &[0, 0, 67, 81, 81, 213, 255, 104, 81, 81, 81, 42, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 255, 47, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 106, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 255, 248, 136, 92, 104, 119, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 235, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 255, 119, 30, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 90, 255, 170, 37, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 26, 215, 187, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ť' / 0x164
        'Ť' => &[
            &[0, 0, 0, 25, 201, 115, 0, 0, 28, 181, 141, 0, 0, 0],
            &[0, 0, 0, 0, 75, 251, 157, 49, 231, 203, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 107, 255, 252, 226, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 137, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 13, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 166, 0,
            ],
            &[
                0, 9, 175, 175, 175, 175, 227, 255, 195, 175, 175, 175, 114, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ť' / 0x165
        'ť' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 48, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 47, 255, 146, 0, 0],
            &[0, 0, 0, 0, 0, 26, 66, 8, 0, 99, 254, 42, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 33, 0, 164, 162, 0, 0, 0],
            &[0, 0, 0, 0, 0, 155, 255, 33, 0, 33, 12, 0, 0, 0],
            &[0, 0, 6, 32, 60, 205, 255, 90, 65, 65, 65, 34, 0, 0],
            &[0, 0, 210, 255, 255, 255, 255, 255, 255, 255, 255, 133, 0, 0],
            &[0, 0, 67, 81, 81, 213, 255, 104, 81, 81, 81, 42, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 255, 47, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 106, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 255, 248, 136, 92, 104, 119, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 235, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 34, 48, 30, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŧ' / 0x166
        'Ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 13, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 166, 0,
            ],
            &[
                0, 9, 175, 175, 175, 175, 227, 255, 195, 175, 175, 175, 114, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 73, 96, 96, 255, 255, 159, 96, 96, 36, 0, 0],
            &[0, 0, 0, 192, 255, 255, 255, 255, 255, 255, 255, 95, 0, 0],
            &[0, 0, 0, 33, 44, 44, 209, 255, 107, 44, 44, 16, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 165, 255, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŧ' / 0x167
        'ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 66, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 155, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 32, 60, 205, 255, 90, 65, 65, 65, 34, 0, 0],
            &[0, 0, 210, 255, 255, 255, 255, 255, 255, 255, 255, 133, 0, 0],
            &[0, 0, 67, 81, 81, 213, 255, 104, 81, 81, 81, 42, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 56, 68, 68, 255, 255, 101, 68, 68, 68, 56, 0, 0],
            &[0, 0, 210, 255, 255, 255, 255, 255, 255, 255, 255, 213, 0, 0],
            &[0, 0, 25, 30, 30, 224, 255, 64, 30, 30, 30, 25, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 255, 47, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 106, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 255, 248, 136, 92, 104, 119, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 235, 255, 255, 255, 204, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 34, 48, 30, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ũ' / 0x168
        'Ũ' => &[
            &[0, 0, 0, 0, 19, 8, 0, 0, 0, 44, 2, 0, 0, 0],
            &[0, 0, 0, 154, 255, 246, 135, 18, 86, 237, 1, 0, 0, 0],
            &[0, 0, 58, 246, 112, 141, 247, 254, 255, 128, 0, 0, 0, 0],
            &[0, 0, 57, 97, 0, 0, 30, 113, 86, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 0, 193, 254, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 134, 255, 93, 0, 0, 0, 0, 0, 195, 255, 32, 0, 0],
            &[0, 119, 255, 108, 0, 0, 0, 0, 0, 210, 255, 17, 0, 0],
            &[0, 86, 255, 156, 0, 0, 0, 0, 8, 247, 236, 1, 0, 0],
            &[0, 21, 249, 242, 33, 0, 0, 0, 117, 255, 162, 0, 0, 0],
            &[0, 0, 134, 255, 237, 149, 129, 178, 254, 241, 36, 0, 0, 0],
            &[0, 0, 1, 123, 236, 255, 255, 255, 194, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 32, 44, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ũ' / 0x169
        'ũ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 86, 65, 1, 0, 7, 110, 3, 0, 0],
            &[0, 0, 0, 5, 208, 255, 255, 201, 83, 146, 218, 0, 0, 0],
            &[0, 0, 0, 79, 231, 45, 74, 209, 255, 250, 81, 0, 0, 0],
            &[0, 0, 0, 31, 45, 0, 0, 2, 46, 24, 0, 0, 0, 0],
            &[0, 0, 27, 65, 30, 0, 0, 0, 0, 0, 63, 60, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 249, 235, 0, 0],
            &[0, 0, 108, 255, 119, 0, 0, 0, 0, 7, 255, 235, 0, 0],
            &[0, 0, 98, 255, 135, 0, 0, 0, 0, 52, 255, 235, 0, 0],
            &[0, 0, 64, 255, 201, 0, 0, 0, 0, 163, 255, 235, 0, 0],
            &[0, 0, 6, 219, 255, 179, 99, 99, 177, 228, 232, 235, 0, 0],
            &[0, 0, 0, 46, 203, 255, 255, 255, 198, 42, 174, 235, 0, 0],
            &[0, 0, 0, 0, 0, 17, 45, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ū' / 0x16a
        'Ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 43, 52, 52, 52, 52, 52, 21, 0, 0, 0, 0],
            &[0, 0, 0, 212, 255, 255, 255, 255, 255, 104, 0, 0, 0, 0],
            &[0, 0, 0, 60, 72, 72, 72, 72, 72, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 0, 193, 254, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 134, 255, 93, 0, 0, 0, 0, 0, 195, 255, 32, 0, 0],
            &[0, 119, 255, 108, 0, 0, 0, 0, 0, 210, 255, 17, 0, 0],
            &[0, 86, 255, 156, 0, 0, 0, 0, 8, 247, 236, 1, 0, 0],
            &[0, 21, 249, 242, 33, 0, 0, 0, 117, 255, 162, 0, 0, 0],
            &[0, 0, 134, 255, 237, 149, 129, 178, 254, 241, 36, 0, 0, 0],
            &[0, 0, 1, 123, 236, 255, 255, 255, 194, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 32, 44, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ū' / 0x16b
        'ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 120, 120, 120, 120, 120, 48, 0, 0, 0],
            &[0, 0, 0, 0, 212, 255, 255, 255, 255, 255, 104, 0, 0, 0],
            &[0, 0, 0, 0, 4, 5, 5, 5, 5, 5, 2, 0, 0, 0],
            &[0, 0, 27, 65, 30, 0, 0, 0, 0, 0, 63, 60, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 249, 235, 0, 0],
            &[0, 0, 108, 255, 119, 0, 0, 0, 0, 7, 255, 235, 0, 0],
            &[0, 0, 98, 255, 135, 0, 0, 0, 0, 52, 255, 235, 0, 0],
            &[0, 0, 64, 255, 201, 0, 0, 0, 0, 163, 255, 235, 0, 0],
            &[0, 0, 6, 219, 255, 179, 99, 99, 177, 228, 232, 235, 0, 0],
            &[0, 0, 0, 46, 203, 255, 255, 255, 198, 42, 174, 235, 0, 0],
            &[0, 0, 0, 0, 0, 17, 45, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => &[
            &[0, 0, 0, 67, 22, 0, 0, 0, 52, 37, 0, 0, 0, 0],
            &[0, 0, 0, 205, 172, 6, 0, 36, 241, 98, 0, 0, 0, 0],
            &[0, 0, 0, 69, 247, 235, 217, 254, 197, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 109, 125, 82, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 0, 193, 254, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 134, 255, 93, 0, 0, 0, 0, 0, 195, 255, 32, 0, 0],
            &[0, 119, 255, 108, 0, 0, 0, 0, 0, 210, 255, 17, 0, 0],
            &[0, 86, 255, 156, 0, 0, 0, 0, 8, 247, 236, 1, 0, 0],
            &[0, 21, 249, 242, 33, 0, 0, 0, 117, 255, 162, 0, 0, 0],
            &[0, 0, 134, 255, 237, 149, 129, 178, 254, 241, 36, 0, 0, 0],
            &[0, 0, 1, 123, 236, 255, 255, 255, 194, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 32, 44, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŭ' / 0x16d
        'ŭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 126, 52, 0, 0, 0, 108, 68, 0, 0, 0],
            &[0, 0, 0, 0, 181, 209, 52, 29, 102, 251, 74, 0, 0, 0],
            &[0, 0, 0, 0, 33, 210, 255, 255, 251, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 42, 58, 19, 0, 0, 0, 0, 0],
            &[0, 0, 27, 65, 30, 0, 0, 0, 0, 0, 63, 60, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 249, 235, 0, 0],
            &[0, 0, 108, 255, 119, 0, 0, 0, 0, 7, 255, 235, 0, 0],
            &[0, 0, 98, 255, 135, 0, 0, 0, 0, 52, 255, 235, 0, 0],
            &[0, 0, 64, 255, 201, 0, 0, 0, 0, 163, 255, 235, 0, 0],
            &[0, 0, 6, 219, 255, 179, 99, 99, 177, 228, 232, 235, 0, 0],
            &[0, 0, 0, 46, 203, 255, 255, 255, 198, 42, 174, 235, 0, 0],
            &[0, 0, 0, 0, 0, 17, 45, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ů' / 0x16e
        'Ů' => &[
            &[0, 0, 0, 0, 214, 160, 72, 229, 109, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 70, 0, 178, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 170, 223, 170, 249, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 98, 124, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 0, 193, 254, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 134, 255, 93, 0, 0, 0, 0, 0, 195, 255, 32, 0, 0],
            &[0, 119, 255, 108, 0, 0, 0, 0, 0, 210, 255, 17, 0, 0],
            &[0, 86, 255, 156, 0, 0, 0, 0, 8, 247, 236, 1, 0, 0],
            &[0, 21, 249, 242, 33, 0, 0, 0, 117, 255, 162, 0, 0, 0],
            &[0, 0, 134, 255, 237, 149, 129, 178, 254, 241, 36, 0, 0, 0],
            &[0, 0, 1, 123, 236, 255, 255, 255, 194, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 32, 44, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 191, 216, 143, 5, 0, 0, 0, 0, 0],
        ],
        // letter: 'ů' / 0x16f
        'ů' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 28, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 96, 247, 250, 209, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 233, 109, 11, 206, 130, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 238, 90, 2, 196, 134, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 118, 252, 235, 229, 33, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 57, 10, 0, 0, 0, 0, 0],
            &[0, 0, 27, 65, 30, 0, 0, 0, 0, 0, 63, 60, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 249, 235, 0, 0],
            &[0, 0, 108, 255, 119, 0, 0, 0, 0, 7, 255, 235, 0, 0],
            &[0, 0, 98, 255, 135, 0, 0, 0, 0, 52, 255, 235, 0, 0],
            &[0, 0, 64, 255, 201, 0, 0, 0, 0, 163, 255, 235, 0, 0],
            &[0, 0, 6, 219, 255, 179, 99, 99, 177, 228, 232, 235, 0, 0],
            &[0, 0, 0, 46, 203, 255, 255, 255, 198, 42, 174, 235, 0, 0],
            &[0, 0, 0, 0, 0, 17, 45, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ű' / 0x170
        'Ű' => &[
            &[0, 0, 0, 0, 94, 210, 130, 0, 157, 210, 66, 0, 0, 0],
            &[0, 0, 0, 12, 229, 222, 20, 62, 255, 165, 0, 0, 0, 0],
            &[0, 0, 0, 138, 233, 40, 5, 210, 189, 7, 0, 0, 0, 0],
            &[0, 0, 4, 133, 38, 0, 43, 125, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 0, 193, 254, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 134, 255, 93, 0, 0, 0, 0, 0, 195, 255, 32, 0, 0],
            &[0, 119, 255, 108, 0, 0, 0, 0, 0, 210, 255, 17, 0, 0],
            &[0, 86, 255, 156, 0, 0, 0, 0, 8, 247, 236, 1, 0, 0],
            &[0, 21, 249, 242, 33, 0, 0, 0, 117, 255, 162, 0, 0, 0],
            &[0, 0, 134, 255, 237, 149, 129, 178, 254, 241, 36, 0, 0, 0],
            &[0, 0, 1, 123, 236, 255, 255, 255, 194, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 32, 44, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ű' / 0x171
        'ű' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 22, 17, 0, 12, 22, 10, 0, 0],
            &[0, 0, 0, 0, 0, 138, 255, 129, 2, 212, 251, 56, 0, 0],
            &[0, 0, 0, 0, 32, 247, 183, 4, 100, 255, 109, 0, 0, 0],
            &[0, 0, 0, 0, 181, 195, 13, 23, 236, 130, 0, 0, 0, 0],
            &[0, 0, 0, 4, 70, 9, 0, 25, 58, 0, 0, 0, 0, 0],
            &[0, 0, 27, 65, 30, 0, 0, 0, 0, 0, 63, 60, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0],
            &[0, 0, 108, 255, 118, 0, 0, 0, 0, 0, 249, 235, 0, 0],
            &[0, 0, 108, 255, 119, 0, 0, 0, 0, 7, 255, 235, 0, 0],
            &[0, 0, 98, 255, 135, 0, 0, 0, 0, 52, 255, 235, 0, 0],
            &[0, 0, 64, 255, 201, 0, 0, 0, 0, 163, 255, 235, 0, 0],
            &[0, 0, 6, 219, 255, 179, 99, 99, 177, 228, 232, 235, 0, 0],
            &[0, 0, 0, 46, 203, 255, 255, 255, 198, 42, 174, 235, 0, 0],
            &[0, 0, 0, 0, 0, 17, 45, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ų' / 0x172
        'Ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 254, 91, 0, 0, 0, 0, 0, 193, 254, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 135, 255, 92, 0, 0, 0, 0, 0, 194, 255, 33, 0, 0],
            &[0, 134, 255, 93, 0, 0, 0, 0, 0, 195, 255, 32, 0, 0],
            &[0, 119, 255, 108, 0, 0, 0, 0, 0, 210, 255, 17, 0, 0],
            &[0, 86, 255, 156, 0, 0, 0, 0, 8, 247, 236, 1, 0, 0],
            &[0, 21, 249, 242, 33, 0, 0, 0, 117, 255, 162, 0, 0, 0],
            &[0, 0, 134, 255, 237, 149, 129, 178, 254, 241, 36, 0, 0, 0],
            &[0, 0, 1, 123, 236, 255, 255, 255, 251, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 60, 255, 167, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 209, 182, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 255, 126, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 135, 149, 43, 0, 0, 0, 0, 0],
        ],
        // letter: 'ų' / 0x173
        'ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 27, 65, 30, 0, 0, 0, 0, 0, 63, 60, 0, 0, 0],
            &[0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0, 0],
            &[0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0, 0],
            &[0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0, 0],
            &[0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0, 0],
            &[0, 108, 255, 118, 0, 0, 0, 0, 0, 248, 235, 0, 0, 0],
            &[0, 108, 255, 118, 0, 0, 0, 0, 0, 249, 235, 0, 0, 0],
            &[0, 108, 255, 119, 0, 0, 0, 0, 7, 255, 235, 0, 0, 0],
            &[0, 98, 255, 135, 0, 0, 0, 0, 52, 255, 235, 0, 0, 0],
            &[0, 64, 255, 201, 0, 0, 0, 0, 163, 255, 235, 0, 0, 0],
            &[0, 6, 219, 255, 179, 99, 99, 177, 228, 232, 235, 0, 0, 0],
            &[0, 0, 46, 203, 255, 255, 255, 198, 42, 203, 255, 0, 0, 0],
            &[0, 0, 0, 0, 17, 45, 23, 0, 37, 232, 97, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 196, 174, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 254, 121, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 42, 139, 148, 40, 0, 0],
        ],
        // letter: 'Ŵ' / 0x174
        'Ŵ' => &[
            &[0, 0, 0, 0, 11, 190, 210, 107, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 164, 248, 198, 248, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 130, 244, 71, 6, 174, 235, 41, 0, 0, 0, 0],
            &[0, 0, 23, 137, 46, 0, 0, 1, 106, 105, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[106, 254, 80, 0, 0, 0, 0, 0, 0, 0, 181, 253, 7, 0],
            &[78, 255, 105, 0, 0, 0, 0, 0, 0, 0, 204, 236, 0, 0],
            &[50, 255, 129, 0, 0, 0, 0, 0, 0, 0, 224, 210, 0, 0],
            &[21, 255, 153, 0, 0, 40, 65, 16, 0, 0, 246, 184, 0, 0],
            &[0, 247, 178, 0, 0, 193, 255, 103, 0, 13, 255, 159, 0, 0],
            &[0, 219, 202, 0, 6, 247, 242, 164, 0, 36, 255, 133, 0, 0],
            &[0, 191, 226, 0, 57, 255, 159, 226, 0, 58, 255, 107, 0, 0],
            &[0, 162, 249, 0, 116, 240, 70, 255, 31, 79, 255, 81, 0, 0],
            &[0, 133, 255, 15, 176, 186, 13, 252, 92, 98, 255, 55, 0, 0],
            &[0, 105, 255, 35, 233, 126, 0, 204, 153, 115, 255, 30, 0, 0],
            &[0, 76, 255, 86, 255, 67, 0, 142, 213, 130, 254, 5, 0, 0],
            &[0, 47, 255, 156, 252, 12, 0, 81, 254, 158, 234, 0, 0, 0],
            &[0, 19, 255, 220, 204, 0, 0, 20, 254, 219, 208, 0, 0, 0],
            &[0, 0, 246, 255, 145, 0, 0, 0, 213, 255, 182, 0, 0, 0],
            &[0, 0, 217, 255, 86, 0, 0, 0, 152, 255, 156, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŵ' / 0x175
        'ŵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 22, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 30, 240, 255, 167, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 205, 225, 135, 255, 113, 0, 0, 0, 0, 0],
            &[0, 0, 4, 181, 213, 32, 0, 104, 248, 91, 0, 0, 0, 0],
            &[0, 0, 15, 70, 13, 0, 0, 0, 42, 60, 0, 0, 0, 0],
            &[42, 65, 7, 0, 0, 0, 0, 0, 0, 0, 29, 65, 17, 0],
            &[138, 255, 54, 0, 0, 0, 0, 0, 0, 0, 142, 255, 41, 0],
            &[97, 255, 91, 0, 0, 26, 34, 12, 0, 0, 183, 250, 4, 0],
            &[56, 255, 129, 0, 2, 236, 255, 129, 0, 0, 223, 213, 0, 0],
            &[15, 255, 167, 0, 56, 255, 241, 199, 0, 10, 254, 171, 0, 0],
            &[0, 230, 204, 0, 129, 253, 136, 252, 17, 48, 255, 129, 0, 0],
            &[0, 189, 241, 0, 202, 204, 60, 255, 85, 87, 255, 87, 0, 0],
            &[0, 148, 255, 34, 253, 134, 5, 245, 153, 121, 255, 44, 0, 0],
            &[0, 107, 255, 113, 255, 66, 0, 185, 215, 146, 252, 6, 0, 0],
            &[0, 66, 255, 181, 247, 8, 0, 120, 254, 182, 216, 0, 0, 0],
            &[0, 25, 255, 243, 188, 0, 0, 54, 255, 243, 174, 0, 0, 0],
            &[0, 0, 239, 255, 122, 0, 0, 3, 241, 255, 132, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŷ' / 0x176
        'Ŷ' => &[
            &[0, 0, 0, 0, 0, 11, 190, 210, 107, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 164, 248, 198, 248, 58, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 244, 71, 6, 174, 235, 41, 0, 0, 0],
            &[0, 0, 0, 23, 137, 46, 0, 0, 1, 106, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 239, 244, 20, 0, 0, 0, 0, 0, 97, 254, 155, 0],
            &[0, 0, 129, 255, 129, 0, 0, 0, 0, 1, 214, 250, 32, 0],
            &[0, 0, 15, 238, 236, 10, 0, 0, 0, 79, 255, 152, 0, 0],
            &[0, 0, 0, 124, 255, 110, 0, 0, 0, 198, 248, 29, 0, 0],
            &[0, 0, 0, 13, 236, 224, 4, 0, 62, 255, 147, 0, 0, 0],
            &[0, 0, 0, 0, 119, 255, 90, 0, 181, 247, 26, 0, 0, 0],
            &[0, 0, 0, 0, 11, 233, 208, 45, 254, 142, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 114, 255, 214, 245, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 231, 255, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 156, 255, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŷ' / 0x177
        'ŷ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 22, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 242, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 208, 222, 138, 255, 108, 0, 0, 0, 0],
            &[0, 0, 0, 6, 185, 210, 29, 0, 108, 248, 87, 0, 0, 0],
            &[0, 0, 0, 17, 70, 12, 0, 0, 0, 43, 59, 0, 0, 0],
            &[0, 0, 64, 55, 0, 0, 0, 0, 0, 0, 22, 65, 38, 0],
            &[0, 0, 189, 252, 28, 0, 0, 0, 0, 0, 140, 255, 90, 0],
            &[0, 0, 83, 255, 129, 0, 0, 0, 0, 1, 225, 241, 9, 0],
            &[0, 0, 4, 228, 229, 4, 0, 0, 0, 56, 255, 156, 0, 0],
            &[0, 0, 0, 126, 255, 81, 0, 0, 0, 141, 255, 61, 0, 0],
            &[0, 0, 0, 24, 250, 185, 0, 0, 1, 226, 221, 0, 0, 0],
            &[0, 0, 0, 0, 169, 254, 35, 0, 57, 255, 127, 0, 0, 0],
            &[0, 0, 0, 0, 62, 255, 135, 0, 141, 254, 33, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 228, 2, 223, 193, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 105, 255, 108, 255, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 242, 241, 245, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 148, 255, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 125, 255, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 224, 224, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 113, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 30, 19, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ÿ' / 0x178
        'Ÿ' => &[
            &[0, 0, 0, 0, 27, 14, 0, 0, 0, 35, 6, 0, 0, 0],
            &[0, 0, 0, 24, 251, 220, 0, 0, 69, 255, 172, 0, 0, 0],
            &[0, 0, 0, 19, 247, 211, 0, 0, 61, 254, 162, 0, 0, 0],
            &[0, 0, 0, 0, 18, 8, 0, 0, 0, 24, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 239, 244, 20, 0, 0, 0, 0, 0, 97, 254, 155, 0],
            &[0, 0, 129, 255, 129, 0, 0, 0, 0, 1, 214, 250, 32, 0],
            &[0, 0, 15, 238, 236, 10, 0, 0, 0, 79, 255, 152, 0, 0],
            &[0, 0, 0, 124, 255, 110, 0, 0, 0, 198, 248, 29, 0, 0],
            &[0, 0, 0, 13, 236, 224, 4, 0, 62, 255, 147, 0, 0, 0],
            &[0, 0, 0, 0, 119, 255, 90, 0, 181, 247, 26, 0, 0, 0],
            &[0, 0, 0, 0, 11, 233, 208, 45, 254, 142, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 114, 255, 214, 245, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 231, 255, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 156, 255, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 154, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ź' / 0x179
        'Ź' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 115, 210, 153, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 33, 245, 226, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 189, 229, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 34, 136, 29, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 148, 254, 254, 254, 254, 254, 254, 254, 153, 0, 0],
            &[0, 0, 0, 96, 164, 164, 164, 164, 164, 205, 255, 136, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 2, 209, 243, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 99, 255, 125, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 231, 228, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 130, 255, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 246, 208, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 162, 255, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 254, 182, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 252, 43, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 255, 153, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 218, 243, 23, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 255, 124, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 255, 183, 174, 174, 174, 174, 174, 105, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 154, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ź' / 0x17a
        'ź' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 8, 22, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 175, 255, 144, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 75, 255, 178, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 229, 178, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 24, 68, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 65, 65, 65, 65, 65, 65, 65, 35, 0, 0],
            &[0, 0, 0, 198, 255, 255, 255, 255, 255, 255, 255, 138, 0, 0],
            &[0, 0, 0, 38, 49, 49, 49, 49, 51, 213, 255, 80, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 140, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 74, 254, 191, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 238, 228, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 205, 249, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 156, 255, 105, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 255, 161, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 247, 207, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 222, 255, 183, 152, 152, 152, 152, 152, 108, 0, 0],
            &[0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ż' / 0x17b
        'Ż' => &[
            &[0, 0, 0, 0, 0, 0, 58, 156, 66, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 191, 255, 207, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 105, 224, 118, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 148, 254, 254, 254, 254, 254, 254, 254, 153, 0, 0],
            &[0, 0, 0, 96, 164, 164, 164, 164, 164, 205, 255, 136, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 2, 209, 243, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 99, 255, 125, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 231, 228, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 130, 255, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 246, 208, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 162, 255, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 254, 182, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 252, 43, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 255, 153, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 218, 243, 23, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 255, 124, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 255, 183, 174, 174, 174, 174, 174, 105, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 154, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ż' / 0x17c
        'ż' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 113, 223, 109, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 202, 255, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 63, 157, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 65, 65, 65, 65, 65, 65, 65, 35, 0, 0],
            &[0, 0, 0, 198, 255, 255, 255, 255, 255, 255, 255, 138, 0, 0],
            &[0, 0, 0, 38, 49, 49, 49, 49, 51, 213, 255, 80, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 140, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 74, 254, 191, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 238, 228, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 205, 249, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 156, 255, 105, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 255, 161, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 247, 207, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 222, 255, 183, 152, 152, 152, 152, 152, 108, 0, 0],
            &[0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ž' / 0x17d
        'Ž' => &[
            &[0, 0, 0, 0, 173, 158, 11, 0, 7, 148, 186, 8, 0, 0],
            &[0, 0, 0, 0, 32, 230, 205, 36, 196, 237, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 248, 252, 251, 61, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 82, 137, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 148, 254, 254, 254, 254, 254, 254, 254, 153, 0, 0],
            &[0, 0, 0, 96, 164, 164, 164, 164, 164, 205, 255, 136, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 2, 209, 243, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 99, 255, 125, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 231, 228, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 130, 255, 97, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 246, 208, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 162, 255, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 254, 182, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 252, 43, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 255, 153, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 218, 243, 23, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 112, 255, 124, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 255, 183, 174, 174, 174, 174, 174, 105, 0, 0],
            &[0, 0, 1, 255, 255, 255, 255, 255, 255, 255, 255, 154, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ž' / 0x17e
        'ž' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 6, 0, 0, 0, 6, 22, 1, 0, 0],
            &[0, 0, 0, 2, 184, 212, 39, 0, 42, 215, 185, 3, 0, 0],
            &[0, 0, 0, 0, 10, 202, 235, 99, 238, 198, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 233, 255, 230, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 38, 70, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 65, 65, 65, 65, 65, 65, 65, 35, 0, 0],
            &[0, 0, 0, 198, 255, 255, 255, 255, 255, 255, 255, 138, 0, 0],
            &[0, 0, 0, 38, 49, 49, 49, 49, 51, 213, 255, 80, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 130, 255, 140, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 74, 254, 191, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 238, 228, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 205, 249, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 156, 255, 105, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 255, 161, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 247, 207, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 222, 255, 183, 152, 152, 152, 152, 152, 108, 0, 0],
            &[0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ſ' / 0x17f
        'ſ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 7, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 142, 229, 255, 255, 233, 76, 0, 0, 0],
            &[0, 0, 0, 0, 183, 255, 211, 134, 133, 170, 31, 0, 0, 0],
            &[0, 0, 0, 42, 255, 217, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 255, 146, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 91, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        _ => panic!("unsupported char"),
    }
}
