//! Module for letters with the font weight regular and size 22.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 22;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 13;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight regular and font size 19px
#[inline]
pub const fn get_char(c: char) -> &'static [&'static [u8]] {
    match c {
        // letter: ' ' / 0x20
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '!' / 0x21
        '!' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 144, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 74, 255, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 66, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 58, 255, 166, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 255, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 42, 255, 150, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 255, 134, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 255, 126, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 255, 118, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 41, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 75, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 204, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 254, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '"' / 0x22
        '"' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 135, 119, 0, 61, 144, 48, 0, 0, 0, 0],
            &[0, 0, 0, 224, 195, 0, 92, 255, 70, 0, 0, 0, 0],
            &[0, 0, 0, 204, 175, 0, 72, 255, 51, 0, 0, 0, 0],
            &[0, 0, 0, 184, 155, 0, 53, 255, 31, 0, 0, 0, 0],
            &[0, 0, 0, 165, 135, 0, 33, 255, 11, 0, 0, 0, 0],
            &[0, 0, 0, 50, 41, 0, 6, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '#' / 0x23
        '#' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 115, 0, 0, 122, 67, 0, 0, 0],
            &[0, 0, 0, 0, 183, 166, 0, 5, 248, 81, 0, 0, 0],
            &[0, 0, 0, 0, 231, 118, 0, 46, 255, 33, 0, 0, 0],
            &[0, 0, 0, 24, 255, 69, 0, 94, 239, 0, 0, 0, 0],
            &[0, 126, 137, 165, 255, 153, 137, 197, 232, 137, 137, 2, 0],
            &[0, 168, 183, 221, 241, 183, 183, 246, 218, 183, 183, 3, 0],
            &[0, 0, 0, 169, 172, 0, 4, 248, 92, 0, 0, 0, 0],
            &[0, 0, 0, 220, 122, 0, 46, 255, 42, 0, 0, 0, 0],
            &[38, 62, 70, 254, 122, 62, 130, 251, 65, 62, 23, 0, 0],
            &[158, 255, 255, 255, 255, 255, 255, 255, 255, 255, 95, 0, 0],
            &[4, 7, 118, 230, 7, 7, 198, 151, 7, 7, 2, 0, 0],
            &[0, 0, 160, 183, 0, 2, 243, 99, 0, 0, 0, 0, 0],
            &[0, 0, 206, 135, 0, 40, 255, 48, 0, 0, 0, 0, 0],
            &[0, 4, 248, 87, 0, 90, 248, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '$' / 0x24
        '$' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 107, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 255, 57, 0, 0, 0, 0, 0],
            &[0, 0, 0, 35, 173, 243, 255, 253, 227, 164, 32, 0, 0],
            &[0, 0, 16, 229, 243, 146, 255, 147, 160, 230, 20, 0, 0],
            &[0, 0, 96, 255, 92, 0, 255, 54, 0, 1, 0, 0, 0],
            &[0, 0, 113, 255, 57, 0, 255, 54, 0, 0, 0, 0, 0],
            &[0, 0, 56, 255, 188, 33, 255, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 122, 253, 253, 255, 126, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 141, 255, 255, 245, 143, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 116, 178, 255, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 54, 1, 219, 211, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 54, 2, 224, 198, 0, 0],
            &[0, 0, 62, 156, 83, 46, 255, 114, 172, 255, 104, 0, 0],
            &[0, 0, 57, 241, 255, 255, 255, 255, 240, 113, 0, 0, 0],
            &[0, 0, 0, 5, 51, 78, 255, 91, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '%' / 0x25
        '%' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[1, 100, 169, 142, 23, 0, 0, 0, 115, 96, 0, 0, 0],
            &[120, 240, 121, 188, 218, 6, 0, 49, 255, 69, 0, 0, 0],
            &[220, 136, 0, 29, 255, 74, 0, 176, 197, 0, 0, 0, 0],
            &[247, 109, 0, 3, 255, 102, 49, 254, 70, 0, 0, 0, 0],
            &[218, 137, 0, 30, 255, 75, 175, 198, 0, 0, 0, 0, 0],
            &[112, 239, 116, 186, 221, 55, 254, 71, 0, 0, 0, 0, 0],
            &[0, 96, 170, 144, 25, 174, 199, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 254, 71, 11, 55, 27, 0, 0, 0],
            &[0, 0, 0, 0, 174, 199, 55, 234, 233, 250, 120, 0, 0],
            &[0, 0, 0, 47, 254, 72, 197, 173, 0, 83, 252, 35, 0],
            &[0, 0, 0, 173, 200, 1, 247, 105, 0, 15, 255, 86, 0],
            &[0, 0, 46, 254, 73, 0, 243, 108, 0, 18, 255, 82, 0],
            &[0, 0, 172, 201, 0, 0, 182, 188, 5, 102, 249, 29, 0],
            &[0, 45, 254, 74, 0, 0, 36, 214, 250, 248, 102, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 29, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '&' / 0x26
        '&' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 129, 185, 166, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 173, 248, 156, 216, 249, 43, 0, 0, 0, 0, 0],
            &[0, 10, 253, 140, 0, 17, 254, 127, 0, 0, 0, 0, 0],
            &[0, 10, 253, 130, 0, 13, 254, 132, 0, 0, 0, 0, 0],
            &[0, 0, 209, 199, 0, 137, 255, 54, 0, 0, 0, 0, 0],
            &[0, 0, 93, 255, 197, 255, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 205, 255, 255, 202, 8, 0, 42, 161, 49, 0, 0],
            &[0, 167, 253, 77, 68, 253, 160, 0, 123, 254, 23, 0, 0],
            &[23, 253, 155, 0, 0, 122, 255, 112, 220, 204, 0, 0, 0],
            &[58, 255, 104, 0, 0, 1, 179, 255, 255, 93, 0, 0, 0],
            &[38, 255, 154, 0, 0, 0, 73, 255, 255, 32, 0, 0, 0],
            &[0, 202, 252, 136, 79, 131, 248, 255, 255, 204, 8, 0, 0],
            &[0, 32, 197, 255, 255, 247, 157, 18, 97, 255, 162, 0, 0],
            &[0, 0, 0, 25, 41, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ''' / 0x27
        '\'' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 27, 144, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 255, 130, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 255, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 249, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 229, 71, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '(' / 0x28
        '(' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 89, 138, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 251, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 195, 231, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 255, 132, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 143, 255, 37, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 204, 230, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 251, 175, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 255, 150, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 255, 131, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 37, 255, 139, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 159, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 235, 201, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 177, 251, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 106, 255, 84, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 241, 189, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 254, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 198, 210, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ')' / 0x29
        ')' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 138, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 246, 153, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 135, 254, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 30, 253, 166, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 188, 246, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 125, 255, 57, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 70, 255, 109, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 46, 255, 131, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 255, 148, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 255, 137, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 59, 255, 118, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 102, 255, 76, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 162, 254, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 234, 200, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 255, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 210, 213, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 117, 249, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '*' / 0x2a
        '*' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 44, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 254, 101, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 237, 83, 0, 0, 0, 0, 0],
            &[0, 0, 13, 126, 62, 6, 217, 66, 42, 112, 87, 0, 0],
            &[0, 0, 45, 230, 254, 246, 247, 239, 255, 252, 155, 0, 0],
            &[0, 0, 0, 0, 8, 148, 244, 238, 48, 5, 0, 0, 0],
            &[0, 0, 0, 0, 66, 252, 79, 221, 176, 1, 0, 0, 0],
            &[0, 0, 0, 12, 232, 196, 0, 93, 255, 108, 0, 0, 0],
            &[0, 0, 0, 0, 54, 53, 0, 2, 108, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '+' / 0x2b
        '+' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 246, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 246, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 246, 103, 0, 0, 0, 0, 0],
            &[0, 0, 45, 95, 95, 95, 250, 159, 95, 95, 83, 0, 0],
            &[0, 0, 120, 250, 250, 250, 255, 252, 250, 250, 220, 0, 0],
            &[0, 0, 0, 0, 0, 0, 246, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 246, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 246, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 173, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ',' / 0x2c
        ',' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 114, 115, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 247, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 255, 174, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 129, 255, 80, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 229, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 94, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '-' / 0x2d
        '-' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 227, 227, 227, 227, 227, 125, 0, 0, 0],
            &[0, 0, 0, 24, 161, 161, 161, 161, 161, 88, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '.' / 0x2e
        '.' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 75, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 204, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 254, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '/' / 0x2f
        '/' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 90, 141, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 231, 181, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 78, 255, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 176, 235, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 251, 143, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 117, 255, 45, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 215, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 57, 255, 104, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 155, 246, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 242, 164, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 96, 255, 65, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 194, 222, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 37, 255, 125, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 135, 253, 29, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '0' / 0x30
        '0' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 45, 148, 183, 163, 90, 2, 0, 0, 0, 0],
            &[0, 0, 83, 253, 245, 202, 229, 255, 171, 2, 0, 0, 0],
            &[0, 13, 241, 214, 24, 0, 3, 250, 255, 94, 0, 0, 0],
            &[0, 100, 255, 73, 0, 0, 35, 250, 255, 198, 0, 0, 0],
            &[0, 157, 248, 4, 0, 0, 179, 240, 163, 249, 7, 0, 0],
            &[0, 195, 212, 0, 0, 74, 255, 110, 97, 255, 40, 0, 0],
            &[0, 210, 196, 0, 6, 218, 210, 3, 80, 255, 56, 0, 0],
            &[0, 214, 194, 0, 120, 255, 63, 0, 74, 255, 62, 0, 0],
            &[0, 200, 212, 27, 244, 166, 0, 0, 85, 255, 52, 0, 0],
            &[0, 174, 236, 168, 243, 27, 0, 0, 116, 255, 23, 0, 0],
            &[0, 122, 255, 255, 118, 0, 0, 0, 176, 230, 0, 0, 0],
            &[0, 38, 255, 255, 6, 0, 0, 42, 250, 150, 0, 0, 0],
            &[0, 0, 156, 255, 156, 88, 116, 232, 239, 31, 0, 0, 0],
            &[0, 0, 6, 146, 250, 255, 255, 208, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 38, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '1' / 0x31
        '1' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 119, 144, 1, 0, 0, 0, 0],
            &[0, 0, 0, 13, 124, 239, 255, 255, 2, 0, 0, 0, 0],
            &[0, 0, 7, 226, 253, 169, 177, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 87, 45, 0, 151, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 156, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 18, 52, 69, 242, 255, 78, 59, 38, 0, 0],
            &[0, 0, 0, 121, 255, 255, 255, 255, 255, 255, 225, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '2' / 0x32
        '2' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 88, 156, 185, 170, 99, 4, 0, 0, 0, 0],
            &[0, 45, 218, 255, 231, 198, 232, 255, 189, 5, 0, 0, 0],
            &[0, 26, 206, 88, 1, 0, 5, 158, 255, 91, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 41, 255, 148, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 32, 255, 149, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 97, 255, 99, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 220, 237, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 169, 255, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 151, 255, 130, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 143, 255, 150, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 147, 255, 148, 1, 0, 0, 0, 0, 0, 0],
            &[0, 1, 151, 255, 141, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 255, 231, 133, 133, 133, 133, 133, 133, 47, 0, 0],
            &[0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '3' / 0x33
        '3' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 106, 167, 187, 170, 103, 5, 0, 0, 0],
            &[0, 0, 62, 240, 255, 220, 197, 233, 255, 195, 6, 0, 0],
            &[0, 0, 12, 153, 48, 0, 0, 5, 163, 255, 91, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 58, 255, 134, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 80, 255, 109, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 38, 217, 230, 19, 0, 0],
            &[0, 0, 0, 0, 118, 197, 215, 245, 158, 31, 0, 0, 0],
            &[0, 0, 0, 0, 114, 191, 202, 241, 233, 128, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 122, 255, 131, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 220, 226, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 210, 239, 0, 0],
            &[0, 0, 9, 0, 0, 0, 0, 0, 50, 252, 192, 0, 0],
            &[0, 0, 151, 199, 128, 96, 93, 137, 241, 251, 64, 0, 0],
            &[0, 0, 86, 218, 255, 255, 255, 253, 195, 61, 0, 0, 0],
            &[0, 0, 0, 0, 13, 37, 37, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '4' / 0x34
        '4' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 108, 164, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 44, 251, 255, 121, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 194, 196, 255, 121, 0, 0, 0],
            &[0, 0, 0, 0, 0, 95, 254, 78, 255, 121, 0, 0, 0],
            &[0, 0, 0, 0, 16, 233, 155, 35, 255, 121, 0, 0, 0],
            &[0, 0, 0, 0, 150, 237, 20, 39, 255, 121, 0, 0, 0],
            &[0, 0, 0, 52, 253, 103, 0, 40, 255, 121, 0, 0, 0],
            &[0, 0, 2, 202, 203, 2, 0, 40, 255, 121, 0, 0, 0],
            &[0, 0, 104, 253, 53, 0, 0, 40, 255, 121, 0, 0, 0],
            &[0, 6, 238, 224, 128, 128, 128, 149, 255, 189, 128, 57, 0],
            &[0, 12, 245, 245, 245, 245, 245, 247, 255, 250, 245, 110, 0],
            &[0, 0, 0, 0, 0, 0, 0, 40, 255, 121, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 40, 255, 121, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 40, 255, 121, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '5' / 0x35
        '5' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 102, 144, 144, 144, 144, 144, 144, 11, 0, 0],
            &[0, 0, 0, 196, 253, 244, 244, 244, 244, 244, 20, 0, 0],
            &[0, 0, 0, 217, 186, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 237, 163, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 254, 139, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 255, 142, 69, 77, 50, 2, 0, 0, 0, 0],
            &[0, 0, 37, 255, 255, 255, 255, 255, 227, 77, 0, 0, 0],
            &[0, 0, 0, 53, 50, 24, 32, 99, 234, 250, 56, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 63, 255, 168, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 239, 209, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 248, 200, 0, 0],
            &[0, 0, 8, 0, 0, 0, 0, 0, 98, 255, 137, 0, 0],
            &[0, 0, 122, 190, 119, 89, 102, 166, 252, 226, 22, 0, 0],
            &[0, 0, 65, 203, 255, 255, 255, 246, 166, 27, 0, 0, 0],
            &[0, 0, 0, 0, 11, 37, 34, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '6' / 0x36
        '6' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 131, 170, 189, 174, 21, 0, 0, 0],
            &[0, 0, 0, 135, 254, 249, 199, 180, 200, 32, 0, 0, 0],
            &[0, 0, 118, 255, 160, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 19, 244, 186, 1, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 255, 66, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 160, 254, 9, 61, 126, 142, 100, 13, 0, 0, 0, 0],
            &[0, 195, 231, 143, 248, 201, 214, 253, 229, 43, 0, 0, 0],
            &[0, 212, 252, 186, 21, 0, 0, 55, 244, 206, 0, 0, 0],
            &[0, 214, 246, 17, 0, 0, 0, 0, 144, 255, 34, 0, 0],
            &[0, 196, 222, 0, 0, 0, 0, 0, 105, 255, 60, 0, 0],
            &[0, 150, 250, 16, 0, 0, 0, 0, 125, 255, 43, 0, 0],
            &[0, 66, 255, 134, 0, 0, 0, 3, 209, 232, 3, 0, 0],
            &[0, 0, 177, 254, 157, 77, 88, 192, 255, 102, 0, 0, 0],
            &[0, 0, 9, 150, 246, 255, 255, 228, 100, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 39, 30, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '7' / 0x37
        '7' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 124, 144, 144, 144, 144, 144, 144, 144, 144, 37, 0, 0],
            &[0, 209, 244, 244, 244, 244, 244, 244, 249, 255, 63, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 180, 236, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 37, 254, 137, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 253, 32, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 14, 242, 177, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 112, 255, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 221, 216, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 78, 255, 109, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 189, 243, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 45, 255, 149, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 255, 42, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 247, 190, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 122, 255, 82, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '8' / 0x38
        '8' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 64, 153, 186, 174, 110, 10, 0, 0, 0, 0],
            &[0, 0, 128, 255, 228, 180, 204, 255, 213, 16, 0, 0, 0],
            &[0, 26, 252, 188, 5, 0, 0, 99, 255, 120, 0, 0, 0],
            &[0, 61, 255, 108, 0, 0, 0, 14, 255, 157, 0, 0, 0],
            &[0, 25, 253, 161, 0, 0, 0, 73, 255, 116, 0, 0, 0],
            &[0, 0, 129, 255, 147, 14, 74, 236, 210, 11, 0, 0, 0],
            &[0, 0, 1, 123, 255, 247, 255, 158, 14, 0, 0, 0, 0],
            &[0, 0, 38, 193, 247, 174, 249, 225, 74, 0, 0, 0, 0],
            &[0, 25, 233, 223, 34, 0, 32, 192, 253, 86, 0, 0, 0],
            &[0, 134, 255, 63, 0, 0, 0, 7, 220, 227, 0, 0, 0],
            &[0, 173, 254, 6, 0, 0, 0, 0, 159, 255, 14, 0, 0],
            &[0, 145, 255, 52, 0, 0, 0, 5, 213, 237, 0, 0, 0],
            &[0, 42, 245, 224, 99, 59, 81, 188, 255, 113, 0, 0, 0],
            &[0, 0, 58, 202, 255, 255, 255, 224, 102, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 42, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '9' / 0x39
        '9' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 77, 160, 186, 167, 91, 2, 0, 0, 0, 0],
            &[0, 0, 161, 255, 224, 180, 215, 255, 180, 4, 0, 0, 0],
            &[0, 90, 255, 145, 3, 0, 0, 132, 255, 111, 0, 0, 0],
            &[0, 180, 243, 8, 0, 0, 0, 0, 205, 216, 0, 0, 0],
            &[0, 212, 209, 0, 0, 0, 0, 0, 129, 255, 20, 0, 0],
            &[0, 203, 223, 0, 0, 0, 0, 0, 133, 255, 51, 0, 0],
            &[0, 146, 254, 53, 0, 0, 0, 23, 229, 255, 62, 0, 0],
            &[0, 31, 237, 236, 123, 86, 122, 228, 201, 255, 49, 0, 0],
            &[0, 0, 39, 175, 242, 249, 208, 86, 149, 255, 29, 0, 0],
            &[0, 0, 0, 0, 0, 1, 0, 0, 192, 238, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 32, 251, 162, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 190, 252, 47, 0, 0, 0],
            &[0, 0, 58, 74, 75, 116, 218, 255, 111, 0, 0, 0, 0],
            &[0, 0, 140, 255, 255, 254, 201, 74, 0, 0, 0, 0, 0],
            &[0, 0, 8, 39, 40, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ':' / 0x3a
        ':' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 69, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 96, 255, 203, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 78, 254, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 75, 27, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 200, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 254, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ';' / 0x3b
        ';' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 62, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 242, 255, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 226, 251, 45, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 114, 115, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 255, 247, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 255, 174, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 129, 255, 80, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 229, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 94, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '<' / 0x3c
        '<' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 63, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 83, 220, 175, 0, 0],
            &[0, 0, 0, 0, 0, 0, 54, 195, 251, 149, 22, 0, 0],
            &[0, 0, 0, 0, 31, 165, 254, 174, 38, 0, 0, 0, 0],
            &[0, 0, 12, 135, 247, 198, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 255, 222, 48, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 64, 190, 255, 180, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 60, 187, 255, 189, 66, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 57, 183, 255, 150, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 53, 110, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '=' / 0x3d
        '=' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 64, 64, 64, 64, 64, 64, 64, 64, 4, 0, 0],
            &[0, 171, 255, 255, 255, 255, 255, 255, 255, 255, 17, 0, 0],
            &[0, 45, 68, 68, 68, 68, 68, 68, 68, 68, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 110, 164, 164, 164, 164, 164, 164, 164, 164, 11, 0, 0],
            &[0, 150, 224, 224, 224, 224, 224, 224, 224, 224, 15, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '>' / 0x3e
        '>' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 34, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 73, 249, 140, 16, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 93, 225, 236, 110, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 118, 239, 217, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 143, 249, 192, 49, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 154, 255, 171, 0, 0],
            &[0, 0, 0, 0, 0, 19, 130, 239, 231, 115, 10, 0, 0],
            &[0, 0, 0, 25, 140, 244, 229, 111, 9, 0, 0, 0, 0],
            &[0, 0, 56, 247, 227, 107, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 53, 104, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '?' / 0x3f
        '?' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 39, 125, 174, 188, 171, 107, 8, 0, 0, 0],
            &[0, 0, 29, 253, 254, 214, 191, 218, 255, 211, 16, 0, 0],
            &[0, 0, 0, 92, 30, 0, 0, 0, 105, 255, 124, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 248, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 16, 255, 141, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 146, 254, 59, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 158, 255, 127, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 187, 251, 102, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 96, 255, 87, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 149, 233, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 35, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 70, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 233, 255, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 210, 251, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '@' / 0x40
        '@' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 81, 128, 128, 81, 3, 0, 0, 0, 0],
            &[0, 0, 53, 222, 244, 197, 189, 246, 213, 30, 0, 0, 0],
            &[0, 36, 238, 170, 20, 0, 0, 21, 193, 215, 5, 0, 0],
            &[0, 177, 197, 4, 0, 0, 0, 0, 18, 236, 106, 0, 0],
            &[28, 254, 68, 0, 96, 201, 214, 180, 60, 140, 190, 0, 0],
            &[101, 240, 2, 96, 254, 124, 105, 255, 108, 78, 248, 0, 0],
            &[147, 189, 0, 221, 175, 0, 21, 255, 94, 49, 255, 14, 0],
            &[170, 167, 23, 255, 107, 0, 35, 255, 80, 44, 255, 23, 0],
            &[178, 157, 36, 255, 85, 0, 59, 255, 67, 56, 255, 6, 0],
            &[160, 174, 11, 254, 110, 0, 104, 255, 78, 92, 217, 0, 0],
            &[132, 212, 0, 187, 216, 83, 218, 187, 167, 198, 138, 0, 0],
            &[65, 255, 28, 31, 192, 228, 117, 26, 212, 195, 14, 0, 0],
            &[5, 223, 154, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 80, 254, 125, 5, 0, 0, 0, 8, 24, 0, 0, 0],
            &[0, 0, 98, 247, 228, 172, 159, 188, 241, 106, 0, 0, 0],
            &[0, 0, 0, 22, 115, 155, 165, 136, 71, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'A' / 0x41
        'A' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 159, 110, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 255, 239, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 228, 214, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 97, 242, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 252, 18, 173, 235, 3, 0, 0, 0, 0],
            &[0, 0, 0, 222, 191, 0, 94, 255, 65, 0, 0, 0, 0],
            &[0, 0, 49, 255, 112, 0, 18, 251, 149, 0, 0, 0, 0],
            &[0, 0, 132, 255, 32, 0, 0, 191, 230, 1, 0, 0, 0],
            &[0, 0, 215, 255, 174, 174, 174, 255, 255, 59, 0, 0, 0],
            &[0, 41, 255, 255, 214, 214, 214, 247, 255, 142, 0, 0, 0],
            &[0, 124, 255, 50, 0, 0, 0, 0, 208, 225, 0, 0, 0],
            &[0, 207, 225, 0, 0, 0, 0, 0, 129, 255, 53, 0, 0],
            &[34, 255, 147, 0, 0, 0, 0, 0, 49, 255, 136, 0, 0],
            &[117, 255, 67, 0, 0, 0, 0, 0, 0, 225, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'B' / 0x42
        'B' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 144, 144, 144, 141, 117, 75, 4, 0, 0, 0, 0],
            &[0, 49, 255, 243, 234, 239, 254, 255, 227, 48, 0, 0, 0],
            &[0, 49, 255, 107, 0, 0, 9, 110, 255, 194, 0, 0, 0],
            &[0, 49, 255, 107, 0, 0, 0, 0, 214, 241, 0, 0, 0],
            &[0, 49, 255, 107, 0, 0, 0, 0, 213, 223, 0, 0, 0],
            &[0, 49, 255, 107, 0, 0, 0, 93, 255, 130, 0, 0, 0],
            &[0, 49, 255, 230, 211, 214, 238, 227, 112, 2, 0, 0, 0],
            &[0, 49, 255, 204, 167, 168, 187, 243, 200, 59, 0, 0, 0],
            &[0, 49, 255, 107, 0, 0, 0, 21, 218, 240, 16, 0, 0],
            &[0, 49, 255, 107, 0, 0, 0, 0, 131, 255, 76, 0, 0],
            &[0, 49, 255, 107, 0, 0, 0, 0, 118, 255, 78, 0, 0],
            &[0, 49, 255, 107, 0, 0, 0, 6, 206, 253, 28, 0, 0],
            &[0, 49, 255, 179, 123, 124, 148, 220, 255, 145, 0, 0, 0],
            &[0, 49, 255, 255, 255, 255, 244, 205, 104, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'C' / 0x43
        'C' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 143, 181, 179, 149, 79, 5, 0, 0],
            &[0, 0, 1, 146, 254, 251, 208, 210, 243, 255, 59, 0, 0],
            &[0, 0, 124, 255, 177, 22, 0, 0, 6, 68, 0, 0, 0],
            &[0, 18, 247, 214, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 255, 103, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 154, 255, 39, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 179, 255, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 255, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 255, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 255, 69, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 255, 154, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 203, 250, 70, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 237, 252, 169, 114, 112, 148, 186, 0, 0, 0],
            &[0, 0, 0, 41, 172, 252, 255, 255, 250, 168, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 34, 35, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'D' / 0x44
        'D' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 144, 142, 119, 73, 3, 0, 0, 0, 0, 0],
            &[0, 171, 255, 235, 239, 255, 255, 225, 59, 0, 0, 0, 0],
            &[0, 171, 255, 10, 0, 12, 97, 238, 249, 51, 0, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 48, 250, 206, 0, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 43, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 92, 255, 102, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 65, 255, 125, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 57, 255, 129, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 78, 255, 107, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 134, 255, 63, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 12, 229, 233, 3, 0, 0],
            &[0, 171, 255, 10, 0, 0, 15, 180, 255, 99, 0, 0, 0],
            &[0, 171, 255, 128, 131, 166, 239, 255, 139, 0, 0, 0, 0],
            &[0, 171, 255, 255, 248, 221, 165, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'E' / 0x45
        'E' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 144, 144, 144, 144, 144, 144, 144, 113, 0, 0],
            &[0, 0, 49, 255, 250, 244, 244, 244, 244, 244, 191, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 236, 216, 216, 216, 216, 216, 91, 0, 0],
            &[0, 0, 49, 255, 215, 172, 172, 172, 172, 172, 73, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 196, 133, 133, 133, 133, 133, 104, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'F' / 0x46
        'F' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 144, 144, 144, 144, 144, 144, 144, 113, 0, 0],
            &[0, 0, 49, 255, 250, 244, 244, 244, 244, 244, 191, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 163, 66, 66, 66, 66, 66, 27, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 108, 0, 0],
            &[0, 0, 49, 255, 164, 67, 67, 67, 67, 67, 28, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'G' / 0x47
        'G' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 90, 156, 184, 168, 126, 33, 0, 0],
            &[0, 0, 0, 31, 214, 255, 243, 210, 226, 253, 184, 0, 0],
            &[0, 0, 6, 216, 253, 117, 7, 0, 0, 31, 43, 0, 0],
            &[0, 0, 108, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 191, 249, 14, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 249, 198, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 255, 169, 0, 0, 2, 41, 41, 41, 41, 2, 0],
            &[0, 32, 255, 158, 0, 0, 13, 255, 255, 255, 255, 17, 0],
            &[0, 15, 255, 174, 0, 0, 4, 91, 91, 187, 255, 17, 0],
            &[0, 0, 243, 217, 0, 0, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 178, 255, 43, 0, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 77, 255, 193, 7, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 0, 162, 255, 219, 121, 91, 109, 209, 255, 17, 0],
            &[0, 0, 0, 1, 114, 224, 255, 255, 255, 240, 168, 8, 0],
            &[0, 0, 0, 0, 0, 0, 20, 41, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'H' / 0x48
        'H' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 0, 92, 144, 10, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 218, 216, 216, 216, 216, 241, 255, 17, 0, 0],
            &[0, 171, 255, 175, 172, 172, 172, 172, 225, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'I' / 0x49
        'I' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 144, 144, 144, 144, 144, 144, 44, 0, 0, 0],
            &[0, 0, 152, 189, 218, 255, 236, 197, 175, 49, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 51, 78, 126, 255, 185, 86, 64, 15, 0, 0, 0],
            &[0, 0, 232, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'J' / 0x4a
        'J' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 126, 120, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 225, 213, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 243, 199, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 73, 255, 154, 0, 0, 0],
            &[0, 0, 24, 200, 137, 109, 147, 241, 251, 50, 0, 0, 0],
            &[0, 0, 16, 208, 255, 255, 255, 223, 80, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 40, 29, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'K' / 0x4b
        'K' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 1, 127, 141, 13, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 121, 255, 125, 0, 0, 0],
            &[0, 171, 255, 10, 0, 0, 60, 251, 189, 2, 0, 0, 0],
            &[0, 171, 255, 10, 0, 20, 227, 233, 25, 0, 0, 0, 0],
            &[0, 171, 255, 10, 1, 181, 254, 70, 0, 0, 0, 0, 0],
            &[0, 171, 255, 10, 117, 255, 135, 0, 0, 0, 0, 0, 0],
            &[0, 171, 255, 58, 249, 230, 4, 0, 0, 0, 0, 0, 0],
            &[0, 171, 255, 212, 241, 255, 83, 0, 0, 0, 0, 0, 0],
            &[0, 171, 255, 215, 30, 227, 230, 15, 0, 0, 0, 0, 0],
            &[0, 171, 255, 37, 0, 80, 255, 152, 0, 0, 0, 0, 0],
            &[0, 171, 255, 10, 0, 0, 177, 254, 60, 0, 0, 0, 0],
            &[0, 171, 255, 10, 0, 0, 29, 243, 214, 6, 0, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 113, 255, 127, 0, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 3, 206, 249, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'L' / 0x4c
        'L' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 244, 147, 147, 147, 147, 147, 115, 0, 0, 0],
            &[0, 0, 208, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'M' / 0x4d
        'M' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 144, 144, 69, 0, 0, 0, 14, 144, 144, 65, 0],
            &[0, 12, 255, 255, 180, 0, 0, 0, 82, 255, 255, 115, 0],
            &[0, 12, 255, 221, 245, 7, 0, 0, 155, 226, 251, 115, 0],
            &[0, 12, 255, 168, 255, 69, 0, 0, 227, 165, 255, 115, 0],
            &[0, 12, 255, 132, 233, 142, 0, 44, 255, 103, 255, 115, 0],
            &[0, 12, 255, 139, 158, 214, 0, 117, 245, 45, 255, 115, 0],
            &[0, 12, 255, 143, 82, 255, 31, 190, 178, 41, 255, 115, 0],
            &[0, 12, 255, 144, 13, 249, 117, 249, 102, 42, 255, 115, 0],
            &[0, 12, 255, 144, 0, 186, 238, 255, 27, 42, 255, 115, 0],
            &[0, 12, 255, 144, 0, 110, 255, 207, 0, 42, 255, 115, 0],
            &[0, 12, 255, 144, 0, 30, 158, 90, 0, 42, 255, 115, 0],
            &[0, 12, 255, 144, 0, 0, 0, 0, 0, 42, 255, 115, 0],
            &[0, 12, 255, 144, 0, 0, 0, 0, 0, 42, 255, 115, 0],
            &[0, 12, 255, 144, 0, 0, 0, 0, 0, 42, 255, 115, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'N' / 0x4e
        'N' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 103, 0, 0, 0, 0, 84, 144, 10, 0, 0],
            &[0, 171, 255, 248, 22, 0, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 217, 247, 126, 0, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 215, 155, 231, 6, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 234, 44, 255, 94, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 246, 0, 187, 205, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 74, 255, 61, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 1, 216, 173, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 106, 252, 32, 146, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 11, 238, 141, 140, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 137, 240, 141, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 29, 251, 220, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 0, 168, 255, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 0, 56, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'O' / 0x4f
        'O' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 157, 192, 181, 112, 8, 0, 0, 0],
            &[0, 0, 0, 108, 255, 243, 197, 216, 255, 205, 12, 0, 0],
            &[0, 0, 41, 253, 203, 21, 0, 0, 101, 255, 146, 0, 0],
            &[0, 0, 152, 255, 50, 0, 0, 0, 0, 203, 246, 10, 0],
            &[0, 0, 217, 223, 0, 0, 0, 0, 0, 126, 255, 71, 0],
            &[0, 8, 255, 180, 0, 0, 0, 0, 0, 84, 255, 112, 0],
            &[0, 25, 255, 162, 0, 0, 0, 0, 0, 65, 255, 131, 0],
            &[0, 33, 255, 158, 0, 0, 0, 0, 0, 56, 255, 135, 0],
            &[0, 20, 255, 170, 0, 0, 0, 0, 0, 68, 255, 122, 0],
            &[0, 0, 243, 204, 0, 0, 0, 0, 0, 100, 255, 88, 0],
            &[0, 0, 188, 249, 17, 0, 0, 0, 0, 161, 255, 31, 0],
            &[0, 0, 97, 255, 134, 0, 0, 0, 36, 245, 195, 0, 0],
            &[0, 0, 4, 199, 254, 162, 96, 120, 231, 248, 53, 0, 0],
            &[0, 0, 0, 17, 164, 250, 255, 255, 210, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'P' / 0x50
        'P' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 144, 144, 144, 144, 132, 92, 15, 0, 0, 0, 0],
            &[0, 49, 255, 245, 234, 237, 254, 255, 239, 67, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 6, 89, 248, 233, 9, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 146, 255, 66, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 108, 255, 84, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 140, 255, 53, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 58, 240, 219, 1, 0, 0],
            &[0, 49, 255, 219, 180, 185, 215, 255, 236, 55, 0, 0, 0],
            &[0, 49, 255, 227, 198, 194, 172, 119, 23, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Q' / 0x51
        'Q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 157, 192, 181, 112, 8, 0, 0, 0],
            &[0, 0, 0, 108, 255, 243, 197, 216, 255, 205, 12, 0, 0],
            &[0, 0, 41, 253, 203, 21, 0, 0, 101, 255, 146, 0, 0],
            &[0, 0, 152, 255, 50, 0, 0, 0, 0, 203, 246, 10, 0],
            &[0, 0, 217, 223, 0, 0, 0, 0, 0, 126, 255, 71, 0],
            &[0, 8, 255, 180, 0, 0, 0, 0, 0, 84, 255, 112, 0],
            &[0, 25, 255, 162, 0, 0, 0, 0, 0, 65, 255, 131, 0],
            &[0, 33, 255, 158, 0, 0, 0, 0, 0, 56, 255, 135, 0],
            &[0, 20, 255, 170, 0, 0, 0, 0, 0, 68, 255, 122, 0],
            &[0, 0, 243, 204, 0, 0, 0, 0, 0, 100, 255, 88, 0],
            &[0, 0, 188, 249, 17, 0, 0, 0, 0, 161, 255, 31, 0],
            &[0, 0, 97, 255, 134, 0, 0, 0, 36, 245, 195, 0, 0],
            &[0, 0, 4, 199, 254, 162, 96, 123, 255, 248, 53, 0, 0],
            &[0, 0, 0, 17, 164, 250, 255, 255, 255, 117, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 40, 55, 249, 189, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 149, 255, 68, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 28, 248, 202, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 78, 107, 18, 0],
        ],
        // letter: 'R' / 0x52
        'R' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 144, 144, 144, 135, 108, 40, 0, 0, 0, 0, 0],
            &[0, 49, 255, 245, 234, 249, 255, 254, 149, 1, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 41, 193, 255, 94, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 45, 255, 169, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 12, 255, 183, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 58, 255, 144, 0, 0, 0],
            &[0, 49, 255, 135, 7, 18, 71, 217, 244, 37, 0, 0, 0],
            &[0, 49, 255, 255, 255, 255, 255, 199, 51, 0, 0, 0, 0],
            &[0, 49, 255, 187, 115, 143, 255, 129, 0, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 189, 248, 35, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 43, 251, 181, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 143, 255, 79, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 15, 233, 223, 9, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 98, 255, 130, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'S' / 0x53
        'S' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 129, 177, 184, 157, 109, 25, 0, 0, 0],
            &[0, 0, 62, 245, 254, 224, 205, 234, 255, 151, 0, 0, 0],
            &[0, 0, 210, 247, 70, 0, 0, 0, 46, 32, 0, 0, 0],
            &[0, 9, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 252, 210, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 177, 255, 145, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 200, 255, 237, 129, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 89, 215, 255, 243, 124, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 80, 218, 255, 133, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 234, 240, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 188, 255, 6, 0, 0],
            &[0, 3, 0, 0, 0, 0, 0, 14, 236, 228, 0, 0, 0],
            &[0, 98, 204, 128, 85, 74, 104, 210, 255, 112, 0, 0, 0],
            &[0, 53, 194, 252, 255, 255, 255, 224, 105, 0, 0, 0, 0],
            &[0, 0, 0, 5, 28, 43, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'T' / 0x54
        'T' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 144, 144, 144, 144, 144, 144, 144, 78, 0],
            &[0, 35, 244, 244, 244, 245, 255, 250, 244, 244, 244, 133, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'U' / 0x55
        'U' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 0, 92, 144, 10, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 164, 255, 16, 0, 0, 0, 0, 176, 255, 11, 0, 0],
            &[0, 142, 255, 47, 0, 0, 0, 0, 206, 243, 0, 0, 0],
            &[0, 82, 255, 138, 0, 0, 0, 36, 252, 181, 0, 0, 0],
            &[0, 4, 214, 254, 151, 93, 115, 225, 252, 60, 0, 0, 0],
            &[0, 0, 27, 183, 254, 255, 255, 219, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 40, 24, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'V' / 0x56
        'V' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 62, 144, 50, 0, 0, 0, 0, 0, 0, 138, 121, 0],
            &[0, 46, 255, 148, 0, 0, 0, 0, 0, 48, 255, 150, 0],
            &[0, 0, 219, 224, 0, 0, 0, 0, 0, 122, 255, 69, 0],
            &[0, 0, 137, 255, 45, 0, 0, 0, 0, 197, 239, 5, 0],
            &[0, 0, 55, 255, 122, 0, 0, 0, 18, 252, 163, 0, 0],
            &[0, 0, 1, 227, 199, 0, 0, 0, 90, 255, 82, 0, 0],
            &[0, 0, 0, 146, 253, 22, 0, 0, 164, 246, 10, 0, 0],
            &[0, 0, 0, 63, 255, 96, 0, 2, 236, 176, 0, 0, 0],
            &[0, 0, 0, 3, 234, 173, 0, 57, 255, 95, 0, 0, 0],
            &[0, 0, 0, 0, 154, 242, 6, 130, 251, 18, 0, 0, 0],
            &[0, 0, 0, 0, 72, 255, 65, 200, 189, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 239, 143, 251, 108, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 163, 243, 254, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 255, 201, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'W' / 0x57
        'W' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 71, 144, 11, 0, 0, 0, 0, 0, 0, 98, 129, 0],
            &[0, 103, 255, 39, 0, 0, 0, 0, 0, 0, 190, 209, 0],
            &[0, 75, 255, 63, 0, 0, 0, 0, 0, 0, 211, 183, 0],
            &[0, 46, 255, 88, 0, 5, 47, 26, 0, 0, 233, 157, 0],
            &[0, 17, 255, 112, 0, 65, 255, 179, 0, 2, 253, 131, 0],
            &[0, 0, 244, 136, 0, 125, 237, 238, 1, 22, 255, 105, 0],
            &[0, 0, 216, 160, 0, 184, 151, 254, 45, 44, 255, 80, 0],
            &[0, 0, 187, 184, 2, 241, 82, 216, 106, 65, 255, 54, 0],
            &[0, 0, 158, 204, 47, 255, 24, 156, 168, 83, 255, 28, 0],
            &[0, 0, 130, 223, 104, 220, 0, 94, 228, 99, 253, 4, 0],
            &[0, 0, 101, 240, 158, 161, 0, 33, 255, 144, 232, 0, 0],
            &[0, 0, 72, 253, 210, 101, 0, 0, 227, 207, 207, 0, 0],
            &[0, 0, 44, 255, 254, 42, 0, 0, 165, 253, 181, 0, 0],
            &[0, 0, 15, 255, 237, 1, 0, 0, 104, 255, 155, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'X' / 0x58
        'X' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 126, 135, 2, 0, 0, 0, 0, 74, 144, 40, 0],
            &[0, 0, 104, 255, 99, 0, 0, 0, 13, 232, 205, 2, 0],
            &[0, 0, 2, 203, 231, 12, 0, 0, 135, 254, 56, 0, 0],
            &[0, 0, 0, 53, 253, 130, 0, 32, 247, 157, 0, 0, 0],
            &[0, 0, 0, 0, 153, 245, 28, 169, 238, 21, 0, 0, 0],
            &[0, 0, 0, 0, 18, 236, 200, 255, 107, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 100, 255, 206, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 104, 255, 205, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 237, 199, 254, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 157, 245, 28, 161, 239, 21, 0, 0, 0],
            &[0, 0, 0, 57, 254, 127, 0, 26, 243, 159, 0, 0, 0],
            &[0, 0, 3, 207, 226, 10, 0, 0, 121, 254, 59, 0, 0],
            &[0, 0, 109, 255, 89, 0, 0, 0, 7, 222, 208, 3, 0],
            &[0, 23, 240, 197, 0, 0, 0, 0, 0, 82, 255, 110, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Y' / 0x59
        'Y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 27, 144, 98, 0, 0, 0, 0, 0, 31, 144, 85, 0],
            &[0, 0, 202, 245, 20, 0, 0, 0, 0, 149, 254, 50, 0],
            &[0, 0, 72, 255, 128, 0, 0, 0, 22, 246, 174, 0, 0],
            &[0, 0, 0, 197, 236, 10, 0, 0, 132, 254, 46, 0, 0],
            &[0, 0, 0, 67, 255, 109, 0, 12, 238, 170, 0, 0, 0],
            &[0, 0, 0, 0, 193, 223, 4, 114, 253, 42, 0, 0, 0],
            &[0, 0, 0, 0, 63, 255, 95, 227, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 188, 245, 252, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 255, 165, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Z' / 0x5a
        'Z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 118, 144, 144, 144, 144, 144, 144, 86, 0, 0],
            &[0, 0, 0, 191, 234, 234, 234, 234, 238, 255, 148, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 127, 254, 51, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 244, 163, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 247, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 253, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 190, 234, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 78, 255, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 216, 215, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 109, 255, 77, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 236, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 140, 254, 51, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 249, 229, 133, 133, 133, 133, 133, 79, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 152, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '[' / 0x5b
        '[' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 144, 144, 144, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 215, 205, 171, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 125, 93, 78, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 255, 255, 255, 213, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '\' / 0x5c
        '\\' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 88, 142, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 255, 82, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 232, 180, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 252, 26, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 41, 255, 121, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 198, 218, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 255, 61, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 244, 160, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 159, 244, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 255, 100, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 218, 198, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 121, 255, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 252, 139, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 180, 232, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ']' / 0x5d
        ']' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 193, 193, 193, 162, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 156, 156, 235, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 203, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 203, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 203, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 203, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 203, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 203, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 203, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 203, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 203, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 203, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 203, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 203, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 203, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 61, 142, 142, 232, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 207, 207, 207, 174, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '^' / 0x5e
        '^' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 168, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 136, 255, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 241, 157, 249, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 122, 222, 3, 200, 157, 0, 0, 0, 0],
            &[0, 0, 0, 9, 234, 109, 0, 70, 252, 39, 0, 0, 0],
            &[0, 0, 0, 109, 238, 11, 0, 0, 196, 169, 0, 0, 0],
            &[0, 0, 5, 225, 134, 0, 0, 0, 67, 254, 48, 0, 0],
            &[0, 0, 95, 249, 25, 0, 0, 0, 0, 193, 180, 0, 0],
            &[0, 0, 196, 152, 0, 0, 0, 0, 0, 63, 236, 49, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '_' / 0x5f
        '_' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 60, 0],
            &[236, 236, 236, 236, 236, 236, 236, 236, 236, 236, 236, 94, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '`' / 0x60
        '`' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 45, 141, 104, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 182, 253, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 190, 210, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 114, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'a' / 0x61
        'a' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 69, 90, 69, 24, 0, 0, 0, 0],
            &[0, 0, 0, 148, 248, 255, 255, 255, 250, 142, 2, 0, 0],
            &[0, 0, 0, 105, 115, 41, 12, 36, 167, 255, 91, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 8, 251, 160, 0, 0],
            &[0, 0, 0, 0, 0, 3, 24, 38, 45, 231, 176, 0, 0],
            &[0, 0, 0, 57, 184, 247, 255, 255, 255, 255, 176, 0, 0],
            &[0, 0, 48, 248, 212, 90, 38, 19, 10, 223, 176, 0, 0],
            &[0, 0, 145, 255, 38, 0, 0, 0, 0, 240, 176, 0, 0],
            &[0, 0, 156, 252, 7, 0, 0, 0, 85, 255, 176, 0, 0],
            &[0, 0, 96, 255, 170, 65, 64, 135, 231, 239, 176, 0, 0],
            &[0, 0, 1, 141, 249, 255, 255, 199, 47, 190, 176, 0, 0],
            &[0, 0, 0, 0, 8, 38, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'b' / 0x62
        'b' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 54, 112, 19, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 255, 43, 10, 73, 88, 43, 0, 0, 0, 0, 0],
            &[0, 122, 255, 102, 236, 255, 255, 255, 172, 8, 0, 0, 0],
            &[0, 122, 255, 249, 127, 26, 34, 155, 255, 144, 0, 0, 0],
            &[0, 122, 255, 171, 0, 0, 0, 0, 216, 245, 11, 0, 0],
            &[0, 122, 255, 84, 0, 0, 0, 0, 137, 255, 60, 0, 0],
            &[0, 122, 255, 54, 0, 0, 0, 0, 110, 255, 84, 0, 0],
            &[0, 122, 255, 60, 0, 0, 0, 0, 115, 255, 81, 0, 0],
            &[0, 122, 255, 94, 0, 0, 0, 0, 147, 255, 51, 0, 0],
            &[0, 122, 255, 190, 0, 0, 0, 8, 231, 234, 5, 0, 0],
            &[0, 122, 255, 235, 168, 73, 81, 194, 255, 111, 0, 0, 0],
            &[0, 122, 242, 39, 199, 255, 255, 243, 122, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 40, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'c' / 0x63
        'c' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 67, 89, 69, 24, 0, 0, 0],
            &[0, 0, 0, 2, 127, 246, 255, 255, 255, 252, 98, 0, 0],
            &[0, 0, 0, 123, 255, 194, 71, 35, 52, 124, 38, 0, 0],
            &[0, 0, 8, 241, 221, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 134, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 87, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 86, 255, 103, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 255, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 238, 230, 16, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 107, 255, 218, 110, 76, 95, 143, 98, 0, 0],
            &[0, 0, 0, 0, 98, 221, 255, 255, 255, 231, 85, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 42, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'd' / 0x64
        'd' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 99, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 207, 225, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 207, 225, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 207, 225, 0, 0],
            &[0, 0, 0, 0, 11, 69, 86, 45, 0, 206, 225, 0, 0],
            &[0, 0, 0, 91, 241, 255, 255, 255, 163, 195, 225, 0, 0],
            &[0, 0, 49, 253, 231, 74, 15, 54, 216, 251, 225, 0, 0],
            &[0, 0, 156, 255, 78, 0, 0, 0, 60, 255, 225, 0, 0],
            &[0, 0, 215, 243, 2, 0, 0, 0, 1, 240, 225, 0, 0],
            &[0, 0, 237, 216, 0, 0, 0, 0, 0, 213, 225, 0, 0],
            &[0, 0, 234, 215, 0, 0, 0, 0, 0, 214, 225, 0, 0],
            &[0, 0, 205, 242, 3, 0, 0, 0, 2, 242, 225, 0, 0],
            &[0, 0, 138, 255, 83, 0, 0, 0, 80, 255, 225, 0, 0],
            &[0, 0, 28, 241, 242, 110, 63, 107, 239, 233, 225, 0, 0],
            &[0, 0, 0, 55, 212, 255, 255, 239, 98, 152, 225, 0, 0],
            &[0, 0, 0, 0, 0, 29, 36, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'e' / 0x65
        'e' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 81, 79, 28, 0, 0, 0, 0, 0],
            &[0, 0, 23, 185, 255, 255, 255, 252, 145, 3, 0, 0, 0],
            &[0, 6, 209, 245, 105, 23, 21, 133, 255, 124, 0, 0, 0],
            &[0, 101, 255, 105, 0, 0, 0, 0, 192, 236, 4, 0, 0],
            &[0, 178, 252, 46, 35, 35, 35, 35, 146, 255, 43, 0, 0],
            &[0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 64, 0, 0],
            &[0, 205, 239, 59, 59, 59, 59, 59, 59, 59, 15, 0, 0],
            &[0, 171, 254, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 176, 255, 168, 83, 62, 89, 145, 184, 0, 0, 0],
            &[0, 0, 4, 124, 228, 255, 255, 255, 224, 122, 0, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'f' / 0x66
        'f' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 121, 131, 111, 53, 0, 0],
            &[0, 0, 0, 0, 1, 176, 255, 248, 228, 246, 163, 0, 0],
            &[0, 0, 0, 0, 62, 255, 182, 12, 0, 2, 13, 0, 0],
            &[0, 0, 0, 0, 101, 255, 87, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 26, 131, 255, 107, 47, 47, 47, 5, 0, 0],
            &[0, 184, 255, 255, 255, 255, 255, 255, 255, 255, 32, 0, 0],
            &[0, 46, 61, 61, 143, 255, 117, 61, 61, 61, 7, 0, 0],
            &[0, 0, 0, 0, 107, 255, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 255, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 255, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 255, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 255, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 255, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 255, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 255, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'g' / 0x67
        'g' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 75, 88, 45, 0, 23, 41, 0, 0],
            &[0, 0, 0, 79, 239, 255, 255, 255, 160, 159, 225, 0, 0],
            &[0, 0, 38, 248, 226, 67, 14, 49, 201, 253, 225, 0, 0],
            &[0, 0, 148, 255, 68, 0, 0, 0, 30, 251, 225, 0, 0],
            &[0, 0, 211, 240, 1, 0, 0, 0, 0, 203, 225, 0, 0],
            &[0, 0, 237, 213, 0, 0, 0, 0, 0, 175, 225, 0, 0],
            &[0, 0, 236, 218, 0, 0, 0, 0, 0, 175, 225, 0, 0],
            &[0, 0, 206, 246, 6, 0, 0, 0, 0, 205, 225, 0, 0],
            &[0, 0, 138, 255, 94, 0, 0, 0, 41, 253, 225, 0, 0],
            &[0, 0, 25, 237, 247, 118, 63, 94, 224, 249, 225, 0, 0],
            &[0, 0, 0, 49, 206, 255, 255, 251, 146, 181, 225, 0, 0],
            &[0, 0, 0, 0, 0, 26, 42, 13, 0, 192, 218, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 232, 190, 0, 0],
            &[0, 0, 52, 48, 0, 0, 0, 3, 136, 255, 110, 0, 0],
            &[0, 0, 1, 55, 120, 148, 160, 137, 70, 0, 0, 0, 0],
        ],
        // letter: 'h' / 0x68
        'h' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 112, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 58, 2, 57, 85, 52, 1, 0, 0, 0],
            &[0, 0, 122, 255, 77, 207, 255, 255, 255, 198, 25, 0, 0],
            &[0, 0, 122, 255, 225, 138, 23, 24, 149, 255, 163, 0, 0],
            &[0, 0, 122, 255, 180, 0, 0, 0, 4, 233, 230, 0, 0],
            &[0, 0, 122, 255, 93, 0, 0, 0, 0, 195, 248, 0, 0],
            &[0, 0, 122, 255, 64, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'i' / 0x69
        'i' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 80, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 61, 255, 231, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 215, 159, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 47, 47, 47, 47, 40, 0, 0, 0, 0, 0, 0],
            &[0, 44, 255, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 30, 49, 226, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 75, 95, 233, 234, 95, 73, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'j' / 0x6a
        'j' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 19, 90, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 144, 255, 146, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 84, 226, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 47, 47, 47, 47, 25, 0, 0, 0],
            &[0, 0, 0, 0, 125, 255, 255, 255, 255, 138, 0, 0, 0],
            &[0, 0, 0, 0, 1, 17, 36, 87, 255, 138, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 44, 255, 138, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 76, 255, 117, 0, 0, 0],
            &[0, 0, 0, 13, 0, 0, 6, 187, 255, 63, 0, 0, 0],
            &[0, 0, 0, 90, 144, 161, 136, 71, 0, 0, 0, 0, 0],
        ],
        // letter: 'k' / 0x6b
        'k' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 112, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 0, 26, 47, 22, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 43, 238, 217, 21, 0, 0],
            &[0, 1, 255, 180, 0, 0, 29, 228, 227, 30, 0, 0, 0],
            &[0, 1, 255, 179, 0, 18, 214, 236, 42, 0, 0, 0, 0],
            &[0, 1, 255, 173, 9, 198, 243, 54, 0, 0, 0, 0, 0],
            &[0, 1, 255, 162, 171, 255, 195, 2, 0, 0, 0, 0, 0],
            &[0, 1, 255, 238, 232, 139, 255, 123, 0, 0, 0, 0, 0],
            &[0, 1, 255, 232, 38, 0, 175, 252, 61, 0, 0, 0, 0],
            &[0, 1, 255, 180, 0, 0, 18, 227, 228, 20, 0, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 63, 253, 181, 1, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 0, 130, 255, 117, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'l' / 0x6c
        'l' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 112, 112, 112, 112, 107, 0, 0, 0, 0, 0, 0],
            &[0, 70, 204, 221, 239, 254, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 74, 94, 219, 249, 96, 74, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'm' / 0x6d
        'm' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 47, 6, 41, 86, 21, 0, 47, 86, 21, 0, 0],
            &[0, 12, 255, 135, 252, 255, 214, 81, 254, 255, 227, 11, 0],
            &[0, 12, 255, 250, 64, 141, 255, 252, 93, 129, 255, 77, 0],
            &[0, 12, 255, 184, 0, 33, 255, 197, 0, 29, 255, 110, 0],
            &[0, 12, 255, 142, 0, 14, 255, 151, 0, 15, 255, 119, 0],
            &[0, 12, 255, 126, 0, 10, 255, 131, 0, 13, 255, 120, 0],
            &[0, 12, 255, 120, 0, 10, 255, 122, 0, 13, 255, 120, 0],
            &[0, 12, 255, 120, 0, 10, 255, 122, 0, 13, 255, 120, 0],
            &[0, 12, 255, 120, 0, 10, 255, 122, 0, 13, 255, 120, 0],
            &[0, 12, 255, 120, 0, 10, 255, 122, 0, 13, 255, 120, 0],
            &[0, 12, 255, 120, 0, 10, 255, 122, 0, 13, 255, 120, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'n' / 0x6e
        'n' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 42, 0, 3, 60, 86, 57, 3, 0, 0, 0],
            &[0, 0, 122, 250, 45, 220, 255, 255, 255, 208, 25, 0, 0],
            &[0, 0, 122, 255, 233, 123, 30, 20, 126, 255, 163, 0, 0],
            &[0, 0, 122, 255, 170, 0, 0, 0, 0, 224, 223, 0, 0],
            &[0, 0, 122, 255, 92, 0, 0, 0, 0, 191, 248, 0, 0],
            &[0, 0, 122, 255, 66, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'o' / 0x6f
        'o' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 55, 87, 70, 13, 0, 0, 0, 0, 0],
            &[0, 0, 53, 218, 255, 255, 255, 244, 116, 0, 0, 0, 0],
            &[0, 28, 238, 231, 76, 16, 37, 170, 255, 105, 0, 0, 0],
            &[0, 139, 255, 68, 0, 0, 0, 3, 217, 233, 3, 0, 0],
            &[0, 209, 231, 0, 0, 0, 0, 0, 134, 255, 54, 0, 0],
            &[0, 235, 199, 0, 0, 0, 0, 0, 103, 255, 80, 0, 0],
            &[0, 233, 199, 0, 0, 0, 0, 0, 106, 255, 80, 0, 0],
            &[0, 198, 239, 2, 0, 0, 0, 0, 148, 255, 47, 0, 0],
            &[0, 119, 255, 93, 0, 0, 0, 16, 235, 222, 2, 0, 0],
            &[0, 12, 216, 246, 119, 60, 84, 207, 254, 82, 0, 0, 0],
            &[0, 0, 23, 172, 251, 255, 255, 220, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 42, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'p' / 0x70
        'p' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 44, 0, 11, 68, 87, 45, 0, 0, 0, 0, 0],
            &[0, 122, 254, 65, 231, 255, 255, 255, 176, 10, 0, 0, 0],
            &[0, 122, 255, 240, 105, 21, 33, 144, 255, 148, 0, 0, 0],
            &[0, 122, 255, 151, 0, 0, 0, 0, 207, 246, 12, 0, 0],
            &[0, 122, 255, 81, 0, 0, 0, 0, 133, 255, 61, 0, 0],
            &[0, 122, 255, 57, 0, 0, 0, 0, 109, 255, 84, 0, 0],
            &[0, 122, 255, 68, 0, 0, 0, 0, 115, 255, 78, 0, 0],
            &[0, 122, 255, 99, 0, 0, 0, 0, 146, 255, 51, 0, 0],
            &[0, 122, 255, 185, 0, 0, 0, 6, 229, 235, 2, 0, 0],
            &[0, 122, 255, 236, 158, 71, 81, 192, 255, 112, 0, 0, 0],
            &[0, 122, 255, 76, 189, 255, 255, 242, 123, 0, 0, 0, 0],
            &[0, 122, 255, 53, 0, 18, 38, 7, 0, 0, 0, 0, 0],
            &[0, 122, 255, 53, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 255, 53, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 143, 30, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'q' / 0x71
        'q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 74, 86, 37, 0, 23, 41, 0, 0],
            &[0, 0, 0, 76, 236, 255, 255, 254, 140, 156, 225, 0, 0],
            &[0, 0, 39, 247, 222, 61, 16, 61, 221, 247, 225, 0, 0],
            &[0, 0, 149, 255, 64, 0, 0, 0, 63, 255, 225, 0, 0],
            &[0, 0, 212, 239, 0, 0, 0, 0, 1, 242, 225, 0, 0],
            &[0, 0, 237, 213, 0, 0, 0, 0, 0, 217, 225, 0, 0],
            &[0, 0, 235, 217, 0, 0, 0, 0, 0, 214, 225, 0, 0],
            &[0, 0, 204, 245, 7, 0, 0, 0, 0, 241, 225, 0, 0],
            &[0, 0, 135, 255, 94, 0, 0, 0, 73, 255, 225, 0, 0],
            &[0, 0, 24, 237, 246, 114, 63, 105, 235, 238, 225, 0, 0],
            &[0, 0, 0, 50, 207, 255, 255, 238, 95, 202, 225, 0, 0],
            &[0, 0, 0, 0, 0, 28, 39, 7, 0, 207, 225, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 207, 225, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 207, 225, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 116, 126, 0, 0],
        ],
        // letter: 'r' / 0x72
        'r' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 24, 47, 47, 47, 30, 0, 23, 83, 77, 22, 0, 0],
            &[0, 132, 255, 255, 255, 185, 62, 244, 255, 255, 185, 0, 0],
            &[0, 2, 21, 44, 201, 226, 227, 150, 46, 60, 60, 0, 0],
            &[0, 0, 0, 0, 185, 255, 168, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 1, 0, 0, 0, 0, 0, 0],
            &[0, 37, 66, 90, 214, 255, 97, 71, 46, 0, 0, 0, 0],
            &[0, 210, 255, 255, 255, 255, 255, 255, 247, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 's' / 0x73
        's' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 49, 85, 85, 58, 5, 0, 0, 0],
            &[0, 0, 0, 33, 210, 255, 255, 255, 255, 240, 23, 0, 0],
            &[0, 0, 0, 170, 250, 89, 24, 22, 58, 109, 0, 0, 0],
            &[0, 0, 0, 204, 232, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 128, 255, 174, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 116, 241, 253, 176, 60, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 123, 232, 255, 151, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 173, 255, 73, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 94, 255, 92, 0, 0],
            &[0, 0, 1, 214, 123, 71, 60, 92, 220, 249, 31, 0, 0],
            &[0, 0, 0, 180, 248, 255, 255, 255, 214, 71, 0, 0, 0],
            &[0, 0, 0, 0, 5, 29, 43, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 't' / 0x74
        't' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 202, 126, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 251, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 24, 73, 255, 161, 47, 47, 47, 24, 0, 0],
            &[0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0],
            &[0, 0, 57, 61, 93, 255, 168, 61, 61, 61, 32, 0, 0],
            &[0, 0, 0, 0, 41, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 144, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 255, 134, 64, 67, 83, 0, 0],
            &[0, 0, 0, 0, 0, 41, 202, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 20, 43, 29, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'u' / 0x75
        'u' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 47, 6, 0, 0, 0, 0, 39, 41, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 219, 225, 0, 0],
            &[0, 0, 144, 255, 41, 0, 0, 0, 3, 244, 225, 0, 0],
            &[0, 0, 115, 255, 86, 0, 0, 0, 75, 255, 225, 0, 0],
            &[0, 0, 44, 252, 226, 91, 58, 104, 231, 231, 225, 0, 0],
            &[0, 0, 0, 98, 239, 255, 255, 230, 87, 147, 225, 0, 0],
            &[0, 0, 0, 0, 2, 31, 32, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'v' / 0x76
        'v' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 47, 29, 0, 0, 0, 0, 0, 13, 47, 23, 0],
            &[0, 2, 226, 215, 0, 0, 0, 0, 0, 126, 255, 69, 0],
            &[0, 0, 133, 255, 50, 0, 0, 0, 0, 214, 227, 2, 0],
            &[0, 0, 39, 255, 140, 0, 0, 0, 46, 255, 134, 0, 0],
            &[0, 0, 0, 200, 228, 2, 0, 0, 133, 255, 39, 0, 0],
            &[0, 0, 0, 106, 255, 64, 0, 0, 220, 199, 0, 0, 0],
            &[0, 0, 0, 18, 249, 154, 0, 53, 255, 103, 0, 0, 0],
            &[0, 0, 0, 0, 172, 238, 6, 142, 247, 16, 0, 0, 0],
            &[0, 0, 0, 0, 78, 255, 80, 230, 168, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 234, 220, 255, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 145, 255, 230, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'w' / 0x77
        'w' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[32, 43, 0, 0, 0, 0, 0, 0, 0, 21, 47, 4, 0],
            &[150, 251, 4, 0, 0, 0, 0, 0, 0, 141, 250, 5, 0],
            &[108, 255, 37, 0, 19, 67, 45, 0, 0, 181, 214, 0, 0],
            &[67, 255, 75, 0, 120, 255, 215, 0, 0, 222, 172, 0, 0],
            &[26, 255, 113, 0, 193, 218, 255, 30, 9, 253, 130, 0, 0],
            &[0, 241, 150, 15, 251, 105, 249, 101, 47, 255, 88, 0, 0],
            &[0, 200, 185, 81, 255, 30, 194, 170, 83, 255, 46, 0, 0],
            &[0, 159, 213, 144, 216, 0, 128, 235, 113, 252, 7, 0, 0],
            &[0, 118, 233, 195, 149, 0, 63, 255, 171, 218, 0, 0, 0],
            &[0, 77, 250, 242, 82, 0, 7, 246, 237, 176, 0, 0, 0],
            &[0, 36, 255, 253, 18, 0, 0, 188, 255, 134, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'x' / 0x78
        'x' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 41, 46, 1, 0, 0, 0, 0, 29, 47, 13, 0],
            &[0, 0, 115, 255, 116, 0, 0, 0, 30, 239, 208, 8, 0],
            &[0, 0, 1, 181, 249, 49, 0, 2, 194, 244, 39, 0, 0],
            &[0, 0, 0, 20, 229, 216, 10, 124, 255, 94, 0, 0, 0],
            &[0, 0, 0, 0, 64, 253, 186, 251, 160, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 137, 255, 223, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 201, 255, 250, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 141, 255, 91, 219, 222, 15, 0, 0, 0],
            &[0, 0, 0, 75, 254, 148, 0, 53, 250, 172, 0, 0, 0],
            &[0, 0, 28, 236, 211, 8, 0, 0, 120, 255, 106, 0, 0],
            &[0, 3, 194, 246, 43, 0, 0, 0, 2, 190, 247, 47, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'y' / 0x79
        'y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 47, 26, 0, 0, 0, 0, 0, 11, 47, 23, 0],
            &[0, 1, 218, 205, 0, 0, 0, 0, 0, 115, 255, 69, 0],
            &[0, 0, 113, 255, 54, 0, 0, 0, 0, 200, 227, 2, 0],
            &[0, 0, 17, 246, 158, 0, 0, 0, 31, 255, 135, 0, 0],
            &[0, 0, 0, 156, 245, 16, 0, 0, 116, 255, 40, 0, 0],
            &[0, 0, 0, 50, 255, 110, 0, 0, 201, 201, 0, 0, 0],
            &[0, 0, 0, 0, 199, 213, 0, 32, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 93, 255, 55, 114, 249, 18, 0, 0, 0],
            &[0, 0, 0, 0, 7, 235, 143, 196, 172, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 136, 231, 253, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 253, 233, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 247, 143, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 113, 255, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 232, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 162, 136, 37, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'z' / 0x7a
        'z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 47, 47, 47, 47, 47, 47, 25, 0, 0],
            &[0, 0, 0, 252, 255, 255, 255, 255, 255, 255, 137, 0, 0],
            &[0, 0, 0, 32, 32, 32, 32, 32, 176, 253, 69, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 94, 255, 125, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 246, 179, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 220, 220, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 177, 245, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 121, 255, 91, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 252, 146, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 234, 245, 118, 113, 113, 113, 113, 78, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '{' / 0x7b
        '{' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 27, 107, 101, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 75, 246, 253, 164, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 218, 236, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 254, 172, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 255, 152, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 152, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 86, 255, 227, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 129, 225, 223, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 47, 255, 138, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 244, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 156, 255, 169, 78, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 133, 218, 183, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '|' / 0x7c
        '|' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 110, 45, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 141, 57, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '}' / 0x7d
        '}' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 124, 64, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 71, 237, 255, 172, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 155, 255, 65, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 66, 255, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 107, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 107, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 255, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 231, 218, 46, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 35, 176, 255, 187, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 143, 252, 166, 87, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 255, 152, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 58, 255, 108, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 107, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 107, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 104, 255, 91, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 137, 245, 237, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 237, 174, 42, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '~' / 0x7e
        '~' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 179, 153, 28, 0, 0, 54, 177, 0, 0],
            &[0, 0, 30, 243, 116, 171, 232, 56, 0, 125, 177, 0, 0],
            &[0, 0, 107, 181, 0, 0, 120, 247, 172, 237, 70, 0, 0],
            &[0, 0, 77, 79, 0, 0, 0, 54, 125, 65, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ' ' / 0xa0
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¡' / 0xa1
        '¡' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 34, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 189, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 98, 255, 201, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 62, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 54, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 255, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 255, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 255, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 255, 137, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 145, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 255, 153, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 64, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 255, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 255, 177, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 131, 94, 0, 0, 0, 0, 0],
        ],
        // letter: '¢' / 0xa2
        '¢' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 173, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 113, 235, 42, 10, 0, 0, 0],
            &[0, 0, 0, 0, 73, 214, 255, 255, 255, 250, 117, 0, 0],
            &[0, 0, 0, 71, 252, 219, 99, 60, 77, 127, 56, 0, 0],
            &[0, 0, 0, 211, 240, 23, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 31, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 61, 255, 118, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 66, 255, 115, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 255, 147, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 233, 229, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 110, 255, 198, 72, 36, 53, 95, 80, 0, 0],
            &[0, 0, 0, 0, 117, 242, 255, 255, 255, 254, 114, 0, 0],
            &[0, 0, 0, 0, 0, 10, 134, 235, 69, 27, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 101, 224, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '£' / 0xa3
        '£' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 107, 168, 171, 128, 28, 0, 0],
            &[0, 0, 0, 0, 0, 184, 255, 205, 190, 241, 151, 0, 0],
            &[0, 0, 0, 0, 71, 255, 135, 0, 0, 14, 22, 0, 0],
            &[0, 0, 0, 0, 118, 255, 53, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 85, 174, 255, 110, 85, 85, 43, 0, 0, 0],
            &[0, 0, 54, 235, 246, 255, 238, 235, 235, 120, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 255, 37, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 254, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 224, 180, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 35, 206, 255, 156, 133, 133, 133, 133, 129, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 249, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¤' / 0xa4
        '¤' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 0, 0, 0, 0, 0, 0, 9, 0, 0, 0],
            &[0, 93, 221, 40, 102, 151, 130, 38, 148, 195, 1, 0, 0],
            &[0, 20, 212, 255, 234, 175, 204, 254, 252, 86, 0, 0, 0],
            &[0, 0, 161, 205, 17, 0, 0, 107, 248, 26, 0, 0, 0],
            &[0, 2, 246, 80, 0, 0, 0, 0, 223, 103, 0, 0, 0],
            &[0, 7, 253, 65, 0, 0, 0, 0, 209, 118, 0, 0, 0],
            &[0, 0, 199, 161, 0, 0, 0, 50, 252, 59, 0, 0, 0],
            &[0, 0, 167, 255, 181, 107, 136, 239, 245, 30, 0, 0, 0],
            &[0, 101, 250, 116, 177, 228, 208, 118, 211, 203, 0, 0, 0],
            &[0, 6, 57, 0, 0, 0, 0, 0, 19, 49, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¥' / 0xa5
        '¥' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 27, 144, 82, 0, 0, 0, 0, 0, 22, 144, 85, 0],
            &[0, 0, 201, 233, 10, 0, 0, 0, 0, 139, 254, 47, 0],
            &[0, 0, 70, 255, 113, 0, 0, 0, 21, 244, 167, 0, 0],
            &[0, 0, 0, 195, 230, 8, 0, 0, 138, 251, 37, 0, 0],
            &[0, 0, 0, 64, 255, 108, 0, 21, 244, 156, 0, 0, 0],
            &[0, 0, 0, 0, 189, 227, 6, 137, 248, 29, 0, 0, 0],
            &[0, 0, 0, 0, 58, 255, 124, 244, 144, 0, 0, 0, 0],
            &[0, 0, 0, 98, 112, 231, 255, 255, 135, 112, 30, 0, 0],
            &[0, 0, 0, 172, 198, 206, 255, 226, 198, 198, 53, 0, 0],
            &[0, 0, 0, 0, 0, 34, 255, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 148, 170, 182, 255, 213, 170, 170, 46, 0, 0],
            &[0, 0, 0, 122, 140, 156, 255, 197, 140, 140, 38, 0, 0],
            &[0, 0, 0, 0, 0, 34, 255, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 255, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¦' / 0xa6
        '¦' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 112, 43, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 204, 78, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 168, 64, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 148, 56, 0, 0, 0, 0, 0],
        ],
        // letter: '§' / 0xa7
        '§' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 93, 125, 122, 95, 29, 0, 0, 0, 0],
            &[0, 0, 48, 239, 235, 187, 194, 229, 237, 6, 0, 0, 0],
            &[0, 0, 160, 237, 13, 0, 0, 1, 33, 0, 0, 0, 0],
            &[0, 0, 155, 251, 50, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 226, 248, 156, 42, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 174, 233, 226, 254, 177, 25, 0, 0, 0, 0],
            &[0, 0, 141, 213, 15, 2, 75, 227, 223, 12, 0, 0, 0],
            &[0, 0, 190, 168, 0, 0, 0, 57, 255, 69, 0, 0, 0],
            &[0, 0, 142, 246, 88, 0, 0, 55, 255, 46, 0, 0, 0],
            &[0, 0, 11, 173, 255, 208, 115, 227, 156, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 167, 252, 251, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 34, 205, 248, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 85, 255, 62, 0, 0, 0],
            &[0, 0, 154, 82, 28, 13, 48, 193, 241, 15, 0, 0, 0],
            &[0, 0, 171, 250, 255, 255, 255, 199, 57, 0, 0, 0, 0],
            &[0, 0, 0, 8, 31, 40, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¨' / 0xa8
        '¨' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 66, 239, 84, 0, 4, 218, 167, 0, 0, 0],
            &[0, 0, 0, 82, 255, 103, 0, 9, 239, 193, 0, 0, 0],
            &[0, 0, 0, 0, 22, 0, 0, 0, 15, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '©' / 0xa9
        '©' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 140, 171, 158, 103, 13, 0, 0, 0],
            &[0, 0, 1, 149, 209, 104, 66, 82, 162, 221, 42, 0, 0],
            &[0, 0, 111, 184, 7, 109, 200, 200, 72, 94, 215, 4, 0],
            &[0, 1, 225, 36, 89, 217, 54, 52, 29, 0, 193, 76, 0],
            &[0, 22, 228, 0, 175, 106, 0, 0, 0, 0, 127, 126, 0],
            &[0, 32, 221, 0, 190, 89, 0, 0, 0, 0, 118, 133, 0],
            &[0, 6, 241, 16, 140, 159, 0, 0, 0, 0, 160, 97, 0],
            &[0, 0, 163, 129, 22, 205, 210, 206, 79, 33, 231, 17, 0],
            &[0, 0, 23, 222, 131, 18, 32, 35, 70, 218, 89, 0, 0],
            &[0, 0, 0, 19, 144, 222, 227, 228, 180, 59, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ª' / 0xaa
        'ª' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 144, 175, 154, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 120, 81, 152, 233, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 57, 255, 29, 0, 0, 0],
            &[0, 0, 0, 0, 98, 227, 194, 178, 255, 33, 0, 0, 0],
            &[0, 0, 0, 8, 246, 73, 0, 30, 255, 33, 0, 0, 0],
            &[0, 0, 0, 6, 245, 143, 75, 179, 254, 33, 0, 0, 0],
            &[0, 0, 0, 0, 68, 171, 150, 50, 132, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '«' / 0xab
        '«' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 143, 145, 0, 10, 195, 85, 0, 0],
            &[0, 0, 0, 0, 96, 255, 110, 0, 169, 242, 41, 0, 0],
            &[0, 0, 0, 56, 248, 158, 0, 126, 254, 81, 0, 0, 0],
            &[0, 0, 25, 230, 200, 7, 80, 254, 131, 0, 0, 0, 0],
            &[0, 0, 24, 228, 203, 7, 78, 254, 137, 0, 0, 0, 0],
            &[0, 0, 0, 52, 246, 157, 0, 121, 255, 85, 0, 0, 0],
            &[0, 0, 0, 0, 89, 255, 103, 0, 164, 244, 44, 0, 0],
            &[0, 0, 0, 0, 0, 132, 138, 0, 8, 188, 82, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¬' / 0xac
        '¬' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 95, 95, 95, 95, 95, 95, 95, 74, 0, 0],
            &[0, 0, 143, 250, 250, 250, 250, 250, 250, 253, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 144, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 144, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 144, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 81, 114, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '®' / 0xae
        '®' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 140, 171, 158, 103, 13, 0, 0, 0],
            &[0, 0, 1, 149, 209, 104, 66, 82, 162, 221, 42, 0, 0],
            &[0, 0, 111, 184, 8, 205, 203, 169, 34, 94, 215, 4, 0],
            &[0, 1, 225, 36, 2, 246, 23, 136, 169, 0, 193, 76, 0],
            &[0, 22, 228, 0, 2, 245, 8, 132, 162, 0, 127, 126, 0],
            &[0, 32, 221, 0, 2, 254, 218, 229, 20, 0, 118, 133, 0],
            &[0, 6, 241, 16, 2, 245, 3, 208, 71, 0, 160, 97, 0],
            &[0, 0, 163, 129, 2, 245, 0, 63, 219, 42, 231, 17, 0],
            &[0, 0, 23, 222, 131, 34, 0, 1, 85, 222, 89, 0, 0],
            &[0, 0, 0, 19, 144, 222, 227, 228, 180, 59, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¯' / 0xaf
        '¯' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 23, 23, 23, 23, 23, 12, 0, 0, 0],
            &[0, 0, 0, 28, 255, 255, 255, 255, 255, 135, 0, 0, 0],
            &[0, 0, 0, 7, 66, 66, 66, 66, 66, 35, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '°' / 0xb0
        '°' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 119, 184, 158, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 165, 211, 111, 154, 238, 28, 0, 0, 0],
            &[0, 0, 0, 20, 254, 34, 0, 0, 183, 121, 0, 0, 0],
            &[0, 0, 0, 31, 255, 17, 0, 0, 163, 133, 0, 0, 0],
            &[0, 0, 0, 0, 208, 171, 45, 86, 245, 56, 0, 0, 0],
            &[0, 0, 0, 0, 33, 186, 247, 223, 95, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '±' / 0xb1
        '±' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 255, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 255, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 255, 88, 0, 0, 0, 0, 0],
            &[0, 0, 61, 109, 109, 112, 255, 160, 109, 109, 96, 0, 0],
            &[0, 0, 130, 235, 235, 235, 255, 242, 235, 235, 207, 0, 0],
            &[0, 0, 0, 0, 0, 5, 255, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 255, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 255, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 164, 57, 0, 0, 0, 0, 0],
            &[0, 0, 42, 89, 89, 89, 89, 89, 89, 89, 78, 0, 0],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 225, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '²' / 0xb2
        '²' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 45, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 243, 244, 255, 165, 2, 0, 0, 0],
            &[0, 0, 0, 0, 96, 61, 1, 49, 254, 65, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 8, 251, 66, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 132, 215, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 234, 43, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 113, 232, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 236, 59, 20, 20, 13, 0, 0, 0],
            &[0, 0, 0, 13, 255, 255, 255, 255, 255, 165, 0, 0, 0],
            &[0, 0, 0, 1, 24, 24, 24, 24, 24, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '³' / 0xb3
        '³' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 42, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 250, 240, 254, 146, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 43, 0, 103, 255, 33, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 76, 253, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 174, 243, 105, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 99, 114, 190, 221, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 247, 103, 0, 0, 0],
            &[0, 0, 0, 6, 68, 2, 0, 51, 254, 77, 0, 0, 0],
            &[0, 0, 0, 10, 230, 244, 222, 250, 156, 1, 0, 0, 0],
            &[0, 0, 0, 0, 3, 42, 54, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '´' / 0xb4
        '´' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 96, 141, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 249, 195, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 197, 203, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 118, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'µ' / 0xb5
        'µ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 47, 6, 0, 0, 0, 0, 39, 41, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 193, 255, 45, 0, 0, 0, 0, 219, 225, 0, 0],
            &[0, 0, 255, 255, 75, 0, 0, 0, 3, 244, 225, 0, 0],
            &[0, 0, 255, 255, 121, 0, 0, 0, 75, 255, 225, 0, 0],
            &[0, 0, 191, 255, 255, 91, 58, 104, 231, 231, 225, 0, 0],
            &[0, 0, 147, 255, 255, 255, 255, 230, 87, 147, 225, 0, 0],
            &[0, 0, 147, 255, 36, 31, 32, 4, 0, 0, 0, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 82, 143, 19, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¶' / 0xb6
        '¶' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 59, 103, 112, 112, 112, 112, 112, 27, 0],
            &[0, 0, 11, 192, 255, 255, 255, 255, 179, 159, 242, 61, 0],
            &[0, 0, 134, 255, 255, 255, 255, 255, 51, 0, 220, 61, 0],
            &[0, 0, 218, 255, 255, 255, 255, 255, 51, 0, 220, 61, 0],
            &[0, 1, 251, 255, 255, 255, 255, 255, 51, 0, 220, 61, 0],
            &[0, 7, 255, 255, 255, 255, 255, 255, 51, 0, 220, 61, 0],
            &[0, 0, 246, 255, 255, 255, 255, 255, 51, 0, 220, 61, 0],
            &[0, 0, 196, 255, 255, 255, 255, 255, 51, 0, 220, 61, 0],
            &[0, 0, 88, 255, 255, 255, 255, 255, 51, 0, 220, 61, 0],
            &[0, 0, 0, 100, 212, 246, 232, 251, 51, 0, 220, 61, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 51, 0, 220, 61, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 51, 0, 220, 61, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 51, 0, 220, 61, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 51, 0, 220, 61, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 51, 0, 220, 61, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 51, 0, 220, 61, 0],
            &[0, 0, 0, 0, 0, 0, 0, 230, 51, 0, 220, 61, 0],
            &[0, 0, 0, 0, 0, 0, 0, 95, 21, 0, 90, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '·' / 0xb7
        '·' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 167, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 113, 255, 215, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 185, 107, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¸' / 0xb8
        '¸' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 237, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 231, 184, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 239, 125, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 138, 123, 51, 0, 0, 0, 0, 0],
        ],
        // letter: '¹' / 0xb9
        '¹' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 142, 250, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 229, 218, 195, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 61, 6, 158, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 160, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 132, 241, 255, 255, 241, 147, 0, 0, 0],
            &[0, 0, 0, 0, 14, 24, 39, 39, 24, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'º' / 0xba
        'º' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 109, 171, 149, 34, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 208, 89, 153, 230, 15, 0, 0, 0],
            &[0, 0, 0, 0, 243, 65, 0, 1, 226, 90, 0, 0, 0],
            &[0, 0, 0, 15, 255, 32, 0, 0, 197, 114, 0, 0, 0],
            &[0, 0, 0, 0, 243, 62, 0, 0, 225, 85, 0, 0, 0],
            &[0, 0, 0, 0, 142, 203, 86, 146, 228, 11, 0, 0, 0],
            &[0, 0, 0, 0, 3, 111, 172, 145, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '»' / 0xbb
        '»' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 205, 74, 0, 61, 207, 24, 0, 0, 0, 0],
            &[0, 0, 0, 169, 241, 42, 18, 223, 199, 7, 0, 0, 0],
            &[0, 0, 0, 11, 211, 220, 19, 49, 246, 161, 0, 0, 0],
            &[0, 0, 0, 0, 36, 239, 191, 2, 94, 255, 113, 0, 0],
            &[0, 0, 0, 0, 45, 244, 175, 1, 111, 255, 97, 0, 0],
            &[0, 0, 0, 17, 221, 210, 12, 66, 251, 145, 0, 0, 0],
            &[0, 0, 2, 184, 234, 32, 33, 236, 188, 4, 0, 0, 0],
            &[0, 0, 18, 187, 61, 0, 53, 192, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¼' / 0xbc
        '¼' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 66, 144, 44, 0, 0, 0, 0, 78, 130, 0, 0],
            &[0, 152, 249, 249, 78, 0, 0, 0, 10, 231, 127, 0, 0],
            &[0, 68, 46, 234, 78, 0, 0, 0, 118, 237, 14, 0, 0],
            &[0, 0, 0, 237, 78, 0, 0, 14, 236, 119, 0, 0, 0],
            &[0, 0, 0, 237, 78, 0, 0, 127, 232, 10, 0, 0, 0],
            &[0, 0, 5, 255, 90, 0, 18, 241, 110, 0, 0, 0, 0],
            &[0, 96, 255, 255, 255, 210, 137, 226, 7, 0, 0, 0, 0],
            &[0, 2, 6, 13, 9, 30, 245, 102, 0, 93, 148, 0, 0],
            &[0, 0, 0, 0, 0, 146, 220, 4, 46, 247, 234, 0, 0],
            &[0, 0, 0, 0, 30, 248, 93, 8, 212, 162, 234, 0, 0],
            &[0, 0, 0, 0, 155, 214, 2, 146, 174, 76, 234, 0, 0],
            &[0, 0, 0, 36, 251, 84, 52, 254, 161, 174, 246, 126, 0],
            &[0, 0, 0, 165, 207, 1, 39, 132, 132, 169, 245, 120, 0],
            &[0, 0, 44, 253, 76, 0, 0, 0, 0, 76, 234, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '½' / 0xbd
        '½' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 66, 144, 44, 0, 0, 0, 0, 78, 130, 0, 0],
            &[0, 152, 249, 249, 78, 0, 0, 0, 10, 231, 127, 0, 0],
            &[0, 68, 46, 234, 78, 0, 0, 0, 118, 237, 14, 0, 0],
            &[0, 0, 0, 237, 78, 0, 0, 14, 236, 119, 0, 0, 0],
            &[0, 0, 0, 237, 78, 0, 0, 127, 232, 10, 0, 0, 0],
            &[0, 0, 5, 255, 90, 0, 18, 241, 110, 0, 0, 0, 0],
            &[0, 96, 255, 255, 255, 210, 137, 226, 7, 0, 0, 0, 0],
            &[0, 2, 6, 13, 9, 30, 245, 118, 130, 179, 138, 15, 0],
            &[0, 0, 0, 0, 0, 146, 220, 90, 179, 110, 226, 148, 0],
            &[0, 0, 0, 0, 30, 248, 93, 0, 0, 0, 156, 174, 0],
            &[0, 0, 0, 0, 155, 214, 2, 0, 0, 50, 243, 72, 0],
            &[0, 0, 0, 36, 251, 84, 0, 0, 55, 237, 101, 0, 0],
            &[0, 0, 0, 165, 207, 1, 0, 65, 243, 130, 45, 41, 0],
            &[0, 0, 44, 253, 76, 0, 0, 181, 255, 255, 255, 231, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¾' / 0xbe
        '¾' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 136, 178, 144, 22, 0, 0, 0, 78, 130, 0, 0],
            &[0, 70, 167, 104, 223, 168, 0, 0, 10, 231, 127, 0, 0],
            &[0, 0, 0, 5, 187, 175, 0, 0, 118, 237, 14, 0, 0],
            &[0, 0, 169, 244, 250, 53, 0, 14, 236, 119, 0, 0, 0],
            &[0, 0, 30, 57, 188, 206, 0, 127, 232, 10, 0, 0, 0],
            &[0, 40, 21, 8, 160, 227, 18, 241, 110, 0, 0, 0, 0],
            &[0, 107, 254, 251, 238, 84, 137, 226, 7, 0, 0, 0, 0],
            &[0, 0, 15, 26, 3, 24, 245, 102, 0, 93, 148, 0, 0],
            &[0, 0, 0, 0, 0, 146, 220, 4, 46, 247, 234, 0, 0],
            &[0, 0, 0, 0, 30, 248, 93, 8, 212, 162, 234, 0, 0],
            &[0, 0, 0, 0, 155, 214, 2, 146, 174, 76, 234, 0, 0],
            &[0, 0, 0, 36, 251, 84, 52, 254, 161, 174, 246, 126, 0],
            &[0, 0, 0, 165, 207, 1, 39, 132, 132, 169, 245, 120, 0],
            &[0, 0, 44, 253, 76, 0, 0, 0, 0, 76, 234, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¿' / 0xbf
        '¿' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 32, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 255, 204, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 255, 215, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 60, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 49, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 248, 133, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 255, 164, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 154, 255, 133, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 255, 121, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 157, 249, 4, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 181, 236, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 134, 255, 103, 0, 0, 0, 46, 99, 0, 0, 0],
            &[0, 0, 18, 211, 255, 227, 204, 230, 255, 248, 18, 0, 0],
            &[0, 0, 0, 7, 99, 160, 175, 159, 108, 24, 0, 0, 0],
        ],
        // letter: 'À' / 0xc0
        'À' => &[
            &[0, 0, 0, 135, 239, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 188, 241, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 184, 189, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 159, 110, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 255, 239, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 228, 214, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 97, 242, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 252, 18, 173, 235, 3, 0, 0, 0, 0],
            &[0, 0, 0, 222, 191, 0, 94, 255, 65, 0, 0, 0, 0],
            &[0, 0, 49, 255, 112, 0, 18, 251, 149, 0, 0, 0, 0],
            &[0, 0, 132, 255, 32, 0, 0, 191, 230, 1, 0, 0, 0],
            &[0, 0, 215, 255, 174, 174, 174, 255, 255, 59, 0, 0, 0],
            &[0, 41, 255, 255, 214, 214, 214, 247, 255, 142, 0, 0, 0],
            &[0, 124, 255, 50, 0, 0, 0, 0, 208, 225, 0, 0, 0],
            &[0, 207, 225, 0, 0, 0, 0, 0, 129, 255, 53, 0, 0],
            &[34, 255, 147, 0, 0, 0, 0, 0, 49, 255, 136, 0, 0],
            &[117, 255, 67, 0, 0, 0, 0, 0, 0, 225, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Á' / 0xc1
        'Á' => &[
            &[0, 0, 0, 0, 0, 0, 129, 239, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 248, 174, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 201, 172, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 26, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 159, 110, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 255, 239, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 228, 214, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 97, 242, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 252, 18, 173, 235, 3, 0, 0, 0, 0],
            &[0, 0, 0, 222, 191, 0, 94, 255, 65, 0, 0, 0, 0],
            &[0, 0, 49, 255, 112, 0, 18, 251, 149, 0, 0, 0, 0],
            &[0, 0, 132, 255, 32, 0, 0, 191, 230, 1, 0, 0, 0],
            &[0, 0, 215, 255, 174, 174, 174, 255, 255, 59, 0, 0, 0],
            &[0, 41, 255, 255, 214, 214, 214, 247, 255, 142, 0, 0, 0],
            &[0, 124, 255, 50, 0, 0, 0, 0, 208, 225, 0, 0, 0],
            &[0, 207, 225, 0, 0, 0, 0, 0, 129, 255, 53, 0, 0],
            &[34, 255, 147, 0, 0, 0, 0, 0, 49, 255, 136, 0, 0],
            &[117, 255, 67, 0, 0, 0, 0, 0, 0, 225, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Â' / 0xc2
        'Â' => &[
            &[0, 0, 0, 0, 120, 239, 199, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 70, 250, 136, 234, 158, 0, 0, 0, 0, 0],
            &[0, 0, 49, 235, 97, 0, 41, 218, 135, 0, 0, 0, 0],
            &[0, 0, 13, 18, 0, 0, 0, 8, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 159, 110, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 255, 239, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 228, 214, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 97, 242, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 252, 18, 173, 235, 3, 0, 0, 0, 0],
            &[0, 0, 0, 222, 191, 0, 94, 255, 65, 0, 0, 0, 0],
            &[0, 0, 49, 255, 112, 0, 18, 251, 149, 0, 0, 0, 0],
            &[0, 0, 132, 255, 32, 0, 0, 191, 230, 1, 0, 0, 0],
            &[0, 0, 215, 255, 174, 174, 174, 255, 255, 59, 0, 0, 0],
            &[0, 41, 255, 255, 214, 214, 214, 247, 255, 142, 0, 0, 0],
            &[0, 124, 255, 50, 0, 0, 0, 0, 208, 225, 0, 0, 0],
            &[0, 207, 225, 0, 0, 0, 0, 0, 129, 255, 53, 0, 0],
            &[34, 255, 147, 0, 0, 0, 0, 0, 49, 255, 136, 0, 0],
            &[117, 255, 67, 0, 0, 0, 0, 0, 0, 225, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ã' / 0xc3
        'Ã' => &[
            &[0, 0, 0, 27, 69, 13, 0, 0, 80, 7, 0, 0, 0],
            &[0, 0, 45, 243, 255, 235, 115, 112, 232, 1, 0, 0, 0],
            &[0, 0, 150, 146, 21, 124, 240, 246, 86, 0, 0, 0, 0],
            &[0, 0, 18, 7, 0, 0, 3, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 159, 110, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 255, 239, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 228, 214, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 97, 242, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 252, 18, 173, 235, 3, 0, 0, 0, 0],
            &[0, 0, 0, 222, 191, 0, 94, 255, 65, 0, 0, 0, 0],
            &[0, 0, 49, 255, 112, 0, 18, 251, 149, 0, 0, 0, 0],
            &[0, 0, 132, 255, 32, 0, 0, 191, 230, 1, 0, 0, 0],
            &[0, 0, 215, 255, 174, 174, 174, 255, 255, 59, 0, 0, 0],
            &[0, 41, 255, 255, 214, 214, 214, 247, 255, 142, 0, 0, 0],
            &[0, 124, 255, 50, 0, 0, 0, 0, 208, 225, 0, 0, 0],
            &[0, 207, 225, 0, 0, 0, 0, 0, 129, 255, 53, 0, 0],
            &[34, 255, 147, 0, 0, 0, 0, 0, 49, 255, 136, 0, 0],
            &[117, 255, 67, 0, 0, 0, 0, 0, 0, 225, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ä' / 0xc4
        'Ä' => &[
            &[0, 0, 11, 81, 13, 0, 0, 66, 40, 0, 0, 0, 0],
            &[0, 0, 104, 255, 116, 0, 17, 253, 206, 0, 0, 0, 0],
            &[0, 0, 42, 180, 49, 0, 0, 160, 110, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 159, 110, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 255, 239, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 228, 214, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 97, 242, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 252, 18, 173, 235, 3, 0, 0, 0, 0],
            &[0, 0, 0, 222, 191, 0, 94, 255, 65, 0, 0, 0, 0],
            &[0, 0, 49, 255, 112, 0, 18, 251, 149, 0, 0, 0, 0],
            &[0, 0, 132, 255, 32, 0, 0, 191, 230, 1, 0, 0, 0],
            &[0, 0, 215, 255, 174, 174, 174, 255, 255, 59, 0, 0, 0],
            &[0, 41, 255, 255, 214, 214, 214, 247, 255, 142, 0, 0, 0],
            &[0, 124, 255, 50, 0, 0, 0, 0, 208, 225, 0, 0, 0],
            &[0, 207, 225, 0, 0, 0, 0, 0, 129, 255, 53, 0, 0],
            &[34, 255, 147, 0, 0, 0, 0, 0, 49, 255, 136, 0, 0],
            &[117, 255, 67, 0, 0, 0, 0, 0, 0, 225, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Å' / 0xc5
        'Å' => &[
            &[0, 0, 0, 70, 228, 43, 155, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 213, 6, 123, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 190, 246, 233, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 159, 110, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 255, 239, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 228, 214, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 97, 242, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 252, 18, 173, 235, 3, 0, 0, 0, 0],
            &[0, 0, 0, 222, 191, 0, 94, 255, 65, 0, 0, 0, 0],
            &[0, 0, 49, 255, 112, 0, 18, 251, 149, 0, 0, 0, 0],
            &[0, 0, 132, 255, 32, 0, 0, 191, 230, 1, 0, 0, 0],
            &[0, 0, 215, 255, 174, 174, 174, 255, 255, 59, 0, 0, 0],
            &[0, 41, 255, 255, 214, 214, 214, 247, 255, 142, 0, 0, 0],
            &[0, 124, 255, 50, 0, 0, 0, 0, 208, 225, 0, 0, 0],
            &[0, 207, 225, 0, 0, 0, 0, 0, 129, 255, 53, 0, 0],
            &[34, 255, 147, 0, 0, 0, 0, 0, 49, 255, 136, 0, 0],
            &[117, 255, 67, 0, 0, 0, 0, 0, 0, 225, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 141, 220, 185, 33, 0, 0, 0, 0, 0],
        ],
        // letter: 'Æ' / 0xc6
        'Æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 144, 144, 144, 144, 144, 144, 45, 0],
            &[0, 0, 0, 0, 209, 251, 252, 254, 244, 244, 244, 77, 0],
            &[0, 0, 0, 13, 253, 123, 188, 229, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 255, 68, 188, 229, 0, 0, 0, 0, 0],
            &[0, 0, 0, 125, 254, 15, 188, 229, 0, 0, 0, 0, 0],
            &[0, 0, 0, 183, 215, 0, 188, 229, 0, 0, 0, 0, 0],
            &[0, 0, 1, 238, 161, 0, 188, 251, 216, 216, 207, 0, 0],
            &[0, 0, 41, 255, 106, 0, 188, 246, 167, 167, 159, 0, 0],
            &[0, 0, 99, 255, 211, 159, 255, 229, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 220, 214, 255, 229, 0, 0, 0, 0, 0],
            &[0, 0, 213, 199, 0, 0, 188, 229, 0, 0, 0, 0, 0],
            &[0, 16, 254, 144, 0, 0, 188, 229, 0, 0, 0, 0, 0],
            &[0, 72, 255, 90, 0, 0, 188, 243, 133, 133, 133, 42, 0],
            &[0, 129, 255, 35, 0, 0, 188, 255, 255, 255, 255, 81, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ç' / 0xc7
        'Ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 143, 181, 179, 149, 79, 5, 0, 0],
            &[0, 0, 1, 146, 254, 251, 208, 210, 243, 255, 59, 0, 0],
            &[0, 0, 124, 255, 177, 22, 0, 0, 6, 68, 0, 0, 0],
            &[0, 18, 247, 214, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 255, 103, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 154, 255, 39, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 179, 255, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 255, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 255, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 255, 69, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 255, 154, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 203, 250, 70, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 237, 252, 169, 114, 112, 148, 186, 0, 0, 0],
            &[0, 0, 0, 41, 172, 252, 255, 255, 250, 168, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 255, 53, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 90, 234, 174, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 251, 102, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 138, 120, 42, 0, 0, 0, 0, 0],
        ],
        // letter: 'È' / 0xc8
        'È' => &[
            &[0, 0, 0, 0, 108, 239, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 164, 250, 47, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 162, 211, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 26, 3, 0, 0, 0, 0],
            &[0, 0, 28, 144, 144, 144, 144, 144, 144, 144, 113, 0, 0],
            &[0, 0, 49, 255, 250, 244, 244, 244, 244, 244, 191, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 236, 216, 216, 216, 216, 216, 91, 0, 0],
            &[0, 0, 49, 255, 215, 172, 172, 172, 172, 172, 73, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 196, 133, 133, 133, 133, 133, 104, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'É' / 0xc9
        'É' => &[
            &[0, 0, 0, 0, 0, 0, 0, 101, 239, 147, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 236, 196, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 176, 192, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 5, 0, 0, 0, 0, 0],
            &[0, 0, 28, 144, 144, 144, 144, 144, 144, 144, 113, 0, 0],
            &[0, 0, 49, 255, 250, 244, 244, 244, 244, 244, 191, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 236, 216, 216, 216, 216, 216, 91, 0, 0],
            &[0, 0, 49, 255, 215, 172, 172, 172, 172, 172, 73, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 196, 133, 133, 133, 133, 133, 104, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ê' / 0xca
        'Ê' => &[
            &[0, 0, 0, 0, 0, 93, 239, 217, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 244, 149, 220, 184, 4, 0, 0, 0],
            &[0, 0, 0, 32, 228, 122, 0, 27, 203, 163, 0, 0, 0],
            &[0, 0, 0, 10, 21, 0, 0, 0, 5, 26, 0, 0, 0],
            &[0, 0, 28, 144, 144, 144, 144, 144, 144, 144, 113, 0, 0],
            &[0, 0, 49, 255, 250, 244, 244, 244, 244, 244, 191, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 236, 216, 216, 216, 216, 216, 91, 0, 0],
            &[0, 0, 49, 255, 215, 172, 172, 172, 172, 172, 73, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 196, 133, 133, 133, 133, 133, 104, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ë' / 0xcb
        'Ë' => &[
            &[0, 0, 0, 5, 80, 21, 0, 0, 55, 51, 0, 0, 0],
            &[0, 0, 0, 75, 255, 145, 0, 1, 239, 234, 0, 0, 0],
            &[0, 0, 0, 25, 178, 67, 0, 0, 139, 132, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 144, 144, 144, 144, 144, 144, 144, 113, 0, 0],
            &[0, 0, 49, 255, 250, 244, 244, 244, 244, 244, 191, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 236, 216, 216, 216, 216, 216, 91, 0, 0],
            &[0, 0, 49, 255, 215, 172, 172, 172, 172, 172, 73, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 196, 133, 133, 133, 133, 133, 104, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ì' / 0xcc
        'Ì' => &[
            &[0, 0, 0, 135, 239, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 188, 241, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 184, 189, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 144, 144, 144, 144, 144, 144, 44, 0, 0, 0],
            &[0, 0, 152, 189, 218, 255, 236, 197, 175, 49, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 51, 78, 126, 255, 185, 86, 64, 15, 0, 0, 0],
            &[0, 0, 232, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Í' / 0xcd
        'Í' => &[
            &[0, 0, 0, 0, 0, 0, 129, 239, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 248, 174, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 201, 172, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 26, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 144, 144, 144, 144, 144, 144, 44, 0, 0, 0],
            &[0, 0, 152, 189, 218, 255, 236, 197, 175, 49, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 51, 78, 126, 255, 185, 86, 64, 15, 0, 0, 0],
            &[0, 0, 232, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Î' / 0xce
        'Î' => &[
            &[0, 0, 0, 0, 120, 239, 199, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 70, 250, 136, 234, 158, 0, 0, 0, 0, 0],
            &[0, 0, 49, 235, 97, 0, 41, 218, 135, 0, 0, 0, 0],
            &[0, 0, 13, 18, 0, 0, 0, 8, 24, 0, 0, 0, 0],
            &[0, 0, 131, 144, 144, 144, 144, 144, 144, 44, 0, 0, 0],
            &[0, 0, 152, 189, 218, 255, 236, 197, 175, 49, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 51, 78, 126, 255, 185, 86, 64, 15, 0, 0, 0],
            &[0, 0, 232, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ï' / 0xcf
        'Ï' => &[
            &[0, 0, 11, 81, 13, 0, 0, 66, 40, 0, 0, 0, 0],
            &[0, 0, 104, 255, 116, 0, 17, 253, 206, 0, 0, 0, 0],
            &[0, 0, 42, 180, 49, 0, 0, 160, 110, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 144, 144, 144, 144, 144, 144, 44, 0, 0, 0],
            &[0, 0, 152, 189, 218, 255, 236, 197, 175, 49, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 51, 78, 126, 255, 185, 86, 64, 15, 0, 0, 0],
            &[0, 0, 232, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ð' / 0xd0
        'Ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 124, 86, 10, 0, 0, 0, 0],
            &[0, 0, 122, 255, 239, 236, 253, 255, 240, 92, 0, 0, 0],
            &[0, 0, 122, 255, 58, 0, 5, 75, 217, 255, 94, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 19, 231, 240, 14, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 114, 255, 92, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 43, 255, 150, 0],
            &[0, 114, 237, 255, 173, 114, 62, 0, 0, 16, 255, 174, 0],
            &[0, 255, 255, 255, 255, 255, 140, 0, 0, 8, 255, 178, 0],
            &[0, 8, 131, 255, 67, 8, 4, 0, 0, 30, 255, 155, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 85, 255, 112, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 192, 254, 30, 0],
            &[0, 0, 122, 255, 58, 0, 0, 5, 141, 255, 148, 0, 0],
            &[0, 0, 122, 255, 153, 128, 156, 227, 255, 177, 10, 0, 0],
            &[0, 0, 122, 255, 255, 251, 226, 181, 86, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ñ' / 0xd1
        'Ñ' => &[
            &[0, 0, 0, 21, 70, 18, 0, 0, 72, 15, 0, 0, 0],
            &[0, 0, 29, 236, 255, 241, 125, 98, 243, 11, 0, 0, 0],
            &[0, 0, 126, 167, 20, 110, 234, 250, 106, 0, 0, 0, 0],
            &[0, 0, 16, 9, 0, 0, 1, 7, 0, 0, 0, 0, 0],
            &[0, 97, 144, 103, 0, 0, 0, 0, 84, 144, 10, 0, 0],
            &[0, 171, 255, 248, 22, 0, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 217, 247, 126, 0, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 215, 155, 231, 6, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 234, 44, 255, 94, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 246, 0, 187, 205, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 74, 255, 61, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 1, 216, 173, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 106, 252, 32, 146, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 11, 238, 141, 140, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 137, 240, 141, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 29, 251, 220, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 0, 168, 255, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 0, 56, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ò' / 0xd2
        'Ò' => &[
            &[0, 0, 0, 0, 135, 239, 111, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 188, 241, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 184, 189, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 157, 192, 181, 112, 8, 0, 0, 0],
            &[0, 0, 0, 108, 255, 243, 197, 216, 255, 205, 12, 0, 0],
            &[0, 0, 41, 253, 203, 21, 0, 0, 101, 255, 146, 0, 0],
            &[0, 0, 152, 255, 50, 0, 0, 0, 0, 203, 246, 10, 0],
            &[0, 0, 217, 223, 0, 0, 0, 0, 0, 126, 255, 71, 0],
            &[0, 8, 255, 180, 0, 0, 0, 0, 0, 84, 255, 112, 0],
            &[0, 25, 255, 162, 0, 0, 0, 0, 0, 65, 255, 131, 0],
            &[0, 33, 255, 158, 0, 0, 0, 0, 0, 56, 255, 135, 0],
            &[0, 20, 255, 170, 0, 0, 0, 0, 0, 68, 255, 122, 0],
            &[0, 0, 243, 204, 0, 0, 0, 0, 0, 100, 255, 88, 0],
            &[0, 0, 188, 249, 17, 0, 0, 0, 0, 161, 255, 31, 0],
            &[0, 0, 97, 255, 134, 0, 0, 0, 36, 245, 195, 0, 0],
            &[0, 0, 4, 199, 254, 162, 96, 120, 231, 248, 53, 0, 0],
            &[0, 0, 0, 17, 164, 250, 255, 255, 210, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ó' / 0xd3
        'Ó' => &[
            &[0, 0, 0, 0, 0, 0, 0, 129, 239, 120, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 248, 174, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 201, 172, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 26, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 157, 192, 181, 112, 8, 0, 0, 0],
            &[0, 0, 0, 108, 255, 243, 197, 216, 255, 205, 12, 0, 0],
            &[0, 0, 41, 253, 203, 21, 0, 0, 101, 255, 146, 0, 0],
            &[0, 0, 152, 255, 50, 0, 0, 0, 0, 203, 246, 10, 0],
            &[0, 0, 217, 223, 0, 0, 0, 0, 0, 126, 255, 71, 0],
            &[0, 8, 255, 180, 0, 0, 0, 0, 0, 84, 255, 112, 0],
            &[0, 25, 255, 162, 0, 0, 0, 0, 0, 65, 255, 131, 0],
            &[0, 33, 255, 158, 0, 0, 0, 0, 0, 56, 255, 135, 0],
            &[0, 20, 255, 170, 0, 0, 0, 0, 0, 68, 255, 122, 0],
            &[0, 0, 243, 204, 0, 0, 0, 0, 0, 100, 255, 88, 0],
            &[0, 0, 188, 249, 17, 0, 0, 0, 0, 161, 255, 31, 0],
            &[0, 0, 97, 255, 134, 0, 0, 0, 36, 245, 195, 0, 0],
            &[0, 0, 4, 199, 254, 162, 96, 120, 231, 248, 53, 0, 0],
            &[0, 0, 0, 17, 164, 250, 255, 255, 210, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ô' / 0xd4
        'Ô' => &[
            &[0, 0, 0, 0, 0, 120, 239, 199, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 250, 136, 234, 158, 0, 0, 0, 0],
            &[0, 0, 0, 49, 235, 97, 0, 41, 218, 135, 0, 0, 0],
            &[0, 0, 0, 13, 18, 0, 0, 0, 8, 24, 0, 0, 0],
            &[0, 0, 0, 0, 55, 157, 192, 181, 112, 8, 0, 0, 0],
            &[0, 0, 0, 108, 255, 243, 197, 216, 255, 205, 12, 0, 0],
            &[0, 0, 41, 253, 203, 21, 0, 0, 101, 255, 146, 0, 0],
            &[0, 0, 152, 255, 50, 0, 0, 0, 0, 203, 246, 10, 0],
            &[0, 0, 217, 223, 0, 0, 0, 0, 0, 126, 255, 71, 0],
            &[0, 8, 255, 180, 0, 0, 0, 0, 0, 84, 255, 112, 0],
            &[0, 25, 255, 162, 0, 0, 0, 0, 0, 65, 255, 131, 0],
            &[0, 33, 255, 158, 0, 0, 0, 0, 0, 56, 255, 135, 0],
            &[0, 20, 255, 170, 0, 0, 0, 0, 0, 68, 255, 122, 0],
            &[0, 0, 243, 204, 0, 0, 0, 0, 0, 100, 255, 88, 0],
            &[0, 0, 188, 249, 17, 0, 0, 0, 0, 161, 255, 31, 0],
            &[0, 0, 97, 255, 134, 0, 0, 0, 36, 245, 195, 0, 0],
            &[0, 0, 4, 199, 254, 162, 96, 120, 231, 248, 53, 0, 0],
            &[0, 0, 0, 17, 164, 250, 255, 255, 210, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Õ' / 0xd5
        'Õ' => &[
            &[0, 0, 0, 0, 27, 69, 13, 0, 0, 80, 7, 0, 0],
            &[0, 0, 0, 45, 243, 255, 235, 115, 112, 232, 1, 0, 0],
            &[0, 0, 0, 150, 146, 21, 124, 240, 246, 86, 0, 0, 0],
            &[0, 0, 0, 18, 7, 0, 0, 3, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 157, 192, 181, 112, 8, 0, 0, 0],
            &[0, 0, 0, 108, 255, 243, 197, 216, 255, 205, 12, 0, 0],
            &[0, 0, 41, 253, 203, 21, 0, 0, 101, 255, 146, 0, 0],
            &[0, 0, 152, 255, 50, 0, 0, 0, 0, 203, 246, 10, 0],
            &[0, 0, 217, 223, 0, 0, 0, 0, 0, 126, 255, 71, 0],
            &[0, 8, 255, 180, 0, 0, 0, 0, 0, 84, 255, 112, 0],
            &[0, 25, 255, 162, 0, 0, 0, 0, 0, 65, 255, 131, 0],
            &[0, 33, 255, 158, 0, 0, 0, 0, 0, 56, 255, 135, 0],
            &[0, 20, 255, 170, 0, 0, 0, 0, 0, 68, 255, 122, 0],
            &[0, 0, 243, 204, 0, 0, 0, 0, 0, 100, 255, 88, 0],
            &[0, 0, 188, 249, 17, 0, 0, 0, 0, 161, 255, 31, 0],
            &[0, 0, 97, 255, 134, 0, 0, 0, 36, 245, 195, 0, 0],
            &[0, 0, 4, 199, 254, 162, 96, 120, 231, 248, 53, 0, 0],
            &[0, 0, 0, 17, 164, 250, 255, 255, 210, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ö' / 0xd6
        'Ö' => &[
            &[0, 0, 0, 11, 81, 13, 0, 0, 66, 40, 0, 0, 0],
            &[0, 0, 0, 104, 255, 116, 0, 17, 253, 206, 0, 0, 0],
            &[0, 0, 0, 42, 180, 49, 0, 0, 160, 110, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 157, 192, 181, 112, 8, 0, 0, 0],
            &[0, 0, 0, 108, 255, 243, 197, 216, 255, 205, 12, 0, 0],
            &[0, 0, 41, 253, 203, 21, 0, 0, 101, 255, 146, 0, 0],
            &[0, 0, 152, 255, 50, 0, 0, 0, 0, 203, 246, 10, 0],
            &[0, 0, 217, 223, 0, 0, 0, 0, 0, 126, 255, 71, 0],
            &[0, 8, 255, 180, 0, 0, 0, 0, 0, 84, 255, 112, 0],
            &[0, 25, 255, 162, 0, 0, 0, 0, 0, 65, 255, 131, 0],
            &[0, 33, 255, 158, 0, 0, 0, 0, 0, 56, 255, 135, 0],
            &[0, 20, 255, 170, 0, 0, 0, 0, 0, 68, 255, 122, 0],
            &[0, 0, 243, 204, 0, 0, 0, 0, 0, 100, 255, 88, 0],
            &[0, 0, 188, 249, 17, 0, 0, 0, 0, 161, 255, 31, 0],
            &[0, 0, 97, 255, 134, 0, 0, 0, 36, 245, 195, 0, 0],
            &[0, 0, 4, 199, 254, 162, 96, 120, 231, 248, 53, 0, 0],
            &[0, 0, 0, 17, 164, 250, 255, 255, 210, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '×' / 0xd7
        '×' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 137, 15, 0, 0, 0, 0, 104, 56, 0, 0],
            &[0, 0, 31, 229, 201, 15, 0, 0, 109, 254, 108, 0, 0],
            &[0, 0, 0, 37, 229, 200, 14, 111, 254, 113, 0, 0, 0],
            &[0, 0, 0, 0, 37, 229, 222, 254, 113, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 138, 255, 224, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 113, 255, 139, 232, 201, 15, 0, 0, 0],
            &[0, 0, 0, 110, 255, 115, 0, 40, 230, 200, 14, 0, 0],
            &[0, 0, 34, 250, 117, 0, 0, 0, 37, 228, 136, 0, 0],
            &[0, 0, 0, 36, 0, 0, 0, 0, 0, 30, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ø' / 0xd8
        'Ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 0, 0],
            &[0, 0, 0, 0, 55, 157, 192, 181, 112, 57, 245, 55, 0],
            &[0, 0, 0, 108, 255, 243, 197, 216, 255, 255, 224, 3, 0],
            &[0, 0, 41, 253, 203, 21, 0, 0, 185, 255, 216, 0, 0],
            &[0, 0, 152, 255, 50, 0, 0, 7, 222, 255, 246, 10, 0],
            &[0, 0, 217, 223, 0, 0, 0, 120, 250, 165, 255, 71, 0],
            &[0, 8, 255, 180, 0, 0, 24, 242, 143, 84, 255, 112, 0],
            &[0, 25, 255, 162, 0, 0, 157, 236, 16, 65, 255, 131, 0],
            &[0, 33, 255, 158, 0, 49, 253, 107, 0, 56, 255, 135, 0],
            &[0, 20, 255, 170, 0, 193, 213, 4, 0, 68, 255, 122, 0],
            &[0, 0, 243, 204, 84, 255, 71, 0, 0, 100, 255, 88, 0],
            &[0, 0, 188, 255, 240, 181, 0, 0, 0, 161, 255, 31, 0],
            &[0, 0, 97, 255, 255, 40, 0, 0, 36, 245, 195, 0, 0],
            &[0, 0, 29, 255, 255, 162, 96, 120, 231, 248, 53, 0, 0],
            &[0, 0, 158, 255, 182, 250, 255, 255, 210, 62, 0, 0, 0],
            &[0, 0, 73, 87, 0, 13, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ù' / 0xd9
        'Ù' => &[
            &[0, 0, 0, 135, 239, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 188, 241, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 184, 189, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 26, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 0, 92, 144, 10, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 164, 255, 16, 0, 0, 0, 0, 176, 255, 11, 0, 0],
            &[0, 142, 255, 47, 0, 0, 0, 0, 206, 243, 0, 0, 0],
            &[0, 82, 255, 138, 0, 0, 0, 36, 252, 181, 0, 0, 0],
            &[0, 4, 214, 254, 151, 93, 115, 225, 252, 60, 0, 0, 0],
            &[0, 0, 27, 183, 254, 255, 255, 219, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 40, 24, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ú' / 0xda
        'Ú' => &[
            &[0, 0, 0, 0, 0, 0, 129, 239, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 248, 174, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 201, 172, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 26, 2, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 0, 92, 144, 10, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 164, 255, 16, 0, 0, 0, 0, 176, 255, 11, 0, 0],
            &[0, 142, 255, 47, 0, 0, 0, 0, 206, 243, 0, 0, 0],
            &[0, 82, 255, 138, 0, 0, 0, 36, 252, 181, 0, 0, 0],
            &[0, 4, 214, 254, 151, 93, 115, 225, 252, 60, 0, 0, 0],
            &[0, 0, 27, 183, 254, 255, 255, 219, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 40, 24, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Û' / 0xdb
        'Û' => &[
            &[0, 0, 0, 0, 120, 239, 199, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 70, 250, 136, 234, 158, 0, 0, 0, 0, 0],
            &[0, 0, 49, 235, 97, 0, 41, 218, 135, 0, 0, 0, 0],
            &[0, 0, 13, 18, 0, 0, 0, 8, 24, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 0, 92, 144, 10, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 164, 255, 16, 0, 0, 0, 0, 176, 255, 11, 0, 0],
            &[0, 142, 255, 47, 0, 0, 0, 0, 206, 243, 0, 0, 0],
            &[0, 82, 255, 138, 0, 0, 0, 36, 252, 181, 0, 0, 0],
            &[0, 4, 214, 254, 151, 93, 115, 225, 252, 60, 0, 0, 0],
            &[0, 0, 27, 183, 254, 255, 255, 219, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 40, 24, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ü' / 0xdc
        'Ü' => &[
            &[0, 0, 11, 81, 13, 0, 0, 66, 40, 0, 0, 0, 0],
            &[0, 0, 104, 255, 116, 0, 17, 253, 206, 0, 0, 0, 0],
            &[0, 0, 42, 180, 49, 0, 0, 160, 110, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 0, 92, 144, 10, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 164, 255, 16, 0, 0, 0, 0, 176, 255, 11, 0, 0],
            &[0, 142, 255, 47, 0, 0, 0, 0, 206, 243, 0, 0, 0],
            &[0, 82, 255, 138, 0, 0, 0, 36, 252, 181, 0, 0, 0],
            &[0, 4, 214, 254, 151, 93, 115, 225, 252, 60, 0, 0, 0],
            &[0, 0, 27, 183, 254, 255, 255, 219, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 40, 24, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ý' / 0xdd
        'Ý' => &[
            &[0, 0, 0, 0, 0, 0, 0, 129, 239, 120, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 248, 174, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 201, 172, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 26, 2, 0, 0, 0, 0, 0],
            &[0, 27, 144, 98, 0, 0, 0, 0, 0, 31, 144, 85, 0],
            &[0, 0, 202, 245, 20, 0, 0, 0, 0, 149, 254, 50, 0],
            &[0, 0, 72, 255, 128, 0, 0, 0, 22, 246, 174, 0, 0],
            &[0, 0, 0, 197, 236, 10, 0, 0, 132, 254, 46, 0, 0],
            &[0, 0, 0, 67, 255, 109, 0, 12, 238, 170, 0, 0, 0],
            &[0, 0, 0, 0, 193, 223, 4, 114, 253, 42, 0, 0, 0],
            &[0, 0, 0, 0, 63, 255, 95, 227, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 188, 245, 252, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 255, 165, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Þ' / 0xde
        'Þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 144, 74, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 255, 143, 24, 23, 12, 0, 0, 0, 0, 0, 0],
            &[0, 49, 255, 255, 255, 255, 255, 227, 134, 8, 0, 0, 0],
            &[0, 49, 255, 179, 98, 102, 128, 215, 255, 172, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 5, 205, 255, 39, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 120, 255, 82, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 115, 255, 73, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 2, 196, 251, 19, 0, 0],
            &[0, 49, 255, 161, 60, 65, 95, 190, 255, 140, 0, 0, 0],
            &[0, 49, 255, 255, 255, 255, 255, 235, 124, 2, 0, 0, 0],
            &[0, 49, 255, 162, 62, 58, 37, 4, 0, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ß' / 0xdf
        'ß' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 111, 132, 120, 70, 3, 0, 0, 0, 0],
            &[0, 1, 155, 255, 255, 241, 251, 255, 211, 25, 0, 0, 0],
            &[0, 77, 255, 183, 27, 0, 8, 125, 255, 145, 0, 0, 0],
            &[0, 134, 255, 41, 0, 0, 0, 23, 255, 162, 0, 0, 0],
            &[0, 147, 255, 24, 0, 0, 0, 138, 255, 82, 0, 0, 0],
            &[0, 147, 255, 24, 0, 1, 157, 254, 124, 0, 0, 0, 0],
            &[0, 147, 255, 24, 0, 93, 255, 113, 0, 0, 0, 0, 0],
            &[0, 147, 255, 24, 0, 133, 255, 48, 0, 0, 0, 0, 0],
            &[0, 147, 255, 24, 0, 47, 245, 233, 75, 0, 0, 0, 0],
            &[0, 147, 255, 24, 0, 0, 55, 225, 254, 136, 0, 0, 0],
            &[0, 147, 255, 24, 0, 0, 0, 17, 176, 255, 112, 0, 0],
            &[0, 147, 255, 24, 0, 0, 0, 0, 5, 226, 217, 0, 0],
            &[0, 147, 255, 24, 0, 0, 0, 0, 0, 209, 225, 0, 0],
            &[0, 147, 255, 24, 84, 155, 86, 59, 125, 255, 160, 0, 0],
            &[0, 147, 255, 24, 57, 225, 255, 255, 253, 176, 16, 0, 0],
            &[0, 0, 0, 0, 0, 1, 24, 42, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'à' / 0xe0
        'à' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 141, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 229, 221, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 42, 231, 145, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 123, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 69, 90, 69, 24, 0, 0, 0, 0],
            &[0, 0, 0, 148, 248, 255, 255, 255, 250, 142, 2, 0, 0],
            &[0, 0, 0, 105, 115, 41, 12, 36, 167, 255, 91, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 8, 251, 160, 0, 0],
            &[0, 0, 0, 0, 0, 3, 24, 38, 45, 231, 176, 0, 0],
            &[0, 0, 0, 57, 184, 247, 255, 255, 255, 255, 176, 0, 0],
            &[0, 0, 48, 248, 212, 90, 38, 19, 10, 223, 176, 0, 0],
            &[0, 0, 145, 255, 38, 0, 0, 0, 0, 240, 176, 0, 0],
            &[0, 0, 156, 252, 7, 0, 0, 0, 85, 255, 176, 0, 0],
            &[0, 0, 96, 255, 170, 65, 64, 135, 231, 239, 176, 0, 0],
            &[0, 0, 1, 141, 249, 255, 255, 199, 47, 190, 176, 0, 0],
            &[0, 0, 0, 0, 8, 38, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'á' / 0xe1
        'á' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 47, 141, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 199, 242, 47, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 113, 243, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 48, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 69, 90, 69, 24, 0, 0, 0, 0],
            &[0, 0, 0, 148, 248, 255, 255, 255, 250, 142, 2, 0, 0],
            &[0, 0, 0, 105, 115, 41, 12, 36, 167, 255, 91, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 8, 251, 160, 0, 0],
            &[0, 0, 0, 0, 0, 3, 24, 38, 45, 231, 176, 0, 0],
            &[0, 0, 0, 57, 184, 247, 255, 255, 255, 255, 176, 0, 0],
            &[0, 0, 48, 248, 212, 90, 38, 19, 10, 223, 176, 0, 0],
            &[0, 0, 145, 255, 38, 0, 0, 0, 0, 240, 176, 0, 0],
            &[0, 0, 156, 252, 7, 0, 0, 0, 85, 255, 176, 0, 0],
            &[0, 0, 96, 255, 170, 65, 64, 135, 231, 239, 176, 0, 0],
            &[0, 0, 1, 141, 249, 255, 255, 199, 47, 190, 176, 0, 0],
            &[0, 0, 0, 0, 8, 38, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'â' / 0xe2
        'â' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 141, 116, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 207, 224, 254, 101, 0, 0, 0, 0],
            &[0, 0, 0, 3, 176, 205, 19, 96, 246, 73, 0, 0, 0],
            &[0, 0, 0, 42, 116, 10, 0, 0, 59, 115, 0, 0, 0],
            &[0, 0, 0, 0, 16, 69, 90, 69, 24, 0, 0, 0, 0],
            &[0, 0, 0, 148, 248, 255, 255, 255, 250, 142, 2, 0, 0],
            &[0, 0, 0, 105, 115, 41, 12, 36, 167, 255, 91, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 8, 251, 160, 0, 0],
            &[0, 0, 0, 0, 0, 3, 24, 38, 45, 231, 176, 0, 0],
            &[0, 0, 0, 57, 184, 247, 255, 255, 255, 255, 176, 0, 0],
            &[0, 0, 48, 248, 212, 90, 38, 19, 10, 223, 176, 0, 0],
            &[0, 0, 145, 255, 38, 0, 0, 0, 0, 240, 176, 0, 0],
            &[0, 0, 156, 252, 7, 0, 0, 0, 85, 255, 176, 0, 0],
            &[0, 0, 96, 255, 170, 65, 64, 135, 231, 239, 176, 0, 0],
            &[0, 0, 1, 141, 249, 255, 255, 199, 47, 190, 176, 0, 0],
            &[0, 0, 0, 0, 8, 38, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ã' / 0xe3
        'ã' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 160, 228, 162, 30, 12, 226, 26, 0, 0],
            &[0, 0, 0, 95, 219, 118, 206, 252, 243, 184, 0, 0, 0],
            &[0, 0, 0, 71, 55, 0, 2, 78, 99, 10, 0, 0, 0],
            &[0, 0, 0, 0, 16, 69, 90, 69, 24, 0, 0, 0, 0],
            &[0, 0, 0, 148, 248, 255, 255, 255, 250, 142, 2, 0, 0],
            &[0, 0, 0, 105, 115, 41, 12, 36, 167, 255, 91, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 8, 251, 160, 0, 0],
            &[0, 0, 0, 0, 0, 3, 24, 38, 45, 231, 176, 0, 0],
            &[0, 0, 0, 57, 184, 247, 255, 255, 255, 255, 176, 0, 0],
            &[0, 0, 48, 248, 212, 90, 38, 19, 10, 223, 176, 0, 0],
            &[0, 0, 145, 255, 38, 0, 0, 0, 0, 240, 176, 0, 0],
            &[0, 0, 156, 252, 7, 0, 0, 0, 85, 255, 176, 0, 0],
            &[0, 0, 96, 255, 170, 65, 64, 135, 231, 239, 176, 0, 0],
            &[0, 0, 1, 141, 249, 255, 255, 199, 47, 190, 176, 0, 0],
            &[0, 0, 0, 0, 8, 38, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ä' / 0xe4
        'ä' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 238, 101, 0, 0, 203, 186, 0, 0, 0],
            &[0, 0, 0, 64, 255, 123, 0, 2, 226, 212, 0, 0, 0],
            &[0, 0, 0, 0, 21, 0, 0, 0, 12, 9, 0, 0, 0],
            &[0, 0, 0, 0, 16, 69, 90, 69, 24, 0, 0, 0, 0],
            &[0, 0, 0, 148, 248, 255, 255, 255, 250, 142, 2, 0, 0],
            &[0, 0, 0, 105, 115, 41, 12, 36, 167, 255, 91, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 8, 251, 160, 0, 0],
            &[0, 0, 0, 0, 0, 3, 24, 38, 45, 231, 176, 0, 0],
            &[0, 0, 0, 57, 184, 247, 255, 255, 255, 255, 176, 0, 0],
            &[0, 0, 48, 248, 212, 90, 38, 19, 10, 223, 176, 0, 0],
            &[0, 0, 145, 255, 38, 0, 0, 0, 0, 240, 176, 0, 0],
            &[0, 0, 156, 252, 7, 0, 0, 0, 85, 255, 176, 0, 0],
            &[0, 0, 96, 255, 170, 65, 64, 135, 231, 239, 176, 0, 0],
            &[0, 0, 1, 141, 249, 255, 255, 199, 47, 190, 176, 0, 0],
            &[0, 0, 0, 0, 8, 38, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'å' / 0xe5
        'å' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 120, 95, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 240, 146, 202, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 64, 221, 0, 75, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 237, 161, 211, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 111, 86, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 69, 90, 69, 24, 0, 0, 0, 0],
            &[0, 0, 0, 148, 248, 255, 255, 255, 250, 142, 2, 0, 0],
            &[0, 0, 0, 105, 115, 41, 12, 36, 167, 255, 91, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 8, 251, 160, 0, 0],
            &[0, 0, 0, 0, 0, 3, 24, 38, 45, 231, 176, 0, 0],
            &[0, 0, 0, 57, 184, 247, 255, 255, 255, 255, 176, 0, 0],
            &[0, 0, 48, 248, 212, 90, 38, 19, 10, 223, 176, 0, 0],
            &[0, 0, 145, 255, 38, 0, 0, 0, 0, 240, 176, 0, 0],
            &[0, 0, 156, 252, 7, 0, 0, 0, 85, 255, 176, 0, 0],
            &[0, 0, 96, 255, 170, 65, 64, 135, 231, 239, 176, 0, 0],
            &[0, 0, 1, 141, 249, 255, 255, 199, 47, 190, 176, 0, 0],
            &[0, 0, 0, 0, 8, 38, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'æ' / 0xe6
        'æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 85, 51, 0, 1, 62, 82, 15, 0, 0, 0],
            &[0, 166, 255, 255, 255, 127, 166, 255, 255, 228, 27, 0, 0],
            &[0, 91, 66, 16, 149, 253, 247, 65, 52, 247, 149, 0, 0],
            &[0, 0, 0, 0, 43, 255, 164, 0, 0, 174, 220, 0, 0],
            &[0, 0, 0, 18, 50, 255, 131, 25, 25, 158, 250, 0, 0],
            &[0, 110, 235, 255, 255, 255, 255, 255, 255, 255, 255, 4, 0],
            &[64, 255, 178, 59, 58, 255, 136, 59, 59, 59, 59, 1, 0],
            &[141, 251, 11, 0, 44, 255, 124, 0, 0, 0, 0, 0, 0],
            &[146, 241, 0, 0, 99, 255, 196, 0, 0, 0, 0, 0, 0],
            &[100, 255, 129, 85, 236, 155, 253, 153, 67, 93, 152, 0, 0],
            &[6, 188, 255, 254, 161, 4, 108, 244, 255, 252, 138, 0, 0],
            &[0, 0, 32, 21, 0, 0, 0, 10, 40, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ç' / 0xe7
        'ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 67, 89, 69, 24, 0, 0, 0],
            &[0, 0, 0, 2, 127, 246, 255, 255, 255, 252, 98, 0, 0],
            &[0, 0, 0, 123, 255, 194, 71, 35, 52, 124, 38, 0, 0],
            &[0, 0, 8, 241, 221, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 134, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 87, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 86, 255, 103, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 255, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 238, 230, 16, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 107, 255, 218, 110, 76, 95, 143, 98, 0, 0],
            &[0, 0, 0, 0, 98, 221, 255, 255, 255, 231, 85, 0, 0],
            &[0, 0, 0, 0, 0, 0, 140, 213, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 168, 232, 116, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 110, 255, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 136, 101, 14, 0, 0, 0, 0],
        ],
        // letter: 'è' / 0xe8
        'è' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 88, 141, 61, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 231, 218, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 233, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 36, 123, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 81, 79, 28, 0, 0, 0, 0, 0],
            &[0, 0, 23, 185, 255, 255, 255, 252, 145, 3, 0, 0, 0],
            &[0, 6, 209, 245, 105, 23, 21, 133, 255, 124, 0, 0, 0],
            &[0, 101, 255, 105, 0, 0, 0, 0, 192, 236, 4, 0, 0],
            &[0, 178, 252, 46, 35, 35, 35, 35, 146, 255, 43, 0, 0],
            &[0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 64, 0, 0],
            &[0, 205, 239, 59, 59, 59, 59, 59, 59, 59, 15, 0, 0],
            &[0, 171, 254, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 176, 255, 168, 83, 62, 89, 145, 184, 0, 0, 0],
            &[0, 0, 4, 124, 228, 255, 255, 255, 224, 122, 0, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'é' / 0xe9
        'é' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 50, 141, 102, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 203, 240, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 118, 241, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 118, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 81, 79, 28, 0, 0, 0, 0, 0],
            &[0, 0, 23, 185, 255, 255, 255, 252, 145, 3, 0, 0, 0],
            &[0, 6, 209, 245, 105, 23, 21, 133, 255, 124, 0, 0, 0],
            &[0, 101, 255, 105, 0, 0, 0, 0, 192, 236, 4, 0, 0],
            &[0, 178, 252, 46, 35, 35, 35, 35, 146, 255, 43, 0, 0],
            &[0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 64, 0, 0],
            &[0, 205, 239, 59, 59, 59, 59, 59, 59, 59, 15, 0, 0],
            &[0, 171, 254, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 176, 255, 168, 83, 62, 89, 145, 184, 0, 0, 0],
            &[0, 0, 4, 124, 228, 255, 255, 255, 224, 122, 0, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ê' / 0xea
        'ê' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 141, 114, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 210, 224, 254, 97, 0, 0, 0, 0, 0],
            &[0, 0, 4, 180, 202, 18, 100, 246, 70, 0, 0, 0, 0],
            &[0, 0, 45, 115, 9, 0, 0, 61, 113, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 81, 79, 28, 0, 0, 0, 0, 0],
            &[0, 0, 23, 185, 255, 255, 255, 252, 145, 3, 0, 0, 0],
            &[0, 6, 209, 245, 105, 23, 21, 133, 255, 124, 0, 0, 0],
            &[0, 101, 255, 105, 0, 0, 0, 0, 192, 236, 4, 0, 0],
            &[0, 178, 252, 46, 35, 35, 35, 35, 146, 255, 43, 0, 0],
            &[0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 64, 0, 0],
            &[0, 205, 239, 59, 59, 59, 59, 59, 59, 59, 15, 0, 0],
            &[0, 171, 254, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 176, 255, 168, 83, 62, 89, 145, 184, 0, 0, 0],
            &[0, 0, 4, 124, 228, 255, 255, 255, 224, 122, 0, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ë' / 0xeb
        'ë' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 238, 97, 0, 1, 207, 181, 0, 0, 0, 0],
            &[0, 0, 68, 255, 118, 0, 3, 230, 208, 0, 0, 0, 0],
            &[0, 0, 0, 22, 0, 0, 0, 13, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 81, 79, 28, 0, 0, 0, 0, 0],
            &[0, 0, 23, 185, 255, 255, 255, 252, 145, 3, 0, 0, 0],
            &[0, 6, 209, 245, 105, 23, 21, 133, 255, 124, 0, 0, 0],
            &[0, 101, 255, 105, 0, 0, 0, 0, 192, 236, 4, 0, 0],
            &[0, 178, 252, 46, 35, 35, 35, 35, 146, 255, 43, 0, 0],
            &[0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 64, 0, 0],
            &[0, 205, 239, 59, 59, 59, 59, 59, 59, 59, 15, 0, 0],
            &[0, 171, 254, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 176, 255, 168, 83, 62, 89, 145, 184, 0, 0, 0],
            &[0, 0, 4, 124, 228, 255, 255, 255, 224, 122, 0, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ì' / 0xec
        'ì' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 99, 141, 51, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 239, 204, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 57, 240, 121, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 119, 1, 0, 0, 0, 0, 0],
            &[0, 8, 47, 47, 47, 47, 40, 0, 0, 0, 0, 0, 0],
            &[0, 44, 255, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 30, 49, 226, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 75, 95, 233, 234, 95, 73, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'í' / 0xed
        'í' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 141, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 218, 232, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 137, 235, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 122, 36, 0, 0, 0, 0, 0, 0],
            &[0, 8, 47, 47, 47, 47, 40, 0, 0, 0, 0, 0, 0],
            &[0, 44, 255, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 30, 49, 226, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 75, 95, 233, 234, 95, 73, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'î' / 0xee
        'î' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 141, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 222, 222, 254, 78, 0, 0, 0, 0, 0],
            &[0, 0, 9, 195, 189, 11, 117, 243, 55, 0, 0, 0, 0],
            &[0, 0, 54, 109, 5, 0, 0, 71, 103, 0, 0, 0, 0],
            &[0, 8, 47, 47, 47, 47, 40, 0, 0, 0, 0, 0, 0],
            &[0, 44, 255, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 30, 49, 226, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 75, 95, 233, 234, 95, 73, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ï' / 0xef
        'ï' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 239, 80, 0, 6, 221, 162, 0, 0, 0, 0],
            &[0, 0, 87, 255, 98, 0, 11, 242, 188, 0, 0, 0, 0],
            &[0, 0, 0, 22, 0, 0, 0, 16, 5, 0, 0, 0, 0],
            &[0, 8, 47, 47, 47, 47, 40, 0, 0, 0, 0, 0, 0],
            &[0, 44, 255, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 30, 49, 226, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 75, 95, 233, 234, 95, 73, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ð' / 0xf0
        'ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 77, 0, 0, 0, 59, 0, 0, 0, 0],
            &[0, 0, 0, 88, 255, 186, 74, 193, 248, 41, 0, 0, 0],
            &[0, 0, 0, 0, 93, 255, 255, 219, 35, 0, 0, 0, 0],
            &[0, 0, 0, 114, 241, 238, 255, 233, 28, 0, 0, 0, 0],
            &[0, 0, 0, 123, 108, 4, 20, 224, 195, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 53, 253, 86, 0, 0, 0],
            &[0, 0, 18, 145, 222, 246, 228, 151, 188, 195, 0, 0, 0],
            &[0, 16, 219, 247, 146, 100, 117, 213, 252, 254, 17, 0, 0],
            &[0, 137, 255, 72, 0, 0, 0, 9, 211, 255, 64, 0, 0],
            &[0, 213, 222, 0, 0, 0, 0, 0, 112, 255, 84, 0, 0],
            &[0, 237, 193, 0, 0, 0, 0, 0, 92, 255, 79, 0, 0],
            &[0, 217, 214, 0, 0, 0, 0, 0, 120, 255, 45, 0, 0],
            &[0, 145, 255, 51, 0, 0, 0, 2, 206, 225, 2, 0, 0],
            &[0, 24, 230, 233, 104, 54, 70, 179, 255, 87, 0, 0, 0],
            &[0, 0, 30, 176, 251, 255, 255, 223, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 40, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ñ' / 0xf1
        'ñ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 136, 228, 177, 44, 5, 201, 59, 0, 0],
            &[0, 0, 0, 61, 233, 119, 189, 254, 241, 213, 4, 0, 0],
            &[0, 0, 0, 54, 72, 0, 0, 65, 103, 20, 0, 0, 0],
            &[0, 0, 22, 42, 0, 3, 60, 86, 57, 3, 0, 0, 0],
            &[0, 0, 122, 250, 45, 220, 255, 255, 255, 208, 25, 0, 0],
            &[0, 0, 122, 255, 233, 123, 30, 20, 126, 255, 163, 0, 0],
            &[0, 0, 122, 255, 170, 0, 0, 0, 0, 224, 223, 0, 0],
            &[0, 0, 122, 255, 92, 0, 0, 0, 0, 191, 248, 0, 0],
            &[0, 0, 122, 255, 66, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ò' / 0xf2
        'ò' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 99, 141, 51, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 42, 239, 204, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 57, 240, 121, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 119, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 55, 87, 70, 13, 0, 0, 0, 0, 0],
            &[0, 0, 53, 218, 255, 255, 255, 244, 116, 0, 0, 0, 0],
            &[0, 28, 238, 231, 76, 16, 37, 170, 255, 105, 0, 0, 0],
            &[0, 139, 255, 68, 0, 0, 0, 3, 217, 233, 3, 0, 0],
            &[0, 209, 231, 0, 0, 0, 0, 0, 134, 255, 54, 0, 0],
            &[0, 235, 199, 0, 0, 0, 0, 0, 103, 255, 80, 0, 0],
            &[0, 233, 199, 0, 0, 0, 0, 0, 106, 255, 80, 0, 0],
            &[0, 198, 239, 2, 0, 0, 0, 0, 148, 255, 47, 0, 0],
            &[0, 119, 255, 93, 0, 0, 0, 16, 235, 222, 2, 0, 0],
            &[0, 12, 216, 246, 119, 60, 84, 207, 254, 82, 0, 0, 0],
            &[0, 0, 23, 172, 251, 255, 255, 220, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 42, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ó' / 0xf3
        'ó' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 141, 91, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 218, 232, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 137, 235, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 122, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 55, 87, 70, 13, 0, 0, 0, 0, 0],
            &[0, 0, 53, 218, 255, 255, 255, 244, 116, 0, 0, 0, 0],
            &[0, 28, 238, 231, 76, 16, 37, 170, 255, 105, 0, 0, 0],
            &[0, 139, 255, 68, 0, 0, 0, 3, 217, 233, 3, 0, 0],
            &[0, 209, 231, 0, 0, 0, 0, 0, 134, 255, 54, 0, 0],
            &[0, 235, 199, 0, 0, 0, 0, 0, 103, 255, 80, 0, 0],
            &[0, 233, 199, 0, 0, 0, 0, 0, 106, 255, 80, 0, 0],
            &[0, 198, 239, 2, 0, 0, 0, 0, 148, 255, 47, 0, 0],
            &[0, 119, 255, 93, 0, 0, 0, 16, 235, 222, 2, 0, 0],
            &[0, 12, 216, 246, 119, 60, 84, 207, 254, 82, 0, 0, 0],
            &[0, 0, 23, 172, 251, 255, 255, 220, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 42, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ô' / 0xf4
        'ô' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 52, 141, 103, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 222, 222, 254, 78, 0, 0, 0, 0, 0],
            &[0, 0, 9, 195, 189, 11, 117, 243, 55, 0, 0, 0, 0],
            &[0, 0, 54, 109, 5, 0, 0, 71, 103, 0, 0, 0, 0],
            &[0, 0, 0, 1, 55, 87, 70, 13, 0, 0, 0, 0, 0],
            &[0, 0, 53, 218, 255, 255, 255, 244, 116, 0, 0, 0, 0],
            &[0, 28, 238, 231, 76, 16, 37, 170, 255, 105, 0, 0, 0],
            &[0, 139, 255, 68, 0, 0, 0, 3, 217, 233, 3, 0, 0],
            &[0, 209, 231, 0, 0, 0, 0, 0, 134, 255, 54, 0, 0],
            &[0, 235, 199, 0, 0, 0, 0, 0, 103, 255, 80, 0, 0],
            &[0, 233, 199, 0, 0, 0, 0, 0, 106, 255, 80, 0, 0],
            &[0, 198, 239, 2, 0, 0, 0, 0, 148, 255, 47, 0, 0],
            &[0, 119, 255, 93, 0, 0, 0, 16, 235, 222, 2, 0, 0],
            &[0, 12, 216, 246, 119, 60, 84, 207, 254, 82, 0, 0, 0],
            &[0, 0, 23, 172, 251, 255, 255, 220, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 42, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'õ' / 0xf5
        'õ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 173, 227, 150, 22, 24, 233, 8, 0, 0, 0],
            &[0, 0, 120, 207, 118, 216, 250, 246, 160, 0, 0, 0, 0],
            &[0, 0, 83, 44, 0, 5, 86, 93, 5, 0, 0, 0, 0],
            &[0, 0, 0, 1, 55, 87, 70, 13, 0, 0, 0, 0, 0],
            &[0, 0, 53, 218, 255, 255, 255, 244, 116, 0, 0, 0, 0],
            &[0, 28, 238, 231, 76, 16, 37, 170, 255, 105, 0, 0, 0],
            &[0, 139, 255, 68, 0, 0, 0, 3, 217, 233, 3, 0, 0],
            &[0, 209, 231, 0, 0, 0, 0, 0, 134, 255, 54, 0, 0],
            &[0, 235, 199, 0, 0, 0, 0, 0, 103, 255, 80, 0, 0],
            &[0, 233, 199, 0, 0, 0, 0, 0, 106, 255, 80, 0, 0],
            &[0, 198, 239, 2, 0, 0, 0, 0, 148, 255, 47, 0, 0],
            &[0, 119, 255, 93, 0, 0, 0, 16, 235, 222, 2, 0, 0],
            &[0, 12, 216, 246, 119, 60, 84, 207, 254, 82, 0, 0, 0],
            &[0, 0, 23, 172, 251, 255, 255, 220, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 42, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ö' / 0xf6
        'ö' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 239, 80, 0, 6, 221, 162, 0, 0, 0, 0],
            &[0, 0, 87, 255, 98, 0, 11, 242, 188, 0, 0, 0, 0],
            &[0, 0, 0, 22, 0, 0, 0, 16, 5, 0, 0, 0, 0],
            &[0, 0, 0, 1, 55, 87, 70, 13, 0, 0, 0, 0, 0],
            &[0, 0, 53, 218, 255, 255, 255, 244, 116, 0, 0, 0, 0],
            &[0, 28, 238, 231, 76, 16, 37, 170, 255, 105, 0, 0, 0],
            &[0, 139, 255, 68, 0, 0, 0, 3, 217, 233, 3, 0, 0],
            &[0, 209, 231, 0, 0, 0, 0, 0, 134, 255, 54, 0, 0],
            &[0, 235, 199, 0, 0, 0, 0, 0, 103, 255, 80, 0, 0],
            &[0, 233, 199, 0, 0, 0, 0, 0, 106, 255, 80, 0, 0],
            &[0, 198, 239, 2, 0, 0, 0, 0, 148, 255, 47, 0, 0],
            &[0, 119, 255, 93, 0, 0, 0, 16, 235, 222, 2, 0, 0],
            &[0, 12, 216, 246, 119, 60, 84, 207, 254, 82, 0, 0, 0],
            &[0, 0, 23, 172, 251, 255, 255, 220, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 42, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '÷' / 0xf7
        '÷' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 247, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 255, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 30, 3, 0, 0, 0, 0, 0],
            &[0, 0, 45, 95, 95, 95, 95, 95, 95, 95, 83, 0, 0],
            &[0, 0, 120, 250, 250, 250, 250, 250, 250, 250, 220, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 129, 50, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 255, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 152, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ø' / 0xf8
        'ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 55, 87, 70, 13, 50, 111, 2, 0, 0],
            &[0, 0, 53, 218, 255, 255, 255, 255, 255, 189, 1, 0, 0],
            &[0, 28, 238, 231, 76, 16, 37, 255, 255, 132, 0, 0, 0],
            &[0, 139, 255, 68, 0, 0, 81, 255, 255, 233, 3, 0, 0],
            &[0, 209, 231, 0, 0, 27, 237, 154, 134, 255, 54, 0, 0],
            &[0, 235, 199, 0, 1, 189, 216, 10, 103, 255, 80, 0, 0],
            &[0, 233, 199, 0, 118, 249, 50, 0, 106, 255, 80, 0, 0],
            &[0, 198, 239, 53, 249, 117, 0, 0, 148, 255, 47, 0, 0],
            &[0, 119, 255, 255, 188, 1, 0, 16, 235, 222, 2, 0, 0],
            &[0, 12, 255, 255, 145, 60, 84, 207, 254, 82, 0, 0, 0],
            &[0, 78, 255, 252, 251, 255, 255, 220, 83, 0, 0, 0, 0],
            &[0, 15, 95, 0, 12, 42, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ù' / 0xf9
        'ù' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 99, 141, 51, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 239, 204, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 240, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 119, 1, 0, 0, 0, 0],
            &[0, 0, 27, 47, 6, 0, 0, 0, 0, 39, 41, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 219, 225, 0, 0],
            &[0, 0, 144, 255, 41, 0, 0, 0, 3, 244, 225, 0, 0],
            &[0, 0, 115, 255, 86, 0, 0, 0, 75, 255, 225, 0, 0],
            &[0, 0, 44, 252, 226, 91, 58, 104, 231, 231, 225, 0, 0],
            &[0, 0, 0, 98, 239, 255, 255, 230, 87, 147, 225, 0, 0],
            &[0, 0, 0, 0, 2, 31, 32, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ú' / 0xfa
        'ú' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 61, 141, 91, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 218, 232, 32, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 137, 235, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 122, 36, 0, 0, 0, 0, 0],
            &[0, 0, 27, 47, 6, 0, 0, 0, 0, 39, 41, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 219, 225, 0, 0],
            &[0, 0, 144, 255, 41, 0, 0, 0, 3, 244, 225, 0, 0],
            &[0, 0, 115, 255, 86, 0, 0, 0, 75, 255, 225, 0, 0],
            &[0, 0, 44, 252, 226, 91, 58, 104, 231, 231, 225, 0, 0],
            &[0, 0, 0, 98, 239, 255, 255, 230, 87, 147, 225, 0, 0],
            &[0, 0, 0, 0, 2, 31, 32, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'û' / 0xfb
        'û' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 141, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 222, 222, 254, 78, 0, 0, 0, 0],
            &[0, 0, 0, 9, 195, 189, 11, 117, 243, 55, 0, 0, 0],
            &[0, 0, 0, 54, 109, 5, 0, 0, 71, 103, 0, 0, 0],
            &[0, 0, 27, 47, 6, 0, 0, 0, 0, 39, 41, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 219, 225, 0, 0],
            &[0, 0, 144, 255, 41, 0, 0, 0, 3, 244, 225, 0, 0],
            &[0, 0, 115, 255, 86, 0, 0, 0, 75, 255, 225, 0, 0],
            &[0, 0, 44, 252, 226, 91, 58, 104, 231, 231, 225, 0, 0],
            &[0, 0, 0, 98, 239, 255, 255, 230, 87, 147, 225, 0, 0],
            &[0, 0, 0, 0, 2, 31, 32, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ü' / 0xfc
        'ü' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 70, 239, 80, 0, 6, 221, 162, 0, 0, 0],
            &[0, 0, 0, 87, 255, 98, 0, 11, 242, 188, 0, 0, 0],
            &[0, 0, 0, 0, 22, 0, 0, 0, 16, 5, 0, 0, 0],
            &[0, 0, 27, 47, 6, 0, 0, 0, 0, 39, 41, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 219, 225, 0, 0],
            &[0, 0, 144, 255, 41, 0, 0, 0, 3, 244, 225, 0, 0],
            &[0, 0, 115, 255, 86, 0, 0, 0, 75, 255, 225, 0, 0],
            &[0, 0, 44, 252, 226, 91, 58, 104, 231, 231, 225, 0, 0],
            &[0, 0, 0, 98, 239, 255, 255, 230, 87, 147, 225, 0, 0],
            &[0, 0, 0, 0, 2, 31, 32, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ý' / 0xfd
        'ý' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 61, 141, 91, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 218, 232, 32, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 137, 235, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 122, 36, 0, 0, 0, 0, 0],
            &[0, 5, 47, 26, 0, 0, 0, 0, 0, 11, 47, 23, 0],
            &[0, 1, 218, 205, 0, 0, 0, 0, 0, 115, 255, 69, 0],
            &[0, 0, 113, 255, 54, 0, 0, 0, 0, 200, 227, 2, 0],
            &[0, 0, 17, 246, 158, 0, 0, 0, 31, 255, 135, 0, 0],
            &[0, 0, 0, 156, 245, 16, 0, 0, 116, 255, 40, 0, 0],
            &[0, 0, 0, 50, 255, 110, 0, 0, 201, 201, 0, 0, 0],
            &[0, 0, 0, 0, 199, 213, 0, 32, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 93, 255, 55, 114, 249, 18, 0, 0, 0],
            &[0, 0, 0, 0, 7, 235, 143, 196, 172, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 136, 231, 253, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 253, 233, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 247, 143, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 113, 255, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 232, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 162, 136, 37, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'þ' / 0xfe
        'þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 54, 112, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 255, 53, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 255, 53, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 255, 53, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 255, 53, 11, 68, 87, 45, 0, 0, 0, 0, 0],
            &[0, 122, 255, 102, 231, 255, 255, 255, 176, 10, 0, 0, 0],
            &[0, 122, 255, 241, 105, 21, 33, 144, 255, 148, 0, 0, 0],
            &[0, 122, 255, 151, 0, 0, 0, 0, 207, 246, 12, 0, 0],
            &[0, 122, 255, 81, 0, 0, 0, 0, 133, 255, 61, 0, 0],
            &[0, 122, 255, 57, 0, 0, 0, 0, 109, 255, 84, 0, 0],
            &[0, 122, 255, 68, 0, 0, 0, 0, 115, 255, 78, 0, 0],
            &[0, 122, 255, 99, 0, 0, 0, 0, 146, 255, 51, 0, 0],
            &[0, 122, 255, 185, 0, 0, 0, 6, 229, 235, 2, 0, 0],
            &[0, 122, 255, 236, 158, 71, 81, 192, 255, 112, 0, 0, 0],
            &[0, 122, 255, 76, 189, 255, 255, 242, 123, 0, 0, 0, 0],
            &[0, 122, 255, 53, 0, 18, 38, 7, 0, 0, 0, 0, 0],
            &[0, 122, 255, 53, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 255, 53, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 143, 30, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ÿ' / 0xff
        'ÿ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 70, 239, 80, 0, 6, 221, 162, 0, 0, 0],
            &[0, 0, 0, 87, 255, 98, 0, 11, 242, 188, 0, 0, 0],
            &[0, 0, 0, 0, 22, 0, 0, 0, 16, 5, 0, 0, 0],
            &[0, 5, 47, 26, 0, 0, 0, 0, 0, 11, 47, 23, 0],
            &[0, 1, 218, 205, 0, 0, 0, 0, 0, 115, 255, 69, 0],
            &[0, 0, 113, 255, 54, 0, 0, 0, 0, 200, 227, 2, 0],
            &[0, 0, 17, 246, 158, 0, 0, 0, 31, 255, 135, 0, 0],
            &[0, 0, 0, 156, 245, 16, 0, 0, 116, 255, 40, 0, 0],
            &[0, 0, 0, 50, 255, 110, 0, 0, 201, 201, 0, 0, 0],
            &[0, 0, 0, 0, 199, 213, 0, 32, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 93, 255, 55, 114, 249, 18, 0, 0, 0],
            &[0, 0, 0, 0, 7, 235, 143, 196, 172, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 136, 231, 253, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 253, 233, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 247, 143, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 113, 255, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 232, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 162, 136, 37, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ā' / 0x100
        'Ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 121, 121, 121, 121, 121, 61, 0, 0, 0, 0],
            &[0, 0, 29, 224, 224, 224, 224, 224, 114, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 159, 110, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 255, 239, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 228, 214, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 97, 242, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 252, 18, 173, 235, 3, 0, 0, 0, 0],
            &[0, 0, 0, 222, 191, 0, 94, 255, 65, 0, 0, 0, 0],
            &[0, 0, 49, 255, 112, 0, 18, 251, 149, 0, 0, 0, 0],
            &[0, 0, 132, 255, 32, 0, 0, 191, 230, 1, 0, 0, 0],
            &[0, 0, 215, 255, 174, 174, 174, 255, 255, 59, 0, 0, 0],
            &[0, 41, 255, 255, 214, 214, 214, 247, 255, 142, 0, 0, 0],
            &[0, 124, 255, 50, 0, 0, 0, 0, 208, 225, 0, 0, 0],
            &[0, 207, 225, 0, 0, 0, 0, 0, 129, 255, 53, 0, 0],
            &[34, 255, 147, 0, 0, 0, 0, 0, 49, 255, 136, 0, 0],
            &[117, 255, 67, 0, 0, 0, 0, 0, 0, 225, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ā' / 0x101
        'ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 23, 23, 23, 23, 14, 0, 0, 0],
            &[0, 0, 0, 9, 255, 255, 255, 255, 255, 155, 0, 0, 0],
            &[0, 0, 0, 2, 66, 66, 66, 66, 66, 40, 0, 0, 0],
            &[0, 0, 0, 0, 16, 69, 90, 69, 24, 0, 0, 0, 0],
            &[0, 0, 0, 148, 248, 255, 255, 255, 250, 142, 2, 0, 0],
            &[0, 0, 0, 105, 115, 41, 12, 36, 167, 255, 91, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 8, 251, 160, 0, 0],
            &[0, 0, 0, 0, 0, 3, 24, 38, 45, 231, 176, 0, 0],
            &[0, 0, 0, 57, 184, 247, 255, 255, 255, 255, 176, 0, 0],
            &[0, 0, 48, 248, 212, 90, 38, 19, 10, 223, 176, 0, 0],
            &[0, 0, 145, 255, 38, 0, 0, 0, 0, 240, 176, 0, 0],
            &[0, 0, 156, 252, 7, 0, 0, 0, 85, 255, 176, 0, 0],
            &[0, 0, 96, 255, 170, 65, 64, 135, 231, 239, 176, 0, 0],
            &[0, 0, 1, 141, 249, 255, 255, 199, 47, 190, 176, 0, 0],
            &[0, 0, 0, 0, 8, 38, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ă' / 0x102
        'Ă' => &[
            &[0, 0, 24, 106, 0, 0, 0, 63, 67, 0, 0, 0, 0],
            &[0, 0, 16, 243, 127, 35, 74, 228, 101, 0, 0, 0, 0],
            &[0, 0, 0, 76, 228, 255, 247, 144, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 159, 110, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 255, 239, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 228, 214, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 97, 242, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 252, 18, 173, 235, 3, 0, 0, 0, 0],
            &[0, 0, 0, 222, 191, 0, 94, 255, 65, 0, 0, 0, 0],
            &[0, 0, 49, 255, 112, 0, 18, 251, 149, 0, 0, 0, 0],
            &[0, 0, 132, 255, 32, 0, 0, 191, 230, 1, 0, 0, 0],
            &[0, 0, 215, 255, 174, 174, 174, 255, 255, 59, 0, 0, 0],
            &[0, 41, 255, 255, 214, 214, 214, 247, 255, 142, 0, 0, 0],
            &[0, 124, 255, 50, 0, 0, 0, 0, 208, 225, 0, 0, 0],
            &[0, 207, 225, 0, 0, 0, 0, 0, 129, 255, 53, 0, 0],
            &[34, 255, 147, 0, 0, 0, 0, 0, 49, 255, 136, 0, 0],
            &[117, 255, 67, 0, 0, 0, 0, 0, 0, 225, 219, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ă' / 0x103
        'ă' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 14, 0, 0, 0, 6, 11, 0, 0, 0],
            &[0, 0, 0, 14, 253, 61, 0, 0, 164, 159, 0, 0, 0],
            &[0, 0, 0, 0, 137, 249, 194, 224, 238, 42, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 112, 95, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 69, 90, 69, 24, 0, 0, 0, 0],
            &[0, 0, 0, 148, 248, 255, 255, 255, 250, 142, 2, 0, 0],
            &[0, 0, 0, 105, 115, 41, 12, 36, 167, 255, 91, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 8, 251, 160, 0, 0],
            &[0, 0, 0, 0, 0, 3, 24, 38, 45, 231, 176, 0, 0],
            &[0, 0, 0, 57, 184, 247, 255, 255, 255, 255, 176, 0, 0],
            &[0, 0, 48, 248, 212, 90, 38, 19, 10, 223, 176, 0, 0],
            &[0, 0, 145, 255, 38, 0, 0, 0, 0, 240, 176, 0, 0],
            &[0, 0, 156, 252, 7, 0, 0, 0, 85, 255, 176, 0, 0],
            &[0, 0, 96, 255, 170, 65, 64, 135, 231, 239, 176, 0, 0],
            &[0, 0, 1, 141, 249, 255, 255, 199, 47, 190, 176, 0, 0],
            &[0, 0, 0, 0, 8, 38, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ą' / 0x104
        'Ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 159, 110, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 255, 239, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 228, 214, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 97, 242, 155, 0, 0, 0, 0, 0],
            &[0, 0, 0, 139, 252, 18, 173, 235, 3, 0, 0, 0, 0],
            &[0, 0, 0, 222, 191, 0, 94, 255, 65, 0, 0, 0, 0],
            &[0, 0, 49, 255, 112, 0, 18, 251, 149, 0, 0, 0, 0],
            &[0, 0, 132, 255, 32, 0, 0, 191, 230, 1, 0, 0, 0],
            &[0, 0, 215, 255, 174, 174, 174, 255, 255, 59, 0, 0, 0],
            &[0, 41, 255, 255, 214, 214, 214, 247, 255, 142, 0, 0, 0],
            &[0, 124, 255, 50, 0, 0, 0, 0, 208, 225, 0, 0, 0],
            &[0, 207, 225, 0, 0, 0, 0, 0, 129, 255, 53, 0, 0],
            &[34, 255, 147, 0, 0, 0, 0, 0, 49, 255, 136, 0, 0],
            &[117, 255, 67, 0, 0, 0, 0, 0, 0, 230, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 2, 180, 149, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 95, 236, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 150, 218, 8, 1, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 14, 39, 8, 0],
        ],
        // letter: 'ą' / 0x105
        'ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 69, 90, 69, 24, 0, 0, 0, 0, 0],
            &[0, 0, 148, 248, 255, 255, 255, 250, 142, 2, 0, 0, 0],
            &[0, 0, 105, 115, 41, 12, 36, 167, 255, 91, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 8, 251, 160, 0, 0, 0],
            &[0, 0, 0, 0, 3, 24, 38, 45, 231, 176, 0, 0, 0],
            &[0, 0, 57, 184, 247, 255, 255, 255, 255, 176, 0, 0, 0],
            &[0, 48, 248, 212, 90, 38, 19, 10, 223, 176, 0, 0, 0],
            &[0, 145, 255, 38, 0, 0, 0, 0, 240, 176, 0, 0, 0],
            &[0, 156, 252, 7, 0, 0, 0, 85, 255, 176, 0, 0, 0],
            &[0, 96, 255, 170, 65, 64, 135, 231, 239, 176, 0, 0, 0],
            &[0, 1, 141, 249, 255, 255, 199, 47, 222, 189, 0, 0, 0],
            &[0, 0, 0, 8, 38, 20, 0, 53, 230, 48, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 209, 127, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 255, 113, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 32, 29, 0, 0, 0],
        ],
        // letter: 'Ć' / 0x106
        'Ć' => &[
            &[0, 0, 0, 0, 0, 0, 0, 100, 239, 149, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 235, 198, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 174, 194, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 143, 181, 179, 149, 79, 5, 0, 0],
            &[0, 0, 1, 146, 254, 251, 208, 210, 243, 255, 59, 0, 0],
            &[0, 0, 124, 255, 177, 22, 0, 0, 6, 68, 0, 0, 0],
            &[0, 18, 247, 214, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 255, 103, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 154, 255, 39, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 179, 255, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 255, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 255, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 255, 69, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 255, 154, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 203, 250, 70, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 237, 252, 169, 114, 112, 148, 186, 0, 0, 0],
            &[0, 0, 0, 41, 172, 252, 255, 255, 250, 168, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 34, 35, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ć' / 0x107
        'ć' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 126, 141, 25, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 90, 255, 145, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 234, 160, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 61, 101, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 67, 89, 69, 24, 0, 0, 0],
            &[0, 0, 0, 2, 127, 246, 255, 255, 255, 252, 98, 0, 0],
            &[0, 0, 0, 123, 255, 194, 71, 35, 52, 124, 38, 0, 0],
            &[0, 0, 8, 241, 221, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 134, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 87, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 86, 255, 103, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 255, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 238, 230, 16, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 107, 255, 218, 110, 76, 95, 143, 98, 0, 0],
            &[0, 0, 0, 0, 98, 221, 255, 255, 255, 231, 85, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 42, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĉ' / 0x108
        'Ĉ' => &[
            &[0, 0, 0, 0, 0, 91, 239, 217, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 243, 150, 219, 185, 4, 0, 0, 0],
            &[0, 0, 0, 31, 227, 123, 0, 26, 202, 165, 0, 0, 0],
            &[0, 0, 0, 10, 21, 0, 0, 0, 5, 26, 0, 0, 0],
            &[0, 0, 0, 0, 54, 143, 181, 179, 149, 79, 5, 0, 0],
            &[0, 0, 1, 146, 254, 251, 208, 210, 243, 255, 59, 0, 0],
            &[0, 0, 124, 255, 177, 22, 0, 0, 6, 68, 0, 0, 0],
            &[0, 18, 247, 214, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 255, 103, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 154, 255, 39, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 179, 255, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 255, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 255, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 255, 69, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 255, 154, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 203, 250, 70, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 237, 252, 169, 114, 112, 148, 186, 0, 0, 0],
            &[0, 0, 0, 41, 172, 252, 255, 255, 250, 168, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 34, 35, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĉ' / 0x109
        'ĉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 118, 141, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 106, 252, 226, 203, 9, 0, 0, 0],
            &[0, 0, 0, 0, 72, 245, 87, 24, 210, 176, 4, 0, 0],
            &[0, 0, 0, 0, 112, 56, 0, 0, 12, 117, 45, 0, 0],
            &[0, 0, 0, 0, 0, 11, 67, 89, 69, 24, 0, 0, 0],
            &[0, 0, 0, 2, 127, 246, 255, 255, 255, 252, 98, 0, 0],
            &[0, 0, 0, 123, 255, 194, 71, 35, 52, 124, 38, 0, 0],
            &[0, 0, 8, 241, 221, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 134, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 87, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 86, 255, 103, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 255, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 238, 230, 16, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 107, 255, 218, 110, 76, 95, 143, 98, 0, 0],
            &[0, 0, 0, 0, 98, 221, 255, 255, 255, 231, 85, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 42, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ċ' / 0x10a
        'Ċ' => &[
            &[0, 0, 0, 0, 0, 19, 179, 119, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 255, 229, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 119, 70, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 143, 181, 179, 149, 79, 5, 0, 0],
            &[0, 0, 1, 146, 254, 251, 208, 210, 243, 255, 59, 0, 0],
            &[0, 0, 124, 255, 177, 22, 0, 0, 6, 68, 0, 0, 0],
            &[0, 18, 247, 214, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 255, 103, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 154, 255, 39, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 179, 255, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 255, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 255, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 255, 69, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 255, 154, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 203, 250, 70, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 237, 252, 169, 114, 112, 148, 186, 0, 0, 0],
            &[0, 0, 0, 41, 172, 252, 255, 255, 250, 168, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 34, 35, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ċ' / 0x10b
        'ċ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 82, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 219, 255, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 149, 218, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 67, 89, 69, 24, 0, 0, 0],
            &[0, 0, 0, 2, 127, 246, 255, 255, 255, 252, 98, 0, 0],
            &[0, 0, 0, 123, 255, 194, 71, 35, 52, 124, 38, 0, 0],
            &[0, 0, 8, 241, 221, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 134, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 87, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 86, 255, 103, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 255, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 238, 230, 16, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 107, 255, 218, 110, 76, 95, 143, 98, 0, 0],
            &[0, 0, 0, 0, 98, 221, 255, 255, 255, 231, 85, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 42, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Č' / 0x10c
        'Č' => &[
            &[0, 0, 0, 45, 224, 92, 0, 13, 168, 186, 2, 0, 0],
            &[0, 0, 0, 0, 79, 252, 129, 204, 214, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 123, 255, 242, 34, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 26, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 143, 181, 179, 149, 79, 5, 0, 0],
            &[0, 0, 1, 146, 254, 251, 208, 210, 243, 255, 59, 0, 0],
            &[0, 0, 124, 255, 177, 22, 0, 0, 6, 68, 0, 0, 0],
            &[0, 18, 247, 214, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 255, 103, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 154, 255, 39, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 179, 255, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 184, 255, 5, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 165, 255, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 129, 255, 69, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 61, 255, 154, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 203, 250, 70, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 237, 252, 169, 114, 112, 148, 186, 0, 0, 0],
            &[0, 0, 0, 41, 172, 252, 255, 255, 250, 168, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 34, 35, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'č' / 0x10d
        'č' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 74, 0, 0, 20, 134, 51, 0, 0],
            &[0, 0, 0, 0, 66, 247, 120, 42, 226, 168, 2, 0, 0],
            &[0, 0, 0, 0, 0, 93, 255, 245, 194, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 101, 124, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 67, 89, 69, 24, 0, 0, 0],
            &[0, 0, 0, 2, 127, 246, 255, 255, 255, 252, 98, 0, 0],
            &[0, 0, 0, 123, 255, 194, 71, 35, 52, 124, 38, 0, 0],
            &[0, 0, 8, 241, 221, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 255, 134, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 87, 255, 101, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 86, 255, 103, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 255, 140, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 238, 230, 16, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 107, 255, 218, 110, 76, 95, 143, 98, 0, 0],
            &[0, 0, 0, 0, 98, 221, 255, 255, 255, 231, 85, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 42, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ď' / 0x10e
        'Ď' => &[
            &[0, 0, 170, 176, 16, 0, 84, 226, 59, 0, 0, 0, 0],
            &[0, 0, 12, 206, 208, 125, 253, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 237, 255, 133, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 26, 3, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 144, 142, 119, 73, 3, 0, 0, 0, 0, 0],
            &[0, 171, 255, 235, 239, 255, 255, 225, 59, 0, 0, 0, 0],
            &[0, 171, 255, 10, 0, 12, 97, 238, 249, 51, 0, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 48, 250, 206, 0, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 43, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 92, 255, 102, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 65, 255, 125, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 57, 255, 129, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 78, 255, 107, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 134, 255, 63, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 12, 229, 233, 3, 0, 0],
            &[0, 171, 255, 10, 0, 0, 15, 180, 255, 99, 0, 0, 0],
            &[0, 171, 255, 128, 131, 166, 239, 255, 139, 0, 0, 0, 0],
            &[0, 171, 255, 255, 248, 221, 165, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ď' / 0x10f
        'ď' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 64, 112, 13, 55, 112, 18],
            &[0, 0, 0, 0, 0, 0, 0, 146, 255, 30, 157, 231, 3],
            &[0, 0, 0, 0, 0, 0, 0, 146, 255, 30, 213, 117, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 255, 38, 146, 7, 0],
            &[0, 0, 0, 41, 85, 65, 6, 145, 255, 30, 0, 0, 0],
            &[0, 4, 160, 255, 255, 255, 213, 151, 255, 30, 0, 0, 0],
            &[0, 111, 255, 176, 29, 24, 165, 251, 255, 30, 0, 0, 0],
            &[0, 212, 244, 15, 0, 0, 9, 239, 255, 30, 0, 0, 0],
            &[11, 255, 188, 0, 0, 0, 0, 178, 255, 30, 0, 0, 0],
            &[30, 255, 161, 0, 0, 0, 0, 152, 255, 30, 0, 0, 0],
            &[28, 255, 164, 0, 0, 0, 0, 152, 255, 30, 0, 0, 0],
            &[7, 251, 190, 0, 0, 0, 0, 180, 255, 30, 0, 0, 0],
            &[0, 198, 249, 20, 0, 0, 18, 246, 255, 30, 0, 0, 0],
            &[0, 84, 255, 200, 76, 80, 203, 233, 255, 30, 0, 0, 0],
            &[0, 0, 115, 244, 255, 253, 158, 95, 255, 30, 0, 0, 0],
            &[0, 0, 0, 9, 41, 19, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Đ' / 0x110
        'Đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 144, 144, 144, 124, 86, 10, 0, 0, 0, 0],
            &[0, 0, 122, 255, 239, 236, 253, 255, 240, 92, 0, 0, 0],
            &[0, 0, 122, 255, 58, 0, 5, 75, 217, 255, 94, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 19, 231, 240, 14, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 114, 255, 92, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 43, 255, 150, 0],
            &[0, 114, 237, 255, 173, 114, 62, 0, 0, 16, 255, 174, 0],
            &[0, 255, 255, 255, 255, 255, 140, 0, 0, 8, 255, 178, 0],
            &[0, 8, 131, 255, 67, 8, 4, 0, 0, 30, 255, 155, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 85, 255, 112, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 192, 254, 30, 0],
            &[0, 0, 122, 255, 58, 0, 0, 5, 141, 255, 148, 0, 0],
            &[0, 0, 122, 255, 153, 128, 156, 227, 255, 177, 10, 0, 0],
            &[0, 0, 122, 255, 255, 251, 226, 181, 86, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'đ' / 0x111
        'đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 91, 99, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 207, 225, 0, 0, 0],
            &[0, 0, 0, 0, 32, 186, 186, 186, 255, 255, 186, 74, 0],
            &[0, 0, 0, 0, 23, 134, 134, 134, 255, 255, 134, 53, 0],
            &[0, 0, 0, 3, 45, 62, 24, 0, 207, 225, 0, 0, 0],
            &[0, 0, 71, 225, 255, 255, 253, 139, 195, 225, 0, 0, 0],
            &[0, 39, 249, 240, 97, 39, 78, 227, 246, 225, 0, 0, 0],
            &[0, 151, 255, 86, 0, 0, 0, 71, 255, 225, 0, 0, 0],
            &[0, 213, 245, 3, 0, 0, 0, 2, 242, 225, 0, 0, 0],
            &[0, 236, 217, 0, 0, 0, 0, 0, 214, 225, 0, 0, 0],
            &[0, 235, 214, 0, 0, 0, 0, 0, 213, 225, 0, 0, 0],
            &[0, 206, 242, 2, 0, 0, 0, 1, 241, 225, 0, 0, 0],
            &[0, 141, 255, 81, 0, 0, 0, 78, 255, 225, 0, 0, 0],
            &[0, 30, 242, 241, 109, 62, 106, 238, 233, 225, 0, 0, 0],
            &[0, 0, 56, 212, 255, 255, 239, 98, 152, 225, 0, 0, 0],
            &[0, 0, 0, 0, 29, 36, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ē' / 0x112
        'Ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 121, 121, 121, 121, 121, 75, 0, 0, 0],
            &[0, 0, 0, 3, 224, 224, 224, 224, 224, 140, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 144, 144, 144, 144, 144, 144, 144, 113, 0, 0],
            &[0, 0, 49, 255, 250, 244, 244, 244, 244, 244, 191, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 236, 216, 216, 216, 216, 216, 91, 0, 0],
            &[0, 0, 49, 255, 215, 172, 172, 172, 172, 172, 73, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 196, 133, 133, 133, 133, 133, 104, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ē' / 0x113
        'ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 23, 23, 23, 23, 23, 13, 0, 0, 0, 0],
            &[0, 0, 14, 255, 255, 255, 255, 255, 150, 0, 0, 0, 0],
            &[0, 0, 3, 66, 66, 66, 66, 66, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 81, 79, 28, 0, 0, 0, 0, 0],
            &[0, 0, 23, 185, 255, 255, 255, 252, 145, 3, 0, 0, 0],
            &[0, 6, 209, 245, 105, 23, 21, 133, 255, 124, 0, 0, 0],
            &[0, 101, 255, 105, 0, 0, 0, 0, 192, 236, 4, 0, 0],
            &[0, 178, 252, 46, 35, 35, 35, 35, 146, 255, 43, 0, 0],
            &[0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 64, 0, 0],
            &[0, 205, 239, 59, 59, 59, 59, 59, 59, 59, 15, 0, 0],
            &[0, 171, 254, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 176, 255, 168, 83, 62, 89, 145, 184, 0, 0, 0],
            &[0, 0, 4, 124, 228, 255, 255, 255, 224, 122, 0, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĕ' / 0x114
        'Ĕ' => &[
            &[0, 0, 0, 11, 112, 7, 0, 0, 50, 80, 0, 0, 0],
            &[0, 0, 0, 1, 228, 150, 36, 65, 212, 131, 0, 0, 0],
            &[0, 0, 0, 0, 58, 218, 255, 250, 163, 9, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 14, 3, 0, 0, 0, 0, 0],
            &[0, 0, 28, 144, 144, 144, 144, 144, 144, 144, 113, 0, 0],
            &[0, 0, 49, 255, 250, 244, 244, 244, 244, 244, 191, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 236, 216, 216, 216, 216, 216, 91, 0, 0],
            &[0, 0, 49, 255, 215, 172, 172, 172, 172, 172, 73, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 196, 133, 133, 133, 133, 133, 104, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĕ' / 0x115
        'ĕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 14, 0, 0, 0, 6, 10, 0, 0, 0, 0],
            &[0, 0, 18, 253, 57, 0, 0, 169, 154, 0, 0, 0, 0],
            &[0, 0, 0, 142, 249, 194, 225, 236, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 113, 95, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 81, 79, 28, 0, 0, 0, 0, 0],
            &[0, 0, 23, 185, 255, 255, 255, 252, 145, 3, 0, 0, 0],
            &[0, 6, 209, 245, 105, 23, 21, 133, 255, 124, 0, 0, 0],
            &[0, 101, 255, 105, 0, 0, 0, 0, 192, 236, 4, 0, 0],
            &[0, 178, 252, 46, 35, 35, 35, 35, 146, 255, 43, 0, 0],
            &[0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 64, 0, 0],
            &[0, 205, 239, 59, 59, 59, 59, 59, 59, 59, 15, 0, 0],
            &[0, 171, 254, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 176, 255, 168, 83, 62, 89, 145, 184, 0, 0, 0],
            &[0, 0, 4, 124, 228, 255, 255, 255, 224, 122, 0, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ė' / 0x116
        'Ė' => &[
            &[0, 0, 0, 0, 0, 20, 180, 118, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 76, 255, 227, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 120, 70, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 144, 144, 144, 144, 144, 144, 144, 113, 0, 0],
            &[0, 0, 49, 255, 250, 244, 244, 244, 244, 244, 191, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 236, 216, 216, 216, 216, 216, 91, 0, 0],
            &[0, 0, 49, 255, 215, 172, 172, 172, 172, 172, 73, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 196, 133, 133, 133, 133, 133, 104, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ė' / 0x117
        'ė' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 84, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 255, 212, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 219, 142, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 81, 79, 28, 0, 0, 0, 0, 0],
            &[0, 0, 23, 185, 255, 255, 255, 252, 145, 3, 0, 0, 0],
            &[0, 6, 209, 245, 105, 23, 21, 133, 255, 124, 0, 0, 0],
            &[0, 101, 255, 105, 0, 0, 0, 0, 192, 236, 4, 0, 0],
            &[0, 178, 252, 46, 35, 35, 35, 35, 146, 255, 43, 0, 0],
            &[0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 64, 0, 0],
            &[0, 205, 239, 59, 59, 59, 59, 59, 59, 59, 15, 0, 0],
            &[0, 171, 254, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 176, 255, 168, 83, 62, 89, 145, 184, 0, 0, 0],
            &[0, 0, 4, 124, 228, 255, 255, 255, 224, 122, 0, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ę' / 0x118
        'Ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 144, 144, 144, 144, 144, 144, 144, 113, 0, 0],
            &[0, 0, 49, 255, 250, 244, 244, 244, 244, 244, 191, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 236, 216, 216, 216, 216, 216, 91, 0, 0],
            &[0, 0, 49, 255, 215, 172, 172, 172, 172, 172, 73, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 196, 133, 133, 133, 133, 133, 104, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 24, 223, 84, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 162, 174, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 217, 158, 1, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 24, 35, 2, 0, 0],
        ],
        // letter: 'ę' / 0x119
        'ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 81, 79, 28, 0, 0, 0, 0, 0],
            &[0, 0, 23, 185, 255, 255, 255, 252, 145, 3, 0, 0, 0],
            &[0, 6, 209, 245, 105, 23, 21, 133, 255, 124, 0, 0, 0],
            &[0, 101, 255, 105, 0, 0, 0, 0, 192, 236, 4, 0, 0],
            &[0, 178, 252, 46, 35, 35, 35, 35, 146, 255, 43, 0, 0],
            &[0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 64, 0, 0],
            &[0, 205, 239, 59, 59, 59, 59, 59, 59, 59, 15, 0, 0],
            &[0, 171, 254, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 176, 255, 168, 83, 62, 89, 145, 184, 0, 0, 0],
            &[0, 0, 4, 124, 228, 255, 255, 255, 255, 187, 0, 0, 0],
            &[0, 0, 0, 0, 1, 25, 42, 177, 165, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 69, 250, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 115, 246, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 8, 38, 12, 0, 0, 0],
        ],
        // letter: 'Ě' / 0x11a
        'Ě' => &[
            &[0, 0, 0, 46, 224, 91, 0, 13, 169, 185, 1, 0, 0],
            &[0, 0, 0, 0, 81, 252, 128, 205, 212, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 125, 255, 241, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 26, 18, 0, 0, 0, 0, 0],
            &[0, 0, 28, 144, 144, 144, 144, 144, 144, 144, 113, 0, 0],
            &[0, 0, 49, 255, 250, 244, 244, 244, 244, 244, 191, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 236, 216, 216, 216, 216, 216, 91, 0, 0],
            &[0, 0, 49, 255, 215, 172, 172, 172, 172, 172, 73, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 131, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 255, 196, 133, 133, 133, 133, 133, 104, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 200, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ě' / 0x11b
        'ě' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 50, 133, 18, 0, 0, 78, 128, 0, 0, 0, 0],
            &[0, 0, 3, 172, 222, 35, 130, 246, 63, 0, 0, 0, 0],
            &[0, 0, 0, 8, 201, 245, 255, 84, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 124, 97, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 81, 79, 28, 0, 0, 0, 0, 0],
            &[0, 0, 23, 185, 255, 255, 255, 252, 145, 3, 0, 0, 0],
            &[0, 6, 209, 245, 105, 23, 21, 133, 255, 124, 0, 0, 0],
            &[0, 101, 255, 105, 0, 0, 0, 0, 192, 236, 4, 0, 0],
            &[0, 178, 252, 46, 35, 35, 35, 35, 146, 255, 43, 0, 0],
            &[0, 208, 255, 255, 255, 255, 255, 255, 255, 255, 64, 0, 0],
            &[0, 205, 239, 59, 59, 59, 59, 59, 59, 59, 15, 0, 0],
            &[0, 171, 254, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 255, 135, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 176, 255, 168, 83, 62, 89, 145, 184, 0, 0, 0],
            &[0, 0, 4, 124, 228, 255, 255, 255, 224, 122, 0, 0, 0],
            &[0, 0, 0, 0, 1, 25, 41, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => &[
            &[0, 0, 0, 0, 0, 31, 228, 239, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 204, 199, 170, 233, 33, 0, 0, 0],
            &[0, 0, 0, 3, 181, 184, 14, 3, 149, 218, 23, 0, 0],
            &[0, 0, 0, 2, 26, 2, 0, 0, 0, 24, 8, 0, 0],
            &[0, 0, 0, 0, 5, 90, 156, 184, 168, 126, 33, 0, 0],
            &[0, 0, 0, 31, 214, 255, 243, 210, 226, 253, 184, 0, 0],
            &[0, 0, 6, 216, 253, 117, 7, 0, 0, 31, 43, 0, 0],
            &[0, 0, 108, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 191, 249, 14, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 249, 198, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 255, 169, 0, 0, 2, 41, 41, 41, 41, 2, 0],
            &[0, 32, 255, 158, 0, 0, 13, 255, 255, 255, 255, 17, 0],
            &[0, 15, 255, 174, 0, 0, 4, 91, 91, 187, 255, 17, 0],
            &[0, 0, 243, 217, 0, 0, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 178, 255, 43, 0, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 77, 255, 193, 7, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 0, 162, 255, 219, 121, 91, 109, 209, 255, 17, 0],
            &[0, 0, 0, 1, 114, 224, 255, 255, 255, 240, 168, 8, 0],
            &[0, 0, 0, 0, 0, 0, 20, 41, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĝ' / 0x11d
        'ĝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 141, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 222, 222, 254, 78, 0, 0, 0, 0],
            &[0, 0, 0, 9, 195, 189, 11, 117, 243, 55, 0, 0, 0],
            &[0, 0, 0, 54, 109, 5, 0, 0, 71, 103, 0, 0, 0],
            &[0, 0, 0, 0, 14, 75, 88, 45, 0, 23, 41, 0, 0],
            &[0, 0, 0, 79, 239, 255, 255, 255, 160, 159, 225, 0, 0],
            &[0, 0, 38, 248, 226, 67, 14, 49, 201, 253, 225, 0, 0],
            &[0, 0, 148, 255, 68, 0, 0, 0, 30, 251, 225, 0, 0],
            &[0, 0, 211, 240, 1, 0, 0, 0, 0, 203, 225, 0, 0],
            &[0, 0, 237, 213, 0, 0, 0, 0, 0, 175, 225, 0, 0],
            &[0, 0, 236, 218, 0, 0, 0, 0, 0, 175, 225, 0, 0],
            &[0, 0, 206, 246, 6, 0, 0, 0, 0, 205, 225, 0, 0],
            &[0, 0, 138, 255, 94, 0, 0, 0, 41, 253, 225, 0, 0],
            &[0, 0, 25, 237, 247, 118, 63, 94, 224, 249, 225, 0, 0],
            &[0, 0, 0, 49, 206, 255, 255, 251, 146, 181, 225, 0, 0],
            &[0, 0, 0, 0, 0, 26, 42, 13, 0, 192, 218, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 232, 190, 0, 0],
            &[0, 0, 52, 48, 0, 0, 0, 3, 136, 255, 110, 0, 0],
            &[0, 0, 1, 55, 120, 148, 160, 137, 70, 0, 0, 0, 0],
        ],
        // letter: 'Ğ' / 0x11e
        'Ğ' => &[
            &[0, 0, 0, 0, 90, 41, 0, 0, 16, 112, 2, 0, 0],
            &[0, 0, 0, 0, 152, 204, 52, 47, 161, 208, 0, 0, 0],
            &[0, 0, 0, 0, 18, 182, 253, 254, 203, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 90, 156, 184, 168, 126, 33, 0, 0],
            &[0, 0, 0, 31, 214, 255, 243, 210, 226, 253, 184, 0, 0],
            &[0, 0, 6, 216, 253, 117, 7, 0, 0, 31, 43, 0, 0],
            &[0, 0, 108, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 191, 249, 14, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 249, 198, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 255, 169, 0, 0, 2, 41, 41, 41, 41, 2, 0],
            &[0, 32, 255, 158, 0, 0, 13, 255, 255, 255, 255, 17, 0],
            &[0, 15, 255, 174, 0, 0, 4, 91, 91, 187, 255, 17, 0],
            &[0, 0, 243, 217, 0, 0, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 178, 255, 43, 0, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 77, 255, 193, 7, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 0, 162, 255, 219, 121, 91, 109, 209, 255, 17, 0],
            &[0, 0, 0, 1, 114, 224, 255, 255, 255, 240, 168, 8, 0],
            &[0, 0, 0, 0, 0, 0, 20, 41, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ğ' / 0x11f
        'ğ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 13, 0, 0, 0, 7, 9, 0, 0, 0],
            &[0, 0, 0, 36, 251, 41, 0, 2, 186, 134, 0, 0, 0],
            &[0, 0, 0, 0, 162, 245, 193, 229, 228, 27, 0, 0, 0],
            &[0, 0, 0, 0, 0, 70, 114, 91, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 75, 88, 45, 0, 23, 41, 0, 0],
            &[0, 0, 0, 79, 239, 255, 255, 255, 160, 159, 225, 0, 0],
            &[0, 0, 38, 248, 226, 67, 14, 49, 201, 253, 225, 0, 0],
            &[0, 0, 148, 255, 68, 0, 0, 0, 30, 251, 225, 0, 0],
            &[0, 0, 211, 240, 1, 0, 0, 0, 0, 203, 225, 0, 0],
            &[0, 0, 237, 213, 0, 0, 0, 0, 0, 175, 225, 0, 0],
            &[0, 0, 236, 218, 0, 0, 0, 0, 0, 175, 225, 0, 0],
            &[0, 0, 206, 246, 6, 0, 0, 0, 0, 205, 225, 0, 0],
            &[0, 0, 138, 255, 94, 0, 0, 0, 41, 253, 225, 0, 0],
            &[0, 0, 25, 237, 247, 118, 63, 94, 224, 249, 225, 0, 0],
            &[0, 0, 0, 49, 206, 255, 255, 251, 146, 181, 225, 0, 0],
            &[0, 0, 0, 0, 0, 26, 42, 13, 0, 192, 218, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 232, 190, 0, 0],
            &[0, 0, 52, 48, 0, 0, 0, 3, 136, 255, 110, 0, 0],
            &[0, 0, 1, 55, 120, 148, 160, 137, 70, 0, 0, 0, 0],
        ],
        // letter: 'Ġ' / 0x120
        'Ġ' => &[
            &[0, 0, 0, 0, 0, 0, 139, 172, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 248, 255, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 84, 112, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 90, 156, 184, 168, 126, 33, 0, 0],
            &[0, 0, 0, 31, 214, 255, 243, 210, 226, 253, 184, 0, 0],
            &[0, 0, 6, 216, 253, 117, 7, 0, 0, 31, 43, 0, 0],
            &[0, 0, 108, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 191, 249, 14, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 249, 198, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 255, 169, 0, 0, 2, 41, 41, 41, 41, 2, 0],
            &[0, 32, 255, 158, 0, 0, 13, 255, 255, 255, 255, 17, 0],
            &[0, 15, 255, 174, 0, 0, 4, 91, 91, 187, 255, 17, 0],
            &[0, 0, 243, 217, 0, 0, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 178, 255, 43, 0, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 77, 255, 193, 7, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 0, 162, 255, 219, 121, 91, 109, 209, 255, 17, 0],
            &[0, 0, 0, 1, 114, 224, 255, 255, 255, 240, 168, 8, 0],
            &[0, 0, 0, 0, 0, 0, 20, 41, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ġ' / 0x121
        'ġ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 87, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 100, 255, 192, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 222, 125, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 75, 88, 45, 0, 23, 41, 0, 0],
            &[0, 0, 0, 79, 239, 255, 255, 255, 160, 159, 225, 0, 0],
            &[0, 0, 38, 248, 226, 67, 14, 49, 201, 253, 225, 0, 0],
            &[0, 0, 148, 255, 68, 0, 0, 0, 30, 251, 225, 0, 0],
            &[0, 0, 211, 240, 1, 0, 0, 0, 0, 203, 225, 0, 0],
            &[0, 0, 237, 213, 0, 0, 0, 0, 0, 175, 225, 0, 0],
            &[0, 0, 236, 218, 0, 0, 0, 0, 0, 175, 225, 0, 0],
            &[0, 0, 206, 246, 6, 0, 0, 0, 0, 205, 225, 0, 0],
            &[0, 0, 138, 255, 94, 0, 0, 0, 41, 253, 225, 0, 0],
            &[0, 0, 25, 237, 247, 118, 63, 94, 224, 249, 225, 0, 0],
            &[0, 0, 0, 49, 206, 255, 255, 251, 146, 181, 225, 0, 0],
            &[0, 0, 0, 0, 0, 26, 42, 13, 0, 192, 218, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 232, 190, 0, 0],
            &[0, 0, 52, 48, 0, 0, 0, 3, 136, 255, 110, 0, 0],
            &[0, 0, 1, 55, 120, 148, 160, 137, 70, 0, 0, 0, 0],
        ],
        // letter: 'Ģ' / 0x122
        'Ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 90, 156, 184, 168, 126, 33, 0, 0],
            &[0, 0, 0, 31, 214, 255, 243, 210, 226, 253, 184, 0, 0],
            &[0, 0, 6, 216, 253, 117, 7, 0, 0, 31, 43, 0, 0],
            &[0, 0, 108, 255, 129, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 191, 249, 14, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 249, 198, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 255, 169, 0, 0, 2, 41, 41, 41, 41, 2, 0],
            &[0, 32, 255, 158, 0, 0, 13, 255, 255, 255, 255, 17, 0],
            &[0, 15, 255, 174, 0, 0, 4, 91, 91, 187, 255, 17, 0],
            &[0, 0, 243, 217, 0, 0, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 178, 255, 43, 0, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 77, 255, 193, 7, 0, 0, 0, 149, 255, 17, 0],
            &[0, 0, 0, 162, 255, 219, 121, 91, 109, 209, 255, 17, 0],
            &[0, 0, 0, 1, 114, 224, 255, 255, 255, 240, 168, 8, 0],
            &[0, 0, 0, 0, 0, 0, 20, 41, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 101, 215, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 173, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 41, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ģ' / 0x123
        'ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 28, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 180, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 53, 255, 83, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 124, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 75, 88, 45, 0, 23, 41, 0, 0],
            &[0, 0, 0, 79, 239, 255, 255, 255, 160, 159, 225, 0, 0],
            &[0, 0, 38, 248, 226, 67, 14, 49, 201, 253, 225, 0, 0],
            &[0, 0, 148, 255, 68, 0, 0, 0, 30, 251, 225, 0, 0],
            &[0, 0, 211, 240, 1, 0, 0, 0, 0, 203, 225, 0, 0],
            &[0, 0, 237, 213, 0, 0, 0, 0, 0, 175, 225, 0, 0],
            &[0, 0, 236, 218, 0, 0, 0, 0, 0, 175, 225, 0, 0],
            &[0, 0, 206, 246, 6, 0, 0, 0, 0, 205, 225, 0, 0],
            &[0, 0, 138, 255, 94, 0, 0, 0, 41, 253, 225, 0, 0],
            &[0, 0, 25, 237, 247, 118, 63, 94, 224, 249, 225, 0, 0],
            &[0, 0, 0, 49, 206, 255, 255, 251, 146, 181, 225, 0, 0],
            &[0, 0, 0, 0, 0, 26, 42, 13, 0, 192, 218, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 4, 232, 190, 0, 0],
            &[0, 0, 52, 48, 0, 0, 0, 3, 136, 255, 110, 0, 0],
            &[0, 0, 1, 55, 120, 148, 160, 137, 70, 0, 0, 0, 0],
        ],
        // letter: 'Ĥ' / 0x124
        'Ĥ' => &[
            &[0, 0, 0, 0, 120, 239, 199, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 70, 250, 136, 234, 158, 0, 0, 0, 0, 0],
            &[0, 0, 49, 235, 97, 0, 41, 218, 135, 0, 0, 0, 0],
            &[0, 0, 13, 18, 0, 0, 0, 8, 24, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 0, 92, 144, 10, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 218, 216, 216, 216, 216, 241, 255, 17, 0, 0],
            &[0, 171, 255, 175, 172, 172, 172, 172, 225, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĥ' / 0x125
        'ĥ' => &[
            &[0, 0, 117, 247, 167, 248, 60, 0, 0, 0, 0, 0, 0],
            &[0, 88, 237, 66, 0, 122, 235, 45, 0, 0, 0, 0, 0],
            &[0, 48, 27, 0, 0, 0, 42, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 112, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 122, 255, 58, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 122, 255, 58, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 122, 255, 58, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 122, 255, 58, 2, 57, 85, 52, 1, 0, 0],
            &[0, 0, 0, 122, 255, 77, 207, 255, 255, 255, 198, 25, 0],
            &[0, 0, 0, 122, 255, 225, 138, 23, 24, 149, 255, 163, 0],
            &[0, 0, 0, 122, 255, 180, 0, 0, 0, 4, 233, 230, 0],
            &[0, 0, 0, 122, 255, 93, 0, 0, 0, 0, 195, 248, 0],
            &[0, 0, 0, 122, 255, 64, 0, 0, 0, 0, 188, 249, 0],
            &[0, 0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0],
            &[0, 0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0],
            &[0, 0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0],
            &[0, 0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0],
            &[0, 0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 158, 207, 106, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ħ' / 0x126
        'Ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 0, 92, 144, 10, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[78, 249, 255, 88, 78, 78, 78, 78, 241, 255, 95, 31, 0],
            &[
                255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 102, 0,
            ],
            &[16, 187, 255, 26, 16, 16, 16, 16, 179, 255, 34, 6, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 218, 216, 216, 216, 216, 241, 255, 17, 0, 0],
            &[0, 171, 255, 175, 172, 172, 172, 172, 225, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ħ' / 0x127
        'ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 112, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 186, 255, 255, 245, 186, 186, 107, 0, 0, 0, 0, 0],
            &[0, 134, 255, 255, 192, 134, 134, 76, 0, 0, 0, 0, 0],
            &[0, 0, 122, 255, 58, 2, 57, 85, 52, 1, 0, 0, 0],
            &[0, 0, 122, 255, 77, 207, 255, 255, 255, 198, 25, 0, 0],
            &[0, 0, 122, 255, 225, 138, 23, 24, 149, 255, 163, 0, 0],
            &[0, 0, 122, 255, 180, 0, 0, 0, 4, 233, 230, 0, 0],
            &[0, 0, 122, 255, 93, 0, 0, 0, 0, 195, 248, 0, 0],
            &[0, 0, 122, 255, 64, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĩ' / 0x128
        'Ĩ' => &[
            &[0, 0, 0, 27, 69, 13, 0, 0, 80, 7, 0, 0, 0],
            &[0, 0, 45, 243, 255, 235, 115, 112, 232, 1, 0, 0, 0],
            &[0, 0, 150, 146, 21, 124, 240, 246, 86, 0, 0, 0, 0],
            &[0, 0, 18, 7, 0, 0, 3, 5, 0, 0, 0, 0, 0],
            &[0, 0, 131, 144, 144, 144, 144, 144, 144, 44, 0, 0, 0],
            &[0, 0, 152, 189, 218, 255, 236, 197, 175, 49, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 51, 78, 126, 255, 185, 86, 64, 15, 0, 0, 0],
            &[0, 0, 232, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĩ' / 0x129
        'ĩ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 173, 227, 150, 22, 24, 233, 8, 0, 0, 0],
            &[0, 0, 120, 207, 118, 216, 250, 246, 160, 0, 0, 0, 0],
            &[0, 0, 83, 44, 0, 5, 86, 93, 5, 0, 0, 0, 0],
            &[0, 8, 47, 47, 47, 47, 40, 0, 0, 0, 0, 0, 0],
            &[0, 44, 255, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 30, 49, 226, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 75, 95, 233, 234, 95, 73, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ī' / 0x12a
        'Ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 121, 121, 121, 121, 121, 61, 0, 0, 0, 0],
            &[0, 0, 29, 224, 224, 224, 224, 224, 114, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 144, 144, 144, 144, 144, 144, 44, 0, 0, 0],
            &[0, 0, 152, 189, 218, 255, 236, 197, 175, 49, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 51, 78, 126, 255, 185, 86, 64, 15, 0, 0, 0],
            &[0, 0, 232, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ī' / 0x12b
        'ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 23, 23, 23, 23, 23, 11, 0, 0, 0, 0],
            &[0, 0, 33, 255, 255, 255, 255, 255, 131, 0, 0, 0, 0],
            &[0, 0, 8, 66, 66, 66, 66, 66, 33, 0, 0, 0, 0],
            &[0, 8, 47, 47, 47, 47, 40, 0, 0, 0, 0, 0, 0],
            &[0, 44, 255, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 30, 49, 226, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 75, 95, 233, 234, 95, 73, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => &[
            &[0, 0, 24, 106, 0, 0, 0, 63, 67, 0, 0, 0, 0],
            &[0, 0, 16, 243, 127, 35, 74, 228, 101, 0, 0, 0, 0],
            &[0, 0, 0, 76, 228, 255, 247, 144, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 144, 144, 144, 144, 144, 144, 44, 0, 0, 0],
            &[0, 0, 152, 189, 218, 255, 236, 197, 175, 49, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 51, 78, 126, 255, 185, 86, 64, 15, 0, 0, 0],
            &[0, 0, 232, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĭ' / 0x12d
        'ĭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 13, 0, 0, 0, 7, 9, 0, 0, 0, 0],
            &[0, 0, 36, 251, 41, 0, 2, 186, 134, 0, 0, 0, 0],
            &[0, 0, 0, 162, 245, 193, 229, 228, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 114, 91, 13, 0, 0, 0, 0, 0],
            &[0, 8, 47, 47, 47, 47, 40, 0, 0, 0, 0, 0, 0],
            &[0, 44, 255, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 30, 49, 226, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 75, 95, 233, 234, 95, 73, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Į' / 0x12e
        'Į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 144, 144, 144, 144, 144, 144, 44, 0, 0, 0],
            &[0, 0, 152, 189, 218, 255, 236, 197, 175, 49, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 51, 78, 126, 255, 185, 86, 64, 15, 0, 0, 0],
            &[0, 0, 232, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 84, 221, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 241, 86, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 72, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 36, 22, 0, 0, 0, 0],
        ],
        // letter: 'į' / 0x12f
        'į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 80, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 61, 255, 231, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 215, 159, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 47, 47, 47, 47, 40, 0, 0, 0, 0, 0, 0],
            &[0, 44, 255, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 30, 49, 226, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 75, 95, 233, 234, 95, 73, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 24, 223, 84, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 162, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 217, 158, 1, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 24, 35, 2, 0, 0, 0],
        ],
        // letter: 'İ' / 0x130
        'İ' => &[
            &[0, 0, 0, 0, 36, 185, 97, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 105, 255, 198, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 125, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 144, 144, 144, 144, 144, 144, 44, 0, 0, 0],
            &[0, 0, 152, 189, 218, 255, 236, 197, 175, 49, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 51, 78, 126, 255, 185, 86, 64, 15, 0, 0, 0],
            &[0, 0, 232, 255, 255, 255, 255, 255, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ı' / 0x131
        'ı' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 47, 47, 47, 47, 40, 0, 0, 0, 0, 0, 0],
            &[0, 44, 255, 255, 255, 255, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 30, 49, 226, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 219, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 75, 95, 233, 234, 95, 73, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĳ' / 0x132
        'Ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 144, 88, 0, 0, 0, 30, 144, 72, 0, 0],
            &[0, 0, 25, 255, 156, 0, 0, 0, 53, 255, 127, 0, 0],
            &[0, 0, 25, 255, 156, 0, 0, 0, 53, 255, 127, 0, 0],
            &[0, 0, 25, 255, 156, 0, 0, 0, 53, 255, 127, 0, 0],
            &[0, 0, 25, 255, 156, 0, 0, 0, 53, 255, 127, 0, 0],
            &[0, 0, 25, 255, 156, 0, 0, 0, 53, 255, 127, 0, 0],
            &[0, 0, 25, 255, 156, 0, 0, 0, 53, 255, 127, 0, 0],
            &[0, 0, 25, 255, 156, 0, 0, 0, 53, 255, 127, 0, 0],
            &[0, 0, 25, 255, 156, 0, 0, 0, 53, 255, 127, 0, 0],
            &[0, 0, 25, 255, 156, 0, 0, 0, 55, 255, 127, 0, 0],
            &[0, 0, 5, 51, 31, 0, 0, 0, 78, 255, 112, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 174, 255, 62, 0, 0],
            &[0, 0, 15, 202, 140, 109, 112, 190, 255, 197, 1, 0, 0],
            &[0, 0, 10, 210, 254, 255, 255, 249, 166, 20, 0, 0, 0],
            &[0, 0, 0, 0, 12, 39, 39, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĳ' / 0x133
        'ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 74, 0, 0, 0, 34, 86, 7, 0, 0],
            &[0, 0, 6, 244, 255, 43, 0, 0, 194, 255, 96, 0, 0],
            &[0, 0, 0, 175, 210, 12, 0, 0, 125, 222, 48, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 43, 0, 0, 0, 21, 47, 11, 0, 0],
            &[0, 0, 0, 198, 238, 0, 0, 0, 116, 255, 64, 0, 0],
            &[0, 0, 0, 198, 238, 0, 0, 0, 116, 255, 64, 0, 0],
            &[0, 0, 0, 198, 238, 0, 0, 0, 116, 255, 64, 0, 0],
            &[0, 0, 0, 198, 238, 0, 0, 0, 116, 255, 64, 0, 0],
            &[0, 0, 0, 198, 238, 0, 0, 0, 116, 255, 64, 0, 0],
            &[0, 0, 0, 198, 238, 0, 0, 0, 116, 255, 64, 0, 0],
            &[0, 0, 0, 198, 238, 0, 0, 0, 116, 255, 64, 0, 0],
            &[0, 0, 0, 198, 238, 0, 0, 0, 116, 255, 64, 0, 0],
            &[0, 0, 0, 198, 238, 0, 0, 0, 116, 255, 64, 0, 0],
            &[0, 0, 0, 198, 238, 0, 0, 0, 116, 255, 64, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 118, 255, 64, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 153, 255, 53, 0, 0],
            &[0, 0, 0, 13, 0, 0, 0, 33, 231, 240, 8, 0, 0],
            &[0, 0, 0, 51, 130, 156, 161, 128, 40, 0, 0, 0, 0],
        ],
        // letter: 'Ĵ' / 0x134
        'Ĵ' => &[
            &[0, 0, 0, 0, 0, 0, 55, 237, 233, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 226, 174, 195, 214, 16, 0, 0],
            &[0, 0, 0, 0, 13, 206, 158, 4, 11, 176, 196, 10, 0],
            &[0, 0, 0, 0, 6, 25, 0, 0, 0, 1, 26, 5, 0],
            &[0, 0, 0, 0, 0, 0, 0, 126, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 223, 213, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 225, 213, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 243, 199, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 73, 255, 154, 0, 0, 0, 0],
            &[0, 24, 200, 137, 109, 147, 241, 251, 50, 0, 0, 0, 0],
            &[0, 16, 208, 255, 255, 255, 223, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 40, 29, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĵ' / 0x135
        'ĵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 86, 141, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 249, 222, 237, 34, 0, 0, 0],
            &[0, 0, 0, 0, 33, 231, 138, 6, 171, 217, 22, 0, 0],
            &[0, 0, 0, 0, 84, 85, 0, 0, 1, 99, 74, 0, 0],
            &[0, 0, 0, 23, 47, 47, 47, 47, 25, 0, 0, 0, 0],
            &[0, 0, 0, 125, 255, 255, 255, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 1, 17, 36, 87, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 42, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 44, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 255, 117, 0, 0, 0, 0],
            &[0, 0, 13, 0, 0, 6, 187, 255, 63, 0, 0, 0, 0],
            &[0, 0, 90, 144, 161, 136, 71, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ķ' / 0x136
        'Ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 1, 127, 141, 13, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 121, 255, 125, 0, 0, 0],
            &[0, 171, 255, 10, 0, 0, 60, 251, 189, 2, 0, 0, 0],
            &[0, 171, 255, 10, 0, 20, 227, 233, 25, 0, 0, 0, 0],
            &[0, 171, 255, 10, 1, 181, 254, 70, 0, 0, 0, 0, 0],
            &[0, 171, 255, 10, 117, 255, 135, 0, 0, 0, 0, 0, 0],
            &[0, 171, 255, 58, 249, 230, 4, 0, 0, 0, 0, 0, 0],
            &[0, 171, 255, 212, 241, 255, 83, 0, 0, 0, 0, 0, 0],
            &[0, 171, 255, 215, 30, 227, 230, 15, 0, 0, 0, 0, 0],
            &[0, 171, 255, 37, 0, 80, 255, 152, 0, 0, 0, 0, 0],
            &[0, 171, 255, 10, 0, 0, 177, 254, 60, 0, 0, 0, 0],
            &[0, 171, 255, 10, 0, 0, 29, 243, 214, 6, 0, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 113, 255, 127, 0, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 3, 206, 249, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 209, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 255, 66, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 14, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ķ' / 0x137
        'ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 112, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 0, 26, 47, 22, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 43, 238, 217, 21, 0, 0],
            &[0, 1, 255, 180, 0, 0, 29, 228, 227, 30, 0, 0, 0],
            &[0, 1, 255, 179, 0, 18, 214, 236, 42, 0, 0, 0, 0],
            &[0, 1, 255, 173, 9, 198, 243, 54, 0, 0, 0, 0, 0],
            &[0, 1, 255, 162, 171, 255, 195, 2, 0, 0, 0, 0, 0],
            &[0, 1, 255, 238, 232, 139, 255, 123, 0, 0, 0, 0, 0],
            &[0, 1, 255, 232, 38, 0, 175, 252, 61, 0, 0, 0, 0],
            &[0, 1, 255, 180, 0, 0, 18, 227, 228, 20, 0, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 63, 253, 181, 1, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 0, 130, 255, 117, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 215, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 164, 206, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 42, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĸ' / 0x138
        'ĸ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 47, 33, 0, 0, 0, 0, 26, 47, 22, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 43, 238, 217, 21, 0, 0],
            &[0, 1, 255, 180, 0, 0, 29, 228, 227, 30, 0, 0, 0],
            &[0, 1, 255, 179, 0, 18, 214, 236, 42, 0, 0, 0, 0],
            &[0, 1, 255, 173, 9, 198, 243, 54, 0, 0, 0, 0, 0],
            &[0, 1, 255, 162, 172, 255, 195, 2, 0, 0, 0, 0, 0],
            &[0, 1, 255, 238, 232, 139, 255, 123, 0, 0, 0, 0, 0],
            &[0, 1, 255, 232, 38, 0, 175, 252, 61, 0, 0, 0, 0],
            &[0, 1, 255, 180, 0, 0, 18, 227, 228, 20, 0, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 63, 253, 181, 1, 0, 0],
            &[0, 1, 255, 180, 0, 0, 0, 0, 130, 255, 117, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĺ' / 0x139
        'Ĺ' => &[
            &[0, 0, 0, 44, 237, 200, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 194, 232, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 227, 43, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 11, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 244, 147, 147, 147, 147, 147, 115, 0, 0, 0],
            &[0, 0, 208, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĺ' / 0x13a
        'ĺ' => &[
            &[0, 0, 0, 0, 0, 26, 240, 197, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 182, 197, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 58, 8, 0, 0, 0, 0, 0, 0],
            &[0, 41, 112, 112, 112, 112, 107, 0, 0, 0, 0, 0, 0],
            &[0, 70, 204, 221, 239, 254, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 74, 94, 219, 249, 96, 74, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 104, 207, 114, 0, 0, 0, 0],
        ],
        // letter: 'Ļ' / 0x13b
        'Ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 244, 147, 147, 147, 147, 147, 115, 0, 0, 0],
            &[0, 0, 208, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 209, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 207, 163, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 34, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ļ' / 0x13c
        'ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 112, 112, 112, 112, 107, 0, 0, 0, 0, 0, 0],
            &[0, 70, 204, 221, 239, 254, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 74, 94, 219, 249, 96, 74, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 162, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 241, 124, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 26, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ľ' / 0x13d
        'Ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 129, 0, 0, 0, 17, 144, 75, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 66, 255, 62, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 125, 197, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 83, 42, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 244, 147, 147, 147, 147, 147, 115, 0, 0, 0],
            &[0, 0, 208, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ľ' / 0x13e
        'ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 112, 112, 112, 112, 107, 0, 0, 110, 74, 0, 0],
            &[0, 70, 204, 221, 239, 254, 244, 0, 29, 255, 106, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 86, 234, 11, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 84, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 74, 94, 219, 249, 96, 74, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 118, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 7, 69, 17, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 123, 255, 178, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 108, 255, 163, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 32, 2, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 229, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 208, 244, 147, 147, 147, 147, 147, 115, 0, 0, 0],
            &[0, 0, 208, 255, 255, 255, 255, 255, 255, 200, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŀ' / 0x140
        'ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[41, 112, 112, 112, 112, 107, 0, 0, 0, 0, 0, 0, 0],
            &[70, 204, 221, 239, 254, 244, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 7, 0, 0],
            &[0, 0, 0, 0, 193, 244, 0, 0, 0, 143, 251, 95, 0],
            &[0, 0, 0, 0, 193, 244, 0, 0, 0, 183, 255, 132, 0],
            &[0, 0, 0, 0, 193, 244, 0, 0, 0, 31, 99, 16, 0],
            &[0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0, 0],
            &[19, 55, 74, 94, 219, 249, 96, 74, 52, 10, 0, 0, 0],
            &[122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ł' / 0x141
        'Ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 144, 94, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 255, 167, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 255, 167, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 255, 167, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 255, 167, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 255, 167, 3, 107, 23, 0, 0, 0, 0, 0],
            &[0, 0, 14, 255, 230, 209, 252, 90, 0, 0, 0, 0, 0],
            &[0, 0, 41, 255, 255, 189, 45, 0, 0, 0, 0, 0, 0],
            &[0, 115, 255, 255, 250, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 136, 137, 255, 167, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 255, 167, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 255, 167, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 255, 218, 147, 147, 147, 147, 147, 80, 0, 0],
            &[0, 0, 14, 255, 255, 255, 255, 255, 255, 255, 139, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ł' / 0x142
        'ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 112, 112, 112, 112, 107, 0, 0, 0, 0, 0, 0],
            &[0, 70, 204, 221, 239, 254, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 147, 240, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 255, 255, 232, 91, 0, 0, 0, 0],
            &[0, 0, 0, 34, 183, 255, 255, 16, 0, 0, 0, 0, 0],
            &[0, 0, 20, 243, 208, 249, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 66, 3, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 244, 0, 0, 0, 0, 0, 0],
            &[0, 19, 55, 74, 94, 219, 249, 96, 74, 52, 10, 0, 0],
            &[0, 122, 255, 255, 255, 255, 255, 255, 255, 255, 66, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ń' / 0x143
        'Ń' => &[
            &[0, 0, 0, 0, 0, 0, 106, 239, 143, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 24, 238, 193, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 180, 189, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 4, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 103, 0, 0, 0, 0, 84, 144, 10, 0, 0],
            &[0, 171, 255, 248, 22, 0, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 217, 247, 126, 0, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 215, 155, 231, 6, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 234, 44, 255, 94, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 246, 0, 187, 205, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 74, 255, 61, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 1, 216, 173, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 106, 252, 32, 146, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 11, 238, 141, 140, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 137, 240, 141, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 29, 251, 220, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 0, 168, 255, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 0, 56, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ń' / 0x144
        'ń' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 28, 141, 123, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 168, 251, 72, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 79, 251, 88, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 65, 0, 0, 0, 0, 0],
            &[0, 0, 22, 42, 0, 3, 60, 86, 57, 3, 0, 0, 0],
            &[0, 0, 122, 250, 45, 220, 255, 255, 255, 208, 25, 0, 0],
            &[0, 0, 122, 255, 233, 123, 30, 20, 126, 255, 163, 0, 0],
            &[0, 0, 122, 255, 170, 0, 0, 0, 0, 224, 223, 0, 0],
            &[0, 0, 122, 255, 92, 0, 0, 0, 0, 191, 248, 0, 0],
            &[0, 0, 122, 255, 66, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ņ' / 0x145
        'Ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 103, 0, 0, 0, 0, 84, 144, 10, 0, 0],
            &[0, 171, 255, 248, 22, 0, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 217, 247, 126, 0, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 215, 155, 231, 6, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 234, 44, 255, 94, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 246, 0, 187, 205, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 74, 255, 61, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 1, 216, 173, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 106, 252, 32, 146, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 11, 238, 141, 140, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 137, 240, 141, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 29, 251, 220, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 0, 168, 255, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 0, 56, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 215, 96, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 97, 247, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 5, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ņ' / 0x146
        'ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 42, 0, 3, 60, 86, 57, 3, 0, 0, 0],
            &[0, 0, 122, 250, 45, 220, 255, 255, 255, 208, 25, 0, 0],
            &[0, 0, 122, 255, 233, 123, 30, 20, 126, 255, 163, 0, 0],
            &[0, 0, 122, 255, 170, 0, 0, 0, 0, 224, 223, 0, 0],
            &[0, 0, 122, 255, 92, 0, 0, 0, 0, 191, 248, 0, 0],
            &[0, 0, 122, 255, 66, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 215, 120, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 254, 47, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 11, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ň' / 0x147
        'Ň' => &[
            &[0, 0, 50, 225, 88, 0, 15, 172, 181, 0, 0, 0, 0],
            &[0, 0, 0, 85, 253, 125, 208, 209, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 130, 255, 239, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 26, 17, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 103, 0, 0, 0, 0, 84, 144, 10, 0, 0],
            &[0, 171, 255, 248, 22, 0, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 217, 247, 126, 0, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 215, 155, 231, 6, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 234, 44, 255, 94, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 246, 0, 187, 205, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 74, 255, 61, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 1, 216, 173, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 106, 252, 32, 146, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 11, 238, 141, 140, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 137, 240, 141, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 29, 251, 220, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 0, 168, 255, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 0, 56, 255, 255, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ň' / 0x148
        'ň' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 139, 33, 0, 0, 57, 138, 11, 0, 0],
            &[0, 0, 0, 0, 137, 238, 58, 96, 249, 94, 0, 0, 0],
            &[0, 0, 0, 0, 1, 169, 246, 254, 123, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 123, 114, 1, 0, 0, 0, 0],
            &[0, 0, 22, 42, 0, 3, 60, 86, 57, 3, 0, 0, 0],
            &[0, 0, 122, 250, 45, 220, 255, 255, 255, 208, 25, 0, 0],
            &[0, 0, 122, 255, 233, 123, 30, 20, 126, 255, 163, 0, 0],
            &[0, 0, 122, 255, 170, 0, 0, 0, 0, 224, 223, 0, 0],
            &[0, 0, 122, 255, 92, 0, 0, 0, 0, 191, 248, 0, 0],
            &[0, 0, 122, 255, 66, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŉ' / 0x149
        'ŉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 144, 143, 3, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 51, 255, 207, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 100, 255, 122, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 153, 252, 38, 47, 8, 0, 50, 85, 60, 5, 0, 0],
            &[0, 209, 176, 49, 255, 81, 187, 255, 255, 255, 213, 29, 0],
            &[0, 21, 10, 49, 255, 235, 168, 37, 19, 130, 255, 165, 0],
            &[0, 0, 0, 49, 255, 234, 5, 0, 0, 1, 225, 224, 0],
            &[0, 0, 0, 49, 255, 164, 0, 0, 0, 0, 192, 248, 0],
            &[0, 0, 0, 49, 255, 139, 0, 0, 0, 0, 188, 249, 0],
            &[0, 0, 0, 49, 255, 131, 0, 0, 0, 0, 188, 249, 0],
            &[0, 0, 0, 49, 255, 131, 0, 0, 0, 0, 188, 249, 0],
            &[0, 0, 0, 49, 255, 131, 0, 0, 0, 0, 188, 249, 0],
            &[0, 0, 0, 49, 255, 131, 0, 0, 0, 0, 188, 249, 0],
            &[0, 0, 0, 49, 255, 131, 0, 0, 0, 0, 188, 249, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 103, 0, 0, 0, 0, 84, 144, 10, 0, 0],
            &[0, 171, 255, 248, 22, 0, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 217, 247, 126, 0, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 215, 155, 231, 6, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 234, 44, 255, 94, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 246, 0, 187, 205, 0, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 74, 255, 61, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 1, 216, 173, 0, 149, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 106, 252, 32, 146, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 11, 238, 141, 140, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 137, 240, 141, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 29, 251, 220, 255, 17, 0, 0],
            &[0, 171, 251, 0, 0, 0, 0, 168, 255, 255, 27, 0, 0],
            &[0, 171, 251, 0, 0, 0, 0, 56, 255, 255, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 154, 254, 7, 0, 0],
            &[0, 0, 0, 0, 0, 4, 0, 55, 238, 207, 0, 0, 0],
            &[0, 0, 0, 0, 0, 235, 238, 255, 243, 68, 0, 0, 0],
            &[0, 0, 0, 0, 0, 118, 148, 116, 27, 0, 0, 0, 0],
        ],
        // letter: 'ŋ' / 0x14b
        'ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 42, 0, 3, 60, 86, 57, 3, 0, 0, 0],
            &[0, 0, 122, 250, 45, 220, 255, 255, 255, 208, 25, 0, 0],
            &[0, 0, 122, 255, 233, 123, 30, 20, 126, 255, 163, 0, 0],
            &[0, 0, 122, 255, 170, 0, 0, 0, 0, 224, 223, 0, 0],
            &[0, 0, 122, 255, 92, 0, 0, 0, 0, 191, 248, 0, 0],
            &[0, 0, 122, 255, 66, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 255, 255, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 255, 255, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 255, 255, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 255, 255, 0, 0],
            &[0, 0, 122, 255, 58, 0, 0, 0, 0, 255, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 188, 249, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 190, 247, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 14, 230, 218, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 122, 134, 75, 0, 0, 0],
        ],
        // letter: 'Ō' / 0x14c
        'Ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 121, 121, 121, 121, 121, 61, 0, 0, 0],
            &[0, 0, 0, 29, 224, 224, 224, 224, 224, 114, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 157, 192, 181, 112, 8, 0, 0, 0],
            &[0, 0, 0, 108, 255, 243, 197, 216, 255, 205, 12, 0, 0],
            &[0, 0, 41, 253, 203, 21, 0, 0, 101, 255, 146, 0, 0],
            &[0, 0, 152, 255, 50, 0, 0, 0, 0, 203, 246, 10, 0],
            &[0, 0, 217, 223, 0, 0, 0, 0, 0, 126, 255, 71, 0],
            &[0, 8, 255, 180, 0, 0, 0, 0, 0, 84, 255, 112, 0],
            &[0, 25, 255, 162, 0, 0, 0, 0, 0, 65, 255, 131, 0],
            &[0, 33, 255, 158, 0, 0, 0, 0, 0, 56, 255, 135, 0],
            &[0, 20, 255, 170, 0, 0, 0, 0, 0, 68, 255, 122, 0],
            &[0, 0, 243, 204, 0, 0, 0, 0, 0, 100, 255, 88, 0],
            &[0, 0, 188, 249, 17, 0, 0, 0, 0, 161, 255, 31, 0],
            &[0, 0, 97, 255, 134, 0, 0, 0, 36, 245, 195, 0, 0],
            &[0, 0, 4, 199, 254, 162, 96, 120, 231, 248, 53, 0, 0],
            &[0, 0, 0, 17, 164, 250, 255, 255, 210, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ō' / 0x14d
        'ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 23, 23, 23, 23, 23, 11, 0, 0, 0, 0],
            &[0, 0, 33, 255, 255, 255, 255, 255, 131, 0, 0, 0, 0],
            &[0, 0, 8, 66, 66, 66, 66, 66, 33, 0, 0, 0, 0],
            &[0, 0, 0, 1, 55, 87, 70, 13, 0, 0, 0, 0, 0],
            &[0, 0, 53, 218, 255, 255, 255, 244, 116, 0, 0, 0, 0],
            &[0, 28, 238, 231, 76, 16, 37, 170, 255, 105, 0, 0, 0],
            &[0, 139, 255, 68, 0, 0, 0, 3, 217, 233, 3, 0, 0],
            &[0, 209, 231, 0, 0, 0, 0, 0, 134, 255, 54, 0, 0],
            &[0, 235, 199, 0, 0, 0, 0, 0, 103, 255, 80, 0, 0],
            &[0, 233, 199, 0, 0, 0, 0, 0, 106, 255, 80, 0, 0],
            &[0, 198, 239, 2, 0, 0, 0, 0, 148, 255, 47, 0, 0],
            &[0, 119, 255, 93, 0, 0, 0, 16, 235, 222, 2, 0, 0],
            &[0, 12, 216, 246, 119, 60, 84, 207, 254, 82, 0, 0, 0],
            &[0, 0, 23, 172, 251, 255, 255, 220, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 42, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => &[
            &[0, 0, 0, 24, 106, 0, 0, 0, 63, 67, 0, 0, 0],
            &[0, 0, 0, 16, 243, 127, 35, 74, 228, 101, 0, 0, 0],
            &[0, 0, 0, 0, 76, 228, 255, 247, 144, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 16, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 157, 192, 181, 112, 8, 0, 0, 0],
            &[0, 0, 0, 108, 255, 243, 197, 216, 255, 205, 12, 0, 0],
            &[0, 0, 41, 253, 203, 21, 0, 0, 101, 255, 146, 0, 0],
            &[0, 0, 152, 255, 50, 0, 0, 0, 0, 203, 246, 10, 0],
            &[0, 0, 217, 223, 0, 0, 0, 0, 0, 126, 255, 71, 0],
            &[0, 8, 255, 180, 0, 0, 0, 0, 0, 84, 255, 112, 0],
            &[0, 25, 255, 162, 0, 0, 0, 0, 0, 65, 255, 131, 0],
            &[0, 33, 255, 158, 0, 0, 0, 0, 0, 56, 255, 135, 0],
            &[0, 20, 255, 170, 0, 0, 0, 0, 0, 68, 255, 122, 0],
            &[0, 0, 243, 204, 0, 0, 0, 0, 0, 100, 255, 88, 0],
            &[0, 0, 188, 249, 17, 0, 0, 0, 0, 161, 255, 31, 0],
            &[0, 0, 97, 255, 134, 0, 0, 0, 36, 245, 195, 0, 0],
            &[0, 0, 4, 199, 254, 162, 96, 120, 231, 248, 53, 0, 0],
            &[0, 0, 0, 17, 164, 250, 255, 255, 210, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŏ' / 0x14f
        'ŏ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 13, 0, 0, 0, 7, 9, 0, 0, 0, 0],
            &[0, 0, 36, 251, 41, 0, 2, 186, 134, 0, 0, 0, 0],
            &[0, 0, 0, 162, 245, 193, 229, 228, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 114, 91, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 55, 87, 70, 13, 0, 0, 0, 0, 0],
            &[0, 0, 53, 218, 255, 255, 255, 244, 116, 0, 0, 0, 0],
            &[0, 28, 238, 231, 76, 16, 37, 170, 255, 105, 0, 0, 0],
            &[0, 139, 255, 68, 0, 0, 0, 3, 217, 233, 3, 0, 0],
            &[0, 209, 231, 0, 0, 0, 0, 0, 134, 255, 54, 0, 0],
            &[0, 235, 199, 0, 0, 0, 0, 0, 103, 255, 80, 0, 0],
            &[0, 233, 199, 0, 0, 0, 0, 0, 106, 255, 80, 0, 0],
            &[0, 198, 239, 2, 0, 0, 0, 0, 148, 255, 47, 0, 0],
            &[0, 119, 255, 93, 0, 0, 0, 16, 235, 222, 2, 0, 0],
            &[0, 12, 216, 246, 119, 60, 84, 207, 254, 82, 0, 0, 0],
            &[0, 0, 23, 172, 251, 255, 255, 220, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 42, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ő' / 0x150
        'Ő' => &[
            &[0, 0, 0, 0, 7, 215, 211, 10, 139, 238, 64, 0, 0],
            &[0, 0, 0, 0, 119, 246, 56, 37, 249, 135, 0, 0, 0],
            &[0, 0, 0, 32, 235, 74, 0, 189, 151, 1, 0, 0, 0],
            &[0, 0, 0, 9, 17, 0, 0, 26, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 157, 192, 181, 112, 8, 0, 0, 0],
            &[0, 0, 0, 108, 255, 243, 197, 216, 255, 205, 12, 0, 0],
            &[0, 0, 41, 253, 203, 21, 0, 0, 101, 255, 146, 0, 0],
            &[0, 0, 152, 255, 50, 0, 0, 0, 0, 203, 246, 10, 0],
            &[0, 0, 217, 223, 0, 0, 0, 0, 0, 126, 255, 71, 0],
            &[0, 8, 255, 180, 0, 0, 0, 0, 0, 84, 255, 112, 0],
            &[0, 25, 255, 162, 0, 0, 0, 0, 0, 65, 255, 131, 0],
            &[0, 33, 255, 158, 0, 0, 0, 0, 0, 56, 255, 135, 0],
            &[0, 20, 255, 170, 0, 0, 0, 0, 0, 68, 255, 122, 0],
            &[0, 0, 243, 204, 0, 0, 0, 0, 0, 100, 255, 88, 0],
            &[0, 0, 188, 249, 17, 0, 0, 0, 0, 161, 255, 31, 0],
            &[0, 0, 97, 255, 134, 0, 0, 0, 36, 245, 195, 0, 0],
            &[0, 0, 4, 199, 254, 162, 96, 120, 231, 248, 53, 0, 0],
            &[0, 0, 0, 17, 164, 250, 255, 255, 210, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 40, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ő' / 0x151
        'ő' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 118, 139, 10, 68, 141, 56, 0, 0, 0],
            &[0, 0, 0, 64, 255, 127, 7, 222, 207, 8, 0, 0, 0],
            &[0, 0, 5, 209, 160, 1, 127, 223, 26, 0, 0, 0, 0],
            &[0, 0, 36, 106, 2, 0, 117, 27, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 55, 87, 70, 13, 0, 0, 0, 0, 0],
            &[0, 0, 53, 218, 255, 255, 255, 244, 116, 0, 0, 0, 0],
            &[0, 28, 238, 231, 76, 16, 37, 170, 255, 105, 0, 0, 0],
            &[0, 139, 255, 68, 0, 0, 0, 3, 217, 233, 3, 0, 0],
            &[0, 209, 231, 0, 0, 0, 0, 0, 134, 255, 54, 0, 0],
            &[0, 235, 199, 0, 0, 0, 0, 0, 103, 255, 80, 0, 0],
            &[0, 233, 199, 0, 0, 0, 0, 0, 106, 255, 80, 0, 0],
            &[0, 198, 239, 2, 0, 0, 0, 0, 148, 255, 47, 0, 0],
            &[0, 119, 255, 93, 0, 0, 0, 16, 235, 222, 2, 0, 0],
            &[0, 12, 216, 246, 119, 60, 84, 207, 254, 82, 0, 0, 0],
            &[0, 0, 23, 172, 251, 255, 255, 220, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 42, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Œ' / 0x152
        'Œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 126, 183, 177, 146, 144, 144, 144, 120, 0],
            &[0, 0, 7, 204, 255, 214, 229, 255, 250, 244, 244, 202, 0],
            &[0, 0, 114, 255, 117, 0, 28, 255, 138, 0, 0, 0, 0],
            &[0, 0, 206, 240, 3, 0, 28, 255, 138, 0, 0, 0, 0],
            &[0, 9, 253, 185, 0, 0, 28, 255, 138, 0, 0, 0, 0],
            &[0, 39, 255, 153, 0, 0, 28, 255, 138, 0, 0, 0, 0],
            &[0, 54, 255, 139, 0, 0, 28, 255, 238, 216, 216, 101, 0],
            &[0, 57, 255, 133, 0, 0, 28, 255, 215, 167, 167, 78, 0],
            &[0, 44, 255, 142, 0, 0, 28, 255, 138, 0, 0, 0, 0],
            &[0, 21, 255, 166, 0, 0, 28, 255, 138, 0, 0, 0, 0],
            &[0, 0, 233, 211, 0, 0, 28, 255, 138, 0, 0, 0, 0],
            &[0, 0, 159, 254, 42, 0, 28, 255, 138, 0, 0, 0, 0],
            &[0, 0, 49, 251, 215, 99, 136, 255, 199, 133, 133, 110, 0],
            &[0, 0, 0, 93, 242, 255, 255, 255, 255, 255, 255, 212, 0],
            &[0, 0, 0, 0, 5, 39, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'œ' / 0x153
        'œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 40, 92, 50, 0, 0, 43, 82, 21, 0, 0, 0],
            &[0, 100, 253, 253, 255, 108, 101, 253, 249, 235, 33, 0, 0],
            &[17, 244, 173, 11, 107, 243, 241, 101, 20, 225, 155, 0, 0],
            &[88, 255, 53, 0, 4, 243, 243, 5, 0, 142, 222, 0, 0],
            &[129, 255, 6, 0, 0, 215, 213, 5, 5, 117, 251, 0, 0],
            &[145, 246, 0, 0, 0, 196, 255, 255, 255, 255, 255, 4, 0],
            &[138, 249, 0, 0, 0, 199, 211, 49, 49, 49, 49, 0, 0],
            &[116, 255, 13, 0, 0, 217, 220, 0, 0, 0, 0, 0, 0],
            &[63, 255, 71, 0, 8, 247, 253, 21, 0, 0, 0, 0, 0],
            &[2, 218, 207, 57, 143, 231, 221, 171, 38, 69, 102, 0, 0],
            &[0, 51, 227, 255, 242, 74, 59, 235, 255, 249, 113, 0, 0],
            &[0, 0, 5, 37, 12, 0, 0, 8, 39, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŕ' / 0x154
        'Ŕ' => &[
            &[0, 0, 0, 0, 0, 0, 129, 239, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 248, 174, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 201, 172, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 26, 2, 0, 0, 0, 0, 0, 0],
            &[0, 28, 144, 144, 144, 135, 108, 40, 0, 0, 0, 0, 0],
            &[0, 49, 255, 245, 234, 249, 255, 254, 149, 1, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 41, 193, 255, 94, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 45, 255, 169, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 12, 255, 183, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 58, 255, 144, 0, 0, 0],
            &[0, 49, 255, 135, 7, 18, 71, 217, 244, 37, 0, 0, 0],
            &[0, 49, 255, 255, 255, 255, 255, 199, 51, 0, 0, 0, 0],
            &[0, 49, 255, 187, 115, 143, 255, 129, 0, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 189, 248, 35, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 43, 251, 181, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 143, 255, 79, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 15, 233, 223, 9, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 98, 255, 130, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŕ' / 0x155
        'ŕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 95, 141, 57, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 38, 249, 196, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 195, 204, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 119, 12, 0, 0, 0, 0, 0],
            &[0, 24, 47, 47, 47, 30, 0, 23, 83, 77, 22, 0, 0],
            &[0, 132, 255, 255, 255, 185, 62, 244, 255, 255, 185, 0, 0],
            &[0, 2, 21, 44, 201, 226, 227, 150, 46, 60, 60, 0, 0],
            &[0, 0, 0, 0, 185, 255, 168, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 1, 0, 0, 0, 0, 0, 0],
            &[0, 37, 66, 90, 214, 255, 97, 71, 46, 0, 0, 0, 0],
            &[0, 210, 255, 255, 255, 255, 255, 255, 247, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŗ' / 0x156
        'Ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 144, 144, 144, 135, 108, 40, 0, 0, 0, 0, 0],
            &[0, 49, 255, 245, 234, 249, 255, 254, 149, 1, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 41, 193, 255, 94, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 45, 255, 169, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 12, 255, 183, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 58, 255, 144, 0, 0, 0],
            &[0, 49, 255, 135, 7, 18, 71, 217, 244, 37, 0, 0, 0],
            &[0, 49, 255, 255, 255, 255, 255, 199, 51, 0, 0, 0, 0],
            &[0, 49, 255, 187, 115, 143, 255, 129, 0, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 189, 248, 35, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 43, 251, 181, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 143, 255, 79, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 15, 233, 223, 9, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 98, 255, 130, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 209, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 207, 163, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 34, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŗ' / 0x157
        'ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 24, 47, 47, 47, 30, 0, 23, 83, 77, 22, 0, 0],
            &[0, 132, 255, 255, 255, 185, 62, 244, 255, 255, 185, 0, 0],
            &[0, 2, 21, 44, 201, 226, 227, 150, 46, 60, 60, 0, 0],
            &[0, 0, 0, 0, 185, 255, 168, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 1, 0, 0, 0, 0, 0, 0],
            &[0, 37, 66, 90, 214, 255, 97, 71, 46, 0, 0, 0, 0],
            &[0, 210, 255, 255, 255, 255, 255, 255, 247, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 156, 192, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 236, 132, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 28, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ř' / 0x158
        'Ř' => &[
            &[0, 0, 68, 223, 71, 0, 23, 187, 159, 0, 0, 0, 0],
            &[0, 0, 0, 107, 251, 115, 221, 193, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 255, 227, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 26, 15, 0, 0, 0, 0, 0, 0],
            &[0, 28, 144, 144, 144, 135, 108, 40, 0, 0, 0, 0, 0],
            &[0, 49, 255, 245, 234, 249, 255, 254, 149, 1, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 41, 193, 255, 94, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 45, 255, 169, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 12, 255, 183, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 58, 255, 144, 0, 0, 0],
            &[0, 49, 255, 135, 7, 18, 71, 217, 244, 37, 0, 0, 0],
            &[0, 49, 255, 255, 255, 255, 255, 199, 51, 0, 0, 0, 0],
            &[0, 49, 255, 187, 115, 143, 255, 129, 0, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 189, 248, 35, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 43, 251, 181, 0, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 143, 255, 79, 0, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 15, 233, 223, 9, 0, 0],
            &[0, 49, 255, 131, 0, 0, 0, 0, 98, 255, 130, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ř' / 0x159
        'ř' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 95, 105, 1, 0, 5, 118, 83, 0, 0, 0],
            &[0, 0, 0, 29, 227, 171, 22, 194, 211, 18, 0, 0, 0],
            &[0, 0, 0, 0, 45, 245, 245, 231, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 124, 57, 0, 0, 0, 0, 0],
            &[0, 24, 47, 47, 47, 30, 0, 23, 83, 77, 22, 0, 0],
            &[0, 132, 255, 255, 255, 185, 62, 244, 255, 255, 185, 0, 0],
            &[0, 2, 21, 44, 201, 226, 227, 150, 46, 60, 60, 0, 0],
            &[0, 0, 0, 0, 185, 255, 168, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 1, 0, 0, 0, 0, 0, 0],
            &[0, 37, 66, 90, 214, 255, 97, 71, 46, 0, 0, 0, 0],
            &[0, 210, 255, 255, 255, 255, 255, 255, 247, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ś' / 0x15a
        'Ś' => &[
            &[0, 0, 0, 0, 0, 0, 44, 237, 200, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 194, 232, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 227, 43, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 19, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 129, 177, 184, 157, 109, 25, 0, 0, 0],
            &[0, 0, 62, 245, 254, 224, 205, 234, 255, 151, 0, 0, 0],
            &[0, 0, 210, 247, 70, 0, 0, 0, 46, 32, 0, 0, 0],
            &[0, 9, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 252, 210, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 177, 255, 145, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 200, 255, 237, 129, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 89, 215, 255, 243, 124, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 80, 218, 255, 133, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 234, 240, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 188, 255, 6, 0, 0],
            &[0, 3, 0, 0, 0, 0, 0, 14, 236, 228, 0, 0, 0],
            &[0, 98, 204, 128, 85, 74, 104, 210, 255, 112, 0, 0, 0],
            &[0, 53, 194, 252, 255, 255, 255, 224, 105, 0, 0, 0, 0],
            &[0, 0, 0, 5, 28, 43, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ś' / 0x15b
        'ś' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 47, 141, 104, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 199, 242, 47, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 113, 243, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 48, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 49, 85, 85, 58, 5, 0, 0, 0],
            &[0, 0, 0, 33, 210, 255, 255, 255, 255, 240, 23, 0, 0],
            &[0, 0, 0, 170, 250, 89, 24, 22, 58, 109, 0, 0, 0],
            &[0, 0, 0, 204, 232, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 128, 255, 174, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 116, 241, 253, 176, 60, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 123, 232, 255, 151, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 173, 255, 73, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 94, 255, 92, 0, 0],
            &[0, 0, 1, 214, 123, 71, 60, 92, 220, 249, 31, 0, 0],
            &[0, 0, 0, 180, 248, 255, 255, 255, 214, 71, 0, 0, 0],
            &[0, 0, 0, 0, 5, 29, 43, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => &[
            &[0, 0, 0, 0, 40, 233, 237, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 214, 189, 180, 225, 25, 0, 0, 0, 0],
            &[0, 0, 7, 192, 173, 9, 6, 161, 210, 17, 0, 0, 0],
            &[0, 0, 3, 26, 1, 0, 0, 0, 25, 7, 0, 0, 0],
            &[0, 0, 0, 30, 129, 177, 184, 157, 109, 25, 0, 0, 0],
            &[0, 0, 62, 245, 254, 224, 205, 234, 255, 151, 0, 0, 0],
            &[0, 0, 210, 247, 70, 0, 0, 0, 46, 32, 0, 0, 0],
            &[0, 9, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 252, 210, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 177, 255, 145, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 200, 255, 237, 129, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 89, 215, 255, 243, 124, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 80, 218, 255, 133, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 234, 240, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 188, 255, 6, 0, 0],
            &[0, 3, 0, 0, 0, 0, 0, 14, 236, 228, 0, 0, 0],
            &[0, 98, 204, 128, 85, 74, 104, 210, 255, 112, 0, 0, 0],
            &[0, 53, 194, 252, 255, 255, 255, 224, 105, 0, 0, 0, 0],
            &[0, 0, 0, 5, 28, 43, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŝ' / 0x15d
        'ŝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 141, 116, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 207, 224, 254, 101, 0, 0, 0, 0],
            &[0, 0, 0, 3, 176, 205, 19, 96, 246, 73, 0, 0, 0],
            &[0, 0, 0, 42, 116, 10, 0, 0, 59, 115, 0, 0, 0],
            &[0, 0, 0, 0, 1, 49, 85, 85, 58, 5, 0, 0, 0],
            &[0, 0, 0, 33, 210, 255, 255, 255, 255, 240, 23, 0, 0],
            &[0, 0, 0, 170, 250, 89, 24, 22, 58, 109, 0, 0, 0],
            &[0, 0, 0, 204, 232, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 128, 255, 174, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 116, 241, 253, 176, 60, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 123, 232, 255, 151, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 173, 255, 73, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 94, 255, 92, 0, 0],
            &[0, 0, 1, 214, 123, 71, 60, 92, 220, 249, 31, 0, 0],
            &[0, 0, 0, 180, 248, 255, 255, 255, 214, 71, 0, 0, 0],
            &[0, 0, 0, 0, 5, 29, 43, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ş' / 0x15e
        'Ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 129, 177, 184, 157, 109, 25, 0, 0, 0],
            &[0, 0, 62, 245, 254, 224, 205, 234, 255, 151, 0, 0, 0],
            &[0, 0, 210, 247, 70, 0, 0, 0, 46, 32, 0, 0, 0],
            &[0, 9, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 252, 210, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 177, 255, 145, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 200, 255, 237, 129, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 89, 215, 255, 243, 124, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 80, 218, 255, 133, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 234, 240, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 188, 255, 6, 0, 0],
            &[0, 3, 0, 0, 0, 0, 0, 14, 236, 228, 0, 0, 0],
            &[0, 98, 204, 128, 85, 74, 104, 210, 255, 112, 0, 0, 0],
            &[0, 53, 194, 252, 255, 255, 255, 224, 105, 0, 0, 0, 0],
            &[0, 0, 0, 5, 51, 255, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 233, 177, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 247, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 138, 121, 45, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ş' / 0x15f
        'ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 49, 85, 85, 58, 5, 0, 0, 0],
            &[0, 0, 0, 33, 210, 255, 255, 255, 255, 240, 23, 0, 0],
            &[0, 0, 0, 170, 250, 89, 24, 22, 58, 109, 0, 0, 0],
            &[0, 0, 0, 204, 232, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 128, 255, 174, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 116, 241, 253, 176, 60, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 123, 232, 255, 151, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 173, 255, 73, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 94, 255, 92, 0, 0],
            &[0, 0, 1, 214, 123, 71, 60, 92, 220, 249, 31, 0, 0],
            &[0, 0, 0, 180, 248, 255, 255, 255, 214, 71, 0, 0, 0],
            &[0, 0, 0, 0, 5, 39, 255, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 64, 229, 188, 33, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 233, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 137, 124, 55, 0, 0, 0, 0, 0],
        ],
        // letter: 'Š' / 0x160
        'Š' => &[
            &[0, 0, 12, 206, 140, 3, 1, 124, 218, 26, 0, 0, 0],
            &[0, 0, 0, 36, 234, 172, 160, 242, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 64, 253, 255, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 129, 177, 184, 157, 109, 25, 0, 0, 0],
            &[0, 0, 62, 245, 254, 224, 205, 234, 255, 151, 0, 0, 0],
            &[0, 0, 210, 247, 70, 0, 0, 0, 46, 32, 0, 0, 0],
            &[0, 9, 255, 184, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 252, 210, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 177, 255, 145, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 200, 255, 237, 129, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 89, 215, 255, 243, 124, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 80, 218, 255, 133, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 234, 240, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 188, 255, 6, 0, 0],
            &[0, 3, 0, 0, 0, 0, 0, 14, 236, 228, 0, 0, 0],
            &[0, 98, 204, 128, 85, 74, 104, 210, 255, 112, 0, 0, 0],
            &[0, 53, 194, 252, 255, 255, 255, 224, 105, 0, 0, 0, 0],
            &[0, 0, 0, 5, 28, 43, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'š' / 0x161
        'š' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 134, 19, 0, 0, 76, 130, 0, 0, 0],
            &[0, 0, 0, 2, 168, 225, 38, 125, 247, 66, 0, 0, 0],
            &[0, 0, 0, 0, 7, 197, 245, 255, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 124, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 49, 85, 85, 58, 5, 0, 0, 0],
            &[0, 0, 0, 33, 210, 255, 255, 255, 255, 240, 23, 0, 0],
            &[0, 0, 0, 170, 250, 89, 24, 22, 58, 109, 0, 0, 0],
            &[0, 0, 0, 204, 232, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 128, 255, 174, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 116, 241, 253, 176, 60, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 123, 232, 255, 151, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 173, 255, 73, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 94, 255, 92, 0, 0],
            &[0, 0, 1, 214, 123, 71, 60, 92, 220, 249, 31, 0, 0],
            &[0, 0, 0, 180, 248, 255, 255, 255, 214, 71, 0, 0, 0],
            &[0, 0, 0, 0, 5, 29, 43, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ţ' / 0x162
        'Ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 144, 144, 144, 144, 144, 144, 144, 78, 0],
            &[0, 35, 244, 244, 244, 245, 255, 250, 244, 244, 244, 133, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 239, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 83, 233, 177, 21, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 247, 110, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 138, 121, 45, 0, 0, 0, 0, 0],
        ],
        // letter: 'ţ' / 0x163
        'ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 202, 126, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 251, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 24, 73, 255, 161, 47, 47, 47, 24, 0, 0],
            &[0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0],
            &[0, 0, 57, 61, 93, 255, 168, 61, 61, 61, 32, 0, 0],
            &[0, 0, 0, 0, 41, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 144, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 255, 134, 64, 67, 83, 0, 0],
            &[0, 0, 0, 0, 0, 41, 202, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 187, 162, 29, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 205, 220, 80, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 162, 214, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 116, 132, 86, 5, 0, 0, 0],
        ],
        // letter: 'Ť' / 0x164
        'Ť' => &[
            &[0, 0, 0, 68, 223, 71, 0, 23, 187, 159, 0, 0, 0],
            &[0, 0, 0, 0, 107, 251, 115, 221, 193, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 255, 227, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 26, 15, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 144, 144, 144, 144, 144, 144, 144, 78, 0],
            &[0, 35, 244, 244, 244, 245, 255, 250, 244, 244, 244, 133, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ť' / 0x165
        'ť' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 161, 110, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 32, 255, 92, 0, 0],
            &[0, 0, 0, 0, 0, 202, 126, 0, 92, 219, 4, 0, 0],
            &[0, 0, 0, 0, 4, 251, 140, 0, 54, 47, 0, 0, 0],
            &[0, 0, 1, 24, 73, 255, 161, 47, 47, 47, 24, 0, 0],
            &[0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0],
            &[0, 0, 57, 61, 93, 255, 168, 61, 61, 61, 32, 0, 0],
            &[0, 0, 0, 0, 41, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 144, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 255, 134, 64, 67, 83, 0, 0],
            &[0, 0, 0, 0, 0, 41, 202, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 20, 43, 29, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŧ' / 0x166
        'Ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 144, 144, 144, 144, 144, 144, 144, 144, 144, 78, 0],
            &[0, 35, 244, 244, 244, 245, 255, 250, 244, 244, 244, 133, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 143, 148, 188, 255, 255, 148, 148, 57, 0, 0],
            &[0, 0, 0, 204, 211, 250, 255, 255, 211, 211, 81, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŧ' / 0x167
        'ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 202, 126, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 251, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 24, 73, 255, 161, 47, 47, 47, 24, 0, 0],
            &[0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0],
            &[0, 0, 57, 61, 93, 255, 168, 61, 61, 61, 32, 0, 0],
            &[0, 0, 0, 0, 41, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 137, 146, 188, 255, 255, 146, 146, 146, 118, 0, 0],
            &[0, 0, 162, 174, 215, 255, 255, 174, 174, 174, 139, 0, 0],
            &[0, 0, 0, 0, 41, 255, 140, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 144, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 211, 255, 134, 64, 67, 83, 0, 0],
            &[0, 0, 0, 0, 0, 41, 202, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 20, 43, 29, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ũ' / 0x168
        'Ũ' => &[
            &[0, 0, 0, 27, 69, 13, 0, 0, 80, 7, 0, 0, 0],
            &[0, 0, 45, 243, 255, 235, 115, 112, 232, 1, 0, 0, 0],
            &[0, 0, 150, 146, 21, 124, 240, 246, 86, 0, 0, 0, 0],
            &[0, 0, 18, 7, 0, 0, 3, 5, 0, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 0, 92, 144, 10, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 164, 255, 16, 0, 0, 0, 0, 176, 255, 11, 0, 0],
            &[0, 142, 255, 47, 0, 0, 0, 0, 206, 243, 0, 0, 0],
            &[0, 82, 255, 138, 0, 0, 0, 36, 252, 181, 0, 0, 0],
            &[0, 4, 214, 254, 151, 93, 115, 225, 252, 60, 0, 0, 0],
            &[0, 0, 27, 183, 254, 255, 255, 219, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 40, 24, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ũ' / 0x169
        'ũ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 173, 227, 150, 22, 24, 233, 8, 0, 0],
            &[0, 0, 0, 120, 207, 118, 216, 250, 246, 160, 0, 0, 0],
            &[0, 0, 0, 83, 44, 0, 5, 86, 93, 5, 0, 0, 0],
            &[0, 0, 27, 47, 6, 0, 0, 0, 0, 39, 41, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 219, 225, 0, 0],
            &[0, 0, 144, 255, 41, 0, 0, 0, 3, 244, 225, 0, 0],
            &[0, 0, 115, 255, 86, 0, 0, 0, 75, 255, 225, 0, 0],
            &[0, 0, 44, 252, 226, 91, 58, 104, 231, 231, 225, 0, 0],
            &[0, 0, 0, 98, 239, 255, 255, 230, 87, 147, 225, 0, 0],
            &[0, 0, 0, 0, 2, 31, 32, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ū' / 0x16a
        'Ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 121, 121, 121, 121, 121, 61, 0, 0, 0, 0],
            &[0, 0, 29, 224, 224, 224, 224, 224, 114, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 0, 92, 144, 10, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 164, 255, 16, 0, 0, 0, 0, 176, 255, 11, 0, 0],
            &[0, 142, 255, 47, 0, 0, 0, 0, 206, 243, 0, 0, 0],
            &[0, 82, 255, 138, 0, 0, 0, 36, 252, 181, 0, 0, 0],
            &[0, 4, 214, 254, 151, 93, 115, 225, 252, 60, 0, 0, 0],
            &[0, 0, 27, 183, 254, 255, 255, 219, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 40, 24, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ū' / 0x16b
        'ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 23, 23, 23, 23, 23, 11, 0, 0, 0],
            &[0, 0, 0, 33, 255, 255, 255, 255, 255, 131, 0, 0, 0],
            &[0, 0, 0, 8, 66, 66, 66, 66, 66, 33, 0, 0, 0],
            &[0, 0, 27, 47, 6, 0, 0, 0, 0, 39, 41, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 219, 225, 0, 0],
            &[0, 0, 144, 255, 41, 0, 0, 0, 3, 244, 225, 0, 0],
            &[0, 0, 115, 255, 86, 0, 0, 0, 75, 255, 225, 0, 0],
            &[0, 0, 44, 252, 226, 91, 58, 104, 231, 231, 225, 0, 0],
            &[0, 0, 0, 98, 239, 255, 255, 230, 87, 147, 225, 0, 0],
            &[0, 0, 0, 0, 2, 31, 32, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => &[
            &[0, 0, 24, 106, 0, 0, 0, 63, 67, 0, 0, 0, 0],
            &[0, 0, 16, 243, 127, 35, 74, 228, 101, 0, 0, 0, 0],
            &[0, 0, 0, 76, 228, 255, 247, 144, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 2, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 0, 92, 144, 10, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 164, 255, 16, 0, 0, 0, 0, 176, 255, 11, 0, 0],
            &[0, 142, 255, 47, 0, 0, 0, 0, 206, 243, 0, 0, 0],
            &[0, 82, 255, 138, 0, 0, 0, 36, 252, 181, 0, 0, 0],
            &[0, 4, 214, 254, 151, 93, 115, 225, 252, 60, 0, 0, 0],
            &[0, 0, 27, 183, 254, 255, 255, 219, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 40, 24, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŭ' / 0x16d
        'ŭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 13, 0, 0, 0, 7, 9, 0, 0, 0],
            &[0, 0, 0, 36, 251, 41, 0, 2, 186, 134, 0, 0, 0],
            &[0, 0, 0, 0, 162, 245, 193, 229, 228, 27, 0, 0, 0],
            &[0, 0, 0, 0, 0, 70, 114, 91, 13, 0, 0, 0, 0],
            &[0, 0, 27, 47, 6, 0, 0, 0, 0, 39, 41, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 219, 225, 0, 0],
            &[0, 0, 144, 255, 41, 0, 0, 0, 3, 244, 225, 0, 0],
            &[0, 0, 115, 255, 86, 0, 0, 0, 75, 255, 225, 0, 0],
            &[0, 0, 44, 252, 226, 91, 58, 104, 231, 231, 225, 0, 0],
            &[0, 0, 0, 98, 239, 255, 255, 230, 87, 147, 225, 0, 0],
            &[0, 0, 0, 0, 2, 31, 32, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ů' / 0x16e
        'Ů' => &[
            &[0, 0, 0, 70, 228, 43, 155, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 213, 6, 123, 182, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 190, 246, 233, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 2, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 0, 92, 144, 10, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 164, 255, 16, 0, 0, 0, 0, 176, 255, 11, 0, 0],
            &[0, 142, 255, 47, 0, 0, 0, 0, 206, 243, 0, 0, 0],
            &[0, 82, 255, 138, 0, 0, 0, 36, 252, 181, 0, 0, 0],
            &[0, 4, 214, 254, 151, 93, 115, 225, 252, 60, 0, 0, 0],
            &[0, 0, 27, 183, 254, 255, 255, 219, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 40, 24, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 141, 220, 185, 33, 0, 0, 0, 0, 0],
        ],
        // letter: 'ů' / 0x16f
        'ů' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 49, 122, 87, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 243, 141, 213, 130, 0, 0, 0, 0],
            &[0, 0, 0, 0, 88, 197, 0, 99, 190, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 242, 155, 220, 123, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 112, 77, 0, 0, 0, 0, 0],
            &[0, 0, 27, 47, 6, 0, 0, 0, 0, 39, 41, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 219, 225, 0, 0],
            &[0, 0, 144, 255, 41, 0, 0, 0, 3, 244, 225, 0, 0],
            &[0, 0, 115, 255, 86, 0, 0, 0, 75, 255, 225, 0, 0],
            &[0, 0, 44, 252, 226, 91, 58, 104, 231, 231, 225, 0, 0],
            &[0, 0, 0, 98, 239, 255, 255, 230, 87, 147, 225, 0, 0],
            &[0, 0, 0, 0, 2, 31, 32, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ű' / 0x170
        'Ű' => &[
            &[0, 0, 0, 7, 215, 211, 10, 139, 238, 64, 0, 0, 0],
            &[0, 0, 0, 119, 246, 56, 37, 249, 135, 0, 0, 0, 0],
            &[0, 0, 32, 235, 74, 0, 189, 151, 1, 0, 0, 0, 0],
            &[0, 0, 9, 17, 0, 0, 26, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 0, 92, 144, 10, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 164, 255, 16, 0, 0, 0, 0, 176, 255, 11, 0, 0],
            &[0, 142, 255, 47, 0, 0, 0, 0, 206, 243, 0, 0, 0],
            &[0, 82, 255, 138, 0, 0, 0, 36, 252, 181, 0, 0, 0],
            &[0, 4, 214, 254, 151, 93, 115, 225, 252, 60, 0, 0, 0],
            &[0, 0, 27, 183, 254, 255, 255, 219, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 40, 24, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ű' / 0x171
        'ű' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 118, 139, 10, 68, 141, 56, 0, 0],
            &[0, 0, 0, 0, 64, 255, 127, 7, 222, 207, 8, 0, 0],
            &[0, 0, 0, 5, 209, 160, 1, 127, 223, 26, 0, 0, 0],
            &[0, 0, 0, 36, 106, 2, 0, 117, 27, 0, 0, 0, 0],
            &[0, 0, 27, 47, 6, 0, 0, 0, 0, 39, 41, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0],
            &[0, 0, 147, 255, 34, 0, 0, 0, 0, 219, 225, 0, 0],
            &[0, 0, 144, 255, 41, 0, 0, 0, 3, 244, 225, 0, 0],
            &[0, 0, 115, 255, 86, 0, 0, 0, 75, 255, 225, 0, 0],
            &[0, 0, 44, 252, 226, 91, 58, 104, 231, 231, 225, 0, 0],
            &[0, 0, 0, 98, 239, 255, 255, 230, 87, 147, 225, 0, 0],
            &[0, 0, 0, 0, 2, 31, 32, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ų' / 0x172
        'Ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 97, 144, 5, 0, 0, 0, 0, 92, 144, 10, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 171, 255, 10, 0, 0, 0, 0, 163, 255, 17, 0, 0],
            &[0, 164, 255, 16, 0, 0, 0, 0, 176, 255, 11, 0, 0],
            &[0, 142, 255, 47, 0, 0, 0, 0, 206, 243, 0, 0, 0],
            &[0, 82, 255, 138, 0, 0, 0, 36, 252, 181, 0, 0, 0],
            &[0, 4, 214, 254, 151, 93, 115, 225, 252, 60, 0, 0, 0],
            &[0, 0, 27, 183, 254, 255, 255, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 185, 216, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 244, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 134, 233, 13, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 39, 10, 0, 0, 0, 0, 0],
        ],
        // letter: 'ų' / 0x173
        'ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 27, 47, 6, 0, 0, 0, 0, 39, 41, 0, 0, 0],
            &[0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0, 0],
            &[0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0, 0],
            &[0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0, 0],
            &[0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0, 0],
            &[0, 147, 255, 34, 0, 0, 0, 0, 212, 225, 0, 0, 0],
            &[0, 147, 255, 34, 0, 0, 0, 0, 219, 225, 0, 0, 0],
            &[0, 144, 255, 41, 0, 0, 0, 3, 244, 225, 0, 0, 0],
            &[0, 115, 255, 86, 0, 0, 0, 75, 255, 225, 0, 0, 0],
            &[0, 44, 252, 226, 91, 58, 104, 231, 231, 225, 0, 0, 0],
            &[0, 0, 98, 239, 255, 255, 230, 87, 168, 249, 0, 0, 0],
            &[0, 0, 0, 2, 31, 32, 4, 21, 220, 90, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 155, 180, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 210, 164, 2, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 23, 36, 2, 0, 0],
        ],
        // letter: 'Ŵ' / 0x174
        'Ŵ' => &[
            &[0, 0, 0, 0, 0, 120, 239, 199, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 250, 136, 234, 158, 0, 0, 0, 0],
            &[0, 0, 0, 49, 235, 97, 0, 41, 218, 135, 0, 0, 0],
            &[0, 0, 0, 13, 18, 0, 0, 0, 8, 24, 0, 0, 0],
            &[0, 71, 144, 11, 0, 0, 0, 0, 0, 0, 98, 129, 0],
            &[0, 103, 255, 39, 0, 0, 0, 0, 0, 0, 190, 209, 0],
            &[0, 75, 255, 63, 0, 0, 0, 0, 0, 0, 211, 183, 0],
            &[0, 46, 255, 88, 0, 5, 47, 26, 0, 0, 233, 157, 0],
            &[0, 17, 255, 112, 0, 65, 255, 179, 0, 2, 253, 131, 0],
            &[0, 0, 244, 136, 0, 125, 237, 238, 1, 22, 255, 105, 0],
            &[0, 0, 216, 160, 0, 184, 151, 254, 45, 44, 255, 80, 0],
            &[0, 0, 187, 184, 2, 241, 82, 216, 106, 65, 255, 54, 0],
            &[0, 0, 158, 204, 47, 255, 24, 156, 168, 83, 255, 28, 0],
            &[0, 0, 130, 223, 104, 220, 0, 94, 228, 99, 253, 4, 0],
            &[0, 0, 101, 240, 158, 161, 0, 33, 255, 144, 232, 0, 0],
            &[0, 0, 72, 253, 210, 101, 0, 0, 227, 207, 207, 0, 0],
            &[0, 0, 44, 255, 254, 42, 0, 0, 165, 253, 181, 0, 0],
            &[0, 0, 15, 255, 237, 1, 0, 0, 104, 255, 155, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŵ' / 0x175
        'ŵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 141, 106, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 219, 223, 255, 83, 0, 0, 0, 0, 0],
            &[0, 0, 7, 191, 192, 13, 112, 244, 59, 0, 0, 0, 0],
            &[0, 0, 52, 111, 6, 0, 0, 68, 106, 0, 0, 0, 0],
            &[32, 43, 0, 0, 0, 0, 0, 0, 0, 21, 47, 4, 0],
            &[150, 251, 4, 0, 0, 0, 0, 0, 0, 141, 250, 5, 0],
            &[108, 255, 37, 0, 19, 67, 45, 0, 0, 181, 214, 0, 0],
            &[67, 255, 75, 0, 120, 255, 215, 0, 0, 222, 172, 0, 0],
            &[26, 255, 113, 0, 193, 218, 255, 30, 9, 253, 130, 0, 0],
            &[0, 241, 150, 15, 251, 105, 249, 101, 47, 255, 88, 0, 0],
            &[0, 200, 185, 81, 255, 30, 194, 170, 83, 255, 46, 0, 0],
            &[0, 159, 213, 144, 216, 0, 128, 235, 113, 252, 7, 0, 0],
            &[0, 118, 233, 195, 149, 0, 63, 255, 171, 218, 0, 0, 0],
            &[0, 77, 250, 242, 82, 0, 7, 246, 237, 176, 0, 0, 0],
            &[0, 36, 255, 253, 18, 0, 0, 188, 255, 134, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŷ' / 0x176
        'Ŷ' => &[
            &[0, 0, 0, 0, 0, 120, 239, 199, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 70, 250, 136, 234, 158, 0, 0, 0, 0],
            &[0, 0, 0, 49, 235, 97, 0, 41, 218, 135, 0, 0, 0],
            &[0, 0, 0, 13, 18, 0, 0, 0, 8, 24, 0, 0, 0],
            &[0, 27, 144, 98, 0, 0, 0, 0, 0, 31, 144, 85, 0],
            &[0, 0, 202, 245, 20, 0, 0, 0, 0, 149, 254, 50, 0],
            &[0, 0, 72, 255, 128, 0, 0, 0, 22, 246, 174, 0, 0],
            &[0, 0, 0, 197, 236, 10, 0, 0, 132, 254, 46, 0, 0],
            &[0, 0, 0, 67, 255, 109, 0, 12, 238, 170, 0, 0, 0],
            &[0, 0, 0, 0, 193, 223, 4, 114, 253, 42, 0, 0, 0],
            &[0, 0, 0, 0, 63, 255, 95, 227, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 188, 245, 252, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 255, 165, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŷ' / 0x177
        'ŷ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 141, 103, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 222, 222, 254, 78, 0, 0, 0, 0],
            &[0, 0, 0, 9, 195, 189, 11, 117, 243, 55, 0, 0, 0],
            &[0, 0, 0, 54, 109, 5, 0, 0, 71, 103, 0, 0, 0],
            &[0, 5, 47, 26, 0, 0, 0, 0, 0, 11, 47, 23, 0],
            &[0, 1, 218, 205, 0, 0, 0, 0, 0, 115, 255, 69, 0],
            &[0, 0, 113, 255, 54, 0, 0, 0, 0, 200, 227, 2, 0],
            &[0, 0, 17, 246, 158, 0, 0, 0, 31, 255, 135, 0, 0],
            &[0, 0, 0, 156, 245, 16, 0, 0, 116, 255, 40, 0, 0],
            &[0, 0, 0, 50, 255, 110, 0, 0, 201, 201, 0, 0, 0],
            &[0, 0, 0, 0, 199, 213, 0, 32, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 93, 255, 55, 114, 249, 18, 0, 0, 0],
            &[0, 0, 0, 0, 7, 235, 143, 196, 172, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 136, 231, 253, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 253, 233, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 247, 143, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 113, 255, 44, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 232, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 162, 136, 37, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ÿ' / 0x178
        'Ÿ' => &[
            &[0, 0, 0, 11, 81, 13, 0, 0, 66, 40, 0, 0, 0],
            &[0, 0, 0, 104, 255, 116, 0, 17, 253, 206, 0, 0, 0],
            &[0, 0, 0, 42, 180, 49, 0, 0, 160, 110, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 27, 144, 98, 0, 0, 0, 0, 0, 31, 144, 85, 0],
            &[0, 0, 202, 245, 20, 0, 0, 0, 0, 149, 254, 50, 0],
            &[0, 0, 72, 255, 128, 0, 0, 0, 22, 246, 174, 0, 0],
            &[0, 0, 0, 197, 236, 10, 0, 0, 132, 254, 46, 0, 0],
            &[0, 0, 0, 67, 255, 109, 0, 12, 238, 170, 0, 0, 0],
            &[0, 0, 0, 0, 193, 223, 4, 114, 253, 42, 0, 0, 0],
            &[0, 0, 0, 0, 63, 255, 95, 227, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 188, 245, 252, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 255, 165, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ź' / 0x179
        'Ź' => &[
            &[0, 0, 0, 0, 0, 0, 0, 74, 239, 175, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 220, 215, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 146, 210, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 118, 144, 144, 144, 144, 144, 144, 86, 0, 0],
            &[0, 0, 0, 191, 234, 234, 234, 234, 238, 255, 148, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 127, 254, 51, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 244, 163, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 247, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 253, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 190, 234, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 78, 255, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 216, 215, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 109, 255, 77, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 236, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 140, 254, 51, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 249, 229, 133, 133, 133, 133, 133, 79, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 152, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ź' / 0x17a
        'ź' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 34, 141, 118, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 177, 249, 64, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 89, 249, 81, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 104, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 47, 47, 47, 47, 47, 47, 25, 0, 0],
            &[0, 0, 0, 252, 255, 255, 255, 255, 255, 255, 137, 0, 0],
            &[0, 0, 0, 32, 32, 32, 32, 32, 176, 253, 69, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 94, 255, 125, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 246, 179, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 220, 220, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 177, 245, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 121, 255, 91, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 252, 146, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 234, 245, 118, 113, 113, 113, 113, 78, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ż' / 0x17b
        'Ż' => &[
            &[0, 0, 0, 0, 0, 6, 171, 140, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 47, 255, 249, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 118, 144, 144, 144, 144, 144, 144, 86, 0, 0],
            &[0, 0, 0, 191, 234, 234, 234, 234, 238, 255, 148, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 127, 254, 51, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 244, 163, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 247, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 253, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 190, 234, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 78, 255, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 216, 215, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 109, 255, 77, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 236, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 140, 254, 51, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 249, 229, 133, 133, 133, 133, 133, 79, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 152, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ż' / 0x17c
        'ż' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 50, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 255, 238, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 213, 168, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 47, 47, 47, 47, 47, 47, 25, 0, 0],
            &[0, 0, 0, 252, 255, 255, 255, 255, 255, 255, 137, 0, 0],
            &[0, 0, 0, 32, 32, 32, 32, 32, 176, 253, 69, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 94, 255, 125, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 246, 179, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 220, 220, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 177, 245, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 121, 255, 91, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 252, 146, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 234, 245, 118, 113, 113, 113, 113, 78, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ž' / 0x17d
        'Ž' => &[
            &[0, 0, 0, 28, 219, 114, 0, 6, 149, 202, 11, 0, 0],
            &[0, 0, 0, 0, 58, 247, 146, 186, 228, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 96, 255, 250, 52, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 118, 144, 144, 144, 144, 144, 144, 86, 0, 0],
            &[0, 0, 0, 191, 234, 234, 234, 234, 238, 255, 148, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 127, 254, 51, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 244, 163, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 159, 247, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 253, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 190, 234, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 78, 255, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 216, 215, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 109, 255, 77, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 236, 191, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 140, 254, 51, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 249, 229, 133, 133, 133, 133, 133, 79, 0, 0],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 152, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ž' / 0x17e
        'ž' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 138, 29, 0, 0, 62, 136, 7, 0, 0],
            &[0, 0, 0, 0, 146, 235, 52, 104, 249, 86, 0, 0, 0],
            &[0, 0, 0, 0, 2, 178, 246, 255, 113, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 19, 124, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 47, 47, 47, 47, 47, 47, 25, 0, 0],
            &[0, 0, 0, 252, 255, 255, 255, 255, 255, 255, 137, 0, 0],
            &[0, 0, 0, 32, 32, 32, 32, 32, 176, 253, 69, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 94, 255, 125, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 246, 179, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 220, 220, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 177, 245, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 121, 255, 91, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 252, 146, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 234, 245, 118, 113, 113, 113, 113, 78, 0, 0],
            &[0, 0, 74, 255, 255, 255, 255, 255, 255, 255, 176, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ſ' / 0x17f
        'ſ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 79, 125, 128, 100, 22, 0, 0, 0],
            &[0, 0, 0, 19, 210, 255, 238, 228, 252, 47, 0, 0, 0],
            &[0, 0, 0, 136, 255, 116, 0, 0, 11, 0, 0, 0, 0],
            &[0, 0, 0, 190, 247, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 204, 224, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 204, 222, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 204, 222, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 204, 222, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 204, 222, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 204, 222, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 204, 222, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 204, 222, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 204, 222, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 204, 222, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 204, 222, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        _ => panic!("unsupported char"),
    }
}
