//! Module for letters with the font weight regular and size 16.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 16;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 9;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight regular and font size 14px
#[inline]
pub const fn get_char(c: char) -> &'static [&'static [u8]] {
    match c {
        // letter: ' ' / 0x20
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '!' / 0x21
        '!' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 137, 243, 0, 0, 0, 0],
            &[0, 0, 0, 130, 236, 0, 0, 0, 0],
            &[0, 0, 0, 122, 228, 0, 0, 0, 0],
            &[0, 0, 0, 114, 220, 0, 0, 0, 0],
            &[0, 0, 0, 106, 212, 0, 0, 0, 0],
            &[0, 0, 0, 98, 204, 0, 0, 0, 0],
            &[0, 0, 0, 90, 196, 0, 0, 0, 0],
            &[0, 0, 0, 14, 32, 0, 0, 0, 0],
            &[0, 0, 0, 79, 154, 0, 0, 0, 0],
            &[0, 0, 0, 138, 236, 3, 0, 0, 0],
            &[0, 0, 0, 1, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '"' / 0x22
        '"' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 183, 135, 33, 254, 30, 0, 0],
            &[0, 0, 164, 116, 13, 255, 11, 0, 0],
            &[0, 0, 144, 96, 0, 240, 0, 0, 0],
            &[0, 0, 79, 49, 0, 128, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '#' / 0x23
        '#' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 201, 55, 13, 232, 0, 0],
            &[0, 0, 3, 243, 10, 59, 187, 0, 0],
            &[1, 7, 48, 217, 7, 111, 142, 7, 0],
            &[46, 229, 240, 243, 229, 248, 236, 229, 27],
            &[0, 0, 139, 112, 0, 214, 39, 0, 0],
            &[0, 0, 190, 62, 11, 241, 2, 0, 0],
            &[159, 221, 250, 225, 226, 250, 221, 142, 0],
            &[13, 51, 221, 18, 120, 152, 18, 12, 0],
            &[0, 80, 172, 0, 158, 96, 0, 0, 0],
            &[0, 126, 124, 0, 208, 46, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '$' / 0x24
        '$' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 92, 0, 0, 0, 0],
            &[0, 0, 76, 177, 211, 127, 57, 0, 0],
            &[0, 119, 237, 181, 200, 158, 154, 0, 0],
            &[0, 214, 104, 81, 147, 0, 0, 0, 0],
            &[0, 193, 156, 85, 147, 0, 0, 0, 0],
            &[0, 50, 223, 247, 196, 46, 0, 0, 0],
            &[0, 0, 3, 126, 231, 247, 165, 1, 0],
            &[0, 0, 0, 81, 147, 34, 252, 51, 0],
            &[0, 6, 0, 81, 147, 22, 251, 47, 0],
            &[0, 183, 188, 186, 220, 230, 163, 0, 0],
            &[0, 32, 100, 166, 192, 58, 0, 0, 0],
            &[0, 0, 0, 66, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '%' / 0x25
        '%' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 7, 8, 0, 0, 0, 0, 0, 0],
            &[94, 218, 217, 90, 0, 31, 230, 12, 0],
            &[226, 38, 39, 226, 0, 153, 122, 0, 0],
            &[244, 16, 18, 246, 31, 231, 13, 0, 0],
            &[181, 117, 118, 185, 152, 123, 0, 0, 0],
            &[19, 135, 136, 51, 231, 13, 0, 0, 0],
            &[0, 0, 0, 152, 124, 93, 145, 62, 0],
            &[0, 0, 30, 231, 97, 198, 69, 226, 33],
            &[0, 0, 151, 124, 143, 117, 0, 173, 91],
            &[0, 29, 231, 14, 124, 138, 0, 193, 72],
            &[0, 150, 125, 0, 23, 209, 193, 188, 3],
            &[0, 0, 0, 0, 0, 1, 19, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '&' / 0x26
        '&' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 28, 3, 0, 0, 0, 0],
            &[0, 36, 226, 234, 223, 32, 0, 0, 0],
            &[0, 138, 174, 0, 176, 132, 0, 0, 0],
            &[0, 130, 162, 0, 188, 123, 0, 0, 0],
            &[0, 46, 240, 131, 226, 21, 0, 0, 0],
            &[0, 5, 255, 255, 42, 0, 0, 0, 0],
            &[0, 164, 210, 229, 133, 0, 140, 122, 0],
            &[66, 247, 22, 35, 238, 90, 231, 69, 0],
            &[109, 204, 0, 0, 78, 255, 255, 1, 0],
            &[70, 247, 51, 7, 102, 255, 255, 17, 0],
            &[0, 152, 253, 251, 211, 61, 189, 184, 2],
            &[0, 0, 15, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ''' / 0x27
        '\'' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 110, 208, 0, 0, 0, 0],
            &[0, 0, 0, 91, 189, 0, 0, 0, 0],
            &[0, 0, 0, 71, 170, 0, 0, 0, 0],
            &[0, 0, 0, 34, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '(' / 0x28
        '(' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 122, 181, 0, 0, 0],
            &[0, 0, 0, 33, 247, 35, 0, 0, 0],
            &[0, 0, 0, 139, 181, 0, 0, 0, 0],
            &[0, 0, 0, 217, 103, 0, 0, 0, 0],
            &[0, 0, 15, 255, 48, 0, 0, 0, 0],
            &[0, 0, 38, 255, 24, 0, 0, 0, 0],
            &[0, 0, 44, 255, 19, 0, 0, 0, 0],
            &[0, 0, 24, 255, 39, 0, 0, 0, 0],
            &[0, 0, 0, 234, 88, 0, 0, 0, 0],
            &[0, 0, 0, 166, 158, 0, 0, 0, 0],
            &[0, 0, 0, 60, 245, 16, 0, 0, 0],
            &[0, 0, 0, 0, 164, 145, 0, 0, 0],
            &[0, 0, 0, 0, 11, 51, 0, 0, 0],
        ],
        // letter: ')' / 0x29
        ')' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 76, 216, 10, 0, 0, 0, 0],
            &[0, 0, 0, 185, 135, 0, 0, 0, 0],
            &[0, 0, 0, 76, 238, 9, 0, 0, 0],
            &[0, 0, 0, 7, 247, 68, 0, 0, 0],
            &[0, 0, 0, 0, 200, 120, 0, 0, 0],
            &[0, 0, 0, 0, 176, 142, 0, 0, 0],
            &[0, 0, 0, 0, 173, 145, 0, 0, 0],
            &[0, 0, 0, 0, 194, 124, 0, 0, 0],
            &[0, 0, 0, 2, 241, 76, 0, 0, 0],
            &[0, 0, 0, 58, 246, 15, 0, 0, 0],
            &[0, 0, 0, 161, 157, 0, 0, 0, 0],
            &[0, 0, 49, 233, 25, 0, 0, 0, 0],
            &[0, 0, 30, 33, 0, 0, 0, 0, 0],
        ],
        // letter: '*' / 0x2a
        '*' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 38, 83, 0, 0, 0, 0],
            &[0, 0, 0, 73, 174, 0, 0, 0, 0],
            &[0, 77, 77, 68, 160, 56, 125, 5, 0],
            &[0, 94, 164, 225, 254, 188, 155, 18, 0],
            &[0, 0, 31, 221, 174, 123, 0, 0, 0],
            &[0, 0, 177, 128, 31, 245, 40, 0, 0],
            &[0, 0, 13, 12, 0, 29, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '+' / 0x2b
        '+' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 34, 0, 0, 0, 0],
            &[0, 0, 0, 74, 183, 0, 0, 0, 0],
            &[0, 0, 0, 74, 183, 0, 0, 0, 0],
            &[0, 97, 110, 152, 214, 110, 110, 30, 0],
            &[0, 126, 143, 176, 224, 143, 143, 40, 0],
            &[0, 0, 0, 74, 183, 0, 0, 0, 0],
            &[0, 0, 0, 74, 183, 0, 0, 0, 0],
            &[0, 0, 0, 22, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ',' / 0x2c
        ',' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 73, 206, 48, 0, 0, 0],
            &[0, 0, 0, 131, 240, 7, 0, 0, 0],
            &[0, 0, 0, 183, 153, 0, 0, 0, 0],
            &[0, 0, 1, 228, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '-' / 0x2d
        '-' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 78, 87, 87, 87, 26, 0, 0],
            &[0, 0, 179, 199, 199, 199, 59, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '.' / 0x2e
        '.' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 79, 154, 0, 0, 0, 0],
            &[0, 0, 0, 138, 236, 3, 0, 0, 0],
            &[0, 0, 0, 1, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '/' / 0x2f
        '/' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 172, 134, 0, 0],
            &[0, 0, 0, 0, 20, 249, 38, 0, 0],
            &[0, 0, 0, 0, 113, 195, 0, 0, 0],
            &[0, 0, 0, 0, 211, 96, 0, 0, 0],
            &[0, 0, 0, 53, 242, 11, 0, 0, 0],
            &[0, 0, 0, 151, 156, 0, 0, 0, 0],
            &[0, 0, 9, 240, 58, 0, 0, 0, 0],
            &[0, 0, 92, 215, 0, 0, 0, 0, 0],
            &[0, 0, 190, 117, 0, 0, 0, 0, 0],
            &[0, 34, 250, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '0' / 0x30
        '0' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 22, 0, 0, 0, 0],
            &[0, 6, 164, 254, 254, 215, 48, 0, 0],
            &[0, 128, 214, 34, 8, 230, 235, 3, 0],
            &[0, 223, 87, 0, 27, 246, 255, 67, 0],
            &[15, 255, 30, 0, 167, 157, 175, 116, 0],
            &[33, 255, 9, 63, 239, 22, 152, 136, 0],
            &[33, 255, 17, 209, 109, 0, 156, 137, 0],
            &[13, 255, 143, 210, 3, 0, 177, 118, 0],
            &[0, 236, 255, 62, 0, 4, 236, 70, 0],
            &[0, 121, 255, 40, 14, 143, 225, 5, 0],
            &[0, 4, 158, 254, 254, 220, 52, 0, 0],
            &[0, 0, 0, 11, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '1' / 0x31
        '1' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 171, 253, 42, 0, 0, 0],
            &[0, 97, 244, 142, 254, 42, 0, 0, 0],
            &[0, 18, 25, 5, 255, 42, 0, 0, 0],
            &[0, 0, 0, 9, 255, 42, 0, 0, 0],
            &[0, 0, 0, 9, 255, 42, 0, 0, 0],
            &[0, 0, 0, 9, 255, 42, 0, 0, 0],
            &[0, 0, 0, 9, 255, 42, 0, 0, 0],
            &[0, 0, 0, 9, 255, 42, 0, 0, 0],
            &[0, 0, 0, 9, 255, 42, 0, 0, 0],
            &[0, 29, 225, 252, 255, 255, 228, 60, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '2' / 0x32
        '2' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 25, 0, 0, 0, 0],
            &[0, 68, 204, 255, 254, 221, 57, 0, 0],
            &[0, 132, 118, 15, 11, 156, 223, 0, 0],
            &[0, 0, 0, 0, 0, 53, 255, 17, 0],
            &[0, 0, 0, 0, 0, 88, 244, 4, 0],
            &[0, 0, 0, 0, 4, 207, 153, 0, 0],
            &[0, 0, 0, 0, 153, 218, 17, 0, 0],
            &[0, 0, 0, 140, 229, 33, 0, 0, 0],
            &[0, 0, 142, 227, 37, 0, 0, 0, 0],
            &[0, 146, 237, 64, 30, 30, 30, 19, 0],
            &[5, 255, 255, 255, 255, 255, 255, 161, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '3' / 0x33
        '3' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 20, 25, 1, 0, 0, 0],
            &[0, 85, 216, 254, 254, 223, 61, 0, 0],
            &[0, 101, 86, 8, 11, 160, 224, 0, 0],
            &[0, 0, 0, 0, 0, 74, 255, 4, 0],
            &[0, 0, 0, 0, 3, 169, 184, 0, 0],
            &[0, 0, 91, 188, 224, 161, 16, 0, 0],
            &[0, 0, 49, 104, 132, 226, 155, 0, 0],
            &[0, 0, 0, 0, 0, 23, 252, 59, 0],
            &[0, 0, 0, 0, 0, 4, 249, 79, 0],
            &[0, 105, 32, 3, 17, 145, 244, 20, 0],
            &[0, 185, 252, 253, 255, 212, 64, 0, 0],
            &[0, 0, 8, 29, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '4' / 0x34
        '4' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 13, 3, 0, 0],
            &[0, 0, 0, 0, 132, 255, 62, 0, 0],
            &[0, 0, 0, 39, 227, 236, 62, 0, 0],
            &[0, 0, 0, 187, 109, 237, 62, 0, 0],
            &[0, 0, 87, 209, 3, 243, 62, 0, 0],
            &[0, 13, 228, 59, 0, 245, 62, 0, 0],
            &[0, 142, 160, 0, 0, 245, 62, 0, 0],
            &[40, 250, 104, 81, 81, 248, 123, 56, 0],
            &[58, 194, 194, 194, 194, 253, 209, 136, 0],
            &[0, 0, 0, 0, 0, 245, 62, 0, 0],
            &[0, 0, 0, 0, 0, 245, 62, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '5' / 0x35
        '5' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 254, 254, 254, 254, 176, 0, 0],
            &[0, 105, 198, 31, 31, 31, 21, 0, 0],
            &[0, 126, 168, 0, 0, 0, 0, 0, 0],
            &[0, 146, 148, 27, 20, 0, 0, 0, 0],
            &[0, 149, 250, 232, 250, 210, 52, 0, 0],
            &[0, 3, 5, 0, 13, 163, 231, 7, 0],
            &[0, 0, 0, 0, 0, 25, 255, 55, 0],
            &[0, 0, 0, 0, 0, 29, 255, 49, 0],
            &[0, 91, 24, 1, 32, 176, 217, 2, 0],
            &[0, 168, 252, 253, 254, 190, 36, 0, 0],
            &[0, 0, 7, 29, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '6' / 0x36
        '6' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 30, 12, 0, 0],
            &[0, 0, 51, 195, 253, 242, 185, 0, 0],
            &[0, 39, 242, 126, 16, 0, 5, 0, 0],
            &[0, 164, 159, 0, 0, 0, 0, 0, 0],
            &[0, 239, 71, 35, 73, 39, 0, 0, 0],
            &[22, 255, 158, 207, 177, 234, 161, 0, 0],
            &[37, 255, 147, 0, 0, 20, 236, 78, 0],
            &[27, 255, 37, 0, 0, 0, 176, 130, 0],
            &[1, 237, 70, 0, 0, 0, 191, 122, 0],
            &[0, 139, 209, 30, 0, 77, 249, 43, 0],
            &[0, 8, 162, 251, 244, 234, 92, 0, 0],
            &[0, 0, 0, 12, 28, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '7' / 0x37
        '7' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[40, 254, 254, 254, 254, 254, 254, 142, 0],
            &[5, 31, 31, 31, 31, 34, 231, 89, 0],
            &[0, 0, 0, 0, 0, 79, 231, 5, 0],
            &[0, 0, 0, 0, 0, 190, 129, 0, 0],
            &[0, 0, 0, 0, 46, 250, 26, 0, 0],
            &[0, 0, 0, 0, 156, 169, 0, 0, 0],
            &[0, 0, 0, 20, 247, 61, 0, 0, 0],
            &[0, 0, 0, 122, 209, 0, 0, 0, 0],
            &[0, 0, 5, 228, 101, 0, 0, 0, 0],
            &[0, 0, 88, 239, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '8' / 0x38
        '8' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 27, 1, 0, 0, 0],
            &[0, 19, 183, 250, 244, 228, 75, 0, 0],
            &[0, 145, 199, 15, 0, 116, 239, 5, 0],
            &[0, 168, 145, 0, 0, 45, 255, 15, 0],
            &[0, 81, 234, 53, 11, 174, 180, 0, 0],
            &[0, 0, 112, 253, 237, 170, 9, 0, 0],
            &[0, 44, 224, 138, 127, 236, 102, 0, 0],
            &[0, 213, 129, 0, 0, 48, 249, 53, 0],
            &[5, 255, 60, 0, 0, 0, 215, 102, 0],
            &[0, 211, 154, 5, 0, 76, 250, 48, 0],
            &[0, 43, 205, 246, 238, 232, 94, 0, 0],
            &[0, 0, 0, 20, 26, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '9' / 0x39
        '9' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 25, 0, 0, 0, 0],
            &[0, 30, 192, 251, 249, 215, 52, 0, 0],
            &[0, 195, 168, 11, 8, 135, 229, 9, 0],
            &[21, 255, 37, 0, 0, 2, 222, 80, 0],
            &[27, 255, 23, 0, 0, 0, 186, 126, 0],
            &[0, 229, 100, 0, 0, 55, 246, 139, 0],
            &[0, 73, 238, 188, 182, 199, 205, 125, 0],
            &[0, 0, 17, 68, 60, 4, 226, 85, 0],
            &[0, 0, 0, 0, 0, 59, 249, 19, 0],
            &[0, 3, 1, 1, 63, 224, 136, 0, 0],
            &[0, 49, 245, 248, 236, 123, 1, 0, 0],
            &[0, 1, 26, 26, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ':' / 0x3a
        ':' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 59, 125, 0, 0, 0, 0],
            &[0, 0, 0, 142, 245, 3, 0, 0, 0],
            &[0, 0, 0, 11, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 79, 153, 0, 0, 0, 0],
            &[0, 0, 0, 138, 235, 2, 0, 0, 0],
            &[0, 0, 0, 1, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ';' / 0x3b
        ';' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 142, 24, 0, 0, 0],
            &[0, 0, 0, 63, 255, 74, 0, 0, 0],
            &[0, 0, 0, 1, 43, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 73, 206, 48, 0, 0, 0],
            &[0, 0, 0, 131, 240, 7, 0, 0, 0],
            &[0, 0, 0, 183, 153, 0, 0, 0, 0],
            &[0, 0, 1, 228, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '<' / 0x3c
        '<' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 43, 16, 0],
            &[0, 0, 0, 0, 22, 152, 224, 22, 0],
            &[0, 0, 8, 121, 228, 122, 9, 0, 0],
            &[0, 81, 220, 148, 21, 0, 0, 0, 0],
            &[0, 125, 230, 120, 13, 0, 0, 0, 0],
            &[0, 0, 29, 148, 231, 129, 18, 0, 0],
            &[0, 0, 0, 0, 26, 145, 234, 26, 0],
            &[0, 0, 0, 0, 0, 0, 24, 12, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '=' / 0x3d
        '=' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[4, 209, 209, 209, 209, 209, 209, 87, 0],
            &[1, 77, 77, 77, 77, 77, 77, 32, 0],
            &[0, 40, 40, 40, 40, 40, 40, 16, 0],
            &[4, 246, 246, 246, 246, 246, 246, 103, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '>' / 0x3e
        '>' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 2, 0, 0, 0, 0, 0, 0],
            &[0, 148, 207, 66, 0, 0, 0, 0, 0],
            &[0, 0, 66, 205, 178, 40, 0, 0, 0],
            &[0, 0, 0, 1, 91, 220, 148, 9, 0],
            &[0, 0, 0, 0, 70, 197, 202, 18, 0],
            &[0, 1, 80, 203, 199, 73, 0, 0, 0],
            &[0, 167, 195, 69, 0, 0, 0, 0, 0],
            &[0, 36, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '?' / 0x3f
        '?' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 25, 1, 0, 0, 0],
            &[0, 103, 232, 255, 252, 226, 73, 0, 0],
            &[0, 56, 69, 12, 4, 119, 240, 7, 0],
            &[0, 0, 0, 0, 0, 24, 255, 25, 0],
            &[0, 0, 0, 0, 0, 95, 228, 1, 0],
            &[0, 0, 0, 0, 94, 237, 67, 0, 0],
            &[0, 0, 0, 88, 232, 51, 0, 0, 0],
            &[0, 0, 0, 184, 105, 0, 0, 0, 0],
            &[0, 0, 0, 34, 13, 0, 0, 0, 0],
            &[0, 0, 0, 150, 83, 0, 0, 0, 0],
            &[0, 0, 1, 233, 143, 0, 0, 0, 0],
            &[0, 0, 0, 11, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '@' / 0x40
        '@' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 114, 219, 239, 207, 75, 0, 0],
            &[0, 141, 183, 31, 1, 59, 224, 64, 0],
            &[39, 223, 9, 18, 63, 38, 63, 194, 0],
            &[126, 127, 48, 229, 170, 246, 55, 228, 12],
            &[175, 72, 177, 114, 0, 231, 42, 198, 38],
            &[196, 51, 225, 57, 1, 248, 28, 197, 39],
            &[189, 57, 215, 61, 35, 255, 26, 217, 12],
            &[159, 94, 138, 196, 170, 175, 165, 183, 0],
            &[87, 181, 5, 100, 75, 11, 104, 22, 0],
            &[3, 209, 113, 1, 0, 0, 18, 0, 0],
            &[0, 25, 187, 223, 185, 205, 203, 0, 0],
            &[0, 0, 0, 26, 55, 33, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'A' / 0x41
        'A' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 8, 0, 0, 0, 0],
            &[0, 0, 0, 165, 249, 15, 0, 0, 0],
            &[0, 0, 6, 241, 224, 91, 0, 0, 0],
            &[0, 0, 75, 224, 125, 174, 0, 0, 0],
            &[0, 0, 157, 146, 47, 246, 11, 0, 0],
            &[0, 4, 236, 67, 0, 223, 85, 0, 0],
            &[0, 67, 255, 52, 47, 182, 168, 0, 0],
            &[0, 150, 255, 239, 239, 255, 248, 8, 0],
            &[2, 231, 84, 0, 0, 4, 237, 78, 0],
            &[60, 248, 12, 0, 0, 0, 162, 161, 0],
            &[142, 181, 0, 0, 0, 0, 82, 239, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'B' / 0x42
        'B' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 254, 254, 242, 203, 84, 0, 0],
            &[0, 171, 144, 24, 37, 124, 252, 36, 0],
            &[0, 171, 133, 0, 0, 0, 247, 79, 0],
            &[0, 171, 133, 0, 0, 50, 251, 34, 0],
            &[0, 171, 227, 196, 207, 225, 74, 0, 0],
            &[0, 171, 172, 82, 90, 165, 219, 30, 0],
            &[0, 171, 133, 0, 0, 0, 204, 140, 0],
            &[0, 171, 133, 0, 0, 0, 186, 149, 0],
            &[0, 171, 144, 23, 33, 100, 252, 73, 0],
            &[0, 171, 255, 255, 252, 218, 103, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'C' / 0x43
        'C' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 24, 1, 0, 0],
            &[0, 0, 57, 206, 255, 255, 242, 103, 0],
            &[0, 41, 245, 138, 20, 7, 54, 38, 0],
            &[0, 166, 184, 0, 0, 0, 0, 0, 0],
            &[0, 236, 96, 0, 0, 0, 0, 0, 0],
            &[11, 255, 61, 0, 0, 0, 0, 0, 0],
            &[9, 255, 62, 0, 0, 0, 0, 0, 0],
            &[0, 238, 94, 0, 0, 0, 0, 0, 0],
            &[0, 171, 178, 0, 0, 0, 0, 0, 0],
            &[0, 50, 249, 137, 28, 15, 58, 28, 0],
            &[0, 0, 70, 214, 255, 255, 239, 52, 0],
            &[0, 0, 0, 0, 16, 26, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'D' / 0x44
        'D' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 254, 254, 249, 218, 128, 6, 0, 0],
            &[5, 255, 80, 32, 82, 219, 185, 1, 0],
            &[5, 255, 61, 0, 0, 30, 246, 81, 0],
            &[5, 255, 61, 0, 0, 0, 177, 158, 0],
            &[5, 255, 61, 0, 0, 0, 141, 187, 0],
            &[5, 255, 61, 0, 0, 0, 141, 183, 0],
            &[5, 255, 61, 0, 0, 0, 181, 151, 0],
            &[5, 255, 61, 0, 0, 36, 249, 70, 0],
            &[5, 255, 79, 33, 91, 224, 169, 0, 0],
            &[5, 255, 255, 245, 209, 114, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'E' / 0x45
        'E' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 254, 254, 254, 254, 254, 53, 0],
            &[0, 171, 163, 31, 31, 31, 31, 6, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 233, 200, 200, 200, 188, 0, 0],
            &[0, 171, 186, 86, 86, 86, 81, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 163, 30, 30, 30, 30, 6, 0],
            &[0, 171, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'F' / 0x46
        'F' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 254, 254, 254, 254, 254, 53, 0],
            &[0, 171, 163, 31, 31, 31, 31, 6, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 187, 89, 89, 89, 84, 0, 0],
            &[0, 171, 231, 197, 197, 197, 186, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'G' / 0x47
        'G' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 17, 0, 0, 0],
            &[0, 0, 104, 229, 255, 255, 220, 31, 0],
            &[0, 102, 247, 99, 14, 17, 80, 1, 0],
            &[2, 233, 116, 0, 0, 0, 0, 0, 0],
            &[51, 255, 25, 0, 0, 0, 0, 0, 0],
            &[81, 245, 0, 0, 43, 71, 71, 29, 0],
            &[87, 241, 0, 0, 133, 215, 247, 107, 0],
            &[65, 255, 16, 0, 0, 0, 204, 107, 0],
            &[11, 248, 95, 0, 0, 0, 204, 107, 0],
            &[0, 139, 237, 75, 7, 13, 212, 107, 0],
            &[0, 3, 135, 241, 254, 255, 229, 73, 0],
            &[0, 0, 0, 3, 26, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'H' / 0x48
        'H' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 254, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 213, 200, 200, 200, 246, 107, 0],
            &[5, 255, 127, 86, 86, 86, 228, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'I' / 0x49
        'I' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 104, 244, 254, 254, 251, 199, 0, 0],
            &[0, 0, 0, 117, 210, 2, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 117, 210, 1, 0, 0, 0],
            &[0, 104, 244, 255, 255, 251, 199, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'J' / 0x4a
        'J' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 129, 0, 0],
            &[0, 0, 0, 0, 0, 191, 130, 0, 0],
            &[0, 0, 0, 0, 0, 191, 130, 0, 0],
            &[0, 0, 0, 0, 0, 191, 130, 0, 0],
            &[0, 0, 0, 0, 0, 191, 130, 0, 0],
            &[0, 0, 0, 0, 0, 191, 130, 0, 0],
            &[0, 0, 0, 0, 0, 192, 130, 0, 0],
            &[0, 0, 0, 0, 0, 212, 117, 0, 0],
            &[0, 55, 38, 18, 107, 254, 55, 0, 0],
            &[0, 126, 252, 255, 246, 124, 0, 0, 0],
            &[0, 0, 9, 29, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'K' / 0x4b
        'K' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 254, 61, 0, 0, 64, 247, 56, 0],
            &[5, 255, 61, 0, 22, 230, 119, 0, 0],
            &[5, 255, 61, 1, 185, 184, 1, 0, 0],
            &[5, 255, 61, 122, 230, 22, 0, 0, 0],
            &[5, 255, 114, 251, 102, 0, 0, 0, 0],
            &[5, 255, 240, 186, 211, 5, 0, 0, 0],
            &[5, 255, 113, 11, 225, 122, 0, 0, 0],
            &[5, 255, 61, 0, 77, 247, 37, 0, 0],
            &[5, 255, 61, 0, 0, 173, 190, 0, 0],
            &[5, 255, 61, 0, 0, 27, 242, 97, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'L' / 0x4c
        'L' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 99, 221, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 228, 41, 41, 41, 41, 8, 0],
            &[0, 99, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'M' / 0x4d
        'M' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[76, 254, 165, 0, 0, 66, 254, 178, 0],
            &[76, 232, 235, 2, 0, 139, 232, 179, 0],
            &[76, 209, 224, 54, 0, 212, 174, 179, 0],
            &[76, 222, 148, 127, 29, 240, 125, 179, 0],
            &[76, 227, 72, 200, 101, 169, 124, 179, 0],
            &[76, 227, 7, 241, 194, 94, 125, 179, 0],
            &[76, 227, 0, 176, 253, 20, 125, 179, 0],
            &[76, 227, 0, 60, 110, 0, 125, 179, 0],
            &[76, 227, 0, 0, 0, 0, 125, 179, 0],
            &[76, 227, 0, 0, 0, 0, 125, 179, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'N' / 0x4e
        'N' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 254, 218, 1, 0, 0, 203, 107, 0],
            &[5, 255, 222, 76, 0, 0, 204, 107, 0],
            &[5, 255, 130, 188, 0, 0, 204, 107, 0],
            &[5, 255, 53, 235, 45, 0, 204, 107, 0],
            &[5, 255, 50, 133, 156, 0, 204, 107, 0],
            &[5, 255, 50, 26, 241, 20, 203, 107, 0],
            &[5, 255, 50, 0, 164, 124, 199, 107, 0],
            &[5, 255, 50, 0, 52, 230, 193, 107, 0],
            &[5, 255, 50, 0, 0, 195, 247, 107, 0],
            &[5, 255, 50, 0, 0, 83, 255, 107, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'O' / 0x4f
        'O' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 32, 3, 0, 0, 0],
            &[0, 12, 175, 255, 253, 228, 68, 0, 0],
            &[0, 157, 206, 31, 11, 128, 242, 21, 0],
            &[13, 250, 67, 0, 0, 0, 224, 113, 0],
            &[63, 253, 9, 0, 0, 0, 163, 168, 0],
            &[86, 240, 0, 0, 0, 0, 141, 191, 0],
            &[88, 241, 0, 0, 0, 0, 138, 190, 0],
            &[65, 254, 10, 0, 0, 0, 161, 166, 0],
            &[16, 251, 69, 0, 0, 0, 220, 111, 0],
            &[0, 160, 209, 36, 9, 129, 239, 19, 0],
            &[0, 13, 174, 254, 255, 222, 61, 0, 0],
            &[0, 0, 0, 15, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'P' / 0x50
        'P' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 254, 254, 250, 216, 106, 0, 0],
            &[0, 171, 161, 24, 36, 107, 252, 69, 0],
            &[0, 171, 150, 0, 0, 0, 189, 142, 0],
            &[0, 171, 150, 0, 0, 0, 185, 143, 0],
            &[0, 171, 150, 0, 4, 83, 249, 62, 0],
            &[0, 171, 249, 241, 251, 234, 105, 0, 0],
            &[0, 171, 166, 38, 26, 3, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Q' / 0x51
        'Q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 32, 3, 0, 0, 0],
            &[0, 12, 175, 255, 253, 228, 68, 0, 0],
            &[0, 157, 206, 31, 11, 128, 242, 21, 0],
            &[13, 250, 67, 0, 0, 0, 224, 113, 0],
            &[63, 253, 9, 0, 0, 0, 163, 168, 0],
            &[86, 240, 0, 0, 0, 0, 141, 191, 0],
            &[88, 241, 0, 0, 0, 0, 138, 190, 0],
            &[65, 254, 10, 0, 0, 0, 161, 166, 0],
            &[16, 251, 69, 0, 0, 0, 220, 111, 0],
            &[0, 160, 209, 36, 9, 146, 239, 19, 0],
            &[0, 13, 174, 254, 255, 255, 76, 0, 0],
            &[0, 0, 0, 15, 28, 214, 130, 0, 0],
            &[0, 0, 0, 0, 0, 86, 243, 21, 0],
            &[0, 0, 0, 0, 0, 1, 125, 58, 0],
        ],
        // letter: 'R' / 0x52
        'R' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 254, 253, 237, 177, 38, 0, 0],
            &[0, 171, 161, 26, 53, 184, 221, 3, 0],
            &[0, 171, 150, 0, 0, 48, 255, 36, 0],
            &[0, 171, 150, 0, 0, 62, 255, 24, 0],
            &[0, 171, 169, 47, 73, 212, 163, 0, 0],
            &[0, 171, 246, 233, 251, 153, 5, 0, 0],
            &[0, 171, 150, 0, 126, 215, 5, 0, 0],
            &[0, 171, 150, 0, 8, 223, 118, 0, 0],
            &[0, 171, 150, 0, 0, 81, 244, 28, 0],
            &[0, 171, 150, 0, 0, 0, 186, 170, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'S' / 0x53
        'S' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 28, 9, 0, 0, 0],
            &[0, 2, 145, 247, 255, 255, 208, 16, 0],
            &[0, 96, 245, 67, 5, 24, 80, 0, 0],
            &[0, 142, 196, 0, 0, 0, 0, 0, 0],
            &[0, 101, 242, 51, 0, 0, 0, 0, 0],
            &[0, 4, 159, 252, 170, 48, 0, 0, 0],
            &[0, 0, 0, 50, 173, 252, 151, 1, 0],
            &[0, 0, 0, 0, 0, 64, 255, 68, 0],
            &[0, 0, 0, 0, 0, 1, 242, 91, 0],
            &[0, 84, 36, 3, 9, 112, 250, 37, 0],
            &[0, 154, 249, 251, 253, 226, 87, 0, 0],
            &[0, 0, 3, 25, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'T' / 0x54
        'T' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[94, 254, 254, 254, 254, 254, 254, 196, 0],
            &[11, 31, 31, 128, 217, 31, 31, 24, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'U' / 0x55
        'U' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 254, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[2, 255, 67, 0, 0, 0, 220, 104, 0],
            &[0, 234, 100, 0, 0, 3, 247, 80, 0],
            &[0, 159, 208, 29, 6, 125, 241, 16, 0],
            &[0, 21, 188, 255, 255, 228, 74, 0, 0],
            &[0, 0, 0, 16, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'V' / 0x56
        'V' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[124, 207, 0, 0, 0, 0, 108, 226, 1],
            &[43, 255, 29, 0, 0, 0, 182, 147, 0],
            &[0, 216, 105, 0, 0, 10, 247, 66, 0],
            &[0, 134, 182, 0, 0, 75, 237, 4, 0],
            &[0, 51, 247, 11, 0, 150, 160, 0, 0],
            &[0, 0, 224, 79, 0, 224, 79, 0, 0],
            &[0, 0, 142, 156, 42, 244, 9, 0, 0],
            &[0, 0, 60, 228, 112, 173, 0, 0, 0],
            &[0, 0, 2, 231, 211, 92, 0, 0, 0],
            &[0, 0, 0, 151, 250, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'W' / 0x57
        'W' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[151, 142, 0, 0, 0, 0, 39, 251, 3],
            &[123, 167, 0, 0, 0, 0, 61, 230, 0],
            &[94, 191, 0, 57, 113, 0, 83, 204, 0],
            &[66, 216, 0, 159, 247, 16, 105, 178, 0],
            &[37, 240, 0, 217, 175, 75, 127, 152, 0],
            &[9, 255, 30, 218, 101, 136, 147, 126, 0],
            &[0, 235, 106, 159, 40, 197, 164, 101, 0],
            &[0, 207, 178, 100, 0, 227, 185, 75, 0],
            &[0, 178, 239, 41, 0, 173, 239, 49, 0],
            &[0, 149, 236, 1, 0, 111, 255, 23, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'X' / 0x58
        'X' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[19, 235, 85, 0, 0, 8, 224, 102, 0],
            &[0, 101, 222, 7, 0, 122, 203, 2, 0],
            &[0, 1, 201, 116, 24, 241, 54, 0, 0],
            &[0, 0, 51, 237, 173, 155, 0, 0, 0],
            &[0, 0, 0, 155, 241, 20, 0, 0, 0],
            &[0, 0, 1, 200, 250, 52, 0, 0, 0],
            &[0, 0, 101, 212, 109, 202, 2, 0, 0],
            &[0, 19, 236, 69, 3, 210, 102, 0, 0],
            &[0, 154, 177, 0, 0, 66, 237, 19, 0],
            &[55, 247, 37, 0, 0, 0, 174, 155, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Y' / 0x59
        'Y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[65, 250, 31, 0, 0, 0, 169, 167, 0],
            &[0, 191, 145, 0, 0, 36, 250, 40, 0],
            &[0, 61, 244, 18, 0, 152, 163, 0, 0],
            &[0, 0, 186, 126, 23, 243, 37, 0, 0],
            &[0, 0, 56, 234, 144, 158, 0, 0, 0],
            &[0, 0, 0, 182, 250, 33, 0, 0, 0],
            &[0, 0, 0, 103, 205, 0, 0, 0, 0],
            &[0, 0, 0, 102, 205, 0, 0, 0, 0],
            &[0, 0, 0, 102, 205, 0, 0, 0, 0],
            &[0, 0, 0, 102, 205, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Z' / 0x5a
        'Z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 99, 254, 254, 254, 254, 254, 17, 0],
            &[0, 9, 24, 24, 24, 136, 201, 1, 0],
            &[0, 0, 0, 0, 20, 239, 60, 0, 0],
            &[0, 0, 0, 0, 148, 173, 0, 0, 0],
            &[0, 0, 0, 40, 245, 37, 0, 0, 0],
            &[0, 0, 0, 179, 145, 0, 0, 0, 0],
            &[0, 0, 67, 239, 19, 0, 0, 0, 0],
            &[0, 2, 208, 116, 0, 0, 0, 0, 0],
            &[0, 98, 233, 37, 30, 30, 30, 2, 0],
            &[0, 171, 255, 255, 255, 255, 255, 17, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '[' / 0x5b
        '[' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 163, 254, 254, 129, 0, 0, 0],
            &[0, 0, 163, 145, 3, 1, 0, 0, 0],
            &[0, 0, 163, 144, 0, 0, 0, 0, 0],
            &[0, 0, 163, 144, 0, 0, 0, 0, 0],
            &[0, 0, 163, 144, 0, 0, 0, 0, 0],
            &[0, 0, 163, 144, 0, 0, 0, 0, 0],
            &[0, 0, 163, 144, 0, 0, 0, 0, 0],
            &[0, 0, 163, 144, 0, 0, 0, 0, 0],
            &[0, 0, 163, 144, 0, 0, 0, 0, 0],
            &[0, 0, 163, 144, 0, 0, 0, 0, 0],
            &[0, 0, 163, 144, 0, 0, 0, 0, 0],
            &[0, 0, 163, 233, 203, 103, 0, 0, 0],
            &[0, 0, 34, 54, 54, 27, 0, 0, 0],
        ],
        // letter: '\' / 0x5c
        '\\' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 34, 249, 23, 0, 0, 0, 0, 0],
            &[0, 0, 190, 117, 0, 0, 0, 0, 0],
            &[0, 0, 92, 215, 0, 0, 0, 0, 0],
            &[0, 0, 9, 240, 57, 0, 0, 0, 0],
            &[0, 0, 0, 152, 155, 0, 0, 0, 0],
            &[0, 0, 0, 53, 242, 11, 0, 0, 0],
            &[0, 0, 0, 0, 211, 96, 0, 0, 0],
            &[0, 0, 0, 0, 113, 194, 0, 0, 0],
            &[0, 0, 0, 0, 21, 249, 37, 0, 0],
            &[0, 0, 0, 0, 0, 172, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ']' / 0x5d
        ']' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 34, 34, 34, 1, 0, 0],
            &[0, 0, 24, 223, 228, 255, 10, 0, 0],
            &[0, 0, 0, 0, 41, 255, 10, 0, 0],
            &[0, 0, 0, 0, 41, 255, 10, 0, 0],
            &[0, 0, 0, 0, 41, 255, 10, 0, 0],
            &[0, 0, 0, 0, 41, 255, 10, 0, 0],
            &[0, 0, 0, 0, 41, 255, 10, 0, 0],
            &[0, 0, 0, 0, 41, 255, 10, 0, 0],
            &[0, 0, 0, 0, 41, 255, 10, 0, 0],
            &[0, 0, 0, 0, 41, 255, 10, 0, 0],
            &[0, 0, 0, 0, 41, 255, 10, 0, 0],
            &[0, 0, 0, 0, 41, 255, 10, 0, 0],
            &[0, 0, 26, 239, 242, 255, 10, 0, 0],
            &[0, 0, 2, 18, 18, 18, 0, 0, 0],
        ],
        // letter: '^' / 0x5e
        '^' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 6, 0, 0, 0, 0],
            &[0, 0, 0, 146, 176, 0, 0, 0, 0],
            &[0, 0, 21, 223, 206, 55, 0, 0, 0],
            &[0, 0, 132, 125, 76, 188, 0, 0, 0],
            &[0, 14, 229, 19, 0, 202, 65, 0, 0],
            &[0, 119, 150, 0, 0, 73, 199, 0, 0],
            &[8, 229, 37, 0, 0, 0, 198, 77, 0],
            &[15, 57, 0, 0, 0, 0, 30, 42, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '_' / 0x5f
        '_' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[246, 246, 246, 246, 246, 246, 246, 246, 98],
            &[39, 39, 39, 39, 39, 39, 39, 39, 15],
        ],
        // letter: '`' / 0x60
        '`' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 183, 58, 0, 0, 0, 0],
            &[0, 0, 0, 112, 210, 6, 0, 0, 0],
            &[0, 0, 0, 0, 86, 49, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'a' / 0x61
        'a' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 92, 141, 153, 107, 10, 0, 0],
            &[0, 67, 187, 118, 109, 204, 185, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 16, 0],
            &[0, 0, 46, 110, 133, 142, 255, 35, 0],
            &[0, 124, 240, 142, 103, 90, 255, 35, 0],
            &[0, 243, 77, 0, 0, 22, 255, 35, 0],
            &[0, 235, 100, 0, 16, 163, 255, 35, 0],
            &[0, 93, 242, 235, 235, 94, 239, 35, 0],
            &[0, 0, 5, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'b' / 0x62
        'b' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 144, 55, 0, 0, 0, 0, 0, 0],
            &[0, 225, 86, 0, 0, 0, 0, 0, 0],
            &[0, 225, 86, 0, 0, 0, 0, 0, 0],
            &[0, 225, 86, 106, 155, 117, 19, 0, 0],
            &[0, 225, 219, 164, 111, 199, 215, 9, 0],
            &[0, 225, 178, 0, 0, 11, 240, 95, 0],
            &[0, 225, 106, 0, 0, 0, 186, 147, 0],
            &[0, 225, 95, 0, 0, 0, 176, 156, 0],
            &[0, 225, 131, 0, 0, 0, 210, 127, 0],
            &[0, 225, 230, 36, 0, 84, 253, 45, 0],
            &[0, 225, 107, 227, 239, 242, 105, 0, 0],
            &[0, 0, 0, 3, 26, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'c' / 0x63
        'c' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 110, 155, 146, 87, 1, 0],
            &[0, 20, 222, 207, 126, 133, 172, 0, 0],
            &[0, 132, 216, 6, 0, 0, 0, 0, 0],
            &[0, 188, 140, 0, 0, 0, 0, 0, 0],
            &[0, 200, 127, 0, 0, 0, 0, 0, 0],
            &[0, 173, 164, 0, 0, 0, 0, 0, 0],
            &[0, 85, 248, 77, 1, 5, 56, 0, 0],
            &[0, 0, 123, 240, 249, 253, 218, 0, 0],
            &[0, 0, 0, 4, 28, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'd' / 0x64
        'd' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 158, 45, 0],
            &[0, 0, 0, 0, 0, 0, 247, 71, 0],
            &[0, 0, 0, 0, 0, 0, 247, 71, 0],
            &[0, 0, 81, 151, 143, 43, 244, 71, 0],
            &[0, 124, 247, 134, 115, 212, 245, 71, 0],
            &[7, 244, 109, 0, 0, 71, 255, 71, 0],
            &[44, 255, 36, 0, 0, 15, 255, 71, 0],
            &[53, 255, 22, 0, 0, 0, 250, 71, 0],
            &[25, 255, 53, 0, 0, 25, 255, 71, 0],
            &[0, 198, 177, 8, 3, 152, 255, 71, 0],
            &[0, 38, 210, 248, 242, 139, 211, 71, 0],
            &[0, 0, 0, 18, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'e' / 0x65
        'e' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 37, 131, 156, 112, 11, 0, 0],
            &[0, 62, 242, 152, 111, 196, 201, 5, 0],
            &[0, 210, 131, 0, 0, 9, 235, 85, 0],
            &[20, 255, 159, 133, 133, 133, 225, 133, 0],
            &[33, 255, 143, 124, 124, 124, 124, 69, 0],
            &[7, 246, 74, 0, 0, 0, 0, 0, 0],
            &[0, 147, 210, 38, 0, 3, 62, 26, 0],
            &[0, 7, 144, 243, 236, 250, 213, 36, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'f' / 0x66
        'f' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 97, 168, 173, 130, 0],
            &[0, 0, 0, 96, 247, 116, 90, 93, 0],
            &[0, 0, 0, 154, 174, 0, 0, 0, 0],
            &[6, 93, 112, 207, 209, 128, 128, 59, 0],
            &[10, 139, 139, 212, 213, 139, 139, 64, 0],
            &[0, 0, 0, 160, 162, 0, 0, 0, 0],
            &[0, 0, 0, 160, 162, 0, 0, 0, 0],
            &[0, 0, 0, 160, 162, 0, 0, 0, 0],
            &[0, 0, 0, 160, 162, 0, 0, 0, 0],
            &[0, 0, 0, 160, 162, 0, 0, 0, 0],
            &[0, 0, 0, 160, 162, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'g' / 0x67
        'g' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 80, 155, 143, 41, 99, 36, 0],
            &[0, 112, 245, 130, 113, 206, 240, 71, 0],
            &[5, 239, 102, 0, 0, 45, 255, 71, 0],
            &[43, 255, 34, 0, 0, 0, 234, 71, 0],
            &[54, 255, 23, 0, 0, 0, 222, 71, 0],
            &[26, 255, 59, 0, 0, 4, 246, 71, 0],
            &[0, 196, 185, 11, 0, 125, 255, 71, 0],
            &[0, 34, 207, 250, 240, 174, 230, 71, 0],
            &[0, 0, 0, 24, 11, 0, 240, 63, 0],
            &[0, 16, 0, 0, 0, 62, 252, 22, 0],
            &[0, 10, 70, 100, 96, 39, 0, 0, 0],
        ],
        // letter: 'h' / 0x68
        'h' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 144, 62, 0, 0, 0, 0, 0, 0],
            &[0, 225, 97, 0, 0, 0, 0, 0, 0],
            &[0, 225, 97, 0, 0, 0, 0, 0, 0],
            &[0, 225, 96, 83, 152, 132, 28, 0, 0],
            &[0, 225, 197, 168, 107, 193, 227, 12, 0],
            &[0, 225, 186, 0, 0, 17, 253, 67, 0],
            &[0, 225, 111, 0, 0, 0, 234, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'i' / 0x69
        'i' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 48, 136, 2, 0, 0, 0],
            &[0, 0, 0, 116, 249, 19, 0, 0, 0],
            &[0, 0, 0, 0, 13, 0, 0, 0, 0],
            &[0, 84, 128, 128, 128, 6, 0, 0, 0],
            &[0, 65, 115, 157, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 56, 255, 17, 0, 0, 0],
            &[0, 196, 242, 255, 255, 254, 237, 123, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'j' / 0x6a
        'j' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 125, 60, 0, 0],
            &[0, 0, 0, 0, 4, 239, 137, 0, 0],
            &[0, 0, 0, 0, 0, 12, 1, 0, 0],
            &[0, 0, 19, 128, 128, 128, 71, 0, 0],
            &[0, 0, 14, 105, 124, 220, 142, 0, 0],
            &[0, 0, 0, 0, 0, 179, 142, 0, 0],
            &[0, 0, 0, 0, 0, 179, 142, 0, 0],
            &[0, 0, 0, 0, 0, 179, 142, 0, 0],
            &[0, 0, 0, 0, 0, 179, 142, 0, 0],
            &[0, 0, 0, 0, 0, 179, 142, 0, 0],
            &[0, 0, 0, 0, 0, 179, 142, 0, 0],
            &[0, 0, 0, 0, 0, 184, 141, 0, 0],
            &[0, 0, 0, 0, 10, 229, 109, 0, 0],
            &[0, 8, 79, 103, 79, 8, 0, 0, 0],
        ],
        // letter: 'k' / 0x6b
        'k' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 86, 119, 0, 0, 0, 0, 0, 0],
            &[0, 135, 186, 0, 0, 0, 0, 0, 0],
            &[0, 135, 186, 0, 0, 0, 0, 0, 0],
            &[0, 135, 186, 0, 0, 4, 118, 71, 0],
            &[0, 135, 186, 0, 1, 161, 205, 14, 0],
            &[0, 135, 186, 0, 137, 217, 21, 0, 0],
            &[0, 135, 178, 110, 243, 31, 0, 0, 0],
            &[0, 135, 221, 210, 234, 95, 0, 0, 0],
            &[0, 135, 223, 21, 75, 244, 41, 0, 0],
            &[0, 135, 186, 0, 0, 144, 211, 9, 0],
            &[0, 135, 186, 0, 0, 7, 207, 155, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'l' / 0x6c
        'l' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 163, 163, 163, 20, 0, 0, 0],
            &[0, 51, 81, 118, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 38, 255, 35, 0, 0, 0],
            &[0, 196, 241, 255, 255, 254, 238, 123, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'm' / 0x6d
        'm' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[38, 80, 104, 139, 20, 108, 145, 15, 0],
            &[76, 238, 172, 215, 218, 173, 216, 126, 0],
            &[76, 246, 9, 106, 243, 9, 117, 168, 0],
            &[76, 221, 0, 89, 213, 0, 104, 182, 0],
            &[76, 210, 0, 88, 198, 0, 103, 182, 0],
            &[76, 209, 0, 88, 198, 0, 103, 182, 0],
            &[76, 209, 0, 88, 198, 0, 103, 182, 0],
            &[76, 209, 0, 88, 198, 0, 103, 182, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'n' / 0x6e
        'n' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 113, 24, 92, 152, 136, 34, 0, 0],
            &[0, 225, 199, 163, 107, 181, 232, 15, 0],
            &[0, 225, 177, 0, 0, 10, 251, 71, 0],
            &[0, 225, 113, 0, 0, 0, 233, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'o' / 0x6f
        'o' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 56, 142, 154, 97, 4, 0, 0],
            &[0, 95, 245, 136, 109, 207, 185, 2, 0],
            &[4, 235, 102, 0, 0, 16, 240, 82, 0],
            &[41, 255, 25, 0, 0, 0, 182, 142, 0],
            &[52, 255, 10, 0, 0, 0, 170, 155, 0],
            &[20, 254, 51, 0, 0, 0, 210, 123, 0],
            &[0, 177, 185, 13, 0, 99, 248, 34, 0],
            &[0, 20, 180, 246, 237, 229, 79, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'p' / 0x70
        'p' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 113, 29, 104, 156, 127, 20, 0, 0],
            &[0, 225, 212, 152, 110, 194, 217, 10, 0],
            &[0, 225, 163, 0, 0, 8, 236, 97, 0],
            &[0, 225, 106, 0, 0, 0, 185, 148, 0],
            &[0, 225, 100, 0, 0, 0, 176, 154, 0],
            &[0, 225, 133, 0, 0, 0, 209, 126, 0],
            &[0, 225, 227, 30, 0, 81, 253, 45, 0],
            &[0, 225, 142, 223, 239, 243, 106, 0, 0],
            &[0, 225, 93, 1, 25, 7, 0, 0, 0],
            &[0, 225, 93, 0, 0, 0, 0, 0, 0],
            &[0, 81, 33, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'q' / 0x71
        'q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 76, 154, 139, 30, 99, 36, 0],
            &[0, 112, 242, 127, 120, 203, 238, 71, 0],
            &[5, 239, 99, 0, 0, 72, 255, 71, 0],
            &[44, 255, 33, 0, 0, 9, 255, 71, 0],
            &[53, 255, 22, 0, 0, 0, 250, 71, 0],
            &[25, 255, 59, 0, 0, 23, 255, 71, 0],
            &[0, 194, 184, 9, 2, 148, 255, 71, 0],
            &[0, 34, 206, 248, 240, 128, 241, 71, 0],
            &[0, 0, 0, 18, 14, 0, 247, 71, 0],
            &[0, 0, 0, 0, 0, 0, 247, 71, 0],
            &[0, 0, 0, 0, 0, 0, 88, 25, 0],
        ],
        // letter: 'r' / 0x72
        'r' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 117, 128, 128, 24, 94, 157, 121, 0],
            &[0, 93, 125, 238, 169, 204, 127, 132, 0],
            &[0, 0, 0, 217, 220, 9, 0, 0, 0],
            &[0, 0, 0, 217, 132, 0, 0, 0, 0],
            &[0, 0, 0, 217, 109, 0, 0, 0, 0],
            &[0, 0, 0, 217, 108, 0, 0, 0, 0],
            &[0, 0, 1, 219, 111, 0, 0, 0, 0],
            &[28, 229, 251, 255, 255, 242, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 's' / 0x73
        's' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 51, 138, 159, 136, 60, 0, 0],
            &[0, 46, 247, 143, 109, 137, 116, 0, 0],
            &[0, 93, 230, 3, 0, 0, 0, 0, 0],
            &[0, 19, 209, 204, 82, 2, 0, 0, 0],
            &[0, 0, 7, 104, 221, 219, 56, 0, 0],
            &[0, 0, 0, 0, 3, 150, 218, 0, 0],
            &[0, 53, 33, 0, 1, 139, 213, 0, 0],
            &[0, 101, 244, 244, 242, 224, 66, 0, 0],
            &[0, 0, 3, 25, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 't' / 0x74
        't' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 123, 11, 0, 0, 0, 0],
            &[0, 0, 12, 255, 22, 0, 0, 0, 0],
            &[18, 102, 146, 255, 140, 128, 128, 1, 0],
            &[30, 139, 159, 255, 149, 139, 139, 1, 0],
            &[0, 0, 44, 255, 22, 0, 0, 0, 0],
            &[0, 0, 44, 255, 22, 0, 0, 0, 0],
            &[0, 0, 44, 255, 22, 0, 0, 0, 0],
            &[0, 0, 42, 255, 28, 0, 0, 0, 0],
            &[0, 0, 17, 251, 106, 0, 1, 4, 0],
            &[0, 0, 0, 116, 245, 238, 244, 53, 0],
            &[0, 0, 0, 0, 1, 26, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'u' / 0x75
        'u' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 39, 0, 0, 0, 126, 36, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 3, 253, 71, 0],
            &[0, 238, 86, 0, 0, 33, 255, 71, 0],
            &[0, 200, 166, 2, 4, 147, 255, 71, 0],
            &[0, 62, 232, 242, 236, 130, 209, 71, 0],
            &[0, 0, 1, 26, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'v' / 0x76
        'v' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[35, 126, 3, 0, 0, 0, 83, 85, 0],
            &[11, 243, 69, 0, 0, 2, 229, 97, 0],
            &[0, 161, 159, 0, 0, 63, 244, 13, 0],
            &[0, 66, 241, 8, 0, 150, 162, 0, 0],
            &[0, 1, 226, 83, 4, 234, 67, 0, 0],
            &[0, 0, 133, 173, 71, 225, 1, 0, 0],
            &[0, 0, 39, 247, 176, 131, 0, 0, 0],
            &[0, 0, 0, 200, 255, 37, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'w' / 0x77
        'w' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[94, 56, 0, 0, 0, 0, 1, 127, 19],
            &[157, 140, 0, 11, 20, 0, 29, 253, 8],
            &[116, 177, 0, 171, 252, 16, 70, 220, 0],
            &[75, 215, 4, 239, 196, 83, 111, 178, 0],
            &[34, 249, 63, 210, 114, 153, 149, 136, 0],
            &[2, 246, 148, 140, 49, 218, 179, 94, 0],
            &[0, 207, 217, 73, 2, 237, 217, 52, 0],
            &[0, 166, 250, 12, 0, 174, 254, 11, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'x' / 0x78
        'x' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[9, 125, 47, 0, 0, 3, 120, 57, 0],
            &[0, 125, 218, 12, 0, 128, 215, 11, 0],
            &[0, 2, 190, 158, 59, 243, 46, 0, 0],
            &[0, 0, 25, 234, 238, 103, 0, 0, 0],
            &[0, 0, 3, 200, 253, 49, 0, 0, 0],
            &[0, 0, 132, 209, 124, 218, 12, 0, 0],
            &[0, 68, 244, 42, 1, 186, 164, 0, 0],
            &[23, 232, 103, 0, 0, 25, 235, 98, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'y' / 0x79
        'y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[34, 122, 1, 0, 0, 0, 79, 85, 0],
            &[7, 235, 69, 0, 0, 0, 221, 98, 0],
            &[0, 137, 173, 0, 0, 51, 245, 13, 0],
            &[0, 33, 248, 25, 0, 137, 163, 0, 0],
            &[0, 0, 180, 125, 0, 222, 68, 0, 0],
            &[0, 0, 73, 222, 51, 227, 2, 0, 0],
            &[0, 0, 2, 221, 183, 134, 0, 0, 0],
            &[0, 0, 0, 116, 255, 40, 0, 0, 0],
            &[0, 0, 0, 100, 200, 0, 0, 0, 0],
            &[0, 0, 3, 212, 94, 0, 0, 0, 0],
            &[8, 101, 89, 11, 0, 0, 0, 0, 0],
        ],
        // letter: 'z' / 0x7a
        'z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 128, 128, 128, 128, 128, 3, 0],
            &[0, 61, 118, 118, 118, 192, 233, 3, 0],
            &[0, 0, 0, 0, 45, 241, 64, 0, 0],
            &[0, 0, 0, 14, 218, 115, 0, 0, 0],
            &[0, 0, 0, 174, 170, 0, 0, 0, 0],
            &[0, 0, 118, 213, 12, 0, 0, 0, 0],
            &[0, 64, 242, 55, 16, 16, 16, 2, 0],
            &[0, 187, 255, 255, 255, 255, 255, 35, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '{' / 0x7b
        '{' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 110, 224, 110, 0, 0],
            &[0, 0, 0, 48, 255, 78, 5, 0, 0],
            &[0, 0, 0, 84, 230, 0, 0, 0, 0],
            &[0, 0, 0, 84, 226, 0, 0, 0, 0],
            &[0, 0, 0, 97, 218, 0, 0, 0, 0],
            &[0, 4, 124, 219, 104, 0, 0, 0, 0],
            &[0, 6, 174, 209, 70, 0, 0, 0, 0],
            &[0, 0, 0, 113, 211, 0, 0, 0, 0],
            &[0, 0, 0, 84, 226, 0, 0, 0, 0],
            &[0, 0, 0, 84, 226, 0, 0, 0, 0],
            &[0, 0, 0, 64, 248, 25, 0, 0, 0],
            &[0, 0, 0, 2, 169, 244, 92, 0, 0],
            &[0, 0, 0, 0, 0, 24, 21, 0, 0],
        ],
        // letter: '|' / 0x7c
        '|' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 49, 117, 0, 0, 0, 0],
            &[0, 0, 0, 77, 183, 0, 0, 0, 0],
            &[0, 0, 0, 77, 183, 0, 0, 0, 0],
            &[0, 0, 0, 77, 183, 0, 0, 0, 0],
            &[0, 0, 0, 77, 183, 0, 0, 0, 0],
            &[0, 0, 0, 77, 183, 0, 0, 0, 0],
            &[0, 0, 0, 77, 183, 0, 0, 0, 0],
            &[0, 0, 0, 77, 183, 0, 0, 0, 0],
            &[0, 0, 0, 77, 183, 0, 0, 0, 0],
            &[0, 0, 0, 77, 183, 0, 0, 0, 0],
            &[0, 0, 0, 77, 183, 0, 0, 0, 0],
            &[0, 0, 0, 77, 183, 0, 0, 0, 0],
            &[0, 0, 0, 77, 183, 0, 0, 0, 0],
            &[0, 0, 0, 28, 66, 0, 0, 0, 0],
        ],
        // letter: '}' / 0x7d
        '}' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 241, 174, 22, 0, 0, 0, 0],
            &[0, 0, 27, 197, 151, 0, 0, 0, 0],
            &[0, 0, 0, 126, 186, 0, 0, 0, 0],
            &[0, 0, 0, 124, 187, 0, 0, 0, 0],
            &[0, 0, 0, 117, 199, 0, 0, 0, 0],
            &[0, 0, 0, 35, 223, 149, 47, 0, 0],
            &[0, 0, 0, 13, 164, 207, 73, 0, 0],
            &[0, 0, 0, 107, 214, 2, 0, 0, 0],
            &[0, 0, 0, 124, 187, 0, 0, 0, 0],
            &[0, 0, 0, 124, 187, 0, 0, 0, 0],
            &[0, 0, 2, 180, 166, 0, 0, 0, 0],
            &[0, 8, 230, 225, 45, 0, 0, 0, 0],
            &[0, 2, 41, 2, 0, 0, 0, 0, 0],
        ],
        // letter: '~' / 0x7e
        '~' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 98, 64, 0, 0, 66, 35, 0],
            &[0, 151, 146, 185, 138, 4, 194, 33, 0],
            &[0, 212, 3, 3, 137, 224, 143, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ' ' / 0xa0
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¡' / 0xa1
        '¡' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 46, 99, 0, 0, 0, 0],
            &[0, 0, 0, 148, 247, 3, 0, 0, 0],
            &[0, 0, 0, 24, 57, 0, 0, 0, 0],
            &[0, 0, 0, 52, 109, 0, 0, 0, 0],
            &[0, 0, 0, 97, 195, 0, 0, 0, 0],
            &[0, 0, 0, 105, 203, 0, 0, 0, 0],
            &[0, 0, 0, 113, 211, 0, 0, 0, 0],
            &[0, 0, 0, 121, 219, 0, 0, 0, 0],
            &[0, 0, 0, 129, 227, 0, 0, 0, 0],
            &[0, 0, 0, 137, 235, 0, 0, 0, 0],
            &[0, 0, 0, 84, 142, 0, 0, 0, 0],
        ],
        // letter: '¢' / 0xa2
        '¢' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 30, 2, 0, 0, 0],
            &[0, 0, 0, 0, 222, 21, 0, 0, 0],
            &[0, 0, 65, 200, 254, 246, 201, 7, 0],
            &[0, 42, 249, 119, 24, 29, 66, 0, 0],
            &[0, 142, 190, 0, 0, 0, 0, 0, 0],
            &[0, 180, 141, 0, 0, 0, 0, 0, 0],
            &[0, 177, 145, 0, 0, 0, 0, 0, 0],
            &[0, 133, 209, 1, 0, 0, 0, 0, 0],
            &[0, 28, 237, 175, 86, 92, 142, 2, 0],
            &[0, 0, 33, 146, 246, 190, 135, 1, 0],
            &[0, 0, 0, 0, 222, 17, 0, 0, 0],
            &[0, 0, 0, 0, 31, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '£' / 0xa3
        '£' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 18, 0, 0, 0],
            &[0, 0, 0, 137, 250, 247, 220, 16, 0],
            &[0, 0, 59, 248, 40, 1, 53, 0, 0],
            &[0, 0, 106, 209, 0, 0, 0, 0, 0],
            &[0, 0, 112, 203, 0, 0, 0, 0, 0],
            &[0, 72, 170, 224, 103, 103, 28, 0, 0],
            &[0, 92, 186, 230, 133, 133, 36, 0, 0],
            &[0, 0, 112, 203, 0, 0, 0, 0, 0],
            &[0, 0, 133, 178, 0, 0, 0, 0, 0],
            &[0, 41, 227, 95, 30, 30, 30, 10, 0],
            &[0, 189, 255, 255, 255, 255, 255, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¤' / 0xa4
        '¤' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 0, 0, 0, 0, 16, 0, 0],
            &[0, 195, 142, 181, 201, 131, 225, 45, 0],
            &[0, 63, 233, 73, 41, 165, 172, 0, 0],
            &[0, 129, 123, 0, 0, 15, 235, 0, 0],
            &[0, 125, 129, 0, 0, 20, 235, 0, 0],
            &[0, 65, 243, 103, 72, 187, 174, 0, 0],
            &[0, 189, 118, 155, 177, 109, 215, 43, 0],
            &[0, 6, 0, 0, 0, 0, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¥' / 0xa5
        '¥' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[64, 243, 20, 0, 0, 0, 159, 165, 0],
            &[0, 190, 134, 0, 0, 35, 247, 37, 0],
            &[0, 59, 241, 17, 0, 159, 154, 0, 0],
            &[0, 0, 184, 129, 35, 243, 28, 0, 0],
            &[0, 0, 53, 238, 172, 143, 0, 0, 0],
            &[0, 82, 190, 248, 255, 206, 158, 0, 0],
            &[0, 16, 38, 128, 209, 38, 31, 0, 0],
            &[0, 77, 179, 211, 239, 179, 149, 0, 0],
            &[0, 21, 49, 135, 212, 49, 41, 0, 0],
            &[0, 0, 0, 106, 201, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¦' / 0xa6
        '¦' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 52, 115, 0, 0, 0, 0],
            &[0, 0, 0, 81, 180, 0, 0, 0, 0],
            &[0, 0, 0, 81, 180, 0, 0, 0, 0],
            &[0, 0, 0, 81, 180, 0, 0, 0, 0],
            &[0, 0, 0, 81, 180, 0, 0, 0, 0],
            &[0, 0, 0, 56, 125, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 172, 0, 0, 0, 0],
            &[0, 0, 0, 81, 180, 0, 0, 0, 0],
            &[0, 0, 0, 81, 180, 0, 0, 0, 0],
            &[0, 0, 0, 81, 180, 0, 0, 0, 0],
            &[0, 0, 0, 30, 67, 0, 0, 0, 0],
        ],
        // letter: '§' / 0xa7
        '§' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 65, 156, 174, 150, 65, 0, 0],
            &[0, 41, 247, 89, 57, 90, 75, 0, 0],
            &[0, 48, 242, 36, 0, 0, 0, 0, 0],
            &[0, 0, 148, 248, 158, 41, 0, 0, 0],
            &[0, 29, 221, 74, 119, 234, 91, 0, 0],
            &[0, 78, 186, 0, 0, 73, 203, 0, 0],
            &[0, 19, 226, 145, 27, 128, 164, 0, 0],
            &[0, 0, 23, 139, 238, 245, 27, 0, 0],
            &[0, 0, 0, 0, 15, 175, 167, 0, 0],
            &[0, 31, 9, 0, 0, 123, 193, 0, 0],
            &[0, 92, 237, 204, 210, 215, 55, 0, 0],
            &[0, 0, 5, 27, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¨' / 0xa8
        '¨' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 7, 0, 48, 19, 0, 0],
            &[0, 16, 255, 71, 0, 218, 124, 0, 0],
            &[0, 0, 48, 4, 0, 38, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '©' / 0xa9
        '©' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 56, 137, 150, 103, 10, 0, 0],
            &[0, 116, 148, 61, 84, 104, 191, 14, 0],
            &[27, 174, 46, 195, 125, 73, 79, 128, 0],
            &[84, 100, 143, 65, 0, 0, 3, 184, 0],
            &[83, 105, 144, 64, 0, 0, 4, 181, 0],
            &[24, 185, 48, 195, 133, 65, 85, 117, 0],
            &[0, 107, 164, 67, 81, 110, 177, 7, 0],
            &[0, 0, 43, 118, 129, 80, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ª' / 0xaa
        'ª' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 16, 0, 0, 0, 0],
            &[0, 0, 89, 183, 189, 161, 0, 0, 0],
            &[0, 0, 0, 21, 57, 234, 0, 0, 0],
            &[0, 0, 130, 167, 106, 236, 0, 0, 0],
            &[0, 0, 208, 63, 69, 249, 0, 0, 0],
            &[0, 0, 60, 152, 97, 105, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '«' / 0xab
        '«' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 98, 22, 6, 111, 4, 0],
            &[0, 0, 90, 226, 23, 158, 176, 1, 0],
            &[0, 51, 238, 52, 113, 214, 13, 0, 0],
            &[0, 150, 190, 2, 220, 122, 0, 0, 0],
            &[0, 10, 206, 118, 45, 236, 57, 0, 0],
            &[0, 0, 29, 229, 42, 78, 221, 5, 0],
            &[0, 0, 0, 19, 0, 0, 21, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¬' / 0xac
        '¬' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 105, 110, 110, 110, 110, 110, 23, 0],
            &[0, 136, 143, 143, 143, 143, 231, 53, 0],
            &[0, 0, 0, 0, 0, 0, 200, 53, 0],
            &[0, 0, 0, 0, 0, 0, 200, 53, 0],
            &[0, 0, 0, 0, 0, 0, 41, 11, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '®' / 0xae
        '®' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 56, 137, 150, 103, 10, 0, 0],
            &[0, 116, 151, 97, 70, 93, 191, 14, 0],
            &[27, 174, 15, 204, 146, 136, 79, 128, 0],
            &[84, 100, 15, 182, 87, 144, 3, 184, 0],
            &[83, 105, 15, 207, 196, 40, 4, 181, 0],
            &[24, 185, 16, 167, 39, 174, 85, 117, 0],
            &[0, 107, 165, 72, 38, 125, 178, 7, 0],
            &[0, 0, 43, 118, 129, 80, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¯' / 0xaf
        '¯' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 144, 165, 165, 165, 47, 0, 0],
            &[0, 0, 77, 89, 89, 89, 25, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '°' / 0xb0
        '°' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 15, 0, 0, 0, 0],
            &[0, 0, 66, 216, 197, 154, 1, 0, 0],
            &[0, 0, 209, 21, 0, 168, 58, 0, 0],
            &[0, 0, 199, 26, 0, 170, 48, 0, 0],
            &[0, 0, 65, 218, 202, 153, 1, 0, 0],
            &[0, 0, 0, 6, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '±' / 0xb1
        '±' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 40, 0, 0, 0, 0],
            &[0, 0, 0, 84, 172, 0, 0, 0, 0],
            &[0, 0, 0, 84, 172, 0, 0, 0, 0],
            &[0, 113, 121, 165, 212, 121, 121, 33, 0],
            &[0, 124, 133, 173, 216, 133, 133, 37, 0],
            &[0, 0, 0, 84, 172, 0, 0, 0, 0],
            &[0, 0, 0, 84, 172, 0, 0, 0, 0],
            &[0, 0, 0, 22, 45, 0, 0, 0, 0],
            &[0, 223, 254, 254, 254, 254, 254, 71, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '²' / 0xb2
        '²' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 77, 200, 216, 134, 0, 0, 0],
            &[0, 0, 48, 19, 6, 231, 26, 0, 0],
            &[0, 0, 0, 0, 24, 221, 5, 0, 0],
            &[0, 0, 0, 17, 197, 68, 0, 0, 0],
            &[0, 0, 19, 196, 70, 0, 0, 0, 0],
            &[0, 0, 189, 220, 177, 177, 65, 0, 0],
            &[0, 0, 37, 45, 45, 45, 16, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '³' / 0xb3
        '³' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 203, 215, 121, 0, 0, 0],
            &[0, 0, 38, 12, 22, 249, 2, 0, 0],
            &[0, 0, 1, 80, 137, 171, 0, 0, 0],
            &[0, 0, 2, 119, 160, 182, 8, 0, 0],
            &[0, 0, 0, 0, 0, 220, 47, 0, 0],
            &[0, 0, 171, 147, 162, 204, 6, 0, 0],
            &[0, 0, 13, 63, 54, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '´' / 0xb4
        '´' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 79, 178, 19, 0, 0, 0],
            &[0, 0, 16, 226, 87, 0, 0, 0, 0],
            &[0, 0, 62, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'µ' / 0xb5
        'µ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 39, 0, 0, 0, 126, 36, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 255, 93, 0, 0, 3, 253, 71, 0],
            &[0, 255, 166, 0, 0, 33, 255, 71, 0],
            &[0, 255, 246, 2, 4, 147, 255, 71, 0],
            &[0, 255, 255, 242, 236, 130, 209, 71, 0],
            &[0, 243, 81, 26, 10, 0, 0, 0, 0],
            &[0, 243, 79, 0, 0, 0, 0, 0, 0],
            &[0, 87, 28, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¶' / 0xb6
        '¶' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 117, 159, 163, 163, 163, 89, 0],
            &[0, 190, 255, 255, 255, 103, 95, 139, 0],
            &[36, 255, 255, 255, 255, 78, 68, 139, 0],
            &[69, 255, 255, 255, 255, 78, 68, 139, 0],
            &[65, 255, 255, 255, 255, 78, 68, 139, 0],
            &[22, 254, 255, 255, 255, 78, 68, 139, 0],
            &[0, 138, 255, 255, 255, 78, 68, 139, 0],
            &[0, 0, 41, 72, 157, 78, 68, 139, 0],
            &[0, 0, 0, 0, 129, 78, 68, 139, 0],
            &[0, 0, 0, 0, 129, 78, 68, 139, 0],
            &[0, 0, 0, 0, 129, 78, 68, 139, 0],
            &[0, 0, 0, 0, 129, 78, 68, 139, 0],
            &[0, 0, 0, 0, 100, 60, 52, 108, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '·' / 0xb7
        '·' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 71, 141, 0, 0, 0, 0],
            &[0, 0, 0, 142, 241, 3, 0, 0, 0],
            &[0, 0, 0, 5, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¸' / 0xb8
        '¸' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 12, 0, 0, 0, 0],
            &[0, 0, 0, 96, 137, 0, 0, 0, 0],
            &[0, 0, 0, 51, 197, 101, 0, 0, 0],
            &[0, 0, 16, 87, 50, 0, 0, 0, 0],
        ],
        // letter: '¹' / 0xb9
        '¹' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 145, 182, 0, 0, 0, 0],
            &[0, 0, 168, 126, 223, 0, 0, 0, 0],
            &[0, 0, 0, 9, 223, 0, 0, 0, 0],
            &[0, 0, 0, 9, 223, 0, 0, 0, 0],
            &[0, 0, 0, 9, 223, 0, 0, 0, 0],
            &[0, 0, 68, 163, 255, 151, 52, 0, 0],
            &[0, 0, 21, 46, 84, 45, 16, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'º' / 0xba
        'º' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 16, 0, 0, 0, 0],
            &[0, 0, 65, 205, 187, 154, 0, 0, 0],
            &[0, 0, 189, 39, 0, 200, 36, 0, 0],
            &[0, 0, 208, 13, 0, 174, 52, 0, 0],
            &[0, 0, 148, 112, 50, 214, 11, 0, 0],
            &[0, 0, 10, 123, 147, 47, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '»' / 0xbb
        '»' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 60, 0, 104, 20, 0, 0, 0],
            &[0, 69, 232, 33, 132, 192, 5, 0, 0],
            &[0, 0, 118, 211, 15, 184, 153, 0, 0],
            &[0, 0, 26, 246, 71, 87, 237, 16, 0],
            &[0, 6, 199, 133, 37, 235, 67, 0, 0],
            &[0, 124, 176, 1, 193, 111, 0, 0, 0],
            &[0, 7, 11, 0, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¼' / 0xbc
        '¼' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[104, 236, 111, 0, 0, 7, 223, 40, 0],
            &[87, 141, 111, 0, 0, 110, 161, 0, 0],
            &[0, 121, 111, 0, 10, 227, 34, 0, 0],
            &[0, 121, 111, 0, 120, 153, 0, 0, 0],
            &[104, 255, 255, 122, 230, 28, 0, 0, 0],
            &[0, 0, 0, 129, 144, 11, 201, 69, 0],
            &[0, 0, 19, 231, 23, 155, 211, 78, 0],
            &[0, 0, 138, 136, 81, 155, 150, 78, 0],
            &[0, 25, 231, 18, 182, 207, 233, 217, 7],
            &[0, 148, 127, 0, 0, 0, 150, 78, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '½' / 0xbd
        '½' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[104, 236, 111, 0, 0, 7, 223, 40, 0],
            &[87, 141, 111, 0, 0, 110, 161, 0, 0],
            &[0, 121, 111, 0, 10, 227, 34, 0, 0],
            &[0, 121, 111, 0, 120, 153, 0, 0, 0],
            &[104, 255, 255, 122, 230, 28, 0, 0, 0],
            &[0, 0, 0, 129, 168, 193, 218, 119, 0],
            &[0, 0, 19, 231, 24, 28, 27, 223, 0],
            &[0, 0, 138, 136, 0, 0, 143, 129, 0],
            &[0, 25, 231, 18, 1, 146, 140, 0, 0],
            &[0, 148, 127, 0, 85, 253, 222, 222, 7],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¾' / 0xbe
        '¾' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 10, 0, 0, 0, 0, 0, 0],
            &[111, 196, 226, 69, 0, 7, 223, 40, 0],
            &[2, 11, 163, 114, 0, 110, 161, 0, 0],
            &[14, 195, 235, 59, 10, 227, 34, 0, 0],
            &[13, 0, 119, 156, 120, 153, 0, 0, 0],
            &[124, 209, 205, 67, 230, 28, 0, 0, 0],
            &[0, 0, 0, 129, 144, 11, 201, 69, 0],
            &[0, 0, 19, 231, 23, 155, 211, 78, 0],
            &[0, 0, 138, 136, 81, 155, 150, 78, 0],
            &[0, 25, 231, 18, 182, 207, 233, 217, 7],
            &[0, 148, 127, 0, 0, 0, 150, 78, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¿' / 0xbf
        '¿' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 104, 0, 0, 0],
            &[0, 0, 0, 0, 137, 252, 9, 0, 0],
            &[0, 0, 0, 0, 21, 60, 0, 0, 0],
            &[0, 0, 0, 0, 39, 122, 0, 0, 0],
            &[0, 0, 0, 0, 134, 177, 0, 0, 0],
            &[0, 0, 0, 129, 230, 42, 0, 0, 0],
            &[0, 0, 128, 212, 29, 0, 0, 0, 0],
            &[0, 1, 243, 65, 0, 0, 0, 0, 0],
            &[0, 9, 255, 48, 0, 0, 0, 0, 0],
            &[0, 0, 179, 211, 106, 113, 167, 135, 0],
            &[0, 0, 10, 117, 176, 178, 134, 44, 0],
        ],
        // letter: 'À' / 0xc0
        'À' => &[
            &[0, 0, 54, 245, 63, 0, 0, 0, 0],
            &[0, 0, 0, 68, 208, 12, 0, 0, 0],
            &[0, 0, 0, 4, 12, 1, 0, 0, 0],
            &[0, 0, 0, 165, 249, 15, 0, 0, 0],
            &[0, 0, 6, 241, 224, 91, 0, 0, 0],
            &[0, 0, 75, 224, 125, 174, 0, 0, 0],
            &[0, 0, 157, 146, 47, 246, 11, 0, 0],
            &[0, 4, 236, 67, 0, 223, 85, 0, 0],
            &[0, 67, 255, 52, 47, 182, 168, 0, 0],
            &[0, 150, 255, 239, 239, 255, 248, 8, 0],
            &[2, 231, 84, 0, 0, 4, 237, 78, 0],
            &[60, 248, 12, 0, 0, 0, 162, 161, 0],
            &[142, 181, 0, 0, 0, 0, 82, 239, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 49, 0, 0, 0, 0, 0],
        ],
        // letter: 'Á' / 0xc1
        'Á' => &[
            &[0, 0, 0, 0, 117, 224, 23, 0, 0],
            &[0, 0, 0, 38, 213, 36, 0, 0, 0],
            &[0, 0, 0, 6, 11, 0, 0, 0, 0],
            &[0, 0, 0, 165, 249, 15, 0, 0, 0],
            &[0, 0, 6, 241, 224, 91, 0, 0, 0],
            &[0, 0, 75, 224, 125, 174, 0, 0, 0],
            &[0, 0, 157, 146, 47, 246, 11, 0, 0],
            &[0, 4, 236, 67, 0, 223, 85, 0, 0],
            &[0, 67, 255, 52, 47, 182, 168, 0, 0],
            &[0, 150, 255, 239, 239, 255, 248, 8, 0],
            &[2, 231, 84, 0, 0, 4, 237, 78, 0],
            &[60, 248, 12, 0, 0, 0, 162, 161, 0],
            &[142, 181, 0, 0, 0, 0, 82, 239, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 55, 23, 0, 0],
        ],
        // letter: 'Â' / 0xc2
        'Â' => &[
            &[0, 0, 21, 223, 230, 86, 0, 0, 0],
            &[0, 9, 190, 84, 29, 201, 61, 0, 0],
            &[0, 1, 4, 4, 8, 1, 3, 0, 0],
            &[0, 0, 0, 165, 249, 15, 0, 0, 0],
            &[0, 0, 6, 241, 224, 91, 0, 0, 0],
            &[0, 0, 75, 224, 125, 174, 0, 0, 0],
            &[0, 0, 157, 146, 47, 246, 11, 0, 0],
            &[0, 4, 236, 67, 0, 223, 85, 0, 0],
            &[0, 67, 255, 52, 47, 182, 168, 0, 0],
            &[0, 150, 255, 239, 239, 255, 248, 8, 0],
            &[2, 231, 84, 0, 0, 4, 237, 78, 0],
            &[60, 248, 12, 0, 0, 0, 162, 161, 0],
            &[142, 181, 0, 0, 0, 0, 82, 239, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 49, 0, 0, 0, 0],
        ],
        // letter: 'Ã' / 0xc3
        'Ã' => &[
            &[0, 0, 127, 177, 67, 39, 126, 0, 0],
            &[0, 49, 179, 83, 203, 235, 57, 0, 0],
            &[0, 1, 2, 4, 8, 0, 0, 0, 0],
            &[0, 0, 0, 165, 249, 15, 0, 0, 0],
            &[0, 0, 6, 241, 224, 91, 0, 0, 0],
            &[0, 0, 75, 224, 125, 174, 0, 0, 0],
            &[0, 0, 157, 146, 47, 246, 11, 0, 0],
            &[0, 4, 236, 67, 0, 223, 85, 0, 0],
            &[0, 67, 255, 52, 47, 182, 168, 0, 0],
            &[0, 150, 255, 239, 239, 255, 248, 8, 0],
            &[2, 231, 84, 0, 0, 4, 237, 78, 0],
            &[60, 248, 12, 0, 0, 0, 162, 161, 0],
            &[142, 181, 0, 0, 0, 0, 82, 239, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ä' / 0xc4
        'Ä' => &[
            &[0, 10, 187, 41, 0, 160, 79, 0, 0],
            &[0, 8, 178, 37, 0, 151, 73, 0, 0],
            &[0, 0, 0, 4, 8, 0, 0, 0, 0],
            &[0, 0, 0, 165, 249, 15, 0, 0, 0],
            &[0, 0, 6, 241, 224, 91, 0, 0, 0],
            &[0, 0, 75, 224, 125, 174, 0, 0, 0],
            &[0, 0, 157, 146, 47, 246, 11, 0, 0],
            &[0, 4, 236, 67, 0, 223, 85, 0, 0],
            &[0, 67, 255, 52, 47, 182, 168, 0, 0],
            &[0, 150, 255, 239, 239, 255, 248, 8, 0],
            &[2, 231, 84, 0, 0, 4, 237, 78, 0],
            &[60, 248, 12, 0, 0, 0, 162, 161, 0],
            &[142, 181, 0, 0, 0, 0, 82, 239, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Å' / 0xc5
        'Å' => &[
            &[0, 0, 76, 138, 39, 177, 0, 0, 0],
            &[0, 0, 20, 200, 204, 89, 0, 0, 0],
            &[0, 0, 0, 4, 9, 0, 0, 0, 0],
            &[0, 0, 0, 165, 249, 15, 0, 0, 0],
            &[0, 0, 6, 241, 224, 91, 0, 0, 0],
            &[0, 0, 75, 224, 125, 174, 0, 0, 0],
            &[0, 0, 157, 146, 47, 246, 11, 0, 0],
            &[0, 4, 236, 67, 0, 223, 85, 0, 0],
            &[0, 67, 255, 52, 47, 182, 168, 0, 0],
            &[0, 150, 255, 239, 239, 255, 248, 8, 0],
            &[2, 231, 84, 0, 0, 4, 237, 78, 0],
            &[60, 248, 12, 0, 0, 0, 162, 161, 0],
            &[142, 181, 0, 0, 0, 0, 82, 239, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 184, 196, 74, 0, 0, 0],
        ],
        // letter: 'Æ' / 0xc6
        'Æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 150, 254, 254, 254, 254, 153, 0],
            &[0, 0, 208, 125, 255, 50, 31, 19, 0],
            &[0, 13, 252, 55, 255, 20, 0, 0, 0],
            &[0, 67, 225, 31, 255, 20, 0, 0, 0],
            &[0, 124, 171, 31, 255, 204, 200, 67, 0],
            &[0, 182, 153, 68, 255, 97, 82, 27, 0],
            &[1, 255, 255, 255, 255, 20, 0, 0, 0],
            &[40, 252, 11, 31, 255, 20, 0, 0, 0],
            &[98, 209, 0, 31, 255, 49, 30, 18, 0],
            &[155, 154, 0, 31, 255, 255, 255, 154, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ç' / 0xc7
        'Ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 24, 1, 0, 0],
            &[0, 0, 57, 206, 255, 255, 242, 103, 0],
            &[0, 41, 245, 138, 20, 7, 54, 38, 0],
            &[0, 166, 184, 0, 0, 0, 0, 0, 0],
            &[0, 236, 96, 0, 0, 0, 0, 0, 0],
            &[11, 255, 61, 0, 0, 0, 0, 0, 0],
            &[9, 255, 62, 0, 0, 0, 0, 0, 0],
            &[0, 238, 94, 0, 0, 0, 0, 0, 0],
            &[0, 171, 178, 0, 0, 0, 0, 0, 0],
            &[0, 50, 249, 137, 28, 15, 58, 28, 0],
            &[0, 0, 70, 214, 255, 255, 239, 52, 0],
            &[0, 0, 0, 0, 197, 80, 3, 0, 0],
            &[0, 0, 0, 0, 89, 232, 28, 0, 0],
            &[0, 0, 0, 45, 82, 25, 0, 0, 0],
        ],
        // letter: 'È' / 0xc8
        'È' => &[
            &[0, 0, 40, 239, 84, 0, 0, 0, 0],
            &[0, 0, 0, 54, 213, 21, 0, 0, 0],
            &[0, 0, 0, 0, 3, 1, 0, 0, 0],
            &[0, 170, 254, 254, 254, 254, 254, 53, 0],
            &[0, 171, 163, 31, 31, 31, 31, 6, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 233, 200, 200, 200, 188, 0, 0],
            &[0, 171, 186, 86, 86, 86, 81, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 163, 30, 30, 30, 30, 6, 0],
            &[0, 171, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 29, 53, 0, 0, 0, 0, 0],
        ],
        // letter: 'É' / 0xc9
        'É' => &[
            &[0, 0, 0, 0, 96, 234, 34, 0, 0],
            &[0, 0, 0, 25, 214, 48, 0, 0, 0],
            &[0, 0, 0, 1, 3, 0, 0, 0, 0],
            &[0, 170, 254, 254, 254, 254, 254, 53, 0],
            &[0, 171, 163, 31, 31, 31, 31, 6, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 233, 200, 200, 200, 188, 0, 0],
            &[0, 171, 186, 86, 86, 86, 81, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 163, 30, 30, 30, 30, 6, 0],
            &[0, 171, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 54, 28, 0, 0],
        ],
        // letter: 'Ê' / 0xca
        'Ê' => &[
            &[0, 0, 13, 212, 227, 108, 0, 0, 0],
            &[0, 3, 179, 101, 20, 194, 78, 0, 0],
            &[0, 0, 4, 0, 0, 1, 3, 0, 0],
            &[0, 170, 254, 254, 254, 254, 254, 53, 0],
            &[0, 171, 163, 31, 31, 31, 31, 6, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 233, 200, 200, 200, 188, 0, 0],
            &[0, 171, 186, 86, 86, 86, 81, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 163, 30, 30, 30, 30, 6, 0],
            &[0, 171, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 53, 0, 0, 0, 0],
        ],
        // letter: 'Ë' / 0xcb
        'Ë' => &[
            &[0, 2, 180, 56, 0, 143, 96, 0, 0],
            &[0, 1, 171, 51, 0, 135, 89, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 254, 254, 254, 254, 254, 53, 0],
            &[0, 171, 163, 31, 31, 31, 31, 6, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 233, 200, 200, 200, 188, 0, 0],
            &[0, 171, 186, 86, 86, 86, 81, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 163, 30, 30, 30, 30, 6, 0],
            &[0, 171, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ì' / 0xcc
        'Ì' => &[
            &[0, 0, 54, 245, 63, 0, 0, 0, 0],
            &[0, 0, 0, 68, 208, 12, 0, 0, 0],
            &[0, 0, 0, 0, 4, 1, 0, 0, 0],
            &[0, 104, 244, 254, 254, 251, 199, 0, 0],
            &[0, 0, 0, 117, 210, 2, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 117, 210, 1, 0, 0, 0],
            &[0, 104, 244, 255, 255, 251, 199, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 49, 0, 0, 0, 0, 0],
        ],
        // letter: 'Í' / 0xcd
        'Í' => &[
            &[0, 0, 0, 0, 117, 224, 23, 0, 0],
            &[0, 0, 0, 38, 213, 36, 0, 0, 0],
            &[0, 0, 0, 2, 2, 0, 0, 0, 0],
            &[0, 104, 244, 254, 254, 251, 199, 0, 0],
            &[0, 0, 0, 117, 210, 2, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 117, 210, 1, 0, 0, 0],
            &[0, 104, 244, 255, 255, 251, 199, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 55, 23, 0, 0],
        ],
        // letter: 'Î' / 0xce
        'Î' => &[
            &[0, 0, 21, 223, 230, 86, 0, 0, 0],
            &[0, 9, 190, 84, 29, 201, 61, 0, 0],
            &[0, 1, 4, 0, 0, 1, 3, 0, 0],
            &[0, 104, 244, 254, 254, 251, 199, 0, 0],
            &[0, 0, 0, 117, 210, 2, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 117, 210, 1, 0, 0, 0],
            &[0, 104, 244, 255, 255, 251, 199, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 49, 0, 0, 0, 0],
        ],
        // letter: 'Ï' / 0xcf
        'Ï' => &[
            &[0, 10, 187, 41, 0, 160, 79, 0, 0],
            &[0, 8, 178, 37, 0, 151, 73, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 104, 244, 254, 254, 251, 199, 0, 0],
            &[0, 0, 0, 117, 210, 2, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 117, 210, 1, 0, 0, 0],
            &[0, 104, 244, 255, 255, 251, 199, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ð' / 0xd0
        'Ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 224, 254, 251, 225, 145, 16, 0, 0],
            &[0, 225, 112, 29, 71, 200, 212, 9, 0],
            &[0, 225, 97, 0, 0, 12, 228, 117, 0],
            &[0, 225, 97, 0, 0, 0, 141, 194, 0],
            &[124, 255, 222, 124, 10, 0, 105, 223, 0],
            &[154, 255, 251, 154, 13, 0, 105, 219, 0],
            &[0, 225, 97, 0, 0, 0, 145, 187, 0],
            &[0, 225, 97, 0, 0, 16, 233, 105, 0],
            &[0, 225, 111, 30, 78, 208, 200, 5, 0],
            &[0, 225, 255, 248, 217, 132, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ñ' / 0xd1
        'Ñ' => &[
            &[0, 0, 114, 179, 76, 28, 138, 0, 0],
            &[0, 32, 191, 79, 195, 239, 70, 0, 0],
            &[0, 1, 2, 0, 0, 0, 0, 0, 0],
            &[5, 254, 218, 1, 0, 0, 203, 107, 0],
            &[5, 255, 222, 76, 0, 0, 204, 107, 0],
            &[5, 255, 130, 188, 0, 0, 204, 107, 0],
            &[5, 255, 53, 235, 45, 0, 204, 107, 0],
            &[5, 255, 50, 133, 156, 0, 204, 107, 0],
            &[5, 255, 50, 26, 241, 20, 203, 107, 0],
            &[5, 255, 50, 0, 164, 124, 199, 107, 0],
            &[5, 255, 50, 0, 52, 230, 193, 107, 0],
            &[5, 255, 50, 0, 0, 195, 247, 107, 0],
            &[5, 255, 50, 0, 0, 83, 255, 107, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ò' / 0xd2
        'Ò' => &[
            &[0, 0, 54, 245, 63, 0, 0, 0, 0],
            &[0, 0, 0, 68, 208, 12, 0, 0, 0],
            &[0, 0, 0, 16, 36, 4, 0, 0, 0],
            &[0, 12, 175, 255, 253, 228, 68, 0, 0],
            &[0, 157, 206, 31, 11, 128, 242, 21, 0],
            &[13, 250, 67, 0, 0, 0, 224, 113, 0],
            &[63, 253, 9, 0, 0, 0, 163, 168, 0],
            &[86, 240, 0, 0, 0, 0, 141, 191, 0],
            &[88, 241, 0, 0, 0, 0, 138, 190, 0],
            &[65, 254, 10, 0, 0, 0, 161, 166, 0],
            &[16, 251, 69, 0, 0, 0, 220, 111, 0],
            &[0, 160, 209, 36, 9, 129, 239, 19, 0],
            &[0, 13, 174, 254, 255, 222, 61, 0, 0],
            &[0, 0, 0, 15, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 49, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ó' / 0xd3
        'Ó' => &[
            &[0, 0, 0, 0, 117, 224, 23, 0, 0],
            &[0, 0, 0, 38, 213, 36, 0, 0, 0],
            &[0, 0, 0, 18, 35, 3, 0, 0, 0],
            &[0, 12, 175, 255, 253, 228, 68, 0, 0],
            &[0, 157, 206, 31, 11, 128, 242, 21, 0],
            &[13, 250, 67, 0, 0, 0, 224, 113, 0],
            &[63, 253, 9, 0, 0, 0, 163, 168, 0],
            &[86, 240, 0, 0, 0, 0, 141, 191, 0],
            &[88, 241, 0, 0, 0, 0, 138, 190, 0],
            &[65, 254, 10, 0, 0, 0, 161, 166, 0],
            &[16, 251, 69, 0, 0, 0, 220, 111, 0],
            &[0, 160, 209, 36, 9, 129, 239, 19, 0],
            &[0, 13, 174, 254, 255, 222, 61, 0, 0],
            &[0, 0, 0, 15, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 55, 23, 0, 0],
        ],
        // letter: 'Ô' / 0xd4
        'Ô' => &[
            &[0, 0, 21, 223, 230, 86, 0, 0, 0],
            &[0, 9, 190, 84, 29, 201, 61, 0, 0],
            &[0, 1, 4, 16, 32, 5, 3, 0, 0],
            &[0, 12, 175, 255, 253, 228, 68, 0, 0],
            &[0, 157, 206, 31, 11, 128, 242, 21, 0],
            &[13, 250, 67, 0, 0, 0, 224, 113, 0],
            &[63, 253, 9, 0, 0, 0, 163, 168, 0],
            &[86, 240, 0, 0, 0, 0, 141, 191, 0],
            &[88, 241, 0, 0, 0, 0, 138, 190, 0],
            &[65, 254, 10, 0, 0, 0, 161, 166, 0],
            &[16, 251, 69, 0, 0, 0, 220, 111, 0],
            &[0, 160, 209, 36, 9, 129, 239, 19, 0],
            &[0, 13, 174, 254, 255, 222, 61, 0, 0],
            &[0, 0, 0, 15, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 49, 0, 0, 0, 0],
        ],
        // letter: 'Õ' / 0xd5
        'Õ' => &[
            &[0, 0, 127, 177, 67, 39, 126, 0, 0],
            &[0, 49, 179, 83, 203, 235, 57, 0, 0],
            &[0, 1, 2, 16, 32, 3, 0, 0, 0],
            &[0, 12, 175, 255, 253, 228, 68, 0, 0],
            &[0, 157, 206, 31, 11, 128, 242, 21, 0],
            &[13, 250, 67, 0, 0, 0, 224, 113, 0],
            &[63, 253, 9, 0, 0, 0, 163, 168, 0],
            &[86, 240, 0, 0, 0, 0, 141, 191, 0],
            &[88, 241, 0, 0, 0, 0, 138, 190, 0],
            &[65, 254, 10, 0, 0, 0, 161, 166, 0],
            &[16, 251, 69, 0, 0, 0, 220, 111, 0],
            &[0, 160, 209, 36, 9, 129, 239, 19, 0],
            &[0, 13, 174, 254, 255, 222, 61, 0, 0],
            &[0, 0, 0, 15, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ö' / 0xd6
        'Ö' => &[
            &[0, 10, 187, 41, 0, 160, 79, 0, 0],
            &[0, 8, 178, 37, 0, 151, 73, 0, 0],
            &[0, 0, 0, 16, 32, 3, 0, 0, 0],
            &[0, 12, 175, 255, 253, 228, 68, 0, 0],
            &[0, 157, 206, 31, 11, 128, 242, 21, 0],
            &[13, 250, 67, 0, 0, 0, 224, 113, 0],
            &[63, 253, 9, 0, 0, 0, 163, 168, 0],
            &[86, 240, 0, 0, 0, 0, 141, 191, 0],
            &[88, 241, 0, 0, 0, 0, 138, 190, 0],
            &[65, 254, 10, 0, 0, 0, 161, 166, 0],
            &[16, 251, 69, 0, 0, 0, 220, 111, 0],
            &[0, 160, 209, 36, 9, 129, 239, 19, 0],
            &[0, 13, 174, 254, 255, 222, 61, 0, 0],
            &[0, 0, 0, 15, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '×' / 0xd7
        '×' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 124, 90, 0, 0, 22, 179, 12, 0],
            &[0, 39, 227, 88, 23, 212, 117, 0, 0],
            &[0, 0, 39, 227, 223, 117, 0, 0, 0],
            &[0, 0, 24, 216, 240, 90, 0, 0, 0],
            &[0, 22, 213, 118, 42, 228, 88, 0, 0],
            &[0, 135, 120, 0, 0, 39, 204, 11, 0],
            &[0, 1, 0, 0, 0, 0, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ø' / 0xd8
        'Ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 32, 3, 38, 33, 0],
            &[0, 12, 175, 255, 253, 229, 255, 106, 0],
            &[0, 157, 206, 31, 11, 213, 255, 25, 0],
            &[13, 250, 67, 0, 8, 223, 255, 113, 0],
            &[63, 253, 9, 0, 121, 180, 163, 168, 0],
            &[86, 240, 0, 24, 238, 39, 141, 191, 0],
            &[88, 241, 0, 158, 144, 0, 138, 190, 0],
            &[65, 254, 61, 234, 17, 0, 161, 166, 0],
            &[16, 251, 255, 108, 0, 0, 220, 111, 0],
            &[0, 246, 255, 40, 9, 129, 239, 19, 0],
            &[8, 237, 246, 254, 255, 222, 61, 0, 0],
            &[5, 81, 0, 15, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ù' / 0xd9
        'Ù' => &[
            &[0, 0, 54, 245, 63, 0, 0, 0, 0],
            &[0, 0, 0, 68, 208, 12, 0, 0, 0],
            &[0, 0, 0, 0, 4, 1, 0, 0, 0],
            &[5, 254, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[2, 255, 67, 0, 0, 0, 220, 104, 0],
            &[0, 234, 100, 0, 0, 3, 247, 80, 0],
            &[0, 159, 208, 29, 6, 125, 241, 16, 0],
            &[0, 21, 188, 255, 255, 228, 74, 0, 0],
            &[0, 0, 0, 16, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 49, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ú' / 0xda
        'Ú' => &[
            &[0, 0, 0, 0, 117, 224, 23, 0, 0],
            &[0, 0, 0, 38, 213, 36, 0, 0, 0],
            &[0, 0, 0, 2, 2, 0, 0, 0, 0],
            &[5, 254, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[2, 255, 67, 0, 0, 0, 220, 104, 0],
            &[0, 234, 100, 0, 0, 3, 247, 80, 0],
            &[0, 159, 208, 29, 6, 125, 241, 16, 0],
            &[0, 21, 188, 255, 255, 228, 74, 0, 0],
            &[0, 0, 0, 16, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 55, 23, 0, 0],
        ],
        // letter: 'Û' / 0xdb
        'Û' => &[
            &[0, 0, 21, 223, 230, 86, 0, 0, 0],
            &[0, 9, 190, 84, 29, 201, 61, 0, 0],
            &[0, 1, 4, 0, 0, 1, 3, 0, 0],
            &[5, 254, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[2, 255, 67, 0, 0, 0, 220, 104, 0],
            &[0, 234, 100, 0, 0, 3, 247, 80, 0],
            &[0, 159, 208, 29, 6, 125, 241, 16, 0],
            &[0, 21, 188, 255, 255, 228, 74, 0, 0],
            &[0, 0, 0, 16, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 49, 0, 0, 0, 0],
        ],
        // letter: 'Ü' / 0xdc
        'Ü' => &[
            &[0, 10, 187, 41, 0, 160, 79, 0, 0],
            &[0, 8, 178, 37, 0, 151, 73, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 254, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[2, 255, 67, 0, 0, 0, 220, 104, 0],
            &[0, 234, 100, 0, 0, 3, 247, 80, 0],
            &[0, 159, 208, 29, 6, 125, 241, 16, 0],
            &[0, 21, 188, 255, 255, 228, 74, 0, 0],
            &[0, 0, 0, 16, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ý' / 0xdd
        'Ý' => &[
            &[0, 0, 0, 0, 117, 224, 23, 0, 0],
            &[0, 0, 0, 38, 213, 36, 0, 0, 0],
            &[0, 0, 0, 2, 2, 0, 0, 0, 0],
            &[65, 250, 31, 0, 0, 0, 169, 167, 0],
            &[0, 191, 145, 0, 0, 36, 250, 40, 0],
            &[0, 61, 244, 18, 0, 152, 163, 0, 0],
            &[0, 0, 186, 126, 23, 243, 37, 0, 0],
            &[0, 0, 56, 234, 144, 158, 0, 0, 0],
            &[0, 0, 0, 182, 250, 33, 0, 0, 0],
            &[0, 0, 0, 103, 205, 0, 0, 0, 0],
            &[0, 0, 0, 102, 205, 0, 0, 0, 0],
            &[0, 0, 0, 102, 205, 0, 0, 0, 0],
            &[0, 0, 0, 102, 205, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 55, 23, 0, 0],
        ],
        // letter: 'Þ' / 0xde
        'Þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 169, 45, 40, 11, 0, 0, 0],
            &[0, 171, 247, 234, 246, 251, 152, 1, 0],
            &[0, 171, 150, 0, 0, 68, 247, 91, 0],
            &[0, 171, 150, 0, 0, 0, 183, 148, 0],
            &[0, 171, 150, 0, 0, 0, 196, 134, 0],
            &[0, 171, 163, 31, 45, 129, 249, 43, 0],
            &[0, 171, 252, 248, 236, 191, 65, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ß' / 0xdf
        'ß' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 104, 167, 174, 133, 27, 0, 0, 0],
            &[151, 230, 117, 100, 190, 227, 7, 0, 0],
            &[232, 99, 0, 0, 48, 255, 24, 0, 0],
            &[243, 72, 0, 11, 180, 179, 0, 0, 0],
            &[243, 72, 0, 181, 174, 7, 0, 0, 0],
            &[243, 72, 0, 242, 89, 0, 0, 0, 0],
            &[243, 72, 0, 109, 247, 117, 1, 0, 0],
            &[243, 72, 0, 0, 67, 231, 141, 0, 0],
            &[243, 72, 0, 0, 0, 73, 245, 2, 0],
            &[243, 72, 50, 13, 0, 99, 242, 1, 0],
            &[243, 72, 116, 250, 234, 241, 96, 0, 0],
            &[0, 0, 0, 10, 29, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'à' / 0xe0
        'à' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 71, 183, 21, 0, 0, 0, 0],
            &[0, 0, 1, 164, 164, 0, 0, 0, 0],
            &[0, 0, 0, 2, 111, 21, 0, 0, 0],
            &[0, 10, 92, 141, 153, 107, 10, 0, 0],
            &[0, 67, 187, 118, 109, 204, 185, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 16, 0],
            &[0, 0, 46, 110, 133, 142, 255, 35, 0],
            &[0, 124, 240, 142, 103, 90, 255, 35, 0],
            &[0, 243, 77, 0, 0, 22, 255, 35, 0],
            &[0, 235, 100, 0, 16, 163, 255, 35, 0],
            &[0, 93, 242, 235, 235, 94, 239, 35, 0],
            &[0, 0, 5, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'á' / 0xe1
        'á' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 185, 59, 0, 0],
            &[0, 0, 0, 0, 181, 148, 0, 0, 0],
            &[0, 0, 0, 28, 104, 0, 0, 0, 0],
            &[0, 10, 92, 141, 153, 107, 10, 0, 0],
            &[0, 67, 187, 118, 109, 204, 185, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 16, 0],
            &[0, 0, 46, 110, 133, 142, 255, 35, 0],
            &[0, 124, 240, 142, 103, 90, 255, 35, 0],
            &[0, 243, 77, 0, 0, 22, 255, 35, 0],
            &[0, 235, 100, 0, 16, 163, 255, 35, 0],
            &[0, 93, 242, 235, 235, 94, 239, 35, 0],
            &[0, 0, 5, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'â' / 0xe2
        'â' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 116, 182, 28, 0, 0, 0],
            &[0, 0, 83, 207, 120, 200, 11, 0, 0],
            &[0, 5, 117, 14, 0, 73, 67, 0, 0],
            &[0, 10, 92, 141, 153, 107, 10, 0, 0],
            &[0, 67, 187, 118, 109, 204, 185, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 16, 0],
            &[0, 0, 46, 110, 133, 142, 255, 35, 0],
            &[0, 124, 240, 142, 103, 90, 255, 35, 0],
            &[0, 243, 77, 0, 0, 22, 255, 35, 0],
            &[0, 235, 100, 0, 16, 163, 255, 35, 0],
            &[0, 93, 242, 235, 235, 94, 239, 35, 0],
            &[0, 0, 5, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ã' / 0xe3
        'ã' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 53, 1, 0, 53, 0, 0],
            &[0, 8, 214, 205, 205, 157, 144, 0, 0],
            &[0, 25, 76, 0, 65, 110, 11, 0, 0],
            &[0, 10, 92, 141, 153, 107, 10, 0, 0],
            &[0, 67, 187, 118, 109, 204, 185, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 16, 0],
            &[0, 0, 46, 110, 133, 142, 255, 35, 0],
            &[0, 124, 240, 142, 103, 90, 255, 35, 0],
            &[0, 243, 77, 0, 0, 22, 255, 35, 0],
            &[0, 235, 100, 0, 16, 163, 255, 35, 0],
            &[0, 93, 242, 235, 235, 94, 239, 35, 0],
            &[0, 0, 5, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ä' / 0xe4
        'ä' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 57, 10, 0, 44, 23, 0, 0],
            &[0, 6, 250, 86, 0, 204, 138, 0, 0],
            &[0, 0, 46, 6, 0, 35, 17, 0, 0],
            &[0, 10, 92, 141, 153, 107, 10, 0, 0],
            &[0, 67, 187, 118, 109, 204, 185, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 16, 0],
            &[0, 0, 46, 110, 133, 142, 255, 35, 0],
            &[0, 124, 240, 142, 103, 90, 255, 35, 0],
            &[0, 243, 77, 0, 0, 22, 255, 35, 0],
            &[0, 235, 100, 0, 16, 163, 255, 35, 0],
            &[0, 93, 242, 235, 235, 94, 239, 35, 0],
            &[0, 0, 5, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'å' / 0xe5
        'å' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 59, 98, 14, 0, 0, 0],
            &[0, 0, 37, 199, 111, 171, 0, 0, 0],
            &[0, 0, 42, 193, 95, 177, 0, 0, 0],
            &[0, 0, 0, 76, 118, 23, 0, 0, 0],
            &[0, 10, 92, 141, 153, 107, 10, 0, 0],
            &[0, 67, 187, 118, 109, 204, 185, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 16, 0],
            &[0, 0, 46, 110, 133, 142, 255, 35, 0],
            &[0, 124, 240, 142, 103, 90, 255, 35, 0],
            &[0, 243, 77, 0, 0, 22, 255, 35, 0],
            &[0, 235, 100, 0, 16, 163, 255, 35, 0],
            &[0, 93, 242, 235, 235, 94, 239, 35, 0],
            &[0, 0, 5, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'æ' / 0xe6
        'æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[2, 92, 153, 108, 12, 135, 140, 19, 0],
            &[14, 174, 109, 219, 218, 154, 179, 175, 0],
            &[0, 0, 0, 113, 227, 1, 35, 250, 6],
            &[0, 59, 119, 177, 223, 126, 132, 255, 26],
            &[84, 239, 132, 165, 217, 124, 124, 124, 14],
            &[172, 123, 0, 125, 204, 0, 0, 0, 0],
            &[163, 150, 6, 200, 235, 69, 0, 49, 0],
            &[59, 243, 244, 139, 75, 242, 239, 206, 0],
            &[0, 11, 18, 0, 0, 8, 25, 1, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ç' / 0xe7
        'ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 110, 155, 146, 87, 1, 0],
            &[0, 20, 222, 207, 126, 133, 172, 0, 0],
            &[0, 132, 216, 6, 0, 0, 0, 0, 0],
            &[0, 188, 140, 0, 0, 0, 0, 0, 0],
            &[0, 200, 127, 0, 0, 0, 0, 0, 0],
            &[0, 173, 164, 0, 0, 0, 0, 0, 0],
            &[0, 85, 248, 77, 1, 5, 56, 0, 0],
            &[0, 0, 123, 240, 255, 255, 218, 0, 0],
            &[0, 0, 0, 16, 243, 25, 0, 0, 0],
            &[0, 0, 0, 16, 129, 204, 0, 0, 0],
            &[0, 0, 0, 69, 74, 9, 0, 0, 0],
        ],
        // letter: 'è' / 0xe8
        'è' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 182, 19, 0, 0, 0, 0],
            &[0, 0, 2, 167, 160, 0, 0, 0, 0],
            &[0, 0, 0, 3, 112, 19, 0, 0, 0],
            &[0, 0, 37, 131, 156, 112, 11, 0, 0],
            &[0, 62, 242, 152, 111, 196, 201, 5, 0],
            &[0, 210, 131, 0, 0, 9, 235, 85, 0],
            &[20, 255, 159, 133, 133, 133, 225, 133, 0],
            &[33, 255, 143, 124, 124, 124, 124, 69, 0],
            &[7, 246, 74, 0, 0, 0, 0, 0, 0],
            &[0, 147, 210, 38, 0, 3, 62, 26, 0],
            &[0, 7, 144, 243, 236, 250, 213, 36, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'é' / 0xe9
        'é' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 185, 57, 0, 0],
            &[0, 0, 0, 0, 184, 145, 0, 0, 0],
            &[0, 0, 0, 30, 103, 0, 0, 0, 0],
            &[0, 0, 37, 131, 156, 112, 11, 0, 0],
            &[0, 62, 242, 152, 111, 196, 201, 5, 0],
            &[0, 210, 131, 0, 0, 9, 235, 85, 0],
            &[20, 255, 159, 133, 133, 133, 225, 133, 0],
            &[33, 255, 143, 124, 124, 124, 124, 69, 0],
            &[7, 246, 74, 0, 0, 0, 0, 0, 0],
            &[0, 147, 210, 38, 0, 3, 62, 26, 0],
            &[0, 7, 144, 243, 236, 250, 213, 36, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ê' / 0xea
        'ê' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 119, 182, 26, 0, 0, 0],
            &[0, 0, 86, 205, 121, 198, 10, 0, 0],
            &[0, 6, 117, 13, 0, 75, 66, 0, 0],
            &[0, 0, 37, 131, 156, 112, 11, 0, 0],
            &[0, 62, 242, 152, 111, 196, 201, 5, 0],
            &[0, 210, 131, 0, 0, 9, 235, 85, 0],
            &[20, 255, 159, 133, 133, 133, 225, 133, 0],
            &[33, 255, 143, 124, 124, 124, 124, 69, 0],
            &[7, 246, 74, 0, 0, 0, 0, 0, 0],
            &[0, 147, 210, 38, 0, 3, 62, 26, 0],
            &[0, 7, 144, 243, 236, 250, 213, 36, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ë' / 0xeb
        'ë' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 58, 9, 0, 45, 22, 0, 0],
            &[0, 8, 252, 82, 0, 208, 135, 0, 0],
            &[0, 0, 46, 6, 0, 36, 16, 0, 0],
            &[0, 0, 37, 131, 156, 112, 11, 0, 0],
            &[0, 62, 242, 152, 111, 196, 201, 5, 0],
            &[0, 210, 131, 0, 0, 9, 235, 85, 0],
            &[20, 255, 159, 133, 133, 133, 225, 133, 0],
            &[33, 255, 143, 124, 124, 124, 124, 69, 0],
            &[7, 246, 74, 0, 0, 0, 0, 0, 0],
            &[0, 147, 210, 38, 0, 3, 62, 26, 0],
            &[0, 7, 144, 243, 236, 250, 213, 36, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ì' / 0xec
        'ì' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 84, 179, 13, 0, 0, 0, 0],
            &[0, 0, 4, 179, 146, 0, 0, 0, 0],
            &[0, 0, 0, 6, 116, 13, 0, 0, 0],
            &[0, 84, 128, 128, 128, 6, 0, 0, 0],
            &[0, 65, 115, 157, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 56, 255, 17, 0, 0, 0],
            &[0, 196, 242, 255, 255, 254, 237, 123, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'í' / 0xed
        'í' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 46, 185, 46, 0, 0],
            &[0, 0, 0, 2, 197, 131, 0, 0, 0],
            &[0, 0, 0, 38, 96, 0, 0, 0, 0],
            &[0, 84, 128, 128, 128, 6, 0, 0, 0],
            &[0, 65, 115, 157, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 56, 255, 17, 0, 0, 0],
            &[0, 196, 242, 255, 255, 254, 237, 123, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'î' / 0xee
        'î' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 179, 18, 0, 0, 0],
            &[0, 0, 99, 197, 130, 188, 6, 0, 0],
            &[0, 10, 116, 9, 0, 82, 58, 0, 0],
            &[0, 84, 128, 128, 128, 6, 0, 0, 0],
            &[0, 65, 115, 157, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 56, 255, 17, 0, 0, 0],
            &[0, 196, 242, 255, 255, 254, 237, 123, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ï' / 0xef
        'ï' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 61, 6, 0, 49, 18, 0, 0],
            &[0, 19, 255, 68, 0, 222, 120, 0, 0],
            &[0, 0, 48, 4, 0, 39, 13, 0, 0],
            &[0, 84, 128, 128, 128, 6, 0, 0, 0],
            &[0, 65, 115, 157, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 56, 255, 17, 0, 0, 0],
            &[0, 196, 242, 255, 255, 254, 237, 123, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ð' / 0xf0
        'ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 113, 5, 37, 90, 0, 0],
            &[0, 0, 32, 203, 255, 246, 90, 0, 0],
            &[0, 0, 87, 227, 255, 224, 13, 0, 0],
            &[0, 0, 55, 33, 0, 159, 154, 0, 0],
            &[0, 0, 68, 143, 142, 91, 239, 28, 0],
            &[0, 122, 233, 123, 104, 192, 248, 107, 0],
            &[14, 247, 69, 0, 0, 7, 220, 148, 0],
            &[50, 255, 11, 0, 0, 0, 166, 155, 0],
            &[33, 255, 39, 0, 0, 0, 188, 123, 0],
            &[0, 197, 160, 8, 0, 67, 245, 36, 0],
            &[0, 26, 184, 244, 231, 230, 82, 0, 0],
            &[0, 0, 0, 14, 28, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ñ' / 0xf1
        'ñ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 56, 4, 0, 54, 0, 0],
            &[0, 1, 201, 201, 214, 147, 168, 0, 0],
            &[0, 12, 89, 0, 54, 113, 18, 0, 0],
            &[0, 113, 24, 92, 152, 136, 34, 0, 0],
            &[0, 225, 199, 163, 107, 181, 232, 15, 0],
            &[0, 225, 177, 0, 0, 10, 251, 71, 0],
            &[0, 225, 113, 0, 0, 0, 233, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ò' / 0xf2
        'ò' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 84, 179, 13, 0, 0, 0, 0],
            &[0, 0, 4, 179, 146, 0, 0, 0, 0],
            &[0, 0, 0, 6, 116, 13, 0, 0, 0],
            &[0, 0, 56, 142, 154, 97, 4, 0, 0],
            &[0, 95, 245, 136, 109, 207, 185, 2, 0],
            &[4, 235, 102, 0, 0, 16, 240, 82, 0],
            &[41, 255, 25, 0, 0, 0, 182, 142, 0],
            &[52, 255, 10, 0, 0, 0, 170, 155, 0],
            &[20, 254, 51, 0, 0, 0, 210, 123, 0],
            &[0, 177, 185, 13, 0, 99, 248, 34, 0],
            &[0, 20, 180, 246, 237, 229, 79, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ó' / 0xf3
        'ó' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 46, 185, 46, 0, 0],
            &[0, 0, 0, 2, 197, 131, 0, 0, 0],
            &[0, 0, 0, 38, 96, 0, 0, 0, 0],
            &[0, 0, 56, 142, 154, 97, 4, 0, 0],
            &[0, 95, 245, 136, 109, 207, 185, 2, 0],
            &[4, 235, 102, 0, 0, 16, 240, 82, 0],
            &[41, 255, 25, 0, 0, 0, 182, 142, 0],
            &[52, 255, 10, 0, 0, 0, 170, 155, 0],
            &[20, 254, 51, 0, 0, 0, 210, 123, 0],
            &[0, 177, 185, 13, 0, 99, 248, 34, 0],
            &[0, 20, 180, 246, 237, 229, 79, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ô' / 0xf4
        'ô' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 179, 18, 0, 0, 0],
            &[0, 0, 99, 197, 130, 188, 6, 0, 0],
            &[0, 10, 116, 9, 0, 82, 58, 0, 0],
            &[0, 0, 56, 142, 154, 97, 4, 0, 0],
            &[0, 95, 245, 136, 109, 207, 185, 2, 0],
            &[4, 235, 102, 0, 0, 16, 240, 82, 0],
            &[41, 255, 25, 0, 0, 0, 182, 142, 0],
            &[52, 255, 10, 0, 0, 0, 170, 155, 0],
            &[20, 254, 51, 0, 0, 0, 210, 123, 0],
            &[0, 177, 185, 13, 0, 99, 248, 34, 0],
            &[0, 20, 180, 246, 237, 229, 79, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'õ' / 0xf5
        'õ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 51, 0, 3, 50, 0, 0],
            &[0, 16, 220, 209, 197, 165, 126, 0, 0],
            &[0, 34, 66, 0, 73, 106, 6, 0, 0],
            &[0, 0, 56, 142, 154, 97, 4, 0, 0],
            &[0, 95, 245, 136, 109, 207, 185, 2, 0],
            &[4, 235, 102, 0, 0, 16, 240, 82, 0],
            &[41, 255, 25, 0, 0, 0, 182, 142, 0],
            &[52, 255, 10, 0, 0, 0, 170, 155, 0],
            &[20, 254, 51, 0, 0, 0, 210, 123, 0],
            &[0, 177, 185, 13, 0, 99, 248, 34, 0],
            &[0, 20, 180, 246, 237, 229, 79, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ö' / 0xf6
        'ö' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 61, 6, 0, 49, 18, 0, 0],
            &[0, 19, 255, 68, 0, 222, 120, 0, 0],
            &[0, 0, 48, 4, 0, 39, 13, 0, 0],
            &[0, 0, 56, 142, 154, 97, 4, 0, 0],
            &[0, 95, 245, 136, 109, 207, 185, 2, 0],
            &[4, 235, 102, 0, 0, 16, 240, 82, 0],
            &[41, 255, 25, 0, 0, 0, 182, 142, 0],
            &[52, 255, 10, 0, 0, 0, 170, 155, 0],
            &[20, 254, 51, 0, 0, 0, 210, 123, 0],
            &[0, 177, 185, 13, 0, 99, 248, 34, 0],
            &[0, 20, 180, 246, 237, 229, 79, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '÷' / 0xf7
        '÷' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 16, 0, 0, 0, 0],
            &[0, 0, 0, 127, 219, 0, 0, 0, 0],
            &[0, 0, 0, 42, 88, 0, 0, 0, 0],
            &[0, 97, 110, 110, 110, 110, 110, 30, 0],
            &[0, 126, 143, 143, 143, 143, 143, 40, 0],
            &[0, 0, 0, 26, 60, 0, 0, 0, 0],
            &[0, 0, 0, 130, 222, 0, 0, 0, 0],
            &[0, 0, 0, 16, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ø' / 0xf8
        'ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 56, 142, 154, 97, 157, 49, 0],
            &[0, 95, 245, 136, 109, 255, 255, 8, 0],
            &[4, 235, 102, 0, 36, 249, 255, 82, 0],
            &[41, 255, 25, 4, 202, 101, 182, 142, 0],
            &[52, 255, 10, 134, 173, 0, 170, 155, 0],
            &[20, 254, 114, 225, 18, 0, 210, 123, 0],
            &[0, 195, 255, 79, 0, 99, 248, 34, 0],
            &[0, 188, 255, 246, 237, 229, 79, 0, 0],
            &[0, 54, 5, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ù' / 0xf9
        'ù' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 84, 179, 13, 0, 0, 0, 0],
            &[0, 0, 4, 179, 146, 0, 0, 0, 0],
            &[0, 0, 0, 6, 116, 13, 0, 0, 0],
            &[0, 122, 39, 0, 0, 0, 126, 36, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 3, 253, 71, 0],
            &[0, 238, 86, 0, 0, 33, 255, 71, 0],
            &[0, 200, 166, 2, 4, 147, 255, 71, 0],
            &[0, 62, 232, 242, 236, 130, 209, 71, 0],
            &[0, 0, 1, 26, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ú' / 0xfa
        'ú' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 46, 185, 46, 0, 0],
            &[0, 0, 0, 2, 197, 131, 0, 0, 0],
            &[0, 0, 0, 38, 96, 0, 0, 0, 0],
            &[0, 122, 39, 0, 0, 0, 126, 36, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 3, 253, 71, 0],
            &[0, 238, 86, 0, 0, 33, 255, 71, 0],
            &[0, 200, 166, 2, 4, 147, 255, 71, 0],
            &[0, 62, 232, 242, 236, 130, 209, 71, 0],
            &[0, 0, 1, 26, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'û' / 0xfb
        'û' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 179, 18, 0, 0, 0],
            &[0, 0, 99, 197, 130, 188, 6, 0, 0],
            &[0, 10, 116, 9, 0, 82, 58, 0, 0],
            &[0, 122, 39, 0, 0, 0, 126, 36, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 3, 253, 71, 0],
            &[0, 238, 86, 0, 0, 33, 255, 71, 0],
            &[0, 200, 166, 2, 4, 147, 255, 71, 0],
            &[0, 62, 232, 242, 236, 130, 209, 71, 0],
            &[0, 0, 1, 26, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ü' / 0xfc
        'ü' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 61, 6, 0, 49, 18, 0, 0],
            &[0, 19, 255, 68, 0, 222, 120, 0, 0],
            &[0, 0, 48, 4, 0, 39, 13, 0, 0],
            &[0, 122, 39, 0, 0, 0, 126, 36, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 3, 253, 71, 0],
            &[0, 238, 86, 0, 0, 33, 255, 71, 0],
            &[0, 200, 166, 2, 4, 147, 255, 71, 0],
            &[0, 62, 232, 242, 236, 130, 209, 71, 0],
            &[0, 0, 1, 26, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ý' / 0xfd
        'ý' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 46, 185, 46, 0, 0],
            &[0, 0, 0, 2, 197, 131, 0, 0, 0],
            &[0, 0, 0, 38, 96, 0, 0, 0, 0],
            &[34, 122, 1, 0, 0, 0, 79, 85, 0],
            &[7, 235, 69, 0, 0, 0, 221, 98, 0],
            &[0, 137, 173, 0, 0, 51, 245, 13, 0],
            &[0, 33, 248, 25, 0, 137, 163, 0, 0],
            &[0, 0, 180, 125, 0, 222, 68, 0, 0],
            &[0, 0, 73, 222, 51, 227, 2, 0, 0],
            &[0, 0, 2, 221, 183, 134, 0, 0, 0],
            &[0, 0, 0, 116, 255, 40, 0, 0, 0],
            &[0, 0, 0, 100, 200, 0, 0, 0, 0],
            &[0, 0, 3, 212, 94, 0, 0, 0, 0],
            &[8, 101, 89, 11, 0, 0, 0, 0, 0],
        ],
        // letter: 'þ' / 0xfe
        'þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 144, 59, 0, 0, 0, 0, 0, 0],
            &[0, 225, 93, 0, 0, 0, 0, 0, 0],
            &[0, 225, 93, 0, 0, 0, 0, 0, 0],
            &[0, 225, 96, 104, 156, 127, 20, 0, 0],
            &[0, 225, 220, 152, 110, 194, 217, 10, 0],
            &[0, 225, 163, 0, 0, 8, 236, 97, 0],
            &[0, 225, 106, 0, 0, 0, 185, 148, 0],
            &[0, 225, 100, 0, 0, 0, 176, 154, 0],
            &[0, 225, 133, 0, 0, 0, 209, 126, 0],
            &[0, 225, 227, 30, 0, 81, 253, 45, 0],
            &[0, 225, 142, 223, 239, 243, 106, 0, 0],
            &[0, 225, 93, 1, 25, 7, 0, 0, 0],
            &[0, 225, 93, 0, 0, 0, 0, 0, 0],
            &[0, 81, 33, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ÿ' / 0xff
        'ÿ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 61, 6, 0, 49, 18, 0, 0],
            &[0, 19, 255, 68, 0, 222, 120, 0, 0],
            &[0, 0, 48, 4, 0, 39, 13, 0, 0],
            &[34, 122, 1, 0, 0, 0, 79, 85, 0],
            &[7, 235, 69, 0, 0, 0, 221, 98, 0],
            &[0, 137, 173, 0, 0, 51, 245, 13, 0],
            &[0, 33, 248, 25, 0, 137, 163, 0, 0],
            &[0, 0, 180, 125, 0, 222, 68, 0, 0],
            &[0, 0, 73, 222, 51, 227, 2, 0, 0],
            &[0, 0, 2, 221, 183, 134, 0, 0, 0],
            &[0, 0, 0, 116, 255, 40, 0, 0, 0],
            &[0, 0, 0, 100, 200, 0, 0, 0, 0],
            &[0, 0, 3, 212, 94, 0, 0, 0, 0],
            &[8, 101, 89, 11, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ā' / 0x100
        'Ā' => &[
            &[0, 0, 31, 35, 35, 35, 9, 0, 0],
            &[0, 0, 194, 219, 219, 219, 59, 0, 0],
            &[0, 0, 0, 4, 8, 0, 0, 0, 0],
            &[0, 0, 0, 165, 249, 15, 0, 0, 0],
            &[0, 0, 6, 241, 224, 91, 0, 0, 0],
            &[0, 0, 75, 224, 125, 174, 0, 0, 0],
            &[0, 0, 157, 146, 47, 246, 11, 0, 0],
            &[0, 4, 236, 67, 0, 223, 85, 0, 0],
            &[0, 67, 255, 52, 47, 182, 168, 0, 0],
            &[0, 150, 255, 239, 239, 255, 248, 8, 0],
            &[2, 231, 84, 0, 0, 4, 237, 78, 0],
            &[60, 248, 12, 0, 0, 0, 162, 161, 0],
            &[142, 181, 0, 0, 0, 0, 82, 239, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ā' / 0x101
        'ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 134, 165, 165, 165, 56, 0, 0],
            &[0, 0, 72, 89, 89, 89, 30, 0, 0],
            &[0, 10, 92, 141, 153, 107, 10, 0, 0],
            &[0, 67, 187, 118, 109, 204, 185, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 16, 0],
            &[0, 0, 46, 110, 133, 142, 255, 35, 0],
            &[0, 124, 240, 142, 103, 90, 255, 35, 0],
            &[0, 243, 77, 0, 0, 22, 255, 35, 0],
            &[0, 235, 100, 0, 16, 163, 255, 35, 0],
            &[0, 93, 242, 235, 235, 94, 239, 35, 0],
            &[0, 0, 5, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ă' / 0x102
        'Ă' => &[
            &[0, 0, 186, 24, 1, 145, 60, 0, 0],
            &[0, 0, 84, 224, 229, 160, 3, 0, 0],
            &[0, 0, 0, 4, 9, 0, 0, 0, 0],
            &[0, 0, 0, 165, 249, 15, 0, 0, 0],
            &[0, 0, 6, 241, 224, 91, 0, 0, 0],
            &[0, 0, 75, 224, 125, 174, 0, 0, 0],
            &[0, 0, 157, 146, 47, 246, 11, 0, 0],
            &[0, 4, 236, 67, 0, 223, 85, 0, 0],
            &[0, 67, 255, 52, 47, 182, 168, 0, 0],
            &[0, 150, 255, 239, 239, 255, 248, 8, 0],
            &[2, 231, 84, 0, 0, 4, 237, 78, 0],
            &[60, 248, 12, 0, 0, 0, 162, 161, 0],
            &[142, 181, 0, 0, 0, 0, 82, 239, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ă' / 0x103
        'ă' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 77, 1, 0, 41, 37, 0, 0],
            &[0, 0, 162, 155, 110, 214, 45, 0, 0],
            &[0, 0, 7, 98, 119, 47, 0, 0, 0],
            &[0, 10, 92, 141, 153, 107, 10, 0, 0],
            &[0, 67, 187, 118, 109, 204, 185, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 16, 0],
            &[0, 0, 46, 110, 133, 142, 255, 35, 0],
            &[0, 124, 240, 142, 103, 90, 255, 35, 0],
            &[0, 243, 77, 0, 0, 22, 255, 35, 0],
            &[0, 235, 100, 0, 16, 163, 255, 35, 0],
            &[0, 93, 242, 235, 235, 94, 239, 35, 0],
            &[0, 0, 5, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ą' / 0x104
        'Ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 8, 0, 0, 0, 0],
            &[0, 0, 0, 165, 249, 15, 0, 0, 0],
            &[0, 0, 6, 241, 224, 91, 0, 0, 0],
            &[0, 0, 75, 224, 125, 174, 0, 0, 0],
            &[0, 0, 157, 146, 47, 246, 11, 0, 0],
            &[0, 4, 236, 67, 0, 223, 85, 0, 0],
            &[0, 67, 255, 52, 47, 182, 168, 0, 0],
            &[0, 150, 255, 239, 239, 255, 248, 8, 0],
            &[2, 231, 84, 0, 0, 4, 237, 78, 0],
            &[60, 248, 12, 0, 0, 0, 162, 161, 0],
            &[142, 181, 0, 0, 0, 0, 82, 255, 6],
            &[0, 0, 0, 0, 0, 0, 71, 170, 2],
            &[0, 0, 0, 0, 0, 0, 193, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 2],
        ],
        // letter: 'ą' / 0x105
        'ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 92, 141, 153, 107, 10, 0, 0],
            &[0, 67, 187, 118, 109, 204, 185, 0, 0],
            &[0, 0, 0, 0, 0, 31, 255, 16, 0],
            &[0, 0, 46, 110, 133, 142, 255, 35, 0],
            &[0, 124, 240, 142, 103, 90, 255, 35, 0],
            &[0, 243, 77, 0, 0, 22, 255, 35, 0],
            &[0, 235, 100, 0, 16, 163, 255, 35, 0],
            &[0, 93, 242, 235, 235, 94, 255, 35, 0],
            &[0, 0, 5, 27, 4, 85, 158, 0, 0],
            &[0, 0, 0, 0, 0, 209, 48, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 1, 0],
        ],
        // letter: 'Ć' / 0x106
        'Ć' => &[
            &[0, 0, 0, 0, 0, 162, 196, 6, 0],
            &[0, 0, 0, 0, 73, 198, 16, 0, 0],
            &[0, 0, 0, 0, 21, 26, 1, 0, 0],
            &[0, 0, 57, 206, 255, 255, 242, 103, 0],
            &[0, 41, 245, 138, 20, 7, 54, 38, 0],
            &[0, 166, 184, 0, 0, 0, 0, 0, 0],
            &[0, 236, 96, 0, 0, 0, 0, 0, 0],
            &[11, 255, 61, 0, 0, 0, 0, 0, 0],
            &[9, 255, 62, 0, 0, 0, 0, 0, 0],
            &[0, 238, 94, 0, 0, 0, 0, 0, 0],
            &[0, 171, 178, 0, 0, 0, 0, 0, 0],
            &[0, 50, 249, 137, 28, 15, 58, 28, 0],
            &[0, 0, 70, 214, 255, 255, 239, 52, 0],
            &[0, 0, 0, 0, 16, 26, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 55, 14, 0],
        ],
        // letter: 'ć' / 0x107
        'ć' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 157, 119, 0, 0],
            &[0, 0, 0, 0, 99, 212, 19, 0, 0],
            &[0, 0, 0, 0, 115, 19, 0, 0, 0],
            &[0, 0, 15, 110, 155, 146, 87, 1, 0],
            &[0, 20, 222, 207, 126, 133, 172, 0, 0],
            &[0, 132, 216, 6, 0, 0, 0, 0, 0],
            &[0, 188, 140, 0, 0, 0, 0, 0, 0],
            &[0, 200, 127, 0, 0, 0, 0, 0, 0],
            &[0, 173, 164, 0, 0, 0, 0, 0, 0],
            &[0, 85, 248, 77, 1, 5, 56, 0, 0],
            &[0, 0, 123, 240, 249, 253, 218, 0, 0],
            &[0, 0, 0, 4, 28, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĉ' / 0x108
        'Ĉ' => &[
            &[0, 0, 0, 46, 231, 233, 50, 0, 0],
            &[0, 0, 27, 203, 54, 53, 205, 33, 0],
            &[0, 0, 2, 3, 17, 24, 5, 2, 0],
            &[0, 0, 57, 206, 255, 255, 242, 103, 0],
            &[0, 41, 245, 138, 20, 7, 54, 38, 0],
            &[0, 166, 184, 0, 0, 0, 0, 0, 0],
            &[0, 236, 96, 0, 0, 0, 0, 0, 0],
            &[11, 255, 61, 0, 0, 0, 0, 0, 0],
            &[9, 255, 62, 0, 0, 0, 0, 0, 0],
            &[0, 238, 94, 0, 0, 0, 0, 0, 0],
            &[0, 171, 178, 0, 0, 0, 0, 0, 0],
            &[0, 50, 249, 137, 28, 15, 58, 28, 0],
            &[0, 0, 70, 214, 255, 255, 239, 52, 0],
            &[0, 0, 0, 0, 16, 26, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 39, 0, 0, 0],
        ],
        // letter: 'ĉ' / 0x109
        'ĉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 57, 185, 85, 0, 0, 0],
            &[0, 0, 27, 217, 104, 221, 51, 0, 0],
            &[0, 0, 86, 50, 0, 31, 109, 0, 0],
            &[0, 0, 15, 110, 155, 146, 87, 1, 0],
            &[0, 20, 222, 207, 126, 133, 172, 0, 0],
            &[0, 132, 216, 6, 0, 0, 0, 0, 0],
            &[0, 188, 140, 0, 0, 0, 0, 0, 0],
            &[0, 200, 127, 0, 0, 0, 0, 0, 0],
            &[0, 173, 164, 0, 0, 0, 0, 0, 0],
            &[0, 85, 248, 77, 1, 5, 56, 0, 0],
            &[0, 0, 123, 240, 249, 253, 218, 0, 0],
            &[0, 0, 0, 4, 28, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ċ' / 0x10a
        'Ċ' => &[
            &[0, 0, 0, 0, 189, 195, 0, 0, 0],
            &[0, 0, 0, 0, 92, 96, 0, 0, 0],
            &[0, 0, 0, 0, 17, 24, 1, 0, 0],
            &[0, 0, 57, 206, 255, 255, 242, 103, 0],
            &[0, 41, 245, 138, 20, 7, 54, 38, 0],
            &[0, 166, 184, 0, 0, 0, 0, 0, 0],
            &[0, 236, 96, 0, 0, 0, 0, 0, 0],
            &[11, 255, 61, 0, 0, 0, 0, 0, 0],
            &[9, 255, 62, 0, 0, 0, 0, 0, 0],
            &[0, 238, 94, 0, 0, 0, 0, 0, 0],
            &[0, 171, 178, 0, 0, 0, 0, 0, 0],
            &[0, 50, 249, 137, 28, 15, 58, 28, 0],
            &[0, 0, 70, 214, 255, 255, 239, 52, 0],
            &[0, 0, 0, 0, 16, 26, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 6, 0, 0, 0],
        ],
        // letter: 'ċ' / 0x10b
        'ċ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 140, 32, 0, 0, 0],
            &[0, 0, 0, 46, 254, 84, 0, 0, 0],
            &[0, 0, 0, 0, 13, 0, 0, 0, 0],
            &[0, 0, 15, 110, 155, 146, 87, 1, 0],
            &[0, 20, 222, 207, 126, 133, 172, 0, 0],
            &[0, 132, 216, 6, 0, 0, 0, 0, 0],
            &[0, 188, 140, 0, 0, 0, 0, 0, 0],
            &[0, 200, 127, 0, 0, 0, 0, 0, 0],
            &[0, 173, 164, 0, 0, 0, 0, 0, 0],
            &[0, 85, 248, 77, 1, 5, 56, 0, 0],
            &[0, 0, 123, 240, 249, 253, 218, 0, 0],
            &[0, 0, 0, 4, 28, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Č' / 0x10c
        'Č' => &[
            &[0, 0, 13, 199, 103, 100, 202, 17, 0],
            &[0, 0, 0, 23, 231, 233, 26, 0, 0],
            &[0, 0, 0, 0, 21, 28, 1, 0, 0],
            &[0, 0, 57, 206, 255, 255, 242, 103, 0],
            &[0, 41, 245, 138, 20, 7, 54, 38, 0],
            &[0, 166, 184, 0, 0, 0, 0, 0, 0],
            &[0, 236, 96, 0, 0, 0, 0, 0, 0],
            &[11, 255, 61, 0, 0, 0, 0, 0, 0],
            &[9, 255, 62, 0, 0, 0, 0, 0, 0],
            &[0, 238, 94, 0, 0, 0, 0, 0, 0],
            &[0, 171, 178, 0, 0, 0, 0, 0, 0],
            &[0, 50, 249, 137, 28, 15, 58, 28, 0],
            &[0, 0, 70, 214, 255, 255, 239, 52, 0],
            &[0, 0, 0, 0, 16, 26, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 35, 0, 0, 34, 21, 0],
        ],
        // letter: 'č' / 0x10d
        'č' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 99, 0, 71, 140, 0, 0],
            &[0, 0, 10, 201, 173, 221, 26, 0, 0],
            &[0, 0, 0, 30, 132, 51, 0, 0, 0],
            &[0, 0, 15, 110, 155, 146, 87, 1, 0],
            &[0, 20, 222, 207, 126, 133, 172, 0, 0],
            &[0, 132, 216, 6, 0, 0, 0, 0, 0],
            &[0, 188, 140, 0, 0, 0, 0, 0, 0],
            &[0, 200, 127, 0, 0, 0, 0, 0, 0],
            &[0, 173, 164, 0, 0, 0, 0, 0, 0],
            &[0, 85, 248, 77, 1, 5, 56, 0, 0],
            &[0, 0, 123, 240, 249, 253, 218, 0, 0],
            &[0, 0, 0, 4, 28, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ď' / 0x10e
        'Ď' => &[
            &[0, 28, 213, 74, 132, 181, 5, 0, 0],
            &[0, 0, 45, 245, 211, 10, 0, 0, 0],
            &[0, 0, 0, 4, 3, 0, 0, 0, 0],
            &[5, 254, 254, 249, 218, 128, 6, 0, 0],
            &[5, 255, 80, 32, 82, 219, 185, 1, 0],
            &[5, 255, 61, 0, 0, 30, 246, 81, 0],
            &[5, 255, 61, 0, 0, 0, 177, 158, 0],
            &[5, 255, 61, 0, 0, 0, 141, 187, 0],
            &[5, 255, 61, 0, 0, 0, 141, 183, 0],
            &[5, 255, 61, 0, 0, 0, 181, 151, 0],
            &[5, 255, 61, 0, 0, 36, 249, 70, 0],
            &[5, 255, 79, 33, 91, 224, 169, 0, 0],
            &[5, 255, 255, 245, 209, 114, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 27, 0, 0, 42, 13, 0, 0],
        ],
        // letter: 'ď' / 0x10f
        'ď' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 86, 117, 45, 149],
            &[0, 0, 0, 0, 0, 134, 184, 113, 149],
            &[0, 0, 0, 0, 0, 134, 184, 98, 24],
            &[0, 9, 114, 155, 105, 134, 184, 0, 0],
            &[0, 176, 220, 109, 164, 229, 184, 0, 0],
            &[36, 255, 56, 0, 0, 209, 184, 0, 0],
            &[81, 248, 2, 0, 0, 156, 184, 0, 0],
            &[90, 240, 0, 0, 0, 138, 184, 0, 0],
            &[65, 255, 13, 0, 0, 166, 184, 0, 0],
            &[8, 237, 124, 0, 43, 243, 184, 0, 0],
            &[0, 67, 234, 240, 217, 129, 184, 0, 0],
            &[0, 0, 1, 26, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Đ' / 0x110
        'Đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 224, 254, 251, 225, 145, 16, 0, 0],
            &[0, 225, 112, 29, 71, 200, 212, 9, 0],
            &[0, 225, 97, 0, 0, 12, 228, 117, 0],
            &[0, 225, 97, 0, 0, 0, 141, 194, 0],
            &[124, 255, 222, 124, 10, 0, 105, 223, 0],
            &[154, 255, 251, 154, 13, 0, 105, 219, 0],
            &[0, 225, 97, 0, 0, 0, 145, 187, 0],
            &[0, 225, 97, 0, 0, 16, 233, 105, 0],
            &[0, 225, 111, 30, 78, 208, 200, 5, 0],
            &[0, 225, 255, 248, 217, 132, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'đ' / 0x111
        'đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 158, 45, 0],
            &[0, 0, 0, 43, 97, 97, 255, 168, 38],
            &[0, 0, 0, 61, 139, 139, 255, 210, 55],
            &[0, 0, 63, 133, 125, 31, 245, 71, 0],
            &[0, 111, 251, 151, 132, 213, 243, 71, 0],
            &[5, 241, 116, 0, 0, 79, 255, 71, 0],
            &[43, 255, 38, 0, 0, 16, 255, 71, 0],
            &[53, 255, 21, 0, 0, 0, 250, 71, 0],
            &[26, 255, 53, 0, 0, 24, 255, 71, 0],
            &[0, 200, 176, 8, 3, 151, 255, 71, 0],
            &[0, 39, 210, 248, 242, 139, 211, 71, 0],
            &[0, 0, 0, 18, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ē' / 0x112
        'Ē' => &[
            &[0, 0, 28, 35, 35, 35, 12, 0, 0],
            &[0, 0, 175, 219, 219, 219, 77, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 254, 254, 254, 254, 254, 53, 0],
            &[0, 171, 163, 31, 31, 31, 31, 6, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 233, 200, 200, 200, 188, 0, 0],
            &[0, 171, 186, 86, 86, 86, 81, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 163, 30, 30, 30, 30, 6, 0],
            &[0, 171, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ē' / 0x113
        'ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 137, 165, 165, 165, 54, 0, 0],
            &[0, 0, 73, 89, 89, 89, 29, 0, 0],
            &[0, 0, 37, 131, 156, 112, 11, 0, 0],
            &[0, 62, 242, 152, 111, 196, 201, 5, 0],
            &[0, 210, 131, 0, 0, 9, 235, 85, 0],
            &[20, 255, 159, 133, 133, 133, 225, 133, 0],
            &[33, 255, 143, 124, 124, 124, 124, 69, 0],
            &[7, 246, 74, 0, 0, 0, 0, 0, 0],
            &[0, 147, 210, 38, 0, 3, 62, 26, 0],
            &[0, 7, 144, 243, 236, 250, 213, 36, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĕ' / 0x114
        'Ĕ' => &[
            &[0, 0, 173, 37, 0, 128, 78, 0, 0],
            &[0, 0, 69, 220, 229, 174, 8, 0, 0],
            &[0, 0, 0, 0, 1, 0, 0, 0, 0],
            &[0, 170, 254, 254, 254, 254, 254, 53, 0],
            &[0, 171, 163, 31, 31, 31, 31, 6, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 233, 200, 200, 200, 188, 0, 0],
            &[0, 171, 186, 86, 86, 86, 81, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 163, 30, 30, 30, 30, 6, 0],
            &[0, 171, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĕ' / 0x115
        'ĕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 78, 1, 0, 43, 35, 0, 0],
            &[0, 0, 166, 153, 111, 215, 43, 0, 0],
            &[0, 0, 7, 99, 119, 46, 0, 0, 0],
            &[0, 0, 37, 131, 156, 112, 11, 0, 0],
            &[0, 62, 242, 152, 111, 196, 201, 5, 0],
            &[0, 210, 131, 0, 0, 9, 235, 85, 0],
            &[20, 255, 159, 133, 133, 133, 225, 133, 0],
            &[33, 255, 143, 124, 124, 124, 124, 69, 0],
            &[7, 246, 74, 0, 0, 0, 0, 0, 0],
            &[0, 147, 210, 38, 0, 3, 62, 26, 0],
            &[0, 7, 144, 243, 236, 250, 213, 36, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ė' / 0x116
        'Ė' => &[
            &[0, 0, 0, 123, 247, 14, 0, 0, 0],
            &[0, 0, 0, 52, 134, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 254, 254, 254, 254, 254, 53, 0],
            &[0, 171, 163, 31, 31, 31, 31, 6, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 233, 200, 200, 200, 188, 0, 0],
            &[0, 171, 186, 86, 86, 86, 81, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 163, 30, 30, 30, 30, 6, 0],
            &[0, 171, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 0, 0, 0, 0],
        ],
        // letter: 'ė' / 0x117
        'ė' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 130, 0, 0, 0, 0],
            &[0, 0, 0, 130, 244, 9, 0, 0, 0],
            &[0, 0, 0, 0, 12, 0, 0, 0, 0],
            &[0, 0, 37, 131, 156, 112, 11, 0, 0],
            &[0, 62, 242, 152, 111, 196, 201, 5, 0],
            &[0, 210, 131, 0, 0, 9, 235, 85, 0],
            &[20, 255, 159, 133, 133, 133, 225, 133, 0],
            &[33, 255, 143, 124, 124, 124, 124, 69, 0],
            &[7, 246, 74, 0, 0, 0, 0, 0, 0],
            &[0, 147, 210, 38, 0, 3, 62, 26, 0],
            &[0, 7, 144, 243, 236, 250, 213, 36, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ę' / 0x118
        'Ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 254, 254, 254, 254, 254, 53, 0],
            &[0, 171, 163, 31, 31, 31, 31, 6, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 233, 200, 200, 200, 188, 0, 0],
            &[0, 171, 186, 86, 86, 86, 81, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 163, 30, 30, 30, 30, 6, 0],
            &[0, 171, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 17, 197, 30, 0, 0],
            &[0, 0, 0, 0, 107, 150, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 7, 0, 0],
        ],
        // letter: 'ę' / 0x119
        'ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 37, 131, 156, 112, 11, 0, 0],
            &[0, 62, 242, 152, 111, 196, 201, 5, 0],
            &[0, 210, 131, 0, 0, 9, 235, 85, 0],
            &[20, 255, 159, 133, 133, 133, 225, 133, 0],
            &[33, 255, 143, 124, 124, 124, 124, 69, 0],
            &[7, 246, 74, 0, 0, 0, 0, 0, 0],
            &[0, 147, 210, 38, 0, 3, 62, 26, 0],
            &[0, 7, 144, 243, 236, 252, 255, 42, 0],
            &[0, 0, 0, 5, 28, 184, 71, 0, 0],
            &[0, 0, 0, 0, 34, 223, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 12, 0, 0],
        ],
        // letter: 'Ě' / 0x11a
        'Ě' => &[
            &[0, 0, 150, 160, 56, 217, 50, 0, 0],
            &[0, 0, 2, 186, 253, 71, 0, 0, 0],
            &[0, 0, 0, 2, 5, 0, 0, 0, 0],
            &[0, 170, 254, 254, 254, 254, 254, 53, 0],
            &[0, 171, 163, 31, 31, 31, 31, 6, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 233, 200, 200, 200, 188, 0, 0],
            &[0, 171, 186, 86, 86, 86, 81, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 150, 0, 0, 0, 0, 0, 0],
            &[0, 171, 163, 30, 30, 30, 30, 6, 0],
            &[0, 171, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 49, 0, 0, 19, 35, 0, 0],
        ],
        // letter: 'ě' / 0x11b
        'ě' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 156, 44, 3, 130, 78, 0, 0],
            &[0, 0, 52, 229, 185, 164, 1, 0, 0],
            &[0, 0, 0, 75, 129, 9, 0, 0, 0],
            &[0, 0, 37, 131, 156, 112, 11, 0, 0],
            &[0, 62, 242, 152, 111, 196, 201, 5, 0],
            &[0, 210, 131, 0, 0, 9, 235, 85, 0],
            &[20, 255, 159, 133, 133, 133, 225, 133, 0],
            &[33, 255, 143, 124, 124, 124, 124, 69, 0],
            &[7, 246, 74, 0, 0, 0, 0, 0, 0],
            &[0, 147, 210, 38, 0, 3, 62, 26, 0],
            &[0, 7, 144, 243, 236, 250, 213, 36, 0],
            &[0, 0, 0, 5, 28, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => &[
            &[0, 0, 0, 168, 226, 164, 0, 0, 0],
            &[0, 0, 130, 151, 6, 157, 131, 0, 0],
            &[0, 0, 5, 0, 21, 17, 5, 0, 0],
            &[0, 0, 104, 229, 255, 255, 220, 31, 0],
            &[0, 102, 247, 99, 14, 17, 80, 1, 0],
            &[2, 233, 116, 0, 0, 0, 0, 0, 0],
            &[51, 255, 25, 0, 0, 0, 0, 0, 0],
            &[81, 245, 0, 0, 43, 71, 71, 29, 0],
            &[87, 241, 0, 0, 133, 215, 247, 107, 0],
            &[65, 255, 16, 0, 0, 0, 204, 107, 0],
            &[11, 248, 95, 0, 0, 0, 204, 107, 0],
            &[0, 139, 237, 75, 7, 13, 212, 107, 0],
            &[0, 3, 135, 241, 254, 255, 229, 73, 0],
            &[0, 0, 0, 3, 26, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 55, 11, 0, 0, 0],
        ],
        // letter: 'ĝ' / 0x11d
        'ĝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 179, 18, 0, 0, 0],
            &[0, 0, 99, 197, 130, 188, 6, 0, 0],
            &[0, 10, 116, 9, 0, 82, 58, 0, 0],
            &[0, 0, 80, 155, 143, 41, 99, 36, 0],
            &[0, 112, 245, 130, 113, 206, 240, 71, 0],
            &[5, 239, 102, 0, 0, 45, 255, 71, 0],
            &[43, 255, 34, 0, 0, 0, 234, 71, 0],
            &[54, 255, 23, 0, 0, 0, 222, 71, 0],
            &[26, 255, 59, 0, 0, 4, 246, 71, 0],
            &[0, 196, 185, 11, 0, 125, 255, 71, 0],
            &[0, 34, 207, 250, 240, 174, 230, 71, 0],
            &[0, 0, 0, 24, 11, 0, 240, 63, 0],
            &[0, 16, 0, 0, 0, 62, 252, 22, 0],
            &[0, 10, 70, 100, 96, 39, 0, 0, 0],
        ],
        // letter: 'Ğ' / 0x11e
        'Ğ' => &[
            &[0, 0, 126, 83, 0, 79, 127, 0, 0],
            &[0, 0, 34, 204, 227, 202, 33, 0, 0],
            &[0, 0, 0, 0, 22, 17, 0, 0, 0],
            &[0, 0, 104, 229, 255, 255, 220, 31, 0],
            &[0, 102, 247, 99, 14, 17, 80, 1, 0],
            &[2, 233, 116, 0, 0, 0, 0, 0, 0],
            &[51, 255, 25, 0, 0, 0, 0, 0, 0],
            &[81, 245, 0, 0, 43, 71, 71, 29, 0],
            &[87, 241, 0, 0, 133, 215, 247, 107, 0],
            &[65, 255, 16, 0, 0, 0, 204, 107, 0],
            &[11, 248, 95, 0, 0, 0, 204, 107, 0],
            &[0, 139, 237, 75, 7, 13, 212, 107, 0],
            &[0, 3, 135, 241, 254, 255, 229, 73, 0],
            &[0, 0, 0, 3, 26, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ğ' / 0x11f
        'ğ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 0, 0, 48, 30, 0, 0],
            &[0, 0, 180, 144, 113, 217, 32, 0, 0],
            &[0, 0, 10, 103, 118, 40, 0, 0, 0],
            &[0, 0, 80, 155, 143, 41, 99, 36, 0],
            &[0, 112, 245, 130, 113, 206, 240, 71, 0],
            &[5, 239, 102, 0, 0, 45, 255, 71, 0],
            &[43, 255, 34, 0, 0, 0, 234, 71, 0],
            &[54, 255, 23, 0, 0, 0, 222, 71, 0],
            &[26, 255, 59, 0, 0, 4, 246, 71, 0],
            &[0, 196, 185, 11, 0, 125, 255, 71, 0],
            &[0, 34, 207, 250, 240, 174, 230, 71, 0],
            &[0, 0, 0, 24, 11, 0, 240, 63, 0],
            &[0, 16, 0, 0, 0, 62, 252, 22, 0],
            &[0, 10, 70, 100, 96, 39, 0, 0, 0],
        ],
        // letter: 'Ġ' / 0x120
        'Ġ' => &[
            &[0, 0, 0, 66, 254, 63, 0, 0, 0],
            &[0, 0, 0, 23, 142, 22, 0, 0, 0],
            &[0, 0, 0, 0, 21, 17, 0, 0, 0],
            &[0, 0, 104, 229, 255, 255, 220, 31, 0],
            &[0, 102, 247, 99, 14, 17, 80, 1, 0],
            &[2, 233, 116, 0, 0, 0, 0, 0, 0],
            &[51, 255, 25, 0, 0, 0, 0, 0, 0],
            &[81, 245, 0, 0, 43, 71, 71, 29, 0],
            &[87, 241, 0, 0, 133, 215, 247, 107, 0],
            &[65, 255, 16, 0, 0, 0, 204, 107, 0],
            &[11, 248, 95, 0, 0, 0, 204, 107, 0],
            &[0, 139, 237, 75, 7, 13, 212, 107, 0],
            &[0, 3, 135, 241, 254, 255, 229, 73, 0],
            &[0, 0, 0, 3, 26, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 0, 0, 0, 0],
        ],
        // letter: 'ġ' / 0x121
        'ġ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 64, 122, 0, 0, 0, 0],
            &[0, 0, 0, 144, 236, 3, 0, 0, 0],
            &[0, 0, 0, 1, 11, 0, 0, 0, 0],
            &[0, 0, 80, 155, 143, 41, 99, 36, 0],
            &[0, 112, 245, 130, 113, 206, 240, 71, 0],
            &[5, 239, 102, 0, 0, 45, 255, 71, 0],
            &[43, 255, 34, 0, 0, 0, 234, 71, 0],
            &[54, 255, 23, 0, 0, 0, 222, 71, 0],
            &[26, 255, 59, 0, 0, 4, 246, 71, 0],
            &[0, 196, 185, 11, 0, 125, 255, 71, 0],
            &[0, 34, 207, 250, 240, 174, 230, 71, 0],
            &[0, 0, 0, 24, 11, 0, 240, 63, 0],
            &[0, 16, 0, 0, 0, 62, 252, 22, 0],
            &[0, 10, 70, 100, 96, 39, 0, 0, 0],
        ],
        // letter: 'Ģ' / 0x122
        'Ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 17, 0, 0, 0],
            &[0, 0, 104, 229, 255, 255, 220, 31, 0],
            &[0, 102, 247, 99, 14, 17, 80, 1, 0],
            &[2, 233, 116, 0, 0, 0, 0, 0, 0],
            &[51, 255, 25, 0, 0, 0, 0, 0, 0],
            &[81, 245, 0, 0, 43, 71, 71, 29, 0],
            &[87, 241, 0, 0, 133, 215, 247, 107, 0],
            &[65, 255, 16, 0, 0, 0, 204, 107, 0],
            &[11, 248, 95, 0, 0, 0, 204, 107, 0],
            &[0, 139, 237, 75, 7, 13, 212, 107, 0],
            &[0, 3, 135, 241, 254, 255, 229, 73, 0],
            &[0, 0, 0, 3, 59, 28, 0, 0, 0],
            &[0, 0, 0, 6, 246, 37, 0, 0, 0],
            &[0, 0, 0, 10, 6, 0, 0, 0, 0],
        ],
        // letter: 'ģ' / 0x123
        'ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 149, 7, 0, 0, 0],
            &[0, 0, 0, 85, 189, 0, 0, 0, 0],
            &[0, 0, 0, 85, 73, 0, 0, 0, 0],
            &[0, 0, 80, 155, 143, 41, 99, 36, 0],
            &[0, 112, 245, 130, 113, 206, 240, 71, 0],
            &[5, 239, 102, 0, 0, 45, 255, 71, 0],
            &[43, 255, 34, 0, 0, 0, 234, 71, 0],
            &[54, 255, 23, 0, 0, 0, 222, 71, 0],
            &[26, 255, 59, 0, 0, 4, 246, 71, 0],
            &[0, 196, 185, 11, 0, 125, 255, 71, 0],
            &[0, 34, 207, 250, 240, 174, 230, 71, 0],
            &[0, 0, 0, 24, 11, 0, 240, 63, 0],
            &[0, 16, 0, 0, 0, 62, 252, 22, 0],
            &[0, 10, 70, 100, 96, 39, 0, 0, 0],
        ],
        // letter: 'Ĥ' / 0x124
        'Ĥ' => &[
            &[0, 0, 21, 223, 230, 86, 0, 0, 0],
            &[0, 9, 190, 84, 29, 201, 61, 0, 0],
            &[0, 1, 4, 0, 0, 1, 3, 0, 0],
            &[5, 254, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 213, 200, 200, 200, 246, 107, 0],
            &[5, 255, 127, 86, 86, 86, 228, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 49, 0, 0, 0, 0],
        ],
        // letter: 'ĥ' / 0x125
        'ĥ' => &[
            &[21, 210, 83, 187, 115, 0, 0, 0, 0],
            &[52, 45, 0, 4, 89, 7, 0, 0, 0],
            &[0, 0, 144, 62, 0, 0, 0, 0, 0],
            &[0, 0, 225, 97, 0, 0, 0, 0, 0],
            &[0, 0, 225, 97, 0, 0, 0, 0, 0],
            &[0, 0, 225, 96, 83, 152, 132, 28, 0],
            &[0, 0, 225, 197, 168, 107, 193, 227, 12],
            &[0, 0, 225, 186, 0, 0, 17, 253, 67],
            &[0, 0, 225, 111, 0, 0, 0, 234, 89],
            &[0, 0, 225, 97, 0, 0, 0, 232, 89],
            &[0, 0, 225, 97, 0, 0, 0, 232, 89],
            &[0, 0, 225, 97, 0, 0, 0, 232, 89],
            &[0, 0, 225, 97, 0, 0, 0, 232, 89],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 217, 150, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ħ' / 0x126
        'Ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 254, 61, 0, 0, 0, 214, 107, 0],
            &[89, 255, 145, 84, 84, 84, 255, 191, 33],
            &[178, 255, 234, 173, 173, 173, 255, 255, 69],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 213, 200, 200, 200, 246, 107, 0],
            &[5, 255, 127, 86, 86, 86, 228, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ħ' / 0x127
        'ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 144, 62, 0, 0, 0, 0, 0, 0],
            &[97, 255, 194, 97, 82, 0, 0, 0, 0],
            &[139, 255, 236, 139, 117, 0, 0, 0, 0],
            &[0, 225, 96, 83, 152, 132, 28, 0, 0],
            &[0, 225, 197, 168, 107, 193, 227, 12, 0],
            &[0, 225, 186, 0, 0, 17, 253, 67, 0],
            &[0, 225, 111, 0, 0, 0, 234, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĩ' / 0x128
        'Ĩ' => &[
            &[0, 0, 127, 177, 67, 39, 126, 0, 0],
            &[0, 49, 179, 83, 203, 235, 57, 0, 0],
            &[0, 1, 2, 0, 0, 0, 0, 0, 0],
            &[0, 104, 244, 254, 254, 251, 199, 0, 0],
            &[0, 0, 0, 117, 210, 2, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 117, 210, 1, 0, 0, 0],
            &[0, 104, 244, 255, 255, 251, 199, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĩ' / 0x129
        'ĩ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 51, 0, 3, 50, 0, 0],
            &[0, 16, 220, 209, 197, 165, 126, 0, 0],
            &[0, 34, 66, 0, 73, 106, 6, 0, 0],
            &[0, 84, 128, 128, 128, 6, 0, 0, 0],
            &[0, 65, 115, 157, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 56, 255, 17, 0, 0, 0],
            &[0, 196, 242, 255, 255, 254, 237, 123, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ī' / 0x12a
        'Ī' => &[
            &[0, 0, 31, 35, 35, 35, 9, 0, 0],
            &[0, 0, 194, 219, 219, 219, 59, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 104, 244, 254, 254, 251, 199, 0, 0],
            &[0, 0, 0, 117, 210, 2, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 117, 210, 1, 0, 0, 0],
            &[0, 104, 244, 255, 255, 251, 199, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ī' / 0x12b
        'ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 146, 165, 165, 165, 44, 0, 0],
            &[0, 0, 78, 89, 89, 89, 24, 0, 0],
            &[0, 84, 128, 128, 128, 6, 0, 0, 0],
            &[0, 65, 115, 157, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 56, 255, 17, 0, 0, 0],
            &[0, 196, 242, 255, 255, 254, 237, 123, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => &[
            &[0, 0, 186, 24, 1, 145, 60, 0, 0],
            &[0, 0, 84, 224, 229, 160, 3, 0, 0],
            &[0, 0, 0, 0, 1, 0, 0, 0, 0],
            &[0, 104, 244, 254, 254, 251, 199, 0, 0],
            &[0, 0, 0, 117, 210, 2, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 117, 210, 1, 0, 0, 0],
            &[0, 104, 244, 255, 255, 251, 199, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĭ' / 0x12d
        'ĭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 0, 0, 48, 30, 0, 0],
            &[0, 0, 180, 144, 113, 217, 32, 0, 0],
            &[0, 0, 10, 103, 118, 40, 0, 0, 0],
            &[0, 84, 128, 128, 128, 6, 0, 0, 0],
            &[0, 65, 115, 157, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 56, 255, 17, 0, 0, 0],
            &[0, 196, 242, 255, 255, 254, 237, 123, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Į' / 0x12e
        'Į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 104, 244, 254, 254, 251, 199, 0, 0],
            &[0, 0, 0, 117, 210, 2, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 117, 210, 1, 0, 0, 0],
            &[0, 104, 244, 255, 255, 255, 201, 0, 0],
            &[0, 0, 0, 0, 55, 183, 5, 0, 0],
            &[0, 0, 0, 0, 172, 85, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 3, 0, 0],
        ],
        // letter: 'į' / 0x12f
        'į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 48, 136, 2, 0, 0, 0],
            &[0, 0, 0, 116, 249, 19, 0, 0, 0],
            &[0, 0, 0, 0, 13, 0, 0, 0, 0],
            &[0, 84, 128, 128, 128, 6, 0, 0, 0],
            &[0, 65, 115, 157, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 56, 255, 17, 0, 0, 0],
            &[0, 196, 242, 255, 255, 255, 245, 123, 0],
            &[0, 0, 0, 0, 17, 197, 30, 0, 0],
            &[0, 0, 0, 0, 107, 150, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 7, 0, 0],
        ],
        // letter: 'İ' / 0x130
        'İ' => &[
            &[0, 0, 0, 144, 236, 3, 0, 0, 0],
            &[0, 0, 0, 64, 123, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 104, 244, 254, 254, 251, 199, 0, 0],
            &[0, 0, 0, 117, 210, 2, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 113, 208, 0, 0, 0, 0],
            &[0, 0, 0, 117, 210, 1, 0, 0, 0],
            &[0, 104, 244, 255, 255, 251, 199, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 11, 0, 0, 0, 0],
        ],
        // letter: 'ı' / 0x131
        'ı' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 84, 128, 128, 128, 6, 0, 0, 0],
            &[0, 65, 115, 157, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 52, 255, 13, 0, 0, 0],
            &[0, 0, 0, 56, 255, 17, 0, 0, 0],
            &[0, 196, 242, 255, 255, 254, 237, 123, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĳ' / 0x132
        'Ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 152, 168, 0, 0, 66, 254, 0, 0],
            &[0, 153, 168, 0, 0, 66, 255, 0, 0],
            &[0, 153, 168, 0, 0, 66, 255, 0, 0],
            &[0, 153, 168, 0, 0, 66, 255, 0, 0],
            &[0, 153, 168, 0, 0, 66, 255, 0, 0],
            &[0, 153, 168, 0, 0, 66, 255, 0, 0],
            &[0, 153, 168, 0, 0, 67, 255, 0, 0],
            &[0, 30, 33, 0, 0, 90, 241, 0, 0],
            &[0, 52, 40, 12, 53, 218, 172, 0, 0],
            &[0, 121, 252, 255, 255, 190, 25, 0, 0],
            &[0, 0, 8, 30, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĳ' / 0x133
        'ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 51, 134, 1, 0, 74, 112, 0, 0],
            &[0, 120, 248, 16, 0, 159, 222, 0, 0],
            &[0, 0, 13, 0, 0, 3, 10, 0, 0],
            &[0, 46, 115, 0, 0, 57, 105, 0, 0],
            &[0, 92, 229, 0, 0, 113, 209, 0, 0],
            &[0, 92, 229, 0, 0, 113, 209, 0, 0],
            &[0, 92, 229, 0, 0, 113, 209, 0, 0],
            &[0, 92, 229, 0, 0, 113, 209, 0, 0],
            &[0, 92, 229, 0, 0, 113, 209, 0, 0],
            &[0, 92, 229, 0, 0, 113, 209, 0, 0],
            &[0, 92, 229, 0, 0, 113, 209, 0, 0],
            &[0, 0, 0, 0, 0, 118, 207, 0, 0],
            &[0, 0, 0, 0, 0, 178, 174, 0, 0],
            &[0, 9, 79, 103, 85, 20, 0, 0, 0],
        ],
        // letter: 'Ĵ' / 0x134
        'Ĵ' => &[
            &[0, 0, 0, 75, 230, 228, 27, 0, 0],
            &[0, 0, 48, 202, 33, 78, 198, 16, 0],
            &[0, 0, 2, 2, 0, 0, 3, 1, 0],
            &[0, 0, 0, 0, 191, 129, 0, 0, 0],
            &[0, 0, 0, 0, 191, 130, 0, 0, 0],
            &[0, 0, 0, 0, 191, 130, 0, 0, 0],
            &[0, 0, 0, 0, 191, 130, 0, 0, 0],
            &[0, 0, 0, 0, 191, 130, 0, 0, 0],
            &[0, 0, 0, 0, 191, 130, 0, 0, 0],
            &[0, 0, 0, 0, 192, 130, 0, 0, 0],
            &[0, 0, 0, 0, 212, 117, 0, 0, 0],
            &[55, 38, 18, 107, 254, 55, 0, 0, 0],
            &[126, 252, 255, 246, 124, 0, 0, 0, 0],
            &[0, 9, 29, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 46, 31, 0, 0, 0],
        ],
        // letter: 'ĵ' / 0x135
        'ĵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 182, 116, 0, 0, 0],
            &[0, 0, 10, 199, 117, 210, 84, 0, 0],
            &[0, 0, 64, 72, 0, 15, 118, 7, 0],
            &[0, 19, 128, 128, 128, 71, 0, 0, 0],
            &[0, 14, 105, 124, 220, 142, 0, 0, 0],
            &[0, 0, 0, 0, 179, 142, 0, 0, 0],
            &[0, 0, 0, 0, 179, 142, 0, 0, 0],
            &[0, 0, 0, 0, 179, 142, 0, 0, 0],
            &[0, 0, 0, 0, 179, 142, 0, 0, 0],
            &[0, 0, 0, 0, 179, 142, 0, 0, 0],
            &[0, 0, 0, 0, 179, 142, 0, 0, 0],
            &[0, 0, 0, 0, 184, 141, 0, 0, 0],
            &[0, 0, 0, 10, 229, 109, 0, 0, 0],
            &[8, 79, 103, 79, 8, 0, 0, 0, 0],
        ],
        // letter: 'Ķ' / 0x136
        'Ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 254, 61, 0, 0, 64, 247, 56, 0],
            &[5, 255, 61, 0, 22, 230, 119, 0, 0],
            &[5, 255, 61, 1, 185, 184, 1, 0, 0],
            &[5, 255, 61, 122, 230, 22, 0, 0, 0],
            &[5, 255, 114, 251, 102, 0, 0, 0, 0],
            &[5, 255, 240, 186, 211, 5, 0, 0, 0],
            &[5, 255, 113, 11, 225, 122, 0, 0, 0],
            &[5, 255, 61, 0, 77, 247, 37, 0, 0],
            &[5, 255, 61, 0, 0, 173, 190, 0, 0],
            &[5, 255, 61, 0, 0, 27, 242, 97, 0],
            &[0, 0, 0, 9, 36, 0, 0, 0, 0],
            &[0, 0, 0, 96, 194, 0, 0, 0, 0],
            &[0, 0, 0, 16, 0, 0, 0, 0, 0],
        ],
        // letter: 'ķ' / 0x137
        'ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 86, 119, 0, 0, 0, 0, 0, 0],
            &[0, 135, 186, 0, 0, 0, 0, 0, 0],
            &[0, 135, 186, 0, 0, 0, 0, 0, 0],
            &[0, 135, 186, 0, 0, 4, 118, 71, 0],
            &[0, 135, 186, 0, 1, 161, 205, 14, 0],
            &[0, 135, 186, 0, 137, 217, 21, 0, 0],
            &[0, 135, 178, 110, 243, 31, 0, 0, 0],
            &[0, 135, 221, 210, 234, 95, 0, 0, 0],
            &[0, 135, 223, 21, 75, 244, 41, 0, 0],
            &[0, 135, 186, 0, 0, 144, 211, 9, 0],
            &[0, 135, 186, 0, 0, 7, 207, 155, 0],
            &[0, 0, 0, 0, 31, 13, 0, 0, 0],
            &[0, 0, 0, 4, 243, 43, 0, 0, 0],
            &[0, 0, 0, 10, 6, 0, 0, 0, 0],
        ],
        // letter: 'ĸ' / 0x138
        'ĸ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 94, 0, 0, 4, 118, 71, 0],
            &[0, 135, 186, 0, 1, 161, 205, 14, 0],
            &[0, 135, 186, 0, 137, 217, 21, 0, 0],
            &[0, 135, 178, 110, 243, 31, 0, 0, 0],
            &[0, 135, 221, 211, 234, 95, 0, 0, 0],
            &[0, 135, 223, 21, 75, 244, 41, 0, 0],
            &[0, 135, 186, 0, 0, 144, 211, 9, 0],
            &[0, 135, 186, 0, 0, 7, 207, 155, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĺ' / 0x139
        'Ĺ' => &[
            &[0, 0, 103, 231, 30, 0, 0, 0, 0],
            &[0, 29, 214, 44, 0, 0, 0, 0, 0],
            &[0, 1, 3, 0, 0, 0, 0, 0, 0],
            &[0, 99, 221, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 228, 41, 41, 41, 41, 8, 0],
            &[0, 99, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 55, 26, 0, 0, 0, 0],
        ],
        // letter: 'ĺ' / 0x13a
        'ĺ' => &[
            &[0, 0, 0, 8, 212, 99, 0, 0, 0],
            &[0, 0, 0, 32, 63, 0, 0, 0, 0],
            &[0, 130, 163, 163, 163, 20, 0, 0, 0],
            &[0, 51, 81, 118, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 38, 255, 35, 0, 0, 0],
            &[0, 196, 241, 255, 255, 254, 238, 123, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 217, 46, 0, 0],
        ],
        // letter: 'Ļ' / 0x13b
        'Ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 99, 221, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 228, 41, 41, 41, 41, 8, 0],
            &[0, 99, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 36, 9, 0, 0, 0],
            &[0, 0, 0, 24, 247, 19, 0, 0, 0],
            &[0, 0, 0, 13, 3, 0, 0, 0, 0],
        ],
        // letter: 'ļ' / 0x13c
        'ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 163, 163, 163, 20, 0, 0, 0],
            &[0, 51, 81, 118, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 38, 255, 35, 0, 0, 0],
            &[0, 196, 241, 255, 255, 254, 238, 123, 0],
            &[0, 0, 0, 3, 37, 5, 0, 0, 0],
            &[0, 0, 0, 53, 232, 5, 0, 0, 0],
            &[0, 0, 0, 15, 1, 0, 0, 0, 0],
        ],
        // letter: 'Ľ' / 0x13d
        'Ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 99, 221, 0, 0, 62, 231, 2, 0],
            &[0, 99, 222, 0, 0, 118, 120, 0, 0],
            &[0, 99, 222, 0, 0, 38, 7, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 228, 41, 41, 41, 41, 8, 0],
            &[0, 99, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ľ' / 0x13e
        'ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 163, 163, 163, 20, 63, 132, 0],
            &[0, 51, 81, 118, 255, 31, 140, 122, 0],
            &[0, 0, 0, 34, 255, 31, 111, 11, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 38, 255, 35, 0, 0, 0],
            &[0, 196, 241, 255, 255, 254, 238, 123, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 99, 221, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 18, 0, 0],
            &[0, 99, 222, 0, 0, 104, 253, 25, 0],
            &[0, 99, 222, 0, 0, 51, 164, 7, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 222, 0, 0, 0, 0, 0, 0],
            &[0, 99, 228, 41, 41, 41, 41, 8, 0],
            &[0, 99, 255, 255, 255, 255, 255, 53, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŀ' / 0x140
        'ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[130, 163, 163, 163, 20, 0, 0, 0, 0],
            &[51, 81, 118, 255, 31, 0, 0, 0, 0],
            &[0, 0, 34, 255, 31, 0, 0, 0, 0],
            &[0, 0, 34, 255, 31, 0, 0, 0, 0],
            &[0, 0, 34, 255, 31, 0, 40, 157, 12],
            &[0, 0, 34, 255, 31, 0, 91, 255, 40],
            &[0, 0, 34, 255, 31, 0, 0, 26, 0],
            &[0, 0, 34, 255, 31, 0, 0, 0, 0],
            &[0, 0, 34, 255, 31, 0, 0, 0, 0],
            &[0, 0, 38, 255, 35, 0, 0, 0, 0],
            &[196, 241, 255, 255, 254, 238, 123, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ł' / 0x141
        'Ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 211, 109, 0, 0, 0, 0, 0],
            &[0, 0, 212, 110, 0, 0, 0, 0, 0],
            &[0, 0, 212, 110, 0, 0, 0, 0, 0],
            &[0, 0, 212, 110, 23, 21, 0, 0, 0],
            &[0, 0, 217, 224, 235, 86, 0, 0, 0],
            &[0, 67, 255, 255, 38, 0, 0, 0, 0],
            &[17, 204, 255, 110, 0, 0, 0, 0, 0],
            &[0, 0, 212, 110, 0, 0, 0, 0, 0],
            &[0, 0, 212, 133, 41, 41, 41, 31, 0],
            &[0, 0, 212, 255, 255, 255, 255, 197, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ł' / 0x142
        'ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 130, 163, 163, 163, 20, 0, 0, 0],
            &[0, 51, 81, 118, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 109, 113, 0, 0],
            &[0, 0, 0, 63, 255, 242, 58, 0, 0],
            &[0, 0, 111, 255, 255, 36, 0, 0, 0],
            &[0, 0, 101, 65, 255, 31, 0, 0, 0],
            &[0, 0, 0, 34, 255, 31, 0, 0, 0],
            &[0, 0, 0, 38, 255, 35, 0, 0, 0],
            &[0, 196, 241, 255, 255, 254, 238, 123, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ń' / 0x143
        'Ń' => &[
            &[0, 0, 0, 0, 99, 233, 32, 0, 0],
            &[0, 0, 0, 27, 214, 46, 0, 0, 0],
            &[0, 0, 0, 1, 3, 0, 0, 0, 0],
            &[5, 254, 218, 1, 0, 0, 203, 107, 0],
            &[5, 255, 222, 76, 0, 0, 204, 107, 0],
            &[5, 255, 130, 188, 0, 0, 204, 107, 0],
            &[5, 255, 53, 235, 45, 0, 204, 107, 0],
            &[5, 255, 50, 133, 156, 0, 204, 107, 0],
            &[5, 255, 50, 26, 241, 20, 203, 107, 0],
            &[5, 255, 50, 0, 164, 124, 199, 107, 0],
            &[5, 255, 50, 0, 52, 230, 193, 107, 0],
            &[5, 255, 50, 0, 0, 195, 247, 107, 0],
            &[5, 255, 50, 0, 0, 83, 255, 107, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 55, 27, 0, 0],
        ],
        // letter: 'ń' / 0x144
        'ń' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 181, 77, 0, 0],
            &[0, 0, 0, 0, 156, 171, 3, 0, 0],
            &[0, 0, 0, 16, 114, 4, 0, 0, 0],
            &[0, 113, 24, 92, 152, 136, 34, 0, 0],
            &[0, 225, 199, 163, 107, 181, 232, 15, 0],
            &[0, 225, 177, 0, 0, 10, 251, 71, 0],
            &[0, 225, 113, 0, 0, 0, 233, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ņ' / 0x145
        'Ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 254, 218, 1, 0, 0, 203, 107, 0],
            &[5, 255, 222, 76, 0, 0, 204, 107, 0],
            &[5, 255, 130, 188, 0, 0, 204, 107, 0],
            &[5, 255, 53, 235, 45, 0, 204, 107, 0],
            &[5, 255, 50, 133, 156, 0, 204, 107, 0],
            &[5, 255, 50, 26, 241, 20, 203, 107, 0],
            &[5, 255, 50, 0, 164, 124, 199, 107, 0],
            &[5, 255, 50, 0, 52, 230, 193, 107, 0],
            &[5, 255, 50, 0, 0, 195, 247, 107, 0],
            &[5, 255, 50, 0, 0, 83, 255, 107, 0],
            &[0, 0, 0, 14, 30, 0, 0, 0, 0],
            &[0, 0, 0, 131, 158, 0, 0, 0, 0],
            &[0, 0, 0, 16, 0, 0, 0, 0, 0],
        ],
        // letter: 'ņ' / 0x146
        'ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 113, 24, 92, 152, 136, 34, 0, 0],
            &[0, 225, 199, 163, 107, 181, 232, 15, 0],
            &[0, 225, 177, 0, 0, 10, 251, 71, 0],
            &[0, 225, 113, 0, 0, 0, 233, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 0, 0, 11, 34, 0, 0, 0, 0],
            &[0, 0, 0, 110, 180, 0, 0, 0, 0],
            &[0, 0, 0, 16, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ň' / 0x147
        'Ň' => &[
            &[0, 0, 153, 157, 58, 217, 48, 0, 0],
            &[0, 0, 3, 189, 252, 68, 0, 0, 0],
            &[0, 0, 0, 2, 5, 0, 0, 0, 0],
            &[5, 254, 218, 1, 0, 0, 203, 107, 0],
            &[5, 255, 222, 76, 0, 0, 204, 107, 0],
            &[5, 255, 130, 188, 0, 0, 204, 107, 0],
            &[5, 255, 53, 235, 45, 0, 204, 107, 0],
            &[5, 255, 50, 133, 156, 0, 204, 107, 0],
            &[5, 255, 50, 26, 241, 20, 203, 107, 0],
            &[5, 255, 50, 0, 164, 124, 199, 107, 0],
            &[5, 255, 50, 0, 52, 230, 193, 107, 0],
            &[5, 255, 50, 0, 0, 195, 247, 107, 0],
            &[5, 255, 50, 0, 0, 83, 255, 107, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 48, 0, 0, 20, 35, 0, 0],
        ],
        // letter: 'ň' / 0x148
        'ň' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 147, 60, 0, 112, 99, 0, 0],
            &[0, 0, 34, 227, 177, 188, 6, 0, 0],
            &[0, 0, 0, 60, 131, 21, 0, 0, 0],
            &[0, 113, 24, 92, 152, 136, 34, 0, 0],
            &[0, 225, 199, 163, 107, 181, 232, 15, 0],
            &[0, 225, 177, 0, 0, 10, 251, 71, 0],
            &[0, 225, 113, 0, 0, 0, 233, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 225, 97, 0, 0, 0, 232, 89, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŉ' / 0x149
        'ŉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[25, 254, 104, 0, 0, 0, 0, 0, 0],
            &[72, 253, 27, 0, 0, 0, 0, 0, 0],
            &[125, 183, 86, 48, 71, 150, 139, 37, 0],
            &[72, 46, 171, 201, 184, 108, 184, 234, 16],
            &[0, 0, 171, 225, 3, 0, 11, 251, 71],
            &[0, 0, 171, 167, 0, 0, 0, 233, 89],
            &[0, 0, 171, 151, 0, 0, 0, 232, 89],
            &[0, 0, 171, 150, 0, 0, 0, 232, 89],
            &[0, 0, 171, 150, 0, 0, 0, 232, 89],
            &[0, 0, 171, 150, 0, 0, 0, 232, 89],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 254, 218, 1, 0, 0, 203, 107, 0],
            &[5, 255, 222, 76, 0, 0, 204, 107, 0],
            &[5, 255, 130, 188, 0, 0, 204, 107, 0],
            &[5, 255, 53, 235, 45, 0, 204, 107, 0],
            &[5, 255, 50, 133, 156, 0, 204, 107, 0],
            &[5, 255, 50, 26, 241, 20, 203, 107, 0],
            &[5, 255, 50, 0, 164, 124, 199, 107, 0],
            &[5, 255, 50, 0, 52, 230, 193, 107, 0],
            &[5, 255, 50, 0, 0, 195, 255, 123, 0],
            &[5, 255, 50, 0, 0, 83, 255, 214, 0],
            &[0, 0, 0, 0, 0, 0, 227, 85, 0],
            &[0, 0, 0, 39, 122, 176, 237, 20, 0],
            &[0, 0, 0, 36, 160, 139, 36, 0, 0],
        ],
        // letter: 'ŋ' / 0x14b
        'ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 113, 24, 92, 152, 136, 34, 0, 0],
            &[0, 225, 199, 163, 107, 181, 232, 15, 0],
            &[0, 225, 177, 0, 0, 10, 251, 71, 0],
            &[0, 225, 113, 0, 0, 0, 233, 89, 0],
            &[0, 225, 97, 0, 0, 0, 255, 133, 0],
            &[0, 225, 97, 0, 0, 0, 255, 179, 0],
            &[0, 225, 97, 0, 0, 0, 255, 179, 0],
            &[0, 225, 97, 0, 0, 0, 255, 179, 0],
            &[0, 0, 0, 0, 0, 0, 232, 89, 0],
            &[0, 0, 0, 0, 0, 3, 243, 71, 0],
            &[0, 0, 0, 0, 47, 84, 23, 0, 0],
        ],
        // letter: 'Ō' / 0x14c
        'Ō' => &[
            &[0, 0, 31, 35, 35, 35, 9, 0, 0],
            &[0, 0, 194, 219, 219, 219, 59, 0, 0],
            &[0, 0, 0, 16, 32, 3, 0, 0, 0],
            &[0, 12, 175, 255, 253, 228, 68, 0, 0],
            &[0, 157, 206, 31, 11, 128, 242, 21, 0],
            &[13, 250, 67, 0, 0, 0, 224, 113, 0],
            &[63, 253, 9, 0, 0, 0, 163, 168, 0],
            &[86, 240, 0, 0, 0, 0, 141, 191, 0],
            &[88, 241, 0, 0, 0, 0, 138, 190, 0],
            &[65, 254, 10, 0, 0, 0, 161, 166, 0],
            &[16, 251, 69, 0, 0, 0, 220, 111, 0],
            &[0, 160, 209, 36, 9, 129, 239, 19, 0],
            &[0, 13, 174, 254, 255, 222, 61, 0, 0],
            &[0, 0, 0, 15, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ō' / 0x14d
        'ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 146, 165, 165, 165, 44, 0, 0],
            &[0, 0, 78, 89, 89, 89, 24, 0, 0],
            &[0, 0, 56, 142, 154, 97, 4, 0, 0],
            &[0, 95, 245, 136, 109, 207, 185, 2, 0],
            &[4, 235, 102, 0, 0, 16, 240, 82, 0],
            &[41, 255, 25, 0, 0, 0, 182, 142, 0],
            &[52, 255, 10, 0, 0, 0, 170, 155, 0],
            &[20, 254, 51, 0, 0, 0, 210, 123, 0],
            &[0, 177, 185, 13, 0, 99, 248, 34, 0],
            &[0, 20, 180, 246, 237, 229, 79, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => &[
            &[0, 0, 186, 24, 1, 145, 60, 0, 0],
            &[0, 0, 84, 224, 229, 160, 3, 0, 0],
            &[0, 0, 0, 16, 33, 3, 0, 0, 0],
            &[0, 12, 175, 255, 253, 228, 68, 0, 0],
            &[0, 157, 206, 31, 11, 128, 242, 21, 0],
            &[13, 250, 67, 0, 0, 0, 224, 113, 0],
            &[63, 253, 9, 0, 0, 0, 163, 168, 0],
            &[86, 240, 0, 0, 0, 0, 141, 191, 0],
            &[88, 241, 0, 0, 0, 0, 138, 190, 0],
            &[65, 254, 10, 0, 0, 0, 161, 166, 0],
            &[16, 251, 69, 0, 0, 0, 220, 111, 0],
            &[0, 160, 209, 36, 9, 129, 239, 19, 0],
            &[0, 13, 174, 254, 255, 222, 61, 0, 0],
            &[0, 0, 0, 15, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŏ' / 0x14f
        'ŏ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 0, 0, 48, 30, 0, 0],
            &[0, 0, 180, 144, 113, 217, 32, 0, 0],
            &[0, 0, 10, 103, 118, 40, 0, 0, 0],
            &[0, 0, 56, 142, 154, 97, 4, 0, 0],
            &[0, 95, 245, 136, 109, 207, 185, 2, 0],
            &[4, 235, 102, 0, 0, 16, 240, 82, 0],
            &[41, 255, 25, 0, 0, 0, 182, 142, 0],
            &[52, 255, 10, 0, 0, 0, 170, 155, 0],
            &[20, 254, 51, 0, 0, 0, 210, 123, 0],
            &[0, 177, 185, 13, 0, 99, 248, 34, 0],
            &[0, 20, 180, 246, 237, 229, 79, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ő' / 0x150
        'Ő' => &[
            &[0, 0, 53, 240, 38, 188, 142, 0, 0],
            &[0, 1, 190, 65, 85, 169, 3, 0, 0],
            &[0, 0, 4, 16, 36, 4, 0, 0, 0],
            &[0, 12, 175, 255, 253, 228, 68, 0, 0],
            &[0, 157, 206, 31, 11, 128, 242, 21, 0],
            &[13, 250, 67, 0, 0, 0, 224, 113, 0],
            &[63, 253, 9, 0, 0, 0, 163, 168, 0],
            &[86, 240, 0, 0, 0, 0, 141, 191, 0],
            &[88, 241, 0, 0, 0, 0, 138, 190, 0],
            &[65, 254, 10, 0, 0, 0, 161, 166, 0],
            &[16, 251, 69, 0, 0, 0, 220, 111, 0],
            &[0, 160, 209, 36, 9, 129, 239, 19, 0],
            &[0, 13, 174, 254, 255, 222, 61, 0, 0],
            &[0, 0, 0, 15, 26, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 48, 28, 22, 53, 0, 0],
        ],
        // letter: 'ő' / 0x151
        'ő' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 177, 65, 100, 151, 0, 0],
            &[0, 0, 127, 171, 25, 227, 48, 0, 0],
            &[0, 2, 111, 5, 65, 53, 0, 0, 0],
            &[0, 0, 56, 142, 154, 97, 4, 0, 0],
            &[0, 95, 245, 136, 109, 207, 185, 2, 0],
            &[4, 235, 102, 0, 0, 16, 240, 82, 0],
            &[41, 255, 25, 0, 0, 0, 182, 142, 0],
            &[52, 255, 10, 0, 0, 0, 170, 155, 0],
            &[20, 254, 51, 0, 0, 0, 210, 123, 0],
            &[0, 177, 185, 13, 0, 99, 248, 34, 0],
            &[0, 20, 180, 246, 237, 229, 79, 0, 0],
            &[0, 0, 0, 15, 28, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Œ' / 0x152
        'Œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 3, 28, 8, 0, 0, 0, 0],
            &[0, 50, 227, 254, 255, 254, 254, 249, 0],
            &[0, 213, 151, 7, 179, 156, 31, 31, 0],
            &[45, 255, 30, 0, 168, 142, 0, 0, 0],
            &[89, 239, 0, 0, 168, 142, 0, 0, 0],
            &[107, 223, 0, 0, 168, 231, 200, 142, 0],
            &[105, 221, 0, 0, 168, 179, 82, 59, 0],
            &[88, 238, 0, 0, 168, 142, 0, 0, 0],
            &[45, 255, 26, 0, 168, 142, 0, 0, 0],
            &[0, 217, 144, 6, 180, 156, 30, 30, 0],
            &[0, 58, 234, 255, 255, 255, 255, 250, 0],
            &[0, 0, 4, 28, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'œ' / 0x153
        'œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 62, 152, 103, 4, 117, 144, 24, 0],
            &[38, 239, 113, 208, 187, 171, 151, 181, 0],
            &[128, 159, 0, 80, 255, 36, 12, 250, 7],
            &[161, 120, 0, 45, 255, 115, 112, 249, 27],
            &[171, 114, 0, 36, 255, 119, 117, 117, 14],
            &[148, 135, 0, 54, 255, 24, 0, 0, 0],
            &[83, 211, 5, 127, 249, 107, 0, 25, 0],
            &[3, 178, 238, 222, 63, 232, 223, 181, 0],
            &[0, 0, 24, 5, 0, 6, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŕ' / 0x154
        'Ŕ' => &[
            &[0, 0, 0, 0, 117, 224, 23, 0, 0],
            &[0, 0, 0, 38, 213, 36, 0, 0, 0],
            &[0, 0, 0, 2, 2, 0, 0, 0, 0],
            &[0, 170, 254, 253, 237, 177, 38, 0, 0],
            &[0, 171, 161, 26, 53, 184, 221, 3, 0],
            &[0, 171, 150, 0, 0, 48, 255, 36, 0],
            &[0, 171, 150, 0, 0, 62, 255, 24, 0],
            &[0, 171, 169, 47, 73, 212, 163, 0, 0],
            &[0, 171, 246, 233, 251, 153, 5, 0, 0],
            &[0, 171, 150, 0, 126, 215, 5, 0, 0],
            &[0, 171, 150, 0, 8, 223, 118, 0, 0],
            &[0, 171, 150, 0, 0, 81, 244, 28, 0],
            &[0, 171, 150, 0, 0, 0, 186, 170, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 55, 23, 0, 0],
        ],
        // letter: 'ŕ' / 0x155
        'ŕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 127, 150, 0, 0],
            &[0, 0, 0, 0, 58, 232, 40, 0, 0],
            &[0, 0, 0, 0, 96, 38, 0, 0, 0],
            &[0, 117, 128, 128, 24, 94, 157, 121, 0],
            &[0, 93, 125, 238, 169, 204, 127, 132, 0],
            &[0, 0, 0, 217, 220, 9, 0, 0, 0],
            &[0, 0, 0, 217, 132, 0, 0, 0, 0],
            &[0, 0, 0, 217, 109, 0, 0, 0, 0],
            &[0, 0, 0, 217, 108, 0, 0, 0, 0],
            &[0, 0, 1, 219, 111, 0, 0, 0, 0],
            &[28, 229, 251, 255, 255, 242, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŗ' / 0x156
        'Ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 170, 254, 253, 237, 177, 38, 0, 0],
            &[0, 171, 161, 26, 53, 184, 221, 3, 0],
            &[0, 171, 150, 0, 0, 48, 255, 36, 0],
            &[0, 171, 150, 0, 0, 62, 255, 24, 0],
            &[0, 171, 169, 47, 73, 212, 163, 0, 0],
            &[0, 171, 246, 233, 251, 153, 5, 0, 0],
            &[0, 171, 150, 0, 126, 215, 5, 0, 0],
            &[0, 171, 150, 0, 8, 223, 118, 0, 0],
            &[0, 171, 150, 0, 0, 81, 244, 28, 0],
            &[0, 171, 150, 0, 0, 0, 186, 170, 0],
            &[0, 0, 0, 0, 36, 9, 0, 0, 0],
            &[0, 0, 0, 24, 247, 19, 0, 0, 0],
            &[0, 0, 0, 13, 3, 0, 0, 0, 0],
        ],
        // letter: 'ŗ' / 0x157
        'ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 117, 128, 128, 24, 94, 157, 121, 0],
            &[0, 93, 125, 238, 169, 204, 127, 132, 0],
            &[0, 0, 0, 217, 220, 9, 0, 0, 0],
            &[0, 0, 0, 217, 132, 0, 0, 0, 0],
            &[0, 0, 0, 217, 109, 0, 0, 0, 0],
            &[0, 0, 0, 217, 108, 0, 0, 0, 0],
            &[0, 0, 1, 219, 111, 0, 0, 0, 0],
            &[28, 229, 251, 255, 255, 242, 135, 0, 0],
            &[0, 0, 0, 30, 15, 0, 0, 0, 0],
            &[0, 0, 1, 234, 55, 0, 0, 0, 0],
            &[0, 0, 8, 8, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ř' / 0x158
        'Ř' => &[
            &[0, 2, 169, 142, 67, 215, 37, 0, 0],
            &[0, 0, 7, 203, 249, 54, 0, 0, 0],
            &[0, 0, 0, 2, 5, 0, 0, 0, 0],
            &[0, 170, 254, 253, 237, 177, 38, 0, 0],
            &[0, 171, 161, 26, 53, 184, 221, 3, 0],
            &[0, 171, 150, 0, 0, 48, 255, 36, 0],
            &[0, 171, 150, 0, 0, 62, 255, 24, 0],
            &[0, 171, 169, 47, 73, 212, 163, 0, 0],
            &[0, 171, 246, 233, 251, 153, 5, 0, 0],
            &[0, 171, 150, 0, 126, 215, 5, 0, 0],
            &[0, 171, 150, 0, 8, 223, 118, 0, 0],
            &[0, 171, 150, 0, 0, 81, 244, 28, 0],
            &[0, 171, 150, 0, 0, 0, 186, 170, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 45, 0, 0, 24, 31, 0, 0],
        ],
        // letter: 'ř' / 0x159
        'ř' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 77, 128, 2, 46, 157, 7, 0],
            &[0, 0, 1, 167, 183, 230, 50, 0, 0],
            &[0, 0, 0, 11, 129, 73, 0, 0, 0],
            &[0, 117, 128, 128, 24, 94, 157, 121, 0],
            &[0, 93, 125, 238, 169, 204, 127, 132, 0],
            &[0, 0, 0, 217, 220, 9, 0, 0, 0],
            &[0, 0, 0, 217, 132, 0, 0, 0, 0],
            &[0, 0, 0, 217, 109, 0, 0, 0, 0],
            &[0, 0, 0, 217, 108, 0, 0, 0, 0],
            &[0, 0, 1, 219, 111, 0, 0, 0, 0],
            &[28, 229, 251, 255, 255, 242, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ś' / 0x15a
        'Ś' => &[
            &[0, 0, 0, 0, 51, 247, 66, 0, 0],
            &[0, 0, 0, 6, 202, 80, 0, 0, 0],
            &[0, 0, 0, 8, 33, 9, 0, 0, 0],
            &[0, 2, 145, 247, 255, 255, 208, 16, 0],
            &[0, 96, 245, 67, 5, 24, 80, 0, 0],
            &[0, 142, 196, 0, 0, 0, 0, 0, 0],
            &[0, 101, 242, 51, 0, 0, 0, 0, 0],
            &[0, 4, 159, 252, 170, 48, 0, 0, 0],
            &[0, 0, 0, 50, 173, 252, 151, 1, 0],
            &[0, 0, 0, 0, 0, 64, 255, 68, 0],
            &[0, 0, 0, 0, 0, 1, 242, 91, 0],
            &[0, 84, 36, 3, 9, 112, 250, 37, 0],
            &[0, 154, 249, 251, 253, 226, 87, 0, 0],
            &[0, 0, 3, 25, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 38, 0, 0],
        ],
        // letter: 'ś' / 0x15b
        'ś' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 185, 59, 0, 0],
            &[0, 0, 0, 0, 181, 148, 0, 0, 0],
            &[0, 0, 0, 28, 104, 0, 0, 0, 0],
            &[0, 0, 51, 138, 159, 136, 60, 0, 0],
            &[0, 46, 247, 143, 109, 137, 116, 0, 0],
            &[0, 93, 230, 3, 0, 0, 0, 0, 0],
            &[0, 19, 209, 204, 82, 2, 0, 0, 0],
            &[0, 0, 7, 104, 221, 219, 56, 0, 0],
            &[0, 0, 0, 0, 3, 150, 218, 0, 0],
            &[0, 53, 33, 0, 1, 139, 213, 0, 0],
            &[0, 101, 244, 244, 242, 224, 66, 0, 0],
            &[0, 0, 3, 25, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => &[
            &[0, 0, 1, 178, 226, 154, 0, 0, 0],
            &[0, 0, 141, 142, 7, 166, 120, 0, 0],
            &[0, 0, 5, 7, 28, 9, 5, 0, 0],
            &[0, 2, 145, 247, 255, 255, 208, 16, 0],
            &[0, 96, 245, 67, 5, 24, 80, 0, 0],
            &[0, 142, 196, 0, 0, 0, 0, 0, 0],
            &[0, 101, 242, 51, 0, 0, 0, 0, 0],
            &[0, 4, 159, 252, 170, 48, 0, 0, 0],
            &[0, 0, 0, 50, 173, 252, 151, 1, 0],
            &[0, 0, 0, 0, 0, 64, 255, 68, 0],
            &[0, 0, 0, 0, 0, 1, 242, 91, 0],
            &[0, 84, 36, 3, 9, 112, 250, 37, 0],
            &[0, 154, 249, 251, 253, 226, 87, 0, 0],
            &[0, 0, 3, 25, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 55, 8, 0, 0, 0],
        ],
        // letter: 'ŝ' / 0x15d
        'ŝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 116, 182, 28, 0, 0, 0],
            &[0, 0, 83, 207, 120, 200, 11, 0, 0],
            &[0, 5, 117, 14, 0, 73, 67, 0, 0],
            &[0, 0, 51, 138, 159, 136, 60, 0, 0],
            &[0, 46, 247, 143, 109, 137, 116, 0, 0],
            &[0, 93, 230, 3, 0, 0, 0, 0, 0],
            &[0, 19, 209, 204, 82, 2, 0, 0, 0],
            &[0, 0, 7, 104, 221, 219, 56, 0, 0],
            &[0, 0, 0, 0, 3, 150, 218, 0, 0],
            &[0, 53, 33, 0, 1, 139, 213, 0, 0],
            &[0, 101, 244, 244, 242, 224, 66, 0, 0],
            &[0, 0, 3, 25, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ş' / 0x15e
        'Ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 28, 9, 0, 0, 0],
            &[0, 2, 145, 247, 255, 255, 208, 16, 0],
            &[0, 96, 245, 67, 5, 24, 80, 0, 0],
            &[0, 142, 196, 0, 0, 0, 0, 0, 0],
            &[0, 101, 242, 51, 0, 0, 0, 0, 0],
            &[0, 4, 159, 252, 170, 48, 0, 0, 0],
            &[0, 0, 0, 50, 173, 252, 151, 1, 0],
            &[0, 0, 0, 0, 0, 64, 255, 68, 0],
            &[0, 0, 0, 0, 0, 1, 242, 91, 0],
            &[0, 84, 36, 3, 9, 112, 250, 37, 0],
            &[0, 154, 249, 254, 255, 226, 87, 0, 0],
            &[0, 0, 3, 132, 144, 0, 0, 0, 0],
            &[0, 0, 0, 56, 203, 91, 0, 0, 0],
            &[0, 0, 20, 86, 47, 0, 0, 0, 0],
        ],
        // letter: 'ş' / 0x15f
        'ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 51, 138, 159, 136, 60, 0, 0],
            &[0, 46, 247, 143, 109, 137, 116, 0, 0],
            &[0, 93, 230, 3, 0, 0, 0, 0, 0],
            &[0, 19, 209, 204, 82, 2, 0, 0, 0],
            &[0, 0, 7, 104, 221, 219, 56, 0, 0],
            &[0, 0, 0, 0, 3, 150, 218, 0, 0],
            &[0, 53, 33, 0, 1, 139, 213, 0, 0],
            &[0, 101, 244, 246, 255, 224, 66, 0, 0],
            &[0, 0, 3, 115, 169, 0, 0, 0, 0],
            &[0, 0, 0, 48, 193, 108, 0, 0, 0],
            &[0, 0, 14, 87, 52, 0, 0, 0, 0],
        ],
        // letter: 'Š' / 0x160
        'Š' => &[
            &[0, 0, 104, 194, 44, 207, 85, 0, 0],
            &[0, 0, 0, 142, 254, 117, 0, 0, 0],
            &[0, 0, 0, 8, 34, 10, 0, 0, 0],
            &[0, 2, 145, 247, 255, 255, 208, 16, 0],
            &[0, 96, 245, 67, 5, 24, 80, 0, 0],
            &[0, 142, 196, 0, 0, 0, 0, 0, 0],
            &[0, 101, 242, 51, 0, 0, 0, 0, 0],
            &[0, 4, 159, 252, 170, 48, 0, 0, 0],
            &[0, 0, 0, 50, 173, 252, 151, 1, 0],
            &[0, 0, 0, 0, 0, 64, 255, 68, 0],
            &[0, 0, 0, 0, 0, 1, 242, 91, 0],
            &[0, 84, 36, 3, 9, 112, 250, 37, 0],
            &[0, 154, 249, 251, 253, 226, 87, 0, 0],
            &[0, 0, 3, 25, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 4, 0, 9, 45, 0, 0],
        ],
        // letter: 'š' / 0x161
        'š' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 155, 46, 2, 127, 81, 0, 0],
            &[0, 0, 50, 229, 184, 167, 1, 0, 0],
            &[0, 0, 0, 73, 129, 11, 0, 0, 0],
            &[0, 0, 51, 138, 159, 136, 60, 0, 0],
            &[0, 46, 247, 143, 109, 137, 116, 0, 0],
            &[0, 93, 230, 3, 0, 0, 0, 0, 0],
            &[0, 19, 209, 204, 82, 2, 0, 0, 0],
            &[0, 0, 7, 104, 221, 219, 56, 0, 0],
            &[0, 0, 0, 0, 3, 150, 218, 0, 0],
            &[0, 53, 33, 0, 1, 139, 213, 0, 0],
            &[0, 101, 244, 244, 242, 224, 66, 0, 0],
            &[0, 0, 3, 25, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ţ' / 0x162
        'Ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[94, 254, 254, 254, 254, 254, 254, 196, 0],
            &[11, 31, 31, 128, 217, 31, 31, 24, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 113, 223, 0, 0, 0, 0],
            &[0, 0, 0, 107, 126, 0, 0, 0, 0],
            &[0, 0, 0, 56, 203, 91, 0, 0, 0],
            &[0, 0, 20, 86, 47, 0, 0, 0, 0],
        ],
        // letter: 'ţ' / 0x163
        'ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 123, 11, 0, 0, 0, 0],
            &[0, 0, 12, 255, 22, 0, 0, 0, 0],
            &[18, 102, 146, 255, 140, 128, 128, 1, 0],
            &[30, 139, 159, 255, 149, 139, 139, 1, 0],
            &[0, 0, 44, 255, 22, 0, 0, 0, 0],
            &[0, 0, 44, 255, 22, 0, 0, 0, 0],
            &[0, 0, 44, 255, 22, 0, 0, 0, 0],
            &[0, 0, 42, 255, 28, 0, 0, 0, 0],
            &[0, 0, 17, 251, 106, 0, 1, 4, 0],
            &[0, 0, 0, 116, 247, 250, 244, 53, 0],
            &[0, 0, 0, 0, 100, 163, 14, 0, 0],
            &[0, 0, 0, 0, 52, 198, 99, 0, 0],
            &[0, 0, 0, 17, 87, 49, 0, 0, 0],
        ],
        // letter: 'Ť' / 0x164
        'Ť' => &[
            &[0, 2, 169, 142, 67, 215, 37, 0, 0],
            &[0, 0, 7, 203, 249, 54, 0, 0, 0],
            &[0, 0, 0, 2, 5, 0, 0, 0, 0],
            &[94, 254, 254, 254, 254, 254, 254, 196, 0],
            &[11, 31, 31, 128, 217, 31, 31, 24, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 45, 0, 0, 24, 31, 0, 0],
        ],
        // letter: 'ť' / 0x165
        'ť' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 201, 15, 0],
            &[0, 0, 0, 123, 11, 76, 176, 0, 0],
            &[0, 0, 12, 255, 22, 55, 33, 0, 0],
            &[18, 102, 146, 255, 140, 128, 128, 1, 0],
            &[30, 139, 159, 255, 149, 139, 139, 1, 0],
            &[0, 0, 44, 255, 22, 0, 0, 0, 0],
            &[0, 0, 44, 255, 22, 0, 0, 0, 0],
            &[0, 0, 44, 255, 22, 0, 0, 0, 0],
            &[0, 0, 42, 255, 28, 0, 0, 0, 0],
            &[0, 0, 17, 251, 106, 0, 1, 4, 0],
            &[0, 0, 0, 116, 245, 238, 244, 53, 0],
            &[0, 0, 0, 0, 1, 26, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŧ' / 0x166
        'Ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[94, 254, 254, 254, 254, 254, 254, 196, 0],
            &[11, 31, 31, 128, 217, 31, 31, 24, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 75, 149, 255, 255, 149, 137, 0, 0],
            &[0, 57, 115, 225, 255, 115, 105, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 110, 212, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŧ' / 0x167
        'ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 123, 11, 0, 0, 0, 0],
            &[0, 0, 12, 255, 22, 0, 0, 0, 0],
            &[18, 102, 146, 255, 140, 128, 128, 1, 0],
            &[30, 139, 159, 255, 149, 139, 139, 1, 0],
            &[0, 0, 44, 255, 22, 0, 0, 0, 0],
            &[46, 215, 255, 255, 238, 215, 215, 48, 0],
            &[4, 20, 64, 255, 42, 20, 20, 4, 0],
            &[0, 0, 42, 255, 28, 0, 0, 0, 0],
            &[0, 0, 17, 251, 106, 0, 1, 4, 0],
            &[0, 0, 0, 116, 245, 238, 244, 53, 0],
            &[0, 0, 0, 0, 1, 26, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ũ' / 0x168
        'Ũ' => &[
            &[0, 0, 127, 177, 67, 39, 126, 0, 0],
            &[0, 49, 179, 83, 203, 235, 57, 0, 0],
            &[0, 1, 2, 0, 0, 0, 0, 0, 0],
            &[5, 254, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[2, 255, 67, 0, 0, 0, 220, 104, 0],
            &[0, 234, 100, 0, 0, 3, 247, 80, 0],
            &[0, 159, 208, 29, 6, 125, 241, 16, 0],
            &[0, 21, 188, 255, 255, 228, 74, 0, 0],
            &[0, 0, 0, 16, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ũ' / 0x169
        'ũ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 22, 51, 0, 3, 50, 0, 0],
            &[0, 16, 220, 209, 197, 165, 126, 0, 0],
            &[0, 34, 66, 0, 73, 106, 6, 0, 0],
            &[0, 122, 39, 0, 0, 0, 126, 36, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 3, 253, 71, 0],
            &[0, 238, 86, 0, 0, 33, 255, 71, 0],
            &[0, 200, 166, 2, 4, 147, 255, 71, 0],
            &[0, 62, 232, 242, 236, 130, 209, 71, 0],
            &[0, 0, 1, 26, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ū' / 0x16a
        'Ū' => &[
            &[0, 0, 31, 35, 35, 35, 9, 0, 0],
            &[0, 0, 194, 219, 219, 219, 59, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 254, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[2, 255, 67, 0, 0, 0, 220, 104, 0],
            &[0, 234, 100, 0, 0, 3, 247, 80, 0],
            &[0, 159, 208, 29, 6, 125, 241, 16, 0],
            &[0, 21, 188, 255, 255, 228, 74, 0, 0],
            &[0, 0, 0, 16, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ū' / 0x16b
        'ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 146, 165, 165, 165, 44, 0, 0],
            &[0, 0, 78, 89, 89, 89, 24, 0, 0],
            &[0, 122, 39, 0, 0, 0, 126, 36, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 3, 253, 71, 0],
            &[0, 238, 86, 0, 0, 33, 255, 71, 0],
            &[0, 200, 166, 2, 4, 147, 255, 71, 0],
            &[0, 62, 232, 242, 236, 130, 209, 71, 0],
            &[0, 0, 1, 26, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => &[
            &[0, 0, 186, 24, 1, 145, 60, 0, 0],
            &[0, 0, 84, 224, 229, 160, 3, 0, 0],
            &[0, 0, 0, 0, 1, 0, 0, 0, 0],
            &[5, 254, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[2, 255, 67, 0, 0, 0, 220, 104, 0],
            &[0, 234, 100, 0, 0, 3, 247, 80, 0],
            &[0, 159, 208, 29, 6, 125, 241, 16, 0],
            &[0, 21, 188, 255, 255, 228, 74, 0, 0],
            &[0, 0, 0, 16, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŭ' / 0x16d
        'ŭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 79, 0, 0, 48, 30, 0, 0],
            &[0, 0, 180, 144, 113, 217, 32, 0, 0],
            &[0, 0, 10, 103, 118, 40, 0, 0, 0],
            &[0, 122, 39, 0, 0, 0, 126, 36, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 3, 253, 71, 0],
            &[0, 238, 86, 0, 0, 33, 255, 71, 0],
            &[0, 200, 166, 2, 4, 147, 255, 71, 0],
            &[0, 62, 232, 242, 236, 130, 209, 71, 0],
            &[0, 0, 1, 26, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ů' / 0x16e
        'Ů' => &[
            &[0, 0, 76, 138, 39, 177, 0, 0, 0],
            &[0, 0, 20, 200, 204, 89, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 254, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[2, 255, 67, 0, 0, 0, 220, 104, 0],
            &[0, 234, 100, 0, 0, 3, 247, 80, 0],
            &[0, 159, 208, 29, 6, 125, 241, 16, 0],
            &[0, 21, 188, 255, 255, 228, 74, 0, 0],
            &[0, 0, 0, 16, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 184, 196, 74, 0, 0, 0],
        ],
        // letter: 'ů' / 0x16f
        'ů' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 66, 95, 10, 0, 0, 0],
            &[0, 0, 53, 190, 122, 153, 0, 0, 0],
            &[0, 0, 59, 181, 107, 159, 0, 0, 0],
            &[0, 0, 0, 85, 115, 17, 0, 0, 0],
            &[0, 122, 39, 0, 0, 0, 126, 36, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 3, 253, 71, 0],
            &[0, 238, 86, 0, 0, 33, 255, 71, 0],
            &[0, 200, 166, 2, 4, 147, 255, 71, 0],
            &[0, 62, 232, 242, 236, 130, 209, 71, 0],
            &[0, 0, 1, 26, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ű' / 0x170
        'Ű' => &[
            &[0, 0, 53, 240, 38, 188, 142, 0, 0],
            &[0, 1, 190, 65, 85, 169, 3, 0, 0],
            &[0, 0, 4, 0, 3, 0, 0, 0, 0],
            &[5, 254, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[2, 255, 67, 0, 0, 0, 220, 104, 0],
            &[0, 234, 100, 0, 0, 3, 247, 80, 0],
            &[0, 159, 208, 29, 6, 125, 241, 16, 0],
            &[0, 21, 188, 255, 255, 228, 74, 0, 0],
            &[0, 0, 0, 16, 25, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 48, 28, 22, 53, 0, 0],
        ],
        // letter: 'ű' / 0x171
        'ű' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 177, 65, 100, 151, 0, 0],
            &[0, 0, 127, 171, 25, 227, 48, 0, 0],
            &[0, 2, 111, 5, 65, 53, 0, 0, 0],
            &[0, 122, 39, 0, 0, 0, 126, 36, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 3, 253, 71, 0],
            &[0, 238, 86, 0, 0, 33, 255, 71, 0],
            &[0, 200, 166, 2, 4, 147, 255, 71, 0],
            &[0, 62, 232, 242, 236, 130, 209, 71, 0],
            &[0, 0, 1, 26, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ų' / 0x172
        'Ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 254, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[5, 255, 61, 0, 0, 0, 214, 107, 0],
            &[2, 255, 67, 0, 0, 0, 220, 104, 0],
            &[0, 234, 100, 0, 0, 3, 247, 80, 0],
            &[0, 159, 208, 29, 6, 125, 241, 16, 0],
            &[0, 21, 188, 255, 255, 255, 74, 0, 0],
            &[0, 0, 0, 66, 213, 26, 0, 0, 0],
            &[0, 0, 0, 167, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 3, 0, 0, 0],
        ],
        // letter: 'ų' / 0x173
        'ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 39, 0, 0, 0, 126, 36, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 0, 250, 71, 0],
            &[0, 243, 79, 0, 0, 3, 253, 71, 0],
            &[0, 238, 86, 0, 0, 33, 255, 71, 0],
            &[0, 200, 166, 2, 4, 147, 255, 71, 0],
            &[0, 62, 232, 242, 236, 130, 232, 73, 0],
            &[0, 0, 1, 26, 10, 53, 184, 6, 0],
            &[0, 0, 0, 0, 0, 169, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 3, 0],
        ],
        // letter: 'Ŵ' / 0x174
        'Ŵ' => &[
            &[0, 0, 21, 223, 230, 86, 0, 0, 0],
            &[0, 9, 190, 84, 29, 201, 61, 0, 0],
            &[0, 1, 4, 0, 0, 1, 3, 0, 0],
            &[151, 142, 0, 0, 0, 0, 39, 251, 3],
            &[123, 167, 0, 0, 0, 0, 61, 230, 0],
            &[94, 191, 0, 57, 113, 0, 83, 204, 0],
            &[66, 216, 0, 159, 247, 16, 105, 178, 0],
            &[37, 240, 0, 217, 175, 75, 127, 152, 0],
            &[9, 255, 30, 218, 101, 136, 147, 126, 0],
            &[0, 235, 106, 159, 40, 197, 164, 101, 0],
            &[0, 207, 178, 100, 0, 227, 185, 75, 0],
            &[0, 178, 239, 41, 0, 173, 239, 49, 0],
            &[0, 149, 236, 1, 0, 111, 255, 23, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 49, 0, 0, 0, 0],
        ],
        // letter: 'ŵ' / 0x175
        'ŵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 127, 179, 20, 0, 0, 0],
            &[0, 0, 96, 199, 128, 191, 7, 0, 0],
            &[0, 9, 117, 10, 0, 80, 60, 0, 0],
            &[94, 56, 0, 0, 0, 0, 1, 127, 19],
            &[157, 140, 0, 11, 20, 0, 29, 253, 8],
            &[116, 177, 0, 171, 252, 16, 70, 220, 0],
            &[75, 215, 4, 239, 196, 83, 111, 178, 0],
            &[34, 249, 63, 210, 114, 153, 149, 136, 0],
            &[2, 246, 148, 140, 49, 218, 179, 94, 0],
            &[0, 207, 217, 73, 2, 237, 217, 52, 0],
            &[0, 166, 250, 12, 0, 174, 254, 11, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŷ' / 0x176
        'Ŷ' => &[
            &[0, 0, 21, 223, 230, 86, 0, 0, 0],
            &[0, 9, 190, 84, 29, 201, 61, 0, 0],
            &[0, 1, 4, 0, 0, 1, 3, 0, 0],
            &[65, 250, 31, 0, 0, 0, 169, 167, 0],
            &[0, 191, 145, 0, 0, 36, 250, 40, 0],
            &[0, 61, 244, 18, 0, 152, 163, 0, 0],
            &[0, 0, 186, 126, 23, 243, 37, 0, 0],
            &[0, 0, 56, 234, 144, 158, 0, 0, 0],
            &[0, 0, 0, 182, 250, 33, 0, 0, 0],
            &[0, 0, 0, 103, 205, 0, 0, 0, 0],
            &[0, 0, 0, 102, 205, 0, 0, 0, 0],
            &[0, 0, 0, 102, 205, 0, 0, 0, 0],
            &[0, 0, 0, 102, 205, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 49, 0, 0, 0, 0],
        ],
        // letter: 'ŷ' / 0x177
        'ŷ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 179, 18, 0, 0, 0],
            &[0, 0, 99, 197, 130, 188, 6, 0, 0],
            &[0, 10, 116, 9, 0, 82, 58, 0, 0],
            &[34, 122, 1, 0, 0, 0, 79, 85, 0],
            &[7, 235, 69, 0, 0, 0, 221, 98, 0],
            &[0, 137, 173, 0, 0, 51, 245, 13, 0],
            &[0, 33, 248, 25, 0, 137, 163, 0, 0],
            &[0, 0, 180, 125, 0, 222, 68, 0, 0],
            &[0, 0, 73, 222, 51, 227, 2, 0, 0],
            &[0, 0, 2, 221, 183, 134, 0, 0, 0],
            &[0, 0, 0, 116, 255, 40, 0, 0, 0],
            &[0, 0, 0, 100, 200, 0, 0, 0, 0],
            &[0, 0, 3, 212, 94, 0, 0, 0, 0],
            &[8, 101, 89, 11, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ÿ' / 0x178
        'Ÿ' => &[
            &[0, 10, 187, 41, 0, 160, 79, 0, 0],
            &[0, 8, 178, 37, 0, 151, 73, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[65, 250, 31, 0, 0, 0, 169, 167, 0],
            &[0, 191, 145, 0, 0, 36, 250, 40, 0],
            &[0, 61, 244, 18, 0, 152, 163, 0, 0],
            &[0, 0, 186, 126, 23, 243, 37, 0, 0],
            &[0, 0, 56, 234, 144, 158, 0, 0, 0],
            &[0, 0, 0, 182, 250, 33, 0, 0, 0],
            &[0, 0, 0, 103, 205, 0, 0, 0, 0],
            &[0, 0, 0, 102, 205, 0, 0, 0, 0],
            &[0, 0, 0, 102, 205, 0, 0, 0, 0],
            &[0, 0, 0, 102, 205, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ź' / 0x179
        'Ź' => &[
            &[0, 0, 0, 0, 74, 242, 48, 0, 0],
            &[0, 0, 0, 15, 211, 62, 0, 0, 0],
            &[0, 0, 0, 1, 3, 0, 0, 0, 0],
            &[0, 99, 254, 254, 254, 254, 254, 17, 0],
            &[0, 9, 24, 24, 24, 136, 201, 1, 0],
            &[0, 0, 0, 0, 20, 239, 60, 0, 0],
            &[0, 0, 0, 0, 148, 173, 0, 0, 0],
            &[0, 0, 0, 40, 245, 37, 0, 0, 0],
            &[0, 0, 0, 179, 145, 0, 0, 0, 0],
            &[0, 0, 67, 239, 19, 0, 0, 0, 0],
            &[0, 2, 208, 116, 0, 0, 0, 0, 0],
            &[0, 98, 233, 37, 30, 30, 30, 2, 0],
            &[0, 171, 255, 255, 255, 255, 255, 17, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 33, 0, 0],
        ],
        // letter: 'ź' / 0x17a
        'ź' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 183, 72, 0, 0],
            &[0, 0, 0, 0, 164, 165, 2, 0, 0],
            &[0, 0, 0, 19, 111, 2, 0, 0, 0],
            &[0, 66, 128, 128, 128, 128, 128, 3, 0],
            &[0, 61, 118, 118, 118, 192, 233, 3, 0],
            &[0, 0, 0, 0, 45, 241, 64, 0, 0],
            &[0, 0, 0, 14, 218, 115, 0, 0, 0],
            &[0, 0, 0, 174, 170, 0, 0, 0, 0],
            &[0, 0, 118, 213, 12, 0, 0, 0, 0],
            &[0, 64, 242, 55, 16, 16, 16, 2, 0],
            &[0, 187, 255, 255, 255, 255, 255, 35, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ż' / 0x17b
        'Ż' => &[
            &[0, 0, 0, 101, 252, 30, 0, 0, 0],
            &[0, 0, 0, 41, 140, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 99, 254, 254, 254, 254, 254, 17, 0],
            &[0, 9, 24, 24, 24, 136, 201, 1, 0],
            &[0, 0, 0, 0, 20, 239, 60, 0, 0],
            &[0, 0, 0, 0, 148, 173, 0, 0, 0],
            &[0, 0, 0, 40, 245, 37, 0, 0, 0],
            &[0, 0, 0, 179, 145, 0, 0, 0, 0],
            &[0, 0, 67, 239, 19, 0, 0, 0, 0],
            &[0, 2, 208, 116, 0, 0, 0, 0, 0],
            &[0, 98, 233, 37, 30, 30, 30, 2, 0],
            &[0, 171, 255, 255, 255, 255, 255, 17, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 0, 0, 0, 0],
        ],
        // letter: 'ż' / 0x17c
        'ż' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 44, 138, 4, 0, 0, 0],
            &[0, 0, 0, 108, 251, 24, 0, 0, 0],
            &[0, 0, 0, 0, 13, 0, 0, 0, 0],
            &[0, 66, 128, 128, 128, 128, 128, 3, 0],
            &[0, 61, 118, 118, 118, 192, 233, 3, 0],
            &[0, 0, 0, 0, 45, 241, 64, 0, 0],
            &[0, 0, 0, 14, 218, 115, 0, 0, 0],
            &[0, 0, 0, 174, 170, 0, 0, 0, 0],
            &[0, 0, 118, 213, 12, 0, 0, 0, 0],
            &[0, 64, 242, 55, 16, 16, 16, 2, 0],
            &[0, 187, 255, 255, 255, 255, 255, 35, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ž' / 0x17d
        'Ž' => &[
            &[0, 0, 129, 177, 48, 214, 65, 0, 0],
            &[0, 0, 0, 167, 254, 92, 0, 0, 0],
            &[0, 0, 0, 1, 6, 0, 0, 0, 0],
            &[0, 99, 254, 254, 254, 254, 254, 17, 0],
            &[0, 9, 24, 24, 24, 136, 201, 1, 0],
            &[0, 0, 0, 0, 20, 239, 60, 0, 0],
            &[0, 0, 0, 0, 148, 173, 0, 0, 0],
            &[0, 0, 0, 40, 245, 37, 0, 0, 0],
            &[0, 0, 0, 179, 145, 0, 0, 0, 0],
            &[0, 0, 67, 239, 19, 0, 0, 0, 0],
            &[0, 2, 208, 116, 0, 0, 0, 0, 0],
            &[0, 98, 233, 37, 30, 30, 30, 2, 0],
            &[0, 171, 255, 255, 255, 255, 255, 17, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 1, 0, 15, 40, 0, 0],
        ],
        // letter: 'ž' / 0x17e
        'ž' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 150, 56, 0, 116, 94, 0, 0],
            &[0, 0, 38, 228, 178, 182, 4, 0, 0],
            &[0, 0, 0, 64, 131, 18, 0, 0, 0],
            &[0, 66, 128, 128, 128, 128, 128, 3, 0],
            &[0, 61, 118, 118, 118, 192, 233, 3, 0],
            &[0, 0, 0, 0, 45, 241, 64, 0, 0],
            &[0, 0, 0, 14, 218, 115, 0, 0, 0],
            &[0, 0, 0, 174, 170, 0, 0, 0, 0],
            &[0, 0, 118, 213, 12, 0, 0, 0, 0],
            &[0, 64, 242, 55, 16, 16, 16, 2, 0],
            &[0, 187, 255, 255, 255, 255, 255, 35, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ſ' / 0x17f
        'ſ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 110, 171, 170, 82, 0, 0],
            &[0, 0, 143, 225, 99, 93, 54, 0, 0],
            &[0, 0, 220, 106, 0, 0, 0, 0, 0],
            &[0, 0, 231, 83, 0, 0, 0, 0, 0],
            &[0, 0, 231, 83, 0, 0, 0, 0, 0],
            &[0, 0, 231, 83, 0, 0, 0, 0, 0],
            &[0, 0, 231, 83, 0, 0, 0, 0, 0],
            &[0, 0, 231, 83, 0, 0, 0, 0, 0],
            &[0, 0, 231, 83, 0, 0, 0, 0, 0],
            &[0, 0, 231, 83, 0, 0, 0, 0, 0],
            &[0, 0, 231, 83, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        _ => panic!("unsupported char"),
    }
}
