//! Module for letters with the font weight regular and size 14.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 14;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 8;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight regular and font size 12px
#[inline]
pub const fn get_char(c: char) -> &'static [&'static [u8]] {
    match c {
        // letter: ' ' / 0x20
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '!' / 0x21
        '!' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 145, 36, 0, 0, 0],
            &[0, 0, 3, 255, 57, 0, 0, 0],
            &[0, 0, 0, 251, 49, 0, 0, 0],
            &[0, 0, 0, 243, 41, 0, 0, 0],
            &[0, 0, 0, 235, 34, 0, 0, 0],
            &[0, 0, 0, 227, 26, 0, 0, 0],
            &[0, 0, 0, 127, 11, 0, 0, 0],
            &[0, 0, 0, 106, 16, 0, 0, 0],
            &[0, 0, 14, 250, 61, 0, 0, 0],
            &[0, 0, 0, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '"' / 0x22
        '"' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 145, 5, 121, 37, 0, 0],
            &[0, 2, 246, 0, 198, 50, 0, 0],
            &[0, 0, 209, 0, 179, 30, 0, 0],
            &[0, 0, 94, 0, 86, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '#' / 0x23
        '#' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 32, 93, 0, 120, 0, 0],
            &[0, 0, 94, 126, 8, 203, 0, 0],
            &[1, 6, 144, 83, 57, 162, 5, 0],
            &[62, 196, 244, 198, 226, 217, 188, 0],
            &[0, 0, 215, 0, 158, 59, 0, 0],
            &[60, 98, 210, 79, 217, 92, 44, 0],
            &[95, 175, 187, 132, 229, 125, 69, 0],
            &[0, 131, 84, 51, 166, 0, 0, 0],
            &[0, 178, 37, 101, 116, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '$' / 0x24
        '$' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 0, 0, 0, 0],
            &[0, 0, 26, 209, 43, 4, 0, 0],
            &[0, 126, 227, 238, 197, 184, 0, 0],
            &[3, 249, 25, 196, 0, 3, 0, 0],
            &[0, 230, 89, 197, 0, 0, 0, 0],
            &[0, 47, 199, 250, 133, 25, 0, 0],
            &[0, 0, 0, 204, 138, 226, 15, 0],
            &[0, 0, 0, 196, 0, 216, 51, 0],
            &[0, 156, 108, 218, 149, 213, 7, 0],
            &[0, 78, 134, 230, 102, 12, 0, 0],
            &[0, 0, 0, 136, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '%' / 0x25
        '%' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[46, 155, 106, 0, 0, 124, 9, 0],
            &[207, 51, 176, 65, 67, 168, 0, 0],
            &[223, 0, 131, 101, 193, 43, 0, 0],
            &[179, 117, 208, 109, 169, 0, 0, 0],
            &[10, 88, 45, 192, 43, 0, 0, 0],
            &[0, 0, 66, 170, 149, 200, 137, 0],
            &[0, 0, 192, 81, 189, 0, 199, 28],
            &[0, 65, 170, 43, 181, 0, 201, 35],
            &[0, 191, 44, 1, 176, 184, 174, 0],
            &[0, 0, 0, 0, 0, 15, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '&' / 0x26
        '&' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 30, 155, 153, 28, 0, 0, 0],
            &[0, 176, 124, 125, 170, 0, 0, 0],
            &[0, 187, 63, 83, 181, 0, 0, 0],
            &[0, 102, 200, 218, 57, 0, 0, 0],
            &[0, 71, 255, 160, 0, 21, 25, 0],
            &[36, 228, 72, 225, 50, 158, 97, 0],
            &[123, 150, 0, 66, 244, 253, 19, 0],
            &[111, 181, 1, 14, 255, 255, 4, 0],
            &[16, 202, 225, 231, 121, 178, 146, 0],
            &[0, 0, 21, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ''' / 0x27
        '\'' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 138, 20, 0, 0, 0],
            &[0, 0, 0, 228, 20, 0, 0, 0],
            &[0, 0, 0, 206, 2, 0, 0, 0],
            &[0, 0, 0, 94, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '(' / 0x28
        '(' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 132, 5, 0, 0],
            &[0, 0, 0, 138, 130, 0, 0, 0],
            &[0, 0, 14, 237, 22, 0, 0, 0],
            &[0, 0, 80, 194, 0, 0, 0, 0],
            &[0, 0, 129, 144, 0, 0, 0, 0],
            &[0, 0, 147, 125, 0, 0, 0, 0],
            &[0, 0, 138, 135, 0, 0, 0, 0],
            &[0, 0, 104, 172, 0, 0, 0, 0],
            &[0, 0, 38, 234, 5, 0, 0, 0],
            &[0, 0, 0, 188, 87, 0, 0, 0],
            &[0, 0, 0, 39, 190, 4, 0, 0],
        ],
        // letter: ')' / 0x29
        ')' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 112, 34, 0, 0, 0, 0],
            &[0, 0, 78, 194, 0, 0, 0, 0],
            &[0, 0, 1, 218, 58, 0, 0, 0],
            &[0, 0, 0, 141, 136, 0, 0, 0],
            &[0, 0, 0, 92, 182, 0, 0, 0],
            &[0, 0, 0, 73, 199, 0, 0, 0],
            &[0, 0, 0, 85, 187, 0, 0, 0],
            &[0, 0, 0, 123, 151, 0, 0, 0],
            &[0, 0, 0, 190, 84, 0, 0, 0],
            &[0, 0, 38, 227, 5, 0, 0, 0],
            &[0, 0, 156, 78, 0, 0, 0, 0],
        ],
        // letter: '*' / 0x2a
        '*' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 205, 8, 0, 0, 0],
            &[0, 50, 11, 187, 7, 66, 1, 0],
            &[0, 137, 187, 243, 205, 172, 15, 0],
            &[0, 0, 133, 156, 189, 3, 0, 0],
            &[0, 14, 184, 5, 160, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '+' / 0x2b
        '+' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 0, 0, 0, 0],
            &[0, 0, 0, 209, 11, 0, 0, 0],
            &[0, 0, 0, 209, 11, 0, 0, 0],
            &[6, 167, 167, 240, 171, 167, 40, 0],
            &[2, 50, 50, 218, 59, 50, 12, 0],
            &[0, 0, 0, 209, 11, 0, 0, 0],
            &[0, 0, 0, 114, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ',' / 0x2c
        ',' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 121, 70, 0, 0, 0],
            &[0, 0, 6, 249, 71, 0, 0, 0],
            &[0, 0, 51, 230, 3, 0, 0, 0],
            &[0, 0, 67, 98, 0, 0, 0, 0],
        ],
        // letter: '-' / 0x2d
        '-' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 36, 184, 184, 184, 73, 0, 0],
            &[0, 12, 61, 61, 61, 24, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '.' / 0x2e
        '.' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 106, 16, 0, 0, 0],
            &[0, 0, 14, 250, 61, 0, 0, 0],
            &[0, 0, 0, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '/' / 0x2f
        '/' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 55, 94, 0, 0],
            &[0, 0, 0, 0, 174, 89, 0, 0],
            &[0, 0, 0, 22, 233, 8, 0, 0],
            &[0, 0, 0, 114, 149, 0, 0, 0],
            &[0, 0, 0, 212, 50, 0, 0, 0],
            &[0, 0, 55, 208, 0, 0, 0, 0],
            &[0, 0, 153, 110, 0, 0, 0, 0],
            &[0, 10, 234, 19, 0, 0, 0, 0],
            &[0, 94, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '0' / 0x30
        '0' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 115, 166, 131, 17, 0, 0],
            &[0, 149, 182, 77, 184, 207, 0, 0],
            &[10, 241, 16, 0, 186, 255, 51, 0],
            &[51, 206, 0, 80, 196, 149, 101, 0],
            &[67, 190, 8, 220, 48, 129, 118, 0],
            &[59, 200, 127, 149, 0, 141, 113, 0],
            &[25, 255, 228, 17, 0, 178, 80, 0],
            &[0, 249, 183, 0, 48, 235, 15, 0],
            &[0, 45, 221, 228, 237, 85, 0, 0],
            &[0, 0, 1, 22, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '1' / 0x31
        '1' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 125, 62, 0, 0, 0],
            &[0, 112, 229, 217, 109, 0, 0, 0],
            &[0, 49, 17, 150, 109, 0, 0, 0],
            &[0, 0, 0, 154, 109, 0, 0, 0],
            &[0, 0, 0, 154, 109, 0, 0, 0],
            &[0, 0, 0, 154, 109, 0, 0, 0],
            &[0, 0, 0, 154, 109, 0, 0, 0],
            &[0, 0, 0, 154, 109, 0, 0, 0],
            &[0, 75, 199, 255, 255, 196, 44, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '2' / 0x32
        '2' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 31, 131, 168, 136, 21, 0, 0],
            &[1, 196, 133, 78, 159, 201, 0, 0],
            &[0, 1, 0, 0, 11, 255, 13, 0],
            &[0, 0, 0, 0, 45, 241, 3, 0],
            &[0, 0, 0, 1, 180, 127, 0, 0],
            &[0, 0, 0, 143, 189, 6, 0, 0],
            &[0, 0, 139, 196, 13, 0, 0, 0],
            &[0, 143, 191, 12, 0, 0, 0, 0],
            &[39, 255, 248, 245, 245, 245, 132, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '3' / 0x33
        '3' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 38, 140, 170, 137, 23, 0, 0],
            &[1, 174, 117, 76, 161, 203, 0, 0],
            &[0, 0, 0, 0, 28, 251, 2, 0],
            &[0, 0, 0, 5, 131, 177, 0, 0],
            &[0, 0, 197, 249, 198, 29, 0, 0],
            &[0, 0, 11, 28, 103, 222, 12, 0],
            &[0, 0, 0, 0, 0, 219, 63, 0],
            &[9, 39, 0, 0, 50, 246, 32, 0],
            &[19, 217, 233, 222, 231, 98, 0, 0],
            &[0, 0, 15, 22, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '4' / 0x34
        '4' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 154, 55, 0, 0],
            &[0, 0, 0, 140, 235, 90, 0, 0],
            &[0, 0, 44, 205, 169, 90, 0, 0],
            &[0, 0, 193, 62, 172, 90, 0, 0],
            &[0, 95, 163, 0, 174, 90, 0, 0],
            &[16, 219, 24, 0, 174, 90, 0, 0],
            &[99, 243, 215, 215, 243, 230, 129, 0],
            &[8, 20, 20, 20, 180, 103, 12, 0],
            &[0, 0, 0, 0, 174, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '5' / 0x35
        '5' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 145, 145, 145, 106, 0, 0],
            &[0, 159, 156, 100, 100, 73, 0, 0],
            &[0, 179, 73, 0, 0, 0, 0, 0],
            &[0, 200, 91, 60, 21, 0, 0, 0],
            &[0, 148, 171, 179, 239, 117, 0, 0],
            &[0, 0, 0, 0, 46, 249, 23, 0],
            &[0, 0, 0, 0, 0, 234, 50, 0],
            &[3, 36, 0, 0, 77, 239, 11, 0],
            &[6, 214, 227, 228, 221, 67, 0, 0],
            &[0, 0, 14, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '6' / 0x36
        '6' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 37, 135, 169, 126, 0, 0],
            &[0, 60, 229, 115, 63, 57, 0, 0],
            &[0, 206, 82, 0, 0, 0, 0, 0],
            &[26, 239, 23, 93, 81, 9, 0, 0],
            &[60, 231, 188, 123, 155, 213, 10, 0],
            &[67, 239, 12, 0, 0, 179, 92, 0],
            &[42, 220, 0, 0, 0, 152, 114, 0],
            &[1, 217, 81, 0, 15, 220, 58, 0],
            &[0, 52, 219, 212, 231, 126, 0, 0],
            &[0, 0, 0, 23, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '7' / 0x37
        '7' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[40, 145, 145, 145, 145, 145, 69, 0],
            &[28, 100, 100, 100, 100, 216, 95, 0],
            &[0, 0, 0, 0, 28, 233, 9, 0],
            &[0, 0, 0, 0, 134, 139, 0, 0],
            &[0, 0, 0, 9, 234, 34, 0, 0],
            &[0, 0, 0, 100, 179, 0, 0, 0],
            &[0, 0, 0, 211, 72, 0, 0, 0],
            &[0, 0, 66, 218, 1, 0, 0, 0],
            &[0, 0, 177, 112, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '8' / 0x38
        '8' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 123, 170, 140, 28, 0, 0],
            &[0, 172, 160, 62, 126, 220, 1, 0],
            &[0, 218, 51, 0, 5, 252, 13, 0],
            &[0, 129, 180, 27, 141, 175, 0, 0],
            &[0, 5, 200, 252, 213, 14, 0, 0],
            &[0, 173, 145, 18, 136, 201, 9, 0],
            &[34, 242, 2, 0, 0, 195, 81, 0],
            &[19, 244, 30, 0, 14, 224, 59, 0],
            &[0, 102, 228, 202, 227, 129, 0, 0],
            &[0, 0, 4, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '9' / 0x39
        '9' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 129, 169, 132, 18, 0, 0],
            &[1, 205, 142, 67, 147, 203, 2, 0],
            &[50, 220, 0, 0, 0, 200, 63, 0],
            &[57, 205, 0, 0, 0, 171, 109, 0],
            &[11, 229, 71, 8, 80, 238, 117, 0],
            &[0, 50, 180, 207, 141, 179, 97, 0],
            &[0, 0, 0, 0, 6, 232, 39, 0],
            &[0, 0, 0, 3, 139, 187, 0, 0],
            &[0, 102, 207, 232, 173, 19, 0, 0],
            &[0, 5, 26, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ':' / 0x3a
        ':' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 115, 19, 0, 0, 0],
            &[0, 0, 12, 245, 59, 0, 0, 0],
            &[0, 0, 0, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 106, 15, 0, 0, 0],
            &[0, 0, 14, 250, 58, 0, 0, 0],
            &[0, 0, 0, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ';' / 0x3b
        ';' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 51, 0, 0, 0],
            &[0, 0, 0, 191, 127, 0, 0, 0],
            &[0, 0, 0, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 121, 70, 0, 0, 0],
            &[0, 0, 6, 249, 71, 0, 0, 0],
            &[0, 0, 51, 230, 3, 0, 0, 0],
            &[0, 0, 67, 98, 0, 0, 0, 0],
        ],
        // letter: '<' / 0x3c
        '<' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 8, 0],
            &[0, 0, 0, 2, 99, 210, 20, 0],
            &[0, 0, 69, 200, 135, 15, 0, 0],
            &[0, 172, 180, 29, 0, 0, 0, 0],
            &[0, 82, 201, 150, 32, 0, 0, 0],
            &[0, 0, 1, 81, 201, 160, 10, 0],
            &[0, 0, 0, 0, 0, 77, 17, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '=' / 0x3d
        '=' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[34, 215, 215, 215, 215, 215, 77, 0],
            &[4, 29, 29, 29, 29, 29, 10, 0],
            &[23, 144, 144, 144, 144, 144, 51, 0],
            &[16, 101, 101, 101, 101, 101, 36, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '>' / 0x3e
        '>' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 0, 0, 0, 0, 0, 0],
            &[0, 194, 128, 11, 0, 0, 0, 0],
            &[0, 4, 107, 208, 97, 2, 0, 0],
            &[0, 0, 0, 13, 144, 208, 15, 0],
            &[0, 0, 16, 126, 209, 109, 4, 0],
            &[0, 129, 212, 106, 7, 0, 0, 0],
            &[0, 89, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '?' / 0x3f
        '?' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 142, 169, 138, 27, 0, 0],
            &[0, 125, 106, 72, 133, 219, 3, 0],
            &[0, 0, 0, 0, 0, 241, 21, 0],
            &[0, 0, 0, 0, 60, 222, 0, 0],
            &[0, 0, 0, 77, 221, 50, 0, 0],
            &[0, 0, 21, 228, 32, 0, 0, 0],
            &[0, 0, 33, 106, 0, 0, 0, 0],
            &[0, 0, 28, 94, 0, 0, 0, 0],
            &[0, 0, 92, 233, 0, 0, 0, 0],
            &[0, 0, 0, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '@' / 0x40
        '@' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 71, 135, 117, 23, 0, 0],
            &[0, 145, 164, 71, 102, 213, 31, 0],
            &[59, 177, 0, 44, 41, 75, 158, 0],
            &[148, 68, 150, 175, 213, 83, 211, 0],
            &[193, 41, 223, 1, 164, 69, 206, 1],
            &[205, 54, 191, 0, 190, 56, 200, 0],
            &[187, 43, 222, 47, 220, 107, 184, 0],
            &[129, 96, 89, 164, 41, 148, 50, 0],
            &[25, 212, 37, 0, 0, 13, 0, 0],
            &[0, 53, 197, 200, 201, 179, 0, 0],
            &[0, 0, 0, 6, 3, 0, 0, 0],
        ],
        // letter: 'A' / 0x41
        'A' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 154, 40, 0, 0, 0],
            &[0, 0, 85, 242, 133, 0, 0, 0],
            &[0, 0, 168, 127, 216, 0, 0, 0],
            &[0, 7, 237, 15, 218, 44, 0, 0],
            &[0, 77, 186, 0, 138, 127, 0, 0],
            &[0, 160, 242, 150, 207, 216, 0, 0],
            &[5, 245, 123, 95, 97, 250, 37, 0],
            &[70, 204, 0, 0, 0, 156, 120, 0],
            &[153, 124, 0, 0, 0, 76, 204, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'B' / 0x42
        'B' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 125, 145, 141, 113, 28, 0, 0],
            &[0, 220, 120, 98, 147, 234, 19, 0],
            &[0, 220, 40, 0, 0, 215, 65, 0],
            &[0, 220, 40, 0, 40, 233, 25, 0],
            &[0, 220, 242, 243, 247, 96, 0, 0],
            &[0, 220, 40, 3, 48, 223, 69, 0],
            &[0, 220, 40, 0, 0, 156, 131, 0],
            &[0, 220, 40, 0, 27, 219, 87, 0],
            &[0, 220, 242, 241, 234, 135, 1, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'C' / 0x43
        'C' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 143, 169, 139, 43, 0],
            &[0, 62, 241, 128, 79, 115, 60, 0],
            &[0, 207, 104, 0, 0, 0, 0, 0],
            &[23, 255, 12, 0, 0, 0, 0, 0],
            &[48, 237, 0, 0, 0, 0, 0, 0],
            &[37, 243, 0, 0, 0, 0, 0, 0],
            &[4, 242, 44, 0, 0, 0, 0, 0],
            &[0, 144, 189, 18, 0, 13, 12, 0],
            &[0, 7, 155, 245, 231, 240, 46, 0],
            &[0, 0, 0, 5, 23, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'D' / 0x44
        'D' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 145, 145, 128, 61, 0, 0, 0],
            &[40, 242, 94, 119, 220, 144, 0, 0],
            &[40, 235, 0, 0, 18, 232, 61, 0],
            &[40, 235, 0, 0, 0, 148, 138, 0],
            &[40, 235, 0, 0, 0, 118, 162, 0],
            &[40, 235, 0, 0, 0, 131, 148, 0],
            &[40, 235, 0, 0, 0, 199, 93, 0],
            &[40, 235, 0, 15, 138, 214, 8, 0],
            &[40, 254, 239, 231, 159, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'E' / 0x45
        'E' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 125, 145, 145, 145, 145, 26, 0],
            &[0, 220, 134, 100, 100, 100, 18, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 244, 244, 232, 0, 0],
            &[0, 220, 57, 1, 1, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 245, 245, 245, 44, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'F' / 0x46
        'F' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 125, 145, 145, 145, 145, 26, 0],
            &[0, 220, 134, 100, 100, 100, 18, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 172, 149, 149, 142, 0, 0],
            &[0, 220, 131, 96, 96, 91, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'G' / 0x47
        'G' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 153, 166, 120, 14, 0],
            &[0, 116, 226, 106, 83, 136, 10, 0],
            &[19, 244, 41, 0, 0, 0, 0, 0],
            &[83, 201, 0, 0, 0, 0, 0, 0],
            &[110, 170, 0, 32, 134, 134, 48, 0],
            &[104, 177, 0, 27, 111, 210, 92, 0],
            &[65, 227, 2, 0, 0, 175, 92, 0],
            &[5, 219, 127, 2, 0, 175, 92, 0],
            &[0, 40, 200, 237, 224, 234, 67, 0],
            &[0, 0, 0, 13, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'H' / 0x48
        'H' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 133, 0, 0, 0, 104, 52, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 254, 244, 244, 244, 252, 92, 0],
            &[40, 235, 1, 1, 1, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'I' / 0x49
        'I' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 145, 145, 145, 127, 0, 0],
            &[0, 35, 72, 247, 98, 48, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 134, 217, 254, 224, 176, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'J' / 0x4a
        'J' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 72, 84, 0, 0],
            &[0, 0, 0, 0, 127, 148, 0, 0],
            &[0, 0, 0, 0, 127, 148, 0, 0],
            &[0, 0, 0, 0, 127, 148, 0, 0],
            &[0, 0, 0, 0, 127, 148, 0, 0],
            &[0, 0, 0, 0, 127, 148, 0, 0],
            &[0, 0, 0, 0, 140, 144, 0, 0],
            &[0, 24, 0, 13, 212, 103, 0, 0],
            &[0, 182, 238, 248, 181, 8, 0, 0],
            &[0, 0, 17, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'K' / 0x4b
        'K' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 133, 0, 0, 4, 134, 40, 0],
            &[40, 235, 0, 0, 138, 178, 0, 0],
            &[40, 235, 0, 75, 225, 19, 0, 0],
            &[40, 235, 28, 232, 60, 0, 0, 0],
            &[40, 236, 186, 206, 0, 0, 0, 0],
            &[40, 255, 129, 206, 97, 0, 0, 0],
            &[40, 237, 0, 53, 234, 22, 0, 0],
            &[40, 235, 0, 0, 146, 165, 0, 0],
            &[40, 235, 0, 0, 13, 228, 72, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'L' / 0x4c
        'L' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 66, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 255, 254, 254, 254, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'M' / 0x4d
        'M' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[58, 145, 33, 0, 6, 144, 87, 0],
            &[102, 244, 115, 0, 67, 244, 153, 0],
            &[102, 192, 188, 0, 140, 190, 153, 0],
            &[102, 155, 223, 11, 211, 127, 153, 0],
            &[102, 158, 155, 107, 202, 107, 153, 0],
            &[102, 158, 79, 236, 126, 107, 153, 0],
            &[102, 158, 11, 217, 48, 107, 153, 0],
            &[102, 158, 0, 0, 0, 107, 153, 0],
            &[102, 158, 0, 0, 0, 107, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'N' / 0x4e
        'N' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 145, 56, 0, 0, 99, 52, 0],
            &[40, 240, 188, 0, 0, 175, 92, 0],
            &[40, 206, 204, 44, 0, 175, 92, 0],
            &[40, 221, 92, 155, 0, 175, 92, 0],
            &[40, 226, 5, 221, 20, 175, 92, 0],
            &[40, 226, 0, 123, 123, 173, 92, 0],
            &[40, 226, 0, 20, 220, 171, 92, 0],
            &[40, 226, 0, 0, 155, 235, 92, 0],
            &[40, 226, 0, 0, 43, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'O' / 0x4f
        'O' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 121, 173, 143, 26, 0, 0],
            &[0, 172, 176, 75, 144, 219, 8, 0],
            &[37, 239, 9, 0, 0, 200, 91, 0],
            &[93, 187, 0, 0, 0, 138, 146, 0],
            &[113, 168, 0, 0, 0, 118, 165, 0],
            &[106, 175, 0, 0, 0, 124, 157, 0],
            &[69, 216, 0, 0, 0, 163, 118, 0],
            &[7, 231, 81, 0, 39, 238, 35, 0],
            &[0, 61, 226, 225, 235, 95, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'P' / 0x50
        'P' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 125, 145, 142, 116, 30, 0, 0],
            &[0, 220, 129, 97, 140, 241, 36, 0],
            &[0, 220, 56, 0, 0, 168, 119, 0],
            &[0, 220, 56, 0, 0, 164, 118, 0],
            &[0, 220, 103, 69, 128, 238, 32, 0],
            &[0, 220, 196, 175, 143, 41, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Q' / 0x51
        'Q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 121, 173, 143, 26, 0, 0],
            &[0, 172, 176, 75, 144, 219, 8, 0],
            &[37, 239, 9, 0, 0, 200, 91, 0],
            &[93, 187, 0, 0, 0, 138, 146, 0],
            &[113, 168, 0, 0, 0, 118, 165, 0],
            &[106, 175, 0, 0, 0, 124, 157, 0],
            &[69, 216, 0, 0, 0, 163, 118, 0],
            &[7, 231, 81, 0, 41, 238, 35, 0],
            &[0, 61, 226, 225, 255, 125, 0, 0],
            &[0, 0, 2, 23, 144, 157, 0, 0],
            &[0, 0, 0, 0, 0, 30, 18, 0],
        ],
        // letter: 'R' / 0x52
        'R' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 125, 145, 138, 96, 8, 0, 0],
            &[0, 220, 129, 104, 188, 194, 0, 0],
            &[0, 220, 56, 0, 12, 254, 29, 0],
            &[0, 220, 56, 0, 25, 253, 18, 0],
            &[0, 220, 144, 129, 210, 126, 0, 0],
            &[0, 220, 155, 164, 195, 0, 0, 0],
            &[0, 220, 56, 4, 211, 83, 0, 0],
            &[0, 220, 56, 0, 65, 225, 11, 0],
            &[0, 220, 56, 0, 0, 170, 135, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'S' / 0x53
        'S' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 100, 165, 160, 111, 8, 0],
            &[0, 127, 212, 87, 86, 138, 3, 0],
            &[0, 193, 98, 0, 0, 0, 0, 0],
            &[0, 149, 186, 18, 0, 0, 0, 0],
            &[0, 7, 154, 240, 140, 20, 0, 0],
            &[0, 0, 0, 35, 162, 222, 18, 0],
            &[0, 0, 0, 0, 0, 212, 79, 0],
            &[0, 39, 0, 0, 30, 241, 47, 0],
            &[0, 207, 234, 215, 236, 119, 0, 0],
            &[0, 0, 9, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'T' / 0x54
        'T' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[66, 145, 145, 145, 145, 145, 95, 0],
            &[46, 100, 100, 246, 122, 100, 66, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'U' / 0x55
        'U' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 133, 0, 0, 0, 104, 52, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[29, 251, 3, 0, 0, 202, 80, 0],
            &[1, 229, 81, 0, 35, 243, 27, 0],
            &[0, 73, 232, 223, 237, 107, 0, 0],
            &[0, 0, 3, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'V' / 0x56
        'V' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[88, 74, 0, 0, 0, 46, 117, 0],
            &[91, 191, 0, 0, 0, 140, 143, 0],
            &[15, 245, 16, 0, 0, 215, 62, 0],
            &[0, 182, 88, 0, 33, 234, 3, 0],
            &[0, 99, 165, 0, 108, 156, 0, 0],
            &[0, 20, 234, 3, 182, 75, 0, 0],
            &[0, 0, 190, 69, 233, 7, 0, 0],
            &[0, 0, 108, 191, 169, 0, 0, 0],
            &[0, 0, 27, 254, 88, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'W' / 0x57
        'W' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[97, 46, 0, 0, 0, 17, 126, 0],
            &[148, 101, 0, 0, 0, 47, 202, 0],
            &[119, 126, 0, 105, 19, 69, 176, 0],
            &[91, 150, 30, 246, 89, 92, 151, 0],
            &[62, 174, 90, 163, 151, 113, 125, 0],
            &[33, 196, 149, 61, 200, 133, 99, 0],
            &[6, 212, 197, 6, 185, 166, 73, 0],
            &[0, 212, 189, 0, 125, 230, 47, 0],
            &[0, 203, 135, 0, 63, 255, 22, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'X' / 0x58
        'X' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[33, 132, 1, 0, 0, 101, 62, 0],
            &[0, 192, 92, 0, 41, 227, 14, 0],
            &[0, 43, 222, 9, 182, 92, 0, 0],
            &[0, 0, 140, 183, 194, 0, 0, 0],
            &[0, 0, 18, 251, 69, 0, 0, 0],
            &[0, 0, 118, 210, 171, 0, 0, 0],
            &[0, 29, 227, 20, 198, 69, 0, 0],
            &[0, 172, 110, 0, 53, 215, 6, 0],
            &[71, 214, 4, 0, 0, 159, 122, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Y' / 0x59
        'Y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[63, 107, 0, 0, 0, 73, 92, 0],
            &[21, 239, 30, 0, 2, 219, 60, 0],
            &[0, 134, 144, 0, 85, 186, 0, 0],
            &[0, 18, 230, 18, 204, 56, 0, 0],
            &[0, 0, 130, 186, 181, 0, 0, 0],
            &[0, 0, 16, 246, 58, 0, 0, 0],
            &[0, 0, 0, 234, 29, 0, 0, 0],
            &[0, 0, 0, 234, 29, 0, 0, 0],
            &[0, 0, 0, 234, 29, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Z' / 0x5a
        'Z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 145, 145, 145, 145, 8, 0],
            &[0, 58, 94, 94, 131, 229, 5, 0],
            &[0, 0, 0, 0, 176, 98, 0, 0],
            &[0, 0, 0, 64, 209, 2, 0, 0],
            &[0, 0, 2, 206, 69, 0, 0, 0],
            &[0, 0, 95, 183, 0, 0, 0, 0],
            &[0, 10, 225, 44, 0, 0, 0, 0],
            &[0, 127, 154, 0, 0, 0, 0, 0],
            &[0, 218, 248, 245, 245, 245, 14, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '[' / 0x5b
        '[' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 145, 145, 84, 0, 0],
            &[0, 0, 30, 240, 75, 43, 0, 0],
            &[0, 0, 30, 233, 0, 0, 0, 0],
            &[0, 0, 30, 233, 0, 0, 0, 0],
            &[0, 0, 30, 233, 0, 0, 0, 0],
            &[0, 0, 30, 233, 0, 0, 0, 0],
            &[0, 0, 30, 233, 0, 0, 0, 0],
            &[0, 0, 30, 233, 0, 0, 0, 0],
            &[0, 0, 30, 233, 0, 0, 0, 0],
            &[0, 0, 30, 233, 0, 0, 0, 0],
            &[0, 0, 27, 228, 221, 128, 0, 0],
        ],
        // letter: '\' / 0x5c
        '\\' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 84, 0, 0, 0, 0, 0],
            &[0, 39, 223, 1, 0, 0, 0, 0],
            &[0, 0, 196, 67, 0, 0, 0, 0],
            &[0, 0, 97, 166, 0, 0, 0, 0],
            &[0, 0, 12, 235, 16, 0, 0, 0],
            &[0, 0, 0, 157, 106, 0, 0, 0],
            &[0, 0, 0, 59, 204, 0, 0, 0],
            &[0, 0, 0, 0, 216, 47, 0, 0],
            &[0, 0, 0, 0, 118, 145, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ']' / 0x5d
        ']' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 66, 176, 176, 56, 0, 0],
            &[0, 0, 17, 45, 195, 81, 0, 0],
            &[0, 0, 0, 0, 182, 81, 0, 0],
            &[0, 0, 0, 0, 182, 81, 0, 0],
            &[0, 0, 0, 0, 182, 81, 0, 0],
            &[0, 0, 0, 0, 182, 81, 0, 0],
            &[0, 0, 0, 0, 182, 81, 0, 0],
            &[0, 0, 0, 0, 182, 81, 0, 0],
            &[0, 0, 0, 0, 182, 81, 0, 0],
            &[0, 0, 8, 22, 188, 81, 0, 0],
            &[0, 0, 75, 198, 198, 63, 0, 0],
        ],
        // letter: '^' / 0x5e
        '^' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 139, 2, 0, 0, 0],
            &[0, 0, 93, 223, 91, 0, 0, 0],
            &[0, 1, 201, 34, 204, 4, 0, 0],
            &[0, 80, 146, 0, 127, 102, 0, 0],
            &[0, 197, 34, 0, 15, 210, 8, 0],
            &[43, 145, 0, 0, 0, 109, 79, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '_' / 0x5f
        '_' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[28, 28, 28, 28, 28, 28, 28, 5],
            &[217, 217, 217, 217, 217, 217, 217, 43],
        ],
        // letter: '`' / 0x60
        '`' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 35, 28, 0, 0, 0, 0],
            &[0, 0, 73, 221, 12, 0, 0, 0],
            &[0, 0, 0, 80, 94, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'a' / 0x61
        'a' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 101, 132, 106, 15, 0, 0],
            &[0, 104, 129, 91, 151, 203, 0, 0],
            &[0, 0, 0, 0, 0, 234, 20, 0],
            &[0, 59, 178, 207, 197, 247, 30, 0],
            &[12, 240, 67, 1, 0, 226, 30, 0],
            &[26, 238, 6, 0, 73, 254, 30, 0],
            &[0, 158, 234, 211, 154, 212, 30, 0],
            &[0, 0, 14, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'b' / 0x62
        'b' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[1, 30, 0, 0, 0, 0, 0, 0],
            &[10, 255, 1, 0, 0, 0, 0, 0],
            &[10, 255, 1, 0, 0, 0, 0, 0],
            &[10, 255, 31, 125, 113, 19, 0, 0],
            &[10, 252, 192, 99, 145, 221, 8, 0],
            &[10, 255, 57, 0, 0, 196, 95, 0],
            &[10, 255, 9, 0, 0, 152, 133, 0],
            &[10, 255, 21, 0, 0, 163, 123, 0],
            &[10, 255, 113, 0, 18, 232, 53, 0],
            &[10, 229, 153, 208, 231, 128, 0, 0],
            &[0, 0, 0, 16, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'c' / 0x63
        'c' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 122, 130, 80, 1, 0],
            &[0, 79, 235, 127, 108, 145, 0, 0],
            &[0, 202, 92, 0, 0, 0, 0, 0],
            &[0, 242, 38, 0, 0, 0, 0, 0],
            &[0, 235, 49, 0, 0, 0, 0, 0],
            &[0, 170, 151, 1, 0, 13, 0, 0],
            &[0, 27, 195, 230, 216, 221, 0, 0],
            &[0, 0, 0, 16, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'd' / 0x64
        'd' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 7, 0],
            &[0, 0, 0, 0, 0, 211, 61, 0],
            &[0, 0, 0, 0, 0, 211, 61, 0],
            &[0, 10, 102, 132, 63, 210, 61, 0],
            &[0, 190, 184, 94, 177, 238, 61, 0],
            &[47, 242, 12, 0, 12, 250, 61, 0],
            &[80, 205, 0, 0, 0, 221, 61, 0],
            &[71, 212, 0, 0, 0, 225, 61, 0],
            &[16, 240, 47, 0, 56, 255, 61, 0],
            &[0, 88, 232, 206, 186, 196, 61, 0],
            &[0, 0, 2, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'e' / 0x65
        'e' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 131, 111, 17, 0, 0],
            &[0, 132, 203, 97, 144, 215, 10, 0],
            &[20, 245, 20, 0, 0, 191, 87, 0],
            &[62, 250, 221, 221, 221, 240, 118, 0],
            &[52, 226, 0, 0, 0, 0, 0, 0],
            &[3, 224, 83, 0, 0, 17, 12, 0],
            &[0, 46, 205, 224, 210, 218, 34, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'f' / 0x66
        'f' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 39, 10, 0],
            &[0, 0, 0, 160, 225, 187, 153, 0],
            &[0, 0, 17, 254, 19, 0, 0, 0],
            &[15, 84, 118, 251, 110, 110, 43, 0],
            &[24, 119, 134, 252, 119, 119, 47, 0],
            &[0, 0, 27, 248, 0, 0, 0, 0],
            &[0, 0, 27, 248, 0, 0, 0, 0],
            &[0, 0, 27, 248, 0, 0, 0, 0],
            &[0, 0, 27, 248, 0, 0, 0, 0],
            &[0, 0, 27, 248, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'g' / 0x67
        'g' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 96, 126, 58, 73, 26, 0],
            &[0, 170, 179, 97, 172, 235, 61, 0],
            &[37, 240, 9, 0, 4, 233, 61, 0],
            &[75, 204, 0, 0, 0, 192, 61, 0],
            &[64, 215, 0, 0, 0, 201, 61, 0],
            &[16, 240, 54, 0, 36, 248, 61, 0],
            &[0, 83, 231, 205, 204, 216, 61, 0],
            &[0, 0, 1, 18, 0, 208, 53, 0],
            &[0, 163, 217, 224, 197, 62, 0, 0],
        ],
        // letter: 'h' / 0x68
        'h' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[1, 30, 1, 0, 0, 0, 0, 0],
            &[10, 255, 10, 0, 0, 0, 0, 0],
            &[10, 255, 10, 0, 0, 0, 0, 0],
            &[10, 255, 28, 115, 123, 33, 0, 0],
            &[10, 255, 177, 98, 140, 235, 15, 0],
            &[10, 255, 64, 0, 0, 219, 64, 0],
            &[10, 255, 14, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'i' / 0x69
        'i' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 0, 0, 0, 0],
            &[0, 0, 3, 242, 84, 0, 0, 0],
            &[0, 0, 0, 72, 16, 0, 0, 0],
            &[0, 93, 110, 110, 36, 0, 0, 0],
            &[0, 74, 105, 220, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[7, 195, 215, 248, 234, 205, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'j' / 0x6a
        'j' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 8, 0, 0],
            &[0, 0, 0, 0, 136, 191, 0, 0],
            &[0, 0, 0, 0, 33, 54, 0, 0],
            &[0, 0, 45, 110, 110, 84, 0, 0],
            &[0, 0, 34, 97, 158, 195, 0, 0],
            &[0, 0, 0, 0, 80, 195, 0, 0],
            &[0, 0, 0, 0, 80, 195, 0, 0],
            &[0, 0, 0, 0, 80, 195, 0, 0],
            &[0, 0, 0, 0, 80, 195, 0, 0],
            &[0, 0, 0, 0, 80, 195, 0, 0],
            &[0, 0, 0, 0, 87, 192, 0, 0],
            &[0, 54, 220, 228, 172, 21, 0, 0],
        ],
        // letter: 'k' / 0x6b
        'k' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 10, 0, 0, 0, 0, 0],
            &[0, 189, 87, 0, 0, 0, 0, 0],
            &[0, 189, 87, 0, 0, 0, 0, 0],
            &[0, 189, 87, 0, 0, 90, 52, 0],
            &[0, 189, 87, 0, 119, 197, 10, 0],
            &[0, 189, 85, 95, 209, 17, 0, 0],
            &[0, 189, 142, 250, 84, 0, 0, 0],
            &[0, 189, 200, 77, 225, 21, 0, 0],
            &[0, 189, 87, 0, 127, 183, 1, 0],
            &[0, 189, 87, 0, 3, 194, 119, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'l' / 0x6c
        'l' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 29, 30, 30, 12, 0, 0, 0],
            &[0, 162, 185, 237, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[7, 194, 214, 246, 236, 205, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'm' / 0x6d
        'm' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[44, 54, 122, 59, 62, 117, 11, 0],
            &[102, 230, 134, 238, 180, 190, 112, 0],
            &[102, 172, 0, 231, 58, 95, 148, 0],
            &[102, 148, 0, 222, 29, 89, 156, 0],
            &[102, 143, 0, 222, 23, 89, 156, 0],
            &[102, 143, 0, 222, 23, 89, 156, 0],
            &[102, 143, 0, 222, 23, 89, 156, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'n' / 0x6e
        'n' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[4, 95, 24, 117, 126, 39, 0, 0],
            &[10, 247, 181, 99, 129, 238, 18, 0],
            &[10, 255, 59, 0, 0, 213, 67, 0],
            &[10, 255, 16, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'o' / 0x6f
        'o' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 87, 133, 100, 10, 0, 0],
            &[0, 166, 184, 94, 153, 204, 7, 0],
            &[39, 238, 9, 0, 0, 198, 88, 0],
            &[79, 195, 0, 0, 0, 147, 130, 0],
            &[69, 206, 0, 0, 0, 160, 121, 0],
            &[12, 237, 53, 0, 23, 234, 50, 0],
            &[0, 71, 221, 202, 226, 104, 0, 0],
            &[0, 0, 0, 18, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'p' / 0x70
        'p' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[4, 97, 31, 124, 113, 20, 0, 0],
            &[10, 249, 181, 97, 141, 222, 9, 0],
            &[10, 255, 49, 0, 0, 191, 96, 0],
            &[10, 255, 11, 0, 0, 151, 133, 0],
            &[10, 255, 25, 0, 0, 163, 122, 0],
            &[10, 255, 109, 0, 17, 231, 59, 0],
            &[10, 253, 150, 207, 233, 139, 0, 0],
            &[10, 255, 7, 13, 12, 0, 0, 0],
            &[9, 225, 6, 0, 0, 0, 0, 0],
        ],
        // letter: 'q' / 0x71
        'q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 95, 131, 51, 73, 26, 0],
            &[0, 170, 175, 96, 173, 230, 61, 0],
            &[41, 240, 7, 0, 13, 248, 61, 0],
            &[81, 203, 0, 0, 0, 219, 61, 0],
            &[72, 215, 0, 0, 0, 224, 61, 0],
            &[15, 239, 53, 0, 53, 254, 61, 0],
            &[0, 84, 231, 206, 178, 217, 61, 0],
            &[0, 0, 2, 21, 0, 211, 61, 0],
            &[0, 0, 0, 0, 0, 186, 54, 0],
        ],
        // letter: 'r' / 0x72
        'r' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[7, 110, 110, 65, 52, 133, 89, 0],
            &[5, 92, 155, 203, 202, 111, 97, 0],
            &[0, 0, 76, 251, 34, 0, 0, 0],
            &[0, 0, 76, 210, 0, 0, 0, 0],
            &[0, 0, 76, 202, 0, 0, 0, 0],
            &[0, 0, 76, 202, 0, 0, 0, 0],
            &[47, 202, 232, 250, 212, 127, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 's' / 0x73
        's' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 82, 134, 124, 58, 0, 0],
            &[0, 115, 210, 105, 108, 105, 0, 0],
            &[0, 135, 160, 2, 0, 0, 0, 0],
            &[0, 17, 170, 217, 105, 6, 0, 0],
            &[0, 0, 0, 42, 178, 172, 0, 0],
            &[0, 27, 0, 0, 59, 227, 0, 0],
            &[0, 157, 227, 203, 229, 98, 0, 0],
            &[0, 0, 10, 25, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 't' / 0x74
        't' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 19, 0, 0, 0, 0],
            &[0, 0, 113, 128, 0, 0, 0, 0],
            &[24, 93, 191, 183, 110, 110, 1, 0],
            &[39, 119, 198, 188, 119, 119, 1, 0],
            &[0, 0, 147, 128, 0, 0, 0, 0],
            &[0, 0, 147, 128, 0, 0, 0, 0],
            &[0, 0, 147, 129, 0, 0, 0, 0],
            &[0, 0, 132, 172, 0, 0, 0, 0],
            &[0, 0, 34, 218, 213, 209, 42, 0],
            &[0, 0, 0, 0, 17, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'u' / 0x75
        'u' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[11, 108, 0, 0, 0, 92, 26, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 215, 61, 0],
            &[25, 251, 0, 0, 0, 235, 61, 0],
            &[5, 251, 41, 0, 52, 255, 61, 0],
            &[0, 124, 237, 201, 178, 193, 61, 0],
            &[0, 0, 9, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'v' / 0x76
        'v' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[42, 79, 0, 0, 0, 61, 62, 0],
            &[31, 239, 7, 0, 0, 204, 76, 0],
            &[0, 190, 83, 0, 36, 232, 4, 0],
            &[0, 96, 173, 0, 124, 141, 0, 0],
            &[0, 12, 237, 15, 212, 46, 0, 0],
            &[0, 0, 163, 142, 206, 0, 0, 0],
            &[0, 0, 68, 254, 111, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'w' / 0x77
        'w' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[85, 25, 0, 0, 0, 0, 107, 0],
            &[168, 86, 1, 56, 12, 28, 221, 0],
            &[127, 123, 52, 251, 99, 69, 179, 0],
            &[86, 161, 126, 159, 169, 109, 137, 0],
            &[45, 194, 194, 40, 224, 145, 95, 0],
            &[7, 215, 213, 0, 182, 207, 53, 0],
            &[0, 217, 157, 0, 117, 253, 12, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'x' / 0x78
        'x' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[19, 107, 5, 0, 0, 91, 42, 0],
            &[0, 167, 138, 0, 86, 209, 8, 0],
            &[0, 14, 218, 97, 227, 39, 0, 0],
            &[0, 0, 61, 255, 110, 0, 0, 0],
            &[0, 0, 155, 198, 198, 4, 0, 0],
            &[0, 89, 207, 6, 164, 137, 0, 0],
            &[36, 231, 39, 0, 14, 220, 72, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'y' / 0x79
        'y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[40, 75, 0, 0, 0, 58, 62, 0],
            &[24, 233, 9, 0, 0, 196, 77, 0],
            &[0, 167, 97, 0, 27, 231, 4, 0],
            &[0, 61, 201, 0, 111, 143, 0, 0],
            &[0, 0, 210, 47, 196, 48, 0, 0],
            &[0, 0, 104, 161, 207, 0, 0, 0],
            &[0, 0, 12, 240, 114, 0, 0, 0],
            &[0, 0, 5, 229, 23, 0, 0, 0],
            &[49, 220, 191, 20, 0, 0, 0, 0],
        ],
        // letter: 'z' / 0x7a
        'z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 80, 110, 110, 110, 110, 2, 0],
            &[0, 73, 101, 101, 159, 225, 2, 0],
            &[0, 0, 0, 24, 223, 52, 0, 0],
            &[0, 0, 4, 193, 100, 0, 0, 0],
            &[0, 0, 140, 155, 0, 0, 0, 0],
            &[0, 83, 202, 7, 0, 0, 0, 0],
            &[0, 228, 242, 233, 233, 233, 28, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '{' / 0x7b
        '{' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 103, 72, 0, 0],
            &[0, 0, 0, 162, 179, 45, 0, 0],
            &[0, 0, 0, 217, 53, 0, 0, 0],
            &[0, 0, 0, 219, 48, 0, 0, 0],
            &[0, 0, 11, 238, 30, 0, 0, 0],
            &[0, 65, 232, 116, 0, 0, 0, 0],
            &[0, 9, 71, 234, 16, 0, 0, 0],
            &[0, 0, 0, 220, 47, 0, 0, 0],
            &[0, 0, 0, 219, 48, 0, 0, 0],
            &[0, 0, 0, 195, 109, 1, 0, 0],
            &[0, 0, 0, 48, 181, 112, 0, 0],
        ],
        // letter: '|' / 0x7c
        '|' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 1, 0, 0, 0],
            &[0, 0, 0, 212, 11, 0, 0, 0],
            &[0, 0, 0, 212, 11, 0, 0, 0],
            &[0, 0, 0, 212, 11, 0, 0, 0],
            &[0, 0, 0, 212, 11, 0, 0, 0],
            &[0, 0, 0, 212, 11, 0, 0, 0],
            &[0, 0, 0, 212, 11, 0, 0, 0],
            &[0, 0, 0, 212, 11, 0, 0, 0],
            &[0, 0, 0, 212, 11, 0, 0, 0],
            &[0, 0, 0, 212, 11, 0, 0, 0],
            &[0, 0, 0, 212, 11, 0, 0, 0],
            &[0, 0, 0, 187, 9, 0, 0, 0],
        ],
        // letter: '}' / 0x7d
        '}' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 44, 109, 23, 0, 0, 0, 0],
            &[0, 25, 140, 214, 0, 0, 0, 0],
            &[0, 0, 2, 253, 12, 0, 0, 0],
            &[0, 0, 0, 252, 14, 0, 0, 0],
            &[0, 0, 0, 236, 44, 0, 0, 0],
            &[0, 0, 0, 78, 228, 105, 0, 0],
            &[0, 0, 0, 202, 113, 18, 0, 0],
            &[0, 0, 0, 252, 15, 0, 0, 0],
            &[0, 0, 0, 253, 14, 0, 0, 0],
            &[0, 0, 68, 242, 3, 0, 0, 0],
            &[0, 71, 201, 76, 0, 0, 0, 0],
        ],
        // letter: '~' / 0x7e
        '~' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 124, 29, 0, 68, 33, 0],
            &[0, 194, 72, 192, 82, 190, 17, 0],
            &[9, 93, 0, 15, 120, 68, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ' ' / 0xa0
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¡' / 0xa1
        '¡' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 93, 13, 0, 0, 0],
            &[0, 0, 15, 253, 64, 0, 0, 0],
            &[0, 0, 0, 19, 0, 0, 0, 0],
            &[0, 0, 0, 208, 16, 0, 0, 0],
            &[0, 0, 0, 233, 25, 0, 0, 0],
            &[0, 0, 0, 241, 33, 0, 0, 0],
            &[0, 0, 0, 249, 41, 0, 0, 0],
            &[0, 0, 1, 255, 49, 0, 0, 0],
            &[0, 0, 3, 56, 13, 0, 0, 0],
        ],
        // letter: '¢' / 0xa2
        '¢' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 80, 62, 0, 0, 0],
            &[0, 0, 76, 202, 204, 141, 5, 0],
            &[0, 90, 221, 78, 56, 89, 0, 0],
            &[0, 198, 85, 0, 0, 0, 0, 0],
            &[0, 229, 46, 0, 0, 0, 0, 0],
            &[0, 214, 67, 0, 0, 0, 0, 0],
            &[0, 136, 189, 20, 3, 44, 1, 0],
            &[0, 7, 144, 235, 239, 194, 1, 0],
            &[0, 0, 0, 117, 88, 0, 0, 0],
            &[0, 0, 0, 14, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '£' / 0xa3
        '£' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 118, 163, 120, 9, 0],
            &[0, 0, 136, 179, 66, 113, 3, 0],
            &[0, 0, 198, 71, 0, 0, 0, 0],
            &[0, 0, 205, 64, 0, 0, 0, 0],
            &[0, 142, 237, 185, 161, 60, 0, 0],
            &[0, 36, 213, 95, 40, 15, 0, 0],
            &[0, 0, 212, 57, 0, 0, 0, 0],
            &[0, 43, 218, 8, 0, 0, 0, 0],
            &[0, 232, 251, 245, 245, 245, 73, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¤' / 0xa4
        '¤' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 0, 0, 0, 1, 0, 0],
            &[6, 196, 117, 173, 128, 191, 33, 0],
            &[0, 133, 152, 34, 104, 189, 0, 0],
            &[0, 189, 25, 0, 0, 213, 0, 0],
            &[0, 151, 84, 0, 32, 204, 0, 0],
            &[5, 194, 201, 206, 205, 214, 26, 0],
            &[0, 45, 0, 7, 0, 42, 5, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¥' / 0xa5
        '¥' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[63, 98, 0, 0, 0, 67, 91, 0],
            &[20, 234, 24, 0, 2, 215, 57, 0],
            &[0, 133, 140, 0, 90, 178, 0, 0],
            &[0, 17, 229, 23, 213, 46, 0, 0],
            &[0, 11, 140, 207, 177, 14, 0, 0],
            &[0, 117, 179, 250, 187, 153, 0, 0],
            &[0, 53, 80, 243, 99, 69, 0, 0],
            &[0, 75, 115, 245, 130, 99, 0, 0],
            &[0, 0, 0, 237, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¦' / 0xa6
        '¦' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 0, 0, 0, 0],
            &[0, 0, 0, 215, 8, 0, 0, 0],
            &[0, 0, 0, 215, 8, 0, 0, 0],
            &[0, 0, 0, 215, 8, 0, 0, 0],
            &[0, 0, 0, 215, 8, 0, 0, 0],
            &[0, 0, 0, 97, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 146, 5, 0, 0, 0],
            &[0, 0, 0, 215, 8, 0, 0, 0],
            &[0, 0, 0, 215, 8, 0, 0, 0],
            &[0, 0, 0, 192, 7, 0, 0, 0],
        ],
        // letter: '§' / 0xa7
        '§' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 37, 26, 0, 0, 0],
            &[0, 56, 218, 167, 176, 133, 0, 0],
            &[0, 123, 144, 0, 0, 0, 0, 0],
            &[0, 26, 225, 176, 60, 0, 0, 0],
            &[0, 90, 161, 68, 190, 123, 0, 0],
            &[0, 132, 109, 0, 20, 213, 0, 0],
            &[0, 26, 198, 169, 161, 112, 0, 0],
            &[0, 0, 0, 68, 201, 120, 0, 0],
            &[0, 8, 0, 0, 56, 205, 0, 0],
            &[0, 144, 195, 172, 209, 87, 0, 0],
            &[0, 0, 11, 23, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¨' / 0xa8
        '¨' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 62, 161, 0, 115, 108, 0, 0],
            &[0, 29, 88, 0, 61, 56, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '©' / 0xa9
        '©' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 92, 66, 3, 0, 0],
            &[0, 140, 115, 101, 116, 171, 10, 0],
            &[62, 108, 144, 120, 86, 60, 115, 0],
            &[112, 46, 176, 0, 0, 0, 158, 0],
            &[91, 74, 179, 17, 13, 22, 140, 0],
            &[13, 176, 64, 140, 92, 157, 39, 0],
            &[0, 26, 137, 144, 143, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ª' / 0xaa
        'ª' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 159, 106, 0, 0, 0],
            &[0, 0, 29, 9, 189, 24, 0, 0],
            &[0, 4, 171, 141, 206, 34, 0, 0],
            &[0, 27, 197, 68, 220, 34, 0, 0],
            &[0, 0, 64, 78, 43, 11, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '«' / 0xab
        '«' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 38, 91, 3, 123, 4, 0],
            &[0, 15, 211, 65, 144, 144, 0, 0],
            &[0, 177, 115, 92, 197, 4, 0, 0],
            &[0, 115, 174, 45, 220, 29, 0, 0],
            &[0, 0, 160, 118, 76, 200, 4, 0],
            &[0, 0, 7, 31, 0, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¬' / 0xac
        '¬' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[16, 167, 167, 167, 167, 167, 30, 0],
            &[5, 50, 50, 50, 50, 188, 46, 0],
            &[0, 0, 0, 0, 0, 171, 46, 0],
            &[0, 0, 0, 0, 0, 79, 21, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '®' / 0xae
        '®' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 48, 92, 66, 3, 0, 0],
            &[0, 140, 137, 103, 102, 171, 10, 0],
            &[62, 108, 122, 120, 170, 60, 115, 0],
            &[112, 46, 122, 136, 145, 0, 158, 0],
            &[91, 74, 122, 83, 141, 22, 140, 0],
            &[13, 176, 88, 17, 95, 166, 39, 0],
            &[0, 26, 137, 144, 143, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¯' / 0xaf
        '¯' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 68, 68, 68, 26, 0, 0],
            &[0, 26, 149, 149, 149, 57, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '°' / 0xb0
        '°' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 65, 158, 94, 0, 0, 0],
            &[0, 15, 185, 33, 150, 59, 0, 0],
            &[0, 30, 163, 0, 106, 81, 0, 0],
            &[0, 0, 128, 190, 161, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '±' / 0xb1
        '±' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 0, 0, 0, 0],
            &[0, 0, 0, 219, 2, 0, 0, 0],
            &[0, 0, 0, 219, 2, 0, 0, 0],
            &[15, 177, 177, 244, 177, 177, 42, 0],
            &[3, 40, 40, 224, 42, 40, 9, 0],
            &[0, 0, 0, 219, 2, 0, 0, 0],
            &[0, 0, 0, 112, 1, 0, 0, 0],
            &[8, 218, 218, 218, 218, 218, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '²' / 0xb2
        '²' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 48, 17, 0, 0, 0],
            &[0, 1, 167, 133, 212, 26, 0, 0],
            &[0, 0, 0, 0, 159, 51, 0, 0],
            &[0, 0, 0, 71, 165, 0, 0, 0],
            &[0, 0, 73, 166, 5, 0, 0, 0],
            &[0, 25, 248, 192, 188, 86, 0, 0],
            &[0, 0, 2, 2, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '³' / 0xb3
        '³' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 50, 15, 0, 0, 0],
            &[0, 3, 164, 130, 217, 15, 0, 0],
            &[0, 0, 0, 17, 208, 18, 0, 0],
            &[0, 0, 74, 188, 184, 8, 0, 0],
            &[0, 0, 0, 0, 154, 69, 0, 0],
            &[0, 27, 182, 165, 199, 18, 0, 0],
            &[0, 0, 5, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '´' / 0xb4
        '´' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 17, 0, 0],
            &[0, 0, 0, 80, 212, 15, 0, 0],
            &[0, 0, 2, 147, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'µ' / 0xb5
        'µ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[11, 108, 0, 0, 0, 92, 26, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 215, 61, 0],
            &[43, 255, 0, 0, 0, 235, 61, 0],
            &[31, 255, 41, 0, 52, 255, 61, 0],
            &[25, 255, 237, 201, 178, 193, 61, 0],
            &[25, 250, 9, 19, 0, 0, 0, 0],
            &[22, 220, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¶' / 0xb6
        '¶' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 30, 30, 30, 14, 0],
            &[0, 152, 253, 255, 204, 167, 119, 0],
            &[53, 255, 255, 255, 140, 58, 119, 0],
            &[94, 255, 255, 255, 140, 58, 119, 0],
            &[90, 255, 255, 255, 140, 58, 119, 0],
            &[36, 253, 255, 255, 140, 58, 119, 0],
            &[0, 96, 200, 209, 140, 58, 119, 0],
            &[0, 0, 0, 37, 140, 58, 119, 0],
            &[0, 0, 0, 37, 140, 58, 119, 0],
            &[0, 0, 0, 37, 140, 58, 119, 0],
            &[0, 0, 0, 37, 140, 58, 119, 0],
            &[0, 0, 0, 19, 73, 30, 62, 0],
        ],
        // letter: '·' / 0xb7
        '·' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 204, 45, 0, 0, 0],
            &[0, 0, 5, 162, 31, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¸' / 0xb8
        '¸' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 0, 0, 0, 0],
            &[0, 0, 7, 203, 16, 0, 0, 0],
            &[0, 0, 103, 180, 84, 0, 0, 0],
        ],
        // letter: '¹' / 0xb9
        '¹' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 5, 0, 0, 0],
            &[0, 12, 161, 237, 45, 0, 0, 0],
            &[0, 1, 61, 153, 45, 0, 0, 0],
            &[0, 0, 0, 154, 45, 0, 0, 0],
            &[0, 0, 0, 154, 45, 0, 0, 0],
            &[0, 0, 135, 255, 214, 72, 0, 0],
            &[0, 0, 1, 3, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'º' / 0xba
        'º' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 68, 155, 87, 0, 0, 0],
            &[0, 10, 191, 10, 161, 42, 0, 0],
            &[0, 32, 158, 0, 113, 79, 0, 0],
            &[0, 4, 191, 73, 186, 28, 0, 0],
            &[0, 0, 23, 91, 37, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '»' / 0xbb
        '»' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 113, 21, 56, 75, 0, 0, 0],
            &[0, 87, 195, 30, 216, 44, 0, 0],
            &[0, 0, 145, 150, 58, 221, 11, 0],
            &[0, 8, 202, 81, 127, 165, 0, 0],
            &[0, 156, 125, 72, 202, 9, 0, 0],
            &[0, 36, 0, 12, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¼' / 0xbc
        '¼' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[36, 139, 12, 0, 0, 103, 29, 0],
            &[136, 204, 22, 0, 35, 198, 0, 0],
            &[0, 176, 22, 0, 162, 71, 0, 0],
            &[6, 201, 38, 42, 192, 0, 0, 0],
            &[93, 255, 165, 181, 62, 29, 12, 0],
            &[0, 0, 49, 185, 26, 232, 67, 0],
            &[0, 0, 180, 56, 170, 156, 67, 0],
            &[0, 58, 177, 84, 213, 206, 164, 0],
            &[0, 189, 47, 6, 16, 137, 77, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '½' / 0xbd
        '½' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[36, 139, 12, 0, 0, 103, 29, 0],
            &[136, 204, 22, 0, 35, 198, 0, 0],
            &[0, 176, 22, 0, 162, 71, 0, 0],
            &[6, 201, 38, 42, 192, 0, 0, 0],
            &[93, 255, 165, 181, 78, 55, 5, 0],
            &[0, 0, 49, 185, 139, 143, 164, 0],
            &[0, 0, 180, 54, 0, 57, 160, 0],
            &[0, 58, 177, 0, 51, 180, 16, 0],
            &[0, 189, 47, 4, 241, 200, 169, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¾' / 0xbe
        '¾' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[76, 162, 101, 0, 0, 103, 29, 0],
            &[35, 19, 230, 1, 35, 198, 0, 0],
            &[35, 191, 193, 0, 162, 71, 0, 0],
            &[33, 18, 217, 63, 192, 0, 0, 0],
            &[94, 163, 98, 171, 62, 29, 12, 0],
            &[0, 0, 49, 185, 26, 232, 67, 0],
            &[0, 0, 180, 56, 170, 156, 67, 0],
            &[0, 58, 177, 84, 213, 206, 164, 0],
            &[0, 189, 47, 6, 16, 137, 77, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¿' / 0xbf
        '¿' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 16, 0, 0],
            &[0, 0, 0, 8, 250, 73, 0, 0],
            &[0, 0, 0, 0, 19, 0, 0, 0],
            &[0, 0, 0, 0, 199, 27, 0, 0],
            &[0, 0, 0, 119, 197, 1, 0, 0],
            &[0, 0, 136, 182, 13, 0, 0, 0],
            &[0, 23, 236, 9, 0, 0, 0, 0],
            &[0, 41, 228, 4, 0, 0, 5, 0],
            &[0, 0, 1, 55, 79, 49, 2, 0],
        ],
        // letter: 'À' / 0xc0
        'À' => &[
            &[0, 0, 77, 31, 0, 0, 0, 0],
            &[0, 0, 100, 195, 1, 0, 0, 0],
            &[0, 0, 0, 90, 47, 0, 0, 0],
            &[0, 0, 11, 154, 40, 0, 0, 0],
            &[0, 0, 85, 242, 133, 0, 0, 0],
            &[0, 0, 168, 127, 216, 0, 0, 0],
            &[0, 7, 237, 15, 218, 44, 0, 0],
            &[0, 77, 186, 0, 138, 127, 0, 0],
            &[0, 160, 242, 150, 207, 216, 0, 0],
            &[5, 245, 123, 95, 97, 250, 37, 0],
            &[70, 204, 0, 0, 0, 156, 120, 0],
            &[153, 124, 0, 0, 0, 76, 204, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Á' / 0xc1
        'Á' => &[
            &[0, 0, 0, 0, 70, 39, 0, 0],
            &[0, 0, 0, 62, 216, 20, 0, 0],
            &[0, 0, 0, 112, 25, 0, 0, 0],
            &[0, 0, 11, 154, 40, 0, 0, 0],
            &[0, 0, 85, 242, 133, 0, 0, 0],
            &[0, 0, 168, 127, 216, 0, 0, 0],
            &[0, 7, 237, 15, 218, 44, 0, 0],
            &[0, 77, 186, 0, 138, 127, 0, 0],
            &[0, 160, 242, 150, 207, 216, 0, 0],
            &[5, 245, 123, 95, 97, 250, 37, 0],
            &[70, 204, 0, 0, 0, 156, 120, 0],
            &[153, 124, 0, 0, 0, 76, 204, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Â' / 0xc2
        'Â' => &[
            &[0, 0, 6, 83, 20, 0, 0, 0],
            &[0, 0, 147, 174, 184, 4, 0, 0],
            &[0, 39, 98, 0, 73, 69, 0, 0],
            &[0, 0, 11, 154, 40, 0, 0, 0],
            &[0, 0, 85, 242, 133, 0, 0, 0],
            &[0, 0, 168, 127, 216, 0, 0, 0],
            &[0, 7, 237, 15, 218, 44, 0, 0],
            &[0, 77, 186, 0, 138, 127, 0, 0],
            &[0, 160, 242, 150, 207, 216, 0, 0],
            &[5, 245, 123, 95, 97, 250, 37, 0],
            &[70, 204, 0, 0, 0, 156, 120, 0],
            &[153, 124, 0, 0, 0, 76, 204, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ã' / 0xc3
        'Ã' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 211, 177, 89, 148, 0, 0],
            &[0, 75, 28, 56, 134, 28, 0, 0],
            &[0, 0, 11, 154, 40, 0, 0, 0],
            &[0, 0, 85, 242, 133, 0, 0, 0],
            &[0, 0, 168, 127, 216, 0, 0, 0],
            &[0, 7, 237, 15, 218, 44, 0, 0],
            &[0, 77, 186, 0, 138, 127, 0, 0],
            &[0, 160, 242, 150, 207, 216, 0, 0],
            &[5, 245, 123, 95, 97, 250, 37, 0],
            &[70, 204, 0, 0, 0, 156, 120, 0],
            &[153, 124, 0, 0, 0, 76, 204, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ä' / 0xc4
        'Ä' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 76, 186, 0, 138, 124, 0, 0],
            &[0, 18, 59, 0, 42, 36, 0, 0],
            &[0, 0, 11, 154, 40, 0, 0, 0],
            &[0, 0, 85, 242, 133, 0, 0, 0],
            &[0, 0, 168, 127, 216, 0, 0, 0],
            &[0, 7, 237, 15, 218, 44, 0, 0],
            &[0, 77, 186, 0, 138, 127, 0, 0],
            &[0, 160, 242, 150, 207, 216, 0, 0],
            &[5, 245, 123, 95, 97, 250, 37, 0],
            &[70, 204, 0, 0, 0, 156, 120, 0],
            &[153, 124, 0, 0, 0, 76, 204, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Å' / 0xc5
        'Å' => &[
            &[0, 0, 95, 173, 132, 0, 0, 0],
            &[0, 0, 169, 40, 188, 0, 0, 0],
            &[0, 0, 39, 140, 62, 0, 0, 0],
            &[0, 0, 11, 154, 40, 0, 0, 0],
            &[0, 0, 85, 242, 133, 0, 0, 0],
            &[0, 0, 168, 127, 216, 0, 0, 0],
            &[0, 7, 237, 15, 218, 44, 0, 0],
            &[0, 77, 186, 0, 138, 127, 0, 0],
            &[0, 160, 242, 150, 207, 216, 0, 0],
            &[5, 245, 123, 95, 97, 250, 37, 0],
            &[70, 204, 0, 0, 0, 156, 120, 0],
            &[153, 124, 0, 0, 0, 76, 204, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Æ' / 0xc6
        'Æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 131, 145, 145, 145, 75, 0],
            &[0, 20, 230, 205, 155, 100, 51, 0],
            &[0, 77, 170, 172, 91, 0, 0, 0],
            &[0, 134, 116, 172, 91, 0, 0, 0],
            &[0, 192, 61, 172, 247, 242, 69, 0],
            &[4, 255, 152, 255, 91, 0, 0, 0],
            &[51, 253, 95, 230, 91, 0, 0, 0],
            &[108, 154, 0, 172, 91, 0, 0, 0],
            &[165, 99, 0, 172, 249, 245, 126, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ç' / 0xc7
        'Ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 43, 143, 169, 139, 43, 0],
            &[0, 62, 241, 128, 79, 115, 60, 0],
            &[0, 207, 104, 0, 0, 0, 0, 0],
            &[23, 255, 12, 0, 0, 0, 0, 0],
            &[48, 237, 0, 0, 0, 0, 0, 0],
            &[37, 243, 0, 0, 0, 0, 0, 0],
            &[4, 242, 44, 0, 0, 0, 0, 0],
            &[0, 144, 189, 18, 0, 13, 12, 0],
            &[0, 7, 155, 247, 240, 240, 46, 0],
            &[0, 0, 0, 95, 160, 5, 0, 0],
            &[0, 0, 6, 170, 172, 19, 0, 0],
        ],
        // letter: 'È' / 0xc8
        'È' => &[
            &[0, 0, 71, 37, 0, 0, 0, 0],
            &[0, 0, 84, 207, 5, 0, 0, 0],
            &[0, 0, 0, 79, 58, 0, 0, 0],
            &[0, 125, 145, 145, 145, 145, 26, 0],
            &[0, 220, 134, 100, 100, 100, 18, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 244, 244, 232, 0, 0],
            &[0, 220, 57, 1, 1, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 245, 245, 245, 44, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'É' / 0xc9
        'É' => &[
            &[0, 0, 0, 0, 64, 45, 0, 0],
            &[0, 0, 0, 47, 222, 28, 0, 0],
            &[0, 0, 0, 103, 33, 0, 0, 0],
            &[0, 125, 145, 145, 145, 145, 26, 0],
            &[0, 220, 134, 100, 100, 100, 18, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 244, 244, 232, 0, 0],
            &[0, 220, 57, 1, 1, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 245, 245, 245, 44, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ê' / 0xca
        'Ê' => &[
            &[0, 0, 2, 81, 26, 0, 0, 0],
            &[0, 0, 129, 177, 194, 9, 0, 0],
            &[0, 29, 108, 0, 62, 80, 0, 0],
            &[0, 125, 145, 145, 145, 145, 26, 0],
            &[0, 220, 134, 100, 100, 100, 18, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 244, 244, 232, 0, 0],
            &[0, 220, 57, 1, 1, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 245, 245, 245, 44, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ë' / 0xcb
        'Ë' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 60, 202, 0, 120, 142, 0, 0],
            &[0, 13, 65, 0, 34, 43, 0, 0],
            &[0, 125, 145, 145, 145, 145, 26, 0],
            &[0, 220, 134, 100, 100, 100, 18, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 244, 244, 232, 0, 0],
            &[0, 220, 57, 1, 1, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 245, 245, 245, 44, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ì' / 0xcc
        'Ì' => &[
            &[0, 0, 77, 31, 0, 0, 0, 0],
            &[0, 0, 100, 195, 1, 0, 0, 0],
            &[0, 0, 0, 90, 47, 0, 0, 0],
            &[0, 98, 145, 145, 145, 127, 0, 0],
            &[0, 35, 72, 247, 98, 48, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 134, 217, 254, 224, 176, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Í' / 0xcd
        'Í' => &[
            &[0, 0, 0, 0, 70, 39, 0, 0],
            &[0, 0, 0, 62, 216, 20, 0, 0],
            &[0, 0, 0, 112, 25, 0, 0, 0],
            &[0, 98, 145, 145, 145, 127, 0, 0],
            &[0, 35, 72, 247, 98, 48, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 134, 217, 254, 224, 176, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Î' / 0xce
        'Î' => &[
            &[0, 0, 6, 83, 20, 0, 0, 0],
            &[0, 0, 147, 174, 184, 4, 0, 0],
            &[0, 39, 98, 0, 73, 69, 0, 0],
            &[0, 98, 145, 145, 145, 127, 0, 0],
            &[0, 35, 72, 247, 98, 48, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 134, 217, 254, 224, 176, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ï' / 0xcf
        'Ï' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 76, 186, 0, 138, 124, 0, 0],
            &[0, 18, 59, 0, 42, 36, 0, 0],
            &[0, 98, 145, 145, 145, 127, 0, 0],
            &[0, 35, 72, 247, 98, 48, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 134, 217, 254, 224, 176, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ð' / 0xd0
        'Ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 145, 145, 131, 74, 1, 0, 0],
            &[10, 255, 100, 113, 207, 173, 1, 0],
            &[10, 255, 10, 0, 7, 212, 91, 0],
            &[10, 255, 10, 0, 0, 117, 169, 0],
            &[190, 255, 190, 90, 0, 88, 193, 0],
            &[69, 255, 69, 29, 0, 100, 179, 0],
            &[10, 255, 10, 0, 0, 168, 123, 0],
            &[10, 255, 10, 10, 114, 231, 21, 0],
            &[10, 255, 240, 234, 172, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ñ' / 0xd1
        'Ñ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 33, 208, 183, 88, 156, 0, 0],
            &[0, 66, 37, 48, 134, 36, 0, 0],
            &[23, 145, 56, 0, 0, 99, 52, 0],
            &[40, 240, 188, 0, 0, 175, 92, 0],
            &[40, 206, 204, 44, 0, 175, 92, 0],
            &[40, 221, 92, 155, 0, 175, 92, 0],
            &[40, 226, 5, 221, 20, 175, 92, 0],
            &[40, 226, 0, 123, 123, 173, 92, 0],
            &[40, 226, 0, 20, 220, 171, 92, 0],
            &[40, 226, 0, 0, 155, 235, 92, 0],
            &[40, 226, 0, 0, 43, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ò' / 0xd2
        'Ò' => &[
            &[0, 0, 77, 31, 0, 0, 0, 0],
            &[0, 0, 100, 195, 1, 0, 0, 0],
            &[0, 0, 0, 90, 47, 0, 0, 0],
            &[0, 9, 121, 173, 143, 26, 0, 0],
            &[0, 172, 176, 75, 144, 219, 8, 0],
            &[37, 239, 9, 0, 0, 200, 91, 0],
            &[93, 187, 0, 0, 0, 138, 146, 0],
            &[113, 168, 0, 0, 0, 118, 165, 0],
            &[106, 175, 0, 0, 0, 124, 157, 0],
            &[69, 216, 0, 0, 0, 163, 118, 0],
            &[7, 231, 81, 0, 39, 238, 35, 0],
            &[0, 61, 226, 225, 235, 95, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ó' / 0xd3
        'Ó' => &[
            &[0, 0, 0, 0, 70, 39, 0, 0],
            &[0, 0, 0, 62, 216, 20, 0, 0],
            &[0, 0, 0, 112, 25, 0, 0, 0],
            &[0, 9, 121, 173, 143, 26, 0, 0],
            &[0, 172, 176, 75, 144, 219, 8, 0],
            &[37, 239, 9, 0, 0, 200, 91, 0],
            &[93, 187, 0, 0, 0, 138, 146, 0],
            &[113, 168, 0, 0, 0, 118, 165, 0],
            &[106, 175, 0, 0, 0, 124, 157, 0],
            &[69, 216, 0, 0, 0, 163, 118, 0],
            &[7, 231, 81, 0, 39, 238, 35, 0],
            &[0, 61, 226, 225, 235, 95, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ô' / 0xd4
        'Ô' => &[
            &[0, 0, 6, 83, 20, 0, 0, 0],
            &[0, 0, 147, 174, 184, 4, 0, 0],
            &[0, 39, 98, 0, 73, 69, 0, 0],
            &[0, 9, 121, 173, 143, 26, 0, 0],
            &[0, 172, 176, 75, 144, 219, 8, 0],
            &[37, 239, 9, 0, 0, 200, 91, 0],
            &[93, 187, 0, 0, 0, 138, 146, 0],
            &[113, 168, 0, 0, 0, 118, 165, 0],
            &[106, 175, 0, 0, 0, 124, 157, 0],
            &[69, 216, 0, 0, 0, 163, 118, 0],
            &[7, 231, 81, 0, 39, 238, 35, 0],
            &[0, 61, 226, 225, 235, 95, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Õ' / 0xd5
        'Õ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 211, 177, 89, 148, 0, 0],
            &[0, 75, 28, 56, 134, 28, 0, 0],
            &[0, 9, 121, 173, 143, 26, 0, 0],
            &[0, 172, 176, 75, 144, 219, 8, 0],
            &[37, 239, 9, 0, 0, 200, 91, 0],
            &[93, 187, 0, 0, 0, 138, 146, 0],
            &[113, 168, 0, 0, 0, 118, 165, 0],
            &[106, 175, 0, 0, 0, 124, 157, 0],
            &[69, 216, 0, 0, 0, 163, 118, 0],
            &[7, 231, 81, 0, 39, 238, 35, 0],
            &[0, 61, 226, 225, 235, 95, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ö' / 0xd6
        'Ö' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 76, 186, 0, 138, 124, 0, 0],
            &[0, 18, 59, 0, 42, 36, 0, 0],
            &[0, 9, 121, 173, 143, 26, 0, 0],
            &[0, 172, 176, 75, 144, 219, 8, 0],
            &[37, 239, 9, 0, 0, 200, 91, 0],
            &[93, 187, 0, 0, 0, 138, 146, 0],
            &[113, 168, 0, 0, 0, 118, 165, 0],
            &[106, 175, 0, 0, 0, 124, 157, 0],
            &[69, 216, 0, 0, 0, 163, 118, 0],
            &[7, 231, 81, 0, 39, 238, 35, 0],
            &[0, 61, 226, 225, 235, 95, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '×' / 0xd7
        '×' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 134, 23, 0, 5, 142, 9, 0],
            &[0, 74, 207, 29, 177, 118, 0, 0],
            &[0, 0, 81, 247, 132, 0, 0, 0],
            &[0, 6, 179, 162, 208, 23, 0, 0],
            &[0, 167, 120, 0, 75, 205, 8, 0],
            &[0, 30, 0, 0, 0, 30, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ø' / 0xd8
        'Ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 121, 173, 143, 138, 88, 0],
            &[0, 172, 176, 75, 165, 255, 25, 0],
            &[37, 239, 9, 0, 151, 255, 91, 0],
            &[93, 187, 0, 44, 209, 142, 146, 0],
            &[113, 168, 0, 187, 71, 118, 165, 0],
            &[106, 175, 78, 181, 0, 124, 157, 0],
            &[69, 222, 213, 40, 0, 163, 118, 0],
            &[7, 255, 226, 0, 39, 238, 35, 0],
            &[21, 255, 244, 225, 235, 95, 0, 0],
            &[10, 53, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ù' / 0xd9
        'Ù' => &[
            &[0, 0, 77, 31, 0, 0, 0, 0],
            &[0, 0, 100, 195, 1, 0, 0, 0],
            &[0, 0, 0, 90, 47, 0, 0, 0],
            &[23, 133, 0, 0, 0, 104, 52, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[29, 251, 3, 0, 0, 202, 80, 0],
            &[1, 229, 81, 0, 35, 243, 27, 0],
            &[0, 73, 232, 223, 237, 107, 0, 0],
            &[0, 0, 3, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ú' / 0xda
        'Ú' => &[
            &[0, 0, 0, 0, 70, 39, 0, 0],
            &[0, 0, 0, 62, 216, 20, 0, 0],
            &[0, 0, 0, 112, 25, 0, 0, 0],
            &[23, 133, 0, 0, 0, 104, 52, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[29, 251, 3, 0, 0, 202, 80, 0],
            &[1, 229, 81, 0, 35, 243, 27, 0],
            &[0, 73, 232, 223, 237, 107, 0, 0],
            &[0, 0, 3, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Û' / 0xdb
        'Û' => &[
            &[0, 0, 6, 83, 20, 0, 0, 0],
            &[0, 0, 147, 174, 184, 4, 0, 0],
            &[0, 39, 98, 0, 73, 69, 0, 0],
            &[23, 133, 0, 0, 0, 104, 52, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[29, 251, 3, 0, 0, 202, 80, 0],
            &[1, 229, 81, 0, 35, 243, 27, 0],
            &[0, 73, 232, 223, 237, 107, 0, 0],
            &[0, 0, 3, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ü' / 0xdc
        'Ü' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 76, 186, 0, 138, 124, 0, 0],
            &[0, 18, 59, 0, 42, 36, 0, 0],
            &[23, 133, 0, 0, 0, 104, 52, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[29, 251, 3, 0, 0, 202, 80, 0],
            &[1, 229, 81, 0, 35, 243, 27, 0],
            &[0, 73, 232, 223, 237, 107, 0, 0],
            &[0, 0, 3, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ý' / 0xdd
        'Ý' => &[
            &[0, 0, 0, 0, 70, 39, 0, 0],
            &[0, 0, 0, 62, 216, 20, 0, 0],
            &[0, 0, 0, 112, 25, 0, 0, 0],
            &[63, 107, 0, 0, 0, 73, 92, 0],
            &[21, 239, 30, 0, 2, 219, 60, 0],
            &[0, 134, 144, 0, 85, 186, 0, 0],
            &[0, 18, 230, 18, 204, 56, 0, 0],
            &[0, 0, 130, 186, 181, 0, 0, 0],
            &[0, 0, 16, 246, 58, 0, 0, 0],
            &[0, 0, 0, 234, 29, 0, 0, 0],
            &[0, 0, 0, 234, 29, 0, 0, 0],
            &[0, 0, 0, 234, 29, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Þ' / 0xde
        'Þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 125, 31, 0, 0, 0, 0, 0],
            &[0, 220, 57, 1, 0, 0, 0, 0],
            &[0, 220, 241, 239, 228, 132, 0, 0],
            &[0, 220, 56, 0, 32, 224, 83, 0],
            &[0, 220, 56, 0, 0, 153, 129, 0],
            &[0, 220, 56, 0, 11, 206, 90, 0],
            &[0, 220, 191, 182, 229, 171, 3, 0],
            &[0, 220, 108, 62, 31, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ß' / 0xdf
        'ß' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 41, 18, 0, 0, 0],
            &[0, 124, 236, 195, 230, 157, 0, 0],
            &[10, 251, 35, 0, 16, 254, 21, 0],
            &[25, 244, 0, 0, 112, 199, 0, 0],
            &[25, 244, 0, 89, 200, 14, 0, 0],
            &[25, 244, 0, 121, 182, 9, 0, 0],
            &[25, 244, 0, 7, 161, 213, 34, 0],
            &[25, 244, 0, 0, 0, 119, 187, 0],
            &[25, 244, 4, 14, 0, 71, 217, 0],
            &[25, 244, 13, 223, 204, 227, 96, 0],
            &[0, 0, 0, 2, 21, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'à' / 0xe0
        'à' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 19, 0, 0, 0, 0],
            &[0, 0, 115, 191, 0, 0, 0, 0],
            &[0, 0, 0, 114, 60, 0, 0, 0],
            &[0, 22, 101, 132, 106, 15, 0, 0],
            &[0, 104, 129, 91, 151, 203, 0, 0],
            &[0, 0, 0, 0, 0, 234, 20, 0],
            &[0, 59, 178, 207, 197, 247, 30, 0],
            &[12, 240, 67, 1, 0, 226, 30, 0],
            &[26, 238, 6, 0, 73, 254, 30, 0],
            &[0, 158, 234, 211, 154, 212, 30, 0],
            &[0, 0, 14, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'á' / 0xe1
        'á' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 27, 0, 0],
            &[0, 0, 0, 33, 230, 43, 0, 0],
            &[0, 0, 0, 121, 52, 0, 0, 0],
            &[0, 22, 101, 132, 106, 15, 0, 0],
            &[0, 104, 129, 91, 151, 203, 0, 0],
            &[0, 0, 0, 0, 0, 234, 20, 0],
            &[0, 59, 178, 207, 197, 247, 30, 0],
            &[12, 240, 67, 1, 0, 226, 30, 0],
            &[26, 238, 6, 0, 73, 254, 30, 0],
            &[0, 158, 234, 211, 154, 212, 30, 0],
            &[0, 0, 14, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'â' / 0xe2
        'â' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 12, 0, 0, 0],
            &[0, 0, 104, 207, 176, 2, 0, 0],
            &[0, 31, 137, 4, 94, 84, 0, 0],
            &[0, 22, 101, 132, 106, 15, 0, 0],
            &[0, 104, 129, 91, 151, 203, 0, 0],
            &[0, 0, 0, 0, 0, 234, 20, 0],
            &[0, 59, 178, 207, 197, 247, 30, 0],
            &[12, 240, 67, 1, 0, 226, 30, 0],
            &[26, 238, 6, 0, 73, 254, 30, 0],
            &[0, 158, 234, 211, 154, 212, 30, 0],
            &[0, 0, 14, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ã' / 0xe3
        'ã' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 21, 181, 149, 53, 137, 0, 0],
            &[0, 78, 64, 82, 169, 54, 0, 0],
            &[0, 22, 101, 132, 106, 15, 0, 0],
            &[0, 104, 129, 91, 151, 203, 0, 0],
            &[0, 0, 0, 0, 0, 234, 20, 0],
            &[0, 59, 178, 207, 197, 247, 30, 0],
            &[12, 240, 67, 1, 0, 226, 30, 0],
            &[26, 238, 6, 0, 73, 254, 30, 0],
            &[0, 158, 234, 211, 154, 212, 30, 0],
            &[0, 0, 14, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ä' / 0xe4
        'ä' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 52, 170, 0, 104, 118, 0, 0],
            &[0, 23, 93, 0, 54, 63, 0, 0],
            &[0, 22, 101, 132, 106, 15, 0, 0],
            &[0, 104, 129, 91, 151, 203, 0, 0],
            &[0, 0, 0, 0, 0, 234, 20, 0],
            &[0, 59, 178, 207, 197, 247, 30, 0],
            &[12, 240, 67, 1, 0, 226, 30, 0],
            &[26, 238, 6, 0, 73, 254, 30, 0],
            &[0, 158, 234, 211, 154, 212, 30, 0],
            &[0, 0, 14, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'å' / 0xe5
        'å' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 62, 170, 116, 0, 0, 0],
            &[0, 0, 158, 29, 189, 0, 0, 0],
            &[0, 0, 50, 167, 99, 0, 0, 0],
            &[0, 22, 101, 132, 106, 15, 0, 0],
            &[0, 104, 129, 91, 151, 203, 0, 0],
            &[0, 0, 0, 0, 0, 234, 20, 0],
            &[0, 59, 178, 207, 197, 247, 30, 0],
            &[12, 240, 67, 1, 0, 226, 30, 0],
            &[26, 238, 6, 0, 73, 254, 30, 0],
            &[0, 158, 234, 211, 154, 212, 30, 0],
            &[0, 0, 14, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'æ' / 0xe6
        'æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[7, 98, 120, 24, 80, 121, 14, 0],
            &[31, 135, 129, 224, 174, 152, 157, 0],
            &[0, 0, 0, 237, 37, 21, 228, 0],
            &[24, 171, 202, 250, 217, 216, 241, 0],
            &[164, 117, 0, 243, 24, 0, 0, 0],
            &[183, 73, 41, 251, 94, 0, 15, 0],
            &[92, 233, 215, 71, 217, 207, 172, 0],
            &[0, 15, 6, 0, 3, 21, 1, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ç' / 0xe7
        'ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 122, 130, 80, 1, 0],
            &[0, 79, 235, 127, 108, 145, 0, 0],
            &[0, 202, 92, 0, 0, 0, 0, 0],
            &[0, 242, 38, 0, 0, 0, 0, 0],
            &[0, 235, 49, 0, 0, 0, 0, 0],
            &[0, 170, 151, 1, 0, 13, 0, 0],
            &[0, 27, 195, 236, 221, 221, 0, 0],
            &[0, 0, 0, 164, 98, 0, 0, 0],
            &[0, 0, 44, 172, 150, 1, 0, 0],
        ],
        // letter: 'è' / 0xe8
        'è' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 44, 19, 0, 0, 0, 0],
            &[0, 0, 118, 188, 0, 0, 0, 0],
            &[0, 0, 0, 116, 58, 0, 0, 0],
            &[0, 0, 70, 131, 111, 17, 0, 0],
            &[0, 132, 203, 97, 144, 215, 10, 0],
            &[20, 245, 20, 0, 0, 191, 87, 0],
            &[62, 250, 221, 221, 221, 240, 118, 0],
            &[52, 226, 0, 0, 0, 0, 0, 0],
            &[3, 224, 83, 0, 0, 17, 12, 0],
            &[0, 46, 205, 224, 210, 218, 34, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'é' / 0xe9
        'é' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 27, 0, 0],
            &[0, 0, 0, 35, 230, 41, 0, 0],
            &[0, 0, 0, 123, 50, 0, 0, 0],
            &[0, 0, 70, 131, 111, 17, 0, 0],
            &[0, 132, 203, 97, 144, 215, 10, 0],
            &[20, 245, 20, 0, 0, 191, 87, 0],
            &[62, 250, 221, 221, 221, 240, 118, 0],
            &[52, 226, 0, 0, 0, 0, 0, 0],
            &[3, 224, 83, 0, 0, 17, 12, 0],
            &[0, 46, 205, 224, 210, 218, 34, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ê' / 0xea
        'ê' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 11, 0, 0, 0],
            &[0, 0, 107, 207, 174, 1, 0, 0],
            &[0, 32, 136, 4, 96, 82, 0, 0],
            &[0, 0, 70, 131, 111, 17, 0, 0],
            &[0, 132, 203, 97, 144, 215, 10, 0],
            &[20, 245, 20, 0, 0, 191, 87, 0],
            &[62, 250, 221, 221, 221, 240, 118, 0],
            &[52, 226, 0, 0, 0, 0, 0, 0],
            &[3, 224, 83, 0, 0, 17, 12, 0],
            &[0, 46, 205, 224, 210, 218, 34, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ë' / 0xeb
        'ë' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 55, 168, 0, 107, 116, 0, 0],
            &[0, 25, 92, 0, 56, 61, 0, 0],
            &[0, 0, 70, 131, 111, 17, 0, 0],
            &[0, 132, 203, 97, 144, 215, 10, 0],
            &[20, 245, 20, 0, 0, 191, 87, 0],
            &[62, 250, 221, 221, 221, 240, 118, 0],
            &[52, 226, 0, 0, 0, 0, 0, 0],
            &[3, 224, 83, 0, 0, 17, 12, 0],
            &[0, 46, 205, 224, 210, 218, 34, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ì' / 0xec
        'ì' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 47, 16, 0, 0, 0, 0],
            &[0, 0, 130, 176, 0, 0, 0, 0],
            &[0, 0, 0, 124, 49, 0, 0, 0],
            &[0, 93, 110, 110, 36, 0, 0, 0],
            &[0, 74, 105, 220, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[7, 195, 215, 248, 234, 205, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'í' / 0xed
        'í' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 25, 0, 0],
            &[0, 0, 0, 44, 228, 34, 0, 0],
            &[0, 0, 0, 130, 43, 0, 0, 0],
            &[0, 93, 110, 110, 36, 0, 0, 0],
            &[0, 74, 105, 220, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[7, 195, 215, 248, 234, 205, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'î' / 0xee
        'î' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 48, 9, 0, 0, 0],
            &[0, 0, 119, 206, 163, 0, 0, 0],
            &[0, 39, 131, 2, 105, 73, 0, 0],
            &[0, 93, 110, 110, 36, 0, 0, 0],
            &[0, 74, 105, 220, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[7, 195, 215, 248, 234, 205, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ï' / 0xef
        'ï' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 64, 158, 0, 117, 105, 0, 0],
            &[0, 30, 87, 0, 62, 54, 0, 0],
            &[0, 93, 110, 110, 36, 0, 0, 0],
            &[0, 74, 105, 220, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[7, 195, 215, 248, 234, 205, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ð' / 0xf0
        'ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 0, 0, 4, 0, 0],
            &[0, 0, 148, 181, 156, 160, 0, 0],
            &[0, 2, 103, 255, 255, 13, 0, 0],
            &[0, 0, 100, 11, 114, 156, 0, 0],
            &[0, 19, 130, 164, 123, 219, 31, 0],
            &[4, 212, 139, 51, 101, 250, 103, 0],
            &[63, 220, 0, 0, 0, 162, 132, 0],
            &[77, 201, 0, 0, 0, 155, 120, 0],
            &[22, 239, 44, 0, 12, 219, 52, 0],
            &[0, 79, 221, 197, 223, 115, 0, 0],
            &[0, 0, 2, 23, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ñ' / 0xf1
        'ñ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 173, 159, 54, 143, 0, 0],
            &[0, 62, 78, 71, 168, 68, 0, 0],
            &[4, 95, 24, 117, 126, 39, 0, 0],
            &[10, 247, 181, 99, 129, 238, 18, 0],
            &[10, 255, 59, 0, 0, 213, 67, 0],
            &[10, 255, 16, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ò' / 0xf2
        'ò' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 47, 16, 0, 0, 0, 0],
            &[0, 0, 130, 176, 0, 0, 0, 0],
            &[0, 0, 0, 124, 49, 0, 0, 0],
            &[0, 2, 87, 133, 100, 10, 0, 0],
            &[0, 166, 184, 94, 153, 204, 7, 0],
            &[39, 238, 9, 0, 0, 198, 88, 0],
            &[79, 195, 0, 0, 0, 147, 130, 0],
            &[69, 206, 0, 0, 0, 160, 121, 0],
            &[12, 237, 53, 0, 23, 234, 50, 0],
            &[0, 71, 221, 202, 226, 104, 0, 0],
            &[0, 0, 0, 18, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ó' / 0xf3
        'ó' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 25, 0, 0],
            &[0, 0, 0, 44, 228, 34, 0, 0],
            &[0, 0, 0, 130, 43, 0, 0, 0],
            &[0, 2, 87, 133, 100, 10, 0, 0],
            &[0, 166, 184, 94, 153, 204, 7, 0],
            &[39, 238, 9, 0, 0, 198, 88, 0],
            &[79, 195, 0, 0, 0, 147, 130, 0],
            &[69, 206, 0, 0, 0, 160, 121, 0],
            &[12, 237, 53, 0, 23, 234, 50, 0],
            &[0, 71, 221, 202, 226, 104, 0, 0],
            &[0, 0, 0, 18, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ô' / 0xf4
        'ô' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 48, 9, 0, 0, 0],
            &[0, 0, 119, 206, 163, 0, 0, 0],
            &[0, 39, 131, 2, 105, 73, 0, 0],
            &[0, 2, 87, 133, 100, 10, 0, 0],
            &[0, 166, 184, 94, 153, 204, 7, 0],
            &[39, 238, 9, 0, 0, 198, 88, 0],
            &[79, 195, 0, 0, 0, 147, 130, 0],
            &[69, 206, 0, 0, 0, 160, 121, 0],
            &[12, 237, 53, 0, 23, 234, 50, 0],
            &[0, 71, 221, 202, 226, 104, 0, 0],
            &[0, 0, 0, 18, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'õ' / 0xf5
        'õ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 185, 142, 54, 131, 0, 0],
            &[0, 89, 55, 90, 169, 45, 0, 0],
            &[0, 2, 87, 133, 100, 10, 0, 0],
            &[0, 166, 184, 94, 153, 204, 7, 0],
            &[39, 238, 9, 0, 0, 198, 88, 0],
            &[79, 195, 0, 0, 0, 147, 130, 0],
            &[69, 206, 0, 0, 0, 160, 121, 0],
            &[12, 237, 53, 0, 23, 234, 50, 0],
            &[0, 71, 221, 202, 226, 104, 0, 0],
            &[0, 0, 0, 18, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ö' / 0xf6
        'ö' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 64, 158, 0, 117, 105, 0, 0],
            &[0, 30, 87, 0, 62, 54, 0, 0],
            &[0, 2, 87, 133, 100, 10, 0, 0],
            &[0, 166, 184, 94, 153, 204, 7, 0],
            &[39, 238, 9, 0, 0, 198, 88, 0],
            &[79, 195, 0, 0, 0, 147, 130, 0],
            &[69, 206, 0, 0, 0, 160, 121, 0],
            &[12, 237, 53, 0, 23, 234, 50, 0],
            &[0, 71, 221, 202, 226, 104, 0, 0],
            &[0, 0, 0, 18, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '÷' / 0xf7
        '÷' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 228, 36, 0, 0, 0],
            &[0, 0, 0, 88, 7, 0, 0, 0],
            &[6, 167, 167, 167, 167, 167, 40, 0],
            &[2, 50, 50, 50, 50, 50, 12, 0],
            &[0, 0, 4, 205, 32, 0, 0, 0],
            &[0, 0, 0, 111, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ø' / 0xf8
        'ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 87, 133, 100, 122, 36, 0],
            &[0, 166, 184, 94, 216, 255, 11, 0],
            &[39, 238, 9, 16, 212, 232, 88, 0],
            &[79, 195, 0, 169, 94, 147, 130, 0],
            &[69, 206, 96, 167, 0, 160, 121, 0],
            &[12, 255, 255, 15, 23, 234, 50, 0],
            &[3, 255, 255, 202, 226, 104, 0, 0],
            &[0, 44, 0, 18, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ù' / 0xf9
        'ù' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 47, 16, 0, 0, 0, 0],
            &[0, 0, 130, 176, 0, 0, 0, 0],
            &[0, 0, 0, 124, 49, 0, 0, 0],
            &[11, 108, 0, 0, 0, 92, 26, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 215, 61, 0],
            &[25, 251, 0, 0, 0, 235, 61, 0],
            &[5, 251, 41, 0, 52, 255, 61, 0],
            &[0, 124, 237, 201, 178, 193, 61, 0],
            &[0, 0, 9, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ú' / 0xfa
        'ú' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 25, 0, 0],
            &[0, 0, 0, 44, 228, 34, 0, 0],
            &[0, 0, 0, 130, 43, 0, 0, 0],
            &[11, 108, 0, 0, 0, 92, 26, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 215, 61, 0],
            &[25, 251, 0, 0, 0, 235, 61, 0],
            &[5, 251, 41, 0, 52, 255, 61, 0],
            &[0, 124, 237, 201, 178, 193, 61, 0],
            &[0, 0, 9, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'û' / 0xfb
        'û' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 48, 9, 0, 0, 0],
            &[0, 0, 119, 206, 163, 0, 0, 0],
            &[0, 39, 131, 2, 105, 73, 0, 0],
            &[11, 108, 0, 0, 0, 92, 26, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 215, 61, 0],
            &[25, 251, 0, 0, 0, 235, 61, 0],
            &[5, 251, 41, 0, 52, 255, 61, 0],
            &[0, 124, 237, 201, 178, 193, 61, 0],
            &[0, 0, 9, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ü' / 0xfc
        'ü' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 64, 158, 0, 117, 105, 0, 0],
            &[0, 30, 87, 0, 62, 54, 0, 0],
            &[11, 108, 0, 0, 0, 92, 26, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 215, 61, 0],
            &[25, 251, 0, 0, 0, 235, 61, 0],
            &[5, 251, 41, 0, 52, 255, 61, 0],
            &[0, 124, 237, 201, 178, 193, 61, 0],
            &[0, 0, 9, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ý' / 0xfd
        'ý' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 25, 0, 0],
            &[0, 0, 0, 44, 228, 34, 0, 0],
            &[0, 0, 0, 130, 43, 0, 0, 0],
            &[40, 75, 0, 0, 0, 58, 62, 0],
            &[24, 233, 9, 0, 0, 196, 77, 0],
            &[0, 167, 97, 0, 27, 231, 4, 0],
            &[0, 61, 201, 0, 111, 143, 0, 0],
            &[0, 0, 210, 47, 196, 48, 0, 0],
            &[0, 0, 104, 161, 207, 0, 0, 0],
            &[0, 0, 12, 240, 114, 0, 0, 0],
            &[0, 0, 5, 229, 23, 0, 0, 0],
            &[49, 220, 191, 20, 0, 0, 0, 0],
        ],
        // letter: 'þ' / 0xfe
        'þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[1, 30, 0, 0, 0, 0, 0, 0],
            &[10, 255, 7, 0, 0, 0, 0, 0],
            &[10, 255, 7, 0, 0, 0, 0, 0],
            &[10, 255, 37, 124, 113, 20, 0, 0],
            &[10, 255, 181, 97, 141, 222, 9, 0],
            &[10, 255, 49, 0, 0, 191, 96, 0],
            &[10, 255, 11, 0, 0, 151, 133, 0],
            &[10, 255, 25, 0, 0, 163, 122, 0],
            &[10, 255, 109, 0, 17, 231, 59, 0],
            &[10, 253, 150, 207, 233, 139, 0, 0],
            &[10, 255, 7, 13, 12, 0, 0, 0],
            &[9, 225, 6, 0, 0, 0, 0, 0],
        ],
        // letter: 'ÿ' / 0xff
        'ÿ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 64, 158, 0, 117, 105, 0, 0],
            &[0, 30, 87, 0, 62, 54, 0, 0],
            &[40, 75, 0, 0, 0, 58, 62, 0],
            &[24, 233, 9, 0, 0, 196, 77, 0],
            &[0, 167, 97, 0, 27, 231, 4, 0],
            &[0, 61, 201, 0, 111, 143, 0, 0],
            &[0, 0, 210, 47, 196, 48, 0, 0],
            &[0, 0, 104, 161, 207, 0, 0, 0],
            &[0, 0, 12, 240, 114, 0, 0, 0],
            &[0, 0, 5, 229, 23, 0, 0, 0],
            &[49, 220, 191, 20, 0, 0, 0, 0],
        ],
        // letter: 'Ā' / 0x100
        'Ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 19, 103, 103, 103, 38, 0, 0],
            &[0, 21, 114, 114, 114, 43, 0, 0],
            &[0, 0, 11, 154, 40, 0, 0, 0],
            &[0, 0, 85, 242, 133, 0, 0, 0],
            &[0, 0, 168, 127, 216, 0, 0, 0],
            &[0, 7, 237, 15, 218, 44, 0, 0],
            &[0, 77, 186, 0, 138, 127, 0, 0],
            &[0, 160, 242, 150, 207, 216, 0, 0],
            &[5, 245, 123, 95, 97, 250, 37, 0],
            &[70, 204, 0, 0, 0, 156, 120, 0],
            &[153, 124, 0, 0, 0, 76, 204, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ā' / 0x101
        'ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 8, 68, 68, 68, 29, 0, 0],
            &[0, 19, 149, 149, 149, 65, 0, 0],
            &[0, 22, 101, 132, 106, 15, 0, 0],
            &[0, 104, 129, 91, 151, 203, 0, 0],
            &[0, 0, 0, 0, 0, 234, 20, 0],
            &[0, 59, 178, 207, 197, 247, 30, 0],
            &[12, 240, 67, 1, 0, 226, 30, 0],
            &[26, 238, 6, 0, 73, 254, 30, 0],
            &[0, 158, 234, 211, 154, 212, 30, 0],
            &[0, 0, 14, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ă' / 0x102
        'Ă' => &[
            &[0, 1, 1, 0, 1, 1, 0, 0],
            &[0, 35, 187, 53, 152, 80, 0, 0],
            &[0, 0, 72, 141, 93, 0, 0, 0],
            &[0, 0, 11, 154, 40, 0, 0, 0],
            &[0, 0, 85, 242, 133, 0, 0, 0],
            &[0, 0, 168, 127, 216, 0, 0, 0],
            &[0, 7, 237, 15, 218, 44, 0, 0],
            &[0, 77, 186, 0, 138, 127, 0, 0],
            &[0, 160, 242, 150, 207, 216, 0, 0],
            &[5, 245, 123, 95, 97, 250, 37, 0],
            &[70, 204, 0, 0, 0, 156, 120, 0],
            &[153, 124, 0, 0, 0, 76, 204, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ă' / 0x103
        'ă' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 24, 166, 20, 107, 92, 0, 0],
            &[0, 0, 94, 176, 134, 7, 0, 0],
            &[0, 22, 101, 132, 106, 15, 0, 0],
            &[0, 104, 129, 91, 151, 203, 0, 0],
            &[0, 0, 0, 0, 0, 234, 20, 0],
            &[0, 59, 178, 207, 197, 247, 30, 0],
            &[12, 240, 67, 1, 0, 226, 30, 0],
            &[26, 238, 6, 0, 73, 254, 30, 0],
            &[0, 158, 234, 211, 154, 212, 30, 0],
            &[0, 0, 14, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ą' / 0x104
        'Ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 154, 40, 0, 0, 0],
            &[0, 0, 85, 242, 133, 0, 0, 0],
            &[0, 0, 168, 127, 216, 0, 0, 0],
            &[0, 7, 237, 15, 218, 44, 0, 0],
            &[0, 77, 186, 0, 138, 127, 0, 0],
            &[0, 160, 242, 150, 207, 216, 0, 0],
            &[5, 245, 123, 95, 97, 250, 37, 0],
            &[70, 204, 0, 0, 0, 156, 120, 0],
            &[153, 124, 0, 0, 0, 76, 222, 0],
            &[0, 0, 0, 0, 0, 73, 137, 0],
            &[0, 0, 0, 0, 0, 55, 156, 37],
        ],
        // letter: 'ą' / 0x105
        'ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 101, 132, 106, 15, 0, 0],
            &[0, 104, 129, 91, 151, 203, 0, 0],
            &[0, 0, 0, 0, 0, 234, 20, 0],
            &[0, 59, 178, 207, 197, 247, 30, 0],
            &[12, 240, 67, 1, 0, 226, 30, 0],
            &[26, 238, 6, 0, 73, 254, 30, 0],
            &[0, 158, 234, 211, 154, 230, 30, 0],
            &[0, 0, 14, 12, 55, 155, 0, 0],
            &[0, 0, 0, 0, 42, 155, 50, 0],
        ],
        // letter: 'Ć' / 0x106
        'Ć' => &[
            &[0, 0, 0, 0, 11, 83, 14, 0],
            &[0, 0, 0, 0, 136, 161, 0, 0],
            &[0, 0, 0, 12, 121, 3, 0, 0],
            &[0, 0, 43, 143, 169, 139, 43, 0],
            &[0, 62, 241, 128, 79, 115, 60, 0],
            &[0, 207, 104, 0, 0, 0, 0, 0],
            &[23, 255, 12, 0, 0, 0, 0, 0],
            &[48, 237, 0, 0, 0, 0, 0, 0],
            &[37, 243, 0, 0, 0, 0, 0, 0],
            &[4, 242, 44, 0, 0, 0, 0, 0],
            &[0, 144, 189, 18, 0, 13, 12, 0],
            &[0, 7, 155, 245, 231, 240, 46, 0],
            &[0, 0, 0, 5, 23, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ć' / 0x107
        'ć' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 41, 0, 0],
            &[0, 0, 0, 3, 205, 98, 0, 0],
            &[0, 0, 0, 72, 101, 0, 0, 0],
            &[0, 0, 44, 122, 130, 80, 1, 0],
            &[0, 79, 235, 127, 108, 145, 0, 0],
            &[0, 202, 92, 0, 0, 0, 0, 0],
            &[0, 242, 38, 0, 0, 0, 0, 0],
            &[0, 235, 49, 0, 0, 0, 0, 0],
            &[0, 170, 151, 1, 0, 13, 0, 0],
            &[0, 27, 195, 230, 216, 221, 0, 0],
            &[0, 0, 0, 16, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĉ' / 0x108
        'Ĉ' => &[
            &[0, 0, 0, 30, 78, 0, 0, 0],
            &[0, 0, 12, 198, 181, 117, 0, 0],
            &[0, 0, 84, 54, 2, 114, 26, 0],
            &[0, 0, 43, 143, 169, 139, 43, 0],
            &[0, 62, 241, 128, 79, 115, 60, 0],
            &[0, 207, 104, 0, 0, 0, 0, 0],
            &[23, 255, 12, 0, 0, 0, 0, 0],
            &[48, 237, 0, 0, 0, 0, 0, 0],
            &[37, 243, 0, 0, 0, 0, 0, 0],
            &[4, 242, 44, 0, 0, 0, 0, 0],
            &[0, 144, 189, 18, 0, 13, 12, 0],
            &[0, 7, 155, 245, 231, 240, 46, 0],
            &[0, 0, 0, 5, 23, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĉ' / 0x109
        'ĉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 33, 26, 0, 0, 0],
            &[0, 0, 44, 211, 210, 23, 0, 0],
            &[0, 4, 142, 26, 47, 131, 0, 0],
            &[0, 0, 44, 122, 130, 80, 1, 0],
            &[0, 79, 235, 127, 108, 145, 0, 0],
            &[0, 202, 92, 0, 0, 0, 0, 0],
            &[0, 242, 38, 0, 0, 0, 0, 0],
            &[0, 235, 49, 0, 0, 0, 0, 0],
            &[0, 170, 151, 1, 0, 13, 0, 0],
            &[0, 27, 195, 230, 216, 221, 0, 0],
            &[0, 0, 0, 16, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ċ' / 0x10a
        'Ċ' => &[
            &[0, 0, 0, 7, 37, 0, 0, 0],
            &[0, 0, 0, 92, 242, 1, 0, 0],
            &[0, 0, 0, 7, 40, 0, 0, 0],
            &[0, 0, 43, 143, 169, 139, 43, 0],
            &[0, 62, 241, 128, 79, 115, 60, 0],
            &[0, 207, 104, 0, 0, 0, 0, 0],
            &[23, 255, 12, 0, 0, 0, 0, 0],
            &[48, 237, 0, 0, 0, 0, 0, 0],
            &[37, 243, 0, 0, 0, 0, 0, 0],
            &[4, 242, 44, 0, 0, 0, 0, 0],
            &[0, 144, 189, 18, 0, 13, 12, 0],
            &[0, 7, 155, 245, 231, 240, 46, 0],
            &[0, 0, 0, 5, 23, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ċ' / 0x10b
        'ċ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 3, 0, 0, 0],
            &[0, 0, 0, 184, 145, 0, 0, 0],
            &[0, 0, 0, 51, 36, 0, 0, 0],
            &[0, 0, 44, 122, 130, 80, 1, 0],
            &[0, 79, 235, 127, 108, 145, 0, 0],
            &[0, 202, 92, 0, 0, 0, 0, 0],
            &[0, 242, 38, 0, 0, 0, 0, 0],
            &[0, 235, 49, 0, 0, 0, 0, 0],
            &[0, 170, 151, 1, 0, 13, 0, 0],
            &[0, 27, 195, 230, 216, 221, 0, 0],
            &[0, 0, 0, 16, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Č' / 0x10c
        'Č' => &[
            &[0, 0, 56, 18, 0, 52, 24, 0],
            &[0, 0, 43, 204, 128, 172, 3, 0],
            &[0, 0, 0, 68, 146, 11, 0, 0],
            &[0, 0, 43, 143, 169, 139, 43, 0],
            &[0, 62, 241, 128, 79, 115, 60, 0],
            &[0, 207, 104, 0, 0, 0, 0, 0],
            &[23, 255, 12, 0, 0, 0, 0, 0],
            &[48, 237, 0, 0, 0, 0, 0, 0],
            &[37, 243, 0, 0, 0, 0, 0, 0],
            &[4, 242, 44, 0, 0, 0, 0, 0],
            &[0, 144, 189, 18, 0, 13, 12, 0],
            &[0, 7, 155, 245, 231, 240, 46, 0],
            &[0, 0, 0, 5, 23, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'č' / 0x10d
        'č' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 4, 37, 0, 0, 42, 0, 0],
            &[0, 0, 151, 123, 158, 116, 0, 0],
            &[0, 0, 3, 159, 134, 0, 0, 0],
            &[0, 0, 44, 122, 130, 80, 1, 0],
            &[0, 79, 235, 127, 108, 145, 0, 0],
            &[0, 202, 92, 0, 0, 0, 0, 0],
            &[0, 242, 38, 0, 0, 0, 0, 0],
            &[0, 235, 49, 0, 0, 0, 0, 0],
            &[0, 170, 151, 1, 0, 13, 0, 0],
            &[0, 27, 195, 230, 216, 221, 0, 0],
            &[0, 0, 0, 16, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ď' / 0x10e
        'Ď' => &[
            &[0, 55, 19, 0, 51, 25, 0, 0],
            &[0, 40, 205, 126, 175, 4, 0, 0],
            &[0, 0, 66, 147, 12, 0, 0, 0],
            &[23, 145, 145, 128, 61, 0, 0, 0],
            &[40, 242, 94, 119, 220, 144, 0, 0],
            &[40, 235, 0, 0, 18, 232, 61, 0],
            &[40, 235, 0, 0, 0, 148, 138, 0],
            &[40, 235, 0, 0, 0, 118, 162, 0],
            &[40, 235, 0, 0, 0, 131, 148, 0],
            &[40, 235, 0, 0, 0, 199, 93, 0],
            &[40, 235, 0, 15, 138, 214, 8, 0],
            &[40, 254, 239, 231, 159, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ď' / 0x10f
        'ď' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 23, 10, 21],
            &[0, 0, 0, 0, 78, 194, 114, 130],
            &[0, 0, 0, 0, 78, 194, 129, 20],
            &[0, 26, 122, 118, 94, 194, 0, 0],
            &[10, 224, 158, 104, 221, 194, 0, 0],
            &[80, 215, 1, 0, 132, 194, 0, 0],
            &[111, 170, 0, 0, 88, 194, 0, 0],
            &[96, 179, 0, 0, 91, 194, 0, 0],
            &[42, 236, 17, 0, 171, 194, 0, 0],
            &[0, 133, 230, 210, 141, 194, 0, 0],
            &[0, 0, 10, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Đ' / 0x110
        'Đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 145, 145, 131, 74, 1, 0, 0],
            &[10, 255, 100, 113, 207, 173, 1, 0],
            &[10, 255, 10, 0, 7, 212, 91, 0],
            &[10, 255, 10, 0, 0, 117, 169, 0],
            &[190, 255, 190, 90, 0, 88, 193, 0],
            &[69, 255, 69, 29, 0, 100, 179, 0],
            &[10, 255, 10, 0, 0, 168, 123, 0],
            &[10, 255, 10, 10, 114, 231, 21, 0],
            &[10, 255, 240, 234, 172, 35, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'đ' / 0x111
        'đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 7, 0],
            &[0, 0, 0, 9, 10, 222, 71, 2],
            &[0, 0, 0, 183, 192, 255, 253, 38],
            &[0, 6, 87, 116, 50, 210, 61, 0],
            &[0, 180, 195, 109, 186, 235, 61, 0],
            &[44, 244, 14, 0, 16, 251, 61, 0],
            &[80, 206, 0, 0, 0, 222, 61, 0],
            &[72, 212, 0, 0, 0, 224, 61, 0],
            &[17, 240, 46, 0, 56, 255, 61, 0],
            &[0, 89, 232, 206, 186, 196, 61, 0],
            &[0, 0, 2, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ē' / 0x112
        'Ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 103, 103, 103, 46, 0, 0],
            &[0, 13, 114, 114, 114, 51, 0, 0],
            &[0, 125, 145, 145, 145, 145, 26, 0],
            &[0, 220, 134, 100, 100, 100, 18, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 244, 244, 232, 0, 0],
            &[0, 220, 57, 1, 1, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 245, 245, 245, 44, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ē' / 0x113
        'ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 68, 68, 68, 29, 0, 0],
            &[0, 20, 149, 149, 149, 63, 0, 0],
            &[0, 0, 70, 131, 111, 17, 0, 0],
            &[0, 132, 203, 97, 144, 215, 10, 0],
            &[20, 245, 20, 0, 0, 191, 87, 0],
            &[62, 250, 221, 221, 221, 240, 118, 0],
            &[52, 226, 0, 0, 0, 0, 0, 0],
            &[3, 224, 83, 0, 0, 17, 12, 0],
            &[0, 46, 205, 224, 210, 218, 34, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĕ' / 0x114
        'Ĕ' => &[
            &[0, 0, 2, 0, 0, 2, 0, 0],
            &[0, 21, 195, 55, 139, 98, 0, 0],
            &[0, 0, 62, 141, 101, 2, 0, 0],
            &[0, 125, 145, 145, 145, 145, 26, 0],
            &[0, 220, 134, 100, 100, 100, 18, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 244, 244, 232, 0, 0],
            &[0, 220, 57, 1, 1, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 245, 245, 245, 44, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĕ' / 0x115
        'ĕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 164, 19, 109, 89, 0, 0],
            &[0, 0, 96, 176, 133, 6, 0, 0],
            &[0, 0, 70, 131, 111, 17, 0, 0],
            &[0, 132, 203, 97, 144, 215, 10, 0],
            &[20, 245, 20, 0, 0, 191, 87, 0],
            &[62, 250, 221, 221, 221, 240, 118, 0],
            &[52, 226, 0, 0, 0, 0, 0, 0],
            &[3, 224, 83, 0, 0, 17, 12, 0],
            &[0, 46, 205, 224, 210, 218, 34, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ė' / 0x116
        'Ė' => &[
            &[0, 0, 0, 39, 5, 0, 0, 0],
            &[0, 0, 6, 249, 81, 0, 0, 0],
            &[0, 0, 0, 41, 5, 0, 0, 0],
            &[0, 125, 145, 145, 145, 145, 26, 0],
            &[0, 220, 134, 100, 100, 100, 18, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 244, 244, 232, 0, 0],
            &[0, 220, 57, 1, 1, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 245, 245, 245, 44, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ė' / 0x117
        'ė' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 0, 0, 0, 0],
            &[0, 0, 9, 248, 72, 0, 0, 0],
            &[0, 0, 0, 75, 12, 0, 0, 0],
            &[0, 0, 70, 131, 111, 17, 0, 0],
            &[0, 132, 203, 97, 144, 215, 10, 0],
            &[20, 245, 20, 0, 0, 191, 87, 0],
            &[62, 250, 221, 221, 221, 240, 118, 0],
            &[52, 226, 0, 0, 0, 0, 0, 0],
            &[3, 224, 83, 0, 0, 17, 12, 0],
            &[0, 46, 205, 224, 210, 218, 34, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ę' / 0x118
        'Ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 125, 145, 145, 145, 145, 26, 0],
            &[0, 220, 134, 100, 100, 100, 18, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 244, 244, 232, 0, 0],
            &[0, 220, 57, 1, 1, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 245, 252, 255, 44, 0],
            &[0, 0, 0, 0, 169, 41, 0, 0],
            &[0, 0, 0, 0, 120, 128, 0, 0],
        ],
        // letter: 'ę' / 0x119
        'ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 131, 111, 17, 0, 0],
            &[0, 132, 203, 97, 144, 215, 10, 0],
            &[20, 245, 20, 0, 0, 191, 87, 0],
            &[62, 250, 221, 221, 221, 240, 118, 0],
            &[52, 226, 0, 0, 0, 0, 0, 0],
            &[3, 224, 83, 0, 0, 17, 12, 0],
            &[0, 46, 205, 224, 210, 255, 37, 0],
            &[0, 0, 0, 16, 140, 85, 0, 0],
            &[0, 0, 0, 0, 77, 155, 11, 0],
        ],
        // letter: 'Ě' / 0x11a
        'Ě' => &[
            &[0, 26, 48, 0, 21, 55, 0, 0],
            &[0, 5, 180, 123, 205, 38, 0, 0],
            &[0, 0, 16, 148, 61, 0, 0, 0],
            &[0, 125, 145, 145, 145, 145, 26, 0],
            &[0, 220, 134, 100, 100, 100, 18, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 244, 244, 232, 0, 0],
            &[0, 220, 57, 1, 1, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 56, 0, 0, 0, 0, 0],
            &[0, 220, 247, 245, 245, 245, 44, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ě' / 0x11b
        'ě' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 22, 0, 9, 33, 0, 0],
            &[0, 16, 194, 87, 197, 54, 0, 0],
            &[0, 0, 32, 184, 80, 0, 0, 0],
            &[0, 0, 70, 131, 111, 17, 0, 0],
            &[0, 132, 203, 97, 144, 215, 10, 0],
            &[20, 245, 20, 0, 0, 191, 87, 0],
            &[62, 250, 221, 221, 221, 240, 118, 0],
            &[52, 226, 0, 0, 0, 0, 0, 0],
            &[3, 224, 83, 0, 0, 17, 12, 0],
            &[0, 46, 205, 224, 210, 218, 34, 0],
            &[0, 0, 0, 16, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => &[
            &[0, 0, 0, 67, 42, 0, 0, 0],
            &[0, 0, 81, 192, 208, 29, 0, 0],
            &[0, 9, 121, 7, 35, 107, 0, 0],
            &[0, 0, 70, 153, 166, 120, 14, 0],
            &[0, 116, 226, 106, 83, 136, 10, 0],
            &[19, 244, 41, 0, 0, 0, 0, 0],
            &[83, 201, 0, 0, 0, 0, 0, 0],
            &[110, 170, 0, 32, 134, 134, 48, 0],
            &[104, 177, 0, 27, 111, 210, 92, 0],
            &[65, 227, 2, 0, 0, 175, 92, 0],
            &[5, 219, 127, 2, 0, 175, 92, 0],
            &[0, 40, 200, 237, 224, 234, 67, 0],
            &[0, 0, 0, 13, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĝ' / 0x11d
        'ĝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 48, 9, 0, 0, 0],
            &[0, 0, 119, 206, 163, 0, 0, 0],
            &[0, 39, 131, 2, 105, 73, 0, 0],
            &[0, 4, 96, 126, 58, 73, 26, 0],
            &[0, 170, 179, 97, 172, 235, 61, 0],
            &[37, 240, 9, 0, 4, 233, 61, 0],
            &[75, 204, 0, 0, 0, 192, 61, 0],
            &[64, 215, 0, 0, 0, 201, 61, 0],
            &[16, 240, 54, 0, 36, 248, 61, 0],
            &[0, 83, 231, 205, 204, 216, 61, 0],
            &[0, 0, 1, 18, 0, 208, 53, 0],
            &[0, 163, 217, 224, 197, 62, 0, 0],
        ],
        // letter: 'Ğ' / 0x11e
        'Ğ' => &[
            &[0, 0, 2, 0, 0, 2, 0, 0],
            &[0, 0, 197, 63, 100, 147, 0, 0],
            &[0, 0, 39, 137, 118, 12, 0, 0],
            &[0, 0, 70, 153, 166, 120, 14, 0],
            &[0, 116, 226, 106, 83, 136, 10, 0],
            &[19, 244, 41, 0, 0, 0, 0, 0],
            &[83, 201, 0, 0, 0, 0, 0, 0],
            &[110, 170, 0, 32, 134, 134, 48, 0],
            &[104, 177, 0, 27, 111, 210, 92, 0],
            &[65, 227, 2, 0, 0, 175, 92, 0],
            &[5, 219, 127, 2, 0, 175, 92, 0],
            &[0, 40, 200, 237, 224, 234, 67, 0],
            &[0, 0, 0, 13, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ğ' / 0x11f
        'ğ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 36, 157, 19, 119, 78, 0, 0],
            &[0, 0, 104, 176, 128, 3, 0, 0],
            &[0, 4, 96, 126, 58, 73, 26, 0],
            &[0, 170, 179, 97, 172, 235, 61, 0],
            &[37, 240, 9, 0, 4, 233, 61, 0],
            &[75, 204, 0, 0, 0, 192, 61, 0],
            &[64, 215, 0, 0, 0, 201, 61, 0],
            &[16, 240, 54, 0, 36, 248, 61, 0],
            &[0, 83, 231, 205, 204, 216, 61, 0],
            &[0, 0, 1, 18, 0, 208, 53, 0],
            &[0, 163, 217, 224, 197, 62, 0, 0],
        ],
        // letter: 'Ġ' / 0x120
        'Ġ' => &[
            &[0, 0, 0, 31, 13, 0, 0, 0],
            &[0, 0, 0, 206, 130, 0, 0, 0],
            &[0, 0, 0, 32, 14, 0, 0, 0],
            &[0, 0, 70, 153, 166, 120, 14, 0],
            &[0, 116, 226, 106, 83, 136, 10, 0],
            &[19, 244, 41, 0, 0, 0, 0, 0],
            &[83, 201, 0, 0, 0, 0, 0, 0],
            &[110, 170, 0, 32, 134, 134, 48, 0],
            &[104, 177, 0, 27, 111, 210, 92, 0],
            &[65, 227, 2, 0, 0, 175, 92, 0],
            &[5, 219, 127, 2, 0, 175, 92, 0],
            &[0, 40, 200, 237, 224, 234, 67, 0],
            &[0, 0, 0, 13, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ġ' / 0x121
        'ġ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 0, 0, 0, 0],
            &[0, 0, 18, 252, 60, 0, 0, 0],
            &[0, 0, 0, 78, 9, 0, 0, 0],
            &[0, 4, 96, 126, 58, 73, 26, 0],
            &[0, 170, 179, 97, 172, 235, 61, 0],
            &[37, 240, 9, 0, 4, 233, 61, 0],
            &[75, 204, 0, 0, 0, 192, 61, 0],
            &[64, 215, 0, 0, 0, 201, 61, 0],
            &[16, 240, 54, 0, 36, 248, 61, 0],
            &[0, 83, 231, 205, 204, 216, 61, 0],
            &[0, 0, 1, 18, 0, 208, 53, 0],
            &[0, 163, 217, 224, 197, 62, 0, 0],
        ],
        // letter: 'Ģ' / 0x122
        'Ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 153, 166, 120, 14, 0],
            &[0, 116, 226, 106, 83, 136, 10, 0],
            &[19, 244, 41, 0, 0, 0, 0, 0],
            &[83, 201, 0, 0, 0, 0, 0, 0],
            &[110, 170, 0, 32, 134, 134, 48, 0],
            &[104, 177, 0, 27, 111, 210, 92, 0],
            &[65, 227, 2, 0, 0, 175, 92, 0],
            &[5, 219, 127, 2, 0, 175, 92, 0],
            &[0, 40, 200, 237, 224, 234, 67, 0],
            &[0, 0, 0, 47, 57, 0, 0, 0],
            &[0, 0, 0, 119, 0, 0, 0, 0],
        ],
        // letter: 'ģ' / 0x123
        'ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 17, 0, 0, 0],
            &[0, 0, 0, 169, 44, 0, 0, 0],
            &[0, 0, 15, 173, 0, 0, 0, 0],
            &[0, 4, 96, 126, 58, 73, 26, 0],
            &[0, 170, 179, 97, 172, 235, 61, 0],
            &[37, 240, 9, 0, 4, 233, 61, 0],
            &[75, 204, 0, 0, 0, 192, 61, 0],
            &[64, 215, 0, 0, 0, 201, 61, 0],
            &[16, 240, 54, 0, 36, 248, 61, 0],
            &[0, 83, 231, 205, 204, 216, 61, 0],
            &[0, 0, 1, 18, 0, 208, 53, 0],
            &[0, 163, 217, 224, 197, 62, 0, 0],
        ],
        // letter: 'Ĥ' / 0x124
        'Ĥ' => &[
            &[0, 0, 6, 83, 20, 0, 0, 0],
            &[0, 0, 147, 174, 184, 4, 0, 0],
            &[0, 39, 98, 0, 73, 69, 0, 0],
            &[23, 133, 0, 0, 0, 104, 52, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 254, 244, 244, 244, 252, 92, 0],
            &[40, 235, 1, 1, 1, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĥ' / 0x125
        'ĥ' => &[
            &[0, 80, 221, 79, 0, 0, 0, 0],
            &[49, 183, 35, 185, 51, 0, 0, 0],
            &[5, 3, 30, 3, 5, 0, 0, 0],
            &[0, 10, 255, 10, 0, 0, 0, 0],
            &[0, 10, 255, 10, 0, 0, 0, 0],
            &[0, 10, 255, 28, 115, 123, 33, 0],
            &[0, 10, 255, 177, 98, 140, 235, 15],
            &[0, 10, 255, 64, 0, 0, 219, 64],
            &[0, 10, 255, 14, 0, 0, 199, 76],
            &[0, 10, 255, 10, 0, 0, 199, 76],
            &[0, 10, 255, 10, 0, 0, 199, 76],
            &[0, 10, 255, 10, 0, 0, 199, 76],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ħ' / 0x126
        'Ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 133, 0, 0, 0, 104, 52, 0],
            &[76, 255, 35, 35, 35, 220, 127, 7],
            &[226, 255, 185, 185, 185, 255, 255, 37],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 254, 244, 244, 244, 252, 92, 0],
            &[40, 235, 1, 1, 1, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ħ' / 0x127
        'ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[1, 30, 1, 0, 0, 0, 0, 0],
            &[20, 255, 20, 10, 1, 0, 0, 0],
            &[202, 255, 202, 192, 29, 0, 0, 0],
            &[10, 255, 28, 115, 123, 33, 0, 0],
            &[10, 255, 177, 98, 140, 235, 15, 0],
            &[10, 255, 64, 0, 0, 219, 64, 0],
            &[10, 255, 14, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĩ' / 0x128
        'Ĩ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 211, 177, 89, 148, 0, 0],
            &[0, 75, 28, 56, 134, 28, 0, 0],
            &[0, 98, 145, 145, 145, 127, 0, 0],
            &[0, 35, 72, 247, 98, 48, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 134, 217, 254, 224, 176, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĩ' / 0x129
        'ĩ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 185, 142, 54, 131, 0, 0],
            &[0, 89, 55, 90, 169, 45, 0, 0],
            &[0, 93, 110, 110, 36, 0, 0, 0],
            &[0, 74, 105, 220, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[7, 195, 215, 248, 234, 205, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ī' / 0x12a
        'Ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 19, 103, 103, 103, 38, 0, 0],
            &[0, 21, 114, 114, 114, 43, 0, 0],
            &[0, 98, 145, 145, 145, 127, 0, 0],
            &[0, 35, 72, 247, 98, 48, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 134, 217, 254, 224, 176, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ī' / 0x12b
        'ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 68, 68, 68, 25, 0, 0],
            &[0, 28, 149, 149, 149, 56, 0, 0],
            &[0, 93, 110, 110, 36, 0, 0, 0],
            &[0, 74, 105, 220, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[7, 195, 215, 248, 234, 205, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => &[
            &[0, 1, 1, 0, 1, 1, 0, 0],
            &[0, 35, 187, 53, 152, 80, 0, 0],
            &[0, 0, 72, 141, 93, 0, 0, 0],
            &[0, 98, 145, 145, 145, 127, 0, 0],
            &[0, 35, 72, 247, 98, 48, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 134, 217, 254, 224, 176, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĭ' / 0x12d
        'ĭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 36, 157, 19, 119, 78, 0, 0],
            &[0, 0, 104, 176, 128, 3, 0, 0],
            &[0, 93, 110, 110, 36, 0, 0, 0],
            &[0, 74, 105, 220, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[7, 195, 215, 248, 234, 205, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Į' / 0x12e
        'Į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 145, 145, 145, 127, 0, 0],
            &[0, 35, 72, 247, 98, 48, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 134, 217, 254, 238, 180, 0, 0],
            &[0, 0, 0, 14, 181, 14, 0, 0],
            &[0, 0, 0, 11, 144, 93, 0, 0],
        ],
        // letter: 'į' / 0x12f
        'į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 0, 0, 0, 0],
            &[0, 0, 3, 242, 84, 0, 0, 0],
            &[0, 0, 0, 72, 16, 0, 0, 0],
            &[0, 93, 110, 110, 36, 0, 0, 0],
            &[0, 74, 105, 220, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[7, 195, 215, 248, 240, 216, 90, 0],
            &[0, 0, 0, 0, 169, 41, 0, 0],
            &[0, 0, 0, 0, 120, 128, 0, 0],
        ],
        // letter: 'İ' / 0x130
        'İ' => &[
            &[0, 0, 0, 41, 3, 0, 0, 0],
            &[0, 0, 18, 255, 63, 0, 0, 0],
            &[0, 0, 0, 44, 3, 0, 0, 0],
            &[0, 98, 145, 145, 145, 127, 0, 0],
            &[0, 35, 72, 247, 98, 48, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 0, 0, 243, 32, 0, 0, 0],
            &[0, 134, 217, 254, 224, 176, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ı' / 0x131
        'ı' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 93, 110, 110, 36, 0, 0, 0],
            &[0, 74, 105, 220, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[0, 0, 0, 191, 84, 0, 0, 0],
            &[7, 195, 215, 248, 234, 205, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĳ' / 0x132
        'Ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 116, 40, 0, 11, 145, 0, 0],
            &[0, 204, 71, 0, 20, 255, 0, 0],
            &[0, 204, 71, 0, 20, 255, 0, 0],
            &[0, 204, 71, 0, 20, 255, 0, 0],
            &[0, 204, 71, 0, 20, 255, 0, 0],
            &[0, 204, 71, 0, 20, 255, 0, 0],
            &[0, 122, 42, 0, 27, 251, 0, 0],
            &[0, 24, 0, 1, 129, 206, 0, 0],
            &[0, 178, 240, 241, 223, 54, 0, 0],
            &[0, 0, 16, 23, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĳ' / 0x133
        'ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 4, 0, 0, 10, 0, 0],
            &[0, 177, 154, 0, 101, 227, 0, 0],
            &[0, 48, 39, 0, 21, 66, 0, 0],
            &[0, 65, 53, 0, 26, 93, 0, 0],
            &[0, 152, 123, 0, 60, 215, 0, 0],
            &[0, 152, 123, 0, 60, 215, 0, 0],
            &[0, 152, 123, 0, 60, 215, 0, 0],
            &[0, 152, 123, 0, 60, 215, 0, 0],
            &[0, 152, 123, 0, 60, 215, 0, 0],
            &[0, 152, 123, 0, 60, 215, 0, 0],
            &[0, 0, 0, 0, 67, 212, 0, 0],
            &[0, 87, 223, 227, 175, 27, 0, 0],
        ],
        // letter: 'Ĵ' / 0x134
        'Ĵ' => &[
            &[0, 0, 0, 52, 57, 0, 0, 0],
            &[0, 0, 46, 204, 203, 56, 0, 0],
            &[0, 0, 117, 21, 17, 122, 2, 0],
            &[0, 0, 0, 72, 84, 0, 0, 0],
            &[0, 0, 0, 127, 148, 0, 0, 0],
            &[0, 0, 0, 127, 148, 0, 0, 0],
            &[0, 0, 0, 127, 148, 0, 0, 0],
            &[0, 0, 0, 127, 148, 0, 0, 0],
            &[0, 0, 0, 127, 148, 0, 0, 0],
            &[0, 0, 0, 140, 144, 0, 0, 0],
            &[24, 0, 13, 212, 103, 0, 0, 0],
            &[182, 238, 248, 181, 8, 0, 0, 0],
            &[0, 17, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĵ' / 0x135
        'ĵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 33, 0, 0, 0],
            &[0, 0, 24, 209, 213, 43, 0, 0],
            &[0, 0, 129, 44, 29, 144, 5, 0],
            &[0, 45, 110, 110, 84, 0, 0, 0],
            &[0, 34, 97, 158, 195, 0, 0, 0],
            &[0, 0, 0, 80, 195, 0, 0, 0],
            &[0, 0, 0, 80, 195, 0, 0, 0],
            &[0, 0, 0, 80, 195, 0, 0, 0],
            &[0, 0, 0, 80, 195, 0, 0, 0],
            &[0, 0, 0, 80, 195, 0, 0, 0],
            &[0, 0, 0, 87, 192, 0, 0, 0],
            &[54, 220, 228, 172, 21, 0, 0, 0],
        ],
        // letter: 'Ķ' / 0x136
        'Ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 133, 0, 0, 4, 134, 40, 0],
            &[40, 235, 0, 0, 138, 178, 0, 0],
            &[40, 235, 0, 75, 225, 19, 0, 0],
            &[40, 235, 28, 232, 60, 0, 0, 0],
            &[40, 236, 186, 206, 0, 0, 0, 0],
            &[40, 255, 129, 206, 97, 0, 0, 0],
            &[40, 237, 0, 53, 234, 22, 0, 0],
            &[40, 235, 0, 0, 146, 165, 0, 0],
            &[40, 235, 0, 0, 13, 228, 72, 0],
            &[0, 0, 0, 55, 16, 0, 0, 0],
            &[0, 0, 33, 87, 0, 0, 0, 0],
        ],
        // letter: 'ķ' / 0x137
        'ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 22, 10, 0, 0, 0, 0, 0],
            &[0, 189, 87, 0, 0, 0, 0, 0],
            &[0, 189, 87, 0, 0, 0, 0, 0],
            &[0, 189, 87, 0, 0, 90, 52, 0],
            &[0, 189, 87, 0, 119, 197, 10, 0],
            &[0, 189, 85, 95, 209, 17, 0, 0],
            &[0, 189, 142, 250, 84, 0, 0, 0],
            &[0, 189, 200, 77, 225, 21, 0, 0],
            &[0, 189, 87, 0, 127, 183, 1, 0],
            &[0, 189, 87, 0, 3, 194, 119, 0],
            &[0, 0, 0, 31, 40, 0, 0, 0],
            &[0, 0, 0, 119, 1, 0, 0, 0],
        ],
        // letter: 'ĸ' / 0x138
        'ĸ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 37, 0, 0, 90, 52, 0],
            &[0, 189, 87, 0, 119, 197, 10, 0],
            &[0, 189, 85, 95, 209, 17, 0, 0],
            &[0, 189, 143, 250, 84, 0, 0, 0],
            &[0, 189, 200, 77, 225, 21, 0, 0],
            &[0, 189, 87, 0, 127, 183, 1, 0],
            &[0, 189, 87, 0, 3, 194, 119, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĺ' / 0x139
        'Ĺ' => &[
            &[0, 0, 43, 67, 0, 0, 0, 0],
            &[0, 10, 216, 71, 0, 0, 0, 0],
            &[0, 66, 71, 0, 0, 0, 0, 0],
            &[0, 90, 66, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 255, 254, 254, 254, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĺ' / 0x13a
        'ĺ' => &[
            &[0, 0, 0, 14, 209, 59, 0, 0],
            &[0, 0, 0, 152, 102, 0, 0, 0],
            &[0, 29, 30, 38, 12, 0, 0, 0],
            &[0, 162, 185, 237, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[7, 194, 214, 246, 236, 205, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ļ' / 0x13b
        'Ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 66, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 255, 254, 254, 254, 45, 0],
            &[0, 0, 0, 38, 32, 0, 0, 0],
            &[0, 0, 1, 118, 0, 0, 0, 0],
        ],
        // letter: 'ļ' / 0x13c
        'ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 29, 30, 30, 12, 0, 0, 0],
            &[0, 162, 185, 237, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[7, 194, 214, 246, 236, 205, 90, 0],
            &[0, 0, 0, 45, 26, 0, 0, 0],
            &[0, 0, 10, 109, 0, 0, 0, 0],
        ],
        // letter: 'Ľ' / 0x13d
        'Ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 66, 0, 5, 140, 2, 0],
            &[0, 158, 117, 0, 51, 167, 0, 0],
            &[0, 158, 117, 0, 33, 25, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 255, 254, 254, 254, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ľ' / 0x13e
        'ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 29, 30, 30, 12, 9, 23, 0],
            &[0, 162, 185, 237, 100, 101, 143, 0],
            &[0, 0, 0, 176, 100, 121, 28, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[7, 194, 214, 246, 236, 205, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 66, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 3, 49, 0, 0],
            &[0, 158, 117, 0, 56, 255, 24, 0],
            &[0, 158, 117, 0, 5, 66, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 117, 0, 0, 0, 0, 0],
            &[0, 158, 255, 254, 254, 254, 45, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŀ' / 0x140
        'ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 29, 30, 30, 12, 0, 0, 0],
            &[0, 162, 185, 237, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 66, 139],
            &[0, 0, 0, 176, 100, 0, 83, 170],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[7, 194, 214, 246, 236, 205, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ł' / 0x141
        'Ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 136, 0, 0, 0, 0, 0],
            &[0, 35, 240, 0, 0, 0, 0, 0],
            &[0, 35, 240, 0, 0, 0, 0, 0],
            &[0, 35, 240, 14, 53, 0, 0, 0],
            &[0, 37, 255, 217, 100, 0, 0, 0],
            &[28, 232, 255, 23, 0, 0, 0, 0],
            &[8, 86, 240, 0, 0, 0, 0, 0],
            &[0, 35, 240, 0, 0, 0, 0, 0],
            &[0, 35, 255, 254, 254, 254, 168, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ł' / 0x142
        'ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 29, 30, 30, 12, 0, 0, 0],
            &[0, 162, 185, 237, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 169, 129, 0, 0],
            &[0, 0, 23, 255, 255, 33, 0, 0],
            &[0, 21, 211, 255, 100, 0, 0, 0],
            &[0, 0, 6, 176, 100, 0, 0, 0],
            &[0, 0, 0, 176, 100, 0, 0, 0],
            &[7, 194, 214, 246, 236, 205, 90, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ń' / 0x143
        'Ń' => &[
            &[0, 0, 0, 0, 65, 44, 0, 0],
            &[0, 0, 0, 49, 221, 27, 0, 0],
            &[0, 0, 0, 105, 32, 0, 0, 0],
            &[23, 145, 56, 0, 0, 99, 52, 0],
            &[40, 240, 188, 0, 0, 175, 92, 0],
            &[40, 206, 204, 44, 0, 175, 92, 0],
            &[40, 221, 92, 155, 0, 175, 92, 0],
            &[40, 226, 5, 221, 20, 175, 92, 0],
            &[40, 226, 0, 123, 123, 173, 92, 0],
            &[40, 226, 0, 20, 220, 171, 92, 0],
            &[40, 226, 0, 0, 155, 235, 92, 0],
            &[40, 226, 0, 0, 43, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ń' / 0x144
        'ń' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 32, 0, 0],
            &[0, 0, 0, 21, 228, 58, 0, 0],
            &[0, 0, 0, 107, 66, 0, 0, 0],
            &[4, 95, 24, 117, 126, 39, 0, 0],
            &[10, 247, 181, 99, 129, 238, 18, 0],
            &[10, 255, 59, 0, 0, 213, 67, 0],
            &[10, 255, 16, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ņ' / 0x145
        'Ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 145, 56, 0, 0, 99, 52, 0],
            &[40, 240, 188, 0, 0, 175, 92, 0],
            &[40, 206, 204, 44, 0, 175, 92, 0],
            &[40, 221, 92, 155, 0, 175, 92, 0],
            &[40, 226, 5, 221, 20, 175, 92, 0],
            &[40, 226, 0, 123, 123, 173, 92, 0],
            &[40, 226, 0, 20, 220, 171, 92, 0],
            &[40, 226, 0, 0, 155, 235, 92, 0],
            &[40, 226, 0, 0, 43, 255, 92, 0],
            &[0, 0, 0, 63, 8, 0, 0, 0],
            &[0, 0, 53, 67, 0, 0, 0, 0],
        ],
        // letter: 'ņ' / 0x146
        'ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[4, 95, 24, 117, 126, 39, 0, 0],
            &[10, 247, 181, 99, 129, 238, 18, 0],
            &[10, 255, 59, 0, 0, 213, 67, 0],
            &[10, 255, 16, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[0, 0, 0, 58, 12, 0, 0, 0],
            &[0, 0, 41, 79, 0, 0, 0, 0],
        ],
        // letter: 'Ň' / 0x147
        'Ň' => &[
            &[0, 27, 47, 0, 22, 54, 0, 0],
            &[0, 5, 182, 122, 205, 36, 0, 0],
            &[0, 0, 17, 149, 59, 0, 0, 0],
            &[23, 145, 56, 0, 0, 99, 52, 0],
            &[40, 240, 188, 0, 0, 175, 92, 0],
            &[40, 206, 204, 44, 0, 175, 92, 0],
            &[40, 221, 92, 155, 0, 175, 92, 0],
            &[40, 226, 5, 221, 20, 175, 92, 0],
            &[40, 226, 0, 123, 123, 173, 92, 0],
            &[40, 226, 0, 20, 220, 171, 92, 0],
            &[40, 226, 0, 0, 155, 235, 92, 0],
            &[40, 226, 0, 0, 43, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ň' / 0x148
        'ň' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 27, 0, 4, 37, 0, 0],
            &[0, 8, 184, 95, 189, 72, 0, 0],
            &[0, 0, 19, 179, 98, 0, 0, 0],
            &[4, 95, 24, 117, 126, 39, 0, 0],
            &[10, 247, 181, 99, 129, 238, 18, 0],
            &[10, 255, 59, 0, 0, 213, 67, 0],
            &[10, 255, 16, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 199, 76, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŉ' / 0x149
        'ŉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[8, 145, 35, 0, 0, 0, 0, 0],
            &[51, 246, 12, 0, 0, 0, 0, 0],
            &[103, 164, 95, 18, 110, 127, 41, 0],
            &[53, 33, 220, 192, 105, 131, 240, 18],
            &[0, 0, 220, 104, 0, 0, 214, 67],
            &[0, 0, 220, 62, 0, 0, 199, 76],
            &[0, 0, 220, 56, 0, 0, 199, 76],
            &[0, 0, 220, 56, 0, 0, 199, 76],
            &[0, 0, 220, 56, 0, 0, 199, 76],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 145, 56, 0, 0, 99, 52, 0],
            &[40, 240, 188, 0, 0, 175, 92, 0],
            &[40, 206, 204, 44, 0, 175, 92, 0],
            &[40, 221, 92, 155, 0, 175, 92, 0],
            &[40, 226, 5, 221, 20, 175, 92, 0],
            &[40, 226, 0, 123, 123, 173, 92, 0],
            &[40, 226, 0, 20, 220, 171, 92, 0],
            &[40, 226, 0, 0, 155, 235, 92, 0],
            &[40, 226, 0, 0, 43, 255, 182, 0],
            &[0, 0, 0, 0, 0, 203, 70, 0],
            &[0, 0, 0, 45, 55, 5, 0, 0],
        ],
        // letter: 'ŋ' / 0x14b
        'ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[4, 95, 24, 117, 126, 39, 0, 0],
            &[10, 247, 181, 99, 129, 238, 18, 0],
            &[10, 255, 59, 0, 0, 213, 67, 0],
            &[10, 255, 16, 0, 0, 199, 76, 0],
            &[10, 255, 10, 0, 0, 255, 152, 0],
            &[10, 255, 10, 0, 0, 255, 153, 0],
            &[10, 255, 10, 0, 0, 255, 153, 0],
            &[0, 0, 0, 0, 0, 199, 76, 0],
            &[0, 0, 0, 58, 215, 144, 1, 0],
        ],
        // letter: 'Ō' / 0x14c
        'Ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 19, 103, 103, 103, 38, 0, 0],
            &[0, 21, 114, 114, 114, 43, 0, 0],
            &[0, 9, 121, 173, 143, 26, 0, 0],
            &[0, 172, 176, 75, 144, 219, 8, 0],
            &[37, 239, 9, 0, 0, 200, 91, 0],
            &[93, 187, 0, 0, 0, 138, 146, 0],
            &[113, 168, 0, 0, 0, 118, 165, 0],
            &[106, 175, 0, 0, 0, 124, 157, 0],
            &[69, 216, 0, 0, 0, 163, 118, 0],
            &[7, 231, 81, 0, 39, 238, 35, 0],
            &[0, 61, 226, 225, 235, 95, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ō' / 0x14d
        'ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 68, 68, 68, 25, 0, 0],
            &[0, 28, 149, 149, 149, 56, 0, 0],
            &[0, 2, 87, 133, 100, 10, 0, 0],
            &[0, 166, 184, 94, 153, 204, 7, 0],
            &[39, 238, 9, 0, 0, 198, 88, 0],
            &[79, 195, 0, 0, 0, 147, 130, 0],
            &[69, 206, 0, 0, 0, 160, 121, 0],
            &[12, 237, 53, 0, 23, 234, 50, 0],
            &[0, 71, 221, 202, 226, 104, 0, 0],
            &[0, 0, 0, 18, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => &[
            &[0, 1, 1, 0, 1, 1, 0, 0],
            &[0, 35, 187, 53, 152, 80, 0, 0],
            &[0, 0, 72, 141, 93, 0, 0, 0],
            &[0, 9, 121, 173, 143, 26, 0, 0],
            &[0, 172, 176, 75, 144, 219, 8, 0],
            &[37, 239, 9, 0, 0, 200, 91, 0],
            &[93, 187, 0, 0, 0, 138, 146, 0],
            &[113, 168, 0, 0, 0, 118, 165, 0],
            &[106, 175, 0, 0, 0, 124, 157, 0],
            &[69, 216, 0, 0, 0, 163, 118, 0],
            &[7, 231, 81, 0, 39, 238, 35, 0],
            &[0, 61, 226, 225, 235, 95, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŏ' / 0x14f
        'ŏ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 36, 157, 19, 119, 78, 0, 0],
            &[0, 0, 104, 176, 128, 3, 0, 0],
            &[0, 2, 87, 133, 100, 10, 0, 0],
            &[0, 166, 184, 94, 153, 204, 7, 0],
            &[39, 238, 9, 0, 0, 198, 88, 0],
            &[79, 195, 0, 0, 0, 147, 130, 0],
            &[69, 206, 0, 0, 0, 160, 121, 0],
            &[12, 237, 53, 0, 23, 234, 50, 0],
            &[0, 71, 221, 202, 226, 104, 0, 0],
            &[0, 0, 0, 18, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ő' / 0x150
        'Ő' => &[
            &[0, 0, 30, 69, 20, 78, 0, 0],
            &[0, 0, 184, 87, 156, 114, 0, 0],
            &[0, 31, 90, 15, 105, 0, 0, 0],
            &[0, 9, 121, 173, 143, 26, 0, 0],
            &[0, 172, 176, 75, 144, 219, 8, 0],
            &[37, 239, 9, 0, 0, 200, 91, 0],
            &[93, 187, 0, 0, 0, 138, 146, 0],
            &[113, 168, 0, 0, 0, 118, 165, 0],
            &[106, 175, 0, 0, 0, 124, 157, 0],
            &[69, 216, 0, 0, 0, 163, 118, 0],
            &[7, 231, 81, 0, 39, 238, 35, 0],
            &[0, 61, 226, 225, 235, 95, 0, 0],
            &[0, 0, 2, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ő' / 0x151
        'ő' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 42, 10, 48, 0, 0],
            &[0, 0, 166, 113, 136, 142, 0, 0],
            &[0, 31, 123, 16, 136, 2, 0, 0],
            &[0, 2, 87, 133, 100, 10, 0, 0],
            &[0, 166, 184, 94, 153, 204, 7, 0],
            &[39, 238, 9, 0, 0, 198, 88, 0],
            &[79, 195, 0, 0, 0, 147, 130, 0],
            &[69, 206, 0, 0, 0, 160, 121, 0],
            &[12, 237, 53, 0, 23, 234, 50, 0],
            &[0, 71, 221, 202, 226, 104, 0, 0],
            &[0, 0, 0, 18, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Œ' / 0x152
        'Œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 152, 163, 145, 145, 122, 0],
            &[5, 221, 126, 132, 219, 100, 84, 0],
            &[70, 214, 0, 71, 195, 0, 0, 0],
            &[108, 167, 0, 71, 195, 0, 0, 0],
            &[127, 153, 0, 71, 252, 242, 148, 0],
            &[122, 157, 0, 71, 195, 0, 0, 0],
            &[92, 186, 0, 71, 195, 0, 0, 0],
            &[28, 242, 25, 71, 195, 0, 0, 0],
            &[0, 122, 239, 240, 253, 245, 206, 0],
            &[0, 0, 13, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'œ' / 0x153
        'œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 79, 130, 22, 70, 124, 17, 0],
            &[71, 200, 109, 202, 191, 127, 162, 0],
            &[156, 88, 0, 203, 94, 4, 218, 0],
            &[181, 62, 0, 179, 213, 196, 232, 0],
            &[175, 69, 0, 186, 84, 0, 0, 0],
            &[125, 124, 2, 228, 138, 0, 4, 0],
            &[22, 213, 214, 122, 198, 195, 147, 0],
            &[0, 3, 18, 0, 2, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŕ' / 0x154
        'Ŕ' => &[
            &[0, 0, 0, 0, 70, 39, 0, 0],
            &[0, 0, 0, 62, 216, 20, 0, 0],
            &[0, 0, 0, 112, 25, 0, 0, 0],
            &[0, 125, 145, 138, 96, 8, 0, 0],
            &[0, 220, 129, 104, 188, 194, 0, 0],
            &[0, 220, 56, 0, 12, 254, 29, 0],
            &[0, 220, 56, 0, 25, 253, 18, 0],
            &[0, 220, 144, 129, 210, 126, 0, 0],
            &[0, 220, 155, 164, 195, 0, 0, 0],
            &[0, 220, 56, 4, 211, 83, 0, 0],
            &[0, 220, 56, 0, 65, 225, 11, 0],
            &[0, 220, 56, 0, 0, 170, 135, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŕ' / 0x155
        'ŕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 48, 0, 0],
            &[0, 0, 0, 0, 173, 134, 0, 0],
            &[0, 0, 0, 45, 128, 0, 0, 0],
            &[7, 110, 110, 65, 52, 133, 89, 0],
            &[5, 92, 155, 203, 202, 111, 97, 0],
            &[0, 0, 76, 251, 34, 0, 0, 0],
            &[0, 0, 76, 210, 0, 0, 0, 0],
            &[0, 0, 76, 202, 0, 0, 0, 0],
            &[0, 0, 76, 202, 0, 0, 0, 0],
            &[47, 202, 232, 250, 212, 127, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŗ' / 0x156
        'Ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 125, 145, 138, 96, 8, 0, 0],
            &[0, 220, 129, 104, 188, 194, 0, 0],
            &[0, 220, 56, 0, 12, 254, 29, 0],
            &[0, 220, 56, 0, 25, 253, 18, 0],
            &[0, 220, 144, 129, 210, 126, 0, 0],
            &[0, 220, 155, 164, 195, 0, 0, 0],
            &[0, 220, 56, 4, 211, 83, 0, 0],
            &[0, 220, 56, 0, 65, 225, 11, 0],
            &[0, 220, 56, 0, 0, 170, 135, 0],
            &[0, 0, 0, 38, 32, 0, 0, 0],
            &[0, 0, 1, 118, 0, 0, 0, 0],
        ],
        // letter: 'ŗ' / 0x157
        'ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[7, 110, 110, 65, 52, 133, 89, 0],
            &[5, 92, 155, 203, 202, 111, 97, 0],
            &[0, 0, 76, 251, 34, 0, 0, 0],
            &[0, 0, 76, 210, 0, 0, 0, 0],
            &[0, 0, 76, 202, 0, 0, 0, 0],
            &[0, 0, 76, 202, 0, 0, 0, 0],
            &[47, 202, 232, 250, 212, 127, 0, 0],
            &[0, 0, 18, 52, 0, 0, 0, 0],
            &[0, 0, 106, 14, 0, 0, 0, 0],
        ],
        // letter: 'Ř' / 0x158
        'Ř' => &[
            &[0, 32, 42, 0, 27, 49, 0, 0],
            &[0, 9, 190, 119, 204, 28, 0, 0],
            &[0, 0, 24, 151, 50, 0, 0, 0],
            &[0, 125, 145, 138, 96, 8, 0, 0],
            &[0, 220, 129, 104, 188, 194, 0, 0],
            &[0, 220, 56, 0, 12, 254, 29, 0],
            &[0, 220, 56, 0, 25, 253, 18, 0],
            &[0, 220, 144, 129, 210, 126, 0, 0],
            &[0, 220, 155, 164, 195, 0, 0, 0],
            &[0, 220, 56, 4, 211, 83, 0, 0],
            &[0, 220, 56, 0, 65, 225, 11, 0],
            &[0, 220, 56, 0, 0, 170, 135, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ř' / 0x159
        'ř' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 41, 0, 0, 37, 5, 0],
            &[0, 0, 116, 154, 126, 151, 0, 0],
            &[0, 0, 0, 136, 158, 2, 0, 0],
            &[7, 110, 110, 65, 52, 133, 89, 0],
            &[5, 92, 155, 203, 202, 111, 97, 0],
            &[0, 0, 76, 251, 34, 0, 0, 0],
            &[0, 0, 76, 210, 0, 0, 0, 0],
            &[0, 0, 76, 202, 0, 0, 0, 0],
            &[0, 0, 76, 202, 0, 0, 0, 0],
            &[47, 202, 232, 250, 212, 127, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ś' / 0x15a
        'Ś' => &[
            &[0, 0, 0, 0, 51, 58, 0, 0],
            &[0, 0, 0, 22, 224, 52, 0, 0],
            &[0, 0, 0, 82, 55, 0, 0, 0],
            &[0, 2, 100, 165, 160, 111, 8, 0],
            &[0, 127, 212, 87, 86, 138, 3, 0],
            &[0, 193, 98, 0, 0, 0, 0, 0],
            &[0, 149, 186, 18, 0, 0, 0, 0],
            &[0, 7, 154, 240, 140, 20, 0, 0],
            &[0, 0, 0, 35, 162, 222, 18, 0],
            &[0, 0, 0, 0, 0, 212, 79, 0],
            &[0, 39, 0, 0, 30, 241, 47, 0],
            &[0, 207, 234, 215, 236, 119, 0, 0],
            &[0, 0, 9, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ś' / 0x15b
        'ś' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 27, 0, 0],
            &[0, 0, 0, 33, 230, 43, 0, 0],
            &[0, 0, 0, 121, 52, 0, 0, 0],
            &[0, 1, 82, 134, 124, 58, 0, 0],
            &[0, 115, 210, 105, 108, 105, 0, 0],
            &[0, 135, 160, 2, 0, 0, 0, 0],
            &[0, 17, 170, 217, 105, 6, 0, 0],
            &[0, 0, 0, 42, 178, 172, 0, 0],
            &[0, 27, 0, 0, 59, 227, 0, 0],
            &[0, 157, 227, 203, 229, 98, 0, 0],
            &[0, 0, 10, 25, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => &[
            &[0, 0, 0, 70, 39, 0, 0, 0],
            &[0, 0, 89, 189, 207, 24, 0, 0],
            &[0, 12, 120, 5, 40, 102, 0, 0],
            &[0, 2, 100, 165, 160, 111, 8, 0],
            &[0, 127, 212, 87, 86, 138, 3, 0],
            &[0, 193, 98, 0, 0, 0, 0, 0],
            &[0, 149, 186, 18, 0, 0, 0, 0],
            &[0, 7, 154, 240, 140, 20, 0, 0],
            &[0, 0, 0, 35, 162, 222, 18, 0],
            &[0, 0, 0, 0, 0, 212, 79, 0],
            &[0, 39, 0, 0, 30, 241, 47, 0],
            &[0, 207, 234, 215, 236, 119, 0, 0],
            &[0, 0, 9, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŝ' / 0x15d
        'ŝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 47, 12, 0, 0, 0],
            &[0, 0, 104, 207, 176, 2, 0, 0],
            &[0, 31, 137, 4, 94, 84, 0, 0],
            &[0, 1, 82, 134, 124, 58, 0, 0],
            &[0, 115, 210, 105, 108, 105, 0, 0],
            &[0, 135, 160, 2, 0, 0, 0, 0],
            &[0, 17, 170, 217, 105, 6, 0, 0],
            &[0, 0, 0, 42, 178, 172, 0, 0],
            &[0, 27, 0, 0, 59, 227, 0, 0],
            &[0, 157, 227, 203, 229, 98, 0, 0],
            &[0, 0, 10, 25, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ş' / 0x15e
        'Ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 100, 165, 160, 111, 8, 0],
            &[0, 127, 212, 87, 86, 138, 3, 0],
            &[0, 193, 98, 0, 0, 0, 0, 0],
            &[0, 149, 186, 18, 0, 0, 0, 0],
            &[0, 7, 154, 240, 140, 20, 0, 0],
            &[0, 0, 0, 35, 162, 222, 18, 0],
            &[0, 0, 0, 0, 0, 212, 79, 0],
            &[0, 39, 0, 0, 30, 241, 47, 0],
            &[0, 207, 234, 226, 236, 119, 0, 0],
            &[0, 0, 20, 224, 15, 0, 0, 0],
            &[0, 0, 109, 180, 78, 0, 0, 0],
        ],
        // letter: 'ş' / 0x15f
        'ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 82, 134, 124, 58, 0, 0],
            &[0, 115, 210, 105, 108, 105, 0, 0],
            &[0, 135, 160, 2, 0, 0, 0, 0],
            &[0, 17, 170, 217, 105, 6, 0, 0],
            &[0, 0, 0, 42, 178, 172, 0, 0],
            &[0, 27, 0, 0, 59, 227, 0, 0],
            &[0, 157, 227, 213, 230, 98, 0, 0],
            &[0, 0, 15, 229, 22, 0, 0, 0],
            &[0, 0, 99, 179, 88, 0, 0, 0],
        ],
        // letter: 'Š' / 0x160
        'Š' => &[
            &[0, 13, 61, 0, 9, 66, 0, 0],
            &[0, 0, 149, 140, 198, 64, 0, 0],
            &[0, 0, 3, 137, 85, 0, 0, 0],
            &[0, 2, 100, 165, 160, 111, 8, 0],
            &[0, 127, 212, 87, 86, 138, 3, 0],
            &[0, 193, 98, 0, 0, 0, 0, 0],
            &[0, 149, 186, 18, 0, 0, 0, 0],
            &[0, 7, 154, 240, 140, 20, 0, 0],
            &[0, 0, 0, 35, 162, 222, 18, 0],
            &[0, 0, 0, 0, 0, 212, 79, 0],
            &[0, 39, 0, 0, 30, 241, 47, 0],
            &[0, 207, 234, 215, 236, 119, 0, 0],
            &[0, 0, 9, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'š' / 0x161
        'š' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 17, 23, 0, 8, 33, 0, 0],
            &[0, 15, 193, 88, 196, 56, 0, 0],
            &[0, 0, 30, 184, 82, 0, 0, 0],
            &[0, 1, 82, 134, 124, 58, 0, 0],
            &[0, 115, 210, 105, 108, 105, 0, 0],
            &[0, 135, 160, 2, 0, 0, 0, 0],
            &[0, 17, 170, 217, 105, 6, 0, 0],
            &[0, 0, 0, 42, 178, 172, 0, 0],
            &[0, 27, 0, 0, 59, 227, 0, 0],
            &[0, 157, 227, 203, 229, 98, 0, 0],
            &[0, 0, 10, 25, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ţ' / 0x162
        'Ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[66, 145, 145, 145, 145, 145, 95, 0],
            &[46, 100, 100, 246, 122, 100, 66, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 251, 35, 0, 0, 0],
            &[0, 0, 10, 201, 13, 0, 0, 0],
            &[0, 0, 109, 180, 78, 0, 0, 0],
        ],
        // letter: 'ţ' / 0x163
        'ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 19, 0, 0, 0, 0],
            &[0, 0, 113, 128, 0, 0, 0, 0],
            &[24, 93, 191, 183, 110, 110, 1, 0],
            &[39, 119, 198, 188, 119, 119, 1, 0],
            &[0, 0, 147, 128, 0, 0, 0, 0],
            &[0, 0, 147, 128, 0, 0, 0, 0],
            &[0, 0, 147, 129, 0, 0, 0, 0],
            &[0, 0, 132, 172, 0, 0, 0, 0],
            &[0, 0, 34, 218, 224, 209, 42, 0],
            &[0, 0, 0, 26, 209, 22, 0, 0],
            &[0, 0, 0, 128, 180, 59, 0, 0],
        ],
        // letter: 'Ť' / 0x164
        'Ť' => &[
            &[0, 32, 42, 0, 27, 49, 0, 0],
            &[0, 9, 190, 119, 204, 28, 0, 0],
            &[0, 0, 24, 151, 50, 0, 0, 0],
            &[66, 145, 145, 145, 145, 145, 95, 0],
            &[46, 100, 100, 246, 122, 100, 66, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ť' / 0x165
        'ť' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 8, 0],
            &[0, 0, 13, 19, 11, 223, 3, 0],
            &[0, 0, 113, 128, 40, 81, 0, 0],
            &[24, 93, 191, 183, 110, 110, 1, 0],
            &[39, 119, 198, 188, 119, 119, 1, 0],
            &[0, 0, 147, 128, 0, 0, 0, 0],
            &[0, 0, 147, 128, 0, 0, 0, 0],
            &[0, 0, 147, 129, 0, 0, 0, 0],
            &[0, 0, 132, 172, 0, 0, 0, 0],
            &[0, 0, 34, 218, 213, 209, 42, 0],
            &[0, 0, 0, 0, 17, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŧ' / 0x166
        'Ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[66, 145, 145, 145, 145, 145, 95, 0],
            &[46, 100, 100, 246, 122, 100, 66, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 144, 201, 255, 237, 187, 0, 0],
            &[0, 18, 25, 255, 61, 23, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 240, 35, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŧ' / 0x167
        'ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 19, 0, 0, 0, 0],
            &[0, 0, 113, 128, 0, 0, 0, 0],
            &[24, 93, 191, 183, 110, 110, 1, 0],
            &[39, 119, 198, 188, 119, 119, 1, 0],
            &[12, 38, 186, 167, 38, 38, 7, 0],
            &[53, 163, 255, 255, 163, 163, 31, 0],
            &[0, 0, 147, 129, 0, 0, 0, 0],
            &[0, 0, 132, 172, 0, 0, 0, 0],
            &[0, 0, 34, 218, 213, 209, 42, 0],
            &[0, 0, 0, 0, 17, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ũ' / 0x168
        'Ũ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 211, 177, 89, 148, 0, 0],
            &[0, 75, 28, 56, 134, 28, 0, 0],
            &[23, 133, 0, 0, 0, 104, 52, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[29, 251, 3, 0, 0, 202, 80, 0],
            &[1, 229, 81, 0, 35, 243, 27, 0],
            &[0, 73, 232, 223, 237, 107, 0, 0],
            &[0, 0, 3, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ũ' / 0x169
        'ũ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 28, 185, 142, 54, 131, 0, 0],
            &[0, 89, 55, 90, 169, 45, 0, 0],
            &[11, 108, 0, 0, 0, 92, 26, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 215, 61, 0],
            &[25, 251, 0, 0, 0, 235, 61, 0],
            &[5, 251, 41, 0, 52, 255, 61, 0],
            &[0, 124, 237, 201, 178, 193, 61, 0],
            &[0, 0, 9, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ū' / 0x16a
        'Ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 19, 103, 103, 103, 38, 0, 0],
            &[0, 21, 114, 114, 114, 43, 0, 0],
            &[23, 133, 0, 0, 0, 104, 52, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[29, 251, 3, 0, 0, 202, 80, 0],
            &[1, 229, 81, 0, 35, 243, 27, 0],
            &[0, 73, 232, 223, 237, 107, 0, 0],
            &[0, 0, 3, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ū' / 0x16b
        'ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 68, 68, 68, 25, 0, 0],
            &[0, 28, 149, 149, 149, 56, 0, 0],
            &[11, 108, 0, 0, 0, 92, 26, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 215, 61, 0],
            &[25, 251, 0, 0, 0, 235, 61, 0],
            &[5, 251, 41, 0, 52, 255, 61, 0],
            &[0, 124, 237, 201, 178, 193, 61, 0],
            &[0, 0, 9, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => &[
            &[0, 1, 1, 0, 1, 1, 0, 0],
            &[0, 35, 187, 53, 152, 80, 0, 0],
            &[0, 0, 72, 141, 93, 0, 0, 0],
            &[23, 133, 0, 0, 0, 104, 52, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[29, 251, 3, 0, 0, 202, 80, 0],
            &[1, 229, 81, 0, 35, 243, 27, 0],
            &[0, 73, 232, 223, 237, 107, 0, 0],
            &[0, 0, 3, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŭ' / 0x16d
        'ŭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 36, 157, 19, 119, 78, 0, 0],
            &[0, 0, 104, 176, 128, 3, 0, 0],
            &[11, 108, 0, 0, 0, 92, 26, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 215, 61, 0],
            &[25, 251, 0, 0, 0, 235, 61, 0],
            &[5, 251, 41, 0, 52, 255, 61, 0],
            &[0, 124, 237, 201, 178, 193, 61, 0],
            &[0, 0, 9, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ů' / 0x16e
        'Ů' => &[
            &[0, 0, 95, 173, 132, 0, 0, 0],
            &[0, 0, 169, 40, 188, 0, 0, 0],
            &[0, 0, 39, 140, 62, 0, 0, 0],
            &[23, 133, 0, 0, 0, 104, 52, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[29, 251, 3, 0, 0, 202, 80, 0],
            &[1, 229, 81, 0, 35, 243, 27, 0],
            &[0, 73, 232, 223, 237, 107, 0, 0],
            &[0, 0, 3, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ů' / 0x16f
        'ů' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 72, 170, 105, 0, 0, 0],
            &[0, 0, 171, 16, 188, 0, 0, 0],
            &[0, 0, 59, 168, 89, 0, 0, 0],
            &[11, 108, 0, 0, 0, 92, 26, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 215, 61, 0],
            &[25, 251, 0, 0, 0, 235, 61, 0],
            &[5, 251, 41, 0, 52, 255, 61, 0],
            &[0, 124, 237, 201, 178, 193, 61, 0],
            &[0, 0, 9, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ű' / 0x170
        'Ű' => &[
            &[0, 0, 30, 69, 20, 78, 0, 0],
            &[0, 0, 184, 87, 156, 114, 0, 0],
            &[0, 31, 90, 15, 105, 0, 0, 0],
            &[23, 133, 0, 0, 0, 104, 52, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[29, 251, 3, 0, 0, 202, 80, 0],
            &[1, 229, 81, 0, 35, 243, 27, 0],
            &[0, 73, 232, 223, 237, 107, 0, 0],
            &[0, 0, 3, 23, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ű' / 0x171
        'ű' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 42, 10, 48, 0, 0],
            &[0, 0, 166, 113, 136, 142, 0, 0],
            &[0, 31, 123, 16, 136, 2, 0, 0],
            &[11, 108, 0, 0, 0, 92, 26, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 215, 61, 0],
            &[25, 251, 0, 0, 0, 235, 61, 0],
            &[5, 251, 41, 0, 52, 255, 61, 0],
            &[0, 124, 237, 201, 178, 193, 61, 0],
            &[0, 0, 9, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ų' / 0x172
        'Ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 133, 0, 0, 0, 104, 52, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[40, 235, 0, 0, 0, 184, 92, 0],
            &[29, 251, 3, 0, 0, 202, 80, 0],
            &[1, 229, 81, 0, 35, 243, 27, 0],
            &[0, 73, 232, 225, 255, 107, 0, 0],
            &[0, 0, 4, 193, 57, 0, 0, 0],
            &[0, 0, 1, 126, 119, 0, 0, 0],
        ],
        // letter: 'ų' / 0x173
        'ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[11, 108, 0, 0, 0, 92, 26, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 214, 61, 0],
            &[25, 250, 0, 0, 0, 215, 61, 0],
            &[25, 251, 0, 0, 0, 235, 61, 0],
            &[5, 251, 41, 0, 52, 255, 61, 0],
            &[0, 124, 237, 201, 178, 210, 62, 0],
            &[0, 0, 9, 19, 32, 173, 4, 0],
            &[0, 0, 0, 0, 26, 151, 71, 0],
        ],
        // letter: 'Ŵ' / 0x174
        'Ŵ' => &[
            &[0, 0, 6, 83, 20, 0, 0, 0],
            &[0, 0, 147, 174, 184, 4, 0, 0],
            &[0, 39, 98, 0, 73, 69, 0, 0],
            &[97, 46, 0, 0, 0, 17, 126, 0],
            &[148, 101, 0, 0, 0, 47, 202, 0],
            &[119, 126, 0, 105, 19, 69, 176, 0],
            &[91, 150, 30, 246, 89, 92, 151, 0],
            &[62, 174, 90, 163, 151, 113, 125, 0],
            &[33, 196, 149, 61, 200, 133, 99, 0],
            &[6, 212, 197, 6, 185, 166, 73, 0],
            &[0, 212, 189, 0, 125, 230, 47, 0],
            &[0, 203, 135, 0, 63, 255, 22, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŵ' / 0x175
        'ŵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 48, 10, 0, 0, 0],
            &[0, 0, 116, 206, 166, 0, 0, 0],
            &[0, 38, 132, 2, 103, 75, 0, 0],
            &[85, 25, 0, 0, 0, 0, 107, 0],
            &[168, 86, 1, 56, 12, 28, 221, 0],
            &[127, 123, 52, 251, 99, 69, 179, 0],
            &[86, 161, 126, 159, 169, 109, 137, 0],
            &[45, 194, 194, 40, 224, 145, 95, 0],
            &[7, 215, 213, 0, 182, 207, 53, 0],
            &[0, 217, 157, 0, 117, 253, 12, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŷ' / 0x176
        'Ŷ' => &[
            &[0, 0, 6, 83, 20, 0, 0, 0],
            &[0, 0, 147, 174, 184, 4, 0, 0],
            &[0, 39, 98, 0, 73, 69, 0, 0],
            &[63, 107, 0, 0, 0, 73, 92, 0],
            &[21, 239, 30, 0, 2, 219, 60, 0],
            &[0, 134, 144, 0, 85, 186, 0, 0],
            &[0, 18, 230, 18, 204, 56, 0, 0],
            &[0, 0, 130, 186, 181, 0, 0, 0],
            &[0, 0, 16, 246, 58, 0, 0, 0],
            &[0, 0, 0, 234, 29, 0, 0, 0],
            &[0, 0, 0, 234, 29, 0, 0, 0],
            &[0, 0, 0, 234, 29, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŷ' / 0x177
        'ŷ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 48, 9, 0, 0, 0],
            &[0, 0, 119, 206, 163, 0, 0, 0],
            &[0, 39, 131, 2, 105, 73, 0, 0],
            &[40, 75, 0, 0, 0, 58, 62, 0],
            &[24, 233, 9, 0, 0, 196, 77, 0],
            &[0, 167, 97, 0, 27, 231, 4, 0],
            &[0, 61, 201, 0, 111, 143, 0, 0],
            &[0, 0, 210, 47, 196, 48, 0, 0],
            &[0, 0, 104, 161, 207, 0, 0, 0],
            &[0, 0, 12, 240, 114, 0, 0, 0],
            &[0, 0, 5, 229, 23, 0, 0, 0],
            &[49, 220, 191, 20, 0, 0, 0, 0],
        ],
        // letter: 'Ÿ' / 0x178
        'Ÿ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 76, 186, 0, 138, 124, 0, 0],
            &[0, 18, 59, 0, 42, 36, 0, 0],
            &[63, 107, 0, 0, 0, 73, 92, 0],
            &[21, 239, 30, 0, 2, 219, 60, 0],
            &[0, 134, 144, 0, 85, 186, 0, 0],
            &[0, 18, 230, 18, 204, 56, 0, 0],
            &[0, 0, 130, 186, 181, 0, 0, 0],
            &[0, 0, 16, 246, 58, 0, 0, 0],
            &[0, 0, 0, 234, 29, 0, 0, 0],
            &[0, 0, 0, 234, 29, 0, 0, 0],
            &[0, 0, 0, 234, 29, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ź' / 0x179
        'Ź' => &[
            &[0, 0, 0, 0, 58, 51, 0, 0],
            &[0, 0, 0, 34, 225, 38, 0, 0],
            &[0, 0, 0, 94, 43, 0, 0, 0],
            &[0, 90, 145, 145, 145, 145, 8, 0],
            &[0, 58, 94, 94, 131, 229, 5, 0],
            &[0, 0, 0, 0, 176, 98, 0, 0],
            &[0, 0, 0, 64, 209, 2, 0, 0],
            &[0, 0, 2, 206, 69, 0, 0, 0],
            &[0, 0, 95, 183, 0, 0, 0, 0],
            &[0, 10, 225, 44, 0, 0, 0, 0],
            &[0, 127, 154, 0, 0, 0, 0, 0],
            &[0, 218, 248, 245, 245, 245, 14, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ź' / 0x17a
        'ź' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 30, 0, 0],
            &[0, 0, 0, 24, 229, 53, 0, 0],
            &[0, 0, 0, 111, 62, 0, 0, 0],
            &[0, 80, 110, 110, 110, 110, 2, 0],
            &[0, 73, 101, 101, 159, 225, 2, 0],
            &[0, 0, 0, 24, 223, 52, 0, 0],
            &[0, 0, 4, 193, 100, 0, 0, 0],
            &[0, 0, 140, 155, 0, 0, 0, 0],
            &[0, 83, 202, 7, 0, 0, 0, 0],
            &[0, 228, 242, 233, 233, 233, 28, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ż' / 0x17b
        'Ż' => &[
            &[0, 0, 0, 36, 8, 0, 0, 0],
            &[0, 0, 0, 236, 100, 0, 0, 0],
            &[0, 0, 0, 38, 8, 0, 0, 0],
            &[0, 90, 145, 145, 145, 145, 8, 0],
            &[0, 58, 94, 94, 131, 229, 5, 0],
            &[0, 0, 0, 0, 176, 98, 0, 0],
            &[0, 0, 0, 64, 209, 2, 0, 0],
            &[0, 0, 2, 206, 69, 0, 0, 0],
            &[0, 0, 95, 183, 0, 0, 0, 0],
            &[0, 10, 225, 44, 0, 0, 0, 0],
            &[0, 127, 154, 0, 0, 0, 0, 0],
            &[0, 218, 248, 245, 245, 245, 14, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ż' / 0x17c
        'ż' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 0, 0, 0, 0],
            &[0, 0, 1, 238, 90, 0, 0, 0],
            &[0, 0, 0, 70, 18, 0, 0, 0],
            &[0, 80, 110, 110, 110, 110, 2, 0],
            &[0, 73, 101, 101, 159, 225, 2, 0],
            &[0, 0, 0, 24, 223, 52, 0, 0],
            &[0, 0, 4, 193, 100, 0, 0, 0],
            &[0, 0, 140, 155, 0, 0, 0, 0],
            &[0, 83, 202, 7, 0, 0, 0, 0],
            &[0, 228, 242, 233, 233, 233, 28, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ž' / 0x17d
        'Ž' => &[
            &[0, 20, 54, 0, 15, 61, 0, 0],
            &[0, 2, 167, 130, 204, 49, 0, 0],
            &[0, 0, 9, 144, 72, 0, 0, 0],
            &[0, 90, 145, 145, 145, 145, 8, 0],
            &[0, 58, 94, 94, 131, 229, 5, 0],
            &[0, 0, 0, 0, 176, 98, 0, 0],
            &[0, 0, 0, 64, 209, 2, 0, 0],
            &[0, 0, 2, 206, 69, 0, 0, 0],
            &[0, 0, 95, 183, 0, 0, 0, 0],
            &[0, 10, 225, 44, 0, 0, 0, 0],
            &[0, 127, 154, 0, 0, 0, 0, 0],
            &[0, 218, 248, 245, 245, 245, 14, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ž' / 0x17e
        'ž' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 14, 26, 0, 6, 36, 0, 0],
            &[0, 10, 187, 93, 191, 67, 0, 0],
            &[0, 0, 22, 181, 93, 0, 0, 0],
            &[0, 80, 110, 110, 110, 110, 2, 0],
            &[0, 73, 101, 101, 159, 225, 2, 0],
            &[0, 0, 0, 24, 223, 52, 0, 0],
            &[0, 0, 4, 193, 100, 0, 0, 0],
            &[0, 0, 140, 155, 0, 0, 0, 0],
            &[0, 83, 202, 7, 0, 0, 0, 0],
            &[0, 228, 242, 233, 233, 233, 28, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ſ' / 0x17f
        'ſ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 37, 5, 0, 0],
            &[0, 5, 188, 212, 188, 96, 0, 0],
            &[0, 69, 217, 1, 0, 0, 0, 0],
            &[0, 88, 182, 0, 0, 0, 0, 0],
            &[0, 89, 181, 0, 0, 0, 0, 0],
            &[0, 89, 181, 0, 0, 0, 0, 0],
            &[0, 89, 181, 0, 0, 0, 0, 0],
            &[0, 89, 181, 0, 0, 0, 0, 0],
            &[0, 89, 181, 0, 0, 0, 0, 0],
            &[0, 89, 181, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0],
        ],
        _ => panic!("unsupported char"),
    }
}
