//! Module for letters with the font weight light and size 24.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 24;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 13;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight light and font size 21px
#[inline]
pub const fn get_char(c: char) -> &'static [&'static [u8]] {
    match c {
        // letter: ' ' / 0x20
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '!' / 0x21
        '!' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 103, 254, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 99, 255, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 95, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 90, 249, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 86, 245, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 240, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 77, 236, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 227, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 64, 223, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 92, 211, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 133, 255, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '"' / 0x22
        '"' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 254, 47, 0, 148, 170, 0, 0, 0, 0],
            &[0, 0, 0, 1, 254, 35, 0, 137, 159, 0, 0, 0, 0],
            &[0, 0, 0, 0, 244, 23, 0, 125, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 232, 11, 0, 114, 136, 0, 0, 0, 0],
            &[0, 0, 0, 0, 219, 1, 0, 102, 124, 0, 0, 0, 0],
            &[0, 0, 0, 0, 86, 0, 0, 39, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '#' / 0x23
        '#' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 165, 0, 0, 78, 168, 0, 0, 0],
            &[0, 0, 0, 0, 136, 123, 0, 0, 122, 126, 0, 0, 0],
            &[0, 0, 0, 0, 180, 80, 0, 0, 166, 83, 0, 0, 0],
            &[0, 0, 0, 0, 224, 37, 0, 0, 210, 40, 0, 0, 0],
            &[0, 101, 122, 124, 254, 125, 122, 122, 249, 126, 122, 57, 0],
            &[0, 98, 119, 155, 220, 119, 119, 151, 225, 119, 119, 55, 0],
            &[0, 0, 0, 101, 156, 0, 0, 93, 164, 0, 0, 0, 0],
            &[0, 0, 0, 144, 113, 0, 0, 136, 121, 0, 0, 0, 0],
            &[0, 0, 0, 188, 69, 0, 0, 180, 77, 0, 0, 0, 0],
            &[80, 149, 149, 240, 165, 149, 149, 237, 169, 149, 115, 0, 0],
            &[52, 97, 115, 239, 97, 97, 110, 244, 97, 97, 75, 0, 0],
            &[0, 0, 63, 193, 0, 0, 57, 199, 0, 0, 0, 0, 0],
            &[0, 0, 106, 148, 0, 0, 103, 154, 0, 0, 0, 0, 0],
            &[0, 0, 149, 104, 0, 0, 148, 109, 0, 0, 0, 0, 0],
            &[0, 0, 192, 60, 0, 0, 194, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '$' / 0x24
        '$' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 167, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 67, 143, 216, 105, 67, 5, 0, 0, 0],
            &[0, 0, 35, 212, 228, 190, 223, 173, 225, 231, 7, 0, 0],
            &[0, 0, 201, 168, 6, 57, 178, 0, 0, 37, 0, 0, 0],
            &[0, 18, 255, 43, 0, 57, 178, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 34, 0, 57, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 193, 179, 9, 57, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 200, 225, 160, 179, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 73, 190, 253, 183, 78, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 197, 128, 233, 183, 8, 0, 0],
            &[0, 0, 0, 0, 0, 57, 178, 0, 26, 221, 120, 0, 0],
            &[0, 0, 0, 0, 0, 57, 178, 0, 0, 136, 176, 0, 0],
            &[0, 0, 0, 0, 0, 57, 178, 0, 0, 151, 153, 0, 0],
            &[0, 0, 66, 4, 0, 57, 178, 0, 75, 245, 62, 0, 0],
            &[0, 1, 233, 241, 203, 193, 235, 220, 230, 87, 0, 0, 0],
            &[0, 0, 1, 45, 78, 135, 206, 55, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 178, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '%' / 0x25
        '%' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 16, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[19, 188, 204, 209, 172, 10, 0, 0, 0, 193, 89, 0, 0],
            &[150, 136, 0, 0, 160, 123, 0, 0, 68, 213, 2, 0, 0],
            &[226, 38, 0, 0, 63, 200, 0, 0, 198, 86, 0, 0, 0],
            &[246, 19, 0, 0, 45, 221, 0, 72, 210, 1, 0, 0, 0],
            &[215, 52, 0, 0, 77, 192, 0, 202, 81, 0, 0, 0, 0],
            &[117, 176, 8, 14, 195, 98, 77, 206, 1, 0, 0, 0, 0],
            &[3, 131, 215, 213, 122, 2, 206, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 209, 73, 23, 168, 218, 195, 60, 0],
            &[0, 0, 0, 0, 85, 198, 0, 197, 108, 2, 49, 230, 25],
            &[0, 0, 0, 2, 213, 69, 38, 228, 1, 0, 0, 153, 114],
            &[0, 0, 0, 89, 194, 0, 68, 198, 0, 0, 0, 122, 144],
            &[0, 0, 3, 216, 64, 0, 46, 218, 0, 0, 0, 141, 123],
            &[0, 0, 94, 190, 0, 0, 4, 220, 59, 0, 14, 223, 51],
            &[0, 4, 219, 60, 0, 0, 0, 57, 219, 187, 224, 121, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 2, 34, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '&' / 0x26
        '&' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 46, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 105, 239, 206, 239, 110, 0, 0, 0, 0, 0, 0],
            &[0, 34, 245, 45, 0, 39, 244, 36, 0, 0, 0, 0, 0],
            &[0, 87, 201, 0, 0, 0, 199, 89, 0, 0, 0, 0, 0],
            &[0, 70, 218, 0, 0, 0, 216, 80, 0, 0, 0, 0, 0],
            &[0, 18, 244, 39, 0, 77, 240, 13, 0, 0, 0, 0, 0],
            &[0, 0, 144, 167, 58, 240, 85, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 175, 255, 255, 42, 0, 0, 0, 0, 0, 0, 0],
            &[1, 185, 190, 9, 112, 216, 14, 0, 0, 199, 87, 0, 0],
            &[84, 230, 17, 0, 0, 170, 174, 1, 12, 250, 25, 0, 0],
            &[156, 147, 0, 0, 0, 13, 216, 121, 108, 204, 0, 0, 0],
            &[171, 134, 0, 0, 0, 0, 44, 255, 255, 90, 0, 0, 0],
            &[135, 190, 0, 0, 0, 0, 2, 255, 255, 33, 0, 0, 0],
            &[38, 247, 121, 4, 0, 33, 188, 216, 174, 204, 8, 0, 0],
            &[0, 71, 224, 248, 230, 244, 151, 15, 7, 203, 158, 0, 0],
            &[0, 0, 2, 36, 43, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ''' / 0x27
        '\'' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 83, 236, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 72, 225, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 60, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 201, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 36, 189, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 76, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '(' / 0x28
        '(' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 212, 83, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 141, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 30, 246, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 136, 174, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 224, 89, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 255, 28, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 227, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 198, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 138, 179, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 172, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 190, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 104, 211, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 60, 251, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 245, 66, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 172, 142, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 67, 235, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 192, 119, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 40, 232, 34, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 50, 45, 0, 0, 0, 0],
        ],
        // letter: ')' / 0x29
        ')' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 180, 122, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 238, 45, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 131, 181, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 251, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 189, 128, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 128, 196, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 70, 245, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 43, 255, 22, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 24, 255, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 255, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 38, 255, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 60, 250, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 208, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 171, 146, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 238, 65, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 216, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 217, 87, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 181, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 77, 18, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '*' / 0x2a
        '*' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 201, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 243, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 230, 1, 0, 0, 0, 0, 0],
            &[0, 0, 166, 158, 82, 12, 209, 9, 78, 157, 195, 0, 0],
            &[0, 0, 47, 108, 165, 216, 246, 222, 176, 121, 65, 0, 0],
            &[0, 0, 0, 0, 0, 138, 190, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 211, 5, 191, 116, 0, 0, 0, 0],
            &[0, 0, 0, 36, 239, 61, 0, 41, 246, 62, 0, 0, 0],
            &[0, 0, 0, 29, 109, 0, 0, 0, 102, 45, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '+' / 0x2b
        '+' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 55, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 211, 0, 0, 0, 0, 0, 0],
            &[0, 99, 234, 234, 234, 238, 252, 234, 234, 234, 234, 5, 0],
            &[0, 9, 23, 23, 23, 65, 215, 23, 23, 23, 23, 0, 0],
            &[0, 0, 0, 0, 0, 46, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 92, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ',' / 0x2c
        ',' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 149, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 255, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 98, 248, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 153, 180, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 213, 90, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 199, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '-' / 0x2d
        '-' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 65, 76, 76, 76, 76, 76, 34, 0, 0, 0],
            &[0, 0, 0, 172, 202, 202, 202, 202, 202, 91, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '.' / 0x2e
        '.' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 96, 211, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 139, 255, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '/' / 0x2f
        '/' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 163, 141, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 14, 246, 45, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 102, 204, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 199, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 40, 247, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 137, 168, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 230, 71, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 76, 227, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 173, 133, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 249, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 208, 97, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 49, 243, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 147, 159, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 237, 62, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '0' / 0x30
        '0' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 46, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 58, 211, 250, 233, 254, 161, 16, 0, 0, 0],
            &[0, 0, 37, 240, 124, 6, 0, 30, 198, 185, 0, 0, 0],
            &[0, 0, 161, 162, 0, 0, 0, 0, 147, 255, 63, 0, 0],
            &[0, 6, 246, 54, 0, 0, 0, 33, 243, 192, 154, 0, 0],
            &[0, 48, 245, 2, 0, 0, 0, 174, 147, 85, 204, 0, 0],
            &[0, 86, 209, 0, 0, 0, 69, 234, 16, 48, 241, 0, 0],
            &[0, 101, 192, 0, 0, 5, 214, 100, 0, 32, 254, 1, 0],
            &[0, 110, 183, 0, 0, 114, 202, 1, 0, 24, 255, 8, 0],
            &[0, 99, 190, 0, 23, 239, 54, 0, 0, 31, 255, 1, 0],
            &[0, 84, 210, 0, 159, 157, 0, 0, 0, 50, 239, 0, 0],
            &[0, 45, 244, 58, 238, 22, 0, 0, 0, 86, 205, 0, 0],
            &[0, 3, 245, 255, 110, 0, 0, 0, 0, 147, 148, 0, 0],
            &[0, 0, 161, 255, 3, 0, 0, 0, 17, 237, 61, 0, 0],
            &[0, 0, 31, 235, 125, 9, 0, 31, 191, 177, 0, 0, 0],
            &[0, 0, 0, 50, 203, 249, 234, 249, 161, 12, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 45, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '1' / 0x31
        '1' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 222, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 174, 225, 182, 164, 0, 0, 0, 0, 0],
            &[0, 0, 61, 241, 146, 12, 129, 164, 0, 0, 0, 0, 0],
            &[0, 0, 9, 57, 0, 0, 134, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 135, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 136, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 136, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 136, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 136, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 136, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 136, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 136, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 136, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 136, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 107, 227, 240, 255, 255, 241, 228, 123, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '2' / 0x32
        '2' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 28, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 16, 140, 234, 245, 239, 251, 172, 22, 0, 0, 0],
            &[0, 0, 195, 178, 57, 1, 0, 38, 194, 200, 1, 0, 0],
            &[0, 0, 18, 0, 0, 0, 0, 0, 24, 250, 61, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 214, 102, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 214, 98, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 15, 251, 54, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 117, 220, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 33, 241, 86, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 16, 214, 150, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 190, 181, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 182, 193, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 178, 196, 12, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 173, 198, 13, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 168, 219, 43, 28, 28, 28, 28, 28, 28, 5, 0],
            &[0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 49, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '3' / 0x33
        '3' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 33, 46, 21, 0, 0, 0, 0, 0],
            &[0, 0, 27, 149, 237, 245, 236, 254, 196, 45, 0, 0, 0],
            &[0, 1, 208, 174, 51, 1, 0, 23, 163, 234, 19, 0, 0],
            &[0, 0, 11, 0, 0, 0, 0, 0, 5, 229, 104, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 188, 131, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 1, 221, 99, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 111, 229, 15, 0, 0],
            &[0, 0, 0, 4, 86, 88, 110, 185, 193, 39, 0, 0, 0],
            &[0, 0, 0, 10, 198, 199, 213, 224, 151, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 8, 109, 241, 60, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 134, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 71, 246, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 237, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 183, 174, 0, 0],
            &[0, 39, 148, 58, 5, 0, 1, 35, 164, 241, 41, 0, 0],
            &[0, 18, 149, 232, 250, 231, 246, 249, 177, 39, 0, 0, 0],
            &[0, 0, 0, 0, 19, 45, 36, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '4' / 0x34
        '4' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 19, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 158, 255, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 65, 213, 246, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 218, 74, 250, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 134, 174, 0, 255, 39, 0, 0, 0],
            &[0, 0, 0, 0, 46, 236, 27, 3, 255, 39, 0, 0, 0],
            &[0, 0, 0, 2, 201, 106, 0, 5, 255, 39, 0, 0, 0],
            &[0, 0, 0, 110, 198, 2, 0, 5, 255, 39, 0, 0, 0],
            &[0, 0, 29, 237, 44, 0, 0, 5, 255, 39, 0, 0, 0],
            &[0, 0, 179, 132, 0, 0, 0, 5, 255, 39, 0, 0, 0],
            &[0, 86, 218, 8, 0, 0, 0, 5, 255, 39, 0, 0, 0],
            &[0, 209, 228, 201, 201, 201, 201, 202, 255, 210, 201, 89, 0],
            &[0, 56, 67, 67, 67, 67, 67, 70, 255, 96, 67, 29, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 255, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 255, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 5, 255, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '5' / 0x35
        '5' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 254, 254, 254, 254, 254, 254, 248, 0, 0, 0],
            &[0, 0, 47, 244, 25, 25, 25, 25, 25, 24, 0, 0, 0],
            &[0, 0, 69, 221, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 198, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 175, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 135, 152, 0, 8, 7, 0, 0, 0, 0, 0, 0],
            &[0, 0, 155, 232, 239, 250, 254, 225, 135, 11, 0, 0, 0],
            &[0, 0, 22, 63, 15, 0, 8, 71, 212, 200, 7, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 22, 237, 99, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 157, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 133, 185, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 169, 159, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 32, 242, 83, 0, 0],
            &[0, 14, 156, 53, 2, 0, 7, 74, 217, 184, 2, 0, 0],
            &[0, 6, 143, 232, 246, 232, 253, 235, 135, 7, 0, 0, 0],
            &[0, 0, 0, 0, 19, 46, 31, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '6' / 0x36
        '6' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 17, 46, 49, 15, 0, 0, 0],
            &[0, 0, 0, 0, 61, 195, 248, 222, 221, 207, 0, 0, 0],
            &[0, 0, 0, 92, 240, 108, 13, 0, 0, 7, 0, 0, 0],
            &[0, 0, 34, 246, 63, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 155, 157, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 237, 59, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 42, 249, 5, 2, 55, 87, 74, 17, 0, 0, 0, 0],
            &[0, 81, 218, 40, 201, 202, 172, 203, 243, 117, 0, 0, 0],
            &[0, 98, 218, 189, 49, 0, 0, 0, 55, 237, 91, 0, 0],
            &[0, 105, 255, 55, 0, 0, 0, 0, 0, 115, 207, 0, 0],
            &[0, 92, 231, 0, 0, 0, 0, 0, 0, 53, 254, 4, 0],
            &[0, 59, 238, 0, 0, 0, 0, 0, 0, 39, 255, 9, 0],
            &[0, 10, 246, 45, 0, 0, 0, 0, 0, 72, 236, 0, 0],
            &[0, 0, 162, 166, 0, 0, 0, 0, 0, 167, 161, 0, 0],
            &[0, 0, 30, 234, 143, 10, 0, 8, 128, 241, 33, 0, 0],
            &[0, 0, 0, 41, 195, 245, 223, 252, 199, 47, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 46, 23, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '7' / 0x37
        '7' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 254, 254, 254, 254, 254, 254, 254, 254, 254, 6, 0],
            &[0, 0, 25, 25, 25, 25, 25, 25, 25, 121, 203, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 206, 100, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 55, 242, 11, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 160, 151, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 18, 247, 48, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 114, 201, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 217, 99, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 67, 241, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 172, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 251, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 126, 199, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 227, 97, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 240, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 184, 147, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '8' / 0x38
        '8' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 36, 43, 12, 0, 0, 0, 0, 0],
            &[0, 0, 1, 120, 236, 238, 229, 248, 166, 18, 0, 0, 0],
            &[0, 0, 122, 228, 58, 0, 0, 28, 187, 192, 0, 0, 0],
            &[0, 0, 231, 90, 0, 0, 0, 0, 25, 255, 40, 0, 0],
            &[0, 1, 252, 53, 0, 0, 0, 0, 0, 246, 56, 0, 0],
            &[0, 0, 207, 116, 0, 0, 0, 0, 58, 244, 14, 0, 0],
            &[0, 0, 65, 244, 101, 1, 0, 56, 232, 100, 0, 0, 0],
            &[0, 0, 0, 54, 207, 209, 177, 216, 72, 0, 0, 0, 0],
            &[0, 0, 0, 54, 194, 205, 211, 215, 82, 0, 0, 0, 0],
            &[0, 0, 94, 235, 86, 1, 1, 83, 224, 147, 0, 0, 0],
            &[0, 30, 248, 63, 0, 0, 0, 0, 23, 230, 84, 0, 0],
            &[0, 106, 209, 0, 0, 0, 0, 0, 0, 137, 170, 0, 0],
            &[0, 120, 195, 0, 0, 0, 0, 0, 0, 131, 181, 0, 0],
            &[0, 74, 242, 18, 0, 0, 0, 0, 2, 200, 130, 0, 0],
            &[0, 2, 197, 190, 33, 0, 0, 20, 158, 226, 21, 0, 0],
            &[0, 0, 15, 154, 244, 229, 225, 247, 171, 30, 0, 0, 0],
            &[0, 0, 0, 0, 9, 40, 42, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '9' / 0x39
        '9' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 39, 46, 10, 0, 0, 0, 0, 0],
            &[0, 0, 1, 120, 237, 237, 223, 243, 139, 6, 0, 0, 0],
            &[0, 0, 132, 221, 57, 0, 0, 36, 202, 158, 0, 0, 0],
            &[0, 22, 251, 57, 0, 0, 0, 0, 22, 243, 54, 0, 0],
            &[0, 91, 218, 0, 0, 0, 0, 0, 0, 152, 148, 0, 0],
            &[0, 117, 186, 0, 0, 0, 0, 0, 0, 90, 206, 0, 0],
            &[0, 109, 202, 0, 0, 0, 0, 0, 0, 79, 240, 0, 0],
            &[0, 55, 244, 20, 0, 0, 0, 0, 0, 165, 253, 0, 0],
            &[0, 0, 193, 175, 14, 0, 0, 5, 130, 181, 248, 0, 0],
            &[0, 0, 21, 186, 237, 188, 182, 228, 129, 76, 229, 0, 0],
            &[0, 0, 0, 0, 37, 76, 71, 20, 0, 114, 193, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 176, 134, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 27, 249, 48, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 178, 177, 0, 0, 0],
            &[0, 0, 10, 2, 0, 0, 49, 185, 217, 21, 0, 0, 0],
            &[0, 0, 95, 244, 225, 241, 240, 147, 17, 0, 0, 0, 0],
            &[0, 0, 1, 31, 49, 36, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ':' / 0x3a
        ':' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 65, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 141, 255, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 70, 175, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 96, 211, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 139, 255, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ';' / 0x3b
        ';' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 55, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 255, 126, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 167, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 149, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 77, 255, 50, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 125, 232, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 180, 153, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 238, 63, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 185, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '<' / 0x3c
        '<' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 70, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 78, 208, 179, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 202, 194, 66, 0, 0, 0],
            &[0, 0, 0, 0, 65, 196, 196, 67, 0, 0, 0, 0, 0],
            &[0, 0, 59, 190, 197, 68, 0, 0, 0, 0, 0, 0, 0],
            &[0, 70, 255, 141, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 75, 198, 205, 84, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 65, 188, 215, 96, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 56, 178, 224, 108, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 47, 169, 211, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '=' / 0x3d
        '=' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 4, 4, 4, 4, 4, 4, 4, 4, 4, 0, 0],
            &[0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0],
            &[0, 10, 19, 19, 19, 19, 19, 19, 19, 19, 19, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 122, 230, 230, 230, 230, 230, 230, 230, 230, 230, 29, 0],
            &[0, 28, 54, 54, 54, 54, 54, 54, 54, 54, 54, 7, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '>' / 0x3e
        '>' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 36, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 48, 229, 157, 32, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 117, 228, 150, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 11, 118, 227, 144, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 120, 225, 138, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 27, 217, 224, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 132, 231, 149, 31, 0, 0],
            &[0, 0, 0, 0, 29, 145, 233, 139, 24, 0, 0, 0, 0],
            &[0, 0, 38, 157, 233, 129, 18, 0, 0, 0, 0, 0, 0],
            &[0, 63, 230, 119, 13, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 19, 9, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '?' / 0x3f
        '?' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 44, 42, 11, 0, 0, 0, 0, 0],
            &[0, 0, 75, 197, 254, 241, 236, 248, 163, 18, 0, 0, 0],
            &[0, 0, 62, 106, 22, 0, 0, 37, 196, 187, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 29, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 243, 62, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 22, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 145, 190, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 137, 225, 33, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 174, 202, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 143, 191, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 241, 51, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 255, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 13, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 214, 62, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 76, 255, 95, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '@' / 0x40
        '@' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 139, 217, 244, 225, 166, 44, 0, 0, 0],
            &[0, 0, 53, 228, 144, 41, 5, 16, 97, 227, 75, 0, 0],
            &[0, 22, 230, 79, 0, 0, 0, 0, 0, 42, 231, 25, 0],
            &[0, 152, 139, 0, 0, 0, 0, 0, 0, 0, 124, 146, 0],
            &[11, 237, 20, 0, 11, 136, 205, 199, 147, 8, 27, 225, 0],
            &[77, 191, 0, 0, 183, 156, 31, 51, 255, 11, 0, 224, 29],
            &[127, 131, 0, 66, 222, 3, 0, 17, 253, 1, 0, 193, 52],
            &[155, 106, 0, 138, 146, 0, 0, 32, 241, 0, 0, 183, 68],
            &[169, 88, 0, 166, 116, 0, 0, 52, 228, 0, 0, 196, 55],
            &[161, 96, 0, 157, 122, 0, 0, 85, 222, 0, 0, 218, 29],
            &[139, 117, 0, 118, 168, 0, 0, 156, 235, 0, 27, 222, 0],
            &[97, 170, 0, 26, 236, 110, 112, 198, 158, 131, 164, 115, 0],
            &[25, 237, 8, 0, 49, 149, 135, 19, 20, 148, 125, 3, 0],
            &[0, 175, 121, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 34, 234, 76, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 230, 164, 71, 36, 45, 79, 158, 93, 0, 0],
            &[0, 0, 0, 18, 121, 189, 216, 201, 169, 94, 11, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'A' / 0x41
        'A' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 134, 254, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 222, 220, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 238, 90, 210, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 158, 10, 244, 43, 0, 0, 0, 0],
            &[0, 0, 0, 3, 231, 70, 0, 167, 132, 0, 0, 0, 0],
            &[0, 0, 0, 68, 234, 4, 0, 80, 221, 0, 0, 0, 0],
            &[0, 0, 0, 157, 151, 0, 0, 7, 241, 55, 0, 0, 0],
            &[0, 0, 7, 239, 63, 0, 0, 0, 162, 144, 0, 0, 0],
            &[0, 0, 80, 255, 68, 66, 66, 66, 141, 241, 2, 0, 0],
            &[0, 0, 169, 255, 218, 218, 218, 218, 224, 255, 67, 0, 0],
            &[0, 12, 246, 56, 0, 0, 0, 0, 0, 156, 156, 0, 0],
            &[0, 92, 223, 0, 0, 0, 0, 0, 0, 69, 238, 6, 0],
            &[0, 181, 137, 0, 0, 0, 0, 0, 0, 3, 234, 78, 0],
            &[19, 250, 49, 0, 0, 0, 0, 0, 0, 0, 151, 168, 0],
            &[103, 217, 0, 0, 0, 0, 0, 0, 0, 0, 64, 245, 11],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'B' / 0x42
        'B' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 202, 254, 254, 254, 248, 225, 172, 65, 0, 0, 0, 0],
            &[0, 203, 109, 19, 20, 34, 61, 146, 252, 90, 0, 0, 0],
            &[0, 203, 97, 0, 0, 0, 0, 0, 142, 215, 0, 0, 0],
            &[0, 203, 97, 0, 0, 0, 0, 0, 75, 249, 0, 0, 0],
            &[0, 203, 97, 0, 0, 0, 0, 0, 93, 224, 0, 0, 0],
            &[0, 203, 97, 0, 0, 0, 0, 9, 205, 127, 0, 0, 0],
            &[0, 203, 160, 102, 102, 113, 144, 213, 134, 3, 0, 0, 0],
            &[0, 203, 203, 171, 171, 178, 204, 204, 119, 7, 0, 0, 0],
            &[0, 203, 97, 0, 0, 0, 0, 24, 185, 187, 1, 0, 0],
            &[0, 203, 97, 0, 0, 0, 0, 0, 23, 252, 58, 0, 0],
            &[0, 203, 97, 0, 0, 0, 0, 0, 0, 228, 95, 0, 0],
            &[0, 203, 97, 0, 0, 0, 0, 0, 5, 250, 78, 0, 0],
            &[0, 203, 97, 0, 0, 0, 0, 0, 107, 245, 18, 0, 0],
            &[0, 203, 112, 23, 23, 34, 64, 152, 249, 101, 0, 0, 0],
            &[0, 203, 255, 255, 255, 251, 229, 171, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'C' / 0x43
        'C' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 42, 35, 9, 0, 0, 0],
            &[0, 0, 0, 0, 69, 196, 253, 237, 244, 254, 193, 59, 0],
            &[0, 0, 0, 118, 245, 121, 22, 0, 0, 26, 111, 37, 0],
            &[0, 0, 69, 250, 63, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 204, 141, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 255, 36, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 228, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 128, 192, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 127, 193, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 104, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 255, 25, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 224, 124, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 241, 47, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 153, 237, 112, 22, 0, 4, 38, 108, 13, 0],
            &[0, 0, 0, 1, 110, 224, 255, 245, 253, 251, 191, 13, 0],
            &[0, 0, 0, 0, 0, 0, 28, 49, 34, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'D' / 0x44
        'D' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 254, 254, 254, 241, 213, 148, 44, 0, 0, 0, 0],
            &[0, 65, 246, 19, 20, 37, 73, 163, 248, 104, 0, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 77, 250, 68, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 137, 213, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 30, 255, 41, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 0, 220, 99, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 0, 194, 122, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 0, 180, 134, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 0, 192, 115, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 0, 229, 90, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 39, 254, 24, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 154, 191, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 98, 248, 44, 0, 0],
            &[0, 65, 246, 23, 25, 42, 88, 181, 240, 75, 0, 0, 0],
            &[0, 65, 255, 255, 255, 235, 205, 128, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'E' / 0x45
        'E' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 196, 0, 0],
            &[0, 0, 203, 122, 25, 25, 25, 25, 25, 25, 19, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 170, 107, 107, 107, 107, 107, 107, 38, 0, 0],
            &[0, 0, 203, 207, 171, 171, 171, 171, 171, 171, 60, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 124, 28, 28, 28, 28, 28, 28, 22, 0, 0],
            &[0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'F' / 0x46
        'F' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 196, 0, 0],
            &[0, 0, 203, 122, 25, 25, 25, 25, 25, 25, 19, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 215, 186, 186, 186, 186, 186, 186, 69, 0, 0],
            &[0, 0, 203, 161, 93, 93, 93, 93, 93, 93, 34, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'G' / 0x47
        'G' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 32, 48, 20, 0, 0, 0, 0],
            &[0, 0, 0, 20, 144, 244, 250, 236, 255, 213, 83, 0, 0],
            &[0, 0, 25, 214, 196, 59, 3, 0, 11, 92, 64, 0, 0],
            &[0, 0, 176, 191, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 53, 252, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 193, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 193, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 218, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 236, 84, 0, 0, 0, 63, 164, 164, 164, 144, 0, 0],
            &[0, 228, 93, 0, 0, 0, 46, 120, 120, 163, 224, 0, 0],
            &[0, 208, 114, 0, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 161, 169, 0, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 90, 239, 15, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 5, 218, 149, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 0, 58, 241, 162, 39, 1, 0, 22, 140, 224, 0, 0],
            &[0, 0, 0, 44, 175, 253, 247, 243, 255, 206, 108, 0, 0],
            &[0, 0, 0, 0, 0, 8, 38, 45, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'H' / 0x48
        'H' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 86, 223, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 250, 107, 107, 107, 107, 107, 107, 158, 224, 0, 0],
            &[0, 65, 252, 177, 177, 177, 177, 177, 177, 203, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'I' / 0x49
        'I' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 254, 254, 252, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 255, 255, 253, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'J' / 0x4a
        'J' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 232, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 236, 76, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 251, 60, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 65, 252, 17, 0, 0, 0],
            &[0, 0, 94, 48, 3, 0, 34, 211, 169, 0, 0, 0, 0],
            &[0, 0, 114, 240, 248, 229, 253, 180, 17, 0, 0, 0, 0],
            &[0, 0, 0, 6, 38, 56, 26, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'K' / 0x4b
        'K' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 153, 209, 13, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 120, 231, 30, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 88, 245, 54, 0, 0, 0],
            &[0, 65, 245, 0, 0, 0, 60, 246, 84, 0, 0, 0, 0],
            &[0, 65, 245, 0, 0, 38, 237, 120, 0, 0, 0, 0, 0],
            &[0, 65, 245, 0, 21, 220, 158, 0, 0, 0, 0, 0, 0],
            &[0, 65, 245, 6, 198, 235, 6, 0, 0, 0, 0, 0, 0],
            &[0, 65, 245, 146, 192, 228, 105, 0, 0, 0, 0, 0, 0],
            &[0, 65, 254, 200, 13, 73, 243, 33, 0, 0, 0, 0, 0],
            &[0, 65, 250, 18, 0, 0, 158, 191, 1, 0, 0, 0, 0],
            &[0, 65, 245, 0, 0, 0, 15, 227, 109, 0, 0, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 72, 244, 35, 0, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 156, 194, 2, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 14, 226, 112, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 71, 245, 37, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'L' / 0x4c
        'L' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 254, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 36, 34, 34, 34, 34, 34, 26, 0, 0],
            &[0, 0, 52, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'M' / 0x4d
        'M' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 214, 250, 22, 0, 0, 0, 0, 0, 126, 254, 113, 0],
            &[0, 216, 229, 110, 0, 0, 0, 0, 0, 217, 231, 113, 0],
            &[0, 216, 155, 201, 0, 0, 0, 0, 53, 236, 173, 113, 0],
            &[0, 216, 88, 243, 37, 0, 0, 0, 144, 149, 177, 113, 0],
            &[0, 216, 83, 162, 127, 0, 0, 3, 232, 57, 181, 113, 0],
            &[0, 216, 84, 69, 218, 0, 0, 70, 220, 0, 181, 113, 0],
            &[0, 216, 84, 3, 230, 54, 0, 162, 129, 0, 181, 113, 0],
            &[0, 216, 84, 0, 140, 145, 10, 242, 37, 0, 181, 113, 0],
            &[0, 216, 84, 0, 48, 232, 92, 200, 0, 0, 181, 113, 0],
            &[0, 216, 84, 0, 0, 211, 231, 108, 0, 0, 181, 113, 0],
            &[0, 216, 84, 0, 0, 118, 248, 21, 0, 0, 181, 113, 0],
            &[0, 216, 84, 0, 0, 0, 0, 0, 0, 0, 181, 113, 0],
            &[0, 216, 84, 0, 0, 0, 0, 0, 0, 0, 181, 113, 0],
            &[0, 216, 84, 0, 0, 0, 0, 0, 0, 0, 181, 113, 0],
            &[0, 216, 84, 0, 0, 0, 0, 0, 0, 0, 181, 113, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'N' / 0x4e
        'N' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 254, 132, 0, 0, 0, 0, 0, 81, 223, 0, 0],
            &[0, 65, 234, 239, 16, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 212, 169, 127, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 225, 46, 237, 14, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 232, 0, 175, 122, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 50, 235, 12, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 181, 117, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 56, 233, 10, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 186, 113, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 61, 230, 8, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 192, 108, 78, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 67, 227, 78, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 197, 165, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 72, 251, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 0, 202, 224, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'O' / 0x4f
        'O' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 49, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 217, 250, 230, 251, 184, 31, 0, 0, 0],
            &[0, 0, 79, 245, 105, 5, 0, 21, 157, 226, 21, 0, 0],
            &[0, 8, 231, 105, 0, 0, 0, 0, 1, 190, 150, 0, 0],
            &[0, 94, 229, 4, 0, 0, 0, 0, 0, 70, 244, 8, 0],
            &[0, 159, 156, 0, 0, 0, 0, 0, 0, 6, 246, 66, 0],
            &[0, 207, 108, 0, 0, 0, 0, 0, 0, 0, 209, 107, 0],
            &[0, 226, 88, 0, 0, 0, 0, 0, 0, 0, 186, 129, 0],
            &[0, 239, 77, 0, 0, 0, 0, 0, 0, 0, 179, 137, 0],
            &[0, 230, 88, 0, 0, 0, 0, 0, 0, 0, 190, 127, 0],
            &[0, 206, 110, 0, 0, 0, 0, 0, 0, 0, 211, 103, 0],
            &[0, 162, 159, 0, 0, 0, 0, 0, 0, 7, 250, 61, 0],
            &[0, 92, 232, 5, 0, 0, 0, 0, 0, 77, 241, 5, 0],
            &[0, 9, 233, 111, 0, 0, 0, 0, 4, 198, 143, 0, 0],
            &[0, 0, 82, 249, 112, 10, 0, 31, 173, 221, 17, 0, 0],
            &[0, 0, 0, 77, 218, 254, 242, 252, 175, 25, 0, 0, 0],
            &[0, 0, 0, 0, 1, 30, 45, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'P' / 0x50
        'P' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 202, 254, 254, 254, 250, 232, 182, 83, 0, 0, 0, 0],
            &[0, 203, 119, 19, 19, 30, 57, 135, 244, 133, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 90, 252, 28, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 4, 249, 86, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 227, 94, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 11, 250, 63, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 130, 226, 3, 0, 0],
            &[0, 203, 135, 46, 46, 57, 92, 176, 239, 60, 0, 0, 0],
            &[0, 203, 242, 232, 232, 221, 196, 130, 26, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Q' / 0x51
        'Q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 49, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 217, 250, 230, 251, 184, 31, 0, 0, 0],
            &[0, 0, 79, 245, 105, 5, 0, 21, 157, 226, 21, 0, 0],
            &[0, 8, 231, 105, 0, 0, 0, 0, 1, 190, 150, 0, 0],
            &[0, 94, 229, 4, 0, 0, 0, 0, 0, 70, 244, 8, 0],
            &[0, 159, 156, 0, 0, 0, 0, 0, 0, 6, 246, 66, 0],
            &[0, 207, 108, 0, 0, 0, 0, 0, 0, 0, 209, 107, 0],
            &[0, 226, 88, 0, 0, 0, 0, 0, 0, 0, 186, 129, 0],
            &[0, 239, 77, 0, 0, 0, 0, 0, 0, 0, 179, 137, 0],
            &[0, 230, 88, 0, 0, 0, 0, 0, 0, 0, 190, 127, 0],
            &[0, 206, 110, 0, 0, 0, 0, 0, 0, 0, 211, 103, 0],
            &[0, 162, 159, 0, 0, 0, 0, 0, 0, 7, 250, 61, 0],
            &[0, 92, 232, 5, 0, 0, 0, 0, 0, 77, 241, 5, 0],
            &[0, 9, 233, 111, 0, 0, 0, 0, 4, 198, 143, 0, 0],
            &[0, 0, 82, 249, 112, 10, 0, 31, 179, 221, 17, 0, 0],
            &[0, 0, 0, 77, 218, 254, 242, 255, 255, 25, 0, 0, 0],
            &[0, 0, 0, 0, 1, 30, 45, 63, 249, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 163, 178, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 32, 248, 64, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 111, 123, 0, 0],
        ],
        // letter: 'R' / 0x52
        'R' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 202, 254, 254, 253, 239, 209, 142, 27, 0, 0, 0, 0],
            &[0, 203, 119, 19, 22, 43, 84, 190, 236, 41, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 3, 193, 170, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 108, 227, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 85, 237, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 134, 197, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 54, 241, 84, 0, 0, 0],
            &[0, 203, 184, 132, 132, 143, 186, 235, 100, 0, 0, 0, 0],
            &[0, 203, 192, 146, 146, 161, 250, 31, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 182, 158, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 37, 247, 57, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 135, 205, 2, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 11, 228, 105, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 88, 238, 20, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 192, 155, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'S' / 0x53
        'S' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 43, 37, 11, 0, 0, 0, 0],
            &[0, 0, 0, 63, 199, 252, 235, 243, 255, 206, 86, 0, 0],
            &[0, 0, 59, 248, 130, 20, 0, 0, 20, 94, 63, 0, 0],
            &[0, 0, 179, 156, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 220, 96, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 205, 121, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 116, 234, 51, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 144, 251, 173, 65, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 151, 245, 210, 100, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 17, 115, 227, 211, 28, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 11, 189, 185, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 66, 251, 3, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 57, 255, 7, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 134, 213, 0, 0],
            &[0, 24, 119, 45, 2, 0, 0, 20, 126, 250, 78, 0, 0],
            &[0, 20, 173, 241, 247, 226, 231, 253, 200, 69, 0, 0, 0],
            &[0, 0, 0, 2, 30, 48, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'T' / 0x54
        'T' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 252, 254, 254, 254, 254, 254, 254, 254, 254, 254, 155, 0],
            &[0, 24, 25, 25, 25, 96, 234, 25, 25, 25, 25, 15, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'U' / 0x55
        'U' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 86, 223, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 64, 247, 0, 0, 0, 0, 0, 0, 88, 223, 0, 0],
            &[0, 50, 255, 5, 0, 0, 0, 0, 0, 102, 208, 0, 0],
            &[0, 21, 255, 44, 0, 0, 0, 0, 0, 143, 176, 0, 0],
            &[0, 0, 208, 131, 0, 0, 0, 0, 7, 227, 104, 0, 0],
            &[0, 0, 80, 249, 98, 4, 0, 30, 178, 217, 7, 0, 0],
            &[0, 0, 0, 94, 228, 251, 240, 253, 179, 26, 0, 0, 0],
            &[0, 0, 0, 0, 3, 31, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'V' / 0x56
        'V' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[36, 253, 34, 0, 0, 0, 0, 0, 0, 0, 135, 194, 0],
            &[0, 207, 117, 0, 0, 0, 0, 0, 0, 0, 217, 111, 0],
            &[0, 121, 200, 0, 0, 0, 0, 0, 0, 43, 254, 27, 0],
            &[0, 36, 254, 29, 0, 0, 0, 0, 0, 125, 197, 0, 0],
            &[0, 0, 207, 111, 0, 0, 0, 0, 0, 207, 113, 0, 0],
            &[0, 0, 121, 194, 0, 0, 0, 0, 33, 254, 29, 0, 0],
            &[0, 0, 36, 253, 23, 0, 0, 0, 115, 199, 0, 0, 0],
            &[0, 0, 0, 207, 104, 0, 0, 0, 197, 115, 0, 0, 0],
            &[0, 0, 0, 121, 187, 0, 0, 25, 253, 31, 0, 0, 0],
            &[0, 0, 0, 36, 251, 19, 0, 104, 202, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 98, 0, 186, 117, 0, 0, 0, 0],
            &[0, 0, 0, 0, 121, 177, 14, 249, 33, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 244, 88, 204, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 207, 213, 119, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 121, 255, 35, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'W' / 0x57
        'W' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[60, 232, 0, 0, 0, 0, 0, 0, 0, 0, 74, 219, 0],
            &[26, 255, 12, 0, 0, 0, 0, 0, 0, 0, 108, 186, 0],
            &[1, 246, 45, 0, 0, 0, 0, 0, 0, 0, 142, 152, 0],
            &[0, 212, 79, 0, 0, 13, 40, 0, 0, 0, 176, 118, 0],
            &[0, 178, 113, 0, 0, 113, 254, 17, 0, 0, 211, 84, 0],
            &[0, 143, 147, 0, 0, 172, 203, 75, 0, 0, 245, 50, 0],
            &[0, 108, 180, 0, 0, 227, 107, 134, 0, 24, 255, 16, 0],
            &[0, 73, 214, 0, 34, 207, 43, 193, 0, 58, 238, 0, 0],
            &[0, 39, 247, 1, 93, 150, 1, 230, 5, 92, 204, 0, 0],
            &[0, 6, 253, 25, 152, 92, 0, 182, 56, 121, 170, 0, 0],
            &[0, 0, 225, 55, 209, 34, 0, 123, 116, 144, 136, 0, 0],
            &[0, 0, 190, 93, 228, 0, 0, 65, 177, 167, 102, 0, 0],
            &[0, 0, 156, 167, 174, 0, 0, 10, 226, 187, 67, 0, 0],
            &[0, 0, 121, 235, 116, 0, 0, 0, 203, 233, 33, 0, 0],
            &[0, 0, 86, 255, 58, 0, 0, 0, 145, 251, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'X' / 0x58
        'X' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 94, 230, 13, 0, 0, 0, 0, 0, 61, 240, 27, 0],
            &[0, 0, 194, 138, 0, 0, 0, 0, 3, 210, 113, 0, 0],
            &[0, 0, 43, 244, 38, 0, 0, 0, 111, 207, 3, 0, 0],
            &[0, 0, 0, 137, 182, 0, 0, 23, 239, 56, 0, 0, 0],
            &[0, 0, 0, 11, 225, 76, 0, 162, 152, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 219, 67, 232, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 179, 246, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 121, 253, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 29, 239, 160, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 175, 141, 6, 217, 86, 0, 0, 0, 0],
            &[0, 0, 0, 78, 229, 13, 0, 71, 229, 14, 0, 0, 0],
            &[0, 0, 11, 225, 87, 0, 0, 0, 173, 145, 0, 0, 0],
            &[0, 0, 140, 188, 0, 0, 0, 0, 30, 242, 49, 0, 0],
            &[0, 46, 245, 40, 0, 0, 0, 0, 0, 123, 201, 2, 0],
            &[1, 199, 135, 0, 0, 0, 0, 0, 0, 7, 218, 105, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Y' / 0x59
        'Y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[2, 210, 126, 0, 0, 0, 0, 0, 0, 2, 214, 116, 0],
            &[0, 80, 239, 15, 0, 0, 0, 0, 0, 88, 230, 9, 0],
            &[0, 0, 201, 126, 0, 0, 0, 0, 1, 212, 106, 0, 0],
            &[0, 0, 68, 239, 15, 0, 0, 0, 85, 223, 5, 0, 0],
            &[0, 0, 0, 190, 126, 0, 0, 1, 210, 95, 0, 0, 0],
            &[0, 0, 0, 56, 238, 15, 0, 82, 214, 3, 0, 0, 0],
            &[0, 0, 0, 0, 177, 126, 1, 208, 84, 0, 0, 0, 0],
            &[0, 0, 0, 0, 45, 237, 95, 205, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 165, 251, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 235, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Z' / 0x5a
        'Z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 254, 254, 254, 254, 254, 254, 254, 94, 0, 0],
            &[0, 0, 5, 19, 19, 19, 19, 19, 65, 246, 27, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 179, 140, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 58, 241, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 193, 127, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 235, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 205, 114, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 86, 227, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 217, 102, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 219, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 227, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 114, 208, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 235, 76, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 210, 29, 28, 28, 28, 28, 28, 14, 0, 0],
            &[0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 127, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '[' / 0x5b
        '[' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 249, 254, 254, 254, 125, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 52, 3, 3, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 49, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 251, 196, 181, 181, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 81, 81, 81, 40, 0, 0, 0, 0],
        ],
        // letter: '\' / 0x5c
        '\\' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 6, 236, 61, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 147, 158, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 243, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 209, 97, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 112, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 249, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 173, 132, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 76, 227, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 231, 70, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 138, 167, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 41, 247, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 199, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 102, 203, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 15, 246, 44, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 164, 141, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ']' / 0x5d
        ']' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 25, 25, 25, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 207, 237, 237, 248, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 177, 203, 203, 233, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 54, 54, 54, 32, 0, 0, 0, 0],
        ],
        // letter: '^' / 0x5e
        '^' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 234, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 222, 181, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 104, 165, 28, 230, 14, 0, 0, 0, 0],
            &[0, 0, 0, 7, 224, 40, 0, 148, 127, 0, 0, 0, 0],
            &[0, 0, 0, 107, 167, 0, 0, 27, 233, 17, 0, 0, 0],
            &[0, 0, 7, 226, 42, 0, 0, 0, 147, 132, 0, 0, 0],
            &[0, 0, 109, 169, 0, 0, 0, 0, 27, 235, 20, 0, 0],
            &[0, 8, 228, 44, 0, 0, 0, 0, 0, 147, 138, 0, 0],
            &[0, 111, 171, 0, 0, 0, 0, 0, 0, 26, 237, 23, 0],
            &[0, 51, 24, 0, 0, 0, 0, 0, 0, 0, 51, 25, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '_' / 0x5f
        '_' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 52],
            &[
                197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 118,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '`' / 0x60
        '`' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 14, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 119, 237, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 151, 190, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 150, 147, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 63, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'a' / 0x61
        'a' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 82, 89, 59, 11, 0, 0, 0, 0],
            &[0, 0, 42, 205, 234, 191, 175, 212, 232, 85, 0, 0, 0],
            &[0, 0, 11, 81, 4, 0, 0, 0, 97, 250, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 209, 101, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 171, 125, 0, 0],
            &[0, 0, 0, 0, 32, 88, 115, 130, 139, 215, 127, 0, 0],
            &[0, 0, 33, 191, 231, 162, 126, 107, 98, 195, 127, 0, 0],
            &[0, 6, 220, 143, 5, 0, 0, 0, 0, 166, 127, 0, 0],
            &[0, 59, 241, 4, 0, 0, 0, 0, 0, 200, 127, 0, 0],
            &[0, 63, 238, 0, 0, 0, 0, 0, 53, 254, 127, 0, 0],
            &[0, 14, 236, 123, 3, 0, 3, 79, 200, 180, 127, 0, 0],
            &[0, 0, 63, 218, 240, 216, 239, 173, 33, 134, 127, 0, 0],
            &[0, 0, 0, 1, 27, 47, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'b' / 0x62
        'b' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 245, 31, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 33, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 33, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 33, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 32, 0, 40, 85, 81, 22, 0, 0, 0, 0],
            &[0, 12, 255, 40, 172, 215, 172, 194, 246, 123, 0, 0, 0],
            &[0, 12, 255, 179, 111, 0, 0, 0, 52, 241, 94, 0, 0],
            &[0, 12, 255, 180, 0, 0, 0, 0, 0, 133, 216, 0, 0],
            &[0, 12, 255, 90, 0, 0, 0, 0, 0, 43, 255, 30, 0],
            &[0, 12, 255, 47, 0, 0, 0, 0, 0, 11, 255, 64, 0],
            &[0, 12, 255, 38, 0, 0, 0, 0, 0, 0, 250, 72, 0],
            &[0, 12, 255, 60, 0, 0, 0, 0, 0, 10, 255, 55, 0],
            &[0, 12, 255, 96, 0, 0, 0, 0, 0, 57, 253, 14, 0],
            &[0, 12, 255, 194, 0, 0, 0, 0, 0, 154, 185, 0, 0],
            &[0, 12, 255, 175, 127, 6, 0, 4, 111, 247, 51, 0, 0],
            &[0, 12, 249, 8, 155, 237, 215, 240, 217, 66, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 45, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'c' / 0x63
        'c' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 70, 91, 72, 25, 0, 0, 0],
            &[0, 0, 0, 19, 167, 249, 212, 188, 206, 248, 96, 0, 0],
            &[0, 0, 11, 215, 190, 34, 0, 0, 0, 18, 18, 0, 0],
            &[0, 0, 134, 215, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 226, 104, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 255, 45, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 255, 27, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 228, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 139, 214, 7, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 218, 194, 51, 0, 0, 1, 48, 49, 0, 0],
            &[0, 0, 0, 18, 156, 244, 243, 227, 250, 229, 80, 0, 0],
            &[0, 0, 0, 0, 0, 6, 34, 46, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'd' / 0x64
        'd' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 135, 164, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 0, 0, 0, 36, 81, 87, 31, 0, 140, 170, 0, 0],
            &[0, 0, 13, 175, 244, 184, 178, 229, 119, 135, 170, 0, 0],
            &[0, 0, 176, 203, 23, 0, 0, 10, 177, 194, 170, 0, 0],
            &[0, 49, 254, 38, 0, 0, 0, 0, 22, 248, 170, 0, 0],
            &[0, 120, 202, 0, 0, 0, 0, 0, 0, 192, 170, 0, 0],
            &[0, 154, 160, 0, 0, 0, 0, 0, 0, 154, 170, 0, 0],
            &[0, 169, 148, 0, 0, 0, 0, 0, 0, 143, 170, 0, 0],
            &[0, 158, 169, 0, 0, 0, 0, 0, 0, 164, 170, 0, 0],
            &[0, 120, 204, 0, 0, 0, 0, 0, 0, 200, 170, 0, 0],
            &[0, 45, 255, 50, 0, 0, 0, 0, 41, 255, 170, 0, 0],
            &[0, 0, 165, 219, 49, 0, 0, 32, 195, 158, 170, 0, 0],
            &[0, 0, 7, 150, 248, 235, 220, 223, 70, 95, 170, 0, 0],
            &[0, 0, 0, 0, 11, 45, 39, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'e' / 0x65
        'e' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 61, 89, 70, 14, 0, 0, 0, 0],
            &[0, 0, 0, 90, 232, 217, 176, 197, 240, 101, 0, 0, 0],
            &[0, 0, 87, 242, 83, 0, 0, 0, 56, 239, 74, 0, 0],
            &[0, 7, 233, 101, 0, 0, 0, 0, 0, 118, 201, 0, 0],
            &[0, 74, 242, 4, 0, 0, 0, 0, 0, 42, 254, 15, 0],
            &[0, 121, 222, 114, 114, 114, 114, 114, 114, 124, 255, 40, 0],
            &[0, 140, 220, 143, 143, 143, 143, 143, 143, 143, 143, 27, 0],
            &[0, 122, 194, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 75, 241, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 230, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 243, 124, 19, 0, 0, 3, 59, 107, 0, 0],
            &[0, 0, 0, 52, 184, 249, 229, 226, 252, 215, 113, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'f' / 0x66
        'f' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 3, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 204, 253, 252, 247, 163, 0],
            &[0, 0, 0, 0, 0, 43, 250, 124, 17, 0, 14, 33, 0],
            &[0, 0, 0, 0, 0, 127, 203, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 152, 164, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 22, 33, 170, 174, 44, 44, 44, 44, 6, 0],
            &[0, 71, 224, 224, 224, 243, 243, 224, 224, 224, 224, 33, 0],
            &[0, 0, 0, 0, 0, 154, 157, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 157, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 157, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 157, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 157, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 157, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 157, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 157, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 157, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 157, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'g' / 0x67
        'g' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 87, 82, 31, 0, 14, 29, 0, 0],
            &[0, 0, 11, 175, 243, 183, 177, 226, 137, 100, 170, 0, 0],
            &[0, 0, 168, 197, 20, 0, 0, 4, 150, 216, 170, 0, 0],
            &[0, 43, 255, 47, 0, 0, 0, 0, 6, 229, 170, 0, 0],
            &[0, 118, 206, 0, 0, 0, 0, 0, 0, 160, 170, 0, 0],
            &[0, 157, 170, 0, 0, 0, 0, 0, 0, 129, 170, 0, 0],
            &[0, 169, 150, 0, 0, 0, 0, 0, 0, 119, 170, 0, 0],
            &[0, 158, 172, 0, 0, 0, 0, 0, 0, 137, 170, 0, 0],
            &[0, 119, 210, 0, 0, 0, 0, 0, 0, 174, 170, 0, 0],
            &[0, 42, 255, 57, 0, 0, 0, 0, 23, 248, 170, 0, 0],
            &[0, 0, 158, 215, 45, 0, 0, 28, 190, 175, 170, 0, 0],
            &[0, 0, 5, 144, 246, 225, 219, 227, 89, 123, 170, 0, 0],
            &[0, 0, 0, 0, 10, 44, 36, 3, 0, 129, 168, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 160, 147, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 14, 235, 87, 0, 0],
            &[0, 0, 0, 0, 18, 37, 26, 4, 0, 0, 0, 0, 0],
        ],
        // letter: 'h' / 0x68
        'h' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 245, 42, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 43, 0, 40, 88, 75, 26, 0, 0, 0, 0],
            &[0, 12, 255, 48, 156, 218, 174, 199, 248, 131, 0, 0, 0],
            &[0, 12, 255, 168, 107, 1, 0, 0, 66, 252, 87, 0, 0],
            &[0, 12, 255, 171, 0, 0, 0, 0, 0, 169, 167, 0, 0],
            &[0, 12, 255, 84, 0, 0, 0, 0, 0, 126, 194, 0, 0],
            &[0, 12, 255, 52, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'i' / 0x69
        'i' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 150, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 114, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 80, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 44, 44, 44, 44, 12, 0, 0, 0, 0, 0],
            &[0, 0, 136, 192, 204, 217, 253, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'j' / 0x6a
        'j' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 82, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 188, 254, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 35, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 44, 44, 44, 44, 40, 0, 0, 0, 0],
            &[0, 0, 0, 21, 184, 196, 209, 230, 234, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 83, 229, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 128, 193, 0, 0, 0, 0],
            &[0, 0, 0, 4, 27, 28, 1, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'k' / 0x6b
        'k' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 102, 196, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 0, 9, 44, 12, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 8, 188, 180, 5, 0, 0],
            &[0, 106, 204, 0, 0, 0, 5, 179, 190, 8, 0, 0, 0],
            &[0, 106, 204, 0, 0, 3, 169, 200, 12, 0, 0, 0, 0],
            &[0, 106, 203, 0, 1, 159, 209, 17, 0, 0, 0, 0, 0],
            &[0, 106, 198, 0, 147, 255, 41, 0, 0, 0, 0, 0, 0],
            &[0, 106, 191, 130, 200, 176, 176, 0, 0, 0, 0, 0, 0],
            &[0, 106, 240, 193, 13, 17, 226, 108, 0, 0, 0, 0, 0],
            &[0, 106, 224, 10, 0, 0, 61, 245, 47, 0, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 130, 215, 11, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 3, 198, 158, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 0, 32, 240, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'l' / 0x6c
        'l' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 183, 236, 245, 245, 245, 70, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 15, 239, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'm' / 0x6d
        'm' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 5, 24, 83, 41, 0, 1, 69, 81, 9, 0, 0],
            &[0, 189, 83, 215, 207, 251, 59, 164, 207, 236, 198, 1, 0],
            &[0, 189, 217, 28, 0, 159, 214, 158, 0, 27, 248, 52, 0],
            &[0, 189, 163, 0, 0, 82, 255, 44, 0, 0, 206, 91, 0],
            &[0, 189, 125, 0, 0, 59, 248, 2, 0, 0, 185, 102, 0],
            &[0, 189, 102, 0, 0, 52, 235, 0, 0, 0, 181, 103, 0],
            &[0, 189, 95, 0, 0, 52, 232, 0, 0, 0, 181, 103, 0],
            &[0, 189, 95, 0, 0, 52, 232, 0, 0, 0, 181, 103, 0],
            &[0, 189, 95, 0, 0, 52, 232, 0, 0, 0, 181, 103, 0],
            &[0, 189, 95, 0, 0, 52, 232, 0, 0, 0, 181, 103, 0],
            &[0, 189, 95, 0, 0, 52, 232, 0, 0, 0, 181, 103, 0],
            &[0, 189, 95, 0, 0, 52, 232, 0, 0, 0, 181, 103, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'n' / 0x6e
        'n' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 41, 0, 0, 41, 88, 77, 30, 0, 0, 0, 0],
            &[0, 12, 252, 14, 166, 221, 178, 196, 249, 141, 1, 0, 0],
            &[0, 12, 255, 173, 105, 0, 0, 0, 55, 249, 94, 0, 0],
            &[0, 12, 255, 173, 0, 0, 0, 0, 0, 162, 172, 0, 0],
            &[0, 12, 255, 87, 0, 0, 0, 0, 0, 123, 195, 0, 0],
            &[0, 12, 255, 57, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'o' / 0x6f
        'o' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 72, 89, 60, 4, 0, 0, 0, 0],
            &[0, 0, 3, 135, 240, 191, 174, 213, 226, 70, 0, 0, 0],
            &[0, 0, 143, 213, 32, 0, 0, 0, 89, 246, 53, 0, 0],
            &[0, 35, 252, 44, 0, 0, 0, 0, 0, 141, 192, 0, 0],
            &[0, 116, 200, 0, 0, 0, 0, 0, 0, 46, 253, 20, 0],
            &[0, 152, 154, 0, 0, 0, 0, 0, 0, 3, 254, 56, 0],
            &[0, 168, 143, 0, 0, 0, 0, 0, 0, 0, 245, 71, 0],
            &[0, 152, 161, 0, 0, 0, 0, 0, 0, 8, 255, 54, 0],
            &[0, 105, 215, 0, 0, 0, 0, 0, 0, 62, 250, 10, 0],
            &[0, 22, 248, 69, 0, 0, 0, 0, 0, 172, 169, 0, 0],
            &[0, 0, 111, 236, 72, 0, 0, 13, 140, 237, 32, 0, 0],
            &[0, 0, 0, 95, 225, 236, 219, 246, 189, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 47, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'p' / 0x70
        'p' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 42, 0, 0, 41, 88, 83, 26, 0, 0, 0, 0],
            &[0, 12, 254, 10, 156, 214, 173, 204, 247, 132, 0, 0, 0],
            &[0, 12, 255, 156, 103, 0, 0, 0, 55, 235, 102, 0, 0],
            &[0, 12, 255, 186, 0, 0, 0, 0, 0, 121, 221, 0, 0],
            &[0, 12, 255, 91, 0, 0, 0, 0, 0, 37, 255, 32, 0],
            &[0, 12, 255, 58, 0, 0, 0, 0, 0, 10, 255, 64, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 250, 70, 0],
            &[0, 12, 255, 67, 0, 0, 0, 0, 0, 9, 255, 52, 0],
            &[0, 12, 255, 105, 0, 0, 0, 0, 0, 53, 251, 15, 0],
            &[0, 12, 255, 205, 0, 0, 0, 0, 0, 146, 187, 0, 0],
            &[0, 12, 255, 156, 139, 8, 0, 3, 101, 246, 53, 0, 0],
            &[0, 12, 255, 40, 127, 237, 215, 238, 217, 69, 0, 0, 0],
            &[0, 12, 255, 38, 0, 11, 46, 30, 1, 0, 0, 0, 0],
            &[0, 12, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'q' / 0x71
        'q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 87, 79, 25, 0, 14, 29, 0, 0],
            &[0, 0, 10, 173, 240, 179, 176, 227, 119, 99, 170, 0, 0],
            &[0, 0, 168, 193, 16, 0, 0, 9, 173, 198, 170, 0, 0],
            &[0, 44, 253, 33, 0, 0, 0, 0, 27, 251, 170, 0, 0],
            &[0, 118, 200, 0, 0, 0, 0, 0, 0, 191, 170, 0, 0],
            &[0, 157, 159, 0, 0, 0, 0, 0, 0, 161, 170, 0, 0],
            &[0, 169, 148, 0, 0, 0, 0, 0, 0, 144, 170, 0, 0],
            &[0, 158, 170, 0, 0, 0, 0, 0, 0, 165, 170, 0, 0],
            &[0, 119, 208, 0, 0, 0, 0, 0, 0, 203, 170, 0, 0],
            &[0, 43, 255, 51, 0, 0, 0, 0, 50, 254, 170, 0, 0],
            &[0, 0, 162, 210, 37, 0, 0, 41, 201, 164, 170, 0, 0],
            &[0, 0, 7, 150, 247, 221, 223, 216, 61, 137, 170, 0, 0],
            &[0, 0, 0, 0, 12, 45, 31, 2, 0, 140, 170, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 6, 0, 0],
        ],
        // letter: 'r' / 0x72
        'r' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 41, 44, 44, 44, 12, 0, 27, 79, 77, 32, 0],
            &[0, 0, 172, 198, 213, 247, 80, 106, 240, 200, 206, 207, 0],
            &[0, 0, 0, 0, 0, 191, 147, 201, 26, 0, 0, 5, 0],
            &[0, 0, 0, 0, 0, 191, 246, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 183, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 136, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 120, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 194, 124, 1, 0, 0, 0, 0, 0],
            &[0, 66, 231, 244, 255, 255, 255, 253, 242, 229, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 's' / 0x73
        's' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 71, 92, 77, 43, 0, 0, 0, 0],
            &[0, 0, 0, 111, 245, 204, 177, 191, 233, 199, 0, 0, 0],
            &[0, 0, 39, 252, 60, 0, 0, 0, 2, 32, 0, 0, 0],
            &[0, 0, 82, 232, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 40, 254, 78, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 105, 244, 177, 75, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 118, 221, 224, 117, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 71, 220, 190, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 44, 255, 44, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 17, 255, 49, 0, 0],
            &[0, 0, 93, 73, 5, 0, 0, 15, 158, 221, 4, 0, 0],
            &[0, 0, 92, 215, 249, 222, 226, 248, 188, 37, 0, 0, 0],
            &[0, 0, 0, 0, 13, 40, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 't' / 0x74
        't' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 254, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 2, 17, 33, 85, 255, 49, 44, 44, 44, 22, 0, 0],
            &[0, 119, 224, 224, 230, 255, 225, 224, 224, 224, 112, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 192, 180, 11, 0, 0, 7, 0, 0],
            &[0, 0, 0, 0, 0, 32, 198, 245, 215, 232, 138, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 25, 48, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'u' / 0x75
        'u' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 44, 2, 0, 0, 0, 0, 0, 24, 29, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 17, 0, 0, 0, 0, 0, 152, 170, 0, 0],
            &[0, 24, 255, 35, 0, 0, 0, 0, 0, 189, 170, 0, 0],
            &[0, 0, 244, 87, 0, 0, 0, 0, 33, 249, 170, 0, 0],
            &[0, 0, 143, 218, 40, 0, 0, 41, 197, 155, 170, 0, 0],
            &[0, 0, 14, 173, 253, 227, 226, 213, 60, 92, 170, 0, 0],
            &[0, 0, 0, 0, 18, 48, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'v' / 0x76
        'v' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 28, 0, 0, 0, 0, 0, 0, 3, 44, 8, 0],
            &[0, 97, 222, 1, 0, 0, 0, 0, 0, 74, 240, 9, 0],
            &[0, 12, 244, 61, 0, 0, 0, 0, 0, 167, 152, 0, 0],
            &[0, 0, 160, 155, 0, 0, 0, 0, 14, 246, 55, 0, 0],
            &[0, 0, 63, 240, 8, 0, 0, 0, 98, 213, 0, 0, 0],
            &[0, 0, 1, 221, 87, 0, 0, 0, 191, 115, 0, 0, 0],
            &[0, 0, 0, 125, 181, 0, 0, 30, 249, 22, 0, 0, 0],
            &[0, 0, 0, 31, 249, 23, 0, 122, 176, 0, 0, 0, 0],
            &[0, 0, 0, 0, 188, 113, 0, 215, 78, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 207, 52, 232, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 241, 183, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 255, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'w' / 0x77
        'w' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[23, 26, 0, 0, 0, 0, 0, 0, 0, 0, 0, 41, 7],
            &[107, 183, 0, 0, 0, 0, 0, 0, 0, 0, 17, 253, 14],
            &[55, 236, 0, 0, 0, 8, 29, 0, 0, 0, 68, 218, 0],
            &[8, 250, 34, 0, 0, 121, 255, 50, 0, 0, 119, 168, 0],
            &[0, 206, 87, 0, 0, 198, 192, 126, 0, 0, 170, 118, 0],
            &[0, 153, 141, 0, 22, 241, 74, 203, 0, 0, 221, 68, 0],
            &[0, 101, 195, 0, 98, 177, 7, 244, 25, 17, 254, 18, 0],
            &[0, 49, 245, 2, 175, 103, 0, 179, 100, 67, 224, 0, 0],
            &[0, 5, 247, 41, 242, 28, 0, 103, 174, 110, 174, 0, 0],
            &[0, 0, 200, 129, 208, 0, 0, 28, 238, 146, 124, 0, 0],
            &[0, 0, 147, 215, 132, 0, 0, 0, 207, 216, 74, 0, 0],
            &[0, 0, 95, 255, 56, 0, 0, 0, 132, 255, 23, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'x' / 0x78
        'x' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 15, 43, 1, 0, 0, 0, 0, 0, 18, 41, 0, 0],
            &[0, 13, 220, 118, 0, 0, 0, 0, 8, 209, 130, 0, 0],
            &[0, 0, 52, 244, 50, 0, 0, 0, 147, 193, 3, 0, 0],
            &[0, 0, 0, 115, 216, 10, 0, 78, 235, 28, 0, 0, 0],
            &[0, 0, 0, 1, 183, 154, 27, 235, 76, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 230, 219, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 161, 255, 55, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 233, 129, 218, 13, 0, 0, 0, 0],
            &[0, 0, 0, 30, 237, 75, 0, 175, 165, 0, 0, 0, 0],
            &[0, 0, 4, 197, 144, 0, 0, 18, 227, 98, 0, 0, 0],
            &[0, 0, 134, 207, 7, 0, 0, 0, 63, 242, 42, 0, 0],
            &[0, 69, 243, 40, 0, 0, 0, 0, 0, 131, 211, 9, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'y' / 0x79
        'y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 23, 29, 0, 0, 0, 0, 0, 0, 2, 44, 8, 0],
            &[0, 78, 228, 3, 0, 0, 0, 0, 0, 68, 241, 9, 0],
            &[0, 3, 228, 75, 0, 0, 0, 0, 0, 160, 154, 0, 0],
            &[0, 0, 130, 174, 0, 0, 0, 0, 9, 242, 58, 0, 0],
            &[0, 0, 31, 248, 23, 0, 0, 0, 87, 217, 0, 0, 0],
            &[0, 0, 0, 183, 117, 0, 0, 0, 179, 121, 0, 0, 0],
            &[0, 0, 0, 81, 216, 0, 0, 20, 248, 27, 0, 0, 0],
            &[0, 0, 0, 4, 231, 60, 0, 107, 184, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 158, 0, 198, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 241, 41, 238, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 186, 195, 150, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 255, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 86, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 182, 117, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 243, 19, 0, 0, 0, 0, 0, 0],
            &[0, 2, 33, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'z' / 0x7a
        'z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 44, 44, 44, 44, 44, 44, 44, 16, 0, 0],
            &[0, 0, 82, 208, 208, 208, 208, 208, 214, 255, 88, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 140, 202, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 71, 240, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 232, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 183, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 114, 216, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 242, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 217, 109, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 158, 177, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 88, 234, 37, 18, 18, 18, 18, 18, 7, 0, 0],
            &[0, 0, 211, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '{' / 0x7b
        '{' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 43, 175, 234, 112, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 234, 162, 43, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 247, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 217, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 216, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 216, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 95, 212, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 148, 175, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 83, 157, 216, 44, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 143, 223, 153, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 186, 150, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 103, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 216, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 216, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 216, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 230, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 253, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 105, 244, 210, 84, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 10, 59, 34, 0, 0, 0],
        ],
        // letter: '|' / 0x7c
        '|' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 202, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 8, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '}' / 0x7d
        '}' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 208, 214, 129, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 64, 217, 143, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 91, 226, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 243, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 243, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 243, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 60, 247, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 253, 43, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 131, 218, 117, 36, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 52, 197, 208, 70, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 237, 95, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 252, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 243, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 243, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 243, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 82, 235, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 180, 184, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 165, 235, 213, 37, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 62, 44, 1, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '~' / 0x7e
        '~' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 149, 167, 63, 0, 0, 0, 15, 160, 0, 0],
            &[0, 0, 192, 100, 76, 207, 107, 0, 0, 66, 173, 0, 0],
            &[0, 30, 204, 0, 0, 18, 201, 148, 57, 190, 78, 0, 0],
            &[0, 55, 124, 0, 0, 0, 10, 127, 179, 101, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ' ' / 0xa0
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¡' / 0xa1
        '¡' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 146, 255, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 182, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 65, 214, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 70, 218, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 74, 222, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 78, 227, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 83, 231, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 235, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 92, 240, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 96, 244, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 100, 249, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 105, 253, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 96, 226, 1, 0, 0, 0, 0, 0],
        ],
        // letter: '¢' / 0xa2
        '¢' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 130, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 173, 180, 91, 44, 0, 0, 0],
            &[0, 0, 0, 25, 186, 243, 178, 157, 179, 237, 59, 0, 0],
            &[0, 0, 10, 218, 172, 14, 0, 0, 0, 3, 2, 0, 0],
            &[0, 0, 123, 216, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 207, 115, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 247, 63, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 255, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 245, 68, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 198, 129, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 108, 234, 19, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 195, 212, 68, 6, 0, 17, 76, 31, 0, 0],
            &[0, 0, 0, 10, 131, 230, 254, 248, 253, 202, 37, 0, 0],
            &[0, 0, 0, 0, 0, 0, 138, 130, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 130, 116, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 27, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '£' / 0xa3
        '£' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 27, 27, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 232, 242, 238, 242, 139, 0, 0],
            &[0, 0, 0, 0, 64, 248, 77, 0, 0, 37, 94, 0, 0],
            &[0, 0, 0, 0, 164, 159, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 202, 113, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 96, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 96, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 96, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 187, 213, 248, 229, 213, 213, 213, 24, 0, 0, 0],
            &[0, 0, 25, 28, 215, 114, 28, 28, 28, 3, 0, 0, 0],
            &[0, 0, 0, 0, 210, 96, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 95, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 227, 71, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 52, 240, 16, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 72, 219, 113, 28, 28, 28, 28, 28, 28, 4, 0],
            &[0, 22, 255, 255, 255, 255, 255, 255, 255, 255, 255, 44, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¤' / 0xa4
        '¤' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 22, 0, 0, 0, 0, 0, 0, 0, 21, 0, 0],
            &[0, 69, 223, 32, 61, 148, 171, 128, 21, 97, 216, 4, 0],
            &[0, 0, 115, 241, 204, 97, 71, 122, 225, 230, 38, 0, 0],
            &[0, 0, 81, 193, 5, 0, 0, 0, 44, 222, 11, 0, 0],
            &[0, 0, 183, 67, 0, 0, 0, 0, 0, 161, 89, 0, 0],
            &[0, 0, 214, 30, 0, 0, 0, 0, 0, 126, 122, 0, 0],
            &[0, 0, 180, 68, 0, 0, 0, 0, 0, 166, 89, 0, 0],
            &[0, 0, 76, 197, 7, 0, 0, 0, 54, 224, 11, 0, 0],
            &[0, 0, 101, 241, 211, 105, 81, 135, 234, 226, 32, 0, 0],
            &[0, 59, 225, 36, 57, 144, 167, 124, 21, 110, 212, 3, 0],
            &[0, 0, 21, 0, 0, 0, 0, 0, 0, 0, 25, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¥' / 0xa5
        '¥' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[9, 225, 95, 0, 0, 0, 0, 0, 0, 0, 191, 137, 0],
            &[0, 92, 227, 9, 0, 0, 0, 0, 0, 78, 234, 14, 0],
            &[0, 1, 203, 121, 0, 0, 0, 0, 4, 215, 103, 0, 0],
            &[0, 0, 61, 240, 20, 0, 0, 0, 105, 210, 3, 0, 0],
            &[0, 0, 0, 173, 147, 0, 0, 13, 233, 67, 0, 0, 0],
            &[0, 0, 0, 36, 244, 37, 0, 133, 178, 0, 0, 0, 0],
            &[0, 0, 0, 0, 142, 173, 28, 240, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 236, 200, 142, 0, 0, 0, 0, 0],
            &[0, 0, 153, 236, 236, 247, 255, 239, 236, 236, 63, 0, 0],
            &[0, 0, 0, 0, 0, 73, 227, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 227, 0, 0, 0, 0, 0, 0],
            &[0, 0, 153, 236, 236, 242, 253, 236, 236, 236, 63, 0, 0],
            &[0, 0, 0, 0, 0, 73, 227, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 227, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 227, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¦' / 0xa6
        '¦' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 205, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 205, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 205, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 205, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 205, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 205, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 205, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 50, 179, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 145, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 205, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 205, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 205, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 205, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 205, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 34, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '§' / 0xa7
        '§' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 150, 223, 235, 234, 206, 132, 1, 0, 0],
            &[0, 0, 0, 209, 138, 16, 0, 4, 42, 92, 0, 0, 0],
            &[0, 0, 32, 253, 7, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 241, 100, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 230, 187, 85, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 202, 152, 190, 230, 128, 10, 0, 0, 0],
            &[0, 0, 11, 224, 46, 0, 0, 30, 184, 199, 2, 0, 0],
            &[0, 0, 57, 204, 0, 0, 0, 0, 5, 239, 56, 0, 0],
            &[0, 0, 35, 248, 27, 0, 0, 0, 0, 212, 55, 0, 0],
            &[0, 0, 0, 139, 233, 105, 13, 0, 91, 214, 3, 0, 0],
            &[0, 0, 0, 0, 58, 167, 237, 191, 217, 32, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 120, 241, 120, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 56, 252, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 5, 251, 33, 0, 0],
            &[0, 0, 76, 67, 7, 0, 0, 13, 146, 204, 0, 0, 0],
            &[0, 0, 69, 193, 237, 221, 221, 232, 154, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 12, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¨' / 0xa8
        '¨' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 24, 0, 0, 0, 36, 1, 0, 0, 0],
            &[0, 0, 0, 172, 215, 0, 0, 55, 255, 75, 0, 0, 0],
            &[0, 0, 0, 86, 115, 0, 0, 18, 154, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '©' / 0xa9
        '©' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 57, 165, 192, 193, 196, 135, 24, 0, 0, 0],
            &[0, 0, 119, 173, 39, 0, 0, 3, 81, 205, 51, 0, 0],
            &[0, 75, 167, 0, 29, 161, 193, 188, 52, 30, 205, 10, 0],
            &[0, 190, 24, 9, 209, 58, 0, 13, 12, 0, 119, 95, 0],
            &[4, 198, 0, 86, 138, 0, 0, 0, 0, 0, 44, 156, 0],
            &[22, 181, 0, 116, 102, 0, 0, 0, 0, 0, 22, 176, 0],
            &[3, 203, 0, 88, 137, 0, 0, 0, 0, 0, 46, 152, 0],
            &[0, 188, 32, 10, 210, 71, 5, 24, 14, 0, 126, 83, 0],
            &[0, 69, 182, 3, 24, 143, 183, 163, 25, 40, 193, 4, 0],
            &[0, 0, 110, 188, 57, 0, 0, 12, 101, 194, 32, 0, 0],
            &[0, 0, 0, 45, 148, 188, 187, 179, 102, 9, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ª' / 0xaa
        'ª' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 32, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 200, 184, 167, 216, 81, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 0, 0, 31, 216, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 45, 74, 84, 234, 0, 0, 0, 0],
            &[0, 0, 0, 29, 202, 147, 106, 92, 234, 0, 0, 0, 0],
            &[0, 0, 0, 131, 119, 0, 0, 10, 240, 0, 0, 0, 0],
            &[0, 0, 0, 114, 159, 12, 8, 142, 237, 0, 0, 0, 0],
            &[0, 0, 0, 9, 151, 204, 182, 61, 148, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '«' / 0xab
        '«' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 165, 32, 0, 17, 173, 8, 0, 0],
            &[0, 0, 0, 0, 148, 181, 1, 3, 187, 134, 0, 0, 0],
            &[0, 0, 0, 100, 221, 17, 0, 144, 188, 3, 0, 0, 0],
            &[0, 0, 59, 240, 48, 0, 97, 226, 21, 0, 0, 0, 0],
            &[0, 0, 171, 178, 0, 0, 217, 133, 0, 0, 0, 0, 0],
            &[0, 0, 22, 225, 95, 0, 48, 239, 57, 0, 0, 0, 0],
            &[0, 0, 0, 48, 235, 47, 0, 83, 227, 22, 0, 0, 0],
            &[0, 0, 0, 0, 83, 218, 15, 0, 126, 190, 3, 0, 0],
            &[0, 0, 0, 0, 0, 81, 14, 0, 1, 94, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¬' / 0xac
        '¬' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 149, 234, 234, 234, 234, 234, 234, 234, 234, 151, 0, 0],
            &[0, 14, 23, 23, 23, 23, 23, 23, 23, 107, 165, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 165, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 165, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 92, 165, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 24, 44, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '®' / 0xae
        '®' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 57, 165, 192, 193, 196, 135, 24, 0, 0, 0],
            &[0, 0, 119, 173, 42, 6, 2, 3, 81, 205, 51, 0, 0],
            &[0, 75, 167, 0, 150, 189, 189, 195, 37, 30, 205, 10, 0],
            &[0, 190, 24, 0, 150, 42, 0, 60, 156, 0, 119, 95, 0],
            &[4, 198, 0, 0, 150, 42, 0, 61, 151, 0, 44, 156, 0],
            &[22, 181, 0, 0, 150, 175, 177, 184, 27, 0, 22, 176, 0],
            &[3, 203, 0, 0, 150, 52, 52, 177, 0, 0, 46, 152, 0],
            &[0, 188, 32, 0, 150, 42, 0, 145, 82, 0, 126, 83, 0],
            &[0, 69, 182, 3, 123, 34, 0, 16, 172, 44, 193, 4, 0],
            &[0, 0, 110, 188, 57, 0, 0, 12, 101, 194, 32, 0, 0],
            &[0, 0, 0, 45, 148, 188, 187, 179, 102, 9, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¯' / 0xaf
        '¯' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 39, 39, 39, 39, 39, 7, 0, 0, 0],
            &[0, 0, 0, 130, 218, 218, 218, 218, 218, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '°' / 0xb0
        '°' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 21, 43, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 156, 212, 181, 219, 73, 0, 0, 0, 0],
            &[0, 0, 0, 112, 159, 2, 0, 27, 221, 19, 0, 0, 0],
            &[0, 0, 0, 184, 52, 0, 0, 0, 149, 81, 0, 0, 0],
            &[0, 0, 0, 174, 68, 0, 0, 0, 162, 72, 0, 0, 0],
            &[0, 0, 0, 76, 205, 31, 0, 76, 218, 8, 0, 0, 0],
            &[0, 0, 0, 0, 91, 208, 224, 181, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '±' / 0xb1
        '±' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 61, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 200, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 200, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 200, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 200, 0, 0, 0, 0, 0, 0],
            &[0, 112, 240, 240, 240, 243, 252, 240, 240, 240, 240, 6, 0],
            &[0, 5, 12, 12, 12, 67, 203, 12, 12, 12, 12, 0, 0],
            &[0, 0, 0, 0, 0, 57, 200, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 200, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 200, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 79, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0],
            &[0, 108, 255, 255, 255, 255, 255, 255, 255, 255, 255, 6, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '²' / 0xb2
        '²' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 121, 201, 220, 181, 40, 0, 0, 0, 0],
            &[0, 0, 0, 54, 133, 21, 0, 92, 215, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 246, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 45, 217, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 203, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 187, 109, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 184, 112, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 184, 110, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 157, 234, 168, 168, 168, 168, 61, 0, 0, 0],
            &[0, 0, 0, 48, 67, 67, 67, 67, 67, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '³' / 0xb3
        '³' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 133, 197, 215, 168, 32, 0, 0, 0, 0],
            &[0, 0, 0, 75, 114, 25, 4, 116, 196, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 22, 242, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 123, 172, 0, 0, 0, 0],
            &[0, 0, 0, 0, 97, 210, 237, 220, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 102, 216, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 216, 50, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 0, 21, 240, 21, 0, 0, 0],
            &[0, 0, 0, 166, 179, 134, 140, 219, 116, 0, 0, 0, 0],
            &[0, 0, 0, 1, 53, 89, 79, 29, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '´' / 0xb4
        '´' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 22, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 122, 230, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 235, 50, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 213, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 31, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'µ' / 0xb5
        'µ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 44, 2, 0, 0, 0, 0, 0, 24, 29, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 68, 255, 30, 0, 0, 0, 0, 0, 152, 170, 0, 0],
            &[0, 63, 255, 52, 0, 0, 0, 0, 0, 189, 170, 0, 0],
            &[0, 39, 255, 103, 0, 0, 0, 0, 33, 249, 170, 0, 0],
            &[0, 38, 255, 235, 40, 0, 0, 41, 197, 155, 170, 0, 0],
            &[0, 38, 255, 190, 253, 227, 226, 213, 60, 92, 170, 0, 0],
            &[0, 38, 255, 16, 18, 48, 34, 0, 0, 0, 0, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¶' / 0xb6
        '¶' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 157, 226, 245, 245, 224, 220, 223, 179, 0, 0],
            &[0, 5, 211, 255, 255, 255, 255, 46, 0, 33, 187, 0, 0],
            &[0, 82, 255, 255, 255, 255, 255, 46, 0, 33, 187, 0, 0],
            &[0, 140, 255, 255, 255, 255, 255, 46, 0, 33, 187, 0, 0],
            &[0, 161, 255, 255, 255, 255, 255, 46, 0, 33, 187, 0, 0],
            &[0, 155, 255, 255, 255, 255, 255, 46, 0, 33, 187, 0, 0],
            &[0, 124, 255, 255, 255, 255, 255, 46, 0, 33, 187, 0, 0],
            &[0, 49, 255, 255, 255, 255, 255, 46, 0, 33, 187, 0, 0],
            &[0, 0, 152, 255, 255, 255, 255, 46, 0, 33, 187, 0, 0],
            &[0, 0, 1, 80, 140, 138, 209, 46, 0, 33, 187, 0, 0],
            &[0, 0, 0, 0, 0, 0, 173, 46, 0, 33, 187, 0, 0],
            &[0, 0, 0, 0, 0, 0, 173, 46, 0, 33, 187, 0, 0],
            &[0, 0, 0, 0, 0, 0, 173, 46, 0, 33, 187, 0, 0],
            &[0, 0, 0, 0, 0, 0, 173, 46, 0, 33, 187, 0, 0],
            &[0, 0, 0, 0, 0, 0, 173, 46, 0, 33, 187, 0, 0],
            &[0, 0, 0, 0, 0, 0, 173, 46, 0, 33, 187, 0, 0],
            &[0, 0, 0, 0, 0, 0, 173, 46, 0, 33, 187, 0, 0],
            &[0, 0, 0, 0, 0, 0, 173, 46, 0, 33, 187, 0, 0],
            &[0, 0, 0, 0, 0, 0, 115, 31, 0, 22, 124, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '·' / 0xb7
        '·' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 191, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 144, 255, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¸' / 0xb8
        '¸' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 208, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 93, 214, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 200, 78, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¹' / 0xb9
        '¹' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 95, 184, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 208, 175, 235, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 63, 65, 0, 230, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 232, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 232, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 232, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 232, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 232, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 129, 143, 255, 159, 131, 29, 0, 0, 0],
            &[0, 0, 0, 7, 67, 67, 129, 71, 67, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'º' / 0xba
        'º' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 37, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 120, 210, 177, 218, 58, 0, 0, 0, 0],
            &[0, 0, 0, 54, 203, 6, 0, 55, 217, 0, 0, 0, 0],
            &[0, 0, 0, 124, 114, 0, 0, 0, 220, 35, 0, 0, 0],
            &[0, 0, 0, 143, 96, 0, 0, 0, 199, 47, 0, 0, 0],
            &[0, 0, 0, 110, 133, 0, 0, 2, 230, 15, 0, 0, 0],
            &[0, 0, 0, 20, 219, 50, 8, 123, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 173, 196, 136, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '»' / 0xbb
        '»' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 92, 0, 0, 64, 65, 0, 0, 0, 0, 0],
            &[0, 0, 26, 228, 67, 0, 51, 230, 36, 0, 0, 0, 0],
            &[0, 0, 0, 63, 233, 36, 0, 102, 214, 14, 0, 0, 0],
            &[0, 0, 0, 0, 116, 214, 15, 0, 160, 181, 2, 0, 0],
            &[0, 0, 0, 0, 0, 214, 134, 0, 15, 245, 89, 0, 0],
            &[0, 0, 0, 0, 100, 223, 20, 0, 147, 194, 5, 0, 0],
            &[0, 0, 0, 51, 236, 46, 0, 91, 225, 22, 0, 0, 0],
            &[0, 0, 18, 222, 83, 0, 46, 235, 49, 0, 0, 0, 0],
            &[0, 0, 51, 113, 0, 0, 80, 86, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¼' / 0xbc
        '¼' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[4, 107, 233, 118, 0, 0, 0, 0, 0, 151, 131, 0, 0],
            &[133, 168, 148, 118, 0, 0, 0, 0, 34, 233, 16, 0, 0],
            &[2, 0, 126, 118, 0, 0, 0, 0, 161, 122, 0, 0, 0],
            &[0, 0, 128, 118, 0, 0, 0, 41, 231, 12, 0, 0, 0],
            &[0, 0, 128, 118, 0, 0, 0, 171, 113, 0, 0, 0, 0],
            &[0, 0, 128, 118, 0, 0, 49, 227, 8, 0, 0, 0, 0],
            &[28, 126, 255, 255, 125, 23, 180, 104, 0, 0, 0, 0, 0],
            &[18, 74, 112, 109, 74, 73, 221, 5, 0, 26, 68, 0, 0],
            &[0, 0, 0, 0, 0, 189, 95, 0, 3, 193, 208, 0, 0],
            &[0, 0, 0, 0, 66, 215, 3, 0, 125, 149, 208, 0, 0],
            &[0, 0, 0, 0, 198, 85, 0, 55, 192, 36, 208, 0, 0],
            &[0, 0, 0, 75, 207, 1, 12, 206, 33, 33, 208, 0, 0],
            &[0, 0, 1, 207, 76, 0, 129, 222, 170, 181, 240, 169, 0],
            &[0, 0, 85, 199, 0, 0, 25, 44, 44, 71, 216, 43, 0],
            &[0, 2, 214, 67, 0, 0, 0, 0, 0, 33, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '½' / 0xbd
        '½' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[4, 107, 233, 118, 0, 0, 0, 0, 0, 151, 131, 0, 0],
            &[133, 168, 148, 118, 0, 0, 0, 0, 34, 233, 16, 0, 0],
            &[2, 0, 126, 118, 0, 0, 0, 0, 161, 122, 0, 0, 0],
            &[0, 0, 128, 118, 0, 0, 0, 41, 231, 12, 0, 0, 0],
            &[0, 0, 128, 118, 0, 0, 0, 171, 113, 0, 0, 0, 0],
            &[0, 0, 128, 118, 0, 0, 49, 227, 8, 0, 0, 0, 0],
            &[28, 126, 255, 255, 125, 23, 180, 104, 0, 0, 0, 0, 0],
            &[18, 74, 112, 109, 74, 73, 221, 5, 58, 105, 70, 0, 0],
            &[0, 0, 0, 0, 0, 189, 95, 109, 183, 120, 205, 137, 0],
            &[0, 0, 0, 0, 66, 215, 3, 4, 0, 0, 55, 213, 0],
            &[0, 0, 0, 0, 198, 85, 0, 0, 0, 0, 104, 163, 0],
            &[0, 0, 0, 75, 207, 1, 0, 0, 0, 54, 216, 24, 0],
            &[0, 0, 1, 207, 76, 0, 0, 0, 52, 214, 38, 0, 0],
            &[0, 0, 85, 199, 0, 0, 0, 54, 211, 37, 0, 0, 0],
            &[0, 2, 214, 67, 0, 0, 0, 214, 245, 236, 236, 236, 12],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¾' / 0xbe
        '¾' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 30, 25, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[47, 217, 190, 222, 167, 1, 0, 0, 0, 151, 131, 0, 0],
            &[5, 23, 0, 7, 239, 37, 0, 0, 34, 233, 16, 0, 0],
            &[0, 0, 0, 57, 234, 6, 0, 0, 161, 122, 0, 0, 0],
            &[0, 106, 217, 252, 102, 0, 0, 41, 231, 12, 0, 0, 0],
            &[0, 0, 0, 21, 219, 44, 0, 171, 113, 0, 0, 0, 0],
            &[0, 0, 0, 0, 191, 77, 49, 227, 8, 0, 0, 0, 0],
            &[102, 160, 121, 164, 210, 11, 180, 104, 0, 0, 0, 0, 0],
            &[8, 72, 100, 74, 5, 57, 221, 5, 0, 26, 68, 0, 0],
            &[0, 0, 0, 0, 0, 189, 95, 0, 3, 193, 208, 0, 0],
            &[0, 0, 0, 0, 66, 215, 3, 0, 125, 149, 208, 0, 0],
            &[0, 0, 0, 0, 198, 85, 0, 55, 192, 36, 208, 0, 0],
            &[0, 0, 0, 75, 207, 1, 12, 206, 33, 33, 208, 0, 0],
            &[0, 0, 1, 207, 76, 0, 129, 222, 170, 181, 240, 169, 0],
            &[0, 0, 85, 199, 0, 0, 25, 44, 44, 71, 216, 43, 0],
            &[0, 2, 214, 67, 0, 0, 0, 0, 0, 33, 208, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¿' / 0xbf
        '¿' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 61, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 76, 255, 107, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 34, 186, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 37, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 244, 36, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 251, 9, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 190, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 207, 169, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 229, 129, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 182, 150, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 254, 42, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 37, 255, 12, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 249, 65, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 145, 224, 75, 11, 10, 45, 124, 94, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 17, 2, 0, 0, 0, 0],
        ],
        // letter: 'À' / 0xc0
        'À' => &[
            &[0, 0, 0, 25, 217, 104, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 238, 45, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 211, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 56, 61, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 134, 254, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 222, 220, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 238, 90, 210, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 158, 10, 244, 43, 0, 0, 0, 0],
            &[0, 0, 0, 3, 231, 70, 0, 167, 132, 0, 0, 0, 0],
            &[0, 0, 0, 68, 234, 4, 0, 80, 221, 0, 0, 0, 0],
            &[0, 0, 0, 157, 151, 0, 0, 7, 241, 55, 0, 0, 0],
            &[0, 0, 7, 239, 63, 0, 0, 0, 162, 144, 0, 0, 0],
            &[0, 0, 80, 255, 68, 66, 66, 66, 141, 241, 2, 0, 0],
            &[0, 0, 169, 255, 218, 218, 218, 218, 224, 255, 67, 0, 0],
            &[0, 12, 246, 56, 0, 0, 0, 0, 0, 156, 156, 0, 0],
            &[0, 92, 223, 0, 0, 0, 0, 0, 0, 69, 238, 6, 0],
            &[0, 181, 137, 0, 0, 0, 0, 0, 0, 3, 234, 78, 0],
            &[19, 250, 49, 0, 0, 0, 0, 0, 0, 0, 151, 168, 0],
            &[103, 217, 0, 0, 0, 0, 0, 0, 0, 0, 64, 245, 11],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Á' / 0xc1
        'Á' => &[
            &[0, 0, 0, 0, 0, 0, 0, 104, 216, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 238, 68, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 212, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 134, 254, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 222, 220, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 238, 90, 210, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 158, 10, 244, 43, 0, 0, 0, 0],
            &[0, 0, 0, 3, 231, 70, 0, 167, 132, 0, 0, 0, 0],
            &[0, 0, 0, 68, 234, 4, 0, 80, 221, 0, 0, 0, 0],
            &[0, 0, 0, 157, 151, 0, 0, 7, 241, 55, 0, 0, 0],
            &[0, 0, 7, 239, 63, 0, 0, 0, 162, 144, 0, 0, 0],
            &[0, 0, 80, 255, 68, 66, 66, 66, 141, 241, 2, 0, 0],
            &[0, 0, 169, 255, 218, 218, 218, 218, 224, 255, 67, 0, 0],
            &[0, 12, 246, 56, 0, 0, 0, 0, 0, 156, 156, 0, 0],
            &[0, 92, 223, 0, 0, 0, 0, 0, 0, 69, 238, 6, 0],
            &[0, 181, 137, 0, 0, 0, 0, 0, 0, 3, 234, 78, 0],
            &[19, 250, 49, 0, 0, 0, 0, 0, 0, 0, 151, 168, 0],
            &[103, 217, 0, 0, 0, 0, 0, 0, 0, 0, 64, 245, 11],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Â' / 0xc2
        'Â' => &[
            &[0, 0, 0, 0, 0, 131, 225, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 219, 113, 215, 15, 0, 0, 0, 0],
            &[0, 0, 0, 43, 212, 36, 0, 111, 183, 4, 0, 0, 0],
            &[0, 0, 0, 83, 35, 0, 0, 0, 79, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 134, 254, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 222, 220, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 238, 90, 210, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 158, 10, 244, 43, 0, 0, 0, 0],
            &[0, 0, 0, 3, 231, 70, 0, 167, 132, 0, 0, 0, 0],
            &[0, 0, 0, 68, 234, 4, 0, 80, 221, 0, 0, 0, 0],
            &[0, 0, 0, 157, 151, 0, 0, 7, 241, 55, 0, 0, 0],
            &[0, 0, 7, 239, 63, 0, 0, 0, 162, 144, 0, 0, 0],
            &[0, 0, 80, 255, 68, 66, 66, 66, 141, 241, 2, 0, 0],
            &[0, 0, 169, 255, 218, 218, 218, 218, 224, 255, 67, 0, 0],
            &[0, 12, 246, 56, 0, 0, 0, 0, 0, 156, 156, 0, 0],
            &[0, 92, 223, 0, 0, 0, 0, 0, 0, 69, 238, 6, 0],
            &[0, 181, 137, 0, 0, 0, 0, 0, 0, 3, 234, 78, 0],
            &[19, 250, 49, 0, 0, 0, 0, 0, 0, 0, 151, 168, 0],
            &[103, 217, 0, 0, 0, 0, 0, 0, 0, 0, 64, 245, 11],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ã' / 0xc3
        'Ã' => &[
            &[0, 0, 0, 0, 2, 0, 0, 0, 0, 13, 0, 0, 0],
            &[0, 0, 0, 123, 245, 220, 78, 0, 31, 199, 0, 0, 0],
            &[0, 0, 27, 214, 17, 52, 208, 198, 225, 86, 0, 0, 0],
            &[0, 0, 21, 47, 0, 0, 4, 62, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 134, 254, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 222, 220, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 238, 90, 210, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 158, 10, 244, 43, 0, 0, 0, 0],
            &[0, 0, 0, 3, 231, 70, 0, 167, 132, 0, 0, 0, 0],
            &[0, 0, 0, 68, 234, 4, 0, 80, 221, 0, 0, 0, 0],
            &[0, 0, 0, 157, 151, 0, 0, 7, 241, 55, 0, 0, 0],
            &[0, 0, 7, 239, 63, 0, 0, 0, 162, 144, 0, 0, 0],
            &[0, 0, 80, 255, 68, 66, 66, 66, 141, 241, 2, 0, 0],
            &[0, 0, 169, 255, 218, 218, 218, 218, 224, 255, 67, 0, 0],
            &[0, 12, 246, 56, 0, 0, 0, 0, 0, 156, 156, 0, 0],
            &[0, 92, 223, 0, 0, 0, 0, 0, 0, 69, 238, 6, 0],
            &[0, 181, 137, 0, 0, 0, 0, 0, 0, 3, 234, 78, 0],
            &[19, 250, 49, 0, 0, 0, 0, 0, 0, 0, 151, 168, 0],
            &[103, 217, 0, 0, 0, 0, 0, 0, 0, 0, 64, 245, 11],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ä' / 0xc4
        'Ä' => &[
            &[0, 0, 0, 0, 2, 0, 0, 0, 2, 0, 0, 0, 0],
            &[0, 0, 0, 155, 197, 0, 0, 44, 244, 61, 0, 0, 0],
            &[0, 0, 0, 119, 155, 0, 0, 29, 200, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 134, 254, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 222, 220, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 238, 90, 210, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 158, 10, 244, 43, 0, 0, 0, 0],
            &[0, 0, 0, 3, 231, 70, 0, 167, 132, 0, 0, 0, 0],
            &[0, 0, 0, 68, 234, 4, 0, 80, 221, 0, 0, 0, 0],
            &[0, 0, 0, 157, 151, 0, 0, 7, 241, 55, 0, 0, 0],
            &[0, 0, 7, 239, 63, 0, 0, 0, 162, 144, 0, 0, 0],
            &[0, 0, 80, 255, 68, 66, 66, 66, 141, 241, 2, 0, 0],
            &[0, 0, 169, 255, 218, 218, 218, 218, 224, 255, 67, 0, 0],
            &[0, 12, 246, 56, 0, 0, 0, 0, 0, 156, 156, 0, 0],
            &[0, 92, 223, 0, 0, 0, 0, 0, 0, 69, 238, 6, 0],
            &[0, 181, 137, 0, 0, 0, 0, 0, 0, 3, 234, 78, 0],
            &[19, 250, 49, 0, 0, 0, 0, 0, 0, 0, 151, 168, 0],
            &[103, 217, 0, 0, 0, 0, 0, 0, 0, 0, 64, 245, 11],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Å' / 0xc5
        'Å' => &[
            &[0, 0, 0, 0, 190, 68, 4, 166, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 214, 16, 0, 116, 114, 0, 0, 0, 0],
            &[0, 0, 0, 0, 124, 188, 128, 214, 33, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 86, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 134, 254, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 222, 220, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 238, 90, 210, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 158, 10, 244, 43, 0, 0, 0, 0],
            &[0, 0, 0, 3, 231, 70, 0, 167, 132, 0, 0, 0, 0],
            &[0, 0, 0, 68, 234, 4, 0, 80, 221, 0, 0, 0, 0],
            &[0, 0, 0, 157, 151, 0, 0, 7, 241, 55, 0, 0, 0],
            &[0, 0, 7, 239, 63, 0, 0, 0, 162, 144, 0, 0, 0],
            &[0, 0, 80, 255, 68, 66, 66, 66, 141, 241, 2, 0, 0],
            &[0, 0, 169, 255, 218, 218, 218, 218, 224, 255, 67, 0, 0],
            &[0, 12, 246, 56, 0, 0, 0, 0, 0, 156, 156, 0, 0],
            &[0, 92, 223, 0, 0, 0, 0, 0, 0, 69, 238, 6, 0],
            &[0, 181, 137, 0, 0, 0, 0, 0, 0, 3, 234, 78, 0],
            &[19, 250, 49, 0, 0, 0, 0, 0, 0, 0, 151, 168, 0],
            &[103, 217, 0, 0, 0, 0, 0, 0, 0, 0, 64, 245, 11],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 190, 205, 138, 1, 0, 0, 0, 0],
        ],
        // letter: 'Æ' / 0xc6
        'Æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 95, 254, 254, 254, 254, 254, 254, 214, 0],
            &[0, 0, 0, 0, 170, 130, 201, 120, 25, 25, 25, 21, 0],
            &[0, 0, 0, 4, 240, 48, 195, 105, 0, 0, 0, 0, 0],
            &[0, 0, 0, 62, 230, 0, 195, 105, 0, 0, 0, 0, 0],
            &[0, 0, 0, 137, 158, 0, 195, 105, 0, 0, 0, 0, 0],
            &[0, 0, 0, 211, 85, 0, 195, 105, 0, 0, 0, 0, 0],
            &[0, 0, 29, 251, 16, 0, 195, 168, 107, 107, 107, 52, 0],
            &[0, 0, 103, 196, 0, 0, 195, 206, 171, 171, 171, 83, 0],
            &[0, 0, 179, 189, 66, 66, 239, 105, 0, 0, 0, 0, 0],
            &[0, 7, 255, 255, 208, 208, 255, 105, 0, 0, 0, 0, 0],
            &[0, 70, 232, 1, 0, 0, 195, 105, 0, 0, 0, 0, 0],
            &[0, 144, 161, 0, 0, 0, 195, 105, 0, 0, 0, 0, 0],
            &[0, 218, 89, 0, 0, 0, 195, 105, 0, 0, 0, 0, 0],
            &[37, 253, 19, 0, 0, 0, 195, 122, 28, 28, 28, 24, 0],
            &[111, 199, 0, 0, 0, 0, 195, 255, 255, 255, 255, 215, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ç' / 0xc7
        'Ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 42, 35, 9, 0, 0, 0],
            &[0, 0, 0, 0, 69, 196, 253, 237, 244, 254, 193, 59, 0],
            &[0, 0, 0, 118, 245, 121, 22, 0, 0, 26, 111, 37, 0],
            &[0, 0, 69, 250, 63, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 204, 141, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 255, 36, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 228, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 128, 192, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 127, 193, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 104, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 255, 25, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 224, 124, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 241, 47, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 153, 237, 112, 22, 0, 4, 38, 108, 13, 0],
            &[0, 0, 0, 1, 110, 224, 255, 255, 253, 251, 191, 13, 0],
            &[0, 0, 0, 0, 0, 0, 111, 194, 34, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 168, 190, 63, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 42, 240, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 4, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'È' / 0xc8
        'È' => &[
            &[0, 0, 0, 7, 196, 143, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 234, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 218, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 81, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 196, 0, 0],
            &[0, 0, 203, 122, 25, 25, 25, 25, 25, 25, 19, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 170, 107, 107, 107, 107, 107, 107, 38, 0, 0],
            &[0, 0, 203, 207, 171, 171, 171, 171, 171, 171, 60, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 124, 28, 28, 28, 28, 28, 28, 22, 0, 0],
            &[0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'É' / 0xc9
        'É' => &[
            &[0, 0, 0, 0, 0, 0, 0, 66, 228, 51, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 226, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 190, 113, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 42, 75, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 196, 0, 0],
            &[0, 0, 203, 122, 25, 25, 25, 25, 25, 25, 19, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 170, 107, 107, 107, 107, 107, 107, 38, 0, 0],
            &[0, 0, 203, 207, 171, 171, 171, 171, 171, 171, 60, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 124, 28, 28, 28, 28, 28, 28, 22, 0, 0],
            &[0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ê' / 0xca
        'Ê' => &[
            &[0, 0, 0, 0, 0, 93, 226, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 227, 102, 229, 35, 0, 0, 0, 0],
            &[0, 0, 0, 21, 208, 63, 0, 75, 206, 17, 0, 0, 0],
            &[0, 0, 0, 63, 54, 0, 0, 0, 60, 58, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 196, 0, 0],
            &[0, 0, 203, 122, 25, 25, 25, 25, 25, 25, 19, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 170, 107, 107, 107, 107, 107, 107, 38, 0, 0],
            &[0, 0, 203, 207, 171, 171, 171, 171, 171, 171, 60, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 124, 28, 28, 28, 28, 28, 28, 22, 0, 0],
            &[0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ë' / 0xcb
        'Ë' => &[
            &[0, 0, 0, 0, 3, 0, 0, 0, 2, 0, 0, 0, 0],
            &[0, 0, 0, 112, 233, 6, 0, 11, 237, 101, 0, 0, 0],
            &[0, 0, 0, 83, 188, 2, 0, 5, 193, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 196, 0, 0],
            &[0, 0, 203, 122, 25, 25, 25, 25, 25, 25, 19, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 170, 107, 107, 107, 107, 107, 107, 38, 0, 0],
            &[0, 0, 203, 207, 171, 171, 171, 171, 171, 171, 60, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 124, 28, 28, 28, 28, 28, 28, 22, 0, 0],
            &[0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ì' / 0xcc
        'Ì' => &[
            &[0, 0, 0, 25, 217, 104, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 238, 45, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 211, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 56, 61, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 254, 254, 252, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 255, 255, 253, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Í' / 0xcd
        'Í' => &[
            &[0, 0, 0, 0, 0, 0, 0, 104, 216, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 238, 68, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 212, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 254, 254, 252, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 255, 255, 253, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Î' / 0xce
        'Î' => &[
            &[0, 0, 0, 0, 0, 131, 225, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 219, 113, 215, 15, 0, 0, 0, 0],
            &[0, 0, 0, 43, 212, 36, 0, 111, 183, 4, 0, 0, 0],
            &[0, 0, 0, 83, 35, 0, 0, 0, 79, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 254, 254, 252, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 255, 255, 253, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ï' / 0xcf
        'Ï' => &[
            &[0, 0, 0, 0, 2, 0, 0, 0, 2, 0, 0, 0, 0],
            &[0, 0, 0, 155, 197, 0, 0, 44, 244, 61, 0, 0, 0],
            &[0, 0, 0, 119, 155, 0, 0, 29, 200, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 254, 254, 252, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 255, 255, 253, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ð' / 0xd0
        'Ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 254, 254, 254, 246, 219, 168, 67, 0, 0, 0, 0],
            &[0, 12, 255, 60, 19, 32, 61, 142, 243, 149, 3, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 44, 230, 121, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 83, 248, 19, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 1, 231, 95, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 166, 153, 0],
            &[21, 33, 255, 65, 21, 20, 0, 0, 0, 0, 140, 175, 0],
            &[252, 255, 255, 255, 252, 243, 0, 0, 0, 0, 126, 188, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 139, 169, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 175, 144, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 6, 235, 76, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 100, 236, 8, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 58, 241, 92, 0, 0],
            &[0, 12, 255, 63, 24, 37, 75, 157, 248, 117, 0, 0, 0],
            &[0, 12, 255, 255, 255, 240, 213, 149, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ñ' / 0xd1
        'Ñ' => &[
            &[0, 0, 0, 0, 2, 0, 0, 0, 0, 13, 0, 0, 0],
            &[0, 0, 0, 100, 241, 229, 96, 0, 13, 216, 0, 0, 0],
            &[0, 0, 8, 226, 24, 40, 194, 205, 219, 111, 0, 0, 0],
            &[0, 0, 12, 56, 0, 0, 1, 57, 48, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 254, 132, 0, 0, 0, 0, 0, 81, 223, 0, 0],
            &[0, 65, 234, 239, 16, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 212, 169, 127, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 225, 46, 237, 14, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 232, 0, 175, 122, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 50, 235, 12, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 181, 117, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 56, 233, 10, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 186, 113, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 61, 230, 8, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 192, 108, 78, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 67, 227, 78, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 197, 165, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 72, 251, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 0, 202, 224, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ò' / 0xd2
        'Ò' => &[
            &[0, 0, 0, 25, 217, 104, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 238, 45, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 211, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 56, 61, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 49, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 217, 250, 230, 251, 184, 31, 0, 0, 0],
            &[0, 0, 79, 245, 105, 5, 0, 21, 157, 226, 21, 0, 0],
            &[0, 8, 231, 105, 0, 0, 0, 0, 1, 190, 150, 0, 0],
            &[0, 94, 229, 4, 0, 0, 0, 0, 0, 70, 244, 8, 0],
            &[0, 159, 156, 0, 0, 0, 0, 0, 0, 6, 246, 66, 0],
            &[0, 207, 108, 0, 0, 0, 0, 0, 0, 0, 209, 107, 0],
            &[0, 226, 88, 0, 0, 0, 0, 0, 0, 0, 186, 129, 0],
            &[0, 239, 77, 0, 0, 0, 0, 0, 0, 0, 179, 137, 0],
            &[0, 230, 88, 0, 0, 0, 0, 0, 0, 0, 190, 127, 0],
            &[0, 206, 110, 0, 0, 0, 0, 0, 0, 0, 211, 103, 0],
            &[0, 162, 159, 0, 0, 0, 0, 0, 0, 7, 250, 61, 0],
            &[0, 92, 232, 5, 0, 0, 0, 0, 0, 77, 241, 5, 0],
            &[0, 9, 233, 111, 0, 0, 0, 0, 4, 198, 143, 0, 0],
            &[0, 0, 82, 249, 112, 10, 0, 31, 173, 221, 17, 0, 0],
            &[0, 0, 0, 77, 218, 254, 242, 252, 175, 25, 0, 0, 0],
            &[0, 0, 0, 0, 1, 30, 45, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ó' / 0xd3
        'Ó' => &[
            &[0, 0, 0, 0, 0, 0, 0, 104, 216, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 238, 68, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 212, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 49, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 217, 250, 230, 251, 184, 31, 0, 0, 0],
            &[0, 0, 79, 245, 105, 5, 0, 21, 157, 226, 21, 0, 0],
            &[0, 8, 231, 105, 0, 0, 0, 0, 1, 190, 150, 0, 0],
            &[0, 94, 229, 4, 0, 0, 0, 0, 0, 70, 244, 8, 0],
            &[0, 159, 156, 0, 0, 0, 0, 0, 0, 6, 246, 66, 0],
            &[0, 207, 108, 0, 0, 0, 0, 0, 0, 0, 209, 107, 0],
            &[0, 226, 88, 0, 0, 0, 0, 0, 0, 0, 186, 129, 0],
            &[0, 239, 77, 0, 0, 0, 0, 0, 0, 0, 179, 137, 0],
            &[0, 230, 88, 0, 0, 0, 0, 0, 0, 0, 190, 127, 0],
            &[0, 206, 110, 0, 0, 0, 0, 0, 0, 0, 211, 103, 0],
            &[0, 162, 159, 0, 0, 0, 0, 0, 0, 7, 250, 61, 0],
            &[0, 92, 232, 5, 0, 0, 0, 0, 0, 77, 241, 5, 0],
            &[0, 9, 233, 111, 0, 0, 0, 0, 4, 198, 143, 0, 0],
            &[0, 0, 82, 249, 112, 10, 0, 31, 173, 221, 17, 0, 0],
            &[0, 0, 0, 77, 218, 254, 242, 252, 175, 25, 0, 0, 0],
            &[0, 0, 0, 0, 1, 30, 45, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ô' / 0xd4
        'Ô' => &[
            &[0, 0, 0, 0, 0, 131, 225, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 219, 113, 215, 15, 0, 0, 0, 0],
            &[0, 0, 0, 43, 212, 36, 0, 111, 183, 4, 0, 0, 0],
            &[0, 0, 0, 83, 35, 0, 0, 0, 79, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 49, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 217, 250, 230, 251, 184, 31, 0, 0, 0],
            &[0, 0, 79, 245, 105, 5, 0, 21, 157, 226, 21, 0, 0],
            &[0, 8, 231, 105, 0, 0, 0, 0, 1, 190, 150, 0, 0],
            &[0, 94, 229, 4, 0, 0, 0, 0, 0, 70, 244, 8, 0],
            &[0, 159, 156, 0, 0, 0, 0, 0, 0, 6, 246, 66, 0],
            &[0, 207, 108, 0, 0, 0, 0, 0, 0, 0, 209, 107, 0],
            &[0, 226, 88, 0, 0, 0, 0, 0, 0, 0, 186, 129, 0],
            &[0, 239, 77, 0, 0, 0, 0, 0, 0, 0, 179, 137, 0],
            &[0, 230, 88, 0, 0, 0, 0, 0, 0, 0, 190, 127, 0],
            &[0, 206, 110, 0, 0, 0, 0, 0, 0, 0, 211, 103, 0],
            &[0, 162, 159, 0, 0, 0, 0, 0, 0, 7, 250, 61, 0],
            &[0, 92, 232, 5, 0, 0, 0, 0, 0, 77, 241, 5, 0],
            &[0, 9, 233, 111, 0, 0, 0, 0, 4, 198, 143, 0, 0],
            &[0, 0, 82, 249, 112, 10, 0, 31, 173, 221, 17, 0, 0],
            &[0, 0, 0, 77, 218, 254, 242, 252, 175, 25, 0, 0, 0],
            &[0, 0, 0, 0, 1, 30, 45, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Õ' / 0xd5
        'Õ' => &[
            &[0, 0, 0, 0, 2, 0, 0, 0, 0, 13, 0, 0, 0],
            &[0, 0, 0, 123, 245, 220, 78, 0, 31, 199, 0, 0, 0],
            &[0, 0, 27, 214, 17, 52, 208, 198, 225, 86, 0, 0, 0],
            &[0, 0, 21, 47, 0, 0, 4, 62, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 49, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 217, 250, 230, 251, 184, 31, 0, 0, 0],
            &[0, 0, 79, 245, 105, 5, 0, 21, 157, 226, 21, 0, 0],
            &[0, 8, 231, 105, 0, 0, 0, 0, 1, 190, 150, 0, 0],
            &[0, 94, 229, 4, 0, 0, 0, 0, 0, 70, 244, 8, 0],
            &[0, 159, 156, 0, 0, 0, 0, 0, 0, 6, 246, 66, 0],
            &[0, 207, 108, 0, 0, 0, 0, 0, 0, 0, 209, 107, 0],
            &[0, 226, 88, 0, 0, 0, 0, 0, 0, 0, 186, 129, 0],
            &[0, 239, 77, 0, 0, 0, 0, 0, 0, 0, 179, 137, 0],
            &[0, 230, 88, 0, 0, 0, 0, 0, 0, 0, 190, 127, 0],
            &[0, 206, 110, 0, 0, 0, 0, 0, 0, 0, 211, 103, 0],
            &[0, 162, 159, 0, 0, 0, 0, 0, 0, 7, 250, 61, 0],
            &[0, 92, 232, 5, 0, 0, 0, 0, 0, 77, 241, 5, 0],
            &[0, 9, 233, 111, 0, 0, 0, 0, 4, 198, 143, 0, 0],
            &[0, 0, 82, 249, 112, 10, 0, 31, 173, 221, 17, 0, 0],
            &[0, 0, 0, 77, 218, 254, 242, 252, 175, 25, 0, 0, 0],
            &[0, 0, 0, 0, 1, 30, 45, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ö' / 0xd6
        'Ö' => &[
            &[0, 0, 0, 0, 2, 0, 0, 0, 2, 0, 0, 0, 0],
            &[0, 0, 0, 155, 197, 0, 0, 44, 244, 61, 0, 0, 0],
            &[0, 0, 0, 119, 155, 0, 0, 29, 200, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 49, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 217, 250, 230, 251, 184, 31, 0, 0, 0],
            &[0, 0, 79, 245, 105, 5, 0, 21, 157, 226, 21, 0, 0],
            &[0, 8, 231, 105, 0, 0, 0, 0, 1, 190, 150, 0, 0],
            &[0, 94, 229, 4, 0, 0, 0, 0, 0, 70, 244, 8, 0],
            &[0, 159, 156, 0, 0, 0, 0, 0, 0, 6, 246, 66, 0],
            &[0, 207, 108, 0, 0, 0, 0, 0, 0, 0, 209, 107, 0],
            &[0, 226, 88, 0, 0, 0, 0, 0, 0, 0, 186, 129, 0],
            &[0, 239, 77, 0, 0, 0, 0, 0, 0, 0, 179, 137, 0],
            &[0, 230, 88, 0, 0, 0, 0, 0, 0, 0, 190, 127, 0],
            &[0, 206, 110, 0, 0, 0, 0, 0, 0, 0, 211, 103, 0],
            &[0, 162, 159, 0, 0, 0, 0, 0, 0, 7, 250, 61, 0],
            &[0, 92, 232, 5, 0, 0, 0, 0, 0, 77, 241, 5, 0],
            &[0, 9, 233, 111, 0, 0, 0, 0, 4, 198, 143, 0, 0],
            &[0, 0, 82, 249, 112, 10, 0, 31, 173, 221, 17, 0, 0],
            &[0, 0, 0, 77, 218, 254, 242, 252, 175, 25, 0, 0, 0],
            &[0, 0, 0, 0, 1, 30, 45, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '×' / 0xd7
        '×' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 9, 172, 22, 0, 0, 0, 0, 0, 83, 123, 0, 0],
            &[0, 0, 122, 212, 22, 0, 0, 0, 83, 229, 43, 0, 0],
            &[0, 0, 0, 122, 212, 21, 0, 83, 229, 44, 0, 0, 0],
            &[0, 0, 0, 0, 122, 211, 101, 230, 45, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 166, 255, 69, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 230, 150, 213, 23, 0, 0, 0, 0],
            &[0, 0, 0, 83, 230, 46, 0, 124, 213, 23, 0, 0, 0],
            &[0, 0, 83, 230, 46, 0, 0, 0, 122, 213, 23, 0, 0],
            &[0, 9, 207, 46, 0, 0, 0, 0, 0, 119, 141, 0, 0],
            &[0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ø' / 0xd8
        'Ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 49, 20, 0, 0, 95, 9, 0],
            &[0, 0, 0, 76, 217, 250, 230, 251, 184, 102, 221, 7, 0],
            &[0, 0, 79, 245, 105, 5, 0, 21, 162, 255, 104, 0, 0],
            &[0, 8, 231, 105, 0, 0, 0, 0, 109, 255, 150, 0, 0],
            &[0, 94, 229, 4, 0, 0, 0, 17, 234, 119, 244, 8, 0],
            &[0, 159, 156, 0, 0, 0, 0, 144, 156, 6, 246, 66, 0],
            &[0, 207, 108, 0, 0, 0, 39, 237, 23, 0, 209, 107, 0],
            &[0, 226, 88, 0, 0, 0, 181, 120, 0, 0, 186, 129, 0],
            &[0, 239, 77, 0, 0, 71, 222, 7, 0, 0, 179, 137, 0],
            &[0, 230, 88, 0, 4, 213, 84, 0, 0, 0, 190, 127, 0],
            &[0, 206, 110, 0, 108, 193, 0, 0, 0, 0, 211, 103, 0],
            &[0, 162, 159, 17, 234, 50, 0, 0, 0, 7, 250, 61, 0],
            &[0, 92, 232, 151, 158, 0, 0, 0, 0, 77, 241, 5, 0],
            &[0, 9, 255, 255, 24, 0, 0, 0, 4, 198, 143, 0, 0],
            &[0, 0, 255, 255, 112, 10, 0, 31, 173, 221, 17, 0, 0],
            &[0, 72, 223, 85, 218, 254, 242, 252, 175, 25, 0, 0, 0],
            &[0, 62, 71, 0, 1, 30, 45, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ù' / 0xd9
        'Ù' => &[
            &[0, 0, 0, 25, 217, 104, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 238, 45, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 211, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 56, 61, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 86, 223, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 64, 247, 0, 0, 0, 0, 0, 0, 88, 223, 0, 0],
            &[0, 50, 255, 5, 0, 0, 0, 0, 0, 102, 208, 0, 0],
            &[0, 21, 255, 44, 0, 0, 0, 0, 0, 143, 176, 0, 0],
            &[0, 0, 208, 131, 0, 0, 0, 0, 7, 227, 104, 0, 0],
            &[0, 0, 80, 249, 98, 4, 0, 30, 178, 217, 7, 0, 0],
            &[0, 0, 0, 94, 228, 251, 240, 253, 179, 26, 0, 0, 0],
            &[0, 0, 0, 0, 3, 31, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ú' / 0xda
        'Ú' => &[
            &[0, 0, 0, 0, 0, 0, 0, 104, 216, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 238, 68, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 212, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 86, 223, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 64, 247, 0, 0, 0, 0, 0, 0, 88, 223, 0, 0],
            &[0, 50, 255, 5, 0, 0, 0, 0, 0, 102, 208, 0, 0],
            &[0, 21, 255, 44, 0, 0, 0, 0, 0, 143, 176, 0, 0],
            &[0, 0, 208, 131, 0, 0, 0, 0, 7, 227, 104, 0, 0],
            &[0, 0, 80, 249, 98, 4, 0, 30, 178, 217, 7, 0, 0],
            &[0, 0, 0, 94, 228, 251, 240, 253, 179, 26, 0, 0, 0],
            &[0, 0, 0, 0, 3, 31, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Û' / 0xdb
        'Û' => &[
            &[0, 0, 0, 0, 0, 131, 225, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 219, 113, 215, 15, 0, 0, 0, 0],
            &[0, 0, 0, 43, 212, 36, 0, 111, 183, 4, 0, 0, 0],
            &[0, 0, 0, 83, 35, 0, 0, 0, 79, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 86, 223, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 64, 247, 0, 0, 0, 0, 0, 0, 88, 223, 0, 0],
            &[0, 50, 255, 5, 0, 0, 0, 0, 0, 102, 208, 0, 0],
            &[0, 21, 255, 44, 0, 0, 0, 0, 0, 143, 176, 0, 0],
            &[0, 0, 208, 131, 0, 0, 0, 0, 7, 227, 104, 0, 0],
            &[0, 0, 80, 249, 98, 4, 0, 30, 178, 217, 7, 0, 0],
            &[0, 0, 0, 94, 228, 251, 240, 253, 179, 26, 0, 0, 0],
            &[0, 0, 0, 0, 3, 31, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ü' / 0xdc
        'Ü' => &[
            &[0, 0, 0, 0, 2, 0, 0, 0, 2, 0, 0, 0, 0],
            &[0, 0, 0, 155, 197, 0, 0, 44, 244, 61, 0, 0, 0],
            &[0, 0, 0, 119, 155, 0, 0, 29, 200, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 86, 223, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 64, 247, 0, 0, 0, 0, 0, 0, 88, 223, 0, 0],
            &[0, 50, 255, 5, 0, 0, 0, 0, 0, 102, 208, 0, 0],
            &[0, 21, 255, 44, 0, 0, 0, 0, 0, 143, 176, 0, 0],
            &[0, 0, 208, 131, 0, 0, 0, 0, 7, 227, 104, 0, 0],
            &[0, 0, 80, 249, 98, 4, 0, 30, 178, 217, 7, 0, 0],
            &[0, 0, 0, 94, 228, 251, 240, 253, 179, 26, 0, 0, 0],
            &[0, 0, 0, 0, 3, 31, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ý' / 0xdd
        'Ý' => &[
            &[0, 0, 0, 0, 0, 0, 0, 104, 216, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 45, 238, 68, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 212, 77, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 62, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[2, 210, 126, 0, 0, 0, 0, 0, 0, 2, 214, 116, 0],
            &[0, 80, 239, 15, 0, 0, 0, 0, 0, 88, 230, 9, 0],
            &[0, 0, 201, 126, 0, 0, 0, 0, 1, 212, 106, 0, 0],
            &[0, 0, 68, 239, 15, 0, 0, 0, 85, 223, 5, 0, 0],
            &[0, 0, 0, 190, 126, 0, 0, 1, 210, 95, 0, 0, 0],
            &[0, 0, 0, 56, 238, 15, 0, 82, 214, 3, 0, 0, 0],
            &[0, 0, 0, 0, 177, 126, 1, 208, 84, 0, 0, 0, 0],
            &[0, 0, 0, 0, 45, 237, 95, 205, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 165, 251, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 235, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Þ' / 0xde
        'Þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 202, 107, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 203, 128, 35, 35, 32, 14, 0, 0, 0, 0, 0, 0],
            &[0, 203, 246, 238, 238, 247, 255, 221, 120, 5, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 20, 99, 233, 160, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 72, 255, 37, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 2, 246, 89, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 229, 92, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 21, 252, 52, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 2, 162, 204, 0, 0, 0],
            &[0, 203, 156, 83, 83, 95, 131, 215, 215, 34, 0, 0, 0],
            &[0, 203, 221, 196, 195, 183, 158, 90, 7, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ß' / 0xdf
        'ß' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 84, 202, 252, 255, 240, 163, 25, 0, 0, 0],
            &[0, 0, 92, 246, 104, 18, 4, 45, 189, 208, 1, 0, 0],
            &[0, 0, 216, 119, 0, 0, 0, 0, 37, 255, 34, 0, 0],
            &[0, 6, 254, 47, 0, 0, 0, 0, 54, 251, 11, 0, 0],
            &[0, 17, 255, 33, 0, 0, 0, 20, 205, 142, 0, 0, 0],
            &[0, 17, 255, 33, 0, 0, 34, 226, 135, 2, 0, 0, 0],
            &[0, 17, 255, 33, 0, 0, 178, 152, 0, 0, 0, 0, 0],
            &[0, 17, 255, 33, 0, 0, 215, 93, 0, 0, 0, 0, 0],
            &[0, 17, 255, 33, 0, 0, 136, 217, 32, 0, 0, 0, 0],
            &[0, 17, 255, 33, 0, 0, 3, 150, 240, 87, 0, 0, 0],
            &[0, 17, 255, 33, 0, 0, 0, 0, 85, 241, 122, 0, 0],
            &[0, 17, 255, 33, 0, 0, 0, 0, 0, 54, 247, 54, 0],
            &[0, 17, 255, 33, 0, 0, 0, 0, 0, 0, 195, 123, 0],
            &[0, 17, 255, 33, 0, 0, 0, 0, 0, 0, 194, 120, 0],
            &[0, 17, 255, 33, 12, 87, 7, 0, 0, 83, 249, 44, 0],
            &[0, 17, 255, 33, 12, 198, 251, 222, 232, 231, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 44, 37, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'à' / 0xe0
        'à' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 22, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 207, 165, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 222, 96, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 210, 58, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 48, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 82, 89, 59, 11, 0, 0, 0, 0],
            &[0, 0, 42, 205, 234, 191, 175, 212, 232, 85, 0, 0, 0],
            &[0, 0, 11, 81, 4, 0, 0, 0, 97, 250, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 209, 101, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 171, 125, 0, 0],
            &[0, 0, 0, 0, 32, 88, 115, 130, 139, 215, 127, 0, 0],
            &[0, 0, 33, 191, 231, 162, 126, 107, 98, 195, 127, 0, 0],
            &[0, 6, 220, 143, 5, 0, 0, 0, 0, 166, 127, 0, 0],
            &[0, 59, 241, 4, 0, 0, 0, 0, 0, 200, 127, 0, 0],
            &[0, 63, 238, 0, 0, 0, 0, 0, 53, 254, 127, 0, 0],
            &[0, 14, 236, 123, 3, 0, 3, 79, 200, 180, 127, 0, 0],
            &[0, 0, 63, 218, 240, 216, 239, 173, 33, 134, 127, 0, 0],
            &[0, 0, 0, 1, 27, 47, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'á' / 0xe1
        'á' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 22, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 111, 235, 31, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 51, 236, 57, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 211, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 82, 89, 59, 11, 0, 0, 0, 0],
            &[0, 0, 42, 205, 234, 191, 175, 212, 232, 85, 0, 0, 0],
            &[0, 0, 11, 81, 4, 0, 0, 0, 97, 250, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 209, 101, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 171, 125, 0, 0],
            &[0, 0, 0, 0, 32, 88, 115, 130, 139, 215, 127, 0, 0],
            &[0, 0, 33, 191, 231, 162, 126, 107, 98, 195, 127, 0, 0],
            &[0, 6, 220, 143, 5, 0, 0, 0, 0, 166, 127, 0, 0],
            &[0, 59, 241, 4, 0, 0, 0, 0, 0, 200, 127, 0, 0],
            &[0, 63, 238, 0, 0, 0, 0, 0, 53, 254, 127, 0, 0],
            &[0, 14, 236, 123, 3, 0, 3, 79, 200, 180, 127, 0, 0],
            &[0, 0, 63, 218, 240, 216, 239, 173, 33, 134, 127, 0, 0],
            &[0, 0, 0, 1, 27, 47, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'â' / 0xe2
        'â' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 142, 255, 101, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 212, 60, 227, 49, 0, 0, 0, 0],
            &[0, 0, 0, 52, 207, 28, 0, 54, 210, 27, 0, 0, 0],
            &[0, 0, 0, 46, 22, 0, 0, 0, 34, 35, 0, 0, 0],
            &[0, 0, 0, 0, 38, 82, 89, 59, 11, 0, 0, 0, 0],
            &[0, 0, 42, 205, 234, 191, 175, 212, 232, 85, 0, 0, 0],
            &[0, 0, 11, 81, 4, 0, 0, 0, 97, 250, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 209, 101, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 171, 125, 0, 0],
            &[0, 0, 0, 0, 32, 88, 115, 130, 139, 215, 127, 0, 0],
            &[0, 0, 33, 191, 231, 162, 126, 107, 98, 195, 127, 0, 0],
            &[0, 6, 220, 143, 5, 0, 0, 0, 0, 166, 127, 0, 0],
            &[0, 59, 241, 4, 0, 0, 0, 0, 0, 200, 127, 0, 0],
            &[0, 63, 238, 0, 0, 0, 0, 0, 53, 254, 127, 0, 0],
            &[0, 14, 236, 123, 3, 0, 3, 79, 200, 180, 127, 0, 0],
            &[0, 0, 63, 218, 240, 216, 239, 173, 33, 134, 127, 0, 0],
            &[0, 0, 0, 1, 27, 47, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ã' / 0xe3
        'ã' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 36, 22, 0, 0, 0, 51, 0, 0, 0],
            &[0, 0, 0, 139, 228, 235, 141, 8, 31, 211, 0, 0, 0],
            &[0, 0, 14, 220, 3, 11, 150, 240, 240, 79, 0, 0, 0],
            &[0, 0, 6, 24, 0, 0, 0, 14, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 82, 89, 59, 11, 0, 0, 0, 0],
            &[0, 0, 42, 205, 234, 191, 175, 212, 232, 85, 0, 0, 0],
            &[0, 0, 11, 81, 4, 0, 0, 0, 97, 250, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 209, 101, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 171, 125, 0, 0],
            &[0, 0, 0, 0, 32, 88, 115, 130, 139, 215, 127, 0, 0],
            &[0, 0, 33, 191, 231, 162, 126, 107, 98, 195, 127, 0, 0],
            &[0, 6, 220, 143, 5, 0, 0, 0, 0, 166, 127, 0, 0],
            &[0, 59, 241, 4, 0, 0, 0, 0, 0, 200, 127, 0, 0],
            &[0, 63, 238, 0, 0, 0, 0, 0, 53, 254, 127, 0, 0],
            &[0, 14, 236, 123, 3, 0, 3, 79, 200, 180, 127, 0, 0],
            &[0, 0, 63, 218, 240, 216, 239, 173, 33, 134, 127, 0, 0],
            &[0, 0, 0, 1, 27, 47, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ä' / 0xe4
        'ä' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 29, 0, 0, 0, 34, 3, 0, 0, 0],
            &[0, 0, 0, 145, 241, 1, 0, 29, 255, 102, 0, 0, 0],
            &[0, 0, 0, 69, 132, 0, 0, 5, 152, 42, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 82, 89, 59, 11, 0, 0, 0, 0],
            &[0, 0, 42, 205, 234, 191, 175, 212, 232, 85, 0, 0, 0],
            &[0, 0, 11, 81, 4, 0, 0, 0, 97, 250, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 209, 101, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 171, 125, 0, 0],
            &[0, 0, 0, 0, 32, 88, 115, 130, 139, 215, 127, 0, 0],
            &[0, 0, 33, 191, 231, 162, 126, 107, 98, 195, 127, 0, 0],
            &[0, 6, 220, 143, 5, 0, 0, 0, 0, 166, 127, 0, 0],
            &[0, 59, 241, 4, 0, 0, 0, 0, 0, 200, 127, 0, 0],
            &[0, 63, 238, 0, 0, 0, 0, 0, 53, 254, 127, 0, 0],
            &[0, 14, 236, 123, 3, 0, 3, 79, 200, 180, 127, 0, 0],
            &[0, 0, 63, 218, 240, 216, 239, 173, 33, 134, 127, 0, 0],
            &[0, 0, 0, 1, 27, 47, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'å' / 0xe5
        'å' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 215, 195, 196, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 177, 64, 0, 112, 128, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 52, 0, 102, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 69, 217, 170, 211, 31, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 42, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 82, 89, 59, 11, 0, 0, 0, 0],
            &[0, 0, 42, 205, 234, 191, 175, 212, 232, 85, 0, 0, 0],
            &[0, 0, 11, 81, 4, 0, 0, 0, 97, 250, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 209, 101, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 171, 125, 0, 0],
            &[0, 0, 0, 0, 32, 88, 115, 130, 139, 215, 127, 0, 0],
            &[0, 0, 33, 191, 231, 162, 126, 107, 98, 195, 127, 0, 0],
            &[0, 6, 220, 143, 5, 0, 0, 0, 0, 166, 127, 0, 0],
            &[0, 59, 241, 4, 0, 0, 0, 0, 0, 200, 127, 0, 0],
            &[0, 63, 238, 0, 0, 0, 0, 0, 53, 254, 127, 0, 0],
            &[0, 14, 236, 123, 3, 0, 3, 79, 200, 180, 127, 0, 0],
            &[0, 0, 63, 218, 240, 216, 239, 173, 33, 134, 127, 0, 0],
            &[0, 0, 0, 1, 27, 47, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'æ' / 0xe6
        'æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 83, 68, 3, 0, 2, 69, 80, 11, 0, 0],
            &[0, 137, 232, 179, 215, 194, 10, 186, 205, 197, 221, 20, 0],
            &[0, 40, 8, 0, 4, 195, 171, 162, 0, 0, 167, 136, 0],
            &[0, 0, 0, 0, 0, 109, 255, 37, 0, 0, 74, 210, 0],
            &[0, 0, 0, 0, 0, 87, 234, 0, 0, 0, 38, 247, 0],
            &[0, 0, 16, 81, 110, 165, 231, 119, 119, 119, 135, 255, 5],
            &[0, 96, 244, 184, 134, 165, 228, 132, 132, 132, 132, 132, 4],
            &[30, 249, 69, 0, 0, 90, 209, 0, 0, 0, 0, 0, 0],
            &[94, 204, 0, 0, 0, 107, 241, 1, 0, 0, 0, 0, 0],
            &[96, 192, 0, 0, 0, 168, 227, 62, 0, 0, 0, 0, 0],
            &[45, 246, 38, 0, 70, 206, 76, 209, 25, 0, 12, 78, 0],
            &[0, 134, 247, 221, 218, 44, 0, 123, 243, 219, 243, 120, 0],
            &[0, 0, 21, 38, 3, 0, 0, 0, 14, 44, 14, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ç' / 0xe7
        'ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 70, 91, 72, 25, 0, 0, 0],
            &[0, 0, 0, 19, 167, 249, 212, 188, 206, 248, 96, 0, 0],
            &[0, 0, 11, 215, 190, 34, 0, 0, 0, 18, 18, 0, 0],
            &[0, 0, 134, 215, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 226, 104, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 255, 45, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 255, 27, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 228, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 139, 214, 7, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 218, 194, 51, 0, 0, 1, 48, 49, 0, 0],
            &[0, 0, 0, 18, 156, 244, 249, 235, 250, 229, 80, 0, 0],
            &[0, 0, 0, 0, 0, 6, 197, 111, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 19, 224, 152, 25, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 119, 169, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 1, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'è' / 0xe8
        'è' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 22, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 210, 159, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 224, 91, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 211, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 47, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 61, 89, 70, 14, 0, 0, 0, 0],
            &[0, 0, 0, 90, 232, 217, 176, 197, 240, 101, 0, 0, 0],
            &[0, 0, 87, 242, 83, 0, 0, 0, 56, 239, 74, 0, 0],
            &[0, 7, 233, 101, 0, 0, 0, 0, 0, 118, 201, 0, 0],
            &[0, 74, 242, 4, 0, 0, 0, 0, 0, 42, 254, 15, 0],
            &[0, 121, 222, 114, 114, 114, 114, 114, 114, 124, 255, 40, 0],
            &[0, 140, 220, 143, 143, 143, 143, 143, 143, 143, 143, 27, 0],
            &[0, 122, 194, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 75, 241, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 230, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 243, 124, 19, 0, 0, 3, 59, 107, 0, 0],
            &[0, 0, 0, 52, 184, 249, 229, 226, 252, 215, 113, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'é' / 0xe9
        'é' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 22, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 116, 233, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 55, 235, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 212, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 35, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 61, 89, 70, 14, 0, 0, 0, 0],
            &[0, 0, 0, 90, 232, 217, 176, 197, 240, 101, 0, 0, 0],
            &[0, 0, 87, 242, 83, 0, 0, 0, 56, 239, 74, 0, 0],
            &[0, 7, 233, 101, 0, 0, 0, 0, 0, 118, 201, 0, 0],
            &[0, 74, 242, 4, 0, 0, 0, 0, 0, 42, 254, 15, 0],
            &[0, 121, 222, 114, 114, 114, 114, 114, 114, 124, 255, 40, 0],
            &[0, 140, 220, 143, 143, 143, 143, 143, 143, 143, 143, 27, 0],
            &[0, 122, 194, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 75, 241, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 230, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 243, 124, 19, 0, 0, 3, 59, 107, 0, 0],
            &[0, 0, 0, 52, 184, 249, 229, 226, 252, 215, 113, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ê' / 0xea
        'ê' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 148, 255, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 209, 61, 228, 45, 0, 0, 0, 0],
            &[0, 0, 0, 56, 206, 25, 0, 58, 209, 25, 0, 0, 0],
            &[0, 0, 0, 48, 20, 0, 0, 0, 35, 33, 0, 0, 0],
            &[0, 0, 0, 0, 6, 61, 89, 70, 14, 0, 0, 0, 0],
            &[0, 0, 0, 90, 232, 217, 176, 197, 240, 101, 0, 0, 0],
            &[0, 0, 87, 242, 83, 0, 0, 0, 56, 239, 74, 0, 0],
            &[0, 7, 233, 101, 0, 0, 0, 0, 0, 118, 201, 0, 0],
            &[0, 74, 242, 4, 0, 0, 0, 0, 0, 42, 254, 15, 0],
            &[0, 121, 222, 114, 114, 114, 114, 114, 114, 124, 255, 40, 0],
            &[0, 140, 220, 143, 143, 143, 143, 143, 143, 143, 143, 27, 0],
            &[0, 122, 194, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 75, 241, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 230, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 243, 124, 19, 0, 0, 3, 59, 107, 0, 0],
            &[0, 0, 0, 52, 184, 249, 229, 226, 252, 215, 113, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ë' / 0xeb
        'ë' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 28, 0, 0, 0, 35, 2, 0, 0, 0],
            &[0, 0, 0, 151, 236, 0, 0, 34, 255, 96, 0, 0, 0],
            &[0, 0, 0, 72, 129, 0, 0, 8, 153, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 61, 89, 70, 14, 0, 0, 0, 0],
            &[0, 0, 0, 90, 232, 217, 176, 197, 240, 101, 0, 0, 0],
            &[0, 0, 87, 242, 83, 0, 0, 0, 56, 239, 74, 0, 0],
            &[0, 7, 233, 101, 0, 0, 0, 0, 0, 118, 201, 0, 0],
            &[0, 74, 242, 4, 0, 0, 0, 0, 0, 42, 254, 15, 0],
            &[0, 121, 222, 114, 114, 114, 114, 114, 114, 124, 255, 40, 0],
            &[0, 140, 220, 143, 143, 143, 143, 143, 143, 143, 143, 27, 0],
            &[0, 122, 194, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 75, 241, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 230, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 243, 124, 19, 0, 0, 3, 59, 107, 0, 0],
            &[0, 0, 0, 52, 184, 249, 229, 226, 252, 215, 113, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ì' / 0xec
        'ì' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 22, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 223, 138, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 232, 72, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 213, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 27, 41, 0, 0, 0, 0, 0],
            &[0, 0, 33, 44, 44, 44, 44, 12, 0, 0, 0, 0, 0],
            &[0, 0, 136, 192, 204, 217, 253, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'í' / 0xed
        'í' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 22, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 138, 222, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 232, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 214, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 44, 44, 44, 44, 12, 0, 0, 0, 0, 0],
            &[0, 0, 136, 192, 204, 217, 253, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'î' / 0xee
        'î' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 169, 254, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 196, 68, 226, 32, 0, 0, 0, 0],
            &[0, 0, 0, 72, 198, 17, 0, 73, 203, 16, 0, 0, 0],
            &[0, 0, 0, 54, 14, 0, 0, 0, 41, 27, 0, 0, 0],
            &[0, 0, 33, 44, 44, 44, 44, 12, 0, 0, 0, 0, 0],
            &[0, 0, 136, 192, 204, 217, 253, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ï' / 0xef
        'ï' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 24, 0, 0, 0, 36, 1, 0, 0, 0],
            &[0, 0, 0, 172, 215, 0, 0, 55, 255, 75, 0, 0, 0],
            &[0, 0, 0, 86, 115, 0, 0, 18, 154, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 44, 44, 44, 44, 12, 0, 0, 0, 0, 0],
            &[0, 0, 136, 192, 204, 217, 253, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ð' / 0xf0
        'ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 147, 133, 13, 0, 65, 164, 2, 0, 0],
            &[0, 0, 0, 0, 24, 162, 248, 215, 211, 65, 0, 0, 0],
            &[0, 0, 0, 0, 8, 125, 255, 255, 52, 0, 0, 0, 0],
            &[0, 0, 0, 17, 219, 155, 21, 83, 228, 19, 0, 0, 0],
            &[0, 0, 0, 0, 32, 0, 0, 0, 136, 167, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 9, 227, 48, 0, 0],
            &[0, 0, 0, 33, 139, 195, 210, 180, 92, 122, 151, 0, 0],
            &[0, 0, 70, 240, 151, 68, 47, 75, 178, 205, 232, 0, 0],
            &[0, 18, 237, 97, 0, 0, 0, 0, 0, 159, 255, 31, 0],
            &[0, 106, 214, 0, 0, 0, 0, 0, 0, 31, 255, 61, 0],
            &[0, 153, 156, 0, 0, 0, 0, 0, 0, 0, 243, 71, 0],
            &[0, 163, 146, 0, 0, 0, 0, 0, 0, 1, 251, 56, 0],
            &[0, 132, 183, 0, 0, 0, 0, 0, 0, 43, 253, 13, 0],
            &[0, 48, 249, 37, 0, 0, 0, 0, 0, 150, 180, 0, 0],
            &[0, 0, 144, 218, 57, 0, 0, 11, 126, 241, 41, 0, 0],
            &[0, 0, 1, 110, 227, 233, 218, 246, 196, 49, 0, 0, 0],
            &[0, 0, 0, 0, 2, 30, 47, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ñ' / 0xf1
        'ñ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 30, 28, 0, 0, 0, 45, 6, 0, 0],
            &[0, 0, 0, 113, 233, 232, 159, 15, 15, 224, 2, 0, 0],
            &[0, 0, 1, 223, 13, 5, 132, 241, 241, 101, 0, 0, 0],
            &[0, 0, 2, 28, 0, 0, 0, 10, 15, 0, 0, 0, 0],
            &[0, 2, 41, 0, 0, 41, 88, 77, 30, 0, 0, 0, 0],
            &[0, 12, 252, 14, 166, 221, 178, 196, 249, 141, 1, 0, 0],
            &[0, 12, 255, 173, 105, 0, 0, 0, 55, 249, 94, 0, 0],
            &[0, 12, 255, 173, 0, 0, 0, 0, 0, 162, 172, 0, 0],
            &[0, 12, 255, 87, 0, 0, 0, 0, 0, 123, 195, 0, 0],
            &[0, 12, 255, 57, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ò' / 0xf2
        'ò' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 22, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 223, 138, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 232, 72, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 213, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 27, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 72, 89, 60, 4, 0, 0, 0, 0],
            &[0, 0, 3, 135, 240, 191, 174, 213, 226, 70, 0, 0, 0],
            &[0, 0, 143, 213, 32, 0, 0, 0, 89, 246, 53, 0, 0],
            &[0, 35, 252, 44, 0, 0, 0, 0, 0, 141, 192, 0, 0],
            &[0, 116, 200, 0, 0, 0, 0, 0, 0, 46, 253, 20, 0],
            &[0, 152, 154, 0, 0, 0, 0, 0, 0, 3, 254, 56, 0],
            &[0, 168, 143, 0, 0, 0, 0, 0, 0, 0, 245, 71, 0],
            &[0, 152, 161, 0, 0, 0, 0, 0, 0, 8, 255, 54, 0],
            &[0, 105, 215, 0, 0, 0, 0, 0, 0, 62, 250, 10, 0],
            &[0, 22, 248, 69, 0, 0, 0, 0, 0, 172, 169, 0, 0],
            &[0, 0, 111, 236, 72, 0, 0, 13, 140, 237, 32, 0, 0],
            &[0, 0, 0, 95, 225, 236, 219, 246, 189, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 47, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ó' / 0xf3
        'ó' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 22, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 138, 222, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 232, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 214, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 72, 89, 60, 4, 0, 0, 0, 0],
            &[0, 0, 3, 135, 240, 191, 174, 213, 226, 70, 0, 0, 0],
            &[0, 0, 143, 213, 32, 0, 0, 0, 89, 246, 53, 0, 0],
            &[0, 35, 252, 44, 0, 0, 0, 0, 0, 141, 192, 0, 0],
            &[0, 116, 200, 0, 0, 0, 0, 0, 0, 46, 253, 20, 0],
            &[0, 152, 154, 0, 0, 0, 0, 0, 0, 3, 254, 56, 0],
            &[0, 168, 143, 0, 0, 0, 0, 0, 0, 0, 245, 71, 0],
            &[0, 152, 161, 0, 0, 0, 0, 0, 0, 8, 255, 54, 0],
            &[0, 105, 215, 0, 0, 0, 0, 0, 0, 62, 250, 10, 0],
            &[0, 22, 248, 69, 0, 0, 0, 0, 0, 172, 169, 0, 0],
            &[0, 0, 111, 236, 72, 0, 0, 13, 140, 237, 32, 0, 0],
            &[0, 0, 0, 95, 225, 236, 219, 246, 189, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 47, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ô' / 0xf4
        'ô' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 169, 254, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 196, 68, 226, 32, 0, 0, 0, 0],
            &[0, 0, 0, 72, 198, 17, 0, 73, 203, 16, 0, 0, 0],
            &[0, 0, 0, 54, 14, 0, 0, 0, 41, 27, 0, 0, 0],
            &[0, 0, 0, 0, 18, 72, 89, 60, 4, 0, 0, 0, 0],
            &[0, 0, 3, 135, 240, 191, 174, 213, 226, 70, 0, 0, 0],
            &[0, 0, 143, 213, 32, 0, 0, 0, 89, 246, 53, 0, 0],
            &[0, 35, 252, 44, 0, 0, 0, 0, 0, 141, 192, 0, 0],
            &[0, 116, 200, 0, 0, 0, 0, 0, 0, 46, 253, 20, 0],
            &[0, 152, 154, 0, 0, 0, 0, 0, 0, 3, 254, 56, 0],
            &[0, 168, 143, 0, 0, 0, 0, 0, 0, 0, 245, 71, 0],
            &[0, 152, 161, 0, 0, 0, 0, 0, 0, 8, 255, 54, 0],
            &[0, 105, 215, 0, 0, 0, 0, 0, 0, 62, 250, 10, 0],
            &[0, 22, 248, 69, 0, 0, 0, 0, 0, 172, 169, 0, 0],
            &[0, 0, 111, 236, 72, 0, 0, 13, 140, 237, 32, 0, 0],
            &[0, 0, 0, 95, 225, 236, 219, 246, 189, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 47, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'õ' / 0xf5
        'õ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 17, 0, 0, 0, 51, 0, 0, 0],
            &[0, 0, 0, 165, 224, 236, 123, 4, 53, 189, 0, 0, 0],
            &[0, 0, 38, 199, 0, 19, 168, 238, 236, 59, 0, 0, 0],
            &[0, 0, 10, 20, 0, 0, 0, 18, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 72, 89, 60, 4, 0, 0, 0, 0],
            &[0, 0, 3, 135, 240, 191, 174, 213, 226, 70, 0, 0, 0],
            &[0, 0, 143, 213, 32, 0, 0, 0, 89, 246, 53, 0, 0],
            &[0, 35, 252, 44, 0, 0, 0, 0, 0, 141, 192, 0, 0],
            &[0, 116, 200, 0, 0, 0, 0, 0, 0, 46, 253, 20, 0],
            &[0, 152, 154, 0, 0, 0, 0, 0, 0, 3, 254, 56, 0],
            &[0, 168, 143, 0, 0, 0, 0, 0, 0, 0, 245, 71, 0],
            &[0, 152, 161, 0, 0, 0, 0, 0, 0, 8, 255, 54, 0],
            &[0, 105, 215, 0, 0, 0, 0, 0, 0, 62, 250, 10, 0],
            &[0, 22, 248, 69, 0, 0, 0, 0, 0, 172, 169, 0, 0],
            &[0, 0, 111, 236, 72, 0, 0, 13, 140, 237, 32, 0, 0],
            &[0, 0, 0, 95, 225, 236, 219, 246, 189, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 47, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ö' / 0xf6
        'ö' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 24, 0, 0, 0, 36, 1, 0, 0, 0],
            &[0, 0, 0, 172, 215, 0, 0, 55, 255, 75, 0, 0, 0],
            &[0, 0, 0, 86, 115, 0, 0, 18, 154, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 72, 89, 60, 4, 0, 0, 0, 0],
            &[0, 0, 3, 135, 240, 191, 174, 213, 226, 70, 0, 0, 0],
            &[0, 0, 143, 213, 32, 0, 0, 0, 89, 246, 53, 0, 0],
            &[0, 35, 252, 44, 0, 0, 0, 0, 0, 141, 192, 0, 0],
            &[0, 116, 200, 0, 0, 0, 0, 0, 0, 46, 253, 20, 0],
            &[0, 152, 154, 0, 0, 0, 0, 0, 0, 3, 254, 56, 0],
            &[0, 168, 143, 0, 0, 0, 0, 0, 0, 0, 245, 71, 0],
            &[0, 152, 161, 0, 0, 0, 0, 0, 0, 8, 255, 54, 0],
            &[0, 105, 215, 0, 0, 0, 0, 0, 0, 62, 250, 10, 0],
            &[0, 22, 248, 69, 0, 0, 0, 0, 0, 172, 169, 0, 0],
            &[0, 0, 111, 236, 72, 0, 0, 13, 140, 237, 32, 0, 0],
            &[0, 0, 0, 95, 225, 236, 219, 246, 189, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 47, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '÷' / 0xf7
        '÷' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 98, 233, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 78, 198, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 99, 234, 234, 234, 234, 234, 234, 234, 234, 234, 5, 0],
            &[0, 9, 23, 23, 23, 23, 23, 23, 23, 23, 23, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 55, 150, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 118, 252, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ø' / 0xf8
        'ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 72, 89, 60, 4, 2, 137, 19, 0],
            &[0, 0, 3, 135, 240, 191, 174, 213, 226, 197, 190, 3, 0],
            &[0, 0, 143, 213, 32, 0, 0, 0, 157, 255, 77, 0, 0],
            &[0, 35, 252, 44, 0, 0, 0, 26, 230, 205, 192, 0, 0],
            &[0, 116, 200, 0, 0, 0, 4, 194, 122, 46, 253, 20, 0],
            &[0, 152, 154, 0, 0, 0, 137, 182, 1, 3, 254, 56, 0],
            &[0, 168, 143, 0, 0, 76, 225, 19, 0, 0, 245, 71, 0],
            &[0, 152, 161, 0, 32, 232, 57, 0, 0, 8, 255, 54, 0],
            &[0, 105, 215, 6, 202, 113, 0, 0, 0, 62, 250, 10, 0],
            &[0, 22, 248, 217, 174, 0, 0, 0, 0, 172, 169, 0, 0],
            &[0, 0, 198, 255, 88, 0, 0, 13, 140, 237, 32, 0, 0],
            &[0, 38, 234, 145, 225, 236, 219, 246, 189, 41, 0, 0, 0],
            &[0, 29, 79, 0, 0, 33, 47, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ù' / 0xf9
        'ù' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 22, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 223, 138, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 232, 72, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 213, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 27, 41, 0, 0, 0, 0, 0],
            &[0, 6, 44, 2, 0, 0, 0, 0, 0, 24, 29, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 17, 0, 0, 0, 0, 0, 152, 170, 0, 0],
            &[0, 24, 255, 35, 0, 0, 0, 0, 0, 189, 170, 0, 0],
            &[0, 0, 244, 87, 0, 0, 0, 0, 33, 249, 170, 0, 0],
            &[0, 0, 143, 218, 40, 0, 0, 41, 197, 155, 170, 0, 0],
            &[0, 0, 14, 173, 253, 227, 226, 213, 60, 92, 170, 0, 0],
            &[0, 0, 0, 0, 18, 48, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ú' / 0xfa
        'ú' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 22, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 138, 222, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 232, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 214, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 27, 0, 0, 0, 0, 0, 0],
            &[0, 6, 44, 2, 0, 0, 0, 0, 0, 24, 29, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 17, 0, 0, 0, 0, 0, 152, 170, 0, 0],
            &[0, 24, 255, 35, 0, 0, 0, 0, 0, 189, 170, 0, 0],
            &[0, 0, 244, 87, 0, 0, 0, 0, 33, 249, 170, 0, 0],
            &[0, 0, 143, 218, 40, 0, 0, 41, 197, 155, 170, 0, 0],
            &[0, 0, 14, 173, 253, 227, 226, 213, 60, 92, 170, 0, 0],
            &[0, 0, 0, 0, 18, 48, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'û' / 0xfb
        'û' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 169, 254, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 196, 68, 226, 32, 0, 0, 0, 0],
            &[0, 0, 0, 72, 198, 17, 0, 73, 203, 16, 0, 0, 0],
            &[0, 0, 0, 54, 14, 0, 0, 0, 41, 27, 0, 0, 0],
            &[0, 6, 44, 2, 0, 0, 0, 0, 0, 24, 29, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 17, 0, 0, 0, 0, 0, 152, 170, 0, 0],
            &[0, 24, 255, 35, 0, 0, 0, 0, 0, 189, 170, 0, 0],
            &[0, 0, 244, 87, 0, 0, 0, 0, 33, 249, 170, 0, 0],
            &[0, 0, 143, 218, 40, 0, 0, 41, 197, 155, 170, 0, 0],
            &[0, 0, 14, 173, 253, 227, 226, 213, 60, 92, 170, 0, 0],
            &[0, 0, 0, 0, 18, 48, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ü' / 0xfc
        'ü' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 24, 0, 0, 0, 36, 1, 0, 0, 0],
            &[0, 0, 0, 172, 215, 0, 0, 55, 255, 75, 0, 0, 0],
            &[0, 0, 0, 86, 115, 0, 0, 18, 154, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 44, 2, 0, 0, 0, 0, 0, 24, 29, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 17, 0, 0, 0, 0, 0, 152, 170, 0, 0],
            &[0, 24, 255, 35, 0, 0, 0, 0, 0, 189, 170, 0, 0],
            &[0, 0, 244, 87, 0, 0, 0, 0, 33, 249, 170, 0, 0],
            &[0, 0, 143, 218, 40, 0, 0, 41, 197, 155, 170, 0, 0],
            &[0, 0, 14, 173, 253, 227, 226, 213, 60, 92, 170, 0, 0],
            &[0, 0, 0, 0, 18, 48, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ý' / 0xfd
        'ý' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 3, 22, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 138, 222, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 232, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 214, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 27, 0, 0, 0, 0, 0, 0],
            &[0, 23, 29, 0, 0, 0, 0, 0, 0, 2, 44, 8, 0],
            &[0, 78, 228, 3, 0, 0, 0, 0, 0, 68, 241, 9, 0],
            &[0, 3, 228, 75, 0, 0, 0, 0, 0, 160, 154, 0, 0],
            &[0, 0, 130, 174, 0, 0, 0, 0, 9, 242, 58, 0, 0],
            &[0, 0, 31, 248, 23, 0, 0, 0, 87, 217, 0, 0, 0],
            &[0, 0, 0, 183, 117, 0, 0, 0, 179, 121, 0, 0, 0],
            &[0, 0, 0, 81, 216, 0, 0, 20, 248, 27, 0, 0, 0],
            &[0, 0, 0, 4, 231, 60, 0, 107, 184, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 158, 0, 198, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 241, 41, 238, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 186, 195, 150, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 255, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 86, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 182, 117, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 243, 19, 0, 0, 0, 0, 0, 0],
            &[0, 2, 33, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'þ' / 0xfe
        'þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 122, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 38, 0, 41, 88, 83, 26, 0, 0, 0, 0],
            &[0, 12, 255, 45, 156, 214, 173, 204, 247, 132, 0, 0, 0],
            &[0, 12, 255, 170, 103, 0, 0, 0, 55, 235, 102, 0, 0],
            &[0, 12, 255, 186, 0, 0, 0, 0, 0, 121, 221, 0, 0],
            &[0, 12, 255, 91, 0, 0, 0, 0, 0, 37, 255, 32, 0],
            &[0, 12, 255, 58, 0, 0, 0, 0, 0, 10, 255, 64, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 250, 70, 0],
            &[0, 12, 255, 67, 0, 0, 0, 0, 0, 9, 255, 52, 0],
            &[0, 12, 255, 105, 0, 0, 0, 0, 0, 53, 251, 15, 0],
            &[0, 12, 255, 205, 0, 0, 0, 0, 0, 146, 187, 0, 0],
            &[0, 12, 255, 156, 139, 8, 0, 3, 101, 246, 53, 0, 0],
            &[0, 12, 255, 40, 127, 237, 215, 238, 217, 69, 0, 0, 0],
            &[0, 12, 255, 38, 0, 11, 46, 30, 1, 0, 0, 0, 0],
            &[0, 12, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 12, 255, 38, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ÿ' / 0xff
        'ÿ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 24, 0, 0, 0, 36, 1, 0, 0, 0],
            &[0, 0, 0, 172, 215, 0, 0, 55, 255, 75, 0, 0, 0],
            &[0, 0, 0, 86, 115, 0, 0, 18, 154, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 23, 29, 0, 0, 0, 0, 0, 0, 2, 44, 8, 0],
            &[0, 78, 228, 3, 0, 0, 0, 0, 0, 68, 241, 9, 0],
            &[0, 3, 228, 75, 0, 0, 0, 0, 0, 160, 154, 0, 0],
            &[0, 0, 130, 174, 0, 0, 0, 0, 9, 242, 58, 0, 0],
            &[0, 0, 31, 248, 23, 0, 0, 0, 87, 217, 0, 0, 0],
            &[0, 0, 0, 183, 117, 0, 0, 0, 179, 121, 0, 0, 0],
            &[0, 0, 0, 81, 216, 0, 0, 20, 248, 27, 0, 0, 0],
            &[0, 0, 0, 4, 231, 60, 0, 107, 184, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 158, 0, 198, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 241, 41, 238, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 186, 195, 150, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 255, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 86, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 182, 117, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 243, 19, 0, 0, 0, 0, 0, 0],
            &[0, 2, 33, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ā' / 0x100
        'Ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 149, 249, 249, 249, 249, 249, 49, 0, 0, 0],
            &[0, 0, 0, 4, 8, 8, 8, 8, 8, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 134, 254, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 222, 220, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 238, 90, 210, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 158, 10, 244, 43, 0, 0, 0, 0],
            &[0, 0, 0, 3, 231, 70, 0, 167, 132, 0, 0, 0, 0],
            &[0, 0, 0, 68, 234, 4, 0, 80, 221, 0, 0, 0, 0],
            &[0, 0, 0, 157, 151, 0, 0, 7, 241, 55, 0, 0, 0],
            &[0, 0, 7, 239, 63, 0, 0, 0, 162, 144, 0, 0, 0],
            &[0, 0, 80, 255, 68, 66, 66, 66, 141, 241, 2, 0, 0],
            &[0, 0, 169, 255, 218, 218, 218, 218, 224, 255, 67, 0, 0],
            &[0, 12, 246, 56, 0, 0, 0, 0, 0, 156, 156, 0, 0],
            &[0, 92, 223, 0, 0, 0, 0, 0, 0, 69, 238, 6, 0],
            &[0, 181, 137, 0, 0, 0, 0, 0, 0, 3, 234, 78, 0],
            &[19, 250, 49, 0, 0, 0, 0, 0, 0, 0, 151, 168, 0],
            &[103, 217, 0, 0, 0, 0, 0, 0, 0, 0, 64, 245, 11],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ā' / 0x101
        'ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 39, 39, 39, 39, 39, 11, 0, 0, 0],
            &[0, 0, 0, 107, 218, 218, 218, 218, 218, 66, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 82, 89, 59, 11, 0, 0, 0, 0],
            &[0, 0, 42, 205, 234, 191, 175, 212, 232, 85, 0, 0, 0],
            &[0, 0, 11, 81, 4, 0, 0, 0, 97, 250, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 209, 101, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 171, 125, 0, 0],
            &[0, 0, 0, 0, 32, 88, 115, 130, 139, 215, 127, 0, 0],
            &[0, 0, 33, 191, 231, 162, 126, 107, 98, 195, 127, 0, 0],
            &[0, 6, 220, 143, 5, 0, 0, 0, 0, 166, 127, 0, 0],
            &[0, 59, 241, 4, 0, 0, 0, 0, 0, 200, 127, 0, 0],
            &[0, 63, 238, 0, 0, 0, 0, 0, 53, 254, 127, 0, 0],
            &[0, 14, 236, 123, 3, 0, 3, 79, 200, 180, 127, 0, 0],
            &[0, 0, 63, 218, 240, 216, 239, 173, 33, 134, 127, 0, 0],
            &[0, 0, 0, 1, 27, 47, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ă' / 0x102
        'Ă' => &[
            &[0, 0, 0, 35, 18, 0, 0, 0, 42, 11, 0, 0, 0],
            &[0, 0, 0, 111, 152, 0, 0, 13, 224, 13, 0, 0, 0],
            &[0, 0, 0, 9, 197, 191, 152, 217, 109, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 57, 77, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 134, 254, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 222, 220, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 238, 90, 210, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 158, 10, 244, 43, 0, 0, 0, 0],
            &[0, 0, 0, 3, 231, 70, 0, 167, 132, 0, 0, 0, 0],
            &[0, 0, 0, 68, 234, 4, 0, 80, 221, 0, 0, 0, 0],
            &[0, 0, 0, 157, 151, 0, 0, 7, 241, 55, 0, 0, 0],
            &[0, 0, 7, 239, 63, 0, 0, 0, 162, 144, 0, 0, 0],
            &[0, 0, 80, 255, 68, 66, 66, 66, 141, 241, 2, 0, 0],
            &[0, 0, 169, 255, 218, 218, 218, 218, 224, 255, 67, 0, 0],
            &[0, 12, 246, 56, 0, 0, 0, 0, 0, 156, 156, 0, 0],
            &[0, 92, 223, 0, 0, 0, 0, 0, 0, 69, 238, 6, 0],
            &[0, 181, 137, 0, 0, 0, 0, 0, 0, 3, 234, 78, 0],
            &[19, 250, 49, 0, 0, 0, 0, 0, 0, 0, 151, 168, 0],
            &[103, 217, 0, 0, 0, 0, 0, 0, 0, 0, 64, 245, 11],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ă' / 0x103
        'ă' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 49, 47, 0, 0, 0, 65, 29, 0, 0, 0],
            &[0, 0, 0, 67, 201, 12, 0, 21, 219, 25, 0, 0, 0],
            &[0, 0, 0, 0, 132, 231, 197, 227, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 33, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 82, 89, 59, 11, 0, 0, 0, 0],
            &[0, 0, 42, 205, 234, 191, 175, 212, 232, 85, 0, 0, 0],
            &[0, 0, 11, 81, 4, 0, 0, 0, 97, 250, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 209, 101, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 171, 125, 0, 0],
            &[0, 0, 0, 0, 32, 88, 115, 130, 139, 215, 127, 0, 0],
            &[0, 0, 33, 191, 231, 162, 126, 107, 98, 195, 127, 0, 0],
            &[0, 6, 220, 143, 5, 0, 0, 0, 0, 166, 127, 0, 0],
            &[0, 59, 241, 4, 0, 0, 0, 0, 0, 200, 127, 0, 0],
            &[0, 63, 238, 0, 0, 0, 0, 0, 53, 254, 127, 0, 0],
            &[0, 14, 236, 123, 3, 0, 3, 79, 200, 180, 127, 0, 0],
            &[0, 0, 63, 218, 240, 216, 239, 173, 33, 134, 127, 0, 0],
            &[0, 0, 0, 1, 27, 47, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ą' / 0x104
        'Ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 134, 254, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 222, 220, 121, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 238, 90, 210, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 145, 158, 10, 244, 43, 0, 0, 0, 0],
            &[0, 0, 0, 3, 231, 70, 0, 167, 132, 0, 0, 0, 0],
            &[0, 0, 0, 68, 234, 4, 0, 80, 221, 0, 0, 0, 0],
            &[0, 0, 0, 157, 151, 0, 0, 7, 241, 55, 0, 0, 0],
            &[0, 0, 7, 239, 63, 0, 0, 0, 162, 144, 0, 0, 0],
            &[0, 0, 80, 255, 68, 66, 66, 66, 141, 241, 2, 0, 0],
            &[0, 0, 169, 255, 218, 218, 218, 218, 224, 255, 67, 0, 0],
            &[0, 12, 246, 56, 0, 0, 0, 0, 0, 156, 156, 0, 0],
            &[0, 92, 223, 0, 0, 0, 0, 0, 0, 69, 238, 6, 0],
            &[0, 181, 137, 0, 0, 0, 0, 0, 0, 3, 234, 78, 0],
            &[19, 250, 49, 0, 0, 0, 0, 0, 0, 0, 151, 168, 0],
            &[103, 217, 0, 0, 0, 0, 0, 0, 0, 0, 64, 255, 13],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 94, 174, 5],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 36, 208, 7, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 110, 148, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 141, 73],
        ],
        // letter: 'ą' / 0x105
        'ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 82, 89, 59, 11, 0, 0, 0, 0],
            &[0, 0, 42, 205, 234, 191, 175, 212, 232, 85, 0, 0, 0],
            &[0, 0, 11, 81, 4, 0, 0, 0, 97, 250, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 209, 101, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 171, 125, 0, 0],
            &[0, 0, 0, 0, 32, 88, 115, 130, 139, 215, 127, 0, 0],
            &[0, 0, 33, 191, 231, 162, 126, 107, 98, 195, 127, 0, 0],
            &[0, 6, 220, 143, 5, 0, 0, 0, 0, 166, 127, 0, 0],
            &[0, 59, 241, 4, 0, 0, 0, 0, 0, 200, 127, 0, 0],
            &[0, 63, 238, 0, 0, 0, 0, 0, 53, 254, 127, 0, 0],
            &[0, 14, 236, 123, 3, 0, 3, 79, 200, 180, 127, 0, 0],
            &[0, 0, 63, 218, 240, 216, 239, 173, 33, 159, 138, 0, 0],
            &[0, 0, 0, 1, 27, 47, 18, 0, 46, 199, 27, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 210, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 44, 214, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 59, 137, 108, 0, 0],
        ],
        // letter: 'Ć' / 0x106
        'Ć' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 47, 229, 69, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 11, 212, 127, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 172, 136, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 86, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 42, 35, 9, 0, 0, 0],
            &[0, 0, 0, 0, 69, 196, 253, 237, 244, 254, 193, 59, 0],
            &[0, 0, 0, 118, 245, 121, 22, 0, 0, 26, 111, 37, 0],
            &[0, 0, 69, 250, 63, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 204, 141, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 255, 36, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 228, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 128, 192, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 127, 193, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 104, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 255, 25, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 224, 124, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 241, 47, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 153, 237, 112, 22, 0, 4, 38, 108, 13, 0],
            &[0, 0, 0, 1, 110, 224, 255, 245, 253, 251, 191, 13, 0],
            &[0, 0, 0, 0, 0, 0, 28, 49, 34, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ć' / 0x107
        'ć' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 12, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 13, 224, 140, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 169, 171, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 123, 171, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 70, 91, 72, 25, 0, 0, 0],
            &[0, 0, 0, 19, 167, 249, 212, 188, 206, 248, 96, 0, 0],
            &[0, 0, 11, 215, 190, 34, 0, 0, 0, 18, 18, 0, 0],
            &[0, 0, 134, 215, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 226, 104, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 255, 45, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 255, 27, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 228, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 139, 214, 7, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 218, 194, 51, 0, 0, 1, 48, 49, 0, 0],
            &[0, 0, 0, 18, 156, 244, 243, 227, 250, 229, 80, 0, 0],
            &[0, 0, 0, 0, 0, 6, 34, 46, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĉ' / 0x108
        'Ĉ' => &[
            &[0, 0, 0, 0, 0, 0, 72, 226, 106, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 225, 103, 229, 51, 0, 0, 0],
            &[0, 0, 0, 0, 12, 199, 81, 0, 58, 213, 27, 0, 0],
            &[0, 0, 0, 0, 53, 65, 0, 0, 0, 49, 69, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 42, 35, 9, 0, 0, 0],
            &[0, 0, 0, 0, 69, 196, 253, 237, 244, 254, 193, 59, 0],
            &[0, 0, 0, 118, 245, 121, 22, 0, 0, 26, 111, 37, 0],
            &[0, 0, 69, 250, 63, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 204, 141, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 255, 36, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 228, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 128, 192, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 127, 193, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 104, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 255, 25, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 224, 124, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 241, 47, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 153, 237, 112, 22, 0, 4, 38, 108, 13, 0],
            &[0, 0, 0, 1, 110, 224, 255, 245, 253, 251, 191, 13, 0],
            &[0, 0, 0, 0, 0, 0, 28, 49, 34, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĉ' / 0x109
        'ĉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 239, 219, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 200, 105, 151, 167, 0, 0, 0, 0],
            &[0, 0, 0, 1, 166, 120, 0, 2, 162, 128, 0, 0, 0],
            &[0, 0, 0, 11, 57, 0, 0, 0, 2, 66, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 70, 91, 72, 25, 0, 0, 0],
            &[0, 0, 0, 19, 167, 249, 212, 188, 206, 248, 96, 0, 0],
            &[0, 0, 11, 215, 190, 34, 0, 0, 0, 18, 18, 0, 0],
            &[0, 0, 134, 215, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 226, 104, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 255, 45, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 255, 27, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 228, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 139, 214, 7, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 218, 194, 51, 0, 0, 1, 48, 49, 0, 0],
            &[0, 0, 0, 18, 156, 244, 243, 227, 250, 229, 80, 0, 0],
            &[0, 0, 0, 0, 0, 6, 34, 46, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ċ' / 0x10a
        'Ċ' => &[
            &[0, 0, 0, 0, 0, 0, 5, 72, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 84, 255, 115, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 27, 157, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 42, 35, 9, 0, 0, 0],
            &[0, 0, 0, 0, 69, 196, 253, 237, 244, 254, 193, 59, 0],
            &[0, 0, 0, 118, 245, 121, 22, 0, 0, 26, 111, 37, 0],
            &[0, 0, 69, 250, 63, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 204, 141, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 255, 36, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 228, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 128, 192, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 127, 193, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 104, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 255, 25, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 224, 124, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 241, 47, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 153, 237, 112, 22, 0, 4, 38, 108, 13, 0],
            &[0, 0, 0, 1, 110, 224, 255, 245, 253, 251, 191, 13, 0],
            &[0, 0, 0, 0, 0, 0, 28, 49, 34, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ċ' / 0x10b
        'ċ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 123, 92, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 247, 206, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 60, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 70, 91, 72, 25, 0, 0, 0],
            &[0, 0, 0, 19, 167, 249, 212, 188, 206, 248, 96, 0, 0],
            &[0, 0, 11, 215, 190, 34, 0, 0, 0, 18, 18, 0, 0],
            &[0, 0, 134, 215, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 226, 104, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 255, 45, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 255, 27, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 228, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 139, 214, 7, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 218, 194, 51, 0, 0, 1, 48, 49, 0, 0],
            &[0, 0, 0, 18, 156, 244, 243, 227, 250, 229, 80, 0, 0],
            &[0, 0, 0, 0, 0, 6, 34, 46, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Č' / 0x10c
        'Č' => &[
            &[0, 0, 0, 0, 68, 169, 7, 0, 2, 152, 97, 0, 0],
            &[0, 0, 0, 0, 0, 127, 181, 5, 161, 154, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 178, 218, 197, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 115, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 42, 35, 9, 0, 0, 0],
            &[0, 0, 0, 0, 69, 196, 253, 237, 244, 254, 193, 59, 0],
            &[0, 0, 0, 118, 245, 121, 22, 0, 0, 26, 111, 37, 0],
            &[0, 0, 69, 250, 63, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 204, 141, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 41, 255, 36, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 98, 228, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 128, 192, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 141, 180, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 127, 193, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 104, 219, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 46, 255, 25, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 224, 124, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 93, 241, 47, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 153, 237, 112, 22, 0, 4, 38, 108, 13, 0],
            &[0, 0, 0, 1, 110, 224, 255, 245, 253, 251, 191, 13, 0],
            &[0, 0, 0, 0, 0, 0, 28, 49, 34, 8, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'č' / 0x10d
        'č' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 11, 0, 0, 0, 0, 14, 0, 0, 0],
            &[0, 0, 0, 11, 198, 75, 0, 0, 118, 172, 0, 0, 0],
            &[0, 0, 0, 0, 26, 223, 67, 115, 197, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 60, 242, 233, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 55, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 70, 91, 72, 25, 0, 0, 0],
            &[0, 0, 0, 19, 167, 249, 212, 188, 206, 248, 96, 0, 0],
            &[0, 0, 11, 215, 190, 34, 0, 0, 0, 18, 18, 0, 0],
            &[0, 0, 134, 215, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 226, 104, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 255, 45, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 32, 255, 27, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 16, 255, 44, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 228, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 139, 214, 7, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 218, 194, 51, 0, 0, 1, 48, 49, 0, 0],
            &[0, 0, 0, 18, 156, 244, 243, 227, 250, 229, 80, 0, 0],
            &[0, 0, 0, 0, 0, 6, 34, 46, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ď' / 0x10e
        'Ď' => &[
            &[0, 0, 0, 171, 73, 0, 0, 46, 189, 16, 0, 0, 0],
            &[0, 0, 0, 22, 219, 71, 44, 225, 46, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 237, 230, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 77, 93, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 254, 254, 254, 241, 213, 148, 44, 0, 0, 0, 0],
            &[0, 65, 246, 19, 20, 37, 73, 163, 248, 104, 0, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 77, 250, 68, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 137, 213, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 30, 255, 41, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 0, 220, 99, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 0, 194, 122, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 0, 180, 134, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 0, 192, 115, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 0, 229, 90, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 39, 254, 24, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 154, 191, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 98, 248, 44, 0, 0],
            &[0, 65, 246, 23, 25, 42, 88, 181, 240, 75, 0, 0, 0],
            &[0, 65, 255, 255, 255, 235, 205, 128, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ď' / 0x10f
        'ď' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 95, 203, 0, 44, 239, 11],
            &[0, 0, 0, 0, 0, 0, 0, 99, 211, 0, 101, 175, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 211, 0, 173, 61, 0],
            &[0, 0, 0, 0, 0, 0, 0, 99, 211, 0, 79, 0, 0],
            &[0, 0, 7, 68, 91, 53, 0, 99, 211, 0, 0, 0, 0],
            &[0, 56, 229, 218, 175, 214, 163, 98, 211, 0, 0, 0, 0],
            &[15, 236, 136, 1, 0, 1, 128, 195, 211, 0, 0, 0, 0],
            &[112, 221, 3, 0, 0, 0, 2, 220, 211, 0, 0, 0, 0],
            &[178, 145, 0, 0, 0, 0, 0, 148, 211, 0, 0, 0, 0],
            &[209, 106, 0, 0, 0, 0, 0, 113, 211, 0, 0, 0, 0],
            &[223, 94, 0, 0, 0, 0, 0, 102, 211, 0, 0, 0, 0],
            &[213, 112, 0, 0, 0, 0, 0, 121, 211, 0, 0, 0, 0],
            &[178, 143, 0, 0, 0, 0, 0, 156, 211, 0, 0, 0, 0],
            &[109, 227, 3, 0, 0, 0, 8, 238, 211, 0, 0, 0, 0],
            &[13, 229, 159, 9, 0, 11, 161, 160, 211, 0, 0, 0, 0],
            &[0, 44, 208, 251, 218, 233, 115, 54, 211, 0, 0, 0, 0],
            &[0, 0, 0, 32, 43, 10, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Đ' / 0x110
        'Đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 254, 254, 254, 246, 219, 168, 67, 0, 0, 0, 0],
            &[0, 12, 255, 60, 19, 32, 61, 142, 243, 149, 3, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 44, 230, 121, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 83, 248, 19, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 1, 231, 95, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 166, 153, 0],
            &[21, 33, 255, 65, 21, 20, 0, 0, 0, 0, 140, 175, 0],
            &[252, 255, 255, 255, 252, 243, 0, 0, 0, 0, 126, 188, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 139, 169, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 175, 144, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 6, 235, 76, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 100, 236, 8, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 58, 241, 92, 0, 0],
            &[0, 12, 255, 63, 24, 37, 75, 157, 248, 117, 0, 0, 0],
            &[0, 12, 255, 255, 255, 240, 213, 149, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'đ' / 0x111
        'đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 135, 164, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 140, 170, 0, 0, 0],
            &[0, 0, 0, 0, 147, 236, 236, 236, 255, 255, 236, 141, 0],
            &[0, 0, 0, 0, 6, 11, 11, 11, 151, 181, 11, 6, 0],
            &[0, 0, 0, 23, 65, 71, 20, 0, 140, 170, 0, 0, 0],
            &[0, 8, 159, 247, 200, 194, 233, 105, 135, 170, 0, 0, 0],
            &[0, 167, 211, 32, 0, 0, 17, 187, 187, 170, 0, 0, 0],
            &[45, 254, 43, 0, 0, 0, 0, 26, 249, 170, 0, 0, 0],
            &[119, 205, 0, 0, 0, 0, 0, 0, 193, 170, 0, 0, 0],
            &[153, 161, 0, 0, 0, 0, 0, 0, 155, 170, 0, 0, 0],
            &[169, 148, 0, 0, 0, 0, 0, 0, 143, 170, 0, 0, 0],
            &[158, 168, 0, 0, 0, 0, 0, 0, 163, 170, 0, 0, 0],
            &[121, 204, 0, 0, 0, 0, 0, 0, 198, 170, 0, 0, 0],
            &[47, 255, 49, 0, 0, 0, 0, 40, 254, 170, 0, 0, 0],
            &[0, 168, 219, 49, 0, 0, 32, 194, 158, 170, 0, 0, 0],
            &[0, 7, 150, 248, 235, 220, 223, 70, 95, 170, 0, 0, 0],
            &[0, 0, 0, 11, 45, 39, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ē' / 0x112
        'Ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 107, 249, 249, 249, 249, 249, 91, 0, 0, 0],
            &[0, 0, 0, 3, 8, 8, 8, 8, 8, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 196, 0, 0],
            &[0, 0, 203, 122, 25, 25, 25, 25, 25, 25, 19, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 170, 107, 107, 107, 107, 107, 107, 38, 0, 0],
            &[0, 0, 203, 207, 171, 171, 171, 171, 171, 171, 60, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 124, 28, 28, 28, 28, 28, 28, 22, 0, 0],
            &[0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ē' / 0x113
        'ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 20, 39, 39, 39, 39, 39, 11, 0, 0, 0],
            &[0, 0, 0, 112, 218, 218, 218, 218, 218, 61, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 61, 89, 70, 14, 0, 0, 0, 0],
            &[0, 0, 0, 90, 232, 217, 176, 197, 240, 101, 0, 0, 0],
            &[0, 0, 87, 242, 83, 0, 0, 0, 56, 239, 74, 0, 0],
            &[0, 7, 233, 101, 0, 0, 0, 0, 0, 118, 201, 0, 0],
            &[0, 74, 242, 4, 0, 0, 0, 0, 0, 42, 254, 15, 0],
            &[0, 121, 222, 114, 114, 114, 114, 114, 114, 124, 255, 40, 0],
            &[0, 140, 220, 143, 143, 143, 143, 143, 143, 143, 143, 27, 0],
            &[0, 122, 194, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 75, 241, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 230, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 243, 124, 19, 0, 0, 3, 59, 107, 0, 0],
            &[0, 0, 0, 52, 184, 249, 229, 226, 252, 215, 113, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĕ' / 0x114
        'Ĕ' => &[
            &[0, 0, 0, 25, 28, 0, 0, 0, 32, 21, 0, 0, 0],
            &[0, 0, 0, 68, 192, 3, 0, 2, 197, 51, 0, 0, 0],
            &[0, 0, 0, 0, 163, 207, 152, 205, 148, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 47, 79, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 196, 0, 0],
            &[0, 0, 203, 122, 25, 25, 25, 25, 25, 25, 19, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 170, 107, 107, 107, 107, 107, 107, 38, 0, 0],
            &[0, 0, 203, 207, 171, 171, 171, 171, 171, 171, 60, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 124, 28, 28, 28, 28, 28, 28, 22, 0, 0],
            &[0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĕ' / 0x115
        'ĕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 51, 45, 0, 0, 0, 68, 26, 0, 0, 0],
            &[0, 0, 0, 72, 197, 10, 0, 23, 221, 21, 0, 0, 0],
            &[0, 0, 0, 0, 137, 231, 198, 226, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 32, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 61, 89, 70, 14, 0, 0, 0, 0],
            &[0, 0, 0, 90, 232, 217, 176, 197, 240, 101, 0, 0, 0],
            &[0, 0, 87, 242, 83, 0, 0, 0, 56, 239, 74, 0, 0],
            &[0, 7, 233, 101, 0, 0, 0, 0, 0, 118, 201, 0, 0],
            &[0, 74, 242, 4, 0, 0, 0, 0, 0, 42, 254, 15, 0],
            &[0, 121, 222, 114, 114, 114, 114, 114, 114, 124, 255, 40, 0],
            &[0, 140, 220, 143, 143, 143, 143, 143, 143, 143, 143, 27, 0],
            &[0, 122, 194, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 75, 241, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 230, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 243, 124, 19, 0, 0, 3, 59, 107, 0, 0],
            &[0, 0, 0, 52, 184, 249, 229, 226, 252, 215, 113, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ė' / 0x116
        'Ė' => &[
            &[0, 0, 0, 0, 0, 10, 72, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 108, 255, 92, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 158, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 196, 0, 0],
            &[0, 0, 203, 122, 25, 25, 25, 25, 25, 25, 19, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 170, 107, 107, 107, 107, 107, 107, 38, 0, 0],
            &[0, 0, 203, 207, 171, 171, 171, 171, 171, 171, 60, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 124, 28, 28, 28, 28, 28, 28, 22, 0, 0],
            &[0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ė' / 0x117
        'ė' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 149, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 255, 71, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 78, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 61, 89, 70, 14, 0, 0, 0, 0],
            &[0, 0, 0, 90, 232, 217, 176, 197, 240, 101, 0, 0, 0],
            &[0, 0, 87, 242, 83, 0, 0, 0, 56, 239, 74, 0, 0],
            &[0, 7, 233, 101, 0, 0, 0, 0, 0, 118, 201, 0, 0],
            &[0, 74, 242, 4, 0, 0, 0, 0, 0, 42, 254, 15, 0],
            &[0, 121, 222, 114, 114, 114, 114, 114, 114, 124, 255, 40, 0],
            &[0, 140, 220, 143, 143, 143, 143, 143, 143, 143, 143, 27, 0],
            &[0, 122, 194, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 75, 241, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 230, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 243, 124, 19, 0, 0, 3, 59, 107, 0, 0],
            &[0, 0, 0, 52, 184, 249, 229, 226, 252, 215, 113, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ę' / 0x118
        'Ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 196, 0, 0],
            &[0, 0, 203, 122, 25, 25, 25, 25, 25, 25, 19, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 170, 107, 107, 107, 107, 107, 107, 38, 0, 0],
            &[0, 0, 203, 207, 171, 171, 171, 171, 171, 171, 60, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 124, 28, 28, 28, 28, 28, 28, 22, 0, 0],
            &[0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 27, 199, 47, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 183, 68, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 245, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 43, 134, 128, 0, 0, 0],
        ],
        // letter: 'ę' / 0x119
        'ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 61, 89, 70, 14, 0, 0, 0, 0],
            &[0, 0, 0, 90, 232, 217, 176, 197, 240, 101, 0, 0, 0],
            &[0, 0, 87, 242, 83, 0, 0, 0, 56, 239, 74, 0, 0],
            &[0, 7, 233, 101, 0, 0, 0, 0, 0, 118, 201, 0, 0],
            &[0, 74, 242, 4, 0, 0, 0, 0, 0, 42, 254, 15, 0],
            &[0, 121, 222, 114, 114, 114, 114, 114, 114, 124, 255, 40, 0],
            &[0, 140, 220, 143, 143, 143, 143, 143, 143, 143, 143, 27, 0],
            &[0, 122, 194, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 75, 241, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 230, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 243, 124, 19, 0, 0, 3, 59, 107, 0, 0],
            &[0, 0, 0, 52, 184, 249, 229, 226, 252, 255, 184, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 129, 143, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 60, 200, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 134, 138, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 93, 140, 65, 0, 0],
        ],
        // letter: 'Ě' / 0x11a
        'Ě' => &[
            &[0, 0, 0, 88, 153, 3, 0, 7, 168, 76, 0, 0, 0],
            &[0, 0, 0, 0, 150, 161, 5, 181, 130, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 198, 218, 177, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 115, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 196, 0, 0],
            &[0, 0, 203, 122, 25, 25, 25, 25, 25, 25, 19, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 170, 107, 107, 107, 107, 107, 107, 38, 0, 0],
            &[0, 0, 203, 207, 171, 171, 171, 171, 171, 171, 60, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 203, 124, 28, 28, 28, 28, 28, 28, 22, 0, 0],
            &[0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ě' / 0x11b
        'ě' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 2, 0, 0, 0, 7, 8, 0, 0, 0],
            &[0, 0, 0, 97, 179, 8, 0, 32, 208, 51, 0, 0, 0],
            &[0, 0, 0, 0, 134, 179, 30, 220, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 189, 244, 124, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 22, 69, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 61, 89, 70, 14, 0, 0, 0, 0],
            &[0, 0, 0, 90, 232, 217, 176, 197, 240, 101, 0, 0, 0],
            &[0, 0, 87, 242, 83, 0, 0, 0, 56, 239, 74, 0, 0],
            &[0, 7, 233, 101, 0, 0, 0, 0, 0, 118, 201, 0, 0],
            &[0, 74, 242, 4, 0, 0, 0, 0, 0, 42, 254, 15, 0],
            &[0, 121, 222, 114, 114, 114, 114, 114, 114, 124, 255, 40, 0],
            &[0, 140, 220, 143, 143, 143, 143, 143, 143, 143, 143, 27, 0],
            &[0, 122, 194, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 75, 241, 7, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 230, 112, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 243, 124, 19, 0, 0, 3, 59, 107, 0, 0],
            &[0, 0, 0, 52, 184, 249, 229, 226, 252, 215, 113, 0, 0],
            &[0, 0, 0, 0, 0, 11, 38, 42, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => &[
            &[0, 0, 0, 0, 0, 27, 216, 162, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 193, 128, 205, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 138, 0, 23, 208, 66, 0, 0, 0],
            &[0, 0, 0, 24, 93, 0, 0, 0, 21, 97, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 32, 48, 20, 0, 0, 0, 0],
            &[0, 0, 0, 20, 144, 244, 250, 236, 255, 213, 83, 0, 0],
            &[0, 0, 25, 214, 196, 59, 3, 0, 11, 92, 64, 0, 0],
            &[0, 0, 176, 191, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 53, 252, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 193, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 193, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 218, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 236, 84, 0, 0, 0, 63, 164, 164, 164, 144, 0, 0],
            &[0, 228, 93, 0, 0, 0, 46, 120, 120, 163, 224, 0, 0],
            &[0, 208, 114, 0, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 161, 169, 0, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 90, 239, 15, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 5, 218, 149, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 0, 58, 241, 162, 39, 1, 0, 22, 140, 224, 0, 0],
            &[0, 0, 0, 44, 175, 253, 247, 243, 255, 206, 108, 0, 0],
            &[0, 0, 0, 0, 0, 8, 38, 45, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĝ' / 0x11d
        'ĝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 169, 254, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 196, 68, 226, 32, 0, 0, 0, 0],
            &[0, 0, 0, 72, 198, 17, 0, 73, 203, 16, 0, 0, 0],
            &[0, 0, 0, 54, 14, 0, 0, 0, 41, 27, 0, 0, 0],
            &[0, 0, 0, 0, 39, 87, 82, 31, 0, 14, 29, 0, 0],
            &[0, 0, 11, 175, 243, 183, 177, 226, 137, 100, 170, 0, 0],
            &[0, 0, 168, 197, 20, 0, 0, 4, 150, 216, 170, 0, 0],
            &[0, 43, 255, 47, 0, 0, 0, 0, 6, 229, 170, 0, 0],
            &[0, 118, 206, 0, 0, 0, 0, 0, 0, 160, 170, 0, 0],
            &[0, 157, 170, 0, 0, 0, 0, 0, 0, 129, 170, 0, 0],
            &[0, 169, 150, 0, 0, 0, 0, 0, 0, 119, 170, 0, 0],
            &[0, 158, 172, 0, 0, 0, 0, 0, 0, 137, 170, 0, 0],
            &[0, 119, 210, 0, 0, 0, 0, 0, 0, 174, 170, 0, 0],
            &[0, 42, 255, 57, 0, 0, 0, 0, 23, 248, 170, 0, 0],
            &[0, 0, 158, 215, 45, 0, 0, 28, 190, 175, 170, 0, 0],
            &[0, 0, 5, 144, 246, 225, 219, 227, 89, 123, 170, 0, 0],
            &[0, 0, 0, 0, 10, 44, 36, 3, 0, 129, 168, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 160, 147, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 14, 235, 87, 0, 0],
            &[0, 0, 0, 0, 18, 37, 26, 4, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ğ' / 0x11e
        'Ğ' => &[
            &[0, 0, 0, 5, 49, 0, 0, 0, 11, 41, 0, 0, 0],
            &[0, 0, 0, 1, 230, 31, 0, 0, 114, 137, 0, 0, 0],
            &[0, 0, 0, 0, 87, 226, 158, 175, 209, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 76, 61, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 32, 48, 20, 0, 0, 0, 0],
            &[0, 0, 0, 20, 144, 244, 250, 236, 255, 213, 83, 0, 0],
            &[0, 0, 25, 214, 196, 59, 3, 0, 11, 92, 64, 0, 0],
            &[0, 0, 176, 191, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 53, 252, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 193, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 193, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 218, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 236, 84, 0, 0, 0, 63, 164, 164, 164, 144, 0, 0],
            &[0, 228, 93, 0, 0, 0, 46, 120, 120, 163, 224, 0, 0],
            &[0, 208, 114, 0, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 161, 169, 0, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 90, 239, 15, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 5, 218, 149, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 0, 58, 241, 162, 39, 1, 0, 22, 140, 224, 0, 0],
            &[0, 0, 0, 44, 175, 253, 247, 243, 255, 206, 108, 0, 0],
            &[0, 0, 0, 0, 0, 8, 38, 45, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ğ' / 0x11f
        'ğ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 36, 0, 0, 0, 76, 17, 0, 0, 0],
            &[0, 0, 0, 94, 180, 6, 0, 34, 224, 7, 0, 0, 0],
            &[0, 0, 0, 1, 154, 229, 198, 224, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 31, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 87, 82, 31, 0, 14, 29, 0, 0],
            &[0, 0, 11, 175, 243, 183, 177, 226, 137, 100, 170, 0, 0],
            &[0, 0, 168, 197, 20, 0, 0, 4, 150, 216, 170, 0, 0],
            &[0, 43, 255, 47, 0, 0, 0, 0, 6, 229, 170, 0, 0],
            &[0, 118, 206, 0, 0, 0, 0, 0, 0, 160, 170, 0, 0],
            &[0, 157, 170, 0, 0, 0, 0, 0, 0, 129, 170, 0, 0],
            &[0, 169, 150, 0, 0, 0, 0, 0, 0, 119, 170, 0, 0],
            &[0, 158, 172, 0, 0, 0, 0, 0, 0, 137, 170, 0, 0],
            &[0, 119, 210, 0, 0, 0, 0, 0, 0, 174, 170, 0, 0],
            &[0, 42, 255, 57, 0, 0, 0, 0, 23, 248, 170, 0, 0],
            &[0, 0, 158, 215, 45, 0, 0, 28, 190, 175, 170, 0, 0],
            &[0, 0, 5, 144, 246, 225, 219, 227, 89, 123, 170, 0, 0],
            &[0, 0, 0, 0, 10, 44, 36, 3, 0, 129, 168, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 160, 147, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 14, 235, 87, 0, 0],
            &[0, 0, 0, 0, 18, 37, 26, 4, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ġ' / 0x120
        'Ġ' => &[
            &[0, 0, 0, 0, 0, 0, 62, 27, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 255, 178, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 147, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 32, 48, 20, 0, 0, 0, 0],
            &[0, 0, 0, 20, 144, 244, 250, 236, 255, 213, 83, 0, 0],
            &[0, 0, 25, 214, 196, 59, 3, 0, 11, 92, 64, 0, 0],
            &[0, 0, 176, 191, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 53, 252, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 193, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 193, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 218, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 236, 84, 0, 0, 0, 63, 164, 164, 164, 144, 0, 0],
            &[0, 228, 93, 0, 0, 0, 46, 120, 120, 163, 224, 0, 0],
            &[0, 208, 114, 0, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 161, 169, 0, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 90, 239, 15, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 5, 218, 149, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 0, 58, 241, 162, 39, 1, 0, 22, 140, 224, 0, 0],
            &[0, 0, 0, 44, 175, 253, 247, 243, 255, 206, 108, 0, 0],
            &[0, 0, 0, 0, 0, 8, 38, 45, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ġ' / 0x121
        'ġ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 146, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 152, 255, 50, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 76, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 87, 82, 31, 0, 14, 29, 0, 0],
            &[0, 0, 11, 175, 243, 183, 177, 226, 137, 100, 170, 0, 0],
            &[0, 0, 168, 197, 20, 0, 0, 4, 150, 216, 170, 0, 0],
            &[0, 43, 255, 47, 0, 0, 0, 0, 6, 229, 170, 0, 0],
            &[0, 118, 206, 0, 0, 0, 0, 0, 0, 160, 170, 0, 0],
            &[0, 157, 170, 0, 0, 0, 0, 0, 0, 129, 170, 0, 0],
            &[0, 169, 150, 0, 0, 0, 0, 0, 0, 119, 170, 0, 0],
            &[0, 158, 172, 0, 0, 0, 0, 0, 0, 137, 170, 0, 0],
            &[0, 119, 210, 0, 0, 0, 0, 0, 0, 174, 170, 0, 0],
            &[0, 42, 255, 57, 0, 0, 0, 0, 23, 248, 170, 0, 0],
            &[0, 0, 158, 215, 45, 0, 0, 28, 190, 175, 170, 0, 0],
            &[0, 0, 5, 144, 246, 225, 219, 227, 89, 123, 170, 0, 0],
            &[0, 0, 0, 0, 10, 44, 36, 3, 0, 129, 168, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 160, 147, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 14, 235, 87, 0, 0],
            &[0, 0, 0, 0, 18, 37, 26, 4, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ģ' / 0x122
        'Ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 32, 48, 20, 0, 0, 0, 0],
            &[0, 0, 0, 20, 144, 244, 250, 236, 255, 213, 83, 0, 0],
            &[0, 0, 25, 214, 196, 59, 3, 0, 11, 92, 64, 0, 0],
            &[0, 0, 176, 191, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 53, 252, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 131, 193, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 193, 131, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 218, 101, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 236, 84, 0, 0, 0, 63, 164, 164, 164, 144, 0, 0],
            &[0, 228, 93, 0, 0, 0, 46, 120, 120, 163, 224, 0, 0],
            &[0, 208, 114, 0, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 161, 169, 0, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 90, 239, 15, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 5, 218, 149, 0, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 0, 58, 241, 162, 39, 1, 0, 22, 140, 224, 0, 0],
            &[0, 0, 0, 44, 175, 253, 247, 243, 255, 206, 108, 0, 0],
            &[0, 0, 0, 0, 0, 8, 38, 45, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 111, 112, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 205, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 64, 66, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ģ' / 0x123
        'ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 113, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 234, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 90, 210, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 51, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 87, 82, 31, 0, 14, 29, 0, 0],
            &[0, 0, 11, 175, 243, 183, 177, 226, 137, 100, 170, 0, 0],
            &[0, 0, 168, 197, 20, 0, 0, 4, 150, 216, 170, 0, 0],
            &[0, 43, 255, 47, 0, 0, 0, 0, 6, 229, 170, 0, 0],
            &[0, 118, 206, 0, 0, 0, 0, 0, 0, 160, 170, 0, 0],
            &[0, 157, 170, 0, 0, 0, 0, 0, 0, 129, 170, 0, 0],
            &[0, 169, 150, 0, 0, 0, 0, 0, 0, 119, 170, 0, 0],
            &[0, 158, 172, 0, 0, 0, 0, 0, 0, 137, 170, 0, 0],
            &[0, 119, 210, 0, 0, 0, 0, 0, 0, 174, 170, 0, 0],
            &[0, 42, 255, 57, 0, 0, 0, 0, 23, 248, 170, 0, 0],
            &[0, 0, 158, 215, 45, 0, 0, 28, 190, 175, 170, 0, 0],
            &[0, 0, 5, 144, 246, 225, 219, 227, 89, 123, 170, 0, 0],
            &[0, 0, 0, 0, 10, 44, 36, 3, 0, 129, 168, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 160, 147, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 14, 235, 87, 0, 0],
            &[0, 0, 0, 0, 18, 37, 26, 4, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĥ' / 0x124
        'Ĥ' => &[
            &[0, 0, 0, 0, 0, 131, 225, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 219, 113, 215, 15, 0, 0, 0, 0],
            &[0, 0, 0, 43, 212, 36, 0, 111, 183, 4, 0, 0, 0],
            &[0, 0, 0, 83, 35, 0, 0, 0, 79, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 86, 223, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 250, 107, 107, 107, 107, 107, 107, 158, 224, 0, 0],
            &[0, 65, 252, 177, 177, 177, 177, 177, 177, 203, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĥ' / 0x125
        'ĥ' => &[
            &[0, 27, 225, 111, 230, 43, 0, 0, 0, 0, 0, 0, 0],
            &[11, 198, 84, 0, 67, 211, 21, 0, 0, 0, 0, 0, 0],
            &[57, 69, 0, 0, 0, 57, 70, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 245, 42, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 255, 43, 0, 40, 88, 75, 26, 0, 0, 0],
            &[0, 0, 12, 255, 48, 156, 218, 174, 199, 248, 131, 0, 0],
            &[0, 0, 12, 255, 168, 107, 1, 0, 0, 66, 252, 87, 0],
            &[0, 0, 12, 255, 171, 0, 0, 0, 0, 0, 169, 167, 0],
            &[0, 0, 12, 255, 84, 0, 0, 0, 0, 0, 126, 194, 0],
            &[0, 0, 12, 255, 52, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 218, 96, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ħ' / 0x126
        'Ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 86, 223, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[13, 79, 255, 13, 13, 13, 13, 13, 13, 100, 238, 13, 8],
            &[
                249, 255, 255, 249, 249, 249, 249, 249, 249, 255, 255, 249, 149,
            ],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 250, 107, 107, 107, 107, 107, 107, 158, 224, 0, 0],
            &[0, 65, 252, 177, 177, 177, 177, 177, 177, 203, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ħ' / 0x127
        'ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 245, 42, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 236, 248, 255, 255, 236, 236, 236, 52, 0, 0, 0, 0],
            &[0, 11, 23, 255, 54, 11, 11, 11, 2, 0, 0, 0, 0],
            &[0, 0, 12, 255, 43, 0, 40, 88, 75, 26, 0, 0, 0],
            &[0, 0, 12, 255, 48, 156, 218, 174, 199, 248, 131, 0, 0],
            &[0, 0, 12, 255, 168, 107, 1, 0, 0, 66, 252, 87, 0],
            &[0, 0, 12, 255, 171, 0, 0, 0, 0, 0, 169, 167, 0],
            &[0, 0, 12, 255, 84, 0, 0, 0, 0, 0, 126, 194, 0],
            &[0, 0, 12, 255, 52, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĩ' / 0x128
        'Ĩ' => &[
            &[0, 0, 0, 0, 2, 0, 0, 0, 0, 13, 0, 0, 0],
            &[0, 0, 0, 123, 245, 220, 78, 0, 31, 199, 0, 0, 0],
            &[0, 0, 27, 214, 17, 52, 208, 198, 225, 86, 0, 0, 0],
            &[0, 0, 21, 47, 0, 0, 4, 62, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 254, 254, 252, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 255, 255, 253, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĩ' / 0x129
        'ĩ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 17, 0, 0, 0, 51, 0, 0, 0],
            &[0, 0, 0, 165, 224, 236, 123, 4, 53, 189, 0, 0, 0],
            &[0, 0, 38, 199, 0, 19, 168, 238, 236, 59, 0, 0, 0],
            &[0, 0, 10, 20, 0, 0, 0, 18, 8, 0, 0, 0, 0],
            &[0, 0, 33, 44, 44, 44, 44, 12, 0, 0, 0, 0, 0],
            &[0, 0, 136, 192, 204, 217, 253, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ī' / 0x12a
        'Ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 149, 249, 249, 249, 249, 249, 49, 0, 0, 0],
            &[0, 0, 0, 4, 8, 8, 8, 8, 8, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 254, 254, 252, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 255, 255, 253, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ī' / 0x12b
        'ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 39, 39, 39, 39, 39, 7, 0, 0, 0],
            &[0, 0, 0, 130, 218, 218, 218, 218, 218, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 44, 44, 44, 44, 12, 0, 0, 0, 0, 0],
            &[0, 0, 136, 192, 204, 217, 253, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => &[
            &[0, 0, 0, 35, 18, 0, 0, 0, 42, 11, 0, 0, 0],
            &[0, 0, 0, 111, 152, 0, 0, 13, 224, 13, 0, 0, 0],
            &[0, 0, 0, 9, 197, 191, 152, 217, 109, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 57, 77, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 254, 254, 252, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 255, 255, 253, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĭ' / 0x12d
        'ĭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 36, 0, 0, 0, 76, 17, 0, 0, 0],
            &[0, 0, 0, 94, 180, 6, 0, 34, 224, 7, 0, 0, 0],
            &[0, 0, 0, 1, 154, 229, 198, 224, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 31, 3, 0, 0, 0, 0, 0],
            &[0, 0, 33, 44, 44, 44, 44, 12, 0, 0, 0, 0, 0],
            &[0, 0, 136, 192, 204, 217, 253, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Į' / 0x12e
        'Į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 254, 254, 252, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 255, 255, 254, 255, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 168, 103, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 191, 67, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 16, 119, 138, 31, 0, 0, 0],
        ],
        // letter: 'į' / 0x12f
        'į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 150, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 87, 255, 114, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 80, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 44, 44, 44, 44, 12, 0, 0, 0, 0, 0],
            &[0, 0, 136, 192, 204, 217, 253, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 255, 255, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 27, 199, 47, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 183, 68, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 245, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 43, 134, 128, 0, 0, 0],
        ],
        // letter: 'İ' / 0x130
        'İ' => &[
            &[0, 0, 0, 0, 0, 20, 68, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 151, 255, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 64, 153, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 254, 254, 252, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 233, 1, 0, 0, 0, 0, 0],
            &[0, 0, 36, 230, 245, 255, 255, 253, 239, 172, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ı' / 0x131
        'ı' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 44, 44, 44, 44, 12, 0, 0, 0, 0, 0],
            &[0, 0, 136, 192, 204, 217, 253, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĳ' / 0x132
        'Ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 186, 123, 0, 0, 0, 0, 0, 220, 89, 0, 0],
            &[0, 0, 187, 124, 0, 0, 0, 0, 0, 221, 90, 0, 0],
            &[0, 0, 187, 124, 0, 0, 0, 0, 0, 221, 90, 0, 0],
            &[0, 0, 187, 124, 0, 0, 0, 0, 0, 221, 90, 0, 0],
            &[0, 0, 187, 124, 0, 0, 0, 0, 0, 221, 90, 0, 0],
            &[0, 0, 187, 124, 0, 0, 0, 0, 0, 221, 90, 0, 0],
            &[0, 0, 187, 124, 0, 0, 0, 0, 0, 221, 90, 0, 0],
            &[0, 0, 187, 124, 0, 0, 0, 0, 0, 221, 90, 0, 0],
            &[0, 0, 187, 124, 0, 0, 0, 0, 0, 221, 90, 0, 0],
            &[0, 0, 187, 124, 0, 0, 0, 0, 0, 221, 90, 0, 0],
            &[0, 0, 187, 124, 0, 0, 0, 0, 0, 224, 88, 0, 0],
            &[0, 0, 64, 43, 0, 0, 0, 0, 2, 244, 71, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 66, 253, 22, 0, 0],
            &[0, 0, 89, 60, 5, 0, 0, 57, 222, 162, 0, 0, 0],
            &[0, 0, 107, 237, 250, 228, 238, 248, 155, 10, 0, 0, 0],
            &[0, 0, 0, 1, 34, 57, 49, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĳ' / 0x133
        'ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 149, 16, 0, 0, 0, 66, 146, 5, 0, 0],
            &[0, 0, 137, 255, 66, 0, 0, 0, 162, 255, 40, 0, 0],
            &[0, 0, 19, 78, 4, 0, 0, 0, 26, 75, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 41, 0, 0, 0, 0, 12, 41, 0, 0, 0],
            &[0, 0, 69, 242, 0, 0, 0, 0, 72, 238, 0, 0, 0],
            &[0, 0, 69, 242, 0, 0, 0, 0, 72, 238, 0, 0, 0],
            &[0, 0, 69, 242, 0, 0, 0, 0, 72, 238, 0, 0, 0],
            &[0, 0, 69, 242, 0, 0, 0, 0, 72, 238, 0, 0, 0],
            &[0, 0, 69, 242, 0, 0, 0, 0, 72, 238, 0, 0, 0],
            &[0, 0, 69, 242, 0, 0, 0, 0, 72, 238, 0, 0, 0],
            &[0, 0, 69, 242, 0, 0, 0, 0, 72, 238, 0, 0, 0],
            &[0, 0, 69, 242, 0, 0, 0, 0, 72, 238, 0, 0, 0],
            &[0, 0, 69, 242, 0, 0, 0, 0, 72, 238, 0, 0, 0],
            &[0, 0, 69, 242, 0, 0, 0, 0, 72, 238, 0, 0, 0],
            &[0, 0, 69, 242, 0, 0, 0, 0, 72, 238, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 72, 238, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 79, 232, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 129, 193, 0, 0, 0],
            &[0, 0, 0, 0, 13, 34, 17, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĵ' / 0x134
        'Ĵ' => &[
            &[0, 0, 0, 0, 0, 0, 44, 224, 137, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 212, 113, 221, 77, 0, 0, 0],
            &[0, 0, 0, 0, 3, 178, 110, 0, 37, 214, 47, 0, 0],
            &[0, 0, 0, 0, 37, 80, 0, 0, 0, 33, 84, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 232, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 234, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 236, 76, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 251, 60, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 65, 252, 17, 0, 0, 0, 0],
            &[0, 94, 48, 3, 0, 34, 211, 169, 0, 0, 0, 0, 0],
            &[0, 114, 240, 248, 229, 253, 180, 17, 0, 0, 0, 0, 0],
            &[0, 0, 6, 38, 56, 26, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĵ' / 0x135
        'ĵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 212, 243, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 159, 153, 103, 206, 10, 0, 0, 0],
            &[0, 0, 0, 0, 119, 165, 3, 0, 117, 173, 2, 0, 0],
            &[0, 0, 0, 0, 66, 2, 0, 0, 0, 55, 13, 0, 0],
            &[0, 0, 5, 44, 44, 44, 44, 40, 0, 0, 0, 0, 0],
            &[0, 0, 21, 184, 196, 209, 230, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 234, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 229, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 128, 193, 0, 0, 0, 0, 0],
            &[0, 0, 4, 27, 28, 1, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ķ' / 0x136
        'Ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 153, 209, 13, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 120, 231, 30, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 88, 245, 54, 0, 0, 0],
            &[0, 65, 245, 0, 0, 0, 60, 246, 84, 0, 0, 0, 0],
            &[0, 65, 245, 0, 0, 38, 237, 120, 0, 0, 0, 0, 0],
            &[0, 65, 245, 0, 21, 220, 158, 0, 0, 0, 0, 0, 0],
            &[0, 65, 245, 6, 198, 235, 6, 0, 0, 0, 0, 0, 0],
            &[0, 65, 245, 146, 192, 228, 105, 0, 0, 0, 0, 0, 0],
            &[0, 65, 254, 200, 13, 73, 243, 33, 0, 0, 0, 0, 0],
            &[0, 65, 250, 18, 0, 0, 158, 191, 1, 0, 0, 0, 0],
            &[0, 65, 245, 0, 0, 0, 15, 227, 109, 0, 0, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 72, 244, 35, 0, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 156, 194, 2, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 14, 226, 112, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 71, 245, 37, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 181, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 232, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 8, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ķ' / 0x137
        'ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 102, 196, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 0, 9, 44, 12, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 8, 188, 180, 5, 0, 0],
            &[0, 106, 204, 0, 0, 0, 5, 179, 190, 8, 0, 0, 0],
            &[0, 106, 204, 0, 0, 3, 169, 200, 12, 0, 0, 0, 0],
            &[0, 106, 203, 0, 1, 159, 209, 17, 0, 0, 0, 0, 0],
            &[0, 106, 198, 0, 147, 255, 41, 0, 0, 0, 0, 0, 0],
            &[0, 106, 191, 130, 200, 176, 176, 0, 0, 0, 0, 0, 0],
            &[0, 106, 240, 193, 13, 17, 226, 108, 0, 0, 0, 0, 0],
            &[0, 106, 224, 10, 0, 0, 61, 245, 47, 0, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 130, 215, 11, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 3, 198, 158, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 0, 32, 240, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 100, 123, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 189, 98, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 54, 77, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĸ' / 0x138
        'ĸ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 35, 0, 0, 0, 0, 0, 9, 44, 12, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 8, 188, 180, 5, 0, 0],
            &[0, 106, 204, 0, 0, 0, 5, 179, 190, 8, 0, 0, 0],
            &[0, 106, 204, 0, 0, 3, 169, 200, 12, 0, 0, 0, 0],
            &[0, 106, 203, 0, 1, 159, 209, 17, 0, 0, 0, 0, 0],
            &[0, 106, 198, 0, 147, 255, 41, 0, 0, 0, 0, 0, 0],
            &[0, 106, 191, 130, 200, 176, 176, 0, 0, 0, 0, 0, 0],
            &[0, 106, 240, 193, 13, 17, 226, 108, 0, 0, 0, 0, 0],
            &[0, 106, 224, 10, 0, 0, 61, 245, 47, 0, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 130, 215, 11, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 3, 198, 158, 0, 0, 0],
            &[0, 106, 204, 0, 0, 0, 0, 0, 32, 240, 90, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĺ' / 0x139
        'Ĺ' => &[
            &[0, 0, 0, 0, 81, 224, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 232, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 200, 98, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 50, 68, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 254, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 36, 34, 34, 34, 34, 34, 26, 0, 0],
            &[0, 0, 52, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĺ' / 0x13a
        'ĺ' => &[
            &[0, 0, 0, 0, 0, 0, 40, 237, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 210, 84, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 65, 62, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 183, 236, 245, 245, 245, 70, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 15, 239, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 98, 210, 24, 0, 0, 0],
        ],
        // letter: 'Ļ' / 0x13b
        'Ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 254, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 36, 34, 34, 34, 34, 34, 26, 0, 0],
            &[0, 0, 52, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 107, 115, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 200, 87, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 70, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ļ' / 0x13c
        'ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 183, 236, 245, 245, 245, 70, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 15, 239, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 173, 50, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 35, 241, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 116, 14, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ľ' / 0x13d
        'Ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 254, 2, 0, 0, 0, 1, 241, 61, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 45, 226, 3, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 118, 115, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 64, 8, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 36, 34, 34, 34, 34, 34, 26, 0, 0],
            &[0, 0, 52, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ľ' / 0x13e
        'ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 183, 236, 245, 245, 245, 70, 0, 32, 243, 18, 0],
            &[0, 0, 0, 0, 3, 15, 239, 73, 0, 89, 187, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 161, 73, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 254, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 29, 164, 46, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 83, 255, 112, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 7, 83, 14, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 2, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 52, 255, 36, 34, 34, 34, 34, 34, 26, 0, 0],
            &[0, 0, 52, 255, 255, 255, 255, 255, 255, 255, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŀ' / 0x140
        'ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 183, 236, 245, 245, 245, 70, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 15, 239, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 5, 84, 21],
            &[0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 66, 255, 129],
            &[0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 19, 161, 53],
            &[0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0, 0],
            &[24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ł' / 0x141
        'Ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 93, 216, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 93, 217, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 93, 217, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 93, 217, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 93, 217, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 93, 217, 0, 0, 76, 4, 0, 0, 0, 0],
            &[0, 0, 0, 93, 217, 34, 182, 216, 32, 0, 0, 0, 0],
            &[0, 0, 0, 100, 255, 237, 123, 6, 0, 0, 0, 0, 0],
            &[0, 0, 66, 255, 255, 33, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 225, 175, 217, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 11, 93, 217, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 93, 217, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 93, 217, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 93, 222, 34, 34, 34, 34, 34, 34, 21, 0],
            &[0, 0, 0, 93, 255, 255, 255, 255, 255, 255, 255, 156, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ł' / 0x142
        'ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 183, 236, 245, 245, 245, 70, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 15, 239, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 75, 107, 195, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 255, 194, 41, 0, 0, 0],
            &[0, 0, 0, 0, 8, 133, 255, 175, 1, 0, 0, 0, 0],
            &[0, 0, 0, 64, 219, 169, 255, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 85, 75, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 238, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 239, 79, 0, 0, 0, 0, 0],
            &[0, 24, 227, 238, 250, 255, 255, 255, 253, 241, 229, 52, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ń' / 0x143
        'Ń' => &[
            &[0, 0, 0, 0, 0, 0, 0, 80, 225, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 232, 90, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 200, 99, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 49, 68, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 254, 132, 0, 0, 0, 0, 0, 81, 223, 0, 0],
            &[0, 65, 234, 239, 16, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 212, 169, 127, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 225, 46, 237, 14, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 232, 0, 175, 122, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 50, 235, 12, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 181, 117, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 56, 233, 10, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 186, 113, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 61, 230, 8, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 192, 108, 78, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 67, 227, 78, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 197, 165, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 72, 251, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 0, 202, 224, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ń' / 0x144
        'ń' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 84, 245, 48, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 233, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 203, 83, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 41, 0, 0, 0, 0, 0, 0],
            &[0, 2, 41, 0, 0, 41, 88, 77, 30, 0, 0, 0, 0],
            &[0, 12, 252, 14, 166, 221, 178, 196, 249, 141, 1, 0, 0],
            &[0, 12, 255, 173, 105, 0, 0, 0, 55, 249, 94, 0, 0],
            &[0, 12, 255, 173, 0, 0, 0, 0, 0, 162, 172, 0, 0],
            &[0, 12, 255, 87, 0, 0, 0, 0, 0, 123, 195, 0, 0],
            &[0, 12, 255, 57, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ņ' / 0x145
        'Ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 254, 132, 0, 0, 0, 0, 0, 81, 223, 0, 0],
            &[0, 65, 234, 239, 16, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 212, 169, 127, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 225, 46, 237, 14, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 232, 0, 175, 122, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 50, 235, 12, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 181, 117, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 56, 233, 10, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 186, 113, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 61, 230, 8, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 192, 108, 78, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 67, 227, 78, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 197, 165, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 72, 251, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 0, 202, 224, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 164, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 132, 155, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 113, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ņ' / 0x146
        'ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 41, 0, 0, 41, 88, 77, 30, 0, 0, 0, 0],
            &[0, 12, 252, 14, 166, 221, 178, 196, 249, 141, 1, 0, 0],
            &[0, 12, 255, 173, 105, 0, 0, 0, 55, 249, 94, 0, 0],
            &[0, 12, 255, 173, 0, 0, 0, 0, 0, 162, 172, 0, 0],
            &[0, 12, 255, 87, 0, 0, 0, 0, 0, 123, 195, 0, 0],
            &[0, 12, 255, 57, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 183, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 89, 198, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ň' / 0x147
        'Ň' => &[
            &[0, 0, 0, 102, 141, 1, 0, 11, 177, 64, 0, 0, 0],
            &[0, 0, 0, 0, 166, 146, 8, 193, 114, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 210, 218, 162, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 115, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 254, 132, 0, 0, 0, 0, 0, 81, 223, 0, 0],
            &[0, 65, 234, 239, 16, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 212, 169, 127, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 225, 46, 237, 14, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 232, 0, 175, 122, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 50, 235, 12, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 181, 117, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 56, 233, 10, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 186, 113, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 61, 230, 8, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 192, 108, 78, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 67, 227, 78, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 197, 165, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 72, 251, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 0, 202, 224, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ň' / 0x148
        'ň' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 4, 0, 0, 0, 4, 10, 0, 0, 0],
            &[0, 0, 0, 69, 196, 18, 0, 18, 197, 75, 0, 0, 0],
            &[0, 0, 0, 0, 102, 203, 25, 204, 102, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 159, 244, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 70, 11, 0, 0, 0, 0, 0],
            &[0, 2, 41, 0, 0, 41, 88, 77, 30, 0, 0, 0, 0],
            &[0, 12, 252, 14, 166, 221, 178, 196, 249, 141, 1, 0, 0],
            &[0, 12, 255, 173, 105, 0, 0, 0, 55, 249, 94, 0, 0],
            &[0, 12, 255, 173, 0, 0, 0, 0, 0, 162, 172, 0, 0],
            &[0, 12, 255, 87, 0, 0, 0, 0, 0, 123, 195, 0, 0],
            &[0, 12, 255, 57, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŉ' / 0x149
        'ŉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 118, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 161, 183, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 214, 105, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[21, 249, 21, 29, 14, 0, 22, 82, 83, 36, 0, 0, 0],
            &[93, 166, 0, 171, 95, 102, 224, 184, 192, 249, 151, 0, 0],
            &[21, 15, 0, 171, 178, 173, 12, 0, 0, 49, 248, 88, 0],
            &[0, 0, 0, 171, 245, 20, 0, 0, 0, 0, 161, 160, 0],
            &[0, 0, 0, 171, 183, 0, 0, 0, 0, 0, 123, 192, 0],
            &[0, 0, 0, 171, 153, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 0, 171, 140, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 0, 171, 140, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 0, 171, 140, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 0, 171, 140, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 0, 171, 140, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 0, 171, 140, 0, 0, 0, 0, 0, 113, 197, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 254, 132, 0, 0, 0, 0, 0, 81, 223, 0, 0],
            &[0, 65, 234, 239, 16, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 212, 169, 127, 0, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 225, 46, 237, 14, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 232, 0, 175, 122, 0, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 50, 235, 12, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 181, 117, 0, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 56, 233, 10, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 186, 113, 0, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 61, 230, 8, 81, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 192, 108, 78, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 67, 227, 78, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 197, 165, 224, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 72, 255, 250, 0, 0],
            &[0, 65, 235, 0, 0, 0, 0, 0, 0, 255, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 91, 203, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 171, 159, 0, 0],
            &[0, 0, 0, 0, 0, 38, 93, 89, 165, 246, 38, 0, 0],
            &[0, 0, 0, 0, 0, 46, 187, 199, 158, 39, 0, 0, 0],
        ],
        // letter: 'ŋ' / 0x14b
        'ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 2, 41, 0, 0, 41, 88, 77, 30, 0, 0, 0, 0],
            &[0, 12, 252, 14, 166, 221, 178, 196, 249, 141, 1, 0, 0],
            &[0, 12, 255, 173, 105, 0, 0, 0, 55, 249, 94, 0, 0],
            &[0, 12, 255, 173, 0, 0, 0, 0, 0, 162, 172, 0, 0],
            &[0, 12, 255, 87, 0, 0, 0, 0, 0, 123, 195, 0, 0],
            &[0, 12, 255, 57, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 161, 255, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 227, 255, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 227, 255, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 227, 255, 0, 0],
            &[0, 12, 255, 43, 0, 0, 0, 0, 0, 227, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 113, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 114, 197, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 132, 183, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0],
        ],
        // letter: 'Ō' / 0x14c
        'Ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 149, 249, 249, 249, 249, 249, 49, 0, 0, 0],
            &[0, 0, 0, 4, 8, 8, 8, 8, 8, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 49, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 217, 250, 230, 251, 184, 31, 0, 0, 0],
            &[0, 0, 79, 245, 105, 5, 0, 21, 157, 226, 21, 0, 0],
            &[0, 8, 231, 105, 0, 0, 0, 0, 1, 190, 150, 0, 0],
            &[0, 94, 229, 4, 0, 0, 0, 0, 0, 70, 244, 8, 0],
            &[0, 159, 156, 0, 0, 0, 0, 0, 0, 6, 246, 66, 0],
            &[0, 207, 108, 0, 0, 0, 0, 0, 0, 0, 209, 107, 0],
            &[0, 226, 88, 0, 0, 0, 0, 0, 0, 0, 186, 129, 0],
            &[0, 239, 77, 0, 0, 0, 0, 0, 0, 0, 179, 137, 0],
            &[0, 230, 88, 0, 0, 0, 0, 0, 0, 0, 190, 127, 0],
            &[0, 206, 110, 0, 0, 0, 0, 0, 0, 0, 211, 103, 0],
            &[0, 162, 159, 0, 0, 0, 0, 0, 0, 7, 250, 61, 0],
            &[0, 92, 232, 5, 0, 0, 0, 0, 0, 77, 241, 5, 0],
            &[0, 9, 233, 111, 0, 0, 0, 0, 4, 198, 143, 0, 0],
            &[0, 0, 82, 249, 112, 10, 0, 31, 173, 221, 17, 0, 0],
            &[0, 0, 0, 77, 218, 254, 242, 252, 175, 25, 0, 0, 0],
            &[0, 0, 0, 0, 1, 30, 45, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ō' / 0x14d
        'ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 39, 39, 39, 39, 39, 7, 0, 0, 0],
            &[0, 0, 0, 130, 218, 218, 218, 218, 218, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 72, 89, 60, 4, 0, 0, 0, 0],
            &[0, 0, 3, 135, 240, 191, 174, 213, 226, 70, 0, 0, 0],
            &[0, 0, 143, 213, 32, 0, 0, 0, 89, 246, 53, 0, 0],
            &[0, 35, 252, 44, 0, 0, 0, 0, 0, 141, 192, 0, 0],
            &[0, 116, 200, 0, 0, 0, 0, 0, 0, 46, 253, 20, 0],
            &[0, 152, 154, 0, 0, 0, 0, 0, 0, 3, 254, 56, 0],
            &[0, 168, 143, 0, 0, 0, 0, 0, 0, 0, 245, 71, 0],
            &[0, 152, 161, 0, 0, 0, 0, 0, 0, 8, 255, 54, 0],
            &[0, 105, 215, 0, 0, 0, 0, 0, 0, 62, 250, 10, 0],
            &[0, 22, 248, 69, 0, 0, 0, 0, 0, 172, 169, 0, 0],
            &[0, 0, 111, 236, 72, 0, 0, 13, 140, 237, 32, 0, 0],
            &[0, 0, 0, 95, 225, 236, 219, 246, 189, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 47, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => &[
            &[0, 0, 0, 35, 18, 0, 0, 0, 42, 11, 0, 0, 0],
            &[0, 0, 0, 111, 152, 0, 0, 13, 224, 13, 0, 0, 0],
            &[0, 0, 0, 9, 197, 191, 152, 217, 109, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 57, 77, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 49, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 217, 250, 230, 251, 184, 31, 0, 0, 0],
            &[0, 0, 79, 245, 105, 5, 0, 21, 157, 226, 21, 0, 0],
            &[0, 8, 231, 105, 0, 0, 0, 0, 1, 190, 150, 0, 0],
            &[0, 94, 229, 4, 0, 0, 0, 0, 0, 70, 244, 8, 0],
            &[0, 159, 156, 0, 0, 0, 0, 0, 0, 6, 246, 66, 0],
            &[0, 207, 108, 0, 0, 0, 0, 0, 0, 0, 209, 107, 0],
            &[0, 226, 88, 0, 0, 0, 0, 0, 0, 0, 186, 129, 0],
            &[0, 239, 77, 0, 0, 0, 0, 0, 0, 0, 179, 137, 0],
            &[0, 230, 88, 0, 0, 0, 0, 0, 0, 0, 190, 127, 0],
            &[0, 206, 110, 0, 0, 0, 0, 0, 0, 0, 211, 103, 0],
            &[0, 162, 159, 0, 0, 0, 0, 0, 0, 7, 250, 61, 0],
            &[0, 92, 232, 5, 0, 0, 0, 0, 0, 77, 241, 5, 0],
            &[0, 9, 233, 111, 0, 0, 0, 0, 4, 198, 143, 0, 0],
            &[0, 0, 82, 249, 112, 10, 0, 31, 173, 221, 17, 0, 0],
            &[0, 0, 0, 77, 218, 254, 242, 252, 175, 25, 0, 0, 0],
            &[0, 0, 0, 0, 1, 30, 45, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŏ' / 0x14f
        'ŏ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 36, 0, 0, 0, 76, 17, 0, 0, 0],
            &[0, 0, 0, 94, 180, 6, 0, 34, 224, 7, 0, 0, 0],
            &[0, 0, 0, 1, 154, 229, 198, 224, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 31, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 72, 89, 60, 4, 0, 0, 0, 0],
            &[0, 0, 3, 135, 240, 191, 174, 213, 226, 70, 0, 0, 0],
            &[0, 0, 143, 213, 32, 0, 0, 0, 89, 246, 53, 0, 0],
            &[0, 35, 252, 44, 0, 0, 0, 0, 0, 141, 192, 0, 0],
            &[0, 116, 200, 0, 0, 0, 0, 0, 0, 46, 253, 20, 0],
            &[0, 152, 154, 0, 0, 0, 0, 0, 0, 3, 254, 56, 0],
            &[0, 168, 143, 0, 0, 0, 0, 0, 0, 0, 245, 71, 0],
            &[0, 152, 161, 0, 0, 0, 0, 0, 0, 8, 255, 54, 0],
            &[0, 105, 215, 0, 0, 0, 0, 0, 0, 62, 250, 10, 0],
            &[0, 22, 248, 69, 0, 0, 0, 0, 0, 172, 169, 0, 0],
            &[0, 0, 111, 236, 72, 0, 0, 13, 140, 237, 32, 0, 0],
            &[0, 0, 0, 95, 225, 236, 219, 246, 189, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 47, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ő' / 0x150
        'Ő' => &[
            &[0, 0, 0, 0, 1, 192, 123, 0, 119, 194, 3, 0, 0],
            &[0, 0, 0, 0, 103, 208, 9, 32, 235, 52, 0, 0, 0],
            &[0, 0, 0, 34, 215, 31, 2, 189, 89, 0, 0, 0, 0],
            &[0, 0, 0, 67, 38, 0, 30, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 49, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 76, 217, 250, 230, 251, 184, 31, 0, 0, 0],
            &[0, 0, 79, 245, 105, 5, 0, 21, 157, 226, 21, 0, 0],
            &[0, 8, 231, 105, 0, 0, 0, 0, 1, 190, 150, 0, 0],
            &[0, 94, 229, 4, 0, 0, 0, 0, 0, 70, 244, 8, 0],
            &[0, 159, 156, 0, 0, 0, 0, 0, 0, 6, 246, 66, 0],
            &[0, 207, 108, 0, 0, 0, 0, 0, 0, 0, 209, 107, 0],
            &[0, 226, 88, 0, 0, 0, 0, 0, 0, 0, 186, 129, 0],
            &[0, 239, 77, 0, 0, 0, 0, 0, 0, 0, 179, 137, 0],
            &[0, 230, 88, 0, 0, 0, 0, 0, 0, 0, 190, 127, 0],
            &[0, 206, 110, 0, 0, 0, 0, 0, 0, 0, 211, 103, 0],
            &[0, 162, 159, 0, 0, 0, 0, 0, 0, 7, 250, 61, 0],
            &[0, 92, 232, 5, 0, 0, 0, 0, 0, 77, 241, 5, 0],
            &[0, 9, 233, 111, 0, 0, 0, 0, 4, 198, 143, 0, 0],
            &[0, 0, 82, 249, 112, 10, 0, 31, 173, 221, 17, 0, 0],
            &[0, 0, 0, 77, 218, 254, 242, 252, 175, 25, 0, 0, 0],
            &[0, 0, 0, 0, 1, 30, 45, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ő' / 0x151
        'ő' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 16, 0, 6, 22, 1, 0, 0],
            &[0, 0, 0, 0, 8, 224, 115, 0, 150, 194, 1, 0, 0],
            &[0, 0, 0, 0, 132, 180, 2, 53, 230, 29, 0, 0, 0],
            &[0, 0, 0, 57, 200, 14, 11, 203, 57, 0, 0, 0, 0],
            &[0, 0, 0, 44, 17, 0, 21, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 72, 89, 60, 4, 0, 0, 0, 0],
            &[0, 0, 3, 135, 240, 191, 174, 213, 226, 70, 0, 0, 0],
            &[0, 0, 143, 213, 32, 0, 0, 0, 89, 246, 53, 0, 0],
            &[0, 35, 252, 44, 0, 0, 0, 0, 0, 141, 192, 0, 0],
            &[0, 116, 200, 0, 0, 0, 0, 0, 0, 46, 253, 20, 0],
            &[0, 152, 154, 0, 0, 0, 0, 0, 0, 3, 254, 56, 0],
            &[0, 168, 143, 0, 0, 0, 0, 0, 0, 0, 245, 71, 0],
            &[0, 152, 161, 0, 0, 0, 0, 0, 0, 8, 255, 54, 0],
            &[0, 105, 215, 0, 0, 0, 0, 0, 0, 62, 250, 10, 0],
            &[0, 22, 248, 69, 0, 0, 0, 0, 0, 172, 169, 0, 0],
            &[0, 0, 111, 236, 72, 0, 0, 13, 140, 237, 32, 0, 0],
            &[0, 0, 0, 95, 225, 236, 219, 246, 189, 41, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 47, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Œ' / 0x152
        'Œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 4, 129, 234, 254, 255, 254, 254, 254, 254, 241, 0],
            &[0, 0, 159, 213, 55, 3, 153, 168, 25, 25, 25, 23, 0],
            &[0, 52, 249, 35, 0, 0, 146, 159, 0, 0, 0, 0, 0],
            &[0, 141, 179, 0, 0, 0, 146, 159, 0, 0, 0, 0, 0],
            &[0, 197, 120, 0, 0, 0, 146, 159, 0, 0, 0, 0, 0],
            &[0, 229, 93, 0, 0, 0, 146, 159, 0, 0, 0, 0, 0],
            &[0, 245, 78, 0, 0, 0, 146, 200, 107, 107, 107, 66, 0],
            &[0, 250, 66, 0, 0, 0, 146, 224, 171, 171, 171, 105, 0],
            &[0, 239, 75, 0, 0, 0, 146, 159, 0, 0, 0, 0, 0],
            &[0, 224, 94, 0, 0, 0, 146, 159, 0, 0, 0, 0, 0],
            &[0, 183, 135, 0, 0, 0, 146, 159, 0, 0, 0, 0, 0],
            &[0, 129, 203, 0, 0, 0, 146, 159, 0, 0, 0, 0, 0],
            &[0, 35, 253, 60, 0, 0, 146, 159, 0, 0, 0, 0, 0],
            &[0, 0, 143, 228, 72, 8, 155, 170, 28, 28, 28, 27, 0],
            &[0, 0, 3, 126, 239, 255, 255, 255, 255, 255, 255, 242, 0],
            &[0, 0, 0, 0, 0, 22, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'œ' / 0x153
        'œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 83, 69, 2, 0, 0, 44, 80, 12, 0, 0],
            &[0, 37, 229, 190, 196, 177, 0, 116, 212, 179, 212, 9, 0],
            &[0, 182, 135, 0, 0, 178, 97, 212, 10, 0, 185, 106, 0],
            &[18, 252, 23, 0, 0, 78, 244, 120, 0, 0, 96, 178, 0],
            &[65, 220, 0, 0, 0, 33, 255, 72, 0, 0, 64, 215, 0],
            &[90, 200, 0, 0, 0, 12, 255, 137, 109, 109, 136, 229, 0],
            &[96, 189, 0, 0, 0, 6, 255, 154, 132, 132, 132, 120, 0],
            &[84, 203, 0, 0, 0, 15, 255, 53, 0, 0, 0, 0, 0],
            &[54, 228, 0, 0, 0, 41, 255, 83, 0, 0, 0, 0, 0],
            &[9, 245, 37, 0, 0, 96, 228, 146, 0, 0, 0, 0, 0],
            &[0, 154, 170, 3, 13, 208, 64, 228, 44, 0, 6, 57, 0],
            &[0, 16, 198, 235, 236, 122, 0, 78, 234, 205, 231, 113, 0],
            &[0, 0, 1, 36, 21, 0, 0, 0, 11, 43, 13, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŕ' / 0x154
        'Ŕ' => &[
            &[0, 0, 0, 0, 0, 0, 104, 216, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 238, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 212, 77, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 62, 56, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 202, 254, 254, 253, 239, 209, 142, 27, 0, 0, 0, 0],
            &[0, 203, 119, 19, 22, 43, 84, 190, 236, 41, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 3, 193, 170, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 108, 227, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 85, 237, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 134, 197, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 54, 241, 84, 0, 0, 0],
            &[0, 203, 184, 132, 132, 143, 186, 235, 100, 0, 0, 0, 0],
            &[0, 203, 192, 146, 146, 161, 250, 31, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 182, 158, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 37, 247, 57, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 135, 205, 2, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 11, 228, 105, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 88, 238, 20, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 192, 155, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŕ' / 0x155
        'ŕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 5, 22, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 163, 206, 9, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 95, 222, 27, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 57, 211, 33, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 20, 0, 0, 0, 0, 0],
            &[0, 0, 41, 44, 44, 44, 12, 0, 27, 79, 77, 32, 0],
            &[0, 0, 172, 198, 213, 247, 80, 106, 240, 200, 206, 207, 0],
            &[0, 0, 0, 0, 0, 191, 147, 201, 26, 0, 0, 5, 0],
            &[0, 0, 0, 0, 0, 191, 246, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 183, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 136, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 120, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 194, 124, 1, 0, 0, 0, 0, 0],
            &[0, 66, 231, 244, 255, 255, 255, 253, 242, 229, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŗ' / 0x156
        'Ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 202, 254, 254, 253, 239, 209, 142, 27, 0, 0, 0, 0],
            &[0, 203, 119, 19, 22, 43, 84, 190, 236, 41, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 3, 193, 170, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 108, 227, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 85, 237, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 134, 197, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 54, 241, 84, 0, 0, 0],
            &[0, 203, 184, 132, 132, 143, 186, 235, 100, 0, 0, 0, 0],
            &[0, 203, 192, 146, 146, 161, 250, 31, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 182, 158, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 37, 247, 57, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 135, 205, 2, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 11, 228, 105, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 88, 238, 20, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 192, 155, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 107, 115, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 200, 87, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 61, 70, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŗ' / 0x157
        'ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 41, 44, 44, 44, 12, 0, 27, 79, 77, 32, 0],
            &[0, 0, 172, 198, 213, 247, 80, 106, 240, 200, 206, 207, 0],
            &[0, 0, 0, 0, 0, 191, 147, 201, 26, 0, 0, 5, 0],
            &[0, 0, 0, 0, 0, 191, 246, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 183, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 136, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 120, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 194, 124, 1, 0, 0, 0, 0, 0],
            &[0, 66, 231, 244, 255, 255, 255, 253, 242, 229, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 83, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 240, 43, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 41, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ř' / 0x158
        'Ř' => &[
            &[0, 0, 126, 118, 0, 0, 20, 186, 44, 0, 0, 0, 0],
            &[0, 0, 4, 188, 119, 17, 210, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 226, 219, 135, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 53, 112, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 202, 254, 254, 253, 239, 209, 142, 27, 0, 0, 0, 0],
            &[0, 203, 119, 19, 22, 43, 84, 190, 236, 41, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 3, 193, 170, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 108, 227, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 85, 237, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 134, 197, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 54, 241, 84, 0, 0, 0],
            &[0, 203, 184, 132, 132, 143, 186, 235, 100, 0, 0, 0, 0],
            &[0, 203, 192, 146, 146, 161, 250, 31, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 182, 158, 0, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 37, 247, 57, 0, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 135, 205, 2, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 11, 228, 105, 0, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 88, 238, 20, 0, 0],
            &[0, 203, 108, 0, 0, 0, 0, 0, 0, 192, 155, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ř' / 0x159
        'ř' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 0, 0, 0, 0, 10, 5, 0, 0],
            &[0, 0, 0, 0, 142, 140, 0, 0, 58, 208, 24, 0, 0],
            &[0, 0, 0, 0, 2, 178, 136, 52, 227, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 223, 244, 78, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 34, 60, 0, 0, 0, 0, 0],
            &[0, 0, 41, 44, 44, 44, 12, 0, 27, 79, 77, 32, 0],
            &[0, 0, 172, 198, 213, 247, 80, 106, 240, 200, 206, 207, 0],
            &[0, 0, 0, 0, 0, 191, 147, 201, 26, 0, 0, 5, 0],
            &[0, 0, 0, 0, 0, 191, 246, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 183, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 136, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 120, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 191, 119, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 194, 124, 1, 0, 0, 0, 0, 0],
            &[0, 66, 231, 244, 255, 255, 255, 253, 242, 229, 45, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ś' / 0x15a
        'Ś' => &[
            &[0, 0, 0, 0, 0, 0, 0, 21, 218, 106, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 181, 166, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 133, 172, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 101, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 43, 37, 11, 0, 0, 0, 0],
            &[0, 0, 0, 63, 199, 252, 235, 243, 255, 206, 86, 0, 0],
            &[0, 0, 59, 248, 130, 20, 0, 0, 20, 94, 63, 0, 0],
            &[0, 0, 179, 156, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 220, 96, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 205, 121, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 116, 234, 51, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 144, 251, 173, 65, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 151, 245, 210, 100, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 17, 115, 227, 211, 28, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 11, 189, 185, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 66, 251, 3, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 57, 255, 7, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 134, 213, 0, 0],
            &[0, 24, 119, 45, 2, 0, 0, 20, 126, 250, 78, 0, 0],
            &[0, 20, 173, 241, 247, 226, 231, 253, 200, 69, 0, 0, 0],
            &[0, 0, 0, 2, 30, 48, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ś' / 0x15b
        'ś' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 22, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 122, 230, 25, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 59, 235, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 213, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 31, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 71, 92, 77, 43, 0, 0, 0, 0],
            &[0, 0, 0, 111, 245, 204, 177, 191, 233, 199, 0, 0, 0],
            &[0, 0, 39, 252, 60, 0, 0, 0, 2, 32, 0, 0, 0],
            &[0, 0, 82, 232, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 40, 254, 78, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 105, 244, 177, 75, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 118, 221, 224, 117, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 71, 220, 190, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 44, 255, 44, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 17, 255, 49, 0, 0],
            &[0, 0, 93, 73, 5, 0, 0, 15, 158, 221, 4, 0, 0],
            &[0, 0, 92, 215, 249, 222, 226, 248, 188, 37, 0, 0, 0],
            &[0, 0, 0, 0, 13, 40, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => &[
            &[0, 0, 0, 0, 0, 40, 222, 143, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 208, 116, 218, 83, 0, 0, 0, 0],
            &[0, 0, 0, 2, 173, 117, 0, 33, 214, 51, 0, 0, 0],
            &[0, 0, 0, 34, 83, 0, 0, 0, 30, 87, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 43, 37, 11, 0, 0, 0, 0],
            &[0, 0, 0, 63, 199, 252, 235, 243, 255, 206, 86, 0, 0],
            &[0, 0, 59, 248, 130, 20, 0, 0, 20, 94, 63, 0, 0],
            &[0, 0, 179, 156, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 220, 96, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 205, 121, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 116, 234, 51, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 144, 251, 173, 65, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 151, 245, 210, 100, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 17, 115, 227, 211, 28, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 11, 189, 185, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 66, 251, 3, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 57, 255, 7, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 134, 213, 0, 0],
            &[0, 24, 119, 45, 2, 0, 0, 20, 126, 250, 78, 0, 0],
            &[0, 20, 173, 241, 247, 226, 231, 253, 200, 69, 0, 0, 0],
            &[0, 0, 0, 2, 30, 48, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŝ' / 0x15d
        'ŝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 153, 255, 91, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 92, 206, 62, 228, 42, 0, 0, 0, 0],
            &[0, 0, 0, 60, 204, 23, 0, 62, 208, 22, 0, 0, 0],
            &[0, 0, 0, 49, 19, 0, 0, 0, 37, 32, 0, 0, 0],
            &[0, 0, 0, 0, 16, 71, 92, 77, 43, 0, 0, 0, 0],
            &[0, 0, 0, 111, 245, 204, 177, 191, 233, 199, 0, 0, 0],
            &[0, 0, 39, 252, 60, 0, 0, 0, 2, 32, 0, 0, 0],
            &[0, 0, 82, 232, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 40, 254, 78, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 105, 244, 177, 75, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 118, 221, 224, 117, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 71, 220, 190, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 44, 255, 44, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 17, 255, 49, 0, 0],
            &[0, 0, 93, 73, 5, 0, 0, 15, 158, 221, 4, 0, 0],
            &[0, 0, 92, 215, 249, 222, 226, 248, 188, 37, 0, 0, 0],
            &[0, 0, 0, 0, 13, 40, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ş' / 0x15e
        'Ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 43, 37, 11, 0, 0, 0, 0],
            &[0, 0, 0, 63, 199, 252, 235, 243, 255, 206, 86, 0, 0],
            &[0, 0, 59, 248, 130, 20, 0, 0, 20, 94, 63, 0, 0],
            &[0, 0, 179, 156, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 220, 96, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 205, 121, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 116, 234, 51, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 144, 251, 173, 65, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 151, 245, 210, 100, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 17, 115, 227, 211, 28, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 11, 189, 185, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 66, 251, 3, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 57, 255, 7, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 134, 213, 0, 0],
            &[0, 24, 119, 45, 2, 0, 0, 20, 126, 250, 78, 0, 0],
            &[0, 20, 173, 241, 247, 227, 244, 253, 200, 69, 0, 0, 0],
            &[0, 0, 0, 2, 30, 139, 183, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 174, 187, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 238, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 4, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ş' / 0x15f
        'ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 71, 92, 77, 43, 0, 0, 0, 0],
            &[0, 0, 0, 111, 245, 204, 177, 191, 233, 199, 0, 0, 0],
            &[0, 0, 39, 252, 60, 0, 0, 0, 2, 32, 0, 0, 0],
            &[0, 0, 82, 232, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 40, 254, 78, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 105, 244, 177, 75, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 118, 221, 224, 117, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 71, 220, 190, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 44, 255, 44, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 17, 255, 49, 0, 0],
            &[0, 0, 93, 73, 5, 0, 0, 15, 158, 221, 4, 0, 0],
            &[0, 0, 92, 215, 249, 223, 240, 248, 188, 37, 0, 0, 0],
            &[0, 0, 0, 0, 13, 114, 198, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 160, 193, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 240, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Š' / 0x160
        'Š' => &[
            &[0, 0, 0, 39, 185, 20, 0, 0, 118, 133, 0, 0, 0],
            &[0, 0, 0, 0, 87, 210, 16, 121, 189, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 138, 219, 224, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 113, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 43, 37, 11, 0, 0, 0, 0],
            &[0, 0, 0, 63, 199, 252, 235, 243, 255, 206, 86, 0, 0],
            &[0, 0, 59, 248, 130, 20, 0, 0, 20, 94, 63, 0, 0],
            &[0, 0, 179, 156, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 220, 96, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 205, 121, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 116, 234, 51, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 144, 251, 173, 65, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 151, 245, 210, 100, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 17, 115, 227, 211, 28, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 11, 189, 185, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 66, 251, 3, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 57, 255, 7, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 134, 213, 0, 0],
            &[0, 24, 119, 45, 2, 0, 0, 20, 126, 250, 78, 0, 0],
            &[0, 20, 173, 241, 247, 226, 231, 253, 200, 69, 0, 0, 0],
            &[0, 0, 0, 2, 30, 48, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'š' / 0x161
        'š' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 2, 0, 0, 0, 7, 7, 0, 0, 0],
            &[0, 0, 0, 102, 175, 7, 0, 34, 209, 48, 0, 0, 0],
            &[0, 0, 0, 0, 139, 174, 31, 222, 69, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 194, 244, 118, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 69, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 71, 92, 77, 43, 0, 0, 0, 0],
            &[0, 0, 0, 111, 245, 204, 177, 191, 233, 199, 0, 0, 0],
            &[0, 0, 39, 252, 60, 0, 0, 0, 2, 32, 0, 0, 0],
            &[0, 0, 82, 232, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 40, 254, 78, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 105, 244, 177, 75, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 118, 221, 224, 117, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 71, 220, 190, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 44, 255, 44, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 17, 255, 49, 0, 0],
            &[0, 0, 93, 73, 5, 0, 0, 15, 158, 221, 4, 0, 0],
            &[0, 0, 92, 215, 249, 222, 226, 248, 188, 37, 0, 0, 0],
            &[0, 0, 0, 0, 13, 40, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ţ' / 0x162
        'Ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 252, 254, 254, 254, 254, 254, 254, 254, 254, 254, 155, 0],
            &[0, 24, 25, 25, 25, 96, 234, 25, 25, 25, 25, 15, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 245, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 90, 138, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 174, 187, 59, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 238, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 4, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ţ' / 0x163
        'ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 254, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 2, 17, 33, 85, 255, 49, 44, 44, 44, 22, 0, 0],
            &[0, 119, 224, 224, 230, 255, 225, 224, 224, 224, 112, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 192, 180, 11, 0, 0, 7, 0, 0],
            &[0, 0, 0, 0, 0, 32, 198, 251, 223, 232, 138, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 192, 109, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 224, 151, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 122, 166, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 1, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ť' / 0x164
        'Ť' => &[
            &[0, 0, 0, 126, 118, 0, 0, 20, 186, 44, 0, 0, 0],
            &[0, 0, 0, 4, 188, 119, 17, 210, 88, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 226, 219, 135, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 53, 112, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 252, 254, 254, 254, 254, 254, 254, 254, 254, 254, 155, 0],
            &[0, 24, 25, 25, 25, 96, 234, 25, 25, 25, 25, 15, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ť' / 0x165
        'ť' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 16, 10, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 222, 82, 0, 0],
            &[0, 0, 0, 0, 0, 12, 0, 0, 27, 237, 9, 0, 0],
            &[0, 0, 0, 0, 4, 254, 6, 0, 102, 128, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 6, 0, 41, 10, 0, 0, 0],
            &[0, 2, 17, 33, 85, 255, 49, 44, 44, 44, 22, 0, 0],
            &[0, 119, 224, 224, 230, 255, 225, 224, 224, 224, 112, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 192, 180, 11, 0, 0, 7, 0, 0],
            &[0, 0, 0, 0, 0, 32, 198, 245, 215, 232, 138, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 25, 48, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŧ' / 0x166
        'Ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 252, 254, 254, 254, 254, 254, 254, 254, 254, 254, 155, 0],
            &[0, 24, 25, 25, 25, 96, 234, 25, 25, 25, 25, 15, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 48, 48, 127, 255, 48, 48, 48, 16, 0, 0],
            &[0, 0, 156, 220, 220, 255, 255, 220, 220, 220, 73, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŧ' / 0x167
        'ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 254, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 2, 17, 33, 85, 255, 49, 44, 44, 44, 22, 0, 0],
            &[0, 119, 224, 224, 230, 255, 225, 224, 224, 224, 112, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 120, 227, 227, 255, 255, 234, 227, 227, 227, 147, 0, 0],
            &[0, 7, 14, 14, 63, 255, 21, 14, 14, 14, 9, 0, 0],
            &[0, 0, 0, 0, 48, 255, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 44, 255, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 255, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 192, 180, 11, 0, 0, 7, 0, 0],
            &[0, 0, 0, 0, 0, 32, 198, 245, 215, 232, 138, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 25, 48, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ũ' / 0x168
        'Ũ' => &[
            &[0, 0, 0, 0, 2, 0, 0, 0, 0, 13, 0, 0, 0],
            &[0, 0, 0, 123, 245, 220, 78, 0, 31, 199, 0, 0, 0],
            &[0, 0, 27, 214, 17, 52, 208, 198, 225, 86, 0, 0, 0],
            &[0, 0, 21, 47, 0, 0, 4, 62, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 86, 223, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 64, 247, 0, 0, 0, 0, 0, 0, 88, 223, 0, 0],
            &[0, 50, 255, 5, 0, 0, 0, 0, 0, 102, 208, 0, 0],
            &[0, 21, 255, 44, 0, 0, 0, 0, 0, 143, 176, 0, 0],
            &[0, 0, 208, 131, 0, 0, 0, 0, 7, 227, 104, 0, 0],
            &[0, 0, 80, 249, 98, 4, 0, 30, 178, 217, 7, 0, 0],
            &[0, 0, 0, 94, 228, 251, 240, 253, 179, 26, 0, 0, 0],
            &[0, 0, 0, 0, 3, 31, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ũ' / 0x169
        'ũ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 17, 0, 0, 0, 51, 0, 0, 0],
            &[0, 0, 0, 165, 224, 236, 123, 4, 53, 189, 0, 0, 0],
            &[0, 0, 38, 199, 0, 19, 168, 238, 236, 59, 0, 0, 0],
            &[0, 0, 10, 20, 0, 0, 0, 18, 8, 0, 0, 0, 0],
            &[0, 6, 44, 2, 0, 0, 0, 0, 0, 24, 29, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 17, 0, 0, 0, 0, 0, 152, 170, 0, 0],
            &[0, 24, 255, 35, 0, 0, 0, 0, 0, 189, 170, 0, 0],
            &[0, 0, 244, 87, 0, 0, 0, 0, 33, 249, 170, 0, 0],
            &[0, 0, 143, 218, 40, 0, 0, 41, 197, 155, 170, 0, 0],
            &[0, 0, 14, 173, 253, 227, 226, 213, 60, 92, 170, 0, 0],
            &[0, 0, 0, 0, 18, 48, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ū' / 0x16a
        'Ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 149, 249, 249, 249, 249, 249, 49, 0, 0, 0],
            &[0, 0, 0, 4, 8, 8, 8, 8, 8, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 86, 223, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 64, 247, 0, 0, 0, 0, 0, 0, 88, 223, 0, 0],
            &[0, 50, 255, 5, 0, 0, 0, 0, 0, 102, 208, 0, 0],
            &[0, 21, 255, 44, 0, 0, 0, 0, 0, 143, 176, 0, 0],
            &[0, 0, 208, 131, 0, 0, 0, 0, 7, 227, 104, 0, 0],
            &[0, 0, 80, 249, 98, 4, 0, 30, 178, 217, 7, 0, 0],
            &[0, 0, 0, 94, 228, 251, 240, 253, 179, 26, 0, 0, 0],
            &[0, 0, 0, 0, 3, 31, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ū' / 0x16b
        'ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 23, 39, 39, 39, 39, 39, 7, 0, 0, 0],
            &[0, 0, 0, 130, 218, 218, 218, 218, 218, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 44, 2, 0, 0, 0, 0, 0, 24, 29, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 17, 0, 0, 0, 0, 0, 152, 170, 0, 0],
            &[0, 24, 255, 35, 0, 0, 0, 0, 0, 189, 170, 0, 0],
            &[0, 0, 244, 87, 0, 0, 0, 0, 33, 249, 170, 0, 0],
            &[0, 0, 143, 218, 40, 0, 0, 41, 197, 155, 170, 0, 0],
            &[0, 0, 14, 173, 253, 227, 226, 213, 60, 92, 170, 0, 0],
            &[0, 0, 0, 0, 18, 48, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => &[
            &[0, 0, 0, 35, 18, 0, 0, 0, 42, 11, 0, 0, 0],
            &[0, 0, 0, 111, 152, 0, 0, 13, 224, 13, 0, 0, 0],
            &[0, 0, 0, 9, 197, 191, 152, 217, 109, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 57, 77, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 86, 223, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 64, 247, 0, 0, 0, 0, 0, 0, 88, 223, 0, 0],
            &[0, 50, 255, 5, 0, 0, 0, 0, 0, 102, 208, 0, 0],
            &[0, 21, 255, 44, 0, 0, 0, 0, 0, 143, 176, 0, 0],
            &[0, 0, 208, 131, 0, 0, 0, 0, 7, 227, 104, 0, 0],
            &[0, 0, 80, 249, 98, 4, 0, 30, 178, 217, 7, 0, 0],
            &[0, 0, 0, 94, 228, 251, 240, 253, 179, 26, 0, 0, 0],
            &[0, 0, 0, 0, 3, 31, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŭ' / 0x16d
        'ŭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 36, 0, 0, 0, 76, 17, 0, 0, 0],
            &[0, 0, 0, 94, 180, 6, 0, 34, 224, 7, 0, 0, 0],
            &[0, 0, 0, 1, 154, 229, 198, 224, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 31, 3, 0, 0, 0, 0, 0],
            &[0, 6, 44, 2, 0, 0, 0, 0, 0, 24, 29, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 17, 0, 0, 0, 0, 0, 152, 170, 0, 0],
            &[0, 24, 255, 35, 0, 0, 0, 0, 0, 189, 170, 0, 0],
            &[0, 0, 244, 87, 0, 0, 0, 0, 33, 249, 170, 0, 0],
            &[0, 0, 143, 218, 40, 0, 0, 41, 197, 155, 170, 0, 0],
            &[0, 0, 14, 173, 253, 227, 226, 213, 60, 92, 170, 0, 0],
            &[0, 0, 0, 0, 18, 48, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ů' / 0x16e
        'Ů' => &[
            &[0, 0, 0, 0, 190, 68, 4, 166, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 214, 16, 0, 116, 114, 0, 0, 0, 0],
            &[0, 0, 0, 0, 124, 188, 128, 214, 33, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 86, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 86, 223, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 64, 247, 0, 0, 0, 0, 0, 0, 88, 223, 0, 0],
            &[0, 50, 255, 5, 0, 0, 0, 0, 0, 102, 208, 0, 0],
            &[0, 21, 255, 44, 0, 0, 0, 0, 0, 143, 176, 0, 0],
            &[0, 0, 208, 131, 0, 0, 0, 0, 7, 227, 104, 0, 0],
            &[0, 0, 80, 249, 98, 4, 0, 30, 178, 217, 7, 0, 0],
            &[0, 0, 0, 94, 228, 251, 240, 253, 179, 26, 0, 0, 0],
            &[0, 0, 0, 0, 3, 31, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 190, 205, 138, 1, 0, 0, 0, 0],
        ],
        // letter: 'ů' / 0x16f
        'ů' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 67, 219, 198, 182, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 204, 38, 0, 139, 101, 0, 0, 0, 0],
            &[0, 0, 0, 0, 209, 28, 0, 129, 108, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 215, 173, 203, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 41, 3, 0, 0, 0, 0, 0],
            &[0, 6, 44, 2, 0, 0, 0, 0, 0, 24, 29, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 17, 0, 0, 0, 0, 0, 152, 170, 0, 0],
            &[0, 24, 255, 35, 0, 0, 0, 0, 0, 189, 170, 0, 0],
            &[0, 0, 244, 87, 0, 0, 0, 0, 33, 249, 170, 0, 0],
            &[0, 0, 143, 218, 40, 0, 0, 41, 197, 155, 170, 0, 0],
            &[0, 0, 14, 173, 253, 227, 226, 213, 60, 92, 170, 0, 0],
            &[0, 0, 0, 0, 18, 48, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ű' / 0x170
        'Ű' => &[
            &[0, 0, 0, 0, 1, 192, 123, 0, 119, 194, 3, 0, 0],
            &[0, 0, 0, 0, 103, 208, 9, 32, 235, 52, 0, 0, 0],
            &[0, 0, 0, 34, 215, 31, 2, 189, 89, 0, 0, 0, 0],
            &[0, 0, 0, 67, 38, 0, 30, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 86, 223, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 64, 247, 0, 0, 0, 0, 0, 0, 88, 223, 0, 0],
            &[0, 50, 255, 5, 0, 0, 0, 0, 0, 102, 208, 0, 0],
            &[0, 21, 255, 44, 0, 0, 0, 0, 0, 143, 176, 0, 0],
            &[0, 0, 208, 131, 0, 0, 0, 0, 7, 227, 104, 0, 0],
            &[0, 0, 80, 249, 98, 4, 0, 30, 178, 217, 7, 0, 0],
            &[0, 0, 0, 94, 228, 251, 240, 253, 179, 26, 0, 0, 0],
            &[0, 0, 0, 0, 3, 31, 44, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ű' / 0x171
        'ű' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 16, 0, 6, 22, 1, 0, 0],
            &[0, 0, 0, 0, 8, 224, 115, 0, 150, 194, 1, 0, 0],
            &[0, 0, 0, 0, 132, 180, 2, 53, 230, 29, 0, 0, 0],
            &[0, 0, 0, 57, 200, 14, 11, 203, 57, 0, 0, 0, 0],
            &[0, 0, 0, 44, 17, 0, 21, 39, 0, 0, 0, 0, 0],
            &[0, 6, 44, 2, 0, 0, 0, 0, 0, 24, 29, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 17, 0, 0, 0, 0, 0, 152, 170, 0, 0],
            &[0, 24, 255, 35, 0, 0, 0, 0, 0, 189, 170, 0, 0],
            &[0, 0, 244, 87, 0, 0, 0, 0, 33, 249, 170, 0, 0],
            &[0, 0, 143, 218, 40, 0, 0, 41, 197, 155, 170, 0, 0],
            &[0, 0, 14, 173, 253, 227, 226, 213, 60, 92, 170, 0, 0],
            &[0, 0, 0, 0, 18, 48, 34, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ų' / 0x172
        'Ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 244, 0, 0, 0, 0, 0, 0, 86, 223, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 65, 245, 0, 0, 0, 0, 0, 0, 87, 224, 0, 0],
            &[0, 64, 247, 0, 0, 0, 0, 0, 0, 88, 223, 0, 0],
            &[0, 50, 255, 5, 0, 0, 0, 0, 0, 102, 208, 0, 0],
            &[0, 21, 255, 44, 0, 0, 0, 0, 0, 143, 176, 0, 0],
            &[0, 0, 208, 131, 0, 0, 0, 0, 7, 227, 104, 0, 0],
            &[0, 0, 80, 249, 98, 4, 0, 30, 178, 217, 7, 0, 0],
            &[0, 0, 0, 94, 228, 251, 240, 255, 233, 26, 0, 0, 0],
            &[0, 0, 0, 0, 3, 36, 203, 138, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 152, 127, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 234, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 127, 134, 16, 0, 0, 0, 0],
        ],
        // letter: 'ų' / 0x173
        'ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 6, 44, 2, 0, 0, 0, 0, 0, 24, 29, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 16, 0, 0, 0, 0, 0, 140, 170, 0, 0],
            &[0, 38, 255, 17, 0, 0, 0, 0, 0, 152, 170, 0, 0],
            &[0, 24, 255, 35, 0, 0, 0, 0, 0, 189, 170, 0, 0],
            &[0, 0, 244, 87, 0, 0, 0, 0, 33, 249, 170, 0, 0],
            &[0, 0, 143, 218, 40, 0, 0, 41, 197, 155, 170, 0, 0],
            &[0, 0, 14, 173, 253, 227, 226, 213, 60, 108, 190, 0, 0],
            &[0, 0, 0, 0, 18, 48, 34, 0, 23, 198, 51, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 175, 76, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 7, 247, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 40, 133, 130, 2, 0],
        ],
        // letter: 'Ŵ' / 0x174
        'Ŵ' => &[
            &[0, 0, 0, 0, 0, 131, 225, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 219, 113, 215, 15, 0, 0, 0, 0],
            &[0, 0, 0, 43, 212, 36, 0, 111, 183, 4, 0, 0, 0],
            &[0, 0, 0, 83, 35, 0, 0, 0, 79, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[60, 232, 0, 0, 0, 0, 0, 0, 0, 0, 74, 219, 0],
            &[26, 255, 12, 0, 0, 0, 0, 0, 0, 0, 108, 186, 0],
            &[1, 246, 45, 0, 0, 0, 0, 0, 0, 0, 142, 152, 0],
            &[0, 212, 79, 0, 0, 13, 40, 0, 0, 0, 176, 118, 0],
            &[0, 178, 113, 0, 0, 113, 254, 17, 0, 0, 211, 84, 0],
            &[0, 143, 147, 0, 0, 172, 203, 75, 0, 0, 245, 50, 0],
            &[0, 108, 180, 0, 0, 227, 107, 134, 0, 24, 255, 16, 0],
            &[0, 73, 214, 0, 34, 207, 43, 193, 0, 58, 238, 0, 0],
            &[0, 39, 247, 1, 93, 150, 1, 230, 5, 92, 204, 0, 0],
            &[0, 6, 253, 25, 152, 92, 0, 182, 56, 121, 170, 0, 0],
            &[0, 0, 225, 55, 209, 34, 0, 123, 116, 144, 136, 0, 0],
            &[0, 0, 190, 93, 228, 0, 0, 65, 177, 167, 102, 0, 0],
            &[0, 0, 156, 167, 174, 0, 0, 10, 226, 187, 67, 0, 0],
            &[0, 0, 121, 235, 116, 0, 0, 0, 203, 233, 33, 0, 0],
            &[0, 0, 86, 255, 58, 0, 0, 0, 145, 251, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŵ' / 0x175
        'ŵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 148, 255, 96, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 209, 61, 228, 45, 0, 0, 0, 0],
            &[0, 0, 0, 56, 206, 25, 0, 58, 209, 25, 0, 0, 0],
            &[0, 0, 0, 48, 20, 0, 0, 0, 35, 33, 0, 0, 0],
            &[23, 26, 0, 0, 0, 0, 0, 0, 0, 0, 0, 41, 7],
            &[107, 183, 0, 0, 0, 0, 0, 0, 0, 0, 17, 253, 14],
            &[55, 236, 0, 0, 0, 8, 29, 0, 0, 0, 68, 218, 0],
            &[8, 250, 34, 0, 0, 121, 255, 50, 0, 0, 119, 168, 0],
            &[0, 206, 87, 0, 0, 198, 192, 126, 0, 0, 170, 118, 0],
            &[0, 153, 141, 0, 22, 241, 74, 203, 0, 0, 221, 68, 0],
            &[0, 101, 195, 0, 98, 177, 7, 244, 25, 17, 254, 18, 0],
            &[0, 49, 245, 2, 175, 103, 0, 179, 100, 67, 224, 0, 0],
            &[0, 5, 247, 41, 242, 28, 0, 103, 174, 110, 174, 0, 0],
            &[0, 0, 200, 129, 208, 0, 0, 28, 238, 146, 124, 0, 0],
            &[0, 0, 147, 215, 132, 0, 0, 0, 207, 216, 74, 0, 0],
            &[0, 0, 95, 255, 56, 0, 0, 0, 132, 255, 23, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŷ' / 0x176
        'Ŷ' => &[
            &[0, 0, 0, 0, 0, 131, 225, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 73, 219, 113, 215, 15, 0, 0, 0, 0],
            &[0, 0, 0, 43, 212, 36, 0, 111, 183, 4, 0, 0, 0],
            &[0, 0, 0, 83, 35, 0, 0, 0, 79, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[2, 210, 126, 0, 0, 0, 0, 0, 0, 2, 214, 116, 0],
            &[0, 80, 239, 15, 0, 0, 0, 0, 0, 88, 230, 9, 0],
            &[0, 0, 201, 126, 0, 0, 0, 0, 1, 212, 106, 0, 0],
            &[0, 0, 68, 239, 15, 0, 0, 0, 85, 223, 5, 0, 0],
            &[0, 0, 0, 190, 126, 0, 0, 1, 210, 95, 0, 0, 0],
            &[0, 0, 0, 56, 238, 15, 0, 82, 214, 3, 0, 0, 0],
            &[0, 0, 0, 0, 177, 126, 1, 208, 84, 0, 0, 0, 0],
            &[0, 0, 0, 0, 45, 237, 95, 205, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 165, 251, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 235, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŷ' / 0x177
        'ŷ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 15, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 169, 254, 75, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 196, 68, 226, 32, 0, 0, 0, 0],
            &[0, 0, 0, 72, 198, 17, 0, 73, 203, 16, 0, 0, 0],
            &[0, 0, 0, 54, 14, 0, 0, 0, 41, 27, 0, 0, 0],
            &[0, 23, 29, 0, 0, 0, 0, 0, 0, 2, 44, 8, 0],
            &[0, 78, 228, 3, 0, 0, 0, 0, 0, 68, 241, 9, 0],
            &[0, 3, 228, 75, 0, 0, 0, 0, 0, 160, 154, 0, 0],
            &[0, 0, 130, 174, 0, 0, 0, 0, 9, 242, 58, 0, 0],
            &[0, 0, 31, 248, 23, 0, 0, 0, 87, 217, 0, 0, 0],
            &[0, 0, 0, 183, 117, 0, 0, 0, 179, 121, 0, 0, 0],
            &[0, 0, 0, 81, 216, 0, 0, 20, 248, 27, 0, 0, 0],
            &[0, 0, 0, 4, 231, 60, 0, 107, 184, 0, 0, 0, 0],
            &[0, 0, 0, 0, 133, 158, 0, 198, 87, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 241, 41, 238, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 186, 195, 150, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 255, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 86, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 182, 117, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 243, 19, 0, 0, 0, 0, 0, 0],
            &[0, 2, 33, 23, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ÿ' / 0x178
        'Ÿ' => &[
            &[0, 0, 0, 0, 2, 0, 0, 0, 2, 0, 0, 0, 0],
            &[0, 0, 0, 155, 197, 0, 0, 44, 244, 61, 0, 0, 0],
            &[0, 0, 0, 119, 155, 0, 0, 29, 200, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[2, 210, 126, 0, 0, 0, 0, 0, 0, 2, 214, 116, 0],
            &[0, 80, 239, 15, 0, 0, 0, 0, 0, 88, 230, 9, 0],
            &[0, 0, 201, 126, 0, 0, 0, 0, 1, 212, 106, 0, 0],
            &[0, 0, 68, 239, 15, 0, 0, 0, 85, 223, 5, 0, 0],
            &[0, 0, 0, 190, 126, 0, 0, 1, 210, 95, 0, 0, 0],
            &[0, 0, 0, 56, 238, 15, 0, 82, 214, 3, 0, 0, 0],
            &[0, 0, 0, 0, 177, 126, 1, 208, 84, 0, 0, 0, 0],
            &[0, 0, 0, 0, 45, 237, 95, 205, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 165, 251, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 235, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 232, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ź' / 0x179
        'Ź' => &[
            &[0, 0, 0, 0, 0, 0, 0, 37, 226, 81, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 202, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 149, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 92, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 254, 254, 254, 254, 254, 254, 254, 94, 0, 0],
            &[0, 0, 5, 19, 19, 19, 19, 19, 65, 246, 27, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 179, 140, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 58, 241, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 193, 127, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 235, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 205, 114, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 86, 227, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 217, 102, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 219, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 227, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 114, 208, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 235, 76, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 210, 29, 28, 28, 28, 28, 28, 14, 0, 0],
            &[0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 127, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ź' / 0x17a
        'ź' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 84, 245, 48, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 233, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 203, 83, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 26, 41, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 44, 44, 44, 44, 44, 44, 44, 16, 0, 0],
            &[0, 0, 82, 208, 208, 208, 208, 208, 214, 255, 88, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 140, 202, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 71, 240, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 232, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 183, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 114, 216, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 242, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 217, 109, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 158, 177, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 88, 234, 37, 18, 18, 18, 18, 18, 7, 0, 0],
            &[0, 0, 211, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ż' / 0x17b
        'Ż' => &[
            &[0, 0, 0, 0, 0, 3, 71, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 70, 255, 129, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 156, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 254, 254, 254, 254, 254, 254, 254, 94, 0, 0],
            &[0, 0, 5, 19, 19, 19, 19, 19, 65, 246, 27, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 179, 140, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 58, 241, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 193, 127, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 235, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 205, 114, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 86, 227, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 217, 102, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 219, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 227, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 114, 208, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 235, 76, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 210, 29, 28, 28, 28, 28, 28, 14, 0, 0],
            &[0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 127, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ż' / 0x17c
        'ż' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 150, 34, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 98, 255, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 80, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 44, 44, 44, 44, 44, 44, 44, 16, 0, 0],
            &[0, 0, 82, 208, 208, 208, 208, 208, 214, 255, 88, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 140, 202, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 71, 240, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 232, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 183, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 114, 216, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 242, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 217, 109, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 158, 177, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 88, 234, 37, 18, 18, 18, 18, 18, 7, 0, 0],
            &[0, 0, 211, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ž' / 0x17d
        'Ž' => &[
            &[0, 0, 0, 57, 176, 11, 0, 1, 141, 109, 0, 0, 0],
            &[0, 0, 0, 0, 113, 192, 8, 148, 167, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 165, 218, 207, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 115, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 74, 254, 254, 254, 254, 254, 254, 254, 94, 0, 0],
            &[0, 0, 5, 19, 19, 19, 19, 19, 65, 246, 27, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 179, 140, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 58, 241, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 193, 127, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 72, 235, 13, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 205, 114, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 86, 227, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 217, 102, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 219, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 227, 89, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 114, 208, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 235, 76, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 210, 29, 28, 28, 28, 28, 28, 14, 0, 0],
            &[0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 127, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ž' / 0x17e
        'ž' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 4, 0, 0, 0, 4, 10, 0, 0, 0],
            &[0, 0, 0, 69, 196, 18, 0, 18, 197, 75, 0, 0, 0],
            &[0, 0, 0, 0, 102, 203, 25, 204, 102, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 159, 244, 156, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 70, 11, 0, 0, 0, 0, 0],
            &[0, 0, 17, 44, 44, 44, 44, 44, 44, 44, 16, 0, 0],
            &[0, 0, 82, 208, 208, 208, 208, 208, 214, 255, 88, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 140, 202, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 71, 240, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 232, 89, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 183, 158, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 114, 216, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 242, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 217, 109, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 158, 177, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 88, 234, 37, 18, 18, 18, 18, 18, 7, 0, 0],
            &[0, 0, 211, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ſ' / 0x17f
        'ſ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 75, 205, 253, 253, 229, 5, 0, 0, 0],
            &[0, 0, 0, 63, 248, 104, 15, 1, 27, 0, 0, 0, 0],
            &[0, 0, 0, 179, 153, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 225, 90, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 240, 67, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 240, 66, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 240, 66, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 240, 66, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 240, 66, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 240, 66, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 240, 66, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 240, 66, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 240, 66, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 240, 66, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 240, 66, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 240, 66, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        _ => panic!("unsupported char"),
    }
}
