//! Module for letters with the font weight bold and size 24.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 24;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 15;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight bold and font size 21px
#[inline]
pub const fn get_char(c: char) -> &'static [&'static [u8]] {
    match c {
        // letter: ' ' / 0x20
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '!' / 0x21
        '!' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 254, 254, 254, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 101, 255, 255, 255, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 90, 255, 255, 249, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 79, 255, 255, 238, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 255, 255, 227, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 255, 255, 215, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 255, 255, 204, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 34, 255, 255, 193, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 255, 255, 181, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 255, 255, 170, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 5, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 82, 113, 37, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 103, 255, 255, 237, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 145, 255, 255, 255, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 246, 255, 195, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 41, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '"' / 0x22
        '"' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 59, 254, 254, 155, 4, 253, 254, 211, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 255, 137, 0, 240, 255, 194, 0, 0, 0, 0],
            &[0, 0, 0, 22, 255, 255, 119, 0, 221, 255, 175, 0, 0, 0, 0],
            &[0, 0, 0, 4, 254, 255, 100, 0, 202, 255, 156, 0, 0, 0, 0],
            &[0, 0, 0, 0, 240, 255, 81, 0, 183, 255, 137, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 108, 28, 0, 72, 108, 52, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '#' / 0x23
        '#' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 149, 254, 135, 0, 167, 254, 111, 0, 0, 0],
            &[0, 0, 0, 0, 0, 193, 255, 93, 0, 210, 255, 71, 0, 0, 0],
            &[0, 0, 0, 0, 0, 236, 255, 50, 3, 248, 255, 29, 0, 0, 0],
            &[0, 0, 0, 0, 23, 255, 253, 9, 37, 255, 243, 0, 0, 0, 0],
            &[
                0, 16, 219, 219, 226, 255, 253, 219, 228, 255, 250, 219, 219, 62, 0,
            ],
            &[
                0, 19, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 72, 0,
            ],
            &[
                0, 4, 56, 56, 172, 255, 164, 56, 186, 255, 146, 56, 56, 16, 0,
            ],
            &[0, 0, 0, 0, 186, 255, 100, 0, 204, 255, 76, 0, 0, 0, 0],
            &[0, 45, 66, 66, 231, 255, 113, 66, 245, 255, 95, 66, 44, 0, 0],
            &[
                0, 175, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 172, 0, 0,
            ],
            &[
                0, 140, 204, 219, 255, 246, 204, 224, 255, 243, 204, 204, 138, 0, 0,
            ],
            &[0, 0, 0, 96, 255, 184, 0, 122, 255, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 136, 255, 144, 0, 164, 255, 127, 0, 0, 0, 0, 0],
            &[0, 0, 0, 176, 255, 105, 0, 207, 255, 85, 0, 0, 0, 0, 0],
            &[0, 0, 0, 215, 255, 65, 2, 246, 255, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '$' / 0x24
        '$' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 79, 240, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 58, 157, 255, 118, 79, 19, 0, 0, 0, 0],
            &[0, 0, 0, 31, 201, 255, 255, 255, 255, 255, 250, 126, 0, 0, 0],
            &[0, 0, 0, 206, 255, 255, 249, 255, 241, 255, 255, 91, 0, 0, 0],
            &[0, 0, 47, 255, 255, 161, 87, 255, 9, 35, 120, 11, 0, 0, 0],
            &[0, 0, 68, 255, 255, 110, 84, 255, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 252, 255, 239, 167, 255, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 255, 255, 255, 136, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 227, 255, 255, 255, 248, 124, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 132, 255, 255, 255, 255, 109, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 84, 255, 72, 243, 255, 209, 0, 0, 0],
            &[0, 0, 8, 0, 0, 0, 84, 255, 8, 199, 255, 222, 0, 0, 0],
            &[0, 0, 76, 214, 109, 38, 88, 255, 96, 251, 255, 175, 0, 0, 0],
            &[
                0, 0, 76, 255, 255, 255, 255, 255, 255, 255, 250, 48, 0, 0, 0,
            ],
            &[0, 0, 36, 175, 246, 255, 255, 255, 254, 204, 55, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 32, 123, 255, 31, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 84, 255, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 55, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '%' / 0x25
        '%' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 18, 17, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 184, 255, 255, 188, 20, 0, 0, 89, 254, 222, 5, 0, 0],
            &[0, 147, 255, 227, 225, 255, 152, 0, 3, 216, 255, 99, 0, 0, 0],
            &[0, 224, 255, 54, 56, 255, 226, 0, 94, 255, 220, 4, 0, 0, 0],
            &[0, 247, 255, 25, 25, 255, 248, 4, 220, 255, 94, 0, 0, 0, 0],
            &[0, 218, 255, 74, 75, 255, 221, 100, 255, 216, 3, 0, 0, 0, 0],
            &[0, 123, 255, 250, 248, 255, 135, 225, 255, 89, 0, 0, 0, 0, 0],
            &[0, 5, 133, 221, 221, 137, 112, 255, 212, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 228, 255, 84, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 112, 255, 208, 34, 176, 225, 193, 61, 0, 0],
            &[0, 0, 0, 0, 10, 232, 255, 83, 216, 255, 247, 255, 247, 26, 0],
            &[
                0, 0, 0, 0, 117, 255, 204, 63, 255, 221, 12, 183, 255, 113, 0,
            ],
            &[0, 0, 0, 12, 235, 255, 74, 94, 255, 178, 0, 129, 255, 144, 0],
            &[0, 0, 0, 123, 255, 199, 0, 75, 255, 202, 0, 156, 255, 125, 0],
            &[
                0, 0, 15, 238, 255, 69, 0, 18, 240, 255, 192, 250, 255, 55, 0,
            ],
            &[0, 0, 129, 255, 194, 0, 0, 0, 78, 239, 255, 251, 121, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 36, 9, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '&' / 0x26
        '&' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 33, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 185, 255, 255, 244, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 179, 255, 255, 244, 255, 255, 76, 0, 0, 0, 0, 0],
            &[0, 0, 11, 254, 255, 145, 6, 226, 255, 155, 0, 0, 0, 0, 0],
            &[0, 0, 37, 255, 255, 105, 0, 212, 255, 171, 0, 0, 0, 0, 0],
            &[0, 0, 12, 251, 255, 160, 71, 255, 255, 106, 0, 0, 0, 0, 0],
            &[0, 0, 0, 172, 255, 255, 255, 255, 215, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 40, 255, 255, 255, 240, 40, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 164, 255, 255, 255, 189, 0, 22, 234, 234, 139, 0, 0],
            &[
                0, 0, 125, 255, 255, 255, 255, 255, 91, 84, 255, 255, 98, 0, 0,
            ],
            &[
                0, 15, 246, 255, 220, 40, 233, 255, 244, 214, 255, 255, 35, 0, 0,
            ],
            &[
                0, 74, 255, 255, 113, 0, 73, 254, 255, 255, 255, 195, 0, 0, 0,
            ],
            &[0, 86, 255, 255, 105, 0, 0, 157, 255, 255, 255, 69, 0, 0, 0],
            &[
                0, 49, 255, 255, 227, 86, 89, 215, 255, 255, 255, 99, 0, 0, 0,
            ],
            &[
                0, 0, 199, 255, 255, 255, 255, 255, 255, 255, 255, 246, 46, 0, 0,
            ],
            &[
                0, 0, 25, 185, 254, 255, 255, 223, 108, 103, 255, 255, 218, 13, 0,
            ],
            &[0, 0, 0, 0, 21, 47, 30, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ''' / 0x27
        '\'' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 254, 254, 181, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 255, 255, 163, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 251, 255, 144, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 233, 255, 125, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 214, 255, 106, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 85, 108, 39, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '(' / 0x28
        '(' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 93, 254, 250, 43, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 20, 235, 255, 145, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 134, 255, 250, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 234, 255, 172, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 80, 255, 255, 82, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 149, 255, 255, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 199, 255, 224, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 232, 255, 194, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 251, 255, 176, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 255, 255, 168, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 246, 255, 185, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 221, 255, 205, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 180, 255, 247, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 122, 255, 255, 56, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 42, 255, 255, 130, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 192, 255, 227, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 255, 255, 83, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 173, 255, 219, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 23, 118, 118, 35, 0, 0, 0, 0],
        ],
        // letter: ')' / 0x29
        ')' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 139, 254, 230, 17, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 235, 255, 159, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 124, 255, 254, 40, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 27, 252, 255, 146, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 188, 255, 238, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 127, 255, 255, 53, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 70, 255, 255, 102, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 40, 255, 255, 133, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 255, 255, 150, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 255, 255, 152, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 32, 255, 255, 141, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 52, 255, 255, 114, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 99, 255, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 155, 255, 252, 16, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 225, 255, 187, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 255, 82, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 180, 255, 220, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 71, 255, 255, 68, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 118, 94, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '*' / 0x2a
        '*' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 193, 218, 109, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 207, 255, 108, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 37, 0, 0, 187, 255, 86, 0, 11, 36, 0, 0, 0],
            &[
                0, 0, 25, 255, 229, 162, 210, 255, 162, 189, 249, 186, 0, 0, 0,
            ],
            &[
                0, 0, 79, 255, 255, 255, 255, 255, 255, 255, 255, 236, 0, 0, 0,
            ],
            &[0, 0, 0, 14, 48, 163, 255, 255, 252, 91, 40, 10, 0, 0, 0],
            &[0, 0, 0, 0, 44, 245, 252, 180, 255, 188, 3, 0, 0, 0, 0],
            &[0, 0, 0, 13, 216, 255, 157, 23, 244, 255, 135, 0, 0, 0, 0],
            &[0, 0, 0, 10, 133, 241, 30, 0, 130, 220, 72, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 0, 0, 10, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '+' / 0x2b
        '+' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 87, 110, 50, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 202, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 202, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 202, 255, 116, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 71, 134, 134, 134, 230, 255, 189, 134, 134, 134, 17, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 94, 179, 179, 179, 239, 255, 214, 179, 179, 179, 23, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 202, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 202, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 202, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 138, 63, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ',' / 0x2c
        ',' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 188, 219, 219, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 252, 255, 236, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 255, 255, 166, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 91, 255, 255, 88, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 139, 255, 246, 12, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 194, 255, 161, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 70, 26, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '-' / 0x2d
        '-' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 202, 237, 237, 237, 237, 237, 107, 0, 0, 0, 0],
            &[0, 0, 0, 0, 217, 255, 255, 255, 255, 255, 115, 0, 0, 0, 0],
            &[0, 0, 0, 0, 151, 177, 177, 177, 177, 177, 80, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '.' / 0x2e
        '.' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 82, 113, 37, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 103, 255, 255, 237, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 145, 255, 255, 255, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 246, 255, 195, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 41, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '/' / 0x2f
        '/' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 2, 19, 19, 9, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 88, 255, 255, 71, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 185, 255, 227, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 29, 253, 255, 133, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 124, 255, 255, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 221, 255, 195, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 63, 255, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 161, 255, 244, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 245, 255, 159, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 100, 255, 255, 62, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 197, 255, 220, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 255, 255, 124, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 136, 255, 253, 29, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 230, 255, 185, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 75, 255, 255, 88, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 173, 255, 239, 8, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 21, 16, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '0' / 0x30
        '0' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 27, 41, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 103, 229, 255, 255, 254, 172, 32, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 255, 255, 255, 255, 255, 223, 24, 0, 0, 0],
            &[
                0, 0, 22, 246, 255, 253, 144, 101, 227, 255, 255, 150, 0, 0, 0,
            ],
            &[0, 0, 114, 255, 255, 155, 0, 0, 183, 255, 255, 247, 9, 0, 0],
            &[0, 0, 172, 255, 255, 68, 0, 57, 254, 255, 255, 255, 61, 0, 0],
            &[
                0, 0, 213, 255, 255, 27, 3, 207, 255, 255, 255, 255, 106, 0, 0,
            ],
            &[
                0, 0, 228, 255, 255, 6, 109, 255, 243, 154, 255, 255, 123, 0, 0,
            ],
            &[
                0, 0, 239, 255, 255, 23, 240, 255, 115, 118, 255, 255, 137, 0, 0,
            ],
            &[
                0, 0, 226, 255, 255, 172, 255, 210, 4, 127, 255, 255, 131, 0, 0,
            ],
            &[
                0, 0, 209, 255, 255, 255, 254, 59, 0, 141, 255, 255, 112, 0, 0,
            ],
            &[0, 0, 164, 255, 255, 255, 156, 0, 0, 183, 255, 255, 75, 0, 0],
            &[0, 0, 104, 255, 255, 255, 19, 0, 17, 240, 255, 252, 17, 0, 0],
            &[
                0, 0, 14, 239, 255, 255, 127, 85, 191, 255, 255, 174, 0, 0, 0,
            ],
            &[0, 0, 0, 96, 255, 255, 255, 255, 255, 255, 240, 36, 0, 0, 0],
            &[0, 0, 0, 0, 88, 221, 255, 255, 255, 193, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 24, 45, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '1' / 0x31
        '1' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 206, 254, 254, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 55, 191, 255, 255, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 87, 255, 255, 255, 255, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 10, 229, 255, 217, 209, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 87, 110, 5, 177, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 184, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 255, 255, 80, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 62, 89, 255, 255, 255, 185, 77, 49, 4, 0, 0],
            &[
                0, 0, 0, 160, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 0, 160, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '2' / 0x32
        '2' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 38, 46, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 32, 151, 245, 255, 255, 255, 213, 85, 0, 0, 0, 0],
            &[
                0, 0, 82, 241, 255, 255, 255, 255, 255, 255, 255, 102, 0, 0, 0,
            ],
            &[
                0, 0, 95, 255, 255, 219, 136, 127, 221, 255, 255, 238, 9, 0, 0,
            ],
            &[0, 0, 0, 132, 127, 4, 0, 0, 31, 255, 255, 255, 50, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 244, 255, 255, 49, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 44, 255, 255, 244, 11, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 196, 255, 255, 149, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 146, 255, 255, 224, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 134, 255, 255, 235, 43, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 130, 255, 255, 230, 42, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 129, 255, 255, 221, 36, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 129, 255, 255, 202, 22, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 124, 255, 255, 255, 170, 154, 154, 154, 154, 154, 101, 0, 0,
            ],
            &[
                0, 0, 205, 255, 255, 255, 255, 255, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[
                0, 0, 205, 255, 255, 255, 255, 255, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '3' / 0x33
        '3' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 37, 45, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 81, 186, 248, 255, 255, 254, 203, 67, 0, 0, 0, 0],
            &[
                0, 0, 82, 255, 255, 255, 255, 255, 255, 255, 251, 60, 0, 0, 0,
            ],
            &[
                0, 0, 1, 197, 254, 185, 127, 118, 219, 255, 255, 169, 0, 0, 0,
            ],
            &[0, 0, 0, 35, 39, 0, 0, 0, 57, 255, 255, 195, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 60, 255, 255, 157, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 50, 202, 255, 249, 47, 0, 0, 0],
            &[0, 0, 0, 0, 76, 237, 253, 255, 255, 219, 68, 0, 0, 0, 0],
            &[0, 0, 0, 0, 83, 255, 255, 255, 252, 178, 86, 0, 0, 0, 0],
            &[0, 0, 0, 0, 42, 131, 148, 177, 251, 255, 255, 141, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 58, 255, 255, 252, 19, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 237, 255, 255, 53, 0, 0],
            &[0, 0, 27, 5, 0, 0, 0, 0, 38, 254, 255, 255, 31, 0, 0],
            &[
                0, 0, 130, 237, 161, 122, 97, 126, 229, 255, 255, 215, 1, 0, 0,
            ],
            &[
                0, 0, 130, 255, 255, 255, 255, 255, 255, 255, 247, 65, 0, 0, 0,
            ],
            &[0, 0, 60, 174, 242, 255, 255, 255, 248, 180, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 22, 46, 37, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '4' / 0x34
        '4' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 82, 254, 254, 210, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 229, 255, 255, 211, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 151, 255, 255, 255, 211, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 254, 248, 250, 255, 211, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 214, 255, 144, 250, 255, 211, 0, 0, 0, 0],
            &[0, 0, 0, 0, 127, 255, 234, 22, 255, 255, 211, 0, 0, 0, 0],
            &[0, 0, 0, 41, 249, 255, 98, 14, 255, 255, 211, 0, 0, 0, 0],
            &[0, 0, 1, 195, 255, 200, 1, 17, 255, 255, 211, 0, 0, 0, 0],
            &[0, 0, 103, 255, 253, 51, 0, 17, 255, 255, 211, 0, 0, 0, 0],
            &[
                0, 17, 240, 255, 229, 144, 144, 152, 255, 255, 236, 144, 109, 0, 0,
            ],
            &[
                0, 40, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 194, 0, 0,
            ],
            &[
                0, 36, 228, 228, 228, 228, 228, 230, 255, 255, 251, 228, 173, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 17, 255, 255, 211, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 17, 255, 255, 211, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 17, 255, 255, 211, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '5' / 0x35
        '5' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 153, 254, 254, 254, 254, 254, 254, 254, 52, 0, 0, 0],
            &[0, 0, 0, 173, 255, 255, 255, 255, 255, 255, 255, 52, 0, 0, 0],
            &[0, 0, 0, 193, 255, 255, 171, 156, 156, 156, 156, 32, 0, 0, 0],
            &[0, 0, 0, 213, 255, 255, 25, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 232, 255, 255, 5, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 251, 255, 249, 115, 131, 107, 32, 0, 0, 0, 0, 0],
            &[0, 0, 16, 255, 255, 255, 255, 255, 255, 253, 126, 0, 0, 0, 0],
            &[
                0, 0, 22, 236, 255, 251, 246, 255, 255, 255, 255, 88, 0, 0, 0,
            ],
            &[0, 0, 0, 13, 44, 1, 1, 53, 220, 255, 255, 199, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 77, 255, 255, 245, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 53, 255, 255, 248, 1, 0, 0],
            &[0, 0, 26, 8, 0, 0, 0, 0, 119, 255, 255, 214, 0, 0, 0],
            &[
                0, 0, 103, 239, 173, 137, 120, 164, 251, 255, 255, 119, 0, 0, 0,
            ],
            &[
                0, 0, 103, 255, 255, 255, 255, 255, 255, 255, 189, 6, 0, 0, 0,
            ],
            &[0, 0, 47, 179, 249, 255, 255, 255, 221, 119, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 31, 46, 25, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '6' / 0x36
        '6' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 19, 37, 31, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 51, 167, 236, 255, 255, 255, 101, 0, 0, 0],
            &[0, 0, 0, 0, 124, 254, 255, 255, 255, 255, 255, 101, 0, 0, 0],
            &[0, 0, 0, 96, 255, 255, 254, 178, 105, 84, 98, 49, 0, 0, 0],
            &[0, 0, 8, 236, 255, 255, 93, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 88, 255, 255, 175, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 148, 255, 255, 85, 63, 156, 183, 154, 56, 0, 0, 0, 0],
            &[
                0, 0, 190, 255, 255, 137, 255, 255, 255, 255, 253, 88, 0, 0, 0,
            ],
            &[
                0, 0, 206, 255, 255, 255, 241, 191, 236, 255, 255, 237, 9, 0, 0,
            ],
            &[0, 0, 210, 255, 255, 217, 20, 0, 24, 240, 255, 255, 72, 0, 0],
            &[0, 0, 195, 255, 255, 116, 0, 0, 0, 175, 255, 255, 105, 0, 0],
            &[0, 0, 156, 255, 255, 89, 0, 0, 0, 172, 255, 255, 101, 0, 0],
            &[0, 0, 91, 255, 255, 166, 0, 0, 7, 229, 255, 255, 68, 0, 0],
            &[
                0, 0, 10, 231, 255, 255, 174, 123, 199, 255, 255, 235, 5, 0, 0,
            ],
            &[0, 0, 0, 77, 252, 255, 255, 255, 255, 255, 255, 89, 0, 0, 0],
            &[0, 0, 0, 0, 69, 213, 255, 255, 255, 217, 82, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 52, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '7' / 0x37
        '7' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 236, 248, 248, 248, 248, 248, 248, 248, 248, 248, 136, 0, 0,
            ],
            &[
                0, 0, 243, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[
                0, 0, 148, 156, 156, 156, 156, 156, 156, 245, 255, 255, 87, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 54, 255, 255, 226, 4, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 171, 255, 255, 119, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 34, 252, 255, 245, 18, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 147, 255, 255, 151, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 245, 255, 254, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 124, 255, 255, 184, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 232, 255, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 100, 255, 255, 215, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 215, 255, 255, 104, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 77, 255, 255, 238, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 193, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 54, 255, 255, 251, 29, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '8' / 0x38
        '8' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 28, 40, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 127, 231, 255, 255, 254, 203, 71, 0, 0, 0, 0],
            &[0, 0, 0, 167, 255, 255, 255, 255, 255, 255, 253, 74, 0, 0, 0],
            &[
                0, 0, 37, 255, 255, 253, 127, 86, 189, 255, 255, 192, 0, 0, 0,
            ],
            &[0, 0, 66, 255, 255, 190, 0, 0, 36, 255, 255, 221, 0, 0, 0],
            &[0, 0, 30, 255, 255, 230, 12, 0, 85, 255, 255, 182, 0, 0, 0],
            &[0, 0, 0, 171, 255, 255, 198, 110, 240, 255, 251, 62, 0, 0, 0],
            &[0, 0, 0, 18, 196, 255, 255, 255, 255, 236, 79, 0, 0, 0, 0],
            &[0, 0, 0, 9, 154, 255, 255, 255, 255, 215, 53, 0, 0, 0, 0],
            &[0, 0, 6, 193, 255, 255, 220, 182, 255, 255, 248, 71, 0, 0, 0],
            &[0, 0, 103, 255, 255, 202, 10, 0, 112, 255, 255, 233, 9, 0, 0],
            &[0, 0, 163, 255, 255, 75, 0, 0, 0, 187, 255, 255, 59, 0, 0],
            &[0, 0, 183, 255, 255, 75, 0, 0, 0, 176, 255, 255, 78, 0, 0],
            &[
                0, 0, 137, 255, 255, 224, 97, 73, 137, 254, 255, 251, 28, 0, 0,
            ],
            &[
                0, 0, 26, 231, 255, 255, 255, 255, 255, 255, 255, 129, 0, 0, 0,
            ],
            &[0, 0, 0, 33, 168, 246, 255, 255, 255, 213, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 37, 47, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '9' / 0x39
        '9' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 40, 47, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 145, 243, 255, 255, 249, 163, 21, 0, 0, 0, 0],
            &[0, 0, 1, 187, 255, 255, 255, 255, 255, 255, 213, 14, 0, 0, 0],
            &[
                0, 0, 85, 255, 255, 253, 154, 131, 225, 255, 255, 141, 0, 0, 0,
            ],
            &[0, 0, 168, 255, 255, 136, 0, 0, 29, 247, 255, 239, 7, 0, 0],
            &[0, 0, 202, 255, 255, 71, 0, 0, 0, 196, 255, 255, 56, 0, 0],
            &[0, 0, 208, 255, 255, 71, 0, 0, 0, 214, 255, 255, 94, 0, 0],
            &[0, 0, 176, 255, 255, 158, 0, 0, 79, 254, 255, 255, 109, 0, 0],
            &[
                0, 0, 95, 255, 255, 255, 209, 202, 253, 255, 255, 255, 104, 0, 0,
            ],
            &[
                0, 0, 2, 186, 255, 255, 255, 255, 230, 176, 255, 255, 86, 0, 0,
            ],
            &[0, 0, 0, 6, 105, 174, 180, 135, 24, 195, 255, 255, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 23, 249, 255, 236, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 6, 178, 255, 255, 143, 0, 0, 0],
            &[0, 0, 0, 87, 76, 77, 115, 213, 255, 255, 233, 22, 0, 0, 0],
            &[0, 0, 0, 214, 255, 255, 255, 255, 255, 235, 54, 0, 0, 0, 0],
            &[0, 0, 0, 214, 255, 255, 255, 227, 138, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 25, 49, 44, 22, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ':' / 0x3a
        ':' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 131, 161, 74, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 121, 255, 255, 249, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 141, 255, 255, 255, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 216, 248, 149, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 82, 113, 37, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 103, 255, 255, 237, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 145, 255, 255, 255, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 246, 255, 195, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 41, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ';' / 0x3b
        ';' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 71, 161, 133, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 247, 255, 255, 130, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 23, 255, 255, 255, 151, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 144, 248, 218, 43, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 188, 219, 219, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 252, 255, 236, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 46, 255, 255, 166, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 91, 255, 255, 88, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 139, 255, 246, 12, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 194, 255, 161, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 70, 26, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '<' / 0x3c
        '<' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 41, 129, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 35, 162, 253, 208, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 154, 251, 255, 255, 176, 0, 0, 0],
            &[0, 0, 0, 0, 24, 146, 248, 255, 255, 201, 75, 0, 0, 0, 0],
            &[0, 0, 12, 137, 245, 255, 255, 197, 71, 0, 0, 0, 0, 0, 0],
            &[0, 0, 55, 255, 255, 255, 127, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 19, 165, 252, 255, 255, 196, 78, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 158, 251, 255, 255, 213, 96, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 32, 151, 248, 255, 255, 193, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 27, 144, 246, 208, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 22, 102, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '=' / 0x3d
        '=' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 90, 170, 170, 170, 170, 170, 170, 170, 170, 170, 22, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 115, 218, 218, 218, 218, 218, 218, 218, 218, 218, 28, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 34, 66, 66, 66, 66, 66, 66, 66, 66, 66, 8, 0, 0],
            &[
                0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 35, 67, 67, 67, 67, 67, 67, 67, 67, 67, 8, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '>' / 0x3e
        '>' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 118, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 55, 255, 231, 109, 7, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 38, 240, 255, 255, 226, 101, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 125, 238, 255, 255, 220, 93, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 122, 236, 255, 255, 214, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 21, 208, 255, 255, 208, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 125, 234, 255, 255, 231, 104, 0, 0, 0],
            &[0, 0, 0, 30, 144, 244, 255, 255, 227, 109, 8, 0, 0, 0, 0],
            &[0, 0, 44, 251, 255, 255, 222, 101, 6, 0, 0, 0, 0, 0, 0],
            &[0, 0, 55, 255, 216, 94, 4, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 35, 87, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '?' / 0x3f
        '?' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 36, 47, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 80, 181, 247, 255, 255, 255, 213, 78, 0, 0, 0, 0],
            &[
                0, 0, 31, 251, 255, 255, 255, 255, 255, 255, 253, 75, 0, 0, 0,
            ],
            &[0, 0, 0, 157, 247, 164, 109, 91, 187, 255, 255, 193, 0, 0, 0],
            &[0, 0, 0, 21, 18, 0, 0, 0, 25, 255, 255, 226, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 72, 255, 255, 195, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 48, 229, 255, 255, 95, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 79, 245, 255, 255, 136, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 42, 249, 255, 246, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 135, 255, 255, 89, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 165, 255, 255, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 5, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 38, 113, 82, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 237, 255, 255, 104, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 255, 255, 255, 148, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 196, 255, 246, 62, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 41, 18, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '@' / 0x40
        '@' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 131, 212, 243, 230, 181, 64, 0, 0, 0, 0],
            &[0, 0, 0, 58, 232, 255, 243, 201, 210, 255, 255, 120, 0, 0, 0],
            &[0, 0, 36, 238, 255, 138, 10, 0, 0, 57, 234, 255, 76, 0, 0],
            &[0, 0, 181, 255, 142, 0, 0, 0, 0, 0, 71, 255, 217, 0, 0],
            &[
                0, 44, 255, 236, 11, 30, 171, 225, 207, 156, 37, 213, 255, 48, 0,
            ],
            &[
                0, 132, 255, 144, 8, 219, 255, 252, 255, 255, 82, 152, 255, 110, 0,
            ],
            &[
                0, 189, 255, 80, 100, 255, 202, 21, 194, 255, 69, 122, 255, 134, 0,
            ],
            &[
                0, 230, 255, 41, 166, 255, 98, 0, 206, 255, 56, 115, 255, 146, 0,
            ],
            &[
                0, 246, 255, 22, 189, 255, 67, 0, 228, 255, 43, 130, 255, 124, 0,
            ],
            &[
                0, 246, 255, 25, 175, 255, 83, 13, 255, 255, 39, 164, 255, 88, 0,
            ],
            &[
                0, 223, 255, 45, 132, 255, 169, 118, 255, 255, 117, 234, 247, 13, 0,
            ],
            &[
                0, 187, 255, 97, 35, 245, 255, 255, 208, 251, 255, 255, 142, 0, 0,
            ],
            &[0, 110, 255, 181, 0, 61, 165, 147, 15, 81, 171, 121, 3, 0, 0],
            &[0, 22, 239, 255, 73, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 99, 255, 246, 108, 10, 0, 1, 30, 108, 70, 0, 0, 0],
            &[
                0, 0, 0, 103, 250, 255, 254, 232, 249, 255, 255, 106, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 27, 130, 184, 213, 200, 170, 93, 11, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'A' / 0x41
        'A' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 14, 14, 14, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 255, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 254, 255, 255, 255, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 255, 223, 255, 237, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 251, 116, 255, 255, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 230, 255, 201, 46, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 255, 138, 2, 239, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 255, 74, 0, 180, 255, 251, 15, 0, 0, 0],
            &[0, 0, 0, 186, 255, 252, 14, 0, 118, 255, 255, 81, 0, 0, 0],
            &[0, 0, 9, 247, 255, 255, 60, 60, 117, 255, 255, 152, 0, 0, 0],
            &[
                0, 0, 71, 255, 255, 255, 255, 255, 255, 255, 255, 222, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 211, 255, 255, 87, 72, 72, 72, 182, 255, 255, 108, 0, 0,
            ],
            &[0, 26, 255, 255, 203, 0, 0, 0, 0, 65, 255, 255, 179, 0, 0],
            &[0, 96, 255, 255, 139, 0, 0, 0, 0, 9, 250, 255, 243, 6, 0],
            &[0, 167, 255, 255, 75, 0, 0, 0, 0, 0, 198, 255, 255, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'B' / 0x42
        'B' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 254, 254, 254, 254, 251, 237, 196, 113, 8, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 190, 2, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 194, 129, 136, 190, 255, 255, 255, 68, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 131, 0, 0, 0, 204, 255, 255, 106, 0, 0],
            &[0, 0, 81, 255, 255, 131, 0, 0, 0, 177, 255, 255, 87, 0, 0],
            &[0, 0, 81, 255, 255, 143, 23, 29, 82, 245, 255, 244, 18, 0, 0],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 228, 74, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 189, 58, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 192, 125, 131, 177, 255, 255, 250, 54, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 131, 0, 0, 0, 161, 255, 255, 158, 0, 0],
            &[0, 0, 81, 255, 255, 131, 0, 0, 0, 114, 255, 255, 188, 0, 0],
            &[0, 0, 81, 255, 255, 131, 0, 0, 0, 163, 255, 255, 168, 0, 0],
            &[
                0, 0, 81, 255, 255, 193, 127, 131, 175, 255, 255, 255, 94, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 185, 3, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 255, 255, 255, 241, 203, 112, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'C' / 0x43
        'C' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 38, 41, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 85, 197, 252, 255, 255, 255, 209, 88, 0, 0],
            &[
                0, 0, 0, 4, 163, 255, 255, 255, 255, 255, 255, 255, 107, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 188, 131, 132, 177, 239, 15, 0, 0,
            ],
            &[0, 0, 26, 251, 255, 254, 91, 0, 0, 0, 0, 9, 0, 0, 0],
            &[0, 0, 121, 255, 255, 176, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 182, 255, 255, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 221, 255, 255, 63, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 237, 255, 255, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 228, 255, 255, 48, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 207, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 255, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 80, 255, 255, 249, 60, 0, 0, 0, 0, 19, 11, 0, 0],
            &[
                0, 0, 2, 205, 255, 255, 250, 173, 127, 137, 181, 250, 44, 0, 0,
            ],
            &[
                0, 0, 0, 42, 228, 255, 255, 255, 255, 255, 255, 255, 44, 0, 0,
            ],
            &[0, 0, 0, 0, 28, 151, 244, 255, 255, 255, 239, 164, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 27, 46, 25, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'D' / 0x44
        'D' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 254, 254, 251, 227, 185, 89, 2, 0, 0, 0, 0],
            &[
                0, 14, 255, 255, 255, 255, 255, 255, 255, 255, 184, 15, 0, 0, 0,
            ],
            &[
                0, 14, 255, 255, 250, 150, 155, 193, 254, 255, 255, 166, 0, 0, 0,
            ],
            &[0, 14, 255, 255, 242, 0, 0, 0, 70, 250, 255, 255, 58, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 154, 255, 255, 145, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 78, 255, 255, 205, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 44, 255, 255, 228, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 33, 255, 255, 242, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 45, 255, 255, 222, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 85, 255, 255, 196, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 168, 255, 255, 126, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 90, 254, 255, 251, 38, 0, 0],
            &[
                0, 14, 255, 255, 250, 149, 156, 202, 255, 255, 255, 136, 0, 0, 0,
            ],
            &[
                0, 14, 255, 255, 255, 255, 255, 255, 255, 255, 153, 5, 0, 0, 0,
            ],
            &[0, 14, 255, 255, 255, 255, 249, 221, 171, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'E' / 0x45
        'E' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 254, 254, 254, 254, 254, 254, 254, 254, 254, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 220, 161, 161, 161, 161, 161, 161, 20, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 167, 23, 23, 23, 23, 23, 19, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 210, 136, 136, 136, 136, 136, 110, 0, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 219, 159, 159, 159, 159, 159, 159, 20, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'F' / 0x46
        'F' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 254, 254, 254, 254, 254, 254, 254, 254, 254, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 220, 161, 161, 161, 161, 161, 161, 20, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 203, 118, 118, 118, 118, 118, 96, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 174, 41, 41, 41, 41, 41, 33, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'G' / 0x47
        'G' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 36, 48, 33, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 164, 245, 255, 255, 255, 243, 165, 46, 0, 0],
            &[
                0, 0, 0, 60, 236, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0,
            ],
            &[
                0, 0, 19, 235, 255, 255, 243, 155, 117, 131, 185, 212, 0, 0, 0,
            ],
            &[0, 0, 145, 255, 255, 220, 29, 0, 0, 0, 0, 8, 0, 0, 0],
            &[0, 1, 230, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 255, 255, 237, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 255, 255, 193, 0, 0, 19, 113, 113, 113, 113, 74, 0, 0],
            &[
                0, 88, 255, 255, 177, 0, 0, 44, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[
                0, 76, 255, 255, 188, 0, 0, 44, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 54, 255, 255, 226, 0, 0, 3, 19, 42, 255, 255, 167, 0, 0],
            &[0, 8, 245, 255, 255, 49, 0, 0, 0, 24, 255, 255, 167, 0, 0],
            &[0, 0, 175, 255, 255, 194, 8, 0, 0, 24, 255, 255, 167, 0, 0],
            &[
                0, 0, 44, 251, 255, 255, 217, 123, 101, 139, 255, 255, 167, 0, 0,
            ],
            &[
                0, 0, 0, 98, 251, 255, 255, 255, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 0, 0, 0, 59, 178, 252, 255, 255, 255, 250, 195, 79, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 33, 46, 26, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'H' / 0x48
        'H' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 251, 0, 0, 0, 0, 99, 254, 254, 166, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[
                0, 14, 255, 255, 253, 34, 34, 34, 34, 120, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 254, 125, 125, 125, 125, 176, 255, 255, 167, 0, 0,
            ],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'I' / 0x49
        'I' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 254, 254, 254, 254, 254, 254, 254, 254, 180, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 3, 45, 74, 130, 255, 255, 215, 90, 62, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 43, 73, 129, 255, 255, 215, 89, 60, 25, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'J' / 0x4a
        'J' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 176, 254, 254, 73, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 73, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 205, 255, 255, 59, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 23, 247, 255, 255, 29, 0, 0, 0],
            &[0, 0, 44, 182, 141, 118, 146, 224, 255, 255, 209, 0, 0, 0, 0],
            &[0, 0, 55, 255, 255, 255, 255, 255, 255, 252, 66, 0, 0, 0, 0],
            &[0, 0, 43, 234, 255, 255, 255, 254, 197, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 46, 40, 15, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'K' / 0x4b
        'K' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 235, 0, 0, 0, 2, 196, 254, 254, 136, 0, 0],
            &[0, 14, 255, 255, 237, 0, 0, 0, 114, 255, 255, 206, 6, 0, 0],
            &[0, 14, 255, 255, 237, 0, 0, 38, 246, 255, 246, 41, 0, 0, 0],
            &[0, 14, 255, 255, 237, 0, 2, 197, 255, 255, 107, 0, 0, 0, 0],
            &[0, 14, 255, 255, 237, 0, 114, 255, 255, 182, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 237, 35, 246, 255, 234, 23, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 237, 177, 255, 255, 92, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 253, 255, 255, 255, 153, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 255, 255, 253, 255, 252, 43, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 255, 135, 125, 255, 255, 181, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 237, 0, 12, 233, 255, 255, 67, 0, 0, 0, 0],
            &[0, 14, 255, 255, 237, 0, 0, 110, 255, 255, 207, 2, 0, 0, 0],
            &[0, 14, 255, 255, 237, 0, 0, 6, 224, 255, 255, 95, 0, 0, 0],
            &[0, 14, 255, 255, 237, 0, 0, 0, 95, 255, 255, 227, 9, 0, 0],
            &[0, 14, 255, 255, 237, 0, 0, 0, 2, 213, 255, 255, 123, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'L' / 0x4c
        'L' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 254, 254, 237, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 250, 176, 176, 176, 176, 176, 176, 22, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'M' / 0x4d
        'M' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 40, 254, 254, 254, 252, 9, 0, 64, 254, 254, 254, 187, 0, 0,
            ],
            &[
                0, 40, 255, 255, 255, 255, 46, 0, 110, 255, 255, 255, 189, 0, 0,
            ],
            &[
                0, 40, 255, 255, 247, 255, 84, 0, 156, 255, 247, 255, 189, 0, 0,
            ],
            &[
                0, 40, 255, 255, 201, 255, 123, 0, 201, 255, 201, 255, 189, 0, 0,
            ],
            &[
                0, 40, 255, 255, 173, 255, 161, 2, 245, 252, 177, 255, 189, 0, 0,
            ],
            &[
                0, 40, 255, 255, 144, 255, 200, 36, 255, 215, 185, 255, 189, 0, 0,
            ],
            &[
                0, 40, 255, 255, 115, 255, 238, 82, 255, 170, 201, 255, 189, 0, 0,
            ],
            &[
                0, 40, 255, 255, 102, 236, 255, 149, 255, 125, 215, 255, 189, 0, 0,
            ],
            &[
                0, 40, 255, 255, 117, 188, 255, 233, 255, 80, 228, 255, 189, 0, 0,
            ],
            &[
                0, 40, 255, 255, 130, 139, 255, 255, 255, 35, 239, 255, 189, 0, 0,
            ],
            &[
                0, 40, 255, 255, 139, 90, 255, 255, 244, 1, 247, 255, 189, 0, 0,
            ],
            &[
                0, 40, 255, 255, 144, 25, 120, 120, 100, 0, 252, 255, 189, 0, 0,
            ],
            &[0, 40, 255, 255, 145, 0, 0, 0, 0, 0, 253, 255, 189, 0, 0],
            &[0, 40, 255, 255, 145, 0, 0, 0, 0, 0, 253, 255, 189, 0, 0],
            &[0, 40, 255, 255, 145, 0, 0, 0, 0, 0, 253, 255, 189, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'N' / 0x4e
        'N' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 254, 254, 62, 0, 0, 13, 254, 254, 166, 0, 0],
            &[
                0, 14, 255, 255, 255, 255, 145, 0, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 250, 255, 228, 1, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 188, 255, 255, 57, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 141, 234, 255, 141, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 152, 154, 255, 224, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 162, 71, 255, 255, 52, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 166, 5, 238, 255, 136, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 161, 255, 219, 10, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 78, 255, 255, 52, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 8, 243, 255, 131, 251, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 168, 255, 214, 239, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 85, 255, 255, 254, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 11, 246, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 14, 255, 255, 167, 0, 0, 0, 175, 255, 255, 255, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'O' / 0x4f
        'O' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 36, 55, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 133, 241, 255, 255, 255, 212, 73, 0, 0, 0, 0],
            &[0, 0, 2, 177, 255, 255, 255, 255, 255, 255, 254, 85, 0, 0, 0],
            &[
                0, 0, 89, 255, 255, 250, 147, 116, 202, 255, 255, 236, 15, 0, 0,
            ],
            &[0, 0, 202, 255, 255, 117, 0, 0, 12, 217, 255, 255, 101, 0, 0],
            &[0, 14, 252, 255, 251, 16, 0, 0, 0, 124, 255, 255, 172, 0, 0],
            &[0, 59, 255, 255, 217, 0, 0, 0, 0, 66, 255, 255, 214, 0, 0],
            &[0, 77, 255, 255, 194, 0, 0, 0, 0, 46, 255, 255, 235, 0, 0],
            &[0, 91, 255, 255, 186, 0, 0, 0, 0, 33, 255, 255, 245, 0, 0],
            &[0, 82, 255, 255, 194, 0, 0, 0, 0, 41, 255, 255, 235, 0, 0],
            &[0, 60, 255, 255, 219, 0, 0, 0, 0, 67, 255, 255, 213, 0, 0],
            &[0, 18, 254, 255, 252, 18, 0, 0, 0, 120, 255, 255, 170, 0, 0],
            &[0, 0, 201, 255, 255, 123, 0, 0, 9, 219, 255, 255, 99, 0, 0],
            &[
                0, 0, 93, 255, 255, 252, 156, 119, 204, 255, 255, 233, 13, 0, 0,
            ],
            &[0, 0, 2, 176, 255, 255, 255, 255, 255, 255, 252, 79, 0, 0, 0],
            &[0, 0, 0, 6, 129, 235, 255, 255, 255, 203, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 34, 46, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'P' / 0x50
        'P' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 254, 254, 254, 254, 253, 240, 199, 110, 4, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 181, 2, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 215, 150, 154, 197, 255, 255, 255, 87, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 152, 255, 255, 163, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 79, 255, 255, 188, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 108, 255, 255, 174, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 3, 56, 226, 255, 255, 123, 0, 0],
            &[
                0, 0, 81, 255, 255, 247, 234, 247, 255, 255, 255, 242, 28, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 238, 72, 0, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 223, 170, 167, 146, 100, 17, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Q' / 0x51
        'Q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 36, 55, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 133, 241, 255, 255, 255, 212, 73, 0, 0, 0, 0],
            &[0, 0, 2, 177, 255, 255, 255, 255, 255, 255, 254, 85, 0, 0, 0],
            &[
                0, 0, 89, 255, 255, 250, 147, 116, 202, 255, 255, 236, 15, 0, 0,
            ],
            &[0, 0, 202, 255, 255, 117, 0, 0, 12, 217, 255, 255, 101, 0, 0],
            &[0, 14, 252, 255, 251, 16, 0, 0, 0, 124, 255, 255, 172, 0, 0],
            &[0, 59, 255, 255, 217, 0, 0, 0, 0, 66, 255, 255, 214, 0, 0],
            &[0, 77, 255, 255, 194, 0, 0, 0, 0, 46, 255, 255, 235, 0, 0],
            &[0, 91, 255, 255, 186, 0, 0, 0, 0, 33, 255, 255, 245, 0, 0],
            &[0, 82, 255, 255, 194, 0, 0, 0, 0, 41, 255, 255, 235, 0, 0],
            &[0, 60, 255, 255, 219, 0, 0, 0, 0, 67, 255, 255, 213, 0, 0],
            &[0, 18, 254, 255, 252, 18, 0, 0, 0, 120, 255, 255, 170, 0, 0],
            &[0, 0, 201, 255, 255, 123, 0, 0, 9, 219, 255, 255, 99, 0, 0],
            &[
                0, 0, 93, 255, 255, 252, 156, 119, 204, 255, 255, 233, 13, 0, 0,
            ],
            &[0, 0, 2, 176, 255, 255, 255, 255, 255, 255, 252, 79, 0, 0, 0],
            &[0, 0, 0, 6, 129, 235, 255, 255, 255, 255, 129, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 34, 126, 255, 255, 201, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 205, 255, 255, 85, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 78, 255, 255, 219, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 170, 199, 199, 71, 0, 0],
        ],
        // letter: 'R' / 0x52
        'R' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 254, 254, 254, 253, 237, 204, 127, 14, 0, 0, 0, 0, 0],
            &[
                0, 81, 255, 255, 255, 255, 255, 255, 255, 225, 29, 0, 0, 0, 0,
            ],
            &[
                0, 81, 255, 255, 226, 153, 179, 248, 255, 255, 177, 0, 0, 0, 0,
            ],
            &[0, 81, 255, 255, 185, 0, 0, 57, 255, 255, 253, 8, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 0, 246, 255, 255, 32, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 37, 255, 255, 253, 13, 0, 0, 0],
            &[0, 81, 255, 255, 201, 62, 99, 218, 255, 255, 188, 0, 0, 0, 0],
            &[
                0, 81, 255, 255, 255, 255, 255, 255, 255, 236, 42, 0, 0, 0, 0,
            ],
            &[0, 81, 255, 255, 255, 255, 255, 255, 216, 26, 0, 0, 0, 0, 0],
            &[0, 81, 255, 255, 210, 91, 237, 255, 247, 33, 0, 0, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 112, 255, 255, 173, 0, 0, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 6, 220, 255, 255, 65, 0, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 85, 255, 255, 210, 3, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 0, 198, 255, 255, 106, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 0, 58, 255, 255, 236, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'S' / 0x53
        'S' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 46, 32, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 222, 255, 255, 255, 254, 212, 129, 9, 0, 0],
            &[
                0, 0, 0, 130, 255, 255, 255, 255, 255, 255, 255, 221, 1, 0, 0,
            ],
            &[
                0, 0, 14, 247, 255, 255, 183, 96, 102, 146, 227, 118, 0, 0, 0,
            ],
            &[0, 0, 49, 255, 255, 245, 5, 0, 0, 0, 1, 6, 0, 0, 0],
            &[0, 0, 40, 255, 255, 250, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 222, 255, 255, 203, 44, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 249, 255, 255, 253, 165, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 217, 255, 255, 255, 247, 131, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 124, 240, 255, 255, 255, 169, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 17, 158, 255, 255, 255, 56, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 213, 255, 255, 104, 0, 0],
            &[0, 0, 47, 15, 0, 0, 0, 0, 0, 210, 255, 255, 95, 0, 0],
            &[
                0, 0, 135, 244, 172, 113, 86, 99, 174, 255, 255, 252, 30, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 122, 0, 0, 0,
            ],
            &[0, 0, 56, 179, 244, 255, 255, 255, 254, 202, 84, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 21, 45, 43, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'T' / 0x54
        'T' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 94, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 246, 0, 0,
            ],
            &[
                0, 94, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 248, 0, 0,
            ],
            &[
                0, 59, 161, 161, 161, 183, 255, 255, 238, 161, 161, 161, 156, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'U' / 0x55
        'U' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 251, 0, 0, 0, 0, 99, 254, 254, 166, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 13, 255, 255, 254, 1, 0, 0, 0, 102, 255, 255, 167, 0, 0],
            &[0, 2, 252, 255, 255, 21, 0, 0, 0, 123, 255, 255, 151, 0, 0],
            &[0, 0, 221, 255, 255, 99, 0, 0, 1, 196, 255, 255, 115, 0, 0],
            &[
                0, 0, 138, 255, 255, 246, 156, 115, 183, 255, 255, 251, 33, 0, 0,
            ],
            &[
                0, 0, 15, 222, 255, 255, 255, 255, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[0, 0, 0, 21, 161, 244, 255, 255, 255, 213, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 35, 45, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'V' / 0x56
        'V' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 163, 254, 254, 102, 0, 0, 0, 0, 0, 213, 254, 254, 62, 0],
            &[0, 90, 255, 255, 168, 0, 0, 0, 0, 23, 255, 255, 239, 4, 0],
            &[0, 18, 252, 255, 233, 1, 0, 0, 0, 85, 255, 255, 168, 0, 0],
            &[0, 0, 196, 255, 255, 44, 0, 0, 0, 149, 255, 255, 93, 0, 0],
            &[0, 0, 121, 255, 255, 110, 0, 0, 0, 212, 255, 253, 21, 0, 0],
            &[0, 0, 46, 255, 255, 176, 0, 0, 21, 254, 255, 200, 0, 0, 0],
            &[0, 0, 0, 226, 255, 239, 2, 0, 83, 255, 255, 125, 0, 0, 0],
            &[0, 0, 0, 152, 255, 255, 51, 0, 146, 255, 255, 50, 0, 0, 0],
            &[0, 0, 0, 77, 255, 255, 117, 0, 210, 255, 230, 1, 0, 0, 0],
            &[0, 0, 0, 10, 247, 255, 182, 14, 253, 255, 156, 0, 0, 0, 0],
            &[0, 0, 0, 0, 183, 255, 238, 63, 255, 255, 81, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 255, 255, 141, 255, 249, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 255, 255, 239, 255, 187, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 214, 255, 255, 255, 112, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 255, 255, 255, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'W' / 0x57
        'W' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 191, 254, 229, 0, 0, 0, 0, 0, 0, 74, 254, 254, 90, 0],
            &[0, 173, 255, 252, 2, 0, 0, 0, 0, 0, 92, 255, 255, 73, 0],
            &[0, 154, 255, 255, 23, 0, 0, 0, 0, 0, 110, 255, 255, 55, 0],
            &[
                0, 135, 255, 255, 47, 19, 119, 119, 96, 0, 128, 255, 255, 37, 0,
            ],
            &[
                0, 116, 255, 255, 67, 71, 255, 255, 234, 0, 147, 255, 255, 19, 0,
            ],
            &[
                0, 97, 255, 255, 85, 110, 255, 255, 255, 16, 165, 255, 254, 2, 0,
            ],
            &[
                0, 78, 255, 255, 101, 150, 255, 241, 255, 55, 183, 255, 239, 0, 0,
            ],
            &[
                0, 59, 255, 255, 111, 189, 255, 172, 255, 93, 197, 255, 221, 0, 0,
            ],
            &[
                0, 40, 255, 255, 117, 228, 254, 108, 255, 132, 204, 255, 203, 0, 0,
            ],
            &[
                0, 21, 255, 255, 129, 254, 224, 64, 255, 174, 204, 255, 185, 0, 0,
            ],
            &[
                0, 3, 254, 255, 163, 255, 185, 26, 255, 216, 197, 255, 167, 0, 0,
            ],
            &[
                0, 0, 239, 255, 194, 255, 147, 0, 243, 251, 193, 255, 149, 0, 0,
            ],
            &[
                0, 0, 220, 255, 236, 255, 109, 0, 206, 255, 234, 255, 131, 0, 0,
            ],
            &[
                0, 0, 201, 255, 255, 255, 72, 0, 169, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 182, 255, 255, 255, 34, 0, 131, 255, 255, 255, 95, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'X' / 0x58
        'X' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 235, 254, 246, 28, 0, 0, 0, 132, 254, 254, 152, 0, 0],
            &[0, 0, 108, 255, 255, 154, 0, 0, 21, 243, 255, 239, 21, 0, 0],
            &[0, 0, 3, 212, 255, 251, 37, 0, 140, 255, 255, 112, 0, 0, 0],
            &[0, 0, 0, 68, 255, 255, 167, 25, 246, 255, 215, 4, 0, 0, 0],
            &[0, 0, 0, 0, 176, 255, 254, 186, 255, 255, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 35, 248, 255, 255, 255, 179, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 136, 255, 255, 249, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 66, 255, 255, 218, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 202, 255, 255, 255, 105, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 97, 255, 255, 250, 255, 236, 17, 0, 0, 0, 0],
            &[0, 0, 0, 14, 232, 255, 224, 97, 255, 255, 145, 0, 0, 0, 0],
            &[0, 0, 0, 139, 255, 255, 96, 2, 212, 255, 251, 42, 0, 0, 0],
            &[0, 0, 38, 249, 255, 214, 3, 0, 82, 255, 255, 186, 0, 0, 0],
            &[0, 0, 181, 255, 255, 81, 0, 0, 1, 205, 255, 255, 79, 0, 0],
            &[0, 74, 255, 255, 201, 0, 0, 0, 0, 74, 255, 255, 220, 7, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Y' / 0x59
        'Y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 87, 254, 254, 213, 0, 0, 0, 0, 56, 254, 254, 231, 8, 0],
            &[0, 3, 219, 255, 255, 67, 0, 0, 0, 163, 255, 255, 120, 0, 0],
            &[0, 0, 100, 255, 255, 175, 0, 0, 21, 248, 255, 240, 14, 0, 0],
            &[0, 0, 6, 228, 255, 252, 31, 0, 121, 255, 255, 134, 0, 0, 0],
            &[0, 0, 0, 112, 255, 255, 135, 3, 225, 255, 246, 22, 0, 0, 0],
            &[0, 0, 0, 11, 235, 255, 235, 88, 255, 255, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 255, 255, 241, 255, 251, 31, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 242, 255, 255, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 137, 255, 255, 254, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 255, 255, 215, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Z' / 0x5a
        'Z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 254, 22, 0, 0,
            ],
            &[
                0, 0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 255, 22, 0, 0,
            ],
            &[
                0, 0, 0, 119, 150, 150, 150, 150, 196, 255, 255, 209, 3, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 3, 211, 255, 255, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 108, 255, 255, 167, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 237, 255, 244, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 149, 255, 255, 122, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 252, 255, 220, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 189, 255, 255, 76, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 255, 255, 181, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 222, 255, 249, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 123, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 244, 255, 255, 172, 159, 159, 159, 159, 159, 20, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '[' / 0x5b
        '[' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 36, 36, 36, 36, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 255, 255, 222, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 255, 255, 222, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 221, 194, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 255, 255, 222, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 102, 102, 102, 102, 89, 0, 0, 0, 0, 0],
        ],
        // letter: '\' / 0x5c
        '\\' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 17, 19, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 174, 255, 235, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 77, 255, 255, 83, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 231, 255, 180, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 138, 255, 252, 26, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 255, 255, 119, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 200, 255, 216, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 103, 255, 255, 58, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 246, 255, 156, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 164, 255, 242, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 67, 255, 255, 95, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 224, 255, 192, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 129, 255, 254, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 33, 254, 255, 131, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 191, 255, 226, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 94, 255, 255, 71, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 3, 21, 21, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ']' / 0x5d
        ']' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 73, 73, 73, 73, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 69, 255, 255, 255, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 60, 223, 223, 250, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 213, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 61, 226, 226, 251, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 69, 255, 255, 255, 255, 197, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 65, 65, 65, 65, 50, 0, 0, 0, 0, 0],
        ],
        // letter: '^' / 0x5e
        '^' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 213, 252, 40, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 83, 255, 255, 169, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 255, 254, 47, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 75, 255, 251, 209, 255, 178, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 199, 255, 163, 62, 255, 255, 55, 0, 0, 0, 0],
            &[0, 0, 0, 67, 255, 254, 43, 0, 186, 255, 187, 0, 0, 0, 0],
            &[0, 0, 0, 191, 255, 177, 0, 0, 55, 255, 255, 64, 0, 0, 0],
            &[0, 0, 59, 255, 255, 56, 0, 0, 0, 179, 255, 196, 0, 0, 0],
            &[0, 0, 183, 255, 191, 0, 0, 0, 0, 49, 254, 255, 73, 0, 0],
            &[0, 6, 105, 105, 44, 0, 0, 0, 0, 0, 87, 105, 69, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '_' / 0x5f
        '_' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 151, 151, 151, 151, 151, 151, 151, 151, 151, 151, 151, 151, 91, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 153, 0,
            ],
            &[
                0, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 142, 0,
            ],
        ],
        // letter: '`' / 0x60
        '`' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 22, 22, 22, 4, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 188, 255, 255, 136, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 185, 255, 251, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 166, 255, 215, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 64, 70, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'a' / 0x61
        'a' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 83, 137, 164, 167, 146, 89, 9, 0, 0, 0, 0],
            &[
                0, 0, 21, 240, 255, 255, 255, 255, 255, 255, 225, 45, 0, 0, 0,
            ],
            &[
                0, 0, 0, 182, 255, 240, 209, 205, 247, 255, 255, 212, 0, 0, 0,
            ],
            &[0, 0, 0, 40, 56, 2, 0, 0, 25, 230, 255, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 24, 30, 184, 255, 255, 58, 0, 0],
            &[
                0, 0, 0, 62, 180, 240, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 75, 252, 255, 255, 229, 192, 180, 229, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 196, 255, 255, 114, 0, 0, 0, 174, 255, 255, 60, 0, 0],
            &[0, 0, 234, 255, 255, 18, 0, 0, 7, 233, 255, 255, 60, 0, 0],
            &[
                0, 0, 210, 255, 255, 160, 67, 85, 199, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 255, 255, 253, 210, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 7, 151, 253, 255, 255, 215, 76, 61, 255, 255, 60, 0, 0],
            &[0, 0, 0, 0, 9, 43, 29, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'b' / 0x62
        'b' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 207, 245, 245, 33, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 35, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 35, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 34, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 28, 48, 138, 166, 137, 41, 0, 0, 0, 0],
            &[
                0, 0, 216, 255, 255, 116, 252, 255, 255, 255, 248, 76, 0, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 255, 252, 229, 255, 255, 255, 235, 12, 0, 0,
            ],
            &[0, 0, 216, 255, 255, 214, 26, 0, 52, 241, 255, 255, 95, 0, 0],
            &[0, 0, 216, 255, 255, 94, 0, 0, 0, 146, 255, 255, 155, 0, 0],
            &[0, 0, 216, 255, 255, 52, 0, 0, 0, 101, 255, 255, 184, 0, 0],
            &[0, 0, 216, 255, 255, 43, 0, 0, 0, 91, 255, 255, 190, 0, 0],
            &[0, 0, 216, 255, 255, 69, 0, 0, 0, 118, 255, 255, 172, 0, 0],
            &[0, 0, 216, 255, 255, 154, 0, 0, 1, 201, 255, 255, 127, 0, 0],
            &[
                0, 0, 216, 255, 255, 255, 148, 94, 175, 255, 255, 255, 47, 0, 0,
            ],
            &[
                0, 0, 216, 255, 247, 237, 255, 255, 255, 255, 255, 167, 0, 0, 0,
            ],
            &[
                0, 0, 216, 255, 165, 40, 193, 255, 255, 251, 160, 11, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 23, 46, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'c' / 0x63
        'c' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 121, 154, 165, 144, 98, 23, 0, 0, 0],
            &[0, 0, 0, 4, 147, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0],
            &[0, 0, 0, 130, 255, 255, 255, 252, 238, 254, 255, 98, 0, 0, 0],
            &[0, 0, 13, 248, 255, 255, 148, 14, 0, 10, 76, 13, 0, 0, 0],
            &[0, 0, 83, 255, 255, 221, 3, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 115, 255, 255, 160, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 129, 255, 255, 148, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 255, 178, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 255, 255, 246, 35, 0, 0, 0, 1, 27, 0, 0, 0],
            &[
                0, 0, 5, 232, 255, 255, 232, 131, 103, 133, 215, 144, 0, 0, 0,
            ],
            &[0, 0, 0, 79, 252, 255, 255, 255, 255, 255, 255, 144, 0, 0, 0],
            &[0, 0, 0, 0, 62, 192, 253, 255, 255, 255, 204, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 39, 43, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'd' / 0x64
        'd' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 131, 245, 245, 109, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 137, 255, 255, 113, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 137, 255, 255, 113, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 136, 255, 255, 113, 0, 0],
            &[0, 0, 0, 0, 67, 143, 162, 125, 24, 124, 255, 255, 113, 0, 0],
            &[
                0, 0, 0, 150, 255, 255, 255, 255, 240, 144, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 83, 255, 255, 255, 252, 226, 252, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 192, 255, 255, 198, 19, 0, 39, 235, 255, 255, 113, 0, 0,
            ],
            &[0, 5, 248, 255, 255, 58, 0, 0, 0, 139, 255, 255, 113, 0, 0],
            &[0, 26, 255, 255, 254, 5, 0, 0, 0, 101, 255, 255, 113, 0, 0],
            &[0, 36, 255, 255, 243, 0, 0, 0, 0, 81, 255, 255, 113, 0, 0],
            &[0, 20, 255, 255, 255, 10, 0, 0, 0, 103, 255, 255, 113, 0, 0],
            &[0, 0, 234, 255, 255, 88, 0, 0, 0, 181, 255, 255, 113, 0, 0],
            &[
                0, 0, 157, 255, 255, 242, 122, 92, 167, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 37, 244, 255, 255, 255, 255, 255, 199, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 64, 222, 255, 255, 235, 101, 25, 250, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 36, 38, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'e' / 0x65
        'e' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 87, 144, 165, 144, 79, 4, 0, 0, 0, 0],
            &[0, 0, 0, 42, 219, 255, 255, 255, 255, 255, 202, 25, 0, 0, 0],
            &[
                0, 0, 12, 224, 255, 255, 227, 180, 230, 255, 255, 189, 0, 0, 0,
            ],
            &[0, 0, 115, 255, 255, 182, 8, 0, 7, 190, 255, 255, 49, 0, 0],
            &[0, 0, 193, 255, 255, 54, 0, 0, 0, 80, 255, 255, 112, 0, 0],
            &[
                0, 0, 226, 255, 255, 239, 237, 237, 237, 241, 255, 255, 135, 0, 0,
            ],
            &[
                0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 208, 255, 255, 58, 38, 38, 38, 38, 38, 38, 20, 0, 0],
            &[0, 0, 146, 255, 255, 137, 0, 0, 0, 0, 0, 0, 1, 0, 0],
            &[
                0, 0, 43, 249, 255, 255, 185, 103, 79, 102, 144, 218, 54, 0, 0,
            ],
            &[
                0, 0, 0, 100, 252, 255, 255, 255, 255, 255, 255, 255, 54, 0, 0,
            ],
            &[0, 0, 0, 0, 56, 179, 249, 255, 255, 255, 245, 174, 23, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 33, 47, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'f' / 0x66
        'f' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 9, 5, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 166, 239, 255, 255, 237, 186, 13, 0, 0],
            &[0, 0, 0, 0, 12, 224, 255, 255, 255, 255, 255, 209, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 255, 255, 180, 138, 160, 103, 0, 0, 0],
            &[0, 0, 0, 0, 126, 255, 255, 173, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 67, 185, 255, 255, 190, 119, 119, 119, 25, 0, 0, 0],
            &[
                0, 131, 254, 255, 255, 255, 255, 255, 255, 255, 255, 54, 0, 0, 0,
            ],
            &[
                0, 146, 255, 255, 255, 255, 255, 255, 255, 255, 255, 54, 0, 0, 0,
            ],
            &[0, 7, 13, 13, 141, 255, 255, 139, 13, 13, 13, 2, 0, 0, 0],
            &[0, 0, 0, 0, 134, 255, 255, 132, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 134, 255, 255, 132, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 134, 255, 255, 132, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 134, 255, 255, 132, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 134, 255, 255, 132, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 134, 255, 255, 132, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 134, 255, 255, 132, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 134, 255, 255, 132, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'g' / 0x67
        'g' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 159, 168, 117, 14, 1, 113, 119, 53, 0, 0],
            &[
                0, 0, 1, 166, 255, 255, 255, 255, 220, 96, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 94, 255, 255, 255, 249, 226, 254, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 198, 255, 255, 185, 11, 0, 54, 245, 255, 255, 113, 0, 0,
            ],
            &[0, 6, 251, 255, 255, 49, 0, 0, 0, 160, 255, 255, 113, 0, 0],
            &[0, 30, 255, 255, 252, 3, 0, 0, 0, 117, 255, 255, 113, 0, 0],
            &[0, 36, 255, 255, 244, 0, 0, 0, 0, 107, 255, 255, 113, 0, 0],
            &[0, 20, 255, 255, 255, 15, 0, 0, 0, 126, 255, 255, 113, 0, 0],
            &[0, 0, 233, 255, 255, 100, 0, 0, 0, 200, 255, 255, 113, 0, 0],
            &[
                0, 0, 153, 255, 255, 246, 127, 94, 175, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 31, 240, 255, 255, 255, 255, 255, 230, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 54, 213, 255, 255, 243, 118, 153, 255, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 33, 44, 9, 0, 173, 255, 255, 100, 0, 0],
            &[0, 0, 30, 28, 0, 0, 0, 0, 19, 234, 255, 255, 65, 0, 0],
            &[
                0, 0, 92, 254, 202, 160, 136, 155, 233, 255, 255, 227, 3, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 17, 31, 15, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'h' / 0x68
        'h' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 207, 245, 245, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 50, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 48, 38, 135, 165, 148, 71, 0, 0, 0, 0],
            &[
                0, 0, 216, 255, 255, 110, 250, 255, 255, 255, 255, 141, 0, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 255, 251, 225, 250, 255, 255, 255, 47, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 228, 35, 0, 29, 232, 255, 255, 116, 0, 0,
            ],
            &[0, 0, 216, 255, 255, 119, 0, 0, 0, 150, 255, 255, 138, 0, 0],
            &[0, 0, 216, 255, 255, 69, 0, 0, 0, 127, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 53, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'i' / 0x69
        'i' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 66, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 227, 255, 252, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 255, 255, 255, 119, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 220, 255, 246, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 49, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 119, 119, 119, 119, 119, 119, 15, 0, 0, 0, 0, 0],
            &[0, 0, 44, 255, 255, 255, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 31, 197, 225, 251, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 235, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 66, 108, 245, 255, 255, 130, 68, 21, 0, 0, 0],
            &[
                0, 0, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'j' / 0x6a
        'j' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 37, 65, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 78, 254, 255, 217, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 139, 255, 255, 255, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 72, 251, 255, 201, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 25, 48, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 119, 119, 119, 119, 119, 119, 1, 0, 0, 0],
            &[0, 0, 0, 0, 74, 255, 255, 255, 255, 255, 255, 2, 0, 0, 0],
            &[0, 0, 0, 0, 53, 200, 228, 252, 255, 255, 255, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 255, 255, 255, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 35, 255, 255, 254, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 107, 255, 255, 234, 0, 0, 0, 0],
            &[0, 0, 36, 193, 151, 134, 170, 249, 255, 255, 174, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 25, 30, 8, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'k' / 0x6b
        'k' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 181, 245, 245, 74, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 0, 85, 119, 119, 96, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 86, 254, 255, 244, 52, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 55, 247, 255, 252, 75, 0, 0, 0, 0],
            &[0, 189, 255, 255, 76, 31, 232, 255, 255, 102, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 77, 207, 255, 255, 132, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 197, 255, 255, 243, 15, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 255, 156, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 255, 144, 221, 255, 255, 72, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 115, 0, 66, 255, 255, 227, 15, 0, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 156, 255, 255, 157, 0, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 16, 230, 255, 255, 73, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 0, 80, 255, 255, 227, 15, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'l' / 0x6c
        'l' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 245, 245, 245, 245, 245, 245, 20, 0, 0, 0, 0, 0],
            &[0, 0, 1, 252, 255, 255, 255, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 61, 114, 251, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 67, 110, 250, 255, 255, 124, 67, 20, 0, 0, 0],
            &[
                0, 0, 180, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'm' / 0x6d
        'm' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 44, 119, 88, 7, 123, 161, 78, 0, 63, 159, 136, 14, 0, 0],
            &[
                0, 94, 255, 233, 145, 255, 255, 251, 78, 248, 255, 255, 144, 0, 0,
            ],
            &[
                0, 94, 255, 255, 251, 237, 255, 255, 245, 234, 255, 255, 209, 0, 0,
            ],
            &[
                0, 94, 255, 255, 106, 25, 255, 255, 205, 0, 186, 255, 228, 0, 0,
            ],
            &[
                0, 94, 255, 255, 68, 5, 255, 255, 168, 0, 167, 255, 245, 0, 0,
            ],
            &[
                0, 94, 255, 255, 65, 5, 255, 255, 164, 0, 167, 255, 248, 0, 0,
            ],
            &[
                0, 94, 255, 255, 65, 5, 255, 255, 164, 0, 167, 255, 248, 0, 0,
            ],
            &[
                0, 94, 255, 255, 65, 5, 255, 255, 164, 0, 167, 255, 248, 0, 0,
            ],
            &[
                0, 94, 255, 255, 65, 5, 255, 255, 164, 0, 167, 255, 248, 0, 0,
            ],
            &[
                0, 94, 255, 255, 65, 5, 255, 255, 164, 0, 167, 255, 248, 0, 0,
            ],
            &[
                0, 94, 255, 255, 65, 5, 255, 255, 164, 0, 167, 255, 248, 0, 0,
            ],
            &[
                0, 94, 255, 255, 65, 5, 255, 255, 164, 0, 167, 255, 248, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'n' / 0x6e
        'n' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 100, 119, 64, 0, 44, 134, 165, 150, 75, 0, 0, 0, 0],
            &[
                0, 0, 216, 255, 203, 102, 254, 255, 255, 255, 255, 146, 0, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 255, 255, 229, 251, 255, 255, 255, 49, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 237, 47, 0, 34, 236, 255, 255, 117, 0, 0,
            ],
            &[0, 0, 216, 255, 255, 135, 0, 0, 0, 152, 255, 255, 138, 0, 0],
            &[0, 0, 216, 255, 255, 76, 0, 0, 0, 128, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 55, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'o' / 0x6f
        'o' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 97, 148, 164, 138, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 59, 231, 255, 255, 255, 255, 255, 187, 18, 0, 0, 0],
            &[
                0, 0, 31, 242, 255, 255, 243, 211, 250, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 158, 255, 255, 190, 10, 0, 43, 241, 255, 255, 62, 0, 0],
            &[0, 1, 239, 255, 255, 54, 0, 0, 0, 147, 255, 255, 142, 0, 0],
            &[0, 20, 255, 255, 253, 4, 0, 0, 0, 97, 255, 255, 175, 0, 0],
            &[0, 35, 255, 255, 242, 0, 0, 0, 0, 88, 255, 255, 188, 0, 0],
            &[0, 15, 255, 255, 254, 14, 0, 0, 0, 114, 255, 255, 167, 0, 0],
            &[0, 0, 215, 255, 255, 98, 0, 0, 0, 193, 255, 255, 111, 0, 0],
            &[
                0, 0, 112, 255, 255, 243, 112, 73, 158, 255, 255, 244, 20, 0, 0,
            ],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 253, 93, 0, 0, 0],
            &[0, 0, 0, 7, 132, 236, 255, 255, 255, 205, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 35, 47, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'p' / 0x70
        'p' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 100, 119, 65, 0, 39, 134, 164, 138, 42, 0, 0, 0, 0],
            &[
                0, 0, 216, 255, 206, 94, 253, 255, 255, 255, 248, 77, 0, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 254, 253, 229, 255, 255, 255, 235, 12, 0, 0,
            ],
            &[0, 0, 216, 255, 255, 218, 30, 0, 51, 243, 255, 255, 94, 0, 0],
            &[0, 0, 216, 255, 255, 93, 0, 0, 0, 159, 255, 255, 154, 0, 0],
            &[0, 0, 216, 255, 255, 47, 0, 0, 0, 115, 255, 255, 184, 0, 0],
            &[0, 0, 216, 255, 255, 42, 0, 0, 0, 106, 255, 255, 188, 0, 0],
            &[0, 0, 216, 255, 255, 69, 0, 0, 0, 128, 255, 255, 169, 0, 0],
            &[0, 0, 216, 255, 255, 155, 0, 0, 0, 199, 255, 255, 124, 0, 0],
            &[
                0, 0, 216, 255, 255, 255, 151, 93, 165, 255, 255, 254, 45, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 237, 255, 255, 255, 255, 255, 161, 0, 0, 0,
            ],
            &[0, 0, 216, 255, 255, 55, 190, 255, 255, 251, 153, 6, 0, 0, 0],
            &[0, 0, 216, 255, 255, 34, 0, 21, 44, 13, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 35, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 35, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 10, 10, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'q' / 0x71
        'q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 81, 153, 163, 118, 21, 0, 111, 119, 53, 0, 0],
            &[
                0, 0, 1, 165, 255, 255, 255, 255, 235, 88, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 95, 255, 255, 255, 250, 232, 255, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 198, 255, 255, 197, 15, 0, 71, 250, 255, 255, 113, 0, 0,
            ],
            &[0, 5, 251, 255, 255, 66, 0, 0, 0, 169, 255, 255, 113, 0, 0],
            &[0, 30, 255, 255, 255, 16, 0, 0, 0, 120, 255, 255, 113, 0, 0],
            &[0, 37, 255, 255, 255, 4, 0, 0, 0, 102, 255, 255, 113, 0, 0],
            &[0, 20, 255, 255, 255, 27, 0, 0, 0, 123, 255, 255, 113, 0, 0],
            &[0, 0, 234, 255, 255, 107, 0, 0, 1, 201, 255, 255, 113, 0, 0],
            &[
                0, 0, 157, 255, 255, 246, 126, 94, 176, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 37, 244, 255, 255, 255, 255, 255, 218, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 65, 223, 255, 255, 239, 120, 129, 255, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 37, 41, 6, 0, 137, 255, 255, 113, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 137, 255, 255, 113, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 137, 255, 255, 113, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 5, 10, 10, 4, 0, 0],
        ],
        // letter: 'r' / 0x72
        'r' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 68, 119, 119, 119, 119, 24, 0, 66, 148, 166, 132, 18, 0,
            ],
            &[
                0, 0, 146, 255, 255, 255, 255, 95, 127, 255, 255, 255, 247, 10, 0,
            ],
            &[
                0, 0, 108, 218, 251, 255, 255, 213, 254, 255, 255, 255, 183, 0, 0,
            ],
            &[0, 0, 0, 0, 42, 255, 255, 255, 251, 135, 71, 81, 59, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 255, 107, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 250, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 230, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 227, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 227, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 60, 133, 255, 255, 241, 100, 46, 3, 0, 0, 0, 0],
            &[
                0, 32, 251, 255, 255, 255, 255, 255, 255, 255, 220, 0, 0, 0, 0,
            ],
            &[
                0, 35, 255, 255, 255, 255, 255, 255, 255, 255, 228, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 's' / 0x73
        's' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 112, 156, 167, 146, 119, 64, 3, 0, 0, 0],
            &[0, 0, 0, 89, 249, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0],
            &[
                0, 0, 12, 242, 255, 255, 243, 218, 242, 255, 255, 97, 0, 0, 0,
            ],
            &[0, 0, 50, 255, 255, 248, 13, 0, 0, 32, 107, 10, 0, 0, 0],
            &[0, 0, 28, 254, 255, 255, 165, 48, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 255, 255, 255, 255, 208, 98, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 214, 255, 255, 255, 255, 212, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 70, 172, 254, 255, 255, 167, 0, 0, 0],
            &[0, 0, 19, 13, 0, 0, 0, 0, 102, 255, 255, 224, 0, 0, 0],
            &[
                0, 0, 81, 247, 177, 120, 93, 102, 194, 255, 255, 211, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 47, 192, 249, 255, 255, 255, 255, 217, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 31, 48, 42, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 't' / 0x74
        't' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 82, 64, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 171, 255, 200, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 255, 200, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 93, 253, 255, 226, 119, 119, 119, 119, 10, 0, 0],
            &[
                0, 0, 211, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 0, 0,
            ],
            &[
                0, 0, 226, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 0, 0,
            ],
            &[0, 0, 12, 13, 77, 255, 255, 203, 13, 13, 13, 13, 1, 0, 0],
            &[0, 0, 0, 0, 67, 255, 255, 200, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 67, 255, 255, 200, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 67, 255, 255, 200, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 67, 255, 255, 202, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 255, 255, 241, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 253, 255, 255, 181, 97, 99, 135, 31, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 255, 255, 255, 255, 49, 0, 0],
            &[0, 0, 0, 0, 0, 14, 157, 246, 255, 255, 255, 232, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 8, 39, 47, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'u' / 0x75
        'u' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 119, 119, 11, 0, 0, 0, 71, 119, 119, 53, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 168, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 31, 0, 0, 0, 194, 255, 255, 113, 0, 0],
            &[0, 0, 234, 255, 255, 75, 0, 0, 20, 246, 255, 255, 113, 0, 0],
            &[
                0, 0, 194, 255, 255, 222, 100, 99, 207, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 198, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 105, 241, 255, 255, 233, 109, 16, 249, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 35, 33, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'v' / 0x76
        'v' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 119, 119, 91, 0, 0, 0, 0, 12, 119, 119, 119, 11, 0],
            &[0, 56, 255, 255, 246, 10, 0, 0, 0, 86, 255, 255, 209, 0, 0],
            &[0, 0, 216, 255, 255, 82, 0, 0, 0, 166, 255, 255, 113, 0, 0],
            &[0, 0, 122, 255, 255, 163, 0, 0, 6, 240, 255, 251, 22, 0, 0],
            &[0, 0, 29, 253, 255, 239, 5, 0, 71, 255, 255, 178, 0, 0, 0],
            &[0, 0, 0, 187, 255, 255, 70, 0, 151, 255, 255, 83, 0, 0, 0],
            &[0, 0, 0, 92, 255, 255, 152, 1, 230, 255, 236, 6, 0, 0, 0],
            &[0, 0, 0, 10, 242, 255, 228, 52, 255, 255, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 158, 255, 255, 153, 255, 255, 52, 0, 0, 0, 0],
            &[0, 0, 0, 0, 63, 255, 255, 241, 255, 212, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 222, 255, 255, 255, 116, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 128, 255, 255, 252, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'w' / 0x77
        'w' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 107, 119, 102, 0, 0, 0, 0, 0, 0, 22, 119, 119, 62, 0],
            &[0, 204, 255, 235, 0, 0, 0, 0, 0, 0, 65, 255, 255, 107, 0],
            &[
                0, 168, 255, 254, 4, 12, 200, 201, 141, 0, 88, 255, 255, 71, 0,
            ],
            &[
                0, 132, 255, 255, 25, 60, 255, 255, 224, 0, 111, 255, 255, 35, 0,
            ],
            &[
                0, 96, 255, 255, 47, 112, 255, 255, 255, 20, 133, 255, 251, 4, 0,
            ],
            &[
                0, 60, 255, 255, 70, 163, 255, 231, 255, 70, 156, 255, 219, 0, 0,
            ],
            &[
                0, 24, 255, 255, 93, 214, 255, 150, 255, 121, 178, 255, 183, 0, 0,
            ],
            &[
                0, 0, 243, 255, 125, 252, 237, 84, 255, 170, 198, 255, 147, 0, 0,
            ],
            &[
                0, 0, 208, 255, 184, 255, 191, 38, 255, 212, 221, 255, 111, 0, 0,
            ],
            &[
                0, 0, 172, 255, 243, 255, 144, 2, 246, 249, 248, 255, 75, 0, 0,
            ],
            &[
                0, 0, 136, 255, 255, 255, 96, 0, 202, 255, 255, 255, 39, 0, 0,
            ],
            &[0, 0, 99, 255, 255, 255, 49, 0, 157, 255, 255, 252, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'x' / 0x78
        'x' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 27, 119, 119, 119, 47, 0, 0, 0, 73, 119, 119, 115, 5, 0],
            &[0, 0, 185, 255, 255, 207, 2, 0, 27, 243, 255, 255, 128, 0, 0],
            &[0, 0, 27, 238, 255, 255, 100, 0, 169, 255, 255, 200, 4, 0, 0],
            &[0, 0, 0, 90, 255, 255, 233, 83, 255, 255, 244, 36, 0, 0, 0],
            &[0, 0, 0, 0, 170, 255, 255, 252, 255, 255, 101, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 231, 255, 255, 255, 177, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 197, 255, 255, 255, 122, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 116, 255, 255, 255, 255, 248, 42, 0, 0, 0, 0],
            &[0, 0, 0, 45, 248, 255, 253, 166, 255, 255, 203, 4, 0, 0, 0],
            &[0, 0, 6, 209, 255, 255, 150, 8, 223, 255, 255, 125, 0, 0, 0],
            &[0, 0, 137, 255, 255, 234, 17, 0, 85, 255, 255, 250, 48, 0, 0],
            &[0, 61, 253, 255, 255, 96, 0, 0, 0, 194, 255, 255, 209, 6, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'y' / 0x79
        'y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 58, 119, 119, 97, 0, 0, 0, 0, 19, 119, 119, 119, 11, 0],
            &[0, 50, 255, 255, 251, 19, 0, 0, 0, 97, 255, 255, 212, 0, 0],
            &[0, 0, 204, 255, 255, 101, 0, 0, 0, 173, 255, 255, 119, 0, 0],
            &[0, 0, 103, 255, 255, 186, 0, 0, 6, 242, 255, 253, 28, 0, 0],
            &[0, 0, 13, 244, 255, 251, 20, 0, 69, 255, 255, 188, 0, 0, 0],
            &[0, 0, 0, 156, 255, 255, 102, 0, 145, 255, 255, 95, 0, 0, 0],
            &[0, 0, 0, 54, 255, 255, 187, 0, 220, 255, 245, 13, 0, 0, 0],
            &[0, 0, 0, 0, 208, 255, 251, 56, 255, 255, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 255, 255, 178, 255, 255, 72, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 246, 255, 253, 255, 231, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 160, 255, 255, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 255, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 255, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 198, 255, 255, 105, 0, 0, 0, 0, 0, 0],
            &[0, 0, 110, 141, 210, 255, 255, 224, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 32, 17, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'z' / 0x7a
        'z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 38, 119, 119, 119, 119, 119, 119, 119, 119, 119, 5, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 11, 0, 0,
            ],
            &[
                0, 0, 75, 237, 237, 237, 237, 237, 252, 255, 255, 241, 7, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 69, 252, 255, 253, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 35, 237, 255, 255, 116, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 211, 255, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 173, 255, 255, 199, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 255, 255, 227, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 254, 255, 246, 51, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 41, 242, 255, 255, 178, 100, 100, 100, 100, 100, 23, 0, 0,
            ],
            &[
                0, 0, 160, 255, 255, 255, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 162, 255, 255, 255, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '{' / 0x7b
        '{' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 126, 219, 255, 185, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 109, 255, 255, 255, 185, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 177, 255, 255, 129, 40, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 191, 255, 240, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 191, 255, 229, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 191, 255, 229, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 208, 255, 227, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 106, 254, 255, 192, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 255, 255, 255, 220, 52, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 255, 255, 243, 139, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 98, 185, 255, 255, 163, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 226, 255, 222, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 192, 255, 229, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 191, 255, 229, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 191, 255, 231, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 184, 255, 253, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 142, 255, 255, 253, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 208, 255, 255, 185, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 47, 86, 71, 0, 0, 0, 0],
        ],
        // letter: '|' / 0x7c
        '|' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 215, 245, 111, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 10, 4, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '}' / 0x7d
        '}' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 11, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 255, 247, 193, 62, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 255, 255, 255, 245, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 67, 196, 255, 255, 75, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 255, 255, 89, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 255, 255, 89, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 255, 255, 89, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 74, 255, 255, 105, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 255, 255, 216, 52, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 131, 246, 255, 255, 184, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 57, 185, 255, 255, 185, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 20, 244, 255, 246, 135, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 68, 255, 255, 128, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 255, 255, 90, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 255, 255, 89, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 78, 255, 255, 89, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 150, 255, 255, 82, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 238, 255, 255, 255, 40, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 255, 255, 252, 133, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 94, 74, 23, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '~' / 0x7e
        '~' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 143, 154, 72, 0, 0, 0, 70, 168, 22, 0, 0],
            &[0, 0, 11, 221, 253, 244, 255, 140, 2, 0, 137, 255, 10, 0, 0],
            &[0, 0, 90, 255, 74, 10, 173, 255, 176, 79, 231, 196, 0, 0, 0],
            &[0, 0, 137, 250, 1, 0, 1, 130, 250, 255, 244, 57, 0, 0, 0],
            &[0, 0, 47, 73, 0, 0, 0, 0, 27, 65, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ' ' / 0xa0
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¡' / 0xa1
        '¡' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 70, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 70, 252, 255, 220, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 140, 255, 255, 255, 43, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 255, 255, 232, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 51, 85, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 33, 33, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 255, 255, 166, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 255, 255, 177, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 255, 255, 189, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 255, 255, 200, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 255, 255, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 74, 255, 255, 222, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 85, 255, 255, 234, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 97, 255, 255, 245, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 108, 255, 255, 254, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 105, 226, 226, 226, 10, 0, 0, 0, 0, 0],
        ],
        // letter: '¢' / 0xa2
        '¢' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 37, 50, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 184, 245, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 85, 218, 251, 131, 95, 23, 0, 0, 0],
            &[0, 0, 0, 0, 108, 241, 255, 255, 255, 255, 255, 241, 6, 0, 0],
            &[
                0, 0, 0, 102, 255, 255, 255, 243, 218, 241, 255, 166, 0, 0, 0,
            ],
            &[0, 0, 7, 238, 255, 255, 116, 6, 0, 0, 46, 41, 0, 0, 0],
            &[0, 0, 71, 255, 255, 182, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 255, 114, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 128, 255, 255, 99, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 106, 255, 255, 126, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 255, 255, 215, 6, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 215, 255, 255, 193, 78, 47, 71, 120, 174, 0, 0, 0],
            &[0, 0, 0, 59, 245, 255, 255, 255, 255, 255, 255, 213, 0, 0, 0],
            &[0, 0, 0, 0, 49, 184, 252, 255, 255, 255, 244, 140, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 196, 248, 42, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 184, 245, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 38, 51, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '£' / 0xa3
        '£' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 41, 27, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 183, 254, 255, 255, 242, 157, 11, 0, 0],
            &[0, 0, 0, 0, 25, 237, 255, 255, 255, 255, 255, 217, 0, 0, 0],
            &[0, 0, 0, 0, 124, 255, 255, 230, 119, 100, 159, 113, 0, 0, 0],
            &[0, 0, 0, 0, 168, 255, 255, 90, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 179, 255, 255, 60, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 179, 255, 255, 60, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 61, 193, 237, 255, 255, 208, 193, 193, 127, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 168, 0, 0, 0, 0],
            &[0, 0, 34, 109, 212, 255, 255, 144, 109, 109, 72, 0, 0, 0, 0],
            &[0, 0, 0, 0, 180, 255, 255, 60, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 203, 255, 255, 34, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 41, 249, 255, 189, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 42, 234, 255, 251, 160, 149, 149, 149, 149, 149, 50, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0,
            ],
            &[
                0, 0, 76, 255, 255, 255, 255, 255, 255, 255, 255, 255, 87, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¤' / 0xa4
        '¤' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 133, 34, 0, 1, 9, 0, 0, 112, 60, 0, 0, 0],
            &[
                0, 0, 120, 255, 226, 163, 246, 255, 217, 158, 255, 244, 25, 0, 0,
            ],
            &[
                0, 0, 15, 200, 255, 255, 255, 255, 255, 255, 254, 102, 0, 0, 0,
            ],
            &[0, 0, 0, 143, 255, 227, 69, 35, 122, 254, 254, 40, 0, 0, 0],
            &[0, 0, 0, 225, 255, 88, 0, 0, 0, 189, 255, 117, 0, 0, 0],
            &[0, 0, 0, 238, 255, 68, 0, 0, 0, 170, 255, 131, 0, 0, 0],
            &[0, 0, 0, 185, 255, 174, 4, 0, 42, 239, 255, 79, 0, 0, 0],
            &[0, 0, 0, 142, 255, 255, 229, 200, 248, 255, 252, 34, 0, 0, 0],
            &[
                0, 0, 103, 255, 253, 244, 255, 255, 255, 242, 255, 223, 16, 0, 0,
            ],
            &[0, 0, 26, 216, 97, 15, 83, 98, 52, 15, 200, 134, 0, 0, 0],
            &[0, 0, 0, 6, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¥' / 0xa5
        '¥' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 57, 254, 254, 211, 0, 0, 0, 0, 59, 254, 254, 207, 1, 0],
            &[0, 0, 186, 255, 255, 71, 0, 0, 0, 173, 255, 255, 80, 0, 0],
            &[0, 0, 58, 255, 255, 185, 0, 0, 34, 253, 255, 205, 0, 0, 0],
            &[0, 0, 0, 186, 255, 255, 44, 0, 145, 255, 255, 76, 0, 0, 0],
            &[0, 0, 0, 58, 255, 255, 157, 16, 243, 255, 201, 0, 0, 0, 0],
            &[0, 0, 0, 0, 186, 255, 248, 141, 255, 255, 72, 0, 0, 0, 0],
            &[0, 0, 0, 0, 58, 255, 255, 255, 255, 197, 0, 0, 0, 0, 0],
            &[
                0, 0, 15, 100, 100, 228, 255, 255, 255, 157, 100, 75, 0, 0, 0,
            ],
            &[
                0, 0, 38, 255, 255, 255, 255, 255, 255, 255, 255, 192, 0, 0, 0,
            ],
            &[
                0, 0, 16, 111, 111, 129, 255, 255, 216, 111, 111, 83, 0, 0, 0,
            ],
            &[0, 0, 5, 35, 35, 63, 255, 255, 195, 35, 35, 26, 0, 0, 0],
            &[
                0, 0, 38, 255, 255, 255, 255, 255, 255, 255, 255, 192, 0, 0, 0,
            ],
            &[
                0, 0, 26, 171, 171, 182, 255, 255, 232, 171, 171, 128, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 32, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¦' / 0xa6
        '¦' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 205, 234, 106, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 89, 101, 46, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 113, 129, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 224, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 212, 96, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '§' / 0xa7
        '§' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 60, 182, 241, 255, 245, 209, 131, 20, 0, 0, 0],
            &[0, 0, 0, 60, 249, 255, 255, 234, 252, 255, 237, 11, 0, 0, 0],
            &[0, 0, 0, 153, 255, 255, 58, 0, 7, 79, 96, 0, 0, 0, 0],
            &[0, 0, 0, 146, 255, 255, 103, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 29, 222, 255, 255, 204, 84, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 150, 255, 255, 255, 255, 211, 50, 0, 0, 0, 0],
            &[0, 0, 0, 109, 255, 246, 87, 150, 250, 255, 238, 23, 0, 0, 0],
            &[0, 0, 0, 195, 255, 194, 0, 0, 91, 255, 255, 101, 0, 0, 0],
            &[0, 0, 0, 184, 255, 248, 75, 0, 37, 255, 255, 99, 0, 0, 0],
            &[0, 0, 0, 65, 249, 255, 255, 179, 179, 255, 236, 21, 0, 0, 0],
            &[0, 0, 0, 0, 51, 191, 255, 255, 255, 255, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 66, 191, 255, 255, 214, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 152, 255, 255, 62, 0, 0, 0],
            &[0, 0, 0, 120, 48, 0, 0, 0, 147, 255, 255, 57, 0, 0, 0],
            &[0, 0, 0, 214, 255, 229, 194, 209, 255, 255, 205, 5, 0, 0, 0],
            &[0, 0, 0, 117, 225, 255, 255, 255, 249, 156, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 19, 44, 32, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¨' / 0xa8
        '¨' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 50, 194, 174, 20, 0, 65, 199, 165, 11, 0, 0, 0],
            &[0, 0, 0, 144, 255, 255, 95, 0, 171, 255, 255, 73, 0, 0, 0],
            &[0, 0, 0, 87, 254, 245, 43, 0, 110, 255, 238, 28, 0, 0, 0],
            &[0, 0, 0, 0, 28, 17, 0, 0, 0, 31, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '©' / 0xa9
        '©' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 100, 202, 247, 255, 236, 166, 48, 0, 0, 0, 0],
            &[0, 0, 7, 182, 254, 176, 102, 85, 122, 217, 248, 90, 0, 0, 0],
            &[
                0, 0, 143, 254, 88, 64, 192, 231, 205, 75, 172, 247, 43, 0, 0,
            ],
            &[
                0, 18, 248, 156, 42, 251, 221, 126, 164, 45, 12, 237, 159, 0, 0,
            ],
            &[0, 82, 255, 62, 146, 255, 45, 0, 0, 0, 0, 158, 232, 0, 0],
            &[0, 108, 255, 27, 180, 247, 0, 0, 0, 0, 0, 129, 254, 5, 0],
            &[0, 91, 255, 44, 160, 254, 13, 0, 0, 0, 0, 148, 246, 0, 0],
            &[0, 38, 255, 115, 84, 255, 162, 58, 80, 37, 1, 219, 199, 0, 0],
            &[
                0, 0, 191, 235, 28, 138, 253, 255, 254, 67, 116, 255, 95, 0, 0,
            ],
            &[0, 0, 37, 237, 227, 91, 31, 52, 65, 150, 255, 172, 0, 0, 0],
            &[0, 0, 0, 36, 190, 255, 255, 255, 255, 239, 126, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 35, 75, 89, 62, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ª' / 0xaa
        'ª' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 32, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 197, 254, 255, 254, 162, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 190, 129, 119, 225, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 43, 78, 170, 255, 140, 0, 0, 0, 0],
            &[0, 0, 0, 0, 87, 239, 255, 233, 233, 255, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 232, 255, 68, 0, 124, 255, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 234, 255, 113, 94, 235, 255, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 252, 255, 249, 102, 238, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 56, 18, 0, 25, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '«' / 0xab
        '«' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 52, 32, 0, 0, 3, 79, 2, 0, 0, 0],
            &[0, 0, 0, 0, 25, 233, 249, 119, 0, 132, 255, 210, 28, 0, 0],
            &[0, 0, 0, 3, 193, 255, 253, 61, 72, 254, 255, 185, 1, 0, 0],
            &[0, 0, 0, 135, 255, 255, 137, 28, 235, 255, 238, 27, 0, 0, 0],
            &[0, 0, 74, 254, 255, 209, 11, 197, 255, 255, 90, 0, 0, 0, 0],
            &[0, 0, 152, 255, 255, 139, 34, 248, 255, 246, 24, 0, 0, 0, 0],
            &[0, 0, 13, 217, 255, 251, 55, 104, 255, 255, 180, 0, 0, 0, 0],
            &[0, 0, 0, 45, 246, 255, 218, 11, 165, 255, 255, 102, 0, 0, 0],
            &[0, 0, 0, 0, 98, 255, 255, 145, 12, 215, 255, 244, 28, 0, 0],
            &[0, 0, 0, 0, 0, 155, 149, 23, 0, 44, 204, 79, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¬' / 0xac
        '¬' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 71, 134, 134, 134, 134, 134, 134, 134, 134, 123, 0, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 235, 0, 0, 0,
            ],
            &[
                0, 0, 94, 179, 179, 179, 179, 179, 179, 204, 255, 235, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 83, 255, 235, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 83, 255, 235, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 83, 255, 235, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 40, 122, 113, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '®' / 0xae
        '®' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 99, 201, 247, 255, 235, 164, 46, 0, 0, 0, 0],
            &[0, 0, 6, 179, 254, 177, 102, 85, 122, 217, 247, 86, 0, 0, 0],
            &[
                0, 0, 139, 254, 92, 181, 187, 175, 123, 16, 172, 246, 40, 0, 0,
            ],
            &[
                0, 16, 246, 161, 0, 247, 191, 149, 255, 131, 12, 238, 155, 0, 0,
            ],
            &[0, 77, 255, 67, 0, 247, 139, 12, 251, 159, 0, 162, 228, 0, 0],
            &[
                0, 103, 255, 33, 0, 247, 224, 223, 249, 69, 0, 135, 253, 2, 0,
            ],
            &[0, 85, 255, 49, 0, 247, 179, 180, 222, 2, 0, 154, 241, 0, 0],
            &[0, 32, 254, 119, 0, 247, 139, 61, 255, 80, 2, 224, 194, 0, 0],
            &[
                0, 0, 183, 237, 30, 218, 123, 0, 198, 165, 121, 255, 91, 0, 0,
            ],
            &[0, 0, 31, 233, 230, 96, 24, 9, 48, 156, 255, 169, 0, 0, 0],
            &[0, 0, 0, 30, 182, 254, 255, 255, 255, 236, 124, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 69, 83, 57, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¯' / 0xaf
        '¯' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 36, 36, 36, 36, 36, 17, 0, 0, 0, 0],
            &[0, 0, 0, 0, 222, 255, 255, 255, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 0, 209, 240, 240, 240, 240, 240, 118, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '°' / 0xb0
        '°' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 22, 43, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 165, 253, 255, 238, 94, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 150, 255, 215, 163, 246, 253, 56, 0, 0, 0, 0],
            &[0, 0, 0, 0, 243, 244, 15, 0, 100, 255, 147, 0, 0, 0, 0],
            &[0, 0, 0, 2, 250, 244, 11, 0, 96, 255, 156, 0, 0, 0, 0],
            &[0, 0, 0, 0, 181, 255, 175, 114, 224, 255, 88, 0, 0, 0, 0],
            &[0, 0, 0, 0, 31, 218, 255, 255, 255, 161, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 74, 96, 50, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '±' / 0xb1
        '±' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 42, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 202, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 202, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 202, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 37, 77, 77, 77, 218, 255, 158, 77, 77, 77, 10, 0, 0],
            &[
                0, 0, 124, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 115, 236, 236, 236, 251, 255, 245, 236, 236, 236, 30, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 202, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 202, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 202, 255, 116, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 150, 190, 86, 0, 0, 0, 0, 0, 0],
            &[0, 0, 33, 63, 63, 63, 63, 63, 63, 63, 63, 63, 8, 0, 0],
            &[
                0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '²' / 0xb2
        '²' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 127, 212, 230, 189, 56, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 255, 252, 251, 255, 235, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 140, 14, 30, 252, 255, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 23, 254, 255, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 172, 255, 167, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 166, 255, 201, 11, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 175, 255, 193, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 160, 255, 252, 154, 144, 144, 95, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 255, 255, 255, 255, 255, 168, 0, 0, 0, 0],
            &[0, 0, 0, 0, 51, 67, 67, 67, 67, 67, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '³' / 0xb3
        '³' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 140, 213, 223, 177, 45, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 152, 255, 237, 242, 255, 232, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 90, 0, 13, 247, 255, 34, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 31, 91, 255, 209, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 255, 255, 222, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 130, 163, 255, 226, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 9, 0, 0, 0, 202, 255, 99, 0, 0, 0, 0],
            &[0, 0, 0, 0, 195, 151, 102, 121, 248, 255, 79, 0, 0, 0, 0],
            &[0, 0, 0, 0, 188, 255, 255, 255, 255, 173, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 60, 97, 93, 49, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '´' / 0xb4
        '´' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 5, 22, 22, 22, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 152, 255, 255, 174, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 65, 254, 255, 171, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 17, 226, 255, 150, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 70, 60, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'µ' / 0xb5
        'µ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 119, 119, 11, 0, 0, 0, 71, 119, 119, 53, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 168, 255, 255, 113, 0, 0],
            &[0, 0, 255, 255, 255, 50, 0, 0, 0, 194, 255, 255, 113, 0, 0],
            &[0, 0, 255, 255, 255, 100, 0, 0, 20, 246, 255, 255, 113, 0, 0],
            &[
                0, 0, 255, 255, 255, 246, 100, 99, 207, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 255, 255, 255, 255, 255, 255, 255, 198, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 243, 255, 255, 255, 255, 233, 109, 16, 249, 255, 113, 0, 0,
            ],
            &[0, 0, 243, 255, 255, 60, 33, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 10, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¶' / 0xb6
        '¶' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 23, 143, 213, 238, 245, 245, 245, 245, 245, 135, 0, 0,
            ],
            &[
                0, 0, 28, 232, 255, 255, 255, 255, 255, 227, 209, 255, 140, 0, 0,
            ],
            &[
                0, 0, 164, 255, 255, 255, 255, 255, 255, 113, 17, 255, 140, 0, 0,
            ],
            &[
                0, 3, 243, 255, 255, 255, 255, 255, 255, 113, 17, 255, 140, 0, 0,
            ],
            &[
                0, 23, 255, 255, 255, 255, 255, 255, 255, 113, 17, 255, 140, 0, 0,
            ],
            &[
                0, 34, 255, 255, 255, 255, 255, 255, 255, 113, 17, 255, 140, 0, 0,
            ],
            &[
                0, 16, 255, 255, 255, 255, 255, 255, 255, 113, 17, 255, 140, 0, 0,
            ],
            &[
                0, 0, 220, 255, 255, 255, 255, 255, 255, 113, 17, 255, 140, 0, 0,
            ],
            &[
                0, 0, 117, 255, 255, 255, 255, 255, 255, 113, 17, 255, 140, 0, 0,
            ],
            &[
                0, 0, 3, 153, 255, 255, 255, 255, 255, 113, 17, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 32, 93, 111, 116, 255, 113, 17, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 39, 255, 113, 17, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 39, 255, 113, 17, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 39, 255, 113, 17, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 39, 255, 113, 17, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 39, 255, 113, 17, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 39, 255, 113, 17, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 39, 255, 113, 17, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 26, 170, 75, 11, 170, 93, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '·' / 0xb7
        '·' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 10, 28, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 60, 243, 255, 189, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 145, 255, 255, 255, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 105, 255, 255, 239, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 93, 126, 43, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¸' / 0xb8
        '¸' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 26, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 172, 246, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 208, 255, 234, 40, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 189, 255, 136, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¹' / 0xb9
        '¹' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 184, 130, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 172, 255, 255, 180, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 187, 255, 219, 255, 180, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 49, 101, 73, 255, 180, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 77, 255, 180, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 79, 255, 180, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 79, 255, 180, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 42, 145, 255, 253, 52, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 104, 255, 255, 255, 255, 255, 195, 0, 0, 0, 0],
            &[0, 0, 0, 0, 27, 67, 88, 135, 115, 67, 51, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'º' / 0xba
        'º' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 31, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 8, 155, 252, 255, 230, 74, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 255, 201, 140, 249, 245, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 212, 255, 44, 0, 160, 255, 109, 0, 0, 0, 0],
            &[0, 0, 0, 0, 240, 255, 15, 0, 129, 255, 134, 0, 0, 0, 0],
            &[0, 0, 0, 0, 214, 255, 40, 0, 153, 255, 107, 0, 0, 0, 0],
            &[0, 0, 0, 0, 132, 255, 187, 114, 243, 249, 34, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 173, 255, 255, 243, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 27, 50, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '»' / 0xbb
        '»' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 22, 61, 0, 0, 0, 73, 9, 0, 0, 0, 0, 0],
            &[0, 0, 95, 242, 244, 41, 15, 196, 255, 156, 0, 0, 0, 0, 0],
            &[0, 0, 43, 248, 255, 213, 10, 164, 255, 255, 94, 0, 0, 0, 0],
            &[0, 0, 0, 114, 255, 255, 162, 15, 225, 255, 245, 43, 0, 0, 0],
            &[0, 0, 0, 1, 191, 255, 255, 100, 64, 253, 255, 214, 11, 0, 0],
            &[0, 0, 0, 0, 117, 255, 255, 179, 11, 230, 255, 253, 52, 0, 0],
            &[0, 0, 0, 37, 245, 255, 232, 25, 152, 255, 255, 128, 0, 0, 0],
            &[0, 0, 3, 199, 255, 253, 66, 74, 255, 255, 187, 2, 0, 0, 0],
            &[0, 0, 117, 255, 255, 125, 15, 231, 255, 229, 22, 0, 0, 0, 0],
            &[0, 0, 17, 137, 175, 2, 0, 70, 200, 61, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¼' / 0xbc
        '¼' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 98, 232, 246, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 151, 255, 253, 247, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 113, 176, 247, 0, 0, 0, 0, 3, 188, 244, 128, 0, 0],
            &[0, 0, 0, 180, 247, 0, 0, 0, 0, 134, 255, 197, 4, 0, 0],
            &[0, 0, 0, 181, 247, 0, 0, 0, 73, 254, 235, 28, 0, 0, 0],
            &[0, 23, 84, 255, 255, 90, 39, 28, 236, 254, 73, 0, 0, 0, 0],
            &[0, 83, 255, 255, 255, 255, 144, 197, 255, 134, 0, 0, 0, 0, 0],
            &[0, 3, 10, 17, 20, 10, 145, 255, 194, 3, 11, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 78, 254, 234, 26, 44, 249, 255, 11, 0, 0],
            &[0, 0, 0, 0, 32, 238, 253, 70, 4, 204, 251, 255, 11, 0, 0],
            &[0, 0, 0, 6, 201, 255, 131, 0, 122, 234, 181, 255, 11, 0, 0],
            &[0, 0, 0, 145, 255, 190, 3, 42, 248, 85, 167, 255, 11, 0, 0],
            &[
                0, 0, 83, 255, 232, 24, 0, 172, 249, 181, 230, 255, 184, 25, 0,
            ],
            &[
                0, 0, 104, 119, 54, 0, 0, 128, 178, 178, 229, 255, 182, 24, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 167, 255, 11, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '½' / 0xbd
        '½' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 1, 98, 232, 246, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 151, 255, 253, 247, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 49, 113, 176, 247, 0, 0, 0, 0, 3, 188, 244, 128, 0, 0],
            &[0, 0, 0, 180, 247, 0, 0, 0, 0, 134, 255, 197, 4, 0, 0],
            &[0, 0, 0, 181, 247, 0, 0, 0, 73, 254, 235, 28, 0, 0, 0],
            &[0, 23, 84, 255, 255, 90, 39, 28, 236, 254, 73, 0, 0, 0, 0],
            &[0, 83, 255, 255, 255, 255, 144, 197, 255, 134, 0, 0, 0, 0, 0],
            &[0, 3, 10, 17, 20, 10, 145, 255, 194, 8, 43, 24, 0, 0, 0],
            &[0, 0, 0, 0, 0, 78, 254, 234, 124, 243, 255, 253, 119, 0, 0],
            &[0, 0, 0, 0, 32, 238, 253, 70, 81, 160, 88, 241, 237, 0, 0],
            &[0, 0, 0, 6, 201, 255, 131, 0, 0, 0, 12, 235, 213, 0, 0],
            &[0, 0, 0, 145, 255, 190, 3, 0, 0, 7, 184, 254, 77, 0, 0],
            &[0, 0, 83, 255, 232, 24, 0, 0, 6, 180, 254, 102, 0, 0, 0],
            &[0, 0, 104, 119, 54, 0, 0, 0, 157, 255, 209, 136, 136, 36, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 198, 255, 255, 255, 255, 67, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¾' / 0xbe
        '¾' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 19, 34, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 27, 200, 255, 255, 218, 28, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 157, 90, 143, 255, 112, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 66, 157, 251, 57, 0, 0, 3, 188, 244, 128, 0, 0],
            &[0, 0, 33, 255, 255, 188, 15, 0, 0, 134, 255, 197, 4, 0, 0],
            &[0, 0, 3, 27, 93, 255, 136, 0, 73, 254, 235, 28, 0, 0, 0],
            &[0, 52, 162, 101, 153, 255, 150, 28, 236, 254, 73, 0, 0, 0, 0],
            &[0, 39, 232, 255, 255, 203, 33, 197, 255, 134, 0, 0, 0, 0, 0],
            &[0, 0, 3, 26, 18, 0, 139, 255, 194, 3, 11, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 78, 254, 234, 26, 44, 249, 255, 11, 0, 0],
            &[0, 0, 0, 0, 32, 238, 253, 70, 4, 204, 251, 255, 11, 0, 0],
            &[0, 0, 0, 6, 201, 255, 131, 0, 122, 234, 181, 255, 11, 0, 0],
            &[0, 0, 0, 145, 255, 190, 3, 42, 248, 85, 167, 255, 11, 0, 0],
            &[
                0, 0, 83, 255, 232, 24, 0, 172, 249, 181, 230, 255, 184, 25, 0,
            ],
            &[
                0, 0, 104, 119, 54, 0, 0, 128, 178, 178, 229, 255, 182, 24, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 167, 255, 11, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¿' / 0xbf
        '¿' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 32, 71, 17, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 250, 255, 230, 15, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 124, 255, 255, 255, 60, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 66, 254, 255, 239, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 46, 86, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 32, 33, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 250, 255, 188, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 81, 255, 255, 154, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 249, 255, 251, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 141, 255, 255, 242, 74, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 88, 255, 255, 229, 43, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 175, 255, 255, 84, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 201, 255, 255, 55, 0, 0, 0, 30, 40, 0, 0, 0, 0],
            &[
                0, 0, 161, 255, 255, 218, 122, 133, 185, 252, 195, 0, 0, 0, 0,
            ],
            &[
                0, 0, 42, 241, 255, 255, 255, 255, 255, 255, 255, 55, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 2, 18, 10, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'À' / 0xc0
        'À' => &[
            &[0, 0, 0, 20, 154, 157, 157, 40, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 252, 255, 193, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 249, 255, 112, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 62, 189, 183, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 14, 14, 14, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 255, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 254, 255, 255, 255, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 255, 223, 255, 237, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 251, 116, 255, 255, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 230, 255, 201, 46, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 255, 138, 2, 239, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 255, 74, 0, 180, 255, 251, 15, 0, 0, 0],
            &[0, 0, 0, 186, 255, 252, 14, 0, 118, 255, 255, 81, 0, 0, 0],
            &[0, 0, 9, 247, 255, 255, 60, 60, 117, 255, 255, 152, 0, 0, 0],
            &[
                0, 0, 71, 255, 255, 255, 255, 255, 255, 255, 255, 222, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 211, 255, 255, 87, 72, 72, 72, 182, 255, 255, 108, 0, 0,
            ],
            &[0, 26, 255, 255, 203, 0, 0, 0, 0, 65, 255, 255, 179, 0, 0],
            &[0, 96, 255, 255, 139, 0, 0, 0, 0, 9, 250, 255, 243, 6, 0],
            &[0, 167, 255, 255, 75, 0, 0, 0, 0, 0, 198, 255, 255, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Á' / 0xc1
        'Á' => &[
            &[0, 0, 0, 0, 0, 0, 0, 40, 157, 157, 153, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 193, 255, 252, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 115, 255, 248, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 184, 188, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 14, 14, 14, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 255, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 254, 255, 255, 255, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 255, 223, 255, 237, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 251, 116, 255, 255, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 230, 255, 201, 46, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 255, 138, 2, 239, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 255, 74, 0, 180, 255, 251, 15, 0, 0, 0],
            &[0, 0, 0, 186, 255, 252, 14, 0, 118, 255, 255, 81, 0, 0, 0],
            &[0, 0, 9, 247, 255, 255, 60, 60, 117, 255, 255, 152, 0, 0, 0],
            &[
                0, 0, 71, 255, 255, 255, 255, 255, 255, 255, 255, 222, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 211, 255, 255, 87, 72, 72, 72, 182, 255, 255, 108, 0, 0,
            ],
            &[0, 26, 255, 255, 203, 0, 0, 0, 0, 65, 255, 255, 179, 0, 0],
            &[0, 96, 255, 255, 139, 0, 0, 0, 0, 9, 250, 255, 243, 6, 0],
            &[0, 167, 255, 255, 75, 0, 0, 0, 0, 0, 198, 255, 255, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Â' / 0xc2
        'Â' => &[
            &[0, 0, 0, 0, 0, 67, 157, 157, 152, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 228, 255, 255, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 184, 255, 187, 68, 241, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 79, 190, 144, 6, 0, 42, 182, 181, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 14, 14, 14, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 255, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 254, 255, 255, 255, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 255, 223, 255, 237, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 251, 116, 255, 255, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 230, 255, 201, 46, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 255, 138, 2, 239, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 255, 74, 0, 180, 255, 251, 15, 0, 0, 0],
            &[0, 0, 0, 186, 255, 252, 14, 0, 118, 255, 255, 81, 0, 0, 0],
            &[0, 0, 9, 247, 255, 255, 60, 60, 117, 255, 255, 152, 0, 0, 0],
            &[
                0, 0, 71, 255, 255, 255, 255, 255, 255, 255, 255, 222, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 211, 255, 255, 87, 72, 72, 72, 182, 255, 255, 108, 0, 0,
            ],
            &[0, 26, 255, 255, 203, 0, 0, 0, 0, 65, 255, 255, 179, 0, 0],
            &[0, 96, 255, 255, 139, 0, 0, 0, 0, 9, 250, 255, 243, 6, 0],
            &[0, 167, 255, 255, 75, 0, 0, 0, 0, 0, 198, 255, 255, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ã' / 0xc3
        'Ã' => &[
            &[0, 0, 0, 0, 19, 82, 49, 0, 0, 30, 103, 26, 0, 0, 0],
            &[0, 0, 0, 18, 224, 255, 255, 198, 103, 182, 255, 36, 0, 0, 0],
            &[0, 0, 0, 113, 255, 203, 226, 255, 255, 255, 201, 0, 0, 0, 0],
            &[0, 0, 0, 123, 184, 2, 3, 84, 168, 158, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 14, 14, 14, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 255, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 254, 255, 255, 255, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 255, 223, 255, 237, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 251, 116, 255, 255, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 230, 255, 201, 46, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 255, 138, 2, 239, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 255, 74, 0, 180, 255, 251, 15, 0, 0, 0],
            &[0, 0, 0, 186, 255, 252, 14, 0, 118, 255, 255, 81, 0, 0, 0],
            &[0, 0, 9, 247, 255, 255, 60, 60, 117, 255, 255, 152, 0, 0, 0],
            &[
                0, 0, 71, 255, 255, 255, 255, 255, 255, 255, 255, 222, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 211, 255, 255, 87, 72, 72, 72, 182, 255, 255, 108, 0, 0,
            ],
            &[0, 26, 255, 255, 203, 0, 0, 0, 0, 65, 255, 255, 179, 0, 0],
            &[0, 96, 255, 255, 139, 0, 0, 0, 0, 9, 250, 255, 243, 6, 0],
            &[0, 167, 255, 255, 75, 0, 0, 0, 0, 0, 198, 255, 255, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ä' / 0xc4
        'Ä' => &[
            &[0, 0, 0, 6, 77, 48, 0, 0, 10, 82, 40, 0, 0, 0, 0],
            &[0, 0, 0, 136, 255, 255, 53, 0, 162, 255, 252, 34, 0, 0, 0],
            &[0, 0, 0, 152, 255, 255, 70, 0, 178, 255, 255, 48, 0, 0, 0],
            &[0, 0, 0, 28, 151, 120, 1, 0, 38, 155, 108, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 14, 14, 14, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 255, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 254, 255, 255, 255, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 255, 223, 255, 237, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 251, 116, 255, 255, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 230, 255, 201, 46, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 255, 138, 2, 239, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 255, 74, 0, 180, 255, 251, 15, 0, 0, 0],
            &[0, 0, 0, 186, 255, 252, 14, 0, 118, 255, 255, 81, 0, 0, 0],
            &[0, 0, 9, 247, 255, 255, 60, 60, 117, 255, 255, 152, 0, 0, 0],
            &[
                0, 0, 71, 255, 255, 255, 255, 255, 255, 255, 255, 222, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 211, 255, 255, 87, 72, 72, 72, 182, 255, 255, 108, 0, 0,
            ],
            &[0, 26, 255, 255, 203, 0, 0, 0, 0, 65, 255, 255, 179, 0, 0],
            &[0, 96, 255, 255, 139, 0, 0, 0, 0, 9, 250, 255, 243, 6, 0],
            &[0, 167, 255, 255, 75, 0, 0, 0, 0, 0, 198, 255, 255, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Å' / 0xc5
        'Å' => &[
            &[0, 0, 0, 0, 18, 250, 199, 122, 251, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 100, 0, 220, 191, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 223, 242, 208, 255, 117, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 35, 155, 177, 107, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 14, 14, 14, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 255, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 254, 255, 255, 255, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 255, 223, 255, 237, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 251, 116, 255, 255, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 230, 255, 201, 46, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 255, 138, 2, 239, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 255, 74, 0, 180, 255, 251, 15, 0, 0, 0],
            &[0, 0, 0, 186, 255, 252, 14, 0, 118, 255, 255, 81, 0, 0, 0],
            &[0, 0, 9, 247, 255, 255, 60, 60, 117, 255, 255, 152, 0, 0, 0],
            &[
                0, 0, 71, 255, 255, 255, 255, 255, 255, 255, 255, 222, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 211, 255, 255, 87, 72, 72, 72, 182, 255, 255, 108, 0, 0,
            ],
            &[0, 26, 255, 255, 203, 0, 0, 0, 0, 65, 255, 255, 179, 0, 0],
            &[0, 96, 255, 255, 139, 0, 0, 0, 0, 9, 250, 255, 243, 6, 0],
            &[0, 167, 255, 255, 75, 0, 0, 0, 0, 0, 198, 255, 255, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 98, 240, 254, 193, 24, 0, 0, 0, 0, 0],
        ],
        // letter: 'Æ' / 0xc6
        'Æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 86, 254, 254, 254, 254, 254, 254, 254, 246, 0, 0],
            &[
                0, 0, 0, 0, 147, 255, 255, 255, 255, 255, 255, 255, 248, 0, 0,
            ],
            &[
                0, 0, 0, 0, 208, 255, 227, 207, 255, 255, 159, 156, 151, 0, 0,
            ],
            &[0, 0, 0, 15, 253, 255, 142, 130, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 74, 255, 255, 86, 130, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 0, 135, 255, 255, 29, 130, 255, 255, 30, 23, 14, 0, 0],
            &[
                0, 0, 0, 196, 255, 229, 0, 130, 255, 255, 255, 255, 151, 0, 0,
            ],
            &[
                0, 0, 8, 248, 255, 173, 0, 130, 255, 255, 255, 255, 151, 0, 0,
            ],
            &[
                0, 0, 62, 255, 255, 193, 76, 207, 255, 255, 134, 130, 77, 0, 0,
            ],
            &[0, 0, 122, 255, 255, 255, 255, 255, 255, 255, 7, 0, 0, 0, 0],
            &[0, 0, 183, 255, 255, 255, 255, 255, 255, 255, 7, 0, 0, 0, 0],
            &[0, 3, 241, 255, 255, 77, 77, 208, 255, 255, 7, 0, 0, 0, 0],
            &[
                0, 49, 255, 255, 147, 0, 0, 130, 255, 255, 157, 154, 149, 0, 0,
            ],
            &[
                0, 110, 255, 255, 90, 0, 0, 130, 255, 255, 255, 255, 248, 0, 0,
            ],
            &[
                0, 171, 255, 255, 34, 0, 0, 130, 255, 255, 255, 255, 248, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ç' / 0xc7
        'Ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 38, 41, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 85, 197, 252, 255, 255, 255, 209, 88, 0, 0],
            &[
                0, 0, 0, 4, 163, 255, 255, 255, 255, 255, 255, 255, 107, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 188, 131, 132, 177, 239, 15, 0, 0,
            ],
            &[0, 0, 26, 251, 255, 254, 91, 0, 0, 0, 0, 9, 0, 0, 0],
            &[0, 0, 121, 255, 255, 176, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 182, 255, 255, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 221, 255, 255, 63, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 237, 255, 255, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 228, 255, 255, 48, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 207, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 255, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 80, 255, 255, 249, 60, 0, 0, 0, 0, 19, 11, 0, 0],
            &[
                0, 0, 2, 205, 255, 255, 250, 173, 127, 137, 181, 250, 44, 0, 0,
            ],
            &[
                0, 0, 0, 42, 228, 255, 255, 255, 255, 255, 255, 255, 44, 0, 0,
            ],
            &[0, 0, 0, 0, 28, 151, 244, 255, 255, 255, 239, 164, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 255, 241, 28, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 55, 229, 255, 200, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 11, 246, 255, 68, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'È' / 0xc8
        'È' => &[
            &[0, 0, 0, 8, 146, 157, 157, 60, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 68, 245, 255, 218, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 240, 255, 145, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 43, 183, 189, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 254, 254, 254, 254, 254, 254, 254, 254, 254, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 220, 161, 161, 161, 161, 161, 161, 20, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 167, 23, 23, 23, 23, 23, 19, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 210, 136, 136, 136, 136, 136, 110, 0, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 219, 159, 159, 159, 159, 159, 159, 20, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'É' / 0xc9
        'É' => &[
            &[0, 0, 0, 0, 0, 0, 0, 21, 156, 157, 156, 36, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 163, 255, 255, 122, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 83, 255, 253, 109, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 172, 190, 81, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 254, 254, 254, 254, 254, 254, 254, 254, 254, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 220, 161, 161, 161, 161, 161, 161, 20, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 167, 23, 23, 23, 23, 23, 19, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 210, 136, 136, 136, 136, 136, 110, 0, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 219, 159, 159, 159, 159, 159, 159, 20, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ê' / 0xca
        'Ê' => &[
            &[0, 0, 0, 0, 0, 47, 157, 157, 157, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 208, 255, 255, 255, 185, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 154, 255, 208, 59, 228, 255, 124, 0, 0, 0, 0],
            &[0, 0, 0, 55, 190, 160, 14, 0, 27, 173, 188, 30, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 254, 254, 254, 254, 254, 254, 254, 254, 254, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 220, 161, 161, 161, 161, 161, 161, 20, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 167, 23, 23, 23, 23, 23, 19, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 210, 136, 136, 136, 136, 136, 110, 0, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 219, 159, 159, 159, 159, 159, 159, 20, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ë' / 0xcb
        'Ë' => &[
            &[0, 0, 0, 2, 69, 59, 0, 0, 5, 75, 51, 0, 0, 0, 0],
            &[0, 0, 0, 103, 255, 255, 85, 0, 130, 255, 255, 63, 0, 0, 0],
            &[0, 0, 0, 120, 255, 255, 102, 0, 146, 255, 255, 80, 0, 0, 0],
            &[0, 0, 0, 15, 143, 134, 8, 0, 24, 149, 125, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 254, 254, 254, 254, 254, 254, 254, 254, 254, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 220, 161, 161, 161, 161, 161, 161, 20, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 167, 23, 23, 23, 23, 23, 19, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 210, 136, 136, 136, 136, 136, 110, 0, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 219, 159, 159, 159, 159, 159, 159, 20, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ì' / 0xcc
        'Ì' => &[
            &[0, 0, 0, 20, 154, 157, 157, 40, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 252, 255, 193, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 249, 255, 112, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 62, 189, 183, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 254, 254, 254, 254, 254, 254, 254, 254, 180, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 3, 45, 74, 130, 255, 255, 215, 90, 62, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 43, 73, 129, 255, 255, 215, 89, 60, 25, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Í' / 0xcd
        'Í' => &[
            &[0, 0, 0, 0, 0, 0, 0, 40, 157, 157, 153, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 193, 255, 252, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 115, 255, 248, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 184, 188, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 254, 254, 254, 254, 254, 254, 254, 254, 180, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 3, 45, 74, 130, 255, 255, 215, 90, 62, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 43, 73, 129, 255, 255, 215, 89, 60, 25, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Î' / 0xce
        'Î' => &[
            &[0, 0, 0, 0, 0, 67, 157, 157, 152, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 228, 255, 255, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 184, 255, 187, 68, 241, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 79, 190, 144, 6, 0, 42, 182, 181, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 254, 254, 254, 254, 254, 254, 254, 254, 180, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 3, 45, 74, 130, 255, 255, 215, 90, 62, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 43, 73, 129, 255, 255, 215, 89, 60, 25, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ï' / 0xcf
        'Ï' => &[
            &[0, 0, 0, 6, 77, 48, 0, 0, 10, 82, 40, 0, 0, 0, 0],
            &[0, 0, 0, 136, 255, 255, 53, 0, 162, 255, 252, 34, 0, 0, 0],
            &[0, 0, 0, 152, 255, 255, 70, 0, 178, 255, 255, 48, 0, 0, 0],
            &[0, 0, 0, 28, 151, 120, 1, 0, 38, 155, 108, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 254, 254, 254, 254, 254, 254, 254, 254, 180, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 3, 45, 74, 130, 255, 255, 215, 90, 62, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 43, 73, 129, 255, 255, 215, 89, 60, 25, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ð' / 0xd0
        'Ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 254, 254, 251, 227, 185, 89, 2, 0, 0, 0, 0],
            &[
                0, 14, 255, 255, 255, 255, 255, 255, 255, 255, 184, 15, 0, 0, 0,
            ],
            &[
                0, 14, 255, 255, 250, 150, 155, 193, 254, 255, 255, 166, 0, 0, 0,
            ],
            &[0, 14, 255, 255, 242, 0, 0, 0, 70, 250, 255, 255, 58, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 154, 255, 255, 145, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 78, 255, 255, 205, 0, 0],
            &[
                0, 186, 255, 255, 255, 172, 172, 29, 0, 44, 255, 255, 228, 0, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 44, 0, 33, 255, 255, 242, 0, 0,
            ],
            &[
                0, 230, 255, 255, 255, 216, 216, 37, 0, 45, 255, 255, 222, 0, 0,
            ],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 85, 255, 255, 196, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 168, 255, 255, 126, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 90, 254, 255, 251, 38, 0, 0],
            &[
                0, 14, 255, 255, 250, 149, 156, 202, 255, 255, 255, 136, 0, 0, 0,
            ],
            &[
                0, 14, 255, 255, 255, 255, 255, 255, 255, 255, 153, 5, 0, 0, 0,
            ],
            &[0, 14, 255, 255, 255, 255, 249, 221, 171, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ñ' / 0xd1
        'Ñ' => &[
            &[0, 0, 0, 0, 13, 79, 57, 0, 0, 19, 103, 37, 0, 0, 0],
            &[0, 0, 0, 6, 209, 255, 255, 210, 110, 164, 255, 63, 0, 0, 0],
            &[0, 0, 0, 86, 255, 210, 219, 255, 255, 255, 223, 6, 0, 0, 0],
            &[0, 0, 0, 102, 194, 13, 0, 72, 162, 165, 39, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 254, 254, 62, 0, 0, 13, 254, 254, 166, 0, 0],
            &[
                0, 14, 255, 255, 255, 255, 145, 0, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 250, 255, 228, 1, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 188, 255, 255, 57, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 141, 234, 255, 141, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 152, 154, 255, 224, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 162, 71, 255, 255, 52, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 166, 5, 238, 255, 136, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 161, 255, 219, 10, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 78, 255, 255, 52, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 8, 243, 255, 131, 251, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 168, 255, 214, 239, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 85, 255, 255, 254, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 11, 246, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 14, 255, 255, 167, 0, 0, 0, 175, 255, 255, 255, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ò' / 0xd2
        'Ò' => &[
            &[0, 0, 0, 20, 154, 157, 157, 40, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 252, 255, 193, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 249, 255, 112, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 62, 189, 183, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 36, 55, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 133, 241, 255, 255, 255, 212, 73, 0, 0, 0, 0],
            &[0, 0, 2, 177, 255, 255, 255, 255, 255, 255, 254, 85, 0, 0, 0],
            &[
                0, 0, 89, 255, 255, 250, 147, 116, 202, 255, 255, 236, 15, 0, 0,
            ],
            &[0, 0, 202, 255, 255, 117, 0, 0, 12, 217, 255, 255, 101, 0, 0],
            &[0, 14, 252, 255, 251, 16, 0, 0, 0, 124, 255, 255, 172, 0, 0],
            &[0, 59, 255, 255, 217, 0, 0, 0, 0, 66, 255, 255, 214, 0, 0],
            &[0, 77, 255, 255, 194, 0, 0, 0, 0, 46, 255, 255, 235, 0, 0],
            &[0, 91, 255, 255, 186, 0, 0, 0, 0, 33, 255, 255, 245, 0, 0],
            &[0, 82, 255, 255, 194, 0, 0, 0, 0, 41, 255, 255, 235, 0, 0],
            &[0, 60, 255, 255, 219, 0, 0, 0, 0, 67, 255, 255, 213, 0, 0],
            &[0, 18, 254, 255, 252, 18, 0, 0, 0, 120, 255, 255, 170, 0, 0],
            &[0, 0, 201, 255, 255, 123, 0, 0, 9, 219, 255, 255, 99, 0, 0],
            &[
                0, 0, 93, 255, 255, 252, 156, 119, 204, 255, 255, 233, 13, 0, 0,
            ],
            &[0, 0, 2, 176, 255, 255, 255, 255, 255, 255, 252, 79, 0, 0, 0],
            &[0, 0, 0, 6, 129, 235, 255, 255, 255, 203, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 34, 46, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ó' / 0xd3
        'Ó' => &[
            &[0, 0, 0, 0, 0, 0, 0, 40, 157, 157, 153, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 193, 255, 252, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 115, 255, 248, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 184, 188, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 36, 55, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 133, 241, 255, 255, 255, 212, 73, 0, 0, 0, 0],
            &[0, 0, 2, 177, 255, 255, 255, 255, 255, 255, 254, 85, 0, 0, 0],
            &[
                0, 0, 89, 255, 255, 250, 147, 116, 202, 255, 255, 236, 15, 0, 0,
            ],
            &[0, 0, 202, 255, 255, 117, 0, 0, 12, 217, 255, 255, 101, 0, 0],
            &[0, 14, 252, 255, 251, 16, 0, 0, 0, 124, 255, 255, 172, 0, 0],
            &[0, 59, 255, 255, 217, 0, 0, 0, 0, 66, 255, 255, 214, 0, 0],
            &[0, 77, 255, 255, 194, 0, 0, 0, 0, 46, 255, 255, 235, 0, 0],
            &[0, 91, 255, 255, 186, 0, 0, 0, 0, 33, 255, 255, 245, 0, 0],
            &[0, 82, 255, 255, 194, 0, 0, 0, 0, 41, 255, 255, 235, 0, 0],
            &[0, 60, 255, 255, 219, 0, 0, 0, 0, 67, 255, 255, 213, 0, 0],
            &[0, 18, 254, 255, 252, 18, 0, 0, 0, 120, 255, 255, 170, 0, 0],
            &[0, 0, 201, 255, 255, 123, 0, 0, 9, 219, 255, 255, 99, 0, 0],
            &[
                0, 0, 93, 255, 255, 252, 156, 119, 204, 255, 255, 233, 13, 0, 0,
            ],
            &[0, 0, 2, 176, 255, 255, 255, 255, 255, 255, 252, 79, 0, 0, 0],
            &[0, 0, 0, 6, 129, 235, 255, 255, 255, 203, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 34, 46, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ô' / 0xd4
        'Ô' => &[
            &[0, 0, 0, 0, 0, 67, 157, 157, 152, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 228, 255, 255, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 184, 255, 187, 68, 241, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 79, 190, 144, 6, 0, 42, 182, 181, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 36, 55, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 133, 241, 255, 255, 255, 212, 73, 0, 0, 0, 0],
            &[0, 0, 2, 177, 255, 255, 255, 255, 255, 255, 254, 85, 0, 0, 0],
            &[
                0, 0, 89, 255, 255, 250, 147, 116, 202, 255, 255, 236, 15, 0, 0,
            ],
            &[0, 0, 202, 255, 255, 117, 0, 0, 12, 217, 255, 255, 101, 0, 0],
            &[0, 14, 252, 255, 251, 16, 0, 0, 0, 124, 255, 255, 172, 0, 0],
            &[0, 59, 255, 255, 217, 0, 0, 0, 0, 66, 255, 255, 214, 0, 0],
            &[0, 77, 255, 255, 194, 0, 0, 0, 0, 46, 255, 255, 235, 0, 0],
            &[0, 91, 255, 255, 186, 0, 0, 0, 0, 33, 255, 255, 245, 0, 0],
            &[0, 82, 255, 255, 194, 0, 0, 0, 0, 41, 255, 255, 235, 0, 0],
            &[0, 60, 255, 255, 219, 0, 0, 0, 0, 67, 255, 255, 213, 0, 0],
            &[0, 18, 254, 255, 252, 18, 0, 0, 0, 120, 255, 255, 170, 0, 0],
            &[0, 0, 201, 255, 255, 123, 0, 0, 9, 219, 255, 255, 99, 0, 0],
            &[
                0, 0, 93, 255, 255, 252, 156, 119, 204, 255, 255, 233, 13, 0, 0,
            ],
            &[0, 0, 2, 176, 255, 255, 255, 255, 255, 255, 252, 79, 0, 0, 0],
            &[0, 0, 0, 6, 129, 235, 255, 255, 255, 203, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 34, 46, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Õ' / 0xd5
        'Õ' => &[
            &[0, 0, 0, 0, 19, 82, 49, 0, 0, 30, 103, 26, 0, 0, 0],
            &[0, 0, 0, 18, 224, 255, 255, 198, 103, 182, 255, 36, 0, 0, 0],
            &[0, 0, 0, 113, 255, 203, 226, 255, 255, 255, 201, 0, 0, 0, 0],
            &[0, 0, 0, 123, 184, 2, 3, 84, 168, 158, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 36, 55, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 133, 241, 255, 255, 255, 212, 73, 0, 0, 0, 0],
            &[0, 0, 2, 177, 255, 255, 255, 255, 255, 255, 254, 85, 0, 0, 0],
            &[
                0, 0, 89, 255, 255, 250, 147, 116, 202, 255, 255, 236, 15, 0, 0,
            ],
            &[0, 0, 202, 255, 255, 117, 0, 0, 12, 217, 255, 255, 101, 0, 0],
            &[0, 14, 252, 255, 251, 16, 0, 0, 0, 124, 255, 255, 172, 0, 0],
            &[0, 59, 255, 255, 217, 0, 0, 0, 0, 66, 255, 255, 214, 0, 0],
            &[0, 77, 255, 255, 194, 0, 0, 0, 0, 46, 255, 255, 235, 0, 0],
            &[0, 91, 255, 255, 186, 0, 0, 0, 0, 33, 255, 255, 245, 0, 0],
            &[0, 82, 255, 255, 194, 0, 0, 0, 0, 41, 255, 255, 235, 0, 0],
            &[0, 60, 255, 255, 219, 0, 0, 0, 0, 67, 255, 255, 213, 0, 0],
            &[0, 18, 254, 255, 252, 18, 0, 0, 0, 120, 255, 255, 170, 0, 0],
            &[0, 0, 201, 255, 255, 123, 0, 0, 9, 219, 255, 255, 99, 0, 0],
            &[
                0, 0, 93, 255, 255, 252, 156, 119, 204, 255, 255, 233, 13, 0, 0,
            ],
            &[0, 0, 2, 176, 255, 255, 255, 255, 255, 255, 252, 79, 0, 0, 0],
            &[0, 0, 0, 6, 129, 235, 255, 255, 255, 203, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 34, 46, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ö' / 0xd6
        'Ö' => &[
            &[0, 0, 0, 6, 77, 48, 0, 0, 10, 82, 40, 0, 0, 0, 0],
            &[0, 0, 0, 136, 255, 255, 53, 0, 162, 255, 252, 34, 0, 0, 0],
            &[0, 0, 0, 152, 255, 255, 70, 0, 178, 255, 255, 48, 0, 0, 0],
            &[0, 0, 0, 28, 151, 120, 1, 0, 38, 155, 108, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 36, 55, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 133, 241, 255, 255, 255, 212, 73, 0, 0, 0, 0],
            &[0, 0, 2, 177, 255, 255, 255, 255, 255, 255, 254, 85, 0, 0, 0],
            &[
                0, 0, 89, 255, 255, 250, 147, 116, 202, 255, 255, 236, 15, 0, 0,
            ],
            &[0, 0, 202, 255, 255, 117, 0, 0, 12, 217, 255, 255, 101, 0, 0],
            &[0, 14, 252, 255, 251, 16, 0, 0, 0, 124, 255, 255, 172, 0, 0],
            &[0, 59, 255, 255, 217, 0, 0, 0, 0, 66, 255, 255, 214, 0, 0],
            &[0, 77, 255, 255, 194, 0, 0, 0, 0, 46, 255, 255, 235, 0, 0],
            &[0, 91, 255, 255, 186, 0, 0, 0, 0, 33, 255, 255, 245, 0, 0],
            &[0, 82, 255, 255, 194, 0, 0, 0, 0, 41, 255, 255, 235, 0, 0],
            &[0, 60, 255, 255, 219, 0, 0, 0, 0, 67, 255, 255, 213, 0, 0],
            &[0, 18, 254, 255, 252, 18, 0, 0, 0, 120, 255, 255, 170, 0, 0],
            &[0, 0, 201, 255, 255, 123, 0, 0, 9, 219, 255, 255, 99, 0, 0],
            &[
                0, 0, 93, 255, 255, 252, 156, 119, 204, 255, 255, 233, 13, 0, 0,
            ],
            &[0, 0, 2, 176, 255, 255, 255, 255, 255, 255, 252, 79, 0, 0, 0],
            &[0, 0, 0, 6, 129, 235, 255, 255, 255, 203, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 34, 46, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '×' / 0xd7
        '×' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 70, 188, 13, 0, 0, 0, 66, 193, 17, 0, 0, 0],
            &[0, 0, 26, 245, 255, 199, 15, 0, 70, 246, 255, 175, 0, 0, 0],
            &[0, 0, 0, 109, 255, 255, 203, 91, 248, 255, 224, 32, 0, 0, 0],
            &[0, 0, 0, 0, 109, 255, 255, 255, 255, 224, 32, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 152, 255, 255, 254, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 69, 247, 255, 255, 255, 196, 12, 0, 0, 0, 0],
            &[0, 0, 0, 66, 246, 255, 223, 126, 254, 255, 194, 11, 0, 0, 0],
            &[0, 0, 23, 244, 255, 223, 31, 0, 104, 254, 255, 163, 0, 0, 0],
            &[0, 0, 0, 106, 221, 30, 0, 0, 0, 104, 224, 33, 0, 0, 0],
            &[0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ø' / 0xd8
        'Ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 36, 55, 30, 0, 0, 94, 71, 0, 0],
            &[
                0, 0, 0, 6, 133, 241, 255, 255, 255, 212, 109, 245, 251, 33, 0,
            ],
            &[
                0, 0, 2, 177, 255, 255, 255, 255, 255, 255, 255, 255, 135, 0, 0,
            ],
            &[
                0, 0, 89, 255, 255, 250, 147, 116, 202, 255, 255, 255, 22, 0, 0,
            ],
            &[0, 0, 202, 255, 255, 117, 0, 0, 51, 255, 255, 255, 101, 0, 0],
            &[
                0, 14, 252, 255, 251, 16, 0, 3, 199, 255, 255, 255, 172, 0, 0,
            ],
            &[
                0, 59, 255, 255, 217, 0, 0, 119, 255, 255, 255, 255, 214, 0, 0,
            ],
            &[
                0, 77, 255, 255, 194, 0, 42, 248, 251, 98, 255, 255, 235, 0, 0,
            ],
            &[
                0, 91, 255, 255, 186, 4, 203, 255, 131, 33, 255, 255, 245, 0, 0,
            ],
            &[
                0, 82, 255, 255, 194, 123, 255, 210, 6, 41, 255, 255, 235, 0, 0,
            ],
            &[
                0, 60, 255, 255, 255, 249, 251, 49, 0, 67, 255, 255, 213, 0, 0,
            ],
            &[
                0, 18, 254, 255, 255, 255, 129, 0, 0, 120, 255, 255, 170, 0, 0,
            ],
            &[0, 0, 201, 255, 255, 255, 5, 0, 9, 219, 255, 255, 99, 0, 0],
            &[
                0, 0, 143, 255, 255, 255, 156, 119, 204, 255, 255, 233, 13, 0, 0,
            ],
            &[
                0, 6, 212, 255, 255, 255, 255, 255, 255, 255, 252, 79, 0, 0, 0,
            ],
            &[
                0, 122, 255, 213, 134, 235, 255, 255, 255, 203, 65, 0, 0, 0, 0,
            ],
            &[0, 17, 160, 46, 0, 3, 34, 46, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ù' / 0xd9
        'Ù' => &[
            &[0, 0, 0, 20, 154, 157, 157, 40, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 252, 255, 193, 1, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 249, 255, 112, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 62, 189, 183, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 251, 0, 0, 0, 0, 99, 254, 254, 166, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 13, 255, 255, 254, 1, 0, 0, 0, 102, 255, 255, 167, 0, 0],
            &[0, 2, 252, 255, 255, 21, 0, 0, 0, 123, 255, 255, 151, 0, 0],
            &[0, 0, 221, 255, 255, 99, 0, 0, 1, 196, 255, 255, 115, 0, 0],
            &[
                0, 0, 138, 255, 255, 246, 156, 115, 183, 255, 255, 251, 33, 0, 0,
            ],
            &[
                0, 0, 15, 222, 255, 255, 255, 255, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[0, 0, 0, 21, 161, 244, 255, 255, 255, 213, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 35, 45, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ú' / 0xda
        'Ú' => &[
            &[0, 0, 0, 0, 0, 0, 0, 40, 157, 157, 153, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 193, 255, 252, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 115, 255, 248, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 184, 188, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 251, 0, 0, 0, 0, 99, 254, 254, 166, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 13, 255, 255, 254, 1, 0, 0, 0, 102, 255, 255, 167, 0, 0],
            &[0, 2, 252, 255, 255, 21, 0, 0, 0, 123, 255, 255, 151, 0, 0],
            &[0, 0, 221, 255, 255, 99, 0, 0, 1, 196, 255, 255, 115, 0, 0],
            &[
                0, 0, 138, 255, 255, 246, 156, 115, 183, 255, 255, 251, 33, 0, 0,
            ],
            &[
                0, 0, 15, 222, 255, 255, 255, 255, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[0, 0, 0, 21, 161, 244, 255, 255, 255, 213, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 35, 45, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Û' / 0xdb
        'Û' => &[
            &[0, 0, 0, 0, 0, 67, 157, 157, 152, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 228, 255, 255, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 184, 255, 187, 68, 241, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 79, 190, 144, 6, 0, 42, 182, 181, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 251, 0, 0, 0, 0, 99, 254, 254, 166, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 13, 255, 255, 254, 1, 0, 0, 0, 102, 255, 255, 167, 0, 0],
            &[0, 2, 252, 255, 255, 21, 0, 0, 0, 123, 255, 255, 151, 0, 0],
            &[0, 0, 221, 255, 255, 99, 0, 0, 1, 196, 255, 255, 115, 0, 0],
            &[
                0, 0, 138, 255, 255, 246, 156, 115, 183, 255, 255, 251, 33, 0, 0,
            ],
            &[
                0, 0, 15, 222, 255, 255, 255, 255, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[0, 0, 0, 21, 161, 244, 255, 255, 255, 213, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 35, 45, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ü' / 0xdc
        'Ü' => &[
            &[0, 0, 0, 6, 77, 48, 0, 0, 10, 82, 40, 0, 0, 0, 0],
            &[0, 0, 0, 136, 255, 255, 53, 0, 162, 255, 252, 34, 0, 0, 0],
            &[0, 0, 0, 152, 255, 255, 70, 0, 178, 255, 255, 48, 0, 0, 0],
            &[0, 0, 0, 28, 151, 120, 1, 0, 38, 155, 108, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 251, 0, 0, 0, 0, 99, 254, 254, 166, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 13, 255, 255, 254, 1, 0, 0, 0, 102, 255, 255, 167, 0, 0],
            &[0, 2, 252, 255, 255, 21, 0, 0, 0, 123, 255, 255, 151, 0, 0],
            &[0, 0, 221, 255, 255, 99, 0, 0, 1, 196, 255, 255, 115, 0, 0],
            &[
                0, 0, 138, 255, 255, 246, 156, 115, 183, 255, 255, 251, 33, 0, 0,
            ],
            &[
                0, 0, 15, 222, 255, 255, 255, 255, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[0, 0, 0, 21, 161, 244, 255, 255, 255, 213, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 35, 45, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ý' / 0xdd
        'Ý' => &[
            &[0, 0, 0, 0, 0, 0, 0, 40, 157, 157, 153, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 193, 255, 252, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 115, 255, 248, 82, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 184, 188, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 87, 254, 254, 213, 0, 0, 0, 0, 56, 254, 254, 231, 8, 0],
            &[0, 3, 219, 255, 255, 67, 0, 0, 0, 163, 255, 255, 120, 0, 0],
            &[0, 0, 100, 255, 255, 175, 0, 0, 21, 248, 255, 240, 14, 0, 0],
            &[0, 0, 6, 228, 255, 252, 31, 0, 121, 255, 255, 134, 0, 0, 0],
            &[0, 0, 0, 112, 255, 255, 135, 3, 225, 255, 246, 22, 0, 0, 0],
            &[0, 0, 0, 11, 235, 255, 235, 88, 255, 255, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 255, 255, 241, 255, 251, 31, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 242, 255, 255, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 137, 255, 255, 254, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 255, 255, 215, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Þ' / 0xde
        'Þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 254, 254, 157, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 204, 121, 120, 107, 66, 5, 0, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 227, 65, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 243, 30, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 169, 27, 31, 74, 225, 255, 255, 132, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 102, 255, 255, 181, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 82, 255, 255, 186, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 157, 255, 255, 157, 0, 0],
            &[
                0, 0, 81, 255, 255, 197, 101, 118, 178, 255, 255, 255, 81, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 185, 1, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 255, 255, 255, 255, 231, 132, 6, 0, 0, 0],
            &[0, 0, 81, 255, 255, 176, 47, 44, 23, 1, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ß' / 0xdf
        'ß' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 8, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 137, 223, 254, 255, 233, 160, 36, 0, 0, 0, 0],
            &[
                0, 0, 13, 212, 255, 255, 255, 255, 255, 255, 239, 42, 0, 0, 0,
            ],
            &[
                0, 0, 122, 255, 255, 241, 136, 126, 217, 255, 255, 161, 0, 0, 0,
            ],
            &[0, 0, 191, 255, 255, 93, 0, 0, 60, 255, 255, 190, 0, 0, 0],
            &[0, 0, 214, 255, 255, 37, 0, 0, 109, 255, 255, 140, 0, 0, 0],
            &[0, 0, 216, 255, 255, 29, 0, 73, 247, 255, 226, 20, 0, 0, 0],
            &[0, 0, 216, 255, 255, 29, 44, 247, 255, 223, 33, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 29, 131, 255, 255, 107, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 29, 124, 255, 255, 224, 45, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 29, 22, 232, 255, 255, 246, 84, 0, 0, 0],
            &[0, 0, 216, 255, 255, 29, 0, 35, 213, 255, 255, 254, 72, 0, 0],
            &[0, 0, 216, 255, 255, 29, 0, 0, 10, 169, 255, 255, 202, 0, 0],
            &[0, 0, 216, 255, 255, 29, 0, 0, 0, 22, 255, 255, 242, 0, 0],
            &[
                0, 0, 216, 255, 255, 29, 178, 97, 56, 140, 255, 255, 226, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 29, 229, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 29, 171, 253, 255, 255, 244, 143, 4, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 16, 44, 39, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'à' / 0xe0
        'à' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 22, 22, 22, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 176, 255, 255, 152, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 173, 255, 254, 64, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 153, 255, 225, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 60, 70, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 83, 137, 164, 167, 146, 89, 9, 0, 0, 0, 0],
            &[
                0, 0, 21, 240, 255, 255, 255, 255, 255, 255, 225, 45, 0, 0, 0,
            ],
            &[
                0, 0, 0, 182, 255, 240, 209, 205, 247, 255, 255, 212, 0, 0, 0,
            ],
            &[0, 0, 0, 40, 56, 2, 0, 0, 25, 230, 255, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 24, 30, 184, 255, 255, 58, 0, 0],
            &[
                0, 0, 0, 62, 180, 240, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 75, 252, 255, 255, 229, 192, 180, 229, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 196, 255, 255, 114, 0, 0, 0, 174, 255, 255, 60, 0, 0],
            &[0, 0, 234, 255, 255, 18, 0, 0, 7, 233, 255, 255, 60, 0, 0],
            &[
                0, 0, 210, 255, 255, 160, 67, 85, 199, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 255, 255, 253, 210, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 7, 151, 253, 255, 255, 215, 76, 61, 255, 255, 60, 0, 0],
            &[0, 0, 0, 0, 9, 43, 29, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'á' / 0xe1
        'á' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 20, 22, 22, 9, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 66, 255, 255, 229, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 221, 255, 225, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 158, 255, 210, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 68, 70, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 83, 137, 164, 167, 146, 89, 9, 0, 0, 0, 0],
            &[
                0, 0, 21, 240, 255, 255, 255, 255, 255, 255, 225, 45, 0, 0, 0,
            ],
            &[
                0, 0, 0, 182, 255, 240, 209, 205, 247, 255, 255, 212, 0, 0, 0,
            ],
            &[0, 0, 0, 40, 56, 2, 0, 0, 25, 230, 255, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 24, 30, 184, 255, 255, 58, 0, 0],
            &[
                0, 0, 0, 62, 180, 240, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 75, 252, 255, 255, 229, 192, 180, 229, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 196, 255, 255, 114, 0, 0, 0, 174, 255, 255, 60, 0, 0],
            &[0, 0, 234, 255, 255, 18, 0, 0, 7, 233, 255, 255, 60, 0, 0],
            &[
                0, 0, 210, 255, 255, 160, 67, 85, 199, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 255, 255, 253, 210, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 7, 151, 253, 255, 255, 215, 76, 61, 255, 255, 60, 0, 0],
            &[0, 0, 0, 0, 9, 43, 29, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'â' / 0xe2
        'â' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 22, 22, 22, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 113, 255, 255, 255, 104, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 47, 247, 255, 193, 255, 244, 42, 0, 0, 0, 0],
            &[0, 0, 0, 21, 222, 253, 112, 0, 126, 254, 216, 15, 0, 0, 0],
            &[0, 0, 0, 26, 70, 46, 0, 0, 0, 49, 70, 21, 0, 0, 0],
            &[0, 0, 0, 12, 83, 137, 164, 167, 146, 89, 9, 0, 0, 0, 0],
            &[
                0, 0, 21, 240, 255, 255, 255, 255, 255, 255, 225, 45, 0, 0, 0,
            ],
            &[
                0, 0, 0, 182, 255, 240, 209, 205, 247, 255, 255, 212, 0, 0, 0,
            ],
            &[0, 0, 0, 40, 56, 2, 0, 0, 25, 230, 255, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 24, 30, 184, 255, 255, 58, 0, 0],
            &[
                0, 0, 0, 62, 180, 240, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 75, 252, 255, 255, 229, 192, 180, 229, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 196, 255, 255, 114, 0, 0, 0, 174, 255, 255, 60, 0, 0],
            &[0, 0, 234, 255, 255, 18, 0, 0, 7, 233, 255, 255, 60, 0, 0],
            &[
                0, 0, 210, 255, 255, 160, 67, 85, 199, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 255, 255, 253, 210, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 7, 151, 253, 255, 255, 215, 76, 61, 255, 255, 60, 0, 0],
            &[0, 0, 0, 0, 9, 43, 29, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ã' / 0xe3
        'ã' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 79, 197, 183, 83, 2, 48, 224, 86, 0, 0, 0],
            &[0, 0, 0, 25, 249, 255, 255, 255, 233, 238, 255, 48, 0, 0, 0],
            &[0, 0, 0, 100, 255, 114, 93, 200, 255, 255, 159, 0, 0, 0, 0],
            &[0, 0, 0, 37, 75, 3, 0, 0, 38, 48, 1, 0, 0, 0, 0],
            &[0, 0, 0, 12, 83, 137, 164, 167, 146, 89, 9, 0, 0, 0, 0],
            &[
                0, 0, 21, 240, 255, 255, 255, 255, 255, 255, 225, 45, 0, 0, 0,
            ],
            &[
                0, 0, 0, 182, 255, 240, 209, 205, 247, 255, 255, 212, 0, 0, 0,
            ],
            &[0, 0, 0, 40, 56, 2, 0, 0, 25, 230, 255, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 24, 30, 184, 255, 255, 58, 0, 0],
            &[
                0, 0, 0, 62, 180, 240, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 75, 252, 255, 255, 229, 192, 180, 229, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 196, 255, 255, 114, 0, 0, 0, 174, 255, 255, 60, 0, 0],
            &[0, 0, 234, 255, 255, 18, 0, 0, 7, 233, 255, 255, 60, 0, 0],
            &[
                0, 0, 210, 255, 255, 160, 67, 85, 199, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 255, 255, 253, 210, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 7, 151, 253, 255, 255, 215, 76, 61, 255, 255, 60, 0, 0],
            &[0, 0, 0, 0, 9, 43, 29, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ä' / 0xe4
        'ä' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 34, 186, 184, 34, 0, 49, 193, 176, 23, 0, 0, 0],
            &[0, 0, 0, 117, 255, 255, 122, 0, 144, 255, 255, 100, 0, 0, 0],
            &[0, 0, 0, 64, 251, 250, 65, 0, 85, 253, 246, 47, 0, 0, 0],
            &[0, 0, 0, 0, 23, 22, 0, 0, 0, 27, 18, 0, 0, 0, 0],
            &[0, 0, 0, 12, 83, 137, 164, 167, 146, 89, 9, 0, 0, 0, 0],
            &[
                0, 0, 21, 240, 255, 255, 255, 255, 255, 255, 225, 45, 0, 0, 0,
            ],
            &[
                0, 0, 0, 182, 255, 240, 209, 205, 247, 255, 255, 212, 0, 0, 0,
            ],
            &[0, 0, 0, 40, 56, 2, 0, 0, 25, 230, 255, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 24, 30, 184, 255, 255, 58, 0, 0],
            &[
                0, 0, 0, 62, 180, 240, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 75, 252, 255, 255, 229, 192, 180, 229, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 196, 255, 255, 114, 0, 0, 0, 174, 255, 255, 60, 0, 0],
            &[0, 0, 234, 255, 255, 18, 0, 0, 7, 233, 255, 255, 60, 0, 0],
            &[
                0, 0, 210, 255, 255, 160, 67, 85, 199, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 255, 255, 253, 210, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 7, 151, 253, 255, 255, 215, 76, 61, 255, 255, 60, 0, 0],
            &[0, 0, 0, 0, 9, 43, 29, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'å' / 0xe5
        'å' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 93, 131, 78, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 161, 255, 248, 255, 131, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 249, 162, 4, 191, 230, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 238, 196, 68, 219, 214, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 100, 250, 255, 243, 76, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 59, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 83, 137, 164, 167, 146, 89, 9, 0, 0, 0, 0],
            &[
                0, 0, 21, 240, 255, 255, 255, 255, 255, 255, 225, 45, 0, 0, 0,
            ],
            &[
                0, 0, 0, 182, 255, 240, 209, 205, 247, 255, 255, 212, 0, 0, 0,
            ],
            &[0, 0, 0, 40, 56, 2, 0, 0, 25, 230, 255, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 24, 30, 184, 255, 255, 58, 0, 0],
            &[
                0, 0, 0, 62, 180, 240, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 75, 252, 255, 255, 229, 192, 180, 229, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 196, 255, 255, 114, 0, 0, 0, 174, 255, 255, 60, 0, 0],
            &[0, 0, 234, 255, 255, 18, 0, 0, 7, 233, 255, 255, 60, 0, 0],
            &[
                0, 0, 210, 255, 255, 160, 67, 85, 199, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 255, 255, 253, 210, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 7, 151, 253, 255, 255, 215, 76, 61, 255, 255, 60, 0, 0],
            &[0, 0, 0, 0, 9, 43, 29, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'æ' / 0xe6
        'æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 37, 124, 160, 145, 77, 0, 74, 149, 152, 62, 0, 0, 0],
            &[
                0, 16, 246, 255, 255, 255, 255, 212, 255, 255, 255, 254, 81, 0, 0,
            ],
            &[
                0, 0, 154, 236, 202, 241, 255, 255, 255, 223, 249, 255, 220, 1, 0,
            ],
            &[0, 0, 13, 4, 0, 47, 255, 255, 228, 7, 107, 255, 255, 42, 0],
            &[0, 0, 0, 0, 0, 0, 246, 255, 175, 0, 53, 255, 255, 82, 0],
            &[
                0, 0, 79, 196, 240, 254, 255, 255, 251, 243, 245, 255, 255, 97, 0,
            ],
            &[
                0, 65, 254, 255, 238, 189, 253, 255, 253, 251, 251, 251, 251, 97, 0,
            ],
            &[0, 164, 255, 253, 36, 0, 247, 255, 162, 0, 0, 0, 0, 0, 0],
            &[0, 195, 255, 236, 0, 15, 255, 255, 223, 4, 0, 0, 0, 0, 0],
            &[
                0, 176, 255, 255, 85, 138, 255, 255, 255, 165, 56, 51, 134, 0, 0,
            ],
            &[
                0, 114, 255, 255, 255, 255, 233, 198, 255, 255, 255, 255, 232, 0, 0,
            ],
            &[
                0, 9, 180, 255, 255, 225, 55, 18, 170, 252, 255, 254, 155, 0, 0,
            ],
            &[0, 0, 0, 24, 41, 4, 0, 0, 0, 18, 45, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ç' / 0xe7
        'ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 121, 154, 165, 144, 98, 23, 0, 0, 0],
            &[0, 0, 0, 4, 147, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0],
            &[0, 0, 0, 130, 255, 255, 255, 252, 238, 254, 255, 98, 0, 0, 0],
            &[0, 0, 13, 248, 255, 255, 148, 14, 0, 10, 76, 13, 0, 0, 0],
            &[0, 0, 83, 255, 255, 221, 3, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 115, 255, 255, 160, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 129, 255, 255, 148, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 255, 178, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 255, 255, 246, 35, 0, 0, 0, 1, 27, 0, 0, 0],
            &[
                0, 0, 5, 232, 255, 255, 232, 131, 103, 133, 215, 144, 0, 0, 0,
            ],
            &[0, 0, 0, 79, 252, 255, 255, 255, 255, 255, 255, 144, 0, 0, 0],
            &[0, 0, 0, 0, 62, 192, 253, 255, 255, 255, 204, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 255, 208, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 81, 236, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 34, 255, 255, 35, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'è' / 0xe8
        'è' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 22, 22, 6, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 167, 255, 255, 163, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 3, 164, 255, 255, 73, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 143, 255, 231, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 57, 70, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 87, 144, 165, 144, 79, 4, 0, 0, 0, 0],
            &[0, 0, 0, 42, 219, 255, 255, 255, 255, 255, 202, 25, 0, 0, 0],
            &[
                0, 0, 12, 224, 255, 255, 227, 180, 230, 255, 255, 189, 0, 0, 0,
            ],
            &[0, 0, 115, 255, 255, 182, 8, 0, 7, 190, 255, 255, 49, 0, 0],
            &[0, 0, 193, 255, 255, 54, 0, 0, 0, 80, 255, 255, 112, 0, 0],
            &[
                0, 0, 226, 255, 255, 239, 237, 237, 237, 241, 255, 255, 135, 0, 0,
            ],
            &[
                0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 208, 255, 255, 58, 38, 38, 38, 38, 38, 38, 20, 0, 0],
            &[0, 0, 146, 255, 255, 137, 0, 0, 0, 0, 0, 0, 1, 0, 0],
            &[
                0, 0, 43, 249, 255, 255, 185, 103, 79, 102, 144, 218, 54, 0, 0,
            ],
            &[
                0, 0, 0, 100, 252, 255, 255, 255, 255, 255, 255, 255, 54, 0, 0,
            ],
            &[0, 0, 0, 0, 56, 179, 249, 255, 255, 255, 245, 174, 23, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 33, 47, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'é' / 0xe9
        'é' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 19, 22, 22, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 57, 254, 255, 234, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 214, 255, 230, 44, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 147, 255, 215, 33, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 65, 70, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 87, 144, 165, 144, 79, 4, 0, 0, 0, 0],
            &[0, 0, 0, 42, 219, 255, 255, 255, 255, 255, 202, 25, 0, 0, 0],
            &[
                0, 0, 12, 224, 255, 255, 227, 180, 230, 255, 255, 189, 0, 0, 0,
            ],
            &[0, 0, 115, 255, 255, 182, 8, 0, 7, 190, 255, 255, 49, 0, 0],
            &[0, 0, 193, 255, 255, 54, 0, 0, 0, 80, 255, 255, 112, 0, 0],
            &[
                0, 0, 226, 255, 255, 239, 237, 237, 237, 241, 255, 255, 135, 0, 0,
            ],
            &[
                0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 208, 255, 255, 58, 38, 38, 38, 38, 38, 38, 20, 0, 0],
            &[0, 0, 146, 255, 255, 137, 0, 0, 0, 0, 0, 0, 1, 0, 0],
            &[
                0, 0, 43, 249, 255, 255, 185, 103, 79, 102, 144, 218, 54, 0, 0,
            ],
            &[
                0, 0, 0, 100, 252, 255, 255, 255, 255, 255, 255, 255, 54, 0, 0,
            ],
            &[0, 0, 0, 0, 56, 179, 249, 255, 255, 255, 245, 174, 23, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 33, 47, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ê' / 0xea
        'ê' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 22, 22, 22, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 102, 255, 255, 255, 114, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 243, 255, 193, 255, 248, 49, 0, 0, 0, 0],
            &[0, 0, 0, 16, 215, 254, 122, 0, 116, 253, 222, 19, 0, 0, 0],
            &[0, 0, 0, 23, 70, 49, 0, 0, 0, 46, 70, 24, 0, 0, 0],
            &[0, 0, 0, 0, 6, 87, 144, 165, 144, 79, 4, 0, 0, 0, 0],
            &[0, 0, 0, 42, 219, 255, 255, 255, 255, 255, 202, 25, 0, 0, 0],
            &[
                0, 0, 12, 224, 255, 255, 227, 180, 230, 255, 255, 189, 0, 0, 0,
            ],
            &[0, 0, 115, 255, 255, 182, 8, 0, 7, 190, 255, 255, 49, 0, 0],
            &[0, 0, 193, 255, 255, 54, 0, 0, 0, 80, 255, 255, 112, 0, 0],
            &[
                0, 0, 226, 255, 255, 239, 237, 237, 237, 241, 255, 255, 135, 0, 0,
            ],
            &[
                0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 208, 255, 255, 58, 38, 38, 38, 38, 38, 38, 20, 0, 0],
            &[0, 0, 146, 255, 255, 137, 0, 0, 0, 0, 0, 0, 1, 0, 0],
            &[
                0, 0, 43, 249, 255, 255, 185, 103, 79, 102, 144, 218, 54, 0, 0,
            ],
            &[
                0, 0, 0, 100, 252, 255, 255, 255, 255, 255, 255, 255, 54, 0, 0,
            ],
            &[0, 0, 0, 0, 56, 179, 249, 255, 255, 255, 245, 174, 23, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 33, 47, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ë' / 0xeb
        'ë' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 182, 187, 40, 0, 43, 190, 180, 28, 0, 0, 0],
            &[0, 0, 0, 106, 255, 255, 133, 0, 134, 255, 255, 111, 0, 0, 0],
            &[0, 0, 0, 55, 249, 252, 74, 0, 75, 252, 248, 56, 0, 0, 0],
            &[0, 0, 0, 0, 21, 24, 0, 0, 0, 25, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 87, 144, 165, 144, 79, 4, 0, 0, 0, 0],
            &[0, 0, 0, 42, 219, 255, 255, 255, 255, 255, 202, 25, 0, 0, 0],
            &[
                0, 0, 12, 224, 255, 255, 227, 180, 230, 255, 255, 189, 0, 0, 0,
            ],
            &[0, 0, 115, 255, 255, 182, 8, 0, 7, 190, 255, 255, 49, 0, 0],
            &[0, 0, 193, 255, 255, 54, 0, 0, 0, 80, 255, 255, 112, 0, 0],
            &[
                0, 0, 226, 255, 255, 239, 237, 237, 237, 241, 255, 255, 135, 0, 0,
            ],
            &[
                0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 208, 255, 255, 58, 38, 38, 38, 38, 38, 38, 20, 0, 0],
            &[0, 0, 146, 255, 255, 137, 0, 0, 0, 0, 0, 0, 1, 0, 0],
            &[
                0, 0, 43, 249, 255, 255, 185, 103, 79, 102, 144, 218, 54, 0, 0,
            ],
            &[
                0, 0, 0, 100, 252, 255, 255, 255, 255, 255, 255, 255, 54, 0, 0,
            ],
            &[0, 0, 0, 0, 56, 179, 249, 255, 255, 255, 245, 174, 23, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 33, 47, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ì' / 0xec
        'ì' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 22, 22, 22, 2, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 207, 255, 255, 109, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 203, 255, 242, 32, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 186, 255, 195, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 68, 70, 10, 0, 0, 0, 0, 0],
            &[0, 0, 20, 119, 119, 119, 119, 119, 119, 15, 0, 0, 0, 0, 0],
            &[0, 0, 44, 255, 255, 255, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 31, 197, 225, 251, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 235, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 66, 108, 245, 255, 255, 130, 68, 21, 0, 0, 0],
            &[
                0, 0, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'í' / 0xed
        'í' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 22, 22, 22, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 109, 255, 255, 206, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 243, 255, 201, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 197, 255, 183, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 70, 68, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 119, 119, 119, 119, 119, 119, 15, 0, 0, 0, 0, 0],
            &[0, 0, 44, 255, 255, 255, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 31, 197, 225, 251, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 235, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 66, 108, 245, 255, 255, 130, 68, 21, 0, 0, 0],
            &[
                0, 0, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'î' / 0xee
        'î' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 22, 22, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 156, 255, 255, 254, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 255, 250, 198, 255, 225, 18, 0, 0, 0, 0],
            &[0, 0, 0, 44, 241, 244, 78, 4, 163, 255, 185, 3, 0, 0, 0],
            &[0, 0, 0, 38, 70, 34, 0, 0, 0, 61, 70, 10, 0, 0, 0],
            &[0, 0, 20, 119, 119, 119, 119, 119, 119, 15, 0, 0, 0, 0, 0],
            &[0, 0, 44, 255, 255, 255, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 31, 197, 225, 251, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 235, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 66, 108, 245, 255, 255, 130, 68, 21, 0, 0, 0],
            &[
                0, 0, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ï' / 0xef
        'ï' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 198, 168, 13, 0, 76, 202, 157, 6, 0, 0, 0],
            &[0, 0, 0, 160, 255, 255, 79, 0, 187, 255, 255, 57, 0, 0, 0],
            &[0, 0, 0, 102, 255, 240, 32, 0, 126, 255, 231, 18, 0, 0, 0],
            &[0, 0, 0, 0, 31, 15, 0, 0, 0, 34, 11, 0, 0, 0, 0],
            &[0, 0, 20, 119, 119, 119, 119, 119, 119, 15, 0, 0, 0, 0, 0],
            &[0, 0, 44, 255, 255, 255, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 31, 197, 225, 251, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 235, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 66, 108, 245, 255, 255, 130, 68, 21, 0, 0, 0],
            &[
                0, 0, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ð' / 0xf0
        'ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 209, 138, 19, 7, 125, 185, 1, 0, 0, 0],
            &[0, 0, 0, 0, 92, 255, 255, 255, 255, 255, 242, 45, 0, 0, 0],
            &[0, 0, 0, 0, 0, 117, 255, 255, 255, 255, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 100, 242, 255, 255, 255, 255, 101, 0, 0, 0, 0],
            &[0, 0, 0, 0, 91, 247, 125, 28, 218, 255, 249, 40, 0, 0, 0],
            &[0, 0, 0, 0, 0, 40, 40, 29, 41, 251, 255, 179, 0, 0, 0],
            &[
                0, 0, 0, 35, 181, 253, 255, 255, 220, 207, 255, 253, 32, 0, 0,
            ],
            &[
                0, 0, 30, 237, 255, 255, 255, 255, 255, 255, 255, 255, 107, 0, 0,
            ],
            &[
                0, 0, 167, 255, 255, 186, 40, 19, 101, 252, 255, 255, 160, 0, 0,
            ],
            &[0, 4, 246, 255, 255, 37, 0, 0, 0, 144, 255, 255, 183, 0, 0],
            &[0, 28, 255, 255, 248, 2, 0, 0, 0, 88, 255, 255, 187, 0, 0],
            &[0, 26, 255, 255, 248, 2, 0, 0, 0, 96, 255, 255, 165, 0, 0],
            &[0, 2, 243, 255, 255, 44, 0, 0, 0, 155, 255, 255, 108, 0, 0],
            &[
                0, 0, 153, 255, 255, 195, 39, 14, 93, 249, 255, 242, 18, 0, 0,
            ],
            &[
                0, 0, 18, 220, 255, 255, 255, 255, 255, 255, 252, 85, 0, 0, 0,
            ],
            &[0, 0, 0, 17, 152, 241, 255, 255, 254, 199, 64, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 6, 35, 46, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ñ' / 0xf1
        'ñ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 200, 177, 73, 0, 67, 224, 67, 0, 0, 0],
            &[0, 0, 0, 41, 254, 255, 255, 255, 228, 243, 252, 29, 0, 0, 0],
            &[0, 0, 0, 121, 255, 98, 100, 209, 255, 255, 138, 0, 0, 0, 0],
            &[0, 0, 0, 43, 72, 0, 0, 0, 43, 43, 0, 0, 0, 0, 0],
            &[0, 0, 100, 119, 64, 0, 44, 134, 165, 150, 75, 0, 0, 0, 0],
            &[
                0, 0, 216, 255, 203, 102, 254, 255, 255, 255, 255, 146, 0, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 255, 255, 229, 251, 255, 255, 255, 49, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 237, 47, 0, 34, 236, 255, 255, 117, 0, 0,
            ],
            &[0, 0, 216, 255, 255, 135, 0, 0, 0, 152, 255, 255, 138, 0, 0],
            &[0, 0, 216, 255, 255, 76, 0, 0, 0, 128, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 55, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ò' / 0xf2
        'ò' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 22, 22, 22, 2, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 207, 255, 255, 109, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 203, 255, 242, 32, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 186, 255, 195, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 68, 70, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 97, 148, 164, 138, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 59, 231, 255, 255, 255, 255, 255, 187, 18, 0, 0, 0],
            &[
                0, 0, 31, 242, 255, 255, 243, 211, 250, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 158, 255, 255, 190, 10, 0, 43, 241, 255, 255, 62, 0, 0],
            &[0, 1, 239, 255, 255, 54, 0, 0, 0, 147, 255, 255, 142, 0, 0],
            &[0, 20, 255, 255, 253, 4, 0, 0, 0, 97, 255, 255, 175, 0, 0],
            &[0, 35, 255, 255, 242, 0, 0, 0, 0, 88, 255, 255, 188, 0, 0],
            &[0, 15, 255, 255, 254, 14, 0, 0, 0, 114, 255, 255, 167, 0, 0],
            &[0, 0, 215, 255, 255, 98, 0, 0, 0, 193, 255, 255, 111, 0, 0],
            &[
                0, 0, 112, 255, 255, 243, 112, 73, 158, 255, 255, 244, 20, 0, 0,
            ],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 253, 93, 0, 0, 0],
            &[0, 0, 0, 7, 132, 236, 255, 255, 255, 205, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 35, 47, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ó' / 0xf3
        'ó' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 22, 22, 22, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 109, 255, 255, 206, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 243, 255, 201, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 197, 255, 183, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 70, 68, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 97, 148, 164, 138, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 59, 231, 255, 255, 255, 255, 255, 187, 18, 0, 0, 0],
            &[
                0, 0, 31, 242, 255, 255, 243, 211, 250, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 158, 255, 255, 190, 10, 0, 43, 241, 255, 255, 62, 0, 0],
            &[0, 1, 239, 255, 255, 54, 0, 0, 0, 147, 255, 255, 142, 0, 0],
            &[0, 20, 255, 255, 253, 4, 0, 0, 0, 97, 255, 255, 175, 0, 0],
            &[0, 35, 255, 255, 242, 0, 0, 0, 0, 88, 255, 255, 188, 0, 0],
            &[0, 15, 255, 255, 254, 14, 0, 0, 0, 114, 255, 255, 167, 0, 0],
            &[0, 0, 215, 255, 255, 98, 0, 0, 0, 193, 255, 255, 111, 0, 0],
            &[
                0, 0, 112, 255, 255, 243, 112, 73, 158, 255, 255, 244, 20, 0, 0,
            ],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 253, 93, 0, 0, 0],
            &[0, 0, 0, 7, 132, 236, 255, 255, 255, 205, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 35, 47, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ô' / 0xf4
        'ô' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 22, 22, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 156, 255, 255, 254, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 255, 250, 198, 255, 225, 18, 0, 0, 0, 0],
            &[0, 0, 0, 44, 241, 244, 78, 4, 163, 255, 185, 3, 0, 0, 0],
            &[0, 0, 0, 38, 70, 34, 0, 0, 0, 61, 70, 10, 0, 0, 0],
            &[0, 0, 0, 0, 10, 97, 148, 164, 138, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 59, 231, 255, 255, 255, 255, 255, 187, 18, 0, 0, 0],
            &[
                0, 0, 31, 242, 255, 255, 243, 211, 250, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 158, 255, 255, 190, 10, 0, 43, 241, 255, 255, 62, 0, 0],
            &[0, 1, 239, 255, 255, 54, 0, 0, 0, 147, 255, 255, 142, 0, 0],
            &[0, 20, 255, 255, 253, 4, 0, 0, 0, 97, 255, 255, 175, 0, 0],
            &[0, 35, 255, 255, 242, 0, 0, 0, 0, 88, 255, 255, 188, 0, 0],
            &[0, 15, 255, 255, 254, 14, 0, 0, 0, 114, 255, 255, 167, 0, 0],
            &[0, 0, 215, 255, 255, 98, 0, 0, 0, 193, 255, 255, 111, 0, 0],
            &[
                0, 0, 112, 255, 255, 243, 112, 73, 158, 255, 255, 244, 20, 0, 0,
            ],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 253, 93, 0, 0, 0],
            &[0, 0, 0, 7, 132, 236, 255, 255, 255, 205, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 35, 47, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'õ' / 0xf5
        'õ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 203, 170, 63, 0, 86, 224, 48, 0, 0, 0],
            &[0, 0, 0, 61, 255, 255, 255, 255, 224, 247, 246, 14, 0, 0, 0],
            &[0, 0, 0, 143, 253, 84, 108, 217, 255, 255, 117, 0, 0, 0, 0],
            &[0, 0, 0, 50, 65, 0, 0, 1, 47, 37, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 97, 148, 164, 138, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 59, 231, 255, 255, 255, 255, 255, 187, 18, 0, 0, 0],
            &[
                0, 0, 31, 242, 255, 255, 243, 211, 250, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 158, 255, 255, 190, 10, 0, 43, 241, 255, 255, 62, 0, 0],
            &[0, 1, 239, 255, 255, 54, 0, 0, 0, 147, 255, 255, 142, 0, 0],
            &[0, 20, 255, 255, 253, 4, 0, 0, 0, 97, 255, 255, 175, 0, 0],
            &[0, 35, 255, 255, 242, 0, 0, 0, 0, 88, 255, 255, 188, 0, 0],
            &[0, 15, 255, 255, 254, 14, 0, 0, 0, 114, 255, 255, 167, 0, 0],
            &[0, 0, 215, 255, 255, 98, 0, 0, 0, 193, 255, 255, 111, 0, 0],
            &[
                0, 0, 112, 255, 255, 243, 112, 73, 158, 255, 255, 244, 20, 0, 0,
            ],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 253, 93, 0, 0, 0],
            &[0, 0, 0, 7, 132, 236, 255, 255, 255, 205, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 35, 47, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ö' / 0xf6
        'ö' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 198, 168, 13, 0, 76, 202, 157, 6, 0, 0, 0],
            &[0, 0, 0, 160, 255, 255, 79, 0, 187, 255, 255, 57, 0, 0, 0],
            &[0, 0, 0, 102, 255, 240, 32, 0, 126, 255, 231, 18, 0, 0, 0],
            &[0, 0, 0, 0, 31, 15, 0, 0, 0, 34, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 97, 148, 164, 138, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 59, 231, 255, 255, 255, 255, 255, 187, 18, 0, 0, 0],
            &[
                0, 0, 31, 242, 255, 255, 243, 211, 250, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 158, 255, 255, 190, 10, 0, 43, 241, 255, 255, 62, 0, 0],
            &[0, 1, 239, 255, 255, 54, 0, 0, 0, 147, 255, 255, 142, 0, 0],
            &[0, 20, 255, 255, 253, 4, 0, 0, 0, 97, 255, 255, 175, 0, 0],
            &[0, 35, 255, 255, 242, 0, 0, 0, 0, 88, 255, 255, 188, 0, 0],
            &[0, 15, 255, 255, 254, 14, 0, 0, 0, 114, 255, 255, 167, 0, 0],
            &[0, 0, 215, 255, 255, 98, 0, 0, 0, 193, 255, 255, 111, 0, 0],
            &[
                0, 0, 112, 255, 255, 243, 112, 73, 158, 255, 255, 244, 20, 0, 0,
            ],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 253, 93, 0, 0, 0],
            &[0, 0, 0, 7, 132, 236, 255, 255, 255, 205, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 35, 47, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '÷' / 0xf7
        '÷' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 107, 169, 50, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 255, 255, 173, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 237, 255, 146, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 70, 4, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 71, 134, 134, 134, 134, 134, 134, 134, 134, 134, 17, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 94, 179, 179, 179, 179, 179, 179, 179, 179, 179, 23, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 2, 27, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 216, 255, 123, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 17, 255, 255, 176, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 151, 217, 77, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ø' / 0xf8
        'ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 97, 148, 164, 138, 72, 202, 108, 0, 0, 0],
            &[0, 0, 0, 59, 231, 255, 255, 255, 255, 255, 255, 208, 1, 0, 0],
            &[
                0, 0, 31, 242, 255, 255, 243, 211, 255, 255, 255, 213, 0, 0, 0,
            ],
            &[
                0, 0, 158, 255, 255, 190, 10, 8, 255, 255, 255, 255, 62, 0, 0,
            ],
            &[
                0, 1, 239, 255, 255, 54, 0, 142, 255, 255, 255, 255, 142, 0, 0,
            ],
            &[
                0, 20, 255, 255, 253, 4, 61, 253, 237, 122, 255, 255, 175, 0, 0,
            ],
            &[
                0, 35, 255, 255, 242, 11, 220, 255, 89, 88, 255, 255, 188, 0, 0,
            ],
            &[
                0, 15, 255, 255, 254, 164, 255, 171, 0, 114, 255, 255, 167, 0, 0,
            ],
            &[
                0, 0, 215, 255, 255, 255, 233, 20, 0, 193, 255, 255, 111, 0, 0,
            ],
            &[
                0, 0, 112, 255, 255, 255, 192, 73, 158, 255, 255, 244, 20, 0, 0,
            ],
            &[0, 0, 4, 255, 255, 255, 255, 255, 255, 255, 253, 93, 0, 0, 0],
            &[0, 0, 54, 255, 255, 252, 255, 255, 255, 205, 71, 0, 0, 0, 0],
            &[0, 0, 0, 95, 69, 2, 35, 47, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ù' / 0xf9
        'ù' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 22, 22, 22, 2, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 14, 207, 255, 255, 109, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 203, 255, 242, 32, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 186, 255, 195, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 68, 70, 10, 0, 0, 0, 0, 0],
            &[0, 0, 113, 119, 119, 11, 0, 0, 0, 71, 119, 119, 53, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 168, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 31, 0, 0, 0, 194, 255, 255, 113, 0, 0],
            &[0, 0, 234, 255, 255, 75, 0, 0, 20, 246, 255, 255, 113, 0, 0],
            &[
                0, 0, 194, 255, 255, 222, 100, 99, 207, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 198, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 105, 241, 255, 255, 233, 109, 16, 249, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 35, 33, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ú' / 0xfa
        'ú' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 22, 22, 22, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 109, 255, 255, 206, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 243, 255, 201, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 197, 255, 183, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 70, 68, 3, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 119, 119, 11, 0, 0, 0, 71, 119, 119, 53, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 168, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 31, 0, 0, 0, 194, 255, 255, 113, 0, 0],
            &[0, 0, 234, 255, 255, 75, 0, 0, 20, 246, 255, 255, 113, 0, 0],
            &[
                0, 0, 194, 255, 255, 222, 100, 99, 207, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 198, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 105, 241, 255, 255, 233, 109, 16, 249, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 35, 33, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'û' / 0xfb
        'û' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 22, 22, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 156, 255, 255, 254, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 255, 250, 198, 255, 225, 18, 0, 0, 0, 0],
            &[0, 0, 0, 44, 241, 244, 78, 4, 163, 255, 185, 3, 0, 0, 0],
            &[0, 0, 0, 38, 70, 34, 0, 0, 0, 61, 70, 10, 0, 0, 0],
            &[0, 0, 113, 119, 119, 11, 0, 0, 0, 71, 119, 119, 53, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 168, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 31, 0, 0, 0, 194, 255, 255, 113, 0, 0],
            &[0, 0, 234, 255, 255, 75, 0, 0, 20, 246, 255, 255, 113, 0, 0],
            &[
                0, 0, 194, 255, 255, 222, 100, 99, 207, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 198, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 105, 241, 255, 255, 233, 109, 16, 249, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 35, 33, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ü' / 0xfc
        'ü' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 198, 168, 13, 0, 76, 202, 157, 6, 0, 0, 0],
            &[0, 0, 0, 160, 255, 255, 79, 0, 187, 255, 255, 57, 0, 0, 0],
            &[0, 0, 0, 102, 255, 240, 32, 0, 126, 255, 231, 18, 0, 0, 0],
            &[0, 0, 0, 0, 31, 15, 0, 0, 0, 34, 11, 0, 0, 0, 0],
            &[0, 0, 113, 119, 119, 11, 0, 0, 0, 71, 119, 119, 53, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 168, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 31, 0, 0, 0, 194, 255, 255, 113, 0, 0],
            &[0, 0, 234, 255, 255, 75, 0, 0, 20, 246, 255, 255, 113, 0, 0],
            &[
                0, 0, 194, 255, 255, 222, 100, 99, 207, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 198, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 105, 241, 255, 255, 233, 109, 16, 249, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 35, 33, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ý' / 0xfd
        'ý' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 22, 22, 22, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 109, 255, 255, 206, 14, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 33, 243, 255, 201, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 197, 255, 183, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 70, 68, 3, 0, 0, 0, 0, 0, 0],
            &[0, 58, 119, 119, 97, 0, 0, 0, 0, 19, 119, 119, 119, 11, 0],
            &[0, 50, 255, 255, 251, 19, 0, 0, 0, 97, 255, 255, 212, 0, 0],
            &[0, 0, 204, 255, 255, 101, 0, 0, 0, 173, 255, 255, 119, 0, 0],
            &[0, 0, 103, 255, 255, 186, 0, 0, 6, 242, 255, 253, 28, 0, 0],
            &[0, 0, 13, 244, 255, 251, 20, 0, 69, 255, 255, 188, 0, 0, 0],
            &[0, 0, 0, 156, 255, 255, 102, 0, 145, 255, 255, 95, 0, 0, 0],
            &[0, 0, 0, 54, 255, 255, 187, 0, 220, 255, 245, 13, 0, 0, 0],
            &[0, 0, 0, 0, 208, 255, 251, 56, 255, 255, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 255, 255, 178, 255, 255, 72, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 246, 255, 253, 255, 231, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 160, 255, 255, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 255, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 255, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 198, 255, 255, 105, 0, 0, 0, 0, 0, 0],
            &[0, 0, 110, 141, 210, 255, 255, 224, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 32, 17, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'þ' / 0xfe
        'þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 207, 245, 245, 33, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 35, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 35, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 35, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 32, 39, 134, 164, 138, 42, 0, 0, 0, 0],
            &[
                0, 0, 216, 255, 255, 113, 253, 255, 255, 255, 248, 77, 0, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 255, 253, 229, 255, 255, 255, 235, 12, 0, 0,
            ],
            &[0, 0, 216, 255, 255, 218, 30, 0, 51, 243, 255, 255, 94, 0, 0],
            &[0, 0, 216, 255, 255, 93, 0, 0, 0, 159, 255, 255, 154, 0, 0],
            &[0, 0, 216, 255, 255, 47, 0, 0, 0, 115, 255, 255, 184, 0, 0],
            &[0, 0, 216, 255, 255, 42, 0, 0, 0, 106, 255, 255, 188, 0, 0],
            &[0, 0, 216, 255, 255, 69, 0, 0, 0, 128, 255, 255, 169, 0, 0],
            &[0, 0, 216, 255, 255, 155, 0, 0, 0, 199, 255, 255, 124, 0, 0],
            &[
                0, 0, 216, 255, 255, 255, 151, 93, 165, 255, 255, 254, 45, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 237, 255, 255, 255, 255, 255, 161, 0, 0, 0,
            ],
            &[0, 0, 216, 255, 255, 55, 190, 255, 255, 251, 153, 6, 0, 0, 0],
            &[0, 0, 216, 255, 255, 34, 0, 21, 44, 13, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 35, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 35, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 8, 10, 10, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ÿ' / 0xff
        'ÿ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 60, 198, 168, 13, 0, 76, 202, 157, 6, 0, 0, 0],
            &[0, 0, 0, 160, 255, 255, 79, 0, 187, 255, 255, 57, 0, 0, 0],
            &[0, 0, 0, 102, 255, 240, 32, 0, 126, 255, 231, 18, 0, 0, 0],
            &[0, 0, 0, 0, 31, 15, 0, 0, 0, 34, 11, 0, 0, 0, 0],
            &[0, 58, 119, 119, 97, 0, 0, 0, 0, 19, 119, 119, 119, 11, 0],
            &[0, 50, 255, 255, 251, 19, 0, 0, 0, 97, 255, 255, 212, 0, 0],
            &[0, 0, 204, 255, 255, 101, 0, 0, 0, 173, 255, 255, 119, 0, 0],
            &[0, 0, 103, 255, 255, 186, 0, 0, 6, 242, 255, 253, 28, 0, 0],
            &[0, 0, 13, 244, 255, 251, 20, 0, 69, 255, 255, 188, 0, 0, 0],
            &[0, 0, 0, 156, 255, 255, 102, 0, 145, 255, 255, 95, 0, 0, 0],
            &[0, 0, 0, 54, 255, 255, 187, 0, 220, 255, 245, 13, 0, 0, 0],
            &[0, 0, 0, 0, 208, 255, 251, 56, 255, 255, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 255, 255, 178, 255, 255, 72, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 246, 255, 253, 255, 231, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 160, 255, 255, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 255, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 255, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 198, 255, 255, 105, 0, 0, 0, 0, 0, 0],
            &[0, 0, 110, 141, 210, 255, 255, 224, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 32, 17, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ā' / 0x100
        'Ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 152, 171, 171, 171, 171, 171, 80, 0, 0, 0, 0],
            &[0, 0, 0, 0, 228, 255, 255, 255, 255, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 104, 104, 104, 104, 104, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 14, 14, 14, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 255, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 254, 255, 255, 255, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 255, 223, 255, 237, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 251, 116, 255, 255, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 230, 255, 201, 46, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 255, 138, 2, 239, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 255, 74, 0, 180, 255, 251, 15, 0, 0, 0],
            &[0, 0, 0, 186, 255, 252, 14, 0, 118, 255, 255, 81, 0, 0, 0],
            &[0, 0, 9, 247, 255, 255, 60, 60, 117, 255, 255, 152, 0, 0, 0],
            &[
                0, 0, 71, 255, 255, 255, 255, 255, 255, 255, 255, 222, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 211, 255, 255, 87, 72, 72, 72, 182, 255, 255, 108, 0, 0,
            ],
            &[0, 26, 255, 255, 203, 0, 0, 0, 0, 65, 255, 255, 179, 0, 0],
            &[0, 96, 255, 255, 139, 0, 0, 0, 0, 9, 250, 255, 243, 6, 0],
            &[0, 167, 255, 255, 75, 0, 0, 0, 0, 0, 198, 255, 255, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ā' / 0x101
        'ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 36, 36, 36, 36, 36, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 185, 255, 255, 255, 255, 255, 163, 0, 0, 0, 0],
            &[0, 0, 0, 0, 173, 240, 240, 240, 240, 240, 153, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 83, 137, 164, 167, 146, 89, 9, 0, 0, 0, 0],
            &[
                0, 0, 21, 240, 255, 255, 255, 255, 255, 255, 225, 45, 0, 0, 0,
            ],
            &[
                0, 0, 0, 182, 255, 240, 209, 205, 247, 255, 255, 212, 0, 0, 0,
            ],
            &[0, 0, 0, 40, 56, 2, 0, 0, 25, 230, 255, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 24, 30, 184, 255, 255, 58, 0, 0],
            &[
                0, 0, 0, 62, 180, 240, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 75, 252, 255, 255, 229, 192, 180, 229, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 196, 255, 255, 114, 0, 0, 0, 174, 255, 255, 60, 0, 0],
            &[0, 0, 234, 255, 255, 18, 0, 0, 7, 233, 255, 255, 60, 0, 0],
            &[
                0, 0, 210, 255, 255, 160, 67, 85, 199, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 255, 255, 253, 210, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 7, 151, 253, 255, 255, 215, 76, 61, 255, 255, 60, 0, 0],
            &[0, 0, 0, 0, 9, 43, 29, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ă' / 0x102
        'Ă' => &[
            &[0, 0, 0, 28, 173, 109, 0, 0, 4, 169, 131, 0, 0, 0, 0],
            &[0, 0, 0, 6, 247, 242, 109, 81, 166, 255, 146, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 255, 255, 255, 255, 231, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 159, 177, 132, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 14, 14, 14, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 255, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 254, 255, 255, 255, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 255, 223, 255, 237, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 251, 116, 255, 255, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 230, 255, 201, 46, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 255, 138, 2, 239, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 255, 74, 0, 180, 255, 251, 15, 0, 0, 0],
            &[0, 0, 0, 186, 255, 252, 14, 0, 118, 255, 255, 81, 0, 0, 0],
            &[0, 0, 9, 247, 255, 255, 60, 60, 117, 255, 255, 152, 0, 0, 0],
            &[
                0, 0, 71, 255, 255, 255, 255, 255, 255, 255, 255, 222, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 211, 255, 255, 87, 72, 72, 72, 182, 255, 255, 108, 0, 0,
            ],
            &[0, 26, 255, 255, 203, 0, 0, 0, 0, 65, 255, 255, 179, 0, 0],
            &[0, 96, 255, 255, 139, 0, 0, 0, 0, 9, 250, 255, 243, 6, 0],
            &[0, 167, 255, 255, 75, 0, 0, 0, 0, 0, 198, 255, 255, 64, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ă' / 0x103
        'ă' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 38, 27, 0, 0, 0, 29, 37, 0, 0, 0, 0],
            &[0, 0, 0, 0, 239, 229, 14, 0, 17, 235, 221, 0, 0, 0, 0],
            &[0, 0, 0, 0, 160, 255, 234, 197, 244, 255, 135, 0, 0, 0, 0],
            &[0, 0, 0, 0, 18, 183, 255, 255, 255, 161, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 31, 59, 23, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 83, 137, 164, 167, 146, 89, 9, 0, 0, 0, 0],
            &[
                0, 0, 21, 240, 255, 255, 255, 255, 255, 255, 225, 45, 0, 0, 0,
            ],
            &[
                0, 0, 0, 182, 255, 240, 209, 205, 247, 255, 255, 212, 0, 0, 0,
            ],
            &[0, 0, 0, 40, 56, 2, 0, 0, 25, 230, 255, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 24, 30, 184, 255, 255, 58, 0, 0],
            &[
                0, 0, 0, 62, 180, 240, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 75, 252, 255, 255, 229, 192, 180, 229, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 196, 255, 255, 114, 0, 0, 0, 174, 255, 255, 60, 0, 0],
            &[0, 0, 234, 255, 255, 18, 0, 0, 7, 233, 255, 255, 60, 0, 0],
            &[
                0, 0, 210, 255, 255, 160, 67, 85, 199, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 255, 255, 253, 210, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 7, 151, 253, 255, 255, 215, 76, 61, 255, 255, 60, 0, 0],
            &[0, 0, 0, 0, 9, 43, 29, 1, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ą' / 0x104
        'Ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 9, 14, 14, 14, 3, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 255, 255, 98, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 254, 255, 255, 255, 169, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 255, 255, 223, 255, 237, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 251, 116, 255, 255, 54, 0, 0, 0, 0],
            &[0, 0, 0, 0, 230, 255, 201, 46, 255, 255, 125, 0, 0, 0, 0],
            &[0, 0, 0, 46, 255, 255, 138, 2, 239, 255, 196, 0, 0, 0, 0],
            &[0, 0, 0, 116, 255, 255, 74, 0, 180, 255, 251, 15, 0, 0, 0],
            &[0, 0, 0, 186, 255, 252, 14, 0, 118, 255, 255, 81, 0, 0, 0],
            &[0, 0, 9, 247, 255, 255, 60, 60, 117, 255, 255, 152, 0, 0, 0],
            &[
                0, 0, 71, 255, 255, 255, 255, 255, 255, 255, 255, 222, 0, 0, 0,
            ],
            &[
                0, 0, 141, 255, 255, 255, 255, 255, 255, 255, 255, 255, 37, 0, 0,
            ],
            &[
                0, 0, 211, 255, 255, 87, 72, 72, 72, 182, 255, 255, 108, 0, 0,
            ],
            &[0, 26, 255, 255, 203, 0, 0, 0, 0, 65, 255, 255, 179, 0, 0],
            &[0, 96, 255, 255, 139, 0, 0, 0, 0, 9, 250, 255, 243, 6, 0],
            &[0, 167, 255, 255, 75, 0, 0, 0, 0, 0, 198, 255, 255, 68, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 31, 222, 222, 20, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 186, 255, 72, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 253, 255, 42, 1, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 68, 210, 243, 183, 0],
        ],
        // letter: 'ą' / 0x105
        'ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 83, 137, 164, 167, 146, 89, 9, 0, 0, 0, 0],
            &[
                0, 0, 21, 240, 255, 255, 255, 255, 255, 255, 225, 45, 0, 0, 0,
            ],
            &[
                0, 0, 0, 182, 255, 240, 209, 205, 247, 255, 255, 212, 0, 0, 0,
            ],
            &[0, 0, 0, 40, 56, 2, 0, 0, 25, 230, 255, 255, 36, 0, 0],
            &[0, 0, 0, 0, 0, 0, 11, 24, 30, 184, 255, 255, 58, 0, 0],
            &[
                0, 0, 0, 62, 180, 240, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 75, 252, 255, 255, 229, 192, 180, 229, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 196, 255, 255, 114, 0, 0, 0, 174, 255, 255, 60, 0, 0],
            &[0, 0, 234, 255, 255, 18, 0, 0, 7, 233, 255, 255, 60, 0, 0],
            &[
                0, 0, 210, 255, 255, 160, 67, 85, 199, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 126, 255, 255, 255, 255, 255, 253, 210, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 7, 151, 253, 255, 255, 215, 76, 61, 255, 255, 60, 0, 0],
            &[0, 0, 0, 0, 9, 43, 29, 1, 0, 139, 255, 101, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 72, 255, 186, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 149, 255, 156, 0, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 11, 166, 236, 232, 58, 0, 0],
        ],
        // letter: 'Ć' / 0x106
        'Ć' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 20, 156, 157, 157, 37, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 161, 255, 255, 124, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 81, 255, 254, 111, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 172, 190, 83, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 38, 41, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 85, 197, 252, 255, 255, 255, 209, 88, 0, 0],
            &[
                0, 0, 0, 4, 163, 255, 255, 255, 255, 255, 255, 255, 107, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 188, 131, 132, 177, 239, 15, 0, 0,
            ],
            &[0, 0, 26, 251, 255, 254, 91, 0, 0, 0, 0, 9, 0, 0, 0],
            &[0, 0, 121, 255, 255, 176, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 182, 255, 255, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 221, 255, 255, 63, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 237, 255, 255, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 228, 255, 255, 48, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 207, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 255, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 80, 255, 255, 249, 60, 0, 0, 0, 0, 19, 11, 0, 0],
            &[
                0, 0, 2, 205, 255, 255, 250, 173, 127, 137, 181, 250, 44, 0, 0,
            ],
            &[
                0, 0, 0, 42, 228, 255, 255, 255, 255, 255, 255, 255, 44, 0, 0,
            ],
            &[0, 0, 0, 0, 28, 151, 244, 255, 255, 255, 239, 164, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 27, 46, 25, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ć' / 0x107
        'ć' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 12, 22, 22, 17, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 9, 221, 255, 254, 100, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 142, 255, 253, 101, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 68, 254, 246, 83, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 43, 70, 38, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 121, 154, 165, 144, 98, 23, 0, 0, 0],
            &[0, 0, 0, 4, 147, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0],
            &[0, 0, 0, 130, 255, 255, 255, 252, 238, 254, 255, 98, 0, 0, 0],
            &[0, 0, 13, 248, 255, 255, 148, 14, 0, 10, 76, 13, 0, 0, 0],
            &[0, 0, 83, 255, 255, 221, 3, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 115, 255, 255, 160, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 129, 255, 255, 148, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 255, 178, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 255, 255, 246, 35, 0, 0, 0, 1, 27, 0, 0, 0],
            &[
                0, 0, 5, 232, 255, 255, 232, 131, 103, 133, 215, 144, 0, 0, 0,
            ],
            &[0, 0, 0, 79, 252, 255, 255, 255, 255, 255, 255, 144, 0, 0, 0],
            &[0, 0, 0, 0, 62, 192, 253, 255, 255, 255, 204, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 39, 43, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĉ' / 0x108
        'Ĉ' => &[
            &[0, 0, 0, 0, 0, 0, 46, 157, 157, 157, 29, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 206, 255, 255, 255, 186, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 152, 255, 209, 59, 227, 255, 126, 0, 0, 0],
            &[0, 0, 0, 0, 53, 190, 160, 15, 0, 27, 172, 189, 31, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 38, 41, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 85, 197, 252, 255, 255, 255, 209, 88, 0, 0],
            &[
                0, 0, 0, 4, 163, 255, 255, 255, 255, 255, 255, 255, 107, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 188, 131, 132, 177, 239, 15, 0, 0,
            ],
            &[0, 0, 26, 251, 255, 254, 91, 0, 0, 0, 0, 9, 0, 0, 0],
            &[0, 0, 121, 255, 255, 176, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 182, 255, 255, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 221, 255, 255, 63, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 237, 255, 255, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 228, 255, 255, 48, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 207, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 255, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 80, 255, 255, 249, 60, 0, 0, 0, 0, 19, 11, 0, 0],
            &[
                0, 0, 2, 205, 255, 255, 250, 173, 127, 137, 181, 250, 44, 0, 0,
            ],
            &[
                0, 0, 0, 42, 228, 255, 255, 255, 255, 255, 255, 255, 44, 0, 0,
            ],
            &[0, 0, 0, 0, 28, 151, 244, 255, 255, 255, 239, 164, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 27, 46, 25, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĉ' / 0x109
        'ĉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 16, 22, 22, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 244, 255, 255, 193, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 198, 255, 206, 242, 255, 122, 0, 0, 0, 0],
            &[0, 0, 0, 0, 152, 255, 188, 13, 56, 233, 252, 70, 0, 0, 0],
            &[0, 0, 0, 1, 70, 67, 3, 0, 0, 24, 70, 46, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 121, 154, 165, 144, 98, 23, 0, 0, 0],
            &[0, 0, 0, 4, 147, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0],
            &[0, 0, 0, 130, 255, 255, 255, 252, 238, 254, 255, 98, 0, 0, 0],
            &[0, 0, 13, 248, 255, 255, 148, 14, 0, 10, 76, 13, 0, 0, 0],
            &[0, 0, 83, 255, 255, 221, 3, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 115, 255, 255, 160, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 129, 255, 255, 148, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 255, 178, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 255, 255, 246, 35, 0, 0, 0, 1, 27, 0, 0, 0],
            &[
                0, 0, 5, 232, 255, 255, 232, 131, 103, 133, 215, 144, 0, 0, 0,
            ],
            &[0, 0, 0, 79, 252, 255, 255, 255, 255, 255, 255, 144, 0, 0, 0],
            &[0, 0, 0, 0, 62, 192, 253, 255, 255, 255, 204, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 39, 43, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ċ' / 0x10a
        'Ċ' => &[
            &[0, 0, 0, 0, 0, 0, 8, 156, 201, 137, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 92, 255, 255, 255, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 84, 255, 255, 255, 43, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 128, 170, 101, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 38, 41, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 85, 197, 252, 255, 255, 255, 209, 88, 0, 0],
            &[
                0, 0, 0, 4, 163, 255, 255, 255, 255, 255, 255, 255, 107, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 188, 131, 132, 177, 239, 15, 0, 0,
            ],
            &[0, 0, 26, 251, 255, 254, 91, 0, 0, 0, 0, 9, 0, 0, 0],
            &[0, 0, 121, 255, 255, 176, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 182, 255, 255, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 221, 255, 255, 63, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 237, 255, 255, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 228, 255, 255, 48, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 207, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 255, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 80, 255, 255, 249, 60, 0, 0, 0, 0, 19, 11, 0, 0],
            &[
                0, 0, 2, 205, 255, 255, 250, 173, 127, 137, 181, 250, 44, 0, 0,
            ],
            &[
                0, 0, 0, 42, 228, 255, 255, 255, 255, 255, 255, 255, 44, 0, 0,
            ],
            &[0, 0, 0, 0, 28, 151, 244, 255, 255, 255, 239, 164, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 27, 46, 25, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ċ' / 0x10b
        'ċ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 63, 44, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 195, 255, 255, 102, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 254, 255, 255, 162, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 186, 255, 253, 87, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 3, 47, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 121, 154, 165, 144, 98, 23, 0, 0, 0],
            &[0, 0, 0, 4, 147, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0],
            &[0, 0, 0, 130, 255, 255, 255, 252, 238, 254, 255, 98, 0, 0, 0],
            &[0, 0, 13, 248, 255, 255, 148, 14, 0, 10, 76, 13, 0, 0, 0],
            &[0, 0, 83, 255, 255, 221, 3, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 115, 255, 255, 160, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 129, 255, 255, 148, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 255, 178, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 255, 255, 246, 35, 0, 0, 0, 1, 27, 0, 0, 0],
            &[
                0, 0, 5, 232, 255, 255, 232, 131, 103, 133, 215, 144, 0, 0, 0,
            ],
            &[0, 0, 0, 79, 252, 255, 255, 255, 255, 255, 255, 144, 0, 0, 0],
            &[0, 0, 0, 0, 62, 192, 253, 255, 255, 255, 204, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 39, 43, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Č' / 0x10c
        'Č' => &[
            &[0, 0, 0, 0, 51, 157, 127, 5, 0, 14, 140, 157, 31, 0, 0],
            &[0, 0, 0, 0, 2, 176, 255, 186, 35, 213, 255, 149, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 223, 255, 251, 255, 202, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 65, 190, 190, 190, 41, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 12, 38, 41, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 85, 197, 252, 255, 255, 255, 209, 88, 0, 0],
            &[
                0, 0, 0, 4, 163, 255, 255, 255, 255, 255, 255, 255, 107, 0, 0,
            ],
            &[
                0, 0, 0, 131, 255, 255, 255, 188, 131, 132, 177, 239, 15, 0, 0,
            ],
            &[0, 0, 26, 251, 255, 254, 91, 0, 0, 0, 0, 9, 0, 0, 0],
            &[0, 0, 121, 255, 255, 176, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 182, 255, 255, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 221, 255, 255, 63, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 237, 255, 255, 41, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 228, 255, 255, 48, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 207, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 156, 255, 255, 152, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 80, 255, 255, 249, 60, 0, 0, 0, 0, 19, 11, 0, 0],
            &[
                0, 0, 2, 205, 255, 255, 250, 173, 127, 137, 181, 250, 44, 0, 0,
            ],
            &[
                0, 0, 0, 42, 228, 255, 255, 255, 255, 255, 255, 255, 44, 0, 0,
            ],
            &[0, 0, 0, 0, 28, 151, 244, 255, 255, 255, 239, 164, 20, 0, 0],
            &[0, 0, 0, 0, 0, 0, 2, 27, 46, 25, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'č' / 0x10d
        'č' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 22, 19, 0, 0, 0, 5, 22, 14, 0, 0, 0],
            &[0, 0, 0, 0, 187, 255, 143, 2, 32, 209, 255, 100, 0, 0, 0],
            &[0, 0, 0, 0, 16, 223, 255, 171, 227, 255, 154, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 57, 253, 255, 255, 215, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 55, 70, 70, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 121, 154, 165, 144, 98, 23, 0, 0, 0],
            &[0, 0, 0, 4, 147, 255, 255, 255, 255, 255, 255, 191, 0, 0, 0],
            &[0, 0, 0, 130, 255, 255, 255, 252, 238, 254, 255, 98, 0, 0, 0],
            &[0, 0, 13, 248, 255, 255, 148, 14, 0, 10, 76, 13, 0, 0, 0],
            &[0, 0, 83, 255, 255, 221, 3, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 115, 255, 255, 160, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 129, 255, 255, 148, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 111, 255, 255, 178, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 255, 255, 246, 35, 0, 0, 0, 1, 27, 0, 0, 0],
            &[
                0, 0, 5, 232, 255, 255, 232, 131, 103, 133, 215, 144, 0, 0, 0,
            ],
            &[0, 0, 0, 79, 252, 255, 255, 255, 255, 255, 255, 144, 0, 0, 0],
            &[0, 0, 0, 0, 62, 192, 253, 255, 255, 255, 204, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 13, 39, 43, 18, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ď' / 0x10e
        'Ď' => &[
            &[0, 0, 0, 125, 157, 58, 0, 0, 72, 157, 114, 0, 0, 0, 0],
            &[0, 0, 0, 51, 247, 249, 82, 105, 253, 242, 42, 0, 0, 0, 0],
            &[0, 0, 0, 0, 103, 255, 252, 255, 255, 88, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 154, 190, 190, 142, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 254, 254, 251, 227, 185, 89, 2, 0, 0, 0, 0],
            &[
                0, 14, 255, 255, 255, 255, 255, 255, 255, 255, 184, 15, 0, 0, 0,
            ],
            &[
                0, 14, 255, 255, 250, 150, 155, 193, 254, 255, 255, 166, 0, 0, 0,
            ],
            &[0, 14, 255, 255, 242, 0, 0, 0, 70, 250, 255, 255, 58, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 154, 255, 255, 145, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 78, 255, 255, 205, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 44, 255, 255, 228, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 33, 255, 255, 242, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 45, 255, 255, 222, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 85, 255, 255, 196, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 168, 255, 255, 126, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 90, 254, 255, 251, 38, 0, 0],
            &[
                0, 14, 255, 255, 250, 149, 156, 202, 255, 255, 255, 136, 0, 0, 0,
            ],
            &[
                0, 14, 255, 255, 255, 255, 255, 255, 255, 255, 153, 5, 0, 0, 0,
            ],
            &[0, 14, 255, 255, 255, 255, 249, 221, 171, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ď' / 0x10f
        'ď' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 144, 245, 245, 97, 144, 245, 244, 36],
            &[0, 0, 0, 0, 0, 0, 0, 150, 255, 255, 101, 196, 255, 173, 0],
            &[0, 0, 0, 0, 0, 0, 0, 150, 255, 255, 107, 246, 250, 40, 0],
            &[0, 0, 0, 0, 0, 0, 0, 148, 255, 255, 115, 102, 73, 0, 0],
            &[0, 0, 14, 118, 162, 138, 32, 137, 255, 255, 101, 0, 0, 0, 0],
            &[
                0, 23, 223, 255, 255, 255, 241, 152, 255, 255, 101, 0, 0, 0, 0,
            ],
            &[
                0, 158, 255, 255, 254, 226, 249, 255, 255, 255, 101, 0, 0, 0, 0,
            ],
            &[8, 248, 255, 255, 93, 0, 32, 244, 255, 255, 101, 0, 0, 0, 0],
            &[56, 255, 255, 229, 1, 0, 0, 149, 255, 255, 101, 0, 0, 0, 0],
            &[81, 255, 255, 198, 0, 0, 0, 112, 255, 255, 101, 0, 0, 0, 0],
            &[90, 255, 255, 190, 0, 0, 0, 93, 255, 255, 101, 0, 0, 0, 0],
            &[76, 255, 255, 210, 0, 0, 0, 113, 255, 255, 101, 0, 0, 0, 0],
            &[38, 255, 255, 252, 23, 0, 0, 182, 255, 255, 101, 0, 0, 0, 0],
            &[
                0, 224, 255, 255, 198, 95, 154, 255, 255, 255, 101, 0, 0, 0, 0,
            ],
            &[
                0, 104, 255, 255, 255, 255, 255, 203, 255, 255, 101, 0, 0, 0, 0,
            ],
            &[0, 0, 137, 250, 255, 245, 111, 34, 253, 255, 101, 0, 0, 0, 0],
            &[0, 0, 0, 17, 45, 13, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Đ' / 0x110
        'Đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 254, 254, 251, 227, 185, 89, 2, 0, 0, 0, 0],
            &[
                0, 14, 255, 255, 255, 255, 255, 255, 255, 255, 184, 15, 0, 0, 0,
            ],
            &[
                0, 14, 255, 255, 250, 150, 155, 193, 254, 255, 255, 166, 0, 0, 0,
            ],
            &[0, 14, 255, 255, 242, 0, 0, 0, 70, 250, 255, 255, 58, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 154, 255, 255, 145, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 78, 255, 255, 205, 0, 0],
            &[
                0, 186, 255, 255, 255, 172, 172, 29, 0, 44, 255, 255, 228, 0, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 44, 0, 33, 255, 255, 242, 0, 0,
            ],
            &[
                0, 230, 255, 255, 255, 216, 216, 37, 0, 45, 255, 255, 222, 0, 0,
            ],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 85, 255, 255, 196, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 0, 168, 255, 255, 126, 0, 0],
            &[0, 14, 255, 255, 242, 0, 0, 0, 90, 254, 255, 251, 38, 0, 0],
            &[
                0, 14, 255, 255, 250, 149, 156, 202, 255, 255, 255, 136, 0, 0, 0,
            ],
            &[
                0, 14, 255, 255, 255, 255, 255, 255, 255, 255, 153, 5, 0, 0, 0,
            ],
            &[0, 14, 255, 255, 255, 255, 249, 221, 171, 68, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'đ' / 0x111
        'đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 131, 245, 245, 109, 0, 0],
            &[0, 0, 0, 0, 0, 0, 37, 71, 71, 209, 255, 255, 185, 42, 0],
            &[0, 0, 0, 0, 0, 0, 132, 255, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 92, 177, 177, 255, 255, 255, 255, 106, 0],
            &[0, 0, 0, 0, 25, 90, 109, 71, 2, 127, 255, 255, 113, 0, 0],
            &[
                0, 0, 0, 104, 247, 255, 255, 255, 208, 124, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 61, 254, 255, 255, 255, 255, 255, 245, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 179, 255, 255, 223, 67, 24, 87, 248, 255, 255, 113, 0, 0,
            ],
            &[0, 3, 243, 255, 255, 74, 0, 0, 0, 151, 255, 255, 113, 0, 0],
            &[0, 24, 255, 255, 255, 7, 0, 0, 0, 105, 255, 255, 113, 0, 0],
            &[0, 36, 255, 255, 243, 0, 0, 0, 0, 81, 255, 255, 113, 0, 0],
            &[0, 22, 255, 255, 255, 8, 0, 0, 0, 100, 255, 255, 113, 0, 0],
            &[0, 0, 236, 255, 255, 84, 0, 0, 0, 174, 255, 255, 113, 0, 0],
            &[
                0, 0, 161, 255, 255, 241, 121, 92, 163, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 39, 245, 255, 255, 255, 255, 255, 199, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 65, 222, 255, 255, 235, 101, 25, 250, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 36, 38, 5, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ē' / 0x112
        'Ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 131, 171, 171, 171, 171, 171, 102, 0, 0, 0, 0],
            &[0, 0, 0, 0, 196, 255, 255, 255, 255, 255, 152, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 104, 104, 104, 104, 104, 62, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 254, 254, 254, 254, 254, 254, 254, 254, 254, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 220, 161, 161, 161, 161, 161, 161, 20, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 167, 23, 23, 23, 23, 23, 19, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 210, 136, 136, 136, 136, 136, 110, 0, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 219, 159, 159, 159, 159, 159, 159, 20, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ē' / 0x113
        'ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 24, 36, 36, 36, 36, 36, 24, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 255, 255, 255, 255, 255, 174, 0, 0, 0, 0],
            &[0, 0, 0, 0, 163, 240, 240, 240, 240, 240, 163, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 87, 144, 165, 144, 79, 4, 0, 0, 0, 0],
            &[0, 0, 0, 42, 219, 255, 255, 255, 255, 255, 202, 25, 0, 0, 0],
            &[
                0, 0, 12, 224, 255, 255, 227, 180, 230, 255, 255, 189, 0, 0, 0,
            ],
            &[0, 0, 115, 255, 255, 182, 8, 0, 7, 190, 255, 255, 49, 0, 0],
            &[0, 0, 193, 255, 255, 54, 0, 0, 0, 80, 255, 255, 112, 0, 0],
            &[
                0, 0, 226, 255, 255, 239, 237, 237, 237, 241, 255, 255, 135, 0, 0,
            ],
            &[
                0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 208, 255, 255, 58, 38, 38, 38, 38, 38, 38, 20, 0, 0],
            &[0, 0, 146, 255, 255, 137, 0, 0, 0, 0, 0, 0, 1, 0, 0],
            &[
                0, 0, 43, 249, 255, 255, 185, 103, 79, 102, 144, 218, 54, 0, 0,
            ],
            &[
                0, 0, 0, 100, 252, 255, 255, 255, 255, 255, 255, 255, 54, 0, 0,
            ],
            &[0, 0, 0, 0, 56, 179, 249, 255, 255, 255, 245, 174, 23, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 33, 47, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĕ' / 0x114
        'Ĕ' => &[
            &[0, 0, 0, 7, 172, 130, 0, 0, 0, 152, 153, 0, 0, 0, 0],
            &[0, 0, 0, 0, 221, 251, 123, 77, 147, 255, 179, 0, 0, 0, 0],
            &[0, 0, 0, 0, 84, 254, 255, 255, 255, 243, 43, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 154, 179, 141, 30, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 254, 254, 254, 254, 254, 254, 254, 254, 254, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 220, 161, 161, 161, 161, 161, 161, 20, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 167, 23, 23, 23, 23, 23, 19, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 210, 136, 136, 136, 136, 136, 110, 0, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 219, 159, 159, 159, 159, 159, 159, 20, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĕ' / 0x115
        'ĕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 38, 28, 0, 0, 0, 27, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 229, 236, 18, 0, 13, 229, 232, 0, 0, 0, 0],
            &[0, 0, 0, 0, 149, 255, 236, 197, 243, 255, 145, 0, 0, 0, 0],
            &[0, 0, 0, 0, 13, 176, 255, 255, 255, 167, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 29, 59, 25, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 87, 144, 165, 144, 79, 4, 0, 0, 0, 0],
            &[0, 0, 0, 42, 219, 255, 255, 255, 255, 255, 202, 25, 0, 0, 0],
            &[
                0, 0, 12, 224, 255, 255, 227, 180, 230, 255, 255, 189, 0, 0, 0,
            ],
            &[0, 0, 115, 255, 255, 182, 8, 0, 7, 190, 255, 255, 49, 0, 0],
            &[0, 0, 193, 255, 255, 54, 0, 0, 0, 80, 255, 255, 112, 0, 0],
            &[
                0, 0, 226, 255, 255, 239, 237, 237, 237, 241, 255, 255, 135, 0, 0,
            ],
            &[
                0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 208, 255, 255, 58, 38, 38, 38, 38, 38, 38, 20, 0, 0],
            &[0, 0, 146, 255, 255, 137, 0, 0, 0, 0, 0, 0, 1, 0, 0],
            &[
                0, 0, 43, 249, 255, 255, 185, 103, 79, 102, 144, 218, 54, 0, 0,
            ],
            &[
                0, 0, 0, 100, 252, 255, 255, 255, 255, 255, 255, 255, 54, 0, 0,
            ],
            &[0, 0, 0, 0, 56, 179, 249, 255, 255, 255, 245, 174, 23, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 33, 47, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ė' / 0x116
        'Ė' => &[
            &[0, 0, 0, 0, 0, 9, 157, 201, 136, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 94, 255, 255, 255, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 86, 255, 255, 255, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 129, 170, 100, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 254, 254, 254, 254, 254, 254, 254, 254, 254, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 220, 161, 161, 161, 161, 161, 161, 20, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 167, 23, 23, 23, 23, 23, 19, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 210, 136, 136, 136, 136, 136, 110, 0, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 219, 159, 159, 159, 159, 159, 159, 20, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ė' / 0x117
        'ė' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 66, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 31, 244, 255, 245, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 82, 255, 255, 255, 81, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 28, 238, 255, 234, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 50, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 87, 144, 165, 144, 79, 4, 0, 0, 0, 0],
            &[0, 0, 0, 42, 219, 255, 255, 255, 255, 255, 202, 25, 0, 0, 0],
            &[
                0, 0, 12, 224, 255, 255, 227, 180, 230, 255, 255, 189, 0, 0, 0,
            ],
            &[0, 0, 115, 255, 255, 182, 8, 0, 7, 190, 255, 255, 49, 0, 0],
            &[0, 0, 193, 255, 255, 54, 0, 0, 0, 80, 255, 255, 112, 0, 0],
            &[
                0, 0, 226, 255, 255, 239, 237, 237, 237, 241, 255, 255, 135, 0, 0,
            ],
            &[
                0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 208, 255, 255, 58, 38, 38, 38, 38, 38, 38, 20, 0, 0],
            &[0, 0, 146, 255, 255, 137, 0, 0, 0, 0, 0, 0, 1, 0, 0],
            &[
                0, 0, 43, 249, 255, 255, 185, 103, 79, 102, 144, 218, 54, 0, 0,
            ],
            &[
                0, 0, 0, 100, 252, 255, 255, 255, 255, 255, 255, 255, 54, 0, 0,
            ],
            &[0, 0, 0, 0, 56, 179, 249, 255, 255, 255, 245, 174, 23, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 33, 47, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ę' / 0x118
        'Ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 254, 254, 254, 254, 254, 254, 254, 254, 254, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 220, 161, 161, 161, 161, 161, 161, 20, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 167, 23, 23, 23, 23, 23, 19, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 210, 136, 136, 136, 136, 136, 110, 0, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 219, 159, 159, 159, 159, 159, 159, 20, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 92, 251, 153, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 30, 246, 233, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 97, 255, 206, 2, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 134, 230, 239, 101, 0, 0, 0],
        ],
        // letter: 'ę' / 0x119
        'ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 87, 144, 165, 144, 79, 4, 0, 0, 0, 0],
            &[0, 0, 0, 42, 219, 255, 255, 255, 255, 255, 202, 25, 0, 0, 0],
            &[
                0, 0, 12, 224, 255, 255, 227, 180, 230, 255, 255, 189, 0, 0, 0,
            ],
            &[0, 0, 115, 255, 255, 182, 8, 0, 7, 190, 255, 255, 49, 0, 0],
            &[0, 0, 193, 255, 255, 54, 0, 0, 0, 80, 255, 255, 112, 0, 0],
            &[
                0, 0, 226, 255, 255, 239, 237, 237, 237, 241, 255, 255, 135, 0, 0,
            ],
            &[
                0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 208, 255, 255, 58, 38, 38, 38, 38, 38, 38, 20, 0, 0],
            &[0, 0, 146, 255, 255, 137, 0, 0, 0, 0, 0, 0, 1, 0, 0],
            &[
                0, 0, 43, 249, 255, 255, 185, 103, 79, 102, 144, 218, 54, 0, 0,
            ],
            &[
                0, 0, 0, 100, 252, 255, 255, 255, 255, 255, 255, 255, 54, 0, 0,
            ],
            &[0, 0, 0, 0, 56, 179, 249, 255, 255, 255, 255, 255, 30, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 33, 88, 255, 216, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 198, 255, 70, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 1, 252, 255, 61, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 53, 201, 242, 199, 0, 0, 0],
        ],
        // letter: 'Ě' / 0x11a
        'Ě' => &[
            &[0, 0, 0, 52, 157, 126, 4, 0, 15, 141, 157, 29, 0, 0, 0],
            &[0, 0, 0, 2, 178, 255, 185, 35, 214, 255, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 224, 255, 251, 255, 201, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 66, 190, 190, 190, 39, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 254, 254, 254, 254, 254, 254, 254, 254, 254, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 220, 161, 161, 161, 161, 161, 161, 20, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 167, 23, 23, 23, 23, 23, 19, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 208, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 210, 136, 136, 136, 136, 136, 110, 0, 0, 0,
            ],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 81, 255, 255, 158, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 81, 255, 255, 219, 159, 159, 159, 159, 159, 159, 20, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ě' / 0x11b
        'ě' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 22, 12, 0, 0, 0, 12, 22, 7, 0, 0, 0],
            &[0, 0, 0, 32, 236, 242, 78, 0, 79, 242, 239, 36, 0, 0, 0],
            &[0, 0, 0, 0, 67, 253, 251, 150, 252, 254, 75, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 135, 255, 255, 255, 143, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 70, 70, 70, 9, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 87, 144, 165, 144, 79, 4, 0, 0, 0, 0],
            &[0, 0, 0, 42, 219, 255, 255, 255, 255, 255, 202, 25, 0, 0, 0],
            &[
                0, 0, 12, 224, 255, 255, 227, 180, 230, 255, 255, 189, 0, 0, 0,
            ],
            &[0, 0, 115, 255, 255, 182, 8, 0, 7, 190, 255, 255, 49, 0, 0],
            &[0, 0, 193, 255, 255, 54, 0, 0, 0, 80, 255, 255, 112, 0, 0],
            &[
                0, 0, 226, 255, 255, 239, 237, 237, 237, 241, 255, 255, 135, 0, 0,
            ],
            &[
                0, 0, 234, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 208, 255, 255, 58, 38, 38, 38, 38, 38, 38, 20, 0, 0],
            &[0, 0, 146, 255, 255, 137, 0, 0, 0, 0, 0, 0, 1, 0, 0],
            &[
                0, 0, 43, 249, 255, 255, 185, 103, 79, 102, 144, 218, 54, 0, 0,
            ],
            &[
                0, 0, 0, 100, 252, 255, 255, 255, 255, 255, 255, 255, 54, 0, 0,
            ],
            &[0, 0, 0, 0, 56, 179, 249, 255, 255, 255, 245, 174, 23, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 33, 47, 27, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => &[
            &[0, 0, 0, 0, 0, 0, 134, 157, 157, 97, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 103, 255, 255, 255, 249, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 247, 252, 91, 152, 255, 220, 16, 0, 0, 0],
            &[0, 0, 0, 0, 162, 189, 69, 0, 0, 116, 190, 112, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 36, 48, 33, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 164, 245, 255, 255, 255, 243, 165, 46, 0, 0],
            &[
                0, 0, 0, 60, 236, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0,
            ],
            &[
                0, 0, 19, 235, 255, 255, 243, 155, 117, 131, 185, 212, 0, 0, 0,
            ],
            &[0, 0, 145, 255, 255, 220, 29, 0, 0, 0, 0, 8, 0, 0, 0],
            &[0, 1, 230, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 255, 255, 237, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 255, 255, 193, 0, 0, 19, 113, 113, 113, 113, 74, 0, 0],
            &[
                0, 88, 255, 255, 177, 0, 0, 44, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[
                0, 76, 255, 255, 188, 0, 0, 44, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 54, 255, 255, 226, 0, 0, 3, 19, 42, 255, 255, 167, 0, 0],
            &[0, 8, 245, 255, 255, 49, 0, 0, 0, 24, 255, 255, 167, 0, 0],
            &[0, 0, 175, 255, 255, 194, 8, 0, 0, 24, 255, 255, 167, 0, 0],
            &[
                0, 0, 44, 251, 255, 255, 217, 123, 101, 139, 255, 255, 167, 0, 0,
            ],
            &[
                0, 0, 0, 98, 251, 255, 255, 255, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 0, 0, 0, 59, 178, 252, 255, 255, 255, 250, 195, 79, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 33, 46, 26, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĝ' / 0x11d
        'ĝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 22, 22, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 156, 255, 255, 254, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 255, 250, 198, 255, 225, 18, 0, 0, 0, 0],
            &[0, 0, 0, 44, 241, 244, 78, 4, 163, 255, 185, 3, 0, 0, 0],
            &[0, 0, 0, 38, 70, 34, 0, 0, 0, 61, 70, 10, 0, 0, 0],
            &[0, 0, 0, 0, 85, 159, 168, 117, 14, 1, 113, 119, 53, 0, 0],
            &[
                0, 0, 1, 166, 255, 255, 255, 255, 220, 96, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 94, 255, 255, 255, 249, 226, 254, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 198, 255, 255, 185, 11, 0, 54, 245, 255, 255, 113, 0, 0,
            ],
            &[0, 6, 251, 255, 255, 49, 0, 0, 0, 160, 255, 255, 113, 0, 0],
            &[0, 30, 255, 255, 252, 3, 0, 0, 0, 117, 255, 255, 113, 0, 0],
            &[0, 36, 255, 255, 244, 0, 0, 0, 0, 107, 255, 255, 113, 0, 0],
            &[0, 20, 255, 255, 255, 15, 0, 0, 0, 126, 255, 255, 113, 0, 0],
            &[0, 0, 233, 255, 255, 100, 0, 0, 0, 200, 255, 255, 113, 0, 0],
            &[
                0, 0, 153, 255, 255, 246, 127, 94, 175, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 31, 240, 255, 255, 255, 255, 255, 230, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 54, 213, 255, 255, 243, 118, 153, 255, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 33, 44, 9, 0, 173, 255, 255, 100, 0, 0],
            &[0, 0, 30, 28, 0, 0, 0, 0, 19, 234, 255, 255, 65, 0, 0],
            &[
                0, 0, 92, 254, 202, 160, 136, 155, 233, 255, 255, 227, 3, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 17, 31, 15, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ğ' / 0x11e
        'Ğ' => &[
            &[0, 0, 0, 0, 103, 173, 33, 0, 0, 76, 173, 56, 0, 0, 0],
            &[0, 0, 0, 0, 109, 255, 191, 79, 103, 224, 254, 37, 0, 0, 0],
            &[0, 0, 0, 0, 10, 215, 255, 255, 255, 255, 144, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 124, 176, 162, 86, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 36, 48, 33, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 164, 245, 255, 255, 255, 243, 165, 46, 0, 0],
            &[
                0, 0, 0, 60, 236, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0,
            ],
            &[
                0, 0, 19, 235, 255, 255, 243, 155, 117, 131, 185, 212, 0, 0, 0,
            ],
            &[0, 0, 145, 255, 255, 220, 29, 0, 0, 0, 0, 8, 0, 0, 0],
            &[0, 1, 230, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 255, 255, 237, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 255, 255, 193, 0, 0, 19, 113, 113, 113, 113, 74, 0, 0],
            &[
                0, 88, 255, 255, 177, 0, 0, 44, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[
                0, 76, 255, 255, 188, 0, 0, 44, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 54, 255, 255, 226, 0, 0, 3, 19, 42, 255, 255, 167, 0, 0],
            &[0, 8, 245, 255, 255, 49, 0, 0, 0, 24, 255, 255, 167, 0, 0],
            &[0, 0, 175, 255, 255, 194, 8, 0, 0, 24, 255, 255, 167, 0, 0],
            &[
                0, 0, 44, 251, 255, 255, 217, 123, 101, 139, 255, 255, 167, 0, 0,
            ],
            &[
                0, 0, 0, 98, 251, 255, 255, 255, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 0, 0, 0, 59, 178, 252, 255, 255, 255, 250, 195, 79, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 33, 46, 26, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ğ' / 0x11f
        'ğ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 38, 20, 0, 0, 0, 35, 31, 0, 0, 0, 0],
            &[0, 0, 0, 27, 255, 195, 5, 0, 41, 254, 178, 0, 0, 0, 0],
            &[0, 0, 0, 0, 203, 255, 225, 201, 250, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 206, 255, 255, 250, 130, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 38, 57, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 159, 168, 117, 14, 1, 113, 119, 53, 0, 0],
            &[
                0, 0, 1, 166, 255, 255, 255, 255, 220, 96, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 94, 255, 255, 255, 249, 226, 254, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 198, 255, 255, 185, 11, 0, 54, 245, 255, 255, 113, 0, 0,
            ],
            &[0, 6, 251, 255, 255, 49, 0, 0, 0, 160, 255, 255, 113, 0, 0],
            &[0, 30, 255, 255, 252, 3, 0, 0, 0, 117, 255, 255, 113, 0, 0],
            &[0, 36, 255, 255, 244, 0, 0, 0, 0, 107, 255, 255, 113, 0, 0],
            &[0, 20, 255, 255, 255, 15, 0, 0, 0, 126, 255, 255, 113, 0, 0],
            &[0, 0, 233, 255, 255, 100, 0, 0, 0, 200, 255, 255, 113, 0, 0],
            &[
                0, 0, 153, 255, 255, 246, 127, 94, 175, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 31, 240, 255, 255, 255, 255, 255, 230, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 54, 213, 255, 255, 243, 118, 153, 255, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 33, 44, 9, 0, 173, 255, 255, 100, 0, 0],
            &[0, 0, 30, 28, 0, 0, 0, 0, 19, 234, 255, 255, 65, 0, 0],
            &[
                0, 0, 92, 254, 202, 160, 136, 155, 233, 255, 255, 227, 3, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 17, 31, 15, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ġ' / 0x120
        'Ġ' => &[
            &[0, 0, 0, 0, 0, 0, 84, 197, 182, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 237, 255, 255, 164, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 229, 255, 255, 153, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 63, 167, 149, 24, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 36, 48, 33, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 164, 245, 255, 255, 255, 243, 165, 46, 0, 0],
            &[
                0, 0, 0, 60, 236, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0,
            ],
            &[
                0, 0, 19, 235, 255, 255, 243, 155, 117, 131, 185, 212, 0, 0, 0,
            ],
            &[0, 0, 145, 255, 255, 220, 29, 0, 0, 0, 0, 8, 0, 0, 0],
            &[0, 1, 230, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 255, 255, 237, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 255, 255, 193, 0, 0, 19, 113, 113, 113, 113, 74, 0, 0],
            &[
                0, 88, 255, 255, 177, 0, 0, 44, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[
                0, 76, 255, 255, 188, 0, 0, 44, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 54, 255, 255, 226, 0, 0, 3, 19, 42, 255, 255, 167, 0, 0],
            &[0, 8, 245, 255, 255, 49, 0, 0, 0, 24, 255, 255, 167, 0, 0],
            &[0, 0, 175, 255, 255, 194, 8, 0, 0, 24, 255, 255, 167, 0, 0],
            &[
                0, 0, 44, 251, 255, 255, 217, 123, 101, 139, 255, 255, 167, 0, 0,
            ],
            &[
                0, 0, 0, 98, 251, 255, 255, 255, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 0, 0, 0, 59, 178, 252, 255, 255, 255, 250, 195, 79, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 33, 46, 26, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ġ' / 0x121
        'ġ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 65, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 75, 253, 255, 219, 4, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 135, 255, 255, 255, 27, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 69, 251, 255, 203, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 48, 5, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 159, 168, 117, 14, 1, 113, 119, 53, 0, 0],
            &[
                0, 0, 1, 166, 255, 255, 255, 255, 220, 96, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 94, 255, 255, 255, 249, 226, 254, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 198, 255, 255, 185, 11, 0, 54, 245, 255, 255, 113, 0, 0,
            ],
            &[0, 6, 251, 255, 255, 49, 0, 0, 0, 160, 255, 255, 113, 0, 0],
            &[0, 30, 255, 255, 252, 3, 0, 0, 0, 117, 255, 255, 113, 0, 0],
            &[0, 36, 255, 255, 244, 0, 0, 0, 0, 107, 255, 255, 113, 0, 0],
            &[0, 20, 255, 255, 255, 15, 0, 0, 0, 126, 255, 255, 113, 0, 0],
            &[0, 0, 233, 255, 255, 100, 0, 0, 0, 200, 255, 255, 113, 0, 0],
            &[
                0, 0, 153, 255, 255, 246, 127, 94, 175, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 31, 240, 255, 255, 255, 255, 255, 230, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 54, 213, 255, 255, 243, 118, 153, 255, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 33, 44, 9, 0, 173, 255, 255, 100, 0, 0],
            &[0, 0, 30, 28, 0, 0, 0, 0, 19, 234, 255, 255, 65, 0, 0],
            &[
                0, 0, 92, 254, 202, 160, 136, 155, 233, 255, 255, 227, 3, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 17, 31, 15, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ģ' / 0x122
        'Ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 36, 48, 33, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 34, 164, 245, 255, 255, 255, 243, 165, 46, 0, 0],
            &[
                0, 0, 0, 60, 236, 255, 255, 255, 255, 255, 255, 255, 57, 0, 0,
            ],
            &[
                0, 0, 19, 235, 255, 255, 243, 155, 117, 131, 185, 212, 0, 0, 0,
            ],
            &[0, 0, 145, 255, 255, 220, 29, 0, 0, 0, 0, 8, 0, 0, 0],
            &[0, 1, 230, 255, 255, 73, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 43, 255, 255, 237, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 68, 255, 255, 193, 0, 0, 19, 113, 113, 113, 113, 74, 0, 0],
            &[
                0, 88, 255, 255, 177, 0, 0, 44, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[
                0, 76, 255, 255, 188, 0, 0, 44, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 54, 255, 255, 226, 0, 0, 3, 19, 42, 255, 255, 167, 0, 0],
            &[0, 8, 245, 255, 255, 49, 0, 0, 0, 24, 255, 255, 167, 0, 0],
            &[0, 0, 175, 255, 255, 194, 8, 0, 0, 24, 255, 255, 167, 0, 0],
            &[
                0, 0, 44, 251, 255, 255, 217, 123, 101, 139, 255, 255, 167, 0, 0,
            ],
            &[
                0, 0, 0, 98, 251, 255, 255, 255, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 0, 0, 0, 59, 178, 252, 255, 255, 255, 250, 195, 79, 0, 0],
            &[0, 0, 0, 0, 0, 0, 6, 33, 46, 26, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 94, 183, 183, 46, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 168, 255, 216, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 164, 136, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ģ' / 0x123
        'ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 16, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 30, 245, 238, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 166, 255, 179, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 254, 255, 127, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 29, 70, 70, 27, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 85, 159, 168, 117, 14, 1, 113, 119, 53, 0, 0],
            &[
                0, 0, 1, 166, 255, 255, 255, 255, 220, 96, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 94, 255, 255, 255, 249, 226, 254, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 198, 255, 255, 185, 11, 0, 54, 245, 255, 255, 113, 0, 0,
            ],
            &[0, 6, 251, 255, 255, 49, 0, 0, 0, 160, 255, 255, 113, 0, 0],
            &[0, 30, 255, 255, 252, 3, 0, 0, 0, 117, 255, 255, 113, 0, 0],
            &[0, 36, 255, 255, 244, 0, 0, 0, 0, 107, 255, 255, 113, 0, 0],
            &[0, 20, 255, 255, 255, 15, 0, 0, 0, 126, 255, 255, 113, 0, 0],
            &[0, 0, 233, 255, 255, 100, 0, 0, 0, 200, 255, 255, 113, 0, 0],
            &[
                0, 0, 153, 255, 255, 246, 127, 94, 175, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 31, 240, 255, 255, 255, 255, 255, 230, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 54, 213, 255, 255, 243, 118, 153, 255, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 33, 44, 9, 0, 173, 255, 255, 100, 0, 0],
            &[0, 0, 30, 28, 0, 0, 0, 0, 19, 234, 255, 255, 65, 0, 0],
            &[
                0, 0, 92, 254, 202, 160, 136, 155, 233, 255, 255, 227, 3, 0, 0,
            ],
            &[0, 0, 0, 0, 1, 17, 31, 15, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĥ' / 0x124
        'Ĥ' => &[
            &[0, 0, 0, 0, 0, 67, 157, 157, 152, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 228, 255, 255, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 184, 255, 187, 68, 241, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 79, 190, 144, 6, 0, 42, 182, 181, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 251, 0, 0, 0, 0, 99, 254, 254, 166, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[
                0, 14, 255, 255, 253, 34, 34, 34, 34, 120, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 254, 125, 125, 125, 125, 176, 255, 255, 167, 0, 0,
            ],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĥ' / 0x125
        'ĥ' => &[
            &[0, 0, 137, 255, 255, 255, 231, 21, 0, 0, 0, 0, 0, 0, 0],
            &[0, 76, 254, 244, 77, 192, 255, 186, 3, 0, 0, 0, 0, 0, 0],
            &[12, 186, 192, 51, 0, 8, 149, 199, 83, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 207, 245, 245, 49, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 216, 255, 255, 51, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 216, 255, 255, 51, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 216, 255, 255, 50, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 216, 255, 255, 48, 38, 135, 165, 148, 71, 0, 0, 0],
            &[
                0, 0, 0, 216, 255, 255, 110, 250, 255, 255, 255, 255, 141, 0, 0,
            ],
            &[
                0, 0, 0, 216, 255, 255, 255, 251, 225, 250, 255, 255, 255, 47, 0,
            ],
            &[
                0, 0, 0, 216, 255, 255, 228, 35, 0, 29, 232, 255, 255, 116, 0,
            ],
            &[0, 0, 0, 216, 255, 255, 119, 0, 0, 0, 150, 255, 255, 138, 0],
            &[0, 0, 0, 216, 255, 255, 69, 0, 0, 0, 127, 255, 255, 140, 0],
            &[0, 0, 0, 216, 255, 255, 53, 0, 0, 0, 126, 255, 255, 140, 0],
            &[0, 0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0],
            &[0, 0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0],
            &[0, 0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0],
            &[0, 0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0],
            &[0, 0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 142, 148, 148, 67, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ħ' / 0x126
        'Ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 204, 254, 254, 61, 0, 0, 0, 157, 254, 254, 107, 0, 0],
            &[0, 0, 205, 255, 255, 61, 0, 0, 0, 158, 255, 255, 108, 0, 0],
            &[
                0, 244, 255, 255, 255, 255, 244, 244, 244, 255, 255, 255, 255, 146, 0,
            ],
            &[
                0, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 153, 0,
            ],
            &[0, 9, 215, 255, 255, 71, 9, 9, 9, 168, 255, 255, 118, 5, 0],
            &[
                0, 0, 205, 255, 255, 88, 34, 34, 34, 172, 255, 255, 108, 0, 0,
            ],
            &[
                0, 0, 205, 255, 255, 255, 255, 255, 255, 255, 255, 255, 108, 0, 0,
            ],
            &[
                0, 0, 205, 255, 255, 255, 255, 255, 255, 255, 255, 255, 108, 0, 0,
            ],
            &[
                0, 0, 205, 255, 255, 156, 125, 125, 125, 206, 255, 255, 108, 0, 0,
            ],
            &[0, 0, 205, 255, 255, 61, 0, 0, 0, 158, 255, 255, 108, 0, 0],
            &[0, 0, 205, 255, 255, 61, 0, 0, 0, 158, 255, 255, 108, 0, 0],
            &[0, 0, 205, 255, 255, 61, 0, 0, 0, 158, 255, 255, 108, 0, 0],
            &[0, 0, 205, 255, 255, 61, 0, 0, 0, 158, 255, 255, 108, 0, 0],
            &[0, 0, 205, 255, 255, 61, 0, 0, 0, 158, 255, 255, 108, 0, 0],
            &[0, 0, 205, 255, 255, 61, 0, 0, 0, 158, 255, 255, 108, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ħ' / 0x127
        'ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 207, 245, 245, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 71, 255, 255, 255, 122, 71, 71, 22, 0, 0, 0, 0, 0, 0],
            &[0, 255, 255, 255, 255, 255, 255, 255, 78, 0, 0, 0, 0, 0, 0],
            &[0, 177, 255, 255, 255, 228, 177, 177, 54, 0, 0, 0, 0, 0, 0],
            &[0, 0, 216, 255, 255, 49, 8, 81, 112, 94, 25, 0, 0, 0, 0],
            &[
                0, 0, 216, 255, 255, 80, 226, 255, 255, 255, 248, 94, 0, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 241, 255, 255, 255, 255, 255, 249, 29, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 245, 84, 23, 78, 245, 255, 255, 107, 0, 0,
            ],
            &[0, 0, 216, 255, 255, 137, 0, 0, 0, 160, 255, 255, 136, 0, 0],
            &[0, 0, 216, 255, 255, 75, 0, 0, 0, 130, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 55, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĩ' / 0x128
        'Ĩ' => &[
            &[0, 0, 0, 0, 19, 82, 49, 0, 0, 30, 103, 26, 0, 0, 0],
            &[0, 0, 0, 18, 224, 255, 255, 198, 103, 182, 255, 36, 0, 0, 0],
            &[0, 0, 0, 113, 255, 203, 226, 255, 255, 255, 201, 0, 0, 0, 0],
            &[0, 0, 0, 123, 184, 2, 3, 84, 168, 158, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 254, 254, 254, 254, 254, 254, 254, 254, 180, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 3, 45, 74, 130, 255, 255, 215, 90, 62, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 43, 73, 129, 255, 255, 215, 89, 60, 25, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĩ' / 0x129
        'ĩ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 203, 170, 63, 0, 86, 224, 48, 0, 0, 0],
            &[0, 0, 0, 61, 255, 255, 255, 255, 224, 247, 246, 14, 0, 0, 0],
            &[0, 0, 0, 143, 253, 84, 108, 217, 255, 255, 117, 0, 0, 0, 0],
            &[0, 0, 0, 50, 65, 0, 0, 1, 47, 37, 0, 0, 0, 0, 0],
            &[0, 0, 20, 119, 119, 119, 119, 119, 119, 15, 0, 0, 0, 0, 0],
            &[0, 0, 44, 255, 255, 255, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 31, 197, 225, 251, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 235, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 66, 108, 245, 255, 255, 130, 68, 21, 0, 0, 0],
            &[
                0, 0, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ī' / 0x12a
        'Ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 152, 171, 171, 171, 171, 171, 80, 0, 0, 0, 0],
            &[0, 0, 0, 0, 228, 255, 255, 255, 255, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 104, 104, 104, 104, 104, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 254, 254, 254, 254, 254, 254, 254, 254, 180, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 3, 45, 74, 130, 255, 255, 215, 90, 62, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 43, 73, 129, 255, 255, 215, 89, 60, 25, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ī' / 0x12b
        'ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 36, 36, 36, 36, 36, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 228, 255, 255, 255, 255, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 214, 240, 240, 240, 240, 240, 113, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 119, 119, 119, 119, 119, 119, 15, 0, 0, 0, 0, 0],
            &[0, 0, 44, 255, 255, 255, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 31, 197, 225, 251, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 235, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 66, 108, 245, 255, 255, 130, 68, 21, 0, 0, 0],
            &[
                0, 0, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => &[
            &[0, 0, 0, 28, 173, 109, 0, 0, 4, 169, 131, 0, 0, 0, 0],
            &[0, 0, 0, 6, 247, 242, 109, 81, 166, 255, 146, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 255, 255, 255, 255, 231, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 159, 177, 132, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 254, 254, 254, 254, 254, 254, 254, 254, 180, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 3, 45, 74, 130, 255, 255, 215, 90, 62, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 43, 73, 129, 255, 255, 215, 89, 60, 25, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĭ' / 0x12d
        'ĭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 38, 20, 0, 0, 0, 35, 31, 0, 0, 0, 0],
            &[0, 0, 0, 27, 255, 195, 5, 0, 41, 254, 178, 0, 0, 0, 0],
            &[0, 0, 0, 0, 203, 255, 225, 201, 250, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 206, 255, 255, 250, 130, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 38, 57, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 119, 119, 119, 119, 119, 119, 15, 0, 0, 0, 0, 0],
            &[0, 0, 44, 255, 255, 255, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 31, 197, 225, 251, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 235, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 66, 108, 245, 255, 255, 130, 68, 21, 0, 0, 0],
            &[
                0, 0, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Į' / 0x12e
        'Į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 254, 254, 254, 254, 254, 254, 254, 254, 180, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 3, 45, 74, 130, 255, 255, 215, 90, 62, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 43, 73, 129, 255, 255, 215, 89, 60, 25, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 103, 253, 140, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 39, 250, 222, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 110, 255, 194, 1, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 143, 232, 237, 90, 0, 0, 0, 0],
        ],
        // letter: 'į' / 0x12f
        'į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 15, 66, 33, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 227, 255, 252, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 44, 255, 255, 255, 119, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 220, 255, 246, 51, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 49, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 119, 119, 119, 119, 119, 119, 15, 0, 0, 0, 0, 0],
            &[0, 0, 44, 255, 255, 255, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 31, 197, 225, 251, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 235, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 66, 108, 245, 255, 255, 130, 68, 21, 0, 0, 0],
            &[
                0, 0, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 92, 251, 153, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 30, 246, 233, 5, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 97, 255, 206, 2, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 134, 230, 239, 101, 0, 0, 0],
        ],
        // letter: 'İ' / 0x130
        'İ' => &[
            &[0, 0, 0, 0, 0, 22, 169, 200, 112, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 126, 255, 255, 254, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 119, 255, 255, 249, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 13, 141, 169, 80, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 254, 254, 254, 254, 254, 254, 254, 254, 180, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 3, 45, 74, 130, 255, 255, 215, 90, 62, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 255, 255, 186, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 43, 73, 129, 255, 255, 215, 89, 60, 25, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ı' / 0x131
        'ı' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 119, 119, 119, 119, 119, 119, 15, 0, 0, 0, 0, 0],
            &[0, 0, 44, 255, 255, 255, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 31, 197, 225, 251, 255, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 235, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 234, 255, 255, 32, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 66, 108, 245, 255, 255, 130, 68, 21, 0, 0, 0],
            &[
                0, 0, 179, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĳ' / 0x132
        'Ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 54, 254, 254, 179, 0, 0, 42, 254, 254, 207, 0, 0, 0],
            &[0, 0, 55, 255, 255, 180, 0, 0, 42, 255, 255, 208, 0, 0, 0],
            &[0, 0, 55, 255, 255, 180, 0, 0, 42, 255, 255, 208, 0, 0, 0],
            &[0, 0, 55, 255, 255, 180, 0, 0, 42, 255, 255, 208, 0, 0, 0],
            &[0, 0, 55, 255, 255, 180, 0, 0, 42, 255, 255, 208, 0, 0, 0],
            &[0, 0, 55, 255, 255, 180, 0, 0, 42, 255, 255, 208, 0, 0, 0],
            &[0, 0, 55, 255, 255, 180, 0, 0, 42, 255, 255, 208, 0, 0, 0],
            &[0, 0, 55, 255, 255, 180, 0, 0, 42, 255, 255, 208, 0, 0, 0],
            &[0, 0, 55, 255, 255, 180, 0, 0, 42, 255, 255, 208, 0, 0, 0],
            &[0, 0, 55, 255, 255, 180, 0, 0, 42, 255, 255, 208, 0, 0, 0],
            &[0, 0, 25, 118, 118, 83, 0, 0, 58, 255, 255, 193, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 142, 255, 255, 162, 0, 0, 0],
            &[
                0, 0, 40, 183, 144, 119, 120, 177, 255, 255, 255, 85, 0, 0, 0,
            ],
            &[0, 0, 49, 255, 255, 255, 255, 255, 255, 255, 190, 2, 0, 0, 0],
            &[0, 0, 39, 234, 255, 255, 255, 255, 232, 137, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 45, 48, 27, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĳ' / 0x133
        'ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 66, 20, 0, 0, 0, 45, 63, 7, 0, 0, 0],
            &[0, 0, 44, 248, 255, 239, 22, 0, 105, 255, 255, 192, 0, 0, 0],
            &[0, 0, 99, 255, 255, 255, 64, 0, 167, 255, 255, 251, 1, 0, 0],
            &[0, 0, 39, 243, 255, 227, 16, 0, 97, 254, 255, 175, 0, 0, 0],
            &[0, 0, 0, 18, 50, 10, 0, 0, 0, 31, 45, 1, 0, 0, 0],
            &[0, 0, 15, 119, 119, 108, 0, 0, 42, 119, 119, 82, 0, 0, 0],
            &[0, 0, 33, 255, 255, 233, 0, 0, 91, 255, 255, 176, 0, 0, 0],
            &[0, 0, 33, 255, 255, 233, 0, 0, 91, 255, 255, 176, 0, 0, 0],
            &[0, 0, 33, 255, 255, 233, 0, 0, 91, 255, 255, 176, 0, 0, 0],
            &[0, 0, 33, 255, 255, 233, 0, 0, 91, 255, 255, 176, 0, 0, 0],
            &[0, 0, 33, 255, 255, 233, 0, 0, 91, 255, 255, 176, 0, 0, 0],
            &[0, 0, 33, 255, 255, 233, 0, 0, 91, 255, 255, 176, 0, 0, 0],
            &[0, 0, 33, 255, 255, 233, 0, 0, 91, 255, 255, 176, 0, 0, 0],
            &[0, 0, 33, 255, 255, 233, 0, 0, 91, 255, 255, 176, 0, 0, 0],
            &[0, 0, 33, 255, 255, 233, 0, 0, 91, 255, 255, 176, 0, 0, 0],
            &[0, 0, 33, 255, 255, 233, 0, 0, 91, 255, 255, 176, 0, 0, 0],
            &[0, 0, 33, 255, 255, 233, 0, 0, 91, 255, 255, 176, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 116, 255, 255, 173, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 187, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 104, 178, 142, 140, 197, 255, 255, 255, 92, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 30, 25, 2, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĵ' / 0x134
        'Ĵ' => &[
            &[0, 0, 0, 0, 0, 0, 7, 148, 157, 157, 77, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 136, 255, 255, 255, 237, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 77, 254, 244, 72, 180, 255, 197, 6, 0, 0],
            &[0, 0, 0, 0, 9, 177, 185, 49, 0, 4, 137, 190, 87, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 176, 254, 254, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 74, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 177, 255, 255, 73, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 205, 255, 255, 59, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 23, 247, 255, 255, 29, 0, 0, 0, 0],
            &[0, 44, 182, 141, 118, 146, 224, 255, 255, 209, 0, 0, 0, 0, 0],
            &[0, 55, 255, 255, 255, 255, 255, 255, 252, 66, 0, 0, 0, 0, 0],
            &[0, 43, 234, 255, 255, 255, 254, 197, 62, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 26, 46, 40, 15, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĵ' / 0x135
        'ĵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 7, 22, 22, 17, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 175, 255, 255, 250, 47, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 101, 255, 246, 202, 255, 213, 11, 0, 0, 0],
            &[0, 0, 0, 0, 57, 248, 238, 64, 9, 178, 255, 168, 0, 0, 0],
            &[0, 0, 0, 0, 43, 70, 29, 0, 0, 1, 65, 70, 4, 0, 0],
            &[0, 0, 0, 34, 119, 119, 119, 119, 119, 119, 1, 0, 0, 0, 0],
            &[0, 0, 0, 74, 255, 255, 255, 255, 255, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 53, 200, 228, 252, 255, 255, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 255, 255, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 9, 255, 255, 255, 2, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 35, 255, 255, 254, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 107, 255, 255, 234, 0, 0, 0, 0, 0],
            &[0, 36, 193, 151, 134, 170, 249, 255, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 25, 30, 8, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ķ' / 0x136
        'Ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 235, 0, 0, 0, 2, 196, 254, 254, 136, 0, 0],
            &[0, 14, 255, 255, 237, 0, 0, 0, 114, 255, 255, 206, 6, 0, 0],
            &[0, 14, 255, 255, 237, 0, 0, 38, 246, 255, 246, 41, 0, 0, 0],
            &[0, 14, 255, 255, 237, 0, 2, 197, 255, 255, 107, 0, 0, 0, 0],
            &[0, 14, 255, 255, 237, 0, 114, 255, 255, 182, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 237, 35, 246, 255, 234, 23, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 237, 177, 255, 255, 92, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 253, 255, 255, 255, 153, 0, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 255, 255, 253, 255, 252, 43, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 255, 135, 125, 255, 255, 181, 0, 0, 0, 0, 0],
            &[0, 14, 255, 255, 237, 0, 12, 233, 255, 255, 67, 0, 0, 0, 0],
            &[0, 14, 255, 255, 237, 0, 0, 110, 255, 255, 207, 2, 0, 0, 0],
            &[0, 14, 255, 255, 237, 0, 0, 6, 224, 255, 255, 95, 0, 0, 0],
            &[0, 14, 255, 255, 237, 0, 0, 0, 95, 255, 255, 227, 9, 0, 0],
            &[0, 14, 255, 255, 237, 0, 0, 0, 2, 213, 255, 255, 123, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 182, 183, 137, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 41, 255, 255, 89, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 95, 164, 53, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ķ' / 0x137
        'ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 181, 245, 245, 74, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 0, 85, 119, 119, 96, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 86, 254, 255, 244, 52, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 55, 247, 255, 252, 75, 0, 0, 0, 0],
            &[0, 189, 255, 255, 76, 31, 232, 255, 255, 102, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 77, 207, 255, 255, 132, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 197, 255, 255, 243, 15, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 255, 156, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 255, 144, 221, 255, 255, 72, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 115, 0, 66, 255, 255, 227, 15, 0, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 156, 255, 255, 157, 0, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 16, 230, 255, 255, 73, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 0, 80, 255, 255, 227, 15, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 136, 183, 179, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 227, 255, 159, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 50, 164, 98, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĸ' / 0x138
        'ĸ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 88, 119, 119, 36, 0, 0, 0, 85, 119, 119, 96, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 86, 254, 255, 244, 52, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 55, 247, 255, 252, 75, 0, 0, 0, 0],
            &[0, 189, 255, 255, 76, 31, 232, 255, 255, 102, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 77, 207, 255, 255, 132, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 197, 255, 255, 243, 15, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 255, 255, 255, 255, 156, 0, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 255, 144, 221, 255, 255, 72, 0, 0, 0, 0, 0],
            &[0, 189, 255, 255, 115, 0, 66, 255, 255, 227, 15, 0, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 156, 255, 255, 157, 0, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 16, 230, 255, 255, 73, 0, 0, 0],
            &[0, 189, 255, 255, 78, 0, 0, 0, 80, 255, 255, 227, 15, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĺ' / 0x139
        'Ĺ' => &[
            &[0, 0, 0, 0, 27, 157, 157, 156, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 255, 255, 111, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 94, 255, 252, 99, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 5, 177, 190, 73, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 254, 254, 237, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 250, 176, 176, 176, 176, 176, 176, 22, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĺ' / 0x13a
        'ĺ' => &[
            &[0, 0, 0, 0, 0, 0, 1, 188, 255, 253, 100, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 109, 255, 250, 90, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 191, 197, 66, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 245, 245, 245, 245, 245, 245, 20, 0, 0, 0, 0, 0],
            &[0, 0, 1, 252, 255, 255, 255, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 61, 114, 251, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 67, 110, 250, 255, 255, 124, 67, 20, 0, 0, 0],
            &[
                0, 0, 180, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 36, 148, 148, 146, 20, 0, 0, 0],
        ],
        // letter: 'Ļ' / 0x13b
        'Ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 254, 254, 237, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 250, 176, 176, 176, 176, 176, 176, 22, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 101, 183, 183, 38, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 179, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 19, 164, 129, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ļ' / 0x13c
        'ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 245, 245, 245, 245, 245, 245, 20, 0, 0, 0, 0, 0],
            &[0, 0, 1, 252, 255, 255, 255, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 61, 114, 251, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 67, 110, 250, 255, 255, 124, 67, 20, 0, 0, 0],
            &[
                0, 0, 180, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 159, 183, 164, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 251, 255, 127, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 164, 77, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ľ' / 0x13d
        'Ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 254, 254, 237, 0, 0, 0, 215, 254, 222, 1, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 10, 252, 255, 103, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 63, 255, 215, 5, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 36, 93, 42, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 250, 176, 176, 176, 176, 176, 176, 22, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ľ' / 0x13e
        'ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 1, 245, 245, 245, 245, 245, 245, 20, 91, 245, 245, 87, 0, 0,
            ],
            &[
                0, 1, 252, 255, 255, 255, 255, 255, 21, 141, 255, 222, 5, 0, 0,
            ],
            &[0, 0, 11, 61, 114, 251, 255, 255, 21, 197, 255, 89, 0, 0, 0],
            &[0, 0, 0, 0, 0, 245, 255, 255, 21, 94, 95, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 24, 67, 110, 250, 255, 255, 124, 67, 20, 0, 0, 0, 0],
            &[
                0, 180, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0, 0,
            ],
            &[
                0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 27, 254, 254, 237, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 9, 189, 250, 199, 17, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 76, 255, 255, 255, 99, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 54, 255, 255, 255, 74, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 98, 163, 108, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 28, 255, 255, 239, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 28, 255, 255, 250, 176, 176, 176, 176, 176, 176, 22, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 28, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŀ' / 0x140
        'ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[1, 245, 245, 245, 245, 245, 245, 20, 0, 0, 0, 0, 0, 0, 0],
            &[1, 252, 255, 255, 255, 255, 255, 21, 0, 0, 0, 0, 0, 0, 0],
            &[0, 11, 61, 114, 251, 255, 255, 21, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 115, 191, 149, 10, 0],
            &[0, 0, 0, 0, 245, 255, 255, 21, 0, 42, 255, 255, 255, 108, 0],
            &[0, 0, 0, 0, 245, 255, 255, 21, 0, 49, 255, 255, 255, 115, 0],
            &[0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 141, 222, 178, 16, 0],
            &[0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0, 0, 0],
            &[0, 24, 67, 110, 250, 255, 255, 124, 67, 20, 0, 0, 0, 0, 0],
            &[
                180, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0, 0, 0,
            ],
            &[
                189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ł' / 0x141
        'Ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 254, 254, 197, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 255, 255, 198, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 255, 255, 198, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 255, 255, 198, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 255, 255, 198, 19, 57, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 255, 255, 255, 236, 230, 13, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 255, 255, 255, 255, 241, 61, 0, 0, 0, 0, 0],
            &[0, 0, 14, 213, 255, 255, 255, 163, 24, 0, 0, 0, 0, 0, 0],
            &[0, 7, 224, 255, 255, 255, 255, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 141, 255, 255, 255, 198, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 95, 255, 255, 198, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 255, 255, 198, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 69, 255, 255, 237, 176, 176, 176, 176, 176, 170, 0, 0,
            ],
            &[
                0, 0, 0, 69, 255, 255, 255, 255, 255, 255, 255, 255, 248, 0, 0,
            ],
            &[
                0, 0, 0, 69, 255, 255, 255, 255, 255, 255, 255, 255, 248, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ł' / 0x142
        'ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 245, 245, 245, 245, 245, 245, 20, 0, 0, 0, 0, 0],
            &[0, 0, 1, 252, 255, 255, 255, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 61, 114, 251, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 47, 78, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 255, 244, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 255, 255, 255, 255, 233, 63, 0, 0, 0],
            &[0, 0, 0, 0, 16, 150, 255, 255, 255, 171, 16, 0, 0, 0, 0],
            &[0, 0, 0, 63, 232, 255, 255, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 247, 240, 255, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 23, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 245, 255, 255, 21, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 67, 110, 250, 255, 255, 124, 67, 20, 0, 0, 0],
            &[
                0, 0, 180, 255, 255, 255, 255, 255, 255, 255, 255, 255, 132, 0, 0,
            ],
            &[
                0, 0, 189, 255, 255, 255, 255, 255, 255, 255, 255, 255, 140, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ń' / 0x143
        'Ń' => &[
            &[0, 0, 0, 0, 0, 0, 0, 24, 156, 157, 156, 33, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 168, 255, 255, 117, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 88, 255, 253, 105, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 175, 190, 77, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 254, 254, 62, 0, 0, 13, 254, 254, 166, 0, 0],
            &[
                0, 14, 255, 255, 255, 255, 145, 0, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 250, 255, 228, 1, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 188, 255, 255, 57, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 141, 234, 255, 141, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 152, 154, 255, 224, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 162, 71, 255, 255, 52, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 166, 5, 238, 255, 136, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 161, 255, 219, 10, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 78, 255, 255, 52, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 8, 243, 255, 131, 251, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 168, 255, 214, 239, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 85, 255, 255, 254, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 11, 246, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 14, 255, 255, 167, 0, 0, 0, 175, 255, 255, 255, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ń' / 0x144
        'ń' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 22, 22, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 88, 255, 255, 218, 23, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 21, 233, 255, 214, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 179, 255, 197, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 70, 69, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 100, 119, 64, 0, 44, 134, 165, 150, 75, 0, 0, 0, 0],
            &[
                0, 0, 216, 255, 203, 102, 254, 255, 255, 255, 255, 146, 0, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 255, 255, 229, 251, 255, 255, 255, 49, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 237, 47, 0, 34, 236, 255, 255, 117, 0, 0,
            ],
            &[0, 0, 216, 255, 255, 135, 0, 0, 0, 152, 255, 255, 138, 0, 0],
            &[0, 0, 216, 255, 255, 76, 0, 0, 0, 128, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 55, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ņ' / 0x145
        'Ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 254, 254, 62, 0, 0, 13, 254, 254, 166, 0, 0],
            &[
                0, 14, 255, 255, 255, 255, 145, 0, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 250, 255, 228, 1, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 188, 255, 255, 57, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 141, 234, 255, 141, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 152, 154, 255, 224, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 162, 71, 255, 255, 52, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 166, 5, 238, 255, 136, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 161, 255, 219, 10, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 78, 255, 255, 52, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 8, 243, 255, 131, 251, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 168, 255, 214, 239, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 85, 255, 255, 254, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 11, 246, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 14, 255, 255, 167, 0, 0, 0, 175, 255, 255, 255, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 33, 183, 183, 106, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 84, 255, 254, 48, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 123, 163, 26, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ņ' / 0x146
        'ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 100, 119, 64, 0, 44, 134, 165, 150, 75, 0, 0, 0, 0],
            &[
                0, 0, 216, 255, 203, 102, 254, 255, 255, 255, 255, 146, 0, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 255, 255, 229, 251, 255, 255, 255, 49, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 237, 47, 0, 34, 236, 255, 255, 117, 0, 0,
            ],
            &[0, 0, 216, 255, 255, 135, 0, 0, 0, 152, 255, 255, 138, 0, 0],
            &[0, 0, 216, 255, 255, 76, 0, 0, 0, 128, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 55, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 18, 183, 183, 122, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 63, 255, 255, 68, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 109, 164, 39, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ň' / 0x147
        'Ň' => &[
            &[0, 0, 0, 56, 157, 123, 3, 0, 16, 142, 156, 26, 0, 0, 0],
            &[0, 0, 0, 3, 183, 255, 181, 36, 217, 255, 141, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 227, 255, 251, 255, 197, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 70, 190, 190, 190, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 254, 254, 62, 0, 0, 13, 254, 254, 166, 0, 0],
            &[
                0, 14, 255, 255, 255, 255, 145, 0, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 250, 255, 228, 1, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 188, 255, 255, 57, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 141, 234, 255, 141, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 152, 154, 255, 224, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 162, 71, 255, 255, 52, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 166, 5, 238, 255, 136, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 161, 255, 219, 10, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 78, 255, 255, 52, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 8, 243, 255, 131, 251, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 168, 255, 214, 239, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 85, 255, 255, 254, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 11, 246, 255, 255, 255, 167, 0, 0,
            ],
            &[0, 14, 255, 255, 167, 0, 0, 0, 175, 255, 255, 255, 167, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ň' / 0x148
        'ň' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 22, 10, 0, 0, 0, 15, 22, 4, 0, 0, 0],
            &[0, 0, 0, 53, 247, 231, 56, 0, 104, 249, 224, 18, 0, 0, 0],
            &[0, 0, 0, 0, 96, 255, 244, 154, 255, 248, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 167, 255, 255, 255, 110, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 16, 70, 70, 68, 1, 0, 0, 0, 0, 0],
            &[0, 0, 100, 119, 64, 0, 44, 134, 165, 150, 75, 0, 0, 0, 0],
            &[
                0, 0, 216, 255, 203, 102, 254, 255, 255, 255, 255, 146, 0, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 255, 255, 229, 251, 255, 255, 255, 49, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 237, 47, 0, 34, 236, 255, 255, 117, 0, 0,
            ],
            &[0, 0, 216, 255, 255, 135, 0, 0, 0, 152, 255, 255, 138, 0, 0],
            &[0, 0, 216, 255, 255, 76, 0, 0, 0, 128, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 55, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŉ' / 0x149
        'ŉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 114, 254, 254, 140, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 160, 255, 255, 71, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 209, 255, 233, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                9, 251, 255, 140, 75, 119, 90, 0, 29, 126, 163, 153, 80, 0, 0,
            ],
            &[
                62, 255, 253, 34, 162, 255, 245, 72, 247, 255, 255, 255, 255, 145, 0,
            ],
            &[
                14, 38, 34, 0, 162, 255, 255, 255, 255, 232, 250, 255, 255, 253, 43,
            ],
            &[
                0, 0, 0, 0, 162, 255, 255, 252, 78, 0, 29, 233, 255, 255, 104,
            ],
            &[0, 0, 0, 0, 162, 255, 255, 186, 0, 0, 0, 151, 255, 255, 135],
            &[0, 0, 0, 0, 162, 255, 255, 129, 0, 0, 0, 127, 255, 255, 140],
            &[0, 0, 0, 0, 162, 255, 255, 108, 0, 0, 0, 126, 255, 255, 140],
            &[0, 0, 0, 0, 162, 255, 255, 104, 0, 0, 0, 126, 255, 255, 140],
            &[0, 0, 0, 0, 162, 255, 255, 104, 0, 0, 0, 126, 255, 255, 140],
            &[0, 0, 0, 0, 162, 255, 255, 104, 0, 0, 0, 126, 255, 255, 140],
            &[0, 0, 0, 0, 162, 255, 255, 104, 0, 0, 0, 126, 255, 255, 140],
            &[0, 0, 0, 0, 162, 255, 255, 104, 0, 0, 0, 126, 255, 255, 140],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 254, 254, 62, 0, 0, 13, 254, 254, 166, 0, 0],
            &[
                0, 14, 255, 255, 255, 255, 145, 0, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 250, 255, 228, 1, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 188, 255, 255, 57, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 141, 234, 255, 141, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 152, 154, 255, 224, 0, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 162, 71, 255, 255, 52, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 166, 5, 238, 255, 136, 13, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 161, 255, 219, 10, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 78, 255, 255, 52, 255, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 8, 243, 255, 131, 251, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 168, 255, 214, 239, 255, 167, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 85, 255, 255, 255, 255, 219, 0, 0,
            ],
            &[
                0, 14, 255, 255, 167, 0, 0, 11, 246, 255, 255, 255, 255, 0, 0,
            ],
            &[0, 14, 255, 255, 167, 0, 0, 0, 175, 255, 255, 255, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 243, 255, 157, 0, 0],
            &[0, 0, 0, 0, 0, 0, 1, 0, 1, 119, 255, 255, 115, 0, 0],
            &[0, 0, 0, 0, 0, 0, 243, 220, 245, 255, 255, 238, 22, 0, 0],
            &[0, 0, 0, 0, 0, 0, 58, 86, 84, 45, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŋ' / 0x14b
        'ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 100, 119, 64, 0, 44, 134, 165, 150, 75, 0, 0, 0, 0],
            &[
                0, 0, 216, 255, 203, 102, 254, 255, 255, 255, 255, 146, 0, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 255, 255, 229, 251, 255, 255, 255, 49, 0, 0,
            ],
            &[
                0, 0, 216, 255, 255, 237, 47, 0, 34, 236, 255, 255, 117, 0, 0,
            ],
            &[0, 0, 216, 255, 255, 135, 0, 0, 0, 152, 255, 255, 138, 0, 0],
            &[0, 0, 216, 255, 255, 76, 0, 0, 0, 128, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 55, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 168, 255, 255, 187, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 253, 255, 255, 255, 0, 0],
            &[0, 0, 216, 255, 255, 51, 0, 0, 0, 253, 255, 255, 255, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 126, 255, 255, 140, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 147, 255, 255, 134, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 117, 129, 241, 255, 255, 96, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ō' / 0x14c
        'Ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 152, 171, 171, 171, 171, 171, 80, 0, 0, 0, 0],
            &[0, 0, 0, 0, 228, 255, 255, 255, 255, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 104, 104, 104, 104, 104, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 36, 55, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 133, 241, 255, 255, 255, 212, 73, 0, 0, 0, 0],
            &[0, 0, 2, 177, 255, 255, 255, 255, 255, 255, 254, 85, 0, 0, 0],
            &[
                0, 0, 89, 255, 255, 250, 147, 116, 202, 255, 255, 236, 15, 0, 0,
            ],
            &[0, 0, 202, 255, 255, 117, 0, 0, 12, 217, 255, 255, 101, 0, 0],
            &[0, 14, 252, 255, 251, 16, 0, 0, 0, 124, 255, 255, 172, 0, 0],
            &[0, 59, 255, 255, 217, 0, 0, 0, 0, 66, 255, 255, 214, 0, 0],
            &[0, 77, 255, 255, 194, 0, 0, 0, 0, 46, 255, 255, 235, 0, 0],
            &[0, 91, 255, 255, 186, 0, 0, 0, 0, 33, 255, 255, 245, 0, 0],
            &[0, 82, 255, 255, 194, 0, 0, 0, 0, 41, 255, 255, 235, 0, 0],
            &[0, 60, 255, 255, 219, 0, 0, 0, 0, 67, 255, 255, 213, 0, 0],
            &[0, 18, 254, 255, 252, 18, 0, 0, 0, 120, 255, 255, 170, 0, 0],
            &[0, 0, 201, 255, 255, 123, 0, 0, 9, 219, 255, 255, 99, 0, 0],
            &[
                0, 0, 93, 255, 255, 252, 156, 119, 204, 255, 255, 233, 13, 0, 0,
            ],
            &[0, 0, 2, 176, 255, 255, 255, 255, 255, 255, 252, 79, 0, 0, 0],
            &[0, 0, 0, 6, 129, 235, 255, 255, 255, 203, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 34, 46, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ō' / 0x14d
        'ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 36, 36, 36, 36, 36, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 228, 255, 255, 255, 255, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 214, 240, 240, 240, 240, 240, 113, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 97, 148, 164, 138, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 59, 231, 255, 255, 255, 255, 255, 187, 18, 0, 0, 0],
            &[
                0, 0, 31, 242, 255, 255, 243, 211, 250, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 158, 255, 255, 190, 10, 0, 43, 241, 255, 255, 62, 0, 0],
            &[0, 1, 239, 255, 255, 54, 0, 0, 0, 147, 255, 255, 142, 0, 0],
            &[0, 20, 255, 255, 253, 4, 0, 0, 0, 97, 255, 255, 175, 0, 0],
            &[0, 35, 255, 255, 242, 0, 0, 0, 0, 88, 255, 255, 188, 0, 0],
            &[0, 15, 255, 255, 254, 14, 0, 0, 0, 114, 255, 255, 167, 0, 0],
            &[0, 0, 215, 255, 255, 98, 0, 0, 0, 193, 255, 255, 111, 0, 0],
            &[
                0, 0, 112, 255, 255, 243, 112, 73, 158, 255, 255, 244, 20, 0, 0,
            ],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 253, 93, 0, 0, 0],
            &[0, 0, 0, 7, 132, 236, 255, 255, 255, 205, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 35, 47, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => &[
            &[0, 0, 0, 28, 173, 109, 0, 0, 4, 169, 131, 0, 0, 0, 0],
            &[0, 0, 0, 6, 247, 242, 109, 81, 166, 255, 146, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 255, 255, 255, 255, 231, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 159, 177, 132, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 36, 55, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 133, 241, 255, 255, 255, 212, 73, 0, 0, 0, 0],
            &[0, 0, 2, 177, 255, 255, 255, 255, 255, 255, 254, 85, 0, 0, 0],
            &[
                0, 0, 89, 255, 255, 250, 147, 116, 202, 255, 255, 236, 15, 0, 0,
            ],
            &[0, 0, 202, 255, 255, 117, 0, 0, 12, 217, 255, 255, 101, 0, 0],
            &[0, 14, 252, 255, 251, 16, 0, 0, 0, 124, 255, 255, 172, 0, 0],
            &[0, 59, 255, 255, 217, 0, 0, 0, 0, 66, 255, 255, 214, 0, 0],
            &[0, 77, 255, 255, 194, 0, 0, 0, 0, 46, 255, 255, 235, 0, 0],
            &[0, 91, 255, 255, 186, 0, 0, 0, 0, 33, 255, 255, 245, 0, 0],
            &[0, 82, 255, 255, 194, 0, 0, 0, 0, 41, 255, 255, 235, 0, 0],
            &[0, 60, 255, 255, 219, 0, 0, 0, 0, 67, 255, 255, 213, 0, 0],
            &[0, 18, 254, 255, 252, 18, 0, 0, 0, 120, 255, 255, 170, 0, 0],
            &[0, 0, 201, 255, 255, 123, 0, 0, 9, 219, 255, 255, 99, 0, 0],
            &[
                0, 0, 93, 255, 255, 252, 156, 119, 204, 255, 255, 233, 13, 0, 0,
            ],
            &[0, 0, 2, 176, 255, 255, 255, 255, 255, 255, 252, 79, 0, 0, 0],
            &[0, 0, 0, 6, 129, 235, 255, 255, 255, 203, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 34, 46, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŏ' / 0x14f
        'ŏ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 38, 20, 0, 0, 0, 35, 31, 0, 0, 0, 0],
            &[0, 0, 0, 27, 255, 195, 5, 0, 41, 254, 178, 0, 0, 0, 0],
            &[0, 0, 0, 0, 203, 255, 225, 201, 250, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 206, 255, 255, 250, 130, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 38, 57, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 97, 148, 164, 138, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 59, 231, 255, 255, 255, 255, 255, 187, 18, 0, 0, 0],
            &[
                0, 0, 31, 242, 255, 255, 243, 211, 250, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 158, 255, 255, 190, 10, 0, 43, 241, 255, 255, 62, 0, 0],
            &[0, 1, 239, 255, 255, 54, 0, 0, 0, 147, 255, 255, 142, 0, 0],
            &[0, 20, 255, 255, 253, 4, 0, 0, 0, 97, 255, 255, 175, 0, 0],
            &[0, 35, 255, 255, 242, 0, 0, 0, 0, 88, 255, 255, 188, 0, 0],
            &[0, 15, 255, 255, 254, 14, 0, 0, 0, 114, 255, 255, 167, 0, 0],
            &[0, 0, 215, 255, 255, 98, 0, 0, 0, 193, 255, 255, 111, 0, 0],
            &[
                0, 0, 112, 255, 255, 243, 112, 73, 158, 255, 255, 244, 20, 0, 0,
            ],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 253, 93, 0, 0, 0],
            &[0, 0, 0, 7, 132, 236, 255, 255, 255, 205, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 35, 47, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ő' / 0x150
        'Ő' => &[
            &[0, 0, 0, 0, 0, 131, 157, 151, 23, 155, 157, 123, 0, 0, 0],
            &[0, 0, 0, 0, 73, 255, 255, 107, 131, 255, 245, 55, 0, 0, 0],
            &[0, 0, 0, 6, 217, 255, 116, 34, 247, 244, 65, 0, 0, 0, 0],
            &[0, 0, 0, 70, 190, 89, 0, 116, 187, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 36, 55, 30, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 133, 241, 255, 255, 255, 212, 73, 0, 0, 0, 0],
            &[0, 0, 2, 177, 255, 255, 255, 255, 255, 255, 254, 85, 0, 0, 0],
            &[
                0, 0, 89, 255, 255, 250, 147, 116, 202, 255, 255, 236, 15, 0, 0,
            ],
            &[0, 0, 202, 255, 255, 117, 0, 0, 12, 217, 255, 255, 101, 0, 0],
            &[0, 14, 252, 255, 251, 16, 0, 0, 0, 124, 255, 255, 172, 0, 0],
            &[0, 59, 255, 255, 217, 0, 0, 0, 0, 66, 255, 255, 214, 0, 0],
            &[0, 77, 255, 255, 194, 0, 0, 0, 0, 46, 255, 255, 235, 0, 0],
            &[0, 91, 255, 255, 186, 0, 0, 0, 0, 33, 255, 255, 245, 0, 0],
            &[0, 82, 255, 255, 194, 0, 0, 0, 0, 41, 255, 255, 235, 0, 0],
            &[0, 60, 255, 255, 219, 0, 0, 0, 0, 67, 255, 255, 213, 0, 0],
            &[0, 18, 254, 255, 252, 18, 0, 0, 0, 120, 255, 255, 170, 0, 0],
            &[0, 0, 201, 255, 255, 123, 0, 0, 9, 219, 255, 255, 99, 0, 0],
            &[
                0, 0, 93, 255, 255, 252, 156, 119, 204, 255, 255, 233, 13, 0, 0,
            ],
            &[0, 0, 2, 176, 255, 255, 255, 255, 255, 255, 252, 79, 0, 0, 0],
            &[0, 0, 0, 6, 129, 235, 255, 255, 255, 203, 65, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 34, 46, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ő' / 0x151
        'ő' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 22, 22, 3, 20, 22, 19, 0, 0, 0],
            &[0, 0, 0, 0, 16, 237, 255, 206, 64, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 0, 143, 255, 221, 31, 200, 255, 182, 7, 0, 0, 0],
            &[0, 0, 0, 44, 251, 210, 29, 100, 255, 173, 7, 0, 0, 0, 0],
            &[0, 0, 0, 32, 70, 13, 0, 49, 66, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 10, 97, 148, 164, 138, 66, 1, 0, 0, 0, 0],
            &[0, 0, 0, 59, 231, 255, 255, 255, 255, 255, 187, 18, 0, 0, 0],
            &[
                0, 0, 31, 242, 255, 255, 243, 211, 250, 255, 255, 182, 0, 0, 0,
            ],
            &[0, 0, 158, 255, 255, 190, 10, 0, 43, 241, 255, 255, 62, 0, 0],
            &[0, 1, 239, 255, 255, 54, 0, 0, 0, 147, 255, 255, 142, 0, 0],
            &[0, 20, 255, 255, 253, 4, 0, 0, 0, 97, 255, 255, 175, 0, 0],
            &[0, 35, 255, 255, 242, 0, 0, 0, 0, 88, 255, 255, 188, 0, 0],
            &[0, 15, 255, 255, 254, 14, 0, 0, 0, 114, 255, 255, 167, 0, 0],
            &[0, 0, 215, 255, 255, 98, 0, 0, 0, 193, 255, 255, 111, 0, 0],
            &[
                0, 0, 112, 255, 255, 243, 112, 73, 158, 255, 255, 244, 20, 0, 0,
            ],
            &[0, 0, 4, 189, 255, 255, 255, 255, 255, 255, 253, 93, 0, 0, 0],
            &[0, 0, 0, 7, 132, 236, 255, 255, 255, 205, 71, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 35, 47, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Œ' / 0x152
        'Œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 25, 48, 20, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 42, 200, 255, 255, 255, 254, 254, 254, 254, 254, 45, 0,
            ],
            &[
                0, 0, 22, 232, 255, 255, 255, 255, 255, 255, 255, 255, 255, 46, 0,
            ],
            &[
                0, 0, 139, 255, 255, 232, 133, 203, 255, 255, 186, 156, 156, 28, 0,
            ],
            &[0, 0, 227, 255, 255, 68, 0, 141, 255, 255, 77, 0, 0, 0, 0],
            &[0, 31, 255, 255, 235, 0, 0, 141, 255, 255, 77, 0, 0, 0, 0],
            &[0, 66, 255, 255, 187, 0, 0, 141, 255, 255, 94, 23, 19, 0, 0],
            &[
                0, 84, 255, 255, 171, 0, 0, 141, 255, 255, 255, 255, 205, 0, 0,
            ],
            &[
                0, 91, 255, 255, 159, 0, 0, 141, 255, 255, 255, 255, 205, 0, 0,
            ],
            &[
                0, 80, 255, 255, 163, 0, 0, 141, 255, 255, 168, 130, 104, 0, 0,
            ],
            &[0, 66, 255, 255, 179, 0, 0, 141, 255, 255, 77, 0, 0, 0, 0],
            &[0, 27, 255, 255, 214, 0, 0, 141, 255, 255, 77, 0, 0, 0, 0],
            &[0, 0, 231, 255, 254, 38, 0, 141, 255, 255, 77, 0, 0, 0, 0],
            &[
                0, 0, 140, 255, 255, 219, 124, 205, 255, 255, 185, 154, 154, 27, 0,
            ],
            &[
                0, 0, 25, 232, 255, 255, 255, 255, 255, 255, 255, 255, 255, 46, 0,
            ],
            &[
                0, 0, 0, 48, 204, 255, 255, 255, 255, 255, 255, 255, 255, 46, 0,
            ],
            &[0, 0, 0, 0, 0, 22, 47, 19, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'œ' / 0x153
        'œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 64, 150, 152, 64, 0, 24, 130, 147, 67, 0, 0, 0],
            &[
                0, 0, 99, 254, 255, 255, 254, 108, 229, 255, 255, 255, 92, 0, 0,
            ],
            &[
                0, 12, 239, 255, 203, 199, 255, 255, 255, 184, 169, 255, 227, 3, 0,
            ],
            &[
                0, 80, 255, 255, 38, 16, 248, 255, 252, 20, 13, 250, 255, 47, 0,
            ],
            &[0, 128, 255, 243, 0, 0, 198, 255, 224, 0, 0, 230, 255, 84, 0],
            &[
                0, 151, 255, 226, 0, 0, 179, 255, 250, 227, 227, 251, 255, 97, 0,
            ],
            &[
                0, 154, 255, 222, 0, 0, 170, 255, 254, 251, 251, 251, 251, 97, 0,
            ],
            &[0, 138, 255, 232, 0, 0, 187, 255, 215, 0, 0, 0, 0, 0, 0],
            &[0, 100, 255, 253, 9, 0, 220, 255, 245, 5, 0, 0, 0, 0, 0],
            &[
                0, 37, 254, 255, 94, 75, 255, 255, 255, 138, 2, 8, 104, 16, 0,
            ],
            &[
                0, 0, 178, 255, 255, 255, 255, 219, 255, 255, 241, 251, 255, 24, 0,
            ],
            &[
                0, 0, 21, 195, 255, 255, 200, 23, 133, 252, 255, 254, 178, 11, 0,
            ],
            &[0, 0, 0, 0, 31, 36, 0, 0, 0, 20, 44, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŕ' / 0x154
        'Ŕ' => &[
            &[0, 0, 0, 0, 0, 0, 40, 157, 157, 153, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 193, 255, 252, 93, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 115, 255, 248, 82, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 14, 184, 188, 59, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 254, 254, 254, 253, 237, 204, 127, 14, 0, 0, 0, 0, 0],
            &[
                0, 81, 255, 255, 255, 255, 255, 255, 255, 225, 29, 0, 0, 0, 0,
            ],
            &[
                0, 81, 255, 255, 226, 153, 179, 248, 255, 255, 177, 0, 0, 0, 0,
            ],
            &[0, 81, 255, 255, 185, 0, 0, 57, 255, 255, 253, 8, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 0, 246, 255, 255, 32, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 37, 255, 255, 253, 13, 0, 0, 0],
            &[0, 81, 255, 255, 201, 62, 99, 218, 255, 255, 188, 0, 0, 0, 0],
            &[
                0, 81, 255, 255, 255, 255, 255, 255, 255, 236, 42, 0, 0, 0, 0,
            ],
            &[0, 81, 255, 255, 255, 255, 255, 255, 216, 26, 0, 0, 0, 0, 0],
            &[0, 81, 255, 255, 210, 91, 237, 255, 247, 33, 0, 0, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 112, 255, 255, 173, 0, 0, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 6, 220, 255, 255, 65, 0, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 85, 255, 255, 210, 3, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 0, 198, 255, 255, 106, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 0, 58, 255, 255, 236, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŕ' / 0x155
        'ŕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 10, 22, 22, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 201, 255, 255, 126, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 115, 255, 255, 126, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 248, 252, 105, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 36, 70, 46, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 68, 119, 119, 119, 119, 24, 0, 66, 148, 166, 132, 18, 0,
            ],
            &[
                0, 0, 146, 255, 255, 255, 255, 95, 127, 255, 255, 255, 247, 10, 0,
            ],
            &[
                0, 0, 108, 218, 251, 255, 255, 213, 254, 255, 255, 255, 183, 0, 0,
            ],
            &[0, 0, 0, 0, 42, 255, 255, 255, 251, 135, 71, 81, 59, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 255, 107, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 250, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 230, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 227, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 227, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 60, 133, 255, 255, 241, 100, 46, 3, 0, 0, 0, 0],
            &[
                0, 32, 251, 255, 255, 255, 255, 255, 255, 255, 220, 0, 0, 0, 0,
            ],
            &[
                0, 35, 255, 255, 255, 255, 255, 255, 255, 255, 228, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŗ' / 0x156
        'Ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 254, 254, 254, 253, 237, 204, 127, 14, 0, 0, 0, 0, 0],
            &[
                0, 81, 255, 255, 255, 255, 255, 255, 255, 225, 29, 0, 0, 0, 0,
            ],
            &[
                0, 81, 255, 255, 226, 153, 179, 248, 255, 255, 177, 0, 0, 0, 0,
            ],
            &[0, 81, 255, 255, 185, 0, 0, 57, 255, 255, 253, 8, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 0, 246, 255, 255, 32, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 37, 255, 255, 253, 13, 0, 0, 0],
            &[0, 81, 255, 255, 201, 62, 99, 218, 255, 255, 188, 0, 0, 0, 0],
            &[
                0, 81, 255, 255, 255, 255, 255, 255, 255, 236, 42, 0, 0, 0, 0,
            ],
            &[0, 81, 255, 255, 255, 255, 255, 255, 216, 26, 0, 0, 0, 0, 0],
            &[0, 81, 255, 255, 210, 91, 237, 255, 247, 33, 0, 0, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 112, 255, 255, 173, 0, 0, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 6, 220, 255, 255, 65, 0, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 85, 255, 255, 210, 3, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 0, 198, 255, 255, 106, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 0, 58, 255, 255, 236, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 121, 183, 183, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 206, 255, 181, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 164, 112, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŗ' / 0x157
        'ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 68, 119, 119, 119, 119, 24, 0, 66, 148, 166, 132, 18, 0,
            ],
            &[
                0, 0, 146, 255, 255, 255, 255, 95, 127, 255, 255, 255, 247, 10, 0,
            ],
            &[
                0, 0, 108, 218, 251, 255, 255, 213, 254, 255, 255, 255, 183, 0, 0,
            ],
            &[0, 0, 0, 0, 42, 255, 255, 255, 251, 135, 71, 81, 59, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 255, 107, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 250, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 230, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 227, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 227, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 60, 133, 255, 255, 241, 100, 46, 3, 0, 0, 0, 0],
            &[
                0, 32, 251, 255, 255, 255, 255, 255, 255, 255, 220, 0, 0, 0, 0,
            ],
            &[
                0, 35, 255, 255, 255, 255, 255, 255, 255, 255, 228, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 183, 183, 136, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 255, 87, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 96, 164, 52, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ř' / 0x158
        'Ř' => &[
            &[0, 0, 72, 157, 110, 0, 0, 26, 149, 154, 13, 0, 0, 0, 0],
            &[0, 0, 9, 203, 255, 159, 45, 230, 255, 115, 0, 0, 0, 0, 0],
            &[0, 0, 0, 33, 240, 255, 251, 255, 174, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 90, 190, 190, 185, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 254, 254, 254, 253, 237, 204, 127, 14, 0, 0, 0, 0, 0],
            &[
                0, 81, 255, 255, 255, 255, 255, 255, 255, 225, 29, 0, 0, 0, 0,
            ],
            &[
                0, 81, 255, 255, 226, 153, 179, 248, 255, 255, 177, 0, 0, 0, 0,
            ],
            &[0, 81, 255, 255, 185, 0, 0, 57, 255, 255, 253, 8, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 0, 246, 255, 255, 32, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 37, 255, 255, 253, 13, 0, 0, 0],
            &[0, 81, 255, 255, 201, 62, 99, 218, 255, 255, 188, 0, 0, 0, 0],
            &[
                0, 81, 255, 255, 255, 255, 255, 255, 255, 236, 42, 0, 0, 0, 0,
            ],
            &[0, 81, 255, 255, 255, 255, 255, 255, 216, 26, 0, 0, 0, 0, 0],
            &[0, 81, 255, 255, 210, 91, 237, 255, 247, 33, 0, 0, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 112, 255, 255, 173, 0, 0, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 6, 220, 255, 255, 65, 0, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 85, 255, 255, 210, 3, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 0, 198, 255, 255, 106, 0, 0, 0],
            &[0, 81, 255, 255, 185, 0, 0, 0, 58, 255, 255, 236, 17, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ř' / 0x159
        'ř' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 20, 21, 0, 0, 0, 3, 22, 17, 0, 0, 0],
            &[0, 0, 0, 0, 161, 255, 165, 7, 21, 193, 255, 127, 0, 0, 0],
            &[0, 0, 0, 0, 7, 205, 255, 185, 213, 255, 180, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 246, 255, 255, 231, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 48, 70, 70, 38, 0, 0, 0, 0, 0],
            &[
                0, 0, 68, 119, 119, 119, 119, 24, 0, 66, 148, 166, 132, 18, 0,
            ],
            &[
                0, 0, 146, 255, 255, 255, 255, 95, 127, 255, 255, 255, 247, 10, 0,
            ],
            &[
                0, 0, 108, 218, 251, 255, 255, 213, 254, 255, 255, 255, 183, 0, 0,
            ],
            &[0, 0, 0, 0, 42, 255, 255, 255, 251, 135, 71, 81, 59, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 255, 107, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 250, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 230, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 227, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 255, 227, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 60, 133, 255, 255, 241, 100, 46, 3, 0, 0, 0, 0],
            &[
                0, 32, 251, 255, 255, 255, 255, 255, 255, 255, 220, 0, 0, 0, 0,
            ],
            &[
                0, 35, 255, 255, 255, 255, 255, 255, 255, 255, 228, 0, 0, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ś' / 0x15a
        'Ś' => &[
            &[0, 0, 0, 0, 0, 0, 0, 4, 146, 157, 157, 63, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 120, 255, 255, 162, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 46, 249, 255, 148, 1, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 142, 190, 113, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 46, 32, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 222, 255, 255, 255, 254, 212, 129, 9, 0, 0],
            &[
                0, 0, 0, 130, 255, 255, 255, 255, 255, 255, 255, 221, 1, 0, 0,
            ],
            &[
                0, 0, 14, 247, 255, 255, 183, 96, 102, 146, 227, 118, 0, 0, 0,
            ],
            &[0, 0, 49, 255, 255, 245, 5, 0, 0, 0, 1, 6, 0, 0, 0],
            &[0, 0, 40, 255, 255, 250, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 222, 255, 255, 203, 44, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 249, 255, 255, 253, 165, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 217, 255, 255, 255, 247, 131, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 124, 240, 255, 255, 255, 169, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 17, 158, 255, 255, 255, 56, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 213, 255, 255, 104, 0, 0],
            &[0, 0, 47, 15, 0, 0, 0, 0, 0, 210, 255, 255, 95, 0, 0],
            &[
                0, 0, 135, 244, 172, 113, 86, 99, 174, 255, 255, 252, 30, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 122, 0, 0, 0,
            ],
            &[0, 0, 56, 179, 244, 255, 255, 255, 254, 202, 84, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 21, 45, 43, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ś' / 0x15b
        'ś' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 21, 22, 22, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 82, 255, 255, 221, 25, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 19, 231, 255, 217, 30, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 174, 255, 200, 22, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 70, 70, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 112, 156, 167, 146, 119, 64, 3, 0, 0, 0],
            &[0, 0, 0, 89, 249, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0],
            &[
                0, 0, 12, 242, 255, 255, 243, 218, 242, 255, 255, 97, 0, 0, 0,
            ],
            &[0, 0, 50, 255, 255, 248, 13, 0, 0, 32, 107, 10, 0, 0, 0],
            &[0, 0, 28, 254, 255, 255, 165, 48, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 255, 255, 255, 255, 208, 98, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 214, 255, 255, 255, 255, 212, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 70, 172, 254, 255, 255, 167, 0, 0, 0],
            &[0, 0, 19, 13, 0, 0, 0, 0, 102, 255, 255, 224, 0, 0, 0],
            &[
                0, 0, 81, 247, 177, 120, 93, 102, 194, 255, 255, 211, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 47, 192, 249, 255, 255, 255, 255, 217, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 31, 48, 42, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => &[
            &[0, 0, 0, 0, 0, 22, 156, 157, 157, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 172, 255, 255, 255, 217, 12, 0, 0, 0, 0],
            &[0, 0, 0, 0, 111, 255, 231, 60, 205, 255, 167, 0, 0, 0, 0],
            &[0, 0, 0, 26, 187, 176, 31, 0, 12, 155, 190, 60, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 46, 32, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 222, 255, 255, 255, 254, 212, 129, 9, 0, 0],
            &[
                0, 0, 0, 130, 255, 255, 255, 255, 255, 255, 255, 221, 1, 0, 0,
            ],
            &[
                0, 0, 14, 247, 255, 255, 183, 96, 102, 146, 227, 118, 0, 0, 0,
            ],
            &[0, 0, 49, 255, 255, 245, 5, 0, 0, 0, 1, 6, 0, 0, 0],
            &[0, 0, 40, 255, 255, 250, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 222, 255, 255, 203, 44, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 249, 255, 255, 253, 165, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 217, 255, 255, 255, 247, 131, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 124, 240, 255, 255, 255, 169, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 17, 158, 255, 255, 255, 56, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 213, 255, 255, 104, 0, 0],
            &[0, 0, 47, 15, 0, 0, 0, 0, 0, 210, 255, 255, 95, 0, 0],
            &[
                0, 0, 135, 244, 172, 113, 86, 99, 174, 255, 255, 252, 30, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 122, 0, 0, 0,
            ],
            &[0, 0, 56, 179, 244, 255, 255, 255, 254, 202, 84, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 21, 45, 43, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŝ' / 0x15d
        'ŝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 3, 22, 22, 21, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 129, 255, 255, 255, 87, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 251, 254, 194, 255, 238, 32, 0, 0, 0, 0],
            &[0, 0, 0, 28, 230, 250, 99, 1, 140, 255, 205, 9, 0, 0, 0],
            &[0, 0, 0, 30, 70, 42, 0, 0, 0, 54, 70, 17, 0, 0, 0],
            &[0, 0, 0, 0, 28, 112, 156, 167, 146, 119, 64, 3, 0, 0, 0],
            &[0, 0, 0, 89, 249, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0],
            &[
                0, 0, 12, 242, 255, 255, 243, 218, 242, 255, 255, 97, 0, 0, 0,
            ],
            &[0, 0, 50, 255, 255, 248, 13, 0, 0, 32, 107, 10, 0, 0, 0],
            &[0, 0, 28, 254, 255, 255, 165, 48, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 255, 255, 255, 255, 208, 98, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 214, 255, 255, 255, 255, 212, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 70, 172, 254, 255, 255, 167, 0, 0, 0],
            &[0, 0, 19, 13, 0, 0, 0, 0, 102, 255, 255, 224, 0, 0, 0],
            &[
                0, 0, 81, 247, 177, 120, 93, 102, 194, 255, 255, 211, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 47, 192, 249, 255, 255, 255, 255, 217, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 31, 48, 42, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ş' / 0x15e
        'Ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 46, 32, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 222, 255, 255, 255, 254, 212, 129, 9, 0, 0],
            &[
                0, 0, 0, 130, 255, 255, 255, 255, 255, 255, 255, 221, 1, 0, 0,
            ],
            &[
                0, 0, 14, 247, 255, 255, 183, 96, 102, 146, 227, 118, 0, 0, 0,
            ],
            &[0, 0, 49, 255, 255, 245, 5, 0, 0, 0, 1, 6, 0, 0, 0],
            &[0, 0, 40, 255, 255, 250, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 222, 255, 255, 203, 44, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 249, 255, 255, 253, 165, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 217, 255, 255, 255, 247, 131, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 124, 240, 255, 255, 255, 169, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 17, 158, 255, 255, 255, 56, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 213, 255, 255, 104, 0, 0],
            &[0, 0, 47, 15, 0, 0, 0, 0, 0, 210, 255, 255, 95, 0, 0],
            &[
                0, 0, 135, 244, 172, 113, 86, 99, 174, 255, 255, 252, 30, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 122, 0, 0, 0,
            ],
            &[0, 0, 56, 179, 244, 255, 255, 255, 255, 202, 84, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 21, 223, 255, 29, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 211, 255, 232, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 195, 255, 130, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ş' / 0x15f
        'ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 112, 156, 167, 146, 119, 64, 3, 0, 0, 0],
            &[0, 0, 0, 89, 249, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0],
            &[
                0, 0, 12, 242, 255, 255, 243, 218, 242, 255, 255, 97, 0, 0, 0,
            ],
            &[0, 0, 50, 255, 255, 248, 13, 0, 0, 32, 107, 10, 0, 0, 0],
            &[0, 0, 28, 254, 255, 255, 165, 48, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 255, 255, 255, 255, 208, 98, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 214, 255, 255, 255, 255, 212, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 70, 172, 254, 255, 255, 167, 0, 0, 0],
            &[0, 0, 19, 13, 0, 0, 0, 0, 102, 255, 255, 224, 0, 0, 0],
            &[
                0, 0, 81, 247, 177, 120, 93, 102, 194, 255, 255, 211, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 47, 192, 249, 255, 255, 255, 255, 217, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 31, 199, 255, 51, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 195, 254, 241, 54, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 168, 255, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Š' / 0x160
        'Š' => &[
            &[0, 0, 0, 26, 156, 142, 15, 0, 5, 125, 157, 56, 0, 0, 0],
            &[0, 0, 0, 0, 138, 255, 213, 35, 187, 255, 186, 3, 0, 0, 0],
            &[0, 0, 0, 0, 3, 194, 255, 251, 255, 229, 20, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 35, 190, 190, 190, 71, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 25, 46, 32, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 222, 255, 255, 255, 254, 212, 129, 9, 0, 0],
            &[
                0, 0, 0, 130, 255, 255, 255, 255, 255, 255, 255, 221, 1, 0, 0,
            ],
            &[
                0, 0, 14, 247, 255, 255, 183, 96, 102, 146, 227, 118, 0, 0, 0,
            ],
            &[0, 0, 49, 255, 255, 245, 5, 0, 0, 0, 1, 6, 0, 0, 0],
            &[0, 0, 40, 255, 255, 250, 16, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 1, 222, 255, 255, 203, 44, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 68, 249, 255, 255, 253, 165, 31, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 59, 217, 255, 255, 255, 247, 131, 4, 0, 0, 0],
            &[0, 0, 0, 0, 0, 8, 124, 240, 255, 255, 255, 169, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 17, 158, 255, 255, 255, 56, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 213, 255, 255, 104, 0, 0],
            &[0, 0, 47, 15, 0, 0, 0, 0, 0, 210, 255, 255, 95, 0, 0],
            &[
                0, 0, 135, 244, 172, 113, 86, 99, 174, 255, 255, 252, 30, 0, 0,
            ],
            &[
                0, 0, 135, 255, 255, 255, 255, 255, 255, 255, 255, 122, 0, 0, 0,
            ],
            &[0, 0, 56, 179, 244, 255, 255, 255, 254, 202, 84, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 21, 45, 43, 13, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'š' / 0x161
        'š' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 22, 10, 0, 0, 0, 14, 22, 5, 0, 0, 0],
            &[0, 0, 0, 49, 246, 234, 60, 0, 100, 248, 227, 21, 0, 0, 0],
            &[0, 0, 0, 0, 91, 255, 245, 153, 255, 249, 53, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 162, 255, 255, 255, 116, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 70, 70, 69, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 28, 112, 156, 167, 146, 119, 64, 3, 0, 0, 0],
            &[0, 0, 0, 89, 249, 255, 255, 255, 255, 255, 255, 181, 0, 0, 0],
            &[
                0, 0, 12, 242, 255, 255, 243, 218, 242, 255, 255, 97, 0, 0, 0,
            ],
            &[0, 0, 50, 255, 255, 248, 13, 0, 0, 32, 107, 10, 0, 0, 0],
            &[0, 0, 28, 254, 255, 255, 165, 48, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 151, 255, 255, 255, 255, 208, 98, 7, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 214, 255, 255, 255, 255, 212, 26, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 70, 172, 254, 255, 255, 167, 0, 0, 0],
            &[0, 0, 19, 13, 0, 0, 0, 0, 102, 255, 255, 224, 0, 0, 0],
            &[
                0, 0, 81, 247, 177, 120, 93, 102, 194, 255, 255, 211, 0, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 113, 0, 0, 0,
            ],
            &[0, 0, 47, 192, 249, 255, 255, 255, 255, 217, 104, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 31, 48, 42, 18, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ţ' / 0x162
        'Ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 94, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 246, 0, 0,
            ],
            &[
                0, 94, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 248, 0, 0,
            ],
            &[
                0, 59, 161, 161, 161, 183, 255, 255, 238, 161, 161, 161, 156, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 177, 243, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 211, 255, 232, 36, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 195, 255, 130, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ţ' / 0x163
        'ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 82, 64, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 171, 255, 200, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 255, 200, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 93, 253, 255, 226, 119, 119, 119, 119, 10, 0, 0],
            &[
                0, 0, 211, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 0, 0,
            ],
            &[
                0, 0, 226, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 0, 0,
            ],
            &[0, 0, 12, 13, 77, 255, 255, 203, 13, 13, 13, 13, 1, 0, 0],
            &[0, 0, 0, 0, 67, 255, 255, 200, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 67, 255, 255, 200, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 67, 255, 255, 200, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 67, 255, 255, 202, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 255, 255, 241, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 253, 255, 255, 181, 97, 99, 135, 31, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 255, 255, 255, 255, 49, 0, 0],
            &[0, 0, 0, 0, 0, 14, 157, 248, 255, 255, 255, 232, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 87, 255, 150, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 133, 247, 254, 113, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 96, 255, 229, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 2, 2, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ť' / 0x164
        'Ť' => &[
            &[0, 0, 0, 72, 157, 110, 0, 0, 26, 149, 154, 13, 0, 0, 0],
            &[0, 0, 0, 9, 203, 255, 159, 45, 230, 255, 115, 0, 0, 0, 0],
            &[0, 0, 0, 0, 33, 240, 255, 251, 255, 174, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 90, 190, 190, 185, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 94, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 246, 0, 0,
            ],
            &[
                0, 94, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 248, 0, 0,
            ],
            &[
                0, 59, 161, 161, 161, 183, 255, 255, 238, 161, 161, 161, 156, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ť' / 0x165
        'ť' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 37, 48, 48, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 222, 255, 206, 0, 0, 0],
            &[0, 0, 0, 0, 45, 82, 64, 0, 16, 254, 255, 79, 0, 0, 0],
            &[0, 0, 0, 0, 171, 255, 200, 0, 73, 255, 192, 0, 0, 0, 0],
            &[0, 0, 0, 0, 217, 255, 200, 0, 16, 43, 17, 0, 0, 0, 0],
            &[0, 0, 36, 93, 253, 255, 226, 119, 119, 119, 119, 10, 0, 0, 0],
            &[
                0, 211, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 0, 0, 0,
            ],
            &[
                0, 226, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 0, 0, 0,
            ],
            &[0, 12, 13, 77, 255, 255, 203, 13, 13, 13, 13, 1, 0, 0, 0],
            &[0, 0, 0, 67, 255, 255, 200, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 255, 255, 200, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 255, 255, 200, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 67, 255, 255, 202, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 56, 255, 255, 241, 2, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 253, 255, 255, 181, 97, 99, 135, 31, 0, 0, 0],
            &[0, 0, 0, 0, 170, 255, 255, 255, 255, 255, 255, 49, 0, 0, 0],
            &[0, 0, 0, 0, 14, 157, 246, 255, 255, 255, 232, 38, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 8, 39, 47, 27, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŧ' / 0x166
        'Ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 94, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 246, 0, 0,
            ],
            &[
                0, 94, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 248, 0, 0,
            ],
            &[
                0, 59, 161, 161, 161, 183, 255, 255, 238, 161, 161, 161, 156, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 140, 172, 231, 255, 255, 255, 172, 172, 71, 0, 0, 0],
            &[
                0, 0, 0, 208, 255, 255, 255, 255, 255, 255, 255, 106, 0, 0, 0,
            ],
            &[0, 0, 0, 172, 211, 255, 255, 255, 255, 211, 211, 87, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 207, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŧ' / 0x167
        'ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 82, 64, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 171, 255, 200, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 217, 255, 200, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 93, 253, 255, 226, 119, 119, 119, 119, 10, 0, 0],
            &[
                0, 0, 211, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 0, 0,
            ],
            &[
                0, 0, 226, 255, 255, 255, 255, 255, 255, 255, 255, 255, 22, 0, 0,
            ],
            &[0, 0, 12, 13, 77, 255, 255, 203, 13, 13, 13, 13, 1, 0, 0],
            &[0, 0, 36, 41, 108, 255, 255, 241, 41, 41, 41, 41, 7, 0, 0],
            &[
                0, 0, 226, 255, 255, 255, 255, 255, 255, 255, 255, 255, 49, 0, 0,
            ],
            &[
                0, 0, 184, 208, 255, 255, 255, 255, 208, 208, 208, 208, 40, 0, 0,
            ],
            &[0, 0, 0, 0, 67, 255, 255, 202, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 56, 255, 255, 241, 2, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 253, 255, 255, 181, 97, 99, 135, 31, 0, 0],
            &[0, 0, 0, 0, 0, 170, 255, 255, 255, 255, 255, 255, 49, 0, 0],
            &[0, 0, 0, 0, 0, 14, 157, 246, 255, 255, 255, 232, 38, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 8, 39, 47, 27, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ũ' / 0x168
        'Ũ' => &[
            &[0, 0, 0, 0, 19, 82, 49, 0, 0, 30, 103, 26, 0, 0, 0],
            &[0, 0, 0, 18, 224, 255, 255, 198, 103, 182, 255, 36, 0, 0, 0],
            &[0, 0, 0, 113, 255, 203, 226, 255, 255, 255, 201, 0, 0, 0, 0],
            &[0, 0, 0, 123, 184, 2, 3, 84, 168, 158, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 251, 0, 0, 0, 0, 99, 254, 254, 166, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 13, 255, 255, 254, 1, 0, 0, 0, 102, 255, 255, 167, 0, 0],
            &[0, 2, 252, 255, 255, 21, 0, 0, 0, 123, 255, 255, 151, 0, 0],
            &[0, 0, 221, 255, 255, 99, 0, 0, 1, 196, 255, 255, 115, 0, 0],
            &[
                0, 0, 138, 255, 255, 246, 156, 115, 183, 255, 255, 251, 33, 0, 0,
            ],
            &[
                0, 0, 15, 222, 255, 255, 255, 255, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[0, 0, 0, 21, 161, 244, 255, 255, 255, 213, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 35, 45, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ũ' / 0x169
        'ũ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 108, 203, 170, 63, 0, 86, 224, 48, 0, 0, 0],
            &[0, 0, 0, 61, 255, 255, 255, 255, 224, 247, 246, 14, 0, 0, 0],
            &[0, 0, 0, 143, 253, 84, 108, 217, 255, 255, 117, 0, 0, 0, 0],
            &[0, 0, 0, 50, 65, 0, 0, 1, 47, 37, 0, 0, 0, 0, 0],
            &[0, 0, 113, 119, 119, 11, 0, 0, 0, 71, 119, 119, 53, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 168, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 31, 0, 0, 0, 194, 255, 255, 113, 0, 0],
            &[0, 0, 234, 255, 255, 75, 0, 0, 20, 246, 255, 255, 113, 0, 0],
            &[
                0, 0, 194, 255, 255, 222, 100, 99, 207, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 198, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 105, 241, 255, 255, 233, 109, 16, 249, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 35, 33, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ū' / 0x16a
        'Ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 152, 171, 171, 171, 171, 171, 80, 0, 0, 0, 0],
            &[0, 0, 0, 0, 228, 255, 255, 255, 255, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 93, 104, 104, 104, 104, 104, 49, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 251, 0, 0, 0, 0, 99, 254, 254, 166, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 13, 255, 255, 254, 1, 0, 0, 0, 102, 255, 255, 167, 0, 0],
            &[0, 2, 252, 255, 255, 21, 0, 0, 0, 123, 255, 255, 151, 0, 0],
            &[0, 0, 221, 255, 255, 99, 0, 0, 1, 196, 255, 255, 115, 0, 0],
            &[
                0, 0, 138, 255, 255, 246, 156, 115, 183, 255, 255, 251, 33, 0, 0,
            ],
            &[
                0, 0, 15, 222, 255, 255, 255, 255, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[0, 0, 0, 21, 161, 244, 255, 255, 255, 213, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 35, 45, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ū' / 0x16b
        'ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 32, 36, 36, 36, 36, 36, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 228, 255, 255, 255, 255, 255, 120, 0, 0, 0, 0],
            &[0, 0, 0, 0, 214, 240, 240, 240, 240, 240, 113, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 119, 119, 11, 0, 0, 0, 71, 119, 119, 53, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 168, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 31, 0, 0, 0, 194, 255, 255, 113, 0, 0],
            &[0, 0, 234, 255, 255, 75, 0, 0, 20, 246, 255, 255, 113, 0, 0],
            &[
                0, 0, 194, 255, 255, 222, 100, 99, 207, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 198, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 105, 241, 255, 255, 233, 109, 16, 249, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 35, 33, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => &[
            &[0, 0, 0, 28, 173, 109, 0, 0, 4, 169, 131, 0, 0, 0, 0],
            &[0, 0, 0, 6, 247, 242, 109, 81, 166, 255, 146, 0, 0, 0, 0],
            &[0, 0, 0, 0, 115, 255, 255, 255, 255, 231, 23, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 73, 159, 177, 132, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 251, 0, 0, 0, 0, 99, 254, 254, 166, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 13, 255, 255, 254, 1, 0, 0, 0, 102, 255, 255, 167, 0, 0],
            &[0, 2, 252, 255, 255, 21, 0, 0, 0, 123, 255, 255, 151, 0, 0],
            &[0, 0, 221, 255, 255, 99, 0, 0, 1, 196, 255, 255, 115, 0, 0],
            &[
                0, 0, 138, 255, 255, 246, 156, 115, 183, 255, 255, 251, 33, 0, 0,
            ],
            &[
                0, 0, 15, 222, 255, 255, 255, 255, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[0, 0, 0, 21, 161, 244, 255, 255, 255, 213, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 35, 45, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŭ' / 0x16d
        'ŭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 38, 20, 0, 0, 0, 35, 31, 0, 0, 0, 0],
            &[0, 0, 0, 27, 255, 195, 5, 0, 41, 254, 178, 0, 0, 0, 0],
            &[0, 0, 0, 0, 203, 255, 225, 201, 250, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 206, 255, 255, 250, 130, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 2, 38, 57, 16, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 119, 119, 11, 0, 0, 0, 71, 119, 119, 53, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 168, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 31, 0, 0, 0, 194, 255, 255, 113, 0, 0],
            &[0, 0, 234, 255, 255, 75, 0, 0, 20, 246, 255, 255, 113, 0, 0],
            &[
                0, 0, 194, 255, 255, 222, 100, 99, 207, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 198, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 105, 241, 255, 255, 233, 109, 16, 249, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 35, 33, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ů' / 0x16e
        'Ů' => &[
            &[0, 0, 0, 0, 18, 250, 199, 122, 251, 157, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 43, 255, 100, 0, 220, 191, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 6, 223, 242, 208, 255, 117, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 35, 155, 177, 107, 2, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 251, 0, 0, 0, 0, 99, 254, 254, 166, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 13, 255, 255, 254, 1, 0, 0, 0, 102, 255, 255, 167, 0, 0],
            &[0, 2, 252, 255, 255, 21, 0, 0, 0, 123, 255, 255, 151, 0, 0],
            &[0, 0, 221, 255, 255, 99, 0, 0, 1, 196, 255, 255, 115, 0, 0],
            &[
                0, 0, 138, 255, 255, 246, 156, 115, 183, 255, 255, 251, 33, 0, 0,
            ],
            &[
                0, 0, 15, 222, 255, 255, 255, 255, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[0, 0, 0, 21, 161, 244, 255, 255, 255, 213, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 35, 45, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 98, 240, 254, 193, 24, 0, 0, 0, 0, 0],
        ],
        // letter: 'ů' / 0x16f
        'ů' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 11, 107, 129, 59, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 202, 255, 248, 255, 89, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 40, 255, 120, 9, 228, 187, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 255, 166, 73, 243, 171, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 139, 255, 255, 231, 45, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 34, 57, 11, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 119, 119, 11, 0, 0, 0, 71, 119, 119, 53, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 168, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 31, 0, 0, 0, 194, 255, 255, 113, 0, 0],
            &[0, 0, 234, 255, 255, 75, 0, 0, 20, 246, 255, 255, 113, 0, 0],
            &[
                0, 0, 194, 255, 255, 222, 100, 99, 207, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 198, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 105, 241, 255, 255, 233, 109, 16, 249, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 35, 33, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ű' / 0x170
        'Ű' => &[
            &[0, 0, 0, 0, 0, 131, 157, 151, 23, 155, 157, 123, 0, 0, 0],
            &[0, 0, 0, 0, 73, 255, 255, 107, 131, 255, 245, 55, 0, 0, 0],
            &[0, 0, 0, 6, 217, 255, 116, 34, 247, 244, 65, 0, 0, 0, 0],
            &[0, 0, 0, 70, 190, 89, 0, 116, 187, 52, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 251, 0, 0, 0, 0, 99, 254, 254, 166, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 13, 255, 255, 254, 1, 0, 0, 0, 102, 255, 255, 167, 0, 0],
            &[0, 2, 252, 255, 255, 21, 0, 0, 0, 123, 255, 255, 151, 0, 0],
            &[0, 0, 221, 255, 255, 99, 0, 0, 1, 196, 255, 255, 115, 0, 0],
            &[
                0, 0, 138, 255, 255, 246, 156, 115, 183, 255, 255, 251, 33, 0, 0,
            ],
            &[
                0, 0, 15, 222, 255, 255, 255, 255, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[0, 0, 0, 21, 161, 244, 255, 255, 255, 213, 93, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 35, 45, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ű' / 0x171
        'ű' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 15, 22, 22, 3, 20, 22, 19, 0, 0, 0],
            &[0, 0, 0, 0, 16, 237, 255, 206, 64, 255, 255, 152, 0, 0, 0],
            &[0, 0, 0, 0, 143, 255, 221, 31, 200, 255, 182, 7, 0, 0, 0],
            &[0, 0, 0, 44, 251, 210, 29, 100, 255, 173, 7, 0, 0, 0, 0],
            &[0, 0, 0, 32, 70, 13, 0, 49, 66, 2, 0, 0, 0, 0, 0],
            &[0, 0, 113, 119, 119, 11, 0, 0, 0, 71, 119, 119, 53, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 168, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 31, 0, 0, 0, 194, 255, 255, 113, 0, 0],
            &[0, 0, 234, 255, 255, 75, 0, 0, 20, 246, 255, 255, 113, 0, 0],
            &[
                0, 0, 194, 255, 255, 222, 100, 99, 207, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 198, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 105, 241, 255, 255, 233, 109, 16, 249, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 35, 33, 3, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ų' / 0x172
        'Ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 13, 254, 254, 251, 0, 0, 0, 0, 99, 254, 254, 166, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 14, 255, 255, 253, 0, 0, 0, 0, 99, 255, 255, 167, 0, 0],
            &[0, 13, 255, 255, 254, 1, 0, 0, 0, 102, 255, 255, 167, 0, 0],
            &[0, 2, 252, 255, 255, 21, 0, 0, 0, 123, 255, 255, 151, 0, 0],
            &[0, 0, 221, 255, 255, 99, 0, 0, 1, 196, 255, 255, 115, 0, 0],
            &[
                0, 0, 138, 255, 255, 246, 156, 115, 183, 255, 255, 251, 33, 0, 0,
            ],
            &[
                0, 0, 15, 222, 255, 255, 255, 255, 255, 255, 255, 123, 0, 0, 0,
            ],
            &[0, 0, 0, 21, 161, 244, 255, 255, 255, 255, 98, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 7, 132, 255, 221, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 37, 251, 230, 14, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 108, 255, 193, 1, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 1, 142, 231, 237, 88, 0, 0, 0, 0, 0],
        ],
        // letter: 'ų' / 0x173
        'ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 113, 119, 119, 11, 0, 0, 0, 71, 119, 119, 53, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 153, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 24, 0, 0, 0, 168, 255, 255, 113, 0, 0],
            &[0, 0, 243, 255, 255, 31, 0, 0, 0, 194, 255, 255, 113, 0, 0],
            &[0, 0, 234, 255, 255, 75, 0, 0, 20, 246, 255, 255, 113, 0, 0],
            &[
                0, 0, 194, 255, 255, 222, 100, 99, 207, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 91, 255, 255, 255, 255, 255, 255, 198, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 0, 105, 241, 255, 255, 233, 109, 16, 255, 255, 113, 0, 0,
            ],
            &[0, 0, 0, 0, 2, 35, 33, 3, 0, 90, 250, 155, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 29, 245, 234, 5, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 95, 255, 208, 2, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 132, 229, 239, 103, 0, 0],
        ],
        // letter: 'Ŵ' / 0x174
        'Ŵ' => &[
            &[0, 0, 0, 0, 0, 67, 157, 157, 152, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 228, 255, 255, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 184, 255, 187, 68, 241, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 79, 190, 144, 6, 0, 42, 182, 181, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 191, 254, 229, 0, 0, 0, 0, 0, 0, 74, 254, 254, 90, 0],
            &[0, 173, 255, 252, 2, 0, 0, 0, 0, 0, 92, 255, 255, 73, 0],
            &[0, 154, 255, 255, 23, 0, 0, 0, 0, 0, 110, 255, 255, 55, 0],
            &[
                0, 135, 255, 255, 47, 19, 119, 119, 96, 0, 128, 255, 255, 37, 0,
            ],
            &[
                0, 116, 255, 255, 67, 71, 255, 255, 234, 0, 147, 255, 255, 19, 0,
            ],
            &[
                0, 97, 255, 255, 85, 110, 255, 255, 255, 16, 165, 255, 254, 2, 0,
            ],
            &[
                0, 78, 255, 255, 101, 150, 255, 241, 255, 55, 183, 255, 239, 0, 0,
            ],
            &[
                0, 59, 255, 255, 111, 189, 255, 172, 255, 93, 197, 255, 221, 0, 0,
            ],
            &[
                0, 40, 255, 255, 117, 228, 254, 108, 255, 132, 204, 255, 203, 0, 0,
            ],
            &[
                0, 21, 255, 255, 129, 254, 224, 64, 255, 174, 204, 255, 185, 0, 0,
            ],
            &[
                0, 3, 254, 255, 163, 255, 185, 26, 255, 216, 197, 255, 167, 0, 0,
            ],
            &[
                0, 0, 239, 255, 194, 255, 147, 0, 243, 251, 193, 255, 149, 0, 0,
            ],
            &[
                0, 0, 220, 255, 236, 255, 109, 0, 206, 255, 234, 255, 131, 0, 0,
            ],
            &[
                0, 0, 201, 255, 255, 255, 72, 0, 169, 255, 255, 255, 113, 0, 0,
            ],
            &[
                0, 0, 182, 255, 255, 255, 34, 0, 131, 255, 255, 255, 95, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŵ' / 0x175
        'ŵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 4, 22, 22, 20, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 145, 255, 255, 255, 72, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 72, 254, 252, 196, 255, 231, 23, 0, 0, 0, 0],
            &[0, 0, 0, 37, 237, 247, 86, 3, 154, 255, 193, 5, 0, 0, 0],
            &[0, 0, 0, 35, 70, 37, 0, 0, 0, 58, 70, 12, 0, 0, 0],
            &[0, 107, 119, 102, 0, 0, 0, 0, 0, 0, 22, 119, 119, 62, 0],
            &[0, 204, 255, 235, 0, 0, 0, 0, 0, 0, 65, 255, 255, 107, 0],
            &[
                0, 168, 255, 254, 4, 12, 200, 201, 141, 0, 88, 255, 255, 71, 0,
            ],
            &[
                0, 132, 255, 255, 25, 60, 255, 255, 224, 0, 111, 255, 255, 35, 0,
            ],
            &[
                0, 96, 255, 255, 47, 112, 255, 255, 255, 20, 133, 255, 251, 4, 0,
            ],
            &[
                0, 60, 255, 255, 70, 163, 255, 231, 255, 70, 156, 255, 219, 0, 0,
            ],
            &[
                0, 24, 255, 255, 93, 214, 255, 150, 255, 121, 178, 255, 183, 0, 0,
            ],
            &[
                0, 0, 243, 255, 125, 252, 237, 84, 255, 170, 198, 255, 147, 0, 0,
            ],
            &[
                0, 0, 208, 255, 184, 255, 191, 38, 255, 212, 221, 255, 111, 0, 0,
            ],
            &[
                0, 0, 172, 255, 243, 255, 144, 2, 246, 249, 248, 255, 75, 0, 0,
            ],
            &[
                0, 0, 136, 255, 255, 255, 96, 0, 202, 255, 255, 255, 39, 0, 0,
            ],
            &[0, 0, 99, 255, 255, 255, 49, 0, 157, 255, 255, 252, 6, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŷ' / 0x176
        'Ŷ' => &[
            &[0, 0, 0, 0, 0, 67, 157, 157, 152, 12, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 228, 255, 255, 255, 154, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 184, 255, 187, 68, 241, 255, 92, 0, 0, 0, 0],
            &[0, 0, 0, 79, 190, 144, 6, 0, 42, 182, 181, 13, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 87, 254, 254, 213, 0, 0, 0, 0, 56, 254, 254, 231, 8, 0],
            &[0, 3, 219, 255, 255, 67, 0, 0, 0, 163, 255, 255, 120, 0, 0],
            &[0, 0, 100, 255, 255, 175, 0, 0, 21, 248, 255, 240, 14, 0, 0],
            &[0, 0, 6, 228, 255, 252, 31, 0, 121, 255, 255, 134, 0, 0, 0],
            &[0, 0, 0, 112, 255, 255, 135, 3, 225, 255, 246, 22, 0, 0, 0],
            &[0, 0, 0, 11, 235, 255, 235, 88, 255, 255, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 255, 255, 241, 255, 251, 31, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 242, 255, 255, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 137, 255, 255, 254, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 255, 255, 215, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŷ' / 0x177
        'ŷ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 22, 22, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 156, 255, 255, 254, 62, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 255, 250, 198, 255, 225, 18, 0, 0, 0, 0],
            &[0, 0, 0, 44, 241, 244, 78, 4, 163, 255, 185, 3, 0, 0, 0],
            &[0, 0, 0, 38, 70, 34, 0, 0, 0, 61, 70, 10, 0, 0, 0],
            &[0, 58, 119, 119, 97, 0, 0, 0, 0, 19, 119, 119, 119, 11, 0],
            &[0, 50, 255, 255, 251, 19, 0, 0, 0, 97, 255, 255, 212, 0, 0],
            &[0, 0, 204, 255, 255, 101, 0, 0, 0, 173, 255, 255, 119, 0, 0],
            &[0, 0, 103, 255, 255, 186, 0, 0, 6, 242, 255, 253, 28, 0, 0],
            &[0, 0, 13, 244, 255, 251, 20, 0, 69, 255, 255, 188, 0, 0, 0],
            &[0, 0, 0, 156, 255, 255, 102, 0, 145, 255, 255, 95, 0, 0, 0],
            &[0, 0, 0, 54, 255, 255, 187, 0, 220, 255, 245, 13, 0, 0, 0],
            &[0, 0, 0, 0, 208, 255, 251, 56, 255, 255, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 107, 255, 255, 178, 255, 255, 72, 0, 0, 0, 0],
            &[0, 0, 0, 0, 15, 246, 255, 253, 255, 231, 3, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 160, 255, 255, 255, 142, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 59, 255, 255, 255, 49, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 71, 255, 255, 211, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 7, 198, 255, 255, 105, 0, 0, 0, 0, 0, 0],
            &[0, 0, 110, 141, 210, 255, 255, 224, 9, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 32, 17, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ÿ' / 0x178
        'Ÿ' => &[
            &[0, 0, 0, 6, 77, 48, 0, 0, 10, 82, 40, 0, 0, 0, 0],
            &[0, 0, 0, 136, 255, 255, 53, 0, 162, 255, 252, 34, 0, 0, 0],
            &[0, 0, 0, 152, 255, 255, 70, 0, 178, 255, 255, 48, 0, 0, 0],
            &[0, 0, 0, 28, 151, 120, 1, 0, 38, 155, 108, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 87, 254, 254, 213, 0, 0, 0, 0, 56, 254, 254, 231, 8, 0],
            &[0, 3, 219, 255, 255, 67, 0, 0, 0, 163, 255, 255, 120, 0, 0],
            &[0, 0, 100, 255, 255, 175, 0, 0, 21, 248, 255, 240, 14, 0, 0],
            &[0, 0, 6, 228, 255, 252, 31, 0, 121, 255, 255, 134, 0, 0, 0],
            &[0, 0, 0, 112, 255, 255, 135, 3, 225, 255, 246, 22, 0, 0, 0],
            &[0, 0, 0, 11, 235, 255, 235, 88, 255, 255, 147, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 255, 255, 241, 255, 251, 31, 0, 0, 0, 0],
            &[0, 0, 0, 0, 16, 242, 255, 255, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 137, 255, 255, 254, 41, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 56, 255, 255, 215, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 213, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ź' / 0x179
        'Ź' => &[
            &[0, 0, 0, 0, 0, 0, 0, 7, 150, 157, 157, 56, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 131, 255, 255, 153, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 54, 251, 255, 139, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 150, 190, 105, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 254, 22, 0, 0,
            ],
            &[
                0, 0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 255, 22, 0, 0,
            ],
            &[
                0, 0, 0, 119, 150, 150, 150, 150, 196, 255, 255, 209, 3, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 3, 211, 255, 255, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 108, 255, 255, 167, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 237, 255, 244, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 149, 255, 255, 122, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 252, 255, 220, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 189, 255, 255, 76, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 255, 255, 181, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 222, 255, 249, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 123, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 244, 255, 255, 172, 159, 159, 159, 159, 159, 20, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ź' / 0x17a
        'ź' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 20, 22, 22, 9, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 66, 255, 255, 229, 34, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 12, 221, 255, 225, 38, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 158, 255, 210, 28, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 68, 70, 13, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 38, 119, 119, 119, 119, 119, 119, 119, 119, 119, 5, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 11, 0, 0,
            ],
            &[
                0, 0, 75, 237, 237, 237, 237, 237, 252, 255, 255, 241, 7, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 69, 252, 255, 253, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 35, 237, 255, 255, 116, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 211, 255, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 173, 255, 255, 199, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 255, 255, 227, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 254, 255, 246, 51, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 41, 242, 255, 255, 178, 100, 100, 100, 100, 100, 23, 0, 0,
            ],
            &[
                0, 0, 160, 255, 255, 255, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 162, 255, 255, 255, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ż' / 0x17b
        'Ż' => &[
            &[0, 0, 0, 0, 0, 1, 141, 202, 154, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 61, 255, 255, 255, 84, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 54, 255, 255, 255, 73, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 112, 171, 118, 1, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 254, 22, 0, 0,
            ],
            &[
                0, 0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 255, 22, 0, 0,
            ],
            &[
                0, 0, 0, 119, 150, 150, 150, 150, 196, 255, 255, 209, 3, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 3, 211, 255, 255, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 108, 255, 255, 167, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 237, 255, 244, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 149, 255, 255, 122, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 252, 255, 220, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 189, 255, 255, 76, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 255, 255, 181, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 222, 255, 249, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 123, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 244, 255, 255, 172, 159, 159, 159, 159, 159, 20, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ż' / 0x17c
        'ż' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 26, 66, 22, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 247, 255, 241, 25, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 92, 255, 255, 255, 70, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 35, 242, 255, 230, 19, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 17, 50, 11, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 38, 119, 119, 119, 119, 119, 119, 119, 119, 119, 5, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 11, 0, 0,
            ],
            &[
                0, 0, 75, 237, 237, 237, 237, 237, 252, 255, 255, 241, 7, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 69, 252, 255, 253, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 35, 237, 255, 255, 116, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 211, 255, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 173, 255, 255, 199, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 255, 255, 227, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 254, 255, 246, 51, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 41, 242, 255, 255, 178, 100, 100, 100, 100, 100, 23, 0, 0,
            ],
            &[
                0, 0, 160, 255, 255, 255, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 162, 255, 255, 255, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ž' / 0x17d
        'Ž' => &[
            &[0, 0, 0, 33, 157, 138, 12, 0, 7, 129, 157, 49, 0, 0, 0],
            &[0, 0, 0, 0, 148, 255, 206, 33, 195, 255, 177, 1, 0, 0, 0],
            &[0, 0, 0, 0, 6, 202, 255, 251, 255, 223, 16, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 42, 190, 190, 190, 63, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 0, 202, 254, 254, 254, 254, 254, 254, 254, 254, 22, 0, 0,
            ],
            &[
                0, 0, 0, 203, 255, 255, 255, 255, 255, 255, 255, 255, 22, 0, 0,
            ],
            &[
                0, 0, 0, 119, 150, 150, 150, 150, 196, 255, 255, 209, 3, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 3, 211, 255, 255, 62, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 108, 255, 255, 167, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 18, 237, 255, 244, 28, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 149, 255, 255, 122, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 45, 252, 255, 220, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 189, 255, 255, 76, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 82, 255, 255, 181, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 222, 255, 249, 37, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 123, 255, 255, 136, 0, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 27, 244, 255, 255, 172, 159, 159, 159, 159, 159, 20, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 33, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ž' / 0x17e
        'ž' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 22, 12, 0, 0, 0, 13, 22, 6, 0, 0, 0],
            &[0, 0, 0, 39, 240, 239, 70, 0, 87, 245, 235, 29, 0, 0, 0],
            &[0, 0, 0, 0, 76, 254, 249, 150, 253, 252, 66, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 146, 255, 255, 255, 132, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 10, 70, 70, 70, 6, 0, 0, 0, 0, 0],
            &[
                0, 0, 38, 119, 119, 119, 119, 119, 119, 119, 119, 119, 5, 0, 0,
            ],
            &[
                0, 0, 81, 255, 255, 255, 255, 255, 255, 255, 255, 255, 11, 0, 0,
            ],
            &[
                0, 0, 75, 237, 237, 237, 237, 237, 252, 255, 255, 241, 7, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 69, 252, 255, 253, 74, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 35, 237, 255, 255, 116, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 12, 211, 255, 255, 161, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 173, 255, 255, 199, 7, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 125, 255, 255, 227, 24, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 78, 254, 255, 246, 51, 0, 0, 0, 0, 0, 0, 0],
            &[
                0, 0, 41, 242, 255, 255, 178, 100, 100, 100, 100, 100, 23, 0, 0,
            ],
            &[
                0, 0, 160, 255, 255, 255, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[
                0, 0, 162, 255, 255, 255, 255, 255, 255, 255, 255, 255, 60, 0, 0,
            ],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ſ' / 0x17f
        'ſ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 4, 8, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 98, 214, 254, 255, 241, 165, 0, 0, 0, 0],
            &[0, 0, 0, 0, 80, 255, 255, 255, 255, 255, 161, 0, 0, 0, 0],
            &[0, 0, 0, 0, 174, 255, 255, 213, 125, 146, 67, 0, 0, 0, 0],
            &[0, 0, 0, 0, 203, 255, 255, 79, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 210, 255, 255, 46, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        _ => panic!("unsupported char"),
    }
}
