//! Module for letters with the font weight bold and size 14.
//!
//! The bitmap font contains all unicode symbols from 0x00 to 0x17f with
//! the exception of control characters. It includes Basic Latin
//! Latin-1 Supplement and Latin extended A. This means the typical letters
//! and symbols from a QWERTZ/QWERTY keyboard plus symbols such as Ö, Ä,
//! and Ü.
//!
//! # Font License
//! * Downloaded from: <https://fonts.google.com/noto/specimen/Noto+Sans+Mono>
//! * License: SIL Open Font License (OFL) <https://scripts.sil.org/cms/scripts/page.php?site_id=nrsi&id=OFL>

/// The height of each bitmap character for the given font weight
/// and size. This size corresponds to the size of the module name.
/// The font size will be a few percent smaller, because each
/// bitmap contains a small vertical padding.
#[allow(dead_code)]
pub const BITMAP_HEIGHT: usize = 14;

/// The width of each bitmap character for the given font weight
/// and size. This is a few percent less than [`BITMAP_HEIGHT`],
/// because the bitmap font doesn't contain horizontal padding.
pub const BITMAP_WIDTH: usize = 9;

/// Returns the bitmap of the given character of the pre rendered
/// "Noto Sans Mono" raster for font weight bold and font size 12px
#[inline]
pub const fn get_char(c: char) -> &'static [&'static [u8]] {
    match c {
        // letter: ' ' / 0x20
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '!' / 0x21
        '!' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 57, 145, 88, 0, 0, 0],
            &[0, 0, 0, 92, 255, 146, 0, 0, 0],
            &[0, 0, 0, 81, 255, 135, 0, 0, 0],
            &[0, 0, 0, 69, 255, 124, 0, 0, 0],
            &[0, 0, 0, 58, 255, 112, 0, 0, 0],
            &[0, 0, 0, 47, 255, 101, 0, 0, 0],
            &[0, 0, 0, 6, 39, 14, 0, 0, 0],
            &[0, 0, 0, 74, 231, 103, 0, 0, 0],
            &[0, 0, 0, 88, 253, 127, 0, 0, 0],
            &[0, 0, 0, 0, 14, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '"' / 0x22
        '"' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 60, 145, 31, 145, 89, 0, 0],
            &[0, 0, 92, 255, 38, 242, 143, 0, 0],
            &[0, 0, 73, 255, 19, 224, 124, 0, 0],
            &[0, 0, 31, 135, 2, 110, 58, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '#' / 0x23
        '#' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 69, 106, 34, 140, 0, 0, 0],
            &[0, 0, 156, 153, 92, 214, 0, 0, 0],
            &[24, 52, 207, 144, 155, 193, 52, 1, 0],
            &[118, 251, 255, 252, 254, 253, 251, 5, 0],
            &[0, 24, 255, 30, 217, 89, 0, 0, 0],
            &[120, 170, 254, 148, 252, 172, 99, 0, 0],
            &[125, 201, 228, 178, 253, 153, 103, 0, 0],
            &[0, 148, 158, 91, 221, 0, 0, 0, 0],
            &[0, 187, 119, 133, 179, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '$' / 0x24
        '$' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 21, 0, 0, 0, 0],
            &[0, 0, 0, 19, 209, 61, 11, 0, 0],
            &[0, 0, 115, 252, 255, 255, 239, 8, 0],
            &[0, 1, 245, 168, 201, 42, 79, 0, 0],
            &[0, 0, 225, 210, 217, 5, 0, 0, 0],
            &[0, 0, 57, 221, 255, 197, 53, 0, 0],
            &[0, 0, 0, 4, 205, 215, 244, 22, 0],
            &[0, 2, 42, 0, 194, 92, 255, 49, 0],
            &[0, 7, 255, 216, 239, 243, 209, 5, 0],
            &[0, 0, 69, 126, 229, 110, 13, 0, 0],
            &[0, 0, 0, 0, 135, 3, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '%' / 0x25
        '%' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[46, 155, 110, 0, 10, 143, 33, 0, 0],
            &[211, 157, 239, 67, 117, 210, 1, 0, 0],
            &[246, 56, 202, 114, 235, 81, 0, 0, 0],
            &[185, 206, 246, 168, 205, 0, 0, 0, 0],
            &[12, 89, 63, 238, 76, 0, 0, 0, 0],
            &[0, 0, 128, 201, 149, 233, 143, 0, 0],
            &[0, 18, 241, 107, 254, 67, 255, 29, 0],
            &[0, 134, 196, 45, 252, 30, 255, 37, 0],
            &[21, 243, 66, 1, 187, 252, 182, 0, 0],
            &[0, 0, 0, 0, 0, 15, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '&' / 0x26
        '&' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 31, 149, 151, 37, 0, 0, 0, 0],
            &[0, 195, 215, 203, 208, 0, 0, 0, 0],
            &[0, 232, 139, 130, 231, 0, 0, 0, 0],
            &[0, 174, 255, 253, 130, 0, 0, 0, 0],
            &[0, 144, 255, 255, 4, 122, 83, 0, 0],
            &[60, 253, 252, 255, 125, 255, 116, 0, 0],
            &[145, 245, 3, 173, 255, 255, 27, 0, 0],
            &[139, 255, 115, 155, 255, 255, 11, 0, 0],
            &[30, 218, 255, 245, 197, 255, 163, 0, 0],
            &[0, 1, 23, 6, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ''' / 0x27
        '\'' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 31, 145, 59, 0, 0, 0],
            &[0, 0, 0, 41, 255, 89, 0, 0, 0],
            &[0, 0, 0, 22, 255, 70, 0, 0, 0],
            &[0, 0, 0, 4, 135, 29, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '(' / 0x28
        '(' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 30, 145, 33, 0, 0, 0],
            &[0, 0, 0, 176, 199, 0, 0, 0, 0],
            &[0, 0, 36, 254, 91, 0, 0, 0, 0],
            &[0, 0, 111, 254, 18, 0, 0, 0, 0],
            &[0, 0, 159, 226, 0, 0, 0, 0, 0],
            &[0, 0, 178, 208, 0, 0, 0, 0, 0],
            &[0, 0, 170, 217, 0, 0, 0, 0, 0],
            &[0, 0, 138, 245, 4, 0, 0, 0, 0],
            &[0, 0, 76, 255, 58, 0, 0, 0, 0],
            &[0, 0, 4, 227, 154, 0, 0, 0, 0],
            &[0, 0, 0, 87, 244, 33, 0, 0, 0],
        ],
        // letter: ')' / 0x29
        ')' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 7, 141, 60, 0, 0, 0, 0],
            &[0, 0, 0, 151, 224, 6, 0, 0, 0],
            &[0, 0, 0, 43, 255, 90, 0, 0, 0],
            &[0, 0, 0, 0, 223, 166, 0, 0, 0],
            &[0, 0, 0, 0, 175, 212, 0, 0, 0],
            &[0, 0, 0, 0, 156, 230, 0, 0, 0],
            &[0, 0, 0, 0, 166, 219, 0, 0, 0],
            &[0, 0, 0, 0, 199, 186, 0, 0, 0],
            &[0, 0, 0, 12, 248, 123, 0, 0, 0],
            &[0, 0, 0, 99, 251, 28, 0, 0, 0],
            &[0, 0, 8, 219, 135, 0, 0, 0, 0],
        ],
        // letter: '*' / 0x2a
        '*' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 15, 4, 0, 0, 0],
            &[0, 0, 0, 13, 255, 66, 0, 0, 0],
            &[0, 0, 109, 61, 250, 88, 117, 15, 0],
            &[0, 5, 191, 232, 255, 241, 203, 42, 0],
            &[0, 0, 2, 185, 210, 223, 18, 0, 0],
            &[0, 0, 50, 225, 22, 206, 92, 0, 0],
            &[0, 0, 0, 5, 0, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '+' / 0x2b
        '+' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 6, 0, 0, 0],
            &[0, 0, 0, 6, 255, 66, 0, 0, 0],
            &[0, 0, 0, 6, 255, 66, 0, 0, 0],
            &[0, 35, 223, 223, 255, 231, 223, 80, 0],
            &[0, 16, 102, 106, 255, 142, 102, 36, 0],
            &[0, 0, 0, 6, 255, 66, 0, 0, 0],
            &[0, 0, 0, 3, 152, 39, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ',' / 0x2c
        ',' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 161, 104, 0, 0, 0],
            &[0, 0, 0, 53, 255, 110, 0, 0, 0],
            &[0, 0, 0, 98, 255, 32, 0, 0, 0],
            &[0, 0, 0, 128, 175, 0, 0, 0, 0],
        ],
        // letter: '-' / 0x2d
        '-' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 49, 245, 245, 245, 98, 0, 0],
            &[0, 0, 27, 138, 138, 138, 55, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '.' / 0x2e
        '.' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 74, 231, 103, 0, 0, 0],
            &[0, 0, 0, 88, 253, 127, 0, 0, 0],
            &[0, 0, 0, 0, 14, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '/' / 0x2f
        '/' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 97, 139, 0, 0],
            &[0, 0, 0, 0, 4, 232, 147, 0, 0],
            &[0, 0, 0, 0, 78, 255, 50, 0, 0],
            &[0, 0, 0, 0, 175, 209, 0, 0, 0],
            &[0, 0, 0, 22, 250, 112, 0, 0, 0],
            &[0, 0, 0, 114, 250, 20, 0, 0, 0],
            &[0, 0, 0, 212, 173, 0, 0, 0, 0],
            &[0, 0, 53, 255, 76, 0, 0, 0, 0],
            &[0, 0, 151, 231, 4, 0, 0, 0, 0],
            &[0, 0, 9, 8, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '0' / 0x30
        '0' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 10, 120, 167, 132, 20, 0, 0],
            &[0, 0, 173, 255, 210, 255, 209, 3, 0],
            &[0, 32, 255, 158, 3, 255, 255, 76, 0],
            &[0, 81, 255, 95, 109, 255, 255, 129, 0],
            &[0, 98, 255, 101, 240, 135, 255, 148, 0],
            &[0, 89, 255, 244, 197, 40, 255, 144, 0],
            &[0, 53, 255, 255, 46, 72, 255, 111, 0],
            &[0, 3, 242, 255, 92, 198, 252, 35, 0],
            &[0, 0, 55, 224, 255, 243, 104, 0, 0],
            &[0, 0, 0, 1, 22, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '1' / 0x31
        '1' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 36, 137, 109, 0, 0, 0],
            &[0, 9, 150, 250, 255, 192, 0, 0, 0],
            &[0, 0, 198, 173, 248, 192, 0, 0, 0],
            &[0, 0, 9, 0, 250, 192, 0, 0, 0],
            &[0, 0, 0, 0, 252, 192, 0, 0, 0],
            &[0, 0, 0, 0, 252, 192, 0, 0, 0],
            &[0, 0, 0, 0, 252, 192, 0, 0, 0],
            &[0, 0, 40, 85, 255, 255, 78, 21, 0],
            &[0, 0, 164, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '2' / 0x32
        '2' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 30, 131, 168, 146, 42, 0, 0],
            &[0, 28, 242, 254, 229, 255, 242, 30, 0],
            &[0, 0, 105, 39, 0, 130, 255, 101, 0],
            &[0, 0, 0, 0, 0, 141, 255, 75, 0],
            &[0, 0, 0, 0, 49, 246, 206, 4, 0],
            &[0, 0, 0, 42, 234, 220, 29, 0, 0],
            &[0, 0, 42, 232, 209, 25, 0, 0, 0],
            &[0, 33, 232, 252, 139, 124, 124, 82, 0],
            &[0, 80, 255, 255, 255, 255, 255, 168, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '3' / 0x33
        '3' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 53, 141, 168, 135, 35, 0, 0],
            &[0, 8, 229, 246, 214, 253, 232, 8, 0],
            &[0, 0, 44, 14, 0, 152, 255, 29, 0],
            &[0, 0, 0, 24, 54, 209, 202, 0, 0],
            &[0, 0, 0, 230, 255, 225, 50, 0, 0],
            &[0, 0, 0, 68, 95, 209, 244, 38, 0],
            &[0, 0, 0, 0, 0, 105, 255, 96, 0],
            &[0, 32, 169, 108, 101, 217, 255, 54, 0],
            &[0, 25, 214, 255, 255, 238, 112, 0, 0],
            &[0, 0, 0, 11, 24, 2, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '4' / 0x34
        '4' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 26, 145, 89, 0, 0],
            &[0, 0, 0, 0, 174, 255, 157, 0, 0],
            &[0, 0, 0, 81, 241, 252, 157, 0, 0],
            &[0, 0, 14, 229, 120, 255, 157, 0, 0],
            &[0, 0, 150, 215, 15, 255, 157, 0, 0],
            &[0, 59, 254, 113, 54, 255, 175, 33, 0],
            &[0, 133, 255, 255, 255, 255, 255, 184, 0],
            &[0, 29, 57, 57, 65, 255, 179, 41, 0],
            &[0, 0, 0, 0, 10, 255, 157, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '5' / 0x35
        '5' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 91, 145, 145, 145, 120, 0, 0],
            &[0, 0, 176, 254, 235, 235, 195, 0, 0],
            &[0, 0, 196, 232, 0, 0, 0, 0, 0],
            &[0, 0, 215, 230, 109, 82, 5, 0, 0],
            &[0, 0, 225, 253, 254, 255, 188, 1, 0],
            &[0, 0, 14, 5, 17, 207, 255, 51, 0],
            &[0, 0, 0, 0, 0, 147, 255, 66, 0],
            &[0, 20, 178, 117, 119, 238, 241, 15, 0],
            &[0, 15, 218, 255, 255, 215, 63, 0, 0],
            &[0, 0, 0, 15, 19, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '6' / 0x36
        '6' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 27, 121, 161, 152, 0, 0],
            &[0, 0, 60, 243, 249, 207, 193, 0, 0],
            &[0, 1, 218, 236, 38, 0, 0, 0, 0],
            &[0, 44, 255, 142, 107, 130, 48, 0, 0],
            &[0, 78, 255, 233, 230, 249, 248, 36, 0],
            &[0, 80, 255, 180, 0, 93, 255, 116, 0],
            &[0, 48, 255, 131, 0, 66, 255, 128, 0],
            &[0, 1, 215, 233, 115, 197, 255, 69, 0],
            &[0, 0, 47, 218, 255, 249, 136, 0, 0],
            &[0, 0, 0, 1, 26, 10, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '7' / 0x37
        '7' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 56, 142, 142, 142, 142, 142, 85, 0],
            &[0, 94, 235, 235, 235, 242, 255, 143, 0],
            &[0, 0, 0, 0, 0, 149, 255, 51, 0],
            &[0, 0, 0, 0, 19, 245, 195, 0, 0],
            &[0, 0, 0, 0, 125, 255, 83, 0, 0],
            &[0, 0, 0, 8, 233, 224, 3, 0, 0],
            &[0, 0, 0, 102, 255, 115, 0, 0, 0],
            &[0, 0, 1, 216, 243, 16, 0, 0, 0],
            &[0, 0, 79, 255, 148, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '8' / 0x38
        '8' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 125, 167, 141, 36, 0, 0],
            &[0, 0, 198, 254, 210, 249, 236, 14, 0],
            &[0, 1, 251, 195, 0, 143, 255, 49, 0],
            &[0, 0, 179, 248, 116, 228, 216, 5, 0],
            &[0, 0, 30, 244, 255, 248, 47, 0, 0],
            &[0, 6, 210, 235, 111, 228, 230, 23, 0],
            &[0, 59, 255, 118, 0, 75, 255, 108, 0],
            &[0, 42, 255, 198, 81, 171, 255, 87, 0],
            &[0, 0, 122, 243, 255, 247, 146, 3, 0],
            &[0, 0, 0, 4, 25, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '9' / 0x39
        '9' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 134, 172, 130, 15, 0, 0],
            &[0, 3, 211, 255, 223, 254, 201, 3, 0],
            &[0, 60, 255, 151, 0, 120, 255, 71, 0],
            &[0, 77, 255, 115, 0, 84, 255, 123, 0],
            &[0, 32, 252, 219, 114, 218, 255, 134, 0],
            &[0, 0, 110, 233, 233, 150, 255, 114, 0],
            &[0, 0, 0, 0, 0, 138, 255, 56, 0],
            &[0, 0, 69, 86, 146, 252, 191, 0, 0],
            &[0, 0, 195, 255, 246, 160, 16, 0, 0],
            &[0, 0, 16, 26, 4, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ':' / 0x3a
        ':' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 37, 149, 54, 0, 0, 0],
            &[0, 0, 0, 113, 255, 148, 0, 0, 0],
            &[0, 0, 0, 12, 93, 28, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 74, 231, 103, 0, 0, 0],
            &[0, 0, 0, 88, 253, 127, 0, 0, 0],
            &[0, 0, 0, 0, 14, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ';' / 0x3b
        ';' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 8, 139, 93, 0, 0, 0],
            &[0, 0, 0, 45, 255, 216, 0, 0, 0],
            &[0, 0, 0, 0, 78, 55, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 12, 161, 104, 0, 0, 0],
            &[0, 0, 0, 53, 255, 110, 0, 0, 0],
            &[0, 0, 0, 98, 255, 32, 0, 0, 0],
            &[0, 0, 0, 128, 175, 0, 0, 0, 0],
        ],
        // letter: '<' / 0x3c
        '<' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 14, 13, 0, 0],
            &[0, 0, 0, 10, 118, 236, 46, 0, 0],
            &[0, 7, 110, 232, 226, 107, 5, 0, 0],
            &[0, 223, 252, 114, 6, 0, 0, 0, 0],
            &[0, 102, 223, 236, 128, 20, 0, 0, 0],
            &[0, 0, 4, 96, 217, 245, 35, 0, 0],
            &[0, 0, 0, 0, 2, 89, 29, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '=' / 0x3d
        '=' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 3, 24, 24, 24, 24, 24, 8, 0],
            &[0, 40, 255, 255, 255, 255, 255, 92, 0],
            &[0, 14, 88, 88, 88, 88, 88, 31, 0],
            &[0, 23, 147, 147, 147, 147, 147, 53, 0],
            &[0, 35, 221, 221, 221, 221, 221, 79, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '>' / 0x3e
        '>' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 27, 0, 0, 0, 0, 0, 0, 0],
            &[0, 243, 145, 23, 0, 0, 0, 0, 0],
            &[0, 81, 207, 245, 136, 19, 0, 0, 0],
            &[0, 0, 0, 81, 240, 241, 32, 0, 0],
            &[0, 8, 105, 220, 238, 128, 9, 0, 0],
            &[0, 229, 234, 121, 14, 0, 0, 0, 0],
            &[0, 110, 10, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '?' / 0x3f
        '?' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 45, 133, 167, 147, 40, 0, 0],
            &[0, 0, 207, 240, 205, 249, 237, 15, 0],
            &[0, 0, 32, 8, 0, 138, 255, 51, 0],
            &[0, 0, 0, 0, 33, 222, 231, 12, 0],
            &[0, 0, 0, 33, 234, 219, 41, 0, 0],
            &[0, 0, 0, 121, 255, 34, 0, 0, 0],
            &[0, 0, 0, 20, 39, 1, 0, 0, 0],
            &[0, 0, 0, 144, 227, 37, 0, 0, 0],
            &[0, 0, 0, 168, 252, 47, 0, 0, 0],
            &[0, 0, 0, 0, 15, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '@' / 0x40
        '@' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 65, 133, 122, 32, 0, 0, 0],
            &[0, 145, 224, 125, 147, 239, 55, 0, 0],
            &[75, 232, 23, 46, 35, 116, 201, 0, 0],
            &[177, 127, 146, 240, 250, 105, 255, 21, 0],
            &[232, 80, 251, 40, 224, 72, 252, 45, 0],
            &[251, 82, 255, 7, 246, 68, 255, 28, 0],
            &[235, 74, 242, 159, 255, 181, 213, 0, 0],
            &[178, 135, 82, 158, 76, 159, 47, 0, 0],
            &[58, 242, 79, 0, 6, 62, 0, 0, 0],
            &[0, 74, 211, 246, 249, 186, 0, 0, 0],
            &[0, 0, 0, 7, 2, 0, 0, 0, 0],
        ],
        // letter: 'A' / 0x41
        'A' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 92, 154, 121, 0, 0, 0, 0],
            &[0, 0, 209, 251, 248, 10, 0, 0, 0],
            &[0, 24, 255, 164, 255, 72, 0, 0, 0],
            &[0, 94, 255, 52, 246, 143, 0, 0, 0],
            &[0, 164, 237, 1, 190, 214, 0, 0, 0],
            &[1, 233, 255, 144, 255, 255, 29, 0, 0],
            &[49, 255, 255, 224, 255, 255, 99, 0, 0],
            &[119, 255, 47, 0, 10, 250, 170, 0, 0],
            &[190, 237, 1, 0, 0, 199, 238, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'B' / 0x42
        'B' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 145, 145, 143, 122, 47, 0, 0],
            &[0, 10, 255, 240, 222, 251, 252, 61, 0],
            &[0, 10, 255, 148, 0, 93, 255, 127, 0],
            &[0, 10, 255, 169, 52, 150, 255, 81, 0],
            &[0, 10, 255, 255, 255, 255, 178, 2, 0],
            &[0, 10, 255, 178, 73, 146, 255, 117, 0],
            &[0, 10, 255, 148, 0, 40, 255, 174, 0],
            &[0, 10, 255, 194, 110, 179, 255, 131, 0],
            &[0, 10, 255, 255, 255, 240, 159, 10, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'C' / 0x43
        'C' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 41, 137, 169, 149, 59, 0],
            &[0, 0, 77, 250, 254, 225, 245, 112, 0],
            &[0, 6, 231, 233, 40, 0, 10, 10, 0],
            &[0, 63, 255, 138, 0, 0, 0, 0, 0],
            &[0, 95, 255, 101, 0, 0, 0, 0, 0],
            &[0, 89, 255, 106, 0, 0, 0, 0, 0],
            &[0, 47, 255, 172, 0, 0, 0, 0, 0],
            &[0, 0, 200, 255, 162, 110, 148, 80, 0],
            &[0, 0, 26, 186, 255, 255, 238, 72, 0],
            &[0, 0, 0, 0, 10, 22, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'D' / 0x44
        'D' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 145, 145, 135, 85, 5, 0, 0],
            &[0, 117, 255, 238, 244, 255, 193, 6, 0],
            &[0, 117, 255, 65, 3, 139, 255, 110, 0],
            &[0, 117, 255, 65, 0, 15, 254, 185, 0],
            &[0, 117, 255, 65, 0, 0, 241, 208, 0],
            &[0, 117, 255, 65, 0, 4, 253, 195, 0],
            &[0, 117, 255, 65, 0, 79, 255, 139, 0],
            &[0, 117, 255, 156, 148, 243, 241, 29, 0],
            &[0, 117, 255, 255, 242, 185, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'E' / 0x45
        'E' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 145, 145, 145, 145, 145, 52, 0],
            &[0, 10, 255, 249, 238, 238, 238, 85, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 181, 50, 50, 50, 9, 0],
            &[0, 10, 255, 255, 255, 255, 255, 46, 0],
            &[0, 10, 255, 191, 77, 77, 77, 14, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 209, 127, 127, 127, 46, 0],
            &[0, 10, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'F' / 0x46
        'F' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 145, 145, 145, 145, 145, 52, 0],
            &[0, 10, 255, 249, 238, 238, 238, 85, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 240, 213, 213, 213, 38, 0],
            &[0, 10, 255, 224, 169, 169, 169, 30, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'G' / 0x47
        'G' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 77, 153, 166, 131, 38, 0],
            &[0, 0, 144, 255, 243, 215, 246, 84, 0],
            &[0, 51, 255, 196, 13, 0, 14, 5, 0],
            &[0, 127, 255, 69, 0, 0, 0, 0, 0],
            &[0, 156, 255, 31, 21, 210, 210, 139, 0],
            &[0, 149, 255, 41, 15, 157, 247, 168, 0],
            &[0, 103, 255, 112, 0, 0, 233, 168, 0],
            &[0, 17, 234, 245, 127, 100, 243, 168, 0],
            &[0, 0, 49, 203, 255, 255, 244, 123, 0],
            &[0, 0, 0, 0, 12, 22, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'H' / 0x48
        'H' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 145, 40, 0, 11, 145, 95, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 112, 56, 72, 255, 168, 0],
            &[0, 117, 255, 255, 255, 255, 255, 168, 0],
            &[0, 117, 255, 123, 71, 86, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'I' / 0x49
        'I' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 133, 145, 145, 145, 145, 17, 0, 0],
            &[0, 161, 213, 255, 224, 179, 19, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 59, 131, 255, 160, 69, 6, 0, 0],
            &[0, 235, 255, 255, 255, 255, 30, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'J' / 0x4a
        'J' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 119, 127, 0, 0],
            &[0, 0, 0, 0, 0, 210, 225, 0, 0],
            &[0, 0, 0, 0, 0, 210, 225, 0, 0],
            &[0, 0, 0, 0, 0, 210, 225, 0, 0],
            &[0, 0, 0, 0, 0, 210, 225, 0, 0],
            &[0, 0, 0, 0, 0, 210, 225, 0, 0],
            &[0, 0, 0, 0, 0, 230, 215, 0, 0],
            &[0, 0, 134, 107, 159, 255, 159, 0, 0],
            &[0, 0, 240, 255, 255, 194, 25, 0, 0],
            &[0, 0, 3, 24, 14, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'K' / 0x4b
        'K' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 145, 35, 0, 44, 145, 86, 0],
            &[0, 117, 255, 62, 4, 206, 238, 27, 0],
            &[0, 117, 255, 62, 126, 255, 85, 0, 0],
            &[0, 117, 255, 107, 250, 160, 0, 0, 0],
            &[0, 117, 255, 235, 255, 68, 0, 0, 0],
            &[0, 117, 255, 218, 243, 198, 0, 0, 0],
            &[0, 117, 255, 66, 130, 255, 85, 0, 0],
            &[0, 117, 255, 62, 14, 236, 221, 6, 0],
            &[0, 117, 255, 62, 0, 115, 255, 113, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'L' / 0x4c
        'L' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 133, 119, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 234, 137, 137, 137, 49, 0, 0],
            &[0, 235, 255, 255, 255, 255, 92, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'M' / 0x4d
        'M' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 75, 145, 127, 0, 83, 145, 102, 0],
            &[0, 133, 255, 250, 3, 182, 255, 181, 0],
            &[0, 133, 223, 255, 36, 228, 223, 181, 0],
            &[0, 133, 219, 231, 93, 255, 195, 181, 0],
            &[0, 133, 237, 182, 177, 227, 193, 181, 0],
            &[0, 133, 252, 134, 248, 182, 206, 181, 0],
            &[0, 133, 255, 90, 251, 135, 215, 181, 0],
            &[0, 133, 255, 10, 0, 0, 218, 181, 0],
            &[0, 133, 255, 10, 0, 0, 218, 181, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'N' / 0x4e
        'N' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 145, 141, 3, 0, 129, 95, 0],
            &[0, 117, 255, 255, 64, 0, 227, 168, 0],
            &[0, 117, 253, 214, 148, 0, 227, 168, 0],
            &[0, 117, 255, 143, 230, 1, 227, 168, 0],
            &[0, 117, 255, 69, 255, 59, 227, 168, 0],
            &[0, 117, 255, 22, 220, 143, 223, 168, 0],
            &[0, 117, 255, 22, 137, 226, 217, 168, 0],
            &[0, 117, 255, 22, 54, 255, 248, 168, 0],
            &[0, 117, 255, 22, 1, 226, 255, 168, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'O' / 0x4f
        'O' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 130, 174, 148, 38, 0, 0],
            &[0, 4, 208, 255, 216, 251, 238, 29, 0],
            &[0, 82, 255, 136, 0, 90, 255, 136, 0],
            &[0, 139, 255, 60, 0, 8, 254, 192, 0],
            &[0, 158, 255, 38, 0, 0, 241, 211, 0],
            &[0, 152, 255, 50, 0, 0, 247, 203, 0],
            &[0, 113, 255, 90, 0, 40, 255, 164, 0],
            &[0, 30, 249, 225, 109, 199, 255, 75, 0],
            &[0, 0, 85, 234, 255, 245, 125, 0, 0],
            &[0, 0, 0, 3, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'P' / 0x50
        'P' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 145, 145, 144, 121, 44, 0, 0],
            &[0, 10, 255, 247, 233, 253, 252, 63, 0],
            &[0, 10, 255, 163, 0, 52, 255, 160, 0],
            &[0, 10, 255, 163, 0, 24, 255, 173, 0],
            &[0, 10, 255, 212, 139, 209, 255, 105, 0],
            &[0, 10, 255, 251, 242, 221, 130, 3, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Q' / 0x51
        'Q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 18, 130, 174, 148, 38, 0, 0],
            &[0, 4, 208, 255, 216, 251, 238, 29, 0],
            &[0, 82, 255, 136, 0, 90, 255, 136, 0],
            &[0, 139, 255, 60, 0, 8, 254, 192, 0],
            &[0, 158, 255, 38, 0, 0, 241, 211, 0],
            &[0, 152, 255, 50, 0, 0, 247, 203, 0],
            &[0, 113, 255, 90, 0, 40, 255, 164, 0],
            &[0, 30, 249, 225, 109, 203, 255, 75, 0],
            &[0, 0, 85, 234, 255, 255, 171, 0, 0],
            &[0, 0, 0, 3, 50, 254, 182, 0, 0],
            &[0, 0, 0, 0, 0, 11, 40, 23, 0],
        ],
        // letter: 'R' / 0x52
        'R' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 145, 145, 139, 101, 11, 0, 0, 0],
            &[10, 255, 248, 241, 255, 213, 9, 0, 0],
            &[10, 255, 179, 0, 145, 255, 77, 0, 0],
            &[10, 255, 179, 0, 131, 255, 73, 0, 0],
            &[10, 255, 233, 194, 252, 217, 9, 0, 0],
            &[10, 255, 238, 229, 253, 35, 0, 0, 0],
            &[10, 255, 179, 55, 254, 146, 0, 0, 0],
            &[10, 255, 179, 0, 168, 251, 42, 0, 0],
            &[10, 255, 179, 0, 33, 249, 186, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'S' / 0x53
        'S' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 118, 168, 159, 115, 19, 0],
            &[0, 0, 177, 255, 215, 213, 252, 33, 0],
            &[0, 0, 245, 220, 1, 0, 20, 0, 0],
            &[0, 0, 203, 250, 95, 2, 0, 0, 0],
            &[0, 0, 36, 218, 255, 208, 60, 0, 0],
            &[0, 0, 0, 8, 118, 242, 249, 56, 0],
            &[0, 0, 0, 0, 0, 87, 255, 131, 0],
            &[0, 38, 185, 111, 90, 183, 255, 89, 0],
            &[0, 25, 210, 255, 255, 242, 131, 0, 0],
            &[0, 0, 0, 10, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'T' / 0x54
        'T' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 93, 145, 145, 145, 145, 145, 122, 0],
            &[0, 152, 238, 243, 255, 246, 238, 200, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'U' / 0x55
        'U' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 145, 40, 0, 11, 145, 95, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 72, 0, 21, 255, 168, 0],
            &[0, 107, 255, 97, 0, 45, 255, 158, 0],
            &[0, 47, 254, 221, 113, 194, 255, 93, 0],
            &[0, 0, 112, 241, 255, 248, 146, 1, 0],
            &[0, 0, 0, 4, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'V' / 0x56
        'V' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[115, 137, 0, 0, 0, 111, 142, 2, 0],
            &[145, 255, 37, 0, 3, 242, 196, 0, 0],
            &[70, 255, 103, 0, 53, 255, 121, 0, 0],
            &[7, 244, 169, 0, 116, 255, 46, 0, 0],
            &[0, 176, 233, 1, 180, 226, 0, 0, 0],
            &[0, 101, 255, 46, 240, 152, 0, 0, 0],
            &[0, 27, 255, 143, 255, 77, 0, 0, 0],
            &[0, 0, 207, 235, 247, 10, 0, 0, 0],
            &[0, 0, 132, 255, 183, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'W' / 0x57
        'W' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[124, 95, 0, 0, 0, 65, 145, 8, 0],
            &[204, 187, 0, 0, 0, 129, 254, 2, 0],
            &[185, 211, 36, 141, 68, 148, 239, 0, 0],
            &[166, 230, 96, 255, 153, 166, 221, 0, 0],
            &[147, 244, 136, 224, 191, 182, 203, 0, 0],
            &[128, 250, 175, 154, 230, 189, 185, 0, 0],
            &[109, 250, 210, 79, 255, 202, 167, 0, 0],
            &[90, 251, 245, 28, 230, 239, 149, 0, 0],
            &[71, 255, 245, 1, 192, 255, 131, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'X' / 0x58
        'X' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 62, 145, 48, 0, 22, 145, 91, 0],
            &[0, 15, 234, 190, 0, 142, 252, 47, 0],
            &[0, 0, 101, 255, 96, 247, 151, 0, 0],
            &[0, 0, 2, 206, 253, 239, 20, 0, 0],
            &[0, 0, 0, 87, 255, 138, 0, 0, 0],
            &[0, 0, 0, 185, 255, 227, 10, 0, 0],
            &[0, 0, 78, 255, 130, 253, 130, 0, 0],
            &[0, 7, 220, 205, 1, 166, 247, 31, 0],
            &[0, 120, 255, 71, 0, 38, 252, 171, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Y' / 0x59
        'Y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 145, 28, 0, 3, 139, 119, 0],
            &[0, 64, 255, 134, 0, 79, 255, 116, 0],
            &[0, 0, 198, 235, 7, 187, 238, 12, 0],
            &[0, 0, 76, 255, 134, 254, 129, 0, 0],
            &[0, 0, 1, 209, 255, 244, 19, 0, 0],
            &[0, 0, 0, 95, 255, 149, 0, 0, 0],
            &[0, 0, 0, 67, 255, 121, 0, 0, 0],
            &[0, 0, 0, 67, 255, 121, 0, 0, 0],
            &[0, 0, 0, 67, 255, 121, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Z' / 0x5a
        'Z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 145, 145, 145, 145, 48, 0],
            &[0, 0, 171, 232, 232, 247, 255, 74, 0],
            &[0, 0, 0, 0, 12, 230, 205, 2, 0],
            &[0, 0, 0, 0, 136, 254, 58, 0, 0],
            &[0, 0, 0, 35, 248, 162, 0, 0, 0],
            &[0, 0, 0, 177, 242, 25, 0, 0, 0],
            &[0, 0, 69, 255, 116, 0, 0, 0, 0],
            &[0, 2, 213, 252, 133, 127, 127, 46, 0],
            &[0, 10, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '[' / 0x5b
        '[' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 40, 166, 166, 106, 0, 0, 0],
            &[0, 0, 61, 255, 173, 93, 0, 0, 0],
            &[0, 0, 61, 255, 63, 0, 0, 0, 0],
            &[0, 0, 61, 255, 63, 0, 0, 0, 0],
            &[0, 0, 61, 255, 63, 0, 0, 0, 0],
            &[0, 0, 61, 255, 63, 0, 0, 0, 0],
            &[0, 0, 61, 255, 63, 0, 0, 0, 0],
            &[0, 0, 61, 255, 63, 0, 0, 0, 0],
            &[0, 0, 61, 255, 63, 0, 0, 0, 0],
            &[0, 0, 61, 255, 119, 47, 0, 0, 0],
            &[0, 0, 57, 241, 241, 154, 0, 0, 0],
        ],
        // letter: '\' / 0x5c
        '\\' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 107, 128, 0, 0, 0, 0, 0],
            &[0, 0, 96, 254, 33, 0, 0, 0, 0],
            &[0, 0, 11, 243, 129, 0, 0, 0, 0],
            &[0, 0, 0, 158, 224, 2, 0, 0, 0],
            &[0, 0, 0, 60, 255, 68, 0, 0, 0],
            &[0, 0, 0, 0, 218, 165, 0, 0, 0],
            &[0, 0, 0, 0, 122, 247, 16, 0, 0],
            &[0, 0, 0, 0, 28, 253, 105, 0, 0],
            &[0, 0, 0, 0, 0, 184, 202, 0, 0],
            &[0, 0, 0, 0, 0, 6, 12, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ']' / 0x5d
        ']' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 82, 188, 188, 82, 0, 0, 0],
            &[0, 0, 56, 134, 255, 112, 0, 0, 0],
            &[0, 0, 0, 12, 255, 112, 0, 0, 0],
            &[0, 0, 0, 12, 255, 112, 0, 0, 0],
            &[0, 0, 0, 12, 255, 112, 0, 0, 0],
            &[0, 0, 0, 12, 255, 112, 0, 0, 0],
            &[0, 0, 0, 12, 255, 112, 0, 0, 0],
            &[0, 0, 0, 12, 255, 112, 0, 0, 0],
            &[0, 0, 0, 12, 255, 112, 0, 0, 0],
            &[0, 0, 40, 100, 255, 112, 0, 0, 0],
            &[0, 0, 96, 220, 220, 96, 0, 0, 0],
        ],
        // letter: '^' / 0x5e
        '^' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 147, 13, 0, 0, 0],
            &[0, 0, 0, 110, 255, 125, 0, 0, 0],
            &[0, 0, 6, 228, 202, 240, 17, 0, 0],
            &[0, 0, 102, 238, 18, 220, 134, 0, 0],
            &[0, 4, 222, 131, 0, 94, 244, 22, 0],
            &[0, 67, 203, 20, 0, 3, 183, 105, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '_' / 0x5f
        '_' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[50, 50, 50, 50, 50, 50, 50, 10, 0],
            &[62, 62, 62, 62, 62, 62, 62, 12, 0],
        ],
        // letter: '`' / 0x60
        '`' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 49, 28, 0, 0, 0, 0],
            &[0, 0, 11, 192, 229, 16, 0, 0, 0],
            &[0, 0, 0, 8, 152, 102, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'a' / 0x61
        'a' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 142, 164, 140, 44, 0, 0],
            &[0, 0, 185, 218, 190, 239, 245, 34, 0],
            &[0, 0, 6, 0, 11, 89, 255, 97, 0],
            &[0, 3, 149, 241, 254, 250, 255, 107, 0],
            &[0, 76, 255, 139, 8, 69, 255, 107, 0],
            &[0, 85, 255, 170, 79, 197, 255, 107, 0],
            &[0, 10, 202, 255, 242, 122, 252, 107, 0],
            &[0, 0, 0, 20, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'b' / 0x62
        'b' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 30, 11, 0, 0, 0, 0, 0],
            &[0, 87, 255, 93, 0, 0, 0, 0, 0],
            &[0, 87, 255, 93, 0, 0, 0, 0, 0],
            &[0, 87, 255, 104, 126, 162, 70, 0, 0],
            &[0, 87, 255, 241, 211, 246, 251, 51, 0],
            &[0, 87, 255, 154, 0, 74, 255, 144, 0],
            &[0, 87, 255, 100, 0, 18, 255, 176, 0],
            &[0, 87, 255, 120, 0, 38, 255, 161, 0],
            &[0, 87, 255, 229, 98, 183, 255, 97, 0],
            &[0, 87, 255, 131, 241, 255, 174, 6, 0],
            &[0, 0, 0, 0, 2, 20, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'c' / 0x63
        'c' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 83, 155, 163, 123, 15, 0],
            &[0, 0, 131, 255, 245, 215, 234, 7, 0],
            &[0, 6, 243, 228, 17, 0, 6, 0, 0],
            &[0, 33, 255, 167, 0, 0, 0, 0, 0],
            &[0, 24, 255, 191, 0, 0, 0, 0, 0],
            &[0, 0, 212, 255, 147, 102, 168, 8, 0],
            &[0, 0, 47, 210, 255, 255, 219, 6, 0],
            &[0, 0, 0, 0, 17, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'd' / 0x64
        'd' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 5, 30, 16, 0],
            &[0, 0, 0, 0, 0, 41, 255, 138, 0],
            &[0, 0, 0, 0, 0, 41, 255, 138, 0],
            &[0, 0, 41, 149, 153, 77, 255, 138, 0],
            &[0, 17, 235, 255, 212, 236, 255, 138, 0],
            &[0, 96, 255, 136, 0, 74, 255, 138, 0],
            &[0, 127, 255, 71, 0, 16, 255, 138, 0],
            &[0, 111, 255, 85, 0, 28, 255, 138, 0],
            &[0, 50, 255, 210, 94, 178, 255, 138, 0],
            &[0, 0, 139, 254, 249, 124, 241, 138, 0],
            &[0, 0, 0, 16, 10, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'e' / 0x65
        'e' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 114, 163, 142, 39, 0, 0],
            &[0, 1, 190, 252, 185, 229, 242, 32, 0],
            &[0, 57, 255, 129, 0, 31, 255, 124, 0],
            &[0, 95, 255, 249, 245, 245, 255, 151, 0],
            &[0, 77, 255, 143, 58, 58, 58, 35, 0],
            &[0, 11, 233, 236, 118, 87, 125, 72, 0],
            &[0, 0, 50, 203, 255, 255, 239, 75, 0],
            &[0, 0, 0, 0, 13, 22, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'f' / 0x66
        'f' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 32, 37, 7, 0],
            &[0, 0, 0, 16, 208, 255, 255, 205, 0],
            &[0, 0, 0, 96, 255, 186, 85, 61, 0],
            &[0, 9, 85, 183, 255, 177, 141, 57, 0],
            &[0, 41, 227, 239, 255, 235, 227, 92, 0],
            &[0, 0, 0, 113, 255, 75, 0, 0, 0],
            &[0, 0, 0, 113, 255, 75, 0, 0, 0],
            &[0, 0, 0, 113, 255, 75, 0, 0, 0],
            &[0, 0, 0, 113, 255, 75, 0, 0, 0],
            &[0, 0, 0, 113, 255, 75, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'g' / 0x67
        'g' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 41, 149, 151, 28, 126, 76, 0],
            &[0, 13, 236, 253, 211, 229, 255, 138, 0],
            &[0, 93, 255, 129, 0, 88, 255, 138, 0],
            &[0, 124, 255, 69, 0, 32, 255, 138, 0],
            &[0, 110, 255, 89, 0, 41, 255, 138, 0],
            &[0, 48, 255, 214, 95, 183, 255, 138, 0],
            &[0, 0, 131, 251, 249, 169, 255, 138, 0],
            &[0, 1, 1, 9, 7, 77, 255, 124, 0],
            &[0, 9, 181, 228, 232, 193, 75, 0, 0],
        ],
        // letter: 'h' / 0x68
        'h' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 30, 12, 0, 0, 0, 0, 0],
            &[0, 87, 255, 102, 0, 0, 0, 0, 0],
            &[0, 87, 255, 102, 0, 0, 0, 0, 0],
            &[0, 87, 255, 111, 129, 160, 93, 0, 0],
            &[0, 87, 255, 240, 218, 241, 255, 80, 0],
            &[0, 87, 255, 176, 0, 68, 255, 143, 0],
            &[0, 87, 255, 114, 0, 36, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'i' / 0x69
        'i' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 51, 23, 0, 0, 0],
            &[0, 0, 0, 41, 255, 194, 0, 0, 0],
            &[0, 0, 0, 9, 152, 89, 0, 0, 0],
            &[0, 0, 135, 141, 141, 90, 0, 0, 0],
            &[0, 0, 180, 218, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 7, 53, 109, 255, 201, 72, 20, 0],
            &[0, 71, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'j' / 0x6a
        'j' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 25, 49, 0, 0, 0],
            &[0, 0, 0, 0, 205, 255, 30, 0, 0],
            &[0, 0, 0, 0, 100, 147, 4, 0, 0],
            &[0, 0, 63, 141, 141, 141, 20, 0, 0],
            &[0, 0, 81, 201, 241, 255, 37, 0, 0],
            &[0, 0, 0, 0, 151, 255, 37, 0, 0],
            &[0, 0, 0, 0, 151, 255, 37, 0, 0],
            &[0, 0, 0, 0, 151, 255, 37, 0, 0],
            &[0, 0, 0, 0, 151, 255, 37, 0, 0],
            &[0, 0, 0, 0, 151, 255, 37, 0, 0],
            &[0, 0, 0, 0, 178, 255, 26, 0, 0],
            &[0, 97, 221, 232, 191, 53, 0, 0, 0],
        ],
        // letter: 'k' / 0x6b
        'k' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[8, 30, 14, 0, 0, 0, 0, 0, 0],
            &[71, 255, 117, 0, 0, 0, 0, 0, 0],
            &[71, 255, 117, 0, 0, 0, 0, 0, 0],
            &[71, 255, 117, 0, 54, 141, 87, 0, 0],
            &[71, 255, 117, 32, 234, 219, 20, 0, 0],
            &[71, 255, 126, 212, 234, 35, 0, 0, 0],
            &[71, 255, 236, 255, 180, 0, 0, 0, 0],
            &[71, 255, 225, 155, 255, 89, 0, 0, 0],
            &[71, 255, 117, 4, 206, 236, 23, 0, 0],
            &[71, 255, 117, 0, 49, 251, 175, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'l' / 0x6c
        'l' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 30, 30, 19, 0, 0, 0],
            &[0, 0, 219, 255, 255, 158, 0, 0, 0],
            &[0, 0, 11, 82, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 7, 53, 113, 255, 198, 71, 20, 0],
            &[0, 71, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'm' / 0x6d
        'm' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 90, 85, 125, 134, 36, 149, 55, 0],
            &[0, 163, 242, 223, 255, 204, 245, 184, 0],
            &[0, 163, 227, 42, 255, 102, 171, 209, 0],
            &[0, 163, 220, 39, 255, 94, 168, 214, 0],
            &[0, 163, 220, 39, 255, 94, 168, 214, 0],
            &[0, 163, 220, 39, 255, 94, 168, 214, 0],
            &[0, 163, 220, 39, 255, 94, 168, 214, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'n' / 0x6e
        'n' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 48, 141, 28, 131, 161, 95, 0, 0],
            &[0, 87, 255, 235, 216, 242, 255, 82, 0],
            &[0, 87, 255, 178, 0, 71, 255, 144, 0],
            &[0, 87, 255, 111, 0, 36, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'o' / 0x6f
        'o' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 119, 164, 137, 33, 0, 0],
            &[0, 6, 205, 253, 204, 247, 238, 31, 0],
            &[0, 83, 255, 132, 0, 80, 255, 137, 0],
            &[0, 125, 255, 70, 0, 21, 255, 177, 0],
            &[0, 113, 255, 93, 0, 41, 255, 163, 0],
            &[0, 38, 251, 218, 94, 186, 255, 84, 0],
            &[0, 0, 90, 235, 255, 246, 131, 0, 0],
            &[0, 0, 0, 3, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'p' / 0x70
        'p' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 48, 141, 26, 129, 162, 71, 0, 0],
            &[0, 87, 255, 234, 212, 245, 251, 51, 0],
            &[0, 87, 255, 155, 0, 79, 255, 144, 0],
            &[0, 87, 255, 98, 0, 26, 255, 176, 0],
            &[0, 87, 255, 120, 0, 43, 255, 166, 0],
            &[0, 87, 255, 229, 99, 179, 255, 102, 0],
            &[0, 87, 255, 175, 243, 255, 178, 6, 0],
            &[0, 87, 255, 92, 6, 19, 0, 0, 0],
            &[0, 76, 225, 81, 0, 0, 0, 0, 0],
        ],
        // letter: 'q' / 0x71
        'q' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 41, 154, 142, 42, 121, 76, 0],
            &[0, 14, 236, 254, 210, 233, 254, 138, 0],
            &[0, 93, 255, 138, 0, 89, 255, 138, 0],
            &[0, 124, 255, 79, 0, 27, 255, 138, 0],
            &[0, 111, 255, 96, 0, 40, 255, 138, 0],
            &[0, 50, 255, 215, 95, 184, 255, 138, 0],
            &[0, 0, 140, 254, 250, 158, 255, 138, 0],
            &[0, 0, 0, 17, 11, 41, 255, 138, 0],
            &[0, 0, 0, 0, 0, 36, 225, 121, 0],
        ],
        // letter: 'r' / 0x72
        'r' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 141, 141, 81, 54, 158, 141, 0],
            &[0, 32, 200, 249, 218, 244, 255, 183, 0],
            &[0, 0, 0, 205, 255, 126, 12, 12, 0],
            &[0, 0, 0, 205, 250, 6, 0, 0, 0],
            &[0, 0, 0, 205, 239, 0, 0, 0, 0],
            &[0, 17, 74, 226, 246, 82, 24, 0, 0],
            &[0, 129, 255, 255, 255, 255, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 's' / 0x73
        's' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 24, 130, 166, 151, 97, 9, 0],
            &[0, 0, 205, 254, 203, 220, 237, 8, 0],
            &[0, 0, 233, 236, 51, 0, 14, 0, 0],
            &[0, 0, 94, 243, 255, 205, 66, 0, 0],
            &[0, 0, 0, 17, 113, 235, 251, 27, 0],
            &[0, 8, 179, 115, 94, 205, 255, 47, 0],
            &[0, 7, 224, 255, 255, 249, 150, 0, 0],
            &[0, 0, 0, 16, 26, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 't' / 0x74
        't' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 41, 0, 0, 0, 0],
            &[0, 0, 0, 144, 224, 0, 0, 0, 0],
            &[0, 21, 97, 218, 241, 141, 141, 47, 0],
            &[0, 82, 227, 252, 252, 227, 227, 76, 0],
            &[0, 0, 0, 221, 224, 0, 0, 0, 0],
            &[0, 0, 0, 221, 224, 0, 0, 0, 0],
            &[0, 0, 0, 220, 229, 0, 0, 0, 0],
            &[0, 0, 0, 194, 255, 121, 96, 48, 0],
            &[0, 0, 0, 65, 233, 255, 254, 91, 0],
            &[0, 0, 0, 0, 2, 24, 12, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'u' / 0x75
        'u' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 56, 141, 48, 0, 28, 141, 76, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 53, 255, 138, 0],
            &[0, 100, 255, 94, 0, 82, 255, 138, 0],
            &[0, 70, 255, 197, 92, 209, 255, 138, 0],
            &[0, 2, 177, 255, 248, 127, 237, 138, 0],
            &[0, 0, 0, 16, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'v' / 0x76
        'v' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 92, 141, 28, 0, 0, 134, 121, 0],
            &[0, 94, 255, 114, 0, 52, 255, 145, 0],
            &[0, 11, 243, 195, 0, 133, 255, 49, 0],
            &[0, 0, 160, 253, 23, 213, 209, 0, 0],
            &[0, 0, 64, 255, 133, 255, 114, 0, 0],
            &[0, 0, 1, 224, 238, 251, 23, 0, 0],
            &[0, 0, 0, 130, 255, 178, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'w' / 0x77
        'w' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[130, 91, 0, 0, 0, 57, 141, 19, 0],
            &[208, 182, 28, 151, 62, 122, 254, 8, 0],
            &[172, 205, 89, 255, 146, 144, 226, 0, 0],
            &[136, 227, 140, 224, 197, 167, 190, 0, 0],
            &[100, 250, 189, 140, 244, 190, 154, 0, 0],
            &[64, 255, 242, 54, 249, 241, 118, 0, 0],
            &[28, 255, 250, 6, 209, 255, 82, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'x' / 0x78
        'x' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 65, 141, 84, 0, 41, 141, 108, 0],
            &[0, 13, 224, 242, 25, 194, 252, 59, 0],
            &[0, 0, 64, 254, 216, 255, 133, 0, 0],
            &[0, 0, 0, 162, 255, 224, 5, 0, 0],
            &[0, 0, 25, 236, 252, 255, 69, 0, 0],
            &[0, 0, 183, 254, 78, 235, 226, 15, 0],
            &[0, 105, 255, 161, 0, 105, 255, 159, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'y' / 0x79
        'y' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 91, 141, 32, 0, 2, 137, 121, 0],
            &[0, 87, 255, 124, 0, 57, 255, 147, 0],
            &[0, 6, 235, 210, 0, 133, 255, 54, 0],
            &[0, 0, 140, 255, 40, 208, 217, 0, 0],
            &[0, 0, 39, 255, 148, 255, 124, 0, 0],
            &[0, 0, 0, 193, 246, 254, 32, 0, 0],
            &[0, 0, 0, 92, 255, 194, 0, 0, 0],
            &[0, 0, 0, 103, 255, 98, 0, 0, 0],
            &[0, 64, 235, 194, 42, 0, 0, 0, 0],
        ],
        // letter: 'z' / 0x7a
        'z' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 141, 141, 141, 141, 141, 44, 0],
            &[0, 8, 208, 208, 208, 253, 255, 63, 0],
            &[0, 0, 0, 0, 111, 255, 148, 0, 0],
            &[0, 0, 0, 66, 251, 189, 4, 0, 0],
            &[0, 0, 33, 236, 220, 18, 0, 0, 0],
            &[0, 10, 209, 255, 136, 94, 94, 39, 0],
            &[0, 56, 255, 255, 255, 255, 255, 107, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '{' / 0x7b
        '{' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 41, 134, 85, 0, 0],
            &[0, 0, 0, 0, 216, 243, 100, 0, 0],
            &[0, 0, 0, 0, 254, 138, 0, 0, 0],
            &[0, 0, 0, 0, 255, 131, 0, 0, 0],
            &[0, 0, 1, 64, 255, 118, 0, 0, 0],
            &[0, 0, 91, 255, 202, 17, 0, 0, 0],
            &[0, 0, 31, 145, 255, 93, 0, 0, 0],
            &[0, 0, 0, 3, 255, 130, 0, 0, 0],
            &[0, 0, 0, 0, 255, 132, 0, 0, 0],
            &[0, 0, 0, 0, 244, 215, 58, 0, 0],
            &[0, 0, 0, 0, 106, 219, 132, 0, 0],
        ],
        // letter: '|' / 0x7c
        '|' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 30, 7, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 16, 225, 58, 0, 0, 0],
        ],
        // letter: '}' / 0x7d
        '}' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 55, 142, 65, 0, 0, 0, 0],
            &[0, 0, 64, 233, 252, 23, 0, 0, 0],
            &[0, 0, 0, 92, 255, 50, 0, 0, 0],
            &[0, 0, 0, 79, 255, 51, 0, 0, 0],
            &[0, 0, 0, 67, 255, 113, 3, 0, 0],
            &[0, 0, 0, 3, 165, 255, 142, 0, 0],
            &[0, 0, 0, 44, 253, 176, 50, 0, 0],
            &[0, 0, 0, 79, 255, 54, 0, 0, 0],
            &[0, 0, 0, 80, 255, 51, 0, 0, 0],
            &[0, 0, 33, 178, 255, 31, 0, 0, 0],
            &[0, 0, 85, 225, 145, 0, 0, 0, 0],
        ],
        // letter: '~' / 0x7e
        '~' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 51, 120, 31, 0, 73, 47, 0],
            &[0, 9, 224, 123, 223, 77, 201, 54, 0],
            &[0, 33, 140, 0, 43, 170, 128, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: ' ' / 0xa0
        ' ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¡' / 0xa1
        '¡' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 104, 30, 0, 0, 0],
            &[0, 0, 0, 98, 255, 165, 0, 0, 0],
            &[0, 0, 0, 26, 142, 58, 0, 0, 0],
            &[0, 0, 0, 23, 128, 47, 0, 0, 0],
            &[0, 0, 0, 54, 255, 102, 0, 0, 0],
            &[0, 0, 0, 65, 255, 113, 0, 0, 0],
            &[0, 0, 0, 76, 255, 124, 0, 0, 0],
            &[0, 0, 0, 88, 255, 136, 0, 0, 0],
            &[0, 0, 0, 23, 56, 33, 0, 0, 0],
        ],
        // letter: '¢' / 0xa2
        '¢' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 72, 96, 0, 0, 0],
            &[0, 0, 1, 100, 203, 225, 157, 32, 0],
            &[0, 0, 139, 255, 209, 167, 211, 20, 0],
            &[0, 7, 245, 195, 3, 0, 0, 0, 0],
            &[0, 33, 255, 134, 0, 0, 0, 0, 0],
            &[0, 15, 254, 173, 0, 0, 0, 0, 0],
            &[0, 0, 183, 255, 151, 106, 150, 39, 0],
            &[0, 0, 21, 167, 241, 251, 207, 29, 0],
            &[0, 0, 0, 0, 105, 140, 0, 0, 0],
            &[0, 0, 0, 0, 12, 16, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '£' / 0xa3
        '£' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 24, 135, 167, 131, 25, 0],
            &[0, 0, 0, 205, 254, 208, 236, 32, 0],
            &[0, 0, 18, 255, 162, 0, 4, 0, 0],
            &[0, 0, 30, 255, 144, 1, 0, 0, 0],
            &[0, 10, 255, 255, 255, 255, 133, 0, 0],
            &[0, 2, 84, 255, 171, 62, 32, 0, 0],
            &[0, 0, 48, 255, 121, 0, 0, 0, 0],
            &[0, 2, 200, 249, 135, 121, 121, 58, 0],
            &[0, 7, 255, 255, 255, 255, 255, 122, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¤' / 0xa4
        '¤' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 0, 0, 0, 26, 0, 0],
            &[0, 19, 240, 177, 221, 178, 243, 54, 0],
            &[0, 0, 161, 217, 99, 187, 209, 0, 0],
            &[0, 0, 207, 114, 0, 63, 253, 2, 0],
            &[0, 0, 163, 209, 83, 178, 211, 0, 0],
            &[0, 18, 239, 195, 236, 195, 246, 48, 0],
            &[0, 0, 36, 0, 0, 0, 39, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¥' / 0xa5
        '¥' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 81, 145, 27, 0, 3, 140, 109, 0],
            &[0, 44, 254, 138, 0, 87, 255, 91, 0],
            &[0, 0, 170, 239, 12, 201, 214, 2, 0],
            &[0, 0, 44, 254, 168, 255, 87, 0, 0],
            &[0, 0, 54, 200, 255, 232, 59, 8, 0],
            &[0, 0, 198, 219, 255, 228, 209, 30, 0],
            &[0, 0, 87, 128, 255, 160, 93, 13, 0],
            &[0, 0, 161, 189, 255, 206, 170, 24, 0],
            &[0, 0, 0, 55, 255, 106, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¦' / 0xa6
        '¦' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 24, 6, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 12, 167, 43, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 0, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 18, 255, 66, 0, 0, 0],
            &[0, 0, 0, 14, 194, 50, 0, 0, 0],
        ],
        // letter: '§' / 0xa7
        '§' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 6, 36, 23, 0, 0, 0],
            &[0, 0, 62, 234, 249, 254, 166, 0, 0],
            &[0, 0, 156, 244, 10, 18, 41, 0, 0],
            &[0, 0, 59, 248, 214, 93, 3, 0, 0],
            &[0, 0, 105, 233, 140, 246, 156, 0, 0],
            &[0, 0, 173, 213, 12, 138, 232, 0, 0],
            &[0, 0, 57, 233, 230, 230, 131, 0, 0],
            &[0, 0, 0, 12, 119, 245, 165, 0, 0],
            &[0, 0, 78, 28, 12, 197, 215, 0, 0],
            &[0, 0, 155, 254, 255, 232, 78, 0, 0],
            &[0, 0, 0, 9, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¨' / 0xa8
        '¨' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 0, 2, 3, 0, 0],
            &[0, 0, 125, 236, 6, 176, 192, 0, 0],
            &[0, 0, 62, 136, 0, 94, 105, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '©' / 0xa9
        '©' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 69, 111, 84, 8, 0, 0],
            &[0, 5, 185, 154, 139, 154, 205, 24, 0],
            &[0, 107, 140, 184, 168, 129, 83, 155, 0],
            &[0, 166, 81, 224, 0, 0, 6, 216, 0],
            &[0, 151, 87, 239, 17, 11, 26, 205, 0],
            &[0, 60, 196, 106, 200, 142, 159, 112, 0],
            &[0, 0, 95, 209, 187, 209, 129, 1, 0],
            &[0, 0, 0, 0, 11, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ª' / 0xaa
        'ª' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 97, 159, 131, 8, 0, 0],
            &[0, 0, 0, 77, 73, 218, 100, 0, 0],
            &[0, 0, 13, 195, 184, 225, 120, 0, 0],
            &[0, 0, 56, 245, 94, 232, 120, 0, 0],
            &[0, 0, 1, 109, 122, 62, 60, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '«' / 0xab
        '«' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 164, 129, 38, 207, 48, 0],
            &[0, 0, 108, 255, 89, 209, 216, 9, 0],
            &[0, 35, 249, 173, 139, 255, 65, 0, 0],
            &[0, 2, 188, 236, 77, 249, 152, 0, 0],
            &[0, 0, 22, 230, 167, 107, 255, 58, 0],
            &[0, 0, 0, 42, 17, 0, 59, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¬' / 0xac
        '¬' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 35, 223, 223, 223, 223, 223, 53, 0],
            &[0, 16, 102, 102, 102, 109, 255, 61, 0],
            &[0, 0, 0, 0, 0, 11, 255, 61, 0],
            &[0, 0, 0, 0, 0, 6, 143, 34, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '®' / 0xae
        '®' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 2, 69, 111, 83, 8, 0, 0],
            &[0, 5, 183, 174, 118, 130, 203, 22, 0],
            &[0, 105, 143, 178, 169, 200, 84, 153, 0],
            &[0, 163, 61, 178, 147, 204, 9, 213, 0],
            &[0, 148, 78, 178, 133, 152, 29, 202, 0],
            &[0, 56, 197, 128, 30, 150, 165, 110, 0],
            &[0, 0, 90, 207, 190, 209, 128, 1, 0],
            &[0, 0, 0, 0, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¯' / 0xaf
        '¯' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 35, 166, 166, 166, 70, 0, 0],
            &[0, 0, 29, 137, 137, 137, 58, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '°' / 0xb0
        '°' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 69, 158, 99, 0, 0, 0],
            &[0, 0, 29, 243, 112, 223, 79, 0, 0],
            &[0, 0, 56, 228, 14, 181, 111, 0, 0],
            &[0, 0, 3, 170, 251, 213, 23, 0, 0],
            &[0, 0, 0, 0, 11, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '±' / 0xb1
        '±' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 134, 34, 0, 0, 0],
            &[0, 0, 0, 6, 255, 66, 0, 0, 0],
            &[0, 10, 80, 85, 255, 126, 80, 29, 0],
            &[0, 33, 244, 244, 255, 247, 244, 88, 0],
            &[0, 0, 0, 6, 255, 66, 0, 0, 0],
            &[0, 0, 0, 6, 255, 66, 0, 0, 0],
            &[0, 11, 72, 73, 108, 81, 72, 26, 0],
            &[0, 40, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '²' / 0xb2
        '²' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 7, 50, 20, 0, 0, 0],
            &[0, 0, 10, 223, 225, 242, 25, 0, 0],
            &[0, 0, 0, 27, 10, 254, 51, 0, 0],
            &[0, 0, 0, 1, 152, 193, 3, 0, 0],
            &[0, 0, 2, 156, 201, 23, 4, 0, 0],
            &[0, 0, 38, 255, 255, 255, 133, 0, 0],
            &[0, 0, 0, 2, 2, 2, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '³' / 0xb3
        '³' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 49, 16, 0, 0, 0],
            &[0, 0, 10, 221, 212, 238, 23, 0, 0],
            &[0, 0, 0, 17, 44, 247, 29, 0, 0],
            &[0, 0, 0, 71, 236, 185, 9, 0, 0],
            &[0, 0, 15, 25, 2, 235, 81, 0, 0],
            &[0, 0, 33, 240, 244, 221, 29, 0, 0],
            &[0, 0, 0, 3, 18, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '´' / 0xb4
        '´' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 9, 49, 36, 0, 0, 0],
            &[0, 0, 0, 146, 242, 61, 0, 0, 0],
            &[0, 0, 31, 181, 50, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'µ' / 0xb5
        'µ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 56, 141, 48, 0, 28, 141, 76, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 53, 255, 138, 0],
            &[0, 175, 255, 157, 0, 82, 255, 138, 0],
            &[0, 172, 255, 255, 92, 209, 255, 138, 0],
            &[0, 104, 255, 255, 248, 127, 237, 138, 0],
            &[0, 102, 255, 103, 8, 0, 0, 0, 0],
            &[0, 90, 225, 76, 0, 0, 0, 0, 0],
        ],
        // letter: '¶' / 0xb6
        '¶' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 29, 30, 30, 18, 0],
            &[0, 1, 153, 251, 255, 250, 236, 153, 0],
            &[0, 76, 255, 255, 255, 210, 82, 153, 0],
            &[0, 124, 255, 255, 255, 210, 82, 153, 0],
            &[0, 119, 255, 255, 255, 210, 82, 153, 0],
            &[0, 54, 255, 255, 255, 210, 82, 153, 0],
            &[0, 0, 94, 194, 208, 210, 82, 153, 0],
            &[0, 0, 0, 0, 22, 210, 82, 153, 0],
            &[0, 0, 0, 0, 22, 210, 82, 153, 0],
            &[0, 0, 0, 0, 22, 210, 82, 153, 0],
            &[0, 0, 0, 0, 22, 210, 82, 153, 0],
            &[0, 0, 0, 0, 11, 110, 43, 80, 0],
        ],
        // letter: '·' / 0xb7
        '·' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 37, 3, 0, 0, 0],
            &[0, 0, 0, 103, 255, 140, 0, 0, 0],
            &[0, 0, 0, 59, 206, 88, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¸' / 0xb8
        '¸' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 12, 1, 0, 0, 0],
            &[0, 0, 0, 19, 244, 61, 0, 0, 0],
            &[0, 0, 0, 129, 216, 114, 0, 0, 0],
        ],
        // letter: '¹' / 0xb9
        '¹' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 19, 13, 0, 0, 0],
            &[0, 0, 4, 123, 248, 103, 0, 0, 0],
            &[0, 0, 14, 158, 205, 103, 0, 0, 0],
            &[0, 0, 0, 0, 190, 103, 0, 0, 0],
            &[0, 0, 0, 0, 191, 103, 0, 0, 0],
            &[0, 0, 0, 195, 255, 255, 117, 0, 0],
            &[0, 0, 0, 1, 3, 2, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'º' / 0xba
        'º' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 73, 156, 89, 0, 0, 0],
            &[0, 0, 25, 248, 99, 231, 65, 0, 0],
            &[0, 0, 60, 237, 0, 190, 109, 0, 0],
            &[0, 0, 19, 243, 128, 240, 56, 0, 0],
            &[0, 0, 0, 52, 134, 76, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '»' / 0xbb
        '»' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 19, 196, 77, 88, 195, 10, 0, 0],
            &[0, 0, 176, 238, 71, 246, 159, 0, 0],
            &[0, 0, 22, 247, 191, 122, 255, 80, 0],
            &[0, 0, 100, 255, 101, 204, 224, 17, 0],
            &[0, 19, 242, 158, 116, 249, 54, 0, 0],
            &[0, 0, 51, 9, 6, 54, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¼' / 0xbc
        '¼' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 20, 130, 38, 0, 0, 0, 0, 0],
            &[0, 160, 224, 68, 0, 0, 70, 66, 0],
            &[0, 0, 176, 68, 0, 58, 239, 40, 0],
            &[0, 50, 255, 158, 32, 227, 91, 0, 0],
            &[0, 71, 195, 148, 202, 154, 6, 2, 0],
            &[0, 0, 0, 122, 208, 42, 244, 79, 0],
            &[0, 0, 63, 238, 39, 179, 198, 79, 0],
            &[0, 22, 230, 87, 84, 211, 216, 167, 0],
            &[0, 13, 29, 0, 26, 65, 191, 120, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '½' / 0xbd
        '½' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[20, 130, 38, 0, 0, 0, 0, 0, 0],
            &[160, 224, 68, 0, 0, 70, 66, 0, 0],
            &[0, 176, 68, 0, 58, 239, 40, 0, 0],
            &[50, 255, 158, 32, 227, 91, 0, 0, 0],
            &[71, 195, 148, 202, 154, 21, 1, 0, 0],
            &[0, 0, 122, 208, 166, 213, 168, 0, 0],
            &[0, 63, 238, 37, 3, 88, 183, 0, 0],
            &[22, 230, 87, 0, 62, 214, 32, 0, 0],
            &[13, 29, 0, 0, 218, 237, 224, 1, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¾' / 0xbe
        '¾' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 54, 158, 126, 1, 0, 0, 0, 0],
            &[0, 39, 63, 239, 19, 0, 70, 66, 0],
            &[0, 0, 188, 206, 9, 58, 239, 40, 0],
            &[0, 80, 98, 229, 64, 227, 91, 0, 0],
            &[0, 47, 123, 83, 183, 154, 6, 2, 0],
            &[0, 0, 0, 122, 208, 42, 244, 79, 0],
            &[0, 0, 63, 238, 39, 179, 198, 79, 0],
            &[0, 22, 230, 87, 84, 211, 216, 167, 0],
            &[0, 13, 29, 0, 26, 65, 191, 120, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '¿' / 0xbf
        '¿' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 103, 33, 0, 0, 0],
            &[0, 0, 0, 89, 255, 174, 0, 0, 0],
            &[0, 0, 0, 22, 141, 63, 0, 0, 0],
            &[0, 0, 0, 16, 128, 55, 0, 0, 0],
            &[0, 0, 0, 108, 255, 79, 0, 0, 0],
            &[0, 0, 129, 255, 159, 1, 0, 0, 0],
            &[0, 47, 255, 158, 0, 0, 0, 0, 0],
            &[0, 71, 255, 157, 38, 90, 78, 0, 0],
            &[0, 0, 4, 62, 76, 39, 0, 0, 0],
        ],
        // letter: 'À' / 0xc0
        'À' => &[
            &[0, 21, 53, 28, 0, 0, 0, 0, 0],
            &[0, 15, 199, 222, 11, 0, 0, 0, 0],
            &[0, 0, 11, 155, 90, 0, 0, 0, 0],
            &[0, 0, 92, 154, 121, 0, 0, 0, 0],
            &[0, 0, 209, 251, 248, 10, 0, 0, 0],
            &[0, 24, 255, 164, 255, 72, 0, 0, 0],
            &[0, 94, 255, 52, 246, 143, 0, 0, 0],
            &[0, 164, 237, 1, 190, 214, 0, 0, 0],
            &[1, 233, 255, 144, 255, 255, 29, 0, 0],
            &[49, 255, 255, 224, 255, 255, 99, 0, 0],
            &[119, 255, 47, 0, 10, 250, 170, 0, 0],
            &[190, 237, 1, 0, 0, 199, 238, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Á' / 0xc1
        'Á' => &[
            &[0, 0, 0, 5, 53, 44, 0, 0, 0],
            &[0, 0, 0, 124, 247, 75, 0, 0, 0],
            &[0, 0, 18, 176, 62, 0, 0, 0, 0],
            &[0, 0, 92, 154, 121, 0, 0, 0, 0],
            &[0, 0, 209, 251, 248, 10, 0, 0, 0],
            &[0, 24, 255, 164, 255, 72, 0, 0, 0],
            &[0, 94, 255, 52, 246, 143, 0, 0, 0],
            &[0, 164, 237, 1, 190, 214, 0, 0, 0],
            &[1, 233, 255, 144, 255, 255, 29, 0, 0],
            &[49, 255, 255, 224, 255, 255, 99, 0, 0],
            &[119, 255, 47, 0, 10, 250, 170, 0, 0],
            &[190, 237, 1, 0, 0, 199, 238, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Â' / 0xc2
        'Â' => &[
            &[0, 0, 18, 53, 29, 0, 0, 0, 0],
            &[0, 2, 190, 233, 228, 20, 0, 0, 0],
            &[0, 75, 153, 12, 126, 112, 0, 0, 0],
            &[0, 0, 92, 154, 121, 0, 0, 0, 0],
            &[0, 0, 209, 251, 248, 10, 0, 0, 0],
            &[0, 24, 255, 164, 255, 72, 0, 0, 0],
            &[0, 94, 255, 52, 246, 143, 0, 0, 0],
            &[0, 164, 237, 1, 190, 214, 0, 0, 0],
            &[1, 233, 255, 144, 255, 255, 29, 0, 0],
            &[49, 255, 255, 224, 255, 255, 99, 0, 0],
            &[119, 255, 47, 0, 10, 250, 170, 0, 0],
            &[190, 237, 1, 0, 0, 199, 238, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ã' / 0xc3
        'Ã' => &[
            &[0, 0, 7, 0, 0, 19, 0, 0, 0],
            &[0, 73, 251, 210, 147, 196, 0, 0, 0],
            &[0, 115, 80, 96, 169, 57, 0, 0, 0],
            &[0, 0, 92, 154, 121, 0, 0, 0, 0],
            &[0, 0, 209, 251, 248, 10, 0, 0, 0],
            &[0, 24, 255, 164, 255, 72, 0, 0, 0],
            &[0, 94, 255, 52, 246, 143, 0, 0, 0],
            &[0, 164, 237, 1, 190, 214, 0, 0, 0],
            &[1, 233, 255, 144, 255, 255, 29, 0, 0],
            &[49, 255, 255, 224, 255, 255, 99, 0, 0],
            &[119, 255, 47, 0, 10, 250, 170, 0, 0],
            &[190, 237, 1, 0, 0, 199, 238, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ä' / 0xc4
        'Ä' => &[
            &[0, 0, 6, 0, 3, 3, 0, 0, 0],
            &[0, 136, 232, 2, 188, 185, 0, 0, 0],
            &[0, 65, 126, 0, 97, 95, 0, 0, 0],
            &[0, 0, 92, 154, 121, 0, 0, 0, 0],
            &[0, 0, 209, 251, 248, 10, 0, 0, 0],
            &[0, 24, 255, 164, 255, 72, 0, 0, 0],
            &[0, 94, 255, 52, 246, 143, 0, 0, 0],
            &[0, 164, 237, 1, 190, 214, 0, 0, 0],
            &[1, 233, 255, 144, 255, 255, 29, 0, 0],
            &[49, 255, 255, 224, 255, 255, 99, 0, 0],
            &[119, 255, 47, 0, 10, 250, 170, 0, 0],
            &[190, 237, 1, 0, 0, 199, 238, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Å' / 0xc5
        'Å' => &[
            &[0, 0, 129, 223, 170, 0, 0, 0, 0],
            &[0, 0, 201, 70, 235, 1, 0, 0, 0],
            &[0, 0, 64, 171, 90, 0, 0, 0, 0],
            &[0, 0, 92, 154, 121, 0, 0, 0, 0],
            &[0, 0, 209, 251, 248, 10, 0, 0, 0],
            &[0, 24, 255, 164, 255, 72, 0, 0, 0],
            &[0, 94, 255, 52, 246, 143, 0, 0, 0],
            &[0, 164, 237, 1, 190, 214, 0, 0, 0],
            &[1, 233, 255, 144, 255, 255, 29, 0, 0],
            &[49, 255, 255, 224, 255, 255, 99, 0, 0],
            &[119, 255, 47, 0, 10, 250, 170, 0, 0],
            &[190, 237, 1, 0, 0, 199, 238, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 3, 0, 0, 0, 0, 0],
        ],
        // letter: 'Æ' / 0xc6
        'Æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 131, 145, 145, 145, 122, 0],
            &[0, 0, 24, 255, 247, 255, 238, 197, 0],
            &[0, 0, 85, 255, 115, 255, 40, 0, 0],
            &[0, 0, 145, 239, 76, 255, 83, 31, 0],
            &[0, 0, 206, 184, 74, 255, 255, 159, 0],
            &[0, 14, 253, 255, 228, 255, 103, 46, 0],
            &[0, 72, 255, 255, 255, 255, 40, 0, 0],
            &[0, 133, 254, 16, 74, 255, 145, 104, 0],
            &[0, 194, 214, 0, 74, 255, 255, 214, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ç' / 0xc7
        'Ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 41, 137, 169, 149, 59, 0],
            &[0, 0, 77, 250, 254, 225, 245, 112, 0],
            &[0, 6, 231, 233, 40, 0, 10, 10, 0],
            &[0, 63, 255, 138, 0, 0, 0, 0, 0],
            &[0, 95, 255, 101, 0, 0, 0, 0, 0],
            &[0, 89, 255, 106, 0, 0, 0, 0, 0],
            &[0, 47, 255, 172, 0, 0, 0, 0, 0],
            &[0, 0, 200, 255, 162, 110, 148, 80, 0],
            &[0, 0, 26, 186, 255, 255, 238, 72, 0],
            &[0, 0, 0, 0, 169, 183, 7, 0, 0],
            &[0, 0, 0, 38, 219, 184, 17, 0, 0],
        ],
        // letter: 'È' / 0xc8
        'È' => &[
            &[0, 0, 17, 53, 32, 0, 0, 0, 0],
            &[0, 0, 9, 187, 232, 19, 0, 0, 0],
            &[0, 0, 0, 7, 146, 103, 0, 0, 0],
            &[0, 5, 145, 145, 145, 145, 145, 52, 0],
            &[0, 10, 255, 249, 238, 238, 238, 85, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 181, 50, 50, 50, 9, 0],
            &[0, 10, 255, 255, 255, 255, 255, 46, 0],
            &[0, 10, 255, 191, 77, 77, 77, 14, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 209, 127, 127, 127, 46, 0],
            &[0, 10, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'É' / 0xc9
        'É' => &[
            &[0, 0, 0, 0, 1, 53, 48, 0, 0],
            &[0, 0, 0, 0, 106, 251, 90, 0, 0],
            &[0, 0, 0, 10, 171, 74, 0, 0, 0],
            &[0, 5, 145, 145, 145, 145, 145, 52, 0],
            &[0, 10, 255, 249, 238, 238, 238, 85, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 181, 50, 50, 50, 9, 0],
            &[0, 10, 255, 255, 255, 255, 255, 46, 0],
            &[0, 10, 255, 191, 77, 77, 77, 14, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 209, 127, 127, 127, 46, 0],
            &[0, 10, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ê' / 0xca
        'Ê' => &[
            &[0, 0, 0, 14, 53, 33, 0, 0, 0],
            &[0, 0, 0, 174, 233, 238, 29, 0, 0],
            &[0, 0, 62, 160, 17, 114, 126, 0, 0],
            &[0, 5, 145, 145, 145, 145, 145, 52, 0],
            &[0, 10, 255, 249, 238, 238, 238, 85, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 181, 50, 50, 50, 9, 0],
            &[0, 10, 255, 255, 255, 255, 255, 46, 0],
            &[0, 10, 255, 191, 77, 77, 77, 14, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 209, 127, 127, 127, 46, 0],
            &[0, 10, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ë' / 0xcb
        'Ë' => &[
            &[0, 0, 0, 7, 0, 2, 5, 0, 0],
            &[0, 0, 118, 242, 10, 169, 204, 0, 0],
            &[0, 0, 55, 136, 0, 85, 107, 0, 0],
            &[0, 5, 145, 145, 145, 145, 145, 52, 0],
            &[0, 10, 255, 249, 238, 238, 238, 85, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 181, 50, 50, 50, 9, 0],
            &[0, 10, 255, 255, 255, 255, 255, 46, 0],
            &[0, 10, 255, 191, 77, 77, 77, 14, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 209, 127, 127, 127, 46, 0],
            &[0, 10, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ì' / 0xcc
        'Ì' => &[
            &[0, 21, 53, 28, 0, 0, 0, 0, 0],
            &[0, 15, 199, 222, 11, 0, 0, 0, 0],
            &[0, 0, 11, 155, 90, 0, 0, 0, 0],
            &[0, 133, 145, 145, 145, 145, 17, 0, 0],
            &[0, 161, 213, 255, 224, 179, 19, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 59, 131, 255, 160, 69, 6, 0, 0],
            &[0, 235, 255, 255, 255, 255, 30, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Í' / 0xcd
        'Í' => &[
            &[0, 0, 0, 5, 53, 44, 0, 0, 0],
            &[0, 0, 0, 124, 247, 75, 0, 0, 0],
            &[0, 0, 18, 176, 62, 0, 0, 0, 0],
            &[0, 133, 145, 145, 145, 145, 17, 0, 0],
            &[0, 161, 213, 255, 224, 179, 19, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 59, 131, 255, 160, 69, 6, 0, 0],
            &[0, 235, 255, 255, 255, 255, 30, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Î' / 0xce
        'Î' => &[
            &[0, 0, 18, 53, 29, 0, 0, 0, 0],
            &[0, 2, 190, 233, 228, 20, 0, 0, 0],
            &[0, 75, 153, 12, 126, 112, 0, 0, 0],
            &[0, 133, 145, 145, 145, 145, 17, 0, 0],
            &[0, 161, 213, 255, 224, 179, 19, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 59, 131, 255, 160, 69, 6, 0, 0],
            &[0, 235, 255, 255, 255, 255, 30, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ï' / 0xcf
        'Ï' => &[
            &[0, 0, 6, 0, 3, 3, 0, 0, 0],
            &[0, 136, 232, 2, 188, 185, 0, 0, 0],
            &[0, 65, 126, 0, 97, 95, 0, 0, 0],
            &[0, 133, 145, 145, 145, 145, 17, 0, 0],
            &[0, 161, 213, 255, 224, 179, 19, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 59, 131, 255, 160, 69, 6, 0, 0],
            &[0, 235, 255, 255, 255, 255, 30, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ð' / 0xd0
        'Ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 145, 145, 135, 85, 5, 0, 0],
            &[0, 117, 255, 238, 244, 255, 193, 6, 0],
            &[0, 117, 255, 65, 3, 139, 255, 110, 0],
            &[0, 117, 255, 65, 0, 15, 254, 185, 0],
            &[0, 255, 255, 255, 129, 0, 241, 208, 0],
            &[0, 241, 255, 189, 65, 4, 253, 195, 0],
            &[0, 117, 255, 65, 0, 79, 255, 139, 0],
            &[0, 117, 255, 156, 148, 243, 241, 29, 0],
            &[0, 117, 255, 255, 242, 185, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ñ' / 0xd1
        'Ñ' => &[
            &[0, 0, 0, 6, 0, 0, 19, 0, 0],
            &[0, 0, 61, 248, 216, 141, 211, 0, 0],
            &[0, 0, 103, 88, 90, 168, 66, 0, 0],
            &[0, 66, 145, 141, 3, 0, 129, 95, 0],
            &[0, 117, 255, 255, 64, 0, 227, 168, 0],
            &[0, 117, 253, 214, 148, 0, 227, 168, 0],
            &[0, 117, 255, 143, 230, 1, 227, 168, 0],
            &[0, 117, 255, 69, 255, 59, 227, 168, 0],
            &[0, 117, 255, 22, 220, 143, 223, 168, 0],
            &[0, 117, 255, 22, 137, 226, 217, 168, 0],
            &[0, 117, 255, 22, 54, 255, 248, 168, 0],
            &[0, 117, 255, 22, 1, 226, 255, 168, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ò' / 0xd2
        'Ò' => &[
            &[0, 0, 21, 53, 28, 0, 0, 0, 0],
            &[0, 0, 15, 199, 222, 11, 0, 0, 0],
            &[0, 0, 0, 11, 155, 90, 0, 0, 0],
            &[0, 0, 18, 130, 174, 148, 38, 0, 0],
            &[0, 4, 208, 255, 216, 251, 238, 29, 0],
            &[0, 82, 255, 136, 0, 90, 255, 136, 0],
            &[0, 139, 255, 60, 0, 8, 254, 192, 0],
            &[0, 158, 255, 38, 0, 0, 241, 211, 0],
            &[0, 152, 255, 50, 0, 0, 247, 203, 0],
            &[0, 113, 255, 90, 0, 40, 255, 164, 0],
            &[0, 30, 249, 225, 109, 199, 255, 75, 0],
            &[0, 0, 85, 234, 255, 245, 125, 0, 0],
            &[0, 0, 0, 3, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ó' / 0xd3
        'Ó' => &[
            &[0, 0, 0, 0, 5, 53, 44, 0, 0],
            &[0, 0, 0, 0, 124, 247, 75, 0, 0],
            &[0, 0, 0, 18, 176, 62, 0, 0, 0],
            &[0, 0, 18, 130, 174, 148, 38, 0, 0],
            &[0, 4, 208, 255, 216, 251, 238, 29, 0],
            &[0, 82, 255, 136, 0, 90, 255, 136, 0],
            &[0, 139, 255, 60, 0, 8, 254, 192, 0],
            &[0, 158, 255, 38, 0, 0, 241, 211, 0],
            &[0, 152, 255, 50, 0, 0, 247, 203, 0],
            &[0, 113, 255, 90, 0, 40, 255, 164, 0],
            &[0, 30, 249, 225, 109, 199, 255, 75, 0],
            &[0, 0, 85, 234, 255, 245, 125, 0, 0],
            &[0, 0, 0, 3, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ô' / 0xd4
        'Ô' => &[
            &[0, 0, 0, 18, 53, 29, 0, 0, 0],
            &[0, 0, 2, 190, 233, 228, 20, 0, 0],
            &[0, 0, 75, 153, 12, 126, 112, 0, 0],
            &[0, 0, 18, 130, 174, 148, 38, 0, 0],
            &[0, 4, 208, 255, 216, 251, 238, 29, 0],
            &[0, 82, 255, 136, 0, 90, 255, 136, 0],
            &[0, 139, 255, 60, 0, 8, 254, 192, 0],
            &[0, 158, 255, 38, 0, 0, 241, 211, 0],
            &[0, 152, 255, 50, 0, 0, 247, 203, 0],
            &[0, 113, 255, 90, 0, 40, 255, 164, 0],
            &[0, 30, 249, 225, 109, 199, 255, 75, 0],
            &[0, 0, 85, 234, 255, 245, 125, 0, 0],
            &[0, 0, 0, 3, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Õ' / 0xd5
        'Õ' => &[
            &[0, 0, 0, 7, 0, 0, 19, 0, 0],
            &[0, 0, 73, 251, 210, 147, 196, 0, 0],
            &[0, 0, 115, 80, 96, 169, 57, 0, 0],
            &[0, 0, 18, 130, 174, 148, 38, 0, 0],
            &[0, 4, 208, 255, 216, 251, 238, 29, 0],
            &[0, 82, 255, 136, 0, 90, 255, 136, 0],
            &[0, 139, 255, 60, 0, 8, 254, 192, 0],
            &[0, 158, 255, 38, 0, 0, 241, 211, 0],
            &[0, 152, 255, 50, 0, 0, 247, 203, 0],
            &[0, 113, 255, 90, 0, 40, 255, 164, 0],
            &[0, 30, 249, 225, 109, 199, 255, 75, 0],
            &[0, 0, 85, 234, 255, 245, 125, 0, 0],
            &[0, 0, 0, 3, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ö' / 0xd6
        'Ö' => &[
            &[0, 0, 0, 6, 0, 3, 3, 0, 0],
            &[0, 0, 136, 232, 2, 188, 185, 0, 0],
            &[0, 0, 65, 126, 0, 97, 95, 0, 0],
            &[0, 0, 18, 130, 174, 148, 38, 0, 0],
            &[0, 4, 208, 255, 216, 251, 238, 29, 0],
            &[0, 82, 255, 136, 0, 90, 255, 136, 0],
            &[0, 139, 255, 60, 0, 8, 254, 192, 0],
            &[0, 158, 255, 38, 0, 0, 241, 211, 0],
            &[0, 152, 255, 50, 0, 0, 247, 203, 0],
            &[0, 113, 255, 90, 0, 40, 255, 164, 0],
            &[0, 30, 249, 225, 109, 199, 255, 75, 0],
            &[0, 0, 85, 234, 255, 245, 125, 0, 0],
            &[0, 0, 0, 3, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '×' / 0xd7
        '×' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 122, 73, 0, 40, 153, 5, 0],
            &[0, 0, 152, 249, 112, 232, 191, 10, 0],
            &[0, 0, 1, 166, 255, 205, 10, 0, 0],
            &[0, 0, 39, 231, 230, 248, 71, 0, 0],
            &[0, 0, 200, 189, 11, 149, 237, 13, 0],
            &[0, 0, 31, 9, 0, 0, 40, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ø' / 0xd8
        'Ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 18, 130, 174, 148, 93, 184, 0, 0],
            &[4, 208, 255, 216, 255, 255, 143, 0, 0],
            &[82, 255, 136, 0, 230, 255, 138, 0, 0],
            &[139, 255, 60, 58, 251, 255, 192, 0, 0],
            &[158, 255, 48, 218, 111, 241, 211, 0, 0],
            &[152, 255, 194, 194, 2, 247, 203, 0, 0],
            &[113, 255, 255, 35, 40, 255, 164, 0, 0],
            &[42, 255, 255, 109, 199, 255, 75, 0, 0],
            &[145, 255, 236, 255, 245, 125, 0, 0, 0],
            &[40, 32, 3, 24, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ù' / 0xd9
        'Ù' => &[
            &[0, 0, 21, 53, 28, 0, 0, 0, 0],
            &[0, 0, 15, 199, 222, 11, 0, 0, 0],
            &[0, 0, 0, 11, 155, 90, 0, 0, 0],
            &[0, 66, 145, 40, 0, 11, 145, 95, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 72, 0, 21, 255, 168, 0],
            &[0, 107, 255, 97, 0, 45, 255, 158, 0],
            &[0, 47, 254, 221, 113, 194, 255, 93, 0],
            &[0, 0, 112, 241, 255, 248, 146, 1, 0],
            &[0, 0, 0, 4, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ú' / 0xda
        'Ú' => &[
            &[0, 0, 0, 0, 5, 53, 44, 0, 0],
            &[0, 0, 0, 0, 124, 247, 75, 0, 0],
            &[0, 0, 0, 18, 176, 62, 0, 0, 0],
            &[0, 66, 145, 40, 0, 11, 145, 95, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 72, 0, 21, 255, 168, 0],
            &[0, 107, 255, 97, 0, 45, 255, 158, 0],
            &[0, 47, 254, 221, 113, 194, 255, 93, 0],
            &[0, 0, 112, 241, 255, 248, 146, 1, 0],
            &[0, 0, 0, 4, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Û' / 0xdb
        'Û' => &[
            &[0, 0, 0, 18, 53, 29, 0, 0, 0],
            &[0, 0, 2, 190, 233, 228, 20, 0, 0],
            &[0, 0, 75, 153, 12, 126, 112, 0, 0],
            &[0, 66, 145, 40, 0, 11, 145, 95, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 72, 0, 21, 255, 168, 0],
            &[0, 107, 255, 97, 0, 45, 255, 158, 0],
            &[0, 47, 254, 221, 113, 194, 255, 93, 0],
            &[0, 0, 112, 241, 255, 248, 146, 1, 0],
            &[0, 0, 0, 4, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ü' / 0xdc
        'Ü' => &[
            &[0, 0, 0, 6, 0, 3, 3, 0, 0],
            &[0, 0, 136, 232, 2, 188, 185, 0, 0],
            &[0, 0, 65, 126, 0, 97, 95, 0, 0],
            &[0, 66, 145, 40, 0, 11, 145, 95, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 72, 0, 21, 255, 168, 0],
            &[0, 107, 255, 97, 0, 45, 255, 158, 0],
            &[0, 47, 254, 221, 113, 194, 255, 93, 0],
            &[0, 0, 112, 241, 255, 248, 146, 1, 0],
            &[0, 0, 0, 4, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ý' / 0xdd
        'Ý' => &[
            &[0, 0, 0, 0, 5, 53, 44, 0, 0],
            &[0, 0, 0, 0, 124, 247, 75, 0, 0],
            &[0, 0, 0, 18, 176, 62, 0, 0, 0],
            &[0, 90, 145, 28, 0, 3, 139, 119, 0],
            &[0, 64, 255, 134, 0, 79, 255, 116, 0],
            &[0, 0, 198, 235, 7, 187, 238, 12, 0],
            &[0, 0, 76, 255, 134, 254, 129, 0, 0],
            &[0, 0, 1, 209, 255, 244, 19, 0, 0],
            &[0, 0, 0, 95, 255, 149, 0, 0, 0],
            &[0, 0, 0, 67, 255, 121, 0, 0, 0],
            &[0, 0, 0, 67, 255, 121, 0, 0, 0],
            &[0, 0, 0, 67, 255, 121, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Þ' / 0xde
        'Þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 145, 93, 0, 0, 0, 0, 0],
            &[0, 10, 255, 175, 32, 9, 0, 0, 0],
            &[0, 10, 255, 255, 255, 254, 184, 15, 0],
            &[0, 10, 255, 195, 90, 155, 255, 132, 0],
            &[0, 10, 255, 163, 0, 13, 255, 175, 0],
            &[0, 10, 255, 171, 26, 115, 255, 147, 0],
            &[0, 10, 255, 255, 255, 255, 230, 34, 0],
            &[0, 10, 255, 199, 99, 77, 12, 0, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ß' / 0xdf
        'ß' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 15, 40, 13, 0, 0, 0],
            &[0, 0, 137, 251, 255, 251, 134, 0, 0],
            &[0, 49, 255, 190, 73, 203, 255, 27, 0],
            &[0, 85, 255, 95, 2, 188, 239, 10, 0],
            &[0, 87, 255, 90, 146, 247, 66, 0, 0],
            &[0, 87, 255, 90, 220, 231, 28, 0, 0],
            &[0, 87, 255, 90, 95, 252, 233, 48, 0],
            &[0, 87, 255, 90, 0, 64, 249, 188, 0],
            &[0, 87, 255, 105, 114, 80, 247, 202, 0],
            &[0, 87, 255, 106, 241, 255, 240, 78, 0],
            &[0, 0, 0, 0, 5, 24, 5, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'à' / 0xe0
        'à' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 49, 30, 0, 0, 0, 0],
            &[0, 0, 8, 185, 234, 21, 0, 0, 0],
            &[0, 0, 0, 6, 147, 109, 0, 0, 0],
            &[0, 0, 70, 142, 164, 140, 44, 0, 0],
            &[0, 0, 185, 218, 190, 239, 245, 34, 0],
            &[0, 0, 6, 0, 11, 89, 255, 97, 0],
            &[0, 3, 149, 241, 254, 250, 255, 107, 0],
            &[0, 76, 255, 139, 8, 69, 255, 107, 0],
            &[0, 85, 255, 170, 79, 197, 255, 107, 0],
            &[0, 10, 202, 255, 242, 122, 252, 107, 0],
            &[0, 0, 0, 20, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'á' / 0xe1
        'á' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 45, 0, 0],
            &[0, 0, 0, 0, 97, 253, 99, 0, 0],
            &[0, 0, 0, 8, 172, 82, 0, 0, 0],
            &[0, 0, 70, 142, 164, 140, 44, 0, 0],
            &[0, 0, 185, 218, 190, 239, 245, 34, 0],
            &[0, 0, 6, 0, 11, 89, 255, 97, 0],
            &[0, 3, 149, 241, 254, 250, 255, 107, 0],
            &[0, 76, 255, 139, 8, 69, 255, 107, 0],
            &[0, 85, 255, 170, 79, 197, 255, 107, 0],
            &[0, 10, 202, 255, 242, 122, 252, 107, 0],
            &[0, 0, 0, 20, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'â' / 0xe2
        'â' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 11, 49, 31, 0, 0, 0],
            &[0, 0, 0, 165, 235, 239, 31, 0, 0],
            &[0, 0, 57, 167, 20, 114, 132, 0, 0],
            &[0, 0, 70, 142, 164, 140, 44, 0, 0],
            &[0, 0, 185, 218, 190, 239, 245, 34, 0],
            &[0, 0, 6, 0, 11, 89, 255, 97, 0],
            &[0, 3, 149, 241, 254, 250, 255, 107, 0],
            &[0, 76, 255, 139, 8, 69, 255, 107, 0],
            &[0, 85, 255, 170, 79, 197, 255, 107, 0],
            &[0, 10, 202, 255, 242, 122, 252, 107, 0],
            &[0, 0, 0, 20, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ã' / 0xe3
        'ã' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 0, 0, 16, 0, 0],
            &[0, 0, 52, 244, 216, 136, 219, 0, 0],
            &[0, 0, 99, 98, 90, 172, 75, 0, 0],
            &[0, 0, 70, 142, 164, 140, 44, 0, 0],
            &[0, 0, 185, 218, 190, 239, 245, 34, 0],
            &[0, 0, 6, 0, 11, 89, 255, 97, 0],
            &[0, 3, 149, 241, 254, 250, 255, 107, 0],
            &[0, 76, 255, 139, 8, 69, 255, 107, 0],
            &[0, 85, 255, 170, 79, 197, 255, 107, 0],
            &[0, 10, 202, 255, 242, 122, 252, 107, 0],
            &[0, 0, 0, 20, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ä' / 0xe4
        'ä' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 0, 1, 4, 0, 0],
            &[0, 0, 110, 243, 14, 161, 208, 0, 0],
            &[0, 0, 53, 143, 1, 84, 115, 0, 0],
            &[0, 0, 70, 142, 164, 140, 44, 0, 0],
            &[0, 0, 185, 218, 190, 239, 245, 34, 0],
            &[0, 0, 6, 0, 11, 89, 255, 97, 0],
            &[0, 3, 149, 241, 254, 250, 255, 107, 0],
            &[0, 76, 255, 139, 8, 69, 255, 107, 0],
            &[0, 85, 255, 170, 79, 197, 255, 107, 0],
            &[0, 10, 202, 255, 242, 122, 252, 107, 0],
            &[0, 0, 0, 20, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'å' / 0xe5
        'å' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 0, 0, 0, 0],
            &[0, 0, 0, 103, 225, 186, 2, 0, 0],
            &[0, 0, 0, 177, 85, 223, 18, 0, 0],
            &[0, 0, 0, 52, 173, 110, 0, 0, 0],
            &[0, 0, 70, 142, 164, 140, 44, 0, 0],
            &[0, 0, 185, 218, 190, 239, 245, 34, 0],
            &[0, 0, 6, 0, 11, 89, 255, 97, 0],
            &[0, 3, 149, 241, 254, 250, 255, 107, 0],
            &[0, 76, 255, 139, 8, 69, 255, 107, 0],
            &[0, 85, 255, 170, 79, 197, 255, 107, 0],
            &[0, 10, 202, 255, 242, 122, 252, 107, 0],
            &[0, 0, 0, 20, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'æ' / 0xe6
        'æ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[41, 144, 158, 74, 132, 152, 29, 0, 0],
            &[60, 208, 223, 255, 243, 235, 189, 0, 0],
            &[0, 0, 41, 255, 117, 113, 252, 6, 0],
            &[72, 214, 252, 255, 251, 251, 255, 19, 0],
            &[204, 207, 45, 255, 126, 33, 33, 2, 0],
            &[212, 211, 124, 255, 213, 69, 83, 0, 0],
            &[113, 255, 234, 90, 227, 255, 180, 0, 0],
            &[0, 16, 7, 0, 3, 21, 1, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ç' / 0xe7
        'ç' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 83, 155, 163, 123, 15, 0],
            &[0, 0, 131, 255, 245, 215, 234, 7, 0],
            &[0, 6, 243, 228, 17, 0, 6, 0, 0],
            &[0, 33, 255, 167, 0, 0, 0, 0, 0],
            &[0, 24, 255, 191, 0, 0, 0, 0, 0],
            &[0, 0, 212, 255, 147, 102, 168, 8, 0],
            &[0, 0, 47, 210, 255, 255, 219, 6, 0],
            &[0, 0, 0, 0, 194, 165, 2, 0, 0],
            &[0, 0, 0, 53, 219, 176, 10, 0, 0],
        ],
        // letter: 'è' / 0xe8
        'è' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 49, 31, 0, 0, 0, 0],
            &[0, 0, 7, 181, 237, 24, 0, 0, 0],
            &[0, 0, 0, 5, 144, 113, 0, 0, 0],
            &[0, 0, 9, 114, 163, 142, 39, 0, 0],
            &[0, 1, 190, 252, 185, 229, 242, 32, 0],
            &[0, 57, 255, 129, 0, 31, 255, 124, 0],
            &[0, 95, 255, 249, 245, 245, 255, 151, 0],
            &[0, 77, 255, 143, 58, 58, 58, 35, 0],
            &[0, 11, 233, 236, 118, 87, 125, 72, 0],
            &[0, 0, 50, 203, 255, 255, 239, 75, 0],
            &[0, 0, 0, 0, 13, 22, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'é' / 0xe9
        'é' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 47, 47, 0, 0],
            &[0, 0, 0, 0, 91, 254, 104, 0, 0],
            &[0, 0, 0, 6, 169, 87, 0, 0, 0],
            &[0, 0, 9, 114, 163, 142, 39, 0, 0],
            &[0, 1, 190, 252, 185, 229, 242, 32, 0],
            &[0, 57, 255, 129, 0, 31, 255, 124, 0],
            &[0, 95, 255, 249, 245, 245, 255, 151, 0],
            &[0, 77, 255, 143, 58, 58, 58, 35, 0],
            &[0, 11, 233, 236, 118, 87, 125, 72, 0],
            &[0, 0, 50, 203, 255, 255, 239, 75, 0],
            &[0, 0, 0, 0, 13, 22, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ê' / 0xea
        'ê' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 49, 32, 0, 0, 0],
            &[0, 0, 0, 159, 235, 242, 34, 0, 0],
            &[0, 0, 53, 169, 23, 109, 136, 0, 0],
            &[0, 0, 9, 114, 163, 142, 39, 0, 0],
            &[0, 1, 190, 252, 185, 229, 242, 32, 0],
            &[0, 57, 255, 129, 0, 31, 255, 124, 0],
            &[0, 95, 255, 249, 245, 245, 255, 151, 0],
            &[0, 77, 255, 143, 58, 58, 58, 35, 0],
            &[0, 11, 233, 236, 118, 87, 125, 72, 0],
            &[0, 0, 50, 203, 255, 255, 239, 75, 0],
            &[0, 0, 0, 0, 13, 22, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ë' / 0xeb
        'ë' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 0, 0, 4, 0, 0],
            &[0, 0, 104, 245, 18, 155, 214, 0, 0],
            &[0, 0, 50, 145, 3, 80, 119, 0, 0],
            &[0, 0, 9, 114, 163, 142, 39, 0, 0],
            &[0, 1, 190, 252, 185, 229, 242, 32, 0],
            &[0, 57, 255, 129, 0, 31, 255, 124, 0],
            &[0, 95, 255, 249, 245, 245, 255, 151, 0],
            &[0, 77, 255, 143, 58, 58, 58, 35, 0],
            &[0, 11, 233, 236, 118, 87, 125, 72, 0],
            &[0, 0, 50, 203, 255, 255, 239, 75, 0],
            &[0, 0, 0, 0, 13, 22, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ì' / 0xec
        'ì' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 49, 25, 0, 0, 0, 0],
            &[0, 0, 16, 202, 220, 10, 0, 0, 0],
            &[0, 0, 0, 13, 159, 91, 0, 0, 0],
            &[0, 0, 135, 141, 141, 90, 0, 0, 0],
            &[0, 0, 180, 218, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 7, 53, 109, 255, 201, 72, 20, 0],
            &[0, 71, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'í' / 0xed
        'í' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 49, 41, 0, 0],
            &[0, 0, 0, 0, 122, 248, 78, 0, 0],
            &[0, 0, 0, 18, 179, 65, 0, 0, 0],
            &[0, 0, 135, 141, 141, 90, 0, 0, 0],
            &[0, 0, 180, 218, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 7, 53, 109, 255, 201, 72, 20, 0],
            &[0, 71, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'î' / 0xee
        'î' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 49, 27, 0, 0, 0],
            &[0, 0, 2, 187, 235, 227, 18, 0, 0],
            &[0, 0, 75, 157, 13, 130, 114, 0, 0],
            &[0, 0, 135, 141, 141, 90, 0, 0, 0],
            &[0, 0, 180, 218, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 7, 53, 109, 255, 201, 72, 20, 0],
            &[0, 71, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ï' / 0xef
        'ï' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 0, 2, 2, 0, 0],
            &[0, 0, 134, 230, 2, 186, 183, 0, 0],
            &[0, 0, 68, 130, 0, 100, 98, 0, 0],
            &[0, 0, 135, 141, 141, 90, 0, 0, 0],
            &[0, 0, 180, 218, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 7, 53, 109, 255, 201, 72, 20, 0],
            &[0, 71, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ð' / 0xf0
        'ð' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 10, 0, 0, 8, 0, 0],
            &[0, 0, 0, 209, 198, 188, 182, 0, 0],
            &[0, 0, 2, 152, 255, 255, 53, 0, 0],
            &[0, 0, 7, 169, 72, 213, 208, 3, 0],
            &[0, 0, 63, 180, 198, 180, 255, 82, 0],
            &[0, 35, 247, 223, 132, 207, 255, 153, 0],
            &[0, 114, 255, 86, 0, 32, 255, 179, 0],
            &[0, 122, 255, 74, 0, 28, 255, 162, 0],
            &[0, 57, 255, 185, 57, 156, 255, 81, 0],
            &[0, 0, 109, 239, 255, 245, 125, 0, 0],
            &[0, 0, 0, 4, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ñ' / 0xf1
        'ñ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 0, 0, 16, 0, 0],
            &[0, 0, 61, 247, 211, 138, 209, 0, 0],
            &[0, 0, 108, 91, 95, 172, 67, 0, 0],
            &[0, 48, 141, 28, 131, 161, 95, 0, 0],
            &[0, 87, 255, 235, 216, 242, 255, 82, 0],
            &[0, 87, 255, 178, 0, 71, 255, 144, 0],
            &[0, 87, 255, 111, 0, 36, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ò' / 0xf2
        'ò' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 49, 25, 0, 0, 0, 0],
            &[0, 0, 16, 202, 220, 10, 0, 0, 0],
            &[0, 0, 0, 13, 159, 91, 0, 0, 0],
            &[0, 0, 13, 119, 164, 137, 33, 0, 0],
            &[0, 6, 205, 253, 204, 247, 238, 31, 0],
            &[0, 83, 255, 132, 0, 80, 255, 137, 0],
            &[0, 125, 255, 70, 0, 21, 255, 177, 0],
            &[0, 113, 255, 93, 0, 41, 255, 163, 0],
            &[0, 38, 251, 218, 94, 186, 255, 84, 0],
            &[0, 0, 90, 235, 255, 246, 131, 0, 0],
            &[0, 0, 0, 3, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ó' / 0xf3
        'ó' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 49, 41, 0, 0],
            &[0, 0, 0, 0, 122, 248, 78, 0, 0],
            &[0, 0, 0, 18, 179, 65, 0, 0, 0],
            &[0, 0, 13, 119, 164, 137, 33, 0, 0],
            &[0, 6, 205, 253, 204, 247, 238, 31, 0],
            &[0, 83, 255, 132, 0, 80, 255, 137, 0],
            &[0, 125, 255, 70, 0, 21, 255, 177, 0],
            &[0, 113, 255, 93, 0, 41, 255, 163, 0],
            &[0, 38, 251, 218, 94, 186, 255, 84, 0],
            &[0, 0, 90, 235, 255, 246, 131, 0, 0],
            &[0, 0, 0, 3, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ô' / 0xf4
        'ô' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 49, 27, 0, 0, 0],
            &[0, 0, 2, 187, 235, 227, 18, 0, 0],
            &[0, 0, 75, 157, 13, 130, 114, 0, 0],
            &[0, 0, 13, 119, 164, 137, 33, 0, 0],
            &[0, 6, 205, 253, 204, 247, 238, 31, 0],
            &[0, 83, 255, 132, 0, 80, 255, 137, 0],
            &[0, 125, 255, 70, 0, 21, 255, 177, 0],
            &[0, 113, 255, 93, 0, 41, 255, 163, 0],
            &[0, 38, 251, 218, 94, 186, 255, 84, 0],
            &[0, 0, 90, 235, 255, 246, 131, 0, 0],
            &[0, 0, 0, 3, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'õ' / 0xf5
        'õ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 0, 0, 16, 0, 0],
            &[0, 0, 71, 250, 206, 143, 197, 0, 0],
            &[0, 0, 117, 84, 100, 173, 59, 0, 0],
            &[0, 0, 13, 119, 164, 137, 33, 0, 0],
            &[0, 6, 205, 253, 204, 247, 238, 31, 0],
            &[0, 83, 255, 132, 0, 80, 255, 137, 0],
            &[0, 125, 255, 70, 0, 21, 255, 177, 0],
            &[0, 113, 255, 93, 0, 41, 255, 163, 0],
            &[0, 38, 251, 218, 94, 186, 255, 84, 0],
            &[0, 0, 90, 235, 255, 246, 131, 0, 0],
            &[0, 0, 0, 3, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ö' / 0xf6
        'ö' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 0, 2, 2, 0, 0],
            &[0, 0, 134, 230, 2, 186, 183, 0, 0],
            &[0, 0, 68, 130, 0, 100, 98, 0, 0],
            &[0, 0, 13, 119, 164, 137, 33, 0, 0],
            &[0, 6, 205, 253, 204, 247, 238, 31, 0],
            &[0, 83, 255, 132, 0, 80, 255, 137, 0],
            &[0, 125, 255, 70, 0, 21, 255, 177, 0],
            &[0, 113, 255, 93, 0, 41, 255, 163, 0],
            &[0, 38, 251, 218, 94, 186, 255, 84, 0],
            &[0, 0, 90, 235, 255, 246, 131, 0, 0],
            &[0, 0, 0, 3, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: '÷' / 0xf7
        '÷' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 51, 6, 0, 0, 0],
            &[0, 0, 0, 30, 255, 84, 0, 0, 0],
            &[0, 0, 0, 1, 103, 18, 0, 0, 0],
            &[0, 35, 223, 223, 223, 223, 223, 80, 0],
            &[0, 16, 102, 102, 102, 102, 102, 36, 0],
            &[0, 0, 0, 20, 225, 63, 0, 0, 0],
            &[0, 0, 0, 12, 187, 46, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ø' / 0xf8
        'ø' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 13, 119, 164, 142, 200, 27, 0],
            &[0, 6, 205, 253, 204, 255, 255, 35, 0],
            &[0, 83, 255, 132, 49, 255, 255, 137, 0],
            &[0, 125, 255, 77, 210, 134, 255, 177, 0],
            &[0, 113, 255, 228, 193, 43, 255, 163, 0],
            &[0, 38, 255, 255, 128, 186, 255, 84, 0],
            &[0, 3, 255, 255, 255, 246, 131, 0, 0],
            &[0, 0, 53, 3, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ù' / 0xf9
        'ù' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 20, 49, 25, 0, 0, 0, 0],
            &[0, 0, 16, 202, 220, 10, 0, 0, 0],
            &[0, 0, 0, 13, 159, 91, 0, 0, 0],
            &[0, 56, 141, 48, 0, 28, 141, 76, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 53, 255, 138, 0],
            &[0, 100, 255, 94, 0, 82, 255, 138, 0],
            &[0, 70, 255, 197, 92, 209, 255, 138, 0],
            &[0, 2, 177, 255, 248, 127, 237, 138, 0],
            &[0, 0, 0, 16, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ú' / 0xfa
        'ú' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 49, 41, 0, 0],
            &[0, 0, 0, 0, 122, 248, 78, 0, 0],
            &[0, 0, 0, 18, 179, 65, 0, 0, 0],
            &[0, 56, 141, 48, 0, 28, 141, 76, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 53, 255, 138, 0],
            &[0, 100, 255, 94, 0, 82, 255, 138, 0],
            &[0, 70, 255, 197, 92, 209, 255, 138, 0],
            &[0, 2, 177, 255, 248, 127, 237, 138, 0],
            &[0, 0, 0, 16, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'û' / 0xfb
        'û' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 49, 27, 0, 0, 0],
            &[0, 0, 2, 187, 235, 227, 18, 0, 0],
            &[0, 0, 75, 157, 13, 130, 114, 0, 0],
            &[0, 56, 141, 48, 0, 28, 141, 76, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 53, 255, 138, 0],
            &[0, 100, 255, 94, 0, 82, 255, 138, 0],
            &[0, 70, 255, 197, 92, 209, 255, 138, 0],
            &[0, 2, 177, 255, 248, 127, 237, 138, 0],
            &[0, 0, 0, 16, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ü' / 0xfc
        'ü' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 0, 2, 2, 0, 0],
            &[0, 0, 134, 230, 2, 186, 183, 0, 0],
            &[0, 0, 68, 130, 0, 100, 98, 0, 0],
            &[0, 56, 141, 48, 0, 28, 141, 76, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 53, 255, 138, 0],
            &[0, 100, 255, 94, 0, 82, 255, 138, 0],
            &[0, 70, 255, 197, 92, 209, 255, 138, 0],
            &[0, 2, 177, 255, 248, 127, 237, 138, 0],
            &[0, 0, 0, 16, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ý' / 0xfd
        'ý' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 4, 49, 41, 0, 0],
            &[0, 0, 0, 0, 122, 248, 78, 0, 0],
            &[0, 0, 0, 18, 179, 65, 0, 0, 0],
            &[0, 91, 141, 32, 0, 2, 137, 121, 0],
            &[0, 87, 255, 124, 0, 57, 255, 147, 0],
            &[0, 6, 235, 210, 0, 133, 255, 54, 0],
            &[0, 0, 140, 255, 40, 208, 217, 0, 0],
            &[0, 0, 39, 255, 148, 255, 124, 0, 0],
            &[0, 0, 0, 193, 246, 254, 32, 0, 0],
            &[0, 0, 0, 92, 255, 194, 0, 0, 0],
            &[0, 0, 0, 103, 255, 98, 0, 0, 0],
            &[0, 64, 235, 194, 42, 0, 0, 0, 0],
        ],
        // letter: 'þ' / 0xfe
        'þ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 30, 11, 0, 0, 0, 0, 0],
            &[0, 87, 255, 93, 0, 0, 0, 0, 0],
            &[0, 87, 255, 93, 0, 0, 0, 0, 0],
            &[0, 87, 255, 105, 129, 162, 71, 0, 0],
            &[0, 87, 255, 241, 212, 245, 251, 51, 0],
            &[0, 87, 255, 155, 0, 79, 255, 144, 0],
            &[0, 87, 255, 98, 0, 26, 255, 176, 0],
            &[0, 87, 255, 120, 0, 43, 255, 166, 0],
            &[0, 87, 255, 229, 99, 179, 255, 102, 0],
            &[0, 87, 255, 175, 243, 255, 178, 6, 0],
            &[0, 87, 255, 92, 6, 19, 0, 0, 0],
            &[0, 76, 225, 81, 0, 0, 0, 0, 0],
        ],
        // letter: 'ÿ' / 0xff
        'ÿ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 5, 0, 2, 2, 0, 0],
            &[0, 0, 134, 230, 2, 186, 183, 0, 0],
            &[0, 0, 68, 130, 0, 100, 98, 0, 0],
            &[0, 91, 141, 32, 0, 2, 137, 121, 0],
            &[0, 87, 255, 124, 0, 57, 255, 147, 0],
            &[0, 6, 235, 210, 0, 133, 255, 54, 0],
            &[0, 0, 140, 255, 40, 208, 217, 0, 0],
            &[0, 0, 39, 255, 148, 255, 124, 0, 0],
            &[0, 0, 0, 193, 246, 254, 32, 0, 0],
            &[0, 0, 0, 92, 255, 194, 0, 0, 0],
            &[0, 0, 0, 103, 255, 98, 0, 0, 0],
            &[0, 64, 235, 194, 42, 0, 0, 0, 0],
        ],
        // letter: 'Ā' / 0x100
        'Ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 38, 170, 170, 170, 70, 0, 0, 0],
            &[0, 29, 133, 133, 133, 54, 0, 0, 0],
            &[0, 0, 92, 154, 121, 0, 0, 0, 0],
            &[0, 0, 209, 251, 248, 10, 0, 0, 0],
            &[0, 24, 255, 164, 255, 72, 0, 0, 0],
            &[0, 94, 255, 52, 246, 143, 0, 0, 0],
            &[0, 164, 237, 1, 190, 214, 0, 0, 0],
            &[1, 233, 255, 144, 255, 255, 29, 0, 0],
            &[49, 255, 255, 224, 255, 255, 99, 0, 0],
            &[119, 255, 47, 0, 10, 250, 170, 0, 0],
            &[190, 237, 1, 0, 0, 199, 238, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ā' / 0x101
        'ā' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 21, 166, 166, 166, 84, 0, 0],
            &[0, 0, 17, 137, 137, 137, 69, 0, 0],
            &[0, 0, 70, 142, 164, 140, 44, 0, 0],
            &[0, 0, 185, 218, 190, 239, 245, 34, 0],
            &[0, 0, 6, 0, 11, 89, 255, 97, 0],
            &[0, 3, 149, 241, 254, 250, 255, 107, 0],
            &[0, 76, 255, 139, 8, 69, 255, 107, 0],
            &[0, 85, 255, 170, 79, 197, 255, 107, 0],
            &[0, 10, 202, 255, 242, 122, 252, 107, 0],
            &[0, 0, 0, 20, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ă' / 0x102
        'Ă' => &[
            &[0, 24, 38, 0, 27, 36, 0, 0, 0],
            &[0, 59, 231, 122, 213, 107, 0, 0, 0],
            &[0, 0, 99, 171, 120, 4, 0, 0, 0],
            &[0, 0, 92, 154, 121, 0, 0, 0, 0],
            &[0, 0, 209, 251, 248, 10, 0, 0, 0],
            &[0, 24, 255, 164, 255, 72, 0, 0, 0],
            &[0, 94, 255, 52, 246, 143, 0, 0, 0],
            &[0, 164, 237, 1, 190, 214, 0, 0, 0],
            &[1, 233, 255, 144, 255, 255, 29, 0, 0],
            &[49, 255, 255, 224, 255, 255, 99, 0, 0],
            &[119, 255, 47, 0, 10, 250, 170, 0, 0],
            &[190, 237, 1, 0, 0, 199, 238, 2, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ă' / 0x103
        'ă' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 41, 0, 20, 40, 0, 0],
            &[0, 0, 38, 238, 121, 199, 133, 0, 0],
            &[0, 0, 0, 88, 174, 135, 10, 0, 0],
            &[0, 0, 70, 142, 164, 140, 44, 0, 0],
            &[0, 0, 185, 218, 190, 239, 245, 34, 0],
            &[0, 0, 6, 0, 11, 89, 255, 97, 0],
            &[0, 3, 149, 241, 254, 250, 255, 107, 0],
            &[0, 76, 255, 139, 8, 69, 255, 107, 0],
            &[0, 85, 255, 170, 79, 197, 255, 107, 0],
            &[0, 10, 202, 255, 242, 122, 252, 107, 0],
            &[0, 0, 0, 20, 6, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ą' / 0x104
        'Ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 92, 154, 121, 0, 0, 0, 0],
            &[0, 0, 209, 251, 248, 10, 0, 0, 0],
            &[0, 24, 255, 164, 255, 72, 0, 0, 0],
            &[0, 94, 255, 52, 246, 143, 0, 0, 0],
            &[0, 164, 237, 1, 190, 214, 0, 0, 0],
            &[1, 233, 255, 144, 255, 255, 29, 0, 0],
            &[49, 255, 255, 224, 255, 255, 99, 0, 0],
            &[119, 255, 47, 0, 10, 250, 170, 0, 0],
            &[190, 237, 1, 0, 0, 199, 255, 3, 0],
            &[0, 0, 0, 0, 0, 114, 172, 0, 0],
            &[0, 0, 0, 0, 0, 110, 208, 65, 0],
        ],
        // letter: 'ą' / 0x105
        'ą' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 70, 142, 164, 140, 44, 0, 0],
            &[0, 0, 185, 218, 190, 239, 245, 34, 0],
            &[0, 0, 6, 0, 11, 89, 255, 97, 0],
            &[0, 3, 149, 241, 254, 250, 255, 107, 0],
            &[0, 76, 255, 139, 8, 69, 255, 107, 0],
            &[0, 85, 255, 170, 79, 197, 255, 107, 0],
            &[0, 10, 202, 255, 242, 122, 255, 108, 0],
            &[0, 0, 0, 20, 6, 88, 194, 4, 0],
            &[0, 0, 0, 0, 0, 88, 207, 88, 0],
        ],
        // letter: 'Ć' / 0x106
        'Ć' => &[
            &[0, 0, 0, 0, 0, 24, 53, 25, 0],
            &[0, 0, 0, 0, 5, 208, 211, 22, 0],
            &[0, 0, 0, 0, 77, 162, 17, 0, 0],
            &[0, 0, 0, 41, 137, 169, 149, 59, 0],
            &[0, 0, 77, 250, 254, 225, 245, 112, 0],
            &[0, 6, 231, 233, 40, 0, 10, 10, 0],
            &[0, 63, 255, 138, 0, 0, 0, 0, 0],
            &[0, 95, 255, 101, 0, 0, 0, 0, 0],
            &[0, 89, 255, 106, 0, 0, 0, 0, 0],
            &[0, 47, 255, 172, 0, 0, 0, 0, 0],
            &[0, 0, 200, 255, 162, 110, 148, 80, 0],
            &[0, 0, 26, 186, 255, 255, 238, 72, 0],
            &[0, 0, 0, 0, 10, 22, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ć' / 0x107
        'ć' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 39, 49, 6, 0],
            &[0, 0, 0, 0, 49, 251, 148, 0, 0],
            &[0, 0, 0, 0, 142, 120, 0, 0, 0],
            &[0, 0, 0, 83, 155, 163, 123, 15, 0],
            &[0, 0, 131, 255, 245, 215, 234, 7, 0],
            &[0, 6, 243, 228, 17, 0, 6, 0, 0],
            &[0, 33, 255, 167, 0, 0, 0, 0, 0],
            &[0, 24, 255, 191, 0, 0, 0, 0, 0],
            &[0, 0, 212, 255, 147, 102, 168, 8, 0],
            &[0, 0, 47, 210, 255, 255, 219, 6, 0],
            &[0, 0, 0, 0, 17, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĉ' / 0x108
        'Ĉ' => &[
            &[0, 0, 0, 0, 36, 53, 10, 0, 0],
            &[0, 0, 0, 38, 244, 233, 158, 0, 0],
            &[0, 0, 0, 139, 100, 24, 167, 48, 0],
            &[0, 0, 0, 41, 137, 169, 149, 59, 0],
            &[0, 0, 77, 250, 254, 225, 245, 112, 0],
            &[0, 6, 231, 233, 40, 0, 10, 10, 0],
            &[0, 63, 255, 138, 0, 0, 0, 0, 0],
            &[0, 95, 255, 101, 0, 0, 0, 0, 0],
            &[0, 89, 255, 106, 0, 0, 0, 0, 0],
            &[0, 47, 255, 172, 0, 0, 0, 0, 0],
            &[0, 0, 200, 255, 162, 110, 148, 80, 0],
            &[0, 0, 26, 186, 255, 255, 238, 72, 0],
            &[0, 0, 0, 0, 10, 22, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĉ' / 0x109
        'ĉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 49, 41, 0, 0, 0],
            &[0, 0, 0, 113, 239, 250, 68, 0, 0],
            &[0, 0, 25, 175, 44, 76, 164, 5, 0],
            &[0, 0, 0, 83, 155, 163, 123, 15, 0],
            &[0, 0, 131, 255, 245, 215, 234, 7, 0],
            &[0, 6, 243, 228, 17, 0, 6, 0, 0],
            &[0, 33, 255, 167, 0, 0, 0, 0, 0],
            &[0, 24, 255, 191, 0, 0, 0, 0, 0],
            &[0, 0, 212, 255, 147, 102, 168, 8, 0],
            &[0, 0, 47, 210, 255, 255, 219, 6, 0],
            &[0, 0, 0, 0, 17, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ċ' / 0x10a
        'Ċ' => &[
            &[0, 0, 0, 0, 22, 58, 1, 0, 0],
            &[0, 0, 0, 0, 184, 255, 52, 0, 0],
            &[0, 0, 0, 0, 83, 148, 11, 0, 0],
            &[0, 0, 0, 41, 137, 169, 149, 59, 0],
            &[0, 0, 77, 250, 254, 225, 245, 112, 0],
            &[0, 6, 231, 233, 40, 0, 10, 10, 0],
            &[0, 63, 255, 138, 0, 0, 0, 0, 0],
            &[0, 95, 255, 101, 0, 0, 0, 0, 0],
            &[0, 89, 255, 106, 0, 0, 0, 0, 0],
            &[0, 47, 255, 172, 0, 0, 0, 0, 0],
            &[0, 0, 200, 255, 162, 110, 148, 80, 0],
            &[0, 0, 26, 186, 255, 255, 238, 72, 0],
            &[0, 0, 0, 0, 10, 22, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ċ' / 0x10b
        'ċ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 45, 30, 0, 0, 0],
            &[0, 0, 0, 19, 253, 218, 0, 0, 0],
            &[0, 0, 0, 1, 144, 105, 0, 0, 0],
            &[0, 0, 0, 83, 155, 163, 123, 15, 0],
            &[0, 0, 131, 255, 245, 215, 234, 7, 0],
            &[0, 6, 243, 228, 17, 0, 6, 0, 0],
            &[0, 33, 255, 167, 0, 0, 0, 0, 0],
            &[0, 24, 255, 191, 0, 0, 0, 0, 0],
            &[0, 0, 212, 255, 147, 102, 168, 8, 0],
            &[0, 0, 47, 210, 255, 255, 219, 6, 0],
            &[0, 0, 0, 0, 17, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Č' / 0x10c
        'Č' => &[
            &[0, 0, 0, 50, 13, 0, 40, 23, 0],
            &[0, 0, 0, 125, 213, 132, 228, 24, 0],
            &[0, 0, 0, 1, 154, 182, 65, 0, 0],
            &[0, 0, 0, 41, 137, 169, 149, 59, 0],
            &[0, 0, 77, 250, 254, 225, 245, 112, 0],
            &[0, 6, 231, 233, 40, 0, 10, 10, 0],
            &[0, 63, 255, 138, 0, 0, 0, 0, 0],
            &[0, 95, 255, 101, 0, 0, 0, 0, 0],
            &[0, 89, 255, 106, 0, 0, 0, 0, 0],
            &[0, 47, 255, 172, 0, 0, 0, 0, 0],
            &[0, 0, 200, 255, 162, 110, 148, 80, 0],
            &[0, 0, 26, 186, 255, 255, 238, 72, 0],
            &[0, 0, 0, 0, 10, 22, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'č' / 0x10d
        'č' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 14, 43, 0, 4, 49, 4, 0],
            &[0, 0, 10, 207, 148, 190, 168, 0, 0],
            &[0, 0, 0, 39, 186, 176, 13, 0, 0],
            &[0, 0, 0, 83, 155, 163, 123, 15, 0],
            &[0, 0, 131, 255, 245, 215, 234, 7, 0],
            &[0, 6, 243, 228, 17, 0, 6, 0, 0],
            &[0, 33, 255, 167, 0, 0, 0, 0, 0],
            &[0, 24, 255, 191, 0, 0, 0, 0, 0],
            &[0, 0, 212, 255, 147, 102, 168, 8, 0],
            &[0, 0, 47, 210, 255, 255, 219, 6, 0],
            &[0, 0, 0, 0, 17, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ď' / 0x10e
        'Ď' => &[
            &[0, 0, 41, 21, 0, 31, 31, 0, 0],
            &[0, 0, 85, 232, 119, 239, 48, 0, 0],
            &[0, 0, 0, 127, 182, 94, 0, 0, 0],
            &[0, 66, 145, 145, 135, 85, 5, 0, 0],
            &[0, 117, 255, 238, 244, 255, 193, 6, 0],
            &[0, 117, 255, 65, 3, 139, 255, 110, 0],
            &[0, 117, 255, 65, 0, 15, 254, 185, 0],
            &[0, 117, 255, 65, 0, 0, 241, 208, 0],
            &[0, 117, 255, 65, 0, 4, 253, 195, 0],
            &[0, 117, 255, 65, 0, 79, 255, 139, 0],
            &[0, 117, 255, 156, 148, 243, 241, 29, 0],
            &[0, 117, 255, 255, 242, 185, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ď' / 0x10f
        'ď' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 28, 13, 30, 4],
            &[0, 0, 0, 0, 195, 240, 138, 228, 7],
            &[0, 0, 0, 0, 195, 240, 149, 89, 0],
            &[0, 73, 161, 111, 186, 240, 0, 0, 0],
            &[45, 251, 247, 225, 243, 240, 0, 0, 0],
            &[130, 255, 81, 2, 217, 240, 0, 0, 0],
            &[158, 255, 37, 0, 169, 240, 0, 0, 0],
            &[144, 255, 53, 0, 179, 240, 0, 0, 0],
            &[89, 255, 180, 109, 248, 240, 0, 0, 0],
            &[6, 186, 255, 195, 164, 240, 0, 0, 0],
            &[0, 0, 19, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Đ' / 0x110
        'Đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 145, 145, 135, 85, 5, 0, 0],
            &[0, 117, 255, 238, 244, 255, 193, 6, 0],
            &[0, 117, 255, 65, 3, 139, 255, 110, 0],
            &[0, 117, 255, 65, 0, 15, 254, 185, 0],
            &[0, 255, 255, 255, 129, 0, 241, 208, 0],
            &[0, 241, 255, 189, 65, 4, 253, 195, 0],
            &[0, 117, 255, 65, 0, 79, 255, 139, 0],
            &[0, 117, 255, 156, 148, 243, 241, 29, 0],
            &[0, 117, 255, 255, 242, 185, 47, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'đ' / 0x111
        'đ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 5, 30, 16, 0, 0],
            &[0, 0, 0, 35, 82, 255, 179, 8, 0],
            &[0, 0, 0, 215, 255, 255, 255, 49, 0],
            &[0, 25, 119, 123, 61, 255, 138, 0, 0],
            &[11, 225, 255, 242, 233, 255, 138, 0, 0],
            &[91, 255, 153, 0, 92, 255, 138, 0, 0],
            &[126, 255, 72, 0, 17, 255, 138, 0, 0],
            &[112, 255, 83, 0, 26, 255, 138, 0, 0],
            &[52, 255, 209, 93, 175, 255, 138, 0, 0],
            &[0, 141, 254, 249, 124, 241, 138, 0, 0],
            &[0, 0, 16, 10, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ē' / 0x112
        'Ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 25, 170, 170, 170, 82, 0, 0],
            &[0, 0, 20, 133, 133, 133, 64, 0, 0],
            &[0, 5, 145, 145, 145, 145, 145, 52, 0],
            &[0, 10, 255, 249, 238, 238, 238, 85, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 181, 50, 50, 50, 9, 0],
            &[0, 10, 255, 255, 255, 255, 255, 46, 0],
            &[0, 10, 255, 191, 77, 77, 77, 14, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 209, 127, 127, 127, 46, 0],
            &[0, 10, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ē' / 0x113
        'ē' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 17, 166, 166, 166, 88, 0, 0],
            &[0, 0, 14, 137, 137, 137, 72, 0, 0],
            &[0, 0, 9, 114, 163, 142, 39, 0, 0],
            &[0, 1, 190, 252, 185, 229, 242, 32, 0],
            &[0, 57, 255, 129, 0, 31, 255, 124, 0],
            &[0, 95, 255, 249, 245, 245, 255, 151, 0],
            &[0, 77, 255, 143, 58, 58, 58, 35, 0],
            &[0, 11, 233, 236, 118, 87, 125, 72, 0],
            &[0, 0, 50, 203, 255, 255, 239, 75, 0],
            &[0, 0, 0, 0, 13, 22, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĕ' / 0x114
        'Ĕ' => &[
            &[0, 0, 20, 43, 0, 23, 41, 0, 0],
            &[0, 0, 42, 237, 124, 204, 125, 0, 0],
            &[0, 0, 0, 88, 170, 128, 8, 0, 0],
            &[0, 5, 145, 145, 145, 145, 145, 52, 0],
            &[0, 10, 255, 249, 238, 238, 238, 85, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 181, 50, 50, 50, 9, 0],
            &[0, 10, 255, 255, 255, 255, 255, 46, 0],
            &[0, 10, 255, 191, 77, 77, 77, 14, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 209, 127, 127, 127, 46, 0],
            &[0, 10, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĕ' / 0x115
        'ĕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 16, 43, 0, 18, 41, 0, 0],
            &[0, 0, 33, 239, 122, 196, 139, 0, 0],
            &[0, 0, 0, 84, 174, 137, 12, 0, 0],
            &[0, 0, 9, 114, 163, 142, 39, 0, 0],
            &[0, 1, 190, 252, 185, 229, 242, 32, 0],
            &[0, 57, 255, 129, 0, 31, 255, 124, 0],
            &[0, 95, 255, 249, 245, 245, 255, 151, 0],
            &[0, 77, 255, 143, 58, 58, 58, 35, 0],
            &[0, 11, 233, 236, 118, 87, 125, 72, 0],
            &[0, 0, 50, 203, 255, 255, 239, 75, 0],
            &[0, 0, 0, 0, 13, 22, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ė' / 0x116
        'Ė' => &[
            &[0, 0, 0, 3, 61, 18, 0, 0, 0],
            &[0, 0, 0, 75, 255, 160, 0, 0, 0],
            &[0, 0, 0, 23, 153, 66, 0, 0, 0],
            &[0, 5, 145, 145, 145, 145, 145, 52, 0],
            &[0, 10, 255, 249, 238, 238, 238, 85, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 181, 50, 50, 50, 9, 0],
            &[0, 10, 255, 255, 255, 255, 255, 46, 0],
            &[0, 10, 255, 191, 77, 77, 77, 14, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 209, 127, 127, 127, 46, 0],
            &[0, 10, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ė' / 0x117
        'ė' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 1, 55, 18, 0, 0, 0],
            &[0, 0, 0, 63, 255, 172, 0, 0, 0],
            &[0, 0, 0, 19, 156, 76, 0, 0, 0],
            &[0, 0, 9, 114, 163, 142, 39, 0, 0],
            &[0, 1, 190, 252, 185, 229, 242, 32, 0],
            &[0, 57, 255, 129, 0, 31, 255, 124, 0],
            &[0, 95, 255, 249, 245, 245, 255, 151, 0],
            &[0, 77, 255, 143, 58, 58, 58, 35, 0],
            &[0, 11, 233, 236, 118, 87, 125, 72, 0],
            &[0, 0, 50, 203, 255, 255, 239, 75, 0],
            &[0, 0, 0, 0, 13, 22, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ę' / 0x118
        'Ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 5, 145, 145, 145, 145, 145, 52, 0],
            &[0, 10, 255, 249, 238, 238, 238, 85, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 181, 50, 50, 50, 9, 0],
            &[0, 10, 255, 255, 255, 255, 255, 46, 0],
            &[0, 10, 255, 191, 77, 77, 77, 14, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 209, 127, 127, 127, 46, 0],
            &[0, 10, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 4, 197, 84, 0, 0],
            &[0, 0, 0, 0, 8, 172, 204, 0, 0],
        ],
        // letter: 'ę' / 0x119
        'ę' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 9, 114, 163, 142, 39, 0, 0],
            &[0, 1, 190, 252, 185, 229, 242, 32, 0],
            &[0, 57, 255, 129, 0, 31, 255, 124, 0],
            &[0, 95, 255, 249, 245, 245, 255, 151, 0],
            &[0, 77, 255, 143, 58, 58, 58, 35, 0],
            &[0, 11, 233, 236, 118, 87, 125, 72, 0],
            &[0, 0, 50, 203, 255, 255, 255, 99, 0],
            &[0, 0, 0, 0, 13, 182, 133, 0, 0],
            &[0, 0, 0, 0, 0, 127, 208, 45, 0],
        ],
        // letter: 'Ě' / 0x11a
        'Ě' => &[
            &[0, 0, 27, 35, 0, 17, 45, 0, 0],
            &[0, 0, 34, 234, 122, 226, 107, 0, 0],
            &[0, 0, 0, 79, 182, 142, 0, 0, 0],
            &[0, 5, 145, 145, 145, 145, 145, 52, 0],
            &[0, 10, 255, 249, 238, 238, 238, 85, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 181, 50, 50, 50, 9, 0],
            &[0, 10, 255, 255, 255, 255, 255, 46, 0],
            &[0, 10, 255, 191, 77, 77, 77, 14, 0],
            &[0, 10, 255, 163, 0, 0, 0, 0, 0],
            &[0, 10, 255, 209, 127, 127, 127, 46, 0],
            &[0, 10, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ě' / 0x11b
        'ě' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 34, 0, 13, 44, 0, 0],
            &[0, 0, 29, 231, 123, 218, 122, 0, 0],
            &[0, 0, 0, 73, 186, 154, 1, 0, 0],
            &[0, 0, 9, 114, 163, 142, 39, 0, 0],
            &[0, 1, 190, 252, 185, 229, 242, 32, 0],
            &[0, 57, 255, 129, 0, 31, 255, 124, 0],
            &[0, 95, 255, 249, 245, 245, 255, 151, 0],
            &[0, 77, 255, 143, 58, 58, 58, 35, 0],
            &[0, 11, 233, 236, 118, 87, 125, 72, 0],
            &[0, 0, 50, 203, 255, 255, 239, 75, 0],
            &[0, 0, 0, 0, 13, 22, 2, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĝ' / 0x11c
        'Ĝ' => &[
            &[0, 0, 0, 1, 52, 46, 0, 0, 0],
            &[0, 0, 0, 110, 239, 248, 76, 0, 0],
            &[0, 0, 22, 171, 45, 68, 164, 7, 0],
            &[0, 0, 0, 77, 153, 166, 131, 38, 0],
            &[0, 0, 144, 255, 243, 215, 246, 84, 0],
            &[0, 51, 255, 196, 13, 0, 14, 5, 0],
            &[0, 127, 255, 69, 0, 0, 0, 0, 0],
            &[0, 156, 255, 31, 21, 210, 210, 139, 0],
            &[0, 149, 255, 41, 15, 157, 247, 168, 0],
            &[0, 103, 255, 112, 0, 0, 233, 168, 0],
            &[0, 17, 234, 245, 127, 100, 243, 168, 0],
            &[0, 0, 49, 203, 255, 255, 244, 123, 0],
            &[0, 0, 0, 0, 12, 22, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĝ' / 0x11d
        'ĝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 49, 27, 0, 0, 0],
            &[0, 0, 2, 187, 235, 227, 18, 0, 0],
            &[0, 0, 75, 157, 13, 130, 114, 0, 0],
            &[0, 0, 41, 149, 151, 28, 126, 76, 0],
            &[0, 13, 236, 253, 211, 229, 255, 138, 0],
            &[0, 93, 255, 129, 0, 88, 255, 138, 0],
            &[0, 124, 255, 69, 0, 32, 255, 138, 0],
            &[0, 110, 255, 89, 0, 41, 255, 138, 0],
            &[0, 48, 255, 214, 95, 183, 255, 138, 0],
            &[0, 0, 131, 251, 249, 169, 255, 138, 0],
            &[0, 1, 1, 9, 7, 77, 255, 124, 0],
            &[0, 9, 181, 228, 232, 193, 75, 0, 0],
        ],
        // letter: 'Ğ' / 0x11e
        'Ğ' => &[
            &[0, 0, 4, 59, 0, 7, 57, 0, 0],
            &[0, 0, 1, 231, 141, 171, 190, 0, 0],
            &[0, 0, 0, 50, 164, 151, 28, 0, 0],
            &[0, 0, 0, 77, 153, 166, 131, 38, 0],
            &[0, 0, 144, 255, 243, 215, 246, 84, 0],
            &[0, 51, 255, 196, 13, 0, 14, 5, 0],
            &[0, 127, 255, 69, 0, 0, 0, 0, 0],
            &[0, 156, 255, 31, 21, 210, 210, 139, 0],
            &[0, 149, 255, 41, 15, 157, 247, 168, 0],
            &[0, 103, 255, 112, 0, 0, 233, 168, 0],
            &[0, 17, 234, 245, 127, 100, 243, 168, 0],
            &[0, 0, 49, 203, 255, 255, 244, 123, 0],
            &[0, 0, 0, 0, 12, 22, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ğ' / 0x11f
        'ğ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 36, 0, 25, 34, 0, 0],
            &[0, 0, 61, 230, 118, 211, 108, 0, 0],
            &[0, 0, 0, 103, 175, 125, 4, 0, 0],
            &[0, 0, 41, 149, 151, 28, 126, 76, 0],
            &[0, 13, 236, 253, 211, 229, 255, 138, 0],
            &[0, 93, 255, 129, 0, 88, 255, 138, 0],
            &[0, 124, 255, 69, 0, 32, 255, 138, 0],
            &[0, 110, 255, 89, 0, 41, 255, 138, 0],
            &[0, 48, 255, 214, 95, 183, 255, 138, 0],
            &[0, 0, 131, 251, 249, 169, 255, 138, 0],
            &[0, 1, 1, 9, 7, 77, 255, 124, 0],
            &[0, 9, 181, 228, 232, 193, 75, 0, 0],
        ],
        // letter: 'Ġ' / 0x120
        'Ġ' => &[
            &[0, 0, 0, 0, 47, 35, 0, 0, 0],
            &[0, 0, 0, 14, 252, 225, 0, 0, 0],
            &[0, 0, 0, 0, 137, 106, 0, 0, 0],
            &[0, 0, 0, 77, 153, 166, 131, 38, 0],
            &[0, 0, 144, 255, 243, 215, 246, 84, 0],
            &[0, 51, 255, 196, 13, 0, 14, 5, 0],
            &[0, 127, 255, 69, 0, 0, 0, 0, 0],
            &[0, 156, 255, 31, 21, 210, 210, 139, 0],
            &[0, 149, 255, 41, 15, 157, 247, 168, 0],
            &[0, 103, 255, 112, 0, 0, 233, 168, 0],
            &[0, 17, 234, 245, 127, 100, 243, 168, 0],
            &[0, 0, 49, 203, 255, 255, 244, 123, 0],
            &[0, 0, 0, 0, 12, 22, 3, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ġ' / 0x121
        'ġ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 58, 12, 0, 0, 0],
            &[0, 0, 0, 93, 255, 142, 0, 0, 0],
            &[0, 0, 0, 34, 158, 58, 0, 0, 0],
            &[0, 0, 41, 149, 151, 28, 126, 76, 0],
            &[0, 13, 236, 253, 211, 229, 255, 138, 0],
            &[0, 93, 255, 129, 0, 88, 255, 138, 0],
            &[0, 124, 255, 69, 0, 32, 255, 138, 0],
            &[0, 110, 255, 89, 0, 41, 255, 138, 0],
            &[0, 48, 255, 214, 95, 183, 255, 138, 0],
            &[0, 0, 131, 251, 249, 169, 255, 138, 0],
            &[0, 1, 1, 9, 7, 77, 255, 124, 0],
            &[0, 9, 181, 228, 232, 193, 75, 0, 0],
        ],
        // letter: 'Ģ' / 0x122
        'Ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 77, 153, 166, 131, 38, 0],
            &[0, 0, 144, 255, 243, 215, 246, 84, 0],
            &[0, 51, 255, 196, 13, 0, 14, 5, 0],
            &[0, 127, 255, 69, 0, 0, 0, 0, 0],
            &[0, 156, 255, 31, 21, 210, 210, 139, 0],
            &[0, 149, 255, 41, 15, 157, 247, 168, 0],
            &[0, 103, 255, 112, 0, 0, 233, 168, 0],
            &[0, 17, 234, 245, 127, 100, 243, 168, 0],
            &[0, 0, 49, 203, 255, 255, 244, 123, 0],
            &[0, 0, 0, 0, 71, 73, 3, 0, 0],
            &[0, 0, 0, 25, 157, 7, 0, 0, 0],
        ],
        // letter: 'ģ' / 0x123
        'ģ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 23, 28, 0, 0, 0],
            &[0, 0, 0, 2, 208, 115, 0, 0, 0],
            &[0, 0, 0, 49, 186, 50, 0, 0, 0],
            &[0, 0, 41, 149, 151, 28, 126, 76, 0],
            &[0, 13, 236, 253, 211, 229, 255, 138, 0],
            &[0, 93, 255, 129, 0, 88, 255, 138, 0],
            &[0, 124, 255, 69, 0, 32, 255, 138, 0],
            &[0, 110, 255, 89, 0, 41, 255, 138, 0],
            &[0, 48, 255, 214, 95, 183, 255, 138, 0],
            &[0, 0, 131, 251, 249, 169, 255, 138, 0],
            &[0, 1, 1, 9, 7, 77, 255, 124, 0],
            &[0, 9, 181, 228, 232, 193, 75, 0, 0],
        ],
        // letter: 'Ĥ' / 0x124
        'Ĥ' => &[
            &[0, 0, 0, 18, 53, 29, 0, 0, 0],
            &[0, 0, 2, 190, 233, 228, 20, 0, 0],
            &[0, 0, 75, 153, 12, 126, 112, 0, 0],
            &[0, 66, 145, 40, 0, 11, 145, 95, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 112, 56, 72, 255, 168, 0],
            &[0, 117, 255, 255, 255, 255, 255, 168, 0],
            &[0, 117, 255, 123, 71, 86, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĥ' / 0x125
        'ĥ' => &[
            &[0, 111, 194, 133, 0, 0, 0, 0, 0],
            &[63, 236, 102, 231, 87, 0, 0, 0, 0],
            &[26, 31, 30, 29, 30, 0, 0, 0, 0],
            &[0, 87, 255, 102, 0, 0, 0, 0, 0],
            &[0, 87, 255, 102, 0, 0, 0, 0, 0],
            &[0, 87, 255, 111, 129, 160, 93, 0, 0],
            &[0, 87, 255, 240, 218, 241, 255, 80, 0],
            &[0, 87, 255, 176, 0, 68, 255, 143, 0],
            &[0, 87, 255, 114, 0, 36, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ħ' / 0x126
        'Ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[45, 145, 61, 0, 30, 145, 76, 0, 0],
            &[184, 255, 211, 103, 157, 255, 238, 20, 0],
            &[255, 255, 255, 188, 242, 255, 255, 37, 0],
            &[80, 255, 140, 56, 98, 255, 135, 0, 0],
            &[80, 255, 255, 255, 255, 255, 135, 0, 0],
            &[80, 255, 149, 71, 110, 255, 135, 0, 0],
            &[80, 255, 108, 0, 54, 255, 135, 0, 0],
            &[80, 255, 108, 0, 54, 255, 135, 0, 0],
            &[80, 255, 108, 0, 54, 255, 135, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ħ' / 0x127
        'ħ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 10, 30, 12, 0, 0, 0, 0, 0],
            &[0, 127, 255, 143, 40, 7, 0, 0, 0],
            &[0, 255, 255, 255, 247, 43, 0, 0, 0],
            &[0, 87, 255, 105, 99, 130, 63, 0, 0],
            &[0, 87, 255, 228, 244, 252, 255, 66, 0],
            &[0, 87, 255, 191, 4, 83, 255, 140, 0],
            &[0, 87, 255, 118, 0, 37, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĩ' / 0x128
        'Ĩ' => &[
            &[0, 0, 7, 0, 0, 19, 0, 0, 0],
            &[0, 73, 251, 210, 147, 196, 0, 0, 0],
            &[0, 115, 80, 96, 169, 57, 0, 0, 0],
            &[0, 133, 145, 145, 145, 145, 17, 0, 0],
            &[0, 161, 213, 255, 224, 179, 19, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 59, 131, 255, 160, 69, 6, 0, 0],
            &[0, 235, 255, 255, 255, 255, 30, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĩ' / 0x129
        'ĩ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 0, 0, 16, 0, 0],
            &[0, 0, 71, 250, 206, 143, 197, 0, 0],
            &[0, 0, 117, 84, 100, 173, 59, 0, 0],
            &[0, 0, 135, 141, 141, 90, 0, 0, 0],
            &[0, 0, 180, 218, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 7, 53, 109, 255, 201, 72, 20, 0],
            &[0, 71, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ī' / 0x12a
        'Ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 38, 170, 170, 170, 70, 0, 0, 0],
            &[0, 29, 133, 133, 133, 54, 0, 0, 0],
            &[0, 133, 145, 145, 145, 145, 17, 0, 0],
            &[0, 161, 213, 255, 224, 179, 19, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 59, 131, 255, 160, 69, 6, 0, 0],
            &[0, 235, 255, 255, 255, 255, 30, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ī' / 0x12b
        'ī' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 37, 166, 166, 166, 68, 0, 0],
            &[0, 0, 30, 137, 137, 137, 56, 0, 0],
            &[0, 0, 135, 141, 141, 90, 0, 0, 0],
            &[0, 0, 180, 218, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 7, 53, 109, 255, 201, 72, 20, 0],
            &[0, 71, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĭ' / 0x12c
        'Ĭ' => &[
            &[0, 24, 38, 0, 27, 36, 0, 0, 0],
            &[0, 59, 231, 122, 213, 107, 0, 0, 0],
            &[0, 0, 99, 171, 120, 4, 0, 0, 0],
            &[0, 133, 145, 145, 145, 145, 17, 0, 0],
            &[0, 161, 213, 255, 224, 179, 19, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 59, 131, 255, 160, 69, 6, 0, 0],
            &[0, 235, 255, 255, 255, 255, 30, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĭ' / 0x12d
        'ĭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 36, 0, 25, 34, 0, 0],
            &[0, 0, 61, 230, 118, 211, 108, 0, 0],
            &[0, 0, 0, 103, 175, 125, 4, 0, 0],
            &[0, 0, 135, 141, 141, 90, 0, 0, 0],
            &[0, 0, 180, 218, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 7, 53, 109, 255, 201, 72, 20, 0],
            &[0, 71, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Į' / 0x12e
        'Į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 133, 145, 145, 145, 145, 17, 0, 0],
            &[0, 161, 213, 255, 224, 179, 19, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 59, 131, 255, 160, 69, 6, 0, 0],
            &[0, 235, 255, 255, 255, 255, 30, 0, 0],
            &[0, 0, 0, 101, 184, 2, 0, 0, 0],
            &[0, 0, 0, 99, 207, 77, 0, 0, 0],
        ],
        // letter: 'į' / 0x12f
        'į' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 51, 23, 0, 0, 0],
            &[0, 0, 0, 41, 255, 194, 0, 0, 0],
            &[0, 0, 0, 9, 152, 89, 0, 0, 0],
            &[0, 0, 135, 141, 141, 90, 0, 0, 0],
            &[0, 0, 180, 218, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 7, 53, 109, 255, 201, 72, 20, 0],
            &[0, 71, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 4, 197, 84, 0, 0],
            &[0, 0, 0, 0, 8, 172, 204, 0, 0],
        ],
        // letter: 'İ' / 0x130
        'İ' => &[
            &[0, 0, 5, 62, 14, 0, 0, 0, 0],
            &[0, 0, 94, 255, 142, 0, 0, 0, 0],
            &[0, 0, 32, 154, 56, 0, 0, 0, 0],
            &[0, 133, 145, 145, 145, 145, 17, 0, 0],
            &[0, 161, 213, 255, 224, 179, 19, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 0, 64, 255, 106, 0, 0, 0, 0],
            &[0, 59, 131, 255, 160, 69, 6, 0, 0],
            &[0, 235, 255, 255, 255, 255, 30, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ı' / 0x131
        'ı' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 135, 141, 141, 90, 0, 0, 0],
            &[0, 0, 180, 218, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 0, 0, 24, 255, 164, 0, 0, 0],
            &[0, 7, 53, 109, 255, 201, 72, 20, 0],
            &[0, 71, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĳ' / 0x132
        'Ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 142, 100, 0, 76, 145, 26, 0, 0],
            &[0, 250, 176, 0, 134, 255, 46, 0, 0],
            &[0, 250, 176, 0, 134, 255, 46, 0, 0],
            &[0, 250, 176, 0, 134, 255, 46, 0, 0],
            &[0, 250, 176, 0, 134, 255, 46, 0, 0],
            &[0, 250, 176, 0, 134, 255, 46, 0, 0],
            &[0, 102, 71, 0, 155, 255, 35, 0, 0],
            &[0, 133, 107, 131, 247, 231, 3, 0, 0],
            &[0, 237, 255, 255, 224, 66, 0, 0, 0],
            &[0, 2, 23, 20, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĳ' / 0x133
        'ĳ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 40, 35, 0, 20, 54, 0, 0],
            &[0, 7, 248, 235, 0, 184, 255, 50, 0],
            &[0, 0, 134, 117, 0, 86, 152, 12, 0],
            &[0, 0, 131, 114, 0, 89, 141, 15, 0],
            &[0, 0, 238, 206, 0, 161, 255, 27, 0],
            &[0, 0, 238, 206, 0, 161, 255, 27, 0],
            &[0, 0, 238, 206, 0, 161, 255, 27, 0],
            &[0, 0, 238, 206, 0, 161, 255, 27, 0],
            &[0, 0, 238, 206, 0, 161, 255, 27, 0],
            &[0, 0, 238, 206, 0, 161, 255, 27, 0],
            &[0, 0, 0, 0, 0, 188, 255, 16, 0],
            &[0, 0, 105, 223, 231, 187, 48, 0, 0],
        ],
        // letter: 'Ĵ' / 0x134
        'Ĵ' => &[
            &[0, 0, 0, 0, 50, 50, 0, 0, 0],
            &[0, 0, 0, 92, 243, 245, 93, 0, 0],
            &[0, 0, 14, 169, 55, 57, 169, 13, 0],
            &[0, 0, 0, 0, 119, 127, 0, 0, 0],
            &[0, 0, 0, 0, 210, 225, 0, 0, 0],
            &[0, 0, 0, 0, 210, 225, 0, 0, 0],
            &[0, 0, 0, 0, 210, 225, 0, 0, 0],
            &[0, 0, 0, 0, 210, 225, 0, 0, 0],
            &[0, 0, 0, 0, 210, 225, 0, 0, 0],
            &[0, 0, 0, 0, 230, 215, 0, 0, 0],
            &[0, 134, 107, 159, 255, 159, 0, 0, 0],
            &[0, 240, 255, 255, 194, 25, 0, 0, 0],
            &[0, 3, 24, 14, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĵ' / 0x135
        'ĵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 39, 49, 3, 0, 0],
            &[0, 0, 0, 58, 249, 238, 125, 0, 0],
            &[0, 0, 3, 160, 82, 40, 175, 29, 0],
            &[0, 0, 63, 141, 141, 141, 20, 0, 0],
            &[0, 0, 81, 201, 241, 255, 37, 0, 0],
            &[0, 0, 0, 0, 151, 255, 37, 0, 0],
            &[0, 0, 0, 0, 151, 255, 37, 0, 0],
            &[0, 0, 0, 0, 151, 255, 37, 0, 0],
            &[0, 0, 0, 0, 151, 255, 37, 0, 0],
            &[0, 0, 0, 0, 151, 255, 37, 0, 0],
            &[0, 0, 0, 0, 178, 255, 26, 0, 0],
            &[0, 97, 221, 232, 191, 53, 0, 0, 0],
        ],
        // letter: 'Ķ' / 0x136
        'Ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 145, 35, 0, 44, 145, 86, 0],
            &[0, 117, 255, 62, 4, 206, 238, 27, 0],
            &[0, 117, 255, 62, 126, 255, 85, 0, 0],
            &[0, 117, 255, 107, 250, 160, 0, 0, 0],
            &[0, 117, 255, 235, 255, 68, 0, 0, 0],
            &[0, 117, 255, 218, 243, 198, 0, 0, 0],
            &[0, 117, 255, 66, 130, 255, 85, 0, 0],
            &[0, 117, 255, 62, 14, 236, 221, 6, 0],
            &[0, 117, 255, 62, 0, 115, 255, 113, 0],
            &[0, 0, 0, 9, 68, 31, 0, 0, 0],
            &[0, 0, 0, 73, 117, 0, 0, 0, 0],
        ],
        // letter: 'ķ' / 0x137
        'ķ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[8, 30, 14, 0, 0, 0, 0, 0, 0],
            &[71, 255, 117, 0, 0, 0, 0, 0, 0],
            &[71, 255, 117, 0, 0, 0, 0, 0, 0],
            &[71, 255, 117, 0, 54, 141, 87, 0, 0],
            &[71, 255, 117, 32, 234, 219, 20, 0, 0],
            &[71, 255, 126, 212, 234, 35, 0, 0, 0],
            &[71, 255, 236, 255, 180, 0, 0, 0, 0],
            &[71, 255, 225, 155, 255, 89, 0, 0, 0],
            &[71, 255, 117, 4, 206, 236, 23, 0, 0],
            &[71, 255, 117, 0, 49, 251, 175, 0, 0],
            &[0, 0, 0, 67, 42, 0, 0, 0, 0],
            &[0, 0, 47, 142, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĸ' / 0x138
        'ĸ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[39, 141, 64, 0, 54, 141, 87, 0, 0],
            &[71, 255, 117, 32, 234, 219, 20, 0, 0],
            &[71, 255, 126, 212, 234, 35, 0, 0, 0],
            &[71, 255, 236, 255, 180, 0, 0, 0, 0],
            &[71, 255, 225, 155, 255, 89, 0, 0, 0],
            &[71, 255, 117, 4, 206, 236, 23, 0, 0],
            &[71, 255, 117, 0, 49, 251, 175, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ĺ' / 0x139
        'Ĺ' => &[
            &[0, 0, 41, 53, 8, 0, 0, 0, 0],
            &[0, 45, 250, 151, 1, 0, 0, 0, 0],
            &[0, 134, 121, 0, 0, 0, 0, 0, 0],
            &[0, 133, 119, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 234, 137, 137, 137, 49, 0, 0],
            &[0, 235, 255, 255, 255, 255, 92, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ĺ' / 0x13a
        'ĺ' => &[
            &[0, 0, 0, 0, 53, 194, 114, 0, 0],
            &[0, 0, 0, 8, 212, 167, 5, 0, 0],
            &[0, 0, 26, 40, 71, 20, 0, 0, 0],
            &[0, 0, 219, 255, 255, 158, 0, 0, 0],
            &[0, 0, 11, 82, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 7, 53, 113, 255, 198, 71, 20, 0],
            &[0, 71, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ļ' / 0x13b
        'Ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 133, 119, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 234, 137, 137, 137, 49, 0, 0],
            &[0, 235, 255, 255, 255, 255, 92, 0, 0],
            &[0, 0, 0, 60, 49, 0, 0, 0, 0],
            &[0, 0, 29, 155, 5, 0, 0, 0, 0],
        ],
        // letter: 'ļ' / 0x13c
        'ļ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 30, 30, 19, 0, 0, 0],
            &[0, 0, 219, 255, 255, 158, 0, 0, 0],
            &[0, 0, 11, 82, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 0, 0, 0],
            &[0, 7, 53, 113, 255, 198, 71, 20, 0],
            &[0, 71, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 3, 68, 37, 0, 0, 0],
            &[0, 0, 0, 59, 131, 0, 0, 0, 0],
        ],
        // letter: 'Ľ' / 0x13d
        'Ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 133, 119, 0, 49, 145, 31, 0, 0],
            &[0, 235, 209, 0, 125, 210, 2, 0, 0],
            &[0, 235, 209, 0, 57, 40, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 234, 137, 137, 137, 49, 0, 0],
            &[0, 235, 255, 255, 255, 255, 92, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ľ' / 0x13e
        'ľ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 30, 30, 19, 14, 30, 4, 0],
            &[0, 219, 255, 255, 158, 143, 225, 5, 0],
            &[0, 11, 82, 255, 158, 153, 85, 0, 0],
            &[0, 0, 30, 255, 158, 0, 0, 0, 0],
            &[0, 0, 30, 255, 158, 0, 0, 0, 0],
            &[0, 0, 30, 255, 158, 0, 0, 0, 0],
            &[0, 0, 30, 255, 158, 0, 0, 0, 0],
            &[0, 0, 30, 255, 158, 0, 0, 0, 0],
            &[7, 53, 113, 255, 198, 71, 20, 0, 0],
            &[71, 255, 255, 255, 255, 255, 153, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŀ' / 0x13f
        'Ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 133, 119, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 209, 0, 62, 163, 41, 0, 0],
            &[0, 235, 209, 0, 147, 255, 111, 0, 0],
            &[0, 235, 209, 0, 16, 83, 12, 0, 0],
            &[0, 235, 209, 0, 0, 0, 0, 0, 0],
            &[0, 235, 234, 137, 137, 137, 49, 0, 0],
            &[0, 235, 255, 255, 255, 255, 92, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŀ' / 0x140
        'ŀ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 26, 30, 30, 19, 0, 0, 0, 0],
            &[0, 219, 255, 255, 158, 0, 0, 0, 0],
            &[0, 11, 82, 255, 158, 0, 0, 0, 0],
            &[0, 0, 30, 255, 158, 0, 0, 0, 0],
            &[0, 0, 30, 255, 158, 0, 72, 122, 6],
            &[0, 0, 30, 255, 158, 0, 211, 255, 54],
            &[0, 0, 30, 255, 158, 0, 55, 112, 3],
            &[0, 0, 30, 255, 158, 0, 0, 0, 0],
            &[7, 53, 113, 255, 198, 71, 20, 0, 0],
            &[71, 255, 255, 255, 255, 255, 153, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ł' / 0x141
        'Ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 63, 145, 43, 0, 0, 0, 0],
            &[0, 0, 112, 255, 76, 0, 0, 0, 0],
            &[0, 0, 112, 255, 76, 0, 0, 0, 0],
            &[0, 0, 112, 255, 226, 126, 0, 0, 0],
            &[0, 0, 206, 255, 255, 79, 0, 0, 0],
            &[0, 78, 255, 255, 87, 0, 0, 0, 0],
            &[0, 1, 147, 255, 76, 0, 0, 0, 0],
            &[0, 0, 112, 255, 172, 137, 137, 115, 0],
            &[0, 0, 112, 255, 255, 255, 255, 214, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ł' / 0x142
        'ł' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 30, 30, 19, 0, 0, 0],
            &[0, 0, 219, 255, 255, 158, 0, 0, 0],
            &[0, 0, 11, 82, 255, 158, 0, 0, 0],
            &[0, 0, 0, 30, 255, 158, 1, 0, 0],
            &[0, 0, 0, 30, 255, 255, 175, 0, 0],
            &[0, 0, 0, 105, 255, 255, 95, 0, 0],
            &[0, 0, 96, 255, 255, 175, 0, 0, 0],
            &[0, 0, 14, 84, 255, 158, 0, 0, 0],
            &[0, 7, 53, 113, 255, 198, 71, 20, 0],
            &[0, 71, 255, 255, 255, 255, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ń' / 0x143
        'Ń' => &[
            &[0, 0, 0, 0, 2, 53, 47, 0, 0],
            &[0, 0, 0, 0, 109, 251, 87, 0, 0],
            &[0, 0, 0, 11, 172, 72, 0, 0, 0],
            &[0, 66, 145, 141, 3, 0, 129, 95, 0],
            &[0, 117, 255, 255, 64, 0, 227, 168, 0],
            &[0, 117, 253, 214, 148, 0, 227, 168, 0],
            &[0, 117, 255, 143, 230, 1, 227, 168, 0],
            &[0, 117, 255, 69, 255, 59, 227, 168, 0],
            &[0, 117, 255, 22, 220, 143, 223, 168, 0],
            &[0, 117, 255, 22, 137, 226, 217, 168, 0],
            &[0, 117, 255, 22, 54, 255, 248, 168, 0],
            &[0, 117, 255, 22, 1, 226, 255, 168, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ń' / 0x144
        'ń' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 2, 49, 43, 0, 0],
            &[0, 0, 0, 0, 109, 251, 88, 0, 0],
            &[0, 0, 0, 12, 176, 73, 0, 0, 0],
            &[0, 48, 141, 28, 131, 161, 95, 0, 0],
            &[0, 87, 255, 235, 216, 242, 255, 82, 0],
            &[0, 87, 255, 178, 0, 71, 255, 144, 0],
            &[0, 87, 255, 111, 0, 36, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ņ' / 0x145
        'Ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 145, 141, 3, 0, 129, 95, 0],
            &[0, 117, 255, 255, 64, 0, 227, 168, 0],
            &[0, 117, 253, 214, 148, 0, 227, 168, 0],
            &[0, 117, 255, 143, 230, 1, 227, 168, 0],
            &[0, 117, 255, 69, 255, 59, 227, 168, 0],
            &[0, 117, 255, 22, 220, 143, 223, 168, 0],
            &[0, 117, 255, 22, 137, 226, 217, 168, 0],
            &[0, 117, 255, 22, 54, 255, 248, 168, 0],
            &[0, 117, 255, 22, 1, 226, 255, 168, 0],
            &[0, 0, 0, 16, 68, 24, 0, 0, 0],
            &[0, 0, 0, 89, 101, 0, 0, 0, 0],
        ],
        // letter: 'ņ' / 0x146
        'ņ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 48, 141, 28, 131, 161, 95, 0, 0],
            &[0, 87, 255, 235, 216, 242, 255, 82, 0],
            &[0, 87, 255, 178, 0, 71, 255, 144, 0],
            &[0, 87, 255, 111, 0, 36, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 0, 0, 12, 68, 28, 0, 0, 0],
            &[0, 0, 0, 81, 109, 0, 0, 0, 0],
        ],
        // letter: 'Ň' / 0x147
        'Ň' => &[
            &[0, 0, 28, 35, 0, 18, 45, 0, 0],
            &[0, 0, 36, 235, 122, 227, 104, 0, 0],
            &[0, 0, 0, 81, 182, 140, 0, 0, 0],
            &[0, 66, 145, 141, 3, 0, 129, 95, 0],
            &[0, 117, 255, 255, 64, 0, 227, 168, 0],
            &[0, 117, 253, 214, 148, 0, 227, 168, 0],
            &[0, 117, 255, 143, 230, 1, 227, 168, 0],
            &[0, 117, 255, 69, 255, 59, 227, 168, 0],
            &[0, 117, 255, 22, 220, 143, 223, 168, 0],
            &[0, 117, 255, 22, 137, 226, 217, 168, 0],
            &[0, 117, 255, 22, 54, 255, 248, 168, 0],
            &[0, 117, 255, 22, 1, 226, 255, 168, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ň' / 0x148
        'ň' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 31, 0, 17, 41, 0, 0],
            &[0, 0, 40, 236, 117, 227, 104, 0, 0],
            &[0, 0, 0, 86, 186, 142, 0, 0, 0],
            &[0, 48, 141, 28, 131, 161, 95, 0, 0],
            &[0, 87, 255, 235, 216, 242, 255, 82, 0],
            &[0, 87, 255, 178, 0, 71, 255, 144, 0],
            &[0, 87, 255, 111, 0, 36, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŉ' / 0x149
        'ŉ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 120, 129, 0, 0, 0, 0, 0, 0],
            &[2, 245, 167, 0, 0, 0, 0, 0, 0],
            &[43, 255, 103, 141, 37, 123, 161, 98, 0],
            &[30, 93, 57, 255, 235, 220, 241, 255, 83],
            &[0, 0, 56, 255, 203, 2, 69, 255, 144],
            &[0, 0, 56, 255, 142, 0, 36, 255, 153],
            &[0, 0, 56, 255, 133, 0, 35, 255, 153],
            &[0, 0, 56, 255, 133, 0, 35, 255, 153],
            &[0, 0, 56, 255, 133, 0, 35, 255, 153],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŋ' / 0x14a
        'Ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 145, 141, 3, 0, 129, 95, 0],
            &[0, 117, 255, 255, 64, 0, 227, 168, 0],
            &[0, 117, 253, 214, 148, 0, 227, 168, 0],
            &[0, 117, 255, 143, 230, 1, 227, 168, 0],
            &[0, 117, 255, 69, 255, 59, 227, 168, 0],
            &[0, 117, 255, 22, 220, 143, 223, 168, 0],
            &[0, 117, 255, 22, 137, 226, 217, 168, 0],
            &[0, 117, 255, 22, 54, 255, 255, 222, 0],
            &[0, 117, 255, 22, 1, 226, 255, 255, 0],
            &[0, 0, 0, 0, 0, 8, 228, 147, 0],
            &[0, 0, 0, 13, 117, 107, 34, 0, 0],
        ],
        // letter: 'ŋ' / 0x14b
        'ŋ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 48, 141, 28, 131, 161, 95, 0, 0],
            &[0, 87, 255, 235, 216, 242, 255, 82, 0],
            &[0, 87, 255, 178, 0, 71, 255, 144, 0],
            &[0, 87, 255, 111, 0, 36, 255, 153, 0],
            &[0, 87, 255, 102, 0, 35, 255, 153, 0],
            &[0, 87, 255, 102, 0, 47, 255, 204, 0],
            &[0, 87, 255, 102, 0, 71, 255, 255, 0],
            &[0, 0, 0, 0, 0, 44, 255, 151, 0],
            &[0, 0, 0, 0, 115, 219, 166, 13, 0],
        ],
        // letter: 'Ō' / 0x14c
        'Ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 38, 170, 170, 170, 70, 0, 0],
            &[0, 0, 29, 133, 133, 133, 54, 0, 0],
            &[0, 0, 18, 130, 174, 148, 38, 0, 0],
            &[0, 4, 208, 255, 216, 251, 238, 29, 0],
            &[0, 82, 255, 136, 0, 90, 255, 136, 0],
            &[0, 139, 255, 60, 0, 8, 254, 192, 0],
            &[0, 158, 255, 38, 0, 0, 241, 211, 0],
            &[0, 152, 255, 50, 0, 0, 247, 203, 0],
            &[0, 113, 255, 90, 0, 40, 255, 164, 0],
            &[0, 30, 249, 225, 109, 199, 255, 75, 0],
            &[0, 0, 85, 234, 255, 245, 125, 0, 0],
            &[0, 0, 0, 3, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ō' / 0x14d
        'ō' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 37, 166, 166, 166, 68, 0, 0],
            &[0, 0, 30, 137, 137, 137, 56, 0, 0],
            &[0, 0, 13, 119, 164, 137, 33, 0, 0],
            &[0, 6, 205, 253, 204, 247, 238, 31, 0],
            &[0, 83, 255, 132, 0, 80, 255, 137, 0],
            &[0, 125, 255, 70, 0, 21, 255, 177, 0],
            &[0, 113, 255, 93, 0, 41, 255, 163, 0],
            &[0, 38, 251, 218, 94, 186, 255, 84, 0],
            &[0, 0, 90, 235, 255, 246, 131, 0, 0],
            &[0, 0, 0, 3, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŏ' / 0x14e
        'Ŏ' => &[
            &[0, 0, 24, 38, 0, 27, 36, 0, 0],
            &[0, 0, 59, 231, 122, 213, 107, 0, 0],
            &[0, 0, 0, 99, 171, 120, 4, 0, 0],
            &[0, 0, 18, 130, 174, 148, 38, 0, 0],
            &[0, 4, 208, 255, 216, 251, 238, 29, 0],
            &[0, 82, 255, 136, 0, 90, 255, 136, 0],
            &[0, 139, 255, 60, 0, 8, 254, 192, 0],
            &[0, 158, 255, 38, 0, 0, 241, 211, 0],
            &[0, 152, 255, 50, 0, 0, 247, 203, 0],
            &[0, 113, 255, 90, 0, 40, 255, 164, 0],
            &[0, 30, 249, 225, 109, 199, 255, 75, 0],
            &[0, 0, 85, 234, 255, 245, 125, 0, 0],
            &[0, 0, 0, 3, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŏ' / 0x14f
        'ŏ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 36, 0, 25, 34, 0, 0],
            &[0, 0, 61, 230, 118, 211, 108, 0, 0],
            &[0, 0, 0, 103, 175, 125, 4, 0, 0],
            &[0, 0, 13, 119, 164, 137, 33, 0, 0],
            &[0, 6, 205, 253, 204, 247, 238, 31, 0],
            &[0, 83, 255, 132, 0, 80, 255, 137, 0],
            &[0, 125, 255, 70, 0, 21, 255, 177, 0],
            &[0, 113, 255, 93, 0, 41, 255, 163, 0],
            &[0, 38, 251, 218, 94, 186, 255, 84, 0],
            &[0, 0, 90, 235, 255, 246, 131, 0, 0],
            &[0, 0, 0, 3, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ő' / 0x150
        'Ő' => &[
            &[0, 0, 0, 31, 53, 26, 53, 11, 0],
            &[0, 0, 9, 225, 152, 195, 185, 3, 0],
            &[0, 0, 75, 129, 50, 151, 8, 0, 0],
            &[0, 0, 18, 130, 174, 148, 38, 0, 0],
            &[0, 4, 208, 255, 216, 251, 238, 29, 0],
            &[0, 82, 255, 136, 0, 90, 255, 136, 0],
            &[0, 139, 255, 60, 0, 8, 254, 192, 0],
            &[0, 158, 255, 38, 0, 0, 241, 211, 0],
            &[0, 152, 255, 50, 0, 0, 247, 203, 0],
            &[0, 113, 255, 90, 0, 40, 255, 164, 0],
            &[0, 30, 249, 225, 109, 199, 255, 75, 0],
            &[0, 0, 85, 234, 255, 245, 125, 0, 0],
            &[0, 0, 0, 3, 24, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ő' / 0x151
        'ő' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 49, 24, 49, 10, 0],
            &[0, 0, 8, 224, 155, 193, 188, 4, 0],
            &[0, 0, 76, 133, 51, 155, 9, 0, 0],
            &[0, 0, 13, 119, 164, 137, 33, 0, 0],
            &[0, 6, 205, 253, 204, 247, 238, 31, 0],
            &[0, 83, 255, 132, 0, 80, 255, 137, 0],
            &[0, 125, 255, 70, 0, 21, 255, 177, 0],
            &[0, 113, 255, 93, 0, 41, 255, 163, 0],
            &[0, 38, 251, 218, 94, 186, 255, 84, 0],
            &[0, 0, 90, 235, 255, 246, 131, 0, 0],
            &[0, 0, 0, 3, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Œ' / 0x152
        'Œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 42, 153, 165, 145, 145, 139, 0],
            &[0, 15, 234, 252, 231, 255, 241, 226, 0],
            &[0, 98, 255, 108, 80, 255, 80, 0, 0],
            &[0, 144, 255, 38, 80, 255, 115, 37, 0],
            &[0, 160, 255, 20, 80, 255, 255, 190, 0],
            &[0, 153, 255, 28, 80, 255, 131, 55, 0],
            &[0, 121, 255, 62, 80, 255, 80, 0, 0],
            &[0, 50, 255, 194, 156, 255, 166, 119, 0],
            &[0, 0, 132, 253, 255, 255, 255, 245, 0],
            &[0, 0, 0, 11, 18, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'œ' / 0x153
        'œ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 102, 149, 42, 103, 150, 32, 0, 0],
            &[88, 249, 194, 241, 232, 193, 194, 0, 0],
            &[169, 185, 17, 255, 137, 64, 253, 6, 0],
            &[194, 164, 0, 246, 247, 242, 255, 19, 0],
            &[185, 171, 2, 252, 150, 33, 33, 2, 0],
            &[133, 223, 92, 255, 207, 33, 75, 0, 0],
            &[24, 221, 253, 140, 205, 255, 203, 0, 0],
            &[0, 3, 14, 0, 0, 19, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŕ' / 0x154
        'Ŕ' => &[
            &[0, 0, 0, 5, 53, 44, 0, 0, 0],
            &[0, 0, 0, 124, 247, 75, 0, 0, 0],
            &[0, 0, 18, 176, 62, 0, 0, 0, 0],
            &[5, 145, 145, 139, 101, 11, 0, 0, 0],
            &[10, 255, 248, 241, 255, 213, 9, 0, 0],
            &[10, 255, 179, 0, 145, 255, 77, 0, 0],
            &[10, 255, 179, 0, 131, 255, 73, 0, 0],
            &[10, 255, 233, 194, 252, 217, 9, 0, 0],
            &[10, 255, 238, 229, 253, 35, 0, 0, 0],
            &[10, 255, 179, 55, 254, 146, 0, 0, 0],
            &[10, 255, 179, 0, 168, 251, 42, 0, 0],
            &[10, 255, 179, 0, 33, 249, 186, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŕ' / 0x155
        'ŕ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 36, 49, 9, 0],
            &[0, 0, 0, 0, 38, 247, 161, 2, 0],
            &[0, 0, 0, 0, 131, 130, 1, 0, 0],
            &[0, 25, 141, 141, 81, 54, 158, 141, 0],
            &[0, 32, 200, 249, 218, 244, 255, 183, 0],
            &[0, 0, 0, 205, 255, 126, 12, 12, 0],
            &[0, 0, 0, 205, 250, 6, 0, 0, 0],
            &[0, 0, 0, 205, 239, 0, 0, 0, 0],
            &[0, 17, 74, 226, 246, 82, 24, 0, 0],
            &[0, 129, 255, 255, 255, 255, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŗ' / 0x156
        'Ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[5, 145, 145, 139, 101, 11, 0, 0, 0],
            &[10, 255, 248, 241, 255, 213, 9, 0, 0],
            &[10, 255, 179, 0, 145, 255, 77, 0, 0],
            &[10, 255, 179, 0, 131, 255, 73, 0, 0],
            &[10, 255, 233, 194, 252, 217, 9, 0, 0],
            &[10, 255, 238, 229, 253, 35, 0, 0, 0],
            &[10, 255, 179, 55, 254, 146, 0, 0, 0],
            &[10, 255, 179, 0, 168, 251, 42, 0, 0],
            &[10, 255, 179, 0, 33, 249, 186, 0, 0],
            &[0, 0, 0, 64, 45, 0, 0, 0, 0],
            &[0, 0, 39, 148, 2, 0, 0, 0, 0],
        ],
        // letter: 'ŗ' / 0x157
        'ŗ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 25, 141, 141, 81, 54, 158, 141, 0],
            &[0, 32, 200, 249, 218, 244, 255, 183, 0],
            &[0, 0, 0, 205, 255, 126, 12, 12, 0],
            &[0, 0, 0, 205, 250, 6, 0, 0, 0],
            &[0, 0, 0, 205, 239, 0, 0, 0, 0],
            &[0, 17, 74, 226, 246, 82, 24, 0, 0],
            &[0, 129, 255, 255, 255, 255, 166, 0, 0],
            &[0, 0, 0, 49, 60, 0, 0, 0, 0],
            &[0, 0, 4, 163, 22, 0, 0, 0, 0],
        ],
        // letter: 'Ř' / 0x158
        'Ř' => &[
            &[0, 31, 32, 0, 21, 42, 0, 0, 0],
            &[0, 45, 237, 118, 234, 89, 0, 0, 0],
            &[0, 0, 92, 182, 129, 0, 0, 0, 0],
            &[5, 145, 145, 139, 101, 11, 0, 0, 0],
            &[10, 255, 248, 241, 255, 213, 9, 0, 0],
            &[10, 255, 179, 0, 145, 255, 77, 0, 0],
            &[10, 255, 179, 0, 131, 255, 73, 0, 0],
            &[10, 255, 233, 194, 252, 217, 9, 0, 0],
            &[10, 255, 238, 229, 253, 35, 0, 0, 0],
            &[10, 255, 179, 55, 254, 146, 0, 0, 0],
            &[10, 255, 179, 0, 168, 251, 42, 0, 0],
            &[10, 255, 179, 0, 33, 249, 186, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ř' / 0x159
        'ř' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 46, 0, 2, 48, 7, 0],
            &[0, 0, 6, 196, 159, 179, 181, 2, 0],
            &[0, 0, 0, 30, 184, 181, 19, 0, 0],
            &[0, 25, 141, 141, 81, 54, 158, 141, 0],
            &[0, 32, 200, 249, 218, 244, 255, 183, 0],
            &[0, 0, 0, 205, 255, 126, 12, 12, 0],
            &[0, 0, 0, 205, 250, 6, 0, 0, 0],
            &[0, 0, 0, 205, 239, 0, 0, 0, 0],
            &[0, 17, 74, 226, 246, 82, 24, 0, 0],
            &[0, 129, 255, 255, 255, 255, 166, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ś' / 0x15a
        'Ś' => &[
            &[0, 0, 0, 0, 0, 49, 52, 0, 0],
            &[0, 0, 0, 0, 81, 254, 111, 0, 0],
            &[0, 0, 0, 3, 161, 92, 0, 0, 0],
            &[0, 0, 11, 118, 168, 159, 115, 19, 0],
            &[0, 0, 177, 255, 215, 213, 252, 33, 0],
            &[0, 0, 245, 220, 1, 0, 20, 0, 0],
            &[0, 0, 203, 250, 95, 2, 0, 0, 0],
            &[0, 0, 36, 218, 255, 208, 60, 0, 0],
            &[0, 0, 0, 8, 118, 242, 249, 56, 0],
            &[0, 0, 0, 0, 0, 87, 255, 131, 0],
            &[0, 38, 185, 111, 90, 183, 255, 89, 0],
            &[0, 25, 210, 255, 255, 242, 131, 0, 0],
            &[0, 0, 0, 10, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ś' / 0x15b
        'ś' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 49, 44, 0, 0],
            &[0, 0, 0, 0, 106, 251, 91, 0, 0],
            &[0, 0, 0, 11, 175, 75, 0, 0, 0],
            &[0, 0, 24, 130, 166, 151, 97, 9, 0],
            &[0, 0, 205, 254, 203, 220, 237, 8, 0],
            &[0, 0, 233, 236, 51, 0, 14, 0, 0],
            &[0, 0, 94, 243, 255, 205, 66, 0, 0],
            &[0, 0, 0, 17, 113, 235, 251, 27, 0],
            &[0, 8, 179, 115, 94, 205, 255, 47, 0],
            &[0, 7, 224, 255, 255, 249, 150, 0, 0],
            &[0, 0, 0, 16, 26, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŝ' / 0x15c
        'Ŝ' => &[
            &[0, 0, 0, 9, 53, 38, 0, 0, 0],
            &[0, 0, 0, 150, 233, 246, 44, 0, 0],
            &[0, 0, 44, 168, 26, 96, 143, 0, 0],
            &[0, 0, 11, 118, 168, 159, 115, 19, 0],
            &[0, 0, 177, 255, 215, 213, 252, 33, 0],
            &[0, 0, 245, 220, 1, 0, 20, 0, 0],
            &[0, 0, 203, 250, 95, 2, 0, 0, 0],
            &[0, 0, 36, 218, 255, 208, 60, 0, 0],
            &[0, 0, 0, 8, 118, 242, 249, 56, 0],
            &[0, 0, 0, 0, 0, 87, 255, 131, 0],
            &[0, 38, 185, 111, 90, 183, 255, 89, 0],
            &[0, 25, 210, 255, 255, 242, 131, 0, 0],
            &[0, 0, 0, 10, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŝ' / 0x15d
        'ŝ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 13, 49, 30, 0, 0, 0],
            &[0, 0, 0, 174, 235, 235, 26, 0, 0],
            &[0, 0, 64, 163, 17, 120, 125, 0, 0],
            &[0, 0, 24, 130, 166, 151, 97, 9, 0],
            &[0, 0, 205, 254, 203, 220, 237, 8, 0],
            &[0, 0, 233, 236, 51, 0, 14, 0, 0],
            &[0, 0, 94, 243, 255, 205, 66, 0, 0],
            &[0, 0, 0, 17, 113, 235, 251, 27, 0],
            &[0, 8, 179, 115, 94, 205, 255, 47, 0],
            &[0, 7, 224, 255, 255, 249, 150, 0, 0],
            &[0, 0, 0, 16, 26, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ş' / 0x15e
        'Ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 11, 118, 168, 159, 115, 19, 0],
            &[0, 0, 177, 255, 215, 213, 252, 33, 0],
            &[0, 0, 245, 220, 1, 0, 20, 0, 0],
            &[0, 0, 203, 250, 95, 2, 0, 0, 0],
            &[0, 0, 36, 218, 255, 208, 60, 0, 0],
            &[0, 0, 0, 8, 118, 242, 249, 56, 0],
            &[0, 0, 0, 0, 0, 87, 255, 131, 0],
            &[0, 38, 185, 111, 90, 183, 255, 89, 0],
            &[0, 25, 210, 255, 255, 243, 131, 0, 0],
            &[0, 0, 0, 31, 255, 60, 0, 0, 0],
            &[0, 0, 0, 132, 216, 111, 0, 0, 0],
        ],
        // letter: 'ş' / 0x15f
        'ş' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 24, 130, 166, 151, 97, 9, 0],
            &[0, 0, 205, 254, 203, 220, 237, 8, 0],
            &[0, 0, 233, 236, 51, 0, 14, 0, 0],
            &[0, 0, 94, 243, 255, 205, 66, 0, 0],
            &[0, 0, 0, 17, 113, 235, 251, 27, 0],
            &[0, 8, 179, 115, 94, 205, 255, 47, 0],
            &[0, 7, 224, 255, 255, 251, 150, 0, 0],
            &[0, 0, 0, 28, 255, 77, 0, 0, 0],
            &[0, 0, 0, 118, 217, 124, 0, 0, 0],
        ],
        // letter: 'Š' / 0x160
        'Š' => &[
            &[0, 0, 22, 40, 0, 12, 51, 0, 0],
            &[0, 0, 22, 225, 132, 214, 131, 0, 0],
            &[0, 0, 0, 62, 182, 157, 2, 0, 0],
            &[0, 0, 11, 118, 168, 159, 115, 19, 0],
            &[0, 0, 177, 255, 215, 213, 252, 33, 0],
            &[0, 0, 245, 220, 1, 0, 20, 0, 0],
            &[0, 0, 203, 250, 95, 2, 0, 0, 0],
            &[0, 0, 36, 218, 255, 208, 60, 0, 0],
            &[0, 0, 0, 8, 118, 242, 249, 56, 0],
            &[0, 0, 0, 0, 0, 87, 255, 131, 0],
            &[0, 38, 185, 111, 90, 183, 255, 89, 0],
            &[0, 25, 210, 255, 255, 242, 131, 0, 0],
            &[0, 0, 0, 10, 22, 1, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'š' / 0x161
        'š' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 26, 32, 0, 16, 41, 0, 0],
            &[0, 0, 38, 235, 117, 225, 107, 0, 0],
            &[0, 0, 0, 84, 186, 144, 0, 0, 0],
            &[0, 0, 24, 130, 166, 151, 97, 9, 0],
            &[0, 0, 205, 254, 203, 220, 237, 8, 0],
            &[0, 0, 233, 236, 51, 0, 14, 0, 0],
            &[0, 0, 94, 243, 255, 205, 66, 0, 0],
            &[0, 0, 0, 17, 113, 235, 251, 27, 0],
            &[0, 8, 179, 115, 94, 205, 255, 47, 0],
            &[0, 7, 224, 255, 255, 249, 150, 0, 0],
            &[0, 0, 0, 16, 26, 6, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ţ' / 0x162
        'Ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 93, 145, 145, 145, 145, 145, 122, 0],
            &[0, 152, 238, 243, 255, 246, 238, 200, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 119, 0, 0, 0],
            &[0, 0, 0, 20, 244, 59, 0, 0, 0],
            &[0, 0, 0, 132, 216, 111, 0, 0, 0],
        ],
        // letter: 'ţ' / 0x163
        'ţ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 41, 0, 0, 0, 0],
            &[0, 0, 0, 144, 224, 0, 0, 0, 0],
            &[0, 21, 97, 218, 241, 141, 141, 47, 0],
            &[0, 82, 227, 252, 252, 227, 227, 76, 0],
            &[0, 0, 0, 221, 224, 0, 0, 0, 0],
            &[0, 0, 0, 221, 224, 0, 0, 0, 0],
            &[0, 0, 0, 220, 229, 0, 0, 0, 0],
            &[0, 0, 0, 194, 255, 121, 96, 48, 0],
            &[0, 0, 0, 65, 233, 255, 254, 91, 0],
            &[0, 0, 0, 0, 69, 249, 46, 0, 0],
            &[0, 0, 0, 0, 178, 209, 72, 0, 0],
        ],
        // letter: 'Ť' / 0x164
        'Ť' => &[
            &[0, 0, 31, 32, 0, 21, 42, 0, 0],
            &[0, 0, 45, 237, 118, 234, 89, 0, 0],
            &[0, 0, 0, 92, 182, 129, 0, 0, 0],
            &[0, 93, 145, 145, 145, 145, 145, 122, 0],
            &[0, 152, 238, 243, 255, 246, 238, 200, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ť' / 0x165
        'ť' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 47, 55, 0],
            &[0, 0, 0, 21, 41, 0, 217, 146, 0],
            &[0, 0, 0, 144, 224, 6, 168, 21, 0],
            &[0, 21, 97, 218, 241, 141, 141, 47, 0],
            &[0, 82, 227, 252, 252, 227, 227, 76, 0],
            &[0, 0, 0, 221, 224, 0, 0, 0, 0],
            &[0, 0, 0, 221, 224, 0, 0, 0, 0],
            &[0, 0, 0, 220, 229, 0, 0, 0, 0],
            &[0, 0, 0, 194, 255, 121, 96, 48, 0],
            &[0, 0, 0, 65, 233, 255, 254, 91, 0],
            &[0, 0, 0, 0, 2, 24, 12, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŧ' / 0x166
        'Ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 93, 145, 145, 145, 145, 145, 122, 0],
            &[0, 152, 238, 243, 255, 246, 238, 200, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 183, 255, 255, 255, 232, 0, 0],
            &[0, 0, 90, 191, 255, 239, 114, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 70, 255, 118, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŧ' / 0x167
        'ŧ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 21, 41, 0, 0, 0, 0],
            &[0, 0, 0, 144, 224, 0, 0, 0, 0],
            &[0, 21, 97, 218, 241, 141, 141, 47, 0],
            &[0, 82, 227, 252, 252, 227, 227, 76, 0],
            &[0, 8, 23, 244, 247, 23, 23, 9, 0],
            &[0, 93, 255, 255, 255, 255, 255, 101, 0],
            &[0, 3, 9, 229, 238, 9, 9, 3, 0],
            &[0, 0, 0, 194, 255, 121, 96, 48, 0],
            &[0, 0, 0, 65, 233, 255, 254, 91, 0],
            &[0, 0, 0, 0, 2, 24, 12, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ũ' / 0x168
        'Ũ' => &[
            &[0, 0, 0, 7, 0, 0, 19, 0, 0],
            &[0, 0, 73, 251, 210, 147, 196, 0, 0],
            &[0, 0, 115, 80, 96, 169, 57, 0, 0],
            &[0, 66, 145, 40, 0, 11, 145, 95, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 72, 0, 21, 255, 168, 0],
            &[0, 107, 255, 97, 0, 45, 255, 158, 0],
            &[0, 47, 254, 221, 113, 194, 255, 93, 0],
            &[0, 0, 112, 241, 255, 248, 146, 1, 0],
            &[0, 0, 0, 4, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ũ' / 0x169
        'ũ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 0, 0, 16, 0, 0],
            &[0, 0, 71, 250, 206, 143, 197, 0, 0],
            &[0, 0, 117, 84, 100, 173, 59, 0, 0],
            &[0, 56, 141, 48, 0, 28, 141, 76, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 53, 255, 138, 0],
            &[0, 100, 255, 94, 0, 82, 255, 138, 0],
            &[0, 70, 255, 197, 92, 209, 255, 138, 0],
            &[0, 2, 177, 255, 248, 127, 237, 138, 0],
            &[0, 0, 0, 16, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ū' / 0x16a
        'Ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 38, 170, 170, 170, 70, 0, 0],
            &[0, 0, 29, 133, 133, 133, 54, 0, 0],
            &[0, 66, 145, 40, 0, 11, 145, 95, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 72, 0, 21, 255, 168, 0],
            &[0, 107, 255, 97, 0, 45, 255, 158, 0],
            &[0, 47, 254, 221, 113, 194, 255, 93, 0],
            &[0, 0, 112, 241, 255, 248, 146, 1, 0],
            &[0, 0, 0, 4, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ū' / 0x16b
        'ū' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 37, 166, 166, 166, 68, 0, 0],
            &[0, 0, 30, 137, 137, 137, 56, 0, 0],
            &[0, 56, 141, 48, 0, 28, 141, 76, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 53, 255, 138, 0],
            &[0, 100, 255, 94, 0, 82, 255, 138, 0],
            &[0, 70, 255, 197, 92, 209, 255, 138, 0],
            &[0, 2, 177, 255, 248, 127, 237, 138, 0],
            &[0, 0, 0, 16, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŭ' / 0x16c
        'Ŭ' => &[
            &[0, 0, 24, 38, 0, 27, 36, 0, 0],
            &[0, 0, 59, 231, 122, 213, 107, 0, 0],
            &[0, 0, 0, 99, 171, 120, 4, 0, 0],
            &[0, 66, 145, 40, 0, 11, 145, 95, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 72, 0, 21, 255, 168, 0],
            &[0, 107, 255, 97, 0, 45, 255, 158, 0],
            &[0, 47, 254, 221, 113, 194, 255, 93, 0],
            &[0, 0, 112, 241, 255, 248, 146, 1, 0],
            &[0, 0, 0, 4, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŭ' / 0x16d
        'ŭ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 23, 36, 0, 25, 34, 0, 0],
            &[0, 0, 61, 230, 118, 211, 108, 0, 0],
            &[0, 0, 0, 103, 175, 125, 4, 0, 0],
            &[0, 56, 141, 48, 0, 28, 141, 76, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 53, 255, 138, 0],
            &[0, 100, 255, 94, 0, 82, 255, 138, 0],
            &[0, 70, 255, 197, 92, 209, 255, 138, 0],
            &[0, 2, 177, 255, 248, 127, 237, 138, 0],
            &[0, 0, 0, 16, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ů' / 0x16e
        'Ů' => &[
            &[0, 0, 0, 129, 223, 170, 0, 0, 0],
            &[0, 0, 0, 201, 70, 235, 1, 0, 0],
            &[0, 0, 0, 64, 171, 90, 0, 0, 0],
            &[0, 66, 145, 40, 0, 11, 145, 95, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 72, 0, 21, 255, 168, 0],
            &[0, 107, 255, 97, 0, 45, 255, 158, 0],
            &[0, 47, 254, 221, 113, 194, 255, 93, 0],
            &[0, 0, 112, 241, 255, 248, 146, 1, 0],
            &[0, 0, 0, 4, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 3, 0, 0, 0, 0],
        ],
        // letter: 'ů' / 0x16f
        'ů' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 1, 0, 0, 0, 0],
            &[0, 0, 0, 126, 225, 166, 0, 0, 0],
            &[0, 0, 0, 202, 66, 235, 1, 0, 0],
            &[0, 0, 0, 66, 175, 94, 0, 0, 0],
            &[0, 56, 141, 48, 0, 28, 141, 76, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 53, 255, 138, 0],
            &[0, 100, 255, 94, 0, 82, 255, 138, 0],
            &[0, 70, 255, 197, 92, 209, 255, 138, 0],
            &[0, 2, 177, 255, 248, 127, 237, 138, 0],
            &[0, 0, 0, 16, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ű' / 0x170
        'Ű' => &[
            &[0, 0, 0, 31, 53, 26, 53, 11, 0],
            &[0, 0, 9, 225, 152, 195, 185, 3, 0],
            &[0, 0, 75, 129, 50, 151, 8, 0, 0],
            &[0, 66, 145, 40, 0, 11, 145, 95, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 72, 0, 21, 255, 168, 0],
            &[0, 107, 255, 97, 0, 45, 255, 158, 0],
            &[0, 47, 254, 221, 113, 194, 255, 93, 0],
            &[0, 0, 112, 241, 255, 248, 146, 1, 0],
            &[0, 0, 0, 4, 24, 7, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ű' / 0x171
        'ű' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 28, 49, 24, 49, 10, 0],
            &[0, 0, 8, 224, 155, 193, 188, 4, 0],
            &[0, 0, 76, 133, 51, 155, 9, 0, 0],
            &[0, 56, 141, 48, 0, 28, 141, 76, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 53, 255, 138, 0],
            &[0, 100, 255, 94, 0, 82, 255, 138, 0],
            &[0, 70, 255, 197, 92, 209, 255, 138, 0],
            &[0, 2, 177, 255, 248, 127, 237, 138, 0],
            &[0, 0, 0, 16, 8, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ų' / 0x172
        'Ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 66, 145, 40, 0, 11, 145, 95, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 71, 0, 20, 255, 168, 0],
            &[0, 117, 255, 72, 0, 21, 255, 168, 0],
            &[0, 107, 255, 97, 0, 45, 255, 158, 0],
            &[0, 47, 254, 221, 113, 194, 255, 93, 0],
            &[0, 0, 112, 241, 255, 255, 153, 1, 0],
            &[0, 0, 0, 24, 238, 86, 0, 0, 0],
            &[0, 0, 0, 28, 188, 166, 0, 0, 0],
        ],
        // letter: 'ų' / 0x173
        'ų' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 56, 141, 48, 0, 28, 141, 76, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 51, 255, 138, 0],
            &[0, 102, 255, 87, 0, 53, 255, 138, 0],
            &[0, 100, 255, 94, 0, 82, 255, 138, 0],
            &[0, 70, 255, 197, 92, 209, 255, 138, 0],
            &[0, 2, 177, 255, 248, 127, 255, 140, 0],
            &[0, 0, 0, 16, 8, 62, 211, 13, 0],
            &[0, 0, 0, 0, 0, 66, 204, 114, 0],
        ],
        // letter: 'Ŵ' / 0x174
        'Ŵ' => &[
            &[0, 0, 18, 53, 29, 0, 0, 0, 0],
            &[0, 2, 190, 233, 228, 20, 0, 0, 0],
            &[0, 75, 153, 12, 126, 112, 0, 0, 0],
            &[124, 95, 0, 0, 0, 65, 145, 8, 0],
            &[204, 187, 0, 0, 0, 129, 254, 2, 0],
            &[185, 211, 36, 141, 68, 148, 239, 0, 0],
            &[166, 230, 96, 255, 153, 166, 221, 0, 0],
            &[147, 244, 136, 224, 191, 182, 203, 0, 0],
            &[128, 250, 175, 154, 230, 189, 185, 0, 0],
            &[109, 250, 210, 79, 255, 202, 167, 0, 0],
            &[90, 251, 245, 28, 230, 239, 149, 0, 0],
            &[71, 255, 245, 1, 192, 255, 131, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŵ' / 0x175
        'ŵ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 15, 49, 28, 0, 0, 0, 0],
            &[0, 1, 182, 235, 230, 21, 0, 0, 0],
            &[0, 71, 160, 15, 126, 118, 0, 0, 0],
            &[130, 91, 0, 0, 0, 57, 141, 19, 0],
            &[208, 182, 28, 151, 62, 122, 254, 8, 0],
            &[172, 205, 89, 255, 146, 144, 226, 0, 0],
            &[136, 227, 140, 224, 197, 167, 190, 0, 0],
            &[100, 250, 189, 140, 244, 190, 154, 0, 0],
            &[64, 255, 242, 54, 249, 241, 118, 0, 0],
            &[28, 255, 250, 6, 209, 255, 82, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ŷ' / 0x176
        'Ŷ' => &[
            &[0, 0, 0, 18, 53, 29, 0, 0, 0],
            &[0, 0, 2, 190, 233, 228, 20, 0, 0],
            &[0, 0, 75, 153, 12, 126, 112, 0, 0],
            &[0, 90, 145, 28, 0, 3, 139, 119, 0],
            &[0, 64, 255, 134, 0, 79, 255, 116, 0],
            &[0, 0, 198, 235, 7, 187, 238, 12, 0],
            &[0, 0, 76, 255, 134, 254, 129, 0, 0],
            &[0, 0, 1, 209, 255, 244, 19, 0, 0],
            &[0, 0, 0, 95, 255, 149, 0, 0, 0],
            &[0, 0, 0, 67, 255, 121, 0, 0, 0],
            &[0, 0, 0, 67, 255, 121, 0, 0, 0],
            &[0, 0, 0, 67, 255, 121, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ŷ' / 0x177
        'ŷ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 16, 49, 27, 0, 0, 0],
            &[0, 0, 2, 187, 235, 227, 18, 0, 0],
            &[0, 0, 75, 157, 13, 130, 114, 0, 0],
            &[0, 91, 141, 32, 0, 2, 137, 121, 0],
            &[0, 87, 255, 124, 0, 57, 255, 147, 0],
            &[0, 6, 235, 210, 0, 133, 255, 54, 0],
            &[0, 0, 140, 255, 40, 208, 217, 0, 0],
            &[0, 0, 39, 255, 148, 255, 124, 0, 0],
            &[0, 0, 0, 193, 246, 254, 32, 0, 0],
            &[0, 0, 0, 92, 255, 194, 0, 0, 0],
            &[0, 0, 0, 103, 255, 98, 0, 0, 0],
            &[0, 64, 235, 194, 42, 0, 0, 0, 0],
        ],
        // letter: 'Ÿ' / 0x178
        'Ÿ' => &[
            &[0, 0, 0, 6, 0, 3, 3, 0, 0],
            &[0, 0, 136, 232, 2, 188, 185, 0, 0],
            &[0, 0, 65, 126, 0, 97, 95, 0, 0],
            &[0, 90, 145, 28, 0, 3, 139, 119, 0],
            &[0, 64, 255, 134, 0, 79, 255, 116, 0],
            &[0, 0, 198, 235, 7, 187, 238, 12, 0],
            &[0, 0, 76, 255, 134, 254, 129, 0, 0],
            &[0, 0, 1, 209, 255, 244, 19, 0, 0],
            &[0, 0, 0, 95, 255, 149, 0, 0, 0],
            &[0, 0, 0, 67, 255, 121, 0, 0, 0],
            &[0, 0, 0, 67, 255, 121, 0, 0, 0],
            &[0, 0, 0, 67, 255, 121, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ź' / 0x179
        'Ź' => &[
            &[0, 0, 0, 0, 0, 50, 52, 0, 0],
            &[0, 0, 0, 0, 87, 254, 106, 0, 0],
            &[0, 0, 0, 4, 164, 87, 0, 0, 0],
            &[0, 0, 107, 145, 145, 145, 145, 48, 0],
            &[0, 0, 171, 232, 232, 247, 255, 74, 0],
            &[0, 0, 0, 0, 12, 230, 205, 2, 0],
            &[0, 0, 0, 0, 136, 254, 58, 0, 0],
            &[0, 0, 0, 35, 248, 162, 0, 0, 0],
            &[0, 0, 0, 177, 242, 25, 0, 0, 0],
            &[0, 0, 69, 255, 116, 0, 0, 0, 0],
            &[0, 2, 213, 252, 133, 127, 127, 46, 0],
            &[0, 10, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ź' / 0x17a
        'ź' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 48, 45, 0, 0],
            &[0, 0, 0, 0, 97, 253, 99, 0, 0],
            &[0, 0, 0, 8, 172, 82, 0, 0, 0],
            &[0, 5, 141, 141, 141, 141, 141, 44, 0],
            &[0, 8, 208, 208, 208, 253, 255, 63, 0],
            &[0, 0, 0, 0, 111, 255, 148, 0, 0],
            &[0, 0, 0, 66, 251, 189, 4, 0, 0],
            &[0, 0, 33, 236, 220, 18, 0, 0, 0],
            &[0, 10, 209, 255, 136, 94, 94, 39, 0],
            &[0, 56, 255, 255, 255, 255, 255, 107, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ż' / 0x17b
        'Ż' => &[
            &[0, 0, 0, 1, 58, 22, 0, 0, 0],
            &[0, 0, 0, 57, 255, 179, 0, 0, 0],
            &[0, 0, 0, 15, 151, 77, 0, 0, 0],
            &[0, 0, 107, 145, 145, 145, 145, 48, 0],
            &[0, 0, 171, 232, 232, 247, 255, 74, 0],
            &[0, 0, 0, 0, 12, 230, 205, 2, 0],
            &[0, 0, 0, 0, 136, 254, 58, 0, 0],
            &[0, 0, 0, 35, 248, 162, 0, 0, 0],
            &[0, 0, 0, 177, 242, 25, 0, 0, 0],
            &[0, 0, 69, 255, 116, 0, 0, 0, 0],
            &[0, 2, 213, 252, 133, 127, 127, 46, 0],
            &[0, 10, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ż' / 0x17c
        'ż' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 2, 56, 17, 0, 0, 0],
            &[0, 0, 0, 69, 255, 166, 0, 0, 0],
            &[0, 0, 0, 22, 156, 72, 0, 0, 0],
            &[0, 5, 141, 141, 141, 141, 141, 44, 0],
            &[0, 8, 208, 208, 208, 253, 255, 63, 0],
            &[0, 0, 0, 0, 111, 255, 148, 0, 0],
            &[0, 0, 0, 66, 251, 189, 4, 0, 0],
            &[0, 0, 33, 236, 220, 18, 0, 0, 0],
            &[0, 10, 209, 255, 136, 94, 94, 39, 0],
            &[0, 56, 255, 255, 255, 255, 255, 107, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'Ž' / 0x17d
        'Ž' => &[
            &[0, 0, 23, 39, 0, 13, 49, 0, 0],
            &[0, 0, 24, 228, 129, 217, 125, 0, 0],
            &[0, 0, 0, 66, 182, 154, 1, 0, 0],
            &[0, 0, 107, 145, 145, 145, 145, 48, 0],
            &[0, 0, 171, 232, 232, 247, 255, 74, 0],
            &[0, 0, 0, 0, 12, 230, 205, 2, 0],
            &[0, 0, 0, 0, 136, 254, 58, 0, 0],
            &[0, 0, 0, 35, 248, 162, 0, 0, 0],
            &[0, 0, 0, 177, 242, 25, 0, 0, 0],
            &[0, 0, 69, 255, 116, 0, 0, 0, 0],
            &[0, 2, 213, 252, 133, 127, 127, 46, 0],
            &[0, 10, 255, 255, 255, 255, 255, 92, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ž' / 0x17e
        'ž' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 24, 33, 0, 14, 43, 0, 0],
            &[0, 0, 33, 233, 120, 221, 116, 0, 0],
            &[0, 0, 0, 77, 186, 150, 0, 0, 0],
            &[0, 5, 141, 141, 141, 141, 141, 44, 0],
            &[0, 8, 208, 208, 208, 253, 255, 63, 0],
            &[0, 0, 0, 0, 111, 255, 148, 0, 0],
            &[0, 0, 0, 66, 251, 189, 4, 0, 0],
            &[0, 0, 33, 236, 220, 18, 0, 0, 0],
            &[0, 10, 209, 255, 136, 94, 94, 39, 0],
            &[0, 56, 255, 255, 255, 255, 255, 107, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        // letter: 'ſ' / 0x17f
        'ſ' => &[
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 4, 35, 23, 0, 0, 0],
            &[0, 0, 49, 238, 255, 255, 31, 0, 0],
            &[0, 0, 143, 255, 128, 79, 0, 0, 0],
            &[0, 0, 156, 255, 27, 0, 0, 0, 0],
            &[0, 0, 156, 255, 26, 0, 0, 0, 0],
            &[0, 0, 156, 255, 26, 0, 0, 0, 0],
            &[0, 0, 156, 255, 26, 0, 0, 0, 0],
            &[0, 0, 156, 255, 26, 0, 0, 0, 0],
            &[0, 0, 156, 255, 26, 0, 0, 0, 0],
            &[0, 0, 156, 255, 26, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
            &[0, 0, 0, 0, 0, 0, 0, 0, 0],
        ],
        _ => panic!("unsupported char"),
    }
}
