use crate::{messages::MessageFromWindowing, notification::Notification};
use egui::CtxRef;
use egui_wgpu_backend::epi::{App, Frame};
use std::sync::{Arc, Mutex};
use tokio::sync::{mpsc, oneshot};
use winit::event::WindowEvent;
use winit::platform::unix::{EventLoopExtUnix, MonitorHandleExtUnix, WindowBuilderExtUnix};
use winit::{
    event::Event,
    event_loop::{ControlFlow, EventLoop, EventLoopProxy, EventLoopWindowTarget},
    window::WindowId,
};

#[derive(Debug, Clone)]
pub enum UserEvent {
    Exit,
    DisplayNotification(Notification),
    RequestRedraw(WindowId),
    CloseNotification(u32),
}

struct CustomRepaintSignal(Mutex<EventLoopProxy<UserEvent>>, WindowId);

impl egui_wgpu_backend::epi::RepaintSignal for CustomRepaintSignal {
    fn request_repaint(&self) {
        self.0
            .lock()
            .unwrap()
            .send_event(UserEvent::RequestRedraw(self.1))
            .ok();
    }
}

pub fn run_window_loop(
    event_proxy_tx: oneshot::Sender<EventLoopProxy<UserEvent>>,
    control_tx: mpsc::Sender<MessageFromWindowing>,
) {
    let event_loop = EventLoop::new_any_thread();
    let proxy = event_loop.create_proxy();
    event_proxy_tx.send(proxy).unwrap();

    let instance = wgpu::Instance::new(wgpu::BackendBit::PRIMARY);
    let event_proxy = event_loop.create_proxy();

    let mut notifications = Vec::new();
    let mut notifs_to_remove = Vec::new();

    event_loop.run(move |event, window_target, control_flow| {
        match event {
            Event::UserEvent(ev) => match ev {
                UserEvent::Exit => {
                    *control_flow = ControlFlow::Exit;
                }
                UserEvent::DisplayNotification(notif) => {
                    match NotificationWindow::new(notif, window_target, &instance, &event_proxy) {
                        Ok(window) => {
                            notifications.push(window);
                        }
                        Err(e) => {
                            log::error!("Error building window for notification: {}", e);
                        }
                    }
                }
                UserEvent::RequestRedraw(window_id) => {
                    if let Some(window) = notifications
                        .iter()
                        .find(|w| w.backing_window.id() == window_id)
                    {
                        window.backing_window.request_redraw();
                    }
                }
                UserEvent::CloseNotification(notif_id) => {
                    if let Some(window) =
                        notifications.iter().find(|w| w.notification.id == notif_id)
                    {
                        notifs_to_remove.push(window.backing_window.id());
                    }
                }
            },
            Event::WindowEvent { event, window_id } => {
                if let Some(window) = notifications
                    .iter_mut()
                    .find(|w| w.backing_window.id() == window_id)
                {
                    let orig_event = Event::WindowEvent { event, window_id };
                    window.platform.handle_event::<()>(&orig_event);
                    let event = if let Event::WindowEvent { event, .. } = orig_event {
                        event
                    } else {
                        unreachable!()
                    };
                    match event {
                        WindowEvent::Resized(size) => {
                            log::debug!("Resized window to {:?}", size);
                            window.sc_desc.height = size.height;
                            window.sc_desc.width = size.width;
                            window.swap_chain = window
                                .device
                                .create_swap_chain(&window.surface, &window.sc_desc);
                        }
                        WindowEvent::CloseRequested => {
                            notifs_to_remove.push(window_id);
                        }
                        _ => {}
                    }
                }
            }
            Event::RedrawRequested(window_id) => {
                if let Some(window) = notifications
                    .iter_mut()
                    .find(|w| w.backing_window.id() == window_id)
                {
                    log::debug!("Requested window redraw");
                    window.render();
                }
            }
            _ => {}
        }
        for remove_id in notifs_to_remove.drain(..) {
            if let Some((index, _)) = notifications
                .iter()
                .enumerate()
                .find(|(_, w)| w.backing_window.id() == remove_id)
            {
                let window = notifications.remove(index);
                control_tx.blocking_send(MessageFromWindowing::NotificationClosed(
                    window.notification.id,
                ));
            }
        }
    });
}

struct NotificationWindow {
    notification: Notification,
    app: NotificationWindowApp,
    start_time: std::time::Instant,
    frame_time: Option<f32>,
    window_size: winit::dpi::PhysicalSize<u32>,
    backing_window: winit::window::Window,
    surface: wgpu::Surface,
    device: wgpu::Device,
    queue: wgpu::Queue,
    sc_desc: wgpu::SwapChainDescriptor,
    swap_chain: wgpu::SwapChain,
    repaint_signal: Arc<CustomRepaintSignal>,
    platform: egui_winit_platform::Platform,
    render_pass: egui_wgpu_backend::RenderPass,
}

impl NotificationWindow {
    fn new(
        notification: Notification,
        window_target: &EventLoopWindowTarget<UserEvent>,
        wgpu_instance: &wgpu::Instance,
        event_proxy: &EventLoopProxy<UserEvent>,
    ) -> anyhow::Result<Self> {
        let min_size = winit::dpi::PhysicalSize {
            width: 300,
            height: 50,
        };
        let max_size = winit::dpi::PhysicalSize {
            width: min_size.width,
            height: 500,
        };
        let window_builder = winit::window::WindowBuilder::new()
            .with_decorations(false)
            .with_resizable(false)
            .with_class("notification".into(), "notification".into())
            .with_transparent(true)
            .with_x11_window_type(vec![winit::platform::unix::XWindowType::Notification])
            .with_x11_screen(0)
            .with_min_inner_size(min_size)
            .with_max_inner_size(max_size)
            .with_inner_size(min_size)
            .with_visible(false);

        let backing_window = window_builder.build(window_target)?;

        if let Some(primary) = window_target.primary_monitor() {
            log::debug!("primary monitor: {:?}", primary);
            let mut position = primary.position();
            position.x += primary.size().width as i32;
            position.x -= 300;
            position.y += 42;
            log::debug!("final position: {:?}", position);
            backing_window.set_outer_position(position);
        } else {
            log::warn!("Failed to find primary monitor!");
        }

        let surface = unsafe { wgpu_instance.create_surface(&backing_window) };
        let adapter = futures_lite::future::block_on(wgpu_instance.request_adapter(
            &wgpu::RequestAdapterOptions {
                power_preference: wgpu::PowerPreference::LowPower,
                compatible_surface: Some(&surface),
            },
        ))
        .unwrap();
        let (device, queue) = futures_lite::future::block_on(adapter.request_device(
            &wgpu::DeviceDescriptor {
                features: wgpu::Features::default(),
                limits: wgpu::Limits::default(),
                label: None,
            },
            None,
        ))
        .unwrap();
        let size = backing_window.inner_size();
        let sc_desc = wgpu::SwapChainDescriptor {
            usage: wgpu::TextureUsage::RENDER_ATTACHMENT,
            format: wgpu::TextureFormat::Bgra8UnormSrgb,
            height: size.height,
            width: size.width,
            present_mode: wgpu::PresentMode::Fifo,
        };
        let swap_chain = device.create_swap_chain(&surface, &sc_desc);

        let repaint_signal = Arc::new(CustomRepaintSignal(
            Mutex::new(event_proxy.clone()),
            backing_window.id(),
        ));
        let platform =
            egui_winit_platform::Platform::new(egui_winit_platform::PlatformDescriptor {
                physical_width: size.width,
                physical_height: size.height,
                scale_factor: backing_window.scale_factor(),
                style: Default::default(),
                font_definitions: egui::FontDefinitions::default(),
            });
        let render_pass = egui_wgpu_backend::RenderPass::new(&device, sc_desc.format);

        let app = NotificationWindowApp::new(notification.clone());

        let notification_window = Self {
            notification,
            app,
            start_time: std::time::Instant::now(),
            frame_time: None,
            window_size: min_size,
            backing_window,
            surface,
            sc_desc,
            swap_chain,
            device,
            queue,
            repaint_signal,
            platform,
            render_pass,
        };

        Ok(notification_window)
    }

    fn render(&mut self) {
        self.platform
            .update_time(self.start_time.elapsed().as_secs_f64());
        let output_frame = match self.swap_chain.get_current_frame() {
            Ok(frame) => frame,
            Err(e) => {
                log::error!("Dropped frame: {}", e);
                return;
            }
        };

        let egui_start = std::time::Instant::now();
        self.platform.begin_frame();
        let mut app_output = egui_wgpu_backend::epi::backend::AppOutput::default();
        let mut frame = egui_wgpu_backend::epi::backend::FrameBuilder {
            repaint_signal: Arc::clone(&self.repaint_signal) as _,
            tex_allocator: &mut self.render_pass,
            info: egui_wgpu_backend::epi::IntegrationInfo {
                web_info: None,
                cpu_usage: self.frame_time,
                seconds_since_midnight: None,
                native_pixels_per_point: Some(self.backing_window.scale_factor() as _),
            },
            output: &mut app_output,
        }
        .build();

        self.app.update(&self.platform.context(), &mut frame);

        let (_output, paint_commands) = self.platform.end_frame();
        let paint_jobs = self.platform.context().tessellate(paint_commands);

        let frame_time = (std::time::Instant::now() - egui_start).as_secs_f64() as _;
        self.frame_time = Some(frame_time);

        let mut encoder = self
            .device
            .create_command_encoder(&wgpu::CommandEncoderDescriptor { label: None });
        let screen_descriptor = egui_wgpu_backend::ScreenDescriptor {
            physical_height: self.sc_desc.height,
            physical_width: self.sc_desc.width,
            scale_factor: self.backing_window.scale_factor() as _,
        };
        self.render_pass.update_texture(
            &self.device,
            &self.queue,
            &self.platform.context().texture(),
        );
        self.render_pass
            .update_user_textures(&self.device, &self.queue);
        self.render_pass
            .update_buffers(&self.device, &self.queue, &paint_jobs, &screen_descriptor);
        self.render_pass.execute(
            &mut encoder,
            &output_frame.output.view,
            &paint_jobs,
            &screen_descriptor,
            Some(wgpu::Color::TRANSPARENT),
        );

        self.queue.submit(std::iter::once(encoder.finish()));

        let result_size = self.platform.context().used_size();
        let new_size = winit::dpi::PhysicalSize {
            height: result_size.y as u32,
            width: result_size.x as u32,
        };
        let current_size = self.backing_window.inner_size();
        if current_size.width != new_size.width || current_size.height != new_size.height {
            log::debug!("Setting new inner size: {:?}", result_size);
            self.backing_window.set_inner_size(new_size);
            self.backing_window.request_redraw();
        } else {
            self.backing_window.set_visible(true);
        }
    }
}

struct NotificationWindowApp {
    notification: Notification,
}

impl NotificationWindowApp {
    fn new(notification: Notification) -> Self {
        Self { notification }
    }
}

impl egui_wgpu_backend::epi::App for NotificationWindowApp {
    fn update(&mut self, ctx: &CtxRef, frame: &mut Frame<'_>) {
        let mut panel_frame =
            egui::Frame::default().fill(egui::Color32::from_rgba_premultiplied(30, 30, 30, 127));
        panel_frame.corner_radius = 10.0;
        panel_frame.margin = egui::Vec2::new(5.0, 5.0);
        egui::CentralPanel::default()
            .frame(panel_frame)
            .show(ctx, |ui| {
                ui.label(egui::Label::new(&self.notification.summary).strong());
                ui.label(egui::Label::new(&self.notification.body));
                ui.with_layout(egui::Layout::right_to_left(), |ui| {
                    ui.label(egui::Label::new(&self.notification.app_name).small());
                });
            });
    }

    fn name(&self) -> &str {
        "notification-window"
    }
}
