mod notification;

use crate::{
    config::Config,
    messages::{MessageFromDBus, MessageToDBus},
};
use core::convert::TryFrom;
use futures_lite::stream::StreamExt;
use std::sync::Arc;
use tokio::sync::mpsc;
use zbus::export::names::{BusName, WellKnownName};
use zbus::Result;

pub type MessageSender = mpsc::Sender<MessageFromDBus>;
pub type MessageReceiver = mpsc::Receiver<MessageToDBus>;
pub use notification::{CloseReason, DBusNotification};

enum DBusLoopEvent {
    DBusMessage(zbus::Result<Arc<zbus::Message>>),
    ControlMessage(MessageToDBus),
}

pub async fn run_dbus(
    config: Arc<Config>,
    tx: MessageSender,
    mut rx: MessageReceiver,
) -> Result<()> {
    let connection = zbus::Connection::session()?;
    if let Some(name) = connection.unique_name() {
        log::debug!("Connected to dbus at {}", name.as_str());
    }
    let dbus_proxy = zbus::fdo::DBusProxy::new(&connection)?;
    dbus_proxy.connect_name_lost(|name: BusName| {
        log::info!("Lost name {}", name);
        Ok(())
    })?;
    dbus_proxy.connect_name_acquired(|name: BusName| {
        log::info!("Acquired name {}", name);
        Ok(())
    })?;
    let result = dbus_proxy.request_name(
        WellKnownName::try_from("org.freedesktop.Notifications").unwrap(),
        zbus::fdo::RequestNameFlags::AllowReplacement
            | zbus::fdo::RequestNameFlags::ReplaceExisting,
    )?;
    match result {
        zbus::fdo::RequestNameReply::PrimaryOwner | zbus::fdo::RequestNameReply::AlreadyOwner => {}
        zbus::fdo::RequestNameReply::Exists | zbus::fdo::RequestNameReply::InQueue => {
            log::debug!("Waiting for notifications name")
        }
    }
    let mut object_server = zbus::ObjectServer::new(&connection);
    let server_info = notification::ServerInformation {
        name: config.crate_setup.application_name().into(),
        vendor: "999eagle".into(),
        version: version::version!().into(),
        spec_version: "1.2".into(),
    };
    let notification_server = notification::NotificationServer::new(tx, server_info);
    object_server.at("/org/freedesktop/Notifications", notification_server)?;

    let mut async_connection = zbus::azync::Connection::from(connection);

    loop {
        let result = tokio::select! {
            Some(msg) = async_connection.next() => {
                DBusLoopEvent::DBusMessage(msg)
            }
            Some(msg) = rx.recv() => {
                DBusLoopEvent::ControlMessage(msg)
            }
        };
        match result {
            DBusLoopEvent::ControlMessage(msg) => match msg {
                MessageToDBus::Exit => {
                    break;
                }
                MessageToDBus::NotificationClosed(id, reason) => {
                    object_server.with(
                        "/org/freedesktop/Notifications",
                        |iface: &notification::NotificationServer| {
                            iface.emit_notification_closed(id, reason)
                        },
                    )?;
                }
                MessageToDBus::ActionInvoked(id, action_key) => {
                    object_server.with(
                        "/org/freedesktop/Notifications",
                        |iface: &notification::NotificationServer| {
                            iface.emit_action_invoked(id, &action_key)
                        },
                    )?;
                }
            },
            DBusLoopEvent::DBusMessage(msg) => match msg {
                Ok(msg) => {
                    if !dbus_proxy.handle_signal(&msg)? {
                        object_server.dispatch_message(&msg)?;
                    }
                }
                Err(e) => return Err(e),
            },
        }
    }

    Ok(())
}
