use std::fs::OpenOptions;
use std::io::{self, Cursor};
use anyhow::Result;
use bytes::Bytes;
use clap::{value_t, ArgMatches};
use log::{debug, info, error};
use rusoto_core::{Client, HttpClient, Region};
use rusoto_lambda::LambdaClient;
use sha2::{Sha512, Digest};
use tokio::runtime::Runtime;
use zstd::stream::decode_all;
use crate::{Artifact, Arch, Version, System, Target};
use super::invoke::invoke;
use super::notary::{Item, Select, Provider};
use super::server::{Command, Verify};
use super::upload::fetch;

pub fn exec(args: &ArgMatches, region: Region, role: Option<String>) -> Result<()> {
    let name    = value_t!(args, "name",    String)?;
    let version = value_t!(args, "version", Version)?;
    let arch    = value_t!(args, "arch",    Arch)?;
    let system  = value_t!(args, "system",  System)?;
    let binary  = value_t!(args, "binary",  String)?;

    let target  = Target::new(arch, system);
    let select  = Select { name, version, target };

    let provider  = Provider::new(region.clone(), role)?;
    let client    = Client::new_with(provider, HttpClient::new()?);
    let client    = LambdaClient::new_with_client(client, region);

    let runtime = Runtime::new()?;

    match runtime.block_on(verify(client, select, binary)) {
        Ok(()) => debug!("artifact valid"),
        Err(e) => error!("error: {}", e),
    };

    Ok(())
}

async fn verify(client: LambdaClient, select: Select, binary: String) -> Result<()> {
    let Item {
        artifact: Artifact { name, version, signature, .. },
        location,
    } = invoke(&client, Command::Select(select)).await?;

    info!("verifying {} {}", name, version);

    let blob  = fetch(&location).await?;
    let bytes = decode_all(&*blob)?;
    let hash  = Sha512::digest(&bytes);

    invoke(&client, Command::Verify(Verify {
        hash: Bytes::copy_from_slice(&hash[..]),
        sig:  signature.clone(),
    })).await?;

    debug!("signature: {}", base64::encode(&signature));

    let mut file  = OpenOptions::new().write(true).create_new(true).open(&binary)?;
    let mut bytes = Cursor::new(bytes);
    io::copy(&mut bytes, &mut file)?;

    Ok(())
}
