use anyhow::{anyhow, Result};
use bytes::Bytes;
use hyper::{Body, Client, Request};
use hyper::body::to_bytes;
use hyper::header::HeaderValue;
use hyper_rustls::HttpsConnectorBuilder;
use log::debug;

pub async fn upload(url: &str, body: Vec<u8>, signature: &Bytes, checksum: String) -> Result<()> {
    let https = HttpsConnectorBuilder::new()
            .with_native_roots()
            .https_or_http()
            .enable_http1()
            .enable_http2()
            .build();

    let client = Client::builder().build(https);

    let body      = Body::from(body);
    let checksum  = HeaderValue::from_str(&checksum)?;
    let signature = base64::encode(signature);
    let signature = HeaderValue::from_str(&signature)?;

    let mut req = Request::builder().method("PUT").uri(url).body(body)?;
    req.headers_mut().insert("content-md5", checksum);
    req.headers_mut().insert("x-amz-meta-signature", signature);

    let res = client.request(req).await?;

    if !res.status().is_success() {
        let body = to_bytes(res.into_body()).await?;
        let body = String::from_utf8(body.to_vec())?;
        return Err(anyhow!("error: {}", body));
    }

    Ok(())
}

pub async fn fetch(url: &str) -> Result<Bytes> {
    let https = HttpsConnectorBuilder::new()
            .with_native_roots()
            .https_or_http()
            .enable_http1()
            .enable_http2()
            .build();

    let client = Client::builder().build(https);

    let body   = Body::empty();
    let req    = Request::builder().method("GET").uri(url).body(body)?;
    let res    = client.request(req).await?;
    let status = res.status();

    let meta = match res.headers().get("x-amz-meta-signature") {
        Some(value) => value.to_str()?.to_owned(),
        None        => String::new(),
    };

    debug!("signature: {}", meta);

    let body = to_bytes(res.into_body()).await?;

    match status  {
        s if s.is_success() => Ok(body),
        s                   => Err(anyhow!("error: {}", s)),
    }
}
