use anyhow::Result;
use clap::{value_t, ArgMatches};
use log::{debug, error, info};
use rusoto_core::{Client, HttpClient, Region};
use rusoto_lambda::LambdaClient;
use tokio::runtime::Runtime;
use crate::Version;
use super::invoke::invoke;
use super::notary::{Update, Provider};
use super::server::Command;

pub fn exec(args: &ArgMatches, region: Region, role: Option<String>) -> Result<()> {
    let name    = value_t!(args, "name",    String)?;
    let version = value_t!(args, "version", Version)?;
    let release = args.is_present("release");

    let change  = Update {
        name:    name,
        version: version,
        release: release,
    };

    let provider = Provider::new(region.clone(), role)?;
    let client   = Client::new_with(provider, HttpClient::new()?);
    let client   = LambdaClient::new_with_client(client, region);

    let runtime = Runtime::new()?;

    match runtime.block_on(update(client, change)) {
        Ok(()) => debug!("update successful"),
        Err(e) => error!("update failed: {}", e),
    };

    Ok(())
}

async fn update(client: LambdaClient, change: Update) -> Result<()> {
    let Update { name, version, .. } = &change;
    info!("updating {} {}", name, version);
    invoke(&client, Command::Update(change)).await
}
