use std::fs;
use std::io::Cursor;
use std::time::SystemTime;
use anyhow::Result;
use clap::{value_t, ArgMatches};
use log::{debug, error, info};
use sha2::{Sha512, Digest};
use rusoto_core::{Client, HttpClient, Region};
use rusoto_lambda::LambdaClient;
use tokio::runtime::Runtime;
use zstd::stream::encode_all;
use crate::{Artifact, Arch, Version, System, Target};
use crate::time::Timestamp;
use super::invoke::invoke;
use super::notary::{Commit, Submit, Signed, Provider};
use super::server::Command;
use super::upload::upload;

pub fn exec(args: &ArgMatches, region: Region, role: Option<String>) -> Result<()> {
    let name    = value_t!(args, "name",    String)?;
    let version = value_t!(args, "version", Version)?;
    let arch    = value_t!(args, "arch",    Arch)?;
    let system  = value_t!(args, "system",  System)?;
    let binary  = value_t!(args, "binary",  String)?;

    let metadata  = fs::metadata(&binary)?;
    let timestamp = metadata.modified()?.duration_since(SystemTime::UNIX_EPOCH)?.as_secs();
    let binary    = fs::read(binary)?;

    let artifact = Artifact {
        name:      name,
        version:   version,
        target:    Target::new(arch, system),
        timestamp: Timestamp::new(timestamp),
        ..Default::default()
    };

    let provider  = Provider::new(region.clone(), role)?;
    let client    = Client::new_with(provider, HttpClient::new()?);
    let client    = LambdaClient::new_with_client(client, region);

    let runtime = Runtime::new()?;

    match runtime.block_on(submit(client, artifact, binary)) {
        Ok(()) => debug!("submit successful"),
        Err(e) => error!("submit failed: {}", e),
    };

    Ok(())
}

async fn submit(client: LambdaClient, mut artifact: Artifact, binary: Vec<u8>) -> Result<()> {
    info!("submitting {} {} to notary", artifact.name, artifact.version);

    let hash = Sha512::digest(&binary);
    let blob = encode_all(Cursor::new(binary), 6)?;
    let md5  = md5::compute(&blob);

    let checksum = base64::encode(&*md5);

    let signed: Signed = invoke(&client, Command::Submit(Submit {
        artifact: artifact.clone(),
        hash:     hash.to_vec(),
        checksum: checksum.clone(),
    })).await?;

    info!("uploading compressed object");
    upload(&signed.presigned, blob, &signed.signature, checksum).await?;

    artifact.signature = signed.signature;

    invoke(&client, Command::Commit(Commit {
        artifact: artifact,
        hash:     hash.to_vec(),
    })).await?;

    Ok(())
}
