use std::convert::TryFrom;
use std::fmt::Debug;
use std::env;
use anyhow::{anyhow, Error, Result};
use bytes::Bytes;
use ed25519_compact::Signature;
use lambda_runtime::{handler_fn, run, Context};
use log::error;
use rusoto_core::Region;
use serde::{Serialize, Deserialize};
use serde_json::Value;
use tokio::runtime::Runtime;
use tokio::task::JoinError;
use crate::Target;
use super::Notary;
use super::notary::{Commit, Search, Select, Submit, Update, SECRET_FN};

pub fn exec() -> Result<()> {
    let runtime = Runtime::new()?;
    let handler = handler_fn(wrap);
    runtime.block_on(run(handler)).map_err(|e| {
        anyhow!("request failed: {:?}", e)
    })
}

#[derive(Debug, Deserialize, Serialize)]
pub enum Command {
    Commit(Commit),
    Submit(Submit),
    Update(Update),
    Search(Search),
    Select(Select),
    Verify(Verify),
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Query {
    pub name:    String,
    pub version: String,
    pub arch:    String,
    pub system:  String,
    pub release: bool,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Verify {
    pub hash: Bytes,
    pub sig:  Bytes,
}

#[derive(Debug, Serialize, Deserialize)]
#[serde(untagged)]
pub enum Response<T> {
    Success(T),
    Failure(Failure),
}

#[derive(Debug, Serialize, Deserialize)]
#[serde(rename_all = "camelCase")]
pub struct Failure {
    error: String,
}

async fn wrap(request: Value, ctx: Context) -> Result<Response<Value>> {
    Ok(match tokio::spawn(handle(request, ctx)).await {
        Ok(Ok(v))  => success(v),
        Ok(Err(e)) => failure(e),
        Err(e)     => failure(e),
    })
}

async fn handle(request: Value, _ctx: Context) -> Result<Value> {
    let region = Region::default();
    let role   = env::var("AWS_ROLE_ARN").ok();
    let notary = Notary::new(region, role)?;

    if env::var("AWS_LAMBDA_FUNCTION_NAME")? != SECRET_FN {
        let query: Query = serde_json::from_value(request)?;
        let search = Search::try_from(query)?;
        return latest(search, notary).await;
    }

    Ok(match serde_json::from_value(request)? {
        Command::Commit(c) => commit(c, notary).await?,
        Command::Submit(s) => submit(s, notary).await?,
        Command::Update(u) => update(u, notary).await?,
        Command::Search(s) => search(s, notary).await?,
        Command::Select(s) => select(s, notary).await?,
        Command::Verify(v) => verify(v, notary).await?,
    })
}

async fn commit(c: Commit, notary: Notary) -> Result<Value> {
    notary.commit(c).await?;
    Ok(serde_json::to_value(())?)
}

async fn submit(s: Submit, notary: Notary) -> Result<Value> {
    let signed = notary.submit(s).await?;
    Ok(serde_json::to_value(&signed)?)
}

async fn update(u: Update, notary: Notary) -> Result<Value> {
    notary.update(u).await?;
    Ok(serde_json::to_value(())?)
}

async fn search(s: Search, notary: Notary) -> Result<Value> {
    let item = notary.search(s).await?;
    Ok(serde_json::to_value(&item)?)
}

async fn select(s: Select, notary: Notary) -> Result<Value> {
    let items = notary.select(s).await?;
    Ok(serde_json::to_value(&items)?)
}

async fn verify(v: Verify, notary: Notary) -> Result<Value> {
    let Verify { hash, sig } = v;

    let public = notary.public().await?;
    let sig    = Signature::from_slice(&sig)?;
    let result = public.verify(&hash, &sig)?;

    Ok(serde_json::to_value(&result)?)
}

async fn latest(search: Search, notary: Notary) -> Result<Value> {
    let items = notary.search(search).await?;
    Ok(serde_json::to_value(&items.get(0))?)
}

fn success(value: Value) -> Response<Value> {
    Response::Success(value)
}

fn failure<T: Into<Failure> + Debug>(error: T) -> Response<Value> {
    error!("{:?}", error);
    Response::Failure(error.into())
}

impl TryFrom<Query> for Search {
    type Error = Error;

    fn try_from(q: Query) -> Result<Self, Self::Error> {
        let Query { name, version, arch, system, release } = q;

        let version = version.parse()?;
        let arch    = arch.parse()?;
        let system  = system.parse()?;

        Ok(Search {
            name:    name,
            version: version,
            target:  Target::new(arch, system),
            release: release,
            limit:   Some(1),
        })
    }
}

impl From<Failure> for Error {
    fn from(f: Failure) -> Self {
        Error::msg(f.error)
    }
}

impl From<Error> for Failure {
    fn from(e: Error) -> Self {
        Self { error: e.to_string() }
    }
}

impl From<JoinError> for Failure {
    fn from(e: JoinError) -> Self {
        Self { error: e.to_string() }
    }
}
