use std::convert::TryFrom;
use anyhow::Result;
use clap::{value_t, ArgMatches};
use rusoto_core::{Client, HttpClient, Region};
use rusoto_lambda::LambdaClient;
use time::{format_description::parse, OffsetDateTime};
use tokio::runtime::Runtime;
use crate::{Artifact, Arch, Version, System, Target};
use super::invoke::invoke;
use super::notary::{Item, Search, Provider};
use super::server::Command;

pub fn exec(args: &ArgMatches, region: Region, role: Option<String>) -> Result<()> {
    let name    = value_t!(args, "name",    String)?;
    let version = value_t!(args, "version", Version)?;
    let arch    = value_t!(args, "arch",    Arch)?;
    let system  = value_t!(args, "system",  System)?;
    let limit   = value_t!(args, "limit",   i64)?;
    let release = args.is_present("release");

    let query = Search {
        name:    name,
        version: version,
        release: release,
        target:  Target::new(arch, system),
        limit:   Some(limit),
    };

    let provider = Provider::new(region.clone(), role)?;
    let client   = Client::new_with(provider, HttpClient::new()?);
    let client   = LambdaClient::new_with_client(client, region);

    let runtime = Runtime::new()?;

    Ok(runtime.block_on(search(client, query))?)
}

async fn search(client: LambdaClient, query: Search) -> Result<()> {
    let name    = query.name.clone();
    let version = query.version.clone();

    let command = Command::Search(query);
    let items   = invoke::<Vec<Item>>(&client, command).await?;

    println!("{} updates > {}:", name, version);

    let format = parse("[year]-[month]-[day] [hour]:[minute]")?;

    for item in items {
        let Artifact { version, timestamp, .. } = item.artifact;
        let timestamp = i64::try_from(timestamp.as_secs())?;
        let timestamp = OffsetDateTime::from_unix_timestamp(timestamp)?;
        let timestamp = timestamp.format(&format)?;
        println!("  {} {}", version, timestamp);
    }

    Ok(())
}
