use std::collections::HashMap;
use std::time::Duration;
use anyhow::Result;
use rusoto_core::{Client, Region};
use rusoto_credential::ProvideAwsCredentials;
use rusoto_s3::{*, util::{PreSignedRequest, PreSignedRequestOption}};
use crate::Artifact;
use crate::notary::notary::Provider;

pub const ARTIFACTS: &str = "kentik-notary-artifacts";

pub struct ObjectStore {
    provider: Provider,
    region:   Region,
    store:    S3Client,
}

impl ObjectStore {
    pub fn new(client: Client, provider: Provider, region: Region) -> Result<Self> {
        let store = S3Client::new_with_client(client, region.clone());
        Ok(Self { provider, region, store })
    }

    pub async fn init(&self) -> Result<()> {
        Ok(self.create(ARTIFACTS).await?)
    }

    pub async fn create(&self, name: &str) -> Result<()> {
        let region = self.region.name().to_owned();
        self.store.create_bucket(CreateBucketRequest {
            acl:                         Some("private".to_owned()),
            bucket:                      name.to_owned(),
            create_bucket_configuration: Some(CreateBucketConfiguration {
                location_constraint: Some(region),
            }),
            ..Default::default()
        }).await?;
        Ok(())
    }

    pub async fn get(&self, artifact: &Artifact) -> Result<GetObjectOutput> {
        Ok(self.store.get_object(GetObjectRequest {
            bucket: ARTIFACTS.to_owned(),
            key:    key(artifact),
            ..Default::default()
        }).await?)
    }

    pub async fn put(&self, artifact: &Artifact, body: Vec<u8>) -> Result<PutObjectOutput> {
        Ok(self.store.put_object(PutObjectRequest {
            bucket:   ARTIFACTS.to_owned(),
            key:      key(artifact),
            metadata: Some(meta(artifact)),
            body:     Some(body.into()),
            ..Default::default()
        }).await?)
    }

    pub async fn presign(&self, artifact: &Artifact, md5: String) -> Result<String> {
        let opts = PreSignedRequestOption {
            expires_in: Duration::from_secs(60 * 5),
        };

        let credentials = self.provider.credentials().await?;

        Ok(PutObjectRequest {
            bucket:      ARTIFACTS.to_owned(),
            key:         key(artifact),
            metadata:    Some(meta(artifact)),
            content_md5: Some(md5),
            ..Default::default()
        }.get_presigned_url(&self.region, &credentials, &opts))
    }

    pub async fn location(&self, artifact: &Artifact) -> Result<String> {
        let opts = PreSignedRequestOption {
            expires_in: Duration::from_secs(60 * 5),
        };

        let credentials = self.provider.credentials().await?;

        Ok(GetObjectRequest {
            bucket: ARTIFACTS.to_owned(),
            key:    key(artifact),
            ..Default::default()
        }.get_presigned_url(&self.region, &credentials, &opts))
    }

    pub async fn list(&self) -> Result<ListBucketsOutput> {
        Ok(self.store.list_buckets().await?)
    }
}

fn key(artifact: &Artifact) -> String {
    format!("{}/{}", artifact.artifact(), artifact.version)
}

fn meta(artifact: &Artifact) -> HashMap<String, String> {
    let signature = base64::encode(&artifact.signature);
    let mut meta  = HashMap::new();
    meta.insert("signature".to_owned(), signature);
    meta
}
