use anyhow::Result;
use bytes::Bytes;
use ed25519_compact::{KeyPair, Seed, Signature};
use rusoto_core::{Client, Region};
use rusoto_kms::*;
use crate::notary::Key;

pub struct KeyStore {
    store: KmsClient,
}

impl KeyStore {
    pub fn new(client: Client, region: Region) -> Result<Self> {
        let store = KmsClient::new_with_client(client, region);
        Ok(Self { store })
    }

    pub async fn init(&self) -> Result<KeyMetadata> {
        let desc  = "notary kek";
        let alias = "alias/notary/kek";

        let metadata = self.store.create_key(CreateKeyRequest {
            description: Some(desc.to_owned()),
            key_usage:   Some("ENCRYPT_DECRYPT".to_owned()),
            ..Default::default()
        }).await?.key_metadata.unwrap_or_default();

        self.store.create_alias(CreateAliasRequest {
            alias_name:    alias.to_owned(),
            target_key_id: metadata.key_id.clone(),
        }).await?;

        Ok(metadata)
    }

    pub async fn generate(&self, id: &str) -> Result<Key> {
        let seed   = Seed::generate();
        let keys   = KeyPair::from_seed(Seed::from_slice(&seed[..])?);

        let kms_id = id.to_owned();
        let sealed = self.encrypt(id, &keys[..]).await?;
        let seed   = self.encrypt(id, &seed[..]).await?;
        let public = keys.pk;

        Ok(Key { kms_id, public, sealed, seed })
    }

    pub async fn sign(&self, key: &Key, msg: &[u8]) -> Result<Signature> {
        let bytes = self.decrypt(&key.kms_id, &key.sealed).await?;
        let keys  = KeyPair::from_slice(&bytes)?;
        Ok(keys.sk.sign(msg, None))
    }

    pub async fn verify(&self, key: &Key, msg: &[u8], sig: &Signature) -> Result<()> {
        let bytes = self.decrypt(&key.kms_id, &key.sealed).await?;
        let keys  = KeyPair::from_slice(&bytes)?;
        Ok(keys.pk.verify(msg, sig)?)
    }

    pub async fn lookup(&self, id: &str) -> Result<KeyMetadata> {
        Ok(self.store.describe_key(DescribeKeyRequest {
            key_id: id.to_owned(),
            ..Default::default()
        }).await?.key_metadata.unwrap_or_default())
    }

    async fn decrypt(&self, id: &str, data: &[u8]) -> Result<Bytes> {
        Ok(self.store.decrypt(DecryptRequest {
            key_id:          Some(id.to_owned()),
            ciphertext_blob: Bytes::copy_from_slice(data),
            ..Default::default()
        }).await?.plaintext.unwrap_or_default())
    }

    async fn encrypt(&self, id: &str, data: &[u8]) -> Result<Bytes> {
        Ok(self.store.encrypt(EncryptRequest {
            key_id:    id.to_owned(),
            plaintext: Bytes::copy_from_slice(data),
            ..Default::default()
        }).await?.ciphertext_blob.unwrap_or_default())
    }
}
