use anyhow::Result;
use bytes::Bytes;
use log::debug;
use rusoto_core::{Client, Region};
use rusoto_iam::*;
use rusoto_lambda::*;

pub const PUBLIC_FN:   &str = "NotaryPublicFn";
pub const SECRET_FN:   &str = "NotarySecretFn";

pub const PUBLIC_ROLE: &str = "NotaryPublicRole";
pub const SECRET_ROLE: &str = "NotarySecretRole";

pub struct FuncStore {
    store: LambdaClient,
    roles: IamClient,
}

impl FuncStore {
    pub fn new(client: Client, region: Region) -> Result<Self> {
        let store = LambdaClient::new_with_client(client.clone(), region);
        let roles = IamClient::new_with_client(client, Region::UsEast1);
        Ok(Self { store, roles })
    }

    pub async fn init(&self, code: &Bytes) -> Result<(FunctionConfiguration, FunctionConfiguration)> {
        let public = self.create(PUBLIC_FN, PUBLIC_ROLE, code).await?;
        let secret = self.create(SECRET_FN, SECRET_ROLE, code).await?;
        Ok((public, secret))
    }

    async fn create(&self, name: &str, role: &str, code: &Bytes) -> Result<FunctionConfiguration> {
        let code = FunctionCode {
            zip_file: Some(code.clone()),
            ..Default::default()
        };

        let role = self.role(role).await?;

        debug!("creating {}, role: {}", name, role.arn);

        Ok(self.store.create_function(CreateFunctionRequest {
            function_name: name.to_owned(),
            code:          code,
            role:          role.arn.to_owned(),
            publish:       Some(true),
            timeout:       Some(60),
            handler:       Some("any".to_owned()),
            runtime:       Some("provided".to_owned()),
            ..Default::default()
        }).await?)
    }

    pub async fn update(&self, name: &str, code: &Bytes) -> Result<FunctionConfiguration> {
        Ok(self.store.update_function_code(UpdateFunctionCodeRequest {
            function_name: name.to_owned(),
            zip_file:      Some(code.clone()),
            publish:       Some(true),
            ..Default::default()
        }).await?)
    }

    async fn role(&self, name: &str) -> Result<Role> {
        Ok(self.roles.get_role(GetRoleRequest {
            role_name: name.to_owned(),
        }).await?.role)
    }
}
