use std::collections::HashMap;
use anyhow::{anyhow, Error, Result};
use bytes::Bytes;
use ed25519_compact::PublicKey;
use rusoto_dynamodb::*;
use crate::{Arch, Artifact, System, Target, Version};
use crate::notary::Key;
use crate::time::Timestamp;
use super::schema::{DATA, META};

pub trait Get: Sized {
    fn get(item: HashMap<String, AttributeValue>) -> Result<Self>;
}

pub trait Put {
    fn put(&self) -> Result<PutItemInput>;
}

impl Get for Artifact {
    fn get(item: HashMap<String, AttributeValue>) -> Result<Self> {
        let mut values = Values(item);

        let name      = values.get("name")?;
        let version   = values.get("version")?;
        let arch      = values.get("arch")?;
        let system    = values.get("system")?;
        let signature = values.get("signature")?;
        let timestamp = values.get("timestamp")?;
        let timestamp = Timestamp::new(timestamp);
        let target    = Target::new(arch, system);

        Ok(Self { name, version, target, signature, timestamp })
    }
}

impl Put for Artifact {
    fn put(&self) -> Result<PutItemInput>  {
        let mut values = Values::default();

        let artifact = self.artifact();
        let revision = self.revision();

        values.put("artifact",  artifact);
        values.put("revision",  revision);
        values.put("name",      self.name.clone());
        values.put("version",   self.version);
        values.put("arch",      self.target.arch);
        values.put("system",    self.target.system);
        values.put("signature", self.signature.clone());
        values.put("timestamp", self.timestamp.as_secs());

        Ok(PutItemInput {
            table_name: DATA.to_owned(),
            item:       values.into(),
            ..Default::default()
        })
    }
}

impl Get for Key {
    fn get(item: HashMap<String, AttributeValue>) -> Result<Self> {
        let mut values = Values(item);

        let kms_id = values.get("kms_id")?;
        let public = values.get("public")?;
        let sealed = values.get("sealed")?;
        let seed   = values.get("seed")?;

        Ok(Self { kms_id, public, sealed, seed })
    }
}

impl Put for Key {
    fn put(&self) -> Result<PutItemInput> {
        let mut values = Values::default();

        values.put("id",     "key".to_string());
        values.put("kms_id", self.kms_id.clone());
        values.put("public", self.public);
        values.put("sealed", self.sealed.clone());
        values.put("seed",   self.seed.clone());

        Ok(PutItemInput {
            table_name: META.to_owned(),
            item:       values.into(),
            ..Default::default()
        })
    }
}

#[derive(Default)]
pub struct Names(HashMap<String, String>);

#[derive(Default)]
pub struct Values(HashMap<String, AttributeValue>);

pub trait Value: Sized {
    fn from(av: AttributeValue) -> Option<Result<Self>>;
    fn into(self) -> AttributeValue;
}

impl Names {
    pub fn put(&mut self, var: &str, name: &str)  {
        self.0.insert(var.to_owned(), name.into());
    }
}

impl Values {
    pub fn get<T: Value>(&mut self, name: &str) -> Result<T> {
        match self.0.remove(name).and_then(Value::from) {
            Some(Ok(value)) => Ok(value),
            Some(Err(e))    => Err(e),
            None            => Err(anyhow!("missing {}", name)),
        }
    }

    pub fn put<T: Value>(&mut self, name: &str, value: T) {
        self.0.insert(name.to_owned(), value.into());
    }
}

impl Value for bool {
    fn from(av: AttributeValue) -> Option<Result<Self>> {
        av.bool.map(Ok)
    }

    fn into(self) -> AttributeValue {
        AttributeValue {
            bool: Some(self),
            ..Default::default()
        }
    }
}

impl Value for i64 {
    fn from(av: AttributeValue) -> Option<Result<Self>> {
        Some(av.n?.parse().map_err(Error::from))
    }

    fn into(self) -> AttributeValue {
        AttributeValue {
            n: Some(self.to_string()),
            ..Default::default()
        }
    }
}

impl Value for u64 {
    fn from(av: AttributeValue) -> Option<Result<Self>> {
        Some(av.n?.parse().map_err(Error::from))
    }

    fn into(self) -> AttributeValue {
        AttributeValue {
            n: Some(self.to_string()),
            ..Default::default()
        }
    }
}

impl Value for String {
    fn from(av: AttributeValue) -> Option<Result<Self>> {
        av.s.map(Ok)
    }

    fn into(self) -> AttributeValue {
        AttributeValue {
            s: Some(self),
            ..Default::default()
        }
    }
}

impl Value for Bytes {
    fn from(av: AttributeValue) -> Option<Result<Self>> {
        av.b.map(Ok)
    }

    fn into(self) -> AttributeValue {
        AttributeValue {
            b: Some(self),
            ..Default::default()
        }
    }
}

impl Value for PublicKey {
    fn from(av: AttributeValue) -> Option<Result<Self>> {
        Some(Self::from_slice(&av.b?).map_err(Error::from))
    }

    fn into(self) -> AttributeValue {
        Value::into(Bytes::copy_from_slice(&self[..]))
    }
}

impl Value for Arch {
    fn from(av: AttributeValue) -> Option<Result<Self>> {
        Some(av.s?.parse())
    }

    fn into(self) -> AttributeValue {
        AttributeValue {
            s: Some(self.to_string()),
            ..Default::default()
        }
    }
}

impl Value for System {
    fn from(av: AttributeValue) -> Option<Result<Self>> {
        Some(av.s?.parse())
    }

    fn into(self) -> AttributeValue {
        AttributeValue {
            s: Some(self.to_string()),
            ..Default::default()
        }
    }
}

impl Value for Version {
    fn from(av: AttributeValue) -> Option<Result<Self>> {
        Some(av.s?.parse())
    }

    fn into(self) -> AttributeValue {
        AttributeValue {
            s: Some(self.to_string()),
            ..Default::default()
        }
    }
}

impl From<Names> for HashMap<String, String> {
    fn from(names: Names) -> Self {
        names.0
    }
}

impl From<Values> for HashMap<String, AttributeValue> {
    fn from(values: Values) -> Self {
        values.0
    }
}
