use std::collections::HashMap;
use anyhow::Result;
use bytes::Bytes;
use ed25519_compact::{KeyPair, Seed};
use rusoto_dynamodb::AttributeValue;
use crate::{Artifact, notary::Key};
use super::change::{Change, ChangeSet};
use super::schema::{DATA, META};
use super::wire::{Get, Put, Names, Values};

#[test]
fn artifact_get() -> Result<()> {
    let expected = Artifact {
        name:      "test".to_owned(),
        version:   "1.0.0".parse()?,
        ..Default::default()
    };

    let input  = expected.put()?;
    let actual = Artifact::get(input.item)?;

    assert_eq!(expected, actual);

    Ok(())
}

#[test]
fn artifact_put() -> Result<()> {
    let artifact = Artifact {
        name:      "test".to_owned(),
        version:   "1.0.0".parse()?,
        ..Default::default()
    };

    let input = artifact.put()?;

    assert_eq!(input.item["artifact"], AttributeValue {
        s: Some(artifact.artifact()),
        ..Default::default()
    });

    assert_eq!(input.item["revision"], AttributeValue {
        n: Some(artifact.revision().to_string()),
        ..Default::default()
    });

    assert_eq!(input.item["name"], AttributeValue {
        s: Some(artifact.name.clone()),
        ..Default::default()
    });

    assert_eq!(input.item["version"], AttributeValue {
        s: Some(artifact.version.to_string()),
        ..Default::default()
    });

    assert_eq!(input.item["arch"], AttributeValue {
        s: Some(artifact.target.arch.to_string()),
        ..Default::default()
    });

    assert_eq!(input.item["system"], AttributeValue {
        s: Some(artifact.target.system.to_string()),
        ..Default::default()
    });

    assert_eq!(input.item["signature"], AttributeValue {
        b: Some(artifact.signature.clone()),
        ..Default::default()
    });

    assert_eq!(input.item["timestamp"], AttributeValue {
        n: Some(artifact.timestamp.to_string()),
        ..Default::default()
    });

    assert_eq!(&input.table_name, DATA);

    Ok(())
}

#[test]
fn key_get() -> Result<()> {
    let seed   = Seed::generate();
    let pair   = KeyPair::from_seed(Seed::from_slice(&seed[..])?);
    let public = pair.pk;
    let secret = pair.sk;

    let expected = Key {
        kms_id: "test".to_owned(),
        public: public,
        sealed: Bytes::copy_from_slice(&secret[..]),
        seed:   Bytes::copy_from_slice(&seed[..]),
    };

    let input  = expected.put()?;
    let actual = Key::get(input.item)?;

    assert_eq!(expected, actual);

    Ok(())
}

#[test]
fn key_put() -> Result<()> {
    let seed   = Seed::generate();
    let pair   = KeyPair::from_seed(Seed::from_slice(&seed[..])?);
    let public = pair.pk;
    let secret = pair.sk;

    let key = Key {
        kms_id: "test".to_owned(),
        public: public,
        sealed: Bytes::copy_from_slice(&secret[..]),
        seed:   Bytes::copy_from_slice(&seed[..]),
    };

    let input = key.put()?;

    assert_eq!(input.item["id"], AttributeValue {
        s: Some("key".to_owned()),
        ..Default::default()
    });

    assert_eq!(input.item["public"], AttributeValue {
        b: Some(Bytes::copy_from_slice(&public[..])),
        ..Default::default()
    });

    assert_eq!(input.item["sealed"], AttributeValue {
        b: Some(Bytes::copy_from_slice(&secret[..])),
        ..Default::default()
    });

    assert_eq!(input.item["seed"], AttributeValue {
        b: Some(Bytes::copy_from_slice(&seed[..])),
        ..Default::default()
    });

    assert_eq!(&input.table_name, META);

    Ok(())
}

#[test]
fn changeset_empty() -> Result<()> {
    let mut key = Values::default();
    key.put("key", 42u64);

    let table = "Table";
    let key   = HashMap::from(key);
    let set   = ChangeSet::new();

    let update = set.update(table, key.clone());

    assert_eq!(table.to_owned(), update.table_name);
    assert_eq!(key,              update.key);

    assert_eq!(None, update.update_expression);
    assert_eq!(None, update.expression_attribute_names);
    assert_eq!(None, update.expression_attribute_values);

    Ok(())
}

#[test]
fn changeset_set_one() -> Result<()> {
    let table = "Table";
    let key   = HashMap::new();
    let foo   = 42u64;

    let mut names = Names::default();
    names.put("#foo", "foo");

    let mut values = Values::default();
    values.put(":foo", foo);

    let expression = "SET #foo = :foo";

    let mut set = ChangeSet::new();
    set.add("foo", Change::set(foo));
    let update = set.update(table, key);

    assert_eq!(Some(expression.to_owned()), update.update_expression);
    assert_eq!(Some(HashMap::from(names)),  update.expression_attribute_names);
    assert_eq!(Some(HashMap::from(values)), update.expression_attribute_values);

    Ok(())
}

#[test]
fn changeset_set_many() -> Result<()> {
    let table = "Table";
    let key   = HashMap::new();
    let foo   = 42u64;
    let bar   = true;

    let mut names = Names::default();
    names.put("#foo", "foo");
    names.put("#bar", "bar");

    let mut values = Values::default();
    values.put(":foo", foo);
    values.put(":bar", bar);

    let expression = "SET #bar = :bar, #foo = :foo";

    let mut set = ChangeSet::new();
    set.add("foo", Change::set(foo));
    set.add("bar", Change::set(bar));
    let update = set.update(table, key);

    assert_eq!(Some(expression.to_owned()), update.update_expression);
    assert_eq!(Some(HashMap::from(names)),  update.expression_attribute_names);
    assert_eq!(Some(HashMap::from(values)), update.expression_attribute_values);

    Ok(())
}

#[test]
fn changeset_remove_one() -> Result<()> {
    let table = "Table";
    let key   = HashMap::new();

    let mut names = Names::default();
    names.put("#foo", "foo");

    let expression = "REMOVE #foo";

    let mut set = ChangeSet::new();
    set.add("foo", Change::remove());
    let update = set.update(table, key);

    assert_eq!(Some(expression.to_owned()), update.update_expression);
    assert_eq!(Some(HashMap::from(names)),  update.expression_attribute_names);
    assert_eq!(None,                        update.expression_attribute_values);

    Ok(())
}

#[test]
fn changeset_set_remove() -> Result<()> {
    let table = "Table";
    let key   = HashMap::new();
    let foo   = 42u64;

    let mut names = Names::default();
    names.put("#foo", "foo");
    names.put("#bar", "bar");

    let mut values = Values::default();
    values.put(":foo", foo);

    let expression = "SET #foo = :foo REMOVE #bar";

    let mut set = ChangeSet::new();
    set.add("foo", Change::set(foo));
    set.add("bar", Change::remove());
    let update = set.update(table, key);

    assert_eq!(Some(expression.to_owned()), update.update_expression);
    assert_eq!(Some(HashMap::from(names)),  update.expression_attribute_names);
    assert_eq!(Some(HashMap::from(values)), update.expression_attribute_values);

    Ok(())
}
