use std::collections::{BTreeMap, BTreeSet, HashMap};
use rusoto_dynamodb::{AttributeValue, UpdateItemInput};
use super::wire::Value;

#[derive(Debug)]
pub enum Change {
    Set(AttributeValue),
    Remove,
}

#[derive(Default)]
pub struct ChangeSet {
    names:  HashMap<String, String>,
    values: HashMap<String, AttributeValue>,
    set:    BTreeMap<String, String>,
    remove: BTreeSet<String>,
}

impl Change {
    pub fn set<V: Value>(value: V) -> Self {
        Self::Set(value.into())
    }

    pub fn remove() -> Self {
        Self::Remove
    }
}

impl ChangeSet {
    pub fn new() -> Self {
        Self::default()
    }

    pub fn add(&mut self, name: &str, change: Change) {
        match change {
            Change::Set(v) => self.set(name, v),
            Change::Remove => self.remove(name),
        }
    }

    pub fn update(self, table: &str, key: HashMap<String, AttributeValue>) -> UpdateItemInput {
        let mut update = Vec::new();

        if !self.set.is_empty() {
            let updates = self.set.into_iter().map(|(name, value)| {
                format!("{} = {}", name, value)
            }).collect::<Vec<_>>().join(", ");
            update.push(format!("SET {}", updates));
        }

        if !self.remove.is_empty() {
            let updates = self.remove.into_iter().map(|name| {
                name
            }).collect::<Vec<_>>().join(", ");
            update.push(format!("REMOVE {}", updates));
        }

        UpdateItemInput {
            table_name:                  table.to_owned(),
            key:                         key,
            update_expression:           update.join(" ").optional(),
            expression_attribute_names:  self.names.optional(),
            expression_attribute_values: self.values.optional(),
            ..Default::default()
        }
    }

    fn set(&mut self, name: &str, value: AttributeValue) {
        let value = self.value(name, value);
        let name  = self.name(name);
        self.set.insert(name, value);
    }

    fn remove(&mut self, name: &str) {
        let name = self.name(name);
        self.remove.insert(name);
    }

    fn name(&mut self, name: &str) -> String {
        let placeholder = format!("#{}", name);
        self.names.insert(placeholder.clone(), name.to_owned());
        placeholder
    }

    fn value(&mut self, name: &str, value: AttributeValue) -> String {
        let placeholder = format!(":{}", name);
        self.values.insert(placeholder.clone(), value);
        placeholder
    }
}

trait Optional: Sized {
    fn optional(self) -> Option<Self>;
}

impl Optional for String {
    fn optional(self) -> Option<Self> {
        match self {
            s if !s.is_empty() => Some(s),
            _                  => None,
        }
    }
}

impl<K, V> Optional for HashMap<K, V> {
    fn optional(self) -> Option<Self> {
        match self {
            c if !c.is_empty() => Some(c),
            _                  => None,
        }
    }
}
