use std::collections::HashMap;
use std::time::Duration;
use anyhow::{anyhow, Result};
use log::debug;
use rusoto_core::{Client, Region};
use rusoto_dynamodb::*;
use serde::{Deserialize, Serialize};
use tokio::time::sleep;
use crate::{Artifact, Target, Version};
use crate::notary::Key;
use super::db::{Get, Put, Names, Values};
use super::db::change::{Change, ChangeSet};
use super::db::schema::{self, DATA, META, INDEX};

pub struct DataStore {
    store: DynamoDbClient,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Search {
    pub name:    String,
    pub version: Version,
    pub target:  Target,
    pub release: bool,
    pub limit:   Option<i64>,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Select {
    pub name:    String,
    pub version: Version,
    pub target:  Target,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Update {
    pub name:    String,
    pub version: Version,
    pub release: bool,
}

impl DataStore {
    pub fn new(client: Client, region: Region) -> Result<Self> {
        let store = DynamoDbClient::new_with_client(client, region);
        Ok(Self { store })
    }

    pub async fn init(&self) -> Result<(TableDescription, TableDescription)> {
        let data = self.create(schema::data()).await?;
        let meta = self.create(schema::meta()).await?;
        Ok((data, meta))
    }

    pub async fn key(&self) -> Result<Key> {
        let mut key = Values::default();
        key.put("id", "key".to_string());

        self.get(GetItemInput {
            table_name: META.to_owned(),
            key:        key.into(),
            ..Default::default()
        }).await
    }

    pub async fn search(&self, search: Search) -> Result<Vec<Artifact>> {
        let Search { name, version, target, release, limit } = search;

        let artifact = Artifact::new(name, version, target);

        let mut names  = Names::default();
        let mut values = Values::default();
        let mut index  = None;

        let query = "artifact = :artifact and #index > :revision";

        values.put(":artifact", artifact.artifact());
        values.put(":revision", artifact.revision());

        if release {
            index = Some(INDEX.to_owned());
            names.put("#index", "released");
        } else {
            names.put("#index", "revision");
        }

        let (items, _) = self.query(QueryInput {
            table_name:                  DATA.to_owned(),
            index_name:                  index,
            key_condition_expression:    Some(query.into()),
            expression_attribute_names:  Some(names.into()),
            expression_attribute_values: Some(values.into()),
            scan_index_forward:          Some(false),
            limit:                       limit,
            ..Default::default()
        }).await?;

        Ok(items)
    }

    pub async fn select(&self, select: Select) -> Result<Artifact> {
        let Select { name, version, target } = select;

        let artifact = Artifact::new(name, version, target);

        let mut key = Values::default();
        key.put("artifact", artifact.artifact());
        key.put("revision", artifact.revision());

        self.get(GetItemInput {
            table_name: DATA.to_owned(),
            key:        key.into(),
            ..Default::default()
        }).await
    }

    pub async fn update(&self, update: Update) -> Result<()> {
        let select = update.select();
        let output = self.store.scan(select).await?;
        for key in output.items.unwrap_or_default() {
            let update = update.update(key);
            self.store.update_item(update).await?;
        }
        Ok(())
    }

    pub async fn query<G: Get>(&self, query: QueryInput) -> Result<(Vec<G>, QueryOutput)> {
        let mut output = self.store.query(query).await?;
        let items = output.items.take().unwrap_or_default().into_iter();
        let items = items.map(G::get).collect::<Result<Vec<_>>>()?;
        Ok((items, output))
    }

    pub async fn get<G: Get>(&self, get: GetItemInput) -> Result<G> {
        match self.store.get_item(get).await?.item {
            Some(item) => G::get(item),
            None       => Err(anyhow!("missing item")),
        }
    }

    pub async fn put<P: Put>(&self, item: &P) -> Result<()> {
        self.store.put_item(item.put()?).await?;
        Ok(())
    }

    pub async fn create(&self, table: CreateTableInput) -> Result<TableDescription> {
        let name = table.table_name.clone();
        let desc = self.store.create_table(table).await?.table_description;

        let mut table = desc.unwrap_or_default();

        let active = Some("ACTIVE".to_owned());
        let delay  = Duration::from_secs(10);

        while table.table_status != active {
            debug!("waiting for {} activation", name);
            sleep(delay).await;
            table = self.lookup(&name).await?;
        }

        Ok(table)
    }

    pub async fn lookup(&self, name: &str) -> Result<TableDescription> {
        Ok(self.store.describe_table(DescribeTableInput {
            table_name: name.to_owned()
        }).await?.table.unwrap_or_default())
    }

    pub async fn list(&self) -> Result<ListTablesOutput> {
        Ok(self.store.list_tables(Default::default()).await?)
    }
}

impl Update {
    fn select(&self) -> ScanInput  {
        let filter = "begins_with(artifact, :artifact) and revision = :revision";

        let mut values = Values::default();
        values.put(":artifact", format!("{}/", self.name));
        values.put(":revision", u64::from(self.version));

        let select = "artifact, revision";

        ScanInput {
            table_name:                  DATA.to_owned(),
            expression_attribute_values: Some(values.into()),
            filter_expression:           Some(filter.into()),
            projection_expression:       Some(select.into()),
            ..Default::default()
        }
    }

    fn update(&self, key: HashMap<String, AttributeValue>) -> UpdateItemInput {
        let Self { version, release, .. } = *self;

        let mut change = ChangeSet::new();
        change.add("released", match release {
            true  => Change::set(u64::from(version)),
            false => Change::remove(),
        });

        change.update(DATA, key)
    }
}
