use std::sync::Arc;
use anyhow::Error;
use async_trait::async_trait;
use rusoto_core::Region;
use rusoto_credential::*;
use rusoto_sts::{StsAssumeRoleSessionCredentialsProvider, StsClient};

#[derive(Clone)]
pub enum Provider {
    Assume(Arc<AutoRefreshingProvider<StsAssumeRoleSessionCredentialsProvider>>),
    Static(Arc<ChainProvider>),
}

impl Provider {
    pub fn new(region: Region, role: Option<String>) -> Result<Self, Error> {
        let provider = role.map(|role| {
            AutoRefreshingProvider::new(
                StsAssumeRoleSessionCredentialsProvider::new(
                    StsClient::new(region),
                    role,
                    "default".to_owned(),
                    None,
                    None,
                    None,
                    None,
                )
            )
        }).transpose()?;

        Ok(match provider {
            Some(provider) => Provider::Assume(Arc::new(provider)),
            None           => Provider::Static(Arc::new(ChainProvider::new())),
        })
    }
}

#[async_trait]
impl ProvideAwsCredentials for Provider {
    async fn credentials(&self) -> Result<AwsCredentials, CredentialsError> {
        match self {
            Provider::Assume(p) => p.credentials().await,
            Provider::Static(p) => p.credentials().await,
        }
    }
}
