use std::collections::HashMap;
use anyhow::{anyhow, Result};
use bytes::Bytes;
use ed25519_compact::{PublicKey, Signature};
use log::info;
use rusoto_core::{Client, HttpClient, Region};
use serde::{Serialize, Deserialize};
use crate::Artifact;
use super::{Provider, Search, Select, Update};
use super::store::{DataStore, FuncStore, KeyStore, ObjectStore};

pub struct Notary {
    ds: DataStore,
    fs: FuncStore,
    ks: KeyStore,
    os: ObjectStore,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Commit {
    pub artifact: Artifact,
    pub hash:     Vec<u8>,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Submit {
    pub artifact: Artifact,
    pub hash:     Vec<u8>,
    pub checksum: String,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Signed {
    pub presigned: String,
    pub signature: Bytes,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Item {
    pub artifact: Artifact,
    pub location: String,
}

impl Notary {
    pub fn new(region: Region, role: Option<String>) -> Result<Self> {
        let provider = Provider::new(region.clone(), role)?;
        let client   = HttpClient::new()?;
        let client   = Client::new_with(provider.clone(), client);

        Ok(Self {
            ds: DataStore::new(client.clone(), region.clone())?,
            fs: FuncStore::new(client.clone(), region.clone())?,
            ks: KeyStore::new(client.clone(), region.clone())?,
            os: ObjectStore::new(client, provider, region)?,
        })
    }

    pub async fn create(&self, code: &Bytes) -> Result<()> {
        let (data, meta) = self.ds.init().await?;

        info!("created data table: {:?}", data.table_arn);
        info!("created meta table: {:?}", meta.table_arn);

        let meta = self.ks.init().await?;

        info!("created master key: {}", meta.key_id);

        let key  = self.ks.generate(&meta.key_id).await?;
        let hash = hex::encode(&key.public[..]);
        self.ds.put(&key).await?;

        info!("initialized keypair: {}", hash);

        self.os.init().await?;

        info!("created buckets");

        let (public, secret) = self.fs.init(code).await?;

        info!("created function: {:?}", public.function_arn);
        info!("created function: {:?}", secret.function_arn);

        Ok(())
    }

    pub async fn submit(&self, s: Submit) -> Result<Signed> {
        let Submit { mut artifact, hash, checksum } = s;

        let key = self.ds.key().await?;
        let sig = self.ks.sign(&key, &hash).await?;
        let sig = Bytes::copy_from_slice(&sig[..]);

        artifact.signature = sig.clone();

        let presigned = self.os.presign(&artifact, checksum).await?;

        Ok(Signed { presigned, signature: sig })
    }

    pub async fn commit(&self, c: Commit) -> Result<()> {
        let Commit { artifact, hash } = c;

        let out = self.os.get(&artifact).await?;
        let key = self.ds.key().await?;

        let sig = signature(out.metadata).unwrap_or_default();
        let sig = Signature::from_slice(&sig)?;

        if self.ks.verify(&key, &hash, &sig).await.is_err() {
            return Err(anyhow!("invalid or missing signature"));
        }

        self.ds.put(&artifact).await?;

        Ok(())
    }

    pub async fn update(&self, u: Update) -> Result<()> {
        self.ds.update(u).await
    }

    pub async fn search(&self, s: Search) -> Result<Vec<Item>> {
        let artifacts = self.ds.search(s).await?;
        let mut items = Vec::new();

        for artifact in artifacts {
            let location = self.os.location(&artifact).await?;
            items.push(Item { artifact, location });
        }

        Ok(items)
    }

    pub async fn select(&self, s: Select) -> Result<Item> {
        let artifact = self.ds.select(s).await?;
        let location = self.os.location(&artifact).await?;
        Ok(Item { artifact, location })
    }

    pub async fn public(&self) -> Result<PublicKey> {
        Ok(self.ds.key().await?.public)
    }
}

fn signature(metadata: Option<HashMap<String, String>>) -> Option<Vec<u8>> {
    base64::decode(metadata?.remove("signature")?).ok()
}
