use std::env;
use std::fs::File;
use anyhow::Result;
use bytes::Bytes;
use clap::ArgMatches;
use log::{debug, info};
use rusoto_core::{Client, HttpClient, Region};
use tokio::runtime::Runtime;
use super::{code, Notary};
use super::notary::{FuncStore, Provider, PUBLIC_FN, SECRET_FN};

pub fn exec(args: &ArgMatches, region: Region, role: Option<String>) -> Result<()> {
    let runtime = Runtime::new()?;
    Ok(runtime.block_on(async move {
        match args.is_present("create") {
            true  => create(region, role).await,
            false => update(region, role).await,
        }
    })?)
}

async fn create(region: Region, role: Option<String>) -> Result<()> {
    let notary = Notary::new(region, role)?;
    notary.create(&code()?).await?;
    info!("notary initialized");
    Ok(())
}

async fn update(region: Region, role: Option<String>) -> Result<()> {
    let provider = Provider::new(region.clone(), role)?;
    let client   = Client::new_with(provider, HttpClient::new()?);
    let store    = FuncStore::new(client, region)?;

    info!("updating functions");

    let code   = code()?;
    let public = store.update(PUBLIC_FN, &code).await?;
    let secret = store.update(SECRET_FN, &code).await?;

    info!("updated {}", public.function_arn.unwrap_or_default());
    info!("updated {}", secret.function_arn.unwrap_or_default());

    Ok(())
}

fn code() -> Result<Bytes> {
    let path = env::current_exe()?;
    let file = File::open(&path)?;

    debug!("function code from {:?}", path);

    code::archive(&file)
}
