use std::convert::TryInto;
use std::fs::File;
use std::io::{copy, BufReader, Cursor};
use std::time::UNIX_EPOCH;
use anyhow::{anyhow, Result};
use bytes::Bytes;
use time::OffsetDateTime;
use zip::DateTime;
use zip::write::{FileOptions, ZipWriter};

pub fn archive(file: &File) -> Result<Bytes> {
    let sink  = Cursor::new(Vec::new());
    let mtime = mtime(file)?;

    let mut exe = BufReader::new(file);
    let mut zip = ZipWriter::new(sink);

    let options = FileOptions::default()
        .last_modified_time(mtime)
        .unix_permissions(0o777);

    zip.start_file("bootstrap", options)?;
    copy(&mut exe, &mut zip)?;
    let sink = zip.finish()?.into_inner();

    Ok(Bytes::from(sink))
}

fn mtime(file: &File) -> Result<DateTime> {
    let meta  = file.metadata()?;

    let mtime = meta.modified()?.duration_since(UNIX_EPOCH)?.as_secs();
    let mtime = OffsetDateTime::from_unix_timestamp(mtime.try_into()?)?;

    let date  = mtime.date();
    let time  = mtime.time();

    DateTime::from_date_and_time(
        date.year().try_into()?,
        date.month().into(),
        date.day(),
        time.hour(),
        time.minute(),
        time.second(),
    ).map_err(|_| anyhow!("invalid mtime {}", mtime))
}
