use std::io::{Error, Write};
use std::ops::Deref;
use hmac_sha512::Hash;
use zstd::stream::write::Decoder;

pub struct Expand<'a, W: Sized> {
    dec:  Decoder<'a, Vec<u8>>,
    sink: W,
    hash: Hash,
}

pub struct Digest([u8; 64]);

impl<'a, W: Write + Sized> Expand<'a, W> {
    pub fn new(sink: W) -> Result<Self, Error> {
        Ok(Self {
            dec:  Decoder::new(Vec::new())?,
            sink: sink,
            hash: Hash::new(),
        })
    }

    pub fn update(&mut self, bytes: &[u8]) -> Result<(), Error> {
        self.dec.write_all(bytes)?;

        let chunk = self.dec.get_mut();
        self.hash.update(&chunk);
        self.sink.write_all(&chunk)?;

        chunk.clear();

        Ok(())
    }

    pub fn finish(mut self) -> Result<Digest, Error> {
        self.dec.flush()?;

        let chunk = self.dec.into_inner();
        self.hash.update(&chunk);
        self.sink.write_all(&chunk)?;

        let digest = Digest(self.hash.finalize());

        Ok(digest)
    }
}

impl Deref for Digest {
    type Target = [u8];

    fn deref(&self) -> &Self::Target {
        &self.0
    }
}
