use std::process;
use anyhow::Error;
use clap::{load_yaml, App};
use env_logger::Builder;
use log::{error, LevelFilter};
use rusoto_core::Region;
use notary::notary::{deploy, search, submit, update, verify, server};

fn main() {
    let yaml = load_yaml!("args.yml");
    let ver  = env!("CARGO_PKG_VERSION");
    let args = App::from_yaml(&yaml).version(ver).get_matches();

    let mut builder = Builder::from_default_env();
    let mut filter  = |this, rest| {
        builder.filter(Some(module_path!()), this);
        builder.filter(None,                 rest);
    };

    match args.occurrences_of("verbose") {
        0 => filter(LevelFilter::Info,  LevelFilter::Warn),
        1 => filter(LevelFilter::Debug, LevelFilter::Info),
        2 => filter(LevelFilter::Trace, LevelFilter::Info),
        3 => filter(LevelFilter::Trace, LevelFilter::Debug),
        _ => filter(LevelFilter::Trace, LevelFilter::Trace),
    };

    builder.init();

    let role   = args.value_of("role").map(str::to_owned);
    let region = args.value_of("region").map(|name| {
        name.parse().unwrap_or_else(|_| {
            Region::Custom {
                name:     String::from("custom"),
                endpoint: String::from(name),
            }
        })
    }).unwrap_or_default();

    match args.subcommand() {
        ("deploy", Some(args)) => deploy::exec(args, region, role),
        ("search", Some(args)) => search::exec(args, region, role),
        ("submit", Some(args)) => submit::exec(args, region, role),
        ("update", Some(args)) => update::exec(args, region, role),
        ("verify", Some(args)) => verify::exec(args, region, role),
        _                      => server::exec(),
    }.unwrap_or_else(abort);
}

fn abort(e: Error) {
    match e.downcast_ref::<clap::Error>() {
        Some(e) => println!("{}", e.message),
        None    => error!("{:?}", e),
    }
    process::exit(1);
}
