use std::convert::TryFrom;
use std::fmt;
use std::num::TryFromIntError;
use std::str::FromStr;
use anyhow::{anyhow, Error};
use bytes::Bytes;
use serde::{Serialize, Deserialize};
use crate::time::Timestamp;

#[derive(Clone, Debug, PartialEq, Deserialize, Serialize)]
pub struct Artifact {
    pub name:      String,
    pub version:   Version,
    pub target:    Target,
    pub signature: Bytes,
    pub timestamp: Timestamp,
}

#[derive(Copy, Clone, Debug, PartialEq, Deserialize, Serialize)]
pub struct Version {
    major: u16,
    minor: u16,
    patch: u16,
}

#[derive(Clone, Debug, PartialEq, Deserialize, Serialize)]
pub struct Target {
    pub arch:   Arch,
    pub system: System,
}

#[derive(Copy, Clone, Debug, PartialEq, Deserialize, Serialize)]
pub enum Arch {
    X86_64,
    AArch64,
}

#[derive(Copy, Clone, Debug, PartialEq, Deserialize, Serialize)]
pub enum System {
    Linux,
}

impl Artifact {
    pub fn new(name: String, version: Version, target: Target) -> Self {
        Self { name, version, target, ..Default::default() }
    }

    pub fn artifact(&self) -> String {
        format!("{}/{}", self.name, self.target)
    }

    pub fn revision(&self) -> u64 {
        u64::from(self.version)
    }
}

impl Version {
    pub fn new(major: u16, minor: u16, patch: u16) -> Self {
        Self { major, minor, patch }
    }
}

impl Target {
    pub fn new(arch: Arch, system: System) -> Self {
        Self { arch, system }
    }
}

impl Default for Artifact {
    fn default() -> Self {
        Self {
            name:      String::new(),
            version:   Version::new(0, 0, 0),
            target:    Target::new(Arch::X86_64, System::Linux),
            signature: Bytes::new(),
            timestamp: Timestamp::new(0),
        }
    }
}

impl From<Version> for u64 {
    fn from(Version { major, minor, patch }: Version) -> u64 {
        let x = u64::from(major) << 32;
        let y = u64::from(minor) << 16;
        let z = u64::from(patch);
        x | y | z
    }
}

impl TryFrom<u64> for Version {
    type Error = TryFromIntError;

    fn try_from(n: u64) -> Result<Self, Self::Error> {
        const MASK: u64 = 0xFFFF;
        let major = u16::try_from(n >> 32 & MASK)?;
        let minor = u16::try_from(n >> 16 & MASK)?;
        let patch = u16::try_from(n       & MASK)?;
        Ok(Self { major, minor, patch })
    }
}

impl fmt::Display for Version {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "{}.{}.{}", self.major, self.minor, self.patch)
    }
}

impl fmt::Display for Target {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "{}/{}", self.arch, self.system)
    }
}

impl fmt::Display for Arch {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Arch::X86_64  => f.write_str(X86_64),
            Arch::AArch64 => f.write_str(AARCH64),
        }
    }
}

impl fmt::Display for System {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            System::Linux => f.write_str(LINUX),
        }
    }
}

impl FromStr for Version {
    type Err = Error;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        let mut split = s.split('.');

        let mut next = || {
            match split.next().map(u16::from_str) {
                Some(Ok(n))  => Ok(n),
                Some(Err(_)) => Err(anyhow!("invalid")),
                None         => Err(anyhow!("missing")),
            }
        };

        Ok(Self {
            major: next()?,
            minor: next()?,
            patch: next()?,
        })
    }
}

impl FromStr for Arch {
    type Err = Error;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s {
            X86_64  => Ok(Self::X86_64),
            AARCH64 => Ok(Self::AArch64),
            _       => Err(anyhow!("invalid arch {}", s)),
        }
    }
}

impl FromStr for System {
    type Err = Error;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s {
            LINUX => Ok(Self::Linux),
            _     => Err(anyhow!("invalid system {}", s)),
        }
    }
}

const X86_64:  &str = "x86_64";
const AARCH64: &str = "aarch64";
const LINUX:   &str = "linux";
